(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     38655,       1388]
NotebookOptionsPosition[     32083,       1164]
NotebookOutlinePosition[     33572,       1204]
CellTagsIndexPosition[     33486,       1199]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Covariance" :> 
          Documentation`HelpLookup["paclet:ref/Covariance"], "Variance" :> 
          Documentation`HelpLookup["paclet:ref/Variance"], "CentralMoment" :> 
          Documentation`HelpLookup["paclet:ref/CentralMoment"], 
          "ExpectedValue" :> 
          Documentation`HelpLookup["paclet:ref/ExpectedValue"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Covariance\"\>", 
       2->"\<\"Variance\"\>", 3->"\<\"CentralMoment\"\>", 
       4->"\<\"ExpectedValue\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Descriptive Statistics" :> 
          Documentation`HelpLookup["paclet:guide/DescriptiveStatistics"], 
          "New in 6.0: Statistics" :> 
          Documentation`HelpLookup["paclet:guide/NewIn60Statistics"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"Descriptive Statistics\"\>", 
       2->"\<\"New in 6.0: Statistics\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["Correlation", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["Correlation",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Correlation"], "[", 
       RowBox[{
        SubscriptBox[
         StyleBox["v", "TI"], 
         StyleBox["1", "TR"]], ",", 
        SubscriptBox[
         StyleBox["v", "TI"], 
         StyleBox["2", "TR"]]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the correlation between the vectors ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["v", "TI"], 
       StyleBox["1", "TR"]]], "InlineFormula"],
     " and ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["v", "TI"], 
       StyleBox["2", "TR"]]], "InlineFormula"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["Correlation",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Correlation"], "[", 
       StyleBox["m", "TI"], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the correlation matrix for the matrix ",
     Cell[BoxData[
      StyleBox["m", "TI"]], "InlineFormula"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["Correlation",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Correlation"], "[", 
       RowBox[{
        SubscriptBox[
         StyleBox["m", "TI"], 
         StyleBox["1", "TR"]], ",", 
        SubscriptBox[
         StyleBox["m", "TI"], 
         StyleBox["2", "TR"]]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the correlation matrix for the matrices ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["m", "TI"], 
       StyleBox["1", "TR"]]], "InlineFormula"],
     " and ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["m", "TI"], "2"]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, {None}}, "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["Correlation",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Correlation"], "[", 
   RowBox[{
    SubscriptBox[
     StyleBox["v", "TI"], 
     StyleBox["1", "TR"]], ",", 
    SubscriptBox[
     StyleBox["v", "TI"], 
     StyleBox["2", "TR"]]}], "]"}]], "InlineFormula"],
 " gives Pearson's correlation coefficient between ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["v", "TI"], 
   StyleBox["1", "TR"]]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["v", "TI"], 
   StyleBox["2", "TR"]]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->1067943069],

Cell[TextData[{
 "The lists ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["v", "TI"], 
   StyleBox["1", "TR"]]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["v", "TI"], 
   StyleBox["2", "TR"]]], "InlineFormula"],
 " must be the same length."
}], "Notes",
 CellID->160145381],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["Correlation",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Correlation"], "[", 
   RowBox[{
    SubscriptBox[
     StyleBox["v", "TI"], 
     StyleBox["1", "TR"]], ",", 
    SubscriptBox[
     StyleBox["v", "TI"], 
     StyleBox["2", "TR"]]}], "]"}]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    ButtonBox["Covariance",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Covariance"], "[", 
    RowBox[{
     SubscriptBox[
      StyleBox["v", "TI"], 
      StyleBox["1", "TR"]], ",", 
     SubscriptBox[
      StyleBox["v", "TI"], 
      StyleBox["2", "TR"]]}], "]"}], "/", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      ButtonBox["StandardDeviation",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/StandardDeviation"], "[", 
      SubscriptBox[
       StyleBox["v", "TI"], 
       StyleBox["1", "TR"]], "]"}], 
     RowBox[{
      ButtonBox["StandardDeviation",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/StandardDeviation"], "[", 
      SubscriptBox[
       StyleBox["v", "TI"], 
       StyleBox["2", "TR"]], "]"}]}], ")"}]}]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->322703784],

Cell[TextData[{
 "For a matrix ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 " with ",
 Cell[BoxData[
  StyleBox["p", "TI"]], "InlineFormula"],
 " columns ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Correlation",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Correlation"], "[", 
   StyleBox["m", "TI"], "]"}]], "InlineFormula"],
 " is a ",
 Cell[BoxData[
  RowBox[{
   StyleBox["p", "TI"], "\[Times]", 
   StyleBox["p", "TI"]}]], "InlineFormula"],
 " matrix of the covariances between columns of ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 "."
}], "Notes",
 CellID->166488486],

Cell[TextData[{
 "For an ",
 Cell[BoxData[
  RowBox[{
   StyleBox["n", "TI"], "\[Times]", 
   StyleBox["p", "TI"]}]], "InlineFormula"],
 " matrix ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["m", "TI"], 
   StyleBox["1", "TR"]]], "InlineFormula"],
 " and an ",
 Cell[BoxData[
  RowBox[{
   StyleBox["n", "TI"], "\[Times]", 
   StyleBox["q", "TI"]}]], "InlineFormula"],
 " matrix ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["m", "TI"], 
   StyleBox["2", "TR"]]], "InlineFormula"],
 " ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Correlation",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Correlation"], "[", 
   RowBox[{
    SubscriptBox[
     StyleBox["m", "TI"], 
     StyleBox["1", "TR"]], ",", 
    SubscriptBox[
     StyleBox["m", "TI"], 
     StyleBox["2", "TR"]]}], "]"}]], "InlineFormula"],
 " is a ",
 Cell[BoxData[
  RowBox[{
   StyleBox["p", "TI"], "\[Times]", 
   StyleBox["q", "TI"]}]], "InlineFormula"],
 " matrix of the correlations between columns of ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["m", "TI"], "1"]], "InlineFormula"],
 " and columns of ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["m", "TI"], 
   StyleBox["2", "TR"]]], "InlineFormula"],
 "."
}], "Notes",
 CellID->314720011],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Correlation",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Correlation"]], "InlineFormula"],
 " works with ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 " objects. "
}], "Notes",
 CellID->426786876]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->14013230],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(3)", "ExampleCount"]
}], "ExampleSection",
 CellID->207464607],

Cell["Correlation between two vectors:", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->9742],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Correlation", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"a", ",", "b"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->71413352],

Cell[BoxData[
 FractionBox[
  RowBox[{
   RowBox[{"(", 
    RowBox[{"a", "-", "b"}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"Conjugate", "[", "x", "]"}], "-", 
     RowBox[{"Conjugate", "[", "y", "]"}]}], ")"}]}], 
  RowBox[{
   SqrtBox[
    RowBox[{
     RowBox[{"(", 
      RowBox[{"a", "-", "b"}], ")"}], " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"Conjugate", "[", "a", "]"}], "-", 
       RowBox[{"Conjugate", "[", "b", "]"}]}], ")"}]}]], " ", 
   SqrtBox[
    RowBox[{
     RowBox[{"(", 
      RowBox[{"x", "-", "y"}], ")"}], " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"Conjugate", "[", "x", "]"}], "-", 
       RowBox[{"Conjugate", "[", "y", "]"}]}], ")"}]}]]}]]], "Output",
 ImageSize->{522, 35},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->583207714]
}, Open  ]],

Cell["Real values: ", "ExampleText",
 CellID->154165225],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Refine", "[", 
  RowBox[{"%", ",", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"a", ",", "b", ",", "x", ",", "y"}], "}"}], "\[Element]", 
    "Reals"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->244411556],

Cell[BoxData[
 FractionBox[
  RowBox[{
   RowBox[{"(", 
    RowBox[{"a", "-", "b"}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{"x", "-", "y"}], ")"}]}], 
  RowBox[{
   RowBox[{"Abs", "[", 
    RowBox[{"a", "-", "b"}], "]"}], " ", 
   RowBox[{"Abs", "[", 
    RowBox[{"x", "-", "y"}], "]"}]}]]], "Output",
 ImageSize->{135, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->220374171]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->32140598],

Cell["Correlation matrix for a matrix:", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->319539475],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Correlation", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"a", ",", "b"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"c", ",", "d"}], "}"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->84039308],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"a", "-", "c"}], ")"}], " ", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"Conjugate", "[", "b", "]"}], "-", 
          RowBox[{"Conjugate", "[", "d", "]"}]}], ")"}]}], ")"}], "/", 
      RowBox[{"(", 
       RowBox[{
        SqrtBox[
         RowBox[{
          RowBox[{"(", 
           RowBox[{"a", "-", "c"}], ")"}], " ", 
          RowBox[{"(", 
           RowBox[{
            RowBox[{"Conjugate", "[", "a", "]"}], "-", 
            RowBox[{"Conjugate", "[", "c", "]"}]}], ")"}]}]], " ", 
        SqrtBox[
         RowBox[{
          RowBox[{"(", 
           RowBox[{"b", "-", "d"}], ")"}], " ", 
          RowBox[{"(", 
           RowBox[{
            RowBox[{"Conjugate", "[", "b", "]"}], "-", 
            RowBox[{"Conjugate", "[", "d", "]"}]}], ")"}]}]]}], ")"}]}]}], 
    "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"b", "-", "d"}], ")"}], " ", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"Conjugate", "[", "a", "]"}], "-", 
          RowBox[{"Conjugate", "[", "c", "]"}]}], ")"}]}], ")"}], "/", 
      RowBox[{"(", 
       RowBox[{
        SqrtBox[
         RowBox[{
          RowBox[{"(", 
           RowBox[{"a", "-", "c"}], ")"}], " ", 
          RowBox[{"(", 
           RowBox[{
            RowBox[{"Conjugate", "[", "a", "]"}], "-", 
            RowBox[{"Conjugate", "[", "c", "]"}]}], ")"}]}]], " ", 
        SqrtBox[
         RowBox[{
          RowBox[{"(", 
           RowBox[{"b", "-", "d"}], ")"}], " ", 
          RowBox[{"(", 
           RowBox[{
            RowBox[{"Conjugate", "[", "b", "]"}], "-", 
            RowBox[{"Conjugate", "[", "d", "]"}]}], ")"}]}]]}], ")"}]}], ",", 
     "1"}], "}"}]}], "}"}]], "Output",
 ImageSize->{585, 88},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->774270832]
}, Open  ]],

Cell["Real values: ", "ExampleText",
 CellID->524039816],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Refine", "[", 
  RowBox[{"%", ",", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"a", ",", "b", ",", "c", ",", "d"}], "}"}], "\[Element]", 
    "Reals"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1386183],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", 
     FractionBox[
      RowBox[{
       RowBox[{"(", 
        RowBox[{"a", "-", "c"}], ")"}], " ", 
       RowBox[{"(", 
        RowBox[{"b", "-", "d"}], ")"}]}], 
      RowBox[{
       RowBox[{"Abs", "[", 
        RowBox[{"a", "-", "c"}], "]"}], " ", 
       RowBox[{"Abs", "[", 
        RowBox[{"b", "-", "d"}], "]"}]}]]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     FractionBox[
      RowBox[{
       RowBox[{"(", 
        RowBox[{"a", "-", "c"}], ")"}], " ", 
       RowBox[{"(", 
        RowBox[{"b", "-", "d"}], ")"}]}], 
      RowBox[{
       RowBox[{"Abs", "[", 
        RowBox[{"a", "-", "c"}], "]"}], " ", 
       RowBox[{"Abs", "[", 
        RowBox[{"b", "-", "d"}], "]"}]}]], ",", "1"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{355, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->789388811]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->12192744],

Cell["Correlation matrix for two matrices:", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->164543815],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Correlation", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"a", ",", "b"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"c", ",", "d"}], "}"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "x", "}"}], ",", 
      RowBox[{"{", "y", "}"}]}], "}"}]}], "]"}], "//", 
  "MatrixForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->189402817],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztHD1X2zDQsZ0EKKUtC32vCyN/oms79UdQXktCXx95lIGlZWRkzMhWRujH
e+mWkTEjIyMjI6Mr6WRLnM6WnARHMcl7xD5ZOp3uQ7o7X/iwfdj59HX7sLuz
vfn+YLvX6e5823y3f8CaokYQNL4HQfD2dcDuE3Yn/vjnB/sLgpf8O7q9vRVt
SXCJ4IHo9UK0jkajk5MT1UvB/0Svpmg9OzuTY9sp3O/3+e2SgNP5E4lVwS3i
OQf5uJBd1+W1La8blucRwrdioWclZ9y06Ao1/NQ8qwI+Pj6WsHgexOL2+vp6
MBikXA0Z64MIlkFR+ArBMYLXEWyTTBvBGxOOD4nxtjWtWnDMeo2YHhv8LJHa
yNadIGkrLgiq4tPTU3YdDodMDQRf4gwP+8RS1yJCp/Csuu7xq5xWfETfpo45
7RDJpxEe4aDVMdHWrpyKUBuP+WFogclvzFHFf2n3RThnJAMDi0n58kzkYOON
uWff398fHR1lz2Uv1ip3ciGBJju1Li4u2O3d3d35+bmxg6AVwY1gCBxd2oPS
dAu4hXHncbGwd5SBGUepvk1HGfOVxJqkMi56SW0T8bXlJNlU28J0rbE+Nz4Z
PJUloiCzHmOv90CSbYQ5oa0Gw3DmMaeRmW7GaZDe1dUV7KNM0NKrBKqUP0nK
8YH2tIyWGdox3Zs4k1x3a+yB+Ewr1miA3WWbWrR+tmaGPQ/yzLPmgrhiZrRO
ZM0sLGFeUka/7MUOZggi5dkMBg8fJnVy7WnM5OLH5vlUlD8SZYojPok6K1JP
pmikuQdT9OWdSI9EzxiebctZDqYlY3o8k84Ye2eF8sF9LXICq1Lxvm5VKjcA
vZgvJGHav1o1+CDknWPvm4iSJRIn6aFFFiw4Tn9AE97TybVpz8nsRpk1FUT9
ZVdjy2xA1oJZHRx1hdIcjUZyJ9X9GNAa3kPXJGpufvsctb1JTG2NZVu61+aN
iYmxdM4KUV6QJTH2yYL50yyBjc68fN+agFUupVHAeyYhiDBkblPBv1F/Ggvw
QensJYJ/VY4FyFdYFFwGCxNrr9djrcFP/hWxAEw+b1ueLznNwr8Op9wTdGw4
HEqd5br7USjv1taWjDxaU55zLymn94+VQ/7iRG0XjfItX1w2l7rntOqOWJp+
TniYOa1F1q7rLg+MtV75uSeQ05lA1nXJ3TyBWH98KXsc1dcqbnSTUJdc9xTi
xApjKrfTfk9bE3Bw0hhq0njEzTfjX9P2iGfbE97h5deY/HHCsizg/BqUvyWw
LGpSXOOJNSeuutRv+BZjLGpSTBh2NBdp22skYnmKRIH9rU1FsZXhdWM6KqpL
sb5NwPCiKsWv+BZX6tF24l6VktKOI0EqhvQwyqX12ccY1/o2g35rtqg/8UyO
Y0axbnYL0nWvVcAw5uY81izQe7W/FSg2O17Un9QvJ1XmFLbVPejydvFYK85U
jSHNR6aotA+7qD+Zjzyim1XVqUajTmtxi+DJepOCPEKVVRv+V5S4cXdRUUJj
qXdFCeRoXOZUvqfes9/vw4K9qSgBOm5ubqT2hhq1g8FA/nqtbhUl3VTiRG0z
E5DkBfW+08O8Zy1ybsBrZl8PfjuCfy+5S2BNqZzT7Fqe71+jvAzIVu2SSrb8
DQy2sjrmW+oao3N7DNluCeEMSEpZ62ditMdReK3ivF0ki0iTkHo3j/fTeYqF
OmgtsMIe6jXLeGJfUMRv4b9xBI3/ocVLLQ==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{381, 61},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//MatrixForm=",
 CellID->159243067]
}, Open  ]],

Cell["Real values: ", "ExampleText",
 CellID->278507023],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Refine", "[", 
  RowBox[{"%", ",", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"a", ",", "b", ",", "c", ",", "d", ",", "x", ",", "y"}], "}"}], 
    "\[Element]", "Reals"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->486636137],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    FractionBox[
     RowBox[{
      RowBox[{"(", 
       RowBox[{"a", "-", "c"}], ")"}], " ", 
      RowBox[{"(", 
       RowBox[{"x", "-", "y"}], ")"}]}], 
     RowBox[{
      RowBox[{"Abs", "[", 
       RowBox[{"a", "-", "c"}], "]"}], " ", 
      RowBox[{"Abs", "[", 
       RowBox[{"x", "-", "y"}], "]"}]}]], "}"}], ",", 
   RowBox[{"{", 
    FractionBox[
     RowBox[{
      RowBox[{"(", 
       RowBox[{"b", "-", "d"}], ")"}], " ", 
      RowBox[{"(", 
       RowBox[{"x", "-", "y"}], ")"}]}], 
     RowBox[{
      RowBox[{"Abs", "[", 
       RowBox[{"b", "-", "d"}], "]"}], " ", 
      RowBox[{"Abs", "[", 
       RowBox[{"x", "-", "y"}], "]"}]}]], "}"}]}], "}"}]], "Output",
 ImageSize->{318, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->505523296]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->17173],

Cell["Correlations for machine precision reals:", "ExampleText",
 CellID->280986211],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Correlation", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1.5", ",", "3", ",", "5", ",", "10"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"2", ",", "1.25", ",", "15", ",", "8"}], "}"}]}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->270843],

Cell[BoxData["0.47597629019097476`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->80155071]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Correlation", "[", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "2"}], "}"}]}], "]"}], "]"}], "//", 
  "MatrixForm"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->457246],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztlsFuwjAMQEOSsrGWOwcOfAcX0CbBYRr/UKFJcEBMwG9w2J1/DU7qjtSY
NVShJyo1ri3XeU7cuF/5YfW9yQ/rZT6a7/Kf1Xq5H822OzCpjhDiBPdwIODZ
wJO7T8aALMbEjspa3SWORNfOSxNrBoNVrV2C7HJvebpCfzpbgnaJMgWZgFQY
d8zEuaZJXag/3flcYtSxSfZt6uWi6ItexPc93hjW8ByUl/k1TVaTA11XVaWF
q8usPiWesKQ0zyweIbtn/0WhxFOiy4jEZU3S+pHVtUVq69CPSEdjadarx82A
OreeMQl7rJX/mmhEfvf9HYhH38fdsbWTIGH5pWlPtxUqPXuK+nvQLL9PL7Rq
Uz3D5+YxveI1iIbOVlLNzK06bbNn0BzDcvgw7feKZqSf8Qjv7hVtEt/fK/jq
rcuBp3tMr3jhdqEhYfu9Ioy+PJMWplmP4E6X0H/q4v9bdM5p6jdb\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{156, 28},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]//MatrixForm=",
 CellID->55824733]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->92163613],

Cell["Use arbitrary precision:", "ExampleText",
 CellID->79274148],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Correlation", "[", 
  RowBox[{
   RowBox[{"N", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "2", ",", "5", ",", "6"}], "}"}], ",", "20"}], "]"}], 
   ",", 
   RowBox[{"N", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2", ",", "3", ",", "6", ",", "8"}], "}"}], ",", "20"}], 
    "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->22156135],

Cell[BoxData["0.99156103055279677007995911793`18.84473136994383"], "Output",
 ImageSize->{144, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->390791861]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->296033878],

Cell["Find the covariance between vectors of complexes:", "ExampleText",
 CellID->226826659],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Correlation", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"2", "+", "I"}], ",", 
     RowBox[{"3", "-", 
      RowBox[{"2", "I"}]}], ",", 
     RowBox[{"5", "+", 
      RowBox[{"4", "I"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"I", ",", 
     RowBox[{"1", "+", 
      RowBox[{"2", "I"}]}], ",", 
     RowBox[{"10", "-", 
      RowBox[{"5", "I"}]}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->74872810],

Cell[BoxData[
 RowBox[{"-", 
  FractionBox[
   RowBox[{
    FractionBox["7", "2"], "-", 
    RowBox[{"28", " ", "\[ImaginaryI]"}]}], 
   SqrtBox["1139"]]}]], "Output",
 ImageSize->{64, 42},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->172869950]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->31756798],

Cell[TextData[{
 "Compute the correlation matrix for a ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->70409678],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SparseArray", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2", ",", "2"}], "}"}], "\[Rule]", "1"}], ",", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"5", ",", "3"}], "}"}], "\[Rule]", "2"}], ",", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"10", ",", "1"}], "}"}], "\[Rule]", "3"}]}], "}"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->169287888],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "3", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 3, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"10", ",", "3"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{174, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->611735762]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Correlation", "[", "%", "]"}], "//", "MatrixForm"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->76458288],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzlVz1PAkEQPThAREXQEKkEJSaiMQYrYy9URms6QkygMBqkoKOkpKTkJ1BS
UlJSUlJSUlKuc/vB7U3WOF6AmHjJ7e5s5r2Z2b2d23mqNGuvb5VmvVrJlhqV
j1q9+pktvjdgyg5YllWG9zFtwZjBiL9lxqAXbQraIEymZZ/ns2Gntfv9PuOP
dchlByvkEyRnudxut6Vsc5YQH06n0+FwqFh0W+fKymQywaherye54potaD22
rRXLcrlstVoui61mBZEnAhNLjsudTgcgzjC4YgmOx+Mk8lxoQ1gQnFebez6f
z53hM4pf13LX6h6aMHAeKY/Fsn+j/YBW/hitVsCICkkbYm1Ho5ElHps3IEu9
KFEvZvTCtWf2wuz7ZlBxGUlK9ge+WBIIhb/7zbDuM/GVpdbCltsiCz7hFyTU
yz/QUitzJ/s9EiqJdqGA5CiJJYFsx8ge6xmygORtsviLgLYzp0xkiBuSdhbt
AA1Fzb84r6aIuMgGffeHUvkXZ8iMnN8lseCshNnOkOyPVeXb3FrY8BndJou/
LPNXc+Y6tfLM+2eircw12gX8V6Tl3ytkm5a98pot0411myz+IqDtjMqNJZJ2
BO2AQHW73V+hqHmVeh8W/G69pHsBd/PFYmHyosjl2Ww2GAycoX7jh5JBVkVm
FNg21B5gSHJhVIZ578NRtW6eegfXXmHEgvPtT+fBZQ1CNA6bKc9ilh1HD+Lg
3kgfXJ8uCWhd39ZYRGFGZ3H1Bcst854DfruRlbQV+AKY/eoO\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{90, 69},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]//MatrixForm=",
 CellID->636730223]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->14232],

Cell["\<\
A correlation matrix is a covariance matrix scaled by standard deviations:\
\>", "ExampleText",
 CellID->390818049],

Cell[BoxData[
 RowBox[{
  RowBox[{"data", "=", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"5", ",", 
     RowBox[{"{", 
      RowBox[{"20", ",", "5"}], "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->153647153],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sd", "=", 
  RowBox[{"StandardDeviation", "[", "data", "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->231407642],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1.136296287435647`", ",", "1.518751514903077`", ",", "1.1075930616760488`",
    ",", "1.7656956251985931`", ",", "1.3781199046068495`"}], "}"}]], "Output",\

 ImageSize->{294, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->151779498]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Correlation", "[", "data", "]"}], "\[Equal]", 
  RowBox[{"Transpose", "[", 
   RowBox[{
    RowBox[{"Transpose", "[", 
     RowBox[{
      RowBox[{"Covariance", "[", "data", "]"}], "/", "sd"}], "]"}], "/", 
    "sd"}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->447773619],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->650777796]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->38510067],

Cell["\<\
The diagonal elements of a correlation matrix are equal to 1:\
\>", "ExampleText",
 CellID->186522624],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Diagonal", "[", 
  RowBox[{"Correlation", "[", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"10", ",", 
     RowBox[{"{", 
      RowBox[{"100", ",", "4"}], "}"}]}], "]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19722595],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.9999999999999999`", ",", "1.0000000000000009`", ",", 
   "1.0000000000000004`", ",", "1.0000000000000009`"}], "}"}]], "Output",
 ImageSize->{108, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->708551932]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->2379],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["Covariance",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Covariance"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Variance",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Variance"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["CentralMoment",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/CentralMoment"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["ExpectedValue",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ExpectedValue"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->1662]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Descriptive Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DescriptiveStatistics"]], "MoreAbout",
 CellID->58884418],

Cell[TextData[ButtonBox["New in 6.0: Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60Statistics"]], "MoreAbout",
 CellID->89240810]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Correlation - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 35, 9.7689056}", "context" -> "System`", 
    "keywords" -> {
     "correlation coefficient", "correlation matrix", "Pearson correlation", 
      "sample correlation", "association", "cor"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "Correlation[v_1, v_2] gives the correlation between the vectors v_1 and \
v_2. Correlation[m] gives the correlation matrix for the matrix m. \
Correlation[m_1, m_2] gives the correlation matrix for the matrices m_1 and \
m_2.", "synonyms" -> {"cor"}, "title" -> "Correlation", "type" -> "Symbol", 
    "uri" -> "ref/Correlation"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[10742, 377, 359, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->14013230]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 33343, 1192}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 1659, 38, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2282, 65, 1576, 37, 70, "ObjectNameGrid"],
Cell[3861, 104, 2004, 67, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[5902, 176, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6380, 193, 635, 25, 70, "Notes",
 CellID->1067943069],
Cell[7018, 220, 302, 13, 70, "Notes",
 CellID->160145381],
Cell[7323, 235, 1217, 45, 70, "Notes",
 CellID->322703784],
Cell[8543, 282, 613, 24, 70, "Notes",
 CellID->166488486],
Cell[9159, 308, 1214, 50, 70, "Notes",
 CellID->314720011],
Cell[10376, 360, 329, 12, 70, "Notes",
 CellID->426786876]
}, Closed]],
Cell[CellGroupData[{
Cell[10742, 377, 359, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->14013230],
Cell[CellGroupData[{
Cell[11126, 391, 148, 5, 70, "ExampleSection",
 CellID->207464607],
Cell[11277, 398, 207, 5, 70, "ExampleText",
 CellID->9742],
Cell[CellGroupData[{
Cell[11509, 407, 227, 8, 28, "Input",
 CellID->71413352],
Cell[11739, 417, 853, 30, 56, "Output",
 CellID->583207714]
}, Open  ]],
Cell[12607, 450, 56, 1, 70, "ExampleText",
 CellID->154165225],
Cell[CellGroupData[{
Cell[12688, 455, 238, 8, 28, "Input",
 CellID->244411556],
Cell[12929, 465, 435, 16, 53, "Output",
 CellID->220374171]
}, Open  ]],
Cell[13379, 484, 124, 3, 70, "ExampleDelimiter",
 CellID->32140598],
Cell[13506, 489, 212, 5, 70, "ExampleText",
 CellID->319539475],
Cell[CellGroupData[{
Cell[13743, 498, 255, 9, 70, "Input",
 CellID->84039308],
Cell[14001, 509, 2061, 67, 109, "Output",
 CellID->774270832]
}, Open  ]],
Cell[16077, 579, 56, 1, 70, "ExampleText",
 CellID->524039816],
Cell[CellGroupData[{
Cell[16158, 584, 236, 8, 70, "Input",
 CellID->1386183],
Cell[16397, 594, 945, 34, 53, "Output",
 CellID->789388811]
}, Open  ]],
Cell[17357, 631, 124, 3, 70, "ExampleDelimiter",
 CellID->12192744],
Cell[17484, 636, 216, 5, 70, "ExampleText",
 CellID->164543815],
Cell[CellGroupData[{
Cell[17725, 645, 433, 16, 28, "Input",
 CellID->189402817],
Cell[18161, 663, 1968, 36, 96, "Output",
 Evaluatable->False,
 CellID->159243067]
}, Open  ]],
Cell[20144, 702, 56, 1, 70, "ExampleText",
 CellID->278507023],
Cell[CellGroupData[{
Cell[20225, 707, 258, 8, 70, "Input",
 CellID->486636137],
Cell[20486, 717, 870, 31, 53, "Output",
 CellID->505523296]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[21405, 754, 216, 7, 70, "ExampleSection",
 CellID->17173],
Cell[21624, 763, 84, 1, 70, "ExampleText",
 CellID->280986211],
Cell[CellGroupData[{
Cell[21733, 768, 274, 9, 70, "Input",
 CellID->270843],
Cell[22010, 779, 176, 5, 36, "Output",
 CellID->80155071]
}, Open  ]],
Cell[CellGroupData[{
Cell[22223, 789, 256, 9, 70, "Input",
 CellID->457246],
Cell[22482, 800, 702, 15, 70, "Output",
 Evaluatable->False,
 CellID->55824733]
}, Open  ]],
Cell[23199, 818, 124, 3, 70, "ExampleDelimiter",
 CellID->92163613],
Cell[23326, 823, 66, 1, 70, "ExampleText",
 CellID->79274148],
Cell[CellGroupData[{
Cell[23417, 828, 394, 14, 70, "Input",
 CellID->22156135],
Cell[23814, 844, 207, 5, 36, "Output",
 CellID->390791861]
}, Open  ]],
Cell[24036, 852, 125, 3, 70, "ExampleDelimiter",
 CellID->296033878],
Cell[24164, 857, 92, 1, 70, "ExampleText",
 CellID->226826659],
Cell[CellGroupData[{
Cell[24281, 862, 466, 17, 70, "Input",
 CellID->74872810],
Cell[24750, 881, 297, 11, 63, "Output",
 CellID->172869950]
}, Open  ]],
Cell[25062, 895, 124, 3, 70, "ExampleDelimiter",
 CellID->31756798],
Cell[25189, 900, 225, 8, 70, "ExampleText",
 CellID->70409678],
Cell[CellGroupData[{
Cell[25439, 912, 420, 15, 70, "Input",
 CellID->169287888],
Cell[25862, 929, 480, 17, 36, "Output",
 CellID->611735762]
}, Open  ]],
Cell[CellGroupData[{
Cell[26379, 951, 139, 4, 70, "Input",
 CellID->76458288],
Cell[26521, 957, 1003, 20, 70, "Output",
 Evaluatable->False,
 CellID->636730223]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[27573, 983, 233, 7, 70, "ExampleSection",
 CellID->14232],
Cell[27809, 992, 125, 3, 70, "ExampleText",
 CellID->390818049],
Cell[27937, 997, 230, 8, 70, "Input",
 CellID->153647153],
Cell[CellGroupData[{
Cell[28192, 1009, 140, 4, 70, "Input",
 CellID->231407642],
Cell[28335, 1015, 331, 10, 36, "Output",
 CellID->151779498]
}, Open  ]],
Cell[CellGroupData[{
Cell[28703, 1030, 315, 10, 70, "Input",
 CellID->447773619],
Cell[29021, 1042, 161, 5, 36, "Output",
 CellID->650777796]
}, Open  ]],
Cell[29197, 1050, 124, 3, 70, "ExampleDelimiter",
 CellID->38510067],
Cell[29324, 1055, 112, 3, 70, "ExampleText",
 CellID->186522624],
Cell[CellGroupData[{
Cell[29461, 1062, 260, 8, 70, "Input",
 CellID->19722595],
Cell[29724, 1072, 303, 9, 36, "Output",
 CellID->708551932]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[30088, 1088, 310, 9, 70, "SeeAlsoSection",
 CellID->2379],
Cell[30401, 1099, 964, 35, 70, "SeeAlso",
 CellID->1662]
}, Open  ]],
Cell[CellGroupData[{
Cell[31402, 1139, 299, 8, 70, "MoreAboutSection"],
Cell[31704, 1149, 154, 3, 70, "MoreAbout",
 CellID->58884418],
Cell[31861, 1154, 150, 3, 70, "MoreAbout",
 CellID->89240810]
}, Open  ]],
Cell[32026, 1160, 27, 0, 70, "History"],
Cell[32056, 1162, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

