(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     54110,       1676]
NotebookOptionsPosition[     46621,       1419]
NotebookOutlinePosition[     48108,       1459]
CellTagsIndexPosition[     48022,       1454]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Variance" :> Documentation`HelpLookup["paclet:ref/Variance"], 
          "Correlation" :> Documentation`HelpLookup["paclet:ref/Correlation"],
           "CentralMoment" :> 
          Documentation`HelpLookup["paclet:ref/CentralMoment"], 
          "ExpectedValue" :> 
          Documentation`HelpLookup["paclet:ref/ExpectedValue"], "Dot" :> 
          Documentation`HelpLookup["paclet:ref/Dot"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Variance\"\>", 2->"\<\"Correlation\"\>", 
       3->"\<\"CentralMoment\"\>", 4->"\<\"ExpectedValue\"\>", 
       5->"\<\"Dot\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Descriptive Statistics" :> 
          Documentation`HelpLookup["paclet:guide/DescriptiveStatistics"], 
          "Numerical Data" :> 
          Documentation`HelpLookup["paclet:guide/NumericalData"], 
          "Statistics" :> Documentation`HelpLookup["paclet:guide/Statistics"],
           "New in 6.0: Statistics" :> 
          Documentation`HelpLookup["paclet:guide/NewIn60Statistics"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"Descriptive Statistics\"\>", 
       2->"\<\"Numerical Data\"\>", 3->"\<\"Statistics\"\>", 
       4->"\<\"New in 6.0: Statistics\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["Covariance", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["Covariance",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Covariance"], "[", 
       RowBox[{
        SubscriptBox[
         StyleBox["v", "TI"], 
         StyleBox["1", "TR"]], ",", 
        SubscriptBox[
         StyleBox["v", "TI"], 
         StyleBox["2", "TR"]]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the covariance between lists ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["v", "TI"], 
       StyleBox["1", "TR"]]], "InlineFormula"],
     " and ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["v", "TI"], 
       StyleBox["2", "TR"]]], "InlineFormula"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["Covariance",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Covariance"], "[", 
       StyleBox["m", "TI"], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the covariance matrix for the matrix ",
     Cell[BoxData[
      StyleBox["m", "TI"]], "InlineFormula"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["Covariance",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Covariance"], "[", 
       RowBox[{
        SubscriptBox[
         StyleBox["m", "TI"], 
         StyleBox["1", "TR"]], ",", 
        SubscriptBox[
         StyleBox["m", "TI"], 
         StyleBox["2", "TR"]]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the covariance matrix for the matrices ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["m", "TI"], 
       StyleBox["1", "TR"]]], "InlineFormula"],
     " and ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["m", "TI"], "2"]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, {None}}, "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["Covariance",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Covariance"], "[", 
   RowBox[{
    SubscriptBox[
     StyleBox["v", "TI"], 
     StyleBox["1", "TR"]], ",", 
    SubscriptBox[
     StyleBox["v", "TI"], 
     StyleBox["2", "TR"]]}], "]"}]], "InlineFormula"],
 " gives the unbiased estimate of the covariance between ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["v", "TI"], 
   StyleBox["1", "TR"]]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["v", "TI"], 
   StyleBox["2", "TR"]]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->1067943069],

Cell[TextData[{
 "The lists ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["v", "TI"], 
   StyleBox["1", "TR"]]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["v", "TI"], 
   StyleBox["2", "TR"]]], "InlineFormula"],
 " must be the same length."
}], "Notes",
 CellID->160145381],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["Covariance",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Covariance"], "[", 
   RowBox[{
    SubscriptBox[
     StyleBox["v", "TI"], 
     StyleBox["1", "TR"]], ",", 
    SubscriptBox[
     StyleBox["v", "TI"], 
     StyleBox["2", "TR"]]}], "]"}]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      SubscriptBox[
       StyleBox["v", "TI"], 
       StyleBox["1", "TR"]], "-", 
      RowBox[{
       ButtonBox["Mean",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Mean"], "[", 
       SubscriptBox[
        StyleBox["v", "TI"], 
        StyleBox["1", "TR"]], "]"}]}], ")"}], ".", " ", 
    RowBox[{"Conjugate", "[", 
     RowBox[{
      SubscriptBox[
       StyleBox["v", "TI"], 
       StyleBox["2", "TR"]], "-", 
      RowBox[{
       ButtonBox["Mean",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Mean"], "[", 
       SubscriptBox[
        StyleBox["v", "TI"], 
        StyleBox["2", "TR"]], "]"}]}], "]"}]}], "/", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      ButtonBox["Length",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/Length"], "[", 
      SubscriptBox[
       StyleBox["v", "TI"], 
       StyleBox["1", "TR"]], "]"}], "-", "1"}], ")"}]}]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->322703784],

Cell[TextData[{
 "For a matrix ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 " with ",
 Cell[BoxData[
  StyleBox["p", "TI"]], "InlineFormula"],
 " columns ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Covariance",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Covariance"], "[", 
   StyleBox["m", "TI"], "]"}]], "InlineFormula"],
 " is a ",
 Cell[BoxData[
  RowBox[{
   StyleBox["p", "TI"], "\[Times]", 
   StyleBox["p", "TI"]}]], "InlineFormula"],
 " matrix of the covariances between columns of ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 "."
}], "Notes",
 CellID->166488486],

Cell[TextData[{
 "For an ",
 Cell[BoxData[
  RowBox[{
   StyleBox["n", "TI"], "\[Times]", 
   StyleBox["p", "TI"]}]], "InlineFormula"],
 " matrix ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["m", "TI"], 
   StyleBox["1", "TR"]]], "InlineFormula"],
 " and an ",
 Cell[BoxData[
  RowBox[{
   StyleBox["n", "TI"], "\[Times]", 
   StyleBox["q", "TI"]}]], "InlineFormula"],
 " matrix ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["m", "TI"], 
   StyleBox["2", "TR"]]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Covariance",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Covariance"], "[", 
   RowBox[{
    SubscriptBox[
     StyleBox["m", "TI"], 
     StyleBox["1", "TR"]], ",", 
    SubscriptBox[
     StyleBox["m", "TI"], 
     StyleBox["2", "TR"]]}], "]"}]], "InlineFormula"],
 " is a ",
 Cell[BoxData[
  RowBox[{
   StyleBox["p", "TI"], "\[Times]", 
   StyleBox["q", "TI"]}]], "InlineFormula"],
 " matrix of the covariances between columns of ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["m", "TI"], "1"]], "InlineFormula"],
 " and columns of ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["m", "TI"], 
   StyleBox["2", "TR"]]], "InlineFormula"],
 "."
}], "Notes",
 CellID->314720011],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Covariance",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Covariance"]], "InlineFormula"],
 " works with ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 " objects. "
}], "Notes",
 CellID->426786876]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->13780915],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(3)", "ExampleCount"]
}], "ExampleSection",
 CellID->13722518],

Cell["Covariance between two vectors:", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->9742],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Covariance", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"a", ",", "b"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->71413352],

Cell[BoxData[
 RowBox[{
  FractionBox["1", "2"], " ", 
  RowBox[{"(", 
   RowBox[{"a", "-", "b"}], ")"}], " ", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"Conjugate", "[", "x", "]"}], "-", 
    RowBox[{"Conjugate", "[", "y", "]"}]}], ")"}]}]], "Output",
 ImageSize->{257, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->192990261]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1636889],

Cell["Covariance matrix for a matrix:", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->319539475],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Covariance", "[", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"a", ",", "b"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"c", ",", "d"}], "}"}]}], "}"}], "]"}], "//", 
  "MatrixForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->84039308],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztHLtS20BQSDYQbIZHSAjJZIYqFBl+I6nyER4mM1BkkiEUdJQuXbr0J7h0
6ZKSkpKSkpIhJKPc3kM6rVfWSX7oJE4zSNrT3Wpft7e+XfGtc3H6/Ufn4uyk
c/j1vPPr9Ozk9+GXn+esKVjxPH/P87zjdx67D9kd/1PHW/YXgwLaZOdAdvK1
+xX+tAnnYDAYiCESjnBGo/S2Pxy+urqScMAxNfjt7e3taDSC2zUDTE/q7Tc3
NxhTv9+XvcwxPT4+Xl5expgC1SqQJTB5mkz0tmcOd7tdNgxu/QiTf319/Qq9
dwPBHxDcQvAbqYG2vL4HvOzaktdPGc8bCN+2ZfQI+TATYIaQlB7X6P39Pdzu
oVEfSSwxfIDgNoL3M7g8mnF8kCHVqtG/Hibnqq6leF6/hlND9y9NBuzIK8aJ
Zyk8D6SFwFX5HmEy6sD2A73AruQE5ge0NWSTenMgrw0CR5BC4XapFAGIvSvu
d1AliU/QibkrV96UxHA/IUHm6MUCOG0eIAq9ps490LhFttJvzMKVp7WZ2tfP
iYnzvYZa5bN1I7oxt7RM6sIdOXudnTg7Qb0wLLDE8a2IxLHXEd50PB574uCe
E+DE8yxaoFczVT5GVoc5mUkjnMt45UhvofFptKwW0kVCHlXibMpzzKOI/pzl
5JJvfS2noH+KPRHtnxYeFS1XxmEcYVKRJBVhUpHnlvWU5/UmC49q7JeWWfxS
Az7m5wmiqEtr38T95moh1Btprzkp2+lxIiXjbFnO23KqyZ+JN3G2kkYThitu
LRn6LOZ9YP2C15O/zSRfHj92wuTOUkR9DpsCbx1oMtJ3qVRMoMvPl+3Ujpba
wdJ3tHBEl9aOLaEMuqj4B8PtWmtnkgsb9ZQl1WLzDmuU5zwm/MA+2doiWzFd
nxGMsx44y0HHoMc5sWAqsAaXz2dWxF6M76xdBkylyGnltQqej0zMEMhKU6tT
S+spDgB2Ua+jMJl/3DXsZ5Q3XQCdag6vGdK7GibnPKZ7o5AW4IT5eF4yJiUB
ABR3/4xGKjm3Xe+ENJUK1Frw13hklkbNMNlqZWqUp8lHbyvGXdmVFbZVeph5
IlfpUS79Zqs23sFRUSVEFlRU6Wo95lTrUR2Z16Daw5K8Rm3z3C8ri++qPZyd
mNiJwNLr9SS8tGqPKb/1LcjDqRaXrU/VQgl1Hs5mqmgzBX1S/NWH7pP6/f7D
wwPcUr8KgGS8I2qhpYSumkOjOONXRz11+rIrN4S+7u7uhsNhhEW+czQaye+t
7KjhsjJX7So3EGxHHZiNsnSVG7R+WSRKfAXIggvpk9IqN9R+D1QEUKuybVUA
ae025P5nqdGomh7qWY0hpNzr9RLfI4te6d9I47wEmbeocgXGwnmzo+oC71nH
2vdZDFcoR1lujcXcqJt3ZQXI0mcrk0hXiJkbzy+sNzhh3T2hMYGGSXz4ng9T
PEZg2gwnKyii0ZHQxH998Fb+A23+Hc0=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{532, 45},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//MatrixForm=",
 CellID->60134476]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->385687642],

Cell["Covariance matrix for two matrices:", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->164543815],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Covariance", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"a", ",", "b"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"c", ",", "d"}], "}"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "x", "}"}], ",", 
      RowBox[{"{", "y", "}"}]}], "}"}]}], "]"}], "//", 
  "MatrixForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->189402817],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      RowBox[{
       FractionBox["1", "2"], " ", 
       RowBox[{"(", 
        RowBox[{"a", "-", "c"}], ")"}], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"Conjugate", "[", "x", "]"}], "-", 
         RowBox[{"Conjugate", "[", "y", "]"}]}], ")"}]}]},
     {
      RowBox[{
       FractionBox["1", "2"], " ", 
       RowBox[{"(", 
        RowBox[{"b", "-", "d"}], ")"}], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"Conjugate", "[", "x", "]"}], "-", 
         RowBox[{"Conjugate", "[", "y", "]"}]}], ")"}]}]}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$, BoxForm`opts$665]]]], "Output",
 ImageSize->{276, 45},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//MatrixForm=",
 CellID->411990716]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->9643],

Cell["Covariances for machine precision reals:", "ExampleText",
 CellID->280986211],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Covariance", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1.5", ",", "3", ",", "5", ",", "10"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"2", ",", "1.25", ",", "15", ",", "8"}], "}"}]}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->270843],

Cell[BoxData["11.260416666666666`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->521656074]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Covariance", "[", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "2"}], "}"}]}], "]"}], "]"}], "//", 
  "MatrixForm"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->457246],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWMFSgzAQTQloLdWTOlV76NfoyY9gOs60B0en9m88eO9PYgJdZ/tYICEK
HOxMA7vZhJeXTfLgOdtvXl6z/XadrZ522ftmu/5YPb7tjEtPlFIH818ulLnP
zV3xP+S5uZZlYkttvcVPfYIdF1ExeOemsKb1R6rsxdqa+ROwqZ77sR9qF7Er
b0/1iOcGbC2iTm2R/NhFTImFnl2tRX402JHYKtw7a8R6Js0SsxE11uMouvHJ
Z7OK9UL0IvIogE8chYxDZjJtYaCNYcSNjF+LDCLiuQdztm0s5L4PY25tuzFG
+OpwIkOYY1EwQyFjjo6oi3mPeW1MY/dmJGRmxs8Wj5qesmZurnpnDFcsrsBh
GXPbqzCq/7yTo26duLzMy1M6zaunNlcJqBYSaJeyeN4O+yM14Ibu6z/q6OV8
4ry07fA4D1zVPdT066sKp06jkPVas5rDEYWsHB/vfQesQynPc+k5juzL+2Zb
XhX1J6eH29PsmTh+/TlaPhtYkPN1aF3aP5P0/ll/Wmshp+48EPydZpV3UfFN
sOPaadi3WtcKzuWiR86aletMYiiIt9/fU9yifFZx/9oWM0DmmTQCaUPSCvSF
iL91kqbg9jIP06SY8yVKt69m5Rc2NfkGrz5eBA==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{170, 28},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]//MatrixForm=",
 CellID->479113147]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->92163613],

Cell["Use arbitrary precision:", "ExampleText",
 CellID->79274148],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Covariance", "[", 
  RowBox[{
   RowBox[{"N", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "2", ",", "5", ",", "6"}], "}"}], ",", "20"}], "]"}], 
   ",", 
   RowBox[{"N", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2", ",", "3", ",", "6", ",", "8"}], "}"}], ",", "20"}], 
    "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->22156135],

Cell[BoxData["6.5`19.14390657568428"], "Output",
 ImageSize->{144, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->507853044]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->296033878],

Cell["Find the covariance between vectors of complexes:", "ExampleText",
 CellID->226826659],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Covariance", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"2", "+", "I"}], ",", 
     RowBox[{"3", "-", 
      RowBox[{"2", "I"}]}], ",", 
     RowBox[{"5", "+", 
      RowBox[{"4", "I"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"I", ",", 
     RowBox[{"1", "+", 
      RowBox[{"2", "I"}]}], ",", 
     RowBox[{"10", "-", 
      RowBox[{"5", "I"}]}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->74872810],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", 
   FractionBox["7", "3"]}], "+", 
  FractionBox[
   RowBox[{"56", " ", "\[ImaginaryI]"}], "3"]}]], "Output",
 ImageSize->{65, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->496651772]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->31756798],

Cell[TextData[{
 "Compute the covariance of a ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->70409678],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SparseArray", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2", ",", "2"}], "}"}], "\[Rule]", "1"}], ",", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"5", ",", "3"}], "}"}], "\[Rule]", "2"}]}], "}"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->47165161],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "2", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 2, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"5", ",", "3"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{167, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->770385565]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Covariance", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->84946980],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", 
     FractionBox["1", "5"], ",", 
     RowBox[{"-", 
      FractionBox["1", "10"]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", 
     RowBox[{"-", 
      FractionBox["1", "10"]}], ",", 
     FractionBox["4", "5"]}], "}"}]}], "}"}]], "Output",
 ImageSize->{259, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->560192534]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->407],

Cell["Compute the covariance of two financial time series:", "ExampleText",
 CellID->267330338],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Covariance", "[", 
  RowBox[{
   RowBox[{"FinancialData", "[", 
    RowBox[{"\"\<^GSPC\>\"", ",", "\"\<Price\>\"", ",", 
     RowBox[{"{", 
      RowBox[{"2004", ",", "1", ",", "1"}], "}"}], ",", "\"\<Value\>\""}], 
    "]"}], ",", 
   RowBox[{"FinancialData", "[", 
    RowBox[{"\"\<^DJI\>\"", ",", "\"\<Price\>\"", ",", 
     RowBox[{"{", 
      RowBox[{"2004", ",", "1", ",", "1"}], "}"}], ",", "\"\<Value\>\""}], 
    "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->294789194],

Cell[BoxData["63161.71277113892`"], "Output",
 ImageSize->{56, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->65016236]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->23218],

Cell["\<\
The covariance tends to be large only on the diagonal of a random matrix:\
\>", "ExampleText",
 CellID->334539635],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"Covariance", "[", 
   RowBox[{"RandomReal", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"50", ",", "50"}], "}"}]}], "]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->85261126],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 180},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->675638337]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->216349561],

Cell["The covariance of a list with itself is the variance:", "ExampleText",
 CellID->594410778],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"Variance", "[", 
    RowBox[{"{", 
     RowBox[{"a", ",", "b", ",", "c"}], "}"}], "]"}], "\[Equal]", 
   RowBox[{"Covariance", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"a", ",", "b", ",", "c"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"a", ",", "b", ",", "c"}], "}"}]}], "]"}]}], "//", 
  "Simplify"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->143369198],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1993992]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->621014266],

Cell["The diagonal of a covariance matrix is the variance:", "ExampleText",
 CellID->49753653],

Cell[BoxData[
 RowBox[{
  RowBox[{"data", "=", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"5", ",", 
     RowBox[{"{", 
      RowBox[{"20", ",", "5"}], "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->278146238],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Diagonal", "[", 
  RowBox[{"Covariance", "[", "data", "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->27593145],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "2.0542960575677287`", ",", "2.347937097434482`", ",", "1.68692464328146`", 
   ",", "2.238579162906947`", ",", "1.7189492485683755`"}], "}"}]], "Output",
 ImageSize->{301, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->149606132]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Variance", "[", "data", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->40466575],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "2.054296057567729`", ",", "2.3479370974344826`", ",", "1.68692464328146`", 
   ",", "2.2385791629069476`", ",", "1.718949248568376`"}], "}"}]], "Output",
 ImageSize->{301, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->38260715]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->38510067],

Cell["\<\
A covariance matrix scaled by standard deviations is a correlation matrix:\
\>", "ExampleText",
 CellID->390818049],

Cell[BoxData[
 RowBox[{
  RowBox[{"data", "=", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"5", ",", 
     RowBox[{"{", 
      RowBox[{"20", ",", "5"}], "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->153647153],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sd", "=", 
  RowBox[{"StandardDeviation", "[", "data", "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->231407642],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1.3906243672165266`", ",", "1.3088711105787227`", ",", 
   "1.4982809665538643`", ",", "1.5090784265506267`", ",", 
   "1.4707845262949972`"}], "}"}]], "Output",
 ImageSize->{308, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->360098309]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Correlation", "[", "data", "]"}], "\[Equal]", 
  RowBox[{"Transpose", "[", 
   RowBox[{
    RowBox[{"Transpose", "[", 
     RowBox[{
      RowBox[{"Covariance", "[", "data", "]"}], "/", "sd"}], "]"}], "/", 
    "sd"}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->447773619],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->168318179]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Neat Examples",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->21415],

Cell["Compute the covariance for a GCD array:", "ExampleText",
 CellID->185968618],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixPlot", "[", 
  RowBox[{"Covariance", "[", 
   RowBox[{"Array", "[", 
    RowBox[{"GCD", ",", 
     RowBox[{"{", 
      RowBox[{"50", ",", "50"}], "}"}]}], "]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->514333197],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 184},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->49716551]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->9914],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["Variance",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Variance"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Correlation",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Correlation"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["CentralMoment",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/CentralMoment"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["ExpectedValue",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ExpectedValue"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Dot",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Dot"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->1662]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Descriptive Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DescriptiveStatistics"]], "MoreAbout",
 CellID->34566052],

Cell[TextData[ButtonBox["Numerical Data",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NumericalData"]], "MoreAbout",
 CellID->128263279],

Cell[TextData[ButtonBox["Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/Statistics"]], "MoreAbout",
 CellID->339984597],

Cell[TextData[ButtonBox["New in 6.0: Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60Statistics"]], "MoreAbout",
 CellID->185185442]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Covariance - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 35, 56.2544706}", "context" -> "System`", 
    "keywords" -> {
     "covariance", "covariance coefficient", "covariance matrix", 
      "sample covariance", "complex covariance", "dispersion", "cov"}, 
    "index" -> True, "label" -> "Built-in Mathematica Symbol", "language" -> 
    "en", "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "Covariance[v_1, v_2] gives the covariance between lists v_1 and v_2. \
Covariance[m] gives the covariance matrix for the matrix m. Covariance[m_1, \
m_2] gives the covariance matrix for the matrices m_1 and m_2.", 
    "synonyms" -> {"cov"}, "title" -> "Covariance", "type" -> "Symbol", "uri" -> 
    "ref/Covariance"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[11213, 389, 359, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->13780915]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 47879, 1447}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 1984, 43, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2607, 70, 1575, 37, 70, "ObjectNameGrid"],
Cell[4185, 109, 1989, 67, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[6211, 181, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6689, 198, 639, 25, 70, "Notes",
 CellID->1067943069],
Cell[7331, 225, 302, 13, 70, "Notes",
 CellID->160145381],
Cell[7636, 240, 1381, 52, 70, "Notes",
 CellID->322703784],
Cell[9020, 294, 611, 24, 70, "Notes",
 CellID->166488486],
Cell[9634, 320, 1212, 50, 70, "Notes",
 CellID->314720011],
Cell[10849, 372, 327, 12, 70, "Notes",
 CellID->426786876]
}, Closed]],
Cell[CellGroupData[{
Cell[11213, 389, 359, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->13780915],
Cell[CellGroupData[{
Cell[11597, 403, 147, 5, 70, "ExampleSection",
 CellID->13722518],
Cell[11747, 410, 206, 5, 70, "ExampleText",
 CellID->9742],
Cell[CellGroupData[{
Cell[11978, 419, 226, 8, 28, "Input",
 CellID->71413352],
Cell[12207, 429, 382, 13, 51, "Output",
 CellID->192990261]
}, Open  ]],
Cell[12604, 445, 123, 3, 70, "ExampleDelimiter",
 CellID->1636889],
Cell[12730, 450, 211, 5, 70, "ExampleText",
 CellID->319539475],
Cell[CellGroupData[{
Cell[12966, 459, 296, 11, 28, "Input",
 CellID->84039308],
Cell[13265, 472, 1719, 32, 80, "Output",
 Evaluatable->False,
 CellID->60134476]
}, Open  ]],
Cell[14999, 507, 125, 3, 70, "ExampleDelimiter",
 CellID->385687642],
Cell[15127, 512, 215, 5, 70, "ExampleText",
 CellID->164543815],
Cell[CellGroupData[{
Cell[15367, 521, 432, 16, 70, "Input",
 CellID->189402817],
Cell[15802, 539, 1260, 38, 80, "Output",
 CellID->411990716]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[17111, 583, 215, 7, 70, "ExampleSection",
 CellID->9643],
Cell[17329, 592, 83, 1, 70, "ExampleText",
 CellID->280986211],
Cell[CellGroupData[{
Cell[17437, 597, 273, 9, 70, "Input",
 CellID->270843],
Cell[17713, 608, 176, 5, 36, "Output",
 CellID->521656074]
}, Open  ]],
Cell[CellGroupData[{
Cell[17926, 618, 255, 9, 70, "Input",
 CellID->457246],
Cell[18184, 629, 874, 18, 70, "Output",
 Evaluatable->False,
 CellID->479113147]
}, Open  ]],
Cell[19073, 650, 124, 3, 70, "ExampleDelimiter",
 CellID->92163613],
Cell[19200, 655, 66, 1, 70, "ExampleText",
 CellID->79274148],
Cell[CellGroupData[{
Cell[19291, 660, 393, 14, 70, "Input",
 CellID->22156135],
Cell[19687, 676, 179, 5, 36, "Output",
 CellID->507853044]
}, Open  ]],
Cell[19881, 684, 125, 3, 70, "ExampleDelimiter",
 CellID->296033878],
Cell[20009, 689, 92, 1, 70, "ExampleText",
 CellID->226826659],
Cell[CellGroupData[{
Cell[20126, 694, 465, 17, 70, "Input",
 CellID->74872810],
Cell[20594, 713, 278, 10, 51, "Output",
 CellID->496651772]
}, Open  ]],
Cell[20887, 726, 124, 3, 70, "ExampleDelimiter",
 CellID->31756798],
Cell[21014, 731, 216, 8, 70, "ExampleText",
 CellID->70409678],
Cell[CellGroupData[{
Cell[21255, 743, 324, 12, 70, "Input",
 CellID->47165161],
Cell[21582, 757, 479, 17, 36, "Output",
 CellID->770385565]
}, Open  ]],
Cell[CellGroupData[{
Cell[22098, 779, 105, 3, 70, "Input",
 CellID->84946980],
Cell[22206, 784, 533, 19, 51, "Output",
 CellID->560192534]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[22788, 809, 221, 7, 70, "ExampleSection",
 CellID->407],
Cell[23012, 818, 95, 1, 70, "ExampleText",
 CellID->267330338],
Cell[CellGroupData[{
Cell[23132, 823, 514, 14, 70, "Input",
 CellID->294789194],
Cell[23649, 839, 174, 5, 36, "Output",
 CellID->65016236]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[23872, 850, 233, 7, 70, "ExampleSection",
 CellID->23218],
Cell[24108, 859, 124, 3, 70, "ExampleText",
 CellID->334539635],
Cell[CellGroupData[{
Cell[24257, 866, 335, 11, 70, "Input",
 CellID->85261126],
Cell[24595, 879, 6426, 109, 70, "Output",
 Evaluatable->False,
 CellID->675638337]
}, Open  ]],
Cell[31036, 991, 125, 3, 70, "ExampleDelimiter",
 CellID->216349561],
Cell[31164, 996, 96, 1, 70, "ExampleText",
 CellID->594410778],
Cell[CellGroupData[{
Cell[31285, 1001, 419, 14, 70, "Input",
 CellID->143369198],
Cell[31707, 1017, 159, 5, 36, "Output",
 CellID->1993992]
}, Open  ]],
Cell[31881, 1025, 125, 3, 70, "ExampleDelimiter",
 CellID->621014266],
Cell[32009, 1030, 94, 1, 70, "ExampleText",
 CellID->49753653],
Cell[32106, 1033, 230, 8, 70, "Input",
 CellID->278146238],
Cell[CellGroupData[{
Cell[32361, 1045, 143, 4, 70, "Input",
 CellID->27593145],
Cell[32507, 1051, 327, 9, 36, "Output",
 CellID->149606132]
}, Open  ]],
Cell[CellGroupData[{
Cell[32871, 1065, 106, 3, 70, "Input",
 CellID->40466575],
Cell[32980, 1070, 326, 9, 36, "Output",
 CellID->38260715]
}, Open  ]],
Cell[33321, 1082, 124, 3, 70, "ExampleDelimiter",
 CellID->38510067],
Cell[33448, 1087, 125, 3, 70, "ExampleText",
 CellID->390818049],
Cell[33576, 1092, 230, 8, 70, "Input",
 CellID->153647153],
Cell[CellGroupData[{
Cell[33831, 1104, 140, 4, 70, "Input",
 CellID->231407642],
Cell[33974, 1110, 335, 10, 36, "Output",
 CellID->360098309]
}, Open  ]],
Cell[CellGroupData[{
Cell[34346, 1125, 315, 10, 70, "Input",
 CellID->447773619],
Cell[34664, 1137, 161, 5, 36, "Output",
 CellID->168318179]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[34874, 1148, 224, 7, 70, "ExampleSection",
 CellID->21415],
Cell[35101, 1157, 82, 1, 70, "ExampleText",
 CellID->185968618],
Cell[CellGroupData[{
Cell[35208, 1162, 258, 8, 70, "Input",
 CellID->514333197],
Cell[35469, 1172, 8581, 145, 70, "Output",
 Evaluatable->False,
 CellID->49716551]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[44111, 1324, 310, 9, 70, "SeeAlsoSection",
 CellID->9914],
Cell[44424, 1335, 1201, 44, 70, "SeeAlso",
 CellID->1662]
}, Open  ]],
Cell[CellGroupData[{
Cell[45662, 1384, 299, 8, 70, "MoreAboutSection"],
Cell[45964, 1394, 154, 3, 70, "MoreAbout",
 CellID->34566052],
Cell[46121, 1399, 139, 3, 70, "MoreAbout",
 CellID->128263279],
Cell[46263, 1404, 132, 3, 70, "MoreAbout",
 CellID->339984597],
Cell[46398, 1409, 151, 3, 70, "MoreAbout",
 CellID->185185442]
}, Open  ]],
Cell[46564, 1415, 27, 0, 70, "History"],
Cell[46594, 1417, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

