(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     50402,       1802]
NotebookOptionsPosition[     41417,       1483]
NotebookOutlinePosition[     42860,       1522]
CellTagsIndexPosition[     42774,       1517]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Resultant" :> Documentation`HelpLookup["paclet:ref/Resultant"],
           "Det" :> Documentation`HelpLookup["paclet:ref/Det"], "Tr" :> 
          Documentation`HelpLookup["paclet:ref/Tr"], "SquareFreeQ" :> 
          Documentation`HelpLookup["paclet:ref/SquareFreeQ"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Resultant\"\>", 2->"\<\"Det\"\>", 
       3->"\<\"Tr\"\>", 4->"\<\"SquareFreeQ\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Polynomial Algebra" :> 
          Documentation`HelpLookup["paclet:guide/PolynomialAlgebra"], 
          "Polynomial Systems" :> 
          Documentation`HelpLookup["paclet:guide/PolynomialSystems"], 
          "New in 6.0: Symbolic Computation" :> 
          Documentation`HelpLookup["paclet:guide/NewIn60AlgebraicComputing"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Polynomial Algebra\"\>", 
       2->"\<\"Polynomial Systems\"\>", 
       3->"\<\"New in 6.0: Symbolic Computation\"\>", 
       4->"\<\"New in 6.0: Mathematics & Algorithms\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["Discriminant", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["Discriminant",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Discriminant"], "[", 
       RowBox[{
        StyleBox["poly", "TI"], ",", 
        StyleBox["var", "TI"]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]computes the discriminant of the polynomial ",
     Cell[BoxData[
      StyleBox["poly", "TI"]], "InlineFormula"],
     " with respect to the variable ",
     Cell[BoxData[
      StyleBox["var", "TI"]], "InlineFormula"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["Discriminant",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Discriminant"], "[", 
       RowBox[{
        StyleBox["poly", "TI"], ",", 
        StyleBox["var", "TI"], ",", 
        RowBox[{
         ButtonBox["Modulus",
          BaseStyle->"Link",
          ButtonData->"paclet:ref/Modulus"], "->", 
         StyleBox["p", "TI"]}]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]computes the discriminant modulo ",
     Cell[BoxData[
      FormBox["p", TraditionalForm]], "InlineMath"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->11643]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "The discriminant of a polynomial with leading coefficient one is the \
product over all pairs of roots ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "i"], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "j"], TraditionalForm]], "InlineMath"],
 " of ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox[
    RowBox[{"(", 
     RowBox[{
      SubscriptBox["x", "i"], "-", 
      SubscriptBox["x", "j"]}], ")"}], "2"], TraditionalForm]], 
  "InlineMath"],
 "."
}], "Notes",
 CellID->26390],

Cell[TextData[{
 "A ",
 Cell[BoxData[
  ButtonBox["Method",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Method"]], "InlineFormula"],
 " option can be given, with typical possible values being ",
 Cell[BoxData[
  ButtonBox["Automatic",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<SylvesterMatrix\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<BezoutMatrix\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<Subresultants\>\""], "InlineFormula"],
 " and ",
 Cell[BoxData["\"\<Modular\>\""], "InlineFormula"],
 "."
}], "Notes",
 CellID->22483706]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->186195502],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(1)", "ExampleCount"]
}], "ExampleSection",
 CellID->523908624],

Cell["Discriminant of a quadratic:", "ExampleText",
 CellID->5772],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Discriminant", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"a", " ", 
     RowBox[{"x", "^", "2"}]}], "+", 
    RowBox[{"b", " ", "x"}], "+", "c"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->5214],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["b", "2"], "-", 
  RowBox[{"4", " ", "a", " ", "c"}]}]], "Output",
 ImageSize->{57, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->65738809]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->6794],

Cell["Discriminant of a polynomial with numeric coefficients:", "ExampleText",
 CellID->81871384],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Discriminant", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"x", "^", "10"}], "-", 
    RowBox[{"5", 
     RowBox[{"x", "^", "7"}]}], "-", 
    RowBox[{"3", " ", "x"}], "+", "9"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->115512854],

Cell[BoxData["177945374758153510836"], "Output",
 ImageSize->{164, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->585760718]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->330549332],

Cell["Discriminant of a general cubic:", "ExampleText",
 CellID->15616],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Discriminant", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"a", " ", 
     RowBox[{"x", "^", "3"}]}], "+", 
    RowBox[{"b", " ", 
     RowBox[{"x", "^", "2"}]}], "+", 
    RowBox[{"c", " ", "x"}], " ", "+", "d"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->30175],

Cell[BoxData[
 RowBox[{
  RowBox[{
   SuperscriptBox["b", "2"], " ", 
   SuperscriptBox["c", "2"]}], "-", 
  RowBox[{"4", " ", "a", " ", 
   SuperscriptBox["c", "3"]}], "-", 
  RowBox[{"4", " ", 
   SuperscriptBox["b", "3"], " ", "d"}], "+", 
  RowBox[{"18", " ", "a", " ", "b", " ", "c", " ", "d"}], "-", 
  RowBox[{"27", " ", 
   SuperscriptBox["a", "2"], " ", 
   SuperscriptBox["d", "2"]}]}]], "Output",
 ImageSize->{247, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->63150170]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->4959],

Cell["Discriminant of a general quintic:", "ExampleText",
 CellID->10527],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Discriminant", "[", 
  RowBox[{
   RowBox[{"Sum", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Subscript", "[", 
       RowBox[{"a", ",", "i"}], "]"}], 
      RowBox[{"x", "^", "i"}]}], ",", 
     RowBox[{"{", 
      RowBox[{"i", ",", "5"}], "}"}]}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2281],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{462, 54},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->204918483]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->938],

Cell["Discriminants are squares of differences of roots:", "ExampleText",
 CellID->16683],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Discriminant", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{"x", "-", "a"}], ")"}], 
    RowBox[{"(", 
     RowBox[{"x", "-", "b"}], ")"}], 
    RowBox[{"(", 
     RowBox[{"x", "-", "c"}], ")"}], 
    RowBox[{"(", 
     RowBox[{"x", "-", "d"}], ")"}]}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->25910],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"a", "-", "b"}], ")"}], "2"], " ", 
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"a", "-", "c"}], ")"}], "2"], " ", 
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"b", "-", "c"}], ")"}], "2"], " ", 
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"a", "-", "d"}], ")"}], "2"], " ", 
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"b", "-", "d"}], ")"}], "2"], " ", 
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"c", "-", "d"}], ")"}], "2"]}]], "Output",
 ImageSize->{296, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->431397064]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Options",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->497219442],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Method",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->349944861],

Cell["\<\
This compares timings of the available methods of discriminant computation:\
\>", "ExampleText",
 CellID->178377156],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"Timing", "[", 
     RowBox[{
      RowBox[{"Discriminant", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"x", "^", "100"}], "-", 
         RowBox[{"2", 
          RowBox[{"x", "^", "77"}]}], "+", 
         RowBox[{"3", "x"}], "+", "4"}], ",", "x", ",", 
        RowBox[{"Method", "\[Rule]", "#"}]}], "]"}], "//", "Short"}], "]"}], 
    "&"}], "/@", 
   RowBox[{"{", 
    RowBox[{
    "Automatic", ",", "Modular", " ", ",", "Subresultants", ",", " ", 
     "\"\<BezoutMatrix\>\"", ",", "\"\<SylvesterMatrix\>\""}], "}"}]}], "//", 
  "ColumnForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->130170461],

Cell[BoxData[
 InterpretationBox[GridBox[{
    {
     RowBox[{"{", 
      RowBox[{"0.01599999999998089`", ",", 
       RowBox[{"54228425007162214874034238415994564", 
        RowBox[{"\[LeftSkeleton]", "190", "\[RightSkeleton]"}], 
        "11563551931558053046829820246441557"}]}], "}"}]},
    {
     RowBox[{"{", 
      RowBox[{"1.8179902028236938`*^-14", ",", 
       RowBox[{"54228425007162214874034238415994564", 
        RowBox[{"\[LeftSkeleton]", "190", "\[RightSkeleton]"}], 
        "11563551931558053046829820246441557"}]}], "}"}]},
    {
     RowBox[{"{", 
      RowBox[{"1.8179902028236938`*^-14", ",", 
       RowBox[{"54228425007162214874034238415994564", 
        RowBox[{"\[LeftSkeleton]", "190", "\[RightSkeleton]"}], 
        "11563551931558053046829820246441557"}]}], "}"}]},
    {
     RowBox[{"{", 
      RowBox[{"0.2340000000000182`", ",", 
       RowBox[{"54228425007162214874034238415994564", 
        RowBox[{"\[LeftSkeleton]", "190", "\[RightSkeleton]"}], 
        "11563551931558053046829820246441557"}]}], "}"}]},
    {
     RowBox[{"{", 
      RowBox[{"0.8439999999999809`", ",", 
       RowBox[{"54228425007162214874034238415994564", 
        RowBox[{"\[LeftSkeleton]", "190", "\[RightSkeleton]"}], 
        "11563551931558053046829820246441557"}]}], "}"}]}
   },
   BaselinePosition->{Baseline, {1, 1}},
   GridBoxAlignment->{
    "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
     "RowsIndexed" -> {}}],
  ColumnForm[{{0.01599999999998089, 
     Short[
     5422842500716221487403423841599456404323863786540139830758371140058200324\
016563085794136240107568219605209944485701994483331550866696814350096111028127\
494170413103491054045854055376241847354157800494814400189588851459052847901156\
3551931558053046829820246441557]}, {1.8179902028236938`*^-14, 
     Short[
     5422842500716221487403423841599456404323863786540139830758371140058200324\
016563085794136240107568219605209944485701994483331550866696814350096111028127\
494170413103491054045854055376241847354157800494814400189588851459052847901156\
3551931558053046829820246441557]}, {1.8179902028236938`*^-14, 
     Short[
     5422842500716221487403423841599456404323863786540139830758371140058200324\
016563085794136240107568219605209944485701994483331550866696814350096111028127\
494170413103491054045854055376241847354157800494814400189588851459052847901156\
3551931558053046829820246441557]}, {0.2340000000000182, 
     Short[
     5422842500716221487403423841599456404323863786540139830758371140058200324\
016563085794136240107568219605209944485701994483331550866696814350096111028127\
494170413103491054045854055376241847354157800494814400189588851459052847901156\
3551931558053046829820246441557]}, {0.8439999999999809, 
     Short[
     5422842500716221487403423841599456404323863786540139830758371140058200324\
016563085794136240107568219605209944485701994483331550866696814350096111028127\
494170413103491054045854055376241847354157800494814400189588851459052847901156\
3551931558053046829820246441557]}}],
  Editable->False]], "Output",
 ImageSize->{711, 87},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->323708157]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"Timing", "[", 
     RowBox[{
      RowBox[{"Discriminant", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"a", " ", 
          RowBox[{"x", "^", "10"}]}], "+", 
         RowBox[{"b", " ", 
          RowBox[{"x", "^", "5"}]}], "+", 
         RowBox[{
          RowBox[{"(", 
           RowBox[{"a", "+", "b"}], ")"}], "x"}], "+", 
         RowBox[{"a", " ", "b", " ", "c"}]}], ",", "x", ",", 
        RowBox[{"Method", "\[Rule]", "#"}]}], "]"}], "//", "Short"}], "]"}], 
    "&"}], "/@", 
   RowBox[{"{", 
    RowBox[{
    "Automatic", ",", "Modular", " ", ",", "Subresultants", ",", " ", 
     "\"\<BezoutMatrix\>\"", ",", "\"\<SylvesterMatrix\>\""}], "}"}]}], "//", 
  "ColumnForm"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->95396236],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWs1u00AQdvybUkhTeuFS1BPPgOAOJx4iqkDtAVG1vdLXgKdB3PNQg9fx
NOPxrHdsZyuhTqTYmf2Z3f1mdjxfvF8291dfv2/ury83F59vNzdX15d3F59+
3NZF2SJJkm39/fYmqX9D/av5bgHqe15f01r8096LpjRz18y1aj7JXyanM1q9
Y/IvdS83v9+q1k52Iq6uILK7u/qilWm/irUrPPW7AXcf1Ofuy/aeknGw3I23
ZvqOpdXkjzLTwmebMnmpwqYpLegYuKAUx+jVyn34fBDtaoaOZRcBhLpfQvqU
wRZclld5KsyFW7Pfi/tIynpVolW47WTdR2JpKWkk8uloPJxcJtpZ5W1r3AXS
HLhdZU0nsNsXbkUp8cIXAW05a88RWYDe858S47D/D+wS5Y5y4+TtCnKFV1fd
lbS9MqnVxNgiI0xjM9VxAvtIlAmWyHvIF14cevj3LBbY18EYoY/gHvtFiuex
vFraiSg/jqDW4PdiubSEvVcWAoLnTJZxf31wXHXPtlcj+vrXPwe7qX2bmVes
tK17e0DMcUceQzcPw1ixBjmPq2D/PKD1S0/5KrAeGgHPVDN/kFoR+YNei2de
H0UNiMxPIPkbafUeaF6/Hbw+Ly6AsSQU4YwTyNlH6FlibABRicsGXN9SwG49
GgdjATov5TaehrRxge48hZyeaHNPbGMFADFYgZjDM0ZmrGA+rgO5tbGCAcw5
K8DnBNp7BcYKuhqMFWh9irMC9B2s92X7xgam5i7GCeJzAp017D3BfIYQE2lj
CMMrdyO7eGm8AC0VH3Mqn4kYGS8YiyvXtRqxRmMENItDixsj4BqMEWh9yndm
iGdx9p4A2oUMMwNdVm/MAPGNywxi/qNtzOCpkH6+zCBP/Lm+vSM4RDQ/klAL
vv21dwQhXHlEGrCsvSMYjTnuyJfQzXsw8zdGwDUYI9D6lJ0c8mM0hRHYySGp
l50csuzfsv+xOPd3tZ0cihHR7eQQeHefnRwa29dODoVm/j+xgmTxDyHYTTA=
\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{203, 102},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->80722048]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Modulus",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->7240],

Cell["\<\
By default the discriminant is computed over the rational numbers:\
\>", "ExampleText",
 CellID->133742431],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Discriminant", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"a", " ", 
     RowBox[{"x", "^", "3"}]}], "+", 
    RowBox[{"b", " ", 
     RowBox[{"x", "^", "2"}]}], "+", 
    RowBox[{"c", " ", "x"}], "+", "d"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2515],

Cell[BoxData[
 RowBox[{
  RowBox[{
   SuperscriptBox["b", "2"], " ", 
   SuperscriptBox["c", "2"]}], "-", 
  RowBox[{"4", " ", "a", " ", 
   SuperscriptBox["c", "3"]}], "-", 
  RowBox[{"4", " ", 
   SuperscriptBox["b", "3"], " ", "d"}], "+", 
  RowBox[{"18", " ", "a", " ", "b", " ", "c", " ", "d"}], "-", 
  RowBox[{"27", " ", 
   SuperscriptBox["a", "2"], " ", 
   SuperscriptBox["d", "2"]}]}]], "Output",
 ImageSize->{247, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->235245789]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->180435940],

Cell["\<\
Compute the discriminant of the same polynomial over the integers modulo 2:\
\>", "ExampleText",
 CellID->3114652],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Discriminant", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"a", " ", 
     RowBox[{"x", "^", "3"}]}], "+", 
    RowBox[{"b", " ", 
     RowBox[{"x", "^", "2"}]}], "+", 
    RowBox[{"c", " ", "x"}], "+", "d"}], ",", "x", ",", 
   RowBox[{"Modulus", "->", "2"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->23740],

Cell[BoxData[
 RowBox[{
  RowBox[{
   SuperscriptBox["b", "2"], " ", 
   SuperscriptBox["c", "2"]}], "+", 
  RowBox[{
   SuperscriptBox["a", "2"], " ", 
   SuperscriptBox["d", "2"]}]}]], "Output",
 ImageSize->{73, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->371034894]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->243356014],

Cell["\<\
Compute the discriminant of the same polynomial over the integers modulo 3:\
\>", "ExampleText",
 CellID->19707],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Discriminant", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"a", " ", 
     RowBox[{"x", "^", "3"}]}], "+", 
    RowBox[{"b", " ", 
     RowBox[{"x", "^", "2"}]}], "+", 
    RowBox[{"c", " ", "x"}], "+", "d"}], ",", "x", ",", 
   RowBox[{"Modulus", "->", "3"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->18445],

Cell[BoxData[
 RowBox[{
  RowBox[{
   SuperscriptBox["b", "2"], " ", 
   SuperscriptBox["c", "2"]}], "+", 
  RowBox[{"2", " ", "a", " ", 
   SuperscriptBox["c", "3"]}], "+", 
  RowBox[{"2", " ", 
   SuperscriptBox["b", "3"], " ", "d"}]}]], "Output",
 ImageSize->{123, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->152955585]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->8667],

Cell["Decide whether a polynomial has multiple roots:", "ExampleText",
 CellID->144731868],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Discriminant", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"x", "^", "11"}], "-", 
    RowBox[{"2", 
     RowBox[{"x", "^", "10"}]}], "+", 
    RowBox[{"x", "^", "9"}], "-", 
    RowBox[{"2", 
     RowBox[{"x", "^", "3"}]}], "+", 
    RowBox[{"11", 
     RowBox[{"x", "^", "2"}]}], "-", 
    RowBox[{"16", "x"}], "+", "7"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->53185853],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->652410419]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FactorSquareFree", "[", 
  RowBox[{
   RowBox[{"x", "^", "11"}], "-", 
   RowBox[{"2", 
    RowBox[{"x", "^", "10"}]}], "+", 
   RowBox[{"x", "^", "9"}], "-", 
   RowBox[{"2", 
    RowBox[{"x", "^", "3"}]}], "+", 
   RowBox[{"11", 
    RowBox[{"x", "^", "2"}]}], "-", 
   RowBox[{"16", "x"}], "+", "7"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->483856105],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "x"}], ")"}], "2"], " ", 
  RowBox[{"(", 
   RowBox[{"7", "-", 
    RowBox[{"2", " ", "x"}], "+", 
    SuperscriptBox["x", "9"]}], ")"}]}]], "Output",
 ImageSize->{137, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->319048439]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Discriminant", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"x", "^", "11"}], "-", 
    RowBox[{"2", 
     RowBox[{"x", "^", "10"}]}], "+", 
    RowBox[{"x", "^", "9"}], "-", 
    RowBox[{"2", 
     RowBox[{"x", "^", "3"}]}], "+", 
    RowBox[{"11", 
     RowBox[{"x", "^", "2"}]}], "-", 
    RowBox[{"16", "x"}], "-", "7"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->31950674],

Cell[BoxData["813583724618463631662004"], "Output",
 ImageSize->{187, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->174984856]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FactorSquareFree", "[", 
  RowBox[{
   RowBox[{"x", "^", "11"}], "-", 
   RowBox[{"2", 
    RowBox[{"x", "^", "10"}]}], "+", 
   RowBox[{"x", "^", "9"}], "-", 
   RowBox[{"2", 
    RowBox[{"x", "^", "3"}]}], "+", 
   RowBox[{"11", 
    RowBox[{"x", "^", "2"}]}], "-", 
   RowBox[{"16", "x"}], "-", "7"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->922926328],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "7"}], "-", 
  RowBox[{"16", " ", "x"}], "+", 
  RowBox[{"11", " ", 
   SuperscriptBox["x", "2"]}], "-", 
  RowBox[{"2", " ", 
   SuperscriptBox["x", "3"]}], "+", 
  SuperscriptBox["x", "9"], "-", 
  RowBox[{"2", " ", 
   SuperscriptBox["x", "10"]}], "+", 
  SuperscriptBox["x", "11"]}]], "Output",
 ImageSize->{228, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->399962463]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->179977997],

Cell["Find the condition for a cubic to have multiple roots:", "ExampleText",
 CellID->13911],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Solve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Discriminant", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "^", "3"}], "+", "x", "+", " ", "c"}], " ", ",", "x"}], 
     "]"}], "==", "0"}], ",", "c"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->17740],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"c", "\[Rule]", 
     RowBox[{"-", 
      FractionBox[
       RowBox[{"2", " ", "\[ImaginaryI]"}], 
       RowBox[{"3", " ", 
        SqrtBox["3"]}]]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"c", "\[Rule]", 
     FractionBox[
      RowBox[{"2", " ", "\[ImaginaryI]"}], 
      RowBox[{"3", " ", 
       SqrtBox["3"]}]]}], "}"}]}], "}"}]], "Output",
 ImageSize->{187, 34},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->95297049]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FactorSquareFree", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"x", "^", "3"}], "+", "x", "+", " ", "c"}], "/.", "%"}], ",", 
   RowBox[{"Extension", "\[Rule]", "Automatic"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->612123172],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"-", 
     FractionBox["1", "27"]}], " ", "\[ImaginaryI]", " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"2", " ", 
       SqrtBox["3"]}], "-", 
      RowBox[{"3", " ", "\[ImaginaryI]", " ", "x"}]}], ")"}], " ", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{
       SqrtBox["3"], "+", 
       RowBox[{"3", " ", "\[ImaginaryI]", " ", "x"}]}], ")"}], "2"]}], ",", 
   RowBox[{
    FractionBox["1", "27"], " ", "\[ImaginaryI]", " ", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{
       SqrtBox["3"], "-", 
       RowBox[{"3", " ", "\[ImaginaryI]", " ", "x"}]}], ")"}], "2"], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"2", " ", 
       SqrtBox["3"]}], "+", 
      RowBox[{"3", " ", "\[ImaginaryI]", " ", "x"}]}], ")"}]}]}], 
  "}"}]], "Output",
 ImageSize->{453, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->282383757]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->14268],

Cell["\<\
The discriminant is zero if and only if the polynomial has multiple roots:\
\>", "ExampleText",
 CellID->39217662],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Discriminant", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{"x", "-", "1"}], ")"}], 
    RowBox[{"(", 
     RowBox[{"x", "-", "2"}], ")"}], 
    RowBox[{"(", 
     RowBox[{"x", "-", "3"}], ")"}]}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->91078070],

Cell[BoxData["4"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->276254107]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Discriminant", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{"x", "-", "1"}], ")"}], 
    RowBox[{"(", 
     RowBox[{"x", "-", "2"}], ")"}], 
    RowBox[{"(", 
     RowBox[{"x", "-", "1"}], ")"}]}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->224854246],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->222549095]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->677174966],

Cell[TextData[{
 "The discriminant can be represented in terms of roots as ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"D", "(", "f", ")"}], "==", 
    RowBox[{
     SuperscriptBox[
      SubscriptBox["a", "n"], 
      RowBox[{
       RowBox[{"2", "n"}], "-", "2"}]], 
     RowBox[{
      UnderscriptBox["\[Product]", 
       RowBox[{"1", "\[LessEqual]", "i", "<", "j", "\[LessEqual]", "n"}]], 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{
         SubscriptBox["r", "i"], "-", 
         SubscriptBox["r", "j"]}], ")"}], "2"]}]}]}], TraditionalForm]], 
  "InlineMath"],
 ":"
}], "ExampleText",
 CellID->7917],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"x", "/.", 
  RowBox[{"Solve", "[", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"a", " ", 
       RowBox[{"x", "^", "2"}]}], "+", 
      RowBox[{"b", " ", "x"}], "+", "c"}], "==", "0"}], ",", "x"}], 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->21870],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox[
    RowBox[{
     RowBox[{"-", "b"}], "-", 
     SqrtBox[
      RowBox[{
       SuperscriptBox["b", "2"], "-", 
       RowBox[{"4", " ", "a", " ", "c"}]}]]}], 
    RowBox[{"2", " ", "a"}]], ",", 
   FractionBox[
    RowBox[{
     RowBox[{"-", "b"}], "+", 
     SqrtBox[
      RowBox[{
       SuperscriptBox["b", "2"], "-", 
       RowBox[{"4", " ", "a", " ", "c"}]}]]}], 
    RowBox[{"2", " ", "a"}]]}], "}"}]], "Output",
 ImageSize->{235, 41},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->623992184]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Expand", "[", 
  RowBox[{
   RowBox[{"a", "^", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"2", " ", "2"}], "-", "2"}], ")"}]}], 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"%", "[", 
       RowBox[{"[", "1", "]"}], "]"}], "-", 
      RowBox[{"%", "[", 
       RowBox[{"[", "2", "]"}], "]"}]}], ")"}], "^", "2"}]}], "]"}]], "Input",\

 CellLabel->"In[2]:=",
 CellID->4643],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["b", "2"], "-", 
  RowBox[{"4", " ", "a", " ", "c"}]}]], "Output",
 ImageSize->{57, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->25100635]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Discriminant", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"a", " ", 
     RowBox[{"x", "^", "2"}]}], "+", 
    RowBox[{"b", " ", "x"}], "+", "c"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->22022],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["b", "2"], "-", 
  RowBox[{"4", " ", "a", " ", "c"}]}]], "Output",
 ImageSize->{57, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->20566072]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->25196393],

Cell[TextData[{
 "Equation ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"-", "1"}], ")"}], 
      RowBox[{
       RowBox[{"n", "(", 
        RowBox[{"n", "-", "1"}], ")"}], "/", "2"}]], 
     SubscriptBox["a", "n"], 
     RowBox[{"D", "(", "f", ")"}]}], "==", 
    RowBox[{"R", 
     RowBox[{"(", 
      RowBox[{"f", ",", 
       RowBox[{"f", "'"}]}], ")"}]}]}], TraditionalForm]], "InlineMath"],
 " relates ",
 Cell[BoxData[
  ButtonBox["Discriminant",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Discriminant"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["Resultant",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Resultant"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->182035434],

Cell[BoxData[
 RowBox[{
  RowBox[{"f", "=", 
   RowBox[{
    RowBox[{"3", 
     RowBox[{"x", "^", "7"}]}], "-", 
    RowBox[{"5", "x"}], "+", "4"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->339245623],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Resultant", "[", 
  RowBox[{"f", ",", 
   RowBox[{"D", "[", 
    RowBox[{"f", ",", "x"}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->547246616],

Cell[BoxData["4720053663936"], "Output",
 ImageSize->{104, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->413265142]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"(", 
    RowBox[{"-", "1"}], ")"}], "^", 
   RowBox[{"(", 
    RowBox[{"7", " ", 
     RowBox[{"6", "/", "2"}]}], ")"}]}], 
  RowBox[{"Coefficient", "[", 
   RowBox[{"f", ",", "x", ",", "7"}], "]"}], 
  RowBox[{"Discriminant", "[", 
   RowBox[{"f", ",", "x"}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->57279941],

Cell[BoxData["4720053663936"], "Output",
 ImageSize->{104, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->524676272]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->21466],

Cell["Using exact coefficients, this indicates no common root:", "ExampleText",
 CellID->261083893],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Discriminant", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{"x", "-", "1", "+", 
      RowBox[{"10", "^", 
       RowBox[{"(", 
        RowBox[{"-", "17"}], ")"}]}]}], ")"}], 
    RowBox[{"(", 
     RowBox[{"x", "-", "1"}], ")"}]}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->54611027],

Cell[BoxData[
 FractionBox["1", "10000000000000000000000000000000000"]], "Output",
 ImageSize->{277, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->186683182]
}, Open  ]],

Cell["\<\
With approximate coefficients, this does indicate a common root:\
\>", "ExampleText",
 CellID->234994314],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Discriminant", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{"x", "-", "1.", "+", 
      RowBox[{"10", "^", 
       RowBox[{"(", 
        RowBox[{"-", "17"}], ")"}]}]}], ")"}], 
    RowBox[{"(", 
     RowBox[{"x", "-", "1"}], ")"}]}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->342344003],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->51127383]
}, Open  ]],

Cell["in this case, using higher precision resolves the problem:", \
"ExampleText",
 CellID->101502327],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Discriminant", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{"x", "-", "1`20", "+", 
      RowBox[{"10", "^", 
       RowBox[{"(", 
        RowBox[{"-", "17"}], ")"}]}]}], ")"}], 
    RowBox[{"(", 
     RowBox[{"x", "-", "1"}], ")"}]}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->374741927],

Cell[BoxData["1.`2.3979400086720384*^-34"], "Output",
 ImageSize->{73, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->278670]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->5331],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["Resultant",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Resultant"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Det",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Det"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Tr",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Tr"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SquareFreeQ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SquareFreeQ"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->1410]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->829617021],

Cell[TextData[ButtonBox["Polynomial Algebra",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/PolynomialAlgebra"]], "MoreAbout",
 CellID->324273859],

Cell[TextData[ButtonBox["Polynomial Systems",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/PolynomialSystems"]], "MoreAbout",
 CellID->540518664],

Cell[TextData[ButtonBox["New in 6.0: Symbolic Computation",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60AlgebraicComputing"]], "MoreAbout",
 CellID->39886491],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->133386361]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Discriminant - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 38, 28.2896126}", "context" -> "System`", 
    "keywords" -> {
     "discriminant", "multiple roots", "resultant", "Sylvester matrix", 
      "Bezout matrix", "subresultants", "modular"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "Discriminant[poly, var] computes the discriminant of the polynomial poly \
with respect to the variable var. Discriminant[poly, var, Modulus -> p] \
computes the discriminant modulo p.", "synonyms" -> {}, "title" -> 
    "Discriminant", "type" -> "Symbol", "uri" -> "ref/Discriminant"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[7205, 222, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->186195502]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 42631, 1510}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 1945, 43, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2568, 70, 1577, 37, 70, "ObjectNameGrid"],
Cell[4148, 109, 1345, 42, 70, "Usage",
 CellID->11643]
}, Open  ]],
Cell[CellGroupData[{
Cell[5530, 156, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6008, 173, 542, 21, 70, "Notes",
 CellID->26390],
Cell[6553, 196, 615, 21, 70, "Notes",
 CellID->22483706]
}, Closed]],
Cell[CellGroupData[{
Cell[7205, 222, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->186195502],
Cell[CellGroupData[{
Cell[7590, 236, 148, 5, 70, "ExampleSection",
 CellID->523908624],
Cell[7741, 243, 66, 1, 70, "ExampleText",
 CellID->5772],
Cell[CellGroupData[{
Cell[7832, 248, 236, 8, 28, "Input",
 CellID->5214],
Cell[8071, 258, 236, 8, 36, "Output",
 CellID->65738809]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[8356, 272, 215, 7, 70, "ExampleSection",
 CellID->6794],
Cell[8574, 281, 97, 1, 70, "ExampleText",
 CellID->81871384],
Cell[CellGroupData[{
Cell[8696, 286, 272, 9, 70, "Input",
 CellID->115512854],
Cell[8971, 297, 179, 5, 36, "Output",
 CellID->585760718]
}, Open  ]],
Cell[9165, 305, 125, 3, 70, "ExampleDelimiter",
 CellID->330549332],
Cell[9293, 310, 71, 1, 70, "ExampleText",
 CellID->15616],
Cell[CellGroupData[{
Cell[9389, 315, 303, 10, 70, "Input",
 CellID->30175],
Cell[9695, 327, 537, 17, 36, "Output",
 CellID->63150170]
}, Open  ]],
Cell[10247, 347, 120, 3, 70, "ExampleDelimiter",
 CellID->4959],
Cell[10370, 352, 73, 1, 70, "ExampleText",
 CellID->10527],
Cell[CellGroupData[{
Cell[10468, 357, 351, 12, 70, "Input",
 CellID->2281],
Cell[10822, 371, 2652, 48, 70, "Output",
 Evaluatable->False,
 CellID->204918483]
}, Open  ]],
Cell[13489, 422, 119, 3, 70, "ExampleDelimiter",
 CellID->938],
Cell[13611, 427, 89, 1, 70, "ExampleText",
 CellID->16683],
Cell[CellGroupData[{
Cell[13725, 432, 360, 13, 70, "Input",
 CellID->25910],
Cell[14088, 447, 659, 24, 36, "Output",
 CellID->431397064]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[14796, 477, 222, 7, 70, "ExampleSection",
 CellID->497219442],
Cell[CellGroupData[{
Cell[15043, 488, 227, 7, 70, "ExampleSubsection",
 CellID->349944861],
Cell[15273, 497, 126, 3, 70, "ExampleText",
 CellID->178377156],
Cell[CellGroupData[{
Cell[15424, 504, 670, 21, 70, "Input",
 CellID->130170461],
Cell[16097, 527, 3197, 68, 108, "Output",
 CellID->323708157]
}, Open  ]],
Cell[CellGroupData[{
Cell[19331, 600, 806, 25, 70, "Input",
 CellID->95396236],
Cell[20140, 627, 1249, 25, 70, "Output",
 Evaluatable->False,
 CellID->80722048]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[21438, 658, 223, 7, 70, "ExampleSubsection",
 CellID->7240],
Cell[21664, 667, 117, 3, 70, "ExampleText",
 CellID->133742431],
Cell[CellGroupData[{
Cell[21806, 674, 297, 10, 70, "Input",
 CellID->2515],
Cell[22106, 686, 538, 17, 36, "Output",
 CellID->235245789]
}, Open  ]],
Cell[22659, 706, 125, 3, 70, "ExampleDelimiter",
 CellID->180435940],
Cell[22787, 711, 124, 3, 70, "ExampleText",
 CellID->3114652],
Cell[CellGroupData[{
Cell[22936, 718, 339, 11, 70, "Input",
 CellID->23740],
Cell[23278, 731, 326, 12, 36, "Output",
 CellID->371034894]
}, Open  ]],
Cell[23619, 746, 125, 3, 70, "ExampleDelimiter",
 CellID->243356014],
Cell[23747, 751, 122, 3, 70, "ExampleText",
 CellID->19707],
Cell[CellGroupData[{
Cell[23894, 758, 339, 11, 70, "Input",
 CellID->18445],
Cell[24236, 771, 380, 13, 36, "Output",
 CellID->152955585]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[24677, 791, 222, 7, 70, "ExampleSection",
 CellID->8667],
Cell[24902, 800, 90, 1, 70, "ExampleText",
 CellID->144731868],
Cell[CellGroupData[{
Cell[25017, 805, 416, 14, 70, "Input",
 CellID->53185853],
Cell[25436, 821, 158, 5, 36, "Output",
 CellID->652410419]
}, Open  ]],
Cell[CellGroupData[{
Cell[25631, 831, 388, 13, 70, "Input",
 CellID->483856105],
Cell[26022, 846, 383, 14, 39, "Output",
 CellID->319048439]
}, Open  ]],
Cell[CellGroupData[{
Cell[26442, 865, 416, 14, 70, "Input",
 CellID->31950674],
Cell[26861, 881, 182, 5, 36, "Output",
 CellID->174984856]
}, Open  ]],
Cell[CellGroupData[{
Cell[27080, 891, 388, 13, 70, "Input",
 CellID->922926328],
Cell[27471, 906, 468, 16, 36, "Output",
 CellID->399962463]
}, Open  ]],
Cell[27954, 925, 125, 3, 70, "ExampleDelimiter",
 CellID->179977997],
Cell[28082, 930, 93, 1, 70, "ExampleText",
 CellID->13911],
Cell[CellGroupData[{
Cell[28200, 935, 293, 10, 70, "Input",
 CellID->17740],
Cell[28496, 947, 552, 20, 55, "Output",
 CellID->95297049]
}, Open  ]],
Cell[CellGroupData[{
Cell[29085, 972, 272, 8, 70, "Input",
 CellID->612123172],
Cell[29360, 982, 968, 33, 51, "Output",
 CellID->282383757]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[30377, 1021, 233, 7, 70, "ExampleSection",
 CellID->14268],
Cell[30613, 1030, 124, 3, 70, "ExampleText",
 CellID->39217662],
Cell[CellGroupData[{
Cell[30762, 1037, 307, 11, 70, "Input",
 CellID->91078070],
Cell[31072, 1050, 158, 5, 36, "Output",
 CellID->276254107]
}, Open  ]],
Cell[CellGroupData[{
Cell[31267, 1060, 308, 11, 70, "Input",
 CellID->224854246],
Cell[31578, 1073, 158, 5, 36, "Output",
 CellID->222549095]
}, Open  ]],
Cell[31751, 1081, 125, 3, 70, "ExampleDelimiter",
 CellID->677174966],
Cell[31879, 1086, 632, 22, 70, "ExampleText",
 CellID->7917],
Cell[CellGroupData[{
Cell[32536, 1112, 293, 11, 70, "Input",
 CellID->21870],
Cell[32832, 1125, 608, 23, 62, "Output",
 CellID->623992184]
}, Open  ]],
Cell[CellGroupData[{
Cell[33477, 1153, 414, 16, 70, "Input",
 CellID->4643],
Cell[33894, 1171, 236, 8, 36, "Output",
 CellID->25100635]
}, Open  ]],
Cell[CellGroupData[{
Cell[34167, 1184, 237, 8, 70, "Input",
 CellID->22022],
Cell[34407, 1194, 236, 8, 36, "Output",
 CellID->20566072]
}, Open  ]],
Cell[34658, 1205, 124, 3, 70, "ExampleDelimiter",
 CellID->25196393],
Cell[34785, 1210, 777, 30, 70, "ExampleText",
 CellID->182035434],
Cell[35565, 1242, 210, 8, 70, "Input",
 CellID->339245623],
Cell[CellGroupData[{
Cell[35800, 1254, 192, 6, 70, "Input",
 CellID->547246616],
Cell[35995, 1262, 171, 5, 36, "Output",
 CellID->413265142]
}, Open  ]],
Cell[CellGroupData[{
Cell[36203, 1272, 368, 13, 70, "Input",
 CellID->57279941],
Cell[36574, 1287, 171, 5, 36, "Output",
 CellID->524676272]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[36794, 1298, 226, 7, 70, "ExampleSection",
 CellID->21466],
Cell[37023, 1307, 99, 1, 70, "ExampleText",
 CellID->261083893],
Cell[CellGroupData[{
Cell[37147, 1312, 342, 12, 70, "Input",
 CellID->54611027],
Cell[37492, 1326, 213, 6, 51, "Output",
 CellID->186683182]
}, Open  ]],
Cell[37720, 1335, 115, 3, 70, "ExampleText",
 CellID->234994314],
Cell[CellGroupData[{
Cell[37860, 1342, 344, 12, 70, "Input",
 CellID->342344003],
Cell[38207, 1356, 157, 5, 36, "Output",
 CellID->51127383]
}, Open  ]],
Cell[38379, 1364, 103, 2, 70, "ExampleText",
 CellID->101502327],
Cell[CellGroupData[{
Cell[38507, 1370, 346, 12, 70, "Input",
 CellID->374741927],
Cell[38856, 1384, 180, 5, 36, "Output",
 CellID->278670]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[39097, 1396, 310, 9, 70, "SeeAlsoSection",
 CellID->5331],
Cell[39410, 1407, 926, 35, 70, "SeeAlso",
 CellID->1410]
}, Open  ]],
Cell[CellGroupData[{
Cell[40373, 1447, 319, 9, 70, "MoreAboutSection",
 CellID->829617021],
Cell[40695, 1458, 147, 3, 70, "MoreAbout",
 CellID->324273859],
Cell[40845, 1463, 147, 3, 70, "MoreAbout",
 CellID->540518664],
Cell[40995, 1468, 168, 3, 70, "MoreAbout",
 CellID->39886491],
Cell[41166, 1473, 179, 3, 70, "MoreAbout",
 CellID->133386361]
}, Open  ]],
Cell[41360, 1479, 27, 0, 70, "History"],
Cell[41390, 1481, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

