(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     47601,       1620]
NotebookOptionsPosition[     39660,       1346]
NotebookOutlinePosition[     41188,       1387]
CellTagsIndexPosition[     41102,       1382]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Some Mathematical Functions" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/SomeMathematicalFunctions"], 
          "The Limits of Mathematica" :> 
          Documentation`HelpLookup["paclet:tutorial/TheLimitsOfMathematica"], 
          "Integer and Number\[Hyphen]Theoretical Functions" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/IntegerAndNumberTheoreticalFunctions"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Some Mathematical Functions\"\>", 
       2->"\<\"The Limits of Mathematica\"\>", 
       3->"\<\"Integer and Number\[Hyphen]Theoretical Functions\"\>"}, \
"\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"IntegerExponent" :> 
          Documentation`HelpLookup["paclet:ref/IntegerExponent"], "Prime" :> 
          Documentation`HelpLookup["paclet:ref/Prime"], "PrimeQ" :> 
          Documentation`HelpLookup["paclet:ref/PrimeQ"], "PrimePowerQ" :> 
          Documentation`HelpLookup["paclet:ref/PrimePowerQ"], "SquareFreeQ" :> 
          Documentation`HelpLookup["paclet:ref/SquareFreeQ"], "Divisors" :> 
          Documentation`HelpLookup["paclet:ref/Divisors"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"IntegerExponent\"\>", 
       2->"\<\"Prime\"\>", 3->"\<\"PrimeQ\"\>", 4->"\<\"PrimePowerQ\"\>", 
       5->"\<\"SquareFreeQ\"\>", 6->"\<\"Divisors\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Cryptographic Number Theory" :> 
          Documentation`HelpLookup["paclet:guide/CryptographicNumberTheory"], 
          "Integer Functions" :> 
          Documentation`HelpLookup["paclet:guide/IntegerFunctions"], 
          "Mathematical Functions" :> 
          Documentation`HelpLookup["paclet:guide/MathematicalFunctions"], 
          "Number Theoretic Functions" :> 
          Documentation`HelpLookup["paclet:guide/NumberTheoreticFunctions"], 
          "Number Theory" :> 
          Documentation`HelpLookup["paclet:guide/NumberTheory"], 
          "Prime Numbers" :> 
          Documentation`HelpLookup["paclet:guide/PrimeNumbers"], 
          "New in 6.0: Number Theory & Integer Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60NumberTheoryAndIntegerFunctions"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Cryptographic Number Theory\"\>", 
       2->"\<\"Integer Functions\"\>", 3->"\<\"Mathematical Functions\"\>", 
       4->"\<\"Number Theoretic Functions\"\>", 5->"\<\"Number Theory\"\>", 
       6->"\<\"Prime Numbers\"\>", 
       7->"\<\"New in 6.0: Number Theory & Integer Functions\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["FactorInteger", "ObjectName"], Cell[BoxData[
     TooltipBox[
      TogglerBox[Dynamic[
        FrontEnd`CurrentValue[
         FrontEnd`EvaluationNotebook[], {
         TaggingRules, "ModificationHighlight"}]], {True->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVVly2zAMDQhxEylSsizLaxJn73KafvQAuUDv/1eApJuosceeTDAWF4B8
eFgk71///HoVV1dXK3p+0wNfIzdCLL4I6p1sYBYuPy3SI2AmRBpuaE/TPRsE
3Isg8mrB9l6ILQhhLkbd93MBuwVxAthdZwPr5wVVzBc8wAvMX8TFqPkqE8q6
rDCsKPjZ/g1guz+PKvLFfGsDU1R4A8zDhgLangX9x/WGM7A1qSQ7rraYkaLf
LHZ5FWY9H7rew/n4/6vWPKT2mXFBaH/NGejzqi/VnFEFJ2Rb1Lo+jtofllPD
RVJjdZLr51Eb/ED1K+QI109LcIg+sdSIWqa5g85L6MiNoRloTws+pdBKiFGl
mx3N0jcAlcsKeFivVkteVOgMSNsC18sYS/iIMdZoK43WSdQVK1SLlYy09bZ2
aEO+aoPiACNak0EBbtfLRIz9KJRcroaSKwE11BprpRHqiJ0BrxtJG4pLYqWc
RZO4BQwSbGT4rFivaLjjQdscjSvlih48Mu+WDisOBcB5ehgeDEZQlikkKh3b
KA1FMTDg9yFxLSe4XC37begHTDh5seS1I/gaW025Yo/yUFbmSalShzon1Kch
JTNn2hKS1hyWVIkf5n2LHEQgt0zeUNTkMaYrDdkMZdo3fJCDHx7hxzAC+8rs
XYBokTPfEJk6JVlZx3Yl0UnTYMPvSUt6zghf8dhIi5VDnwO+Hcb1MHK1msK+
oy5qkYS7wVC3cfiq8ojBcMtFMJXnDiIcLPFJClDFGkJBhXGEZV6VcCQFoey7
tyvBF6lw0uKH6nyQ5fqhrKx/Kxe8+3coucpGN4U5+RYux7vnhGO5P7hKenqz
xSO3ivjisf345fg5Pt1ScCRkkzR1U1Dur+NCec4LZ49Ynx9P04GmOvlnqcKZ
79BfJfcjkg==
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlLlKQ0EYRi/iA9gpWIgvIXbjM1iKWAjW2vgeZnWN+AgWZhEULCxELFyi
ZnfJavbeJOr/33xXcj8CNnY5EA7k3GWYmTuzq+uLa2OWZS38/MYt0DbDHH8Q
8uTB64bRNrmsQkaPRoQSGX1ZqZPR9/eEazL6klIlowf8wiUZ3ecV3snowYDw
SEafUmJk9HmlQh4cX4GM7vUId2T0VFIho28or2T0w5DQIKN3PoUaGX1FaZLR
0ymh6sz7hHLk9PMzoU5Gz6SFmvFsCbd9Xzj9qyc0nfe49xfuTyYUEzoQbmh8
+N8e37HTbVpk9MSzUCajbypFMvq0UiOjzyklMnqvKzRMJCwU+/vjzemTSpns
nh97feJmd0e4cvppTKiQ0bsdoW5ecorZDgpPtD6xqFA2fp+Qp/vtdUj1v797
Oqdwnb2Ov++fUVpk9PCJUCC7n4t5wz5xj9/+rj6ceXbPP86NbEb58/k56qP+
n/0bdGyYqQ==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}], False->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVdd22zAMDUBxSaJm5HgljrPTJN177/nWh35AfqD//1aAVGrrJLbbk+BI
FEmQFxcXlDQ9//3zHNfW1gZ0/6IbbsYOhahvCGrO9qHM/mnhmMMLoKv0RD6V
cPyZHTTcDI5NkYnQq4UQ0BdiAkKYZagl1OOw5fWtcgyvPp7AnefsGG0DBIeA
9RZVrNfcwBasb4nlbP9u4cbc/XD/zXcIgC2qYYazJTQiGSYHS0Hr8Yu5LV/h
ePvR5oMOKswAQ0M8R5OloFMWSNRE+IAVeALv38KX26wAV1uU5Ojv16PQy8q+
gMP+9ABW5C9KbkZi6qv19DGc/ID9b+CrZTjbbVagH3p91oIcL+GZ6JDNUet4
HpWUXxx5BaWZxRh1FQgH6JqoKcarF/2/dblez7IE0XmWGlFL/6ygchIqCmPo
CTSmDq9SaCUUhfI7K3pKlwJESZiAveHGxoA7ESYGpM2B62WMJXzEoojRRhpt
IlFHPKFyjGRBQ2fjBG0WttpMcYIFWhNAAXaHA0+M4yiUXK6UxJWAGmKNsdII
cYGVAadTSQPKS2KkEovGc8swk2ALhg8Tww1qdrjRNmSTtOUqHDhk3jktVpwK
QOLoZngwWICyTMFTqdhHMrQTDQPeazzXdgWXK+e4KV3AhH0US1Ergo8x16QV
R5QXZWWeJJW6qLNHPWq8mEFpS0hac1pSeX4YxjlyEhmFZfKGsqaIhd+Sks+Q
0i7lhZx8cwYPmx5wrMA+yaCwyMqnRCb2IiubsF9JTKRJMeX3JKd5VoS3OEyl
xShBFxLebXrDpsfVSlv2FZ2iHMn4NBg6bZy+ihxiZvjIFWAixyeIcLDNT1KC
qogha1Gh14NB6LXpSEpC2bm3y8O3FmHniF9U55INhnttz7pZuWDu79BqFZxJ
F2bhWzjo7Zx6HMvng6ukuztzvGJXa66NmF/+crzrHe1ScmTkk/SouqB8vq42
0jl0EnuF9/RsMR1Io4U/S5Wt+A79AaDyK/w=
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlUtrE1EYhgfxBwguFBTEvyAiFhfzHxRFvC0E17rQhYLiT7D3u/gTXNg2
3sBFFyIutE1qm17UZJKZZJK0KFjBtur7Td4jnhe7dTMPhAfyQObMnO9Mjl69
fvraniAI7v/+7A3IRri7/8VGWJwHVTH72qohZp+ZBnUx+yWjJWYfHwNvxOwX
jKaYfXAAzIrZ+/tARcw+NAhKYvaTRiJmP2gUxH+vLxKzl5cMMXtfL3gvZn84
Cdpi9hvGJzH7yxegJWa/YnTE7Mtl0HTPfZ/xyPWVZZCGvQ/Au65fSV9aNMLJ
CfDW9a0fIBWz/9wBHbcOf/74PX83W99j128ZNTH71BMQidmfFkAiZj9irIvZ
Fz+AWMx+yEjF7KeMVOw/32x/iuHoCHjt+s42aIfTU6DWna/Prm9vgVb4cc0I
h4fAgusZ62L2A0YsZj9h1MX+9bN9LHfP75zrZ42a2L9/3+wXjVTM7q8jm9NE
5q8wA+JwoB/8eX+eNxIx+00jErP3GA2xvy7uG88R+z2jI2Y/bLTE/v1l76WG
mxP/+r7ZN7+Bupj9mFERsz9/Bppi9myeEzH7QgnUxOy3jYaY/a7RFrNfNtpi
9nNGXcw+NgoaYnbOk2///PJ/bXXFcP37JojF7CPDIBazf/0CKmL2+TlQEbPf
MZpi9olxkIrZS0UQidn3G1Ux+3EjErOfMapi9mBX8p73vOc973n/P/0XnFJ2
Rg==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}]}, "\"\<Click me\>\""],
      "Toggle changes in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["FactorInteger",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/FactorInteger"], "[", 
       StyleBox["n", "TI"], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives a list of the prime factors of the integer ",
     Cell[BoxData[
      StyleBox["n", "TI"]], "InlineFormula"],
     ", together with their exponents. "
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["FactorInteger",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/FactorInteger"], "[", 
       RowBox[{
        StyleBox["n", "TI"], ",", 
        StyleBox["k", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]does partial factorization, pulling out at most ",
     Cell[BoxData[
      StyleBox["k", "TI"]], "InlineFormula"],
     " distinct factors."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->7081]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "For negative numbers, the unit ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    RowBox[{"-", "1"}], ",", "1"}], "}"}]], "InlineFormula"],
 " is included in the list of factors. "
}], "Notes",
 CellID->12753],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FactorInteger",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FactorInteger"]], "InlineFormula"],
 " also works on rational numbers. The prime factors of the denominator are \
given with negative exponents. ",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->10024,
  ButtonNote->"10024"]
}], "Notes",
 CellID->26996],

Cell[TextData[{
 "The last element in the list ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["FactorInteger",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FactorInteger"], "[", 
   RowBox[{
    StyleBox["n", "TI"], ",", 
    StyleBox["k", "TI"]}], "]"}]], "InlineFormula"],
 " gives what is left after the partial factorization."
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->455806205],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["FactorInteger",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FactorInteger"], "[", 
   RowBox[{
    StyleBox["n", "TI"], ",", 
    RowBox[{
     ButtonBox["GaussianIntegers",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/GaussianIntegers"], "->", 
     ButtonBox["True",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/True"]}]}], "]"}]], "InlineFormula"],
 " factors over Gaussian integers. ",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->9354,
  ButtonNote->"9354"]
}], "Notes",
 CellID->1387],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["FactorInteger",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FactorInteger"], "[", 
   RowBox[{
    StyleBox["m", "TI"], "+", 
    RowBox[{"I", " ", 
     StyleBox["n", "TI"]}]}], "]"}]], "InlineFormula"],
 " automatically works over the Gaussian integers. "
}], "Notes",
 CellID->5461],

Cell[TextData[{
 "When necessary, a unit of the form ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    RowBox[{"-", "1"}], ",", "1"}], "}"}]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    ButtonBox["I",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/I"], ",", "1"}], "}"}]], "InlineFormula"],
 " or ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    RowBox[{"-", 
     ButtonBox["I",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/I"]}], ",", "1"}], "}"}]], "InlineFormula"],
 " is included in the list of factors. "
}], "Notes",
 CellID->29914],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["FactorInteger",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FactorInteger"], "[", 
   RowBox[{
    StyleBox["n", "TI"], ",", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"]}], "]"}]], "InlineFormula"],
 " pulls out only factors that are easy to find."
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->428367995],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FactorInteger",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FactorInteger"]], "InlineFormula"],
 " uses ",
 Cell[BoxData[
  ButtonBox["PrimeQ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrimeQ"]], "InlineFormula"],
 " to determine whether factors are prime."
}], "Notes",
 CellID->252132328]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->219576864],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->302475635],

Cell["Find prime factors and their exponents:", "ExampleText",
 CellID->13289],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FactorInteger", "[", "2434500", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->7566],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"2", ",", "2"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"3", ",", "2"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"5", ",", "3"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"541", ",", "1"}], "}"}]}], "}"}]], "Output",
 ImageSize->{223, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->13664638]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->105072382],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FactorInteger", "[", 
  RowBox[{
   RowBox[{"10", "^", "100"}], "+", "1"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19558],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"73", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"137", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"401", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1201", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1601", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1676321", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"5964848081", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "129694419029057750551385771184564274499075700947656757821537291527196801"\
, ",", "1"}], "}"}]}], "}"}]], "Output",
 ImageSize->{578, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->586523645]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "SmyRawData",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1372],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FactorInteger",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FactorInteger"]], "InlineFormula"],
 " threads itself element-wise over lists:"
}], "ExampleText",
 CellID->17280],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FactorInteger", "[", 
  RowBox[{"{", 
   RowBox[{"123", ",", "1234", ",", "12345", ",", "123456"}], "}"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->25994],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"3", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"41", ",", "1"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"617", ",", "1"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"3", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"5", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"823", ",", "1"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2", ",", "6"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"643", ",", "1"}], "}"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{366, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->236038241]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->803],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FactorInteger",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FactorInteger"]], "InlineFormula"],
 " works with rational numbers:"
}], "ExampleText",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->10024],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FactorInteger", "[", 
  RowBox[{"3", "/", "8"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->993],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"2", ",", 
     RowBox[{"-", "3"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"3", ",", "1"}], "}"}]}], "}"}]], "Output",
 ImageSize->{116, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->430917184]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FactorInteger", "[", 
  RowBox[{"2345354", "/", "2424245"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->25213],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"2", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"5", ",", 
     RowBox[{"-", "1"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"11", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"17", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"311", ",", 
     RowBox[{"-", "1"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1559", ",", 
     RowBox[{"-", "1"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"6271", ",", "1"}], "}"}]}], "}"}]], "Output",
 ImageSize->{453, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->42545021]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->28593],

Cell["Factor over Gaussian integers:", "ExampleText",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->9354],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FactorInteger", "[", 
  RowBox[{"5", ",", 
   RowBox[{"GaussianIntegers", "\[Rule]", "True"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->9324],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "\[ImaginaryI]"}], ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"1", "+", 
      RowBox[{"2", " ", "\[ImaginaryI]"}]}], ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"2", "+", "\[ImaginaryI]"}], ",", "1"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{218, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->347402847]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(7)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->27025],

Cell[TextData[{
 "Find factorizations of numbers of the form ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox["2", 
     SuperscriptBox["2", "n"]], "+", "1"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->17993],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"FactorInteger", "[", 
     RowBox[{
      RowBox[{"2", "^", 
       RowBox[{"2", "^", "n"}]}], "+", "1"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"n", ",", "7"}], "}"}]}], "]"}], "//", "Column"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->3682],

Cell[BoxData[
 TagBox[GridBox[{
    {
     RowBox[{"{", 
      RowBox[{"{", 
       RowBox[{"5", ",", "1"}], "}"}], "}"}]},
    {
     RowBox[{"{", 
      RowBox[{"{", 
       RowBox[{"17", ",", "1"}], "}"}], "}"}]},
    {
     RowBox[{"{", 
      RowBox[{"{", 
       RowBox[{"257", ",", "1"}], "}"}], "}"}]},
    {
     RowBox[{"{", 
      RowBox[{"{", 
       RowBox[{"65537", ",", "1"}], "}"}], "}"}]},
    {
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"641", ",", "1"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"6700417", ",", "1"}], "}"}]}], "}"}]},
    {
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"274177", ",", "1"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"67280421310721", ",", "1"}], "}"}]}], "}"}]},
    {
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"59649589127497217", ",", "1"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"5704689200685129054721", ",", "1"}], "}"}]}], "}"}]}
   },
   ColumnsEqual->False,
   GridBoxAlignment->{"Columns" -> {{Left}}},
   GridBoxItemSize->{"Columns" -> {{Automatic}}, "Rows" -> {{Automatic}}},
   RowsEqual->False],
  "Column"]], "Output",
 ImageSize->{393, 103},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->426743074]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->6463],

Cell["Display as an explicit product of factors:", "ExampleText",
 CellID->8661],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FactorInteger", "[", 
  RowBox[{"20", "!"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->28484],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"2", ",", "18"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"3", ",", "8"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"5", ",", "4"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"7", ",", "2"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"11", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"13", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"17", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"19", ",", "1"}], "}"}]}], "}"}]], "Output",
 ImageSize->{445, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->184759346]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CenterDot", "@@", 
  RowBox[{"(", 
   RowBox[{"Superscript", "@@@", "%"}], ")"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->2116],

Cell[GraphicsData["CompressedBitmap", "\<\
eJy1VdtugkAQXdjFS+ulFbU0auLXtE/9CGKa6EPTxvqp/s92B2bL9jACiZaE
HebKmdnl8Jaf9u8f+emwy7evx/xrf9h9b18+j86kI6XU2d3TTLln656K+2yt
k+GqdCno0u7OnEpywXJJbnqwxaVStyROjVnOWGqOn7NuGvLIPPzzdlpjyCAE
KvnVGWYqWhdSbqAjmjnocaeqWAX9Cei9og729eiW2E8gsN+12Acd7YjynuP6
EjqhHsZhV13rFXnlBgfWVcsMcWdQb9tJRJG2dIP5hNf47mpvy7h3/3U8XYnW
cD38ytpQyue1xIy5z6DPhJ6aJnbtxNv8Tb1c4geKMKr+bY3EWngGJ0KHpI7F
bM9YD9JcTBhNqVi7QNiDmhw7/afYyHre0Z6J+/VIxWg1TEjx24zA4iNbsXho
p3g6x0kgQ/84sGuh7oT9xlZ/E8l/Kb/seNjQaXhqPbcbgePXoCNfoR/1AAnu
VkeWUwIf3BBVx/ksbcV0oX1zEyQDW+1301xwDuhfcZ01yw1Ipo/oB8Tcuj4=
\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{212, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->569501126]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->29465],

Cell["Find primes that appear only to the first power:", "ExampleText",
 CellID->4798],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FactorInteger", "[", 
  RowBox[{"20", "!"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->29054],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"2", ",", "18"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"3", ",", "8"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"5", ",", "4"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"7", ",", "2"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"11", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"13", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"17", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"19", ",", "1"}], "}"}]}], "}"}]], "Output",
 ImageSize->{445, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->60952340]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Cases", "[", 
  RowBox[{"%", ",", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"p_", ",", "1"}], "}"}], "\[Rule]", "p"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->4025],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"11", ",", "13", ",", "17", ",", "19"}], "}"}]], "Output",
 ImageSize->{108, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->261486306]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->32420],

Cell["\<\
Find all natural numbers up to 100 that are primes or prime powers:\
\>", "ExampleText",
 CellID->21190],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Select", "[", 
  RowBox[{
   RowBox[{"Range", "[", "100", "]"}], ",", 
   RowBox[{
    RowBox[{
     RowBox[{"Length", "[", 
      RowBox[{"FactorInteger", "[", "#", "]"}], "]"}], "==", "1"}], "&"}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->27341],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1", ",", "2", ",", "3", ",", "4", ",", "5", ",", "7", ",", "8", ",", "9", 
   ",", "11", ",", "13", ",", "16", ",", "17", ",", "19", ",", "23", ",", 
   "25", ",", "27", ",", "29", ",", "31", ",", "32", ",", "37", ",", "41", 
   ",", "43", ",", "47", ",", "49", ",", "53", ",", "59", ",", "61", ",", 
   "64", ",", "67", ",", "71", ",", "73", ",", "79", ",", "81", ",", "83", 
   ",", "89", ",", "97"}], "}"}]], "Output",
 ImageSize->{461, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->394992867]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->8645],

Cell["The number of distinct factors of successive numbers:", "ExampleText",
 CellID->24733],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"Length", "[", 
     RowBox[{"FactorInteger", "[", "n", "]"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"n", ",", "100"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->7905],

Cell[BoxData[
 GraphicsBox[{{}, 
   {Hue[0.67, 0.6, 0.6], PointBox[CompressedData["
1:eJxd0clKQ0EQheGDK5cuXLhwoSIiIuI8a9ohGqcYr4nDVnTtK9Sj9SP5CCre
I9QfCJePv6r7ksy+fTYfY5Lef76/z7/PVyc/VbLH4Ql4Ep4q+Zxp9Bl4Dp7H
/gL6IvoS+jL6CryK+TV4HfMb6JvoW/A2vIP9XfQ99H30A/RD9CP049b+3zro
JTtsn3eSHbBO0W2ff4YO6xzd9n43v3900S+wD+sS3fZ+D72HfoUO6xrd9v4N
OqxbdFh36LbP77f279NHv8c+rEGeLwPM2+35FfN6yPMFDtv7tvcb7De5B3qF
9Yh9OGzfb/v+IfaHuQd6Rdcon19GeT7gCusJ99u+3/b7Y17P2IcDrrBesG/7
fvQK6/Xf3+iuhoY=
     "]]}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 1.},
  ImageMargins->0.,
  PlotRange->{{0., 100.}, {1., 3.}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{229, 146},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{225, Automatic}},
 CellLabel->"Out[1]=",
 CellID->27617949]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->8945],

Cell["The highest power of a prime in numbers up to 100: ", "ExampleText",
 CellID->14017],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"Max", "[", 
    RowBox[{"Last", "/@", 
     RowBox[{"FactorInteger", "[", "n", "]"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "2", ",", "100"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->22202],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1", ",", "1", ",", "2", ",", "1", ",", "1", ",", "1", ",", "3", ",", "2", 
   ",", "1", ",", "1", ",", "2", ",", "1", ",", "1", ",", "1", ",", "4", ",", 
   "1", ",", "2", ",", "1", ",", "2", ",", "1", ",", "1", ",", "1", ",", "3", 
   ",", "2", ",", "1", ",", "3", ",", "2", ",", "1", ",", "1", ",", "1", ",", 
   "5", ",", "1", ",", "1", ",", "1", ",", "2", ",", "1", ",", "1", ",", "1", 
   ",", "3", ",", "1", ",", "1", ",", "1", ",", "2", ",", "2", ",", "1", ",", 
   "1", ",", "4", ",", "2", ",", "2", ",", "1", ",", "2", ",", "1", ",", "3", 
   ",", "1", ",", "3", ",", "1", ",", "1", ",", "1", ",", "2", ",", "1", ",", 
   "1", ",", "2", ",", "6", ",", "1", ",", "1", ",", "1", ",", "2", ",", "1", 
   ",", "1", ",", "1", ",", "3", ",", "1", ",", "1", ",", "2", ",", "2", ",", 
   "1", ",", "1", ",", "1", ",", "4", ",", "4", ",", "1", ",", "1", ",", "2", 
   ",", "1", ",", "1", ",", "1", ",", "3", ",", "1", ",", "2", ",", "1", ",", 
   "2", ",", "1", ",", "1", ",", "1", ",", "5", ",", "1", ",", "2", ",", "2", 
   ",", "2"}], "}"}]], "Output",
 ImageSize->{462, 67},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->56830228]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->13878],

Cell["The number of factors with Gaussian integers allowed:", "ExampleText",
 CellID->19549],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"Length", "[", 
     RowBox[{"FactorInteger", "[", 
      RowBox[{"n", ",", 
       RowBox[{"GaussianIntegers", "->", "True"}]}], "]"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"n", ",", "100"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->13033],

Cell[BoxData[
 GraphicsBox[{{}, 
   {Hue[0.67, 0.6, 0.6], PointBox[CompressedData["
1:eJxdkssuREEURU+MDA0MDAwQERER7zddXq1pz9ZeU2HsF+rT6pN8AqKWpFYn
N5WVffbep+7t6bfPwcdIRLz/PL/n3++r056R2nM0tfqY9HHNTYgn5Z+Sf0b6
bGp75qTPK39BvCheUt6y8lY0vype0/y68jZ0n035t+Tflr4j3lX+nvz7mj8Q
H9aT79KRP7X7ZnEctf1ZHMdtXxbHiXQY/6l0cZypH2b/rvzddj7OpcP4e5W5
f0/6hfrFcSkdZv+++vvtfFxJh3n/18qH2e+mcv2+Gab/Vvkw+XctJ5g8uPYX
mP77Vk8w7xOueQXGP2j3S+IM0y+OB/lh+mH6YfqH2n/Y6ll6kR6P8sO8P7ju
VzQfT9pfnGH2h/E/yw9zf+kFxv8ivziLC8z/5/Vf/wapvHmT
     "]]}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 1.},
  PlotRange->{{0., 100.}, {1., 5.}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 123},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->301235655]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->18192],

Cell["\<\
Timings can increase rapidly and unpredictably with the size of the input:\
\>", "ExampleText",
 CellID->11680],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"Timing", "[", 
    RowBox[{
     RowBox[{"FactorInteger", "[", 
      RowBox[{
       RowBox[{"2", "^", "n"}], "-", "1"}], "]"}], ";", "n"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "50", ",", "300", ",", "50"}], "}"}]}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->14427],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1.484923295436147`*^-14", ",", "50"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1.484923295436147`*^-14", ",", "100"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.01000000000001483`", ",", "150"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.04000000000002393`", ",", "200"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"3.7650000000000037`", ",", "250"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.06000000000001712`", ",", "300"}], "}"}]}], "}"}]], "Output",
 ImageSize->{356, 37},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->412520430]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->17978],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["IntegerExponent",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/IntegerExponent"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Prime",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Prime"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["PrimeQ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/PrimeQ"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["PrimePowerQ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/PrimePowerQ"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SquareFreeQ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SquareFreeQ"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Divisors",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Divisors"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->27412]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->22375],

Cell[TextData[ButtonBox["Some Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/SomeMathematicalFunctions"]], "Tutorials",
 CellID->19646],

Cell[TextData[{
 ButtonBox["The Limits of ",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/TheLimitsOfMathematica"],
 StyleBox[ButtonBox["Mathematica",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/TheLimitsOfMathematica"],
  FontSlant->"Italic"]
}], "Tutorials",
 CellID->1984],

Cell[TextData[ButtonBox["Integer and Number\[Hyphen]Theoretical Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/IntegerAndNumberTheoreticalFunctions"]], "Tutorials",
 CellID->4248]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection",
 CellID->26533],

Cell[TextData[{
 ButtonBox["Demonstrations with FactorInteger",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/symbol.html?symbol=FactorInteger"],
     None}],
 " (",
 ButtonBox["Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None}],
 ")"
}], "RelatedLinks"],

Cell[TextData[ButtonBox["Implementation notes: Numerical and Related \
Functions",
 BaseStyle->"Hyperlink",
 ButtonData->{
   URL["http://reference.wolfram.com/mathematica/note/\
SomeNotesOnInternalImplementation.html#12915"], None}]], "RelatedLinks",
 CellID->74473536],

Cell[TextData[StyleBox[ButtonBox["MathWorld",
 BaseStyle->"MathWorldLink",
 ButtonData->{
   URL["http://mathworld.wolfram.com/PrimeFactorizationAlgorithms.html"], 
   None}], "IT"]], "RelatedLinks"],

Cell[TextData[ButtonBox["The Wolfram Functions Site",
 BaseStyle->"FunctionsSiteLink",
 ButtonData->{
   URL["http://functions.wolfram.com/NumberTheoryFunctions/FactorInteger/"], 
   None}]], "RelatedLinks"],

Cell[TextData[{
 ButtonBox["NKS|Online",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.wolframscience.com/nksonline/index/search.cgi?SearchIndex=\
FactorInteger"], None}],
 " (",
 StyleBox[ButtonBox["A New Kind of Science",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.wolframscience.com/nksonline/"], None}],
  FontSlant->"Italic"],
 ")"
}], "RelatedLinks"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->250141172],

Cell[TextData[ButtonBox["Cryptographic Number Theory",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/CryptographicNumberTheory"]], "MoreAbout",
 CellID->82732016],

Cell[TextData[ButtonBox["Integer Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/IntegerFunctions"]], "MoreAbout",
 CellID->81905762],

Cell[TextData[ButtonBox["Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/MathematicalFunctions"]], "MoreAbout",
 CellID->260279914],

Cell[TextData[ButtonBox["Number Theoretic Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NumberTheoreticFunctions"]], "MoreAbout",
 CellID->122918338],

Cell[TextData[ButtonBox["Number Theory",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NumberTheory"]], "MoreAbout",
 CellID->905695462],

Cell[TextData[ButtonBox["Prime Numbers",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/PrimeNumbers"]], "MoreAbout",
 CellID->70222594],

Cell[TextData[ButtonBox["New in 6.0: Number Theory & Integer Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/NewIn60NumberTheoryAndIntegerFunctions"]], "MoreAbout",
 CellID->257732429]
}, Open  ]],

Cell["New in 1  |  Last modified in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"FactorInteger - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 42, 16.1079446}", "context" -> "System`", 
    "keywords" -> {
     "decomposition into primes", "factoring", "factors", "Gaussian integers",
       "ifact", "integer factoring", "Pollard algorithm", "prime factors", 
      "submultiples"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "FactorInteger[n] gives a list of the prime factors of the integer n, \
together with their exponents. FactorInteger[n, k] does partial \
factorization, pulling out at most k distinct factors.", 
    "synonyms" -> {"factor integer"}, "title" -> "FactorInteger", "type" -> 
    "Symbol", "uri" -> "ref/FactorInteger"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[13636, 381, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->219576864]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 40958, 1375}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 3656, 77, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[4279, 104, 4197, 83, 70, "ObjectNameGrid"],
Cell[8479, 189, 1069, 32, 70, "Usage",
 CellID->7081]
}, Open  ]],
Cell[CellGroupData[{
Cell[9585, 226, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[10063, 243, 226, 8, 70, "Notes",
 CellID->12753],
Cell[10292, 253, 390, 12, 70, "Notes",
 CellID->26996],
Cell[10685, 267, 494, 15, 70, "Notes",
 CellID->455806205],
Cell[11182, 284, 603, 21, 70, "Notes",
 CellID->1387],
Cell[11788, 307, 348, 12, 70, "Notes",
 CellID->5461],
Cell[12139, 321, 585, 23, 70, "Notes",
 CellID->29914],
Cell[12727, 346, 522, 16, 70, "Notes",
 CellID->428367995],
Cell[13252, 364, 347, 12, 70, "Notes",
 CellID->252132328]
}, Closed]],
Cell[CellGroupData[{
Cell[13636, 381, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->219576864],
Cell[CellGroupData[{
Cell[14021, 395, 148, 5, 70, "ExampleSection",
 CellID->302475635],
Cell[14172, 402, 78, 1, 70, "ExampleText",
 CellID->13289],
Cell[CellGroupData[{
Cell[14275, 407, 110, 3, 28, "Input",
 CellID->7566],
Cell[14388, 412, 421, 15, 36, "Output",
 CellID->13664638]
}, Open  ]],
Cell[14824, 430, 125, 3, 70, "ExampleDelimiter",
 CellID->105072382],
Cell[CellGroupData[{
Cell[14974, 437, 155, 5, 28, "Input",
 CellID->19558],
Cell[15132, 444, 760, 25, 71, "Output",
 CellID->586523645]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[15941, 475, 220, 7, 70, "ExampleSection",
 CellID->1372],
Cell[16164, 484, 223, 7, 70, "ExampleText",
 CellID->17280],
Cell[CellGroupData[{
Cell[16412, 495, 189, 6, 70, "Input",
 CellID->25994],
Cell[16604, 503, 977, 35, 54, "Output",
 CellID->236038241]
}, Open  ]],
Cell[17596, 541, 119, 3, 70, "ExampleDelimiter",
 CellID->803],
Cell[17718, 546, 260, 8, 70, "ExampleText",
 CellID->10024],
Cell[CellGroupData[{
Cell[18003, 558, 126, 4, 70, "Input",
 CellID->993],
Cell[18132, 564, 323, 12, 36, "Output",
 CellID->430917184]
}, Open  ]],
Cell[CellGroupData[{
Cell[18492, 581, 140, 4, 70, "Input",
 CellID->25213],
Cell[18635, 587, 669, 24, 36, "Output",
 CellID->42545021]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[19353, 617, 239, 7, 70, "ExampleSection",
 CellID->28593],
Cell[19595, 626, 116, 2, 70, "ExampleText",
 CellID->9354],
Cell[CellGroupData[{
Cell[19736, 632, 175, 5, 70, "Input",
 CellID->9324],
Cell[19914, 639, 500, 18, 36, "Output",
 CellID->347402847]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[20463, 663, 223, 7, 70, "ExampleSection",
 CellID->27025],
Cell[20689, 672, 244, 9, 70, "ExampleText",
 CellID->17993],
Cell[CellGroupData[{
Cell[20958, 685, 323, 11, 70, "Input",
 CellID->3682],
Cell[21284, 698, 1331, 49, 124, "Output",
 CellID->426743074]
}, Open  ]],
Cell[22630, 750, 120, 3, 70, "ExampleDelimiter",
 CellID->6463],
Cell[22753, 755, 80, 1, 70, "ExampleText",
 CellID->8661],
Cell[CellGroupData[{
Cell[22858, 760, 124, 4, 70, "Input",
 CellID->28484],
Cell[22985, 766, 661, 23, 36, "Output",
 CellID->184759346]
}, Open  ]],
Cell[CellGroupData[{
Cell[23683, 794, 155, 5, 70, "Input",
 CellID->2116],
Cell[23841, 801, 761, 17, 70, "Output",
 Evaluatable->False,
 CellID->569501126]
}, Open  ]],
Cell[24617, 821, 121, 3, 70, "ExampleDelimiter",
 CellID->29465],
Cell[24741, 826, 86, 1, 70, "ExampleText",
 CellID->4798],
Cell[CellGroupData[{
Cell[24852, 831, 124, 4, 70, "Input",
 CellID->29054],
Cell[24979, 837, 660, 23, 36, "Output",
 CellID->60952340]
}, Open  ]],
Cell[CellGroupData[{
Cell[25676, 865, 201, 7, 70, "Input",
 CellID->4025],
Cell[25880, 874, 228, 7, 36, "Output",
 CellID->261486306]
}, Open  ]],
Cell[26123, 884, 121, 3, 70, "ExampleDelimiter",
 CellID->32420],
Cell[26247, 889, 114, 3, 70, "ExampleText",
 CellID->21190],
Cell[CellGroupData[{
Cell[26386, 896, 284, 10, 70, "Input",
 CellID->27341],
Cell[26673, 908, 595, 13, 54, "Output",
 CellID->394992867]
}, Open  ]],
Cell[27283, 924, 120, 3, 70, "ExampleDelimiter",
 CellID->8645],
Cell[27406, 929, 92, 1, 70, "ExampleText",
 CellID->24733],
Cell[CellGroupData[{
Cell[27523, 934, 286, 9, 70, "Input",
 CellID->7905],
Cell[27812, 945, 921, 25, 167, "Output",
 CellID->27617949]
}, Open  ]],
Cell[28748, 973, 120, 3, 70, "ExampleDelimiter",
 CellID->8945],
Cell[28871, 978, 90, 1, 70, "ExampleText",
 CellID->14017],
Cell[CellGroupData[{
Cell[28986, 983, 284, 9, 70, "Input",
 CellID->22202],
Cell[29273, 994, 1228, 21, 88, "Output",
 CellID->56830228]
}, Open  ]],
Cell[30516, 1018, 121, 3, 70, "ExampleDelimiter",
 CellID->13878],
Cell[30640, 1023, 92, 1, 70, "ExampleText",
 CellID->19549],
Cell[CellGroupData[{
Cell[30757, 1028, 361, 11, 70, "Input",
 CellID->13033],
Cell[31121, 1041, 882, 23, 144, "Output",
 CellID->301235655]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[32052, 1070, 226, 7, 70, "ExampleSection",
 CellID->18192],
Cell[32281, 1079, 121, 3, 70, "ExampleText",
 CellID->11680],
Cell[CellGroupData[{
Cell[32427, 1086, 352, 12, 70, "Input",
 CellID->14427],
Cell[32782, 1100, 668, 19, 58, "Output",
 CellID->412520430]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[33511, 1126, 311, 9, 70, "SeeAlsoSection",
 CellID->17978],
Cell[33825, 1137, 1447, 53, 70, "SeeAlso",
 CellID->27412]
}, Open  ]],
Cell[CellGroupData[{
Cell[35309, 1195, 314, 9, 70, "TutorialsSection",
 CellID->22375],
Cell[35626, 1206, 163, 3, 70, "Tutorials",
 CellID->19646],
Cell[35792, 1211, 290, 9, 70, "Tutorials",
 CellID->1984],
Cell[36085, 1222, 197, 4, 70, "Tutorials",
 CellID->4248]
}, Open  ]],
Cell[CellGroupData[{
Cell[36319, 1231, 321, 9, 70, "RelatedLinksSection",
 CellID->26533],
Cell[36643, 1242, 371, 12, 70, "RelatedLinks"],
Cell[37017, 1256, 270, 6, 70, "RelatedLinks",
 CellID->74473536],
Cell[37290, 1264, 199, 4, 70, "RelatedLinks"],
Cell[37492, 1270, 207, 4, 70, "RelatedLinks"],
Cell[37702, 1276, 391, 13, 70, "RelatedLinks"]
}, Open  ]],
Cell[CellGroupData[{
Cell[38130, 1294, 319, 9, 70, "MoreAboutSection",
 CellID->250141172],
Cell[38452, 1305, 163, 3, 70, "MoreAbout",
 CellID->82732016],
Cell[38618, 1310, 144, 3, 70, "MoreAbout",
 CellID->81905762],
Cell[38765, 1315, 155, 3, 70, "MoreAbout",
 CellID->260279914],
Cell[38923, 1320, 162, 3, 70, "MoreAbout",
 CellID->122918338],
Cell[39088, 1325, 137, 3, 70, "MoreAbout",
 CellID->905695462],
Cell[39228, 1330, 136, 3, 70, "MoreAbout",
 CellID->70222594],
Cell[39367, 1335, 198, 4, 70, "MoreAbout",
 CellID->257732429]
}, Open  ]],
Cell[39580, 1342, 50, 0, 70, "History"],
Cell[39633, 1344, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

