(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     37135,       1365]
NotebookOptionsPosition[     29795,       1107]
NotebookOutlinePosition[     31169,       1145]
CellTagsIndexPosition[     31083,       1140]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Algebraic Operations on Polynomials" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/AlgebraicOperationsOnPolynomials"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"Algebraic Operations on Polynomials\"\>"}, "\<\"tutorials\"\>",
        
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"FactorSquareFreeList" :> 
          Documentation`HelpLookup["paclet:ref/FactorSquareFreeList"], 
          "Factor" :> Documentation`HelpLookup["paclet:ref/Factor"], 
          "FactorTerms" :> Documentation`HelpLookup["paclet:ref/FactorTerms"],
           "SquareFreeQ" :> 
          Documentation`HelpLookup["paclet:ref/SquareFreeQ"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"FactorSquareFreeList\"\>", 
       2->"\<\"Factor\"\>", 3->"\<\"FactorTerms\"\>", 
       4->"\<\"SquareFreeQ\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Polynomial Factoring & Decomposition" :> 
          Documentation`HelpLookup["paclet:guide/PolynomialFactoring"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Polynomial Factoring & Decomposition\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["FactorSquareFree", "ObjectName"], Cell[BoxData[
     TooltipBox[
      TogglerBox[Dynamic[
        FrontEnd`CurrentValue[
         FrontEnd`EvaluationNotebook[], {
         TaggingRules, "ModificationHighlight"}]], {True->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVVly2zAMDQhxEylSsizLaxJn73KafvQAuUDv/1eApJuosceeTDAWF4B8
eFgk71///HoVV1dXK3p+0wNfIzdCLL4I6p1sYBYuPy3SI2AmRBpuaE/TPRsE
3Isg8mrB9l6ILQhhLkbd93MBuwVxAthdZwPr5wVVzBc8wAvMX8TFqPkqE8q6
rDCsKPjZ/g1guz+PKvLFfGsDU1R4A8zDhgLangX9x/WGM7A1qSQ7rraYkaLf
LHZ5FWY9H7rew/n4/6vWPKT2mXFBaH/NGejzqi/VnFEFJ2Rb1Lo+jtofllPD
RVJjdZLr51Eb/ED1K+QI109LcIg+sdSIWqa5g85L6MiNoRloTws+pdBKiFGl
mx3N0jcAlcsKeFivVkteVOgMSNsC18sYS/iIMdZoK43WSdQVK1SLlYy09bZ2
aEO+aoPiACNak0EBbtfLRIz9KJRcroaSKwE11BprpRHqiJ0BrxtJG4pLYqWc
RZO4BQwSbGT4rFivaLjjQdscjSvlih48Mu+WDisOBcB5ehgeDEZQlikkKh3b
KA1FMTDg9yFxLSe4XC37begHTDh5seS1I/gaW025Yo/yUFbmSalShzon1Kch
JTNn2hKS1hyWVIkf5n2LHEQgt0zeUNTkMaYrDdkMZdo3fJCDHx7hxzAC+8rs
XYBokTPfEJk6JVlZx3Yl0UnTYMPvSUt6zghf8dhIi5VDnwO+Hcb1MHK1msK+
oy5qkYS7wVC3cfiq8ojBcMtFMJXnDiIcLPFJClDFGkJBhXGEZV6VcCQFoey7
tyvBF6lw0uKH6nyQ5fqhrKx/Kxe8+3coucpGN4U5+RYux7vnhGO5P7hKenqz
xSO3ivjisf345fg5Pt1ScCRkkzR1U1Dur+NCec4LZ49Ynx9P04GmOvlnqcKZ
79BfJfcjkg==
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlLlKQ0EYRi/iA9gpWIgvIXbjM1iKWAjW2vgeZnWN+AgWZhEULCxELFyi
ZnfJavbeJOr/33xXcj8CNnY5EA7k3GWYmTuzq+uLa2OWZS38/MYt0DbDHH8Q
8uTB64bRNrmsQkaPRoQSGX1ZqZPR9/eEazL6klIlowf8wiUZ3ecV3snowYDw
SEafUmJk9HmlQh4cX4GM7vUId2T0VFIho28or2T0w5DQIKN3PoUaGX1FaZLR
0ymh6sz7hHLk9PMzoU5Gz6SFmvFsCbd9Xzj9qyc0nfe49xfuTyYUEzoQbmh8
+N8e37HTbVpk9MSzUCajbypFMvq0UiOjzyklMnqvKzRMJCwU+/vjzemTSpns
nh97feJmd0e4cvppTKiQ0bsdoW5ecorZDgpPtD6xqFA2fp+Qp/vtdUj1v797
Oqdwnb2Ov++fUVpk9PCJUCC7n4t5wz5xj9/+rj6ceXbPP86NbEb58/k56qP+
n/0bdGyYqQ==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}], False->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVdd22zAMDUBxSaJm5HgljrPTJN177/nWh35AfqD//1aAVGrrJLbbk+BI
FEmQFxcXlDQ9//3zHNfW1gZ0/6IbbsYOhahvCGrO9qHM/mnhmMMLoKv0RD6V
cPyZHTTcDI5NkYnQq4UQ0BdiAkKYZagl1OOw5fWtcgyvPp7AnefsGG0DBIeA
9RZVrNfcwBasb4nlbP9u4cbc/XD/zXcIgC2qYYazJTQiGSYHS0Hr8Yu5LV/h
ePvR5oMOKswAQ0M8R5OloFMWSNRE+IAVeALv38KX26wAV1uU5Ojv16PQy8q+
gMP+9ABW5C9KbkZi6qv19DGc/ID9b+CrZTjbbVagH3p91oIcL+GZ6JDNUet4
HpWUXxx5BaWZxRh1FQgH6JqoKcarF/2/dblez7IE0XmWGlFL/6ygchIqCmPo
CTSmDq9SaCUUhfI7K3pKlwJESZiAveHGxoA7ESYGpM2B62WMJXzEoojRRhpt
IlFHPKFyjGRBQ2fjBG0WttpMcYIFWhNAAXaHA0+M4yiUXK6UxJWAGmKNsdII
cYGVAadTSQPKS2KkEovGc8swk2ALhg8Tww1qdrjRNmSTtOUqHDhk3jktVpwK
QOLoZngwWICyTMFTqdhHMrQTDQPeazzXdgWXK+e4KV3AhH0US1Ergo8x16QV
R5QXZWWeJJW6qLNHPWq8mEFpS0hac1pSeX4YxjlyEhmFZfKGsqaIhd+Sks+Q
0i7lhZx8cwYPmx5wrMA+yaCwyMqnRCb2IiubsF9JTKRJMeX3JKd5VoS3OEyl
xShBFxLebXrDpsfVSlv2FZ2iHMn4NBg6bZy+ihxiZvjIFWAixyeIcLDNT1KC
qogha1Gh14NB6LXpSEpC2bm3y8O3FmHniF9U55INhnttz7pZuWDu79BqFZxJ
F2bhWzjo7Zx6HMvng6ukuztzvGJXa66NmF/+crzrHe1ScmTkk/SouqB8vq42
0jl0EnuF9/RsMR1Io4U/S5Wt+A79AaDyK/w=
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlUtrE1EYhgfxBwguFBTEvyAiFhfzHxRFvC0E17rQhYLiT7D3u/gTXNg2
3sBFFyIutE1qm17UZJKZZJK0KFjBtur7Td4jnhe7dTMPhAfyQObMnO9Mjl69
fvraniAI7v/+7A3IRri7/8VGWJwHVTH72qohZp+ZBnUx+yWjJWYfHwNvxOwX
jKaYfXAAzIrZ+/tARcw+NAhKYvaTRiJmP2gUxH+vLxKzl5cMMXtfL3gvZn84
Cdpi9hvGJzH7yxegJWa/YnTE7Mtl0HTPfZ/xyPWVZZCGvQ/Au65fSV9aNMLJ
CfDW9a0fIBWz/9wBHbcOf/74PX83W99j128ZNTH71BMQidmfFkAiZj9irIvZ
Fz+AWMx+yEjF7KeMVOw/32x/iuHoCHjt+s42aIfTU6DWna/Prm9vgVb4cc0I
h4fAgusZ62L2A0YsZj9h1MX+9bN9LHfP75zrZ42a2L9/3+wXjVTM7q8jm9NE
5q8wA+JwoB/8eX+eNxIx+00jErP3GA2xvy7uG88R+z2jI2Y/bLTE/v1l76WG
mxP/+r7ZN7+Bupj9mFERsz9/Bppi9myeEzH7QgnUxOy3jYaY/a7RFrNfNtpi
9nNGXcw+NgoaYnbOk2///PJ/bXXFcP37JojF7CPDIBazf/0CKmL2+TlQEbPf
MZpi9olxkIrZS0UQidn3G1Ux+3EjErOfMapi9mBX8p73vOc973n/P/0XnFJ2
Rg==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}]}, "\"\<Click me\>\""],
      "Toggle changes in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["FactorSquareFree",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/FactorSquareFree"], "[", 
       StyleBox["poly", "TI"], "]"}]], "InlineFormula"],
     "\[LineSeparator]pulls out any multiple factors in a polynomial. "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->30792]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["FactorSquareFree",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FactorSquareFree"], "[", 
   RowBox[{
    StyleBox["poly", "TI"], ",", 
    RowBox[{
     ButtonBox["Modulus",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Modulus"], "->", 
     StyleBox["p", "TI"]}]}], "]"}]], "InlineFormula"],
 " pulls out multiple factors modulo a prime ",
 Cell[BoxData[
  StyleBox["p", "TI"]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->30100],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["FactorSquareFree",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FactorSquareFree"], "[", 
   RowBox[{
    StyleBox["poly", "TI"], ",", 
    RowBox[{
     ButtonBox["Extension",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Extension"], "->", 
     ButtonBox["Automatic",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Automatic"]}]}], "]"}]], "InlineFormula"],
 " extends the coefficient field to include algebraic numbers that appear in \
the coefficients of ",
 Cell[BoxData[
  StyleBox["poly", "TI"]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->26517],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FactorSquareFree",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FactorSquareFree"]], "InlineFormula"],
 " automatically threads over lists, as well as equations, inequalities and \
logic functions."
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->82730543]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->268689014],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->382362664],

Cell["Pull out multiple factors:", "ExampleText",
 CellID->73605572],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FactorSquareFree", "[", 
  RowBox[{
   RowBox[{"x", "^", "5"}], "-", 
   RowBox[{"x", "^", "3"}], "-", 
   RowBox[{"x", "^", "2"}], "+", "1"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->17656],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "x"}], ")"}], "2"], " ", 
  RowBox[{"(", 
   RowBox[{"1", "+", 
    RowBox[{"2", " ", "x"}], "+", 
    RowBox[{"2", " ", 
     SuperscriptBox["x", "2"]}], "+", 
    SuperscriptBox["x", "3"]}], ")"}]}]], "Output",
 ImageSize->{172, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1821628]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->6818],

Cell["A complete factorization:", "ExampleText",
 CellID->340879707],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Factor", "[", 
  RowBox[{
   RowBox[{"x", "^", "5"}], "-", 
   RowBox[{"x", "^", "3"}], "-", 
   RowBox[{"x", "^", "2"}], "+", "1"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->10884],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "x"}], ")"}], "2"], " ", 
  RowBox[{"(", 
   RowBox[{"1", "+", "x"}], ")"}], " ", 
  RowBox[{"(", 
   RowBox[{"1", "+", "x", "+", 
    SuperscriptBox["x", "2"]}], ")"}]}]], "Output",
 ImageSize->{171, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->246347960]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->11535],

Cell["A univariate polynomial:", "ExampleText",
 CellID->242436578],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FactorSquareFree", "[", 
  RowBox[{
   RowBox[{"x", "^", "4"}], "-", 
   RowBox[{"9", 
    RowBox[{"x", "^", "3"}]}], "+", 
   RowBox[{"29", 
    RowBox[{"x", "^", "2"}]}], "-", 
   RowBox[{"39", "x"}], "+", "18"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->6972226],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "3"}], "+", "x"}], ")"}], "2"], " ", 
  RowBox[{"(", 
   RowBox[{"2", "-", 
    RowBox[{"3", " ", "x"}], "+", 
    SuperscriptBox["x", "2"]}], ")"}]}]], "Output",
 ImageSize->{137, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->45250527]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->606606440],

Cell["A multivariate polynomial:", "ExampleText",
 CellID->242121446],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FactorSquareFree", "[", 
  RowBox[{
   RowBox[{"x", "^", "5"}], "-", 
   RowBox[{
    RowBox[{"x", "^", "3"}], 
    RowBox[{"y", "^", "2"}]}], "-", 
   RowBox[{
    RowBox[{"x", "^", "2"}], 
    RowBox[{"y", "^", "3"}]}], "+", 
   RowBox[{"y", "^", "5"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->685335161],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"x", "-", "y"}], ")"}], "2"], " ", 
  RowBox[{"(", 
   RowBox[{
    SuperscriptBox["x", "3"], "+", 
    RowBox[{"2", " ", 
     SuperscriptBox["x", "2"], " ", "y"}], "+", 
    RowBox[{"2", " ", "x", " ", 
     SuperscriptBox["y", "2"]}], "+", 
    SuperscriptBox["y", "3"]}], ")"}]}]], "Output",
 ImageSize->{195, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->822871859]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->137525866],

Cell["A rational function:", "ExampleText",
 CellID->250609994],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FactorSquareFree", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"x", "^", "3"}], "+", 
      RowBox[{"x", "^", "2"}]}], ")"}], "/", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"x", "^", "2"}], "-", 
      RowBox[{"4", 
       RowBox[{"y", "^", "2"}]}]}], ")"}]}], "-", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"x", "+", "1"}], ")"}], "/", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"x", "^", "2"}], "-", 
      RowBox[{"4", 
       RowBox[{"y", "^", "2"}]}]}], ")"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->582901814],

Cell[BoxData[
 FractionBox[
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "x"}], ")"}], " ", 
   SuperscriptBox[
    RowBox[{"(", 
     RowBox[{"1", "+", "x"}], ")"}], "2"]}], 
  RowBox[{
   SuperscriptBox["x", "2"], "-", 
   RowBox[{"4", " ", 
    SuperscriptBox["y", "2"]}]}]]], "Output",
 ImageSize->{106, 36},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->263088134]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Options",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->26367],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Extension",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->4121],

Cell["\<\
By default algebraic number coefficients are treated as independent \
variables:\
\>", "ExampleText",
 CellID->367222654],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FactorSquareFree", "[", 
  RowBox[{
   RowBox[{"x", "^", "2"}], "+", 
   RowBox[{"2", 
    RowBox[{"Sqrt", "[", "2", "]"}], "x"}], "+", "2"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->532318586],

Cell[BoxData[
 RowBox[{"2", "+", 
  RowBox[{"2", " ", 
   SqrtBox["2"], " ", "x"}], "+", 
  SuperscriptBox["x", "2"]}]], "Output",
 ImageSize->{93, 17},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->27658353]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->6512979],

Cell[TextData[{
 "With ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Extension",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Extension"], "->", 
   ButtonBox["Automatic",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Automatic"]}]], "InlineFormula"],
 " algebraic dependencies between coefficients are recognized:"
}], "ExampleText",
 CellID->200276468],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FactorSquareFree", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"x", "^", "2"}], "+", 
    RowBox[{"2", 
     RowBox[{"Sqrt", "[", "2", "]"}], "x"}], "+", "2"}], ",", 
   RowBox[{"Extension", "\[Rule]", "Automatic"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->621074522],

Cell[BoxData[
 SuperscriptBox[
  RowBox[{"(", 
   RowBox[{
    SqrtBox["2"], "+", "x"}], ")"}], "2"]], "Output",
 ImageSize->{67, 23},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->333506410]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Modulus",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->29805],

Cell["Pull out multiple factors over the integers modulo 2:", "ExampleText",
 CellID->136667834],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FactorSquareFree", "[", 
  RowBox[{
   RowBox[{"1", "+", 
    RowBox[{"x", "^", "4"}]}], ",", 
   RowBox[{"Modulus", "\[Rule]", "2"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->159056695],

Cell[BoxData[
 SuperscriptBox[
  RowBox[{"(", 
   RowBox[{"1", "+", "x"}], ")"}], "4"]], "Output",
 ImageSize->{51, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->429699301]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Trig",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->32506],

Cell["Pull out multiple factors in a trigonometric expression:", "ExampleText",
 CellID->425343965],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FactorSquareFree", "[", 
  RowBox[{
   RowBox[{"1", "-", 
    RowBox[{"Cos", "[", 
     RowBox[{"2", "x"}], "]"}]}], ",", 
   RowBox[{"Trig", "\[Rule]", "True"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->715596512],

Cell[BoxData[
 RowBox[{"2", " ", 
  SuperscriptBox[
   RowBox[{"Sin", "[", "x", "]"}], "2"]}]], "Output",
 ImageSize->{62, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->153319162]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->25351],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FactorSquareFree",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FactorSquareFree"]], "InlineFormula"],
 " only pulls out multiple factors:"
}], "ExampleText",
 CellID->57323525],

Cell[BoxData[
 RowBox[{
  RowBox[{"f", "=", 
   RowBox[{
    RowBox[{"x", "^", "9"}], "+", 
    RowBox[{"9", 
     RowBox[{"x", "^", "8"}]}], "+", 
    RowBox[{"21", 
     RowBox[{"x", "^", "7"}]}], "-", 
    RowBox[{"27", 
     RowBox[{"x", "^", "6"}]}], "-", 
    RowBox[{"153", 
     RowBox[{"x", "^", "5"}]}], "-", 
    RowBox[{"81", 
     RowBox[{"x", "^", "4"}]}], "+", 
    RowBox[{"239", 
     RowBox[{"x", "^", "3"}]}], "+", 
    RowBox[{"207", 
     RowBox[{"x", "^", "2"}]}], "-", 
    RowBox[{"108", "x"}], "-", "108"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->42129648],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FactorSquareFree", "[", "f", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->848171248],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"3", "+", "x"}], ")"}], "3"], " ", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"-", "4"}], "+", 
    SuperscriptBox["x", "2"]}], ")"}], " ", 
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], "+", 
     SuperscriptBox["x", "2"]}], ")"}], "2"]}]], "Output",
 ImageSize->{170, 20},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->48961553]
}, Open  ]],

Cell["Factor gives a complete factorization:", "ExampleText",
 CellID->313938186],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Factor", "[", "f", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->851500563],

Cell[BoxData[
 RowBox[{
  RowBox[{"(", 
   RowBox[{
    RowBox[{"-", "2"}], "+", "x"}], ")"}], " ", 
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "x"}], ")"}], "2"], " ", 
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"1", "+", "x"}], ")"}], "2"], " ", 
  RowBox[{"(", 
   RowBox[{"2", "+", "x"}], ")"}], " ", 
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"3", "+", "x"}], ")"}], "3"]}]], "Output",
 ImageSize->{251, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->13875142]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->16702625],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Expand",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Expand"]], "InlineFormula"],
 " is effectively the inverse of ",
 Cell[BoxData[
  ButtonBox["FactorSquareFree",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FactorSquareFree"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->27222],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FactorSquareFree", "[", 
  RowBox[{
   RowBox[{"x", "^", "5"}], "-", 
   RowBox[{"x", "^", "3"}], "-", 
   RowBox[{"x", "^", "2"}], "+", "1"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->11536],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "x"}], ")"}], "2"], " ", 
  RowBox[{"(", 
   RowBox[{"1", "+", 
    RowBox[{"2", " ", "x"}], "+", 
    RowBox[{"2", " ", 
     SuperscriptBox["x", "2"]}], "+", 
    SuperscriptBox["x", "3"]}], ")"}]}]], "Output",
 ImageSize->{172, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->425160666]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Expand", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->28714],

Cell[BoxData[
 RowBox[{"1", "-", 
  SuperscriptBox["x", "2"], "-", 
  SuperscriptBox["x", "3"], "+", 
  SuperscriptBox["x", "5"]}]], "Output",
 ImageSize->{87, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->456383742]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->80768718],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FactorSquareFreeList",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FactorSquareFreeList"]], "InlineFormula"],
 " gives a list of factors:"
}], "ExampleText",
 CellID->289317600],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FactorSquareFreeList", "[", 
  RowBox[{
   RowBox[{"x", "^", "8"}], "+", 
   RowBox[{"11", 
    RowBox[{"x", "^", "7"}]}], "+", 
   RowBox[{"43", 
    RowBox[{"x", "^", "6"}]}], "+", 
   RowBox[{"59", 
    RowBox[{"x", "^", "5"}]}], "-", 
   RowBox[{"35", 
    RowBox[{"x", "^", "4"}]}], "-", 
   RowBox[{"151", 
    RowBox[{"x", "^", "3"}]}], "-", 
   RowBox[{"63", 
    RowBox[{"x", "^", "2"}]}], "+", 
   RowBox[{"81", "x"}], "+", "54"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->271634407],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"2", "+", "x"}], ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"3", "+", "x"}], ",", "3"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"-", "1"}], "+", 
      SuperscriptBox["x", "2"]}], ",", "2"}], "}"}]}], "}"}]], "Output",
 ImageSize->{281, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->43934735]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->302812401],

Cell[TextData[{
 "A univariate polynomial has multiple factors if an only if its ",
 Cell[BoxData[
  ButtonBox["Discriminant",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Discriminant"]], "InlineFormula"],
 " is zero:"
}], "ExampleText",
 CellID->399952287],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Discriminant", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"x", "^", "5"}], "-", 
    RowBox[{"x", "^", "3"}], "-", 
    RowBox[{"x", "^", "2"}], "+", "1"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->453958394],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->155211764]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FactorSquareFree", "[", 
  RowBox[{
   RowBox[{"x", "^", "5"}], "-", 
   RowBox[{"x", "^", "3"}], "-", 
   RowBox[{"x", "^", "2"}], "+", "1"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->671796515],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "x"}], ")"}], "2"], " ", 
  RowBox[{"(", 
   RowBox[{"1", "+", 
    RowBox[{"2", " ", "x"}], "+", 
    RowBox[{"2", " ", 
     SuperscriptBox["x", "2"]}], "+", 
    SuperscriptBox["x", "3"]}], ")"}]}]], "Output",
 ImageSize->{172, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->10332207]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Discriminant", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"x", "^", "5"}], "-", 
    RowBox[{"x", "^", "3"}], "-", 
    RowBox[{"x", "^", "2"}], "-", "1"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->48901133],

Cell[BoxData["7684"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->536179656]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FactorSquareFree", "[", 
  RowBox[{
   RowBox[{"x", "^", "5"}], "-", 
   RowBox[{"x", "^", "3"}], "-", 
   RowBox[{"x", "^", "2"}], "-", "1"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->731249562],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "1"}], "-", 
  SuperscriptBox["x", "2"], "-", 
  SuperscriptBox["x", "3"], "+", 
  SuperscriptBox["x", "5"]}]], "Output",
 ImageSize->{95, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->173252388]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->30111],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["FactorSquareFreeList",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FactorSquareFreeList"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Factor",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Factor"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["FactorTerms",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FactorTerms"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SquareFreeQ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SquareFreeQ"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->397]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->27604],

Cell[TextData[ButtonBox["Algebraic Operations on Polynomials",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/AlgebraicOperationsOnPolynomials"]], "Tutorials",\

 CellID->18564]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->93722742],

Cell[TextData[ButtonBox["Polynomial Factoring & Decomposition",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/PolynomialFactoring"]], "MoreAbout",
 CellID->131234235]
}, Open  ]],

Cell["New in 1  |  Last modified in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"FactorSquareFree - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 42, 19.7174120}", "context" -> "System`", 
    "keywords" -> {
     "multiple factors", "polynomial factors", "square-free factoring", 
      "square-free factorization", "square-free factors"}, "index" -> True, 
    "label" -> "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "FactorSquareFree[poly] pulls out any multiple factors in a polynomial. ",
     "synonyms" -> {"factor square free"}, "title" -> "FactorSquareFree", 
    "type" -> "Symbol", "uri" -> "ref/FactorSquareFree"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[9632, 262, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->268689014]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 30940, 1133}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2190, 54, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2813, 81, 4200, 83, 70, "ObjectNameGrid"],
Cell[7016, 166, 509, 15, 70, "Usage",
 CellID->30792]
}, Open  ]],
Cell[CellGroupData[{
Cell[7562, 186, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[8040, 203, 505, 18, 70, "Notes",
 CellID->30100],
Cell[8548, 223, 636, 21, 70, "Notes",
 CellID->26517],
Cell[9187, 246, 408, 11, 70, "Notes",
 CellID->82730543]
}, Closed]],
Cell[CellGroupData[{
Cell[9632, 262, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->268689014],
Cell[CellGroupData[{
Cell[10017, 276, 148, 5, 70, "ExampleSection",
 CellID->382362664],
Cell[10168, 283, 68, 1, 70, "ExampleText",
 CellID->73605572],
Cell[CellGroupData[{
Cell[10261, 288, 223, 7, 28, "Input",
 CellID->17656],
Cell[10487, 297, 443, 16, 39, "Output",
 CellID->1821628]
}, Open  ]],
Cell[10945, 316, 120, 3, 70, "ExampleDelimiter",
 CellID->6818],
Cell[11068, 321, 68, 1, 70, "ExampleText",
 CellID->340879707],
Cell[CellGroupData[{
Cell[11161, 326, 213, 7, 70, "Input",
 CellID->10884],
Cell[11377, 335, 415, 15, 39, "Output",
 CellID->246347960]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[11841, 356, 216, 7, 70, "ExampleSection",
 CellID->11535],
Cell[12060, 365, 67, 1, 70, "ExampleText",
 CellID->242436578],
Cell[CellGroupData[{
Cell[12152, 370, 297, 10, 70, "Input",
 CellID->6972226],
Cell[12452, 382, 382, 14, 39, "Output",
 CellID->45250527]
}, Open  ]],
Cell[12849, 399, 125, 3, 70, "ExampleDelimiter",
 CellID->606606440],
Cell[12977, 404, 69, 1, 70, "ExampleText",
 CellID->242121446],
Cell[CellGroupData[{
Cell[13071, 409, 341, 12, 70, "Input",
 CellID->685335161],
Cell[13415, 423, 497, 17, 39, "Output",
 CellID->822871859]
}, Open  ]],
Cell[13927, 443, 125, 3, 70, "ExampleDelimiter",
 CellID->137525866],
Cell[14055, 448, 63, 1, 70, "ExampleText",
 CellID->250609994],
Cell[CellGroupData[{
Cell[14143, 453, 595, 22, 70, "Input",
 CellID->582901814],
Cell[14741, 477, 446, 17, 57, "Output",
 CellID->263088134]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[15236, 500, 218, 7, 70, "ExampleSection",
 CellID->26367],
Cell[CellGroupData[{
Cell[15479, 511, 225, 7, 70, "ExampleSubsection",
 CellID->4121],
Cell[15707, 520, 131, 4, 70, "ExampleText",
 CellID->367222654],
Cell[CellGroupData[{
Cell[15863, 528, 226, 7, 70, "Input",
 CellID->532318586],
Cell[16092, 537, 259, 9, 38, "Output",
 CellID->27658353]
}, Open  ]],
Cell[16366, 549, 123, 3, 70, "ExampleDelimiter",
 CellID->6512979],
Cell[16492, 554, 362, 12, 70, "ExampleText",
 CellID->200276468],
Cell[CellGroupData[{
Cell[16879, 570, 299, 9, 70, "Input",
 CellID->621074522],
Cell[17181, 581, 242, 9, 44, "Output",
 CellID->333506410]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[17472, 596, 224, 7, 70, "ExampleSubsection",
 CellID->29805],
Cell[17699, 605, 96, 1, 70, "ExampleText",
 CellID->136667834],
Cell[CellGroupData[{
Cell[17820, 610, 220, 7, 70, "Input",
 CellID->159056695],
Cell[18043, 619, 228, 8, 36, "Output",
 CellID->429699301]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[18320, 633, 221, 7, 70, "ExampleSubsection",
 CellID->32506],
Cell[18544, 642, 99, 1, 70, "ExampleText",
 CellID->425343965],
Cell[CellGroupData[{
Cell[18668, 647, 248, 8, 70, "Input",
 CellID->715596512],
Cell[18919, 657, 235, 8, 36, "Output",
 CellID->153319162]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[19215, 672, 233, 7, 70, "ExampleSection",
 CellID->25351],
Cell[19451, 681, 225, 7, 70, "ExampleText",
 CellID->57323525],
Cell[19679, 690, 593, 21, 70, "Input",
 CellID->42129648],
Cell[CellGroupData[{
Cell[20297, 715, 112, 3, 70, "Input",
 CellID->848171248],
Cell[20412, 720, 477, 18, 41, "Output",
 CellID->48961553]
}, Open  ]],
Cell[20904, 741, 81, 1, 70, "ExampleText",
 CellID->313938186],
Cell[CellGroupData[{
Cell[21010, 746, 102, 3, 70, "Input",
 CellID->851500563],
Cell[21115, 751, 564, 21, 36, "Output",
 CellID->13875142]
}, Open  ]],
Cell[21694, 775, 124, 3, 70, "ExampleDelimiter",
 CellID->16702625],
Cell[21821, 780, 341, 12, 70, "ExampleText",
 CellID->27222],
Cell[CellGroupData[{
Cell[22187, 796, 223, 7, 70, "Input",
 CellID->11536],
Cell[22413, 805, 445, 16, 39, "Output",
 CellID->425160666]
}, Open  ]],
Cell[CellGroupData[{
Cell[22895, 826, 98, 3, 70, "Input",
 CellID->28714],
Cell[22996, 831, 272, 9, 36, "Output",
 CellID->456383742]
}, Open  ]],
Cell[23283, 843, 124, 3, 70, "ExampleDelimiter",
 CellID->80768718],
Cell[23410, 848, 226, 7, 70, "ExampleText",
 CellID->289317600],
Cell[CellGroupData[{
Cell[23661, 859, 525, 18, 70, "Input",
 CellID->271634407],
Cell[24189, 879, 547, 20, 39, "Output",
 CellID->43934735]
}, Open  ]],
Cell[24751, 902, 125, 3, 70, "ExampleDelimiter",
 CellID->302812401],
Cell[24879, 907, 262, 8, 70, "ExampleText",
 CellID->399952287],
Cell[CellGroupData[{
Cell[25166, 919, 250, 8, 70, "Input",
 CellID->453958394],
Cell[25419, 929, 158, 5, 36, "Output",
 CellID->155211764]
}, Open  ]],
Cell[CellGroupData[{
Cell[25614, 939, 227, 7, 70, "Input",
 CellID->671796515],
Cell[25844, 948, 444, 16, 39, "Output",
 CellID->10332207]
}, Open  ]],
Cell[CellGroupData[{
Cell[26325, 969, 249, 8, 70, "Input",
 CellID->48901133],
Cell[26577, 979, 161, 5, 36, "Output",
 CellID->536179656]
}, Open  ]],
Cell[CellGroupData[{
Cell[26775, 989, 227, 7, 70, "Input",
 CellID->731249562],
Cell[27005, 998, 290, 10, 36, "Output",
 CellID->173252388]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[27356, 1015, 311, 9, 70, "SeeAlsoSection",
 CellID->30111],
Cell[27670, 1026, 971, 35, 70, "SeeAlso",
 CellID->397]
}, Open  ]],
Cell[CellGroupData[{
Cell[28678, 1066, 314, 9, 70, "TutorialsSection",
 CellID->27604],
Cell[28995, 1077, 180, 4, 70, "Tutorials",
 CellID->18564]
}, Open  ]],
Cell[CellGroupData[{
Cell[29212, 1086, 318, 9, 70, "MoreAboutSection",
 CellID->93722742],
Cell[29533, 1097, 167, 3, 70, "MoreAbout",
 CellID->131234235]
}, Open  ]],
Cell[29715, 1103, 50, 0, 70, "History"],
Cell[29768, 1105, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

