(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    273636,       7593]
NotebookOptionsPosition[    247302,       6690]
NotebookOutlinePosition[    249513,       6740]
CellTagsIndexPosition[    249426,       6735]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Manipulating Numerical Data" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/ManipulatingNumericalData"], "Curve Fitting" :> 
          Documentation`HelpLookup["paclet:tutorial/CurveFitting"], 
          "Unconstrained Optimization" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/UnconstrainedOptimizationOverview"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Manipulating Numerical Data\"\>", 
       2->"\<\"Curve Fitting\"\>", 
       3->"\<\"Unconstrained Optimization\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"FindMinimum" :> 
          Documentation`HelpLookup["paclet:ref/FindMinimum"], "Fit" :> 
          Documentation`HelpLookup["paclet:ref/Fit"], "NMinimize" :> 
          Documentation`HelpLookup["paclet:ref/NMinimize"], "LeastSquares" :> 
          Documentation`HelpLookup["paclet:ref/LeastSquares"], 
          "Interpolation" :> 
          Documentation`HelpLookup["paclet:ref/Interpolation"], 
          "ListSurfacePlot3D" :> 
          Documentation`HelpLookup["paclet:ref/ListSurfacePlot3D"], 
          "ChiSquareDistribution" :> 
          Documentation`HelpLookup["paclet:ref/ChiSquareDistribution"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"FindMinimum\"\>", 
       2->"\<\"Fit\"\>", 3->"\<\"NMinimize\"\>", 4->"\<\"LeastSquares\"\>", 
       5->"\<\"Interpolation\"\>", 6->"\<\"ListSurfacePlot3D\"\>", 
       7->"\<\"ChiSquareDistribution\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Curve Fitting & Approximate Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/CurveFittingAndApproximateFunctions"], 
          "Data Transforms and Smoothing" :> 
          Documentation`HelpLookup["paclet:guide/DataTransformsAndSmoothing"],
           "Matrix-Based Minimization" :> 
          Documentation`HelpLookup["paclet:guide/MatrixBasedMinimization"], 
          "Numerical Data" :> 
          Documentation`HelpLookup["paclet:guide/NumericalData"], 
          "Optimization" :> 
          Documentation`HelpLookup["paclet:guide/Optimization"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"], 
          "New in 6.0: Numerical Data Handling" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60NumericalDataHandling"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Curve Fitting & Approximate Functions\"\>", 
       2->"\<\"Data Transforms and Smoothing\"\>", 
       3->"\<\"Matrix-Based Minimization\"\>", 4->"\<\"Numerical Data\"\>", 
       5->"\<\"Optimization\"\>", 
       6->"\<\"New in 6.0: Mathematics & Algorithms\"\>", 
       7->"\<\"New in 6.0: Numerical Data Handling\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["FindFit", "ObjectName"], Cell[BoxData[
     TooltipBox[
      TogglerBox[Dynamic[
        FrontEnd`CurrentValue[
         FrontEnd`EvaluationNotebook[], {
         TaggingRules, "ModificationHighlight"}]], {True->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVVly2zAMDQhxEylSsizLaxJn73KafvQAuUDv/1eApJuosceeTDAWF4B8
eFgk71///HoVV1dXK3p+0wNfIzdCLL4I6p1sYBYuPy3SI2AmRBpuaE/TPRsE
3Isg8mrB9l6ILQhhLkbd93MBuwVxAthdZwPr5wVVzBc8wAvMX8TFqPkqE8q6
rDCsKPjZ/g1guz+PKvLFfGsDU1R4A8zDhgLangX9x/WGM7A1qSQ7rraYkaLf
LHZ5FWY9H7rew/n4/6vWPKT2mXFBaH/NGejzqi/VnFEFJ2Rb1Lo+jtofllPD
RVJjdZLr51Eb/ED1K+QI109LcIg+sdSIWqa5g85L6MiNoRloTws+pdBKiFGl
mx3N0jcAlcsKeFivVkteVOgMSNsC18sYS/iIMdZoK43WSdQVK1SLlYy09bZ2
aEO+aoPiACNak0EBbtfLRIz9KJRcroaSKwE11BprpRHqiJ0BrxtJG4pLYqWc
RZO4BQwSbGT4rFivaLjjQdscjSvlih48Mu+WDisOBcB5ehgeDEZQlikkKh3b
KA1FMTDg9yFxLSe4XC37begHTDh5seS1I/gaW025Yo/yUFbmSalShzon1Kch
JTNn2hKS1hyWVIkf5n2LHEQgt0zeUNTkMaYrDdkMZdo3fJCDHx7hxzAC+8rs
XYBokTPfEJk6JVlZx3Yl0UnTYMPvSUt6zghf8dhIi5VDnwO+Hcb1MHK1msK+
oy5qkYS7wVC3cfiq8ojBcMtFMJXnDiIcLPFJClDFGkJBhXGEZV6VcCQFoey7
tyvBF6lw0uKH6nyQ5fqhrKx/Kxe8+3coucpGN4U5+RYux7vnhGO5P7hKenqz
xSO3ivjisf345fg5Pt1ScCRkkzR1U1Dur+NCec4LZ49Ynx9P04GmOvlnqcKZ
79BfJfcjkg==
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlLlKQ0EYRi/iA9gpWIgvIXbjM1iKWAjW2vgeZnWN+AgWZhEULCxELFyi
ZnfJavbeJOr/33xXcj8CNnY5EA7k3GWYmTuzq+uLa2OWZS38/MYt0DbDHH8Q
8uTB64bRNrmsQkaPRoQSGX1ZqZPR9/eEazL6klIlowf8wiUZ3ecV3snowYDw
SEafUmJk9HmlQh4cX4GM7vUId2T0VFIho28or2T0w5DQIKN3PoUaGX1FaZLR
0ymh6sz7hHLk9PMzoU5Gz6SFmvFsCbd9Xzj9qyc0nfe49xfuTyYUEzoQbmh8
+N8e37HTbVpk9MSzUCajbypFMvq0UiOjzyklMnqvKzRMJCwU+/vjzemTSpns
nh97feJmd0e4cvppTKiQ0bsdoW5ecorZDgpPtD6xqFA2fp+Qp/vtdUj1v797
Oqdwnb2Ov++fUVpk9PCJUCC7n4t5wz5xj9/+rj6ceXbPP86NbEb58/k56qP+
n/0bdGyYqQ==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}], False->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVdd22zAMDUBxSaJm5HgljrPTJN177/nWh35AfqD//1aAVGrrJLbbk+BI
FEmQFxcXlDQ9//3zHNfW1gZ0/6IbbsYOhahvCGrO9qHM/mnhmMMLoKv0RD6V
cPyZHTTcDI5NkYnQq4UQ0BdiAkKYZagl1OOw5fWtcgyvPp7AnefsGG0DBIeA
9RZVrNfcwBasb4nlbP9u4cbc/XD/zXcIgC2qYYazJTQiGSYHS0Hr8Yu5LV/h
ePvR5oMOKswAQ0M8R5OloFMWSNRE+IAVeALv38KX26wAV1uU5Ojv16PQy8q+
gMP+9ABW5C9KbkZi6qv19DGc/ID9b+CrZTjbbVagH3p91oIcL+GZ6JDNUet4
HpWUXxx5BaWZxRh1FQgH6JqoKcarF/2/dblez7IE0XmWGlFL/6ygchIqCmPo
CTSmDq9SaCUUhfI7K3pKlwJESZiAveHGxoA7ESYGpM2B62WMJXzEoojRRhpt
IlFHPKFyjGRBQ2fjBG0WttpMcYIFWhNAAXaHA0+M4yiUXK6UxJWAGmKNsdII
cYGVAadTSQPKS2KkEovGc8swk2ALhg8Tww1qdrjRNmSTtOUqHDhk3jktVpwK
QOLoZngwWICyTMFTqdhHMrQTDQPeazzXdgWXK+e4KV3AhH0US1Ergo8x16QV
R5QXZWWeJJW6qLNHPWq8mEFpS0hac1pSeX4YxjlyEhmFZfKGsqaIhd+Sks+Q
0i7lhZx8cwYPmx5wrMA+yaCwyMqnRCb2IiubsF9JTKRJMeX3JKd5VoS3OEyl
xShBFxLebXrDpsfVSlv2FZ2iHMn4NBg6bZy+ihxiZvjIFWAixyeIcLDNT1KC
qogha1Gh14NB6LXpSEpC2bm3y8O3FmHniF9U55INhnttz7pZuWDu79BqFZxJ
F2bhWzjo7Zx6HMvng6ukuztzvGJXa66NmF/+crzrHe1ScmTkk/SouqB8vq42
0jl0EnuF9/RsMR1Io4U/S5Wt+A79AaDyK/w=
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlUtrE1EYhgfxBwguFBTEvyAiFhfzHxRFvC0E17rQhYLiT7D3u/gTXNg2
3sBFFyIutE1qm17UZJKZZJK0KFjBtur7Td4jnhe7dTMPhAfyQObMnO9Mjl69
fvraniAI7v/+7A3IRri7/8VGWJwHVTH72qohZp+ZBnUx+yWjJWYfHwNvxOwX
jKaYfXAAzIrZ+/tARcw+NAhKYvaTRiJmP2gUxH+vLxKzl5cMMXtfL3gvZn84
Cdpi9hvGJzH7yxegJWa/YnTE7Mtl0HTPfZ/xyPWVZZCGvQ/Au65fSV9aNMLJ
CfDW9a0fIBWz/9wBHbcOf/74PX83W99j128ZNTH71BMQidmfFkAiZj9irIvZ
Fz+AWMx+yEjF7KeMVOw/32x/iuHoCHjt+s42aIfTU6DWna/Prm9vgVb4cc0I
h4fAgusZ62L2A0YsZj9h1MX+9bN9LHfP75zrZ42a2L9/3+wXjVTM7q8jm9NE
5q8wA+JwoB/8eX+eNxIx+00jErP3GA2xvy7uG88R+z2jI2Y/bLTE/v1l76WG
mxP/+r7ZN7+Bupj9mFERsz9/Bppi9myeEzH7QgnUxOy3jYaY/a7RFrNfNtpi
9nNGXcw+NgoaYnbOk2///PJ/bXXFcP37JojF7CPDIBazf/0CKmL2+TlQEbPf
MZpi9olxkIrZS0UQidn3G1Ux+3EjErOfMapi9mBX8p73vOc973n/P/0XnFJ2
Rg==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}]}, "\"\<Click me\>\""],
      "Toggle changes in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["FindFit",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/FindFit"], "[", 
       RowBox[{
        StyleBox["data", "TI"], ",", 
        StyleBox["expr", "TI"], ",", 
        StyleBox["pars", "TI"], ",", 
        StyleBox["vars", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]finds numerical values of the parameters ",
     Cell[BoxData[
      StyleBox["pars", "TI"]], "InlineFormula"],
     " that make ",
     Cell[BoxData[
      StyleBox["expr", "TI"]], "InlineFormula"],
     " give a best fit to ",
     Cell[BoxData[
      StyleBox["data", "TI"]], "InlineFormula"],
     " as a function of ",
     Cell[BoxData[
      StyleBox["vars", "TI"]], "InlineFormula"],
     ". The data can have the form ",
     Cell[BoxData[
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          SubscriptBox[
           StyleBox["x", "TI"], 
           StyleBox["1", "TR"]], ",", 
          SubscriptBox[
           StyleBox["y", "TI"], 
           StyleBox["1", "TR"]], ",", 
          StyleBox["\[Ellipsis]", "TR"], ",", 
          SubscriptBox[
           StyleBox["f", "TI"], 
           StyleBox["1", "TR"]]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          SubscriptBox[
           StyleBox["x", "TI"], 
           StyleBox["2", "TR"]], ",", 
          SubscriptBox[
           StyleBox["y", "TI"], 
           StyleBox["2", "TR"]], ",", 
          StyleBox["\[Ellipsis]", "TR"], ",", 
          SubscriptBox[
           StyleBox["f", "TI"], 
           StyleBox["2", "TR"]]}], "}"}], ",", 
        StyleBox["\[Ellipsis]", "TR"]}], "}"}]], "InlineFormula"],
     ", where the number of coordinates ",
     Cell[BoxData[
      StyleBox["x", "TI"]], "InlineFormula"],
     ", ",
     Cell[BoxData[
      StyleBox["y", "TI"]], "InlineFormula"],
     ", \[Ellipsis] is equal to the number of variables in the list ",
     Cell[BoxData[
      StyleBox["vars", "TI"]], "InlineFormula"],
     ". The data can also be of the form ",
     Cell[BoxData[
      RowBox[{"{", 
       RowBox[{
        SubscriptBox[
         StyleBox["f", "TI"], 
         StyleBox["1", "TR"]], ",", 
        SubscriptBox[
         StyleBox["f", "TI"], 
         StyleBox["2", "TR"]], ",", 
        StyleBox["\[Ellipsis]", "TR"]}], "}"}]], "InlineFormula"],
     ", with a single coordinate assumed to take values ",
     Cell[BoxData["1"], "InlineFormula"],
     ", ",
     Cell[BoxData["2"], "InlineFormula"],
     ", ",
     Cell[BoxData[
      StyleBox["\[Ellipsis]", "TR"]], "InlineFormula"],
     ". "
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["FindFit",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/FindFit"], "[", 
       RowBox[{
        StyleBox["data", "TI"], ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["expr", "TI"], ",", 
          StyleBox["cons", "TI"]}], "}"}], ",", 
        StyleBox["pars", "TI"], ",", 
        StyleBox["vars", "TI"]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]finds a best fit subject to the parameter constraints \
",
     Cell[BoxData[
      StyleBox["cons", "TI"]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->19752]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FindFit",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindFit"]], "InlineFormula"],
 " returns a list of replacements for ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["par", "TI"], 
   StyleBox["1", "TR"]]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["par", "TI"], 
   StyleBox["2", "TR"]]], "InlineFormula"],
 ", \[Ellipsis]. "
}], "Notes",
 CellID->8382],

Cell[TextData[{
 "The expression ",
 Cell[BoxData[
  StyleBox["expr", "TI"]], "InlineFormula"],
 " must yield a numerical value when ",
 Cell[BoxData[
  StyleBox["pars", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["vars", "TI"]], "InlineFormula"],
 " are all numerical. "
}], "Notes",
 CellID->13616],

Cell[TextData[{
 "The expression ",
 Cell[BoxData[
  StyleBox["expr", "TI"]], "InlineFormula"],
 " can depend either linearly or nonlinearly on the ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["par", "TI"], 
   StyleBox["i", "TI"]]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->28781],

Cell[TextData[{
 "In the linear case, ",
 Cell[BoxData[
  ButtonBox["FindFit",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindFit"]], "InlineFormula"],
 " finds a globally optimal fit. "
}], "Notes",
 CellID->8808],

Cell["\<\
In the nonlinear case, it finds in general only a locally optimal fit. \
\>", "Notes",
 CellID->14609],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["FindFit",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FindFit"], "[", 
   RowBox[{
    StyleBox["data", "TI"], ",", 
    StyleBox["expr", "TI"], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        SubscriptBox[
         StyleBox["par", "TI"], 
         StyleBox["1", "TR"]], ",", 
        SubscriptBox[
         StyleBox["p", "TI"], 
         StyleBox["1", "TR"]]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        SubscriptBox[
         StyleBox["par", "TI"], 
         StyleBox["2", "TR"]], ",", 
        SubscriptBox[
         StyleBox["p", "TI"], 
         StyleBox["2", "TR"]]}], "}"}], ",", 
      StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
    StyleBox["vars", "TI"]}], "]"}]], "InlineFormula", "TemplateInclusion"],
 " starts the search for a fit with ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    RowBox[{
     SubscriptBox[
      StyleBox["par", "TI"], 
      StyleBox["1", "TR"]], "->", 
     SubscriptBox[
      StyleBox["p", "TI"], 
      StyleBox["1", "TR"]]}], ",", 
    RowBox[{
     SubscriptBox[
      StyleBox["par", "TI"], 
      StyleBox["2", "TR"]], "->", 
     SubscriptBox[
      StyleBox["p", "TI"], 
      StyleBox["2", "TR"]]}], ",", 
    StyleBox["\[Ellipsis]", "TR"]}], "}"}]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->10299],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FindFit",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindFit"]], "InlineFormula"],
 " by default finds a least\[Hyphen]squares fit. "
}], "Notes",
 CellID->22146],

Cell[TextData[{
 "The option ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["NormFunction",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NormFunction"], "->", 
   StyleBox["f", "TI"]}]], "InlineFormula"],
 " specifies that the norm ",
 Cell[BoxData[
  RowBox[{
   StyleBox["f", "TI"], "[", 
   StyleBox["residual", "TI"], "]"}]], "InlineFormula"],
 " should be minimized. "
}], "Notes",
 CellID->22732],

Cell[TextData[{
 "The constraints ",
 Cell[BoxData[
  StyleBox["cons", "TI"]], "InlineFormula"],
 " can contain equations, inequalities or logical combinations of these. "
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->19298],

Cell["The following options can be given: ", "Notes",
 CellID->26940],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["AccuracyGoal",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/AccuracyGoal"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "the accuracy sought ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["EvaluationMonitor",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/EvaluationMonitor"], 
    ButtonBox["None",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/None"], Cell[TextData[{
     "expression to evaluate whenever ",
     Cell[BoxData[
      StyleBox["expr", "TI"]], "InlineFormula"],
     " is evaluated "
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["MaxIterations",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/MaxIterations"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "maximum number of iterations to use ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["Method",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Method"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "method to use ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["NormFunction",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/NormFunction"], 
    ButtonBox["Norm",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Norm"], Cell[
    "the norm to minimize ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["PrecisionGoal",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/PrecisionGoal"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "the precision sought ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["StepMonitor",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/StepMonitor"], 
    ButtonBox["None",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/None"], Cell[
    "expression to evaluate whenever a step is taken ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["WorkingPrecision",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/WorkingPrecision"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "the precision used in internal computations ", "TableText"]}
  },
  GridBoxAlignment->{
   "Columns" -> {Left, Left, {Left}}, "ColumnsIndexed" -> {}, 
    "Rows" -> {{Baseline}}, "RowsIndexed" -> {}}]], "3ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, None, None, None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{True, True, True, True, True, True, True, True, True}}}},
 CellID->30188],

Cell[TextData[{
 "The default settings for ",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " are ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["WorkingPrecision",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/WorkingPrecision"], "/", "2"}]], 
  "InlineFormula"],
 ". "
}], "Notes",
 CellID->18960],

Cell[TextData[{
 "The settings for ",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " specify the number of digits to seek in both the values of the parameters \
returned, and the value of the ",
 Cell[BoxData[
  ButtonBox["NormFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NormFunction"]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->637],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FindFit",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindFit"]], "InlineFormula"],
 " continues until either of the goals specified by ",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 " or ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " is achieved. "
}], "Notes",
 CellID->4185],

Cell[TextData[{
 "Possible settings for ",
 Cell[BoxData[
  ButtonBox["Method",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Method"]], "InlineFormula"],
 " include ",
 Cell[BoxData["\"\<ConjugateGradient\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<Gradient\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<LevenbergMarquardt\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<Newton\>\""], "InlineFormula"],
 " and ",
 Cell[BoxData["\"\<QuasiNewton\>\""], "InlineFormula"],
 ", with the default being ",
 Cell[BoxData[
  ButtonBox["Automatic",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->24902]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->348580231],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(1)", "ExampleCount"]
}], "ExampleSection",
 CellID->907094202],

Cell["Find a nonlinear fit to a list of primes:", "ExampleText",
 CellID->6495732],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"Prime", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "20"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->425550223],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "2", ",", "3", ",", "5", ",", "7", ",", "11", ",", "13", ",", "17", ",", 
   "19", ",", "23", ",", "29", ",", "31", ",", "37", ",", "41", ",", "43", 
   ",", "47", ",", "53", ",", "59", ",", "61", ",", "67", ",", "71"}], 
  "}"}]], "Output",
 ImageSize->{483, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->673387210]
}, Open  ]],

Cell[TextData[{
 "Fit the best-fit parameters ",
 Cell[BoxData[
  FormBox["a", TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox["b", TraditionalForm]], "InlineMath"],
 ", and ",
 Cell[BoxData[
  FormBox["c", TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->32517813],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindFit", "[", 
  RowBox[{"%", ",", 
   RowBox[{"a", " ", "x", " ", 
    RowBox[{"Log", "[", 
     RowBox[{"b", "+", 
      RowBox[{"c", " ", "x"}]}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->9895969],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"a", "\[Rule]", "1.4207557242233555`"}], ",", 
   RowBox[{"b", "\[Rule]", "1.655581377626135`"}], ",", 
   RowBox[{"c", "\[Rule]", "0.5346453120672511`"}]}], "}"}]], "Output",
 ImageSize->{264, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->179251908]
}, Open  ]],

Cell["Evaluate the fitted function:", "ExampleText",
 CellID->169120],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"a", " ", "x", " ", 
     RowBox[{"Log", "[", 
      RowBox[{"b", "+", 
       RowBox[{"c", " ", "x"}]}], "]"}]}], "/.", "%"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->613462901],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1.1138796622480749`", ",", "2.8483920410655297`", ",", 
   "5.036205900088848`", ",", "7.578104935694751`", ",", 
   "10.409111411867393`", ",", "13.483671381984252`", ",", 
   "16.768224232631617`", ",", "20.237094693206025`", ",", 
   "23.87004944943169`", ",", "27.65075855256118`"}], "}"}]], "Output",
 ImageSize->{354, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->104697178]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->456238338],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Simple Exponential Fitting",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->11860700],

Cell["Fit data to a model of exponential decay:", "ExampleText",
 CellID->261070299],

Cell[BoxData[
 RowBox[{
  RowBox[{"data", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1.0", ",", "12."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1.9", ",", "10."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2.6", ",", "8.2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"3.4", ",", "6.9"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"5.0", ",", "5.9"}], "}"}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->613023938],

Cell[BoxData[
 RowBox[{
  RowBox[{"model", "=", 
   RowBox[{"a", " ", 
    RowBox[{"Exp", "[", 
     RowBox[{
      RowBox[{"-", "k"}], " ", "t"}], "]"}]}]}], ";"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->211120279],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"fit", "=", 
  RowBox[{"FindFit", "[", 
   RowBox[{"data", ",", "model", ",", 
    RowBox[{"{", 
     RowBox[{"a", ",", "k"}], "}"}], ",", "t"}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->932959552],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"a", "\[Rule]", "14.388856945488829`"}], ",", 
   RowBox[{"k", "\[Rule]", "0.19820818668792958`"}]}], "}"}]], "Output",
 ImageSize->{181, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->304977752]
}, Open  ]],

Cell["The resulting model function:", "ExampleText",
 CellID->71387768],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"modelf", "=", 
  RowBox[{"Function", "[", 
   RowBox[{
    RowBox[{"{", "t", "}"}], ",", 
    RowBox[{"Evaluate", "[", 
     RowBox[{"model", "/.", "fit"}], "]"}]}], "]"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->16189946],

Cell[BoxData[
 RowBox[{"Function", "[", 
  RowBox[{
   RowBox[{"{", "t", "}"}], ",", 
   RowBox[{"14.388856945488829`", " ", 
    SuperscriptBox["\[ExponentialE]", 
     RowBox[{
      RowBox[{"-", "0.19820818668792958`"}], " ", "t"}]]}]}], "]"}]], "Output",\

 ImageSize->{218, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->10060483]
}, Open  ]],

Cell["Show the data with the model:", "ExampleText",
 CellID->629282411],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"modelf", "[", "t", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "6"}], "}"}], ",", 
   RowBox[{"Epilog", "\[Rule]", 
    RowBox[{"Map", "[", 
     RowBox[{"Point", ",", "data"}], "]"}]}]}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->83329518],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 106},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->302447679]
}, Open  ]],

Cell["Separate the time from response:", "ExampleText",
 CellID->368492844],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"tl", ",", "yl"}], "}"}], "=", 
   RowBox[{"Transpose", "[", "data", "]"}]}], ";"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->301688966],

Cell["\<\
Looking at residuals gives a good indication whether it is a good fit:\
\>", "ExampleText",
 CellID->112903641],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"residuals", "=", 
  RowBox[{"yl", "-", 
   RowBox[{"Map", "[", 
    RowBox[{"modelf", ",", "tl"}], "]"}]}]}]], "Input",
 CellLabel->"In[7]:=",
 CellID->829000504],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.19827275908539654`", ",", "0.1264592364406809`", ",", 
   RowBox[{"-", "0.3944169313066155`"}], ",", 
   RowBox[{"-", "0.434184668104475`"}], ",", "0.5589986700962273`"}], 
  "}"}]], "Output",
 ImageSize->{359, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->150673020]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{"residuals", ",", 
   RowBox[{"Filling", "\[Rule]", "Axis"}], ",", 
   RowBox[{"DataRange", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"Min", "[", "tl", "]"}], ",", 
      RowBox[{"Max", "[", "tl", "]"}]}], "}"}]}], ",", 
   RowBox[{"PlotRange", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"0", ",", "6"}], "}"}], ",", "All"}], "}"}]}]}], 
  "]"}]], "Input",
 CellLabel->"In[8]:=",
 CellID->619343705],

Cell[BoxData[
 GraphicsBox[
  GraphicsComplexBox[{{1., 0.19827275908539654`}, {2., 0.1264592364406809}, {
   3., -0.3944169313066155}, {4., -0.434184668104475}, {5., 
   0.5589986700962273}, {1., 0.}, {2., 0.}, {3., 0.}, {4., 0.}, {5., 0.}, {1.,
    0.19827275908539654`}, {2., 0.1264592364406809}, {
   3., -0.3944169313066155}, {4., -0.434184668104475}, {5., 
   0.5589986700962273}}, {{{}, {}, {}, 
     {Hue[0.67, 0.6, 0.6], Opacity[0.2], LineBox[{8, 3}], LineBox[{9, 4}]}, 
     {Hue[0.67, 0.6, 0.6], Opacity[0.2], LineBox[{6, 1}], LineBox[{7, 2}], 
      LineBox[{10, 5}]}}, {{}, 
     {Hue[0.67, 0.6, 0.6], PointBox[{11, 12, 13, 14, 15}]}, {}}}],
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotRange->{{0, 6}, All},
  PlotRangeClipping->True,
  PlotRangePadding->{Automatic, Automatic}]], "Output",
 ImageSize->{184, 97},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->288011517]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->74394520],

Cell["\<\
Use a linear fit on the logarithm of the data for a model of exponential \
decay:\
\>", "ExampleText",
 CellID->485657525],

Cell[BoxData[{
 RowBox[{
  RowBox[{"data", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1.0", ",", "12."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1.9", ",", "10."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2.6", ",", "8.2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"3.4", ",", "6.9"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"5.0", ",", "5.9"}], "}"}]}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"logdata", "=", 
   RowBox[{"Apply", "[", 
    RowBox[{
     RowBox[{"Function", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"t", ",", "v"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"t", ",", 
         RowBox[{"Log", "[", "v", "]"}]}], "}"}]}], "]"}], ",", "data", ",", 
     RowBox[{"{", "1", "}"}]}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->109105924],

Cell["\<\
The logarithm of the exponential model is a linear model for the logarithm of \
the data:\
\>", "ExampleText",
 CellID->160429556],

Cell[BoxData[
 RowBox[{
  RowBox[{"logmodel", "=", 
   RowBox[{"la", "-", 
    RowBox[{"k", " ", "t"}]}]}], ";"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->648832529],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"lfit", "=", 
  RowBox[{"FindFit", "[", 
   RowBox[{"logdata", ",", "logmodel", ",", 
    RowBox[{"{", 
     RowBox[{"la", ",", "k"}], "}"}], ",", "t"}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->164817600],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"la", "\[Rule]", "2.625519881760822`"}], ",", 
   RowBox[{"k", "\[Rule]", "0.1819783992256813`"}]}], "}"}]], "Output",
 ImageSize->{188, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->349688790]
}, Open  ]],

Cell["\<\
The exponential model formed by exponentiating the model for the logarithm:\
\>", "ExampleText",
 CellID->189393166],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Exp", "[", 
   RowBox[{"logmodel", "/.", "lfit"}], "]"}], "/.", 
  RowBox[{
   RowBox[{"Exp", "[", 
    RowBox[{"a_", "+", 
     RowBox[{"b_", " ", "t"}]}], "]"}], "\[RuleDelayed]", 
   RowBox[{
    RowBox[{"Exp", "[", "a", "]"}], 
    RowBox[{"Exp", "[", 
     RowBox[{"b", " ", "t"}], "]"}]}]}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->70031323],

Cell[BoxData[
 RowBox[{"13.811752798255432`", " ", 
  SuperscriptBox["\[ExponentialE]", 
   RowBox[{
    RowBox[{"-", "0.1819783992256813`"}], " ", "t"}]]}]], "Output",
 ImageSize->{116, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->288413531]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->270889856],

Cell["Exponential fit as parameters in a differential equation:", \
"ExampleText",
 CellID->703708898],

Cell[BoxData[
 RowBox[{
  RowBox[{"data", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1.0", ",", "12."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1.9", ",", "10."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2.6", ",", "8.2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"3.4", ",", "6.9"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"5.0", ",", "5.9"}], "}"}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->93684799],

Cell["The model function is the solution a differential equation:", \
"ExampleText",
 CellID->1703653],

Cell[BoxData[
 RowBox[{
  RowBox[{"demodel", "[", 
   RowBox[{
    RowBox[{"a_", "?", "NumberQ"}], ",", 
    RowBox[{"k_", "?", "NumberQ"}]}], "]"}], ":=", 
  RowBox[{"First", "[", 
   RowBox[{"x", "/.", 
    RowBox[{"NDSolve", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{
           RowBox[{"x", "'"}], "[", "t", "]"}], "+", 
          RowBox[{"k", " ", 
           RowBox[{"x", "[", "t", "]"}]}]}], "\[Equal]", "0"}], ",", 
        RowBox[{
         RowBox[{"x", "[", "0", "]"}], "\[Equal]", "a"}]}], "}"}], ",", "x", 
      ",", 
      RowBox[{"{", 
       RowBox[{"t", ",", "0", ",", "6"}], "}"}]}], "]"}]}], "]"}]}]], "Input",\

 CellLabel->"In[2]:=",
 CellID->507606713],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Timing", "[", 
   RowBox[{"FindFit", "[", 
    RowBox[{"data", ",", 
     RowBox[{
      RowBox[{"demodel", "[", 
       RowBox[{"a", ",", "k"}], "]"}], "[", "t", "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"a", ",", "k"}], "}"}], ",", "t"}], "]"}], "]"}], "//", 
  "Quiet"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->63559203],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.8430000000000133`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"a", "\[Rule]", "14.388864728714069`"}], ",", 
     RowBox[{"k", "\[Rule]", "0.19820841641073797`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{241, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->407759799]
}, Open  ]],

Cell["Using caching is faster: ", "ExampleText",
 CellID->174378436],

Cell[BoxData[
 RowBox[{
  RowBox[{"demodel", "[", 
   RowBox[{
    RowBox[{"a_", "?", "NumberQ"}], ",", 
    RowBox[{"k_", "?", "NumberQ"}]}], "]"}], ":=", 
  RowBox[{
   RowBox[{"demodel", "[", 
    RowBox[{"a", ",", "k"}], "]"}], "=", 
   RowBox[{"First", "[", 
    RowBox[{"x", "/.", 
     RowBox[{"NDSolve", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{
          RowBox[{
           RowBox[{
            RowBox[{"x", "'"}], "[", "t", "]"}], "+", 
           RowBox[{"k", " ", 
            RowBox[{"x", "[", "t", "]"}]}]}], "\[Equal]", "0"}], ",", 
         RowBox[{
          RowBox[{"x", "[", "0", "]"}], "\[Equal]", "a"}]}], "}"}], ",", "x", 
       ",", 
       RowBox[{"{", 
        RowBox[{"t", ",", "0", ",", "6"}], "}"}]}], "]"}]}], 
    "]"}]}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->245175184],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Timing", "[", 
   RowBox[{"FindFit", "[", 
    RowBox[{"data", ",", 
     RowBox[{
      RowBox[{"demodel", "[", 
       RowBox[{"a", ",", "k"}], "]"}], "[", "t", "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"a", ",", "k"}], "}"}], ",", "t"}], "]"}], "]"}], "//", 
  "Quiet"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->339205902],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.15699999999999556`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"a", "\[Rule]", "14.388864728714069`"}], ",", 
     RowBox[{"k", "\[Rule]", "0.19820841641073797`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{241, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->371695152]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Constraints and Starting Values",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->30298558],

Cell["Find a constrained fit:", "ExampleText",
 CellID->197867878],

Cell[BoxData[
 RowBox[{
  RowBox[{"model", "=", 
   RowBox[{"a", "  ", 
    RowBox[{"Cos", "[", 
     RowBox[{"\[Omega]", " ", "t"}], "]"}]}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->15080643],

Cell[BoxData[
 RowBox[{
  RowBox[{"data", "=", 
   RowBox[{
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"t", ",", 
        RowBox[{"Cos", "[", 
         RowBox[{"2.1", " ", "t"}], "]"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"t", ",", "0", ",", "10", ",", ".25"}], "}"}]}], "]"}], "+", 
    RowBox[{"RandomReal", "[", 
     RowBox[{".1", ",", "41"}], "]"}]}]}], ";"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->218875459],

Cell["\<\
Fit to the model with positive amplitude and frequency between 1 and 2:\
\>", "ExampleText",
 CellID->161917982],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"fit", "=", 
  RowBox[{"FindFit", "[", 
   RowBox[{"data", ",", 
    RowBox[{"{", 
     RowBox[{"model", ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"a", ">", "0"}], ",", 
        RowBox[{"1", "<", "\[Omega]", "<", "2"}]}], "}"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"a", ",", "\[Omega]"}], "}"}], ",", "t"}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->74490737],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"a", "\[Rule]", "0.8635662050477086`"}], ",", 
   RowBox[{"\[Omega]", "\[Rule]", "1.9999999993911868`"}]}], "}"}]], "Output",\

 ImageSize->{146, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->894649802]
}, Open  ]],

Cell["Compare the fitted model to the data:", "ExampleText",
 CellID->47073673],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", 
  RowBox[{
   RowBox[{"Plot", "[", 
    RowBox[{
     RowBox[{"model", "/.", "fit"}], ",", 
     RowBox[{"{", 
      RowBox[{"t", ",", "0", ",", "10"}], "}"}]}], "]"}], ",", 
   RowBox[{"ListPlot", "[", 
    RowBox[{"data", ",", 
     RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->139019433],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 98},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->197745188]
}, Open  ]],

Cell["\<\
The residuals show a pattern, indicating that the frequency constraint is too \
strict:\
\>", "ExampleText",
 CellID->50181010],

Cell[BoxData[
 RowBox[{
  RowBox[{"residuals", "=", 
   RowBox[{"Apply", "[", 
    RowBox[{
     RowBox[{"Function", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"t", ",", "y"}], "}"}], ",", 
       RowBox[{"Evaluate", "[", 
        RowBox[{"{", 
         RowBox[{"t", ",", 
          RowBox[{
           RowBox[{"y", "-", "model"}], "/.", "fit"}]}], "}"}], "]"}]}], 
      "]"}], ",", "data", ",", "1"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->696071961],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{"residuals", ",", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->158684639],

Cell[GraphicsData["CompressedBitmap", "\<\
eJy9WM9PE0EUXrr9RUCtolBarDUECIRgNOhJIcZfHBBN1BiPTdWAIBgkJHhQ
vGP0oAeriRf1H9DoH+BFT149+4+s+2Z3Zna/fTNdqrFJ252Z9773vfe+me72
cmNj8c79xsZSs1G/tN54sLjUfFi/uLbuT7ldjuN89t/NsuNfe/6VeA94nue4
dCVeTos+HJrN+JNv1Chq81yMP7z/OXvm9bOdb07RYnXzxkff6sK5txErQn4h
Vn98/722+unrl18GjJcQybVY6UhuLNIriBSsFsJVF7BoLg9zd4HFcYEgrLLa
ynGyiZkci6SZBkghJfGi+IFbBCY1ss5ydxwJqGXxcJmqmOyRGY5bsfrL75xV
ca02s7v1bEJHp2NIt6FL00nF5BRW2CGsThP6MW2oFvplWf7pFJJN2U2Jz/G2
dZXbG8tQycmO/HWtJxPqwOrQ/ApUtwYsetiu70Asm5XG1lb/Qp/cuUN9cJmd
swpZjVtqG8XicXTe4yk1xWmThqtQIRMv7B6etWUv0OwQzGfoI6/yVhd9Ye2G
wu9KyCcTrWkHPAahfjwfVT91ccAQSXeslLCoJGYInX6cu+jbkAF/alPOVSYX
MnS6FQvVKN0xE3Otkc6YS3XnPX63I1tSgFNQlfc/NYOygaPOYn+0Bm2Y4d7c
B53qZuqYkQerClkOFZeMmKY6PAedcbLvFcky3KeHASEZNS0Pyq0E9ZTdq0Z4
UdRaZF7uO7nf9L7D7A5ChY9AtIbZS+815b4dY35IWCksZb7NYg5AZN7qBPAd
hfFp4UX83oXVTGLgyYu/5uw6dUuFQQ8jQphLqgiWdTEuqrEqkseenH8brd06
ryTxbFBQJfJHuiu9soJKAxXwxtMcz6AS9Llo5ETKE+dEQQXzoifWHi+us3te
fH/vtWQXLbxGOAXczkIEHjELXjPAYwLWR2IZ5sB6BmJOwPqIhYP2WoYx7zUF
zJaEVb+c9Q1p9hgwCKyKnvnOypVdC47RmIKqEBO7GKAXpM5Cpsm7Llf20b8O
+qgZVoHxlgETfy8FZj65O6vA0YSH+5d+C2g4DDlv7MI/Oh6GvHKAu2LATd7P
4MkwDBmiKrFrCzENyPsPeZc0BtYZQMf+LLDqHAWrOqD0QpRZC4r2QjVrL2J+
C6x5zPTPH7FdkniKNT+rp9BpZLwJlThq0UGW6f4m1MTmj08+NJyH+GNQxR5Y
7zPgS/1gXeaB30kYd4an+eEJPgXrfbFecs83NH8dWJ1nVXINsHmruJauAr85
1ucKxLdZ6fhzsUiMauHfurSabX/m/C9P/f8M78H9F7AFFa9ZFGa749qCntQS
SsqELOj7EfQGz/d+tqNPIYbNSmNrK4r8BCINshiPIZLNSkcajEWy/aPjdP0B
y0ffKg==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 99},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->388765332]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->568401064],

Cell["\<\
Give starting values for some parameters to get a better fit:\
\>", "ExampleText",
 CellID->339171382],

Cell[BoxData[
 RowBox[{
  RowBox[{"model", "=", 
   RowBox[{
    RowBox[{"a", " ", 
     RowBox[{"Exp", "[", 
      RowBox[{
       RowBox[{"-", "b"}], " ", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"x", "-", "c"}], ")"}], "^", "2"}]}], "]"}]}], "+", 
    RowBox[{"d", " ", 
     RowBox[{"Sin", "[", 
      RowBox[{
       RowBox[{"\[Omega]", " ", "x"}], "+", "\[Phi]"}], "]"}]}]}]}], 
  ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->68570158],

Cell[BoxData[
 RowBox[{
  RowBox[{"data", "=", 
   RowBox[{
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"x", ",", 
        RowBox[{"model", "/.", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"a", "\[Rule]", "2"}], ",", 
           RowBox[{"b", "\[Rule]", "1"}], ",", 
           RowBox[{"c", "\[Rule]", "0"}], ",", 
           RowBox[{"d", "\[Rule]", "2"}], ",", 
           RowBox[{"\[Omega]", "\[Rule]", "0.67"}], ",", 
           RowBox[{"\[Phi]", "\[Rule]", "0.1"}]}], "}"}]}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", 
        RowBox[{"-", "5"}], ",", "5", ",", ".1"}], "}"}]}], "]"}], "+", 
    RowBox[{"RandomReal", "[", 
     RowBox[{".25", ",", "101"}], "]"}]}]}], ";"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->504300830],

Cell["Search with all parameters starting at 1:", "ExampleText",
 CellID->152303581],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"f1", "=", 
  RowBox[{"FindFit", "[", 
   RowBox[{"data", ",", "model", ",", 
    RowBox[{"{", 
     RowBox[{
     "a", ",", "b", ",", "c", ",", "d", ",", "\[Omega]", ",", "\[Phi]"}], 
     "}"}], ",", "x"}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->628502664],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"a", "\[Rule]", "3.1943010052440544`"}], ",", 
   RowBox[{"b", "\[Rule]", "0.3914865391691187`"}], ",", 
   RowBox[{"c", "\[Rule]", "3.6856402243107382`"}], ",", 
   RowBox[{"d", "\[Rule]", "2.1749210749156584`"}], ",", 
   RowBox[{"\[Omega]", "\[Rule]", "0.8888468674782414`"}], ",", 
   RowBox[{"\[Phi]", "\[Rule]", "0.8236021044658991`"}]}], "}"}]], "Output",
 ImageSize->{520, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->2584664]
}, Open  ]],

Cell["Search with a better starting value for the parameter c:", "ExampleText",
 CellID->72785105],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"f2", "=", 
  RowBox[{"FindFit", "[", 
   RowBox[{"data", ",", "model", ",", 
    RowBox[{"{", 
     RowBox[{"a", ",", "b", ",", 
      RowBox[{"{", 
       RowBox[{"c", ",", "0"}], "}"}], ",", "d", ",", "\[Omega]", ",", 
      "\[Phi]"}], "}"}], ",", "x"}], "]"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->109648129],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"a", "\[Rule]", "2.1981540549377483`"}], ",", 
   RowBox[{"b", "\[Rule]", "0.7992366426595594`"}], ",", 
   RowBox[{"c", "\[Rule]", "0.08503673793250188`"}], ",", 
   RowBox[{"d", "\[Rule]", "2.023993487769597`"}], ",", 
   RowBox[{"\[Omega]", "\[Rule]", "0.6669270095559184`"}], ",", 
   RowBox[{"\[Phi]", "\[Rule]", 
    RowBox[{"-", "0.053769559734109164`"}]}]}], "}"}]], "Output",
 ImageSize->{556, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->90699819]
}, Open  ]],

Cell["Compare the quality of the two fits:", "ExampleText",
 CellID->171800968],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", 
  RowBox[{
   RowBox[{"Plot", "[", 
    RowBox[{
     RowBox[{"Evaluate", "[", 
      RowBox[{"model", "/.", 
       RowBox[{"{", 
        RowBox[{"f1", ",", "f2"}], "}"}]}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", 
       RowBox[{"-", "5"}], ",", "5"}], "}"}], ",", 
     RowBox[{"PlotStyle", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{"Red", ",", "Green"}], "}"}]}]}], "]"}], ",", 
   RowBox[{"ListPlot", "[", 
    RowBox[{"data", ",", 
     RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->61710716],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 112},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->132412364]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Multivariate Data",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->127486635],

Cell["Fit a surface to data in 2-dimensions:", "ExampleText",
 CellID->3453890],

Cell[BoxData[
 RowBox[{
  RowBox[{"model", "=", 
   RowBox[{"a", " ", 
    RowBox[{"Exp", "[", 
     RowBox[{
      RowBox[{"-", "b"}], " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{
         RowBox[{"(", 
          RowBox[{"x", "-", "x0"}], ")"}], "^", "2"}], "+", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{"y", "-", "y0"}], ")"}], "^", "2"}]}], ")"}]}], "]"}]}]}], 
  ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->53072381],

Cell[BoxData[
 RowBox[{
  RowBox[{"data", "=", 
   RowBox[{"MapThread", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"#1", "[", 
         RowBox[{"[", "1", "]"}], "]"}], ",", 
        RowBox[{"#1", "[", 
         RowBox[{"[", "2", "]"}], "]"}], ",", 
        RowBox[{
         RowBox[{"1.2", " ", 
          RowBox[{"Exp", "[", 
           RowBox[{
            RowBox[{"-", "34"}], 
            RowBox[{"(", 
             RowBox[{
              RowBox[{"(", 
               RowBox[{"#1", "-", ".56"}], ")"}], ".", 
              RowBox[{"(", 
               RowBox[{"#1", "-", ".56"}], ")"}]}], ")"}]}], "]"}]}], "+", 
         "#2"}]}], "}"}], "&"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"RandomReal", "[", 
        RowBox[{"1", ",", 
         RowBox[{"{", 
          RowBox[{"100", ",", "2"}], "}"}]}], "]"}], ",", 
       RowBox[{"RandomReal", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{"-", ".1"}], ",", ".1"}], "}"}], ",", "100"}], "]"}]}], 
      "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->292962567],

Cell["\<\
Find the fit starting from an approximate position for the peak:\
\>", "ExampleText",
 CellID->103597510],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"fit", "=", 
  RowBox[{"FindFit", "[", 
   RowBox[{"data", ",", "model", ",", 
    RowBox[{"{", 
     RowBox[{"a", ",", "b", ",", 
      RowBox[{"{", 
       RowBox[{"x0", ",", ".5"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"y0", ",", ".6"}], "}"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "y"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->84847018],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"a", "\[Rule]", "1.1677452920709623`"}], ",", 
   RowBox[{"b", "\[Rule]", "32.59675575701265`"}], ",", 
   RowBox[{"x0", "\[Rule]", "0.5695547956119585`"}], ",", 
   RowBox[{"y0", "\[Rule]", "0.5600725787915997`"}]}], "}"}]], "Output",
 ImageSize->{368, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->503473188]
}, Open  ]],

Cell["Show the fitted surface with the data:", "ExampleText",
 CellID->670475389],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", 
  RowBox[{
   RowBox[{"Plot3D", "[", 
    RowBox[{
     RowBox[{"model", "/.", "fit"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
     RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}], ",", 
   RowBox[{"ListPointPlot3D", "[", 
    RowBox[{"data", ",", 
     RowBox[{"PlotStyle", "\[Rule]", 
      RowBox[{"Directive", "[", 
       RowBox[{
        RowBox[{"PointSize", "[", "Medium", "]"}], ",", "Red"}], "]"}]}]}], 
    "]"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->391796545],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 141},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->16310007]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->435170072],

Cell["Fit a model to data in four dimensions:", "ExampleText",
 CellID->129853208],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"model", "[", 
    RowBox[{"x1_", ",", "x2_", ",", "x3_", ",", "x4_"}], "]"}], ":=", 
   RowBox[{"a0", " ", 
    RowBox[{"Cos", "[", 
     RowBox[{
      RowBox[{"a1", " ", 
       RowBox[{"x1", "^", "2"}]}], "+", 
      RowBox[{"a2", " ", 
       RowBox[{"x2", "^", "2"}]}], "+", 
      RowBox[{"a3", " ", 
       RowBox[{"x3", "^", "3"}]}], "+", 
      RowBox[{"a4", " ", 
       RowBox[{"x4", "^", "4"}]}]}], "]"}]}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->329142581],

Cell[BoxData[{
 RowBox[{
  RowBox[{"g", "=", 
   RowBox[{"N", "[", 
    RowBox[{
     RowBox[{"Range", "[", 
      RowBox[{
       RowBox[{"-", "5"}], ",", "5"}], "]"}], "/", "5"}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"data", "=", 
   RowBox[{"Block", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"a0", "=", "1"}], ",", 
       RowBox[{"a1", "=", "1"}], ",", 
       RowBox[{"a2", "=", "2"}], ",", 
       RowBox[{"a3", "=", "3"}], ",", 
       RowBox[{"a4", "=", "4"}]}], "}"}], ",", 
     RowBox[{"Flatten", "[", 
      RowBox[{
       RowBox[{"Outer", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"{", 
           RowBox[{"##", ",", 
            RowBox[{"model", "[", "##", "]"}]}], "}"}], "&"}], ",", "g", ",", 
         "g", ",", "g", ",", "g"}], "]"}], ",", "3"}], "]"}]}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"data", "[", 
    RowBox[{"[", 
     RowBox[{"All", ",", 
      RowBox[{"-", "1"}]}], "]"}], "]"}], "+=", 
   RowBox[{"RandomReal", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "0.1"}], ",", "0.1"}], "}"}], ",", 
     RowBox[{"Length", "[", "data", "]"}]}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[2]:=",
 CellID->262167145],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"fit", "=", 
  RowBox[{"FindFit", "[", 
   RowBox[{"data", ",", 
    RowBox[{"model", "[", 
     RowBox[{"x1", ",", "x2", ",", "x3", ",", "x4"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"a0", ",", "a1", ",", "a2", ",", "a3", ",", "a4"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x1", ",", "x2", ",", "x3", ",", "x4"}], "}"}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->531529081],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"a0", "\[Rule]", "1.000548513531908`"}], ",", 
   RowBox[{"a1", "\[Rule]", "0.9994638926784591`"}], ",", 
   RowBox[{"a2", "\[Rule]", "1.9997925979743774`"}], ",", 
   RowBox[{"a3", "\[Rule]", "3.000491786512234`"}], ",", 
   RowBox[{"a4", "\[Rule]", "4.000269812862884`"}]}], "}"}]], "Output",
 ImageSize->{465, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->328507755]
}, Open  ]],

Cell["\<\
The residual is comparable in size to the randomness added to the data:\
\>", "ExampleText",
 CellID->166969836],

Cell[BoxData[
 RowBox[{
  RowBox[{"residual", "=", 
   RowBox[{"Apply", "[", 
    RowBox[{
     RowBox[{"Function", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"x1", ",", "x2", ",", "x3", ",", "x4", ",", "r"}], "}"}], ",",
        
       RowBox[{"Evaluate", "[", 
        RowBox[{"r", "-", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"model", "[", 
            RowBox[{"x1", ",", "x2", ",", "x3", ",", "x4"}], "]"}], "/.", 
           "fit"}], ")"}]}], "]"}]}], "]"}], ",", "data", ",", 
     RowBox[{"{", "1", "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->926626446],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Norm", "[", 
  RowBox[{"residual", ",", "Infinity"}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->134455144],

Cell[BoxData["0.10063020861425276`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->254192682]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Options",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->223291971],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "EvaluationMonitor",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->78762153],

Cell["\<\
Count evaluations of the model with numerical values of the parameters:\
\>", "ExampleText",
 CellID->245920735],

Cell[BoxData[
 RowBox[{
  RowBox[{"data", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"4.00", ",", "126."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"5.00", ",", "125."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"7.00", ",", "123."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"12.0", ",", "120."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"14.0", ",", "119."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"16.0", ",", "118."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"20.0", ",", "116."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"24.0", ",", "115."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"28.0", ",", "114."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"31.0", ",", "113."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"34.0", ",", "112."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"37.5", ",", "111."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"41.0", ",", "110."}], "}"}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->39132109],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Block", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"count", "=", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"FindFit", "[", 
      RowBox[{"data", ",", 
       RowBox[{"60", "+", 
        RowBox[{"70", " ", 
         RowBox[{"Exp", "[", 
          RowBox[{
           RowBox[{"-", "a"}], " ", "x"}], "]"}]}]}], ",", "a", ",", "x", ",",
        
       RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
        RowBox[{"count", "++"}]}]}], "]"}], ",", "count"}], "}"}]}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->65187881],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"a", "\[Rule]", "0.009415496283978395`"}], "}"}], ",", "17"}], 
  "}"}]], "Output",
 ImageSize->{144, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->274659422]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Gradient",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->442346239],

Cell["\<\
Specify the model gradient to avoid problems with a removable singularity:\
\>", "ExampleText",
 CellID->261036267],

Cell[BoxData[
 RowBox[{
  RowBox[{"data", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"0.", ",", 
       RowBox[{"-", "1."}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{".2", ",", "2."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{".5", ",", "5."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{".6", ",", "4."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1.", ",", "0."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2.", ",", "0."}], "}"}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->90513975],

Cell[TextData[{
 "With symbolic derivatives ",
 Cell[BoxData[
  ButtonBox["FindFit",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindFit"]], "InlineFormula"],
 " fails since the derivative for Sinc is given as a generic formula:"
}], "ExampleText",
 CellID->179578763],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindFit", "[", 
  RowBox[{"data", ",", 
   RowBox[{"a", " ", 
    RowBox[{"Sinc", "[", 
     RowBox[{"\[Omega]", " ", 
      RowBox[{"(", 
       RowBox[{"x", "-", "x0"}], ")"}]}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"a", ",", "\[Omega]", ",", "x0"}], "}"}], ",", "x"}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->251205743],

Cell[BoxData[
 RowBox[{
  RowBox[{"Power", "::", "\<\"infy\"\>"}], ":", 
  " ", "\<\"Infinite expression \\!\\(1\\/0.`\\) encountered. \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/General/infy\\\", \
ButtonNote -> \\\"Power::infy\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->476426630],

Cell[BoxData[
 RowBox[{
  RowBox[{"Power", "::", "\<\"infy\"\>"}], ":", 
  " ", "\<\"Infinite expression \\!\\(1\\/0.`\\^2\\) encountered. \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/General/infy\\\", \
ButtonNote -> \\\"Power::infy\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->678958854],

Cell[BoxData[
 RowBox[{
  RowBox[{"\[Infinity]", "::", "\<\"indet\"\>"}], ":", 
  " ", "\<\"Indeterminate expression \\!\\(0.`\\\\ ComplexInfinity\\) \
encountered. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/General/indet\\\", ButtonNote -> \
\\\"Infinity::indet\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->659664696],

Cell[BoxData[
 RowBox[{
  RowBox[{"FindFit", "::", "\<\"nrjnum\"\>"}], ":", 
  " ", "\<\"The Jacobian is not a matrix of real numbers at \\!\\({a, \
\[Omega], x0}\\) = \\!\\({1.`, 1.`, 1.`}\\). \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/General/nrjnum\\\", ButtonNote -> \
\\\"FindFit::nrjnum\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->480711551],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"a", "\[Rule]", "1.`"}], ",", 
   RowBox[{"\[Omega]", "\[Rule]", "1.`"}], ",", 
   RowBox[{"x0", "\[Rule]", "1.`"}]}], "}"}]], "Output",
 ImageSize->{159, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->732772992]
}, Open  ]],

Cell[TextData[{
 "The gradient has a singularity when ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x0", "=", "1"}], TraditionalForm]], "InlineMath"],
 " since there is a point in the data with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "=", "1"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->170632306],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"D", "[", 
  RowBox[{
   RowBox[{"a", " ", 
    RowBox[{"Sinc", "[", 
     RowBox[{"\[Omega]", " ", 
      RowBox[{"(", 
       RowBox[{"x", " ", "-", " ", "x0"}], ")"}]}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{"a", ",", "\[Omega]", ",", "x0"}], "}"}], "}"}]}], 
  "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->31397592],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"Sinc", "[", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"x", "-", "x0"}], ")"}], " ", "\[Omega]"}], "]"}], ",", 
   RowBox[{"a", " ", 
    RowBox[{"(", 
     RowBox[{"x", "-", "x0"}], ")"}], " ", 
    RowBox[{"(", 
     RowBox[{
      FractionBox[
       RowBox[{"Cos", "[", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{"x", "-", "x0"}], ")"}], " ", "\[Omega]"}], "]"}], 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"x", "-", "x0"}], ")"}], " ", "\[Omega]"}]], "-", 
      FractionBox[
       RowBox[{"Sin", "[", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{"x", "-", "x0"}], ")"}], " ", "\[Omega]"}], "]"}], 
       RowBox[{
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{"x", "-", "x0"}], ")"}], "2"], " ", 
        SuperscriptBox["\[Omega]", "2"]}]]}], ")"}]}], ",", 
   RowBox[{
    RowBox[{"-", "a"}], " ", "\[Omega]", " ", 
    RowBox[{"(", 
     RowBox[{
      FractionBox[
       RowBox[{"Cos", "[", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{"x", "-", "x0"}], ")"}], " ", "\[Omega]"}], "]"}], 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"x", "-", "x0"}], ")"}], " ", "\[Omega]"}]], "-", 
      FractionBox[
       RowBox[{"Sin", "[", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{"x", "-", "x0"}], ")"}], " ", "\[Omega]"}], "]"}], 
       RowBox[{
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{"x", "-", "x0"}], ")"}], "2"], " ", 
        SuperscriptBox["\[Omega]", "2"]}]]}], ")"}]}]}], "}"}]], "Output",
 ImageSize->{410, 68},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->536748248]
}, Open  ]],

Cell["\<\
Specify finite differences to avoid the removable singularity:\
\>", "ExampleText",
 CellID->341588118],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"fit", "=", 
  RowBox[{"FindFit", "[", 
   RowBox[{"data", ",", 
    RowBox[{"a", " ", 
     RowBox[{"Sinc", "[", 
      RowBox[{"\[Omega]", " ", 
       RowBox[{"(", 
        RowBox[{"x", "-", "x0"}], ")"}]}], "]"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"a", ",", "\[Omega]", ",", "x0"}], "}"}], ",", "x", ",", 
    RowBox[{"Gradient", "\[Rule]", "\"\<FiniteDifference\>\""}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->61959277],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"a", "\[Rule]", "4.683124035841452`"}], ",", 
   RowBox[{"\[Omega]", "\[Rule]", "6.721526284826763`"}], ",", 
   RowBox[{"x0", "\[Rule]", "0.5209820263795437`"}]}], "}"}]], "Output",
 ImageSize->{271, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->242178391]
}, Open  ]],

Cell["Show the data with the fit:", "ExampleText",
 CellID->4019287],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", 
  RowBox[{
   RowBox[{"ListPlot", "[", "data", "]"}], ",", 
   RowBox[{"Plot", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"a", " ", 
       RowBox[{"Sinc", "[", 
        RowBox[{"\[Omega]", " ", 
         RowBox[{"(", 
          RowBox[{"x", "-", "x0"}], ")"}]}], "]"}]}], "/.", "fit"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "2"}], "}"}]}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->80969814],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 112},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->353661454]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "MaxIterations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->365923182],

Cell[TextData[{
 "When convergence is slow, increasing ",
 Cell[BoxData[
  ButtonBox["MaxIterations",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MaxIterations"]], "InlineFormula"],
 " may allow convergence to the best fit:"
}], "ExampleText",
 CellID->111416151],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"model", "[", "x_", "]"}], ":=", 
   RowBox[{"d", "+", 
    RowBox[{"33", " ", "c", " ", 
     RowBox[{"(", 
      RowBox[{
       FractionBox[
        RowBox[{"0.7818074748209513", " ", "b"}], 
        RowBox[{
         RowBox[{"4", " ", 
          SuperscriptBox["b", "2"]}], "+", 
         SuperscriptBox[
          RowBox[{"(", 
           RowBox[{
            RowBox[{
             RowBox[{"-", "47"}], " ", "a"}], "+", 
            RowBox[{"2", " ", "x", " ", "\[Pi]"}]}], ")"}], "2"]}]], "+", 
       FractionBox[
        RowBox[{"0.6686926778600345", " ", "b"}], 
        RowBox[{
         RowBox[{"32", " ", 
          SuperscriptBox["b", "2"]}], "+", 
         SuperscriptBox[
          RowBox[{"(", 
           RowBox[{
            RowBox[{
             RowBox[{"-", "46"}], " ", "a"}], "+", 
            RowBox[{"2", " ", "x", " ", "\[Pi]"}]}], ")"}], "2"]}]], "+", 
       FractionBox[
        RowBox[{"0.8311998722869463", " ", "b"}], 
        RowBox[{
         RowBox[{"21", " ", 
          SuperscriptBox["b", "2"]}], "+", 
         SuperscriptBox[
          RowBox[{"(", 
           RowBox[{
            RowBox[{
             RowBox[{"-", "44"}], " ", "a"}], "+", 
            RowBox[{"2", " ", "x", " ", "\[Pi]"}]}], ")"}], "2"]}]]}], 
      ")"}]}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"xv", "=", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", "500"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"data", "=", 
   RowBox[{"Block", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"a", "=", "2"}], ",", 
       RowBox[{"b", "=", "3"}], ",", 
       RowBox[{"c", "=", "4"}], ",", 
       RowBox[{"d", "=", "5"}]}], "}"}], ",", 
     RowBox[{"Transpose", "[", 
      RowBox[{"{", 
       RowBox[{"xv", ",", 
        RowBox[{"model", "[", "xv", "]"}]}], "}"}], "]"}]}], "]"}]}], 
  ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->579614116],

Cell["With the default convergence is not reached:", "ExampleText",
 CellID->118008607],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindFit", "[", 
  RowBox[{"data", ",", 
   RowBox[{"model", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"a", ",", "2.3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"b", ",", "3.2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"c", ",", "3.9"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"d", ",", "4.9"}], "}"}]}], "}"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->162448948],

Cell[BoxData[
 RowBox[{
  RowBox[{"FindFit", "::", "\<\"cvmit\"\>"}], ":", 
  " ", "\<\"Failed to converge to the requested accuracy or precision within \
\\!\\(100\\) iterations. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/FindFit/cvmit\\\", ButtonNote -> \
\\\"FindFit::cvmit\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->182612031],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"a", "\[Rule]", "1.999278732943677`"}], ",", 
   RowBox[{"b", "\[Rule]", "3.0182550241690564`"}], ",", 
   RowBox[{"c", "\[Rule]", "3.973557685910686`"}], ",", 
   RowBox[{"d", "\[Rule]", "5.000010919329603`"}]}], "}"}]], "Output",
 ImageSize->{340, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->145944404]
}, Open  ]],

Cell["Convergence is reached before 1000 iterations are used:", "ExampleText",
 CellID->11419712],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindFit", "[", 
  RowBox[{"data", ",", 
   RowBox[{"model", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"a", ",", "2.3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"b", ",", "3.2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"c", ",", "3.9"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"d", ",", "4.9"}], "}"}]}], "}"}], ",", "x", ",", 
   RowBox[{"MaxIterations", "\[Rule]", "1000"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->239578397],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"a", "\[Rule]", "1.999999999954946`"}], ",", 
   RowBox[{"b", "\[Rule]", "3.000000001146417`"}], ",", 
   RowBox[{"c", "\[Rule]", "3.9999999983297174`"}], ",", 
   RowBox[{"d", "\[Rule]", "5.000000000000679`"}]}], "}"}]], "Output",
 ImageSize->{200, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->149704356]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Method",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->308617728],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Method",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Method"], "->", 
   ButtonBox["NMinimize",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NMinimize"]}]], "InlineFormula"],
 " to search more exhaustively for a global minimum:"
}], "ExampleText",
 CellID->58344041],

Cell[BoxData[
 RowBox[{
  RowBox[{"model", "=", 
   RowBox[{
    RowBox[{"a", " ", 
     RowBox[{"Exp", "[", 
      RowBox[{
       RowBox[{"-", "b"}], " ", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"x", "-", "c"}], ")"}], "^", "2"}]}], "]"}]}], "+", 
    RowBox[{"d", " ", 
     RowBox[{"Sin", "[", 
      RowBox[{
       RowBox[{"\[Omega]", " ", "x"}], "+", "\[Phi]"}], "]"}]}]}]}], 
  ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->112853019],

Cell[BoxData[
 RowBox[{
  RowBox[{"data", "=", 
   RowBox[{
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"x", ",", 
        RowBox[{"model", "/.", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"a", "\[Rule]", "2"}], ",", 
           RowBox[{"b", "\[Rule]", "1"}], ",", 
           RowBox[{"c", "\[Rule]", "0"}], ",", 
           RowBox[{"d", "\[Rule]", "2"}], ",", 
           RowBox[{"\[Omega]", "\[Rule]", "0.67"}], ",", 
           RowBox[{"\[Phi]", "\[Rule]", "0.1"}]}], "}"}]}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", 
        RowBox[{"-", "5"}], ",", "5", ",", ".1"}], "}"}]}], "]"}], "+", 
    RowBox[{"RandomReal", "[", 
     RowBox[{".25", ",", "101"}], "]"}]}]}], ";"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->198224689],

Cell["Find a local minimum:", "ExampleText",
 CellID->320088632],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"f1", "=", 
  RowBox[{"FindFit", "[", 
   RowBox[{"data", ",", "model", ",", 
    RowBox[{"{", 
     RowBox[{
     "a", ",", "b", ",", "c", ",", "d", ",", "\[Omega]", ",", "\[Phi]"}], 
     "}"}], ",", "x"}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->724733624],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"a", "\[Rule]", "3.273674931787074`"}], ",", 
   RowBox[{"b", "\[Rule]", "0.4054293610017216`"}], ",", 
   RowBox[{"c", "\[Rule]", "3.591596155401839`"}], ",", 
   RowBox[{"d", "\[Rule]", "2.1588884914008695`"}], ",", 
   RowBox[{"\[Omega]", "\[Rule]", "0.9091057125373316`"}], ",", 
   RowBox[{"\[Phi]", "\[Rule]", "0.8363317022475234`"}]}], "}"}]], "Output",
 ImageSize->{520, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->309184904]
}, Open  ]],

Cell["Search for a global minimum:", "ExampleText",
 CellID->12121573],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"f2", "=", 
  RowBox[{"FindFit", "[", 
   RowBox[{"data", ",", "model", ",", 
    RowBox[{"{", 
     RowBox[{
     "a", ",", "b", ",", "c", ",", "d", ",", "\[Omega]", ",", "\[Phi]"}], 
     "}"}], ",", "x", ",", 
    RowBox[{"Method", "\[Rule]", "NMinimize"}]}], "]"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->152087054],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"a", "\[Rule]", "2.2494249231013725`"}], ",", 
   RowBox[{"b", "\[Rule]", "0.7170344334958937`"}], ",", 
   RowBox[{"c", "\[Rule]", "0.0937987829007177`"}], ",", 
   RowBox[{"d", "\[Rule]", "2.0275781109656457`"}], ",", 
   RowBox[{"\[Phi]", "\[Rule]", 
    RowBox[{"-", "0.08079389629862424`"}]}], ",", 
   RowBox[{"\[Omega]", "\[Rule]", "0.6792958638303535`"}]}], "}"}]], "Output",\

 ImageSize->{556, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->11893923]
}, Open  ]],

Cell["\<\
Compare the quality of the two fits (the global best fit shown in red):\
\>", "ExampleText",
 CellID->103264514],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"Plot", "[", 
     RowBox[{
      RowBox[{"Evaluate", "[", 
       RowBox[{"model", "/.", 
        RowBox[{"{", 
         RowBox[{"f1", ",", "f2"}], "}"}]}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", 
        RowBox[{"-", "5"}], ",", "5"}], "}"}], ",", 
      RowBox[{"PlotStyle", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", "Green", "}"}], ",", 
         RowBox[{"{", "Red", "}"}]}], "}"}]}]}], "]"}], ",", 
    RowBox[{"ListPlot", "[", "data", "]"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->68814404],

Cell[GraphicsData["CompressedBitmap", "\<\
eJydWXtwVNUZv2ZfSRAIiAjII4KVDBYIaiVCkt1kIQQIsgksWaxAAIUgkKxA
NBo0FZSALCwFwiY8GoGsgSoTqTRrqS2MOE0ydpqxdiZ26phppzOp/SfO1M5W
bXv6O9939t6bmw3tNCPrfZzzO9/j9z3OuSs27ql8eufGPds2b8xcsmtjsHLb
5t2ZRdW78Mh2h6Zp1/AvOEHDtcAV/bsmBP6/Vv7a5D39aQF6Okv+OnpET1iE
60U9/usUnfQ0RVNztCz5a28RLZkiM1tkB0UwBeM0e47ImSAmNIiGuIjPTIr+
AM3sF/0e4akQFX2iz5F4UitqgYXFpuKJHRPtSRG+Q08hHxaLiZiWRve9oheI
8n6qaZY2LMr9CRTMghBaqrx3ic5O0rj+2/q6S/VZH8X2i3hcu3sYeWz4kbeM
OIMeAQtKDIgBequ5oq097rzmbVvb5W/RUfweWbhLS8ESmjNacRj3LxQ89U3G
CFFRIfr7J1tWSEkq+3R6CovBAlJ2kiut3P8m0Dx5zfrvYu9ZvIEEWgruNAfu
i+L5n7XsFVlZIhzW7v3fV5sIM90QN/gF63QkdKv7jevVua80rj9WveNq48lf
VVe9G+v4FAOUKO7TuIWAEPOLgd+LYFAqCXuOJ48b6wzno/sk5+ANsqSDQRe7
m0ROjujt1Zw052DDTTwNVr3L0jvAgGJRLEFaWkRxsXaP4q0j6QqZ9BSUg3qg
LPvM1t31p+qiw7FNrzIrbGpl71l+76gUlRERUWtUVGhTBsXGNPIG0LJEFpQ3
7HXoJgTCFLWKbkXTqmRAvneeb/0I7zdtvSRHBcJgjJUdAX09aSXSf4SSNT8i
YjF9Bd0lUgeKFxffu3We+HyivX0II3iFqYkIA7OllXil7g8+qy5siB16y4yO
a0MPWikV93ghV8rHYk1HXntf0q+/f1LStaaYbLMjdCERRd4qXRMDfxS9w/22
qit5JceUBG1tIhDQJg3yCaGmrfCf5THKMP5QeVFEWWUc2byy65XSqkZgqxHb
L0wcXsq0R7pWbK5qw2jd6IZshOeCtZAhEakly97AiILcJsn/CUkxybuudtHu
Ez5+0X36WrXntdjVT3iGEwkZmdb3+AVAefObEMK3QXJiYaYfbC2ys/E70STT
e10fS0nbukVmppTpnqRIxAenWSbh8YgbN+i5i+VROSGwX7FZRreRFWmkA8RB
qpXjwmHkAaWrbh3JoLKLyhPjk0oyKSFJcesuuVDN21KSeHyKyTKKHAjdF8/e
BiUV0pT6pT8W5zaKvj49TO4dIlX0cIfS6u6keGTRNF2qsoKTWHpQtJGmdl0+
vKBcbGMuy5djBzGVEO2GhHmnUizZQbcdKi3qjAyvgtMILJ3FQ6QkljjYi92/
6K3OeQl+57zuhFgQTlnu0brYhQ9HKv8lz8mEZVdOWL4P6YzqnhNUg9WEnuz8
oVGW+bakeMQ7O3JLrv8w60v13QFRIbCyY24jZfvhavBwqNCt7oOwVPfCh4Tq
RAlDple2XLQFOXJkEjyDvVyp2Nb/CG4De2da+MwVIBnKUKmYkzD34Z7vIyK5
i3KilKCgyGEDA2LCBHD6zv8dTUryVc7s6IGriqfUXznNHEY2RE4cIW5XB4nd
zvrWy+xVnb6sLRoztHRyOOfWwYzluYWrwnJK4RmdqDTXCRlgOzk3HhcZGfgd
k1SCu2glJs8TJdXVZRHJ9QdNXJCD0T+gft4GQdXqRXmN2mwxOJ+iHiaqiDb6
NghQYIOn5u1D57U5pAHEl4aMx8vnH2SrJJ9NUtlZg+e3h7p8E+cKrsxE2m7f
M9WBc1Bq9CCGcQ3TU5ASP/fkz/qatbmkvWFBtn6GZfXVwqiYUrc7DFyn7kgo
taPsZO06j5VLU1RcpZstnXDVEDslRmdanmeYbC/zkFkGnRCoQAOZGfvjdVYZ
rB17usVzxcXRly8r7ayRZp1rlYzGO42ONSEMSzha6UPdtEKQkyeq51ZJ082M
iETKC0+wwiNVRFhXT9hrlMU+lMWU1URPz5fZ90HVQf23lGS6BS3N4FNfX3dW
IWtitYhVhskWVPKpXRmi6Tqyjs3wlorAI4v3NrTPhwfIAinDyKI59KwvWzr0
5Nacn8weKZplLfTF6EH6+zWzHMpWeaeAn+jM5cxpJL1570DZ0o6mH62/qK0V
DQ1WKaYMmTPNLIfd2FnU19sNGchGC+rrml5GArTmzakKgaxvbxNtARFAY4Sm
xLp6YuR9ybikIv/5mlh15ZWOB0v2xvfIHZ3JEnb05ejOsZlAkfwe2Ybr0Zfx
v8hyMTCgJSr242JQzKNgnQzfj8zNOrGk0TfkruKF+gvYlM2n8aoD8PmiL0Y5
SEZYJCVcBzOmrenHf8i+U54IOEyo5Uua5IbPG2kRLYyK8ji/dSMeFrtVi21F
XUn36GBu+cZ93n6U93PsTQfY0P3wip2Vb03v8Erv59DTPH9Idr65p1QnlBwx
2JkzkMOl1UC0y91nOAzxKkTFY/Tk5a7I0qojsewyENdacUljG0KtP3sC4lNz
MW1sKoTw9w7+cQ6hSHDqcYPnuMj3NjotmFZeWO+t42lNG8S9Htsld7LWzsfK
yP+23v8ljwv3cMDc+MxvsmZET/xSZWGnSXM9UkGQ5VXhZzuOLyT7otcqiudz
IbF6armJ+7vX74VzjNxoxpacy30lduKnekOTS0iIhq+K81DSh3CAkY32HkTS
O2cTsoOjCpcculoe4SBY3g175Q5lhIopwrNhp/fMstdjweMctXwu4JR1Ehur
3l7ccf4hFHmWVRBew6ZKt8i3jEag53uufz1vvUyIjui2kzyNE6vmptkF/mPc
W2lWtCUmO74euokp55ckehzGdSUkLX/sdTYwd3OMfKzr8vqiOuxZhkFmrxR4
m1p79pQvDCkPDUIOBrtrj7OBuS56mLftGXGfPIdR53NWZN61f7djxb+y53S/
eVN5iJGdjNzfH52zlpXhXqCAkH4wEIxnpCaQZWZdSs+Rx37SvglO5ZyUQPH5
yperbT93N4wCQX+d4xCdncnlkz6cGZ/276yZKLWM52C8nh74nDl1qaMbKntp
/qX6rN76cnmZasErIjwz10cZeHZJoJ4eHiprO6PdurH/E8+44dFW+y8qfg8M
MBpnB/QkolLuJVQvoToVXBbyavF486ylbFOXYjgh2jZ11W0u3heriXB/oDof
yXCuzYyn92TaIrovcx9WnHBZ5Fxk0tpbuwVezrDgdnYacuodvsK9WXtwzbLn
JCOS4+pdPkpuhsFGm94ra4t5nd7eP2eNlH1MUhxbW9fN5U/UxZZu5+6dRkkv
EAdsf88cv6WvSCTJnTzbH1+BIdI6Y0wyGDZny2LTsqdlnKzqTsVX1gEVcte6
1eyLMQZjTTZmKaI1MpetCh0aIgVxxf5sT4E/V7nBjKOf2Cqc8lWt8uQBfylJ
9tHED3nq/9ucEfJYl+NxjMFUeVaOzaa8pLCm7ih//56jO5PsKgnNFonM/byS
Fr/LhKOfOKi4RdPVX79VUygShCP0d/3vf5qdJq07zsRwmFZ1hJxJOR4xCdnX
qhPhOG4FJn/UttvYH401oek7W5Xn0VBhuuzFrViU1ZwAApwwHfCMNbFaL1Ja
CctWUUF7AnkOZ91JuOWPC+Hbl5UKTVm8stJWaXUzJtBQoeRliWKPZtsQauTT
pCEn9lR5083nvn+sKDze8pBZYvZFGdmGOYGHSu4UtQZV2XR9ytexqwe8Ky07
KDMOcyF29B0+6BoiF3VcaUavSyePf82+d0nZCfOpn7ZKri+/jTj08y16rrFm
aeZz5m89uetvTFXnmLDaeEOeVGWBnGrcGXxTKKYdaTh8MzgPbycac136mROo
Z7BNWoW0MPaSyGB/8zzyaHzOJNPKCntVjTwZSiyOuQtorsIuPIN8WNo3DyHF
c1dzkonF5BcCIXjZxCyHfl51ObYRjOUz1cScxF6dF0voKPvdL8Iv8IE9j/fz
eDoN4jW4t+NuGJXjn5VPYzx2GZON8XY+dGA3Shs8zDlJj6YUHgXJfb6n4+vQ
YE8xZrv0Iy02v2LZQ4PZhTve3289dLq0tY6BWYY1LDN9nGAIbtcYwfBjLFbu
lj2S23tqjT/KsCYEl64FUxJ7H7mDIV1S5Tuf79IHP3dX7b/S8Rsd1DRf1qvo
LL9MnxyBxreQeeQheawdCHwc70bSld/+aN+rUTtg40MCyrt0j4QDJ8pLPl+S
NgoE+PzcHarBdNopaqWCzw7U+S99ajNQsDa0AIH4nMspy2lOzjdtF4r8jbI0
es9MN1hp2v2nmPyn6SccxlfLWWwT9F4k0J69p/Fid1eIbTJdj1LaGYjEH/tS
0+sK77Pk5QPmjOQPiayssiUn2FPHWq+zr2cYfHboqNIPz7ST6CyjEccjLRy0
fk9LsdxTD3WntHNLS7nnh7IjcZ+aMzC1qGvDhqqLxvcb71mTJEm/i0YiTHd5
r04cCFqIIV8JTfcpSd6noHNBqwXeVFZ+lTP764z0WxX3VxxasLCqfmvHQfhj
6JEVf2EmXgzz9VTpa7cwZNAO1QRo0zPWjqvI3aKhQdTWHsx/HsOC+ftsEjr1
nFtu4reXbdkZcaw88CSut4TO6vXM0OkBnW2DTpnl7iG/Wd2DX+obvfW7qp2j
tK9PbYBmrpdfd2pro0+F+dv12EUFzfA+pmuonX75Hd3rPQ0VdQxz0dM1ZKnW
ikTl9OZFlNXsJokQkap3M2xNPpsoowp5IzOz3Cvl8nrO8Bor/T+SHHY3zeso
Ay/RZ5kLOKbqn6UNHWfpspD35DqBF8EoXhkpAd5CXmFbZRjqYHvp80UXbjE7
r7jkDB8yeLrWrqqKXOz4kPyhvv7qYpqPf5VeJikSsYa4oA+ybAM7MiO21+iB
cEGoI43zMYxrVx9lCr3NSMSpInV511PlVWeQObnvVBXw8ZLzlqVnD1lZ6bh8
HzN0NDpf2CBDZGDxja2ypS1dEDqyfTMacgTbjK7cx6peWt1RgxIgM+8Ii8ey
DXwn4xvfwiORpDlC7ZwPhN6/NtBaWiD3twWFTRqdsdrAU9XHF+jIktJaGhas
r3vPTd/OrV/XGVfde5p5B6v6uqKhtk9kT2b5SGnjeFzXSTVMT56rXlAPsujd
Gnf0jJRuRsLMTRVvrXviUmzTq3LjFgxSN5VaXnBcWYL2mvoCJhzryZJL1gCO
xkC4e/+56sorUoeSQRHlssQxncBpmyye4afaHf8BHP7teA==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 112},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->698701185]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "NormFunction",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->31388],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"data", "=", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"Prime", "[", "x", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "20"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->24887865],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "2", ",", "3", ",", "5", ",", "7", ",", "11", ",", "13", ",", "17", ",", 
   "19", ",", "23", ",", "29", ",", "31", ",", "37", ",", "41", ",", "43", 
   ",", "47", ",", "53", ",", "59", ",", "61", ",", "67", ",", "71"}], 
  "}"}]], "Output",
 ImageSize->{483, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->109594591]
}, Open  ]],

Cell["The default is to find the best least-squares fit:", "ExampleText",
 CellID->117745648],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindFit", "[", 
  RowBox[{"data", ",", 
   RowBox[{"a", " ", "x", " ", 
    RowBox[{"Log", "[", 
     RowBox[{"b", "+", 
      RowBox[{"c", "x"}]}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->3720920],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"a", "\[Rule]", "1.4207557242233555`"}], ",", 
   RowBox[{"b", "\[Rule]", "1.655581377626135`"}], ",", 
   RowBox[{"c", "\[Rule]", "0.5346453120672511`"}]}], "}"}]], "Output",
 ImageSize->{264, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->65550885]
}, Open  ]],

Cell["Use the \[Infinity]-norm instead:", "ExampleText",
 CellID->487219474],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindFit", "[", 
  RowBox[{"data", ",", 
   RowBox[{"a", " ", "x", " ", 
    RowBox[{"Log", "[", 
     RowBox[{"b", "+", 
      RowBox[{"c", "x"}]}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}], ",", "x", ",", 
   RowBox[{"NormFunction", "->", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"Norm", "[", 
       RowBox[{"#", ",", "Infinity"}], "]"}], "&"}], ")"}]}]}], 
  "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->443269537],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"a", "\[Rule]", "1.1507695890710887`"}], ",", 
   RowBox[{"b", "\[Rule]", "1.0022987426644`"}], ",", 
   RowBox[{"c", "\[Rule]", "1.0468558066626197`"}]}], "}"}]], "Output",
 ImageSize->{250, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->692538862]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindFit", "[", 
  RowBox[{"data", ",", 
   RowBox[{"a", " ", "x", " ", 
    RowBox[{"Log", "[", 
     RowBox[{"b", "+", 
      RowBox[{"c", "x"}]}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}], ",", "x", ",", 
   RowBox[{"NormFunction", "->", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"Norm", "[", 
       RowBox[{"#", ",", "1"}], "]"}], "&"}], ")"}]}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->14151221],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"a", "\[Rule]", "1.1477982444745727`"}], ",", 
   RowBox[{"b", "\[Rule]", "1.0047401048565836`"}], ",", 
   RowBox[{"c", "\[Rule]", "1.0517840067414115`"}]}], "}"}]], "Output",
 ImageSize->{250, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->840827165]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "StepMonitor",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->244192407],

Cell["\<\
Show steps taken in parameter space to fit the Michaelis\[Dash]Menten model \
to experimental data:\
\>", "ExampleText",
 CellID->398990986],

Cell[BoxData[
 RowBox[{
  RowBox[{"model", "=", 
   FractionBox[
    RowBox[{
     SubscriptBox["\[Theta]", "1"], "x"}], 
    RowBox[{
     SubscriptBox["\[Theta]", "2"], "+", "x"}]]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->73123166],

Cell["\<\
Data from an experiment on the rate of an enzymatic reaction:\
\>", "ExampleText",
 CellID->545405375],

Cell[BoxData[
 RowBox[{
  RowBox[{"data", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"0.02", ",", "76."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0.02", ",", "47."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0.06", ",", "97."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0.06", ",", "107."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0.11", ",", "123."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0.11", ",", "139."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0.22", ",", "159."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0.22", ",", "152."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0.56", ",", "191."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0.56", ",", "201."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1.1", ",", "207."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1.1", ",", "200."}], "}"}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->947481304],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"fit", ",", "steps"}], "}"}], "=", 
   RowBox[{"Reap", "[", 
    RowBox[{"FindFit", "[", 
     RowBox[{"data", ",", "model", ",", 
      RowBox[{"{", 
       RowBox[{
        SubscriptBox["\[Theta]", "1"], ",", 
        SubscriptBox["\[Theta]", "2"]}], "}"}], ",", "x", ",", 
      RowBox[{"StepMonitor", "\[RuleDelayed]", 
       RowBox[{"Sow", "[", 
        RowBox[{"{", 
         RowBox[{
          SubscriptBox["\[Theta]", "1"], ",", 
          SubscriptBox["\[Theta]", "2"]}], "}"}], "]"}]}]}], "]"}], "]"}]}], 
  ";"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->139363387],

Cell["Show how the parameters evolve during the search:", "ExampleText",
 CellID->145906611],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{"steps", ",", 
   RowBox[{"Epilog", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"Red", ",", 
      RowBox[{"Point", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          SubscriptBox["\[Theta]", "1"], ",", 
          SubscriptBox["\[Theta]", "2"]}], "}"}], "/.", "fit"}], "]"}]}], 
     "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->461283651],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzdWDtPG0EQPu7hgIgUKwiDRYGTBqU3ShUhFB5KQUGBaGMBESgKIIcobVpQ
6BDIgqRAJD0Kgh9AAxUtNU1+Q6rN7M7e7e7c3vnsoLOJJe/dzH7z+HZn9nye
q22trnyoba0t1Sqz9drm6trSx8rMRh1UXo/jOGfwfT/swD2DO/E9Ywyu8bEK
owcAD64uXEek7Eo51A9LveMLM/kJZ4fg6lus+gT4JR+5sTR1NZnflvngK9lE
2CxKRPaa4Af5EETzFpTNqiD0g0Q72nE2pZYj2JkIrffj5GbiVWP36yVurSY/
6TjToZT57DwF6lHEC6TFhZ9wP/36SHJWcr/ghFiscTfGcqQ93sqCYXfEPZfb
5/s40mLyb2a/IyXQXF/dbaz/uji/RbaRiGwl+7Cf74VroHO1exa4gq7J5l9x
5msoRCQwNXmItEqEZbjW4fmmn1D6+cavvrY3up7i+9TKcw645p8/XWDIhprl
nKPyqjKzw5ZNXJRylVSlhtNWYpzwRNRTGI9F5cbje8RDQGS/RTlv+27L10vY
vwrBPSfysyZyv/KraY+aaHldvmNmhVWtNivMrK80lKovOypLVt/assw/yjZZ
vWIKSq1e0chih6ya3QfGlwXkRCek/tSSpxBX0bqT1sZ5Kmo1rqHPwS7ywsVs
1uFp3N460IxpLntk14PY3hSY+XSwe1AVEVj3HFGqMhSK+zwgWXimD6ndFwQg
UqMyD9On5Qk3huMZIA5i/e4dgOk/bkBwMgeM+mW8flMc23nx1nGtmT/Mbv7X
KGndqb8jZalK8asltSqz9FWeXpK6s8NMLF0Z/iaz70O3V9n/FKVZx5j/HmTu
m4Q3RtZKreXgI2PHsBZPjdi7FM0luTOSnlcPo5pyqtmev4I20NE=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 107},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->347974402]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "WorkingPrecision",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->276747749],

Cell[TextData[{
 "Use a higher ",
 Cell[BoxData[
  ButtonBox["WorkingPrecision",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/WorkingPrecision"]], "InlineFormula"],
 " to get a more precise parameter estimate:"
}], "ExampleText",
 CellID->429534],

Cell[BoxData[
 RowBox[{
  RowBox[{"data", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"4", ",", "126"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"5", ",", "125"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"7", ",", "123"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"12", ",", "120"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"14", ",", "119"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"16", ",", "118"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"20", ",", "116"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"24", ",", "115"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"28", ",", "114"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"31", ",", "113"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"34", ",", "112"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"75", "/", "2"}], ",", "111"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"41", ",", "110"}], "}"}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->11960218],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindFit", "[", 
  RowBox[{"data", ",", 
   RowBox[{"60", "+", 
    RowBox[{"70", " ", 
     RowBox[{"Exp", "[", 
      RowBox[{
       RowBox[{"-", "a"}], " ", "x"}], "]"}]}]}], ",", "a", ",", "x", ",", 
   RowBox[{"WorkingPrecision", "\[Rule]", "50"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->195890952],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "a", "\[Rule]", 
   "0.0094154962840453400771940379222510706270007523073428862669685677597`50.\
"}], "}"}]], "Output",
 ImageSize->{419, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->33927605]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->107405395],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Mathematical Data",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->308456881],

Cell["Fit a list of Fibonacci numbers:", "ExampleText",
 CellID->31387452],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"Fibonacci", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "15"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->61543534],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1", ",", "1", ",", "2", ",", "3", ",", "5", ",", "8", ",", "13", ",", "21",
    ",", "34", ",", "55", ",", "89", ",", "144", ",", "233", ",", "377", ",", 
   "610"}], "}"}]], "Output",
 ImageSize->{371, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->283340045]
}, Open  ]],

Cell["The exponent is approximately the golden ratio:", "ExampleText",
 CellID->628619116],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindFit", "[", 
  RowBox[{"%", ",", 
   RowBox[{
    RowBox[{"a", " ", 
     RowBox[{"c", "^", "x"}]}], "+", 
    RowBox[{"b", " ", 
     RowBox[{"c", "^", 
      RowBox[{"-", "x"}]}]}]}], ",", 
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->652836235],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"a", "\[Rule]", "0.44713428447330056`"}], ",", 
   RowBox[{"b", "\[Rule]", "0.20087203717207816`"}], ",", 
   RowBox[{"c", "\[Rule]", "1.6180537282911562`"}]}], "}"}]], "Output",
 ImageSize->{271, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->345233619]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->92107301],

Cell["Generate data with some noise present:", "ExampleText",
 CellID->554009824],

Cell[BoxData[
 RowBox[{
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"Sqrt", "[", "x", "]"}], " ", 
      RowBox[{"Cos", "[", "x", "]"}]}], "+", 
     RowBox[{"RandomReal", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1"}], ",", "1"}], "}"}], "]"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "100"}], "}"}]}], "]"}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->196749448],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->434322074],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], LineBox[CompressedData["
1:eJw1lAtQVGUUx5eXi5DKa1ue8ZAcRB6CPFeBP4I8BZZ9gVM6iA+UkIVECc1h
UZBHDigNYEVYJm0MO0WACsrAogihDJKSGKlhxGtLQwTWIKGaPffO3Lnzv993
fuf/nXPutU+WCvZps1isvf/d/z8113SQUnpQL7O1pYNeYNItwi3J7S5pfXhb
OA8YJS6SNgKH+3J5fm6atBkEKaoWk6nZII02x9jRks8P3dGGRltD5m+7x3v2
L1q3Q7M8p29g8nuKd8BmweJNp2OGSo12xJaJjNyJVi3S65AUle0VYaOieCdE
TY13nx5jE98ZycWW/VI9A9IusEurdPZXDNF+N4icSqo//EeHeBthM66KyNq0
hrQH2DlTLZ0jy+THE7Ov5MIb+1nE2wTx8TODaWmrSXsha0jcNliySHxvnNtp
OZ2qMiCeD26ZBnipXBm+L+z1dJV+Xrqk/XBjVuG7f4Tx7w9ujdtQYQiHNA85
19h1PlcMSW8Gr5/rfbfqNfnbAlODrOyAXhPiBSDP/W2eS6wx6UDwsq/LGypG
yV8Qki/8pDjyzJh4QHnRF8WifaYaLQOE9wsy1WZMfYMxoMX2+KOeq+HJgqHy
ucl/sNaM+FtR9KVrkPphn8aPbCuuua+bZzmaED8EvJS67lXelsQPwR5LrtVs
G1PPUDypyVjdzqf6yEKRNRHcosinfKxtmO4L2LBRZ4n421Dn9KDmK+hTfBj6
q75doRvJJX4Y6nfc69KuYPofjtuVWnsXKmh+ZOFoFGZ01OTaED8C5yVZFplT
RrQeAVHto47gPqYfkShoLR/h21gQPxKRWr0negLNaT0KRjOHfm2QT5K/KGQo
pLKo41bEj4anecoP6zMtiB+N8eBPYgcePaP6bkerN+eXp0LG/3bs4pxr+i3T
ivgxaHAyXGncP6bZL4uBT1N7mYvakvixOJh0qpZrY0f8WBj8fFVZfZiZjzhY
vbeoyI8mv7I41Dz+WH31sjXx+Xi8pK8Ou6Cr0eCjS4GyviEO8fhQdazYdcKB
6qXk4/2Rev17dasofzxOKtvC1+TQfCIe5i6nOyePOFC+eIxebn99VkDzrIzH
TPTCnfwl5nsW4Hktb3c1216jIUBV8oHBU0fpfDIBPhp+KE+N/1tzfqUAv5fm
frA7xpb8C2HwMnB+uYE0hLiSLD5su/yC+iHEsPTF8M4cqo9SiCIrvcD09baU
XwTWzLHC5eoDmv0QoVxHW8vzCTOvIlinm2woOGNH/kX45un1Fp+hBeqfGK/u
298qN3uT/ItxqVPcWOq8lvyL4f5OxmTzEM23Uoz52kblu6PUb5YEHit7+WWp
juRfgqCOwpxPb9P/RSbBQteIkXTqDYqXwLC0ca6gx4H8J6Cn2fJisJzmDQmI
e57O/u4Sza8sAV9r80662lE/lAmYy/sstsnChvInQv/PyrN+HG2KT4RvaL79
+YuM/0TkJavVuuNM/RIhMTQThMbM0Pl3QGDUXVz841v4F6FPyBA=
     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  ImageMargins->0.,
  ImageSize->Small,
  PlotRange->{{0., 100.}, {-9.817566916843232, 10.20812466100742}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 107},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->82449866]
}, Open  ]],

Cell["Fit the data:", "ExampleText",
 CellID->256354851],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindFit", "[", 
  RowBox[{"%%", ",", 
   RowBox[{"a", " ", 
    RowBox[{"x", "^", "b"}], " ", 
    RowBox[{"Cos", "[", 
     RowBox[{"c", " ", "x"}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->831553048],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"a", "\[Rule]", "0.909615805946977`"}], ",", 
   RowBox[{"b", "\[Rule]", "0.5241771181544256`"}], ",", 
   RowBox[{"c", "\[Rule]", "1.0001962361694303`"}]}], "}"}]], "Output",
 ImageSize->{264, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->293369963]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Experimental Data",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->56107519],

Cell["Determine the approximate complexity of a computation:", "ExampleText",
 CellID->233528021],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"data", "=", 
  RowBox[{"Block", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"n", ",", "v"}], "}"}], ",", 
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"n", "=", 
        RowBox[{"2", "^", "k"}]}], ";", 
       RowBox[{"v", "=", 
        RowBox[{"RandomComplex", "[", 
         RowBox[{
          RowBox[{"1", "+", "I"}], ",", "n"}], "]"}]}], ";", 
       RowBox[{"{", 
        RowBox[{"n", ",", 
         RowBox[{"First", "[", 
          RowBox[{"Timing", "[", 
           RowBox[{"Fourier", "[", "v", "]"}], "]"}], "]"}]}], "}"}]}], ",", 
      RowBox[{"{", 
       RowBox[{"k", ",", "16", ",", "20"}], "}"}]}], "]"}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->50305132],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"65536", ",", "0.0780000000000255`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"131072", ",", "0.14100000000002544`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"262144", ",", "0.20400000000001361`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"524288", ",", "0.45299999999997453`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1048576", ",", "1.0309999999999744`"}], "}"}]}], "}"}]], "Output",\

 ImageSize->{363, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->59492223]
}, Open  ]],

Cell["Fit to the theoretical model for the complexity of an FFT.", \
"ExampleText",
 CellID->547014642],

Cell[BoxData[
 RowBox[{
  RowBox[{"model", "=", 
   RowBox[{"c", " ", "n", " ", 
    RowBox[{"Log", "[", "n", "]"}]}]}], ";"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->167968953],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"fit", "=", 
  RowBox[{"FindFit", "[", 
   RowBox[{"data", ",", "model", ",", "c", ",", "n"}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->120560214],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"c", "\[Rule]", "6.990849737626811`*^-8"}], "}"}]], "Output",
 ImageSize->{126, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->19214972]
}, Open  ]],

Cell["Show the timing data and the fit in a log-log plot:", "ExampleText",
 CellID->70723954],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LogLogPlot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{"model", "/.", "fit"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", 
     RowBox[{"2", "^", "16"}], ",", 
     RowBox[{"2", "^", "20"}]}], "}"}], ",", 
   RowBox[{"Epilog", "\[Rule]", 
    RowBox[{"Map", "[", 
     RowBox[{"Point", ",", 
      RowBox[{"Log", "[", "data", "]"}]}], "]"}]}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->524019058],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{403, 241},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{400, Automatic}},
 CellLabel->"Out[4]=",
 CellID->201963988]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->430150368],

Cell["Fit a model for lubricant viscosity to experimental data:", \
"ExampleText",
 CellID->497012135],

Cell[BoxData[
 RowBox[{
  RowBox[{"data", " ", "=", " ", 
   RowBox[{"Import", "[", 
    RowBox[{"\"\<ExampleData/lubricant.tsv\>\"", ",", "\"\<TSV\>\"", ",", 
     RowBox[{"\"\<HeaderLines\>\"", "\[Rule]", "7"}]}], "]"}]}], 
  ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->90706713],

Cell[TextData[{
 "Model for lubricant viscosity as a function of temperature ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "1"], TraditionalForm]], "InlineMath"],
 " and pressure/1000 ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "2"], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->235035511],

Cell[BoxData[
 RowBox[{
  RowBox[{"model", "=", 
   RowBox[{
    FractionBox[
     SubscriptBox["\[Theta]", "1"], 
     RowBox[{
      SubscriptBox["\[Theta]", "2"], "+", 
      SubscriptBox["x", "1"]}]], "+", 
    RowBox[{
     SubscriptBox["\[Theta]", "3"], 
     SubscriptBox["x", "2"]}], "+", 
    RowBox[{
     SubscriptBox["\[Theta]", "4"], 
     SuperscriptBox[
      SubscriptBox["x", "2"], "2"]}], "+", 
    RowBox[{
     SubscriptBox["\[Theta]", "5"], 
     SuperscriptBox[
      SubscriptBox["x", "2"], "3"]}], "+", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       SubscriptBox["\[Theta]", "6"], "+", 
       RowBox[{
        SubscriptBox["\[Theta]", "7"], 
        SuperscriptBox[
         SubscriptBox["x", "2"], "2"]}]}], ")"}], 
     SubscriptBox["x", "2"], " ", 
     RowBox[{"Exp", "[", 
      FractionBox[
       RowBox[{"-", 
        SubscriptBox["x", "1"]}], 
       RowBox[{
        SubscriptBox["\[Theta]", "8"], "+", 
        RowBox[{
         SubscriptBox["\[Theta]", "9"], 
         SuperscriptBox[
          SubscriptBox["x", "2"], "2"]}]}]], "]"}]}]}]}], ";"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->13701714],

Cell["Scale the pressure to improve conditioning of the model:", "ExampleText",
 CellID->36776710],

Cell[BoxData[{
 RowBox[{
  RowBox[{"sdata", "=", "data"}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"sdata", "[", 
    RowBox[{"[", 
     RowBox[{"All", ",", "2"}], "]"}], "]"}], "*=", "0.001"}], 
  ";"}]}], "Input",
 CellLabel->"In[3]:=",
 CellID->183188250],

Cell[TextData[{
 "Find the parameters ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[Theta]", "i"], TraditionalForm]], "InlineMath"],
 " that best fit the scaled data:"
}], "ExampleText",
 CellID->840290591],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"fit", "=", 
  RowBox[{"FindFit", "[", 
   RowBox[{"sdata", ",", "model", ",", 
    RowBox[{"{", 
     RowBox[{
      SubscriptBox["\[Theta]", "1"], ",", 
      SubscriptBox["\[Theta]", "2"], ",", 
      SubscriptBox["\[Theta]", "3"], ",", 
      SubscriptBox["\[Theta]", "4"], ",", 
      SubscriptBox["\[Theta]", "5"], ",", 
      SubscriptBox["\[Theta]", "6"], ",", 
      SubscriptBox["\[Theta]", "7"], ",", 
      SubscriptBox["\[Theta]", "8"], ",", 
      SubscriptBox["\[Theta]", "9"]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      SubscriptBox["x", "1"], ",", 
      SubscriptBox["x", "2"]}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->124043390],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    SubscriptBox["\[Theta]", "1"], "\[Rule]", "1054.8625781979827`"}], ",", 
   RowBox[{
    SubscriptBox["\[Theta]", "2"], "\[Rule]", "206.6110671176068`"}], ",", 
   RowBox[{
    SubscriptBox["\[Theta]", "3"], "\[Rule]", "1.459978302907963`"}], ",", 
   RowBox[{
    SubscriptBox["\[Theta]", "4"], "\[Rule]", 
    RowBox[{"-", "0.2595485763212314`"}]}], ",", 
   RowBox[{
    SubscriptBox["\[Theta]", "5"], "\[Rule]", "0.022553203705195927`"}], ",", 
   
   RowBox[{
    SubscriptBox["\[Theta]", "6"], "\[Rule]", "0.4017968455972736`"}], ",", 
   RowBox[{
    SubscriptBox["\[Theta]", "7"], "\[Rule]", "0.03526749690737607`"}], ",", 
   RowBox[{
    SubscriptBox["\[Theta]", "8"], "\[Rule]", "57.4342581228009`"}], ",", 
   RowBox[{
    SubscriptBox["\[Theta]", "9"], "\[Rule]", 
    RowBox[{"-", "0.4759943284212547`"}]}]}], "}"}]], "Output",
 ImageSize->{500, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->148477185]
}, Open  ]],

Cell["Show a plot of the fit with the scaled experimental data:", \
"ExampleText",
 CellID->27339907],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"Plot3D", "[", 
     RowBox[{
      RowBox[{"Evaluate", "[", 
       RowBox[{"model", "/.", "fit"}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{
        SubscriptBox["x", "1"], ",", "0", ",", "100"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        SubscriptBox["x", "2"], ",", "0", ",", "8"}], "}"}]}], "]"}], ",", 
    RowBox[{"Graphics3D", "[", 
     RowBox[{"{", 
      RowBox[{"Red", ",", 
       RowBox[{"PointSize", "[", ".025", "]"}], ",", 
       RowBox[{"Map", "[", 
        RowBox[{"Point", ",", "sdata"}], "]"}]}], "}"}], "]"}]}], "}"}], 
  "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->934645470],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 148},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->58759160]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->616353932],

Cell[TextData[{
 "Fit a model for the rate of isomerization of ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 "-Pentane to isopentane with experimental data:"
}], "ExampleText",
 CellID->370945209],

Cell[BoxData[
 RowBox[{
  RowBox[{"data", "=", 
   RowBox[{"Import", "[", 
    RowBox[{
    "\"\<ExampleData/isomerization.dat\>\"", ",", "\"\<Table\>\"", ",", 
     RowBox[{"\"\<HeaderLines\>\"", "\[Rule]", "7"}]}], "]"}]}], 
  ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->448973570],

Cell[TextData[{
 "Nonlinear model for rate as a function of Hydrogen ",
 Cell[BoxData[
  FormBox[
   RowBox[{"(", 
    SubscriptBox["x", "1"], ")"}], TraditionalForm]], "InlineMath"],
 ", ",
 StyleBox["n",
  FontSlant->"Italic"],
 "-Pentane ",
 Cell[BoxData[
  FormBox[
   RowBox[{"(", 
    SubscriptBox["x", "2"], ")"}], TraditionalForm]], "InlineMath"],
 ", and isopentane ",
 Cell[BoxData[
  FormBox[
   RowBox[{"(", 
    SubscriptBox["x", "3"], ")"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->558097503],

Cell[BoxData[
 RowBox[{
  RowBox[{"model", "=", 
   RowBox[{
    SubscriptBox["\[Theta]", "1"], " ", 
    SubscriptBox["\[Theta]", "3"], 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       SubscriptBox["x", "2"], "-", 
       RowBox[{
        SubscriptBox["x", "3"], "/", "1.632"}]}], ")"}], "/", 
     RowBox[{"(", 
      RowBox[{"1", "+", 
       RowBox[{
        SubscriptBox["\[Theta]", "2"], 
        SubscriptBox["x", "1"]}], "+", 
       RowBox[{
        SubscriptBox["\[Theta]", "3"], 
        SubscriptBox["x", "2"]}], "+", 
       RowBox[{
        SubscriptBox["\[Theta]", "4"], 
        SubscriptBox["x", "3"]}]}], ")"}]}]}]}], ";"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->560143462],

Cell[TextData[{
 "Find the parameters ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[Theta]", "i"], TraditionalForm]], "InlineMath"],
 " that best fit the data starting with values having the right size:"
}], "ExampleText",
 CellID->202452456],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindFit", "[", 
  RowBox[{"data", ",", "model", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       SubscriptBox["\[Theta]", "1"], ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       SubscriptBox["\[Theta]", "2"], ",", ".1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       SubscriptBox["\[Theta]", "3"], ",", ".1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       SubscriptBox["\[Theta]", "4"], ",", ".1"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     SubscriptBox["x", "1"], ",", 
     SubscriptBox["x", "2"], ",", 
     SubscriptBox["x", "3"]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->280600424],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    SubscriptBox["\[Theta]", "1"], "\[Rule]", "36.027658701825864`"}], ",", 
   RowBox[{
    SubscriptBox["\[Theta]", "2"], "\[Rule]", "0.08154214879852004`"}], ",", 
   RowBox[{
    SubscriptBox["\[Theta]", "3"], "\[Rule]", "0.04296370785877883`"}], ",", 
   RowBox[{
    SubscriptBox["\[Theta]", "4"], "\[Rule]", "0.19268114041489706`"}]}], 
  "}"}]], "Output",
 ImageSize->{397, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->90732187]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Differential Equations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->89447407],

Cell["\<\
Find parameters in a differential equation model that gives the best fit:\
\>", "ExampleText",
 CellID->759987326],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"sol", "=", 
   RowBox[{"First", "[", 
    RowBox[{"x", "/.", 
     RowBox[{"NDSolve", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{
          RowBox[{
           RowBox[{
            RowBox[{"x", "''"}], "[", "t", "]"}], "+", 
           RowBox[{".33", " ", 
            RowBox[{
             RowBox[{"x", "'"}], "[", "t", "]"}]}], "+", 
           RowBox[{".72", 
            RowBox[{
             RowBox[{"x", "[", "t", "]"}], "^", "3"}]}]}], "\[Equal]", "0"}], 
         ",", 
         RowBox[{
          RowBox[{"x", "[", "0", "]"}], "\[Equal]", "2"}], ",", 
         RowBox[{
          RowBox[{
           RowBox[{"x", "'"}], "[", "0", "]"}], "\[Equal]", "0"}]}], "}"}], 
       ",", "x", ",", 
       RowBox[{"{", 
        RowBox[{"t", ",", "20"}], "}"}]}], "]"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"times", "=", 
   RowBox[{"N", "[", 
    RowBox[{
     RowBox[{"Range", "[", 
      RowBox[{"0", ",", "100"}], "]"}], "/", "5"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"data", "=", 
   RowBox[{"Transpose", "[", 
    RowBox[{"{", 
     RowBox[{"times", ",", 
      RowBox[{
       RowBox[{"sol", "[", "times", "]"}], "+", 
       RowBox[{"RandomReal", "[", 
        RowBox[{".1", ",", "101"}], "]"}]}]}], "}"}], "]"}]}], ";"}], "\n", 
 RowBox[{"lp", "=", 
  RowBox[{"ListPlot", "[", 
   RowBox[{"data", ",", 
    RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->185936742],

Cell[BoxData[
 GraphicsBox[{{}, 
   {Hue[0.67, 0.6, 0.6], PointBox[CompressedData["
1:eJw10wswXGcUAOAVJHaoeia0gl32ei77ENp43ENK03RIyEQblVRSrzYz2jEi
pCutSBAiXUmnXY+iHksrkxRJkNiuYGMMyXrWa1kqkayiVGS9Sit79szcuXPn
nv/77z3n/LTTX4ZE7aBQKFH/X9t3dRA30uKPuVIgP287OkjqDwt1ReGbpOp5
mGypVa4lSNdI7uuYIse1M+7LFpfwvYLkKYI/kyfOkCptgbzUkHtU0ijD/CUy
lLPrj4RH1WT063hFjg+tXEiv6Rer1q+SF2MnymR3n4ofd23HBhmrrdi67j0v
xs+D9oHxPKN3lzBfAxoeZt9+mb8sVvma4DXG8qekK3G9FhSY/jPSIVCKVfvt
hCbXO9lZA8vo6cCnkmRj31K1R4WNktvljoEL6OmCMvATHrNCgZ4eJB1Lcz8S
OYmePuRWT5a7SwfQM4Cj/WcJ67lmzDcAi03hudnWSqyPIWjq6cXe1OnG/zeC
ueAsf63KEayPMbwpkzTddJjC+pnA7z8Zt/v5TpMqzwTW5XH3/7WYQc8UeHNJ
9aZDf6O3G2JsZN++YC2itwd2CTJ6vsleQM8M3BtuOa5cWETPDPalPo60ZSyi
Zw4+Ae15tQUL6L0FtJTwK7Y1s+i9DWPKa7lEp7q/FpBVFqjovaxAzwLyf23e
nG59it5eiJk3aSvgT6JnCUEluiKKRD0PVtD1fYCX3Yt+9KwBAps4Zg7q+lgD
mbFYku/Xhr41WA5YnRlfFuF6GpQ6UY8LGutxPxrIwsR+/nZ89OigldsfrD9a
hf2iw9lLC9KuXgn2hw4pB0R2zzw7sd824BaWfrpDtwfnwQZKE2wydCy6sb+2
8CBlymNjaRA9W6ixEMQRPQPo2cJcxPkH65mD6DFAOJ49G/TGOHoMeGaoPF5h
P4YeATr673gYb8rRI0BQbCZtGJGjR0BS0ynqjRA5enZwZw/wQxIm0bODLmHf
oUNXJ9CzB8ectLtDFLVnD+znPwaX0ybRswcms/jVNeoYeg7QfS8t9WrmKHoO
cDLvZEcVVYaeI1T019A+bJWh5wi79+UcrKePoucI2rOztWvECHpOIKz3zjQJ
V59nJ1i9bmUVWdyLnjOcuSgSJhZK0XMGL+Yl93Oeveg5w5p4NUxe9QQ9Jtjb
efKeV0vRY8J6soENnf8EPRfw7id4rsmd6LkA/56PmzBFff5cQPy5aPq3skfo
uUIcNbTuI98W9FyBVajxXrT8IXos0BQb1RJRzZjPgs39bdHDGpXos6DEOCi0
t7AM17Ngx1bl15ejanA/FphuDZe0F9Whxwb+qQqeuVMGemzYy/iLsbPhZ5xv
Nsy/PzT8XWcpzi8bEk8QoUfis9Bjg8BDMh2Z/wvOMweKDuYc9oqvxfnnwBLB
MzJ/WYkeB+R9oTMrMXXoccCp2vBj/4BUPD8cmPhg+it+RDF6XIg4UbBfz6cA
PS70mc9Krng0oscFc2t3y5jhWvS4cL570PCLllvoceFPRhq9qb4aPTeQig43
L3tVkv8BtAnXJg==
     "]]}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotRange->{All, All},
  PlotRangeClipping->True,
  PlotRangePadding->{Automatic, Automatic}]], "Output",
 ImageSize->{184, 104},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->602594220]
}, Open  ]],

Cell[TextData[{
 "Define model so the solution for a set of parameters ",
 Cell[BoxData[
  FormBox[
   RowBox[{"{", 
    RowBox[{"\[Gamma]", ",", "a", ",", "b"}], "}"}], TraditionalForm]], 
  "InlineMath"],
 " is reused for all ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 " values:"
}], "ExampleText",
 CellID->246277593],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"model", "[", 
    RowBox[{
     RowBox[{"\[Gamma]_", "?", "NumberQ"}], ",", 
     RowBox[{"a_", "?", "NumberQ"}], ",", 
     RowBox[{"b_", "?", "NumberQ"}]}], "]"}], ":=", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"model", "[", 
      RowBox[{"\[Gamma]", ",", "a", ",", "b"}], "]"}], "=", 
     RowBox[{"First", "[", 
      RowBox[{"x", "/.", 
       RowBox[{"NDSolve", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{
            RowBox[{
             RowBox[{
              RowBox[{"x", "''"}], "[", "t", "]"}], "+", 
             RowBox[{"\[Gamma]", " ", 
              RowBox[{
               RowBox[{"x", "'"}], "[", "t", "]"}]}], "+", 
             RowBox[{"a", " ", 
              RowBox[{"x", "[", "t", "]"}]}], "+", 
             RowBox[{"b", " ", 
              RowBox[{
               RowBox[{"x", "[", "t", "]"}], "^", "3"}]}]}], "\[Equal]", 
            "0"}], ",", 
           RowBox[{
            RowBox[{"x", "[", "0", "]"}], "\[Equal]", "2"}], ",", 
           RowBox[{
            RowBox[{
             RowBox[{"x", "'"}], "[", "0", "]"}], "\[Equal]", "0"}]}], "}"}], 
         ",", "x", ",", 
         RowBox[{"{", 
          RowBox[{"t", ",", "20"}], "}"}]}], "]"}]}], "]"}]}], ")"}]}], 
  ";"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->19020788],

Cell["\<\
Find the fit with tolerance appropriate to the precision of the solutions of \
the ODE:\
\>", "ExampleText",
 CellID->619280420],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"fit", "=", 
  RowBox[{"FindFit", "[", 
   RowBox[{"data", ",", 
    RowBox[{
     RowBox[{"model", "[", 
      RowBox[{"\[Gamma]", ",", "a", ",", "b"}], "]"}], "[", "x", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"\[Gamma]", ",", ".1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"a", ",", ".1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"b", ",", "1"}], "}"}]}], "}"}], ",", "x", ",", 
    RowBox[{"PrecisionGoal", "\[Rule]", "4"}], ",", 
    RowBox[{"AccuracyGoal", "\[Rule]", "4"}]}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->145117175],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"\[Gamma]", "\[Rule]", "0.3386539633274514`"}], ",", 
   RowBox[{"a", "\[Rule]", "0.035869838195409495`"}], ",", 
   RowBox[{"b", "\[Rule]", "0.7054357237241016`"}]}], "}"}]], "Output",
 ImageSize->{286, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->191573774]
}, Open  ]],

Cell["Show the fitted model with the data and the residuals:", "ExampleText",
 CellID->240791162],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"Show", "[", 
    RowBox[{
     RowBox[{"Plot", "[", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"model", "[", 
          RowBox[{"\[Gamma]", ",", "a", ",", "b"}], "]"}], "[", "x", "]"}], "/.",
         "fit"}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "20"}], "}"}]}], "]"}], ",", "lp"}], 
    "]"}], ",", 
   RowBox[{"ListPlot", "[", 
    RowBox[{"Transpose", "[", 
     RowBox[{"{", 
      RowBox[{"times", ",", 
       RowBox[{
        RowBox[{"data", "[", 
         RowBox[{"[", 
          RowBox[{"All", ",", "2"}], "]"}], "]"}], "-", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{
           RowBox[{"model", "[", 
            RowBox[{"\[Gamma]", ",", "a", ",", "b"}], "]"}], "[", "times", 
           "]"}], "/.", "fit"}], ")"}]}]}], "}"}], "]"}], "]"}]}], 
  "}"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->639620771],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{386, 108},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->271178654]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->5874383],

Cell["\<\
Find how a nonsoliton initial condition for the KdV equation breaks down into \
two solitons:\
\>", "ExampleText",
 CellID->283959408],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"xl", "=", "25"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"tend", "=", "15"}], ";"}], "\n", 
 RowBox[{"sol", "=", 
  RowBox[{"First", "[", 
   RowBox[{"u", "/.", 
    RowBox[{"NDSolve", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{"D", "[", 
           RowBox[{
            RowBox[{"u", "[", 
             RowBox[{"t", ",", "x"}], "]"}], ",", "t"}], "]"}], "+", 
          RowBox[{"D", "[", 
           RowBox[{
            RowBox[{"u", "[", 
             RowBox[{"t", ",", "x"}], "]"}], ",", 
            RowBox[{"{", 
             RowBox[{"x", ",", "3"}], "}"}]}], "]"}], "+", 
          RowBox[{"3", " ", 
           RowBox[{"u", "[", 
            RowBox[{"t", ",", "x"}], "]"}], " ", 
           RowBox[{"D", "[", 
            RowBox[{
             RowBox[{"u", "[", 
              RowBox[{"t", ",", "x"}], "]"}], ",", "x"}], "]"}]}]}], 
         "\[Equal]", "0"}], ",", 
        RowBox[{
         RowBox[{"u", "[", 
          RowBox[{"0", ",", "x"}], "]"}], "\[Equal]", 
         RowBox[{"3", " ", 
          RowBox[{
           RowBox[{"Sech", "[", 
            RowBox[{"x", "/", "2"}], "]"}], "^", "2"}]}]}], ",", 
        RowBox[{
         RowBox[{"u", "[", 
          RowBox[{"t", ",", 
           RowBox[{"-", "xl"}]}], "]"}], "\[Equal]", 
         RowBox[{"u", "[", 
          RowBox[{"t", ",", 
           RowBox[{"3", " ", "xl"}]}], "]"}]}]}], "}"}], ",", "u", ",", 
      RowBox[{"{", 
       RowBox[{"t", ",", "0", ",", "tend"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", 
        RowBox[{"-", "xl"}], ",", 
        RowBox[{"3", "xl"}]}], "}"}], ",", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<MethodOfLines\>\"", ",", 
         RowBox[{"\"\<SpatialDiscretization\>\"", "\[Rule]", 
          RowBox[{"{", 
           RowBox[{"\"\<TensorProductGrid\>\"", ",", 
            RowBox[{
            "\"\<DifferenceOrder\>\"", "\[Rule]", 
             "\"\<Pseudospectral\>\""}]}], "}"}]}]}], "}"}]}]}], "]"}]}], 
   "]"}]}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->499266246],

Cell[BoxData[
 TagBox[
  RowBox[{"InterpolatingFunction", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"0.`", ",", "15.`"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "25.`"}], ",", "75.`"}], "}"}]}], "}"}], 
    ",", "\<\"<>\"\>"}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{351, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->759326705]
}, Open  ]],

Cell["The initial wave breaks into two separate waves:", "ExampleText",
 CellID->628084645],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"sol", "[", 
    RowBox[{"t", ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "tend"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "xl"}], ",", 
     RowBox[{"3", " ", "xl"}]}], "}"}], ",", 
   RowBox[{"PlotRange", "\[Rule]", "All"}], ",", 
   RowBox[{"PlotPoints", "\[Rule]", "100"}], ",", 
   RowBox[{"Mesh", "\[Rule]", "False"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->157219196],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 154},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->29905889]
}, Open  ]],

Cell[TextData[{
 "Get the spatial grid used by ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " to solve the PDE:"
}], "ExampleText",
 CellID->365473181],

Cell[BoxData[
 RowBox[{
  RowBox[{"xg", "=", 
   RowBox[{"sol", "[", 
    RowBox[{"[", 
     RowBox[{"3", ",", "2"}], "]"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->161037294],

Cell["The discretized solution at the final time:", "ExampleText",
 CellID->200967086],

Cell[BoxData[
 RowBox[{
  RowBox[{"dsol", "=", 
   RowBox[{"Transpose", "[", 
    RowBox[{"{", 
     RowBox[{"xg", ",", 
      RowBox[{"sol", "[", 
       RowBox[{"tend", ",", "xg"}], "]"}]}], "}"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->210495540],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{"dsol", ",", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->74395396],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzdWM1PU0EQf7z3WvATqygtUOXkx0kPRsVYW4pgUq0igl4kWpCPagSC3Lzh
QQ9604OFg0QwMRw8SPRqYmL05NWz/8hzduexu2/edtunBImk3bc785uZ38zO
bluKpbnJsQelufJoqfPibGlmsjz6sLNvehZEToNlWWvwfpS0YO7BjL/XPA+e
4XE7jC4AbHja8Owi6zRDMYG3/sek7f7T8Z8d3NcONsYQK0buyXKEB+t0Xai2
YFwNopUNcRWhx6W42PXzcmv6PVMTkfR9WTER188iFYnTLkWqq0FXDX1Spw8w
31w+Kb8b9DwcrabdW+/CenieipCPngnvq5jaz9UywX5Wu/wsWevPRocSW54N
PG3HSZx5gxQ7dWX5RzbzCl79V97Y8LBi5Yn3KIHXjcEVi0tdIYVJvG4JxBQS
9A5hhHcQ+t6fP/uCRGDSHOQFICfoHUiFmTJfsepSR0itRrJu4WsI7FeniVcL
4zN7h+Bdskb97eFV397G/WDbznfS6ro59BbmuUwld74CYBh7uhdwAnKcZzOV
7mxAC6qh68v5HBdmAiY4CWuFN9DmswuoAhgIGQERV2GimjAYZid7JUZ6h2d3
EjISNVb5BFwxuUxK1fomSlJCC6pwyqD1qxFMSniD5dXi0oWexWLhtcPQjSjE
l1BZNtO5QgcusP96exbFYeV7lhD9h5xIAoKHWj7fO0RDUuCU+0KP575/+zU9
9eHTx5/hCYANWrCOpNV6Qxio6nEycmdVqQ49GyewNsVLSwAavvUOnP4Z0Uha
bEEtPsHUjaJHQT7Qz26ZGagF79f96h2KvZvLVl6++GqIrQpxBx2yDt2rO/mB
ARTsPHYOVM7axlEYFRRWnqzjfP30yWefcavWN//sdIVvGJuUe3Lg2jLQyhOW
cRJJ73k3sWoiVpeJXu8lGWIzrsU1K7crqw7lOGGwojWTnHqJ10MGL9IqRqz6
SBS9lz3EyiVWfSSKS/BJ4ZV9eieItUusi0SfNHCSVoNkrbfSfzs4QPiWtagE
QdmEp8lKomzC02QlUVMRstFL97IhLtiLyT0DWrAWE4lmO7mPo8IXid6nnlcL
qWqBrA9yqyMeftuln82cgfzmBAwLpN5of5TYuVo/4jMeLqUC2YHqfpz1SgTu
qvt1xW0hUeqzGic1SkWwkpVJicqyvYxr0ZJZ6q87cCOljPHj/yTKRsY+7OHv
aHpO5sVeq7829ahqPY7aY0TrkPWIF+zNdAQr2ZtpkU3wtyZljHayS9MiT11P
/6turb7D9CS3aW3GSHVMKFkLPWor93yULneq7m+tXouCUjtJnoFaN22UCOZc
tuY+bd7te9cLno8Orc2oFzwfJpQ8H3pUFK6UXbsBJdmZUJKdHmVih/8Psxp+
Aximtyg=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 123},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->196737441]
}, Open  ]],

Cell["Model of two solitons with different velocity:", "ExampleText",
 CellID->381346136],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"model", "[", 
    RowBox[{"t_", ",", "x_"}], "]"}], "=", 
   RowBox[{
    RowBox[{
     RowBox[{"k1", "^", "2"}], " ", 
     RowBox[{
      RowBox[{"Sech", "[", 
       RowBox[{"k1", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{"x", "-", 
           RowBox[{
            RowBox[{"k1", "^", "2"}], " ", "t"}]}], ")"}], "/", "2"}]}], 
       "]"}], "^", "2"}]}], "+", 
    RowBox[{
     RowBox[{"k2", "^", "2"}], " ", 
     RowBox[{
      RowBox[{"Sech", "[", 
       RowBox[{"k2", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{"x", "-", 
           RowBox[{
            RowBox[{"k2", "^", "2"}], " ", "t"}]}], ")"}], "/", "2"}]}], 
       "]"}], "^", "2"}]}]}]}], ";"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->432777935],

Cell["\<\
Find fit with starting conditions estimated from the graph of the discretized \
solution:\
\>", "ExampleText",
 CellID->106690545],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"fit", "=", 
  RowBox[{"FindFit", "[", 
   RowBox[{"dsol", ",", 
    RowBox[{"model", "[", 
     RowBox[{"tend", ",", "x"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"k1", ",", "1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"k2", ",", "2"}], "}"}]}], "}"}], ",", "x"}], "]"}]}]], "Input",\

 CellLabel->"In[7]:=",
 CellID->4291732],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"k1", "\[Rule]", "0.9633722990536158`"}], ",", 
   RowBox[{"k2", "\[Rule]", "2.0097459668147026`"}]}], "}"}]], "Output",
 ImageSize->{195, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->185531360]
}, Open  ]],

Cell["\<\
Plots of the two soliton model and residuals at intermediate times:\
\>", "ExampleText",
 CellID->202748041],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"Show", "[", 
       RowBox[{
        RowBox[{"ListPlot", "[", 
         RowBox[{
          RowBox[{"sol", "[", 
           RowBox[{"t", ",", "xg"}], "]"}], ",", 
          RowBox[{"PlotRange", "\[Rule]", "All"}], ",", 
          RowBox[{"DataRange", "\[Rule]", 
           RowBox[{"{", 
            RowBox[{
             RowBox[{"-", "xl"}], ",", 
             RowBox[{"3", " ", "xl"}]}], "}"}]}]}], "]"}], ",", 
        RowBox[{"Plot", "[", 
         RowBox[{
          RowBox[{
           RowBox[{"model", "[", 
            RowBox[{"t", ",", "x"}], "]"}], "/.", "fit"}], ",", 
          RowBox[{"{", 
           RowBox[{"x", ",", 
            RowBox[{"-", "xl"}], ",", 
            RowBox[{"3", " ", "xl"}]}], "}"}], ",", 
          RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]}], "]"}], ",", 
      RowBox[{"ListPlot", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"sol", "[", 
          RowBox[{"t", ",", "xg"}], "]"}], "-", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"model", "[", 
            RowBox[{"t", ",", "xg"}], "]"}], "/.", "fit"}], ")"}]}], ",", 
        RowBox[{"PlotRange", "\[Rule]", "All"}], ",", 
        RowBox[{"DataRange", "\[Rule]", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"-", "xl"}], ",", 
           RowBox[{"3", " ", "xl"}]}], "}"}]}]}], "]"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "tend", ",", 
      RowBox[{"tend", "/", "3"}]}], "}"}]}], "]"}], "//", "Grid"}]], "Input",
 CellLabel->"In[8]:=",
 CellID->89257198],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{368, 515},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->94696131]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->174534589],

Cell[TextData[{
 "For linear models, ",
 Cell[BoxData[
  ButtonBox["Fit",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Fit"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["FindFit",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindFit"]], "InlineFormula"],
 " are equivalent and use ",
 Cell[BoxData[
  ButtonBox["LeastSquares",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LeastSquares"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->441279876],

Cell[BoxData[
 RowBox[{
  RowBox[{"data", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"0", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "4"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4", ",", "10"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"5", ",", "20"}], "}"}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->50948481],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["Fit",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Fit"]], "InlineFormula"],
 " with the basis ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{"1", ",", " ", 
    StyleBox["x", "TI"], ",", " ", 
    SuperscriptBox[
     StyleBox["x", "TI"], "2"]}], "}"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->359675166],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"basis", "=", 
   RowBox[{"{", 
    RowBox[{"1", ",", "x", ",", 
     RowBox[{"x", "^", "2"}]}], "}"}]}], ";"}], "\n", 
 RowBox[{"Fit", "[", 
  RowBox[{"data", ",", "basis", ",", "x"}], "]"}]}], "Input",
 CellLabel->"In[2]:=",
 CellID->16038186],

Cell[BoxData[
 RowBox[{"1.321428571428571`", "\[InvisibleSpace]", "-", 
  RowBox[{"2.3821428571428567`", " ", "x"}], "+", 
  RowBox[{"1.1964285714285712`", " ", 
   SuperscriptBox["x", "2"]}]}]], "Output",
 ImageSize->{202, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->520964260]
}, Open  ]],

Cell[TextData[{
 "This is equivalent to ",
 Cell[BoxData[
  ButtonBox["FindFit",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindFit"]], "InlineFormula"],
 " with the model function a linear combination of the basis:"
}], "ExampleText",
 CellID->517816379],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"model", "=", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"a", "[", "0", "]"}], ",", 
      RowBox[{"a", "[", "1", "]"}], ",", 
      RowBox[{"a", "[", "2", "]"}]}], "}"}], ".", "basis"}]}], ";"}], "\n", 
 RowBox[{"FindFit", "[", 
  RowBox[{"data", ",", "model", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"a", "[", "0", "]"}], ",", 
     RowBox[{"a", "[", "1", "]"}], ",", 
     RowBox[{"a", "[", "2", "]"}]}], "}"}], ",", "x"}], "]"}]}], "Input",
 CellLabel->"In[3]:=",
 CellID->94537532],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"a", "[", "0", "]"}], "\[Rule]", "1.321428571428571`"}], ",", 
   RowBox[{
    RowBox[{"a", "[", "1", "]"}], "\[Rule]", 
    RowBox[{"-", "2.3821428571428567`"}]}], ",", 
   RowBox[{
    RowBox[{"a", "[", "2", "]"}], "\[Rule]", "1.1964285714285712`"}]}], 
  "}"}]], "Output",
 ImageSize->{328, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->567333560]
}, Open  ]],

Cell[TextData[{
 "Construct a design matrix and use ",
 Cell[BoxData[
  ButtonBox["LeastSquares",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LeastSquares"]], "InlineFormula"],
 " to find the coefficients:"
}], "ExampleText",
 CellID->525347866],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"m", "=", 
  RowBox[{"Map", "[", 
   RowBox[{
    RowBox[{"Function", "[", 
     RowBox[{
      RowBox[{"{", "x", "}"}], ",", 
      RowBox[{"Evaluate", "[", "basis", "]"}]}], "]"}], ",", 
    RowBox[{"data", "[", 
     RowBox[{"[", 
      RowBox[{"All", ",", "1"}], "]"}], "]"}]}], "]"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->600072301],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "1", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "4"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "3", ",", "9"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "4", ",", "16"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "5", ",", "25"}], "}"}]}], "}"}]], "Output",
 ImageSize->{433, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->42230904]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LeastSquares", "[", 
  RowBox[{"m", ",", 
   RowBox[{"N", "[", 
    RowBox[{"data", "[", 
     RowBox[{"[", 
      RowBox[{"All", ",", "2"}], "]"}], "]"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->156648758],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.3214285714285734`", ",", 
   RowBox[{"-", "2.3821428571428567`"}], ",", "1.196428571428571`"}], 
  "}"}]], "Output",
 ImageSize->{196, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->134729421]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->55697299],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FindFit",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindFit"]], "InlineFormula"],
 " effectively uses ",
 Cell[BoxData[
  ButtonBox["FindMinimum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindMinimum"]], "InlineFormula"],
 " to minimize a norm of the residual function:"
}], "ExampleText",
 CellID->129377990],

Cell[BoxData[{
 RowBox[{
  RowBox[{"data", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"0", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "4"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4", ",", "10"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"5", ",", "20"}], "}"}]}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"xp", ",", "response"}], "}"}], "=", 
   RowBox[{"Transpose", "[", "data", "]"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->238300117],

Cell[BoxData[
 RowBox[{
  RowBox[{"model", "=", 
   RowBox[{"a", " ", 
    RowBox[{"Exp", "[", 
     RowBox[{"b", " ", "x"}], "]"}]}]}], ";"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->293034305],

Cell["\<\
The residual is the difference between response and model prediction:\
\>", "ExampleText",
 CellID->80546646],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"residual", "=", 
  RowBox[{"response", "-", 
   RowBox[{"Map", "[", 
    RowBox[{
     RowBox[{"Function", "[", 
      RowBox[{
       RowBox[{"{", "x", "}"}], ",", 
       RowBox[{"Evaluate", "[", "model", "]"}]}], "]"}], ",", "xp"}], 
    "]"}]}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->405386996],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"1", "-", "a"}], ",", 
   RowBox[{
    RowBox[{"-", "a"}], " ", 
    SuperscriptBox["\[ExponentialE]", "b"]}], ",", 
   RowBox[{"3", "-", 
    RowBox[{"a", " ", 
     SuperscriptBox["\[ExponentialE]", 
      RowBox[{"2", " ", "b"}]]}]}], ",", 
   RowBox[{"4", "-", 
    RowBox[{"a", " ", 
     SuperscriptBox["\[ExponentialE]", 
      RowBox[{"3", " ", "b"}]]}]}], ",", 
   RowBox[{"10", "-", 
    RowBox[{"a", " ", 
     SuperscriptBox["\[ExponentialE]", 
      RowBox[{"4", " ", "b"}]]}]}], ",", 
   RowBox[{"20", "-", 
    RowBox[{"a", " ", 
     SuperscriptBox["\[ExponentialE]", 
      RowBox[{"5", " ", "b"}]]}]}]}], "}"}]], "Output",
 ImageSize->{342, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->360238027]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{"residual", ".", "residual"}], ",", 
   RowBox[{"{", 
    RowBox[{"a", ",", "b"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->2657863],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"2.510894928183351`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"a", "\[Rule]", "0.5072247144521088`"}], ",", 
     RowBox[{"b", "\[Rule]", "0.7357440870932601`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{262, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->5152634]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindFit", "[", 
  RowBox[{"data", ",", "model", ",", 
   RowBox[{"{", 
    RowBox[{"a", ",", "b"}], "}"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->86975213],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"a", "\[Rule]", "0.5072247145192513`"}], ",", 
   RowBox[{"b", "\[Rule]", "0.7357440870657572`"}]}], "}"}]], "Output",
 ImageSize->{188, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->511241654]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->309013018],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Starting Values",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->739487787],

Cell["\<\
Good starting values may be necessary to get a reasonable fit:\
\>", "ExampleText",
 CellID->3297576],

Cell[BoxData[
 RowBox[{
  RowBox[{"model", " ", "=", " ", 
   RowBox[{
    RowBox[{"a1", " ", 
     RowBox[{"Exp", "[", 
      RowBox[{"-", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"b1", 
          RowBox[{"(", 
           RowBox[{"x", "-", "x1"}], ")"}]}], ")"}], "^", "2"}]}], "]"}]}], 
    " ", "+", " ", 
    RowBox[{"a2", " ", 
     RowBox[{"Exp", "[", 
      RowBox[{"-", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"b2", 
          RowBox[{"(", 
           RowBox[{"x", "-", "x2"}], ")"}]}], ")"}], "^", "2"}]}], 
      "]"}]}]}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->25917882],

Cell[BoxData[
 RowBox[{
  RowBox[{"data", " ", "=", " ", 
   RowBox[{"Block", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"a1", "=", "1"}], ",", 
       RowBox[{"b1", "=", "5"}], ",", 
       RowBox[{"x1", "=", 
        RowBox[{"-", ".5"}]}], ",", 
       RowBox[{"a2", "=", "2"}], ",", 
       RowBox[{"b2", "=", "10"}], ",", 
       RowBox[{"x2", "=", ".25"}], ",", 
       RowBox[{"x", "=", 
        RowBox[{"Sort", "[", 
         RowBox[{"RandomReal", "[", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{
             RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", "100"}], "]"}], 
         "]"}]}]}], "}"}], ",", "\[IndentingNewLine]", 
     RowBox[{"Transpose", "[", 
      RowBox[{"{", 
       RowBox[{"x", ",", " ", 
        RowBox[{"model", " ", "+", " ", 
         RowBox[{"RandomReal", "[", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{
             RowBox[{"-", ".1"}], ",", ".1"}], "}"}], ",", " ", "100"}], 
          "]"}]}]}], "}"}], "]"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->95944592],

Cell["\<\
Find the fit with the default starting values of 1. The local minimum is \
nowhere near:\
\>", "ExampleText",
 CellID->289703782],

Cell[BoxData[
 RowBox[{
  RowBox[{"fit", " ", "=", " ", 
   RowBox[{"FindFit", "[", 
    RowBox[{"data", ",", " ", "model", ",", " ", 
     RowBox[{"{", 
      RowBox[{
      "a1", ",", "b1", ",", " ", "x1", ",", " ", "a2", ",", " ", "b2", ",", 
       " ", "x2"}], "}"}], ",", "x"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->220126268],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", " ", 
  RowBox[{
   RowBox[{"ListPlot", "[", 
    RowBox[{"data", ",", " ", 
     RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}], ",", 
   RowBox[{"Plot", "[", 
    RowBox[{
     RowBox[{"Evaluate", "[", 
      RowBox[{"model", " ", "/.", " ", "fit"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", 
       RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", " ", 
     RowBox[{"PlotStyle", "\[Rule]", 
      RowBox[{"Directive", "[", 
       RowBox[{"Thick", ",", "Red"}], "]"}]}]}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->101239967],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzNWd1vVEUUH/Zud9uFthBpi62wpULSUEKIMX4ES7fbD0rZVtBIY4RoxQ/6
ICDWR9NHidE3fbA1WEg1EhJ9gGDiE4mJwQfjq8/+I9eZOXPnnP3d2dtLA8Qm
u/fOmXN+5zfnY+be7dzi8sX3P1pcXrqwOHji6uKVi0sXPhmcvnxVi6JtSqm7
+vPxHqXvY31nP3fjWF/ld1l/R3qqYEbmJrZ/qqBvq/rGXCMxp4yu/kxmzFUz
MK2+aocZKy3yWKm9KUkZbGqbauzd1EuBLpEj7/hHm9r16UsxsX9CLP9fMdsK
m7aYamY/2FZbag4+ZjbSBn2FWaU0crAru9WU3LXqrokcPQ86edHKf9j4e2z0
26++/N3WZhvIdqEO+KLSXrDMvJabO2CkJS9tzK7rm/GxVaqJhTd+1MOpie9c
dNmHwVZtND9RW3NSZGu11A6QzghbjU1hUgctEy89PX/TQI87Jn8++PfypTu/
3vvHMWFmxITm5xs3nDQSqyWR5MLWlkv7tc/v6/GVS3eIi6dqs9LuUV47s6EZ
eCoFWFdnkCmPiROFeOz4qhZJTmjHuM+HotoO8fNDy7lIQ7FysWbkjfnpaI7d
xPiazQHJo3hlJaaj5FW7b9K5Ym5Px8pl3OjfSqpXVZp7QhMpaFgVce4jMy5R
Veo/7VoL9bAPugK7L9e4w4/1X1JWa84zxmIg4epFxTw+jIWPd0Hk0/vgyvY3
th7aZEeqZDemmJW9Fw327ju3z5+7tfLZb8pGr3juzZ+aPaa6cSSwL6R72Ndd
U0S4Wp4R9aTHny7fS2eLACtCszZm6tuupeQ9+JsIfNjKUt3mu9vXHc1JejSl
u5D7fCRU21z9mpXFLslIEh6tyw/rtTVaF1Ul1SmBVqA/d8FYsHc5qB1f1bky
cwWxg5w8cR3WNAJIhYSTViJVg2326Wct+s0bf81MX5+bXZfowv4QMxHSomA2
Ob4mdyNRDUdCqxJ1ftZi94JWG8T/EKB2BlH6bXXLymdEew7wOaVvIrFbmMnt
wY5kn/Pso6kShpsr2e9DVKtmcgcgyw712VM9dko3gh6/ff62ekH4k7mz/sqy
+qmHzSq7YnG+p3JDHuq2kw1F8jAAsR8NsRXztHvqmjk1u/7FtfsO5S2wwjO7
B8YvZlhx3ntgzFb8xMLMdosc06FK+lXw/TJ4HQAvLwkrlg4Byu4gCrORKCwd
AlREoUqeDK0spVUHRr0uLgtilj31BjEOAkY/eE6eJENvY3hm9Yfi7DwwD7vn
ZL4p4NtdGXB3AiL5OSC7aub7FnVXyeU98nL2shMiI726/Xl0tUXd5vOaXvMH
7/2sEV4/s6Eawhtx+ObrP5r6mpmlvSVvPZgx8sI7Ank5FbRP9yni8J5AOI24
+W0EragCk25e8KO0zjBk8WiGFkc9S4uj1Q3Y+4JW4ROIfe1zq3ga9I4Ja9bG
Z3r0uQf0JQpbHQla8fyxYEQ7gOEsY7vuWfrwFzd32K2qAnykDdZ+BfCrQf/M
UmLRG5M+z2QlC7+IJXPDWvjmk2XVOprhOB0F6XBmhA7b5pBPJKyBT0pDueJE
iPItjTT4CZ6yR9NPxa1/L8OePsv4+HZZ9s87lGZ6tEmdSulfKcId/5z5LvkH
F7+R9eTCMwqEM7dF/0/Org9y/Eos98XklxTMRB/UElkl+yR2Oc+G99aHkbbe
h7fDSmYytOB3kBZavIKwVv7TAaW9wHU0Q4u5ZmnB03EuVibL2HutnpzC9VWC
MXZxfqvWPbM1H8U4XLmttbPGHK9H87SA0k6oh+kMLa6H6SZ/XVAHYYyH6bou
YDWVocWspppY4Tt5GONR7wWkY54S/X8qcp0tDSFVSYYf4349j8yCdtwb+fSl
Bv5GXveRCdjlQOI8133Esk4K/M2onjun9D86te0/+fJDYw==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 113},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->1884422]
}, Open  ]],

Cell["\<\
Typically, better starting values are required to get a good fit:\
\>", "ExampleText",
 CellID->142231484],

Cell[BoxData[
 RowBox[{
  RowBox[{"sfit", " ", "=", " ", 
   RowBox[{"FindFit", "[", 
    RowBox[{"data", ",", " ", "model", ",", " ", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"a1", ",", "1"}], "}"}], ",", " ", 
       RowBox[{"{", 
        RowBox[{"b1", ",", "5"}], "}"}], ",", " ", 
       RowBox[{"{", 
        RowBox[{"x1", ",", 
         RowBox[{"-", ".5"}]}], "}"}], ",", 
       RowBox[{"{", " ", 
        RowBox[{"a2", ",", "2"}], "}"}], ",", " ", 
       RowBox[{"{", 
        RowBox[{"b2", ",", "10"}], "}"}], ",", " ", 
       RowBox[{"{", 
        RowBox[{"x2", ",", ".25"}], "}"}]}], "}"}], ",", "x"}], "]"}]}], 
  ";"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->169431678],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", " ", 
  RowBox[{
   RowBox[{"ListPlot", "[", 
    RowBox[{"data", ",", " ", 
     RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}], ",", 
   RowBox[{"Plot", "[", 
    RowBox[{
     RowBox[{"Evaluate", "[", 
      RowBox[{"model", " ", "/.", " ", "sfit"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", 
       RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", " ", 
     RowBox[{"PlotStyle", "\[Rule]", "Red"}]}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->442472748],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 113},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->95882332]
}, Open  ]]
}, Closed]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->20192],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["FindMinimum",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FindMinimum"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Fit",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Fit"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["NMinimize",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NMinimize"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["LeastSquares",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/LeastSquares"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Interpolation",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Interpolation"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["ListSurfacePlot3D",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ListSurfacePlot3D"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["ChiSquareDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ChiSquareDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->32076]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->26380],

Cell[TextData[ButtonBox["Manipulating Numerical Data",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/ManipulatingNumericalData"]], "Tutorials",
 CellID->17773],

Cell[TextData[ButtonBox["Curve Fitting",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/CurveFitting"]], "Tutorials",
 CellID->3867],

Cell[TextData[ButtonBox["Unconstrained Optimization",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/UnconstrainedOptimizationOverview"]], "Tutorials",
 CellID->123070281]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection",
 CellID->28044],

Cell[TextData[{
 ButtonBox["Demonstrations with FindFit",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/symbol.html?symbol=FindFit"], 
    None}],
 " (",
 ButtonBox["Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None}],
 ")"
}], "RelatedLinks"],

Cell[TextData[ButtonBox["Implementation notes: Numerical and Related \
Functions",
 BaseStyle->"Hyperlink",
 ButtonData->{
   URL["http://reference.wolfram.com/mathematica/note/\
SomeNotesOnInternalImplementation.html#20880"], None}]], "RelatedLinks",
 CellID->8698]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->13071],

Cell[TextData[ButtonBox["Curve Fitting & Approximate Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/CurveFittingAndApproximateFunctions"]], "MoreAbout",\

 CellID->378120448],

Cell[TextData[ButtonBox["Data Transforms and Smoothing",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DataTransformsAndSmoothing"]], "MoreAbout",
 CellID->165492331],

Cell[TextData[ButtonBox["Matrix-Based Minimization",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/MatrixBasedMinimization"]], "MoreAbout",
 CellID->66543557],

Cell[TextData[ButtonBox["Numerical Data",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NumericalData"]], "MoreAbout",
 CellID->498447318],

Cell[TextData[ButtonBox["Optimization",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/Optimization"]], "MoreAbout",
 CellID->28288771],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->38511423],

Cell[TextData[ButtonBox["New in 6.0: Numerical Data Handling",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60NumericalDataHandling"]], "MoreAbout",
 CellID->607692000]
}, Open  ]],

Cell["New in 5  |  Last modified in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"FindFit - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 42, 53.3432728}", "context" -> "System`", 
    "keywords" -> {
     "approximation of functions", "conjugate gradient", "curves", 
      "fitting curves", "fitting differential equations", "fitting surfaces", 
      "Gauss-Newton algorithm", "GNA", "gradient descent", 
      "least squares curve fitting", "least-squares fits", 
      "Levenberg Marquardt", "Levenberg-Marquardt algorithm", "LMA", "Newton",
       "Newton's method", "nonlinear fitting", "nonlinear programming", 
      "polynomial fit", "quasi-Newton", "residuals", "steepest-descent", 
      "testing models", "trend line"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "FindFit[data, expr, pars, vars] finds numerical values of the parameters \
pars that make expr give a best fit to data as a function of vars. The data \
can have the form {{x_1, y_1, ..., f_1}, {x_2, y_2, ..., f_2}, ...}, where \
the number of coordinates x, y, ... is equal to the number of variables in \
the list vars. The data can also be of the form {f_1, f_2, ...}, with a \
single coordinate assumed to take values 1, 2, .... FindFit[data, {expr, \
cons}, pars, vars] finds a best fit subject to the parameter constraints \
cons.", "synonyms" -> {"find fit", "find fit"}, "title" -> "FindFit", "type" -> 
    "Symbol", "uri" -> "ref/FindFit"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[21595, 649, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->348580231]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 249282, 6728}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 3869, 83, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[4492, 110, 4191, 83, 70, "ObjectNameGrid"],
Cell[8686, 195, 3431, 107, 70, "Usage",
 CellID->19752]
}, Open  ]],
Cell[CellGroupData[{
Cell[12154, 307, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[12632, 324, 431, 17, 70, "Notes",
 CellID->8382],
Cell[13066, 343, 318, 12, 70, "Notes",
 CellID->13616],
Cell[13387, 357, 287, 11, 70, "Notes",
 CellID->28781],
Cell[13677, 370, 220, 8, 70, "Notes",
 CellID->8808],
Cell[13900, 380, 112, 3, 70, "Notes",
 CellID->14609],
Cell[14015, 385, 1368, 50, 70, "Notes",
 CellID->10299],
Cell[15386, 437, 212, 7, 70, "Notes",
 CellID->22146],
Cell[15601, 446, 403, 15, 70, "Notes",
 CellID->22732],
Cell[16007, 463, 330, 9, 70, "Notes",
 CellID->19298],
Cell[16340, 474, 69, 1, 70, "Notes",
 CellID->26940],
Cell[16412, 477, 2867, 80, 70, "3ColumnTableMod",
 CellID->30188],
Cell[19282, 559, 520, 20, 70, "Notes",
 CellID->18960],
Cell[19805, 581, 576, 19, 70, "Notes",
 CellID->637],
Cell[20384, 602, 498, 17, 70, "Notes",
 CellID->4185],
Cell[20885, 621, 673, 23, 70, "Notes",
 CellID->24902]
}, Closed]],
Cell[CellGroupData[{
Cell[21595, 649, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->348580231],
Cell[CellGroupData[{
Cell[21980, 663, 148, 5, 70, "ExampleSection",
 CellID->907094202],
Cell[22131, 670, 82, 1, 70, "ExampleText",
 CellID->6495732],
Cell[CellGroupData[{
Cell[22238, 675, 207, 7, 28, "Input",
 CellID->425550223],
Cell[22448, 684, 414, 11, 36, "Output",
 CellID->673387210]
}, Open  ]],
Cell[22877, 698, 301, 12, 70, "ExampleText",
 CellID->32517813],
Cell[CellGroupData[{
Cell[23203, 714, 322, 10, 28, "Input",
 CellID->9895969],
Cell[23528, 726, 357, 10, 36, "Output",
 CellID->179251908]
}, Open  ]],
Cell[23900, 739, 69, 1, 70, "ExampleText",
 CellID->169120],
Cell[CellGroupData[{
Cell[23994, 744, 322, 11, 70, "Input",
 CellID->613462901],
Cell[24319, 757, 479, 12, 54, "Output",
 CellID->104697178]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[24847, 775, 220, 7, 70, "ExampleSection",
 CellID->456238338],
Cell[CellGroupData[{
Cell[25092, 786, 246, 7, 70, "ExampleSubsection",
 CellID->11860700],
Cell[25341, 795, 84, 1, 70, "ExampleText",
 CellID->261070299],
Cell[25428, 798, 477, 16, 70, "Input",
 CellID->613023938],
Cell[25908, 816, 218, 8, 70, "Input",
 CellID->211120279],
Cell[CellGroupData[{
Cell[26151, 828, 230, 7, 70, "Input",
 CellID->932959552],
Cell[26384, 837, 301, 9, 36, "Output",
 CellID->304977752]
}, Open  ]],
Cell[26700, 849, 71, 1, 70, "ExampleText",
 CellID->71387768],
Cell[CellGroupData[{
Cell[26796, 854, 250, 8, 70, "Input",
 CellID->16189946],
Cell[27049, 864, 390, 13, 39, "Output",
 CellID->10060483]
}, Open  ]],
Cell[27454, 880, 72, 1, 70, "ExampleText",
 CellID->629282411],
Cell[CellGroupData[{
Cell[27551, 885, 325, 10, 70, "Input",
 CellID->83329518],
Cell[27879, 897, 2299, 42, 70, "Output",
 Evaluatable->False,
 CellID->302447679]
}, Open  ]],
Cell[30193, 942, 75, 1, 70, "ExampleText",
 CellID->368492844],
Cell[30271, 945, 201, 7, 70, "Input",
 CellID->301688966],
Cell[30475, 954, 121, 3, 70, "ExampleText",
 CellID->112903641],
Cell[CellGroupData[{
Cell[30621, 961, 186, 6, 70, "Input",
 CellID->829000504],
Cell[30810, 969, 365, 10, 36, "Output",
 CellID->150673020]
}, Open  ]],
Cell[CellGroupData[{
Cell[31212, 984, 502, 16, 70, "Input",
 CellID->619343705],
Cell[31717, 1002, 992, 22, 118, "Output",
 CellID->288011517]
}, Open  ]],
Cell[32724, 1027, 124, 3, 70, "ExampleDelimiter",
 CellID->74394520],
Cell[32851, 1032, 132, 4, 70, "ExampleText",
 CellID->485657525],
Cell[32986, 1038, 849, 28, 70, "Input",
 CellID->109105924],
Cell[33838, 1068, 140, 4, 70, "ExampleText",
 CellID->160429556],
Cell[33981, 1074, 167, 6, 70, "Input",
 CellID->648832529],
Cell[CellGroupData[{
Cell[34173, 1084, 238, 7, 70, "Input",
 CellID->164817600],
Cell[34414, 1093, 300, 9, 36, "Output",
 CellID->349688790]
}, Open  ]],
Cell[34729, 1105, 126, 3, 70, "ExampleText",
 CellID->189393166],
Cell[CellGroupData[{
Cell[34880, 1112, 386, 13, 70, "Input",
 CellID->70031323],
Cell[35269, 1127, 299, 9, 36, "Output",
 CellID->288413531]
}, Open  ]],
Cell[35583, 1139, 125, 3, 70, "ExampleDelimiter",
 CellID->270889856],
Cell[35711, 1144, 102, 2, 70, "ExampleText",
 CellID->703708898],
Cell[35816, 1148, 476, 16, 70, "Input",
 CellID->93684799],
Cell[36295, 1166, 102, 2, 70, "ExampleText",
 CellID->1703653],
Cell[36400, 1170, 741, 25, 70, "Input",
 CellID->507606713],
Cell[CellGroupData[{
Cell[37166, 1199, 365, 12, 70, "Input",
 CellID->63559203],
Cell[37534, 1213, 375, 12, 36, "Output",
 CellID->407759799]
}, Open  ]],
Cell[37924, 1228, 68, 1, 70, "ExampleText",
 CellID->174378436],
Cell[37995, 1231, 844, 28, 70, "Input",
 CellID->245175184],
Cell[CellGroupData[{
Cell[38864, 1263, 366, 12, 70, "Input",
 CellID->339205902],
Cell[39233, 1277, 376, 12, 36, "Output",
 CellID->371695152]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[39658, 1295, 251, 7, 70, "ExampleSubsection",
 CellID->30298558],
Cell[39912, 1304, 66, 1, 70, "ExampleText",
 CellID->197867878],
Cell[39981, 1307, 203, 7, 70, "Input",
 CellID->15080643],
Cell[40187, 1316, 465, 15, 70, "Input",
 CellID->218875459],
Cell[40655, 1333, 122, 3, 70, "ExampleText",
 CellID->161917982],
Cell[CellGroupData[{
Cell[40802, 1340, 417, 13, 70, "Input",
 CellID->74490737],
Cell[41222, 1355, 309, 10, 36, "Output",
 CellID->894649802]
}, Open  ]],
Cell[41546, 1368, 79, 1, 70, "ExampleText",
 CellID->47073673],
Cell[CellGroupData[{
Cell[41650, 1373, 383, 12, 70, "Input",
 CellID->139019433],
Cell[42036, 1387, 4290, 74, 70, "Output",
 Evaluatable->False,
 CellID->197745188]
}, Open  ]],
Cell[46341, 1464, 137, 4, 70, "ExampleText",
 CellID->50181010],
Cell[46481, 1470, 487, 16, 70, "Input",
 CellID->696071961],
Cell[CellGroupData[{
Cell[46993, 1490, 175, 5, 70, "Input",
 CellID->158684639],
Cell[47171, 1497, 1623, 31, 70, "Output",
 Evaluatable->False,
 CellID->388765332]
}, Open  ]],
Cell[48809, 1531, 125, 3, 70, "ExampleDelimiter",
 CellID->568401064],
Cell[48937, 1536, 112, 3, 70, "ExampleText",
 CellID->339171382],
Cell[49052, 1541, 460, 17, 70, "Input",
 CellID->68570158],
Cell[49515, 1560, 801, 23, 70, "Input",
 CellID->504300830],
Cell[50319, 1585, 84, 1, 70, "ExampleText",
 CellID->152303581],
Cell[CellGroupData[{
Cell[50428, 1590, 293, 9, 70, "Input",
 CellID->628502664],
Cell[50724, 1601, 542, 13, 36, "Output",
 CellID->2584664]
}, Open  ]],
Cell[51281, 1617, 98, 1, 70, "ExampleText",
 CellID->72785105],
Cell[CellGroupData[{
Cell[51404, 1622, 343, 10, 70, "Input",
 CellID->109648129],
Cell[51750, 1634, 565, 14, 36, "Output",
 CellID->90699819]
}, Open  ]],
Cell[52330, 1651, 79, 1, 70, "ExampleText",
 CellID->171800968],
Cell[CellGroupData[{
Cell[52434, 1656, 611, 19, 70, "Input",
 CellID->61710716],
Cell[53048, 1677, 5479, 94, 70, "Output",
 Evaluatable->False,
 CellID->132412364]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[58576, 1777, 238, 7, 70, "ExampleSubsection",
 CellID->127486635],
Cell[58817, 1786, 79, 1, 70, "ExampleText",
 CellID->3453890],
Cell[58899, 1789, 457, 17, 70, "Input",
 CellID->53072381],
Cell[59359, 1808, 1129, 37, 70, "Input",
 CellID->292962567],
Cell[60491, 1847, 115, 3, 70, "ExampleText",
 CellID->103597510],
Cell[CellGroupData[{
Cell[60631, 1854, 414, 13, 70, "Input",
 CellID->84847018],
Cell[61048, 1869, 417, 11, 36, "Output",
 CellID->503473188]
}, Open  ]],
Cell[61480, 1883, 81, 1, 70, "ExampleText",
 CellID->670475389],
Cell[CellGroupData[{
Cell[61586, 1888, 636, 19, 70, "Input",
 CellID->391796545],
Cell[62225, 1909, 15035, 251, 70, "Output",
 Evaluatable->False,
 CellID->16310007]
}, Open  ]],
Cell[77275, 2163, 125, 3, 70, "ExampleDelimiter",
 CellID->435170072],
Cell[77403, 2168, 82, 1, 70, "ExampleText",
 CellID->129853208],
Cell[77488, 2171, 528, 17, 70, "Input",
 CellID->329142581],
Cell[78019, 2190, 1240, 43, 70, "Input",
 CellID->262167145],
Cell[CellGroupData[{
Cell[79284, 2237, 424, 12, 70, "Input",
 CellID->531529081],
Cell[79711, 2251, 476, 12, 36, "Output",
 CellID->328507755]
}, Open  ]],
Cell[80202, 2266, 122, 3, 70, "ExampleText",
 CellID->166969836],
Cell[80327, 2271, 621, 19, 70, "Input",
 CellID->926626446],
Cell[CellGroupData[{
Cell[80973, 2294, 137, 4, 70, "Input",
 CellID->134455144],
Cell[81113, 2300, 177, 5, 36, "Output",
 CellID->254192682]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[81351, 2312, 222, 7, 70, "ExampleSection",
 CellID->223291971],
Cell[CellGroupData[{
Cell[81598, 2323, 237, 7, 70, "ExampleSubsection",
 CellID->78762153],
Cell[81838, 2332, 122, 3, 70, "ExampleText",
 CellID->245920735],
Cell[81963, 2337, 1038, 32, 70, "Input",
 CellID->39132109],
Cell[CellGroupData[{
Cell[83026, 2373, 577, 19, 70, "Input",
 CellID->65187881],
Cell[83606, 2394, 283, 10, 36, "Output",
 CellID->274659422]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[83938, 2410, 229, 7, 70, "ExampleSubsection",
 CellID->442346239],
Cell[84170, 2419, 125, 3, 70, "ExampleText",
 CellID->261036267],
Cell[84298, 2424, 554, 19, 70, "Input",
 CellID->90513975],
Cell[84855, 2445, 273, 8, 70, "ExampleText",
 CellID->179578763],
Cell[CellGroupData[{
Cell[85153, 2457, 365, 12, 70, "Input",
 CellID->251205743],
Cell[85521, 2471, 369, 7, 70, "Message",
 CellID->476426630],
Cell[85893, 2480, 373, 7, 70, "Message",
 CellID->678958854],
Cell[86269, 2489, 404, 8, 70, "Message",
 CellID->659664696],
Cell[86676, 2499, 434, 8, 70, "Message",
 CellID->480711551],
Cell[87113, 2509, 318, 10, 36, "Output",
 CellID->732772992]
}, Open  ]],
Cell[87446, 2522, 323, 11, 70, "ExampleText",
 CellID->170632306],
Cell[CellGroupData[{
Cell[87794, 2537, 371, 13, 70, "Input",
 CellID->31397592],
Cell[88168, 2552, 1717, 56, 89, "Output",
 CellID->536748248]
}, Open  ]],
Cell[89900, 2611, 113, 3, 70, "ExampleText",
 CellID->341588118],
Cell[CellGroupData[{
Cell[90038, 2618, 469, 14, 70, "Input",
 CellID->61959277],
Cell[90510, 2634, 364, 10, 36, "Output",
 CellID->242178391]
}, Open  ]],
Cell[90889, 2647, 68, 1, 70, "ExampleText",
 CellID->4019287],
Cell[CellGroupData[{
Cell[90982, 2652, 471, 15, 70, "Input",
 CellID->80969814],
Cell[91456, 2669, 2734, 49, 70, "Output",
 Evaluatable->False,
 CellID->353661454]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[94239, 2724, 234, 7, 70, "ExampleSubsection",
 CellID->365923182],
Cell[94476, 2733, 268, 8, 70, "ExampleText",
 CellID->111416151],
Cell[94747, 2743, 1931, 62, 70, "Input",
 CellID->579614116],
Cell[96681, 2807, 87, 1, 70, "ExampleText",
 CellID->118008607],
Cell[CellGroupData[{
Cell[96793, 2812, 471, 15, 70, "Input",
 CellID->162448948],
Cell[97267, 2829, 418, 8, 70, "Message",
 CellID->182612031],
Cell[97688, 2839, 413, 11, 36, "Output",
 CellID->145944404]
}, Open  ]],
Cell[98116, 2853, 97, 1, 70, "ExampleText",
 CellID->11419712],
Cell[CellGroupData[{
Cell[98238, 2858, 526, 16, 70, "Input",
 CellID->239578397],
Cell[98767, 2876, 413, 11, 36, "Output",
 CellID->149704356]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[99229, 2893, 227, 7, 70, "ExampleSubsection",
 CellID->308617728],
Cell[99459, 2902, 344, 12, 70, "ExampleText",
 CellID->58344041],
Cell[99806, 2916, 461, 17, 70, "Input",
 CellID->112853019],
Cell[100270, 2935, 801, 23, 70, "Input",
 CellID->198224689],
Cell[101074, 2960, 64, 1, 70, "ExampleText",
 CellID->320088632],
Cell[CellGroupData[{
Cell[101163, 2965, 293, 9, 70, "Input",
 CellID->724733624],
Cell[101459, 2976, 542, 13, 36, "Output",
 CellID->309184904]
}, Open  ]],
Cell[102016, 2992, 70, 1, 70, "ExampleText",
 CellID->12121573],
Cell[CellGroupData[{
Cell[102111, 2997, 347, 10, 70, "Input",
 CellID->152087054],
Cell[102461, 3009, 566, 15, 36, "Output",
 CellID->11893923]
}, Open  ]],
Cell[103042, 3027, 122, 3, 70, "ExampleText",
 CellID->103264514],
Cell[CellGroupData[{
Cell[103189, 3034, 641, 20, 70, "Input",
 CellID->68814404],
Cell[103833, 3056, 5507, 94, 70, "Output",
 Evaluatable->False,
 CellID->698701185]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[109389, 3156, 229, 7, 70, "ExampleSubsection",
 CellID->31388],
Cell[CellGroupData[{
Cell[109643, 3167, 236, 8, 70, "Input",
 CellID->24887865],
Cell[109882, 3177, 414, 11, 36, "Output",
 CellID->109594591]
}, Open  ]],
Cell[110311, 3191, 93, 1, 70, "ExampleText",
 CellID->117745648],
Cell[CellGroupData[{
Cell[110429, 3196, 320, 10, 70, "Input",
 CellID->3720920],
Cell[110752, 3208, 356, 10, 36, "Output",
 CellID->65550885]
}, Open  ]],
Cell[111123, 3221, 76, 1, 70, "ExampleText",
 CellID->487219474],
Cell[CellGroupData[{
Cell[111224, 3226, 487, 16, 70, "Input",
 CellID->443269537],
Cell[111714, 3244, 355, 10, 36, "Output",
 CellID->692538862]
}, Open  ]],
Cell[CellGroupData[{
Cell[112106, 3259, 476, 15, 70, "Input",
 CellID->14151221],
Cell[112585, 3276, 358, 10, 36, "Output",
 CellID->840827165]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[112992, 3292, 232, 7, 70, "ExampleSubsection",
 CellID->244192407],
Cell[113227, 3301, 150, 4, 70, "ExampleText",
 CellID->398990986],
Cell[113380, 3307, 244, 9, 70, "Input",
 CellID->73123166],
Cell[113627, 3318, 112, 3, 70, "ExampleText",
 CellID->545405375],
Cell[113742, 3323, 965, 30, 70, "Input",
 CellID->947481304],
Cell[114710, 3355, 642, 20, 70, "Input",
 CellID->139363387],
Cell[115355, 3377, 92, 1, 70, "ExampleText",
 CellID->145906611],
Cell[CellGroupData[{
Cell[115472, 3382, 425, 14, 70, "Input",
 CellID->461283651],
Cell[115900, 3398, 1119, 22, 70, "Output",
 Evaluatable->False,
 CellID->347974402]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[117068, 3426, 237, 7, 70, "ExampleSubsection",
 CellID->276747749],
Cell[117308, 3435, 250, 8, 70, "ExampleText",
 CellID->429534],
Cell[117561, 3445, 1024, 33, 70, "Input",
 CellID->11960218],
Cell[CellGroupData[{
Cell[118610, 3482, 339, 10, 70, "Input",
 CellID->195890952],
Cell[118952, 3494, 290, 10, 36, "Output",
 CellID->33927605]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[119303, 3511, 227, 7, 70, "ExampleSection",
 CellID->107405395],
Cell[CellGroupData[{
Cell[119555, 3522, 238, 7, 70, "ExampleSubsection",
 CellID->308456881],
Cell[119796, 3531, 74, 1, 70, "ExampleText",
 CellID->31387452],
Cell[CellGroupData[{
Cell[119895, 3536, 210, 7, 70, "Input",
 CellID->61543534],
Cell[120108, 3545, 358, 10, 36, "Output",
 CellID->283340045]
}, Open  ]],
Cell[120481, 3558, 90, 1, 70, "ExampleText",
 CellID->628619116],
Cell[CellGroupData[{
Cell[120596, 3563, 353, 12, 70, "Input",
 CellID->652836235],
Cell[120952, 3577, 360, 10, 36, "Output",
 CellID->345233619]
}, Open  ]],
Cell[121327, 3590, 124, 3, 70, "ExampleDelimiter",
 CellID->92107301],
Cell[121454, 3595, 81, 1, 70, "ExampleText",
 CellID->554009824],
Cell[121538, 3598, 435, 15, 70, "Input",
 CellID->196749448],
Cell[CellGroupData[{
Cell[121998, 3617, 108, 3, 70, "Input",
 CellID->434322074],
Cell[122109, 3622, 2052, 43, 128, "Output",
 CellID->82449866]
}, Open  ]],
Cell[124176, 3668, 56, 1, 70, "ExampleText",
 CellID->256354851],
Cell[CellGroupData[{
Cell[124257, 3673, 323, 10, 70, "Input",
 CellID->831553048],
Cell[124583, 3685, 357, 10, 36, "Output",
 CellID->293369963]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[124989, 3701, 237, 7, 70, "ExampleSubsection",
 CellID->56107519],
Cell[125229, 3710, 97, 1, 70, "ExampleText",
 CellID->233528021],
Cell[CellGroupData[{
Cell[125351, 3715, 741, 24, 70, "Input",
 CellID->50305132],
Cell[126095, 3741, 598, 18, 54, "Output",
 CellID->59492223]
}, Open  ]],
Cell[126708, 3762, 103, 2, 70, "ExampleText",
 CellID->547014642],
Cell[126814, 3766, 180, 6, 70, "Input",
 CellID->167968953],
Cell[CellGroupData[{
Cell[127019, 3776, 179, 5, 70, "Input",
 CellID->120560214],
Cell[127201, 3783, 230, 7, 39, "Output",
 CellID->19214972]
}, Open  ]],
Cell[127446, 3793, 93, 1, 70, "ExampleText",
 CellID->70723954],
Cell[CellGroupData[{
Cell[127564, 3798, 454, 14, 70, "Input",
 CellID->524019058],
Cell[128021, 3814, 4136, 72, 70, "Output",
 Evaluatable->False,
 CellID->201963988]
}, Open  ]],
Cell[132172, 3889, 125, 3, 70, "ExampleDelimiter",
 CellID->430150368],
Cell[132300, 3894, 102, 2, 70, "ExampleText",
 CellID->497012135],
Cell[132405, 3898, 286, 8, 70, "Input",
 CellID->90706713],
Cell[132694, 3908, 321, 11, 70, "ExampleText",
 CellID->235035511],
Cell[133018, 3921, 1146, 40, 70, "Input",
 CellID->13701714],
Cell[134167, 3963, 98, 1, 70, "ExampleText",
 CellID->36776710],
Cell[134268, 3966, 264, 10, 70, "Input",
 CellID->183188250],
Cell[134535, 3978, 208, 7, 70, "ExampleText",
 CellID->840290591],
Cell[CellGroupData[{
Cell[134768, 3989, 701, 20, 70, "Input",
 CellID->124043390],
Cell[135472, 4011, 1028, 28, 54, "Output",
 CellID->148477185]
}, Open  ]],
Cell[136515, 4042, 101, 2, 70, "ExampleText",
 CellID->27339907],
Cell[CellGroupData[{
Cell[136641, 4048, 703, 22, 70, "Input",
 CellID->934645470],
Cell[137347, 4072, 15689, 261, 70, "Output",
 Evaluatable->False,
 CellID->58759160]
}, Open  ]],
Cell[153051, 4336, 125, 3, 70, "ExampleDelimiter",
 CellID->616353932],
Cell[153179, 4341, 211, 6, 70, "ExampleText",
 CellID->370945209],
Cell[153393, 4349, 288, 9, 70, "Input",
 CellID->448973570],
Cell[153684, 4360, 533, 21, 70, "ExampleText",
 CellID->558097503],
Cell[154220, 4383, 701, 24, 70, "Input",
 CellID->560143462],
Cell[154924, 4409, 244, 7, 70, "ExampleText",
 CellID->202452456],
Cell[CellGroupData[{
Cell[155193, 4420, 707, 23, 70, "Input",
 CellID->280600424],
Cell[155903, 4445, 545, 16, 36, "Output",
 CellID->90732187]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[156497, 4467, 242, 7, 70, "ExampleSubsection",
 CellID->89447407],
Cell[156742, 4476, 124, 3, 70, "ExampleText",
 CellID->759987326],
Cell[CellGroupData[{
Cell[156891, 4483, 1505, 48, 70, "Input",
 CellID->185936742],
Cell[158399, 4533, 1980, 40, 125, "Output",
 CellID->602594220]
}, Open  ]],
Cell[160394, 4576, 345, 12, 70, "ExampleText",
 CellID->246277593],
Cell[160742, 4590, 1353, 41, 70, "Input",
 CellID->19020788],
Cell[162098, 4633, 138, 4, 70, "ExampleText",
 CellID->619280420],
Cell[CellGroupData[{
Cell[162261, 4641, 624, 18, 70, "Input",
 CellID->145117175],
Cell[162888, 4661, 367, 10, 36, "Output",
 CellID->191573774]
}, Open  ]],
Cell[163270, 4674, 97, 1, 70, "ExampleText",
 CellID->240791162],
Cell[CellGroupData[{
Cell[163392, 4679, 935, 31, 70, "Input",
 CellID->639620771],
Cell[164330, 4712, 5349, 92, 70, "Output",
 Evaluatable->False,
 CellID->271178654]
}, Open  ]],
Cell[169694, 4807, 123, 3, 70, "ExampleDelimiter",
 CellID->5874383],
Cell[169820, 4812, 144, 4, 70, "ExampleText",
 CellID->283959408],
Cell[CellGroupData[{
Cell[169989, 4820, 2133, 63, 70, "Input",
 CellID->499266246],
Cell[172125, 4885, 479, 18, 36, "Output",
 CellID->759326705]
}, Open  ]],
Cell[172619, 4906, 91, 1, 70, "ExampleText",
 CellID->628084645],
Cell[CellGroupData[{
Cell[172735, 4911, 509, 15, 70, "Input",
 CellID->157219196],
Cell[173247, 4928, 27967, 463, 70, "Output",
 Evaluatable->False,
 CellID->29905889]
}, Open  ]],
Cell[201229, 5394, 227, 8, 70, "ExampleText",
 CellID->365473181],
Cell[201459, 5404, 193, 7, 70, "Input",
 CellID->161037294],
Cell[201655, 5413, 86, 1, 70, "ExampleText",
 CellID->200967086],
Cell[201744, 5416, 268, 9, 70, "Input",
 CellID->210495540],
Cell[CellGroupData[{
Cell[202037, 5429, 169, 5, 70, "Input",
 CellID->74395396],
Cell[202209, 5436, 1624, 31, 70, "Output",
 Evaluatable->False,
 CellID->196737441]
}, Open  ]],
Cell[203848, 5470, 89, 1, 70, "ExampleText",
 CellID->381346136],
Cell[203940, 5473, 796, 29, 70, "Input",
 CellID->432777935],
Cell[204739, 5504, 140, 4, 70, "ExampleText",
 CellID->106690545],
Cell[CellGroupData[{
Cell[204904, 5512, 401, 14, 70, "Input",
 CellID->4291732],
Cell[205308, 5528, 302, 9, 36, "Output",
 CellID->185531360]
}, Open  ]],
Cell[205625, 5540, 118, 3, 70, "ExampleText",
 CellID->202748041],
Cell[CellGroupData[{
Cell[205768, 5547, 1635, 47, 70, "Input",
 CellID->89257198],
Cell[207406, 5596, 13729, 229, 70, "Output",
 Evaluatable->False,
 CellID->94696131]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[221196, 5832, 237, 7, 70, "ExampleSection",
 CellID->174534589],
Cell[221436, 5841, 475, 18, 70, "ExampleText",
 CellID->441279876],
Cell[221914, 5861, 521, 18, 70, "Input",
 CellID->50948481],
Cell[222438, 5881, 372, 15, 70, "ExampleText",
 CellID->359675166],
Cell[CellGroupData[{
Cell[222835, 5900, 280, 9, 70, "Input",
 CellID->16038186],
Cell[223118, 5911, 336, 9, 36, "Output",
 CellID->520964260]
}, Open  ]],
Cell[223469, 5923, 261, 8, 70, "ExampleText",
 CellID->517816379],
Cell[CellGroupData[{
Cell[223755, 5935, 546, 17, 70, "Input",
 CellID->94537532],
Cell[224304, 5954, 470, 15, 36, "Output",
 CellID->567333560]
}, Open  ]],
Cell[224789, 5972, 250, 8, 70, "ExampleText",
 CellID->525347866],
Cell[CellGroupData[{
Cell[225064, 5984, 367, 12, 70, "Input",
 CellID->600072301],
Cell[225434, 5998, 599, 19, 36, "Output",
 CellID->42230904]
}, Open  ]],
Cell[CellGroupData[{
Cell[226070, 6022, 248, 8, 70, "Input",
 CellID->156648758],
Cell[226321, 6032, 289, 9, 36, "Output",
 CellID->134729421]
}, Open  ]],
Cell[226625, 6044, 124, 3, 70, "ExampleDelimiter",
 CellID->55697299],
Cell[226752, 6049, 368, 12, 70, "ExampleText",
 CellID->129377990],
Cell[227123, 6063, 672, 23, 70, "Input",
 CellID->238300117],
Cell[227798, 6088, 196, 7, 70, "Input",
 CellID->293034305],
Cell[227997, 6097, 119, 3, 70, "ExampleText",
 CellID->80546646],
Cell[CellGroupData[{
Cell[228141, 6104, 328, 11, 70, "Input",
 CellID->405386996],
Cell[228472, 6117, 822, 27, 39, "Output",
 CellID->360238027]
}, Open  ]],
Cell[CellGroupData[{
Cell[229331, 6149, 215, 7, 70, "Input",
 CellID->2657863],
Cell[229549, 6158, 371, 12, 36, "Output",
 CellID->5152634]
}, Open  ]],
Cell[CellGroupData[{
Cell[229957, 6175, 201, 6, 70, "Input",
 CellID->86975213],
Cell[230161, 6183, 300, 9, 36, "Output",
 CellID->511241654]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[230510, 6198, 230, 7, 70, "ExampleSection",
 CellID->309013018],
Cell[CellGroupData[{
Cell[230765, 6209, 236, 7, 70, "ExampleSubsection",
 CellID->739487787],
Cell[231004, 6218, 111, 3, 70, "ExampleText",
 CellID->3297576],
Cell[231118, 6223, 625, 23, 70, "Input",
 CellID->25917882],
Cell[231746, 6248, 1085, 33, 70, "Input",
 CellID->95944592],
Cell[232834, 6283, 139, 4, 70, "ExampleText",
 CellID->289703782],
Cell[232976, 6289, 353, 10, 70, "Input",
 CellID->220126268],
Cell[CellGroupData[{
Cell[233354, 6303, 593, 17, 70, "Input",
 CellID->101239967],
Cell[233950, 6322, 2150, 39, 70, "Output",
 Evaluatable->False,
 CellID->1884422]
}, Open  ]],
Cell[236115, 6364, 116, 3, 70, "ExampleText",
 CellID->142231484],
Cell[236234, 6369, 718, 22, 70, "Input",
 CellID->169431678],
Cell[CellGroupData[{
Cell[236977, 6395, 522, 15, 70, "Input",
 CellID->442472748],
Cell[237502, 6412, 4246, 74, 70, "Output",
 Evaluatable->False,
 CellID->95882332]
}, Open  ]]
}, Closed]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[241821, 6494, 311, 9, 70, "SeeAlsoSection",
 CellID->20192],
Cell[242135, 6505, 1736, 62, 70, "SeeAlso",
 CellID->32076]
}, Open  ]],
Cell[CellGroupData[{
Cell[243908, 6572, 314, 9, 70, "TutorialsSection",
 CellID->26380],
Cell[244225, 6583, 163, 3, 70, "Tutorials",
 CellID->17773],
Cell[244391, 6588, 135, 3, 70, "Tutorials",
 CellID->3867],
Cell[244529, 6593, 177, 4, 70, "Tutorials",
 CellID->123070281]
}, Open  ]],
Cell[CellGroupData[{
Cell[244743, 6602, 321, 9, 70, "RelatedLinksSection",
 CellID->28044],
Cell[245067, 6613, 359, 12, 70, "RelatedLinks"],
Cell[245429, 6627, 266, 6, 70, "RelatedLinks",
 CellID->8698]
}, Open  ]],
Cell[CellGroupData[{
Cell[245732, 6638, 315, 9, 70, "MoreAboutSection",
 CellID->13071],
Cell[246050, 6649, 186, 4, 70, "MoreAbout",
 CellID->378120448],
Cell[246239, 6655, 167, 3, 70, "MoreAbout",
 CellID->165492331],
Cell[246409, 6660, 159, 3, 70, "MoreAbout",
 CellID->66543557],
Cell[246571, 6665, 139, 3, 70, "MoreAbout",
 CellID->498447318],
Cell[246713, 6670, 135, 3, 70, "MoreAbout",
 CellID->28288771],
Cell[246851, 6675, 178, 3, 70, "MoreAbout",
 CellID->38511423],
Cell[247032, 6680, 175, 3, 70, "MoreAbout",
 CellID->607692000]
}, Open  ]],
Cell[247222, 6686, 50, 0, 70, "History"],
Cell[247275, 6688, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

