(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    391942,       8300]
NotebookOptionsPosition[    377936,       7824]
NotebookOutlinePosition[    379995,       7873]
CellTagsIndexPosition[    379908,       7868]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Unconstrained Optimization" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/UnconstrainedOptimizationOverview"], 
          "Numerical Optimization" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/NumericalOptimization"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Unconstrained Optimization\"\>", 
       2->"\<\"Numerical Optimization\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"FindMinimum" :> 
          Documentation`HelpLookup["paclet:ref/FindMinimum"], "NMaximize" :> 
          Documentation`HelpLookup["paclet:ref/NMaximize"], "Maximize" :> 
          Documentation`HelpLookup["paclet:ref/Maximize"], "FindFit" :> 
          Documentation`HelpLookup["paclet:ref/FindFit"], "LinearProgramming" :> 
          Documentation`HelpLookup["paclet:ref/LinearProgramming"], "D" :> 
          Documentation`HelpLookup["paclet:ref/D"], "FindRoot" :> 
          Documentation`HelpLookup["paclet:ref/FindRoot"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"FindMinimum\"\>", 
       2->"\<\"NMaximize\"\>", 3->"\<\"Maximize\"\>", 4->"\<\"FindFit\"\>", 
       5->"\<\"LinearProgramming\"\>", 6->"\<\"D\"\>", 
       7->"\<\"FindRoot\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Optimization" :> 
          Documentation`HelpLookup["paclet:guide/Optimization"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Optimization\"\>", 
       2->"\<\"New in 6.0: Mathematics & Algorithms\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["FindMaximum", "ObjectName"], Cell[BoxData[
     TooltipBox[
      TogglerBox[Dynamic[
        FrontEnd`CurrentValue[
         FrontEnd`EvaluationNotebook[], {
         TaggingRules, "ModificationHighlight"}]], {True->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVVly2zAMDQhxEylSsizLaxJn73KafvQAuUDv/1eApJuosceeTDAWF4B8
eFgk71///HoVV1dXK3p+0wNfIzdCLL4I6p1sYBYuPy3SI2AmRBpuaE/TPRsE
3Isg8mrB9l6ILQhhLkbd93MBuwVxAthdZwPr5wVVzBc8wAvMX8TFqPkqE8q6
rDCsKPjZ/g1guz+PKvLFfGsDU1R4A8zDhgLangX9x/WGM7A1qSQ7rraYkaLf
LHZ5FWY9H7rew/n4/6vWPKT2mXFBaH/NGejzqi/VnFEFJ2Rb1Lo+jtofllPD
RVJjdZLr51Eb/ED1K+QI109LcIg+sdSIWqa5g85L6MiNoRloTws+pdBKiFGl
mx3N0jcAlcsKeFivVkteVOgMSNsC18sYS/iIMdZoK43WSdQVK1SLlYy09bZ2
aEO+aoPiACNak0EBbtfLRIz9KJRcroaSKwE11BprpRHqiJ0BrxtJG4pLYqWc
RZO4BQwSbGT4rFivaLjjQdscjSvlih48Mu+WDisOBcB5ehgeDEZQlikkKh3b
KA1FMTDg9yFxLSe4XC37begHTDh5seS1I/gaW025Yo/yUFbmSalShzon1Kch
JTNn2hKS1hyWVIkf5n2LHEQgt0zeUNTkMaYrDdkMZdo3fJCDHx7hxzAC+8rs
XYBokTPfEJk6JVlZx3Yl0UnTYMPvSUt6zghf8dhIi5VDnwO+Hcb1MHK1msK+
oy5qkYS7wVC3cfiq8ojBcMtFMJXnDiIcLPFJClDFGkJBhXGEZV6VcCQFoey7
tyvBF6lw0uKH6nyQ5fqhrKx/Kxe8+3coucpGN4U5+RYux7vnhGO5P7hKenqz
xSO3ivjisf345fg5Pt1ScCRkkzR1U1Dur+NCec4LZ49Ynx9P04GmOvlnqcKZ
79BfJfcjkg==
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlLlKQ0EYRi/iA9gpWIgvIXbjM1iKWAjW2vgeZnWN+AgWZhEULCxELFyi
ZnfJavbeJOr/33xXcj8CNnY5EA7k3GWYmTuzq+uLa2OWZS38/MYt0DbDHH8Q
8uTB64bRNrmsQkaPRoQSGX1ZqZPR9/eEazL6klIlowf8wiUZ3ecV3snowYDw
SEafUmJk9HmlQh4cX4GM7vUId2T0VFIho28or2T0w5DQIKN3PoUaGX1FaZLR
0ymh6sz7hHLk9PMzoU5Gz6SFmvFsCbd9Xzj9qyc0nfe49xfuTyYUEzoQbmh8
+N8e37HTbVpk9MSzUCajbypFMvq0UiOjzyklMnqvKzRMJCwU+/vjzemTSpns
nh97feJmd0e4cvppTKiQ0bsdoW5ecorZDgpPtD6xqFA2fp+Qp/vtdUj1v797
Oqdwnb2Ov++fUVpk9PCJUCC7n4t5wz5xj9/+rj6ceXbPP86NbEb58/k56qP+
n/0bdGyYqQ==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}], False->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVdd22zAMDUBxSaJm5HgljrPTJN177/nWh35AfqD//1aAVGrrJLbbk+BI
FEmQFxcXlDQ9//3zHNfW1gZ0/6IbbsYOhahvCGrO9qHM/mnhmMMLoKv0RD6V
cPyZHTTcDI5NkYnQq4UQ0BdiAkKYZagl1OOw5fWtcgyvPp7AnefsGG0DBIeA
9RZVrNfcwBasb4nlbP9u4cbc/XD/zXcIgC2qYYazJTQiGSYHS0Hr8Yu5LV/h
ePvR5oMOKswAQ0M8R5OloFMWSNRE+IAVeALv38KX26wAV1uU5Ojv16PQy8q+
gMP+9ABW5C9KbkZi6qv19DGc/ID9b+CrZTjbbVagH3p91oIcL+GZ6JDNUet4
HpWUXxx5BaWZxRh1FQgH6JqoKcarF/2/dblez7IE0XmWGlFL/6ygchIqCmPo
CTSmDq9SaCUUhfI7K3pKlwJESZiAveHGxoA7ESYGpM2B62WMJXzEoojRRhpt
IlFHPKFyjGRBQ2fjBG0WttpMcYIFWhNAAXaHA0+M4yiUXK6UxJWAGmKNsdII
cYGVAadTSQPKS2KkEovGc8swk2ALhg8Tww1qdrjRNmSTtOUqHDhk3jktVpwK
QOLoZngwWICyTMFTqdhHMrQTDQPeazzXdgWXK+e4KV3AhH0US1Ergo8x16QV
R5QXZWWeJJW6qLNHPWq8mEFpS0hac1pSeX4YxjlyEhmFZfKGsqaIhd+Sks+Q
0i7lhZx8cwYPmx5wrMA+yaCwyMqnRCb2IiubsF9JTKRJMeX3JKd5VoS3OEyl
xShBFxLebXrDpsfVSlv2FZ2iHMn4NBg6bZy+ihxiZvjIFWAixyeIcLDNT1KC
qogha1Gh14NB6LXpSEpC2bm3y8O3FmHniF9U55INhnttz7pZuWDu79BqFZxJ
F2bhWzjo7Zx6HMvng6ukuztzvGJXa66NmF/+crzrHe1ScmTkk/SouqB8vq42
0jl0EnuF9/RsMR1Io4U/S5Wt+A79AaDyK/w=
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlUtrE1EYhgfxBwguFBTEvyAiFhfzHxRFvC0E17rQhYLiT7D3u/gTXNg2
3sBFFyIutE1qm17UZJKZZJK0KFjBtur7Td4jnhe7dTMPhAfyQObMnO9Mjl69
fvraniAI7v/+7A3IRri7/8VGWJwHVTH72qohZp+ZBnUx+yWjJWYfHwNvxOwX
jKaYfXAAzIrZ+/tARcw+NAhKYvaTRiJmP2gUxH+vLxKzl5cMMXtfL3gvZn84
Cdpi9hvGJzH7yxegJWa/YnTE7Mtl0HTPfZ/xyPWVZZCGvQ/Au65fSV9aNMLJ
CfDW9a0fIBWz/9wBHbcOf/74PX83W99j128ZNTH71BMQidmfFkAiZj9irIvZ
Fz+AWMx+yEjF7KeMVOw/32x/iuHoCHjt+s42aIfTU6DWna/Prm9vgVb4cc0I
h4fAgusZ62L2A0YsZj9h1MX+9bN9LHfP75zrZ42a2L9/3+wXjVTM7q8jm9NE
5q8wA+JwoB/8eX+eNxIx+00jErP3GA2xvy7uG88R+z2jI2Y/bLTE/v1l76WG
mxP/+r7ZN7+Bupj9mFERsz9/Bppi9myeEzH7QgnUxOy3jYaY/a7RFrNfNtpi
9nNGXcw+NgoaYnbOk2///PJ/bXXFcP37JojF7CPDIBazf/0CKmL2+TlQEbPf
MZpi9olxkIrZS0UQidn3G1Ux+3EjErOfMapi9mBX8p73vOc973n/P/0XnFJ2
Rg==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}]}, "\"\<Click me\>\""],
      "Toggle changes in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["FindMaximum",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/FindMaximum"], "[", 
       RowBox[{
        StyleBox["f", "TI"], ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["x", "TI"], ",", 
          SubscriptBox[
           StyleBox["x", "TI"], 
           StyleBox["0", "TR"]]}], "}"}]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]searches for a local maximum in ",
     Cell[BoxData[
      StyleBox["f", "TI"]], "InlineFormula"],
     ", starting from the point ",
     Cell[BoxData[
      RowBox[{
       StyleBox["x", "TI"], "=", 
       SubscriptBox[
        StyleBox["x", "TI"], 
        StyleBox["0", "TR"]]}]], "InlineFormula"],
     ". "
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["FindMaximum",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/FindMaximum"], "[", 
       RowBox[{
        StyleBox["f", "TI"], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            StyleBox["x", "TI"], ",", 
            SubscriptBox[
             StyleBox["x", "TI"], 
             StyleBox["0", "TR"]]}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{
            StyleBox["y", "TI"], ",", 
            SubscriptBox[
             StyleBox["y", "TI"], 
             StyleBox["0", "TR"]]}], "}"}], ",", 
          StyleBox["\[Ellipsis]", "TR"]}], "}"}]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]searches for a local maximum in a function of several \
variables. "
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["FindMaximum",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/FindMaximum"], "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          StyleBox["f", "TI"], ",", 
          StyleBox["cons", "TI"]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            StyleBox["x", "TI"], ",", 
            SubscriptBox[
             StyleBox["x", "TI"], 
             StyleBox["0", "TR"]]}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{
            StyleBox["y", "TI"], ",", 
            SubscriptBox[
             StyleBox["y", "TI"], 
             StyleBox["0", "TR"]]}], "}"}], ",", 
          StyleBox["\[Ellipsis]", "TR"]}], "}"}]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]searches for a local maximum subject to the \
constraints ",
     Cell[BoxData[
      StyleBox["cons", "TI"]], "InlineFormula"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["FindMaximum",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/FindMaximum"], "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          StyleBox["f", "TI"], ",", 
          StyleBox["cons", "TI"]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["x", "TI"], ",", 
          StyleBox["y", "TI"], ",", 
          StyleBox["\[Ellipsis]", "TR"]}], "}"}]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]starts from a point within the region defined by the \
constraints."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, {None}}, "RowsIndexed" -> {}}},
 CellID->12439]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FindMaximum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindMaximum"]], "InlineFormula"],
 " returns a list of the form ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    SubscriptBox[
     StyleBox["f", "TI"], 
     StyleBox["max", "TI"]], ",", 
    RowBox[{"{", 
     RowBox[{
      StyleBox["x", "TI"], "->", 
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["max", "TI"]]}], "}"}]}], "}"}]], "InlineFormula"],
 ", where ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["f", "TI"], 
   StyleBox["max", "TI"]]], "InlineFormula"],
 " is the maximum value of ",
 Cell[BoxData[
  StyleBox["f", "TI"]], "InlineFormula"],
 " found, and ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["x", "TI"], 
   StyleBox["max", "TI"]]], "InlineFormula"],
 " is the value of ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " for which it is found. "
}], "Notes",
 CellID->17026],

Cell["\<\
If the starting point for a variable is given as a list, the values of the \
variable are taken to be lists with the same dimensions. \
\>", "Notes",
 CellID->19890],

Cell[TextData[{
 "The constraints ",
 Cell[BoxData[
  StyleBox["cons", "TI"]], "InlineFormula"],
 " can contain equations, inequalities or logical combinations of these. "
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->19298],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FindMaximum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindMaximum"]], "InlineFormula"],
 " first localizes the values of all variables, then evaluates ",
 Cell[BoxData[
  StyleBox["f", "TI"]], "InlineFormula"],
 " with the variables being symbolic, and then repeatedly evaluates the \
result numerically."
}], "Notes",
 CellID->1915],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FindMaximum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindMaximum"]], "InlineFormula"],
 " has attribute ",
 Cell[BoxData[
  ButtonBox["HoldAll",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HoldAll"]], "InlineFormula"],
 ", and effectively uses ",
 Cell[BoxData[
  ButtonBox["Block",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Block"]], "InlineFormula"],
 " to localize variables."
}], "Notes",
 CellID->11648],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["FindMaximum",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FindMaximum"], "[", 
   RowBox[{
    StyleBox["f", "TI"], ",", 
    RowBox[{"{", 
     RowBox[{
      StyleBox["x", "TI"], ",", 
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["0", "TR"]], ",", 
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["1", "TR"]]}], "}"}]}], "]"}]], "InlineFormula", 
  "TemplateInclusion"],
 " searches for a local maximum in ",
 Cell[BoxData[
  StyleBox["f", "TI"]], "InlineFormula"],
 " using ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["x", "TI"], 
   StyleBox["0", "TR"]]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["x", "TI"], 
   StyleBox["1", "TR"]]], "InlineFormula"],
 " as the first two values of ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 ", avoiding the use of derivatives. "
}], "Notes",
 CellID->20737],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["FindMaximum",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FindMaximum"], "[", 
   RowBox[{
    StyleBox["f", "TI"], ",", 
    RowBox[{"{", 
     RowBox[{
      StyleBox["x", "TI"], ",", 
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["0", "TR"]], ",", 
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["min", "TI"]], ",", 
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["max", "TI"]]}], "}"}]}], "]"}]], "InlineFormula", 
  "TemplateInclusion"],
 " searches for a local maximum, stopping the search if ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " ever gets outside the range ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["x", "TI"], 
   StyleBox["min", "TI"]]], "InlineFormula"],
 " to ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["x", "TI"], 
   StyleBox["max", "TI"]]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->14015],

Cell[TextData[{
 "Except when ",
 Cell[BoxData[
  StyleBox["f", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["cons", "TI"]], "InlineFormula"],
 " are both linear, the results found by ",
 Cell[BoxData[
  ButtonBox["FindMaximum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindMaximum"]], "InlineFormula"],
 " may correspond only to local, but not global, maxima. "
}], "Notes",
 CellID->8689],

Cell["By default, all variables are assumed to be real. ", "Notes",
 CellID->23646],

Cell[TextData[{
 "For linear ",
 Cell[BoxData[
  StyleBox["f", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["cons", "TI"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  RowBox[{
   StyleBox["x", "TI"], "\[Element]", 
   ButtonBox["Integers",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Integers"]}]], "InlineFormula"],
 " can be used to specify that a variable can take on only integer values. "
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->21374],

Cell["The following options can be given: ", "Notes",
 CellID->9099],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["AccuracyGoal",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/AccuracyGoal"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "the accuracy sought ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["EvaluationMonitor",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/EvaluationMonitor"], 
    ButtonBox["None",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/None"], Cell[TextData[{
     "expression to evaluate whenever ",
     Cell[BoxData[
      StyleBox["f", "TI"]], "InlineFormula"],
     " is evaluated "
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "Gradient", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[TextData[{
     "the list of gradient functions ",
     Cell[BoxData[
      RowBox[{"{", 
       RowBox[{
        RowBox[{"D", "[", 
         RowBox[{
          StyleBox["f", "TI"], ",", 
          StyleBox["x", "TI"]}], "]"}], ",", 
        RowBox[{"D", "[", 
         RowBox[{
          StyleBox["f", "TI"], ",", 
          StyleBox["y", "TI"]}], "]"}], ",", 
        StyleBox["\[Ellipsis]", "TR"]}], "}"}]], "InlineFormula"]
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["MaxIterations",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/MaxIterations"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "maximum number of iterations to use ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["Method",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Method"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "method to use ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["PrecisionGoal",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/PrecisionGoal"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "the precision sought ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["StepMonitor",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/StepMonitor"], 
    ButtonBox["None",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/None"], Cell[
    "expression to evaluate whenever a step is taken ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["WorkingPrecision",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/WorkingPrecision"], 
    ButtonBox["MachinePrecision",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/MachinePrecision"], Cell[
    "the precision used in internal computations ", "TableText"]}
  },
  GridBoxAlignment->{
   "Columns" -> {Left, Left, {Left}}, "ColumnsIndexed" -> {}, 
    "Rows" -> {{Baseline}}, "RowsIndexed" -> {}}]], "3ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, None, None, None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{True, True, True, True, True, True, True, True, True}}}},
 CellID->19426],

Cell[TextData[{
 "The default settings for ",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " are ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["WorkingPrecision",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/WorkingPrecision"], "/", "2"}]], 
  "InlineFormula"],
 ". "
}], "Notes",
 CellID->10073],

Cell[TextData[{
 "The settings for ",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " specify the number of digits to seek in both the value of the position of \
the maximum, and the value of the function at the maximum. "
}], "Notes",
 CellID->17325],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FindMaximum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindMaximum"]], "InlineFormula"],
 " continues until either of the goals specified by ",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 " or ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " is achieved. "
}], "Notes",
 CellID->7673],

Cell[TextData[{
 "Possible settings for ",
 Cell[BoxData[
  ButtonBox["Method",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Method"]], "InlineFormula"],
 " include ",
 Cell[BoxData["\"\<ConjugateGradient\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<Gradient\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<LevenbergMarquardt\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<Newton\>\""], "InlineFormula"],
 " and ",
 Cell[BoxData["\"\<QuasiNewton\>\""], "InlineFormula"],
 ", with the default being ",
 Cell[BoxData[
  ButtonBox["Automatic",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->3822]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->672757868],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(3)", "ExampleCount"]
}], "ExampleSection",
 CellID->289512076],

Cell[TextData[{
 "Find a local maximum starting the search at ",
 Cell[BoxData[
  RowBox[{
   StyleBox["x", "TI"], "=", "2"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->499988759],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMaximum", "[", 
  RowBox[{
   RowBox[{"x", " ", 
    RowBox[{"Cos", "[", "x", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->376441841],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.5610963381910451`", ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", "0.8603335884296213`"}], "}"}]}], "}"}]], "Output",\

 ImageSize->{179, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->648154644]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"x", " ", 
    RowBox[{"Cos", "[", "x", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "14"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->347120420],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 104},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->485044888]
}, Open  ]],

Cell[TextData[{
 "Extract the value of ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " at the local maximum:"
}], "ExampleText",
 CellID->452807755],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"x", "/.", 
  RowBox[{"Last", "[", 
   RowBox[{"FindMaximum", "[", 
    RowBox[{
     RowBox[{"x", " ", 
      RowBox[{"Cos", "[", "x", "]"}]}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "2"}], "}"}]}], "]"}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->28565156],

Cell[BoxData["0.8603335884296213`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->156355731]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->986451359],

Cell[TextData[{
 "Find a local maximum, starting at ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "=", "7"}], TraditionalForm]], "InlineMath"],
 ", subject to constraints ",
 Cell[BoxData[
  FormBox[
   RowBox[{"1", "\[LessEqual]", " ", "x", " ", "\[LessEqual]", " ", "15"}], 
   TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->1560461396],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMaximum", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", " ", 
      RowBox[{"Cos", "[", "x", "]"}]}], ",", 
     RowBox[{"1", "\[LessEqual]", " ", "x", "\[LessEqual]", " ", "15"}]}], 
    "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "7"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->678564151],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"6.361003944833853`", ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", "6.437298189878469`"}], "}"}]}], "}"}]], "Output",\

 ImageSize->{144, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->219625470]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1519398755],

Cell["\<\
Find the maximum of a linear function, subject to linear and integer \
constraints:\
\>", "ExampleText",
 CellID->1896901598],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMaximum", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"-", "x"}], "-", "y"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "+", 
        RowBox[{"2", "y"}]}], "\[GreaterEqual]", "3"}], "&&", 
      RowBox[{"x", "\[GreaterEqual]", "0"}], "&&", 
      RowBox[{"y", "\[GreaterEqual]", "0"}], "&&", 
      RowBox[{"y", "\[Element]", "Integers"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1202084729],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "2.`"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "1.`"}], ",", 
     RowBox[{"y", "\[Rule]", "1"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{144, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->50585186]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(6)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->13645],

Cell["With different starting points get different local maxima:", \
"ExampleText",
 CellID->62007641],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMaximum", "[", 
  RowBox[{
   RowBox[{"x", " ", 
    RowBox[{"Cos", "[", "x", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->66723803],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"6.361003944833853`", ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", "6.437298179138412`"}], "}"}]}], "}"}]], "Output",\

 ImageSize->{144, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->10453814]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMaximum", "[", 
  RowBox[{
   RowBox[{"x", " ", 
    RowBox[{"Cos", "[", "x", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->386171924],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"12.605931297892674`", ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", "12.645287223190522`"}], "}"}]}], "}"}]], "Output",\

 ImageSize->{165, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->174136289]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->404322872],

Cell[TextData[{
 "Local maximum of a 2-variable function starting from ",
 Cell[BoxData[
  RowBox[{
   StyleBox["x", "TI"], "=", "2"}]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  RowBox[{
   StyleBox["y", "TI"], "=", "2"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->516388244],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMaximum", "[", 
  RowBox[{
   RowBox[{"Sin", "[", 
    RowBox[{"x", " ", "y"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"x", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", "2"}], "}"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->46634681],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "1.2533141373152121`"}], ",", 
     RowBox[{"y", "\[Rule]", "1.2533141373152121`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{213, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->128600046]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->242324886],

Cell["Local maximum constrained within a disk:", "ExampleText",
 CellID->681774368],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMaximum", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Sin", "[", 
      RowBox[{"x", " ", "y"}], "]"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "^", "2"}], "+", 
       RowBox[{"y", "^", "2"}]}], "<", "3"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"x", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", "2"}], "}"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->852578689],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.9974949866040596`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "1.2247448713916185`"}], ",", 
     RowBox[{"y", "\[Rule]", "1.2247448713916185`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{255, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->128327230]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1308882988],

Cell["Starting point does not have to be provided:", "ExampleText",
 CellID->374615681],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMaximum", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Sin", "[", 
      RowBox[{"x", " ", "y"}], "]"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "^", "2"}], "+", 
       RowBox[{"y", "^", "2"}]}], "<", "3"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1153958179],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.997494642394625`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "1.2247428849092166`"}], ",", 
     RowBox[{"y", "\[Rule]", "1.2247428849092166`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{255, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->123814084]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->685666782],

Cell["\<\
For linear objective and constraints, integer constraints can be imposed:\
\>", "ExampleText",
 CellID->169051774],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMaximum", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"-", "x"}], "-", "y"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"3", "x"}], "+", 
        RowBox[{"2", "y"}]}], "\[GreaterEqual]", "7"}], "&&", 
      RowBox[{"x", "\[GreaterEqual]", "0"}], "&&", 
      RowBox[{"y", "\[GreaterEqual]", "0"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1991020309],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "2.3333333333333335`"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "2.3333333333333335`"}], ",", 
     RowBox[{"y", "\[Rule]", "0.`"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{221, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->624608140]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMaximum", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"-", "x"}], "-", "y"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"3", "x"}], "+", 
        RowBox[{"2", "y"}]}], "\[GreaterEqual]", "7"}], "&&", 
      RowBox[{"x", "\[GreaterEqual]", "0"}], "&&", 
      RowBox[{"y", "\[GreaterEqual]", "0"}], "&&", 
      RowBox[{"x", "\[Element]", "Integers"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->833915335],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "2.5`"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "2"}], ",", 
     RowBox[{"y", "\[Rule]", "0.5`"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{158, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->17622987]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->5114145],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Or",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Or"]], "InlineFormula"],
 " constraints can be specified:"
}], "ExampleText",
 CellID->284316099],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMaximum", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "+", "y"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"x", "^", "2"}], "+", 
        RowBox[{"y", "^", "2"}]}], "\[LessEqual]", "1"}], "||", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"(", 
          RowBox[{"x", "+", "2"}], ")"}], "^", "2"}], "+", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{"y", "+", "2"}], ")"}], "^", "2"}]}], "\[LessEqual]", 
       "1"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1963030529],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.4142121471429892`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "0.7071060735714946`"}], ",", 
     RowBox[{"y", "\[Rule]", "0.7071060735714946`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{262, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->620671711]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Options",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->6246],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "AccuracyGoal & PrecisionGoal",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->25956],

Cell[TextData[{
 "This enforces a convergence criteria ",
 Cell[BoxData[
  FormBox[
   RowBox[{"||", 
    RowBox[{
     SubscriptBox["x", "k"], "-", 
     SuperscriptBox["x", "*"]}], "||", " ", 
    RowBox[{"\[LessEqual]", " ", 
     RowBox[{"max", "(", 
      RowBox[{
       SuperscriptBox["10", 
        RowBox[{"-", "9"}]], ",", 
       RowBox[{
        SuperscriptBox["10", 
         RowBox[{"-", "8"}]], "||", 
        SubscriptBox["x", "k"], "||"}]}], ")"}]}]}], TraditionalForm]], 
  "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"\[EmptyDownTriangle]f", "(", 
     SubscriptBox["x", "k"], ")"}], "\[LessEqual]", 
    SuperscriptBox["10", 
     RowBox[{"-", "9"}]]}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->764461149],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMaximum", "[", 
  RowBox[{
   RowBox[{"Sin", "[", 
    RowBox[{"x", "/", "2"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "1"}], "}"}], ",", 
   RowBox[{"AccuracyGoal", "\[Rule]", "9"}], ",", 
   RowBox[{"PrecisionGoal", "\[Rule]", "8"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1625295724],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.`", ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", "3.1415926535940666`"}], "}"}]}], "}"}]], "Output",\

 ImageSize->{130, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->494162563]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1055000278],

Cell[TextData[{
 "This enforces a convergence criteria ",
 Cell[BoxData[
  FormBox[
   RowBox[{"||", 
    RowBox[{
     SubscriptBox["x", "k"], "-", 
     SuperscriptBox["x", "*"]}], "||", " ", 
    RowBox[{"\[LessEqual]", " ", 
     RowBox[{"max", "(", 
      RowBox[{
       SuperscriptBox["10", 
        RowBox[{"-", "20"}]], ",", 
       RowBox[{
        SuperscriptBox["10", 
         RowBox[{"-", "18"}]], "||", 
        SubscriptBox["x", "k"], "||"}]}], ")"}]}]}], TraditionalForm]], 
  "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"\[EmptyDownTriangle]f", "(", 
     SubscriptBox["x", "k"], ")"}], "\[LessEqual]", 
    SuperscriptBox["10", 
     RowBox[{"-", "20"}]]}], TraditionalForm]], "InlineMath"],
 ", which is not achievable with the default machine precision computation:"
}], "ExampleText",
 CellID->1329553122],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMaximum", "[", 
  RowBox[{
   RowBox[{"Sin", "[", 
    RowBox[{"x", "/", "2"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "1"}], "}"}], ",", 
   RowBox[{"AccuracyGoal", "\[Rule]", "20"}], ",", 
   RowBox[{"PrecisionGoal", "\[Rule]", "18"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1468387723],

Cell[BoxData[
 RowBox[{
  RowBox[{"FindMaximum", "::", "\<\"lstol\"\>"}], ":", 
  " ", "\<\"The line search decreased the step size to within tolerance \
specified by AccuracyGoal and PrecisionGoal but was unable to find a \
sufficient increase in the function.  You may need more than \
\\!\\(MachinePrecision\\) digits of working precision to meet these \
tolerances.\"\>"}]], "Message", "MSG",
 CellID->7355808],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.`", ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", "3.141592653594063`"}], "}"}]}], "}"}]], "Output",\

 ImageSize->{130, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->787252468]
}, Open  ]],

Cell[TextData[{
 "Setting a high ",
 Cell[BoxData[
  ButtonBox["WorkingPrecision",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/WorkingPrecision"]], "InlineFormula"],
 " makes the process convergent:"
}], "ExampleText",
 CellID->1199314865],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMaximum", "[", 
  RowBox[{
   RowBox[{"Sin", "[", 
    RowBox[{"x", "/", "2"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "1"}], "}"}], ",", 
   RowBox[{"AccuracyGoal", "\[Rule]", "20"}], ",", 
   RowBox[{"PrecisionGoal", "\[Rule]", "18"}], ",", 
   RowBox[{"WorkingPrecision", "\[Rule]", "40"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->39134653],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.9999999999999999999999999999999999999999999999997555881504`40.", ",", 
   RowBox[{"{", 
    RowBox[{
    "x", "\[Rule]", 
     "3.1415926535897932384626419849609423925982679573717249489545`40."}], 
    "}"}]}], "}"}]], "Output",
 ImageSize->{342, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->647176890]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "EvaluationMonitor",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->23123],

Cell["Plot convergence to the local maximum:", "ExampleText",
 CellID->28034064],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{"Last", "[", 
   RowBox[{"Reap", "[", 
    RowBox[{"FindMaximum", "[", 
     RowBox[{
      RowBox[{"x", " ", 
       RowBox[{"Cos", "[", "x", "]"}]}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", "5"}], "}"}], ",", 
      RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
       RowBox[{"Sow", "[", "x", "]"}]}]}], "]"}], "]"}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19607354],

Cell[BoxData[
 GraphicsBox[{{}, 
   {Hue[0.67, 0.6, 0.6], 
    PointBox[{{1., 5.}, {2., 7.5}, {3., 6.07895713587439}, {4., 
     6.9116884229275435`}, {5., 6.4311438487116765`}, {6., 
     6.437794605287185}, {7., 6.437297723022912}, {8., 
     6.437298179138412}}]}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 5.},
  ImageMargins->0.,
  ImageSize->Small,
  PlotRange->{{0, 8.}, {5., 7.5}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 121},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->68827344]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Gradient",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->12053],

Cell["\<\
Use the given gradient and the Hessian is computed using finite differences \
of the gradient:\
\>", "ExampleText",
 CellID->858592029],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMaximum", "[", 
  RowBox[{
   RowBox[{"-", 
    RowBox[{"Sin", "[", 
     RowBox[{"x", " ", "y"}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}], ",", 
   RowBox[{"Gradient", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"y", " ", 
       RowBox[{"Cos", "[", 
        RowBox[{"x", " ", "y"}], "]"}]}], ",", " ", 
      RowBox[{"x", " ", 
       RowBox[{"Cos", "[", 
        RowBox[{"x", " ", "y"}], "]"}]}]}], "}"}]}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<Newton\>\""}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2076542040],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "1.60711446908485`"}]}], ",", 
     RowBox[{"y", "\[Rule]", "0.9774016456312932`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{228, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->26026175]
}, Open  ]],

Cell["This uses supplied gradient and Hessian:", "ExampleText",
 CellID->1859244329],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMaximum", "[", 
  RowBox[{
   RowBox[{"-", 
    RowBox[{"Sin", "[", 
     RowBox[{"x", " ", "y"}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"x", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", "1"}], "}"}]}], "}"}], ",", 
   RowBox[{"Gradient", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"y", " ", 
       RowBox[{"Cos", "[", 
        RowBox[{"x", " ", "y"}], "]"}]}], ",", " ", 
      RowBox[{"x", " ", 
       RowBox[{"Cos", "[", 
        RowBox[{"x", " ", "y"}], "]"}]}]}], "}"}]}], ",", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<Newton\>\"", ",", 
      RowBox[{"Hessian", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{
            RowBox[{"-", 
             RowBox[{"y", "^", "2"}]}], 
            RowBox[{"Sin", "[", 
             RowBox[{"x", " ", "y"}], "]"}]}], ",", 
           RowBox[{
            RowBox[{"-", " ", "x"}], " ", "y", " ", 
            RowBox[{"Cos", "[", 
             RowBox[{"x", " ", "y"}], "]"}]}]}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{
            RowBox[{"-", "x"}], " ", "y", " ", 
            RowBox[{"Cos", "[", 
             RowBox[{"x", " ", "y"}], "]"}]}], ",", " ", 
           RowBox[{
            RowBox[{"-", 
             RowBox[{"x", "^", "2"}]}], 
            RowBox[{"Sin", "[", 
             RowBox[{"x", " ", "y"}], "]"}]}]}], "}"}]}], "}"}]}]}], 
     "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1427703802],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "8.118188590213629`"}]}], ",", 
     RowBox[{"y", "\[Rule]", "11.028986285836702`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{214, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->128052500]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Method",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->16079],

Cell["\<\
In this case the default derivative based methods have difficulties:\
\>", "ExampleText",
 CellID->993924491],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMaximum", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"-", 
     RowBox[{"Abs", "[", 
      RowBox[{"x", "+", "1"}], "]"}]}], "-", 
    RowBox[{"Abs", "[", 
     RowBox[{"x", "+", "1.01"}], "]"}], "-", 
    RowBox[{"Abs", "[", 
     RowBox[{"y", "+", "1"}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->782571855],

Cell[BoxData[
 RowBox[{
  RowBox[{"FindMaximum", "::", "\<\"lstol\"\>"}], ":", 
  " ", "\<\"The line search decreased the step size to within tolerance \
specified by AccuracyGoal and PrecisionGoal but was unable to find a \
sufficient increase in the function.  You may need more than \
\\!\\(MachinePrecision\\) digits of working precision to meet these \
tolerances.\"\>"}]], "Message", "MSG",
 CellID->322138632],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "0.01000000452670391`"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "1.0068264489265601`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "1.000000004526704`"}]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{215, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->51247189]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMaximum", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"-", 
     RowBox[{"Abs", "[", 
      RowBox[{"x", "+", "1"}], "]"}]}], "-", 
    RowBox[{"Abs", "[", 
     RowBox[{"x", "+", "1.01"}], "]"}], "-", 
    RowBox[{"Abs", "[", 
     RowBox[{"y", "+", "1"}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<ConjugateGradient\>\""}]}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->939473422],

Cell[BoxData[
 RowBox[{
  RowBox[{"FindMaximum", "::", "\<\"sdprec\"\>"}], ":", 
  " ", "\<\"Line search unable to find a sufficient increase in the function \
value with \\!\\(MachinePrecision\\) digit precision.\"\>"}]], "Message", \
"MSG",
 CellID->49025433],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "0.7373153897224387`"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "0.6363423052199573`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "1.0000000001623532`"}]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{250, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->17329102]
}, Open  ]],

Cell["\<\
Direct search methods which do not require derivatives can be helpful in \
these cases:\
\>", "ExampleText",
 CellID->1614271486],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMaximum", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"-", 
     RowBox[{"Abs", "[", 
      RowBox[{"x", "+", "1"}], "]"}]}], "-", 
    RowBox[{"Abs", "[", 
     RowBox[{"x", "+", "1.01"}], "]"}], "-", 
    RowBox[{"Abs", "[", 
     RowBox[{"y", "+", "1"}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<PrincipalAxis\>\""}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->736570868],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "0.010000000000000009`"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "1.0067107133234492`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "1.`"}]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{215, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->456027668]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NMaximize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NMaximize"]], "InlineFormula"],
 " also uses a range of direct search methods:"
}], "ExampleText",
 CellID->1954260595],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMaximize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", 
      RowBox[{"Abs", "[", 
       RowBox[{"x", "+", "1"}], "]"}]}], "-", 
     RowBox[{"Abs", "[", 
      RowBox[{"x", "+", "1.01"}], "]"}], "-", 
     RowBox[{"Abs", "[", 
      RowBox[{"y", "+", "1"}], "]"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->46889232],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "0.010000000000001452`"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "1.0073451755251817`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "0.9999999999999986`"}]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{215, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->148029175]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "StepMonitor",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->2025],

Cell[TextData[{
 "Steps taken by ",
 Cell[BoxData[
  ButtonBox["FindMaximum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindMaximum"]], "InlineFormula"],
 " in finding the maximum a function:"
}], "ExampleText",
 CellID->1691254454],

Cell[BoxData[{
 RowBox[{
  RowBox[{"pts", " ", "=", " ", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"FindMaximum", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"-", 
         RowBox[{
          RowBox[{"(", 
           RowBox[{"1", "-", "x"}], ")"}], "^", "2"}]}], "-", 
        RowBox[{"100", 
         RowBox[{
          RowBox[{"(", 
           RowBox[{
            RowBox[{"-", 
             RowBox[{"x", "^", "2"}]}], "-", "y"}], ")"}], "^", "2"}]}]}], 
       ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"x", ",", 
           RowBox[{"-", "1.2"}]}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"y", ",", "1"}], "}"}]}], "}"}], ",", 
       RowBox[{"StepMonitor", "\[RuleDelayed]", 
        RowBox[{"Sow", "[", 
         RowBox[{"{", 
          RowBox[{"x", ",", "y"}], "}"}], "]"}]}]}], "]"}], "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "1"}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"pts", " ", "=", 
   RowBox[{"Join", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1.2"}], ",", "1"}], "}"}], "}"}], ",", "pts"}], "]"}]}],
   ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->1514338379],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ContourPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"-", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"1", "-", "x"}], ")"}], "^", "2"}]}], "-", 
    RowBox[{"100", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", 
         RowBox[{"x", "^", "2"}]}], "-", "y"}], ")"}], "^", "2"}]}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1.3"}], ",", "1.5"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "1.5"}], ",", "1.4"}], "}"}], ",", 
   RowBox[{"Epilog", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"Arrow", "[", "pts", "]"}], ",", 
      RowBox[{"Point", "[", "pts", "]"}]}], "}"}]}], ",", 
   RowBox[{"Contours", "\[Rule]", 
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{"-", 
       RowBox[{"10", "^", 
        RowBox[{"-", "i"}]}]}], ",", 
      RowBox[{"{", 
       RowBox[{"i", ",", 
        RowBox[{"-", "2"}], ",", "10"}], "}"}]}], "]"}]}], ",", 
   RowBox[{"ColorFunction", "\[Rule]", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"Hue", "[", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          RowBox[{"Log", "[", 
           RowBox[{"10", ",", 
            RowBox[{"-", "#"}]}], "]"}], "+", "10"}], ")"}], "/", "12"}], 
       "]"}], "&"}], ")"}]}], ",", 
   RowBox[{"ColorFunctionScaling", "\[Rule]", "False"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1350533918],

Cell[BoxData[
 GraphicsBox[GraphicsComplexBox[CompressedData["
1:eJxknQeUVTUTxwHpUpUiKlWQJqCgiCKGLiD4oQKCgNIFpKoIiNKki4BIkd6k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   "], {{
     {Hue[1.0462076885174267`], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmot31dWVx8+V3Jub5L5ycwNoWyUBdaZrzT8wTqc6jh0HnE5bC1Spy7Vq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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwllulv1UUUhqe0tyv3d2/vvbghsrrxzT/ABUGCZW8pi8aYiKCggiAllrKD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         "],
         VertexColors->None]}]}, 
     {Hue[0.9783635574578536], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNnXe4VFXytRv6dvcNHW83jjIqg4wKoqOCAQMZRUAUEEExB7IgJgyAqKAY
UAYTgjlgQBhFUTHMCI5xzKMgSRETJkwgKOj8vvXeVTx+f5zn7N5n51C7alXV
7uanntlvdONEIrF3TSKR1HuPTCLRNZ1I9EolEguyicSzjRKJDno+0HOjnjuV
ZrneM/TMIZ/S7qmMZyjcvEki0TmfSNysvM/pe0c9+yq+k9776X2Ingv1ba3e
++sZlUskntG39npe0NNHTzfFv6Uy/1uVSIzVs5PK7KIyZyrf+Qq/qPBwtfUo
tW1H1dtG6b8pq9xKIjFa8QcprrWeVnouU56+jVxvC+V9v159qU4kpijduYr/
n+KPKSUSnyh+N/2+VPHH6v2j4s/TOOyhMo5XeJriG+l5VXlXqE0vKc1J0WbK
P0ppJqVc3zr6prruKao+/d5N4fPU5qbKd4HKPFh9G6c0a1RWt1r3c1v1ZZzK
f1MT0VnfL1PceKUZorI38dazQuFb9Dys8GS1d7WeuWrfNyrjaz1H6dtiPUfq
6ao0i/Q+Qk8XhRfUJRKP65mp9OPUnqPVtmFq22yNfx/VfZTCFyh8rMKHKM1d
ep5S3j31nK+4W5W3rdrUWmlWqrymer7NuE0z9ZytMleozJFq+2CVdZ3SXqXn
IqW/UM9ita9K5ST1PK1+D9BzjOLWK80Ypb9cZaT0vK/8hyt+guKn6HdazyTV
9YTiFuhpo9/3qC1vK/xXhW8tJBKNVfe5St9Wv9cobSs9FyjuirLnbLzCx6pt
I1TPHI3vAUr3mdI0Vh8b6emg8DYq85+K/4a8auMd+n2Aynxc+b9Tvr2VboPi
1+s5SfGv6f2qnmVKf6/W3ccFr4s31P6z9T5Tz7Xq93bKp2YkKip7id7KkjhA
cbvqqVHcoWrX50rQI+X91lPPLvr2H31L6jlP6e9S+UtU/msqs7+ef+n7UvVj
qcZgiZ6FjbzXOsXeeTb2HWvgmdi7rIG1sf/YhwdpTOZqTa5WfWfkPNd76XlS
4Sf0DNUzpsptZq5pL+3Skkt8orbcrzb1Ut09lO6ilPf07loPr6fdh561/r5e
6Xup0/c19hh0Vpk3qZzZ+v284nfRs5L9T3s0Fl8qbzuVsb+elnqGKf3A2Msf
ZTx37FPWRw+V30/hy9WW+9SmixW3g9qzvZ6WKne4fl8etIA11TPWFWtwTdCf
D1Xf3MZeOwerDQ+qDR+nTFugSZ2VZmbOa6Wnfo/Uehih55Qqr0fWMuO5UuEV
/Faaq5T3yqzbt0Xtul7zu1np26mM/fWsV33dVFdR3/6r3+8ozzV6blM5V7Om
9b424X3DHpqq8Au13kPXKHy3+nuPni76/a7ST9Vzu+KHas0uDRq9j/K+kjSd
XqLnhqAb42NdsafWpL0nblIdm9SPjdXet6ernOuU5ko90/U8lHCb3lf4+qD5
1+qd0XOHwn9VOzapnMmN3FbSztKzOm0ac4PK7Kr+5tXfd5Men56xNnbPmaas
UrhG31tqgb1Bu9WGlipvteLTKiel5wyVNUNpH9b4tda3ASmPwWux19hzOyVM
C6BRLfV8njEdmdjINAZ6Ml3x76mvi6q8VxfHudNP8fNU93FJ083dVGaroImL
gq72VZotdZ6PqVXu29ToL/2eGnN3u/rysfpyvr63VxvGpbxP7le9rZX3PsVP
U7njFV6e8P48ItYbtLwztIFzttb7mjO0W5yjh8Re7hzhDaproZ4ZKvNqrYun
I3yY2tldz780Xh9pzU5U/U8q/H6t6cM+yruvxrWVnt/1+019f0PPoQofqzb/
oPD3KdORI4OG7KW0X3C2q5xeOe+/redr36A/vPvFmdhR5UxO+Wz8TvWu09Oq
sdP1CRrFXu0T6R9X269SH3Zhrev9lH5/kvZ5zLncQ++/pR0+RuHjU24bc8c5
OSjiL1WZOzNH0G29T0g5LXG7RPwojdEcvQ/W85D6Mivn8AXwQZyrKqtJI69N
aGCLRl5XnC8PKO9Zsc72j/nYL+aEMHNXqzVfEY19WumOVNlzVf8mhesU30Tx
zyn8H7WzXuPxZ+Ze3xsrbrN+z1cbt4/130F52+vZR79/13icIn5li9LtobgN
ittZT3+Fj8o5zzqN23yl+1Dj1CN4GcZtz7THjjGaqzQ/K833KuecpNNx9g3J
uIwdFX604vqo60S18/Y6j03LKJcyN6sf54q+PaQyLuLMjrKgs6yB/vAJrKlq
n5usrwOjLujjK9VOB208vdb0kTFcF2cVPAu8y/0akw+1nz6qd1m/q20/ilZP
Vlsu1XuwxuTRWKsdYx2yXjvEGlsZ8wdN6JI2jwUfNkRzMQi+JWU+dqLaeELC
+4Q19oPCLwcPxro6RemurDJ/9qPG5gG142uFlynvD5w3eqrqzFv+nvA+Pyzp
c7Sb3l2T5pXHqN5G6stFGu+HVcbGmIvBih+g9gxPubyBteZRT1P8IYrvpviz
FG6svOOUt33SvAl86mTNzaQa86bXBX2G1m2n9vw36Hkl7fOGcwa6D82Cvi8J
2n570PnpcR4tjbMDOt8keJg/69mY8tnDufV2Y58rjaGZWiet9KxT+Ey181fN
yxi185eUzwzOqirN3zL9fkJt7qc0swreS33Utt61PoOOUvwDir9E4dnMv54a
1bOP8rTV87HCczRuPbT+vlWa/9Z6rtuqbY/Vev9wfryr9zt6tlH6YWrXvSpn
RtLnP+uBvbxC7Vmu5wn9nqcyV6jMKzSOZyj9ID1HKP0n8AxZ88NLq81HtlL4
DNV9m74fmPRZtnOU+aDCt+Qcv1T5+urZQeHXkWH09NfTRm3ZonK+Vp63NEaL
FddHaeYr7cZGbj9837kln3EH6fu4Wu+V9kpzcDZ43uBVkIOeVPiJrM+Ns1P+
TnuQo6Af7RLuO2HoFmfQdVmXy9nTPug8/DD79CDOylqPL/xhTmk/qPUZcnGt
zy/a1UJj9kLe/DljCZ+2u/K2TjtMOXsovEZp56nPu6f9jTrgO2aqnDv17THR
kvl62tQE/9vIPDByEeFzE94P7C9kot8jTJoPgv/irOyr/p6a9Nr8pmg+Gx57
CHRP5Y9X+TdqzEtJr6vXg0eAL4JHmBDh3sq3Pu9y26bNd/PtNJUzpmAZ4Rm1
f4da05aTFD+iYFmJ9o+PNjOuz8TYZrV3pyrNIKXpojXSOWe6tw8ylp7Dk+Zb
OX+QTQ5UmcPU5mXVpn2VWtPSpjnLX/B1uyKPJM1Ld4i62AuPKv6RrOnsmSrn
2qh3L4VvUjuSin9EcSsr5mda6/2y+vtPjc9DasscPWvU51lVXpucNweo3nY5
y2GNo4+M1Vkpy3Lj9LuRyrhUZe2q+Bmq/+as5/hZvZ/Jmkaer+9flc1TP19r
2jm8kdfUhOCLvtSYf6P4u/VUlKdcZ9whr7n7Wt9+Qu5TW9rkLBvvnDW9mpsw
zbohaFpF9UypmHZfrH4MTZpPHqLyblFZL+v3CIUfUVuGJY1x3Bh0rxNyfMVt
WqL3Uj0TFR6j90PqZ/uU+V/4YGTYXtX+znle1dgyPGv1H4pvpvjh4Ctqz8SK
+c/bwQEUtzllnvG2gvnG5UnL69CCS5T29LL5VngY6oLfbpnxPmQPckYNj7pm
FywXIRNNUniV8h6h8FKlW5Jymybpe7eSsZa+eprWWub7c85ri7F8XM9jWdO8
hRqHE/SMTpnuvxtj+1vWZ8p1Ci8O2ZX9+LDGck6Nf7+osqdXeZ9tzjrvtJif
pTFfN0aYMQdHYJ7AnFZpTM6reN1zLnE+IYOwv16PtfdnzV1TPS8lfY4tiX1x
Q4Rp5/WRl3Pt5Sibee6uMbxbfbor5bPyLc5Cxf9FbWgOFhNrmzDrey+l75sx
LX9bT7+Meee0+nee8h8BfU07P3nJQxj5g6dZ2jTmI4396pLz8/utOK8PhrfR
unqvxuf0O9Hm+SnPZ/8oH3pEWcfVud63kj4v4WHA4m7QvI+tWFZdmDQfAw+z
rdL+qc48225pywXw/3trnB/XM1vtyarueSljU81Uxg56Hq3x/oDmLFQ5O9QZ
e/o//d5W38drjW2A9iv/1Yo/V/k/T/qcZx+MU3veUPn3wnsr/VX6PTD27/KY
U35DdwbovRpcpeKyGmhGY6/tB+ucBroEfgQNID8yBHjcUSFbINd1DR7w2Qgj
Pz0XZ0THiN8nwsTDNy5Tncv1XKZ6u2u9f6DwpQoPS5tXBifaVXHbFLxHn9Jc
XaTfzZKWRRYFnT8y2tAvZKIXonxkMtIgS4KXUE9fjdU9Gtu7azxPYBgvRd47
9O3ElGWY4co7K2Naj/wLXwB2k836nAYLWqt5eEzrqrXS/aZ2bdEzT98GKs0S
5R+VMP/7fp35WOLej/hz4MXiXHte73V58043JP2G90upvCMV/47iHtC7hdJ9
pG+X6T1S32pU13dqwzo9Z6k/A2ocNz9pPg5+jrO5ndr3Tcp76wLlW6NnPzA5
vd9Rud/p+wUqc6h+pzgfavw+EZ5H4dU1iQbAuhreLXAxMBmwwl2DD3xczw4K
/1zn34TJgzwMjz9F8V/U+jdhyuzO/KneLqw//f62bHwZbHms4gdDt5Xme9U7
rdby7Y8VyyHIIPDoa6JtlPdxlDlPZczVs1uNx+GB4GnHq8xhyp8Go6rxezZ7
hPEMXiKnsZih+Fkqe5bet9aaXjGOjCc8xvcq4wc9t+nbN3p/W3Ge88RLj9V4
dlFd75RNd6A5X8V8UAZn9Kwok/ZBa+jXCn1fWWfe+GuVd27RbUjnvG4OSBnb
QKZH5n1J7f1f3r8vVV3HlXyGn6O4hL4dqPiTwVAD7y6qHb+qf7/o+VXp7xB9
qNG4fqS6Tiqahmzh7Cqbz/oya/wQmTmn/NfCg6ZML6ALyAXwv6wD5HHauDTa
CV8NrgRfWp1z3+jXmUmHkbUZp0EVjxVxyyO+FOPMfN2gtpyqdBXVP6TW441s
Dxa4INbeF+rH53oOV9wtes+ot9yDDHZd0PlMhO+MMPLR3xPmg6H74EicCVdG
+qmRhvPx9vhNeIL6Mj7ncxV57uo4Lw5SOw8sB3+h9p5RMW82Ab6g4vMeeerB
oKXkuSbOyokq7+Kcz/YJSjMk6XbAI0+JM7dO87quynzI6QpvV2tMpafyHF1l
mnBejWln75THPJPzmJ6i9NvWWn6anrN8BR07HHpbMB37smx9D7qeFQrfy7eU
eb3r43yHH5weZ0cqztfpcXY3izN369l/W/D2U+J8h2e7Isb89pDhwBWZiyui
HPYEfOul+r2f2rCvnlnVlk++1fO68txXZ7kSfURzeDxwVNaoxriR+vIUaZTv
oILxnceDFiHfkfYveuoUnq/3o9U+vynvniizHXuzzvIi9AAZA5rAOcv+OCfl
spB5Waufaa19Wm8+dEvSfMSeCZ8t8NnwiHeoPfsU/P0QhdeXjP1MV9l/17On
0g1SfCfF/ydtOv6I4o4GJ1H8HPQWKbfzvpCpkQv4jYwAvX055PAu0Jycsc9t
0dMhi6muTgp3LllnAm97WvC3Y/RuqfX6mcJ9Ff4iZwyht8K/5YyZXKvw1LL5
lAy0s2Id3eEF69vQte2VtU5zutp2gtKWVOYb1cbSwNS+TFh39HnOmDHyNHLx
GuKRmdT2XxobW0Nfga6H84S5uz8wupcDp3slwpwvhyvvpsbWeZD/lYjnO+HZ
oQu5L87Tf9caTwbb5wHnB0d+W219Q+3PKu09WadBJwiNvD3oJHH/Dix6Ua3n
d2jC6SdkXdbuGpN9Ky6rrd6t9funpOVO8IdmSv9A0Toh9EHg61sCY+8Nz5B0
f7byf/B+4B+bAwOZG5glOjUwTXQ89Ju0j4Y89VKtZYDJCb9fCnkA+YDf8PHt
1KcXS+YLkCmR3TnXwE87VbkN8CDgOYwbfV0UcuP2mtsDKm7PV+rXQQo356xX
mf9Xst61l9ZkTz0ZtefojPU8H+npkDWeMjf6/XCMCfgfPAI458CMdVNr9Rxd
YywVDLx/xjQJueo31fO9xvEyffufwr+XrEMeoOdTff8kZZ6AMsEVd1Hbdq43
P/dXvVvUWxZFt8neB3fuUWOsBMzkNn2/td78zt7wmwWf4Wcr7qx6Y0gL9N6h
YL77R9X/qcbi9Iz1mXeGTnMnMFSN1wzFX6bwJfXWrTGHc2Iex2g8zsxa/wnu
S/6Hkz4H34x52ajy16v8kaGXIy9n69uBuUG7eL8d83W19uC2yKJJYzmUA34D
rkga+rtK41ejdTgR+SzqB98cAR0qeC10T/nc4sxqkjPfCw//lNI8pDRLNV4j
VVedxuiWGvMX1yq9im6QGx4M+QL545qQQZYEpkU5YFz83pvzRGkn1/ksg999
qsZ4OjqFn0OvUK8ydql4n2QjL+VMUnueYS7Sli8oH7mDs+29wNPAaykL3QTr
a2CN19WCauuZ2Nfvq+G9NV9DVU6nvHlQ+M8N6uPpqveiGrdlo56rU6YJ5IVW
3Kj0Zyv9lMbmw4+O8lm/hFmHA4NHJ75b1mF49Y3RR/ht3hsiDB9zXI0x3LtU
/kq1Y1CN9wJv9sNOOZ+NHzb2+E2qc/gNjcmdedtFZNSupmr/3BpjBbyhDx9V
e97AVb9Qmf+osVwPL/1R8NXoTecFtnCMyh5YZxlpsMbq4GLscfhc1fWjxnBv
tWcvPUPUv//pvWPFdOyxKpfHOIChgK1uaGwZF/oFplQfOCJY10rlH5r1evqz
yti+4vZxFg5I2p4AjGdZxjgPunWwU7A4cMOlUQ7rFtsavr2qdv5XbT417b0B
9vZN7J13Yx/NUNyzNZY3R3FmFkxrsSVY1dh2O02SLh/MgbrRWfdO+I0+C/11
S2hkxWVRB/FgjE/q9wI9bZLe+4R7x/fGkWZezBNjju4UXLyF4kdlTWOwGZmr
uXu42lh3v6x1uYS76zk0aXudoxXfX09zhTsmbVdQjnNhTqwB5EL6Q1/mRL3E
Y1cEtlsfONOcwIk/ytruZzc9XTjbolxshrCZoKztsraz2crL84aOPRHlMS/I
zIelTGMqJZ+LnImUiQ0E48w+/iD28rIIQysGZG2PQ79ewOYFuwo9h6ds20Gb
9hWtbVZv7Hmt1uafKqGLrPEYgSFQ3rLAsqmzZdK2Ik+rrIU568BbRnuwx2Dd
tYo5At9YHrIP9gjLo51vVjsP6cHwPqwYx2Ou0Ktvr2/TwhaIMDYveyWtC4MX
vAmZVeO0n9Zom7S/sUYejzVzfJ1xKDCoeVW2vdhd4ZEqbwTYL/J61vMEn908
57VLv7rmrPvbqDzHqowxNR6PZVmfAYPqbDMBRs65jK6C9O1iDA6IeWF/E0da
9GF7R/tXIPchc4Z8iP4IOxzGrHWMGzR7acwj8iJyIzLLdTmPB3Y8+2esyxyl
5ycV0ElxTZN+d4w0zPlBMR8vKvxv7Nty5uv3iPZ8mrXe7m96zte3sTmXh50M
bVsKH6W4Z3O2PeG84rzb2vZ9Ik0HzUP7sL2bkjYvgCw/rM62IGDcg+tsz4H8
lhTNqGje761xu2gDbRqd8xjz+7Os24cevKj1/4PWyqS0126XGOdza1x319jP
7Gt0T4fGPt874XWCvpO29UFHn/Jv7D/uC4xrger9OezNsDc6NPKOiL2wtexD
ovxdYpxZw9iDdIox7xhzUB8yDjgaPPCJ6tMJdda5L9Fe21yyrA920Ctpnfv3
ihtesM4FuQUdBzq8+6Kcl2MPdgl6wn7GHulvCeuesU8a8P+1AzqGjvSwWGOc
BwNijVHevdE2yqWtm4KPvTvi2fMrYs9OR2ZX2yY2Nu91cNDwWUp7a433FzYj
2Kt0Tlj3R150cMjKz4S8fKriT8natvFHtb9K/NUJyjNWjS1qPdyh+C0ah15F
j8+31Y7DNoJz79SgkyPBiurCdlLvc+osg/yqfKWK7XnAiZsHVvwe55qeQ7Vv
1qjeRN5y0BRN1FX1th+Ed2R9nKg6TlF5J9fZ7uTDjNuyOujYyugX9GRW9B25
dlbItj+p7FzF4wI+C06LjcGwjO0S+A32nK8Yf/4M3rhk3hx+s1/ZPCc2abyx
aWxcsAyP/E49M6Mu9jdhaAd6dfARsBH0Cegs0GMsrLZ+nt8PFqzPBptHVkcv
yxq7Uu25s958SB+15ciS7QiwvTmnxuOwttb4JPghuNdPsX53K1iW4zf87tga
78fTVMY/88ZR7ypbzt8S5wt0lf0FVr0w77PmdJUzJO8y+Y6cj4zP/mafgwnD
c9EG9ATrlTep+b5bvz/OumzG5Kwon3p/VJ0/gd3UGP85N9p2TNa/oavgx9mC
eWpsRnjDV8MDwTPs0cj8LjgSfGSLnGV5wjupDX+t+PzZDWyt4HNnbOhB0MO0
Uvz1BZeFrndCjbHH11XvxrJ/j4320TZkIXhK9FcvBV/eL/h8wvC9C4N+vRx8
EbYw7Mdtc64DPBNsc3zU9Z3qWlywDo+x4vyDpwX7AvcCcwLX4Y2sjY0tv9nv
6PzIB64DbvRu0dgRZ/j0ODcbaww+zXv8Pqg2j0i+F7QGFuvZL2PMiDIpn/V6
edCi49SG9+I39jpTYvzvDFlvSuAX4P7IbpdmXG822nlbnTEoeNULg5fkfUGE
sRfiN+Ug5zE2yHo7Fiy7clatzVpORa/zQNh4EM94T44xfyVvOQcZBxyQNQWP
eWLadkk3xBjeGuNInRdFvbdFPOMJ5sL+AcOBhjHP6ChPSnsP8e1b1VVbMZZ1
RbSBtYAM0qBXTJiXxH4cXUN38KSKsUEwlcGBqxydM9ZDndgJc87oUwMv2rHa
ZXSKtjPGXfO2TadcMEvkBHDKwbXGl+GLoLfYTGPTOUvpF+WNQTXoOOocRve2
vtr8JPhlKjDMrwKnvafKul/svzaFnI7szfzCV4BJcWZVa+28WDAfV612LSla
99UisDHsHDhD/pP3OYJ+AKwceo0tyeDgz5jfL7Kmnzntxx71XkdvVZsf3Tfa
mQkM9q3o9/VxhtI27O7gFZEdkXNYgy1jHZKmEu0HB6ecO4MHGB59pB2DYwzh
eY8P/TVrH1mLtXpN2dgEv6eqjZvAIpPW76VLlrmwZdhcDnuGkMPYa29rDH4p
G+NiTUFDkEsKBdv48hseDH0zbYM3Q3eHvpu99160YUTO9gfM70yN0YMp58kV
bDsL3WH+0XeShjP9shqPP5jEZYFLnK34s7KWYaAfrI2WISeBsSErsb6QdSYl
jMMQx/kOnnR5YEp3x7gtjzTsXfbLZPX1gqL3Azh4V+wwUh5v7HKGBG4Gr4Gu
mXZdGm2bqHz7aXwfUfopKqdL0ZhJWm1M1ZlmrtI499S3J6qth8TOBszqgsCh
oLfIIcgjrJ8vNf7T6o23s/+hq8gX8EvoIuCZ4Nkb7GWS1rNiJwyfhj4JnwT0
Ntsy73nr3cAvwX/BMG8Gc8pbpgO7JZ59jQ05edkL5Oc39lS/Vdl2AzurZ/LO
T15kXvTw8LTYvHwddi8XFUx3oDn5eu8T9gh9+irvfmErja0kNpWbsA3L2m6E
ttBW2jNd6X9V+geStuu5OWx7oK+80cFjowTOTZuxJxoaNkXYRmJ/x14Do8dm
ETsQbA0mhL1Bx8DMxwYvBU/FuUFfx0Z/aRf8POM7s2DbPsrFtmRMlI/OB3yH
eYKWJGMMkRGRGTmj4P2RAZrFe2SER4dcSBiskvMQvBfeBD7pEuwNsj7/scHA
r+KL8K3A7hi7anSUyHHDarwuatXOu+vNP/AdewZsT2eXrbOhDmg5+gx414cV
d1i963gs4tEDMLe0lfnFfvXrKOfbKBPfHGxwPgw7HLBszgDo/+K0zyHmclHa
Zxvf0KfBi48KGQNZA7nyNXwsQvZAN8NZztjtjc1VwXV9ono+hY7VWBe7LvpO
2ny16c496kep4HBDe6Od9yr+vnqno+30ARsPMC3KQ6Yh7XeRfgcw9rwxw/u0
p3/L244eXTb4Ndh1J81Lx6x/X6nnqhrTWOxb+Q3t4D0lwtjCXhHn75v13p/s
zV/yXuusc2zXeWOvnhSdqMK+DZwhY9sWbA7h61fnzdODu7we2Au65dqS9Wes
ZfLBA43OWh+AjID9J5jWnoHbceY32LvBP4aty3KV/b+ybRPgFy4PvgUe6vbg
ozhLjyn7PL1MdV5asv0aNuqXZG3L9bTiFpZsW7RA78dL9qc5T/nOK7o/QzO2
+aBNS/T9/ZJtl5Dt4E2ht6wtztuhjcxXse+x9Xqv3vPEHDXVuy20JWPbScYc
2g722zVrXBgdQ++wq8CPgPlE1mR97ZX3vD9W7/lgDqDf2PRxTmDbiK0hdobE
Q9+HBnZKHPgq9ZKe84I9MbPebWedsG6Y6141Dm+1F2XtoJvoWjQPBM+DPWEq
yr9eZbxdtk0W+pZekRfbF8LEbat8FxaMqZ4UeWnznLApGhhYIBgjsi96bfYf
58ntoQNp8AOq83plDUNzwE2Ql/HJu6lgHJB9MKn+D58S4pDTV2ELUfEeAO8a
HXLTFOVrX7TM81LamA7tgGfsEHwjtHFUyHFgmsSxNl9Ju+0DQl6iDHimKwu2
teIba4S1As+ALhy9OfYCPcK/AH1o+1j/0HN8cVqFP04LdMJ5Y8LMA3uZuUD3
R5j5Ys90CD1ad/Zo2efhncr7vtrRUfNwuMrYFT1W2NxhO4/d4Ax4M6VvpzRX
QI+Vbm3GurFbw97pq/gNroVOBX87fDCXMc46a37QtyeVt3nBevi/VMyfNdhz
li0PI6/eqTHeXeG/V/tcw8aNvjfNGiOGPoCzgreCJ+MPh18cPpIvBAZL/C2R
Bjy2hFwStizo8GaGTemXGf8Gf4OvfCPvNYN+AfwD2b8nOHDZebB5Ih78Ad0h
fQLDh359F/E7q191BWP2DXYzVdZtnZC2nxm2BLx/iTC2cdjIYTeLz+XAtPlI
dCO/FKwfgackjrToU/E9gA4zxtBi6DC2/RtizPEdwW+Bb31UzqaC82A/xLwy
p6mc7YSwvUE/Al7GOYbelzqxJemgfncsmy87DN1W0fIUGAT2ldjJY5tGWdiq
zYjxB3+rDdtg7Eh5fxvh3wr2XUE+hH50inMHu4EvYy6w3cH2aHOs306xho/I
GS/EJn87tat1vX//pDIfrLge1vWaeq9tZDjqQS+dDTkLWQa/hMvKbhNlYueN
DIIdF/ZTpLsYzKls3zvOnd+CjwI3+irWOf4mqRjD+aL3D4fdxcLQLTAOYLlg
zeDJz0YYzHlXteEn9Asqa+e054Z5ARPFzpw+PhV6CvgKcK3FsbZ5vxDhhRF+
KuwF2gSewL5YlDOOi41K25C1nwncG2y5bditUBd+cFPDFw6s+d/RZnTzP2Vs
s8q6WJ/x2sA3GvsO9ic4GXo79gt+HNijg9eyHlmX+Juci52bnr2z1jmiu1Vz
Gnzj8IvDJw4dInZ+YFqcc5x36P4+y9lmHfmOfbwxYx0fvCXpwd/ADjfkTefW
FOz3y/7opLy/Z03LDmaMFH6yyusBvxfkbtYqPjPYsWN7N73aNsMrVEZfMPZa
93tt9J1xfSJoDuP7fM6+jcOCFoGXQ38ejDkaGunByvFlvzT82dG3kAaZ6CW1
vVAxf8Q6eS7G/8XQNWxdP8wfNAf680zM46xIPzOeZyMN87tPzDt2VfhbIr/T
j8+jL+DlT8Uaw6YYuz1s9tA14X8O1vd8rCX6i/65W/AhpHk+4tHVMl/wMLy7
RRibxxFNbIdHnV9EGx7PuXzyLlfcZxnjzIzh4xFPvx6KvqP3IIzMOFnlvZb3
uQ2dgL5DZ5+I9c44I+ssiHmZHWVCr04u2XYWmrBBc9um4rMc+yn4O2gbfqX4
l8JbIn/hA9sqYT3wXfXGQRepnEX6fWDKvn74F2EjAZ0eGL4P6LbAzOnXZtX5
StF7AuyYePR0zZTvLznbNOJ7RNmM+aqoF7kMH1fahJ4FnyfaRhxyGvIauOWa
SEObW0d7aDP+/vhu4be1GVpR9G9sAMBuwCq5QwDcExkdfR3x4DnsVdqGrRZ2
cJxhnFlvq4xHKv6NLIRMBK0lXbPAgrfR/OxTsexURv6oON3pynta0fLgVvmz
acjmo4O3r1P6V4seR9bGKaF3QAajPPBAxu+UGFt8mvATY0+hQwbzBdME6yWM
PQY2gc1inLlDAN4RfhH/Vuy5wZOx42hfMraNPeYhedtk8h3fdc5W/OcIkw98
kXORMxFdKzY56GbBvtHTE4ePLvHwZi9FWZTzner5oGibkKUKH5W3vuIrhU8u
ei1gh4evBboLyhgT5TB3YO7wktg50l7C+OvjH4JvCD4ihBt4kviNTSZ2Ddg3
YAMDn9M00sBbbZf7Y/y2i3F7Ne/9xl77QG07Om8/JjAssCx0nWB2yO3oXrE1
Wh5zgc8cdpDoVdBFoZOC3+sdOBj4Eu+zIwz2hH0ZNufY4NBObDDgs9mj6Lqg
TZxV+GsjVyMzg9Vhh982YzsTeJOL0x7DmWrv10X/xr4VO1fsiq+GZ8jbTgm7
xvfD5xyslDTYwYKFYucHnnmSeIGTy/7d4KeUNc9zv8p4uGQbJPqdifLvVfyc
kv27x+V81wY2f2BjbwW+Sn7isAX6J2lL7kMDTxp6YeQ/MHFkQPBy9ir7lLjT
Aiv/UvkOK1qPi80yOmXwTN6DIzwkwpTJwzf0YvhH4aeEfQ16dXhCeDr6T192
T7iehvqSPj85Rzk3q9THddDAhLEq9BxgPryXRriDvj9fMo+EDRrxlDlPcf8o
uW72InsSLJGxYmwnJCynE88+XRT4GtgamBG2INiBgGsR3yJhW2vaR9ueVdsy
Jesi8WfC/hqbbPT54NvoowdlLMdzJwb5SDMtMG32GDIatKxF0En4hi2RpkWk
IZ57R7ARAMPEnoQ7erApabCnjv4iX60IP1bqfyvWFXj5mxHG3oCyWBtgouuj
zFtzvv+FM42xmRjjg16deukLuB7fOe9WpZyGdQrvg504vCXYLPekUHbHos8w
zq+JwSPB/3wdeUhP/RuiPfg5UQ426LRnY4w587ExcEXOENYnuid8EPeIM7Fn
8HyMJ3ItduEPJWxXfn3OZxd2DjfkjA1ypwS0hL6PDpuO5gnbF+ObfVHQFPCh
TxKmBfCtYGjcWUSZ2HTBzz4a35ChuQcFHBh68HXJNOFA1kjePCQ6wL/nzJf1
zJnWUDZ2YtjBYwN/YcRR1w+KSyvvwIT1zBeF/IIulvMNvy3sSfndI2F+lXh+
Y5MIzg7d23ofDbj3MUXfibP17g/e3F8B7t4rsHf4GGgbfAI67HEhj2C3BZYN
zoltzLXB/zDH8AHcV/J7zj7q5EEHCS/At4Fhf4vfE7ZSlAUegn0NNkjYUIzM
eXz2jLmbHuN8VKSnXuyroe9gqJNU/sqMbdHaYtOOHI1/ecG+u9hTPB28Mnzy
gSG3I79g1zYz+FjOAmw3ObOgA5OiTNLOiPTwd8OCx1tasK8Ctpz0Cdti6PDa
hOePufss73uIuIOooA1UrHe53CFFHDILfWIs0BGAs2Azx5jjL0yf4cd+Vzkn
Fo3B/qT6M3nfSYWuD99C9HzgT6xteFjs5jm/WFvw7+cFhtA8fBLJQ71HxfjD
k5MWDAQbsOtiTknPOYkNHnF82z7OUOLhW7BjwJ4BPpD3ObFfFkeY7x+GzQPx
+Lsh67MOudeDexfYpx+nfecB6xA/BOw/sP3A7n9VxnY5n4W9CPwYOCLnClgi
/AD2y+C9/9bYXF3v/Hw/KNJAr1ij0DR492NjbDlX8WHjbMUPHd+baxP2QcbO
jPzvcv7mrQ+6cOs5m3Cdq6Ne5GvuViA/NgXYOmD3MLve+kV0i9htcZ7Az6Bv
xd4L/SV+NtBHaCM2Rj1jfJAXL474g0MGA3tEDpsVMgv1DIy6sLPhbIdPwV6L
84HyvtXaub9ou4tuRd8lgY8atJbyobfsqV7Bay3A96Js/guaCm1lfP6W8bnD
+UOd1E2Z4CLrC8ZG4DWpH7voRQVjteC00O9b43yBH2EM8U0ABwEPwYeIM4Fx
oF3YlZfLnm/Oo3HB23BGUQ7nQo86Y9PsO2zL4E2w5VgM3l9vmy70MkfH/uKu
Hu4juiZhnrB39PfaWPPQH+4EuSCwiP5F35VGnk2cw3nf28ZZcUacF+Cf+Hmj
j+oe7TkieAF8suAH8O1G998kcFnGk3OKMRgXawka0j1oI76h3aOcc4q+/478
S4vWxw8LXGF88EjwKqwr+ExkZOYWQz2wInx1wIvQ02I7Wx9r8LBYh/Aa8DS0
96yi7++7OcqeEOVzX1G3tPWC2FFir8V4I68jmyMD4weJ3yA+g+iM8A1el7Cd
9/0Zn1PD1ZYRZfMo+Grjs408ghyAbIAsgE4H/2N0CsQ1jXjK2xFMJ2XfXmQj
5KkP8rZpA9NGJ025yP7ouqcHxgh2zP0IcxPG+pgzsOta9Mb1xnrw+24a7YFP
h05zDqB3wVeaPY5/EPg4GB06w91Cb4jP0MERD/ZNGu7doL20G7p9nujMTnnb
l84pW4+IDrGeu3xytm3GjgUcDSylbZS5JmHek7aw5mkL/tvYuE2rNoYNb4zt
xHbBb4NfM3bYNzP2s2P86d92MT7ImIwhvtLcD9EqwscVfScOdJn7FsDMHor0
4GzIlQPV9qPL5vXwMWP9M6bIgsiHyHoHBi/A2p4YPCW84tSE46Gx8H2UAe8H
BtAscAB8BSYHdsc+4TdYAXpxvmMDMiN4vDkJ67S5jwkcBT0z6xTbEMoEX0Ae
P0fj/Je8fS5o486xHm4I2RW5lfMHGsEZRDsJQyuQxVtF32n7ddF+dCc3RfqR
wUsxVgOLvseH8lh3u0RdvPmNbPtO0RgH+EZJbduYs/3n8MDfwd7hw/ADwQcE
mSkVctPFQUOR7d4tmS5AE8BQsTGEdvVQ3HNF71V4SvzE6TtrAvkbmyywNe6l
4UynbLBmyoSG9QidX/fg2aCxpIU/wZ6fcJ/gVfBzx38F2gU96BMyAvquAcFj
vFWybwZ+GcfH+UD6ftEGysHPjzuDOPPxwYLvAt8m7YmR/ufwcSHu8JzvIOPO
TeaZtcEa6ad6mpTdH3SlA2IMJpVtvwKdGBN2jujROPc4/9DFzw7+jvb0jnjo
JHLrwDjveudM57CLAM8DC6AszuSZcUYPVz8mFO0XDM3i/jPo4fCc0/aPcugD
37B5Y1ygaWDbY4LnIQ/9O0Tvw7BlKNtPFN0sfC62NzsEXSRtk7rYt42MV84N
HBs77n/lbct9esl+2tgFTNb75Lz9mntHeeh8y1EO6xlfE+4sw9+EuCZBFw4Q
/Tuw3nUk0qZH0KIGjCdwHux99g+bHzDvaYF7g99yVoG94yPNmcVv7h36U9C3
tkEP0R8gI7cJ+0/au0+92wzt4o4beGnW3bTgXa/TOE0rG2vG7ntL2H5/mzOm
QDvRf3M/HWMH/8T8jkpYz88YoOvHF/z88J3Ht+/RvP37oEG0CdkcukXb+I2P
Gve+4afG3Ub055uE9UBPhn4KH1Rs8vhNX7eJ8wX9aqFk/I17V8A4sDHDxoY7
WrCzgUdFXwQeyF1ZhKFH2Amy1tAxse6alv0b3cm84O3xY94YeeG94GWhc9yL
83DB9BnZALkC3p47OLmvjfYjszDO3CN0gcb+wnrz1NyDgoz7euTbEHmxa+A3
sglnOTp57DR+Dh4OHhyc+diQ8S9Sm7cp2+7tiPhG/CVF362AjM1da5eUjR9M
UP0X1xu7xfcI2wj2+rEav+PyPnNXFn0vLfgmsh4+Udiu4K87JOSLfxVNL6AV
3M8zPGQl6PyFwZuhx6VdYKTQWWzKsYFi7XAHBLSgDXYFZf8eUjT2Cp3Cpg/b
vltCF4N8i57t7YxxS+Qk7uokDTpYaAM0ArrE/l8cMhRy5/CgTdAMaAf0p1fQ
EMLMFXN2aMxduwgPyBtX5dxBl3lC6OPwMW0X+MmkKBda82TIuehHsHMfGfI4
svgZIY/zfWi0E5trbK/hhdjf0AL2OHP+Zsw7cdz99Ub0b1jk5Z6qUVEmGA1h
ZF/4d3wz4OEPDf0Wawa9BGsDHQSYC+sCOnZd7A3O5U8D60HHiF8LYTAfcF7a
zzl4r9b7+Po//PSWxdpYEjgAcv3osM/A3gJ8D3mM/ct9h9jlwBtiv8o9AqmQ
40iDPMU9DfDo6ILRE2Brgp0JtANfYvyI0QmjG2YuxofteINOIee8yA585+5I
5g2fuM0hR7C3iWe/HxnlEAbbPibjtNzPsCnaQP2Uj40l/NiqkIWxD8KPExsh
fJjxFwFr4rwdGPII5d1W7bqxgcG2G7smvuN7yZmOjQy2atjAgEnfkzfPQH/X
RpnYHnN3KPbH7TR+B5Rd7tqIJw1+pfiR0ib4joFR/h4q7/yy62gVa5t7KMfE
HO0XNPvrkHFy0Omi7VfBY8E94TNXBr2AxoA9Eg++iR7opIx1amcp7xUV34GC
Pm1dyTo19vn8OPfBU8dGmfjr4TMMb4/9CvQd2r4k4vEXRI+0PLAL6iMvmOqx
QQ+hdZODJ2Avcrc093lirw69IrxjzAXzwPzOz/veZPyGsVHE7py0vPvn/sjX
P8phjBlr/Hx5fxnhs6I8cB7sIrljbqufG28wUnzp0Dvhv4ZvCn1BX4bOCP3T
d/EQRufI+4cIw3Pwm3OtUdrrjLnmHogJTayD2kV9GV32nejQE3wJ2bPY9VxT
bzsY8GEwFnDjhntCc/YHxCahc+wj6E0izi38CtljpF0T8Zz7+P18XLbvDz6G
1IUPI7QPerr1/tpeQVcZN9pNm8HJwG05T/EL5R4B7hBYGmlY49x3g00A5xQ2
sx0j/abYk2Ad68I+APmavU6YfUpbjo/2gOFyPxe4LunrIg04SafAhehf19j7
0B7sDKBRnPvUC88wNtqBnNo5xoSxwja+a4wP9+iAG9K/b8JegfLBtLEPA9Ok
bYNiXrbSVtYFdsDsRfgE2tsq2twuwtB/3nxDDuXM4a6QDRHfMtJzdwjxnEvY
eAzPmhfGfhy7Newb6QNtAmOHh0JPxF7jOzbx4ADYnBMmHzbz+E6gG0RX0C7q
RU9IPGcu/tz4hWNDxXl9TsT3jL4zd7z5DS6PTgT/w6bxTkeY/UIYbIiHb2BK
fbW2Ly/bLgp8kDXMHT3QOGgda4e+0mfswK+NNtEesE3uyQSv4A7FnQNjxJ+d
NGCb98Z4bYj9sXPskQMjL7zlh4xT0f/rcITa807JPDZ4MHfJNYu52jfmCx9f
7sNt8P9N+97nhvsu8tYlg5O0jbzQDfYX93tSJ2XxDT8D7rKhHGRS7nmD18HG
DB4E/0zOd3xY8TeD9yYtdeM3Cu1oE23AxrFt1EU9u0dd7E3axplCHN+w+YRf
g67AsyFDg+uhA90z6qM93APcPuoFN+wQ9bIe94/1DN8NTWI/kfbgyIsvMT6u
tBF/B+5VRpYZFGHWKr7I+CQzR8TtGWnAY/aNvUD/9tva30gzKGSfE2J8KGPv
KIc5J8x6ZSzbBm85IsaWMHe7dYl+0d7O0XfuDMNeGDuwVtxLVLQfLJhC50jf
PvKSj7K4awlsBCyZtQM2Db4CHoiPa5toAzgMNvz4Z6IP5YwC94RXQYc+IuLh
Jwlj549PJemJ467fXiXjF9dHvZTJeQ52yZnOnS6f5Iy7Di2az4bHhh88I3hC
xoXxwgcc+3js5LE/eixvDAJeAl3xP4vWF7eOtcT+ogzuMkCOGBVlwifAL4yK
MHPF3IN/7oh/btn2MdiKnBT2NujfwD2ZX3hn+gfPD73C/g9ajk8H9+Dj18F/
BRDGf3brb+Qs7g65r2JbV+xqRwfGyFmIrIKcgg0OdG3nkLmwKcf/C78r7kzE
Lhocirv4wbPQaeCjBe3/uWj8EeyRewCWBrYzONLAO/GeFmH0hOhpwLu4X7x/
yXeMo595K/AK9CRnlKwrwXef87VN4OrsQ/Bz7i+lndi9c6aQhrPkCZW5oGDc
lv4j24NJcv8o8jCyMPdxcVcdsiznLnf4gLlix31S8Lc1qr9csr6J+w+4D477
EObl/R8P5CEOf2l8irlzFH95MDruq3uo4jMcW2H8NrF9+qlonA6M7jm18YSS
759FLr4y5g4/ZcqkTu76gSbCb3Dv3GPRfmwTCO+e8N2T2Mqgc+TuQ3x34WOR
my+JecRGgvz4cIFTIHPCHyLLkLfhf3mqPT5by54f5TO+lINs8ih+ISp/Vdq+
CMeGPwJ3nHDmsdeZw7dDJmUtsCbABpHLxxVNUxvsO+osE6NvRO8IP4C+jHXC
+gIbQrcHdoSeHJkO2Q7dAfZq8MCtIg4d3BzNy/CS9bmskSdinVD251E+Nojc
oYDeE9ss7gGEp8UGH1t8bFjA00mPDILfE/6H+EJwBpGXc+Sp2Bvsi/eK9kkD
mzqsZN0q5YNH0n/6zh4iPb4UvNdGGP7ry+CRwBY/CfpDfwkjzzI+yJ6fxvpm
P8Avod+GLoAHcE8PfYY+g9Uuj/F5ML6js4aXYC6ge+jKPgweEj6FucPeCdpD
XugSWDV7B0yMfs+PcRtQNnYP5s45sjL4B3TL4CZgJg9Fe9Et7pD3fyNhZ4Wv
BmMxJGHdK3nhU5CLoE9g44w7c4auEjyc/xcBE8dG7YOgA9BJ1sBJYe8zMmx+
FsQ+ZT/ynwjMLz7p8HdfBZ8M/sR+Wxf7dVjdH7g6fvvoHcDd8KHCZvBI9bVP
2TzkVt9+MBDGHzsMMI0NOc8944Oe+qOgJ/j24R+Ib1rfkvV/rEl4fHgO+Hxo
Dv4njM3w8EuGfnJGoudGh4tcTVp8UbHxZS5Yj+jAufcOu4xni7avQn8J3cBP
BR4GvBXfMGwSsb2gz9wt0ENpe5b8X0ngcHNLlufZz/iDcd4uC5mE+unrx9Ff
+kc/oY3491EGPn7cm3V4yXcJnB1p0EHhb8Gds9TfPG89FnLuLzmPC2PSOm8b
F+xb8J/+JvwZkTuRRfFPXM05WXbf8APAh43zmbuquKsOPJI7JIp501LoNX6V
8Pmshf/Efue+wG5ljx06EPKyd1ibz8a+5myAV4ZPRqbPh1xPGa9FOWtirBnn
N7bS6kaec+rFDhw9KH442AYeVfQ96VNDhwwdRTfEXVb81xfnPTgj9z5gL4Nf
AXbD+GBQN2cFmDz7eG7s5eGxhhlbMIZfC95Pq2MdMkd8Jx06THzbwV7whUU+
5YzkfGTdMG/wsPSb8xEeb7nafHjZdkOvRt/xQ78y9hH5O6je0RX7D91Y8J3R
6Mw581g32MuAIZAf/OG1mAt82pEzbo664A0HhZ0b8j53WCDz/z/REoVH
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtmXecVdURx++y5e3bZe+9rxCRBBAbKiRRRAUiUsRGUYgCilhQioCVjvQF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         "],
         VertexColors->None]}]}, 
     {Hue[0.8950302241245203], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFnXW8lNX2/+fknJk5Z/qIXuOKqIgiKBIiBqIIKmF/RVAp9dqg2F4lFOva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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtmnnATdXXx69ncJ97H845dxAlTZSxknlI/MwaFEkJmSmkgUy/IpSSSspY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         "],
         VertexColors->None]}]}, 
     {Hue[0.811696890791187], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmgm4VNWRx1vee939Frrvvd3NIgiICEoAQaOC4BaXuERUcJdVQFAUAffd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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtmXnc1XMWx389Pfc+9ynub7m/p10lbdKqlCzZKoRKi9BCRSLaZE0YS/u+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         "],
         VertexColors->None]}]}, 
     {Hue[0.7283635574578536], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFlmmIllUUx686M+87Or2+z31mdBynXHLLNDUCM61AUT9E5ZJmggmSZjOm
pX4oKlFJw70N0jZaFFqoDCvRplyCMrVSKXBpsUWDli8tbpn1+/u/0YfDPc+5
5/7PuWd7brdpc8bNbh1C+BFqA1WXQ3guC6G2EMKGqhC2sI5GYWH7EPbFEIYW
Q3gvD2EafE1lCIMrQrgK2dlW7KPfHtlE9MeyPy76fGfO7of/BgP16HaAvm5j
Gzn6XdBvh8066Ht86FMXwtXov87+cHRHJPzerH2gRZzdjg+3R59v5twebPyB
/W217EHNbUPIuMtf0T7Jx6EJ52dk+9Afy3dbcJ6HZsGf5nwPdBfgTz8w+kPb
qkNYy/6TUD18d9YjnA34thf+TbAaOXsdawV0P/JZ2L4TWS+w7kDWFb1dyPvj
T2VuGy+wXqs9MKeX8AVbGXbPVfncEvj98L0T/y1nj0LzkQ3j7Eb4Z7HRE18a
MufqLvBmQ59g6x/WoBih8zn4J9Bfhs5HyG7G1kjsjswcI8VHd/o1+l6t+F5d
5bzn5CJCP3CnlkrvT2ltHzdV2bdF7GfQlhr7ey7JD4NZyv0t+yMy+3AAeTF3
vI/i299l29tbsEzxV247lp3ffvB1nDlI3gew7uL7ZfSWFrz2auXcb0g+3wtm
fWZ+ORidVYfgNCG/pM65fk31FV2LjyHvm9ln1UtMMemS2Z5sSE+1oLpV7ZfK
rv9G+D658Yej3yuz/w1g9sjs2zVgXJo5Xp0y50y+NoC5vdp90QuMR8Fcgewk
+TkFvUTMFxe9Cn8V+91y50H1pFW5uBvZXOgYuu9zrgVq4sxZfBuCrZ5F1/qD
BcfwEXxYnno8y50n5Uj7iql8lu+dSuZVL7vyVDNtbecL+r4AfhFawJl1wsO/
C+Ef4NwA5C8WHPvGFP9h0b2tvt4C3n3oDQGzd4X7/HyPI++ZOw5j8GtHdM93
zRxT+bMG+aHouTAJ2UD49WAOxs8nUp82oHM4Ol9NRded+m4y8pnRNjtc4Lqe
ULSv61N83gVjIbjDwBmnGRA9e1qQb80tn4E/S0uO/+Si61d904zsw+iZ9lXZ
9mRrHuf6sncc3dGcHwUNRP52O/eU4qn+kT+qi/9yvLXStvsWPR92gvNG9ByV
7rHkv+prXnSNqV6FrVgpX7IpnZz9LDrP/dF/KPl/Cj/ntPcsH6jaz93HO7n7
jlrPs+tZNyMfjN/3lDzvNOvasN6Umx+gnsg9G74Er1P0vTQ7Z5acl37qs5L/
Ecr9lVC3Cs81rbrjZdhaXOfe0GwYEx3fD1jHq5eKntnqZ/XybnT+ZO8WdFZk
/j+pN+dmnjWaKZqV6qWtaYYp7rr7LDAezpyvraw3gDWI/Ytzzw7V7cesJxN+
U/RMV266oPNp2ZhT8PlW6Ey14zum6Fr6DPs/RX9rnncvOLZjkY+PvvOSGvuj
WD2FD09DFxVcN/XpX6mcDCr77PGy8yReNb0szUz9fzsXPLty9teC06hv+MfV
P/B18M8k/DPRd9O9doO/puTafIt71MBPKrrX5Z9805xbkWKivP2ecncQvJUp
DqpJ9ZVqWP+rndG5nc+6J3eNba6xjmpSvaIcKP6qcemqBg4g/yz6vy3SOc1J
+aR+U68dS7NIOLqz5pzqqojd05ydWnSv78j/nzFaFc8jyA9Gy06wVqc5pr48
k3pzAnGYWOt7trD/SzS+6q93eodMRzYqc+xvxG4L/OXgVCH/rmSdqchal+2z
1t+iedWoYq83zxWZ3wG6+6roHlNPqS8VO91/DTqroI4Fvy/mpDfGocz/Bs0M
9bRmk2r7tpJxhak4jUyzQv+Q1Un/1ehZqZisyz1D1aevRM9xzcXz/ib/V2Z+
e+m/rBwqlw3Jl03p7aLeHVDr/p0BdnN0PI8nH1RL43PHSPERhnSFszH6vahc
610lmXj15P70rlAsGpNcOdFMUV70FltY5713cs8szat/AVDzUwc=
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlltsVFUUhjfQMjNtmZmzzxQBQShqkVIK+IIXJAK2fVMoQpAIJMZbKIiC
L16C1IiJooAhgvCACBpRI5J4gYJc1YiCAkYT8IKiwoNGYwIKEkW/P/95WJm1
zrrt9e+11p6GOxZ03NczhLAaqoJq0xCGF0Pomw9hPvR1DOEbaCL8iV4hLKkP
4TFoAPLAvPkuqLvO3xbDt1RC2FsIYRvfpsB3QMPQJST6k1hnoZ/IdVvvEH6P
9pHtNGwGJPjwrTsXwivom5B3QGOQl+KfYJtCP9c457JyCIM48+P4dleH8BG+
k3Re6EOojlp68fso/jOwOQP/A9+WE3sm8g2lEPrgvyCrdzW5WqESsS7FZzP8
aXIMQx6KPBn/h/Bv7+kYs9HPgXpU+1s98X7FZgqxjoFXC/wM7F/GdxP0ObHG
kW89+h349EB+p+KzKcaP2H6AzwjkLvwPYnsoNdZjuaBN8J3YDCbXEDCeCf+d
zofunx4h7EO/H+qX9zdh/ZcwzhnzNejeg8bi2w/6IxoTYaEz/Ab/LTV8pWbA
5xyxz5edW/H7ky/F5gl0jcgH4O/GfjT618BjObZDU/u+Tz0F+D3U18l9zSN+
HnkkNleg+xv7EfDdxNgONcEvB8tlUB36S4iR53tt4rOrhk7kT4nRXnDORcir
iu6Nt4nXG92XyE/BzyVfpFcmcL5TNe6ZDviT5HlQvYV+OvqRxJ+F/7u1nJV4
k7G5Ku+eXYnuOaiSM+b7dbfoG6oco5lY4/m2Bf096nH4w+gn541RVTTGwlY9
KFthIizkcxDb/ol7/SxURDeKmI3UfxF5AfYb0S8m92fYr4IvlH2eATn37viS
a1cPb4nuafWyZng22DckxuoCmC0Fm7uwOYL9dGIuxL8F/VJm6RT2T6v3y46t
HM/AH098l+pXYflAalthugFdX2zSatu/iNwHuQ5aD78G2lByvoHo+5N7Ef6n
8+6xdvS3YHs1uqPYtKK/CP0L3QQV0V2I3hXqienUc6Js/M9QTyOxp1XMXwPt
jb7jpuz+5iEvTNxrh8gxhNyDU9dyALke3VpoUDYfb2H/ZjR2miHtwmbi7yn4
W4XYr0fPsuZ7FnyO8/SGboe/ueyaVIt21hHsYzYv2glF4h0suxbl2yld0Xdx
JfoJ0Rio9huhR9BtT13btVAb3+Zn8y/MdbdDEs+i7ngn/Esl5xem5zn75al3
42bua2rJGAkb7Zg67Qts5tW451YQ+xO+tRU8Y5fBn6vYVzu2gG+e88/Jex73
IZ8seldpZo5E7yTtiuuRt5YdQ77aeVO121PXojsczvyNy+ZHM7GLXFWp98Nc
ck6inolZ/2umteuPJb4/xewi13/Rs6cYR+G/gK5D/p4zjaHeWRX3vmpaif3a
1LuvteBa9a0104/Cdmu0r96wNnKlZfeuelh3WYGezPlO1Yt3Fv22qZ5tqXtS
96UdOwZ9knqW9YbcCj81NV7aBwdSn7EtO49iD068S5VDd70x9a7XnTeiuz96
FwgTYf9s9r4pZjW2uyvGThiOhH8j+u1WD4+Gb47uXb1JTUW/wXp7NZPHo3e+
7m8S8mHkJSXfpTA9hryiZJ3+E+yO7ln16i7hQry50bOlnardoh2r3aodo13w
fOK3VXjei/3HqWvXnah3X0381qqH1fsPF42lMBX/QnZ/+rYu8UxpljTDmq0V
id8OzVg/+M7Eu1Xv8y/RM6z6taOjao2ePb356r1c6rdD+HVEv/Hy1X+S85r3
7D3VDLTgX5Ptc71x+yr+D6X9q5nS7p4fvXt0X/8DluInMw==
         "],
         VertexColors->None], 
        PolygonBox[{{4103, 4298, 1625, 4855, 4102}, {4035, 4704, 1966, 4669, 
         4034}, {4703, 5052, 2364, 3712, 4702}, {4501, 3999, 3998, 4502, 
         1819}, {5050, 4701, 4700, 5051, 2364}, {4519, 4325, 4324, 4520, 
         1826}, {4118, 4511, 4510, 4492, 1299}, {3716, 4798, 4797, 4796, 
         598}, {4867, 4866, 4865, 4654, 1955}, {4301, 4506, 1821, 4116, 
         4300}, {4516, 4719, 1997, 4718, 4515}, {4857, 4035, 4034, 4856, 
         2143}, {4793, 4140, 2151, 4864, 4792}, {4866, 5309, 2666, 5308, 
         4865}, {4107, 4499, 1818, 4500, 4106}, {3999, 4539, 1842, 4540, 
         3998}, {4109, 4507, 1822, 4509, 4108}, {4901, 4139, 4138, 4902, 
         2187}, {4325, 4503, 1820, 4504, 4324}, {4704, 4703, 4702, 3713, 
         1966}, {4675, 4795, 2085, 3897, 4674}, {4671, 4805, 4804, 5027, 
         2336}, {4717, 4518, 4517, 4716, 1996}, {4862, 4678, 1299, 4119, 
         4861}, {4871, 3795, 3794, 4870, 2154}, {4518, 4903, 2187, 4242, 
         4517}, {3795, 4867, 1955, 4655, 3794}, {4508, 4301, 4300, 4117, 
         1822}, {5312, 4516, 4515, 4142, 2667}, {4493, 4107, 4106, 4494, 
         1815}},
         VertexColors->None]}]}, 
     {Hue[0.6450302241245203], EdgeForm[None], 
      GraphicsGroupBox[{
        PolygonBox[{{4873, 4670, 1315}, {4495, 4299, 1626}, {4139, 4901, 
         1315}, {4798, 3716, 1967}, {4107, 4493, 1291}, {4795, 4675, 1967}, {
         3598, 4506, 1626}, {4511, 4118, 1823}, {4506, 4301, 1626}, {4675, 
         4805, 1967}, {4104, 4617, 1290}, {5311, 4873, 1315}, {4539, 3999, 
         1144}, {4105, 4499, 1291}, {4103, 4701, 1290}, {4503, 4325, 1144}, {
         5052, 4703, 1985}, {3999, 4501, 1144}, {4000, 4503, 1144}, {4719, 
         4516, 1825}, {4867, 3795, 2152}, {4499, 4107, 1291}, {4717, 4514, 
         1825}, {4670, 4140, 1315}, {4704, 4035, 1985}, {4903, 4518, 1825}, {
         4301, 4508, 1626}, {4514, 4719, 1825}, {4673, 4798, 1967}, {4518, 
         4717, 1825}, {4678, 4862, 1968}, {5309, 4866, 2152}, {4852, 4853, 
         2141}, {3716, 4795, 1967}, {4507, 4109, 1291}, {3693, 4678, 1968}, {
         4862, 4677, 1968}, {4866, 4867, 2152}, {4868, 5309, 2152}, {4001, 
         4539, 1144}, {3795, 4871, 2152}, {4805, 4671, 1967}, {4871, 4868, 
         2152}, {4793, 4139, 1315}, {4701, 5050, 1290}, {4140, 4793, 1315}, {
         4519, 4001, 1144}, {4298, 4103, 1290}, {4501, 4000, 1144}, {4035, 
         4857, 1985}, {4299, 3598, 1626}, {4325, 4519, 1144}, {4516, 5312, 
         1825}, {5026, 3693, 1968}, {4118, 4512, 1823}, {4854, 4852, 2141}, {
         4703, 4704, 1985}},
         VertexColors->None], 
        PolygonBox[{{4512, 4105, 1291, 1823}, {4677, 4673, 1967, 1968}, {4508,
          4507, 1291, 1626}, {4857, 4104, 1290, 1985}, {4493, 4495, 1626, 
         1291}, {5050, 5052, 1985, 1290}, {4853, 4298, 1290, 2141}, {4671, 
         5026, 1968, 1967}, {4901, 4903, 1825, 1315}, {5312, 5311, 1315, 
         1825}, {4109, 4511, 1823, 1291}, {4617, 4854, 2141, 1290}},
         VertexColors->None]}]}}, {{}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{5312, 4516, 4719, 4514, 4717, 4518, 4903, 4901, 4139, 4793, 
         4140, 4670, 4873, 5311, 5312}],
        RowBox[{"-", 
          FractionBox["1", "100"]}]],
       Annotation[#, 
        Rational[-1, 100], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{4798, 4673, 4677, 4862, 4678, 3693, 5026, 4671, 4805, 4675, 
         4795, 3716, 4798}],
        RowBox[{"-", 
          FractionBox["1", "100"]}]],
       Annotation[#, 
        Rational[-1, 100], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{4035, 4704, 4703, 5052, 5050, 4701, 4103, 4298, 4853, 4852, 
         4854, 4617, 4104, 4857, 4035}],
        RowBox[{"-", 
          FractionBox["1", "100"]}]],
       Annotation[#, 
        Rational[-1, 100], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{4118, 4511, 4109, 4507, 4508, 4301, 4506, 3598, 4299, 4495, 
         4493, 4107, 4499, 4105, 4512, 4118}],
        RowBox[{"-", 
          FractionBox["1", "100"]}]],
       Annotation[#, 
        Rational[-1, 100], "Tooltip"]& ], 
      TagBox[
       TooltipBox[LineBox[{3795, 4867, 4866, 5309, 4868, 4871, 3795}],
        RowBox[{"-", 
          FractionBox["1", "100"]}]],
       Annotation[#, 
        Rational[-1, 100], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{4325, 4503, 4000, 4501, 3999, 4539, 4001, 4519, 4325}],
        RowBox[{"-", 
          FractionBox["1", "100"]}]],
       Annotation[#, 
        Rational[-1, 100], "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[LineBox[CompressedData["
1:eJwl1FmIj1EYgPEz9syYMVuMfSwTYzZFUsjuQraxJkvJPrJfWMKMQtm3bEVp
UiJbyTq2IXu2XNhDcmFLFJMlficXzzzvec973nO+M9//y544u3hWQgihkz+N
00JYUz+EVukhVGWE8APNxMfkuvJdbs4tMF3tDKxKDmFKagirOZ9HZoZQwPPU
zEXPlBDS5NKRijIsRymOW39HzW0sEM9WW2WPH3hivNH4MT/DUzQyPsT5zpWH
dvZMtPayuBJJ4kJrk/mp2lTuYJzCRfwOB5ztLe/kdO7Hk/VcqVee+KO4AufR
A33wBw3UTuVLeq1TtxdJcns4k+8gA7viGImoo742qlDpPN24refuyu14hj33
6VcXBdaWWzPL3BzkiHuhN76be6S2lrqHXGh+KR/kDH3f8734vJyOLHOtrWvE
LTgb493POHRQd0OfUeLrPE3Nbt6JxeLTvITP8DC9hqqv5FfO8BrFxpeMh/EV
fhPPFcfqh6O6OEnuhP4J6r8aV+MJ8Qwojj3j/1LcRo8svs0l3BCLrB0S79ya
V+Jv4i+ogQFYKLeeX/NlHmLtRS6231BMlu+ozyR+qcdIuRHIUXfK+QaKT/IO
dBbv4k1qb/JGvsUl+k2J98SF3Fy/DHEa7ht/4EzO4yKMtq6AE6yt0PMCaorz
7V8n/s/UztNjppr9vEU+Rz6XT8sF8V8u02c8l/I5ubHiWvH3wDf5JyfzxPh+
ml+r1wbUldvGKbyGN2NT3E+fa2pLuFz9VXEvdHGe3/F51P/iVO6rfnt8r81V
8GC5/jyIz3IutuozX317cXd19fTsxkfkDuNofM/j7zG+V+Ix5l5YP8a6+/G5
jO/xQzxAY+Nj8Y7d14r43eBl/FkuN+6lT5b4ubpP8TvALeU26JnN6+N3wlnW
8WNuGu9E3JabpP3/nv0DAlGhjQ==
         "]],
        RowBox[{"-", 
          FractionBox["1", "10"]}]],
       Annotation[#, 
        Rational[-1, 10], "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[LineBox[CompressedData["
1:eJwV1nmQz3Ucx/EvpdFMtBEb6z5y1LLt2rXIMROSe7RYdh27KlfWESnnTMpR
ucK6zWhSzIgKI0z5AysahGI6yS2qaYqZjqHH+4/nvD7v9+v1/ny/39/32K1f
PL5fSbkkSeqXT5IRDybJ4SpJUvhwkhSgkXp51SRZgVLk6T2Lf1OS5D/c4XeX
74leGPVQkpThMGqjCT+bnpPtY64vdthnJ3YhjVcLfeR68nrT3vRupSQ5j1zr
dL0c2oKOQo9qSTKSjqycJBfsWwGDHHswCrDAfrN5A63zsQWbkY5c2Wzza/kT
5CbitL2q8aojFW300vjPO7+96mz+bppDB6Kb+QH0rr0S2RRaxB+B5/Cy3hEc
RVO0lX2HXozztH7c/Bz7N9F7jT6mztc/yL9HfcN177F+Um+zc9iCEv1iv8F4
Op93y3oBvU3foLVk5tE0OpdWt3cN8+nqjeopcV6YivO8DfY5y2uBM8hxDt/Q
UziJ2WZH2/siLmC6mfVmvuadRjt7HtO/SkuRaX4F/we5Zerm8czQZnQ5/Uc2
h9dKZrreNFzWu4QfHeskPYXj+AonUCg7hg6hS+WXoJbslHhGHG8ybWnPbH6q
+pz6J9RwvHz1I7R8PF90sLo2vRLPEK2pzjWXwW9Na6gv8y4hLe6TumacN9bg
xXhm6Eqsxlj1Kpplvo35NPU1s1dRV79AXSfuJ39ovCPqYfTpwDV1jeeI/inT
TqaDPerKjItj4YTee3qbrFu67iz3rZp8ddxndo26DrrEXtgu8xGG81rY5y2z
O8x2te6it9O6xHoX/RjtZe+au4PBjlVR/pbnbj/vpvoG9sXvTHthFT+Ldx3F
+kOQYb+WWMQrUlfAEuxQF9JneAV0aqzjutXd6SZ+W+u51petKzqHK/R++jYt
pV/wVtDD8Z47/mT9I+qj4aG18/iSbuX3439I8+y5Lb4PcTxeD3THQrMzZa5a
X4tjxXsU10K7yf4e77XZ3+is+HagGX+4XlM6zPyjoeom9Fe5ElpP3ZGXGe+G
epx6A22FnPgmqLPpJJkGdIi6If3F/Nh4Z9VteU+Yb0drq2/ybqAxf6i6USiv
Hi1U14/z5s8wM9p6lN4Y2llvXTyP6lL5LfyVtJU6U3+b+/2U6+2Mg9YHMJ+/
UW4BLVMfwj7+3vidzCxFuvk3+d/JfY8leo3kFtO/5VL1KqFyfPfjO6e3HMuw
BxPMHqSHcCDuJ8rwrmzfeD/id/fclVkfQkZ87+g0z/Z0bDO/3d7b6c+843qz
zG6lM2mx7Eh7FNFX5abF99Y6RbYNrULL6Z2VXyR/hk52/JfQSd2RdqSd4m+d
YzSTbUxT7Pmt/ivqqdhvn8/xgvlPZXfH9zd+Q6TKNVffim+07G36QNxT2dHW
R2mJzBFaJDccuer28RzTDvEd9RuUl53jnP9Sv07n22MeZukPMNvYOkt+PM2k
6+U2YC1Oxv2nDXgN0V+dr86jg+I9t99EOgnrnfM6rEUGf5XeMVqkLkad+Lul
vu6Yx+k1ukZmNUp4M5xPnl5/VMAHqCr3Pr2X/0f06Wd6C80sxiLswoj4LsXf
d94nNHFec2XfiP8BzPXG/3P5JQI=
         "]],
        RowBox[{"-", "1"}]],
       Annotation[#, -1, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[LineBox[CompressedData["
1:eJwV1Hloz3Ecx/GPf8hi7KDlP5QhfzhyZfwhJrIchea+RW0zmW2UaHOfc5Wa
ko1ph1iOUfgDuf2BcmTYnNtstYUiWx7fP557vd+vz/v49P3u++u7PGt2ZqcQ
QrI/ObEhDO4VwvQeIfTDZHF/WsaviAvhHB3fM4RdvPHyNPlLeZr8UkII1cjl
zU8MYRPN0b8JGzE1PoTuiEW8s5FqhuhbqmcJFqORf5hfhDnyeZiLInU5vHd6
36MWV9TecYeF9tc6W0RrUN89hHHyBpqqtw2tuKs+1z2+i6+Kj5g5yJyB6Co+
TZ/rf2DmUtpAGyPEJ9XXimtwHRd4zea04KazFbw6O/OQjyb+D9Q42yy/Rhvl
DdHziZ4Njtp5DNednaA36HH6wT0K8VBe6L4PaIH8Hy21dwzvjLjczhI6Vp7C
X4VnvGreBF6V/Jb8Jp7jBV7zXqFFvJquQZLaUXrWRPe3v8Md38jb6Vtaxf/N
r6C/aCX9SEtoHS2lN2iBmX9oM71v7l/xPP07zcmz47O4s9ouWMDrKY8T/1F7
jtarP0s/0WlmTEWpeAfvLN1Js/VU6q2KZiIfw3lP1Y6gfTynreYl27dAns5f
LG8X9+INpdvNGkZ7yzvoPflKdauwXW2rPdtoG012/pg/kK63KwuHcBDpvP30
ADKQiQG8SeoL9Bab+9F9TtE6WuZ8Cv2pJtb5E/EWZ8Xy8/alRO9MXK+/ko6T
l/ET5Qnoihgs4b2P3gE+4B7viPps8zagLXoPyNL/1Tcwy96ZmBE9L16T2hh9
P6L/Tftn8PJpkrybuJnejr5LM5ahXXxR/b/oG6cJzkfzE6M56h/x18nXopw3
Mfp9iJ6F2nhnm6N3L7/srFrNN/FXfEGq2iacd7ZX7Sy1++geZOjfTTPpBH3/
AYcKt9E=
         "]],
        RowBox[{"-", "10"}]],
       Annotation[#, -10, "Tooltip"]& ], 
      TagBox[
       TooltipBox[LineBox[CompressedData["
1:eJwV1HlM13Ucx/GP6ayJBCgYtZLQWil2eIJEyaELN93K1dZhq5BIMxPFaS0Q
M1334VGmOfEuhcqjdEtLcK7WxNQB3WXOAy0VDLXLrR6fP56/1/t4vT+fz/f7
+/x+mSVTxz3VKYRQ5+PE5SG80SOEk3QbzUwK4RN6LT2QHMJ+fJQawr/yhT1D
WIQS3nkpITxKh/PmohteRwK641X+qeZeo+V0A+8HZj5EgTWb1PJpM03Umyvu
EHem22gX2kbb8URaCJNQpXaV/a9Elj3y1RrMVovbrDvJXgPVFqg9KR4jnkIn
4+G4Ds3Qe1bcO3qsMxkJcR95jXglVmAIb576WfXu9A+aSHfQS+11Gfbw3Md7
Px5AvlqS/hrnWYvVSOFZR1/Rfxnp8izrfMHXIf+SnqOlaovF0/XHOncFnYYJ
6uX0Zr1K8U10Fv9MJMfvRb5RXIti+xSa7ca/StwfS8Qj1HbyXc+fIO9Dr0Nf
3KKe5NzJaPd+P5e3infRKvOzcV5+AR2JIfS0z2i+ndiBE+rFdDvdEN95XIen
DXeqf2uPU3Q9SpzjDrqL57O4F8YhjTcVL/DWWuNpez6DOr4R+vW0njZgt7gB
/zjLbnk7/2yz1XE/Z260RjGdqXdGPIu28DfjbvX5fGfNvKc3R/wcHoxnj9+x
+vu0g26ko+O91P9KvkQ+XvwQUtTHm5mD3+Pz4TfMsNdpupn/Hr0t9EX5VvoS
bTV3BU1HpXWqMDLeQ/ko2mS+GsM8ezZysMc6x5HDOxzZuA25mM5bgT7ea6H5
Fv4i2lfeHH/DWI530YrB8V5jEIZhKMrNT0OmmQKzefY6Jx8UfyPx90Qz6Hzc
xTOPPsI3xHpDkYG9qNDLj+9d/zEzZThv/b1qF2gFDXq9+fLEz4t/Uv8ZS81f
Iu9hphP9WH4NvVX+lvhtLMY78T5jlfnDzrmS/mj+B6xx5k/jfXAvuqpf1G+T
bxIfE2+mt9u7Tu17/u8wN/432KMp7of/eBbR4ziGkbwTzIyiRbwFvIW0kGbx
5lp3AE3lKY1eNKrVqGWrrY7v0ToDsRVb8JfeAvo3XUj3O8cB/GLuUHwmM1ny
GjqAnvI8XXhPx/uMozxH8KveoTiHi/F/AF+L96HSGSfK91k/HX+af5MexRG0
6H2D9bzrsDbeKTMHrX91/E/Xy8VE9cdRFn/P6jeq9cMN6MXf1TnOqDeI7/W8
9bTIGsvU6uP3rXbY+YP4IGrjXTczBY28CfppajnyXvFOWbe/+uD4fuM9QinK
9PrR7db+H3XC9II=
         "]],
        RowBox[{"-", "10"}]],
       Annotation[#, -10, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{3551, 5260, 3550, 4930, 4929, 5246, 5245, 4291, 3994, 5458, 
         3993, 4030, 3557, 5378, 3995, 5250, 4942, 5115, 5114, 5249, 3585, 
         3556, 5377, 3778, 3619, 5391, 3827, 4173, 4056, 5390, 3618, 4469, 
         3617, 5389, 4055, 4172, 4054, 3616, 4467, 4468, 4053, 5280, 5282, 
         5281, 3562, 4596, 3555, 4593, 4592, 4624, 4623, 3607, 4963, 4964, 
         5248, 5247, 4931, 4932, 4417, 4928, 4927, 4926, 5337, 5336, 3992, 
         4531, 4532, 4271, 4278, 3549, 4533, 4534}],
        RowBox[{"-", "100"}]],
       Annotation[#, -100, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{3560, 4933, 4595, 3779, 4746, 3797, 4602, 4603, 4171, 5476, 
         4170, 5279, 4052, 4969, 4970, 3613, 4440, 4834, 4737, 4738, 4736, 
         3781, 5135, 3834, 4980, 4981, 4213, 5413, 3662, 4560, 4561, 4310, 
         5384, 4309, 3589, 3567, 5123, 5124, 5122, 3811, 5125, 3663, 3852, 
         3798, 5116, 3799, 3569, 5380, 3786, 3645, 3853, 5443, 3854, 3844, 
         5139, 3843, 5140, 4338, 4369, 3574, 5381, 4010, 4432, 4431, 3664, 
         4087, 3803, 5119, 3804, 5120, 4748, 5111, 3788, 5441, 4641, 4642, 
         3654, 3855, 5444, 3856, 3848, 5145, 3847, 5146, 4341, 4370, 3579, 
         5383, 4021, 4020, 4437, 3666, 4088, 3806, 4742, 3790, 4741, 3789, 
         5147, 5148, 4995, 4996, 4216, 5414, 3668, 4566, 3667, 4313, 4940, 
         4941, 3580, 5488, 4609, 4599, 4644, 4643, 3658, 4887, 4218, 4886, 
         4219, 4888, 4217, 5501, 4315, 5490, 4314, 3594, 3581, 5489, 4610, 
         4600, 4646, 4645, 3660, 4647, 4220, 3670, 4567, 3669, 4316, 5385, 
         4946, 3807, 5121, 3792, 4743, 4745, 4744, 4767, 3582, 5150, 5151, 
         5149, 5153, 5152, 3791, 5112, 5113, 4766, 4765, 4847, 5294, 3595, 
         4296, 4274, 4482, 4481, 4028, 4445, 3659, 5412, 4084, 4024, 5258, 
         4025, 5259, 4023, 4187, 5478, 4186, 4439, 4184, 5500, 4185, 4022, 
         4183, 5477, 4182, 4438, 4180, 5499, 4181, 4016, 4435, 3577, 5382, 
         4179, 4015, 5339, 5340, 4938, 4939, 3576, 4433, 3575, 3593, 4608, 
         4607, 3805, 4008, 5459, 4007, 4427, 4006, 5338, 4177, 4936, 4937, 
         3572, 4424, 3570, 3591, 4606, 4605, 3800, 5507, 4747, 4935, 5252, 
         4003, 4732, 5090, 5091, 4528, 4529, 4267, 4913, 4266, 5373, 4337, 
         4415, 3565}],
        RowBox[{"-", "100"}]],
       Annotation[#, -100, "Tooltip"]& ]}, {}, {}}}],
  AspectRatio->1,
  Epilog->{
    ArrowBox[{{-1.2, 1}, {-0.8377335312327795, 
      0.1383578663016639}, {-0.38546788559588496`, 
      0.05641791466247163}, {-0.3069314914683546, -0.08277877382185601}, \
{-0.11656763796105155`, 0.021647383890019883`}, {0.07862586389284601, 
      0.031835306905749886`}, {0.2718555676826467, -0.036128088682171475`}, {
      0.4584963880522191, -0.17473278087913358`}, {
      0.6357975829441427, -0.37220556411385985`}, {
      0.8027271046850972, -0.6161274459842689}, {
      0.9594205771014539, -0.8958511169360397}, {1., -0.9983533104372209}, {
      1., -1.}}], 
    PointBox[{{-1.2, 1}, {-0.8377335312327795, 
      0.1383578663016639}, {-0.38546788559588496`, 
      0.05641791466247163}, {-0.3069314914683546, -0.08277877382185601}, \
{-0.11656763796105155`, 0.021647383890019883`}, {0.07862586389284601, 
      0.031835306905749886`}, {0.2718555676826467, -0.036128088682171475`}, {
      0.4584963880522191, -0.17473278087913358`}, {
      0.6357975829441427, -0.37220556411385985`}, {
      0.8027271046850972, -0.6161274459842689}, {
      0.9594205771014539, -0.8958511169360397}, {1., -0.9983533104372209}, {
      1., -1.}}]},
  Frame->True,
  PlotRange->{{-1.3, 1.5}, {-1.5, 1.4}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 169},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->539862624]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "WorkingPrecision",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->26397],

Cell[TextData[{
 "Set the working precision to ",
 Cell[BoxData[
  FormBox["20", TraditionalForm]], "InlineMath"],
 ", by default AccuracyGoal and PrecisionGoal is set to ",
 Cell[BoxData[
  FormBox[
   FractionBox["20", "2"], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->549268746],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMaximum", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Cos", "[", 
     RowBox[{
      RowBox[{"x", "^", "2"}], "-", 
      RowBox[{"3", " ", "y"}]}], "]"}], "+", 
    RowBox[{"Sin", "[", 
     RowBox[{
      RowBox[{"x", "^", "2"}], "+", 
      RowBox[{"y", "^", "2"}]}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<Newton\>\""}], ",", 
   RowBox[{"WorkingPrecision", "\[Rule]", "20"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1164918439],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.99999999999999999999999999992196252681`20.", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "1.16792859722225029061295112602695264056`20."}],
      ",", 
     RowBox[{
     "y", "\[Rule]", "0.4546857360698436846282762264969351912`20."}]}], 
    "}"}]}], "}"}]], "Output",
 ImageSize->{549, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->524230119]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->21468],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FindMaximum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindMaximum"]], "InlineFormula"],
 " tries to find a local maximum, ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["NMaximize",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NMaximize"], " "}]], "InlineFormula"],
 "attempts to find a global maximum:"
}], "ExampleText",
 CellID->1650805990],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMaximum", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     FormBox[
      RowBox[{
       RowBox[{"100", "/", 
        RowBox[{"(", 
         RowBox[{
          SuperscriptBox[
           RowBox[{"(", 
            RowBox[{"x", "-", "1"}], ")"}], "2"], "+", 
          SuperscriptBox[
           RowBox[{"(", 
            RowBox[{"y", "-", "1"}], ")"}], "2"], "+", "1"}], ")"}]}], "+", 
       RowBox[{"200", "/", 
        RowBox[{"(", 
         RowBox[{
          SuperscriptBox[
           RowBox[{"(", 
            RowBox[{"x", "+", "1"}], ")"}], "2"], "+", 
          SuperscriptBox[
           RowBox[{"(", 
            RowBox[{"y", "+", "2"}], ")"}], "2"], "+", "1"}], ")"}]}]}],
      TraditionalForm], ",", " ", 
     FormBox[
      RowBox[{
       RowBox[{
        SuperscriptBox["x", "2"], "+", 
        SuperscriptBox["y", "2"]}], ">", "3"}],
      TraditionalForm]}], " ", "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"x", ",", "2"}], "}"}], ",", " ", "y"}], "}"}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2092335573],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"103.06261052078153`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "1.2303719117773388`"}], ",", 
     RowBox[{"y", "\[Rule]", "1.2190918583675474`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{248, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->487542325]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMaximize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     FormBox[
      RowBox[{" ", 
       RowBox[{
        RowBox[{"100", "/", 
         RowBox[{"(", 
          RowBox[{
           SuperscriptBox[
            RowBox[{"(", 
             RowBox[{"x", "-", "1"}], ")"}], "2"], "+", 
           SuperscriptBox[
            RowBox[{"(", 
             RowBox[{"y", "-", "1"}], ")"}], "2"], "+", "1"}], ")"}]}], "+", 
        RowBox[{"200", "/", 
         RowBox[{"(", 
          RowBox[{
           SuperscriptBox[
            RowBox[{"(", 
             RowBox[{"x", "+", "1"}], ")"}], "2"], "+", 
           SuperscriptBox[
            RowBox[{"(", 
             RowBox[{"y", "+", "2"}], ")"}], "2"], "+", "1"}], ")"}]}]}]}],
      TraditionalForm], ",", " ", 
     FormBox[
      RowBox[{
       RowBox[{
        SuperscriptBox["x", "2"], "+", 
        SuperscriptBox["y", "2"]}], ">", "3"}],
      TraditionalForm]}], " ", "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", " ", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->827788029],

Cell[BoxData[
 RowBox[{
  RowBox[{"NMaximize", "::", "\<\"incst\"\>"}], ":", 
  " ", "\<\"\\!\\(NMaximize\\) was unable to generate any initial points \
satisfying the inequality constraints \\!\\({\\(\\(3 - x\\^2 - y\\^2\\)\\) \
\[LessEqual] 0}\\). The initial region specified may not contain any feasible \
points. Changing the initial region or specifying explicit initial points may \
provide a better solution. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NMaximize/incst\\\", ButtonNote -> \
\\\"NMaximize::incst\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->31348708],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"207.15989698200875`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "0.9948613347456736`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "1.9922920020949584`"}]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{263, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->2905743]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ContourPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"100", "/", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{
        RowBox[{"(", 
         RowBox[{"(", 
          RowBox[{"x", "-", "1"}], ")"}], ")"}], "^", "2"}], "+", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"(", 
          RowBox[{"y", "-", "1"}], ")"}], ")"}], "^", "2"}], "+", "1"}], 
      ")"}]}], "+", 
    RowBox[{"200", "/", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{
        RowBox[{"(", 
         RowBox[{"(", 
          RowBox[{"x", "+", "1"}], ")"}], ")"}], "^", "2"}], "+", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"(", 
          RowBox[{"y", "+", "2"}], ")"}], ")"}], "^", "2"}], "+", "1"}], 
      ")"}]}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "3"}], ",", "2"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "3"}], ",", "2"}], "}"}], ",", 
   RowBox[{"RegionFunction", "\[Rule]", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"#1", "^", "2"}], "+", 
        RowBox[{"#2", "^", "2"}]}], ">", "3"}], "&"}], ")"}]}], ",", 
   RowBox[{"Contours", "\[Rule]", "10"}], ",", 
   RowBox[{"Epilog", "\[Rule]", 
    RowBox[{"(", 
     RowBox[{"{", 
      RowBox[{"Red", ",", 
       RowBox[{"PointSize", "[", ".02", "]"}], ",", 
       RowBox[{"Text", "[", 
        RowBox[{"\"\<global maximum\>\"", ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"-", ".995"}], ",", 
           RowBox[{"-", "2.092"}]}], "}"}]}], "]"}], ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", ".995"}], ",", 
          RowBox[{"-", "1.992"}]}], "}"}], "]"}], ",", 
       RowBox[{"Text", "[", 
        RowBox[{"\"\<local maximum\>\"", ",", 
         RowBox[{"{", 
          RowBox[{"0.5304", ",", "1.2191"}], "}"}]}], "]"}], ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{"1.2304", ",", "1.2191"}], "}"}], "]"}]}], "}"}], ")"}]}], 
   ",", 
   RowBox[{"ContourLabels", "\[Rule]", "True"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->200111634],

Cell[BoxData[
 GraphicsBox[GraphicsComplexBox[CompressedData["
1:eJyEfQVclMn//+7CLuqZ2N16nl18zXuwu8/kzu5ODE6xFbvPTkwO9ezCeWwx
zm5FUJAQpJcNdvntPg/vGWY4/n9fvnz87Mwzz8Sn5zOfqTh8Su9ROo1G07qo
RuN8qn9yyXh22lTF8VdPYQXspKPwhUkfHH/tBHBHpYKZwu+VCskUfqvA3yls
znD++UzhpUr7Nykco9R/KAHeppR/ofB05+cvxFG4utLhFApvVepbKJw5QM+c
xik+MV7AGC9gjBcwxgsY4wWM8QLGeAFjvIAxXsAYL2CMFzDGy6+bxjOndRTH
KT4xXsAYL2CMFzDGCxjjBYzxAsZ4AWO8gDFewBgvYIyXx0uNZ054Kq6jOE7x
ifECxngBY7yAMV7AGC9gjBcwxgsY4wWM8QLGeHm603jmRIcinorrKI5TfGK8
gDFewOgvYPQXMPrL8wWNZ058QqQjEc/EdRD7ie8Dxvd5PqTxzIkviXQr4rW4
7uK84Hs8n9N45sT3RL4g0o2IV87x+c21SBUcz+ChobT9nPioyGdEOhTxFO3l
xIdFPiXSsYjnaC8nPi7yOZEPiHTinN+2NSwSLwc0njnJBZFvinxFpDsRr7Ge
vJzReOYkd0S+LPItka5FuhHxWHyiP7yc03jmJPdEuSDyTZGviHQr0pGI58C7
qY5fvVayeRL7CVh8ov85yWlRjol8XuSDIp8R6V6kS55uRLwX8VbEMxEPxHUQ
xwW9gz3HK+9HEKUbGVZSTnn/JYUfKe8flAD3VMb3jsKuSjNRFN6o1DfSZ2Wl
vQSFzrt5qL+P9GC/T1W+nyihfogy3jQJ5Wg3p352Vvpjk1AOOKdxoDyncaE8
p3Gin+g3+gsY/RZ/Byz2U/yu8p1dcbS+OE5xHMAPwMAn9Bv4BBj1AQOf/nOc
VhOVa+K64ne8h+8C38Rx5tQvlOfUT3GcYr+dj2CLifYb3xX5g/gdwGcV+v1C
lJ83WTLlUQp94nfU66a8FyHhdxfHV+W3JumckwrXRGXihZW2C/iL0uAL+h7a
wXtoJ6f3ADv5i/shC6npbGd7CI83jvFj3ICvKO2ZpKvKM4b+ntN7zq+sLZtE
JjuqW00p9Inf0Y81Sv33CuzjbZWGKr8/Jl0cv/Y+aJUaOov9ZfodrBfev670
J4TCoAP0E/3O+h1nPXwH8F1leQNINWe1s+y76AfW3zmOCSstZKZTTuwIUcb5
+KCFwyPnuvB6XgqZ5niULmDJti5o1/nd3AutFD9FvgUYcmG98qEfZKLSThrl
I6gPGPW3KPUTJNQHXwAdgt8D7qq8lyyh/6iPcak2tlGyK99LpHSAfgIPAGMd
0G9xXIAh1zCPFZVnFP0+nugvYPQX/cJ6oF8inuAJ+hHbRTtoF/1CP9Ev9BPz
jPFgnsXxiusCGPWxTqIcE9cJ64ByrAP6jX7ktC4YF76L9RHbwRN4IOIFyjFO
tINx4fvQLwCDPrEO0CNAv1WV/trpuonriPXA7/g+2sO8YN4BY97Rb4wf7WCe
AWOe0S7mB+PDfGLdxPGAL+KJ3/E+YBWPDDKek5VyFwqr862hMMYFGP0W66Md
1AeM+oAxXpHO8Lv4xHfU7jP/E35Xqjn0W74/dto/4CNgyFsRRrvQX1AOGP2H
PoNywCgHXqEcsDh+flxsXdBfcf4wz+gvyrPKCyeMctRH/wGjv4DRP3HdAef0
xDrjmdPvWfVpZznGI/JhlPP17YQfj53w47ETfjx2wo+H4YGIlznhaU79FPVJ
UT6q9Guj/Bow2hfnQeSbIp0CRv9RD3jKwxpPkW7wO/g4+Cmeop2J74Gvoh7K
RXrF7xgvjwcpVC6Jejf4GuQZ5g8wT68WWh/tiXIP/RblCH5X+WVqZr+TJFHu
pyvtR2f6eZi8xfry/IL1D+vAt2eg9CjyD+XhsMuBr4CBr4DFdcb8iXgKvofv
i/SPcnwfML4PGN8HjO/+N93bhXUW/bsagY6Yf1WkN96fqqEw6ol+QpGexO8C
FvUAkc+q+J0s8U8TfUJvxvcwn4DbK+1HZuJfeja9C/iK+sAXvr14+j7wEniK
9tVxmQV81MiAMb+or5az9RD5eE5yUv3dTcYT/QcM/Uish/6A3tV2tRQWy0X+
iPqYP9QHjP7BfsL76D/w7JxSL5aOH/QMGP2H3iSOC/ohYPGJ9cP8ie+hXfwO
PAA9Yxwi/aIceI5y0Av6Cfsf5ZiPj0rDrtnoCPiK9nm/uEbm/QdaAZ80Ms+f
2foAFukUfBj8kffzi+upkbE+ysOXfQ/1AWPeeHtUI29Q2kuk6wt6Aoz3RPkN
uuDxjb0P/AKfwHry+G0R6IvJJ15exFN5gu+JeMfz2ZTM77rQ8QPG+4CxXoDx
vrh+gLE+mD9eLttovwCDnsCfsF4iLL6PecP7GD/mC/XRP6wL2lPXOYPCIn/C
fPHzw/zeqAd8Af5jvkAvmE/AwFfAaBfzxa+3Vphvu+BH0FL6wvuAUZ/fR9JQ
ekZ9wKgPfvff+4nMjw8+hP6LfErEa8C8f5/hG68PaDxFfoffsa6YP+CXCIvz
g3LwK4wf8gz8AfigvOahlXn8YHo0L/9sVA9V6/9Q+UEw+53nX0lU3oK/AH95
/s/oWdTPMH8YL9aPr2ej6yfqD2gf30c5+ofx8fzGTuuj/Q7O3/0T6Xjxvvg9
yBPQ60Xn04vxF5RjvlEOfBT9BsAX0CHWW+lXXALtD/qP9cf8ijD6i/eV391t
lK7xXTzxuyhHgScijPrAA57uzdn0DJRDnwYMekZ96Ncoh58V5dC3UY7vY/6A
n4DxPsaFcjE+QbRLxX0PES95O0bjKfp9eTmj8cS4wD9EPxvsSeCXuI8n+nnE
fRfU4+eH2a/ie6IfCfXV8UHv0gt6KvAG+yZ6gZ/rBX1FL/Bvhl+8HQP8cqHt
A1bLU1S89ddRegMMfo/68N+hfYxP/C7wGN9DPwDjiXni7SpRbzdk8xOJ/lHR
fwB7CeWi30D8HTD2R8CvDirPAArnV8YvS4CxnwQYflnAPF6nUH0f7X5SnumS
m9LuR9ofyDHwI/F3wKADtIv+iHQg2o+Yb4wXMNYbMPoJGOMHjO8BRn9FfxTG
j/VGu5gf9BP1nftNz+ck0H0s+JPXKeXhVB45x5Wvrrr/FbfmFR039jMw7p+d
vy+Kor+jXmsF/kTbcwry2m5PFXz+PsFK16eWUuEy9X+L+ypoF/XV74ZKp5zt
pT5T8PjyACtdJ9HPJNrloHO0p87PV4mPY1L1A58UI6nhAP3yxdN2xX1/zAP0
f2U+NpkpfqBf+B6/T5SYjU5Efz5+R33AkJui3AUs6unABxEW+SPaxXr8t3/N
KvTLmu133h5h/BTf5+1PPeWHKAf9oBww+CXoBfWB96iPcowH5Xgf9IX6gFGO
9VTG4+5Kv495Rn8Boxx8Ft/7b/6spTDWiR+vnfDjsWfTM0W7Du3y/N1C+QC/
f2PJZnfnhGeifzmrXDizMUn6Lz9zUu4USdyngzxUx2ui+mlWenPCKOf1W1Zf
7a9R0M/dhLhD0f+hp+MU/TGYN7wv6sNq+zq6voB5fZDpnWgP/BT6OuwbrDfv
x9RTfUEBPYAfqURlI1pajvYAQ1++pLxok/Dk/Xk6mdf7XGRnPMDAkCSFrgt2
ZfvyIj8Q/U+8/m/J5m8AzO/f6CkMeuT1K41A72gviY4f3+PttzQKA595fTiF
4hfmU23fLMgFhl9Yf+Af9DXwA8CwjwDz+M3sHXyf3wfS0/fBX4Av8IcABv3z
/gENH4fr+D7kGmCskwJXYesJGHyClxfJFEY51ounBzfBH5VLoA83ur74Ht9/
BqMc/Frk3yiHPFTHYaAwysV9M8hpEUZ9zA/oA+PHeDEeEV9BP7w801F+j/q8
PNPR/mL90B/AvH7P4sxFecGXp1D6QP8Boz94H/OD7/GwneqPKOdh1h/eH8/i
UlBftKPFfRBRfol+VVF+ifHu4j65uO8NGN/5/+07ivt4YpyAGC8u7rOJ+9Q5
7cOjPXHfOqc4AzGuAN8T97kBQ96Kdgn0Nl4eW6l9I8Yp8fElLtR+FukGv6Me
9HaUQ89FOc93NDnY6XY6b/gdT6wDYHXdwii/5OPqNZR/oD4fJ6+h/UM5+gd5
gv4BTzD/vN9IlINsn1TU+/j9B3beRIyrEeOwUS7uS2A9oX/BTws/CcoBA89E
P2nW/bRPfVX/om5farZ993cO8PVro+If8hsZny2uTMWzRAn6Gb4j7quK+3pY
B+e/3iSVXHbAtWuz+GTnuvl7GCXIY/W7Rk7vdOqhTn76ukcq1TcVfWtRuPId
D2IW6NKcLU4B8yDuU4h2Pa/3p9In+DH/u4Xw/MJKsuqFSdfVfpVeHK7Gm88y
KbBXgRi6Pjx/TKV6Sqjye7QEe7OV49H7xCfVHj5rIby+mKKs7x8hSYpeXaQr
+x3fgX6D+cF8A+b3y0x0vwgwHw9jon4HzBv8K5gXwHy8XKKE9/Ee75cz030i
yE/RPhX5HuqBv+UU56v4icZ/UN5b+B7xiya6P4/+4n30F/o2+CnKIVfQPuZL
fY/ZybxdrhP861rKX4Gf0C9QzvMpLfWL83Sul3m5bRD2ew3Cfq+B6rvqMhoE
uedG5Rr6w8cnGyh/RX3AqA/8gv4LvqFUc9g/vB7P9odQH3SJ+QJdoZz3L7sI
ctyFyg/o+9D3RL0C84rvo398vAaTN6jPx9Fr6fjF8v/2U2uFc6VMDvLrrM32
O9aZ169s2fRI8fec3hOfeJ/fZ7BRuxz2KGDwPX6/z054/4s9G55incTvi/v7
4F+gI9HfARjrKp5XFOPzUA4/BvwaoGNRjxLj4/A78APj5/Y3s8R3oxz8mJef
ydnK8T7oE+sBegLs7Md3k+qHmV82ifIfkY/y7cWo3/GHnIoW4PBscfOgd94/
xmDoOfw+g15+pfznjTq+YFd5jPL9ZxTm9WTmnwP/gN4LGHiAdca6i34zrDu/
v8f2Q8E/AIv8RRmGw77GegKGvYX2+XhhRu8o5/m3uH9tF/xcGRQG/eB90e8i
xnn8v+N2tFQOOP1BO7YnS3iCX/L712H0+4BB75gPZ3tT6qj6wtXG31R9ciTi
F75l248Hvoh8jff/GKg/TOmWr4bCmB98X62vpfjExwfp6fxAX4a8hj4Fea7K
s1gKq/pDPIUxL8Bv0W7IiY+JcZAifqIc+AQY+83A9yTle/covfDxp1oq3zAf
gCG/eDvdItjlZqrHgf+IfIjno250Pfh4gdRs/kzgKdYf5aLdwPvl0qm+if6q
6x2bOT5mX2E9uHisYFeZ189Zf3j8SMmG38A33h/P7HrIS+Ab8BEwT98muh5Y
Lz4eKYz2F/wG3+fOlTpg3r+skRH/gfnAeLEegIFvmC/wO8yn8vBzofxN9Iej
PuQHry/F0PdRH+9DnqAcdILxgw9h/bj9gWCmLwKG3EZ/sH54n99vdxP8tW5C
HgE3mY/DZ+fUwZ/QX/BvzDtgjBff5+LLg5k8AczRjwPm9ksdMK8/M/8hYOgb
gNE/4JvoVwF+AebjV7SC/9VO7XLAfB4AjXAuXvTnmIU4CDPtD2DeH2qn50xB
/7AnRDsG6wxYjBsFH4N9JcZp5RRXKcatinHX6LcYZynG/Yv5JER/HvxU2MeG
PcjHM5sFOcLOh4l+HsDwA0Buob54ngMw6iMeHu/hd9GPhN/xBP8Bf+b9rulU
3isPr3TKD1EO/gu+D7sbdrgTv7fZLDROgY+v0ngCX7HOmFfAKBfP9UFOY/4h
14B/KMeTP09goXoD1gkwyoHXiCtA++D/4jxD73DOQ/muKaSS4/dxIez8Mh+/
Eqri82jmFwdf5+0Ys3D+wET9dvDj8ecR0mi8pBpvZ6L+GdAf/G6i/w34gfkF
PsP/gnp4jz+Xk0bh/z5XlX0fWPRD8eeGtII8MVL5kBOfBIzvqMttoPJc9E9A
/kI+8nZlGK0PmC//pvY7UCPEcWkFewB6fHY6wfry+pIpW5wz2skprh/4Jfq1
AEPPhZ8L+IpzK1hn+Mn+20/3I1O/sxDuPJJDvsEeAp8EX4e+APwGzJ1vcNhb
/Hwwe0jcN0M7mH/Ek6IceCHq+eI5YpTnZCegH6I+j/FhHTF+zj/igLnzXcFs
vxX4BlidT22mPpHpX8jNYOAXYOAB8ATrw+M/03v/285kerJ4/gb0wr/Hvof3
ePvCTs4r65pG7UEuniOLviTiNcqhX/H+0FQ6LtAv5g/94v0FduE8jV2wF+yC
PyCS6q/8eetcMuSHik8sHgnrjzg3tA/5Cj4J+gI/xHqjHsYLusP3+LxLXyld
8vHyZiL6SSAfIU+BJ6JeBhhyAN+Ffo55B38Xz72DX0NfEOmZtxe1shhXBznB
x/eYqf0H/qKuc2a8Qm0Xio8YNxd/Eyj6RVOzxW2IcTrgx/y+ijkbvsP+4/fP
MogYJ8efv7XS9cf6iXwR9bAeqAd+xLefSMvBrzH/GJeKH+GUH6K+qJfy+7ga
T/Bx6FnYDwO/5PfHLEJekCTKbzh7t5OVg5d2t1C5ltM+E+YL8keMixLtetAD
+gV7FfgDPQv94e0vK30P+iDKc8ojgXlE3KeY94PfV0rN9jvwC/jL+WOCXXl/
pgOGvsqfm2bnSES84/U05q8W/WnAb8yP8rmnLB8NH+8dSuUixiHqfRgnxgP+
KuoxkNfKa72Z/wP94/SpuAyqx4FucjrXhXI+j55btn1+Lg9gMDtvKe4bo9/8
eWUTlUPiOUcxr5p4jpCLC3PMq5hPjqdTE6VLzIsCLdTz/p6FzB8CGHoKnnhf
efhDTzZTGPV4v4KZrgevV5iz9QP+fuAl9G3QbyulnXOUXjj/W5Y8WqBH4LW4
DyXmCYO8Al6qfOMbfeJ35/wELVDjS3scSqHx4WgPdI/6/H6gge5niDDkG2Do
I3hCTvH6k1nIo2Pi49P9BHvCMW7IP+X9kVoZsKhfYT15fxCTX5g3PLF+0NdA
b/DnqdjnKldSGo5Tx7vLldox/31emck10Cef38BNOB+TS+b1lVxCfFEuId49
e5wO/30WXybuE4rnp0S/NOYReAC8AH8Vzy3B3wI6Bd3y8eYs/yHoHnoJ8Al6
FOgK8lykZ/5cgJXiH/RhkZ6hd4FugZeA0Q7GC30C+Aw5C/7Pn8c0UfxR8p65
p2eO+5NUS/nu00x6zpD4eEorr79nyx+SSuNFeLzJEPQMM+UT4DfQJzEe8bwF
v79jovMLPACMdgGjfcDg44DBLwDD3yvCWB/AKnppKazig5bu/6Ac6yHyY9QH
jPVBPdEu/G89NomuP5c/0j3T/+TP4vHV/hh4OZdlfKBP6J8qvrH8FeJ8g49A
/v+nPwZ5eoLRHyvVb5X+e2RQGPwLMOQ0H0/L9hvV9XAT4msNVK7x+yeR2fYb
UJ9bfwe/4teX+ZswfshDwJBj4J/gK+CXfB41Nt/oH+xPwLzfUiuDLwHGfKps
Vyv/rSzwYopvaA/4xcdzaSiM+c2aF5KdW0nPhn+Yd/QDfIGPL2HnRvi8A27Z
zuGL54jEOG1errBzQZC7oj0tnhsDjPbF+FcxXhb4zOtRGiFei8W3oh6/P6MR
zi+xeFbA+J4oBwFjfnk5z/J4cXpFln1tMR4VsLgPAX6EdoDfYvwpT+cp2fg7
5DL4a9Z4P+e5FZxjAT2r+MjOwfLnKFJ4vTE4mepN4GeAVb6RQNvDeXTAKBf9
5Zgv2Ds8HjH/OfRK8Vyj6JcF3wX+ifHZYrwgYNAp9AfwfcCgP57fMv837GyR
P4Fu+PyyBhn9B//DOgKvsA7oH+gU+gnPHxmMceF34Df6j3UFHxHlBsYB/x3k
A/QprCf0V8BcPiFHfZWfvqD67Q6l3dcq/w1m9ivGg35AnqN9RS8Md+Ht2+c6
Gfo15BVgyEfwWd7eDKP0JPqpsd7oD+YL/eH2SbPIBzEeAfMPeSLqi6Az9Asw
H5fE8pcAn9TxpFL/Nbcv68BHlGP8fDyPRub1T7YPK+6/YDzgD8pnemuofIO/
EPIcMPAFMLef7yfk68lCP8AHXu/PxZ+XWqin8RLAJ/gzAXPnER0w5I1It6Az
dZ401M4HHwZe8HyG7f/+934w7M1v1F6/rOCRNVNvfy/9qfx+R7CrwyR1fS10
3eGfhH4I+gL9gX+q7RslPBU7yj1NyY+8yT9VAj0o4/ZEfG8oheHHBN7w9GMm
vF8qge5XoT+or/bbTutj/rh4Qvg1HHov1gd2tion2PdBh1y+lV2u1P4HvqAe
8BXrDXrZqbx/nb4P/MG6YJ3wnvr9JNqfCsr8f6X4i+9xcf5e6VT+Yf1AXxiP
6GcS5SreV/H5C+Wfit0VZ5ZclPdMFF9gF3L+aLSvyX5+EXoe7HnA6C/fLzOl
Rz4+002Ix0SeKRchvieFvs/F4wa70vaBL+CfkLv8/JmpfcH/Lu4j2LPZX/z+
fga1k8S4KbQrxvmK96mI53HAl/C7mPdD9GsAhhzh6KFK9vnH+vH3RbhQ+xXt
YB6xzqIdq76HfaoUau/z88rOeYlxL6gvxrHgd8SXiPHs4K/4XYxzEfNmwK/I
2atZ8mGIdAp9kff/ZY8D5/1USdQOwTwCxjyD3sGPQZfgj85+Bjf+pqxr7pFq
vE/4Gittj493D6V0qs5rKI2rQrsob66U3xHiguKJGPeBdsC/4FfCumBeQJ9o
D/yQj2OGPqGV+XhAlp8H7YFu+Py7bF8Ov2M+IQf487QGag+r32XnOtE+f/7D
QPdP+PjJcKoPAj+grwGGvsyfI9VS/wDGje/w+ll2uxL0hflAfcCi3Yf60Ld4
OjUIfldrDnTD9nXE87u8HNZSPyfsbthLXByeP/NLc/sxcQkU3wGL/Qd/Agz+
hPXk9Tkj5eegP8Dc+w7+hf6DXwHm943SKT/j95OZnxXzBX7Mx9d/ofu1GD/w
D98T+4P+oh3IL3wP48V68/yc5S9Fv/l4YrMQr2Oi9SEPUQ5YbA94yZ3XWph5
ztuhF6rjEO87YfFtvD4dL+yX2vj4God+Bv0J9Af9BnoQ3kc8Nerz8RNsfxF0
wMcDu1H6Aww8hBzG/Kvrxcp5Ok6l+AUY+gnmA+WiH4u/j4nly+HyoPumUP2Y
t69Tsn0H5aL/R4zDBB+E/AU/xXiBr3w+WgP1D6CcP69goP1VqjnsZcgp4Bfw
l18HK9UjoAegHvg95h3rhfFCPxfPDcDPBnkF+QC5C7wDvoPuwK9BV1gfWl8p
f0r1ePHeB8hFxKfx8SKJtB6/v5+Y43vwi4r9E2HMF39OPZXaR+CbfP5lM42v
UOnOpOgj0eXU/LGGcyYC+Yb2MV+AvZ3gt+vKurQ8bKXzg3LoG7Aj8D3wf+48
QidWX8y/Je73QL/Ae+AffDwki5vA73x+kFzCeTE34V5BN0H/dRPuCXSTYV+j
HOuB/vDx0A7+mjk/4Afwl4D/4/ugD8w36IObf19GP3gfMPAX9QGjHk8vqXT+
+X06q+Dnt1B8EvUuMf8Ufof/BvjF2Z+4f8Kd7QfBHgbM51c10/poD/UBgx9B
3nDniRfqhf10th8o5hMC/4XdJvoLwWf4/SOWr1vUP7D+sPOVn0vrZK3SL5uE
J8YL/OD3UXT8+Zss8bXgv7xfVU/xgY+vZnn+gX8YH/bzAfPygOEX8A/6AL7P
x1ewfGuAsX4YP9rj9J4s59iAd+J3gZf8/RCMnvA9/nyXhvpjlPF763h/RZbz
XIC5/W4HDPpWHlX0vD9yEnsf64/3AXPnZx395eMuWBzGf59DZPmHxPN3GC/v
32XrpY6f7e/x9kZ2PVfUg/lzqmKcsIb3w2SJ+8H8ijB/jkbcF2Pnmvh8sy7U
PgP+V8nUS9AO8JD3S7B9POAPysGfUY71RTmePF5m0LgP0BufF5rtg4lxF/y8
sXPifNwni1MDXwPeOu2LL43UeMt86TE07pezx33NXH3n/gz4MfAE8hP9we/w
Hyqv5WbnUTHf3D2/WeKTsD6gd/APPr+ylso/fA/lVN5lllN5l1Wf8GXrC3oC
vqB9vIdywDz/YucCxXqAefmqo/SGctAHr3/ac5CzbF8I/eXpxi74Ee20XdT/
b/pLp/ERsJt4e8lO50/cVxTPDXBxVQ55DTzgz9tYhPPbBkHeGfj8nw55a1H6
80rlY0ZXar9DPuH7kE+AxXzaYvwp8I8//6zjz3NkkY+i/Aa+8n5BHcU/rJ+4
juKT53dp1P5XuuXFztuJ5+mxDvgu8BffE+Ut5gv8GP3HfGJ8oE+8j/0XxBuK
fgzwF6wvfhf9mWgf9TE+7JfifcAin+fxh50P5/cbLcSm4H1m/KvM9u/5/ZgU
QZ9i5YDhrwM+AR8BQ35jPUT9EPiM+QTM80M3/jy4wx7AuoJuRTqGPo71hrxB
Ofol7lvy8XJ2uh8g3suBOEfIA7U/bJ8L9grkGG/PZND9Qj6PmZmehwT+8Oep
2PlKVR59laC/w2+MeEL+HtpP1K/Mn59k+5zwL2Fd+bhPE41rA17ivf8+X5Hp
z/bPvK87OI3uZ0EfFOlepEvo7+Bn8G/x/qofFIbfFDDokssD0zszH0oxFyp/
UR+weF4S5dh3xT4s5gv4D31A/Q7b3wE/gX0LPOPs3Sz3zFfJ1LOwznifz++f
i8a3gR74e+PdZF6f0vH5McT9POR/CWb2AvgSYIxHlOPiPhf3XpY8D8B7lIt5
wcGfsN8u7tsDhj+H9y+w+94xX+gPnjx/1PL5BxzjBx5DHqJfgPn7Y9j9lShH
vwDz5xHY/gXmDfwOMOQH4l0QTwT9Bft4gKHn4n1xXwTzIeYZgt0JuhD16qx+
rbmfE5TfU6xM/856L4nPavW922VZPlPxXgN8H3oUd47YwfeAV9Cbwbcxvzwf
z55XkJfzqfSJ9cA8Ao+gxzv1Jd87rxV6L1ae3U8Auexcr+djPyhwo/cWxU6K
e/5Kwd9tNdl9BArZdLTScxI4N+EszrcvVZHT0X1ZnDbkDS8v2Hkk8TwZ1p2P
r/8q3PP7VeLzd1gz5z2EynvAzvkbfkjVD9xPxlJ9CvoE+Br81eA7kBeI3+Pz
nLPzzqJcAAw6AH/i8w270X1KyH9+v4nt50A/wvyAD/J5CI3Ubyaew8zpPdTD
e+AL6B8vrwyC3GJ2MWDeHmJ6igijfd6eEe0GA5VjeJ/Pb2ag+piqzxmE8+Fu
1F+EeebwzJ3lG8O6IL4P8g36E+pDH1bb1wl+Bb2Ql1dP4wPRf+xP8edf4/l8
ZR5Mv+T3Q7SCP0xL4weVz+ZmsPp9lp8I9dEPwNAfwafwO+gJT7QHGH4F0a8G
esfvfLxCOtWnlPa8GAx9Dv1DfTFuUbxXTMzzJtphfB7IeBofgX0RMY8n/KSq
3sjuleXP6Rqz6X2AoTfz+aoi6XgBY/5gb3DnqiYxf7V47pjTV7zS+fhlep/e
D8r/AKvrkmkf+WfmtRlpyeZXwn4n+KGib3hlno/1ZvkA0T6Xz8mx/u2U96Np
/az5DRu9MWXLgwv9mO8vyy/I7W8sZPQkwpD7oEfA3HneYFc+j3awK683OOgZ
9A0+yN8voqP8AHwQ/QOMdeDPLUfS/XrufK175j00wWp+vE3h6YKcNwh2uk6I
s2NxemgP31O7zc4Hga/y9w3aM+X/U+o/UL+TQfcJ+ft+4lU9zwPnfOOpXQd7
jZffsbS+Kg+/031FrI/od4b8AB1DXvxnPHUVFn8Je4m7b96IeNovdD54fcki
+FHNwv15Jqq3AIZeBn6I+QUf5e538sW5k6c03hj1MU7UB4z6OJ8EGPoxYLV9
O9VfYF9ivKK9CHkCOx98ETD4KvQyvA86AV/GesEe4/Mb5hLiy90oPQDmzwsZ
ZD5eksHAfz6fu17wnxkEfy+D8T7kEdYX6w+7FfQEGHYV6B7zi7hV3p+XIXHx
3I73+HyvGRLP7zJoHDHmG/YSH0diofgDPol5h77Myw8Wr4B5gl6P+AToq2Kc
N8q5PPHBSdI0ZZ7M1A4R4xP4vBkWMtpZ3+cz2ewUKwUsNG5JjFsQ4+iwLnye
KRMBv4FcAoxxQg8Hv4SeDnmp3Ft2K1ly5rky5k6hej3K+fwkSTQuEfSu5CFz
V/MvuG9Kot9T9cUPVD6r8uU59Ucp7QSnS4jT5vMJRVC7z1lv5F6rBH1TfB9+
LgW/3DPznZRmedCV+9sap9H7xrC+sFMBo5y7X8xB96BH6ENiHgr8zt3f4qjP
xcc57BrIX+Advgd+w/NLe7Y4Mejv0DegT6jj1VF/LKfvdGL70eCHymf8cN7J
QvO78PGKGlqO+Qb9oF3wfz7vaBLh5JN7Mv0e+CVg9Bf0Ie5Loj76wZ+3YuWg
e/izwd/hLwLMxZdkuV8H+hGfx0dP/WUYD+gA/Af6LJ8v5ws9/4J4YqyPmEeJ
57Ps/Cj0I/Av1IdejflHOege+jmf1zs923l8wNBjoE9jXrl1y5JHFHKLy0uW
5d45yBExL7yYR168B0i8J0jMew45Af0UdAG8Bl6KT/jLoB9w8Vm+zD/Gy4+c
4gotgl1uJqIcEu/54vUyM+HjuJmfJCe/CfgQynk/i4nWF/P65JSHkqPXLOdt
YEfz+SoSCH+e0ELzx0zK1Ad5+9lM24XewH8niWSNz/5+jcWncvvvvuwp/u78
XrEI9T7o1UlplL+DH4p5JcU8lLxcNFMY/AP1xHuswaehR+R0bz34B5+f6huV
D4BzuqdTkQ8Bn5R1izvL9kVUfSqF6kdivny8DxjtQ9/iy2MJ9l3Ad0B36viZ
/4rfZ2f+GMg3fh7cqD9G5XduMneeayS7/wF8BTDoiLPHPFk+IxFGe5w9eFsj
6LNa4bytVrhvQ0vpH/wWMOYFv/PykN37gHrAG/48EosPAL7x+SnZfRIo58/j
Wai/B7+r/oXMczRGpl9CPoh+JLwHOuTilOLYvqt4z6B4r7l475WY11TExwuZ
+gd3zsgrM79tXJTgHwnPFs8AvivKGegnwEfAotxBOZ9XzSDEoxkEf59BiE9j
/hAuLszfSOU7t5/o4POwf6DfcPtHfsw/Cvzj87nossXR8PjD4mp4vU6bLe+H
eC+MeK+DeM8lr1eJ9wiw/VHwb/SLv1/MTvenYG8Dhp7B319r5v17vuq5u349
IqlexN3/4Zt5z7p7przZxeJ/MC/oN9aT1ycNVB+F/xtyCTD0Y4wH53DBPwHz
+ypR1A/aXfk9hsKgP/hXYMcBX3DuC+WYL+Xn3Foa38Gd3/FneVtAx8AvUe/j
zp+62/j9OepP+07jOyCPMa+Qu/AHAcZ8g75Rrv7O+Az0ADHeBjDKMd9cfLsv
O5/+n/pcIIOV7/Zm/iPxHCTGA7wCzMXPlNZRGOPh82BoZC4fTiCLtwXM5+fR
8OfXPV35/EYalt8f/AT1AaMc+AI+hCf4N5e3b6Ge+qEAw44F/sNfBT0H9ifs
G9CLeG4JdA19C/Mr+gXFe9BEfiOeyxPvsQUMv4p6/jmZ+lnwhPxX1s0vncBf
CPwH/wfM4Ye7jcKQT7BjoTdifkT9EviNenye13ga38f7ZdKp3wV6OfAN+rZo
52N+sM7AW6wrLy/0Qpytnp7HAn+D/wT4yOWv82X34AAG/wB9cfZBIIuPhp0t
3gMEPqjGxSSo/hGveLqfAvmNdhCfCPwEngBGOfCbyyemYf4G0A+H714ufP43
jQuf3+I5u5cUMMqVbvjrBP0vme5HKPgtZfJ5D3umnPiRo9wALMoNvK/aEXE0
bgP9hf6B8YOP8nFX7L4vrAPkIX8fjIHmzwB+QH/h5SOLK+f33Wz8fVw0Xj+a
rhfH3yYx/odyzn8fyOLA+fPILJ80+A8fT2Wn+Ax9R8xvD37E5zFnfm70H3qG
KK/h94B/TsUHlg+BlmfqcyiH/YH1g/0BmM//w+KLwCf5+x+1Mn8uUCfz5/G0
Qr4zOz0vD37En2dIzkZHWFfQM94T760R71US7/nBPh/6KZ67EvNyiXHpmFcn
ny9976tSb+1X5m+BfABeivfwgi9jnxn7a/DXivd4oB78teDX6HfW+zmMV+KV
c3HRl4xUPsAOyho/49TToL/gfZRD3mAeUQ4/OMrV+bNmy/sMfxl/H0SCMl9X
XxsVv0LvkfHC/jvbn1fvDUmg72W979P5HtrBe1nbcb6Hdpz1rHvM1I+h3Cu7
z0T5G574Hesi7lNADon7GqKfDPSBdUb/copTgBxTP8viJDh+ksHOY6EdPMW8
Btx5I3+G14CBP2gP9SF/UB8w9huVz/i5ZIsrQf8A8/mpM/ezs8SVwB4Q7X7o
GcB/4D3vZ8zMW+KRQvUt/M7nZWMw9r/gnxLbRTtoF35J7KNhn1CNl2X7buI+
G2Au3iPLOSess8pPsp9D5vUiBoNvi3l0xTwZ4jkh8C3YQ8rPubV0PXl9/wfl
WzgHBP2Ey2eVJW4H/QAfFOU39jMwDsgzzCP0Y+CjqI9ingFjnqG34H2MG+sO
GHSY9R6irHllUR/9wvpBTiDuATDoDP0FvfHylp0Hgf6L+QW9cfeJ+esEvz3z
02Gec9IToOdx96fRe+vjaTnv/9TR9VTA5zqq/2C9ODnqzfY90F/gNX6HHOfz
c9qJaJeK55TFPNBiHCXkMPAbcA0nG8qn2is+KUbKj7HOIh8A3YMP8HzaKNzv
yvKTIV8Z5g3fx7zy68dgzn+cm/l/lXpJGqr/8eeBGB6gfXXe2H03gPlzl3bK
R6EXAp/RX+zvifuLgMU819BD+DzpscJ+QDwtF+/VQb+A/4CVblTRU3sA+quY
5wr2oDpPFgpjfkQ7DvYB9G1ufzVL/JF4rxB/7xO7t+azs1vh35X58ElKo+MH
/4F8BH+A/BTvD4UdDvkHGPQIvR78AfIPMPQL+Hcx34BzOhcj5iXF/gzOffD3
b9uz2WXiviv0fy4fB/bvS7PzF4CBd3xeJS2FufN4FB9M9LwJf75LS9cT7YOv
433Qv5j/CeWYJ5Tz/mHE04bQ/VfePtHIXD4vXwYDH8VzDdDTsd7QM0R9HnoG
8B/zDnzn4mo/2AlvT9ipXwLjhb0CPAReKvm/zlkp3gKPIMdEuQY+jt/BL9Au
/A+As8Zz5VtoJQeU+v/S+BmcDwPfQn3AqK/87JVOxyvmbQP+YL3Br6G/i/fm
YH6xn4lywChX1y2S6nncfQ4Ouj2t9OcttZewnijn9QgL1TsAi3FPoA/IPdCV
Kj/chH08A4XVp57KD+hbvN5goOcJ0E/x3hbgMd4Xz8EDD3j+mcDH+cQlUHsA
sHifNuieO188UitDHkPvxhO/83lYEnPMe4P1F+O4sW8MOc7fT2Pl7EsnjH0K
yFHAwE9eT0ii7wPmz9Ml0d9zeg/18B78wcD748rzAZVLqn6QQe1ShY/FMf8d
1pP3gzMYco2XtyyeFngFvQsw8AlynMcXu5A/38biQZX/YL+X5R8D/XL31WjY
fqTyvecsnzL0KbWexpOzg+n9rzbKp/j9bJbvg49LtBPEi6M+4t1QDnuG3yc3
UftB9ANA/gHP4KfC+gPG+sNeEO0HZ33rCavE3S/nGD/nv8ty3hL6D/RowPBX
Qh9S0cZC9VHA4LewB/A97DdC/wIMfoZxYP0Ag/75eDB2vxn0DvV3lieVl6Mp
dJ8ddAg85O+J08viveDcPYJZ7pkBDD89+s2fk3MT9t2yx0+hnPf7sv1hzBd/
f5CFQP+CPQH8A38E/kHfwXoD5u8f1An5g9m+ck5xHfzTTPU62AsYH+Q+7weI
oP4Z+ANAnyLMx7uxfVbAWH/oX+B30L/A7/j71+00PzvK+fsf7UTMg83le/ZK
5+PHqN8qVYhjYfvBfP8E/TCQnb+CXYH+IZ6fuw/ZAYPf8ee/orPZ38BLcd9U
eQSzuH2sNx83ohXiUKz0vD3mH/wD9QGDvrGe3PkBByzagaKdCDsJdpPzd9df
TVT/xb4R9CvgO9YDMPi7et7nJdUfsQ6QP4jfAp/g/R2Zcepe8TReAPSJcj7/
iIXG+yjfKc3y0oJ/AQb+gv9iXnn5rZEhv4EvkBuif5iLo3ToA8ifDTmPcsh5
lINPQr+DPu309+deyO4bx3zCfsJ8AgaeAP/A/+EvAH4C5uUxi5dFf8S8weI9
HKgHfxX0Bf7+Mhafp/on3GQuXoved/+N0j8Xb+fgF8BLcV+F34di99CA/lV8
0dJ1B8ztP3oyfwPkAPQR8G3oI4C5uEngvUeCkKcpnsKoD/rB+QHQEfxzoAfI
RcDceo800P1YzBd3PpP6WyOofQyYzx/wldr/iDNQhtNbz+cP683i0yHXYW9B
XnL29T4z1W/FewJFWLQHYe8CRjnmDeuPeYNdIN6zCvsP5Vgf5TU/Zv8A5u/z
TRDiCxOFew6SqB4MvgN5ATnMnRdxrAfVC5X6T7P5mQCD7mHfifc2YH6Ah+C3
4K+Qj+DLiH/g7yGOpzDGi3lCfcwX1luh79wJ2fgv+I2C7x6M34I/wr+Gc7rg
d9AvwL+4PHD+LM9R1ns6l29Ppk/wHfAj0DvsUcDgBzR/Q+b88/fLGqj/HPQF
v494v4OYfwr0CRj95vIpBepl0AvmCzDmF/4GyCPYl+BrTvzxlpKo/o16vJ2U
TGH4n3h/IuNjmAcRL1Gfy99N78OIpPyGj4+0UzkJegFfBwx6Br6BP4h5hgDz
96MxuQgY/jbgAZ9PI57iM86/oj7mE3wb4xf5OsYLeubuI8oSV43158/TGOj6
Kz/nNlB7CusGuxL9cdKb9WYylbfO9TfeZOfgMH5Fr9uUpJR7FGX3jCqf87WS
rH67uJ/Zvb/on8I3e0RKE5z9XBtGsvpZvNaFUb/LNWe9MpuU/tW4YFXzhFjV
eM+lnxPUc22XjAodB19R9TRvmm9QzXuwy82qnjec8VbBE8nRzl/O5ittUsYR
WNNKajrhF6+o/9E5S0Md7WBfyek37b3XqsTDejj6jXE4x+XvweIgsN+j8Kmd
cQp9+fmapaznedxHpRExnw/m0UkvvjvjaDt4z1nfy1f178at+K7o3aXzmYiT
j8k/q+tR2zVzfQZm3r+d9kyx47y/mhU8jburxsE0cLynxJWs/E7r4T0nn/Fo
biVZ8+EEDrASk3PiUjLvUTprJcecDb8OUL+7NtMv++aemufFZqF+EKc8HDrL
pMjJ3gXUvECe7y1qXMjYDwpdlm7O8rko5xzdM+GLb9R5mWhV87boM89FH7aS
Xs7vf7uu4vXEzDzPhqf0O/gu2lXG34y1i+8o6xv/UIHD57Pv4LvcvcA1LMr+
jfcVs8K3fXeEK3jxPECVHw3Gpil43PZSklK//BSVX/gtSFXww29BojLvu96k
qevcSs2zZBxhVvRk/8bfCNrFd5z0WHqsmvfQP0DNO+NsB+3iO/gu2sV3ssaR
u/dQ+aT0xkTtFKW+A6+B5075GNiR8em/nPPyd5hKf20tNJ5UUce8md/N2U55
B15k1Zs8Ms9LksbsnBL81AodnYxV6/VMJTy/Vv1d/pLq16ywT70P3No3keI5
8B6/o94O5/vnzyj8q9jpTDyNCVDmIemMlYxyDnvnFWU8x4+zOCKcQ84qJ5z9
OuYs97+jrNO2LVZyQjEHgpX1Tlqm3nuw0DEPmBfl/HARli81ax5GZx4htIN2
le8st5IWynIE03nCvDn7F1hSzXc/rr6F5qF16hH59loU+o0L+6zoT7dXWbjf
nfWU9n8V43aTKZ+HvuzE8/A9ZtoO2lX4Y1Kawk+9Ir4TLh+G3UJ2OtdpwXvl
u4+j2P6pEx8+O+YB9pgyv8dTpaz5BeNOsvztTv2/doDKZ3xnqPshZ9ys3L5U
g1jVP1ihu1Vy8nOv+OcKvtaOtdC89E46r/3ErIzHv164wm8W1s+04x3zqMiP
a8kKft2ekazQD2mrntfO/SZFKujsT8GDKv6ct1K9X3nvnLo/k3uikeqrTrt4
3K9GZfxrVyTQJ35X+MM5azb7QeG/vzJ7XJGnoxhdZM0b+rxUqoLvnsZEVS9x
8FnnvLZ18N0GzgrlbynrWmy3leo1yjw5+qms/1lGp9DvlHXJk6LKrycRqjyZ
Z6b+aSf/2+bgA/gu+gH9TJm/0are7pxHzKuSL/+sVbqrmO8BavslvyjyfGR9
9V4C5zl28BsljnBOAtVHnHywbQE1fmTgLJOaT2zNK0Xu5atrVedtUVS2/XLs
Q4J+nf1/3SNVOY81/ZcoBS+PJJgU/JhfVt0/+W5KUeyB2rUTVTlJUil+K+O3
Weg9R7j3aIqTbhxyYoiizj0kWeOzuznkplNvM45Mo+uar/N2/YaxeupvO3g5
wbx+to7Cv7m2+VXfld13NfifqR4v196lMOTPh0JLrIZ3bvKzx8sO1TkRRfef
oY91mT574MAs9wWUetk76OljV3l2dVLr04tbNA5EXvPqgHezUOrv/ePY/hUd
U2Mp3Kz2Fc+PY8wU3pJxcXCJdey+MkuFfT3PbAylML53o1fJKR46nbz0s2GV
teYXIf42Tvp+JoPEz9HS/H9bXh0rfOW7Rn594PiqmIYsP+KafbYtJU+y+88u
alOsz7Ylk6z6lnM+/AuO2DztuI38eHT1iE8blk+xUdOth7osYfed4b2SBzoe
zxhnor/XJzdn2fOye85qPq3we90pLG5gePV3ASGhLL9pbPn2IUe2JRHbtRrS
AHMGfW90tZBFu6uydlDfkFakz8hu7F4L1y+rPkyd+J2Qq4Ev8w1ieS/xPVuf
n88F/3ONtLy34NGKXTrhXgedXPbYvTJtaz+VcsU2PLHyIssHadv6zvf6Hpbv
sWDDmtdGb8ug+nOFm7MnxfVh+R+3v6ow54/pyDOn8RyyZcjeDr/pZTzDpg0K
ODBGJ2//3Hh4Ace8amIzQub/z0yqSu5FctXRy76x6S6vHHyjdO2Jf1ScoZe3
xrY5fybjO9Gdub7y0CSdvPbG0V2fE26TEfk/t94/QycPPTi0RP+1sdLrM3uP
Ltusk4teN70uutVMFu841TaxSwZZuuqKf6ulocR8ponfurIZpN9bzc2/fv4i
vd6cUe7cd4d989q8dfGEDNLJ2vrno+fM5HFo7hdb+uvlL32axF99kUz6Ll05
2kOrkft+uvqwq/Gz1GLRvyV9Tpil61tu/Llybqy0ZmavS+NuZEgVd+zsvC6P
RWo0+lXTA2aztLfVmqN/ns6QBh63RN64dFqaPCzO3HpuOlm0vdov+XYayYHT
EyzNTalkxd67/0t6kkCGzhnx2z9uRvJi4+jxa9ukE/li9Ha5bgRpVKq727q/
rWRSo935ywyMlEqkdN654XUqMZ64NKtD20Tpe5kek648SiDFvp7e1K5hmtR9
Uesnp7cnkNzxZZeXm5kmPTYfDZwX/I38WbFsdFigTSpzb9Wtxf5HpSbLdva/
aE6XAkyF3Qcsuk79rXlalK48q88X6s8G39ze8d2vLV1YXq/8RfvZm26/S4IL
Ddr6r1sqfc59JJnIoxSSnH/2rNIxKUS6PCS45y0LiermrZnf8it59PhpcvUV
FjJ85OASpxO/kN2VEpLjl9mkIZ/HnmjbMlKSzs6c+WGSWTrz5VjzTReipTx9
g6L2hEZK9yLv/NamkU1a9SWk/PaHKSRmdt7aP/VKIptnjLbnuZZO9ehRPqe7
36kRI0V8S4hp1cgs2UuPWuz/vy9Sub2BwxtusBCvGxM7lhuVQPXWXJVr3r63
MEEKfbd934IfaWTnnuemy97pZEUAqbm+tMP+G9KrVPyGz1K+Ie+K+PhZpdLT
zEsPtEgjy2cW7/NqT7x0RA4nfp8TyQnlmSrV7T614t8+KeTO9Mtnn95OklLb
V/K6eeQHibN32NUij1nyudb8yPs+iZL0z76Cm1fYKJ1ND2ou/e+gTbra4v3M
Xs77XJ7Yg75mGMmgv3+5sjRVI7uXMkxLeuoi/3x03ORxJbRy+oGfL3ye6yK3
fdC41NfH6VKvQYXntPKxUb44r5XbiMTWBjmucu3SYd1Z/kLzCb/8bskRxHPr
Hy1+XcLyE1Y6viZt9hs3eds1v4D8t1+R84POzig40SBnXPa5O2v9B2mkPOzT
/1xd5bW5TtZ691cCqXz19rRDl13kw+nDfYNWJ5DARd09K15xkW/NTkrc9CSC
PPot5q9dYRp52Opbwf1vRJBLrd7u+/etRo6/ndQu/4mT0v+KnPgjMa9WHtl/
dIl3Iwyy9xC3W59ya2XzzQ/R66ZbSdM1ca2qVdLKJ1s1+3K2i5X0vH6nbNdz
6aTx0tFbE8caidf+L+kNQq1k75Tfkho43v/prM/IWoPTSbuN/r1i97nIPbdU
aRJSO52cKLqo9JxLGvn4S9v9AuuNZM/Pp6Z6moyUb9JzPAVfPCkw10SuPX7d
okchrVzCb1eJ5dUSyKm15jwFh6bQeMwznVzydA1OEvJuJktxu0bE/9YvRap/
cWerhcXN0quOo677RiVJFbu9qJm3XRSlG9hFNdqfLDT+A8untzCs+vuhL63E
cjpvoTu7GZ3tvfFwTuVRP8huS89/Oi21kAo/Ha09LYT5K6UvRSc6DGfJb2az
8xurJEnBxdr3LuQWQ0x1JzYP7Gol/f7UTS30KZqM+tbh/a5B7PuVbrt3fKE1
k6kniyQvrveGynvMx53VJR9Ud2H5XdLKtR4cspnltX3TrM64C9N08uFKHVde
O7eVypHBJSrvbtqf3auBcXT0jD/2cYWJTHIZ9LLco2QJz0Mr14YkNfjG7rH8
lCf5+DsHfP7W9h4f00n/i57bHx7+RqSF4wbn2W0mTWu8j2hXL5XkTjg01N5d
I4d3CiCVW6aSER3n+k2OzCBbr71+efJKCrGOmR40a52NNG4wvs6n75EktHuZ
SuFXtfLQCaY6y6s/Je2NM83XZBsZ3c/L+3+pejlvnos/OixJo/HKRvfWRUdu
Y/d/Tmz3KU/7xS7yqVd7T5c7xu6fwzi/vYua1+AvF9nr0fLx0xLZfXOBH6fO
HDYx+31yS9wvjzvhkH91In+amJIWSsvDjF+LNe/+hbyrtuHR1vNaWZNr4GbL
MxOdx5gq78eWcfCpIxmTa+TyYvclGm6XL3q5Ugqp6eXfYG/lVHLKK/hYic4O
e65flbVVfrB4eti/hoezCgQlamS73DXxn6VppNCy+FXzf9LIfy4tfTHAgR/6
sQ899k5yla+17LGlY/VUpp9tGVPLd8xV2p5npStXt69/Qi7cdS1XrrVGLtCi
ee+eA56Qll0/Wubl0cgjTjU68WD4E/o+7M914+sVdN3imF/p2jNtgI10tm7p
9/saV3nE4LQB/csjLw27J71uM7fnsy+5yOGpT6t83BlD6gcUkKs41qNFz3WP
ugZFk6PvLi4pNdFFtoz65eVUwu6/6x58PGDODJYfDv1ubn464swaN/ld/sDm
lTRWGsf9cO3VgXcemSm/BD3828uv5oMtbvKbPFsX5PZ8Tus36tV4y/zGl+n5
tJa7L5W0xj6h53v/qOA/6EBlG9WPg369sEfX4w7dnzp57srOdm90Mh/XaiZ3
0ywfShlSyUv7upppwy2UTy36kjB3TgyTGzOvjdt7ICyd8iMaT3ljvS39YbIU
cWnv1oh9L6g+9iT5zJCkDuz+u0n1PKt1cbWSQcPOXJxcxCDfKLskqPfMH2R8
fN5aB5/q5TvLa/97W/+FTI80JL4paJDjfx9sHtE9jsx7veTSyxU6uZw8pmS7
5Q/IucS2o3I3dJFvlj+3x1Q/Rhr8tfLWgW1c5akL1lUdWcAk/U+6537GzVVe
P7be6aqnXOStocNmVv6qkbULh734vMxCOrjfmnVmr4U8XPduaKlR6ZJv+rni
Kz2tJGqy/Hb2cI1srFb4dZP+qcRtzItnWxtp5Ca35x8efTWFzI9e7O+/1CAP
bJ+v6qSMl8R/m+v8q3Ea+fq7SfM0F99Ja1oYyz9wT5bm3cod2jjOJBmiTlUv
Vv8dseqLeF85ayNFvZP+uT/puZR/xaNRZwJtZLPcJY/3b8+l9k2rHh1d3Uba
fFsxo0N7E/k8PM8/2q4pZNfygFEfmsVL2ifdt1VZYZaKBic3TA1NkJq7Lx/w
6wymN9e/bN4S72um+co+5msaUPQw8zcUrnjlasjBdHpu+9Dl+WV8WhhJYTKh
b53jzF6tOr/Xivs+LD9dUIf358PqsfPTsOPHHF/easORH/S9JQUHTB/7IEG6
+sq34seDRpL09K2l/KQEqWH8zIdvf48h3nV+blV5t4n8seR4uTXhH0iXqMgy
1XY45ntu+f3kyTOpXK2DRX/eZyWHy35+VOBMuhRobza04vggMv5JvH9YS5ZX
53HToPs9J6ZL78a0+Ng0JJTayx79qyTPfJouPR3R/dSmzwdIwom1nQ467PZm
G2oMsE57Tu4+i1gTO9tKOo3cNdurtoW8Czf+HTcyibwJN7VaVddV/ssYVLPX
CjfZcqXjpvikNPKwkWZhqZpu8o7OZV7PG2MiFS4c05VbopcvVWnt1nVHGvnd
/YDLkUd6Ob1Bc4/KByzkfs3W4ZVnucq1tlwr1miFXZpdLWzI3QCr9KlzhvXy
Iau06dL7M9V97dI074yxK3snkflnvBbXOKOX7x6dnGf2/USy64Z7i+NEL3+e
0MpayoH/bofrtrq8wE0++qDiuv1+0aTv2tkf/9qjlzdsb9q9bVowISZP13LL
3ORZw748kCp9IqXa3+/R2XnftCnj6Ll7t6SyVYI/7nOUTxrh9z2tb5B0Y0Kh
Mm5LXOWtry+HhE95KcnR04zVr+rlLRsm29x7fSCHjcbE9me18t/+bcKrXHpP
Kph7lxq3VCv/1jZ1w9oRb8niWinPrj/JIMaRvcKPTbhEjD6fr7zZk0EiBq2p
b3e3En3vAgGHnmvly/der7+x4BLlv4OmPYyeVtVFrnC9WyuiNZGZv24quXpV
klRhbC39oQsp0sP6X8u6/J0smTfrz8+PTJZ25rmjuxeUROQ+/asX7GklF6oc
net3Pol0/DBguXdnK2nlV7NxmcVJRJ/bXNj7oImUWbmtW6WyiWR0qavbzrUy
UXnad4VlxtcEtt8fdOFA7SWX2X11uYbNXrDqhl4udvrRL7cPhtL9RdBH2e6L
KxXY7iK3WnBzcO777D66nqsS3Ee7CvfNOfSpHd4+JWus08v7E4O2nK6fQor8
WbZS0PF0klwtMjrPjOz3y0Vf6rCy5Sid/HHNhPY2bbJ0sc27iuempFB6ujbs
jD6stkH+t+qZpINLnlD5Dr9s7RKGJxErXeSvD/P279ElLZu8d99u3T7BeR+d
b54uNz3esXNFt85Ey/mTSeSJCTuOl9TJvx8uVmJ+UZbP2v1+7goRoSz/X2pA
SOFKYUnE2jc+98apLvKZAn/H3ynsKHd7nfbFTycbH1v3+QQkkm0tu9wc7bCP
63bK877Rp3jy9kKBH9adOrn86C3Hf18fT4wppbefccARxq4Xao98SvuzuN21
lEZTUsjNSoPn7fyfq7x5XOj7P4pHUXn3KbbDlkZ+H8nKLrk3VdK60bi7paWm
XW0W4iqfzjNgaY3RT8iNQdNezZH1MgkcWnzB0m/S5pnjT4xtaKDxFfMKVa/e
Y5Ze7rf7aNKHbkYptHnd0146g7yl0cLb98umUX0U+1T9LIZSHW+6yh3OTkuq
vJTdMz/h/r+3+r38mi3PneuhkPmFkrTy9dE/nYwaeoEsdJ8wbfTPrnLn/mPX
RF1leXyxjn5xC8ptrsviDD+eNvf8OFIrhx71qfu7GzvHNGZOBePnp8nk+/Hv
/3y6pZWblim/tqtPcrb8J5Euv1TLCEsnQyYY/re4G7tPrlq1vsagojZqf0IO
H7K+rL04zUT1+s87Ch/N5aC7f7z7r+uwjeURqV7yj6VNKrjIy8PLFh5TKIqU
exRaq2aohewu9/eioYFx5OmZ3LXvz3aT+XsIdPJkb++lPfOHk8q2alcLHmZ5
GK7F13e93tthlzUJuXz3zVvKJ37f+efWzg1ekCq7auSa1iGD+iHPNBjwvlyE
i3wnde+VjHeBknl1ITlUq6V5rPPk3i6vmxNNRl693NxeI13K77tWuzi3QR5Q
6EznInVc5Fn3gqvFVjTIrY4U2rDosk4ec2BVRBu/EGl1yV+atcvj0J8DGyUX
OBcipVRf3aGIVSdXnPxQavqW3RO+u+uZZyGe7LlsU4X6Vxx650+bfh423qCR
zTUjBlZZGEr1xwI7q+43z31Gz8PcOxpU46dFYRTeF5o/sEF/g5xax1zwhZ+V
HJoVtamvh0Hu1HTOxLkfzcQw2JQ6fX+Wc8KDdCndHXq+zbYtb6RsJW9d2+ky
8mto+ejaD8vIs2yUfmvcGRn34kMGWX3MO3hHhoXc3v5q0bN7zM4Kfhx+2few
XRqUVLurv0NPb1B7y/ydPVl+whddZr4fdCt7vkH3yYeORTaySAtam+4Pm8vs
FugDFV+0srw8m0CeX5zifWhtglSn4wa/36enkbFxQd+KHUqQjtYInNpisZFs
X5I8PM95G+nVKuL47MQnxDj7Q8jZpokkZmZkpNvfRtqPeiuWjbR8SpKuzl3a
b+39FMnm93SP64okun/88PCl+HNWMyk7aNVCLx3LV4j+XGn2dGu18GSiT/r0
3Lunje5nNflf89PLr6aSgl7R238al0oKzDl2/chYK9k8f1XTCnnCScDD2/aU
viZy8t/k/WcXx5DTk9KOX9ydSkIv9Iqa+HcS+WmW168tjkVJ4xMidAFmk7T6
4O6jnx86vrP475i281NIxs5O0qgA2D8sju/06fv3r+9LIWN+OdU7qFIC+dmy
QxpTzUTed7q0y+OvWBJX5WO3XV1M1F7Jv+DLhX01TeSX/Us3e5RKoP6xuiM2
/mjaKkEiK94vHLzKQmoaTk86LiVIa8fU7HR7l4Xul3wY4fHR18NC+l5t8tPd
MQnSCmujpwl6Zh97ey35LWJqPPlcvO4wa5c04ZyHUTp4vfkjuW0a1bt+OW6J
TX3A/F5zJq+LPdMwXZrfp++lGqM/SMs6h7ZP+WCRCrv+2TvKlCZ5bdu/a17Z
VLqvRC7PuvZiv02av2xNwVPXHHph8ph7ZbbbpO0/P/o77Xr2/IYvJz64Vv3v
dOnslMKH9zSLofE206uQI/sHaOXuH2ot7n9XJ38++NfXH2Faedq90FmFaxpo
vG6BxYffjzlgoPI20nvVypMhBvlQy/3j7h6IJ90GzyiTnmSQSwatsrs8iicd
Th3M0CQb5EJtaw6JyvuS8vtlxy7P+alAGjl9qvAvw37Vyl5DJh2eXC6NkJqa
0v7VtfKuenUu7N/3jawP6+G3rKHDnuy9sUzvgK+Uz0M+3G91sduY1Q59ImBB
+zb/fqP6A/hmrSMn7tS+qZP9lwT4zCah2eLuL/kuD7dscpF/vZgcl+vMN1LX
fcnt+X20cv70I/euhDyVgqeVPDjOUyufr1BpTfDBUGnnCPJ07WetfKSCfkG1
xs8lbcaPeRprBun9p++nXmnPpJ0Dv0QN7uPgT6Wb9LEX0siHyYJSn/y18pRv
YybfL6GRu+x51cl3uVZuEBP0udEDrTzzS2Lpbu8yyPxVx9d+uKuVD7xofqbK
QI08qGGzI1WJVm63Oihkyj0bmT2hxoq7F7Ry4+lSsWtVbCROGjbtVD2NXNU9
smbiMhspvH6A1HaSVnZ5VW6Q1NFChoaN1o31s5ExIfkvVXTwvUov/i29YIeN
aK4OWzrqT6389p/Dfo3THHZSjULhh0kGubMn8qcqP2wkaNmwsPsjNXLwqcTU
GS9tBM9iM4I/frtkI3sthd4vr2QjKZdrfqkRYSTeFVev/ucZuy/wSnxSSk2N
nXjdHd03aTfLS+sfFvny6dTn1K+5a8njAWf2pVN6rvNk08s0B92eiHxvIO3T
SLFvJ8vUueTC5yV2rPvt0msb5R79nRS8d9zl/LdkqcmvrfQlTiXTOLAydVMe
Pf5slBr+mJrv9BWWb3H4mrtRkyumkHD94KGVXdJovGuPf7o+C5rL8vU38spr
KRxgIb7rWxdanT/VYXfW7ZjqkiRtf+o7YH29aNItSvq7w3aWj7Hg+xMJLqdZ
PHeuJ5cXF7/E8itWDp2eUvIFg5/nrjt2lYde/iU5/MvbXG9IrpaDesmvXOW6
25vMs90YQeM079XrnRFb5D2N5zvmUsUjaHUaKbfFI3xqjE5+sbLv7c7D2P2R
Wzyu3utY9gedJ8QneidHpp084SIfPGHMu/FsOLldb+4vaQ79smzhO2cP3DtK
z3+MPm0Ke3yf5VM89u182zUJNqpHQM72Kjv4f42rpZHuX3d1jLCyPFhJ1hFL
K8xn90naLdfcE49r5S67T1Q+/TSRrJvs+ffGOmFUf/q9x/JFb7+HklUh9coc
+mojM88W7zI1NYyU/xR23j49nTQskezTvkmS1PyjYUabsBSaLzNoz6Uilsks
v35o1/mLfywwyGs7T/19tAPP8Szao4pNf0QrN2yj6bpmgVau3nLa3xfckol9
WcEWne4Y5I4+pxY0cNg7wYaYEcMd6zP13rPEeqkp5J3n0JJr/tXIC94ODz5f
P5WozwxS7XKPscNdDXK36fue3s7F4iQOX6305r6vTXp4q8cdj4exUtlyA/aM
3GmTusvtrg87/l3acvVd1zWrbVK1otHfq178IVU51XLlRX+btLz0wIb7N8VL
pWvcuDHaAbsdqJcrvw+LI7135+aVCU/TifeKE1MG/aOj55ZXuv3u0+qki/wm
MqBdV8M1yk8bFF5a4/aYMLJkeosbgxtr5SXv/Qe9HB9GFrtH59vyP6186XWf
qOJ+JjK1bd19y0ZaycnZO0uGeZvI9RFlfrNIVtL1QMVxjXam0PMEaTNrRT2r
wPJX0ftHp+tG96P3gcSSIT4bVqc66CXJnj458Md3Uj9u4OTbzy0k/P7hggNJ
PGk0bm0N/T1XuWHIm1y+VbPnQwzxmTUidBOLdw+Uj5dp9sZG/nrx54bWphTK
P0pN3TZtzP1QcvLZrPd9jmvk96ssPVIPmciHCueKvs7F7vc9Jm3/s1sfV7ld
7mvHxvmE0vkxlxy8d8eACFVO+2rk4pW71isqJ5KfqrVPaejob2p0/XtTJiUS
v/w+A7dcsJCoVSvef36RTHLFrPEt0tBFTr40ZlOJSez+TvT/YnyVgOklXeTB
z/8tUebDd7Jmy+Byi3e5yKZu46f32R1LfDfv/FjkqIt862SL3xbdiiHXTh0t
YV3G8heWaupT5EUHF3nyHH/7lposf2G7SfMbtV+vl7tdfPhHyGVXObRIu31V
L+plw+A3tyZ3cpVPjunxtkHdRFKxfdfzNa+5ypW8S9ufj85+P2in/eUqre3A
7POqG0eeaiS5ySXzzT/xtnsoPVfy4NzeTcuvpVJ/5yiLtnqzZuye1+V1j+QN
vODQ27aX7zF3l6t8WXdrZ+qqWPIyd5vVhvMsr53bom3rLUdc6X0501zCh8xw
8Ks718qOvOLJ7hNtm/CgZp0qMstr2GB4m4PDf1B9fsdf62ekOTATcHv/tuv2
VWNxq5W7Xa4U3pP57c/PCqm6qscXGrfbofkf1Z7s08tbn795c6J3KjmRh/Q+
tpzd/93rr8PPP4zTyJMLtptobxhH7cUm/yt5Inx3Itm+W/++ywmWj3HX6n5t
h/fQ0nlEv2ZFTOxd+eBHovWPeJW7oCvNWwQ+vPHzguhNL89JF11+js91xCzN
PVXW5l8xVkq/fXuzgyVIXiV7Hm413Cw13XHlVfd1sdKbde9056aYpfF52h5L
as7shOvdV+d+s95Mnz/O7/ot9fpT8sDflvwyJZ0Ubv14RIEyz0jxWS0m/Lwh
neTdVO5doZMJ0qxE/4xVFYzkzxHF0p42tdJzf8tH5Zs9Ma9DXj48lXC74Fvp
dIcf8V8G3JTqL9xyt1YHFtf/sr1rS32xDGlOvZX7JgUFSuI5/0FPBg+s9JNd
OuPeOP/bny3krabiqtW5YqhdHNG/wsT0iO/SuE7HbxvPsvtWNW/+nr+tq5WI
5zRKTdX2btMtWtqrm9fMJ9BEyl0b1jPwfqzUekLfyu6DzFKay8trblWjWd7w
61FLbnb/KB2MjPafHMDuY53+/svAkgWSieSSv+SRX1OoXRVXsHWrAnVMZHuu
iprO1eLJtMcHq0yuaiKTPa64NHv9g8TUPl+pVnUTabP+bMCIi/HS48Kf/qrx
wUbyzO2XNul7vPTw3ydtO5WykUMh5N63AKOEJ/D9pwmNC+sms/u7POqt/b3K
m1SpS5K7qeR7dg680+gJ/QYsM5GKs7oUX1Kf5c3R9g32XTFXL7+fVqPRFncd
PQcdeLXF/ZmLWb7KR5cqPPfplUiCYhc1Xx1tkNcNe7Ukj/EZiSnZz61uKVe5
b8bGMeGWdNK86sVfPLunkj/W52qxsFca6R/yfF6cn5noJvXvPPKxTeqa4bPZ
mJggLeuRe0r0b2apeVB41f71kqTKg5sY7kZapEkJv/cYkNdE4wGTLrUImdXD
wa++1Vhd7tdk4b65CFKzcM1HC+879JP+ox59Lx1Bz2tt2FB8i2HFN5LROFKT
WM5GPkfFWEtF2yT7Lj35X+54afLduuX3vLDRuLdS3Y7py5+zSf6z3OaWLB8t
9QrZ9nErsUmBr4zrRx+OljZ/uNnvr682qZumxPPZA2OlHb8WX2TtwfI/Dp0x
Z5bfSyNpM1jfbdxZV3nnOI9Kk7axfJAuk3q5FiluJ1fe17q6KMlI9HP7VvC+
H0tONX2UaooyyP5n59w9G+OQb6OW949wzO+NllFhbdukkWW/tRjTLcxGNswp
ka9c3TQyse7yqJe7bRS/KlRf0L/mbXZ/7of1Gc8a3mL5HE+Xu2Qsv86xPo1O
v3o6TyNHDTCPSXWUJ+4P1tnH28mzG6RD4XgXOd7Y8XOLJnYyx8d/SrkyDv7e
MFGq5BUu3euavv5+MTe5RpEXQ3N52Ehs1/g+jc+H0/M/TV4vqXRgv1Ga3nzo
yDxNWX7HBhMP75xTm+Vz3N+yRdVHMTbSZG79CzGLE0nUnjJ6j+1m6m+JDG9V
/u+lafR8acVB15bM/t1EgndennM8t6u8ctjcYguLR9Dy4GmP3j8OY/kkZj3O
s7lWKw2NX0O7h04NeVK8h15uWuXsj/D3X+n9urftu0K6XbRQOgA/qCZ9K/jb
ZJ3cLPzi4JoFb9L1m1mnf/5d60yUn3eLfjrs8dzv1L/6R/cCrSZ42Sn/xvdH
31rU8lH+WBrn1Dhtl0/8q0QyrFD94kVjM+g5FJzveLUi2Oy9x0rj79191s7O
q7FKze2X3R85YJxL6nH+e71pbzNIsQu1DEUO6eW2L2JqzxuRQhZ/3p1vzVC9
3Hpm06iJxX+QQ4v6P8r4Vy8/vvcpwf3LdzJ3nO8//a7r5Xb9o/92aeFCn4Ez
yz/uPdJF7rM6tEHgLp18xJS3zYHNGSRPzLABvx7WyR7NXLfGbMkgZMf3Da0O
6eTpwTMfBNe3kOqeib3lJzp5dO8ye/867SKfDPeMrPFKI58cv7xa6x1mUrNr
5LmZLzPI3PyN2j8cZ6FPbc8R6Y8MFjJhdZ9p+U+bya6FUdMsk/Vy4Y19Ru1p
n0L+DZy95Pymd6TmwMvFS3vaSLf9UfMOTfxC/R7wX2xdd+fL0Q5W0sf17JlO
DlhdTyvJV/9Tq0+1k4R7G5NIye+3Rs+dYiSlS7oklPG303Ygx+fnva+PHRBO
/TTrn60pkPrgm9TmUR35XnI6jUse+EeXvbWGfZBm19kzKNcHixQ5+Xlza3uH
nt1lYZ2R49i9EeENxozZEx4qGU/kaRPQ/S1p+CSyRg/JRX49Kuan5B1vSfnx
CXPnDNLIQ0fP+nloxluSkLq/RuMzLE+8JlfwEnstM42jKzdsXZGJj0zSuBqd
a595aaHnZU+uM08Kio6h5/VwPnTPgZAH9Rx0mzRdyvPS6Z/v0rb7n79r5QE1
Tq7ZXz2F7J7/WHdzg1Z+PdjXckrL7gUwFXCRxi1LI38Vad/n6q0MMvbHpMp7
isaRUUurXNx+xVW+NWjFpUt1fpDyZ97u8rvpSvfHu/R51vWGwWEf/JE7yHKC
5SFucrlTyaIHk+l+APTFt4dqlOz6xk3++Xu3uMAladnyGIyeN6L44rNuNC/p
s+evj3+96ya//dGikPeHMLp/jvWb43/iws1HzO962VRlygIfF/nQtba11/ql
U34dZlnUv2iyidp1oP+8+X9/a19oJHO/vfz0RJMm4E8aCTz4emeZz0Yaf9PW
2Gnjo3+MpMUvfQouKpVC7ZabeTrW2WdkeezQP6+IzZuTXRJJifBJIcv+sNE4
IPCHakU9K/sdNtNzDvW8nk+obWZxk9Mqrrm+6c8M6S/f8abnflZ6LuwSaTZ4
wc1UcrHY6jzG7gZ5zsIFozqVSyW3iGe3D1MNsj5jyIIT8xPI0Y43R5R7ppc/
/ClLpYdHkLgaB3+72VQvj7OWm757t4Our3YfV+qZq1zv08wq5T/q5LxbvmS8
cuj/v1gD5px+ZSUu7/KeOHvARf5+eezADVVfkjn9TgZc7aeTGxQb8r7TUhe5
35BE+6ZiOjn63pETbeeaybfepZYX+Ucjm91+nXd8vZlU7PZPWFkHvpcMGNhk
3A6bJKV8//1MzSSpcbGJm12n2KRh+T/Zy79PkmY32uMSGPiW/DOt3ov1M+1E
17LnT3KtL1KsefCM5o/tJG5sqQaJiVbp3v+6HS4W/0kKullnruHFE2lZSIe8
/WZkSK59dAEZ1V5IuapHNOz9I0N6tjDSp3e/r/Q85oM1N07HBD2j/AT5CzrM
OOg32JBOfp393XvIwgypRJ2BvYv9SCKt4ov5f3ifIQUVjPaqUjWJuG+dNv/p
HnauqmUtzc1Ihx0O/RJPw9OG+Ws1sUm7Si2tdHeyiYTVX31oxUK9/G5Sz+5x
qUnEI7hrwkqH3fC9TvDlY8cTyHOLFBv0RC+PbhXbpe31GDKooNV94ym9HPMx
yrxv7Tdy3L3+xDaD9PJ2Q5Xe64dFku+V5CB5ql5eHNSs2eX6n8jYO5VC/Bc5
+H25kRbrrBByaX+6+6h5Ovl1ZbenB8d/JMdHbnyQ4qqTm5Ytcrj72FvEbVaZ
xPr/08lXdnc7X3v8W3KiR6nJNftkkFupT1aNHGImM8v/8+a1SSP/OLv08sNF
ZlKwq0/7RgvspHVc0WGFBpjJy0JDkhsfsJGV3t/a2kpmEG+/kld130wk1/aI
tCBvM1m5ZsTO0c9NpGrk8JqjWmSQCS2/jSZ5k8m/L8o26HjaRuPi0zYu2dKl
UjKZ2anhu/VbjGR/s40TWycmkTtLKm9cvcRIgufHt4uNSKB5Sp4UXNiq5upP
VH4jf0WlTz6vi9TXybUDR6w93DeC/PX6n7vPT9hJo/c+O//dFEG+ut88s93D
Tq55vdo3miTTfH3WJlN1Goc+ifOrlkW7rjb1MVL8yDcgqt+Gpcnk0eBlS6tF
W8nLWkGVntaKIJp2A36fUSGDNBubULRq5QiyusAp06SOGaTSNtl1aTEHfWn/
/WQyZZBLl6r321g0gvz5fmC9H18ccIc9s1xbRVB/V/GSxf91DUgmrS3fyiac
Nsj5//12/X5gKrn+snux3mfc5LtXCpRYPiiVzHk36skM2Y3Flb+dXqTZZYPc
ILlrhagG30j+eYNepIe6ySsLnhrxYUIMuZS0SCc/N8iF1pS8bu0bTUJLpjxo
fMtA8wuMWTvkyIZEN7nfFnfjsvafSbnZP/VKre8mS68e9nQ3BUg/7Smn6d5F
L1/fW2HZwN4p0sNfOtTL5Wh/4qVKJf98apQWF+p8ou4Mhz4x0adz1T8/kYhJ
Y273c3OV/6+s646n8u3/ZzkiREmhNFCilNFQdBtFyogiLUXTiAYlm5IoDZVR
ViSypWxdF4lE9szeex7H2fzO73m+5z6vb89f53Wf2+u4x+f6XO/Per+HQ047
nBZLA9OBOKpHMR6e3lDgOLxmEnm6+1fwtU04aNmGc3oQQUMWFmXxjXNYaDIu
fTvYPBxp0huQSVXGQb9Ngk4/awvBXMiFwvviWLgxb5dyBu0biFdfOvfwChbu
/hmMeaM4jay5rO99NHQJmAyrZww/nEK+jal9HEvAwNLlMlpiN6fR/K9GoxV/
Mw8DOb9+XnFD2BLYxS87uOwMHRl1vRHEqMfADASTF6LGBKGmMi8YY1TQXrnc
Q+sXDSyNWVk6x5GBf6C5k0E1Fspd/3Xv4ngvsF21M8bi2QLA/vT/7StHAgHK
SlZtybywZfcBt7fHe8FRF1FTqhYPvFl6FF96GwJ1D0irusWOS7T6TTWov5Hv
Pav7SwhU5L7RbPiuHjrCa90lYxLSj7witS0xPJjI7vXr9Pcs9SPg1Eb9gZdM
ZBdFRuV9Pw/KF8qJy5SdU6ZjavEw/v36QuWIDjQP4KL785OVDZdH7W1g9EKs
KgY95uBTvpXBu2b2Y2F3+esC1pMMNP8R1WFqnXWBiO6j2Rib85cj8egxJ48i
Vb3YW+LL1fU0P2vrv3aBie53DBvRVpthbp7ROOarVFzLAvAY/HLJTWqR26/0
zz72dq1WSyt730LnkVZ4shb0aYib6hdJ/JU29HcdKl+WPz5b+5dO0zzogz9P
OdpS0GPJnZ4Dm2fnwRu9FK+wRzPcvidpmtHyH7MorxMnH5/+O876ctgCsjH3
tqDG3XnkEJZH+dEyEkLoDs+hXB4B+84nBo8K0oDF9MYRsHYQSJ4otzqgQ0fx
Q4pEfMSXpT5w7l5l+fOJebAQ3BnAOD8HWNp+vle/k4CdlPqMlus8qJMsExE5
RAImO6pW7jOaB49NxzZbPJwBAf/5pAP756/crw5NA+lp6cgHrnTAVO9OXM/e
b46ZHY5x8qeD4xLN+ScNZwDv48xbJXJUENvXavtQlIycOf5l09w0BbH2SSoS
ieHqQBW9f/DmgVE/iksvPd+yWsWmB81rcPguV99s8mQ2M9D5f8rum27ZrQxk
62ISlSqCR+2Pg6Nw/A0CCwN4OFP5wHYXFoueVw+6U/shEwsFeDJdJm8TYeO7
5MptrxaBykW64lYHIiQSE2+9d2Dj6TdUbP9ZLo+VuZ7DsdgZIlws7jd1/jKJ
+vsPW+vXlEUNovlClFfeb4jHme3PrHvd9qTx9SEq6QI2m2d44X2CTry8Ohm1
S079xKfdSm5/Ex4mKYoFpAr1ofkEjj8kzQ0SY1ew73Mx8E6i1TBq55zzP5/J
TDl7YOF5nKmPQgYeXU8cnol91rKHVancvtmw5p76nEPTYGVeihD8yO1LTd+k
YGJkPwJ07Q1CcJMM0KvKahBSnQKGJvaHv/nTUV5/W9wVFYwxAzilh2R16DcC
Efkgm9K7BNi4W2iVfV8Lkldm5WvIJEA33dviEnzs/fZrp9mGT3iox9v86pzO
FICxJ24Y5rL9A6lAf//OITCbt3hkpBcHqwOhXYYbBZz4VpcbJrYEZO1KZU77
UEBHa3R72bklYGwgcWqFDQWEdx2MeR+0CG70pn6UuEIBek93CJ+4uwjGatXn
Srew/UalVX1/EBV8Tk2o7xbGwstnjePBJirKi88Sp3b4irFAjXDerZa8RUBP
kT0m/BEPmRtJjzfPMoGP46MCB38cWtdhbn9CtbQgwNUJv3QWjlLR/Zgzj3ne
ZYdALxuPllm+6zjA24e+H87/e7alsVlPDQv7+FYK/Oj+g56Pa1PdCr2461So
4OzUexFu3wwnjm89ckCOvpNbV/K5nlh6R5AOjmj4fF4eTEH7Ezjx+W3r3gML
nUQYFb6ibO/oPEh64PzNtI/7nonuy3QO1vLATwpvdD0l2Hg8fdPW8xGL4O5H
szpfAhksq4y6lGC4CNymjx4vu16D2uuJPxFHLG+2oscDHcucB5o70efBud/1
px6f2RxFgLlg9mjNzkGwu63JxEwXA6MtL7i90B4EVRttdSatMdBWM9ZmQxwJ
TMsVO+Zp4qHYUXrahkGuHkvv8owYr/ZhYPKcevEz+5gWEN+6ln08qdET934H
Hs0nPU407Ne8w9Upa/iwN0argNtvZ7tXu7LvEwEyo5bxSinMo/EZp/7Av7w8
WO8JDxR9cIHVJTKC4q89bY1N23LGgPxk33Gfd2z8MMrTuLFrHNSskRblTyVA
Cccy70sn+hCt+IP3NvAwQcYd3rog4z7E8LmQuq8tC9i1NKx7VMTGcRQ1SzPB
BeC17JeXURANze9x+GsGdelbddePI54sCe8BEhM5++Xx2XdW88jq4OkYOo6G
xmOcvxe5HnHsLPt3+vMfZc4JDKN8angw5/jLloEQCLs1m0/+QS4sq5H2Cmcg
37Mutj4LbEfiw7OX/97Xh0Q+ax92esvVrfTVp5Zn+XH5u/Q2rb6JyFDBj0TL
zJyzFGR0dqYJM0VGnAdM9a/GUMDFggP67tYEWKBXo5e2twdMDNzXr7xAgIdL
4tUju1hoXwaHv8VR9biN07UZBJ1rx62v5nvEtvtVRVQVZwqKNzl/L0r/fXHZ
63F0X7wjkBd9v4QHHsrLlht/xQJbN2MOz2xmx7Ma91YbnKSg9TEOv9zQGStf
iu4iWp/zLivdEpxLgEP3rNU9bDEwOWXILKGLALts0n2DG1h/8QougpyRu08E
thMg8aug7hlfANzbrUz3hBBgqNNFyzqPYvAqnP+CYyEBmiNWWeLlYyA9OqPt
6SUGEgswIuv458Dk5SM+y5AF8NZjTP3t3n7gtEbFoDYaDz1Kf823m7P98Sfh
kpcP50HQHkGp10JM0PxT6doWlXmw1alyw4FAnn/zXrHt8NiN/ZI88fMgp9bj
itXrPuCjZNJyv4MB+tq8Vmtqs3Ga9Mr9AY1U5KfHaEjh+16kOe2q3XN+JtJ/
/3tL6MVe5GGvvOdNGgNx96AxfR1p4Ns3d8/iHG4dKQt+eW1UQ4ROCU5b5Eg0
cKt49S9sEwFq7RbrYt3Ho/UvkVSV9f5OJEDJPM6UusXVhXlkHbvv7m0CdFxh
VmP3DgLnwOIMmVQuT7qZ6ozHLzsa0PQD+jnn5sE3V83sI3Fc3skiW5dme3f2
dS0Wyb1vWQIpHdRD+7tJXF6/Lw+ctXzZcVGNfubFMgL8UqN3MsVoFggvbXI5
x0sH2GNGqi8eTKB+8BLgz7Q0mEDzOh6Fe9Y2fewHc8qmKzT7uX2e8c0HeEUG
WOAp30LV6/hhcO/O9Zk9hAHAW2VRnLplCblz4nvRmO48EhJh1qfSxp1L+PS2
p6imiInWd9rFTIoU9rDjje9fg9bepKLvjXM99y8oFl7vYcdvlTY/99X/RPM9
pMHFlHP5ZWA07qTnRC4O6g+LL8LCJdT/G2La1/j6LIGg7ODvqhM08MehPN47
dRHlffDXUylRg4vIScf94h6p88Di+xDTXGQJEVIqySqdIKH7PU+79Zcfan8Q
lSp8tfrPz2hecWfUGatNUdy+Fc73AuEutM/DQ+j3jevKbTBai0gDvZqxcvMg
iovHhhMHXWb/gOaZFtHV8kygpGUTKXOLAlabFCTfyCKBUAGPnZuxFDQvxclD
yRXv0V60Z8eJ06T6fZlkNA+qNHSO789xJjrvzsGFKjcbmZ0Hi8Ezt5Khh0X9
YGWtX+FJPANYNvJlix4ZQXwSp/QyR6iI9d7h4s3+DGQc+bWi8l4F0Nn1aqgk
g4Hy+h587GOXaMdEcaXMav57KTbs9ygvOmXR0IO8Oru8/c8u9rr5czm3s7ob
rfdIpNnLNJ2dB51KXQ3Ry5jI5kmr12HDpUDS4+3U02EG0mR9+OKQUBlYnvlo
dVUlGa17cepcP+ymRL9XURDJvNpveLE55MoxMm/IfhpicqA5cJUcC1RfZY2a
buHqiKh471RJUOeFjTHG39ek0rl6Z6GIaozHIoqPOZ+Peg2DLzouIgZ8vB5H
bNl2Cp0sllq4cxbzsjc0JS8RIe1E87Wv5BAweu/+VjVJInT9YlJZNf4N7a+J
nZAadmv8gwTbpSXk7SXAS63RIxktA2BXBFPo4xQentKszHlY9geQQ4k2OxPw
UFrayyhdpR3tt+HkOxJnvi+7x8DD6g77YvvvdSjeCZN5VHoQ+YTImlUU3LyL
hSlzpzddNuf21RVtOP82aicGzZtaCD/WWq3BjetSpN6ObbzPQO20wW6V6i4v
bp9vl+Cjb/UOXB1ihoJcmd06BnhD3ViVu50CbPXMCxndZPA695MOYSMVfa6S
aSGWy/AzYDXp/j59RW7d6vXKs99DvBkIbkpE6fZeCiK+Gu6OWjn6b31Ntv8y
F1g1l0egAKQysvDQ7lngHKtBl/tIR3X51lI9A8df08Fs/2llC3obOOe6uclO
kQd2v1QSbYtvQ/sVOHznqlGtL7F9BHiAFuL5LeElInbv+fuwrzzQ94+S/lTE
Alj/gYhRLMTC8YZo/tVXFtB+7Q8+vU/vrWkG02sXlM/U4GBq/hNvZzMs3JOu
9FItigbCH+69pKKF/Yv3CgsX+y9SpMapoPMofTA4kQm2dErtlP1NA6SHX/yw
n5lgZ2lnlF4PDYzzWbXyf2YA7zqzN7UEKkhLecpzNqcXxR98AS99xN/3goel
P+HTWiY4bTcSZAjmAbl8rbDaQyJsx3TUxd/Gwnb7pYHy6ziYXLn1kJw3Foon
dGXi9+KgCUZCNkWEBXhy6JNPFwjwfJnaBfo3Bhg87jI0ls/lry5MaLuYGMoD
o7TzpYcUmKi/DWmTDfoRTwMJcZjYcDaeJIWo2Bz0ov2bL52Np5teNUxYYghw
hcOmT/oyKeCFgPdm7Bc8nEJieMICUsC/8xnzoLTE9dZBPjoIlBGLXoqbR/cZ
Dn7q7nI53FDLg+q7WB6n3xIoJ6B2mGqzyWcVG39Qes07os1YaP2OE9//0JVr
9cknQPcnfA177WfButmR2bc5BKiaPdC4qWoWpHoVuO9O5+p4qgBNnN0mXvjl
Y0bhTAIGLu5r4nlQxQtTJYL/vBPFovzhpB3FmmdCGYAWcnFjaiVXp/7hqnP+
hw8TYciod36HDRn49kmKnnuBg9pWI08XBsh/1R/J4MzVz29lXHHo83jLfH/5
WQMORq4mlbprkdH69ibGVtndGuVIppepaQMbj3P6JzolFI74sfe3J6GBPKa0
iv/RdVuIn87IP0eAfGVxAqQ0HPzQ+yrS34N7HYKP7NONp0hg1QE+5+5HuH/j
I/bvpOTGRe7fRgGk7wVjDhuXkH0nrkfGrc9Bpmb3/el7vPg/OiH3BKM/3mHE
objaWC7xVPpOFjrfJozfYHd6BzueXxspErB1EJxvfRLkxPYrJUGlPyIn+sFG
bOtGliERdlNyXrkkzwPRww4OPCM46KW5eHtTFjfv9GhkfRolnQF2+hUtu6uw
BPwkDDZ1sM+LIW+yMKwl8LYkmv+3IAuhWn3yP3WABqivZ5tImDmk/tiFzCQH
9vfeL4r9xUhIVFxubYQyCwkOT8DdOJ+P+r8zy629YkcLwR/8CdtAVyZwOvyS
If+5ALz/2En90s0E+sv79m63zELkQqbXNO5dBNmYbadu7ctCAMuwal5oEbxa
Fnxq9YZ65NX4wsXEE0wQXMt8U57O5dOsjlTs9M+jIZjR8RbR5ilEIu63R2cO
DVHWfcfId5xGblFOnugLpSHH0koUw3lmEbt9i96UcBoChF/HmVBnkFMFamda
t80iDUEWv25epyGwskHESmga6R0cfbzLn4aMHWBIGuz4g/qPWHd8VXZ6JrLn
J/KKMMgE2qp0vd966Wjdk5OX6/vOjOgrZttl+rYTPiwyWs939z6aHD7PQkr8
6iukup+BsuUvqmbY+2LdLE5GhfkOCT+lTXLnHwMv+Be2qaZQgf8rjMh1m3Hg
e+SDZpcjez+w35D6qnYEpMhvWsqZo4LYmstGv0+PAkm6+YxLHRXUSc2fMmbv
d5zrCdT//FKiEABeTL1YJNsvEpspvtWLdej1+tjtuwRmxsCQp2JzIj8DSF8s
GmStHPs3/2b2GEhKGLJjijOA0fzRQGkRJtpH8x9+oJVMxHYGYo8NdKF1iT4P
O9ejbixkY9Lols7nVOCZoYD9tUiEtStu4TLAABAPUpTdXcILCaUfDuwSGgB2
Sb8tjbO4fDcPb+YclpHkhcWPKccvB/1A65uc9erutevoplkiFBULgsRl5Wh/
wobKxohns9nI+33Lv0l1EKBx4/TF3L2/keu675NKr+Dg2OlE/u0BDSB5aPk0
Rg4Hm/Uz/uxzrweCAdU7ljdhYXJZzGvP97no+tjZ5kTKv5wLXhnuTT/2aAlU
FL0JMqfmgXMB0pujHmGgrM/S432VecBfDlv9yxUD668sg6rVdHBA7FmpogIO
HqigH8qPYYDZPc2Mre6LQP5AY0ngUwbIcAtfu72XBTR8bP1mt+UBtdkR9RF2
XEf8KKu9SiEP8Pu8VpNnH5cZVmDvqeDgrcs3mCLhJOA9sXd8oY4M9vjEVd7l
49qbdJa8fTl7HStL+lvW4hbQ9/dJQxHusSWjeJ1jh1m1qieVv9HBYYPFS3wD
LOR461FFyStMRPqTiSmug4Ucmeu/e8yBiUwYFsoJKLDQfjJUx0XjEOhz4OoJ
yu65Ijciz0Li5doiZT1IwGNj3iPp1TQUN7/UC5KzOMYA98cYtRabSGBuJj/C
9w0DPe/Z91LBOYcKSu1KWy8cJAFlPTfdA2QqeGJ2hHUcTwLb1r7KW9dKBQay
Ir9aBOaAXIdn/DNXKlghluTqrNEDlEZ2WQhX8aD5nWvkM2Ef6TzQpWUtSK+e
BQp6OoYqUXh0P9vds0r9RSoe7ReYxtVdcjqOhUJXDZLX248C88i3N9NX4tD8
S6aa9xbdQSzUmlI+OpozguIZDh5KvvjsyRZRPNTx6k/7KrwAxk1CNk5ux8EM
gQq52q0shPO5t6zU9MyWOZDmn1xgcg6H8qdjN8b81NmDg607ij9Ff10AZ26H
aieLEGGjtnNv23UKWgfgXO+hg66HBhg88GHUO7rzcgbYLmBzc6MSAZImjqa0
aDGA2Y5XmbFCBIh9l6VGyGeAf+ulMYBG2sYXmnxs/CYx7BVIr0TxBxC68ejS
ZQZ4UPtz2PUQFvoUSbzij2CAe1XXMXc+YWBh1Jou6VQGGCtZewZjxJ23/H7m
KyXIgwizJoUbJezawEun2ClFClf3OupoksTYaSz8tF7W0HuyDcV1n+tEk0yU
SMBhm4TT6HIc9Bs4TDQ3ZIAL77J6BQqYoLk/+ixtjh1nXpn7aRzNQO24akJW
pTefDmS3Bco/r6Oj9t5WFx2W7IKBjvGmpNkYMghPH1BkVoyh+zQnH2ezrcjs
E3t9ChAChi/I1gGLEPcufn32+s+ZU3pcWwvI5geJIhJse6i10ilQGQeHpQxc
4k5x+51aRbdInPEgwBOev/dTltFRvBR8Sqjtun8/qGsJ9X5FI8C2H0tzvyz7
0POj9IikRqlMdN6Sw5/Hq8VyF6QR0H7lKiPLd7ZHCRDTq7WZr7YPtbfjd6ZE
zsw3oPfTJOBf0FeAhw9r3UzdBxfBG5zV41T28bUN/j4Vddx5y3Kj0OWKthi4
eedS8cUCru5L7bnWtIRMPIoDv1jX3WoXooEzmSO5N/yx0DT05d6sZAYav2TV
8M2fS2MgoSUSZrvvsoC8QKK35WcG8nrYckrNhgVcA9bt/HWIBhKBM9nTkAXi
Xn6WDF/zvzp0V086PCMv4+owl9ZZy1Q+mQd7SG8vABxX57L6fLDB1Wk62BV7
IytcmARk7xRsEH9CR/0G531gisZfqGlPgtXGfLbyunQ0r3a+W1c+eSsGWjdP
AtX5LrSuOZ2RELh7qheR/DH8fn1iF4r3Uh5ONH+4w9U5in+X8OSHE9uuf2aM
OIktgRLPjE8B7PND5drV9TwYeGLlNEZAh1sXjD84Zv9Yi21/ZVcXT69eArqd
BHmrrVkI5zxD4+OXbN8sxPHxK5axzxI4hvvtdeV2FhIc2WWp92EJ4Oo+NY4R
MfCOB/+azFIGaMyJsnEoWQR1uTUU2TV9iP1T9+0mK6hoXMfZx/Vll2FWNQ0A
zTdm4+Yh3LkXo0/PD+65TQVl+xLvecWMgdgAvxtrpWkgKH6nzHfbcRDeiAmW
zaSC0256Dm1DY8BeNye77tYIoq9/8pfoLBUZNshWp7LjsHdaR+fH7MdBcutc
DeUYOx768kmjAcfLxpOnom+IUtD13B0YeD/6AAVszYq5o0HDQLKpgrCYTQ/Q
ViC75Q/iYKyuxoSMWTeKxzn+WqBYKuidEh4mW73p6ZhrRf1Rdx1OcsepWnT+
JSm5+tgFTB2y3rgmce02tn/SyBTx2lqPPt8MhioFf60e8Y0YdZRYvwRGq/l/
SKfUITnCK04eiMNA3bCBE6fYccWXO++KlZlYuD18PPD0USJcR9v3bd8fLFyd
3SH7TQyDxvu53Qbj+Jcs7jyErJFc1vgSmPpwrNNOAgdNz/RpVv3Cou//jqN0
wEgFFp7MZ3mlHF0CG9+IL7ZALMzfdLKzKwUDXyd+LUqLxcKub7rKlRuZwPvA
/c9zBzHwsKTyQMrHRaB9co3Mjp8sYKu+qz1xGRZOjnokG1K5OkWEtetkssks
0F0dsGR/aQk0ZV2/xNPPAmrTnbfLYjBwXsX9N8GFBcQ9hOYM+ZlgcSafnHqr
HtEodPCwi1gEXgnYW51s+7WTzzcOiVoEbgtPTHIdiXD7OVef1s8soPPNxKfF
mIzWcTh4IaYl+IPmIAVc3Z+Waiw5AhoMjvnyZhBhjcsyr0sbOtH4lqMHdqa7
eWkblQca+0p+4TlJBgWD5vZbSex4K35ywmyBjPolzu9jb9QPy3Vy49m3GlVL
+uIEGPf15afp2Xm0fnMsS3Vp+2Yq8BVedva3Jw6Nlyv938obaHDrYRtP66tI
/MbBtKfVZ8tmKKi9HRXZr5DbuwDO33Wp0jfAQ8FOJ+Ld61ydNc7+WtQqn/Hl
AFef7Eu2iFyePQ6WnHjz9ldENVKaZG6uIoaD3yNs8cKZNYjvmpU5ns1Y+OpT
sTuxDAvbUtyJpDuLwFS8tVFpiAUOLr9VNGe9iO5Tbu1PXvfZU8BhpTI3y1V9
YLfG9toGUybYnRm0U7dnEOUH9xs5uvJBLhauEXz9IQGLRfdvLUHXy9dIPJCy
QzziKpWGXj8n7vebOj3zUQwPFd/O7b0XEw4+1rEmnPfjIfVnGvGHTS+6Xv1e
HS2KdKNyr8vvE99hEy5PFuf7U8mPLuddIaF481nY0euOr/HQ562A8VjJArpe
yfvMisTHqMD0dM0Tl/3s+Cg9SVhJmAauUdV3u8Yywb4NuK6rhEkw9F4298dH
Otg4QYiezZ5E/TiHv/8RPQLr/YaO6o/0pCTOrKjg8tG2r9N9xppg44KGH0Jh
tuOoPkv866PDG15NgJSvZwRuZtOBi+Kzg43OE+DgGr0NY2y88Fpu7fOGxzPg
mOjTcptSAqpPSTtV5sFDw0EJ1188eyJngJhA6qa07wToFaPosX05GaQpGKXc
UudB7TWC8FLpuDgPpE3dhdniPaDhukJftjYv6s/OVJJcpux5oauacVmmExGd
pxHOUAk4GJeF7kPGSsafCCSurunB/SGZwft54OzVQP5sLRJQf+xSb++wiM4t
5na0HD9mswTqW50SGCZjoGqw3jWBnwzML4FP7/0pgCxmEn9v7ici3Wm2PLN3
ETkYSVFQ7/iN8uIPRLQ5Cof+Ly9+QtNZxz1vFpFZz1Jnug0NCOpJCh+8Oo48
8t45kjVNRXVSOPYFJ/Ht9q/GkP0xrLWKJ/qQHSXx6gw6O97sEVI+qt+HPMVt
t97Lthv6qvfl+ekMpHv9qTDfdR2IsF0Xo6Kbq0NaThuEIjtIyO6mCMXJfiri
HKLv85iPCrYTZPctDMyAiT/LBV6LUoGI7DzJbnwG/Bo4dnOT+gJadxY0SL7A
eL8AqmxL3AoqqSD3/XIDphMNjUM59cgyx3slc5tmkV3rVFz6znDnPjh94aQj
8pM64xR0nxG9mxMD2HifOPjMp+gFFtXx7W0UTN5PJYNbY4/M1jMJcGHhbJN2
ORlcLAA3V/PzwPMffj4+lDYNakIE2xbL2P6rEcR4XZgCQS7J1UFse/JePVUb
fAELbTP+VH6lzAABvw8aVO15YE1D6Mor8DB8nUWDizoFbPshfpj0hACT9Gym
E/0oaB6YUz900fqzxcSLq88XNrXi1444Agy9OBekwd6P/7Me3xGgYrSEfvwo
GeE3fyxz7hQF4U/yrHRsqgCWq4/gP1lz9eEqBYe6chAeeDva3E/x5ggis991
LFKTB04XZktePU1Glj2QiWkhE+ENuxmL44x5NN4WjTpS5lC7hOJRYnxCj7wT
Ccym8nfNzGDR+tvlgse2kMR+rrH3Yg43F/+lW7wISnYI0isvUkGgcoxQTS9X
R8+kf/ytqt4S8m9/tIRIK6R/M7Rhr4829TZ+y0kwvljyNbSBhuoZIYmzaZTN
dDRftnvi+LsXBUso7jZoGP/pCnhgmSISFLWWB+0Lmxn5nPpTZhztB8s6ZHVo
HRxHvjlbxSmo4eHue9eHEs2p6PmO+6UFUY1UZPr+E49reDz8kSYVdqmVivjF
vJ9xXcTBhGeCT/vu9aB1wtphADd79AA1M+XSnBNLIMa8K1hxpBfFL9se3POS
ne5FJrfdfLhSCAOVUhS6jrgSUR7AXn2KcsVWIiQY54fqH6wDN532KRwf4oHf
HIe2fln2G0j5Mwx+GsyCpmy/oPtrF0DE+8bL2xdmwMIKknOAwAJg5l9KSZ5b
AGJbSV5F+jPAJJKg1vBsAcSFbARLa7n91rVCs84Zx6vZcZ3QnJI1C6wuzMzF
raoGxpvldkeYs5CE9ukdri0ZKG+BYfKUkGTiF7DiRmre/ChXp5DjL1vOLoUq
m5CB3L31+bq6ZPAZOb7p5lYy8L8j9CZHjIZIriFdMvIdRq7XuEheVaMh4nhH
v9aDI0jkuBNUcu9A96HplTWSSl86QFEpSVdgBwO47mxaF/OlDlEzd74RcoaB
/j++vHiGbcIcuNXVCtzvkkHeTdayFe/mgJ/l2IE/UmQQOZYw4nBnDmDeFb44
689AfC4Y3JC+WwH603rVVuxjIRKnk66FsQbQ/Ma1frEjRwIHEOqSs1oBlsMH
148cTYr/8yGnE2x5PvdTMIcOeJQbjAilnUC/3r1yNJMOfApZTzzrJoDPjL1Y
0mEqWHT+EnBDYxJ8VcbOsZE1qD+sa8WqmwRuehPa0spUwHtAr8JfYR55e+HF
2RQnLi+JvYqfitBBKrINGfM4XDSPzi9w6kCdnf6uzto0JO1NIUO7YhL8W/d0
FukUUxm5c5yGrPPDViw/RUDz77AoVdz7AgHeVQiondzPC6Xgui+BHylgXMx9
N1OTF0YoXS2eYuPsoe+B/D5GePjlyw4XVzUmypMRtjZCRm05E1HQGMw+f5LL
nyFtLgY91jORwIS0AE+dBYTqe4Vyms5Ayh9EihSEYeF/P4nQvqr/HTaSq0cb
8ib/wMf9S+Aj9hiNbzkRXtkrRCfOLKL6J5aVmJ5DfSxwtaF9y4sqrr7orwDy
o6pEIprfSdwU/3LFGBE2vvyw11p7FnjckEwzYx9HFJ6z7C2tQ/Ekp3893bmK
p38FASY4Jp+t+1YHIrMbrh4VIcDlH7xTfQ0Awihom5G4ToDM+dwHD98FIrte
CMWk3sNB65UxAcrxFojjimWiM3ls/NCrWUdMa0BY9W++3uvn1lvk+EOn7hfh
YUbP6V8ztdMARnSXXmQfx4b72TXdGkLxWOjofaXlu4ZA35oy9+drsTBW2f3P
ih+DaJ2ccO+j2cuzxH8+sdD1tn7Z4g4MdF6Q7u08wwKrzs1VG3lgoZ50fY1G
IR10Wn7BOjzj6uJx1pGnabv9RDITbKpx1Ty+jgU6d/mB9H46cB6QOCfctoDG
6zGqNhTh7AUgJLzxyoQ9E4Ra/OlNS1oA1xJkpUzMmOD3y5NXJ2yJcMeROUuW
FAuUYnI/9Wbg4SPszRPfrVjg2FlihbMEHsXlUW88bV1X4eGujLywoecUsMVX
cmL4MAFep7w0vjgyBYRsxMc7K1mofu8D54KdGytYyF7qBn/DlbPIu1vlp5Wy
WciJ7ljh6JezyE4/8R9GChSk+JP1qTtO3H298P6aVucjDGRDgOrl683cff7r
n9TpqdxhsOfDoROKR5mo/s6es+q3Zvawj532nslVnAe2fStFtvBTwG1/JbEP
YfNAe+hTTIU2BTRtcz7hp0pC66uFtcsaLLpJ4Emx5F21w0xUN2rBrG89ZgcT
1Wc3iJhc3+ZOBykXr+KEW6eA+/0fsiaeXD6LYgVYaKVPBYYlr6UVrs2C9eKb
j2YYU1G9PeYj4Trx5wxAXMda8AofBUYb4w6Z3mKANy2KtguXB0DCZj9h5Rg6
cKqosyv2HEDxNEdP+c+lS2+9jemoboFHscepj+x4beSNQ8e3a7GoXizDuM9K
p6UJsZcltX2/xYPqjfKm33YPO8UDM09oSUtVtCHLjDPNj/gSUB6qDj4L4lgS
AVrEG0qapLcinHwXddeBg7PnmtD4SwZK4NozPiIvn5ssDsdh0TnsbIOqeRM2
3hrNiLsZ++09qlfq91X9Gv8sA8UDkp5Pk8xDhlA9zoOpQ+9Mq4ZAtDnJcNsQ
E1i3HcVW2XHmkohoHsmj+avmhygiDHiSvyvaDYv2C2lY3Fqevn8etfe6UfOT
hdfnAVl51IWHwQSwgHBsnxsbnwUFtQmvYQFdQqOH1kANmsdyOUC1LV9XC5ot
13ccaGQBdZEc69kRFqoXpWFg80uLfWxhrHlz/8FpxGxVZXFNFwuheJ72Lzk8
g3xF9nro9rNQ3cFddtvWZPWxkI/bUuvC908ghHPpAxsqmGj9nXOdRi6CG6N3
8sDUGWr+dcUl4F4s5OuxDQ/HdXQmxbFLaDxWdtLV3+v6Akgt7BWobyWgPFjD
Z4Njx/oI0H+D4YAkH1ff2eA7sCbAPqBVdfVRxxAWVviuETShzYIVqubVV5yo
6Jzm7ZGa5GN3qYD04JX084ma/+ahvDCw1ozXyjGvBlhQf8HmCxi4oqm77aVG
DYp/iiN77yzp1ADH96fiJ78vgVDZy7z3T9QAppiILXbFEvh5rvAkkX1+ISyz
/PvFJfDjs0LIJ/8aUFFfNXvi/SJolWtT+5NeAyjtiZs/X2LjRr/VHz7TJkFI
Uujvry9wkBnI/4la0wsCpnulXmcQYMbVQtMtx6fAQnJD57tcPGz7AQXlIqdB
/cUvh48UcfVL0y4Rr+OCCTDq20b7votzQEHMU0QuigAPVP+6am06B5yK+i1J
0QRY3m8sdqiXAGPfFs3SjYhQr+qNMBTggbkqttMIOx4PWWcLH15mgi5lz7mr
brxoP8BbZ4UVdrG8UG2F5Ul7dpz03/XAC2/L9kyxnDJAw+JjYQt3IqTKOhJW
5RcADetXTfxtRCjtin12SDYLtWcOrlq5T3Ddg+9E6PNbdbxTsBoJyn/25lgh
EZ6o7+p4Y1uHBNtDwqAJEdXXThR86vr7LhFy5uWycyXESLvKkUWNivGC+0so
T0B4TO50kdQoImO753dfBzu+Cexa17+FidYLOf08ZrMTETfZ+w3Pj94FmYNc
nmNOvTpWXtNxwzAV1TM3kAOv1kjTENXTITOlH+YQYWFCldtKGuJDumIucrUe
7fdacrG4L3WHjur4cOq8LZ5bKEq+NLT+dFNjzZ2b4aFAE1fmFkyiAIltTQH6
u6aQUP/iU8fdKODawZ2/H6yfRlIuKr0frCMBKfURoeLHXN2zvaevfNdikZAn
KqtCr7wlAVpHlA+ugoQcfn6kJnQr9z505Ywrdwqx/cP2XmyoFgmhOUx8VX04
Bzblev7xECOjc8hZzwSjS7PJSPgwQTXKfg7su/dEl6zMPv8bbzc6O4fqrCl2
4QbefZgDksd3zzlMzSN6kdNyMuXtAJbM3fuQyACpw7eV5VvqEMtW/8LyNAbI
tPp5fXNKPUhvff879TgDhDoLHXUvrgN3xMd+nmIxwLEUp3ehbUwkZCZxUry7
D4l+fsksY4KJSDarPlqj14/c826ycZhg2/uASvlZhIrqmtmdKHvX4kZBapWW
Dz5bPgM4eLKgtOnN+dkpwOGz4OCXPrJbb9IFKvJx2aYiHXEC/GWS1fs0mRce
XN0u+jqFiaiUtVvo8ZORDc6ztkOPmEhzX9Wmk9ZkhJZee15dm4y4KAytiHZi
Iqd34ovXGJCR2OKt/Ek3mWh8Nh3pHy3xhhfOrru0YpdbP7o++jfHnlMZCuD2
UfyTX6zWzLkoMomDOfzLLyk1NaB8TJz1UUc1s2qXXwSX3n879ATBQx/jBOAW
TwWbw/zEyb9mQPs11yH+3VTANFG+wiuzgOLLW4YvBpx/VwPDQ/plt+LxMCZA
L7yQSgeCVxaJYekMUOIyY77qIwb1oxy/tzfIZqqzkPwf/bDgD2x7eEu1sztP
ResUHsIU0or7eGi6nrbjvsUcKDXIb44KxMPJCPk95HNzQLFwxmQkAA9D69MR
StI8Godx8lN3Ijw/mGxgAb4vx7YGI3PA/ybLNpiPgfrj/T7xbkrjdEBSS5z1
5plD82Gvk3a9PZE1h8ZPHP3mnOf1RtZrGGBHfjK/m+oEcBIx0+i9zeWf//ji
0v4QSxw8M6xL9YMD6P7DyYc+DGT4LonyoPPGkjfWvJuL44Fxfo+u9QVXovng
+4ULpmN7h9H3Y6fl1/lMlYK8OLXr6CIvD8QeEVkZmEtB+/vubTdF3hT1guPH
+9bf1cPDztTS+r5r/cC/0egY7xYc3Hw3P/+KFu3fOuFsvPXsDPHjSDkWvtay
uh6vyUDza/OXQucUxFnI3H8+aUAu5fkYIZSF9m3fqNH9bbdhEe37HLdp77bY
tAjOyunbqOEpQDzHcmR7DR1Y6T9QAqQ5UOegnS4JCXBF3oGqlXtnwZrDC4aB
63jh1Y09XxZS21E/jSnW7kiMaQLKcrQWQSaXV8xwCyJV6o6DXo1dGudHBkCb
qVxHFY7Lo8qx86nxRY0Il2Ewsb06PXBPJ3JSpUkA8cXDbTePXNpqR4QFr1+S
j33BwObdI3VGt9hxU3z1u6eHMbA0/fOr/t0YuP+4of26Xxg4OileUn2/B/wW
iwqTImKgxPfKE/h7PeCzvEv1r99LQDUv9NFx50zEq3G6dDgMAx1cXXSRdZ+R
x0orMo3tMXA7KUycP6wXOWuVvnZrOga+qd+ZZ9nci/wcsgU6WhgYHg5e/9Qj
onXRFW3dtqvZ64h2fUdmbxsN9GcXMgQ98eh53VM5cylX8DD8rTQyXUYDd6oH
Nz+0w8MdO2eevWXQwPAFR/H3vlTk5IEt/u5q3DhXWU9Q+VfrPMqzE5t9wl/F
m4S41hFr3n2mojqcyyUu04prqchQbV2jy0HunL+gtmh9TDkdiZHUwSBBNDQv
SJP8FTk6PQueaq7aJaS2AE4H1nvM3J4FP7+BjxYbFoC2osk6dQUK6JbQWiZ4
bQYkxiz9uf6eAoYf6OcUJMyAlb5qmzpbBwF/ccQLhwkaONXzLLDy3BAY356t
Y55DQ/Xh49et+IWj00Hey6TdR590suOBzuHoX3Sgf6uJWmI0D74+k+fVyJwH
pYVPU5vquTrYr7vPXfmmOYPy7th9b0uLWz+DNEe9WOf1//oVqpJPW9l+KTx/
peud0SmgdhWrvWMbE52T6Lqfu9L9MAPl1eT4AbkjtnOhRlPA5zf+q+WnQSRj
XnXV9j4mG4d3TGnfHEbzHhw9WUyKxrNbRBaScHj/9sfSIyDoBbVKaZKJ8r3N
xexgFd1nIW6SQapZmFG0L3ofvVttZtUEOKzrq6UVQUcsIm+IqjiMgxIB7/7i
JTryqt/0afhhLDoXx1nXHrebTKlyeBgn6CWgfpaTf+X2zRrpFX64ReCFFc+F
HasiWOi8G+e+HfPFJNeTiHAX2bn8+5MZ0NjDCKDNcOd3xQ5dXfA5RIDLRKaY
Zi9HQHPiQxFXtv+ROf3xp9erYSBzOMiNYUKA9kYPeCVzB8HgFZ7l4f8/T/UP
nhKRrGp5eQMLHWSOOx0g9aB54w0266K3ufUgOJfzveU8ONjcX3Qt26AXGdLw
1AqQwMLb854d1+R6EcmiDRTxo1goo/trK60Qi87D5a5Ust5VgEXvk2fmATJx
D4uun+uh05r+ZSx0f4i8h3dWYjHReI8TH+i6vq5axrZH55fXntBHmEDyVS9R
/iIdNNmFWh+NGwSef15/r5BYBMtlt4TVHWEB0p+2a4o5ZPR3dYLtNTdPkMER
rLFFg8M//2+SDK5Nxd5IzcHDBduidgfbcWBip3v++R4uT9q+WaHK0FrGXzrX
C8jUcEXJljwGai+WJ7d2f/rIBKL+kh0m7/7Raz/L5dGI5fcX9Q9jgiz+ZJHZ
ORJ6XuODCdOhpx/d34x2UXyPxg+hfpfz+5KRk2YWkUSowne8blqoDTy3zyx8
nkeE/Z0mph2XWtC/5/TvDKVih1Rfse3r1FeiYXg7ak+bbacYtPh6EDI6/Y6y
lpuH3fxAPUfLgAizz8sWritrQPFDS79CVUx4Ldqvw8kv6Uvl/hpg8sC3q0bi
kwiTYD6TpPQ0BQ+HdUS730JOfQYHp4/dYV6aw0C6lH/f1GMKGMsyI3bWc3lN
doxsPWOUh0XrRVH79vrELGDhXJd+XtrPOaDuNrRt2wwWJpn+qF3InAPHLvLM
b6vuQePVkY9vah4l9gDNt8LG8v6LIHGt04bTKT0gKqqr9ZvrIrB6nFwWJ9oL
VldXX9F8ywKK5/lMdrD3Ic3zb7LzT9DAfotNUtZrR1AetvtWTlbWQcMg9LlX
iLAhDZzMrb3RvTQENoSpkT+70cAWR8EYR/4hkIcJn/cOZYEjyj4y1xcHgTSt
f49lPAtQfjiexcVh0Xo7By/Fxidfca0mwrWV3QNYcRJgwC0Xz7HjHb7ydKf7
MnPAvGH1Jtd2Iuz5z4MeRU5fCVz6cZG9zwf6Es4qjyDFLVcqgQYLCVP/2XWc
fxRxML6nd9ichaytPmaceHAMWWc9wZy/zUJGhyTu6i6NIsbybmJnbrCQ+JzI
Q/MCLLBDJrPTj0aA4TzXawiVMWg8vSvv+hoQVwqWwupjRhYJ8H6qVNIRkV4g
tGunTgD7vZV8411vmkUFa6hXvl3OZwDljdjqy2xcnxBiL2j6gwTUH741fs7G
mzpPLlS9+ICHFn1poufWUoB347eyM2YscLREew2D/Z45+Q33q4pSu2+OA74L
gvn3JujAS7jx5FrZeaC5zFb+Kft5uTCtlRZb/uFjgFjYemIt7+uZebAlZZhq
t2IR7LTQY3Q2zaP1ZIdbNhUvk3pQ3Bqp9+e4e30PGH65RzSajSf6Pgg7GD3t
Ab4FiQPxehh40mcbvfx1D/gU7YytPoOBngqRfJuLqGj9Zr9Unb8ImQoU8Op+
fElYWFC+cP5BBxW8qWu+kl6BhaOtfRI/7QZBNOZrY2kwt18s6HdCrXIdBnZP
HpNa/fifuZ5UDOpHlBxULr9m++OLfNeuvv8xAhI0rn4MNcJD7820rVHtw+j8
JSc/9oTSunLlBTycudOytimZB+1DN9Re6jntzwN7rh6541lLgNeaN+R1W5LA
6oTX9pudCVDqy3wHc3IO7I4TeKUXwOV973y27u1UHh7V125rvfWimb3eOXYw
GRSYWOFCA947Wzwj4hZBrHhBfMoUBaiStxlKXaCBlZF0xp0fs8DtrICeQSJX
N5VTP9VNkaJclp0Dn5KTf7itxkJrYc+A5zu7AOLimpX0kNs3pxNQcCueQkWa
UgmEX5okRD5wD3+YGNt/tiBGXelMwCA61UfM13L9/p91wZGva0HwHv/LW2VZ
aN7sI1bs+EmtSaRQxbE38M80Gm83khh7J15MIwI6LzTBmUXAqzH0umPlDPJ0
2Jqlt4cJ0r7Ldz9hx70c3MXBWYX9187IvePwzVEBvve4U9TmWeBC+XpTbS0V
nSuprZe4JLcVB8mSqf638njQOqrqpce7ddYQIScvVE7euOJPLQ+UvhtyxnVw
FkzN5i5z6iHC7vpWn2ZWPZi4N+rSwt6vEZPjtxscKaj9MtT3bw/xpACf3r7R
L5oYuEYMzhi5UIBZ5MPivIsYWC3t9u7ZBTZee1lkWM7Gr133L2y/Z0UBPzKS
N35OwkAdg2Hvp/l4uN11p5yb+zhovHZa2QpDgNS5b16C0pPguwD57JZH06DW
KvPd1H06EIt+Gf1yxzQIvV9x0tuNDrpumdKHYpuA64aXLcIJBHR/UGp9WjHu
TYA6R0j2qsZNaD2WHBjfmJzQjPYD0v9zI02goKxqw8EsAszGklpE2fYpu0rq
a8QJ/H+fz0oSmA/+vXKcjZv34pbErrHX05vH9ZaOOTi4sbREdufA4H/3s3pO
f+MAcDhr9vlTJA6q+jhHTG/BoXXYlvHennNjLCSj5XtxXuMkoj9wC/uxgIVw
/Kj2guZp23oWUuP5kXG1YgxJnBOZoY2ykJd1+oyS8EmE875Am1844yPuv/nV
DwvAMGuuxLCUAGf2ZKwW72CBUa+iH5EvF0DYhI4JKYWF9o1w/NCVdZY2n5wW
0D69a2dfMIJHidAnyShqa/w4mL3/NelPNxEynckdIg+4fYJG8nPGOlGjQG/7
Ld4tjwjwbLRpT/qrEbQ+zvEHzxKSMK2eBGh8eq2GUPYY+vyZ326KrlesBU+/
y1WuycTAeHJWIlmmFrCEDyzu+sS2B7lV+rk6i2h9KDpZxFCFvS9KXE+dTu7l
4tOJ17hHXk9n0L6wlU+Z0tigWRCXrGezxw2L1qE4fnxp75mvrTfIwFxY+UB9
CA287Td+cgjHA89FZpQE8vUgSl3iT8Vv4qCYa6GNpPoUcqGp42sREwOT4o3e
5prQEU3FrvycbgxUdGaeXmvch2y5oAcybrJAjJKodDj7d0yKNxrM6cyBeVNe
Ob6kRRDvuV97JXkEkIcOHEwLIUCNjN6tXTvfA0zgttUzbL/JFxNmiu0uQeKf
f3a8nYmFY2OOqthrjYh729gBB/NMhONP2oyrTi3smgUcHPdb1OoPeXMh6m8s
w5sTtZ8VgdsbGTzvdJlofBGrd6I/qK8f/DzOGyZ0hYbaF+fz2PDKnakSNGTl
vYKoL84zqB/h1H222+xLSwpmIapftL85HrmLcOoHWQFXsodtGWj8/1B52clr
Bgz0epCu6eUHdRbAdlaZVeQdKtAzbZj4Jj+PvILX+C2ukRBrzBbDzpg5BH9y
RNAgeR5x8OZLdrImIYfL8tUt5OaRlSLLoox0aGjecpvamSC4iYz6m+i7EooM
DTKaz3ZKwMxHMnjQ+pahtHKYHjuOU9tjbflzExZy5ie6YsgP+h/i4ctPt0/t
bGGi8wyc6y5x0xA1OMULOfxle5CcMMtpHnhy87XnorQ6hIMvOevlLPaFuac3
Di4py7ulXZr4b/7+Lg7Ki4uJaQvMIQ7LVdz71GloHUJcTLrd5f/1s8Q739xq
m0ROyXsJL3rTkJP6EVti7y4ANb7lF+68nUE0PyVfqNtKBnQxfzEEP4c4+mNd
krp3I8dUhrUt9Zgo34u2xbICu4c4NN5SMavoefIOB0UcEoK+CiyBTD3rim2/
G4Cxy6pjmqJENL6LLv/wMDJ0BuHg2Y+VFX2Ps+eRXYXIbGrCHHLkimD1OiIF
+RNgflHLeAHZIjASlWCNhZz339nO1/7ckju3/WP1kbDPvAPg0PUKYZ/TOBiK
wd7ttR9AnwenrrL80uFjj95w6zIn7pl60+NZCCdfV/Kz6M/Pcm6/WaVS5Un7
GDyUPXdOfs3lRfBcxOup+T0K2gd17sPBPX52NMRsiwLRAJ+LDHcXPz7/gYVw
6n0557JNf8pSQYzU+udhdu1IbuQJpdYOOrJd6M2o0oN25InOFdeqHjoiVPRK
Z1tVJ6I4UbeuToeB1ATkVKekUBA/SrCfxGMyMnyiS7pPnYL6zTsOD9QbGqlg
tUP6/Y0WXL7Oc8Kh0enLuhF5zxONHU8XkR0PN79NW4hFjFW0+7tsBsFa8qnu
X3aLSOPn0jB9pyGUx+BE9J9bxmqzoCbMq3MmDQ+NkCqZT79m0P5Mzj6e1EP3
Gv2Ch9aVIgsjYTS0Psvp14s8POFMfIqDN8flxc2cMGifHN6/JGX2Lqc/dxEU
PFSTSQ9uQuuNl5ZXrMI+rQWOb+Zz93Sw9+EqSr7UZga4YmHSwrOVCXiTHpLi
vmPROj/rZrzg/h4WGs9QhvA/fQhLqA59hdabu+sO1qP+K3VTzO3xFBoizjLv
Kbv0D4/Rnhmw7b5pZYY7HfU3CvmGqlRHGsrfk7fi3PLA30S0v3Q0V3TPJPt6
BT3nZmdTlpCEFfY3b4aVIGb8RZjRPO46G/KL1HY/SAd+w6FZ1teJ8PDl1quX
9/QhjJE2odHHdOC6febCSQcy4PRlcOomL8eeZxpazCFBvlV2SvIk0C46++W7
0Tyy7iqUzUhigCjS4ctxpFKg/J75Eb+tD/E9vlnMaicdZOWe/50RygBEHpL/
49BScFrbnD7+IhVptzokfPszA0QEBBQHMwG47hQiPBbHAOMY27NryXPAcINe
9Kk980DufntuwEY6yOt4+nyDa9//8A/8zV/293z33/7hbz4hTt6Yo7/99/zb
33qMf+dT/p4n+zs+/ru/++/59b/nC//WX/sbj/w9L/e3funf/Ud/5//+1pv9
Pyy+OCA=
   "], {{
     {RGBColor[0.293416, 0.0574044, 0.529412], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFl0ts1VUQxs+ltPT29r56CywoUBNhYUQQRKOSUF6piTEmJECNsCkCBQIt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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtk11ozXEYx58529n5z87b/3+4IVFIQsxGuNgopVwoF0Zy43XtxBw1L0Wy
LZHZS0kRYy8im00uvCyuFJe4ENtc2OTtgrxdIp+nrzpP/X/n+T7f5/l+f89v
1vaGjfsmmdl6opi4GJndS5qVF5nVBmY3UmZLiJpSs3Uxs9a02e3QbHnC7CoF
Y5xHiV5yfcRbcvPBr6W2AKYia/aQ8xPq78fNbsHfT3TTtB5MG7WD1FQl/nPS
uzJDTYnZYfj2810As4jvX/z3CmwlnA/g2gE+l9MM3nsEvjX0+sJ/MyZLw3io
mXyWA8TsKWZfyc8k34/ek/SLcqrdQ/4T2M9EE9r2MmMF/V+H6rUB/NKsNLkW
n8F7rU6Jy3u61tPUZOPSPAR2If/lnYsZO8h1ZqQtw3ztzN6Rlpc3TV5XUTMc
l+cB855nxnpyLfB/JN9DvcXleTn5FeAfc34BPmSeD46JCdPjXkfqtQ2Ng3wP
w9fDPPlAXsyBY2qRPHlHfiKS9i7yebATobxbAGU7uQ7iTqk0/WT2H2nNnuF+
noPdTQ0/25zQ3T2lx90y3WEXtVeILYF6dgMciLQLjdS3Mvtx8FvBTwP/nnM2
J22u6QT4g+j/XaI7+gb/97R6u0cv4WqjZnpMOzgPb8aYqY5ZajifovYRHNdi
uqOj1F4H/4f6ZvhWhfLcvfY78N08C2ZZQjtax/ezUNp8xl769UXy9gj9O8mf
Yd48uYJJeyEp79yDoVAYzzlnNb0vUL8pUM0h5mtJ6q39ZabF4EfJnwvk+dKM
dtJ3cVeJamtSeqvO4W/HPXUv/Q25V01JaXPP/O0NRbp7f4Nz8acRTC18A+jd
mdEduPfu4Ruwl1LaXffMdyfwnUpoh1Zm5bF76zvoXh5DX0uxPC0DO4KG6kA1
3eTHI+2W93RtDRm9bdd42ftF8tJ3pDkpT9wL53Su0VBv1Tn/AdZSpV0=
         "],
         VertexColors->None]}]}, 
     {RGBColor[
      0.37797123538561295`, 0.20442311892575815`, 0.6482646122357866], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmXmQVNUVxl/PMMz0dPdML2MFlyoDREIELBZlMwhotKIJJnGrCEZkERji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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlmuQjmUYx2+W3X3Z53nf533XpNGXFGFoSE3ogA5C6UPRDDo61FZCpIyt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         "],
         VertexColors->None], PolygonBox[{{4240, 3512, 3333, 3376, 1560}},
         VertexColors->None]}]}, 
     {RGBColor[0.4580053998557777, 0.34358091651257827`, 0.7607623123723553], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmgm0ldV1x7937xvvve/OttWiAWcgiqgYhJgE1FQJVEBUkkBiqoADyFRe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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtl3twVdUVxneSexMSuPfcx6EtliqBOkBVKPhKwWId2xHiOAVs0WoYeclD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         "],
         VertexColors->None], PolygonBox[{{5415, 3495, 896, 3362, 3361}},
         VertexColors->None]}]}, 
     {RGBColor[0.5380395643259425, 0.48273871409939845`, 0.8732600125089239], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmgmUlNWVx79eqrqrqququ6qalmhOgiub4EAQ2g1Rc1gEQmRmEkU4g6Bz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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtl3twVdUVxk8e3NyE3HvuufdCQNoZQiAVUqADRh6C1icPIVqsrRXsFIh2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         "],
         VertexColors->None]}]}, 
     {RGBColor[0.6037092719961173, 0.5916550035389223, 0.9104080740871237], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmXmQVcUVxi8z781bZt5785YZxBUIGuNuClRUVJQZFBUsZTEaRxZ3LbWM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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtl3uQ1WUZx989u3vOnrN7Lr/zO+yCpC7OIkqk0iCIBnIRtJFLU4IKisuC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         "],
         VertexColors->None]}]}, 
     {RGBColor[0.6625529021381817, 0.6862003437629793, 0.9117495276135048], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmHuQ1WUZx3/s7jlnz+65/c45CHlhUDFBvABxyQRbNZtmZLIaYuUiEIQg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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlnds1VUUxy8tbxXe+L33Kg6mEwEBiYrKkCHGRKNikAoKxRKmsYBxoInK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         "],
         VertexColors->None]}]}, 
     {RGBColor[0.7213965322802462, 0.7807456839870364, 0.9130909811398861], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmHuQV2UZx497+V129/f7nd/ZBcFYaQETlDL/wcTMKcUknEmNXfKKMIly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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwllmdslWUUx5/29o5eeu+7rshoKxQxjCIhJigorgRihEREBBVkNGKwpVAE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         "],
         VertexColors->None]}]}, 
     {RGBColor[0.7784452230078847, 0.8522716219882353, 0.9069462422641089], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmGuMVdUVx88wc++dAebec8+9YIGCrcNjtEpi4EMLCQY0KFIaBXmNlZbX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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlnuQzmUUx59td99dat/39/7eRRKmDBrMlPij0jAuw6i2cb9GdtEfkVsu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         "],
         VertexColors->None]}]}, 
     {RGBColor[0.8312907966554459, 0.8698942126795219, 0.8832714726266442], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmGmM1dUZxv8zc5e5M9z9Ug3K4IDFxAXIKGLrluq4wIgMAyZaxQVS7aJg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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlnlsVVUQxk9fXx+1pO/e996VGNGyBhKJQARcE4xQXMCyFEgQZXHXgKBQ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         "],
         VertexColors->None]}]}, 
     {RGBColor[0.8841363703030073, 0.8875168033708087, 0.8595967029891795], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmXuwllUVxt9z/S7nO9+dGUCRKBKdgTimICmHQyjgBQUBEwwE00RLoQIz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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwllneMVVUQxs/W9/at77779l0iYlkxWEiIEAVEAqiACyiCLlUELDEICqzC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         "],
         VertexColors->None]}]}, 
     {RGBColor[0.9411759999999999, 0.906538, 0.834043], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmFtslGUax9+ZTkuhM9NpO7VgDWQTgVJkb6qiATysJ2KWVhdty0ETudBd
8RAXMQG58BB3NZuY7EZEIVFQEuXGC6Vk642YgAYPJSlUqIImxdjTYhEKmk0E
/z+ev+HizffO+z3vc/w/h2/+sObxvzyWTSn9J5NShZ5nSim9WE5pu37/Ukxp
dkNKe/ViW31KW7WuqE7p89qU7hVdd2VKP0zVmZ4fi+bPOt8imutzKb1cSKlP
d2eLZ0c+zk6L57iEza/RO+3bRf9XyVqsd3t1lhWPMzr/o/YLtD7S/pj4LJes
ddrfrrPbtA5Ih93i+aHO3oVPZTzf0fohG/wuaH9ca6fW21q9Wlu0HsiEzp+J
xz7RPaf9vypD9lgubLlMPI6JblBrj1ZBfuiUDo3W/Srrdpf0f1j636B7I9l4
JtO02kbswJ6izvv1u0frXq0j3ndq3SMey7VmTwve3MdPJ6XTj1pVopmQvqum
xO910idXHzRNWhmt3aI5IH9/pvWCbDpeCr3R+YTuDlVHTBdYN/T/f1VKZ8Vz
pn02URl+a7UO0HSLT7tWF/4Tv8GGoDuruDxXDJqvtAZsy525kF9REXyQd1Dn
Hegjfx3W+UN63yx93shGHG53TMd1dzQXWJqtd0O6t0PrEb3bKJq1ej4vmTXF
8Cf+gBd85unZWBO0r8g3r2o1aX+FnufrUvqnaBcJM7PMd342sPaW1oaa4A9W
L5eOd2CjdNgkOZW6/3M2ZEOHr6v1nKr1fibize82bStrYs/5HPE4UAi7kImv
8fOA/FAhvqOZ0GmD5RLrCZ09oTUiLIyWwx7igXywBEbJk9FsPD/2frg6fmMP
8iutz63S4Ratcb1fqueQ9Dmt/XWyMWsdrpWcvxeDT59jvzwTepNr5D5PYkRc
iM9t3v9XtnxTCJ2JP3L/reefJOtmrRHxOCKaqmL4qU9xOFgXceMd+qP7VdIh
b2yQAy8JX2XJuacU+AN7r4lHv2LRWhHYoZ4Qd2RWWe7qfNzblYk8QQ407HmS
R9BOqQnbXxO/h8rB7+tC2IMtm3Rvo9ZkJmzFZvwwYnx+qfPDWsO5qDnV2ahF
1JdD4tloPfHBwnLYil7fZ0O3KscInJC/I+ZJrIkBtvwuj7wmL8ZMg/xx7ws1
Ue+gJ643VgVux0xDjB6U3ybl86enBOaxi1pKfm6yjcPmf5nfP20azoYt627F
52gpahz18veaRy3AVnQnJjOqo85fq3cLs1F7d/o3+3k+n+Ma2+uazG9oTgoq
/5gauqHjDJ21aR313Tbz6ZZvV2jl3H+o6dTzL3T2eUPw6CpFr6JPoVeTdbsS
TDhe5Ax2kC/UHGjWZyJWFeKxhvwTn21aT2n/QSn6IvGhdiEXP1MXyGfiNen6
Qm0B3+zX2Qf4Ahtfrw/8gb2Xp4UdyKRGkg/kAr4HZ/h/Xz7eow+0FaZv0Zqb
jRpMT3iyGH1hof05xzJ3WO428dmqtUxrzHkLfk4J/++VApOntX+/FLZwf9J1
kvxalQ/duuTTpbnoL8wLV3tm2KP3PVortc7p91n3aOg73FuJ6aD78v582EY/
32GsgAd0mSiEPtQDfIS9b0pOk3w06lo04txEHnMAtYJeh3+2u2aOuNaTG+QI
tXazfNVbiljfJ7zc3xA1kHs530VXdCbHebZ7f6EufIR/wN4+506v54WrU2C1
1bhdYdvxFb0o1QdWDnkGoN4+XBe9Bx2nuo9QM/eYBp495k+fJb7znAut3hMj
5LUYDy3WgXPw+10hMDzFNRPf4g/qIz6hT810Dv4qP3fVRj8Hg9OzgcPF8tOS
hpgpqD0XjB9qHFgH5+TRWHX02ddLkT/IbXPs0RmfLbXfMp5jmIuIZ5PnGfrG
9bXhh049vy3FnVOZqLfkAvMLOs50j8UGZNHzxpzL9NWTnqMu1sjqsBlcJOf+
As9ozCrk0LPOI2yYZZ8whx4vhI3EtNuYGLYd79h28os6wztwCf43yGd1jTGn
LHMOgvf2fOTkMs+zp1yXfpKcteWoteB5wvZyZ8i5PGH6NcY5tNh0zv4BM/Qd
8ud7Y3mFdU6lmL+Z+6jfPzuW9GRiSS7M92xB7tB7D7mvTXe8wEN3bWCFGPxo
/cjf/0n/87Uhe6ttPG17O7wH//ibuDMHkRMZ17MW43azaBfnI997XCfwJz3t
poboVcQZmiHHfUk+zla6FkHPk9q0yzrgb/DAN8qs+pjBmBHafY7OxBJ7sGuZ
40UM5xYjjuCCmfIpv4eOu/BrNs+NnknZc4c9OOO7p8P+h2aD51ti1OVagU1L
bDv7RbYRHmvNp+jZFD8uFb8LtZGX+LvdfsaHr/j+ecX920LkBRjBn+CHeE53
Dk667oMx6jQ5S82hpsx1vsxzbyFG15SjLpyw74kBOqNTxjzP+O5brpljnk/2
O0a7nKf0ZvAPnuj/F+tRIb5LqcnU7POu28y228shj7pNnqwxH/bMcsy5zNyj
juOVngNXGQ8rjRVivd73TvgutBOZS/TQdrmf0dfId77/mI2Yhzg757xbYtzC
u831mdpM79paCv9QV/uNf/rwEdf2GabtdG1v8W/6dZ/7BX1g0DTk7rjn9w+t
w7v2N34nXouMAbA60z39pHtKv3WmtnZVXpoD55uO7+jmbOQ1WMu4TvbbD9z9
Wzm+Dy/Og/7mW+57x63roHm9bXu/sm7g7rD9AM/D9i16goFJy+nPXJq3ey23
WBNzMPHlWfRMXOVzehOz9oBlNXvuxBb8sdn51WQMkC89jgX6I+eY/b3T+nO3
3fUfPBy1LZ2+N+C7B13j2FNX9/p7p8fn2NvhPKX2MD93eP+psP1JQ/w/Qe3e
5xzptf3kCP/FzPC30IR7+BPG6mrT7/HMtsv+qbKvBowndNvi3MCuU84fzugd
qz3v9Zkef/K9eJ/npUdVhx+pi967rhx9Ghwttm/BXbf7TqN93+z5D5v2W7dn
ivGfwkfGMDYSX2Svct7t9qxIjvM9yvxNDXlSctdrtU+LWrLffYIZgzP0+Q2T
i5Kd
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlMtvTHEUx09npmam7Z25M3MbpN1YVEPEyquJV0RigrRCaUVCYqFBEalK
0AUVFZF0Va9a8BdYYGXDQtBSNhLPtlQinZRMS0RioT4nX4uTnPM75/s9557H
XbD/2PajMTMrIgmkLm9Wj1xMmjVg16L3ZMx+E7Q4bnYna9ZeIDZt1l9l1hya
tSCv8bVVmJUisxH8nSmw1Wa9YGd4W4r+EP8y9OXIF/y3sT+hTwdmd9HTxOwD
+xTphT8WF/Y8HJmYOBrxDRP/mGKH4RjJmd2gxjr0DjA7iP+G/1Il9VNfNdhT
YBalxfEM/QXShx2H38DOwnG/Qt88FPzPAd8QEgc/AWeG3CViKonv4m0qpp68
5dvfIIa+lpqeg+3A35NUjnneL2Qzvl+8PcE/Q47LlfrGVzl9g9e+B39nThiP
/TNHXD95O50UZwP4hQXV9oj6d5K7NVTv2+Ecp9ZVWdWapOZZdMO/mvxfqbED
/1bemhLqWTkQxmN9BivRVyAlYid9Rui33CbXFPZn4l/mVGsZu76W/sN/EuxA
DRzo90LV5j1dT60fwXShnwEzhr6xIK5u3g7DvS2r3nlP3uG7ntGulbFb8B2M
pPs3PAhVg+deR38GQ9Xgub3GD/gOUV8Z/Tj8V5nXlbxyzQWzCf4i8iOlmDb4
h7AvpDWjUfAbCvp2r7kGfx/4KKEdb8pqZ31Xfea+G8Wseus7sstnEWoWtTFh
WyPtonNcw77pN5bSDAbRt4Cfn9LbbnJ/D3QbfmON9GIy0i35Du8t6Eb8NnxG
5/BP419SrZ0cI/doqNzFhLgH8vp2z9FNbFckX3OVsGczuk3n8NvtD3T7fsNr
sN8H6qXf7ESgGXjvvYd/s/on+L/Ab264oBvz2/Ke+u4eyGj2vsPj4I/ktNu+
M37LJyLV4jf9DyAzoKA=
         "],
         VertexColors->None]}]}}, {{}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{5346, 3912, 4095, 4096, 3918, 5224, 3919, 5225, 3735, 5081, 
         3736, 4818, 4819, 4629, 3708, 4613, 3699, 4612, 3700, 3713, 5360, 
         3712, 3573, 3714, 5361, 3715, 3595, 4985, 5393, 3623, 5352, 5353, 
         5069, 3470, 5070, 4639, 4635, 4636, 4606, 4608, 4607, 3570, 4274, 
         4275, 3832, 4901, 4902, 3529, 4903, 3530, 5236, 5237, 5235, 3926, 
         4318, 4317, 3597, 4182, 4183, 4043, 4181, 4180, 4184, 4174, 4175, 
         3853, 5164, 3852, 4742, 5204, 5203, 3564, 4094, 4014, 4952, 4953, 
         3565, 4339, 5256, 5172, 5173, 4924, 4925, 3546, 4280, 3547, 4458, 
         3866, 4935, 4936, 3554, 4282, 3553, 4558, 4559, 4557, 4761, 4760, 
         5223, 3916, 4759, 4763, 4762, 3915, 4756, 4758, 4757, 4281, 4350, 
         3550, 4932, 4931, 5280, 5279, 3863, 4250, 5404, 4249, 4343, 4251, 
         3551, 4461, 4460, 3864, 3928, 5373, 3927, 5065, 5066, 5064, 5068, 
         5067, 3687, 4595, 3686, 4596, 4928, 3548, 4179, 3549, 4930, 4929, 
         5175, 5174, 3698, 4957, 4958, 4959, 5063, 5062, 3569, 5227, 3920, 
         5226, 3921, 4312, 4311, 3590, 3897, 5344, 3568, 4270, 3567, 5343, 
         3896, 3586, 4309, 3587, 5346}],
        "187"],
       Annotation[#, 187, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{4475, 3888, 4741, 4075, 4370, 3642, 4143, 4418, 5176, 3867, 
         4937, 4938, 4939, 4244, 3784, 5121, 3785, 5122, 3566, 5202, 3890, 
         5199, 5201, 5200, 4243, 3555, 4552, 4553, 4550, 3651, 4551, 3450, 
         4052, 3797, 4867, 4868, 4869, 3798, 3992, 5378, 3991, 4344, 3796, 
         4252, 3990, 5052, 3689, 4933, 4934, 5058, 3710, 4975, 4976, 4600, 
         3688, 4597, 4599, 4598, 3449, 4594, 3444, 4383, 4382, 4490, 4489, 
         3585, 4587, 4588, 4584, 4586, 4585, 3535, 4178, 4912, 4911, 4910, 
         3851, 5222, 3914, 5221, 3913, 4027, 5383, 4026, 4355, 4356, 4291, 
         4290, 4351, 4288, 4289, 4041, 4098, 3588, 5303, 5273, 5274, 5272, 
         5276, 5275, 3705, 4690, 3824, 4689, 3823, 4605, 3695, 4466, 4467, 
         3557, 4610, 4504, 4505, 4503, 4507, 4506, 4523, 4765, 3930, 4805, 
         3931, 4320, 3929, 4319, 4010, 5286, 4011, 5287, 3711, 5059, 5061, 
         5060, 3694, 4603, 4604, 5389, 5391, 5390, 3527, 5246, 3528, 4900, 
         4899, 4988, 4987, 3598, 4286, 3876, 5188, 3877, 3933, 5374, 3932, 
         3558, 4093, 3934, 5347, 3599, 4323, 3600, 5241, 3936, 5239, 3935, 
         5240, 4321, 4322, 4273, 3828, 4287, 5289, 4012, 5288, 4013, 5290, 
         3878, 5370, 4735, 5369, 3728, 5079, 3729, 4647, 4974, 3581, 4300, 
         4301, 3909, 4748, 3908, 5217, 3580, 5229, 3922, 5228, 5231, 5230, 
         3761, 5099, 5100, 4835, 4836, 3474, 3437, 5196, 5197, 5195, 3887, 
         5198, 3886, 4294, 4476, 4475}],
        "170"],
       Annotation[#, 170, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{3484, 4843, 4842, 5118, 5117, 3644, 4364, 4078, 4079, 4170, 
         4169, 4116, 4117, 3862, 4459, 4926, 4927, 4237, 4777, 4718, 4719, 
         4454, 4792, 5338, 3540, 4793, 3539, 4090, 4006, 4307, 4306, 3583, 
         4304, 4305, 3911, 4749, 3910, 5218, 3582, 4754, 4755, 4659, 3737, 
         4658, 3738, 4164, 4163, 3436, 3473, 4070, 4040, 4135, 4134, 3578, 
         4162, 5080, 3733, 4654, 3734, 5233, 5234, 5232, 3923, 4729, 3875, 
         4728, 4731, 4730, 3874, 4733, 4734, 4732, 4753, 4752, 5220, 4724, 
         4725, 4723, 4727, 4726, 3717, 4646, 3727, 4378, 4379, 5244, 5245, 
         4989, 4990, 4991, 4325, 4497, 4496, 4495, 3939, 4766, 4028, 4764, 
         3917, 4492, 4802, 4803, 4029, 4100, 4099, 4357, 3947, 4330, 3948, 
         4329, 3946, 5307, 4030, 5304, 5306, 5305, 3589, 5238, 5149, 5150, 
         5148, 3825, 5151, 3706, 4177, 4441, 4271, 3826, 4365, 3827, 4272, 
         5078, 4693, 3830, 4691, 3829, 4692, 4158, 4172, 4173, 4159, 3677, 
         4302, 3678, 4627, 4628, 4626, 3707, 4089, 4088, 4314, 3679, 4161, 
         4451, 4138, 3635, 4235, 4074, 4543, 3643, 4411, 4554, 5029, 3652, 
         4855, 4856, 5123, 5124, 4865, 4866}],
        "153"],
       Annotation[#, 153, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{5031, 3659, 5032, 4144, 4145, 3972, 5257, 3971, 5258, 4236, 
         3571, 4961, 4960, 4984, 4983, 3594, 4981, 4982, 4979, 4980, 3593, 
         4313, 3592, 4978, 4977, 5309, 5308, 3716, 4638, 4637, 4445, 4573, 
         4572, 4648, 4132, 3559, 4947, 4468, 3696, 4609, 4087, 4694, 3831, 
         4442, 4443, 4444, 4197, 4315, 3924, 4347, 3925, 4316, 4986, 4643, 
         3720, 4642, 5072, 5071, 4128, 3579, 4155, 4156, 4328, 4327, 3671, 
         4154, 3691, 4063, 4062, 4310, 3621, 4127, 4491, 4153, 3669, 4324, 
         3668, 4152, 3697, 4176, 3469, 4061, 3941, 4767, 3940, 4768, 4801, 
         4800, 4973, 4972, 4971, 4022, 5297, 4021, 4809, 4810, 4806, 4808, 
         4807, 4346, 4297, 4298, 4042, 4486, 3577, 4285, 3576, 4485, 4484, 
         4746, 4745, 4068, 4049, 4048, 4578, 3681, 4449, 4622, 5336, 3533, 
         4907, 4623, 4624, 4524, 4526, 4525, 4655, 4579, 4580, 4450, 4630, 
         5337, 3534, 4908, 5398, 3709, 5055, 4631, 4633, 4632, 3628, 4303, 
         4071, 4072, 4381, 3443, 4166, 3448, 4051, 3856, 4115, 4114, 3483, 
         4546, 4547, 3788, 4851, 4852, 3496, 4238, 3497, 3794, 5125, 3795, 
         5126, 3793, 4247, 4425, 4248, 3989, 5420}],
        "136"],
       Annotation[#, 136, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{3503, 4246, 4860, 4859, 4858, 3791, 4415, 3494, 4212, 4414, 
         4413, 4412, 3786, 4680, 3638, 4112, 3613, 4210, 3614, 4113, 4453, 
         4740, 3884, 4474, 4951, 3562, 4209, 3561, 4472, 4473, 5162, 5163, 
         5161, 4909, 4230, 3769, 4714, 3850, 5160, 3849, 5400, 4739, 3881, 
         4712, 3847, 4711, 3848, 4713, 3758, 4710, 3846, 4703, 3845, 4109, 
         3610, 4296, 3905, 4111, 4110, 4131, 4619, 4620, 4618, 4617, 4433, 
         3665, 4640, 4641, 4376, 4397, 5316, 3426, 3468, 4395, 5384, 4396, 
         4151, 5407, 3663, 4353, 3664, 4615, 4616, 4614, 3702, 5053, 4125, 
         4126, 3891, 4477, 4954, 4955, 4185, 4188, 5207, 3894, 5206, 3895, 
         5208, 3819, 5270, 4003, 4890, 4891, 4171, 3690, 5147, 3821, 5146, 
         3820, 4265, 4266, 4025, 5301, 4024, 5302, 4308, 4190, 5406, 4191, 
         3868, 4720, 5178, 5177, 3523, 4625, 4521, 4522, 4520, 3622, 4602, 
         3693, 4601, 3692, 4157, 3673, 4721, 3870, 4462, 4463, 4464, 3871, 
         5284, 5285, 5281, 5283, 5282, 4283, 3822, 5271, 4004, 4892, 4894, 
         4893, 5397, 3703, 5054, 3704, 4621, 3624, 4398, 4399, 3471, 3432, 
         4160, 3438, 4820, 4380, 3951, 4203, 4202, 3591, 4576, 4577, 4574, 
         3680, 4575, 3682, 5050, 4581, 4582, 4530, 4532, 5328, 4531, 5329, 
         3492, 4555, 5419}],
        "119"],
       Annotation[#, 119, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{3718, 4192, 4189, 3524, 4564, 4565, 5046, 5045, 3670, 4770, 
         3943, 4769, 3942, 4193, 3719, 5182, 3869, 5179, 5181, 5180, 3525, 
         5184, 5185, 5183, 5187, 5186, 3721, 5073, 5074, 4812, 4813, 3427, 
         5216, 3431, 4050, 4362, 4361, 3907, 4299, 3730, 4198, 4488, 4487, 
         3906, 5215, 4747, 5047, 5048, 4897, 5335, 4896, 3560, 4201, 4950, 
         4949, 4948, 3879, 4446, 3531, 4225, 5101, 5102, 4675, 4677, 4676, 
         4772, 4533, 4534, 4406, 4544, 5326, 3482, 4545, 5418}],
        "102"],
       Annotation[#, 102, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{3865, 3552, 3783, 5365, 3782, 4242, 3542, 5339, 3859, 3563, 
         3685, 5341, 3544, 3701, 3543, 5340, 3684, 3889}],
        "102"],
       Annotation[#, 102, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{4103, 4104, 4187, 4186, 3606, 4105, 4878, 4432, 3662, 4563, 
         4375, 4374, 3605, 5410, 4502, 3620, 4744, 3904, 4481, 4482, 4483, 
         4264, 4968, 4967, 5214, 4480, 4494, 4493, 3596, 4359, 4360, 4033, 
         5310, 4032, 5311, 3575, 4295, 3574, 4970, 4969, 4020, 5296, 4066, 
         4067, 4817, 4704, 3837, 4702, 3838, 4224, 4223, 5098, 3759, 4673, 
         3760, 4674, 3950, 4709, 3843, 4448, 4905, 4229, 3768, 4678, 3770, 
         4400, 4401, 3476, 3442, 4165, 4822, 4821, 5249, 3958, 5375, 4660, 
         5086, 5085, 4023, 5298, 5300, 5299, 4354, 5167, 5166, 5165, 4038, 
         5315, 4005, 4277, 3855, 4366, 4367, 4097, 3584, 5219, 4750, 4751, 
         4663, 5087, 3741, 4664, 3660, 5252, 3961, 5250, 3960, 5251, 3999, 
         5265, 5266, 4791, 4790, 3777, 5120, 3776, 5119, 4844, 4845}],
        "102"],
       Annotation[#, 102, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{3604, 4501, 4996, 4995, 4994, 5349, 5348, 4431, 5144, 5145, 
         4888, 4889, 3522, 4263, 3521, 4887, 4886, 4002, 5269, 4060, 4046, 
         4045, 4358, 4031, 4102, 4101, 3518, 4261, 3519, 4879, 5142, 5141, 
         3816, 5367, 4478, 4956, 4479, 4799, 4798, 3892, 5205, 3893, 4743, 
         5157, 4707, 3840, 4705, 3839, 4706, 3880, 4738, 5399, 3841, 5158, 
         3842, 4708, 3766, 4228, 4906, 5159, 3844, 5368, 4469, 4470, 4471, 
         4292, 4293, 3883, 5190, 3882, 5191, 3538, 5056, 5057, 4634, 5394, 
         3630, 5009, 3631, 4661, 4662, 4384, 4385, 3447, 4167, 3452, 4053, 
         4039, 4119, 4118, 3509, 5130, 3804, 5129, 5132, 5131, 3744, 4665, 
         4666, 4387, 4388, 3454, 4687, 3456, 4684, 4685, 4683, 3790}],
        "85"],
       Annotation[#, 85, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{3607, 4106, 3672, 4567, 4566, 4998, 5000, 4999, 4997, 4437, 
         4438, 4722, 3872, 4465, 4940, 4941, 4194, 4644, 4645, 4570, 3675, 
         4439, 4895, 4511, 3611, 4508, 4510, 4509, 4571, 5189, 4736, 4737, 
         4695, 5152, 3833, 4696, 3762, 4699, 3836, 4697, 3835, 4698, 3854, 
         4715, 5395, 3636, 5013, 3637, 5402, 4535, 4556, 5030, 3653, 4857}],
        "85"],
       Annotation[#, 85, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{4419, 4686, 3792, 5332, 3505, 4234, 3967, 5376, 3968, 3641, 
         5356, 4542, 5355, 3639, 5015, 3640, 5405, 4541, 4875, 5354, 4430, 
         5011, 5012, 3634, 5010, 4789, 5263, 5264, 4870, 4871, 4872, 4253, 
         4429, 4428, 4811, 3806, 4254, 3995, 4036, 5366, 3805, 4345, 3997, 
         5382, 3998, 3810, 5111, 3773, 5110, 5113, 5112, 5255, 4337, 3545, 
         5342, 4008, 4794, 4795, 4539, 5392, 4540, 5403, 4536, 4538, 5409, 
         4537, 3461, 3479, 3966, 4335, 4334, 3501, 5331, 3986, 4854, 4853}],
        "85"],
       Annotation[#, 85, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{3602, 4499, 3603, 4500, 4688, 5143, 3818, 4884, 4885, 4262, 
         3817, 5268, 4001, 5267, 4000, 3467, 3425, 4129, 4130, 4260, 4259, 
         4064, 4065, 4377, 4133, 3627, 4276, 4069, 4651, 4652, 4649, 3731, 
         4650, 3732, 4653, 4914, 3537, 4227, 3767, 5107, 5109, 5108, 5248, 
         3955, 5051, 3683, 4915, 4916, 4515, 3612, 4512, 4514, 4513, 4583, 
         5291, 5193, 5194, 5192, 3885, 5313, 5314, 4804, 4037, 5312, 3541, 
         4278, 4279, 3959, 4054, 3453, 4776, 3455, 5322, 4775, 4391, 4408, 
         4407, 3485, 4681, 4682, 3789, 5330, 3498, 4245, 3499, 5014, 5421}],
        "68"],
       Annotation[#, 68, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{3608, 4107, 3674, 4569, 4568, 5001, 3609, 5003, 5002, 5351, 
         5350, 4898, 5049, 3676, 5396, 4942, 4943, 3556, 4284, 4946, 4945, 
         4944, 3873, 4440, 3526, 4221, 5095, 3756, 4669, 3757, 3949, 5317, 
         3433, 5154, 3834, 5153, 5156, 5155, 3763, 5103, 5104, 4837, 4838, 
         3475, 3439, 4139, 3445, 5320, 3956, 4332, 4331, 3493, 4785, 4786, 
         4077, 4076, 4342, 3506, 5411, 5416}],
        "68"],
       Annotation[#, 68, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{3975, 3974, 4340, 4341, 3988, 4787, 3987, 4788, 4140, 4560, 
         4561, 4424, 4423, 4422, 3656, 4086, 3504, 5033, 5034, 4864, 4863, 
         4862, 3655, 4085, 3500, 4149, 4017, 5293, 4018, 5294, 4016, 4352, 
         4966, 4965, 4019, 5295, 3615, 5292, 4015, 4962, 4963, 4964, 4348, 
         4349, 4007, 4918, 4919, 4920, 4168, 4136, 4137, 4120, 3617, 4336, 
         3616, 5041, 5042, 4880, 4881, 5170, 3860, 4457, 4796, 4797, 4009, 
         4092, 4091, 4371, 3969, 4338, 3970, 5116, 3775, 5114, 3774, 5115, 
         4216, 4923, 4922, 4921, 3861, 5171, 4883, 4882, 5044, 5043, 4394, 
         4393, 3619, 4519, 4529, 4528, 4562, 3515, 5334, 3812, 3811, 3903, 
         5372, 3902, 3516, 3900, 5371, 3901, 3899, 5212, 5213, 5209, 5211, 
         5210, 4150, 3502, 5134, 5135, 5133, 3807, 4218, 4217, 4255, 3808, 
         4372, 4373, 4257, 4256, 3510, 3996, 5381, 5380, 5379, 3809, 4084}],
        "68"],
       Annotation[#, 68, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{3465, 3481, 4123, 4124, 3667, 4435, 4436, 3520, 4593, 4611, 
         4591, 4592, 4590, 4589, 4434, 3666, 4057, 3463, 3480, 4121}],
        "51"],
       Annotation[#, 51, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{3618, 4122, 4142, 4141, 4779, 3977, 4778, 4781, 4780, 3976, 
         3979, 5377, 3978, 3491, 3657, 5359, 3658}],
        "51"],
       Annotation[#, 51, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{3973, 3645, 3779, 5364, 3778, 4239, 3508, 5333, 3803, 3507, 
         3802, 3800, 5127, 3801, 5128, 3799, 5262, 3994, 5260, 3993, 5261, 
         4206, 4204, 4205, 3952, 5318, 3440, 4199, 3434, 3472, 4831, 4830, 
         3625, 5006, 3429, 4814, 3723, 5075, 3722, 4267, 4268, 4034, 4047, 
         3428, 4269, 4108}],
        "51"],
       Annotation[#, 51, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{4044, 5040, 3661, 4876, 4877, 4498, 3601, 4992, 4993, 4829, 
         4828, 3466, 3424, 5243, 3937, 5242, 3938, 4220, 4219, 3754, 5363, 
         3755, 3945, 4700, 4701, 4447, 4904, 3532, 4222, 5096, 5097, 4670, 
         4672, 4671, 4771, 4716, 4717, 4452, 4913, 3536, 4226, 3764, 5105, 
         3765, 5106, 3954, 3898, 5345, 3572, 4231, 3771, 4333, 3957, 4368, 
         4369, 4073, 3477, 3446, 4774, 3451, 5321, 4773, 4386, 5088, 3742, 
         4455, 4917, 4456, 5278, 5277, 3857, 5168, 3858, 5169, 3743, 5094, 
         3747, 5092, 3746, 5093, 4211, 3962, 5253, 3963, 5254, 3748, 3750, 
         5362, 3749, 3780, 4846, 4847, 3787, 3495, 5415, 5417}],
        "51"],
       Annotation[#, 51, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{4861, 4420, 4421, 4427, 4426, 4416, 4417, 3654, 4850, 4849, 
         4848, 3646, 5016, 4548, 4549, 4517, 5386, 4518, 5401, 4516, 5005, 
         5408, 5004}],
        "51"],
       Annotation[#, 51, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{3464, 4827, 4392, 4410, 5387, 4409, 5388, 3490, 5137, 3813, 
         5136, 3814, 5138, 3753, 4258, 3815, 4363, 4058, 4059, 5140, 5139, 
         3462, 4826, 4825, 4874, 4873, 3514, 5038, 5039, 5035, 5037, 5036, 
         3489, 4148, 3488, 5327, 3781, 4241, 4240, 3512, 5025, 5026, 5024, 
         5028, 5027, 5414, 5020, 5021, 5019, 5023, 5022, 3513, 5413, 5358, 
         5412, 3487, 3649, 5357, 3650, 3511, 5018, 3648, 5017, 3647, 5259, 
         3985, 4783, 3981, 4782, 3982, 4784, 3980, 3984, 3983, 3486, 4082, 
         4083, 4035, 4081, 4080, 4679, 4668, 3478, 5385, 4667, 4405, 3633, 
         3752, 5325, 3460, 4215, 3459, 5324, 3965, 3751, 3964, 5323, 3458, 
         4214, 3457, 4841, 4840, 4839, 3632, 5090, 5091, 5089, 3745, 4213, 
         4390, 4389}],
        "34"],
       Annotation[#, 34, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{4055, 4056, 4147, 4146, 4824, 4823, 4404, 4403, 4402, 3772, 
         4233, 4232, 3517, 5084, 3740, 5083, 3739, 4527, 3629, 5007, 5008, 
         4657, 4656, 5082, 4207, 4208, 3953, 5319, 3441, 4200, 3435, 4834, 
         4833, 4832, 3626, 5077, 3726, 5076, 3725, 4196, 3724, 5247, 3944, 
         4815, 4816, 3430, 4195, 4326}],
        "34"],
       Annotation[#, 34, "Tooltip"]& ]}, {}, {}}, {InsetBox["34", 5422], 
     InsetBox["34", 5423], InsetBox["51", 5424], InsetBox["51", 5425], 
     InsetBox["51", 5426], InsetBox["51", 5427], InsetBox["51", 5428], 
     InsetBox["68", 5429], InsetBox["68", 5430], InsetBox["68", 5431], 
     InsetBox["85", 5432], InsetBox["85", 5433], InsetBox["85", 5434], 
     InsetBox["102", 5435], InsetBox["102", 5436], InsetBox["102", 5437], 
     InsetBox["119", 5438], InsetBox["136", 5439], InsetBox["153", 5440], 
     InsetBox["170", 5441], InsetBox["187", 5442]}}],
  AspectRatio->1,
  Epilog->{
    RGBColor[1, 0, 0], 
    PointSize[0.02], 
    InsetBox[
     FormBox["\"global maximum\"", TraditionalForm], {-0.995, -2.092}], 
    PointBox[{-0.995, -1.992}], 
    InsetBox[
     FormBox["\"local maximum\"", TraditionalForm], {0.5304, 1.2191}], 
    PointBox[{1.2304, 1.2191}]},
  Frame->True,
  ImageSize->{266., Automatic},
  PlotRange->{{-3, 2}, {-3, 2}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{270, 267},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->4551709]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Maximize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Maximize"]], "InlineFormula"],
 "  finds a global maximum and can work in infinite precision:"
}], "ExampleText",
 CellID->1708706374],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Maximize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     FormBox[
      RowBox[{" ", 
       RowBox[{
        RowBox[{"100", "/", 
         RowBox[{"(", 
          RowBox[{
           SuperscriptBox[
            RowBox[{"(", 
             RowBox[{"x", "-", "1"}], ")"}], "2"], "+", 
           SuperscriptBox[
            RowBox[{"(", 
             RowBox[{"y", "-", "1"}], ")"}], "2"], "+", "1"}], ")"}]}], "+", 
        RowBox[{"200", "/", 
         RowBox[{"(", 
          RowBox[{
           SuperscriptBox[
            RowBox[{"(", 
             RowBox[{"x", "+", "1"}], ")"}], "2"], "+", 
           SuperscriptBox[
            RowBox[{"(", 
             RowBox[{"y", "+", "2"}], ")"}], "2"], "+", "1"}], ")"}]}]}]}],
      TraditionalForm], ",", " ", 
     FormBox[
      RowBox[{
       RowBox[{
        SuperscriptBox["x", "2"], "+", 
        SuperscriptBox["y", "2"]}], ">", "3"}],
      TraditionalForm]}], " ", "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", " ", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->1528684992],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", 
    RowBox[{"Root", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"9450000000000", "-", 
        RowBox[{"1185300000000", " ", "#1"}], "+", 
        RowBox[{"39312000000", " ", 
         SuperscriptBox["#1", "2"]}], "+", 
        RowBox[{"1872000000", " ", 
         SuperscriptBox["#1", "3"]}], "+", 
        RowBox[{"18099900", " ", 
         SuperscriptBox["#1", "4"]}], "+", 
        RowBox[{"48841", " ", 
         SuperscriptBox["#1", "5"]}]}], "&"}], ",", "1"}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"Root", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"289", "-", 
          RowBox[{"1785", " ", "#1"}], "+", 
          RowBox[{"234", " ", 
           SuperscriptBox["#1", "2"]}], "+", 
          RowBox[{"1326", " ", 
           SuperscriptBox["#1", "3"]}], "-", 
          RowBox[{"507", " ", 
           SuperscriptBox["#1", "4"]}], "+", 
          RowBox[{"507", " ", 
           SuperscriptBox["#1", "5"]}]}], "&"}], ",", "1"}], "]"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"Root", "[", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{"-", "595"}], "-", 
          RowBox[{"2077", " ", "#1"}], "+", 
          RowBox[{"1586", " ", 
           SuperscriptBox["#1", "2"]}], "+", 
          RowBox[{"910", " ", 
           SuperscriptBox["#1", "3"]}], "+", 
          RowBox[{"169", " ", 
           SuperscriptBox["#1", "4"]}], "+", 
          RowBox[{"169", " ", 
           SuperscriptBox["#1", "5"]}]}], "&"}], ",", "1"}], "]"}]}]}], 
    "}"}]}], "}"}]], "Output",
 ImageSize->{467, 75},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->315667043]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", "%", "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->1056055790],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"207.1598969820089`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "0.9948613347360095`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "1.9922920021040142`"}]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{263, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->752448616]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->25229],

Cell["\<\
If the constraint region is empty, the algorithm will not converge:\
\>", "ExampleText",
 CellID->687871453],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMaximum", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "+", "y"}], ",", " ", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"x", "^", "2"}], "+", 
        RowBox[{"y", "^", "2"}]}], "\[GreaterEqual]", " ", "2"}], " ", "&&", 
      " ", 
      RowBox[{
       RowBox[{
        RowBox[{"x", "^", "2"}], "+", 
        RowBox[{"y", "^", "2"}]}], "\[LessEqual]", " ", "1"}]}]}], "}"}], ",",
    
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1311741547],

Cell[BoxData[
 RowBox[{
  RowBox[{"FindMaximum", "::", "\<\"eit\"\>"}], ":", 
  " ", "\<\"The algorithm does not converge to the tolerance of \
\\!\\(4.806217383937354`*^-6\\) in \\!\\(500\\) iterations. The best \
estimated solution, with feasibility residual, KKT residual or complementary \
residual of \\!\\({0.5094107859477752`, 1.0546072559086298`*^-6, \
8.290002787469562`*^-12}\\), is returned.\"\>"}]], "Message", "MSG",
 CellID->121139156],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.7374756314177864`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "0.8687378157094133`"}], ",", 
     RowBox[{"y", "\[Rule]", "0.8687378157083729`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{262, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->202906829]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1437064019],

Cell["\<\
If the maximum value is not finite, the algorithm will not converge:\
\>", "ExampleText",
 CellID->107225626],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMaximum", "[", 
  RowBox[{
   RowBox[{"Tan", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->749723501],

Cell[BoxData[
 RowBox[{
  RowBox[{"FindMaximum", "::", "\<\"sdprec\"\>"}], ":", 
  " ", "\<\"Line search unable to find a sufficient increase in the function \
value with \\!\\(MachinePrecision\\) digit precision.\"\>"}]], "Message", \
"MSG",
 CellID->738282979],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"4.801901335062553`*^14", ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", "7.853981633974481`"}], "}"}]}], "}"}]], "Output",\

 ImageSize->{193, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->520123313]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMaximum", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"x", "^", "2"}], "+", 
      RowBox[{"y", "^", "2"}]}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "^", "2"}], "+", 
       RowBox[{"y", "^", "2"}]}], ">", "10"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1357681257],

Cell[BoxData[
 RowBox[{
  RowBox[{"FindMaximum", "::", "\<\"eit\"\>"}], ":", 
  " ", "\<\"The algorithm does not converge to the tolerance of \
\\!\\(4.806217383937354`*^-6\\) in \\!\\(500\\) iterations. The best \
estimated solution, with feasibility residual, KKT residual or complementary \
residual of \\!\\({0.43151582754954465`, 7.312348927125921`, \
0.1593467131318663`}\\), is returned.\"\>"}]], "Message", "MSG",
 CellID->376267146],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"26.208433894735613`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "3.61997471639897`"}], ",", 
     RowBox[{"y", "\[Rule]", "3.619974716398971`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{248, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->654101941]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1615498026],

Cell["\<\
Integer linear programming algorithm is only available for machine number \
problems:\
\>", "ExampleText",
 CellID->2001017660],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMaximum", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"-", "x"}], "-", "y"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "+", 
        RowBox[{"2", "y"}]}], "\[GreaterEqual]", "3"}], "&&", 
      RowBox[{"x", ">", "1"}], "&&", 
      RowBox[{"y", ">", "1"}], "&&", 
      RowBox[{"x", "\[Element]", "Integers"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}], ",", 
   RowBox[{"WorkingPrecision", "\[Rule]", "20"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1252111367],

Cell[BoxData[
 RowBox[{
  RowBox[{"FindMaximum", "::", "\<\"lpip\"\>"}], ":", 
  " ", "\<\"Warning: Problem specified contains integer variables. Only \
machine number algorithm is available; the problem will be converted into \
machine precision.\"\>"}]], "Message", "MSG",
 CellID->227133727],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "2.`"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "1"}], ",", 
     RowBox[{"y", "\[Rule]", "1.`"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{144, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->467710250]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1986880533],

Cell["\<\
Sometimes providing a suitable starting point can help the algorithm to \
converge:\
\>", "ExampleText",
 CellID->585941183],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMaximum", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"-", 
       RowBox[{"Cos", "[", "x", "]"}]}], "+", 
      RowBox[{"Exp", "[", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"x", "-", "0.5"}], ")"}], " ", "y"}], "]"}]}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "^", "2"}], "+", 
       RowBox[{"y", "^", "2"}]}], "<", "1"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->831389318],

Cell[BoxData[
 RowBox[{
  RowBox[{"FindMaximum", "::", "\<\"eit\"\>"}], ":", 
  " ", "\<\"The algorithm does not converge to the tolerance of \
\\!\\(4.806217383937354`*^-6\\) in \\!\\(500\\) iterations. The best \
estimated solution, with feasibility residual, KKT residual or complementary \
residual of \\!\\({2.6522780976498024`, 0.9410342449205663`, \
0.05992969652258977`}\\), is returned.\"\>"}]], "Message", "MSG",
 CellID->340079755],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.7117655966513841`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "1.762852369357621`"}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "0.5164658455857531`"}]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{270, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->89699571]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMaximum", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"-", 
       RowBox[{"Cos", "[", "x", "]"}]}], "+", 
      RowBox[{"Exp", "[", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"x", "-", "0.5"}], ")"}], " ", "y"}], "]"}]}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "^", "2"}], "+", 
       RowBox[{"y", "^", "2"}]}], "<", "1"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"x", ",", 
       RowBox[{"-", "1"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", 
       RowBox[{"-", "1"}]}], "}"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->765500604],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.6004629968908979`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "0.6572187833876794`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "0.7536998547122109`"}]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{263, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->446818958]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->27716],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["FindMinimum",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FindMinimum"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["NMaximize",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NMaximize"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Maximize",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Maximize"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["FindFit",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FindFit"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["LinearProgramming",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/LinearProgramming"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["D",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/D"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["FindRoot",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FindRoot"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->773]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->21363],

Cell[TextData[ButtonBox["Unconstrained Optimization",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/UnconstrainedOptimizationOverview"]], "Tutorials",
 CellID->123070281],

Cell[TextData[ButtonBox["Numerical Optimization",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/NumericalOptimization"]], "Tutorials",
 CellID->15079]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->279426012],

Cell[TextData[ButtonBox["Optimization",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/Optimization"]], "MoreAbout",
 CellID->840103691],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->313970380]
}, Open  ]],

Cell["New in 5  |  Last modified in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"FindMaximum - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 42, 58.7496612}", "context" -> "System`", 
    "keywords" -> {
     "conjugate gradient", "constrained local optimization", 
      "constrained optimization", "convex programming", 
      "Gauss-Newton algorithm", "GNA", "gradient", "gradient descent", 
      "interior point algorithm", "Levenberg-Marquardt method", 
      "linear fractional programming", "linear programming", 
      "local maximization", "local optimization", "Newton's method", 
      "numerical maximum", "quadratic programming", "quasi-Newton", 
      "steepest-descent"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "FindMaximum[f, {x, x_0}] searches for a local maximum in f, starting \
from the point x = x_0. FindMaximum[f, {{x, x_0}, {y, y_0}, ...}] searches \
for a local maximum in a function of several variables. FindMaximum[{f, \
cons}, {{x, x_0}, {y, y_0}, ...}] searches for a local maximum subject to the \
constraints cons. FindMaximum[{f, cons}, {x, y, ...}] starts from a point \
within the region defined by the constraints.", 
    "synonyms" -> {"find maximum"}, "title" -> "FindMaximum", "type" -> 
    "Symbol", "uri" -> "ref/FindMaximum"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[22274, 688, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->672757868]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 379764, 7861}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2731, 62, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3354, 89, 4195, 83, 70, "ObjectNameGrid"],
Cell[7552, 174, 3402, 109, 70, "Usage",
 CellID->12439]
}, Open  ]],
Cell[CellGroupData[{
Cell[10991, 288, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[11469, 305, 936, 36, 70, "Notes",
 CellID->17026],
Cell[12408, 343, 175, 4, 70, "Notes",
 CellID->19890],
Cell[12586, 349, 330, 9, 70, "Notes",
 CellID->19298],
Cell[12919, 360, 386, 11, 70, "Notes",
 CellID->1915],
Cell[13308, 373, 474, 17, 70, "Notes",
 CellID->11648],
Cell[13785, 392, 947, 36, 70, "Notes",
 CellID->20737],
Cell[14735, 430, 958, 36, 70, "Notes",
 CellID->14015],
Cell[15696, 468, 415, 14, 70, "Notes",
 CellID->8689],
Cell[16114, 484, 83, 1, 70, "Notes",
 CellID->23646],
Cell[16200, 487, 575, 19, 70, "Notes",
 CellID->21374],
Cell[16778, 508, 68, 1, 70, "Notes",
 CellID->9099],
Cell[16849, 511, 3206, 90, 70, "3ColumnTableMod",
 CellID->19426],
Cell[20058, 603, 520, 20, 70, "Notes",
 CellID->10073],
Cell[20581, 625, 472, 14, 70, "Notes",
 CellID->17325],
Cell[21056, 641, 506, 17, 70, "Notes",
 CellID->7673],
Cell[21565, 660, 672, 23, 70, "Notes",
 CellID->3822]
}, Closed]],
Cell[CellGroupData[{
Cell[22274, 688, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->672757868],
Cell[CellGroupData[{
Cell[22659, 702, 148, 5, 70, "ExampleSection",
 CellID->289512076],
Cell[22810, 709, 189, 7, 70, "ExampleText",
 CellID->499988759],
Cell[CellGroupData[{
Cell[23024, 720, 235, 8, 28, "Input",
 CellID->376441841],
Cell[23262, 730, 297, 10, 36, "Output",
 CellID->648154644]
}, Open  ]],
Cell[CellGroupData[{
Cell[23596, 745, 239, 8, 28, "Input",
 CellID->347120420],
Cell[23838, 755, 2929, 52, 125, "Output",
 Evaluatable->False,
 CellID->485044888]
}, Open  ]],
Cell[26782, 810, 163, 6, 70, "ExampleText",
 CellID->452807755],
Cell[CellGroupData[{
Cell[26970, 820, 300, 10, 70, "Input",
 CellID->28565156],
Cell[27273, 832, 176, 5, 36, "Output",
 CellID->156355731]
}, Open  ]],
Cell[27464, 840, 125, 3, 70, "ExampleDelimiter",
 CellID->986451359],
Cell[27592, 845, 357, 12, 70, "ExampleText",
 CellID->1560461396],
Cell[CellGroupData[{
Cell[27974, 861, 362, 12, 70, "Input",
 CellID->678564151],
Cell[28339, 875, 295, 10, 36, "Output",
 CellID->219625470]
}, Open  ]],
Cell[28649, 888, 126, 3, 70, "ExampleDelimiter",
 CellID->1519398755],
Cell[28778, 893, 135, 4, 70, "ExampleText",
 CellID->1896901598],
Cell[CellGroupData[{
Cell[28938, 901, 544, 17, 70, "Input",
 CellID->1202084729],
Cell[29485, 920, 339, 12, 36, "Output",
 CellID->50585186]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[29873, 938, 216, 7, 70, "ExampleSection",
 CellID->13645],
Cell[30092, 947, 102, 2, 70, "ExampleText",
 CellID->62007641],
Cell[CellGroupData[{
Cell[30219, 953, 234, 8, 70, "Input",
 CellID->66723803],
Cell[30456, 963, 294, 10, 36, "Output",
 CellID->10453814]
}, Open  ]],
Cell[CellGroupData[{
Cell[30787, 978, 236, 8, 70, "Input",
 CellID->386171924],
Cell[31026, 988, 297, 10, 36, "Output",
 CellID->174136289]
}, Open  ]],
Cell[31338, 1001, 125, 3, 70, "ExampleDelimiter",
 CellID->404322872],
Cell[31466, 1006, 286, 11, 70, "ExampleText",
 CellID->516388244],
Cell[CellGroupData[{
Cell[31777, 1021, 340, 12, 70, "Input",
 CellID->46634681],
Cell[32120, 1035, 358, 12, 36, "Output",
 CellID->128600046]
}, Open  ]],
Cell[32493, 1050, 125, 3, 70, "ExampleDelimiter",
 CellID->242324886],
Cell[32621, 1055, 83, 1, 70, "ExampleText",
 CellID->681774368],
Cell[CellGroupData[{
Cell[32729, 1060, 503, 18, 70, "Input",
 CellID->852578689],
Cell[33235, 1080, 374, 12, 36, "Output",
 CellID->128327230]
}, Open  ]],
Cell[33624, 1095, 126, 3, 70, "ExampleDelimiter",
 CellID->1308882988],
Cell[33753, 1100, 87, 1, 70, "ExampleText",
 CellID->374615681],
Cell[CellGroupData[{
Cell[33865, 1105, 398, 14, 70, "Input",
 CellID->1153958179],
Cell[34266, 1121, 373, 12, 36, "Output",
 CellID->123814084]
}, Open  ]],
Cell[34654, 1136, 125, 3, 70, "ExampleDelimiter",
 CellID->685666782],
Cell[34782, 1141, 124, 3, 70, "ExampleText",
 CellID->169051774],
Cell[CellGroupData[{
Cell[34931, 1148, 514, 17, 70, "Input",
 CellID->1991020309],
Cell[35448, 1167, 374, 12, 36, "Output",
 CellID->624608140]
}, Open  ]],
Cell[CellGroupData[{
Cell[35859, 1184, 567, 18, 70, "Input",
 CellID->833915335],
Cell[36429, 1204, 341, 12, 36, "Output",
 CellID->17622987]
}, Open  ]],
Cell[36785, 1219, 123, 3, 70, "ExampleDelimiter",
 CellID->5114145],
Cell[36911, 1224, 195, 7, 70, "ExampleText",
 CellID->284316099],
Cell[CellGroupData[{
Cell[37131, 1235, 660, 23, 70, "Input",
 CellID->1963030529],
Cell[37794, 1260, 374, 12, 36, "Output",
 CellID->620671711]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[38217, 1278, 217, 7, 70, "ExampleSection",
 CellID->6246],
Cell[CellGroupData[{
Cell[38459, 1289, 245, 7, 70, "ExampleSubsection",
 CellID->25956],
Cell[38707, 1298, 782, 28, 70, "ExampleText",
 CellID->764461149],
Cell[CellGroupData[{
Cell[39514, 1330, 339, 10, 70, "Input",
 CellID->1625295724],
Cell[39856, 1342, 281, 10, 36, "Output",
 CellID->494162563]
}, Open  ]],
Cell[40152, 1355, 126, 3, 70, "ExampleDelimiter",
 CellID->1055000278],
Cell[40281, 1360, 858, 28, 70, "ExampleText",
 CellID->1329553122],
Cell[CellGroupData[{
Cell[41164, 1392, 341, 10, 70, "Input",
 CellID->1468387723],
Cell[41508, 1404, 414, 8, 70, "Message",
 CellID->7355808],
Cell[41925, 1414, 280, 10, 36, "Output",
 CellID->787252468]
}, Open  ]],
Cell[42220, 1427, 244, 8, 70, "ExampleText",
 CellID->1199314865],
Cell[CellGroupData[{
Cell[42489, 1439, 395, 11, 70, "Input",
 CellID->39134653],
Cell[42887, 1452, 404, 13, 54, "Output",
 CellID->647176890]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[43340, 1471, 234, 7, 70, "ExampleSubsection",
 CellID->23123],
Cell[43577, 1480, 80, 1, 70, "ExampleText",
 CellID->28034064],
Cell[CellGroupData[{
Cell[43682, 1485, 453, 14, 70, "Input",
 CellID->19607354],
Cell[44138, 1501, 675, 21, 142, "Output",
 CellID->68827344]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[44862, 1528, 225, 7, 70, "ExampleSubsection",
 CellID->12053],
Cell[45090, 1537, 145, 4, 70, "ExampleText",
 CellID->858592029],
Cell[CellGroupData[{
Cell[45260, 1545, 600, 19, 70, "Input",
 CellID->2076542040],
Cell[45863, 1566, 377, 13, 36, "Output",
 CellID->26026175]
}, Open  ]],
Cell[46255, 1582, 84, 1, 70, "ExampleText",
 CellID->1859244329],
Cell[CellGroupData[{
Cell[46364, 1587, 1605, 51, 70, "Input",
 CellID->1427703802],
Cell[47972, 1640, 379, 13, 36, "Output",
 CellID->128052500]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[48400, 1659, 223, 7, 70, "ExampleSubsection",
 CellID->16079],
Cell[48626, 1668, 119, 3, 70, "ExampleText",
 CellID->993924491],
Cell[CellGroupData[{
Cell[48770, 1675, 412, 14, 70, "Input",
 CellID->782571855],
Cell[49185, 1691, 416, 8, 70, "Message",
 CellID->322138632],
Cell[49604, 1701, 433, 14, 36, "Output",
 CellID->51247189]
}, Open  ]],
Cell[CellGroupData[{
Cell[50074, 1720, 484, 16, 70, "Input",
 CellID->939473422],
Cell[50561, 1738, 261, 6, 70, "Message",
 CellID->49025433],
Cell[50825, 1746, 433, 14, 36, "Output",
 CellID->17329102]
}, Open  ]],
Cell[51273, 1763, 139, 4, 70, "ExampleText",
 CellID->1614271486],
Cell[CellGroupData[{
Cell[51437, 1771, 477, 15, 70, "Input",
 CellID->736570868],
Cell[51917, 1788, 420, 14, 36, "Output",
 CellID->456027668]
}, Open  ]],
Cell[52352, 1805, 224, 7, 70, "ExampleText",
 CellID->1954260595],
Cell[CellGroupData[{
Cell[52601, 1816, 441, 15, 70, "Input",
 CellID->46889232],
Cell[53045, 1833, 436, 14, 36, "Output",
 CellID->148029175]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[53530, 1853, 227, 7, 70, "ExampleSubsection",
 CellID->2025],
Cell[53760, 1862, 239, 8, 70, "ExampleText",
 CellID->1691254454],
Cell[54002, 1872, 1233, 42, 70, "Input",
 CellID->1514338379],
Cell[CellGroupData[{
Cell[55260, 1918, 1426, 47, 70, "Input",
 CellID->1350533918],
Cell[56689, 1967, 147343, 2472, 70, "Output",
 CellID->539862624]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[204081, 4445, 233, 7, 70, "ExampleSubsection",
 CellID->26397],
Cell[204317, 4454, 304, 10, 70, "ExampleText",
 CellID->549268746],
Cell[CellGroupData[{
Cell[204646, 4468, 542, 17, 70, "Input",
 CellID->1164918439],
Cell[205191, 4487, 462, 14, 36, "Output",
 CellID->524230119]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[205714, 4508, 233, 7, 70, "ExampleSection",
 CellID->21468],
Cell[205950, 4517, 397, 13, 70, "ExampleText",
 CellID->1650805990],
Cell[CellGroupData[{
Cell[206372, 4534, 1109, 38, 70, "Input",
 CellID->2092335573],
Cell[207484, 4574, 374, 12, 36, "Output",
 CellID->487542325]
}, Open  ]],
Cell[CellGroupData[{
Cell[207895, 4591, 1091, 36, 70, "Input",
 CellID->827788029],
Cell[208989, 4629, 658, 11, 70, "Message",
 CellID->31348708],
Cell[209650, 4642, 413, 13, 36, "Output",
 CellID->2905743]
}, Open  ]],
Cell[CellGroupData[{
Cell[210100, 4660, 2144, 68, 70, "Input",
 CellID->200111634],
Cell[212247, 4730, 149965, 2518, 70, "Output",
 CellID->4551709]
}, Open  ]],
Cell[362227, 7251, 238, 7, 70, "ExampleText",
 CellID->1708706374],
Cell[CellGroupData[{
Cell[362490, 7262, 1091, 36, 70, "Input",
 CellID->1528684992],
Cell[363584, 7300, 1770, 53, 96, "Output",
 CellID->315667043]
}, Open  ]],
Cell[CellGroupData[{
Cell[365391, 7358, 98, 3, 70, "Input",
 CellID->1056055790],
Cell[365492, 7363, 414, 13, 36, "Output",
 CellID->752448616]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[365955, 7382, 226, 7, 70, "ExampleSection",
 CellID->25229],
Cell[366184, 7391, 118, 3, 70, "ExampleText",
 CellID->687871453],
Cell[CellGroupData[{
Cell[366327, 7398, 569, 20, 70, "Input",
 CellID->1311741547],
Cell[366899, 7420, 449, 8, 70, "Message",
 CellID->121139156],
Cell[367351, 7430, 374, 12, 36, "Output",
 CellID->202906829]
}, Open  ]],
Cell[367740, 7445, 126, 3, 70, "ExampleDelimiter",
 CellID->1437064019],
Cell[367869, 7450, 119, 3, 70, "ExampleText",
 CellID->107225626],
Cell[CellGroupData[{
Cell[368013, 7457, 210, 7, 70, "Input",
 CellID->749723501],
Cell[368226, 7466, 262, 6, 70, "Message",
 CellID->738282979],
Cell[368491, 7474, 299, 10, 39, "Output",
 CellID->520123313]
}, Open  ]],
Cell[CellGroupData[{
Cell[368827, 7489, 419, 15, 70, "Input",
 CellID->1357681257],
Cell[369249, 7506, 441, 8, 70, "Message",
 CellID->376267146],
Cell[369693, 7516, 371, 12, 36, "Output",
 CellID->654101941]
}, Open  ]],
Cell[370079, 7531, 126, 3, 70, "ExampleDelimiter",
 CellID->1615498026],
Cell[370208, 7536, 137, 4, 70, "ExampleText",
 CellID->2001017660],
Cell[CellGroupData[{
Cell[370370, 7544, 572, 18, 70, "Input",
 CellID->1252111367],
Cell[370945, 7564, 294, 6, 70, "Message",
 CellID->227133727],
Cell[371242, 7572, 340, 12, 36, "Output",
 CellID->467710250]
}, Open  ]],
Cell[371597, 7587, 126, 3, 70, "ExampleDelimiter",
 CellID->1986880533],
Cell[371726, 7592, 134, 4, 70, "ExampleText",
 CellID->585941183],
Cell[CellGroupData[{
Cell[371885, 7600, 543, 19, 70, "Input",
 CellID->831389318],
Cell[372431, 7621, 442, 8, 70, "Message",
 CellID->340079755],
Cell[372876, 7631, 391, 12, 36, "Output",
 CellID->89699571]
}, Open  ]],
Cell[CellGroupData[{
Cell[373304, 7648, 695, 25, 70, "Input",
 CellID->765500604],
Cell[374002, 7675, 415, 13, 36, "Output",
 CellID->446818958]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[374478, 7695, 311, 9, 70, "SeeAlsoSection",
 CellID->27716],
Cell[374792, 7706, 1684, 62, 70, "SeeAlso",
 CellID->773]
}, Open  ]],
Cell[CellGroupData[{
Cell[376513, 7773, 314, 9, 70, "TutorialsSection",
 CellID->21363],
Cell[376830, 7784, 177, 4, 70, "Tutorials",
 CellID->123070281],
Cell[377010, 7790, 154, 3, 70, "Tutorials",
 CellID->15079]
}, Open  ]],
Cell[CellGroupData[{
Cell[377201, 7798, 319, 9, 70, "MoreAboutSection",
 CellID->279426012],
Cell[377523, 7809, 136, 3, 70, "MoreAbout",
 CellID->840103691],
Cell[377662, 7814, 179, 3, 70, "MoreAbout",
 CellID->313970380]
}, Open  ]],
Cell[377856, 7820, 50, 0, 70, "History"],
Cell[377909, 7822, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

