(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[   1077799,      19246]
NotebookOptionsPosition[   1065241,      18815]
NotebookOutlinePosition[   1066767,      18855]
CellTagsIndexPosition[   1066678,      18850]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"FindMinimum" :> 
          Documentation`HelpLookup["paclet:ref/FindMinimum"], "NMinimize" :> 
          Documentation`HelpLookup["paclet:ref/NMinimize"], "Nearest" :> 
          Documentation`HelpLookup["paclet:ref/Nearest"], "FindClusters" :> 
          Documentation`HelpLookup["paclet:ref/FindClusters"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"FindMinimum\"\>", 
       2->"\<\"NMinimize\"\>", 3->"\<\"Nearest\"\>", 
       4->"\<\"FindClusters\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Discrete Mathematics" :> 
          Documentation`HelpLookup["paclet:guide/DiscreteMathematics"], 
          "Optimization" :> 
          Documentation`HelpLookup["paclet:guide/Optimization"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Discrete Mathematics\"\>", 
       2->"\<\"Optimization\"\>", 
       3->"\<\"New in 6.0: Mathematics & Algorithms\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["FindShortestTour", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["FindShortestTour",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/FindShortestTour"], "[", 
       RowBox[{"{", 
        RowBox[{
         SubscriptBox[
          StyleBox["e", "TI"], 
          StyleBox["1", "TR"]], ",", 
         SubscriptBox[
          StyleBox["e", "TI"], 
          StyleBox["2", "TR"]], ",", 
         StyleBox["\[Ellipsis]", "TR"]}], "}"}], "]"}]], "InlineFormula"],
     " \[LineSeparator]attempts to find an ordering of the ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["e", "TI"], 
       StyleBox["i", "TI"]]], "InlineFormula"],
     " that minimizes the total distance on a tour that visits all the ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["e", "TI"], 
       StyleBox["i", "TI"]]], "InlineFormula"],
     " once."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["The following options can be given:", "Notes",
 CellID->1067943069],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "DistanceFunction", Cell[
    "the distance function to apply to pairs of objects", "TableText"]},
   {Cell["      ", "TableRowIcon"], "Method", Cell[
    "the method to use", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{"Rows" -> {{True, True, True}}}},
 CellID->173046494],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["e", "TI"], 
   StyleBox["i", "TI"]]], "InlineFormula"],
 " can be numbers or lists of numbers, in which case the default distance \
function used is ",
 Cell[BoxData[
  ButtonBox["EuclideanDistance",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/EuclideanDistance"]], "InlineFormula"],
 "."
}], "Notes",
 CellID->931768780],

Cell[TextData[{
 "If the ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["e", "TI"], 
   StyleBox["i", "TI"]]], "InlineFormula"],
 " are strings, the default distance function used is ",
 Cell[BoxData[
  ButtonBox["EditDistance",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/EditDistance"]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->155968330],

Cell[TextData[{
 "For small numbers of points, ",
 Cell[BoxData[
  ButtonBox["FindShortestTour",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindShortestTour"]], "InlineFormula"],
 " will usually find the shortest possible tour. For larger numbers of points \
it will normally find a tour whose length is at least close to the minimum."
}], "Notes",
 CellID->146039061]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->505797371],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->44293591],

Cell["\<\
Find the length and ordering of a shortest tour through six points in the \
plane:\
\>", "ExampleText",
 CellID->309019030],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindShortestTour", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"4", ",", "3"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"2", ",", "3"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"3", ",", 
      RowBox[{"-", "5"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"-", "1"}], ",", "2"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"3", ",", "4"}], "}"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->286910787],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"11", "+", 
    SqrtBox["2"], "+", 
    SqrtBox["5"], "+", 
    RowBox[{"3", " ", 
     SqrtBox["10"]}]}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "3", ",", "5", ",", "2", ",", "4", ",", "6"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{282, 22},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->291797466]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->48238906],

Cell["Specify a list of points:", "ExampleText",
 CellID->662267519],

Cell[BoxData[
 RowBox[{
  RowBox[{"pts", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "4"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "5"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "5"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "4"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "5"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4", ",", "5"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"5", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"5", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"5", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"5", ",", "4"}], "}"}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->36542884],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindShortestTour", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->13271528],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"14", "+", 
    RowBox[{"5", " ", 
     SqrtBox["2"]}]}], ",", 
   RowBox[{"{", 
    RowBox[{
    "1", ",", "6", ",", "10", ",", "9", ",", "13", ",", "16", ",", "17", ",", 
     "14", ",", "18", ",", "19", ",", "15", ",", "12", ",", "11", ",", "8", 
     ",", "5", ",", "4", ",", "7", ",", "3", ",", "2"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{509, 22},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->510388478]
}, Open  ]],

Cell["Order the points according to the tour found:", "ExampleText",
 CellID->287257706],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"pts", "[", 
  RowBox[{"[", 
   RowBox[{"Last", "[", "%", "]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->8620312],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"2", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"3", ",", "2"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"3", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"4", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"5", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"5", ",", "2"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"4", ",", "3"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"5", ",", "3"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"5", ",", "4"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"4", ",", "5"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"3", ",", "5"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"3", ",", "4"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"2", ",", "5"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "5"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "4"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"2", ",", "3"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "3"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2"}], "}"}]}], "}"}]], "Output",
 ImageSize->{511, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->419391993]
}, Open  ]],

Cell["Plot the tour:", "ExampleText",
 CellID->414721653],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics", "[", 
  RowBox[{"Line", "[", "%", "]"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->778367117],

Cell[BoxData[
 GraphicsBox[
  LineBox[{{1, 1}, {2, 1}, {3, 2}, {3, 1}, {4, 1}, {5, 1}, {5, 2}, {4, 3}, {5,
    3}, {5, 4}, {4, 5}, {3, 5}, {3, 4}, {2, 5}, {1, 5}, {1, 4}, {2, 3}, {1, 
   3}, {1, 2}}],
  ImageSize->{87., Automatic}]], "Output",
 ImageSize->{91, 87},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->30697977]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->529878775],

Cell["Find the shortest tour through points in 3D space:", "ExampleText",
 CellID->339568718],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindShortestTour", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1", ",", "4", ",", "3"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "2", ",", "3"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"2", ",", "3", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"3", ",", 
      RowBox[{"-", "5"}], ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"2", ",", 
      RowBox[{"-", "1"}], ",", "2"}], "}"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->151278327],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"3", " ", 
     SqrtBox["2"]}], "+", 
    RowBox[{"3", " ", 
     SqrtBox["3"]}], "+", 
    RowBox[{"2", " ", 
     SqrtBox["6"]}], "+", 
    SqrtBox["57"]}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "3", ",", "2", ",", "4", ",", "5"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{294, 22},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->586957610]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->107061988],

Cell["Find the shortest tour through a list of strings:", "ExampleText",
 CellID->162803421],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindShortestTour", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "\"\<cat\>\"", ",", "\"\<dog\>\"", ",", "\"\<do\>\"", ",", "\"\<cap\>\"", 
     ",", "\"\<dock\>\""}], "}"}], ",", 
   RowBox[{"DistanceFunction", "\[Rule]", "EditDistance"}]}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->32969921],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"11", ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "3", ",", "5", ",", "4"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{137, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->280895951]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->440544881],

Cell[TextData[{
 "Use a ",
 Cell[BoxData[
  ButtonBox["DistanceFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DistanceFunction"]], "InlineFormula"],
 " based on a (symmetric) connectivity matrix: "
}], "ExampleText",
 CellID->22766153],

Cell[BoxData[{
 RowBox[{
  RowBox[{"co", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "4"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "4"}], "}"}]}], "}"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"co", "=", 
   RowBox[{"Join", "[", 
    RowBox[{"co", ",", 
     RowBox[{"Map", "[", 
      RowBox[{"Reverse", ",", "co"}], "]"}]}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->104298834],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindShortestTour", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "3", ",", "4"}], "}"}], ",", 
   RowBox[{"DistanceFunction", "\[Rule]", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"If", "[", 
       RowBox[{
        RowBox[{"MemberQ", "[", 
         RowBox[{"co", ",", 
          RowBox[{"{", 
           RowBox[{"#1", ",", "#2"}], "}"}]}], "]"}], ",", "1", ",", 
        "Infinity"}], "]"}], "&"}], ")"}]}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->17256916],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"4", ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "3", ",", "2", ",", "4"}], "}"}]}], "}"}]], "Output",
 ImageSize->{112, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->310399958]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Options",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->273407831],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Method",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->65045005],

Cell[TextData[{
 "This finds all points on a ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", " ", "\[Cross]", "n"}], TraditionalForm]], "InlineMath"],
 " grid with coordinates that are coprime:"
}], "ExampleText",
 CellID->37322884],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"pts", "[", "n_", "]"}], ":=", 
   RowBox[{"Select", "[", 
    RowBox[{
     RowBox[{"Tuples", "[", 
      RowBox[{
       RowBox[{"Range", "[", "n", "]"}], ",", "2"}], "]"}], ",", 
     RowBox[{
      RowBox[{"CoprimeQ", "@@", "#"}], "&"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->406655626],

Cell[TextData[{
 "Find the shortest tour using ",
 Cell[BoxData["\"\<OrZweig\>\""], "InlineFormula"],
 " method, the default for 2D real inputs:"
}], "ExampleText",
 CellID->437056151],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"len", ",", "tour"}], "}"}], "=", 
  RowBox[{"FindShortestTour", "[", 
   RowBox[{
    RowBox[{"pts", "[", "10", "]"}], ",", 
    RowBox[{"Method", "\[Rule]", "\"\<OrZweig\>\""}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->308876977],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"57", "+", 
    RowBox[{"15", " ", 
     SqrtBox["2"]}], "+", 
    SqrtBox["5"]}], ",", 
   RowBox[{"{", 
    RowBox[{
    "1", ",", "11", ",", "17", ",", "16", ",", "23", ",", "28", ",", "36", 
     ",", "39", ",", "48", ",", "54", ",", "53", ",", "60", ",", "61", ",", 
     "49", ",", "55", ",", "56", ",", "50", ",", "37", ",", "25", ",", "19", 
     ",", "18", ",", "31", ",", "24", ",", "30", ",", "29", ",", "40", ",", 
     "41", ",", "42", ",", "43", ",", "44", ",", "51", ",", "57", ",", "58", 
     ",", "62", ",", "63", ",", "59", ",", "47", ",", "52", ",", "46", ",", 
     "45", ",", "38", ",", "26", ",", "20", ",", "32", ",", "33", ",", "34", 
     ",", "35", ",", "27", ",", "21", ",", "15", ",", "22", ",", "10", ",", 
     "9", ",", "8", ",", "14", ",", "7", ",", "6", ",", "13", ",", "5", ",", 
     "4", ",", "12", ",", "3", ",", "2"}], "}"}]}], "}"}]], "Output",
 ImageSize->{576, 61},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->217572224]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"pts", "[", "10", "]"}], "[", 
    RowBox[{"[", "tour", "]"}], "]"}], ",", 
   RowBox[{"Mesh", "\[Rule]", "All"}], ",", 
   RowBox[{"PlotLabel", "\[Rule]", 
    RowBox[{"N", "[", "len", "]"}]}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->140408445],

Cell[BoxData[
 GraphicsBox[GraphicsComplexBox[CompressedData["
1:eJyFkzEKAzEMBEUqlypcuLjCMf7H+Q15QiB1vp4nJCFs4YElB4cZaY0stLre
n7fHJSLOz/89f9/r3M9YO5e1xwvyCa7gBj7AHTxQbyA/DetdHaz7Ca7Qixs4
wcUwdYnT3ROr38NwASe4gtVHR3wYlm4iP6Ebf3QdOldXuoY4+2Lfbk7Mc66q
Tx/QN/Sd8s539C19zTm6PWjQc4/ICT33khzrDaI+PS0=
   "], {{{}, {}, 
     {Hue[0.67, 0.6, 0.6], 
      LineBox[{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 
       19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36,
        37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 
       54, 55, 56, 57, 58, 59, 60, 61, 62, 63}]}}, {
     {Hue[0.67, 0.6, 0.6], 
      PointBox[{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18,
        19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 
       36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53,
        54, 55, 56, 57, 58, 59, 60, 61, 62, 63}]}, {}, {}}}],
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotLabel->FormBox["80.44927141309621`", TraditionalForm],
  PlotRange->{{0., 10.}, {0., 10.}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 137},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->32759629]
}, Open  ]],

Cell[TextData[{
 "Finding shortest tour using ",
 Cell[BoxData["\"\<OrOpt\>\""], "InlineFormula"],
 " method, the default for non-2D or not real inputs:"
}], "ExampleText",
 CellID->179988326],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"len", ",", "tour"}], "}"}], "=", 
  RowBox[{"FindShortestTour", "[", 
   RowBox[{
    RowBox[{"pts", "[", "10", "]"}], ",", 
    RowBox[{"Method", "\[Rule]", "\"\<OrOpt\>\""}]}], "]"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->290723926],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"46", "+", 
    RowBox[{"21", " ", 
     SqrtBox["2"]}]}], ",", 
   RowBox[{"{", 
    RowBox[{
    "1", ",", "11", ",", "17", ",", "16", ",", "23", ",", "28", ",", "36", 
     ",", "39", ",", "48", ",", "53", ",", "60", ",", "54", ",", "61", ",", 
     "55", ",", "56", ",", "50", ",", "37", ",", "31", ",", "25", ",", "19", 
     ",", "18", ",", "24", ",", "30", ",", "29", ",", "40", ",", "49", ",", 
     "41", ",", "42", ",", "43", ",", "44", ",", "51", ",", "57", ",", "62", 
     ",", "58", ",", "63", ",", "59", ",", "52", ",", "47", ",", "46", ",", 
     "45", ",", "38", ",", "32", ",", "33", ",", "34", ",", "35", ",", "27", 
     ",", "26", ",", "20", ",", "21", ",", "15", ",", "22", ",", "10", ",", 
     "9", ",", "8", ",", "14", ",", "7", ",", "6", ",", "13", ",", "5", ",", 
     "4", ",", "12", ",", "3", ",", "2"}], "}"}]}], "}"}]], "Output",
 ImageSize->{565, 61},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->368079397]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"pts", "[", "10", "]"}], "[", 
    RowBox[{"[", "tour", "]"}], "]"}], ",", 
   RowBox[{"Mesh", "\[Rule]", "All"}], ",", 
   RowBox[{"PlotLabel", "\[Rule]", 
    RowBox[{"N", "[", "len", "]"}]}]}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->284882423],

Cell[BoxData[
 GraphicsBox[GraphicsComplexBox[CompressedData["
1:eJyNkzEKwzAMRUUmjxoyePCgGt8jPkOPUOjcq+cIaSl/8ANBAkE8/R+hWPLj
9Xm+NzM7vu8v/p/zWKPNlctc8wW6g3dwBTdwgDt4JLr6ka6+pDtYfQW4ghUd
UfmSMP2ZT6z+Gzjgbwk7eAfrvwL5Dh6JHtA74oDvbl2xvqvIZ3OhzjmqPufO
c8rOVfU0hw6de5jtJfeac87uQYWf94js8PNekm1ew/49LQ==
   "], {{{}, {}, 
     {Hue[0.67, 0.6, 0.6], 
      LineBox[{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 
       19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36,
        37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 
       54, 55, 56, 57, 58, 59, 60, 61, 62, 63}]}}, {
     {Hue[0.67, 0.6, 0.6], 
      PointBox[{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18,
        19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 
       36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53,
        54, 55, 56, 57, 58, 59, 60, 61, 62, 63}]}, {}, {}}}],
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotLabel->FormBox["75.698484809835`", TraditionalForm],
  PlotRange->{{0., 10.}, {0., 10.}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 137},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->326702914]
}, Open  ]],

Cell[TextData[{
 "The ",
 Cell[BoxData["\"\<TwoOpt\>\""], "InlineFormula"],
 " algorithm performs exchanges of edge endpoints for improvement:"
}], "ExampleText",
 CellID->82904636],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"len", ",", "tour"}], "}"}], "=", 
  RowBox[{"FindShortestTour", "[", 
   RowBox[{
    RowBox[{"pts", "[", "10", "]"}], ",", 
    RowBox[{"Method", "\[Rule]", "\"\<TwoOpt\>\""}]}], "]"}]}]], "Input",
 CellLabel->"In[6]:=",
 CellID->552708790],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"43", "+", 
    RowBox[{"22", " ", 
     SqrtBox["2"]}], "+", 
    SqrtBox["5"]}], ",", 
   RowBox[{"{", 
    RowBox[{
    "1", ",", "11", ",", "17", ",", "16", ",", "23", ",", "28", ",", "29", 
     ",", "30", ",", "31", ",", "24", ",", "18", ",", "19", ",", "25", ",", 
     "20", ",", "26", ",", "32", ",", "37", ",", "42", ",", "41", ",", "40", 
     ",", "36", ",", "39", ",", "48", ",", "54", ",", "53", ",", "60", ",", 
     "61", ",", "49", ",", "55", ",", "56", ",", "50", ",", "43", ",", "44", 
     ",", "51", ",", "57", ",", "62", ",", "58", ",", "63", ",", "59", ",", 
     "52", ",", "47", ",", "46", ",", "45", ",", "38", ",", "33", ",", "34", 
     ",", "35", ",", "27", ",", "21", ",", "15", ",", "22", ",", "10", ",", 
     "9", ",", "8", ",", "14", ",", "7", ",", "6", ",", "13", ",", "5", ",", 
     "4", ",", "12", ",", "3", ",", "2"}], "}"}]}], "}"}]], "Output",
 ImageSize->{576, 61},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->83119295]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"pts", "[", "10", "]"}], "[", 
    RowBox[{"[", "tour", "]"}], "]"}], ",", 
   RowBox[{"Mesh", "\[Rule]", "All"}], ",", 
   RowBox[{"PlotLabel", "\[Rule]", 
    RowBox[{"N", "[", "len", "]"}]}]}], "]"}]], "Input",
 CellLabel->"In[7]:=",
 CellID->49853226],

Cell[BoxData[
 GraphicsBox[GraphicsComplexBox[CompressedData["
1:eJyNkzEKwzAMRUUmjx48aPDgGN8jPkOOEOjcq/cITSl/8ANBAkE86X8FWfF+
vc/XZmbH/f7i//kca7S5cpprPqGewSVg+cUJnBETIvP0sW9FvULniAU6+thX
88iveSu4gTv8HfURsL7fwPJncIGe85EdeuU7eAT1hnpHHNA97SuWz5GP5uLc
6sdz4rlyL6pHe+FeuXf1439Bduije8R7F91Lss0vIaI9LQ==
   "], {{{}, {}, 
     {Hue[0.67, 0.6, 0.6], 
      LineBox[{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 
       19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36,
        37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 
       54, 55, 56, 57, 58, 59, 60, 61, 62, 63}]}}, {
     {Hue[0.67, 0.6, 0.6], 
      PointBox[{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18,
        19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 
       36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53,
        54, 55, 56, 57, 58, 59, 60, 61, 62, 63}]}, {}, {}}}],
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotLabel->FormBox["76.34876634970789`", TraditionalForm],
  PlotRange->{{0., 10.}, {0., 10.}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 137},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->298123795]
}, Open  ]],

Cell[TextData[{
 " ",
 Cell[BoxData["\"\<CCA\>\""], "InlineFormula"],
 " (Convex hull, Cheapest insertion and Angle selection) intended for points \
in ",
 Cell[BoxData[
  SuperscriptBox["\[DoubleStruckCapitalR]", "n"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->61580125],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"len", ",", "tour"}], "}"}], "=", 
  RowBox[{"FindShortestTour", "[", 
   RowBox[{
    RowBox[{"pts", "[", "10", "]"}], ",", 
    RowBox[{"Method", "\[Rule]", "\"\<CCA\>\""}]}], "]"}]}]], "Input",
 CellLabel->"In[8]:=",
 CellID->3306480],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"57", "+", 
    RowBox[{"15", " ", 
     SqrtBox["2"]}], "+", 
    SqrtBox["5"]}], ",", 
   RowBox[{"{", 
    RowBox[{
    "1", ",", "11", ",", "17", ",", "16", ",", "23", ",", "28", ",", "36", 
     ",", "39", ",", "48", ",", "54", ",", "53", ",", "60", ",", "61", ",", 
     "49", ",", "55", ",", "56", ",", "50", ",", "37", ",", "25", ",", "19", 
     ",", "18", ",", "31", ",", "24", ",", "30", ",", "29", ",", "40", ",", 
     "41", ",", "42", ",", "43", ",", "44", ",", "51", ",", "57", ",", "58", 
     ",", "62", ",", "63", ",", "59", ",", "47", ",", "52", ",", "46", ",", 
     "45", ",", "38", ",", "26", ",", "20", ",", "32", ",", "33", ",", "34", 
     ",", "35", ",", "27", ",", "21", ",", "15", ",", "22", ",", "10", ",", 
     "9", ",", "8", ",", "14", ",", "7", ",", "6", ",", "13", ",", "5", ",", 
     "4", ",", "12", ",", "3", ",", "2"}], "}"}]}], "}"}]], "Output",
 ImageSize->{576, 61},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->512845898]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"pts", "[", "10", "]"}], "[", 
    RowBox[{"[", "tour", "]"}], "]"}], ",", 
   RowBox[{"Mesh", "\[Rule]", "All"}], ",", 
   RowBox[{"PlotLabel", "\[Rule]", 
    RowBox[{"N", "[", "len", "]"}]}]}], "]"}]], "Input",
 CellLabel->"In[9]:=",
 CellID->94849089],

Cell[BoxData[
 GraphicsBox[GraphicsComplexBox[CompressedData["
1:eJyFkzEKAzEMBEUqlypcuLjCMf7H+Q15QiB1vp4nJCFs4YElB4cZaY0stLre
n7fHJSLOz/89f9/r3M9YO5e1xwvyCa7gBj7AHTxQbyA/DetdHaz7Ca7Qixs4
wcUwdYnT3ROr38NwASe4gtVHR3wYlm4iP6Ebf3QdOldXuoY4+2Lfbk7Mc66q
Tx/QN/Sd8s539C19zTm6PWjQc4/ICT33khzrDaI+PS0=
   "], {{{}, {}, 
     {Hue[0.67, 0.6, 0.6], 
      LineBox[{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 
       19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36,
        37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 
       54, 55, 56, 57, 58, 59, 60, 61, 62, 63}]}}, {
     {Hue[0.67, 0.6, 0.6], 
      PointBox[{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18,
        19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 
       36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53,
        54, 55, 56, 57, 58, 59, 60, 61, 62, 63}]}, {}, {}}}],
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotLabel->FormBox["80.44927141309621`", TraditionalForm],
  PlotRange->{{0., 10.}, {0., 10.}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 137},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[9]=",
 CellID->21707778]
}, Open  ]],

Cell[TextData[{
 "The ",
 Cell[BoxData["\"\<Greedy\>\""], "InlineFormula"],
 " algorithm moves from one point to the nearest unvisited neighbor:"
}], "ExampleText",
 CellID->220647538],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"len", ",", "tour"}], "}"}], "=", 
  RowBox[{"FindShortestTour", "[", 
   RowBox[{
    RowBox[{"pts", "[", "10", "]"}], ",", 
    RowBox[{"Method", "\[Rule]", "\"\<Greedy\>\""}]}], "]"}]}]], "Input",
 CellLabel->"In[10]:=",
 CellID->402198568],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"44", "+", 
    RowBox[{"22", " ", 
     SqrtBox["2"]}], "+", 
    RowBox[{"2", " ", 
     SqrtBox["10"]}]}], ",", 
   RowBox[{"{", 
    RowBox[{
    "1", ",", "11", ",", "16", ",", "23", ",", "28", ",", "36", ",", "39", 
     ",", "48", ",", "53", ",", "60", ",", "54", ",", "61", ",", "55", ",", 
     "56", ",", "50", ",", "43", ",", "44", ",", "51", ",", "57", ",", "62", 
     ",", "58", ",", "63", ",", "59", ",", "52", ",", "46", ",", "47", ",", 
     "45", ",", "38", ",", "33", ",", "34", ",", "35", ",", "27", ",", "22", 
     ",", "15", ",", "9", ",", "10", ",", "8", ",", "7", ",", "14", ",", "20",
      ",", "26", ",", "32", ",", "37", ",", "42", ",", "41", ",", "49", ",", 
     "40", ",", "29", ",", "30", ",", "31", ",", "25", ",", "19", ",", "18", 
     ",", "24", ",", "17", ",", "12", ",", "3", ",", "4", ",", "5", ",", "13",
      ",", "6", ",", "21", ",", "2"}], "}"}]}], "}"}]], "Output",
 ImageSize->{573, 61},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[10]=",
 CellID->568591244]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"pts", "[", "10", "]"}], "[", 
    RowBox[{"[", "tour", "]"}], "]"}], ",", 
   RowBox[{"Mesh", "\[Rule]", "All"}], ",", 
   RowBox[{"PlotLabel", "\[Rule]", 
    RowBox[{"N", "[", "len", "]"}]}]}], "]"}]], "Input",
 CellLabel->"In[11]:=",
 CellID->334097482],

Cell[BoxData[
 GraphicsBox[GraphicsComplexBox[CompressedData["
1:eJyNkzEKwzAMRUUnjR48eMjgGt/DPUOOEMicq+cIaSmf4geiCYTPs2T5y0bP
7Vj3h5mN9//R73eOWe01s4MTOIMLeAFXcAP3IC5fisuX4gksXxW8BFyQr/UG
7kG8It6gHXms++8c7StYj/pi36rPe9I96jxD/O478V3lR/XEDk5g+S1Q9sl9
qlvB9JEDdnCCRnke5DvUoJyru3NG35w73sNvri89Rz0t
   "], {{{}, {}, 
     {Hue[0.67, 0.6, 0.6], 
      LineBox[{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 
       19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36,
        37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 
       54, 55, 56, 57, 58, 59, 60, 61, 62, 63}]}}, {
     {Hue[0.67, 0.6, 0.6], 
      PointBox[{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18,
        19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 
       36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53,
        54, 55, 56, 57, 58, 59, 60, 61, 62, 63}]}, {}, {}}}],
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotLabel->FormBox["81.43725369254486`", TraditionalForm],
  PlotRange->{{0., 10.}, {0., 10.}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 137},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[11]=",
 CellID->193990691]
}, Open  ]],

Cell[TextData[{
 " ",
 Cell[BoxData["\"\<GreedyCycle\>\""], "InlineFormula"],
 " is a variant of ",
 Cell[BoxData["\"\<Greedy\>\""], "InlineFormula"],
 " algorithm with known upper bound: "
}], "ExampleText",
 CellID->70982352],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"len", ",", "tour"}], "}"}], "=", 
  RowBox[{"FindShortestTour", "[", 
   RowBox[{
    RowBox[{"pts", "[", "10", "]"}], ",", 
    RowBox[{"Method", "\[Rule]", "\"\<GreedyCycle\>\""}]}], "]"}]}]], "Input",\

 CellLabel->"In[12]:=",
 CellID->737289853],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"65", "+", 
    RowBox[{"20", " ", 
     SqrtBox["2"]}], "+", 
    SqrtBox["5"], "+", 
    RowBox[{"2", " ", 
     SqrtBox["10"]}]}], ",", 
   RowBox[{"{", 
    RowBox[{
    "1", ",", "11", ",", "16", ",", "23", ",", "28", ",", "36", ",", "39", 
     ",", "48", ",", "53", ",", "60", ",", "61", ",", "54", ",", "40", ",", 
     "41", ",", "49", ",", "42", ",", "43", ",", "50", ",", "51", ",", "52", 
     ",", "63", ",", "46", ",", "47", ",", "59", ",", "45", ",", "57", ",", 
     "62", ",", "58", ",", "56", ",", "55", ",", "44", ",", "37", ",", "29", 
     ",", "30", ",", "31", ",", "24", ",", "17", ",", "2", ",", "3", ",", 
     "12", ",", "4", ",", "5", ",", "13", ",", "19", ",", "25", ",", "18", 
     ",", "6", ",", "7", ",", "14", ",", "20", ",", "26", ",", "33", ",", 
     "38", ",", "34", ",", "35", ",", "27", ",", "32", ",", "21", ",", "22", 
     ",", "8", ",", "9", ",", "15", ",", "10"}], "}"}]}], "}"}]], "Output",
 ImageSize->{536, 78},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[12]=",
 CellID->64571147]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"pts", "[", "10", "]"}], "[", 
    RowBox[{"[", "tour", "]"}], "]"}], ",", 
   RowBox[{"Mesh", "\[Rule]", "All"}], ",", 
   RowBox[{"PlotLabel", "\[Rule]", 
    RowBox[{"N", "[", "len", "]"}]}]}], "]"}]], "Input",
 CellLabel->"In[13]:=",
 CellID->274712270],

Cell[BoxData[
 GraphicsBox[GraphicsComplexBox[CompressedData["
1:eJyNkzEKgDAQBA+rlClSWKSIIf/QN/gEwdqv+wQV2SIDhwZkGe+yXMJm2o51
H8xsvr9H33XOvdrScwBHcAKP4Awu4ApuDmsO9WvO7LD6C1j1CE7oJ2ew5mjg
7DD7Gur05T76sj+Beb4RmqAGDuAIDVDmxsuR/v/N1d+5WKefl0svN1854b1+
5ajCr6IuHy+3fBfeO9J5KuptuQDZGT0t
   "], {{{}, {}, 
     {Hue[0.67, 0.6, 0.6], 
      LineBox[{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 
       19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36,
        37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 
       54, 55, 56, 57, 58, 59, 60, 61, 62, 63}]}}, {
     {Hue[0.67, 0.6, 0.6], 
      PointBox[{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18,
        19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 
       36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53,
        54, 55, 56, 57, 58, 59, 60, 61, 62, 63}]}, {}, {}}}],
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotLabel->FormBox["101.84489454529844`", TraditionalForm],
  PlotRange->{{0., 10.}, {0., 10.}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 137},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[13]=",
 CellID->275041580]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData["\"\<SimulatedAnnealing\>\""], "InlineFormula"],
 " uses simulated annealing to minimize the tour length:"
}], "ExampleText",
 CellID->329753893],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"len", ",", "tour"}], "}"}], "=", 
  RowBox[{"FindShortestTour", "[", 
   RowBox[{
    RowBox[{"pts", "[", "5", "]"}], ",", 
    RowBox[{"Method", "\[Rule]", "\"\<SimulatedAnnealing\>\""}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[14]:=",
 CellID->439870596],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"14", "+", 
    RowBox[{"5", " ", 
     SqrtBox["2"]}]}], ",", 
   RowBox[{"{", 
    RowBox[{
    "15", ",", "19", ",", "14", ",", "18", ",", "17", ",", "16", ",", "13", 
     ",", "10", ",", "9", ",", "6", ",", "1", ",", "2", ",", "3", ",", "7", 
     ",", "4", ",", "5", ",", "8", ",", "12", ",", "11"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{509, 22},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[14]=",
 CellID->342557596]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"pts", "[", "5", "]"}], "[", 
    RowBox[{"[", "tour", "]"}], "]"}], ",", 
   RowBox[{"Mesh", "\[Rule]", "All"}], ",", 
   RowBox[{"AxesOrigin", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"0", ",", "0"}], "}"}]}], ",", 
   RowBox[{"PlotLabel", "\[Rule]", 
    RowBox[{"N", "[", "len", "]"}]}]}], "]"}]], "Input",
 CellLabel->"In[15]:=",
 CellID->142661838],

Cell[BoxData[
 GraphicsBox[
  GraphicsComplexBox[{{4., 5.}, {5., 4.}, {4., 3.}, {5., 3.}, {5., 2.}, {5., 
   1.}, {4., 1.}, {3., 2.}, {3., 1.}, {2., 1.}, {1., 1.}, {1., 2.}, {1., 
   3.}, {2., 3.}, {1., 4.}, {1., 5.}, {2., 5.}, {3., 5.}, {3., 
   4.}}, {{{}, {}, 
     {Hue[0.67, 0.6, 0.6], 
      LineBox[{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 
       19}]}}, {
     {Hue[0.67, 0.6, 0.6], 
      PointBox[{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18,
        19}]}, {}, {}}}],
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotLabel->FormBox["21.071067811865476`", TraditionalForm],
  PlotRange->{{0, 5.}, {0, 5.}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 141},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[15]=",
 CellID->149114842]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "DistanceFunction",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->4743078],

Cell[TextData[{
 "By default ",
 Cell[BoxData[
  ButtonBox["EditDistance",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/EditDistance"]], "InlineFormula"],
 " is used for strings:"
}], "ExampleText",
 CellID->801521965],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindShortestTour", "[", 
  RowBox[{"{", 
   RowBox[{
   "\"\<This\>\"", ",", "\"\<finds\>\"", ",", "\"\<the\>\"", ",", 
    "\"\<shortest\>\"", ",", "\"\<tour\>\"", ",", "\"\<through\>\"", ",", 
    "\"\<a\>\"", ",", "\"\<list\>\"", ",", "\"\<of\>\"", ",", 
    "\"\<strings\>\""}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->157216613],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"39", ",", 
   RowBox[{"{", 
    RowBox[{
    "1", ",", "8", ",", "9", ",", "7", ",", "3", ",", "5", ",", "6", ",", "4",
      ",", "10", ",", "2"}], "}"}]}], "}"}]], "Output",
 ImageSize->{235, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->47871281]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{
   "\"\<This\>\"", ",", "\"\<finds\>\"", ",", "\"\<the\>\"", ",", 
    "\"\<shortest\>\"", ",", "\"\<tour\>\"", ",", "\"\<through\>\"", ",", 
    "\"\<a\>\"", ",", "\"\<list\>\"", ",", "\"\<of\>\"", ",", 
    "\"\<strings\>\""}], "}"}], "[", 
  RowBox[{"[", 
   RowBox[{"%", "[", 
    RowBox[{"[", "2", "]"}], "]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->540832990],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"This\"\>", ",", "\<\"list\"\>", ",", "\<\"of\"\>", 
   ",", "\<\"a\"\>", ",", "\<\"the\"\>", ",", "\<\"tour\"\>", 
   ",", "\<\"through\"\>", ",", "\<\"shortest\"\>", ",", "\<\"strings\"\>", 
   ",", "\<\"finds\"\>"}], "}"}]], "Output",
 ImageSize->{434, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->240013749]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->220547840],

Cell["\<\
This finds the shortest tour through 100 points, with a penalty added to \
cross a \"river\": \
\>", "ExampleText",
 CellID->367818203],

Cell[BoxData[
 RowBox[{
  RowBox[{"pts", "=", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", 
     RowBox[{"{", 
      RowBox[{"100", ",", "2"}], "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->369630947],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"len", ",", "tour"}], "}"}], "=", 
  RowBox[{"FindShortestTour", "[", 
   RowBox[{"pts", ",", 
    RowBox[{"DistanceFunction", "\[Rule]", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{"If", "[", 
         RowBox[{
          RowBox[{
           RowBox[{"(", 
            RowBox[{
             RowBox[{
              RowBox[{"#1", "[", 
               RowBox[{"[", "1", "]"}], "]"}], "\[LessEqual]", ".5"}], "&&", 
             RowBox[{
              RowBox[{"#2", "[", 
               RowBox[{"[", "1", "]"}], "]"}], "\[GreaterEqual]", ".5"}]}], 
            ")"}], "||", 
           RowBox[{"(", 
            RowBox[{
             RowBox[{
              RowBox[{"#2", "[", 
               RowBox[{"[", "1", "]"}], "]"}], "\[LessEqual]", ".5"}], "&&", 
             RowBox[{
              RowBox[{"#1", "[", 
               RowBox[{"[", "1", "]"}], "]"}], "\[GreaterEqual]", ".5"}]}], 
            ")"}]}], ",", "100000", ",", 
          RowBox[{"EuclideanDistance", "[", 
           RowBox[{"#1", ",", "#2"}], "]"}]}], "]"}], ")"}], "&"}], ")"}]}]}],
    "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->649096336],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"200007.53638119693`", ",", 
   RowBox[{"{", 
    RowBox[{
    "1", ",", "21", ",", "45", ",", "91", ",", "78", ",", "64", ",", "77", 
     ",", "79", ",", "53", ",", "6", ",", "31", ",", "28", ",", "29", ",", 
     "80", ",", "33", ",", "83", ",", "18", ",", "35", ",", "99", ",", "97", 
     ",", "44", ",", "95", ",", "16", ",", "51", ",", "13", ",", "48", ",", 
     "52", ",", "84", ",", "41", ",", "24", ",", "32", ",", "73", ",", "75", 
     ",", "26", ",", "55", ",", "82", ",", "22", ",", "61", ",", "3", ",", 
     "87", ",", "34", ",", "20", ",", "25", ",", "54", ",", "96", ",", "50", 
     ",", "86", ",", "74", ",", "8", ",", "15", ",", "63", ",", "12", ",", 
     "7", ",", "47", ",", "90", ",", "58", ",", "4", ",", "23", ",", "60", 
     ",", "59", ",", "72", ",", "94", ",", "37", ",", "38", ",", "98", ",", 
     "2", ",", "70", ",", "9", ",", "71", ",", "42", ",", "88", ",", "46", 
     ",", "30", ",", "39", ",", "43", ",", "93", ",", "68", ",", "62", ",", 
     "92", ",", "40", ",", "49", ",", "57", ",", "27", ",", "36", ",", "81", 
     ",", "76", ",", "19", ",", "56", ",", "11", ",", "14", ",", "85", ",", 
     "5", ",", "65", ",", "66", ",", "89", ",", "10", ",", "17", ",", "69", 
     ",", "67", ",", "100"}], "}"}]}], "}"}]], "Output",
 ImageSize->{541, 84},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->260795598]
}, Open  ]],

Cell["This plots the tour, and the \"river\" in red :", "ExampleText",
 CellID->568247361],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"GraphicsComplex", "[", 
      RowBox[{"pts", ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"Point", "[", 
          RowBox[{"Range", "[", "100", "]"}], "]"}], ",", 
         RowBox[{"Line", "[", "tour", "]"}]}], "}"}]}], "]"}], ",", "Red", 
     ",", 
     RowBox[{"Line", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"0.5", ",", "0"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"0.5", ",", "1"}], "}"}]}], "}"}], "]"}]}], "}"}], ",", 
   RowBox[{"Frame", "\[Rule]", "True"}], ",", 
   RowBox[{"FrameTicks", "\[Rule]", "Automatic"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->297895584],

Cell[BoxData[
 GraphicsBox[{GraphicsComplexBox[CompressedData["
1:eJwVk308FAYDx+8xU/tIeVKsUM1LbqTUNZnid5U4L5lYy+pwhFHKeU6N283b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    "], {PointBox[{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 
      18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 
      36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 
      54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 
      72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 
      90, 91, 92, 93, 94, 95, 96, 97, 98, 99, 100}], 
     LineBox[{1, 21, 45, 91, 78, 64, 77, 79, 53, 6, 31, 28, 29, 80, 33, 83, 
      18, 35, 99, 97, 44, 95, 16, 51, 13, 48, 52, 84, 41, 24, 32, 73, 75, 26, 
      55, 82, 22, 61, 3, 87, 34, 20, 25, 54, 96, 50, 86, 74, 8, 15, 63, 12, 7,
       47, 90, 58, 4, 23, 60, 59, 72, 94, 37, 38, 98, 2, 70, 9, 71, 42, 88, 
      46, 30, 39, 43, 93, 68, 62, 92, 40, 49, 57, 27, 36, 81, 76, 19, 56, 11, 
      14, 85, 5, 65, 66, 89, 10, 17, 69, 67, 100}]}], 
   {RGBColor[1, 0, 0], LineBox[{{0.5, 0}, {0.5, 1}}]}},
  Frame->True,
  FrameTicks->Automatic]], "Output",
 ImageSize->{184, 179},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->321749635]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->13379203],

Cell["\<\
This defines a sparse distance matrix among six points and find the shortest \
tour:\
\>", "ExampleText",
 CellID->244475536],

Cell[BoxData[
 RowBox[{
  RowBox[{"d", "=", 
   RowBox[{"SparseArray", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"{", 
         RowBox[{"1", ",", "2"}], "}"}], "\[Rule]", "1"}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"2", ",", "1"}], "}"}], "\[Rule]", "1"}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"6", ",", "1"}], "}"}], "\[Rule]", "1"}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"6", ",", "2"}], "}"}], "\[Rule]", "1"}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"5", ",", "1"}], "}"}], "\[Rule]", "1"}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"1", ",", "5"}], "}"}], "\[Rule]", "1"}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"2", ",", "6"}], "}"}], "\[Rule]", "1"}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"2", ",", "3"}], "}"}], "\[Rule]", "10"}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"3", ",", "2"}], "}"}], "\[Rule]", "10"}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"3", ",", "5"}], "}"}], "\[Rule]", "1"}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"5", ",", "3"}], "}"}], "\[Rule]", "1"}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"3", ",", "4"}], "}"}], "\[Rule]", "1"}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"4", ",", "3"}], "}"}], "\[Rule]", "1"}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"4", ",", "5"}], "}"}], "\[Rule]", "15"}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"4", ",", "1"}], "}"}], "\[Rule]", "1"}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"5", ",", "4"}], "}"}], "\[Rule]", "15"}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"5", ",", "2"}], "}"}], "\[Rule]", "1"}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"1", ",", "4"}], "}"}], "\[Rule]", "1"}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"2", ",", "5"}], "}"}], "\[Rule]", "1"}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"1", ",", "6"}], "}"}], "\[Rule]", "1"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"6", ",", "6"}], "}"}], ",", "Infinity"}], "]"}]}], 
  ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->785067326],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"len", ",", "tour"}], "}"}], "=", 
  RowBox[{"FindShortestTour", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1", ",", "2", ",", "3", ",", "4", ",", "5", ",", "6"}], "}"}], 
    ",", 
    RowBox[{"DistanceFunction", "\[Rule]", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"d", "[", 
        RowBox[{"[", 
         RowBox[{"#1", ",", "#2"}], "]"}], "]"}], "&"}], ")"}]}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->230946992],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"6", ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "4", ",", "3", ",", "5", ",", "2", ",", "6"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{149, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->287264757]
}, Open  ]],

Cell["\<\
This plot the shortest tour in red, as well as the distance on each edge:\
\>", "ExampleText",
 CellID->552808591],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"GraphPlot", "[", 
  RowBox[{"d", ",", 
   RowBox[{"VertexLabeling", "\[Rule]", "True"}], ",", 
   RowBox[{"EdgeRenderingFunction", "\[Rule]", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"If", "[", 
         RowBox[{
          RowBox[{"MemberQ", "[", 
           RowBox[{
            RowBox[{"Thread", "[", 
             RowBox[{"{", 
              RowBox[{"tour", ",", 
               RowBox[{"RotateLeft", "[", "tour", "]"}]}], "}"}], "]"}], ",", 
            
            RowBox[{"#2", "|", 
             RowBox[{"Reverse", "[", "#2", "]"}]}]}], "]"}], ",", "Red", ",", 
          "Yellow"}], "]"}], ",", 
        RowBox[{"Line", "[", "#1", "]"}], ",", "Black", ",", 
        RowBox[{"Text", "[", 
         RowBox[{
          RowBox[{"d", "[", 
           RowBox[{"[", 
            RowBox[{
             RowBox[{"#2", "[", 
              RowBox[{"[", "1", "]"}], "]"}], ",", 
             RowBox[{"#2", "[", 
              RowBox[{"[", "2", "]"}], "]"}]}], "]"}], "]"}], ",", 
          RowBox[{"Mean", "[", "#1", "]"}], ",", 
          RowBox[{"Background", "\[Rule]", "White"}]}], "]"}]}], "}"}], "&"}],
      ")"}]}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->392599670],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 92},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->275214439]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(6)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->258228157],

Cell["Find a shortest tour visiting 50 random points in the plane:", \
"ExampleText",
 CellID->266728988],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"With", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"p", "=", 
     RowBox[{"RandomReal", "[", 
      RowBox[{"10", ",", 
       RowBox[{"{", 
        RowBox[{"50", ",", "2"}], "}"}]}], "]"}]}], "}"}], ",", 
   RowBox[{"Graphics", "[", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"Line", "[", 
       RowBox[{"p", "[", 
        RowBox[{"[", 
         RowBox[{"Last", "[", 
          RowBox[{"FindShortestTour", "[", "p", "]"}], "]"}], "]"}], "]"}], 
       "]"}], ",", 
      RowBox[{"PointSize", "[", "Large", "]"}], ",", "Red", ",", 
      RowBox[{"Point", "[", "p", "]"}]}], "}"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->809342350],

Cell[BoxData[
 GraphicsBox[{
   LineBox[{{8.949691839708398, 8.265355512496779}, {8.01164738002604, 
    8.266194882078048}, {5.7442481013694335`, 8.84278732681696}, {
    5.53924857572206, 7.381380779532338}, {5.942441274822237, 
    6.279938961168763}, {4.974352212270117, 6.1260762090344345`}, {
    4.567057973287865, 6.480574631094161}, {3.4769409277557184`, 
    6.149554583198073}, {4.340228830169424, 7.858927572319647}, {
    3.67040750900103, 8.792053845024173}, {2.545916064415288, 
    9.792430163427566}, {2.627138177938866, 9.067177862547386}, {
    2.9134344084242514`, 8.045242315723183}, {3.0591864307588557`, 
    7.891303646573311}, {2.795963112509221, 7.254573598155791}, {
    2.6357568784535212`, 7.469300479622262}, {1.5825073257602806`, 
    7.940462566488837}, {0.40716482311047963`, 7.033919748606106}, {
    2.05709781872363, 4.517602099408604}, {0.5790329268743455, 
    3.425247887104481}, {1.8227697880068061`, 1.4297399732652472`}, {
    1.994217359921473, 1.1781619112911712`}, {4.892316759638056, 
    4.066841209679306}, {5.0934213540168605`, 2.78434665854955}, {
    4.915834308465666, 2.1122767411755894`}, {4.868097296023704, 
    0.2316182796725741}, {5.558163727532066, 0.5729357636942023}, {
    6.115301101683254, 0.7688764364978584}, {6.012246736809848, 
    0.9365127404774198}, {6.054179372808828, 1.5885410140626788`}, {
    6.325599975593876, 1.6206795363157322`}, {6.9847697914364115`, 
    1.7688223069998643`}, {7.264117142748687, 2.5426822952775385`}, {
    7.58403217367669, 3.1154396420807284`}, {8.141273301757124, 
    3.138865286434074}, {8.513254883564407, 0.30051991590644533`}, {
    9.247555108671792, 0.0861976965100153}, {9.237574827844014, 
    1.3992483206329887`}, {9.832431679913025, 1.8379720905399033`}, {
    9.539775194370128, 3.322553894570559}, {9.153272893477364, 
    4.030634206866834}, {8.975735161401692, 4.606755104297353}, {
    8.836779748837236, 4.434369020225912}, {8.486780460759443, 
    3.929042746557563}, {7.938571877275116, 4.525885189208254}, {
    8.096501833478374, 5.003926848075029}, {6.296802984854082, 
    4.837759887710524}, {6.9779990721095375`, 5.634670616102908}, {
    8.22857911045826, 6.513954338259285}, {9.685197870412805, 
    8.369603428881742}}], 
   {RGBColor[1, 0, 0], PointSize[Large], 
    PointBox[{{8.949691839708398, 8.265355512496779}, {2.9134344084242514`, 
     8.045242315723183}, {9.832431679913025, 1.8379720905399033`}, {
     8.975735161401692, 4.606755104297353}, {8.096501833478374, 
     5.003926848075029}, {2.795963112509221, 7.254573598155791}, {
     4.974352212270117, 6.1260762090344345`}, {8.01164738002604, 
     8.266194882078048}, {6.054179372808828, 1.5885410140626788`}, {
     4.567057973287865, 6.480574631094161}, {7.58403217367669, 
     3.1154396420807284`}, {6.9847697914364115`, 1.7688223069998643`}, {
     8.22857911045826, 6.513954338259285}, {4.340228830169424, 
     7.858927572319647}, {8.836779748837236, 4.434369020225912}, {
     5.0934213540168605`, 2.78434665854955}, {6.296802984854082, 
     4.837759887710524}, {8.513254883564407, 0.30051991590644533`}, {
     5.558163727532066, 0.5729357636942023}, {7.264117142748687, 
     2.5426822952775385`}, {3.0591864307588557`, 7.891303646573311}, {
     9.153272893477364, 4.030634206866834}, {5.7442481013694335`, 
     8.84278732681696}, {0.40716482311047963`, 7.033919748606106}, {
     5.942441274822237, 6.279938961168763}, {1.5825073257602806`, 
     7.940462566488837}, {8.141273301757124, 3.138865286434074}, {
     4.892316759638056, 4.066841209679306}, {4.868097296023704, 
     0.2316182796725741}, {2.627138177938866, 9.067177862547386}, {
     1.994217359921473, 1.1781619112911712`}, {6.9779990721095375`, 
     5.634670616102908}, {0.5790329268743455, 3.425247887104481}, {
     8.486780460759443, 3.929042746557563}, {6.012246736809848, 
     0.9365127404774198}, {3.4769409277557184`, 6.149554583198073}, {
     9.539775194370128, 3.322553894570559}, {2.05709781872363, 
     4.517602099408604}, {6.115301101683254, 0.7688764364978584}, {
     1.8227697880068061`, 1.4297399732652472`}, {2.6357568784535212`, 
     7.469300479622262}, {3.67040750900103, 8.792053845024173}, {
     5.53924857572206, 7.381380779532338}, {6.325599975593876, 
     1.6206795363157322`}, {9.237574827844014, 1.3992483206329887`}, {
     9.685197870412805, 8.369603428881742}, {7.938571877275116, 
     4.525885189208254}, {9.247555108671792, 0.0861976965100153}, {
     2.545916064415288, 9.792430163427566}, {4.915834308465666, 
     2.1122767411755894`}}]}}]], "Output",
 ImageSize->{184, 185},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1262940337]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->210630102],

Cell["Find a shortest tour visiting 100 random points in 3D:", "ExampleText",
 CellID->121435194],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"With", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"p", "=", 
     RowBox[{"RandomReal", "[", 
      RowBox[{"20", ",", 
       RowBox[{"{", 
        RowBox[{"100", ",", "3"}], "}"}]}], "]"}]}], "}"}], ",", 
   RowBox[{"Graphics3D", "[", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"Line", "[", 
       RowBox[{"p", "[", 
        RowBox[{"[", 
         RowBox[{"Last", "[", 
          RowBox[{"FindShortestTour", "[", "p", "]"}], "]"}], "]"}], "]"}], 
       "]"}], ",", 
      RowBox[{"Sphere", "/@", "p"}]}], "}"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->117433653],

Cell[BoxData[
 Graphics3DBox[{Line3DBox[CompressedData["
1:eJwNlnk81Ikbx10JM+6MwWCMY2bM/Z1OYj62dXQhS9KiTFKOSuGnVkSHamtj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    "]], {
    SphereBox[{17.335847779044645`, 14.084512108705628`, 
     13.134956646875695`}], 
    SphereBox[{14.538669238693513`, 8.18469165042015, 12.906774629700912`}], 
    SphereBox[{17.142803777870064`, 16.270883688164194`, 
     18.160778752073163`}], 
    SphereBox[{17.720800584274183`, 15.215751971691788`, 
     17.144749091372894`}], 
    SphereBox[{18.990777586626855`, 18.01326501270571, 12.82442771706933}], 
    SphereBox[{4.399145169042291, 12.47081363903279, 8.704988643541753}], 
    SphereBox[{0.5741746839814166, 9.19065810055287, 16.78644498917421}], 
    SphereBox[{16.529013191224017`, 15.9014920958519, 1.3874273302524687`}], 
    SphereBox[{6.20704361209909, 11.003415561912018`, 18.92128882780451}], 
    SphereBox[{13.559425000792974`, 14.536554993458818`, 14.86406346496488}], 
    SphereBox[{1.4540805773309007`, 17.586483313553714`, 0.5862085908007497}],
     SphereBox[{5.7496116671347375`, 9.231013385055338, 2.221857693395264}], 
    SphereBox[{4.710636093656748, 6.063885455822247, 6.061159089816246}], 
    SphereBox[{14.626281626217033`, 1.213493514401498, 14.344081576866433`}], 
    SphereBox[{18.99507879118213, 7.033538092252751, 12.530951683734578`}], 
    SphereBox[{17.640425907877205`, 12.392589172848204`, 7.379719871556527}], 
    SphereBox[{8.199067357026015, 4.34510775852694, 11.521733290866386`}], 
    SphereBox[{3.8466032654095406`, 13.378112882270226`, 6.231723720778115}], 
    SphereBox[{17.20485688418072, 19.09362464368337, 3.5520520895093544`}], 
    SphereBox[{14.999039927245716`, 8.794610667929064, 3.931596624951448}], 
    SphereBox[{2.6704551395727183`, 18.663158900405833`, 9.541805381609802}], 
    SphereBox[{2.5681727418140587`, 16.944315414460572`, 7.204124623116619}], 
    SphereBox[{4.528447417136685, 7.284124586710652, 5.15652343765415}], 
    SphereBox[{1.4868544086432012`, 16.527696541940166`, 
     19.758837511300253`}], 
    SphereBox[{9.375395476118724, 5.833949381800649, 17.26705573475105}], 
    SphereBox[{1.8302675512479238`, 8.205230320456792, 16.797900823821404`}], 
    SphereBox[{15.340758703452062`, 16.25938327816133, 17.035853391236863`}], 
    SphereBox[{14.923737180746887`, 7.261662410624163, 10.3990583422733}], 
    SphereBox[{18.066927716102263`, 8.483034093610256, 8.786471407565788}], 
    SphereBox[{16.331824893303928`, 15.699400118944723`, 13.98048817852155}], 
    SphereBox[{3.671447172586211, 9.274581477198765, 7.427404112649594}], 
    SphereBox[{16.766338817717667`, 19.939588629333635`, 2.620099284620223}], 
    SphereBox[{8.91551388798634, 6.96874840788646, 1.4109845663645082`}], 
    SphereBox[{5.8007594114379835`, 12.173962153118833`, 
     3.0042935787357727`}], 
    SphereBox[{17.32599696974844, 11.005534677100428`, 3.683160279558626}], 
    SphereBox[{0.09282203448075954, 11.510886248996034`, 19.6415488546172}], 
    SphereBox[{14.512386995099892`, 8.29887559988677, 5.79721916265453}], 
    SphereBox[{16.053876771309504`, 8.054010109240854, 10.856590928757432`}], 
    SphereBox[{11.710876824340932`, 8.832926520548114, 6.542872802450659}], 
    SphereBox[{1.5871177451301532`, 11.809753570315786`, 
     15.938700473457924`}], 
    SphereBox[{9.05543621678186, 15.673631884180189`, 18.302142058977342`}], 
    SphereBox[{2.199941842813846, 10.940061123843389`, 14.989507119962347`}], 
    SphereBox[{18.643553297701693`, 4.710898382954931, 1.7125111613118138`}], 
    SphereBox[{14.870099259992164`, 2.9763252051132483`, 5.677166946890906}], 
    SphereBox[{9.660209582034941, 6.323805472554347, 8.933014178414446}], 
    SphereBox[{12.331346313060653`, 11.4148112293715, 1.3471951554450179`}], 
    SphereBox[{13.37519521429353, 14.047976527357754`, 14.669109156387012`}], 
    SphereBox[{16.0207575798788, 16.095512002383522`, 3.7821235758069305`}], 
    SphereBox[{18.767897152108645`, 12.07942377593961, 17.487942687102596`}], 
    SphereBox[{19.733887468056892`, 18.086606439764054`, 7.428704810901352}], 
    SphereBox[{14.316628167382692`, 9.797172177783624, 13.652758498994778`}], 
    SphereBox[{12.676084700573856`, 19.031424595287206`, 
     11.569916652495529`}], 
    SphereBox[{6.123986433499549, 19.304467819016544`, 15.510155669686313`}], 
    SphereBox[{12.980832139336442`, 0.023023009038718456`, 
     19.214326632400752`}], 
    SphereBox[{7.380534709592115, 1.7719596130392912`, 7.171592113625893}], 
    SphereBox[{19.641965788811504`, 12.65876783303613, 5.540015613807228}], 
    SphereBox[{18.617439795958294`, 18.40599671710489, 13.780618460366178`}], 
    SphereBox[{12.024253385348285`, 5.9747265950690664`, 5.048827859408789}], 
    SphereBox[{18.07674566287013, 2.6688349068183426`, 3.70133732408811}], 
    SphereBox[{18.63357628621088, 9.22865423739439, 9.751722831772035}], 
    SphereBox[{12.516228200034348`, 3.0745796334534914`, 
     13.061753155749614`}], 
    SphereBox[{3.647503166687973, 6.146473930072736, 5.129098363574882}], 
    SphereBox[{12.116663239913969`, 16.309833613936064`, 2.784229825166107}], 
    SphereBox[{13.904463519317481`, 11.912205453945695`, 
     14.746567303524639`}], 
    SphereBox[{16.92143801699295, 17.546241850010375`, 8.311959664163307}], 
    SphereBox[{15.910942445733456`, 18.0393506383301, 5.9586919443579855`}], 
    SphereBox[{3.208263492017185, 4.590000540567067, 18.333652016438975`}], 
    SphereBox[{14.380064761599751`, 12.029507771025042`, 10.04059118309144}], 
    SphereBox[{7.797914581359771, 14.132406612777135`, 17.213695924706958`}], 
    SphereBox[{1.167331654207775, 11.179752012194925`, 7.605882434229594}], 
    SphereBox[{18.99129457245985, 13.45236694349532, 2.9774010128237727`}], 
    SphereBox[{6.02333513488404, 12.249616081995613`, 10.244189999546096`}], 
    SphereBox[{16.052125908377427`, 12.604337279071247`, 5.687405378002133}], 
    SphereBox[{13.909386062084547`, 0.02907300665546675, 
     15.931700885725482`}], 
    SphereBox[{8.069105699348839, 12.735767306924636`, 11.74396837629413}], 
    SphereBox[{5.517519721652633, 7.8774750972371255`, 0.9955112859379334}], 
    SphereBox[{9.727892408017933, 7.530621715383248, 12.347013482777989`}], 
    SphereBox[{6.450577102390014, 0.1696484376017171, 17.66594782053206}], 
    SphereBox[{15.224909034227627`, 17.60377742705807, 9.397542698493266}], 
    SphereBox[{7.181311209727164, 11.746951468310339`, 15.29309879811902}], 
    SphereBox[{1.6249765066570543`, 7.900793412837568, 17.24098824491678}], 
    SphereBox[{7.743305784439687, 11.843422211640222`, 14.352858808456773`}], 
    SphereBox[{14.366908489490896`, 1.232801186638075, 12.886357368808445`}], 
    SphereBox[{13.840849996166336`, 1.1373442221674157`, 8.923544562762302}], 
    SphereBox[{18.236095660883898`, 15.133262982220312`, 
     16.713832557582545`}], 
    SphereBox[{7.948464827771291, 4.977595033623126, 6.523651951578167}], 
    SphereBox[{6.428444874693624, 7.536064911461726, 10.181769079897581`}], 
    SphereBox[{12.856841926816758`, 13.684107920086564`, 9.474327892611129}], 
    SphereBox[{6.082050129611738, 4.854768250652017, 2.3668666598097765`}], 
    SphereBox[{0.6868674723708423, 9.225088622826462, 15.632759904740897`}], 
    SphereBox[{5.497040035547989, 1.442938617366325, 12.706809264748582`}], 
    SphereBox[{13.586480479576366`, 7.901168442809561, 13.76375407203755}], 
    SphereBox[{9.607724971723446, 11.422405786953437`, 3.7939205370217444`}], 
    SphereBox[{18.63376131297627, 4.01191355700381, 11.54724965935765}], 
    SphereBox[{4.904213313109422, 4.975653484657263, 3.174133019582608}], 
    SphereBox[{8.098676325137193, 3.2148671716157784`, 18.015947658176074`}], 
    SphereBox[{5.283240220177814, 0.9062511688766195, 15.89369790068175}], 
    SphereBox[{18.228485106794512`, 15.781335846001753`, 15.77979754850686}], 
    SphereBox[{15.719515819120353`, 14.301112922040913`, 8.313501825207265}], 
    SphereBox[{16.031787183116588`, 18.824176456857558`, 
     9.224125945685389}]}}]], "Output",
 ImageSize->{184, 194},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->159269566,
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->45718568],

Cell["\<\
Find the shortest tour through 2D points whose coordinates are relatively \
prime:\
\>", "ExampleText",
 CellID->130508201],

Cell[BoxData[
 RowBox[{
  RowBox[{"pts", "=", 
   RowBox[{"Select", "[", 
    RowBox[{
     RowBox[{"Tuples", "[", 
      RowBox[{
       RowBox[{"Range", "[", "10", "]"}], ",", "2"}], "]"}], ",", 
     RowBox[{
      RowBox[{"CoprimeQ", "@@", "#"}], "&"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->269274],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindShortestTour", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->167449473],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"75.76297991208098`", ",", 
   RowBox[{"{", 
    RowBox[{
    "1", ",", "11", ",", "17", ",", "16", ",", "23", ",", "28", ",", "29", 
     ",", "36", ",", "39", ",", "40", ",", "48", ",", "53", ",", "60", ",", 
     "54", ",", "61", ",", "55", ",", "56", ",", "50", ",", "49", ",", "41", 
     ",", "42", ",", "43", ",", "37", ",", "31", ",", "30", ",", "24", ",", 
     "18", ",", "25", ",", "19", ",", "13", ",", "14", ",", "21", ",", "20", 
     ",", "26", ",", "32", ",", "33", ",", "34", ",", "38", ",", "44", ",", 
     "51", ",", "57", ",", "62", ",", "58", ",", "63", ",", "59", ",", "52", 
     ",", "45", ",", "46", ",", "47", ",", "35", ",", "27", ",", "22", ",", 
     "15", ",", "10", ",", "9", ",", "8", ",", "7", ",", "6", ",", "5", ",", 
     "4", ",", "3", ",", "12", ",", "2"}], "}"}]}], "}"}]], "Output",
 ImageSize->{564, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->504581693]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics", "[", 
  RowBox[{"Line", "[", 
   RowBox[{"pts", "[", 
    RowBox[{"[", 
     RowBox[{"Last", "[", "%", "]"}], "]"}], "]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->507375310],

Cell[BoxData[
 GraphicsBox[
  LineBox[{{1, 1}, {2, 1}, {3, 2}, {3, 1}, {4, 1}, {5, 1}, {5, 2}, {6, 1}, {7,
    1}, {7, 2}, {8, 1}, {9, 1}, {10, 1}, {9, 2}, {10, 3}, {9, 4}, {9, 5}, {8, 
   5}, {8, 3}, {7, 3}, {7, 4}, {7, 5}, {6, 5}, {5, 4}, {5, 3}, {4, 3}, {3, 
   4}, {4, 5}, {3, 5}, {2, 5}, {2, 7}, {3, 8}, {3, 7}, {4, 7}, {5, 6}, {5, 
   7}, {5, 8}, {6, 7}, {7, 6}, {8, 7}, {9, 7}, {10, 7}, {9, 8}, {10, 9}, {9, 
   10}, {8, 9}, {7, 8}, {7, 9}, {7, 10}, {5, 9}, {4, 9}, {3, 10}, {2, 9}, {1, 
   10}, {1, 9}, {1, 8}, {1, 7}, {1, 6}, {1, 5}, {1, 4}, {1, 3}, {2, 3}, {1, 
   2}}]]], "Output",
 ImageSize->{184, 180},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->346948077]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->27805920],

Cell["\<\
Find a shortest tour for 3D points with relatively prime coordinates:\
\>", "ExampleText",
 CellID->284343987],

Cell[BoxData[
 RowBox[{
  RowBox[{"pts", "=", 
   RowBox[{"Select", "[", 
    RowBox[{
     RowBox[{"Tuples", "[", 
      RowBox[{
       RowBox[{"Range", "[", "10", "]"}], ",", "3"}], "]"}], ",", 
     RowBox[{
      RowBox[{"CoprimeQ", "@@", "#"}], "&"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2880373],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics3D", "[", 
  RowBox[{"Line", "[", 
   RowBox[{"pts", "[", 
    RowBox[{"[", 
     RowBox[{"Last", "[", 
      RowBox[{"FindShortestTour", "[", "%", "]"}], "]"}], "]"}], "]"}], "]"}],
   "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->413704880],

Cell[BoxData[
 Graphics3DBox[
  Line3DBox[{{1, 1, 1}, {2, 1, 1}, {1, 2, 1}, {1, 3, 2}, {1, 4, 3}, {1, 5, 
   4}, {1, 5, 3}, {2, 5, 3}, {1, 5, 2}, {1, 5, 1}, {1, 6, 1}, {1, 7, 1}, {1, 
   8, 1}, {1, 7, 2}, {2, 7, 1}, {2, 5, 1}, {1, 4, 1}, {1, 3, 1}, {2, 3, 1}, {
   3, 2, 1}, {3, 1, 1}, {3, 1, 2}, {2, 1, 3}, {1, 2, 3}, {1, 3, 4}, {1, 3, 
   5}, {1, 4, 5}, {1, 5, 6}, {1, 4, 7}, {1, 5, 8}, {1, 4, 9}, {1, 5, 9}, {2, 
   5, 9}, {3, 7, 10}, {4, 7, 9}, {5, 7, 9}, {5, 7, 8}, {5, 6, 7}, {6, 5, 7}, {
   7, 5, 8}, {8, 5, 7}, {9, 5, 8}, {9, 5, 7}, {9, 4, 7}, {10, 3, 7}, {9, 2, 
   7}, {9, 1, 7}, {10, 1, 7}, {9, 1, 8}, {10, 1, 9}, {9, 1, 10}, {8, 1, 9}, {
   7, 1, 10}, {7, 1, 9}, {7, 1, 8}, {8, 1, 7}, {7, 1, 6}, {6, 1, 5}, {7, 1, 
   4}, {7, 1, 3}, {7, 1, 2}, {7, 2, 1}, {7, 1, 1}, {8, 1, 1}, {9, 1, 1}, {10, 
   1, 1}, {9, 2, 1}, {10, 3, 1}, {9, 4, 1}, {9, 5, 1}, {9, 5, 2}, {8, 5, 3}, {
   7, 5, 3}, {7, 5, 4}, {7, 6, 5}, {8, 7, 5}, {9, 8, 5}, {9, 7, 5}, {9, 7, 
   4}, {10, 7, 3}, {9, 7, 2}, {9, 7, 1}, {10, 7, 1}, {9, 8, 1}, {10, 9, 1}, {
   9, 10, 1}, {8, 9, 1}, {7, 9, 1}, {7, 10, 1}, {7, 9, 2}, {7, 10, 3}, {7, 9, 
   4}, {8, 9, 5}, {7, 9, 5}, {7, 8, 5}, {6, 7, 5}, {5, 7, 6}, {5, 8, 7}, {5, 
   9, 7}, {5, 9, 8}, {5, 8, 9}, {7, 8, 9}, {7, 9, 10}, {7, 10, 9}, {7, 9, 
   8}, {8, 9, 7}, {9, 10, 7}, {10, 9, 7}, {9, 8, 7}, {9, 7, 8}, {10, 7, 9}, {
   9, 7, 10}, {8, 7, 9}, {8, 5, 9}, {7, 5, 9}, {7, 4, 9}, {7, 3, 10}, {7, 2, 
   9}, {7, 3, 8}, {8, 3, 7}, {7, 5, 6}, {7, 4, 5}, {7, 3, 5}, {8, 3, 5}, {9, 
   4, 5}, {9, 5, 4}, {8, 7, 3}, {7, 8, 3}, {7, 8, 1}, {8, 7, 1}, {7, 6, 1}, {
   7, 5, 2}, {7, 4, 3}, {7, 3, 4}, {7, 2, 5}, {7, 1, 5}, {8, 1, 5}, {9, 2, 
   5}, {9, 1, 5}, {9, 1, 4}, {10, 1, 3}, {9, 1, 2}, {8, 1, 3}, {7, 2, 3}, {7, 
   3, 2}, {7, 3, 1}, {8, 3, 1}, {7, 4, 1}, {8, 5, 1}, {7, 5, 1}, {6, 5, 1}, {
   5, 6, 1}, {6, 7, 1}, {5, 7, 1}, {5, 8, 1}, {5, 9, 1}, {5, 9, 2}, {5, 9, 
   4}, {4, 9, 5}, {3, 8, 5}, {3, 7, 5}, {4, 7, 5}, {5, 7, 4}, {5, 7, 3}, {5, 
   8, 3}, {4, 7, 3}, {5, 7, 2}, {4, 7, 1}, {3, 7, 1}, {3, 7, 2}, {3, 8, 1}, {
   4, 9, 1}, {3, 10, 1}, {2, 9, 1}, {1, 9, 1}, {1, 10, 1}, {1, 9, 2}, {1, 10, 
   3}, {1, 9, 4}, {2, 9, 5}, {1, 9, 5}, {1, 8, 5}, {1, 7, 5}, {2, 7, 5}, {3, 
   7, 4}, {2, 7, 3}, {1, 7, 3}, {1, 8, 3}, {1, 7, 4}, {1, 6, 5}, {1, 7, 6}, {
   1, 8, 7}, {1, 9, 7}, {1, 10, 7}, {2, 9, 7}, {3, 10, 7}, {4, 9, 7}, {3, 8, 
   7}, {3, 7, 8}, {2, 7, 9}, {1, 7, 9}, {1, 7, 10}, {1, 8, 9}, {1, 9, 10}, {1,
    10, 9}, {1, 9, 8}, {1, 7, 8}, {1, 6, 7}, {1, 5, 7}, {2, 5, 7}, {3, 5, 
   7}, {4, 5, 7}, {5, 4, 7}, {5, 3, 7}, {5, 3, 8}, {5, 4, 9}, {4, 5, 9}, {3, 
   5, 8}, {3, 4, 7}, {4, 3, 7}, {5, 2, 7}, {5, 1, 7}, {6, 1, 7}, {5, 1, 6}, {
   5, 1, 4}, {5, 1, 3}, {5, 2, 3}, {5, 3, 2}, {5, 4, 3}, {5, 3, 4}, {4, 3, 
   5}, {3, 4, 5}, {3, 5, 4}, {4, 5, 3}, {3, 5, 2}, {3, 5, 1}, {4, 5, 1}, {3, 
   4, 1}, {4, 3, 1}, {5, 4, 1}, {5, 3, 1}, {5, 2, 1}, {6, 1, 1}, {5, 1, 1}, {
   4, 1, 1}, {5, 1, 2}, {4, 1, 3}, {3, 1, 4}, {4, 1, 5}, {3, 2, 5}, {3, 1, 
   5}, {2, 1, 5}, {1, 2, 5}, {2, 3, 5}, {2, 3, 7}, {3, 2, 7}, {3, 1, 7}, {4, 
   1, 7}, {5, 1, 8}, {5, 2, 9}, {5, 1, 9}, {4, 1, 9}, {3, 1, 10}, {2, 1, 9}, {
   3, 1, 8}, {2, 1, 7}, {1, 1, 7}, {1, 1, 8}, {1, 1, 9}, {1, 1, 10}, {1, 3, 
   10}, {1, 2, 9}, {1, 3, 8}, {1, 3, 7}, {1, 2, 7}, {1, 1, 6}, {1, 1, 5}, {1, 
   1, 4}, {1, 1, 3}, {1, 1, 2}}]]], "Output",
 ImageSize->{184, 195},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->286195536,
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->227172317],

Cell["\<\
Find a \"shortest tour\" through the names of countries in Europe:\
\>", "ExampleText",
 CellID->441274894],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"#", "[", 
    RowBox[{"[", 
     RowBox[{"Last", "[", 
      RowBox[{"FindShortestTour", "[", 
       RowBox[{"#", ",", 
        RowBox[{"DistanceFunction", "\[Rule]", "EditDistance"}]}], "]"}], 
      "]"}], "]"}], "]"}], "&"}], "[", 
  RowBox[{"CountryData", "[", "\"\<Europe\>\"", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->145928136],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"Albania\"\>", ",", "\<\"Romania\"\>", ",", "\<\"Slovenia\"\>", 
   ",", "\<\"Slovakia\"\>", ",", "\<\"Croatia\"\>", ",", "\<\"Latvia\"\>", 
   ",", "\<\"Serbia\"\>", ",", "\<\"Spain\"\>", ",", "\<\"Ukraine\"\>", 
   ",", "\<\"France\"\>", ",", "\<\"Greece\"\>", ",", "\<\"Sweden\"\>", 
   ",", "\<\"Ireland\"\>", ",", "\<\"Iceland\"\>", ",", "\<\"Poland\"\>", 
   ",", "\<\"Finland\"\>", ",", "\<\"Svalbard\"\>", ",", "\<\"Italy\"\>", 
   ",", "\<\"Malta\"\>", ",", "\<\"Moldova\"\>", ",", "\<\"Andorra\"\>", 
   ",", "\<\"Cyprus\"\>", ",", "\<\"Belarus\"\>", ",", "\<\"Belgium\"\>", 
   ",", "\<\"Bulgaria\"\>", ",", "\<\"Hungary\"\>", ",", "\<\"Denmark\"\>", 
   ",", "\<\"Germany\"\>", ",", "\<\"Norway\"\>", ",", "\<\"Portugal\"\>", 
   ",", "\<\"Montenegro\"\>", ",", "\<\"Monaco\"\>", ",", "\<\"SanMarino\"\>",
    ",", "\<\"VaticanCity\"\>", ",", "\<\"Lithuania\"\>", 
   ",", "\<\"Liechtenstein\"\>", ",", "\<\"CzechRepublic\"\>", 
   ",", "\<\"BosniaHerzegovina\"\>", ",", "\<\"UnitedKingdom\"\>", 
   ",", "\<\"Switzerland\"\>", ",", "\<\"Netherlands\"\>", 
   ",", "\<\"Luxembourg\"\>", ",", "\<\"Macedonia\"\>", 
   ",", "\<\"Estonia\"\>", ",", "\<\"Austria\"\>"}], "}"}]], "Output",
 ImageSize->{535, 101},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->131074360]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->168813060],

Cell["Find a traveling-salesman tour of Europe:", "ExampleText",
 CellID->933112649],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics", "[", 
  RowBox[{"{", 
   RowBox[{"Gray", ",", 
    RowBox[{
     RowBox[{
      RowBox[{"CountryData", "[", 
       RowBox[{"#", ",", "\"\<Polygon\>\""}], "]"}], "&"}], "/@", 
     RowBox[{"CountryData", "[", "\"\<Europe\>\"", "]"}]}], ",", "Thick", ",",
     "Red", ",", 
    RowBox[{"Line", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"#", "[", 
        RowBox[{"[", 
         RowBox[{"Last", "[", 
          RowBox[{"FindShortestTour", "[", "#", "]"}], "]"}], "]"}], "]"}], 
       "&"}], "[", 
      RowBox[{
       RowBox[{
        RowBox[{"Reverse", "[", 
         RowBox[{"CountryData", "[", 
          RowBox[{"#", ",", "\"\<CenterCoordinates\>\""}], "]"}], "]"}], 
        "&"}], "/@", 
       RowBox[{"CountryData", "[", "\"\<Europe\>\"", "]"}]}], "]"}], "]"}]}], 
   "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->934699026],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{241, 174},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->142644075]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Neat Examples",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->589267740],

Cell["Plan a tour through every country of the world:", "ExampleText",
 CellID->191358578],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"SC", "[", 
    RowBox[{"{", 
     RowBox[{"lat_", ",", "lon_"}], "}"}], "]"}], ":=", 
   RowBox[{"r", " ", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"Cos", "[", 
        RowBox[{"lon", " ", "\[Degree]"}], "]"}], " ", 
       RowBox[{"Cos", "[", 
        RowBox[{"lat", " ", "\[Degree]"}], "]"}]}], ",", 
      RowBox[{
       RowBox[{"Sin", "[", 
        RowBox[{"lon", "  ", "\[Degree]"}], "]"}], " ", 
       RowBox[{"Cos", "[", 
        RowBox[{"lat", "  ", "\[Degree]"}], "]"}]}], ",", 
      RowBox[{"Sin", "[", 
       RowBox[{"lat", " ", "\[Degree]"}], "]"}]}], "}"}]}]}], ";"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->131383970],

Cell[BoxData[{
 RowBox[{
  RowBox[{"r", "=", "6378.7"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"places", "=", 
   RowBox[{"CountryData", "[", "\"\<Countries\>\"", "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"centers", "=", 
   RowBox[{"Map", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"CountryData", "[", 
       RowBox[{"#", ",", "\"\<CenterCoordinates\>\""}], "]"}], "&"}], ",", 
     "places"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"distfun", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"lat1_", ",", "lon1_"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"lat2_", ",", "lon2_"}], "}"}]}], "]"}], ":=", 
   RowBox[{
    RowBox[{"VectorAngle", "[", 
     RowBox[{
      RowBox[{"SC", "[", 
       RowBox[{"{", 
        RowBox[{"lat1", ",", "lon1"}], "}"}], "]"}], ",", 
      RowBox[{"SC", "[", 
       RowBox[{"{", 
        RowBox[{"lat2", ",", "lon2"}], "}"}], "]"}]}], "]"}], "r"}]}], 
  ";"}]}], "Input",
 CellLabel->"In[2]:=",
 CellID->358139632],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"dist", ",", "route"}], "}"}], "=", 
   RowBox[{"FindShortestTour", "[", 
    RowBox[{"centers", ",", 
     RowBox[{"DistanceFunction", "\[Rule]", "distfun"}]}], "]"}]}], 
  ";"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->208856423],

Cell["Visualize the tour: ", "ExampleText",
 CellID->88976648],

Cell[BoxData[
 RowBox[{
  RowBox[{"surfaceCenters", "=", 
   RowBox[{"Map", "[", 
    RowBox[{"SC", ",", 
     RowBox[{"centers", "[", 
      RowBox[{"[", "route", "]"}], "]"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->121823911],

Cell[BoxData[
 RowBox[{
  RowBox[{"GreatCircleArc", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"lat1_", ",", "lon1_"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"lat2_", ",", "lon2_"}], "}"}]}], "]"}], ":=", 
  "\[IndentingNewLine]", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"u", "=", 
       RowBox[{"SC", "[", 
        RowBox[{"{", 
         RowBox[{"lat1", ",", "lon1"}], "}"}], "]"}]}], ",", 
      RowBox[{"v", "=", 
       RowBox[{"SC", "[", 
        RowBox[{"{", 
         RowBox[{"lat2", ",", "lon2"}], "}"}], "]"}]}], ",", "a"}], "}"}], 
    ",", 
    RowBox[{
     RowBox[{"a", "=", 
      RowBox[{"VectorAngle", "[", 
       RowBox[{"u", ",", "v"}], "]"}]}], ";", " ", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"Evaluate", "[", 
        RowBox[{
         RowBox[{"RotationTransform", "[", 
          RowBox[{"\[Theta]", ",", 
           RowBox[{"{", 
            RowBox[{"u", ",", "v"}], "}"}]}], "]"}], "[", "u", "]"}], "]"}], 
       ",", 
       RowBox[{"{", 
        RowBox[{"\[Theta]", ",", "0", ",", "a", ",", 
         RowBox[{"a", "/", 
          RowBox[{"Ceiling", "[", 
           RowBox[{"10", "a"}], "]"}]}]}], "}"}]}], "]"}]}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[5]:=",
 CellID->258852470],

Cell[BoxData[
 RowBox[{
  RowBox[{"tourLine", "=", 
   RowBox[{"Apply", "[", 
    RowBox[{"GreatCircleArc", ",", 
     RowBox[{"Partition", "[", 
      RowBox[{
       RowBox[{"centers", "[", 
        RowBox[{"[", "route", "]"}], "]"}], ",", "2", ",", "1"}], "]"}], ",", 
     
     RowBox[{"{", "1", "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->131431459],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics3D", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Sphere", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"0", ",", "0", ",", "0"}], "}"}], ",", 
       RowBox[{"0.99", " ", "r"}]}], "]"}], ",", 
     RowBox[{"Map", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"Line", "[", 
         RowBox[{"Map", "[", 
          RowBox[{"SC", ",", 
           RowBox[{"CountryData", "[", 
            RowBox[{"#", ",", "\"\<SchematicCoordinates\>\""}], "]"}], ",", 
           RowBox[{"{", 
            RowBox[{"-", "2"}], "}"}]}], "]"}], "]"}], "&"}], ",", "places"}],
       "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"Red", ",", "Thick", ",", 
       RowBox[{"Line", "[", "tourLine", "]"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"Yellow", ",", 
       RowBox[{"PointSize", "[", "Medium", "]"}], ",", 
       RowBox[{"Map", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"Tooltip", "[", 
           RowBox[{
            RowBox[{"Point", "[", 
             RowBox[{"SC", "[", 
              RowBox[{"CountryData", "[", 
               RowBox[{"#", ",", "\"\<CenterCoordinates\>\""}], "]"}], "]"}], 
             "]"}], ",", "#"}], "]"}], "&"}], ",", "places"}], "]"}]}], 
      "}"}]}], "}"}], ",", 
   RowBox[{"Boxed", "\[Rule]", "False"}], ",", 
   RowBox[{"SphericalRegion", "\[Rule]", "True"}]}], "]"}]], "Input",
 CellLabel->"In[7]:=",
 CellID->114665653],

Cell[BoxData[
 Graphics3DBox[{
   SphereBox[{0, 0, 0}, 6314.913], {
    Line3DBox[{{{2492.014815342531, 4548.616524639334, 3713.187789742302}, {
     2256.8057622804436`, 4640.834789498933, 3749.3057887979985`}, {
     2073.929206586604, 4611.113916549065, 3888.9921296750654`}, {
     1922.05496477713, 4702.0304975213585`, 3858.034162984386}, {
     1630.1157399536394`, 4720.812878034248, 3967.9291998398107`}, {
     1607.608115373053, 4856.21888603035, 3810.5836781717367`}, {
     1321.5668674136705`, 4903.615474039338, 3859.5116255449184`}, {
     1355.073455710602, 4908.407628312745, 3841.760556561247}, {
     1616.3743331573578`, 4868.455242148314, 3791.2123734115335`}, {
     1816.9783487239497`, 4965.126694664225, 3568.461892216461}, {
     1910.5450598582647`, 5065.020542200001, 3373.8995496818197`}, {
     2156.302760820424, 5010.89076698226, 3305.9258635139054`}, {
     2228.174716680869, 5063.912248110751, 3174.8769842905795`}, {
     2692.986895438178, 4831.72545373678, 3176.486173543079}, {
     2579.067740174896, 4820.036342332001, 3286.863693582058}, {
     2625.5208887613926`, 4656.381370230966, 3480.3112343070084`}, {
     2492.014815342531, 4548.616524639334, 3713.187789742302}}}], 
    Line3DBox[{{{4612.419227341069, 1680.302906599994, 4073.0804931066637`}, {
     4482.598121690123, 1575.6413065658874`, 4255.75871525437}, {
     4413.378681277919, 1620.8926261542215`, 4310.755084683012}, {
     4505.03327916477, 1727.8168607464763`, 4172.186206215886}, {
     4612.419227341069, 1680.302906599994, 4073.0804931066637`}}}], 
    Line3DBox[{{{5041.169464994508, 763.9052239273175, 3832.8674545389035`}, {
       5171.361142168633, 0, 3734.278729188319}, {
       5215.892361893048, -201.89387062315882`, 3666.267778565903}, {
       5402.384204307434, -111.59164721028046`, 3389.6321191719953`}, {
       5415.0323870509765`, -361.24820301712793`, 3351.8260206946743`}, {
       5458.52984935199, -346.6099201752429, 3282.091884297186}, {
       5530.285249604942, -842.9601269619327, 3064.8290282419234`}, {
       5584.878127433277, -851.2815095214812, 2961.8017798739006`}, {
       5604.342449484579, -854.2483813277349, 2923.9389702769327`}, {
       5760.790470840381, -483.747091627257, 2695.7551061653994`}, {
       5921.153404138738, 0, 2372.2892013108867`}, {6009.152264008688, 
       446.5571511481072, 2092.484042643517}, {5721.098236755274, 
       1216.0569701811273`, 2545.2021728478826`}, {5648.9123279981695`, 
       935.1713716369939, 2811.237753421236}, {5434.9707772208685`, 
       912.7529903003774, 3211.8200945029457`}, {5250.130856427837, 
       691.1928399093782, 3556.1484978052185`}, {5041.169464994508, 
       763.9052239273175, 3832.8674545389035`}}}], 
    Line3DBox[{{{-6101.6400274719135`, -991.6382448777152, \
-1573.0404497178013`}, {-6101.403728342352, -995.2071543666208, \
-1571.7025652672096`}, {-6101.112752193295, -999.095528702843, \
-1570.3646072012534`}, {-6100.606283024439, -1002.1834512564977`, \
-1570.3646072012534`}, {-6099.836066240707, -1005.8840423697233`, \
-1570.990435456255}, {-6099.093068125243, -1010.2451510416646`, \
-1571.076755331646}, {-6098.746427026884, -1014.125233672289, \
-1569.9222016571189`}, {-6098.4241441301065`, -1016.4781706650963`, \
-1569.6524382334817`}, {-6098.111509034514, -1017.5199828813991`, \
-1570.1919620918459`}, {-6097.746754386939, -1017.2403441914742`, \
-1571.7888826152089`}, {-6097.80198169437, -1015.3900440694097`, \
-1572.770720877314}, {-6097.978069762137, -1011.7007827265398`, \
-1574.4645683312613`}, {-6098.315676325866, -1008.1480816284336`, \
-1575.4355103866073`}, {-6098.675027867543, -1004.1619344563773`, \
-1576.589802026403}, {-6098.307366622254, -1004.4293641196724`, \
-1577.841121499531}, {-6098.148536091306, -1003.5286630889904`, \
-1579.0276578091623`}, {-6098.375224376804, -1000.9424890736291`, \
-1579.7934821596932`}, {-6098.520128835777, -998.2338011692036, \
-1580.9475659831824`}, {-6098.433367940284, -997.3453038007656, \
-1581.8427519128065`}, {-6098.372278007271, -996.0239571474982, \
-1582.9104603927058`}, {-6098.61343344785, -993.5500632181565, \
-1583.5359646211418`}, {-6098.9671228308125`, -991.5316231785498, \
-1583.4389046836475`}, {-6099.489211589188, -989.8682631529148, \
-1582.4682838242718`}, {-6100.121432335113, -987.6763047367211, \
-1581.400555781936}, {-6100.80427496285, -985.1645504127404, \
-1580.3327805260114`}, {-6101.2678454119, -982.2895048496385, \
-1580.3327805260114`}, {-6101.730060976226, -979.4142411533478, \
-1580.3327805260114`}, {-6102.144191062635, -977.8420858719992, \
-1579.7071933045156`}, {-6102.6057548620365`, -977.2605867840571, \
-1578.2833827431741`}, {-6102.884310090447, -977.5236892196984, \
-1577.0428734394488`}, {-6102.887866410718, -978.9445373413226, \
-1576.147509856367}, {-6102.780956611095, -980.8940639949648, \
-1575.349205999869}, {-6102.162529314177, -985.1653240222437, \
-1575.0795028118873`}, {-6101.705346291611, -988.6977792327989, \
-1574.6371830921862`}, {-6101.694421306872, -989.8982447212438, \
-1573.9251392904305`}, {-6101.6400274719135`, -991.6382448777152, \
-1573.0404497178013`}}}], 
    Line3DBox[{{{4700.677486206689, 143.61868657907868`, 
     4309.3872573180115`}, {4693.733513008255, 122.9097376382984, 
     4317.588747814788}, {4693.769067418801, 121.54438072393816`, 
     4317.588747814788}, {4700.677486206689, 143.61868657907868`, 
     4309.3872573180115`}}}], 
    Line3DBox[{{{6178.062912112082, 
      1447.1557128478455`, -651.9913241424521}, {6198.169837048939, 
      1345.754590657383, -677.8266318324306}, {6063.666702243502, 
      1489.3793927497236`, -1304.418965883228}, {5963.77411688758, 
      1242.276755635775, -1891.5497442966744`}, {5911.9942064249135`, 
      1457.596675668733, -1900.4078836601648`}, {5575.507744748854, 
      2422.3715286987513`, -1932.2637643953342`}, {5680.354450692, 
      2295.0121701190237`, -1776.036639123518}, {5762.646381530199, 
      2328.2602683522687`, -1434.8952903432116`}, {5677.145228867054, 
      2529.6069350993857`, -1434.8952903432116`}, {5723.482649705074, 
      2546.2639906745176`, -1202.5388718537388`}, {5801.326312975912, 
      2363.5411911058472`, -1202.5388718537388`}, {5875.429034479812, 
      2348.021674668257, -808.6665357733685}, {6005.982678123847, 
      1951.4620673560976`, -898.7666881602586}, {6051.749389733513, 
      1848.2807083662462`, -804.9853644267173}, {6083.283897964468, 
      1803.8757708519038`, -653.8370669361536}, {6178.2473089713985`, 
      1447.1989061549043`, -650.1455261798856}}, {{6214.639544848828, 
      1324.7417629169984`, -557.7887406399523}, {6202.908824883739, 
      1341.1163584640892`, -642.7617857647199}, {6192.816632456668, 
      1439.2147328120893`, -515.263815559252}, {6214.639544848828, 
      1324.7417629169984`, -557.7887406399523}}}], 
    Line3DBox[{{{2749.6036053947914`, -5398.054474133169, 
     1997.123330565952}, {2751.093270501841, -5397.1819590364275`, 
     1997.4299507244789`}, {2752.385549293739, -5396.597404940503, 
     1997.2290622330802`}, {2754.318984547807, -5395.919991484066, 
     1996.3937654836861`}, {2754.9312520180256`, -5396.0728289004255`, 
     1995.1354620318282`}, {2754.295565561034, -5396.89756952267, 
     1993.7818967006176`}, {2752.82332883477, -5397.804901748827, 
     1993.3588871247923`}, {2749.783229906493, -5399.432322506907, 
     1993.1473786934794`}, {2744.752215556113, -5403.375394122734, 
     1989.392700144642}, {2744.002201385842, -5404.678630647729, 
     1986.885630379726}, {2742.2360257980886`, -5406.037547415421, 
     1985.6266780917695`}, {2736.08242592357, -5409.154586791029, 
     1985.6266780917695`}, {2734.293953044308, -5409.790887899658, 
     1986.3566692674221`}, {2733.9505276283717`, -5409.463190100402, 
     1987.7213580955754`}, {2735.657169881372, -5407.985782478339, 
     1989.392700144642}, {2749.6036053947914`, -5398.054474133169, 
     1997.123330565952}}}], 
    Line3DBox[{{{2895.447403467711, -5367.236620598814, 1870.018471099983}, {
      2896.6779000371657`, -5366.984179579, 1868.8369777979583`}, {
      2896.8916044791463`, -5367.38013259787, 1867.3679964302448`}, {
      2896.1920757794733`, -5368.168558527909, 1866.1863458466783`}, {
      2894.548047064188, -5369.314174382269, 1865.4411247784142`}, {
      2891.468607918324, -5371.072960623249, 1865.153674924581}, {
      2890.1988291494986`, -5372.218243817837, 1863.8228345992702`}, {
      2890.6254014767455`, -5373.011144159375, 1860.873376422094}, {
      2890.1222763905043`, -5373.716803861162, 1859.616796388851}, {
      2888.785129528928, -5374.513116918581, 1859.3931594696821`}, {
      2887.4342292768024`, -5375.0363594188975`, 1859.978869913721}, {
      2884.84588657454, -5376.0685402797435`, 1861.0118083796185`}, {
      2877.0657475366265`, -5380.081468116696, 1861.4590435405253`}, {
      2875.25501255793, -5380.691981936377, 1862.4919055625683`}, {
      2874.8327020484276`, -5380.353492752508, 1864.1209505444708`}, {
      2876.0894917114742`, -5378.4816176478935`, 1867.5808989948805`}, {
      2875.7046374280662`, -5376.836608912645, 1872.902722869365}, {
      2878.558583928072, -5374.411273734183, 1875.4779736517637`}, {
      2881.0764605493478`, -5372.779586388889, 1876.286661081211}, {
      2884.8650095026337`, -5370.950727034175, 1875.70143005715}, {
      2895.447403467711, -5367.236620598814, 1870.018471099983}}, {{
      2872.1938957299303`, -5360.280084119896, 1925.0229440501018`}, {
      2870.173070455991, -5361.385276198272, 1924.9592608140078`}, {
      2867.6385912588034`, -5362.520309668494, 1925.574856585987}, {
      2865.218694894991, -5363.283835169912, 1927.050083373382}, {
      2864.233825048318, -5362.8139812798045`, 1929.8198070387105`}, {
      2864.38840639943, -5360.559013178146, 1935.846020095365}, {
      2864.931791728971, -5360.0003997116555`, 1936.5885556388882`}, {
      2866.2132088882586`, -5359.472417349366, 1936.1536454697516`}, {
      2869.2916051829225`, -5359.356560406398, 1931.9100995790955`}, {
      2870.675878026041, -5358.6152185488845`, 1931.9100995790955`}, {
      2871.2094718142625`, -5358.038540271379, 1932.7164458167697`}, {
      2870.7933308785578`, -5357.261968867037, 1935.4853495477773`}, {
      2866.848204872519, -5357.12934296533, 1941.6900532677876`}, {
      2867.0170873523025`, -5356.658330490521, 1942.7398799675752`}, {
      2868.1031886598703`, -5355.94609510635, 1943.100413132093}, {
      2870.084205522382, -5354.973283886956, 1942.8565237988923`}, {
      2876.207662401511, -5353.68952479333, 1937.3310622764002`}, {
      2880.5539014498727`, -5353.797815401204, 1930.5625767239976`}, {
      2881.805535600532, -5355.340042512997, 1924.4073305927263`}, {
      2881.1944529119664`, -5356.354985234996, 1922.4966805322497`}, {
      2879.910837051742, -5357.285203532926, 1921.8279081746898`}, {
      2878.1168476459484`, -5358.230172006341, 1921.880986194893}, {
      2872.1938957299303`, -5360.280084119896, 1925.0229440501018`}}}], 
    Line3DBox[{{{
      2836.460677664176, -4574.742747860242, -3422.5769976678403`}, {
      2747.040134777426, -4485.690456531043, -3608.346618066}, {
      2803.0170145119264`, -4261.780324371757, -3829.900491286216}, {
      2622.5373519954987`, -4249.013939841491, -3969.381824216603}, {
      2304.401543105862, -4404.792374828655, -3996.9177315504835`}, {
      2243.9932468820375`, -4274.177293769021, -4169.378426021934}, {
      2028.9535389284983`, -4377.67329611649, -4172.186206215886}, {
      2088.3788929256043`, -4203.941045114292, -4318.954384892966}, {
      1984.685733136027, -4249.715600385734, -4323.049102950882}, {
      1862.7332353260122`, -4106.37262488936, -4511.73386289253}, {
      1689.7210525924936`, -4096.188167012429, -4588.452784420154}, {
      1775.0213910441962`, -3937.342509881722, -4694.086355321323}, {
      1467.7395216071272`, -3826.9191256350987`, -4887.560167666559}, {
      1431.1509225774814`, -3620.8276674243257`, -5052.645616864509}, {
      1218.3451133028361`, -3734.8882702802234`, -5025.341628527785}, {
      1169.9956488031064`, -3971.0390609573114`, -4852.810798715839}, {
      1472.8553548764696`, -4301.858463403055, -4473.536023601378}, {
      1447.8440616699952`, -4491.544047576324, -4291.572361241112}, {
      1819.52900118743, -4949.777638087551, -3588.429912555949}, {
      1813.0596946616815`, -5138.91754400682, -3315.4418573535863`}, {
      2098.971439714925, -5278.951632869642, -2900.8278547828363`}, {
      2280.0012724481317`, -5419.495201581031, -2473.5560732829726`}, {
      2387.1347732829595`, -5420.898346128181, -2367.121117545119}, {
      2547.250887658689, -5296.738827385517, -2478.6860228332102`}, {
      2598.8100681815913`, -5312.652347228116, -2389.5030774220777`}, {
      2714.175321627789, -5243.6392984739705`, -2413.5685034749054`}, {
      3080.241823310734, -4881.887823235372, -2714.239355928943}, {
      2952.755129918785, -4837.389447863771, -2927.236608549868}, {
      3204.2000110548915`, -4676.6972188983655`, -2923.9389702769327`}, {
      3333.2543195487383`, -4690.344474874739, -2752.7982196706116`}, {
      3352.6964728744033`, -4580.878817463749, -2909.087367283329}, {
      2949.777555147586, -4654.088337706507, -3213.4230673802986`}, {
      2836.460677664176, -4574.742747860242, -3422.5769976678403`}}, {{
      1410.96835253236, -3603.45201241123, -5070.71154695291}, {
      1551.8942036376939`, -3348.369335874729, -5202.678239261923}, {
      1340.171842728278, -3425.5805004918466`, -5211.252378901671}, {
      1336.856047915205, -3417.1050783906235`, -5217.664466060418}, {
      1410.96835253236, -3603.45201241123, -5070.71154695291}}}], 
    Line3DBox[{{{3533.323304528547, 3314.1452331311184`, 4149.68450480151}, {
     3392.7214437040525`, 3394.6958234100375`, 4201.582459259027}, {
     3357.1281289326703`, 3519.17724643873, 4127.092914398336}, {
     3333.4516692190705`, 3635.700334060455, 4044.4525884096215`}, {
     3392.3239669346713`, 3608.2491227428004`, 4020.011201674247}, {
     3405.496526342634, 3547.08579047857, 4063.0763583832972`}, {
     3484.763809730408, 3462.5339767734013`, 4068.7950722643905`}, {
     3533.323304528547, 3314.1452331311184`, 4149.68450480151}}}], 
    Line3DBox[{{{2120.5846753980763`, -5854.059745842723, 
     1385.972155770335}, {2120.5728787966264`, -5852.400344108611, 
     1392.9804621708086`}, {2120.6979278306458`, -5851.661279578904, 
     1395.8919220239732`}, {2122.1741448788316`, -5850.952391929542, 
     1396.6197392926176`}, {2124.4556696508907`, -5850.329174692656, 
     1395.7615646764784`}, {2128.191085255508, -5849.952393737533, 
     1391.6441303880388`}, {2139.651883238911, -5847.411154435221, 
     1384.7332954539445`}, {2146.3672418784736`, -5846.152226575824, 
     1379.6468739114116`}, {2146.827473716538, -5846.86956685808, 
     1375.8858078505887`}, {2145.1328214786063`, -5848.089505163658, 
     1373.3419122017333`}, {2142.790404529011, -5849.091113667511, 
     1372.7330826763591`}, {2120.5846753980763`, -5854.059745842723, 
     1385.972155770335}}}], Line3DBox[{CompressedData["
1:eJx11Ps/03scwHGXTEqSy5mamksl6mSkOA/KByvrSHWktJLHQcxyaUkODspx
7Tglp+Yy8ZjNmt0v37lk6H3qWLpQOl2PrB09clKnEqvOMpz+gfPD8x94/fBy
TTgalWRmYmKy8QvzLzZtlQgYOhmc2KMJXM++hB59IjTVvpTC9YePO3QbZZDp
9XXUkUV8lJzqa5a8VQb+ZcLhgjAp+N3IGdTZChDFn9WbzZPCWNh0YaZUCtPq
p52UACG62d4uqA2UQM/QcpqrXAp5H5ziLc+IEJPNWFRsEEGq9+tV84QSSNG+
oUzeEyE3bu/nD7fEcN3NVTXjLIPn1doAwRYx8jrgPSG+IQSHNtO5nP0SCOoI
tjqcKEYRCU43I38VQ/+klu+3TgL0Vz5LdzGkyHXhaHZItRCaeC+We8yKoDIF
x7xvkCGWhad1FU4IS+nkoJreVgh3wffYLVAg5J4W7tgiBFpnm3JW2gJ5KyuO
lI4pUGene+zpIRGI5kYvL1zKhRij54l1cwq0WrMwRyIXgyUzxDie1QxWmtaJ
Y01KRJzQsjNXiiE1nPWpMI4DewNC88g2GKKRv7t9cq8QBoymBfY1zaAZO9B/
5ZwKHTMd6sr9yIOPkSpzi0Mc6Nu8Pn82SYUO66zGVk7xgFKWsKHChgN35/wp
FiMqpLcMXdKf3ALPrDHns484cKtCX5Lj0oaukbM83Eo5QLVy3jZiw4M1hm6p
z30Vwv2wbU/38WbQWbcFJG5qhbHgjig3ogqxd6PBUwI27GttGxHPCsBI0PVf
tlUhVkeJBc/xIijx1P4VsUIou2vXcKIdQ5fwu+IdGpqArJr3ue+UEFJK35PV
bzHU1/kL47miAabu08b9VSJIn3R8zq7HEGUzO267hgVx2mUNE8fEgCf24iLP
YehaAPVh3PtaOL+hu2EkQQIDRVHT/X4YijBo46dxdUCuTqe0MyQwf/cCdaRe
iey63+YcHWBBeddmZpVRAkTvnQ5hT5VovKa0IIJWD4WPK5r3m8vgXg2fkktV
oh2G8vosBgsodbfX6sek8Ns3uYSki0o0PD6Y6PWqFuhmIRmNH2QQuZV0+l28
EoVGe+cnmzLB0Szbfl6xHFz5dxoz/1ag1iNr+/jzmRCwReBdSZUD7afVfUFr
lcjPqfzqk4PngYCZfRSkK2A0ddQlbU6OatRdBemBTJBGniXibimAXqIPIrDk
6HbuksE3yUxQeqblr/FQwKF9ZQ87RuUo1f3wTOXv9VD8tisvY6MSBki1TcTj
UuRHmVHTo5vgtOheov0zJcQ88cWvuilBf/yI87AkfelwN26ufCcGJu/Rtw44
KfrkS7LD2Z8HY5ZjxZVcDMxf1IvvbJEg3b6IiZeLmfDggiZGn4NB46ZVnntK
xShhiprmeLUOZIjEThjE4DHbmhAbK0KNfI+T3/fUg8Qf4/pexIBVxzZGrxOi
kmqP4AcubNAO01mGGQyIPk9r1we2ot3zva6eCW4GfYyb+9CfGNTt2PaXD5mP
Qm2rt0c7cyFpShhigVcBlUM6K866hCq4ycNjTlwgvmO3F9mo4BRd3bi6uAVp
zFz4hggehKQHT4afwyBixYs1nBQuIr0OXSye5UNlD2H2nzMKEGTIGa/xLahj
8iS7cJkE/uUU3XliooCqwLqfN0xwUcl2WlRGvhQOPgpruPCVHGxseWohjYei
LV2qUrNl8H9/+w98ipkV
      "], {{-5119.883308246634, 
      2604.9600281859607`, -2772.8670814683665`}, {-5175.6617919012415`, 
      2606.8605474857877`, -2665.448703654928}, {-5119.883308246634, 
      2604.9600281859607`, -2772.8670814683665`}}, {{-3846.3973482540005`, 
      2617.2814894672806`, -4363.814699805266}, {-3947.7564773333706`, 
      2795.167392562322, -4158.133202920363}, {-4100.138638347465, 
      2527.3570215909726`, -4182.002310107565}, {-3943.675122932287, 
      2465.8771022397, -4365.167846877706}, {-3847.4466516244156`, 
      2617.995488058578, -4362.461183484501}}, {{-4088.882189752076, 
      4714.789340922371, -1318.9456391576787`}, {-4054.0416456323346`, 
      4766.291174718746, -1238.962656126835}, {-4139.265598727175, 
      4689.572454232099, -1249.8816702797478`}, {-4088.882189752076, 
      4714.789340922371, -1318.9456391576787`}}}], 
    Line3DBox[{{{4244.657476835224, 728.0944369239904, 4705.377252173004}, {
     4198.090380226197, 973.0651021319792, 4702.870948310739}, {
     4082.427516154492, 1005.2590835374566`, 4796.983785704314}, {
     4076.71919769098, 1259.356859985795, 4741.53926192587}, {
     4189.709003827124, 1210.6177465161945`, 4654.949712410433}, {
     4262.966522731917, 1040.5130507499046`, 4629.499185364015}, {
     4288.541793534741, 792.2538419192691, 4654.949712410433}, {
     4283.987056701689, 724.5821743024488, 4670.144458225626}, {
     4268.724433263232, 716.892721403304, 4685.282299804405}, {
     4244.657476835224, 728.0944369239904, 4705.377252173004}}}], 
    Line3DBox[{{{3484.763809730408, 3462.5339767734013`, 
      4068.7950722643905`}, {3405.496526342634, 3547.08579047857, 
      4063.0763583832972`}, {3392.3239669346713`, 3608.2491227428004`, 
      4020.011201674247}, {3441.232406320366, 3584.307449963096, 
      3999.8091604262313`}, {3484.763809730408, 3462.5339767734013`, 
      4068.7950722643905`}}, {{3333.4516692190705`, 3635.700334060455, 
      4044.4525884096215`}, {3357.1281289326703`, 3519.17724643873, 
      4127.092914398336}, {3392.7214437040525`, 3394.6958234100375`, 
      4201.582459259027}, {3292.521532194259, 3536.98030753244, 
      4163.758633033967}, {3284.407183284696, 3436.9357928842437`, 
      4252.9937103150105`}, {3219.9510128380516`, 3557.3447840283284`, 
      4202.978378777322}, {3144.049479913156, 3564.1335646163393`, 
      4254.3763927787395`}, {3104.018743975075, 3752.114113245113, 
      4120.014685439633}, {3285.0036928237423`, 3763.459933083811, 
      3966.4762397129225`}, {3280.1832684695105`, 3660.105542013549, 
      4065.936403409967}, {3333.4516692190705`, 3635.700334060455, 
      4044.4525884096215`}}}], Line3DBox[{CompressedData["
1:eJx1lHs41FkYx4fS0gUxubTlGk+G3IUSr1K5VGTdhiQmWlISlWvF2rZMojMi
26A4M3IpbPmtLvRTGtnUlKlIsrLNaNt6SK2IyZ7/+mv/OP98nu/znvO+5/t+
DaMS/KMVGQyGKzmzyKE3C7PmL+SBt0pR9y0zilb6PD624TIGtbuT3B3KPHjc
df3yHBZFS8CFo1WP4f2dHvPYaQRfA/UsxISrDVqL04jezZAhXtmJYIojjRuy
oujofFENB2NgfnzKGG5DEOvwpXXamqJhuUTZ9wKGlmbuzi00gjFZnPsiB4ou
qr9XNKsAQ3hoNXg8QPD7fVPjyZUUvWb+ZPJgDobTN7bI9pH6r7mlc4WOFP3s
uWaXJBvDT5nsjpk7CAr0+vlThJcmWj8vJ3z8wD+q4goEMZGVrRJS5/sxfNUy
HwP7ov3VF2cR/PwgWv8Eubf2doNQsRhD1RodW98UBGoR8sY8O4pOe76zsF+I
IZ7F9TrojSCKQwUzCHdRZGvEX8GwU9Y/NLocgZnD9ndlhBd3Bqsva8bQquAT
wTBEkGSzOtiMcE736rreGxiCVfLcM4wQxCmLzghtKVrjjmdOUwuGUqNXh+Ls
EGRmbKqcbUPmcHWvtpzG8OLUK47ECcG4OX+kj8xz1dIwHVE7hq3CpMW+Lghu
tXlFqllStMG8lOt19zG0X2HnxwOC0I8plUMWFK3YccH6nhiDlefJ470bEZSs
zT/GItzCfEzm+BjDNf+QNxNsBE5HUitoc4oenmq3/foQg7Sxc+PKCAR1yyI+
BRP+sbc3upDUaVS0SKjcjmAiNbdNSP5dmS8truom/3LMktkRiOBAl3Eik/hn
wISV5v0CQzZvUklO+McET7sTyym6YMVu91MvMZRHfJorj0aQfSld/tiEoqmY
4PZ8KQaHfTe6lXIRBPlZWbroU3RcofqTg5MY/JsUQ57mITC4knlaZSlFb5j1
sGRIQQBP9P6aF3AEgXbkpdXrdCn68KFYvVGmAHi/KWDVWARlsm2jMUyK3vpH
zh19SwG4zL7h7bkLQQ1j1LVNk6L3+8cbOdgIwOnXprurdiMQGrtX2xNu7C3/
sdJWAM5L9askqQgGRdUVDRoULbqt7ccn+hAvX/FthMCUtcdohOg7n8x/9tlU
AKua1L2L+AiUI34RVRPukWhXOmoigKwVr88IKxFUv1UPMiGcbdiQxCV6YX1O
tPlNBM7TiRor1Em/Gasnc1YIIKm+5+g6EQIPWdT5w2rf3t93XsR0fUT60v7b
axfhioUdTt4WAvDzSy//bhDBteTOsH3knXsmSqN0lwhAejFhUZ8UQbJbcp8h
mQPjYu2CJk0B3A3izuR/QDD6xb1gnhbxc2vdhgUqAhirUI5RkiMY2CYLYBGe
6WhWPq4kgBeOpjo7ZvNgS23knE2LKVr8di1LNkb8wzIfG1jJg8yy/ekL9Sj6
TcC19Zwe4mfromm5Dw+cy04d5RpQ9Mh7DjXRgeGH4t2ukxt5cPKc9QGJKUWH
zVzQ0iS5wQpPc7gUxoPXo64LlhD/xAXeu1lTjiFwJIuVEs4DE3s3RhLxm3Az
S5pegoG/+dFUE5sHD5Retnwg/tTI1zj3noch411EaYkDD45jmU062Rd+6ll3
ozwMZcmq17tIvoVxH11ztqdod6XQvn+PYvjzSK1uNOn3uvvxg1kkBzyW+s01
JVyU+tZQ/A6B1UxzjyrJjYKMVcNFWRhCgii3TAkCn+L3LWuJvkHrVmV/LoZh
neYJBZJXz1jSmHBS/1Bba7YjybEB9uSAnRjB8Gf5mmay13vXu/gYnMNg6LKA
mdePABs/3MSz+paHt/uN1m6fQrCBGxq7n+ypW5GuR2MVBpPa9vwmVR7k8pkB
LWQOtb7hT0/XYPi/3P4PwfXeSA==
      "], CompressedData["
1:eJx1lW88lOkax0kqEe0h26qEKaZjJxK1pa3Ln0Sndtvpz8ZWlCZFGyfU2mZV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      "], CompressedData["
1:eJx1lGtM01cYxgvehoNmRphsDAoM6oyIOuMIrMyXibFcdmm5hDpZEQQ1Yw4c
EAFjaEVhBIR1PSggl63/v7AihCmHoch6hgWcIwxhK2B0gVJWCgTKTS0U2Pm4
L/vwfvnlyZNznvO8xyP+K3GiLYfD8aGzgc6m0lZBQxcCccVa4zErJlmKoPQ+
LQPHDY7CYS2CoWdDfmuUb1CmZJZ2MBCn/dH0UQeCqtme7rRVTDjfcQdCf2Xg
tREBN64NgSCuXbNIeWnz2R2rhAGV4ycOkpsI+FtFoe+uYfJwu9PhjE4Gbg6e
1CXkIFjXTtQKqf9E5N0jCYMMWCXkSns0gj3+fi/cKW+0RAa36RnwMsSqNWIE
n36p1SlWMAnQ6D5+/R8GAnVJbQqq559wLtMvY9IdFTJiM8GA50qqeVCKIHws
I1P8H747f0r44CKCIrPM4kJ5VsZejmWEgWftvbt8ixDUNK8d2GfB5FG64ZTn
KAMpwbX3auoQcMu451eWMOGG3/rpspEBl0k7QZ4agUYhUpgWMckIHeBdmWKg
s1zmO1mFYM62CfPmMenK93n7vWUGzLdQeXUFgi/SG5tCKM+9sT9HSvnP2S/s
ldcROG7d2HKHcvsPWy8kWxjQH4zUvl+K4HRQ4Y5o6n8o5KzCd4GBZZfGkOxv
EZCN7p729Dwi+XJZ8hwDmsxt/Ve/QeAx09F8kfJw091lRHlfxaUxh8sIXnrC
nRnq4yyy07+xxEBli9FQJ0PweFu/fHYOExwu+Etuy8K9P52EsmIEeaKqc9Zp
THYOCAMqtrPgVX2jaEsJgvo3ewI4JkxeVUwEb/ZiIb83XRctR2AyJkrfMmLi
vW93VNReFq5ZNVdjjyJoqRapWAMmBfmHrBIhC77Xnzf8va6E0eTYypRxTCL6
zjT0R7MQeH9LZtKSEs40pMZpqF6SXcYXHGeBM7NgV7ughF2nptOWxjD5nmP5
bFzKwtE/ePwwixKc5irRTsofnZfKek+wMB8/f838CoLbBe7qcj3tWxj6xS+O
hQMMd73TBcH4kMAaRvWSFWRzP4YFfYlN37lABLnO/PgCqsdBXoXZESw45fjL
dWG0D5VR6uZRTNxkMVFiyi2bcwZ5iOZp8qjzoT6/H47IehzAwp4Y46Q37Xng
uNqYSXlxTdWg4CALMwlDkFuLoOmYg9tTei9/1y6uD82nepOUb1ONALI/995P
8xx4qOSd5rHg+sPTky8ZBK0qV0P7FCZLdvVISPOPSMwz99C98/pt+J3VGUyG
1U9S7dcYSFJLOLMPEHxwSVXiTN9xPa1bbl5koIBnbHPACFSBUpfC55jUfX0B
pmmfi/096p2bEQSzKW52tOemJ7dnm+je/d8/8C/MNzgU
      "], {{1095.2902362740597`, -5595.611875331073, 2859.5945042919257`}, {
      1093.5923945668474`, -5595.277611362775, 2860.8980786333564`}, {
      1093.3016721747538`, -5594.723780162536, 2862.092061290781}, {
      1095.1686109055727`, -5594.0582783743785`, 2862.6790567295943`}, {
      1102.1815852693298`, -5592.80297093698, 2862.440282586115}, {
      1111.6159112905993`, -5591.7858629614475`, 2860.7786734646324`}, {
      1115.6005557480241`, -5591.720152984018, 2859.355665307675}, {
      1116.4214243034899`, -5592.222827254237, 2858.05191410037}, {
      1116.6282465266768`, -5594.123352209939, 2854.2492829670978`}, {
      1117.942367310097, -5594.348492085204, 2853.293448364029}, {
      1118.9870936769348`, -5594.139618538972, 2853.293448364029}, {
      1120.7822524011594`, -5593.114744966145, 2854.59774440198}, {
      1121.2038322743663`, -5591.57116952917, 2857.454726252625}, {
      1123.5127141281787`, -5590.380095374165, 2858.877972288547}, {
      1129.1599282928012`, -5588.6312049285925`, 2860.072167207707}, {
      1138.1185823920478`, -5586.146163648215, 2861.3756867584225`}, {
      1151.071842873065, -5580.137472501706, 2867.9109279276936`}, {
      1173.0304006636802`, -5572.018884333617, 2874.7902396085096`}, {
      1186.8004947666107`, -5568.306819369157, 2876.330551412453}, {
      1191.8662510259974`, -5567.286326007157, 2876.2113089825234`}, {
      1195.2213025216672`, -5567.670475942248, 2874.0746683618763`}, {
      1196.1235338367712`, -5570.167841740331, 2868.8555900643896`}, {
      1199.0544532640474`, -5570.269893510494, 2867.4335949692468`}, {
      1201.8064991580088`, -5569.492453284734, 2867.791596574961}, {
      1203.178440572584, -5566.56694260116, 2872.8918886169895`}, {
      1206.3805682480006`, -5565.386373509204, 2873.8361345267585`}, {
      1207.794017669443, -5565.197843421337, 2873.607534869785}, {
      1208.1220110688046`, -5565.86550230718, 2872.1761969973045`}, {
      1206.1237072846882`, -5567.580621678577, 2869.690804570246}, {
      1206.2880819092456`, -5569.1379434961, 2866.59821386879}, {
      1206.7561247466792`, -5570.49992024217, 2863.7534782114803`}, {
      1204.6030474877855`, -5572.310195250896, 2861.1368852059595`}, {
      1200.5332229964117`, -5573.734911963938, 2860.072167207707}, {
      1182.092790778264, -5577.0063208873835`, 2861.3756867584225`}, {
      1168.4019546859433`, -5580.31919116495, 2860.539859362604}, {
      1144.4056227160506`, -5587.535441736195, 2856.1507572321525`}, {
      1124.671873365024, -5595.237868094782, 2848.9015544064982`}, {
      1118.4542283861988`, -5596.90992040475, 2848.0648117414617`}, {
      1114.558595894483, -5597.443688264524, 2848.5429579041565`}, {
      1095.2902362740597`, -5595.611875331073, 
      2859.5945042919257`}}, CompressedData["
1:eJx1lH001Xccxy8u5mGUYzckDiVLWrG4Xcr5lIuS58edhh22Up4q7JCJTB5v
XHlc39+y8/tRc9jGXL5tjrYvEm5TytA1HS7SsnloLCxp3//21/74/vM6n/M+
7+/782ARdTbgpCqPx7OnT40+74jbPuwuBrrn464MOmBSar7bL1GXg+4yabq6
FQNtJeOJUiEmITYqCTXqHHzDmA/dWEHwqlsucz2MyeTwykTDDAtuG2uOrU8R
uIieja+6Y3JiwFC8OsDC+7Vji3NKBGVv1SHlUUyc+2Slw70sJEuuP92uQBBf
XTdr4YnJy8jY+MluFnA3f+xyD4IEdd2m7uOY/H3PU+Z9h4W47fyA32QIxKVy
kb03JppplkwR5VUHJCH+ZQiI5DPtYV9Msr6d4WV0sWA+8MWrhQwEtqPbWlf9
MdkxkGf2EWGh+FyT7cpJBBL+vsq+QEx2cWltBZS/sLrJzAYicCvehtggTLTD
1/ZudFL9xujmve4IhOWCgNrg//h1u8r5qQMIzLLFzimhmAgXYkPbfmIh5Yet
1yR7EJjqIUnQB5i8yRxpk1I+Lgo5XGOBQFNL2Y4p31jMcuJ3sOAHj3qGTWkO
4W5bCynHsk6mjvovCtDIL9+CYN9ESaV2COXpynDLRywMNOSmfvjmGvRmN+PP
qZ+uYtts9gkLAmnjThEPQZ71mdkl6v/UVHqLyQQLLk17FvM0EEwbmrXbU/6m
UJFSrmThfJJqARgj8LjaUa9Pc/jY0nj4LK1fG7P5vcUMwW65IiclABONwQvP
ztD6wjQH/CvlG9FFqhaUt08hR70pFtSzjH2fbEZQkHJ/pIrmHDaqXE+aZaHy
0GRwvwECrSvyfn3KXZX1EZ1/sqDoFI/ITBBM6Juz1n6Y+AgCc0/NszAdstEm
sEfwPLK1j/HBxGBTsvzqHNUfcjDPckbw3XJorYL23fl52eQM5cyatUutmP63
a+lSkhcmUY97RJdfsPDup0NmuUcQWM3Lvb+mc5VTtiAUvqL6DhVV0qMI3rGS
z+JjdE7y37tY/5oFUYVRvvQ0gn6ecV0vndvGe76bU3kc2B6qT+hMQrD+OuNx
pxiT9fD98eEaHOjGBppEpyAYWBJmXnTFhOGzAh8dDprfFmkXZSL4WXN+ueII
JjrnYgN66B6ZOyl/tM1HMGoYeF6Hci2v5fselMf4H/e5g6jOjUumTXSPbt6y
e22nycFk/oWWwWYENn2RMgI052pj5z9UORA8SD1R8QhBZj/TMOeCidpf/hrb
qc+OsPWMiRkEtw5mrdsfwqRQd3HIVIWDT1buOhguIfgn+mXOXWdMpvpnvCrU
OHhomHbaaANBXbV69oiI9usrpSdPi4OQ/k3TNaoMVNomV3Y5YhJR5P9LooAD
cbE9KtFkoDoo06CP3oc+/smjMcYcuKTH8Op0GbjdJlbxplxo5JEwTXlEaLYy
bgsDcncnE8V+TMalq5ECIw70dqTyfXYyUFwblvmAcoViIep7Aw5yHYleDb0/
D6e/TLKjOuHBvINSfQ7+7y79CxFKW0E=
      "], CompressedData["
1:eJx11Gs0lVkYB/CznCHkspgcudS4ndxNRKPCPO6kiZVh5JJb7iG3k1RHaSST
EUpbiMUmlyknjnmXYfAillsMYzqDM4YscilkoWVomj2f5tN82F9+a6//ftbz
PO+rHhR7JkSEwWCYkMMkp8IlW3hJogAWGg1bA3UoumzZQnrDD0N6dOIrjz0F
8NpELLiZ+Ltehx/3EHfNk6lIEimATXebDTNdim6YEqh7+GIY/bX1nsMHBArY
qDtc97/7LEGpbdJbBJYawX0tJMepSidNJASDZ5y8ttoQAnH9LuH7QxT9lper
yeNgKGcmvFCuQbAQtqxpp03RzVLtQYXXMczoODea5iCo72ywiiQ5VILP/uU0
DGEpR0QvXkJw1+vagiR5V/bezMGMDAwDslsyPfYIzpyVc/qc+HZog09TPoYo
g/lJS0BQmT58WoL4+sxSsXgBhp2fJRfmiCe2lmwUk3yOoaWAfohBLc6WK3RC
sJc3F88kzoj0X1EqxHD15VcdZv4IOGqczSukTk7ytSEByVf0rkv8KRlB56y8
cJX4VD7bWCcLw0iN//rR68Sr2kxyiT85fDww/w4GSwf2N2PpCJ4HnG/vI32w
sZmzjf4eQ9dtuVOHiMtoOC71sSm6qkXURScPw6Q1v/YNyb8x8jImXIu4lJ6I
XDEGJssiyzAMAY66r2JKXLrSfNy5nPSZJS1yOwBB/G71YwbxbYesNINKDFPR
nbFfk/q99JiDdzQpupY3VfKgCkNTpkExMwJBkYad36Q6RRfUs7huTzB0B3P2
fBGLYIjd/+3mZ6TOLP9hBh/Dd9lWJueiEEwb1bD9DlK0/QO++ekWDFRzr1dD
CJmLWUmp5AGKdl/A8oudGFS406+5pxA48rpXbqpSdOLchhJ/EIO+Y1PYkAMC
T+sYm1kVig6TnbxrMIIh85GVJpfMkTUQbrepRNFMQfs53QniO6uxohYICuZs
1FsVKdojqXV2cB5DUcQFN+1jCNx0g1wCideldjQfX8Sw6n3BulcLgblrforP
foqWUig7ObqEQbRzhjeujODTDfHaNuKPnuqo7i5jyJ0PEE5II/C3V+p4RxyP
RHQJVzDES7KmGeII0j7RM0omrt4zFlK9jkHA99O/LIbgaHpo7DPybny5rEj2
FtnnuCC9WCYCdWfdHTaLopu2+DyrvzHIKxYFlEshGD5xclFMgaI/OlpUZzEq
IE+Puzp1AEHvxGGl1H0UvTyaxl7+gGGi+pfSHNIfVYt9PqbkfrzK+VdbpJ7R
Kue1Ei8EG+y1XFWSH2rsaVk9h+GvW10LSb4Isl3LkhpJnU8LH3t//B1D6lXI
vBGDoKNsIO6+MkWf0Ngb4jqAYe2PxpXEm2SOY2LG/aT/P1hZQ107Bu6VpmP1
txDoaq33C9Uoer7zcu0q2RPB+HiwfSGCOhOJwedkr8KpBH5kDobgs+XuL9oQ
vHFtY9DEdx9+eeRiOgalP/V77IUIBlMMFfvIvg2l/aYwRr5rZ+t2TvQ2At9n
jRP/7m2G+W7K+yQM//df+gdM0k/v
      "], CompressedData["
1:eJx1k2tQ1FUYxpFdsDVCWIzluiKXRdhgTVuEwPZlMoZWIGECRq4RouAFtFa5
aaMiEY5gydhwQGj5H4wBxlkBOysD4qGUhQElKTDiIjCauAgo1wxXOh/70of3
y2+eOfO8z/ucTZ9nRKQYGxkZSdnw2FyPsftWZI7gilbS5e9DaNoHJYprXRyM
mVz0XDVFsNtY6TDLeHXmr8k3OznoEgSMmBojGGspVG3bTuiXR5qaNjG9VqM0
bDSUgrwgaHWB6W0jhan6bg6iR7UPjF+XAsG6RW/GQ3LefhDUy4GqWO8/zEdQ
KX5iCHqP0GcmWamr/RxciotWO3giGLHWKSUyQpfKDt9tG+dg0riyw7AFgccB
L+s8L0IX3J61uj7hoD/RXl21A0HdXy9uDEkJdY5J79PoOZDvqEVPFQhWhKuL
HZ6EvrMlZsIww4GbSpN2m3GDMPM450HocM6yd/ICB5cFTnGFHyIIHUPX/thM
qCTxsfz+EgcKUYajfRSCxMJf1PckhGoshTbu/3AgcNqXeicGQf/Pcbu73Qht
ynqojTBwkFvLm/tpDwKLpQnZqCuh442/T75eg0F5r4LnGI/AvOL60isXQo9e
dlIv8hl/ZT31cQ6Ccyijs2Ajobmnjp+ZW4/hLW3c31eyETTL268uOxAa6rrn
+3UOGC4ZRfYojyHY6/z8gD/ja3nuw4+cMCQUPWqYTUfw28T+FgXjKDdlatoZ
Q07DYKc0AUGm70FxpCOhL3UeJidcMDz2rkp7czuC4V3iPDXjV4c5mUCGodmi
tmqE5RxWyX8jjPHec6rm81sxpIxXG72UIdjV1x2Rz96fJ19tUPlgONWoV9r6
IsjWfWcbaM/8e+SvVPhhePfijfsrLOcYccOdr+0IbZgKK/7sfQzhbep5QSgC
lws5p3tsmf8hr5p0pp84USLVspyX2yVi+//w0DX4BWb+K1em+TMiQpVnLPv8
AjBkPRTEzhxEIHUZGoi3JnQsu0PPV2BYkZh9upqHwC02LN5yA6F58eUJ5YEY
guMNvJ4iBPL51uB8K9a3kW/sRgFDkV4lUpQguDuZoj3G9OljNj63mE/SEtIn
K2C5RfF7ymwI1XqcV2ukGCJcjraaXUBQnzRa/ifzmbEUXhPC8pyXwUfmPyLY
aWWVXc321Qb88HROxPQC3uaeOrZXo1uvL8vH0qylod4KQ9le+c7TjGvaeLHL
jBfXlYWFW2AYUFonlJ5FEK07dLLeidD2m6bj29ZiEFc+dxxg9x3kJ6/LY31b
X/PF4PQyB8WHA08mHULwiXr/whF3Qm9nBje2zHKQFCfsj9qHoCbAb/oW63Px
2a0p3BQH//ff/wX4sPtX
      "], {{1459.6323478209592`, -5669.772112850047, 2531.9895907255586`}, {
      1461.3486675661143`, -5669.389309874148, 2531.8567524422588`}, {
      1461.8732655847418`, -5669.459400258669, 2531.396917551102}, {
      1461.6844485278145`, -5669.9140803380105`, 2530.4874203435015`}, {
      1458.9541683063358`, -5671.114205506654, 2529.373458562851}, {
      1455.3861232968723`, -5672.17675991721, 2529.046406527566}, {
      1449.361620560984, -5673.163196251153, 2530.2932501088176`}, {
      1442.849838843497, -5673.016910825087, 2534.339590907636}, {
      1435.883142493378, -5674.638221801416, 2534.666515100911}, {
      1431.5284112415106`, -5675.4189331474145`, 2535.3816342081964`}, {
      1428.4199168910313`, -5675.499164075634, 2536.9547630191973`}, {
      1416.537362403263, -5674.41684085354, 2546.0222126197827`}, {
      1414.6660107778323`, -5674.590534518063, 2546.6754864937034`}, {
      1413.0515968494224`, -5674.992761328718, 2546.6754864937034`}, {
      1409.346490820824, -5677.140742622603, 2543.9396901785035`}, {
      1407.8483162247533`, -5677.631383881094, 2543.674247475799}, {
      1406.5373109788081`, -5677.604072568436, 2544.4603510329653`}, {
      1406.0964445910354`, -5677.013067745671, 2546.0222126197827`}, {
      1406.8490389999192`, -5675.594437758183, 2548.7677904339166`}, {
      1407.4973925450088`, -5672.528805373274, 2555.226356350735}, {
      1405.8238644124453`, -5670.653371954446, 2560.3053113623228`}, {
      1406.3191421430392`, -5670.2358729447815`, 2560.957888293403}, {
      1407.689247796857, -5669.660986573299, 2561.4778876589576`}, {
      1413.1819159543468`, -5669.059000673225, 2559.785266249007}, {
      1421.9693534798619`, -5670.269308564836, 2552.2270306757036`}, {
      1428.7697225040286`, -5673.249325227438, 2541.785369354025}, {
      1432.8757634723615`, -5673.119272044978, 2539.763465680487}, {
      1439.4210739441914`, -5671.8416032897985`, 2538.915809765333}, {
      1443.1246848266692`, -5671.279851224845, 2538.068100570792}, {
      1447.8189908481434`, -5671.45855689345, 2534.9934313878457`}, {
      1454.9743267592646`, -5670.941865065236, 2532.0509002628364`}, {
      1459.6323478209592`, -5669.772112850047, 2531.9895907255586`}}, {{
      1253.6283114107, -5685.502897568844, 2606.009698478852}, {
      1262.88262667089, -5689.727762593178, 2592.2845424122065`}, {
      1268.7912382926324`, -5692.203446787299, 2583.95092910346}, {
      1268.6222857082523`, -5693.644883601474, 2580.8562776030594`}, {
      1266.8827692887057`, -5694.733398110072, 2579.3086793420225`}, {
      1263.730518569199, -5695.756491177435, 2578.595908560885}, {
      1262.3060875221397`, -5696.482535367549, 2577.68961584925}, {
      1261.9056809185283`, -5697.358943948459, 2575.948137342063}, {
      1265.0477770881043`, -5700.799505760686, 2566.778682863632}, {
      1264.7998126410573`, -5702.835172342906, 2562.3750937424834`}, {
      1262.2863712196138`, -5704.028532661895, 2560.957888293403}, {
      1258.0433918739677`, -5705.052799231994, 2560.7641578516022`}, {
      1253.4785872418875`, -5705.563177989511, 2561.8653249724734`}, {
      1248.1073109900237`, -5705.054990007844, 2565.616766245467}, {
      1243.1977846456434`, -5703.01542256942, 2572.525616624159}, {
      1238.465120926171, -5697.645547960026, 2586.668328963072}, {
      1235.9289147777633`, -5695.555478223597, 2592.477811302043}, {
      1236.2111229850818`, -5694.553947402619, 2594.542558820186}, {
      1240.3910425442723`, -5692.383661908789, 2597.308606463586}, {
      1250.956816975073, -5686.41258966393, 2605.3085406867185`}, {
      1252.1779665656104`, -5685.673454054533, 2606.3348602793158`}, {
      1252.8242241897754`, -5685.442581630572, 2606.5279212517908`}, {
      1253.6283114107, -5685.502897568844, 2606.009698478852}}, {{
      1468.5378338098233`, -5625.485173963751, 2623.952529712224}, {
      1475.0612632688824`, -5626.880069856685, 2617.294144541885}, {
      1475.791281005251, -5628.943905833948, 2612.440294032332}, {
      1474.5191278997463`, -5629.979373400919, 2610.926901799044}, {
      1469.098374324112, -5631.452819246159, 2610.805010176503}, {
      1464.7971823680946`, -5633.034455666786, 2609.8095190640415`}, {
      1461.4607956193847`, -5634.818250417687, 2607.828467830527}, {
      1458.9796189912977`, -5635.409473255679, 2607.9402274799486`}, {
      1452.9372436845858`, -5635.535848843389, 2611.0386347802773`}, {
      1452.114102243653, -5635.150197650934, 2612.328573127765}, {
      1453.0721180448822`, -5634.309819503261, 2613.608227644244}, {
      1463.921294473935, -5628.61746488995, 2619.8117427635134`}, {
      1464.208441135245, -5627.94732547046, 2621.090657081168}, {
      1461.1592536085602`, -5626.341130270247, 2626.235482858192}, {
      1451.4931730632616`, -5623.600620582535, 2637.4414341809515`}, {
      1447.0084068012193`, -5622.158320118486, 2642.974873919364}, {
      1443.240800478166, -5622.273904047242, 2644.7884281553665`}, {
      1437.8061990777105`, -5622.8988447789925`, 2646.4193933080182`}, {
      1436.539383563153, -5622.564573223314, 2647.817197127876}, {
      1436.2171247402998`, -5621.6305348473925`, 2649.974375399529}, {
      1436.4390516117728`, -5619.6772240570435`, 2653.9940539532486`}, {
      1435.361859810192, -5619.182507661164, 2655.623837631567}, {
      1428.4368055329796`, -5619.262396524376, 2659.1863608265344`}, {
      1426.2097173028483`, -5618.673548568407, 2661.6248959777627`}, {
      1426.2867698532425`, -5618.23695047788, 2662.505081389821}, {
      1427.1267405867243`, -5617.640020485722, 2663.314393108612}, {
      1429.901388120529, -5616.852863735726, 2663.4863651602864`}, {
      1437.4744361177081`, -5615.80215540215, 2661.6248959777627`}, {
      1445.0695109913652`, -5615.477641842201, 2658.1946227418207`}, {
      1453.3345289133983`, -5618.494941833409, 2647.2905064726965`}, {
      1459.4768404289312`, -5619.7015691941, 2641.3434679058596`}, {
      1466.240188403922, -5623.79170831692, 2628.862914146549}, {
      1468.5378338098233`, -5625.485173963751, 2623.952529712224}}}], 
    Line3DBox[{{{3654.733060348463, 4418.127362446578, 2794.7970510927616`}, {
     3648.911628009274, 4413.771432223799, 2809.2488969587193`}, {
     3636.889787423927, 4411.837351144031, 2827.8149782441487`}, {
     3634.204963622841, 4413.192306273357, 2829.1521062376187`}, {
     3629.34332566115, 4416.532922002759, 2830.179793466244}, {
     3625.006801902943, 4421.472219351919, 2828.024538375968}, {
     3625.7305015956554`, 4423.58303845525, 2823.792683907572}, {
     3632.035063372147, 4425.175027622411, 2813.1763121678446`}, {
     3643.2484208522096`, 4433.201586601319, 2785.9070922256965`}, {
     3646.6298041712594`, 4432.4407340144335`, 2782.691844380091}, {
     3649.69232341886, 4430.045231239156, 2782.491488495994}, {
     3654.0089902703126`, 4422.121419109345, 2789.4218296410586`}, {
     3654.733060348463, 4418.127362446578, 2794.7970510927616`}}}], 
    Line3DBox[{{{-228.37001818036617`, 5944.639608563876, 
     2301.5040189167203`}, {-152.087474701903, 5873.274272703892, 
     2483.814084756216}, {76.30572247356497, 5828.994850248532, 
     2589.3648183471323`}, {-63.486915475470425`, 5898.473276177768, 
     2427.302229179571}, {99.83578014457801, 5916.854577310983, 
     2380.898657641059}, {98.98212342206955, 5766.814666715875, 
     2724.30997300115}, {157.54783721968485`, 5699.702034689215, 
     2859.4385960791365`}, {15.099691985263904`, 5767.644307714236, 
     2724.30997300115}, {-242.02018060256233`, 5774.426576081314, 
     2699.117937331923}, {-118.98254854457228`, 5842.504854231767, 
     2557.1065428370543`}, {-165.71398474010363`, 5871.459003210278, 
     2487.2317420945164`}, {-232.82030024553308`, 5839.076513146022, 
     2557.1065428370543`}, {-268.31855829859563`, 5908.83348127847, 
     2387.7825972636456`}, {-228.37001818036617`, 5944.639608563876, 
     2301.5040189167203`}}}], 
    Line3DBox[{{{3161.238626876523, -5350.1566113714025`, 
     1438.821833217267}, {3155.814335739646, -5353.876476236721, 
     1436.891173867808}, {3152.6401540599363`, -5355.711290916482, 
     1437.0213350394808`}, {3148.6649427431516`, -5357.702975112875, 
     1438.3120658629384`}, {3139.3658973799647`, -5361.290884921489, 
     1445.2527459325022`}, {3135.153935722279, -5359.051675122762, 
     1462.5965379040852`}, {3136.328930091967, -5357.626913147608, 
     1465.2946433344412`}, {3138.764476094751, -5356.061071361115, 
     1465.8038932461832`}, {3142.678969848266, -5353.975738831717, 
     1465.034597023164}, {3144.7921170119284`, -5353.401497934806, 
     1462.5965379040852`}, {3149.7143936034886`, -5352.642729254766, 
     1454.7607162977342`}, {3153.7273552162496`, -5351.083093464185, 
     1451.8013589339546`}, {3160.517962337497, -5347.807368312481, 
     1449.1018774277961`}, {3162.018961975836, -5347.6832079834985`, 
     1446.28284963785}, {3162.9550802974327`, -5348.137137558776, 
     1442.552601427512}, {3161.238626876523, -5350.1566113714025`, 
     1438.821833217267}}}], 
    Line3DBox[{{{3442.4671924952086`, 1496.8273356939387`, 
     5157.202850937079}, {3376.9915416663935`, 1597.4985753608623`, 
     5170.274665743944}, {3235.274272694051, 1601.3147738918901`, 
     5258.954750267351}, {3145.1222073248446`, 1679.3388622485827`, 
     5289.219316376825}, {3102.8762121493764`, 1790.2431649211608`, 
     5277.7838448093535`}, {3212.3189131516033`, 1985.2596869668416`, 
     5140.774733978607}, {3338.6541609464566`, 2056.6321863293774`, 
     5031.050201871271}, {3521.0377929730857`, 1819.9380640663571`, 
     4997.792712129787}, {3636.0624322749222`, 1588.5566164101226`, 
     4994.331942818639}, {3442.4671924952086`, 1496.8273356939387`, 
     5157.202850937079}}}], 
    Line3DBox[{{{4003.0639489020314`, 178.27767234629303`, 
     4963.0141831898545`}, {3974.05199881478, 234.9421277826681, 
     4983.926825035576}, {4012.176724022939, 422.8767901210853, 
     4940.7921273400525`}, {4065.362848735844, 436.8542490936986, 
     4895.895930589943}, {4118.49694567693, 419.55208296077467`, 
     4852.810798715839}, {4003.0639489020314`, 178.27767234629303`, 
     4963.0141831898545`}}}], 
    Line3DBox[{{{181.2297007230084, -6046.945440067525, 
     2022.2315224579047`}, {115.98559223723758`, -6133.562746215638, 
     1747.5036682734187`}, {91.85450647915407, -6072.085568922601, 
     1951.7052244950712`}, {181.2297007230084, -6046.945440067525, 
     2022.2315224579047`}}}], 
    Line3DBox[{{{6338.613679426155, 180.74410521979135`, 690.7400968670692}, {
     6332.14772511139, 302.308695098596, 707.3389076209747}, {
     6232.941277359054, 388.5030277274849, 1298.9696379978502`}, {
     6236.5223162865595`, 261.38773044091243`, 1313.4989694836752`}, {
     6260.614173539858, 105.63607606587651`, 1217.1133388083663`}, {
     6338.613679426155, 180.74410521979135`, 690.7400968670692}}}], 
    Line3DBox[{{{2286.28895511996, -4890.143718335281, 3398.115789334211}, {
      2284.355904934031, -4890.680415780409, 3398.6433852781593`}, {
      2282.934424106374, -4891.042976491844, 3399.076743373104}, {
      2281.998069879759, -4890.929955961055, 3399.868036401978}, {
      2282.9836520211597`, -4889.61194681658, 3401.101930867804}, {
      2286.213938457571, -4887.492875476337, 3401.977793838515}, {
      2288.8575936546335`, -4886.320952059921, 3401.8836193247025`}, {
      2290.331144907899, -4886.358192778449, 3400.8382126369665`}, {
      2290.563255439796, -4887.474905867971, 3399.076743373104}, {
      2289.2219409979907`, -4888.653504737619, 3398.2853772851363`}, {
      2286.28895511996, -4890.143718335281, 3398.115789334211}}, {{
      2297.7294216922674`, -4875.478060648574, 3411.446449062939}, {
      2297.541349990204, -4874.770115575105, 3412.584615124008}, {
      2298.6358647866423`, -4874.00385615986, 3412.9420243032814`}, {
      2300.439048700082, -4873.218915873366, 3412.8479707115384`}, {
      2302.6041090407853`, -4873.177724257997, 3411.446449062939}, {
      2304.3625797958125`, -4874.124061855452, 3408.9061912112647`}, {
      2304.4323359118744`, -4875.19643212178, 3407.3252042373074`}, {
      2302.9064708465153`, -4875.976584241312, 3407.240500238232}, {
      2301.311672665155, -4876.302024796081, 3407.8522324515093`}, {
      2297.7294216922674`, -4875.478060648574, 3411.446449062939}}, {{
      2304.8897465073696`, -4871.234056239767, 3412.678671626596}, {
      2304.2251335555793`, -4871.060836396547, 3413.374657434274}, {
      2304.6298651819466`, -4870.684814019005, 3413.6379885682218`}, {
      2306.748925744414, -4869.6222539246755`, 3413.7226289870932`}, {
      2309.976164989261, -4869.074602480156, 3412.3212513865415`}, {
      2311.4732824415014`, -4869.464786558123, 3410.7503055822713`}, {
      2311.141176354994, -4870.301012260457, 3409.78125441079}, {
      2309.474986511255, -4871.09133293019, 3409.78125441079}, {
      2306.738314922754, -4871.7754599742175`, 3410.6562277749904`}, {
      2304.8897465073696`, -4871.234056239767, 3412.678671626596}}, {{
      2299.516176978697, -4870.339058900772, 3417.5775767579617`}, {
      2299.8971832250054`, -4869.604976812402, 3418.3671550021622`}, {
      2300.9425182324944`, -4869.045142602074, 3418.4611475184347`}, {
      2302.5859827764843`, -4868.518946182789, 3418.1039704170034`}, {
      2304.5584939071514`, -4868.6875675701785`, 3416.534092841454}, {
      2304.6776198044977`, -4869.554610947501, 3415.2178041483976`}, {
      2303.1791046023704`, -4870.388825463007, 3415.0391492526237`}, {
      2301.7986465809963`, -4870.85680349481, 3415.3024288408496`}, {
      2299.516176978697, -4870.339058900772, 3417.5775767579617`}}, {{
      2282.2186429053904`, -4887.038919918313, 3405.3109037470754`}, {
      2282.2724834540786`, -4886.219924922653, 3406.4498890424347`}, {
      2283.157101051017, -4885.622879312622, 3406.7134342438253`}, {
      2285.1113603751687`, -4885.135772131042, 3406.101620405648}, {
      2286.0835010802784`, -4885.658676928004, 3404.698989433082}, {
      2286.055162519935, -4886.531213545899, 3403.465613599995}, {
      2284.9315298391098`, -4887.2403212404415`, 3403.201968204169}, {
      2283.096135851324, -4887.671843325812, 3403.8139896916828`}, {
      2282.2186429053904`, -4887.038919918313, 3405.3109037470754`}}}], 
    Line3DBox[{{{107.1507652980063, 5666.358463869147, 2927.236608549868}, {
     3.4384536718293273`*^-13, 5615.4209919518225`, 
     3025.700046790298}, {-162.51901201840562`, 5641.866194303816, 
     2971.6572996064547`}, {107.1507652980063, 5666.358463869147, 
     2927.236608549868}}}], 
    Line3DBox[{{{
     2186.3696607662596`, -5868.518455983097, -1211.6486818655826`}, {
     2265.5219439663015`, -5800.78945995964, -1380.6033654269768`}, {
     2130.477608373321, -5698.221203685099, -1918.1120832979068`}, {
     2280.0012724481317`, -5419.495201581031, -2473.5560732829726`}, {
     2387.1347732829595`, -5420.898346128181, -2367.121117545119}, {
     2547.250887658689, -5296.738827385517, -2478.6860228332102`}, {
     2598.8100681815913`, -5312.652347228116, -2389.5030774220777`}, {
     2714.175321627789, -5243.6392984739705`, -2413.5685034749054`}, {
     2843.341196027777, -5291.715301068514, -2144.9880431854845`}, {
     3158.1847113327835`, -5087.01993835785, -2199.070523165084}, {
     3260.272176817753, -5107.7604301293395`, -1992.2907447139778`}, {
     3214.3426340146716`, -5211.24314724949, -1788.507752681226}, {
     3046.23659707523, -5311.860717090205, -1786.726617833142}, {
     3053.490417531991, -5389.70742959446, -1521.5333658564937`}, {
     2691.160312707677, -5616.884867180803, -1376.9801197339118`}, {
     2615.5585205534817`, -5690.992099805619, -1208.0050643561553`}, {
     2614.1717854013673`, -5718.6339880463465`, -1072.9141984206135`}, {
     2186.3696607662596`, -5868.518455983097, -1211.6486818655826`}}}], 
    Line3DBox[{{{4456.359088042351, 1502.6173281602632`, 
     4309.3872573180115`}, {4388.893607484269, 1288.9475777753805`, 
     4445.676633981681}, {4366.170019396594, 1260.230220387981, 
     4476.180608883336}, {4324.145696939212, 1235.8460470086357`, 
     4523.523209815025}, {4274.436765335887, 1427.4419030052027`, 
     4514.356392963427}, {4278.47303147275, 1456.5110269499871`, 
     4501.228480907569}, {4456.359088042351, 1502.6173281602632`, 
     4309.3872573180115`}}}], 
    Line3DBox[{{{5492.316792830914, 2592.298059739058, -1949.938641768162}, {
     5653.312543647596, 2170.103364320896, -2004.625242283426}, {
     5557.556083289681, 2022.7849895825027`, -2389.5030774220777`}, {
     5442.695712348549, 1980.9792334631563`, -2672.1898041438662`}, {
     5326.172980927055, 2007.2831870471716`, -2879.324447546743}, {
     5300.81614493333, 2228.2579291069687`, -2761.164335744264}, {
     5188.520877851617, 2476.6504159461297`, -2762.836858609597}, {
     5146.101822309165, 2897.7074588670225`, -2410.133026878956}, {
     5492.316792830914, 2592.298059739058, -1949.938641768162}}}], 
    Line3DBox[{CompressedData["
1:eJx1lGk4FAgchx1LIcfjcWWUqISVxnaQin+mEqYU60ikFlHtiMpSKEcN6WIi
MmxmGs1hGnMaZ/5Rs13KUVu0rtm0KJEux6rtU33aD++X9/l9fJ+f1S8H/aLU
VFRU3L+i/hWLfFY9rZYD3YMZCmWZECcpHnVn+BWYheE7HGZ4UO2oW5TwRIg4
sKh7aISDJ9QMjfK9KiDUQK3CplCIbQJe1Kb1XDxzPMWr7g0XCLvPlj5gCrGE
65z3JIuHr8rdjXu6+RA1W19CShBiFteR0dHCxuY9encLRNeB8hPJdnJEiMfK
BHe0AsuR9tB01dZDIlB5KMnr9xFgEe+SMJfNxEazLxHa+8XAGy3QIXy6jmfK
ZvI3bvodWU7do6OtYlCJzVevPiJA3/KmchNmIaolht15PyiGMUOnlRPZAsy5
lN/y6FMBMjXuv2uzkAAqJ7jVmt99TfSuG29XSuHW+oX0hjV8vEDtpKSNn0Wv
2CzfThcp3JXVGVOb+FhXlmRWa5mBr1OoQ34GEgi0HRle8rkSc7Zt0KW2H8Yf
9Ej+Y2wRHK5++bH/lAjZ3Q4bxsZ/w1znFpagTgS6bSST+hQRtgvvjyftiMF/
cwPPdcwRAXlpkfsf9mKMHHs9+OO4PyqWrNgfHiqEnrVNa7oVYjwcUNtu3xqD
RObnn/PMhXA/XsAkT4lxr79T1wz3EGYVbpXylwvh6YKJnZrD3/fBPQdJGacr
YcwkoaNvk+Sb/5T4gh+RUAl/ap48reEnQbnqCE07cwfWGt5W7XUQgHGYJWVN
mwR703SmtpP2os2+eXrrdIWwZ1F4+Jc5EoxR6veort0NXnmJvqbtAvC42de8
NU2Cloufap/SToEIh3Vyqo0A/JZXExe0ShCPMn0Xr4kDc+cttROP+aA07xt2
Jku/eeNdbqsjMiugJDZWNHVTigaaLvV0qwMgNZq80JPEBdHscS9yjAy3R859
K3aPgFVZtJsaZlwo9Q4JDouXoacdgbg56Ti0zfjYEUgccIp3m1NQLsPL1wtv
vDXIgKbLg9ccJWzweVl8U++pDNW1rBboc1Nh9YDTEL3iGtiNbuw8b12FIc76
K0xbkuDBrBoPnm058FtevjyRXIUUYpM+T5EKYScoiS7ka5BDD7pPJVVh2QP/
55whCjz6UnBngHYVpp9x9qUMVuFmlb5Usze7wXfKPqzSlgFHfMy/FO2R43hU
UbZCGQ0DgaX23s0McCcz73l5ynFgkExabuoHcyd3B11WYUD/cKaAfkCOfzvc
Ulu9dhssC0+mDWoxoDQ10fFFkBxnrD0C1m1LRWpVl7E3sRRmF9sUk6rlmEyg
BD16ko7Sk5bZeivpoKpMfu/YKMeW26RWK+scbDzp65t7oAQ26O8yS8iWo51h
n9DVMxebK69+Towqg2WsjtAFDnK8QvSk9gbkonaIQcNZlzIYLe7a1mMox7Ve
tvMbdC7iXp2lGxqMGNCmnl1z8nUV3jDTh6I3NKTnrX8f9CsL5kVpb2EGVmHo
McZCTnMuit9Ef5x3kAVpdTA141qFQCxcptFHw1Wd6UZ+muXQ1c6mKsdkSBpJ
Z73KLMCLtapm011siLUgmMt0Zaio4dhbKwpRmQQLjz5gQ87p90F/NUpxUXC6
3uAwHTUbayiVFlzgvx5wc9spRW/G9rFZ43T8cKop5MzXfmKyz1Y0vZOgUdJ0
tZ3VFVR/nkzbxefAP9YCc41eCfo0lhAWOTKQ2W9KDEnmQOaF0bTKixKk+QQe
XlHPxMDhrS1CJReOkLbEFRtJUJDiUR83wMRs22BF2XMuLDWMce1sEGMHcfrt
fDcW3nOKNEi6xIOGkjSjyEQxzrcm6Br6stBK1WFVhjYPyHvSlwX7ibGY4y4s
Si9HV6lpUaQTDz7sc5zqeyFCz7zoOofaaxjnLyOc12HDkTvkUhNHMSpYXiXP
zLloWQA57ZMcOO0ZEKM1W4TnEjy4j+N5+H9//h+otQ6d
      "], {{4234.940264263703, -4769.928017061932, -29.687710684621855`}, {
      4065.2838951851813`, -4911.174647380056, -204.0689209051348}, {
      4050.1644764594143`, -4927.827679309303, -22.26581818228762}, {
      4234.940264263703, -4769.928017061932, -29.687710684621855`}}}], 
    Line3DBox[{{{2622.6296539934056`, -5443.8770137146585`, 
      2042.9954595014644`}, {2618.135569010774, -5444.947040953781, 
      2045.9060470826346`}, {2614.4198241705267`, -5446.343888871289, 
      2046.939402409978}, {2610.199267055564, -5448.296571564656, 
      2047.129195859267}, {2608.813866267959, -5448.674812223077, 
      2047.8883494501943`}, {2608.6625617759387`, -5448.358803102877, 
      2048.9215898711145`}, {2610.0324597461963`, -5447.3139884342, 
      2049.954770349899}, {2614.6650331634382`, -5444.881524431728, 
      2050.5135062355607`}, {2621.1781126405067`, -5442.49157148953, 
      2048.542039129241}, {2623.8506056941687`, -5442.159663534834, 
      2046.0009495581262`}, {2624.9003994189534`, -5442.394426948441, 
      2044.0289833196518`}, {2624.7957558640446`, -5442.832946403417, 
      2042.9954595014644`}, {2623.9584988212955`, -5443.379120621343, 
      2042.6157826784527`}, {2623.5784728748213`, -5443.562294216969, 
      2042.6157826784527`}, {2622.6296539934056`, -5443.8770137146585`, 
      2042.9954595014644`}}, {{2616.529171818459, -5457.914879588223, 
      2013.1452878996288`}, {2615.4145806932647`, -5457.544612931017, 
      2015.5959562781213`}, {2615.426861666422, -5456.910397721252, 
      2017.2964414075147`}, {2620.0175302106186`, -5453.417673812899, 
      2020.78140983486}, {2622.0693621226706`, -5452.153509139527, 
      2021.5311184830055`}, {2629.505412891109, -5449.025392259599, 
      2020.3062263148297`}, {2630.5495442367114`, -5448.905031697818, 
      2019.271339086012}, {2629.6681957700675`, -5449.678620164392, 
      2018.3314413342816`}, {2623.781193950535, -5453.105457649963, 
      2016.7366719578672`}, {2619.4080111097164`, -5456.39350461239, 
      2013.5255856725437`}, {2618.322527195544, -5457.085957256766, 
      2013.0607762040713`}, {2616.529171818459, -5457.914879588223, 
      2013.1452878996288`}}, {{2599.891679632735, -5464.746133342796, 
      2016.1663230724382`}, {2602.4445848022706`, -5463.842586911051, 
      2015.3213288432428`}, {2602.89515183099, -5464.149855486751, 
      2013.9058754963778`}, {2602.6617549417424`, -5464.642470701883, 
      2012.870623454174}, {2599.7083932604532`, -5466.3281403268975`, 
      2012.1099925902233`}, {2592.1688959742437`, -5469.907447339694, 
      2012.1099925902233`}, {2588.486960437459, -5472.376903506686, 
      2010.1343168086576`}, {2587.732398975241, -5472.73375468623, 
      2010.1343168086576`}, {2586.7280413341086`, -5472.932979572825, 
      2010.8844649257999`}, {2587.237038524954, -5472.032120458278, 
      2012.6804687176525`}, {2589.695602958804, -5469.966784351143, 
      2015.1311997264716`}, {2593.4955603725116`, -5467.749491876265, 
      2016.2613824638195`}, {2599.891679632735, -5464.746133342796, 
      2016.1663230724382`}}, {{2571.182254284206, -5479.671823324038, 
      2012.4691833473107`}, {2569.706092892402, -5480.306035090353, 
      2012.6276476049688`}, {2569.57632063632, -5480.029275160429, 
      2013.546713093533}, {2571.6758738255203`, -5478.535565923548, 
      2014.9305057233332`}, {2575.9804472299834`, -5476.345763121047, 
      2015.3847047735}, {2578.2951263655136`, -5475.3108095722255`, 
      2015.2368272591164`}, {2579.0691002890667`, -5475.342746064224, 
      2014.159397628704}, {2578.1367079920606`, -5476.065424383669, 
      2013.3882568381002`}, {2571.182254284206, -5479.671823324038, 
      2012.4691833473107`}}}], 
    Line3DBox[{{{-2594.5999242418993`, 5804.830332581286, 
      509.71534513704256`}, {-2687.711894298745, 5759.442126013055, 
      541.1514209084804}, {-2687.6452128201145`, 5759.29923565421, 
      543.0001972356416}, {-2594.5999242418993`, 5804.830332581286, 
      509.71534513704256`}}, {{-2700.9731788946115`, 5752.885990784736, 
      544.8489276163259}, {-2689.5859380764487`, 5759.085747828161, 
      535.6048178125125}, {-2700.9731788946115`, 5752.885990784736, 
      544.8489276163259}}}], 
    Line3DBox[{{{4046.161351126181, 2204.5121725336708`, 4410.954328918362}, {
     4185.882978304977, 2227.235565520895, 4266.8043214583295`}, {
     4266.639866010629, 2113.3395890083657`, 4244.690062336561}, {
     4409.931250088139, 1865.8520239232048`, 4214.132922003972}, {
     4216.853053132038, 1762.5067460395262`, 4449.666727796158}, {
     4046.161351126181, 2204.5121725336708`, 4410.954328918362}}}], 
    Line3DBox[{{{6284.627508902851, -294.5433452330213, 
       1050.9590772815511`}, {6244.180476339045, -603.0794452278969, 
       1155.1272880725778`}, {6187.597805255001, -429.06173422816937`, 
       1489.0779424276636`}, {6161.35106815008, 0, 1650.929042995449}, {
       6161.299976060484, 25.09163189600567, 1650.929042995449}, {
       6236.140306225268, 261.3717194755278, 1315.3146373894813`}, {
       6260.614173539858, 105.63607606587651`, 1217.1133388083663`}, {
       6258.629532840139, -18.205666684790824`, 1231.681214585082}, {
       6253.850929551926, -309.5114883768675, 1217.1133388083663`}, {
       6284.627508902851, -294.5433452330213, 1050.9590772815511`}}}], 
    Line3DBox[{{{5571.6131884602555`, 
     3090.5146588593902`, -306.0380883990966}, {5539.550537865202, 
     3123.499665319096, -494.9175377664885}, {5487.4855963700375`, 
     3240.9823472075554`, -267.11223244244565`}, {5571.6131884602555`, 
     3090.5146588593902`, -306.0380883990966}}}], 
    Line3DBox[{{{-1565.47607488617, 6075.112497463138, 
     1153.3024287350524`}, {-1396.5643624708398`, 6089.579690647273, 
     1286.250699798711}, {-1324.558920338368, 6058.04068002081, 
     1494.4900386971653`}, {-1621.980752123125, 5963.024830454567, 
     1580.9259948284682`}, {-1860.4623200834596`, 5882.727209046518, 
     1618.6458011230995`}, {-1878.9108924376685`, 5941.061052943609, 
     1364.295097651402}, {-1706.1577778652472`, 6009.412236149287, 
     1289.8852286343704`}, {-1565.47607488617, 6075.112497463138, 
     1153.3024287350524`}}}], 
    Line3DBox[{{{6280.0192889738055`, 1086.6290677077002`, 
     261.55054832600234`}, {6270.215191596337, 1077.420022903925, 
     459.7621508130301}, {6284.707076602275, 950.4725826878434, 
     535.6048178125125}, {6241.405209539873, 1078.0778766264852`, 
     755.2633926366776}, {6194.357747371584, 1303.4741484729236`, 
     786.5754457395809}, {6045.994221790738, 1580.4989265569993`, 
     1278.9803373098352`}, {6026.811542878753, 1510.1034930589688`, 
     1443.933432286449}, {6048.234643111922, 1698.1838731496737`, 
     1105.8222844721363`}, {6103.27547937817, 1516.0615077747616`, 
     1067.4266332937787`}, {6093.65582504921, 1689.920323339351, 
     836.2665105147113}, {6143.4309835600925`, 1579.2699827621407`, 
     672.2914262284892}, {6120.841723438614, 1778.268906464187, 
     246.7184284071085}, {6203.178129531861, 1466.369172562866, 
     241.1560346640822}, {6249.847177541226, 1252.624330884985, 
     241.1560346640822}, {6280.0192889738055`, 1086.6290677077002`, 
     261.55054832600234`}}}], Line3DBox[{CompressedData["
1:eJxFkmk4FIoXxsuWJSSpLjdKokikIlIdWyJLlCIka4gsKamQiCKSfR8zI8Y6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      "], {{-2485.532247842629, -2792.969993580998, 
      5168.100400523438}, {-2588.218847250822, -2804.825899860535, 
      5110.957695012468}, {-2549.947998839148, -2728.113419651859, 
      5171.361142168633}, {-2485.532247842629, -2792.969993580998, 
      5168.100400523438}}, {{-2457.620237719274, -3206.6941064356693`, 
      4936.094545782465}, {-2345.1238660319173`, -3536.3866043337075`, 
      4762.5809733438255`}, {-2458.552923233545, -3205.9790780284875`, 
      4936.094545782465}}, {{-870.2383969709025, -1523.6299945556805`, 
      6132.621826113834}, {-775.45945448244, -1639.2628194368729`, 
      6115.496196815568}, {-1120.2132501085991`, -1738.230709745344, 
      6034.193397959608}, {-1160.673508997101, -1597.530033353446, 
      6065.356418879775}, {-982.6462008899944, -1414.7205414164189`, 
      6141.724995762257}, {-870.681566146414, -1523.3767880091439`, 
      6132.621826113834}}, {{-794.0520735010024, -1734.360150922311, 
      6086.812783506774}, {-611.2352454840052, -1816.2487811660808`, 
      6084.0320125383}, {-617.1689583582406, -1916.5085189971978`, 
      6052.59541547675}, {-567.3037113273505, -1737.3791307808658`, 
      6111.259603800295}, {-506.15499672496884`, -1862.9598793176287`, 
      6079.556011531039}, {-412.04460973427865`, -2126.389712665681, 
      5999.708302864273}, {-514.9698471388651, -2248.4764269727298`, 
      5947.013830813387}, {-637.5673007376995, -2140.9515143853478`, 
      5974.750893558753}, {-926.6978486110676, -2150.0465484972997`, 
      5933.493458888656}, {-1005.1187188706282`, -1936.3125822122518`, 
      5994.017320207073}, {-786.2001103784397, -1997.589097288877, 
      6006.608100653274}, {-980.5188142169154, -1878.220099090857, 
      6016.534368250362}, {-856.0183053003086, -1830.174417344402, 
      6050.248586057342}, {-974.9201798124412, -1751.5858543029965`, 
      6055.51742859354}, {-860.8801422805138, -1664.3594323896418`, 
      6097.262234022965}, {-794.0520735010024, -1734.360150922311, 
      6086.812783506774}}, {{328.77350197276206`, -2684.099818618495, 
      5777.138551055594}, {447.6551862947699, -2815.78027326347, 
      5706.03189413481}, {239.67903551705672`, -2874.314751544314, 
      5689.348148864539}, {139.45313353257472`, -2834.4278244284005`, 
      5712.651347811629}, {198.93883556232151`, -2595.310161887445, 
      5823.452772480332}, {328.77350197276206`, -2684.099818618495, 
      5777.138551055594}}, {{-452.2004168258639, -1750.629888754549, 
      6117.076382196135}, {-495.7949328034141, -1814.3945220296669`, 
      6095.0778168154}, {-531.203763469428, -1733.880768222038, 
      6115.496196815568}, {-452.2004168258639, -1750.629888754549, 
      6117.076382196135}}, {{-300.2676936884607, -2173.19877452297, 
      5989.562595760969}, {-223.7565005492735, -2309.714681774229, 
      5941.629810688185}, {-376.64895450539626`, -2266.9661405967663`, 
      5950.362490845226}, {-300.2676936884607, -2173.19877452297, 
      5989.562595760969}}, {{2236.692085866948, -3268.6456552792765`, 
      5000.097777368365}, {2263.0128150926835`, -3464.84153414205, 
      4854.014815800469}, {2375.725094517177, -3441.6672699472783`, 
      4816.499804656385}, {2475.9987250560534`, -3344.085391528769, 
      4834.701324558186}, {2552.4235306532237`, -3470.470052780054, 
      4704.124299263843}, {2573.559413613749, -3394.638413409697, 
      4747.740059944663}, {2630.06587403735, -3500.7975007326036`, 
      4638.424737676413}, {2498.5146429055585`, -3470.6557778316214`, 
      4732.841296831163}, {2477.4716223304163`, -3589.065508663435, 
      4654.949712410433}, {2403.4545236227523`, -3532.153932817246, 
      4736.571401104493}, {2195.233593410663, -3697.194742610721, 
      4711.636042350146}, {2176.1893985214006`, -3679.7353143980135`, 
      4734.085065536446}, {2190.4114387111813`, -3482.806240445645, 
      4874.420181998684}, {2237.513369361912, -3269.845858357979, 
      4998.945456244334}}, {{2160.3168342084764`, -3789.999835520375, 
      4653.680920797106}, {2149.609748859439, -3902.0557167024717`, 
      4565.18924047602}, {2231.5741771648563`, -3834.2283301777693`, 
      4583.293956738231}, {2115.710630895244, -3899.358757852842, 
      4583.293956738231}, {2160.3168342084764`, -3789.999835520375, 
      4653.680920797106}}, {{913.3589698726577, -2176.338533534152, 
      5925.971622409913}, {1216.19912842115, -2218.368281467367, 
      5855.554246850349}, {1205.9749833762758`, -2424.1069265495375`, 
      5775.564356677635}, {1107.6370898968921`, -2306.6490353438608`, 
      5842.972188435606}, {938.2860773951835, -2386.0478639631824`, 
      5840.736984135057}, {1015.950986983084, -2438.6657860336322`, 
      5805.907893351139}, {1244.8656619494318`, -2625.622189213731, 
      5678.400416772993}, {1011.0372879947598`, -2633.842182952826, 
      5720.882156414559}, {1220.1506201000273`, -2749.1073449446476`, 
      5625.073773759852}, {922.0821277736222, -2739.911602083314, 
      5685.988274030268}, {702.7630312476533, -2659.851315391918, 
      5754.922136043153}, {572.6869119752954, -2694.28008938321, 
      5753.320622979863}, {576.2659968102955, -2585.129492970684, 
      5802.8300591602165`}, {687.4819176309737, -2636.139875257754, 
      5767.664073176113}, {753.4559457381645, -2540.910242599441, 
      5802.059372919675}, {835.5932228027483, -2461.5782568191585`, 
      5824.966106473175}, {900.1073590249033, -2371.627294972255, 
      5852.606633455882}, {736.8785121204462, -2407.715834381609, 
      5860.693492176777}, {535.0090774030043, -2254.4581528288923`, 
      5942.97883339994}, {402.00509068042794`, -2071.353602495998, 
      6019.6096094758395`}, {347.82277582137596`, -2183.731147111547, 
      5983.155629244053}, {78.42129028651236, -2155.7897228969223`, 
      6002.852193905977}, {15.639779011406217`, -2067.8682557970533`, 
      6034.193397959608}, {-0.5769695772304397, -1983.4752450215922`, 
      6062.477984248078}, {119.0119681701063, -1768.4720887600545`, 
      6127.49184517686}, {116.0225362829141, -2075.215095021724, 
      6030.57499501237}, {141.17377359344547`, -1870.2744863518283`, 
      6096.716903494108}, {249.69360113896767`, -1769.1933507742208`, 
      6123.350527540175}, {311.4235816934426, -1958.874608869232, 
      6062.477984248078}, {423.36131309108583`, -1843.563297998364, 
      6091.785719709056}, {529.6014317872365, -1988.0610238089757`, 
      6037.793419707278}, {594.3700472769209, -1991.6583960186485`, 
      6030.57499501237}, {626.0615673824478, -1917.3262575865392`, 
      6051.423024860833}, {631.9605296033507, -2018.646090790525, 
      6017.7659923895}, {690.4896913479619, -2067.6529927904166`, 
      5994.651681085985}, {719.03874085862, -1961.3293065397754`, 
      6026.938221887848}, {781.8312468635389, -2036.740031640125, 
      5994.017320207073}, {761.6193482457018, -2096.3258582124067`, 
      5976.049493985789}, {888.0607300435756, -2068.676277721657, 
      5968.2275666856185`}, {809.1560712184037, -2152.7248402777464`, 
      5949.6937656035325`}, {913.3589698726577, -2176.338533534152, 
      5925.971622409913}}, {{-155.6271976530272, -1749.5160922068617`, 
      6132.111162434998}, {-6.174637649714769, -1768.8962022332232`, 
      6128.521990625046}, {-139.91181739303306`, -1967.923966493936, 
      6065.9305663230725`}, {-155.6271976530272, -1749.5160922068617`, 
      6132.111162434998}}, {{-668.8876636487469, -1224.2834810034042`, 
      6224.26967133982}, {-641.7596352501963, -1321.651805195402, 
      6207.18895848896}, {-790.2415166359823, -1350.531957285434, 
      6183.801053376122}, {-836.3029420634579, -1286.1548471246897`, 
      6191.463218683744}, {-668.8876636487469, -1224.2834810034042`, 
      6224.26967133982}}, {{-279.1946374319135, -1145.3024304196515`, 
      6268.823365457357}, {-214.55025371993074`, -1327.1191862703986`, 
      6235.425931246534}, {-329.3937504468166, -1245.2385225311132`, 
      6247.294972159639}, {-279.1946374319135, -1145.3024304196515`, 
      6268.823365457357}}, {{-265.77886037471546`, -1450.4913574285372`, 
      6205.904455387871}, {-217.1585850163201, -1623.9243971312085`, 
      6164.6999433350975`}, {-352.18925559392494`, -1552.2566765370502`, 
      6176.914733780153}, {-307.3192754058796, -1462.5509696174904`, 
      6201.15418403985}, {-265.77886037471546`, -1450.4913574285372`, 
      6205.904455387871}}}], 
    Line3DBox[{{{5661.486146174854, -2439.439477172054, 
      1638.4515079442326`}, {5662.198723238621, -2439.266125474273, 
      1636.2457265547412`}, {5661.919965391546, -2441.7942742451205`, 
      1633.4370871767105`}, {5660.110565908968, -2448.246143647114, 
      1630.0468986512906`}, {5656.437593810307, -2457.5054561405664`, 
      1628.8629071744551`}, {5653.517738324302, -2463.724104214682, 
      1629.6055997584308`}, {5648.295006876678, -2471.5897515298116`, 
      1635.7937845048784`}, {5643.486563893003, -2475.5202102097314`, 
      1646.412093615176}, {5639.613897772776, -2474.302690585028, 
      1661.4436401551063`}, {5635.72920555289, -2476.7047855037667`, 
      1671.0186764787904`}, {5634.081214916735, -2475.9805515765916`, 
      1677.6360934174163`}, {5634.59540592002, -2473.3088943758125`, 
      1679.848628474155}, {5636.3861196173275`, -2469.028041375854, 
      1680.138604843314}, {5638.1798744776215`, -2465.229222309686, 
      1679.6982688881615`}, {5641.573230011276, -2461.1671997287963`, 
      1674.25242881689}, {5647.058622106006, -2458.2593022202655`, 
      1659.971027321187}, {5653.699111980014, -2451.0154836715724`, 
      1648.036146447825}, {5661.486146174854, -2439.439477172054, 
      1638.4515079442326`}}, {{5662.568036890699, -2362.911039265667, 
      1743.4988781471175`}, {5661.682464412808, -2365.442566295328, 
      1742.9420035442026`}, {5660.331530725879, -2368.673431385539, 
      1742.9420035442026`}, {5656.706095565711, -2376.745099326512, 
      1743.7237657430553`}, {5654.2018442203125`, -2383.10393483603, 
      1743.1668969392015`}, {5652.642558384708, -2386.07247085093, 
      1744.1628252503615`}, {5648.556616672161, -2391.5873336790596`, 
      1749.837667795224}, {5648.207753069186, -2391.439625542154, 
      1751.1651508512662`}, {5648.845686329914, -2388.62930712705, 
      1752.9421368561082`}, {5648.024951718562, -2388.282257330524, 
      1756.0568595819286`}, {5645.687780165117, -2391.1047679295543`, 
      1759.727583371592}, {5645.011415200671, -2391.7246963647826`, 
      1761.0544537443739`}, {5645.109285864921, -2390.673843626779, 
      1762.1672491680245`}, {5646.185213732375, -2387.1442168991352`, 
      1763.5046668367227`}, {5646.522484276503, -2384.2090048628424`, 
      1766.3932024582869`}, {5645.325311504713, -2382.2640432254752`, 
      1772.8321538259688`}, {5646.249720136022, -2379.5782392640267`, 
      1773.495190601036}, {5647.283320641202, -2377.1242279953863`, 
      1773.495190601036}, {5649.285163150873, -2373.441483720548, 
      1772.0514548843714`}, {5650.187805448977, -2371.2918619178563`, 
      1772.0514548843714`}, {5654.223031501273, -2361.236539444486, 
      1772.6075721469708`}, {5655.791686754566, -2357.5571278351786`, 
      1772.5006276533375`}, {5657.180122599038, -2354.8915566929318`, 
      1771.6129675204654`}, {5658.523781048933, -2354.0025984796034`, 
      1768.5005161590857`}, {5660.264200561986, -2353.8227605614948`, 
      1763.1622959044485`}, {5661.511635015716, -2351.815292081164, 
      1761.8355566042123`}, {5661.962495149053, -2351.643385166268, 
      1760.6157396005644`}, {5661.0707067431085`, -2361.567001516277, 
      1750.1695462453058`}, {5662.568036890699, -2362.911039265667, 
      1743.4988781471175`}}, {{5527.768978175408, -2579.8612702157116`, 
      1864.3764749541815`}, {5526.665170966072, -2583.7995584981545`, 
      1862.193765303518}, {5523.7262545605245`, -2593.477080511843, 
      1857.4548682373206`}, {5522.22197934372, -2602.9881564443695`, 
      1848.6023797083742`}, {5519.803259391414, -2608.65688895393, 
      1847.8351937673228`}, {5517.579410543715, -2613.3572727468672`, 
      1847.8351937673228`}, {5516.040093922082, -2615.6109391102873`, 
      1849.2416790270186`}, {5514.279521993274, -2612.866668176159, 
      1858.3494874914252`}, {5513.479078069549, -2613.5479704640584`, 
      1859.7658862804594`}, {5508.962250682988, -2621.4230603106416`, 
      1862.0659895453246`}, {5508.265695327777, -2622.152838000857, 
      1863.0988202023123`}, {5508.218995758458, -2621.706099130444, 
      1863.8654228639796`}, {5509.768222658632, -2615.617533893075, 
      1867.836379068641}, {5510.2655800376515`, -2611.812674434866, 
      1871.6894724055371`}, {5512.4293270262, -2604.112762657085, 
      1876.041930210505}, {5512.906640084675, -2600.51430553669, 
      1879.6274669224558`}, {5514.224965876123, -2596.8782837039753`, 
      1880.787041463007}, {5514.597173726491, -2595.99530133685, 
      1880.9146968868106`}, {5516.287209286488, -2592.957729144045, 
      1880.148751970025}, {5518.2637168691035`, -2589.2199077952537`, 
      1879.4998031824111`}, {5519.497723391059, -2586.402757369514, 
      1879.755129838004}, {5520.736330326813, -2584.222143431205, 
      1879.1168070158544`}, {5526.112752831703, -2579.511152455583, 
      1869.7630190342547`}, {5527.768978175408, -2579.8612702157116`, 
      1864.3764749541815`}}, {{5581.929664281153, -2494.813592716295, 
      1818.1804230098014`}, {5583.889749875892, -2492.0424611391086`, 
      1815.9607162906038`}, {5584.773643343367, -2492.074594575252, 
      1813.1964200521222`}, {5585.134754928829, -2492.235732172835, 
      1811.8621676515602`}, {5584.575736138179, -2494.534637352805, 
      1810.4210781980576`}, {5581.483469551356, -2501.360888481916, 
      1810.538504063889}, {5576.362567268481, -2511.640369945377, 
      1812.0863280836015`}, {5574.157902091762, -2516.852222949269, 
      1811.6380047855312`}, {5573.598581725244, -2518.96567253437, 
      1810.4210781980576`}, {5576.057615260118, -2520.991229806448, 
      1799.9995501495596`}, {5576.104565411527, -2521.2000015388076`, 
      1799.5616460231215`}, {5575.008358174079, -2523.6230658423833`, 
      1799.5616460231215`}, {5572.729865084735, -2527.1518495756472`, 
      1801.6656376750514`}, {5564.344161963126, -2537.0493334816256`, 
      1813.634036046773}, {5562.318848179497, -2538.870355549911, 
      1817.2947033700884`}, {5561.916323766798, -2538.3230122677187`, 
      1819.290186531152}, {5563.987136972786, -2530.6723793769534`, 
      1823.6112902315463`}, {5565.180256022104, -2527.570271950988, 
      1824.2727121599337`}, {5566.584796875065, -2524.552569175631, 
      1824.166032648373}, {5569.807628185893, -2519.441850212377, 
      1821.3921704017155`}, {5578.027316943192, -2501.9984986342133`, 
      1820.2825204532612`}, {5581.929664281153, -2494.813592716295, 
      1818.1804230098014`}}, {{5617.189825350333, -2551.085325922445, 
      1620.7886400897125`}, {5614.035336229816, -2556.5583812127275`, 
      1623.0927813941962`}, {5612.635130454299, -2559.5762927411997`, 
      1623.178913129258}, {5611.500869188989, -2562.1711693953207`, 
      1623.0066493427046`}, {5609.882726911288, -2565.6576746106302`, 
      1623.0927813941962`}, {5603.959929088448, -2573.387281434501, 
      1631.2954063939958`}, {5602.183536611959, -2573.838481120512, 
      1636.6761393804113`}, {5601.776208273319, -2568.2894655304012`, 
      1646.756273304753}, {5603.055202145051, -2564.510111618302, 
      1648.2942635108466`}, {5605.803553299893, -2558.718557669093, 
      1647.9501067842093`}, {5607.35487918497, -2555.2060992387323`, 
      1648.122185790148}, {5609.10356368954, -2550.64924947272, 
      1649.2299136171223`}, {5610.006911333411, -2548.661773720048, 
      1649.2299136171223`}, {5611.144503086289, -2546.6506987706425`, 
      1648.4663399461722`}, {5614.344447791413, -2543.4602568670834`, 
      1642.4859309264618`}, {5617.56207794519, -2541.255649546022, 
      1634.8791129874203`}, {5619.176945704633, -2542.1279518249403`, 
      1627.9587277974454`}, {5619.567931509923, -2544.7037407506546`, 
      1622.5759843393778`}, {5618.944527463544, -2546.9508943173737`, 
      1621.2085705495813`}, {5618.382665012535, -2548.2448825247466`, 
      1621.1224315803781`}, {5617.189825350333, -2551.085325922445, 
      1620.7886400897125`}}, {{5632.442091970896, -2370.367911446079, 
      1828.8700705575718`}, {5635.493186166388, -2368.815892033395, 
      1821.4668562239171`}, {5635.55650268994, -2369.4443645988817`, 
      1820.4532397344694`}, {5634.88505012648, -2371.245517155817, 
      1820.186490233772}, {5633.605402043469, -2374.2841036483187`, 
      1820.186490233772}, {5632.885563632735, -2376.064947769266, 
      1820.0904595651696`}, {5631.796532802232, -2379.624268416515, 
      1818.8100077468016`}, {5631.526536752071, -2380.2631601521084`, 
      1818.8100077468016`}, {5630.900925068491, -2381.1803390014766`, 
      1819.5462772943217`}, {5629.518470048853, -2381.649798059884, 
      1823.2058920429522`}, {5626.723699161962, -2380.4674295227173`, 
      1833.3490993999549`}, {5625.850362397219, -2380.838875947205, 
      1835.5455964626772`}, {5623.182201768066, -2384.1777861137125`, 
      1839.3835651980685`}, {5622.723872037169, -2383.983458502778, 
      1841.0358004146683`}, {5622.007132822173, -2381.2952368854308`, 
      1846.6950163151944`}, {5620.17903018029, -2379.9309220105433`, 
      1854.0038201632108`}, {5620.417487355797, -2378.088630468529, 
      1855.6442071317092`}, {5622.265065324651, -2372.925930777189, 
      1856.656067308968}, {5625.297438536032, -2365.870450255531, 
      1856.4750013419623`}, {5626.76851269225, -2363.8082262257826`, 
      1854.6429479019307`}, {5629.722965956356, -2365.3499542038176`, 
      1843.6790964621707`}, {5628.996391361335, -2367.5764552443316`, 
      1843.0396210102906`}, {5628.22539105043, -2370.8252946220537`, 
      1841.2170051125104`}, {5628.238286535386, -2371.7213717061263`, 
      1840.023156096041}, {5629.179160101044, -2371.9674519239156`, 
      1836.82499996534}, {5630.328689915765, -2370.3692262713325`, 
      1835.3643411206594`}, {5631.752686123665, -2367.6946997562923`, 
      1834.4473775731076`}, {5632.217757297643, -2367.439102513615, 
      1833.3490993999549`}, {5632.319062563313, -2368.8235834863444`, 
      1831.2483441097847`}, {5632.442091970896, -2370.367911446079, 
      1828.8700705575718`}}, {{5660.352491048922, -2417.0543599696753`, 
      1675.133304554549}, {5661.982485856961, -2414.8649926207, 
      1672.780645292159}, {5662.497783481149, -2415.084769738792, 
      1670.717838848561}, {5662.931714066686, -2418.634862411148, 
      1664.0984027836637`}, {5665.429136049621, -2417.538833361251, 
      1657.1760267365582`}, {5665.327558789958, -2421.1080119933104`, 
      1652.305460826423}, {5664.472919790987, -2424.1107810072926`, 
      1650.832260515518}, {5662.4215071102035`, -2427.7943753209656`, 
      1652.4560015249122`}, {5661.625460867355, -2427.9443938213817`, 
      1654.9612839489096`}, {5658.41437566368, -2432.813377027344, 
      1658.788568757153}, {5657.875740423643, -2433.0615182879387`, 
      1660.2612578001065`}, {5657.684905240283, -2430.581202259445, 
      1664.5390419690584`}, {5653.640442359949, -2433.4152658997673`, 
      1674.1127746355555`}, {5653.014641502737, -2433.1459114236773`, 
      1676.6156764977225`}, {5653.247987672803, -2431.7963425967237`, 
      1677.7864667491474`}, {5655.304887821678, -2426.9050825776626`, 
      1677.9368390791558`}, {5660.352491048922, -2417.0543599696753`, 
      1675.133304554549}}, {{5536.04478668893, -2572.624311434116, 
      1849.7637584276697`}, {5536.653939177921, -2571.8264613005326`, 
      1849.0498913604229`}, {5537.114415941485, -2571.394174506754, 
      1848.2720671172076`}, {5537.57647382391, -2571.6087507025672`, 
      1846.5884543833617`}, {5539.930177164252, -2568.8239360964435`, 
      1843.4019929149574`}, {5540.108193808812, -2568.90648107498, 
      1842.7518504626237`}, {5540.125594996057, -2569.2435112869716`, 
      1842.2295897578492`}, {5539.227831835082, -2571.8350233704227`, 
      1841.312936355164}, {5539.141211525112, -2572.4411767995516`, 
      1840.7266827778224`}, {5539.834214256536, -2572.7630152748447`, 
      1838.1896082250473`}, {5540.028505462332, -2573.8289440322064`, 
      1836.110676273861}, {5539.301459914307, -2576.1834807082996`, 
      1835.0018255895252`}, {5534.065384602975, -2587.4122804497606`, 
      1835.0018255895252`}, {5533.212690859032, -2588.9556444265727`, 
      1835.396327474898}, {5532.551959783109, -2589.399817470488, 
      1836.7610317033293`}, {5527.347265041677, -2596.9905786078875`, 
      1841.707315579258}, {5526.336787830402, -2596.833730674404, 
      1844.9579867043083`}, {5526.226362004125, -2594.521437512335, 
      1848.538448661424}, {5526.981190583911, -2592.4039837001947`, 
      1849.2523338438723`}, {5532.627459323866, -2579.965333942123, 
      1849.7637584276697`}, {5536.04478668893, -2572.624311434116, 
      1849.7637584276697`}}, {{5598.202116470243, -2587.360968928289, 
      1628.9597814601073`}, {5598.5291551872415`, -2587.2156623058063`, 
      1628.066370034697}, {5598.8508865690355`, -2587.1627470222543`, 
      1627.0437487619643`}, {5599.564043679404, -2585.9980947763893`, 
      1626.4409194759105`}, {5600.93730752849, -2583.25286400826, 
      1626.074908400172}, {5602.124029937649, -2581.322411152306, 
      1625.0522000471271`}, {5603.068763817713, -2580.667099626514, 
      1622.8343842904967`}, {5602.779610337291, -2582.6203614187093`, 
      1620.7240347370073`}, {5600.305193555759, -2589.03286961817, 
      1619.0442332033724`}, {5598.518059905217, -2592.975781699378, 
      1618.915012727852}, {5596.63963240352, -2596.389691039821, 
      1619.9379887120142`}, {5594.829536180774, -2598.6358702906055`, 
      1622.5867510608514`}, {5594.495411504662, -2598.077086157862, 
      1624.6323384183731`}, {5595.967835382455, -2591.2015641206704`, 
      1630.5312414865257`}, {5596.747380764799, -2589.368279305804, 
      1630.7680276566784`}, {5597.469000550684, -2587.923269848316, 
      1630.585056736302}, {5598.202116470243, -2587.360968928289, 
      1628.9597814601073`}}, {{5549.663221310298, -2563.646220391097, 
      1821.2001198962882`}, {5550.518514727696, -2563.112754958381, 
      1819.3435390367326`}, {5550.577669610373, -2563.91583477183, 
      1818.0310271909407`}, {5549.522084939622, -2566.660861817133, 
      1817.3800755988061`}, {5547.838964895352, -2570.032772243561, 
      1817.7535749333445`}, {5545.8926777631905`, -2572.5178484042926`, 
      1820.1758201816606`}, {5545.2482827496215`, -2572.5247245541, 
      1822.1283473510368`}, {5545.324794355144, -2571.17250914924, 
      1823.8033181528288`}, {5546.102928205158, -2569.228554382183, 
      1824.1767006245348`}, {5547.527015008957, -2566.3493403128296`, 
      1823.8993314384736`}, {5549.663221310298, -2563.646220391097, 
      1821.2001198962882`}}, {{5555.080855575518, -2555.5333443597724`, 
      1816.0781106229188`}, {5558.34566987544, -2548.4244937398544`, 
      1816.0781106229188`}, {5559.712111316047, -2545.9748876751055`, 
      1815.3310443589598`}, {5560.101231431304, -2545.847865603725, 
      1814.3171253772175`}, {5559.714482612798, -2547.361233944997, 
      1813.377871713374}, {5558.243608893979, -2551.7596576748824`, 
      1811.7020515670308`}, {5556.818467783181, -2554.793489406855, 
      1811.798121366762}, {5555.917071546477, -2556.2235158107483`, 
      1812.5453156582187`}, {5554.890234973959, -2557.1374032833733`, 
      1814.4025099520006`}, {5554.704128771191, -2556.7462573967373`, 
      1815.523149700949}, {5555.080855575518, -2555.5333443597724`, 
      1816.0781106229188`}}, {{5593.804792225107, -2589.0025771317282`, 
      1641.410153521874}, {5592.462794052669, -2590.8707084483044`, 
      1643.0345581396627`}, {5592.376785727223, -2590.3329519310487`, 
      1644.1748005000638`}, {5593.560590696473, -2586.708758183213, 
      1645.8527906667673`}, {5595.2170665171825`, -2583.007227333346, 
      1646.0356411993498`}, {5596.429436712885, -2580.6881077481567`, 
      1645.5516219252263`}, {5596.953318429572, -2580.929685942148, 
      1643.3895452821776`}, {5595.725222534127, -2584.924390008251, 
      1641.2918149542625`}, {5593.804792225107, -2589.0025771317282`, 
      1641.410153521874}}, {{5591.064465371425, -2594.7994388162765`, 
      1641.5930392090704`}, {5591.626862394171, -2594.1590473930855`, 
      1640.689354686164}, {5591.585727908678, -2594.9346059256413`, 
      1639.6027349384096`}, {5590.950060217896, -2597.0237797628756`, 
      1638.4622673396057`}, {5589.758679537086, -2599.6616731662457`, 
      1638.3439137160055`}, {5588.664822366961, -2601.632905802254, 
      1638.9464349651516`}, {5588.155006777847, -2601.893997807622, 
      1640.2697748834944`}, {5588.585300808248, -2600.1007124615085`, 
      1641.6468288420451`}, {5589.310974837121, -2598.350077004708, 
      1641.9480484762425`}, {5591.064465371425, -2594.7994388162765`, 
      1641.5930392090704`}}}], 
    Line3DBox[{{{897.7312763749798, -5950.862847262366, 
      2114.0301839119024`}, {897.5779484156325, -5950.6209987225775`, 
      2114.7759328306865`}, {897.7782681641102, -5950.329441337479, 
      2115.5111464524734`}, {898.7285516744861, -5950.010460879331, 
      2116.0047721665487`}, {900.0912476847451, -5949.890374182315, 
      2115.763212425707}, {901.8639767358317, -5950.09237620338, 
      2114.4398246610463`}, {903.7604371001017, -5951.877138320408, 
      2108.5988932633677`}, {906.1742061943794, -5952.09434084728, 
      2106.949205766898}, {908.401037324437, -5951.989208714808, 
      2106.287184770955}, {910.7062319213835, -5951.491876492107, 
      2106.6970102955056`}, {912.5863059605798, -5950.649415271368, 
      2108.262664068157}, {913.3781467144223, -5949.298418798898, 
      2111.7297116774807`}, {914.8644836990288, -5948.458231231185, 
      2113.452468773326}, {930.9843774950075, -5945.956646419158, 
      2113.452468773326}, {935.9929015596124, -5945.957760327665, 
      2111.2359628021763`}, {937.0538217343757, -5946.346262149896, 
      2109.6705799221263`}, {935.8304602601175, -5947.358490905279, 
      2107.359015523036}, {930.1608342469472, -5948.713046868262, 
      2106.045488220124}, {924.2110933615738, -5949.844883072591, 
      2105.4675044470073`}, {919.2607472852178, -5951.804352407351, 
      2102.093794093295}, {917.4309945152019, -5952.146063541131, 
      2101.925616803474}, {913.6104470034768, -5952.473835367836, 
      2102.661380343355}, {909.4501802506334, -5953.08118259039, 
      2102.74546560688}, {903.6222826674036, -5954.258069792766, 
      2101.925616803474}, {901.0251476882709, -5954.447538040052, 
      2102.5037193693847`}, {899.6203367362526, -5954.225605976424, 
      2103.733436741745}, {897.7312763749798, -5950.862847262366, 
      2114.0301839119024`}}, {{1057.6505534372168`, -5912.985834171405, 
      2146.1098577897274`}, {1055.6483920621254`, -5913.343609214199, 
      2146.1098577897274`}, {1053.957133912234, -5913.474052544864, 
      2146.5816266226925`}, {1053.4976294064918`, -5913.2856954612125`, 
      2147.3259460754225`}, {1054.2291624079273`, -5912.610744749619, 
      2148.8249682559162`}, {1061.412855630655, -5909.795910835446, 
      2153.027805713296}, {1064.5482660733128`, -5908.995159020854, 
      2153.6775268986803`}, {1067.1797247400093`, -5908.589229197234, 
      2153.4889007650513`}, {1068.299858374772, -5908.8946338974365`, 
      2152.095097363155}, {1067.1817057317887`, -5909.779603471555, 
      2150.2190435727657`}, {1064.145259579735, -5911.074141755719, 
      2148.164576372318}, {1057.6505534372168`, -5912.985834171405, 
      2146.1098577897274`}}, {{1033.772222847239, -5917.27184758472, 
      2145.9316305630045`}, {1040.6473668933247`, -5915.084855838318, 
      2148.6362875112595`}, {1043.2577763344077`, -5914.556463329921, 
      2148.8249682559162`}, {1044.8324893943316`, -5914.552648774044, 
      2148.0702325543216`}, {1044.6345976986345`, -5915.059542861511, 
      2146.7703304485076`}, {1039.134142634163, -5917.012976204687, 
      2144.05488806544}, {1033.8446808364797`, -5918.478699426236, 
      2142.5659267205874`}, {1032.4483101600072`, -5918.589579172009, 
      2142.932936459805}, {1031.9675904043906`, -5918.335449859177, 
      2143.8661537733938`}, {1032.0555812053885`, -5918.046586336854, 
      2144.6210782442495`}, {1033.772222847239, -5917.27184758472, 
      2145.9316305630045`}}}], 
    Line3DBox[{{{6093.65582504921, 1689.920323339351, 836.2665105147113}, {
     5986.790930181547, 2012.839211876432, 891.4181709100569}, {
     5774.638867623051, 2423.4824344877284`, 1211.6486818655826`}, {
     5770.340617289117, 2435.509518551165, 1208.0050643561553`}, {
     5781.355146301957, 2515.8046932428283`, 966.6814924188999}, {
     5638.860327323834, 2929.1533964182618`, 557.7887406399523}, {
     5884.879700590206, 2417.5655863411644`, 459.7621508130301}, {
     6001.275246090117, 2089.865151793897, 552.2433865457075}, {
     6033.178842113445, 2034.2972576817135`, 387.558274925628}, {
     6102.087653908535, 1817.1787302171754`, 387.558274925628}, {
     6120.841723438614, 1778.268906464187, 246.7184284071085}, {
     6143.4309835600925`, 1579.2699827621407`, 672.2914262284892}, {
     6093.65582504921, 1689.920323339351, 836.2665105147113}}}], 
    Line3DBox[{{{5492.991048529791, 2445.6371829484765`, 
     2129.2538129944546`}, {5623.045533650187, 1612.3823606858646`, 
     2543.500685953468}, {5671.553562056526, 1519.6881967829033`, 
     2492.3566492945306`}, {5748.901189273992, 1648.47088920764, 
     2218.2183241058697`}, {5881.729410247841, 1629.305564399842, 
     1854.302082305173}, {6022.180390066588, 1460.6279081235198`, 
     1512.5221167004963`}, {6026.811542878753, 1510.1034930589688`, 
     1443.933432286449}, {6048.234643111922, 1698.1838731496737`, 
     1105.8222844721363`}, {6103.27547937817, 1516.0615077747616`, 
     1067.4266332937787`}, {6093.65582504921, 1689.920323339351, 
     836.2665105147113}, {5986.790930181547, 2012.839211876432, 
     891.4181709100569}, {5774.638867623051, 2423.4824344877284`, 
     1211.6486818655826`}, {5774.008918790078, 2313.3362828843715`, 
     1413.191401765296}, {5659.494531174293, 2394.5451134102855`, 
     1709.9967957429596`}, {5609.828546255932, 2497.6565884558254`, 
     1726.0790647462413`}, {5492.991048529791, 2445.6371829484765`, 
     2129.2538129944546`}}}], 
    Line3DBox[{{{2030.9412650504535`, -5703.54569018256, -2008.14786007286}, {
      1708.6080006144452`, -5091.72017772365, -3441.3453796441518`}, {
      1424.9117824555556`, -4867.561732155002, -3868.369538433411}, {
      1455.0901698262446`, -4559.400061047707, -4216.917994340557}, {
      1360.5191702482362`, -4341.421951753621, -4470.889924189459}, {
      1229.1472259811962`, -4212.451552193408, -4629.499185364015}, {
      1213.3213864044296`, -4278.285035051032, -4572.957693020134}, {
      1081.8626069035215`, -4244.3160956563215`, -4637.1508353661075`}, {
      1165.6778575964165`, -4208.051508389117, -4649.872183732957}, {
      1232.452091623989, -4086.3738653583214`, -4740.2978982726545`}, {
      1129.4857598717963`, -4114.593665948352, -4741.53926192587}, {
      1077.4115377966452`, -3938.3532787836907`, -4900.650111946007}, {
      1110.8333755723197`, -3762.1635996984746`, -5029.909338230504}, {
      1188.8043316287085`, -3762.771108826962, -5011.597702697009}, {
      1093.4890058949413`, -3679.761941925557, -5094.256327938667}, {
      1230.5233692001964`, -3670.548646199067, -5069.585640241064}, {
      1102.4658285763758`, -3655.3855358456995`, -5109.847294308456}, {
      1246.7089288954837`, -3644.798132470967, -5084.188933364733}, {
      1167.271424037839, -3621.1434334273567`, -5119.825323892467}, {
      1431.1509225774814`, -3620.8276674243257`, -5052.645616864509}, {
      1218.3451133028361`, -3734.8882702802234`, -5025.341628527785}, {
      1169.9956488031064`, -3971.0390609573114`, -4852.810798715839}, {
      1472.8553548764696`, -4301.858463403055, -4473.536023601378}, {
      1447.8440616699952`, -4491.544047576324, -4291.572361241112}, {
      1819.52900118743, -4949.777638087551, -3588.429912555949}, {
      1813.0596946616815`, -5138.91754400682, -3315.4418573535863`}, {
      2098.971439714925, -5278.951632869642, -2900.8278547828363`}, {
      2280.0012724481317`, -5419.495201581031, -2473.5560732829726`}, {
      2130.477608373321, -5698.221203685099, -1918.1120832979068`}, {
      2030.9412650504535`, -5703.54569018256, -2008.14786007286}}, {{
      1423.0593775420803`, -3360.6687517244372`, -5231.502770636031}, {
      1350.506347098892, -3405.200374422674, -5221.930362086159}, {
      1423.0593775420803`, -3360.6687517244372`, -5231.502770636031}}, {{
      1347.965815894495, -3404.573018180102, -5222.995731671615}, {
      1341.1262213930033`, -3333.3541126867754`, -5270.478584315307}, {
      1244.1437263595167`, -3424.454316854883, -5235.745668950528}, {
      1348.9561089973554`, -3404.1807667835137`, -5222.995731671615}}, {{
      1336.856047915205, -3417.1050783906235`, -5217.664466060418}, {
      1143.1478758228418`, -3521.732799337518, -5194.075924942958}, {
      1328.5239164314496`, -3460.9231268797503`, -5190.842802792099}, {
      1293.6215433860555`, -3636.258144054662, -5078.580874839311}, {
      1410.96835253236, -3603.45201241123, -5070.71154695291}, {
      1336.856047915205, -3417.1050783906235`, -5217.664466060418}}, {{
      1285.5964956309906`, -4454.053541532491, -4381.376768731854}, {
      1349.4863338150049`, -4555.789512519538, -4255.75871525437}, {
      1285.5964956309906`, -4454.053541532491, -4381.376768731854}}}], 
    Line3DBox[{CompressedData["
1:eJx1lXs01PsCxZFndCo6odXJI02RSimSR5uuxCQm6SBKKnqI0pVHilRC8s6z
MMPMMF6/mTHDUeqrHHklOUQhSqdIeURJlnOvv+5/94/PP3vtf/Zae6+t5eW/
/4SUhIQEY54F8wTXTzhcuc4m3WvKVkkElcNlXflAsRKFZkvGijGtArI5xCGs
n0nBSVum3WN9OVJ+6ejs1y0gm8bmbqzeRsFG0mQHTZoCM3RQy6c3n5j70lMV
JstR/59eY5/fKSjbSTV1L8gnU0eSLc7GU1ic/npftRSFjwOt9u2mLEIUmPZP
7fnYwhw16+kvg1xSz0dnvXyiOeuz/ZkzH0Zc7SuUaRmWDmpkrz3GJuF3aPLG
hRQ62NFuMf8qg4FFTrrrZD7xe/K5U9DIx/lurXvhzBKc3bzWuOwEhyQ5REX8
OcfHT5c6g+ltPIQmBO7Sai0gj1616RfvFWDhl/dCNx4PnQ1134aSOGSZh3SB
g44Avd6BipsKiuD06uDbL0w24VRV0jRyBWhI6ZHdsbQIKrGGX91mOeT+Gcm4
o3QBxpm6lEFHIa4emWm8F8Ahz83Sh4ZXCNFvlHehrImDyueWfznWskmeZ8j4
wltCnJd8UR3ynQ3pRervpnoKiO9bT5+xfRU4khqm2/Y9H1cDRq0spllEb8hz
82NpETSMs3Ik37FAN3mk7rKVRZzn1B6J5EUY1kxIN7ydDw/55SGf/FnE2lh2
NIsmQnV6ZnBHGwsF4Wy1C6pMsiNMuGRTnAjJb16s+8WGhSmH01SeVh4Ji3l6
cvahCH88pN40bmMhyHnkg0tPLjl90HzZylkR7La7nlE2Y8J9PNAsjZtLsqxq
DOpHRND22rrmfR4TC0xXq7woyCXJy3v+dmkWwYw24EIDC/GVfWuac+4S81Ce
9MxCMWp4M4nf7Fl4NuVv4tl9l+zUD+Qkz+tGVXQtCxMWTp/dUhUZmUGMf6ua
/LhUjFFbbf9dOflw1VV+ZXY4hdy8AOWiG2Io0cuyg+ZYULUcdk/9nESaM4+a
rnsgRm/6r6tOMVgQy31r5dDiSKh0+Ju0YDFCx7/pmrkWIGVdX++HdzeJ/2L3
ps8pYpB28+sHjQsQsfwPJmfjLVLqsmLYWVeMKJlDak9r2Di4NKb52kg0CU8/
u24XEeFonmafXzgX3wOO/QwIjiKZY36MyHoRdlRrLWtK4GKl+lyX/fFI0tD9
KdouWYR+pQdH71wrxOIi1UfZjsdJ/+jQwz4FMZq/Jsc4GnMRlsro19viYv5t
SLP+zJgIEr7Rvt75XDxRubKa0+iGxse2dYe0xdgdXqsT8oaDkHtis1O13nA9
5XpRWVqMsCd1XyTpXNwwsiznfIiF4bgjx8la9L9+9sdyDnj9nYjaMEWjoZQK
lD/X+fqshQdPeuTl0Tvx8DI4MNOWV4GShsxO6h0PsjdTsz//lQC0J+rY2VRg
0j86hN9VjNuOdeLtj2+hIrnCIXV5BT6pE79kbgn0qi5b3ddIAt1x4KVChxDi
oOmbVuMl+G43Y0zXvwM9HHA7qS8E83Sxz40FZdjrHLhopjgVa9VM1EzXChEQ
3zAps7YMzYrtLdfk7sDNTlYh31mA9ZYDF07JU5C5f+CyiWwMLp0PazhmJsC0
U1sUg6IQID91IqErGgX1fkGtR/iI+tVS8lYaH5LuO1bt3xwJhQiX+2p+fOSp
l058LOTjYY1e7BLfCLzR3Wmd1UJB7XB5W4StABdtEkVHj/0b1sKgwMK3FOas
KjUt9szvvih4sMf2JGrlQr376RQiag+1uncKkPFbrWKX6BgmeIOzLTYUHFse
zBm/FkDJP05krc4g3TuR0CzJB0ODzbPdJkB9B9/Q0JtOBDre8qo3+Ih50qOy
p5KPICf/6J6+UHLp3E+NyCd8ZCuv3+B0nI9QhmWOtU0E6dIqppkZCFAwu/tz
xBCFbdnPB2i7I8hUxq662Y0CxDe+ru8bptAWnLhnMDaF/LiV9aNnnI/GlGZ9
lTQK6oe/WAWbZpKR203jg1v5MDG6srN8Pt8HmsK5DVNZZPaC1GhzBYXqHF74
qBEf3MfOvGq3LDL5e6d3+T4KdRb/3Lbl8GGvra+mqsQkV0pR7sUuRZ6uKxFo
CCCidY+Nk1wiE6VZlb+/FGm8xbv2XhdApsdHdUnfXRI4WXy5S6cMPzMO9WVc
EkCqOc/4/Ye7JJrTX2ooLkElTSnpzEIhzAdWx9fsziXKdgFhpyaKQRtf45Og
KsRUrPnLlo05ZOCSR0nuCh6G2UYVqS+F0Bkx8fnzai4ZXLXBM2dDEWIH+oQK
UhXQ713SOH6eSZzqqvmaboVYrSJXlTUshITioqaEjHzCCpJ36rpaBMVq9TRZ
eSG8foiuC93Z5MfGWca5vkIEyaVFOtYJUMG7qDP9D5vQVgq6LjOK8JEeF/Xu
rAAWW9Ys7HbhEs8fvntrErmIPbQibmXGvN/xxfQihUJiOnF9QVDx/E9Ue7Rk
cPl4OZIZ76/LJeceHGhfyuchXNfbY08SHyc223nrTXCJ+dNlG2XGivH8MGMk
tpXC//vf/wKJdsxV
      "], {{-2007.6295067778913`, 5648.475240369014, 
      2179.900206994112}, {-2143.632991746568, 5598.9540308471505`, 
      2178.156341302454}, {-2118.189133595204, 5655.322010916938, 
      2053.8796062957554`}, {-1937.7442992943922`, 5730.307172836659, 
      2023.9912118111613`}, {-2009.272496660344, 5647.891005650811, 
      2179.900206994112}}}], 
    Line3DBox[{{{-1695.924976651886, 
     6040.729513491541, -1149.4516555299533`}, {-1693.495603756583, 
     6041.39436039068, -1149.5392608766383`}, {-1691.308238030883, 
     6042.111233057971, -1148.9917237837524`}, {-1689.4499121467288`, 
     6042.785141419668, -1148.1813528293728`}, {-1689.3778871749098`, 
     6042.934248802632, -1147.5023786348318`}, {-1690.901338117301, 
     6042.689001198669, -1146.5496018689066`}, {-1694.5046741110766`, 
     6041.748109345047, -1146.188196868763}, {-1695.9560238195195`, 
     6041.245266953994, -1146.691972491412}, {-1696.745920354166, 
     6040.817565355618, -1147.7761601690977`}, {-1696.898783936974, 
     6040.5518690922445`, -1148.947920438263}, {-1695.924976651886, 
     6040.729513491541, -1149.4516555299533`}}}], 
    Line3DBox[{{{-577.1164354395528, 
     6206.9919429548, -1352.0359943580422`}, {-576.4853177900804, 
     6206.785042061854, -1353.2545252130876`}, {-574.6094608764045, 
     6206.940009080708, -1353.341561153177}, {-572.8705253425317, 
     6207.27622650919, -1352.5364686423818`}, {-572.828939573789, 
     6207.5359834769715`, -1351.3614282734875`}, {-574.4618996997051, 
     6207.588711225853, -1350.4257136380156`}, {-575.4053461306747, 
     6207.510798629403, -1350.3821912864464`}, {-576.8526698349914, 
     6207.21073083952, -1351.1438229348323`}, {-577.1164354395528, 
     6206.9919429548, -1352.0359943580422`}}, {{-739.9283433125429, 
     6200.991019974065, -1299.2421279206378`}, {-740.0791944142244, 
     6200.783404939427, -1300.1467767229249`}, {-738.9039949584545, 
     6200.77726055038, -1300.844318616092}, {-737.4642284208123, 
     6201.037813184027, -1300.4192559561543`}, {-736.4944699775368, 
     6201.374679831038, -1299.362022703067}, {-736.9295668311784, 
     6201.526170599657, -1298.3919511832935`}, {-738.278385526097, 
     6201.42278834379, -1298.1194535462903`}, {-739.4830936342007, 
     6201.21535431535, -1298.4246507336286`}, {-739.9283433125429, 
     6200.991019974065, -1299.2421279206378`}}}], 
    Line3DBox[{{{1999.7900154829156`, -5913.789076330503, 
     1309.8673003968195`}, {1633.9232525751002`, -6041.611530074788, 
     1231.681214585082}, {1448.066850260756, -6149.720038588862, 
     878.5553677227747}, {1379.162115159291, -6153.199214306842, 
     961.1789527257778}, {1328.2525931859225`, -6186.96350137115, 
     803.1446765036173}, {1427.9363610424964`, -6201.530188624013, 
     435.7003080073898}, {1236.7553588223084`, -6255.6208288132075`, 
     159.55537766762353`}, {
     1620.4380188492848`, -6169.430312032499, -11.132926047555543`}, {
     1870.3597514067849`, -6092.30864838198, -270.8199091050806}, {
     2168.886292762953, -5991.4822398823935`, -293.064000243695}, {
     2101.9101676035007`, -6007.714958972454, -420.88990135620264`}, {
     2180.954242568368, -5975.891334026496, -469.0149738868202}, {
     2176.299494772346, -5995.608131642192, 64.94097966344593}, {
     2273.644856373651, -5959.2882481990955`, 72.36250383864133}, {
     2196.3425076895287`, -5985.5976425584995`, 191.08673505559233`}, {
     2505.4487959742323`, -5864.485987236952, 135.44048970937934`}, {
     2402.141141748889, -5900.982106492706, 309.74474002028967`}, {
     2431.8569187549574`, -5856.5611599494005`, 688.8954902420079}, {
     1956.3642625737734`, -6021.070083641306, 779.2096059494136}, {
     1800.7185076611117`, -6033.978581323916, 1018.0024687057293`}, {
     1999.7900154829156`, -5913.789076330503, 1309.8673003968195`}}}], 
    Line3DBox[{CompressedData["
1:eJx1lGtQ1FUYxhekQYzV0kQkIMCEZd3pAgbDRT1AEBgbwkAQa4aEBBWs1ESQ
UQFL4Mj9fxgYBGLQQFDhHHB3WW57uC25XBzcRTQc4uIkCRqCDcbN3m9+6sP5
8ptnnv97nv/zHttoacgpQx6PJ4KzBc5yXN2Gyw2KSt5TWHpdI2jZ7cXs8AmO
rWlvb30MPFHQEdYMfKHiYfsjLccM/XitDaMUtQXEnj0IPMlRYEpaObbSkzs4
oqNI8SxNHNtCUMMxJ5/zFzmmHd2he0FP0bUxgfU48At+lf5nznNsKlj54M5N
ipyVXWGDcoK2fZwumiri2JBVvtQF/Gd6tvprlARF3S13IPkcW84hondgnmbv
N/lTKoIG5xytDxRwrMuitzhukCKvRNVQSSdBv7vX1sk58K/3Kze4TpGsYSTO
lz33fxwxHjgzQNF9G/Ge0O7n+i+1KbLNfoo8fSJ2jvUQxKIMCgrLOZb+MpU6
A88LSb8/DPyjr/2vnK7lmGdYovF8H9y3zbLiLPBv+Fd1wkaO3U4165/ooag8
VhQ83UtQvPCgslzFsaZYXd0J4CNSYeMl4N85zWou9nKsc2GTrHdTVLy36v1R
8LE1OqfI0XMscvKn8W3A/yiZircEvdWxI6F7JjkWfidBeERNUV1D6domcPdD
Ho+085B/eJqRdxdFb9vGhEnAx6x2art+B2aXXT3GRoGXbfhE5kAOZfxPv3e3
waw7z566A18Kt2/OUBO06FO7ZO2CGb/mk8CsDopUrnbYFvQBLc1Vd30xi7lF
klbaKMrpeD3aCHK7xJtUzQZh9vdvmfEyOUUeT6cdC/sImh/OkPITMNuy+1B7
RAtF4qyjpUX9BM1WPAmdT8PsXTuTJCPgxobRKi/g/KRQlPszZrEigbAX+MDR
fGcz8FlU9NkVnMMsLbdG4qmkKGNpxi0I5lGrAi2e5GIm38wW3IJ5JOM7cxM6
gB8XHJ/Mw+z6Z/sXg2B+G/tEoUE7QdWy1XET0GvXfxiu7qTItMriQ7s2gjZk
fV84ZWF2L1mgfwZ5mn8eUVnWCr2tkRUnpGCWn8XzSIX8uxdOKg5AD0+6NT4t
S8YsxKpw/jD89/piuwQH6L97qt4381vMXimR/KOAnmwXqVfONBO0T6ANnAOf
GDcHzZIGfApeqgmiBHklS1t2gT7zV4s/A4YpGjvdpJFfIWi3Q9O/+kjMxB/c
qPQeoaio4q+Zq8BTsowlv4gxM1pa5y2C3rrMYqS1Efo28fAtL1fMokoPT2YP
UaTWcfeC4bsXTpz6KvVVzG6aD+xbBV5tOo1dYe/e8Nxb72SC2a4f1Q9egz3a
f3l1NgJ44Jq5RDcH+/U/78B/KzwxhQ==
      "], {{4464.395077199569, 4349.659002878108, -1355.528252511794}, {
      4469.264358862665, 4345.577819896187, -1352.569107966832}, {
      4469.947739758976, 4345.301399458667, -1351.1982244238495`}, {
      4469.671857778913, 4345.974041312038, -1349.9469641512271`}, {
      4468.245607936102, 4347.440404552847, -1349.9469641512271`}, {
      4464.834812308721, 4350.801415135873, -1350.4039524704583`}, {
      4461.667413352731, 4354.612626378362, -1348.58683695656}, {
      4460.095118816359, 4357.138195642857, -1345.6269784678814`}, {
      4460.761502370144, 4358.032654268814, -1340.5118027451945`}, {
      4459.490169331663, 4359.651409359118, -1339.4777745477159`}, {
      4457.465687598601, 4361.721291574443, -1339.4777745477159`}, {
      4455.0377980979365`, 4363.883593110559, -1340.5118027451945`}, {
      4451.037506250615, 4366.424132427468, -1345.5181543573995`}, {
      4447.775055063891, 4367.994288900255, -1351.1982244238495`}, {
      4443.600967122102, 4370.832574466304, -1355.7458246204487`}, {
      4443.281896020491, 4370.518728471849, -1357.8017994440013`}, {
      4443.886114423615, 4368.489082344562, -1362.34835099887}, {
      4445.105100516993, 4366.5763900708425`, -1364.5017277503528`}, {
      4446.910468635974, 4364.5235495032275`, -1365.1868589198803`}, {
      4449.966956726962, 4361.5501269695105`, -1364.7301066402918`}, {
      4452.657602875594, 4359.405476424972, -1362.8051414894683`}, {
      4454.321102894232, 4358.415985127872, -1360.5319926017703`}, {
      4461.816980584353, 4352.083246145502, -1356.2353558246298`}, {
      4464.395077199569, 4349.659002878108, -1355.528252511794}}, {{
      4503.634282286465, 4305.857231631369, -1365.534855733061}, {
      4508.162425919162, 4302.632445056816, -1360.7495266488484`}, {
      4510.799745914072, 4300.9090634614295`, -1357.453708270328}, {
      4510.942207721689, 4301.510866734706, -1355.0713457077497`}, {
      4509.558667172532, 4303.498813889133, -1353.3633200969746`}, {
      4508.218087954543, 4305.04676996686, -1352.90637881466}, {
      4505.404709509783, 4307.775686471583, -1353.5917880111988`}, {
      4494.778546253319, 4316.968554759036, -1359.6183315043147`}, {
      4490.704931750391, 4319.917114259321, -1363.7078249691683`}, {
      4488.345149745808, 4320.9665119738365`, -1368.1446959857474`}, {
      4488.057678190959, 4320.689760761486, -1369.9605682215924`}, {
      4489.540616241449, 4319.0384742576425`, -1370.3085065052153`}, {
      4494.108750187846, 4314.681934488763, -1369.05808338483}, {
      4502.467705619706, 4306.863241299617, -1366.2090874363932`}, {
      4503.634282286465, 4305.857231631369, -1365.534855733061}}}], 
    Line3DBox[{{{-6069.712862649469, -1553.582748517321, \
-1196.9879274650004`}, {-6069.262593264219, -1555.2735226024122`, \
-1197.0754086086524`}, {-6068.8493426822915`, -1556.296444496456, \
-1197.8408586587796`}, {-6068.551058258902, -1556.5586011762052`, \
-1199.010869178898}, {-6068.28589670674, -1555.8133294618167`, \
-1201.3179636128912`}, {-6068.407093816679, -1553.699946451086, \
-1203.4390346474963`}, {-6068.8784478171965`, -1551.4506789209074`, \
-1203.9638144461771`}, {-6069.4539318054285`, -1549.7922475736646`, \
-1203.1985077501472`}, {-6069.787233681933, -1549.5388236314857`, \
-1201.8427775256532`}, {-6070.034396321848, -1549.7147680808064`, \
-1200.3667169085868`}, {-6070.069242115754, -1551.3036370844843`, \
-1198.1360989005939`}, {-6069.712862649469, -1553.582748517321, \
-1196.9879274650004`}}, {{-6067.158532427701, -1554.6215034488578`, \
-1208.5334142687798`}, {-6066.661996066272, -1556.0740644266798`, \
-1209.1564917345652`}, {-6066.487036075902, -1556.25488201565, \
-1209.801418963238}, {-6066.429017864808, -1555.111579056846, \
-1211.5612398435082`}, {-6066.621345418421, -1552.7914544403877`, \
-1213.5723465580854`}, {-6067.026387700011, -1550.9772174813818`, \
-1213.867444083357}, {-6067.508883919611, -1549.1827912594542`, \
-1213.7472194910104`}, {-6067.917009570397, -1548.0461559279715`, \
-1213.157019549772}, {-6068.185399760684, -1547.5506247699384`, \
-1212.4465794027517`}, {-6068.202762707514, -1547.8934524399074`, \
-1211.9219366619932`}, {-6067.989083681137, -1549.19258797478, \
-1211.331703411812}, {-6067.989289008504, -1550.2079664433459`, \
-1210.0309662374966`}, {-6067.755541739522, -1552.1790687058551`, \
-1208.675520708817}, {-6067.158532427701, -1554.6215034488578`, \
-1208.5334142687798`}}, CompressedData["
1:eJx11Gk0nGcUB3BLlI4gYhk7k5gJiSppkFhSJdZIgyMVNSIIUgaxhIiKtold
Q15J86KUI21IHGstUzEeOk04lkOUoEOMPWPJtMHEqaXXp+dTP7xffuc5z7nv
ff730gKjPIMlxMTEbOCThG/d0b0iqbEZrUzwPQJ0SNQglVX7folAu5F3GTng
4z0XSqnaJKL3eVt5LhNIIZ5WWwxeEKK+269JooFnoUEPwFO1mD214GcGczbS
NUi0w2SJmvbOv1cd6QFv6D9we1iNRM6cToNO8N23I9J8cHqvqCCPis9XfyPr
IAIvL/sOuani+/PG1pUUmppRtP/kLYoKriet3Gn1CLhmdLBbtxKu3/XLxnOf
gc9255dmHCTR0Qfn1/LBjdTrTjPB+Vvd2S6KJDqbOF0uB55gP2d5AzxCKyZV
9gCJUhKyhdcEBKo6Ih1wFzw0o1/WSoFEdi/MxtveEKjBLjTtEXjiemdemDyJ
cjmlv1PAjbxu2zwDl6zKrSqSI5H8t8lq5osEKgvRUeoD50b4OZ4Cp4jEIgIW
CBSeEb42CW7IdXz7aj+J+ngWl+LnoT+m1GEheMEJ2sN4cGqT+PnsOQJFB9Ia
JZqbUe7zZRsVcLOMS5+WzhIobMneUh68ty1wWQ+c28ndrpwhUJfzAEcdnB1p
W2gEnnu5obVhmkDxPl48bfAtxw7bR+B3eJwnXXwC9Rgf3aSBuyRuzWtCnTsh
fgUTUwSa6vJQYYB/vv13Tj64i1Ru+j+vCdQi3NQ3BC+9/u8hCvRh5oOiUW3w
+o+FnnTwMPcPLVfABfPnDF0mCRQ45L2qCy7Rd5XdD/0UTlfJ/zBBIJaFIEMD
PGWWOFkL/Ter6X03zSOQV5mkmRy4AaXWYBveq7ha2doRvD1W588d6A+j5nsZ
F3j3i8UFC+1/ESjzXfrUKrietZV7vjKJbu7SDXzBb0oJg/f6PKLopTAB+THX
8SkUB7+xQk72gjuozbbRIYfDautRnHECWXzVw2WDy9i/HmJBbn/qaB//ETyk
4mLJz+CcFbGxenUSLQ1IVN4Dn3v8MC0TPLzs3tkrmvh83XS6Yyz4hpamcYUW
vt9kWCHOF7yMZpog0Mb1ZB1bum8HvmRbWH1MF9c/GDJqbAj+SfxpJksP/29z
iZm5PHjx8V+intJwfzp9s6mbMEcKBoK1rw/hflJ/jZPigc9d4c6wD+P+d1y9
NvYH+D7JxAtZ+vi9fsvLk2gEZxdY1S/r4/c1Xrhjv7cHSqJdM1rpOA+q/cX+
2eAxsaovtRg4P4uuKkmJ4IriQxq3GDhvcxH85EjwUxPWb2IYOJ98nulBJriS
8MnjIQbO84n7g9e9wXVlgmJ9GTj/Se4b8W7gdkXatu4MPC8UE6a/M7gGpSV/
lI7na3FUyekMeN3x/QuX6XgeWSPPaXvnWeUtL5P18fxW+jCVHcBHTg6YS+vj
eQ/ft5jiCu72kaCVfhjvhyZP+8EvwIMyRS9e0fA+iZv0cPIDV0tt8xPp4f2j
x9aoCQc3CXXIfqqL99X/7ef/AEY7NQU=
      "], CompressedData["
1:eJx11Gs0m2ccAPCM1UG0YkSoS6umTnXVOTtuNaXtMZc2bqVEpTLaxdK12tDt
6IyetQxxBHnxUikSCYIVJ65zeaRGrWZ2OqM6dy3aYVMMZ2V/n55P+/B++Z33
POd/t4iMCbyiRqFQXOFTh8/xxxvNLuIm1FmZ6phcQqJX3XOKzgMEav2bt3IO
3NuwQsO1mES/Sw493XuQQJsubK8I8IUQt+01MYm6k+rlHuAvPDJrr4J7MeqP
mIIPnSq24oLHK5dO3gE3f3yYP3yfRGz5oh8fPE3IWcgFV2OXTIsK8f8HNmYf
VoN/Y7R8O6AAv69ixed0g0s5EydoJI6Hy2Nkj4HbIudDg3k4fpZNQ9U/4B/P
qT7MziWRz/D6sYvgebyZUeqDJpTrVhLXSZDoueRU+KQ5gU6zKORB8OsdLWeX
RCTSHtTjhIG726fcdAAffHTSxxz8c/7Ovg4zAiXvYXOZ4LU9azF+OSRqj1mN
poJvtsSGXwY/P+9A3MkmUZiuXBVqSqA3317i3gKfUhYvm4P/EuHaKzQh0JRL
/80UcF72+z7tWST6KzbpSc9+AokE1tokOHNxyjEcXG9jvWzFmEDEmeH7EvCc
RmTPBVcIdBPNwTVVRV1K8DyblQYN8EuF3h+4GRFI63jBaDO45y370ERwlLR/
NIJBIGXCEu8ROGurKigNnBWuUy8wJFC0WvJqD7hQ/2okBeJMCC1Oa6ITKN3P
fbkPvPHa1vMvwXWYXMGSAYHelvIX+sEVmuzgRfAoi9EMK/A+0a+dveAVNZ81
D0B91KV2M3x9An3ldDFXBZ5q4m5SC/WkzXk4o/cINNxXzWwDH2paD86Cvix9
zZm1BLd2np7fzat4KyU6BvoYws+fFesRKGTYda0aXK718iUT+r7qeV7XErwr
i/9HEbj/mI5vfj6Jxuu9hR00An3yzFouAJduxmVMwPz8MJl6+Qb4nOxaYDy4
vuF26WGYQ98RE6ETeLKD/9QVcA3avcwvYG4r47V1GeDL9JHAAHDqq/GAuiLI
N5E1SwVfMNYudwHv/Gn69aoY///QznP+KPiLlVTmkwf4/bPUI2oG4PecWwid
EhyP9Ub19hbMrdPMv3XnSnH8mVuPxybB08VGMoEE57vcbVbVD+6xWMlhS3F9
BmJrFprA6Xv7U2zLcD27Bo12CsFf+0YJeTJc/9N/NusKwG386cE/y3C/fvOK
1Y8Dn+bUWRXJcX8pGgqNKPDx9jz0Vo7nIagtajMA3HQoPyyiHM/PiE8I2x1c
2WIVgcrxvKW3qpAt+JBnR6RFBZ7PGEuWxBL8grq0h16B55mzEXvXaDeeBg0b
7Qo8/0qbkB0D8LZPWxiicrwvhaWy23rg8TpOe0zL8X4ZK/rv7gNnbNh9J5Pj
fbzQysug7t7D3m3N43K8v9cbq9tp4BPEKQVThvfdPd1sevcdx3c/Ch4pw/ch
QdRwwhT8zJsCeoYU35Nnx8TS3byeptO0aiT4/qhDhkfBvw96J3ugFN+r/7vP
/wEydBzt
      "]}], 
    Line3DBox[{{{692.7279621760399, -6224.842602420507, 
     1208.0050643561553`}, {813.7518615179529, -6237.130054770872, 
     1060.1086160153413`}, {780.6799538144627, -6267.673104453008, 
     891.4181709100569}, {474.7904377900699, -6265.753873321218, 
     1096.6841519522698`}, {471.1890348802702, -6242.326798766159, 
     1224.3981055255013`}, {692.7279621760399, -6224.842602420507, 
     1208.0050643561553`}}}], 
    Line3DBox[{{{4465.411537226992, 1507.1164363195194`, 
     4298.4315209824645`}, {4342.520349198201, 1128.4444431329678`, 
     4533.976603993657}, {4338.399829031359, 1080.3442218475993`, 
     4549.610640019778}, {4280.746750266003, 1204.6073584097971`, 
     4572.957693020134}, {4201.518439524172, 1247.205736761076, 
     4634.601853722565}, {4235.072898188405, 1365.17088142336, 
     4570.369755451284}, {4199.6828361895205`, 1433.7795184498766`, 
     4582.00327999542}, {4278.47303147275, 1456.5110269499871`, 
     4501.228480907569}, {4274.436765335887, 1427.4419030052027`, 
     4514.356392963427}, {4324.145696939212, 1235.8460470086357`, 
     4523.523209815025}, {4366.170019396594, 1260.230220387981, 
     4476.180608883336}, {4388.893607484269, 1288.9475777753805`, 
     4445.676633981681}, {4456.359088042351, 1502.6173281602632`, 
     4309.3872573180115`}, {4465.411537226992, 1507.1164363195194`, 
     4298.4315209824645`}}}], 
    Line3DBox[{{{524.3348994314875, -5894.421785491871, 2380.898657641059}, {
     903.5670779376075, -5792.102878132891, 2514.542597529811}, {
     1302.5002715730056`, -5779.649515254077, 2363.674726643992}, {
     1636.8453379287012`, -5758.904135435676, 2200.812165670424}, {
     1299.5031914074436`, -5845.583953163507, 2197.3286945833047`}, {
     830.9422126628691, -5826.354179639822, 2459.867009130474}, {
     524.3348994314875, -5894.421785491871, 2380.898657641059}}}], 
    Line3DBox[{{{4266.514601426674, 2941.439937254058, 3719.2200716467196`}, {
     4403.574783641888, 2863.358603662079, 3619.049643443378}, {
     4414.61982310132, 2787.213388607391, 3664.749245698615}, {
     4266.514601426674, 2941.439937254058, 3719.2200716467196`}}}], 
    Line3DBox[{{{4078.828389775395, 1209.4947119335802`, 4752.693467768247}, {
     3951.773925060587, 1266.2402357376716`, 4844.371187327644}, {
     3883.434193196714, 1346.0248887647558`, 4878.008769563295}, {
     3891.610605185196, 1030.630341450288, 4947.825955598295}, {
     3982.5898281281748`, 853.7932795566088, 4908.953960537211}, {
     4082.427516154492, 1005.2590835374566`, 4796.983785704314}, {
     4078.828389775395, 1209.4947119335802`, 4752.693467768247}}}], 
    Line3DBox[{{{6178.062912112082, 
     1447.1557128478455`, -651.9913241424521}, {6202.908824883739, 
     1341.1163584640892`, -642.7617857647199}, {6192.39771986153, 
     1441.0160892332344`, -515.263815559252}, {6120.331498632529, 
     1743.4222891912243`, -435.7003080073898}, {6033.178842113445, 
     2034.2972576817135`, 387.558274925628}, {6001.275246090117, 
     2089.865151793897, 552.2433865457075}, {5884.879700590206, 
     2417.5655863411644`, 459.7621508130301}, {5638.860327323834, 
     2929.1533964182618`, 557.7887406399523}, {5465.127994782492, 
     3266.494799350117, 387.558274925628}, {5446.617733051172, 
     3311.59548902096, 235.59345727024507`}, {5544.63087811683, 
     3149.7887250659937`, -153.99059348067615`}, {5570.7915588250635`, 
     3092.1783182752147`, -304.18472366653396`}, {5539.550537865202, 
     3123.499665319096, -494.9175377664885}, {5425.137982575191, 
     3229.7549794441593`, -907.9506224325476}, {5539.52924205205, 
     2991.048639609298, -1027.1599203345156`}, {5499.121634865678, 
     2969.2306992711233`, -1277.1624757477618`}, {5411.22632577723, 
     3099.0385370812537`, -1342.5362143721625`}, {5383.402634428094, 
     3083.1037587992823`, -1483.6647121605322`}, {5558.106808909259, 
     2856.4788967418194`, -1278.9803373098352`}, {5723.482649705074, 
     2546.2639906745176`, -1202.5388718537388`}, {5801.326312975912, 
     2363.5411911058472`, -1202.5388718537388`}, {5875.429034479812, 
     2348.021674668257, -808.6665357733685}, {6005.982678123847, 
     1951.4620673560976`, -898.7666881602586}, {6051.749389733513, 
     1848.2807083662462`, -804.9853644267173}, {6083.283897964468, 
     1803.8757708519038`, -653.8370669361536}, {6178.062912112082, 
     1447.1557128478455`, -651.9913241424521}}}], 
    Line3DBox[{{{3625.906191822096, 552.6829459707805, 5218.731602538996}, {
     3477.565950538956, 496.99524373308424`, 5324.222429365511}, {
     3417.2354280613904`, 599.4764601110141, 5352.657628970384}, {
     3455.059817068268, 500.95983180573074`, 5338.484297756935}, {
     3347.770513624404, 623.4949146524707, 5393.653712420691}, {
     3625.4949221664383`, 603.4478136078104, 5213.391506073308}, {
     3624.4053046715367`, 552.4541715105333, 5219.7982974296465`}}}], 
    Line3DBox[{{{4550.026969657767, 4285.230297325258, 1273.5264285724982`}, {
     4606.684736602645, 4223.714694231659, 1275.3445061173732`}, {
     4542.0036501561, 4252.81070800451, 1404.1430175914563`}, {
     4661.853293169501, 4163.300133150629, 1273.5264285724982`}, {
     4583.099917185168, 4266.338472225575, 1217.1133388083663`}, {
     4550.026969657767, 4285.230297325258, 1273.5264285724982`}}}], 
    Line3DBox[{{{2952.449942081541, -5399.649059911562, 
     1677.6897982937369`}, {2950.0161083047624`, -5401.332671858951, 
     1676.5512275655258`}, {2947.654207022221, -5402.655320314811, 
     1676.4438122699692`}, {2946.0169800701847`, -5403.354934199558, 
     1677.0668138745816`}, {2944.297152369829, -5403.588464028738, 
     1679.333105718595}, {2943.967913242265, -5402.019227321476, 
     1684.9495195551306`}, {2940.6449396146545`, -5400.766373893154, 
     1694.7399811604469`}, {2932.144765914031, -5402.114991934456, 
     1705.1376412594814`}, {2931.7684609862295`, -5401.736507190267, 
     1706.9827439240087`}, {2932.2942115741653`, -5400.928815368482, 
     1708.634623929601}, {2935.415953021758, -5398.743763617348, 
     1710.179127327942}, {2936.3987737098546`, -5397.655128744832, 
     1711.92728901956}, {2936.5632960046814`, -5395.848523434143, 
     1717.3317707163335`}, {2937.4570428603197`, -5395.068450183288, 
     1718.253831419324}, {2939.335162295902, -5394.011291404183, 
     1718.361045268165}, {2943.3798632394855`, -5392.559860783267, 
     1715.9914972241138`}, {2955.9278250981006`, -5387.342914309927, 
     1710.8011884627106`}, {2959.5222215675753`, -5386.512726671683, 
     1707.1972807650916`}, {2960.7658237425167`, -5387.818012145971, 
     1700.91048998874}, {2959.139257657141, -5390.317802568896, 
     1695.8132359140197`}, {2959.158727924519, -5392.762718255821, 
     1687.9880264795197`}, {2958.4397697971035`, -5394.331914504103, 
     1684.2300955170451`}, {2952.449942081541, -5399.649059911562, 
     1677.6897982937369`}}}], 
    Line3DBox[{{{1879.0017245611984`, -5703.830221818503, 
     2150.2295249033136`}, {2233.681203011526, -5617.749156010146, 
     2034.5457462195989`}, {1895.9189516005079`, -5760.821912165478, 
     1976.4199744174064`}, {1879.0017245611984`, -5703.830221818503, 
     2150.2295249033136`}}}], Line3DBox[{CompressedData["
1:eJx1VWs4lVkbJiVDaSOFhESiLbskIb6lUAxNkQiV2pPKIVGp2VEqpNrvfjeh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      "], {{-3556.832421069578, 
      5197.54118514261, -1011.0996227744703`}, {-3560.3597417428014`, 
      5193.938270958035, -1017.1811229346465`}, {-3560.970992035626, 
      5192.303383708639, -1023.3693641147497`}, {-3558.107599079263, 
      5192.595899609962, -1031.8099760751516`}, {-3554.3636735578857`, 
      5194.128152175207, -1036.9885750928272`}, {-3552.8081782231625`, 
      5194.96715443929, -1038.1156019671216`}, {-3548.110086357953, 
      5198.021225227042, -1038.8954938655343`}, {-3544.7851841728498`, 
      5200.359196864425, -1038.545093703802}, {-3539.8397981017188`, 
      5204.025760682197, -1037.0456961635102`}, {-3535.752258459458, 
      5207.392146245051, -1034.0873715620976`}, {-3531.763796052841, 
      5211.102008032879, -1029.0160546687882`}, {-3528.0959878192994`, 
      5214.700065533505, -1023.3550787767728`}, {-3536.946105225643, 
      5209.737363538698, -1018.0680437367713`}, {-3545.019519839377, 
      5205.191427232558, -1013.22874999798}, {-3556.832421069578, 
      5197.54118514261, -1011.0996227744703`}}}], 
    Line3DBox[{{{1069.236581296401, -6277.278619400431, -374.5930560432121}, {
     1132.520688411983, -6273.342394707526, -224.46776847485307`}, {
     1006.1876348561264`, -6293.561575319383, -257.84264799685195`}, {
     1006.8469182840031`, -6297.685300065339, -115.03405792610378`}, {
     1236.7553588223084`, -6255.6208288132075`, 159.55537766762353`}, {
     1620.4380188492848`, -6169.430312032499, -11.132926047555543`}, {
     1208.8527159215118`, -6238.38273462949, -555.9403362644871}, {
     1051.463687567647, -6272.090729900789, -493.0676217510447}, {
     1069.236581296401, -6277.278619400431, -374.5930560432121}}}], 
    Line3DBox[{{{4768.499830102869, 3416.009075187929, 2506.0137786019113`}, {
     4686.30619096858, 2966.377162982379, 3150.7069843808335`}, {
     4666.871455214114, 3177.559671756598, 2968.3731306784534`}, {
     4547.321249185964, 3180.1276223821396`, 3145.8657714632263`}, {
     4554.017403639684, 3176.92654463004, 3139.407094878251}, {
     4508.278279668345, 3071.497163928443, 3305.9258635139054`}, {
     4505.381140392963, 3063.7700923610114`, 3317.026875222087}, {
     4915.2191647480895`, 2307.6895932996663`, 3347.088734029845}, {
     5360.11057633483, 2499.4606103025344`, 2389.5030774220777`}, {
     4915.56544204786, 3307.275595043339, 2363.674726643992}, {
     4768.499830102869, 3416.009075187929, 2506.0137786019113`}}}], 
    Line3DBox[{{{234.55821973292603`, -6199.748940194648, 
     1481.8600507845056`}, {-10.81387943010952, -6195.890223825407, 
     1516.1270013809603`}, {70.0786213091108, -6176.979969234781, 
     1589.9123046595053`}, {234.55821973292603`, -6199.748940194648, 
     1481.8600507845056`}}}], 
    Line3DBox[{{{6284.664031478106, 1085.5499726264777`, 
     111.32366494150033`}, {6280.0192889738055`, 1086.6290677077002`, 
     261.55054832600234`}, {6249.847177541226, 1252.624330884985, 
     241.1560346640822}, {6253.0011059172775`, 1255.1485573513996`, 
     111.32366494150033`}, {6284.664031478106, 1085.5499726264777`, 
     111.32366494150033`}}}], 
    Line3DBox[{{{4969.348503597715, 3672.6523635341796`, 
     1582.7235249461485`}, {4869.9252424495535`, 3669.751890145693, 
     1872.0477792312827`}, {4739.0992771970205`, 3785.4308247433037`, 
     1974.655717322365}, {4737.54369698675, 3935.513078772548, 
     1659.8886155255543`}, {4542.0036501561, 4252.81070800451, 
     1404.1430175914563`}, {4582.361226608544, 4223.547924581673, 
     1360.6697643574216`}, {4807.659573560055, 3870.0647856711853`, 
     1611.4656898191165`}, {4969.348503597715, 3672.6523635341796`, 
     1582.7235249461485`}}}], 
    Line3DBox[{{{2859.0117336678386`, 1522.2977113636769`, 
     5495.131961538916}, {3010.865721672832, 1574.0425606714707`, 
     5398.600866266215}, {3097.0411541435697`, 1385.3734575591907`, 
     5401.563677550378}, {2994.5671960879517`, 1300.0038520667886`, 
     5479.997352437815}, {2859.0117336678386`, 1522.2977113636769`, 
     5495.131961538916}}}], 
    Line3DBox[{{{4969.348503597715, 3672.6523635341796`, 
     1582.7235249461485`}, {4807.659573560055, 3870.0647856711853`, 
     1611.4656898191165`}, {4582.361226608544, 4223.547924581673, 
     1360.6697643574216`}, {4661.853293169501, 4163.300133150629, 
     1273.5264285724982`}, {4583.099917185168, 4266.338472225575, 
     1217.1133388083663`}, {4530.688062999867, 4377.782280176727, 
     997.8485221521206}, {4228.01102994151, 4692.935964993429, 
     887.7434590939694}, {4497.857757458458, 4490.0143574097965`, 
     544.8489276163259}, {4736.270941387223, 4249.612870048872, 
     443.10463153298207`}, {4911.949250885615, 4051.498828809278, 
     382.001946174733}, {5148.078451672213, 3731.15365957076, 
     513.4143688093942}, {5299.429122884662, 3439.29826982787, 
     880.3931352986056}, {5181.38379327356, 3532.2922635316154`, 
     1167.898558007838}, {5024.6305299715195`, 3670.7443291640334`, 
     1402.3329837054446`}, {4969.348503597715, 3672.6523635341796`, 
     1582.7235249461485`}}}], 
    Line3DBox[{{{
     1901.1987153926514`, -3434.5641030716647`, -5027.626334213876}, {
     2036.7010442149237`, -3418.8585382074075`, -4985.084637417919}, {
     1901.1987153926514`, -3434.5641030716647`, -5027.626334213876}}, {{
     1975.1886826458106`, -3382.388620335922, -5034.47023816508}, {
     2110.55585334829, -3342.873475722359, -5005.8530347193755`}, {
     1975.1886826458106`, -3382.388620335922, -5034.47023816508}}}], 
    Line3DBox[{{{2939.706160078936, -373.3808948713573, 5648.586379771405}, {
      2938.1606864134806`, -371.45995609099657`, 5649.5170565133385`}, {
      2938.4835351475253`, -370.16172555683005`, 5649.4343522692325`}, {
      2942.96892973516, -367.1138648240906, 5647.298024619155}, {
      2948.0734962553597`, -363.9079356942864, 5644.842545635389}, {
      2949.435923104237, -361.38493081980556`, 5644.292901442892}, {
      2952.2765895047773`, -360.06471588781693`, 5642.891991650617}, {
      2955.3143472065376`, -360.37238954691105`, 5641.281994373924}, {
      2956.59343190421, -359.8868509723458, 5640.642731366532}, {
      2958.5545414649064`, -357.04211457089673`, 5639.795177451049}, {
      2960.9495926056566`, -356.3350039540356, 5638.582850770703}, {
      2964.073763254788, -356.645379333258, 5636.921543660585}, {
      2965.9060154773047`, -355.57930778905035`, 5636.025066767163}, {
      2967.811311820205, -353.8112717212139, 5635.133298461985}, {
      2972.9706758649586`, -351.9742364942833, 5632.528134620485}, {
      2978.3492749193547`, -352.35273713727497`, 5629.662231006335}, {
      2979.1114462364208`, -353.7689587421821, 5629.170116876342}, {
      2978.6532085380495`, -356.94671585071035`, 5629.212004828862}, {
      2976.9172030145082`, -358.5465290346261, 5630.028600541342}, {
      2971.9131180953846`, -359.8146818880171, 5632.590833993749}, {
      2966.679391062978, -362.2075438461233, 5635.195895070832}, {
      2959.3100955928453`, -366.1942956430633, 5638.811859422226}, {
      2952.714398692847, -372.53289976591554`, 5641.853472782143}, {
      2948.5386111329994`, -375.40218601440733`, 5643.846821751409}, {
      2946.314686421625, -375.63116463706393`, 5644.992886331182}, {
      2941.3361916308518`, -374.55039837958117`, 5647.660320599692}, {
      2939.706160078936, -373.3808948713573, 5648.586379771405}}, {{
      3001.455414092851, -366.37100753902916`, 5616.480336656361}, {
      3001.7177687236403`, -364.21525899008924`, 5616.480336656361}, {
      3002.4960024727393`, -363.5785522831574, 5616.105597427359}, {
      3004.4615382739735`, -363.978844493031, 5615.028402046948}, {
      3005.6436235744845`, -364.5691750027192, 5614.357435602393}, {
      3006.9568599744102`, -364.48350047630026`, 5613.659760809643}, {
      3007.9256206901127`, -360.6223187053733, 5613.390124839673}, {
      3009.6321046800017`, -358.064484884349, 5612.639139401706}, {
      3010.7101127726205`, -357.4600140023275, 5612.09948639884}, {
      3012.4708911126104`, -357.83169335764234`, 5611.130839628318}, {
      3014.41938160575, -359.93599901016626`, 5609.949693037912}, {
      3015.712346044665, -360.0903851084793, 5609.244837807154}, {
      3017.7112807675494`, -358.4541649339196, 5608.274505368813}, {
      3018.799525636294, -358.5032835766726, 5607.685664307765}, {
      3020.843713910559, -360.7030948164814, 5606.443634204919}, {
      3021.786817608797, -361.222252063719, 5605.9019438034875`}, {
      3022.747711680964, -360.6843017835662, 5605.4185210360865`}, {
      3022.7464578961267`, -359.868180562455, 5605.4716513327485`}, {
      3022.1506187492278`, -358.16577733494336`, 5605.9019438034875`}, {
      3022.167695426553, -357.18901454773743`, 5605.9550585610805`}, {
      3022.8080648484047`, -356.8554699287339, 5605.631031977522}, {
      3026.3539106112394`, -357.35708427934196`, 5603.685538290262}, {
      3029.247492184967, -357.4521710797891, 5602.115784815718}, {
      3031.0829135807485`, -355.90951105312996`, 5601.221159795493}, {
      3032.683963109917, -355.8533550088819, 5600.358029592826}, {
      3033.2880040960686`, -356.6595294687738, 5599.979603020532}, {
      3032.798411862861, -359.140522794315, 5600.086210755075}, {
      3030.5490597094354`, -360.42697700573615`, 5601.221159795493}, {
      3028.4740166935567`, -362.7159311879145, 5602.195638629061}, {
      3027.4749078747104`, -364.31149981769386`, 5602.632104938677}, {
      3025.2514955783613`, -365.35100273368755`, 5603.765316579771}, {
      3022.337923244173, -365.40856652763074`, 5605.333509009818}, {
      3017.6739610402965`, -367.45273576257034`, 5607.712193206777}, {
      3015.1863681712816`, -367.3127162884374, 5609.059299351976}, {
      3013.620962183771, -367.85598848874025`, 5609.86491441813}, {
      3011.24502338738, -370.8200869109933, 5610.945514106162}, {
      3009.8732507347004`, -371.3817863787616, 5611.644346647233}, {
      3007.393920612174, -371.15846391590185`, 5612.988231853736}, {
      3003.443786974621, -368.60093393351286`, 5615.271361206588}, {
      3001.455414092851, -366.37100753902916`, 5616.480336656361}}, {{
      2933.5981232487834`, -363.1997629796838, 5652.424406698662}, {
      2933.378056198815, -360.8751378000625, 5652.68750280157}, {
      2934.723826409528, -358.9792408340271, 5652.109664307164}, {
      2942.69578985279, -353.1978192227662, 5648.327759512451}, {
      2943.993330406927, -352.47792806515184`, 5647.696545561423}, {
      2945.2943923838266`, -352.5554886896473, 5647.013304357197}, {
      2947.1361098939838`, -353.1724751545345, 5646.013783418543}, {
      2948.2279752525437`, -352.50736785299864`, 5645.4852804296315`}, {
      2948.4011897651667`, -350.2212730233874, 5645.537102359104}, {
      2949.3334863024493`, -349.1366481693674, 5645.117295280949}, {
      2950.84396022482, -348.35979841320295`, 5644.375878983806}, {
      2951.87416699009, -348.95940497560406`, 5643.8001316431855`}, {
      2952.9558735477667`, -350.0436572000268, 5643.167083910751}, {
      2953.7483879724596`, -350.21863160299637`, 5642.741449032395}, {
      2955.821878865257, -348.8690016112033, 5641.739193735941}, {
      2957.9737063604716`, -347.68617236432397`, 5640.684317352527}, {
      2959.7628451262763`, -347.89647132944793`, 5639.732762804149}, {
      2964.547036358016, -349.2587933396526, 5637.135163759676}, {
      2964.2627109391524`, -350.7439367174028, 5637.192471558129}, {
      2961.398267500507, -352.00360318558745`, 5638.619286189603}, {
      2957.1121871693445`, -352.4520109169239, 5640.840255005911}, {
      2955.8067796813875`, -353.2539405076256, 5641.474242137899}, {
      2955.7915636703156`, -354.2097085634397, 5641.4222856027}, {
      2957.243080363723, -355.58263541138837`, 5640.5751517939}, {
      2957.1293581207105`, -356.5271466604207, 5640.5751517939}, {
      2955.655654602016, -356.9879820319319, 5641.318367376893}, {
      2952.8412273369872`, -356.9670165287723, 5642.793361913011}, {
      2950.1858703738803`, -358.0017910765092, 5644.116559554057}, {
      2947.0739315186793`, -360.40900312139166`, 5645.588922568847}, {
      2943.940872088798, -363.6847151831582, 5647.013304357197}, {
      2941.242788584902, -365.10008178356173`, 5648.327759512451}, {
      2938.425754335486, -365.862261888385, 5649.744470468008}, {
      2935.9886010179334`, -365.08268318584584`, 5651.061781570838}, {
      2933.5981232487834`, -363.1997629796838, 5652.424406698662}}, {{
      2949.684037362962, -386.2961038382158, 5642.513011937637}, {
      2950.257952802742, -382.8077089457461, 5642.4507051365445`}, {
      2952.0023300525972`, -377.5134973064766, 5641.895026736625}, {
      2956.3232322964686`, -373.5019290787815, 5639.899196365001}, {
      2960.7768621798105`, -370.26681981162557`, 5637.775850858852}, {
      2962.845082124999, -370.2996313993223, 5636.687049350286}, {
      2965.2125452020096`, -370.7899972245127, 5635.4097481669305`}, {
      2966.4227990737636`, -373.38934269422845`, 5634.601127488772}, {
      2965.1065730156793`, -379.9283328207907, 5634.856800538876}, {
      2963.4526265672994`, -381.45664266134935`, 5635.623572406243}, {
      2960.9930721065584`, -382.3011367964857, 5636.859015244312}, {
      2959.946016230312, -381.65124467469246`, 5637.452935363679}, {
      2958.6095350235037`, -380.70463097233153`, 5638.218450292985}, {
      2957.236522895206, -380.6565036559426, 5638.941963159322}, {
      2955.2840340551165`, -383.1714034982213, 5639.795177451049}, {
      2952.979662215722, -386.53374091490696`, 5640.772683921472}, {
      2951.3606955740065`, -387.4457209466984, 5641.557369019747}, {
      2950.091480088532, -387.27910216996133`, 5642.2326118418305`}, {
      2949.684037362962, -386.2961038382158, 5642.513011937637}}, {{
      2977.255478037511, -364.48239183007644`, 5629.468544591173}, {
      2978.0166588140114`, -360.26095523842736`, 5629.337662101725}, {
      2982.3443541368747`, -353.45083502299434`, 5627.477974241166}, {
      2985.4965384152033`, -351.2091698446653, 5625.946696168376}, {
      2987.3958209626735`, -351.03085776001757`, 5624.949531844265}, {
      2990.9833177176542`, -351.45240285768267`, 5623.016422851729}, {
      2992.3242805441314`, -352.8198671033682, 5622.217287818321}, {
      2992.3779495296053`, -354.539336459888, 5622.080554036163}, {
      2990.824911082278, -355.9672276878139, 5622.816676191918}, {
      2987.9284594903124`, -357.1298878129358, 5624.282661299889}, {
      2987.336213642828, -357.86560234176005`, 5624.550492912184}, {
      2987.1575213924525`, -359.3541220993597, 5624.550492912184}, {
      2987.6380076531286`, -360.61807376463383`, 5624.214383369533}, {
      2989.213060379086, -363.124164398821, 5623.216144777715}, {
      2989.0317475990305`, -364.6136187203243, 5623.216144777715}, {
      2988.031179483672, -365.6004355548112, 5623.683817743983}, {
      2986.2581167517087`, -365.78554143207674`, 5624.613505638746}, {
      2982.5328851682043`, -365.4296282091684, 5626.612876830708}, {
      2980.237528247258, -365.75028045159064`, 5627.808161049579}, {
      2978.258246800715, -365.5073727177928, 5628.871633449502}, {
      2977.255478037511, -364.48239183007644`, 5629.468544591173}}, {{
      2939.62268206101, -341.6416328338589, 5650.638297734783}, {
      2939.731650834604, -340.7027138932232, 5650.638297734783}, {
      2941.7398759834305`, -339.9052530468953, 5649.641104618958}, {
      2944.746801675999, -339.21890986148867`, 5648.115658802621}, {
      2946.2635832389187`, -338.28125451787025`, 5647.380842560733}, {
      2946.959679219568, -337.4883280645439, 5647.065075548032}, {
      2950.454934398528, -336.93392215416003`, 5645.272792539365}, {
      2952.55778201505, -337.01484075396024`, 5644.168426878662}, {
      2953.6271297403136`, -337.37451264576947`, 5643.587414639648}, {
      2953.8225826637463`, -338.3526189279469, 5643.4265606464905`}, {
      2952.2171092805793`, -339.12404748608395`, 5644.220292483956}, {
      2950.8529461017174`, -339.6039769036794, 5644.904757332405}, {
      2950.4771544147766`, -340.19731056300617`, 5645.065459422253}, {
      2950.8945726431434`, -340.963038444609, 5644.801069795096}, {
      2952.8456339077625`, -341.5853906985543, 5643.743064064236}, {
      2953.742814314344, -342.4858978500594, 5643.218982695977}, {
      2953.4376062070014`, -343.40654066928755`, 5643.322775109342}, {
      2950.13845806546, -344.0588659621081, 5645.008437991569}, {
      2946.2461306377672`, -342.8101025470828, 5647.116845018672}, {
      2943.8230880670876`, -341.9709792946815, 5648.431212777879}, {
      2940.103240309307, -341.7780951099144, 5650.380019080641}, {
      2939.62268206101, -341.6416328338589, 5650.638297734783}}, {{
      2933.753920993417, -339.4962049527432, 5653.816759312105}, {
      2934.1929641689185`, -338.3612169662723, 5653.6569603995895`}, {
      2937.139415354368, -337.3531038079047, 5652.187065917314}, {
      2938.540649123846, -336.5631435202014, 5651.505798800052}, {
      2939.8792630467406`, -335.44812076348967`, 5650.875876090714}, {
      2940.2268797486586`, -334.13885004534296`, 5650.772585628787}, {
      2941.181725068446, -333.77160513898826`, 5650.297360823264}, {
      2942.8223589435197`, -334.35577274261425`, 5649.4085062894665`}, {
      2943.3042177993675`, -335.44082377531583`, 5649.09315069502}, {
      2942.8751772082537`, -337.457681250438, 5649.196553027392}, {
      2940.768449140208, -339.11947967230367`, 5650.194031805304}, {
      2937.661071141634, -340.6627919024381, 5651.7174366127165`}, {
      2935.7608908700126`, -340.52033132177746`, 5652.713294126514}, {
      2934.423497708922, -339.8902900420718, 5653.445588031148}, {
      2933.753920993417, -339.4962049527432, 5653.816759312105}}, {{
      2968.775768661937, -350.43994950641036`, 5634.835930814283}, {
      2968.7989479137273`, -347.7162902508701, 5634.992447054434}, {
      2970.1659773988345`, -346.3551602683695, 5634.355851351412}, {
      2974.303006343799, -346.27459226337106`, 5632.178017712781}, {
      2976.941101367389, -346.90296492760854`, 5630.745394875705}, {
      2978.1888246615476`, -348.3339409551572, 5629.997200731073}, {
      2977.319217266642, -349.75989267846626`, 5630.36872557813}, {
      2974.439010528363, -351.108449369102, 5631.8069142530685`}, {
      2969.93940530552, -350.816446204002, 5634.199272291165}, {
      2968.775768661937, -350.43994950641036`, 5634.835930814283}}}], 
    Line3DBox[{{{-6083.569112686511, 
     230.16272988776046`, -1903.9500148211016`}, {-6064.942738866877, 
     137.63280441723916`, -1971.1267020194769`}, {-6053.5745662744885`, 
     211.39548202997915`, -1999.3400437974415`}}}], 
    Line3DBox[{{{2384.0688150297246`, 1069.7763090216713`, 
     5818.900946386588}, {2438.7151824963207`, 1160.5962059473643`, 
     5778.710790081342}, {2723.2826521560755`, 1050.8273196781186`, 
     5671.622980301482}, {2916.6132394376273`, 1143.9870503143068`, 
     5556.300435564146}, {2956.0908379094585`, 1247.6884556622088`, 
     5512.949679222095}, {2771.069510704976, 1465.1423230391238`, 
     5555.388863984303}, {2473.970329995247, 1521.0048607307083`, 
     5679.24543490918}, {1995.9596918806546`, 1107.137641756201, 
     5956.35835394364}, {1995.4091225083205`, 1104.5562541700037`, 
     5957.022041690337}, {1920.3552289353001`, 1016.7757547890094`, 
     5997.184051634074}, {2060.4857416266323`, 993.8603400317141, 
     5954.3642669109}, {2134.8934733452916`, 802.4537010215435, 
     5957.022041690337}, {2384.0688150297246`, 1069.7763090216713`, 
     5818.900946386588}}}], 
    Line3DBox[{{{4565.42764037947, 603.7522526363275, 4413.634258510293}, {
       4569.019828967605, 596.1152578966723, 4410.954328918362}, {
       4570.6380761013, 593.6222410026879, 4409.613804210064}, {
       4612.4974708422, 313.10060110072345`, 4394.843431234459}, {
       4700.464620653186, 264.1683774196826, 4303.912302586933}, {
       4700.677486206689, 143.61868657907868`, 4309.3872573180115`}, {
       4693.733513008255, 122.9097376382984, 4317.588747814788}, {
       4632.398866433145, -142.88136616627472`, 4382.725105168172}, {
       4404.753306429094, -85.85735451747685, 4612.8722629374015`}, {
       4210.548714722098, -352.33624886378465`, 4778.593138015486}, {
       4214.355533990674, -99.3167926465276, 4787.186783260142}, {
       4120.500125633704, -139.09096174410251`, 4867.231873356121}, {
       4156.725965483417, 0, 4838.33059410743}, {4003.0639489020314`, 
       178.27767234629303`, 4963.0141831898545`}, {4118.49694567693, 
       419.55208296077467`, 4852.810798715839}, {4119.888736210329, 
       459.69271231581985`, 4847.990625144708}, {4144.442455325039, 
       599.684695014598, 4811.630564690581}, {4264.912486643201, 
       567.7981715214996, 4709.133721576799}, {4389.723568171527, 
       458.7966466040735, 4605.176035951752}, {4402.970064032088, 
       543.2170941383158, 4583.293956738231}, {4565.42764037947, 
       603.7522526363275, 4413.634258510293}}}], 
    Line3DBox[{{{3718.0111299957875`, -5148.833195959614, 
     594.7467086178401}, {3827.342105647412, -5066.777657256236, 
     605.8302305658175}, {3945.0455281605578`, -4992.307676360304, 
     448.65748104134576`}, {3859.263116752308, -5072.125743401079, 
     259.6966091486647}, {3690.4857963685763`, -5196.21842766434, 
     259.6966091486647}, {3718.0111299957875`, -5148.833195959614, 
     594.7467086178401}}}], 
    Line3DBox[{{{-5227.152994821303, -3103.5329332455867`, \
-1932.0374196662756`}, {-5225.478624842957, -3104.7595578001096`, \
-1934.594251919049}, {-5221.4545095887615`, -3106.56238483436, \
-1942.5490066990442`}, {-5219.7011057489035`, -3107.8632670383345`, \
-1945.178647332293}, {-5217.626760040166, -3110.3297879015927`, \
-1946.800784211058}, {-5215.567111285876, -3113.2982833400815`, \
-1947.57469603217}, {-5213.742883176292, -3116.1601433829023`, \
-1947.8821316805643`}, {-5212.10461413173, -3118.515459169282, \
-1948.496988006122}, {-5211.133211023412, -3119.4165657736567`, \
-1949.6524398327274`}, {-5209.939117533102, -3119.813501801128, \
-1952.2069038853892`}, {-5210.182896027467, -3117.8596281445202`, \
-1954.6762440694235`}, {-5212.266756636587, -3113.549585398853, \
-1955.99026759433}, {-5214.170753585391, -3110.120038347642, \
-1956.37174113797}, {-5215.622632896155, -3108.0247902374977`, \
-1955.831318005349}, {-5217.4166025078675`, -3106.1330316406193`, \
-1954.0509910507221`}, {-5220.427013535579, -3104.2241926337924`, \
-1949.0376210363618`}, {-5221.8696761449255`, -3102.861903385015, \
-1947.3414656603006`}, {-5223.632475756019, -3100.0864453057884`, \
-1947.0340212380402`}, {-5224.446801903085, -3098.2274927945673`, \
-1947.807923532636}, {-5225.311842275721, -3095.166679366216, \
-1950.3520369190326`}, {-5228.403898926557, -3088.6217178609822`, \
-1952.440074278094}, {-5231.37907114054, -3083.4858575286817`, \
-1952.5884539383462`}, {-5234.678378940098, -3078.909620896792, \
-1950.966812994584}, {-5238.8392293725865`, -3073.4858132513814`, \
-1948.3485762381808`}, {-5241.111022893025, -3071.8683252839646`, \
-1944.7863450328737`}, {-5242.9509099601355`, -3072.8237613883134`, \
-1938.3068841634881`}, {-5244.299994881675, -3075.5822961030913`, \
-1930.26547230719}, {-5243.751668745351, -3077.9672926592325`, \
-1927.952144595488}, {-5239.373439946027, -3086.7195267117886`, \
-1925.8614203011286`}, {-5235.0681147299, -3094.161052924662, \
-1925.6279242632718`}, {-5230.029336503878, -3100.7978239221193`, \
-1928.641927489045}, {-5227.152994821303, -3103.5329332455867`, \
-1932.0374196662756`}}, {{-5256.797581259514, -3067.0263663946102`, \
-1909.7754180804604`}, {-5255.495756010841, -3068.9723212493373`, \
-1910.2321691243428`}, {-5254.685885525397, -3069.975400178651, \
-1910.8482348836878`}, {-5254.1414562338505`, -3070.2723708023464`, \
-1911.8678869034277`}, {-5254.152881512373, -3069.664000174192, \
-1912.8131413522365`}, {-5254.758615322984, -3068.295894933739, \
-1913.3441629308736`}, {-5255.306250896868, -3066.893979058006, \
-1914.0875686608647`}, {-5255.577241801491, -3065.945528957951, \
-1914.8628042032108`}, {-5255.185996409061, -3065.594354837258, \
-1916.498130637241}, {-5255.301824585381, -3064.6785245210394`, \
-1917.6449003528005`}, {-5256.024404678252, -3063.133317280818, \
-1918.1333186175682`}, {-5256.480454091237, -3061.678708624861, \
-1919.2056718627077`}, {-5256.793996133136, -3060.1413331974245`, \
-1920.798165626809}, {-5257.855587569093, -3058.0569029804556`, \
-1921.2121924457083`}, {-5259.567370186299, -3055.8592574189943`, \
-1920.0231684326666`}, {-5262.516272747424, -3052.6604440474052`, \
-1917.0290510101438`}, {-5264.672140960174, -3050.8416788174236`, \
-1914.0026094517998`}, {-5264.822220725034, -3051.7879302543834`, \
-1912.0803076527873`}, {-5263.14178275928, -3055.356341547166, \
-1911.0075590486213`}, {-5261.385338594611, -3058.7582035440155`, \
-1910.402120254753}, {-5256.797581259514, -3067.0263663946102`, \
-1909.7754180804604`}}, {{-5153.798992785779, -3204.9628945933564`, \
-1963.2581277662034`}, {-5152.7634594681695`, -3206.4394120286556`, \
-1963.5653077397114`}, {-5152.014608310349, -3207.2209449096913`, \
-1964.2537963026411`}, {-5151.753252837092, -3206.9334881248988`, \
-1965.4082819163846`}, {-5151.972599888603, -3205.6978089459058`, \
-1966.8486412829227`}, {-5152.183118642502, -3204.332143448332, \
-1968.5218607978495`}, {-5151.946479891146, -3203.4368216803573`, \
-1970.5972921527318`}, {-5152.219165774919, -3202.3596135754024`, \
-1971.6349213692654`}, {-5152.9569781266155`, -3201.3222290738054`, \
-1971.3914013239814`}, {-5154.882555532971, -3199.277364637059, \
-1969.6760832133825`}, {-5157.9933228612645`, -3196.346770295877, \
-1966.2873379386142`}, {-5158.223726272997, -3196.4895485344014`, \
-1965.4506470081244`}, {-5158.007647274798, -3197.3524908988898`, \
-1964.6139187121114`}, {-5157.156333047562, -3199.317230712346, \
-1963.6500461575401`}, {-5155.71159183207, -3201.9112355729762`, \
-1963.2157577199591`}, {-5153.798992785779, -3204.9628945933564`, \
-1963.2581277662034`}}, CompressedData["
1:eJx11Gs0lHkcB3BMo5GotFGK2N1yiWmEKJUhIwzKNWFS45bkmi3abXelcpg9
RDVT2R4TTc/IjnZqyo7IX9PFHtrVRpou51RbSEa2pSLVfvfVvtoXz5vPmfPM
zPf3/f1shVnhyQZ6enqr8TDwtHY8ukVfVJGH9/iraBeaJJtpf+4Kp0j9rvgJ
/UsqknZN2S3m0MS09CTtCee8cu0Pg6t7vndSL6UJp+qo81dhFHFxuxRZC9fl
rHF5xKaJq1PLRNUGiuxyv+37Bt6dUfnUAD4gf2aqWE8R9+osZVCjiuh/suE6
OtNE2tKbJg+lSKc2XyqFlzeXBkY40eTdRoFxWQhFBvz6TAbhpxkZop+W0KTr
2+LftwVTRHgsZdLtFxXpsx4vmAqP4TzQOvMp0pHcNvAd/BVrZpLQkSaXy588
fR1IEe744qgr8KYZqSU6B5r8EWNvqgjA/23ML/wAV8TWFRXAHf4SxcWvo0h6
akc9W60ikS9l09zgzuoPReb+FEkSv+1OgPdZBpYP29PkAsu4vsuPIuN+jyf3
wavVJ2p64TnO0Yqjaykyd3qiUAIPVxzO+PfzRyQNqdm+FPGxt7lZD9c2n/Ng
4v2alD4bvg9FVvy6N00Jt3XNagqEl31WWOnFpYiMsfCmCn42qkolgnv57j8e
7E0R5aebwkb4kL3wbid8RvaJb7atoUjt8MjKk/CQuMu+25GDwv/Q2OnVFIkx
cSgvhPsej/vIRG6u/Bt2b1dRJKwogUqEjyz3dqmGW/24biQNLk6qoH3hjUPR
DHfMxcNtiRMLHsFh1TPh0VfbRu0wx4FZN2ybvSiS2fCi7g7yZJ4a/XsAfu/h
nIZDcM+WoC0V8GkuFu962P95KOdKRQg8bO+ngDb06nnbcOoZ+DV9o2ZD+GUL
q+V16OFvr7OO9sOL/WKkLehDa6ujpAy9fcqSbArE79G1CiaS4M/jZnZJl9GE
Pb5W1QHndTqHW8Bdl86yKHSlyUajYFkmcmicPWdRO/r5PsmsLsGNJtv5svV2
yI1ZMJaeBt/mv2VGujtN3ieMlush5/UVy3aaw0tdLAW85TTx+ZJd8gqeWxA6
1oY9auEqRq08aJIZZe01inkpr3Nep8N1V9U9o/ANOpHMFHMffFihtIWz5Vrp
CU+a1NquaF+DnuQWaTN7VSqy/+uaVO4KmoiHzr3IRK8Epfm0CF5ckJf/DD5b
ZKJR8ShyYErT4hC4rP2w3aaVNJFcNKyaQD+lnBTdfDjfX7GLgi8QVOVw0fNE
4dD5wQsqYi3YofkTfi63w/8g9iJPGrxbDb+nd9vEwYsm/H6xZWcQRc7EBvLE
8OFjOXe84e/lZ2dOxd6FTvbc2QmPMKjL3gg/v8jBwQN7epWndd4Cf8drSvSB
z2O8KQ7DXh9/e7ciFm6xwL17Et/7hV2R0Vbcge5TU/Oj4YLD+96o4Q0qze1Y
3A0Ga2hwK1wTPxIQBF/S3DHFDXcman6ydSackbs47wFy2KFvGWeAu6Q8tr+6
FJ4jT5T/m1v1/YNcGZwZfKBGDh9I4pXRyDnl/p6MZRHI//NKzl14pu5ZzRPM
xfsly7AabnYrQGOI3MKnyywXwudY8dmP4U/mFvZ6wsfErLWbMff20CzKOJIi
140/5u2BR84rkmSjJ36C3XvM4f1mIvNW+N4jESEH0Stz/c19Y3iPkeaHCSbm
vlDCEFHooWOlzRUVPD88JTke3jxWP1KC3pbU1ulHwP/vPv8D3xkRiA==
      "], CompressedData["
1:eJx11Ps31GkcB/BZsrS74YzL0cWtSCYaJ1LGpQymqAydLjqpbMrmku6dFA0y
EyOjUxLlEUl92VKrYVHyTFOUTelCGUa1LlOxytI6wrZvf8D+8P3ldb7nOc/z
/ryfx3rb7jU7tFgslic+bXzKZO4GhxNyWqMYZcfMZWgZ70NN7UpCV6tmVyfD
LzY8Lz5ry1Atfx/XqgBCd6y4z2+HW7psrFPYMPTMXRbrtD+h4yHByS6pcpog
ib42MIeh4XHfd4WtIPTIW8GHTHhchJelBVzi2GFksJzQUXGFSA3/l1sqjp/N
0IR6b59aP0IjdbaIHdLk9Nnzw7kd1gyNmltWvM6X0KnSHd7x8LwSnyRPeGHt
VasuPqHpO38a+wP+WKIJz7diqH1Db+4eb0L1+xrz9aVy2umvMuXCzeeI5g8t
JbS9YXpqELyQO+Wvu5YMjQjyKEz0IpRzdGXfcbhpzJLueHhGjN+YmSehxbMu
ZVXCZYf1WDx4UjR1u+NO6EivBUsFr2zcHGEDz9f7zD/II/Sy2aneAXj5A/1B
D7hOymcnNzdCg6y65xqmYz8disbJ/7enmX5kLyFUJVm6xRhuaPb10054N7sg
VmsxoWq2zS0T+JjwUHkJvK4uKN3cFfkXXcye9F7vROV7+O2rg/sDFxH6xnRx
is6kRx/1X4vz/lrGMihwIdRhdGT5EPbjpCPxq4XvV8tD2fBGo57QZvjJ1oPq
YOSZX3DW56YzoWO9jvfy4DXj3oZszEXX47EyDi5ct7vxANysn1dTAN9ynj0S
Dl+956XKA37iVmzrJsx3Rt5udSz8ybQ3bsOYi8pGJOWjJ5UfeJuL4ZXpjimV
8AJhfcc/8ObszYd0sZ+BsPSXO+F258TkFfqmLNvEl8JdZXtGzeDrBF+eVqCf
Of0/KhxxXtoXz6tHf8rLn54/ZYecmxJfj8MPGLjr7YMfMeLciJzH0GUu72+/
Q27jCq/SmfAh2W3nZfboZ6sgWIOcL7wLt1Kiz4/OfLEz5jD0y4yAnGmYl/C6
X3sk3CV3u6cGPpy0/r4X5puarx2lD6+vlSl+n491vA8UnkIfqu99fkglcpq5
XRqw0YGhtodmJ/zpQag0tUa+Fz5iwps1Cs+JC5kTgL6tMajosYYr/s4wyHFk
6P2Qjut30M+WPKHxc7GckuzqCdcFDO1pK3blos/GUWmhmfBAnsnMK/COb6lF
19D/HqGz0yY4p1QwSwNXHWueZ4n7krXVt3c63CI+sWcDl6HzAovqJbhfJdrr
J4ZTcK4bdkXRcDnfLqFLQOixFuaGEh4jatW6CTes64zg4P4OOzcry+DHMy9v
PAeP8rwSGTF537XTV1yALxnSbEuEJ3PYBSfxPjgrp3ilwQUDLywm1zfPDPuB
4D1pXdX48QSc1oQFc+AVT16qk1YRqsdL0UuF/3bNfnEf9s8/XtUTtJrQjPbz
Njnwqb42NSL414xHHt/gTQvEXlVwddNgmwa5xVkv7ToZSKis4UlEE9y/lQYl
w/dVvx0agH9y1pr+HXJIGaSr7iH/kruvYm2FeJeSdKzt4Z4Lqx84wx/75ru5
w51Gdtn+DK/Kmt92GPN9ttB9rRNcJupXl8CzThtuPY0+7DITtI1i/dwJebEG
3tIS6N6JXnUPXX14Dt457ekvizBfx5U+ZRPoYenDy7om8BehE8FieN3N/hJL
uCnXKHsvznVQpG/+Gp4hDdX1Q59D1vrJLiGf/3uf/wMt9wzl
      "]}], 
    Line3DBox[{{{6284.664031478106, 1085.5499726264777`, 
     111.32366494150033`}, {6304.639839115316, 
     966.622435822484, -70.50713158688959}, {6243.072935020844, 
     1232.387691524598, -439.40254413111114`}, {6243.608471030748, 
     1279.7361531463318`, -259.6966091486647}, {6173.518764530397, 
     1590.8340673146195`, -211.48693638689255`}, {6174.704016886918, 
     1596.8870369599074`, 103.90276719105113`}, {6209.575198942648, 
     1452.631911208069, 137.295554248364}, {6203.178129531861, 
     1466.369172562866, 241.1560346640822}, {6249.847177541226, 
     1252.624330884985, 241.1560346640822}, {6253.0011059172775`, 
     1255.1485573513996`, 111.32366494150033`}, {6284.664031478106, 
     1085.5499726264777`, 111.32366494150033`}}}], 
    Line3DBox[{{{5942.90133782728, -1767.8917802583878`, 1498.097470842783}, {
     5983.876089355275, -1637.0032556876024`, 1483.6647121605322`}, {
     5950.7108693693835`, -1790.9606902360026`, 1438.5109124363498`}, {
     5949.908478816625, -1790.7191980899795`, 1442.1260476456328`}, {
     6026.349726171628, -1480.2134642958029`, 1476.4453149333215`}, {
     5942.90133782728, -1767.8917802583878`, 1498.097470842783}}}], 
    Line3DBox[{{{4515.660126446829, 3054.275283517868, 3311.801595040085}, {
     4513.00791762871, 3056.9147473849102`, 3312.9813541040676`}, {
     4502.88477747256, 3063.6741672694257`, 3320.5034208101465`}, {
     4479.325069977728, 3075.5399664374436`, 3341.3342428309957`}, {
     4472.54464174001, 3078.344741056749, 3347.827918643816}, {
     4471.574884721156, 3086.437925282042, 3341.6661523425205`}, {
     4480.683837030509, 3089.4357527254615`, 3326.6609043236163`}, {
     4487.727210598467, 3086.0596174052394`, 3320.2942958531144`}, {
     4503.234174742831, 3080.2435650483817`, 3304.6626510744322`}, {
     4520.16452925273, 3069.271866999167, 3291.7315390129993`}, {
     4519.754030884046, 3066.4581707174225`, 3294.916004627094}, {
     4515.660126446829, 3054.275283517868, 3311.801595040085}}}], 
    Line3DBox[{{{3551.287462972689, 2979.884000384882, 4381.376768731854}, {
     3374.0553348852472`, 3252.601901706651, 4327.14052881885}, {
     3284.407183284696, 3436.9357928842437`, 4252.9937103150105`}, {
     3292.521532194259, 3536.98030753244, 4163.758633033967}, {
     3392.7214437040525`, 3394.6958234100375`, 4201.582459259027}, {
     3533.323304528547, 3314.1452331311184`, 4149.68450480151}, {
     3575.271220676256, 3166.842798396205, 4228.044001527208}, {
     3551.287462972689, 2979.884000384882, 4381.376768731854}}}], 
    Line3DBox[{{{3787.9134598904243`, 478.52475299560444`, 
     5109.847294308456}, {3735.5145648462494`, 647.4737030414765, 
     5129.768262766973}, {3624.4053046715367`, 552.4541715105333, 
     5219.7982974296465`}, {3625.4949221664383`, 603.4478136078104, 
     5213.391506073308}, {3623.998834906972, 603.1987964101386, 
     5214.460408000263}, {3686.0842023995206`, 709.8279348596709, 
     5157.202850937079}, {3656.9122230926328`, 929.8698385864046, 
     5142.970811299923}, {3640.0290715442584`, 923.322563275997, 
     5156.110694356559}, {3891.610605185196, 1030.630341450288, 
     4947.825955598295}, {3982.5898281281748`, 853.7932795566088, 
     4908.953960537211}, {4082.427516154492, 1005.2590835374566`, 
     4796.983785704314}, {4198.090380226197, 973.0651021319792, 
     4702.870948310739}, {4244.657476835224, 728.0944369239904, 
     4705.377252173004}, {4246.757456439762, 715.7441658205179, 
     4705.377252173004}, {4264.912486643201, 567.7981715214996, 
     4709.133721576799}, {4144.442455325039, 599.684695014598, 
     4811.630564690581}, {4119.888736210329, 459.69271231581985`, 
     4847.990625144708}, {4065.362848735844, 436.8542490936986, 
     4895.895930589943}, {4013.6058554348338`, 423.0274182079552, 
     4939.618358807037}, {3787.9134598904243`, 478.52475299560444`, 
     5109.847294308456}}}], 
    Line3DBox[{{{6344.314388779525, -343.5957852557803, 565.1818845995894}, {
       6348.075683725083, 0, 624.2986494445696}, {6341.192410880978, 
       132.8290456345542, 677.8266318324306}, {
       6269.849711620262, 0, 1173.3704801533581`}, {
       6267.785615873991, -9.116131078596807, 1184.3116395599811`}, {
       6259.371515807334, 0, 1228.0398678788056`}, {
       6258.634563863894, -16.385104406710447`, 1231.681214585082}, {
       6253.850929551926, -309.5114883768675, 1217.1133388083663`}, {
       6284.627508902851, -294.5433452330213, 1050.9590772815511`}, {
       6344.314388779525, -343.5957852557803, 565.1818845995894}}}], 
    Line3DBox[{{{5127.648284453179, -479.2910725339975, 
     3763.6839437887947`}, {5128.615689876043, -479.5268703429946, 
     3762.335548057394}, {5129.063482293215, -479.39265139556886`, 
     3761.742172238033}, {5130.0024906763465`, -479.7856918683625, 
     3760.411390464197}, {5130.560377513939, -479.5289469918496, 
     3759.6829509522363`}, {5130.715259761546, -479.9164890948522, 
     3759.4221333530613`}, {5130.377269220716, -480.35908662099, 
     3759.8268461943876`}, {5129.778616164389, -480.4872806246629, 
     3760.6272099611897`}, {5129.375220850341, -480.8522812571428, 
     3761.130763126472}, {5128.921437540279, -480.9099782138519, 
     3761.742172238033}, {5128.6902229389725`, -480.77361839715076`, 
     3762.0748284668007`}, {5128.340934409293, -480.9115296514738, 
     3762.5333289035843`}, {5127.851306052906, -480.43495893729545`, 
     3763.261474205908}, {5127.537811497174, -480.4714902397387, 
     3763.6839437887947`}, {5127.648284453179, -479.2910725339975, 
     3763.6839437887947`}}}], 
    Line3DBox[{{{4343.079269106449, 2121.389942910638, 4162.352803688725}, {
     4426.95515286716, 1943.2974482017885`, 4160.946622142048}, {
     4480.70680259036, 1863.6091857880328`, 4139.811715750722}, {
     4588.978422140355, 1902.3811972532492`, 4001.254367253788}, {
     4576.995328953453, 2044.1941806085385`, 3944.641631467231}, {
     4663.82294061001, 1952.5136322098242`, 3888.9921296750654`}, {
     4716.97777425179, 2021.6992563821086`, 3788.2273537841747`}, {
     4699.051974278252, 1813.213251018788, 3913.9471048241758`}, {
     4641.664405247335, 1895.803411448748, 3943.1833160784195`}, {
     4609.403220930995, 1978.7677915239115`, 3940.265684455624}, {
     4628.11426440287, 1719.6479703785967`, 4038.7105496189574`}, {
     4612.419227341069, 1680.302906599994, 4073.0804931066637`}, {
     4505.03327916477, 1727.8168607464763`, 4172.186206215886}, {
     4409.931250088139, 1865.8520239232048`, 4214.132922003972}, {
     4266.024939939057, 2114.5806148073907`, 4244.690062336561}, {
     4343.079269106449, 2121.389942910638, 4162.352803688725}}}], 
    Line3DBox[{CompressedData["
1:eJx1VWlUkwcWzZEIVnHoDMoSUUQhCGLRpmxS8bFXC1URC2qlLBUURvbVIBA2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      "], {{1253.7281075154324`, -1744.7472834839539`, 6005.983386523903}, {
      1371.825427408354, -1745.3676171791976`, 5979.933157454027}, {
      1343.1654174158725`, -1819.607974015696, 5964.289326680635}, {
      1253.7281075154324`, -1744.7472834839539`, 6005.983386523903}}}], 
    Line3DBox[{{{2955.6339988260693`, -5494.835593440207, 
     1326.2063022192415`}, {2954.3640756087866`, -5495.83105789142, 
     1324.9104052230502`}, {2953.1533434966527`, -5496.89097194215, 
     1323.211495708612}, {2950.514363382978, -5498.721760303126, 
     1321.4907053171567`}, {2946.0373558925776`, -5500.928025059016, 
     1322.2966576399492`}, {2944.8672457060243`, -5501.211418346572, 
     1323.7233569825364`}, {2945.0206417521667`, -5500.759583108744, 
     1325.2588873465356`}, {2945.568828922152, -5500.237704684785, 
     1326.2063022192415`}, {2949.461171274059, -5497.19652004593, 
     1330.1589788974723`}, {2949.1361287689733`, -5495.6697351138855`, 
     1337.1701262527608`}, {2951.3170883632706`, -5493.63402120325, 
     1340.7186039632427`}, {2955.556915478287, -5490.579943657415, 
     1343.8857435345474`}, {2960.5633940386183`, -5487.436660065323, 
     1345.703155101329}, {2963.140144124437, -5486.139094693116, 
     1345.322269925687}, {2965.4617143797263`, -5485.284687489956, 
     1343.6898480439102`}, {2966.110317804476, -5485.59185511928, 
     1341.0015927145078`}, {2964.979251425879, -5488.329905531936, 
     1332.2712098735174`}, {2965.260786479631, -5488.851041473856, 
     1329.4947922717004`}, {2963.5955649500193`, -5490.029715607681, 
     1328.3405960645848`}, {2957.9707500967083`, -5493.249250281144, 
     1327.567476950972}, {2955.6339988260693`, -5494.835593440207, 
     1326.2063022192415`}}}], 
    Line3DBox[{{{2914.237396500026, -5391.321058074309, 
      1768.8107122033352`}, {2918.3119803756267`, -5394.424307743395, 
      1752.5567788871444`}, {2916.60245568471, -5398.196165942685, 
      1743.76660121004}, {2913.585592999174, -5400.6785796407, 
      1741.121351795377}, {2909.992216396317, -5402.61215832936, 
      1741.1320619598714`}, {2906.7532314640603`, -5403.848050255408, 
      1742.7063984286212`}, {2901.575167715544, -5404.527370824839, 
      1749.2167201641041`}, {2895.8188413862713`, -5402.9887000397375`, 
      1763.4511717411342`}, {2892.5373085116325`, -5398.812162580024, 
      1781.5355292685804`}, {2893.8374443298553`, -5395.9663640731005`, 
      1788.0339855791779`}, {2896.5453999735487`, -5393.758258987407, 
      1790.310107075643}, {2899.39561912273, -5392.109582020604, 
      1790.6627234891284`}, {2901.161679703824, -5391.219910575821, 
      1790.4810733512154`}, {2908.076879064541, -5388.801834696378, 
      1786.537808674417}, {2911.4426901605552`, -5387.274674658999, 
      1785.6614269475704`}, {2914.2893092646773`, -5385.618355353177, 
      1786.0141215535089`}, {2918.023241079456, -5382.54925213473, 
      1789.166734226858}, {2918.3996297187596`, -5381.584519844913, 
      1791.4534174862022`}, {2914.735256681756, -5381.728961209629, 
      1796.9767559804043`}, {2914.2413921290986`, -5381.086221933568, 
      1799.7004946707582`}, {2916.978334261509, -5378.694800078769, 
      1802.4131976476885`}, {2922.4752422453666`, -5375.032948728173, 
      1804.4314751603624`}, {2926.37849920125, -5373.113235666026, 
      1803.8228089596828`}, {2928.560042637195, -5372.720009382478, 
      1801.4520441715836`}, {2929.200259219074, -5374.162391168095, 
      1796.1008114093513`}, {2930.637391521154, -5374.053305483208, 
      1794.0817259096905`}, {2937.3754792176474`, -5371.834808808742, 
      1789.7010283736342`}, {2940.535094594521, -5371.041745263712, 
      1786.890488556176}, {2948.0889302685428`, -5367.834333243348, 
      1784.079572239428}, {2950.3649710025784`, -5367.048972944254, 
      1782.6793177402892`}, {2949.86783762932, -5368.0531258939955`, 
      1780.4772023532985`}, {2946.584479669117, -5370.291922006437, 
      1779.1622372918916`}, {2928.7875582833462`, -5380.308321735119, 
      1778.2855483618214`}, {2919.0039175482284`, -5385.188171028305, 
      1779.6005680924318`}, {2914.221800678615, -5387.717504392813, 
      1779.782312337317}, {2911.358391157093, -5389.6147640740655`, 
      1778.7238973809401`}, {2910.690064167285, -5390.582256244329, 
      1776.8849087719911`}, {2910.991938642938, -5391.141326140536, 
      1774.6928818067354`}, {2912.6975252614434`, -5391.26121229611, 
      1771.5274079526232`}, {2914.237396500026, -5391.321058074309, 
      1768.8107122033352`}}, {{2956.7436652524, -5373.808129708032, 
      1751.4755987662702`}, {2957.1895530038605`, -5374.152139490598, 
      1749.6663736876983`}, {2956.6515574408622`, -5374.928911990713, 
      1748.1889053900463`}, {2954.937749702204, -5376.389824263121, 
      1746.5935540201763`}, {2952.591591110283, -5377.901162375285, 
      1745.9082661547263`}, {2948.831909183882, -5380.064367528803, 
      1745.5977379183987`}, {2945.3298255419086`, -5381.895557501641, 
      1745.8654349397532`}, {2941.7697733636946`, -5383.352510960214, 
      1747.3751838864139`}, {2939.7208076779025`, -5383.911971631537, 
      1749.0989522117354`}, {2939.395827970331, -5383.584855440905, 
      1750.6512961689273`}, {2940.6298885612114`, -5382.472021591232, 
      1752.0001384954485`}, {2943.2384222791443`, -5380.638135362053, 
      1753.2525590762264`}, {2948.4383854721304`, -5377.720661846866, 
      1753.4666407490615`}, {2953.6210821278064`, -5375.060945551756, 
      1752.8993196457186`}, {2956.7436652524, -5373.808129708032, 
      1751.4755987662702`}}}], 
    Line3DBox[{{{-5059.263979323919, 3590.915190488143, 
     1482.2246024934966`}, {-5058.212880996719, 3591.4968853079845`, 
     1484.4009779510131`}, {-5058.061941625176, 3590.8585965071993`, 
     1486.4580803283031`}, {-5059.046224256039, 3588.9020543543684`, 
     1487.8329997320614`}, {-5063.976873458529, 3580.4555850911297`, 
     1491.4052830357411`}, {-5066.299838082588, 3575.205809562305, 
     1496.1026234574545`}, {-5067.919913434704, 3569.33114890923, 
     1504.6184202110192`}, {-5068.297948959348, 3568.40649607353, 
     1505.537999970342}, {-5070.625163402029, 3565.5463487040865`, 
     1504.477775680685}, {-5075.615229060663, 3559.6638548502897`, 
     1501.5781621347073`}, {-5078.055158425778, 3557.277725559327, 
     1498.98121443359}, {-5078.179281869405, 3558.157482386656, 
     1496.4705816447565`}, {-5076.888030125734, 3561.217500790025, 
     1493.5700625122668`}, {-5071.823946038673, 3571.2862384457144`, 
     1486.7179132158744`}, {-5071.373370448995, 3573.3526596315737`, 
     1483.2857436801419`}, {-5070.524118828028, 3577.524016437542, 
     1476.1167847415884`}, {-5070.766053145603, 3579.816061560055, 
     1469.7150368907608`}, {-5069.147804747112, 3583.580879768323, 
     1466.1181063565173`}, {-5067.0559545816395`, 3587.01177929786, 
     1464.9587496981458`}, {-5064.6091977342085`, 3590.058717274038, 
     1465.9555828028151`}, {-5063.393535342077, 3591.056202681978, 
     1467.7107839882337`}, {-5063.154174149578, 3589.8240086240808`, 
     1471.5458150862428`}, {-5059.263979323919, 3590.915190488143, 
     1482.2246024934966`}}}], 
    Line3DBox[{{{115.98559223723758`, -6133.562746215638, 
     1747.5036682734187`}, {191.08371075478888`, -6137.24550308293, 
     1727.8652552709455`}, {70.0786213091108, -6176.979969234781, 
     1589.9123046595053`}, {-10.81387943010952, -6195.890223825407, 
     1516.1270013809603`}, {-242.39509911875354`, -6169.367242279007, 
     1602.4874838256715`}, {-185.4051554144633, -6126.745374857766, 
     1765.3412502982821`}, {-46.35014249689688, -6128.346714560566, 
     1768.9069790198773`}, {-104.21839417059526`, -6071.8859453216955`, 
     1951.7052244950712`}, {91.85450647915407, -6072.085568922601, 
     1951.7052244950712`}, {115.98559223723758`, -6133.562746215638, 
     1747.5036682734187`}}}], 
    Line3DBox[{{{4143.566978204692, -198.35205804132363`, 
      4845.546702716052}, {4142.70086513755, -195.91192438732887`, 
      4846.386463115735}, {4140.391668818227, -192.77907710259188`, 
      4848.484994945543}, {4139.768510296533, -191.23673527937237`, 
      4849.078147672701}, {4139.76219586977, -189.3466692384303, 
      4849.157709385808}, {4140.965966316485, -188.77163752862495`, 
      4848.152207251062}, {4145.854940186613, -189.24829100590057`, 
      4843.9535081668155`}, {4146.971062782621, -190.56126342897028`, 
      4842.946530710817}, {4148.065784971635, -192.5051700440046, 
      4841.932020697314}, {4147.93964330465, -197.17185412559763`, 
      4841.852296946114}, {4147.541788580737, -199.30052912803149`, 
      4842.105957232429}, {4146.886516721425, -200.78160934253336`, 
      4842.605977444775}, {4145.611622791731, -200.71988224826325`, 
      4843.699979545012}, {4143.91338614647, -199.24941110555335`, 
      4845.213639282605}, {4143.566978204692, -198.35205804132363`, 
      4845.546702716052}}, {{4144.535937305528, -176.95396627536107`, 
      4845.546702716052}, {4144.610919158695, -175.18895680980137`, 
      4845.546702716052}, {4144.847635993378, -174.19163313436366`, 
      4845.380174903315}, {4146.279546705512, -172.73670838092266`, 
      4844.207018713099}, {4147.5549614109295`, -172.78984298884114`, 
      4843.113172561165}, {4148.481876769807, -174.09050270380925`, 
      4842.272638439599}, {4148.412731669962, -175.7304547816546, 
      4842.272638439599}, {4148.0529727544335`, -177.22742837576718`, 
      4842.5262687831055`}, {4146.367565406626, -177.6629203360292, 
      4843.9535081668155`}, {4145.177237153466, -177.98881384097803`, 
      4844.960200531294}, {4144.535937305528, -176.95396627536107`, 
      4845.546702716052}}}], 
    Line3DBox[{{{6130.624028970588, -1449.2181099648774`, 
     1001.5136420230403`}, {6048.70158444518, -1622.6306739562629`, 
     1211.6486818655826`}, {6045.569626669418, -1475.6142376965004`, 
     1400.5228311596964`}, {6106.496402577681, -1225.7410514311023`, 
     1376.9801197339118`}, {6107.70622062802, -1227.8322469964182`, 
     1369.732231400546}, {6153.482292385798, -972.7810553032452, 
     1369.732231400546}, {6218.0202503409055`, -872.0413568467766, 
     1124.0915124220348`}, {6217.949561544056, -870.1872152655358, 
     1125.91791462899}, {6254.244566967937, -930.9835945047535, 
     839.9453156694467}, {6207.896863091848, -1124.4355723826395`, 
     940.9967432328986}, {6161.802253196309, -1221.932476738101, 
     1107.6496308840485`}, {6130.624028970588, -1449.2181099648774`, 
     1001.5136420230403`}}}], 
    Line3DBox[{{{5968.518527532095, -1792.5339798368225`, 
     1360.6697643574216`}, {6027.455010069217, -1615.051702362971, 
     1322.576194510536}, {6026.851782498171, -1671.39392269753, 
     1253.5204972270635`}, {6048.70158444518, -1622.6306739562629`, 
     1211.6486818655826`}, {6045.569626669418, -1475.6142376965004`, 
     1400.5228311596964`}, {5968.518527532095, -1792.5339798368225`, 
     1360.6697643574216`}}}], 
    Line3DBox[{{{3434.923520367794, -5340.186718742869, 609.524329445825}, {
     3294.8871194788744`, -5415.587885167822, 709.1829227468247}, {
     3306.920770288006, -5392.882365845533, 817.8682639582336}, {
     3155.631929290911, -5462.044935953534, 946.5019463604218}, {
     3038.602290693107, -5569.335562156466, 661.2194832987964}, {
     3105.553049444898, -5541.582218077311, 578.1180393075803}, {
     3212.011242465024, -5489.364629405228, 487.5176240025507}, {
     3260.4346921766064`, -5480.298003317949, 153.99059348067615`}, {
     3521.693796109185, -5313.979269887928, 217.05026064817548`}, {
     3367.2516082730035`, -5399.207793152568, 444.95561907165205`}, {
     3434.923520367794, -5340.186718742869, 609.524329445825}}}], 
    Line3DBox[{{{1897.774430692228, -5760.815975038335, 1974.655717322365}, {
     1618.8686604243842`, -5830.864848897769, 2016.9514283155577`}, {
     1833.7309529969302`, -5763.197668948677, 2027.5100765829773`}, {
     1710.1029148533444`, -5754.764418233105, 2155.4693691278217`}, {
     1880.6608219968, -5703.2834010629695`, 2150.2295249033136`}, {
     1897.774430692228, -5760.815975038335, 1974.655717322365}}}], 
    Line3DBox[{{{191.08371075478888`, -6137.24550308293, 
     1727.8652552709455`}, {736.646547978166, -6117.156107894091, 
     1650.929042995449}, {292.79633716029514`, -6208.731335292498, 
     1433.087297097614}, {234.55821973292603`, -6199.748940194648, 
     1481.8600507845056`}, {70.0786213091108, -6176.979969234781, 
     1589.9123046595053`}, {191.08371075478888`, -6137.24550308293, 
     1727.8652552709455`}}}], Line3DBox[{CompressedData["
1:eJx92Xc0l2/8P3CjrOyEFJWQlZFRKtxllAYym1YaklFGUioZWZEoJaH7rYwQ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      "], {{-2424.4886798070497`, 5369.997056246674, 
      2443.931248498926}, {-2423.914324101643, 5370.124354632893, 
      2444.221236957393}, {-2423.266243978444, 5370.51323839092, 
      2444.009401999607}, {-2423.0598298943046`, 5370.555829748062, 
      2444.120461497689}, {-2423.3664677144798`, 5370.185319419988, 
      2444.630501372975}, {-2422.9288012119623`, 5370.065393751496, 
      2445.3276633388314`}, {-2423.1292575659636`, 5369.809647544313, 
      2445.6906264423246`}, {-2423.0659370102376`, 5369.644327585822, 
      2446.1162993868684`}, {-2423.370523660149, 5369.469399450525, 
      2446.198553556019}, {-2423.739910885472, 5369.438037275322, 
      2445.9014081196747`}, {-2425.566212440804, 5368.660590377726, 
      2445.7975599772467`}, {-2425.730139024769, 5368.648823810605, 
      2445.660808293719}, {-2425.5044445761746`, 5368.8735049490615`, 
      2445.391413904367}, {-2424.3375038461, 5369.537274613312, 
      2445.091166804115}, {-2424.007231138137, 5369.605375484318, 
      2445.269053695307}, {-2423.888995940259, 5369.768313209785, 
      2445.0284431490463`}, {-2424.6600455356856`, 5369.851819481556, 
      2444.080356779111}, {-2424.4886798070497`, 5369.997056246674, 
      2443.931248498926}}, {{-2436.5825500200813`, 5363.924333192278, 
      2445.2392346609786`}, {-2437.1079757571397`, 5363.937322286558, 
      2444.687057085682}, {-2436.9851688318763`, 5364.1144913129165`, 
      2444.4207291258676`}, {-2436.006274684982, 5364.346361390835, 
      2444.8875709791437`}, {-2435.838885110416, 5364.251303091377, 
      2445.2628842450677`}, {-2435.878884864708, 5364.016100423146, 
      2445.7389522742474`}, {-2436.052884288709, 5363.9019191543375`, 
      2445.816067622641}, {-2436.5825500200813`, 5363.924333192278, 
      2445.2392346609786`}}, {{-2426.208758808706, 5369.658156957364, 
      2442.9686911831604`}, {-2425.6013243015345`, 5369.812330438254, 
      2443.232989589804}, {-2425.138600044652, 5369.8121555052085`, 
      2443.692671994981}, {-2425.1502399808933`, 5369.688024031172, 
      2443.953871924924}, {-2425.34711329133, 5369.5742977908085`, 
      2444.008373666863}, {-2425.8532694535506`, 5369.445810054555, 
      2443.7883087471187`}, {-2425.9283387025266`, 5369.1624030039275`, 
      2444.3364080357455`}, {-2426.1025807092933`, 5369.048560715608, 
      2444.413531003574}, {-2426.227909972006, 5369.076193866531, 
      2444.228435177153}, {-2426.332479842773, 5369.5573354152175`, 
      2443.0674180113497`}, {-2426.208758808706, 5369.658156957364, 
      2442.9686911831604`}}, {{-2425.41960303565, 5370.059569484635, 
      2442.8699636691426`}, {-2426.6078474055416`, 5369.742146113842, 
      2442.3876287679145`}, {-2426.5870035259495`, 5369.870843307305, 
      2442.1253715804696`}, {-2425.7731491887757`, 5370.542501781124, 
      2441.4568509882206`}, {-2424.8896008448314`, 5371.0850188316035`, 
      2441.1410926444423`}, {-2424.539232859169, 5371.183817725571, 
      2441.2717166529465`}, {-2424.069485780049, 5371.068094783685, 
      2441.992698454851}, {-2424.9149922936995`, 5370.716349243557, 
      2441.9268756695424`}, {-2425.453765044173, 5370.335092648396, 
      2442.2302750359527`}, {-2425.41960303565, 5370.059569484635, 
      2442.8699636691426`}}, {{-2431.1199952620045`, 5370.083351209386, 
      2437.1446530111907`}, {-2431.006187872346, 5369.931744128558, 
      2437.5921865494197`}, {-2430.6607277354137`, 5369.941944602844, 
      2437.9141962423064`}, {-2430.5748481706896`, 5369.777162334337, 
      2438.362734359394}, {-2430.1895115386437`, 5369.774041790415, 
      2438.7536505603416`}, {-2430.817295461888, 5369.339891718838, 
      2439.0838635192695`}, {-2431.100888362752, 5369.218036249465, 
      2439.0694618676375`}, {-2431.4101596452992`, 5369.202757756012, 
      2438.79479901013}, {-2431.3748046356063`, 5369.3491250366515`, 
      2438.5077860961005`}, {-2431.7817737879272`, 5369.449785360898, 
      2437.8802466816624`}, {-2431.6241094561083`, 5369.750035525331, 
      2437.376137629867}, {-2431.1199952620045`, 5370.083351209386, 
      2437.1446530111907`}}, {{-2429.877023980054, 5370.88023436517, 
      2436.6281715599066`}, {-2429.2574365354744`, 5371.008333324662, 
      2436.9635779785726`}, {-2429.046442249725, 5370.891342366295, 
      2437.4316933790624`}, {-2429.1611855694564`, 5370.470998648554, 
      2438.2433999085474`}, {-2429.443938959876, 5370.521969178742, 
      2437.8493833745356`}, {-2429.5888379891107`, 5370.767396463784, 
      2437.164200746505}, {-2429.9593458015142`, 5370.712752228688, 
      2436.915222325149}, {-2430.004834993712, 5370.788333570105, 
      2436.703278590651}, {-2429.877023980054, 5370.88023436517, 
      2436.6281715599066`}}, {{-2420.6815886219747`, 5375.592804447768, 
      2435.3883339779677`}, {-2420.5841934846835`, 5375.476756521754, 
      2435.7412613712613`}, {-2421.219248031705, 5374.857321365719, 
      2436.4769274347964`}, {-2421.5130686159832`, 5375.108741854599, 
      2435.6301364863007`}, {-2421.2171342444126`, 5375.203353950685, 
      2435.715538095239}, {-2420.6815886219747`, 5375.592804447768, 
      2435.3883339779677`}}, {{-2430.0294499758306`, 5376.338456245842, 
      2424.4082507164967`}, {-2429.573554681023, 5376.454870610666, 
      2424.6069901494266`}, {-2429.4722505045625`, 5376.505756016512, 
      2424.5956631054173`}, {-2428.717737054622, 5376.886565394392, 
      2424.5071059078323`}, {-2428.181819211195, 5376.975839014415, 
      2424.8458857229334`}, {-2427.7245020709943`, 5376.888824650278, 
      2425.496650048211}, {-2427.845677389567, 5376.531688643066, 
      2426.1669476352463`}, {-2427.0069752878994`, 5376.22531855259, 
      2427.684525645372}, {-2427.0920703243473`, 5376.0134855553615`, 
      2428.0685277181865`}, {-2427.472019302623, 5375.80427103905, 
      2428.1519155483556`}, {-2427.7345590229193`, 5375.8103263933435`, 
      2427.8760132198845`}, {-2428.5861684319902`, 5375.369855820032, 
      2427.999552232693}, {-2429.557818163193, 5375.120016976103, 
      2427.5805447615767`}, {-2430.168333186317, 5375.070911738998, 
      2427.07813144854}, {-2430.023241308549, 5375.224832074628, 
      2426.8825149455806`}, {-2429.1514317278447`, 5375.3957411121555`, 
      2427.3766988598754`}, {-2428.943535550601, 5375.535708989384, 
      2427.2747748219494`}, {-2429.2688571420663`, 5375.555583175187, 
      2426.9051654141235`}, {-2429.5210421397783`, 5375.738594868981, 
      2426.2472576875007`}, {-2430.2796548478154`, 5375.792020239791, 
      2425.368970766915}, {-2430.0294499758306`, 5376.338456245842, 
      2424.4082507164967`}}, {{-2426.997987555184, 5377.131332422809, 
      2425.686107533289}, {-2426.0767686688523`, 5376.992332833067, 
      2426.9154610698147`}, {-2426.0727955911993`, 5376.833328442261, 
      2427.2716862034267`}, {-2426.3606119253927`, 5376.570054374093, 
      2427.567161028668}, {-2426.7564661732945`, 5376.546178434239, 
      2427.2243273027498`}, {-2427.1965602923597`, 5376.770400475603, 
      2426.287412545009}, {-2427.199401462611, 5377.026948117988, 
      2425.7159674521877`}, {-2426.997987555184, 5377.131332422809, 
      2425.686107533289}}, {{-2432.8551293779433`, 5375.414256773931, 
      2423.6235634955756`}, {-2432.677810654936, 5375.422072272988, 
      2423.7842116136267`}, {-2432.6791176019824`, 5375.250127839049, 
      2424.1641990487205`}, {-2432.2991632976714`, 5375.309678412166, 
      2424.41339944118}, {-2432.433460797868, 5375.031987719608, 
      2424.894282186731}, {-2432.2715461455214`, 5374.998873672177, 
      2425.1300838993557`}, {-2431.97607382254, 5374.945330358348, 
      2425.5450443176055`}, {-2431.8151761045574`, 5375.214191680863, 
      2425.110519712493}, {-2431.2501498730094`, 5375.56407847382, 
      2424.9014901567402`}, {-2430.9688551290787`, 5375.566802932158, 
      2425.1774497150022`}, {-2430.9245548780746`, 5375.243949545501, 
      2425.9373407747607`}, {-2430.2887290176627`, 5375.33720445713, 
      2426.367721922776}, {-2430.2565054882425`, 5375.190957191344, 
      2426.7239606639396`}, {-2430.546936297846, 5375.033605847826, 
      2426.7816169692032`}, {-2430.7613878172265`, 5374.783214978458, 
      2427.1213726311044`}, {-2431.1797211512467`, 5374.60888075426, 
      2427.088426980027}, {-2431.560572604872, 5374.301745553005, 
      2427.386994176944}, {-2431.988425798027, 5374.123491790152, 
      2427.3530196025577`}, {-2432.705744509169, 5374.110353329842, 
      2426.6632153772557`}, {-2434.299637264014, 5373.686561260734, 
      2426.003237290103}, {-2434.7567397264893`, 5373.62263432002, 
      2425.686107533289}, {-2433.9415952476475`, 5374.143862546433, 
      2425.349406905703}, {-2433.648811314191, 5374.495761364534, 
      2424.863390845696}, {-2433.0637253164155`, 5374.651480509487, 
      2425.1053712378343`}, {-2432.979782904976, 5374.76567263669, 
      2424.936500245299}, {-2433.277261633966, 5374.873489600264, 
      2424.3989830074133`}, {-2433.0735875689466`, 5375.247587105852, 
      2423.7739137112376`}, {-2432.8551293779433`, 5375.414256773931, 
      2423.6235634955756`}}, {{-2408.4565803512633`, 5386.896072884056, 
      2422.4989763690373`}, {-2408.1861482633185`, 5386.871540899437, 
      2422.8223556753833`}, {-2408.33141760319, 5386.515203677893, 
      2423.4701222693707`}, {-2407.95533535375, 5386.582328693034, 
      2423.6946196155454`}, {-2408.0614411674032`, 5386.340265435916, 
      2424.1271275464305`}, {-2407.4142824052774`, 5386.5326832766605`, 
      2424.342346877484}, {-2407.1139822250257`, 5386.264533282103, 
      2425.236141921877}, {-2406.562004212919, 5386.316444128606, 
      2425.6686034140093`}, {-2407.1470332457734`, 5385.959953739065, 
      2425.879681075538}, {-2407.8037564506476`, 5385.688193991426, 
      2425.8312879348236`}, {-2408.468980788274, 5385.208451040031, 
      2426.2359319381358`}, {-2409.126881221187, 5384.98977859098, 
      2426.0681038767543`}, {-2409.3308117659317`, 5384.991775506726, 
      2425.8611475515727`}, {-2409.187426355826, 5385.251181715667, 
      2425.427662190526}, {-2409.749666259531, 5385.3228159448545`, 
      2424.7099628668707`}, {-2409.5144243594855`, 5386.309968839243, 
      2422.750265377463}, {-2408.932497930033, 5386.547552913247, 
      2422.8007286239786`}, {-2408.4565803512633`, 5386.896072884056, 
      2422.4989763690373`}}, {{-2405.0239563475834`, 5387.670327443575, 
      2424.186853808563}, {-2404.757493232108, 5387.654428977504, 
      2424.4865111324325`}, {-2404.3838732635973`, 5387.75213183345, 
      2424.6399414993675`}, {-2404.5823091872517`, 5387.4893135633865`, 
      2425.0271141956337`}, {-2404.516320452261, 5387.316201803654, 
      2425.477086361066}, {-2404.8408330612697`, 5387.133799207903, 
      2425.5604892628426`}, {-2405.043547136197, 5387.082708609518, 
      2425.4729676866896`}, {-2405.036315572256, 5387.2685783041725`, 
      2425.0672724679557`}, {-2405.301057686615, 5387.381656135358, 
      2424.553444044468}, {-2405.0239563475834`, 5387.670327443575, 
      2424.186853808563}}, CompressedData["
1:eJx112k8VtseB3Ah5USOCEWpSDgZQ5Fhk5JO5kNuGUqmciOEjA1olhxkbLA6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      "], {{-2430.7628741209423`, 5376.810902707387, 
      2422.62462142245}, {-2429.9736595411377`, 5377.11519007511, 
      2422.740996598637}, {-2429.7589717849582`, 5377.140280084076, 
      2422.9006247322345`}, {-2429.8096115922604`, 5376.977245958273, 
      2423.2116370347953`}, {-2429.3844668527277`, 5376.886675437537, 
      2423.8387903730227`}, {-2429.5762434702065`, 5376.71088821567, 
      2424.036507915725}, {-2430.3148317546297`, 5376.419794899967, 
      2423.9417685994986`}, {-2430.850782803179, 5376.555362757659, 
      2423.103504381733}, {-2430.7628741209423`, 5376.810902707387, 
      2422.62462142245}}, {{-2393.891256455699, 5394.749223383498, 
      2419.4584433439027`}, {-2394.012224409046, 5393.4983129928805`, 
      2422.1261542543566`}, {-2393.1615572245364`, 5393.866568686595, 
      2422.1467524135937`}, {-2391.566295221981, 5395.096486098587, 
      2420.982910151348}, {-2391.824816725895, 5395.425503480258, 
      2419.994085217759}, {-2393.891256455699, 5394.749223383498, 
      2419.4584433439027`}}, {{-2432.3866628515107`, 5376.652302031892, 
      2421.346492230563}, {-2432.2506447419605`, 5376.576530563586, 
      2421.651358504184}, {-2432.459655932244, 5376.114034713458, 
      2422.468079876176}, {-2433.0553879879935`, 5375.781798237128, 
      2422.607113571034}, {-2433.3086045208847`, 5375.716846542823, 
      2422.4969166049127`}, {-2432.8198144528546`, 5376.385351758161, 
      2421.5040759424405`}, {-2432.3866628515107`, 5376.652302031892, 
      2421.346492230563}}, {{-2434.555470257978, 5376.023597717138, 
      2420.562667857504}, {-2433.700784620458, 5375.833983667106, 
      2421.842926572164}, {-2433.8111957732986`, 5375.578428987551, 
      2422.299177874664}, {-2434.009823490461, 5375.517732540504, 
      2422.2342942610926`}, {-2434.9016988177377`, 5375.539776227731, 
      2421.2888141004573`}, {-2434.555470257978, 5376.023597717138, 
      2420.562667857504}}, {{-2413.2236136744878`, 5385.2439687628885`, 
      2421.4278591993575`}, {-2412.904897480096, 5385.263124968899, 
      2421.702855547471}, {-2412.297778762085, 5385.46980423319, 
      2421.848076216185}, {-2412.0672830148455`, 5385.45910242392, 
      2422.1014364243147`}, {-2412.2904081334286`, 5385.302186176561, 
      2422.2281148540505`}, {-2413.5901119464097`, 5384.827748683509, 
      2421.9881458261953`}, {-2413.2236136744878`, 5385.2439687628885`, 
      2421.4278591993575`}}, CompressedData["
1:eJx11vs/1GkbB/BhmhFLS2wqHmWEViikdFh9s8RGsZaKzWHtYwdRotZYlNML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      "], {{-2404.0740943544674`, 5390.900445462707, 
      2417.939996353039}, {-2403.6883057313294`, 5390.768856256, 
      2418.6168273025028`}, {-2403.848136439271, 5390.545512860309, 
      2418.955745919928}, {-2404.1327889841855`, 5390.425038335158, 
      2418.94132401357}, {-2404.49560347943, 5390.353350006745, 
      2418.7404459592485`}, {-2404.459865504615, 5390.551400142184, 
      2418.3345607261745`}, {-2404.6451296061614`, 5390.562113577696, 
      2418.1264628472454`}, {-2404.0740943544674`, 5390.900445462707, 
      2417.939996353039}}, {{-2405.1925491564257`, 5392.219298225437, 
      2413.88349580662}, {-2404.904032192218, 5392.331374583255, 
      2413.920593687927}, {-2404.0530835019435`, 5392.244811603527, 
      2414.9613565106256`}, {-2403.451276415493, 5392.36263153856, 
      2415.297269878651}, {-2403.478682747677, 5392.221642118246, 
      2415.584747809337}, {-2403.6766295582956`, 5392.108942168676, 
      2415.639357664937}, {-2404.8663455774745`, 5391.740924440384, 
      2415.2766619216895`}, {-2405.0702465202567`, 5391.7427485386725`, 
      2415.0695503201628`}, {-2405.1925491564257`, 5392.219298225437, 
      2413.88349580662}}, {{-2428.3641700776498`, 5381.91163389504, 
      2413.6876998543758`}, {-2427.4869601228734`, 5381.946827746009, 
      2414.4914768431204`}, {-2427.4902225841684`, 5381.603205170696, 
      2415.253993135041}, {-2428.225573402316, 5381.153512022703, 
      2415.5167427945357`}, {-2428.657739416208, 5380.983917163972, 
      2415.4600717040275`}, {-2429.36701787317, 5380.952329402032, 
      2414.817096836295}, {-2428.938493786937, 5381.28032998675, 
      2414.51723818608}, {-2429.0649462061256`, 5381.360100327587, 
      2414.2122209356194`}, {-2428.9087867354715`, 5381.540142755367, 
      2413.9679963975823`}, {-2428.6166083971484`, 5381.6442710047095`, 
      2414.029825785034}, {-2428.3641700776498`, 5381.91163389504, 
      2413.6876998543758`}}, {{-2414.941057148027, 5387.589831541565, 
      2414.487355023981}, {-2416.008532659114, 5387.275622334854, 
      2414.120508407764}, {-2416.3111021460286`, 5387.295530380576, 
      2413.773232096857}, {-2416.5871004525443`, 5387.583511295994, 
      2412.853991591262}, {-2415.5269237291855`, 5387.712739248418, 
      2413.6268995185897`}, {-2414.941057148027, 5387.589831541565, 
      2414.487355023981}}, CompressedData["
1:eJx11PszXGcYB3AWuy7NrgmmiSRNpEPCkCqayCzpsZNdRUowIgxBtyKrsSIZ
l7rLaFS3CdIfcqEab7btZjpVupHLKk6LNS51Z3ftiksjqqTbBNmEqH73D+gP
55fPPPN9nvO85z1OH6VHJDNMTEw88ZjhuXJSqvDTEZoVojy1yW6i5hkvo4UD
hBJ7Zn4q0xLa9XL8t2xOE8Vs8PpKDj8x+mzDf4LQz4P5HdHwkDJFtgqubiuN
SNAQOndCNpMCT1p4K008SKg7oVYVDNRbXdIlOMCfFu0S+A0RatJviRcKv8Da
/54efWO7wwKr4YPCtuEouG2XxOIcXHfJf7xvhFC2rwo8CpF/5NBGyvKWJiow
yM0wNE4o/c246Bk1od3e5O9Zgme0dZ6pUBGqx7nSnAcftnMetDS+1yEzST1c
4kJpGsYJnWbwMS+Bi9SaZBf4Sq+aJ4e7j/U9i4MXFA5wO+FldWUPs1SEdj6u
/MAU3j7Vk1OtJlSXh2p6CfXTPS0txvwWlWDZTEMo64vH6a2or6WLHaYwT9CY
xeHr8B12jb/VoP6wKPesDu5eH8Z0nCDUY3bBlW44a5a2uQsfnR+/VQd32Z5V
rUfOBieq5jx8W8HXNrfgibHrAiH2kBYS8AsT3ii2jGBrCfV0dih9Hn7/3rRv
IvyAT++FNcy5/yUdzME5jsoY+/LgK6KHO2PxXs3RSVUd8NNSVuMJuEhbKQ2E
rycJlAY4g7HazoYPaK8NRSFHxz1VEonvpPq2rf3KG02UNpXJy4B3S2quNsCt
jgle2KPe7t841xLkxFS0Rn4Hj3H6a1oIZ4VH17vhvJZpzuZnyD9z5+c9bXDP
qfxU+RihGCo9Pwx++Rt3W1vsU1D1nF88SqjfI007jXMOWhw4aAHf8WHE21x4
iGfgzjzk//GiRmHcW3DmP5lLcGIiuXttktDNlYaqTPiNWvHwgtF9VsK3TxG6
zNu+cg7zL+T3xS6i79Gi4V+9pwld9OdPDCk8SE6SVuF7d/+wt36G0Oc+UVoX
w4+VjzxxhHf1tdRw4cs5mrhmeKuvsrAU+Tn2par7yE8g78Q7oO9ch4MyC15n
dVrhBL99sdzVuGc5U3R+Ed+zjUHyo9MjQi9ezxqpgz8Sa6Q34MuyxkQWPEDB
ze6FN/X3K14NE6r1nmNuFObc/FJeroY/rojPS8f9eu2fkhoI95pdzH4Nb3+w
1qrDffRg1frq0VeWyuCysbcj+Y4P5uCJHasxQuynP6PDchfcy2q3+7vI3/y4
wDocvu7OuxkGz/FO9k3CfXz/6tqcOeZx3De5JRUeFbMt9yD6hvLCbf5G/dbe
KukX8IBkh8Jm5JtGfB7+BOfC5LM438MtTp5VHEX9//1n/gNlMVNA
      "], {{-2403.653994982281, 5394.868403542305, 
      2409.4929074115635`}, {-2403.402350113402, 5394.83548625113, 
      2409.8176092299827`}, {-2403.5711485919282`, 5394.5558239564625`, 
      2410.275271804839}, {-2404.021871286126, 5394.3264247738025`, 
      2410.3391784902465`}, {-2404.10265241754, 5394.2038322127155`, 
      2410.532958326975}, {-2403.7436360165684`, 5394.081902063231, 
      2411.1637555636426`}, {-2403.796429417658, 5393.769923131351, 
      2411.808954341644}, {-2404.476211873067, 5393.371052132692, 
      2412.0233271163956`}, {-2404.644150889181, 5393.3680801474475`, 
      2411.862547833329}, {-2404.4159842582967`, 5393.792833036717, 
      2411.140049629002}, {-2404.5194131347994`, 5393.8729626206605`, 
      2410.85763681054}, {-2405.316310423606, 5393.888459526863, 
      2410.0278884264158`}, {-2405.204896556235, 5393.992945638902, 
      2409.905225933102}, {-2404.3936958622976`, 5394.325118184859, 
      2409.9711958066105`}, {-2404.1650066202446`, 5394.622279652069, 
      2409.5341397922116`}, {-2403.653994982281, 5394.868403542305, 
      2409.4929074115635`}}, {{-2401.4332044485027`, 5396.140364996005, 
      2408.8589447771687`}, {-2401.094339020794, 5396.266307597525, 
      2408.914610900533}, {-2400.4813021130813`, 5396.28316294042, 
      2409.487753355725}, {-2400.3560929972728`, 5396.128498422784, 
      2409.958826478232}, {-2400.6395781776705`, 5395.827433500862, 
      2410.350516743662}, {-2401.007859461181, 5395.716979237557, 
      2410.230949260501}, {-2401.397103412987, 5395.805716904365, 
      2409.644435833196}, {-2401.5814165493666`, 5395.9156284540695`, 
      2409.2145857706114`}, {-2401.4332044485027`, 5396.140364996005, 
      2408.8589447771687`}}, {{-2393.4240986132518`, 5399.745350630346, 
      2408.7517353441176`}, {-2393.869269174379, 5399.70665125138, 
      2408.3960829843036`}, {-2394.583581842822, 5399.486475528348, 
      2408.1795946638326`}, {-2394.5335243103223`, 5399.6024664772285`, 
      2407.969288626211}, {-2393.488087237715, 5400.067798708511, 
      2407.965164947899}, {-2393.4240986132518`, 5399.745350630346, 
      2408.7517353441176`}}, {{-2425.365206005373, 5385.943468683202, 
      2407.702278033409}, {-2425.0435231455995`, 5386.082785146126, 
      2407.714649289264}, {-2424.9556545630753`, 5385.686750559635, 
      2408.6888525189324`}, {-2425.989144029356, 5385.195010886141, 
      2408.7476118885356`}, {-2426.325208942209, 5385.112767633451, 
      2408.592981456754}, {-2426.15095445296, 5385.428701416705, 
      2408.0620710779385`}, {-2425.365206005373, 5385.943468683202, 
      2407.702278033409}}, {{-2426.5640922089215`, 5386.471277136439, 
      2405.3123657825454`}, {-2426.1511649559698`, 5386.408160292041, 
      2405.870184677904}, {-2426.249150163333, 5386.224012092437, 
      2406.183626178715}, {-2425.889250732203, 5386.253459113978, 
      2406.48056451006}, {-2425.8790732813773`, 5386.1304369224235`, 
      2406.7661557104684`}, {-2426.4817028095144`, 5385.711073316037, 
      2407.097104541414}, {-2427.103903496384, 5385.059222425883, 
      2407.928051790285}, {-2428.071650512395, 5385.023488018041, 
      2407.032152562991}, {-2427.9934411944723`, 5385.2513620718555`, 
      2406.6011939704185`}, {-2428.356835240199, 5385.380050560909, 
      2405.9464835616754`}, {-2427.744060744216, 5385.6765853654015`, 
      2405.9011167061817`}, {-2427.2129078798184`, 5385.828011371136, 
      2406.0980490727347`}, {-2426.85526832052, 5386.314309566237, 
      2405.370107735382}, {-2426.5640922089215`, 5386.471277136439, 
      2405.3123657825454`}}, {{-2393.6630116292004`, 5401.022254619834, 
      2405.6495342961975`}, {-2393.341226326785, 5401.008701081968, 
      2406.00009875342}, {-2393.5051525085055`, 5400.844190732232, 
      2406.2063091823898`}, {-2394.1976970242536`, 5400.472905589358, 
      2406.3506547382085`}, {-2394.017181180682, 5400.370990712251, 
      2406.7589386736}, {-2394.2262641421767`, 5400.13033133817, 
      2407.0909186512454`}, {-2394.509750416772, 5399.981196462197, 
      2407.143498633538}, {-2394.83983666357, 5399.962572769263, 
      2406.856883867499}, {-2394.766440347824, 5400.305719459566, 
      2406.159912091487}, {-2394.0662029575947`, 5400.812316840003, 
      2405.7196478653777`}, {-2393.6630116292004`, 5401.022254619834, 
      2405.6495342961975`}}}], 
    Line3DBox[{{{3935.5974163624296`, 1662.461436713762, 4736.571401104493}, {
     3921.1586918773723`, 1596.2049108268254`, 4771.2113855672815`}, {
     4076.71919769098, 1259.356859985795, 4741.53926192587}, {
     4189.709003827124, 1210.6177465161945`, 4654.949712410433}, {
     4235.072898188405, 1365.17088142336, 4570.369755451284}, {
     4147.9225140985145`, 1531.6203515872526`, 4597.465780801244}, {
     3935.5974163624296`, 1662.461436713762, 4736.571401104493}}}], 
    Line3DBox[{{{2618.015577407377, -1024.232465172433, 5725.797410294842}, {
     2474.0156583639623`, -1104.0907038517962`, 5774.776526398054}, {
     2462.450366143286, -979.9280164357697, 5802.059372919675}, {
     2406.3944138162465`, -1098.3483052064732`, 5804.369958540206}, {
     2345.534521166974, -1001.261774542204, 5846.687639925243}, {
     2499.595110207867, -963.6782104982935, 5788.882645350113}, {
     2439.6510455749126`, -707.2452783943867, 5851.12985516593}, {
     2573.953884616514, -624.2903125743559, 5802.8300591602165`}, {
     2707.4461402583916`, -888.4179345703697, 5706.861016454987}, {
     2618.015577407377, -1024.232465172433, 5725.797410294842}}}], 
    Line3DBox[CompressedData["
1:eJx1lFtU0gcAhykts8vWRkst8sJstWZZy1Wmq58Sm2meyHXDLUq0i67I0zLn
OYkmihpDyAsmBKEiGFhpwB9OOSXUTcvCk0UtK41MW6aWU2taa+2ppz185/we
vsfv/HzYB6N2O5FIpAnvWPyO//ZID2l/6sh2FFALm+5STNhgvEtpLSlDmjCq
6fDEH8Ecejl8ydmELdzljsekcrB0Wdd1CcmY5rdTOMAy4TxroLfVpRQedvN9
5TIxMl7xFpeqTOBGMkcZr6QIeb254uugE3hbJZ5BmmnGSn5g1/rIAizwDOR8
5pyP2jBfM9nfDH2bbQpxQYTpb7JeMIIlYOvdDhNOZrzlXx3L8xSiikVX7EuS
YUWthiF6SOCYaumQccspyOp79uxwl2F/UCi7eCoBr5VSxQRHGX4abW+5mXMK
VlrfsrNzCQgEOZWxk8rQkKIUllkVOHuP2TGXRIDKDpjfrSxFdHQ1LVukhNfY
iPYazwgOx62+80kZxsOLg6hSBehXMhkXpUb86S0qXi4sR+i++nO9g0p004eL
1/kZYbt3snL3JyrYFqocWbvluBlADpmZbMQX9HOCzBIVhr1jdrHIp8FrlPZ5
thrQuqfT5dfTamybx0lfwJShsOtBnVOfAb50ge4cS4NA/aHaeUIJ1LHceFOR
ARHjAvd1pjPwKEmg7aQVwa87N4jKNqBcpaPZ5Fr4yENeu+skkGyXV9gv6NFz
qWuUtaUKi8ypoZe8TiCC0WV3vakHkfwyO/R5FYSMRmKVVQBDvmFj4WwDnnpY
OPmaKkzOLpQ9axcBN8S+4d8a8NfBnJSa2zrsishIHSjKA3vp5r/blAZUNZfc
qnZo0XlcvZn9WIzLR6eteFJgwHmb79C1Vi0yTljlY38cR2ro9ZoCuhHNBR2T
V390BmHKOHa8Vx6ip/3j5ZtuRCax8YcDUyrRztN87k/jY1IiVTr0zIg59mtf
ERM12DtjVeC4IhHa/jVcayQBR3ffkgRPNVLkRHD85T1gxjOPfOxM4GhDY/+E
CA0ayNxP1S3RaLGub/yeSuCbtMu+KQ/U+PCMW72MEWfpHHhSd9+VwNWh/FzG
Sg0oHm9uR8ZlWJrvPM0Jzzeic3ptTBGvEqOHYscO/cy3lAxyNmX8ZsTqiz6z
rog06Cdtmzroyrf0j1sF650ISHh5LbRqNabHZLg+epRm2cbdmmv3JxBFEYeL
vlNj73xJm3tToqUj0WXnrVkmnNRKqsWaMnQPbOhZO+eA5WHBDgnZ9q7PmOZf
VoyV40Y+f/QOEWfpJQUfSxokQInuUR/hlKM3IZzsqNthqeZ6rzl2470vTb+y
qQIcS/oHbgEH8gj4+Yexg2dXIKXhefvU5UsQUMN8ocomsOjN7wZFWAXSrH0P
s2tiYVcIzHoaATeGVtS1VY2Fsi8pHXXb0Z70VtjGf+//3z/8C0GKHZY=
     "]], 
    Line3DBox[{{{-579.0698141769305, 6322.153869287789, 
      618.758671340066}, {-830.9265200114313, 6297.3577163530945`, 
      583.6613749682448}, {-1427.556043456679, 6216.41573325871, 
      77.92858322779111}, {-1373.3433346197583`, 6229.042233090095, 
      27.83224093461295}, {-1514.3061074920831`, 6196.264659513724, 
      31.543177922576938`}, {-1584.2632345470258`, 
      6177.757753358385, -115.03405792610378`}, {-1598.825650012387, 
      6167.395973982623, -307.89142723595586`}, {-1764.3111390267438`, 
      6119.279592662188, -359.7737643773222}, {-1726.0242428087977`, 
      6106.4196987235555`, -648.2996732046383}, {-1606.4039158046319`, 
      6137.7932619586545`, -659.3739627423}, {-1280.594954157715, 
      6238.572164987197, -357.9212142815298}, {-579.0698141769305, 
      6322.153869287789, 618.758671340066}}, {{-1019.3493693330607`, 
      6293.653079356829, -196.6506291436387}, {-984.8918895696186, 
      6301.379845672215, -102.04752038140474`}, {-1019.3493693330607`, 
      6293.653079356829, -196.6506291436387}}, {{-1745.1340990397798`, 
      6133.111448412191, -165.12004034620998`}, {-1810.7740391561772`, 
      6106.547209760829, -344.9525243859861}, {-1664.1794932307955`, 
      6153.487318679967, -230.03070046174153`}, {-1746.9048424111782`, 
      6132.557096698987, -166.97490017143048`}}, {{-2143.59282092264, 
      6003.324860852858, 230.03070046174153`}, {-2088.898523953059, 
      6026.70873454793, -55.663951983477816`}, {-2203.046170913243, 
      5976.971552725466, -331.98240425514547`}, {-2371.13200390397, 
      5908.306191632414, -396.8181648821243}, {-2651.6205135910914`, 
      5782.750665392215, -465.3139626196616}, {-2790.872454401768, 
      5722.136514617784, -394.96625341830304`}, {-2816.533282133847, 
      5719.750836820858, -198.50522743600195`}, {-2984.217052516322, 
      5636.205224294315, 124.3098673004821}, {-3093.6221744939207`, 
      5577.215451191174, 109.4684542410724}, {-2893.454355320696, 
      5670.5659154772575`, 400.5218869216312}, {-2945.7753904243955`, 
      5638.743008687742, 463.4633978483163}, {-2774.8107347046102`, 
      5722.968295838805, 485.6675420407969}, {-2682.2322072735287`, 
      5782.778739767729, 230.03070046174153`}, {-2240.571665507062, 
      5971.490445190119, 94.62644836418664}, {-2143.59282092264, 
      6003.324860852858, 230.03070046174153`}}, {{-2150.9191502340805`, 
      5941.84855658495, -869.3654178569822}, {-2620.8557810174098`, 
      5733.2555690454665`, -974.0170661321929}, {-2614.036962083661, 
      5753.712465998198, -865.6889218866974}, {-2273.2482084620365`, 
      5916.8690455865335`, -714.7146074511496}, {-1858.5461676796851`, 
      6066.405546424633, -657.52838639225}, {-1662.554403351369, 
      6112.189171288776, -751.5783926359203}, {-2150.9191502340805`, 
      5941.84855658495, -869.3654178569822}}, {{-2467.521225232723, 
      5831.973180021817, -766.3168538745633}, {-2588.8412812753254`, 
      5778.405778369809, -771.8427112223892}, {-2469.217573044297, 
      5831.255160463913, -766.3168538745633}}, {{-2617.142639274207, 
      5747.225116170957, -898.7666881602586}, {-2738.160418310804, 
      5685.244326085783, -931.819814291055}, {-2617.2508770029526`, 
      5747.462805391189, -896.9296725340929}}, {{-2800.4984432048786`, 
      5658.088410687564, -911.623659439641}, {-2764.092426326972, 
      5663.0477843011695`, -988.6842475955002}, {-2802.144195900505, 
      5657.273539341278, -911.623659439641}}, {{-2886.168312841964, 
      5611.86025875581, -929.9841913409863}, {-3059.411576591074, 
      5519.3245872005355`, -929.9841913409863}, {-2864.1838972331393`, 
      5609.171099744809, -1010.6749559665539`}, {-2886.168312841964, 
      5611.86025875581, -929.9841913409863}}, {{-3053.83706390288, 
      5501.716908456485, -1045.4682856677462`}, {-3218.281760530119, 
      5395.149050304726, -1105.8222844721363`}, {-3053.83706390288, 
      5501.716908456485, -1045.4682856677462`}}, {{-3181.6203795197357`, 
      5451.969660458302, -917.1326360275647}, {-3438.137009304401, 
      5294.266721422099, -915.2963878951355}, {-3246.8444167016723`, 
      5400.0977848072125`, -992.3502094791953}, {-3141.7053491446272`, 
      5463.59305573301, -983.1846777193293}, {-3183.206158577684, 
      5451.043933956708, -917.1326360275647}}, {{-3573.566657492053, 
      5206.057644174844, -902.4404911071656}, {-3632.0636550421123`, 
      5161.518172762294, -924.4768509631348}, {-3573.566657492053, 
      5206.057644174844, -902.4404911071656}}, {{-3617.8360940618636`, 
      5147.660073588303, -1049.1289021316975`}, {-3823.9512892723506`, 
      5019.6535797768865`, -931.819814291055}, {-3630.728334537817, 
      5156.432845917727, -957.5101860163473}, {-3460.1260586575886`, 
      5234.294022698175, -1147.8272658134865`}, {-3617.8360940618636`, 
      5147.660073588303, -1049.1289021316975`}}, {{-4206.488475551798, 
      4729.562425930564, -790.2579669169497}, {-4154.990044362705, 
      4757.364535944873, -889.5808526383772}, {-4206.488475551798, 
      4729.562425930564, -790.2579669169497}}, {{-3133.9746275362054`, 
      5520.531488929394, -624.2986494445696}, {-3065.21768971898, 
      5583.302066031675, -344.9525243859861}, {-3192.313517822777, 
      5521.827833103134, 79.78393004549577}, {-3680.174487511099, 
      5207.3264555033575`, 166.97490017143048`}, {-3589.890280635856, 
      5272.460066434714, 40.82047093647891}, {-3213.3631121073245`, 
      5510.044730042793, 38.965018613820064`}, {-3252.4374916051247`, 
      5484.995553552416, -155.84553478232866`}, {-3515.682550983275, 
      5321.701352036693, -85.34992936468699}, {-3312.221370777363, 
      5447.647127699315, -200.35980893165694`}, {-3454.5353964214255`, 
      5339.898704433002, -489.3676647125381}, {-3326.1930660665325`, 
      5420.772331133702, -489.3676647125381}, {-3259.5774843578356`, 
      5475.231938671689, -291.2104586324077}, {-3213.6078979925746`, 
      5488.430060977971, -487.5176240025507}, {-3180.613862585972, 
      5494.211326967499, -620.6053832734332}}, {{-3524.37352957209, 
      5314.664832496066, -144.71569466070045`}, {-3454.2877023528704`, 
      5360.005306081991, -161.41027893735404`}, {-3524.3501154501682`, 
      5314.629524601111, -146.57069954305618`}}, {{-3603.790994772341, 
      5259.921123711043, -183.66798397144217`}, {-3685.50268887957, 
      5202.012865459768, -209.63245896690722`}, {-3603.790994772341, 
      5259.921123711043, -183.66798397144217`}}, {{-3694.7827016716556`, 
      5195.862791886298, -198.50522743600195`}, {-3778.858916216345, 
      5134.895162366303, -202.21437347367646`}, {-3694.7827016716556`, 
      5195.862791886298, -198.50522743600195`}}, {{-3436.4065125838306`, 
      5349.329832434291, -513.4143688093942}, {-3391.877969518962, 
      5368.877480259276, -598.4414189014025}, {-3436.4065125838306`, 
      5349.329832434291, -513.4143688093942}}, {{-3471.604131852274, 
      5328.817667216623, -489.3676647125381}, {-3439.960740001856, 
      5334.365794618464, -631.684546629416}, {-3470.0538947720715`, 
      5329.827290458011, -489.3676647125381}}, {{-3755.7412443201843`, 
      5144.117756356604, -346.8052831906784}, {-3854.3253205511783`, 
      5068.695103843025, -374.5930560432121}, {-3755.7412443201843`, 
      5144.117756356604, -346.8052831906784}}, {{-3938.248903016842, 
      5007.618326977116, -319.0109076654619}, {-4143.480890651846, 
      4837.129044073339, -348.6580126501124}, {-4161.288455661054, 
      4815.232695772609, -430.1466777094485}, {-3912.212665474473, 
      5022.443737277771, -396.8181648821243}, {-3938.248903016842, 
      5007.618326977116, -319.0109076654619}}, {{-4452.119715657966, 
      4527.874571733073, -603.9831041533176}, {-4431.283038763473, 
      4535.622430253177, -692.5846450445014}, {-4452.119715657966, 
      4527.874571733073, -603.9831041533176}}, {{-4005.563552415253, 
      4964.149328022191, 22.26581818228762}, {-3919.7810958170617`, 
      5032.159981977508, 22.26581818228762}, {-3964.515504706141, 
      4996.000080717436, -102.04752038140474`}, {-3888.889295514712, 
      5055.907633789306, 46.38680655207611}, {-3928.606944052461, 
      5019.352810372879, 244.86431782656345`}, {-3894.3740055283274`, 
      5050.866371094097, 102.04752038140474`}, {-4005.563552415253, 
      4964.149328022191, 22.26581818228762}}, {{-4741.4088173939, 
      4187.473492577575, -819.7084026039689}, {-4758.997789964052, 
      4151.538376029439, -896.9296725340929}, {-4663.163646128026, 
      4250.597066129161, -935.490823495421}, {-4742.626703440406, 
      4186.09409551606, -819.7084026039689}}, {{-4894.331955406063, 
      3963.351870643927, -1012.5069628177123`}, {-4850.804716569261, 
      4048.714976412631, -874.8796097072342}, {-4757.614621787493, 
      4147.895266027471, -920.8048993249078}, {-4724.201309164005, 
      4243.747065901632, -600.2886981648073}, {-4418.353487282144, 
      4580.66907846227, -428.29539460673675`}, {-4414.535830134992, 
      4592.72630216116, -326.4233568411183}, {-4331.091917061545, 
      4660.809992047308, -454.20998887752046`}, {-4259.014810743291, 
      4738.424287450317, -309.74474002028967`}, {-4422.703130969281, 
      4590.52030603708, -233.7392247265105}, {-4179.292987442655, 
      4816.208664143403, -159.55537766762353`}, {-4303.8320012286795`, 
      4707.794124908646, -38.965018613820064`}, {-4448.274456157978, 
      4568.939716624612, -159.55537766762353`}, {-4511.7967765282365`, 
      4493.460089916636, -374.5930560432121}, {-4727.558261389175, 
      4279.17644653298, -163.26516654920053`}, {-4952.077872528578, 
      4010.1135923119414`, -289.3568923800711}, {-4952.077872528578, 
      4010.1135923119414`, -289.3568923800711}, {-4894.331955406063, 
      3963.351870643927, -1012.5069628177123`}}}], 
    Line3DBox[{{{2743.6453946966503`, 5077.795886147398, 
     2715.9183671061287`}, {3065.9414548670925`, 4865.493079784293, 
     2759.4915792403904`}, {3372.2794782336046`, 4593.602211843854, 
     2866.0710964712193`}, {3542.7589262713677`, 4242.104873039306, 
     3184.5280853622126`}, {3659.210613883461, 4140.836984240467, 
     3186.1356598847033`}, {3714.569556803257, 3766.8002194207843`, 
     3563.8466303478867`}, {3571.0374192982454`, 3743.4119728314095`, 
     3731.269521440951}, {3606.796433545107, 3581.7037323942113`, 
     3853.599817095655}, {3484.930902383692, 3460.6860271110713`, 
     4070.2238903246307`}, {3441.232406320366, 3584.307449963096, 
     3999.8091604262313`}, {3280.1832684695105`, 3660.105542013549, 
     4065.936403409967}, {3285.0036928237423`, 3763.459933083811, 
     3966.4762397129225`}, {3210.1896407234844`, 3978.4316337241385`, 
     3815.046250882774}, {2987.6447726541846`, 4097.083634102884, 
     3869.844712336456}, {2712.3345342568923`, 4208.719350615696, 
     3951.92819925039}, {2573.049914746979, 4379.817373300619, 
     3858.034162984386}, {2492.014815342531, 4548.616524639334, 
     3713.187789742302}, {2625.5208887613926`, 4656.381370230966, 
     3480.3112343070084`}, {2579.067740174896, 4820.036342332001, 
     3286.863693582058}, {2692.986895438178, 4831.72545373678, 
     3176.486173543079}, {2547.7023181147824`, 5072.8923976154965`, 
     2909.087367283329}, {2743.6453946966503`, 5077.795886147398, 
     2715.9183671061287`}}}], 
    Line3DBox[{{{3699.3147822537353`, 4101.302080273078, 
     3190.9567653208214`}, {3659.210613883461, 4140.836984240467, 
     3186.1356598847033`}, {3714.569556803257, 3766.8002194207843`, 
     3563.8466303478867`}, {3571.0374192982454`, 3743.4119728314095`, 
     3731.269521440951}, {3606.796433545107, 3581.7037323942113`, 
     3853.599817095655}, {3759.919724764167, 3427.2663289940547`, 
     3847.6827914308333`}, {3971.35951524997, 3452.2501577070775`, 
     3605.285860960956}, {4151.756218606527, 3338.097702790208, 
     3508.2528013084757`}, {4175.358502610053, 3417.491335647547, 
     3402.1975303406302`}, {3956.6688217458964`, 3917.728824098492, 
     3111.9104077482343`}, {3833.0359302511793`, 4046.2429508645123`, 
     3102.1874911073182`}, {3699.3147822537353`, 4101.302080273078, 
     3190.9567653208214`}}}], 
    Line3DBox[{{{3719.4375926296275`, -408.43961824728336`, 
     5165.924386088082}, {3886.8953839605406`, -433.69556702743006`, 
     5039.02432215672}, {3917.5755765585013`, -674.3358834365025, 
     4988.555543267233}, {3845.9398502873973`, -710.4887322712121, 
     5039.02432215672}, {3823.518577123935, -587.3572816162982, 
     5071.837024601685}, {3682.1397429375274`, -656.9941185630482, 
     5167.0126119115985`}, {3624.8960400890946`, -461.1450211326999, 
     5228.315948566127}, {3672.6982403511747`, -525.9727240248639, 
     5188.685191732415}, {3719.4375926296275`, -408.43961824728336`, 
     5165.924386088082}}}], 
    Line3DBox[{{{3700.0549419886156`, -293.02334635614494`, 
     5187.633799214808}, {3699.593260799877, -290.81671708634025`, 
     5188.08722283148}, {3699.99619293357, -288.45414028538585`, 
     5187.93177202913}, {3701.772314656265, -287.4518711677909, 
     5186.720239439036}, {3704.930733776356, -287.8109735940582, 
     5184.4446946031}, {3706.239959418795, -288.7131321655985, 
     5183.458669703317}, {3707.6198133509256`, -288.8206216846362, 
     5182.465789384176}, {3708.7455502667726`, -287.9933603394502, 
     5181.706278609084}, {3710.2274155437367`, -285.24868424757756`, 
     5180.797178338742}, {3711.235156347687, -284.52475385767895`, 
     5180.115150142049}, {3715.0208854381754`, -285.04326238730545`, 
     5177.372291937847}, {3716.293972190271, -286.40017321785086`, 
     5176.383654931601}, {3720.3867703010164`, -289.6975690650736, 
     5173.2592519462}, {3723.141538013776, -291.062705205393, 
     5171.200371244082}, {3725.6790887181805`, -293.9043086589268, 
     5169.211668643468}, {3726.4063829946776`, -296.716983887817, 
     5168.526694353918}, {3726.5544852031517`, -300.17840424565827`, 
     5168.220030575015}, {3727.764054361625, -302.23061338653577`, 
     5167.228028773395}, {3730.399137565545, -304.97734864779017`, 
     5165.164545419826}, {3732.0086801358107`, -307.2958398576331, 
     5163.864267019143}, {3732.355357651045, -308.2458529974697, 
     5163.557075149204}, {3731.772134767509, -310.1552127171049, 
     5163.864267019143}, {3730.981325585243, -312.0467682494246, 
     5164.321722410337}, {3730.9739376840284`, -313.4297893251942, 
     5164.243306960479}, {3731.4775854005675`, -314.96751459724686`, 
     5163.785838355574}, {3731.3323390454466`, -316.68356634352045`, 
     5163.785838355574}, {3730.6259398586294`, -317.5450414788647, 
     5164.243306960479}, {3729.10413480527, -317.99241429289947`, 
     5165.314788687172}, {3727.4445956643, -317.5035979616034, 
     5166.542551991074}, {3725.3675526560273`, -314.56637325187023`, 
     5168.220030575015}, {3723.297494805533, -311.4041184065691, 
     5169.902990406519}, {3720.8938944567944`, -309.7121072591454, 
     5171.734788908964}, {3713.9936038211868`, -308.5633818856906, 
     5176.760940987576}, {3712.3667526963977`, -307.62578013596107`, 
     5177.983503533991}, {3710.1312530647674`, -305.15217849075265`, 
     5179.731838907725}, {3709.4952264414087`, -303.83865135426356`, 
     5180.2645616755535`}, {3709.1947871408524`, -301.0704669158053, 
     5180.641301518396}, {3708.250998715321, -299.50528639664896`, 
     5181.407608357721}, {3704.917093557515, -297.40720481744376`, 
     5183.912805438379}, {3701.093251921098, -294.7014130850229, 
     5186.798001436898}, {3700.0549419886156`, -293.02334635614494`, 
     5187.633799214808}}}], 
    Line3DBox[{{{4505.381140392963, 3063.7700923610114`, 3317.026875222087}, {
     4371.829140465494, 3072.572981770763, 3483.420578722739}, {
     4341.857687568996, 3100.8297544727775`, 3495.8461557503215`}, {
     4336.516268943066, 3106.549115487763, 3497.398023138674}, {
     4333.803642542254, 3110.332256469264, 3497.398023138674}, {
     4362.603302366731, 3129.079234559025, 3444.469372781589}, {
     4428.677750310998, 3157.0018473413425`, 3332.8615947536223`}, {
     4547.321249185964, 3180.1276223821396`, 3145.8657714632263`}, {
     4554.017403639684, 3176.92654463004, 3139.407094878251}, {
     4508.278279668345, 3071.497163928443, 3305.9258635139054`}, {
     4505.381140392963, 3063.7700923610114`, 3317.026875222087}}}], 
    Line3DBox[{{{4335.660256346069, 1058.2562767710156`, 4557.406881573763}, {
      4372.694555458366, 950.7370664322628, 4545.707320668208}, {
      4465.302789622022, 980.3988846225791, 4448.337072887416}, {
      4624.34679056328, 1548.7807024405606`, 4111.509307584602}, {
      4675.445796659236, 1395.2919560684595`, 4108.671397158956}, {
      4835.634917327629, 1392.6871801170432`, 3919.8049766103723`}, {
      4702.536756762832, 1318.8719683286852`, 4102.991405359184}, {
      4622.26632480946, 906.8525989366086, 4301.172639680296}, {
      4516.077209950561, 804.4358730341833, 4432.35188697162}, {
      4565.42764037947, 603.7522526363275, 4413.634258510293}, {
      4402.970064032088, 543.2170941383158, 4583.293956738231}, {
      4287.21007282414, 792.0078233656133, 4656.2181101407705`}, {
      4262.966522731917, 1040.5130507499046`, 4629.499185364015}, {
      4335.660256346069, 1058.2562767710156`, 4557.406881573763}}, {{
      4824.379252359181, 1348.5037671760292`, 3949.014574462638}, {
      4941.300961305326, 1331.7234328329498`, 3807.6070173071544`}, {
      4922.271249548163, 1083.7326647503698`, 3909.55022313624}, {
      4824.379252359181, 1348.5037671760292`, 3949.014574462638}}, {{
      4733.398267319384, 770.8828596454524, 4205.769150778809}, {
      4906.229629586577, 763.9091702310405, 4004.143765076833}, {
      4784.873165974038, 683.8286431308679, 4162.352803688725}, {
      4733.398267319384, 770.8828596454524, 4205.769150778809}}}], 
    Line3DBox[{{{6305.427508958026, -833.8574964364238, 
     483.81741898382285`}, {6344.314388779525, -343.5957852557803, 
     565.1818845995894}, {6344.314388779525, -343.5957852557803, 
     565.1818845995894}, {6284.627508902851, -294.5433452330213, 
     1050.9590772815511`}, {6284.627508902851, -294.5433452330213, 
     1050.9590772815511`}, {6243.845755576621, -603.0471170124532, 
     1156.952049667911}, {6244.180476339045, -603.0794452278969, 
     1155.1272880725778`}, {6217.949561544056, -870.1872152655358, 
     1125.91791462899}, {6217.949561544056, -870.1872152655358, 
     1125.91791462899}, {6254.244566967937, -930.9835945047535, 
     839.9453156694467}, {6254.244566967937, -930.9835945047535, 
     839.9453156694467}, {6305.669801498158, -832.0232866705164, 
     483.81741898382285`}}}], Line3DBox[CompressedData["
1:eJx1lXk4Vukbx/Xal0iRQlmyRnglmrLcjEKkrJOlsmTQplJhMglFZCnnPS+l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     "]], 
    Line3DBox[{{{-3577.246711575731, 4022.0748811472436`, 
      3422.5769976678403`}, {-3472.561819580495, 3992.3793176911213`, 
      3562.307606434905}, {-3395.195846835346, 4106.5183384141465`, 
      3506.703008225865}, {-3563.0095755918014`, 4147.258775255514, 
      3285.2733684287623`}, {-3577.246711575731, 4022.0748811472436`, 
      3422.5769976678403`}}, {{-3709.4664987686533`, 3761.625407149668, 
      3574.611346817878}, {-3661.9349585191617`, 3924.654999319515, 
      3446.030932230818}, {-3567.1671071093056`, 3959.4236371531933`, 
      3505.152918420251}, {-3710.1964491932576`, 3762.365621426859, 
      3573.0744365460723`}}, {{-3723.5220207968773`, 3000.928515848728, 
      4221.09292641215}, {-3879.287887939863, 3023.5694806773386`, 
      4061.645820107859}, {-4015.5852372133063`, 3329.858599917478, 
      3670.821515305554}, {-3953.601125544414, 3455.0264757793093`, 
      3622.104896570217}, {-3808.185954372765, 3569.8759085950633`, 
      3666.267778565903}, {-3814.0990874356853`, 3713.3690957143326`, 
      3514.44900378651}, {-3728.975842286842, 3685.837718907846, 
      3632.7886213647685`}, {-3463.939826556772, 4001.232787073199, 
      3560.7682810938823`}, {-3543.452967296354, 3788.8249172360934`, 
      3711.6789334598393`}, {-3694.3695830737884`, 3473.2907992364785`, 
      3869.844712336456}, {-3799.8251841024694`, 3355.8943865978617`, 
      3871.319558788915}, {-3723.5220207968773`, 3000.928515848728, 
      4221.09292641215}}, {{-3519.6636270761132`, 2754.8123211193174`, 
      4550.910976677815}, {-3694.878399050122, 2707.5446474340883`, 
      4439.018955623395}, {-3836.0554036906624`, 2605.3496521104935`, 
      4380.028061561181}, {-3802.32551721394, 2841.0543093065303`, 
      4261.284402937045}, {-3623.240221807454, 2990.310757154036, 
      4314.856377763405}, {-3705.8827643448176`, 2979.6062415777556`, 
      4251.6106679801815`}, {-3664.2844543793217`, 3071.068740863195, 
      4222.483856249421}, {-3629.0031470190547`, 2896.995152042575, 
      4373.278968687867}, {-3521.507370916937, 2754.6038827382263`, 
      4549.610640019778}}, {{-3525.359418900102, 4466.549908644161, 
      2882.6353517507496`}, {-3499.5207950335353`, 4536.026334405939, 
      2804.5735841128635`}, {-3525.8787782853465`, 4467.207925129597, 
      2880.980021537176}}}], 
    Line3DBox[{{{4157.437231315792, -170.9264961976577, 
     4834.6989038167485`}, {4157.498485091613, -167.13501480258265`, 
     4834.778787420153}, {4158.994960267175, -163.90535581938306`, 
     4833.602139688987}, {4160.086299500869, -161.0360102090973, 
     4832.759364392262}, {4162.582411212493, -157.33471424713917`, 
     4830.731533376405}, {4163.932326970208, -156.43599077083243`, 
     4829.597193059067}, {4165.558052241659, -157.38530013625075`, 
     4828.164213311898}, {4166.491616949845, -158.5602612520433, 
     4827.320161273577}, {4166.336139391729, -160.2001372425905, 
     4827.4002091836}, {4165.531336566755, -163.33650786073562`, 
     4827.9895980915535`}, {4165.345359817321, -165.48445804034984`, 
     4828.076906760642}, {4165.314042389126, -171.061003803754, 
     4827.909563284466}, {4165.052602612712, -172.69594631511114`, 
     4828.076906760642}, {4164.286172415789, -174.69184073082036`, 
     4828.666184880535}, {4163.1924144870745`, -175.02446329947733`, 
     4829.597193059067}, {4160.9592014553755`, -174.55227427043098`, 
     4831.538442087661}, {4158.658702208891, -173.44510933168166`, 
     4833.558552721916}, {4157.437231315792, -170.9264961976577, 
     4834.6989038167485`}}}], 
    Line3DBox[{{{4555.546937436852, 3185.8801824029288`, 
     3128.0941888283996`}, {4546.395995183654, 3181.450249951129, 
     3145.8657714632263`}, {4428.677750310998, 3157.0018473413425`, 
     3332.8615947536223`}, {4362.603302366731, 3129.079234559025, 
     3444.469372781589}, {4150.960721267462, 3337.458106508469, 
     3509.802297536945}, {4175.358502610053, 3417.491335647547, 
     3402.1975303406302`}, {4343.567552715499, 3273.112921243846, 
     3332.8615947536223`}, {4382.576073636162, 3362.8688981718947`, 
     3189.3499999999995`}, {4555.546937436852, 3185.8801824029288`, 
     3128.0941888283996`}}}], 
    Line3DBox[{{{2875.1024892582336`, 3332.7977695676714`, 
     4616.715108540471}, {2719.933500707753, 3382.9150106423167`, 
     4673.934260609424}, {2642.7126316874983`, 3528.305537141963, 
     4610.308414075213}, {2682.950495188501, 3592.9064448976883`, 
     4536.586118281062}, {2827.245862426592, 3562.8365030037285`, 
     4472.213163105699}, {2907.9022758330825`, 3492.2378499841016`, 
     4476.180608883336}, {2877.2447626916164`, 3795.2127876076047`, 
     4243.304863233771}, {2671.467965682896, 3960.6141345535952`, 
     4226.654501553725}, {2522.195988124624, 3739.30932693299, 
     4510.42202515463}, {2324.9348667747463`, 3818.839428343894, 
     4549.610640019778}, {2160.326358959812, 4091.6518285692255`, 
     4390.807331977748}, {1802.0686810337543`, 4441.666074456479, 
     4208.5584992797285`}, {1677.3667444902323`, 4486.321148011639, 
     4212.739850902471}, {1534.6953956295254`, 4461.294391443526, 
     4292.944919924791}, {1254.3506177124223`, 4477.489905393196, 
     4366.520624587327}, {953.0928164815128, 4582.170412547123, 
     4333.952247492935}, {800.7780710811209, 4668.38977792521, 
     4272.31846334505}, {794.9450186791494, 4447.806509898659, 
     4502.542987888102}, {582.1650967177703, 4462.096608668574, 
     4520.906031435734}, {525.5080760325579, 4300.602619121286, 
     4681.503184280846}, {326.21135479315797`, 4321.6580424130125`, 
     4680.242686704497}, {194.30809186522572`, 4171.872657420056, 
     4821.362523779622}, {159.00179960161972`, 4170.559710639441, 
     4823.791435967615}, {457.4435021011886, 4035.803787399115, 
     4918.419148669657}, {726.959715401878, 3909.628436383732, 
     4987.398996631663}, {816.9934670256364, 3625.285920253876, 
     5184.364701797746}, {1006.1097912631902`, 3632.017025235414, 
     5146.261663219665}, {1209.2252605439428`, 3440.168050321333, 
     5233.625105489574}, {1779.931055924787, 3289.626087338947, 
     5167.0126119115985`}, {1962.3975911197597`, 3523.3203942542946`, 
     4941.965477803252}, {2346.9756142658457`, 3244.206028822018, 
     4965.344539767238}, {2488.941641398888, 3080.9098351731327`, 
     5000.097777368365}, {2838.0136353651515`, 3027.4676042645324`, 
     4844.371187327644}, {2875.1024892582336`, 3332.7977695676714`, 
     4616.715108540471}}}], 
    Line3DBox[{{{4770.423706064326, 
     4230.4199181752265`, -185.5226954466847}, {4926.751352856374, 
     4018.160355144586, -518.9625781038676}, {5292.558044242269, 
     3558.6865666914864`, -111.32366494150033`}, {5220.0736181063685`, 
     3659.662629306195, 213.34139591171007`}, {5273.867547716035, 
     3557.2685776333497`, 469.0149738868202}, {5148.078451672213, 
     3731.15365957076, 513.4143688093942}, {4911.949250885615, 
     4051.498828809278, 382.001946174733}, {4735.034578747333, 
     4250.990415606824, 443.10463153298207`}, {4809.396656495407, 
     4178.289189351235, 315.30452048995164`}, {4770.423706064326, 
     4230.4199181752265`, -185.5226954466847}}}], Line3DBox[CompressedData["
1:eJx1lHk0lXkYxy8X2coyIlvFkSVLZdKQZu5PNRXKVlJEst1737fs0iSmOhmF
XISjFPW+WSLGlO4lJo8lKa1ICKkJk3VcXVuW+fXHnP6aP37nvOdzfuf7fH/P
830fHZ8gF38mg8EQwwfh8/U787rQXHOGD/d8o1+8taahXjXQ1aWRg1ptass1
pvjQ/346sO5nGhp2FN4QneGgxX55AUvEh8W+VE6LAw1txM3a93s5iHGDVR09
zocot54jn7xpuHxj43q/1RyUo5zdYTzKhwc11TX3gmmItb4aWSTLQc3Sc7G5
w3wwM7O+tvIEDQydTzKnuthot/qYyAFze5GyufhJGp7LxI+vzWajCUXTobgR
PohK+krOhdPQmbRbZ4M/G11b6NlzUMiH6hWVrIeeNEgOfF/WcZiNIpUkR6ux
z50xdmlJe2jwaFzVceQYG9W6q6o74vd2hVxvOrsF+3laPJIfzUZPLaUC6xf4
MGz8KtNc91td+/KHufXiAvjT01O7ZxkNoVV7VooK2Giy4IR7nJQAtr25WtI4
R4HJ2Z5mKpmNjicbewqXCMBKl3erXkSBlXfksrtFbHTOQXXllLQAWI/nFzP+
oaA7sDXlQwcbpbu+rd8kI4D1mmoXC0co2FuV71NlwkUXgk6+rcT3ny+XjSoe
o+CKU0asPMlFs6etJfqwvsY0bOBNUKBSNloqm8JF+hL56psxH2Iet+6apMDu
mWc5+o2LytSmYtqxz4gg39mJaQqijr5xNj3ARRaVTke9MM8/JT+ah/0rKvAd
5nZxkHHaUGeCpAAetSu3PWbQUBAd7GhmzEH2CiVeHhICKDXwd0qU+sZ3mV0c
iMT9MV1nYpmuQIP3jFDF6CcOivDQ68le5MOhhYZDq7Rp+EV86V0DPw5KNcxd
soj77Mqhd86u+parMh7iyOL7CTO6+89jfklVzYFRivVPu40EYd4z9k5uE9aJ
KShrCxLjIqsale0RmIeNOLk0atKgqGTQ9SODQM1lplUszP8WGWalatGQ8wpZ
vDYkUEmAxpN0XPcvr5AaCusbvSL22dsQyEV5q1vQPPb5R7W0UI8GcUvm4DyX
QLVx7Xc1MG9WKI710KehaIqpeiGeQB5RxYkSmJtmxSnZYr7RpUKS+ZFAmdNn
1JowX+ecOHYQ58eRIavXJUOiB9K+tytw3Xqet1QL9v+rlGmSvDKJnMwW1FzF
BJAb651hp0SDmvfUl8vyJOq/Uy5kMQWQfrAhXFaahnJCwe4HWRLVUrb37mM+
xdJ491iShvGEAx8ZCwRaSl5CJnguR2Zbx+eYNFx9+dLOuppA2XI5zZcx52Xy
KkzEaZhvP+/WyyPQ07LKrU2Yx9i4Mw+L0SDPEdMsjCSQ2pI8oTie+wkbfssB
PPemgG0+ZqEEMjPQjrfC/HBXdJXzIgU7R0JGtUIIdOhd05ZSzFt99z5aOk/B
WvFuW+dwAqUYZdul4lzZ0n6693He3AZ2bJU7RSD16LrETpxPiYviqS5CCib3
hYZ9ySJQY5tcWALOeWfPsCNvlIL+Z8tcKnA/J5R9kocwb+1C7OPDFOxntSWs
WU6i3u5nNYWYayU1qEXg/2XANsDoHItE4CS64op5Cjt4Sw/W0Vpjqx/sR6Jt
r4HDx3WVC27frBinYOb01O8rzpKIO923Wh9zqZmTAt/PFHxaGxruzSNRfkl5
0Wrsf3Yub9JlhoIXQ3XM/ksk2h0yM3cLz6s0We/mIN4DhWRWXGcGiTLiDWfN
8dyjau+7b8f5mZuOXbHsDonSwvyd1L/gnEt22+wzo6FlROp5bRmJBg3zjnXi
/ePQaSGM3UhD5nT7E8VEElEBt9qnpvnQu2b0vaoVDbeT1oV4OJHIRju3uAnz
tuEdIX6baZhUsajLmSGQ0WD8RPRXnWq775ItaYhLq4v55QMXRYp0PgdgLnTr
9RrDOv/tk//b8/8C6Skiew==
     "]], 
    Line3DBox[{{{3718.794456933103, 4191.033887057487, 3048.543329956037}, {
     3699.3147822537353`, 4101.302080273078, 3190.9567653208214`}, {
     3833.0359302511793`, 4046.2429508645123`, 3102.1874911073182`}, {
     3718.794456933103, 4191.033887057487, 3048.543329956037}}}], 
    Line3DBox[{{{1342.2637636826955`, 4732.94795394038, 4060.214938152403}, {
     1566.373334036544, 4685.9258875919795`, 4034.4004318379893`}, {
     1707.5797741602557`, 4583.380203058092, 4094.4610047103843`}, {
     1408.9193578580334`, 4608.367570882941, 4179.199476537026}, {
     1546.0121773287924`, 4550.056218186119, 4194.597531217617}, {
     1534.6953956295254`, 4461.294391443526, 4292.944919924791}, {
     1254.3506177124223`, 4477.489905393196, 4366.520624587327}, {
     953.0928164815128, 4582.170412547123, 4333.952247492935}, {
     800.7780710811209, 4668.38977792521, 4272.31846334505}, {
     1345.8787110156725`, 4693.636856454041, 4104.411924377356}, {
     1342.2637636826955`, 4732.94795394038, 4060.214938152403}}}], 
    Line3DBox[{{{-1047.0817357882042`, 5888.2035456648155`, 
     2218.2183241058697`}, {-1147.1448856779077`, 5820.157156086013, 
     2344.7053076503635`}, {-1239.5594218099343`, 5765.660434749799, 
     2430.733609625493}, {-1357.821502734017, 5804.288521347579, 
     2270.3235931615272`}, {-1509.4468011146107`, 5774.193505793776, 
     2251.2381931403556`}, {-1444.526426325188, 5844.3502578983935`, 
     2108.252156799836}, {-1839.6357487461648`, 5846.437043284817, 
     1767.124189422525}, {-1860.4623200834596`, 5882.727209046518, 
     1618.6458011230995`}, {-1621.980752123125, 5963.024830454567, 
     1580.9259948284682`}, {-1545.441560054943, 5883.899226734046, 
     1918.1120832979068`}, {-1402.4045991677688`, 5886.67834742556, 
     2016.9514283155577`}, {-1151.6714613056565`, 5971.190247783557, 
     1925.1892270601957`}, {-1175.9895711276563`, 5894.17543725044, 
     2136.248612290381}, {-1047.0817357882042`, 5888.2035456648155`, 
     2218.2183241058697`}}}], 
    Line3DBox[{{{3097.0411541435697`, 1385.3734575591907`, 
     5401.563677550378}, {3010.865721672832, 1574.0425606714707`, 
     5398.600866266215}, {3145.1222073248446`, 1679.3388622485827`, 
     5289.219316376825}, {3235.274272694051, 1601.3147738918901`, 
     5258.954750267351}, {3209.6504782859124`, 1472.8821413664289`, 
     5311.927681631486}, {3318.217359216596, 1295.9433011739436`, 
     5291.292678272108}, {3216.7979436759356`, 1261.7288920187864`, 
     5361.722193717921}, {3143.9297546604607`, 1426.1497868070805`, 
     5363.731553065527}, {3097.0411541435697`, 1385.3734575591907`, 
     5401.563677550378}}}], 
    Line3DBox[{{{4247.010899811953, 3081.8605672182175`, 3626.685477278649}, {
     4371.829140465494, 3072.572981770763, 3483.420578722739}, {
     4341.857687568996, 3100.8297544727775`, 3495.8461557503215`}, {
     4336.516268943066, 3106.549115487763, 3497.398023138674}, {
     4247.010899811953, 3081.8605672182175`, 3626.685477278649}}}], 
    Line3DBox[{{{4911.581313017889, 
     2689.012223629512, -3055.0607124547496`}, {4939.376070825465, 
     2518.547905457568, -3153.9331268865058`}, {4839.9066993839215`, 
     2584.2695050142456`, -3253.408667338313}, {4841.514584687587, 
     2733.6224661666897`, -3126.476999551752}, {4911.581313017889, 
     2689.012223629512, -3055.0607124547496`}}}], 
    Line3DBox[{{{6204.936498616434, -1262.4085984215515`, 
     770.0008238755117}, {6305.427508958026, -833.8574964364238, 
     483.81741898382285`}, {6254.244566967937, -930.9835945047535, 
     839.9453156694467}, {6207.896863091848, -1124.4355723826395`, 
     940.9967432328986}, {6204.936498616434, -1262.4085984215515`, 
     770.0008238755117}}}], 
    Line3DBox[{{{5231.687384548728, 1067.568862407011, 3489.6357294670006`}, {
     5240.452007184593, 1494.4302395571244`, 3315.4418573535863`}, {
     5197.657137418801, 1823.5890469922076`, 3216.6281972788593`}, {
     4972.752632002512, 1980.5743999877761`, 3469.419259332193}, {
     4915.2191647480895`, 2307.6895932996663`, 3347.088734029845}, {
     5360.11057633483, 2499.4606103025344`, 2389.5030774220777`}, {
     5360.11057633483, 2499.4606103025344`, 2389.5030774220777`}, {
     5432.424572936224, 2533.1811807095587`, 2181.643888231443}, {
     5492.991048529791, 2445.6371829484765`, 2129.2538129944546`}, {
     5623.045533650187, 1612.3823606858646`, 2543.500685953468}, {
     5673.136883483953, 1516.5755351462883`, 2490.648557594935}, {
     5721.098236755274, 1216.0569701811273`, 2545.2021728478826`}, {
     5648.9123279981695`, 935.1713716369939, 2811.237753421236}, {
     5434.9707772208685`, 912.7529903003774, 3211.8200945029457`}, {
     5231.687384548728, 1067.568862407011, 3489.6357294670006`}}}], 
    Line3DBox[{{{4283.987056701689, 724.5821743024488, 4670.144458225626}, {
     4267.380181662224, 716.6669668005968, 4686.541212221794}, {
     4268.724433263232, 716.892721403304, 4685.282299804405}, {
     4283.987056701689, 724.5821743024488, 4670.144458225626}}}], 
    Line3DBox[{{{3318.217359216596, 1295.9433011739436`, 5291.292678272108}, {
     3209.6504782859124`, 1472.8821413664289`, 5311.927681631486}, {
     3235.274272694051, 1601.3147738918901`, 5258.954750267351}, {
     3376.9915416663935`, 1597.4985753608623`, 5170.274665743944}, {
     3442.4671924952086`, 1496.8273356939387`, 5157.202850937079}, {
     3453.156164882946, 1452.7553133727988`, 5162.657086268466}, {
     3390.3055418601775`, 1430.9591002751567`, 5210.182153837991}, {
     3380.7562687971476`, 1316.9690067334907`, 5246.32189026676}, {
     3318.217359216596, 1295.9433011739436`, 5291.292678272108}}}], 
    Line3DBox[{{{4065.362848735844, 436.8542490936986, 4895.895930589943}, {
     4119.888736210329, 459.69271231581985`, 4847.990625144708}, {
     4117.092414088572, 419.40900305534984`, 4854.014815800469}, {
     4065.362848735844, 436.8542490936986, 4895.895930589943}}}], 
    Line3DBox[{{{-2345.7056349022478`, 5416.974033164469, 
     2416.996294661975}, {-2339.988953663077, 5417.509703966883, 
     2421.333102685261}, {-2338.7430491634072`, 5416.7027619196115`, 
     2424.340287377641}, {-2338.530270136011, 5415.171097153218, 
     2427.964549620526}, {-2340.0914390992048`, 5412.55596063861, 
     2432.2877542778665`}, {-2344.0811487708147`, 5409.33853976994, 
     2435.6013261896737`}, {-2348.5342499798176`, 5406.41903931266, 
     2437.792800471218}, {-2355.3837272907263`, 5402.565541325903, 
     2439.7267795579087`}, {-2363.7671666088445`, 5401.966336144532, 
     2432.9361223063825`}, {-2365.517142173899, 5403.140816442811, 
     2428.6234079764477`}, {-2366.031899502349, 5404.31658739045, 
     2425.5038577156993`}, {-2363.689461268646, 5406.412889218583, 
     2423.1148326025955`}, {-2359.3730628848257`, 5409.91164904928, 
     2419.5099482362907`}, {-2353.2668684979863`, 5413.693530965708, 
     2416.996294661975}, {-2345.7056349022478`, 5416.974033164469, 
     2416.996294661975}}}], 
    Line3DBox[{{{4455.379400490874, 1668.7531167891082`, 4248.843504165365}, {
     4403.418344934455, 1710.9212994562752`, 4286.078496313567}, {
     4352.247236907631, 1833.9870993015084`, 4287.452506841974}, {
     4411.0600131660585`, 1866.3296061697363`, 4212.739850902471}, {
     4505.03327916477, 1727.8168607464763`, 4172.186206215886}, {
     4455.379400490874, 1668.7531167891082`, 4248.843504165365}}}], 
    Line3DBox[{{{4307.067826610783, 4061.141508279743, -2375.733587063497}, {
     4275.369168733936, 4203.838924043122, -2176.412291304031}, {
     4371.82866101454, 4291.187317509772, -1777.8186788238822`}, {
     4127.213143038335, 4583.734568025902, -1625.8237210152968`}, {
     4070.77335579778, 4729.926894644941, -1320.760972712829}, {
     3913.2375114719966`, 4747.136701470763, -1684.9566778875176`}, {
     3980.1158740508836`, 4682.136507800364, -1709.9967957429596`}, {
     3933.360795285081, 4237.741773373718, -2694.0733483551794`}, {
     4155.563758555055, 4022.336114015424, -2690.70914899234}, {
     4307.067826610783, 4061.141508279743, -2375.733587063497}}}], 
    Line3DBox[{{{5277.514174326697, 
     3425.0755973642836`, -1050.9590772815511`}, {5191.519162758384, 
     3545.8417718999867`, -1078.4009464766134`}, {5122.139654449121, 
     3579.9033991289016`, -1278.9803373098352`}, {5110.707977330344, 
     3521.254764651633, -1472.8348661238817`}, {4992.6119381991, 
     3616.2620792618964`, -1638.379778514202}, {4975.892863550718, 
     3520.955925719387, -1879.1416310663117`}, {5085.455727668312, 
     3499.4947376055616`, -1606.0791740747202`}, {5177.561106107444, 
     3390.252482639379, -1544.947502823634}, {5277.514174326697, 
     3425.0755973642836`, -1050.9590772815511`}}}], 
    Line3DBox[{{{-2143.59282092264, 6003.324860852858, 
      230.03070046174153`}, {-2240.571665507062, 5971.490445190119, 
      94.62644836418664}, {-2682.2322072735287`, 5782.778739767729, 
      230.03070046174153`}, {-2774.8107347046102`, 5722.968295838805, 
      485.6675420407969}, {-2947.415509624395, 5637.885878809579, 
      463.4633978483163}, {-3106.4139507091286`, 5539.339528267603, 
      594.7467086178401}, {-2926.654501315003, 5630.045686597543, 
      651.9913241424521}, {-2889.0423282484926`, 5633.555461522414, 
      777.3679807784142}, {-2700.9731788946115`, 5752.885990784736, 
      544.8489276163259}, {-2689.5859380764487`, 5759.085747828161, 
      535.6048178125125}, {-2687.711894298745, 5759.442126013055, 
      541.1514209084804}, {-2594.5999242418993`, 5804.830332581286, 
      509.71534513704256`}, {-2330.063711071198, 5930.382805088065, 
      298.6244756628788}, {-2325.066335001541, 5937.95385191587, 
      150.28067194414908`}, {-2143.59282092264, 6003.324860852858, 
      230.03070046174153`}}, {{-1115.2345321764517`, 6239.862249241015, 
      712.8707728179377}, {-1246.5364029163984`, 6247.763099347066, 
      315.30452048995164`}, {-1490.517787412851, 6200.5070617287865`, 
      141.00564831718532`}, {-1575.0280980532818`, 6179.081396307709, 
      161.41027893735404`}, {-1478.4612911884417`, 6181.995679645766, 
      533.7558592638408}, {-1329.1889801033742`, 6200.112648542793, 
      692.5846450445014}, {-1227.5444858937792`, 6228.072768477318, 
      626.1452033698297}, {-1115.2345321764517`, 6239.862249241015, 
      712.8707728179377}}}], Line3DBox[CompressedData["
1:eJx113kwlnvYB3BLKIkWkSUiUhFZK6SLFC0SIZQ1RDmUk0iWllvP07FEStL9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     "]], 
    Line3DBox[{{{6009.152264008688, 446.5571511481072, 2092.484042643517}, {
       5921.153404138738, 0, 2372.2892013108867`}, {
       5760.790470840381, -483.747091627257, 2695.7551061653994`}, {
       5742.946045524976, -662.7886659955893, 2695.7551061653994`}, {
       6118.4113403183355`, -589.1360044402791, 1704.6334294903768`}, {
       6018.91879559411, -1224.5628690081448`, 1720.719617453839}, {
       6033.039749371222, -1287.865894326196, 1622.2350356031077`}, {
       6027.584193653838, -1308.716816808885, 1625.8237210152968`}, {
       6106.496402577681, -1225.7410514311023`, 1376.9801197339118`}, {
       6107.70622062802, -1227.8322469964182`, 1369.732231400546}, {
       6153.482292385798, -972.7810553032452, 1369.732231400546}, {
       6217.949561544056, -870.1872152655358, 1125.91791462899}, {
       6244.180476339045, -603.0794452278969, 1155.1272880725778`}, {
       6187.597805255001, -429.06173422816937`, 1489.0779424276636`}, {
       6161.35106815008, 0, 1650.929042995449}, {6161.299976060484, 
       25.09163189600567, 1650.929042995449}, {6126.121674985679, 
       415.846814221442, 1727.8652552709455`}, {6009.152264008688, 
       446.5571511481072, 2092.484042643517}}}], 
    Line3DBox[{{{4996.91252791115, 1276.4651929821364`, 3753.573695761105}, {
      4995.635430654499, 1277.9688522669667`, 3754.761769623325}, {
      4995.521668287204, 1280.2623620761592`, 3754.1317552696783`}, {
      4997.4460782664755`, 1284.5393352013348`, 3750.1073426156136`}, {
      4996.883110810165, 1289.0350774787796`, 3749.31479552323}, {
      4998.764113227426, 1306.1619392969776`, 3740.870489423764}, {
      5000.028565888003, 1306.9771195399017`, 3738.8954036323544`}, {
      5001.671226722971, 1305.2057957728896`, 3737.316746069616}, {
      5002.565316413204, 1302.0078339848503`, 3737.23554853875}, {
      5002.164769043395, 1299.4522386936462`, 3738.6608822805933`}, {
      5002.512214303377, 1296.9704324676186`, 3739.057760058045}, {
      5003.444424800202, 1295.870179117845, 3738.1918164814474`}, {
      5005.639303438946, 1295.8233750161469`, 3735.26848226858}, {
      5006.442283310465, 1294.5675650709145`, 3734.6277154955433`}, {
      5006.3650791204755`, 1292.3385249510548`, 3735.5031149084}, {
      5001.392286818848, 1276.7541062053108`, 3747.5042142253565`}, {
      4996.91252791115, 1276.4651929821364`, 3753.573695761105}}, {{
      4994.3140042236955`, 1266.1767398289878`, 3760.5103085526985`}, {
      4992.786484563608, 1268.219585522094, 3761.8500624635667`}, {
      4992.289114050877, 1270.894786609998, 3761.6073071356914`}, {
      4993.044675228162, 1272.7946554326306`, 3759.9617453210644`}, {
      4994.6593744747715`, 1273.2062641212956`, 3757.677105869206}, {
      4996.079904398748, 1271.135616422068, 3756.4895212853035`}, {
      4996.519286565495, 1269.1677559720633`, 3756.570499295849}, {
      4996.122706385746, 1267.2385282422927`, 3757.7490742830446`}, {
      4994.3140042236955`, 1266.1767398289878`, 3760.5103085526985`}}}], 
    Line3DBox[{{{-6242.860301854942, 1132.7191776818859`, 
      656.8532606566667}, {-6243.101013869627, 1131.2997650163773`, 
      657.0116150372994}, {-6243.474564209887, 1129.7918623408705`, 
      656.0559456726999}, {-6244.181291997385, 1131.2704013120515`, 
      646.7155184353717}, {-6244.093555969086, 1133.8435070686915`, 
      643.0460759412961}, {-6243.824087827754, 1137.3971363584028`, 
      639.376418446613}, {-6242.938969558992, 1144.7800700628425`, 
      634.8269870985557}, {-6242.349378180339, 1148.388577766185, 
      634.1069351360928}, {-6242.111422702445, 1149.0206100078635`, 
      635.3033246763368}, {-6242.243748755544, 1147.0175782780836`, 
      637.6184958448143}, {-6242.4153254935545`, 1144.3462090615883`, 
      640.7311041689569}, {-6242.648088683116, 1141.4610999364124`, 
      643.6032071027857}, {-6242.863435735977, 1137.1095667717295`, 
      649.1930729043457}, {-6242.614045743912, 1137.064141503541, 
      651.6660985409824}, {-6242.6152703552825`, 1135.038239113614, 
      655.1766724150845}, {-6242.860301854942, 1132.7191776818859`, 
      656.8532606566667}}, {{-6237.106863632468, 1232.9442747692146`, 
      515.9070429278614}, {-6237.717184498875, 1230.1239890093282`, 
      515.2601167092124}, {-6237.820284180314, 1230.144321029464, 
      513.9618095808137}, {-6237.135209268798, 1234.4204053349554`, 
      512.0198572244554}, {-6236.06196311473, 1240.5429947860268`, 
      510.29203431071653`}, {-6234.982808821501, 1245.6461743486539`, 
      511.04664369400365`}, {-6234.196060612009, 1249.2233790538191`, 
      511.91221618071876`}, {-6234.203475087903, 1248.4326498474863`, 
      513.7476423271423}, {-6235.182068912447, 1243.535933343299, 
      513.7476423271423}, {-6237.106863632468, 1232.9442747692146`, 
      515.9070429278614}}, CompressedData["
1:eJx11PtXjHkcB/CmG0trUCJMmi4ruqohI/KtdsrSOruIdZJF0RLV2lEuXUwz
JJrMeHzTdJ1UJtq01iqaVHS6CKsbqm0XJbVOSZwuLs3suz9gf3h+eZ3nPM/n
+/58vh/urvANu3V1dHRW4dHD01zZHCtoL60Sbc8Z8+dR0pGeH9mRwpDYY8Wn
eXA/dmmPDzy6+8kYL5UhLvLvvrCC17zl+7vBf3cTq9IUDPGLfpoyG66JclWV
uVLycdjWXyedIa4PRyyN4C8P0bFV8Oyp9aw9GQyZrEr/ZhJ8JLPt8iYXSrZm
TjpelcmQXckVH3XhrtxDT0OXUBIZpX3EzkY9B6KXadtKqwbYh0RiZ7yfVzu0
UckQFd898TM8y1NsneFECfum801RDkMEz2pYLHwn82pB3hpH1O8ZYqW4yJDg
W8ZyQ/jMW884I/aU2FTKXNNyGVKzZYPbl/DBXK4sz46SnVmNfZI8hrC2zBTN
gg+NaKM2Labk7vO5puvzGXJG/3TYAvg/Q0GPDRZRIrZRqIfhvdtijZbAQyXf
nlm3kJK55mHtxy4xJOFOgqcHPGe6eYvWhhJT6Y3w53Chm/CrjXBFRszBNdaU
uAoksVYqhthpPjkFT+RfHmyhY0WJyryujQ9/pC3qi4bf72es/biUPGnOL3SE
S/XDVsjhK+pflBlaULJ5hvkoC164/lW5Cn40/4hntTklb5bpJhfgvyV60u5K
+MPZcncRh5IGo52ZDvCQcZ+r7fB7xvcKyHxKbC24NnKcy3/R38aDcHst53LY
PEqmsi1tWpDP3XdeaVM7SqsC4ovinOdScrIjokGDPGXZdWoreJfy9L87zChZ
KSnpnQwP2uktWgU/fPtNzbk5lFS7VEUPoV8e667f+x7+8oDQwwWud6qr+0/0
90FG1aIQ+JrR60XNs5HbCfUgxTxkqN1yYuAyTbixEH7K2ELpk4W+PFanJ8Nz
rl4zFcMjxlZb9mPeeg1MaDa81vH1nHPwUMHeXDnmc091oNk1uLB1nObAg/mO
I3ZpDLls5Hu+HF4+brDMHPW0nu33rcX83z9cKaqDx8Qot6bBvVMsa0MvMOTd
9m1DD+CNfZ3zJTivgNM5yxD3KLGF+DbBB7Oc+vYjny51YWXJeYboFV0St8I/
LfxVvQl56isrq2MYhgSKz1Q8hV9p+pC8EvlPvzml7utzqGdp8doW+FAJz1iG
fn3eG7nBQY57R91vP4Q/a1Zld6G/7p3McnsZQ6KWbDedqJNj7sfjYR5e/XTy
iOdZhizYdbShAt7L9/GXYH5eTeN6H05mSNLB2vY/4NZb43RbLCm5IpDSRilD
Gq3uLM+F8wOFP0oxn56KgYi1cGevsxo6kVt3knU75jmhMPBxTxJDZqaXJRyf
yJkbP1+G+ZfsL35bBK/5S6QfAU8aVI612VJSy4oKSYILIga8t8DD2vwNpbhf
duU8egFebB22wwvu2xSs04z7OJzYc+QRnJ24I9geHjRvBsvUgZLEEwZlHNTj
Yee13ARudmCY/QPuu7Trw2+p8HDn4pCPmFuXjTcKUrEfDDr6nZfivA4vSlg9
cIMVTVf2YZ9wHCv47+FGPgO/1MNbre83F2D/HHydnlmH3F6nOtUUwyN1pvVF
YV/5Ca+X1iPnixaSxBR4gNYszgf7bZ+iS9OKvrSHmwji4baL541fgo+uLlaO
oo/5JpqA/fDOhQp9Q+zPaQHxDe6Yh9uLG37eDA8yFk3ZDU+bHPs+jjLk//bz
f6zc7Co=
      "], {{-6137.30746771726, 1072.235446164448, 
      1368.057705175656}, {-6137.18466542526, 1071.7724630225791`, 
      1368.9710953762165`}, {-6137.369133238968, 1070.5904930029383`, 
      1369.0689568671403`}, {-6138.30268195311, 1066.0070385044305`, 
      1368.4600354362826`}, {-6139.207221545263, 1063.515175000636, 
      1366.3395820475482`}, {-6139.678721513521, 1063.3761090648588`, 
      1364.3277235562468`}, {-6139.46446791551, 1065.6568803212301`, 
      1363.5120647163149`}, {-6138.753217386246, 1068.844809523527, 
      1364.218970394644}, {-6137.926900273641, 1071.129341202025, 
      1366.14383990608}, {-6137.30746771726, 1072.235446164448, 
      1368.057705175656}}, CompressedData["
1:eJx11Ps31GkYAHBR2UZJW7mNaWId6jhbTklbpIdIW3KLLWTYcWlXZYTapsbZ
KWbLG0d9lbsuasrRlmSLxDQ7sSqXoXKZSwyS26IdkQlrH3/A/vD+8jnf877P
97mZszm+EdpaWlrb8OjgybE/euShQ7lY7KCz5tUHCryqeE89FASKPaX6+ehP
hizvXuujgLZG7V3zjgCz16MsBT1VbaG9sJ+C6PvveX4qAt/mmCWeQY84x3Kd
we8Xzrc+OdZNQHW3zfYEOl3fgvMJnXneaUt2LwH2hePiKPQQWeTZIfSSXHOx
Wz8BTtP4+RD0BS5LJN3oST5lZe8HCchXT3UEzN2Tr+d8GN0qTHTr1DABu95L
l/3Qw6lZ2ieM38rYTaMZJRA5a3XdC32J5rBxArqrR9XncDWBe1tDxN7ouywS
pZm9FHxU+O2584mAMt8z0xc9/+KLlaXvKRBZDpaIJgi0Jzev85+Ls7pN3dhD
wdONQQOFkwSMEp9fPID+nL47eqibgrMrBMVhXwjsa6EfY6NPbY79kN1FwQxr
mVo5RcB69b3vjqLLtI8s/15FgY2VZJo5Q2B4tZExF13yy4ippoMClt76JPN/
CTgHd1n8hm60eFav6B0FBp+VOu/Q/c/FeV1G91eMdQQpKRDQ7EV+swT2v7K5
fHOujhWMFH0FvtuXPU7Q8xgnBE/QzSoDA/bJKDigErDi0effCLWrQ19ezerT
bcd7tmzQNkVvHBy16UA/zX4YX9VKwdIQ/7ZYfFc/7ZKDGr3QMEQR14J5uMle
SzD+E51/eek6lotLkg0W27ylwDJBzgufJnCmViBgoAsemDB6XlNwf0RIm8L8
vLnyKssO/VodV5bbTIFwUV1BsIYA/4zK0x198ptA5+omCm7ubXFI/ExAYv2g
JhD91B5Hp2EpBfb8euHpcQI6tfoz0ejG2QNSQ/TN8/iqnWMEdDnlFYno8j7D
l9BIQUPtY676IwFvWlRJOnpjkI+ZvIECed7Vv7kjBBb8Xq17C30sLvhJHPrm
FQa1g0MEwkOv2hejL+N7R/LQ04L2fO05QECYwRl6hi6hL/QwRN/tOupb+IFA
gPvPEXXoixJpG5noLOfR85oeApY5F22b0bUEupIM9K5RftPBLqzXznJ3Jfpa
YdbBa+hMp4zc6g7sNx/D9C70ozlLkw0x/hcx3O1uSgLBfl+ZzbntuIvXNLpx
wTBXKSMwVeIa1YvOELqUx2IeTMx3yTLbCHhkJV7pQU/u9ovZj/k0y9twiN9C
oHXXhXQF+tNGNacI878pbHtk1hsCk1oPt7bP5cfOeDoV6yVyUfoomgmYWOYF
N6LnRO1I6XxDwaM49nx2EwFTjSzz+Vz8sft/ssV+EK+vGF0lJVCRVl1cht7J
ZYXxsX86r/9QwGwksK2UWVyEfrDgWZK0jYLAoknZjw0EoinJ2Tx0UV9EFQP7
82P3r/TBeozHItQuFd2qtWjFYTnuAUGO5WN0R8dVd3joZGIwrgz7nNd/srIe
PaF8hIpANynyeHsM56VUPUK3xvuHhYdu+aLvrtEJqMT5Sug3otWjG4TVpjuh
x69MmFqA82h+t/TZHxjn+Mv2hLXoFH+i2RPnd68n77gc/yueccVtObqCdrsm
A+fd7UaMegfmJ39d1IV56PXSvRYc3A+taaaFPa8JNKvKogdwXhafXJT7CPeJ
0kePLnpLwL3hn4kW9Mk+3dwU3D8xxtLKmlYCtx2tqv5ED51cqhOG+4rJ3pT6
pZ3A/+3n/wAdMf8F
      "], CompressedData["
1:eJx11Ps31HkYB/Apx6SmkmGaGrckdCUmdDE8zalJdFulexKrWp2idrctKemj
yWHG5vjS1uk2oWhyGWuKNWYmSZxcRhJGiYNM5dZW0hD7zB+wP3x/ef3w/bzP
c7MLiQgIm0ij0Xj4GeHXP6inM9YVqc+m8rV/O1LgwxZrMuUEOO9TNDT0RJpn
sBRdNxy1vq+IgOWpKQtHBUXqHW+PZ3HRI/aV3VmjIOBr9fXcJ/TqKlXNQvSA
rIGwfBWBuiGb9F50wZ7ADwUOFEzWjd5gPyGw3ehgdQ96Yk4SYxX64B7ln9EV
BDykNe06dLFCv2rbPAp6etw6G6oIvDo8ktKF7uJ9RfPanoJF3fQTjBoCucub
Q9+hZ1q13BiaS0HXT0tiHTQEWH+UqQz/oeTf385A37+d58JsIJCt+X7vI3rS
HJsri+woqLQNj65vJLCln2waQJ9YnNsrmEOBizjk0tZmAoND3Y8/oycvu9Ub
YkvB7lxNRYqWgP8h5bsx9Hz2JskMGwpi9xp3Cl8TkMdybI2xbiPtebOdrSmQ
+e4acG8jsNsjkctEl0Uyw99YUlDRuNQr4S2BTR3HkjjoYmbPvVEOBZJk078S
2jG/6+KnC9C1kw6U5symwOj942VLOwgUhdp7LUPvdSz7p3YWBSnc0w5x6FWf
JkjXoi+/+PhuLBs9f8qa39Bvi+ZHBqBv7UoTZcykoLMtUMdApxjvdgajV9eI
QitZFFSplfECfDfzTfTqCHT/mn+n9VlQYD5v+lkXzJn2LWhjDPpz19EsJrqs
duXRkjcEAqfxvC6jO07YO77cnIIXCk75WCsB0dL8oOvojLoStzomBc+CPn77
0kLgV6nbiWxD/mjHJQfRm+sjTG42EeBqL/MeoYcNrlX9MKMgLqvQcir2pS6U
qJ+ib9G91KWin/KLcfV5QWBO4qqml+hp1q4iZ/Tn7YkSfh3mZ5lI2tC/ODml
e6BPt8h4YldNQKlq7tChe1hGWfmgvwqsDdBWErD2PpPah95iOrnxIXrQqf6x
C08JhJvx/YfRbbauyFOh56jPf7coI6D+XeLww5Dn6M/xnpg/wbXBOF9JwIw1
ZWyCb5E6YWhbcB56esH9Gf4lBDTnzO/T0eWLR8ydsD5LBMMPPz8ioC9N4hqj
f6pWiKZhPSuFR2wVhQRaz3D2G6GHzBcXfUaXB+0wviMjsHHl+REaemnQg2Mt
2K9ysdxPnktgXva4yRjmObrAJ0+J/d3sXTzwVUqAXZe4T2/oe/EvO9JxHurJ
9eTj2QR0x7ivvqJ7T13sdwnnh15eG8+9RyDGvSzzA/pw09WUFpy3dXTefsjE
vRh2De5Cj6RPmqnA+ZStSBUkpxOQzXaAJvTUi84VU60oEIeeG/C6QyBLVqWt
Rc+Vjgr70J0yriV6SQj0ilQSFfp61liUF+5L/+7SDOo29n1kjbUcXbXdOcoS
96uMYz/ER49wT8pLRxcIT147jPvoy+ClbEEviNZ3pBncM6dwLe7v3UIXzxJ0
mfCkPh7d3Y/WbYf7njXIvHEe3z3TasKKQmdnJ/F/oMP4IdkVzGlxNZgVjn6h
USlvwnvSzIi4NDmDgDkn1mEXelvuhgMyvD83T3fHVmIdqs300zei86+xPSzx
Xo3bqfe8wLqNx1ibAno2n74+Dj3ySLmF030CDaa6zW6GOdR0KnrRfSZJTYsf
EJiVY1pvj27OPpy8De/krTDPDfF5BHjaZ0LDHeDNFbZuQKdV7NTFFRD4v/v8
H7X0z5k=
      "], CompressedData["
1:eJx11FtQU1cUBuA0hJSCFB3FSanYEaoU1CpQEYHMLCGgCChBixbkIuWmNMOt
IFTFQZIQIJx9RNBSKIRqBS2KRLB4AxMIFwuJaBmkI4oCHUTLXQSx2pWn89SH
/fLNnn32/tdaZ1VEfEAUm8Vi8XEZ4Bp/7Gw5BA13Jlscvtxzl4Zwx8yBm28o
sMlYua8b3Q2mm9f+QUO/3eijwrcU8B/PPbiNHsj7/U92Jw0qv1KrwncUKLcr
qy+j75y2VhxCN3Tb/c8FFoEtUye2lqPb53zudh99dbnCpYtNIM1+fTZB52a8
UGzpoiHMgOvDMiTwHW+uWIze4m9i5oA+oRIFOX5IIMw8NjcdXWoZGmWHnno7
z070EYEFwYmDyejZocv6r+L5nPGZzQoTAt5DQ7kidH6rRRYfvSiTo9KYEhjU
GRrGor/pX4huw3ftkdcI+swIvD/bwfkW3Tiqo1KI7uLXe6d7CQGLy76+em9T
Nm0XYT6hi21Ta5cSqOjneUWis+x1UTkdNEye5FqJzAlEJE3PHkJ/tnBm4+Z2
zIfzJHN+OQF/4bVvkvT7Y/wOD7fScDfv1MMDPAKv2t7f+gE9SbYxoVBDg5jK
iDj9CYF8qc06fQ73nMesBC00JFnynxRaEGjnLiml0AcMqrNn1DSQvekk7FMC
m9gxvWXoVi4OVU4qGqhM3+RZ9LDwgpTf0D2WhpORJhqcVlbK/FcQEK71MWlE
/0mb7r+ukYb9Iad1InTfsg61vu5TKWV9CbeY/bYS+eOn6ImCGtvVN5nz+zyP
TbxGdw9hBcRdZ+4D4h9djLc23AnZnWa1qoG5v2WxRGKJnnnpyNnea8x7x1ab
vLNHv5ncME3V01AvnFkejPl48p85bkPPLbig8apj8uQs3XIoBP3MioRt75U0
lKT0fhaJ+esUO+Tx6CoHpZsNetrXWXUXsV5s84AdJ9ATzSpLd9Uy9T0n6tYU
oe8/6pORdoUG+2OcrPvYD5Lp6tFz6PmzB84vRudAW+Mt7B/n5S+FdejhHPVC
VQ0NdlxBajH2m9z1KrdF/92JJdKt6I/ae2OjsT+FEo9XWvS2GywT7xqmnxvN
yjl/ob81yDjvj1405pMzzyGQLlBMDqFP+QXZ7UN3Pz6Y1orz8madV88o+mBu
5REt+sX2f0cVOF+RjcWhE+iXLm8iArxnaWBfeD7OY8JTXoneT1qI3W1rmfnl
aSXBehfZr6kwxXx+ftGwsh7n/ZLGInYc/df4C+JJ9Mjq0IPzcxSk7ghWj6Br
u2rVHph/7uAHLNFrCrRrJBv+Rk9ZlJ6zBuv10LOznDNLAdl87kE/el7bFyWH
sb7GKp7pvRkKmifkv+jQIS173An7YdlIlax7moL04FEnDXpxff1pKfbPYVP2
fS768ehdHdfR934cI++5QUPFXv7z+CkKrIc5kVXorrHl1QXYn97a/nvm6C8q
gnT6Ohp9f2VhJ/ZzF5mQzU1SYKQtH5Cidx7USUuamP11F5/PJaJbeI24BKqY
8zekaKvD0Q2UYusKNXMfWV/cKW/0d/GtAUEtzP0LhEYeTuhyJ0XPWQ3z3qLA
4eZV6OubxNKXOO/ORiUHTDCf7VdpySJ93YcTlV+1M3m+dnwpmMU5ulL0wO4o
/k9kruo49jwF//d//g+d4t3b
      "], CompressedData["
1:eJx11Ps3lHkcB3DZ3EIrkUK0KqSGrUwic/pmtS2lVLMuETEpVo1ObtVSnSR9
x21mHgZrmopcKkpXGtZdpFO7HSkbisxGsVIki7Jvf8D+8PzyOt/zfJ/P5/P+
PN8Fhu0IUlZSUuLg+QaP0TVOcNzasuqLLQW83zMZ0vRCw/5hEiU3bWv+5cMF
9WvYi7IY8pLX6BsupGSGQaAPFz6u3737A/yfFdpF69MoGbXpd7KHt78XmdX+
xhDvbSp+yzIpGTubfmoxXLQ3qGtxNkPOXd7dbSal5EJVvro6fA814I7A6z9J
rBZfoKTN5ovTR7uy6uPCBVoeUobEBJwrNL5EySu34Rsd8F4z08FSuPDalFi9
EOdZs9c9gu9RLAu+Co+P5rQ8vUqJp028XyV8phPb/zw8ZP7Ci8euU6KzIVF6
BX7aYUGp8jmGNGqH2w7fpKREmShy4JK6+yYU5+8YLmqwu0tJaMbcKgmcIx/8
6QS8l4SWrrxHyZyU4YMiuJ9bZK4afMXOz00vyykJnHhdkgA3tpVXT6GuEGbM
cUMlvrOr564AHhbZG28EH+S0RjtVU5JW4i6m8MOfJU4F6Jt5s7vT2xpKHvaL
TKfP2/JK/GzhO0+wHiyto+R2xWjn9L1DZis+70T/0/21Wr/CDQ0mnLPhWy1O
9kZgjjLz011h9ZQ8W1rKzYerpQhtMjIY0jCvqP8I3FN1/7pbcKMKZkQuYchf
wUzhbLjDm6rmmmnX+1unK50h8mXzghzxfuGU6dATeMBB50hV+I4BWb5qLSWJ
IZMeb+HtyaqBVQxD7t727g1GXfy3drJJ+EZ2mlufmCHjbwYj+OgDvRG5ZQ7m
XuaRs0sXnjdf+tSogpIBcb8yC555+F71YyFDuMLldXz02THdhuMMf1ba+GI8
lSG8iQNZv2IueV0lUl8463RVlwV8TZSFHrlFiYvvpueRcC3dC5KfUxiiMxXu
8hhzn3TIHkiFL1CZNRmXjPxEjaw1L0Kfv14WX4FbNlq23kxiiKLg3Zg3cqXH
zipugEdqH9Z8nYjcLk/bfgg5vKK7PawdbtDU5nwSrpA0tEQhtxIXJdcP8ArX
ZpEJfMuRekEUcj4xekJVzb6smrl0v6FCwJDCnly1WOyFmQqXGsP9s3Kbd8G9
+KpzRdijopzVdCU8M2dT0BhlyB9hI3X12LuIj0lcAleOZXNn4rxbzaYek2RK
kled3bcVXmAvK/4Wbrekzz6fUtJ5ZjTXB84qT/MxhKfwHggDzlDS4SW7sR+u
UxSntwTO0TeO3x9HSXGl29ABOG99QX42fP7x1bvkJyl5ur1z6hBcmpHnrIu6
NrqIH3keR67urIwPh7+r7b+UAB+KTYjxiEFdpfrSCPhu+Z+dE3DB61rZ9WOU
aFhUHOTDQ/UsRZ3oc42gQz38KCX7bvX8MH3v5I/xVtswl47xgI9ZRyjZkTfe
GQhPfSjbaI05lrhv7raE/3J2dYsnvG1YQ1sLc38guN5qCvda1P3JFf7++W2f
Prj5KV0TAVxDcTHcEV7OcvaoQ65GX7WH+uJeNX3l++bwwr1rXmkih4VP9OzF
+M7ETL6mPnw8QXO9Aq6l2Kz5Peoq5n5pmwFPSJLyy5Hz0NQ7jBX6MCvGrqAX
cz80sHybEvbivbv1llj0rY3HPv8EHj1xdVUrnMteFW2FPruOVR6Vw6t84qwv
T+9dn5d8IeZiruTKyoHrN79wiMGeNn0x9PLGHP/v//wfKz3cwg==
      "], CompressedData["
1:eJx11Pk31HsYB/CxTBMlamxzx5KuOnNCum7JKctzHEfntBCNS2lhJo62q8Va
p1wth3C+328ZauhmmltoUSJLOjRojnCNJMlSGWRrimGY0Og+8wfcHz6/vH74
LO/P8zx2vOjACF0ajeaBSw/XjYi71c2elZK9FlVMy1kK6FHjD+o0BBS4ZHhL
0INlLoYm6AfDfl/EpJHgzvwkfIruGVzfx0BP13sSfkyXBFHL+flC9JjXV2p+
fqdArE48W69PQv66WAcxevxoV34quohT/taKQULNv88Kc9HXHC2VLkfvfdUZ
FWlAgphz4HQWuo2fDTtXTYH+uGqTaAkJC+7PzgrQdd4xmspmKMhK1q+VGpGQ
fj9yh9aDXD7Q26YpsOmzfVFnTMIjZbBY6+bvb+cpVBSw6hrL7iwnYWzpR2EO
+rB1YpdwioINDh12B5gkjGSXFN1EFw3HPiyfpGCS6aEzZEpC3+LwS3fQ1XMd
rhFKCioyngs8zUlotGedK0I3PZmZYTpBQc/JWuE+CxI4ajK5An3PXsXfL79R
YKZOv+pjSUJ9SFRSA/rXmZqMRV8pcPppP69E/5IS1N6uzTNhYEvlFwoUi9nZ
PBYJ5qyUOLk2h7x1HUfGKLhIk4tI9ONtYVsn0E9lqcFmlIKlQ+wPSejS1Y4m
dK9KiVh5MpU/TEHJid3pG9HZgVn7zdG3WUseWw5RsKSlMeounvvJ7YKIg15D
yJJlgxTE5PpWDeD975m/8nBHb1xV7HN5gALNrGxzL76XZSVg7EAPWvXq3oN+
CoS/Cb6JzHCftKxL4egugenP+HIK1s4zr7thbq00/9DT6Hnvp9LS+yg43br9
ROYKEtyeHhi8jP5E3dNZ+okCrygLaDIhIdysKViAnnaQK9yO3h0hkXYvIyHU
uDm0EP0775Ei5iMFXVtYOtKlJNB4RRMV6N4aYZwRerl9/EyOIQkBN8yd6tGz
U45sNUPPEebahywmwWid7IEM3SJkr681+s1ibxljEQmbP0T+0Y3u9K6Vvxp9
pWtyzCM9Eg6HWdOG0PfrFP7lhL48/hYrSIcE3z8HmxTobaW2qXfQ+++tX9As
EOC4KjdgCn3M9n4YG+8fynSOePqDAKdzuwSz6NzpnYbX0K/VPvRLmCPAs/dd
yhy6PichIB7zKeVWabjfCZB3xzVoPWjC2Wof5nls7kxZ0AwBK3QLDmvdYgv9
DGD+0VODqlgVARsLRi+qtPlfzvbLxP/yMeN6VE0SIGg/GjGJ7s/jxcbh//bm
vcncoCQgOFnv2hi6Y+g/Uw2fMefHNjM94wSoJx1y5ejL/M4VWGL9OK81Hqn+
RkBLFfd6Jzqn9pAoaoSCTBtqpOMrAT77bHMa0IPlBrVrsD4dvX4kcND9D0X7
PEdPnb9hFYv17H/IsLtCQUDza7O8InR6vjVRp6CA25bfkobesZ5ovoU+MiBx
NMZ+STLSXM1BB76b5gp6e+ceZeM4BQshx4YG0Wufi8sT0Hm21V2m2I/udvyW
U3huySkvfiR6klRFP4j963rcot4V718nNJEEor+9sOBbiP3+2G3ldk98b9zd
nSpv9JBpn4FknA8vAwxuJ04QUD3qHO2E3uJg86YR54nw107dfsztQiVfxEZv
+KxPF+P8Gd/NUF2aIoA93MAwQN/k/cv9RJxXZZlp0bumCUhTSrPmsE/pbLd+
OXp9D03lpSagpySjYgj9/K0N4kqceyzb2Bc7ZgkI6/Vv7UE/4l38sxjdOM7u
bdI8Af83n/8DdZTzWw==
      "], CompressedData["
1:eJx11P071XcYB3DTWJKeKC5qSzk1O5w5ZJ4mdytPoSSNHpQcx1Mc4pBEw4h0
xeezShbSWuRiiwo9neo0elCOFAllhRrWg7R1xSrb2x+wH76/vK7v9f18P/f9
vm/joGgfqbqampojngl4pkQZya6KzyrvyczP6SznpPOh7og8mFG5y0aTOrjX
qowl/XBZlJd4BL7ev6OgAr4tv8phlQenYKuVYwVSRr6y1cPFcBeb/Ijz8L9C
B6/7hjBSbLCeeAB+yK5TuwZeFaRQmYcyClScnrYLPjJtyj4jT06zil7eNg5j
5JOmdEqFz8n1tSjF+wpFyu2F4YykPvvjt8OfPuoyEsF1WOEW5whG9XmjTrHw
ygbNC3X4z8beHkHUFkahmZWSCHh2TXrfYrhTbGbxsUhGGwT2OqFwyZ6pU/3d
Ofl5mFn9EcUoS6M7UwJvD/rZYKsbJ4uxLFezaEajFaGXguGRAzdXGrly4h/x
roQYRtfdDE1C4EeGqsuvOnP6JTX1hWIrozN2amfGv6/5es38mGWcHg8GpI7F
Mvp46EFuOFzPvqzFcCmnQsvvV9vJGV2ePPYiBh4w08rBbwmn0LC52dJ4RnEv
59yNG38/vrtCnThVFjWl7EpgpN3fUpgM7x2zNqLFnPJfxrTnbWPUmFg7d7ye
2fnZM3Z+zanCzVeclcjo20LtGgaPufn3LYU9p4ZNyqrg7Yw0CgIti+BvstRF
7205rWz2OilMYrS487j6MbhKHC6otOGk2ujh1wZ3V9w3PgV/O+Ia9tyaU97J
td+s34HvKIq+uwLvSU3LFS1CHrS07tbD70cofVVwteJiyWNLTv76w2WTkhnV
5pQGdMEL7ZS6+8ScJJbhtAieckc6OjBe51bnXhcLTp+n95TYwGcHzZ89Cvd+
nXLtnYjT7xnmT2fC56+ZsVvDEufuTLGdC/9pj/ZQO84VnmuQ6sIbzgRWOZtz
Eu30+TQenn9pSdk8uOjAd/MizTj9u6Lhh2Hc647PgFgENyzRNFGH/zr7x1ve
8K9axV328PxIj6ECIafJHwInHkDdUjLaVEvhrbWp7iXwCL21WgrU2erF9Ws+
8CfLO8/awNs6J025gb747neo9Idnaw7qn4Cf9VSb+Rv6aKVakLwJfu+WxWcC
nNsQ3ni4FH3/RHaoOQQ+xu+nF8H7ylsak5GTLz+sOSGD10gcPPRwr0ShWYJL
HCMWfFsQC++Y7iB4BzcNPFw/AXkbcXG9HA832L/WoBf1iRC1V19EbvuPpTE5
/J+DbzttUOduzWZHOXKu3f3WIQ7u2OaUvhd9eWh/yFQkY6SMO3VlKzx1keX0
PvTxmZzkzzBHXSeCXkfD0/4USG3R9/QmcXE15k6vcFlhGPxxcaPae+SkpaRE
koE55THySim8LTcnxxu5uhEwkBWMuRbucHq+Yfxe08hU146ThrO61jrsgfWy
VZtXw6UW6bb3kNukKO/jQdgbr6Kf1LvBV5Sey8xHzj3XGc3LxJ4Rnb8hdByv
w86jxn6YC+HJBrvL2Et1XipPa/jU0XbzcidOYZfO750Ftz/abbgQTg+nD4dg
7pZqseI87LeLD4pO68P12qM7TDCngj09gwJ40mbLJu3xvqQLNeMw11HxBjMe
SRjpH8yoHkM+FyhN/ZXYA0m7+xNq4Rf6jNVfwZvl13ImY28o7Dz0T8Fr2rqb
HsF3ZYrL/LBnOhKlra1wvTcT3Vvh5qqNX1hhL0VGjjzTxbn/t5//A3dUsy8=

      "], {{-6138.824932766492, 1077.1385001080237`, 
      1357.35580686477}, {-6138.742383408102, 1075.5779238791076`, 
      1358.9656985416984`}, {-6139.243911426029, 1072.5737579721854`, 
      1359.0744717372395`}, {-6139.8421681799755`, 1070.6906237165404`, 
      1357.8561883073262`}, {-6140.392791526827, 1070.3449326610767`, 
      1355.637038772597}, {-6140.440732933301, 1072.1202117423118`, 
      1354.0160807417199`}, {-6139.955775801549, 1075.0173750535748`, 
      1353.918167591422}, {-6139.3802441074595`, 1076.9045858942752`, 
      1355.0278303942366`}, {-6138.824932766492, 1077.1385001080237`, 
      1357.35580686477}}, CompressedData["
1:eJx11Ps7lGkcBvCJVqVsWklRm8n5kBkapSZ5WhM1csg5GinHzRXbOm1qK5to
k1K99BAiUovtsKVR2bAyUlg0u4i10qVNOsygZiPae/6A/eH95XO91/W+z/29
vw97Z4xnmAqLxbLHo4rnijgk27ZIXFtcGSmq3E7J5fseV8cNGDJgI2vUhO8r
yMvdtoOS6X/86vmzIUNSFFoCeaG4dktTxlTWTkqk4sS4ZCOGWMyXc9rhBrsD
m1tDKJlQmawJNmaIKNfmiBiu1hI0lBtKibzf3tbFhCE/zW2sy4IfSItTtIdR
cufixikXU4asNdOPTILPiQ/1yQun5O3RfpmfGUPehgRqbIeXFJhcD4+gJHd/
/0CIOUMaQ42XCOGfNqcG3Ye7W3pPJVkwpC9DL5+n/O6dhgR2JCUzGkwc8y0Z
MijklC+FN/m6sA/Cq94LRTeXM2QhN6pKHc4r/PphLzzEtCnhkRVD/O8+dps8
L659Vfb7m1Y443w9rJPDkAxuaawMvsIqoLQOPuq+wLyLyxCVbXXsYXhp96Yn
G+H8OPX7LdYMCfASBj2HqxVKi9rwn5NLzq+osmHI6rKUqgG44ZG3mVvhGlvV
3Y+vYIi1tPic8v2WY6qTCchhjT8hPjyG3L6UZvEK3sctMcxGbubD5Tc+wit8
vuwegXd9fB57Czlrxzg8zrBliMu75JJxeIKx5GUn5jKSPPf8zJUMiTrFWzAb
5/W3vPnZScxxtXXo6x3wE14bLi6Gm9BCZ2fMPTxn4YsS+FTB7KccZZ4/2Aum
BVOSxo8Jk8DfJArXCODJB5IjqoMoSXZYebIdLm5/tmoHfMZtLUsNESU9o5f4
d+Fuft4vv1N+11G//lEgJe+zanYdhl9vVFyj8MQ9puFLAygxVvCdbOE5vrzl
N+Ae8vGefn9KRM8EJQ04V/HhJkkrvFp/WWKxHyVPdM5ccYCrnraOHYGvG2rV
m/BB/se1woqRj72g30nZZ6edAlatNyWLNK41ypDzbcNOCQeu5pqXl+pFyQHV
bysWw+8d/VHqDg82Kmhy9aQkRXMk3B7zmhH4POYb+LvhpWnaWyh5KNJXeCrn
G83hpcP3rs1PFXpQUmk2ZeOJPqSwD6eWwrWtHHQOuVOiVx8ocEJ/rvRWddbB
v7eovFjpRomAsH3WoG+/SHtv/QU3/3dxw7ArJRXElW+AfjZ3dXBH4bl6mVkn
4Ems5vcs9HnftEHdmRfEtdLMg+1cuFHSpZ4h7IWKnDNvAXyO7IWlA/ycel9N
B/borntzogH8guGZsc3wY03pLRLs3a7gNjcu/NQXWaYB8Lrq+ukS7Gn82Kpi
W/g/rvEvO+D5S2hYG/a6fJZz4Ho4Mc1X9MHtturKBnEPpEemf74OLt/QyP4N
54o9mtE6D94xr8xI+f78SL+zdsih80HXnxtxn3CPmObx4ZyosvhFyG2f7JCo
CPfPHo7PGA8+UjjL6gP8LImy04TPSRlebQwXeuc8dcRc4pwrIrKWMWTUW61D
Gx6t+aDnOOb41bQc7bXwiddJLtPh3l4BV6WYu2j/J50PbIbEpRboyJFnTQWv
SdeXkhwLTk4nfNx5ZUMv/IavXvV29Go/byi6Gd7ddqJBAt9U1T12AT3c9Hdc
/QD80aIJ9mW4POLz0x7o7frolmxdfJe10Mz3JFxsvu5FJnreVVNyLxLu+Org
4G54pcKuiI+9uFnOCeuC2wRtcfWEPzTJ1tqLPfJl6QYH4bz/dz//BwkIzpI=

      "], CompressedData["
1:eJx11Gs81FkYB3BFkigWuetiaoe0FJbNZU9CF6lBSQ0Km5mRe2WLxrZb2Fj5
H5+06SatFCujYpKNxsqthoi0Uy6VMNqSLqvVatvfvPJqX5w33xfPOZ/n+T1n
fmiM746pSkpKLjjKOL5C0R5Xu0pJwm3tyMo5lJx6oF7ft40hPgOqPEt40ENp
p5IBJSG9hf5h2xkysrkhQQ++qD3r9GpDSiyaGj0m4P6nHTM/2VZKJgS7K7KM
KPEXmtwrCWGI0v3ZYjncYSSH1WVMiQ6/7NGuUIb82GK2qx2efjHX08SUkn1z
/lD2CGOImcTg6xr40L4Pum/hvTe4AstvGCJs89hfDL8dd17YbEaJ2P3VOtMd
DDncMPfwMfhd27C6vLmUTFjliHTDGXJUe5tKGlykJhlXnUeJ1vcJHH0eQ6bu
qRbEw58cOtvYDc9oDIg14zPkVmLDeR68plPH4j58C384hiVgyBmZ4z+b4bWt
B1Y6z6cked+0GMsIhgRKo7Z4w1MDXt+wg4+dMJ61ZCdD6oyH+1fBj1WYiWpQ
p4lz9iU7kiG1Uq7UDa4nWPJsNZwzONRiHMWQB5KcOFc4i2Od14H3c45bl03A
PWTnfloODzIfvBQM79Eq8LsXzZBrKSGLnOE6lmdjk9AH3ysNAbkxDPHb5Fii
qJO96NXD4+jbUs2qx2tiGZK9ap6l4l6Jk81fYhNK9s6quvMU/lzGWeMFL/zT
tzscc5marH1dEMeQvFTByU3w0X/fndLHHIVfBkd0wPvTnQe2w6uqdDWbkQf3
zMHkhfHoWxnjGqWYY5GfS5I+JXvsp10Lgr/viL+QqKgjzlzyBXK15WTtZ/vh
Nk1VS48o5vjkpvsu3UkvMFhlcAae8Vr//ec6k3WkbVzpVfhGQe3RaG3MvSiq
lw3P2FTcXq+Y42pePUuLEubOQHMX3hnzLuuXHnhgnP3jyFmU7KyOc4yGRx+s
EL2Gd59JFbI0KXH2LCh+jj6UjuRfUEWe29neku6ZlLSKW/I4cBXfsO9M4Bqd
f+flqFPi41fhkI8+nzSl623hE9xPVhtmUPKIm7PsIeZivqA6ey08KSV25gw1
Sla8KBuZDr/bF1keBB9qWbN42XS8c3ZfiTnmGxgwlBgHX3hoaw1XlRK2kaTE
Gjm5dlAWlgLvvP+cnzKNkjRR8TpH5ErZyf7ZUbhwdD3LDC5Ozr1kjxxmG3qn
FcCX/1z1XqxCSXj0D6XWyK17+WmLcjjviN3tDfB5rWOlFsh5ae6Hrjq420zN
ArkyJZszDbMWYC/UQ64ntMJFXjLjUbiI7TIyF3s0pB4+0AEPXrzSMBx1+E1T
rBZi7zre8jVkcPfKX589go+PZbOXYk/rvLsEPXA5r3yBD95pJc1042Cvi40O
cJ7AAxuzNBvgtPDFiUT8A2/k8it9cKc6oXsR+pBau/xyOf4N3WMadoo6+m7p
PRnomxdPzlKBdw5/1aS4l+UQ2haFPquxh/Ui8P/caqy72gZvTTcaq8ZcIm6W
sPrxj12J99zWAF8hN3bSxHz9xGpMIlyWmmT6G7zm990yrgYlNhsH97Lhy1zU
75TCJT1zKouQEwfPN66jwQzxivnEy4cb2fR2jSFXDqO2T9vhybKP/ZnwnJcB
HZe1Jr2nWW1KMnx2mZb1R+3JOs0Xi7by4fUTH4rEyHlWYOi4Oe4957F1rT/c
oHDKfiU9SlLsw22+hf/f//wfEHm6vQ==
      "], CompressedData["
1:eJx11OlTU2cUBvCQiCyyBCgugIprVbBUMg3gGHwVrBUw41IL6iAupLKWKpUl
GJYEEiUaLwYulEUGZceVQSPKEq0EFCEFrIigLaJCG9lcyogIffgD+uF++c2d
e9/3nOecRQcjdgiYDAaDh4eFpzxjTGIyqlL/uVz+6jshTVjtQ+qHhhT5LfRN
9tCISu3psMzuUxxNVNrWZK4RRQqZ1x5r4ZXjzskXRTQx1+wruw43zmtwuAp3
fFNl6Z+A7yzK4XsbU+QKYfFp+KVNbxPyE2lSetCtagTu7Se3OQbXGy3I4Ytp
ku4VmH1xFkWqNau27YQbNGxY8VlCE1reZB9tQhGVRWOSC5wTe7Gel0KTF694
/T6mFMkRidbbwr15ln+LpDT5wf/m4GoziuzYyVRNDqvUbe8PaWtlNBHUmVRZ
m1NExGN6DMDLBE5PzpygycaCDcKZbIokOduo2uDlc9T3D5ykSXLDksSP8IDA
gIlb8MLn8QpOKk14kzX6oxYUsc6RjJXDF01KzbLgOlHLg15LijQxIz9nw4uN
/LMm4GKfmIg2K4o4LMjpT4N3pvcavYe7Op/tuP4FRVgttRfE8OQnze675TT5
0mFttNKaIk9ds28kwlvLmF19eH+f1eGngtkU2dweHJwwfa9wB+MIeFe6tWjF
HIqMOHE64uEJ9br+cZxfa85f2AX3zZVOTXt1rp1IBr/JnV8QNZciLrOqeFL4
Gs3O5SWog+5u9sEZ8yjyVXfHQDo8yF3htgd1W5tYcygW7ltVa3kO/iHhaGUc
6nz+kPLeY7iDp/PZy/D9xi4JX6MvHK1j0XwbisicqBMa+DsFqY9BH216TxZ5
wWsnvrHugueIk4RO6Ls2+GqDP1xMFB/ewuXGabJY5IRfWibYA+9sbTQyQ39l
z4au34uniZeZoMgVfmT+Kb1VcOeB7mo2cng+aSiQAXcrfdTzLdzLY4HSH7kV
X11feA3nnKqNux8Iz7ypCKuIpcnbSs++zfCYeCuFBB4rN2SPR9Ok03fyThPq
YzfM5F6Ad/9oemJLFE1MNXvtOfDFC0XsWnhjhMTO4Bj6qD1Qdxp11hVI1z6H
p//ybMI3kiaprbruNvRrKitgbAo+RVSCsiM0sZ3J2m0Atxr+VG2LuUsLN70T
9DNN4vqKM9ag730Lbwy6wHuMQlNWROCcoc1p25CTQZ+SwO/hl0tqZAPhNGnc
K2wJQq6KxSYPQ+CpGm7wbHhUT75hDHKo5PpuSYK7SWP3eYbRJGOeHysFuQ1Y
FvIuE77SbeT40VDkOe6s6DRy7u56Y6gEXmHIf20KX3pGqcrFvPTnvX6mgn84
+dGjLIQmErLO4hLmi9NuNqqGGw3q7l6Dp0Y+KW/CPMaHWdTfhzfPWKx3C17n
vmOGDvM71u1q0w6P1vzecQfeE/ZH3ly4ta06tAPuMziQx8V/hynl2C7sAa/S
R/qP4Kql0ppyeIjwyvYi7A1xUHFkC/zfuKxKOe51xtTjoQFc/+4Vopk+Z0+q
ZRjqMLTV3UCIvcTdqrSogz9YUCX1/okmoU0Mlj78qYVIdhmuF95SnIk6sx0V
+SXYe7u6iuecgzPqPw+/QF+2Z76u2Q9fxe7MPwVnFxpuXH2UJuseON/mwA8/
nmsvhCv+as6NQt//UcRk28O5Xk38/fAjL3UVjsjJ8ehfx5fAV/fobd4En6W4
ndmLXPlLem9vgvtF1fmtnK7PGKsiI4YmL80ZkyL4/+3n/wAr6eps
      "], CompressedData["
1:eJx11H00VHkYB/DJ29lNSS+EpJQZnc6Jakjb9PLroI1GzCDrrZYKvdg1SipG
mmRUVOPHTYnSkdGbcdrs2CTDbiprhFAkIaNYu9SJNpL9Tv/vH/fccz7nnnt/
93m+z2Md+rNwpw6LxVqDS/frfWuo8WOlKvyVz0XJNobUtSV2+B2nZMypqujT
I6XKp5Jr3gkPVa5dJ0+mxEFzv+ktnN1rsK4efoo3WDsphZKy5LOfnsEFZart
KrjK7rp/0AlKJkTDsmr4elWIXjH8xUenmVUnKSlxsbe4B6fu7r5c+JPCiRzL
VDyfJeLdgHcJPh9YCHfIbLCNTaMk6LiUnwc/4JvuJt/KkPzxz5Lq05SUf1jg
kQvPvChX3w9miLA1SzH5LCX6u45GZ8K/58muOsMHgstuOcsoOXxdMSMDHvVr
5O2gIIa0nunwDEunRH1C10frJYfEG2MDGeJVkxERTilZ0Fc8T/sei3xpSHoA
Q4oucb2DMyihw7FmWfC5V1hShT9DFPWLpZxMSkp/6c3OhvOSEuzVPzCkcOrG
tB44Z1fL1/OX3YjsGfBjSJ+30csYhpIYpYOuAj77mosgbwtDDPo5+Rr4jSGn
f5XwgxFxZn/4MsSks7Cbe46SKJsu8WP4UqmeRbwPQwxfsff4wtuXeyub4VXb
Rt7neTMkJjA/60f4qqfdif1wvni9frCQIQHWi4zXwZ1VM0zH4OPxtsPJAoYw
I/KGIXy3cO3+hTORB/GHWbNdvRhyfpPJylR4XpSHiAPnWJZTkSdDXOe2mxjA
G7mNAzx4bq/G4vJmhgQf2rLGG/+b3qZIEMJfWIYVPPFgSORd0bcnUbc67l3B
DvjSxqS0SHjjBzH/Euo8aUV2thhuW2+Tf4nPkIdLNNxc9KW4NpOTDu+xSDYl
8DaX17I49DH8julqOTyuwnFy1yaG7JWlVnqg70Ec/fZ78KdFs45I4FdDa6tm
naGEtfjCMjVcU8qtk8Gf3mkJf4BctX2JE/XAa6+wHjnAK91Wd+xGDi9PsvUa
hq865XbOBb7B9Ka38SlKkq3flhrUKFWGdmNvfOAvzvpduI6c7/czcZwOb+pr
Tm2AC2c2TfXCXFi6DyrmwGmEybLNOP8YKzBxGHMkYAtz2fAHPL6wBl7yqHFe
AebuY2AAyx4eb5U25Ir6bI8VDoQkUTIoWtjiBP/n7z7DSniNcEiz6BglOyvq
ud/BE6XnR+Wof05E+SoDCSVhU+YMroQnbRsNSUO/PHRem+kfpSRplGel9XFl
04Zo9PfytRSbaYmUiMz3zHeER/dbLPBFHo5out+4HkEfDSuYZfCDf/UcdkJ+
0vaNPpcmUJL6akWC9vxGKa5XkpC3zhFTyYiYkpffCDlW8HY6Z/d25DP3qntJ
Mvx3PbsaI7ha4ie5hTyr992MXgl3Hs/J14WffF5VmoL8xzTteGsK9+fz9N+h
/oOGdpVPMC/W0fy7hnB+imVON3x9QOcbOeZrU3OGyxLtd3+qzv8TXm0ua+vH
PE6pd5DGwRM36oz+BleITw9VYH5NbIbYnfCiLwO3CrT5lL8XU8y7/OHjzWH4
r2PFsY4yuJFby8QO7IdnzbftzVGHpgO+BYfhOX3GcY7YJxMfd+/thjvp7YkJ
hXuU39mrh73UIWhdrkY9u9qmUU94lKJLshM+b3qDTx3qHx/SaqSdl/alF6Y9
gOssZrPeoV/F/gVf2PBy4XwzG+w9T9a951bo+//t5/8A3HTuow==
      "], {{-6088.315005781844, 1441.460846747021, 
      1241.9439229156}, {-6088.662999255157, 1440.0844037180389`, 
      1241.8347239746759`}, {-6089.072211622403, 1440.5178528129177`, 
      1239.3230439942247`}, {-6088.546424602092, 1445.331716185687, 
      1236.2978426143166`}, {-6087.818382736819, 1448.3019744106396`, 
      1236.407060692353}, {-6087.492453913518, 1448.7857473443544`, 
      1237.4446136445201`}, {-6088.105040269286, 1444.7779643369863`, 
      1239.1155484513881`}, {-6088.315005781844, 1441.460846747021, 
      1241.9439229156}}, CompressedData["
1:eJx11P03lGkYB/BQKiaWJdWmLRPSmtam2mErt1osTeWU14g1XielKZP25ZgN
5cionnukBpOQvDaiwUyWNZ1q0gpFrW17Q1Rb28s5m6kI+50/YH94fvmc+zzP
/VzX97oW8XdviTWcMmXKGjxGeLIcC7a6LlNrXh5KuxVUSYl4tJjHDWDImgHz
PWx4bI8fp7SKkivN0zb4wSttWIdnwc8UjovCaiixO817mgyfG39XqeOoNYPF
LHMrBSVqWbZBAzxq3vLxh/As0md0o5aSnUk9P84MZEjuo/jwDnhJ49sSqzpK
FH5PjqXAGcOhtnp4XN7i5R31lJyt7GWPwhVcC5kc3uIzWGeppOTtJp+gnCCG
nDhbdyETLpxcUhbaQInn1OFMl2CG8J+vchTCr+ZPty1uxH1+u71vCG7X4W0c
Ae8K25y6v4mSXtsZ7MoQhuR3nRB4w2cbJTm3qij5VPnx1b5Qhiw4kFbIhTcu
OMC3V1NS5zqQv2EbQ45fthXYw0NL09+PwQssFWynMIa0x0ywbeBFHr/8EX2R
Eu2uTpZJOEMszf5pngHnuV3P74AL49YGvoJnLB3iTzqrNc2x/7ZdgZc6BJ/q
2c6QwylNHB28fHPNZCs8w0R0SBnBEIHhtbUv4RY3Hu9Vwbdn6RRHIhkyO8et
7W8412Ty6Dr4KM++L+J7htxhe1Y9gRvMvvW+E/fs7ncasI9iSFV2iu4ZvC/R
0egF/ndPZ6fpMDwrNXVUf746IGr1fni/NHIO5TNka+gD9Wv4m9sjCgfULd54
Z+ayaIaInyXZvYUrNl7y8kKd20/wdRfhXwudcifgmvhq3Qj6FR3+rp0Tg/yc
tJLPRB3qbDJsKi5QkmP1xbgUnsvyU1rDrTtPT4Sg76kRwQEP4ZtfTHQ7wk07
csbl5ykZDl1nZhnLkI3GmpaV8MVLW1f7I1f9gbU7XOAW7LhSX3jQfPfsvHOU
uMs0vly4UrWrRN/3MV7U2XvVlCS0GzjbwwXz5dkiOCtSlGiHnGc+esqM4bsJ
Bxu99LnapxXu/bOCkjsRPFkL3Oj4KaMieGrB50cXllMSI000jIPXa6U1TXBO
pHmAoAznm+a+fo86dD6pcNDnPCQyOWzRGUoMVjqJU+AzxUqvfvgh5c27d0uQ
z1+fbhpCnRuCTZ31c2SR3vVMWkxJ+JIaAw+4cU1CNQtz99y8L5Z3mpIHzYPW
EvTrZp9ao5/TDMFlf+MiSjTWaZJ29F1rO7JxFfxg/Sr3L09R4j/CLf2AnHg0
zVrkC//Q8DwwRE7JsXD+BzZcpdO5BMMlqkjvN4WUvB4bYXyQt2/Wb/1hB9zr
el7jXHjr6PnKOOSTNAvvJ8OrYt9JFsJH3SQkDXku4ZmsSYdrGx2b1AWUJH72
5p0cc7GkYdoeCbw0YYugDT5rdblZC+bo2vTvfI/B5y2rzViB90jUHlMfY+7k
QrfHJ+FF0Q+07vBg3oD6E3hR9/SRPHj31Z/HbHH/ssPsc76Y393T/mqR6fcV
P8ZfCh9TR4uzMe+65PoL+vOXglxVItThyr3CpIfYG+PLtWZS+IqwA7VBqNvx
bT7jBF7QYyLLgg+fT1KVo869ZEZ1E/ZPcZSYiOGyiDm2OvQl46SqwRPOZU1d
r6+Dp8jXxKuUkoNmQ6n3sfe+MjVbGg//WOHglIu+q9bnfiuFn+H85LINfmRQ
VNaPnJRx/AXb4a6/p631gXt3XbOWI1cGvcYsD/j/7ef/AB7FzVM=
      "], CompressedData["
1:eJx11Os7lGkcB/CZKEpqnXIYlJpNmG1LEopuTIbVNkiUTAeWajrZ5to0k700
HRyuDp67wxoSJdRIVhSzDZGoLEVXEZtV1qUcouN0MG326w/YF8+bz4vnee7v
7/u77aJ2hsSMY7FYnnh08NQz+ypMXVQ1khe9SQaxlIiSV4Ztm8qQBd1NTSML
VDUFU36dXbuJkhUt3BH2NwxZnXfDrBd+SMdZNnELJewtfjcK4ZwPWXEPx/wS
S9wN39qhHic2YsihsNpJlfB68eIk+62UPPARbVpizJCHVWwnJVxxkRz/Cj+c
MmxgZcIQllBVdgoek7t/Xus2SuzNJUETTBliTNLuyOHGWc+ll7ZT4uze+Ohf
eMqJeye3w9UfrqbJd1ByO/+xxRczhlhqsoLXwZfWcP/pg++RJCRopzFE62p9
dwWcph5mB+2kJNmyIYdtwZDXVtpIPjy0xHCfP3ywNjNK1xLnUo6aeMAbch3M
veEn6jR1OlYMCdbN5MyDm7BC+9zho4o2Py1ceq9dMhvODUl/5wx/pvHrG+Aw
xECVt24m/Kr2VmkV/kfUFNDfaM0Q7syBSlu40ElS7w9PSX+jPGfDkLAsj5YZ
8EvusvwYnHdxgr84xpYhGgdV2Sy4JvEz5wDyKedsGDCdzpCv5T5B9vDBH1ws
ziHPGacDxKXwulNbHjnBU08o1NViSr5uNhO7z2BI2/EWkSt877K42DjMK1+n
NbUAnrH87y/ecN4RaYfdZkpkv6QrP8G71uoUjOUmCHWb/Ag94TTH9s+zY4jA
aJdrJPzdDmlvUgwlrtMEyhC4auHLd2J4/BK3dI+fKCkp93EIhxcElWlkcEnV
x/mvoihx3BXpsAR+UyZvZOCPPVglCzdSkrQ/ucEAXjfZqzsX/rQo7O7gekrO
8esnVuF/RjwWdV2DW/H8k3LXUXJNNv7PcPiAXmjnffgD2RzBYCQldc89C9uR
w1vTy6098FaXwrjzaylp7hUNeMMjFGnJn+G2SfdtIiPQ24ZQxwzkzOeFOxpg
LxLtI3gH11BiaWk00IG5rPy5be9MeNOuXrbXakriPeRKffiTqOlhC+GGp4OH
Y8MpaQjO0HfCfCuK5Oof4ULx9WS7MEoa9f3veKEPN1+cVEbCy+Uhw3+sosRC
98AyAfrDX/mxMw7OVjU394RS8raU3yNAD2fdMJ26Dy67Mj82AU4tzpwNQG/b
jy46kwbfQE28zOALONtEPubYl+mVepnwZ4njy6zhiur2jKXof/FVT988+OUL
wglcuF5n3ypv7Ev9UYVbMVxJNto7wSMSix4EYL+OuXS8qIDrLPOd5gz/TZdn
uAZ7asO/PaqG62fLgi7Ad3fm6O/BXteOE5yvhntGFy6yxrn82Cx+Lu4Bbk/U
kSp4cfzF4FG4ZGpKzlPcG1cqN6uvw6WaoefdyCdlt692Lnx9dhd37LsX1n73
6hbyHJIFxFPcS1yXbwtK4U+85MH5yL+cWyOZBP90qt/qIjw8wrHoPeY137wt
LnsKQ9wK+9Oy4IOdfUJfzF3YmDo3EK69ZjI8lttf0gk8ip503dW8NIYfq7oe
vR9+5Czd2CWipOIJi/fGkCF7f3+ZIxnLrTSw3BE9/OLT1zgEdxnxLI6Gp3wK
ct69gRKfiLwiPbzHlj++ZWzus4fvV36PntuYTZzrDhcqRtWL4YFr9Eqk0ZS0
u7u9TobrNYir58DfL7+trcW+NI9+dB6C/9/9/B8pMquh
      "], CompressedData["
1:eJx11OlTU1cYBvDIIkLREW0Q1LJVXEfBlq2oeDrV2rqRWhEp1AoGFBEsKhQd
KFgFJJLkIGkEFCSAhggJJiRADUpEixBQER2qVcAZcSEEpIAKQw197h/QD/fL
byY35z7vc17XiIPbIs1YLNYaPOZ4cmuShxp96nQhqijTdA0lbDur669PUbJA
kxhaAy/0+CfQEl7zInaDA4+SWPObYQr4lYpva/lqSkL3cAM3n6aENWrcdwme
uSd/oT18rGjAJzWbklOXY3aVwpOGDS88qilpn0i8ruBTIim83XMOHm3lJ61T
UbJtS09Up4ASG6cnOfnwDv0yQ4eSEuuo4qERISXJp/v7zsJz+/YXD1yhJDGx
u35KDiWHCxxfi+F3heMt0+B9W6taJ+FWkrpo5v2/BltXaRSU3OHkdfWfocTZ
bHYv47PVns/uyylRfFc1/GcuJdwMy/gS+OroHI+0Sko4EzX8NBElj6TNPTL4
+BCZ51FByVKz5jb33ynpueVJVfAuqe2LbhklYQ8alZXwjqnjglq4kKX3MpVT
Is1/qHUUU1IxGnmTyXkgveRqlZSSpKGnxj1wUfrd5ffg9UvG2BGXKFE6NT2n
8CPvPjE9hd8bzNzLvoice1Jdz8O9s49ZGOFxF70ELaWU+P/4qTsPPmWT8ep7
5jyHjckDJfj9vvbAIPj3U3nR1r51ullFosRSCSW9f4USc/iBtvrmOfDU0BC/
lmLkaHgcLMD5+TRzxSK43NlG8eYC8rnhqWbBy25dU/nCl6f1vrKHH6Q69k7k
s32uTLQObp7jqJQXUfJRf3yyCHmadEtStsFLTnqtWA+fbv3Hojrkfyht/epw
eKvDqoCuQvQktUB8G/Pqftfa9TM8amxVXwLcrdhLdoPiew/rvZPhi23kyhS4
c3hKyhX0ge3CH82GF3jpnOfAB2cEiIToz7+Pyo6K4KvspMEC+Od/K/v3om+2
ZzR2EvjIBtWEGO5dbs71Qz8zHng7yOC+th/zZ+CcnpqsMAv0ecUdtZ0Sfjxx
in0GfOPOL47cz6KkNTbkZR08q2Vexgd4So3cIMF94SRnXdTCJ1d68p4jn2Zb
17XHMimJy5s2dg1eKOetY3Iuf+WvDMugRGy7vaQBHvKQZCswl/Fyu2db0ynJ
4WeX6eBx4h3huZjjuJu2lXOSkresDy8Z36U9WP0L5j4SaqjmnqBEFhUvZf53
WRuboy+jpN9Fk8f7jZKzzh0OGvi+xbut5qNX7OE37+8dp6RtsSBaBQ9qlwSZ
4JvMoqqXwt2/qlUzOZhiJnK3oreBAYqW0jRKgtiNxgvwExVSdhF6fiC+QOsD
9/vmtIrJ+QjHlGe8TEmTSpdpTMV9qa388hS844H+a3/cI0sXR2EDfKaVeP8x
+KiLgykD9265cpCjhl8Kn+/GzL1JFjGbi3s6uWFySxN85Kay5Cd4PufNW0UV
etse5j4Bn78+QcX0amBU6MPDfdeyDb5bcJ7Ouec5TA8XmRYmcbE3wtPkCxrh
5wY/e+0Ft6j8wSIAe6amd+2dQHxvbERg+AJ4Va9Qb4+9NNPDI34MPudMu3YW
vH8X1Q/CJTbcmfXIszl9Y6cFfPdm7sZg7DdHJ/eYPOTvGlnt+A737nHFce8G
+GXdPLMszCvMMHH0JbNXD/FOLMT+XOv6YVSA+To/i1zyBD5jx5DaCR7XGO9e
jD5sLnZbyewBZULZIXt4w5qiWTfQn//bz/8BCBztyw==
      "], CompressedData["
1:eJx11PtXjHkcB/DpJllSUym3IxErWqpTi6Z8zZCJTW6JLrooYxRmTlvZXdIq
3evr6TKIQ6OLaWz3UrJUU7nUsqSiC7nUUdqJRZeTaN/zB+wPzy+v85zn+Xw/
n/fnu9D/6I5AdRaL5YBHA09sxn2tGLuqWl7iSO2FwwzJCNb3iDpLiWPYYNIx
+M4T0bqp8ATuRJhlJiUpLB3pIXh+C18nDm6QJDQaukiJVbPxKT94pPaCuTPh
vXVHyiovUyJbNMHbCd9QPdqpCa/SK9WLu0LJpU5/s81w8dSCwcRghiyUh1zb
nkuJ4PIa1/XwYdkeWwO4/Jn01HcySlK5GU2r4Z0crarMIIasOeouqZBTYhCh
22oL17kV/aLsEENmeesHcgsoKX9nl/0DPO6v9t0ceI/3e6WsiBJT2VUzS/iO
oIHE3UKGrKg/LB4socRmb2fpCrjzwIfT4oMMud9Xohwto2Sp2nm+6n2HzzWc
ZAFDupdwsx9VULKyid2r+r7ez57F+QcYYnvWIzW0kpL5nKUa1nDFJxP7u4EM
CdAKH3tQRcnvt5/z7OHj1wpbwwIYwpT//XTwBiXhAv4rAo9JGLWQ7GeIqCxt
5Z1qSkihrjlfVc+UjuLr/gz5R2zj6HKTEvPK3Eo3uNylIlDox5Cg6ycaz8DH
LJrafODHTlqMJfoypM74dEMsPCSJ5xcE1+DVnSnyYUh8TMNNOzhnf59INd/c
bSXrWvcxxCi82OI8/mvwPlMzFi68U6s57s2QiY3L9YtQZ6GT6RuJqv6asuum
cGW/U3EUznUxosZYBn9afsKb78WQitlpdw3Rh0TF7vgK+MtS0/weT4bkLpMY
C9G3ntYd5vfhqU8mtXXg1U56WZHo8+ME+ugZvHx01qDUgyGHDjT4BWAub0Kc
fQbg7OyHInu4leba+XMxx+Fho65h+NnHI/4b4KdjV7Xk/EFJ26bQl9o/VtV2
rfnl84u9DAnb6srVR04k874FGcFtnObl98MbRSPKgKuox/mr1yJ45oyQvI9w
h+AUt5wcSljbvX61gj/Iy6ifgA/dE1R3SimpVPgr1sK1w5LqQ/HfsOjbXtOQ
80k7cRcPrvDwvTIE13eQfuNiL1h1Dwt/grcLGl0FOG/R4j9N489T4rE6meMG
z0puaFb1x0W0pL5bQslimxvLvOHNk+Zt7uinwvB4v2s6JU96nLx94PJVBxc4
ov97H/duH2AwL0bc4Qs/mK7zdhHmaH10U00eRc5ni3h+cDWr9ndTMXcvn1Cb
zGRKLgja81Tv7+d/s1bC3V18jtQlUtKf90RD9X1WQUzNI+RHaLBeNC+Bkjmc
zOee8Kljt+zKkLdbbepf6+MoKZ3hQN3h3f5dHenIp+C3Vey8WErGJSZvVee1
PXVxSy/yXNeS+++DGEoCI+0ubYTn8N811yP/6V32z53gnpOzWA5ww3089yvY
l7Stajs14AGZHR8s4XNEullD2C/OLm3DaXBWpGSPGdwsuV67Gfto+Drb1xvu
mKPlrJqvmm+8Xx721yRKNDYOd3vFVkyBpwwsLYjEvrtLt3W9Rp13Usv7RpAf
pvy1ugfuh6bDJnzdeEr85EJZH5wb6uJhjXuGqxiMOIk+pLHNHVX5dM7PYl+A
xwT3OSxPwnlLG6+p8vz9lnPD6rivhtbei9ZPwb5IN09X5b+Wa939DC4T5bdY
nkHeSj5G5aj2q/fLrnW4D8XyL+bhqZR8Uvayz8GPb5X52sGnH4iIUGLu/3c/
/wdmLM9I
      "], CompressedData["
1:eJx11Hk01WkYB3BLEyqDGpUZ3GsLIYkbWa6H0nLTTImZOooZDVfX0kYnw1HZ
8quRcd/bmCxlGS2cECktw3UYa7aSjhRNTJFmtFGkzLe/+mv+eP/5/PGe932e
7/MYBOzyClRSUFBwwVHGedDYfVCRKuU5hsmi/YsYTV3trM7ncSRuCNYbda2U
f60t77czY6QUkv5LjT5HPINjZvfhd/bUKb4yZ3RqpCFNGf5Xr3tqI1x92CWv
1IKRzCduXbgeRwtsLz8phmcHu9bct2JUJpDbvNflSJbsuyoDPjE0mHrSmtHG
rrDD5+GZjYVPE+CRLTvbfGwYiXs61OPhK780P7ATXnJGHnfI9pMPDQXe9IJH
/51Z4SRgNBDOiUrh0qN+iU7wGc2bwsaXMxre2zupjPcI/PkNRvB3Uu/6UgdG
OV9kz42E+wiTvGbDm4L2H5c4MsqtnSGeh39pXS16/EZYKQ/fo6lW4cTI/1mT
YTtc0iR6OACfkXVNddqZUYK7ktdF1K2v76fYdnhyaHj1WiGjbE0xr5DPUX2A
mrIcrvL0R3UjV0Yahfzvmww4CthQZFkKN+XdmfsevmtgWDhtyJH7UEtOAXyO
86ppMTGKrswscTfGv/xWX8uA27Yn5d+CvxV4VGWZcBQSYyRLhRut1dR2cWN0
W9CQO9OUI9MPai0cXDJWH+MA13FSTYsy40i3nBuNh9/c/PCOLfw3t4nPXphz
NDWk//NHF7hpOslx/5HV7A+JBUdLV64w++hjxfNzPOF9EZ0Rg5YcRabM35kM
V6jVFUjw/iLP8zyfJRx9p+dclwIPfh3afwR1sPU2Uqy25qjL7v52Bu8wq7pe
4MJo9KmLuo4NR7nV6xtPwdskUSPeqGeKWLHKbxlHCyMlEyXw6KrOoBPoS8fK
6G+lthzFFZRq34AXvasz9ljBaO6KiuZCO/TF22pJGzzvUOuWJHtG5suqZPkC
jm7/o3L6Ibw7OOx2A3LSW5kvT1zOka1f0dlXcAe9lno1O0axZRFRbvYcbb+4
0FUFeZhcXGDhuYzRiJlQqQd+4o3+CB9+Sdqow1/K6Jv22QtFDhx57i3/VQB/
JGvZumMJI/t9j+Iz4YtGtzwTwUWh0fyzloyuh7eO18BdizV8guALmvvDYxZ/
8tyNPmMxcIWQF441mDt/jcfXZPBSdZmVDF754XnsTFNG1saac9zg+04e7ymC
N4Tx9D1NGPV3uY23453qYXkOdXC7C4EiqRGjkKNRv9vAJ0/USu/C47YI6tYY
MuKfOTMSiTpUpGUmjsADA8wuv+cz2tpRVc1Qt3sm1Z3T8JF7Bv+W8xhZbCgf
y0CdxdIeoQb2Rp//6bQefUblP0RURqMvN9IrpvThZWO7I6b1kH+1cd5a9HGN
8+uj5nBfldh4GfxRq9UuJfRd66WjxA7ueNnhljl8m+XnO84hJzrDQcmu8JQD
m7yqdRl59B8/LESuZlXYizzgqle6e2vhT85d7G5FDrsHe63Xwy+8TT/cCM+Y
DON5I7epWhPZH91Vo/krZ9yvvL9NfA8533FEquQJtxdeKi6BZ729+zIAc6Ho
pPVgNbzLucdUin9lpqQpP1/EkYVvutQNvk3bIyECdZgpmsVPwtxtHhxIXQ6v
WP8y70/UTbXOV8sSc3prjUbhUvhj3arEswaM3jgW5Axhrp/3HcvkwWXSZBMx
+rLu0N2aMuyB3VNPEufByzOehV4xZnSw3CElC3vj//bzf5pOs8o=
      "], CompressedData["
1:eJx11Pk3lXkcB/BbSiY3nQjFVZZslUu2aPOxJkJpptVypCRZsu8KX+sUzZNl
xoxk6ZyRmpoYLjEhp6NEGZWlJscMl1NppM3ocOd9/4D54fnl9cPzfL+f9/v5
aB0N8zy+kMfjbccjgyd+ZuXeIm9Rq+vqTl91I44Sc839nYIZnQ961J8E97Hq
6Hoq5OjLdVIShTA6XffYIBDe3H0zVNOEI0XLgvRdYYz4XdvGPOFdFWoVQZs4
0mpv95ALZ3SP99J7O9xjtW1FnSlH8y8l7HkEI0t3z4qN8AFhz1CeGUcvM+0S
eqIYTV7af0IFzjak7Rsw52jq4FcaAzGM7vN1Ny6Gj4wUbbhlwdErii57F8fI
NLtm8L2XqFV90d1331py1Oel/M+KREaHSg9eGYeLjyQKP8Ft+/6Wt05mpNMs
2vMMrmWqN/1oM0fKdm6HD51hNHf7udMDeGvUrNsWK46G/Q8MRaQyOiKXkX8X
fqXcwNkCHpQfvjM9ndE1h+TNLfDLjaU5JnCPJl+rmAxGfQENBg3wsWUvFmyE
yx2fSfDMYuT7ODz+V/h9s8w2fXh+Vp7tihxGNi28ZKmbJuwIrMN5tmiL46tz
Gbn1185JfbinMdgWrjqouk79HCMrrSYbETxacNLfF/dyPKDU4Hee0acx+/FG
uA/95jCJ+UhqOqvj8hgtGsoJaYN7ZWwT8uHdlvqRR/IZbW0Z6e2B2w8Xrm3G
/BMybJWWXmCke6+ysx9+dEmf4AXyMjn0IjYRPlXtXSWGL0ic0DuPfEPyG0Pr
4e0L2rQ+wcO671bvQB/GHy9MvQOPOGYRIou8auRD702hP40VLko5cKcbBVcF
cPt4+tcMfassvfmLqvS7RpPeQvgOP03t8Q0cXfD/SzMC55RMl7UR/Gc9u56S
9RylhNvoFuFeGl2XJQfhEutTKcMGHBXtcuZlYg7hUSGXguFqk2UrC/Q5El0t
P+OMub1WFo+dhcvYyBrs0uNIpSHu6TPMOdIx4GIhfDBuJHN+HUcdWRNBjshl
+mrJlxvwsoaYXgcdjrZeez+RjRwlUb37WuHWt8d2B2lzFJ1tGFWL3D+7mm95
Bnd+0sFpanFU7OGa3oqeHP99wHocrjrakuOkyZFPpexoPXq1vru4ZA5ubOpm
OLeGo6GLy4+VoocKTqdHFHxErTl31L7WgatHNiono7fZNirL1eFJLnz5Rg3M
IXN+5hv0vKajU1cf3iWuuekOr6wLr7aIx/yr83TN4OVvJA6jAo4W96/SF8Ti
v1NQXGIDL54sbI+Hfw4oqJePZnSryeuwI9x51eqYFLhRlV/bqkhGaQ+S/F3h
rzuSptPgt2Q74rfhv1ZLiuHvhQdaBulnwfd+X349GXtg/0PFpVKfdzFMnoV/
9EsLG8PeECSa5+6Rvn/ZH9EhOGfTh5Q7kdgz8wcqnF3gy2V4Zzxw3/3FmaeM
TzFisaIEeziFamhuWsuR0ErRWCWIkbD4yZetcPs1z7crYp4J2rv9zE4ySj2n
8FB6326LzQPTcB63c2FWIKOyWZ/runDdwmw/O+Q1573nR2X47b6qURW4gB8W
/B3yfScu8Bk4wajpo6iED4/Q+CnmNPoQ0C+60AuX+SHc5DPy8q63f22E/lyT
nW1ajPdUid4KxfBJ7Vz3RPRtwnBaKxY+qOOV/gTecvmVaSf6OWarUSvAOe2c
zn1olu63dGMFhj4riuVrZ+DuXVNvquF1Z+PePkL/19cv+1MOc/i//fwf2NS3
fg==
      "], CompressedData["
1:eJx11Ps3lWkUB/BDN5eh5qTCDKEodUIpZUrtzKSs0oWphinJJUmlMFSi8mgk
R/I6BkkpB7lLnCGM+6VTFqkI00UT0tDJTEPKLPM9f8D88PzyWc961/Pu795b
39XH3kORx+NZ4UzCiZhaJfF1LKlqSmg6JCjjKOzc45rum4yCW/7odYMbyvwj
RXCt2c95fWmMOvz2WjnB91iRThS80SJpLi+TUdTKpC3b5a52MvA83NA7aMgs
h1H5uHHiZniEOM9BHW5s6NZ8KJ9RpRfxNsKF1dnfxd/lqH+zOCGpkNGA+T9G
VvC//bbdjoZ7Sm231RYzilN92rcKHlRo2K8Jb3uyPrG9hNGLRZpu38DtZtYo
mJdy1Kot1JWWMRIURX6wgBsoJHeWl3B0vj1rZ+JvjCwqrvOXw9+LBnQ2wut6
whavrmYkY0bLV8KzBIohrr9yFJCnEpVey+jV2GS11XIP4rxDJBydDn74vr2e
Ucv1OPE6OD9eMHilmKOavjV8aSOjaw3dgg1wX+8FRiVFHEWs6Z5+7B4jUWRX
vLwOXFamevsdjryMTvg1SxkVf+s/wwFuNDWi4UMhR7bTDqc8vs9oa5ve4R/h
ZgcmBBrwI/Vp5rEPGGlpqPzpDp9itsJv+W2O4n2Vaj7AA+qyvXzgZzwbg3cW
cDQynpI6uZlRf+YC2Sm4VzrfPjCfo7ib4tgq3O+ONrAMhbu0ehxUhqtriGv1
4fOzq3qi4U6Nm14K8jiq0BpWMMV7VE6H+F+DK1uFq+7IRY5zU/cP478W7BdJ
bsErxScO9uZw5Gzn3LKviZFkjbVOqTyXef+OqcCn6bsfFzYg95Gflerhti4B
V1OyOVo93l/hW4f7nl82PZT3Sc3RHgt4btXZMY0aRukFIWov4GuDa4XNWRwN
POnTFFYySl42a3wQLlIKLHWHq+/vrH1Qzmju7g69EXh2QkGZN3zdPU+1l6WM
XJcFZEzA/Q4PKfnBrR/t4B5JGIV0aO9WdiqpKp1y9adT8M4r/JOldxg9NlDY
MhPelO4yJxR+ycEm70IBo4U89r0O3KdL0BQBV43Nmrkjl5F6e4ZMHz7o79rG
w/tXTSEbvSxGI587/YzkzpzvB8JbLabbjKQzUmW3bi2Cx23tDH8Hb0jotexN
ZXQkwzjcBH5pzzlFD9StWXD3gCyFUeqEee9SuFbuuHY3fLDsnpfBNUZfnxe7
m8LPuZqtrEIutOGrC2eTGF00G74svx8WK/NYgRxjZhtqz0pkZPnWM3cJ/Flt
m68Wci/ybct8/Quj5y4XVYzhIjv3oXF4WuXS08MizJ2uU8g8uGOeQ/EL9JWj
9Q0T21j4MRVFeR2WSLbWV6MPndeaTH0Xw2j7torwWfC1IyJ+KvrW8k258PVl
RmNlOTpq8EKj0QyG/u8aLVi1EJ4RXZc4CW53RsnDDfOSXLEroyGa0dUOC80x
5FX4yX6fNeYrP319vQQeLJiIGYKPauyK0cM8/qBo3/EJnrT98/BLeM0zydt/
4X+90spJxPd1hxfbPYLfUAwK6sJc90x76y/EO/eW8LSl8GZjqcwGeyCteYmu
lEPufZcv3YXP6fuobIB9wv/CMcARdQjLaQvNl/etZSg/Cj7DZHC+KerGouan
psIHFj0NHIUHPut8vSmB0Rv7DUkx8E1tIRmt2Ff+GjOOi68wWubqUxQBFwsj
Y6XwAuUzN02TGR393awlCF5d+tHLHHsypS5bUXad0f/t5/8Ar33paA==
      "], CompressedData["
1:eJx11Gk0nFcYB3BbTWNpjqAZ2ySilnFIrSGN5UaaE0sjbYc0SGyhZULGFnuc
hFiO/UZaBLU0E8tBJZYO2oOKNJaEEmZsIdYxhiiCNLR9fLqf+uH98jtz7nvf
//N/Rt2b9bWvmIiIiCU84vDMZrunfn6E065ilK6jtopRfnrZ0xVPjGw3jRMM
wP3bE4rXwV9ShLYvvDD6mf2Gpg7uUKfu5vIGo72YVIkWb4ycH09ayIHHJwxy
2sAfZLHe1V2F30tQXUXBKdO5ilprGK0nLqyU+2C04VCovEXjtNc6mnHUwPWk
slWqfDHSLYsaE4Ab5bq6KoJHa3XYVn+LEZtSlz8F3tcsuisDPjcUdaP2O4xq
Y60vvQRvatIrkgAPkm4prfPDCDvfaXgBPmNfbp0J9+FRtnLK/TE6k+6b0Q0u
EX7cmgre1GvxRQ4TI3HHH8V7wJenI5EJfK9ddWFP6DWMhufMGE/BLxpGfdO2
gtG8BHvYMgCjq0Xb1s/Ad72cgiaEGIV8vPRwF1xofK5y/5y4MNeFd8sYseLC
rOoC4Zy/OyuHwH9QD6ZWCTCq/oNJt7mOUYyFKoUHvpotJ969hNFvod1jz8CP
0y3ez4AXHfGsjOZjZL7YMqXPwkifRltaBS/RSj+gvwi52bQFx4LXn/pz4P1+
bgYRGtPzGO1Qp7MbwBePzXwqBfnLHVVyujcH82BxxHvBKyV5KUrgdpHaVnaz
xD26Gsbp4Bf4Ua//eU3OMY3cYp8Cn6zg32+cJu9tFcgzHcG/fGsoGzhF7inW
LKrjDR5So2Gu+Yp81y+DOlciwDnxwZGvJkgOSff0N5LBCwpHQiQnSG7OTIFK
MXilZI1GxxjJOTBxk/0IfL1jK1MwSuZyI/8tvQucNY8fK4DbSytjP5ijS2+A
7ih4GC3D0JqH0WZB46EUmHuXmmYGH1xt0imAx8XI4iDTtRB60khNCdgB3+hf
EQsBFx6k+7GhV3nnzRw+PMppPzJfkCfNJT0UZ6Sly4M/dLBflueS3lZ5mD+n
gTNO3rRU5ZKenw61ktUFX6kxyfqES/bCzD2o3wS8XziVoMcle9SsuxX+Gfjd
PupYGZfsnbC6uMQanJJbMkTlkT117pcy2vf6+JzBXR7Z61ujip5nwMXCZw+F
QD5F9a03qeA+QXmRp8FH7ujvOkHO5/kfVLh4YCTi9lf0SXDFWKdM9jhG25fz
zj5yx+hifKjyCfCCpDGvZJjXKqdNVQE8tdNNSw/cNtnr9sgk7POwcmneFYym
680GtMCvDfrPNEEfztpoe+iA7z2RXlAGf64vmiUBvZLjpvSMXoZe2LeLfASu
lDt++yvoIeP7uMpS8AuZ/j/9C3M5LG3sXjQDfbWfMk0DtzlMcVgDN1CIucWf
JX6s8/72FHgVK41vNE/OmffhyHaDLzuE5wsWyHvLBA98W8G1cwb6nyySe7Yp
iGmXg8ukJDjKLJHv0iymM++Ce+6sXWcISA5qjQdMY8F/5VtF5C+T3OjNjaVM
8L7Ept+DhCRnt70k+UvgFeknGOdWyFz+7//5P8D024g=
      "], CompressedData["
1:eJx11PtXjGkcAPAZsVGNrkwKSW2lQbHIocs3DYuMLLotXcely662zDYramm3
VUc9z+Ye09YqRGgbYSqpQdvZOtYoaWpQ6SZnSI2abmq/7x+wP7w/vJ/znOf9
vt+bdXjM9r1TWCyWGz46+KxzUExN3CqrMi27kd+ziILaMmjSqYnAyT3sslh0
57WV3hWOFKQL469kKgm8/Kvh1V70A7wul5M8Cqn7Cs4YtRCIyjFyCEb/Vl61
MxX9QPPAxhwVAcHPPW2+zD1au3uJ6AV8B773KwI+w/ViAbpLLudDHLo6vow1
8ZqAe4HIbxM6+2LTsTT8rk3shsrSNgK/nBgzWY9+8EPquxyMUxlq5XHkDYFI
icCLcTuFurfJnsKfRX4P3DsJpLy5EMd4s2pTr6EdBZ+dald2NwF2iGkf40Yl
T7ZstKXAFScM3e4hwI1YaO2Nznr8THLJmkJcWjEvsJfAooe7A7egW5WkaFRW
6FuSs7reEZj017b4oZ9Yma0NmEvBINdTHqQmUK+U+gehhx8c1pTMoRB0Qxhe
+Z6AQYZ+5h50vVXcDmMuBV3zm7LRDwTCX5g1/YB+/xbfKdOUwu4q499mfySw
oVHYx9RF+PnvNGMjCik7zx/X7Seg89F+x3H0RUkxnrEcCpPdme716Pu8yLSz
6HO1xtc1MzAP/m69MQME9GSGosvoFTpSHfEXFN5WJm3qQT9U3dh+Bz2ytTBv
bAqF+LvlKkcNgZVe0PaQcdswbQiLgkYynrMGfUBVdlGB3ur4lWv1ZwJjo2/n
2KKHSbZWd6A/sckLEIwS4KQoq5vx/s7C8CgNuonvS4M7WgLzOkdsQ9E7ygt/
nESPbilxGx8kkF0rlpVj/M+SxYVGPrKqxTncvr14b8SpBHY/5sE1w3HHfPQM
yVNeEZ4roObuI30EZHrbZYvRo+/2BTDvEpmhswrzucA5QeiCflqhbl6InjTe
wc3C/JcXya+vQ9edaWEtwHO5NvfiXbBeYR3dyVvRFbZmC37Cc59GOBkyrG9F
6tVlfuhXXrh2H0MXjC61Woz9kNMQfSQY/UJXe1w6eubnaQ5HsX90PaI0+9A9
N9e5nEOPdX68pK4L5+J7nlk0ujJ0W/pR/G7NmsvFxtifDYHa0Bj0N07H5kRh
nG3yoOU+2M/OZXnAuF+92GI5/u9EgUiajv1/2sfbinH1w0jLUczj9pHeZbU4
L7WcidIDTDxeFjXyTwQarXNXzML5ejTAzYhEj5GrRPsx/9dWD6yNw3lck+Qx
KmTy4JV0yQDrdfhwVmpHM4E764OHgtDPCh94Fo8TMDL/Zlc0zrv78pB4f/Rs
4sv3nSSgX9v4SZ/ZD5ZZ67ahvxu2FA6zKZy7lXZe0UggJHkJh49+t5e19tA0
Ct2rTvwqfY79P714aBX6vWynweHpFDrjTyfKGwiIdPQyl6DfbM3bnKVPwYJf
JR+qJ/Bd4ZkWpu5T/vlXM9uQQvgsa9NA9FK2dI8hOl/Mu3bKhAJPUFPU/oyA
U9sp9QTTb1fHzAxnUwh+VGn5BzpLk9LZj14nTa+5bk7hvsLEn6BXJ+wubEUv
HXQbtMH5tbtV9SoffWQ+W8n0edjJll0V8ymIVhwU9aNLyl87ydFnuOcX++J+
KE6sa4rAeAK+Pm7JzFGafWfCTBsKz2/u38jB/5q3za0xj9mrtwfv539JIcJj
9VQl+pAMbJg5pb8PxO3CfRXgbv/kKeYn13npbWauHz1NX2GM+82AY9X+HvP5
f/v5P8RK60U=
      "], CompressedData["
1:eJx11Ps3lHkcB3CbbluhsCUHhdjUxnbRReRrURtyaLq5hJ1UdmTGMMOMQe7r
up7voRqXojMspdtunGolapU00mW0uTSFNV2ZTSnFlH33B+wPzy+v85znfL/v
z/vzmLN5W/dO0tLScsKjjUfY5juUZXexqaXRyOJoJCVxA65kSyRDQh0nP0uG
S7MGjdV8SvwsyJEKHkM2Xf1nkwD+4cl8r9PRlGhZGObOgFcGuvVw4JycQddP
AkpO7Fx+LJXLkJkvXFoD4EFlGk59DCUO8nncWfDpwU4dnvB+aataJKKk77B+
yckIhgTMu7uawA1meLNaxHjf/GX6Ljj/qVemLXx785SiVAkl4gKfLQvhaXZR
xiZwm7C6bc4JlHxdv11/7ABDbreY/jEd3q7hBIwlUvJKQASv4LptCdkjtheb
tByaH9YmUVK44Y5CDR+2P75xAL62ZnbvRDIlTTEKOx18n3Vmm5sCXh3jE+SR
SgnfQpTlBD+x8fmiq/CbtW06hWmUdJRkPsmEy7jxnDNwd2W5lzKdkjuDxxT9
cCbY7loxXKgpe2z1CyWl4io3FnLQZa9NyYEvELkETsCVsytY9+H+40EiCfzh
BpNNnZmUlAv/PRiKnL+vUMp58HYb1Tc/ZFGiWjL881TMS2KryQ2Ca0zbFhhn
UxIb+OpeHXxa9RrFDvhf753TDeErnn08wucz5FGqdvtmeJzXWr8EeGh+t8mq
KIaU22SIXeA/zvT0joX3vtid9Bluf2n9Ryc4d1dN4iR4qeUo+3Y0Q6pUbssd
4NG1IZI8nOdslPdkmYAhnvxF0tXwsxkhxUbwzLoJK4GQIYbLDN+vgeu8vK5r
j3sFL3bcvzGGIQ9IUso6+POi5Epf5PDYunOxYSxDfP1ttq+HL+Ne53EzKDkp
tV3aA28xzj/65TxmFSP5Och5cMmI+rCIIdU02cEd3m1q6uGAuZi5n1N4iBmy
5ly691Z418KiC3EplPgnq8Vv4QV+BlMC4PtaA52XYO5hjhx2bhxDymrtRaHw
2288fboPUsIyy9piIEHP+8fSouHG44eWmaBXeY/vFufAM+YaJaTCV/bdrAuK
p2T/0ozwIXiVTfD8QrintlPd8ThKyhQVic7x6G3nleZK+Jyq2WIV+p/fIK+L
hZePh647Dz9t1c6XxuI7HcN3SuE+1vdzb8AtS/aEKYWU5FQI807Bex7p1nbD
K1mvRy2wd2aCpbrVcNmb3X0vv8yrtfHZwyjsKfvm/Wz4LbFDrxb2Qv+0r9wM
e532a4G/H1xPqlHpw9+Z8fQe8CgZj3e01oNn6J0vXAgP+BAcOcal5Pkn31V1
uJf6nuN3q+BX3wY/+D2CEl5M13U3uE7ghghX+M7p6Ts7D1Ai7P2zqBl5Jrnm
hbPgr4dkP02EI/99bpEr4B7v47RD4ceaR7qs4Q3ygWqKuXyeqnKPhJczksgG
DiXmlfWWfZjvzOKG0Vj40Dq98WvwxsP9B63hN8yNtdPhhzyemPvDI8yslGz0
RC3y8c/78h+bxbYMge/1tpJI0Tftow1dDNxUvudpBzxnZPelW+ingCmtKYSf
CZ+r2ozzhK2PMnqHPofM2uEnhR+ZZHDrCnxUc/lvC7iyRDZWAr94gfvbSty3
p6ig1hd7kTgwJ7MY7lM9On8e8pHfbb+Xgj2S2bVdKoAb1gcNhyHPU0lJQ5ex
j+yaRgmFyxRlCk/k3zQt+9uv4P/3f/4PeTHOaw==
      "], CompressedData["
1:eJx11Hk01WkYB3DLiLI0Gue0oJJEXFKYnLI8zY00dJhKuZpIJZwz0ZUZS6NQ
ozTy9kP2JelaJ9tVbgu5lshIcdLYKmFKCy0qjcJ8/TV/zR/vP5/znvc87/d9
nldnb8BWbzkZGRlrLHksF+sLhVbmklr3DzWf+TWM/j4q7yCO4+hk26LsFXAD
s1me7bWM2iWuqoGMI+2OkSUacFXbhZ+O1TEaTnjB33SWo+xXD7Jl4a2+yXEm
DYwGwnhhWhxHnrVflo6aSWqTD27o7G1kFKsap6MUz1FDMG9ZD1ws9Qg81cSI
87aRnYZHsHKTFviLpcOjO5oZ7XExWSibyFFBw+F11+DLDNP99G7jnPiLpvLn
UM+Cw3ZF8KctwQNjcC29nE0KSRxJT5c6p8Fj34QL6loYzZdd7qGYzNEu98dD
cXCVrM8CMTxyiXHQ7BR4kU/ocbgfP1NwEe4VeiNGOZWjnxfoJ4fDb2ruXnMV
5+ekio3H4X3OayuD4U0PJua0oc57RnbtA2kcGQum2gLhFdtOrx7CvUpXestL
0jlKi/b19Yf/EPPN9MQtRosUyr5EZnAk3FxxOgD+u4JmvgXyWTcoPmGRydGl
68VFQnjWCn6msJ6RpD86vQOel3ioOWgmB7eQJyVSRk92Wm0QZHEUqtI7+it8
K0tym3uT0d4+95L78PjetUpR8A1m/pbO1YxqhHKBJtkcfbzgZnAG7lhulSK+
xuhNGT/2ALxv7Tu7VPg55daNCyWMmuzSLU7B90woe+XB9d+2W0dcZvTHDlX/
BLgovDC0Ep6hMR01XPHffpdov7iGmZwPe3ltK2M0emS3pgtcynPV6YQrpfw4
VHOJked2UbMK/PqqKdNncKdfHrvxihk5xUS25qP+iBYFmwn42NS9+ekFjOz6
3Tx58Ie2/dLZ6LePbfJpw3mM9ulqRaYhnwa9xtuL4Vfe248ILjLKF1xd/Q45
33SvbzOHC+8Ux7TmMPJv0hSawUOuKLbaw3sV3/O1zzMSD4mD/fBezu5/3hLA
y+3nOQRkMZocGVaJwfsK9kltD8J37Y99Lc1gZFBs6peJfqgarKyKgKsN8hI1
0hk59M1rLEdfbc1ZXXAWvj1nYNoujdHUCkFpKfqw4tDL5PPwfpd6v+BURsml
eqmX0LdjLgPRZXDrr+62F6QwUg0PiSpGn6//qzDgBvwMd2i3BN5tnLcmC3Nh
9KS6qwnesNnruyb4PWcZzYwEjqIdzOs64DEV1RGd8L3NSetFmK/hMR3eQ7i2
Rs6CQXhBS0jiVcxpldrLrwfh/C0dpm2ox8f/w6s7mOseNRp/Ble8zyarUL/u
eOKVT/gHlHz0u2f80U8NoqPIIbm8UGICd91ZtuEl3Cw3IfJAJuarR5l3BP+J
Vt450XP4+OV/LI2yGTmGynV1nOGId2Jj7FN43RGP6RHkr2ZoZeIEl94tGZ2p
c9Yx/c+xuXgv0eSH3liOhupzu7rgYWs8DLeIGKmLt6+KgmdodVu3wcWOS+Oe
5zMqcnPqsYEHBiXq1cELlh2bPl7EaNa8xlXacP4Fkdpl+OKS17ra6MONPa8+
qsOr9juG5c3kHGLqWlnKSH5upq8BfGWxw7OkmXtFSS2/R59bVU04usJ9mu4+
+A2uvtzC4FElo9Zbnd25cEPb/fZB8Oou6wBhFSPht6HnJ+G65t6WXnC9CRup
HOZu0dtI9VDct8T4pNEWuK/vw/r4G4xydyq/n4Pc/u9//hd1FOOI
      "], CompressedData["
1:eJx11PtXzHkYB/BkVahZ3UZFdNN0GTpEx1GTJyrXEptSQpqiTFOiaZVuKq0v
Z8N+iFEtEkkpJduVGbJ2WWupcGJFF+0iRTek2Pf8AfvD55fX+Z7veT7P53ne
5iHRa8LU1dTURDhjcZYcas4Idq5WikWbaj9mMBLWSc/vN+FI6pRusR7+JN4k
dADeF0s1nVM46pvQlLoW/vbh7Se98FNm4056m3LkE35up8rfvfDamQg/XirL
qJ7GkVrAkJvKzdae7NGCj6svl9mbceRg6nkxEL7pnBazTme0Tj3LMteco0b/
A+kqn311heHlNEavu/UeaFtytEmYXBUG/7SkWCNkDyNFgUl3tBVHNov5vBi4
pvF+A91URo66tcO3Z3D0rm2LPBF+tjfSS5nMSId1x/EFHJ3WPJq0D57eE3Qv
OonR882FvittOPq7TsPvBHyPfKfi0m5GPQf6fSJsOTK1v69RCD/o8uVlcAKj
1UVPbCR2HJUmbXNRwEd+cM913MXoxLB0uq89RwMa8qhGeFDg6wJZHKNWaeFL
IyHuy7Mz7VJ9L2rTrY5l1DA3Z8w1+JfiBYHDcNf87PfDOxglB3i9dZ3JUfik
D/kGLtVKk23TfP/azsjw/N1Zp+DKyF94dvCOMiuBfjQjJ+LNbYaLHHXGE7zs
fXakv5SRd6v5+E64ZoBmjh+8qP2Ifa6EkfMf4yR/wqs6C25K4bYRI+K2CEYd
66oDDsG3v21J3AvvG47kCcIZJWjPN3KAZyVcuJsHd+sdnS3dwiiuJUFciPpd
UnxsyuE2g8oKizBGA8Vxzp/Qh4XWtZEK+Fe/ytgloYzOP5hZMhNuYr6gvAnO
17YoGw1htLX+yBV39DNltfa/7fD4pNA5lnDhRbmBC/r/OE/O+uAbPl7UW7aZ
kal0tjYf71VxasrrEfig96B6azCjGMeYmHZrjtLkmsbqomql8asmQTs8/7Dj
4FHMQ0DVei0d+DwjtZ++h2cf3tfljPmZSholevCNeiYGiXAv12D1RguO9hg8
vKnyW5/f7O6FP6rtMA1XzWf9ojH68OVlUWpi1HPVul84OJ2j6G1Li1T/v5EV
K/ZA/d+UViRnYP4rr8p8tOD5FxbcEojxLp0m9/WxL1qPq5WqOvvEKaPj0Z9M
j7DLl7Bfns9Wp37AvVY42Y8Gop81I5n5ftjHb0eitbrhFe5aD4rR/9w5OZYT
jDlq6DGc9wI+UOMSNLKV0a8FjxY1T+bot1a37ffg87O8E+vwvtcWrllcxedI
qHAQ1cNTCyrdEzAPXXGC8CuGHNXwewqK4HWT5PXzMT+PzvhH3TXg6NWxfQnZ
8IyhFsVQFKPSicKzmnCLXJ37aXC/4sagSsznLL48M0SfoyCnZ2Mj4KFuvVtf
YJ77024cb9PjKESjxdMXXumw3DYP82/lf2xtKjzmzfMmV1U9vyvWBWBfdLsk
q1bC+bISkSXc8eOtgXTs127JpAQXeHOhb5423N5KNGch9nEMjxl9B5dc54UM
Yo9kdwz7P2F/Q8377A7BD/bmCDrg3cqhDR7Y90ifqVHv4Z5fuhV34NfzGoVf
kQ+fYyfPiEf9FYvbfKrgO2o//LwMedKZ0bJKgPvKLIokZ1S5YXapnCF/MgNm
tPfBDet2FRxR5VJX894o5NV+ae/Tp+hnQ/fpjXvhPdNq+pci33hv5qV2ov/u
//DPyOC5gnR1S+Th5IZnByYYcdSTUuIhUeVeOdPIgsdP/NHWFe/7f/n8H5k7
sMc=
      "], CompressedData["
1:eJx11Gs0lWkUB/DTuGSaGk2iZcbE0YVCh4ohYaOMMy6nIaWJ0I0UJocoXdxW
STqPN120GClhXJuQyO2wQjqlobBWpUFmzKlVkgil5u+TT/Ph/fJbz3re/e73
vzd/e4jbri94PJ4VHgU8YnOLFBOzCml0lXlvbBoj0bjxHB0xR+k/Wn5cAk8d
K9FcmMFoe8NtveFQjpyl7r7qcP+YAPPzlxklrroe1QYPLo1zVoHvEs+PFlxl
5Ne/0a4JPquzv2zEtEI6OCtrZnbu9Pn5AiPtPvg5p2I7zYLp+w22yee1w43i
vXKLixi9DjvlYIB65g6FxFTDn7Y1OupcZ3R//6eug3BbXeFYPny0J6CytJRR
skvNlYfwsqDi4QvwWVV88/XljBz+2qspCuOo6+WtkHj4B0vZ150VjHJ/29DS
AVeYvXPfr1P1lBtZ7a5ipCzaUeIfztGGDOsRH/jH6i2pC2oYtYT+M2fuAY42
NnseEcE1boiH7tQykgvM9lTATxzbrGINz9t7eDJSykgy9NgsIoKjFLseLRO4
tyeLc6pn5PO7laJZJEcTR4+tXgQvMDev0Wpg9FDYaf/VQY4UVR+0qMMZfZv2
Gq78rF/YATfxGwycDU/ac1XUB1dsfpqQf4ij/QMDJxXgserOeZ1wwTfukrAo
jvzPLls+sbpC2rOwxOsu/N7K/HvCwxwJJA3tI3DHstpVvajn/uKENrUjHKnW
GI0PwZdmHP1jDG8/bz2e3g7PkhnkvYXrzr9fsLKOUaypfgB3lKOzj9bLBuF2
7/lhTuiDVdQzqcUxjnq2mPJG4UUTyRZ56Ke3jZ9mH3zAIyB3DC7UfyP7u5LR
9abjn8XRHF19u86Dh/r59pqx628ymvAVF07A6+tqP6lM9SdoYVxOGaP9W7UH
AmM4Ugt4kTUPHrmluVWlhJGROPXabfitG15NWvB4N8e6vdcYcX6j1mqxHD1Q
CnxhCJeYfKk1jLy5KOotInidsa6iBTx9s22mZx6jNA1hqx88LcQg8yf4rmUS
eW82o/qP7jOD4JIVFj94w9WCFKP2XZk+b8fvtN8Hf67Mf9xxafr+SsuAtmh4
oo7Hh+3pjIQ5ciUVuIbh1g4Obvsh60XrRUbB3yceqET9yzZc2pQ11Qfj8sOW
FxhlGqsW+8B7RWMdt+DFwTdD76Uw0l/ruqYf/bFLcnZthbvHFZ6x5Bi90/U6
7Q2vMSmQdsOPnw57lsCQ81zjXBn672qjVvIWvtPBa2A8iZGHUq7rIviJvova
Cpjf4FeGkY6nGOX3ykcO4f8qbapSnguPVB16Ep7IqFq3dHk18tDwZ1EIf2pv
jFmXNCYwsrdd4zOJXJnNDHITwDUkel7vTmB+8/X5+vCMepfytfCYCwa+uvBR
m3ffbUM+n59V1nCAxzXMOLcCfveJbXcC8nzwocp7V/h4TNfiNfAinfCuWuR/
aKTPwxOuWnLK0wGeOslPlWOOPq/7VOo15YHt4QLUE8mr4bThM5S9U3zh3b7F
rxecZHQo3u22L+ZRJmoe9IGPRi8e7cZ3TUxmj17G/GYk8UTb4DmmL8uuoD+0
W5Q2jj2g7hRcNHX/o+hasb+EkV5Ao83P8H8j2r/+Bd7CZPLlydg/TYWb6rBP
QpfcDXKDv1qaLJSdYaR+uiHHfGrPRNg+EMLf7Bj2CTzHqNB695472FeZ6i6r
CC5srDZUSWXkyrIzguH/t5//A0MYzL0=
      "], CompressedData["
1:eJx11Pk7FVgYB/Cb28SYJ5R2tIxkSpIiN0ovbZbqeahIjcxc65AIlXUs3bTg
nlEdMm6NEEJXRaEm2bIl1YgwqTQpY0wLYuoZzNcfMD+cXz7Pec7znnPe77tA
7GfvriAQCNZiCbFkLlum1I4Wl8frdSdI6zidVOwQ1f3AaLRA2FsEF8q6Btzq
ORkH1v3V580ofUOKRA6XaH2j4wKfVpeTabyP0Y6DU+ZcgDet/9qzBl5arP0q
y5eRRl+UawpcLXuhxx34tpI63ZV+jJSaNA0T4SH+NwaL4ZqBmgZt/oxK7do/
SuFRqa8iLOGmq7RLTgYw8n0Ybh4Pz1tZmtSIOlc/z5/oEMSoo1PofwJuVN/z
swP8XavVI51DjIJjHeTj+3s6zeUBtZwqmy+fHz3M6FVSVhmDezZKzkprOFn7
PNr3IoSRrlat7mn47NkzHufe5XRLzdHkShgjlZ7W5zK4W07BHqdqTgvXHI4K
i2DklG9UlQUvm3DqK6UqTne6o2vXRjJa4exseRU+drTIuriCU8ZPY8dfRDFy
Xewn/BXevzmm2qMc52S01R6MYdS3uEV+Hz596j+zLpVxumSVLXhzhNFZrS/L
2uFJseJqp9ucNojNikRHca+e79L64FuFgUbnbnGKei1/6R3LaEGNo4NgrLi8
WCp07yrl1KriZRNwjFHag/XXVeBDdz2WnCnhVDB9ad6W44zODwuMF8Lb9Pjn
9hucyifdrh6GG5cFbTeCa+k375Re56Q8q2Nm8AlGsnUlsbbw3qbKKy2FOH+u
mds9eK7kQ6IzvHaeRkPcNU7potM5PfDAZosIfzh7qi63vMpJIXzN6za4NFsU
L4Gr6/GUzwWchDorU6VwH7uli07DO2wPjcyHP686oDwXTkuklenwtJKLNlZy
TpI7PVdDUKfp5kabQnihecY+/8ucPMOWT83CfVUj7HOq4QLF99tS8jmNZDzL
y8b7iOzVTz2EW+RX5S6Hn205NhaD99xh1WTaOX6OdvO1ujxO8ptqk0USRjMs
lib3wge9W7/9Hi62LpxWj/9yG3TY2w9PupQv9oLfClJINo1mNGHSkOMgPFqv
O7ETHhmY0ZKMfrigH9r5L9xfUrm+G95b+VTtD/RPTL4BH4M3OGjM24N6IszX
Kc0JZ5QUv3/T+H7HAesP5rhX/QXJyOZQRrs799iO739zbbX6dbhigMDQL5hR
uW6n8hDc/ffl7yfh3Y58yE1n6H8V179Lxus0OccqeuFS1yf8NvLSIND64k94
/ZRVojX4F/H5xycGka+i9nXibri3bKBiHv4xNcXWfMUBRjPHLKsew2VzDVyf
4d+H02eYJCC/G7v8lBrg2rLXb5ehT6oMJ/YPI+8FT5wSyuCZYez+j+irtAcT
B45jPnwMUrPOh+/Mbhg0Qx8O6eSJTHwYRWRrZP8Crwk0aziJvjXbqFqlgDmj
FRHbFQdvqdibtAl9PrlsPx/xYiTWqdhwGO57alfaBOQibkDTQBseGew34glv
0l88dRty5LYx6eFBT0ZNF10S7eAKSs4hisidIHH7zWEPRkcuDsYS/NPbUXt7
5NTCKtknE175zn3/MngCazeRVXKqi/dadAw+sKo6UgNOoSm7u5F390+NOWfg
88uXOSjC54RyF33MjSK+a1UX3CxQO7gfOd01ObM8CHPGV2zWYId6MpXDH72E
P0hQ9d2KuZQabag7BJ95T9XmN7j1M6OtOphjbvrDcbW41//N5/8AR/cQOw==

      "], CompressedData["
1:eJx11Ps31HkYB/Bxm8VK5RCri0ipjuhmWy3ryS0ll6MytpKubm1JQqxk61uq
7ySpPVKN2MpuCiuEcqstHLvGZXLPTGZQKJTBuqR9+wP2h+8vr/M5M8/n+byf
x3BfkMdBRQ6HY41PCV9aaVH0vPMFZWtNBP65ynwyH1g/fkHE0GOO1EEbvtf7
O614FT4Z5QdbCV8x5Gjqwv8aXnYzdOkhLp/cnEvVLRsZGl8UKFCEPzANUXD8
ik8BR8wLcpoYilhT6/I5tqCM94E7aAMfcE+VOLcwpDsnvHYUXjq5Pd0SvvZS
SurHVob2Rjp+OwJXbY+r9MDvh1atCE15zZDzrSu+cvhmzZnph1BPdtgjEx8x
Q0XSU7mf4BMHGjIY1O+k3VWh/YahrrB42SBcK748TqDEp41DiRGiDoY4S8yc
huCFaVOrGxX4FDgjwfm0jCHjqovPh+HcnNZBTQ6f2gzK3JZ3MRSjdjZ4Er6k
836w32eWXnQ4Hn/ezVBvaU6gEu7b1u3gVDbOkoZRVZ/dO9RjtSuTC6+RBPWP
/cvSk2re48wehhINmndowjOP+V/dOsLSul1Gz8d6GQo9aRutD7fpTdxY/Yml
yCGXcwvfM/SaopQWwf3/ONhlMsiSZ6pL7bIPDP0cv13VFG5wwUbnzAeWVNR4
cRr9DEXt9k+1hEeuigmd6mGp8q7u0DP44HBDvR18qPjq5I63LHGTbk1tGmDI
a2Q4xgOuEqqQ1CRjiR/oFv0AviB9mcQbHrUlI8yigyWFleFNzfCqeyl5AfCZ
eXrtV8UsBdU493TATxRqbgiDy26rzx5qY0nb1jg+F744UXf/menzWbUDHi0s
dbfXtf8If+SXO/cKXB6yKiKnkSWjWa71dagz2Mvi6E345Tau6fArlkYn4w7P
gx9eNsshDW6VoHN0nYilhX49UYQ+SHamJGTDn6omWkXWseSuXm5P6NvZrIA9
xfCx6OGk4hqWXI7cjvmmj6H+NpHgJbzCtb6/Q4h3ZCo/1+Nd7KV624Tw6jDP
f7jw8ZYNa/zxjictC6xF8OJdSd4acMuCe9UDePdCQ+UrzfDFsmRfLfhS4dL7
POTE1r5bTwx/Ymj9QA9eLle+l4Vcmfhp5Mvg8St01QzgTVY6Io6UIV96d7kL
vufGSmMu6tQ3PGxoi9zadze5TZ9n7fYmV9eyNDTD4Rc+cq5i5GUyfX6OxZeC
XNx35aK/hdWYC+VAqVAKz4ud0ItCfwrtnSYWtDGUkfE97w3cTezradfA0kWz
rZYRmLv8nT88boFXyYI6eptYOje/b3YH5nRHwG9/NcKvj83+NRvvtVxtRqIP
5tr3kvx8DdzjTkaI12vkTVJeKcceEBdl91bAxwWrpFPIQ4Pr7eO/Y28krymp
KoX3GzqW3EV+uluSH8bUMxT+wndzDlzsr5++vIslWUT77vA6hhTF+g7p8NFr
emdqkE9Jyz7BtVqGPNXE92/BHa2FZtv6UOfHQgtJDUNzOmceT4DnNJ3IHEX+
r4sGTHjwfH5nUSxc+ePqmzcwL60B8WnDQuS2WcPrBNzc28RmvpylzlbBlkq4
Oy/56E/wfQeu15VgHhUrDE49g49cu9PqAz+izBX4jGGOTmm19MMdNxc8nJ4X
t40KT79MsPTloZy/Cf8reRb01gmeVumeVYL90Pqy9EYdfMOn3gvr4SE66495
Y58IJwMUY3GvxpSnOebTOfyTZzZXkU+q6u+z96APeUFG243h5Tw712bsq1iJ
xer96Nv/7ef/AKS+Csg=
      "], CompressedData["
1:eJx11Gs0FGgYB/BZQ47L2trpcoZkpDTYyk6uE9NryiXORkqXUSkWzVKa7CKV
sGyR28uEcumKOJG11p1QNik6m9MWG5WRiiL3wrT7n0/7aT+8X34f3svz/J9X
3zvI3VeJwWDYYjGx6ueXht7gVDWWyIKWxDhR0rzP3+HIa0ryV84F58KZ0xmz
Q/DE/lAD/htKsts3RZyF32JHCJs3UzIdZcNQfkuJ7h3RxxPw5I2RrUbOlETa
5LC64HbGsX1H4X0TW5OWw6+UJ+ytHqRk3XRItw88rCWmIg6uL15WcH6Ikv0Z
p4S74ftdZMui4d8t+GZV7DtKlKaMTVzhj7qOysLhs6fkQb7vsR9bU8UZPqx/
zF4Frq4rqbIfpuRJfayXPbyyhKmcgnsO82vvaY9QcvqDuJoo7lNzslkb3n2t
tHsEPqP+T5oNfFYzpicP7/1jT1J59QdK3heVHhDAVUIaXW87UsLrvMw5NEpJ
T1bCGsU+0uC5phcOlDw1PZbAGaOEYSSQC+Ff9/zw/rM9JRe2X4uvhzPjtRpc
4KmuD4JubqJEZ8pCZ9M4Jasvhae4wS0KA798uJGSljk7QTl8XSCvUwR3ndTM
ixRSwr0cY8yeoEQa/+MSPzj35t5Jnh0lhZ88Cv3hT5zLexV19jNpEg1soCQh
bUGMFJ7z3NwwEi6jaksvCCiJfqjDrYCPy1o/JMEPtmmNbbH9z5fTvPWKvhtp
VRoo22D/6UOjRfA17efUbsKvJFaLa/iUvBSUL46A116xfFcLb8puvP/cmpKs
nPQOPnyez4P6DnhX/raoDCtK9C5qsl/iXdt5Yyl98ENu64ddLSkJlqtPHYEz
w8nHUXheoZ1OqAXu3cvPnkTdXlWUdajqVzUKfs+cMjWnpMCYk7sL/rLttjUb
PsXwSd1tRsmuy2xxKfoSoOGz0gTe/kgYF72OkpHflsrn0MdQk+P5VnDzsut2
2nBzzjKRAM5Kv2TkCJ9hFv5axkOO3zluFiMP2/db3fCAZz16y3GBZyrpWkmR
q6w44Zrv4RkrApJl31KS5p9jeAs55E4cvntYce5Iy9IhePNVdkEncuu29vy9
k3BnJZWBn+EsG41948j5g94Xk9Hwc2pOEZ/g29QvLmTBV/vv0UyEi3SiSxg4
l8163MbDvJi1Rlmmwv04sQ4h8BlRcaQ75mtQ3u+VDtcaCzccgkdsnHftDOY0
xzMm+xxc2pt+uAPv9RCy7jZgrg/olQjPw686nR0Tom7Oufc5DHj3ifoZxT6Z
u2oMV6LOPJbFm50DlEgy1eyT4ZOrPvv9hL587hcP1LyiRPn4gbCzinPrFp3e
gT56t0rczeAbwsJsFe+KT58qy0PfJb6ek1X9yGOWJOAEnGla7x2LnFS5W9t6
wVsvFBlL4OI+mW4ncmVp9VXdfHiN6O8BRZ0HOwM4HOQwR2/w+lMZJabtBno7
4cXPFmsFIrdn8rr4zfCoHu/HTvCD0pHcSuQ8V+tPZiV8VHpH1RrOXZ+srUQo
iWP0Sdrhw748Sy78yXj/MxfM0Y5FX2yZgW/JzPdZDE9ta3olxdy9LeFm2uI+
x4qWJDLho6wJsx7MqczLzS0Dfqal4OoEctsg1/CtwFwfHwoMUUUdCgv4Ti/g
bcGz2XL8A3sdJZ3R8LWl4u6HcM8QTsVj/BvOr7fmLkSdD6oNixvgKzhxi4rx
zzj8MvdXHfz//ud/Ac5U08Q=
      "], CompressedData["
1:eJx11PtbzHkUB/Bmu8wIW+g6Uyqp9ZTMurTkepRKNaWnhnKbroOVcUm6sYUn
CZ/50GCzm/ZRkrTEKrrSVGq6TpipoWwX7ErpYmTTuu2ZP2B/+P7yer7P53s+
73PO1yZ8T4DwGy0trRX4aOMj9vhkkbm1VOqXVX4nZT6F5Um1Q3ekBAJfLnU4
hc7aoj7tspBCqrm/Pq+GAOvSux3J6Gf62nkGiyiEtVkO9tcSaNGPNt6PXlLz
MvAlumVjbNiVOgKttswZUehlbs+Fneg611eowmQEGB0TyQJ0VY275U1nCl2M
rnbHRgJWv5qt3IBuF7e2Nxd9ut2fivdNBPq8xlwD0LNHnudU4Dk9w4rpNS1Y
59eJtd7oHwyCqv5Av9VdvOaQnIBt5l/FHuirJL9804j1ly7sV89/SKAwtE/i
prnvzYeK3gUU0gdi2+SPCBwx1dVyR68s0tdiYw41xi6zBAoCBbUnLD3RBRkd
TD6XQv7IfLZMScBMlZrjhX4sruoMdaKgTAs8at5BoLeorpSHrhckkjc6UvAM
OcLnqQjYC1LO+aOPS0UMPQcKXwRsL/4TAtK+yr0b0ZfoV1lEfkdBt433wvkp
gdqhKIsI9JDTl1xqZ1Mw4YoFA+genJwMETp39fMg21kUsi8Ole/rJLDmVuPZ
GHSnvFNREdYUoPDgh2b0dcIzhYfR+bCR1FlSWLAfTCfQH6/yKhWjs2US5RwO
hdfMywFq9ILPRhc185Cyw8ZBbEYhPOauUyV6uK+hewG6JFEdrTbG79qfcNyM
Xrg+4lUZeu5vdiPBMyisCxToPMA6DSJTuE3oSZ38xVJDCtLR+AcsdK2Ez/92
oPu8i41lGlA4+jpLaI052LxXvelH30O+b9o7hYJsj0MeC3PLdp1sMI5+23BX
YZk+hZOLhhTt7QRMRlzf6ApKpYlJW+K1J1HIXJ8mTMS+cIqIvTG6G2dGsS8T
+xupyNfHPl6cwq+3Rvcmp2vi9CgcHvRLj8G+p2d1pcxFl8YK/8nWpTCcmVus
asNzDt1XLkEv+lj1rEWHwhKrvNUcnKvH6azZ7uhfpzZUj2tTqAw1lgtwDq+l
5m7z1fitfcMMfN8sQ/zwHM7toPm9/PXoY8kJ7Cnoy87ufNzUQEAZUjG2FZ3O
ux1ggu6jx5Z8qSfQecolJwx90nmR9Vf04Yy6rWtwjy4PeC8Voms/CVa+wjq7
FrCuU9y7bl5AyA70aH+zj214r4y3M5vfVuM+pkS6R2nqv9qqLMEc+ut/2B+O
e33IqGXej+jdF5Ie/YS5eYuaD0zcxxzyhTe3oRuVvvF1m4x9DO5hXr1HYNPa
37Mi0JsPtJiwplLgv+jmxlQS2OKTkLAZPdQuvr8A+/hMT+m8q4JA4qLBkQ2a
/JNCX4imUUg+lv/zyXICcSEsL00+rY3nn35rRGF3zGLTjjICy3qsHrmi8+t2
ygtNKBR1S0r80PfFr0xbjF6fVVDta06BzWuIGyslcNxCtNsRvaGZUzSIc3uh
ujdIhn6+8ljkTHSr1ImDaTPxnI/XLOToo1X+yYboATFXnWfbUOg+p3bVwfOX
JksUDHTdjeKJ+7YUVJIUTyG6+MM6JzXO22qG+fFge+yLj1/0J/TcigTu3+i8
Ddd40+dQkDCP3C3Be416mk9rR/dYLmQW4F635uxkX8IcGJwb22Xos1x8y8Pm
4t5lnewpwdzkeezRYnTbT1dk4/MojJYfLBnAnLnVpgLNfr3fvumLHP8zTUk3
HD2qsJ7/+T//Bz02178=
      "], CompressedData["
1:eJx11Gs01GkcB/BBydpK7VRT9jTd1rIr1SDGSuc5JwnlzrjOMHKJSERh0zYO
XdjtP075q2Vt2x7kshty6KKyriWRNtdUZDByLdSU2369+r/aF/83nxfP8/y/
v+/zbPINc/RXZrFYZvhU8GVpDqtvnCir0LpQ63dZTBNuqq2K/wRFZqL0hlfA
hZXLn83CL3EdudwpiogvW/2uBDeK72Ef9KXJN3wzk+4PFFnCkicp3pdVRDWJ
0zzgMU2LnfMVFOG3BknG4PGBGlqO8NC5eb3IzxQp/iX4RD/8vr15sRU8qNpm
idkMRbT/yZnqhXMtEhr+xb6Z31/cMD+L9ctjU3vgL2QnVUVw98K243fmKSI/
U5X9Gr4juylP7kOTkWrd7BssKfnTUildBh+tKuOpwovGVAujlKTEvM6KWlh/
40xPXoo3TXw0ta7wlKXkmUlV3uDC+QsuWhSKaBL5cou4E64u+/rBB/hqcWuH
WEiT9RqW5yJVpCQo33r7HDzQvqpE4kUTc07QhWm4QU1SzVLkM/s8tsjQkyaS
uovHwhZJSa1tbudq+EsXnTUCD5pcifqrrQEudDLc+i2cfT+4SN2dJtpzDZ2r
FkvJpG9oiDE8zihXt8KVJimePbv2wru4xsVW8AEb9q/HBTThZLfd84QP+Tq/
8IJz46guPRfGf3KsGzgKH1z16X2fE7NObE/zp0R4lwZnNtOR2fd8R+rndHjY
V0f63ByYc8pU74UXwGWvCrZK7Jn/mnAf1quGm5r4Fu2yY3JwFWlZt8LjO9ee
DrBlcouYq9ceXOibVbxyig2T8w1adHcGXsBp0Ss/QJP+G9P8bfBG748jKybL
KiJScocH9tMkI7rLLgJzZBW38DbAC01TCyTw0vx3WvmYe+K9x6aG8ERJi/7f
1jRp0T9VehM9aW+67rsHvmiZaY0lvMbA2FuBXtGcE/sPwOeTr2g4wO9aj1kY
oYdvoqddXOA52YIYd3iAIjM0FL1V+tnvsAiecNdoRAz3Vpc4XUPPy2f7IgPh
1r3KgmB4zpq26A7ci4Dbr6eD4CpTbH43/GnsoJCDe9R7W7UiAl6/R+Ijh3fX
PlD2wr1r/1FgcQSeYWe+7zH+a0dU4qHb7yjyoDStJgw+UdJuSJDPWOUXGrrj
FHFrngsPhks4zQ6bkadnxhmjm6MU4W1f6uUHL+0X31FB/ulD3KviEYo8eTWT
JYSHtLtekMEVjnHJ24cpEqG02UQAZ+0+fasac2wcV4vZNESR6HPjN/fB/c7W
Nq9EH/a2dBwyfksRw7S03Xz40cMPvxShP8sGG9NPDlLk8ZPzMm14Up9aeC76
luDW0jwqp4j92TH2evgq9vSxM840iWc/r0+Gq/FfNqpPLtzTtxpP0dvvLK/V
ieAp9YZZH9EHbV5l61pXxqvFCroXnlXi9dDHjVlncmXO8Ub4utH1d3LcmX2v
nloXdws+P97XPeLBnNOK35/5B3xXiZxn4MX810HOo51S+PWxR0vVhUwOyct/
KzwB75uaarYRMblJ973p8IP/oJso2OLN5CzWv7zTAS43ozUVC27BNS7HXC75
TywxgxucbBA24F3yCIlp4mGOOrOZldvgPUrdtD7et6T+9zoVmPv/vc//AWiR
Byo=
      "], CompressedData["
1:eJx11PtXzGkcB/AuGJdUzLJlS6wlW4k105wIPcUqt1SYY7rQuKtJ2zZRmzBY
sqeyefLdpraR6UIzk5hOV6XLSKSyG7HSbeQeKhmEtO/5A/zw/eV1nnOez/fz
eX+e6VvCfLcbGRgYLMZnjK9T6CAWVxdXWexhDuWoGCL50spbXEWJ58gBrRB+
xdTxHCuPIdLyq2M51ZRMnuFX4wWPd/Zfvgse7FcwlltDyX2bPUlucLOhzAv+
cIdPyTyioURhcWszF17b5jZ5LbwqTytcfZWSV+Nj59jqz//HjXWHuwdL4zfW
UjLCIOqjJXxV7/5uJ3jj7PKibdcoies+cYsNT2sYnlaBOhMHnhRsr6MkpGvY
lgXnzFKZXIKLWuMu7rhOSYx05yQj+M9PovkL4BnGA+t8blBi9jov1hjuYXar
f72SIVHGPe9c6ikJ2hJ1ZLiquOrRh6WeHQqGiEvuEvOblKz/eNVaf97K2nfo
Uy5DQjJv/9QEd7ab/a/+3tJQl8op8Lf2n29ENlCSaV5lPh4erjzIWnieIfNt
c50mNlJyz0v+Tl9//Rh2t985hgx37rJKgHMfP9d9B7fXiW7uz2FI2PokRT/c
OuvAJAd4yqLBvzjZDHEc1BVwmyg5c+zUHR78rWWvzcYshigSOrYJ4S4/BM9a
AVcGPh0xIZMh3St8KvbB2aejsvnwyKV99s5yhvSbWCSJ4Hx5UVEIvOQ5O38g
gyEW14NNVsGLAmp0MXCVZZTYGr7RdYzxaHikz4ZrFP7kqDb5gYwhGv8ZZy+g
Tv+w+bU58POd5e9Z8Jy5ojIOfMm4ve1l8GOMwQ6ndIas45rGyNEfX/UYs3/g
lsJ9nVv/Zshanu7hR/RT0Pig4TF85W5N+6k0hlwfz5/qCS8sfFP6Du4WVGgz
D772zeE7EZjX8WEbhWkN+lx4kyNMZUhr0OfAZMx3d2Bj/jS46wrlwWEpQ3gt
6rJc5MHuXrEbB24llXjL4I0jmRcFyM/DUZkvl8Oz99o5LIE3n5xTfQ55W3Mx
j66DR58c5bQSfjwtXZODfF4ebLsWBDfizdXw4QO+E6SpyHMNTzAYCk9PzPPe
Ch+a0BeaiPwb7vzePgbekyJpD4MndC1wP4x9yRCwBRK9Lx4rfQZP6RK7S7Bf
fS+bPifAo0qGZG/hcTNt3Q9hHxNjR5hTuIffci8R/tfzRUTq6UpK0m3UyiS4
6m6L2hv9aanweqO4QknbwgVZp+H71izk18EHjvTK71ZQYlNbP+4UXLsrJDMX
/f+gjlBZwOtLhQL9vdwHPY/+wLzis3Wm4eWUdDVmWcbBNS6SjBDMd5Hj4/Q7
lymJEKdw9PWXdtP6lWcYMjpSrhbAs5d2caLhPnbaZ7ORH8OI1Gm6MkoCu8WT
dsOVLl52v55lSNKBE9+o4TGGHRs2wWWd9xVrkM/8gCbh7/BF5tR4Nbw9IO12
MvLs6Ro2OgxeVLzq6BL4+3tthaHI/8jeqU6/wf1ty8N/hA8WiVzV2Jcb4Q6X
0+ASk7r5lvDKatXhOOzX3IOkrgM+3Vu72RCe2DxTVI99jLSSOzij/owTE4Wv
kbdg1SEzGfY3y+/VtwXwFpbUsxXeISvJ/gX7btS8NWAZ+uMh2DS5Fv4lIMXR
He9GZWD/YA/c5sh9fj485hJrHhvvTNtTrUaGPjf47hDJ4Jdq419Gwv9UCtgR
mFefx5ToY/CK2PzmZXivznNZJ30x36+9z/8DdLsJsQ==
      "], CompressedData["
1:eJx11Ps31HkYB3DNYIYuk+wqEpoYQ9NYk2y7juZxOeTWRkibS2iyZKTOYLfs
2VO+X1QkH2tc0op0d5yUjUptGExKlOwZIUdRy4rKJQ3VPvMH7A+fX14/fL+f
5/08z2d11L4ACUNLS8sZDxPPW444I2ln3T3bQ8OWt0QEUotZ6q5ZCoqdRTrx
6D9J6XDeegJWujymuZoC1tWollB0F5Oa8gUOBDomdPrIHAW764QnAtBzE1q2
SdGFB1x/+OoTBeNSIX8z+m2J9IUKXWHRFXr2MwUf+E60M7rj6wA39w0Ezsfm
ZLtr0eB2pZC1Ed3HkuIDetmjdo/RBTRoP/LaK0KPELXOOKE/8os1KGLSwL8W
xtQ44y0jSInfH8kq6/XUoaGwsydVgB7edXm9Ar0kTaCa0KXhnDQvQuNLJrTa
tqJLW0OmM9g0hJmNZdmjy+LjFyRgvUeeDrkb69NQb5Ws0vg/8bmv1JhPQ0Hi
2dyFNNy0bjz1Lfr+ucoTK9C197wsml5Ew3i4r1JTl61xPe1oT0DtLHzy3RIa
gqq0j3mhf+LNBw3bETDSaz+5i0PDM27vL1vRvV5kDjDRSX17d+xSGkoXVZuG
obO1ckKuryMga8/N8TWggZ1nLY5DT57i2kcLCPRttr7LXkYDJzjlbQp6wvBR
meFaApuMm1VF6Dfaltukoz/sWrxKYUNAkj2whmFIw3OPGFY+uttFJ5ckPgFm
69Psjehb5EbSCnSr5ZFtPGsCQ93+fA/0Zu6dyBr0sWjOXZUVgYG4N1V8dNnI
+JgCXSrvWn3ckoDSzqm/H//7otfJ6G/0bm3HOec1eE/nBwnR6FH3q2tfo9et
7F/3bjUBk/hLNxVYl/Rw0cRH9BTbTZ0VFgS2TFpUzmIORL8/kRVad89MLhYr
zQl0VXZsZ6OHscMumqCXeQtKU80IBGZ1rp/AnPPyJQtt0GVnMv88s4pAxdHC
Z7cW09DDNcgToS8bjPrVAf33lPSTUuxX84rQBhf00jvbbHeYEvDzln6Zw767
dX8/44POyacPvFlJIDdjSc5BPRqe6BdHBKMXbDgxeBh9e+Qu8QiLhibz/uld
6LWJjfuN0GNULYHeOG+98wdT96AnpPOyTdBfBhmFlGvTMJocYhyP7q40vGmG
/mCyNnuSQcPQYddMjV87ZfgjQb8aONPojvM/6pq9+AC64/18mRw9cAe3ueQL
BX81D9lpfKFitk8P7/+umXleG/erTVSsI0VPe2VaMYb+ycW/MGmeAvbWYoXm
+4ckZk0RmMMtJ6/3atzrVSMRejHoXMrC3BXz1BXHzco/UmBRWPVSU9fYY0+B
Jeb/LPZ4fQC+DxwfsSAInbXbXpKE/eoP8lav/UCBd1wVbEG/JFcMBGN/heM1
1dwZCt4xaio0Oe/n+O67wCXgc/3clHiaguQ4zwwHdIm64Y8POCel5bZNaVMU
zF9SPeahuya6hnnivMl0r4xOTVLg11dSvgJ91MexQc4jUC6M3HEMPWuN3x09
Td+3l5wexrmdYk2JgtHbZjeFq3GuDlb7mzvg/Bfsbt3mgZ7x2jdmFD32aYvd
EVt8H36ubI9Gry77t7MH/fKgy/0O3KNe46+Hr6LH3La5rETfWzM0Y4r76N9k
aGCD90z3NGTUogvOPGmKFRLgj9QpW9EnrT43aPaow0Hf/AbudfJ4YE8e1jtR
ZOpQgD648eEF8g2BhLu/JaVjPt4GOh/T0Bt3vvd4ju8Go7Q25DTm+X/v83+F
Abnu
      "], CompressedData["
1:eJx11P03lGkYB/AxYmJMIW9z0pqhOidhsNN76k4llRI2S7axtE3SYmlJaWvT
LnKK53kKbZteJCl2d8pWo50i04Ypx0taO2lHaqR1iompphpjv/6A/eH55XOe
c537vq7vdQvjU8K3slksVgA+c3zfbzqj5LrL6yfX9azi2zPEa+VqbZuRIrm7
T8oMQnn9hrX3qCF49DR9+qlRivQtaXEdgLsbCwYjpzBkatm621kmirwTWS7t
hs9g0w634PM08ubYMYrEcRxi2uEnPF9WXYPPHeZuCGbRJIObO7MB/sOfcdQ0
B4b89lImFpvR5F11q6scrowI5DrBd6tGXARsmtRk9gfUwLmlztPOoM7KnE4j
15wmqz+7l3cRnh+W13Uc7myXaK6D26QJPM7DnSfeWOEBT/M2XGqbQJOclvXz
zsKrA/J4v+Jebs9OO1VY0ETSt1hXBjdln21W2TEkvycp6BtL1Fmjdz0Dz+wI
3RoJX8i92z2bA3/i73YOrg214H9ryxAzzazrXfAElUJbAT+mTIllJjPkoLjg
p9SJNHGWvb5UBU+JkkovT2JIydblfSPw3jGR5Bq8fiisJJ7HkML3TvO/sqJJ
Zc30KgX8zs/Jymwb/F82hWmAF7jJg5rh6XsS4+dwGbLF5tlbC2uajOYlcLrg
Pcr2xy+sGPLJ3wVSH/iMh4rcJ/DULeJkLrzxUde+ZXD9praNOvinMamp9RyG
aGdZlC2Azzx+m2uGPIzZdwozLBly1649yRXuy3n83B7uWPWh2suCIWuj9q7Q
4jwZy/0KBfAJ8+qjwicwREqLyovgM+RN4WJ4rFQXam3OkIIvew/7wgfeTNUu
h+/xZiX4sRnySvGj6Xf0gaMcLower2+VpRpgMaTyNe+9O5xH+fclwl/LZP5T
4PQy2+7v0Gff9Sz13nEvOGm7eIwm3MAr7irMK0rJCsqHbw+53PbARBPZpOet
HLhpKElSCl+ovZk7OkqTqZtrhQsw91hipq2CdwyJXIrhET2Fhi+Qk9qoEdUN
eMvDbp0P/Jd/izt2IVdST135HXiilA6eC9/ZL0yikM+NN0WdD+B3mWORrUaa
PFLyDpUjz3yTneYfuOv2lva/4NYHSsrlyH+lMaG/H/4i41SIBv759Lq6+9iX
MN93B17CWemXTBGoHy4cLOnHfrF99QM6eKBYcLgZnnFu4qAlPC7YRq2Hh3lG
3FqC+8ZfyN7vhz090hgWYoDzvNcU1cAbHieEbMNeWw+mDb6Bu7U4xRSjb418
hwfVHylS1e0wOgy32rGrJhN9Xqj6unPsA0Wyej/4DMH1x2q3bTJjyCzJW942
eKnaInNg/JzZtcmLMMfawI2nn72nyGSti6oHXq1vCM7H3PPKUlbvhBuUq9LU
cM26LH4ictLavsHGCU4OXVffh4c2OeZcR66cBfvOdxgoMpAuXz/eZ5G27UoR
cjhyi+V4Ee7p43zhKrz9uG9xL3KrsRUYi+ANxqXq8/BF/fdKvZDz+M1+6lJ4
kIjnVAT3kLx9lWHNkIamJ94K+MzervkHx+db6LajHnt0O/NFlB4+Ony0MB1+
RLX5zgnsHTv74MdAnFPz/Glj3Hh9iX3HU+xpNVXpWAn3XlQuDof7K1wVf2Cv
V1asveCB/lQ0L8heDNfVlUaz8T5E8iWyq3BBzNE5s+FEGG5Qw+M8DlyJRv+Z
JpcAPjzncLfjZbwzOq/9yTzM6//e5/8AgPzDbA==
      "], CompressedData["
1:eJx11GdUk1cYB/AUnAiyZAUxUuLgKGgVsETAS92iDYq4alhhBIFAmKkERDig
uAq+Nr21BheVViNDVFxgCI5jqyKKAopFZagsJSaKgto//eKnfrhffuee8z7v
c//PYx8SuypMj8VieeLo4+R73otcc7RClbctwbFVSknPzJgxrMcMUd7f0bkU
7u8TVrD0R0q67OeqY+Fe2t+vzIVPa9ghKIaXh+cMsv5hyHFDoakLPM4kOvQY
vE2THqKA8wZTF3Dhkd9PiBm9mRKDjl6ObwtDVD8YCezhR6aFjWBwf/+SSheb
Jwx5Y3fghSW8ScKevR3+tEEvqQc+/kjWejN4ZxHf0gSuKU82bHzKkD2syhtG
8FuR3u9/Qf0OTSlH1c8YUpOaN8EYTj+17T2VQonR7tT8mlaG7M9YmDAavu92
xkp3eDXVpV9pY0ipXHnFFP4qNOLq8yRKluvao662M6TWVWdhDi8JXRSgB7/4
2/J11zoYovhsG2oLbx8vrStNoIST7bjg+nOGqN/t8hz63xcrpj8IjKdklNUT
/coXDPHb5GHhBK89G/9wrISSQbl1zamXDDnv6id1g2tuusdVxVKy6KD3GkUn
Q77KtdAS+EZdlGOsmJKDU1vyJV2okyYO+Az1rcQicWIMJaxn8sO8boZYjZq5
IwD+dvNex/AoSkweJWf3w0eat7hGwSelXZtqs4mS+/2GZod6GHL4kDozA77U
uvlMsIiSnTFa28m9DHlYHxf0EzxdNH3AMoKSkj6PD4fgx7bHrT0Ir6vMEd4K
o8R6nHTr8FcMkeU56ErhmbcVGdmhlBSLvXKWwLMi+Moa+Pz0ZJWXkBLTG2Yu
SfCfJ/INGuF3NihPvw+m5NK32wIz4OxVvNwOOC8oXmsf/OW++tyyr9/CzXPZ
a32CKBG/LC/z++/+3yEjCytUfNUMcXIgJXGcV0oTuPabkEobOLPmZtGRAEpW
p8xYWYH6WflzeJPho8Tmzh7we9GigsVw2fzCABd4KfuCqEFACTttTORZ9OdP
mb7JfHjWBoFLAvxy/+eRbLha3NfqAz+ukKyQwd97nuZHoM9vpZ/q/OFFjxfl
5sCLlxiTArzXBgnHfz08NiF+60d45u7sYX/hfRNyWBoBfDhXfCAJ9TiI4iUt
yIOHZPaeELgdf0pFD5wRdD/WIj+uujKfYLibXS/nHv432blM8wZ56xqWxYqA
702zFSxHf7i87hFvkE/HcdyaAPj5nguvFehn3jYFW4M8XyyxShv6bm+1al1W
CCUn7K469SH/dZFSj6H6p7ed7KrHe4n0heQ15qXp44NwP3jOpQvV5/G+zfFu
6R8wX1WJxvJlcO28TLVBOCX12p0Xh8En+Fp78OAp6yw2Xkd+DDVOXDbm9Hlq
rWoGXN6ZUGQZSUnq5MHWhZjrco5ZBRfu/2t7UBjyKZeRWhn2Q91NZwczeG3p
+lOcaEoW6I3uqMI+eVewO3Y4fFbDlLPRyL+d77zVVvAnpjKr18iJeoyXMxdz
5D7v4cJ87KstA/ucnsIbx8o7xXGU9I+dJbSEVz1qnloHH9HrnHUO8+jcP7ux
rJkhIu9C2eWheQxfG87CXAfOqb8jhp/srj6jhBtlSCctTqSkqsnd/ju49Ykt
iym8ouOG8y7sB4MDq3Ru8MyYlMpcuNNO325h8pf7yx5JbDbD79r2ZfOwf6r0
ewuF8AHlH7dF8JaTdyONsceK/fjKY/D/28//AqU/7yI=
      "], CompressedData["
1:eJx11Ps7lGkYB/AJTRHlkJw6GOUwUww7JslFT2W5UkzkkMbKjOSKzl0ppzTr
KUqnd2Kb6mpSqWRXay0l20GpGCElUmoTV1OWRcmZ2q8/YH94f/n88Dz3e9/f
5+ZItwVEabBYLHd8mvj8vhd1RkaUlLEaVyobtORk5OUn3s55lLxi51+WwvMf
coIlcH/iYbzbhpJ10ezAMHjrqpS8MLjX3B87t9pRskJPzBHDTYNv7/ikKScr
++InxfIoMTik9FsPDzru9EYLHhdxZTRwASVmv/ZJw+HH0hKaTmjISW35QPRi
B0quhmRojNeTcWyBIn+CnCjPLTJgOVIyrPhWHw0PbZmVXMWSkymp3q+KnSjZ
2FFtugXOS8q5+fE7Q5yIXrC3gBLj5pCmnfCrqTa+E+Ez3b8sK3SmJDVnlLMP
3jQ1+/KfYwxZ3jN965iQkjVFpz9Q+BvripFnowyp5VsUznChRPPMoIiBPzbV
5clGGBJ1p5YzaRElOXLnUCX8iGttjeMwQ7Yezsh7BHeyceu8AjdnJSesHmKI
4yC73NcV/dn1SFAMV7es1p4wyJA1F74l58LvdqjtHsIVq/mmhf0MKfr0uLIa
LtFIuVkLj/Tiu9X3MSSmTtv8Bnw6SZn2Dv7CKl504CtDLs15siQW3tH/ruEf
uGHeWMjVXoZ4G02Z3It6pC4xPgNwnVZli+oLQ6zj9CI84TXcx1FakpKySyXv
2zo/M2Rt4/YjEvxvxXUdQ0P4uWsn/fXhK07OF/svpCT2IEdkCR9zy+Q69zBE
msOrNkLfRh7MU3PhopYqdUU3Q67XSfN/R//ttZc0C+A3tKa1i+EpugkGDj9Q
kuLO9fSAC/4NIj1dON/Qn5eO+Q5OkX1eDr8+MXLLV7hjO1/2AHn4uerWJBFc
Ydyw/BA8+Nl9OzXy4zUQzw+Cv9z3tOoo/H3/5O4h5G3ttU2bxfDW3nCJHJ4j
uadmcynZ0VJeGgY/81djLhv1vE0zGdO0pSQzMKjjJ7i288XXCfBdJwutdawp
CTprbbEe7hRgodsNzzhW5as/F/M6WDo7Ar53n+esSPRh6oVfDOZxKJngIWwb
v/drvSrHC30TrSn5SOZQkt7KiRm/d39b+bpSuFAtX5w4i5J6Q+dLIXBp7+ak
c5hL1gdPbrkFJbZUOzQAnrz3jft+zPGLPLxTYE5JWWLNHu/xczwlwueYe4Po
yHClKSUr+a2hS+GadRfFRciJi/PdA4kmlJzqU8QI4Yv7ph6dOMCQyHctsg0z
KCntucN1gF/LPvpWDWfz/FU7jClJYhmbcOAObk2LXJDb5q4mWfF05D9+KMwI
nmnVbpqGnCud6jbaw7dEZwl14McVHprReBfbd5rnPTdCrpSVy0aQt/MD7aPF
eEf26R/MfoOnLawq6IGf0a9vzsS74z41Uj2A60oqAlrgSt9tqtZvDClc9jBH
D+cHF6TJXsB9rxy/XYb3++S2bG46/Fn+Lf1K+Nnd91yV2ANZBcOHnFF/blLT
51L4tPuChYnYGxsdT0hM8L9RjaVL8+Eb/GwrQrBnws0CT9miP35JPqzz8PKz
8Z0C7KUZyYL5UeinVK99Uxb8ue0t+2x4sFO/n8qMkm4Fe3cGvGGo304Xe6+r
saDbB/P6Q3haSwbnr7pZuQeeG2Jj2TuTkmpxgjAOvuCwl1UbPHRdbE3JbEo2
x2/z3A7/nt195294nI+Im2VJyXsaZhsD70rVdX0Nf6LqtUyxQp3/s5//A4uj
xek=
      "], CompressedData["
1:eJx11G00lGkYB3Da0aJdu7ZY29QRqu14GYOitbG3tzZJMmNWnRRLWFS23bbO
Edocb93YUU/RG4VG5Wxi28FGmEKWkvasYuP05mVGGakx7YxJ+59Pz6f98Hz5
nec8z3Vf1/+6baKTBbFzDAwMvPC8h2fc6NvNsc4NrV++f3sg5V+GdHyoW+qc
S8mmyJufiuAHqxc4hGkYEhM4IQ87TMlAoVe6N3yrFz1zWssQx10qyzJKSZzN
G0MneH2Mk/zHGYZk9KbsNM+npMTdZREXnreYNxikY4j8K82ZUwWUhBUPPzKG
22l5uXZvGTLfKNZutZiSzPBUPzWvofXKYc9hLfyv8s99XxRSIlhnz1XADa3b
o2NmGdKvbHapOUpJ1g82tx/CDznOlXXDA3fU2KcdoyS1ICGhBy7Q1putfMcQ
cfCcixuLKCn4OEfXCo/5pkF4Bn5y/+p42xOU8HWOu/6Ap+V7zi+Cj07puBMn
KbEWTmhq4ZoorUIMd6roMa49TYnhntD9NfASTXEQF74hWRAXX0LJrasbJZfg
ffF3L1jAk9rWyzlnKXk6YDF+Ec5Jq64uQ51+zj738s5RcsB+X5Xe10i1XB78
++7nW6bLKJFtHS6phhdcqQvagD7UhUZl+FRQolo27CqFbw+IMUxCP60vjZL4
85S8+y4voUl/3mQDL4r+S8ePH4+TUJK72ljUBne9EGZWhXmlMyUznpWU/KLN
Mb4Dj5qxzerCfJs9HZWP4ESdzx+Em4gqVanIg6795pXwC5RESYTho/B76cml
pW8YEmb6JOM8nH8/yOoNvD7nSaRAzZDp9QJuI/zmQ1Ksn+922fLmudPs+117
E7UL4cyzyv7G1+z3Jw+8NefB+z/pzdnziq1HeqfYbQ3ce2xuStEUW382KZzY
BN9XcflQ8EuGLOd490TivGqPNNs4ON/IX8mZZPvzcsTeIhU+1dyibppg+7mU
CVbnw1cNJDsMv2D7vyVxd3Ip3Po3xeA8+K+5z0yyMK+Qa2s1UvjeSTu39nGG
cEKeL5ktpWRV4IKcG3BhoCU/Ar7D/0DEduShNnyC2wffUN/0Z6YCefNK8qlC
fsKPWPs8hRdaVfkugl/+qThsFHlzqK92m4bLH6WvCJAzZHCw9ywX+RQKjRQ6
eMgyyc9BcNnhE4+/Rp4DSy1CTPkNrVVji3nPxlC/ffeSvcj/TIbfkDm8VzQS
q4DPG96UeAn7EhoTsfEzuE56PWsSbpFRnzKC/dp8LCR1Cbxzp3D3NNzjyaut
LtjH67celNvAaVbntQT8d8zJhDmC/S33sImyhSs6hmqG4CtUHIkp9t0y2r1M
78qpKRKqYO+Hlm0OcXpvPWeW1KZg75Mb9xc+4MJFg4XXJOPs/WP79rxkAbwo
uDYz+zlDLGUe3c45lPzzQiT9QF+/pbI2DnO5rONXbMum5OBddcQs+hOp8VD9
jvmunJOaWJdFSZFUoVTBxc27xhglQ4Y+Wl7pDW9qaTcbgfeZhIw9Rk4YQeN8
VSbqDBoQ/A0/FbDY1Ql5k3NkKUPw8tm+vE74F7sd3E1fse/79/mtqoPzjNfx
RK/Z7zfLd1pX6Oco3pN7VsXWs7a4g4j19ay4uEYxzdZvHiETp8CtukZ8XbFf
hpuvHnXHef/vfv4PtlXxFg==
      "], CompressedData["
1:eJx11P03FFgYB3A0UWlb5bVSZnZDWdmcWb2I3ITdppAxYSQjRtN4ibISGulF
sY66eZntRbGWZCZWTjSxZwa7WW2iLEVllaUkNZOcdtvS7Hf+gP3h/vI599xz
7/N8n8uKTOBGG+jp6bljTcPK+36ap4ubolk7x9uzazcllu3G79z8KdlxILKb
Cb9BbuVUJFCyte3jEGcLJWd2OZWYwI35eRdmJ1Kyt+eIsTCAEkmj9hYDnvVg
0xkGfKekd14WlxJht2bqzVpFs212uFcsPP68fv/lQEoWp5+99wwu53dNj9Kd
o5+p7OVRcji1yfA+/Kzhh5qbcI3h0SNTWynxp37NXXBphLqGC7+X7T3KCqbE
WrPU6le4nrSyjQN3NF8dvy6EkhWPYvoVcO7rQdsN8CpeSfcWPiXc0dSJOrhF
h4X5bbzrxEWr1MBQSlTyabXV8Iyv130bDJ9Ro4jw2QZ/2LqsCt6kEgYOoT5r
R29vtA/DO5gD62XwN32Obz7GU+JSPHz1Ndw3rSW/Eu4tyJuXCx8fNi2r2E7J
uXncCt1+M4eyWVVxOGd6xxWPcEoultu/rYGbvuh78FssJS+Ti4VNcPHYwqCr
cK3ZoqhnMbh/iLXrIgElR+50fN4Et3cU/WUED5UEsMPg9a2ayVZ45JlC02Vi
SsLSnBIOw+39xtkd8DavU2s37aJEMXanMwu+LV9ecQ8u9l/F3i2ipPf6+dex
8EWx77cOwYc9dxjk76RkyKr2GhvOTbrjpYYr3YYvNURj3zYtcwD3pDk8CwP0
/ZGSuscLKSEHBb5i+AbNs+S58LHt6pyCKEocages76MOx1Q+GhY88blJUWMk
Jct9ZqqXwYP+9eSw4beX9K4c2kGJF3tllAj1fPTJ/scE3udF+KlwYWkc/wT6
0tejehkIX33YmlkWQYmdwN25DH38QcZbFQVntAyKXOFFGU+/K0Hfp06qylPg
78uCFgrxnsdFX1WcQk5eLSjsO67LvyJEbwr3lvtYZe5HrgIkdT8VwH9Ob9cy
sF8Y65weHERJ2zmb9HI4ac7hXMD+02GhDV8inxOC+T/WwvMcwjMvwvssZ941
RJ6t/BviWnTzonK9qev3ktGZxoOYC9nEyfp2uDz3YGsDnBQtL2/EHJkFxfF7
4Jc/WJUo4ZKE+dYXMHeFV47nD8IPRZ1mt8G7GbK6XMwpK06eMgKvPJ3k7oZ7
vupNM8rxo+Rh9YZrY/AY24H7V+B7j5UukPoi74qRNWp4yo13E3aoj1G0gWP9
ZkqSb1SzXsFNXUJ4s1Bnl5DwprFNqN/I8Q4N/IUs0TkDfjfmVvtKeAqDdf0l
vHy+1i0CfTwwy+ZqMYeS8vQs0XO45O2fe9aj7x89Fkwx4bvH+w2H4YEtnWoW
ciJP+myJaiMlNnvs1gzozp/LStFHrlLlpUsz4Yu1G3f1wn//ws//MVzaEvyC
DxdJMx064IsFlSuUyKf5P4W5QfDMvwsadXVurO70PIc82zipxIlwb0/l7AZ4
l6FMkoL89xsoKy7BGSaTzCq4pWzOEy7m5Y+ebA8t3CKUHSCFX/rmUEgx5suW
Y2+ZhPtPPmmbPAafYFua7cOcio2TNk9HHRiBQsk+Xf5VGZ/6Y67HOdKDdXA1
r8k2WpcTnmCvPf6Bmg9PTZJRZzUpKAyGi7tncCl85Jc1TD/0ZbxfL9sLvr6o
LFek+2c8Gn1d0cf/+5//A1/PwP0=
      "], CompressedData["
1:eJx11Hs0lHkYB/Cx7K5Wo2XrmMqlNNtGZcplWsn4yXTTdtlCqtlK0eTg1EZK
szWkC1K/ihcp2rbVuJZFSXJp2vYsSxiMa3GYUC6DYVrS2f321/61f7z/fP54
3+d9vs/zzN9/eJvfJywWyxmP7scnzr4sbaKwPPWlaN2yvxky90auQv2KEl9V
wsMEuEd2noaB82Z/75DVScnV/AX+F+Cb1qTuew8/5eLFCemiZOFiU/5peJTR
6tFRuKn+7vdrVJSUFKjZx+B6bc2X3sKbdSNTZvVQotBrbwmAn1OdN++Cf81e
IO3tpaQocqezP/ye9cl1gfCi2drUjDeUpJx1+PkgXGDGO6J9x5DZXMvSgH5K
9gi8Q3zhEXv1dCLgkSHc3SaDlHj11xfth08I9CYN4IfrZoZmD1Giw05vFcMl
tcJKKy1Dumckb+YOU1KV/+F4ELw/xoy9fpwhJZTjFzlCSc5OVWUw3Mfhqfbg
GEMErSMnSkcpOeU0XUcCNw4ezj2vYci8LwqiqzSULK7fbRQNT/r2savjKEOM
co96l4xRskPGL736sc+a4u27RhjCifyOGzFOiY2ftfw2PEd2557hML6bUT1g
qKWEVR4oyYNbH310/9kQQ9q+mpt/Au6/WodVCheFuop6Bxli5bo0NBteUhhQ
Uwu3qV20NnXgP+9zVUao4PyOhH88+xkSUhYRLIWfmzrmOAFv1suQGb5liNfF
0hxz+MkVA0H6k4XlxTE9eQ5vGGJhJ0tLRJ17BSvemcN1oyZ1h3sZIv58ZXIr
/ov0Zc5cBh/dfKiaAy8eNawaRR/+FHpoXOHjN8fWru5hSM8Gz2IV+qbdMj/O
A77eKuxI4GuGlIVpxLno88CtxD4x3NfFzyVRxZCLoSNG3sjlwaCyXQI3cJCn
yrsZIhX+5NiJHNujFlrGwM84JjiK4JO2KeYbkbvTvU1ZSfA6/jRjbRdD3O+6
n0nHnJzdYfebDL7M5jPRFfjMpzkLBjFX9w83Tj6EJ8dLu63hXNkJN24fJaar
TvLK4Z3XrNzt4NUjap9tmNtzyUkHKuDTu71zneBKcYE0HHNem1lzUQEvcs4y
EMIvF4y052IvzlzhLW2EWxRadhTBbRRRj4ewRxVjg9VN8H5vUR4P9W+aG5a4
pIOSWI7nVAtc7RJrlga3vU/Hw19SMsQ//qgNzr31ZN4c9Kd2p6qhv42SlRNM
ihJe4fP79il4nJupNriVktu/2n5ZB59VVWH+Cn3+xkCeZNxCidslRXklXF62
yLEMuYQrBaebmiiRNBy/9AzeFFQaeQs5PncWiiuUlHC6Y0MfwANOczUDfQzZ
y/5D0dpIiTRVXpkNF+on9/yFOel0qdMxgQuPeFTchMe3XEkwwbzZO4U6RDcg
94Rmz2vwfXnT3o7BjZ/aBVrAi0/Zx4bDgwZ+mBBgnhN90mX99ZRELCJPguAr
8rs4UUP/ebZ6hv4u+J7A6+JaNUMiLG9mcfAeN/IkbiNcW99olIk9evhsVkgY
3DzokAn/Y/3NDaXD2LtDG9qlLNQ5dr0g3gIe7eUp4WN/52SwtxbDnWuC/A3h
rLqsdDb2vUPE801DH2IuP78whX2565Fmo4IvGV8uyUffnqsy1W/ggy/M2I9x
Nzb++CH+dTMl8oYtMUr4oxpTHsWd2drZuJwgl23SFzfk8DvrhxUHcJcMp0lr
HyBHlf1lzwL4L5+ybCvhZ/etGnFH7v93n/8Fmts2dg==
      "], CompressedData["
1:eJx11Ps71HkUB/ApplJPdluU8ZSiG61Lac0olbNqTYWN0Sor2VzSqmgspWyt
20h9fLZBxuRSaUPhaZguI2uVLUqSUQrVoAnlfgkxxB5/wP7w+eX1fJ/Pc77n
vD/HwCuQ5zudwWBsxKOG59faUH2BT+H9HXo8hUEahSHu4PqT7wi0rTKeexTd
g3amFl+koF9bJ1FXEqgLXr5jP/pHDXasawaFJcfLBVL0dzPXyn9CD2P4nOn9
i4KZIgnIewKHiFvKFvTAQ365p7Mo9B+Fi3EtBO4F6XZYoM+LSeIsvk5BydbQ
ut5KwPf+iYol6OLyL+vu5FKw6u/Z09hGoLJokKmJLrMbDXa4gXUGxJdxPhJw
j5i1btK78P5Iac8KtXwK26v9xy63E9gMzYpedLm8atXdAgri/OFK404CZQNZ
h5vRx4+oawfepCDy6Fss6yKgd9GwVo6evrKzYtltCq+Z5mGOPQTCzOqdytAf
CeL1x9DZJTmKrl4CGY6VhXfRJWt83WruUAjUjnog7CfAaNUJlqKzao6deoze
1sNcvvoTgQmLg7ty0ctmuNTeQ9e4vO9U+SCBLzUmEVfRL3mp/pChm/k1aTgO
E/ARcT+ko4dkrwyRoNsuPapR/pnAZMWBF6noph11iY+xnv1PuNbmowR0tV44
pKCniXQ8lbcoZO1mPg9VEdjT7WZzAd3O06ZkHP931fok5+wxAlkHeprS0Cdv
i0OtpBS2Xu4bKx4n4DbhtT0D3dnQzzgE+9miOSM88wsBWcZ7Tha601z7E1Ls
/8b5G3KPTBDguU1j5qM3/qztPS+Pwidlp2zBJIGaYElmETrb5ZkjH+dbfO3K
nnPoUtdjz0vRPXiv3AuyKZQoeW8a0Bf4qhRV6FqBLgytTAoN4jRrBiMOouyv
9jaie++eeVCEeVOqd4SM4vfawqHkLnSHzTpMFebTxnLn1gfoN5dem1Sht33e
cnov5llTcIXthT78Q9A3szE/b3UL6x9coBCz48LD11j/Mht7OQv9nODbUeNk
zIOn6YAJuo+7ZqwZ+m7NrdUFiRTy/omw3ol92MhV3rVBvxKmV8CKp7BI11/J
w74Jdw17O6N792UujfyTQpP690Fs7PO/L8/beqJrzBlyfBJHoVDcaDIN59L3
dfEYH93Po2jGd2cpWK64+jh7hIDlrcOl4eixEpXdyVgKPR4S0Sace47VuJYQ
vY4rX/0ohgKn4WzF7SECT7s6jl+aekcT89fME1D4bZ9BqBHmylaPP1OCbqQG
v7hH4btIS9ZPGCBQ2nIsugi9yUh9KDqSgmdOa/FAH4H+E6Y65ehnjL/KuBFB
IahY8epHzDk71NStCt0hscNfhm4wYBEs6SbwJH5bXv1UHwTZlc3hFCKgXJeF
72iBjBfehM4PqJ3bjt4lP3Q2oYPAhpf+Te+m9kNW0CY53hNns+U0C99piqvr
5/dT+2eEz5JhPWk1XOf8DwTiJ/kRrVP19LlsS8f6ewWsO/txD6QmTE9oQ3cS
nmuLjKZgYd69xgr3xtAckUqJnhyfbsnF/kTdUEYa4p6RKdLEjejP3U2Es7Gf
/rPiVOtwL+nxq6UN6NaDBfZVZyikP9xbGYB7zH7Z8f5q9JQMtbAAnFfoG5u/
H+E+/L3+4/kKdJd2Md8c5/uyXqXlhO7F6Vxbgk6tODnPhBR0W64lMtGDn3qs
lqIv3MXUP4z56W6OtmhqJpDEad+XiW6l25w6W0TBznDvSCu6p9GiqET0/Ld5
DjGYz1MTK1YuxHv+bz//B6PFAak=
      "], CompressedData["
1:eJx11Ps3lHkcB3CXrQlFrWpryUmrRrrJktZtv3IomRLtTMJ0Ksmy1CDpIuV6
aubrMuo5mqJCY7SisRXDdDGdtnJJWoRMWXQ0oUVZHbOX9j1/wP7w/PI6z3m+
n+fzeX8/1nsPBoYb6OnpueMxxLOie378VjNF/bfFQzn6Fxly3ChEKr9ESVp+
0Yv18Lm2Zsa+BQyJbUkO23WZkk2+ccwSuGCVejSnkCG5H+tc7a9QwpraIDWB
nxJrnV9cYkjD+qKDtkWUjIoNp0+YKuoboj36La4wJEVi+dmnmJJS7Znfe+DR
WzzN9xQx5Oyn7IHUEkoqxtP2NcCvH3ikcC5miFHPgKX6KiWqzJiUavgqro2+
aQlD/GbZDPqVUhK89e+VJfD8ylkDA/D3nepT7TJKjEb8LuTAv2gV0NqrDHk+
9upq1DVKwrMCG47D92vc/8mWMoSssAg0LaekL3Pb5xi4g4DvPAl37OFMU16n
5H2AynE3/H7PLQG/lCF5lp9qIipRDyf3QhC8OthewoMnvF0YNVdOiXxmuVUA
3LltVO0PXzvZz6urooQR85p84a1+a7194Rmdd9123KQkaWZP2wb4ll8t9Lzg
y6Pkbb23KHk0r2mXziu82ydaUE+uKpzPq6ZksUOWt87HjKLdQuAeH7afV9RQ
kvwwUanzuMem0zT43+McsdtfCkosXC4N6s6l+XVOLPgVp2bR4jq8n+l0W1cn
t7h5ahn6FpESoLJWUpJZ3OEaDL8mqQnbiP6fkbGCtXCZ3Gh/GLznnf6eHzGv
cOma7so7lNguMgiLhx9Ln9+nmy97qNPj+7uUCLxtuOlw//xG91bkwdBrZ7IE
fpNVPMHAFTaCdRnIz8K0l3ufwttr/HaXwffW5g+6IG8y/0J2O9xA8Ue1Eh7I
eRI5foEh/kG9LBk8buiG/Bn8ojnvZJmEIapWRQsXnpGkPvUGfsbEgbv7PEPW
dK117ECdJ9TsRVr4uydTfQvyGZKoaTvKhpczR0NnI7chb8r1fmPwnU9DUi7+
N7qa5cuGv/6JWyc6x5DTqS2H+eibGftBvwfcMLvR1OcsQ+wORY541FLiPRbf
sQ3uuPHW+wN5DDl2ePosQ/Q/IeFyYiTceNJqtZ2YIbtmiLuuYY4HTY78kAT/
Uzb8ansuQ4S59eYutyl5ZrK5KAfedVI8fiKHIfbWmqQq5MRh6qFdCbw+xNi1
LJshKzO1IRa/4B65TT6Sw505ha0E/ubisDwROayRSET34ClPvxF2ZzHko73m
30bkdsedG/7N8DnK/g1x8Btb+ByrCkoGS9abv4T7WrzUGsP1tZslsbgXdqeH
s/rgC9zvVcyBy6VZPzfhHj02/75XA1dtPOe6AL7Upkm1ugz5DDVrH4Xz0rMP
W8GHBSvZpbinb9NGOWPwxhP8k3nwjJglkjVSSkRC70idpzRnOk1H/a37+Tuf
Yg94cgaa3sGHtQWpb+EFNfPlQuyNXoNlbbpzhTO+8gxGf7wyH4SEY88kN+jN
1NWpmJid7oJ+BrmEWu/DXto0tYJ2wblHuq2/Rv9FUo90EfbY9tVfujfAP2gy
vCIwrwjOOv1O7L11n6sO3Ycr47lKb8y3KjfgvA88IfKBsFK395bbxoiRB9nz
Ot5IISUxBh/GC+B5QT6pauQnLEY/VgEfNikUCXX1LJ3buwx5W3h53vUq+P0O
y+8SdX1ePhIhQD4V8oHm13DzKK1nGDxW2BtaizzfFoyw7HHu/+3n/wC6stXd

      "], CompressedData["
1:eJx11Pk31WkcB/A7uVlSthjKUjmSJYxrKQnPMUVRtlEpWZubtVIoMpmrtFDO
o0Lf3NNipsStVNc6JBchSxkjLlNNdJxItpslztycefsD5ofnl9f5nuf7fD7P
+/msCT3iw13EYrEcsGSwfnW8HG0oLhdd0xNx9zsyRDbk4tvaZ5SERrRXm8Dr
8valJTgwpNMz01MgomS4dcTECH7GsqT1O/jW5pyUzDpKKpqGhtfDY04ahuht
Zkh0dUdM6HNKFNaMdVvCNcKWGtjbM2RS83igaSMlVrZsZ0f4lP7lsM92+K/0
iry4iZJg3VUWW+D1i80sFeAhrDGL6GZK+ge7cr3hW0ZXJFVuYIiwf3b5QAsl
xlHdz4PgN7X1zkfbMiSRP11s00aJmZFabRj8wzbZyEs2DPGTyqiEvKQkyUYu
4AR8SfX0mJM1QxpPxIeHv6JE3JwmewFeM612borDkJLKyiq7dkpa2D+evQ4/
OyszU2jJkKqwGbc++DfXrvH78JbK3F3BPzCkIrwje8+flGSV9u2vgQeJ8qo0
LRhy2kmizYfHGQd8aIOPNl9zdDBnSFNa4DsBXHm3VvAHePLUwNOZ9QzRXzZ4
PRUuHVzcJYHPuRjY6sG7d+7xsYTvklXauainXOTOGSx1MWVIioG8iwDnGUhb
+2I5/J+X7M0xJgzZsXsobhx1vbgXYmAIJz2s/FxjhoT58m7JwTWk3U528Mkb
BzoajBiibu53bg59a5qXT3GBO3q9NgyCdwz3hJW2UvJ7+abtvvAbfzWz59Yx
ZGPpIZfd6P87we1Gf7j0qxWfhe8/J9A88QtKuBP5nyPhz7R0/+bjex7/eAUH
95u5mh8VC/d45H78DtyhPHYnr4ESE7/s2/EL5/nEa12JfZrZ+YFF9ZSYuq99
nwxP2i4xMoB7l8UVddVSEpZ55UwK/KEwezwXfuH+oP1oDSVLJtk6qfABUaxA
FfWyZF7LyyHPGr/J1JyD+58KjLkAdysTZek9pcSwzSvhNHyuuc0zGn3r/fhp
lW0lJTM+ZSWn4POp/k880ee8UcVDBysoyVlFexPgic5V/3JwL4865+34ZZT4
8vqCFuoauqacr2HGkFQfcdp4CSVL9e+mH4RnRAd//Rn3Ppdc5L+vGPWOHfl+
L7zVxK+gGDnRzL/7eOAJJQFXy8XecN9ykpaFXJ0/EKxw9TElR5dunnOGi1m6
bf3I4atsKZf7CPmsU/e0gtdETBBzK4ZYFeq6BhdR4qpy4IE+nNeqWJmInKsV
TDy89JCSX+4k6qjC+UP1zs/xLhbbu+0bekBJw5u7gnnkLfZx5MdleF+nTWYV
j8InFa30v8ALbyVFfYLb94nPmMG1pm8WvoPP5qWzLPFOrScktbrwy9Y9yu3w
zJYvUUs2MSSiIkN2C5xXtU1YAVfs0VX3whxQWrNDJRteraltXQBXvyq9sg5z
oyP91GodnJPlXnwvB/40fWL+G9xbSWVDL/z1Yc7W1IV3PTCS2Yn5MyXMVqtF
vXckDaoJcNfgfDd7zLG3osOcdvQnayr5TST8KFcizYP3paprqaLPJSuOyQXA
tfdcVJJ3YogHRz7lmJCSeP+W2x5wz+nG9EPwrqKR9RLc1w4mI8JlYZ+aAgsu
fCpAoJNTSomkMHTMAZ4gjKX74QJlv4xt5ZSMrN7YZQ3nsjM8foLHjfTzVv5B
ya8n66+bwvcmNA5vhwtd4q0nkTevYUMFY7gNz9S8E+d0fF8X3ot8/t98/g/N
SfKn
      "], CompressedData["
1:eJx11Os31WkUB3C51FKYmpU4VBjpruisKQ2aTaeMli7o4jJhkDHHUO5NyKXm
ENXzixzlTkpHI5GoxpBrF2ZKZhwRKySUMCQRNV9/wLx43nzWs56193723jqu
h20OycrIyJjiyOH06//W9dj49r2S7BudPeMcVS0M7dt9nlGrxtqScnjttlFt
qw8c8YVLwivEjOL5WwX5cGV/vfVFcPVLB8QmFxmZWA04ZsIVD08uk8AlJa4+
D1MY6YeVf5UAz+w17MqAxzjozfFIZySvpaISCfdp8glTmODIQz0kXTaLUUbg
XpkA+AF/qd0U7hspbeGLLzEqLlgz6AVXn1jpfRKu0VujoHsF71uPnHeGN1cb
3Q+B54hsE+KvMkrqGXG3gycapB+Thz+Pu3twJI9RY4e1ug08Yq3kzzPI9yfn
czp6+YxET4uFVjPvK/lxavDG6XwXowJGDedySi3gr83l5bLfc7SnW5unUcgo
t/zF6FZ4SJXvEn340zaPuIYiRnI7DDdsh9fH6a7eMcaRo5swSFDM6I6WZr0A
3tJzdKD5HUe9R7WTw28xiroQ6GsJ3zh7oWRslKNc3dvvw0oY8Yf5PrvhBkOn
96nCcx6I641KGZVLf2y3hh8KqryfM8KRs7OmfRF8uPJItANcerDNtu5fjubl
Plv6Cq4gcspzh1fIR/f2DXN06ppD6mN4kih+1hG4hWaOjRK8qEnDNhTuK81s
DIV3mk/XGAxx5JMYV9mBeOp0V36IhRf+7LZ8/yBHqrE2V6cRf1qYcmAy/H5v
WlDoW46SHHfMeYZ8nVVW61yBG4d8/v5L+N3t8zMjbjLaPD9lshS+xynV5psB
jtSuXL7xEvUUTGlUP4J/No0edH3DUa3L8hDeDUYrhdEFz+Cuc7vUP77myPiO
esXi64xqxms9++Gj0ubGBLjX2N6w8WuM9o9JpRNw+bSdivpw/1LDjdcljPxM
xVNzTW7fez+pXVHXz9GoU0g/5TLqeMJCFsMFccE3XeBfhJpfrMxh9HCnmss6
+Ip/NgxO9nHEmrQV1mVjLl6kehvD1yTeU5TH/Zb2lMPJGYyk7Xm538FvfXIR
KMMNYyKl06mMAnYr3dwHX2Z5xGkRfEGQ4bfCZEYHrxqkuMKDF/U7a8FV7Lw8
2pMY+bw1URfCnzruei6Gb3RzfOKdyOhCY7ZGEHyio9A0A66oYvFROYGRp5u9
ayh8oKGzSxX5jnP+a9o4RiPjRqpR8LLuJO3T8K3uLem1ZxklTBuXiOC7qkQZ
s1BnY8vNvFenMb9+nqm/wnWaxlx74AVTkRf5cYzs89oMYuB/+R0NsJ/5L9mX
Wb+fYhTp2XD3JLw160O8Mf531a43fhExjCIC9/fN+Ajv8ng+vCG/TikqmpFe
xYl3x+HWqg72Z9E/X3dbDteIGAk3/ZERDO8PLq7xRr8ZRZml74TP7/Ud84Ef
mBVnZoX+bDonbObBede5Jnd47Ine0lXo83CvJnsD+N+WklP28PIgM+fZmAsV
wSKPSHhUbXWZFTyF1zreCffd3jG9FPFI6gXJW+AneBZZZZgviVOxlgzif3Rn
00c+3Gm58JcVmFO+l3/9MuS7beyyry68ZarVXwZzbV715nh4LCPTgqC0BfCh
LUJRC/zYYnGIJuqpIpNWLgd/oBSgZob9UPgFHX95hpHT+rXvhtCfUftsH87s
Gb6+bEw3Y7THLt3qBbzMqZoXDv/0wzzx7HhG/7ef/wNiafAH
      "], CompressedData["
1:eJx11Gs01HkYB/AZNFsMm8tIk1tnt3TBZknMyXhij5VbhVGJBmHQhTKd6lBO
YaZM82BVVhutaueII3I/WwmxbbK7OXTZY8xZDbHSDRkk7TOverUv/m8+L36/
5/d9nue/PCY5OE6HwWB40KdLX5dshfRwWFPL/dMT0vS9CP23j2sWMxDEceml
e8hr8zenOO5HOLRlp8UsE6E42KxMQD5THLCMeQAh5PbDthldhF/++G3f9+RZ
QpfUHq1zS8ZZLARzt/lEF3JB2tGJzBSESf6zijULEV5UPD+3grz7hd0ZwUGE
i338hO36CD54ZZpLDmbR3r7kKRZtL3LYCH9PDC1nk8seqh4vPYSweo1PaaMR
ws53Gx4xyQ9+82mMTf7zme6wvsUI9n1GdzWCppYCk6ehDPK3nP3Z70wQBNeU
X78mD5Zmbpuk8+0r7UZemyG4BoHPv+RH/JUW+uSvyq2Yz8wRjnv7fRokv5Se
W2JL9cvS98lKLRAkplmSAfJqw9aVbskIR2o8vYCLAO723SryCOvhq1spB/HY
nfiGZQjyT4UPlOTCqd7ZRMpzfEY3baEVQqpHkOs/5OrWBQsUlP/uupD3NtYI
qjhjqfbeoYvz9upEBGFBM3ehDYJ6V/K4ts6NUY0NyxMQzERiRT15R1RNzFvy
Y+49kuh4BK+8TK6DLUKN5uYrbQ5MVlTTlViEtGK994nk+9bsKmFQbtf12ncM
xXz2iKChCiPykoI89IuivhQd7ncmlw4YpC8lf+fNfZy7m95zc6f9PbrXImMk
XtvHXA+D9CcRCFmS0oEl5JF/lVW6avuYbdLgGY7Ac3HMcKV3TTwfLv2OfGx6
buj0DgSMUOvZUA7nM4SxIdq5urzevCcMwcEvr1FJuRXpx6/VzqHA0DfHWoBw
wcMmOJZyHvFou5xK/lO1/+OkEIT1m8OU96kvbq28rizyQB2pXtM2hMwkU86C
JQjRqqrqC+QvGeVLWFsRDE45zi3jIMywP7wvI6/f/8WV0CCEwLrsImtTBEux
6cZG8qqwYwGnAqnvq3QdvzSmuTvqpu4gt9Uwk6sDqO/t8qMvaQ6Tvebreslv
qJ91qPyp/kW2zc00t6dZJxRqcudJkbch+Q/xtZ4ymvNTnFHz1+TreDmbuOQ3
uIntUbQXzuX9KdPks8/rvrIjP/mK38mjPeKU81J1ttP8myt/dCafZCQ9XamH
cK3OzZRNHso7V+BJflW0yMFOByHpvLEPh7xHJGb5k2eo+Gw+7bVJaq5sKbmx
1fXDLlS/pCrwmHheDrUeJzSW2nM2lz2xpPdmbLg4+3RODl2routtyH16JEI9
yqdDbHtW+EEOO5yM46zIO8MX3XmwBeGuJzvZcFYO21p932jP578dq5RT/mtj
5xymp+XgP/c7g0uek+80eDAYoWCqssWQ/ABTetaM/M6gb7hzKELAoZqUGI0c
bpn6rjYil7p/VExQ36v8HjWOTMlhbX+CpzaH0s7wb/NpfjL/TLC8Qe51rzZ/
hnLr64q+FhT+2QsVm0bfkMe4DKv0aT7LxxWTA+SMdfVWQ+TNH4sqOiJpzkQh
14PpXraY16vtI6ulszWS5j82YDePQXWOqveO3CeP1JENc2hfbhWnXR0gNzhQ
ofMruVPhhd6uPQirAvIGp2bk0N3eOqog1/g4ibT72HZyO8+dcus14a8sJB8W
hsxxaH8vi5RlNynnh2d21UrIi+slDQ9o3/PNLtkFUV/+7//8H+knwAQ=
      "], CompressedData["
1:eJx11G041WccB3COEKbDmodyEsrCqpGnjofciY4T51iSxujSwVaHHPRAmMK1
HdqYuK3+JE01EVtaphoiD1fRZkZHnYjoamrV1HSc8tC+51V7sxf/N5/rf90P
v9/3d1uKJEExLDU1NU98Gvhi48xeVtKGlpQnVa36Cxky6LBSzdGRkt0yzugx
eNgl1yuz7zKkxs5YHOlESWSo52+58PnbZXVP4UYSr748Z0oC3qTapcLHevYa
DMGl7pKYThdK7jY71iXCPVsav4qEs+t6FHOulLiO5+iJ4GzxyfFAuFVbpNSV
S4mulmBpGFwzve11nyFDikISn0a7UZJzcmVrELzeLf/0x3CfHmNbqTslijcL
Rf7wjYeGBUMGDBkN0Ymu8qAkzsvGPwA++SLDd4rNEHce27/ek5KgzdPv8+D9
EqUwA156WXiveh0l5fzaX/jwbU1tC0oWMKRgdzqv2IuS4N87N2yEOxZ9PmIF
9z3MidhFsH6S3SnVvtfdljPr9RnCf9X+0mw9JeaCCAvVOb+8ZZK14x2G5F60
cL8EH7Tu3aS6V/2RIsdsPYb8HdKu4ehNSZJCWBQFv6sQGH2vyxD90/wDR+FK
UbmpBE7On9vWrcMQrcA2vx64b4dlYxo8Qba46/l8hpgEe3RMwFdYuukfhj+Z
q5u3+D/+wxd/6ZXAlewN2j7ab9fh19pfOgcvNj3aK9FiyI+crsCv4dWz04ZX
4TWb46PKNBlyP85bvgI+ojaW2wfna0uW35zHEOuRUsNa3Mv+Qr90FO4w/bBg
J5wj8bhpAM+wkYa9hgtsp4ZLNBgS38DS3YK6dXObqV5xQ8vQ3opqLlxc9njJ
YdRZ2cFNN4f7PrP3kLPgoQ0Xy9GXWZuczDXwqjjxTBp8tLR/6gT6mOMkNOLB
XQq5W8zhPS9bDYrQ92WT/5RvhWcRWfMqOLu69VYCcsKyuGEbDQ9oKbb1hLPi
F5QIkSvjLLXePfBryYprAvixIy66xshhrsh5PAu+7MbJtgh4Z23gqXHkNvXY
kbxCeGiZzifx8LSdI4bnkfPjmYv0SuE396298xi+4tfvDiZjLiazCpQVcLct
zlc/xX053mLjTZij24VzD6rh8ns1p0fhGRpJNRzM3cD+oxfq4Nq5yWERqOf9
bzKLNddQcih910QD3H/GN2UAPvOZrPKVPSVLD8n3/wzPN4172Ih+hTHZY8oP
KUl5WpZ8Bf7MKOWME/rr4yk9vggevuN+i+r/JX7a9SbIgyGbc9B3NSVerUPh
qn0DortnlfDKPYHDhaso2VeeLVedkxUpMLmDXAmmfYoUKym5MepwTnUvdXN1
cQPyeW2faUcavGJHewQDT82JzqPI81be4CYTuCv726h8+NnYwUcS5F/b76yw
/wNKkgc0VmfCt8ku/+mHeUmU0ddN8A5NqTwJXjE3VW+h/9adGbFcpKqnSN6o
gKeW1qUMw2/zq1Zthrdz7Wa6MKe7NB/stcK+y1ImROvggWPBeifYb885F5os
tYdfT2TdYuHdUE/iNanjvvbhIVGL4VPxNsF/wL3XCnVOwPdPGATows8c2N7k
hPeHV2Jc9RHqluDcHf4cOZdoWW9QV71j7emdy1Fnp0pJ6j14ZJDDT6p3z6ov
ypWNfs2tbj3QBY/lhr1ohrukVU+zHTAvMdPvNcFnrU9dPA+vnPR7ZIa+/9/7
/C++sbZo
      "], CompressedData["
1:eJx11FlQU1ccBvCgI0Yl1gUnbWMBURYJRGQPm0cQLcSlBkTUhgYDBQ2Kssgi
IosoIvXQ4A0KslRLlRpwgRmsgIAWFcS6IIWpI6gVWQQKAdSihH488dSH+/Kb
O2fO/f7f/y7ZGSYOmsZisVzwTMcTHVj3to59vVYn2uLcfguGfJJyspOXU5Jt
+Nu6UnjFpuK+tQKG1K/5JGDzKRFnPdPOgdff3arirWDIuPLwhhJzStLjlIEn
4KzprQuS4R+SDgyGCChxftWWEA+f6Ehz6Ybbyu8vsrKkRD28z2sPXKs73XCj
JUM2d/aPz7aixKe3f1gCZ1cb9pTBuX4tv/RaU1Jn0lHoA/dWBF+5BPfZkUib
bCnRrTLy9ID/PhQcNX8lQ0xiRuxU9pTck+4Qu8E1ZwQZuXjfZtqD7QlCSqoO
7I0Rwr8pXH0zC341nxO/zomS2128QTv4ZdGAnR58ZsCDL6a7UHIsbsYrG3ha
92VeMe7PiqsZKnKl5Ef150et4aMNTXk2cF9BUd0yQklvY/AZK3jpmFNyLXJr
apDr/LCaki2q1D0O8HPP587rQM7qQv9Nj90oeWOayV8Fd+LGhmrMGZLS6fX1
G3dKGqxu5q6Fz2l0/F4PHkDyhK1rKPH2nfFuI1x9JF2wis8QbaX+y9MelDQV
R630hz8yGq5SL2fIDtmTmhVrKYmJmycPgbs3v+Evgivk8VmZ8CvdG1Nj4Ymj
3tJGE4bkPgnPvwN/pwjOT4V/qlZEDRhPuXBV1MPTcNH5ysQio6lzOOsloyp4
+VEzmWQZQ96+DteYwt38j/1bCT+2VzLryFKGcDxq9BW4p8C9vesx/MKlkngX
Q4ZU6N6x/wPfFZaQl/8K/l2BvDBoCUPM7bqPdiCH+qYxzzH45oC5OQYGDCmM
0M+oQ25leSVndWZdr6WBWiZe+sjnacHxw8g5LpxR6cPLeyu3RephjutbXLmY
yzYPrp81PKLtnG7BVwyRnWfvoZjj8KDl4dXwwMI2Zwc4f3Au0+dMibE2x8wX
vilYq1W2mCF9BuveW6Inx0cqOEHwnEM6P4/zGOIg1LQEo1fj5Z2ukfALe5vL
s+EnpcWDJ9HDrNkDikNwT7O/OgvgTQYl7BL0dmHYztcZ8A3XIhY4wkeZCOkD
9Hz5NWNzJdw+Mc3BDX6jJbGifyUlvHdRlgVwrbGApV7wE3GPM3nYo2kd8Zqf
4BcFLi9r4HM8I+Vi7J3FM5HiV/gjx+hdd+E9PlH3lNjTbzd3qlRwuxku6c74
LtnO3uIhM0pKtT9OXIB75NhO6CGHme8PNofjP6Aezeu7CO+ZiOVmwU9EVyTN
NkV/ROnzz8Or7/anTOZ8oMor5p4xJdWSCd+z8Pjog+ZbMJdG1xcxZUaUXM3O
7zkFvxXp7mCLOWrtTkm6tYwSRfLQ2ckcvESLE3Ux90XeklOapTjfpDsoBT7L
1HpMDXfPYKt3wUMtDQQR8MutsUpX9EqQxHumDe9iGXFD4BzPv3fz0MOByMWf
NRtSco1bPbwFnsBnbQ0ymvJ/PgofusFFLu3SUuOpcyQLDS7ZTM5ROpKahr0Q
hp4u2wevXD+WM9kr5Yfa9oemDFkTsl85E/d/wQ67/Rlc7D/vyEXsl718vnsD
/G0Xa+QDevvcTyQeMGOI8VCo4w3k4MdPFXfB24syZfXY39v3Xzu1IbdSiYXm
KVw20vZnLvZ9e/gGjy+R8//9n/8DCpuj5Q==
      "], CompressedData["
1:eJx11G001FkcB/Ch8jyarFm0aaRNJAzLnJXoOtpdOUlFBsVW21LJRLEc6Zyy
m9KeWvz9ZY0mTZkZxIo8pLYzPSw9sSHJ05YdZR0lLDvYjP3Oi87/1b64bz7n
nnt+9/f73rts98Gt3+qyWCxvrHlY/SHngly765T98ov8EQ5NQucHhC0qo8jO
uVIlD745wdR2yyKalMaPGL7X+rZWCzZ8lcu8xdXwtgAb7uwVijy85yma6apT
0kPPOFwzmrxQP+BpyilSsWng7iBcei5TLxneEz/sMFdBkQheTU43XMXujoqF
p2oSdhpXUsTmvCCiCb6u9fNnj+EK5z15FlcpsjLTy1kJb8qTsEPgx6dvP1pe
RZF/jt45WQ33Lj3y0h8uzq1s/7Qadf6cFieHH5/ycPGBC277Viy+RpFTwSOv
LsI/ypzubkT9wrSM7QtqKPJVzNXIQrhHbXZGENzz+3CDUTj3+akj+fCb6THy
LvTH8eZ+5+u1FFGyqx3F8OtjHSvUC2lSbB+zMrkO9azweafdX7Fz6kY6/GiW
pHFpPUWWK/paz8PNw51XS01pkuBgb1IJfx2kbL4EL/xRQN9h04RjZjm16jpF
zCbfHyyD35pMPjxgQhMTwcPIH+Cb98kl2vuKeDPNenB+Zn1eG7zw3lPWLfje
fLWVozFN1pa0LpmAS2dPvLsPX/za8M8gI5r4Jgq+M2qgiOm6X7m9cOfmsCOG
hsx+iwuB94bgBwZcK/kGzPkhibGyaXhBZ2NHuD5TT84uK0sO8jASohf6sR5T
f8mrvtW28J86H+73XcDc10kzFeEB51roJMfNZ/qzcDZYugE+HpeUWjCP6Sc/
6anu1/CAt6OJ93WZ/geu2ZGWBL8R6Rmt1sF+zdmeAczLIENwLAM+qbGcpHSY
+TZZhHHy4d695nN8HSYPR0dnGirh4qCw3l0smthkdciskJ+3mwK/uQtvX6Lq
mZvLJRL9fokl8rZ7/SXDJ/D0PTUaIxaTT3eu0FsF59l7pZRg/4c86yksD47C
zW6e5tbAP+R/e5Jd0SzceNHaKiW88syT8XG8l3JFh59RT53ygnNa2yP44wmb
v8bwvrKXvQ3jwGUhb8R+LOY97rcue8mFj4uihhpYzPvdEd8Q8wk8OK5Y4Ib7
8q3dZXalFPl3dp8/D66SijgKuJ94ODWyhCIpmvmFtvDEuHCRNfrs3bY37YKC
Il01mvalcKll6owGrvq9o1Mtp8jh8GMT2nOyhT77DmFeJTc2nImGO/IE01Zw
gbeT3lbMV1H8vHBYRpHwba1CM/gbz4sufOTBrqL+sxy40CHKQXvf839Y9bGR
nzWCX9R+8BO59r9Noz8PetcrQ5G3CNoz3haee5LTOaKdY9U7HTfk06R/iGsC
bxFIavvhbxqbg40Nmf1nog0WtMC77l4O3GLEnF8lYq24BXeqL3KzN2bqSQme
qCuHfxk9WKAxZur/W/7ishguzDpwtt2Eue8O9uChdPj7JWNl5qZMfzZe81ot
0v6fsd1XBk2ZfsrcU1qj4Jkt+jkbFzL9d3f6QugPl6rciqzxzxQN+7W7Yl7/
9z//B5976oU=
      "], CompressedData["
1:eJx11GtYk3UYBvCBwjUJPCSglxOGFciQEA8oTsy/ggyCK3TijAUqOoQhyUJU
CjxwPoro6wt6RXL0wCGXMmCh2bSRDFHAOUIgEARUxsmYkBLWzSc/9eH98vuw
67/nvp9nyd5wfpA+g8HYgG8GPgHjizkX71crLmWxGFadNNGSy7s39lHEWb2V
kwH3MzRZFgd/+tGUjtNPEZ5D6esTcI3eOt0RuMG15Er2c4rkD0T4RMInHwZu
DIOzDo15Gr2gSJxr1FcH4ZvMKI7uT5oUFSXPYrykSDFbowuBB8SxsmPgPzkx
fxiYdkn8qX3wWY8O6RvAvXvMHDUDFPmeUVyxC57vvOLVqQ6aBNVZ58i0FFkz
5BkshBd5BOxdAP88ifcoZZAijNKsF/7wdydEUavbaSIZS+3eNkQR4w6fxD1w
L/t4it9Gkxiz4xdmDlNkTrREKoLzlewV3zyhybMdVdbl8Joal9dieF7IA3VW
K0041+uEG0Yo4l30UCCBG43ER1//gyZpx1TOcvhreVHCMfgO78P3glpoYn6U
q106SpFUv9sWSfDz88aiUjQ0KVO7s47At77k5GXDzeOXqTc+psk6/X/rCuEN
JbMlBXBDS+NMkZomfcHuXkr4BsNbtjL4/qNdoxaP3vuESeOvd+CWZxet3dJM
k/79Ybof4ZFrVdIWePxvXQWMJppEu1a6x8J73t063g8PW+k7ymmkSVfNos2f
wfn/jE9MTOcisq71fUgTP/6Sm0/wvwTyLu7chmqFaTHNnf+AJhfNLzJD4UGJ
WqkFfO7f2RPlDTQx87qgbsXcztslDq+AJymf+bbdp0mzdDDCBV4SqPHgwXlp
oZVMOC02uZmKXB47nI3fCa9s/mSyuJ4m7RE9ZQrkyFvd9EswfNMtI7IJzjXc
lzuB3Hu/1sz/Fk5Cdqd0qmhiuT/CaSHcfH3gRBq8ual1Wwycn5lY+Cn6M6Pw
YF42/ENJwVQS/Gyj9RxX9C1Z/pd7IbxcOHj1DNxm+fboL9HbC/rj0dem37lK
ocmF6xmLb+5Fz/3GAw7L4H7S/j5TvOdwF2MyDHuR9MaJpYCHNnSfZsPz6uuD
4rBHtg+EibXwNx2pFjnwuy+XNOX0UuSuOMpaCV+d62gSjTlomXeC5M8octxK
EFEHtw0tzX4Fn1JV2/T2UKTbc0Q3/TvKn7lxjzHnSgOz+Sy4p6hXeBduZXdP
XIVcxP5SQUg3RTaX2K+8Da9VuzbmIMeIc+paxVOKOBjNHJLDjQ2rHKOQ+zWX
gIzl8BuZCek34AkZZpk70RObvD2lsi7MOVZpUgavUgmeO6FXRv3D7K1wcYJ4
TS68LLiVfRI9tFxsPqQHn72KvzkDPtxSvF2F3ursu+fVd1LkaFhFeCzcQ6jI
K0bPuX07M6/CrXjtsoPwnPRhnlbz3l0OfLw4EO7uls9UYo/c9Gf134YfGNtl
6w0X1KReZmIfKy1CaC28tSVRtR7+Qblpeyf8QLpb3Tq8Z9cLvw676Tlwr2TI
sNcn5M9tKLhRqPOWhXCft/KqFNwB2cAgczbmMBlewjaAN0kSAvxxN35fmm9+
Du7RNu4zir1YzBTGOODOZKWlX1mPOXPKpa/a4DPHTy9g4F6deaInGoYPOoZv
uQ9nLYj0EcHXZo97lSOvt+kjMTXwQ5fqY+/BDU8tTD+J3L8b6Ksoga8cc6mw
w/2MPNeh8kVP/u8+/wcZUhl0
      "], CompressedData["
1:eJx11Hs81XcYB3C5zlG5HR2FqCY6SS6FIj2k41LZGOsyhNDSpEw7WqiEFl+h
fiQ2MRnLZbq+dMUmpY5bFJI7yyJtpVoX7OOv7Z/98fvn/Tqv8/1+n+fzPPP8
Q90DpaWkpFbhk8G39HiufJl+eeX2hteHbi/gKMXy7MQsY0aSrv51+XDeYGKr
6cccJc0NCDBfymh7mULHKXhzqVxaFjylsiA4woSRwKh35jF4qauevJw+R7pN
7yubTBkN5meIj8D5J6LjJ/F7K53AMhdzRomrPfIOwtdFfWh+C3dPdhI2LWO0
Izdzxz543LSHkpfwsey25bssGKlFZNWK4eM1VxcdgKu7GDrxrRi5773A+wau
0K4TwoOHGPrpVq1gFKzdaRQOD/L+kJ6Gd13ojbwWbs3ovmmHcQR8Syjf9/x8
jmQnClPnrWJ0KMHWNRKupGJr3zCPo/Sa4dzbtowKjYzSpu7ZV/907YgeRy+0
Dp4NIEYselgQB/eXG9+vCO+ubjMYs2O0/4Ti5FH4ynwreQNdjoJ82tVC1jBq
/DFIIw3u1V8U9lAH///i7qI6B0ZefYN5OXA9u6418dociTPeCNVEjJaMPr5Z
DLf98+cECy2OHpzoPW7pyKji3K3YK/A7jzx9n8zmKGzS4XsLJ0YFMYZ1NXD9
IkFehiZHn3R4umo6M/pONmy8BX79jXWBi4Cj399tEtTAdfniq/1wh+wavXEN
jooj/R09XBhd4YfpvIVneGzptuNzVL8x6lgpPDL18xDFhaiD8abCV2ocnd/g
Le6Di3ZrD/HhVSLBL5r/8eplx6sWwp8k2VTUqXDUZz265SZcmEhJK+AGCXEZ
h5U5UgluUQ+AB5n95egINwmLiSqcydH99Ihd/binqdkjiSc8/jTx6mdwdGrx
vvzl8CHbe6qBU95lbjE2naPmj5LeeaEO+0sUBsLhNiWGrXPgWt1HLwegbpUx
R386ANeQfR5bpsTRWlVZvgh1Vjd5yEuES++uKxfBA6eXeMisZRTCmXWlw69L
5zp18pATS++qAvRR+lli6A9wZ9/X1/rgi60tNU3tUX+zZLcz8NwBm7AhuHyn
m28hcvLZdddjRXDR+savR+Hvm5o2Kq1mlN/ZalkCb/fTn7UN56pU5zkGIoen
/ISTpfAPW79taoW7XmZ7ypHbvfdooAy+LUCSvx7vCt4a/0B2JSN+d+ZoMVx+
p2XPUtRndpL/xEbMRfTLgvZCuLDWfJo66nnxpMHIOcyRcr5b1tQ9xbbeUq/h
q+4V9fCXM7oQ96VuDvxA0VNJG/rSfDFjaxzmVMZ2j3kW/JXigM019HFzhcr5
mWbo+6YY45Spcx+r9iih70FesR7F2AM8vfLMWPhXoSYzvNSRQx/dlB3YG87K
FBMxdZ8vsiVFyFWDgv4CB+yZ5wd7anbCN8jMl3uPHHry5jfbL2HU7dAZ6QMf
1oi87Yzcepof1vYzYuTwwG31p/D0M6crTiLnk0OnUy8tZrTgRtgRO3i0gqP7
IObi6exMbX34zrk22kZwqZGbJ3MwR5r1u5rvChndEXf8pgWvaVRuHMHcSfI2
/5EOl5gM2KvAk32Sa2t1/vXNFc89JjAXF7NVy/iYa73McKcbcC7zStTw1P7p
EbT4YA/43MlJFeDcNueEW53wOcuKx1Zgnww/U/g7DS6v1mrVAB8TPtnGx/4R
9f1avQbvur/9bG8FvOWwinAUruk/cmku6vB/+/kfhheiHg==
      "], CompressedData["
1:eJx11Pk31XkYB/Abt4mYU+neLtetdGW/JTUtVM4jTLrJUpOxFK1ozmQdIqMh
OSrLh5nvpBrjtOHUKVPKCc2ULXXJUjIU2bJESKqrEPPuD5gfvr+8zvd8v5/P
87yfZ9GewK37VXg83no8qnis6wanWWkWFMfV9bCqEY4m3V91PZAw6mja+9IM
XlvrHbv8HUflfGu1gwsYbWnL3aoPzx0Ua5jB29I2XTfWY5Q9/L5GABdVjlxM
gz+YayHqXMSIy+3W1YRHF6TPOgn360v95ZI+o92B03zV4Wuubw4TwBcrtvT6
GjAyHd1Vz4NnZnp91oT//qElSWrEKG97deKkRkHx6feXytJxTrswq4AaY0a7
Hgv2T8DbeS3ZUrhz3pHkUFNGkvzWZ2NwR8WTOblvOeoVSw0+mTF6NjYrZQo+
XJFyo2qYI7VzGy2DlzBKXzAxycd/eZ3Gyr43HG1zjeKXLWU0e8BeQwM+ERFT
qgaf1+Tp8MGckQfPRTEHHtw7ZWI8xFHysak4VQtGX8UvsdSBpz27KHMY5Kio
Ieb2ILzUreNHI3iCyv3GsdccNdxR7v5rOaMoPWnEMvjp5imSwFUtlRu+XcEo
lK6dIPgM4YGkuj6OfhKJtXPgdxOyHF3hym55SPwrjnbu2dH9Ei5z9BDvhbec
NaK1vRyJ3Buy++GOGqKSQLjWsOKRTw9HEYfGMivg6jY7NsfB1/V2SUTdHG13
fuQYCn9RYt6QDp+UV8trX3LkbWhvPoBzrqzMkubAZz48u1jZyVF1nlxhDfec
lmx7Gz7dvzR/PrzTqfaML+57wnWhQQ08IsTvdWM7R/lDazv2L2MUJHl+rw2+
h5WX8eDPhYf0bFDPbndPzRE4BY6vNG3jaNzF7/uP6MvuzriJSfh525Q1pa0c
FcpGvkqVMXJP9Nym9XVBcW/q59SBFxx1O6lazEF/r4lnOenBZ2ToH46HzxiN
qg8xYXQ0J8F7KVzp3Rq2AG4n2Pe+FrnyTS5/uwpeKTfqMYDLirScBYZ43606
yh5+qrBFfq+FI+Vq/cvuixkl/hpr7gQfKDRorIDvTeXbnJKiDhkHX7vADxQc
DliH7wR/3NBUj7mIPZAR7AE3iQmqt4Xr5o4HiRcymrgq+ccL/rdVkXoh3CHE
ISlgPiPR3ayenfBIUbN0Ke47uzTDX6HLSKemOdQH3iTJvnUBbn36VZWtmFFx
dMd9T/hMs+na8ahbxsTojSptRt5uYx7b4fx8YaMv6qwSWdMeKWJkMLdohzN8
m0WYtkMHRym3drrZz2P0Z0jheQf4psT4fGP0UWuoX7BayOiPOk0TG3igzDZX
DXlw1WASZwGjc05p/BXwZgGJ3LqQ84onssS5jLRfGL4xgZ/cp3/lG+TK9GbC
caUWo5977vTrwCfDK0NikMPQ2s1Bv8FPCKXz1eE/+CqSqpBb3SXKDne40ObM
d5/R9xKTIw9ykPMEn5I2Z3iyqV1OPzw0JYbeYi5Wuy0yCYfzh5LFjXChSDpq
iTl6VHgsWgG/fEEnuRReGb2qJnYAfSw/numCc4Z3XXe4+WWuT+XxvDCnarn2
D8fhHd5y8UW4ipncbwXm2rzTf/2/uG9obN+6JPhGVZlVJPbA0SsL7zxFffz4
BpmH4euvOPq4YJ+U8NtiP6Ge4YeybPzhalYPWw2xf4SjiWl26Mso03D1+rLf
mtc2pcAf17T3XtVBblMC3m2E+wb3hO/DHnv62LjMDH3/v/38Hx2RxZ8=
      "], CompressedData["
1:eJx11Ps7VHkcB3CljJ2UBpPWszJmXUKFJJmJvgYT0kSki2pilLYbpSERrVuU
7KFOF7LmQTYyxpIuUju0UVZjsyRKcskmym6p1Vq1b39AP5xfXs95zvM9n/f7
8zUODlu7faqampoTHnU8G8xqRW/Uris1xroo59U00VEIDW6ZU+RXl+C0NrjX
kNItQUSTxKF2ffF8igxPmPrUwasz64vXr6FJb6M6PcuCIqam4QfL4K2z1OIW
+NAkXv/KwwdweWdR1yn4n/7RWft8aZJe1fHggiW+wxF+iIMzQnKiBGtpctY4
YVBqRZHThaxHYXCDrFHBJXj1XEV/wAKKTLTHRG+C5wsab8f40aR2/MAK/kKK
tL4N6PeC+7pvM+6Br+spjTZfRBHBp3QRH37ymGai0J8mw7vTKnWtKfILu6XT
Cn6/OXtqKVyHYXzyP3hdnncZFy4JXP+kAH5/8dGB5zYU2VUyUfA1/L1ubEsO
nKFX0F1rS5H9SX9rseBVliyb0/ALPs3nzi6myEtd3qAmfOKY2Wg6PHNstk6w
HUXyZ2pJGfDvPuqHzYX39WcOGS6hiFhhd1ADTt1U+RTi/IMxJFkF7xVY7pj0
vdapTFt4jVVufKg9RSKz1S1nwrlNVtk+mI/TnMTDb+CJPieL9OBp5m6qcMw5
Y/8nqWQpRQJvNLAN4WW2KzozkUtM/9oWJTx04aI9ZvCiNQ2VlcixfjxewXCg
yK13mU02cGnl5ehHyD0vr1y+BL4jSWklgKemBDqloyfyyw0LVsFzZfZHRPCr
pKCm3JsmQyWvmlbCqwW88SD4k39mB+9cRRPOPZN9FnDHzRm2kfBI3fYSjhdN
eI/Zsz/iPPz257En4NO6NQY7PGjilaxXKYez19XczIHb10r7psBvb58R5QGX
SMbuVcD7je4qaoQ0URXL+A8xB75BwsXGyXyXErtD7jTJHnqmdIc79n1e2QXf
yYryynejicmWuoFCzHmZR4rsAzzs4npDsSv6I2WsHkVehjHvoplTritvSFuF
xwXobdoZXXu4pr724Dx4fpfnaJULTd5nBPFCkXvpD8zf7OC5VW0avYQmbted
jTPQk3hrwwkPeIWjTDwL/uNGzfly9KqzP2nzVrjJ7afdvBU08TuQW9aAHkYU
WLzeCy8N1t7f5kwTbS1N6RP0+SuhxZEj8Lt/+E8JhyuFbP4I+l8c4vYsHR5y
VRnKhDevKjnKgOd2X9Q7A9+jyMvQgfeMPJ7OxR41+np9kwevNvg23QCexj7H
dMHeSba9UPwEzxof43LhfXObRLuxp75vU5sVcIGFrZ8FfK/W9y0y7HWIeMTx
Z3heomtjLtw8SsEewP0wKh/mV8ADRuxfsPBf1+bFHrKH81xPBEz6llMFdsnw
jpd1m3Jwzxx/03vuMtzhQHjJdswnplG2WQ/u5+09pxB+Stszwh1zvlfJjZOZ
UeSE57LSHHj7cGyYCXIJ3Opp5Ao3yxHsyIJ/nsYrnoocJYVczrgp7g2/O8xU
uHzQ/3U33HJGEUcFn3F+uCJy8n2nLkdL9ETr8IDsGnx5m3jTrsn5zzS6OR29
CrrUwamBq0S/vw6Ap0gzrUUraXI+IXlbBzy2PLRcAN/gffUVjX6KO991M3Ge
5RvfRyyER9VG3HnqiX2pYBmtg//lp73MEP44iHp4Bf1vPaTaUg2vYAVxNODT
JZ/U/8W+1A851CzGHL50P/8Phx2ecw==
      "], CompressedData["
1:eJx11Gs0lHkcB3CNTRG5NIdVpivOdnEZlzEu8Z/ksuVSZo51WXazZqVMDloa
dpsyM5JrT48UIVNr2cq0dDFGQrRqR0VsbK1YkeNENu2pdDn269W+2hfPm895
zsz/+f6+v/+amMQQIUNLS2szHm084cKxXzoNVK1bTSPrk7Mo0ud2eUzjTBNP
B1HhVfgnsuFR/yMU2T/rUpfCoUn559FD5fDxBvcbedkUKRljX7R2oYmHSXh1
DrxWFWAZc5QiM8+LHo7CdxvrDCfBQ1wd7atyKGLpIrBXcmkyW9CuEwVflRoV
K86lCNeMMZHvSpMYQcEmf/ghC72TgXkUEUZuV4jcaJI3eUHjArerdwu7DLcp
vpUd5k4T/1qecC3c1mv9uiP5FLn+qDvR24MmMrs3DBP4vQzfkUl4+zpRqO1m
mug1r6vSgdfwUopH4R/2lnqYe9IkcaNP12t9VavPe8/GbQUUaXAfmHkLF/gs
fPsczq8427MFbsOvqxnwosmy4qnav+D6Xw8FucP7DC7Z1hKahBoYveiHrw5j
sJ3gMzeHZKk8mqipwuZe+PqKf/a14H9zdXdecNpCkzC2n1k3PKOIZbId7qAy
PDsOf3anyege/BY9O9qP771kFakn9qaJqdfJoi74lbtMlhDecfhT7zm4fJpf
Pf87kvGIOhny9H2yIyB9K03M/RvCH8Kz1np8N59/uii4sxfOCly2ZhCe7xWv
3Yl5hd6UPjX1oclPtfs/jsKTO7oiJzBf879fV3rDm447z8zAD0SU7a9AHxq1
L82FwxlanCwG8tS2Yl0XoD92y/ndQXDxbLDCAP5ql0WyWE6RNHlpEhdevbk0
ehXcek4UwJZRxLnmtkwLTox32jvMz6XpyvREJkU0j9mu9Thnm3e8qS98eaw6
8txhigwd1RQEwv1LThiGwguP9J3WHKKIhWm2/AFyUGrKVibAmbsNS7IkFJE+
CWXx4M13mQ2Z8La2tFneQYq8vFUo/xE5Z7VHfXMCrkhPtU34gSJNz1QFLzEv
01O+lkq4auSPleu/p8iwiZr7Gfw6X3+0DV6SdGK5IIMihyvjR6Ix98G4EpuH
8AHda2JJOkXKvt32ZQ560rtFnPscTq/urbwgpgjzlCb1DHrlWNQepbVU1bo4
8Nqu/gMUmZIIW1vQT2VGt9oIrlqgKEqCK6YMHvWhz5kb2xRr4BG63hv14Y/1
rrY/Qf/fnZ9NZsPzUjRvqtIoMt1jrD+HfVkxmXzfA85Ma3RRwlfzRZZMOFN9
7EwA/OO5AYY/XK/7/V4n7N3tY6xnoXAL884DwXB1bNfIV9jTBe8sz0XCD46v
LO2Ad+yZyi7GXjs9XdwXC6+aPuOrgSeKE6IHcD984C18GQc3zI9y88I5G3/W
pe3hNUIP6R74iraKO/Xw35lWb2jcM83hnAXz76e8CKukkU+ImVRtAC8t64+P
gUf7LVGmIM9tvzkPljvRZMmQ3C8KXmKx4Qs+8s+ZzrdzhTvlX5TPn19UV/6A
jXm9ajQxfu2Ifcy+qAyC57b2yAwxX86YUcgd+PQVHtcHvqGBpTsJ97wfvPYG
vNecx+HAp6QmXI7kPx8M9yyygm/yi+BK0DfOjuN1Q/CJ3sw/l8HLrwn2daKf
Nqk7K81wHiJ1P60DP3902E6BPutY374RAVeECFQz6IO4ZVHPuBT5c+MTWuA7
HOiWQfjSQ1urbbAvkl/jxG7I4f/u538B6HGsvw==
      "], {{-5941.146027753306, 2054.9840216062535`, 
      1080.573106205265}, {-5941.975989300661, 2052.4847513178274`, 
      1080.759630159369}, {-5942.955600512717, 2050.617480977466, 
      1078.916292881033}, {-5942.113774761719, 2053.925449704846, 
      1077.2594046197883`}, {-5940.675619784844, 2057.8401039475816`, 
      1077.7202684736742`}, {-5940.341386458501, 2057.8404458719315`, 
      1079.5603742551416`}, {-5941.146027753306, 2054.9840216062535`, 
      1080.573106205265}}, CompressedData["
1:eJx11FlQk1cUB/AQmmpJqRIojMWgQqPYUhRBKW7cFhRUpFbHUVBiBFNhIAFZ
S0S2oQK28kXxA0XREUEWBVwnCgikmoKKbShrUcAlCJiIBCgSBNN/nnzqw335
zZm7nHPuWRAYvlXIZDAYa7CMsXoiresH9bL6IknwRXUKTWrSNtxXjFKkRTFH
2wu37eM9eQ6/7RnSf+Nfipzydde3wfk9JSP/wL0L7RRlbymyVRTN+hu+3N+M
EwS/NSAtKNVR5JttuzmN8CFJ8SJ/eFV6ceK5dxR5Zcfbfw8+oJuZ9SKZJqrU
Y6eypikieiLprjfsP9rBEcEdrHaMxuop0qkIVtyBTw43TI8n0aQ5Jk9YyJCS
K/MjIuVwl8HMMnP4/ZEQ6SEjKeEqOqUKw33qDt5ZmkiTWv1eySamlHxb3sY1
nBuQ3mZbc4gmAYPCuybGUvJXhPjwfXggN47flUCTSzt3s2/BPVOWGLfDLUM3
X/j1IE0WtVhc2/yRlMQf2GPWDVdVC/wuSWiSVtG7oxmunM5y6oPH35vBaorH
PfuEU24sKfldtCxhDC7dVvso6WearJTJ6zLgl59+mf4ezraZ6VQQRxMPegFT
DueUdJz8hHGr/pl5fJIilibM4v1BvfBO7WfMuXDnsklOQswHd/ZLd1gMj9ou
/u58NPLmGfu8EZ4Y6RbiCmdOqPc1RNFkyEMgyIVruudofeCFyVp5ciRN2vw/
zveGc2+4mgbA+WzhsZIDNNmxsauzH+966LY0RGzwOO/E5giaHMldmb0fXvvE
zV8CPxIX3xMM33my0qcNebOXuNsdhU9dr5ydHY6+WunAcoIrHaqq8uAyLTX4
NTwnNmBuDOoyORz2ssSwj3HqeoWYJoK7+cfLUMdVbu9bZPDXGvcte+B1uYX9
Taj77FnHWA3w6h8FOToR+k1mNKFEnxxsmJ+hhAdlqM2ZiN/g/PisFn310yyl
5Cl8e7WalY/47qo5acwpihxlOPup4Cq2lcIc8WPn2cEmkxTJDE4VD8GrllxU
zIUXXh54YDpBkctndoVNwGvMPK7y4F4al/LPxykSw+RYT8E3q/LXn4Hn3V5N
LRyjyCyd8LQevuWF+hoH7z3zFfsAGaHIJluLbEN8F70lZxyuqtyXLR6mSFJi
KtsQb1+qtApDPk+vt3GtGaKIPNvs3SQ81Cpm9g+oizr5pLXNa4rU1ZtuHIff
WEskTqij3rY85ZyaIueq/duH4TN1hSvMUXcud2Sx1yuKOLT3HnoFX7vKYtEY
/NlDozbrQYq0hvuKX8CLQn2iW9E/Xuv8Ui0GKJJikuXzGO5d2dB0Hf1m2bjL
2aOfIqfVHYwmuHB+z6cM9C3JfFdc9JIikcevjsrh/aYpFd1wXfuf85bDT9Sa
11yBu9jZuNrjXxhFCnZN9FGkLG8sqhDeJHPpMpZ88AfPYr1y4UvDNVd64PSM
Uj4P+9wuKn+TCi+tjhYtxD9lFVzjHYb/cXR5RhTc6m2Tpx6ue2t51Q73LGiN
4wfBHU+MiTrw33mmdkn98JvvW9hb4dpWzW8VhvlQvKy5D+9V5jSMu8O/b9w7
nYZ5EtgvE5ogbxpfx3XO8HkVyjcauK117urtyPMvNV9kzoM7PY9XyTGvRiJ4
lo80FMlYSPhmcG4rSbPH3Jtx1tEpEHVc/ViQwISvuF6VlAV35LB8rVH3mzFs
oQ5zQBu2xv8w/EIAP/SlliL/N5//A0uR564=
      "], CompressedData["
1:eJx11Ps31HkYB/DZXFtRuR2XJRu7czaLyqxaqT6RSzt71m3axkbt0VJ2N8kS
mySblCxD32G+2cxiEbLSroyOiuKUlKnOVKgtWxi5DIMZM27TvucP2B8+v7zO
95zveZ7n/TwfRh4KiVrCYDA24+ngqfW9rxefF7V6dB/0ODNHE9OPlr7KUFDk
yHzhDRputuP1vuXwA8dYgqNKivyScITNh4+EZ34smMX36YVOUTMU4fVd7OHB
3RoH3trDA7inrgSqKJLl5RyVDWdGLNRUqmnSvsDY4qmmSK5mTVku3Nqv1LBN
RZPqS2Kh9SxFdKjMH87Be6sE830zNOkw1khk8Hoq6FYhPGpZ/8lFJU34usKX
jXMUaVyqv+6C9vtasw0fwCvu8PJi5inCmNELqoarnWNOPZymSUi5hdXKBYp8
vvmz6Hp4X59IPTGFusQWO8rg5vy6pha4M9f/QsUkTR5UlTvaLVKEODJ17sEl
T6yjOuQ0MU+UXEyH6/FV871at7XuSJugyQWJmPECXlWgkzII3/nSWVo+TpOC
2H09phqK8G2Gbs/Ci0yqO/bKaGJkVOz6CXzEliUwKRK1ynyyt9qO0eRpwrUw
FlzuI5myg981nzq8fZQm7D0ndOzhPe5zB1zh520zxLEjNKkzNto1if/a3Aqw
9od/o2fBdR2miWZTKKcO/rjftT0M/rx+neHutzQZPh44z4Hn3nuoiob7pMwk
KIfQT/2HDs/Rh+++EnWnwBc7Tyc7wTdmGNhy4HWvBh/nwyd6Iyo4Upo4nWi/
KUKfy4xq7pbCm//QnZscpMkJPbmRHrzJ/tAXf8Fv+pv08+ATE+Uu/phXzP6K
TdfhCt/SvUXwM7Knjscx39VbRVFiuDItT98THuK/oU2InKxf1RXXCx+086r1
hbPt1ka2IFd5N5oMB+ELA3HSYHiJQ/LTJ8hhp2lu+jB8qtvRoRP+bvrLgBHk
9nZZdYUSfjs0dvwZPM9kN0+BnFfkbrGbg69iHmMGoq4gy+3vvQ8P0F2bqILX
7M//1h192KC5Fr9qmiIhP/qNzsPPrWGevwTPvX+9nzVFEar70wY1vLGtx4uH
PidqrqbtnKSI6/qxcQXcdyCcHY+5pPhc9UuXUyTRUvCbHN4y2tjOwRzPpt7b
2DSB/NcyXUbhQ1UqMw/M3co8OV0PHpv4aK223sg1pmJL5IRvv1cZMU6RreY9
6mfw7JgHEZHIVeSMYesDGUXUq3PuaPu5azarjSCHcdNZr4PhIxYs7k140vg+
TRZye4U9HSYbo4iumJPaAB9zzy6PRs63x3YtUnBBtLRHO9/Fq25n/8RemLFd
nMLhR1eE/poDF9ZILk9hjyyK7bt84X2JnCdpcN7B0G33sXeMN4WlnnAr3lhw
DPzvjf+uMMGexltUqoPg+V0N3Vy4o0ueqB9esPtWZSo85/dZb22eDaR+7k24
Ay/i31xqhScPXaFYcOs9z09m4260qN8Z2KAuj6oOLwftHKuGzCJwZwpCcqR8
eFGIJ9cInjBQbOyKu+T3Nklug76NbPJOXcA+MiTDfflwTpJbsAi+vKPz9Bi8
ZL0yUwEXWgoNv0f/290e6/4Db6+XCr7GPbRM8hxlYo5clp+iE842rjRtgscF
qNRy+LKjRTLtPXlnaLW/Fr7S2CXiEfJwevNP0gb4GbZBcwk8/fBc62XkJ0sk
DNbeqzslS37mwxVhzYFFyNv/3ef/ADTGKes=
      "], CompressedData["
1:eJx11Ps31GkcB3ChbYcmm5VGuZTNqFxSimTiQco22Wo2WUlNjLZZkYo6bc1m
kihrYr9t++RoaEV0k0sNI/cVZTeXJiRJCGd3E4uksPueP2B/eH55ned8z/P9
fD7vz8LAA4JgTQ0NjbU4WjiczDPSermiXLJPFXmphpKb95KnEhsZcugXFr8C
HlZ7hD0Ffzx6nGfexJCdpp2xefBK4zWzA2vhDlszS+ETpZOO6XDXtSxpNfwu
o8fb+4Qh65bmH0yGHw2Y5N+HZ+26MclRMcTa5ZZpEvxMnWd0HrxyIeevZjh/
dbtLLNxwqpGbBR+Se2mnPWVIXmFwfzRc1MPlLoe31Nt+GdHMkOendiRFwWMK
PS2UeOfEV4oErxaG5FxUOkngQqntIg/4EU5Xo3ErQ+zWO46p76+3jtEXPqDE
b3HOnCH4mNf9/NNws9lXNSTVlJR8b+pX/YwhzWadH+PgvQveS5N/o0Q54BIu
a2OI+R6ORwLcSGKoVVhFybCTbqzvc4Z84bBacQme1M5O/q6Skpkz8z112hni
odm9JBN+nh1lZ1JBCXcHR7sEnms98vlNOGvVnBBeOSXi/h0XRS8YEvHMeqkS
Xrz+tXKwlBJrWXbDKNxEFh/SAE/QnG+2poQSViInP7yDIVzTPxd1wm3mnbUc
LKbkljJR3An/91HUu0F4o8M7n2tKSkLi19U6vWRI6o30rzVTFeVBP48Y7i6i
pMnZ4GMs/EJ0m88cePZcVhKnEO9MFV4vgLf6RoQvgVddLLR7co+SULfpmnXw
jmP3DF3gTVryQNldSopsW2ya4U13uIHb4DU1ut78Akp+mm3fXAZ3NTZaIoLr
Rgy8iMinJOUfpUAOd+pTZpyAryio1V+ZR0lMThXfH55nESaXwS8UnRrYnUuJ
RPE3h61+z0F5WjLcbdJg+vgdSta89pVl4X9jRMK623BV1xUrS7j/srEbNvBf
T5ofvg/37Vr7oSKHkv3+9PI11LPNbMq+Du4sEAwEwB0qNjvqwgeVttI2eDur
OOLDbUpKtQauBqFfms7i1m74Q+FiR23cf6c6ocxA3/O/3bjsjfr9QavH2fDM
xPL6PsyJLjtMfwx+OEHlzYEXRFcXmsLt3RWHPsDpOHfTeXjedUnkRswbUxLS
NAn3nvDY+Anef06/dfkRzKfzyfjl09IU5SkyKy8J3KJM9eYK5vyUICZKfT/I
1jJAiPocLDic/QdyYWuj4z4Ofxuq99Id9dy0vSJRG85+Ou/8KHzURD5sgfr3
Nmfz7ZCv3vlbKwfhTAjpmoF+Ve4U7jqEPMbxRrp74JKM2D4f9FfQXxdegFyL
6+8cbYX7bDngl455MK7SyZkGF2asGnoEj9GblfFWQcl0Pt8zGPtBNjrRrq7/
vhTHAGfMW4vnDzNU2DMCLwNndb/29r2SnsF8djXeGvWGi7XtT8rhJicMohow
zxvcZvE6GhgylB2WnQAvP3vX1wjz/948p+00PF3rWdFx+O091z7dg7z09Kc0
usEfbnnME6v/K/2BRVYZJU7uuSMseKS02MgHPhz8Pu4NcqezbGvKcD32T7CH
mMBFCXO/WYH89lcsCO+HpyY16C9V94Xp3R+JvNf7SWun4KaTNQojeNoC8w18
7IfNoa7HrPB90bbLymnwx/6+aT9inyTZWW4Pghtb7S96izz+vlNUpt4/etXC
+Fz4ubjPUtvVud67UmGPfcXb8IpjiDr8337+D0tNBd4=
      "], CompressedData["
1:eJx11Ps3lHkcB/BxS2uk1KIyaNybytkynVI2X5WESUhXUqTaREyKtpHjUuO2
XZ6nEXLWpVwa44hJ7QxqhVJDVLLdTK3VNu2U7bBlMdS+5w/YH55fXuc5z/fz
fD7vz5cdERu0V5fBYHyPRw/PYoOT6t1WsuYVj3ldddto4jJ7tqmFJUWErhmD
vvBfuOnM0e00yZWITcXwlOPmUW7w4y+n7OzeQZPfaA+jDSyK0F7Fbk7wV0o/
TkUITfQK7jQyrCgS4aS0nA6nN9fcVYfSxK92oWMbfJX1wS4NS9ZsVJIrb95J
EyOuUkJZU+SqGR01CO/mXDd3DqPJwNf70/bbUGTS0SX2OZx35e/0Sa3bqP3W
zqNI2NqRAwq4zz+r9kbuwnec69c5syliZMGObIRPtDV4dcDbe9mexrYUuVlV
kVYN77p0xnPJbpqQ+Ot9H+A2QQ/7L8EtL899z4HXRnxMfGBHkRqmu70Inm/c
okPDnS8G11fZU+T93aE95+DLFQd7WHDr8sTrSQ4U2bQ5UJYBX9/5dJk5XFCp
fL7OkSLXYuT3TsJtY6aKS1HPvVqLVgMnigQ8C6lKg7v3DMsXwbdXt4vq4B38
fxekwvdV63s14H8p46FOf2eKZJUuPnMKXmQdOPYMffMwLTmohPfpnZZq/Un4
cOs+eJUH42jwfLz/JqAvB67jruoRov/BkTlCOXzUyjU8H67sy4tyw7zadNWt
+hyKtATxEkvgZV1jQzsw39QkptodHuStuCLW/pcDN/AE8sCanOEeAr/xfsxS
Dk8rEtotQX7icj6PhsP5rwuO3YebPI6pVG1B/araXZvgX30jJl7AR/JqmUWb
kYe7Tx/Ph3u+ky94BzfgaeJbg2lSmtT1UY06V/uGFmpzUth+aYNgE03uLLuQ
KIIf3iJhzECuXDOz+l2DaKIzryTLCX510I3jAK8Tab4ZDMBcwh5evoL+HK1W
qJbCPa1C7M3gnxYVVZjDA5JrpTz4xv3Jmd3+NLlfM07x0f9rmrrfw+CFMxPk
oxtoUs9VDNzEHHN6SQ0f7t8xFGELX377S6EGc+9f5TtyEi45fOsHHo8mnUf0
5rjAVYPN5/PhggvSmgQ/mgRke8u3IT+NDfnCavihY43cUl+a1M1sSRIgb1vN
L2Y0as+dprN+Bdze9a+0POTTLD5ApID3Nyj29PjQhGPGPSBFzqfsYsc+h4eW
h2+PhjsO2z54hL0w+EluMgB38+eXx8NZC2XfDWGPiiVU0gd4ZVl7x3H4eO6o
qRnc7hFb8Bl+NKXp51R4rIhvsgb7mJU92+4LvNjQ0CUTfpvPUR3D/kazur/V
sZY1695Ik03AO5TScTnugSPevVw9eOX0tWVxqD9bKHYwgaeeENTrw3mvK8zf
wG/VzVLH4T55eyh7GwO+ekby/jvoT1nBpPcfcymS/PqVgQbn6uucnyxHP434
TFYU3CrjSP8n+LTsl59Oof8lns2GBnBByprcQbjm7TBzL+a479VtVcMciiyf
uJX+J9yxPuH0mo00KReP/5oGt8pcMvcpXOjMPHsWeTBuM46JhEtGCkY64S2U
T3J0IE0YunHnQuHdK77QTfAXn0VfbyBvhlFin2j4QK/cRqKd76KtUgZyu0d/
pUwEf9g6S6yde1CMrNgHObcoKWx6oq2nsebHbPiHYi+BHfZiZYrHsgWov0J6
gJ2g9aaGsaitNFn6KH9qHvz/7uf/AK9Br88=
      "], CompressedData["
1:eJx11PtfzGkUB/AJua1WarWt1ithNrfUZC2il2NHFLaLS025lOiiJDZlCqlI
zeh5uky6KMtONUvN0G2m2+o+TEm3KRTtxIvY3dhaGmmUPfMH7A/PL+8fnudz
zvecr5l38E6fSQwGwxbPZDx61umqD67ltXEf9X+xaSew009HEhBFISeoUjOA
Pm70dfubDgJPBLubTWMo5L8yftiD/sa+Kq+uEz2heuif8xQsvX4734LeIGxl
X1YSoHul3zTFUvio06fbgN7puUrIQ183XdNUHkehsbKULUV/f9Fdtq+LQOrB
c+E3eBQGFWWNBeiFwjcGLujue39cknGJQteabb5C9OPpR9Lt0PvkLw2jCQVn
6ZZpmeijdy1MbNDHRlomjiRSMBTN8UnVvvv7kkhnfLeIuW+BTTKFs5paZhK6
96Daww/zr/nVdmwkhUKmXdjuBHRriVVNJNa7WlmzPSuVwt1LcfN52nqnzXuW
hv3pyeufY5ZGwePV1l4+unh8unlHKwEF0/p+TDqFGydZLIIeKKxYNusBAa7r
vo+NGRTr9mVr7ze2ZJqYtxCYxUnco8ykEN7imqbNedWoPS62iUBh9oXDVVco
XHMK5Wrr0vC78xvuEYiVN+VysyhwxhWHrqE3yzsMJt8lUFF5vfMzut4tL44I
fUyUXcNuJCBTzHvrnE0hvn5PdDF6rmeJSlRHwLhHwT2OzhX87VyFXi2+2PFn
DYGpf50o8UL3yUuKakLf+TQ+0OkOgRPmbA8j9P12wW5d6MG6vEepVQS6o6YU
ZeO7ww/8zPrR6x4YH31SQeChz4uwYcxf62poOIzeXjbkvbmMQKiRYNAIvaqs
rX4C/fnnLVuJlIBP5LGk6diHw2yDlXpu5bX+oUWzH5UQCJOODCixnxLTUodv
0c+oAm+ZFRPYXdtpdRr77x1xe8EK9KGlJVZBhQRYGWYWn/B7Bft9Z78WPWRJ
iEnGbQJ1i/r+8BRQEOcyzR3Qj+lX9L2QEGCwwu/l4zy85Ndvd0VX9edo9NED
WIaiXpyfCJ2RrEPoHgEzHG3FOFcbn0bpUApzN0+YBKE/flf33B693WKmck4C
7svrUz0R6My1bhWhBQSaxWGLF/IpBCQ4ci6gbzPYtCMafWWTWah1PIV37v5S
os2f9YUsAX1mUkicw0Wck9qnyhR0J3V8XxjeP2o/I9PrAgXdNo4sCz3Y0ro4
Gt3ymOLnJNxH/w1y9TV0g4OMlgDM/1awQtCF+zs3Z50iB/17jqnnrlsEdC+Z
rLc8R4EXIpoQafOM3Vu9HvtTWTUUdPMshXmV++3y0IUsq/eMIgKCgaOSrWco
FF/vHRKiH3q5Si3H/o9vmsHVPU3hoXT4qvZdvRPiel4pgY5l9nM/hFNQZw5Q
bU5BPzt/h4xAgdyVMRt9leaOJE2bRzKW/GU57ov7cKULl4LjqH9DEroL3/eH
NpyfLr4jr/kUhQR/dRsPfUO4ZzrFeStMPhcYjn556YsbZ9EfbXMky6vx+6YM
qNzR3VOW955Ed8jtvfka53nt8pK8I+hMvmmBL7ri9OrF5+sJxHCm6ZSiZ0zo
du9Fnx8fLLfFfeHbySQWmGfzs0nN29Hd+H2j3bhfba2xg4/RnY5yjTeiJ8+M
2UUVBNiD9EwJ1pW2pdCfhS674hVm10zA5cABVV0EBZuY6k8L0W9KmMWa+wSW
LnyVqMa+ZYovqwzR/82Z/VUR/h9qhtf/5IT990g0mjoFXUk51KeNwKI1+pzW
SNzr//k//wcZNvXm
      "], CompressedData["
1:eJx11GswXGcYB3BxS5Qy2CjVJrOpa6msiDSEzUsIYt2i2USHpakIkZBIliG2
GreIS+RE1hlFRt3auhR7sa4bLGEnQTeq0VgpG92QTNHaRpQmfXx6P/XD+fKb
M+95zvP+n4d6OvH4GXU1NTV3eDTgyaMwpxNHRX3EgQr2pU4S6QlC25LTCXSo
SlPJAr/neqyou4tESw02VBvwjx9arDHAA20ymzR7SER3ZmisXCWQesKAvis4
zd9ygtFLosW9QyfHwA+N7VizBx/WtTO3FpNIq5PcJQFf2mf8gSl41Pay4Av3
8Pv98u981MHz3TSLeH34fFmLJPXlQ1GfvuHhn17143o2JyJ5E+AZDqbvu0pw
/b2R3r394EkrrFyDQRL9IJDwp8Hn2k/7t4CfTW1jz4Pflh9d9ucQqN7fmsIF
v5xNC7K8T6JS+nrFI/BRmrn9NfADG4+GX4NfqWnIiv6KQPzkuvcug7ubxEaz
huG/6hLkehkEmjas7YwA15pIjqGNkEihYZrQB17n1NYYCF5doK7DBa/uKz+Y
8TWBnJb9Kg+Dq1qlDevgMcEz6vRrBDr/QszdC65kuYZGSElEfPGKbpxJoKKW
8MdW4C5Zc9zPwFujBB4z4KsXfhszAX89Oe3WAX4ixp7DyyKQZSQZRQGfvMOg
eoL78UM6rmcTaM/uQgtdcFlZmPIg+N0jFqvhOQRqZbc56mzVGdd8VAL1DCma
VFa5BEqRBtC3g0vCR+sDwP1dXv69Am6SPXdfC1y+wyJrCvowZbbS23KdQBvF
DRQ98HmX+JQ16Ftm2rTxqTwCjVUj7tZ35YOBjCzwDnpJnNoNAn1zVfbECNxR
ZeRSM0SiXq2AgNvg/RyLWCo4RXOhjwn3ZZP275ppPoE4vzjX2YJ37FHUpcC9
N840xd0BX7SKP+kE/smg7G3ZAIlElfW0JfDj5W3BXuAaPrtVQZAfiplQ7FZA
oNhL2p5McI4Va1Yb8vaGt236CvizDLZdHHhNS+KgGHJLCyzJKQWXzmq/ywF/
muRYlQI5j9ZLEH8LrnS2+7AAPHWzknm3B79/kXt2uQq8a6yZGtaNz/dQDHeK
wBvLVc8oXbge+yPe6ePgCl2PSlkHrn/kwT9/zoLvCpgRbojw/6YZv5CvgnP7
DT2tRbg/tQOkrj7M0c5c7WilEPfT9rnDEhXcsWT9dyMh7j/6a34bDVyS+YdX
swDfVzo9Xu4F/uCUsucJH99v6RvvhVBwgQYySubjPNRW2Vz8EvyM4YjCkI/z
83x/4VoS+LnidF4zD+etwOSj/engVLXW+XYezuctxTufFoLr5/zqEMrDefYt
fdt4E/xzs2EfKQ/nv1tbeY4EL2I/3pTx8LxELE49LQM3uOm705eP56t7oTis
HHwtUOncy8fz+LOBdKIC3HaDydwnwPM7njuus3U+y0+sMBbiee+eic3fcoMf
Q3TzhHg/nLBiHLsFvhiiZRHbjveJp7nB0DXwgvPu3wtEeP8ETVbqp4JbslWM
kg68r/5vP/8HzCUAcA==
      "], CompressedData["
1:eJx11G001HkUB3BRsg7JEbHUUDZxekKhVHu1MqI2nJQ8bDVFT0bKs2hEVq29
QzUiecpMWBWjpmi3qI3MpCijTITGTLPVSQm1Eqe986pX++L/5nPmnLn3e+/v
WrEO+IVqamhorKJPiz7rphcTPUF1t18mzKjS+xXh+10G6PgAQbd/39on5F5W
Hhaa5FKnpU7arQg2zZpGYvIfpnlaPiX3KggPVbUhzNmVNVBPnn5ZV7eF/JIJ
O6/tMYLjYHrzNfJJoTOHG8g1u6/dr5Mi8KRBSiG5QxJv8tN0BMbPdn8VP0Ew
M63XqSAvM7TzbCH3XLPGOqET4bhPajqf3D1Z1So/hrAIQiTezxAy3pexi8kH
mDtlo2kIiTEt/VO7EVYE/CtTu+SMQaB9KoKr86Cj6DlC57VFdwvIM9dXFq47
ijC3TWG/ohchfqXDwSLyzwZzK9I4CKlLbeWlfQh7fYYqS8hjLMXB9ckInCaZ
27sXCB3R2s3qer7meluPHUa4kufjrduPcE7nn+py8mW1bKVTIkLMl+ecL+RO
JxIiL6nriV5SGR2PoEjgFDUoEAxTNLyuq/sdYhd1xCDkRjmFbVIiLD4c3HWT
vOmj1ZkZ0QivN4zn1JIL+fl8df7x5Tn9sQcpH6nwopLc4NSf0x6TM02yTG8c
+Ob83VrJXeTck4KMcTbCeJvgZBW5fU+V/mvytvlpi1n7EbbNv9LnQX6PKfce
IR/22dpavpf2wUfxtJrq3Nzuk6oRXHd7rUtk3rvdCPsjxkaU1JcxLz5bnzyR
4RfmFIZwM9t64q0cIX+ne6w5+ZyNtc6cXdTXG+b7ZsotULTAyI5c/47DSzEL
QSeHy02inKM+PHddTm6+MnCF0Q6ECO7qPiOaS2z7sR53ctfqSOaq7QjK8aFZ
GTTHhMFhY19y6BN/t2cbwiCr4UFXF8Lv5ofOhpCL+DrzTv+CsPG+29XZtCfm
jK3H95FXWNjcrQ+huktNpq2nvfpQxTsWRV7oOZErIc8IyJPvpz1UDLIvJpGP
XXAo6SC/qDlcnUp7K1ipupdOzuEK8/vI7d1F9edoz+Mm895y1b9/nXT+DXmD
b0nmFXoXvsGOp0+TSywaXDqoHvNuEePZQ5rDVbFpHvmaxkcVt6h+WyUnVpve
3VCUqU0B+eScmhNl1O/yROOz6+8jBKfMGikmv5kcYZFF+Tj4KtxKxAiPBo6+
KCJPji5v8d9J72Rg0/jMZso5c96yQvLlfbJG81CEh4KfPKqbqJ7tVkP55FGz
x6fKaV5+f0gYYY0I885UynPJQ2Ramy7sQfDfcr2VeZf2WTrjs7r+3orec3v2
IYx88eJt/hvBPV64MIt8gXdMu104Qt3UezW8O/Q/C8xyTpC3JQoN39K+Jdl/
EEzcRlj6UrvxCPloWqdWZiSCiSjvagF5YOqUJXHkH3tye9cdQqi4VFpwgHyt
sR5XPa+gcJdiVfQ3D66U2O4g97Ct8eTHIkzx0I3jkwcYdmf6qfsVCEpXJSDc
+TQark/13Kj13MIklwVJzUbpPfZnl7mUkofVChmu5Ho2hltqkhDaxbcYbOpL
N9B4+iLygypn7t4jCDWKU7NCKQfjONZ8S3L/vq/1likIGgv1HH6j3Nh60uOG
5NpM9+mddE/iCkvOPqGcX5md52mTs1RjrZfp/pRuKN+4muYiz48QflLfQ/8f
c1h0xya9yg54SHOMPSKJe0OeIjsqcaa7p3ITVSfQ3P/vPv8H89vuDQ==
      "], CompressedData["
1:eJx11Ps3lGkcAPDJpWiNZWiHRFOyrVIttdW0GV/VNMndFE60bklUZinEduV9
N8P7KrWVa2bKJWdz6eYyRYohdapd5+xat3QqMctUDBFj7NcfsD88v3zOc873
+jxLQkU+4VoMBsMRjzaeQUIRaZJS0yCojIionaHAellL8ZZxAiT60qKZ5JqG
AnakxFuLhqlMwa3YLwRk6w2FD6Iz3Cv5Ch0aQiM3arVMEbBpb7WmE50Zm7VF
MpcGu7I2jb2GgB5KZtGMnjQqsN6gT0N7jWjwHoOELHGyxz10s56CQuV8Ghxj
mcE8bRLEgX4hhehsdrfmugENRF1K30tdEl6zCmMvoqcb6vruMaShpYYVFaJH
gufE1uxk9GeydXIjIxpOaI5whuaT0Dqe7hCPLk79it+PHpAjsw1mktA+sIQR
ha5c561Vb0yDH8dJ2PQ1CbtbQjx+Qp80LfC7xKJhZV7/pzksEpSNSo0Puu+o
mnPYhIayFadMl5ri/SypyA1db/qJfzh6KM3rNf2GhIp2O68t6NunWWk8U8yf
s9O7k03CcBPL0xk9ou6IjT/e37dakR5qjm55vGozemC/5JYP+pmVZ24/XEiC
pL7SwRG9NKfTOBrzSUvxOv+3BQnZMp84HnrG58q9aZg/W530y+1FJBib3y0E
9JbxU49LsN5lW1lWOyxJaGzY3iFInp3vAkUf9o29doyRh+6ya+czV3Sb+6J9
uujWjfLe39Gl7fF/7kLfLDC6w8f+V962io9DLzJ40BWKzrMveVSMc7zBeeI2
jXGzByTXD6O3tZcuU8yjYfWI+2JndDnHhZuInh/9JckO90Gc0SYTYP4932V0
iNGhkJF1XpsGjr2uz3yst9W5anvubL2i88VjDBouFKtvnTMjYSr3H5tS9JBV
JRU/4n7uG46t7cc+p7+6+PE+esCLYP9rago6RC+f6ywgIf99IPcleq6n2RX9
SQryWiW+UzhHpw5D7bfosucJB2PG8f6dZuEzIxL2J6k4KvS46QS2bIwCPfs0
/QRDEtbsDS+bh+/i4o72F0tVFCjMhSwDAxLCJl0OmKH79BbYHxqmQNOUb5uq
T0L9Jo1wFfqNoSW+f32goCxpdfXMXBLUGd5cHrrn1XUDVkoK5Exrh2gdEmyq
Fll6onMXrk84MEiBYfD3N7vnkNCp4jYHoPdHeW248C8FjjdLRrxnCPi4PGrb
oVmXubp1DVDgwLnxrkdNgPKhdtQx9OoEuUgHvZ6pups4ScDYz6eZBHqQH3yx
66fA/b2x0eYJAmL6wt3PoZtymw12v6egkR/wzuIzAbWlE22X0IdeMesC0b0b
TYxtRwmQr3VsyEVPSWypDUOfyAu6HDZCgO0J/msJ+tWHVFMk+qPHmcKuTwQc
D9b1kKJP/1EQ74lxc6hvbU5+JOBKRehIIXp524dHQvS6s7UQ8YGAvLqzFdfR
V/X6x2/A/O/fW+SVriRAcHlHUD56FcskYlJBgZPlndKxIQKs3zjvzkFPjTHQ
7sb+tPKkTVJ0Pw9b/gX0gKNP3+YNUZDE3t+XiS6Wl7uI0V8YNpUH4lzmnEj0
eoIufaqKOYUu2XMtzuITBZwIxXHAuKlC36qj6Jlv7Nd04HxlI1YVavRXrg/0
w9F/69vPjx+lYDioyPwz5i+vPnRs1+z/2V3kxsK9OhjTKF6LfUhffNpRgD7Q
ucL4+QQFJ7sm7MuHCRg2FPB/QA+L9YiOmKJgdN6og6+KgJqo9b8uRVccMOhl
aCjYttF1OXeMgP/7n/8DOTLe8Q==
      "], CompressedData["
1:eJx11OlbzWkYB/DkVFSkpqIUnSxjUpomM8mocw9daUFJJo7TRiURyalwWk4q
7Y/UL1lSlyEmS83M1ZVocRgKWdppoUUhkRqTOoT5+gPmxfPm8+JZ7u99P/zN
u9z8FRUUFGywJmK5X6vcUjWtTEZ/eUx26eTo5ko9l+u7GPGU9sgL4JoeQyVZ
XRypPS3X0N/NaOLI4JtMeJtITbatmyM/pyubMkMZeWVdEsfAeUabtwt6OIrz
yzfnixltUo0/uh3u4u0i037GUb9ZvNPNMEbO72YYeMKtqw3DJfDS6xFDByMY
/SO4WeAAF9QY27n2ciTqrc123McoQ6RYbg1vT/I4Vgr3OVp2w1jCqKl37ZAJ
PMpEQ9Wwj6OeW1LTkUhGLpb2PAO4h2y/Zxx8jtW0DQ+iGV0+fMXzG3gT92ub
BB4WTJU5UkY/rgh8pAr/NGK+Rwy//8TLW3iA0ZEW3UxF+JqTErtg+MkilUiN
eEZ1Z5WrFOCF1RMCxnCfmrt2Dy8kMBIfMPSUa5TJDOrufhqGB1h+PGaRyKhs
mdaZMTiTSHSi4VXDYR5ZSYyEj2SRn+FlVRbHc1GHjelByc3JjNQCIlW/wFWW
L3jwHdw9R1j0LoWRQ2e0kzLO9emQpjqizs2nsxIGUhmVvw3wmwq/ZeCgHIRc
HvTeW1WaxuhCh/2G6fClojDbVOQoHJzJc09nlNj83p4Pt+gMdr2E3Gdcseyv
hGcnDg4uhIt1BBPrnnL0QdAhncAYpcz6UmIDb9zUnZvwhKOtJdYrJsGTO3d4
OsGN+uapn+9A7msbRrqxz8OEF7Vf863syNXxbedokeqO1gy4bl6GIBQecXHJ
BL02juL91c5PhgvNpEdi4VpaozcErRyZRt0JWo/7uzquq8yB/znYP1/+iKNQ
46yZYrx3LLm/5nd4wYibLx+uIHKt8Ed9hmt1fWTw60WzeR3NHDmGX6y3Qj0F
Ken1DfBR/hqxElw4dnh6J3LxnjuvvQ/+ua5wq3kTR9HRXfUBBxlVeFjYy+F1
mg1FGxs5ko3H2rUh923n9i6Yolkm63ufKYpv4OjLlOo3i+Owv0pFDh8ujP0j
rrieIzf1EefQWEaBaaZa38NtJydVO8DT/OMLi2MY5VboyW3hijk9AT11HMnL
HaJbohjtXlJYtRqu7Kz+SgK/PatRm4d+VopbdUYEL67MS9KBixYsfWmyn9H8
0t2O/nBHO7GiITyy66rUdy+j40Zb7gfD3VKyvOfCH+drt+eFM4pqzuCFwy+0
uu5fCPcMOf1hAHOq/M6mNQpu/7pT8AN8r88c39V7GD2NVVgeA0+cr1VxFn7j
wPqMGvwDwnal8a9+eCzwlD7eFfIs/p5/CCPNVlNDKXxW2C1FBn/xajY3B/9J
c9qhE/vg9RltAztRtx6zt4FKOxmdiypw3ws3t7HV7YbzLz9LnRLMiO2yFITA
9WvTf5Oh/oeW5V9bvoNR17cxLlvhU80pNB95ldhY8Qq2MzJ4qSH0hOsePOEd
hXz/rTX2tIF7OafLVn09V5rXeLuFoxX5p/6WBzE6uq1e4Re4Iee3uAD9E1O7
VDYEN3JSClgEb4keHxh8zNGwV/HU6djH0kMypA+fpCrvs0LfKubwdwbBr42u
C1eHr5kk7tdEn/cFaZ8ahps9HzYZRf/8lHDPeSPm4k7G694C3D9bz2D8OXxl
9bImC8yRypsj1ol4b8vPaz8+hB83SrwagblTyf2Yl436/N///B9gK9AK
      "], {{-6140.472111210996, 1064.6174185696211`, 
      1359.7814874155792`}, {-6140.2877214676155`, 1064.916621637695, 
      1360.3797177824285`}, {-6140.3078054930975`, 1064.147376915057, 
      1360.8909228903458`}, {-6140.518449776822, 1060.2102333906491`, 
      1363.0117824010886`}, {-6140.906624666973, 1057.9595446694018`, 
      1363.0117824010886`}, {-6141.40339533558, 1057.162197018379, 
      1361.3912424266155`}, {-6141.396264120312, 1059.1476230909598`, 
      1359.8793805149949`}, {-6140.94591397865, 1062.2708935164792`, 
      1359.476928960771}, {-6140.472111210996, 1064.6174185696211`, 
      1359.7814874155792`}}, {{-5948.458230543473, 1907.092836985149, 
      1290.9900389234363`}, {-5948.781324470068, 1905.708060193415, 
      1291.5460630383966`}, {-5950.452387547931, 1901.3205051446444`, 
      1290.3140742631795`}, {-5951.081131641095, 1900.4912237695091`, 
      1288.6349999128556`}, {-5950.758945556679, 1902.3342119440279`, 
      1287.4028926270348`}, {-5949.087828273249, 1906.4931941448017`, 
      1288.9730028866525`}, {-5948.458230543473, 1907.092836985149, 
      1290.9900389234363`}}, CompressedData["
1:eJx11Ps31GkcB/DRlqFU7GC1lEvHoUNry6XWpZ5UbNnYTLmkTSUVXRQHW2q6
TCqVZ0x9FSI0ynW3M6gzS8zYEWeWLnKJYZjRSLt1ZE+J2NW+5w/YH76/vM5z
nu/n+dxsd8cFR89gsVg++L7At4mXvr50nkTW0FhWWq1liJaxWKtQUOIcdPFd
PjwyeCK9An56d6V5ZQslUYs1ydfhVZ7GW4vgnCGmvPAxJZLCws9X4H93iixz
4UXh1auFTylhCywj03T3O7Gr7eGTTSaZF9ooUStf96fCc7nd+++9ZMjBzSUO
ye2UDK+bW3kCPnXJ52gxfKJH+SGkkxKjM+8kp+CCQWehfJAh959F+Sx5QYl5
009uJ+GaP1NsA+GuZVXT2m5KPh1kHFPgrFsfvu7RMMSL6zVxQUlJ9tvgR7p7
vPgNuR/VDKnO6Hil30dJwJeS0XPwfCNtjin8mqHwyn4VJdZ3F09cgs8UHApz
HWBI6AF9+7x+Sgo9BTVC+PV9+ZLgfsTPXxtfPkDJqleRqdnwA7Jz0/EqhthY
ae/x1ZTse7p/vBieseV9i30fQ2in3csVGkr+3VqbIYYnzj4p8u9liHtEeaoY
7i7jmzyEH9J0rI1VMsRggUw5BXdr0xc369471tSX3oM8B6zMnzNIidbT8Bsl
PEv+3mldN0Pqq/zN3+K8xiSieBhe074jMuYFQ85b+X/OgfOVG5d8hJee708W
dDEk2nGomwPfnNpaNWu+RNZRsTHxQSdDmChjUQji56oy55vDrxrfdejvwHnR
cMFhvPdFToG/A9wkIzxLHz4y71dhOPLDXR4a7woPeqLW5rUzpNO2wM0M+eSV
1K3xg3fPib3f/JwhCdY9rRW9lChiSsu5cG9+ltlO+Pw1LlYOqFd0bda9XfC4
3keTE20MkfAGNGdQ3wCL0MBYuHHi9iMsnOc8id8h7aJEEDsZeAze9X2tSw7O
K1yauK86KDGYdMw7C//x4Zj3bfg/v8Q0fn5OiapfZX4FXrSpJ64MPrKoaJkx
+rNm36QgE240+7vblXDrg5a3bNDPHd9edcuHH960crIGnnDNku2I/m+PCJfc
hme/Ea1Zhng8lhvGr8S82PECRu7AB5/M8yuGr9swVRCG+fpNekxTAleUGZgs
RH6ig7ZwzzdTMqtg1V7dedMsszQW8tkSlNna+oiSvzarUorhSRpOZiKc91XS
ZYdGSrzcn6l1/93+Ws8sDPWaqM/OLPkddXfru58Hd+KpLDxRX7s94a/9Gigp
bzEV3YC3Kw9vsUQ/hPRGp5nKKNmtYAYy4LGFYeNT8Czvne76UsyjMmFRGjzi
B219L/rKcWyj2Kmekg/28sLT8McWHtoa9OH6Nvax5DpKGh7cnEqCWzLj7Gz0
7RtBFYcFH7CyTN8LP6sOvaxFn1e61KZUPaTk8sIh9TZ4qffqERnmYtuzU+N3
4HW+G6IC4Xv9r97MxRyZj5YROdw/5uKQJ3ypTJ48hLmL40lzzHE/2067yxke
ozoTJcWcJommG/PgYxktait4pJ51Qhbm+rgkVBqG+Bf84cw3gtes4MiPYD9M
CbObCd77c6LYQ08X/6jQQwlvct+xKAz50ftkYPMO81JxwpAtxp6R37D1zUE+
FTfEfoNwn7pk2wW6fZUZKZ4hp4SZ6FV1wJUz50pPwwdPuMZcRL2OhxxVNsHP
+tp0DcM9Q5aXLW2iZHo6wl8Cb96z+Egg9mQQJ911FP3wf/v5PzNS+SI=
      "], CompressedData["
1:eJx11FlQU1cYB3ASNBIiRoVatFIJOIJAQWlVBB0OkKlsWquoLcgwjaCILFa0
Bq2WLmY0lPXmggi0qAgEhIqiAZFFhERchmqoRBYhkbAoMoCAWAz0n4fWpz6c
l9/cOfec7/t/hyeI2RbGNDAw2IhliCWL11y0VMnqAtRaQQ2bJjc4q7Iz2yiy
gOH63BSelxFrdQ3OHUtNOtdOkVNLmi0Y8EunpifNjGlSHBkRn9VBkQB5V9BY
q6yOKnSUseEZ0ZXfZndShOXmwemDB9w8+kaHfVy1u5IKn1HEzcvMvxseaX8+
5xTc9kXiSWkXRQyO2Z9Wwo8kbA08BufvcY0u6qYILydB8QBucvzFDAOe6RMc
XKymSF14jJUCvmD1klyxEU003ofHaA1F4tlO8Xq/xS5fmT+HJqxE84T45xSR
/jI1Xg93Nj/Z4AQ/LOloCuihyBZR5NsauDAtvaOSRZM4YUqHhZYiwdOLivTf
h4wm7FTNpon/uPZAIzwncH2/HL43WrX5zSya6Nx2Nn/TSxFFi62n/pyL56vd
P4SLbvDb++DtKTrZY/hluywHF0OazFg1JO/qo4j9ksWqNviV10GpgUyafGDs
cfsSfNewp1kPvFReMHyCQRNeV1RWJ9zVQhQ5BB8ThPteMKDJskPbPp6B8+vD
DhqgLyPWEbOCZiT/OV07k2QCrz6qbP5xWkJ4hVxFP1xcb/zVUvjpeykLi3QS
Em3ZM3gFvvuE40YHuJPC5gvlOwlhzN1fJYBHNEgfbIQ/NilgTk9JyHLXQdZr
3CtC2DngA9fd8vJLh3uT4hVCeO4PsWXB8FdtrSP1f0uIzMvzTjvq5sdtkRzS
75N/PHX4rYRkFHoarYTbfrKJ9ROcWn9LngCvnO/LO4C+hB+4TKXB1waGjNvA
lYPJmnT0cTFXvSEPXilaa9E4KSHr0if8LqPv1xSHtNfhieJ5fAF87OcX5beR
k7yQg0vr4d21akYU3KHp95H7yBUn1LC7BV5T02/JgUd5zGbeRw55J0dWa+Dm
HP/WRXA+k91bg9yqquZM9MNb73bZJ8PjonaMVyPnoYlpn0/Ag8W+6nNwx8lP
VWWYC9MkR7YOvnDIb40pzh8Yu6/3CuZony56A/uprM5ylDtsAS9VMiyrMHd/
ll6N4cJzK67KaXj1sY+/bnxKkUG5XDQPXsB8XhuHeg44lA9oVRR55lKTbQrf
KijIHob7MjRxLHip1/dGC+CJs3yWKdGXd1tkid6tFJlZ16zVf88fKTbyRn97
M3wf0k8oErLtQr7+v49eKsxWIg87JspD3/5FkRXWH/kZwx+X2NmwkZ8ul7kB
sfDMh18qDeGZZ5rc++Ey7p4sQ7idraRsCvc96lSrbkQOO9d07r3eQpHXodWx
o/Cy7Htn9LnNf3REfALuWDHfXF/n3Ib9218i5xXFn7F2w28fsTZ7Au+XuIsb
me+9ySJm71348ldqn98M3+8zwRPa3YRz4pi/foe545o0uRbBe+r7bkjh3prc
p/6Y33/POXqRcf0sfIcqUmqFefeOe2cTDp8nXc4WwUUVbow38FfbC12G4G1n
hXwhfD3Lgx+Md6Mww+O8GHUrLuEM7YHfkQ5YrcL7ExZYssYZdd7EtR7ZDq/K
Go+RwA3S1Jun9PUXtIV5wZPzjNwn4TnR1anN6FeJv6W/M5wZy0kJwvt2YZkz
4w/0/f/e538ANNIPYg==
      "], CompressedData["
1:eJx11Ps31HkYB3CmJKyKVlubVUJFpWkxQjMeZ4zbTLnUMI2pmG3cZpqZqA7p
siOfUn0KCWv3NCfX1QqdxZqNNoVqG6PWUdLWVlbXSaXbHovs4w/YH76/vH74
fJ/P+3mej6NUFSVjmJiYsPGbgp9/zMTYzWxdW5bPsKe7hEJgcpnRuJ7AL9og
w1X0WcqS0umbKEBkjmCbkEDoeEf4b+hd3/7dMIB+w0ElMosh4B2cnN04eY5N
g7x1MwUnXfDMRhEBZlGm7if0+L3KRUVbKKj3t3RKxQT0mqSOCnTr0mZOLrqk
20bsKCHw7vWdG1r0gEa1OAfd4szo0INNBNq4nSPfoYe3qEs06AlzmOrCLQSk
832dCtBP73nStxv9V/HYcFg8gQ1dLofz0Ue7tcMFWI8wJaP/rZTAzIRPfnno
wpNtW2uxfv27bXbZWwnw9QXOuehyuXrOVcyBpexPNUsgcOHGkkMn0InVLqNJ
LAUfTfHsxEQCikeC1ZPu5jZB5ospfM/e+G1tEgHPV17fFKGfFxi40SIKDi65
zN5kAuUf+DY/TOZ2S1+SH02hfqckpjeFAE/5wPc0+gnV2D7DBgqxFxb4nJMT
aD6ufH8GnSdVlbpGUVjQV5wvUxA4mtMuPoduViL6kBBBQWtRlzWI3t90PaQF
XVt5dU/HWgpWdvfd/bcRqLYL6LyMbv/aw3aKgMKKl2+zFeihg24T3eirk4bn
SkMpJEVa6GTozfGu1/rQl7Cs3cuCKbzKsihchT7quJ3zGH2qf2opg4d1FpT2
GfC/8WejXV6jb/6rd4LLpfD4RcQtHnrXoOjaOPod5WJDJVA4v+5F/WG8Vw2D
Ozid6NrSZ90//oxDweZrlumPmENtj+rA5+h1j9Y+XcamkMtf/PspzO3hKivt
QnTbHJFB5UfhUvQIX40537ba77wC/djzN+6NPhS63Ni2TtiXkLR/Zvii69zH
GSPeeE43M04nI6BqSmVz0f+YPSPXAT1scRPDG/vuFTykE6Dby4VpgSwK7OGW
ES3OSfGoUBON/jOjuT7Fi0L/PBbvTRyB5fXLQ+PQrblmxjxPCjmh/LwAnMPy
hHxWAvrgw8TWYvRKP8sAgnOb0+u9TIHOFzoVadHviC1rr8QS2KdSm6SiH5yp
41ei8/4dcrXBvTCtS+9NQ89IHJBT/O+wmZ10I+6RfqIsNQP93qmYgRPoIbLM
1oZozGeez+Xd6E3ll1IysP7khQf2LsI9vczw+DMT3eWKrnEL3rcg75h9Ke71
V4v4B9PRw41RQy6Y2yF1jQc7ikDh+9SOHegXGz3OGH0p8BsCzUciCFRdelM1
6ZFljrk3Mf/Ap7bre8IJsOcEjKnQK7Qv/QuxX8yorKU96wiYVyv0kzm4Wb1t
98G+a/343I9rCVTs1CyMR880ObpuPICCoqLnKaC33iXtYvSguZzP2nB+llh6
nbsoIGD95ce+8MlzLHkF0iAKVQ1yhgzdYdoKSQi6Z2vDNOcQClIJqfFG5+xa
PX8N+gdXztTOMAr3mp+0AvrxjsNBK9HdqzgxMtyL+OpbD3egb68pH3BBF6kF
xc7hFJS1VbG30VdmHXn1BXony1TbHknBWBUWtxnrHHd4LrBG5yw/otSsp8CR
LDDY432NxcyVDHTvroh6jpCCeZBo/wT6M9Oowo84/8y2pXGjuO+Vps92TcM8
PzldV71AT+82VzXj+2B9UrGGFYn1n7378j563TuBTRq+J6uq9V4a7Mv/vc//
Ac72wAk=
      "], CompressedData["
1:eJx11GswnGcUB3ClkrhmE1NSISISVVUhwVZVPSIRy65l437Zum122cQlZKax
605ERN68KLpSOxuXmkZQIkW3bpuO6BCpUMalCXIZKpRiJ4Lt8enpl354vvzm
nTPnOef/vMYRcSyOspKSkiMcFTi8+mrnGrvWrrzatgPBMQQqmerJtUwj0Wjv
SGQ5eN9amoIO/iT3TdyxdBLJtXPV88FV0y99epBPIIpDf9+JDBLp0lbr0sC/
lKvHaoIXnzw4b5NJotxBiXci+JZKOGcT6pRObAkYWSRaExomccHvHF5TLICr
tHkNB2WTSNI8800wuHfy7W+nwCXFRha8HBI1ml9u9gF3oJS5LUcTaH9JknvM
VRIZuaSwaOAs8cmcGfBFuhfvQi6J7Ky2+C7grg2nhHIegTKTJ3QCrpHoxcxA
tiN4jtg6URO8Me29Tpc8EoW2Z5Xs+Pav831nzhMov1m/U+s6iY6ftjG3B9fQ
VdwL4RDI/uZcxyi4ZuO9B5+D+w9qrpRGEkhoPKWTnk8iwc01p506PaZ/pg6H
E+j0+ML6vhsk8vOop30B3jD40/RSGIF0xea8AvCaKYMBBD45t6yw/Arql1Rz
V8D3OST0u4J3KFbV40MJ9PhlqdS2gERama8fMcBl2U7spmACmWU9E0eAz2+N
SoPAtT1Eez8MJNDV0caxi+CvqtCJqJ39KneUsP3x9/yiepM4cLf1crUaX1xf
IHXZLwB/QrbLFs/hfuplvcJr4E3iaZ49C/e/bqQbUwjOTNd+GeGN7/v21shj
CXhheYVeIxPPZ+JNglUTeMJr01vbDDxPa6+Fok7wAbZltgUDzz+uJjr+9538
MNgvhB54X3ryKc/n4FFZGV117ni/1J9/sFgCV96lVj5Jw3koLv67YAOc3xHG
/IuG85M39AtXndralTTGjDWg4by10j1ddcHZ0wYPP6HhfCrczY4eAb9/bPuA
Aw3nuWLzaZoluLFeLMWDhvN/cah1yh58kL3VGUjD7+VutC1yBs+y/q3W1R2/
r+86mYfo4K2VVc0+7vg9muonZrLAX3Wo+iKYwxL3YzE1lUSezylz/uD8uusN
5nQClXlfMPNPIVHmrkW/APCC0SPbyzDP5YyU+QwhibrpJs4h4BLtythxTwKp
tpUekglIdC6Hcj8IfLfqbiuxF4E0mEMP9cEttGe/36kT36LvGQl7l/doVaUk
k+idY4soEFwnuXb+LOTEV9RtqQweKpWU+oBXrqTa7IFc2V55n1N7hURz/T9+
wASvE0nHe/0IxHl3aTgQXJ5sKDoLnl4mX8kOINBHQpnZZ+Bpi91iJ3DK2hmu
cxCB9gQwHhiD88eXj1uDn/racHYsBH8v81NITcGj+IePFrJxfZOIOC998M3U
vhG3MNxP9V01jhb40w1iZjsc98+iqglUwA0e5ceOROD7PuOOlP0D+eHeuHwn
NwrPh6goos6Cz/aKrOzP/2eeQd5+f4C/DabOaHDx/FernTZk4HVmG5PtPLwv
R1797RZwu4AqcX403u///Z//BYiayuE=
      "], CompressedData["
1:eJx11G04lXccB/DTuEzKw1JNFiVnKtq0OUTncP7ytPJ8Nk9dizEqreOQ1gMq
YsQ4OG6585zjENehrebhhJMeVmpFjSuNyBV1pu1KYpfMnO17Xt2v9uJ+83nx
/9+/3+/7/1lEiQQx77BYLGd8WvhW+fX/83VZWzd7/WF1r4wmGROGlne5FEkR
9LWFw20SN68Phj++4N92lUcR28MzMQL4hDLVbSec62Xo3eJMETVVus4Xbngv
M/pBLU3KFH6jjS4Uibh3J9EfvjXNcc2fUpoUCFO8ivkUKfEL7NgFvzKe0pcI
v+9r/8MpQpEmG5MKzTkezelrCmpoopS9eRXuSpEdit36gfA+pdG8/DxNpoJv
atntoMhtic3yIHjYho1xd6tpojimeLUAj2zfuHk3vJvOMJisool1y77MZjeK
VAyODkTAU7zi3HXhHbF7ZoPcKeKcNFu5Fz5/o0K2qZImtT72hir4wfSgwm/h
VrZVtqPlNMlZpj0Y6UGR68l29Ek433GtigWPtRr2vgV3PX656wzcTdh1h11G
E4epHuFKT4pUZQ/rlcEvD4YMjJyjSX72Y54PPK7RZrEeHik/WqMNv9HekB8F
b35NDFrhf/CEZVtoxt25PM97cKce847Js8w5YitX9gg8MLS+cuVZ5l6eiZbF
FHydRY6cFDP/+f5yA/sFzbxacyd/o5i68r+MFBmWt3Vn3Zzz1qWYPli/4Vwz
g7fLr41Ii5i+qYlLgx2cF5On7pfQxDB07KICfS5Iz3B0hx/ZFT+RIGHm0isu
7/aFPxK+7EuGsxc9PnKBz3V9EBcBHxOZPTWFLws6bnIAczfNj7b7Bl73ucp8
o4TJyduwH/cfgZdcGOr5FD4+r59QjVw1pe3pPA3X631YzpcwOcwyrtc9A2/L
lh5tlzC5va/ifJEHP/2ZteCTIibnYrVOlQQeumTvlsYi5l1cevZLWInmfLlV
wgb051Csuu3pdooE1j8zLocHz9S2lMIH7I/5/OWEervjHM7Br1w4cfAk+k8X
Ouithnf6tPcUwzkxb/MiMa+6yJ3Tro6o68XDu4XwUiMTU/cSmkQpry45vo0i
hYLSplz4+IOW5A+RhyyuKOaWA0XSkiJYWfB9tgteOsjPa7e6N+bwmUYdTir8
sWPX5HP4KhG/P8+eIjYL8fwEeCpXe/9W5NM83P9dY3j9GRVnPzz6SNpP+sjz
XIZK3MyhSNDJpiUhmj7Lpp4EVdBEbr36wFdwi2N8hTdc8PFo/1a8l67YF/Qm
+OnE2esucFWlUk8P7yvlyTaPFXCheId0M3za5dBav2rGnaJ8Rabw7yx1F9nn
mXMMotPzlmrqzVlh9vd55t6Il/O5c8jtz5nN7/XWMP9ptigTTsCXPro4VC1l
6lovD+EOwk94KhJ1apk+WLYnjV6DN7QKnj+qZfpWEsBSX9LsB+3mzu3Yb8Mz
9dwOODudFtXAO6Tjv5bLmLlM8JRjhZp91ef+uxpua/L9bQfMkfNvFD8LHiCe
7p2Bz/idGNPD3B9KjU4la/ZSFStHJWNy4pKZ6hsPFy8zGxqGC1gB4gHk6v/2
839M0t8J
      "], CompressedData["
1:eJx11Gs01GkcB3DLSUkXW4mj5Nba1kYk1cHokWxqTWlmSE1FyW7jNmvldtRS
m9LolMef2Vwyo0SyuRZN1MxpouKkZCVqkHVJF/YUJzO17Ncrr/bF8+ZznvP/
/57f8/09FgeEnGBtLS0tFpYOVsoDlYu7/k1F+xymccMlSozLHPaNsRmy+B+X
fHs4Tz9J16iAkoQg44yEbQwxbu8fWQaXjodEv4cbsyxXGG9nSF1lc+UcuOGE
w1HlZUomQ0zeyuEx+kND6tk3FUeUtaZZhZS0nRl7+asPQ/zjrDMHppw17BdR
RMn+xKJ/7Xfgv5LY60/hLtdKV3hcoSRkjxVbAz8rquhWwp/LA9XF8Itcdq6S
wxDrox6J1+F7xsyCDIopEX+2MznPZci+V0uYAviSwvbgGPjkgKwhlseQdU6p
btnwnB9mPY6AR9n2eHj7MuRd5tmXIviHjpK1bXDr5SUKcz+G2FTwFxyDZybz
ZjTDXYuWz3sD97SV8Y7Ar4o82/zhzCpvftVOhvBdtQ2j4L26HXocODd6TXyk
P0O6dipfRUzVc1SzQYX6CzJbxat2MaT8Xum9UHiUt3KjAD4YF+DXDpf0cRqn
9pcnzl6Ugv509g/XHNrNkA6hYFs4vLSvZ4shXF1x+M0w3LFJKoiEF/781+Aa
9LlXrX4v4DPEqmYsOwbeXMXn+OJeRJxrTDPcQvX2/dS5or50iWbhHsWHcl8s
3cMQrVrhj6fhVfZOZ22QB6PulnoePMKSJafw22WqveyLlHhs9+XEwBVOI1wp
PNT9mkY7n5KtXt8fOwZ//LX/RBncJ8s8SCaZ3p+qL71eCz/JLxN35U1/3/nL
E9EjeFq6nVh8Yboeb1uvt91w6eAVzbZcStYYn7rbgfpfhAz9PQz/bX5NWkwO
JQtnPlsaDe8LHxfqIIesuNWp9tmU3Ddd7TUDTp236i2Cn3ML27Eri5KI4ErD
RPTNIqhUYQN3NF3yatF5SqovbUruRf9PzIjQuMKzs7ueuf1BSfqZunpnuA5f
fmc7/HNykINATMm52/PaT+F+OapNuvvgWfuXfvwKXrRglewh8vDavbE7En7h
iL2RbSYlBwoNwrTgkRsTK5Lg3/UbshoyKJlwky5bh1wV7n03Nw2+o1PvQQDc
rL8r/wDyqfw8OSqBmzz+VKNmKAl1uDN2HHmOfsL2KYPTeq47A7cOd48rRf5F
dcICGTy3ZXFzNrwhUKDpwLzMrZgIU8IVqtbRfLj3714lBvDxTwftHsC7RDfN
jPDfi/I2/mbM3fCd+aEt8LqBgB5TeNXBlS6nMb+nU+KlbXCdqMOX0+EsbWeT
Jsz7LSFb/hwucrCT6OK88XmeO63gYy2Biqn9wZYeTwfhXuPBHyjeE1bqBlY7
fMRsZs5u9E3BTa7Th4+MfvNtK7yTNGi7oP9lD/+8kYZ3qVMm0Dya6kPPYokJ
7itJuHDuSviko2nNfbgsY1Cohpe/brir8mZIQN19Xi28KY0f74k8nHnS8uwy
vInj0VcJz8loqLREfgJvuWw5Ad/sacMrgLsZrF0ejrxVO+iZxMNvfBFVZ8HX
KxX1Xhem93Ptbo+nwH+yTs+jedPfL17faxULn21fcrVDMl3PL6UhjwRwK/OF
7CopJTHlfSpz1P+RU9voC6+vPi5XY44SfJJOJsP/733+D9le0aY=
      "], CompressedData["
1:eJx11Hs01GkYB3CRFTrYFpMJe9BkqbXRHjZM3pFLbpGmZVZq16EM3RxNR6hT
wrjk8vMrSWvbtS06Ti4RMxgmg5RL1nSxm5zaSJkugwqx7Hf+6q/94/fP57zn
vM/veb7PaxFxODhKXU1NjY1PA98961ZDRx2RNMbsjsDiNkUUvZZOH91oIqOZ
69fBHSycjwjhcp2MV38Rmrw8QxqM4Gc73TRew/9mzg10cGiS+EIzRAeuGbBQ
NAov0p+vr3aniXExmVnUFknH7IPthuHtz3KKirfSZL5IWTgN33OUI5PDVxbU
JqV50EQ3LLxyHG6122ryDtyQfTMnyZMmXMfeiafwrY85Si784f5KEuZFE0OX
izbDcMG3gW9HuilSW7dx2tmbJpm2+VMP4LzyWS4frjDSPKa/jSbLfWWeg/A5
PQ+Ld7coEqRwvD0EV9bmy+TwCL8TbZ/DF8PzJdk+NElptGsYgMtbH7WVdFHE
J5efssmXJnFXQrapzkt8T69v6aTIvvt9jAH4+aTYxIfw0jbrnkcdFLkRuutL
Pz+aLEkkjqo6V3nE/Dkvo8gPJ4QzTfCr/J/KVf+b/r5Tv7qdIqtnl7lr+9Ok
4x9Fzht4fJdHzcBNilRluIb7w+ueXTecgV9K4qdOSykSHf1g4iDcrUheo4b+
6/MSWAx4g7028zi83DlGYAD/7e79wr7WT+cvJk91mcLX2dY5TUso8qBlhR0H
Ps4YK7WBKxhZW0wkn+qROgXas+GX66YSB5spsp0f1F2N+gM2JLX7w2Pps6c+
NlEkpnWN0g0udrgXHQ4fi7cMtoJfGOSKxeiPwaEnQj7c43iwsElMkdIrNju/
gg+NUKHJcKeFGuWoiCJLjSv2CtD/sHdBjCz4rfhLISfgQQ7lMe2Yl/BN0EIJ
vCQ1MbuikSKj3gcYc5h7hcBp7Bo85fVEpjucKXGNMoHr+qT1SeBrXFKFww0U
SQ+suL4Z+Xk1e+ZKN5yZZp06ChdrVC/xkLctds2a9+D795U+fwWfe/nW9DDy
GXl4cNmY6t7i4iohvJoVpnMcea5zd9ihUNX5OM9gEW4WJ1qXi/xX3miv+aCa
C0ce9xnqEf9c5v479qUq/vniAtxb67Q8Gd53jbVHhP16erTrprquSJovm+2f
gn+9s7ZiCPuYqBTMacGzXpDIaPTBpCrUcjl89QatspVw4+/20o/hpma2Pc5b
aBLVpMU2gD9Tun7YgT4r2C2DQjZNVswccNKDc8fXvNuEuUzcKNZ+4kqTLONV
prrwOAcpywjzvfjix8vb4eb9QoHqXvNet8n3cE5ZXmS3C03y1PeLVXUWsI4o
77egzsLOjFB4rfvOcNV/fbOaKalHfp6vv+qiA89JIcfew7/fdMyKRg4XfEzM
bznTxNL8i+FxuElJdshIG0WypwPy/oAn3ClwG1Hlh7e7tQl5Vuyq31sMT44M
0JCrzntZT53HXsiCxltr4Ekts9bt8FxKvW0Ee/TvWG3AINzfsUVaD98zkhkq
wt4JPEmsEeqZ2GzsUQZvKu1fS2FPzTSU44fgQxUHXS/AO/3UsvjY644Ey3Ny
eBSv51w6/OQO32aCd0CnQdnPQ38ONmQ/PAqP5Nw1Z+Dd2JhWF/0BTtKM8yLg
Fqn9zNdwvVN99r+g/xHLK7248O2TS75cvEt2WY283ZjXvi52vCf8V5amYTP8
ZMpQDQvz/b/3+T9aQ8/V
      "], CompressedData["
1:eJx11Ps3lHkcB/CGmTClhq1Oc9ZlI9TQuISodXx3i0orY05IyTUVKTJYd0O+
RRnbPqHNptSxadtx66aokzSFjgq5FiEbIpaidttk3/MH7A/PL6/znOf5fj+f
9+ezLDBcHKwyZ84cBzyqeNIWaWQc875ZEx20LOn0Goa82Ju3UjWaEgstM5cU
+MUb7QFsO4aI14jT62MoMepZPRULP+xJimbx/qm36ZlFsZS0m5YVRMKlnWTw
E7zv7rpsaTwljcGpvDD4irids9PwvPiYiLBESlwE8vPB8JTLAvNJuH8L9zu3
ZErm7XW28oNPV8p838I/5OprWUkpCUrebuwLjzAZ2xQL/+eJ+ka1NEoUH8bd
dsH1q0/sZMMtTI+dbTxMiZOG+KEPXKe6t/iELUN+edDtkkApsU5oKlT612c2
OOrCxRLHlPlHKVns8zc7EL7uyOPStTYMsTcRKdIzcK/SrDKlmxs7v6qzZkiz
7kNFZyYlhpbu3BB49kAjd3A1Q5LOXJeqH6dkpMZELQLeveS+nA3PXNh9QT2L
koklQSk/wv2vm/9haMUQ+Wih9E/451CdCmWd7f1DrddbMuSG9tTHTBklDtqt
5TL4K3OthvfmDNE+8I6vkk1Jl1mF8BQ8547cZRE89ffnfGf4yVbBxvNwSWyD
o42QIWMdiQofuKZId0OFsl+XX4uGzBjiuoPn6ggvZV7W3YY/sTP00ID36rqF
j+K/iVHxM/Vw43+/9JiaMqTcMzo8CM65UFDaBi/Mf1m7VcAQzojYtBLn/7b/
e/4A3GXbuGXkSoYEfqyWtqIOvV1ivUn4c4+Rr/JWMMRvf6lUcYwSt4DKO7PK
74+V76syYciVz4tNj6KeGdNd9Qt23KzxGonR7DVmyDNxAl+A+nPGjs8uhTcJ
vQMk8M4Rr9aKI5TI6gT3DOF7OnJ01OFVrVP3eOhvRu9onRDu7fDNvAIjhqgG
Xtu0E3mQ/rqBvRYu0QhaZgV36l8ymZtKySRnl8wJHtWfH1G3nCGhfnbjt1Io
CeFcer0F7pmoEDTCr7f0jPQkoT7cdIkYXtzWPr8ZrkrnarxLoORxWl//djiP
HupogzeIWtYsRP6TSnbr74Bf6ZRpuuC/eef19phhXi4nbPL1h7NeDV3bCn//
qGSVF+bLpEFoGwA3zfjtXC18Zqit5ecoSp6mOdUona9zTmSL+7rbHjUYiKTk
kk3JNR94fn6eAR/1lBRVvnE/RMmJYf90pTeUGG7Oghfs7TvXFU5Jwj5usSd8
XmZwykH0pUSak08PUqITIXMVwfXb9niL0Mfk5cNdHgeQnx+i3DfDY7pDqizR
97zIqKseYZR41Pp1fQcXGgnvaiMn6qoPmuP2U+I6dbPYHn6qsdz+Pbws2ser
NRTn4e1+YQH/K3qUPEPeCm0WrPaByzamCAzgF41ShQbIrf7EU8ulcNul2VrK
vnNmzjiwkPPOs1VHWHB2zKCzJnyVHae+F36r5PYXPXhJU3MtCx6+aqDujgVD
WPsMXkbC2xbzvCeQN5W5w3Iu5m76am7fJ7hDiGGxMrdNFZ8eDcE9eIMmxTj/
SXppXTucvV3oq8D8qqXFt2fivjI9fxvlXPB3hSUVYg84FKm9OI36hHKtc27B
wzROrkjE3mA5x9l0oJ4ZY3HVZfDb4yL+MFxl5s0W6whKWPvjRYVwrawJs/vY
PwWnS3Pl6FeAfP1PufD+BdvChdhj9oHJclcJJf+3n/8DwKGztg==
      "], CompressedData["
1:eJx11FlQU1ccBvAkoB0XZHFERJTVxlCVisZWinBCqQXEAGEViLKFgGyiqI1W
QQlQVDjW8YJVgwqDYDRYqQsuFRQmsmgBYdCpAkoQNIytIkgRif3y1Kc+3Jff
3Dn3nO//nWsbkyaScFgs1ho8BnjeRVgWnJ9SU2f+Yq31kxCGBOoeHG/wp6Te
RjF0DC54lexyKZQhM/+ZfcJYRElsXenuTHh/OVsxEsaQFtHbM3sDKRHO4CqT
4CZpjyubNjDE9N7ClKnBlNSUGW8LhXuGNG0oCWeITeO53aUhlFzZb9niAZ+2
W12+NYIhnHqtUXAYJRXe7R1L4el2Rcq1kQzJ882NmRtOyZAbt8AC/pW5Q4Kl
mCEq2yTZqwhK2mJ/iZ4Gp8WVbllwW8fT9KaYkhdfJzmMGNbUmXF2GARtZIjA
8FbcwU2U/JXbOnsI3mM1NrwOvmrZEZk4mhLJvJ6VT+F8m9Mzq+Gb2/uarWMp
mTbpWdcB126uz1XCJfa1YQ/jKJF75xW2wCe2xLiW6dfpUcTujKck6XDGeL3+
/U9xe07CRTeiX36UUuJ+QDt6C569RONeBO/3KeBKEim5zHic1ntJdvspO3iO
k4vtr5spOVsYNaj3gKhtBy/iXCFiK+ZJEiVeW3cZ3IY3nK9duQYuWhUheJpM
yd33PZ+r4a1uqS1i5MZ6v65clUJJgLWHYyv8W10xLxM5j9knskWplOxKUz97
DB/IItwzmEt3yfrQh/DuG2y1Bj6cv8y5AXN8/eFB35w0SkLXXkj8G277cPvA
IOYurGnkroDXmBXxOMg/Z+LA8RL0JHXVSJkVvC+9qs0YvsG00aIh5L/1G4ur
0x3g4QcMf/gxmCGFvcOcCLiNMnlyNfysUFrBD0KvHDPiL2P/j3q9PH3gidmF
MnEgQ5oPyzkanDey8l12NLz87uIvLEQMafpNWdSPfBovue6Twe9275B3+DNk
p/GYphZ5Ktnv7Q/BP5vgp0z6MUQuEWozkL9FR8zP5fBrgq4ZN4QMOeQ79mpK
AiXWFWHR1+HtBqd0/euRv1Obw17Ml3PtyoUO/T51OVGlvgzJdKbz+9AH3r6n
fi/hs/mrjf5Yx5DB5JoCJ/Qn8vgxMgm/3GekG/dBbvkprinom+MRYcLMqei5
GW/BUbiVSME9gX4uL+4st4V3KXv967wZkrV6gVkT+nwnZ7TuS/jIdxOx4fAq
3zDXcfQ/fqtywB0etd86a8SLIZ6JskVc3BdxvkuFF3xx7grBR7hKoWmLw/16
4WToFAof8njTxsBZSyvdqnEf5/NNxJvgA8+MTBRw7b7EZUa4v2YLOx9FwZu3
qy6a4rtcB1VZHu47605XlxT+oWdH31x4hpftRmv8H6qTmpcmwBeRt5kH4ea5
veJO/E8EwkuD+vdvz8k8ooNXLNyiqfSj5Hu26n48/N2Toy+3IAfePJvllUJK
BmVVFzfCuyvMw/2Rp4v363td6yl5rousDobToOlTnJC/T/MeqTPcfVLO9oVr
RmeZGmFeD0zmz6r2paSUzbMTwM3qP5lo4Vah17NS4SstHeV8+Bv+qFSNuY9f
HXIJgN/8yf++HVwuHW0wRn8qDsfUSuFVgQrHOfp8wiQirf9/63+zaYKy4EE5
ao4zeihZclXCw35ah5Pt3mLuaSO/j01Hb3WyY1Pb4H6Dqeo/9f3vVFgEoOfb
U++nncF5k0/uza+Hn8vzSC/CvbCfq3p+Cvn83//5X+7Usns=
      "], CompressedData["
1:eJx11G001FkcB/BowpS2nazHSgyNhxU1Jqk1ddlIaGSH2JStjZqO2jiqldoW
yWQk/W8I66GobJFlLImdjDZFj9pTRA8eNkN1jlWW2dVgv171al/cN58X957f
7/f9Xctv934VoT1t2jQhznScNWPnDU3NapVuM/3Z99MpaZaz287YMWSstV6l
A38Q0q4sPgUXvbT2tmdIyhv3okHTWmW19jrrAYYSuf3iyJmfMyTdYfVkJ3wJ
bapXUEoMfAd3PYcvo4GNjfDXevwtWhmUXPJctajagSEdWt/4XIEr+Y3+bXAP
uwMrcxczZODyxuPZ8BvmlbqlmZRkfW2rn+zIkByJaug4fHB4VRgni5LCrp4v
9jsxpHjfT20x8FOBza198OJ6o9eSJQxJdr4qDIebRHks8jtDSf56bZOQpQzR
fyNsCYTPqYvJlMPdKi0HvuQzJE986bInnNctW2uaTcmbffnPbJwZYjXZy3OD
W6dZD3Hg0osjnBkChogdNpg7w61k99Uz4SOsTbPb4TLPbfn2cHm3uYQFv3t0
W3neMoYcvMvq5cJH83qiJvDuoasPgje4MCRrxQ6hJVylrm+VwasG5Oz38BO5
Fzwt4JuPOtcYwyt1dKcnLWdI4qacA1OuabwZfwH1emv7sj/A5+irFDw4P/5E
URP6JuXwTba6MiSG7W/pCLftqS1Uoc+bg3PEpfCyHuUhF/g7VmaoHjzJpd+0
Gz40o6RjNVxikN1lf5qSJfPLfcfh/d8PxvrA0yzCQkX0o1OWNCwUHvR2rpiF
PIwZF7ap4IpH/Z9K4AlP+nfaIT8sW+7JGrhX1rDeIbjv48fq7pOUHPNIa90L
txHJO1Pgd9LPluvCZ3npWHwCL+BEvCuA6zSldilPUFJbFTCWiXo7hN33foXH
h/uXxKVSYvhQ8GIWvNZVFHAL7rbHO6pYRolAfN8nEv08pxE+fAHvTCqMDUuh
RJXZMlmHuUQKrnFG4bfIKmOz45S0Nzkq/sEc/ygpqGIj/zcLco6skVKyWK+9
kQevqGstWACPdj+7YG8yJRapO9ki5ETga8jhw3WjbljlHsO7WkmjEuRq5KLs
nhc8Lj1iW1MSJecE+3g/IIezE+aWhcK1NePXho5S0hGdrU5DbpduvhUVCbfq
izwig/MqYnxykfPwufGieLi3kc/j6kRKMtQtZqXYi2DR2vp0uGOQW6A//NUm
m3UN2COuxjA5By7udOrYCD/YECt+hr0rDHAJ+Bn+Vnv/878SKCnnrTebwJ6W
mKUGVcI9wp40j8LjzmbPc4APR8gfXYO/j73oNw63E/x5YAf+Ac1tS+Fv8CGt
/QnRuD+v9XyLHP+GkUldrwJ+vTLfvA/uf3XQgAOftU747Dp84XrT5SGoy7a0
3ehHW4bURe8onfISrtPuO3BnL5/ZevCaQXX11D3988rYbuhbgHX0hyIbhoTa
TIbUwBNZWkbz0OeMDqUiGB4Y+JRVDn81URj+L1w/9d4IFx437HrlPLxv0Ye3
bZhXc4Xx7zPg2b7xn+VOzcUh83CV9KOHNPQsT4F3FXDPjCMPE6yXrvPh9Q3y
1MPwpKh3J58iP79oXib6wXeXqXN3wel3L+ZYIoeirfkep+HX+fMTguD62++W
SpDblVumj/4N7+NKFZ5w2/cP3d3TKClbwVm4B/XKaFafEzwjZPsxKfLvp7m9
Wg3/v//5P/lCso8=
      "], CompressedData["
1:eJx11Ps31GkcB3BEo9OFQZtliYqQ3KbpMi593CqiMyFrZaWLnZaKViK3eWKO
W7TqWxQaUdsirKVSm2Rbs6PcL4OzNVONxKQdbNauONnP9w/YH55fXud7vud5
3p+L6aEov3A1FRUVZzyL8GwWht4xD7rfbBAjFQ3doKBi4bFKtQ2B3UulmuvR
W82+NQlH5yxkpDTZEUg70hhI+/PRrtipMgock4WuvQ4E/q0sqqDdLfdkVzJ6
2hS/aXgTAee9jQ83onOd5PLCUgoeOkdeGd1M4Prv6b0s9P1lcR8brlOQaG9p
rdhK4Kj6MhsO+j2DAS9JCQUWE/yELg6BM4ILrYDu/vHL9r+EFAjeuzwtdiLg
MRJUvBP9cMC+ZG3006tYof4uBAxqldI96A80jTNsrlHwJr5BJNtOQP+gV3Qg
unjctdenmALG7oIQb1cClHsL8wh62ZnV+vOFFCxKOEEy3QjwXtSdO4Y+s4Vr
bIge92qP/VV3Al1eAXqJ6J+CZMZtVyiQdeyMSvIg8LZAt0SA7pWhpTpeQEG8
H8dvjSeB+Q5N4Xn0wIdyWIoeywh9UoguPrW0U4g+rlmU33iZgreGPkZD6N7R
NRMV6FqCb5qllyhgKVKVMvTa9tUFd9HZTe37FigKsmWPE39GL59+KfkVPV6l
54gpeupg64dd6JfkJxlP0Y/5/bKr/iIFLWXa8gq8Z9WtLTsG0DsTD5gNXMDv
7VK/luK7VvTnZsnQm/5cP3IUfSBantyHOWSKJi1H6XqxhBNzeZhb0w63H+jc
mLEek+jLfQ1Dvkc3cXjnuxMITN6xZE+jJwZUpl9E141nWzdhXYZGXLQ+oada
vcwxR3/f37ltpTOBFPapYrWv7jeLFhuyrdAtPgtv3ONI4M2P1S9U0cMjM/nX
0EX1o/UR2wik33WRLkZ/l5cjKEVv44mfnN5CgDCYfNpdb9QYrcT7Rw81rDrG
JuBwIsue9tNTh1sy0dmM6cJgFgH/tWoKDXTGA27UPLq619zgDnsCI0HPr9Ku
YW60MhpzG3/DDWbbElimm1myQOfs6aDjjzkrdJKqrTYSkE5GJM6iXzfSVrKx
XhLB+THbDQRmBmdNP6CPGUSU6GN9exo+L/CyJOB+7qeM9+gNz//+Yg592K/O
I2k9AUOd43Uj6E76vEhpPgXSidLbIjMCL7nrTtB1+RDRF/YY+yf72uvtnHUE
tNrGBuk6bj13Wa8U+zBccvtm5xoC8n5nB7rubfWCrqmrFHRG1yblmRJQqtfU
NaMz2HPiXuxnQxvVVr4JgTie69o62t3dxGtwLuRQnV68moCNm8vFcnQ9H9mw
Gs6R9ph16YwxgQzFfA7dt/V+k2VydGawiW0KuosyKYDuc2EHy8ce51dSvqkP
0LN0xNZptB/nmmrjvP+R9FqVg37ZNZcfR9/zUWu2Ev3jZFgsD/1Q9UnHSHT1
jpyEdtwb5nK1hS50+1t7l9BzWl85n8/BPROWxdKNwnsWdUpE9FwfzC66vwr3
laawu9AR37Vkgp1G74EwD+asAN38zOYUT8zBLKBq2JPO/7t45RQ6b0ryKAZz
25Vr98wJnWl94GzoTQoahhPu9azFPldueLUVXWOfImg/umVkxotgrEtM5Z3F
tugbFXVcMXq3LbNvEdZxSWsY34r+T4hjxW/owlr3gGcWBJb76nlvQLdoqzrE
Rc/jrQirsSLgqVfuS+9hm4nuEB/0hH86/MusCfzffv4PdbfL1Q==
      "], CompressedData["
1:eJx11Hs01GkYB3CMXCYbGxWZcYkRMsilsVG9khIqR1ndlNFSqczZilaz5Z5r
5T1dXIsVlbVbgzVGlClUWknb2aIxbJsU7expUaMd1X7nr/7aP35//D7nd877
/J7n+z7WUYLQaC0NDY0leBh48uK07VzsJFJfUfXf1xdS8vbHudEHMihp+ZB0
mwPnu7stb4TntdOygGOU8I9mx7Dg2im7z4jg3mbiPnYmJcKqOQJTePKrjA5v
eHV50dOX8HDz8jxDeOUV3/2ecN1eN1ltFiUfU3xOT4ef7nSpkLpSompOkQmz
KRG9H2N8AZ/MTboqd6EkZ19lxeIcSmxNrmYw4cOq9Yrd8I3ud5V/wsdN+94Y
wKNZhVcynSnxCpME5+ZScnmrqE99bm1zd2cVlxKG4bld9nmUjCa23jCGy8JH
CtudULdWzcyrcNk3FuXq+tv05OZDCyiJFY8FWh6nRCN+2yc2nFtWM8mADz0I
DUyHW7hl+DnAlz91NhA7UBKUqNX7G1xQd6DEFb6rPjjkiT0lF8KnTzBPUHL4
YNuTJXCjQ2fv5czHf649eN4JzmNPMwqGhy+YUC61o+R2su8THty2rnAgDP6v
w6YpPuezbyyyNYyBV11pPDnHFnUs7nOaDRd/zQnbDy87Mm+Djw0l0pSHRgOo
J9Uj5lMaPG+a2fiYNSUs+XhzMVzxqrqVquvXj2wzhYdyYt8SuHXD9u/L4R5B
kSuXWVHSv8Hheg/68/FxvKcIXr7sH2GMJeZaZWMcCGerIvZK1f2R+zJPWOC8
3nmuEvS/R1/bpwuuyA/41hZ+TsU2MYcX/WymkMO3r2phrmHjXeh4bCfmKKtu
kozCj1+qtBliUdLnPaWsRx5WPNYrHYcbDHcLRuHPPtxUjiA/32lW7NCcj5yY
31mcCj+1Z4nSEj7Bs1vJhLed5x3Lgvtnxb0LQw779QsvGsObRIOOJ+Erpra3
JCPPLV2rs1jwd2bN8WfgOZGT18TIf1mi/CUHXtzZYVaq9ksByc/SKZF4MoYc
4PkBHS+MUH86P+DoHPjciLJBLlxk0libAS+QfRCuSUO/b/oNuMIXTTTcew9/
EXjkdVEqJTeEPKUzPKvuSHQ/+lNicmvzcArydcieOsELZnRfvIE+x4TItZbC
E+5OcdXnRnG4nuWYi9s2BeunZEq4G8eD1HWKO/RjUzDHnkyFgAO/a2X62gr+
UbDNP2oeJWkGw6rqJEr07nxlaA6X+S8wWY6ctOjMcg+BF6z99b4hfLZdMTMf
uVKyQi9owiMeLkrQg/dUJnbFIYd106QZnUeRd+d0iUq9N2rTGpqR22zdVT+U
wmu1diYp4BetfKku8p/qab+Lwh+N8BY+h18YEXkN2X/+3tNnUvgI3jqyTsfF
kZIHqjTjevgo397jDrxrymslE/duRoTpwtdwP/OK9Aa49Vh+zXrc37PyW5nO
qNO/ydjvElzv0ZY0Lu77L4712afg1wx7UgvhAydEGrrYD5v3XG5noD+FS7v9
suHpfYdt/oBXSEu8kuCbIvp1E+HrYgaoBHsmqKTA4kv0nzPo0rQXfj50331r
7Ks4dk5CE3xLvWVsJFyaKtDLhZfMXt3Nx3xbdcXua+GCt58MorD3ZvKeF1gh
D8nFww/94VOMDbO2wkusvPmD8LOhv+t4wYv+GiXt8Dc6ITuqkav/28//ARQe
sE8=
      "], CompressedData["
1:eJx11P83lFkcB/CJKM0mFdFXlEpnE04UTeoqTalNnZQhuxQpWhul1rdOIWpd
tStPadA5SZsvGUI204YVavNlZhlEGSbfi5BomyFm3/MH7A/PL6/znPvc+76f
92PsFXDAR43FYtnhUcczXP6sf4+5sMxLXOmX9YYhCqO2yKZeShIrfHW3wts1
RebzWhnSyj/5V9Q7ShYks9os4XV+svth8J9ZnQy3nxJnvfk/msD7elMGOuFV
8obP+h8o8X6v57YEfrhoyR6pah1Nmd/EICWxbkfTtOHbuYM8BylDvHJGk6XD
lDTqD/E14fNLd5faweUBvjUlI5S0h9jHTK0TlnHKty3ZCL9XF293d5SS9XW8
0n/hww8dnljA/TdH3j7zmZIRk4L2MXj2tfVvn+K7tS7D/hu+UKJnedpmFP7T
D3TDTviFqr7cD3JK+ivUFn6Cd9c2T0iQwzj/475L45QYm16MGFG9v93UwhPe
6G+yf3yCkr3L0txVHhZ5r2LgNUOsrThWLpOU9ClTcr+o1pGHu86Cs19fzIia
okRg++iCav/pQo+Da1oYor31cvF1JSXseG4EC+f1fLBfUNTMkN4XTlJ/Vhz5
PuuWZCa8ZnVyT/MrhrzSenHDeFoccdpUItSH22a6baRNDFnZ3qKXCldyVvxt
DHeXkLbsRoZMmVl2f4APPj2WuhZu7xi/RtzAkPQ8k22z1OKIr7CLZwNn5euu
HZEwJDxzUfYo3r/KuTPoABceLezWhfPu6xhmwc+x38h5cIWBz+3qOoYskyUv
N4evC8sxOw4PmajPHP6HIeIBj6BL2P/YS3uzc/BjIQbuevBV9UnOmThvy6NX
g9Hw33NXZnPEDAk1jtFIRT4Vsam7bsLvnjAK9RIxpOQWNyEIeUYa2u5MV53r
SENtbC1DXkYHFxl9pYRTziMF8Odm7U5G8Or3X89m4L5CfSWelXCpge2vu2oY
csxhqkFHQcm3s77bUA8ffSKulFUzZOaka4o35kFrR7d/O/zcqheOwfDOKXn1
HczPcm+ZdAAesNlvUhteukjfX4x5S5qMTlLAJTfDhtOrGLK0flLRj/m0L3a5
N81CWNbVNh6VAz8/j1+l9ZES6Tu+QAsenq+xoBAuPZXANx2ipGHeaOlcuN43
kXl/wnPOJprsQF+6V8dcWggfPHSY/Qz+aaqgPAj92m6t6bMYzuZqd1hjP7UC
x6GH6GOYdXXhcvjjX8KlHHig1YkMBfqrWLurcQVc5P5mfh7cta+D5d1DybVC
q4plcPMQ7pUE5ON9onF8oIuSLHVZr8o7NE1HNJBnG03Tie9E75K4iQbweEmd
Zx/cOfD6Le8OSubuPnhaHz6bb1nihvs6ldt9/uBbSjrq71fOgZutDp6zCffL
biSuoTJKgu7GBc2Axxw/3LoQ82C4JTGlsp0Szfy0TVPI0+ZI4ns5PLwlWJ8L
17ljZzYGNxb1ODZj3ioDOTsVbegvb+mDLnhEXkXOCsynIPfqu1a4QuPmaDN8
2xZxPgtzfqB1VcQn+MUrCo8auENypVIKF8XO0LbD+mFNscoSuOEfhzKE6Ivu
yKnZRfCY4huGD+CLlIsd1dC7fcWCMi/s/0l1qFEK3EfmkiKFi5TOPXY4b2CH
+pE4+IymLKfH6G+0x5gNF/lENgimh8MtRNejfkPfxdfYBWeQ544FJdNPqnr3
Ud3cF/8Hb7k04TnyD3juOtsDvncoT1AP5+Tvbd2C+/q///N/OhQF/w==
      "], CompressedData["
1:eJx11Ps31GkcB/DZGbkUjVQu2wq5RAjJnfEo5ggNo6WkTkmaNBtF1NittW2k
sua78jTalI7FKY6RGuVy0rSqtYqVis1pT+Ve61q7NS6rfc8fsD88v7zO93yf
z/k8n/fHYldyZAKbxWL54XBwrsaKakWcOqWJ67soH39KxKFz8ZFzUjLR6q69
He7p02ZRBueNOTh7f5IS58hthUJ4ucIjYCGh5J3lYjMrFkPsQ+8nBMEFDoaK
dPjl2GF+E3ztwGAND15x57v7B+DPWgRc0WcMafgq+owHPJ0vPrgPrlmqb7yM
zRCL+urfneBhfbvOx8M7hGF3/4Qb+5xpXgmvnDTN2AaPMSwLucJhSHLt0Qxr
+Pj9Rvch1MniLFZkaDCkOPjEQkt4iX3z04Pwqxe2HY2Yx5DJnO58G/iBxdGH
83iU/MU/Vm6nyZDrF25/VH/vu0wyYgwfid0+ztFiiE1yQ5H63s4Ygyo3P0pq
lt+99RxeuFuWZA+3lczRSF9KhJvknyq0GdJTlRStrl+0YSo/2YeSsY08rQwd
hmQe3N/mBT/9SWeFmTf+rzSR+85nSAxv5md1fzIFO9/7eqGe9dFL/obvllqa
h8BjbrbsG/WgxMix93nRAoY8rmE5R8OT7iQlXHKnRO+X3+bcdRkiaTd9uQvu
HSUeDXejpDeNG38XHpTwbeAB+KudHwzZa1GPXtUDLz2GXPhwTXxM/e4zFiOK
NZTsnyrVKYZvMKoVnYTvsZ1I7HKhZHbd8MNRuGPicTeZ+t6+VsVpZ0ruaRTu
t1vIkC0pNvOvwM1Uvw75O1ES/vDU6lB4uVhI6+FNXw42/uNISdvU1NhW9feC
eqsH8KGgzrjP4TF9V+XhcEFqc/oTeDQ3/Ia/AyXLf+rY6AI/0rGh4zXc2l41
kmBPCd/RW3cK9VR6ikYm4MU+Hx/nrqLkYtSiiip4ecG8XpZGndI1TZSusEMf
BvyGwuCTO84WLoBfP2dYEwKXhzb1daM/ltmpmkbw+lbTH17bUlJaXfgyAp53
pMDeEr46JH6+BE6nlT230H9XhxX9dvCAQ04fjsHfF/su5cKF+oZcF7hdwT1+
FjyIa8ONxTtKZH49nnDlzW+scuHnLEOCZZgH+d62NwR+3kDCOwvndI4pH2F+
pOH3nNfDZx/dTtdGnfVCPckM5m3jZlZZEHz7Uvfuo3DT72uvWcMHCsy91sF3
NA4GxKEPuiW9olDM89vKfe2BcBVboOyCT5u08VIw/5z181wD4Ow6P3YD+klT
pWlFyEtUbvghP7iOtdz6IvovCHbUbUW+apteZHvB24NzzDPxXk+lt1bMIo9l
2WIjJ7gwblzWsZqSxoosiRe8f1giWAUX6GV1Xsc8tMQUTGQi7+0K/rPl8K8z
53ayMVf91m62f2A/eOzNajBQ37uyJE6IOTRel2buD9eesF2qCW/eKxsudkXf
NmetEWH/nFhZelmlzkssr34U83xjiUb5EewrHvdN6lu4uaRuyBu50PoiYJn8
XymZlItS1PPT37lVbIIcRbxtzdeHF+3Qv/IYbvjRwjHBk5IXXW66slkpaVe5
tSrVuWuqnKlGHusSc3I2wWteDdpWw1MCGdNp5LcxZJzvDn+lr5VfAp+4oUp/
hrw76a9qCYNvSdnkSeG7I3WtrmE/qHokZ07BAzRcurPhD4s1h09hn4y1nF00
DX+y51LTYfjAyTydeOyf/uM/ns9Dnf+3n/8D+HGUtw==
      "], CompressedData["
1:eJx11Os7lHkYB/Ahh2FXUmRVV8kwpSRlUnak39QqKtcqdCKVTam0o3RQdHLq
7HlSj3O1W2O0M4whES6HDsqhWZ1LLRZ1rZXTGOPQoP36A/bF783nxXPdz31/
73tmoHB9kDaLxVqKNw5vi15aYP1wYcWa3O787HsM4dR0qLU0FNnEK1FWw19z
si0mFTDkh3/O1L0apohiSf/WYvh1cURAOPyqvCw4a5QirPGvLsnhqy/djhDC
L1zhPfBk0SRj1LxEDN/TJ0jeBW84KvJ006LJkGFlWzp8QdOG/K1wS92cehdt
mrg6fjC7Cv+4tqXOB+7k6BPEG0cTR92KpxS8da7oUCPqvN/uPMFahybt9OZz
cfBmWavZWzjpP/z1e12aTJpZrIiGFzaJyXb4xSk6Ol3w4hn8zDNw1dlnyvZ8
hszfmRlSoUcTWsT3HfuOPrXwMBte++YX02h9mhT0tsfEwv/uuut/7S5DXGoW
ljiyaVKaukx6cayeaTGL7uYxxGiPheAFfLN/xNUE+K9iu7KXuQx5IeGK/Qxo
0vrd+O2p8Lw/S8xUcoYYLPG4UAdv+zHY8TZ8/ka+nymcbRB62daQJsah5EsW
3CsvXuiUw5AI38fG++HhC1iSQnjPnYIDm2UMMfFwyE2ElxywiXgE5y0d3Hci
myF7j+esy4BLjxDv5/DRiU3et7IY4i71NEmF72lXGTfC8xUCwyopQ2I1XK9w
uINLn7oDfq7aTNElYUjne42nMzxnWm/tMLxOeDR4MjymRi1pQf02X5/cMRop
rDBL7Oxx/YMhbSo19yB8pad01Qz4uQ7B7OA7DNm2f15UF/rDXxpWvgC+VjNk
lZDJkKS+Y5G+cO3eVrUr3MYttnAO3KQ5/40I/ecM3FCuh3ewY4w3iBmi0eLG
f8a8MrzTvXbANwaMyrozGNJ7sDzDHN7jt+znULgqbnHjefiLz0HDzpg7d5es
/BQ8h1SFWcOjefyJ65AfYXzNOgp+6vGOWeUihiw3pJ7uRN6K7fk2KXABt0yn
Cm6bPcLfj3wWfJo0eGvMj19pfg7P7daTH0aeuXKfmiz4ioqPpfXwwXcSzknk
v5YeSSuATxxMT2mBu7x/Oc/oG/YoKKmjCF4UWd+zCfVseBjwet4IRRwGbs5+
BLfVrnHbAT+ypuOrH/Yx6i+qtRKe+SVq8jt46itpc8oQRUQTbBUP4GmfzaYW
oz+L81I8BgYosmKcfsDYd+I515vs0U9Ty4Q8YT9FtJWsmY/hfkbJ7bfgwYrT
1yeoKWIpS9KUwkdiPpnGYF7e2ssvvFFRpJP3ZXohPHDIIXUn5uv6QRle1kuR
kBD2+hz4symalT8hD7ah2yRvlRSZYuV0Ugy/V/27Hge5ihR237CCX7MtrUyH
dym/DbCQw8xmj8S0HooUFR3SS4CHjUwfboBf9uv0codv8fnGi4MbLEpnFyPP
3Q8NjOfC3bKr/I/Ba62ksxjk3/nlCVoAl5fo/rZvzP+1CBdiX5affGcaC48c
bpT5w7d8vJnrjv2qn2qyVRf1sLWi5Svh+wZEbfHY0/Jjqt1yuMHiFn0X+JwG
+9uDcO6g5ZEr+N9qu9LgOXBX8/PCOuy7bG/Uk2T0J7Cx+pM5vPT16ZAM3IcE
9W4LRR9FnJSbzurA7QTOouO4J4r6S/Z26L9XUlhZP/bIui/5fDPcXcSplGBe
97et9m2D28zKbFiFe7U6bHyDO+b7f/f5P4fOFkI=
      "], CompressedData["
1:eJx11GtUzGkcB/AUum1oyhrDaZZElxnTjG5M5UmlmIo6yXbRmVObljBZ1yiU
NkyXwb/+bdbpglWadJyoMU7pYtpqp5VZQ1qX1mm7iUQ0VI799qpX++J583nx
e57z+32f35IYSUicvp6engeOAY4wmm0o6lA0DD54/dLUlyaur4aJlx1FnE3r
eU7wiYyX/E3raUI9e3xWBp89LOVaw6+mlZrb+NFkV2jFvEG4YIXkjgk8NvZg
93Z/mkgLxCPf21Ok2rPP6d19RcMTtUpeuoEmpsX/8rrhYsUXsxdwVXjO6aMb
aTJf6iOJc6BIa5jNozZ4a0/YD0Eimhz/LmqOPocitjsvVlXCHcc08uwAmmhy
OM+vwUUTiW8L4KEmtTPFgTSJyVTJg7kU8awYPiKFX3H4JlEFH/Qf6Jq7kiLN
3DbuAbiIk7ueDqKJQN3b2QQfnTzFi4O77tfxxuE0P8w5i0eRG5ZZyi3wdJOb
v0Rvogl7a3HMRkeKsJIjrgTAT8sL54fBjd+WyIz4FDGqOc/1gKfdWJByB54x
FlH7O3zY5mSxE7zMak9SFXzE1ECZJKCIQhK5XwCPDu2TesD9Hu2s5qyiyIYz
pyT2cF82w9AF3n1Hk6qFWxbFhtrBtd5/M5vwzjojb6u9ThTJK2xutIU38jrL
g+BVG9uk+s4U0Qm2lvDgz9v/DJGgD97eivwUuLi8ne0IX8zw7J1EP1kPM658
grvttipwg5e+aDZjw5ePLlka4kKRSmnTuB+c8zlarcW8dMmTkmJ43qX89GD4
Y27jqA7zlSsW5fTAu+rvs6Om6gQGLl4MN+M+DpjnSpHA5VKHPfBOb/9wLfJT
0MuvWA4/l8zyT4VPOsrzstdP+9Kn1gtkcOb5S3W3fKfrLLNqbb8MjxfkHpb4
0CSr+o+6YdybxCKUEm4296DAwZsmF923ddTA3xn/vPMBPC/BYqjfiyY3XTNS
f4QzdmSFDMCz5zhd+o3QJJgp1BrAlazwJj3kWVRX2hm7liaPLG10FPrjWy7b
ZgFv2DfriMwT9S+7v2LCI2YytnPhAzImZ5MHTWxf1aacwlzO2OWbe8G1mbXq
o+40GVUyzMcwx0XBq+5vgR+KfbrjmpAm4qUKnyB4Uv8IIwF+uPlQRucamuTE
R/r/ipw0+g6FnYDPeXi7ZBbcTVwvGkCuXDqNW2VwvTWEU7IavsKCbQ9XWNxy
K4J3iQ1q3OGibVbvE5Db6+ODZZVwTYD+2iduNFk366XqOnKeliln3oW/0/do
2QcfWhKZ/xb/4uyiGG4LXNVdRJLhljVZO/hw1sRsZQc8Lm/hmnS4ZkZkwjH8
u8GFUT5P4D+98F+VBXdgDO5S4586fxgb6oK3VvcU6uE9V99YF1rDHYaPmTyF
J8Q/k+6Hz/7n268Z2AM90hnFU67UuOr64bf3tcs+Ym+4RmdWTHnqh889EehD
796gzRL4nnv2AVr4XaHq3mr0M9OxKPEr9tJIX3ZMOzz3Q4c1E/3n7/qr4QL8
i9Im/h58b1l6+kc4U/g6MQBuYjQjQjnVZw9NrwZzHF+mO2EJH3jjE18GL79b
cJuFPLyPmmjot6XISvXC+ovwsUNeFz7C+3QTYg28MIVfdmbqPUfNd7t7TfsD
tfHxA/Ae4bG1aeum67gburSL4YWH555rQW6F1KfNxrj3//bzf4P91AE=
      "], CompressedData["
1:eJx11Ps31GkcB3CXdR1KF9IV02SKGZM7e1JP5RptSyiUKE0Ka7IuaVvaak+X
x+4669tMq5JsUSF1dJmyNaO2TWWJQkIMcouSZbCUfc8fsD98f3mdOc98ns/n
/XkstscH7NRQU1Nzw6eJj77OMSuwlMr/dddhyZYzpGmX9OeUF5QMme1wE8PJ
UESWxJYhn5MvNBe9pGT6sOLuEXjbS9+T8XYMUXvxyEe7npJA8uxjArz2zt0w
L3uG5A4HHIlvoIR/yDJOCL8qFDpdg996qWWqfEUJh2PetBF+eFf7xqMODDlR
opma8ZoSq5r8J57wT+uNpR/gUdKVBXbNlISKSn5zhRsL0zQ2OzJE4deQ3dpC
ST3zy3I7eEKQssof/mxt/08ZrZQky/IUHHhLtsHUHfhBYbAnUVByQ5aTshA+
5pL3tBQuUEa6DLRT4ju10McEfmtIq6IY/jGZ5GR0UmJglKFnDG+93LDbFe5d
2TrB7qJEt4YnnAbfInvaZA/fzc5/lNdNiaK4aMAQzrRmcuSo/+qM/AlWL865
cU9N9fuIsU3tfvBekYM4qI+SErlfwGx4Tlj50Rj0p0zdY+6Rd5REVZ2MVv3v
7a62RePoc0GHgXpWPyUxfK6hqn66PaXNFJ7gwXqQOEDJ4waB2xK4z+CIiSvm
FXm8u9vpPSVXNP6ysoH/EcnrD8F8lV8bLKuCR46PtTrDQw2/if5OwJAf9mSv
dflAiZ9OnXANfJq3PnPWhiHHTEpGEuARk6IA1bzcmgMe+/MZ4pT08dhxuDAw
OnwbXOls5pLEY8jb7m5BLLxGnKHcCx809K+xskZOyoKvWcCP1oq/TYc/MeYl
b7BiiHnt5+prqKdeg3chCz4p+yJKexlDqlM2levBszq4x/Lgq9RNHXhLGTLc
l3jaC/fd+WBITQqfO8W377RkyFhXn0Ew+iMcuRJUAQ986+XEgnNq1m0j6OeE
T7ljo+qcUx7FdksY4lm61YSF/mcc4Nzsg3tN3LoUymGIQ6PN1es9lIxaPkid
hN+RxLccWow80Dn7vDFfrxmpOnpcqdy3v1F7HnyheWK+7C0l4VvO/joffjr8
y7Y1bIZYd9z2Xob8bFjMliyFX3LXjX1twZBG3YbwH5G3HvnMh47w+aMxpglw
vqQw9lUbJfo+zTw3eOOzJZ774J2pk6nWyHP6+/vp6+CjfGamETzz/mhaEvIf
UvR7jz/c/Z+YVSYqP7OSVjRRIm92ZYXASz8X2i6Az9Mqi7LGfnWwU6ZHwDNP
nNdnw8XKXM2T2McDMX0DKo87XzdTDNfZI3U0xv6ypOzoSHhV7L3HergXd3zW
RGEdJXVT7hEqvzddZPQ9XPKueG4E3oeR4JhK1TnqlRc7P8Ar27Ne2OM9SbYI
G1bVY3rZr/o5+raVpeji1FJi9nR1YABcW3P/rOvof+f6VIu1NZScnjdqq7qv
lnfIu0zMazz77M2055TEx/3tthq+WnPHbBHmayewLu2tRn/0y6kzfH8179xX
XIZsv7hCnA7/U9I5gw9XZG0+rMoPb3597hq4wJlYLYBHzfEI24e8cT2mGfLg
8tIzJkbwRReSewORz57jsku+8Ny4goOfkIcBX1boOeS5gq1pJYafWsnSGoT7
rthbth/5TzN/Y6CDOi8LJG+a4EE2C1oeYl+CpixaClX1c/oHK+FGol7dHOwX
b+uotwj3LSo5lXRbtS9eLMMu7KNtIrc5Ev35v/f5P8uFvbc=
      "], CompressedData["
1:eJx11Ps31GkcB/BZtS2lpRmZEJVGt2kqhMqlB2k30jYVq9FOGY2N6TbuDbk/
ZmZX+E6FozbaSpt2EZvShUrLFpbuaYWmbRSnbMmpteXs+/sH7A/fX17ne87z
fD6f9+eZIdu1Tm7E4XA88Y3Bt34gT1wnO39lyXPNuc1ahvQVDG6gtpRwT3Ab
KuAOK1T+LXDL/e8CRNMo8XC1tzoFn531/bJmeOulDrOu6ZR0dZj5HYMPvdpp
0ghvTC/bUWRPSbqsPvUHuFPzK309vDzkS5+1AkpabM+XF8LbW+3PXIBX1fqX
mM6iRBYY9xcDdxk/rK2F3/S0ibk+mxLFuHQp6/cHej/4wA8UR86MmkvJ+0KP
vny4RVW+ql3DkO8qdf4ThJRY1y9+xLqp9P6KTXBXyWhNznxKYqN9phbAPc1X
/bpHzRDFhS+U70SU8OOaylj3SnkzMA7ex0/OCFhIid4k5+xh+JFR20FBNkOi
izkmexZRMm3Bw1lsvV4Z0qW+lCHbh6IstY6UHGpesKgc3hZk/TosiyEdXIdW
hRMlSyWiw2fge4NeDKZlMuSuxRNjR2f8z3lWz/Z/pM82ujSDIcPemq4GeFav
JKkZHlIkaV+bzpCse2UhMxZT8sDzamUHnPO5n6UyjSE6cWOuFF5XOFr8CH5F
oOxlUhlS0qYPlcNt9dn8fnjdg7BU3xSGZPg+m+wNZ0SWzsNw9+ttM+V7GfK7
u/HilzhXcc1g4IRjjlKxizqZIXlbTowo4FZHJgm58JjcUutlSQzpzeFVNKAu
Q4/1ezt4wEnZ8CYVQzRfn9T1ow9THA8FzYcH18a4pe1hiN/11+H96FvanAA7
F/idh39s5LEe3JxwE31uUlZs9oVLVpV7uSUypP1tkF/2AkrmblSZBsKbrEzH
tiQwJKZjTZA95nW7e8KHUPjcRVz5YDxDIlU5H09h7qEj35AIeENWZkEmvCLe
4+yUeZR0O9vcVsAH7sn3q+HVC6uv7J5Didvqg4nxcOvHt8ZPhW/rLB69ihym
xY5Zkwy3vGW+ZDr8367Go586UHK5+iIvA97jai4RwB+8Wd0WOJOSj29cnqvh
T7Wl2XPgMs2Pu5gZlEwU/1ShgRfs1nsehh+JlBR3Yo/+lPBdc+E3a8OdS+Gc
+N4+ZztKgl9EVbLuJbxUw0O9gYn3LuumIm9GFy+z7ml7vC+b9SIr8SQbzN24
Mk0LX8F9oopE34J7puWXWVEikhqG2XOVbox3F7xTN6xaN4WS4qOB/ez9H3G2
edej/+Ne078FfEpOd7omsfVeC3IfKcEcZfu8eiwsKfmtj8bEwbfavfJLx9yL
wsxmOU2mpHN9ovFOuEZ7yF+GnGTKBIbtFpR85uNq2AL3IPvDqpEr+XML824e
JQX648IQtg+pI3E65HAlz0+XDH/sMundGni2ftApGrktFfNCl8M7HNTjCfx0
bNwnV5HzMc139y2B70txyWXzUzTR504J9qJ3S8LdTfAWm0HlPPgNocgpBXuU
X9V6rgou+kV40Yp1keZcC/aOfLt1yBH3dN+wI24ivL81j5zCnp6JavHpho9d
KS9j8x9qfiMiG3udI80Jq0G9TlzFyiHsy7p/jgnC8Q7khRodqEF/EiIik57B
aaz44HK8G+qnZll30M+qk8qlXfD6BmObn+HMUPwFPuYywnfTtsGNjSIMarxL
dQ0bKpKskYe82MJGeLzwq5dD8PwmxuQt5vt/7/N/0+6+/Q==
      "], CompressedData["
1:eJx11Ps71FkcB/AJUW5b+6SGJcldqF3tZis6m9is2jHT04USCTuEkiRbpmgU
tXPcyy2XFJFLo2LWdYTGZXtGK7ZNRaaiJdputoe07+8f0A/fX17PzOd8zue8
zzHy3c/zV2KxWA74lPEt7L3vF8GVSB84z1N58DslXhVVdWViSiZZEQWBcDq5
YXVlLSV7zaqrl12n5HBSWqgXnOekrjhZR0lIQqR98w1KwvT9eVvh8p4VhVHw
0/21goBqShRmb/s3wtsNq95trqfk/nTtybkSSmKmTcMI/My4Dc8JHhk4IGrE
+gOp3G/s4QkSx+f2cGW1jLQg1FvPVX+zHJ6xozHKFr5f0JS1sIGSLy6qBljD
XT23jDrjd8f2BfdXNVIyJ9XqK0v4mnEdsRf6d7Pcc/4HKSV6enkGZvCORPbf
EVjP2/k/kbQZ9T3uUGO4pU+8iKK/PdxlHy1aKLl+2inbCH6suED1Sg0lThmu
Q2GtlLyvnEwygRvKhi8pblKSVPNHY14b+qquNrKAJ5/tUluCudQv4c+vuE2J
9uCbUKbPvlI/D68qSnoDjdjnZJQQPZOsr+GDJi0t2dcoSVRlzee2o6/mPw+v
gsesy9Xur6DkzmjPzDD8+dGXB9fD92oFt6wto2SD61T+jg5KCmeEGj8xc/51
ylFQSomdkB+eDs97cnxiG/z61fSdQ8XYv9+JyYtwxciNpT5wF/HUQ9MiStwf
lB/4Db761f6tQUx9Y7f8oEuos89cbz08Ts2sKApecqG8824BJdbqM5F30c+H
geDwOLiu+dEUdj4lo+d12+3gE7p1i1Pgotu+YT65lFCrQPVA7Lf7TO6pPHhZ
m65DSQ7mu3GYdQTzeZQQJyiHC8TeDa+z0OePpCIQ80znN5vXM7kKZek7ZFJi
83Gz+yrM//jWT/u64JrLpWvjz1NikOHBf3aLkilW3mgf/MvIbSVV55BXqev7
cJzvrNmx5UNMnqX1/EfpyKFv9qe3TZR0vtd5PA7nPuuomgO/zA6P8UZ+TNm9
7lPwa72HOSvTKCn1FAnqkbuzDa9DlXkSafcKb6EjfMbul9WLkLs9Zq5yDfhs
5bOarnChxYEwP+Tr1mCf1gK4PGJt0BZ4waBakAQ50m/4100fvuCol8wLriXU
9tXEfdklLRw2hA+71O5yQT9rTPsehSBHK48cSjCHW23Jf8eBy2we736CHHnL
uuOt4NM9DrJV2K+qbeteX9zf6JFpiQV8MV8n6yPm0zIePfGhkpLM7EUS5veT
xZetnmagbiantQL5cqn1rWPqx967ILyC+QvFGsYny3EvbVttTODHFXMtg3Fe
4krNrkPI3UGZ6jjT55rmqZe2OF+VhCzBmauUJOt0turBU3jJold5lNwYORIj
Rx5nJIkvmDlsdzyxWYz85PyjkWQPL4jYxNWG53TUGIYVUsIpMhiVlVDC5ldu
UoJ/u0tuo4185kc+2y6C50YF933AucyTmxV1Ic/D2yQp0XDpzXtkDL7YdFZd
AOqWcQ300uAT7AEt5tyfOiQoLUXf1j/HHHwMbxgYM/gLvvtF+M6H2OfbpJBF
HPwvID5y7Da8M+SdySHcR+ETra4xuOdM6bpauIci080acz/lzxXVYL858qbe
Mjjn4UisAu/ji77x7CtYJ9VTuT+XydWpUqUsvA8qbUNvqrFOdLuacRJzj2wL
UjjIQVvX7IxRzP/Ad9SRuUc+39cpTPH+KE3w4hyw/ufe5/8BTgLrzg==
      "], CompressedData["
1:eJx11PtXzGkcB/Co6LLYYk+JLpPVZbOFbWp2VzzE5KDEjEoNa4qki6KILpN2
Y42nNDXf0pjt5jI5dhTJblflHqbQLioVRRGTbJZjaLLv+QP2h+8vr/Ocmc/z
+XzeDys0du2W8Xp6el749PFlxN3Y1fNNdZPc4x5n0JghYyVFS6OCKJm/O8Tv
LrzTmn3H34Qh8Q88WIJgSthB+b2X4H1/X3OphqdKvFUCASUffDnfn4evu50X
YGvKkABHqh+1kZKTYQ5fn4ZzvTitFvBpjhVLft5ESU9LCVsOj/kq8KIIPpQs
8ikJpYT1dtQpF75hvmpBIjwl4/mSC5spaR18YCaGH30troyFl04rT1OFUzIr
ar/FAbjQ3O2FBvVw2cShJYIS+e+TFKnwktr05yPwn57aaRsiKRl0jVmTDJ+h
NSlLhVeu/uBVEE2Jz9Ci43vhzV3P3xjBO0Svp6/aTknL6OqEPbp6tPQEg/4E
H1TK+mIpaco2VSfqfv+H0j47+B235qq1OyiJKZAu052XGczqOmPEkHfswp7D
OympjxG/F8GvBDveUE1kyKGGbwcU8ZSsuHy/fR88VKmcEAJPaM+tPZhAybDl
VKHuvhuntu5JnsCQh8KBEPddlMzL876k68+RErOi3wwZssw1ya8cfjUv3b0Q
PkfML7towBCxc6nhINwn50baKbj2y/ywXn2GGEy3njwCt/Cb3VMFN/RS6BvC
Y4vdH9XBzS3E0y7DfzwuCnAez5DyIbmGBz9nKfPQ7UPb8Ipov3EMGe9k3H8B
dZpX3hY+hrdP4m1I0GOIW73Jw07cy4JhyQZ1dU5c+YsxfCI3vKgNfahZtT7y
Izy6bOzu3M9ScppfkpCPvqnusttNXaqb6it7i4PGpMQ7Lc3JI46S5fJggSU8
p2ueclgrJes5i2IVmEv2n9YWLnCXwIIsS/i146r1GswxMu19Eweef8oh8+yo
lNQ1pD2eHUXJyNmBQS58c+u+t8vh0pArRuxtlMQqzHzXwcWXKg71fpKScyeS
T3hspaQ5cPJYGLwmsk+YBL8d7+fqsIWSUy0OpjHwKXaS7nQ4r2rKd2ZhlGzL
lrxLhqtn1D21gjcJolzfYs9z9luY7odrjT3C7eE7W3YMP0IuorrVtofh4fY2
L5zhjVYVFVeRo7njYs1y4JoOZ0Wx7vz1f7jXkbv4LG+RDP5CGFiogDdLRJH3
kdOXNXn35PBuiXnHDNyr/Z3ZrE8BlGTuTvy1EO6b/TJXAj9gq9IuXYf9bCg8
XwTPPmn/yAB9azuasLuchz6oGjnF8NI3SbI9cNarct7CtZTMrC3Q6M5Pzqq6
8Aq+6pa94JM/JZ9vebAK4DdPS86oMK/6OUkB6tWUpORvypDAbXs89c9gvs+C
uSkmcFPRaEMWfEtKqZEN9sF7xJkR+FHS2e/Oz4BnrljSOwofm7P1yTNfSr5Q
xr/aC39SoxnoxL6pQyL7y+CJEbwj2+Ec3oBzNfYzjt9lUwov2uu9NBTO9a+V
MdhndvhMs/vwBelthrr5VjVTbhxyYf3hmuci/O9fjU39un1oX8jnrESORFZF
/t1w/4hj1z3hWYKjAbORu2NTbe3/QP12Sn6HI5xubbw5Bjdp8BFeRB/s2uKu
zoTv+1c5FIH8GqiDcj+uoeRYp7+fbp9fu8mtFuMdGPrMctqAPtepVlprdPlK
dVBa4t3IZD1WvORTEq2ysVbDBYvVnunwZ705V45gjv/3Pv8HqATHnA==
      "], CompressedData["
1:eJx11Gs4lGkYB/BB5yuXsdUm6WqHLYkRFjtt0hPt2A7W1hBiMCOThFQqp12P
sqi02+5bwqxQJjYpm0N2c1yyIcmpHEJWpRybcspm7N+nPvXh+fK73ut97/e+
//fDER/c5aXMYrE24qjg2MhMrpsLC0o9OCJ/sRlDzvIu3OY5U3Ldd7l4DTwp
QXzX0pwhPoa6w+F7KJntMGdkGXxj4N/Tml8yxCO/7V6TCyWJIdLbbLiJXUVk
NLyVm2VgI6RExebizrnw/sUNL9/CV7nbBFW5UWJ/dKhsyrWgdO8vT/U9eAw5
kZya6eJBSVIVN/kd/IhSyktXeCS//eaIiJKEx7PDRuH8CVWZMzw8tC7nrCcl
3aP3+XL4I7sNgt1wd+9LMVwvShqKm7QH4UUTs9QE8P1hc2IrJZTIq3J0X8GT
Hszpt4NnXeGl23lTot9dW9gHr/ReKexAnabFqvL6/ZT801AdOABfHNbd5wNf
8J1fPP8AJV+VGfsNwS+0n8+cQH/yitvJZV9Kos9tNRqD3wqqMVoKZ++Ycnvq
R8mNec6bFPD7jTpvzNDnzuyON6yDlLiI/bWV0B9LtT2+FaYM6ahuylcKoOS/
9RV18+GnXR5e6/mCIWquya+fwccyv3bTgE8uuVHCmDBEwA87lXqIkuOkpJMD
txhpTcsxZoizHnvM5DAla/qm7unD9TIOdTQaMeSN8pIwKXyFeWjyBnhK1aLR
yHX47pLO5ha4c5KmkA+3G93UJDNkiFdjw0/D8Nubty50hI97nolz5H54fv/u
t7liuKA/xzDYgCF//qq4dh6usqXOInCmnr88pfr6DLHi2nTpwQOs6gsoPO1d
eqvtWuRn1LaXQf3HDCyWMfBrg56vWXoM2ScpmteM/42K/801BS7fp12js4Yh
1QL5Azn6NqalOZAFjyk91c3XxdyjJhbI/Sk5sn5z152Z9/gu4vusZojBjvey
KvT/enpeWgU8pDtDNLWKIZJnhfmRmNe0+uWFjfDkiDMcDtxzlqRrHeYbksuy
fQJXHxMn5X7OkJYBZd4d5GHfQ372c/ingxGm38BH9NcO8ZCfnjs+5sPwvied
L9p1GHLBrFnpd+RNI0HaOAGvmmuUeRAu1PJWqCGf5rU7x9/DW6cj+g/DR0/m
TR5Dnlsb9UyU3ApKVTWmNx6D12U862pC/pfqqjnMhrsrZX0fDN8jlbVvwL6E
W9g2q8CDeiXLBuCy55ySW9iv8iBva2W4w7dz091Qpz878ogR9rEl+nk0C24V
rLm4Hl64/IZaMfbXcjv7ogL1BCj7OlqjD5WipSs9sO/GcYmCmTpjn8id8+AX
BWIFB/eDoVmh9chMPsUr4+PQ5/AYaZaKEyU624sG+uHR8YmXjmMu5bmuqSxH
SmTbhq164Nkig71OmKPo5mdeK3ZTkv+40rQNPpA/3cbD3B0WaVV7OFDSG1pk
VgM/8agvIAY50Ro5mVpvjzrXqwrKZnLLfUy8kasVGSFNEvi5f2Mn8+ELX+jk
2iCHfcr9wXpwrkRRdhXOFqUL4pDbyqHM0k/gUbHWP0vhm15ZLj9q+OH5kpQt
PqfhHWXxrHLshdH4VZkffHXtNuEP8LYDllqp2CP1sZuHW+B8L5HQF27cPm4/
hL1zCDS4FYD6/zjXG+oOrzWZvFeDPT3RsM6Wi//l1WqY74LPV9wdzMBeP3Ly
OcVGfyIy3K9sgWvb3m3+EfdAzKshO3X082P38//2Zs0E
      "], CompressedData["
1:eJx11GswnFcYB3CRpqTuHSqqKiEiMtK4tpl1yaG2NqqLXfdLLYMgRZM0MkZc
IuKSIbwGb7dWXbuylUQ6EWtkwyKJsBeX7IpbsgQ1iiiJblVDH5/Op354v/zm
zHmf85z/eQ5FpzBiVVVUVFzg2wtfZaePx7CELzza10enjJDox8VKKo1BIFf1
QFshuHFcT1zuMxK94iS3bPoRiKbTerkVPG7fycMsGYki2Yw+Mfi5Hll6Hbha
FpV+U04irqOTUTO43Xf/+JeDu4V6snNGSdSxYaeoATc8zojLAk+8uyF7+hyv
L2WwsxPAzao1jbTH8f4rnVN/hoCzFpY258dxPUMipok7eKh/J91qEtcfW8+s
tQHPMHuMdsCdIvkG9eDd/r1Nn4LbZNS0yaZIZFA8YKrGJFCEUXm7BriCO+HA
e0HCOfurc8C1NO6NKMV8YRfb88HllyRyWG0o0fcn0NlNy+VZcNucUuSrINGt
zOArXHCT2wWv5ODbopTF++C+8uHf6QEEmvctGOsD969oIoynSSSwOOCtBJe5
rTxvB9+qsHXPBp9WVbdoCyRQVu+LUR74KXfl2hy4ml/bux+CoJ7FBF4d+PmJ
kodT4JFa5qP2wQTS+ZCTzgbnWEkLZOAGjvQ76+BBy24jpeAOhw3HROAxXh8J
BSEEKhruaCgANyirqu8Fj4/pINNDCeRh3k3LBy9rlhfSwa1nPkmhhBFox6Qo
Pm93/7b85XE4F02SqPka/IZsNv8auIWWnk8sOLMik1cXTiDpHX2zXacWmxxf
g76p+hVpekUQiHWo+1oReFt4qKkeeD5VvjUJbvz3/phy8IerqsW20P8THNvf
or8lUJnzloADTvE3SGXAfRnl8KxfglP215b+Av4z98bEBbhfr1SPZHokgYLN
o0JbwD0Eep9XTpDoCO+0803w9yscl4XgAWFVjd6Qn6C9jyqXwNkHBUfE4Bfy
vjj//RiJ7LTT+OYsAg2Y2TYqdv/b3C85Cvl8ov/mGAVcmrZa9Qa80GC1YQZy
/kjYEkMFL3krbd8D+TnhyApQl+P1n1ktHDQEz/VicHqe4f1P5ZocswSvvXeJ
sjSC6/FN+CPDFfxLkU1S0zCuPzznndIH/LHuVvrwED7vbIdLfDQ4werJ/ncQ
9+cbbTfmRfDpEeWO5SDuZ436dnsBeN/K1U2mFPf/6leJbM5uPafpSdkSfF9l
RRl1v4Lfdxjo/FiC75emw13vAtetCauniXEePEfVm/rAzyx5BPqKcX7OXUpq
HAUvVLXkLIlw3ni1EXXz4A+MqDN5IpzPlEH7r1+DD1x0vV4iwnl2eZKa9hd4
68JdPinC+bcOiR5Xggvmk6WaYvxefjowdHJzt/7q7bkrYvy+SiNU4vZI+UKG
8rbedTF+j09hOG7BetPJOOOzEvx+LXmjZ3Z9LdXLSSHB7339vagP3oJzvNPp
XVI8H3T32d9aBo+mJkXVDOJ5wlc4B82By7gbGZlDeP6k9ZdrTIK3lWs3hg/j
efV/8/k/6zb9uA==
      "], CompressedData["
1:eJx11Ps/1XccB3A6W9mEh1yWy4MiKac6NauRls+m6zA6LmWOKB0TlUuZw+Hk
dHzinHL5HoZoYjbVNKVHbjEdLYrEPJZIJSmXg0RzGeuy1/kD9sP3l+cP38fn
83q/3p+l+8O4/Hlqampf4GPhG2vWSXHlVSnOb5C2p04zpHSf8UxZEiW+bqwu
F/h8zhUtEzgjdg/7TkrJaER67za4y3TseUO4STDnH6tTlNR48NoJfN5kPWcR
/POTh1xHTlNyTUthrHIT3qeh+VMMeVO271ZBKiXMB/djHeFp9RmVNvB3Aqsj
bumUuLcMaajczkV/eeUkQ4J7puLHGUrWbF2qvQV+fVyx9/7fDFnqm5EvyKDk
1lfLNqs887zh1n1w/91u+sOZlAhOiDt2wtlio2HRa4awRQLL7VmUpHBnZtzg
4TXKqbMTDOlb/YAvzKYkOk03yxtufcDQr2acISuOvQ08lUOJblNajh/88Xik
dvcrhrBW1g3Hn6Gks63UOwRun9lcKx1jiPbzIFOnXEpmy5KyIuFeCfZ7Lr5k
iMV047UuuM+i3INCeKONRUDTKEPiq4I3OeVRYhRd0ZkIvzVk260cYYjDSf2w
BLhL/06NTPhuBwPvvGGGFNWnaaTCi3J2XMqHN1WbXatVMsRjyocJhxuU7R65
CHeNME/sGWLI9fJ1DVbwB8V298rhLcLqu2rwglft/iU4T73dBZ+bKveUpVsO
MsTUc8L0Pe6lvz2Q2wYv9FL8vG2AIXEHxGa2cBM+a+AxvDpdZhPSz5AjE1PN
9sjn44S06X5431HembcvGCL33mShhzx1R+M8X8HXxz4hDPzY8J3Wjh8oueN8
2+OdKocLrPby5wxZUtTwOgLzapLN47P8qhQ9nNPSanijVBk6I6ckZtrtoRZ8
idrEG1d45liwKBh9+IxvQPXgvONdy7nwScmVQkUaJf6RQXFmcKfLZnMDfQyR
CdTYmuhbnXmO3zK4NNfSeQTOeBaFeaCfjjZBa63gJnZD5v74z9NnXHa2jBJR
+EpHNnz9eLw8EH5ZIu5oT6Yky5+nvgpe+ITv2wm/2vxssTH25Y73TJHKfV7u
0nLGfWcNJw6FUkoybRt0VsP5Wuuz6+Br2bL8FgklekMSnRXwXz1uBxQiT4Nf
UmK3nED+OzQGreGajxxE+si/Ouxm7Z8JlBy1FX1tAfdS/tQxDZ90b3gZdZyS
JUqfdabww5LW9i7MMYijE+gooiQwezLDAF6hrnv4OuauCPk3gBNPyTFZllQb
/rA/pioP/Zmpu2HsGEdJlGJhizp8tqxEZxR929XY2R8jpKRNL1c+i3lxe4vj
WtDPLzXz5vfFUmK+YGPFOFxQo4y4hD73Ze6tj4G3sg9vG1TNPat57hT6XzCW
lbIJnrL4tnGXau6lvcnd2JfEzfLLq+Cu52oP3YMXb5BbV2O/RMK8UBd4wUFn
tqqfIY+u+GRjHw3umtpnw29evfpNBXxB/PdGUdjfj8K+5engnJa2c5Wq/vdH
q7tzse/9TeEDv8MfhVvVnoU71/Q6cPA+zBWvGpLivpEHWFZp8DV7jMo18c4s
Thc7SpBPQO6E/km4+92GrQJ48hRv9CLyTJqyroyDmypZP76Az1d/6DSI/CXJ
BztU+/6++y+5O96x0qc9ls5iSiaVqb8Fw9tuNHrWwkt8zUP+wHyfRi1cuxfe
Gr2/0xrvZNuHJuc8EykJXilM2gP/ZKN3iSV8p5dQ9Ab9+b/3+T+PY/C8
      "], CompressedData["
1:eJx11Hs01GkYB3AV2rKuSSpN2UXIFkOYJG+5xqS2McbkkhmjsGdDyK0Us6fo
NE6//ZVbtpEohai0ReW2IaKtc8ZKSartJky5xGqwX//tP/vH75/P+Z33vO/z
fJ/HUBi5M3SukpKSI755+Gw36PDYmrfqp+5O8bw2UCQrLP+ZQQBNvuTcdXOF
l4h7ymUsioT1fX6pE0iTicJD3znDiy4HyuX2FLlU0/7umyCaaOdItDbCKxdY
uSbCXT5bDE7Dq/u8+1lwPfXHR0/ZUWRVis7w6G6aWI7bP7WH28l8T1baUkSH
r5wrC6aJuTej1Q6+Sv9Tbvt6ilBtJ52qBDRJUnNYO3v+KDdgJMmGIiwtlZvp
Qpp8kHgyZ32N04d72da4f17kzI8hNJmveMt2hH90ZAfdYFLk9KR1v7KIJmxu
d5gLXHPfutJQK4oYpUfEn4OrMN8YucNF2vZRv1hSJFuvvZMZSpPuy/k/74Cf
v+fJtl1HkVquYUkhPC7HhseFZ0edLvBdS5HAxlqv+XtoUi5S7QiGu7dFqi38
gSK60ugQT3hb2eaoMHjW9+E7LSwoImp9vSIeHifINY+GHxPeDN6+hiJX/ZYn
S+A+o6+OJcPbDGZyY8wpkn71MDsN7uXmrTgKzxDw8nLMKGJpW1ofDD+78KI0
E764lRtvAV/Kze0yhVPO4i358OKHRSocU4oYRG+hnuL+i44EFVyAiyPDZe9X
U8T21T+1EfCOlsaMq7P1T4lTPQwfNh25K0d93DlmerVwlbyvQj24NKHTkgOP
buS7NcN7/TxZBvDK992xpah/IHtbnAzeo1tdV2VCEcYZF/9B9Cv8wbR7L/yI
gWX9bThvslLHCN7syKt9Dc80Nnewwjm3eNpntqPvNgeMreWz/5f1hrLgKfo5
TSLk5Hlzcd3YbN9L6i5WwD3VK+SJyJXjYY0uBdz44ZJtJnhvh/mJfjFy6JNY
MzhP61Z9c+Gu4Xz4b+eG/pYgz0+CBMtV4XKhsCIVdXsc/1dvNvJfOxRtqQ5f
+zLPZwKect35yTl/mjhrSh9pwN0yna70oC/TN/SUbu6iSYxeTaQWvPjOxPwG
9NFDPWl1O58mTXxumjacmbNlugh9Zyb7HRrwo8mz9jHG7PnVY4svfUFOymwa
5ujDq8Smkm/hvUUjsd3I1aD499d8Hk0WbGJEqsH/4Pg+v40cbszeNXrBlyaj
+w88VYabBWhMqiHPZlnV1jNcmsTGsyRz4P36LrIB+P3cnpkEuHHnx/sTqI/u
p1ShA+bl7bWeySkfmiRIXrBH4D8Vb76yDPNlf6I1rQiu2ebMfw9X95CW78U8
eolLWb7w03vLr/TBjUKOR7hifjmtpw6ZwVNX1gi64LcVXgpDzPtUQqY3A143
TymjAx6QoG0+/R9/4DOk3ghP3hpu0Y290Xqx4JkTPE5+LawGzpTt2+GK/TPz
tro5Hn62IMSmDM5Q5uxmYF8dM+7c9Bi+YfSgoBDecr5+JB3esL/vzXq8N9Fu
70AWXNqYHjAMlxb/ufQOfIZf+fE4fIfFIoG/A0VafE8d3IZ6bq2IbkiDx3xd
lM6FM1uWOY7BH4WW/hoLD45KUdyB0+vY/tfRlyUhRU0RcC5LNH4D3lVtWHIQ
ffyQ9+ShCG46LlpM4BkDcTsDkIdln7X7+XAHk5Ue9vAXPSKFA/LzbsXlPRy4
mtqCuY24pwe/XWaCvP3ffv4XMgqt/A==
      "], CompressedData["
1:eJx11Hs0lGkcB/AptuZUW+pgERN2LFsUpiiXPEi345TbkobCMut+GzVqx8kW
O2wjU68XOci91B5sCZtIwiph1rpttesS1qqkqCRlv/56/9o/3n8+5z3P83t+
v+/z6PhFuAQsZbFY1vgU8D1qy+IndFc3hIqPjzfY0OTYAOtD6BhF9CKVXE/B
Z4PWv98Hd3nH8Xf8hyLrrN2vCOG6PuoXbBb9Scys2ThFSgKLcsPhohzPgtad
NPnluzof1X8p8txs+bIgOFs/m+cMv8HmzbAmKPL18T3ZwfAmqW9tmDVNHIdV
jHvgll6jHAE8u4m7/50VTcZKs0JKn1FkcqTLNhA+fvWKkzJcb9Li++jnFAl/
lREfAi+4LncztaTJxrjTPL0X+F8gqV+sZ9x79JCTBdY5IHnWB6eksWqL9a/S
ung4YgdNDja7OcZMUmTcQXIuFt7YNOKZup0mH5tZ7nPwypo27cU+PJbXuVeY
0yRXNZcd/JIiq/+wG5TCEx5H7gswowntZeV/D75G4ZgmDV8baN8o2UYTEwPP
tLVTFNlyOVHhEty+1aL62laaDNzSsNsJn1FnL78Kbz4bUSrn0SR5+n2mG1w0
2Gl2G76fv1eWbMq4MEpReB8+4isQV5gw6whNKtr64H+r+Qb0GzP7dhfUmAzB
V3q7u4XD/5wtS+1AncEnp5xfw+dnJTcztzDnusNbwVLswfoex9LubWb6oBX8
slYJ7vyw+oTXZqZvuTp+aRx4clUW/60R02fhxPCcITxdJbxWBu+vVzxqDC83
8H9pCe/p0rq4CU7z75bEYY4JAh7LER5ClIS/GdIkoFUvoxJzX9/ULnOHh2ss
6ZXDfWaK+YPIyaG4PeW+cGufT1WPDJlcOYp3tYfCDXqkGSPw66MeUh3k8MuV
559EwZOCEs/wse9UqjLXDrmdTeiQxMGb9duGAuALuk2pHsh5UPo0fRr+xcTR
9ifwveTaV7G4FzfURIUS+PyDzhpX9KFu+8a6zFGKmBjm5P8IL6x1qTBHP18r
2br+OkKR29GZG87CZ3aaa5XB7ySXKo8+pQhHpTgqCZ5tJLsow1y+KTY20ocL
HH5vTIR7yHtVhJivYmqoQ9AwRab5797+AM/rd6fdkIeZMxOVd4coktNgn71Y
/8FoQfo25Ed33exBPfhulVvcWHhtS1GWCvLWIQizuzRIkQOBJzjR8HK5QPsN
/MW3KT5c+M37LV7+8PiGODOCPIfZfra0bQD5j/1wjg9nT/faayH/XZbKHyTw
Eu1iUxd4pFNS13v4irHJDU5wD/bDTXvgmtKS0W7co5Czu1p3wEVTOW+3wnkj
ZJmmBeOSp7IqfbixTMR5Y8GsEzm//LA6XKdId0c77nW6mHSK4aUe5wLWwN9k
ZrSbWTF1hlzmyT8hz70HPj5fZc2ca2ueDecF/EKC3+pheIt+nmoavMq7zW8Q
niOKMdiHd2nVayOuBvpmlf7qpy64UU39rnL4OjY3uQp+6sjP3CZ48d7mIWW8
e5/PlaofwVw29PXkV8IrwhrjT8DzxdFLNDHHQvUwWRE8ZcY+7y8bZu4dYu0L
WXClhTXV3fAzu6X6ZciJy8J6USq8rm+u8wGcVXjyfApy9X/v839vJ/1c
      "], CompressedData["
1:eJx11G041WccB/DjWQqnsNSRqLRJ0nNK7K5ZsYip0NMUayuGQxOrpqeRp9y0
/n/OqDhknDmkXA5J55BdHq4Lw4mdUiOpyBUhw8ns65VXvbjffF7c93X/ft/f
z9w32OOYKofDscdRw7EM1AkoHpfI5s1aHSDgsmRgpHlzQColC6/n+ojhItml
gatwG3mDf6uAkluiOUP58ORRsc5iuMJou8IqjRKpW6NnDnz57n2jxnDJ6uDo
n9Mp8Y01PjPtuQlmqbf0WWK478vbpdfh1YXPp31HmEHEWrhzjLr3yxuU2By8
uLEA3krubPLQY4not8NnRm5S4ufJzS+Ey12juJ26LFnzeRKvP4MS7i1GtxRu
5vGslgOven1+U3UmJT/yhvg18PRcjbDi2SxJ3ZO/O0JISdZ5J+NmuMtTwf42
HZbUXJgy18+ixCDpbeQT+NaWbW/HZrHk2B6f2+fgg5tbqvvh7w82eBZos6T8
nnKiBf7A9l3eGHxvqf2uFi2WZAuTpkbhKgkpdPaERNb9xdSaWM0Z3x50/7PF
8KBKQQXRmLmnY85Q/Dr4MsNc1XE1lljX2YVEw8cVmUInuI90V3+RKksush3P
jeCsYZO5N9wpW9dGocKSinVytwv41/GYFbFB8A7nxa6/clgyxBPZNqAO0mb5
t+fgMp5RU/kUQ3Z48zuGULfW+48nWHhw7aRK+H8MOR5pNqkC5/sl/Z0Lp788
Cc6YZEjGd/WKYfSliuvoVA6fkK72q//AkEUlB0Jr0UcrNzvNRrjVgUfXR5QM
CbgZ43QFfRdUzn/zGN7UVfE+Af5ph0+aHXJSVMXf8hbeX9pYLJlgyHLtwZXd
yNWsVUupEm7JsKHu8A31Y8rTyOHvghPd2kqJzCRdYtM7zpBzZSn181IoeXE2
i8eFt6041TAAl7YmpGQylKSVOTrNhwvWXzsyCk8+7Ze99holjh41HqbwQ+JA
+Qe4a6AspP4qJQ/FS/Qs4DlsxZQq3l0geMcJTaYkXZH8zbRbH19ZfQqe9LrM
3SyJkqgtHIdpHy7Zad4HF7oeW9qTSMmh2ZcUy+Dq2w6Rw/hvmX1JuOQK8txn
1L4UXmTd1dgE7832ds9LoET0zH2hCdyKdSsoRD0LM/Xyi+IpMWn3khrAe6K0
JxNRf+HKLQmKOEp6v5YU6sAr23xzAtEvYQUbYgF/+JO9rgp8PNOlbhf6q2ax
Li0xlpJ/DbyPDKKeW+MVwyzy8M/DEB1TuGORZkAP/K7bV50nkZ8WpfqD+hhK
jtSU27fCD58NElUgb91VWaU5cOdOz4My+KuGo42a6jO+t79B/y7cn7QzXeoz
9yj9jcQ34PqLqrwtkX8tA77zfLz7vRZhYuGCYtMwPublxry1jy/D+17srD0J
1zOu6yvBfJ0w2MY3wb8uhwzm+cE3xkQtoJjHMHlP3F9w25cbIl3g5+fyXOSY
3xPJ3Dci1M2hPXuVHXz/MHP0D8z75vU+jWmoc1zXMhdL+PKTqQMX5rDk1Z/S
93noS7hDbtsn8L1eGqNe2BvtQ6ZHH6GPeabiuerw8cquO9bYPwWC8EQu+i65
4/tkEPP+1EtbSw37anfGXQdf5KRkTWxzN/wFLzjGH67sjC68h1yZv+FHtMFH
nubJmuHSxFXPlyCHRNxvWgcXhtqP2WJP7oj4wTAeuXV52ZtSDvesLNDIgOtb
RO78wFLysf38P0dbDiA=
      "], CompressedData["
1:eJx11Hs01GkYB3BGx65K2oNVbCpjRURaTDfHG2rlSIxb4xbWZUguKaqJVUMq
2noRJ+QS41ppK3aKUNGwHGHSaVxSUTa6qLQtdpr9zl/9tX/8/vmc33l/7+95
vs+zMiiaHcJQUFCwxqOE594bfY9UPWFLCjftn7YgSsbFJ+3qBZTkNTYEJcJ1
ls2zaYHPrzzeHl5OiWOoeGo/fO+DSqNbcD4/M9WwgpLJ7jNJkXC3tgUaN+Di
Ih/bcTiPsTo9GL7gWB/nCnygtuh+XiUlevvVzwbCb39b+NNGuGJ+paNzFSUd
nPkFHHjmdY1DFvC3/vHditWU+PDEWr7wqy/abNmBlASn9y6thl+6r6DtDd+o
9kD8JICSDymtJj/XUDL6WjvLDz4g6XGX7qbEV7brGwn87gbNLPk5Y4FmF07B
k6S5VbsvUXLn2nUV+X1WSBL6q/wpMek9qT4IP+FYcGAPXM9UhePjR4mX60JV
cpkSJfMubix8dPF0Kc+Xkq4f3bvPw+3aVnkfhG8L+Swo8MG9ncnyYXiwwxIi
r3NPa/6Hnd6oi65EpnIF9Ux2KMuEO+mx8pQ4lPQ5ba80hkdGHF1cAPdaox9i
sIsSo01RT63gNWqyqWr5/ZWtdAc9v/q0QjqnHt743sVT5kHJ4aYvLE24+USD
pQjubzjEq3OnxHIw1Osv3Cdz/Vk1CZxLtnhFulEyoXlEWQBf43U+7iX8zyPL
5yibkmtVuxM94ToxPf3T8NpZXQ+hK95ze8Z/j/qUV3bVKzGFLZ1zDSojLpRY
ndRjH4XLxgbOqcNjCGu/Mvzj8GTtLOofopX9kQn/0qS1wXQnzg8L6OfCb48X
uljAN6spSj2cKXkxcnn2EforMB0SEvh3xfYvX++gJJkG1JrCm2xmi3bA2+cs
VqfA7+aWGPOQnzsGNce94Ymf0/fqwM3jKypEyNumgfjDQXB6U7hiJdzIbB9T
A96VUDwSLT9f0a5J6ERJm0gY6oHcxqmK+hLgGSPimCb4s1Mqz0uQf7anI5sH
v6lt9m4dzsnK6tYfxbz8mt5uz4czHkaWrIeLIwRFZvC4KQ/74/CUw/zWS3CB
4RbtI2WUvIp2bz0Bl4j2ZazA/y5NlnaJS5H3sZ7eVHjfpHMEA/WR9o/9YQV3
cWpgp8GbO2W8A3DjYYXBsovoc/VQgPy7nc3ZpZ6oc1GOt5UuXHKBFZMENzAV
a7DQrwsugYwrJahrzlTSIXiswVyqFvprO+PSbQ3vVK7XiYL/vc7SMwR5aGTs
sXtXTInwxpuaMLizdVqdPfKj3nxIWAwXaF/tkNd5WwU3jiJvQZvvP4qDR9ed
8WXD09zDQqKRz4OmiWvYcIlR1JwdXNzy2+IGL0q0VNQ+bYP7S9fmy/s+Fbpn
njLn6/uLYjV/N5Lf522nyXP4svZdhVz443Jp7xJ41cQmVRPM1ycr1eRz8MB7
BRML4NnbmSJlzOPCD9l9j+HvaG7GDHJbGl6StRPzu46Z5LcK/xv1VGwwCbcJ
e7LBCPO+76IxJwMeL4vdMgRnDr/hM7AfdB8an/4Xbp/q49cFjzu99ftB+Ch3
wJyPOufUZcw0wMNdWU7XsX9CA0WLNNGvZmeHRVfhHSkpwh+wr241vmJeg8+G
6dUUwo/V11lOw5e2s2K9kYexX8Zt5HtgY+ParX7Ye1ljrk0M5Of/9vN/Uo3N
yg==
      "], CompressedData["
1:eJx11PtPU2cYB3A2QIEyaZFyCxQYIoogjZMBcnuVQQSn2GKcggoykClabxUh
ENysXHT4FvGIUXHl4pwGWA0CdgoKRkXY0BovECap00JbaVEsoAiTffsH7Ifz
yycn57zv83yfxytttzDjczMzswg85nh+Wx6XMspWtO+S9k77TjIkjUTkBfAo
KfCZvj8Cn/y2K6PxA0OC9xiiJz0oeWx8EK6HV95S9YbDb0186r7tRYlf9LRc
B+8YvBbX+Z4hj6xEglJvSizmlEo08Iee/XZCeB43ly73ocSxcMWPw/Cwv/JV
ogmGnMya02mYT8lP2QOaN/BmidK2dJwhrqnbD0gWUPJEIro+Bu851Xf68hhD
Vi6Tnfncj5KMnYHeH+Hqfr+LnUaGLOx+HZi2iBL7iCpbM46i3VE8M2/oHUP2
7hZcv+hPyYdeZZMt/KvHhH96lCFrXRZcvR1Aib+xhOUAb9+lMbS8ZUj86v2V
1xdTckJeGOoGHz7in9n7hiHmltn7CgMp0euCwhbBpXHWh46NoA7lMUHefEo4
6wv0S+Gt2sa4egNDfApvayvhhsuZVVHwLcfSxx/oGVIfv6HwBZy/P1cogDdJ
k9OPDDNkS5YdawReFpL3dLPp/Y1Xai69ZkiAV8NNJbz+wERgFtxVEVTZo2OI
Q+6ylCI4e2vjzly4nuEnGbUMaUi/UWcL1+eV5B2GJ1Rt0xbDOySs+T+Yzl8r
Ly6H55pJQ+QahrzUCo4ex30vbN3hcx7eFTToGQPnWn6SnUJ96vLbzl6CO8tX
3vl7iCHu5rG1e1DPeTaZrU3we9vc/90HvzWgrvBF/R2z28Qd8M5fjyls4HpO
w/TNhejjtuZf7sOFBt1bNlzl9H1fFPr7u29CihK+yW1uvRN8oyVnfjXysNqD
n9oHT/2TJePBI5emGcbmUXKwZHX1C3hSVXeXD9w/MG5xGPJmbeDOfQXff2c0
9By8qKZhRw7yGRK1Xj4I36yLf8jGvaK8imrqkefUWX5EC7fPskkugqunW1xf
uVOiWbVB+RrOsy4dmoK/z4kpd3FDfdRJFhpTfpqcIlWoZ0WEhzDdlZKSOv5s
HZz0nk5Yh75cKUvhKpwpiU1U1qjhlZxu72D0MUfLSvvSiRJBpe6RypSfo+Fn
XdB3MQ0NqeXiv3lHtvfD+fKjx6fgvE0z56IdKElQhx56Ao9dkskeQH7GasSs
GXtKhrh7n3XCI1pkNv7IoU2iwvYVh5Jo6b38dnijamKNLXLLe/7CfZhNSXdT
5ioFfPwf875h+JTfF9Ge8J5dGTGX4bpJ57shmAvHAmH+YTtKiFe5VGbqe3HT
tCPm6LCg+a4VvPTj0l4GPjtRnDgGF7A2+t6dQ4mX9CBPAs+uGNFEYk4X+sVW
X4PPlJ2ryIZrIr6pc8VcB6lFI0/g7zkkeDvcj/+ycRweHP84wQPf/+OpbDwJ
zqpYM/4Q+8HzO9GsMyYPYw2Z5iUkWZwZhn3yvCWgLALn5xkjny6HF574zNEB
e2nTSdYSZ9RBmr4iOBguuHHVvwDektbvyUbdhOGDHab5bbX+uVgDT995wjto
LiVtyU5VHnBR9dCqtdiHqma7rwtQ/zruSo09XLk+q1QBz7gT+mwI/Vr3aKqD
BfcQcYWN8FGbDzMZ6K/MpdbSAm7V48LUwcfk/GsWLpS8G461msa+upgtPn8B
HqB/ZryC/LRJnncZ4TlWbqLz8NaogeYdyNv/7ef/AKXwyTU=
      "], CompressedData["
1:eJx11Hs01GkYB/ApdhnGZDAT1rRmpbRTu6GoTbyrGQrdRjuYdGErRXvUIF3U
qV21hbD7y28kl0nkOpM5bETKllRq66zYtkYlE6mYdNlmhbP7nb/6qz/efz7n
Pe95zvN8n1cQHS/ZOJHBYCzEMcGpYB97m3KjvuX+T20vxE40Sfk2xuV4MUWm
6g5o4uA51dlF5XAT3mDVzlMU2dRz7wcZ/JLtJoklnybSqRZzwksoUs6OGl4G
D5c0lDDg34j1A/6lFOGxFvX7w6f1rEr8B+9s8T/cPu80RWLjH9i5w5tX/Voc
hvue4tR9LmUU0RZujRTCTZpnHnuE+0xL4TqHcoqkK37v/hwesjfxSSc87Z6v
jWkFRQZKLw/YwR1k6Wfb4fJHtxU6uLDyVYA1XGX1qioUnjU+4nS5kiJj+0ya
WfDbEzpLH3xGE7cQdWBGFUVkDhkeTLggiy6KgfvOnyT/rpoiBzMTQszgkd5i
5zeONOkIcuaOw1fXvq23hNukNrRy4P3bDPUlKoq8Wv88yeh+mrCbhQ40SbJS
3flSTRFJZgbHBr7UbMLui/Y00QQtvlQKn+qbv9IRXn7ngPrxZJpot72Y8R88
riaJ7wKvb5qy0RTurMrYsPQMRcw3Pz46C84QxRW78Wgyol1hvxs+a34Gb76x
/6GjuqVcmvwR4y7PhrOL9tSL4WbBdTMS7T74jnDNtlC4ld3OhBO2H95Ztjn9
VjQ8pm7NolYbmijG8rz94RX5SlECfO/1EZ6egz68Noy9QZ0HlX0dqfBo/hd6
e/jQUOTmLLhfdPAuBXzHJxXXRNY02ffp+V128GD+9sgSeO5ikcnDSTRh+zDW
ZqJv7yqcPM/Bmd6yJ5bwR5GCpn5jn2+dC7gG9xF3uJaxaeLRbfrCEy6WjLc8
hKeUVaf9ZUWTICbdEos5zqkTDOqNOVFKb8nh27vu5JzE3Ft9Fn7NuFnfwj3E
5lrDkwPzrC8hJ50SwzgHfn5HY7eKRZPlIb1r7iJvldPH9vLhZm7pqQ1woa6h
chD57FInj7vBR5XdwstwKWeKYQJ8JP2UdAFcMPulPArOevyMy0LOtW2Dhwi8
ZOjP93/Du55uihJgLwruXTENgXuripQ6uCUtr/bCHpUUcH8Mg+/01Iv18PfD
LEMw9s7reVruanhb48/XN6D+BQ9rTiRjT4uEcep1xvcjRkO0cPGkf/1OnUSe
z7RURRvr6ei4uQJ9i++P9+1TUqQ4oNhpI9zW0NV+BR4xe2CLBzzwsGjaenhl
prrsNPrPO9j0MqeIIpzrUa+NfnZirLMj5pjirC42hcsUFmEyuNbvWfIIfI98
Ss2RQops5YdPlsC/6jEPuos8ePcniGfDtUkzg5fA89098uqQqz53jnlvAUUi
Atrv+8JPTO/ckI0cOlBqyW/wC8sFEk/4SvMcYRxyu//cqKEQbsHy1kyDzw2Q
6UXcD37MV7jVHl4b7SHkYy8iCsz2t8CbBi7kMeEDqvy+t3Dp3C1zh+DDd11d
R5ATdukVhxvYr0Z+u3wB6nTXeCmewr1ypUuU2FN5apuFEs78vne4Ex6vEyQm
Yq8bM1aMO6IP8fnM1KvwHoW88T78F1FYUDXcVfNgXi38iPOeNSr8D7EXU1oX
o8+hL9dON+a/44LUyQb/jG7hWO4o/PgN26qjcPO0d3W98LO85qutmOPH/uf/
AWYX3Gw=
      "], CompressedData["
1:eJx11Ps31GkcB/AJhZIu1lBmMC4NhhjlUKhPl4Mu55SSpOigKGUpJeVSbVQe
dpWv+2pzak+FKV2oiRBaUqys0qLcaiVF2wWLVvueP2B/eH55nec838/3cxP5
h27YqcTj8ZxxlHHyXaJjihzl988PLz1kIeEo5PjEuJjP6JCpbO5lODeuy+fg
j6/nOs7UZbTutnRvFvyi8qPIcbg89n3y1LmMvPjEOwvnNQgav8CDX8eGzxYw
UrtnonsCLh0rV/ax5OhY2jbPufqMCudV/nQE7hR3brknPPjuYZ65IaOM8Bsq
++GZnZ+PrYNPrH2ZZ2/E6FROtVUo/En9R402vC/0vVLlYMKI3xEiCILHT/mS
3wyvCFupZjGPUfZI1asd8NZ3WUW+cK/FLTdVzBhZlyeYBMATJVn6/RYcfe/A
xirNGfXVzKj3g99YIWo7CJeEVewIlDCy2xykrnDL5EANJbj8g9+vvZaMfiwa
jQ6Et1SPfjEw50jM1AoXzWdUV1k2TRHPY9VM26tmHF3z3ZS8y5qRMNFyOARu
n2azsl7M0WneVhs/G0Y1uaOiMHhYm/yhN1zDuDjHUsooPto5LxIukM0SRc3j
yC9vuK4SPlb+euI4fPWNAkGOKUdH/UOdTW0ZNZ8J3poE5x/W7Csz4eijzp6g
VXD3zoCTGfBlvUmeXcYcOfwue+4ILzqXm3IRXst/F6EMf3ris9o/eN/r2Q7V
AkWcdulm6UYc3WyyH4+Bm2kbvpXDu9NCS0tEHLU1S8zrEL9hku/JWnhjnVZP
pyFHTd9ijN7gfw/aana2KOqetz1nMvxd7Zw3z5Gfs01hGd1wL8GnrmwDjpb9
Pakw1YrRAeOxwE9w7Uz/qkp9jpIdutUlyHOr0YjPBNzq34ZJXvD2Wz+PZFsw
mlRd1T/VSX7f4K74zqCQozWtxSW9qG/M6d6nOvDM7R3F8fAlbn09emJGb0pf
LjOFhwdE6wnhHnMSd0pNGX1oP9BrC9/yg0xWJODI+ELB5kXGjFpqRzwc4Ute
9W4ohed3pUVIRYxEqR2LVsGb9m/mb1Tcd32cam6AebmTXu8Or3k9/OcWeNZS
2WVdIeJpknZ4KLzOKaIRzntx5Ku6HuZo2mcrH/j8LjuVFvigekKX2hz0uVuU
4074xgtDPi/gl9WS03V0GJVOTgwPgq+Od13vjvjdks71OmnDR/4Q74ZXCD+2
/Qb/5VJk+3EtRlTsdCoYXpv93mgx8tYg7RS3z2Kk8m3EZQ/8rbLNwFV4vOuq
bu+ZjLZOt2tWfNfjzokpZ1CXhJQXjaozGEUdvmWv8EZXjwol1LHc95qkfTqj
uL0Ra33h8uqJ66/gb4f643o0GPHKRoXb4BkBZpab0CfjD3K1dOBL1iSd3whf
b6PUtRB9VeM7eP7oNEYBrUJaDfdx2N6jhT6c75+yVx8u9pw8sBQ+/cpuk0/w
RLuhiZGpjNy8vsoXwl0GnhQ1os9zubImVcX9jG1XzeBR0s5sGeYia3TxhTXw
oEuGj/TgBdaRLacxR7VbAsOq4dbC/gAt+O3Q7F1D8LUX37tEIk7JxIKHKvBn
Zypcn2Ae+c1DG7zxv2dTqsOGFHupyi4lH3P98kBI6m5NRl0PRPf64bcHNd21
sQfU1i9vliFvbMbCgVbF/div+wbh4/tcj/KRf7cS/9EGeIlG3wIv7JM4Jb/M
S7MZKZFzaCX8+bXC2Ep4wl8P2r2/Y/R/+/k/Axmuvg==
      "], CompressedData["
1:eJx11Gs01GkcB3BiczlNF203ZNgUW7m0aquRPEXkrEpkKy3JoDQ7Lp0dRamo
SDeHvx6iMYpSbTXENmVGqV1yaQehkUOZcss1olaK/c6+6s2+eN58znOe8zy/
3/f5mfiFuAdMUFNTs8PSwDKSx9UXlUiKTUbtbSbdpqTyXPCNQB5DRmURqYVw
M7ckc3qLEoODK/utfmVIc3Htc9V+M5em7fk3KTmqd8dmBJ5rm57xCO7TOzC5
5ndKbjGBLhI+Q3x6WvpL4JW01rn/BiW2g1zrvcEM+XC5oKICLrh/VHMyPGbn
3c5pIQw5MP3s/Wr4hjPtZRbXKbGWTXDMgk81tPuuGR6SmFfWkkNJENtUtDSU
IbtqBBld8L4GeVXyVUqocrtQDD+yT3vhIPzPLhv23SuUFA3p8dlhDHGsj3BQ
L5UUu1Z+m8fLpmSG/uirA/BAN8FHXXimNGBKQhYlE+KMPkjhy49XrzCEy2QJ
YqfLlFRxY8Wd8Gi1OA1L+Ke6P56NZVKSJEzk6O5jSMN+ryEOfDiIcZr/lb89
YRGyHm70c7Gdq4iSsMoVk8ZxTlEhJ9wbzmm1FGhlUOIqXxzdDH/AWZXNh+t8
k/PFWkhJ2jCLlwbXZAm89sPn5he4PrlIScRQ25at8LflLLN4OP/ouM9gOiX2
T1LzB1GHYzqiiefhY8mvpKfgyXH1b2Lg87iZYdfgmiZRWyVplDhNsZZ8Rp2r
rRdFiOGhpl4TH8CTTMd28OBrei9EF8GzS0xDt8G7m5c9k6OPkVdbzErhG9rm
XfeFl+t7dxvAT3cHO9bD7zgMz/58gZJGBwfNIOThqYbii0J1z/HHwjDs7zgW
2ncD+cm4KJcoVec3KZoi4Sc7FNeVyGH4tqxPrXAvXl3rO/jpjz3+s+G6bX2r
OuGeCTd7AvEuB4mYvWkvQ/zX5h3uhjuv9R1qgmstNG48EcQQlvq19ar9hsZO
JQ9Rt0Ueq5Nle1D/DrXxLrhxu6vLMtQ55ZDw0NhuhuxkLbfrgBfzYuks9KV0
Xb4/gb/g+r17DXfX+d53BJ63zeDg2UCGXHqzIVP1ro0xq0Oc0Xdlu/JhWwBD
9Bs22z2DyytSlphfosTYR8/GBd5Sun/TU1XdWB6h2sgVs0dSJ/FnSJ0iofYB
3Df8TOgW5HCa1vHaVXADzx/X5qr6tW5lkAi5tRw5adXAZcjzW/tEOfBfqvJm
RCHn88sXFB6GD1i94Kf8l5+ymFL8i9wovcu28CO6OnmxcPvmJN6Ua5TMXHfl
ngZ8jXtFVSTcgJ+o3gU3Snfc0evHkNists7dKo8SNVrjn8pamcWv4cu0OS+3
wxPa5+7SwX9PnDoc/w+c4Wu6/AQfXcDjKuEe3Zoz9XG+nulSkS38h8Jy73uY
GzQtZHwz3CT9UctCuNWc6W7nMGeWx1g4ZcL3igtS5sC9qxVFXMylnPR46Qh8
1tmAxSz4XaFlYwX8ufRv2W+oj6CrrlgNzjrF+rBETMmg+fuifvhIdBl5jznw
MrVJJxVulGlnGI36B783v9kKt6wRzPgCn+N5QWGOPvZFaCc3waM9h3YMw6f3
S5Oq4dmJ4QO18HC5h1EvXBRle+gY8tBnb8NRzTGJX3bDG3jaRuFjV+SKPRAZ
/wSeW89zD4bf7mU36iKHPRPDalTzM+i1xaQA+MO/lOw6eMG0lPOqefvCWKre
j3e1uW0OECLP/zef/wXsYQaU
      "], CompressedData["
1:eJx11Gsw1WkcB3B3uYShstpyKbemiGRlLT2KWm12ClHLCUcuh0xkdxZNduWI
dlz+538Wx6XTiLKJY0ScGGKSQ1IrapVLdZJauS07Budgv+eFvNoX/zef+c/z
/J7fzYR5zitUQU5OzhmfIj72PON0x3ZhS9xynhlvPU1yHMtnJC8pkhjzrK0R
fi/mgaq8Fk1uH3f9oeIVRRSHl76qgkcKomZYcK3s5WL/AYp4lmTGlsFtPI5v
OwN38r42qjdIkbrasfpr8DO96v6n4cuSYI0uuM2WgG08+IRlq+AkvL9FoSdp
iCLRkXZ1OXBp4Tw1gnj01aJy9wxThC0665YFHx/IWRmGu9+4KxXDK+cSkzPg
DffagqLg9imvVNJeU6TerGUlG64o/Zt5RZMmtHlWj+Ubitzs37ssOyc+2OfE
JniSnMfbBnh3ZCOfC78hOGprr0GTgcP5ta5vKXKBoSmUxRlXORfuo06TF5OX
v2uCZ/C6EvnwIt2rW39Uo8mlKZ7FDjFFYkZCLcrh7oLmGbN1NDEJ4amlwEcn
tT2q4dzKvVWHVGnyh2b0sSdwVoNdUDNczGzlLijThGHMv6jyjiIm/NqtIvhQ
mRn7S3g4seXshFv6Ljx4Dv++dol+qkgTA0FnuQucPXvP4T2847GjE1thzfOK
33hNwwNPVk+Vya+d8+josxPypsIWL/OsVobc2r0qQd+Y68FduZG7NqxwPscZ
aSTlG8OLlv4tcVzmfH5XkWvBS2u4z1TD7cAlDgkUCDvM4OPxp8UH4FfkdjTp
SzlE2KTcLUTecl/K23nBR2x8uC4Szuc8B8ykOTLgQzVhv/Ytcsh+0UNmHeqy
Pau0IhreNrbLWrrAIaEiNW9T+OZDlvkJcGI79ZCGp347V5aCup8vfX03Fe5Y
4p28A/5bnXXSMPpkdOHn/TTchnUps2WeQ9ofsx12wzsPPurjw5UyzK77wVf7
0O+GXvZNWfw8PpMBjyhOqBChb//x2XlCABftebszBB7XrBuhDSd24WN34e5K
7X9GwFfnIpPd7C+Ep7MWPw3CHcaqB8owR+UOfffvw3n60WfFcMn1Ju485u58
QRvdCtctNdL1RfweyQER3vB95v06zXAreS/1fchP4pbAxjv9FIk/JtogO0fI
Td50G/7kp7xbxnAfl57EBrg4edElA3kOpo378v9Cngfnw2RxXj2puz4addFm
KjgZwi/GTdZWweUPVoZ6oo669fbO1S8o0sHX0i6FG+n1FhSi7pkK/qf84H3j
WtNF8I6NBiOJ6JNKnlhHE26Qe2cwG/40LFatA321wogS9z6niK041jgNXj+z
bHgTffjL9GK3AO6oFXIhXvZ/rL7BDPq2vHiyphDuaeWmyoK/utUS76y05sYT
Hp9kfXJOotBpoLx2jmtV+vuj8COtrdIQlbV7Ez7W6DjJ+qGrIqcSc7caZ1zI
7O9WcNtrJUbpmNPVd+3u6vQ1ghs4GH7BxFyrulgU1cK3Oze5a8Dd2rOUyrEH
Rtxm802Qt8OF3u8kmK/wjR9MUrA3VvPMnU/5IJu79Ma0ryfgrN7L60xRl2PS
Q9Fv4EGn5lIfYv+o8T0318FZXNekPrjEwuqjNfZYu84RkS/q/n/7+T8TFuTx

      "], CompressedData["
1:eJx11Hs4lGkYBvCp4ap2O8ihlEK5ImEUtROiZ0ORRQxFjqmMqUQ6oVpjirbU
m3KYYmOXhJYyLZKcOiir1TZIB2WrbWnTwaGcVuw9/b9/fP/8rm/mep/7vZ9v
TlCY++axHA7HGg8XzyaZ4ZhdgrKatPsGEe/9GM1Zczt5JJfRHuM1aSHwP2et
t7zlz6iVm9AQms9IzrfY4AV/ldfGiQxk5BLLuffpAqP6K3NF38Eb9fxVnTcw
qgnZ5SopYLT3kLhvOfxmp/XROUGMNnjvPW1wkdGLQF45H35Z8Jimw7dYzs5u
usRI8riPawS39xRu7ISf1Q8Ii5cxOiWtuj0X7qqeqvkC/rbleaTZr4zSI9SO
zIQHui5paIG/PiyPbStmpKF/7e9p8MnJq5e/x3mWdb1SPlDKyMfabtoUuDT6
2YFX8AuFKfGqZYyyHCWTJsLn817KhjFXQ2pezMmrjCpFm3ZOgCcuSQ/XgF+Z
Z9Y9WI55TcJ3jYO/TE/q5gUwqs2LeLSigpHw2IEhZTj/ncoMR+TWbdZZGl7J
KHHB4i//s8NVa3ucL6MVxau+jqliFD5qmzcJ/jSy9+KN9YzyJBI1YTWjzqRM
pgqf25bTwPFmdK9ko4pRDaOeoS4/TfiqCdFTaB0jbu73VdXw1KJxHB1467NE
L7Eno8HpjpmG1xm5pUwU68Nfy3NKrwsYZYq0IoLhtfzcCh68LrN/lOvOSM1K
bXA/3DO6lLsUXhIgTFm1BnPWuoX5KXxBlNtKeIj5h/BCZ0Z1W4VPZsCHByY7
usH7beW6vU6MtvG37s7CeT7F8T77wtvv6l6zWM1IUCa+pQSvPmN4QgQ/Pt5U
FOvAqCI4RtkS8y6SJj/bAy+qMBtTv5KR/4TQIQfkszS5/dohOHdA6qRmz6i4
R+m8JfK0ct7ilATvLMrf4WfLSFarsXI88o9yke/OUtx7rGRd3rfo66Iun1Lc
lzDmRkchvLrukrSRGAW3R3fY4X7fOtlUXoU3RWkLPi9n1O5pbFx9hVG8ZkFV
LVylNdtuPjxh2+HQhejPQ49jeo1wpXpLmcCGUazXG+Fx9M3J1Km4TdGHlBuC
GGtGd0za/F9cZuSgpK7dAQ/L5Px2BF5219vdGH1usfmjsBc+JPIVlSxjNPWn
+Q5i9H/kcpS0H/7RN9g2A+8fiw+2bi5k9C//Udwo3DnQSicPbjFvaT8f+/Wm
Nc59nAdyczWVy+BmdVcjsrCPofa9vl/BZzo+Tz2Dc7aG6O7Ww/7+WD52YCK8
QOtcoxhzySTikXLs+1R1ydgpcGVfAx8h8ukSG8rDzjPa6t8/W/H+gMW9GqMV
jBqzu03scnC/M9YNKzzHW++BCvI3TzdtXnKOUZqBaf14eGravLNNdoze1fTd
ds5GX8Sz2rlwzf2bf0jB/Zrnp9ceyWIUJHyqN4K5zvRnB6xFH0K1zZp6fmbk
cTQkUpGDfEuf+jT0J5iXsDABXjzcM/gBnlTRcLMZfTsauNDXBb7PJb9KkXNQ
cfrhU+inoCSnmQ/vKM/gPYH/ZXgn3wR9fh2zU9sTrnZi8Jv78AybFtt/3Bhd
qlQpOQ0vkXYtVtz7x9KMk9nYF3vHCj9VnPP0Pq1YRU/St/uN8cN+Da/dqFoE
n35/9ocCeECj1EcD+xgsG3gQjXlzda1+yVT0p8/j4e9e+L15SI4I+TxPY3dP
wiV63MSD2PdII53RvchzoOAO7yBcR/+tjwW+D7mL7XMKkP//fZ//AxAM1A8=

      "], CompressedData["
1:eJx11Hs0lHkYB3AkKXJd91uzagydro5C5Keo5NJQO3JplqIMOtnUts1mRi7x
zkT4bU3ZLdek01kkl0IbOuzaaJVR7NZJke6oVsy67df+01/7x/vP5/zO+/7e
5/k+D2v3gYBIFSUlJVc8c/C0s0o1166/3pjYMDo6Y0vJEUvnC5VJDLEhKSeW
wR3brpTF2FFyX3f74spkhgQsn6heDtd6nMhX4HzQzThRUwpDhBe50XZwrqOj
YhTu8RW/pzOVIfK4vENL4R9fv5Emw2t2FLi+OsGQJ7sDVVbDQ2tanAs5lGip
dX49k8aQ906aBvbwuoZU+Qo4d3n7cQOGITGCiz1OcNbZ5DI/G0qeD+/Qs5Qw
5MXcMINN8OCRVz9NL6Hk08fyQraUIfdcfjDyhR/J0bhsATeSNvGtTjKk5E76
/SD45fKRiQ5rSpbEZm9Sz2DIVPiW23vhEuWqo6IvKVGRzj3fD3cx58UchXvH
bHJtXERJFtfTqjyTIYLfFSZpcPZgkf83VpQ8G9moCD7FEB/J0OFc+AD3TdE1
C0qCr064K+Deg+s6S+ElKutJpDklZ4rDlgizGFK1p3GiAT49Wr+m1JSSEmX9
2AG4fhev6R58gXZ3yRtjSmS28zdwshkSbpW/7Sm8MCJ0zQUjSjKShjaGws8m
vhz4BF8U5bz1mQElPX6TRrHw6cMLU+a5XW/sLrhYcfqLz+f9ul8kGMNLmL18
L31KfNWduxzgWmKVIVv4vFgjwZQuJWKhx9R73Kfz52D5Ovhq58LOqzqU+AzP
b86CX1MLN/CDt2h4ifdqU6ImtHxtDNd5d3d3CDyJRZIkWshPiGr0cdQhQztB
QwCPzrvZXraQkoGypb5dqGfv8Bq1I3Df8e8DuzQpidgT80AHrngreCiGZ/K8
NPfBtSpyt7igL3ac8X/S4YEHLilPaFDiWHc/bDv6614o5ufAK4Qs21PwNllb
cTDyUBnAjjkDb77GC8yBd/WMWYUgP2PePbW5cMrqrT0N1/PsafVG3lpiTXLO
wetkt+ar47s2p9P3u6QzJPF1bdZZuL962HdCeEZVsvpi5NbTujdVBu+zqXny
Fu5nt2xwIXLumtc/L3vW7/Ac7+J/O0yLb49hXshBtwAJXKCaHlmO+oQza1cP
Y762FqVxRHDzNEnyDOqZ5Hk+fxDz6BPPXjVbn2PxvyT0of7W1pr9744zpDh1
PCUaLk2adFipR0lcr1ShCnf1pfm74BHVgmYx+nuo1q3ANhF5k8iFXDi7pt3s
D+TBv03Xgi9mSOnTSL4r3ESYOXPSkJKdDZvdK0UMsZ+uE62AF3V7/foX8hZd
2fpJH36LVVdhBjfkvbQ6Z0IJx6FblyYwZFUbO3U2b+EyoxU8M7zn5J02Dlx/
0NB+fHYeE653FSH/oRkfPgwcY4iI+nP64Xl3d9iHWVKywClkpAJ+Kd6C6fhv
LhQW5pg7w0xD8wK4sqp33+y8xD5PqIpifT7/+KH8zyvw7uZHytWY3xYZDX8I
33LFY9fsPG5+t3NEaTElUSv3O5rhPg/qZPvS4fUbH4l9sB8szAcOHYObyt0M
voVrFyi1ytiU2KV5+P0N51F5fRT8sPFvN+Kxf7I9zTqlqMPBWK5aCNwy7MeI
bdhXTi86hpxQzwW58ikfuEh4qcMOe2+oPm7fJLxpfRDbA77BtFVHBjcOMh7r
RF9Uqh0mneGCsv16c7BvUyX5fTfQx//bz/8CFvG2SA==
      "], CompressedData["
1:eJx11P831XccB/BKX1y7Q5Yvjdtu6XQpX1IXi9Z9Z7pxJ3UcMbqr495LSIQr
HGGptMnWx6dYUSrdW0Y3RIYst6tC9xYydyFf2l1m1WgjItae/oD98Pnl8cPn
8zqv1/P5WS6K8QudN2fOnM/w6OHhRTEavD6pUUouvWtNOUATm/x0s6lOivBP
+Cjc4JGGi6U6eFCupalaSxHnxjZTO7jqy5wKQSxNhPGZMcW/UiTRwsSZBc8q
rqqugH/rl3w8+wlFrDnE2hBeqXiTaxFHkzzXy6yELopw3R+umw+3fcGdNoYn
y4NNQrop4mXdunlqWY3yp1qGiAGv0JYv3NZDkfMKH4eX8Kn7nrwDcPOT9QXO
TylS4GZSOQhPXCvcP4XvOq1sMmb1UqRihL27Fz7GzBSMwg2OKsOn4FeF3OYu
uPBiik0a3P2SyVFNH0W2vKJJO1y/7lFRIjx9WnEnr58iDmKd5hF8TaTNHj14
ed+O59sGKDIQMWT+GC49pbJajv2sjYrTTMLZg056s+85W75kUBFDk1NRycHU
M4oQhqy5DZ79/e4Adzir16zM5DeKyBMaa7Vwp5Q9lruiaVL9Y01JOlz0bun5
2flfv2g8m7KfJsMCtb0Wbv5PT9pz+DtJY/e5KJocCqy1MNVRxOiQhj8Mj525
rb69jyZLr7eUbIJLjEqd3sLdhO/zn0VingTdlm3wwxGB6/Ww/8s7V/AWwJds
LWZ4wLOGRBwjeP7nbudsI2iyIfyNPxv+gVXLh1bwsRabG77hNBnpu88Zwjx9
xcMGq+HyTT+ck+6liX68i3UefIX5n6s3wKtjCiUFYTTJGDnDsYV3XrQcms3b
6dsuMlUoTYILVJQC+2mz2ZzsD3+vvmkaBOe9FK+wgD8ek46Hwn00diWHJTTx
dFw8GYc9r2WUGifBt4e13SwR4+4f7eY24F5sqXPJMbjj6NxKD/gXTYEJk7iv
Q1VZ8ym44XEX0x4RTfh2ZsEc+APm4J4iuGDlxx3x8BpHh3kC5EQl4PMq4En9
RU1MuOjWslehyFtHdKavEt7TJxiQh9DkjWxVYCry2ZE6ynsAnyHqeAU86avf
RTnIczlLEtc126OT1Vf4cNcnJFqO/G/J2rpOB/cb8F7iC/cUt078jL7o19+Z
eQXfmVmRtRN+gL0rpBP9ko0sNPwbvt9n2QI13PnC3EVv0ccrSS7dk/BsA8fa
dviDRQf1OfCu8ULLGfinkSl5fMzfEHSsSYhem3Cf5L+H92VbDtTBL+Xe+lr2
C0XM5qeensuuURZedr7rgL2VMQdz33ZQRLk3NWj2PeV/VO0wxv61aaPWYni1
t6Jw1jPGbw5lwC8mBlzrf4w7hi0om4DXihbZinHHk7xbijh4r3W98jV846ZD
Vz2QB3u1disLXufp4DgM92oYNq2ES6zE7r3tFBmbFxj+HK6t3W6Xg1wFVBoV
V8LvBdSl9cCLzEcl0cjh68kMphyeq5amtcE1PePNAuT2hszMuxx+zO7Ribtw
ZsNf8auQ8wm/tn/b4VLGPtEN+JFO73sx6Eu94TV7A8yzxjUiTAYvPdrxzBv9
mvAt4Arh7Cxhdj6cd0SnrIKzNG7rW+DXmU9bs+A5ev6Z36GnHDOvh57Yzxn7
2Olk+N5S/sFQ9PrhCQ9eN1zl/yItHB7bqtJtxH+g/xvXqHTsP8R8qngX/O6q
6WQ5/EJPHHc97vV//+f/AFge2Lk=
      "], CompressedData["
1:eJx11H881HccB/ArUQk1q6ZzXN2plrQ8tmzUo3z6oXChu4pupoZlhK6Hfgxp
a62PJd9z4thjJddvVzRUZCqFlXHkKsuP/BxCRIpbl7n2ur/ur/3x+ef5x+fx
+bzeP+YFSUQ7J7JYrJU4Rjg/5HMSbY8U3WMXVx+/Ns6Qqk2pttzVlMRu7Giy
hHPCHphJ3jNEfNlUwF1DiefR3GpTuFfgzVxPlpRMTo6YsH4tJcK05CDdj0X3
+IryLSYTpaThNTs9bh0lrwLOqUfhdXam/h1w+oLztNKNkuD02TNewh1Kaxbd
NpIS1h5drt0GSuLCV7n2wnXHVhwohDdt/iBU5k6J/+KZuzv19/Pdy/LgdYPN
3HcelNgPZIa0wvdcMH94BW6zs6Zgn4ASX9XlQ43wy85D4xfgB5XTlz/fSIl1
7A55HVw1HL5EAed1TC7x86bE+J1G9wQ+pSxM9DveWau8knXLhxKfCpswvd+/
svy7JxOkJKhwygmukJJmew+vevhAst9dc+Rg0RJ+fJeIksDyrk16v2MlP/kx
cktNlnLObqYkSRn/4Bl8R0GLLBg5W0XzjEq3UPLriojkdnh0QUBf5hhDPmxt
01ZspSTe5hNFN7zGXiVv0qIuR2ITin0pyfJQOvbDhxucns1+C3ffIov3w3/z
B0P1OU96eyB27yhDut/HTHPYRsn1iY3tY3BjruZR3huG8Is2TL8Bl/S4/2OC
OkaZhUVbDDPkBrei5iMxJZMzuxIt4HEa+VyvIYYsMxEPC+D7ukkWG77QUzKW
PcAQzdzzGUHwuEtHeXy4qXT0ev8Lhvjlbi0Rwdf0CkSO8D9He+oEvYb7Rf7C
Fy7wFG92C/Pc8J6ynJksd3hlyX2zwU6GLBFUc5zgh9j1aSJ4PveYq+Pfhv+K
hg7KAuAjied6otoN+Uz1eTsQCt+69vWkwlaGrF/iNO0R8lyors7cB9/kOI9q
mxnS6293+iHy7yxwy46Dn9JmNNvA29JkfVmo1xxJp3ECPDs9lRfUxJCZu+6f
2ov6tlap5Cfg23Utlj83MmRsjvn2BegHk5BM4Um44+o255wGhkSuMY9XoX/O
yFfbKeDvg4e68+G5spDar9FvGdkpmvNw577MlJvwlMPRVhr0p5JhdEr9PQf2
f38H/phXf+Qg+tk+y7fnKlxb+YtbOTzm8426qZ6UFCsO11+D52i/nVUJt6ix
cUrHvLRfLA+7Dv9rRNT9G965e2Rv0jLM19WpAqMb8Nw/+CVy/KuX5XBejXlU
cmYszYPXfTqNfokcQtYuDv4J85vh3Tdf71+IeQrXFoaM2zda+mDexa2Rhdnw
6ENueZPaGGLukrhoKfaDunSd9hJ8v0tUSCXqYldzqX8+9kl+UsXzs/AFF1OL
pKhjb9WJ004EcxGdsC0DnqgWxAi7GBJQy08Lc6UkouiBQ4p+/7AiX42gT161
hD4uXUXJ+B5hYSJcZWJ95ib6KvCr/GoXuI+YffIwPDJ8+7/ifoYUdrGFDSsp
eVN8yzhGX3cPB3/rlwyZJS5zuACvdVL1RcD5g9qE2+jz7pG5P0rhURLb/d/A
u6Ks2uMwF68V3tan4O6aouV++j35NPk2Z8RwfxXvM6UXXNKlvvds1PCesh08
6gpPUnckxGJO58zbbHwLHii6078M7tbe5uv8zvBfS7O7LxfBg21lOaNjhnz+
bz//BzFu648=
      "], CompressedData["
1:eJx11Ps31HkYB/CRFC26uBRToXa7IJVkSrdPit2ISlfHKjuEPS7TbJy1NRUl
ZnPpkyydZV1zMHsm08WoSHJGTFIrJo2m4RgJZUyRag3te/6A/eF75pzXOfP5
fr7P834eOzbH7+gUBoOxCY8+nn2jtWH2rNt1nImCvSLGJfLTFqd6zmpK/IbY
P9jBRcndb8vhbruZ1budKSmbbtkzH55fSwtc4Km/Z5mtXUOJrXcMZx68cJ1m
5/2vlKR4iVKsXSgpjxqwN4czo/SsveDLTHLPjcN9ctJYs+Bng4JtZZOUGEvP
5LSvpSReLgg1hnvc2LElCM4pcN0rcMWvyMDIFC5dEjHJm6Bk4ub6CC4L76mV
2H4DF9jtfmAIZ5jtKl61jhKzetcqE3gjqXBcoqWkgi1jdsNnhhkYztb5Iw/m
9nFK3L6OZJ5aT0nHC08zC3hXqWka+19KTtjfMTVyo6QovfSBFTzBNYeb8IWS
d3PmShPh7cGb+Tbw8e8GrAs/U9Jy56/E13CRW5pKV88MQy7P7xMlLl1DxGUD
vltq7usMV4Twdx4fo0S1x+N0JHyVckWRG9y+8XBd5kdKKo8JNqbDf2vVTtkO
z0uP1YhHKZnr7TU/C+6lDfXxhYd0MiXyEZwj0fKS4foZG0X+8GfC8YCJD5Q4
PRxTBsJtLpdoj+rOCXfKttN5jvbBAri0JsYhGh6k+mxb+R7/z2wtacb9HZ0X
e/PgZLiiqFNDyWhax4XD8E5hnE8SvP2y43EOXBh75VgX6vbhobv6D/g1Zq5e
1jAlryMsWD5wM0XeqUJ4QNiGNw7wqUeIshT1jy41bCmDN8R36K+GT8TJX4+g
j8PGwvBK+GRh+dVGNSX5+SKhC1x67oDkHlyuvM5rgc+4p5CFIg/dBuHch7r6
e5aktcGfVJksvIT8lKnbOU/hkfonO+Tw0ws2JN1E3g7otdMX8Fm3eOu74XUs
l/RnyG31oztaBbzFsTFhP+7zvdvHtWrkPL2vbksffO/A/fEAeFPf4VdG8DlT
Y5MH4Z9yNjm0wu8tPti7DPPy7VYeXw2XzSxe7on67FI1CXxXUXJo0OTsENyC
f0u2HHW2DWjWxK2khKaKKzTwN2LDrXnwVPd5YzeccL7Su1Z3jntN/Y4E9EtU
/PnuyApKiM2PLu/g5l2+/cHoezZrm802ODd6n6ofbnVVT+uBnKieDxaVOGJe
/Hc298KLM2SJS5GrY/mTHkx4a4HWuhseui+q2BA5zJl+lZ/vQAlfqQjs1L33
veW0AbhUUC7YBJ9hOSpvg2s1kldNyDObv/md2p4Sq7v7Ix/Dey37Z5ch/5FB
/kIxPDHE2EICHzWILkrCvMSlMNSZ8Lps1nNdf418Hba3Yb4UnktN0+A/m0t5
f8N7uH6RIsxjaqW8JRf+KDZsUQH88U3n7lTM78VDs3wa4FmqoGtX4JrBaS+/
wBcONkcY4J4x1YriFLg4MMZGhj1QGLTowkE4VY3Fn4L/2aPMq8A+6T39slQC
N413PHMcfvtg5BNr7KXcQGbDNtSHzX4hDocLj0b9ch5u1fNU7zm8itG3NBAe
V950YRj+q/7bwJO6viT3M/bA55ouWeePfZjHOOG/En1UfGkJ0c1vZrX+5Xr4
kdSkTDX8eoO7yUa4f8PFqhp4QtNQdA3yINuVHbxGt8f+4RaJ4e8d2pZlID//
t5//Aywy3+k=
      "], CompressedData["
1:eJx11Gs4lGkYB3AittpK2h1zVZcZatNOSSpZxx6TSai2g1OYMZiEagk1KSzT
QYehvbz1WnTaK7VpKSmmbLnGZFw0rENlbOxIJZFJp02mRvufT33aD8+X34fn
fd77/t+3TVTChs3jjIyMPHBMcExbXrEymmTymayntZP9aPKQWyXtDKVI5Ezq
cTJ8IFPFMYVL0qzNy8Io8tX+uhlxcN7IBEudL01SjYzLpeEUCY1M3SCAn264
92EYHuOYEbKNT5FLxTXTA+CD3inCZ3BNyt02IqBIsNfUSz/Cu8bHihLgGSaS
VHYERWY0qhv94fWJGQMfVtGEnWH55h2cFc9JXgUfuiMflsD5G4+41QgpkhHN
vMCDM8KdxZPhGxeNBKZFUsSvo7DWF/4ry/L0fB+a7DKK8XKIosiZ47ti1sL1
T2xW+q2kyVbtY30H/KTogHwNfN7E2vYOHk1qrqnCY6Mpoh2sHmd4v968eOy9
N02UHzXSl/ASbmVIGHz3dnN7JlzXkh4hEFFE9+CQIhquZvbwXVfQhIqV3ZfB
W0fNqrbBCxyap/K5NPm8132m2WaKJFY/LU6HjyoKor7xokk+o+nBYnjMFiv5
IfgVWst0ITRJZldGBsBfp+alnoC7nrmwIWL5F3cIZvicg/ePCHIPetJEHZpg
6gn/U392XgXcPfNf1WUPmnxb98h3Glx8ZYG1Ar6tkDep050mi3gmi//CO6fM
TzrVDg+VGKuM4cHrdr74GS5s3xKhgX8Ua18XuOG/mBqmBVw6R2b2Gu5079p1
pStNmh5mWp1AfUazKmLG4KLeOn4EnME7f8MEzp9V6zS9WSan7LS2OS40uXXM
+VMY6l+h2b3EFt6qOrVmHnyTrtOiBH28xBiqdoQ7cvM+3PmBJpyw9PeD6LvZ
Uf4ON3hXs25CM/xkvVWZDVwhniHxhws885bGwZdbCKNDkJ/cpL9rAw33d40q
k+DNHOngPuRQnbnsqBAuL3ioToNbX8zhFiO3V20FoyK4Surf+hZe2vR9YR1y
vtDvtnMCPGj2SaEebt7rwn6FuWBF3y1OhlsrT4ek4P07kizEFvCfjpFsMVxm
d7ZlEN7TLfrkhvlq0jv27IGX7ShRClGfuRSvPGkTRe5qr5Smw71eKFw74Gne
8rzrIRSZ6O9zPBXuqQxi3ED9i7p7zxnDc94Ord4L79CUHV6Afmn3VT3ZFEyR
fbnSEzvhCknp46/R90oFyWoIogh9sWij4f0eWbLqIThva+EaLvxl/3r9FviZ
lbpfVMiP/dpGcXsgRd5U3smLgD8vj91cgrw1FHR/3g3XuLedCoJ7f9d44yDy
6bf0URUHvn7g2cTVhjqsEh6OQp6fVXTr9AEUiV9XqiVwI05DpwfyPxan3t4N
D7DLWbwU3jfmcN5qxRffcyB8PBu+p3pWIB/zdXlRl+qN4R7ePZtpcFur/FlO
mFOLyp5sFr4bK7DnGMNjaiS9kzDXrD7Nb1FwYtlvNoIc3rqpXL4TviyxfbAc
7hRq1/8UnmKW/NkX+2HunD/kbNShXq1zN+TfNprtYY190nM1SlkETxht+KcO
7jyXpvbDBUV9dvao8/22Q/FV8PAlq/sH4Kaz2c/vw9UlJtt/N8zjZZeba7Hf
jt2eos5Gv/LHOH35cKaj74Rr8Aru+Xc89H2hYLg9Gz6cKFIzsG9b6zvIVOTk
//bzfyJw6v8=
      "], CompressedData["
1:eJx11Gs4lGkYB3A1maQiQytyKGXXoYTd6HDhsUopKqyxbU7rsGrTlZWyNSNR
z7Zrl+1VOY20SSiVWoPtUowpkmrIYWoiJeRQDYYanbT/+dSnPjxffh/e533u
+3/f80N3+kRMVlNTc8Jh4cQ7T/HesOU/UThHUs0ZZ8jyGatFUQ2UaHnGbfKE
H442mZ6rZIhmsaSRf4eS1bYWc9bBf6hgFlvC81zVr+ZIKFn790TnGvjF6LMb
ha8Zwk7KKLzaRMnY1rf5q+F6hRG/uMAdAmhaRzMlw44mp9zgdQJuZcMrhrRw
j5980UrJTk6W1B3uKVpV8GyMIc7d9PErKSW5S1Y4qr6zYaNiRQx8Wuyi3f0P
KDm9LM3UA25pKjiXOsqQ9eZzj117CN+R6ucFX3lC4VusYMgHzX4Nfgclsrdb
ZvnAfYrSg+tHGBK6LuG1biclc6srIv3hpkVjnb3DDBGzE47sf0zJW/G8VYFw
Q83zIhb8TUb8afETSvJmKlrD4Nkr//rKbIghk77JSW/uoiQ2NH/ydvi3C5dM
dpUzhEnZ7lD2lJKWcXlPDFzUpvwx5CVDDA4cKg/rpuSheKuAB6+x1Qk68AL3
buOy2+AcdUaeAC8oceDOgrOPhSu0eyhx5LrbJMNruTd22T3/5Mp1lWbHVX2x
dPvCd5Ah7Z6S5nZ8x7V04R0BXCN8T13PAEOs7r1P+hken3/LowguiRr5Yyr8
6G2qW4v/DHJhRVyCp9zs8sjuR92W25MBvOt+yG/GV+Af1PzjF8N5bYzBE9Th
hRY/Twx/+UrhXtPHEN/0THYO6ma/xjH1Jvx5sKLnBjztbm6UHercyjc2aYYL
1RNjNsNLv0xZ/w/6wvdvnvEALjMXSwLhxiWxfYPoY3vq4m2d8H9nC3ih8NoW
pn62jBKjAgfrXlUfdWx3RcL1pUr7Bfcpeed12KgPnjNkrN8BH9yhN2rQRolV
3h43OdxMe+L7Lri1WOfZtBZKNrNkicNwndbxpVy8q1tg924AuW1qD6lWwAsz
Mjm34TpaF7QbkPNya633Y/CFGx/JXVC3CotDC85hLtjBSjUlXD0hIk8Iz22Z
qpeNOfqoPl0yCv/1TnJFJvpSyT7YdKSeEjNOynrVvWG882l89PHS734OuXWU
TJrBSRqBm+RFlgzDnawsOLduUNJWF+PxEr4iKUi9FXlI5u7L0LxOydPyS8n9
8KbCmW/KkavhEPHpmBpKOoRRNt3wj5ft9mYhh/4TQpaGiJLaRlfPR3CnK4YZ
POR2/Oa9wetVlAgn+ZZJ4ZuuO9gGIuduxpGK0muUTNFbmtgIz4pt8nPGvFwe
knvdu0rJkMPBTFV/LzplvzbBfEXkzmkwg3f6TOmogj9esv/dBDx9sC7uciUl
hzycw8rg5qaVuzsxp++3/5S8B+4VO05UeVNeqO3zGvvkuvrz1p6B06/l7ouw
H6xsLYrPwfuK/uRlwQMt+6dpYp9Ye+/7MBf3Ks8mGKfB63ku0ji4hnZCdA18
6v1dLNW8nP8o0++FB4UErD2BdzkGnPJMhDvqlPV6Y4+dybaJLUAdTlXu7opT
5fNguE0VXBjDF3VXU6JWvEm2A64lMWJZYU9GmTAhG1Bn6cyj/lvhjQN6o+Zw
mbv3d8/ElFyxT9sZCi8JFWTMh58ccakSoI+6GcEGAfCIZdJBI/iE4d6n0ei7
YU15gB+8++5ehT4836yUH4acfG4//w8d1P7D
      "], CompressedData["
1:eJx11FtM01ccB/DWERSxCsplgiJFpFyndQ03lRyogmBcGURhjstAUEERBWsL
yNVS5Fb/J4MyBQkKQyEiGi0DnDeQOJmZjMpIBRUmyn0RQRRZdN8+9WkP5+WT
k19+53c53OjEoNgFLBZrC84XOO/c/yh22th8Vz/4vDr1JCUdNb0OKasZ4j6/
T/Ql3GjHz4PtMkpKzUw7B+GOs58SDeA+37h1nMulROK+zXGPFUNmnfVnZ/jN
d/0FtydfySlJSY1wHIQHVbMXv4JXiJ9x7uRRcoSGxCevYUh1pSrnMXzhjFGS
fj4l/Z12ASbWDEkobL58G55WVzL8HH5+a4fBLbiFc62sAe5sYsL9pYCSuBDn
ugQuQ9qLXKfL4eWLajinCylZvDzogL0NQ9xehOwqgGd2Vr3ZV0SJze+b/Ibg
fk7Hb6bD/7Io4XbB5dVj66rW4r22Xp+OwF92Vbh7FFMSMrU7JtKWITV7lifG
wAuLO45cgPtpqI3FOoYsHFfNhsA1AkGLoYISj6VnqrrhFwOU1wLhm3takvTh
P3aJK+V2DHHhZA74w5NVR7PYcM9aYdh6HvLJLf1BCFcFGj/8F/FfH/3s0QVX
bm094QPnhUke58MfpnZKou0ZknF8conWJyr9HM3hpcLz4lH4pIAn0Lr7Gc3b
GrxLvfOsOsqBIRLZyk9aj/+piv81vKmk/WgnvMY1Ta7NJz61yC8YdctibTDi
OqL+ys1PtPnzC3smklFn954JRQxc7H9gOBSurkjglqIv0kLWPyXwY4ZRN6Lg
TwbU/U2nKOlzid95HZ49HSs+CO85Ze+vQd9XK7ZzWuGHRhrup8I9+xVxxXLd
/dLWe3py+Fxy6PjVXF38AuESdhk8qdLbZ59Ml49v8q03l+DcNVz/VZjbSt9t
zCq4rYx18Ca81276enc2JRyO2vgB3hvtGn7jobYvgdL9H7JQn7VXgsPh6zMH
8p/BzdquPmrMpIQnaTR9jXqaNq5mv4Wb+otq92dQcnK81zwSHqTkjbMx/1nt
z3MV6ZQoly15+gj9UvEyJObwE3vL7fxOUDLiHdzvCO9KL9tuD8/rtbx8OI2S
Bp7MKgfzQFN+jfCAb5ifj1am6uaHNWg4EwAvsPpKfDuFkpbis21m8L8vFJwK
g5uv95S/llKSaRs1tAfzmefUahwHl2tGfdKlunl+cfrdb2K4KIOxNoF3i7yH
BzD/gr6nYTnw5IWHhuollMzn7V60Dp6geXxPob2fbSX3luj2S+/StJ4SLmnm
3PeFL218UNeCfVRVlqsr4Jv6Rhp2wD187MeM4Fs7doRVw0edhbGB8ClR6Nhh
7LuDW5qwBv5tXGB4O5xvHVHfj//Bc1LQpHXpgWy2K/Jc3Ces3wU3lq6Qa/30
nGjDJbhhzIr45/h/WgPCO7XxFbXZH1eiPtllT+qOwWMdnL3OwRuShl/Owa9t
LHFYA18ga3xfCue+/+irQZ33toXW9a1iyETE1Gwx/Ljmytpm9OVzqL1DPTzx
zzbbXPhYvkucEn1kug1Gi+A7LRXiNPhFPmvZMfT9uzC2Qa7WvVQ2h+CRr8jM
nQzdfWmk59vv4cttXTwMs3TxIzItp0TwYaNI/nCWLp8i/QBjL/hH7tAWfg4l
OR+U+TbI///+5/8At4isAQ==
      "], CompressedData["
1:eJx11OlTk1cUBvCIIihaFEeQpYpBUVqsYTBiAPFKdCZUDERZAwqNYpQIRGWp
UBEaFV+gBbwssrhQWUori4ArDkXEDmDESmWxMUHcqLEiGqgyDbQPf0A/vF9+
H+577jnPuUslMdsiDFgs1np80/Exqqpaa/drLbe97caFYko+iXLO0coZUpny
w1sjeEuzpXROKCXn3WZ8sh1emEu6dW7XWqj9g/SOHZSEz3+9ojuGIR0L5k88
h9cHlxVahFHy8sUX5l/B/Q0ibX6DC41vL30YTslmtVu1KXzlVhflDXisfb9v
toQSlY2n9FE0Q3pG/uWUw9mThpu8d1MykVG07jI83ZJvnwXflDBdY7SHksbj
I44X4bPIBvtkuG1BvJOflJKBJMvOVnht3ri7FK7td7Y03UdJ1d7IGSNTnqhU
+MNDRJx+cSQlY96DKmfU876nwYsP/1vStLpCRkmxyi60AN78bUMfB352j0D8
bj8lbcaPo4zRh3r5kzefwvWNmcvcoykhh2Qv8uFHHRpE8+CBBfVqixhKbMt8
nbgHGCI3rwo2gBev/uitg08Uelpp4ZwPtktHXa+1HJkUnOySU3I8OLmv5iBD
ik+nk9fwla2vbrgeoMTaZqF96iGGsBUct6fwQS/t/Qr4z6c4KaGxDJmW4vq9
Bn6LPnh8Af5EWyLbHMcQ8YL8m3/AQ4ReyvNwWnpM4hLPkJQXI7peOM9vvPUM
fOHzneYrEhjS5TOU1w9PE6a1fA7vSK/tWfI1Q9TqPJepc2wMpjVfR50XB/py
Fx1mSB0vSDEIf8fJHuvHvYbXLs+YnYi5dxnp/oS7n1v35Uf0x/RZY5wO7isY
DhqBn/AQFljAVzTNvKBJYgjTGpk3AS/XHPZsR5+5z2Zfv/oN/McMIxP0bY3I
0PIw5tI5cGq94gjm8l1A7SJ4ue611WeYo7FFGWdjMkMybHNu2MOd2rv2q/bi
v5X3+C/hd2e/S3OFz7qcPWcxctKdlHnnyFGGSAvfXhXC1RZn/wmPQP1Ncqke
vuRRWM4OuCxRrjNDDh3rdAlRKQwJGr4qOwh/pveWBSC32Xka9i24Rw63LQ3u
o3cVFCHn0ee2LR+DswpKRWfgpKrAR7OTkqyimHzrVIbw63g7GuA1puomNvbo
Id8qjA3Pvtvq1wafpz0b0RtCSWDJ2laT1Kl6Ylar4BcjSYI19tRya/RJDc4f
nxOr+gseH6qvux9EyWn/Xy4VweclBk1Mwsc6qgXHAinJNUmK48FvalftnI+9
3tIjM+MFUGK3cXR3O+67OE61Zhn8vN8H3rAfJbyhe8O8qf605952gl+oD48z
g1cMiMuK0c+AFH71BrgPiz2+djsl/I9DN19iLmZ1Ub1b4dO3eNWGbKNkncRm
Ghs+ebCyNgD+/pTNobciSsoi+hp8MN83oR7VYfDNOcrtCjjnd404FnkI5AaG
7IJbPW/2PAEXXJfIM5Afj+NPV0nhDunNQgZufsUhsRA5zE/zYsngvqRUnAnn
pncqKpFbk5ON0n3wzl3csEn47sGsthrkvOKnvq6pc2Qll3fJUWfckKPhFezF
pCY5PRyemntnlQj30uuJ6Ffska6U5xwMZ8YMXzmhD8EzWHYq7F3xqGO3CF4w
t6XEzJ8SZabC5T32NFdjX78JLi9VzpSgz71DQWpzePzwiK8LfJYLd/0lzIXM
jQncgn13q8ipcYAP8EY3sIJxjkBZnoX34f/e5/8AlSLILg==
      "], CompressedData["
1:eJx11Pk31WkcB3DqZApzTJax3EuLslcqLUQ9oWRp3OsOirhpmaRVlzvpMmRp
VJ5HrmtnIqc7oUgMkriGKJLlSBKVNqeplGk6Rpt5+wPmh+8vrx+e5/O8P5/P
d96OQ167pykpKTngm45PaUufraVfjcKoxN75FqUkatmCsTEuI3yZVjwXvqp0
r290MiWxj+c7OhgxcmLbwLpv4SkqFXU2KZS47Uh9dH4OI1XTOuiXrTUKpWM5
OS+llJwQ7ldZOI+RojTbW2PwgEaFS0MqJWraJe8r5jPSrzH66zCcWTeJRWmU
zBC4/MRbwEiE3ND7Lnz0eWrOpnRKqmWOgx8WMhJd5MrpgNe8eJu7Ft44EXfv
gikjb3vrWhTwzWWPMrQyKCkp9dDwNmckd1F0Rw28+mrIyCz4zZDCuI8WjMxU
nC28CN+n7KPzCG7if6wqz4oR08R+fhE82HDk+3GcP2FhmW+1mJH7KevfFcDd
AmuNRuGeV1nClSWMVDQJuDlwta/lPc/h3EiOEWcpI7GXBe8y4Rpum+wm8S71
3CarA8sYiTv0lZ8Bnx555Ad9+NBnxWjBckZUgowOpMNnN3g528goOTJ3+/bL
NoyQ9jheNvx6+SedUOQZaBGTULCCkSbh+YdTnmjn3kmR/0Op3GfnSkY6b6Z7
58NTz0ycvo1+de36TU95Fc55XiSXw9Od7/PVGSXBTuWtofBdIkv5JbiFScUr
jyRKpCtK99TD9dzzSv6AbzM+1cROURI0oDPWC3+QLA6ph5u38yK7EylZezrG
4gZcEqUacBue3HxHuvIEJcWZLuw4/KCdalcfXKxqsE0ST0md/6xzuvCqeifN
J/C2IPu5jbGUjJXdfxqL+tPiTtqOwmN7j058c5ySFv+W7nq8d+btsYWT8OXf
WRYd/gX1i9xEQ8jHIyG6Uh1z2LzF2uZqJCW17ZXKnciz2EncZQBfVyPfO01C
SZmMORUif23/inhzeJIwzccjgpKK4s5xX/RrfDI92QbeqqvaHH6UEpV/BcK3
6O+W+FYHZ7jO5bohhZiST43D9ocxD91SnjIP7lO62fxVOCWrX24QDGN+njVM
rtkOfx2v9mplGCW+5sFPnS3Rd3e2bz98VtRAdZCIkjet5ofOYj5T/1YbPArP
CszsSTpCiZm1NOw15tl9+PCHGLgopSwvHe57mm9PTJCbqsiYwntPmuxpDMUc
5h9syca+dGT9+F0a3CNj7uc2eM/IpzPj2C//gcLsXLh3/hv9Xnjw/J7CYOyj
9t0FTWfhc6oeDF7D+bN7DGqfYH95S4SpcngyZ1S/GX7K5xr3GPb9gbu/4gL8
S4TkUjHqZzZ9M80MGSkL2CAshnsGXkw7g/dy/wpb/57DSOZ4lnMJ3O6fO3nh
yEeyr7DtmQEjy2rp7t/hnL2r7q7+mZJh/vH89/qMBJ27c2/q3nAeMeQif7/P
mu1L4Sev1O8qgOvubNk/hH49Vg+bIdNjxOugu24OPF/qV5mH/ramXBIY603t
kQZHBl+Tk2UWEEXJVo3O8j5dRjJuvNNLgju2PRRyoikhav2mCrjWyOKAOPif
iZ5V/TGUvGjvHu2EG2uHyiPg5Z6yQRnm06qvMkkL53e7Zn0JgXNdtD8uTqBE
7BsjiYLnjEuuC6fulbxxfIn5Xz1m6qCJ+j13a2YK4K4bXcPPYV8WrcgS9ME1
GkTPNsJv6Gd/9MN+lXp1VVUjH2sVTx9buCXfTKyJfdyzoWRRC/L8v//zf1aT
yDo=
      "], CompressedData["
1:eJx11Hk41HkcB3BXs0ubI7Y8JoXNsYq0kUTtN5RCzxYzqIxjJradHLUdrpIz
bcrix09W48rRhUZiba6USjSYxBTTPtuB7VFWUex4tO/5S//sH99/Xn98ns/3
cxlywz2ClBQUFDbgKeNpaEkin92pa/nQsM63mEeT3Fmfgfs9FPEy09jYBTet
Cepg7KXJDlaaIEZMEZtY9kgTnBJVruLDl0z/Uef8iCLzvYKXVcGlxht1RXDF
lJhSpV6KBBXrWJXC7UwUG+7AbVtUmjvgZpZ+JA+eYxfkfxOu/bedJOsxRRKm
HXdkwJcF8JWq4Ztm9cY5fRRxCpbdPQsvUTs/ZA8PnSlXNe2nyAFJJC8JXl63
WX0NXH30z8mXcP88vYl4uIUke6ZR/q8vum/kSCgyNbw16QT8SLro9VZ49eXj
R1yfIE6/IFseh0GYxvu4NLmt+WlwCB59u+n7RLhLyUX9iUCakF+3Oyc+pYhy
TDv/F3jbQXuuDtzw5RlP1QGKiE9/cy0drj2uI7QJoIlba5daFHxmzC/hHJz5
fiLJ258mIk8NWgoXrFYLL4JnvVPhRfvRpIe5v33FIEXyNStGLsMX35t3V8Ch
ib9tb3cYXGGm79gNuOvXi9e0+tLkSgg/Lg+e16JXfwtetmPLqpE9NKn0/OFq
Gbxce3+CCL5b0/ya+mfuNHv2wXN5PpWPott30cTAKZmVDu+v7bw/Do/v7lJ/
50OTJuW3JTz4idfaLMW2upbiQmtTps9cniFCfspCeHjx5MPN3jQ52mzMF+Nf
vJrfXZbDk+8+mTrgRZPd/S+ehcJzdY8a2cITwhpXCtg0Kcmtkr1DPdPX6px2
hceGJ71sZ6E+X/JHQ+DT7h+ZPnByxfk5Dz7SoSocQF92LRo2D4U/DW80yfSk
yfS5VH1rePxbz73H4BsG04QWcGHZY048+m4l8hKkwnfxy+3bPWji6CAViDAn
Wqy1itnw6oiFC8Rww6jzQk14T44DqxS+8n1EQRh82IiR64I5pM80O1fBoyfv
TETA1WqZcbGYW78GqXU9nONzfVs8XGojDb6BOZ8xyVl+G36V6ZR/Gl5b6rp9
FPtyv7D4ZCfcjt0eSsHnb4n3t4TLqn/S7oZbJk8mqyL/eW5um4Owd4e2rEgW
y+PIarJi4VMXFyRWY08LuFEfeuGvHDgX3sPTS8+2fOqmyPpPr9374Cqjly7+
iLplWJl68eBDjtPV8vgeDMEbF9Q/26B5sbgLcVKNUh/AR/69V2GKfg1n+s9z
hwfua1snd/s6VePf4NZSrWU9IooU6XH3tcKLxjS8I9H38a90M0LgaRL94/I6
/KyUcYKNORkziNuzFN55ULfwkrxu/Z1D+Zi37wJ0PQYeoo+rZbH5cPbOAVnU
bpr4KQQ+rIezycbuTHizhSSVhbntO/zq8AV4+Mc2xkl4Ut2UlqXvnHOXZhlF
wm3KlqYxOHNx3A+RJi7cOPhUqyv2K0bGih2C12SxSjzgkjSGuSH20c7yZsq3
yDNAeGtmk9xn387/CH+xbWdjBPz4Jd/1VvJ6mrm96MBe67RtdRuEn4lexNGH
a4+NVhXgDkwVUqu9UTf1wAIfTXiFfocNA/ckZOE/TlL4yr4OsQz7tSSz9aAE
fkonoTIKfVFg2F5/I79L9R6jG3CXHFz+0l2E/v7fff4PgqEBkw==
      "], {{-6141.520263637728, 1065.0199692309463`, 
      1354.7232213505754`}, {-6141.375453673713, 1064.44281280683, 
      1355.8328530013837`}, {-6141.336988004275, 1063.1113154722705`, 
      1357.0512225724592`}, {-6141.705219925768, 1060.856698142742, 
      1357.1491250198924`}, {-6142.181253074065, 1059.3933926577376`, 
      1356.1374502529723`}, {-6142.373337057584, 1060.0888980352534`, 
      1354.7232213505754`}, {-6142.256593795832, 1062.0559808461342`, 
      1353.711460954877}, {-6141.899355040392, 1063.9815421913622`, 
      1353.8202541070577`}, {-6141.520263637728, 1065.0199692309463`, 
      1354.7232213505754`}}, CompressedData["
1:eJx11OtbjHkYB/AUnSzlqixNtCmzuBiUDor8YtJhS66sthJNTSFSNCOTtCjM
ZArPJR5UO7El20FOu6OwO5Jh2jHJ2O1ApUHKIWW2Axntd/4AL35vPm+e+7nv
7307xCaHxhsaGBgsxTPCk4gzspoapLLXi2qax1XQRCJrv+8QSZHIeRHNKnjY
iMRTV04Th0yP1jfwgtX9iQ/gL7gqJxG8OY5/uWwdRbwLxo9Rwg+yGyZawdUl
SaKEKIq8d0953AgvtVy2f2EZTa7xImdOW08RnpP7Br0vCmRybv1Gk/iry0V3
4VvdJeX6epTLG4OeXqDJoHro9bYNFMlgT3rfDt8uTmV9LqWJ5Yi2ZRResXRx
ZRd8xg9ph+3ge7aYHT0UTZGAuY5dvfrv8po4S8/TpCOEmBtyKCInITbDcE82
c250CU1yN72YtAmuGjDONH0olelqlT9bF9Nkn+PNDhncJGLBDCv4t3W21e6/
0qSba+05NoYit73k16fDY9+6dK87R5Px4QPfLYTvPlZs4gwPHAqeYn2WJi38
uDw/+Gh9ssUyOKX+8vviIpr0lTNUofAD3q7ZwfDBmTdqORLUY1QZuQRezPqU
EQX/8Dg2QPQLTeJCD1bawnk+2tk8uEeOoGFFIU3OBvTkaFBn0vXEhj1wmXZi
dUsBTbL7c5wl8L11maVH4NNrl9iNhcuttXM84YzO9E9FcDrz5TArnyaMp8Ie
BfpW9E2CaSWcf3aEfe8MTQJT2Umr4E9ipEoZfLLN6BHtaZpk6aL23kT/31xy
D1LBD29R+OfCe4TCMHt4u6mr8B/432sNeHlwubhGm4r5Wlhk7H8OTxBNPM+C
J+6oOaZCTtIGi6+8gdczJtS5wQv6NNUWcPvGlOdauNnOK53e8ONKhWY18ub4
1vXoZziTPy/NDx6mzms8hHz+ywn8a2yjVDZ1KuNkCHz1if8+3ougyOQDFWXG
cHZeqtNdeP+JBd26cIoE2fnZmsJTROkenvhfTYCZ5TL4x1Zuthl8jc+aoCq4
WUKR+76fKHKnXrXHHK6ln710RN/4Vc9PKcMoEjWwdpH+u4L0O5/1fV5p1O7i
BB/VHC/X+7WkXkYaPHlViSp9LUU4dru36lD//Gpv7wjMsUI2q63nR4q4WB+q
GoAHTXfr8sDcTa32zeTCH6rv7HoHj9/7yGoKcjJn+9XSV2soIlhxTtgG7+O1
rOAiV/4WisJMeFmARZsaHvNok6EPcujfpnwwC36fvuGlgJ+p4yqnIbdlzvId
L0Mp8joqrPMPfQ4/aumNyPlOj1vZ1+HHdsWHlsFHzDkCH+yFV7TU5hw897LG
r1C/L+Wsdgb26CTj2pOT8PUXd5iK4LbhvvUx2LvWzTUhlfCO4QTLNPhQx1tf
T+xp/uxanQoeXFLjkAB/V5iinIR9t8mttzdBneeHU6ThcIHBl6ZuuLfjg/RA
eLJN1JOV8GBhM+9P3I2j0tsHq+Db5xv5e+n34tXxIibuzLMMicIefTMukvNn
6XOrcl45As+6EO2XD+832pbMgFcOXjKMxb3qau7zmIe5DFTl906AP6t2Y9+H
LyxdFVcPrx7y/TCKu2EpuGTOwp1MEicrUjBfG3FOjf6eFKc/amPCbTeuZzGQ
kzRf5r0eeA/f56IYPuZLt3ErXLiZe6oTrtWUiLPgpw1dOk4hb3l3x/U2w5kO
3/fuhuvkjP5Y5PZr9/l/+sn/pg==
      "], CompressedData["
1:eJx11GlQk0cYB/Cg2EGJIofGECAcAipyCBrLEdhIQCi2I3gBFtBgPDhEwSAi
6owF1JEjq1aFKClnxQMVhIJApQyImBREFK3gjKKCjchRUImYaf/55Kd+2Hln
frPzzu7z/PexEiWEiKcxGAw+1nSs3hjGbmJb27R5zgy+kSMl67lmNpkFlHQ/
KjRcAfey3cpNg2cMs3UWyCmZ2ifhOsEHfDY67YcvtxexquF3Xx/1tYYbsHOr
X8BTnA45hvxCyQi7by0HzghmtvTCG6bF+o7Ah3vSVrPh/n6O4TvhiUUOYVmF
lGz5LaHbCM5O6GuOhsvPhvMsiihZuEcaYAj/40tY1ehSSpTZIuMbcIvcsEYD
uNq+p/gQvCbTd1RQTEmH12ieMZx/syz5ggMl0Z2FZ+7Bo055MrX+3aspzhK4
vDqDISyhhGsya4AFf7qTFgcuoSS256PbLfjkQLS3OTy0MDlj12JK3ph/cWKX
4pzSdRp7+MQBl0DOIkp4bbKxRHjywwdd2vq8ufTc3cueksHuE7418D7fUKYn
vJwM64/YUlLnzI0Yhmf/q3q+Cp56s/3GXLhHcJQ3pwz1YI+brIVf65XyWmwo
qdCsGfSCdyQEBkbClX2r9qRYUzLWJOMGwBm1Y6nx8Bbn2ghHq6/7vxhVLzmk
7aM4p6SfS8nD1opkO/jo0VOh2fDnjdeF5y0oqay586cuvF/ysCAPHnu5yajR
nJKJ8qW3lNp73Y0PuAL/PK/gTpIZJcdEAtEJOOucelYdPHyqIj+Pg9zElMqd
4RLVjJsKOK+hQr3WlJLWq5/7HqOeB19m7XkGP3LS49h+NiXVKdJyMdxBVlTx
NzxArbO7YAElfi63+Cr0q+Vis94E/HKk5QM3eM/n+q7d8NvubKGuHfpluXfX
ZhYlpTqDE33IA2uoTWYAD1qUenhsPnIy8x0rCL7Ce+G4KfygWv/0MTg1uedR
h7xd9fNtt4O7zjLUsYDnRZnYmsFPBwU2OMIrHineWcMvStwMjiC3SdsMry+H
p3eE99jDXUJt1S+Rfw+prNATXv6psWkp/JWeql8I/2fHNFc+nMp+7pDDz/Em
davwvjbUzxX4wFNvBPGMcf6YZvHwPHhlYHswgfv/JFBlwFWV8oDci5SUbXIV
CeBx/JUFk/C0pNoiPfjkmW4D7f8jFiklf6E+OzTLVmRfwPdpL/9beFhq/Hgd
6mw2ZMwwh1/jbRhyhRcJTVj56Mv1xeLptTJK3nYFcLT3vfa+9sUB9PHDsINP
JPyBq+sOK/jxoe1Jbei7xcZP70zhzcsmzmrrOeOI4ngZciJJvH3+ST4l7evc
t82GP56pYI4jV3mvF4xUa10S4cKAc+7bS30sKZkvlPidgTtvlWnG0F/P/GeR
J5Fbzfu3zBNwM8VHRb92PozVp/RYf92/T3+daTec+XvVSquFlGzzya65DDdY
o+xsg0tjIyw1cOvDzluU8LgcubwGfnB9fKe/HSXsj69SdXH+qhcxIyVw272r
bKR4pzkhnI7V8I5cwa/n4Iqu4MIYvOsP8V1hZ+Hfx21JSde+o+TM15WYA607
MzdOwdlKk0oJ/NK8H0KyMDeyoq+cjEWdVQ38uO1ws/pvxGLMGbLPXT0A93R7
MudHeGRruqYTPrv4ynYR+rimjHk/CL7p6LLwS5hj8zd/Ek7A/28+/wf5hLHK

      "], CompressedData["
1:eJx11FtQk0cYBmBEKgFBxRNl1ApKNdQDgkW0VlihgBYMB6GowFRwQKoQBNRo
pAUTBbW7ECzgoSYc5BgVrU41DRpAoFJEgxanVZAaYShWhcCgQiWhb65y1Yv/
5pmd/b/9vnfXISYpJNbUxMRkHb6J+CxGo7QfRSvq/NMEu2xzGPGlVzsb5JT0
xMUrp8PNknMfvII7RPpHZFykJMysTWYJ9865NvYcPm2Dk8y7mpJzVgqRKbxg
5LPFj+EP5iQ/m3SFkiLdQIZ+u6Iufto/s7XZjKjeDfe2/USJSY5e9Bb+RbN/
XC+8J8xjB7tGiai7uXkI3hRUeknHGJHaREr9fqYk/YfAgAH436E7ySx4QGLI
puHrlAToV7i/gitSx4OWUUYsXFoTTigoGfEc17yGC7fuGIw7wcjY6Cz3iUrs
L+9QvoRXPSpZmHGcEc6iKcmxNZQM+qij+uGf5gUNKbMY2RO9e2blTUryam4m
GvY5yN8ve5TJyPNLtrJfb1GS/8HNuEH4bZfbVtOPMpI/O69VpaLEV6edajiX
dKp5cKCYkaqrbwoltZQUZJccew83dx2VsMOMNBVmb3aro0SxwFY7AX3j672G
WtMZ2e58ZUE5vLNHpOPAA/mFc62/Y+Sb4YN/DsGFVkvNbODaw8V7eWmMuE71
j7eop8Sze98TO/gCcf0didC4fuSrrzMWwu9mtc1qP8DIqhgzp8vwJTLR/OXw
3Ht9kg8Fxno2/nhfvAber34xGrXPWH+uekvMerip0wGeeC8jc6rDcptw3nVz
yn158OXC9MyWFEbsxE9i7qE/k0/xy8Lh3cEJ8sFkRmLLnB9Xo5+zw/3k0fB7
SaICO3jnpHa3BPTfXxpIEuAJ1kIXrz2MmBc9fW+BeYVIraMEcOd9Uv7uJON8
uSaeGhH8SZUDN59vzENYymT+CThXuM1HBm+oajgfhPw8bO0tLoDbh9o+bE5k
pN5jCjcPeVufv6pWBn/2ViB9mGjM50lNS3kJPNNTKarBPm+P+pRZI8+nHDye
yw2ubE5p4hvzv/Lii66rhv9qrG3VcBM/R/ss3Bef1F/KrsPjV2sf3UD98mpW
3IL7tTGHt1AJX+U29nkRznvdvfb1vCrUufrM6lvwGR4Vi46hP5KlFb9lVVAi
WMrLrIHLHb3TeKnoT0zHWptySsr+0F8wrM/gtPq4YS7lLsFj1aWUrFB1WRjW
p1XzDv2LOTrZks0J5ymZP3lJ/Q242jK8T4W5cyWmjZtKKNlfGq03eJxmTex5
5MS9I1gSVkyJtmtD6DVDPRle3DjkSq3zUoiL0OcgVc8luCxgum4xcnineVlt
ZyEllu/N0irgdd19pX3fMlI6L+T1bnjo2QsxRfB2/RZBJXKu8/AVOMCHl5zc
dho+JdLaaSfuhbqup9cSHjx+X5QDT3+W8pej2Li+YUJ0x1F4XsO2s5ojjGwV
xnNj4U8POvscgrd09a+UZhrrGbDqa0+Cjyw2bwo/Zqx/ZXLSu53wpxEvU13x
Dsyt3HWXh/OSvLU2EYYcvrHzfIl34/CZueO+6M+M2pmqYPjp5D3cxu8Z6eco
A3agn0o/zhU/eP4nmuXn8P4MevPdK9H/QOb68Tq45bliaSreK/Pft7pxMK/G
U40cV3ibw66IL7ON8x147HhkEfzyQORZ+xxjHv7vff4PqQ3n3w==
      "], CompressedData["
1:eJx11Ps3lHkcB/Bhm6XaSEStJmHsmt1Zi0TJ5TMVJREppbJpumzZ1nRYoonE
ly6HyUNLJ7dMiqJcYnIbw4jcQiqMWSSZUyqbapOjZT/PH7A/PL+8znO+z/d5
fy7GfMG2Q+oMBsMRn6/wceEl+G/hV9SxnleUTN6moEH4QsCWE2j0GJpPu+6c
mbUE3b197p3GBgLnREm97uipYzLlQnQ9bqvCv5EAz79CTLvyxVZhZiEF9aet
mZ+aCDy031/khd57j19cVUBBUmelV1QzAQ+h7ssd6L9YSJl9tyhIbBtjTLUQ
GOrOFvrS39VwtjuEvn1v64RPG4GlTWWb9qIP/13yOeYmBXLLwI057QQigl9G
8tHZzAFmTj4F77yfDHQ8JLCfZbkiCD0NzjdvzaNgjUva1GAHAesjKTah6POd
np8IukFB+5WGc82dBBrOu9adQq82ao9MvE7BySBVZ2IXgfqTO3vOo6cYHPZ0
zKWAG51wgfuIwH25dlgyus2s/zL/axSoL9nwOhOd4119KR3dyAJcI8UU2KUk
yV6iW0j38a6jVzGVPZk5FIi6I6I0uwnsPjAcVYY+WJlnu/UqBSUB0cdm8f3p
kFpnGbrJgzei49kUhKwxc2lDZ93XTW1F3/De1Cc5iwK2s6rhKDpvUnL6EfoE
y92QjS4tdcztw/sHTl4bG0CffWCY55pJwSmfdAdTdHf5zYZR9C2Tnw0VGRTM
Fn5McMMchAmxWh/QTU9y3L6kUxC9ymG5F+a2S1o0Oo1eX7gu9CK6fNPhHRzM
mX3mwsws+uupxMBk9GnzgKHXWC+zb94Hzj1QUZdRylX/Dj3ht+U1aa0EemL+
tdVCv+0nD+egh51JUphi3fv0Mn7WRZfbx2v/hK6wKt4jfkCgbq230xJ032MO
Ekv6nPXT2t9iX1mZhKax0FXfFyXboKdPKPWF9wn42NaM064c5WnmoydnDvYp
sJ+V3tOOy9FHn9qHG+L/Pu3VHLKvJ+BVywujnRfa2StCDzc/WCqWEeiQzNym
3SDc2k4dc3ON+zVuXi0BEx+/vKXoJCRlzii6b325XUwNgfagJ6do38gOZe3C
/I9tf0w0qgn8aBQwRP+Xjk1MlD3W8TJ7MqmwksALvq8encPv1yZql2HdnVSM
1fsqCIh7tJ5poHdD1t0Z9C1/crOs7hEYEelsVkMvjRzf/wz7x0CgftpYQiCj
YMryH8z/xjOGhxX2YZeUnb+qnMAV16fpb9GnPD3XL8qlzy8SB5cR6PrUIqLr
brx53sh79Mp1OaOP7xIYdlj45i/0r7Ov6D/G/h/5tCKWj56q79XYge7A5MSb
4BzFNhkcXYxeZhK8oJHu/2K3CTWcuwOJVuOTpQRkdW0DVehvB0uZw+hdFi3V
M+g1HH2dYvQvTjvzZTi/GX5x5qvwnAU1tpJc9DHjdyOZOO+qS9sKUtEVgpLG
y+j9ArW7QtwbSnnJOTO8v4DfvyMBPSToppkf7pkmmfqHfvTFktQ/CLrUM/tW
E7o40n5Mgjk0ORta03Ots1rt1UrcV21XLfcWY24KW6PdIeiLTJ9XXUWf3R5v
3os56xVw7QLR5StdfBfcoeBixFiOHtbr80HblgB086Jl1RHocfnezUeqCJz1
sNGk9xLrhHhTGLr2D85H+7HuFW6XFu1EXz3OKA5GFyfHvdojJaDBLZ/xRufc
O64mQPeMFglV2FfOWcJy2rXiL1DTeJ+NH1WhcXUE/m8//wfOY+6s
      "], CompressedData["
1:eJx11P03lGkYB3CxdmppzWlQyRExZKUxpNJu48ZIb5oolJEkUhQqSms3VHel
neiRvE0rg5Q22uqEZcq01EYMqvFWiWWtao1NUY7Ufv0B+8Pzy+c853nu+7q+
12UWHOUdqqmhobEcjxaeU/u7xyoDKxQNrvXKnmcMESzkJp1dQMmugP6b1+BB
cdfr1j5nyAnjoIadCylJMl20rxguaUt1WAm3KfboE9pRkpfHnl0Ab3knkLjB
QyStD6zsKWGJkuQyuEhQvq4J37f3C7vDWUTJhtilwVK4t7ikqB5ewf8gG3Wk
xJqtcsqBs9Jup/rCZQPq8kdLKAmOJyuy4Xyu/MKfTxnSN9qmznWiZOOp3oQs
eIWLYUskvL3B+Kct31Jy4MqI+aSP3S1YP97JkHPuXbs+fkcJL8JLd9L3BO7W
PAnvG9PoOS6gpN7rUtbPcAfJmeyLHQw5PG9j8oAzJVn8+NBc+PsTlm/t4QUc
w4mFLpRkN+oOFsLXH+5w9WpnyOgME5W7KyXRUtb0q3B38zDN6DaG3O985sNz
o2RQs/32TfjfjfYLUlsZsjzlTs0gPHn/spEyuLbT+jUm8PZho4wfhJQoxQaj
1fBPP4oiBCqGHFV4CNvgmdXpqXXwDfwZ44FPGPJvfID0C3dKzhSWtT+G6ys3
6SQ+Zgj/4lrlBN73KzLv6ILXXymWyR4xZPi9PEsBj4x03PcKHul/4lJNC0Ne
ijm6vvDNUzPy38APvk4v9YGz1IvF5Tgnq2+K9BO8RMG+FdfMkLMCo+N9uK+R
je7El1srFKUJXlIduKlGi+Vr1Ofx1SI9A7g4NS+K14Rz/rEkupZQItlrIDGB
zxnpN69RMqRnzJN3EHU29T9yZD68N8eC7wffdkZ8WRt9aatL7rWHh8j1Ja8b
GaIXoR0Sgz6O16bsXQr3YQZyhuBuWroVymWUNBZEJbnBm2uEscfg8vPJ66Yh
J7WWE6Wr4BPS4x9Owq/lfNZehFytaX01LIJ7F5pKTsMPz+we8kMOI3+x9/SF
/x77tV0anPuCqxvrQMndWdH3AuDhmV0vM+C734YbJvMpEQwJWIHwzeGtejo4
f+NvN/hFPEpcPh7ib4V/ONJskQDfFZU/u96WkgGtbmEo/NZX/HNH4fvu7JCO
21DScZqvtQNe+6BY/B4u62pydv6Gkgg7mec2+BURK7gT9UwaipuVO5+SOPV+
8aQ7FU297In6U292j4EVJey0N+oguPO0pnQFfPWU80PXuZQYG7E/+8MHl20q
y0d/R5aGVe2xoGRGkMOBjXCzGxt2HkNO5kt7unzNKek990+MN1xzrNp2FJ5p
VmkQMQ/13JJbuRp+yLpGow15u3i6YaDUjJJbvLQwV3g64xhSjny+ir/LMYXv
zdMMdYL/lRHgkYk89wtV2++ZUpK+5WEdD7778ovKA8h/Y+JznSx41ieTBC58
WtHHEh/M0dNkvY5cuPHUC5lz4fqckOxaeIO8We8Z3OKdgbY+3FcefD4f87iy
ysRfiP+SHyKrWHBbxnp2IubXJVA9vRcuWBuj+jw5F9zElEH44jK+TQXu9USY
4D88uQdiRDb12A9e1SUp11EHw1+7t72Es+2s2hZjzzws7F/VjrrNFBoNvZjc
Y+F+1gXwkRuOa8wtKQmL6BxWwfvjLxjqYY/F2YrmZKIvnBVVOQ/hlV3bFd/D
Oafuz7W0pqTknodaAa9VWS7oh4eHWukr0ff/28//ARBF3E8=
      "], CompressedData["
1:eJx11Pk31WkcB3CuLamUiJpTkSVKIWWEy0emhRZXZRnLucVNppnBbcrUlTIS
+mos329pcxRSmpE0KjfR3JMmNdFRRCjpcu2hsrWatz9gfnh+ef3wnOf5LG+j
4IjNO3hKSkp8HBUcuy6d1n4nqSxGUiEp0+IoQXjXy8aCIec6QYsCLpJLOfMp
HFU/TE6fsoihqsNpda/hZpm+O03h9vpRtmqLGSpw+OOnl/COvA+DRvA+t2MR
upYMeUVMX9ECl43VbL+K++uHX8TYLmFIIL55b8J5efyLTnC9mvJvti1lyOm5
pc4ruLpV482Hkzn6MaAyOdOKIUWgTWo73DORK+zU5CgobqC6y5oh1fX1uj3w
OLnkihq8uXVJ9bJlDPU1eqQNwM8aPLhgPImj+J/vTjtky5BycGn/ELztbvER
Vw2ORLrd+k+XM/QkoOfrOHxz76nrg2ocbdWO2KRnx9D3kR6qWnyprNviVPM5
VY4WXNpQ6f8tQymxmVtmwT/Pt+J7qnD0TufDohR7hsK097svhNs5+rSnKnN0
yeLW1+KVDPkKwy1t4Wb7mkpdlDj6IjC2r3BgyPH1k6bv4Imjne9jv7Lk+dqt
4bYjQ2krOS8v+Cy5i4vNF5bsu493XHRiKCA9qCUEvns2t/CXTyypxNSt+YHP
0G8FMoEYfjm02t7kI0uJsgRPfWeGNt7rXX0IHjn3etGzMZY8hAXFefBPVRZ8
Dq5p9PiGwShLB4cN+rVdGLJhoz1y4Gybv+ajYZYU8Zrx/nDReOKba3DXtHlP
Y4ZYkhRuHIyFG95Yml8Gr+he03PuPUt2wrnSRHjtAydRNVzn0b2ULe9Y6nzq
5rMLLqbqrY1wefjs01FvWSqxNkkyg7ctDv9bAdc1LeGdGWTp33wTv3/wzsjs
mofv4OY5yt3lAyxpudXe5sOVjeSVH+EN5nt5XnC9K6Ij51AHB2teqIqzVOa/
6cmp9n6WAjPS5E2o25DMcUwdXpdbG9wJ5wuT88ZR55dnGlS14ANdu471wFct
iFLXgJ8IjY/Qhm8P3j/+Bh7kbVA9ij7uLkzS0YUXnkzKegsfdWtSaUbfvavO
Z8yEHz37PFiE92wa3GdagDnpyGe6psLnheXz1+Bf3gmmJRGYK80/K15M+Nod
mv634FmiBqnpCobMnq+PmARvXKcWdhb1KZ+Z/Wsj5vZT7siJL/ivYsOIsA/1
LCs0tPsdcy6p7Fw1Bq+dfOjwY9R/T/teSwcbhvqtD7q+gc8RLg+ajz6mbpYP
jWCPdOiqsxx+hBdmFznC0ljRpPJi7F2WSJvXMNHf2CG5DPPgGfWX7kHsqWIq
xT2CJ0yODpjxgaXmOYyzH/Y6w0+2ohTemNS0uwbzVugay3NHDmiMdY9cga8O
7PQx/MxSCGVvEyA3tFSr1DPhO1vdB1owz3uHp6XvRM7oFfX0pcCP1uSaLx3H
f4dSe3PMGfIrua96AL7swMnxXuzLcu/A1qGFDKVXnFcPh4sFRQkOPI7W3daS
hMITpTyxEJ7T4XAyCfv4ImSA/9mMIavZd3wEcF9luUs99tct5/7Fa/D3W7Xz
CM5LfSY2VudIVtqyMhluZlIRbg0PuXzHMRL5YDD4VhwPP66kkz0fPmPqg5Qy
5Im/qXtdNjwofO2FmXCvaC7mNPInyzchVwFX1EuMNeBpaRmqUcixIpFU3x3v
TI7ui5nImWT1PVZt8KuHDW/WwLd5uUpG4bmsx3QB8jBV9qo/HHX4v3z+D4nQ
yDY=
      "], CompressedData["
1:eJx11Ps3lHkcB3CyXeyWtCpCEZWt1ZDuib4iu6XlsLRdmFSTcculDMm2Np3G
puWLDI/LoFyiy6HWdCE7colCMmOiUiRdGFu5zY5Yu+/5A/aH55znvM5znuf7
fD7vz2fxwWC3w1PU1NRscGngsnwXLcl+fKuqjVN8vSSbIRVcxY7uFkpCozSE
WfBFbFfzjXD3j2Wjt1spsVj8KOSC6nl/eamHkCFJ3RU5SVJKZt2TbsyBs3b3
r+jKYki95Vfb/WWULNtXvjcfrmwIUpvIZEg218lydTslTZ68rGL4K494pR5c
x9Zl12gHJZW7tJ1L4DZF/Zx1GQzRWigZy31GyXqpMFgEr5wdv809nSGOlXL9
dZ2UJJdqm96FP94aUXmUYcjyzG/Grr2gJO7PsaiHcL/t718tSWPIpRPh7tO6
KLkzL4YtgaubKc47pDIkUG54xK6bkhoHe5Pn8DptJeuwgCFxLROnXF5R8v2o
+KocbmHynlmQwpB+8a0f1/dQIuhVWCngm3mZUzefZ0hjBvuAHD6UG3REvfVW
FX/snPn+ZIYM2K8PjHqN+rwU3pwFT6xW+J9OYoiBmuKoDG7Go1oGcDP1hqqi
RIY4SAt5Y3ChTt3m5fDhb7OHH1GGRFctFQ3Bb2+pidwAnxF9yUaRwJDp3IxP
InheETfWDm6Ux/I6B5/kyHyd4a+jLxS7wXVTSGZZPEMSygY7S3FO65sWfC94
g6s4zQluvFtL6y/8b9Y/AT5+8J7C6pzXvzOkzUIrahT1CS10Uw+GK2vrffrg
22K78ttRT5len8fP8Aeujh9i4CcGUxoFLynpcN9kfxouV47PiYPv3BEUvAZ9
OTnZ3xsHr9mf/jEJLl49NLvkOSWLNJSD5+DPfDZYT8d59oQasnTQ97wHH6NT
4MXDZ2Zqw2vzCs6ykZM+4xtHUuH/tm4hfPjFXtlFwRNKTIUtexm49OtLxybg
H07urWhqo0QaaZOaAX8xYKI8ivo8UzTPGZGgzpNmjQJ4uP7x8F2os6vkLNcA
Offr1p2qes8KqdPnJviIfsLdrZiL4mm/jiTDHQbMrUrRr3lfLkyOaKbEP7ZR
FA9/+tnc+jz629rfwRU1UmLk4Tv1N7grJ6mFhzwsuKYr1nxISdd3Lp0n4Iap
un21yI+n1mBaWAPev/K4QZiqDq7lhwuRN88E9vDkfUqsTINGAuGOJDgyFvkc
sGUnCusoeTvpI+fAf8j3zucizyeLo1f51VKSdlVT30OVK+fIKWXI/8qMtfP3
1VDyr2N2ohM8807EhWTMi2VMuu2xakqCy0UaqvwMBDatCcF8CQqXmIvvIbc1
rddYqrwNx36+g3mcb3jWaxU8xMzXywQeMPypQoD5dbPZ5N5eRUl10BXNefBU
RUxoCPbA/VP22tfhZcaH2qbBC9I1bXdgbyy9rJoe+HthxiTu+O38cRG8zscu
bQQewI/gfYDLQrj18dg/HrNYB9n4bm8z56de+A1P3cS/4Z72B4fG4bIA7TNP
4c6BO7O9cxjypobFr8R/xXR55zfDxTXSlAa41Sd3IxHqwHk584t78DDrKXUW
uchJgtGEBHVzOTS9vBxuzVvrsBy+58ymzrmov8FlPckf8I4ZxsYmcF0bvS3h
9cihfsHaq/DwsAN2+vCYtzMLxtHHEu93hqq9d2XJop458O3NPU3p6Ltv7mDR
RbhkzxPTLJynbtXcdTubKHG2++VNLtyJf0NnGfxJorG31iNK/m8//we0ghro

      "], CompressedData["
1:eJx11FlQk1cUAGCCWhBi00LAshMERqDYqeA0wBCujRRwqCwq20DZZOsoFAQL
ItggI5hpFPmDCBIUtILIagm0KLKEQVqHFIgQaDUhrCWsIiYCgj15uk99uC/f
3Dlz7j0LLSrJP0ZdTU3NFc4OOIo/GO6ph1s7L8Sl95lM85A13Se06AGBLhp4
R8WCD8feaFGA582NfOsPfi6s93QgeD95jimc4aFPClglVuAP6FliBnjGRfuR
j//lIZegk24U8GjPJvcD4IZThN8UOHuMobMT/Dq/UJcKXhzxbsBCju8L4zjT
JPBbZz51Vshx/JZojV+WUGunkYf0cN88zkd9PUAwCG7AMPDesYjzV5vPn3kC
rgx5gUTgKdK/iifBc55ZKe+Dv9srFFUs8ZC8O8res4ZAsaV22QXgts+Z/KRl
HjqVt7jcCl6uQc1hgWuZhAwNgg/rpZAZDwk0ztRSJICLvhnzLV/hobdFg/t7
wGP6TWeDwe39YrU/gAttdG5H1BKobFxwzAt806uGGv6ah1a/M/tas45ApiRy
Cx08LNHKpAM8tXp1qgE84wul8X7wU21iS7NVHord+XNeYD2BvPQIthn4o/Zr
hlRwZa0kQKuBQHeMZEp91btSJym7wVH8slU7OK2h/p894AU5pie2IL7lSjUt
s5FAC5+Feu4Czw6767kKXlXz5Yp9E4H0Z9SFKj+hfZaTBU5+U3R9GLxHjo5/
BD61SR7SBK873Hkw6xGBRNb6oyqvcUjW58J7b3EbRZa/EmiPfv+0BviIVry2
OfhBwTWPLnALuk2RDvidpXQSE/4ztoAb6dFMIH5/XTMV3KGxKnAA6uLOSM3s
B3e3KU4zB1/Xf1q/BnU86lno4son0OV0p47PwTMlzi0G4CPpfworwMtK1g2d
wdtapaZogYecLgVrroBfcPqN4wvevJk68h76yt9RV8+hhUChYrfccFV8MsPX
CnyD5dgXDl5K0eAkgf9euW3rMwfvLRPQzoEn39SuzQe/os46ToF+3uplH2OB
u5IEPqXgb6/OMlxm8f2mgn27+eCjXxn9uDWN42fKhB594BujbSl20zifOHb7
zZfg+fQOYcgUzn+vW8b2Oji1YjvNfBK/13jeVU6BeQmiRMv8JvD/DOQGiEzA
3yvjI5dk+D+DKIyQQ+D5peEvaTL8/0c4V8a9wc17tANOjuN62d1dWAwFV+iy
i9akuL7Wj4/knlHth8GaXK4U94NgzM0wG9xgSf2soxT3T8r2q4ar4PzThREv
JLjfntZ3MYvBS3PVYqQS3J/3zweKb4M/L9BKkIPnMSX+fdDPSUcH31SDPyuf
NVZIcP+/Tuy93AQe6b/PnCTF88J59dMulettPPw7TYrnK4k1dKkRPMD6A1cu
xfPYGvKYpPLKIWOf8HE8v+zvaYp68GCRLEg0juddZsz1vgeefk+c0CjD+0Em
DqwsAa+KSjzPmcD7RCg8lHwDfO0HC8H2BN4/B+hdtmzwJ5rd3WOTeF/9337+
D9VI83A=
      "], CompressedData["
1:eJx11Ps/lncYB3ALlXpoy6R6vVoo5RA5lCJ0lZCMFhFiNdFUnkSoldN0e7Cv
Q1ILtXKYCuWsnhzm8DzUmkgplM1QOiCKlFD7PH/Afrh/eb/u131f1+e6vl9V
Tz8H72lSUlJmeKTxVAV7dRvGCWue+XgOXjVhNPOid0ZLLkeVKYvCNOBbrlnq
7jdllFnRmPgij6OwM6E98+EjZV5FM9YzOhb1zm/+NY7GPdVaePBz8s0rxcTo
ra2HYE8+R+nzTBxk4A2VU8UXNzAK/OCbcKuAI2uH2ZnvmbBG9KE3fKcFI4eF
CRYbizgKSYx8/ApeFKxmoGzJSCN5ol5UjHpe+vzVDZcTVym+hWtlWvc6lHJ0
S8O2tB1uFvTlQKMVo8b6hk1Pyzjq1FATNMNjv9qbe9ma0bYY7nDIDY6cT07y
G+Axsi0RkZvxPt+lVf4mR4VB0Ra1cO/fe+pC4Bbt24bPl3NUHufsXgl/LRJu
PQKPSrTJUq7kaFV916kbku/npE75wx/K91rEVnE0J0LMK4HP++gy5gu3j0g4
3vcHR8LI3OpiuLFpZEE06nEN21qrVcOR7tnkwXx4V3rIoRPwY0+047bVcrRb
FLOnQPJf945pF9GXuqt+p1sdvt9XpVEq8QOvDz7exEgv+KXvJhFHfHFHeBn8
8BupyTHk2Z/lViYn5sjUX2BaAV84knXccCMjb5G8/xX4Xu3so5J+zaznxh7C
vDZH89JV6jkySX205TbcjtmZ5ZszmpPWFngAfk+cmifJ85F5d+IA9iElYcyQ
wTX79iW3wZtanY9qr2O0UzyoGgk301wz0gV3N+58tt8YdWrV3toON7ppeXcA
3rx6v/sdI0a8Ei/Fj6jnSfwnlzH4D04q82avZhTc0SITAG8RZLhLYX9idIK2
f2vIyLZRLr8e/Ram/XRHAR56zif2mh6j51cmo4aRz6X2sPwFcCoKqB7SZTR9
V+FyKXjCQscF6vC7Vda/GOgwGh6U+dCH/IsfnclZBa/gd386rYU658TtulnN
kWBWfTXBb4QqCdo1GOXpXJrhizn67RsxsoEHG/q+n1zOyO7UY6tZmLucjF6P
E3xorpvahmWMnjoqnk6s4Kh1nWqWJ9yu6aRX1FLMxd5PUQF7FR134voBeO+I
yc+5Sxi1ZrYUhwg5iuP9WxsIdxzIPt2sxmhHqfevg9eRp77oVSj87Z8Ot0dV
GY0Zxf7oVCY5X+cmBXB+h3rPAnjf3/2y4hKOGpW/0ImD131ecWEx/E3OeJYB
zlGXR+isJHjAzuRPS+FKskmG5YUcWa3LaD8Dn7tHxl4L3tw0ZGuFcxp8d9Qx
De5dObxhJTw2c9CpH+e9KOxdpsSLVTwEX6PO8u8V5C9f5SjezcX/AvxFiX3F
fHiI6qR5BO6NzoCXk7/BkyeWaEzBhVni/mDcM+0NbWap8LzceHEt8lki9XlH
fA5H4Vunnp+Fy58Qrs1WZxT+vJF3+wpHD2zq9E7BZ87cLfJB/ssfiO014ToH
L40zeOma9hptTdxX+4YXFV3mSKlJaBgFVzYNUXmN+d5v5kt7wQN1p9+T5Cnt
kt5SsAI5aLvGWMGd70wbDYL75qRNHcL+dEZOONnBO07qp/Ale1Ure14f+5bW
1lV4DF6SnVG3W9JXypHV9w0YjWzM3vsQ/o3CuPEOeJLeekHSKtS/2fKIB+pU
mu2nbwd35UnbfIf9Px+opMRDv7pyixvN4eaaW5YNrWU0GhEx8Q/8/+7n/wCb
JeqU
      "], CompressedData["
1:eJx11Ps31GkcB3CVjApJnZjGVm4bOpJbTCVPZGNDjURKSzlqcz+5bhokuURP
sltfxVZyqiVCLuOSGYxJI9JNiHA6uqx7Ie3qsu/5A/aH7y+vH57v87yf9+fR
OhTi6jdXTk7OGt88fF5nJtKuDQnqE5hHU5EHGLI+jS3nWEaJxP9m3ll4cLaL
r9MvDNmk9+JVUDklPjYeX+Lhy3Im/imDz1SesM2voMTLa7NiOFwY5mTP8WZI
W8+LgKlKStgZyTcPw7eucNFLgOu0z3LcqyhJ78zT8YRLZq4OvodHHe0/K6ym
JEWzdp0rPMPti/pruEtvTINRLSUvbOK4O+DvQlk7euBmJ4qmcu5Rovh5Dncb
vNxRLew5fMxU/1cVIdYXDPtaw5uGUyRtcOcrR7siRZRcLUi3lvn7M08KPeFC
5W0O/fWU1PlpJMq8Lav12xucy+pn8XqTRpxrcHZa5inGPjHH4JHm2suPiynR
8mgc2gI3zFsa/R25BeYPfLzVRMnXHzlvfoJHT6zyXgWvqXomFUsoseoaKXOA
D9/xMinyYsiph9uzmu5T0qteKuXB528ZNXq0nyES3ljvjWZKLj5dEr0fXjLV
LRrfx5B+sy7myANKDFlyHX6yHBRivyyBGxhWbp8vpSRUVCwKhddJ68XmngxR
fZx75zTcq/Ol6im47gPu8yEPhtgrDkn74TvZbufOwcM/j+kthudXp3UqtVBS
r5eXnQMP9V0x19ydISX3duVowvMPB3JL4V0sY96IG0OWJ95X+4p1qsovd4vg
MVnOAjW4NMk1thpO2RH67XCnnYXN3N0M6ZsZ6+PBs0Ptat/AIwqmv03y0Lfv
ExUtONfGawv4n+AaiZnRmvCFMaZxy+GCnNhO1rCgPvm9KMl+F0M8+UKhLfKJ
uNL3OxuuxVGeCt6J//ZMC/Yhz9thId56cPH67kQFuFumg58T8pePt7G1gq+c
DlQ1cWFIrHpk2hrc16Let++2wXMdLSKkzgwR1wQbjuDe3eMl3Ty45NY83UPw
ungr+dwG9NCi1Mgb/ueIhumsE0N2GBxktqI/F3I/FAbCW9O1/f+AL7rtuaYb
PTRQmC4KgzdMXt+cDQ/irSg/UkeJjWFl3nE407dW/jrctFjn5Gf0PFdf99JJ
+OuHvg//gm+3bHROraHEoi0qKAV+2ezAsWK4S5xG72rMy9KYB4zMm19+TNXB
PnMHWP5VAkrMeEFuMh+Qm9zAwJN7amIPYx7r1N6OJ8vWT+wXL0QOTfGb9nAw
vwu4mh+T4NyD5bV8eGI0v2Ic70CMirZiHNyCv3vwAPIs7Xh06cldStaqWhrz
4cLLWqzHcJbyfMVnpZSsNDz6Qzh8LZutWoz7WmD/m+hDCSWtIRy+vywftWh6
Fverrr+p3gj+/FCZtg98kaDAI8AV70mGrXJGMSX3il5z98BJh+VFB/RHs+Fu
3mJ4XCp32BYuf8PkXCb6xg+OT6q4Q0lmRkDQBnjjNSW74D0MudA7GJUCb1nn
424At00PP+iIPocXJUweh9s0v2pcBrcJ+aRzHv1vH5xjnQXv6HmmpQAftXja
4L+XIRVtfz/tgZsrtd+X9TOUsArsMF+P7cwr7LGfJ/++NR6Azz3xPTwV81g6
MjrQCleqOhUg6//5hIlZH8y1sasu5yTO+0qpTEMIv92YMGOJd+C0SskVV+Sm
rGawtxDekpqVrIJ3o/OCe/5G5Px/7/N/5ZQBFg==
      "], CompressedData["
1:eJx11Hs0lHkYB3BnXU7kspWsIu3kVuZ0to5tFjH97AxmMsYYFsklmbBYCu1R
VrMumUperbX2ZWmNykGybLMzWx1Fq6VkRAyT+zXslAxiD9Y++9fvr/3j/edz
3vO+v9/zfJ+HcjKBf+oDDQ0NF3g04ZnbKIkx6Zc1LgZ7Ja4fJZFkvHrHlV4C
dX9JsjTB1d5uZe4cEqnMQjY+Ae8q516deyVrfN7DPJjnRaLCOhS/pCDQM5tb
nv3gZEH64V4uieKGN032gJu/FgrawSPEIVn13ti7dR/MycCTTH/2W+Hh71hU
Fb8Xg7u86y+l8/F/96U6R+SAd/vXsLN88Tl7/duMk8Fzk1NoT/1IZBxmJtTr
I5D1LyyvCPCJD8fyyr4gkVGBfqwYvJm9fb8n+I4cs4oJfxKFzctLOEoCDb0v
jnYEDz2gXLsfQKL5X0sDPnpFoOiJj2m24PXW2uPXAkm0ZUBOroBn1yVJjcEP
FRqejzxGIhMi4vpMP4HOxAY/2gSuar4plIPbbGfazQ8QSBTvnb2mlDVWjta7
0oJItGHZ7qgzRCCygUp5B95LFSlKwa8513haDRPo69Xh4nHwErUnW/s4iayX
aLvZIwRaIE/NDoJnmGqd34D3Bd9FS5JHCVSvKXHsA+8Yyaz+G9xUymeXjREo
5pEirQu8UPhTxwK4SJO0kI8TaGDzD2+egx9rcaK8BQ8K1J9XTxDIUatNKAe3
LZEKU8EpGtJGwykCERdI2xfgI4WJi5vAa9mDhjavof7rTruU4H7GkvJ9cN/K
b5sCqNMEat1980I/uLo4skcGdaPebQ8ynyHQ4OnSPZPg5fYuNCXU+YiNV9E6
uEhcYj8HLv6tUmcV+vJwa4xP6yyBtGi6rRpQz2kzsUcd9DHDkN909i8CcS7L
hzaDm7ffs3wJfVctcGP0VASqMPmRYwGeEKnHuAw58eusaL8CflFkL7QDf5Gg
73oHcvVPYNzUBHi0ruWndPAFXmitwIdE6Q4UGeUNgVoei9L54Bm6DjJzyGd0
FHWDDv7ns7jgKPC+ZT9eDxd7WuHL5m/AZ5yunyO88HcUeyVP8sFPP7alszj4
v/snq8LLwWtUn0sTPPE5463+qLoHPpwja7E7iu/ly0ny7fxvLlZcnKdYuA5G
iQczx8CzUrq4Rixct6zbarQIfsajdt7BA9e5zrQ1VgfmetKArhnhjvvSbPVk
zBTcXL4UnuuG+6ha7nhABacpt1n9zsR97+RcWv0MfOyrGwIeE+fEoHKYxwSX
Wq3fn2bgXIl3PuzkgXemRbmkM3AOb7OWc/3BD3XXbBUxcG7LjLWOBIML1goG
rjJwzsOrlXPh4PR1ZkU+A89Fg8wn7wT4kjP/rDYTz1GoZ5FBGLjrWteeFCae
uwMnmsJCwM+pb92YZeI5bdk2m3AcnCsSjQa74bl2+V5A4YNf0rujOOyO90Br
X+lFFni+f27aTg+8N6q1nqbSwU+6N0hWPPCeKdKn9tmDp/JqvBUsvJfe7srs
3Avu26adfZeN99j/7ed/AdroB2w=
      "], CompressedData["
1:eJx11P03lGkYB/Ah2XJSOqX1EmuiwmwvihoT63ZMzKzqbFN0rJdKO5WwVlSa
immFlOrxaJPBMESL0i51shUHI02lLWtT0SCSikneJkL2+/wB+8Pzy+eH+7nv
6/peFzs4QiTWZbFYrvim4bN7PqQRmFdUL9Vx2WBfSJO9Z29pdmgp0r6mwH4d
fIm0fS4Nj1dx65Z+osiGgyUWy+BWBavkY3C+zsdtI/DVVTvbLeHrPQ30dl6m
yXPqWsi9UYoMWA1dMYbrzLB84wf3HJJxc8cocnPmHMksePmr5nsiuDpz9VeH
P1Pk7KiUNw2uYSlzvOE1PbNGROMUMRX9qf/ZrKI6gL4r5cPj86ZqORMUeRxd
aj4AN71vIXOFHwxb2TQJ9/netlUDn7sip0SFe6pk8/VVkxR5J+qv6oY7Fy0S
1zK+aWoi8QtFXtFPIrvgAq/w1Zvgri/Hb/KmKLJrwTabHvjbykiXsAKaTOQ/
qWiDr7i+vp85x+GywevRSzQ5arc9O4aVSng6/FLGTw/kCRLh3nYSgwn4AklW
WC+8+m5vT0E+TezG6coInVRi0/HAchA+PW4+vz6PJlHlorEG+DHJR7dhuPC2
1XEf+O9Oei4muqkkVcltmYBfU+ttO6Sgycb3K5IF8CPZ6Wv1UTeDrCfKjFya
FFv4q3fDn20eO2kEF0fwNHdyaOKbKbz1M/xgoteUOTw7oX+0Q04Ta+NKhT/8
7YUyri28bJyfOR2+74sDtQZ+vVY33QluOK3wG042TTrrbJK1uKexe3CFB3zI
ZR9vcxZNWpSGyZfgEbzXL0TwFI9cs5hMmggF8wZ58PjJmDPBcD/FutAcGU2s
ml07lKhP0B6//nDmntkzHznBFV1ha53g3Ek/9zh467elAUEZNDmw32Q4HfVn
qz4ZpTB5S5r9fOQiTUTsD+k9TB85B+wy4KyZQafPwC9K2LEr4bGco32F8MC4
vprF8FNHTItDkAduuNfodfgzb+HKqnSa+Cf4m2QhP9O9gx2VcM+480pfeFOa
/owHyOHAKsHeBrjCbsvLQLjvqQ/OA8itPEHR1wJfXvxPofYCTWTV8aEm8JDO
Rs5r5r2hZqpJ+IkGh2PfIf/RylDDt/C2K8+WReGcYSt2qRjz8ka7UfuR8ajK
WRJ4zX7XrDTMV1TS7uQx+HDqkSIp473We+5jTvnD5psn4LbhR9eMwJstcsIN
4GfOh8u/wDOW/iQOwXu9A29d8x2hSH5cMWEtRL+0WUZqOOG3lpQNY76a3KyZ
c2LSam9Uoc6/CBfWmcC/VvN2MR6q8PrBAX05q6u/9bch1M0suFALl17NvzEX
/Y1+ZyZnw2vsD3mNwCVtkUbx8J62we11g/hvuUmTBu4htrQORk6qEp3ND8NV
jU+du+EDvV1bCHK17o8bDQJ4wEKOuhV+0vm9xgI57HIsPWQPf+ghTmmE21jF
PvwMvz0vUbYIHrL17wX1zF5a3lXfjJyzHOXlTvDZLU/Vf8Ff9LtzyjAXO+SO
9T/Cwzr1pFfg3YJVV1MwRy0Oeo9l8BdP64Lk8B1u7em7MXdpnZ6NffBXgWzX
c/COwHl33DC/QoP3+3zx3uOUi/oEvHCx8aJieJL2XNwj+BJ5km00U7f2ccNf
sQfmiGM5AainT9DNf5n8m+Zuj/XDPpH1fYgcgz+SWuj7woUJ3Tq1cG7Y8sYi
9Ov/9vN/fLLjag==
      "], CompressedData["
1:eJx11GtUzHkYB/BOVLsnxiW3kEvNikzt6KKLqX4pWVQoFV3MFirTfS6ayWZE
Z6Wyqd/6c4jNaSm1mDmlNmmoiDVFEbXl0naR+1hEima/88qrffF/83nxO8//
eb7PMz8y0X+bvp6eniu+MfjcNoypLFJXXVGlm8Z4yxjSEXl0pNueEtbrmmlH
4ZV2p9a7wsf15wWVOlCyrtn6dS7c1zar2g7elfqse9dSShRnh4Pl8PSWUC0D
j/J/IghwpKS0tPCUGL7w3NO+HLjpLC7bzomSBu6rp0nwe/e2xE+Dc4uEZ42c
KdmYctwhCr7Gj+c/Di5eMbRtAL5Q74HeFvj7P86HMlKGNKwbUClcKIkNylPz
dXXuH906D/6hI2N66jJKIh/emBMOv9OSqylNYUhiDiFOPEpEffnDIfBnnNgR
B/g1V0vnF/CX6tHNm+A3tx3OrNvBEHutlz51pUSmkHro3nf06Xb5R8KQ1IG5
cWw3SkxSAojOp36x9kmAm7P8TpfCtb5ja3X1r9yZtidXzJC01o3pM90pOaxy
rE+A87T6LkoRQ9Sfmjhy+O46z0gpPJvKJW1ChhRfrl5VD08JnDJlL7xieUr5
UDJDWvqKjTRwF+Msv1/gLD2p0Axe1TWdmUgoSd44YnEMPmMZ+4Jn0ldf47bc
+AzcbP/oJ0EiQ140mTsO452IdKNJSvjB8rYT2gSGWAcoSzrgaXdtZtfB32re
DyyED+Zfzz4EF3TNdL4Dj4jpsg2IZ8hMi/YJPHju44JHvXC1GXs0LY4hf0UE
bm1HH0IaRJw38MTZXmXT4DZZVZvCdH0rkA2Oaaq6MsnjvMwzliHa0Zeam+jz
IqOx0ZPhTVwbq0cChvw8p2eBBXwtY90xD+7lN7ZTCm8+diJ8J+bl0jAhlAvP
SwvLngL3p5/tazH3xX9nRBH48Z7PlxTbGZLrmPrgHXLi3nr7m9Xw/m7V7Wr4
oeDTe63ge7L4BSHwkI/a3UHw38d3DwcibzH5dxu3wkvWNJv8CDcYUiWnI7e8
W42yRHjm0e8sBfDeyCN/nkTOl87ZUCSDV5aYVovgHcOXbJqxF2qnYy/T4Cvr
k849hx/wvSN+hT2yPBuRsw/Ov3xX9Q5ONYOmbPiqM/31WXDXD4PK7fgvwwU3
VCHYx0wXVtIB+N7hyezHcLGxMS2wo2Tlv2VW++HLg1cNXUE/24Vq5pktJX4R
nGu6d6iTLMYB/feLn3rTA25cW+ObDZd3tt8vgR9PqAtTLKHE9qfk3gz4rGbf
4kzM18nSZK4VPNl5NEpX/w8H2iXRyEOPzxNykUtJoSk3QgIvHPFpWIFcmXlJ
8gPh106aLxbC3f1Xl12Am78ViSbDN1s/lEbDg6lGlId8WlbOqOj/npIOHt81
DJ7CCqqJQ56jTuexr8IVLK/otfAl1w/KvYVf3dLNU+4BJ+YCj7nYI7m3R0Yn
fLVwIssOvi62Nv0jvKyt0eQz3J7L07eAf/hV7HYL+9i5SG3rjnqClcl9k3T1
N4UPFWF/Y3LilUfghziKM4ZwXsXFteNwBwSGrfwvcP745jfvkGfvF58yuuFt
DrXxUvSnJU6U/0R3B6723vfGPfHfN7/rW/R5n4H4Qiv8TW2jgSnuEnvL82Il
XMtReTXo7hhHE7ALLqlJFW/CHFsMDOvL4RK+wW89cOsd6wunY+7/d5//A2aU
4JE=
      "], CompressedData["
1:eJx11PtXzGkcB/A0XVyTlFnZbqKSdNli5DYPEZpRUWaaSzUVZYxLYqVGZzmm
dlc8FN9W0bh11E5JsUtIynZ0kZR2OLksUzguIbNKdbrse/6A/eH55fXDcz7P
5/P+PE6x29dtNDYyMlqMw8KReKZ6qidUVI+8ZnETpjOksXT+q4JwSvITF0iz
4LpSB+EDeH1TbNIEASVeR2zsVPAOq1UTOc4MiZhv9OshISXcyx8rd8P7x7tp
1HC/1S0xViJKrOVbFm2HP0pdaXUCzn15cfFZMSWRA315CXC1LCkyC17oXvZ0
npSSghyH/mj4N01A0UH4lzs+KU2RlCz13ZkcAd/pL52ogj9vZsIk0ZQEmGod
18L1HbxTaXBvsYm7TkbJItWGKaHwe/xHgjHwfdsbU0WxlKhH0oUh8Bup48MZ
vKvZjG16J44SQXi4t8En/13/yAn+8E238biNlDSN1a82+FBSeXapE0PyZ4Qo
A+MpiXXWThPAPQ+0Pr3vyJD4Is3ryARKqhR2YiHcPkSQK4ErjsZkSDdRkpar
CDW8S9bS3al0YMik5UumcuSU+PUkum+CF0bd7z1lz5CBKEtxB9xihSxjB7y9
sEdZZYd3/Xz8aNJmSvY/cbyrhM/qCWp4+T1DNO4eP7bBgzKlVzLgrDLWJRY8
Rj6db6mgJLhmjyob7pxbJ3WdxhABb1GrHfyBls06DS/ObTzHs2WIpOrmKnN4
+DSRsgTePPf2X4lTGZK2RtNVifuzlRGZN+BBTKk85zuGdBv/bhIGL/q4Td4A
/6Xdq7aSjfrrt/CrUP95+eWx7Yb7Z+Y1dE5BPdwak1HwgRyh6J2hznOV8ePg
SkHQJQf0Z2906PoBw3sTk0/62jDkce9Xni362c4xrzG1qKhusa8KbLRmyLXm
XvtBzOWwW6SODR/2G0r9dzJDXAq6O69voMSljBM4A768dbRRJlytONyRgPna
cdeyfeDl1SWd0+Htsa4RIzGUiMadDF8C39+nnXnTCvNNKV6+D/mJOrzi6gp4
0UFOfg1cyCVz+qIoqT6r8w2DW98OYYvh763zjymQz8yAz1oxvHXB17sy+JFb
gz1aCSX8rpzkWPhQHHd3Arzkz6dCgvxXH+L1yeGmny6O3gbveP1CWR6B/Et/
C1TAVXWr0jvgp/n6c7OxX7u8JomT4LWy2RVvDXXe0wVfWU/JiW/ptjvhXZff
V0jxroKV6uog7O/plONrDC7aYL6kBW4eejRtZB0lO3z85myFX39mk1WOfpot
SzSqXUtJuWSeZAvc0WXEZwb673th6cPSUEpizIxsNsJfSLgqE8xrvKzNoyyE
km5Oulsc/IyfsedueLsol9EGY9953hJDH4Zz9LPWIw/iV8TSDS4/m9sTCmez
1Wd8kR+dJomdtwb78qHhaiDcP7A/YhLydmrwvLc/fEDn0LgQrvzSu+cTXOK1
LGKQj7kf4Jl4w/f9ceBJI3L7JvlzVhf8Of9BujM8hWWRdwE5Vz77ZsvCPQ9d
tcZT4Lda9Lv2Yy8cRxdfD4FbzBQNmcOLb8x5LsF+Zdz0iLsGH1XZNrkfOcxX
W6bNxT7Wrc6ODkb94pL4vYbcmn0Yo7LA/i5M4W0egz54+nt7GHL+Q0Hy2zfw
pnjP2vdwvcLJuR4e/Y6jqcI/0PLYZase/fzJ5cW1SvihANWwK/6TQvUTtSXm
4pzVeUwD92kzrxuGX/18/J91YZT83//8Hw7rnkw=
      "], CompressedData["
1:eJx11FlUU1cUBuAQiVZAXVLECYdE0CI4g8Ua9ESlBWpEoEmlCqIMVRGtIgUl
TIFAAsihkouslIoSyyo4IIMyqbBAZHJEUKuxiMxaoIo0QK30zxNPfbgv37rr
nnP3/vfm7j3s5sdmsVj2eCbg+aaI8zBHr6RSdbuxdDCJIXnh+34yjaaEHFrp
r4Z/x56k7YPXCL227JZSYiMJPJ8Gn3kksE10kiG/6nsNX4mh5Jy1oVEK/OGr
80Mu8NC/WRfZMkoezmi8SeHrtB5uGnzHTio4I4yj5L2t+isFvMhVFvsYPtr3
WUp6PCXRyurFOud+EjjZB164YnfUMzklyj7zH3X+y0K3wYFEhrzOfP0nJ4GS
xU7DXTqffjnoHwk8Ubg8fWkiJcWLFtYlwCWHVfsM4KvnvKywS6Ikc45s0yl4
vrel39IEhoS1TvVdeZIS8aH31ip4S3b6bWcFQ/q/qIudnkyJg6djThbc+o3d
hQA5Q568m7+nBT5TXs5c0L3vfeV9UjxDOktml0kpJe49GwRX4R3Ly3svxzHE
Kj/y7uQUSurSdj2qgWs9Kgt9ZAyZG+H58wH4vLbBSw/goac1JXGxDNmr/NBZ
BF8h2OilgUe+OB2eG8OQjhrJgWZ4nzCuqR8+V2w87C0dd3eWJ4fNLqmUW9nP
mxXNkD2pHIvL8K68KvU0OKuNZthHjZ/7xJ23aD6c78Dv9IlkyNez7A1Hcf81
U83t1sDrmwtLzSIYYujboj0Kn6SOKt0EN37Q1bM5nCFe2yzO3kMd9H36R9zg
58aeVgVIGHKzsomtDxd5SHJ94MPJTSxlGEPWvjF9xUOdNTXcxmNwA72KrOsn
GBJ3sNPeCn1xSb3iGwVfpX2S6w6vuKsxMUEf+4uz/E7BP+aUmIQfZ0htVer+
AQUle9qzZGfgW9IUShO4ZTaLfwM5SRB7NOXCK1rN+BdDGXLn0m0jKXLFC25w
yoebvIvQL4DfXfX7nM+Rw4CGiIk34dVjz6c5w1d8TPi+M5aSZXXLMm/BbXc+
HXKBD+q9LExFzn8ImvpO52NXM9Kr4S+S2ysdMRcvAqw9G+HbHEdE9XCzsu4p
LMyRxZdnT+rcydD1GR/39Drr+Lw8kpL52r8EOh/1/GMgD34gcf3BlAhKOj61
bG6A93ubd/NQh2PF2hsB4ZQ4+Z8aqocbtRUfZ+DiJZxUTwklI6q86Cr4ifUN
0mDUuTz/6Bu/MEqq2wozyuDWAs4SEfoyNiCsjT1BiVF7v3kBXJVm62iDPv6m
KOuuPE5J2tvulTnwa5uCR43R9/ALhndmwCO3B8kz4UlHDhq+hX87suBacigl
W8OKOBR+9KKriI9cObhMyLaCWwqOWcfCC6YUi2Yjh6eVPR5tIZRkXe8+Egrf
LvZPGoIL1bwZt+BXH3341w/O1WydvTFm3DsaeS26XK2rMdCXYy66Qthze+Dr
0txsHeCNrY/i98rGz3W+YVu0Fi5eEC/gY+5k913tMuBbclepuPCYaLNDMZjT
1RP52xbhvwqK2AbGurnIFdrswlzbti7vrYeHxPvX6ME1L/132GAPpA8/VihR
t90ba6e9xdw9Zu6bGmFvbH46fCkYdR6cZ9bbCldvCDrcDu/lykWB6Iuvwb0d
zfACS4vArdg/KqIQSNFH8T7L7VXw9b3CUR72287S2oFs9H1BsZp7DR4XGrI/
EW7qbOf9HDn5v/38HxihyUo=
      "], CompressedData["
1:eJx11GswnGcUB/AtismghNUKErdgqsEqRRL2SSXrFi26pVS2NrWJWzErFx2r
aknqPuRdFWHjthd26UpdVl3CikgjbjVddEpDQzANzcpMMGh79tPzqR/eL795
55nnOed/jvXF1HCWBolE8oFPE777gQPN5TT54LQibExHyke1NvSC7joCVb31
CSMXPFj+2HURvFf0l+hb8D9ymYevgpfWjvO6W/gocYAzHACee4zmxAL/R0vx
orSVj6Jc1xLcwN95rvI5Dx6XUnR48gc+8rrJP+0Ansn+t+ckuDaFcaFRhv8X
TFEpVuDx4yZXl9rw+f0vKwv0wFlunZrt9/B9OkvZSzvn5IPckuzyvB/x/c+d
1nZYBtf11VfQ2/mIpPPhiHY9gX76LiB6Ary7zPWWbQcfZRdKLiSBj9wga3WD
V3lw2lXgVEeqze/gSsSyqwc/NaG0G+jkI0unHeOIBgJ9PfXxvXLw1q7Zfqsu
PtpW6sSugu8Vb/hwwZ2TKRFccE3TF9KMRgJdMThQJIDXSxblIXI+ksnn7MgC
AolMQnqZ4Nfo9hl+4JLNodE+8DHdbfcw8GPSm22N4Cnh6WlxQgKNq3oEAeCl
tgeDt8GPzBv5vS0i0CFHOc8TvG1PL3AHPPP7FPIoeJPqs1y1m3MGFrngH2QX
rHLEBJqR+We6gL/ON6vNAE96SSu2ayKQP/Xsltr/1khY0QT/Mm/UdQxcpS9e
dwBfmPdq3IV3OXubKNObCZRWdjyeAk6/mGNlDi7vMT5vKCFQUKHxZTdwgSyU
LIa6RU8uDrWAx1nsXHcHP+T1Xrw7OO1K/FM3KYGyNoLKqOBn7HW3IqH+ZN/h
iVbw2QZHSjC47Xbduxzoo55kv8yyhUDThdXRkeAxwldG9dB3xzQzey747cnl
ijjwxGXjyEeQk/DL1swp8OVnv0Wx1X1sexq0Cbk6QTmaatxKIDPdfpcc8KIQ
Zo0puOGqqS0CL2cRvCrwIe/NX5SQ24ouo6wIcI9nKi0J+CyL5kICz+o1kDDB
Sac6NIfVeWiQpypa8P+M2TsGM+CvAm7MbUjx+eYPcpzWwT2U+XfNpfg+QTpx
0QfghcXViiAJvv9gN73kCOR/aPSxqWEzfm/Jz3URruDKcZ+zZ5pwfaqC5Sd8
wZ27LGsWxLievyrydT8F51l8ftxQjOvfn7mVlAw+JbeJ8RPhfi2EPLrEAQ99
OJf8pxD3dyXdn1kM7s6Y5eQIcR46+3nLNeBhHSMnrYU4P/T4pA0B+PwS+5qT
EOetaN8srRM8+e7R958IcD5fF37zxn1w2aUiixkBznOfbcWtEfCGFd6bSwKc
f/a+Z7Pa8xeivooQ4nlZj5a0TYBXUpE4VojnS2lR6qP23THDWKUQzyOZrH9m
HLznC5fEQBGe39FG994n4B89FzzsE+F5r+Z69yjAV9d2a++I8X6QTYtFXeDc
0D2L6014nzA8x/LawWmVOclrTXj/GLJjGGJwe22N4gfNeF/9337+D1kW3rY=

      "], CompressedData["
1:eJx11FlUU1cUBmBCNRFkVGMgcTEogojzBNpQji6ktiYCIQgIgrJiy1RdYLGA
ChUleouI5TLUIRRdUYKgGAlGZQhoZDDYgpCFUGW1BEUKDjwgoCD9efKpD+fl
ezj3nr3/vR0jD4j2GRsZGXnifIFz5JLTJPPLO3WGg39ufxFCk8sZvgPqCoo4
vNfFjW26U9d08uXGul00aeOs3USrKOLVaqUxwE1HfbJMwmjiNjmszqmkyNIy
gWcHXKbS3zXA+VmFbYrb8OqAQC28iheRW7ObJr0uH5s71RS5Js+2VcGjjc+6
FYTTxH17GI93lyJj+XrTEnjwFjsZI4ImkvvpsxLvUeS5Uir4DZ4wxNd1w91P
ZRs9raLIubTDf5+G196+Ouy9hyauFWxqWw1FPvxSH3MMbmvBtSqHV561EFfV
UsTfy7Y3Ec68scredi9Neiy95WvqKMJvSI6NgRviDvOPwy/JpPuv1FPEfUvK
KQl8XDwhTYW/ZMlH2A8o4jZL/iwcvs1PNpkMfzi/bDBJS5EqLokPgQe26YoT
4ayOHvOWhxThHAr2EcP7O7Iz4uF0t0+YZSNFephRxoHwrSvb2j/hP6+X5lrw
myjSqI223gm34xZITsOvdsm+FzdTJG+m8lIQfJ3JQBgXPv+91Fb4iCL2jYKK
6e+W3ddVK1CfzrwIJ2cdRVL9ntTshXu7vrjQhDpnFj4w74Pv6C/qm37XAp2y
Kwhu2jJ3a3ILRdqVNq774XP22aiS0K8cxYDvv/DRyPWMpOk+LiuccR79fSoU
rnB/TJFOZXpaOjzFaEZ3VShNGuapd4XCrdsFXBr+xw+T3x7Y9dnFJtq+QviY
sto1B3nLvLtm0Qo4x9jBpRSur3aNqgymSVpk3kkDvjtz3OFaDVxQWhceE4T8
rItRxcIXD5o76+D83g0FZ3bSRCH139SBdy0S9Vf3wLXDKdHegTSxHblF28Bb
uzcIX8NLCoxSo8U0UcqsfDxRt8gVzPgJ+DzWACs7gCaagy06T9S503JqfDbm
otpyVa9KRJPB+ZqExehL9ilaz4M3OE1NdvvTJL45rm6sAfdPjJktgxd/LY9k
wGPeTcVXou++yyNL+HAWI4azxI8mrdZF3iHICds9I0gIj3r/omuHL02KXUJ/
HbtPkUHdoeZg+OU1Iaq+HTRxSYttzEAOS2vVqyXwuSV/SVPgR54NebGR220d
qbUH4PXmXoes4MX/GMouIOernqiKEuFHA1Y7s+FFZjed3DAXiwNrWcfg37BW
jl4T0sRBs/DiPczRiWCDUgof6AhNKYeHci3lQZg7kYTDOw1PFj5uroCHTQnS
ZmBOyxkbazPh+Y4ntc64f4lZ//J6zHVb/mr6LNyu/4NsKTzibcb6HOyBhcP2
R7PhI7H7Z8rgHL6edwR748KWpjdZcDPhBnNL1CGhJv2749gzrj95vJq+3/Pt
zsqf4Xd6L7IV2Es3dT92H4cHf/TT70E9mdaMuKFbFMm1O+FxGJ6klaQR1D+r
JeytGK7y8BiPh4verSyzRx/LRzJ7DUrkkHl+QdR0f7mK0E/wtfYFW8/Bl6lf
le6Ga7p4ec+QhwD1LX0CnLr+RiaCO75y9L8n/uzhckanD3ypPi+5AHnz/N3U
ogj+suIRcz28PXecY0A+bZa3vh6CO5/5ymIJ/GK+SKBBnjePznkgwX9euaGZ
zZ7OSfTzERPMhYWi0msC/n/7+T+VSdBy
      "], CompressedData["
1:eJx11Ps3lWkUB3BxrFFTVqRJFB1SDZKUsShsaVFHHaGOKHRzCY1hhYxrGVH0
PjRSCAnROa6FjmuHrusshYkYWi7TFJOSVaOppst8/QHzw/vL54fn3c937/3w
D4S4+ioqKChY41PCl+qs/vvn7VLZojM8jcmbjF7vmWgxFHF0KtUj/A1cr7so
+ZaMUUReqdOv7hwdcWwVTsDFYiXTRnjwC8OCbz04CgqW7hmG6/5TOMa1Maox
PxmZ4cmRs1bJWB9c+48kwyR4mJpT+uK9HIkc7by6ZlxutyMaHrzU3fCKF0er
HE/cuw8vWbHSPxQ+vfQez9yHo+y8zV5tcE/xn/X+cFcPu8HGfRwVqDzOa4X3
fRzoiEc9GjWSScsDHA24p7fdgHe77X4cAdcznRV1+SBH/lnvq+vgQ/Gx3Qz3
laTdjf18iKMPxpLjUvixFz8MylsY5S13jnbw48g2ujZy5pxdNidNnjUz0s94
GBjkz5G5ju1gPbxi0EpNAa5QEdcdFcBR7GCpqAkeprRaYN3IaO/5oZeBhzmq
zjBLk8F1HJoioqWMVG9lNZgHcrRWPBl9F76+piursZ5RW1iDxQCc8ie0HsD7
C77kfqxltO2LSZlnEEeplqZHe+CbrRIyN1zHOROr5LVwMa+57Ak8xXa9JLaG
0ZI4/cFR+FBc7vln8OLcv2tlVYwC3avyxuEv2+Si1/CRtfSQV8ko1k8r/TZc
7/CmTx/gmU9PLxCUM3JRrzQNhSdNqZsqC6Wyj/Pmq6eLGR26bnJSAW4ZzrdV
g3eqDKzuK2OkPJQvP4D6s5ODVHTgngFNG3RLGT3qjJHlI4fhmLEEI3iysvHq
gBJGirkZZfXILas5aK45fMVVVWF6MaOJd/G8KuS8PSa03B6+ZPJT68BlRgXz
5i88jb44q3avE8IHntWt4MGfJrZfFPpy9GRO+lYP+LGt9o4mhfiv2lT/NPoe
ImmVHIJPacjluy8xGtG+JjyDOVkT5Or508y9jNfXJBYwWuD072eN/Rz1GzsK
YuA+KctfVuYzGmcXVzLMod10WG0iPLLgSIYUbvT87o7Z3hwllOoKUuE/hgtD
2uFd3h/Cf8acJ0kDhs/CqbpiWwdc6FuVM4G96MgI25ID33gxdqoXXpYZ0u6L
PeJU78/Jgw8rR/a2oJ7hUZ/y59i78+M8nwL4On+pUQnqX7qsSScOe2pWrulX
CJ8Wq+mk4b4Xmv8SGeziiOUoGhXBx2zkjWHIh++hoTHgxlHEaYF5Mdw/ZZ7v
7iJG7mOay+pcOcofiXa6BJ/vZv1KG33xHn9Qe8WFo9bKBR4z7hWV4KxwhZHj
zjeZdTs4GjlbbJALz35UbdSO/nb4mdx+4cxRWcpRi3PwQAOThYlXGc3elbnJ
AZ5mlJjNwW936GvZSxgFRGzrvCPkqEd/jJLh7L2ZsWIFo16rLZIguHJG59x4
eFJE8DIZ5pPXbJ9tAzfzXKwTAdcVHOPHViP/BvbYAv7duVduwTPnn/XkWV5j
9FvJiIE3fF/Xxob98PDRlX1vsS93vjo5VMFdfGW/iGbmR3HRaEUdo24PwVtd
1PnQ6uZRAVx1i32H3w1GoQctrrXA1wi+EVvDN23efkunAX7KRXYcOWydG7/G
FP7pvkNPL/ad/67wyWHk9n3HVy09OL8oXTsN78OFE05JochZkW8zS3Omjzlt
tgfxzsQdEUVloS8JPVdfq8D1Wg02aOJd6k9Qete9E/v+P+/zf0Hfz2Q=
      "], CompressedData["
1:eJx11Gs01GkcB/DJoMgpl3TDMDSTyzCVtdHN0+SeTm6dpRIpXZxdLJHL0koh
Uj07dLEh0U3Z5DIzMjTUWCtp19K2xrAMOlFbqO2y52S/86pX++L/5nP+57l8
f7/fw46ICYjUYDAY6/Ex8QXKh6+aLJPInMc6u87ZU8IPMVRpiSlpCOokBnC2
9eR2DQdKgrdf8MmA+zPXTDLgrBCHij64fNUY20hCyXyFdt5bK4msJCM12Y1P
yVXl3A8VcIH4QcVz+Hf24dNVcPbEm9mCBkqMXi1vVsI7d7V+vWgFJVXHqxcO
wpd51W58Al+ls26xIXxvpmVo9l1KUum7B13wT0sz7+vB//YLL+c3UuLoxWlv
g3M454xmw3WEHmwV/FT9E5978LirCc2z4FFHFu/Pk1LS/tMesRSub8F7l4/z
VDmJbjk2UfJ74nGXeni4+OeyHPhYrfmUAj7ZLeDUwrVFcU7GcNcNWTNJzZRk
+3mkiOCXiW+NI3LQFU736N6jxE5kMFoHNxjRLJMhz8FKl8qLcOlQt0QC11wy
nDTIoyS8+AdPKxkla2XteU1whn+j9owdJayVmxKE8DUVprta4W03jxSw4Evy
+i8zWyhxi5Hcblffy5fn6GqLfZMsbALgMcL7jMfwVHbdb2E2lBgzQ25fgsdG
rvZX5/nBYg43wxo5iPXansIzvP+ao4IrwlIivlhOSVPNpwqtVkoOvp8V/AIe
Wj3D/opLydDFu/368PO98vyP6vUfWkXP5VCStzlxgSlc9e21HVroB1/xsfX8
ZZ/Xeeqma6MPZ0iT9AKtKNHmZTm9wr43BHY72fBtQvlhHUucu6OHJYZzxa7W
DvCi4wPVDmzkVbnr13j4Ac+2+LXwSDtrtyALSgpLzH048KnK283e8NGegJMp
5pSUs/yCZMjtEn2fEwwXbU11KGOh33qvc/3ggbm5LyPU+56gpV/CX3ZYtvSg
Lm6GNf7R8MIl1+WdZqhXx8SEAN6zJbgoHW4d1ZfwxhT1ys9feAd1rzEuS8uF
mwnj3ubC/xGs1lkAtz36MZXCHz7ePX4aviG59t9E9M+MIHugGP7apP0+V/3/
FVtRF/qwyNV55xX44O5eQx780Y/fxHLhJUfiQ6rg/aXu0pXwxNLiqTj083QR
fVIHX9Q1rrsafqLm0K0OzMUk6YwWw98VjjyuhN8oWLTPEr7VqVj7LnzArK+A
hXvJPbPMUzBfOns9WFK4+6V1MxQem1Clr8ScGoX11jfBOdfmzddCPlH9Z102
w/nKzv3qdfRDDxWOwXcYh0Q04B1wNuxLb4QvTpf6hCB/3/PPW1fBT0cWVYvg
OXUHW1xQr5wsL2GriBKLJsnQHbiTrYbzUtS31/PMngC4Inad8CZ8Q1JLyQf4
GUmdgglvTuW7qfPJZO7U+hN9Ut71KLC+npKuAyfmXVC/SyZCHhf9tr3t6do0
+Pf1zq7q/Dk3/ziqif4cDVfu8IOf5aW9zoaXql60D8E3Tiut3OGMBxUNyfDo
V2bH7NH/W3wGJlzg8wvbxtX9cCpmS4OO9ed1trk7OIbBlX7MPUGYL9dcxcN9
8Gd2NiMB8KWsZ1l8zOMNVbeqBK7Y93GBB1yj6PBWXczvL/0yx2H4+PBFubN6
LjTLpSPwDu+yxhW4ryQ9PNEWfvJtZnwz3gc7jQO1hfD/e5//A+WvvRU=
      "], CompressedData["
1:eJx11GswnGcUB3AUkah0NctSOtUisxuXBBWXRR4apJuslW1dKi61u+7EdTUs
Lbu5kJA3nVjeINJ1p+IS9yCVCU2i1UzMimBaoaKb2bFNtSGCac9+ej71w/vl
N+88c57z/M/5kJfKjdHS0NDwhO8t+E4MdFMvzvSPRtF0ksTxJOp4eW5axCAQ
K/hS5tfg8onM15EJJGKMb7ppgRfNyplJ4PNSze2mRBJ1+V20bqYTSKnYJYgE
71PKjkuSSNQW4EWmgPMapwYDwX1jf74XkUwiC16LPAKcw/E5wwQPcjfoqk/B
/wep0Je24EufRt/KP43PN35OvWkOTqOUU4NTcT0XdG9Hvg2ueVU6Y5eG69d5
RonaetI/Sh2+saydTqLGNUaO6QECedtT2l+ANz4uZ86DV9KeWs2DWxerxDPg
tP26Rl0ZUM/J19JeGwIthXYqxsAFTbZ55zOh/svblxptCTRFpVUNgP9iHGqm
Ao9amCsesiOQKHzTog38wFx9SVAWid4tEBXP2hPIUl9oQoJLhRQRQ0iiMpfP
HlEPEWjklrnyW/D2YzIrS3BZYciTYAcC9TTXpp8Fd+j4hCgGd3wn77tmRwJR
WujT+eAv+B0/isFrq0dsND8m0CQ1yO4rcJ9J4ZoIPKbMfOOUM4HyK45dSVP3
wek4QwguD68p7DkMfTP5ey0B3KtzOuo0eFHJqL6BK4GOFr7KSgQPCNluegP1
l/5kKI90I1DlnW53AbjJr+yj/4B37qab1rkTaHF9lZ6svtfw+dV94MnjmuG/
MQmkYJd2p4Bf+2HaVwb9mWn8g/bGg0BCI05BKjiXeXDJHpyu2x+o50WgLXfH
wTPgSbVVCxzo/8ZIubbGEXj3lexCMbjQLzEtHd4r6l+bjDlw13Flfwn4nr/E
GVfh3ZMlq2E1CO6bY9taAZ5tEefcBznZLZdtMb0hh2m+oXXgy4Z7B2chVw+L
Qz7vBk86opR2gCf07jzagRyKDdZjDXwI5HPXkj8MXnvNJvcjcL9RiRMLnKpI
oT4GP/lFtMY85Dy8gN0XAt7gfDBgQe0SMwvNJPy/jfXhNBW4uKXTjp6Iz9/r
O2GuA7l9cN9/aCke14PG70QYgYexeKo98bj+Z3XSsP3gm22yRac4fN9WPd99
ruApp1bKImJxf2Y8HdxZ4MumnQMXYnA/a+ZyJ4LAm4yNgj8AN5+o5BpC/3XI
y2YJ6rmjBbWyBPi9oh1vFwnBR4SejN/5+H3H/GuviMEbqt388vg4DytjojEC
XN9vlTDi4/xMbjp7l4Ozdq47mvNx3rgRlKd14FodLou9PJzPV+9lCFvVe+mB
fcMwD+c5J7fq/S5wrsA26x4P5z+Q63S3E1y60c524eN5WT5b5NEPzpGsqTz5
eL56yam0PvBm5WZwFx/P49xQqaRX3f91RbWVAM+vUbi9bg+415/1nAoBnveG
HA/v78FLT5g9zIZ+7pR/w74J+8H4uv2uFvW+YvsPKWLwPnGxe+lzA3xxweLQ
/Vi8fxafS7Sl4B16HvSGOLyv/m8//weR5Ncg
      "], {{-5949.067376510296, 1920.9231091471734`, 
      1267.464180355903}, {-5949.503067004659, 1919.5732591430033`, 
      1267.464180355903}, {-5950.007033660955, 1919.0479385122658`, 
      1265.8929650949867`}, {-5949.741897683787, 1920.682241586206, 
      1264.6599415659214`}, {-5948.645192685862, 1924.2268157324017`, 
      1264.4307901786178`}, {-5948.195731267555, 1924.654834886884, 
      1265.8929650949867`}, {-5949.067376510296, 1920.9231091471734`, 
      1267.464180355903}}, CompressedData["
1:eJx11G081WcYB/AUn/hYx+SpFGElmRPWKqOTOxRr5pSHHbLWqTRlOz2cgz5G
06E7WSF/ixJ1Ujl5CkeaEh0xceQhDxU5lEYPq9TwmdS0339vtjd78X/zfXHf
//u6ftdluWWX77apU6ZM4eGbhq8lylx2RVChtIh86ZO0miGjExFVTQ2UvMq/
VXgRvnfbvi3j8J8Su45IVJRY51nfvwBPDXLVHIWnuQttrG5RcjN+QXsOfFwo
iRmGfxnbLHvQTMnlndK2LLj0h8UTz+BZB7M7s1op4ZwsMWHd8c4H8/fCt8w4
F+t/m5Lt+8/bpcMPn+xRiOEf39Af0umgRBJdmXcMbnpDqvXegyFJ07TLSjsp
+VHE28i6V+Yy6WG4h15Izud3KHm7Q/UN6xM1xctnwTPnFD1W3aVk60PxRdbL
nMZszrszRFT+Om5pNyXmzRXfsffGG+V+/wk8U562dX8PJbV6YbkZcPvgDI7S
jSHdK/ka8vuUMKL02FPsOUON2X2rGDKwsNympJeSdo1W9fl//kcr9S/CEPE+
20ZGTcl4coGMrds7g2NBR+DRKcTYq4+SZl1ty1K4d6XCutCVIXt7pL134AWO
7QFX4TsNHKNvrWRIY+tknGM/JdUpk9Pr4Ac8r3Ff8FA3h4dDoXBBAI1sgROz
cBHnP14y+GZtJ9xV0yD+9AqG3JefS3OGU/H+kX54lHSsqMaFIdcW2rzox70N
+X1Oz+Eamkt9fnNmiG57yIgArh9q2jwCV65+r4iEa5MWcR7eJfnQw1sjsEKZ
MfaEl/kZQ2rrg9vrUYf114ze6sBTEjQiuXD1gbOV9ahbj3KtvQHc60abfo0T
Qx5bvjCToc5WFofMreDyJ0eLB5czxKxgwRtf9MX4tYW1HTwqnPs0Cn4vIsx2
GH307xnd6AgXdta27YPb2y8aFKHvu7sjZq+Ay0wuH50Jz9zYNdaFnOTEOLSs
gld1ybuM4dXvDUUOyFUVcb/rDR9c49dasIwhJv7OHVLkUOul+KEvvN5ptn8J
XHW4LLgWuTUS1pX7wR872LnMxznzNKsfabZQ8rQh63QQ3E+nV24D99OZFcPH
XHyaNHloEzxi7hJDLvxUx3TDTMxR2ImvZwjha8oeXZfBB/xLPf7A3BXV8Dew
XvDzejdD1CcigO+74SYlobYhgaxb2T9PTIBzU6U63b9SkuCRP5119Z7K9Am4
8NhXYTvrKHGNubSOvfesv6NAhPqfOCNZZVlLia35W3UgvME615iPPk5mXTo9
UUOJsDhuiQBuZsK1rIFzGpiKESUlbl79nXz4JlOxQw5yMlcz3lAfbnBgXrcn
3DSJqOKQK6dnm+uCr2OOXFXpHnBOYcXZEXgfFf2iqsZ7k3mbXeB6r0p9O5BP
514fvd3wtFGHXLaPVUHl2QrkPDz1zEsX+IW+K8sWwpPmJG5Odf3XhY/W2c6F
GxkoindhjobrFW474HskvCtG8JgjudZ98PGbt5/Vwgs4H0nYHGasDE+sxJwm
K/8c+gL/OS6Ilr9DnoefvE44jrnu6o8Nm4AfFCwWjsIveb3K1sIe4McwHDXe
G7FrQPE7u6+We4eK4Ylq80Vq1K1wdKanGu6yfSrXB3umrXRANQ11ftdwL7mD
PUe3gnsVnmPcNLIOffl2RnVKI1ymq/9mAfabhfvSOAX6+CA40eo6nLfp+J5U
uK27btMS9P3/9vPfMsrc7A==
      "], CompressedData["
1:eJx11G001FkcB3CUcVAkZCJZVA6nVjs9SLHn2koyJLG2MGmzYiOtpEwOCW0P
J/+bxx2PzWobFW0PyGRp1KQhmyYPTTtEkqE8tJ04WdTud155tS/+bz7nnnvv
//f7/q713oM7wrQ0NDRc8c3CZ5Ymr7PzEtdXdUb8YyGkpN6Ie/RkO0NMv54e
/AL+KnypqTH8LmuJ1UAHQ940PIkyhdcUuAUOwn+7mnnaR8GQuYdO+BvAe2sG
LHvgUfYZFXefM6ThcvcGHXjpoFVvB/xo9JLuFUqG3Bp73zQbvnbPyLHRC5Tc
I7vcczsZ0j907sMnrrj+LleY3Q+/Gblm+GMXQ6ouDbVMwX04TVmfiynRjV9I
t3cz5JIOMZiESzemmrLhVpMdQ9k9DJmvjBZNwKv/9dHnFFHic2XxRtlLhnhp
XLCbhp87mewYXUDJNUeef38vQ0JdpjLV+/tuM3Q+k0/JdNB09/ArhrQKlIs1
cM/olEN3HgoosVMdf9nexxBl75VgbfgWa4W19i+4v6RSIXjNkMSaUhN9eFdw
of2mHEo+ijp0NvUzpIY4CI3ggyP1m9KyKNHX6ol8ADc7Gyljw5tUxQ4NGZTU
NGtvWapiiPFAg1Bd/04PywnWeUoSogzYe+FeX6l22sM/2ZoPbGUoaeXHd8XA
VW9byznw8IkDwvRzM+ttfE/1uMAHlr091np2Zv+APsOlXnAvB+Ud59OUxKx8
sKIZ9zEUb2b7wSe0GtJ3n6KEx+fpecO5IcYBIfDb19YaiE6iPl4pQSL874GM
/s1RcIGmcdxoKiV8SXhJJ+oj7lr9kA9fsV+c65RCyeThfuEH1LPZ+PvVqfCE
2AZhyAlKblztYI+j/keMDksy4HxfXYurxyl5/Ee+jgL9eu95eV0h/PPY5qS2
JEqGdt48WIL+2omix0Tw19a7nD4lUuIaV+S3B3n43a6m+Bpc7+fbSm2sN6mt
u6HzgiFMxPi4GL57+5nglVhvpqnSEyFvltkVIxK4vHL2D85wUllV4op8GrDO
s2Rwyf39ud/AbfTs4luQ5223XBxa4GFz259x4REtaXN4yL+LOFgoh+dIWc6r
cO48PfOLY5iXoUa3/Qr4Vred2i7wv5NHN+Rgvlihjwr/gq+rvTnPFv9LuVXf
rm9DzjPC/TvhIcs4ibrJlJSVx66deIo50mlv7ILnPdHPegfn9PXlS+UMifS0
dVPvw7vC21eBOmf6RT4sfcIQ4T6LZ2oPCE8OEqAvxfPvOwlbGLLG3dO7A149
t+xVIPqoXefPET9myIRcZqS+/w4PUcoi9D1AgzqN/cmQpIth+x7B16/UzHyB
nDg5xUjd4Ty+TC6Fu2jJY4qQq0GzhKLGZoZImiyqauHPJ99dDEYO5bT4eBh8
NKZUXglvbC54ao7cLigfH3GAx173YZfDp8bXDysoztVfbW8BDyTV0hJ422HV
huyMmfV3TFwLBHD3VE68T9bM/g5tSc/S4WWXnd/oYu784h7kS+AmyQv8U9Q5
j40OleZSsmg4QpPg/lmBi6yOqOco1GQ8AXP93cK2Q6/h+uyw5T+q55efSFfh
HXAW/ep4HfWpkC232QNPnDJmzcG78VKzLi4P9SyrSv5yB/xekI3v/UJKlKZi
TQHqL84S2HrAW8z74vPw/szJ61lwG/16LLevdIVnJ0Ra/oT3yrBTFjOI/s5S
cIfU8+vxyMrbHe9enqz66XLk4f/e5/8Apdjr7Q==
      "], CompressedData["
1:eJx11Pk/1HkcB3CTWqXW9siZRq0jhBwTrR3GfmxiWeaRTOvYiu1gkUbrWjmK
xaa2h2/fdLm1W03KsZtC1hFJJEfF2MhMTTRJUQybc1/zB+wP31+ej8/j8/h8
X+9Ddw9/+/5FCgoKHHyK+HjHVx2/r1dZz752IOKcNU3aprkfJfcoUpi0+5Pb
8EBnn82HbGgS9dGrZmszRR4JtmmXwcOUp9T64XZMDZ1bcInU/eYl+PthvfPO
m2nSeDS1yuo+Rc4auAqz4KUZD+vK4CmvsuhKONudfTQDfkb66oT2FzQRB3EF
bi0USTTKW5wC/5JrZZ8Cb8x+zhTDVef/Dk6AZ2SOfHUE/rSGwU9opYjm+9z2
aDjHIHbLYbhL12S99gOKuL0rHuDDVfoUlKLg1pvCx+rgVVkda4Ph5b2d7/jw
kytUCgPaKMIVGGUGwX31M0Nm8c6KiURPxkPk8Nh/YyBca1MRIx2eVtN4MQ9+
yTx+eB9cqq1Wrg6PNhFl2rZTZN7ukaX8foF6GNsa+eRWmV1thf8wF3D2R/iH
jKMH7iDnyJC/RLwOijz+fVluGFx2pjRFvAnnffQ/PpF7RotiBLze9fbgAosm
Ft9ZqLt3UmS54YqZn+FMlmb5Orgox/bmTbjPA5XsI/CVdTMMYoV8XGQXPuui
SN1kvyQNPvJwV0eAJepVfaDRF77ByKbtNHzAYDJJ1QLnNUY7LsC1Jh25OXDu
i3Y7G3OabPbkNNyD37Iszi+C7+N0zUvNaOL0blZrEL5oLOZEOVzD3LNlOXzE
O2PHGDyiMNapDh78jGduYUqTuGJZjBQufa3Y1gGn/83LfrmBJlPbO+e74Jq8
/bV9cLNm75Gl8FlKZ/g6nDc/lSCF7xnnCjca0yTwhm11JHwpI3hIBv+6aY2j
lxFN3KbUu8zgsacSRYr6lfXxTyM0Yw1xXpERKkQ+YeOau1bBh54Ze+avp8my
thxBFPwKk+nChDsf+YVvDf8jn3JbAucYdF81hPesD5O2GNDEjN1Zmo666I7Y
plrBQ5KF1f5wJz9e7yJ48V77f+zhodSN1zJ9mqzWUL8bjrpP0yzPb+DbFX1Z
v8ELhH9u60P/KLsK3nLh09+bm5yCexUnLHDg4avF6Tvgalbdumfh3LC1JQXo
z+elhw7vhEu0RT7Z8KuXBxNn0M8OnBD2XvjbD3tDC+AmqabtfvAttS49++CV
vc5TqvL3556fq8S8VOqFOwbK/8u4ffev8LvNx66tgS8IpXFBcKHTTOE03KNi
QZiMuUu/uJYjv7/8nPa6fuTjmODXMIb5jaf6BuXeH+sm4yLnGIe8kP1wO63L
vbvhjDaZ0AJ1mWax+EPYD8FN3sm+8Fy+cspK1LGWWW11EO4SEpTsBX8jEJWN
wVueFpUugQuk4y/d5XmmJwV1oh+u6+5suYa9dHxQqWgrvKtg6EqpCU26W7sT
A+AeJTkVtvC4cdH4LPrN8Ns7lC68wvqYjSX8BTno0If+fPIsPXeiiSKtnztK
dOHhE+IkI/S5x4Cxcw9cbD8s05TnkLWyejHmonfAd9kD+GjDBPtT+Gmen+9z
+OSczqtHcP+eyZJZ9KHI7NKoKeZOJbo64DW8N3x51Ri84l5ogRLm9A1lGa2C
90ga42QSuJipdOsFfGkz67IzPNIzLbMHHhrVwKvBHqhpXfzTSfj/7ef/ALmA
4Ek=
      "], CompressedData["
1:eJx11G00m2cYB/COrZqz1ual9BxKjHmNmZ3QKbVbKp15KbEzU+vsREpNFwSj
ow7RVe2UJPKoEnuKFsXoyFgwJfU2Ez2WMmRUWkSrZRTZnFbZ1U/3vuzD8+V3
nnOf+76u/3VZRSWERuvs2rXrCHy68DV4sq13fyiTux5OZDtcJpFY5ymP0k6g
6nIBb/OYTC78bDRdp5hEbyyVRQ+0Eah2LqThPjhlo3G37xUS7dNw1VXgGdzU
k0rwmb+cDM1KSFR86ciJ8+AKe+2mHFzi6Ne9Cr7tnUM7C77uo+T/CK5Jfriv
vxT/35alyioCV9Ksz+mU4fON/z65wQePf2h85o8yfB/VoSoFF9wgwT6w5nt8
/4KI0QMnwIdKfDNTSRLl09frPwV3tdVMHQW3yQjdZl4l0QUT8s1mcHaH+yVn
8Kie/VTjchLtuEtGqB0E4sULpJbgy3fa+tLAj+VlCkTg6dKWoD3gf6675vpW
kOggr7zG5BcCmQZcmNhkyuT5cpVZDThD4/vkBriVIYW9CH6cpWrSqyTRjKzL
xaOTQJZjs+OT4OlOkoXT4E7WIXfugxPH6cFK8NgMVkgkuMsExTvzFoE+Gkzu
k4Pza8e/7QIfeeI4ZdNFIBlXJG0Dr6YleLeAc2cNzg6D+xTGVjSBi1Ob/qkD
N2/vMk7pJtCspaFeM3jkUIm7G/gK17XZTA7vCo7cqAVfFIasOIJvbn8Z1Avu
R7edrwNP6z9672d416ziQecXtwmUe04wWg/uT6coGOAWq7a0LXB+VHxgA7id
1DR2BOr21UB7fEEPgZjNB1kycAtT5eIa1H+82C58Ty+BJr02ptvBKxcyNdng
KRapPufB9Sipcf0v63P1bZNq6KN223l7DbyVtHr9LvjaT+HVQ9B3eeMDdmAf
gZrWqG5qcG1n6dwq5GQ91KBUAu6mPJW0DB6jZWUfAGc4haTdBR900ffYAjcL
0x1GEhJJOaYOO+C/3p7ZMYL+Tjvd89+B3A6axN2k9hPome2LJRvw7MmpUMf/
eFZ3pcoNfK7qg4hPrmBf8CA4fuC91/Ym5BTj84PSri1xwB8zPVqZl/F9Tt98
/7s0cN5TuW1KEb7/hNraIx88b3lVe53A752dTkqqBDd/fip5TIzrk5bM9H85
X/zpheFEMa6nt1cUdQA839ci7Hohrr8uweePga90Z36OCnG/eGFRcUvgE+Fq
RooI9zeAFR27Ba5T/VqYkQjnYaTCKGYv7JPf01NipUKcn9+6GN8Yg9u/11jX
KcR5U4c555uD0+YFjweEOJ/Ph5WGzuA5m4H1MUKc51e09GYa+K1DQWemhDj/
6mesXjq4vsrhB40Qz0t9TnKAF7iAo/BcEeL5UjbOPfIE37/svsUR4Xn0cxcm
HQbPvSGNnBDh+f14KeLiu+AtHAqjtRDP+6ikMPcdcEt2qNZOjPdD63zAi7fA
7crzFLoE3ie0xIJHluBi3gj1awLvn1f19AOMwDsoPSXBRXhf/d9+/hd3rfFl

      "], CompressedData["
1:eJx11Ps31HkYB3CNIhp7SjluJ4Yu0sRJDdEwPhspCcUscRKh6OaylbNqnSiV
Dy3pW7ksZnM0a7ouYkfnsEoiStsUpqWoXIbCtsPMJmXf8wfsD99fXud7vuf5
Ps/7eawi4wP2sLS0tNzwaONpl7h28/jSBju1uPHsVoaIFs6z1H9GSViPIpQL
/0Vr/IgSXjpz+ylLRsnaOcUpVnCuE6tnly9DBDy5ueFzSiKMHdON4Y8dz65q
hXdJzVeveEFJS1vKUgO4xapdXEc/hozaMNbBHZScuj47cC48qKdI6AA/KdOX
neqkZCQjimjD9ZQfve3gQt0CYUMXJa880u2+rpc2HM4wcreF+6v7O6fllOzv
2Wv2CV5apMtbBt8kGtvh8hclqQ0G5Rp/frsishL1RMR+bxjfTcl0u0gwBR9o
vJrvDj+vmswo7aEk60oW+zO8w6N36jH+97RTbVzLK0r4j+QLpuEcy900FN40
/0Fj32tKBoMlAyzU+asiIynZhyEhZiLWUC8lqxNj2frwF4W64vwtDBkuuaiQ
91FSs3Z/zgKNDxBLqTdDJgvG19x5Q0nHMu0fTOEv9+UNd21mSEL055Kkt5Tw
qsRVmj6vdnNTqTcxpMJYULn0HSUTK2rOrYSzB79LNYH70vagG/Awm4UbXeDX
vG8ltW5kiMkGRfjifkrk7IfGHvAj9WXN7z0Zkr84YioGbnXOaCAA3uZ4orDE
gyHK/DUOl+CHlLzyMHiwd/21hg0Myd0ROfsqvFB9OjMOHsfpDDnyLeo/+Sk2
G25T7xV1TPO+yU2vPMKQP9JG94XC+SJJ3Rn4um2HCu+6M2RldMoDHfjfHQZ2
BfCYXeXp8QKGyOtslNmo/26/eZoYLuuLLmHc8J3jP3t8RR/MvZd63oFn+3wz
X+rKkBqF4eEAeCCrbG4dvHrcvsUHnqOnO30G/bwT3yRpg9dG3R1O4OM7gzHD
Zei/aXaKTQdcW3KGNwduEcLl38C8WGmyWb3waE9JS8F6hng5eX0swnw/Bsya
NQyP9elNtofrNNkXpSMPQkVioBI+Eag+2OjCkCut5mPByI9ZF2/5v/BHAkqa
4Z2qx+5c5K1f/bvLF818TyZvaoMf+DRZN/aSEqlyeivLVdowmOzs9xSeU9HD
r0Sez7VqR+jCFUseCp/D463Tao8i/6HzfuzSga8Tuy33Qj0Ovatm3LEv5i8u
+2l8v1OhWgqfv7LAUx/7FR1+IH0OvLo85BIX/7u4yjqsH3u6I42VPBs+sPPE
TDH8txrZ5nbstXNxZrWmTtH7OIsT6GcQe/RBE+6AKTcxSwU/4M/7EI653LcO
tOz8kxJvo5GfJjR3YOD+5Wdwftlp2dRTSg4mdjsMwSdzWbUczL1sS6CVEzyh
WTH+Gj631leihZw46mX5Zbajb9q2pp1wacXFJX3wz98YlbPgfuLjUc3wLzo6
PHvkMJQjLM59QsmHq+cLNXOPzbMdYyO3Ow0n6z3hQ9ZH/avgBrSfvIf76iec
NYYPj2TPlMIrbXPnOWMvFr1uVunBy+TC4jz4cr+9qUZeyOe9zAsc+D0jlUCT
W5OvQ4d3Y7+2b4u6IIRvYfa8SYYnZ1/jCLCno9OLXMrhfanm4wfhTlxVkBn2
2t/Chs1B/dcbxaJwePdb5QYV/F31h2O34Htc/3myHZ4TtLD5Ge7D5M2k8zvR
n4lM51OaPY2MEY3cwD2JypvQW4E+/999/g+beNOz
      "], CompressedData["
1:eJx11Ps71FkcB/BJ5NKmos0tl6YyhbUmGi3iVLKsJFKDXFppwyhR2raii5JT
rZ3pS9tTrFJWPdXjUnlaNvKkmy6LxrgMuQ/rLqbGzsq+5w/YH84vrx/Oc877
vD9ncUSc/y41Fou1Bmsm1gG3BOeQNQ8fr5AEhOmZMeSsdLi4p4mSbGX0gc1w
jSPfeR+F74hK0IuTUrJPzTrRHT7id2e2DH7QQl9tVislo4LNtWvgZ+2ev22H
C9ipf1x/T8n+nukBHnxQ5homhU8sf7Ler50STxtzth383MrtMyXwmD4vDUUH
JWUik+9XwLXiEwtq4PKvslKyuijJq0lsYMO197mceQWXZGnF2/dQMjJ3tH0x
fJFImM6H+5fVZD6RUVLBTr1nAt+Yyyvwh8veafWu76PE/LJ3sMqTIsZXtpky
pP4lzbr7N85zrTBU5e7XP0wL4M2tO0tYA5Qc5RQ1qdzAg9hMLmLIsNbBWN4g
JazJuw2L4EtOO1Wkwlt0DY9uHKIk8ha3Ywn8xNqojjwThvgl94a7DlNSZaXt
aAkfUqtPcYDbBm+tZo1QEt0nS7CFr2oe/DXAmCHmRpP6N+FGNzKVXLiibK9N
txFDnjGB3eajlLSeSLnvDHfrWLBHA57mvjo5Bi6eoZzhAc/midkcQ4bkbkgZ
FMLDe/IZb7hXzITfQwOGBEzxl56Fl4etO8yHi5vkXtKFDCnXqzX2gdvSH+Q7
Vfd1air5/CVDfi4t7hnAecZ0w3XjVX7mfQ4b3hHtsi4Szn+eKfoJPm3whUXJ
AoZozvEN/R33bTP6qDgD38cdmSvVZ4iJrZD/BvmUCiYuiOB7QhwEcfD1/RlX
XyFP/ozfKq/ANd/7OKvDowLLH+Uh/4ZA68rb8KiYSwIrPYZEGoW4RfdTkub9
ub0Evr9arqiYz5AgU0WGFt7xWuyQ5E/45Wz+rSp4dVyoOLOXki6OZcYL+CNj
bnEwXDenslQXPZlyVorEcI/JnKHReQzJ2SZ882M3JYqKU45S+CYbGecj3I+U
GDR3UrJ76J5pJ3yXWD1TCb/QyE23QW8FDsLePnjexZYBFvbXMkt4fbqNEvmF
qcIRuNw0dN0seIy6i04b5mVrtmL1R7iPY+Su2XDrqnpjzxZK1DwzOZ/gRXfH
Ao7Bx6s1eaXNlNjsHvabgu84HW97Ct6xPbTjG8zpLXXL2ypfFvju5ie4nv0A
p6aBkuM9ZvkqvySa1xyD3ATCSu0kCSVKTz0rlbdt8LrTCp8Z4uT7bT0l7Y+r
rf6Fh5U6Gm5G/v1XrZ25YkoOi+3Zcng+z2fSDu/LOfn1Frd3lDSe3yb+AA96
4BWWC1/bFZ17uI4SJ52Dv/TDq7ykHifRE0MX30v1tZQ0B73W6IU3jCbf+wC/
4pH2NBQ+ZnJoWyu88cWn5Fr0sPNicO88eOqr8uk6uL9Z0bMC9Lb3uGbZYA0l
ByJ4e9/A/5IYXDZDzzuX8VPG4axY/bRK+DOeQ74STjPY6XbYRztx4ZwH8H+0
J1wbMS/26c5PRfDCwfHxfPjy2PGg+5i7orZwgRHOr1NRfCQH3uLd3GWBOZ1v
bzG/Hi67IdJR9VZXsn1ECXeQpoQWIofSl0lHVD3X8bXM3o1/4NwNdUkBcstc
JdiUBD8m572tg48JtzyqR87dhwY27oFHst2z7+CfWeA4i+jhvdoDDi2NgLNc
r2ka4L8S1YVxExtx3//5n/8DYfvQZQ==
      "], CompressedData["
1:eJx11Gk0nGcUB3DLa2xjnRElE0yDaoXonLSEkfNwxhpbE5EitcSScOwmDeOU
DMZkNDJeTeN1QkqapQwqEy2ChKjltEkqlclU6NFaQymNLGTtnU/Pp354v/zO
e55zn/v872UfztyXpKGmpuYJnyZ8pj/Wip4Nd/Txo24PxepSqDQ5teRYGYnK
TI63/Q1e85fjAa4ehRIYF5044JF3ZElT4PeOvkkx16cQ/15vsTG4b02s5Ri4
y5PUwsfgAUUK5T8iEsnpk3k3wBte7p13M8DuY72now08S8Y7YWqIfXSNfqIB
PMozsH7JEJ//nBPjXwXek9x+85YRrscmnnlOCD7XvZRYY4zr3xPou5IFbpVz
WDfLhELleQ7jneCiNiffWHBJk89VH1MKhXQN1JuKSVQZm0OEg2ciN6NW8A+V
qOE4eHKj9wIC/1T0i0cxg0Jml8SMdfDs3TbNu8C5XgO1i+Ducrko4ySJMimR
CVt1vmVtQC+TQrLuJpen4HdnlJMMcMFYWMt2M/i/8AhTLCFRCDfB0ACc+UPJ
CAtch/ani205iWzPyA+9HeroWyHcW4rBO79yDO4Hj1gfrNgAv+r523ABeIdd
YErMlyRq1jf3/hfcwYOeyAfvPcK8QTtFImTVnLMEziuzLUwHvz2kGP8W/LvZ
Y/Yz4Js5CotkcMMPuM5BFSSqrhDuVvniVFDqY6g/JbQ+YBE8Ubo2NwlOTj5T
XwYXPtTfUXCaRLKtr/l/gEvoSaN54MSZMEMDKYlC43jzs+C/jkgVFPSnVZCx
dh7ce+tI+jx4xaZjvT0479xUnGUlibx8Jvevguduk1r7Q/+ZtnyqHDz4e9p9
1b2qnOPnH8A7hm9yqlfBZWK+/CX4wouB+E14d4PQdz8LIElEj3hFp0E/Heh6
0SxwzUa/pWrw5+yIOBNwDcXBMAT5mbi+ojUKXsCY2LkNfDCBP5AIeWON33/0
Cjx9zHvjffAvggWccsitnbZfFKOKRGbZU1xXVR6WOdZtdArNDr6tsgIfzW2I
ClLlwVrGTdXH/8dL2MxocNmbrAukHj6/P6J8KBU8/dotzeu6uB6ls91TAfia
xefT0zq4/p9P17FOgRddFvTQdfB9GYEefXXgD3e2Mly1cX8UdCf1RvAQ9mD7
XRrup5rQs64bXLojjfZCC/f/otPyg2Hw6U8iL1Rp4ffySlpZmADvP6iedpPA
7+vl5Wy5CL4ekq0RSeA8+NXSMlbBJw7FJMcSOD88s0jWa3DHop78TU2ct4/H
544SIx19Q0Xy39UInE+WpMFNH/ynr2u7tAmc54knl0tNwUce6b5jROD8+++b
3W8B3pzO3zAj8LysvNfHUzkjJO2KhMDzlXtWyLAB33tHIK4k8DwGZgSIrcEH
lcJOTS08v5y0/m/Mwc/nF7nPaOF5t/eesdgC3lW6lHmAhvdDTI6i3QC8yUSZ
u0sb75MtV/I/Ut1LHClruaSN9090uGuUOvg1Ywu3Eh28r/5vP/8HJaHIUg==

      "], CompressedData["
1:eJx11Ps31GkcB3AyLaYz1GlX2WO6uBTVyJhdRQ3PNu1UDtZyOi7HOKR23QoT
MRXJhA6nZ3cb+ea4TWNt6KJyS1dddEOWJYSxhVBEZlUWte/5A/aH7y+vH77P
5/k8789n5a4orz3zdHR0hPj08Cmv5A2XD9XUxVi3ZbQmMaSpmv1dXx8lgzrB
Z1Tw5JSEeMcjDKHt5v0bXlAyo2LX58I/RZjW2sOXVnvtUL6kJPyVD4vC2/Tf
83LgD52KLy8aoMTMKuqLdHjg1jiPk/AO/khn8iAlNDAoKw2+sGt2gAuvSBHP
m35FydLIplNH4OU171lL4PfmX7YOHaYkidPemQB3L4zcsxB+oHviyeMRSjxv
HVNrXTedJ81D/Sdu89bZvKEkNeuKWAYfL5zps4bLImO3xY5SwvwRZnYQ3nf8
XG1VIkMUrQ49JWOUWI1VhR+CZw4vYovgKz8Old5/S0lfxkyrHH7dxk4Schj9
edvKbxynZPZkPFdbv8vVzuJ3hxjiy5qyq5ygZIul6ukvcMcl5zcvhufrr16V
8I4S1mCSgwIe1WRQoDrIECHf/JTVJCVB+Xc3FcA5zjvl9TKGxFo61xTBey3k
E6XwbO6Kr0YSGHJHpmieg8v5DwTVcAf+lD0HrsmbLXTS4F6NCq978J60YGv7
eIb87inc7Qr3tBlb26KtZ3KkzecAQ1w+2KgFcL3XLbf74Hc6EzYmxTHEJLmO
8wH/b1I6643Dn5Wt2V4cyxDWekaQDY9ocXf7CJ/2co9ygvtmGRobw+1Sp1iG
wzV1Ays6RnftZ0iZ2wJlMO4rLee4fQ0/cL9tKlPKkMKUz8PZ6I+vYkuOFXxJ
5HUTC3iOOGSoGP2U+9nu4MOVG0Snr8Uw5MIoR6lE/29IG8Vb4UGXIjwGoxmy
abi7+RDea9UR9ZwnvGFxfEsi/KytvYzgfXsca9x94HOWF3SPwcVtvZKp15SE
USPFbnj8si5jLrzER3orD/mJfvLaUgqP7n0YWh3FkP5Zkf83yNvpuUpbmbYe
nv4PN+FGC2w0d5HPfdvW85LhuZXr19yHu7qJGr2RZ3W6yCcdvlHPxK4BLrv9
s6q7n5LDWb5ntJ52r36vC879/nTJwUjMyxbuxwAKd2xK1RfDryl2PmNjvo7d
jfbTukl+ynQtPFIukp7DPHIGqidPwN8sTczgoT/CuHzvIDUl58LUXK0XPa8t
V8IDLUyvr+2lZDt3tjkTrpuWGLgYfb7ov3fEuIcS2xTO8gx4F9/rfCq8JVsi
MeimxGtatfwoPPbO0Qe78Y5+bCf/Fc8pYb/RtUqCh4WEtLTD49ghwQFdlBgI
EvVjtedaxfxVgTwsMpl5e7GTkuIKC70w+Nn3zppfkauBANETO7ijUZuHBB5e
n/nTXuSwNU330rMOSuqOyxb8CH8pVLjvQG4ti7aqc+AVar9RETzj2mSAJXJe
2V9mmgG/ap7zaDP8kWa192/wUG+hUgnvof8KbOESs6eacMxRaP90TC/c9HH0
+DJt/Qa5QyLMXeeXAg9n1OMi/zxuBOfJL1mYYU4fio/H1cNv7Sn49hNy/qfF
u7MaeO7ysSEp7juatb9IAzfpGHOWYA+QyiqVEP35u3Rd4Qv4P/GKJAH2RjDZ
V2KLfs53cBlv1+5V112zuXC73maNAP0Patpf/BjOutnwUg97yXvC8JEv3su/
IX/shnZ/+nQFRMDlnlLzLLzv/+3n/wDd5RCj
      "], CompressedData["
1:eJx11FlQk1cUB3AqQcEFCx3BgYAgNrjVkLJHIwcw7IRFquwiEgUE7SjGKWjE
MhcDtzAUseyLMoC2JakwKopRwiYtoFWgqBUMaqlLRRDBKNb05ClPffhefg/3
nuX/Xev4/aHCeTo6Ojz8dPFr6xMUd+W0tO3MGe2WJlAIPJPombWWQG27nqQT
XchwkPXsonDg2u7yuXUE+Oy8iA70ZtbyQ4/iKYSKaxhHvyAAB8+Ft6Ovm3XZ
92Enhavz3QcmNxDoiDGP05yvv2xU7RhH4bjMuCDKjsA3LjXlmvPXT/J5wTso
+HabbZRzCLChUK8HPbx6jpcbQ8HN46h4oT0Bg7KI+72aOrMOXOyMolDYKWS4
OBBgybMf3EIvyl0er46gIDZVvtruSIDJrfcdRN/nnE644RSqplQW4U4EZAoT
s3voNVNqU9E2CooU5gDXmYD6hSphFD3KbZt9UxiF1R6uHu/Qs/Tlrk/QnZ1O
Pp0IpdBaWvko34XAiICf/Bw9ieXBWB+C/VbkCAxcCSRMtq+aQu8+c+lyUhAF
7or+F5HoT49PfK1Cj5OlGJ0NpJAgWCo6jv5sXqKbTm5L21pT2/fj/lrv8eE3
6KPHZIvyWH4UlAFNlmHoGQPv6RL0jrp6c29f7b1bVv6oMkE/NNry+wlvbZ1G
i0vfMdHnr68pO+dF4XrKNbtJ7MtHtC3WBj09KmCgl09BVqbWc0A3Nyznr0X3
fzdUMbEF6/H7MzgE51btf6qAg17eMzlohH7gmGJlCM75QsAw2xW9qys12dFT
u5fpIgc+D72yqGvaFd09M/q8Lu5Rx+uB1B190aDi/mZ0x15V6kXcuz18KvRC
f53RY7IF3SQoXxyKOWlwUnP80fdcWprii85WXlH8jbka3NPEDkDfUGWdxMZ6
DpJ0ZxHm0H7vufhA9Jue108vw76UUW4332BuY6VMsQC9bqC0bQ59+9QDTjrm
fE/I8l+C0HfsHVGN4Xyambo9qtUEpKzxn4LRrxTzZ27gPO39izadsCVw+OSm
ghD0quGp4UYfChZD+mImi4CN1EekOf/eqISdhvsKbU34rmUVgcJgpwWa+mOC
rNZwAyjsNhzKTbUhEK3cu9EbPeiu3VU15sG7vk/OW0lAbFnJ8UQ/2Xx7YRfm
53mG4jMrawKlFZzezejDtuOvczBv8oqwKgsrAovCx1Wa+S/Oi9ol2ErByCt2
nLuCgC3vdq0DuuTXIg/jrygsMLe8LLbEfledHdiA/iR6cekQ5v/ltMudcQsC
vzG6ilej+91tP1yC/8vH5mdn96H/kalUWqMnvy3pj4yk0NLC6DZD/76/VWKG
Lr5/az8zmgI5X3hslklAIq++YIyeGDkrGYnReqBr/5eG6P88/iFZHktho+Xp
YRs8p+NIRLwe+s95hklH47T3MtQ+qjn8X/wezrRuitfW+WF0wewMego7dcoM
3yWmeWg2wb54XKHkFfqNkDGpCr2QoXrriXN4OMGoeYr++ROrx0P4vvHgpYyF
c0tMvrN1DL0x7UJDk5BCqd60koVz7r9RXDeCPq/sIpUKtXvRO2xwegj9L9Ml
3+bupjC7nxGehnv0bDzEvY0+YWdcQtAzqt5WKnDvNbUV+X3oY/XyuEz0mSZH
fxvMyUfOGanmHXvT8O+RMjxfPXjdIB9zlZnf19mNLkmr1jmF3iAruPfJGgL/
9z7/B1vpy/w=
      "], CompressedData["
1:eJx11Gs01GkcB3DNIA2JoRSHyI5Vq5NzauTS8FCNLm6hk85InVm6jft17eTW
hhAaf1HIZZMmOmWWMJxit1VZFlvkEtIoOsxZkiR02u+82n2zL543n/M/z/95
fr/v7zHhh3gG0JSUlDhYdKyQUxpNg6sbWpK5W3uNuSISWVsZzQmmCMvZ+24n
3D2840kafGqY/awTfqTCOu1XeB7n0cx7+P6FgPToEIqo7c3wlsLDsifU5fA9
Q75ORqEUWbs2f6MEHqoyYDYOl6fqL7bC89f5TorhN0ryHUfhhz8XS4LCKPJM
46F5KfwcO/HTADwk3infOJwi66Uzs/lwz9pe7R54VrmH21O4fkqVzRV4Q/qq
Cyfg7UM9L/wjKPKqOy47E35F1kHjwT12fHt8RSRFVmpf602Hf2lcjJzYi31u
O09ch0vSpdvT4K+Xz1dHwr3ibjO/iaLIV63qpEtwhn/JK2qPiNSJxzmVcPH0
AUEGPHCfLMEEzk95aMuKpsh2m2Hfy/A7EZ+HHHeLSPempcE0uKOO7qDinEoW
ZcV8JxF52znuMgZfxZ9LUNxrVGZuNesIH3bhm8VQRFCgXVYM55RcYunCX9qH
7veD8/TkrhVw1jHfWCsiIuJH3rQM+Mkzncx78GF1k7yjDvg+vtWvDH64+sF0
A7zfvVkYZ/+vjxyO8nkE/8srq7WMIyJDjpmWefDNRbYDir7XSOj9j3dhf/rY
zUh4Er31rSIn7sbxbnI79DdqWe4M1+J6M97B/Sf9M5nw1qWCPmW4xsj1xo+K
89zdaGljKyLHaR+oOtzXRH/wvLIm6qOlWnjCRkRuXm7+cAhueYExqQnfRj/5
w7y1iKR6sVky1LnnVnubEbzUobHSCG4idjE/BU+x+mPfFvhP2S/d7+8UkWDz
L8uj6CP5etfLCt6/e8THBX6oinPDHZ6jHTzpBG9m/tI6ZiUiD/coG9QgJ04Z
ixoecAdVbp4Q3raX668O3xK5ov4IfKTuO2YSfCDz5stg5E2coxYQAPerWs/d
AKelGs03Ibevd1UUBMHPqUlZG+GM9lp7XTh/3ZeuKLhlk5kKCy5/rp0aiPzL
Y1Q3xMJbyk0/XoXn3WB3/Y45Gl/R6ZoE18861FMMn7cz1jOEly8aMS7ARcFr
TjBxr58ZDpmJmMfAW2sep8BP13VPJcNNVSzob4Io8mdp+ZmL8ADJtnsLijrE
dfUegxcaTWqkw5+3UOEC1LOF5ybuC6RI24RZVTL8bGJO/EH05Xvjor5TClft
ZCv2lx24HvsAnizsVl0QUKTF1awpEc63YD0oQn/DmO9yiuG5rhrmP8I1kw2s
hchDk960kAcvrIlVi4DbGVa5HUWuXswFUabwhDcRvgJFv9yfKu1EDgNjTw/M
n0X9P+te5cMTqwwndZDbRLb/uVF4gxa/yQceMpLNmPmPL6o7e7rBg6K9H7cj
/0sx45VT8OP3+7sVfb/WQJNWYF7YUoHVavxXrvz1N0VOpPV93CTMl6dFyIwN
3J7TVrQZnhLkq87DnD6VBq6Mg3fwt/oZwHUkZ1V3YK4/2fGEL+BpeTJNJpzO
kb8vhhsIbaa3oW6mNXH1NLjhxOCmGLwPBwULzSXw3PDU57OYC1lA1N9ueGck
CTrN+ugLrz38yTjcw1FLVg8vzZ3TvwP/v/f5H+bNse8=
      "], CompressedData["
1:eJx11Ps31HkYB3CXUTOSWzW1Kpc5bTFLJJcl6pMkx0rRuJVyidpQbivbrkvI
KKfk+8nRdmySbLZatwaFpSbXwrKmyJalw6ak1IYuk2nf3z9gf/j+8vrhe57P
87yfxygkyitMRUlJyRGfKj6Vg0Zt9QY3byem6fq22lOiLnZa/lSNIWb3TgSX
wp8V6/IL11HS4O3it2kOQ3JTe29cgq83yVkyz4ESVdOmP67BU2xXZZ6D9w/+
XRUP/9WGFBrOZchz59mxbPgC5fjZIbja8fxTF+DRIx3/psP9NTo5bo6UkNwq
35Vchvy15E+tZPiD9kBmM9xo546jFfA3mkN9CfCusbePCdy1+HKGE48hjeWV
kTHwJ2Ueqg7wDhGHNwC3am97Hg5n4qb5tvCr4Z4dMeoMOWVnnBIKl41+cLCE
v4jd48Gdx5BbYeW+e+ElUzzNW6jzpOhm5Xm4p85lUQhcIn7y8Ru41Va3GnMN
hoSpaR8OhjtEp6oOoD+F7/gBjXDXyG5FGLyyZ6HsHfq5Pz8v3XU+Qz53mnWw
Hq8ftzsD3ng4J6oNHiFfvDgCnm1lJy22o6TGdm3BBk2GPPTUHIyGWyQyqc1f
U/L4TQDzG1ygCDRh+1BjX1cyaktJUEuWyXwthnxwDuCyfVswNemgBh+U2W0J
grvnDRdlwXvbBQM3rClZZ/CBVwI38NcoofBMmYX7gBUlxvvqLYfgBeHELR+u
PhFpK19LyahKhpSrzRDpj5yuK/D3Y3XO5ZaUxO/uyRPAa5vXDEnghttUN/eu
oWSPWnacEN6z8mlCI3y0qOnqtAUlEV3h01/CPQNS79+FyyMti7+AHzetS9aC
C9y2qT6Cpx10ru5eTcl4+8JD46jnku6Cq2NsH+y2l0yZUaJX/+pSNZzTMDw5
BTflcnT04J9aOjfEwjdI3Itm4cGSNTvKTCnJrriy0ZB9b6lYWcPw5u3tXt/n
3P+KEnnpUm4T+jmHb1S9CB7+TDn3AHzw8Jk4f3hp+8s+A7jbo4BxhZASIU+/
6R/MS6fNOE4I9zxQ+3MuPOqTsH8f/J6LPNQanmQmOi2EN5p7aQ4jD96LR73X
wVetiOWaw3u2rG4TwVfIh6Qu8OCN131aTCgRD/9y5w7ypmsoNdwKfxyULLgH
/67dO9AMXr/8TrIHPNlgX7wj/hNdWZ50Bnk2z7IYFME15n8kTvDWTC09BfJP
jlKeDzxxbPhQNdxickvNt/DWyj5tX7h3hXaqMd77OllR0Yf9mhpTest6xNqs
hHz4Q5m1uis86GJSzA6434tj1xLRz2M5SU8asL8uoQUS1jOtZ00m4WnV+fcd
57L13wpi68ySzdzuxlw4p3jNrbgPjr2fC1zh1zdJORWYb9nJnd3+cB8nYYgT
fEa6dOK0OSXrxQL+e9yf4yvcj7B9e12v4xWFnBiMXOBfhp/3DV9kBQ8dEe/y
QN5s/Aq9AuE/HKkdXAmXVOz1yEc+rTxkxqvgMrraXh9e12+y/wjy3L3M8dws
hyEJ12vHF7JzOe2n8EH+C9KimkbgLcteXVSDN/FjQ4qwL3cbTzx9BBdvndCV
s3ejKoubYkOJS6D15iF4TLq7ZBJunaunswt7J63y/X0aHrSxrWQErqTfetYG
+3vXftPLpajHOe8Btx/eXDdTpoN9TxM8/GkbfLvyTAO7Fy2iJc4T8Aw9cwWF
/999/g/G/5bW
      "], CompressedData["
1:eJx11Pk31WkcB3BFYxs6lSjUtIkZukdDyKj55EYoY4rrRLeyzLWmZSxt2vSd
mHpSyVSuSkTcohC5IYOUq9IiJFlGzCCaGIfJPu/7B8wPzy+v8z3n+T7vz7LQ
Z9cm0VQFBYVVOIo4yZn8M22ctLSrbXHMV0WMPr018wt7zlHlis6kt/BtYW0B
o/CV+vW8xBcc3e1szKqD/1TCqxqCG7naqT59ydHS6Lyz1XCxZl1oTTGjHK+j
OyZfcWR6Iz66Cj49oF9aDR9RU9ayeM2Rq9pF/0fw6Gpql8HnBL+OCa3lqDnW
c10ZvPOYvUYF3NA7aDy7jqOOusi9cg87uXJuK+5V7PKOa6nnaFGkNEnuv8vM
9EcKGYny04I1GziSGqXrlsMdU4VHZ8O3tvq4GrzlSNbac0vuMwcHJkzvMzrJ
Px6yopEjQX5gzEN4sPOHug1SRqf0JY28dxzZi4yzK+BnJ/nTAgsYPQxsFs9o
Qj4m4YZP4BeLwrTT8hmV+CUc7ID/Oqaq8xIu7GrOa7vLyN3pnNb5Zo7qMq9s
rIfbH/a+NT+Xkf6n2wrGLRxZlv882ASP6nefJ8xmVNynwbsO/1wwZbQdru3R
MEV8m9Efpby2cXhOv4tPDzwxun5zQyaj9uTZut+1cqSoddVxAO60ONFO5xaj
4KCmxxvhnqkkHoUbWZ7McpcweiYYyHOAh9UKfJV+kZZGHrJMuZDOqNNetEUP
bpB5dK8GPCcg5u/6NEZ2SzQFb3BvqfUcJR14rsCnUieV0e71XyrvgqeLPg8v
gKd5jul6pDAqnecy8hfeq3BsrocxPNnLolt8jdG3hS9438NDnJQ7lsM9QvJE
D5IYHXH7OBKF3FTMlfts4MuW6I1PXmH0uMVKJR35pyxZaLQWLvC35C+GL8ie
kXUP9dKz8Lm8Hi4pdHhhf5lR7XDPsjzUNyPpBzdXuMhP42ZQIqND//Dtr6Af
GvvKaQs83aGuLFbM6MBH1SMRbzjy3X6C7wvXLPIcyE1g1J3zzQcH9NXqUg9n
f7i3prGkAB5/3oA/E33o9aHPPRh+bmn+tmJ4kPPHxHfoZ0Ol3ohd8OWlQl4Z
PGCbsuHVGo6SbTIuhMq/n1o0/giuotab44W5KKz2OrYXPlnDP/MELqidpW+G
OZKYD7nJ/fGNjqg7+M9roeonpmHuCn/z6oiA20atM4zHu/YnuGzureZoolsg
kPun2aKMfcjB826gTdMzvNdfciAcLi2JeiVEbnFm26vfP+XowYMGm91wUitR
Woj8I7RXqivANy493LYTPjZidUkJ9XKxPmS07AlHWubHBwPgpqq1CrJkRibN
908fruLooIrBPR+4YfF2lVPXURen0I4uGUceE+16Qni4m/INZ/TPvmLTPTvh
QWr3dQTw+AJD2XT0m7vrSPEsuMmIROIMzz+dvfNVBiOza7WVDZUcDXpmldvJ
89wzFB13E3sp5ZFuOVy2xmP/KvjqdQ48V/T/jq+dumRwi/K4O+bwDlvP1bMw
L+fz/G2H4NYDX4jlfWj7Pnuw5g6jDaPnHK1xr/rpqUqL4AnDZ7TjcrAf1rs0
p8LDJ8Xqc+BV/86o+BHzK6yP7zHBezcZV2XL52Kf+twJTcy7yvfCsBp4l5We
miL8cupwy7N7jNbW6t2MRW5rei9pjWHuNplzf0qwNyp8n18NQs7606ykfXAK
aevzw/5x2BdAW1Gvspexjt3w+bn9AXzsqz2m7yO3or7/t5//A27h+KQ=
      "], {{-5953.040750667716, 1903.4072181252318`, 
      1275.2099720745316`}, {-5953.548727645949, 1901.7372914684206`, 
      1275.3299619250138`}, {-5954.50682627611, 1900.096513753369, 
      1273.3009794441791`}, {-5954.556837959781, 1900.6850339395276`, 
      1272.188254648147}, {-5953.926219543465, 1902.6595352212016`, 
      1272.188254648147}, {-5953.101167291996, 1904.3427622010622`, 
      1273.5300648351845`}, {-5953.040750667716, 1903.4072181252318`, 
      1275.2099720745316`}}, {{-6141.845621509259, 1068.3892622396838`, 
      1350.5889218701898`}, {-6141.766925109781, 1067.823392739741, 
      1351.394068932274}, {-6141.95099598267, 1065.8676266339144`, 
      1352.1012741068148`}, {-6142.408701257982, 1063.7385135403135`, 
      1351.6987132942284`}, {-6142.582990541804, 1063.7686968499424`, 
      1350.882694355316}, {-6142.599215818604, 1065.2070736560968`, 
      1349.6749438504391`}, {-6142.27561085175, 1067.5805067149236`, 
      1349.2723490897124`}, {-6142.0406327103565`, 1068.4231849212533`, 
      1349.6749438504391`}, {-6141.845621509259, 1068.3892622396838`, 
      1350.5889218701898`}}, CompressedData["
1:eJx11FtQU1cYBeAYwp1AFR0VlYLYDoPUxoIaJXE2WlqCiNwkVG6GS4BOARU1
KDU2CqE5IkpOBbnZaiUQVKpiK0YLMoiRmKmhihEQFK1WodGOIghy6cqTT304
L9/L2fv/19ruiVnhKUwGg8HHZ4FPcmvokJLfeDVI7OnCd6AJm+O98/w0ilg5
i2IU8JynKxzVcOfgwXUVTIpES5uy8+Hjs1Tj1fBrO9Z7H7SgCD9H2J8LX9jb
oprOpkmQy7nKQhZFmiVTFyVwDXuXyBFeFFS+ociSIqJFxtJseLI0zcMe3lSp
aj5oRZFl4XJJFjx3HveJNVy2veDTg9YU4fKrn2yBqz1SOo/gv0f9PjHm21Dk
2NpvI7aZ/9uzt80DHqhXS6W2FHGxeCDPhPM/5LW7wfclPTqSbkeRKfZE6w54
Vr3zPGJPk7LRVSfW2VNEGOdgMJ8z/u2zgj/saCLhnu5Z4EARTyFz5m74UqnW
LQ7+/M1A3iN4iE8UZx888GXxwj22NDEOlYQWsSmibuDlUfC6EZ/QKhuazOAu
KPR2pIisiB9Gw20Ec3t/t6aJ/lxr8hk4yz84txI++iWnoc+KJnNGyk7MdqKI
/XC7vQquGtB1TVnSpOv4CDMdnpHI/PcX+OMrnDXu8OJ3q+xV8Aa361EauAt7
xvS1LJp0cywzW+Fdy+UqHXxHaf/GYeZ7Zz+NftwJv7j5lH4O3CZ0ze3T8Cr3
wD39cENcRqHfNJrIv79ctRfex009bIKXtLieimfQhD8itlwGt+50/HgMXjHX
krNvSkl8tUqtDvc6dDHDyXp149WW14nD1ZNKUi6pXBoMr7vmsXYmfFJzd7h9
Qklir3jLL2Nufy1V5MyHP7eJXxkH1wzma5zg4l2U2Av+MlJ8UzauJDUR0V1h
mH93WaqrD7zhaGf2LHilbCxVjj1+YCj14cErWCtT1O+UJEFbF3MSe28UliUH
wAVi/2/48Ad6r4LLyIk8tlUTAq/5jH+4Y0xJxmq0Ne3IVRIvoyccvv5Z7s07
cMnZ+Vwdcjhe8eJdFFz/rNzqHjyb98UNHXL7dstPH8XArasGA3vgQYn0rHbk
fPl2Q0ACPHpQt78PritNqW5HL5asuiVIgvcNl9b2w01J7hMG9KhfsPihCG7X
xLsUgXPOn3Yh8h56d+NYt+9muG1HeUkbXB0zKX2JnrYs98+Ih3ttHo5bgTm4
9fK2suEz/eIyYuGKY/GjtfBg/YHXXAZFfO+lXYiEB0eZAhSYc9ZJ4YGdUwri
6PCnKhieaEwLTce+iu1icm9MKkiOYKBNAL+fw+B1w1fGLlvtBXf58bdxAq+v
vdVwCXu/P+KRrJ5QEKutgfoV8IfF97uPIidi1vVUf/i2u6+MnvDDqoa6p8hV
OrOx1TSuICThwZCr+V6NpoA25NBw7ZzmLPy1sKrInJNwOmr3zxY08R8qENJw
1cm8elt4s63vBhnrvYeJSrZMIoeOx7d2x6MXX+3+4ZUaLi2cnWzO7aasJ5an
0K9M2WlPI7y3wy/EnHOXqUN/F6CPi7QDLe445wvNkl/vwF11CxOT0V9b1hSj
EC5KWNOkNffF2Ksk6LvENKPHAXMI6/qOcRX+9X6n+jNw7WhYQAV80z+WIefh
bTzOOgXejTxDxEYB5hxrsulUm73I+fYb+OK07W9Y2Etm5Oe1FfDbZc1OHXiX
DDJRmgH+f+/zfyH/viM=
      "], CompressedData["
1:eJx11Gs01GkcB/CRJlmEoWmVddKUUEk3uydbHhw1WiJqVkwZ1iwpO1sTYmI1
Lqv179902WiGCnGwa1ldpNtW1qWVhmqNS+uyDk3ZyFI5Jfb7fzWv9sXz5vPi
eX7P72YbLgkQz2CxWBtw9HHeHnDLjg69dkfv4ocnfDlNjP/0qn82iyLnN69M
DIOrc/20tqk0yUivKXljQJFK/yUnhPAM87x7C+BtNvKTcwwp0v281iUQXupc
kDkO73Brm3b8iCLxw8lmW+HainncYXi1VdxBfyOK5Gj2z/aGuw6F338Gl2nX
8iTG8MB7fZ7wBoNAQR881DE4+7QJRX4S2bswfnEoaeYk4jzjnO91aQ5Fynlc
wUb4lmj2i9dwTZbTP42m8I7pVxvg6WrWqBn80yeDDzRmFMnkljxi3Gm413/5
EZp0jgf0tplTpDYswpS5J1lpxuOn0ORUz+2CuxyKSEq5eYx7//40OOI7mowM
cQRKC4rY8Nl/M/FMrROwCpJoEtKRcG23JUWmTpk5e8ENv4px+O0wTRI7bTKN
51JErBJ4fwEPafDv+JBIk1bXFan5cL5NQoovXJO5z84aPsRqiVzApcjltMkL
2+Ebwo0XCg/R5OWF7VWx8J+tnP2Y/I/KWU25cTRZyb8bXgm/ym6Ii4C/K2wy
6T5IkzuhjyfV8GNCy+UxTJ6jqvUWSnUuvfqxJA7+MqioOHy/7p4+I/myFPh8
B25/kUT37tGD9Qsy4N0PDmgav9HF+SClZUwBH3IrijWJ0f1r/MR1u1y4UDua
FrCXJukLq30t4Pn2xnQh/JPeFp/4aJoUTXCSxchbsrS0tILJw9K7SbeiaPwn
a1chk+cZ6V3VcAOravf+SJrY6gdHNaIu7RrjLXfgv6g52wzhmqnmZz2o4zvR
+6X3mXsWcYZWfk2T5pC0k92oe/HZDI9H8PDrAS0CMU0eipt/VaNPLt9ykXXB
2aKWzqQImuwQuxtdR1+pkppTe+E1g7RtGnxR3XuhCn3oGXa+bAB+TjR97Ad4
8ahGG4u+XZdTWvICXnf2TIMC7qk3KduJPq8ONlePMO/elq/Jht833W2wFnOh
ErS3jTNxrt2YkAd3DHUImos5ypYG73oNFyt9s+WIU8ryqpzG3K1ZsaiO8Sqp
+Hgk/pXDedz8CnPqd65phHE+X+nqgzy43ZDHjbApcq+5qozxHscaoTPy+dSo
ZWwWPO7fLHvGTzlNrLLcQ5MDHtsml82kyBHuFRcmntR1vMNv4XVlaR179Cny
2WKOYpiJsz4ytAZ1nHO6zODmDIqYrlakauFXdo3VJqLuXc8rGhfD93xQegwy
fW5XuU+IPpHPXr24XI8igT5/8HqY/aANlFl/S5Os2MD2AHhre1OKBv7CfP5A
F/pQoLUamAu3nZD7qeHrj3vmKdG3rWwJNcGiSI+Co6yHuz8JqQyK1fkOZ5no
FvyEapMNN153j8palXMJnrD5885Hh3TvKvTmlZTD30x/qa1I0MU5zbZuL4Dz
xFU798p0/5p6uCInG97heHmjHeY9Ouq9+xW4xQ3v9Uz/W4pWpeol06T0+Gir
CHk7Kko2S4M/renxuIG9oewrdFmCPNdxt/4lgyv6u2Q/Ys/Mu8lzYqEuUW0W
MVL495scoyTYS4nHNJVa+P/t5/8AyDHGpQ==
      "], CompressedData["
1:eJx11Ps31GkcB3CaqJBipDKsS9KUS7ZVQtajQUUq7aotKUYjurCbbMqZ4uxT
QuI7rIbdNt2WXbmtyrRdhpRbKnLWtUSkoSJGiaJ9f/+Afvj+8jrf85zzfJ73
+2MqDN8omqSiouKMj4NvUmRAkEIoKx3RmZVwJYMh/s+E381voaR6R31+Jzz2
rp7CLJMhdcOc1D/hLuUR0lZ46HDo5xT4e92ete6tlDA2ouEmeNkB4exk+NJ2
YYBOGyU8RwO1/+CT/TRHT8LTSzJNhuFjRSVptfC1o0H6Y/C9O3fxPz6hxHah
fWsVXNXUUzQCPx33vSmvnZJ9zNGKavgLiVYbhZ9c4pX2wzNKrKhl5D34pTPz
bsXC63cG9lzqoCRc/aCC9UmDtW3a8C3xsVlTnlOiGezoXAF33S3t/g33PW4t
2Lu/C+d7tW1j/a7SvpIPL8rVmP2mmxLjI3tErBcbBmZclTJkikX/qbAeSt4q
9Sjr+t650SvhQ/d4Dr0vKTFfKU69D6/dp35jx2mGOKcKC3f3UiIfLG+sg9+p
4JuI0xmiEclb/aSPEteNYn49XOnzeYEGvLyZr7vqNSXLR2VXG+GL1+uWL/iV
IcbrSj6de0OJoeXk40/g8r76U+5pDNEs6T6i6Kfk1qMO5jm8R1B3LCiVIa41
tk8N31KSIy6e1Qt/syaxIVbCkEOJE0qrQUqmb+ZVDLDzTMh2zWIYEuzWF754
iJLjfvnj7+F1aTt05CkMEV2O401TUmIXIMqbgHt01tu3JzPE4Hnt0dtwS07O
bfUgWWmy7+WL46cYskluvHvTMCUKs0WBGvCGNed1kuAnJvw/lsLTDwj2c+ET
6e3L8pIYoiJze6j5jpLYlc1zjeCf/aKqHpxkyPpcs9UL4WoZlkvN4DHCMrft
cCde1WEz+Ifxa3UL4GoiT6OBRIakapeZvsP576yMODZw60W+whj4+N8F4Znw
tBtFV+3goX3xHF14jULx9Vx4LtepxglurGX+6kICQ779ZHf4MO4VPPpLpAB+
nqvvsgweJ94emo/5NGnPCfGAr1hS7b+c/X9Ld+59zLOo2KVhDfx6ol2DI1xi
ppp9F/OfTr5hvOH/2lhRZ3h82UDhxQHMhzNW6ANXmByNJvBbcdfJLryjZJ2d
KuuGefHWxfCgLO7gXLy71HNmhi/8p/z10dfgLTenOsheUSKwSPJk/XVBcsRC
3OvcnWdh3sjVjJsbbFl/oS1lfocfO2R977GCkuIrtlasE3GJfAbm6UwrvX2Q
244V7lob4P9s27x/CL7KK/pC4wtK9qRMm8J681aTLBHea1delKUvemERkVjp
Cdcy+WvFarxvz6qDki706yX/4o/u8NzNRVWLkJOOOZkVUZ2UdI0Iul3g9iEp
NtORq+XWD1qM0NMkg71cRzgNiQgagLuba/s2oe9n86s1HeBR/K3NgcjnzqbX
yvNPKXns5nNsCVxe2psgQJ5PWHm1SrA3ZrVw6y3hrjNLDecj/9zqYL8U7Jlo
bWWaOTxzZtdadfTFpFAn7xr2VVi1WochPD/H3OIl3M9J6KEK15d3VOnBp3bK
ZZXonUV2/5ZQ7D29Pz4aaLE5l1a0Z6Onny7Mq5xopuSEQxiHAw9IKJDFoe9t
RsbKYnjBDBt/ti/2PoxkBP7z4MOodHhcc8KyYXiTxYecRuwN8ZDA4wxcLeYr
aT/czqjqkTv2Twhn7GwN/Ev7+X8dSObL
      "], CompressedData["
1:eJx11P8z1HkcB/BFZ32L4diyJV9GqNWlMhpE7ynp1lGRrNZ3lVPd9s2eL4c6
9UE+UpKrLUen22pnNtKeSpMkUkS7Tp2w2yrU6ZuM1TYrcU9/wP3w+eUxO5/5
7PP5er0cE/eG7dBnMBh+eAzwFKjnLbt2vK5RcqCrKOoNTaaTjxVs51FEp/E4
WgXXFrAKeXD/DeLTVCRFnk/1frwIz9ji5+L5libODr/sl26liJGJH/sCfGtz
4thSeHODCbuXT5EHu4v4FfBb+4u7OfD4vNzx6SiKrE72iJjx/WZLhgLxfvMX
/JdLYyhyIr/8r3PwsOfbh9fAVYbjkpBYiogjhcZn4OIlwUP+8K+sWdcFcRRx
82raPeP8c5pE/jBNcsYOv8iOp0iqdbaeCB76ejk79V+aWG9M4xxMoIhdwPfN
M+4+qNEreU2TYnl+375EijDG6+Vn4Yqvv/tUv6LJ7URvZfA2iuyt5HLK4H1S
bsejIeTwRTNssZ0iN+ZHXfgDXuXbMGo4SJOapKljDXCWcZpkJodZ6zJXOQ/Q
ZOcE2ReygyKBJS2tEnjeXBk74QVNfML73zyEq3TOp6XwhHAFOdRPk9qSgGLb
JIpMz03IlsGPHzXQ1T+nSdYq88cRcBPVr+PX4WU5Uz1KFU0q25y4Qnjj/bd1
d+Bxw/5H2EqaeKevyxTAX5clDbXAOWpOxdZemnR61b4PhHdoGWcfwf3OB/Rm
9NBkMd9sQB+uP60SPIF3tS+sutWN33e4LZLgO4OE8QM98M6s1lt9/9DES8h1
WQT/EGktUcMN3+vZTz6liSYv1+MkcjAYufznMFxkHOzp94QmBtFnGpTI8zP7
ndVH+GYpLyyuiyZRBrwKC3hj4Mv2T/ARo+rGnL9pMifojsIdvYhuhtdNwotH
DX8Ud9Jkja/YwQc9mqwUWTJOIM8ycwsJPHqwmbkCvTOeSWOYcCOp0K5dQZMT
GT4qJ8yJaEkP0xRe6CSzU8DZ/vad+pirZdz15VZwd1Zb5picJqNWmk/KaIr0
dfNSWHBLxWOpFl6ikoVUY27DRSYr2HDbdcnNE3DHCk5KPuY8YMFtnxnPYOVf
78f7TT3f5fKwF773M1V28KvsU1cG4XzH03e8sUfpVZ+vzrjAd15SK77/lHVK
jS32zi/AcqEjfFdDvnsHXJtSecQ4giLOd9uLHeBWlS4/1yCf+VrXlcZbKDK1
ntFpD3duSigqRZ5uIcl5rHDsl5d3ywK4U5rhlnTkL3P9csRnM0VM3c1O2cJH
99AHV6HHgfgiYVoYRVYu4o7PgceH6MT26P3ikFLeGkqRLg5nxAa+1racr/+M
Jj9Vu771hmvrJuQWcK82dU4L5ufkFd+9dzchB2aW3my4YX1vzGXMGzPm40MB
PFVXyjaEu8gtpDsxn7XqFNoD/qHRuGMK/VZaz8rnYM7lG43Mv4XrZCL1BHyt
x2o3MzVN7rXplLbwGJsNsjF4Tt85y8fYl6zYyV0EXi/47esHOONCbc8V7Jd/
anIqBa+6ZOP6Cu6pKGUKsI9Xlwd59MN7ynfI+uEPvouY/AH7m25zyDMW/6vy
wELrbnhtqcMmU9yBezUtHC38mwKHEQU8eyzy8Bv4Xf/E81Lk5jR7+cE2+Iap
Pdse4p5wmbH30pDziDD0fhM8OjhUI8b9KVReO8lHL0/dSrtvwzfxFwcdxr2q
mkheE4IeC01WXKqF37jZxIvDffsU2hIdjt7/7z7/ByTl5HU=
      "], CompressedData["
1:eJx11Ps31GkcB3AZdViXjdVg3MMSdazVOWRYz+SyTbmUUltyaUypQ5s11Byp
OGWiNn19a5hWEd1W2VjbMEJiT+WSW3SR6TYq2YpNjaaEfX//gP3h+eX1y/N5
Pp/357EX7IjYrK2lpeWHw8Lxdg/bcctecV0n/oveKSuasBK59OWPFHFNXCa6
AQ8effnA0Jomc/JcwuLhdNO9rkb4Kj+2/A38RrcqyAEeEll4Rg7/acJ/YYcN
TdqlrBNjExTRzj/rXQ0X8Oa+8rGliVIl0L0JvxsldyyHf2pRVJnb0aSxQvbX
BfisVwmbS+BVM/UaNVx3YMr8BFzEptykcOvo2ud37GkyIPG9KoXP9OdHHILP
u2WxvnI+TY65sydOwzXKxV8dgBusnjNq5UCTUnXu7atwl7Q4XzE82TPwogb+
1MQ0UwnnqOK6E+H6enWjfY40qUpc3GWAd3X0zugLmXsdRFuIE02Ckxt28eHj
InH2WviVqKf3ON/SpDnF7+Qx+NoVkdOh8OxQy3oJ3Iuv56+Ca6+xuB0Iv9+e
aTMOX5ozVkQ0FElbnrCSC+/s2b082pkmBnlLa87Bn9koKz2YOguPb2iFG7nz
3I0+UWR/4Mk6Z/jemNmuni40me7wyd4L92hWFs+H+66rjXCHJ191VUzAa/hr
Yq3hcu/6WDd4q7yBn/KZIsYpSaMmcP7RfXek8IQez8G38O9qZJcN4WfHPJ7k
w3P1qKRtkxQprL5bwrhmC0diC98W1zv1HG6RfCVWF84Ov0RbwgP6P+bFfaFQ
lzCLcWlXREw56g/RKErvwzvTQw+x4GrOu9JzcJnANy1kiiK9pnecdOBnnHPa
POHNWewQBbwo9PAA4xr7TFYL+lbJVQzZT1OkwtUsUB/uETvpq8Jc0tXadvvg
D4vn3TKAB7Vzv/kFfr2zv/gu3IQS9BjBFz06FaYF9yla32I7Q5Ho+rJdbHiR
88ud9pi712NFWyxcePGMzAaesWj481LkRKddKD0Oz28znsv0OVv2D78PeWNH
d89uhP88R+XoBleEbs1WI5/OCdYbBuFxOxKfL4Z38QJ45vAm/u4PI/CyH3ON
/eGuSenhXOTcKqMpZRTefzt+hA+/qP3voxjsi91bU68h+K/c/X4r4QMbJavf
Yb8u/JHOa4X/voart5Hpm+WMbB7c0Vicdx5ex08WbYUP10VG+mBPsz6a8HbC
5cP0gmQmt5Oi/k7su59ystYHXl2UodgD3/7QLem9JU3m6o6kjqFvDWrDHyTw
nTnVkoNwibVZ32k463W84Ch8WfMScyv4hze9BQFwp3iOUgafbVY+UsWhifjB
3j2PMcemiH2SMnj+pdQnQXDl8Z4XqfD8I7PSK5j/4cBb/UELmugVllA68MiC
wtw/4SHB1+4/hq9/MrznCPIzNPG6rYbJv5Ztqwr+2/cdb4zhf2+y82qGV3hk
hovhR8qORRUgn4a7F+y6CZ8uOOqfAf/kzmvnwF8/Yx1ifAuRykbhxYN93iXI
f7Coqa0DLozanzMODzq/4oIDXHuTpJZxuZB1U4j6Vy3ZnHoJ+/X1PXFpD/xL
Z2XaVjjXrbbJC/5CXGfKeOrCw+sewqV521vbsdfqrqG13fCcK42mIejbtVxB
uAAefS5gqhVupC545Ya5vFdljbPg//c//wfjvtJI
      "], CompressedData["
1:eJx11Gs01GkcB3CM0rq0S0fsNEy0NOWSVWynkscRi0S2lgySNiezskITlkKh
zk7R/CWb+zWiWMmt2UUjTrnsmIzrwYklylGYI1rKfueVV/viefM5z3kuv+f7
e/ROB//gryAnJ2eFQcOQt8pd3alV11SSEKkxN0CRVPeNpSl0PjHbmOWoB+8M
4LzxGqTIiXnrhiX4Z00vKy04Z6YiuhWuqbRQ7r+FTyaSL+RpwHOXvhsVwg3N
NGLF8Cc8O4ka/OgVE2Yj3OWwgciGwSfqtMX1X8C/8lP2EcAf0vR1H8GD59gb
FOFz7Ucv1MLVtdWnDXX45NSZ5cDVzXVNEp1f+6vgLY79AgredJarIIf5Yjvf
ERu4Ry9fmabLJ88XI8ZkPs39pNOFe13sMco8B0/TC5XKwwfCNONPwlMEDyrE
cOOow2wafF9woslMP0U8E1NemzL5JKOkZ0J2njueYmkUXGl1suE6PMIozVUV
rk4FjWX2UaTGwNC7D953V1i2Ee7AOuBsDOc+kjB0t/JJavHVgc1wTfY1yZFe
iozELzxlw6enjDhMuJtR0v3gHorsPKYamATfdUftLAtebjwp4UsocnJfxlIt
vDn1XLmxbN/6bxv14Ybz+p1i+JdzxXmWcKVNkaV23bjvP8Xj4/BXZQ+JDfz5
MyUG5yVFeJUs2yl4WKvKDmd43MHx1hvitfnraiz+8oDn7Xq7+EfX2vrv4vIb
f4JnFn+83COiyPf69PTH8E6FGYfz8Mmp/cH//k2RgOXfVHnwpKVm50vwlVTF
Ria821Vq6w53MPU/mQC/RcTZVZ0UySrq8tWC732fZZ4CF719fWuggyKMoNwY
Eeo5k6w6mwFf5sR8CIL7m/iNxcJ/pB1jFcnOnyUSKMD5YSrK2+G3K1wSyuEB
ntlZae0UyfXa1v0C7zsZa7a3Dn6oxUFoDHdQzWs8A78nmCdCuEtOyw5hG0Vs
nWJFi8iVZkHKcCucKS0UtsDPh3R85sHf6owpi2X9UnHK3gsexFqu2AJXUQvv
75G976rHOl94fNQmvzLkmbYrOlzm1XIsQzE8vE35tiW8TKVw06Bs3+mSBgnc
aUmuuQ39Yh1CjxuAj45q1tvjnFu39Yax4c/ZIYojcD0lt4NO8GfdJaOz6Mfo
+YIPQ/AETl+/AK4t4nby4Ht0w8174cUrvYEZqE9ZeluqKby6w/K+BL7wTdA9
VdRfKzfjT8nXfKI4PF74El5gzIi7BJe0JOomwi2LhW/a4KFstePeeEdqv6KP
NTxn4GqjrG426yO79iEPvsnMWgX4R8eL5k/gFoHvVrWRn8psDzWRNp9kCx6n
1cCd6oeyrsP1fXbHFMGvXPN8/wDeUOk6648c6tYeLuXBpRHtK/nw/BuvNtgi
t+4nLlMR8DHvFz6/w1XcnqYzkfO+nNndXHjd9rQLN2X1P106v9y9Nr+ePT8S
B2+PaGnuk6ytz6qxHeTCDQblD1ah74TtTcMlcPU7S0c48BEHhksS+lTYOR74
Eh4f6W7vDafnZX8KQF9L68wslHDf0QWt3cfhybYTjB6ZH9Ch28NNVyWe9rKc
uNFZFfg3eFL95ZvwhWiDdCt4hqtFEx3/z45qy7tD8Gifn333wD/x3iwmwFd+
ORNqiff6v//5P3mMzNI=
      "], CompressedData["
1:eJx11Gs0lHkcB3CFXQxFuWS7rdRx2S4bHV2E/2pTNltCaFfJrYgpSrqQnsQy
yZSe2Blym8k1DU4mY+02Up1U7GSSmGqdMu2qaNTM6M5+n1e96sXz5nN+5zn/
5/v//h6b8N1+URN1dHTc8OjiKR/KvFYcJGn94SeR8xsBTdzsZ/WLoygibrtc
kg9Xl/dmRQpp4rheI6/bThGznJSNXPi2tsQ/u+ARH8tEzTsoIgg2PZkNn6Y4
nCaDf1Bs+PVWNEVyIoJYx+B+G2THV56jSV7wtJrBGIqUBMUcPQSXySZyq+Ht
kj3uhrEUuXP+2LH9cKWe2rYSrsq5Lvw+jiL3FkRnJMLX325qPgcPmaoxCGbj
nI8sMxPgZrYbNwjgR7zG2Ed2UeSllBTsgrc91FeXwOO5tyeU7qbI24ajPbFw
nutAUxHcTkNfqI+nyMfkhY4xzHtmVp4shKeN7vWUJFCEt6DBNw6uunPJ2RGe
qVVHnN9DkaG9N0aYeRfRIm97+ICY3pW3lyJ/mHoks+HcOQbuTchh/glLVVgi
RbT1WYQ5J5UdHe4Ft6rkXbLaR5GRs8YO++B5SauF95D/ihblAylcz60++SDc
6HDH8ih4nkj0jUcSRfj5/rqp8DnSWR80ZTTpzVr6bzVccCr9URrzvSOTxtPh
I4XGTzTweTtC9bKY3Ap6iQW8T/XYf85+ith6VJvT8Lglv4S5lNKk0anZ1Qku
7K9Q8OBWtgdsb5R8nq8qNhpnetL3tnZxMHx40zvTl3j/XQfh+xr41uPyyYeK
aeJuPNdCCN9M76htgOf2DZ5kwVcqC7yXwTtVHTUSuFdS489FRZjXZWnq8L1G
9/qOXIff1OtXSM/SRKqf/tQUTnec0e+Efxrl6vvBta5DPgHIc7THsKMbHr/G
N1JZSBPu8VJxKvKXG10QP4Q/eO/XnwR/PdEipRD3JZdNUSjhzxpOcwzh/l4p
nDLc7ytPns0wvM5P1/dsAXp+na/how/dvrGUFv678yyHRfD2W3+v5aA/N7lL
DcbgbLcZtm18mnTdGGuMRt8kM//T6ARLWvf7DB2+Bs+YZmyzFv0UJT7MMYR7
R+ZmBMFTix+0fIs+Z9p3ZRvDm0dWvNwM/6f7t8B3OynywuAUZzL8oF1zYAh8
yWWTV13Yl7l3s3dPgfe+sqjdCtc74Sm4iP0SH9B1soC7X1Eot8HHrU9rBdjH
4tzyJ1ZwiXLV5AjGBx9b87G/H/hJf5nDk5+JY7vhnsThTTn2vSpu53Rm3oTH
9++Bs0wWvmiJRH+K7udOhT99HhS6Bjno+wTGqCIoYqpd2cmc88qqUlcJ/CtV
Zuly+KEzq/tZ8PAFNaP2yLndpEJKh1OkMWQS/TW82l0RxYdbBkxfMQU+rFSy
dOFNLoXlhrhfzuIqdmUYRQY81jkxOZ8a/yg4CA9Q2yeEwAflm+LVcDJQezUY
/Ul1eT62EC4243xi7tFPc351O1xbMWw5G/66NGDZIPy7zlUZS4s/zz9hVxxg
ejJ4M7XPAn22GVjnGQHP3xdq3gNv2tnDS4dL7Y6m18ONzWNjZPBy6/mz1fAl
rW4sa5x/S5CzrI35f24xvXoHe7R93vOwMjjFv5jfAt8UYHffHXs3I6M6gSAf
9ngdqxEuJ8KqWrjMev3Gcbigd7moAt48oU3Ewb5nqn88J0f+X/o//w/R3emb

      "], CompressedData["
1:eJx11FlQk1cUB/AQgoBlcxgtTgCLlsoiWzFlKcKVClKoMSWILLKVpSIT3CJY
C8gw2Gr4+rG2UEoitiQETCARqBB2SCuILC1lsSktoBRcwEphWBToydN96sP3
8ptv7px77v8ci0/OBsZTKRTKIfg04bPNM+V3RTR2uMxe8p2oJ9HUIP1kiy2B
MuZiw1rAGVeXp5oaSNT+UoPzzgECGapKT8rB08n1Le5dEtUEsoYrwJkeDX7V
4Ed4dd+yGknkrC/W97AjECv572IBuLdBtWq7gkQt8qlfJ8HL2wxpxeAPdd36
Z8D5dn1xX9sTyKVDZpULvtMwMK6jmUQ5zMjuUAcCcX9ZMvwSnJLQKS1tIdEB
jy6JrSOBrJPdBjLApeNO3dxWEp0IOMbf5kSgEIGEkwJOt0yVM9tItDxgzZ4D
9ysKWOCA34/mXbZqJ5EBN2hg6F0Cfbb+vDcRXJx61nMf+KNCq5xmZwLR2Db+
MeC1ikLaBrhWLm1bxUECVfQrHCLAA7JUoyvgs7FNmQSDQG7Hup1Dwa8nFw8u
gi8JxhnJ7xHIXeYbfAI8xLxvYR48Oj/e8kMXAmVrzm8LBudfiWBuBz9cXaRP
dyVQTJdnbiD4KTP7BS1wC3r/mb/Ao+tclGzwzWHT5b1wr6RM31eFbgSKWk5C
6vNv1VkGI+hDub52u6s7gQ7V3l5Tny81/ZQWCX0zN55eGQRfsh80Uddpnhjl
nAZ9vnFUJGC9D+8etkMcDn434umLUngX5o/E+UbwXfl6hZHgGyPebymaSETz
Ziy94UGgcP5pvwTwke9kE+uQh5KC0cd+4M3TUbpnwBft9Ezp4OclIg0OeEzS
ft4F8NYfdv9+CnJVlnR4IA18+vpXjFTwyaFNr4x6/H/kalx5JvhQyPrbnXfw
+U7JOp/zwIOqZN/Q5Lie3UWGIXngY2Px6ftluP7Y0el/S8EjZo2mk2rwffk9
orJb4B/vpfbmSnF/nqg0DaTgPmmrbiO3cT/dr/FD68FfRyfPr1Xj/pc8uPS8
GbztfvC6eTV+ryD9FqsecOGL/MR4MX5f7YwaryHwc6nGpjmVOA+2Px28Og5e
nx2mIRfh/DDfNJmYBJf039AbE+K81fTyVx6DO+epGH8IcT5DP1BcnFfnfE9Z
AlWI8+zo5VDyEnz+mU+ZthDnv+HR8TvL4L5mi0o9IZ6XcROFdA087sjmzzuE
eL5MWs/xNtQ5fM35fpcQz2NKlY+nRmRjx/G6i/50IZ5fSZA5mwJuMCkVbArx
vFPYflma4M8EqzepIrwf2DsddKjgllmdaEaE9wk3QslSn1OaXVZ0rxLvn8sc
M+tNqOe3/AvhVWK8rx4W3xS+At9TMFXDq8L7TbBZHroEnug0dcUf3teqPmX4
NLil6/DWP+BGfX922EhIZDz3dJ89uH9v2OwceNZRZc4TyEmF8bBSG1zMquxT
9z9e595ED+Sq1lFrdcWGQPYdW60qcOUMlfuFDP9fmp0uHQWP/ag2P06Ozy+s
Kph4AM7TNXK2qMP1/N9+/g/BBtKT
      "], CompressedData["
1:eJx11FlQk1cUB/BQECskDKgUTdLaBlllCYKpGpYPZatowyKmUKKyBBEURKgg
CsRCLJBPZYkVtSBjKciOjKASFIwidhARWWQpVrEKhAqMNIjMAP3nib704b78
5syZe+45534VHO0j/IRCoTjgqOP0lXaKYpi3mlObai/d6JYS9SmG9E3GJCFz
3J8cBp+cfqOe0yMlaG7D1mrwS9EBdkHwpotbRim9UkJQ9dvTASOS0KXPiP3g
zmJF4xD82torru1wy9y9WTx4ZfoCdefzJV8wEIZvh8865yWY9i3l2X1NMMmB
13sZFGj0SwkN9r6IOTh1MUOPDWcqfI2OwhvLCndb4T6W81dnWHBHhezurgHk
T5ZpH4WHjkdJGPCL0xfaGuCt/pXnWuAhmaWvV8HlVmpFJoNSwj4lv8DYhCSq
9RyV2vB7aVwnKZzRujfxAnz/y7W/zzNuNZdavhiP+kNKmPm+Y60xJYmDH9N+
UMJdZwTWg/B1dZNZV+DxVwbm3sO5OuZa/fDYyZAHlmYk0WvitU0BH7EznXEZ
khLVKy4/aIPfoeVlvILLj//1qgb+Sr/5wUFzkrhYrFb2Av7NMkJYBU/o5mXq
bCCJ6BKtq4MqF5S2lKviOxj8Rnh4vbe8D+6iMeJRCvezTWUGWZCENRnb1wMP
bw+8Xwx33ltm+KklSRjK82afwcdMk9yK4N/bKkYr4GeiVuv2wqWG8hO2cFrI
Z1XeViTBUJ4/1AUvmrCwtYHrWjuNjsJ9mnjNqviCJNkvzai34TCfJbImCQk3
JmAAnlP+7XYvuFy78NxyNknwku8wVXXp0W9TX+Kd23JzlWI4My2YMQzffOew
1hH4x3+yns7As7xC9ryF+zyOtKfAR0SXvwywIYm8eWbNFPxexuw2Fvqbs8r/
4A348uSxeFVf9q14PVGLOaEFv143C68w45+koL+bf+Rm9GDeBrKvr7DYSBIl
h4cfLoO7W+0sDoPnF/4c7gOP6KEe04PnR/h3Z2BuVz5Jmt4Hn9CsjzSAb80M
EjP+42PV7FxDeFhYNMOpdymPfmudvjncX8B268QeUQRGHRvh7sKpEdWch5Ky
aCX2LjCK/2YB9+zcckLbEX7aRWKUDh+Ka7eSwW3zLSM84Z03PR9VdCH/5+Ly
A3DxjjhdXzifNTbhBPfz3JWvCY9Zxp4PgLdt9fLpeiYlCu285efxni61AnYI
vCEw8X0Y3H6OpOvD+2J+PRUJP5AjqZjrRF01rpVn0Edh8eJULJzTIhKdhZ/5
7njiIvrOC2wpPqna0yHTbBb8TfxAYhC8l5cengLfT0t+vx6+7bTZifuYK0kv
zz4d/nDzW+btp1JiTfDfdV/Am46mukjgDrL5Bhlcotw5dgRzG+jlu/4svGOO
w7kLd44bMXyMOfc7YKyTDS+h6jc0wxVrLzGN4OMR3e9y4X+utNl0H65OpfFF
2KOIjXn9WXClR5KaHe4zGnAoexB7N9VVfVIVv3qY78GBm+vU73GBV2k2bVDF
e/cHvSyGB309yajEXg+30KUkPE0RzzXAu02bTlQawc1SFrtUdd0sq439Cf4k
k+Negn/jg8eO52lwYWN5/Af4DHkskgO/zAwtTYTPp3HzetAvrug6vQv/UkVI
uvsx+MoOiqsr+v5owaEzAf5///O/KJvJGw==
      "], CompressedData["
1:eJx11Hs0lHkYB/BJISSLddyiXNa2ttuhQky9Ri7TTq5LUot1yS23dLQouaUM
ufTKdKiDcQaTpVgZlENTGhklt7XJpJU6EtuSWyL7ff/qr/3jPb9zPuc5z3l+
z+95Xj2/SNdAKRqNRse3ljpbaJpnmgVtm9/PCtde5BA6tubPH9uShE7VDukw
eHwgSzoMnu5UWWdhRxIbM/WmveHvvPbY98J99Vcz+fCu3hE3V7j+6UGaxSUO
IXHKzje0J4m3A8cbbOG5FatNxfBbi1+uF8PL869pWMC7Z6pPyWRwCOVNLjwt
B5I4okTGb4P7uY0ah8Otm9Kqr8IFo64bjeD0/f3iE3DD9a/GtJkkoTczFqkJ
P1DpkPoErvmsb7IAfnnOO0sVrshorRLBk5T4s8qHSKJKkjklA5++EOe3g80h
itp9lrPgKeH35NbAK8vZ7K1w7vTC2vU/kQT5cN5suUnQFjpzVmgAf1FGqz4J
75xXP7EAl2F5LOnCc1/PuvfARU+tdD7Ce1d1m7XgVv4l3cYskvBMjMiZg2e6
Pn/PRz2+zZbMNPiUXbXnDFzhT+7dcvhsSWCACH6z+dUwFc/bE7iyFx4TlX5e
/jBJjJe7NFL5BQZ0yW742JS1lRm8olc4uAhf+kFF8Ah93vlG/NgTPsAzS/oC
Vxe+0DgK179utDUS/i66rnYd7msSli81iXfkFrgdiYcnOh6bU4An0M2Xz1/8
Gs/RfWNO9dNSuXFOFb7s+2nRH36+uShbG/5yecNqZTqHKPsxKowBV1T3CDeE
C44Z5+6Hi27fqVGDm3z4ZmA7/Ia+NL//Aoe4Pf5IKMF9g5U62s3gEV55zDC4
WK6kohDe4h57gAFv2iyMloLXXLL6hQW/X6vGOAwPKcvQK0zjENs0OK9n0P8c
w+InnvAzxs89TeABfP6+bOpd2uRHA+CGG0vVxakc4mlYuJ8u/Dcny/koeFzI
0oA/3OZfdXYN5iHfjEGegwfKN2ivpHAIZnZIiince0h2Fxt+XLzgWwDv4l6P
b6TmzSyrqwA+VtPK2wnP3lF7ig4fcuwOLoN7CRQnOpI5hLNJRagQ89wTozLA
h99jTdl1w9dVmZpaUa49FdUAb0hOYAfD7wy9M76PfRk+O+J2D36rZa9PBLwt
LVnfDu64+POHDvjQJ3muLHyCv6Qpxj4Wja70PKXqH/aNVYTPHXJXdoYzawu5
ffAJrq2lClzuH0bQe+y7hZduzF/wt0nKNHV4lOGmlnR4iKCweRj+8aoBXRvO
GqIztsMPNrkxh+DBEy4bsuDx4U6ivoMk4fNdf6KEqj9BZV8O3EFapSoXbhXA
rH8Fb18SBV2Bj2SIsw/DBxrVNlP5e4xiNNagb5+3MXq2wKdSB2mD8GLHCOtI
+MNkDZtVG5Iwlb5ytAf+5WX96DB8pct5bBy+PSd2lPK6Gx6hEvizvMl6CbzG
2L+kE/7s+xunmalf45Mad3k/gLvebU28A+eJ6lVn4Vp/fyt/F64hHRq3BfMT
tXukVA31dErsY6qo/K055xbg0WSIDwPe3f55Axdem28f/Cvm0+6lMysePitV
qnONihfKijvhY6y84Ba4ZYwJ7zKcCHrAM8VesN2N6hXQz/0yN8tS4KF/nFws
gjf+Lr8zEP5//+f/ADf373Q=
      "], CompressedData["
1:eJx11FlQk1cYBuAoii1qjQ5BiaUKRSkgRA2CYjG/QIvIUkUNa4vEUkSWCCYg
CBGDLDUCxQBREZCyubCokbIMAiJoQVsQKWTYDY3suLSIrOn7X7U3vTg3z3zz
zTnne8/R5nCdvBdTKBQLLBWstsBt7MzastopLeGho1slhJVVmSRyh5iguQxd
uQqfth8q8oC3mUsVT+GG2a7SJDjf0pDRAB/wZh/bZComWk47BorIPobTKg/h
XeHD/QL4JwEM/Th4+C2prArODfP/TgYfiZwbEMIXVbsVlcOdvfq6tpmJCbrf
cg0BfL3g7JwUXn9iPD4N3rqzuCUMznacbymBC7X3qL6Bb//cQsSHZ0l98m7D
b9QKom13iommPt/fg+A/9pooC+DsdWrPJfA/TY8a8OBd6o16ZvDQC1sc38J/
sPELOAmXCuLOMeF24RnprF1i4ilT1HkKfqWZUvqIISGKGZwKIbzUmZlP1jub
GafUwld1aHU0wF/t7kgn+08PrnU4CKc5MBvm4ef3Uenk/nV5fiK5sYRIdc6P
NjAXEwPB66ZCyPu0vcjvh1e40yyd4Q4q+bER8L2dddXBcG93u+wAeObNNRrk
vS3kDyxbCue7RlHOw111NhjHwy39vdolRphLKH1rEnxsdPnXifDNvIZcA/jo
kfueV+HJPu7cVLhB9Whk9ZZ/6zMudEelwRNiH8fXwLNnFC2n4bkJz7aTOYlV
Gg0egj86cyeIAx9Xugfkw5dSXjQNGUoInluvnIB3xsVvLYEzw24zBfCNhwJ0
1eHaPY4R5eS8ThVqq8NXermZduN+3DteP3kIt6FzV90ykBA6m/buy4Wz2u3n
6uGGDhcZRfCq91EGXvAcKsO1GS6Z+FnPGj7rSOtZA5e90td4AS9NHXtmA+ce
zkiqx3wLKp4H95BzVIjre/Ux32WaliHwllbP8pfwoOiZCTl8Ba+//lO4/Ued
8wq4WpWD5iB8daPE5hHy9mHVS/kbeJNOdqoAnqPKavKFDxebfDEJT1OtrBHC
Vxq121PhUbP099NwM9lCYBz8/Vxh1m3kP5xtvXsBHhMzcVkE3/yXto4T3MP4
fv8s/EzrsqA5OP1FbO4HvLuN6bQMJfxm7U0aBedaoUhm34Lz0pbIVR4iVyZ8
1hL4umsy2QG4/vHIBQo88YHdJR783v7GC4vhn+XRti+Bq8RcuRsKV9vp619j
IiYSuioayHrOb4yzg/BrmhTHCLhiff5Jsn9dJaE5Ale1tPe0hv/UPf6O3D9x
52i3C+ZoFHLw+lp4jieVPwNXLg7KegK3yHN6OcIUE13XRxV/w7nUBecdyE9f
5fNzbfD0qTz2azJXrnxqDnxP8zc61fAeXsmeYThr/LIVFbmdPJ347h586Bjr
CemFPe2/kv6t/FldATwr7ONMOdzLi/M24j/1X5nqjnXC0ydjU4bgpgqqUyP8
xK4N5mROor8Umh3GOzI9cOluHzxFeVzZRPbhqB2pgbskthKLcC5DhnJ3HdzK
lTauj3f9wKNZaQQvPaOVXAMP8Ulw0oOLymImvOHMx5PWv8D9Aue+T4bvb9rR
kwG3Clh9shBuN/LH2xm4jHN1th/+qkYhvQH3GT1hPwU3mS22ZWBe//c//wNo
TN3Z
      "], CompressedData["
1:eJx11Hk01WkYB3D3TjGarFkbdVwaHSrXlqS5/JoxDKab5XQxGqNku7pMXFxu
ISNT2f3iopnBKceSFgrHNtnKWJuQEQdjK3HKliUj5vv76/41f7z/fM5z3vd5
n/d5XsbZICdvuoSEBAvrE6zCB242R+uqGkpcvKJlLomIzWcer1hMkrAZi36h
D/das/QNg2cacg4nwvltzel6cAO9p8xg+P5KtUdTcKXX7AEduEPIgehAuFPd
Mt3EgCSeJPAytOBqLYk2XLg0I+dyErzRQeC0F17mPyjnA3fh60u8hu80DZLb
Ddc4pzRwBr5YUR5FGJJE8TGnTmW4x0PzvB/gKVM6H7Ph6uRVBUW4s8p3c25w
M8/H9tNw9sP7HFn4V1y2BQdelmifY2pEEhMvjjsrwGv98penLoqIp9yO6Ti4
b4Rtlgx87H6m1Tj863WLI73w8/079lDx4SbjscFwd8+xUyrGJMHiDART+0ue
6BsNgntqVVmx4WHV/dFUPk3y6gUS8PJf443j4esODGVVeK6P40KaUEQsC5IX
y+HznU1C6l5KFfanU+GjNc8v98GFTX8xPofvMI+o1IKzCpzGZuCaPfWtmnDH
pFT5ikgR4ZrcQ9BMSKI7mLZC1b+Pm7xgAw/QSv9pJ/yJ5G+lh+AqgefdhiLE
3k1LmzWBu1prfxsUId5nRedT/mF4wc+7n1G+VK2osIJz9Zbamllw2T/TO7ZF
iPNs7uLusqbyXxhszRaI75Vn8uMsG37l5agdUyCuA6/doNkFzsi/29USLiJC
ayP5bnDT0pHl7+HTdntXW8PFdV6XXD11Dr7pECvhAX/E2e47gnfh22zJ+8G/
YJc9OAN3uEaqpcKlyhsuB8M7nfmta2Eigs1Tbj8OT0y782UY/LouP3wDPk5P
EC6hT6SKje7GwM2MW5NIuLnd+4MFcJa/93wcPMM5oEUEn9CIvWcP59nGGV+D
C7r/Xb8JjxveYi6ib9NsU8KS4CO3BIZ5cL9EXlkWvNA/vjoN3tw75nsbPqjb
YWQJf+ciy8+ATz7mqRbBE6bmJuYxR6PG/1Rmw3nq5SdK4QpeTG8RPGCZrM2k
+na/pUgD97VpPqpiCy+jz4bmwF8NlVprwtf6j7Su65PERvtni1R8iw9XOhU+
2SdVVAVnKS5rU/FmB6SNSPhEW/dVPjynZjiXis9vrH9Px3spVSb6G8InC5Qb
s+AdMd1y2+GVbMmSbfCJKaODN+CczbmWEPiVCeHM34dI4sKsnigF/s6NFjkO
T+C/1bsDN6jlLlB1i0oLYTigf27X7uu4AU+v/EhLgJen05mO8Oc094AoeHZc
Ty9V/0n57JN/wJNbQo+FwLUtLPdEweu6bN/oRYp9hGM1Qr3vhwiZjcxI8T6r
zmMqQfCKwaFf6ELxuZxcj5O+cJfrMbsCheI8fd9e2PKA91Zrq72EaxZl1XTC
h259M3IaHspr3OiHWynti/wAF7FvXnKG3/PgLREXxXWrK+7QsKPmK2HwTQl8
WGHO1R9+VnVuhpoXumlyQyHc5XddQSm8rDuXRs1XeGk9UwH/1SzDXTQH/7//
+T+Zpd7t
      "], CompressedData["
1:eJx11Gk01XkYB3C6qJDl2opzj5imcmJc67WEv2WQ7SRkG26ZhKE4hokQc60p
TG78kZE9ppjErcyMveZkG/fYMidLGLpirDUUMt/7yryZF783nxfP+f2e3/d5
VPxCz/jvERAQMMGh4OhGfFXj0v6kdWPrVShPhiQUB1LKN/TYRFhhzh4HOOdR
kaKrLElo0pYokvpswvetUJ8lfJgRQbTAG6v3s+nwF5QqqjFcOOHMpJocSRTo
DTr7wAlPw99OwKsDK/QE5UmC92uBVDrcnhG1owqvubU4FAwvPOnT9xgelTHT
eAj+44sTK0Pwm5yupkW40ZG4eUl4Gt051UyBJNzPT7zSYrCJddeiChH4e2mJ
0mp4mYhZVxD8i3uzw1ttT1pfLvzEkDlIEhWKlrwHcDJGZN8afNFXsDcWzlIx
FVmGS7c1tM3BOc2bs7PwjoC0d8cM2IRw/MWMWXj28ejOabjOcDg1GK7Sod48
CleK0/B1PEQSRxiu9Fq4jFDn9T/hP8SkEHbwK+nqTivweFkvt2H4dN6gqg1c
lLsZrGvIJqp4pRK98EtBe0/WwcPuqniFwB9lTPp1wUeC1ctq4aqfm9uWw206
Zc78Abc8FehhBj9w3kt/FP7BXkSzB16pmWRlAncK477bZ8QmMofWxPh19Ig7
dGO4QbRFvR58wCdOvA/uFvFsqQfvavSWs3CHu1LUwvn3SdZ6ea4bXvJxhhsF
t+e4bvbDhSzmy33g01cU6Vnw2Rhm0ghcpmaOt4T+31hzzKqAC7dQc8f59bd+
r2PBO06/aX4I/3oilD4DH2uP3ZH7j9e+remahze7yDpWy++6UXDu36vwBzlX
lU3kd+vLDjClPsJp4U0rXORt3q2BWQBneeS47UMeuFsjORvIZ6RngkASPFW0
S0AKHmTLXrgJ9/e2L7kE37C93HQQPl742kEVHuE1PkXAF6zUVg7DzSVox59g
LjLf73dVhrs9rLOmw40OMQcnqPjfuZycVfR/yvr2DUP486WGp5Hwq1rx936B
u1xz5lrAU/N7CsXgtZp+jSw4++ycLH++TBUqE0qkSULtL9NuO3hqn7enG3w5
yPwCA77pJDdGhau6BBxlwl9TmD29UiSRyDJv6UOuut3t1gLhxWb6uhfgPlUX
Sm7Cd3Z+PhYCzxTNPx0Ar2q9lngKnmBw7l0kfMtP6um2JEkkdWf5i8Cfrbxp
jef3jWmufBsu3UnadGAuPKyjQljweu3Hj3Ph1toWnjfgh7kNxcn8/bC3vz8f
3kQaGtvCR8V0HdLgCRb63xTCD2gMO0jBlVn319Ph7qvZgnfhodEhviOY61lt
pbIMOEOBl1cCX4/dCi2GK3gPPc+CD6hn3C+DR3xg1H0Lp6Sn827Biy5rNFfA
vcNCV63hnzK3w/l1xGk6KbJ4L2eyVEcRLrj4z3wqvGds0IoFdz5oPEiBZ0u/
T74OZ9EG+hPhlam0L6ew9z4zVf0uEf7paKP4Cjzo+0C5l3CfRs2LcXD1vGRr
X/zLoFH9TDfcJpF29go8KswgoRs+V7/NaYU3cCybwuBe7sp6BtRdT2wvvRME
v7Sh7V9O3a3zKovi4c3PidLys1iZ3fv8337+F57m0Ac=
      "], CompressedData["
1:eJx11Gk0nGcUB/BBSJixNLZQoRTDEUuMCarMW6eCRGwjYq84FamtUjFCOVGN
JZZQwwwR0dCc5CSS4VgmKQmhVLXWNk2QEnsjTarWYND/+6Fn+iUfni+/8557
73vvfR7dsM+9wyUpFIo9jhTO2ImUYOdWYdtq8DcDy2w+ISmUTZ3U5xJJcUxZ
B7gm98RIiA+fcFJJjNE04BK7JLr/tIPnHk3pCIJPKPaYOsOVLtvQLOFPrts4
d8HzpVnuOXD37UqhEfyRRqynxXE+EVNW5tAOv9o3nmkA5wjmTU3hl2S2Q6QN
ucSk8m+678J54Q4mPHhsT4SADR+WePTxO/Dvd8vf3YH/XXOvnwc/YjmurEB+
X+GaK4L75VAWhuGBFT8FyMGH996ZfAMP3XQhjOhcwqyo33g33OPpYOsK3PNY
amgUXG7r8A+ih8K2M6ZWLSG+fKJy4W7tffhMOZO7DrfJtiwLhKc39Q+I4MVU
jbBVuMaMAccPXsWZW2AZcYmS/MTwDbi29BrjBeLPBy0/T4Qn85ys1uCqv08r
zcJNijYpzfCO8HjlFbj94dT1Kfhi0GbbHFwjr1SNzPv4hqvwLDyfs1KrY8wl
bmYvppJxytOaDeLh1vOviY/gvaXvy5L1f2fv+9kueHXNJCUW/klJ7RuyHjm5
6mApeBdtNPQCfDXavJiCPvxInagpwbwORg+1XYYzC7uFMvBRw/GvjODXmnsX
rsOpQfnmpIuosiI6fKqn+706OE3GcZQGn3Dr8G1miz1cR1lRBU53srD2YIvj
BOh3OpLzdcxLUp/yFudNXSra1oPr8yJTE73FdfrofeloAp8jNk7T4G5aswNh
8FB/x0wGOUeFAJ9rXuI+NCTo32DCByUio6rhG/praSpwAYO5Se7z2W1qmC08
vWez8Bn6zPR/2UXuf+F0/VCfJ5+g7WwMCOCTN2ervOCKOcXu4fCfXRZro+F0
1sxRHzjbxvBlBFygO91mBRdkCOghcPmRW6e2PfgE/3nPt/9gTzSnz+eehjep
8SJK4A8kqzyr4W71hS1RcLczlRdL4SHMBItQ+H4271Us3GSw4nY5PCuMpbQf
Pmum55UEdxdK1ljCz1mstPZhzynnlNJT4OX1I71MuNXQ7cpc+LzPPrs0uFq/
S4It3ODXIx728MWTr6O+hs9IyPI/hMvOvzLbwH3MZ92/kgk3+uDhPRZcS5Sl
IIQfavd3ugiPGf3FxxHOox/QPg8/qK3EyYNLLcVwnODGdY1bdnDnByp/ZZH9
KYi0aoArNp60V4RHMyqKc8i8ybEDTXCG+pLbE7wzVP+g9AzyPWF17TFAP+WF
d1oa4X5lrVVkHNtgCx063LbxmCgbfutTYk86XHV1fLAYfpWhdeULuGRF7D7y
vyIeq/1RAr9U98IuCN7Q2bRC9qdgXU1OCnMPVBqgesEziPEhDjzNZOtQHJxm
3jTqBPdOHqLHwaPb6GPP/ufdM4XL5ByfRhXsuHiL47QPJ0eT3tk0n0T6f3nV
06XqQuF7rS+oNniL61Q3PB7jBx+hu5ro4F6MqJ46UAwfHsww9oTH248ROWxx
H972Pv8Lum/ILg==
      "], CompressedData["
1:eJyN1Gkw3GccB3BnIijKpiqU7dpsNkTdRDL4Lxr733UNVYKISamjMsmIKhJR
1U3q6BZb2UldNRNS1i1GYtYxiKkK29JE0tEyMSJxlZAgavS7r55XnemL581n
nnn+v//vev/cheA4NRUVFXccdZxpuZmpcW9n38OQ0Mr7AinlMmcQPOwooeZK
1XPfhnMCqgMdhVLqVbr4ZRvc9drwgg7c5PSkSjU8VUdHUg43SlkY1oRX01rH
9PyklNsG/5Ni+E5XPHOvp7NP1GnzW+b/8JkJ3som3D+IWn3mR94356U7rsKl
k655wf4kHt6WOP05vGZek9XjT+IP3xZ1z8BfZ9d3HQ2AB5W0TsPtc9Ytp+Gf
H2vhcuHh4nFK1UlCKUq6wh7Bfwgw7yiBd0cp9mzgyaPRBaNwqjCUtwvfx/m5
JwIefdAtcQTeU3Ty6A6cuyDPug73HrVi9MN1Ili2cYFSSvt4YnIvnFtzPrEX
LredLzoH73HXj9yER6kecpHDhyxuJJ+FZw/FZ5xwllCsAWHBPfhlW8FYFHx7
ey8+BW5/50JjMzx2pdJ/CF675HutA15kUazWCB9Vd2kegH/rP566DH919qCi
Dr6i25TfB79Xd0Bi5yKhOhV7Hyrvz9Xn9wfBU7ddWy/BLxrtK6uHf63FYAfA
X//TUCWDr5VorN6GRwyVioRwWaepeBaeuxgtvwWPd9yI4cNvztk3cFwl1IrE
zUf5XV9fjZonyM+nwc+SQuCG+5k3a+GsyMv7JuGnHLxOF8APZY0xZfDN9hxB
AvxAybLxXfjI79t2LfDSgWLGG9RXM22x9gn8ZaGhtElZXwefr7bgbe/MO+/A
mbPGt9vhqV6n/PLhuqkzg2bHJZT7mSpPZT7LHvuVm8FNTO3YjvArQlmMsi53
04JTmtFvBuLJEA94UaE05z4828aD7wX/qe9qLg0/3LW2NQhfCrfu58Gfth1u
/wjuYKJt/QA+EsO89VBI7kfHqp+ZgDd0r4kSheT9irI/xX/ANS9Wxe8KSDyi
JXf6Kdw6yF5RLCDxjy+xChfgDNdKOUdA/veOpx5nSZmfx3UBXAHJD2dVkbAB
r/h16mQ3TfLJpQydduH6mVNNwTTJv+gb52nlPPo0D/mF0qRe7PzAei3MtYN1
Vu8Cn9RXV6b2l3IPyFS8Z5f5pB+24vTz9OBuR7RzVvmkfwqjGE7KfWLYMmy+
zif9dmXKysAUHvPlYnken/Rn90yHtzm8PyOzr5BP+jnpg+oVJlz7hVH0d3zS
/+9uSDLeg1NveR7Z45N5mf/sPFt5v/HviUFVmszXo+tjYZZwdjj9XIMm8+jx
4oYaB26VMSraT5P5TZC0arDgurr6w2k0mfdLRtMcNvx7Vv1MOk32Az+W96MF
XOWLNxXzNNknv6RfNVH+7+DWiciPBWT/WO4w2s3gD5LWl8IEZF/9137+F/BV
zaY=
      "], CompressedData["
1:eJx11FswnGcYB/AVaUf2JE7RJIZEqhRD0WkTiWTTFaJq0ThMgkaCNYgJpnS2
2VlZJFJaXd2waE1CTVQOixKkiDhlt1UqNLHoIAkbVtcxGMTqf69eN734bn4X
7/t8z/t/nv3nL34etY1Cobjh08Unr22MVbbWP3LVmVU5MySsmdGgG4keYtYB
TeLEGDyhOaz8Jnwg+g4rBB4cUbXyHN7HvVRdAr+i1jx3h9vNW00Nw9nNskkm
U8K66vvLr4fgddIw/6dwk29eFV3a4uzglAc98O9tpz6b3OL9Mv8CGVyeVMsL
0CfnM8XGVx/DeQ7pF0/pk3rWQ6uHH8IFLTtMW/RJ/ZQLDk718N2W0Q9td0pY
vwfqaDLhB9lxG1I4bRdvLA8ebrmZXQF/4KJU3IVbGKm7xPBSCsfrGXxmWlNW
Br+p06bWwGOFHgMr8HcrfVZK4P1x66fewHeWjrY7eopZi4FmroVwQ03EHNdA
wno/ylIYCqcOh32bDz/xQ3pQJNzZvir+OnypSDIihs8W/dh0Dq4s+/jMn/Aa
+TgzC97Z+gmnCz5t8laK/kkx65UD81wm/L1Zt1o5XCEVnPeEL//lWJMBHxE5
7XkMd1md56TCl+7v4QvhAaKJtTa4n3N5bBt8/KhdjwAeWSAPfwSP+IkevAbf
59ngmAa35r4U+cFzHKqT3LzErK9X64Sp8MOmbcs+8M16vbPJ8ErlvX4+vPtI
Yag3PHXhtPc9uEVjjKf23mlnK8k/6M+uDyL0huCRav9C7b0Db6+/GYRf1q94
bfCpmGWQuTc/Hb69r6EpDu6VE5PmAle1OyVqzzlA7SqPgfdLucfC4TYn/o6/
qs2Jgh+9jnfvH9vfLYQPGeYuZmv9ZODd7+CdMcV2BXCPJXteLrzZ+p15Czh7
zjRLCu9ZqeKK4IELFUpzuM+XuZMd8CN+Mcna97oRIqqqRj65T7tNe+HG1OOM
YniXjWjTfYsfrFwt+xmu7ErkKBjEmZy+ztvwkO0RlhcY5HxXvq/iDvxMUbN9
PIPUY3PLI60Gflmv6ooOg9Tva2Ju3whv8eKN5NHJ/zoKDM+2wzVUtyZbOulP
w4fbMjq1OW+Q99rTST8TBH113XCDFF9pK430f/ehxcheeBxf6dlBI+/FL322
b1Dbnx6P26dp5H17zVU92nk/Sulgh9JIHsLYQ0YT8KmpWdsFKsnPa2tD70m4
IiHSa4lK8qbKYKyq4HnHm1+sUEk+C810ytXwF9eY/DV4Bu+wIAseZrt3ZA4u
M+cWbVBJ/ksqNrIX4Zwnt1woNDIvUhfd9mX4V0KVvy6NzJdVpmpwAV52rbU4
iUbmMSW2JnMJfv2L5Y+SaWR+zXe4m8zDW/V+23hJI/MudNK0ae99kqPOV9LI
fjCKPCabgTMsixUBdLJPRqL+mNbWb2Um+zeITvbPaOr47DTc+H6sQyed7Kv/
28//Adsg5Z4=
      "], CompressedData["
1:eJx11FlQU2cUB/AoHazKqBRGUBgFxaAkSip7TPUCRsyUICOLgAgCpbIbnVob
Wg1IVYRSNJTeAi4oaoqgQRBEsUGkUIQ4OFj2pWgCGtEYRCJ7+r9P6Usf7syd
38P3nXPmfz7ryIO7o+fTaLQv8BnQqJ/HlY11dx+GqDL8lq4nCb75pCBvfQ5x
JPfO19XwnS85N1/CjXRTT6Vwd+XdqjK4icowwG8DSawammY1wt8cLTAshu8p
lszI4Ceap8/2wXninZvy4e95vk9/tSMJ37pJqzn4oMI98Rx8hUVp4jwGSYyd
Hhu025BDvAwNdzwDz2bWZCfAzbe8EXjDw8i/2oRweoFdvyeTJFpmBlhC+OgI
2/gQ3MxYJrwFv1zzRHMNvjYqc1cMfGxg5fIVG0ki+2h1WRt80Wx1Vjh81sC9
PA2eybok0MEf3KtvCYSHi1g+anjWcCqLaZdD+OSVLPSB+/qqVEGbSCIjb78m
CN5ZIPDiwiePxJ+shwt5W8pOwpMLHOme1Nzouw/VwXdrjQXlcPb+tig2XGWY
bcS0J4nFhUr7f+ALDbM8XOHNnpETG+D5OyreLWbkEL3lHtbO8G3XWUpb+Ljq
mNQVbnLbK90RXk98GkCHh3r4bQuBcxS8d5vhcVb9DTbw10H01u/he3i8wM/h
Tlyp81p44caPYRfg0QruH/ZwW+lxiTVc+LBBLYP/HVXJd4JnBrqoK1D/TGaR
TzvcJX1fA3Xv/X3rYylPMhYfH4RfnTQYoPoy2dzjuANeHJhyawTuVJOhJeCJ
2w/TujHPsr1JA1q4Qj66xAuekKtpiYeLrEKXzGfmELcHxVI+vCx3r6WOqfeK
mKQuf3iAtmRUzNSf87zLbl4ofKKzq5HO1N97I/rt41j4muvyET5DX6dB/rVk
Kidh6RncF8hhQOOdVc3wSpOtDCpX361bfelbeHRyh6wCrgyvNkiHX9H2yS8i
z2r6AnsxPMSKxzkLL40auuEAzz/ndEkAt1JWf/MblVs7WXoT9iXxfsTSXXBW
gbO3hMonz7VSY0sSCZczRRvhw5x8I2qPFB/izU7BPWdra2nwm7ReObV3Q3Mx
Qgu4KkCje4acHB6e+KkOXnXAvreMThIxJZaEBL5aPebdDHdj1XJ2wJtmt6ck
wy+YyY2ewdnB5hd715GEKT/2IR/eHZkq74WLFGydAO6fd4ZmDZfERJgOwQtj
NUpDeIJ/6TIV9qJpoPXZWzjT5Ar3vA1JaPPqfR/A0xrmPryj9r2vKugiPKev
42w2vH2ae2cCvsBv7T0HuLfl8NNIeNz5NfwpKoctX3k5wV1eqwl3eHfi9qQZ
ap5pkSEu8Cmbjymm8PjTKxfMwVmFF165wl2lU5r3eDe4lRIzHdyvx/y9m43+
nbEqCv6MOn/rQTOH3+EM8TStBh4cq1gzTXnqL5xi+CPGVOFV+KpHzNxxuKSj
fmI5+i398yPxM/zYnLGI6qtlvwXrR/hzlnZZGpyT0Rmjgvc4BB8Yhb8O/TB4
FF7fym5+AZeXV60Io+vdjSZi9FH79Wq8pfk/Hnci4pN2eNaiLyedbfXn60Zs
+p/AZZU/0Its9fX83/v8L2fa35M=
      "], CompressedData["
1:eJyF1GlQU1cUB3BWQ9nqNko31sgimzGFqBR5lnaQcYQYraIEJC8Bi4goSkkV
qVSGAQoMkEbUEReYBqwsklJQkDWQAhUIArFYKIpCdWypS6CkgO3/fbqf2n54
X37z5s655/7PcaATBNFGBgYG/viM8b1w3HFV3VTfWrCQ6vu1uIjatstMXuIt
o+bODxq2wdODyycMJUXUWc5unxy4a4p2pAF+tzXCLAE+0pJimgYPV96qqYX/
5KX3HGXcqnQ46T88293OsAp+JcCmd2s0ceNwVpcCPlTy3WRtNDmfL9XkXYaH
+rm/to8h9fBVy1MvwYNv7HthF0Pqf5u3yD0LV7cMLGTDYxQF7A74Z3Xqp7nw
Mo2N2Szcpc6mdhKudyjl5zD3Kug+roNnnikOZK2TUZa3p06dgSfdp/WRB4oo
we0Zo/fha75kXT8B59p2nuqG33se3B4Ft1ckZUnhuqzH4h/gbzhcSMuFv7lb
Op7E3Ou3dFUH3K3+5pFO+Ey+SfohePFby9q8Pi2iDi4Z2L8Iz/t8biQWvndB
bu8Bv2X9KMSHI6PCLj32PgB356047QY30Uz7x8O5bOGiGG7dbbrXGV58uE14
A67l8naK4FYeiTed4BuiWrY9gVd/YX4tEs7+VbvaAX7OvH6T/XoZ1czSvg6H
C/7iJtvCFfHlbmHw7JXGS6LgrMbfN1Sjfn6WzCYfXrmrOTwCPlsddK4CnhEh
ZXXB552ebGTOqWpySv0WXhlSoNbBH2WOUfvhH6jvBfjBg4YPfu/BlVElDsVh
Erj8QY51L/of5Vj/UQj84lCdLw0fbo7X9cBHTDqMT8Ib9oSGMH04ylmlEsKP
W1jILsLbt2+ZZvp56B2OzR/Iz4w076US7i3fOhYH3xHjbDEN951cKuiGX6bl
polwYYFy6jR8vGEgbQgemd7uybxjs5LdvoL5n19YMw7f92D7L6nwk1PzTmWS
//eXT3mrM+A1P6+z9JOQ82UPjfhMDq1ZH+r6xKSelAn3ZXK4Rv8NLRaT+tPW
rnFk8jzrXZggEZP7Nmy6Fsfk38tyo0RPk/5YvsrwL4Ofv287nkeTfsqf6bTl
8F5NbGI+Tfqv6GdzlPBXFcnHnGnyXs/6jXYy89t4591cV5q8b1AHq7IJXugy
mtssInloDwzd08rsk+F+zzYRyY/g7zxDFbxprLFXJSJ5m5tVX++Er6KuxKtF
JJ/KtX9+0gW/sJBi1S0iea5zzc38kcmVvop7R0TybxBUsb4PPjNxYrBPROZF
eFU1qoFHOwccGxCR+eriDWYMwhdV88uHRGQeS2reS+uHmy2NFQTSZH43C+tL
BuDlD3viPqbJvI8mHl7ZC18wH2bX0mQ/ULJFH+ac5Lu+m+tosk+2PHex62Hm
SDF6hC0m+0fylbcZ04ejLQmlhWKyr/5tP/8DMPfisQ==
      "], CompressedData["
1:eJx11G00lGkYB/A5XsIoRjkZSWVnC20pW6c3jEcMCkkqVsJBrby1OGSqrU5l
D7MzXoZM2NNOaVoR0xSilkpIG0pzrDEzZVdjJu805aQy7f/51Kf9cH/5nedc
576v6389dtFH9xzSo1Aobjj6OJO/m3upohseXC9v/ne4kEe4FOe9EHtxifKb
bjGDcEKYX6qB9ws5amsWlxA5GTkq4ScPPA9uLeIR8o+y+xy4mP6IKoc7XzVq
eQjXzWZG6HlziaT5GWN98P4FG9ffhy+/YDbHhptGqB71wunK0YI/4fnh0S+1
8HJxzy9SeOnxVnkjfMtCV49UHy5Rs2BtRA+8xe+Q6x142wDbbgwuDbzn/gxe
//2hylq4+YTqfowvl5CUEXZdcNG2MPoteKqAFaGAs7NS7Z7CN9adrxLDnzBY
m/13cInak3uIJ3CXKHVMNZzpNrSzHu5XuSz7MXzw82hAFdx2U1SI1U68q7Vs
og1ezVpz8DpcQDt9KQVerUyt6YBzjHoSO9C3WjvPFW3wkiX8cLJOXZietB0u
7VaXmvtxiYEUW992+PlNt8+1wS2tXnH84c3bGALyPjSZk8cQn0fssGpIOwO/
YEMbJutvdluUrILXV7zpuQbPTbRo6oTnlWs6DeDOUZUxTXDTgzHppK858Dmd
dDcH+vouuIwnlD2Huy7OOruygEe0GOgcWuHvfjzjTM6Fyjh+1jv/ax0DxaV6
0kXjCoYP3GmgaL4Qnq939xI596mRuJ64PB7xk21sDRtuxXTlKOCRBq8njsD/
VuS0seBKZYnsH/jjOj86N5dHvPDm9xrDs6pPrCbzJsvpkfDggy4TYc3o22T3
4gENXP2eEnKThz4/dOyPhzdwyjKH4R9s/HxuwY+kU/k0+FhTUuEk/M1v38X1
cnmEfVDKqWuYYwI9OH4a3p3SfqUP7l4quMGEW4Yy4rVwhsOMmRwumS6WdiEn
fXwl/z18UXLDDSXcbXmSNhjedVox8onMOSPcigLPllD2KZHD7FZvEx1ca5pb
oQe3Gcipj4Wf+yjYTYlpeDC5PyDBEG64T01TIeexoqlkPbhLfdxnI/j41A+p
ifD1904IDODSl69KqHDG2o8Fb7FftF0tdfPgut2q1Qvg4u5Q7XG4cceuOmO4
cIO42Bzu42iyhAr/4+I3xVS4hafHSwv4Ks38eULs9YejGZbk92+vsq21cEf/
dbNM+F9Lr1SbkPfR+Jq8hz+Oq5CoPLnEvdXWDqbw7UtfS2fgrwvY6wrgNhKt
O1knssahqA/9t1+WXOsN13nZ7CXr9IukgzK4x6RNqwn8kyDEnbxPY5fmSz/8
LnNK3rsd/WQevk2+65Tdi9lGzD2448xJMTxcUWVtBI/Wtaffhc9YlIxchO+9
Y22rD1f87KIpQa4CK+WRv8LtQxN0ZD/TRdQW0mMHt67kw41X1IjIucwo2x5l
5n+tUxQapJyFFz6bdg1B/mcdK881wA+71GwhPfcyJ2M/nG4dKFTCzduHjd/B
t37JTNuIPUpr9Hq6EO9a6jinewtP8td33gDnxXeyA+A3hzrjxuHHbIUSGvZa
nXl5Uymc9bDgwyicGWSmNYNHTRgeG4XPnZV/q4Y3LlE5jMFXRXGbWZjL//2f
/wMUB+G+
      "]}], Line3DBox[CompressedData["
1:eJx11HtUjGkcB/ChVDTddKLGrYuVSxrVJCr2pyLt2ITTGEXRFt2ZjJ0mi0hT
ohpnw/uwR4z0OOqoHV7Xk33XjtIele7HJcrOtmvTRVtRiv35q7/2j/ec93zO
c77P+V3O4xC1e2OMAY/Hm4CfF35f/iMe6P7ZHElh+dtgtXQmy7WF8j5sfs5A
ZdDQntot6M6hyY6zWS6gtfT201YGGgRyXkQoBf2j3PbP6MNv163VvWBgheK8
xCCEQuSEDq35HJZLOiE8PqWPgTeT73SJ0Tstflx0AM9rqjV+14YYmJ+y4qb3
BgomEzzdh2exXJ7cZ5nLCAP3PpaoqjZR+OrlAoUYvaexqrtxmAGRz9rmkjAK
4dLtqT0zWO7d6e2lh/oZUIWNdJVHULC7Fqm+KmA55+7V8pMDDKwTJxXp0JuM
3Eo80HOO6Y/nY3726kqnT9so3JD16nR26EcT6tYbENha1imKCqew+OTWWHs8
r39+5vFyEwJSM/eA77Be0daSI0q894+ZMXNj+QRsJRUv+d4Ucr9W9QkdWG7D
VW1N91wCzQF8uww3Ci62tsYh6Ezi/cDrfgTUFSOdceg//SbPL7PHe/1vt8jE
BPJSvWaPeFGwrNWb9GG9exLPH7SUEhB86FXloZc/q3bNQ5+2ZZFUG0Fg/SFt
faYrhetG7xtPYz/19bkt72QEBgtuFdYtpjAoXNn6JafAUaN0OkCg5kX1yu+X
UGj4bFOci/N90OR4qT2TQHzGtKEUT+zPpY0ddVjv4yc9HY9yCLSXrc2q8aUQ
ZLi7s9p23G2KT0hkARTyNIfP8qaz3B1+84BaRSDGq7jSCvtzY+GfolBr7E//
TLvNaQR6PY8NNeFetecIYkqtWK7QrGH0SSwBxwV//X0zmoLmjFR3xZLlMu66
pYujCXjUBGRNT6KQ7Nyn9jBnuYUDu+SGOwmMbl9WFKygcKIto31wCssZeyWG
rE4g4Pv+1kJpGoULDlcN3NBdpf2/+sZj3yT5Ryz2UxiZ4/MmEP1VkFXJLswx
FGx7VqWkUCt1rTAxZTmL184V83YQkPUO5CjwXhthWZ+VBdZ11EdgtwXn8maD
ozCRQs+x0V0N6P4J8o+tmwhEqU38vPD8HON44x70l2viqk8HE5DcvZhpuJdC
dkHZAk+L8bnzK0bT+HIKS7uOp4egLxMPtshWELD8tDgtJYVCW6VTzXrsQ1GK
QG+9lMBY8P22cvShe95Ps9BnTGxZtdKdQL5GHuiLfai535+pxJypuv5gmZBA
A9N/yOwHCmrl3kYH83HPjeOzqoMU7mZF1BabsZxkW6GkyIVAwY0w84npFJzt
vQ9eRPeYF3Eqyhnd9Obw1Fzcc00ZPxBzLJXPYk51MyB+LWqcl00hUFkfqMU5
Op2bP6uznoEKcfo39pjv5qYojrZhufKOrjXFDxnY4ST4Ngv77F5lEV46bdwv
Xo5bLkql8Ny/5XMzutHDtEnyRwxciZ/c0oV1mSQP/VuA7p5qHffxMQPS3utc
NebsEyX/EoT5Y52zdh7G90fE0zJr0CM3JZnaovPWJVy2xvdnbOLPtab7KAiF
HUav0U8ZuygkHQzs/z373IVkCpNeeVTG4t5qzz4tbNIzEDWb+HTjHirG/JeE
4v6rSuOqruH5/3sP/wMucojc
     "]], 
    Line3DBox[{{{5702.154334080212, -1748.7661715297436`, 
     2261.6512811756497`}, {5748.05343592142, -1669.9638970653423`, 
     2204.2948918623156`}, {5876.611800629145, -1744.4512159019741`, 
     1763.5581617980133`}, {5964.089535633751, -1419.0133254670218`, 
     1761.7749240725964`}, {6033.5009854608115`, -1287.9643538513594`, 
     1620.4404869206587`}, {6018.91879559411, -1224.5628690081448`, 
     1720.719617453839}, {6118.4113403183355`, -589.1360044402791, 
     1704.6334294903768`}, {5742.946045524976, -662.7886659955893, 
     2695.7551061653994`}, {5760.790470840381, -483.747091627257, 
     2695.7551061653994`}, {5603.501384715845, -854.1201810573626, 
     2925.5879131890997`}, {5667.675010042684, -863.9019201379975, 
     2796.238034023488}, {5607.854771613948, -1191.986328599088, 
     2796.238034023488}, {5723.049032038058, -1331.795228233813, 
     2482.104940759333}, {5789.346223570131, -1336.575890435246, 
     2320.527716434667}, {5702.154334080212, -1748.7661715297436`, 
     2261.6512811756497`}}}], Line3DBox[CompressedData["
1:eJx1k/831Xccx30Nx70SdZHNvYjjmOYSq1i8Jkbpki+p3XGYKzY25mvdpXPd
UY6KuqxYK3pbts9btrt73rKYeEcOcRBbUc5yO43y7azGpTXV3vsD9sPrnNd5
nNd5vV7v9+v5dEzOij5oqKenp8/CncV/efh8VazcG4NfjfZcx2UC+aqasZUC
RHVtQbnlvhgqMyXaNsYtZiqunMtB9AvPx5vz3sUgWMjnSxnPVJ4nEemI7ni6
at4ZgEFr7n/Jt4HAaFf6ir0M0WTXz3PTGBeH+sSovyfwbHu3U9oBRIuV6aUm
fhgWEmOe5DYReL1B1uiwC9GJJOPKWh8MlsLNUaVqAoFupjZ6OxENb293OO+C
QWSKX/JaCGRhmU7oj2jfl02RhiIMrzUjt2taCTw/Jhwz24KoVhoUd9EBg6T1
+M3V6wT+sfh09g8RokfS4lyjN2I4quTfre8mEL84s0fAR/SocHpNvgCDxajE
ZKiHgOrN/pEWU0Q1lVdHZVYYpD6/JIp7CcQ8uBfzPuNFgrjUfD6GvtCE0JI+
AvqqWG28GaJi762Vj0wxCL9af8f3FgF+SmzyBA/RcSdbe28TDO3Sx4merN6q
GiXarkeUd1y+u8oYA5e8R9TE5k5n9hsJnBDt7VLeWDTAIC8qHxYyPrR/DS/b
DdGGmvuSrFUO9j2M4nkxnnngsnmtF6Ir6IFR2t8cBC1NTW1k7xJV++bVBCKq
8Fg81r/MwSHr5SRtF4ErH2h+yghGdMNnqv3bFzgoSf3TrvgmgQg7fa5zN6Jn
XeKukVkOPNQGLlLGvRtbJ9+IRDR+Ut2S94SDaZtXL1Ssz0zY186HpYhWzU8a
nJzhYNVefniYEjg7uZc/l4RoXXV2SD7rvy7aJqSlg8CmsS2xYymIBj/rGdy5
xIHxoV4/x3YC8qrfMwZTEX2RmBDducJB+EiZ7Qq7o7PGxK0uA1E7++EJmY4D
obPTjhB297JvdDy1AtES68mDrs85mC/gOHkzARfN07V3ihHNG+xs07F/W93X
sfCQ6ecetZJZliEqat6Wm7MOw9JcYI5PI4H7F08UTJQjmi75MGLMDsPLG0lm
SqZPSYPOaZlxm79yWjWuGH7z/8ii+lumK90pawHj4p6fMyyZL8Ico9xzaglc
c4w8feQ0oiHydhM3pmcrT9t3li4Q+MTrlvhuBaJvGc6e2cT8cnt829wrxgOm
AyIHTiL6yH3+1wFWP75W4RFWx/RTsfSDgu3vXl9aPbUVQ/BAg+q7SwR2dTea
nVAiKnf2NMhiPl0ueDuho57pNuVMtlERotZD712/6oVBUfXjx81szxSzwsKO
QuaX//H1v3I793M=
     "]], 
    Line3DBox[{{{4392.312802496999, 
      4403.3657840073365`, -1416.2526989398507`}, {4393.435032613175, 
      4403.522343717981, -1412.2794598038197`}, {4392.402605319085, 
      4406.115802638572, -1407.3935400323403`}, {4388.89780549368, 
      4411.522592022324, -1401.3772379375212`}, {4387.609495399031, 
      4412.876345958925, -1401.1491573563708`}, {4386.168242597064, 
      4414.091636411312, -1401.833393451841}, {4384.519583263181, 
      4414.358229795699, -1406.1447766914478`}, {4381.494383974502, 
      4416.3815590560525`, -1409.2177184628108`}, {4375.0745924941775`, 
      4421.764662160136, -1412.2794598038197`}, {4373.609603647304, 
      4422.70731007255, -1413.8644823019308`}, {4373.213524427308, 
      4422.5537950765765`, -1415.5688218592056`}, {4374.354894960075, 
      4420.805792245125, -1417.501001830605}, {4375.651303902112, 
      4418.721154102372, -1419.9974362469636`}, {4375.396557726147, 
      4418.463900441491, -1421.5820106956696`}, {4373.4593250232265`, 
      4419.175598877475, -1425.3260148658246`}, {4373.526229144218, 
      4418.394804982698, -1427.5396177652897`}, {4374.235108888115, 
      4416.920955173398, -1429.9266337545766`}, {4373.835894334343, 
      4416.764528372328, -1431.6299662777653`}, {4372.565592184029, 
      4417.177591315644, -1434.2335788940554`}, {4372.449311302461, 
      4416.813423127928, -1435.7088477012278`}, {4373.676025425663, 
      4415.154077100173, -1437.0755686749742`}, {4374.83094580769, 
      4413.900220933492, -1437.4118147721824`}, {4378.424543903207, 
      4409.560667327187, -1439.7871108281931`}, {4380.240460342905, 
      4407.756830301325, -1439.7871108281931`}, {4381.7476572453215`, 
      4406.626944699671, -1438.6591425601714`}, {4382.490538885922, 
      4406.404903829524, -1437.0755686749742`}, {4386.334688044451, 
      4403.762784666106, -1433.4416733330356`}, {4387.027630311223, 
      4403.735934362639, -1431.4021382159237`}, {4386.269412882139, 
      4404.89642413499, -1430.1544742640083`}, {4384.634062477041, 
      4406.6369203129125`, -1429.8072879591853`}, {4382.21132219669, 
      4408.785798573639, -1430.6101495189168`}, {4381.342622480315, 
      4409.835602707335, -1430.0351294748464`}, {4380.863208901455, 
      4411.04650423961, -1427.7674783833504`}, {4381.780471494258, 
      4411.49268023904, -1423.5680250093149`}, {4383.581976961547, 
      4410.39569121235, -1421.4192162038012`}, {4384.020521448591, 
      4410.836918210324, -1418.6949772486198`}, {4384.836617251264, 
      4410.210656321259, -1418.1197047086275`}, {4386.920279489646, 
      4408.211326527034, -1417.8917631756144`}, {4389.906216502553, 
      4405.164424108292, -1418.1197047086275`}, {4391.44227267534, 
      4403.814880224484, -1417.5552745743103`}, {4392.312802496999, 
      4403.3657840073365`, -1416.2526989398507`}}, {{4393.889545228634, 
      4407.299685049113, -1399.020314540319}, {4395.388893693533, 
      4405.911297882661, -1398.6835948081568`}, {4396.681171754542, 
      4405.1456836847965`, -1397.0325226895425`}, {4396.325046691357, 
      4406.00371657395, -1395.446531892489}, {4394.776752603172, 
      4407.943406178886, -1394.1972287674323`}, {4393.568737553891, 
      4409.147482868577, -1394.1972287674323`}, {4392.366753707387, 
      4410.111299657646, -1394.9359539269112`}, {4391.3922289178845`, 
      4410.57982108961, -1396.5219739040103`}, {4391.177537994498, 
      4410.487355603311, -1397.4887498208645`}, {4391.843406686481, 
      4409.462700247166, -1398.6292847904544`}, {4393.889545228634, 
      4407.299685049113, -1399.020314540319}}}], 
    Line3DBox[{{{-712.1358713905541, -5690.375535363868, 
     2792.9021567557975`}, {-807.697794425535, -5833.227566723218, 
     2451.30457465009}, {-617.5120507355987, -6009.733051756049, 
     2046.851534388178}, {-124.80086895540563`, -6041.870157701286, 
     2041.5786613761948`}, {-143.96194597813334`, -6034.278675623625, 
     2062.6607847690757`}, {-34.637018810996274`, -5953.597808202178, 
     2289.3857484264436`}, {298.2974872000681, -5922.574673631553, 
     2349.88117452526}, {179.47070795339178`, -6046.997901815767, 
     2022.2315224579047`}, {91.85450647915407, -6072.085568922601, 
     1951.7052244950712`}, {-104.21839417059526`, -6071.8859453216955`, 
     1951.7052244950712`}, {-46.35014249689688, -6128.346714560566, 
     1768.9069790198773`}, {-185.4051554144633, -6126.745374857766, 
     1765.3412502982821`}, {-242.39509911875354`, -6169.367242279007, 
     1602.4874838256715`}, {-299.0735281367038, -6151.6776978980715`, 
     1659.8886155255543`}, {-242.39509911875354`, -6169.367242279007, 
     1602.4874838256715`}, {-519.0451432540754, -6096.070308066636, 
     1804.5311381000624`}, {-830.5248440846898, -6076.123677599549, 
     1754.6404839374445`}, {-1411.783578964413, -5888.046661057386, 
     2006.3866360642978`}, {-1597.6339281373005`, -5780.46558512709, 
     2172.9236389771972`}, {-1595.9543888043127`, -5671.455114192219, 
     2444.4508926462768`}, {-1910.230336910641, -5424.393051212978, 
     2759.4915792403904`}, {-2105.622847623763, -5168.2641610352875`, 
     3089.20890768543}, {-2138.4212248852814`, -5017.502267380433, 
     3307.512562605369}, {-2290.2291412659433`, -4903.961542927476, 
     3375.4740935140694`}, {-2182.768644560537, -5175.735292646819, 
     3022.432615423128}, {-1959.6554326962994`, -5528.784605730381, 
     2506.0137786019113`}, {-2179.673867582227, -5367.889426977187, 
     2668.819705548814}, {-2387.5633201862506`, -5116.259965948506, 
     2968.3731306784534`}, {-2285.6439044567423`, -5117.606523679264, 
     3045.2830904228226`}, {-2451.1973697747185`, -4786.625703947434, 
     3430.4022380355377`}, {-2244.0030511448717`, -4875.0766166620515`, 
     3447.592200090868}, {-1778.3229551446789`, -5150.0207636726955`, 
     3317.026875222087}, {-1541.4968423966843`, -5198.452726978113, 
     3359.7158243302174`}, {-1349.9997707069497`, -5394.494149462093, 
     3124.8595457252854`}, {-1045.6894478500997`, -5464.422415510501, 
     3120.0055983157035`}, {-901.99131401299, -5641.828301692168, 
     2836.1944174668965`}, {-712.1358713905541, -5690.375535363868, 
     2792.9021567557975`}}}], 
    Line3DBox[{{{-6072.036816573491, 1858.6099510731165`, 
      603.1182619918352}, {-6072.938461134207, 1855.7564533660618`, 
      602.8267769990832}, {-6073.135303739753, 1856.1642914382787`, 
      599.579348926639}, {-6073.743828355547, 1854.8435624320223`, 
      597.4989175031097}, {-6074.889703854352, 1851.0213570337635`, 
      597.7028624737034}, {-6075.616826742736, 1849.1571907199125`, 
      596.0801528396695}, {-6074.409194943295, 1854.5831433003968`, 
      591.5133024492068}, {-6072.984988344678, 1860.291362773741, 
      588.2032530763591}, {-6071.805497452176, 1864.6836901300262`, 
      586.4694594542874}, {-6071.3496868335915`, 1866.2832330780884`, 
      586.1003020961857}, {-6069.927792424196, 1870.8331768011258`, 
      586.3220184647109}, {-6069.127422507373, 1872.674492249865, 
      588.724269455345}, {-6069.567780016301, 1869.562306623733, 
      594.0517110822348}, {-6070.358262573282, 1866.5582864836688`, 
      595.4195303879592}, {-6070.663956325538, 1865.0288145503016`, 
      597.0932425233706}, {-6072.036816573491, 1858.6099510731165`, 
      603.1182619918352}}, CompressedData["
1:eJx11Ps3FGgYB/AZ90sMzTS0GJlh3WpaJ6Sd2FdbkqmtaKNS2dDFmaKVTMpJ
Sa9V0+V0WpuYbVasiqzRnnGpOKN16R5pw1aYXJcts5JQ9jt/wP7w/vI573kv
3+d9Xqdt8aGxegwGwx9DH4M/en3o0EJVLSfLOkT9uYJ09rQOphhT8n3RqzcS
eOo19jMtfBfH8+VXppREjFWmR8JLROUsgauC/LKmPcPZnJKbjFGbJfDRKGVg
JjyxVV7HtaAkW8W0E8H/zGmxSYcrCyTZXBYlTHeh0AM+LRwdUcIZ6tIDjtaU
TAZ+ayWA+2wsLy2BN/uFbXJjU7LleseIHTz97qSkCG49tX6x1yxKOs2yBjnw
kJIoUT784Y2ktQtsKKkYG/CyhJ92GPpaDt/4cHctbzbma6d3WMNtN7xbHgOf
VLc6Mu0oOTu51X6W7jw5CVc84N6GqiU19pQ8CEh2nQ1vYTuIR5DDftXgwRge
JaJiTpornKfIFabDJYlN0h5HSlT8lQlz4b31Tw+EwN07Cg1WOFEi0MYM63Lo
PV7t88JFQXKyJL77+ZSsUVczVsKnYuuiEuBxW/+Okgoo8RCGVOly9gouNTKA
L024ORXijPmr298nwzdomrjRzgpSGJp8dAD+tOiiQxb81vlnTe8FCvKhwEMb
5kJJu2UuMw++unWbqQyu3CIwyYT3ut1Jq4JPVMy6M8xXEFP10OIjcImW3/YQ
3s1J5R6Dy7p8NSFwUbnV/WG4tjZF0+OkIFXl4qI27GsuLNtn4aeq7Xsl8Q2G
G0aHh3vDO877sZzhXuOeCzrnKIhmpqw4AvfiFjgREZx5VSWXwm8fkk2sRQ4G
quTuSPjhZuH2vxwV5OOLhko75FayIi08CT68JLY2CW481jmvAjn7O+svOAWP
8R2PZcFnj7z/YiHqssJmpCsP7ho4Jn/JU5B7AZX9majjYxenqyp4WtklrRR+
lZU5UfcZJav8Qy0b4QEub38LhosrLS922VLSNyP6ynN4kN+ZbFv43s01UW+4
lEgfLzs4AH9km1TY74DzsE+KhzmUtMkXxUzAf630qa+AR6Qkxr2eSQlLLTvK
XKSqNf1hg+Ft+ND86swOK0pqPAMvmcCDu42W1sHbAqMLmy2x/uEreSx4rV5f
XCM89FNac/MM9MU6/zAufPOH1sgHuvWlE1v7zPDe9p/kseEpBwLDjXFOXwlj
ygB9WnImP063zqTUZJTAb8wdaQlCX3vKrHaYw8uW+6xPgQ/VfLn2Z0NK9j36
ZP0R5w/dbZnUD39isfMbngElAfnEdhx+cdCQ74Scj+gtYtzSo2RAua6gH66M
S9pzCm4pN8uQMSlp4Dw6ocutYLE3mYSPGwkvyBiU2DNKWhrgCy/b5+1A3XP7
/xmvmj5O6F56oRi+ervxORO8H+/p6nh7uNUW9b4ceHvYfE0i/KeW/I9ln46T
oOnsP1LhLHPbejbe5+/hKb2pcC+vu5w4+Ns9c7SpcBfN87JjcP+pvHExvKus
vMEe7/+1u15TBTx+0+EWX7h7Y29nBrz4XrKRB/a98K6JbwPvbqw6IUB/zdn5
WP4ErpzPO8uAiy/Ls0/AtexdZtdwL7fNbVmvdP/e4L/L3NCn4QWxhjeQQ3xP
e3kDXNS06rvTcH3/PvEgcus8ZxpUCj+jz7UYg8drZuSIkXP93OadP8KNi7dt
j8S/MS+iNfc+6vJ///N/Yjur8Q==
      "], CompressedData["
1:eJx11P9XzGkUB/DZ2sRkGvomFRpsvqT1vZVUT0lEHbblTLZmk8SMEW2hlDlr
wyMSjb4Qew7Tptg2Glkjq2mqbQkd1KSc6MtoUImaakaTmn3PH7A/fH55nefc
z33uc+/lbN8XEm3CYDC88Jnii9+QrPBdKVNor1aldDpIiPUCzbqODkquM4Xz
lsHrZE+a7BwlxKdUNEuvouTDsSfWi+BXUhWN1nAb/tptczopcWx78ZYDTzoT
cH8NvHXFlNVCNSU9llYbpsGd9Nu5fvDMDAGj4i0lf90eSrCH+80sr/WGR48s
uzDzPSUW8WfyreEtM9o9PeFWJ+4spF2UjFtnt9t4/rr6FN8EznZu4Om6kWf1
/IXG+BOr9Vm1yN9geSWN/4GSQuEei5lwG2l2fQa84mFxb30vJY/L28Zc4OzA
nzih8KCLjGPunyihOu+Xy+FMKX9H91TkU1/HFPchH94MlT/8WWB77WH412Mi
/et+SqYdFXI2wzP02Vw2fNnoerbjACW6KOWNKPjaUO30PHsJsS1QinwHKdGk
PuaL4Kbv2JYMeFmFPGn9ECUqqUqaDs8yVUdkTpEQs5GS9JVaShYLSr2vwRvM
U7r77SRkvP+dqHE6Sl4neEythk8Lj7+xCS5ILcj8G84VfHmkhL/aOzZBbSsh
zU5nElZ/puRhr76wD+4V7n/aH949jtVdDN88Ys5kecoUjvcOxObbSIj8+OeW
fvitd7cXOMMZSw+0eMDjlmwJZg9TkqbdKfaCf8hrEV20lhCOXf+v4+GbeEuT
w+DOkdYpI1YSwr9mWN6COGyrHJdYuDy0hHUOntQsEpyAp7rH7DoOH7trZT4P
butRlzUevvrsKrtL8JQn336vmCwhvovfMHfjXlF+X5ZI4SP2n3u48Kc9dx4X
oT4fj8uYcji3fv2Eb+BZJ37fWIt6muUUDyjhFXEdHuWTJER/ROj2DPVfwXY/
ooI37X/RcgoeLbyUW4b32lAZHNYH97LZLQuFb718NuK0hpKwf9IeDcLrSlyH
w+C2iTGeAXj3HIaSPwxXHjrEi4CXRSw5okH/KMl09zG4Kz+4czv8ls8LN/FH
SpKdDEzTVTLFrIm3Y3ca87GIPO+GPmQP7Qsynq9SiQIuw33z5YL7PZRoR4Oy
jfFLEhs5zfDMYyO6dejzZ7s85MZ8+qc/V0/CfU1UbtmtmJfSRZ4PPsET/fef
C4TLUsJ8Re9Qz3TL0XZ4QfrgVhk81lm93xlzx01yHXgOPz/UqJmN+rN65P82
Yn51Ym5ZFVyTUcQ3vlfNg6CbF95QstFs7qJSeHZNWL4BLrVXe8VhDwQ6aZNz
4WJh9j0J+qFTxvYWYG/UzI1adRIe054ea4n++eE7XufBdko6nNqke+GWjNyu
m/B9lYUWRW2UiO2Lf+PB97AfmjihP20M818OtVIyuTFhth+8gRFSdx/+1Ssy
EAN/dXKOlwv8ptQngIf+9x0brDKDdw2ymibA4z+ORo7Cyyp1TytfI87lnLta
9P+O4UlfdmG+Qg4EO+TBPfdeWfMSHl7NCjHHPDIr8uYVGc9v6XKogf9pKHf5
GV6fEHSwGd405fkfBfC+65FVxnkXKtYy5uO/VzV+4WK4q9RVkQjP+/GuQxE8
4IJVYTJcw9kc1wHfltRbEIj7/lKxIi3aGCfqaXMg9s/oQGu+GeqTy9opDYVn
dkWUHYbPMDTMeg//v/38H6rz2JE=
      "], CompressedData["
1:eJx11Ps7FFgYB/ARuaSZMWKnRq7jfslGLok6LkvKENVspRSty9OWRIhSnm6n
eEjZbhoxo3IpIWU2tRGPUEuxhFZGma2mlN0sq61lv/6Afji/fJ7znOec9/2+
xzhiZ0jkDAaD4YGljFVxeUfmTTdpvefVix3p2mLy1nd/a44lJfPaXUsuwc0a
ghKr4JHb294EWlMSI+12LIFfbE2rS4af7X7aOceWknrf9CExfISZq5cAT1Y5
TWV2lESG5pZdgGsd6D8WC1eaVRZ3yZ6Swd3pAQXwDv+qoUXwEyF62RELKTH9
cItZBJdPSdZ+5ohJ16STiOtIyVm23eQV+JEzO97ehzs3nJxTt4iSPt8So1tw
7/PzylfDVf448NTXmRKfPUkVt+FdWodreHDPfY9m3HehZM3W/2gLXBIrmLqm
JSbBGmUVOosp0fba39wD79v7LncZXL9FY3iFGyWN8xWPnsMzDa0r1eCJu8xK
Nyyh5PfR8nkf4AYNOZoX2GIyIGK/9nanpKiL26K+RFpfO7hC+oklJmkdt59N
wY8U96zkwc95nxrPgss7XjfneFAyXtmYZAsvFJZWmkzvv7PE4iP8s0BBlsFV
hZUyKVNMJp3j3c2XUpJRHSwJgW/5pilNAC+Wea+0h7Nq3zdEwa9lLCgZmi0m
bVwbpRlwbYfWg3vhD4xj/VLh4y/Gthfj/MfhAy9y4C6jiaEcOOOjKNsQnqHg
D16BM7Y5yUs0xURHqBu1DfeX3uXdq4aLu7vUkuDM5FWjFHVYZyN7+RB+fcuP
MSz4MH+T0gHU7WhtfnMv/DxbEtA/C+9dJfstGHV+713p+BreKZv5Uxm8lMO0
mERfxjdZjYzB7e9m6abATaOfdOWij0Lj9oUz3aX1u1kG//rBRxS6yspOlDgJ
nhTpwPOZgUpcOC27mvcDcrK2Nz6bB3/orLNODy6Yav++BrkaeBNbaQRPqNvV
Ywi37Upwm0AOHXueKczh0eunok3hZ0hisOMCSmKtq9zs4L5KtzWs4NzeNlE0
ch6btvqmFXxQ2F8fAQ9//uyGBHPRIdyjO33O5XKrKBH8dLs4V445aoz8VmgC
71/fPtoNtwmXH7S3oCQ5PC/KAK6I4YexUbcbYXz1Y2aU9Dvx1abvP1ByOGk5
XM11jSfDlBKelVyqCTf77tqpGviKl/laR00ocY87WqgK/zMgynQEHnTIZbOD
MSW/9O4P/wf13KA7kbMR/f3Lw8PgiyElgcbX86br38Hm5LXCrVrubBw2wLwv
tgzogS8slC5zRq6qipt4yvCk69ZhD+Bm7IQ4CTzxaSnfT58SkYe09yZ8t5Dn
wEJu5x6SOtbMpyT1zsRVCVylQCUlBe5zL6VzDdzviUJ2As47rrxSPu2XXO8a
wb2+pO/bB98peFUeiDna4e9azoX3nZm9IQbex0s7/DO8NXfxhA9cVyI4txqe
3Md6ZYx5LKau2QVwsnWpy1J4enFWZwbcdshupwXuaZna7WwJb//7jf8o3K2R
daEfbvpi5lkOPP4gMyAU/wOj6FdOHd6b/LHQQQXexuluyoLfH5ir3oK6hXqW
ZQ5j3k9GbFYdg2d/MnEbM6LEzCDEaAAuel57zxz/mH/e6XfL0RfPCr3Ix3B9
ToD7OviEk41XNR/5nx1kUg9vDtMXHodzVRlzfdB3EfttfC3cXJpqeQSuHjd4
6KU5JV/7n/8HhcWcwA==
      "], CompressedData["
1:eJx11Gs01GkcB3D3W0ZkT1uRcWlq5ForbI565KRMzK5bWeN2yiWk0rjFJpl5
IhPl0mJpd4RxqS3CIWdOSLXYjmxLspaVLjua04UVFWm/Xu2+2RfPm8/5n///
/3x/3+cx2X/EO1xJQUHBGUsZa9g+eiY2rKVj1t03ZrWfmPyho131YQ0ljIni
x3FwmSnXeNxXTNZrjkXVGlLSb2bflwAXuywOSOCfPDys/Y0ouTnv0nYcTmvD
zsTC4x/snVYzpkSt25adCU8MOvto0UdMeIJSgxoTSvJmjdxEcM4yzfo78I2x
axV2mVEykzOsVAL3ij5rvw8+N+rjeH8dJSH6eS8k8GPtLiMybzGRmI0Nuq+n
hFmkNnEZvj3yDace/tJQ0ly5Ae+50p3TBjcabE/eAW+aNzacYFOy8Gr6VA88
co/T8ICXmOzVzXyosZESc3bSi0dwyxOqokh49rV8zjILSlaof5TL4Df0aqo/
fC0mE+ciTMfgmRe0Pd7Du0Q+arnwbdasUwWWlGzpljtrhbd0aD651WsCT8xg
ZxlYUdKtWN1pAA9783am+Ssx4dMUh0T45XduQ1bwwwuTQg48eNTkXAVcpaU/
jcDz2ktS/uT+64fSh7gc+LPfGB51cO80scoJOOUcWxMMnxvbzSBwfj/Xz3Tp
Pa0i9Sh4Oiu0ahncXGUwpAL/qXvSIjcenlgZUzPkKSbjvO19M9hXffS0JB0u
veaqVwlPXffk6Sp4hfn3Rjlw9wpz6VH4ZFF4BhO59WWT1vPwGiveNj68+tr5
YUVzPL8wqvUDPE675RSB9zob8G9jLva3fz16CX7GyTHfFd6xOMONxBxzx11X
VcNLvXj9bnBpoHLXW8xdyyNB4wpceYO1CwceoTOYHYueZBWXPVp6vvnqBZ4A
/r7yLH8QvWLY8UYq4BtD/Uuk8LhG5gkr9NAuRuBWCy/v+TukA65uYHQjDb31
SZjTKoMLxdmbbZCPsKt15j56/tMRuUkx/HpUc9pB+HDTaxsTA0oehrmoFi7l
rKVUXQ6Xl5W5Z6ymhC2qKj0N/2SuHqCLObL2CY0ef07Jhe+e2WXABRenuO7w
e8zCJ/4rKREyRz2X8v9x4HFdC7wh+pcdY59RUhmkpBABj+R7tbPQH16wIESg
T8kU57r3PrjS3cC8AjjH+cDbnSsouer5kLcb/orlZquI3poWf+nA1KOk3PK0
zBHex81MPgw/z/2kzNSlxNbe7QUbPhK1GDIC71tZ+tR9OSWTOiK6ailP9RUK
u3FegkoUBYU6lDQHtOSqw7Ulnlub4Bsmbx3XhjumDqnOov+h6xp0jHFOd03J
uhsYlIyvdXB5CR9IvvPOD064KUeF8D35+YW/wxv6ixU7/+NZcbdjfl46R/2d
jRa4N0odqubr4PW1i1NSODup9VAofI3w5BVVfNcvef9CA/zBSfv7hfDeu6my
03BG6A7WRbhdwYjvInxh9KDjF9hXfYO+dh78nm+9fBPut9xNeqzlyOF1dIA8
B/4mWCy1gV/SZ4g0kJuq7oG5dLhDU2nbfnj8nQ+pbOQc5jM2/y3cO/PmdAjc
UM58Ho65KBVGlSfDDxwukgXCq/mxPR2Yo+1ffR/5cNutQZe/gUeaFflvxtyd
Aps1j8CfF7zashcu2f5uSIqeqDHSJ6PgvRcby73hqaKkKV/06v/u538AXDW2
6Q==
      "]}], 
    Line3DBox[{{{3795.6683430248236`, 1906.2562889890423`, 
     4758.876178306062}, {3747.6859097524953`, 2016.5149551407083`, 
     4751.455718780979}, {3791.8836242383827`, 2157.008613251647, 
     4653.680920797106}, {3943.7719328260596`, 2111.678833609084, 
     4547.008811904897}, {3795.6683430248236`, 1906.2562889890423`, 
     4758.876178306062}}}], 
    Line3DBox[{{{4568.846232764547, 597.4443066509062, 4410.954328918362}, {
     4572.082436259511, 592.457482812051, 4408.272906378109}, {
     4570.810560434734, 592.292671184067, 4409.613804210064}, {
     4569.019828967605, 596.1152578966723, 4410.954328918362}}}], 
    Line3DBox[{{{156.470129982885, 4167.8464101744285`, 
     4826.218715476876}, {-68.39487971033321, 4353.795154586398, 
     4661.287760086198}, {-54.79843755896722, 4485.08997190853, 
     4535.281553015936}, {-429.8660969113038, 4547.508300449811, 
     4452.3249079628395`}, {-520.908993253993, 4656.234965034258, 
     4328.5036053051335`}, {-1252.0579121675673`, 4592.495247562485, 
     4246.074902270871}, {-1614.867442807072, 4404.890392932556, 
     4321.684562576521}, {-1725.0443726949684`, 4276.79660048801, 
     4406.931635535957}, {-1686.6822904014641`, 4174.685162862751, 
     4518.287322893451}, {-2182.045106295764, 3794.6927433215888`, 
     4639.698247502005}, {-2032.2352899335503`, 3753.31976415287, 
     4740.2978982726545`}, {-1847.2657142856053`, 3858.4208806738557`, 
     4731.597127652023}, {-1849.722868454352, 3675.1009359255586`, 
     4874.420181998684}, {-1322.387977097877, 3940.7690607482987`, 
     4838.33059410743}, {-940.4644811476066, 3973.2818588647024`, 
     4900.650111946007}, {-607.7777921728785, 3866.469719874063, 
     5036.748132539332}, {-526.2469879846041, 4089.123584415541, 
     4867.231873356121}, {-167.41176243775774`, 4022.288565593636, 
     4947.825955598295}, {156.470129982885, 4167.8464101744285`, 
     4826.218715476876}}}], Line3DBox[CompressedData["
1:eJx11GlMlEcABmDUleKKBouAUkVgEYVFaEXlENcXy1FKCyko3YriAYIn68kh
FEE5VjSRS+cDaxGUo5wzSwUUtYiVqiwrgqGooKh4AJWUqsFw2ekvf/XHJF+e
zPfOO5nJmG1W+G2ZpKWlNYEPez7++zYa+ClpTQhDinbWuQoHgua+0S3+jyke
u4zv1ZMz9Hz+QuTvRdB2RWyws4/irfX80SdeDA2NDyyPBhHoza66ajREIbyR
Dsa6MdwpKm1VbCcoroiZc2KEwsGj8WyaM4P8F6vC0FSe478s5dFEhlRfv3rR
EganQ8XxizIJKhdKZtfpMBSdeu2ULmXoYmmrfU8RJHln5g3r8R6uw7dsFzDE
qFQtS88QzNXXLL89k0Hpsc73qIShuvKA54wyAslcQ11rI4aAkjO1d80YZA21
saUXCSyUCeO/cq+OmBwaZsrg6Fqdda+R4O2FkkMj3H3mzcqVc88yeqVXz12n
aJOh7iyG7H5Zu5sJg69B+PFZrQRRL13HJLMZLpVFDgmfMXy4dH/amk6CjZXi
8X5jhtybDxev43lxFYOptk8I/GWnVXX8/56SMNMD+gzJ2Ur55ccEZopJe+rN
GbTL39WD7+/L4Oc5AS8J7OYkPfOcz/udS5ksETMoevJ+eztMsDZ+mrTQisHB
ZvhBpTaD2QnRoiZtAbo3/iFTbBi67xwMMeZ9al73nv5qWzbCd0UcS+XnkuQ4
llLgxNDyR1DYEo9sbN3wjLQ9pNg82vco3YVhqOZ7hxWWHz0mofmqfBVDplvY
BPf3AnoOWDfbPeDnGDQQrOK+LCfuzbtuATudFJEjfH6uT65ZGxi+NpHuC7wp
oLzb3tuqm6K0LEQ2R8awqnB9ge5FAW9ogSygh8JGM72pdgXD9FLLDsNyAcet
vo1ve07hvcf4QhrPCb00UX9tjgBJ/+WSPD5f5D7y3oDftxdjO17+cErAw6mL
J3l2UmiCKlp1NzAUjGmtXhvK3UX+ZPJtCmVUg/71MIb2zYGebiEC3ETipbbX
KNRxGUPW4Qz5NZpC/wQBJfGRAZ01FCaHncSd+xmedk1J1igFNJ10LbGjFPYG
cce6YxnW7x4XOacIUNte6HpdRBF+q/dIP/cM/Xa1aaIAn56hNdJiisCNvnNj
oxkGw2JkFXYChhSNNdFVFGW/N0o/7GW41ntda0BHQFqxo+vVKxS7ZtZ5Re1m
cD7b7HNeTeA84++I+hsUj6ZaTGG858pPqlrvHSHwq/vx/kQNhcsmO3vJLobF
29xrT0YTjNpnHEpsoShXmLd/upPhqFcCecpduXFwnNylKAhPtPhuO4OasL9e
JRNUryKJHq0U8mBNx/6tDDrLbPRMMwgEZbqXYxtFVp/v8P1Qfo8GsC+R+8Ku
mL0d9yi+UImapNzFfvnno3jOjfQkw5x2iqru2OIN3CO9PewaDhJcvGWsu/1P
CgtVjXEt963v8qNbdhAsiLDMDOygKFKbLNfn6yZeMW0uDfzoQdlubcG8p7kq
f5Paj+DnvPlVdTxfvL+3d94WhqUmDt+0riQ4U5i67k4Xxf+9V/8CFm1T9g==

     "]], 
    Line3DBox[{{{2847.105650472892, -5404.634122884319, 
     1836.2279550207743`}, {2848.959071263917, -5404.880684276066, 
     1832.6239900761236`}, {2848.9040460944175`, -5406.08002917233, 
     1829.168703080982}, {2847.536186964166, -5406.984634130094, 
     1828.624762003028}, {2844.7521568387174`, -5408.2443179821785`, 
     1829.2326951962536`}, {2843.3826707897347`, -5408.275146794765, 
     1831.2696726628483`}, {2842.9092577504157`, -5407.374688336949, 
     1834.6606285986152`}, {2843.7751744348416`, -5406.180833135446, 
     1836.8356613227586`}, {2844.8318377203805`, -5405.556036257647, 
     1837.038226050562}, {2845.6521321984014`, -5405.145993057909, 
     1836.9742584600133`}, {2847.105650472892, -5404.634122884319, 
     1836.2279550207743`}}}], 
    Line3DBox[{{{5214.884961117393, -200.335672437274, 3667.786001208461}, {
     5142.856196043725, -487.65244754251347`, 3741.796216687978}, {
     5193.175825959435, -567.2165484217793, 3660.1917870334355`}, {
     5306.327134634641, -867.359527991227, 3431.9664158384408`}, {
     5473.285834356899, -988.0872190605842, 3123.2418274858614`}, {
     5500.496474752598, -1288.4418644122163`, 2961.8017798739006`}, {
     5584.878127433277, -851.2815095214812, 2961.8017798739006`}, {
     5530.285249604942, -842.9601269619327, 3064.8290282419234`}, {
     5458.52984935199, -346.6099201752429, 3282.091884297186}, {
     5415.0323870509765`, -361.24820301712793`, 3351.8260206946743`}, {
     5402.384204307434, -111.59164721028046`, 3389.6321191719953`}, {
     5214.884961117393, -200.335672437274, 3667.786001208461}}}], 
    Line3DBox[{{{4774.174514835812, 4067.9326522255, -1160.601279014394}, {
     4667.218848492703, 4009.7197570068447`, -1681.3772273179554`}, {
     4947.781856257699, 3408.9994014694576`, -2141.4928142928716`}, {
     4740.346656379623, 3381.2564094243326`, -2604.61750763474}, {
     4834.468805271377, 3117.5932677272826`, -2756.1453658832042`}, {
     4778.297805897626, 3091.219481637884, -2880.980021537176}, {
     4819.941436462129, 3027.452553688502, -2879.324447546743}, {
     4865.811372463408, 3036.5616898874646`, -2791.233863565279}, {
     5038.482652813411, 3063.445470629743, -2432.4489914027895`}, {
     5024.040893392242, 3266.8074062653814`, -2185.1306967529717`}, {
     5124.514247051772, 3325.779546231515, -1834.76369912627}, {
     5297.280772589931, 3109.967843157007, -1718.932843620825}, {
     5177.561106107444, 3390.252482639379, -1544.947502823634}, {
     5085.455727668312, 3499.4947376055616`, -1606.0791740747202`}, {
     4975.892863550718, 3520.955925719387, -1879.1416310663117`}, {
     4992.6119381991, 3616.2620792618964`, -1638.379778514202}, {
     5110.707977330344, 3521.254764651633, -1472.8348661238817`}, {
     5122.139654449121, 3579.9033991289016`, -1278.9803373098352`}, {
     4774.174514835812, 4067.9326522255, -1160.601279014394}}}], 
    Line3DBox[{{{-954.5605276459905, 6201.879333393575, 
     1146.0020170513972`}, {-822.7043788282896, 6058.30583185597, 
     1818.763767716053}, {-729.292880273605, 6041.285633150468, 
     1912.8025207720675`}, {-605.5905603390669, 6103.160480062596, 
     1752.8565023498948`}, {-454.47934447264834`, 6115.758251237166, 
     1754.6404839374445`}, {-428.4650214161634, 6001.876130141343, 
     2117.0059834262192`}, {-228.39570840319598`, 5945.308343967076, 
     2299.7734204782973`}, {-268.31855829859563`, 5908.83348127847, 
     2387.7825972636456`}, {-510.2429289906438, 5679.838033109517, 
     2857.7798657450267`}, {-705.2417189404638, 5635.287288843637, 
     2904.132397130549}, {-719.0561416406115, 5574.509055074211, 
     3015.894684843047}, {-797.4393530567565, 5568.262335633742, 
     3007.7165308185363`}, {-869.8313631496421, 5673.335588118219, 
     2782.888857693123}, {-764.3627259519584, 5779.924962762611, 
     2587.6689776725543`}, {-900.831449426935, 5752.589332522004, 
     2604.61750763474}, {-1147.1448856779077`, 5820.157156086013, 
     2344.7053076503635`}, {-1047.0817357882042`, 5888.2035456648155`, 
     2218.2183241058697`}, {-1147.1448856779077`, 5820.157156086013, 
     2344.7053076503635`}, {-1047.0817357882042`, 5888.2035456648155`, 
     2218.2183241058697`}, {-773.6324556671133, 5997.613927023648, 
     2029.2692517037854`}, {-950.2983703433443, 6045.465828295808, 
     1799.1913784337844`}, {-876.6014761483817, 6095.752845747724, 
     1661.680109278861}, {-1048.3226701210965`, 6154.497141694534, 
     1308.0512995230677`}, {-954.5605276459905, 6201.879333393575, 
     1146.0020170513972`}}}], 
    Line3DBox[{{{5963.77411688758, 1242.276755635775, -1891.5497442966744`}, {
     5696.99026273226, 1476.8784424336702`, -2459.867009130474}, {
     5680.248101696379, 1463.7247468810708`, -2506.0137786019113`}, {
     5371.078395544623, 1589.2865789399548`, -3051.80253766772}, {
     5270.9693030135095`, 1918.475932027788, -3037.127984612995}, {
     5442.695712348549, 1980.9792334631563`, -2672.1898041438662`}, {
     5557.556083289681, 2022.7849895825027`, -2389.5030774220777`}, {
     5653.312543647596, 2170.103364320896, -2004.625242283426}, {
     5492.316792830914, 2592.298059739058, -1949.938641768162}, {
     5575.507744748854, 2422.3715286987513`, -1932.2637643953342`}, {
     5911.9942064249135`, 1457.596675668733, -1900.4078836601648`}, {
     5963.77411688758, 1242.276755635775, -1891.5497442966744`}}}], 
    Line3DBox[{{{-6217.1645719467015`, 
     1425.1770477710938`, -59.571451528870384`}, {-6217.485064628049, 
     1423.7657606993985`, -59.86835380402036}, {-6217.745865375093, 
     1422.569219585875, -61.211592838002815`}, {-6217.830611915485, 
     1422.1318071752544`, -62.74786488115129}, {-6217.690257236881, 
     1422.670697033197, -64.41772166604306}, {-6217.257111417071, 
     1424.5129954941538`, -65.5013450406226}, {-6216.733282048175, 
     1426.791345034332, -65.63114816642226}, {-6216.442821896035, 
     1428.0952966139528`, -64.77774221422229}, {-6216.334166156739, 
     1428.641453614737, -63.14128353117599}, {-6216.4051333039415`, 
     1428.4293106315563`, -60.91480247908692}, {-6216.786737531108, 
     1426.8036135066843`, -60.064952689523125`}, {-6217.1645719467015`, 
     1425.1770477710938`, -59.571451528870384`}}}], 
    Line3DBox[{{{860.8313889593852, 5445.326756533282, 3208.61333357028}, {
     182.04409633318917`, 5636.072118366451, 2981.5037671555165`}, {
     265.9363338651223, 5709.759738858245, 2831.2073892323397`}, {
     712.3980286681275, 5612.947664406155, 2945.355879242311}, {
     963.9264572011389, 5504.141113164684, 3076.213594904928}, {
     860.8313889593852, 5445.326756533282, 3208.61333357028}}}], 
    Line3DBox[{{{3975.567267902869, 233.87124146752168`, 4982.768590933533}, {
     3828.6584176091255`, 317.01560721922874`, 5092.022144104463}, {
     3786.2993227835073`, 479.43984824166284`, 5110.957695012468}, {
     4013.6058554348338`, 423.0274182079552, 4939.618358807037}, {
     3975.567267902869, 233.87124146752168`, 4982.768590933533}}}], 
    Line3DBox[{{{2270.1100257594057`, -5810.751289498571, 
      1330.2569723693707`}, {2260.352572220557, -5814.3495065858915`, 
      1331.1497866582538`}, {2252.726407131691, -5816.877134581796, 
      1333.033317584755}, {2244.2338531020473`, -5818.819466093491, 
      1338.8682248833131`}, {2229.3272390087395`, -5822.261445032167, 
      1348.771818027647}, {2223.5079647826965`, -5824.072780648956, 
      1350.5562802978054`}, {2217.725383865488, -5825.15277599454, 
      1355.3977084537828`}, {2216.4916326387847`, -5823.195852132188, 
      1365.7849758116147`}, {2217.682667593456, -5822.137257272618, 
      1368.3621718440083`}, {2219.3960415891693`, -5821.228641445631, 
      1369.4495261203974`}, {2221.660110827554, -5820.482602298379, 
      1368.9493483323613`}, {2227.3016129931657`, -5819.4651130021775`, 
      1364.0993414369054`}, {2235.079815435796, -5818.9996925265505`, 
      1353.3198021928888`}, {2239.714915045714, -5817.836712051762, 
      1350.6542049038753`}, {2245.901148808619, -5815.70399582137, 
      1349.5661350106097`}, {2252.5685122174887`, -5813.606917994784, 
      1347.4878072952915`}, {2256.9259833747665`, -5812.92419483297, 
      1343.134803619246}, {2262.5812483152727`, -5812.3010977719305`, 
      1336.2992679827819`}, {2266.868637144638, -5811.040599608409, 
      1334.5139271033306`}, {2270.622863761256, -5810.029485340017, 
      1332.5325061867136`}, {2272.2666086751965`, -5809.748713763262, 
      1330.9538053688584`}, {2271.4892294865635`, -5810.2122816459305`, 
      1330.2569723693707`}, {2270.1100257594057`, -5810.751289498571, 
      1330.2569723693707`}}, {{2317.7238900636053`, -5787.4275661990305`, 
      1349.68582471181}, {2319.053029465356, -5787.087911204237, 
      1348.8588675326032`}, {2320.049779725969, -5787.147129725455, 
      1346.8893081853125`}, {2319.1279773514448`, -5788.181920005217, 
      1344.0272227862474`}, {2320.636008642417, -5788.60942291452, 
      1339.5757366933067`}, {2319.9827975765133`, -5789.115474124105, 
      1338.5199050465715`}, {2318.9562714334975`, -5789.597204388259, 
      1338.215121764399}, {2316.8239579404653`, -5790.36276221046, 
      1338.5961003676657`}, {2315.2958801582595`, -5791.114807786074, 
      1337.9865322054034`}, {2315.173735429086, -5791.425080464483, 
      1336.854443141173}, {2317.322695143185, -5791.609239879538, 
      1332.3256467982192`}, {2316.707947451302, -5792.183081886578, 
      1330.899365888673}, {2315.067862825909, -5792.9788505368815`, 
      1330.2896367870653`}, {2313.4704862506396`, -5793.529445591519, 
      1330.670718965158}, {2307.8547180385704`, -5794.466090585681, 
      1336.3319256381665`}, {2307.290690359728, -5794.256090997395, 
      1338.215121764399}, {2309.4190430917392`, -5792.569697815893, 
      1341.839658584559}, {2309.5761959952138`, -5792.052882624085, 
      1343.7986790357788`}, {2306.534346660699, -5792.6508484154765`, 
      1346.443146228596}, {2298.1410221970136`, -5795.160672204908, 
      1349.9904871608605`}, {2295.002036101071, -5795.508658136048, 
      1353.8311333995946`}, {2294.4005149519667`, -5794.935101471059, 
      1357.3014170502986`}, {2296.1385016372874`, -5793.769809012226, 
      1359.3355257173637`}, {2298.5367013725813`, -5792.765194550691, 
      1359.563945993473}, {2301.6746584832713`, -5791.766670904878, 
      1358.508846600319}, {2308.2663852481555`, -5790.589281448587, 
      1352.3297520598098`}, {2311.1164381547196`, -5789.805421528494, 
      1350.8174118400673`}, {2317.7238900636053`, -5787.4275661990305`, 
      1349.68582471181}}, {{2478.020890364083, -5731.170413038445, 
      1304.1901140677003`}, {2479.3710219847485`, -5731.025119553813, 
      1302.2611506090113`}, {2478.4500375082584`, -5732.162945193885, 
      1299.002336919172}, {2476.5025440691443`, -5733.426610898576, 
      1297.1384415933094`}, {2469.678999185809, -5736.790346533931, 
      1295.274430727202}, {2466.9029280743707`, -5737.923119306805, 
      1295.546954140434}, {2465.230913270013, -5738.311722302549, 
      1297.0076375808487`}, {2465.6502793775076`, -5737.769130422173, 
      1298.6099475131134`}, {2467.55368514721, -5736.64965821243, 
      1299.9396908501649`}, {2470.8573486965497`, -5735.227497889219, 
      1299.9396908501649`}, {2473.3801983052526`, -5734.004012403846, 
      1300.5391460344679`}, {2475.9221240787965`, -5732.605074752459, 
      1301.86880384296}, {2477.54901230864, -5731.342179428206, 
      1304.331784445822}, {2478.020890364083, -5731.170413038445, 
      1304.1901140677003`}}, {{2459.5045794447037`, -5740.031141558675, 
      1300.266667890533}, {2456.67561230463, -5741.8145940132, 
      1297.737952702589}, {2449.4816981539534`, -5745.563567984498, 
      1294.7402776843185`}, {2447.0446981885875`, -5746.601913166515, 
      1294.7402776843185`}, {2445.070189880769, -5747.307209846004, 
      1295.3398365712978`}, {2444.9293737086596`, -5746.976211658477, 
      1297.073039658199}, {2446.463260576704, -5745.737233377403, 
      1299.6672072607616`}, {2448.705252184558, -5744.631650871425, 
      1300.3320628708645`}, {2456.4259271305405`, -5741.092489785449, 
      1301.4001606923116`}, {2459.1124730077354`, -5740.00155981704, 
      1301.1385892347228`}, {2459.5045794447037`, -5740.031141558675, 
      1300.266667890533}}, {{2523.981634608422, -5710.804704134934, 
      1305.3888422189416`}, {2523.635815927288, -5711.505014623386, 
      1302.9913378456965`}, {2520.743545165679, -5713.373094896794, 
      1300.397457708599}, {2516.639851818419, -5715.439685117509, 
      1299.2639270076095`}, {2510.211325286112, -5718.204079990536, 
      1299.5364142589153`}, {2510.060405138868, -5718.104322669092, 
      1300.266667890533}, {2510.6537273141103`, -5717.449486251395, 
      1301.99958666945}, {2517.5638854375306`, -5713.833443533409, 
      1304.5279423223694`}, {2519.90917836016, -5712.480813844229, 
      1305.9228056350767`}, {2522.7162817338653`, -5711.149502823542, 
      1306.3259961714261`}, {2523.752648457239, -5710.798775174388, 
      1305.8574228727239`}, {2523.981634608422, -5710.804704134934, 
      1305.3888422189416`}}}], Line3DBox[{CompressedData["
1:eJx1mnc81+/3/4lUZkZZSfaOipTCZeRNSEKUkggZKURmRpSZkdei4vm0U5FE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      "], CompressedData["
1:eJx11X08lWcYB3BhrZfN5nxKRaOW9cJmXtqcCt3jpKxRHVqtRB2lzVtlp3Xk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      "], CompressedData["
1:eJx1lHtQ1FUUx3mPIBAmDwlQBCciAZdaARX0WpA7AiLyDhgF5LGgsItQjUBh
wMpKuwzQlaQauLa8wl2XgN/y7pCwtChgawIuA1gji4komJYSRre/+qs/zj+f
ufM953zPd+72xKxjyXo6OjoetPRptfhFBLt1MrC1/2C9ex9GPk8uNTlGEqgd
7T3u3MVAiKdLiEUDRg0T99KDogkciH6dpdvNQGrrzxnrFzC6nWN1yCOOQHtP
qCaT8udrl8Nb+Bhp7cotlQkE3Ma/wjNUZzVTs9SWjtGz/YGqX1MIfLrO3NLS
vncMZsyWMzBKmk67LswgIM4VNK4pGPDTW6+Ly8ZIHcRLjMgk8KWfavEu5Xvm
5eHyXIxScPLC3GkCRkLj5UXKNaWVfIMCjDZuO8w9SXWk1Qd2dVB+9fSy6koR
Rh+xTK1z0gnYh6otgykXd5r66ZZg1LNXofvmKTrn2RASxTBwZmS7YksxRjYT
O3T5fAJZKZyuT9oZaA5UF1mcx2ifMPqhw1kCb6cHFMpaGaiUCp2lZdSHv1+w
jp4nYNM8mZcgZ2DLxKFqnggj30oYKhQRUAYvDJ2g3Mee082jOmqDS6LYcgKs
8Sa2C9XpPTUdNk/n9+RYvvGemID/Q8l6HeW6vztwf6Q+fG+718uLvq+d7chn
U75WNaBJoFwiez9UUEFAM/zERHmVgbTa3PXgPIxeWr+6yfMigfTrvJsGUgbw
lLtJ0mcYub0SkbyzlkD5yxIJp4UBLdkmDqrC6N0+j/bVegIF3jLRcDMDI2NK
w+kajETW7qKTLQTQmQrVrSYGbgxVJPlIMCqMbyzmSgnMebGnOhrpvdp9H0+1
YTS2uCcyRkZAP3NF5dPAQLf6rwmHaxhtNlvo5LQSaPg4os63nvr85zKbP47R
/V3neKXfEYjRhOwzotx2MmCo6A5GAmG1uEhOIEnhbz8pYUDIGQ00ncUID6Bj
5pQ3H9FYuVPenB/FLpvHKEsxmuFM9c0XLxy1o9zcS+XKevDf+7DH7+T9Qrn/
gDxkeQGjgNGLl1VXCDw1VOc70TnZOgclLncxippLXR1sIhD3YHO2gO5b1dq5
ZHwTI8MaHXYeIXBuoXSnJfWtYia1d5OSzi8f7sqoJvC1z6PXNlCfs7/Zalk8
gJHYm60BekfxB6wju+ldkgtWwvsZ6kNlwYe3BQQiPVPLntE8hNnEjq3JMHok
PTfqWkjgj2tG/tw2mpMXn2vnv8WopIC32JpDwN2xRm5F86l1XJrVb8RIn1ch
eJ5K4Ke3jE24NM9payecmBaMvnAt2IGPU9/0ZlYQ5ZX1U76HaV875f1423gC
TjdKdiupjomMs6HjB4wstEY9nbEErAenubb/6vCehvQOYoRW+JzEGAK/FZvt
V1Au8g6KuUf36mf6RjZGEfi/f+MfAXRQqQ==
      "], CompressedData["
1:eJx11H841VcYAHDP1bJZJuF2x1ooodWV8jPh3MiVO8mt7FbilitlrREKra6m
JuJWXo8WKc+R8ls91neSOO7UxhayzB3SYpgfa1YmbWTv/vLX/jj/fJ7zvO97
3vOeY77nU2kYT0tLywmXNq5i18w/aBnH+NqurmGfAHmybKqZOFHWUrk3JBg9
aSyqr3kvkOe+1X4fraVszSY1OKBnSF/dHQ0CUn/tdZbajTJra6m7HfoNlTDB
PBDIPPmYuMCDskvmMQuU6PvXyc4N+wHxDBzh+RPK4iZe8OXlHHO2f7MrWQRE
1cr96oD7s8wi675D76xLKZjvDuSFaCJ2oTtlFibNYFzBsd62zN5Cl9m8qVvN
W3XR24p5D1ocgPzwTCvtH9yf1uvpdQjjnLBdpnsHfcXQKulLEWXnhhr+ykD/
0TZuS/RqIPopB4rs1lPm4XtyJR/jfCXPWf9gJRC3/jGnHqyzQKd7IAX9Qsbd
a5wlELtlpmG16LsH2/MsbnBMA1nn9d6bPZcwMjKZoRsKhGdFhkBqDfPVqXgu
6f24mS03OabY3V16QHe2zvYys6aD6C0xJ0ZrtIGo3RqO2WMcj80mlYvQs3m1
Lnk8IJtOPP/JAOsMlo7v/C9vZKaA1zsXyNMzTXodGyiTZL3Mu4XnCq8fFFi9
A2Sx7Y7KORLKcky2xldg/9vF0QOP9IFYTXhuLPTHvr06PxVQyrG/u/tn1Fhn
CPc6R7GNMt9WtW9fMcdaw92ON/KB1BhZ3qnYSdnFbycPDxZxDAzTllbjeWO7
7qXPBFO2Wq/zy6BCjq0qqp9KsQbSzB/sSg6lbEbLpjHrOsdODS2ytLMDkiB6
JjQOo+x7999U29DP6DyoqnUCEqG7ID9PQVnA+7W7R69x7GjP9VP23kCO3n66
pWcPZYd8MgVN6EsjTJWcDIhe2IX7xZg3UF1lcwE9ZO+ViclQIGdux/DLgygL
902ajkGPV0ryEz4GQtPP/rkD639SFqt8A334conkaRSQAL+yL1rQG7w4F3kB
xzzHNN3B8UB8xEZPeLuwzhGLV5uvcmyPjsf9I0lApDWyXFUIZXV+VXkP8zkW
GnBgX2sKEHGQoFAqp0x00MR4AF0WO/m7Jg3IcES4bDHuv+IYJbuJcRQxPtGn
04GMp9ZYzcW8vdqjWdmYd63LuEyVAcS2RZhji/1v0rR2OGKdLfOqf7mVDaRw
MsB0Gu/rkTB5Doc+cfuz3tx8ILui9yUtxfu9V2HUqEYfOT6nU1QOxDt5xOot
H+zb9Sb9RvQ2RTzVqQLiuvZhjxjnpGaDakiI/R8Xa+oa7uAcTpsJP8S5euyT
ZtqG7tsdMiStmZ3PfsESv2N4v2BReVLxNZCLBnHS1+sos5TwwQznYdx5x0Jl
KZAx76LJb9AN/IeVnejmuSU+3blAlljLNAJ8pwOLQiP0ca4SrQ6/+xiApC8X
rqnH+JJ4z2NS9O09Eec2pc/+A94920ty0K/Wa/YdOQ2kLvagAR/doF7y9gcl
HOsql2cv/xzI1Sj/sSmMf6VvfWIlerXYsd8oEd/ppZ0r+l0pE4d7Pb6Ic674
uWG+MgGIxdwO9yFnjCN3vRyJ76LJft1+m2isf2aj6qYjZf/3H/4LkHe31w==

      "], {{-5753.047674728544, 
      1274.787910436778, -2442.3701451059223`}, {-5752.988337462225, 
      1271.0884085931257`, -2444.437181963124}, {-5753.229176646423, 
      1266.5085647293458`, -2446.246877658004}, {-5753.529116851435, 
      1262.2586123149154`, -2447.7376467218814`}, {-5753.5821081613, 
      1258.4816786917324`, -2449.5572006065527`}, {-5752.909476889995, 
      1256.6512810608444`, -2452.0754063685367`}, {-5751.6898405911315`, 
      1256.1745166663643`, -2455.178863406654}, {-5750.310861535072, 
      1258.1868187885775`, -2457.377588963172}, {-5749.530875007883, 
      1260.22453285947, -2458.1583622928415`}, {-5749.128012052559, 
      1263.0810797104543`, -2457.634427068042}, {-5749.921525101413, 
      1264.6230692924348`, -2454.98363289632}, {-5750.760646211816, 
      1266.0703797089257`, -2452.2706770675136`}, {-5750.682199581218, 
      1267.4212078390256`, -2451.7568010183213`}, {-5750.254888282265, 
      1268.4846734204812`, -2452.209012927579}, {-5749.293674185289, 
      1268.5883182002165`, -2454.4082009517015`}, {-5747.268643397921, 
      1267.6155404997166`, -2459.647875631516}, {-5746.631050996243, 
      1268.63190679184, -2460.6134070177527`}, {-5746.307706924708, 
      1269.717551456631, -2460.808559571565}, {-5745.938519811118, 
      1272.8970047927087`, -2460.0279331220145`}, {-5745.506097765776, 
      1274.589714229581, -2460.1614643995295`}, {-5745.038136627219, 
      1274.6963117397436`, -2461.198856560751}, {-5744.318629848078, 
      1275.0626439270852`, -2462.688048219256}, {-5743.886184541958, 
      1276.0185857780414`, -2463.2015260179255`}, {-5743.61223663133, 
      1277.7461129955393`, -2462.9447894631408`}, {-5743.9205264572975`, 
      1278.9720748187701`, -2461.589142723922}, {-5744.845414145099, 
      1281.1776786150783`, -2458.2816383389627`}, {-5746.536539587826, 
      1282.3971221567294`, -2453.688878049965}, {-5747.640038713095, 
      1282.0115279051365`, -2451.30457465009}, {-5749.01504657149, 
      1280.4223937472461`, -2448.909589569444}, {-5750.176816148469, 
      1280.7864816182544`, -2445.989832671548}, {-5751.072383329318, 
      1281.8288253217713`, -2443.336856489715}, {-5751.620134185673, 
      1281.2133297485068`, -2442.3701451059223`}, {-5752.294732905407, 
      1279.045478575008, -2441.9176193159224`}, {-5752.735497400871, 
      1276.8255797848158`, -2442.041036868773}, {-5753.047674728544, 
      1274.787910436778, -2442.3701451059223`}}}], 
    Line3DBox[{{{-4737.863070202987, 
      496.57610413588395`, -4241.919305079713}, {-4811.265399721511, 
      614.9148579831805, -4142.634265883742}, {-4706.194010213946, 
      706.144757592446, -4247.459382919521}, {-4295.845595447454, 
      1012.8563511033614`, -4605.176035951752}, {-4383.025046286756, 
      711.2036308707342, -4579.420763487371}, {-4467.714736052133, 
      748.975104740553, -4490.698722887285}, {-4553.654791266014, 
      623.7731150348069, -4423.002242022928}, {-4584.454880239824, 
      730.2079398290585, -4374.629527817854}, {-4607.653043138866, 
      586.1675028611859, -4371.928039508728}, {-4737.863070202987, 
      496.57610413588395`, -4241.919305079713}}, {{-4901.924157414937, 
      262.6186239127533, -4073.0804931066637`}, {-4752.254248597862, 
      396.2738298663382, -4236.373484296047}, {-4855.9895143909, 
      394.9690228048486, -4117.180952684764}, {-4899.034484269378, 
      532.2054567734793, -4050.189151597739}, {-5070.029015296869, 
      448.0287049050989, -3844.722324645685}, {-5219.450681334576, 
      668.6270529009792, -3605.285860960956}, {-5039.207626690614, 
      352.3757240558992, -3894.8724670005854`}, {-5045.361447106861, 
      127.71161115808071`, -3900.747531233361}, {-4901.924157414937, 
      262.6186239127533, -4073.0804931066637`}}}], 
    Line3DBox[{{{736.646547978166, -6117.156107894091, 1650.929042995449}, {
     692.7279621760399, -6224.842602420507, 1208.0050643561553`}, {
     471.1890348802702, -6242.326798766159, 1224.3981055255013`}, {
     289.20357702908416`, -6209.316780654548, 1431.2791825899749`}, {
     736.646547978166, -6117.156107894091, 1650.929042995449}}}], 
    Line3DBox[{{{5673.136883483953, 1516.5755351462883`, 2490.648557594935}, {
     5721.451731335897, 1214.392718737999, 2545.2021728478826`}, {
     6009.152264008688, 446.5571511481072, 2092.484042643517}, {
     6126.121674985679, 415.846814221442, 1727.8652552709455`}, {
     6161.299976060484, 25.09163189600567, 1650.929042995449}, {
     6236.140306225268, 261.3717194755278, 1315.3146373894813`}, {
     6232.941277359054, 388.5030277274849, 1298.9696379978502`}, {
     6233.843155296681, 392.2001678269327, 1293.519320890586}, {
     6172.529279694678, 527.3637601438886, 1519.7313729073328`}, {
     6132.473410088502, 1040.9003922249894`, 1413.191401765296}, {
     6022.180390066588, 1460.6279081235198`, 1512.5221167004963`}, {
     5881.729410247841, 1629.305564399842, 1854.302082305173}, {
     5748.901189273992, 1648.47088920764, 2218.2183241058697`}, {
     5673.136883483953, 1516.5755351462883`, 2490.648557594935}}}], 
    Line3DBox[{{{6285.13708219307, 948.6676081447622, 533.7558592638408}, {
     6325.005517292666, 675.9480933593106, 474.56619275001884`}, {
     6317.954668009482, 584.2453496206161, 655.6827544048112}, {
     6332.235395244586, 300.4667352257394, 707.3389076209747}, {
     6232.941277359054, 388.5030277274849, 1298.9696379978502`}, {
     6233.843155296681, 392.2001678269327, 1293.519320890586}, {
     6172.529279694678, 527.3637601438886, 1519.7313729073328`}, {
     6132.473410088502, 1040.9003922249894`, 1413.191401765296}, {
     6022.180390066588, 1460.6279081235198`, 1512.5221167004963`}, {
     6026.779160137732, 1511.958699049518, 1442.1260476456328`}, {
     6045.994221790738, 1580.4989265569993`, 1278.9803373098352`}, {
     6194.357747371584, 1303.4741484729236`, 786.5754457395809}, {
     6241.405209539873, 1078.0778766264852`, 755.2633926366776}, {
     6285.13708219307, 948.6676081447622, 533.7558592638408}}}], 
    Line3DBox[{{{-5939.550420399394, -1066.0184900458473`, \
-2067.1620817956555`}, {-5938.129737306742, -1074.4309640902345`, \
-2066.888244757681}, {-5937.36295112096, -1077.6101409015132`, \
-2067.435914576901}, {-5935.302810559293, -1080.3396036106324`, \
-2071.921952641406}, {-5933.976021143689, -1080.3121011304556`, \
-2075.733131845906}, {-5932.099057444411, -1081.4679836907392`, \
-2080.490678402336}, {-5930.9041017768595`, -1084.3520923296583`, \
-2082.3954392446008`}, {-5930.501060524272, -1083.422692187399, \
-2084.0264228588576`}, {-5930.403858557598, -1078.699376372172, \
-2086.751403454527}, {-5930.474648753424, -1071.4425225036314`, \
-2090.2858779277076`}, {-5930.981630301729, -1066.5108446894956`, \
-2091.369218761514}, {-5932.077352432993, -1063.608578762956, \
-2089.738922912002}, {-5933.635034899752, -1061.8570646065452`, \
-2086.2043372966577`}, {-5934.589338364663, -1060.5313748123465`, \
-2084.1632081022613`}, {-5935.672029820621, -1058.4799893887482`, \
-2082.1218399672825`}, {-5936.918349587567, -1054.960662434902, \
-2080.353864381093}, {-5937.766396167705, -1055.5389847986078`, \
-2077.6384108193374`}, {-5939.212352635784, -1056.758465936964, \
-2072.8800893850726`}, {-5939.946328514597, -1060.0979728637378`, \
-2069.068290873008}, {-5939.550420399394, -1066.0184900458473`, \
-2067.1620817956555`}}}], 
    Line3DBox[{{{-5449.787720930462, 
      1178.982771905502, -3098.0037298792245`}, {-5450.680988117736, 
      1177.3836158697063`, -3097.0402446292887`}, {-5451.3514566283, 
      1174.83996465591, -3096.8261242381377`}, {-5451.772852412327, 
      1171.5458066120607`, -3097.3322196221798`}, {-5450.7267096709365`, 
      1164.6544106656972`, -3101.769193572722}, {-5449.178305520358, 
      1164.7213623253099`, -3104.4635016036773`}, {-5447.383003517218, 
      1168.91165959252, -3106.038897866606}, {-5446.895159268703, 
      1171.492226658646, -3105.9222103609654`}, {-5446.901587435711, 
      1174.5773487900049`, -3104.745535225032}, {-5447.793582233149, 
      1178.451837337599, -3101.7108250858832`}, {-5449.787720930462, 
      1178.982771905502, -3098.0037298792245`}}, {{-5445.925515594102, 
      1166.8093624287067`, -3109.3833601455167`}, {-5446.349200242232, 
      1164.4148638660222`, -3109.5388892613437`}, {-5446.186730123901, 
      1162.8892305594793`, -3110.3942560503656`}, {-5445.343343937204, 
      1162.8085203268183`, -3111.900689561629}, {-5443.911082202789, 
      1164.9865748752923`, -3113.591511330605}, {-5443.504093519559, 
      1166.886787313064, -3113.591511330605}, {-5443.875812276643, 
      1168.1590786319227`, -3112.4643287143285`}, {-5444.736036401703, 
      1168.3436674535637`, -3110.889946402769}, {-5445.925515594102, 
      1166.8093624287067`, -3109.3833601455167`}}}], 
    Line3DBox[{{{-5092.833542821636, 3450.0523599392127`, 
      1687.6015254759313`}, {-5093.896032355429, 3448.178633778235, 
      1688.2242182568293`}, {-5094.81978550367, 3447.1185263131947`, 
      1687.6015254759313`}, {-5095.0213270294835`, 3447.514159480515, 
      1686.1842981406508`}, {-5094.848235749053, 3448.693536137592, 
      1684.294522489326}, {-5094.409235108021, 3450.2118326065106`, 
      1682.5119748127013`}, {-5093.53393397601, 3452.602356688196, 
      1680.2567422944576`}, {-5094.427409866519, 3453.856868310701, 
      1674.9614290555216`}, {-5093.964977539388, 3454.9710258006876`, 
      1674.0698039448307`}, {-5092.7368965863025`, 3456.7341759229803`, 
      1674.1664878842194`}, {-5091.133778958531, 3458.7620028914066`, 
      1674.8540062053319`}, {-5090.322400003845, 3460.288734320375, 
      1674.1664878842194`}, {-5090.6750445245125`, 3460.5284542764402`, 
      1672.598008547918}, {-5091.404298893435, 3460.1147885053147`, 
      1671.233558057122}, {-5091.031335121416, 3461.5501988356486`, 
      1669.396254863632}, {-5089.275975594175, 3464.254727429827, 
      1669.1383758454674`}, {-5087.757583071812, 3466.0806466451363`, 
      1669.9764719442305`}, {-5086.510500615065, 3467.051065153182, 
      1671.7600093173855`}, {-5086.245717741345, 3466.480552975294, 
      1673.74752116379}, {-5086.781192148029, 3464.3756855843526`, 
      1676.4760369122587`}, {-5086.992822835095, 3462.3108369995152`, 
      1680.0956456166284`}, {-5087.174087350448, 3460.7455305792105`, 
      1682.7697012859935`}, {-5088.1801514524695`, 3458.832481072361, 
      1683.6609825745604`}, {-5089.869391074588, 3456.215406126274, 
      1683.929433871084}, {-5091.102593602981, 3453.937821208955, 
      1684.8743569273458`}, {-5092.833542821636, 3450.0523599392127`, 
      1687.6015254759313`}}, {{-5087.849029168481, 3468.483678263123, 
      1664.7002492976487`}, {-5088.46055464859, 3467.209733618022, 
      1665.4847753401443`}, {-5089.676432915477, 3465.8273879152725`, 
      1664.6465136482204`}, {-5090.458872876877, 3465.7100185282625`, 
      1662.4969837632525`}, {-5091.070610078705, 3465.996463205238, 
      1660.024773984302}, {-5091.692767842141, 3465.5097152165536`, 
      1659.1325630110807`}, {-5093.561016863757, 3463.5302104488514`, 
      1657.5307957301998`}, {-5094.51579667069, 3463.529338681406, 
      1654.5957233009951`}, {-5093.488741596864, 3467.6427250490465`, 
      1649.1331239898184`}, {-5093.179699583675, 3469.6443647570927`, 
      1645.8743025693364`}, {-5092.431272050161, 3471.2169373735132`, 
      1644.873977889132}, {-5091.316788448212, 3472.800422846936, 
      1644.9815417625568`}, {-5090.355547109777, 3473.837348253086, 
      1645.7667428559514`}, {-5089.647714171211, 3473.875128334343, 
      1647.8748218149976`}, {-5089.8189136318, 3472.6899522989092`, 
      1649.842905139215}, {-5088.96332961799, 3472.4967126462416`, 
      1652.8861123696404`}, {-5087.003497450438, 3473.8927005980086`, 
      1655.9826725116689`}, {-5085.956242399815, 3474.3491247624765`, 
      1658.2403172208067`}, {-5086.617756803369, 3472.3276386642783`, 
      1660.4439936176127`}, {-5087.586572173057, 3470.1259258596397`, 
      1662.0778018255328`}, {-5087.849029168481, 3468.483678263123, 
      1664.7002492976487`}}, {{-5088.930731199593, 3510.88034217876, 
      1569.8142966463047`}, {-5090.059408468052, 3509.692538745465, 
      1568.810757147043}, {-5090.234302876416, 3510.0752975893315`, 
      1567.386307543044}, {-5087.886925507999, 3514.4871338997837`, 
      1565.1199662957595`}, {-5086.461432281485, 3517.4379145458693`, 
      1563.1232533706952`}, {-5086.0204863287345`, 3519.3641465270152`, 
      1560.2196341491758`}, {-5082.657586100166, 3525.045302034809, 
      1558.3520690334262`}, {-5080.1784888990705`, 3528.7125030366415`, 
      1558.136156386867}, {-5079.189111197979, 3529.602538050199, 
      1559.3452427469276`}, {-5079.077801180866, 3528.7365166232876`, 
      1561.6660897263168`}, {-5080.990944363178, 3524.808838601193, 
      1564.3105077376078`}, {-5083.077706868556, 3521.0025577712113`, 
      1566.102073335098}, {-5084.034345678297, 3518.1210223880444`, 
      1569.468997398405}, {-5085.371539358744, 3515.503540929996, 
      1571.0012254574283`}, {-5087.50229414363, 3512.3855036057594`, 
      1571.076755331646}, {-5088.930731199593, 3510.88034217876, 
      1569.8142966463047`}}}], 
    Line3DBox[{{{-3093.482217121547, 3907.6712207735645`, 
     3980.9907174836253`}, {-2986.9387877239574`, 3892.6544152854135`, 
     4075.935717298436}, {-3075.4846139429023`, 3577.685284355293, 
     4292.944919924791}, {-3064.5828116593643`, 3575.5101171963893`, 
     4302.54265316548}, {-2754.2467036776325`, 4027.5002340718456`, 
     4108.671397158956}, {-2904.1185414169254`, 4048.9601539397972`, 
     3982.440314581782}, {-2854.515529732598, 4127.572603038655, 
     3937.3467191955706`}, {-3010.7676244442982`, 4039.254646451382, 
     3912.4818086118885`}, {-3093.482217121547, 3907.6712207735645`, 
     3980.9907174836253`}}}], 
    Line3DBox[{{{1893.7291566302995`, 1128.142655703876, 5985.724494136837}, {
      1843.3128630636493`, 1110.49528517233, 6004.732433712106}, {
      1826.8227983218885`, 1016.7965256457002`, 6026.330307821354}, {
      1828.657944711001, 958.0321477978672, 6035.395448107037}, {
      1879.6459260818742`, 891.1817852602674, 6029.970141566551}, {
      2141.4874056952417`, 699.945975813799, 5967.572455545951}, {
      2459.8895044906453`, 576.2070085785347, 5857.025080957204}, {
      2780.486842810392, 467.78866897546936`, 5721.702575994007}, {
      2761.8171898830783`, 559.3886000690129, 5722.52251142616}, {
      2883.20012614992, 348.0546115659431, 5679.24543490918}, {
      2964.8744185752275`, 263.7386984957521, 5641.611052792808}, {
      2986.4550541809376`, 356.99431244240327`, 5625.073773759852}, {
      3070.026118787961, 279.3942638392765, 5584.325578375014}, {
      3042.030727812818, 411.2986913018885, 5591.4842508564725`}, {
      3113.6919610334335`, 282.4549530843777, 5559.942019596416}, {
      3116.936609784333, 388.2349778006497, 5551.737877691838}, {
      3224.196179195511, 292.4792150046982, 5496.07392570836}, {
      3351.539926525362, 387.7864002895942, 5413.373764913782}, {
      3146.31669307897, 586.9228093069399, 5517.610585458878}, {
      3221.639224205622, 651.5471197447586, 5466.657182393286}, {
      2774.4617499641536`, 596.4815724997836, 5712.651347811629}, {
      2135.1268073386077`, 801.8326517419569, 5957.022041690337}, {
      2060.4857416266323`, 993.8603400317141, 5954.3642669109}, {
      1920.3552289353001`, 1016.7757547890094`, 5997.184051634074}, {
      1995.4091225083205`, 1104.5562541700037`, 5957.022041690337}, {
      1995.9596918806546`, 1107.137641756201, 5956.35835394364}, {
      1890.9203190428336`, 1129.4524357424789`, 5986.365444276379}}, {{
      970.2546748379737, 449.34849696364097`, 6288.4422144305}, {
      1094.2864722409429`, 485.30188933968805`, 6265.367737242069}, {
      1040.8014737816188`, 333.83086094383776`, 6284.35382027902}, {
      970.2546748379737, 449.34849696364097`, 6288.4422144305}}, {{
      1138.6913621951396`, 214.81492756089264`, 6272.563289322729}, {
      1270.9867922947296`, 293.4304216815661, 6243.90141269416}, {
      1222.8558090304339`, 380.87717092604504`, 6248.78947804996}, {
      1283.8184423902717`, 310.1930915090763, 6240.465058227524}, {
      1421.9336283020596`, 417.1498133941825, 6204.184433097466}, {
      1155.7415621129805`, 456.4241800989028, 6256.496791290289}, {
      1056.1118520897878`, 308.8287179249492, 6283.077770397268}, {
      1177.7727820004, 347.75449828698623`, 6259.371515807334}, {
      1138.6913621951396`, 214.81492756089264`, 6272.563289322729}}}], 
    Line3DBox[{{{3669.642922330036, 4887.505489210167, 1825.8763938266904`}, {
     3214.5156504481292`, 5104.564226196217, 2073.192437682006}, {
     2970.407921038545, 5103.671802431449, 2411.8508672174557`}, {
     3195.193870504328, 4845.765857477599, 2645.2037869313517`}, {
     3293.4310373129674`, 4852.206418466464, 2509.425943329764}, {
     3249.921193746912, 4941.2651181074125`, 2389.5030774220777`}, {
     3714.547097420422, 4751.553858505543, 2076.7015868358653`}, {
     3669.642922330036, 4887.505489210167, 1825.8763938266904`}}}], 
    Line3DBox[{{{2743.6453946966503`, 5077.795886147398, 
     2715.9183671061287`}, {2284.498506046803, 5304.54618360469, 
     2707.5210159618055`}, {2167.952505094029, 5420.267943636954, 
     2570.6985519189398`}, {1881.3807473802478`, 5495.063492966077, 
     2636.758878179496}, {1988.8421645332683`, 5324.120782540299, 
     2895.8692006886467`}, {1750.2885758595855`, 5355.014631204442, 
     2991.3411525270367`}, {1443.2790391847504`, 5269.716453456238, 
     3291.632999777246}, {1360.8430306106513`, 5218.133720987516, 
     3406.904813159797}, {1450.7743997814325`, 5042.83846457875, 
     3626.685477278649}, {1095.9328708308844`, 5076.0384350009945`, 
     3704.1299435933715`}, {1355.073455710602, 4908.407628312745, 
     3841.760556561247}, {1616.3743331573578`, 4868.455242148314, 
     3791.2123734115335`}, {1816.9783487239497`, 4965.126694664225, 
     3568.461892216461}, {1910.5450598582647`, 5065.020542200001, 
     3373.8995496818197`}, {2156.302760820424, 5010.89076698226, 
     3305.9258635139054`}, {2228.174716680869, 5063.912248110751, 
     3174.8769842905795`}, {2692.986895438178, 4831.72545373678, 
     3176.486173543079}, {2547.7023181147824`, 5072.8923976154965`, 
     2909.087367283329}, {2743.6453946966503`, 5077.795886147398, 
     2715.9183671061287`}}}], 
    Line3DBox[{{{-4454.692161099791, 4484.804840630407, 
      854.4337196612385}, {-4456.441939885619, 4483.122264872744, 
      854.139148283681}, {-4458.302404915603, 4481.5509086942475`, 
      852.6750899994137}, {-4461.967671253176, 4479.289729204777, 
      845.35301060095}, {-4462.24096625582, 4480.502388575853, 
      837.4470699897407}, {-4460.34458566209, 4483.290730865152, 
      832.612808989564}, {-4457.595125852146, 4487.727428747097, 
      823.3844237354475}, {-4455.886365379333, 4490.707437877722, 
      816.3559862071551}, {-4453.615715242213, 4493.278777821334, 
      814.5959585757935}, {-4449.301474669889, 4497.7099373156225`, 
      813.7170253058794}, {-4445.215650337635, 4501.429821465403, 
      815.47598028103}, {-4442.067638196646, 4503.898703437373, 
      818.9903887477635}, {-4440.265160474128, 4504.901042749441, 
      823.2409062447402}, {-4441.442843092739, 4503.579639926445, 
      824.1174603054326}, {-4443.579655588447, 4501.658547233403, 
      823.0940763733495}, {-4445.671946900338, 4499.3780698650235`, 
      824.2642870833652}, {-4447.165799005438, 4497.120637636824, 
      828.5143430848284}, {-4449.298955581771, 4493.62703354722, 
      835.9835986624194}, {-4450.80572150448, 4491.697953239836, 
      838.3266763523927}, {-4456.008049295939, 4486.756064029987, 
      837.1534963792138}, {-4456.994687335744, 4485.556869640258, 
      838.3266763523927}, {-4456.679510025233, 4485.239672549058, 
      841.6926486035579}, {-4453.9911424895745`, 4486.917481287937, 
      846.964053981708}, {-4453.123022883764, 4486.6693748725465`, 
      852.8229321748817}, {-4454.692161099791, 4484.804840630407, 
      854.4337196612385}}, {{-4429.161536511101, 4518.333705422977, 
      809.3221237117473}, {-4431.447484326129, 4516.563933532525, 
      806.6827864612094}, {-4433.863399161763, 4514.610825558522, 
      804.3370819025421}, {-4436.108086495856, 4512.483124746577, 
      803.8997349193745}, {-4436.8008241562475`, 4512.557582093622, 
      799.6475505219838}, {-4434.698634576666, 4514.830736693172, 
      798.4767557633933}, {-4429.561979113718, 4519.68890243641, 
      799.5039639572273}, {-4426.6458765781335`, 4522.078346782359, 
      802.1392640532771}, {-4422.516762892078, 4525.5961838032235`, 
      805.0704025638206}, {-4422.268561859757, 4525.342197734903, 
      807.8567015174656}, {-4424.085766530055, 4523.251743489279, 
      809.6125526493421}, {-4424.875223152941, 4521.848032596717, 
      813.1317972065847}, {-4427.071277434152, 4519.671605394152, 
      813.2786569875099}, {-4429.161536511101, 4518.333705422977, 
      809.3221237117473}}}], 
    Line3DBox[{{{813.7518615179529, -6237.130054770872, 
     1060.1086160153413`}, {1089.5872333931259`, -6200.448036141025, 
     1027.1599203345156`}, {1110.9060666506575`, -6194.832870990318, 
     1038.145992419473}, {1379.1009061805705`, -6152.9261274553155`, 
     963.0132141614962}, {1328.2525931859225`, -6186.96350137115, 
     803.1446765036173}, {1298.901397037227, -6172.6271604507, 
     948.3368540983028}, {1144.7203946126415`, -6196.475162954322, 
     990.5172704441275}, {1137.665300787276, -6198.6507394302125`, 
     985.0179510779877}, {1051.632994285331, -6239.70193986774, 
     804.9853644267173}, {780.6799538144627, -6267.673104453008, 
     891.4181709100569}, {813.7518615179529, -6237.130054770872, 
     1060.1086160153413`}}}], 
    Line3DBox[{{{-4952.077872528578, 
      4010.1135923119414`, -289.3568923800711}, {-5349.441833284738, 
      3410.1583511981685`, -664.9103564064902}, {-5315.261335437104, 
      3427.640889419218, -828.9080523126803}, {-5483.981210907103, 
      3054.4345429328546`, -1133.2225692053714`}, {-5309.093785541013, 
      3354.108622426034, -1118.6117357229582`}, {-5149.639885370117, 
      3670.942784246264, -832.5874223140429}, {-4983.707801668947, 
      3875.0562514809144`, -913.4600623140332}, {-5053.044057220364, 
      3761.851920348767, -1001.5136420230403`}, {-4894.331955406063, 
      3963.351870643927, -1012.5069628177123`}, {-4952.077872528578, 
      4010.1135923119414`, -289.3568923800711}}, {{-5626.818804188301, 
      2968.7770989714213`, -461.61279386057555`}, {-5549.651871070224, 
      3072.002935894825, -672.2914262284892}, {-5406.11327815476, 
      3330.20014497213, -609.524329445825}, {-5549.82218797456, 
      3084.763761405222, -609.524329445825}, {-5626.818804188301, 
      2968.7770989714213`, -461.61279386057555`}}}], 
    Line3DBox[{{{
     3333.2543195487383`, -4690.344474874739, -2752.7982196706116`}, {
     3204.2000110548915`, -4676.6972188983655`, -2923.9389702769327`}, {
     2952.755129918785, -4837.389447863771, -2927.236608549868}, {
     3080.241823310734, -4881.887823235372, -2714.239355928943}, {
     2712.6498939756166`, -5244.42859821239, -2413.5685034749054`}, {
     2843.341196027777, -5291.715301068514, -2144.9880431854845`}, {
     3158.1847113327835`, -5087.01993835785, -2199.070523165084}, {
     3133.676148172658, -5011.685561153924, -2398.1024424521684`}, {
     3313.3453163777403`, -4884.603276089638, -2418.72018630764}, {
     3309.81401324744, -4797.848547462881, -2591.060439920149}, {
     3403.234500056341, -4727.390937841669, -2599.5352562407634`}, {
     3333.2543195487383`, -4690.344474874739, -2752.7982196706116`}}}], 
    Line3DBox[{{{2030.9412650504535`, -5703.54569018256, -2008.14786007286}, {
     1466.523142650837, -6008.323545222668, -1561.144368081445}, {
     1228.6853955211707`, -6186.423009185064, -952.0064286855392}, {
     972.3285018892748, -6268.797954903519, -666.7557086453812}, {
     1069.236581296401, -6277.278619400431, -374.5930560432121}, {
     1051.463687567647, -6272.090729900789, -493.0676217510447}, {
     1208.8527159215118`, -6238.38273462949, -555.9403362644871}, {
     1620.4380188492848`, -6169.430312032499, -11.132926047555543`}, {
     1870.3597514067849`, -6092.30864838198, -270.8199091050806}, {
     2168.886292762953, -5991.4822398823935`, -293.064000243695}, {
     2101.9101676035007`, -6007.714958972454, -420.88990135620264`}, {
     2180.954242568368, -5975.891334026496, -469.0149738868202}, {
     1868.07234899272, -6072.264070904355, -570.7262409709941}, {
     1741.2641911434885`, -6079.183418028523, -836.2665105147113}, {
     1818.8909275460026`, -6024.457403067328, -1041.807315350636}, {
     2098.734364619451, -5932.126416423401, -1045.4682856677462`}, {
     2076.2748799428855`, -5906.868421233011, -1218.9346853537272`}, {
     2186.3696607662596`, -5868.518455983097, -1211.6486818655826`}, {
     2265.5219439663015`, -5800.78945995964, -1380.6033654269768`}, {
     2130.477608373321, -5698.221203685099, -1918.1120832979068`}, {
     2030.9412650504535`, -5703.54569018256, -2008.14786007286}}}], 
    Line3DBox[{{{-3232.7302511635053`, 5116.968476223252, 
      2013.4305119744822`}, {-3204.3026132609407`, 5252.925763049168, 
      1681.3772273179554`}, {-3275.445488149902, 5252.001591510281, 
      1541.3467575247016`}, {-3458.2310637796295`, 5139.928808233711, 
      1519.7313729073328`}, {-3489.202286429618, 5156.760760224919, 
      1386.0373574254222`}, {-3328.986877746064, 5218.760158238847, 
      1539.5461891652837`}, {-3346.3129432243973`, 5232.4837177641775`, 
      1452.9685197352492`}, {-3156.3962941921827`, 5330.094965438728, 
      1521.5333658564937`}, {-3040.824312035091, 5313.161756995904, 
      1792.0695682178507`}, {-3078.628302878663, 5205.676166653947, 
      2027.5100765829773`}, {-3234.218580166093, 5116.02789663608, 
      2013.4305119744822`}}, {{-3414.9054409537694`, 5205.293572960668, 
      1389.6594326088311`}, {-3498.413860099934, 5173.616560205648, 
      1297.1529753373497`}, {-3414.9054409537694`, 5205.293572960668, 
      1389.6594326088311`}}, {{-3135.3201936515516`, 5351.154224560451, 
      1490.8821007207853`}, {-3250.528349658312, 5294.0045715709475`, 
      1447.5478348757642`}, {-3221.5477406131813`, 5336.910092891725, 
      1351.604418955688}, {-3136.8766486472273`, 5350.241970502092, 
      1490.8821007207853`}}, {{-3645.6118956162577`, 5120.405239205583, 
      1085.715332330372}, {-3774.071040039927, 5078.7039001988305`, 
      806.8259842352162}, {-3744.785375076478, 5116.600366131176, 
      696.2735654321709}, {-3689.4847200216905`, 5140.769860029259, 
      804.9853644267173}, {-3677.5304980710234`, 5174.788119721379, 
      620.6053832734332}, {-3504.7716796234245`, 5258.496525582336, 
      867.5272065751587}, {-3343.3391687191224`, 5374.783338655956, 
      788.4167396845808}, {-3470.7948427296596`, 5263.736671025249, 
      966.6814924188999}, {-3503.5000680230014`, 5256.588622359542, 
      884.068446808866}, {-3660.0177868355204`, 5127.999787137909, 
      997.8485221521206}, {-3645.4285671798684`, 5120.147747208886, 
      1087.5436995085292`}}, {{-3550.9222076167857`, 5179.52439420424, 
      1118.6117357229582`}, {-3497.1893437376116`, 5224.039071643632, 
      1080.2296802017731`}, {-3552.4287199359796`, 5178.491253683536, 
      1118.6117357229582`}}, {{-3460.355357987934, 5221.42403723717, 
      1204.3610379807553`}, {-3450.3141330981684`, 5269.286683177724, 
      1008.8428635961741`}, {-3367.6653273746065`, 5306.591432701165, 
      1089.3719746631439`}, {-3460.355357987934, 5221.42403723717, 
      1204.3610379807553`}}, {{-3310.6310611011677`, 5291.264789437506, 
      1315.3146373894813`}, {-3419.9844370500423`, 5232.926015571309, 
      1268.071549989386}, {-3318.116447944371, 5323.870572828674, 
      1155.1272880725778`}, {-3310.6310611011677`, 5291.264789437506, 
      1315.3146373894813`}}, {{-3080.4556091112877`, 5440.993338534569, 
      1262.6157057146258`}, {-3109.045931846, 5447.055486007673, 
      1162.425746456761}, {-2883.272633241408, 5614.258202508541, 
      924.4768509631348}, {-3080.6364290952397`, 5441.312719968032, 
      1260.796877076332}}}], 
    Line3DBox[{{{-3601.8260245296906`, -4560.866439975565, \
-2629.2889334809233`}, {-3601.504640111552, -4560.623132256985, \
-2630.151072234721}, {-3601.058144167807, -4560.548662712236, \
-2630.8914510781956`}, {-3600.3187408043177`, -4560.594107287962, \
-2631.8244704355807`}, {-3599.755290607719, -4560.207667111151, \
-2633.264432327754}, {-3599.31112071374, -4559.808625302532, \
-2634.5622875390586`}, {-3599.1798451649342`, -4559.47868680323, \
-2635.3125501058908`}, {-3599.253494671918, -4559.244739194751, \
-2635.6166980835756`}, {-3599.3641812798505`, -4559.057719385144, \
-2635.789045396912}, {-3599.6995597826008`, -4558.828087827555, \
-2635.7282171983334`}, {-3600.3819428656843`, -4558.547095830754, \
-2635.2821349107417`}, {-3601.0982320880034`, -4558.145390649057, \
-2634.998256272177}, {-3601.6579391166592`, -4557.709007772538, \
-2634.988117632981}, {-3602.243259155353, -4557.305026847144, \
-2634.8867307995556`}, {-3602.996475691523, -4556.949905865728, \
-2634.4710363913264`}, {-3603.4375335655723`, -4557.017286135489, \
-2633.7511434365265`}, {-3604.7488259312536`, -4557.040610665018, \
-2631.9157404250323`}, {-3605.0677123579812`, -4557.280267189402, \
-2631.063861938755}, {-3605.344010079604, -4557.629543889617, \
-2630.080074759633}, {-3605.3929308052548`, -4558.181869113713, \
-2629.0556389286453`}, {-3605.1091976933817`, -4558.967777291971, \
-2628.0818420133123`}, {-3604.9956514958035`, -4559.3148338961755`, \
-2627.635493768235}, {-3604.8706301025336`, -4559.483838397245, \
-2627.5137597391354`}, {-3604.2993383397547`, -4559.906233742003, \
-2627.5644823913312`}, {-3603.384036233991, -4560.711392117362, \
-2627.422458460942}, {-3602.7727552337656`, -4561.410493287871, \
-2627.0471019508955`}, {-3602.363259417449, -4561.874107288405, \
-2626.803621600007}, {-3602.198259816653, -4562.156269161493, \
-2626.539846018784}, {-3602.10734860101, -4562.041130809169, \
-2626.864492119837}, {-3602.221182020113, -4561.694186732518, \
-2627.310867129424}, {-3602.079097637966, -4561.350573853, \
-2628.102130201669}, {-3601.863017636052, -4561.076949478295, \
-2628.873057628206}, {-3601.8260245296906`, -4560.866439975565, \
-2629.2889334809233`}}}], 
    Line3DBox[{{{3640.0290715442584`, 923.322563275997, 5156.110694356559}, {
     3491.9868515279536`, 1244.585278499371, 5190.842802792099}, {
     3442.4671924952086`, 1496.8273356939387`, 5157.202850937079}, {
     3637.393563113869, 1589.1381731740403`, 4993.177507717793}, {
     3858.3849653075363`, 1602.1415839161668`, 4820.147455687986}, {
     3891.610605185196, 1030.630341450288, 4947.825955598295}, {
     3640.0290715442584`, 923.322563275997, 5156.110694356559}}}], 
    Line3DBox[{{{5039.429679591496, -655.9983340332768, 3855.078258634696}, {
     5020.530758422824, -787.6904300119558, 3855.078258634696}, {
     4908.714933282785, -821.4371641724675, 3989.683242950859}, {
     4687.581426374206, -721.4877876856215, 4265.424883095711}, {
     4743.454088727852, -513.9079624776131, 4233.598422646288}, {
     5039.429679591496, -655.9983340332768, 3855.078258634696}}}], 
    Line3DBox[{CompressedData["
1:eJx1lGs4lVkUx8mZTA2eEYkaKkwlJcqlE5OVkTJCJZHbzKB05FbJRMe1nFyO
W2Ia1bv1vl3IJQ77JNfNo5AaGSRpZNTQTaWJx73Z86lP82F9+T3r+f/XXvu/
93Lv4N3758jIyPBpydEq/dS/0d+dhWPDeprejzBpP86vXN/BwHv22hXiwcJW
h+ejv3ZiMvBQxaxvkgHT9O3LhD4s6P49ZqHUjsnc6JQD1QsQNOjU5a/1ZiFX
9mpAbBsmHZ1vd5muROCzWTg96MbCX9nZWuqUd+Lo1H8sEARk+Kvd2sqCqvCp
gvQBJpLO0dZIDwRKPeeV3a1Z8ElLtXGkHIKL7XV9EfSd25sUZMPCpm6dJOX/
9O89m9IIRNAaZtaf4MTCCcmN/PJWTF7ebSodOoqgeYt97FNnqv9OuYTcxQTJ
amfywhHYnux6ILuXhVXv6sputGBi5TBwMu4UgudxhaIJFxYMdYZbKpsx0V9g
sT4uCYGrq8qlcHpe5Fdt8OY29Q19kVWViqC+a6Wk3p+FksvLmJJGTIJq8yLc
khGYv0reEB3FQqZP7YamekxGucD5tpEITiauVvCuYiGKvFbtLceksv5YCk+I
4Ir6cEb/CPVNnAhszcfEfEp66gSd/4Jjxrwv9DmI/dZEMnUBkxmxJJ0XhiC0
wuHeQysO1DmPXQ7nMHlgVDtkGoyg/cK+HY1eHHQbXXXVzcRkUk4wZOeDAATa
u4X+HDTl6OGsdEyKnaNjjnoj0NMynuRFcKCd+DAkQEzvvUqYVyRAsHrrY5O5
URwcIIofQpIx0Tn0cp7IH4G6M3nUFcOBRv6UzTzK3Yylz6b9qG+quZ5PLAe5
47cmrZI/39fBFvkU3RMc7Lw0zshQX2OdSLVISwQtuYtEd4UcTPNf/F5Aef79
Hs/pDQjWJhZIrKm+ScLbtIuUR1f//MMLfQTTvG0DmUkczAG7HuM0TLqcZIsL
ltNzOer7Boo50DrSZ+dEedyO+NzyJQg0Mk7f9Kb9lrxQ9zVUZ7DxaHnQIgQT
5nUtZlQ/ZbBGcTvdTz1z3DVMCYFOfFO1ZTgHgTwNW9VsTDyyN37okkFwzbOB
pxbCQV2hcunAeUzkRR0H53QzgNYILJR/4sBfoejHqEt0nqUbWV/CwPfiiFEz
Qw7Gn5iRN0WYOOOoYcdCBu6oftqjqMuB6MqhR/dKMNliaXTW6DoDNh/9Dbct
5eBPUW1qTSkmKmWHZ8IKGJCN0cqIUeMgqDtY8FpC5+SbL54qY2BVs41JxNcc
VCzu75Gj+bmWMFuULmVgrm1CjuZ7Fuo1as/sr8LEM713nTbtfzV2/g/bRhaU
XOKVDtN8xvjJR1pVMODKfy57+zoLWGBUvJnmX/Sd1Gi8joGUPb58hcssBM9o
Lmmg72VTM3/nUsqdEu8Yl+SwYH1OXSX/PiZpNUee9JUzsPlYXseKs5QbVGX6
0vfYkLr2mVwpAwszDXvXi1lwaBJXLmz/zBsMjZzChCxkxXv4HunApLDhqadm
NQP7/UY+1oWy8FXcbNz8LkzUr7e1JVLuNNt/IzaABVO+fYB5N82nS0eW4CYD
8TUtZ37xZcGeN5al8BiTnnWNCgzdZ+xFJwOpFwv7tuQIhL2YkPnyXl/mMeA2
MuJe5MqCgcysVfqTz/vXr/D6ZmwPC+LTgyqY8nKbTftC6JwvX5WJXSmfGf1t
zQTVsdt9Oty6ioGsyl7FFfTfGLp92KyN+qq6SNOMGxn4v//zX3suvqw=
      "], {{2520.9511687766358`, -5514.675199170644, 1980.1455356351778`}, {
      2521.8462534902137`, -5514.265936773583, 1980.1455356351778`}, {
      2523.2381457533443`, -5513.746980601295, 1979.817462223558}, {
      2525.5491862895897`, -5513.277675000986, 1978.1770082496948`}, {
      2525.7130552038593`, -5513.635401087484, 1976.9703884094056`}, {
      2524.4849260966566`, -5514.512726121462, 1976.0918352534104`}, {
      2520.495956014768, -5516.454628272304, 1975.7636903046555`}, {
      2516.0500839760853`, -5519.385203651004, 1973.244191336261}, {
      2512.0476970943278`, -5521.325251181881, 1972.9159962310646`}, {
      2507.906793332587, -5522.935033425688, 1973.6782469580642`}, {
      2506.2502865766046`, -5523.2555717666355`, 1974.8850801966912`}, {
      2506.248135326516, -5522.866574805184, 1975.9753973853042`}, {
      2520.9511687766358`, -5514.675199170644, 1980.1455356351778`}}}], 
    Line3DBox[{{{3632.342999304361, 4520.415615274849, 2657.017254476135}, {
     3583.9109960742408`, 4465.4605536223535`, 2811.237753421236}, {
     3658.589934206365, 4491.206645682224, 2670.5048678299922`}, {
     3632.342999304361, 4520.415615274849, 2657.017254476135}}}], 
    Line3DBox[{{{6243.072935020844, 
     1232.387691524598, -439.40254413111114`}, {6214.639544848828, 
     1324.7417629169984`, -557.7887406399523}, {6192.816632456668, 
     1439.2147328120893`, -515.263815559252}, {6120.331498632529, 
     1743.4222891912243`, -435.7003080073898}, {6033.178842113445, 
     2034.2972576817135`, 387.558274925628}, {6102.087653908535, 
     1817.1787302171754`, 387.558274925628}, {6120.841723438614, 
     1778.268906464187, 246.7184284071085}, {6203.178129531861, 
     1466.369172562866, 241.1560346640822}, {6209.575198942648, 
     1452.631911208069, 137.295554248364}, {6174.704016886918, 
     1596.8870369599074`, 103.90276719105113`}, {6173.518764530397, 
     1590.8340673146195`, -211.48693638689255`}, {6243.608471030748, 
     1279.7361531463318`, -259.6966091486647}, {6243.072935020844, 
     1232.387691524598, -439.40254413111114`}}}], Line3DBox[CompressedData["
1:eJx1lHtM01cUxwuMymSSQFnpQAwUdIC8UZ5Dj4oYKIoVCguGt2wEAgYmUMdk
dgg4M7KODtEwF+7PIXrH+9aOUMguUxCUKvORbU4mKtCsIJbwHPjY3V/8tT9O
cvPJyfec+73nXOeMo4eyzHg8ngkLTxb/nSsSepx4ZRgqT3yafd2ewI2SEcp7
gegPf/Mz3jmJwU8ikO/YSCAk+fzp5glEFzr006GfY7BcBu2cIwHjxMm22seI
rshlhUHHMTS0RWgMzgR++WC98dRviB5Nito8Jscw8lhSJnIhEO79KmrpLqLJ
1/oLd5RikD7bNCtkvErruiK/g+iFoU8mRJUYHPdUTU84Mf2qDXPaQUSDhDq5
hnGToWplItN3XWqm+n5EN6Wu6PWnMJj+vH/zITGBgeyzvMzriE7W9a5GKjBk
BhJlHtOfkYjtLFj+eILDtzXFGAz+H5uabyEQH9Z5f2QI0UWpKsa0AIN/SP5Y
z/sEOtRFRe03EdW59T6U5mJINbmSP+dOwLBXV1vP8t9WzCdZfITBY9ftA1pP
AtaT9tMlTL+rZiBRko6hqSClrNh7ra5tVMnFkBTWzxfG2Y0+BEyOXci+MYCo
v6+iqzMRg3WP81yn75r+w5kKWbwMQ7F/hFLpR2DrGY1ZOuMvZsKfNUVh2LJc
iqYCCVS83h1ffg3RgrTjc0ERGGaXFU7aYAKytF9ruxjPlo5W2+3E0PjTfJdX
6Fo/K4MJ5bJQDAW8vDeWYQTu9QqcY5jP7u1CD34ghu6v9u07zHiDi9OBWB2i
g43zTc8DMCg8D7ruDCGgz3xVVfYHoqN3fK6o/TDU2mvEvqxue1bg/tgniJoP
G234XhjOaXTKpCACRW2qvt4pREv6E5sNbhgCJE//rGL8qnv94j9GRB3Voka9
EIPwxwa3cFbX5YGqNGYV0Zq2+PFxGww+AkNiKeNTXJitgMfRhVyjzNMaw7o+
75wT7F5xeYPmOj5HLb33WCYxHj00OVrN/LFISf46xo6jRy6/iVtgOqJ7qQNZ
AQS61eeD08QctX+dsyFNgCE2sublTfYujtxSk5kfR+v9yloN72K4eLrz1qQX
gS+fZLR8uJ2jey+18uocMOwSu864biUQfdCXfz+Yo3+JBspXxRgePW0Zl7sR
0J35/TN5CEfnceWKhvnTzl3y6GRzmKao/O4Iy1d2x3WtBjE/VabyYDa3fd88
Wq9i+lcvf2/MTcXgoo2Izmf7mNBRbRHpyFHb3UPZijwMVmPG8To7Av7pJpza
lqNLMzZ31xViWByOPpvDuPVzkeSBFUd9WoSHHY5h8JVavdz2HgHpqrZn21sc
Pbc9MH2Y7aN0iXdLwvRvT+lQ6zKi//cP/AsFGQ0g
     "]], 
    Line3DBox[{{{3904.345560355304, 2223.9899255754126`, 4527.446106390765}, {
     4046.161351126181, 2204.5121725336708`, 4410.954328918362}, {
     4218.047112283897, 1763.0058237365208`, 4448.337072887416}, {
     4147.9225140985145`, 1531.6203515872526`, 4597.465780801244}, {
     3935.5974163624296`, 1662.461436713762, 4736.571401104493}, {
     3795.6683430248236`, 1906.2562889890423`, 4758.876178306062}, {
     3943.7719328260596`, 2111.678833609084, 4547.008811904897}, {
     3904.345560355304, 2223.9899255754126`, 4527.446106390765}}}], 
    Line3DBox[{CompressedData["
1:eJx1V2k41evblRIZ6uhICBFpMEVocNRK0kAjCqEkYylEJJmOkGTKsI1779js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      "], {{3380.7562687971476`, 1316.9690067334907`, 5246.32189026676}, {
      3390.3055418601775`, 1430.9591002751567`, 5210.182153837991}, {
      3453.156164882946, 1452.7553133727988`, 5162.657086268466}, {
      3491.624668611973, 1245.6010036290886`, 5190.842802792099}, {
      3414.159562345102, 1327.6940409485212`, 5221.930362086159}, {
      3380.7562687971476`, 1316.9690067334907`, 
      5246.32189026676}}, {{-3007.6301718589593`, 2363.9395536998, 
      5104.28880705013}, {-3503.0169097649004`, 2728.661153540671, 
      4579.420763487371}, {-3543.5502575040764`, 2610.9398440249306`, 
      4616.715108540471}, {-3327.208174524634, 2484.5441862570674`, 
      4841.9561790574535`}, {-3442.573411633756, 2432.970344597676, 
      4787.186783260142}, {-2985.383974757459, 2248.28694778402, 
      5169.187751831555}, {-3007.6301718589593`, 2363.9395536998, 
      5104.28880705013}}, {{-3719.1497695602725`, 1715.863484445005, 
      4889.9438835561305`}, {-3686.201212713129, 1636.065868354063, 
      4941.965477803252}, {-3719.1497695602725`, 1715.863484445005, 
      4889.9438835561305`}}, {{-1378.9968792107527`, 1110.7246885466582`, 
      6128.007177164524}, {-1471.7132013734351`, 1089.6731343881536`, 
      6110.195283548296}, {-1379.3199175786028`, 1110.3235076278545`, 
      6128.007177164524}}, {{-1324.306841774878, 876.5395771634592, 
      6177.839707251585}, {-1430.697833971485, 794.6816871841772, 
      6165.176267871294}, {-1324.306841774878, 876.5395771634592, 
      6177.839707251585}}, {{-1278.0202238686895`, 1078.7369528717527`, 
      6155.550713290532}, {-1336.3072519560292`, 981.6152646681678, 
      6159.4259546276035`}, {-1313.1514879539986`, 905.8825332544524, 
      6175.98766964675}, {-1142.3030162082123`, 1002.3608456840419`, 
      6195.0165652886835`}, {-1184.8172060993274`, 1108.0860521048003`, 
      6168.968080583801}, {-1281.0296458060009`, 1072.3739583920942`, 
      6156.036941891156}}, {{-219.0216117440884, 1313.5154784119522`, 
      6238.150391867868}, {-336.9965606915932, 1226.2443370206504`, 
      6250.6457133648855`}, {-250.51608728099686`, 1142.5816048255774`, 
      6270.531289797351}, {-218.63951631854619`, 1313.5791336431735`, 
      6238.150391867868}}, {{-195.68775711901696`, 1173.577784697702, 
      6266.75635197171}, {-142.81990059166722`, 1079.9693399242697`, 
      6284.98865478837}, {-57.52524540964819, 1155.5210475674346`, 
      6272.900098420922}, {-195.34636890515813`, 1173.6346583063264`, 
      6266.75635197171}}, {{1244.0491827999572`, 1558.3934948059516`, 
      6059.00693481376}, {1083.4325937184021`, 1478.5173316165685`, 
      6109.66234786981}, {1017.7446968098931`, 1541.5386346830405`, 
      6105.3802551434}, {1126.7470476759142`, 1777.7544903284538`, 
      6021.448642367601}, {1242.9495463399994`, 1557.0160040045084`, 
      6059.5867250603005`}}, {{918.1042012403128, 1412.8565680997583`, 
      6152.132531378226}, {1071.933738640433, 1478.10103358935, 
      6111.790988283748}, {527.878512142169, 1340.097674016123, 
      6213.959783464577}, {918.5151457075837, 1412.5894426419572`, 
      6152.132531378226}}}], 
    Line3DBox[{{{5544.63087811683, 
     3149.7887250659937`, -153.99059348067615`}, {5571.6131884602555`, 
     3090.5146588593902`, -306.0380883990966}, {5487.4855963700375`, 
     3240.9823472075554`, -267.11223244244565`}, {5496.062855482352, 
     3235.2753427199486`, -118.744411975844}, {5544.63087811683, 
     3149.7887250659937`, -153.99059348067615`}}}], 
    Line3DBox[{{{6098.198239038467, -635.4158592697502, -1759.55636828623}, {
     6099.8877351181145`, -631.0238253301861, -1755.2755455543886`}, {
     6101.218601809942, -629.2781071657677, -1751.272202363244}, {
     6102.488932870035, -625.6419117133908, -1748.1460787092028`}, {
     6103.32172493495, -620.216488381521, -1747.1717487061112`}, {
     6103.237404904131, -618.0555643770379, -1748.2317319856852`}, {
     6102.919883323183, -617.8888998107434, -1749.3987239132227`}, {
     6100.4369418606775`, -619.6544088584449, -1757.4160639882173`}, {
     6099.548148888311, -619.5641293441842, -1760.5301357895505`}, {
     6098.51566881087, -623.3577355077837, -1762.7664226581226`}, {
     6097.777611094701, -630.5366244567242, -1762.7664226581226`}, {
     6097.708230717307, -634.6924309156038, -1761.5145588898376`}, {
     6097.9065597710905`, -635.2509873173609, -1760.6264400528073`}, {
     6098.198239038467, -635.4158592697502, -1759.55636828623}}}], 
    Line3DBox[{{{2791.449252136428, -5414.894505664582, 
      1890.6459888806537`}, {2782.457355827596, -5417.425060190487, 
      1896.6418929837553`}, {2775.645362154605, -5419.98051975033, 
      1899.3203202926488`}, {2770.6611923884975`, -5421.181110261761, 
      1903.1672592637271`}, {2771.562416222102, -5420.119507093043, 
      1904.8779466975243`}, {2773.100651605458, -5418.880613420102, 
      1906.163519629193}, {2775.9565009732605`, -5417.418188689067, 
      1906.163519629193}, {2779.6738884747288`, -5415.88932347121, 
      1905.090443953341}, {2785.6845158537258`, -5414.566581524097, 
      1900.064262696194}, {2791.086121733941, -5412.79827886799, 
      1897.1733561664587`}, {2794.418366810347, -5412.95052539544, 
      1891.8261788161778`}, {2794.064945987077, -5413.656673294887, 
      1890.3270064451738`}, {2792.9367262569876`, -5414.275937028474, 
      1890.220677814984}, {2791.449252136428, -5414.894505664582, 
      1890.6459888806537`}}, {{2802.5729477650175`, -5412.282026455898, 
      1881.6486995606126`}, {2803.8919345703243`, -5411.336164140694, 
      1882.4039491869335`}, {2805.797718675645, -5410.459271881869, 
      1882.0848331523357`}, {2807.295129868499, -5410.204072622355, 
      1880.5849186880887`}, {2808.4552357288767`, -5411.053687959095, 
      1876.4037052114707`}, {2807.589813046679, -5411.834738892227, 
      1875.4460511024608`}, {2805.9267450466814`, -5412.81147800054, 
      1875.116181749107}, {2803.7052572069383`, -5413.774499818001, 
      1875.658867126572}, {2802.1278762183506`, -5414.222396918077, 
      1876.722912960439}, {2802.5729477650175`, -5412.282026455898, 
      1881.6486995606126`}}}], 
    Line3DBox[{{{3007.625676463883, -5418.6939264791645`, 
     1509.8866220392135`}, {2991.84695703505, -5425.718188487355, 
     1515.9972344294026`}, {2989.071217161614, -5426.573705978978, 
     1518.4086282399621`}, {2989.8456691224337`, -5423.073708681775, 
     1529.3489186690922`}, {2999.0067300565343`, -5414.228393423149, 
     1542.6934974019298`}, {3000.1548413634537`, -5411.732052647738, 
     1549.206509213193}, {3003.555010571974, -5409.09040799789, 
     1551.8414695364995`}, {3006.7015400107375`, -5407.283122738569, 
     1552.0466390684564`}, {3010.0680497106796`, -5405.893226376872, 
     1550.3620387263106`}, {3011.8761962654, -5406.209815359998, 
     1545.7395966575602`}, {3015.4001996155507`, -5411.246791206308, 
     1521.0468404417766`}, {3013.9204434657217`, -5412.8118836835465`, 
     1518.4086282399621`}, {3009.9998495004857`, -5416.464028205696, 
     1513.1530085093102`}, {3010.2498283163277`, -5416.913862535106, 
     1511.043965937627}, {3009.2805820268877`, -5417.77504675332, 
     1509.8866220392135`}, {3007.625676463883, -5418.6939264791645`, 
     1509.8866220392135`}}}], 
    Line3DBox[{{{2424.207001737679, -3622.700509933611, 4656.938384609868}, {
      2425.0348911481337`, -3623.4177396291184`, 4655.949242727541}, {
      2425.056689375446, -3624.5177099605585`, 4655.081644128025}, {
      2424.1974230583915`, -3626.450812450019, 4654.023533343772}, {
      2421.567758406545, -3629.1349478117304`, 4653.300206521974}, {
      2420.80046796458, -3630.496028133574, 4652.63767921045}, {
      2419.688274685614, -3632.769110244031, 4651.441812492741}, {
      2418.692744480312, -3633.4320088302943`, 4651.441812492741}, {
      2416.5217721356694`, -3634.1157722470152`, 4652.035981019468}, {
      2415.265366240304, -3633.849203086581, 4652.8966107003835`}, {
      2414.2852338688704`, -3632.374560471059, 4654.556429131074}, {
      2412.5828838428106`, -3632.315612270522, 4655.485024296022}, {
      2411.4843170231206`, -3630.8403871154046`, 4657.204651080215}, {
      2411.3879039813696`, -3629.8016533433006`, 4658.064192764906}, {
      2413.088747303257, -3628.4173005149, 4658.261937779214}, {
      2415.0082936177505`, -3626.4561880052515`, 4658.794280523223}, {
      2415.169292614905, -3624.7366850531835`, 4660.048813267912}, {
      2411.55370686523, -3619.4883074090117`, 4665.996871134186}, {
      2409.3313414246627`, -3617.2198563914317`, 4668.90315681921}, {
      2405.161645690939, -3616.675751441275, 4671.473820651673}, {
      2400.062268198664, -3614.354548517046, 4675.890930762096}, {
      2398.6270065436106`, -3612.74040744085, 4677.8744021108305`}, {
      2397.7408482634764`, -3610.8586247600088`, 4679.781245588836}, {
      2398.7875943710887`, -3609.221802620542, 4680.507424049222}, {
      2402.1152288443286`, -3607.2729418244094`, 4680.303199637351}, {
      2411.760552344097, -3607.491684327147, 4675.171491578397}, {
      2412.943936803065, -3609.2617811101554`, 4673.194265517204}, {
      2412.2352903239234`, -3612.4714396104623`, 4671.079606698495}, {
      2414.5921570845258`, -3618.147363721154, 4665.465470812484}, {
      2417.649559999568, -3621.4691789894996`, 4661.302959545003}, {
      2424.207001737679, -3622.700509933611, 4656.938384609868}}, {{
      2427.428928010227, -3625.7351833237917`, 4652.8966107003835`}, {
      2426.9942586947195`, -3625.263865012651, 4653.490568089335}, {
      2427.1361186003937`, -3624.230459162208, 4654.221474174419}, {
      2427.606650898538, -3623.318838419006, 4654.685836194654}, {
      2429.010370965448, -3622.3779584785416`, 4654.685836194654}, {
      2432.070284630188, -3622.114834598708, 4653.292591875006}, {
      2433.4273924298436`, -3623.247755872267, 4651.700819629167}, {
      2433.380803430638, -3625.146541525435, 4650.245596520836}, {
      2431.704511717949, -3627.463209974416, 4649.315811812817}, {
      2429.3517934164533`, -3628.785375377414, 4649.5139803286165`}, {
      2427.5180473418723`, -3628.909047850166, 4650.37513994954}, {
      2426.809447430628, -3627.849758246929, 4651.571318108447}, {
      2427.428928010227, -3625.7351833237917`, 4652.8966107003835`}}}], 
    Line3DBox[{{{2983.0202246397, -5442.349882093908, 1473.1706589064784`}, {
      2984.3311007303605`, -5441.428777790623, 1473.9180531649515`}, {
      2987.5121112076486`, -5439.61544608492, 1474.1671798347577`}, {
      2990.8387063484347`, -5438.08944207178, 1473.051507122361}, {
      2992.745621046135, -5437.465543461691, 1471.4808192348423`}, {
      2992.861289615565, -5438.259882204727, 1468.3066589521711`}, {
      2990.1745460343095`, -5443.112185919649, 1455.7470953917543`}, {
      2986.7883959234664`, -5445.706358719887, 1452.9938136739252`}, {
      2982.4343579641522`, -5448.092140177169, 1452.9938136739252`}, {
      2977.253932320116, -5450.140969517862, 1455.9313599436284`}, {
      2976.390625824591, -5450.143161114832, 1457.687228405573}, {
      2976.2162906063477`, -5449.6204324894225`, 1459.9956929125444`}, {
      2981.120022112872, -5444.924074186688, 1467.4940988296999`}, {
      2983.0202246397, -5442.349882093908, 1473.1706589064784`}}, {{
      2973.6512544838915`, -5474.43718699325, 1369.5800058309142`}, {
      2972.580431325762, -5474.5614478102125`, 1371.4066586717786`}, {
      2972.389566566672, -5474.209934747403, 1373.2223217230387`}, {
      2975.640940929009, -5471.768735585516, 1375.907549559926}, {
      2978.5104090249783`, -5469.218778043741, 1379.8316532184595`}, {
      2980.6701493231085`, -5467.775884148658, 1380.8859589613157`}, {
      2982.009677365077, -5467.212872376366, 1380.2229465645005`}, {
      2983.2033207257045`, -5467.312764017506, 1377.2446324536497`}, {
      2983.2974404615825`, -5468.393192377486, 1372.7439547471802`}, {
      2981.5794255953665`, -5469.739179610738, 1371.1130974130674`}, {
      2978.6133592805004`, -5471.499330141063, 1370.5368386809687`}, {
      2977.225120141743, -5472.565137196633, 1369.2972990322378`}, {
      2975.096360582748, -5473.771661871905, 1369.1015772890485`}, {
      2973.6512544838915`, -5474.43718699325, 1369.5800058309142`}}, {{
      2986.740857690778, -5451.914458662246, 1429.6228434206378`}, {
      2984.920117814797, -5453.062255174254, 1429.0478023699518`}, {
      2981.1625259004927`, -5455.168577587852, 1428.8525026274458`}, {
      2979.5420972084758`, -5455.778080080647, 1429.904934558252}, {
      2979.575726793532, -5454.956952383988, 1432.9643491519205`}, {
      2983.1681918576674`, -5451.9625966025615`, 1436.8803270750507`}, {
      2988.7957194296714`, -5448.802351205452, 1437.173188943072}, {
      2990.1040457051695`, -5448.210374784641, 1436.695930928521}, {
      2990.2318759251866`, -5448.443291740127, 1435.5461381972173`}, {
      2986.740857690778, -5451.914458662246, 1429.6228434206378`}}, {{
      2976.1578491576565`, -5468.51142073241, 1387.689081969179}, {
      2974.897915597657, -5469.19693314807, 1387.689081969179}, {
      2973.9780311200307`, -5469.597931905326, 1388.0802655871173`}, {
      2973.721947808116, -5469.422661098795, 1389.3189775509732`}, {
      2975.283762170789, -5467.768988850171, 1392.4807033245384`}, {
      2977.061245873046, -5466.557786731109, 1393.4367558846338`}, {
      2979.3619955148415`, -5465.376191634133, 1393.1542892302352`}, {
      2982.0633827135352`, -5464.61112963391, 1390.3729260023385`}, {
      2982.1865483149277`, -5464.836829771627, 1389.2211865002248`}, {
      2981.391180216919, -5465.488916960153, 1388.3627836810133`}, {
      2976.1578491576565`, -5468.51142073241, 1387.689081969179}}, {{
      2981.869448009216, -5461.852406595617, 1401.5835949364111`}, {
      2980.6000936235414`, -5461.045601880733, 1407.4152571542074`}, {
      2981.3828415821285`, -5460.371942444652, 1408.3707935351717`}, {
      2983.854948992333, -5458.895355651912, 1408.859407256786}, {
      2985.289438142383, -5458.2371117818475`, 1408.3707935351717`}, {
      2985.8885690428224`, -5458.427354941704, 1406.3619569952546`}, {
      2983.9334269287024`, -5460.850502335327, 1401.094852178656}, {
      2982.767823935444, -5461.707302393547, 1400.236816212317}, {
      2982.2415074787473`, -5461.944580183233, 1400.432320420338}, {
      2981.869448009216, -5461.852406595617, 1401.5835949364111`}}, {{
      2994.8668263578106`, -5451.202778039566, 1415.2648709959628`}, {
      2993.1608541542637`, -5451.095575612518, 1419.2810915023688`}, {
      2993.6689141128168`, -5450.2169415116905`, 1421.5820106956696`}, {
      2995.1394401563525`, -5449.309854529789, 1421.961860720749}, {
      2997.1878861878035`, -5448.282578185123, 1421.5820106956696`}, {
      2997.8389037389466`, -5448.272557939319, 1420.2470671100216`}, {
      2997.8865215840688`, -5448.944518280984, 1417.566129110097}, {
      2996.9548985543765`, -5449.931025359986, 1415.7425065489713`}, {
      2996.2462812704694`, -5450.444684326009, 1415.2648709959628`}, {
      2994.8668263578106`, -5451.202778039566, 1415.2648709959628`}}}], 
    Line3DBox[{CompressedData["
1:eJx11Ws8lGkUAPAxM8nkkjQuyYwSTRdCJVJtpZVLF5HKSqSSkij32LJGl/d9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      "], CompressedData["
1:eJx11H081HccAPDjhlKncokd2vL0mkvRkzyeytaVq5iWLLl0FstEluE0K1tc
7veLVpvfr5SXo7p56MGMX6nNZxwhMnJdDyRqSbrzdFrryT5/8df++P7zfn1e
34fP9/P5LJDEB+8yZLFYHrjYuP5YLNuaomegobglSn1LDpqk1IHjgxRw7dfs
skWfVV9g4zcsB9OKirD1AxRI20tdT40xcFrxdn0ql4CjDrOS9/RTMDwhqeaj
fzNf3ZXpQYDts8QP2Y8pCNL3Zz8eZWDnStmqgM0E6GQTJvUPKdCGO/zrjz56
kFctSZryG96+e3JHGAhrtf5y8U8E3BzPLJz1iAITZVuOH3pTvDjOUDnlLgl3
Q43QP5bqN3hcJiBF87nsci8F0TvUUc+GGZjdxzMeaSVgxrYgT6tuChJ3VLnO
wPhmYWjx0vsEjJT93LxQQ0HA2S0NNRjP9uY+bXtLQJ5yt7WqnYLBlmkd76PH
V1mUHrIhYUZIWIjTLQqWGt47UDzEQOFFpTrNiYQ2V5ZuNe6Tc6LIU69jQKyK
DDZaQkJuhIpzqIcCU/5JX0v0sKPhAweWkVDHBB86/ZyCLkH3KqGWgbEVVVZn
1pLwTrvx5osxCmwrIxo3PGdgundIrXorCVzWXtX37yhIuVl06vdBBgT1haKY
7SSY5Niaco1p6PYcMx59xsDXYQ/SR2NJaJSz99/j0BDv2PhFzwADlbyKHIWM
hMQtGX+yLWn49tJnNWvQvYUrovJPk6D/VRXizqNB6p6pIZ4yYPx69db9FSRk
VBZ7cj6goVNU8uZlPwNeVM6J6GskrN78MiDNkYa7GZeyxeh2lmkK3RUSvBRv
jtS70nCjVh8oQj8T/arhOyUJLZusi2LdaQgYnxMoRWefFORHZJPg257UwfKm
YY8iSDWIXn9j7+7keBJaneeQ03xpeDQmLErE+9ReFXfwN5KQUvDpRYUA9985
HKpB79VMZ7msIEFcyv87y4+G8DsS0VJ8V5e5j27CYSq+lTb6LQGd038sqmUB
CdeORJWfw3MlEuNz72HeOMlpM1/PJ6EbHmRELaNBvfKOPgv9tnB4ec1sEooe
NT4R4btOxT1INcT8/3C7IlY9QUB/91eRNS40ZHEqg+To8Z4DtXV9Ux6Tai5W
o28u13KamojJfTofP6G5+L/+gSVxkjICXjHXFz7Ec9PZXj1O6OPN7Wbncgng
J/i/7POggQps+cQKXWDwLtGHICbzQ0p0BfboUp5qaPgAMZmHRX1v94WjjyiT
naTJBDhcvxSyA/PQa13Ft8J6iytst5DuJkDlY8H/C/9lXanerw29JJq0N9s1
df8TM9ld/2DdPj1r4L4I+7fMRdmwF+vBz7ywKR3rf9wuzPCYkIDj0eNrF2L9
lNYtO2KM/dJzYfl9/XIC2stsDNyw3np0fJkCXSvKC5XOJUCwzS24ah4N3Dfz
xB7Yj86b1nZuG5FPxoeXayJ16MdEVPC1DjmscQ4wip2LdbLEMf8kzo0Ylxi2
9qocCiwV5Waz8b/K3aoDcf4cHuBfMDgvh1qqZJV2GvbF9nVDg+idH/nz9v0i
hy1gdiGRRcNhuyDBjzjfKktEMtMSOUxnOSeUj1PQauP24jw6bZ8ZYnBFDhZ5
iqDqIQr+b07+B4c186c=
      "]}], 
    Line3DBox[{{{4487.908605137364, 974.0841945573075, 4426.923313526861}, {
     4485.270188893155, 972.7574330274891, 4429.8880346026945`}, {
     4481.467177869982, 975.1434175869067, 4433.211131765662}, {
     4477.678986189328, 979.6161554988361, 4436.051935507991}, {
     4475.599617645516, 986.3421342737923, 4436.659886298196}, {
     4476.35270654398, 989.5396323525197, 4435.187871175247}, {
     4480.6797493628965`, 992.5797836319801, 4430.136346296166}, {
     4483.8124631942765`, 993.8402447618615, 4426.682849809252}, {
     4487.0826730247045`, 992.2894488222889, 4423.71613298112}, {
     4490.2472384179055`, 987.2941785830365, 4421.62228518181}, {
     4491.357541245525, 982.2308760574253, 4421.62228518181}, {
     4490.491979025172, 977.8705807330891, 4423.467486446275}, {
     4487.908605137364, 974.0841945573075, 4426.923313526861}}}], 
    Line3DBox[{{{6337.530489335351, 723.499022435253, 8.37563609884283}, {
      6337.879528776674, 720.4350195774138, 8.37563609884283}, {
      6338.0409562509785`, 718.9967695573324, 9.704238831247999}, {
      6338.036617382883, 718.9962773496984, 12.242543485974606`}, {
      6338.1117125409355`, 718.2709143275091, 15.50815857968583}, {
      6339.059660831176, 709.6167281701537, 24.091606920585267`}, {
      6339.129088616486, 708.896346081992, 26.871143777717094`}, {
      6338.950677003349, 710.3382814574816, 30.619118452557792`}, {
      6338.402327670977, 715.0210693245322, 34.849576713996214`}, {
      6337.430866637435, 723.123487398557, 43.31423049796248}, {
      6336.788309824039, 728.5237478692494, 46.69855745026076}, {
      6335.774631644911, 737.1702509450191, 48.513174771825}, {
      6335.100145363819, 742.9365932124833, 48.63184838293899}, {
      6334.313377509774, 749.7826556823135, 45.9749359836604}, {
      6333.904384608679, 753.5682112394047, 40.048566071782254`}, {
      6333.928668002583, 753.7560868937038, 31.82891964398919}, {
      6334.371705553233, 750.3388782106363, 23.245509936733153`}, {
      6334.896558946025, 746.0176597949779, 18.892557472178108`}, {
      6335.052747695065, 744.7581327696729, 15.990547039737415`}, {
      6335.631727718464, 739.8939834863155, 11.882505118626103`}, {
      6336.369551131802, 733.5912587402534, 8.858136918336871}, {
      6337.530489335351, 723.499022435253, 8.37563609884283}}, {{
      6324.241724642191, 814.1090662879426, 170.3136121469332}, {
      6325.164914282765, 806.362217963016, 172.86546956540647`}, {
      6325.43109168435, 803.8214339178458, 174.94655391629738`}, {
      6325.343506812217, 804.2422038357793, 176.1751639644329}, {
      6324.96682229349, 806.9091174030983, 177.4994736283609}, {
      6324.523913971152, 809.999467766767, 179.19769480820776`}, {
      6324.355649032893, 810.9764406382473, 180.71117086513806`}, {
      6324.2373959217775`, 810.9612769611126, 184.86983878967905`}, {
      6323.952239172188, 812.6412566847832, 187.2323545834051}, {
      6323.521194548358, 815.7273482433131, 188.36742134890036`}, {
      6322.534825938433, 823.0334984336589, 189.69165836093612`}, {
      6322.126361474328, 826.2739603917279, 189.2176047040607}, {
      6322.142182390222, 826.5622086017381, 187.4215327906982}, {
      6322.264649488239, 825.8599555409378, 186.3832762488657}, {
      6323.134514979117, 820.2554842116509, 181.56138383618384`}, {
      6323.234628666868, 819.982315914766, 179.29451306503185`}, {
      6322.903802371909, 823.0815297965923, 176.7427261828733}, {
      6322.950378623594, 823.087592841941, 175.04003534448907`}, {
      6323.212776052886, 821.4102869129363, 173.43303986024094`}, {
      6323.884493553512, 816.9187744277767, 170.12441970768964`}, {
      6323.959627422479, 816.3561863342467, 170.03204922704901`}, {
      6324.241724642191, 814.1090662879426, 170.3136121469332}}, {{
      6325.136858540872, 811.1102785967279, 150.1916331972288}, {
      6325.0444909361595`, 812.0578322542336, 148.9562175697738}, {
      6325.013885752017, 812.4690906131373, 148.01017479362005`}, {
      6325.073736941175, 812.2467392485948, 146.66678845945717`}, {
      6325.195228803244, 811.510504916293, 145.49813722236507`}, {
      6325.371721422675, 810.2539214706495, 144.826995300522}, {
      6325.503861146356, 809.221684122936, 144.826995300522}, {
      6325.614857694615, 808.259678754315, 145.35010771617826`}, {
      6325.730341435364, 807.1804367850829, 146.31841960421897`}, {
      6325.7584390073935`, 806.774484555881, 147.33903894193256`}, {
      6325.738198839652, 806.7326329164885, 148.4331119679994}, {
      6325.650956335379, 807.2095774369718, 149.5538923175246}, {
      6325.513264222049, 808.1681552463721, 150.19942408875352`}, {
      6325.320621565064, 809.6470640878167, 150.34745098511343`}, {
      6325.136858540872, 811.1102785967279, 150.1916331972288}}, {{
      6324.920345023034, 812.0082303806602, 154.39868169049547`}, {
      6324.915886692719, 812.1928053074386, 153.6084739488075}, {
      6324.956809452071, 811.9680274926692, 153.11097575138334`}, {
      6325.0660211522045`, 811.1965721725127, 152.68804590202728`}, {
      6325.234427693338, 809.9389863137407, 152.38754270608}, {
      6325.4097545018285`, 808.5308077477266, 152.58787820768617`}, {
      6325.559038255653, 807.310047738788, 152.86278278973546`}, {
      6325.662490969704, 806.3863821595314, 153.45599697186174`}, {
      6325.701524420789, 805.93695835421, 154.2061383320046}, {
      6325.684237156573, 805.9347558403723, 154.92511454694628`}, {
      6325.594523922884, 806.5235737912622, 155.52277593907266`}, {
      6325.416861434649, 807.858506429635, 155.81993668173968`}, {
      6325.219108671035, 809.4096620378998, 155.7976774617952}, {
      6325.082965755835, 810.5591604077931, 155.34804081281396`}, {
      6324.920345023034, 812.0082303806602, 154.39868169049547`}}}], 
    Line3DBox[{{{4555.546937436852, 3185.8801824029288`, 
     3128.0941888283996`}, {4634.219915499943, 3192.9566373740968`, 
     3002.8065832871075`}, {4612.812353936926, 3759.8841377856775`, 
     2296.3116399553705`}, {4491.415727445621, 4156.670402741076, 
     1799.1913784337844`}, {4341.421951753622, 4266.303470018662, 
     1907.4915015638999`}, {4310.128626154076, 4398.798032466842, 
     1661.680109278861}, {3714.547097420422, 4751.553858505543, 
     2076.7015868358653`}, {3249.921193746912, 4941.2651181074125`, 
     2389.5030774220777`}, {3293.4310373129674`, 4852.206418466464, 
     2509.425943329764}, {3390.096146186889, 4811.708232080999, 
     2458.154937942835}, {3658.724393864392, 4545.118062184905, 
     2577.4893405292996`}, {3632.342999304361, 4520.415615274849, 
     2657.017254476135}, {3658.589934206365, 4491.206645682224, 
     2670.5048678299922`}, {3718.794456933103, 4191.033887057487, 
     3048.543329956037}, {3833.0359302511793`, 4046.2429508645123`, 
     3102.1874911073182`}, {3956.6688217458964`, 3917.728824098492, 
     3111.9104077482343`}, {4175.358502610053, 3417.491335647547, 
     3402.1975303406302`}, {4343.567552715499, 3273.112921243846, 
     3332.8615947536223`}, {4382.576073636162, 3362.8688981718947`, 
     3189.3499999999995`}, {4555.546937436852, 3185.8801824029288`, 
     3128.0941888283996`}}}], 
    Line3DBox[{{{5876.119763986633, -1744.3051565826897`, 
     1765.3412502982821`}, {5881.918995525867, -1858.3247765900428`, 
     1624.029447018599}, {5942.90133782728, -1767.8917802583878`, 
     1498.097470842783}, {6026.349726171628, -1480.2134642958029`, 
     1476.4453149333215`}, {5949.908478816625, -1790.7191980899795`, 
     1442.1260476456328`}, {5996.538468992442, -1649.8453660941173`, 
     1416.8099194569295`}, {5968.518527532095, -1792.5339798368225`, 
     1360.6697643574216`}, {6045.569626669418, -1475.6142376965004`, 
     1400.5228311596964`}, {6106.496402577681, -1225.7410514311023`, 
     1376.9801197339118`}, {6027.584193653838, -1308.716816808885, 
     1625.8237210152968`}, {6033.5009854608115`, -1287.9643538513594`, 
     1620.4404869206587`}, {5964.089535633751, -1419.0133254670218`, 
     1761.7749240725964`}, {5876.611800629145, -1744.4512159019741`, 
     1763.5581617980133`}}}], Line3DBox[CompressedData["
1:eJx1mGk4lWsXxx3KkDKUU0ISFRFSiTStTEnmzENSaCBkLJRKklRC7emRQilF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     "]], 
    Line3DBox[{{{3614.4797276728623`, 
     5229.1899546582945`, -528.1310503489638}, {3620.1854497247805`, 
     5225.229935173138, -528.2453257450594}, {3621.204988551361, 
     5224.558522777427, -527.8980611486271}, {3622.4486191077926`, 
     5223.918200468253, -525.6979424922379}, {3623.8212611727035`, 
     5223.152259463428, -523.8450453471524}, {3628.2114748756203`, 
     5220.3121703448105`, -521.7624246059945}, {3629.762678951346, 
     5219.5110514155895`, -518.9807016089072}, {3629.621839404218, 
     5219.930465478027, -515.7372669240445}, {3627.4935096025547`, 
     5221.729094590167, -512.4936980118767}, {3623.9998689507647`, 
     5224.324663160129, -510.75478827088244`}, {3620.7271194386544`, 
     5226.615823180504, -510.5250762114007}, {3618.681534886393, 
     5227.930662703776, -511.56487861649236`}, {3617.68829112809, 
     5228.640498568545, -511.33516890938836`}, {3616.8833164577513`, 
     5229.310417859426, -510.17773255646034`}, {3616.973150723065, 
     5229.440300749856, -508.2057239363961}, {3616.1255442404085`, 
     5230.048660355227, -507.9771142566361}, {3614.1104352250545`, 
     5231.40762010547, -508.32446753996857`}, {3611.737933922242, 
     5232.854367014826, -510.29203431071653`}, {3611.642628168776, 
     5232.716283594225, -512.3782897449655}, {3612.7720578275407`, 
     5231.617789730611, -515.6229730672846}, {3612.670435303915, 
     5231.470631206927, -517.823377900633}, {3609.9538147841818`, 
     5233.023658687304, -521.0667258078034}, {3602.6329280726254`, 
     5237.416951364071, -527.5541234140215}, {3601.232917443414, 
     5237.8285714488275`, -532.9981429951938}, {3600.2989835332805`, 
     5237.997476313551, -537.6275265479873}, {3600.2434871426835`, 
     5237.916735809715, -538.7845506426445}, {3601.6248489951736`, 
     5236.871391413744, -539.7130412476265}, {3604.4796487873314`, 
     5234.89509336792, -539.827299152631}, {3606.806041680808, 
     5233.36408165782, -539.1328746316805}, {3611.4327819365417`, 
     5230.576449872238, -535.1980508148628}, {3612.5716188915626`, 
     5230.097546862583, -532.1838391770219}, {3612.7144293704114`, 
     5230.3043006141725`, -529.1739456886701}, {3613.4647228854537`, 
     5229.86785960441, -528.364038839829}, {3614.4797276728623`, 
     5229.1899546582945`, -528.1310503489638}}}], 
    Line3DBox[{{{6130.624028970588, -1449.2181099648774`, 
     1001.5136420230403`}, {6205.303455868758, -1260.6036021740877`, 
     770.0008238755117}, {6208.2236854931025`, -1122.6297208374278`, 
     940.9967432328986}, {6161.802253196309, -1221.932476738101, 
     1107.6496308840485`}, {6130.624028970588, -1449.2181099648774`, 
     1001.5136420230403`}}}], 
    Line3DBox[{{{-1515.0811822519208`, 6193.966912208853, 
     164.6711621628386}, {-1527.5366872483078`, 6191.020783853834, 
     160.33406687389967`}, {-1530.6814403867404`, 6190.292904852758, 
     158.43538173394612`}, {-1529.8243271585468`, 6190.535823146629, 
     157.21781187952013`}, {-1523.9945678398203`, 6191.996684079197, 
     156.30518720421756`}, {-1520.1195785946513`, 6193.01762010841, 
     153.56729404397862`}, {-1515.913297592843, 6194.089814236799, 
     151.8944941369681}, {-1508.3438003235499`, 6195.933697864158, 
     152.0469720082863}, {-1501.402947930064, 6197.536287142867, 
     155.39255932739786`}, {-1502.3525464230954`, 6197.201825872283, 
     159.49935943846768`}, {-1508.058934418351, 6195.704566304638, 
     163.75856433430297`}, {-1515.0811822519208`, 6193.966912208853, 
     164.6711621628386}}}], 
    Line3DBox[{{{3858.3849653075363`, 1602.1415839161668`, 
     4820.147455687986}, {3883.434193196714, 1346.0248887647558`, 
     4878.008769563295}, {3951.773925060587, 1266.2402357376716`, 
     4844.371187327644}, {4078.828389775395, 1209.4947119335802`, 
     4752.693467768247}, {4076.71919769098, 1259.356859985795, 
     4741.53926192587}, {3921.1586918773723`, 1596.2049108268254`, 
     4771.2113855672815`}, {3858.3849653075363`, 1602.1415839161668`, 
     4820.147455687986}}}], 
    Line3DBox[{{{4338.399829031359, 1080.3442218475993`, 4549.610640019778}, {
     4335.660256346069, 1058.2562767710156`, 4557.406881573763}, {
     4262.966522731917, 1040.5130507499046`, 4629.499185364015}, {
     4189.709003827124, 1210.6177465161945`, 4654.949712410433}, {
     4201.518439524172, 1247.205736761076, 4634.601853722565}, {
     4280.746750266003, 1204.6073584097971`, 4572.957693020134}, {
     4338.399829031359, 1080.3442218475993`, 4549.610640019778}}}], 
    Line3DBox[{CompressedData["
1:eJx112k4VV0bB3CZU+aUoYyRJNH8IC31FqUQjVSGpCgR0aGSBtHe+5AkQ4az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      "], CompressedData["
1:eJx11ms8VdsWAHCnnVcSElK21xZCRGrL45jsyqPyTueikDYqjlBOKInSQ7QW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      "], CompressedData["
1:eJx11mk8VvkXAHBLKpIlS1rsSyQREwr58aRkTYaxziCKlKJEIYWy3vvclETl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      "], CompressedData["
1:eJx1lmk4llsXxw0VKWSoJCqSEpVSxsp24hANEppNB6VBkkTylAz3c8cxRZMi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      "], CompressedData["
1:eJx11mk8llkbAHCiRW9NZB2iNKbSghpblI54FCNbZAmhEGWrLDEtU8pbRvUc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      "], CompressedData["
1:eJx1lnk4lXkbxy1JG2USmUm2SYqS6hQm+VUoKhqTZSINvULSck3lZToa2uQ8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      "], CompressedData["
1:eJx11H041VccAPBLj5fcVkpjMx4Jjy7mpZC6FyeupbjbRWrdSpbXskqbcqfS
5iWldTvXy5A41Cbx7Fpm/SKe+2VLuoWmlYu0qd2UlyeJrPd9/9tf++P883nO
8z3f8/2e77Hdtjs8Tp/H4y3FNQuX2rfOqXA1ByFvvZSLKhmRqBW5690p6JvO
M+9ElxbktUxXMMLTKR+edqYQY2c7RoM5eG3jfdi1hJE76VJPHwGFruYOsxVr
OND6N24bPMGI2aReULALBaOFXskWazl4csR4S+UBRpIaBEONvhR8Xf60z0Y3
UIzr2nYzYu2/puZaKIVbyrKeB+hPL/VWDSQycgyu10aHUaivyrySH8LB9ku/
Pbu6mZFHb+q8HTdSOOhvZFSKXhPe0Ta5jpFlZZ4xWQkU+NeOPfNG73bLnJJL
GAm4mOlXmk6hWZtiqYeu6I1yvBLEiOHsudJHSgp7Za4mFXiu1vSeuEuM+cgb
bgWcovALf8XUNHpUit5XoyJGFtw4ZE3PYD4T9stuYJyoS4Ftzu6MlD2W2X1W
TeF9uU8oh+6kNxbp7srIgXPdokAVhdrKE34i9Bk7zl4f3U4yd09EI4W+DbEJ
OzF+yXhTfIAXIw4fjqk7WyjsWa6rKcB6sj/8VPOFjLzUxCZ2XcZ6RjLJu+gK
zQefiLwZGVPz9Lr7KAwuu71Yhn0hG+/uXefJiLRfYbh3nMJscV/LduwjtF+0
H0MPEaSZvOIpwcLCcX/9RxgnIVs+vJKRam3G0x8NlLBaFfPlcBAHfN0L4wNY
h3ksK2QPeoZsTo8S3cA53MjnY0a+OBvf1v+Cwu9uNPMM+sgmjxnzMEamvk8W
mExRsFWNXdWguxedb4pZj3E81WEDDyjcFQorO9A1RWLHiC2MeLhI/17ST2GJ
f7b+DnSjWumhijhGxC5bJXd6KAh/WJlyV8zBG43Dy9tJjKSu0CRG3KQg6n45
7Ir+16zS0sfJjGjzRrOeayn0RKnc3QI56LLxCF2wj5GRKMnOZB2F8Z8KHU4G
cBDEOZnbpTKSrW41FE/guX58639WcdB32Sm/PP0/v11WvygQ3TjdvOx0LiOq
AV3Xrkc4F6+1STrCQbxsMOebYkbUtKZI0UuhaihLYYP77WURBcWljHR9Kj/c
dJ3C/fKHgpPoC+d45nagVxWFOLe2Unhb8Pm9fMzn4KaR/Y7fMaIksYvOX6Aw
5Fv3bT3m36wxGdHgHAmOP5+Ac/gOi+MU7lgfnvGvl8tyGCnfdTx6oIhC63wu
eh/2sWRLWP2br7E+qjOK3hycxymzuDrse7vV0HLhfkY6o2vafFMpNHmJkoLx
nViv1US5pTHiE9v26uckClk2KfX3cP+tm35BJzGOYaHn2cREClLhe6aO6Bfa
0za3YB1yLKz88+LwXqM5q57huYZLD73gU0Y2bJfsqNtKgUptG0S4v3pBJH8E
7/XE1HZmMc71vOSOBDW6lcf96SNYN0vLxvZQ/Aecs/s95Oi8o1dP8fGfqR2f
zHjHi8L//Uv/Ag3cZeA=
      "]}], 
    Line3DBox[{{{4771.654083491909, 4229.032079207865, -185.5226954466847}, {
     4259.342623302204, 4722.185536993574, 496.76741190401515`}, {
     3912.1081677374605`, 4903.538041933708, 1156.952049667911}, {
     3944.8331243582006`, 4833.971675022547, 1326.2063022192415`}, {
     4442.532721649705, 4429.628676869595, 1153.3024287350524`}, {
     4550.026969657767, 4285.230297325258, 1273.5264285724982`}, {
     4553.058130593881, 4283.089406423966, 1269.8899503546525`}, {
     4583.099917185168, 4266.338472225575, 1217.1133388083663`}, {
     4530.688062999867, 4377.782280176727, 997.8485221521206}, {
     4228.01102994151, 4692.935964993429, 887.7434590939694}, {
     4497.857757458458, 4490.0143574097965`, 544.8489276163259}, {
     4736.270941387223, 4249.612870048872, 443.10463153298207`}, {
     4809.396656495407, 4178.289189351235, 315.30452048995164`}, {
     4771.654083491909, 4229.032079207865, -185.5226954466847}}}], 
    Line3DBox[{{{5371.078395544623, 1589.2865789399548`, -3051.80253766772}, {
     5010.3801803804, 1658.6418421654103`, -3582.291358539866}, {
     4920.987283920183, 1791.0928945494313`, -3641.9341131588867`}, {
     4726.030300755499, 2502.304237154396, -3477.2007119319464`}, {
     4778.297805897626, 3091.219481637884, -2880.980021537176}, {
     4819.941436462129, 3027.452553688502, -2879.324447546743}, {
     4889.671848767678, 2916.757550002774, -2876.012568795672}, {
     4865.122354768524, 3036.1317010170005`, -2792.9021567557975`}, {
     5038.482652813411, 3063.445470629743, -2432.4489914027895`}, {
     5147.555285815295, 2896.5540804618017`, -2408.4149826047624`}, {
     5188.520877851617, 2476.6504159461297`, -2762.836858609597}, {
     5300.81614493333, 2228.2579291069687`, -2761.164335744264}, {
     5326.172980927055, 2007.2831870471716`, -2879.324447546743}, {
     5442.695712348549, 1980.9792334631563`, -2672.1898041438662`}, {
     5270.9693030135095`, 1918.475932027788, -3037.127984612995}, {
     5371.078395544623, 1589.2865789399548`, -3051.80253766772}}}], 
    Line3DBox[{{{-3010.7676244442982`, 4039.254646451382, 
     3912.4818086118885`}, {-3136.2330354725796`, 4235.798254749243, 
     3593.0306403193545`}, {-3299.1729586559945`, 4011.7181484926264`, 
     3702.6192048235184`}, {-3093.482217121547, 3907.6712207735645`, 
     3980.9907174836253`}, {-3010.7676244442982`, 4039.254646451382, 
     3912.4818086118885`}}}], 
    Line3DBox[{{{4700.464620653186, 264.1683774196826, 4303.912302586933}, {
       4889.9438835561305`, 0, 4095.8836049956285`}, {
       4961.838088568091, -10.103395316418041`, 4008.4753203977734`}, {
       4983.926825035576, 0, 3980.9907174836253`}, {
       5108.524520927424, -190.2971694705956, 3815.046250882774}, {
       5128.065761287383, -481.7350539253235, 3762.8030211175833`}, {
       5039.429679591496, -655.9983340332768, 3855.078258634696}, {
       4743.454088727852, -513.9079624776131, 4233.598422646288}, {
       4687.581426374206, -721.4877876856215, 4265.424883095711}, {
       4598.987168896003, -751.7394308419324, 4355.688067165434}, {
       4561.4704434884925`, -632.956329682535, 4413.634258510293}, {
       4632.398866433145, -142.88136616627472`, 4382.725105168172}, {
       4693.769067418801, 121.54438072393816`, 4317.588747814788}, {
       4700.677486206689, 143.61868657907868`, 4309.3872573180115`}, {
       4700.464620653186, 264.1683774196826, 4303.912302586933}}}], 
    Line3DBox[{{{1119.647767074117, 6222.740665504156, 843.6238365331069}, {
     1050.7895087311408`, 6256.906669880739, 659.3739627423}, {
     893.3369832816293, 6263.85034609078, 808.6665357733685}, {
     1100.5991166854824`, 6189.155440042803, 1082.058322522273}, {
     1025.863441574937, 6207.945187551675, 1047.2986382088059`}, {
     1119.647767074117, 6222.740665504156, 843.6238365331069}}}], 
    Line3DBox[{{{4739.0992771970205`, 3785.4308247433037`, 
     1974.655717322365}, {4728.404633824271, 3550.1866211161696`, 
     2392.943431023166}, {4768.499830102869, 3416.009075187929, 
     2506.0137786019113`}, {4915.56544204786, 3307.275595043339, 
     2363.674726643992}, {5360.11057633483, 2499.4606103025344`, 
     2389.5030774220777`}, {5432.424572936224, 2533.1811807095587`, 
     2181.643888231443}, {5492.991048529791, 2445.6371829484765`, 
     2129.2538129944546`}, {5609.828546255932, 2497.6565884558254`, 
     1726.0790647462413`}, {5659.494531174293, 2394.5451134102855`, 
     1709.9967957429596`}, {5774.008918790078, 2313.3362828843715`, 
     1413.191401765296}, {5774.638867623051, 2423.4824344877284`, 
     1211.6486818655826`}, {5770.340617289117, 2435.509518551165, 
     1208.0050643561553`}, {5781.355146301957, 2515.8046932428283`, 
     966.6814924188999}, {5638.860327323834, 2929.1533964182618`, 
     557.7887406399523}, {5465.127994782492, 3266.494799350117, 
     387.558274925628}, {5273.867547716035, 3557.2685776333497`, 
     469.0149738868202}, {5148.078451672213, 3731.15365957076, 
     513.4143688093942}, {5299.429122884662, 3439.29826982787, 
     880.3931352986056}, {5181.38379327356, 3532.2922635316154`, 
     1167.898558007838}, {5024.6305299715195`, 3670.7443291640334`, 
     1402.3329837054446`}, {4869.9252424495535`, 3669.751890145693, 
     1872.0477792312827`}, {4739.0992771970205`, 3785.4308247433037`, 
     1974.655717322365}}}], 
    Line3DBox[{{{3718.0111299957875`, -5148.833195959614, 
     594.7467086178401}, {3434.923520367794, -5340.186718742869, 
     609.524329445825}, {3367.2516082730035`, -5399.207793152568, 
     444.95561907165205`}, {3521.693796109185, -5313.979269887928, 
     217.05026064817548`}, {3691.9971588805392`, -5195.144689035591, 
     259.6966091486647}, {3718.0111299957875`, -5148.833195959614, 
     594.7467086178401}}}], Line3DBox[{CompressedData["
1:eJxE23k4lN8XAHApWSKytkiyhEopkkIOUVSSFCkiouwzZsYyRCEpqnnHDCGh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      "], CompressedData["
1:eJx1nHk4le/z+JWKIpLSYs0eUVIkqhH1JkuRJVuILCVLOYdzjjVClnMe0oJI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      "], CompressedData["
1:eJx12GlcjVvUAPBKmtSVRkOahNCcSoNaFSWaNGnSJBqMJZfSOec5KlREZx+R
JKVEKEPDIQ1Cg26iSd00kTSnOYR33Q/vve+X98P58v89v/3sZ6+911r7KPgd
dtjLw8XF1YC/Bfj7vdCmZ+saNvztflrvFTDg5NH6pK8OHKisV1GglNng1O8n
8Hzdf34lJK3j+3o2tNzl1lARZkDLt4Cz0+geg01j4apsYL3OyVKdoP/riyqY
79rx+cvrPoxytdHBJXmd8SZHDog4ZLycWs0GlSpKgPcvOnjnb3sci54dGCfh
uYwNFSHjPGP5dOj86n/ZxokDfW/o84OL2LDHNPvM1pt0MGsMt5J05sBPSalt
RlxsGOy/NxqZTAcSyxVUjF7dHZw5MkXg52PdoskkOqxqUZJzdOHAIK90xJsR
AlO3Vafr8Pk75/n776GLxO4JzB8g0LGj5L51Dh1Y3fkf6tAX1lklaX0kUPqE
d5czhw5WLw54yu7igNum4uhnfxOo/7hd5nIpHeSv7e8/ju5cmr+j8jUB4+a5
osAiOgg+XTxu78qBmXPGP/o5BMKjIqKtcHydpaxDVm4c+Bwn+fe92wQ0HLpy
j2TQwSH7jsdqdw5M236dykgisPeK1cqYdDqUnnATX+zBAaNVLbHXzhBQTJo0
qUujw45zPZYN6DtcaLMBJwiYHn9XLniNDv0FshXJnhw4N6W9riKYQIFSJFdh
PB0ucCvHeO7mwC0Bl6uPvAl428/OGpykw069HP5f6LkhymfsXAl0qU1fZwTT
odFpvPmIFweUNOWvjO4k8Ohte9+cGx20lSwCWv5xlZIYOVsCYR9sKrK30kHh
udmsnDcH9DqW7mlDNz/UmyGmRwcqjO+DMrr7R/nWMXsCd7zfe5xZTYfzSjUr
ldCzjDlrRtHXKf1iPZGiwxuRGLou+n7l2oRqawKqMxW5I9x0cLdRexSN3maY
L+i7hYDA/I/UuV4aHEifviPmw4Gm1U06jUYEhMvEcw9W0mDrY0nmZfTCb6s/
124isFF6V/XwPRocFg4OHUAfIu2JA3oEOPvMisdSaPDKdY+poC8HZvMy+0d1
MC6iP9S9omhg/LqHC9BvXE3PuKmLcVw0N7cpjAbvtUX3W6GvcFDhadMgEFXE
vmXhS4OmOylmd9CtHSc7c5RxPpLPbpU70WDIMGqLnB8HfNXPlZ+VI+D2R0Wx
igMN0gLEJC6hJ9UKC9suJ/CmplP+kAcNnmofi6lFt1zKWPlSgsBKIzH+jgAa
7E0UHebew4FEu+2Kb4UJ0EK1ToTH0KBKbuSGKTpfycADZwEC7Qe9u8NSaXBf
XY7hiW4umBaVzI3r36x6OiiHBokSVTU30PlltJ3T5liwvVbIp72MBsX13+kz
6NTf/PvVx1nwKT1TQ6yVBj/bevbq+HOgos0qMnaUBTrWv8Z2DtHgY6VZoTG6
Rbdiz+QgCxYKHMk+tZj+rw8sz41go0uP6M8ab6RD1xWTKUl0fqXrO3gHWBBH
LLxzbenArLuSIID+cU33wOYvLBBfX1uXHEoHLa4rzeM4nxj31ri1vSzYoPd+
qXbif/6Kb8F0dzcLck9tHQ3DcxqQVyU9gO7vfKlnQScLQrZMeS8dp0N4nsrk
S/Q9Ut0TQm0soCI5He3cDAh9Ka8ygu4dYJgj2MiCY34eTyyEGDBbY5SjgvOJ
TxbPPvUXC6KP+Bx152OA3tY/tU+hD1uHkuEXLIjgH83zmaHDmqLHFlx7OcB9
oqtK/xkLzGL722366bA/xrP2CLq05JLeG4Us+DIhGhgzQIdNW9zmH6PnivpX
q6O7Dx0Us56jgxIo3b2FfvlcsvuNAhaIVe6kuBYyoM1VISAVnV0VrP4VXWTs
/WQG+ivgfnUF3UA9I+oT+vMvJa6ymLfF6o8rH0D/bJ/UNZ7PgtIlvG7HLBlQ
eal9wB5dr9GWXXGPBf4KWhO37RhwsLGWk4OemlP1Wf06C/Z7OQWk4fNt6ioG
Afs48Ngy21wHvZDHaceB/+PPQi5GRl1hAUylb6/dygD7TxOvc9EfvNpx2+cy
C262l0pesf/PzeOG0qyvsuDyGzvZm14MeCy0Xj4c/ZXuwxbqBgu2LthwqSOC
AbeJsJAg+u8HV4yrsnB//noYN5P43zyFVIyb/bNZ4HLjc4piFgPaLY3k6OhN
Av22sTksUPaLH/lQiuuw4LuRLnrVyrXtA/dZ0Gi7sCyxmwE2vReGuzGOxj92
qFx4zAKN8h9rnnxnQIi4VUQ4uvTbQ8+4SllwOOHz+xQBCprKh/LacZ/Mis9G
r6xhQXGHGntclIJf30KnjNBNWuo9bN/heZlpNTsgSQFvpPmKv/D8Fu7b1drT
ygKrH5lT3EsoqMp1VIxA9/rp/YdvFwskHjg8jRKiwOK3abMB+s4/Sg208Lx0
GZfLPBOk4PzxBZWFmE+s1OQm1KZY8F1s6fJ0YRw/y/MPbXQn11srgIeAQRVz
pdpSClKDPN32Yn6L8mVOmAsSWOglaju9igITGb+UUcyfPX9qBm5aRCBt0Qo/
JRUKhNWTw8vRtfZLR3iIEQgdCZzT1qLAsTpXahe6b5Lo3WwpArtjzEST9SlY
Z7D4kgj6X2dNqXzMVx/KXs5GbqUgSsN1aTHWBbq4SHnvSqxHUf37zLwoiJA6
VGyPflJr2cfvmPeqbM4IjB6m4Mvz+Gk+9Ky508Vl8gT++MlReU9RkCCbyK7H
emQ699z+3loCqZ6bV1VeoUBnbjJaD32D3/dji9UI8CjEMRbfoeCCjHb2U6x3
r2TH7upoEZAoU0toKKLgdVnwwG70t/cLDy/RJ3Dz03LaykoKLhcEi3dj3bR3
qbQ7upnAxQt/JvI1UFAbMfnhDPpLrczTEqYE+FTO+nr0UGDOlRSvjm6xTNiH
bUKgz86sYmSAgsde3s/WoxevUjI6Z0Dg/s8XtnfHKUjpq3njhJ4QEzkxr0lA
Xyvwsv00BduPN+cWomuODeZHKBAQJOInf36jYFFSZGg4znO+ozRhdCmB2vi1
mjzzFNy6Fivbhe6zTpdzAePlYLH6eOxvCmr2bi6OxHUIqYwOo2Pc/Q4Vlz8V
YML60SjXJvTYZpga+cWCFdm/IxxEmZB8LI01hP5s7/GRC1wEeocC4m4sYwKJ
jpIpQTdn2FVf4yVguLVCSm41E/LXN2wORp/QPjCYJELA1SvuZMAmdJONaS9w
PmoZ4z/vYT16IxX8Y8iCifH3e2mKni7mujtvGYFL875F95yY0Pj33e52/N5q
Xu1xHRkCjY19hrf3MGFQRPdEIroGe/swHffD7i8P8jNCmVD+Z4/XZvSldTsr
PbFutp5MO7r1NBMUco1pl7AveiAZm3lgA/YDEdc11iYyIdWml9ihB9XqHzbE
+hvfXHbk/DUmhGWoL57A/sqUmXmJro39kmKOatBNJiRp8i0uQNcL2m06hf1A
IMd7mVUWE2byVPOD0X1Gj5/wMiQgmpP58kUaE2LmOtQ90Vsn96T9wLgf69nY
VpTAhBezb2Qc0fcZpDbdNce+KG0ujj+GCV+mR90B/Vvxtpif2whwOUq/dY5k
wu8dl39Iohu73/mUZUPgUEqt4NqjTLh/PdxyGvvDqTJ6A+WC/ZjF07MxQUxw
ZUgfvIHuHeQDlR4EeCdVhYP9mKBb5VcUgr72ZKNBlg/Gvel7dpcbzmegRNIa
vYjj42+1j0D15qKOTZZMeOcqb66P/nuR0ifWfgIuq7oyioyYIDtre1Ucve+Y
c4pCKAF59Xvlghjf9/xvpOuwjzVxeT0WGkng8oa4sKaNTLjIK77WAf2V5BqP
/LMEfOp+P53SYULXqOLrh9gnL4jzlfl9jYBYwaPiRwZMuL691CgN+/C3Zc98
SrJx/qd1BfI24z759FFCGF06PbDTIJ9AnYFASKoJE2yvn571xz7/jTrHilVK
oFIjYdN9fL7i9CX7XrwXtC6Xfnu9ksDika0zz3Ce/3tfCPu95rTCGwIL+OY/
UjjPQ60FQ9roBpuNJJc2E/jYc2xkWpUJqy8oZg3hfUSTt8eqo5PAj7lHedxr
mBBX/+TUKfTbgexmiS8E682ygn2y+L2q8SEq6AmMhwlJ/QTk2jIzBpYzoUmw
/owi+gNn7iOAPqmh7EfwfPXG7vlhjh5pwe/5qZfArQf0tX28TJAz1ii9iC7y
l7mCHd4jJJJvpKV9p8BMfVqnHp2yHC+rRN91bsmfNWOY/1lXd35B9zGvW/6h
j8CT+Cohz0EK64bvz3L0HbtzHmhM4DhBcwyBL1h3iO8zHfT6iaFfkb8wb2Sa
d+v3Yn2J8JhJ2Yn9c0/QEkVhNnSF5FnSP1GQfjrWdNAev7fBUYJfgg2HRaQ2
5XdjHjbqqIpFbzoQmCUoy4bRwIsjzzDvcW7NRvOgP5IePFKM98HOYeGDUjhO
TV0tf4gdzv/+a6NIdbzf9ZxVGvqA+fbsPEMZ/dvrwYP+G9lgLV96X6OZgowU
9cfztlhHNOZp7YZs0Bz41bK7iYLkPtX0FPScM8ZmvlvYsHFvSH3zOwqK1XgF
NNFlJWQv6luyoaOao+JYQ0G0llayIjrUV8y6oQcY5h4XK6cgrtcxxAA95ZTW
SX1zNgyrHTzIU0iBtq2e3nl0gyGxjy5GbLhanMrkuU9B3vxUPjfOc/8ol8sJ
Nfyu0GGe8xkULB+LbmlBX07bqCyuyIacpArvE6kUdBw8qPzP+jQYMj02L2HD
1+llF28lUeBRmHQkAte516ZVJ4ibDfzOU0KCbArubijRd8C4FITs3v19kMBn
g+fyRxMp+OpV08TA/XlTxpdWi/vk5FgXczv6Hc0u3Xfok3dNbmbhvfKoymir
/AUKllhdKvHAc9GZ3thw7h2Bfm/HfU9iKVhf+3JFD/pb/kefuWsJvGpt296D
dTNIXMrFFs9X7kDD7Hw1gWDRdZeUwymg3RuLpaHbDiuwWXg/NeC30/saRoFp
fLRrGHpSY66/eD3WqZP161zR74+IOxqiTyzLyHZsJZADF93Uj1Og5dLMV4zv
fV2oW1TXTeC35r5U1wiMl5B+mTm67uSSkwq4ny1KfvISfN7kdnGfPrpk8vyI
bwfmT+f0eOoIBa0rTaXj0adEMgWLGzCfjPlfHgugAG5a66nie7X8nVt0awhQ
DJk3zt7Yh4gJ2ZWj8yXGK/U9J9B99Rr/jBMFU4uubjPE/PN2cuyBAeaNA3YD
305sp2DGh0vkDPrsAZvK+TICv0zd416aU3AtV2v9KfR37lN8X18QiGZ0eZYY
U9CQ+MrKE30xu1VK5y3mw5p8g40m1L//J2wbLLY5gHljn0H1qNcWChozQ5Sm
/vmu+KyyAoxvPW9pqN82CtxU3e5uQ/dM+nDAHc9p3tLHhlY4ny/OLX5dGN/e
yqDyuGnMY7Kf+iQtKdjza2ZFKXr44InA67NYp1p3bivG+SSOJas++WefLPPZ
1DBDwDHkgmYk9mP/mw/1wqIqjacIPBRV+OishPO8PVe6Ad97MdjtYjeOb+Yq
XWgq858zv1ufuTCH91nzQUa1NJ7fZaHuEuitApo3XvwmkHnUzuWpGPZRuSZC
L3D8nQ3LLAIWsIF3S+IRbRE8FyVW5eno9alCvsHzeM+9LXhaihv7KMvzSStx
HN6ZhMDoUdy3l4Z5xzsZcLi7x4wH101Mbe1Tlc8ESmYWia17zoCv5odO30QH
p1Ge8R4CL8I1xJUfMaArrGB3D3qq0Itr1l14XixlnmulMkD0tbWyHMblVGKc
rTPutyNho21F8QxYRItyUkZv2UXpPcVxhIbfy3WGMyAlo2qTKrrZzuqhF5jP
F2aM1989yoAbYgrmfTi+/7nht6UjBGaZFwuYhxhwYeFI2nX0DLt27RRct5L3
UYFkP+Pf/b9vzc8Pn34QWJX4sKfGj/Fv3Ium+UZNBNjA1CoVZKAPxnR+3ol+
Qlhv5gzm1e9SQkq38F7z7a/ZD/m4bvE3OxPDZNhQUV59/uIuBpz5et3UGf22
vMrXBswzGov1jz2xYQCjc1G6CPr/97/c/wDbaWqz
      "], CompressedData["
1:eJx11Xs41GkbB/Bx2PEuUVRa7VJT70bKocgxuRE5DePUYSmSwYaKiMzvNyIh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      "]}], 
    Line3DBox[{{{4819.941436462129, 3027.452553688502, -2879.324447546743}, {
     4865.811372463408, 3036.5616898874646`, -2791.233863565279}, {
     4889.671848767678, 2916.757550002774, -2876.012568795672}, {
     4819.941436462129, 3027.452553688502, -2879.324447546743}}}], 
    Line3DBox[{{{3221.639224205622, 651.5471197447586, 5466.657182393286}, {
     3530.898995696809, 834.669153554313, 5246.32189026676}, {
     3423.906844488201, 972.0959494622497, 5293.364249256415}, {
     3188.6326364383185`, 925.3776770311978, 5446.477003962539}, {
     3104.934823298598, 1033.8775753923337`, 5475.243427663155}, {
     3111.7625080361113`, 893.2632236521919, 5496.07392570836}, {
     3038.3409922853825`, 1051.1302039985233`, 5509.212557057524}, {
     2813.1712729979663`, 878.0010128563654, 5657.118992932904}, {
     2384.0688150297246`, 1069.7763090216713`, 5818.900946386588}, {
     2134.8934733452916`, 802.4537010215435, 5957.022041690337}, {
     2774.4617499641536`, 596.4815724997836, 5712.651347811629}, {
     3221.639224205622, 651.5471197447586, 5466.657182393286}}}], 
    Line3DBox[{{{4402.970064032088, 543.2170941383158, 4583.293956738231}, {
     4287.21007282414, 792.0078233656133, 4656.2181101407705`}, {
     4283.987056701689, 724.5821743024488, 4670.144458225626}, {
     4267.380181662224, 716.6669668005968, 4686.541212221794}, {
     4246.757456439762, 715.7441658205179, 4705.377252173004}, {
     4264.912486643201, 567.7981715214996, 4709.133721576799}, {
     4389.723568171527, 458.7966466040735, 4605.176035951752}, {
     4402.970064032088, 543.2170941383158, 4583.293956738231}}}], 
    Line3DBox[{{{4182.850864962058, 3029.733765293675, 3743.298764849738}, {
     4247.010899811953, 3081.8605672182175`, 3626.685477278649}, {
     4333.803642542254, 3110.332256469264, 3497.398023138674}, {
     4362.603302366731, 3129.079234559025, 3444.469372781589}, {
     4150.960721267462, 3337.458106508469, 3509.802297536945}, {
     3971.35951524997, 3452.2501577070775`, 3605.285860960956}, {
     3759.0923945500135`, 3426.5121956099692`, 3849.1625365230275`}, {
     4095.1608461469737`, 3048.7342914024466`, 3823.962676937581}, {
     4182.850864962058, 3029.733765293675, 3743.298764849738}}}], 
    Line3DBox[{{{-2999.080410195423, 4935.854569505386, 
     2707.5210159618055`}, {-3063.497966265944, 4902.621575062428, 
     2695.7551061653994`}, {-3022.3643065425517`, 5086.870342925117, 
     2382.6199451038055`}, {-2932.530075480634, 5055.4899195032085`, 
     2555.4065665179364`}, {-2999.080410195423, 4935.854569505386, 
     2707.5210159618055`}}}], 
    Line3DBox[{{{1922.4795222186483`, 4703.069116122049, 
     3856.5563739726304`}, {1864.4687309763062`, 4665.439775978917, 
     3930.043477867057}, {1877.3874305178074`, 4551.125267680219, 
     4055.920231416835}, {1688.6691928092484`, 4536.654950584567, 
     4153.910435552685}, {1707.5797741602557`, 4583.380203058092, 
     4094.4610047103843`}, {1566.373334036544, 4685.9258875919795`, 
     4034.4004318379893`}, {1342.2637636826955`, 4732.94795394038, 
     4060.214938152403}, {1322.9932154019652`, 4903.230838364026, 
     3859.5116255449184`}, {1607.608115373053, 4856.21888603035, 
     3810.5836781717367`}, {1630.1157399536394`, 4720.812878034248, 
     3967.9291998398107`}, {1922.4795222186483`, 4703.069116122049, 
     3856.5563739726304`}}}], 
    Line3DBox[{{{4774.174514835812, 4067.9326522255, -1160.601279014394}, {
     4944.631153399624, 3973.217925693174, -672.2914262284892}, {
     4926.751352856374, 4018.160355144586, -518.9625781038676}, {
     5292.558044242269, 3558.6865666914864`, -111.32366494150033`}, {
     5496.062855482352, 3235.2753427199486`, -118.744411975844}, {
     5487.4855963700375`, 3240.9823472075554`, -267.11223244244565`}, {
     5539.550537865202, 3123.499665319096, -494.9175377664885}, {
     5425.137982575191, 3229.7549794441593`, -907.9506224325476}, {
     5278.536360587784, 3425.7389902237173`, -1043.6378446633944`}, {
     5191.519162758384, 3545.8417718999867`, -1078.4009464766134`}, {
     5121.098086069483, 3581.393217678045, -1278.9803373098352`}, {
     4774.174514835812, 4067.9326522255, -1160.601279014394}}}], 
    Line3DBox[{{{-1396.5643624708398`, 6089.579690647273, 
     1286.250699798711}, {-1173.3484727059395`, 6112.26074190946, 
     1396.9021707016702`}, {-1105.3851458416693`, 6102.721354535447, 
     1490.8821007207853`}, {-997.781146866132, 6194.734783734783, 
     1147.8272658134865`}, {-1144.29219507562, 6224.559524139982, 
     795.7812467803275}, {-1329.1889801033742`, 6200.112648542793, 
     692.5846450445014}, {-1227.5444858937792`, 6228.072768477318, 
     626.1452033698297}, {-1115.2345321764517`, 6239.862249241015, 
     712.8707728179377}, {-909.5353160273327, 6247.359562443076, 
     911.623659439641}, {-954.5605276459905, 6201.879333393575, 
     1146.0020170513972`}, {-1048.3226701210965`, 6154.497141694534, 
     1308.0512995230677`}, {-876.6014761483817, 6095.752845747724, 
     1661.680109278861}, {-950.2983703433443, 6045.465828295808, 
     1799.1913784337844`}, {-773.6324556671133, 5997.613927023648, 
     2029.2692517037854`}, {-1047.0817357882042`, 5888.2035456648155`, 
     2218.2183241058697`}, {-1175.9895711276563`, 5894.17543725044, 
     2136.248612290381}, {-1151.6714613056565`, 5971.190247783557, 
     1925.1892270601957`}, {-1402.4045991677688`, 5886.67834742556, 
     2016.9514283155577`}, {-1545.441560054943, 5883.899226734046, 
     1918.1120832979068`}, {-1621.980752123125, 5963.024830454567, 
     1580.9259948284682`}, {-1324.558920338368, 6058.04068002081, 
     1494.4900386971653`}, {-1396.5643624708398`, 6089.579690647273, 
     1286.250699798711}}}], 
    Line3DBox[{{{6341.192410880978, 132.8290456345542, 677.8266318324306}, {
       6338.560834923995, 182.58792552550375`, 690.7400968670692}, {
       6260.703974529951, 100.17262000981641`, 1217.1133388083663`}, {
       6258.629532840139, -18.205666684790824`, 1231.681214585082}, {
       6259.371515807334, 0, 1228.0398678788056`}, {
       6267.785615873991, -9.116131078596807, 1184.3116395599811`}, {
       6269.849711620262, 0, 1173.3704801533581`}, {6341.192410880978, 
       132.8290456345542, 677.8266318324306}}}], 
    Line3DBox[{{{-6234.579416996665, -891.8877920399345, -1011.122816264997}, \
{-6233.4788434851735`, -900.3844748526591, -1010.3776498255701`}, \
{-6232.759752956433, -900.166223964177, -1014.9976951503576`}, \
{-6231.370554937585, -901.9859194210436, -1021.8884959833001`}, \
{-6231.431635497948, -899.7456950424076, -1023.4896904063157`}, \
{-6231.990228430266, -895.442847332667, -1023.8620951624505`}, \
{-6233.427175044804, -891.7611490651719, -1018.3130149570488`}, \
{-6233.85188472373, -892.3174320875457, -1015.2211432584941`}, \
{-6234.579416996665, -891.8877920399345, -1011.122816264997}}}], 
    Line3DBox[{CompressedData["
1:eJx11P1TlFUUB/BddkF5CVZQwhUkFlhgqTVG3JVCFFEcYMHAjG2wJm0VedHl
TXDC53mEYdBlERgEqgWCvbzFm0QIzzBIHYHBqVxKCCcFFUmhCXaIoEBZsPMP
9MP95fPDud977rnX7ZQ65rQZh8N5CxcPl+7O4ZNOYhYCx6na3jYG0rrPr/Zw
CTQ2hD3v92RBba/3qZhjoEo+/85d9BDx+1ZS9NksncCBexlGZS4aOz4Ba7co
fawHCxN1x4wxrxh46po4mrKZwFxZne6mOwvl3y2uOD/C+vG61CYbAl2FRB2N
brphafl5BwMc1jLwmR2ByNah2x0iFpQfKWsEXzGwK/O4c6UDgarEMznpbixI
JN3mPToG1iaKIvhCAsEvjczqGyz0+8qL7XMY8M4ozPZ1ITB+0HavET1yf0kz
+x4DBWETyTd3EnCf4z7wxjoCZbt/n4QBerv4bw16nGprQy76kEQpO7BMQ/We
IpM/epK7pVaGefjqzTfmG2n4Ta2zbHMmEA5x16Mw/0b0PunrCTTEijN6up0I
DOf5KZKwD7vln63PyWjInzKYH99K4NSm2Jpq7JtE7nNU8zsFz2TtSR8LCAhj
baL9sf8DVZX2nBYKthidIsZt8VxTwp6TXiw8T0/VFqgpOHj3Tl+YFYGE3seu
Ed4s9I6mV792hALiVJJLLAh8rzFbSvBhgdq2UzsSSEFyVYVYhPfldf7+I4OE
he4l/lhxEAVW0VXVxhd6UKSYvr3ky0JS7uzibQUFAXuPnrX+Vw9jsq6sWvQn
mtz1BdzXNSi/LXlZD3H8oDNK9HxF6Q9/llNwIcV3dQa9tyPm1jjWD3DaLj1U
T8HCdFmeZEUPh9KKndcwj4d2h2duKQVv5/xSYGHSg2rycqEIvZcbahgow/xe
VydjOAR4UmdHNZ4rNWA9Pr+ZggeX1rNO8wjssvry0wHsg/CefH/9Qwo8HIWG
L8xxrgS+VufQGzIvrkzyaIB77QX7sA8Ry2xWBXrLNz9dFYho+DnoV4sanM96
kd+PyVi/dNsnK2I5Da2Ktg8HX+mhZNjEHUG3mckMXVPS0KFqdJ3DnNbF9x8+
RveLqpuK19DgPn49W7GmB4fSOtVF9CvKnq+nh2gYDH4XhtF3v1AsduK+s0VN
wsl/aBi5ZZedsaEH/8SQPgPe74n2hb9aeAxUZkozWs0IiA9zzJpxHroOpBpE
ZgxUlLfxK/C9qHKn5U9wfqLyng6etWXA5uUO10h8L+FLFv616KVXOI3xngx0
zWeMLaKr6pu4TeghFzxnBkMZONfBffMPawJrnLQ9sVjfe6emtvMDBja2HAtf
3ESgP9492BbzpHQ5GppTGLhmfcJUjH0+4jIllaF7+vGuabUMTBg7Oy0w5//9
D/8BpXom+Q==
      "], {{-6016.812802180255, -616.7892118124357, -2026.1659511776356`}, \
{-6016.086533434408, -622.9755554319481, -2026.4298580418276`}, \
{-6016.097687928805, -624.9931857683375, -2025.7753619394318`}, \
{-6016.326418570904, -626.9274936500026, -2024.4979705871585`}, \
{-6016.2942847968025`, -629.6840286223302, -2023.7378270960087`}, \
{-6015.951158836269, -631.7711605146453, -2024.1073452854544`}, \
{-6015.271641051804, -634.8842993363952, -2025.1525129762645`}, \
{-6014.939541556315, -637.1846417400276, -2025.4164346506477`}, \
{-6014.376901203698, -639.0357786118342, -2026.5037512724634`}, \
{-6013.512887487972, -639.2623991846507, -2028.9948316850803`}, \
{-6012.473844539709, -639.0458213053377, -2032.1398566530993`}, \
{-6011.971320415206, -637.931288778903, -2033.975959371678}, \
{-6012.170350363767, -634.7691753167061, -2034.3769223707181`}, \
{-6012.468790367144, -632.7847196063909, -2034.113131930895}, \
{-6013.32523112084, -628.7366731644279, -2032.8363314796457`}, \
{-6013.842446847084, -626.1381523709767, -2032.1081980656768`}, \
{-6014.255560738802, -622.4676745542494, -2032.013221969146}, \
{-6014.418134094757, -619.4078819201825, -2032.4669921273742`}, \
{-6014.32577714848, -617.9132438057967, -2033.1951110153534`}, \
{-6014.312203397519, -615.3660971533329, -2034.0076146705917`}, \
{-6014.294097704117, -612.6066010555933, -2034.8939403970242`}, \
{-6014.429720346466, -610.7113838557377, -2035.0627593865852`}, \
{-6014.9639112078485`, -609.0686607971559, -2033.975959371678}, \
{-6015.652452466091, -608.7141068377761, -2032.0448807236994`}, \
{-6016.390127654942, -606.773279917854, -2030.4407670833139`}, \
{-6017.003379162995, -606.7290439064194, -2028.6359684333454`}, \
{-6017.400953653536, -607.8300655268927, -2027.126553624882}, \
{-6017.497914734732, -610.1740793279844, -2026.134282094696}, \
{-6017.198559620924, -613.4329973422939, -2026.0392745125962`}, \
{-6016.812802180255, -616.7892118124357, -2026.1659511776356`}}, \
{{-5919.892252589064, -528.1280196889562, -2315.98147716155}, \
{-5919.230364194753, -530.672379748902, -2317.0913687134744`}, \
{-5918.300282224186, -531.2137459841217, -2319.3420221051138`}, \
{-5917.1507342136965`, -530.3818325555932, -2322.4633453058586`}, \
{-5916.304873667488, -528.6406378345578, -2325.013851108891}, \
{-5916.067093491965, -527.0582393751661, -2325.977955099375}, \
{-5915.516262864556, -526.6969736703085, -2327.4602750018084`}, \
{-5912.644908834033, -526.6493453517392, -2334.755734349863}, \
{-5911.647819284973, -526.7685261488499, -2337.252376317585}, \
{-5911.429762428488, -525.8131708158793, -2338.0188971159355`}, \
{-5911.624084200505, -524.6865450768078, -2337.780658341559}, \
{-5912.371717152523, -523.504885692565, -2336.154319147377}, \
{-5913.52955391872, -523.0873105002737, -2333.315574568396}, \
{-5914.700193928661, -522.0464541584893, -2330.579929039296}, \
{-5915.501930021077, -520.4524697301079, -2328.9009924573334`}, \
{-5916.377866204754, -519.4889557626369, -2326.890169114803}, \
{-5917.270650612915, -520.5040086233051, -2324.391816024587}, \
{-5918.367967372814, -521.9537849029265, -2321.2708888022735`}, \
{-5919.252346813115, -523.9058838272053, -2318.5745555488757`}, \
{-5919.81271776119, -525.8298668102636, -2316.707583576998}, \
{-5919.833615836822, -526.8730970719023, -2316.417145159784}, \
{-5919.892252589064, -528.1280196889562, -2315.98147716155}}, \
{{-6011.339013954406, -596.6207760992244, -2048.3311741113607`}, \
{-6011.639694534283, -594.6375122425825, -2048.0254154022855`}, \
{-6012.03501405746, -593.6170688591566, -2047.1608279043885`}, \
{-6012.667929392489, -593.7855258590603, -2045.2522607579035`}, \
{-6013.127157656087, -594.254769820864, -2043.7653349648504`}, \
{-6013.342324038047, -596.713616819664, -2042.4153944373538`}, \
{-6013.427258376047, -598.4178932882351, -2041.666555344534}, \
{-6013.257612799488, -600.944888737519, -2041.4239669429894`}, \
{-6012.859557220693, -603.025028624689, -2041.9829700543416`}, \
{-6011.932881966721, -606.2177565226232, -2043.7653349648504`}, \
{-6011.461170521128, -606.9121064807968, -2044.9464492557722`}, \
{-6011.3717474358755`, -605.9492048033953, -2045.494797189148}, \
{-6011.548150678104, -604.5891818019276, -2045.378801915857}, \
{-6012.018982071597, -603.2586843627299, -2044.3875388369372`}, \
{-6012.404463801064, -602.1314659642372, -2043.5860518609927`}, \
{-6012.594000237765, -600.6665949959878, -2043.4594979986184`}, \
{-6012.229551951962, -599.2524712604124, -2044.9464492557722`}, \
{-6011.783559044381, -598.2543228235081, -2046.5492650866386`}, \
{-6011.339013954406, -596.6207760992244, -2048.3311741113607`}}, \
{{-6127.809125563696, -454.08440637835724`, -1712.1846753613643`}, \
{-6127.257520567799, -457.2694980260354, -1713.3107052780488`}, \
{-6126.811769215646, -458.0964699030501, -1714.6833115368574`}, \
{-6126.295948423012, -458.8105618054264, -1716.3346150187526`}, \
{-6125.829882311847, -458.1305688064778, -1718.1787814137037`}, \
{-6125.55769572872, -456.49760776281425`, -1719.5832461400278`}, \
{-6125.54304388232, -454.34647869630703`, -1720.2050415113865`}, \
{-6125.668598985561, -452.52830993585525`, -1720.2372028615302`}, \
{-6125.948938504636, -450.39904569693954`, -1719.7976603278678`}, \
{-6126.4821349773665`, -448.50319727233614`, -1718.393209320748}, \
{-6127.133131848252, -447.36824219601516`, -1716.3667820375335`}, \
{-6127.601966313291, -447.4024738090701, -1714.6833115368574`}, \
{-6128.03464227523, -448.29427062332417`, -1712.9031963310129`}, \
{-6128.08211454939, -450.0182200991674, -1712.281194465047}, \
{-6127.99633453962, -452.4851950803501, -1711.9380135104416`}, \
{-6127.809125563696, -454.08440637835724`, -1712.1846753613643`}}, \
{{-5981.3246945861365`, -508.8896674900481, -2156.9886170900213`}, \
{-5980.425225886649, -510.70559740999767`, -2159.052477442989}, \
{-5979.261920874481, -511.3420951264585, -2162.1216046587424`}, \
{-5978.755099610303, -509.5118913488633, -2163.954431465206}, \
{-5978.946928176732, -506.58527662428054`, -2164.1115215148952`}, \
{-5979.412045788096, -503.68173754128014`, -2163.5040984365164`}, \
{-5980.458696288386, -501.4573524982122, -2161.126557056319}, \
{-5981.306446647595, -502.26433548364287`, -2158.5915358383795`}, \
{-5981.831447705773, -503.8855382749265, -2156.758119390221}, \
{-5981.705561118957, -507.02931105517666`, -2156.370456997001}, \
{-5981.3246945861365`, -508.8896674900481, -2156.9886170900213`}}, \
{{-6011.770706026181, -567.2210363132783, -2055.4043310573143`}, \
{-6011.232024237475, -567.2760607003012, -2056.964052125895}, \
{-6010.917229349344, -565.8704035680422, -2058.2707397903096`}, \
{-6010.7270718454, -563.2066928735093, -2059.556257762168}, \
{-6010.764858583239, -560.9879070433409, -2060.051473379231}, \
{-6011.129177058465, -558.5879593920555, -2059.640550760894}, \
{-6011.735310011892, -556.5277409122154, -2058.428800311385}, \
{-6012.302379302176, -556.0510720773441, -2056.900822859241}, \
{-6012.732099417985, -556.9375427275673, -2055.4043310573143`}, \
{-6013.036267505386, -558.9768846956513, -2053.9604129682853`}, \
{-6012.958628121053, -561.5102344827967, -2053.4966479338736`}, \
{-6012.5871526344845`, -564.4396080841465, -2053.781232458939}, \
{-6012.353759071281, -565.5821804911268, -2054.1501315367223`}, \
{-6011.770706026181, -567.2210363132783, -2055.4043310573143`}}, \
{{-6100.045437170674, -663.7542424247866, -1742.6100137771339`}, \
{-6099.919831217835, -665.3564842267266, -1742.438662223958}, \
{-6099.743308720522, -667.1686521038908, -1742.36369549216}, \
{-6099.5758097686585`, -668.3354003866506, -1742.5029192156396`}, \
{-6099.286798933957, -668.6269271372931, -1743.402497031856}, \
{-6098.770995128638, -668.2472162027907, -1745.3514537235399`}, \
{-6098.430687519016, -665.3017582473032, -1747.6642726792202`}, \
{-6098.396967618211, -662.8209990431494, -1748.7242317107746`}, \
{-6098.580622763356, -660.9024677468408, -1748.8098826863518`}, \
{-6098.827516110865, -659.5292267241646, -1748.4672767385728`}, \
{-6099.203514330857, -659.0314120840616, -1747.3430626698973`}, \
{-6099.614505842154, -659.3989262362674, -1745.7690643949502`}, \
{-6099.843815941347, -660.9316473915572, -1744.3877059296133`}, \
{-6099.953387681784, -662.2361192834045, -1743.5095871333701`}, \
{-6100.045437170674, -663.7542424247866, -1742.6100137771339`}}, \
{{-5973.276111996098, -546.5541125882205, -2170.0379678405798`}, \
{-5972.711770890323, -548.3946221417305, -2171.1266960438306`}, \
{-5971.881250225457, -549.6847947155586, -2173.084133130641}, \
{-5970.995233507042, -549.4981445702435, -2175.5646164961504`}, \
{-5971.036788401293, -547.9255538812604, -2175.8471795974724`}, \
{-5971.47375199967, -545.4433604023172, -2175.2715829918834`}, \
{-5971.9830073639805`, -543.9134083370556, -2174.2563910420117`}, \
{-5972.831399732985, -543.0446986133627, -2172.142079991923}, \
{-5973.37834569424, -543.7251340134889, -2170.4671865368414`}, \
{-5973.433735361907, -545.4121370439453, -2169.8914028138483`}, \
{-5973.276111996098, -546.5541125882205, -2170.0379678405798`}}, \
{{-5969.914171986316, -582.8305928213949, -2169.849526853966}, \
{-5970.428862392976, -583.0912353758116, -2168.3628617767886`}, \
{-5970.390566405165, -584.4550849258214, -2168.1011111764524`}, \
{-5969.930096317061, -585.7775533943611, -2169.0119854503087`}, \
{-5969.362006009892, -587.2996735827813, -2170.163593975144}, \
{-5968.73229785065, -589.1310612566059, -2171.3988669190103`}, \
{-5968.573824607147, -590.2724888422899, -2171.524482737953}, \
{-5968.356054941786, -590.3561385948093, -2172.1002097194196`}, \
{-5968.296758389577, -589.2984354736437, -2172.5503096003954`}, \
{-5968.268100847443, -587.1920489917129, -2173.199269281989}, \
{-5968.398749121055, -584.8911065359708, -2173.4609393741935`}, \
{-5968.73760530368, -582.400227529087, -2173.199269281989}, \
{-5969.18468683042, -581.9179868727776, -2172.1002097194196`}, \
{-5969.4763222748625`, -582.0515949006292, -2171.2627820403072`}, \
{-5969.914171986316, -582.8305928213949, -2169.849526853966}}, \
{{-5946.411758244898, -539.2823246568028, -2244.365270119723}, \
{-5945.505739421439, -537.9447158378773, -2247.0849283480206`}, \
{-5945.314181257345, -535.835154177454, -2248.0955637356537`}, \
{-5945.434719003908, -534.2769003047188, -2248.147656597249}, \
{-5945.923622299802, -532.6470530984467, -2247.2412163812614`}, \
{-5946.462551294652, -532.6953313743162, -2245.8033083915234`}, \
{-5947.018811097602, -533.6868274835285, -2244.094320794432}, \
{-5947.35936842856, -535.4962646465368, -2242.7603490597558`}, \
{-5947.163419257757, -537.8854074345531, -2242.708238516851}, \
{-5946.769911370774, -539.1055176274576, -2243.4586138547043`}, \
{-5946.411758244898, -539.2823246568028, -2244.365270119723}}, \
{{-5983.285887238146, -520.4182272259155, -2148.7830393848208`}, \
{-5982.375179905471, -521.7066688060186, -2151.0051251855066`}, \
{-5981.672167193056, -519.6467356192221, -2153.4574628694713`}, \
{-5981.50553803671, -517.7389844349564, -2154.379616602523}, \
{-5981.77730980378, -515.553716324618, -2154.1490829328695`}, \
{-5982.508225107463, -514.7752014465666, -2152.304699269412}, \
{-5983.220312245391, -515.7832876881165, -2150.0827856796136`}, \
{-5983.4030682555795`, -518.1136538581349, -2149.0136468797677`}, \
{-5983.285887238146, -520.4182272259155, -2148.7830393848208`}}, \
{{-5954.38641398888, -569.5664897008006, -2215.5563943882235`}, \
{-5953.962673817183, -567.5335456144451, -2217.2162374008626`}, \
{-5954.058084091203, -565.550322934568, -2217.4667649136927`}, \
{-5954.562901138066, -563.7107706747275, -2216.5794624636223`}, \
{-5955.240908082881, -563.5652172564745, -2214.7942709413983`}, \
{-5955.6709100929165`, -564.0254204395823, -2213.5205049357373`}, \
{-5955.869630520734, -565.4077332985648, -2212.6329856907996`}, \
{-5955.827288179525, -567.6063881304992, -2212.183986817004}, \
{-5955.585584704162, -569.3665109128533, -2212.382383204789}, \
{-5955.033776778347, -569.9430725399623, -2213.718884929324}, \
{-5954.38641398888, -569.5664897008006, -2215.5563943882235`}}, \
{{-5973.259002090216, -591.9991360860759, -2158.1305815037063`}, \
{-5972.733720781754, -593.2103162012824, -2159.251516472531}, \
{-5972.261314655503, -591.9002570488153, -2160.917065778986}, \
{-5971.923149887358, -589.7617446343901, -2162.435817869575}, \
{-5971.5896319425565`, -587.8345399594193, -2163.8811222677773`}, \
{-5971.697582218529, -586.5823634702298, -2163.923013277287}, \
{-5972.244926924658, -585.3732607993303, -2162.7395516714105`}, \
{-5972.715415831173, -585.5246212624214, -2161.3988917800802`}, \
{-5973.106097413661, -587.1417456150261, -2159.8800451867414`}, \
{-5973.28140722688, -589.0537305231729, -2158.874387877719}, \
{-5973.3416175827915`, -590.3229143422463, -2158.36106026213}, \
{-5973.259002090216, -591.9991360860759, -2158.1305815037063`}}, \
{{-5890.207923754132, -383.48356867006726`, -2417.892606693711}, \
{-5889.804178108377, -386.5543545069999, -2418.387099814416}, \
{-5889.1448477871145`, -388.47245114095284`, -2419.6850634921507`}, \
{-5888.624737638279, -388.334919538724, -2420.972610241874}, \
{-5888.1188672393155`, -386.75337592963666`, -2422.455721260436}, \
{-5888.233360095682, -385.0063188947673, -2422.455721260436}, \
{-5888.753720844854, -382.87282639704364`, -2421.5287947993984`}, \
{-5889.2585553075305`, -381.35735391561366`, -2420.5400073850324`}, \
{-5889.746793032762, -381.5954231111051, -2419.314228665006}, \
{-5890.148421941839, -382.4473241902728, -2418.2016668830092`}, \
{-5890.207923754132, -383.48356867006726`, -2417.892606693711}}, \
{{-5977.570407345686, -589.476689253552, -2146.854198135446}, \
{-5977.420714469254, -590.7260361651827, -2146.927582018184}, \
{-5976.7447923544405`, -591.9232555407885, -2148.4790519374537`}, \
{-5975.75246997681, -593.2994875348215, -2150.8583927485715`}, \
{-5975.0967341669275`, -593.0237576904779, -2152.755334487033}, \
{-5974.768083161771, -592.1486984422924, -2153.9080670642647`}, \
{-5974.865783806923, -590.8947430902367, -2153.981420087371}, \
{-5975.14090027865, -589.8689108094509, -2153.499380050459}, \
{-5975.868161753617, -588.8875741459987, -2151.74924848503}, \
{-5976.763694782871, -588.870496779489, -2149.265215078883}, \
{-5977.215266561645, -588.9149886334886, -2147.9968536627816`}, \
{-5977.570407345686, -589.476689253552, -2146.854198135446}}, \
{{-6015.786770685818, -591.231141977909, -2036.8036126237107`}, \
{-6015.901629759858, -593.2567136688194, -2035.8751785754337`}, \
{-6015.786744613849, -595.6838570423686, -2035.5059016832736`}, \
{-6015.527688106368, -597.0364987888896, -2035.8751785754337`}, \
{-6014.958753398552, -597.29807892618, -2037.478807208858}, \
{-6014.802524275179, -595.2683788369769, -2038.5337478515617`}, \
{-6014.741876868194, -593.5663395828474, -2039.2088772762595`}, \
{-6014.5743699554005`, -591.8539126800571, -2040.2004274876783`}, \
{-6014.698195633722, -590.5942101981167, -2040.2004274876783`}, \
{-6015.072637887392, -589.1470825439546, -2039.5147869207378`}, \
{-6015.529490180869, -589.1918290020707, -2038.153976372118}, \
{-6015.776959957563, -590.2761091866151, -2037.109563319374}, \
{-6015.786770685818, -591.231141977909, -2036.8036126237107`}}, \
{{-6039.092282204473, -593.415186236405, -1965.9696116070509`}, \
{-6038.303254745753, -593.0184386710755, -1968.5112712715072`}, \
{-6037.955754801418, -590.7500199975758, -1970.2584619621102`}, \
{-6038.175245569779, -589.0692236782435, -1970.089044562061}, \
{-6038.784259088847, -587.5326815769281, -1968.6807029733857`}, \
{-6039.286718466461, -588.2199931808941, -1966.9333648542513`}, \
{-6039.715929381834, -590.8158126198416, -1964.836343803137}, \
{-6039.515929516998, -593.1375347928039, -1964.7516107463432`}, \
{-6039.092282204473, -593.415186236405, -1965.9696116070509`}}, \
{{-5968.334429781462, -576.8931100713644, -2175.7739229610647`}, \
{-5967.802399498269, -579.4700691615411, -2176.5483338278113`}, \
{-5967.171777255556, -580.1447458984588, -2178.097046628359}, \
{-5966.664680170709, -579.7800792905168, -2179.582837164314}, \
{-5966.67615426292, -578.2044130757528, -2179.96995778611}, \
{-5966.83998049494, -576.7486581297233, -2179.9071820865925`}, \
{-5967.288392133721, -574.7947492137715, -2179.195707453194}, \
{-5967.784009585464, -574.5271191236238, -2177.908697426856}, \
{-5968.215531606448, -575.1994513235707, -2176.5483338278113`}, \
{-5968.334429781462, -576.8931100713644, -2175.7739229610647`}}}], 
    Line3DBox[{CompressedData["
1:eJx1lWs4lVkbxxmvQzFDERulmJ1T2A4NMuR2KGE7dUIOU5SxMbY3h50UJcaE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      "], {{3069.8594909428284`, -5450.747394111819, 1246.2460597536294`}, {
      3072.0326230305736`, -5449.110623599584, 1248.0475410856927`}, {
      3075.7799470192485`, -5446.798613845208, 1248.910031879075}, {
      3078.6591613501837`, -5445.369481809709, 1248.0475410856927`}, {
      3079.9402440494605`, -5445.195034549757, 1245.6455430013875`}, {
      3079.8052283604275`, -5446.398099496672, 1240.7099528748902`}, {
      3078.1098420438448`, -5447.6597302238815`, 1239.3776478579423`}, {
      3075.461288168784, -5449.277085167964, 1238.842525924315}, {
      3073.9895626664734`, -5450.335721980088, 1237.8377827184697`}, {
      3072.206911037037, -5452.066413869508, 1234.6376814808234`}, {
      3069.7122971369126`, -5453.758055042486, 1233.3706579392938`}, {
      3061.8084395737756`, -5458.2881398791615`, 1232.9774334550013`}, {
      3060.1038572598377`, -5459.3352324031175`, 1232.5732809960882`}, {
      3053.8565455794114`, -5463.538744766437, 1229.438194240671}, {
      3048.6270274333815`, -5466.699183404383, 1228.3676061214546`}, {
      3047.2397245616125`, -5467.396516106746, 1228.7062654342888`}, {
      3045.7127842597733`, -5467.932890129289, 1230.1045625632512`}, {
      3045.2218011251935`, -5467.67995471421, 1232.4422034189834`}, {
      3045.8797508336656`, -5466.998102966219, 1233.840336358475}, {
      3047.7135179278307`, -5465.778718501125, 1234.7141381794281`}, {
      3053.232153611863, -5462.712689780939, 1234.6486038776213`}, {
      3055.351054422842, -5461.377190193414, 1235.3148629690775`}, {
      3058.4959394722378`, -5459.0129505201785`, 1237.979759238741}, {
      3062.5674889493152`, -5456.047595143262, 1240.98295925533}, {
      3067.8898285473015`, -5452.434679077986, 1243.7128930278648`}, {
      3069.8594909428284`, -5450.747394111819, 1246.2460597536294`}}}], 
    Line3DBox[{{{5041.169464994508, 763.9052239273175, 3832.8674545389035`}, {
     4994.325252635012, 975.3223014394606, 3846.2027207633064`}, {
     5197.602449090831, 916.4775467711092, 3582.291358539866}, {
     5231.687384548728, 1067.568862407011, 3489.6357294670006`}, {
     5434.9707772208685`, 912.7529903003774, 3211.8200945029457`}, {
     5250.130856427837, 691.1928399093782, 3556.1484978052185`}, {
     5041.169464994508, 763.9052239273175, 3832.8674545389035`}}}], 
    Line3DBox[{{{4182.850864962058, 3029.733765293675, 3743.298764849738}, {
      4124.0787904553545`, 2998.1519134381374`, 3832.8674545389035`}, {
      4336.807444144102, 2794.88167759478, 3750.806751975005}, {
      4476.6543752582775`, 2551.712988788464, 3759.8058616541844`}, {
      4542.895150720259, 2351.461114118944, 3810.5836781717367`}, {
      4482.00497516068, 2416.675592355151, 3841.760556561247}, {
      4492.319311306951, 2213.74239611366, 3950.4715539929643`}, {
      4444.394692781567, 2280.841016942625, 3966.4762397129225`}, {
      4389.425435426152, 2155.109169198152, 4095.8836049956285`}, {
      4173.451731142366, 2372.4566077619925`, 4200.186184219843}, {
      3879.2841228235798`, 2711.2619636249437`, 4276.450274352492}, {
      3575.271220676256, 3166.842798396205, 4228.044001527208}, {
      3484.763809730408, 3462.5339767734013`, 4068.7950722643905`}, {
      3606.796433545107, 3581.7037323942113`, 3853.599817095655}, {
      3759.919724764167, 3427.2663289940547`, 3847.6827914308333`}, {
      4095.1608461469737`, 3048.7342914024466`, 3823.962676937581}, {
      4182.850864962058, 3029.733765293675, 3743.298764849738}}, {{
      4343.079269106449, 2121.389942910638, 4162.352803688725}, {
      4381.749102919671, 2154.50487214386, 4104.411924377356}, {
      4211.155721962463, 2332.680632157811, 4184.803728221748}, {
      4185.882978304977, 2227.235565520895, 4266.8043214583295`}, {
      4266.639866010629, 2113.3395890083657`, 4244.690062336561}, {
      4343.079269106449, 2121.389942910638, 4162.352803688725}}}], 
    Line3DBox[{{{2877.2447626916164`, 3795.2127876076047`, 
     4243.304863233771}, {2775.866309601174, 3930.1851002574635`, 
     4187.603729931402}, {2940.3533594561445`, 3862.095539152989, 
     4138.399915183348}, {2987.6447726541846`, 4097.083634102884, 
     3869.844712336456}, {2712.3345342568923`, 4208.719350615696, 
     3951.92819925039}, {2573.049914746979, 4379.817373300619, 
     3858.034162984386}, {2492.014815342531, 4548.616524639334, 
     3713.187789742302}, {2256.8057622804436`, 4640.834789498933, 
     3749.3057887979985`}, {2073.929206586604, 4611.113916549065, 
     3888.9921296750654`}, {2203.5976666218803`, 4432.6417402603465`, 
     4022.8917728619454`}, {2365.8001513517765`, 4158.92583692938, 
     4218.309995339978}, {2432.4901038114303`, 3995.5051000538087`, 
     4336.674368718898}, {2671.467965682896, 3960.6141345535952`, 
     4226.654501553725}, {2877.2447626916164`, 3795.2127876076047`, 
     4243.304863233771}}}], 
    Line3DBox[{{{1857.1851914922638`, -5622.88357593647, 2371.045580744019}, {
      1863.3540617713036`, -5620.5370741889055`, 2371.7690292582033`}, {
      1865.7798915722115`, -5620.037539041802, 2371.045580744019}, {
      1866.9021031556351`, -5620.3013013718155`, 2369.5365599693428`}, {
      1865.168939586034, -5622.561563081196, 2365.535877167251}, {
      1865.4547917515624`, -5623.127698571759, 2363.9642542742085`}, {
      1867.882838266292, -5622.925754021546, 2362.526900856685}, {
      1870.7537524807005`, -5622.218921923602, 2361.9374423433806`}, {
      1871.5270393571473`, -5622.348165600797, 2361.017013009525}, {
      1870.574112694636, -5623.546470763597, 2358.9173936530624`}, {
      1866.7563561808772`, -5625.474078949951, 2357.345070989924}, {
      1865.2393063161353`, -5625.89490910236, 2357.541620398132}, {
      1859.9215467193237`, -5626.749057530826, 2359.7034927348745`}, {
      1855.5636582938355`, -5627.966498249659, 2360.2309833146983`}, {
      1848.7586606143916`, -5629.355390648182, 2362.258027943505}, {
      1845.5824540831857`, -5629.459885163702, 2364.491593665107}, {
      1842.3012195903555`, -5630.204502220539, 2365.277397996155}, {
      1841.0841279701963`, -5629.942210580871, 2366.848881795371}, {
      1842.0811301372037`, -5628.56802059208, 2369.3401692371226`}, {
      1846.2325763744325`, -5625.439098671936, 2373.5361617356693`}, {
      1849.1118933279163`, -5624.436618010928, 2373.6704964950463`}, {
      1851.7744433182575`, -5624.031516130587, 2372.554447561373}, {
      1854.0552070149065`, -5624.051482347948, 2370.7251850943735`}, {
      1855.2008400904765`, -5623.730317161974, 2370.59082356295}, {
      1857.1851914922638`, -5622.88357593647, 2371.045580744019}}, {{
      1886.5301450949223`, -5619.398727589188, 2356.0933007877975`}, {
      1891.6581672543603`, -5617.7049295966835`, 2356.0208811139587`}, {
      1893.8550278224006`, -5617.485296266655, 2354.7793463124635`}, {
      1894.4136785149938`, -5618.2029943256575`, 2352.6167599516734`}, {
      1893.248847302858, -5619.963870117056, 2349.3464177862998`}, {
      1885.535256422661, -5624.4705138325935`, 2344.76052632563}, {
      1881.4716693272221`, -5625.964910474464, 2344.4395649342505`}, {
      1879.695987663372, -5626.2046111308655`, 2345.288544646906}, {
      1879.5896076832692`, -5625.886200314163, 2346.1374763222466`}, {
      1881.4479050079055`, -5624.033663425063, 2349.087657666914}, {
      1881.0247625476002`, -5623.682236206701, 2350.267567502884}, {
      1877.6898044307422`, -5623.92267574895, 2352.3580564050208`}, {
      1872.8996341969716`, -5626.012904172646, 2351.178311465916}, {
      1870.2706864138247`, -5626.831180876521, 2351.3128484863278`}, {
      1867.9421094104614`, -5626.977448334832, 2352.8133716515663`}, {
      1867.915325173429, -5626.601179778097, 2353.734307797284}, {
      1874.6284989543349`, -5622.200566550633, 2358.907049967412}, {
      1877.6512927787583`, -5621.0311450630625`, 2359.2897615505576`}, {
      1880.8216767436106`, -5620.270681648176, 2358.576048232767}, {
      1882.7689448812248`, -5620.221918973214, 2357.1381740754796`}, {
      1884.299752669129, -5619.873739214164, 2356.745062025605}, {
      1886.5301450949223`, -5619.398727589188, 2356.0933007877975`}}, {{
      1837.7843910990216`, -5626.23299211776, 2378.2061601632504`}, {
      1833.3478124927956`, -5627.588566904332, 2378.4230931597363`}, {
      1825.0843550765765`, -5628.561834633255, 2382.4719223035213`}, {
      1820.9506911825465`, -5629.808734655762, 2382.688792409846}, {
      1817.4236713954965`, -5631.354631622864, 2381.7283433571984`}, {
      1815.3986498454428`, -5631.872368557036, 2382.048500014872}, {
      1815.1625873009407`, -5631.140036778699, 2383.958967316959}, {
      1818.7260474691434`, -5628.633729457943, 2387.1597336302475`}, {
      1827.2687704720313`, -5625.458756485569, 2388.119827298761}, {
      1832.7366415588597`, -5624.179366237592, 2386.9429295806117`}, {
      1840.91602687768, -5623.447473651225, 2382.3686496994}, {
      1841.3438564796809`, -5624.256376425104, 2380.1274555044524`}, {
      1840.313839611267, -5625.227057204816, 2378.629693043049}, {
      1837.7843910990216`, -5626.23299211776, 2378.2061601632504`}}, {{
      1812.4369262279379`, -5641.3925586942205`, 2361.6892422913934`}, {
      1806.0249718999257`, -5643.071943081234, 2362.5889477611745`}, {
      1800.2799660786934`, -5644.015367181014, 2364.719067618871}, {
      1797.3428174196158`, -5643.685812332219, 2367.737937008683}, {
      1796.877201840451, -5642.223771806247, 2371.5726681614697`}, {
      1797.8000538741474`, -5641.069252268716, 2373.61882942451}, {
      1799.7700779071363`, -5640.410589493595, 2373.6911632726474`}, {
      1801.8480297393771`, -5640.0558676102255`, 2372.9574749581584`}, {
      1803.109525338922, -5640.404634724912, 2371.169603003078}, {
      1803.2709926272487`, -5641.317092934159, 2368.8750228996955`}, {
      1803.527593031783, -5641.712414934214, 2367.737937008683}, {
      1804.5032841489758`, -5641.539245530858, 2367.4071319970867`}, {
      1806.7270346398748`, -5640.449935743108, 2368.306490865779}, {
      1808.9216947621942`, -5639.677023174019, 2368.4718842540037`}, {
      1810.9219303444777`, -5639.482150301281, 2367.4071319970867`}, {
      1815.1293364913186`, -5639.361353775662, 2364.4709140418963`}, {
      1815.2109182532822`, -5640.019831185998, 2362.837132788195}, {
      1814.8278241561613`, -5640.416001423011, 2362.185638251121}, {
      1812.4369262279379`, -5641.3925586942205`, 2361.6892422913934`}}, {{
      1794.034584681869, -5648.63740303854, 2358.4208886370056`}, {
      1795.0594479085416`, -5647.832261559704, 2359.56903136138}, {
      1796.7356395894487`, -5647.437523474622, 2359.238044343961}, {
      1798.55913110347, -5647.505057172092, 2357.6864446189547`}, {
      1798.9530037337202`, -5647.923605267136, 2356.3829759662785`}, {
      1798.5972643522139`, -5650.046140779451, 2351.5612213438044`}, {
      1796.1894102678432`, -5651.763364465818, 2349.2739654037114`}, {
      1794.4707535376954`, -5653.259540383751, 2346.986359942884}, {
      1792.951201653233, -5653.741655287432, 2346.986359942884}, {
      1791.2054472444681`, -5653.921081465941, 2347.886952201222}, {
      1790.6566068016427`, -5653.0111131487565`, 2350.4952586921504`}, {
      1794.034584681869, -5648.63740303854, 2358.4208886370056`}}}], 
    Line3DBox[{{{-6219.075062945229, 
     320.2684250225532, -1381.4294000731143`}, {-6219.191485040694, 
     318.42428376022326`, -1381.3315814492603`}, {-6219.198705670755, 
     316.24809141508126`, -1381.798934020259}, {-6219.314617922249, 
     313.0980489089418, -1381.9945676695124`}, {-6219.487374569373, 
     310.71262798111314`, -1381.755459690769}, {-6219.729261634641, 
     309.20117114030495`, -1381.005516908333}, {-6219.842395271667, 
     307.3567973518789, -1380.9076968077236`}, {-6219.893049061063, 
     304.8564278715107, -1381.2337624845763`}, {-6219.726876130701, 
     302.67201051846695`, -1382.4619091989973`}, {-6219.341942241357, 
     302.4356686653449, -1384.2442565517272`}, {-6218.979922114027, 
     302.96205829718406`, -1385.7548158223865`}, {-6218.7399362681435`, 
     304.2559214024157, -1386.5480984644034`}, {-6218.581796648199, 
     305.55373206140524`, -1386.9718977755101`}, {-6218.501589761681, 
     307.1817424789909, -1386.9718977755101`}, {-6218.473929667865, 
     308.3771619984825, -1386.8306320523955`}, {-6218.539150114117, 
     309.79482383956923`, -1386.2220946227233`}, {-6218.572584642593, 
     311.6461836914643, -1385.6570122949208`}, {-6218.441497044024, 
     314.2509858676602, -1385.6570122949208`}, {-6218.3372746382565`, 
     315.87785498437006`, -1385.7548158223865`}, {-6218.166686208034, 
     317.61013466016783`, -1386.1242927290994`}, {-6218.000782389675, 
     319.99547758152715`, -1386.3198961743105`}, {-6217.928017667111, 
     321.4062889577598, -1386.3198961743105`}, {-6218.052177754404, 
     322.2832367202178, -1385.5592084255582`}, {-6218.31610643608, 
     322.84102688657424`, -1384.2442565517272`}, {-6218.660930647318, 
     322.8589293863099, -1382.690142930132}, {-6218.897084128498, 
     321.8917027693657, -1381.8532768374143`}, {-6219.075062945229, 
     320.2684250225532, -1381.4294000731143`}}}], 
    Line3DBox[{{{5465.127994782492, 3266.494799350117, 387.558274925628}, {
     5446.617733051172, 3311.59548902096, 235.59345727024507`}, {
     5544.63087811683, 3149.7887250659937`, -153.99059348067615`}, {
     5496.062855482352, 3235.2753427199486`, -118.744411975844}, {
     5292.558044242269, 3558.6865666914864`, -111.32366494150033`}, {
     5220.0736181063685`, 3659.662629306195, 213.34139591171007`}, {
     5273.867547716035, 3557.2685776333497`, 469.0149738868202}, {
     5465.127994782492, 3266.494799350117, 387.558274925628}}}], 
    Line3DBox[{{{3636.0624322749222`, 1588.5566164101226`, 
     4994.331942818639}, {3521.0377929730857`, 1819.9380640663571`, 
     4997.792712129787}, {3338.6541609464566`, 2056.6321863293774`, 
     5031.050201871271}, {3283.2076896176627`, 2204.8379850065126`, 
     5004.702830009777}, {3192.402474559581, 2637.8581749668942`, 
     4851.606371006018}, {3286.5300240261417`, 2751.2159634852896`, 
     4724.123708523987}, {3408.0863774899512`, 2688.326516825822, 
     4673.934260609424}, {3481.4076433104, 2626.6057347896635`, 
     4654.949712410433}, {3625.7183536889106`, 2521.504359125562, 
     4602.607508427604}, {3629.8536546903083`, 2672.900687529379, 
     4513.045318865786}, {3792.3306626903823`, 2522.7998983230677`, 
     4465.5931865586945`}, {3778.1014322751985`, 2405.371305280628, 
     4541.80053956277}, {3687.975810404597, 2465.7787041819806`, 
     4583.293956738231}, {3746.5589318227208`, 2322.9649561469278`, 
     4610.308414075213}, {3681.911455345105, 2278.4359427836184`, 
     4684.0229909374075`}, {3904.345560355304, 2223.9899255754126`, 
     4527.446106390765}, {3943.7719328260596`, 2111.678833609084, 
     4547.008811904897}, {3791.8836242383827`, 2157.008613251647, 
     4653.680920797106}, {3747.6859097524953`, 2016.5149551407083`, 
     4751.455718780979}, {3795.6683430248236`, 1906.2562889890423`, 
     4758.876178306062}, {3935.5974163624296`, 1662.461436713762, 
     4736.571401104493}, {3921.1586918773723`, 1596.2049108268254`, 
     4771.2113855672815`}, {3858.3849653075363`, 1602.1415839161668`, 
     4820.147455687986}, {3636.0624322749222`, 1588.5566164101226`, 
     4994.331942818639}}}], 
    Line3DBox[{{{3632.342999304361, 4520.415615274849, 2657.017254476135}, {
     3412.006695849351, 4716.390160073358, 2608.0045735087197`}, {
     3165.014559063785, 4757.722468665648, 2834.5323145218003`}, {
     3195.193870504328, 4845.765857477599, 2645.2037869313517`}, {
     3293.4310373129674`, 4852.206418466464, 2509.425943329764}, {
     3390.096146186889, 4811.708232080999, 2458.154937942835}, {
     3658.724393864392, 4545.118062184905, 2577.4893405292996`}, {
     3632.342999304361, 4520.415615274849, 2657.017254476135}}}], 
    Line3DBox[{{{3307.561720368457, -289.3741544445422, 5446.477003962539}, {
       3315.595168832379, -174.73011109316315`, 5446.477003962539}, {
       3417.8486453660125`, -253.9900283891121, 5379.741015406654}, {
       3416.1604867232413`, -108.35187963708587`, 5385.714538399703}, {
       3558.0205063898607`, -229.0475701148209, 5289.219316376825}, {
       3770.2903456780737`, 0, 5145.1651479118445`}, {3786.7241097152814`, 
       8.812123051014906, 5133.074766004129}, {
       3791.2123734115326`, 0, 5129.768262766973}, {3882.7319459102296`, 
       119.75869541567646`, 5059.433187727701}, {3975.0999937155916`, 
       26.595520300735117`, 4988.555543267233}, {4037.2407067720474`, 
       16.44149093267578, 4938.444172303524}, {
       4035.286105008577, -347.12852727011426`, 4927.85770150478}, {
       3945.59855428512, -164.21984576018508`, 5009.301107009008}, {
       3945.1596828169986`, -296.6380466368305, 5003.552201822497}, {
       3839.0758789377987`, -316.7538091476482, 5084.188933364733}, {
       3804.7619028285935`, -181.64652120220862`, 5116.503209626598}, {
       3655.1197443157007`, -192.62287657633638`, 5224.060659308329}, {
       3657.609717475531, -328.5779426227129, 5215.528868700698}, {
       3529.1503293570854`, -304.6232810166932, 5304.725845834203}, {
       3612.5669042073464`, -368.01289147155853`, 5244.210194522764}, {
       3307.561720368457, -289.3741544445422, 5446.477003962539}}, {{
      3623.387190603682, -460.95307123396043`, 5229.37866510823}, {
      3719.556245534981, -407.35766044372457`, 5165.924386088082}, {
      3717.9437080282255`, -408.2755714952929, 5167.0126119115985`}, {
      3672.6982403511747`, -525.9727240248639, 5188.685191732415}, {
      3623.387190603682, -460.95307123396043`, 5229.37866510823}}}], 
    Line3DBox[{CompressedData["
1:eJx11ms4lYkWB3DX7NqRSDU1lEsSiQojHC2XqBQlFF2o7Mf9kss0ityKcUnY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      "], {{-2297.715883179445, -2738.311158539882, 
      5282.988492245736}, {-2398.8587057208765`, -2790.5720782332933`, 
      5210.182153837991}, {-2334.834753867128, -2684.342703131965, 
      5294.399362937821}, {-2414.545385301302, -2656.6540256990297`, 
      5272.568032757598}, {-2304.7651817121673`, -2278.1028355417125`, 
      5494.189532393662}, {-2383.0073036385666`, -2370.562458670348, 
      5421.210520755309}, {-2421.5341537302716`, -2068.1855498911464`, 
      5526.897372264051}, {-2591.0592805239703`, -1636.9449523930075`, 
      5594.16094849314}, {-2764.7360297547752`, -1618.819487972381, 
      5515.747623045913}, {-2876.3707290133066`, -1530.4668621854032`, 
      5483.792146227629}, {-2615.7345451309725`, -1556.198522654504, 
      5605.710716536879}, {-2962.974250968969, -1480.5143139888075`, 
      5451.300270960051}, {-3317.205204594316, -1311.1425687682772`, 
      5288.181964056845}, {-3522.4209228659897`, -1054.5435985049185`, 
      5212.322163010284}, {-3164.09346129415, -1317.0849878982713`, 
      5379.741015406654}, {-3006.068897502808, -1289.437965665209, 
      5476.195139525929}, {-3100.543566106672, -1119.3217728493755`, 
      5460.912199761121}, {-3159.1569217205065`, -1016.322083798123, 
      5447.442579406382}, {-2966.8703396047276`, -970.6780194592318, 
      5562.668268062767}, {-3079.119617016321, -902.3424563159236, 
      5512.949679222095}, {-2975.2063326733014`, -878.4728271286515, 
      5573.530879076322}, {-2944.8961273063933`, -788.1644714114254, 
      5603.052494433775}, {-2947.8744085188446`, -724.0674021277927, 
      5610.1315990603625`}, {-2768.6970885546493`, -772.1660223277396, 
      5694.373517410986}, {-2691.9440950708545`, -919.0347125895445, 
      5709.345485786319}, {-2554.98885944699, -880.5846322097668, 
      5777.924915020769}, {-2573.7926596134985`, -540.8195075787767, 
      5811.275186698865}, {-2438.5752173319706`, -686.2180414319831, 
      5854.08143085041}, {-2466.266130843233, -810.8670457937258, 
      5826.477468925912}, {-2381.6701246794405`, -751.6994489911438, 
      5869.447081761265}, {-2403.2360684122923`, -863.6391862002516, 
      5845.202943058658}, {-2291.4654281958933`, -536.0520966062555, 
      5928.713859776592}, {-2051.8251750878135`, -677.2522882990758, 
      6001.596160928319}, {-1865.328378013156, -812.3585634675545, 
      6045.530356928145}, {-1724.2854104885407`, -1395.4685079124552`, 
      5980.5786640258275`}, {-2456.078269863966, -1988.892967213057, 
      5540.73984114689}, {-2287.23612989161, -2250.2778518403993`, 
      5512.949679222095}, {-2298.5123283841826`, -2737.642664239628, 
      5282.988492245736}}, {{-5458.656366606435, -2451.3091230222917`, 
      2209.5175818930984`}, {-5439.44700919913, -2559.609472075919, 
      2132.751573572008}, {-5498.1380618143585`, -2486.3563335144813`, 
      2067.9273986338767`}, {-5458.656366606435, -2451.3091230222917`, 
      2209.5175818930984`}}, {{-3018.886297920578, -1520.5428607620945`, 
      5409.444390952994}, {-3040.953873618275, -1594.2803648696233`, 
      5375.749561383714}, {-3113.822396380973, -1465.2535977756222`, 
      5370.750009821892}, {-3018.886297920578, -1520.5428607620945`, 
      5409.444390952994}}}], 
    Line3DBox[{{{2604.373853791307, -5490.559963571052, 
      1938.7629576920574`}, {2584.4945012536464`, -5501.349754730451, 
      1934.7745964625615`}, {2579.275856779548, -5503.98858000386, 
      1934.2335577228548`}, {2577.973156276399, -5504.408719962702, 
      1934.7745964625615`}, {2576.6381303170824`, -5504.459356650572, 
      1936.4082282354793`}, {2578.504030434193, -5502.341558271224, 
      1939.940213294218}, {2580.6403395658035`, -5500.827380803325, 
      1941.393122067277}, {2584.3115462617548`, -5499.039938518897, 
      1941.5734022777172`}, {2587.218761751927, -5497.223162705427, 
      1942.845919843818}, {2589.2100263226507`, -5496.349269768722, 
      1942.6656517020876`}, {2593.8390202530727`, -5495.001877551896, 
      1940.3007995446776`}, {2597.670470237606, -5493.255340443643, 
      1940.120507273435}, {2602.2385827674516`, -5490.834227128335, 
      1940.8522711820235`}, {2604.352237453857, -5489.895751583824, 
      1940.6719841361573`}, {2605.7281844985337`, -5489.628877744498, 
      1939.5796202124968`}, {2605.4511627211555`, -5489.985150457339, 
      1938.9432628194388`}, {2604.373853791307, -5490.559963571052, 
      1938.7629576920574`}}, {{2572.212538356038, -5482.390069547311, 
      2003.7303892622406`}, {2571.542673654018, -5483.152241143177, 
      2002.5042991271953`}, {2570.384538678634, -5483.919086154732, 
      2001.891223276048}, {2567.0659108434083`, -5485.5312079965615`, 
      2001.7326658376455`}, {2558.813907455825, -5488.5445415517715`, 
      2004.0368989649107`}, {2553.8742430007896`, -5490.45504067705, 
      2005.1043581921408`}, {2552.0466705520375`, -5490.911032593384, 
      2006.1823230874402`}, {2552.5189713058203`, -5490.297531341762, 
      2007.2602244020145`}, {2555.7562483026395`, -5488.513015509747, 
      2008.0210576191969`}, {2561.356128261115, -5484.72126315249, 
      2011.2436798513959`}, {2564.410954602016, -5483.1230984959275`, 
      2011.7085357122698`}, {2567.7349082280784`, -5482.075021677672, 
      2010.3244981257835`}, {2572.212538356038, -5482.390069547311, 
      2003.7303892622406`}}, {{2579.5293587446868`, -5479.56941584632, 
      2002.0392089806426`}, {2578.0047323082968`, -5479.556678946129, 
      2004.0368989649107`}, {2578.5774458151323`, -5478.614252068749, 
      2005.8758493393066`}, {2581.551500905818, -5476.818611962092, 
      2006.95376873423}, {2587.25754246661, -5474.350055008599, 
      2006.340841975007}, {2591.58235155766, -5473.260374972294, 
      2003.7303892622406`}, {2591.821708395762, -5473.765881693575, 
      2002.0392089806426`}, {2589.767578526093, -5475.356241186152, 
      2000.3478725759048`}, {2585.6002825616843`, -5477.3253744412505`, 
      2000.3478725759048`}, {2579.5293587446868`, -5479.56941584632, 
      2002.0392089806426`}}, {{2523.465753560028, -5504.770606780306, 
      2004.3539725650521`}, {2523.1166309708897`, -5504.414791011371, 
      2005.7701675449646`}, {2524.8516919525377`, -5503.457358934775, 
      2006.2140269749586`}, {2530.6309099148175`, -5501.480349910049, 
      2004.3539725650521`}, {2532.2161078956924`, -5501.789942626954, 
      2001.5001124462856`}, {2532.319384918003, -5502.418902302809, 
      1999.6395791115083`}, {2528.52923785672, -5504.3997016253, 
      1998.98411900921}, {2525.6215425890855`, -5505.135413992223, 
      2000.6332965385225`}, {2523.465753560028, -5504.770606780306, 
      2004.3539725650521`}}}], 
    Line3DBox[{{{
     3164.6716270712927`, -4256.05879153122, -3543.8158453621363`}, {
     2798.664235904269, -4449.942642490308, -3612.935463472431}, {
     2836.460677664176, -4574.742747860242, -3422.5769976678403`}, {
     2950.2778335242897`, -4654.877664940542, -3211.8200945029457`}, {
     3196.911211031154, -4360.036748247113, -3384.9153538483138`}, {
     3164.6716270712927`, -4256.05879153122, -3543.8158453621363`}}}], 
    Line3DBox[{{{2671.467965682896, 3960.6141345535952`, 4226.654501553725}, {
     2522.195988124624, 3739.30932693299, 4510.42202515463}, {
     2324.9348667747463`, 3818.839428343894, 4549.610640019778}, {
     2160.326358959812, 4091.6518285692255`, 4390.807331977748}, {
     1802.0686810337543`, 4441.666074456479, 4208.5584992797285`}, {
     1677.3667444902323`, 4486.321148011639, 4212.739850902471}, {
     1534.6953956295254`, 4461.294391443526, 4292.944919924791}, {
     1546.0121773287924`, 4550.056218186119, 4194.597531217617}, {
     1408.9193578580334`, 4608.367570882941, 4179.199476537026}, {
     1707.5797741602557`, 4583.380203058092, 4094.4610047103843`}, {
     1688.6691928092484`, 4536.654950584567, 4153.910435552685}, {
     1877.3874305178074`, 4551.125267680219, 4055.920231416835}, {
     1864.4687309763062`, 4665.439775978917, 3930.043477867057}, {
     1922.4795222186483`, 4703.069116122049, 3856.5563739726304`}, {
     2073.929206586604, 4611.113916549065, 3888.9921296750654`}, {
     2203.5976666218803`, 4432.6417402603465`, 4022.8917728619454`}, {
     2365.8001513517765`, 4158.92583692938, 4218.309995339978}, {
     2432.4901038114303`, 3995.5051000538087`, 4336.674368718898}, {
     2671.467965682896, 3960.6141345535952`, 4226.654501553725}}}], 
    Line3DBox[{CompressedData["
1:eJx1lms8ldkXxxESEacYKreJ4tQMklRS2xEaCuWucr8nikhRcYSh3OJ5eB4T
FSXdxKhHzqhVUQ0TTWMmiZrENFPJpVwOqVnz5t+r/4v95vvZn7XW/q3b1g2I
3hosJSEhEYFnBp5VknaZx6I44Fer+yUF08THXkHKezcDjVFJGb67OZg7GuMt
8KVJcvNtg7+9GZBIqfo7Zg8H8sv110Rso8lfvvsrfR0ZeGGmUNMfw0GnvaDj
D2ea/NytKV4tYGC/XvGo/F4OXqvMOP5cQJMej+z+DEsGbHbzI9XjOMjuDVCu
M6OJVvSc5YmrGZif/v2I/D4Ofuv2NC5bRJOtxa1LApYxsKq79vAr5FMft727
qkaTzDPXVP/kM+BR7+XknsDBROlVV8XZNIn1lNS98w0DF6cvP/sKudGIZZKi
PE0iV3a90DdlQHdyi4U+2hkYLby4l0eTeSb8pH/Qr4pH2eZDGE9DRotYpEmT
Lt4bRXYjAybfrXjhhvFbeQnyhLo0cbxOshPcGfCst23mYjm4ne+5IEmHJhWd
pb4pfgy8sUx6fxC5/+YV1ac1aFLvsz29LZyBYx/qhLrIvRom1PuUaVJ+MOS7
6/EMhLtzLj6om09CWUmuDE2UxK8cRUcZ6DSf7SSHOqvY+63smYG67Xqge+8k
A6EZdgk3ojkI2dw801WSJv7nnnB3rzIgu95bqgPzaMRTapaVoEnQXr6FRRMD
g+4WOj27OLg+rPpWHe08OpDif6qdgb1H7INnIA8ysrOwlaWJrGPXbesnDOyZ
HaJfH8mBkkb6gCfePx8bbSv9JwO5ETufFSHXunh37L2YIrun1q1/9A8DZ1zD
vLOQ83YVf5M6TBFVbk+FaJgB321uy+8jf3x6yxl4RpGYjNF9vE8MVOWV3TqE
fjsv6deotlNEV+T9+3sJFtjCdWVfYfyLB3wNSpooUvW4sUtbkoXsVOn0PuRT
nH57uYgizn1p7qrTDPBvXNJjsD5dxRNqUhcpklTjWnIE/XrOdDDZiXom5iuS
6lKKpH5+fcuhlwGneaX+MpjHbSMG8ZEsRYJz7BcybQykKRbY22He2xzG7z/L
R7/G8pJxdxgIy0ww1InnYP1w9TzrLHxv6kOTA40MpJ6wuPwA+XK7wRD7ZIq0
naq9OiDCvC+tkEvAuroabTXqFkMR2dYC5x68//Tqc91NyB/71xqYRVGkV3tg
/gO0b5z6Y94ttDOi4qCZj/dn1zm4zrnPwPc3tq84ivF4lXBicogimXlhCWLM
16EHQoMujH+oq8Zmv5AiS7xSF6T+wcBX4slwLeTJErP421MoIgrbdW0evvfh
hrYPylhvtuyM9LepFCkLb125fJCB4YzHv+5BfaqPWad6I+8/ubS+coKB2W7s
htPI9T4tG7RAv4vW5g4aTDIg5OtMa6Ady/KosJoDFOErpPu6jTGwcLGt4Czy
2PC7aef3UWRQrTzl9RADz13lHWUxnrfrnMM5fFf2pV/tKPS75cc5I4eQy8iM
vdcK+5KvwwFcpSvyK1KLo3IDKfLuQ+/x8+MMNMsUFWejfbH22MYMf4pcb5QW
5EmzUKB8PlKIcd4dXnO5Cu+H904kd6qwMKWn/vIk9ss/S72GngZQRKVWx2qp
NgsLtJQ3LEPulO3Ufm07RQx3vD7cwGehIfKEdA/WT8TrZF6yK0UKK9/5PTVj
IdpUuWgbcmsr4SC3lSIlo6HymetY6DKmqTHsO5NQL3UXd+Rr1vRZWrFw5zdB
rgNyo46V0YoBX3i0hsSpzv/6UbRC1B5KEW/N1T+PClh4LudXmYt8bPKCzULk
Yh8qjrFjwXlCgVeHfeEuGBwv20IRnqTkgEUwC/cXGB44g/zNjrQQQwHmq+RZ
RnIMC4V57Z4McuHM/FeLTSnyPlp4x+wwCzFqBiJP5GsHRiJZE4qYX8sulj7C
Qk5qfsxt7MelXREamqsoYrY9qH1BBsZZ25yvi1xDWPXoI9pfe1aqyQ654+6t
6V/v5KDpQrpxO+rQLPdNWOX3LNj73181HY56Kp0pfYj6Z4PRDb2jLByNft2l
hryz6USk1i6KyAR2b3LPw3zFRcZoh3HwknVRy0FuJJ1ozZxg4TA5lxsVin2d
pf6DdjTm1/hFpOJZFk7OktY0RX7NOrEuCuunLM7CvPI/7vExqTaEg4Nj22uf
HqTIlgZ2NKWCher7O/YrIbd85cZ/g30aeCQvaew06pCa3jIRxMGn7s/mP534
wkdMCgx7AznYJDcafOzKF15y3DB8CfKJkl8E2tcpkm9ZuGM1+tWZn2KW6M/B
zrhvxYsfUMS1wL/d8xILcR++/mDox4F9abPz6m6K1O+rO5j4IwvbEiuc7vlg
nEM3Cy68oUjoZNOv5g0sFGlN5L/fwYGz3451F8cpcsthkdagiIX6SroxFrln
bELNPZznxh+zqli837CgrT4P+dEfmnIOy9H/s78pUxxoivZ5Zfd+L8N999Bh
1qyllSzEf56WfYBcddZG67e414J9JHLTGBaeKOnWTCH/LljM/2UhTaLiX928
mcMCb2hJkD/y8z0e3huX02SJapGNYxoLZ55cjApFrhKb8OimFU3mvlMP+ryP
Berbn3SikL/IKuZ54H5v/LT+petOFs6urCTVyD039KS9cKOJ5cF2G4VAFhTe
6R238+Vgj8UtLw0vmuxPVuz404MF370qkVPIK0RT8nkeuK9zTraYOuGc/2RU
OYx6ul6T8Y3YQpNj0yS90ZYFU9tzMVcC8J9gx5V32NBEaKfuUmfOglWRP78P
8zU9J/B0C6HJXX2Z5IfY194FRX/NCeYggK+bXoi8t+AXuTXzWYgYdL4ZgnWS
uao0LdoS7bf2t16axcJvJadjsrDeHL5NF8Ba3L+lP/Q9FeNchf32w1i3/Zdz
hOPraCLQG5jNvWSgT2HZ76pY50KRzqcxa5pIzbWp8erBeUgc1VSQr57uDI93
oInmlLm1wWMGGi70V51DHjLZkSGJuo2Xv1OoaGFg/eTnA8kROOf3KrlIoA5Z
3fHZzvUM8FpphQ7kT6i5Q4E+NHEpbGnUucTAoiHl8ivYjy5taabjQTT5u//I
zVaWAXcnMJLBfj8CvK8vh9Dkc9ZfxS1p+F/6P/+6fwFc9R18
      "], CompressedData["
1:eJx11mk8VesaAPBtiMrYIKkM0SylDI14CScyRUTaZm3J0G041W5EIUnSu9I9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      "]}], 
    Line3DBox[{{{4641.727909075008, 1034.546866886032, 4251.104384548848}, {
     4640.7807510190205`, 1026.2448062295564`, 4254.1496575294805`}, {
     4637.6539217281115`, 1025.2477133650743`, 4257.7983651788345`}, {
     4634.165724257392, 1027.5904363903526`, 4261.030347833379}, {
     4632.261526240392, 1033.684580178489, 4261.626805718005}, {
     4630.855689498316, 1043.3196487889961`, 4260.806658778581}, {
     4632.448928220982, 1048.6496385631795`, 4257.765206415871}, {
     4635.702527505472, 1049.0799662261156`, 4254.116475949193}, {
     4638.532788147336, 1044.7345356981668`, 4252.100306107753}, {
     4640.2752227113, 1041.0771482118594`, 4251.096084419011}, {
     4641.390137788017, 1036.9803053693945`, 4250.880276498049}, {
     4641.727909075008, 1034.546866886032, 4251.104384548848}}}], 
    Line3DBox[{{{3155.631929290911, -5462.044935953534, 946.5019463604218}, {
     2998.401861865336, -5549.285876602823, 950.1716815917603}, {
     3065.033417314837, -5495.48723084735, 1045.4682856677462`}, {
     2849.364683964155, -5596.216791054535, 1118.6117357229582`}, {
     2956.539728695865, -5525.496177504852, 1189.7808684889155`}, {
     2646.2453093230247`, -5696.507945950612, 1111.304042378783}, {
     2317.6231385251876`, -5823.929768638849, 1182.4883626213389`}, {
     2130.669110273595, -5859.267833802306, 1347.9774791499876`}, {
     2158.8461839022784`, -5867.572106311446, 1264.4344275154936`}, {
     1979.496481505519, -5944.811749777222, 1195.2491913479616`}, {
     1984.8884619532303`, -5978.405150000089, 1003.346074919989}, {
     1999.7900154829156`, -5913.789076330503, 1309.8673003968195`}, {
     1800.7185076611117`, -6033.978581323916, 1018.0024687057293`}, {
     1956.3642625737734`, -6021.070083641306, 779.2096059494136}, {
     2431.8569187549574`, -5856.5611599494005`, 688.8954902420079}, {
     2402.141141748889, -5900.982106492706, 309.74474002028967`}, {
     2505.4487959742323`, -5864.485987236952, 135.44048970937934`}, {
     2854.1102861612294`, -5699.526540255998, 239.30186239592604`}, {
     2708.3325403043764`, -5755.499617028439, 476.41651893477575`}, {
     2901.8045135109246`, -5666.558002825631, 396.8181648821243}, {
     3033.5911171615307`, -5591.057641679024, 474.56619275001884`}, {
     3105.553049444898, -5541.582218077311, 578.1180393075803}, {
     3038.602290693107, -5569.335562156466, 661.2194832987964}, {
     3155.631929290911, -5462.044935953534, 946.5019463604218}}}], 
    Line3DBox[{{{-1565.7266223667762`, 6076.084792188722, 
     1147.8272658134865`}, {-1616.553981690437, 6096.863184440165, 
     950.1716815917603}, {-1712.166396345752, 6043.9566689733, 
     1107.6496308840485`}, {-1791.6283004070742`, 6029.110122162613, 
     1061.9382552751283`}, {-2059.1791974059483`, 5896.583658728427, 
     1295.3362029170194`}, {-1998.4967384792644`, 5820.5516066989485`, 
     1677.7972076630838`}, {-1735.6437538363057`, 5828.293136912417, 
     1925.1892270601957`}, {-1634.4777387654367`, 5782.541643146567, 
     2139.744927965952}, {-1831.6983866351397`, 5642.972727446407, 
     2342.9796216984673`}, {-1846.0939691060223`, 5642.57285244303, 
     2332.6213477766387`}, {-1568.6276557807237`, 5649.906062609799, 
     2511.1317072596785`}, {-1239.5594218099343`, 5765.660434749799, 
     2430.733609625493}, {-1357.821502734017, 5804.288521347579, 
     2270.3235931615272`}, {-1509.4468011146107`, 5774.193505793776, 
     2251.2381931403556`}, {-1444.526426325188, 5844.3502578983935`, 
     2108.252156799836}, {-1839.6357487461648`, 5846.437043284817, 
     1767.124189422525}, {-1860.4623200834596`, 5882.727209046518, 
     1618.6458011230995`}, {-1878.9108924376685`, 5941.061052943609, 
     1364.295097651402}, {-1706.1577778652472`, 6009.412236149287, 
     1289.8852286343704`}, {-1565.7266223667762`, 6076.084792188722, 
     1147.8272658134865`}}}], 
    Line3DBox[{{{-6178.906718076595, -195.7992113263478, \
-1571.8104619043297`}, {-6178.374238369674, -199.9921382705192, \
-1573.374909320705}, {-6177.894439280388, -203.32273113829834`, \
-1574.8313732304548`}, {-6177.4575617802475`, -205.46705293399546`, \
-1576.2661744015988`}, {-6177.162647720313, -205.8889692522694, \
-1577.3664906892175`}, {-6177.063918829841, -205.13017251149122`, \
-1577.851908455269}, {-6177.070659334867, -202.7559860100479, \
-1578.1323676173058`}, {-6177.254824582262, -198.0132476326546, \
-1578.0137122145484`}, {-6177.446610783735, -193.70238855079165`, \
-1577.7979736285163`}, {-6177.59040665344, -192.62765033652397`, \
-1577.3664906892175`}, {-6177.974540083616, -192.31583628352948`, \
-1575.8993912027477`}, {-6178.6420234227835`, -192.33661455248227`, \
-1573.27780862011}, {-6178.940089196204, -192.99357934779505`, \
-1572.0262537421345`}, {-6178.984536941945, -194.29036212697207`, \
-1571.691775577165}, {-6178.906718076595, -195.7992113263478, \
-1571.8104619043297`}}, {{-6176.057676028642, -212.2191857900689, \
-1580.8504956352738`}, {-6175.801189017509, -214.36870510579513`, \
-1581.5623357936834`}, {-6175.444974190302, -214.68007622294525`, \
-1582.9104603927058`}, {-6175.234009297064, -212.94622441205252`, \
-1583.967337393532}, {-6175.2146980055895`, -209.3847310945181, \
-1584.5173264815435`}, {-6175.429062942171, -205.72328646497394`, \
-1584.1614526225624`}, {-6175.652039633463, -204.00424929115783`, \
-1583.5143957799085`}, {-6175.887186860216, -203.5803893024554, \
-1582.6516263117628`}, {-6176.136296201663, -204.23606937676757`, \
-1581.5946916659582`}, {-6176.2089070212005`, -206.2888411769894, \
-1581.0446359410078`}, {-6176.176219623393, -208.9856605296847, \
-1580.8181387659565`}, {-6176.135076263671, -210.92680068931608`, \
-1580.721067897974}, {-6176.057676028642, -212.2191857900689, \
-1580.8504956352738`}}}], 
    Line3DBox[{{{4444.539303611051, 3145.372216832415, 3322.7274468778405`}, {
     4450.756764496039, 3139.0935125539236`, 3320.341824682305}, {
     4462.24761793249, 3123.768265013425, 3319.3721865363077`}, {
     4465.730901952337, 3117.3188648520804`, 3320.7505621749583`}, {
     4465.903474595855, 3113.37208298795, 3324.219324663152}, {
     4458.830668124113, 3109.713571158721, 3337.1131638509287`}, {
     4444.718609701086, 3104.3705703553883`, 3360.82929237535}, {
     4438.645037490062, 3112.731704317515, 3361.122618724498}, {
     4431.416752668432, 3120.9962985674524`, 3362.995890348043}, {
     4430.022241017495, 3117.793816555206, 3367.800224410654}, {
     4430.581641908564, 3112.8368792328247`, 3371.6474560177344`}, {
     4433.681686751075, 3102.825463720122, 3376.7964896123294`}, {
     4425.586910906083, 3102.260892618675, 3387.9155152018675`}, {
     4417.567217632351, 3098.0307708337805`, 3402.222642724462}, {
     4416.907027440217, 3095.049875257153, 3405.7909904484613`}, {
     4414.515062742961, 3091.765716870351, 3411.8697517344763`}, {
     4410.438052105478, 3091.882889932098, 3417.032348908212}, {
     4401.692379343972, 3094.2384144343073`, 3426.1649876011898`}, {
     4391.163695999056, 3099.7031835929047`, 3434.724917451749}, {
     4387.130219498429, 3103.893854615942, 3436.094449580957}, {
     4383.020159140719, 3113.319281488511, 3432.8109511125017`}, {
     4381.007580452955, 3128.035188942694, 3421.9851149223296`}, {
     4379.3372399720865`, 3132.6623550145605`, 3419.8897055414363`}, {
     4386.855019227309, 3139.6954525060078`, 3403.766912670911}, {
     4396.138539951939, 3147.16359890661, 3384.839864641521}, {
     4402.755491292232, 3150.2505298538836`, 3373.348391887712}, {
     4405.355007806606, 3149.913886901861, 3370.2675636061995`}, {
     4408.446269057353, 3149.996784846595, 3366.145486797982}, {
     4412.576191651699, 3150.819331420008, 3359.958717546775}, {
     4421.817891350386, 3152.305759550787, 3346.387398977674}, {
     4436.4227436504325`, 3154.6081881844775`, 3324.8179060905886`}, {
     4444.539303611051, 3145.372216832415, 3322.7274468778405`}}}], 
    Line3DBox[{{{5500.496474752598, -1288.4418644122163`, 
     2961.8017798739006`}, {5702.154334080212, -1748.7661715297436`, 
     2261.6512811756497`}, {5789.346223570131, -1336.575890435246, 
     2320.527716434667}, {5723.049032038058, -1331.795228233813, 
     2482.104940759333}, {5607.854771613948, -1191.986328599088, 
     2796.238034023488}, {5667.675010042684, -863.9019201379975, 
     2796.238034023488}, {5603.501384715845, -854.1201810573626, 
     2925.5879131890997`}, {5584.878127433277, -851.2815095214812, 
     2961.8017798739006`}, {5500.496474752598, -1288.4418644122163`, 
     2961.8017798739006`}}}], 
    Line3DBox[{{{4341.421951753622, 4266.303470018662, 1907.4915015638999`}, {
     4491.031426534392, 4156.314743786219, 1800.9714507636559`}, {
     4510.005492444033, 4287.321582402752, 1402.3329837054446`}, {
     4084.3058351098275`, 4649.067772293458, 1546.7476794584684`}, {
     3671.0644847714266`, 4886.437826588643, 1825.8763938266904`}, {
     3714.547097420422, 4751.553858505543, 2076.7015868358653`}, {
     4310.128626154076, 4398.798032466842, 1661.680109278861}, {
     4341.421951753622, 4266.303470018662, 1907.4915015638999`}}}], 
    Line3DBox[{{{5425.137982575191, 
     3229.7549794441593`, -907.9506224325476}, {5539.52924205205, 
     2991.048639609298, -1027.1599203345156`}, {5499.121634865678, 
     2969.2306992711233`, -1277.1624757477618`}, {5411.22632577723, 
     3099.0385370812537`, -1342.5362143721625`}, {5383.402634428094, 
     3083.1037587992823`, -1483.6647121605322`}, {5558.106808909259, 
     2856.4788967418194`, -1278.9803373098352`}, {5723.482649705074, 
     2546.2639906745176`, -1202.5388718537388`}, {5677.145228867054, 
     2529.6069350993857`, -1434.8952903432116`}, {5762.646381530199, 
     2328.2602683522687`, -1434.8952903432116`}, {5680.354450692, 
     2295.0121701190237`, -1776.036639123518}, {5575.507744748854, 
     2422.3715286987513`, -1932.2637643953342`}, {5492.316792830914, 
     2592.298059739058, -1949.938641768162}, {5404.155496133389, 
     2771.399710787064, -1949.938641768162}, {5297.280772589931, 
     3109.967843157007, -1718.932843620825}, {5177.561106107444, 
     3390.252482639379, -1544.947502823634}, {5277.514174326697, 
     3425.0755973642836`, -1050.9590772815511`}, {5278.536360587784, 
     3425.7389902237173`, -1043.6378446633944`}, {5424.910812684584, 
     3229.6197380756034`, -909.787179427347}}}], 
    Line3DBox[{{{5038.482652813411, 
     3063.445470629743, -2432.4489914027895`}, {5147.555285815295, 
     2896.5540804618017`, -2408.4149826047624`}, {5492.316792830914, 
     2592.298059739058, -1949.938641768162}, {5404.155496133389, 
     2771.399710787064, -1949.938641768162}, {5297.280772589931, 
     3109.967843157007, -1718.932843620825}, {5124.514247051772, 
     3325.779546231515, -1834.76369912627}, {5024.040893392242, 
     3266.8074062653814`, -2185.1306967529717`}, {5038.482652813411, 
     3063.445470629743, -2432.4489914027895`}}}]}, 
   {RGBColor[1, 0, 0], Thickness[Large], 
    Line3DBox[{{{2260.8504659295127`, 4848.409470423099, 3474.089012650353}, {
      1941.7729032111365`, 4778.580146644595, 3752.666180790547}, {
      1613.9002513607768`, 4687.106665485709, 4014.2459783895974`}}, {{
      1613.900251360776, 4687.106665485709, 4014.2459783895974`}, {
      1245.9719650166624`, 4650.030678248673, 4184.803728221755}}, {{
      1245.971965016678, 4650.030678248674, 4184.803728221748}, {
      1426.436884724364, 4315.831516091808, 4475.118973681765}, {
      1598.8896430917916`, 3957.3907356799705`, 4740.297898272654}}, {{
      1598.8896430917925`, 3957.3907356799696`, 4740.2978982726545`}, {
      1858.6581751303684`, 4151.1442365764915`, 4472.270676642775}, {
      2110.3476328608153`, 4326.853858722304, 4184.803728221749}}, {{
      2110.3476328608162`, 4326.853858722305, 4184.803728221748}, {
      2443.183844661506, 4231.718551185209, 4100.14932590753}}, {{
      2443.1838446615134`, 4231.718551185208, 4100.149325907528}, {
      2859.7732004638447`, 4291.611832239427, 3753.8751728435686`}, {
      3255.4849013810917`, 4320.174380241212, 3380.196010764347}}, {{
      3255.484901381094, 4320.174380241212, 3380.196010764344}, {
      3277.6375271247753`, 3961.977256756959, 3774.6048995522688`}, {
      3276.8837785453725`, 3576.0911245425577`, 4142.634265883744}}, {{
      3276.883778545372, 3576.0911245425605`, 4142.634265883742}, {
      3455.183728491137, 3455.1837284911635`, 4100.149325907532}}, {{
      3455.183728491152, 3455.183728491152, 4100.149325907528}, {
      3438.490606370393, 3263.0057484274416`, 4268.183398781269}}, {{
      3438.4906063703957`, 3263.005748427466, 4268.183398781249}, {
      3756.8399673239164`, 3059.0691981350897`, 4149.22436015923}, {
      4060.684904471121, 2843.3221804050477`, 4014.2459783895947`}}, {{
      4060.684904471117, 2843.322180405051, 4014.2459783895974`}, {
      4382.158672864403, 2845.8071167544613`, 3658.672014552403}}, {{
      4382.158672864392, 2845.8071167544617`, 3658.6720145524173`}, {
      4495.162384310365, 3070.1871556328388`, 3324.9480684769037`}}, {{
      4495.16238431037, 3070.1871556328497`, 3324.9480684768873`}, {
      4468.720629793017, 3100.0618079448977`, 3332.8615947536305`}}, {{
      4468.720629793044, 3100.0618079448664`, 3332.8615947536223`}, {
      4423.391885006163, 3213.7823224908093`, 3285.273368428752}}, {{
      4423.391885006174, 3213.7823224907843`, 3285.2733684287623`}, {
      4417.577079988473, 3122.0348110123964`, 3380.1960107643617`}}, {{
      4417.577079988474, 3122.0348110124132`, 3380.196010764344}, {
      4295.68120666094, 3101.90564692882, 3551.4951806656154`}}, {{
      4295.681206660939, 3101.90564692882, 3551.4951806656195`}, {
      4117.454803126962, 3216.908255474139, 3658.672014552417}}, {{
      4117.4548031269605`, 3216.90825547414, 3658.6720145524173`}, {
      3848.200302700416, 3716.1638355269683`, 3474.0890126503555`}}, {{
      3848.200302700413, 3716.163835526973, 3474.089012650353}, {
      3873.949726595061, 3976.7206311749237`, 3141.0221628319932`}}, {{
      3873.9497265950613`, 3976.720631174929, 3141.022162831986}, {
      4087.830604219257, 4087.830604219257, 2695.755106165405}}, {{
      4087.8306042192585`, 4087.8306042192576`, 2695.7551061653994`}, {
      4340.9963949495095`, 4035.5825189704747`, 2357.4642566994025`}, {
      4574.823124746716, 3965.355986117934, 2008.6709463327636`}, {
      4788.269099731428, 3877.463863603541, 1650.92904299545}}, {{
      4788.2690997314285`, 3877.4638636035415`, 1650.929042995449}, {
      5037.158975481318, 3741.9347570561126`, 1145.7606293041379`}, {
      5244.512720153487, 3575.549865867004, 631.1443375700419}, {
      5408.6205063710195`, 3379.6811905395584`, 111.32366494150222`}}, {{
      5408.62050637102, 3379.681190539558, 111.32366494150033`}, {
      5520.751100761118, 3187.40713415335, -222.61341961624748`}}, {{
      5520.7511007611165`, 3187.4071341533527`, -222.61341961624294`}, {
      5513.812652785342, 3183.401219346781, -389.4103191309964}}, {{
      5513.812652785341, 3183.4012193467806`, -389.4103191309915}, {
      5478.930914205872, 3163.262238188122, -814.1877792083749}, {
      5419.533620815009, 3128.9691948597724`, -1235.3221444118667`}, {
      5335.886546652355, 3080.675534075039, -1650.9290429954497`}}, {{
      5335.886546652356, 3080.6755340750396`, -1650.929042995449}, {
      5142.069075438385, 3468.3693855752485`, -1489.0779424276657`}}, {{
      5142.069075438382, 3468.369385575252, -1489.0779424276636`}, {
      4962.296784412864, 3474.637615279692, -1997.5779724178974`}}, {{
      4962.296784412866, 3474.637615279692, -1997.577972417893}, {
      5190.9712700232485`, 2997.0086601035396`, -2181.643888231442}}, {{
      5190.97127002325, 2997.0086601035364`, -2181.643888231443}, {
      5037.607078194518, 2994.8833905753745`, -2518.1346454429777`}, {
      4867.31156574108, 2982.6923636087404`, -2846.161990483539}}, {{
      4867.311565741081, 2982.6923636087404`, -2846.161990483537}, {
      4878.962251929208, 2649.0603630057503`, -3141.0221628319823`}}, {{
      4878.962251929209, 2649.0603630057467`, -3141.022162831986}, {
      5096.612304448899, 2269.157995837083, -3092.455124665309}}, {{
      5096.612304448893, 2269.1579958370894`, -3092.45512466531}, {
      5259.574222352664, 2341.713316300743, -2746.1011332110706`}, {
      5402.915807976756, 2405.5331020181984`, -2389.503077422076}}, {{
      5402.915807976756, 2405.5331020181984`, -2389.5030774220777`}, {
      5538.2390880594185`, 2070.66274634931, -2393.3401940325793`}, {
      5655.8040325317, 1729.1528210625502`, -2389.5030774220777`}}, {{
      5655.804032531699, 1729.1528210625556`, -2389.5030774220777`}, {
      5801.119849097393, 1859.1134731641973`, -1891.697460036249}, {
      5905.6849374870835`, 1976.014538537071, -1380.6033654269813`}}, {{
      5905.684937487084, 1976.014538537072, -1380.6033654269768`}, {
      6028.376052158488, 1472.0443382335934`, -1476.3405196757496`}, {
      6109.847886436721, 958.0089714899577, -1561.9831321590787`}, {
      6149.543373152747, 437.42317722266444`, -1636.945617832665}, {
      6147.19109270979, -86.15351682201685, -1700.7154174783686`}, {
      6102.8071289264935`, -609.1411324920223, -1752.856502349895}}, {{
      6102.8071289264935`, -609.1411324920216, -1752.8565023498948`}, {
      6232.468504258106, -730.2607983118561, -1144.9319629915392`}, {
      6300.697278226811, -844.1823819459564, -525.7219837112561}, {
      6306.820929010826, -949.7829737110657, 98.66996624902076}, {
      6250.779096610377, -1046.0216828183798`, 722.0893396613488}}, {{
      6250.779096610377, -1046.0216828183793`, 722.0893396613466}, {
      6181.987766773028, -1257.7396261591748`, 942.831890772469}}, {{
      6181.987766773036, -1257.7396261591543`, 942.8318907724473}, {
      6166.378979795885, -1087.2989874115524`, 1217.1133388083679`}}, {{
      6166.378979795886, -1087.2989874115533`, 1217.1133388083663`}, {
      6005.378905284615, -1497.3091287711884`, 1543.1471954616063`}}, {{
      6005.378905284619, -1497.309128771181, 1543.1471954616004`}, {
      6026.7107645896685`, -1614.852282387742, 1326.2063022192683`}}, {{
      6026.710764589672, -1614.8522823877563`, 1326.2063022192415`}, {
      5945.812839380494, -1768.7578925109071`, 1485.4692479948715`}}, {{
      5945.812839380499, -1768.7578925108967`, 1485.4692479948608`}, {
      5786.446998686148, -2136.074667605642, 1625.4321996962915`}, {
      5601.495309702436, -2493.946392500755, 1758.208001549895}}, {{
      5601.495309702439, -2493.946392500749, 1758.2080015498925`}, {
      5311.975155952692, -2887.8740933619656`, 2032.4657078116143`}, {
      4981.432412250808, -3259.4997180109785`, 2291.0273678992676`}, {
      4612.4197485619925`, -3605.9533272692133`, 2531.8961974440035`}, {
      4207.786925049885, -3924.5593769308357`, 2753.212047831311}, {
      3770.6587843413363`, -4212.857380086963, 2953.265771500479}, {
      3304.4111194222655`, -4468.620908634999, 3130.512421113929}, {
      2812.6446035286285`, -4689.8747872308795`, 3283.5831806636766`}, {
      2299.156983363196, -4874.910346901104, 3411.2959363750297`}}, {{
      2299.156983363196, -4874.910346901105, 3411.2959363750297`}, {
      2012.3177452180073`, -5159.907388747102, 3164.450461270292}, {
      1714.3786231899726`, -5416.442537337208, 2900.149973031278}, {
      1406.9830404743234`, -5643.100752542699, 2619.852345165589}}, {{
      1406.9830404743216`, -5643.100752542701, 2619.8523451655874`}, {
      1871.7290301470582`, -5621.164106689041, 2363.6747266439943`}}, {{
      1871.7290301470607`, -5621.164106689039, 2363.674726643992}, {
      1996.669736224157, -5691.083743337464, 2076.7015868358617`}}, {{
      1996.669736224156, -5691.0837433374645`, 2076.7015868358653`}, {
      1821.9743492526768`, -5749.394201122296, 2076.7015868358653`}}, {{
      1821.9743492526304`, -5749.394201122311, 2076.7015868358653`}, {
      1311.157809777371, -5914.250682205703, 1997.5779724178935`}}, {{
      1311.1578097773709`, -5914.250682205703, 1997.577972417893}, {
      1030.5766757433205`, -5844.690764865166, 2337.8013749062793`}}, {{
      1030.5766757433241`, -5844.690764865167, 2337.801374906275}, {
      992.4027468906229, -5930.364970034534, 2129.2538129944132`}}, {{
      992.4027468906304, -5930.364970034519, 2129.2538129944546`}, {
      1248.3434642015034`, -6013.9838823816945`, 1720.886442458823}, {
      1496.9433899923256`, -6062.238006790683, 1302.399526328541}, {
      1736.7406507647665`, -6074.843588447663, 876.2539459982532}, {
      1966.3251364418397`, -6051.726501160934, 444.9556190716507}}, {{
      1966.3251364418393`, -6051.7265011609325`, 444.95561907165205`}, {
      2272.2257912655464`, -5922.755510400375, 667.6606979069962}, {
      2569.20200615712, -5770.522185417373, 887.7434590939705}}, {{
      2569.202006157119, -5770.522185417375, 887.7434590939694}, {
      2259.2444851810715`, -5809.637555687026, 1353.4177173843273`}}, {{
      2259.2444851810733`, -5809.637555687024, 1353.4177173843236`}, {
      2133.3309704783496`, -5850.6962840135175`, 1380.6033654269795`}}, {{
      2133.330970478363, -5850.696284013515, 1380.6033654269768`}, {
      2278.281046921913, -5712.669402725163, 1691.9389635575249`}, {
      2415.560378943879, -5555.408534913284, 1997.577972417884}}, {{
      2415.5603789438833`, -5555.408534913279, 1997.577972417893}, {
      2574.9145806286083`, -5480.162644726093, 2006.351409851951}}, {{
      2574.9145806286583`, -5480.162644726069, 2006.3514098519527`}, {
      2604.1926642126564`, -5459.803460947295, 2023.991211811189}}, {{
      2604.1926642126105`, -5459.803460947326, 2023.9912118111613`}, {
      2734.488767060654, -5405.558908839116, 1997.5779724178842`}}, {{
      2734.4887670606126`, -5405.558908839136, 1997.577972417893}, {
      2788.012211321332, -5413.260211390522, 1900.3724590983159`}}, {{
      2788.0122113213392`, -5413.260211390523, 1900.3724590983004`}, {
      2848.717787215426, -5403.07405994985, 1838.3175355469684`}}, {{
      2848.7177872154775`, -5403.074059949842, 1838.3175355469155`}, {
      2881.7805935867927`, -5374.498094243433, 1870.2739198845356`}}, {{
      2881.7805935868428`, -5374.498094243389, 1870.2739198845836`}, {
      2914.2279903969943`, -5386.0058552208875`, 1784.9453317995515`}}, {{
      2914.2279903970184`, -5386.005855220896, 1784.9453317994892`}, {
      2949.874755194966, -5395.431529824981, 1695.6919607151256`}}, {{
      2949.8747551949778`, -5395.431529824983, 1695.6919607150955`}, {
      2991.683597778062, -5397.140079053802, 1615.0919193212594`}}, {{
      2991.683597778035, -5397.140079053801, 1615.0919193213115`}, {
      3005.260390564046, -5414.209804920791, 1530.5410361066456`}}, {{
      3005.26039056404, -5414.209804920783, 1530.541036106685}, {
      3149.2166527023637`, -5353.4213494374535`, 1452.9721331565158`}}, {{
      3149.2166527023273`, -5353.421349437469, 1452.9721331565356`}, {
      2991.1576295440204`, -5440.895529693778, 1462.0005335768021`}}, {{
      2991.1576295440045`, -5440.8955296937875`, 1462.000533576803}, {
      2959.8591375722162`, -5489.478263622251, 1338.9084990950992`}}, {{
      2959.859137572215, -5489.478263622252, 1338.908499095097}, {
      3035.6380139635276`, -5476.4359449074545`, 1217.113338808327}}, {{
      3035.6380139635035`, -5476.43594490746, 1217.1133388083663`}, {
      3159.005409660146, -5469.933693685827, 887.8760602800637}, {
      3272.771911410744, -5446.807142573987, 555.9403362644839}}, {{
      3272.771911410743, -5446.807142573987, 555.9403362644871}, {
      3558.2349263298865`, -5275.3002185754995`, 444.95561907165285`}}, {{
      3558.234926329888, -5275.300218575498, 444.95561907165205`}, {
      3829.4463704360596`, -5081.846975560228, 444.9556190716521}}, {{
      3829.4463704360514`, -5081.8469755602355`, 444.95561907165205`}, {
      3779.3885897432624`, -5137.949233013771, -74.25130767623267}, {
      3703.7654728806015`, -5159.296292305834, -592.9559680927465}, {
      3603.088565660016, -5145.743753162655, -1107.649630884046}}, {{
      3603.088565660015, -5145.743753162655, -1107.6496308840485`}, {
      3459.6356285558268`, -5120.894048547252, -1579.613608190453}, {
      3295.4402367099706`, -5065.341713816083, -2042.106916531961}, {
      3111.486833529156, -4979.419816353311, -2492.35664929453}}, {{
      3111.4868335291562`, -4979.419816353311, -2492.3566492945306`}, {
      3063.4995249769618`, -4725.771949240802, -2994.9730940752565`}, {
      2991.473986829291, -4435.042571194301, -3474.089012650353}}, {{
      2991.4739868292904`, -4435.042571194302, -3474.089012650353}, {
      2778.7390718309043`, -4213.63566533996, -3900.2176016801163`}, {
      2547.1643496617753`, -3963.6603759955956`, -4299.902776758395}, {
      2298.319893293314, -3686.811531688453, -4670.4346786890455`}, {
      2033.8928642229835`, -3384.9661632254406`, -5009.301107009008}}, {{
      2033.8928642229837`, -3384.96616322544, -5009.301107009008}, {
      2125.3363831459587`, -3764.4065635798693`, -4690.415991418336}, {
      2203.5108717346034`, -4120.344790217705, -4342.24738331123}, {
      2267.928266231726, -4450.558627792773, -3966.96899027538}, {
      2318.1863920129144`, -4752.986465040927, -3566.9237733688547`}}, {{
      2318.1863920129144`, -4752.986465040927, -3566.9237733688533`}, {
      2062.350452654904, -4997.810184501521, -3384.7330264183306`}, {
      1798.4763302793203`, -5223.154521639677, -3189.3500000000035`}}, {{
      1798.4763302793153`, -5223.154521639682, -3189.3499999999995`}, {
      2072.848321384554, -5361.2544305596, -2765.5134165898194`}, {
      2333.261890138701, -5463.252009703924, -2323.054050273562}, {
      2577.963428937608, -5528.460414177254, -1864.9513929149239`}}, {{
      2577.963428937609, -5528.460414177254, -1864.951392914921}, {
      2236.911466454505, -5749.468699933968, -1621.0029147796}, {
      1883.5076166198523`, -5938.729287680355, -1368.1035033149305`}, {
      1519.7033221296294`, -6095.1971090286, -1107.6496308840485`}}, {{
      1519.703322129629, -6095.1971090286, -1107.6496308840485`}, {
      1453.3960792332216`, -6175.015385006244, -666.8122088007076}, {
      1379.7623391594134`, -6223.705716279016, -222.61341961624476`}}, {{
      1379.7623391594134`, -6223.705716279017, -222.61341961624294`}, {
      1242.9028687369473`, -6244.300907672085, 389.5026615198584}, {
      1094.0126258886826`, -6204.453915634717, 997.8485221521239}}, {{
      1094.0126258886833`, -6204.453915634717, 997.8485221521206}, {
      656.626191239119, -6247.380893637047, 1107.649630884049}}, {{
      656.6261912391202, -6247.380893637047, 1107.6496308840485`}, {
      541.6916214627536, -6191.563565286875, 1434.8952903432107`}}, {{
      541.6916214627536, -6191.563565286876, 1434.8952903432116`}, {
      376.14148427210057`, -6149.858906414337, 1650.9290429954399`}}, {{
      376.141484272093, -6149.858906414334, 1650.929042995449}, {
      117.10143214447174`, -6192.589772626548, 1525.1009319224718`}}, {{
      117.10143214447996`, -6192.589772626547, 
      1525.1009319224756`}, {-26.819994945639216`, -6146.651059636862, 
      1704.6334294903854`}}, {{-26.819994945631986`, -6146.651059636864, 
      1704.6334294903768`}, {132.891605130633, -6090.336039672769, 
      1891.5497442966725`}}, {{132.89160513063445`, -6090.336039672769, 
      1891.5497442966744`}, {-321.3274309877704, -6012.56961730523, 
      2105.604181500608}, {-773.5075049825235, -5896.650829054025, 
      2306.297645567935}, {-1220.779337872023, -5743.315229981905, 
      2492.3566492945292`}}, {{-1220.7793378720257`, -5743.315229981903, 
      2492.3566492945306`}, {-1025.9305464433617`, -5536.761416203498, 
      2996.5902662717626`}, {-822.6378079477871, -5284.637148866989, 
      3476.1603719836976`}, {-612.5743283629336, -4989.017543071598, 
      3927.1198526597764`}}, {{-612.5743283629333, -4989.017543071597, 
      3927.1198526597764`}, {-535.5783446927737, -4595.635234455161, 
      4391.025656783436}, {-453.6345224919577, -4159.797025014497, 
      4814.365806696042}, {-367.4998841547848, -3685.529323037341, 
      5193.229355052416}, {-277.97016813224366`, -3177.2135603589077`, 
      5524.116243119799}}, {{-277.9701681322438, -3177.2135603589095`, 
      5524.116243119798}, {274.99497012512916`, -3324.6648899950446`, 
      5436.8000538588985`}, {825.6950360856584, -3444.7316957062885`, 
      5304.702153936634}, {1369.594036008406, -3536.42501396628, 
      5128.910604321721}, {1902.21199501803, -3598.989587342846, 
      4910.873360839933}, {2419.161857659642, -3631.9100853851996`, 
      4652.386347684597}}, {{2419.1618576596416`, -3631.910085385199, 
      4652.386347684597}, {2711.8768236341716`, -3166.5979502487266`, 
      4827.64903497721}, {2982.402763387538, -2675.376185169773, 
      4963.4110966919525`}, {3228.526188785479, -2162.2640481510443`, 
      5058.561705109444}, {3448.233277115916, -1631.4599078512788`, 
      5112.322322160871}, {3639.726348520584, -1087.3068918361423`, 
      5124.253069555664}, {3801.43857490997, -534.2573503877179, 
      5094.256327938667}}, {{3801.4385749099715`, -534.2573503877168, 
      5094.256327938667}, {4141.239676532812, -186.8452459929855, 
      4847.993036872087}}, {{4141.239676532811, -186.8452459929857, 
      4847.993036872086}, {4160.781880478465, -157.41695349225665`, 
      4832.279765890558}}, {{4160.78188047849, -157.416953492218, 
      4832.279765890537}, {4097.582634996559, 442.7304775535051, 
      4868.430955001817}}, {{4097.582634996558, 442.7304775535113, 
      4868.430955001817}, {3964.8239491363383`, 627.96642216926, 
      4957.180944345574}}, {{3964.823949136328, 627.966422169275, 
      4957.18094434558}, {4307.926479448986, 605.4395832582978, 
      4665.085856518214}}, {{4307.926479448999, 605.4395832582968, 
      4665.0858565182025`}, {4268.724433263222, 716.8927214033322, 
      4685.282299804411}}, {{4268.724433263232, 716.892721403304, 
      4685.282299804405}, {4206.844261524968, 996.7819173591694, 
      4690.0640569719535`}}, {{4206.844261524969, 996.781917359167, 
      4690.064056971954}, {3971.536450293789, 1101.404535294759, 
      4868.430955001822}}, {{3971.536450293795, 1101.404535294756, 
      4868.430955001817}, {3690.285546486248, 1343.1540948637657`, 
      5026.484194031191}}, {{3690.285546486236, 1343.1540948637758`, 
      5026.484194031197}, {3971.0771078779794`, 1406.232156851083, 
      4789.662975025008}}, {{3971.0771078779785`, 1406.2321568510827`, 
      4789.662975025009}, {4087.909982567451, 1487.8775540142278`, 
      4665.085856518186}}, {{4087.909982567436, 1487.877554014217, 
      4665.0858565182025`}, {4283.68970744126, 1644.3544159380906`, 
      4431.017347646779}}, {{4283.68970744124, 1644.3544159380742`, 
      4431.017347646805}, {4363.877920335436, 1417.9098882728517`, 
      4431.017347646806}}, {{4363.877920335429, 1417.9098882728745`, 
      4431.017347646805}, {4333.716006501668, 1201.845815587863, 
      4523.525826227277}}, {{4333.716006501673, 1201.8458155879005`, 
      4523.525826227262}, {4274.640154261453, 1130.7392584562358`, 
      4597.466038043288}}, {{4274.640154261458, 1130.7392584562397`, 
      4597.466038043284}, {4498.713577036146, 990.4782986589082, 
      4412.294480389586}}, {{4498.713577036162, 990.4782986588971, 
      4412.294480389571}, {4560.87550189033, 1038.993440544732, 
      4336.671647330662}}, {{4560.87550189031, 1038.9934405447163`, 
      4336.671647330688}, {4636.094647619778, 1023.5536601049657`, 
      4259.903521583205}}, {{4636.094647619813, 1023.5536601049577`, 
      4259.903521583165}, {4570.6380761013015`, 593.6222410026962, 
      4409.61380421006}}, {{4570.6380761013, 593.622241002688, 
      4409.613804210064}, {4428.318091600701, 154.64027531292393`, 
      4588.452784420153}}, {{4428.318091600701, 154.64027531292427`, 
      4588.452784420154}, {4701.259391893573, 123.10680971252876`, 
      4309.387257318013}}, {{4701.2593918935745`, 123.10680971252857`, 
      4309.3872573180115`}, {4874.464743219815, -340.85577942440324`, 
      4100.149325907529}}, {{4874.464743219815, -340.8557794244038, 
      4100.149325907528}, {4874.061536841729, -685.0046767785765, 
      4057.35214368577}}, {{4874.06153684173, -685.0046767785576, 
      4057.3521436857727`}, {5129.295245226647, -480.34537513065055`, 
      3761.304600519068}}, {{5129.29524522665, -480.3453751306481, 
      3761.304600519064}, {5388.859821889507, -471.4641437456615, 
      3380.196010764341}}, {{5388.859821889504, -471.46414374566154`, 
      3380.196010764344}, {5544.545508912001, -892.1720942429117, 
      3024.8731781444917`}, {5655.6043325402925`, -1305.6991419754709`, 
      2645.2037869313563`}}, {{5655.604332540294, -1305.6991419754754`, 
      2645.2037869313517`}, {5863.033660517426, -1246.2262758351874`, 
      2181.6438882314405`}}, {{5863.033660517424, -1246.2262758351876`, 
      2181.643888231443}, {5989.243621433488, -837.9916708226715, 
      2028.434000097143}, {6085.121899371641, -425.51317471069126`, 
      1864.9513929149182`}}, {{6085.121899371639, -425.51317471069757`, 
      1864.951392914921}, {6211.428191603847, -216.90785209353965`, 
      1434.8952903432082`}}, {{6211.428191603847, -216.90785209354138`, 
      1434.8952903432116`}, {6260.037197245608, -384.2117492378345, 
      1162.8109523258586`}, {6292.5862727922895`, -550.5299631281719, 
      887.7434590939713}}, {{6292.58627279229, -550.5299631281729, 
      887.7434590939694}, {6312.7750140622875`, -220.44696111935815`, 
      887.7434590939694}}, {{6312.775014062289, -220.44696111933368`, 
      887.7434590939694}, {6315.313407230485, 128.61496526752165`, 
      887.7434590939698}}, {{6315.313407230484, 128.61496526752106`, 
      887.7434590939694}, {6286.369663929699, 246.99213678559153`, 
      1052.7891635035396`}}, {{6286.369663929706, 246.9921367855664, 
      1052.7891635035046`}, {6220.6592343993, 874.2566412016198, 
      1107.6496308840487`}}, {{6220.659234399301, 874.256641201617, 
      1107.6496308840485`}, {6154.728293607759, 864.9906517500598, 
      1434.8952903432064`}, {6071.9276703733585`, 853.353783046819, 
      1758.2080015498823`}}, {{6071.927670373357, 853.3537830468185, 
      1758.2080015498925`}, {5955.773268233409, 670.5592249678325, 
      2183.3297490775167`}, {5806.298381765297, 484.0131207603497, 
      2596.2365244410094`}, {5624.339271079623, 294.7591311316842, 
      2994.6182575423572`}}, {{5624.3392710796215`, 294.75913113168167`, 
      2994.618257542362}, {5436.618540572705, 562.3419931616271, 
      3288.5808515267063`}, {5223.075668844442, 827.2539165826287, 
      3566.923773368853}}, {{5223.07566884444, 827.2539165826297, 
      3566.9237733688533`}, {5004.7525743259275`, 1302.0859159478125`, 
      3734.2787291883196`}}, {{5004.752574325929, 1302.0859159478107`, 
      3734.278729188319}, {5291.102035255617, 1503.1117076716766`, 
      3229.970299673699}, {5528.460414177254, 1690.2199698436402`, 
      2695.755106165398}}, {{5528.460414177254, 1690.2199698436398`, 
      2695.7551061653994`}, {5699.517356135334, 1855.3884710292343`, 
      2181.937033620331}, {5825.671885724591, 2005.9396962202518`, 
      1650.9290429954522`}}, {{5825.671885724593, 2005.9396962202527`, 
      1650.929042995449}, {6044.899869304283, 1670.592822553841, 
      1164.525174187599}, {6205.13050723875, 1318.9412053324331`, 
      666.7557086453868}}, {{6205.13050723875, 1318.941205332429, 
      666.7557086453812}, {6277.966515441289, 1106.9748806566088`, 
      222.6134196162497}}, {{6277.96651544129, 1106.9748806566054`, 
      222.61341961624294`}, {6330.189872033728, 777.2495838696864, 
      111.32366494150413`}}, {{6330.189872033729, 777.2495838696752, 
      111.32366494150033`}, {6244.086205899682, 
      1298.7717986302682`, -111.32366494149426`}}, {{6244.086205899679, 
      1298.7717986302823`, -111.32366494150033`}, {6160.412664539228, 
      1650.6775985057361`, -111.32366494150033`}}, {{6160.412664539227, 
      1650.6775985057466`, -111.32366494150033`}, {5993.772803705286, 
      2181.600118554199, -55.87656295086723}, {5781.06548117068, 
      2695.7551061653985`, -1.5631940186722204`*^-13}}, {{5781.065481170679, 
      2695.7551061653994`, 0.}, {5860.334199818241, 2488.467828398549, 
      389.64679321933465`}, {5910.6415798829685`, 2268.8827265019795`, 
      777.367980778417}}, {{5910.641579882968, 2268.882726501981, 
      777.3679807784142}, {5746.3376856709465`, 2552.167042541241, 
      1074.1788860353606`}, {5554.445582804334, 2823.1983862105385`, 
      1365.8326507166332`}, {5335.886546652357, 3080.6755340750383`, 
      1650.9290429954474`}}, {{5335.886546652356, 3080.6755340750396`, 
      1650.929042995449}, {5223.154521639683, 3015.589669087681, 
      2076.701586835865}, {5084.975812827236, 2935.8121543585403`, 
      2492.3566492945292`}, {4922.023612994678, 2841.7316579201715`, 
      2895.8692006886454`}}, {{4922.023612994677, 2841.7316579201706`, 
      2895.8692006886467`}, {4844.099115968555, 2528.922016706443, 
      3290.4514702529914`}, {4735.2658121953, 2199.9760496204217`, 
      3664.0383129753036`}, {4596.218135763364, 1856.992666507574, 
      4014.245978389595}}, {{4596.218135763363, 1856.9926665075716`, 
      4014.2459783895974`}, {4523.742295581507, 1646.5075430827324`, 
      4184.803728221746}}, {{4523.7422955815055`, 1646.5075430827283`, 
      4184.803728221748}, {4438.574076188791, 1554.3665354895736`, 
      4309.387257318023}}, {{4438.574076188799, 1554.366535489583, 
      4309.3872573180115`}, {4406.643597634345, 1780.3995813528325`, 
      4254.348742656964}}, {{4406.64359763434, 1780.3995813528597`, 
      4254.348742656958}, {4228.003638956987, 1971.550475552845, 
      4350.262939330659}}, {{4228.003638956987, 1971.5504755528448`, 
      4350.262939330659}, {4015.8655266667706`, 1872.6288492053718`, 
      4588.452784420167}}, {{4015.8655266667824`, 1872.6288492053773`, 
      4588.452784420154}, {3804.825697976732, 2109.049323588628, 
      4665.085856518208}}, {{3804.825697976748, 2109.0493235886106`, 
      4665.0858565182025`}, {3548.914834506785, 2217.6081126196345`, 
      4814.065999366986}}, {{3548.9148345067674`, 2217.608112619642, 
      4814.065999366996}, {3389.457001661398, 1802.206255994398, 
      5094.2563279386695`}}, {{3389.4570016614, 1802.206255994403, 
      5094.256327938667}, {3258.5470109185267`, 1450.7986016873278`, 
      5288.181964056837}}, {{3258.5470109185208`, 1450.7986016873133`, 
      5288.181964056845}, {3148.593925023453, 1468.2134596587596`, 
      5349.627951753491}}, {{3148.5939250234815`, 1468.213459658755, 
      5349.627951753476}, {2952.784144008986, 1440.1690540337324`, 
      5467.61306098856}}, {{2952.7841440089815`, 1440.1690540337318`, 
      5467.613060988564}, {2513.2420970156054`, 1225.790073670115, 
      5733.137583128493}}, {{2513.242097015599, 1225.790073670112, 
      5733.137583128497}, {2892.5791148369312`, 775.0642378636948, 
      5632.057810569238}}, {{2892.579114836936, 775.0642378636886, 
      5632.057810569237}, {2949.123277339624, 520.0100032303636, 
      5632.057810569237}}, {{2949.1232773396273`, 520.0100032303494, 
      5632.057810569237}, {3235.