(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     76901,       2457]
NotebookOptionsPosition[     66627,       2100]
NotebookOutlinePosition[     68209,       2143]
CellTagsIndexPosition[     68123,       2138]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Discrete Fourier Transforms" :> 
          Documentation`HelpLookup["paclet:tutorial/FourierTransforms"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Discrete Fourier Transforms\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"FourierDCT" :> 
          Documentation`HelpLookup["paclet:ref/FourierDCT"], "Fourier" :> 
          Documentation`HelpLookup["paclet:ref/Fourier"], 
          "FourierSinTransform" :> 
          Documentation`HelpLookup["paclet:ref/FourierSinTransform"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"FourierDCT\"\>", 
       2->"\<\"Fourier\"\>", 
       3->"\<\"FourierSinTransform\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Image Processing" :> 
          Documentation`HelpLookup["paclet:guide/ImageProcessing"], 
          "Signal Processing" :> 
          Documentation`HelpLookup["paclet:guide/SignalProcessing"], 
          "New in 6.0: Numerical Data Handling" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60NumericalDataHandling"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Image Processing\"\>", 
       2->"\<\"Signal Processing\"\>", 
       3->"\<\"New in 6.0: Numerical Data Handling\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["FourierDST", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["FourierDST",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/FourierDST"], "[", 
       StyleBox["list", "TI"], "]"}]], "InlineFormula"],
     " \[LineSeparator]finds the Fourier discrete sine transform of a list of \
real numbers."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["FourierDST",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/FourierDST"], "[", 
       RowBox[{
        StyleBox["list", "TI"], ",", 
        StyleBox["m", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]finds the Fourier discrete sine transform of type ",
     Cell[BoxData[
      FormBox["m", TraditionalForm]], "InlineMath"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "Possible types ",
 Cell[BoxData[
  FormBox["m", TraditionalForm]], "InlineMath"],
 " of discrete sine transform for a list ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["u", "r"], TraditionalForm]], "InlineMath"],
 " of length ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " giving a result ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["v", "s"], TraditionalForm]], "InlineMath"],
 " are:"
}], "Notes",
 CellID->321171497],

Cell[BoxData[GridBox[{
   {"1.", 
    RowBox[{"DST", "-", "I"}], Cell[BoxData[
     FormBox[
      RowBox[{
       SqrtBox[
        FractionBox["2", 
         RowBox[{"n", "+", "1"}]]], 
       RowBox[{
        UnderoverscriptBox["\[Sum]", 
         RowBox[{"r", "=", "1"}], "n"], 
        RowBox[{
         SubscriptBox["u", "r"], "  ", "sin", 
         RowBox[{"(", 
          RowBox[{
           FractionBox["\[Pi]", 
            RowBox[{"n", "+", "1"}]], " ", "r", " ", "s"}], ")"}]}]}]}], 
      TraditionalForm]], "InlineMath"]},
   {"2.", 
    RowBox[{"DST", "-", "II"}], Cell[BoxData[
     FormBox[
      RowBox[{
       FractionBox["1", 
        SqrtBox["n"]], 
       RowBox[{
        UnderoverscriptBox["\[Sum]", 
         RowBox[{"r", "=", "1"}], "n"], 
        RowBox[{
         SubscriptBox["u", "r"], "  ", "sin", 
         RowBox[{"(", 
          RowBox[{
           FractionBox["\[Pi]", "n"], " ", 
           RowBox[{"(", 
            RowBox[{"r", "-", 
             FractionBox["1", "2"]}], ")"}], " ", "s"}], ")"}]}]}]}], 
      TraditionalForm]], "InlineMath"]},
   {"3.", 
    RowBox[{"DST", "-", "III"}], Cell[BoxData[
     FormBox[
      RowBox[{
       FractionBox["1", 
        SqrtBox["n"]], 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"2", " ", 
          RowBox[{
           UnderoverscriptBox["\[Sum]", 
            RowBox[{"r", "=", "1"}], 
            RowBox[{"n", "-", "1"}]], 
           RowBox[{
            SubscriptBox["u", "r"], " ", 
            RowBox[{"sin", "(", 
             RowBox[{
              FractionBox["\[Pi]", "n"], " ", "r", " ", 
              RowBox[{"(", 
               RowBox[{"s", "-", 
                FractionBox["1", "2"]}], ")"}]}], ")"}]}]}]}], "+", 
         RowBox[{
          SuperscriptBox[
           RowBox[{"(", 
            RowBox[{"-", "1"}], ")"}], 
           RowBox[{"s", "-", "1"}]], 
          SubscriptBox["u", "n"]}]}], " ", ")"}], " "}], TraditionalForm]], 
     "InlineMath"]},
   {"4.", 
    RowBox[{"DST", "-", "IV"}], Cell[BoxData[
     FormBox[
      RowBox[{
       SqrtBox[
        FractionBox["2", "n"]], 
       RowBox[{
        UnderoverscriptBox["\[Sum]", 
         RowBox[{"r", "=", "1"}], "n"], 
        RowBox[{
         SubscriptBox["u", "r"], " ", "sin", 
         RowBox[{"(", 
          RowBox[{
           FractionBox["\[Pi]", "n"], " ", 
           RowBox[{"(", 
            RowBox[{"r", "-", 
             FractionBox["1", "2"]}], ")"}], " ", 
           RowBox[{"(", 
            RowBox[{"s", "-", 
             FractionBox["1", "2"]}], ")"}]}], ")"}]}]}]}], TraditionalForm]],
      "InlineMath"]}
  }]], "Notes",
 GridBoxOptions->{GridBoxAlignment->{"Columns" -> {{"Left"}}},
 GridBoxItemSize->{"Columns" -> {
     Scaled[0.02], 
     Scaled[0.12], 
     Scaled[0.64]}}},
 CellID->187749179],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["FourierDST",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FourierDST"], "[", 
   StyleBox["list", "TI"], "]"}]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["FourierDST",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FourierDST"], "[", 
   RowBox[{
    StyleBox["list", "TI"], ",", "2"}], "]"}]], "InlineFormula"],
 "."
}], "Notes",
 CellID->411046922],

Cell["\<\
The inverse discrete sine transforms for types 1, 2, 3, 4 are types 1, 3, 2, \
4, respectively.\
\>", "Notes",
 CellID->872153026],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  StyleBox["list", "TI"]], "InlineFormula"],
 " given in ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["FourierDST",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FourierDST"], "[", 
   StyleBox["list", "TI"], "]"}]], "InlineFormula"],
 " can be nested to represent an array of data in any number of dimensions. "
}], "Notes",
 CellID->29223],

Cell["The array of data must be rectangular. ", "Notes",
 CellID->15547],

Cell[TextData[{
 "If the elements of ",
 Cell[BoxData[
  StyleBox["list", "TI"]], "InlineFormula"],
 " are exact numbers, ",
 Cell[BoxData[
  ButtonBox["FourierDST",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FourierDST"]], "InlineFormula"],
 " begins by applying ",
 Cell[BoxData[
  ButtonBox["N",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/N"]], "InlineFormula"],
 " to them. "
}], "Notes",
 CellID->21017],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FourierDST",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FourierDST"]], "InlineFormula"],
 " can be used on ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 " objects. "
}], "Notes",
 CellID->20893]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->609290708],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->111463653],

Cell["Find a discrete sine transform:", "ExampleText",
 CellID->786559098],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FourierDST", "[", 
  RowBox[{"{", 
   RowBox[{"0", ",", "0", ",", "1", ",", "0", ",", "1"}], "}"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->147480397],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.5854101966249685`", ",", 
   RowBox[{"-", "0.2628655560595667`"}], ",", 
   RowBox[{"-", "0.08541019662496845`"}], ",", 
   RowBox[{"-", "0.42532540417601994`"}], ",", "0.8944271909999159`"}], 
  "}"}]], "Output",
 ImageSize->{366, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->71362833]
}, Open  ]],

Cell["Find the inverse discrete sine transform:", "ExampleText",
 CellID->169504947],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FourierDST", "[", 
  RowBox[{"%", ",", "3"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->876521411],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "8.326672684688674`*^-17", ",", "2.220446049250313`*^-16", ",", "1.`", ",", 
   "5.551115123125783`*^-17", ",", "1.`"}], "}"}]], "Output",
 ImageSize->{360, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->2729950]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->25285511],

Cell["Find a discrete sine transform of type 1 (DST-I):", "ExampleText",
 CellID->360961894],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FourierDST", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "0", ",", "1", ",", "2"}], "}"}], ",", "1"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->170348084],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.3660254037844388`", ",", 
   RowBox[{"-", "1.`"}], ",", "1.7320508075688779`", ",", 
   RowBox[{"-", "8.326672684688674`*^-17"}], ",", "0.36602540378443893`"}], 
  "}"}]], "Output",
 ImageSize->{336, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->610669744]
}, Open  ]],

Cell["Find the inverse discrete sine transform:", "ExampleText",
 CellID->252433387],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FourierDST", "[", 
  RowBox[{"%", ",", "1"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->366710795],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.0000000000000004`", ",", 
   RowBox[{"-", "2.7755575615628914`*^-17"}], ",", 
   RowBox[{"-", "1.6653345369377348`*^-16"}], ",", "1.`", ",", 
   "2.000000000000001`"}], "}"}]], "Output",
 ImageSize->{300, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->15190786]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1293636265],

Cell["\<\
Use machine arithmetic to compute the discrete sine transform:\
\>", "ExampleText",
 CellID->75805794],

Cell[BoxData[
 RowBox[{
  RowBox[{"v", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
    "0", ",", "1", ",", "2", ",", "3", ",", "4", ",", "3", ",", "2", ",", "1",
      ",", "0"}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->283753173],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FourierDST", "[", "v", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->158871569],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"4.567444499063787`", ",", "4.199928165665723`*^-17", ",", 
   RowBox[{"-", "0.9999999999999999`"}], ",", 
   RowBox[{"-", "2.8970898488349384`*^-16"}], ",", "0.06644681695159468`", 
   ",", "8.508276548579996`*^-17", ",", 
   RowBox[{"-", "0.36610868398461766`"}], ",", 
   RowBox[{"-", "9.73613550891983`*^-17"}], ",", "0.`"}], "}"}]], "Output",
 ImageSize->{391, 37},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->880806092]
}, Open  ]],

Cell["\<\
Use 24\[Hyphen]digit precision arithmetic:\
\>", "ExampleText",
 CellID->8571778],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FourierDST", "[", 
  RowBox[{"N", "[", 
   RowBox[{"v", ",", " ", "24"}], "]"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->343340528],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "4.5674444990637874817114087553900185051`23.492670612758822", ",", 
   "0``22.823846159168234", ",", 
   RowBox[{
   "-", "0.99999999999999999999999999999999999999`22.890960099224372"}], ",", 
   "0``22.88179328403019", ",", 
   "0.0664468169515947902244210433617284481`21.70452736541617", ",", 
   "0``22.834325851418544", ",", 
   RowBox[{
   "-", "0.36610868398461772806417020124825304679`22.399175240510917"}], ",", 
   "0``22.713935290882098", ",", "0``23.273001272063738"}], "}"}]], "Output",
 ImageSize->{571, 37},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->332753730]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->450525711],

Cell["2-dimensional discrete sine transform:", "ExampleText",
 CellID->312951839],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", " ", "=", " ", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", " ", 
     RowBox[{"{", 
      RowBox[{"4", ",", "3"}], "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->62446164],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FourierDST", "[", "m", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->80647354],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "0.7213936819102706`", ",", "0.19342213352502607`", ",", 
     "0.5113942721526287`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "0.05764909182384692`"}], ",", 
     RowBox[{"-", "0.09612452062873043`"}], ",", "0.2548177028691185`"}], 
    "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.12725451865023552`", ",", 
     RowBox[{"-", "0.1930021353027679`"}], ",", "0.07764989835205499`"}], 
    "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.37730944142035794`", ",", 
     RowBox[{"-", "0.33650863767682127`"}], ",", 
     RowBox[{"-", "0.1581014916257782`"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{468, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->452330538]
}, Open  ]],

Cell["4-dimensional discrete sine transform:", "ExampleText",
 CellID->26630158],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", " ", "=", " ", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", " ", 
     RowBox[{"{", 
      RowBox[{"4", ",", "4", ",", "4", ",", "4"}], "}"}]}], "]"}]}], 
  ";"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->3199137],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Max", "[", 
  RowBox[{"FourierDST", "[", "m", "]"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->49014697],

Cell[BoxData["1.416910689896044`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->1589779]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1020263627],

Cell["The list may have complex values:", "ExampleText",
 CellID->294060611],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FourierDST", "[", 
  RowBox[{"{", 
   RowBox[{"1", ",", 
    RowBox[{"2", "I"}], ",", "3", ",", 
    RowBox[{"4", "I"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->174976628],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"1.577161014949475`", "\[InvisibleSpace]", "+", 
    RowBox[{"1.6892463972414662`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "0.7071067811865475`"}], "-", 
    RowBox[{"0.7071067811865475`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "0.11208538229199139`"}], "+", 
    RowBox[{"1.4650756326574836`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{"2.`", "\[InvisibleSpace]", "-", 
    RowBox[{"3.`", " ", "\[ImaginaryI]"}]}]}], "}"}]], "Output",
 ImageSize->{505, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->23604008]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->190477669],

Cell[TextData[{
 "You can use ",
 Cell[BoxData["\"\<I\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<II\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<III\>\""], "InlineFormula"],
 ", or ",
 Cell[BoxData["\"\<IV\>\""], "InlineFormula"],
 " for the types ",
 Cell[BoxData["1"], "InlineFormula"],
 ", ",
 Cell[BoxData["2"], "InlineFormula"],
 ", ",
 Cell[BoxData["3"], "InlineFormula"],
 ", and ",
 Cell[BoxData["4"], "InlineFormula"],
 " respectively."
}], "ExampleText",
 CellID->258193052],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FourierDST", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1", ",", "0", ",", "0"}], "}"}], ",", 
   "\"\<IV\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->184074371],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.4472135954999578`", ",", "0.44721359549995815`", ",", 
   RowBox[{"-", "0.44721359549995804`"}], ",", 
   RowBox[{"-", "0.44721359549995804`"}], ",", "0.44721359549995804`"}], 
  "}"}]], "Output",
 ImageSize->{359, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->121341613]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->258228157],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Sine Series Expansion",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->97695754],

Cell["Get an expansion for an odd function as a sum of sines:", "ExampleText",
 CellID->948737116],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"f", "[", "x_", "]"}], ":=", 
   RowBox[{"Exp", "[", 
    RowBox[{
     RowBox[{"-", "100"}], 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"x", "-", 
        RowBox[{"1", "/", "2"}]}], ")"}], "^", "2"}]}], "]"}]}], 
  ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->292652122],

Cell[TextData[{
 "The function values on a uniformly spaced grid with ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " points on ",
 Cell[BoxData[
  RowBox[{"[", 
   RowBox[{
    RowBox[{"-", "L"}], ",", "L"}], ")"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->47689303],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"n", "=", "20"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"xg", "=", 
   RowBox[{
    RowBox[{"N", "[", 
     RowBox[{"Range", "[", 
      RowBox[{"n", "-", "1"}], "]"}], "]"}], "/", "n"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"fg", "=", 
   RowBox[{"Map", "[", 
    RowBox[{"f", ",", "xg"}], "]"}]}], ";"}], "\n", 
 RowBox[{"fp", "=", 
  RowBox[{"ListPlot", "[", 
   RowBox[{
    RowBox[{"Transpose", "[", 
     RowBox[{"{", 
      RowBox[{"xg", ",", "fg"}], "}"}], "]"}], ",", 
    RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]}]}], "Input",
 CellLabel->"In[2]:=",
 CellID->160587250],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzVWDtPAkEQXjgQO8FKAQ1RSisLKowmBk1MDJXRlhgTLXxEqf0FWmhnYqnW
JhL9ATZa2Vr7R84dbtflvtu9270IUZJ7zM5zZ74ZFjY73f29w073YLdTWz/t
nOwf7J7V1o5P+ZKXYYz1+HU+xfi7z9/6V8/3+TN6X+H3LBfwiMqTpPzwdzYz
yPV+uIzWqjG8mb7+Mt1zisPYGMiWIxK5yIqNTNXgjyRpoTICP3JfTMo5+8uL
jJqyJ/l6657lPk17MFvPp6ha4EWiqwAWKgPrWRF3VuBDZSBAaA10r2JWAyze
330sL91cXrwKvCm6YOJzQ0SfpNUXtBdsXjzo88ivzEBs729ffOXl+VPoKrpg
xd/eeuB0a/U2Evs4ZAR7DWuQxE+SR/9jEJ/dfsx8T1vnH6njoyelxb3yCpDX
UsIuign8CbCH/jDLs9ooJ0BrA+hFZ2TTKnXGdSrN5NVpWU2B5AbQzVAUZchS
A+hmjA+VhQbQeq10e0uXq5If/U6jLvZgxiuNPp6M05I56GF/UQxEmuQRuVlL
P2n1ojNevzOZQznL8dtjNHgetRez73k/3El1oFtaG3N+uKPqQLdCudaf3jDz
gU3Vb3WgW4bK65Bpi5icQwfgKYrij+sAeWLRI8xVXuYS+0O/W3PNVcT/H9W/
N0Xp1aVWttNm+JpxCJR8FwQOysVPUHk61kv9T2wM28tCn1bTtQh0O5RhG4QG
FtXkLQLdjsFGPA4Dy2r+FoF2s/ybk3L43ffXTgl/GeMlUV08qZnzNgmrnmVV
3PXwd8KOgx0SwF8OLvrROFTvpLXjgs7ov2ks8w3dqTTE\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 126},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->439112326]
}, Open  ]],

Cell["Compute the DST-I and renormalize:", "ExampleText",
 CellID->674066408],

Cell[BoxData[
 RowBox[{
  RowBox[{"coef", "=", 
   RowBox[{
    RowBox[{"FourierDST", "[", 
     RowBox[{"fg", ",", "1"}], "]"}], "/", 
    RowBox[{"Sqrt", "[", 
     RowBox[{"n", "/", "2"}], "]"}]}]}], ";"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->60039364],

Cell["\<\
The function has, in effect, been periodized with a particular odd symmetry:\
\>", "ExampleText",
 CellID->275165810],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", 
  RowBox[{"fp", ",", 
   RowBox[{"Plot", "[", 
    RowBox[{
     RowBox[{"Sum", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"coef", "[", 
         RowBox[{"[", "r", "]"}], "]"}], "*", 
        RowBox[{"Sin", "[", 
         RowBox[{"Pi", " ", "r", " ", "x"}], "]"}]}], ",", 
       RowBox[{"{", 
        RowBox[{"r", ",", 
         RowBox[{"n", "-", "1"}]}], "}"}]}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", 
       RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", " ", 
     RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->118986235],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 111},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->457884518]
}, Open  ]],

Cell["Plot the expansion error where the points are defined:", "ExampleText",
 CellID->286287023],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"f", "[", "x", "]"}], "-", 
    RowBox[{"Sum", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"coef", "[", 
        RowBox[{"[", "r", "]"}], "]"}], "*", 
       RowBox[{"Sin", "[", 
        RowBox[{"Pi", " ", "r", " ", "x"}], "]"}]}], ",", 
      RowBox[{"{", 
       RowBox[{"r", ",", 
        RowBox[{"n", "-", "1"}]}], "}"}]}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", 
     RowBox[{"1", "-", 
      RowBox[{"1", "/", "n"}]}]}], "}"}], ",", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->32402675],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 84},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->210549377]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Pseudospectral PDE Discretization",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->85182363],

Cell["\<\
Approximate the second derivative for a function with zero boundary \
conditions:\
\>", "ExampleText",
 CellID->282795626],

Cell[BoxData[
 RowBox[{
  RowBox[{"uxx", "[", 
   RowBox[{"u_", "?", "VectorQ"}], "]"}], " ", ":=", " ", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"n", " ", "=", " ", 
      RowBox[{"Length", "[", "u", "]"}]}], "}"}], ",", "\[IndentingNewLine]", 
    
    RowBox[{"FourierDST", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"-", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{"Pi", " ", 
           RowBox[{"N", "[", 
            RowBox[{
             RowBox[{"Range", "[", "n", "]"}], "/", 
             RowBox[{"(", 
              RowBox[{"n", "+", "1"}], ")"}]}], "]"}]}], ")"}], "^", "2"}]}], 
       " ", 
       RowBox[{"FourierDST", "[", 
        RowBox[{"u", ",", " ", "1"}], "]"}]}], ",", " ", "1"}], "]"}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->52719743],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"n", " ", "=", " ", "20"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"xg", " ", "=", " ", 
   RowBox[{
    RowBox[{"N", "[", 
     RowBox[{"Range", "[", "n", "]"}], "]"}], "/", 
    RowBox[{"(", 
     RowBox[{"n", "+", "1"}], ")"}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"u0", " ", "=", " ", 
   RowBox[{"1", " ", "-", " ", 
    RowBox[{"2", " ", 
     RowBox[{"Abs", "[", 
      RowBox[{"xg", " ", "-", " ", ".5"}], "]"}]}]}]}], ";"}], "\n", 
 RowBox[{"ListPlot", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"Transpose", "[", 
     RowBox[{"{", 
      RowBox[{"xg", ",", "u0"}], "}"}], "]"}], ",", " ", 
    RowBox[{"Transpose", "[", 
     RowBox[{"{", 
      RowBox[{"xg", ",", " ", 
       RowBox[{"uxx", "[", "u0", "]"}]}], "}"}], "]"}]}], "}"}], 
  "]"}]}], "Input",
 CellLabel->"In[2]:=",
 CellID->61936166],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzFV89PE0EUXrbdlpiaGGT5ZaPE2EvDydhLQ2MiVBMix43nhjSBGEGR/gWk
NzxyIOGiQTx4hODFGxc4cZQz/0id2d350W9eZ7dl0U12ut/beW9ev/fezNvV
1s56+0NrZ2OtNf9mu/VxfWPt8/zrrW0myo05jnPK7vczDnvusafwPu312K8+
PuBjjr8LL+crKZ1lg8tgTpM7HLN7Sn/nSZ14xniocXX0+6Dx6Xzvp+MRGC3z
Ry6btrybI/2c4UNeYSdeT3nsGzMKYCPCx8Eu8/BwqRNbUDhvWDAl5irmnFn5
D1yBrH5Nkja5oBjOvLn4c7K5f312GXuscPYe2zkvwoyHtG0tKtMjeShyJ20u
JPslLNL+iGzPiqdh/qPpkQf2R4+gqO6CtpZL8DWlyV2t8oWeigNH38i16P2F
lkZZYt877hJX/vP6bsjKfTb+4EUygB9W50yEde/IDEnAebVPiIwxJJ6URGkU
7/W0VPofS8fB63DVgsRyun5+KA+Gkcp8l6I0EllHfbunwi6Zm48BP+FDUWJ2
qRMjrEztBMHzDG09Ildc1qLH/fIBv0qorCxr8vvR1cvGwZe9c2cfMM1WpPUu
OGazmkuHsZbCbp+XUQQuL262Nk9+nV3HsxWm1xjmX6RnJQpWdFHdT57WsvYg
qFMCDruA7xE2sTvAnCoBw13AlE3c7znPHJaA/y7gtLaSeBrQp1hjIs4ePA1v
G3eUTkJMOoAnLFqK9Q7giT5/fOC1A5heY7S8z6gajH4f41AGntqA54g8SMrt
MrDYBkzZFL2y6OPKwG0b8CAbeh3TOUf3bxnM1bjH6kKG0kT0GUQiAPzUoqW4
DgDbtBS7AWBaK+uzK429BWChCbhq0VIsNAFXjQhS37QY9wVgqQm4SuRQ6j00
hRZXutU+TVpMWwlm/5BUEe5AT0TMaxBNxM/J6L6AaNYA27RUtGqAaa27y/nB
qzSAhQrgOmljEVioAK4bMUqT84vAUgVwPdN8HF5L36Ozr4/sux/+FvcwHzD9
5bAM8fQB27RUvLCnSfpK+fe18BbY8ACv9HmxCix4gFcsaygW8AuH1orjPfYX
cSs0Pw==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 107},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->96955887]
}, Open  ]],

Cell["Solve the wave equation for a plucked string:", "ExampleText",
 CellID->684207310],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", " ", "=", " ", 
  RowBox[{"First", "[", 
   RowBox[{"NDSolve", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"u", "'"}], "[", "t", "]"}], " ", "\[Equal]", " ", 
        RowBox[{"v", "[", "t", "]"}]}], ",", " ", 
       RowBox[{
        RowBox[{
         RowBox[{"v", "'"}], "[", "t", "]"}], " ", "\[Equal]", " ", 
        RowBox[{"uxx", "[", 
         RowBox[{"u", "[", "t", "]"}], "]"}]}], ",", " ", 
       RowBox[{
        RowBox[{"u", "[", "0", "]"}], " ", "\[Equal]", " ", "u0"}], ",", " ", 
       
       RowBox[{
        RowBox[{"v", "[", "0", "]"}], " ", "\[Equal]", " ", 
        RowBox[{"ConstantArray", "[", 
         RowBox[{"0.", ",", "n"}], "]"}]}]}], "}"}], ",", 
     "\[IndentingNewLine]", 
     RowBox[{"{", 
      RowBox[{"u", ",", "v"}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"t", ",", "0", ",", "25"}], "}"}]}], "]"}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->696096691],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"u", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "25.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], ",", 
   RowBox[{"v", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "25.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}]}], "}"}]], "Output",
 ImageSize->{302, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->145688220]
}, Open  ]],

Cell[TextData[{
 "Plot the solution ",
 Cell[BoxData[
  FormBox[
   RowBox[{"u", "(", 
    RowBox[{"x", ",", "t"}], ")"}], TraditionalForm]], "InlineMath"],
 " as a surface:"
}], "ExampleText",
 CellID->299769029],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot3D", "[", 
  RowBox[{
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"u", "[", "t", "]"}], " ", "/.", " ", "sol"}], ",", 
     RowBox[{"{", 
      RowBox[{"t", ",", "0", ",", "25"}], "}"}]}], "]"}], ",", " ", 
   RowBox[{"Mesh", "\[Rule]", "False"}], ",", 
   RowBox[{"DataRange", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"0", ",", "25"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "1"}], "}"}]}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->3529069],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 132},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->913582059]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->2123667759],

Cell["DST-I and DST-IV are their own inverses:", "ExampleText",
 CellID->393142439],

Cell[BoxData[
 RowBox[{
  RowBox[{"data", "=", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", "9"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->279848536],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Chop", "[", 
  RowBox[{
   RowBox[{"FourierDST", "[", 
    RowBox[{
     RowBox[{"FourierDST", "[", 
      RowBox[{"data", ",", "1"}], "]"}], ",", "1"}], "]"}], "-", "data"}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->279425620],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", 
   ",", "0"}], "}"}]], "Output",
 ImageSize->{171, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->20037490]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Chop", "[", 
  RowBox[{
   RowBox[{"FourierDST", "[", 
    RowBox[{
     RowBox[{"FourierDST", "[", 
      RowBox[{"data", ",", "4"}], "]"}], ",", "4"}], "]"}], "-", "data"}], 
  "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->27979728],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", 
   ",", "0"}], "}"}]], "Output",
 ImageSize->{171, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->36100724]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->320017841],

Cell["DST-II and DST-III are inverses of each other:", "ExampleText",
 CellID->19940023],

Cell[BoxData[
 RowBox[{
  RowBox[{"data", "=", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", "15"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->355011233],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Chop", "[", 
  RowBox[{
   RowBox[{"FourierDST", "[", 
    RowBox[{
     RowBox[{"FourierDST", "[", 
      RowBox[{"data", ",", "2"}], "]"}], ",", "3"}], "]"}], "-", "data"}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->678711840],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", 
   ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0"}], 
  "}"}]], "Output",
 ImageSize->{280, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->169351462]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Chop", "[", 
  RowBox[{
   RowBox[{"FourierDST", "[", 
    RowBox[{
     RowBox[{"FourierDST", "[", 
      RowBox[{"data", ",", "3"}], "]"}], ",", "2"}], "]"}], "-", "data"}], 
  "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->132848064],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", 
   ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0"}], 
  "}"}]], "Output",
 ImageSize->{280, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->957096643]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->217596405],

Cell["The DST is equivalent to matrix multiplication:", "ExampleText",
 CellID->73266107],

Cell[BoxData[
 RowBox[{
  RowBox[{"dstII", "[", "n_", "]"}], ":=", 
  RowBox[{
   FractionBox["1", 
    SqrtBox["n"]], 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"Sin", "[", 
      RowBox[{"Pi", 
       RowBox[{"(", 
        RowBox[{"r", "-", 
         RowBox[{"1", "/", "2"}]}], ")"}], 
       RowBox[{"s", "/", "n"}]}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"s", ",", "n"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"r", ",", "n"}], "}"}]}], "]"}]}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->45951218],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixForm", "[", 
  RowBox[{"dstII", "[", "7", "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->253948789],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      FractionBox[
       RowBox[{"Sin", "[", 
        FractionBox["\[Pi]", "14"], "]"}], 
       SqrtBox["7"]], 
      FractionBox[
       RowBox[{"Sin", "[", 
        FractionBox[
         RowBox[{"3", " ", "\[Pi]"}], "14"], "]"}], 
       SqrtBox["7"]], 
      FractionBox[
       RowBox[{"Cos", "[", 
        FractionBox["\[Pi]", "7"], "]"}], 
       SqrtBox["7"]], 
      FractionBox["1", 
       SqrtBox["7"]], 
      FractionBox[
       RowBox[{"Cos", "[", 
        FractionBox["\[Pi]", "7"], "]"}], 
       SqrtBox["7"]], 
      FractionBox[
       RowBox[{"Sin", "[", 
        FractionBox[
         RowBox[{"3", " ", "\[Pi]"}], "14"], "]"}], 
       SqrtBox["7"]], 
      FractionBox[
       RowBox[{"Sin", "[", 
        FractionBox["\[Pi]", "14"], "]"}], 
       SqrtBox["7"]]},
     {
      FractionBox[
       RowBox[{"Sin", "[", 
        FractionBox["\[Pi]", "7"], "]"}], 
       SqrtBox["7"]], 
      FractionBox[
       RowBox[{"Cos", "[", 
        FractionBox["\[Pi]", "14"], "]"}], 
       SqrtBox["7"]], 
      FractionBox[
       RowBox[{"Cos", "[", 
        FractionBox[
         RowBox[{"3", " ", "\[Pi]"}], "14"], "]"}], 
       SqrtBox["7"]], "0", 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Cos", "[", 
         FractionBox[
          RowBox[{"3", " ", "\[Pi]"}], "14"], "]"}], 
        SqrtBox["7"]]}], 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Cos", "[", 
         FractionBox["\[Pi]", "14"], "]"}], 
        SqrtBox["7"]]}], 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Sin", "[", 
         FractionBox["\[Pi]", "7"], "]"}], 
        SqrtBox["7"]]}]},
     {
      FractionBox[
       RowBox[{"Sin", "[", 
        FractionBox[
         RowBox[{"3", " ", "\[Pi]"}], "14"], "]"}], 
       SqrtBox["7"]], 
      FractionBox[
       RowBox[{"Cos", "[", 
        FractionBox["\[Pi]", "7"], "]"}], 
       SqrtBox["7"]], 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Sin", "[", 
         FractionBox["\[Pi]", "14"], "]"}], 
        SqrtBox["7"]]}], 
      RowBox[{"-", 
       FractionBox["1", 
        SqrtBox["7"]]}], 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Sin", "[", 
         FractionBox["\[Pi]", "14"], "]"}], 
        SqrtBox["7"]]}], 
      FractionBox[
       RowBox[{"Cos", "[", 
        FractionBox["\[Pi]", "7"], "]"}], 
       SqrtBox["7"]], 
      FractionBox[
       RowBox[{"Sin", "[", 
        FractionBox[
         RowBox[{"3", " ", "\[Pi]"}], "14"], "]"}], 
       SqrtBox["7"]]},
     {
      FractionBox[
       RowBox[{"Cos", "[", 
        FractionBox[
         RowBox[{"3", " ", "\[Pi]"}], "14"], "]"}], 
       SqrtBox["7"]], 
      FractionBox[
       RowBox[{"Sin", "[", 
        FractionBox["\[Pi]", "7"], "]"}], 
       SqrtBox["7"]], 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Cos", "[", 
         FractionBox["\[Pi]", "14"], "]"}], 
        SqrtBox["7"]]}], "0", 
      FractionBox[
       RowBox[{"Cos", "[", 
        FractionBox["\[Pi]", "14"], "]"}], 
       SqrtBox["7"]], 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Sin", "[", 
         FractionBox["\[Pi]", "7"], "]"}], 
        SqrtBox["7"]]}], 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Cos", "[", 
         FractionBox[
          RowBox[{"3", " ", "\[Pi]"}], "14"], "]"}], 
        SqrtBox["7"]]}]},
     {
      FractionBox[
       RowBox[{"Cos", "[", 
        FractionBox["\[Pi]", "7"], "]"}], 
       SqrtBox["7"]], 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Sin", "[", 
         FractionBox["\[Pi]", "14"], "]"}], 
        SqrtBox["7"]]}], 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Sin", "[", 
         FractionBox[
          RowBox[{"3", " ", "\[Pi]"}], "14"], "]"}], 
        SqrtBox["7"]]}], 
      FractionBox["1", 
       SqrtBox["7"]], 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Sin", "[", 
         FractionBox[
          RowBox[{"3", " ", "\[Pi]"}], "14"], "]"}], 
        SqrtBox["7"]]}], 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Sin", "[", 
         FractionBox["\[Pi]", "14"], "]"}], 
        SqrtBox["7"]]}], 
      FractionBox[
       RowBox[{"Cos", "[", 
        FractionBox["\[Pi]", "7"], "]"}], 
       SqrtBox["7"]]},
     {
      FractionBox[
       RowBox[{"Cos", "[", 
        FractionBox["\[Pi]", "14"], "]"}], 
       SqrtBox["7"]], 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Cos", "[", 
         FractionBox[
          RowBox[{"3", " ", "\[Pi]"}], "14"], "]"}], 
        SqrtBox["7"]]}], 
      FractionBox[
       RowBox[{"Sin", "[", 
        FractionBox["\[Pi]", "7"], "]"}], 
       SqrtBox["7"]], "0", 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Sin", "[", 
         FractionBox["\[Pi]", "7"], "]"}], 
        SqrtBox["7"]]}], 
      FractionBox[
       RowBox[{"Cos", "[", 
        FractionBox[
         RowBox[{"3", " ", "\[Pi]"}], "14"], "]"}], 
       SqrtBox["7"]], 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Cos", "[", 
         FractionBox["\[Pi]", "14"], "]"}], 
        SqrtBox["7"]]}]},
     {
      FractionBox["1", 
       SqrtBox["7"]], 
      RowBox[{"-", 
       FractionBox["1", 
        SqrtBox["7"]]}], 
      FractionBox["1", 
       SqrtBox["7"]], 
      RowBox[{"-", 
       FractionBox["1", 
        SqrtBox["7"]]}], 
      FractionBox["1", 
       SqrtBox["7"]], 
      RowBox[{"-", 
       FractionBox["1", 
        SqrtBox["7"]]}], 
      FractionBox["1", 
       SqrtBox["7"]]}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 ImageSize->{413, 269},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]//MatrixForm=",
 CellID->458920561]
}, Open  ]],

Cell[BoxData[
 RowBox[{
  RowBox[{"data", "=", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", "7"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->71580635],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Chop", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"dstII", "[", "7", "]"}], ".", "data"}], "-", 
   RowBox[{"FourierDST", "[", "data", "]"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->105998950],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0"}], 
  "}"}]], "Output",
 ImageSize->{135, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->875479753]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1305812373],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FourierDST",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FourierDST"]], "InlineFormula"],
 " always returns normalized results:"
}], "ExampleText",
 CellID->146202875],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TableForm", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"FourierDST", "[", 
     RowBox[{
      RowBox[{"ConstantArray", "[", 
       RowBox[{"1", ",", "5"}], "]"}], ",", "type"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"type", ",", "1", ",", "4"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->949419420],

Cell[BoxData[
 TagBox[GridBox[{
    {"2.154700538379252`", "0.`", "0.577350269189626`", "0.`", 
     "0.15470053837925146`"},
    {"1.4472135954999579`", "1.2102406750369443`*^-16", "0.5527864045000419`",
      
     RowBox[{"-", "1.0039712117215771`*^-16"}], "0.4472135954999579`"},
    {"2.8235955159711317`", "0.8777061007329484`", "0.44721359549995776`", 
     "0.22786670826713562`", "0.07083167502878462`"},
    {"2.021471201601977`", "0.6965515053690705`", "0.4472135954999581`", 
     "0.35491071886919645`", "0.32016958489789715`"}
   },
   GridBoxAlignment->{
    "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
     "RowsIndexed" -> {}},
   GridBoxSpacings->{"Columns" -> {
       Offset[0.27999999999999997`], {
        Offset[0.7]}, 
       Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
       Offset[0.2], {
        Offset[0.4]}, 
       Offset[0.2]}, "RowsIndexed" -> {}}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 ImageSize->{392, 61},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//TableForm=",
 CellID->127000706]
}, Open  ]],

Cell["\<\
To get unnormalized results, you can multiply by the normalization:\
\>", "ExampleText",
 CellID->104315647],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"nc", "[", 
    RowBox[{"n_", ",", "1"}], "]"}], "=", 
   RowBox[{"Sqrt", "[", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"n", "+", "1"}], ")"}], "/", "2"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"nc", "[", 
    RowBox[{"n_", ",", 
     RowBox[{"2", "|", "3"}]}], "]"}], "=", 
   RowBox[{"Sqrt", "[", "n", "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"nc", "[", 
    RowBox[{"n_", ",", "4"}], "]"}], "=", 
   RowBox[{"Sqrt", "[", 
    RowBox[{"n", "/", "2"}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[2]:=",
 CellID->77124536],

Cell[BoxData[
 RowBox[{
  RowBox[{"unnormalizedDST", "[", 
   RowBox[{"data_", ",", "type_"}], "]"}], ":=", 
  RowBox[{
   RowBox[{"FourierDST", "[", 
    RowBox[{"data", ",", "type"}], "]"}], "*", 
   RowBox[{"nc", "[", 
    RowBox[{
     RowBox[{"Length", "[", "data", "]"}], ",", "type"}], "]"}]}]}]], "Input",\

 CellLabel->"In[3]:=",
 CellID->388475004],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TableForm", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"unnormalizedDST", "[", 
     RowBox[{
      RowBox[{"ConstantArray", "[", 
       RowBox[{"1", ",", "5"}], "]"}], ",", "type"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"type", ",", "1", ",", "4"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->567054902],

Cell[BoxData[
 TagBox[GridBox[{
    {"3.7320508075688776`", "0.`", "1.0000000000000002`", "0.`", 
     "0.2679491924311226`"},
    {"3.23606797749979`", "2.7061804185178403`*^-16", "1.2360679774997894`", 
     RowBox[{"-", "2.24494787686228`*^-16"}], "1.`"},
    {"6.313751514675044`", "1.9626105055051508`", "0.9999999999999997`", 
     "0.5095254494944286`", "0.1583844403245368`"},
    {"3.196226610749831`", "1.1013446322926335`", "0.7071067811865478`", 
     "0.5611631188171801`", "0.5062325628940018`"}
   },
   GridBoxAlignment->{
    "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
     "RowsIndexed" -> {}},
   GridBoxSpacings->{"Columns" -> {
       Offset[0.27999999999999997`], {
        Offset[0.7]}, 
       Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
       Offset[0.2], {
        Offset[0.4]}, 
       Offset[0.2]}, "RowsIndexed" -> {}}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 ImageSize->{385, 61},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]//TableForm=",
 CellID->159163682]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["FourierDCT",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FourierDCT"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Fourier",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Fourier"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["FourierSinTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FourierSinTransform"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->250839057],

Cell[TextData[ButtonBox["Discrete Fourier Transforms",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/FourierTransforms"]], "Tutorials",
 CellID->1899]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Image Processing",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/ImageProcessing"]], "MoreAbout",
 CellID->27671915],

Cell[TextData[ButtonBox["Signal Processing",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/SignalProcessing"]], "MoreAbout",
 CellID->225153187],

Cell[TextData[ButtonBox["New in 6.0: Numerical Data Handling",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60NumericalDataHandling"]], "MoreAbout",
 CellID->740317552]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"FourierDST - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 44, 3.8450776}", "context" -> "System`", 
    "keywords" -> {
     "discrete Fourier transform", "discrete sine transform", 
      "fast Fourier transform", "fast sine transform", 
      "real odd Fourier transform", "sequence transform"}, "index" -> True, 
    "label" -> "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "FourierDST[list] finds the Fourier discrete sine transform of a list of \
real numbers. FourierDST[list, m] finds the Fourier discrete sine transform \
of type m.", 
    "synonyms" -> {
     "DST", "DSTI", "DSTII", "DSTIII", "DSTIV", "DST-I", "DST-II", "DST-III", 
      "DST-IV", "Fourier DST"}, "title" -> "FourierDST", "type" -> "Symbol", 
    "uri" -> "ref/FourierDST"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[11143, 365, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->609290708]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 67979, 2131}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2327, 57, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2950, 84, 1575, 37, 70, "ObjectNameGrid"],
Cell[4528, 123, 975, 30, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[5540, 158, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6018, 175, 461, 17, 70, "Notes",
 CellID->321171497],
Cell[6482, 194, 2809, 91, 70, "Notes",
 CellID->187749179],
Cell[9294, 287, 455, 17, 70, "Notes",
 CellID->411046922],
Cell[9752, 306, 140, 4, 70, "Notes",
 CellID->872153026],
Cell[9895, 312, 383, 13, 70, "Notes",
 CellID->29223],
Cell[10281, 327, 72, 1, 70, "Notes",
 CellID->15547],
Cell[10356, 330, 420, 16, 70, "Notes",
 CellID->21017],
Cell[10779, 348, 327, 12, 70, "Notes",
 CellID->20893]
}, Closed]],
Cell[CellGroupData[{
Cell[11143, 365, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->609290708],
Cell[CellGroupData[{
Cell[11528, 379, 148, 5, 70, "ExampleSection",
 CellID->111463653],
Cell[11679, 386, 74, 1, 70, "ExampleText",
 CellID->786559098],
Cell[CellGroupData[{
Cell[11778, 391, 186, 6, 28, "Input",
 CellID->147480397],
Cell[11967, 399, 385, 11, 36, "Output",
 CellID->71362833]
}, Open  ]],
Cell[12367, 413, 84, 1, 70, "ExampleText",
 CellID->169504947],
Cell[CellGroupData[{
Cell[12476, 418, 129, 4, 28, "Input",
 CellID->876521411],
Cell[12608, 424, 309, 9, 39, "Output",
 CellID->2729950]
}, Open  ]],
Cell[12932, 436, 124, 3, 70, "ExampleDelimiter",
 CellID->25285511],
Cell[13059, 441, 92, 1, 70, "ExampleText",
 CellID->360961894],
Cell[CellGroupData[{
Cell[13176, 446, 211, 7, 70, "Input",
 CellID->170348084],
Cell[13390, 455, 354, 10, 39, "Output",
 CellID->610669744]
}, Open  ]],
Cell[13759, 468, 84, 1, 70, "ExampleText",
 CellID->252433387],
Cell[CellGroupData[{
Cell[13868, 473, 129, 4, 70, "Input",
 CellID->366710795],
Cell[14000, 479, 358, 10, 39, "Output",
 CellID->15190786]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[14407, 495, 221, 7, 70, "ExampleSection",
 CellID->1293636265],
Cell[14631, 504, 112, 3, 70, "ExampleText",
 CellID->75805794],
Cell[14746, 509, 250, 8, 70, "Input",
 CellID->283753173],
Cell[CellGroupData[{
Cell[15021, 521, 106, 3, 70, "Input",
 CellID->158871569],
Cell[15130, 526, 517, 12, 58, "Output",
 CellID->880806092]
}, Open  ]],
Cell[15662, 541, 91, 3, 70, "ExampleText",
 CellID->8571778],
Cell[CellGroupData[{
Cell[15778, 548, 164, 5, 70, "Input",
 CellID->343340528],
Cell[15945, 555, 671, 17, 58, "Output",
 CellID->332753730]
}, Open  ]],
Cell[16631, 575, 125, 3, 70, "ExampleDelimiter",
 CellID->450525711],
Cell[16759, 580, 81, 1, 70, "ExampleText",
 CellID->312951839],
Cell[16843, 583, 240, 8, 70, "Input",
 CellID->62446164],
Cell[CellGroupData[{
Cell[17108, 595, 105, 3, 70, "Input",
 CellID->80647354],
Cell[17216, 600, 809, 24, 54, "Output",
 CellID->452330538]
}, Open  ]],
Cell[18040, 627, 80, 1, 70, "ExampleText",
 CellID->26630158],
Cell[18123, 630, 262, 9, 70, "Input",
 CellID->3199137],
Cell[CellGroupData[{
Cell[18410, 643, 135, 4, 70, "Input",
 CellID->49014697],
Cell[18548, 649, 173, 5, 36, "Output",
 CellID->1589779]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[18770, 660, 244, 7, 70, "ExampleSection",
 CellID->1020263627],
Cell[19017, 669, 76, 1, 70, "ExampleText",
 CellID->294060611],
Cell[CellGroupData[{
Cell[19118, 674, 213, 7, 70, "Input",
 CellID->174976628],
Cell[19334, 683, 661, 17, 36, "Output",
 CellID->23604008]
}, Open  ]],
Cell[20010, 703, 125, 3, 70, "ExampleDelimiter",
 CellID->190477669],
Cell[20138, 708, 498, 19, 70, "ExampleText",
 CellID->258193052],
Cell[CellGroupData[{
Cell[20661, 731, 221, 7, 70, "Input",
 CellID->184074371],
Cell[20885, 740, 369, 10, 36, "Output",
 CellID->121341613]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[21303, 756, 227, 7, 70, "ExampleSection",
 CellID->258228157],
Cell[CellGroupData[{
Cell[21555, 767, 241, 7, 70, "ExampleSubsection",
 CellID->97695754],
Cell[21799, 776, 98, 1, 70, "ExampleText",
 CellID->948737116],
Cell[21900, 779, 328, 13, 70, "Input",
 CellID->292652122],
Cell[22231, 794, 293, 11, 70, "ExampleText",
 CellID->47689303],
Cell[CellGroupData[{
Cell[22549, 809, 617, 21, 70, "Input",
 CellID->160587250],
Cell[23169, 832, 1050, 21, 70, "Output",
 Evaluatable->False,
 CellID->439112326]
}, Open  ]],
Cell[24234, 856, 77, 1, 70, "ExampleText",
 CellID->674066408],
Cell[24314, 859, 261, 9, 70, "Input",
 CellID->60039364],
Cell[24578, 870, 127, 3, 70, "ExampleText",
 CellID->275165810],
Cell[CellGroupData[{
Cell[24730, 877, 643, 20, 70, "Input",
 CellID->118986235],
Cell[25376, 899, 2567, 46, 70, "Output",
 Evaluatable->False,
 CellID->457884518]
}, Open  ]],
Cell[27958, 948, 97, 1, 70, "ExampleText",
 CellID->286287023],
Cell[CellGroupData[{
Cell[28080, 953, 647, 21, 70, "Input",
 CellID->32402675],
Cell[28730, 976, 5230, 90, 70, "Output",
 Evaluatable->False,
 CellID->210549377]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[34009, 1072, 253, 7, 70, "ExampleSubsection",
 CellID->85182363],
Cell[34265, 1081, 132, 4, 70, "ExampleText",
 CellID->282795626],
Cell[34400, 1087, 827, 27, 70, "Input",
 CellID->52719743],
Cell[CellGroupData[{
Cell[35252, 1118, 843, 28, 70, "Input",
 CellID->61936166],
Cell[36098, 1148, 1318, 26, 70, "Output",
 Evaluatable->False,
 CellID->96955887]
}, Open  ]],
Cell[37431, 1177, 88, 1, 70, "ExampleText",
 CellID->684207310],
Cell[CellGroupData[{
Cell[37544, 1182, 1002, 29, 70, "Input",
 CellID->696096691],
Cell[38549, 1213, 730, 27, 54, "Output",
 CellID->145688220]
}, Open  ]],
Cell[39294, 1243, 213, 8, 70, "ExampleText",
 CellID->299769029],
Cell[CellGroupData[{
Cell[39532, 1255, 572, 18, 70, "Input",
 CellID->3529069],
Cell[40107, 1275, 8179, 138, 70, "Output",
 Evaluatable->False,
 CellID->913582059]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[48347, 1420, 238, 7, 70, "ExampleSection",
 CellID->2123667759],
Cell[48588, 1429, 83, 1, 70, "ExampleText",
 CellID->393142439],
Cell[48674, 1432, 176, 6, 70, "Input",
 CellID->279848536],
Cell[CellGroupData[{
Cell[48875, 1442, 262, 9, 70, "Input",
 CellID->279425620],
Cell[49140, 1453, 280, 9, 36, "Output",
 CellID->20037490]
}, Open  ]],
Cell[CellGroupData[{
Cell[49457, 1467, 261, 9, 70, "Input",
 CellID->27979728],
Cell[49721, 1478, 280, 9, 36, "Output",
 CellID->36100724]
}, Open  ]],
Cell[50016, 1490, 125, 3, 70, "ExampleDelimiter",
 CellID->320017841],
Cell[50144, 1495, 88, 1, 70, "ExampleText",
 CellID->19940023],
Cell[50235, 1498, 177, 6, 70, "Input",
 CellID->355011233],
Cell[CellGroupData[{
Cell[50437, 1508, 262, 9, 70, "Input",
 CellID->678711840],
Cell[50702, 1519, 344, 10, 36, "Output",
 CellID->169351462]
}, Open  ]],
Cell[CellGroupData[{
Cell[51083, 1534, 262, 9, 70, "Input",
 CellID->132848064],
Cell[51348, 1545, 344, 10, 36, "Output",
 CellID->957096643]
}, Open  ]],
Cell[51707, 1558, 125, 3, 70, "ExampleDelimiter",
 CellID->217596405],
Cell[51835, 1563, 89, 1, 70, "ExampleText",
 CellID->73266107],
Cell[51927, 1566, 529, 19, 70, "Input",
 CellID->45951218],
Cell[CellGroupData[{
Cell[52481, 1589, 138, 4, 70, "Input",
 CellID->253948789],
Cell[52622, 1595, 6170, 222, 304, "Output",
 CellID->458920561]
}, Open  ]],
Cell[58807, 1820, 175, 6, 70, "Input",
 CellID->71580635],
Cell[CellGroupData[{
Cell[59007, 1830, 225, 7, 70, "Input",
 CellID->105998950],
Cell[59235, 1839, 257, 8, 36, "Output",
 CellID->875479753]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[59541, 1853, 231, 7, 70, "ExampleSection",
 CellID->1305812373],
Cell[59775, 1862, 216, 7, 70, "ExampleText",
 CellID->146202875],
Cell[CellGroupData[{
Cell[60016, 1873, 365, 11, 70, "Input",
 CellID->949419420],
Cell[60384, 1886, 1139, 28, 96, "Output",
 CellID->127000706]
}, Open  ]],
Cell[61538, 1917, 118, 3, 70, "ExampleText",
 CellID->104315647],
Cell[61659, 1922, 606, 22, 70, "Input",
 CellID->77124536],
Cell[62268, 1946, 358, 12, 70, "Input",
 CellID->388475004],
Cell[CellGroupData[{
Cell[62651, 1962, 370, 11, 70, "Input",
 CellID->567054902],
Cell[63024, 1975, 1108, 27, 96, "Output",
 CellID->159163682]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[64193, 2009, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[64512, 2020, 724, 26, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[65273, 2051, 318, 9, 70, "TutorialsSection",
 CellID->250839057],
Cell[65594, 2062, 154, 3, 70, "Tutorials",
 CellID->1899]
}, Open  ]],
Cell[CellGroupData[{
Cell[65785, 2070, 299, 8, 70, "MoreAboutSection"],
Cell[66087, 2080, 142, 3, 70, "MoreAbout",
 CellID->27671915],
Cell[66232, 2085, 145, 3, 70, "MoreAbout",
 CellID->225153187],
Cell[66380, 2090, 175, 3, 70, "MoreAbout",
 CellID->740317552]
}, Open  ]],
Cell[66570, 2096, 27, 0, 70, "History"],
Cell[66600, 2098, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

