(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     39064,       1433]
NotebookOptionsPosition[     31205,       1151]
NotebookOutlinePosition[     32587,       1190]
CellTagsIndexPosition[     32501,       1185]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Discrete Distributions" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/DiscreteDistributions"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Discrete Distributions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"NegativeBinomialDistribution" :> 
          Documentation`HelpLookup["paclet:ref/NegativeBinomialDistribution"],
           "BernoulliDistribution" :> 
          Documentation`HelpLookup["paclet:ref/BernoulliDistribution"], 
          "BinomialDistribution" :> 
          Documentation`HelpLookup["paclet:ref/BinomialDistribution"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"NegativeBinomialDistribution\"\>", 
       2->"\<\"BernoulliDistribution\"\>", 
       3->"\<\"BinomialDistribution\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Discrete Statistical Distributions" :> 
          Documentation`HelpLookup[
           "paclet:guide/DiscreteStatisticalDistributions"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"Discrete Statistical Distributions\"\>"}, "\<\"more about\"\>",
        
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["GeometricDistribution", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["GeometricDistribution",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/GeometricDistribution"], "[", 
       StyleBox["p", "TI"], "]"}]], "InlineFormula"],
     " \[LineSeparator]represents a geometric distribution with probability \
parameter ",
     Cell[BoxData[
      StyleBox["p", "TI"]], "InlineFormula"],
     ". "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->9463]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "The probability for value ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " in a geometric distribution is ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox[
    RowBox[{"p", "(", 
     RowBox[{"1", "-", "p"}], ")"}], "x"], TraditionalForm]], "InlineMath"],
 " for non-negative integers, and is zero otherwise.",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->28884,
  ButtonNote->"28884"]
}], "Notes",
 CellID->429991538],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["GeometricDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/GeometricDistribution"], "[", "p", "]"}]], 
  "InlineFormula"],
 " is the distribution of the number of failures in a sequence of trials with \
success probability ",
 Cell[BoxData[
  StyleBox["p", "TI"]], "InlineFormula"],
 " before a success occurs."
}], "Notes",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0.7, 0},
 ScriptMinSize->9,
 CellID->23225],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["GeometricDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/GeometricDistribution"]], "InlineFormula"],
 " can be used with such functions as ",
 Cell[BoxData[
  ButtonBox["Mean",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Mean"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["CDF",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CDF"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["RandomInteger",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RandomInteger"]], "InlineFormula"],
 ".",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->10820,
  ButtonNote->"10820"]
}], "Notes",
 CellID->33143606]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->325078534],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->7881384],

Cell["Mean and variance of a geometric distribution:", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 CellFrameLabels->{{None, None}, {None, None}},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->10820],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"GeometricDistribution", "[", "p", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->22266],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "1"}], "+", 
  FractionBox["1", "p"]}]], "Output",
 ImageSize->{43, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->99557703]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Variance", "[", 
  RowBox[{"GeometricDistribution", "[", "p", "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->21378],

Cell[BoxData[
 FractionBox[
  RowBox[{"1", "-", "p"}], 
  SuperscriptBox["p", "2"]]], "Output",
 ImageSize->{35, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->164831187]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->23038],

Cell["Probability density function:", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 CellFrameLabels->{{None, None}, {None, None}},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->28884],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"GeometricDistribution", "[", "p", "]"}], ",", "k"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->13359],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"1", "-", "p"}], ")"}], "k"], " ", "p"}]], "Output",
 ImageSize->{62, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->297859716]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"k", ",", 
       RowBox[{"PDF", "[", 
        RowBox[{
         RowBox[{"GeometricDistribution", "[", "0.2", "]"}], ",", "k"}], 
        "]"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"k", ",", "0", ",", "30"}], "}"}]}], "]"}], ",", 
   RowBox[{"Filling", "->", "Axis"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->11978],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztW71PFEEUX275qDVRQITwP4h2xsKolRWirRdihILDIJ2VjWLEaGNhRAs+
jFYaSbDRREMhYAItFhRgQWFMxIRgJLjOzM7sm3k7+3m3t7PIJre7b9/Xb968
NzP7cRfLowPXhsqjg/3l7gsj5RsDg/03u88Pj5BLdoNlWevk96jNIucOOWO/
dcchR3l/guxtwurkxxI5lsjxGKetRibGt3ZCNEpSgvZLUV2bS9mMeYru6WWH
bVYH3TUC7VqVJdoQbSO6le6aXNrb+6y0SrQHVeK3Iz7Vp+FroMeMkLdVbSEP
1DqfQXx6quuJktDzI7F6OL8rHlKQ4HrHk+Ijp/qoqJHsQXqd8fA1yfgsXlNW
s6yn1z4aab9D01IV80kpJqIGRV2XeBWICg7j05b4a1keQY4gbIsJrroZtrf3
d2dnlzWa0YQgl4De3v6jyEfxD+wJeZc/M71y5vSTB+PzCj1+f57LA9+lp6eW
Y8rHsmcdIvsVnp1BObC19Zunszc+u83k/N3dPcwnKgpdUHteDAl9b+wTl5+a
XCY0xPhK3wtCnzs7wfmYvnxJ0Hr5xYUNQr+bW+V8oDE/qXwse8G9vrn5C6L0
beMnjura2g/gE+GoqBpt7/mzL0rlkF4itNKrnPaiWhmaDe2l4cqs0gucPoz8
d+nwZNQrX1e/G2oPj4eidmaios6j2vtfRE2O0t07H325iaNydR9HhYzDoRUL
UbjuRUGsn4rfejKr4Mrgra3siz6XMx3mXqj/0X3USsjcWwVuFZ5BISOL3CqY
gW4XuBXQF9AKE8dCvAoYK0TMcRUXCTVkhpmo8YrnocEoIQNMRgk9DihNGAse
GxQzWFuahArWfCahgnwyA9VTw1DUo7bwankilxjgUTBPFHCnkA+KSeaVP8xh
m40QiPct9FJTPuiUNxlMx38l5K2RuShtL94moss+hnAvFg+leGtcnzyEESI+
OhkVjmnW90z+Wm521HdxWSN6yVxnPYaBlyzmrHq3oXhe8N3vqwy9QA1m6QVq
CLzUIrdeZ4L5wGoxrabJKPxc4U1NMOLaqqVVqKXaWJ0z0kqavjSvJWHrfvlb
vmrXW9ybslpqZoph66dq16BF8xr/bgCe+9SzjfBcp55e4Q7Zn7NifWs76fL0
vZOmlqrVijN25IPMfK3/KXb4admHBFowt4OWLnZ4zo7nY954KV1b88ITNcu6
QMUWNYZxm/6xN+S766iRN61N3YxVV3y+L7hTR8/3rbUec1jfJn3aEpZD8GQs
TKqvV7z112X+wuf1ytBbWtV6G0m+s053lWJZOvAdUzNonGC0Fbwa59pKTjOZ
yH+2YFpnKWBd1yLXi/vtMc3WJY2FFtSWkoTN3ei3xlRA51+rrXiHb1p13lkt
xPjPREAsEqw2XQ3/P6Sshn+9SGM8\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 144},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->632528177]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->20191],

Cell["\<\
Generate a set of pseudorandom numbers that are geometrically distributed:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->14890],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RandomInteger", "[", 
  RowBox[{
   RowBox[{"GeometricDistribution", "[", "0.45", "]"}], ",", "10"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->3093],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "2", ",", "1", ",", "0", ",", "1", ",", "2", ",", "0", ",", "1", ",", "2", 
   ",", "0", ",", "5"}], "}"}]], "Output",
 ImageSize->{189, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->199313077]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->2809],

Cell["\<\
Properties based on higher\[Hyphen]order moments:\
\>", "ExampleText",
 CellID->20699375],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", 
  RowBox[{"GeometricDistribution", "[", "p", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->9507],

Cell[BoxData[
 FractionBox[
  RowBox[{"2", "-", "p"}], 
  SqrtBox[
   RowBox[{"1", "-", "p"}]]]], "Output",
 ImageSize->{52, 35},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->259951270]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Kurtosis", "[", 
  RowBox[{"GeometricDistribution", "[", "p", "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->18222],

Cell[BoxData[
 RowBox[{"3", "+", 
  FractionBox[
   RowBox[{"6", "-", 
    RowBox[{"6", " ", "p"}], "+", 
    SuperscriptBox["p", "2"]}], 
   RowBox[{"1", "-", "p"}]]}]], "Output",
 ImageSize->{90, 35},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->549465226]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->10150],

Cell["Second moment of a geometric distribution:", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->15225],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ExpectedValue", "[", 
  RowBox[{
   RowBox[{"k", "^", "2"}], ",", 
   RowBox[{"GeometricDistribution", "[", "p", "]"}], ",", "k"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->7262],

Cell[BoxData[
 FractionBox[
  RowBox[{"2", "-", 
   RowBox[{"3", " ", "p"}], "+", 
   SuperscriptBox["p", "2"]}], 
  SuperscriptBox["p", "2"]]], "Output",
 ImageSize->{70, 36},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->39674157]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->113],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  FormBox["0.9", TraditionalForm]], "InlineMath"],
 " quantile of a geometric distribution with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"p", "=", 
    FractionBox["1", "3"]}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->48521834],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Quantile", "[", 
  RowBox[{
   RowBox[{"GeometricDistribution", "[", 
    RowBox[{"1", "/", "3"}], "]"}], ",", ".9"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->4591],

Cell[BoxData["5"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->828270547]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->16001],

Cell[TextData[{
 "The probability of at least 5 failures before 1 success for ",
 Cell[BoxData[
  FormBox[
   RowBox[{"p", "=", "0.4"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->32514],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"1", "-", 
  RowBox[{"CDF", "[", 
   RowBox[{
    RowBox[{"GeometricDistribution", "[", ".4", "]"}], ",", "4"}], 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->10642],

Cell[BoxData["0.07775999999999994`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->270749988]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->86062964],

Cell["\<\
Plot the cumulative distribution function of a geometric distribution:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->13370],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"CDF", "[", 
    RowBox[{
     RowBox[{"GeometricDistribution", "[", 
      RowBox[{"1", "/", "5"}], "]"}], ",", "k"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"k", ",", "0", ",", "15"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->26124],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzlWD1oVEEQ3ty7XCAX9ND8CkIaJTGlFlqIaBER0cpOLY4gJIVGYsQmoII/
YKeNSWMsTKEiKIFYGMFWkAQkAVGbKyQWBsFgERDO2Xnzsn/z9u7eO7lgjnf7
3uzOzM6b+fbbvTtVHB++cLE4PjJU7D0+Vrw8PDJ0pXdwdAy6giYhRAm+97sF
PJfhCb+lchnubtsHbRYUeqQUSE38iC5oMiDuonsn3OVwhuR2kkWAbvCj9weW
v22o1q/1CmZWOYvIKlmIbqfH1clYXnbIpmVDDpVoxg5L17YN49zHxCWn7Kpf
fJ4Y2i1ZOtuOrgasEf1tBBtND9XLjbxSVHYFd8omp3sI2wIbWSfNKyMKokqn
zBMbkfJAM4bx9LGVyjCYS1xBr4Yvtv6yWlt2JFGE0ZrL0FqK1pYud9A9a/kJ
Mayv8oKlseDpbYN2CTqbTJ3FhW9wiaxsODmmRnZsleS09lvNX73jQRw7yLC1
WlPOktZ+s1Wh0fngq9ZdQT5Qo74tFzZmlSy0yMbgY5pGWbq9+ilGx4KU5TzN
xnwm87dwtXBnSG0TxSDvUpXTyVl+bFyniEXLUBSBmZXG1C9cOevrf2g/yuN+
tLz0ncZnjfnyqL2y8uvq+NzopVmRl01w9847cGBqJ3sTB0fNG/o00FxFRZJU
UTJClu5x6NBZo9pYdNSl1NVyY2MuOtFsBQ4Jz5+rq78NxIJM3maYzFZaoyEz
v53/CnhWuAY5hc9quSIObbUix7e/8QiKzr6bo/71QmIcjjDfTK72ovz82cfp
Rx/gEq2yCWaeLBKnTTL5xvzGcoN83IMyuABH0mEbel1bW6fxh1V79bKRN4Y4
q4DJwv+Bqlot96P8eu7TkcNTcIm8bOR+RuP3GM8Ovrx7T4q8pt7TOJt6nnyq
RqCWsRzFkAQhR1EulX5qzH/dqNAx1Lh5Y97g8S+ff5CHiRpm24xcdRplOH8R
V+WRVYC7aHw0UZVCr8BNhldYFTV7rcdelpzX3B4+0zlvNHbdzqEMOTYwBdCj
8aLHCiplnFIU/yurf8F2I1FFT56YVrx2/uxT2tPOVO05ju1UJtMy1zWUX71c
NvILMo0PxuApYhHXV/zZkPNV7133NsrAORRDq/WL6lAir7dQhvLRL682zJJ6
s4NOtXSUxyPlQYQUI2PAsTQ+4LGCdzRqBkcdxyp+5scog6Gx96p1tZu1ehHl
wYhXMXzBYwVvZcSr9nnzn5h51J6afG9oK5bl/y16U+awp1Y8b8Wuuaa/FOok
/w==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 145},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->209777572]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->14720],

Cell["\<\
The probability of getting any real number except a non-negative integer is \
zero:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->25361],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"GeometricDistribution", "[", "p", "]"}], ",", "\[Pi]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->16983],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->424505843]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->6817],

Cell["Moments can be obtained from the characteristic function:", \
"ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->3912],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ExpectedValue", "[", 
  RowBox[{
   RowBox[{"k", "^", "4"}], ",", 
   RowBox[{"GeometricDistribution", "[", "p", "]"}], ",", "k"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->3709],

Cell[BoxData[
 FractionBox[
  RowBox[{"24", "-", 
   RowBox[{"60", " ", "p"}], "+", 
   RowBox[{"50", " ", 
    SuperscriptBox["p", "2"]}], "-", 
   RowBox[{"15", " ", 
    SuperscriptBox["p", "3"]}], "+", 
   SuperscriptBox["p", "4"]}], 
  SuperscriptBox["p", "4"]]], "Output",
 ImageSize->{169, 36},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->760453945]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"cf", "=", 
  RowBox[{"CharacteristicFunction", "[", 
   RowBox[{
    RowBox[{"GeometricDistribution", "[", "p", "]"}], ",", "t"}], 
   "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->182220216],

Cell[BoxData[
 FractionBox["p", 
  RowBox[{"1", "-", 
   RowBox[{
    SuperscriptBox["\[ExponentialE]", 
     RowBox[{"\[ImaginaryI]", " ", "t"}]], " ", 
    RowBox[{"(", 
     RowBox[{"1", "-", "p"}], ")"}]}]}]]], "Output",
 ImageSize->{91, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->521933495]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"D", "[", 
    RowBox[{"cf", ",", 
     RowBox[{"{", 
      RowBox[{"t", ",", "4"}], "}"}]}], "]"}], "/.", 
   RowBox[{"{", 
    RowBox[{"t", "->", "0"}], "}"}]}], "//", "Simplify"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->30497],

Cell[BoxData[
 FractionBox[
  RowBox[{"24", "-", 
   RowBox[{"60", " ", "p"}], "+", 
   RowBox[{"50", " ", 
    SuperscriptBox["p", "2"]}], "-", 
   RowBox[{"15", " ", 
    SuperscriptBox["p", "3"]}], "+", 
   SuperscriptBox["p", "4"]}], 
  SuperscriptBox["p", "4"]]], "Output",
 ImageSize->{169, 36},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->209280104]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->28742],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["GeometricDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/GeometricDistribution"]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  ButtonBox["NegativeBinomialDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NegativeBinomialDistribution"]], "InlineFormula"],
 " with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", "=", "1"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->21252],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"GeometricDistribution", "[", "p", "]"}], ",", "k"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->21367],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"1", "-", "p"}], ")"}], "k"], " ", "p"}]], "Output",
 ImageSize->{62, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->7257267]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"NegativeBinomialDistribution", "[", 
    RowBox[{"1", ",", "p"}], "]"}], ",", "k"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->18103],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"1", "-", "p"}], ")"}], "k"], " ", "p"}]], "Output",
 ImageSize->{62, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->34734385]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->932691717],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["GeometricDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/GeometricDistribution"]], "InlineFormula"],
 " is a mixture of ",
 Cell[BoxData[
  ButtonBox["PoissonDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PoissonDistribution"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["GammaDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/GammaDistribution"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->217493462],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"poispdf", "=", 
  RowBox[{"PDF", "[", 
   RowBox[{
    RowBox[{"PoissonDistribution", "[", "\[Mu]", "]"}], ",", "k"}], 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->364101735],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"-", "\[Mu]"}]], " ", 
   SuperscriptBox["\[Mu]", "k"]}], 
  RowBox[{"k", "!"}]]], "Output",
 ImageSize->{43, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->88294756]
}, Open  ]],

Cell[BoxData[
 RowBox[{
  RowBox[{"gdist", "=", 
   RowBox[{"GammaDistribution", "[", 
    RowBox[{"1", ",", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"1", "-", "p"}], ")"}], "/", "p"}]}], "]"}]}], ";"}]], "Input",\

 CellLabel->"In[2]:=",
 CellID->437860854],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ExpectedValue", "[", 
  RowBox[{"poispdf", ",", "gdist", ",", "\[Mu]", ",", 
   RowBox[{"Assumptions", "\[Rule]", 
    RowBox[{"k", ">", "0"}]}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->221897965],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"1", "-", "p"}], ")"}], "k"], " ", "p"}]], "Output",
 ImageSize->{62, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->3775265]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"GeometricDistribution", "[", "p", "]"}], ",", "k"}], 
  "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->15690664],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"1", "-", "p"}], ")"}], "k"], " ", "p"}]], "Output",
 ImageSize->{62, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->41190763]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[Cell[BoxData[
   InterpretationBox[Cell[TextData[{
     "Possible Issues",
     "\[NonBreakingSpace]\[NonBreakingSpace]",
     Cell["(2)", "ExampleCount"]
    }], "ExampleSection"],
    $Line = 0; Null]]]], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->30650],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["GeometricDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/GeometricDistribution"]], "InlineFormula"],
 " is not defined when ",
 Cell[BoxData[
  StyleBox["p", "TI"]], "InlineFormula"],
 " is not between zero and one:"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->10789],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"GeometricDistribution", "[", "2.4", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->20065],

Cell[BoxData[
 RowBox[{
  RowBox[{"GeometricDistribution", "::", "\<\"probparm\"\>"}], ":", 
  " ", "\<\"Parameter \\!\\(2.4`\\) is expected to be a probability between 0 \
and 1.\"\>"}]], "Message", "MSG",
 CellID->1011720545],

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"GeometricDistribution", "[", "2.4`", "]"}], "]"}]], "Output",
 ImageSize->{228, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->23616454]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->4447],

Cell["\<\
Substitution of invalid parameters into symbolic outputs gives results that \
are not meaningful:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->16727],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Mean", "[", 
   RowBox[{"GeometricDistribution", "[", "p", "]"}], "]"}], "/.", 
  RowBox[{"{", 
   RowBox[{"p", "->", 
    RowBox[{"-", "1000"}]}], "}"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->13503],

Cell[BoxData[
 RowBox[{"-", 
  FractionBox["1001", "1000"]}]], "Output",
 ImageSize->{44, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->422183224]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->29122],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["NegativeBinomialDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NegativeBinomialDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["BernoulliDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BernoulliDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["BinomialDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BinomialDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->25260]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->1286],

Cell[TextData[ButtonBox["Discrete Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/DiscreteDistributions"]], "Tutorials",
 CellID->4738]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Discrete Statistical Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DiscreteStatisticalDistributions"]], "MoreAbout",
 CellID->12182319]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"GeometricDistribution - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 45, 13.1437266}", "context" -> "System`", 
    "keywords" -> {
     "discrete memoryless", "number of trials before success", 
      "Saint Petersburg paradox"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "GeometricDistribution[p] represents a geometric distribution with \
probability parameter p. ", "synonyms" -> {"geometric distribution"}, "title" -> 
    "GeometricDistribution", "type" -> "Symbol", "uri" -> 
    "ref/GeometricDistribution"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[7299, 230, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->325078534]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 32358, 1178}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2181, 55, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2804, 82, 1586, 37, 70, "ObjectNameGrid"],
Cell[4393, 121, 609, 19, 70, "Usage",
 CellID->9463]
}, Open  ]],
Cell[CellGroupData[{
Cell[5039, 145, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[5517, 162, 478, 16, 70, "Notes",
 CellID->429991538],
Cell[5998, 180, 551, 17, 70, "Notes",
 CellID->23225],
Cell[6552, 199, 710, 26, 70, "Notes",
 CellID->33143606]
}, Closed]],
Cell[CellGroupData[{
Cell[7299, 230, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->325078534],
Cell[CellGroupData[{
Cell[7684, 244, 146, 5, 70, "ExampleSection",
 CellID->7881384],
Cell[7833, 251, 270, 6, 70, "ExampleText",
 CellID->10820],
Cell[CellGroupData[{
Cell[8128, 261, 144, 4, 28, "Input",
 CellID->22266],
Cell[8275, 267, 218, 8, 53, "Output",
 CellID->99557703]
}, Open  ]],
Cell[CellGroupData[{
Cell[8530, 280, 148, 4, 28, "Input",
 CellID->21378],
Cell[8681, 286, 225, 8, 54, "Output",
 CellID->164831187]
}, Open  ]],
Cell[8921, 297, 121, 3, 70, "ExampleDelimiter",
 CellID->23038],
Cell[9045, 302, 253, 6, 70, "ExampleText",
 CellID->28884],
Cell[CellGroupData[{
Cell[9323, 312, 170, 6, 70, "Input",
 CellID->13359],
Cell[9496, 320, 253, 9, 36, "Output",
 CellID->297859716]
}, Open  ]],
Cell[CellGroupData[{
Cell[9786, 334, 462, 15, 28, "Input",
 CellID->11978],
Cell[10251, 351, 1644, 31, 165, "Output",
 Evaluatable->False,
 CellID->632528177]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[11944, 388, 216, 7, 70, "ExampleSection",
 CellID->20191],
Cell[12163, 397, 258, 7, 70, "ExampleText",
 CellID->14890],
Cell[CellGroupData[{
Cell[12446, 408, 183, 6, 70, "Input",
 CellID->3093],
Cell[12632, 416, 291, 9, 36, "Output",
 CellID->199313077]
}, Open  ]],
Cell[12938, 428, 120, 3, 70, "ExampleDelimiter",
 CellID->2809],
Cell[13061, 433, 99, 3, 70, "ExampleText",
 CellID->20699375],
Cell[CellGroupData[{
Cell[13185, 440, 147, 4, 70, "Input",
 CellID->9507],
Cell[13335, 446, 237, 9, 56, "Output",
 CellID->259951270]
}, Open  ]],
Cell[CellGroupData[{
Cell[13609, 460, 148, 4, 70, "Input",
 CellID->18222],
Cell[13760, 466, 310, 11, 56, "Output",
 CellID->549465226]
}, Open  ]],
Cell[14085, 480, 121, 3, 70, "ExampleDelimiter",
 CellID->10150],
Cell[14209, 485, 218, 5, 70, "ExampleText",
 CellID->15225],
Cell[CellGroupData[{
Cell[14452, 494, 213, 7, 70, "Input",
 CellID->7262],
Cell[14668, 503, 283, 10, 57, "Output",
 CellID->39674157]
}, Open  ]],
Cell[14966, 516, 119, 3, 70, "ExampleDelimiter",
 CellID->113],
Cell[15088, 521, 429, 15, 70, "ExampleText",
 CellID->48521834],
Cell[CellGroupData[{
Cell[15542, 540, 197, 6, 70, "Input",
 CellID->4591],
Cell[15742, 548, 158, 5, 36, "Output",
 CellID->828270547]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[15949, 559, 223, 7, 70, "ExampleSection",
 CellID->16001],
Cell[16175, 568, 347, 11, 70, "ExampleText",
 CellID->32514],
Cell[CellGroupData[{
Cell[16547, 583, 197, 7, 70, "Input",
 CellID->10642],
Cell[16747, 592, 177, 5, 36, "Output",
 CellID->270749988]
}, Open  ]],
Cell[16939, 600, 124, 3, 70, "ExampleDelimiter",
 CellID->86062964],
Cell[17066, 605, 254, 7, 70, "ExampleText",
 CellID->13370],
Cell[CellGroupData[{
Cell[17345, 616, 313, 10, 70, "Input",
 CellID->26124],
Cell[17661, 628, 1498, 29, 70, "Output",
 Evaluatable->False,
 CellID->209777572]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[19208, 663, 233, 7, 70, "ExampleSection",
 CellID->14720],
Cell[19444, 672, 267, 8, 70, "ExampleText",
 CellID->25361],
Cell[CellGroupData[{
Cell[19736, 684, 174, 6, 70, "Input",
 CellID->16983],
Cell[19913, 692, 158, 5, 36, "Output",
 CellID->424505843]
}, Open  ]],
Cell[20086, 700, 120, 3, 70, "ExampleDelimiter",
 CellID->6817],
Cell[20209, 705, 234, 6, 70, "ExampleText",
 CellID->3912],
Cell[CellGroupData[{
Cell[20468, 715, 213, 7, 70, "Input",
 CellID->3709],
Cell[20684, 724, 409, 14, 57, "Output",
 CellID->760453945]
}, Open  ]],
Cell[CellGroupData[{
Cell[21130, 743, 220, 7, 70, "Input",
 CellID->182220216],
Cell[21353, 752, 354, 12, 53, "Output",
 CellID->521933495]
}, Open  ]],
Cell[CellGroupData[{
Cell[21744, 769, 278, 10, 70, "Input",
 CellID->30497],
Cell[22025, 781, 409, 14, 57, "Output",
 CellID->209280104]
}, Open  ]],
Cell[22449, 798, 121, 3, 70, "ExampleDelimiter",
 CellID->28742],
Cell[22573, 803, 618, 20, 70, "ExampleText",
 CellID->21252],
Cell[CellGroupData[{
Cell[23216, 827, 170, 6, 70, "Input",
 CellID->21367],
Cell[23389, 835, 251, 9, 36, "Output",
 CellID->7257267]
}, Open  ]],
Cell[CellGroupData[{
Cell[23677, 849, 199, 6, 70, "Input",
 CellID->18103],
Cell[23879, 857, 252, 9, 36, "Output",
 CellID->34734385]
}, Open  ]],
Cell[24146, 869, 125, 3, 70, "ExampleDelimiter",
 CellID->932691717],
Cell[24274, 874, 514, 17, 70, "ExampleText",
 CellID->217493462],
Cell[CellGroupData[{
Cell[24813, 895, 208, 7, 70, "Input",
 CellID->364101735],
Cell[25024, 904, 310, 11, 54, "Output",
 CellID->88294756]
}, Open  ]],
Cell[25349, 918, 267, 10, 70, "Input",
 CellID->437860854],
Cell[CellGroupData[{
Cell[25641, 932, 231, 6, 70, "Input",
 CellID->221897965],
Cell[25875, 940, 251, 9, 36, "Output",
 CellID->3775265]
}, Open  ]],
Cell[CellGroupData[{
Cell[26163, 954, 173, 6, 70, "Input",
 CellID->15690664],
Cell[26339, 962, 252, 9, 36, "Output",
 CellID->41190763]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[26640, 977, 327, 9, 70, "ExampleSection",
 CellID->30650],
Cell[26970, 988, 448, 14, 70, "ExampleText",
 CellID->10789],
Cell[CellGroupData[{
Cell[27443, 1006, 146, 4, 70, "Input",
 CellID->20065],
Cell[27592, 1012, 227, 5, 70, "Message",
 CellID->1011720545],
Cell[27822, 1019, 239, 7, 36, "Output",
 CellID->23616454]
}, Open  ]],
Cell[28076, 1029, 120, 3, 70, "ExampleDelimiter",
 CellID->4447],
Cell[28199, 1034, 281, 8, 70, "ExampleText",
 CellID->16727],
Cell[CellGroupData[{
Cell[28505, 1046, 240, 8, 70, "Input",
 CellID->13503],
Cell[28748, 1056, 202, 7, 51, "Output",
 CellID->422183224]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[29011, 1070, 311, 9, 70, "SeeAlsoSection",
 CellID->29122],
Cell[29325, 1081, 786, 26, 70, "SeeAlso",
 CellID->25260]
}, Open  ]],
Cell[CellGroupData[{
Cell[30148, 1112, 313, 9, 70, "TutorialsSection",
 CellID->1286],
Cell[30464, 1123, 153, 3, 70, "Tutorials",
 CellID->4738]
}, Open  ]],
Cell[CellGroupData[{
Cell[30654, 1131, 299, 8, 70, "MoreAboutSection"],
Cell[30956, 1141, 177, 3, 70, "MoreAbout",
 CellID->12182319]
}, Open  ]],
Cell[31148, 1147, 27, 0, 70, "History"],
Cell[31178, 1149, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

