(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    132000,       4066]
NotebookOptionsPosition[    114427,       3449]
NotebookOutlinePosition[    115903,       3488]
CellTagsIndexPosition[    115816,       3483]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"DiracDelta" :> 
          Documentation`HelpLookup["paclet:ref/DiracDelta"], "UnitStep" :> 
          Documentation`HelpLookup["paclet:ref/UnitStep"], "PrincipalValue" :> 
          Documentation`HelpLookup["paclet:ref/PrincipalValue"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"DiracDelta\"\>", 
       2->"\<\"UnitStep\"\>", 
       3->"\<\"PrincipalValue\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Generalized Functions" :> 
          Documentation`HelpLookup["paclet:guide/GeneralizedFunctions"], 
          "Mathematical Functions" :> 
          Documentation`HelpLookup["paclet:guide/MathematicalFunctions"], 
          "New in 6.0: Mathematical Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicalFunctions"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Generalized Functions\"\>", 
       2->"\<\"Mathematical Functions\"\>", 
       3->"\<\"New in 6.0: Mathematical Functions\"\>"}, "\<\"more about\"\>",
        
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["HeavisideTheta", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["HeavisideTheta",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/HeavisideTheta"], "[", 
       StyleBox["x", "TI"], "]"}]], "InlineFormula"],
     "\[LineSeparator]represents the Heaviside theta function ",
     Cell[BoxData[
      FormBox[
       RowBox[{"\[Theta]", "(", "x", ")"}], TraditionalForm]], "InlineMath"],
     ", equal to 0 for ",
     Cell[BoxData[
      FormBox[
       RowBox[{"x", "<", "0"}], TraditionalForm]], "InlineMath"],
     " and 1 for ",
     Cell[BoxData[
      FormBox[
       RowBox[{"x", ">", "0"}], TraditionalForm]], "InlineMath"],
     ". "
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["HeavisideTheta",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/HeavisideTheta"], "[", 
       RowBox[{
        SubscriptBox[
         StyleBox["x", "TI"], 
         StyleBox["1", "TR"]], ",", 
        SubscriptBox[
         StyleBox["x", "TI"], 
         StyleBox["2", "TR"]], ",", 
        StyleBox["\[Ellipsis]", "TR"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]represents the multidimensional Heaviside theta \
function which is 1 only if none of the ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["i", "TI"]]], "InlineFormula"],
     " are not positive. "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->26346]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["HeavisideTheta",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/HeavisideTheta"], "[", 
   StyleBox["x", "TI"], "]"}]], "InlineFormula"],
 " returns 0 or 1 for all numeric ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " other than 0. "
}], "Notes",
 CellID->20509],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["HeavisideTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HeavisideTheta"]], "InlineFormula"],
 " can be used in integrals, integral transforms and differential equations. \
"
}], "Notes",
 CellID->30080],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["HeavisideTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HeavisideTheta"]], "InlineFormula"],
 " has attribute ",
 Cell[BoxData[
  ButtonBox["Orderless",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Orderless"]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->7981],

Cell[TextData[{
 "For exact numeric quantities, ",
 Cell[BoxData[
  ButtonBox["HeavisideTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HeavisideTheta"]], "InlineFormula"],
 " internally uses numerical approximations to establish its result. This \
process can be affected by the setting of the global variable ",
 Cell[BoxData[
  ButtonBox["$MaxExtraPrecision",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/$MaxExtraPrecision"]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->23207]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->122089080],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->465933886],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"HeavisideTheta", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "2"}], ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->255218453],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzzTSzJSM1NLMlMTlRwL0osyMhMLlZwyy8CCjEzMjAwNAFxggQDkP0fyALj
pv//gTQyyQYkWYBS0kCaGUhLAGkmIC0D5UtB5ZnBqtlBJEgYzIOogvFBTDEQ
wYLgY6qQQjMBYi6MAgEmGBdJlwiyK0iyTxqNzwRWxonHVcSZi85npKm5bAT1
ETZZEkMFIk5hMQ6iJdH4MJrlP3ragZBOaPbYIlzMvH3bjfnzzgARgyWIAKn6
+fMPQhXIzFqgIJIvkUxCj+1Bwz9/7iktzEdPq9hDxRLNFbbwWADFkh1qnGCo
IUWUduYZQ3nIIQAuDWDpDEO/0X/01MuKppdYfThTGQF97Gj6QCUJiEuKemLs
YcEugqETFoZsUJp16KYAVrhvoXHJitUE6sUlKERZoDS+OGRBMp8Ud8Higjrq
YWGFnqKYh26MI+cLEsKJ2LxBrE6CcU+2Xcwkxj5yiODP77D6mmVQxT4hnUMt
9knNwdR221Ar71HzOUgJvL4jGNMmaCqZiQwR2uoDKbC0t53JwOzqPOfhw/cg
GUZS7YWKEGuvC9aYdgWLYra8GRgBXKAksg==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{130, 96},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->512734480]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->229644424],

Cell[TextData[{
 "Differentiate to obtain ",
 Cell[BoxData[
  ButtonBox["DiracDelta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DiracDelta"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->198163040],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"D", "[", 
  RowBox[{
   RowBox[{"HeavisideTheta", "[", "x", "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->4240533],

Cell[BoxData[
 RowBox[{"DiracDelta", "[", "x", "]"}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{95, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->914881128]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->18618],

Cell[TextData[{
 "Generate ",
 Cell[BoxData[
  ButtonBox["HeavisideTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HeavisideTheta"]], "InlineFormula"],
 " from an integral:"
}], "ExampleText",
 CellID->552355627],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"DiracDelta", "[", 
    RowBox[{"x", "-", "a"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "Infinity"}], ",", "b"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->389936162],

Cell[BoxData[
 RowBox[{"If", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"a", "\[Element]", "Reals"}], "&&", 
    RowBox[{"b", "\[Element]", "Reals"}]}], ",", 
   RowBox[{
    RowBox[{"HeavisideTheta", "[", 
     RowBox[{"-", "a"}], "]"}], "+", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "1"}], "+", 
       RowBox[{"2", " ", 
        RowBox[{"HeavisideTheta", "[", "b", "]"}]}]}], ")"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"a", "-", 
       RowBox[{"b", " ", 
        RowBox[{"HeavisideTheta", "[", 
         RowBox[{"-", "b"}], "]"}]}]}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{
       RowBox[{"-", "a"}], "+", 
       RowBox[{"b", " ", 
        RowBox[{"HeavisideTheta", "[", "b", "]"}]}]}], "]"}]}]}], ",", 
   RowBox[{"Integrate", "[", 
    RowBox[{
     RowBox[{"DiracDelta", "[", 
      RowBox[{
       RowBox[{"-", "a"}], "+", "x"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", 
       RowBox[{"-", "\[Infinity]"}], ",", "b"}], "}"}], ",", 
     RowBox[{"Assumptions", "\[Rule]", 
      RowBox[{
       RowBox[{
        RowBox[{"Im", "[", "a", "]"}], "<", "0"}], "||", 
       RowBox[{
        RowBox[{"Im", "[", "a", "]"}], ">", "0"}], "||", 
       RowBox[{
        RowBox[{"Im", "[", "b", "]"}], "<", "0"}], "||", 
       RowBox[{
        RowBox[{"Im", "[", "b", "]"}], ">", "0"}]}]}]}], "]"}]}], 
  "]"}]], "Output",
 ImageSize->{500, 67},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->58312727]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->136559208],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["HeavisideTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HeavisideTheta"]], "InlineFormula"],
 " threads element-wise over lists:"
}], "ExampleText",
 CellID->87642531],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"HeavisideTheta", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"-", "1"}], ",", "0", ",", "1"}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->223027789],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0", ",", 
   RowBox[{"HeavisideTheta", "[", "0", "]"}], ",", "1"}], "}"}]], "Output",
 ImageSize->{174, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->8957073]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->647755012],

Cell["Integrate over finite and infinite domains:", "ExampleText",
 CellID->31876294],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"HeavisideTheta", "[", 
    RowBox[{"x", "-", "a"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "y"}], "}"}], ",", 
   RowBox[{"Assumptions", "->", 
    RowBox[{"y", ">", "0"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->185869160],

Cell[BoxData[
 RowBox[{
  RowBox[{"(", 
   RowBox[{
    RowBox[{"y", " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"-", "a"}], "]"}]}], "+", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "a"}], "+", "y"}], ")"}], " ", 
     RowBox[{"HeavisideTheta", "[", "a", "]"}]}]}], ")"}], " ", 
  RowBox[{"HeavisideTheta", "[", 
   RowBox[{
    RowBox[{"-", "a"}], "+", "y"}], "]"}]}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{487, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->542186415]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"HeavisideTheta", "[", 
     RowBox[{"x", "-", "a"}], "]"}], " ", 
    RowBox[{"Exp", "[", 
     RowBox[{"-", 
      RowBox[{"x", "^", "2"}]}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "Infinity"}], ",", "Infinity"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->311609859],

Cell[BoxData[
 RowBox[{
  FractionBox["1", "2"], " ", 
  SqrtBox["\[Pi]"], " ", 
  RowBox[{"Erfc", "[", "a", "]"}]}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{95, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->80028367]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"HeavisideTheta", "[", 
    RowBox[{"x", " ", 
     RowBox[{"Cos", "[", "x", "]"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "5"}], ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->17399961],

Cell[BoxData["5"], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->484103998]
}, Open  ]],

Cell["Numerical integration:", "ExampleText",
 CellID->474749005],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"HeavisideTheta", "[", 
     RowBox[{"x", " ", 
      RowBox[{"Cos", "[", "x", "]"}]}], "]"}], " ", 
    RowBox[{"Sin", "[", "x", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", 
     RowBox[{"Pi", "/", "2"}], ",", 
     RowBox[{"3", 
      RowBox[{"Pi", "/", "2"}]}], ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->183728440],

Cell[BoxData["0.716337814536774`"], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->123425078]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->182430],

Cell[TextData[{
 "Integrate expressions containing symbolic derivatives of ",
 Cell[BoxData[
  ButtonBox["HeavisideTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HeavisideTheta"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->243133372],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"HeavisideTheta", "'''"}], "[", 
     RowBox[{"x", "-", "a"}], "]"}], " ", 
    RowBox[{"f", "[", "x", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "Infinity"}], ",", "Infinity"}], "}"}], ",", 
   RowBox[{"Assumptions", "->", 
    RowBox[{"a", "\[Element]", "Reals"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->480136610],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["f", "\[Prime]\[Prime]",
   MultilineFunction->None], "[", "a", "]"}]], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->725572585]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->611122967],

Cell["Indefinite integral interpretation for real arguments:", "ExampleText",
 CellID->259982338],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"HeavisideTheta", "[", "x", "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->147978867],

Cell[BoxData[
 RowBox[{"x", " ", 
  RowBox[{"HeavisideTheta", "[", "x", "]"}]}]], "Output",
 ImageSize->{133, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->483347866]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->31720],

Cell[TextData[{
 "Multivariate ",
 Cell[BoxData[
  ButtonBox["HeavisideTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HeavisideTheta"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->461353717],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"HeavisideTheta", "[", 
  RowBox[{"1", ",", "1", ",", "1", ",", "1", ",", "1", ",", "1", ",", 
   RowBox[{"-", "1"}], ",", 
   RowBox[{"-", "1"}], ",", "1", ",", "1"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->105408681],

Cell[BoxData["0"], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->133075188]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->141883382],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"HeavisideTheta", "[", 
    RowBox[{"x", ",", "y"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "a"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", "b"}], "}"}], ",", 
   RowBox[{"Assumptions", "->", 
    RowBox[{"Element", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"a", ",", "b"}], "}"}], ",", "Reals"}], "]"}]}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->5135933],

Cell[BoxData[
 RowBox[{"a", " ", "b", " ", 
  RowBox[{"HeavisideTheta", "[", "a", "]"}], " ", 
  RowBox[{"HeavisideTheta", "[", "b", "]"}]}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{265, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->20206465]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->429962081],

Cell[TextData[{
 "Differentiate the multivariate ",
 Cell[BoxData[
  ButtonBox["HeavisideTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HeavisideTheta"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->5507245],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"D", "[", 
  RowBox[{
   RowBox[{"HeavisideTheta", "[", 
    RowBox[{"x", ",", 
     RowBox[{"f", "[", "y", "]"}], ",", "z"}], "]"}], ",", "y"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->240921601],

Cell[BoxData[
 RowBox[{
  RowBox[{"DiracDelta", "[", 
   RowBox[{"f", "[", "y", "]"}], "]"}], " ", 
  RowBox[{"HeavisideTheta", "[", 
   RowBox[{"x", ",", "z"}], "]"}], " ", 
  RowBox[{
   SuperscriptBox["f", "\[Prime]",
    MultilineFunction->None], "[", "y", "]"}]}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{291, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->91629666]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(6)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->30703],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 " with ",
 Cell[BoxData[
  ButtonBox["DiracDelta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DiracDelta"]], "InlineFormula"],
 " source term to find Green's function:"
}], "ExampleText",
 CellID->138952721],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"gf", "[", "z_", "]"}], "=", 
  RowBox[{
   RowBox[{"w", "[", "z", "]"}], "/.", 
   RowBox[{
    RowBox[{"DSolve", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{
           RowBox[{"w", "''"}], "[", "z", "]"}], "+", 
          RowBox[{
           SuperscriptBox["k", "2"], " ", 
           RowBox[{"w", "[", "z", "]"}]}]}], "==", 
         RowBox[{"DiracDelta", "[", "z", "]"}]}], ",", 
        RowBox[{
         RowBox[{"w", "[", "0", "]"}], "==", "0"}], ",", 
        RowBox[{
         RowBox[{
          RowBox[{"w", "'"}], "[", "0", "]"}], "==", "1"}]}], "}"}], ",", " ",
       
      RowBox[{"w", "[", "z", "]"}], ",", "z"}], "]"}], "[", 
    RowBox[{"[", "1", "]"}], "]"}]}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->33955874],

Cell[BoxData[
 FractionBox[
  RowBox[{
   RowBox[{"HeavisideTheta", "[", "z", "]"}], " ", 
   RowBox[{"Sin", "[", 
    RowBox[{"k", " ", "z"}], "]"}]}], "k"]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{183, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->76629991]
}, Open  ]],

Cell["\<\
Solve the inhomogeneous ODE through convolution with the Green's function:\
\>", "ExampleText",
 CellID->48591970],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"gf", "[", 
     RowBox[{"z", "-", "\[Zeta]"}], " ", "]"}], " ", 
    RowBox[{"Exp", "[", 
     RowBox[{"-", "\[Zeta]"}], " ", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"\[Zeta]", ",", " ", "0", ",", " ", "z"}], "}"}], ",", 
   RowBox[{"Assumptions", "->", 
    RowBox[{"z", ">", "0"}]}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->597358624],

Cell[BoxData[
 FractionBox[
  RowBox[{
   RowBox[{
    SuperscriptBox["\[ExponentialE]", 
     RowBox[{"-", "z"}]], " ", "k"}], "-", 
   RowBox[{"k", " ", 
    RowBox[{"Cos", "[", 
     RowBox[{"k", " ", "z"}], "]"}]}], "+", 
   RowBox[{"Sin", "[", 
    RowBox[{"k", " ", "z"}], "]"}]}], 
  RowBox[{"k", "+", 
   SuperscriptBox["k", "3"]}]]], "Output",
 ImageSize->{175, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->225673814]
}, Open  ]],

Cell[TextData[{
 "Compare with the direct result from ",
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->54497589],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"gf", "[", "z_", "]"}], "=", 
  RowBox[{
   RowBox[{
    RowBox[{"w", "[", "z", "]"}], "/.", 
    RowBox[{"First", "[", 
     RowBox[{"DSolve", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{
          RowBox[{
           RowBox[{
            RowBox[{"w", "''"}], "[", "z", "]"}], "+", 
           RowBox[{
            SuperscriptBox["k", "2"], " ", 
            RowBox[{"w", "[", "z", "]"}]}]}], "==", 
          RowBox[{"Exp", "[", 
           RowBox[{"-", "z"}], "]"}]}], ",", 
         RowBox[{
          RowBox[{"w", "[", "0", "]"}], "==", "0"}], ",", 
         RowBox[{
          RowBox[{
           RowBox[{"w", "'"}], "[", "0", "]"}], "==", "0"}]}], "}"}], ",", 
       RowBox[{"w", "[", "z", "]"}], ",", "z"}], "]"}], "]"}]}], "//", 
   "Simplify"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->517161024],

Cell[BoxData[
 FractionBox[
  RowBox[{
   RowBox[{
    SuperscriptBox["\[ExponentialE]", 
     RowBox[{"-", "z"}]], " ", "k"}], "-", 
   RowBox[{"k", " ", 
    RowBox[{"Cos", "[", 
     RowBox[{"k", " ", "z"}], "]"}]}], "+", 
   RowBox[{"Sin", "[", 
    RowBox[{"k", " ", "z"}], "]"}]}], 
  RowBox[{"k", "+", 
   SuperscriptBox["k", "3"]}]]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{175, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->77670538]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->453679385],

Cell["Model a uniform probability distribution:", "ExampleText",
 CellID->158401160],

Cell[BoxData[
 RowBox[{
  RowBox[{"p", "[", "x_", "]"}], ":=", 
  RowBox[{
   RowBox[{"HeavisideTheta", "[", "x", "]"}], 
   RowBox[{"HeavisideTheta", "[", 
    RowBox[{"1", "-", "x"}], "]"}]}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->316758502],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"p", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->458897865],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzzTSzJSM1NLMlMTlRwL0osyMhMLlZwyy8CCjEzMjAwPATiaRIMQPZ/IAuM
H/7/D6SRSXYgyQyUYgLxQIz/YMDABGRKAhkgmhlJjgGkFogt8MhJEDATXQ5Z
HwsOfWB7GDjQZMCiLAg+A4M4hgg7mh4TgirECNoiiaaCDY8JIN+xYJjABKGY
oQqgIcBM0GZB5DAacH+KEzSBaUjFmwQaHzOUyfcP639IjjHHKYPuOsLuZ4Hq
lKaxu5H1mBFUgR6KxPkDI12h6MCVg9ihIcCGFIZMDAhxczS+OAH1EmjykAyK
XGL6o7m1H48oF1j09q035889BSIGJxCBpMoJLP/27VeovDNY/vPnn8jhAnbL
TbiPuQiEHaV89DKIbPOAfqGF+6jm/u3bbgyE+0h1Pzh3YcS6AhpfHi3UccpD
fY1THsVWfzRTIem1ve0A1BQvmqgC5Ttnutg0WFUN5xAAKRos7nH4j70Vywyr
cQiaYA8icdaGDBSYg942YcYbdpj60VtHTGS6g1rmYNbgxIUMLI5gNTIrUbpG
c+So34ZmCKCUSazw/ALJMkSmfnuMvIatNCE3F7Mg0Y4k6EMuFcn1B6yvRB/9
sLhALwOJC72hmf5GWwUjpnRBrtUpyFPk1uz0N4mU9hNMPSUlDb5+MeESh4lo
/4/mxuGcZ4dz7A7F8oiUUoTS9srAl62j/a9Rvw2PEAClZJAibPOLxKVqRzSd
mDMbxOUp+poD7qfZ285kYHZ1nnP1yguy3AEV6adIH3pZMh0sijkrzcAIAAl0
Dpg=\
\>"], "Output",
 Evaluatable->False,
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{225, 150},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->306785250]
}, Open  ]],

Cell["\<\
Calculate the probability distribution for the sum of two uniformly \
distributed variables:\
\>", "ExampleText",
 CellID->233767774],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"DiracDelta", "[", 
     RowBox[{"z", "-", 
      RowBox[{"(", 
       RowBox[{"x", "+", "y"}], ")"}]}], "]"}], 
    RowBox[{"p", "[", "x", "]"}], 
    RowBox[{"p", "[", "y", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "\[Infinity]"}], ",", "\[Infinity]"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "\[Infinity]"}], ",", "\[Infinity]"}], "}"}], ",", 
   RowBox[{"Assumptions", "->", 
    RowBox[{"z", "\[Element]", "Reals"}]}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->71106397],

Cell[BoxData[
 RowBox[{
  RowBox[{"DiscreteDelta", "[", 
   RowBox[{"1", "-", "z"}], "]"}], "+", 
  RowBox[{"DiscreteDelta", "[", 
   RowBox[{
    RowBox[{"-", "1"}], "+", "z"}], "]"}], "+", 
  RowBox[{"z", " ", 
   RowBox[{"HeavisideTheta", "[", 
    RowBox[{
     RowBox[{"1", "-", "z"}], ",", "z"}], "]"}]}], "-", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "2"}], "+", "z"}], ")"}], " ", 
   RowBox[{"HeavisideTheta", "[", 
    RowBox[{
     RowBox[{"2", "-", "z"}], ",", 
     RowBox[{
      RowBox[{"-", "1"}], "+", "z"}]}], "]"}]}]}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{426, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->125769134]
}, Open  ]],

Cell["Plot the distributions for the sum:", "ExampleText",
 CellID->405412712],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{"%", ",", " ", 
   RowBox[{"{", 
    RowBox[{"z", ",", " ", 
     RowBox[{"-", "1"}], ",", "3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->130181674],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{225, 154},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->65940727]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->14364154],

Cell["\<\
Fundamental solution (Green's function) of the 1D wave equation:\
\>", "ExampleText",
 CellID->20125111],

Cell[BoxData[
 RowBox[{
  RowBox[{"\[ScriptCapitalG]", "[", 
   RowBox[{"x_", ",", "t_"}], "]"}], ":=", 
  RowBox[{"HeavisideTheta", "[", 
   RowBox[{"t", "-", 
    RowBox[{"Abs", "[", "x", "]"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->22304118],

Cell["Solution for a given source term:", "ExampleText",
 CellID->138427790],

Cell[BoxData[
 RowBox[{
  RowBox[{"uG", "[", 
   RowBox[{"f_", ",", 
    RowBox[{"{", 
     RowBox[{"x_", ",", "t_"}], "}"}]}], "]"}], ":=", 
  RowBox[{"Integrate", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"\[ScriptCapitalG]", "[", 
      RowBox[{
       RowBox[{"t", "-", "\[Tau]"}], ",", 
       RowBox[{"x", "-", "\[Xi]"}]}], "]"}], 
     RowBox[{"f", "[", 
      RowBox[{"\[Tau]", ",", "\[Xi]"}], "]"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"\[Tau]", ",", "0", ",", "t"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"\[Xi]", ",", 
      RowBox[{"-", "Infinity"}], ",", "Infinity"}], "}"}], ",", 
    RowBox[{"Assumptions", "->", 
     RowBox[{
      RowBox[{"x", "\[Element]", "Reals"}], "&&", 
      RowBox[{"t", ">", "0"}], "&&", 
      RowBox[{"\[Tau]", ">", "0"}]}]}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->377461750],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"solG", "=", 
  RowBox[{"uG", "[", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"HeavisideTheta", "[", 
       RowBox[{"1", "-", "#2"}], "]"}], " ", 
      RowBox[{"HeavisideTheta", "[", 
       RowBox[{"#2", "+", "1"}], "]"}]}], "&"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "t"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->432287539],

Cell[BoxData[
 RowBox[{
  FractionBox["1", "2"], " ", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{
     RowBox[{"-", "4"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "+", "t", "-", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{
       RowBox[{"-", "1"}], "+", "x"}], "]"}]}], "+", 
    RowBox[{"4", " ", "x", " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "+", "t", "-", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{
       RowBox[{"-", "1"}], "+", "x"}], "]"}]}], "+", 
    RowBox[{
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "-", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{
       RowBox[{"-", "1"}], "+", "t", "-", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "+", "x"}], "]"}]}], "+", 
    RowBox[{"2", " ", "x", " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "-", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{
       RowBox[{"-", "1"}], "+", "t", "-", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "+", "x"}], "]"}]}], "+", 
    RowBox[{
     SuperscriptBox["x", "2"], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "-", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{
       RowBox[{"-", "1"}], "+", "t", "-", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "+", "x"}], "]"}]}], "+", 
    RowBox[{"4", " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{
       RowBox[{"-", "1"}], "+", "t", "-", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{
       RowBox[{"-", "1"}], "+", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "+", "x"}], "]"}]}], "+", 
    RowBox[{"4", " ", "t", " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{
       RowBox[{"-", "1"}], "+", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{
       RowBox[{"-", "1"}], "-", "t", "+", "x"}], "]"}]}], "+", 
    RowBox[{"2", " ", "t", " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "-", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "+", "t", "-", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "+", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "-", "t", "+", "x"}], "]"}]}], "-", 
    RowBox[{
     SuperscriptBox["t", "2"], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "-", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "+", "t", "-", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "+", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "-", "t", "+", "x"}], "]"}]}], "+", 
    RowBox[{"2", " ", "t", " ", "x", " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "-", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "+", "t", "-", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "+", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "-", "t", "+", "x"}], "]"}]}], "+", 
    RowBox[{"3", " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "+", "t", "-", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{
       RowBox[{"-", "1"}], "+", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "+", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "-", "t", "+", "x"}], "]"}]}], "+", 
    RowBox[{"2", " ", "t", " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "+", "t", "-", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{
       RowBox[{"-", "1"}], "+", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "+", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "-", "t", "+", "x"}], "]"}]}], "-", 
    RowBox[{
     SuperscriptBox["t", "2"], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "+", "t", "-", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{
       RowBox[{"-", "1"}], "+", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "+", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "-", "t", "+", "x"}], "]"}]}], "-", 
    RowBox[{"2", " ", "x", " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "+", "t", "-", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{
       RowBox[{"-", "1"}], "+", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "+", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "-", "t", "+", "x"}], "]"}]}], "+", 
    RowBox[{"2", " ", "t", " ", "x", " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "+", "t", "-", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{
       RowBox[{"-", "1"}], "+", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "+", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "-", "t", "+", "x"}], "]"}]}], "-", 
    RowBox[{
     SuperscriptBox["x", "2"], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "+", "t", "-", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{
       RowBox[{"-", "1"}], "+", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "+", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "-", "t", "+", "x"}], "]"}]}]}], ")"}]}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{536, 338},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->51535107]
}, Open  ]],

Cell["Plot the solution:", "ExampleText",
 CellID->158417198],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot3D", "[", 
  RowBox[{"solG", ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{
      RowBox[{"-", "4"}], "Pi"}], ",", 
     RowBox[{"4", "Pi"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", 
     RowBox[{"4", "Pi"}]}], "}"}], ",", 
   RowBox[{"PlotPoints", "->", "30"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->75116359],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 146},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->174562102]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->221643494],

Cell[TextData[{
 "Fundamental solution of the Klein\[Dash]Gordon ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[Delta]", 
     RowBox[{"t", " ", "t"}]], "-", "\[CapitalDelta]", "+", 
    SuperscriptBox["m", "2"]}], TraditionalForm]], "InlineMath"],
 " operator:"
}], "ExampleText",
 CellID->150951800],

Cell[BoxData[
 RowBox[{
  RowBox[{"\[ScriptCapitalD]", "[", 
   RowBox[{"t_", ",", "x_List"}], "]"}], ":=", 
  RowBox[{
   RowBox[{
    FractionBox["1", 
     RowBox[{"2", "\[Pi]"}]], 
    RowBox[{"HeavisideTheta", "[", "t", "]"}], 
    RowBox[{"DiracDelta", "[", 
     RowBox[{
      SuperscriptBox["t", "2"], "-", 
      RowBox[{"x", ".", "x"}]}], "]"}]}], "-", 
   RowBox[{
    FractionBox["m", 
     RowBox[{"4", "\[Pi]"}]], 
    RowBox[{"HeavisideTheta", "[", 
     RowBox[{"t", "-", 
      SqrtBox[
       RowBox[{"x", ".", "x"}]]}], "]"}], 
    FractionBox[
     RowBox[{"BesselJ", "[", 
      RowBox[{"1", ",", 
       RowBox[{"m", 
        SqrtBox[
         RowBox[{
          SuperscriptBox["t", "2"], "-", 
          RowBox[{"x", ".", "x"}]}]]}]}], "]"}], 
     SqrtBox[
      RowBox[{
       SuperscriptBox["t", "2"], "-", 
       RowBox[{"x", ".", "x"}]}]]]}]}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->85726970],

Cell["\<\
Visualize the fundamental solution (it is nonvanishing only in the forward \
light cone):\
\>", "ExampleText",
 CellID->233537389],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"-", 
     RowBox[{"\[ScriptCapitalD]", "[", 
      RowBox[{"t", ",", 
       RowBox[{"{", 
        RowBox[{"0", ",", "0", ",", "z"}], "}"}]}], "]"}]}], "/.", 
    RowBox[{"m", "->", "1"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"z", ",", 
     RowBox[{"-", "3"}], ",", "3.02"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", 
     RowBox[{"-", "3"}], ",", "3.01"}], "}"}], ",", 
   RowBox[{"PlotStyle", "->", 
    RowBox[{"{", 
     RowBox[{"Opacity", "[", "0.3", "]"}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->489441445],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 136},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->632024652]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->513749978],

Cell["\<\
A cusp\[Hyphen]containing peakon solution of the Camassa\[Dash]Holm equation:\
\
\>", "ExampleText",
 CellID->713047985],

Cell[BoxData[
 RowBox[{
  RowBox[{"chEq", "[", 
   RowBox[{"x_", ",", "t_"}], "]"}], ":=", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      SubscriptBox["\[PartialD]", "t"], "#"}], "-", 
     RowBox[{
      SubscriptBox["\[PartialD]", 
       RowBox[{"x", ",", "x", ",", "t"}]], "#"}], "+", 
     RowBox[{"3", " ", "#", " ", 
      RowBox[{
       SubscriptBox["\[PartialD]", "x"], "#"}]}], "-", 
     RowBox[{"2", " ", 
      RowBox[{
       SubscriptBox["\[PartialD]", "x"], "#"}], " ", 
      RowBox[{
       SubscriptBox["\[PartialD]", 
        RowBox[{"x", ",", "x"}]], "#"}]}], "-", 
     RowBox[{"#", " ", 
      RowBox[{
       SubscriptBox["\[PartialD]", 
        RowBox[{"x", ",", "x", ",", "x"}]], "#"}]}]}], ")"}], 
   "&"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->101984099],

Cell[BoxData[
 RowBox[{
  RowBox[{"u", "[", 
   RowBox[{"x_", ",", " ", "t_"}], "]"}], " ", ":=", " ", 
  RowBox[{"Exp", "[", 
   RowBox[{
    RowBox[{"-", 
     RowBox[{"(", 
      RowBox[{"x", "-", "t"}], ")"}]}], 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"HeavisideTheta", "[", 
       RowBox[{"x", "-", "t"}], "]"}], "-", 
      RowBox[{"HeavisideTheta", "[", 
       RowBox[{"t", "-", "x"}], "]"}]}], ")"}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->282973152],

Cell["Check the solution:", "ExampleText",
 CellID->394152426],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"chEq", "[", 
    RowBox[{"x", ",", "t"}], "]"}], "@", " ", 
   RowBox[{"u", "[", 
    RowBox[{"x", ",", " ", "t"}], "]"}]}], "//", 
  RowBox[{
   RowBox[{"FullSimplify", "[", 
    RowBox[{"#", ",", 
     RowBox[{
      RowBox[{"x", ">", "0"}], "&&", 
      RowBox[{"t", ">", "x"}]}]}], "]"}], "&"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->135872379],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->440213334]
}, Open  ]],

Cell["Plot the solution:", "ExampleText",
 CellID->36716238],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"u", "[", 
    RowBox[{"x", ",", "t"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "3"}], ",", "3"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->49751253],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 140},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->452373150]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->453076180],

Cell["\<\
Differentiate and integrate a piecewise defined function in a lossless \
manner:\
\>", "ExampleText",
 CellID->326857912],

Cell[BoxData[
 RowBox[{
  RowBox[{"pwF", "[", "x_", "]"}], ":=", 
  RowBox[{
   RowBox[{"2", 
    RowBox[{"HeavisideTheta", "[", 
     RowBox[{"x", "-", "1"}], "]"}]}], "-", 
   RowBox[{"2", 
    RowBox[{"HeavisideTheta", "[", 
     RowBox[{
      RowBox[{"-", "x"}], "-", "1"}], "]"}]}]}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->237338708],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"pwF", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "3"}], ",", "3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->573692869],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzzTSzJSM1NLMlMTlRwL0osyMhMLlZwyy8CCjEzMjAwKjEwMOyQYACy/wNZ
YAwCykAMYaKzOoAkC1AZK4jHDFP/n6EPLNuJJMqAUxUbWPT2rTfnzz0FIoYo
EMH88+cfqCoZqB0saLrR+cimTZ50LD9vMxAxRIII5kD/xdu33QBJqaHpYoaa
3gvWnQQWam87gEc1pp3ousBhw4JQhcvn6PpUgAQTA8wPhFTrDGJVIF/0DyL3
DG1Vo6E5Gk70VwVSNJjcQy1VKlC/EVcmq6Kpwl7roOuS/U9KnQXSzTR/3hn0
2oblP3rdhEsVaWbiVg3yLf4aCKRKdwBlUfP4QLtmNBRGtj9B+QQkqP0fklO1
QKIsQOUMMKCNpk8TqkMLSQeIzw42jxOLauQWtBbUNchqtND04OODzED3CQtE
nBkqDQRMWEouZah70VvAKlBxsAagyyB+Ry+J0d2EHiro6iFuYoemAD00WQ0c
ocCEJbzRTdaEqoOENwcB1bhswscn5HdWrDGtAfY2gk9eXGtimII9tmHpFtZq
goQaIt3iCjVk9YhUO1C5DznNgmh5IGaE+w9ZPSE+O4nqqaYf2udlB/d5EXw2
EvmU6sfOX73qErTYINV/9A5PRKoezK4cbPqx80FRP5RjHb0GIFQjaKH5Gl0e
XT9xtkbhsiU/bzPxtkwkYEriAKqCtWIGi3uGtipELTw43DNYVY2mutFUR1hV
53/sYzzYVXehqULv+xCvC9T0hvTZbt96k5y0hlQ7nj//FB66bGB0df7HPvaD
XTX2eBgcqhCpenC4Z7CqGg2n0dAcDSdqqcI1F467tIWpYiBaF3klezeaKtic
N6Sm+vz5J1C1ve1MhigQwRwbvRIy+Q7tg2DTjczvg/uemDlvXLUMRBbExL3a
gIERABXs+s4=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{290, 184},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->364979244]
}, Open  ]],

Cell["\<\
Differentiating and integrating recovers the original function:\
\>", "ExampleText",
 CellID->534675154],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"D", "[", 
  RowBox[{
   RowBox[{"pwF", "[", "x", "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->105686975],

Cell[BoxData[
 RowBox[{
  RowBox[{"2", " ", 
   RowBox[{"DiracDelta", "[", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "x"}], "]"}]}], "+", 
  RowBox[{"2", " ", 
   RowBox[{"DiracDelta", "[", 
    RowBox[{"1", "+", "x"}], "]"}]}]}]], "Output",
 ImageSize->{266, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->16012691]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{"%", ",", "x"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->35542566],

Cell[BoxData[
 RowBox[{
  RowBox[{"2", " ", 
   RowBox[{"HeavisideTheta", "[", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "x"}], "]"}]}], "+", 
  RowBox[{"2", " ", 
   RowBox[{"HeavisideTheta", "[", 
    RowBox[{"1", "+", "x"}], "]"}]}]}]], "Output",
 ImageSize->{322, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->378209724]
}, Open  ]],

Cell[TextData[{
 "Using ",
 Cell[BoxData[
  ButtonBox["Piecewise",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Piecewise"]], "InlineFormula"],
 " does not recover the original function:"
}], "ExampleText",
 CellID->21430366],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"pwF2", "[", "x_", "]"}], "=", 
  RowBox[{"PiecewiseExpand", "[", 
   RowBox[{
    RowBox[{"pwF", "[", "x", "]"}], "/.", 
    RowBox[{"HeavisideTheta", "->", "UnitStep"}]}], "]"}]}]], "Input",
 CellLabel->"In[5]:=",
 CellID->712114601],

Cell[GraphicsData["CompressedBitmap", "\<\
eJylVNtuwjAMdQldAbXjUkBIvPR9nzG2iYd9RIWQ4AGBoP8fnNRpg+NtRavU
nB5fjhwn7ndZHfansjruyuLrWl4Ox92t+Dxf0aQiANjiu14Bfmv8su9Wa0R/
VbgqdL0gxog5Yg+xRxiT3flHTRaYNG0f64W44TYGYG6w30bVKn7eSOKBesL4
kmobEHZTGTC+8LKd2t8qQ8bzYI/JU1XxqKnQ/+c7Jp/HW+duK4o2FaRi1Lu1
uio/GubHbH6xynr/sWa6vqGp0EFjd371Y3bYsSw4Xd7/jHFZ3U0Pt4+pxiFh
QnH9Tqpjxl89FYOzTioTxtNgz27OeGSrZ4xT5nVz5c6BT47cuVw8B35z69i5
frypdlbovwbRHS27lQo=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{73, 27},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->38686201]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"D", "[", 
  RowBox[{
   RowBox[{"pwF2", "[", "x", "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->301431977],

Cell[GraphicsData["CompressedBitmap", "\<\
eJy9V7tOwzAUdeO47weUBehSJpj4CSSY+AYaVUjtgEClc0dgZ0H8bLBdO01O
ruOkKY2UJvd17r0nftSP0Xrx/BKtl/No+rCK3hbL+fv0/nUlVbzBWOOUMTY5
Z/I9lm/6VtdY3ttXfOPyl0u3pnwK+WRCq82l3pU1MFb1DOUzNFHcyNauFSYW
Ze5F54A6SMnWHgCqregj6UZD6q5j1gZZeevCYntZDIc2FatLb4GX8W2BL08h
7rzCGtXwguxjKnuBHBDdfR2Hv4KqSPvhOSyo4KQMj94+/4dJ4ajfpd+XOTv3
XLmGe3MkKs3SZIyGOz3LWsp9X9IrsHyUYomuhUYXpJZiDWvIspXXFnAXpn1x
hQxBZ7vCldXKQUKDvozFFU1lo5BHDl6zmSzTQSry0913Sh547Lcgn+W+J87j
HzKv3Sf7psprsCtZLdO/ZLT66WuL7VJphokGfQceixsfLd8g4zi/89ivjoiq
dlMlzuLsiOiAX898C9x9McvA+GE89+CHJfERp0n0k6zT5IjKz/ZZjOOzbdBx
l/JVhzOvZXBazgzIgy9DWBqpCgNRqk6LJVh+hfZVJ0wc/r8QBrcLeXp74Nu4
hNeaPXfiw37rNuBW/cLCE19vZHchDtfyql1T87g4IzVjq414jHAzZE9beWae
4uw6clGThxHg4Tmh6rpGnjP0TpjvZVOzdtxXffETsjZ7gtzE7hU+LQ+N301J
/0vjtz3WKtX2VM0af8yQ/9s=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{273, 27},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->424893095]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{"%", ",", "x"}], "]"}]], "Input",
 CellLabel->"In[7]:=",
 CellID->706615872],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->24952956]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->32544],

Cell[TextData[{
 "Expand ",
 Cell[BoxData[
  ButtonBox["HeavisideTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HeavisideTheta"]], "InlineFormula"],
 " into ",
 Cell[BoxData[
  ButtonBox["HeavisideTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HeavisideTheta"]], "InlineFormula"],
 " with simpler arguments:"
}], "ExampleText",
 CellID->875727382],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"HeavisideTheta", "[", 
   RowBox[{
    RowBox[{"x", "^", "5"}], "-", "x", "+", "4"}], "]"}], "//", 
  "FunctionExpand"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->827402583],

Cell[BoxData[
 RowBox[{"HeavisideTheta", "[", 
  RowBox[{"x", "-", 
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"4", "-", "#1", "+", 
       SuperscriptBox["#1", "5"]}], "&"}], ",", "1"}], "]"}]}], 
  "]"}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{273, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->405353561]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->446141051],

Cell[TextData[{
 "Simplify expressions containing ",
 Cell[BoxData[
  ButtonBox["HeavisideTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HeavisideTheta"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->218614874],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Simplify", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", " ", 
      RowBox[{"HeavisideTheta", "[", "x", "]"}]}], ",", 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{
       RowBox[{"2", "x"}], "+", "2"}], "]"}]}], "}"}], ",", 
   RowBox[{"x", ">", "3"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->147094569],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"x", ",", "1"}], "}"}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{44, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->181494354]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->323123013],

Cell["Use in integrals:", "ExampleText",
 CellID->991934362],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"HeavisideTheta", "[", 
    RowBox[{"Sin", "[", "x", "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->349467808],

Cell[BoxData[
 RowBox[{"2", " ", "\[Pi]"}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{22, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->77440148]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"HeavisideTheta", "[", 
    RowBox[{"1", "-", 
     SuperscriptBox["x", "2"], "-", "y"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "2"}], ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->25845485],

Cell[BoxData[
 RowBox[{"4", " ", 
  SqrtBox["3"]}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{38, 17},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->39808079]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->146008241],

Cell["Use in Fourier transforms:", "ExampleText",
 CellID->314406036],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{" ", 
  RowBox[{"FourierTransform", "[", 
   RowBox[{
    RowBox[{"HeavisideTheta", "[", "y", "]"}], ",", "y", ",", "x"}], "]"}], 
  " "}]], "Input",
 CellLabel->"In[1]:=",
 CellID->227003087],

Cell[BoxData[
 RowBox[{
  FractionBox["\[ImaginaryI]", 
   RowBox[{
    SqrtBox[
     RowBox[{"2", " ", "\[Pi]"}]], " ", "x"}]], "+", 
  RowBox[{
   SqrtBox[
    FractionBox["\[Pi]", "2"]], " ", 
   RowBox[{"DiracDelta", "[", "x", "]"}]}]}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{192, 40},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->23624205]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1006516673],

Cell["Use in Laplace transforms:", "ExampleText",
 CellID->930209115],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LaplaceTransform", "[", 
  RowBox[{
   RowBox[{"HeavisideTheta", "[", "x", "]"}], ",", " ", "x", ",", " ", "s"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->646218562],

Cell[BoxData[
 FractionBox["1", "s"]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{16, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->133742640]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(10)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->32050],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["HeavisideTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HeavisideTheta"]], "InlineFormula"],
 " stays unevaluated for vanishing argument:"
}], "ExampleText",
 CellID->255283717],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"HeavisideTheta", "[", "0", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->12162104],

Cell[BoxData[
 RowBox[{"HeavisideTheta", "[", "0", "]"}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{123, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->190755669]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->14924626],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["PiecewiseExpand",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PiecewiseExpand"]], "InlineFormula"],
 " does not operate on ",
 Cell[BoxData[
  ButtonBox["HeavisideTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HeavisideTheta"]], "InlineFormula"],
 " because it is a distribution and not a piecewise\[Hyphen]defined \
function:"
}], "ExampleText",
 CellID->370442904],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PiecewiseExpand", "[", 
  RowBox[{"HeavisideTheta", "[", "x", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->46088639],

Cell[BoxData[
 RowBox[{"HeavisideTheta", "[", "x", "]"}]], "Output",
 ImageSize->{123, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->615391083]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->495916464],

Cell["\<\
The precision of the output does not track the precision of the input:\
\>", "ExampleText",
 CellID->188188594],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"HeavisideTheta", "[", "1.6", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->812151],

Cell[BoxData["1"], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1528832]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"HeavisideTheta", "[", "1.60000000000000000000000", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->351307062],

Cell[BoxData["1"], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->300225805]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->78837792],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["HeavisideTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HeavisideTheta"]], "InlineFormula"],
 " can stay unevaluated for numeric arguments:"
}], "ExampleText",
 CellID->673641459],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"HeavisideTheta", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{"1", "+", "E"}], ")"}], "^", "2"}], "-", "1", "-", 
   RowBox[{"2", " ", "E"}], "-", 
   RowBox[{"E", "^", "2"}], "+", 
   RowBox[{"Exp", "[", 
    RowBox[{"-", 
     RowBox[{
      RowBox[{"Exp", "[", "E", "]"}], "^", "2"}]}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->236660448],

Cell[BoxData[
 RowBox[{"HeavisideTheta", "[", 
  RowBox[{
   RowBox[{"-", "1"}], "-", 
   RowBox[{"2", " ", "\[ExponentialE]"}], "-", 
   SuperscriptBox["\[ExponentialE]", "2"], "+", 
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"-", 
     SuperscriptBox["\[ExponentialE]", 
      RowBox[{"2", " ", "\[ExponentialE]"}]]}]], "+", 
   SuperscriptBox[
    RowBox[{"(", 
     RowBox[{"1", "+", "\[ExponentialE]"}], ")"}], "2"]}], "]"}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{287, 24},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->235289381]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Simplify", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->123622996],

Cell[BoxData["1"], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->19711796]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->110679831],

Cell[TextData[{
 "Machine\[Hyphen]precision numericalization",
 " of ",
 Cell[BoxData[
  ButtonBox["HeavisideTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HeavisideTheta"]], "InlineFormula"],
 " can give wrong results:"
}], "ExampleText",
 CellID->216733786],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"HeavisideTheta", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{"1", "+", "E"}], ")"}], "^", "2"}], "-", "1", "-", 
   RowBox[{"2", " ", "E"}], "-", 
   RowBox[{"E", "^", "2"}], "+", 
   RowBox[{"Exp", "[", 
    RowBox[{"-", 
     RowBox[{
      RowBox[{"Exp", "[", "E", "]"}], "^", "2"}]}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->510635538],

Cell[BoxData[
 RowBox[{"HeavisideTheta", "[", 
  RowBox[{
   RowBox[{"-", "1"}], "-", 
   RowBox[{"2", " ", "\[ExponentialE]"}], "-", 
   SuperscriptBox["\[ExponentialE]", "2"], "+", 
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"-", 
     SuperscriptBox["\[ExponentialE]", 
      RowBox[{"2", " ", "\[ExponentialE]"}]]}]], "+", 
   SuperscriptBox[
    RowBox[{"(", 
     RowBox[{"1", "+", "\[ExponentialE]"}], ")"}], "2"]}], "]"}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{287, 24},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->22536744]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->238486270],

Cell[BoxData["0"], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->244609980]
}, Open  ]],

Cell["\<\
Use arbitrary\[Hyphen]precision arithmetic to obtain the correct result:\
\>", "ExampleText",
 CellID->140926449],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{"%%", ",", "1000"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->497318653],

Cell[BoxData[
 RowBox[{
  RowBox[{"N", "::", "\<\"meprec\"\>"}], ":", 
  " ", "\<\"Internal precision limit $MaxExtraPrecision = \\!\\(50.`\\) \
reached while evaluating \\!\\(HeavisideTheta[\\(\\(\\(\\(-1\\)\\) - \
\\(\\(2\\\\ \[ExponentialE]\\)\\) - \[ExponentialE]\\^2 + \
\[ExponentialE]\\^\\(-\[ExponentialE]\\^\\(2\\\\ \[ExponentialE]\\)\\) + \
\\((1 + \[ExponentialE])\\)\\^2\\)\\)]\\). \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/General/meprec\\\", ButtonNote -> \
\\\"N::meprec\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->477412756],

Cell[BoxData["1"], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->663820968]
}, Open  ]],

Cell[TextData[{
 "A larger setting for ",
 Cell[BoxData[
  ButtonBox["$MaxExtraPrecision",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/$MaxExtraPrecision"]], "InlineFormula"],
 " will not avoid the ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["N",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/N"], "::", "meprec"}]], "InlineFormula"],
 " message because the result is exact:"
}], "ExampleText",
 CellID->17465],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Block", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"$MaxExtraPrecision", "=", "1000"}], "}"}], ",", 
   RowBox[{"N", "[", 
    RowBox[{"%%%", ",", "20"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->809015494],

Cell[BoxData[
 RowBox[{
  RowBox[{"N", "::", "\<\"meprec\"\>"}], ":", 
  " ", "\<\"Internal precision limit $MaxExtraPrecision = \\!\\(1000.`\\) \
reached while evaluating \\!\\(HeavisideTheta[\\(\\(\\(\\(-1\\)\\) - \
\\(\\(2\\\\ \[ExponentialE]\\)\\) - \[ExponentialE]\\^2 + \
\[ExponentialE]\\^\\(-\[ExponentialE]\\^\\(2\\\\ \[ExponentialE]\\)\\) + \
\\((1 + \[ExponentialE])\\)\\^2\\)\\)]\\). \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/General/meprec\\\", ButtonNote -> \
\\\"N::meprec\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->93301592],

Cell[BoxData["1"], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->125706083]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->151128829],

Cell[TextData[{
 "The functions ",
 Cell[BoxData[
  ButtonBox["UnitStep",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/UnitStep"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["HeavisideTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HeavisideTheta"]], "InlineFormula"],
 " are not mathematically equivalent:"
}], "ExampleText",
 CellID->229130295],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"HeavisideTheta", "[", "x", "]"}], ",", 
   RowBox[{"UnitStep", "[", "x", "]"}]}], "}"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->614562461],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"HeavisideTheta", "[", "x", "]"}], ",", 
   RowBox[{"UnitStep", "[", "x", "]"}]}], "}"}]], "Output",
 ImageSize->{226, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->918897257]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"D", "[", 
    RowBox[{"%", ",", "x"}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->99294959],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"HeavisideTheta", "[", "x", "]"}], ",", "0"}], "}"}]], "Output",
 ImageSize->{156, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->89242928]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->52897639],

Cell["\<\
Products of distributions with coincident singular support cannot be defined \
(no Colombeau algebra interpretation):\
\>", "ExampleText",
 CellID->86757341],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"HeavisideTheta", "[", "x", "]"}], " ", 
    RowBox[{"DiracDelta", "[", "x", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->155725090],

Cell[BoxData[
 RowBox[{
  SubsuperscriptBox["\[Integral]", 
   RowBox[{"-", "1"}], "1"], 
  RowBox[{
   RowBox[{
    RowBox[{"DiracDelta", "[", "x", "]"}], " ", 
    RowBox[{"HeavisideTheta", "[", "x", "]"}]}], 
   RowBox[{"\[DifferentialD]", "x"}]}]}]], "Output",
 ImageSize->{254, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->752134003]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->27336247],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["HeavisideTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HeavisideTheta"]], "InlineFormula"],
 " cannot be uniquely defined with complex arguments (no Sato hyperfunction \
interpretation):"
}], "ExampleText",
 CellID->356648096],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"HeavisideTheta", "[", 
  RowBox[{"1", "+", 
   RowBox[{"2", "I"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->385445084],

Cell[BoxData[
 RowBox[{"HeavisideTheta", "[", 
  RowBox[{"1", "+", 
   RowBox[{"2", " ", "\[ImaginaryI]"}]}], "]"}]], "Output",
 ImageSize->{154, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->110414120]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->254059430],

Cell["\<\
Numerical routines can have problems with discontinuous functions:\
\>", "ExampleText",
 CellID->25094924],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindRoot", "[", 
  RowBox[{
   RowBox[{"HeavisideTheta", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "1", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->266815590],

Cell[BoxData[
 RowBox[{
  RowBox[{"FindRoot", "::", "\<\"cvmit\"\>"}], ":", 
  " ", "\<\"Failed to converge to the requested accuracy or precision within \
\\!\\(100\\) iterations. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/FindRoot/cvmit\\\", ButtonNote -> \
\\\"FindRoot::cvmit\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->490300623],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"x", "\[Rule]", "2.`"}], "}"}]], "Output",
 ImageSize->{55, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->312914770]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"HeavisideTheta", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->294724839],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"slwcon\"\>"}], ":", 
  " ", "\<\"Numerical integration converging too slowly; suspect one of the \
following: singularity, value of the integration is 0, highly oscillatory \
integrand, or WorkingPrecision too small. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/slwcon\\\", ButtonNote -> \
\\\"NIntegrate::slwcon\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->148451150],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"ncvb\"\>"}], ":", 
  " ", "\<\"NIntegrate failed to converge to prescribed accuracy after \
\\!\\(9\\) recursive bisections in \\!\\(x\\) near \\!\\({x}\\) = \
\\!\\({0.001906500078402859`}\\). NIntegrate obtained \
\\!\\(1.9997626654955845`\\) and \\!\\(0.0003805636710848472`\\) for the \
integral and error estimates. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/ncvb\\\", ButtonNote -> \
\\\"NIntegrate::ncvb\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->166964921],

Cell[BoxData["1.9997626654955845`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->41941159]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->100562957],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Limit",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Limit"]], "InlineFormula"],
 " does not give ",
 Cell[BoxData[
  ButtonBox["HeavisideTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HeavisideTheta"]], "InlineFormula"],
 " as a limit of smooth functions:"
}], "ExampleText",
 CellID->542079127],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Assuming", "[", 
  RowBox[{
   RowBox[{"x", "\[Element]", "Reals"}], ",", 
   RowBox[{
    RowBox[{
     FractionBox[
      RowBox[{"1", " "}], "\[Pi]"], 
     RowBox[{"Limit", "[", 
      RowBox[{
       RowBox[{"ArcTan", "[", 
        FractionBox["x", "\[CurlyEpsilon]"], "]"}], ",", 
       RowBox[{"\[CurlyEpsilon]", "\[Rule]", 
        RowBox[{"+", "0"}]}]}], "]"}]}], "+", 
    FractionBox["1", "2"]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->530109224],

Cell[BoxData[
 RowBox[{
  FractionBox["1", "2"], "+", 
  FractionBox[
   RowBox[{"Abs", "[", "x", "]"}], 
   RowBox[{"2", " ", "x"}]]}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{74, 31},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->16349085]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"FullSimplify", "[", 
    RowBox[{"%", ",", 
     RowBox[{
      RowBox[{"x", "\[Element]", "Reals"}], "&&", 
      RowBox[{"x", "<", "0"}]}]}], "]"}], ",", 
   RowBox[{"FullSimplify", "[", 
    RowBox[{"%", ",", 
     RowBox[{
      RowBox[{"x", "\[Element]", "Reals"}], "&&", 
      RowBox[{"x", ">", "0"}]}]}], "]"}]}], "}"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->202268354],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0", ",", "1"}], "}"}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{44, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->395979576]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Neat Examples",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->4301],

Cell["Form repeated convolution integrals starting with a product:", \
"ExampleText",
 CellID->168473678],

Cell[BoxData[
 RowBox[{
  RowBox[{"conv", "[", 
   RowBox[{"f_", ",", "g_", ",", "x_"}], "]"}], ":=", 
  RowBox[{"Block", "[", 
   RowBox[{
    RowBox[{"{", "y", "}"}], ",", 
    RowBox[{"Integrate", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"(", 
        RowBox[{"f", "/.", 
         RowBox[{"x", "->", 
          RowBox[{"x", "-", "y"}]}]}], ")"}], 
       RowBox[{"(", 
        RowBox[{"g", "/.", 
         RowBox[{"x", "->", "y"}]}], ")"}]}], ",", 
      RowBox[{"{", 
       RowBox[{"y", ",", 
        RowBox[{"-", "\[Infinity]"}], ",", "\[Infinity]"}], "}"}], ",", 
      RowBox[{"Assumptions", "->", 
       RowBox[{"x", "\[Element]", "Reals"}]}]}], "]"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->288154854],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NestList", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"conv", "[", 
      RowBox[{"#", ",", 
       RowBox[{
        RowBox[{"HeavisideTheta", "[", "x", "]"}], 
        RowBox[{"HeavisideTheta", "[", 
         RowBox[{"1", "-", "x"}], "]"}]}], ",", "x"}], "]"}], "&"}], ",", 
    RowBox[{
     RowBox[{"HeavisideTheta", "[", "x", "]"}], 
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "-", "x"}], "]"}]}], ",", "2"}], "]"}], "//", 
  "Short"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->130872667],

Cell[BoxData[
 TagBox[
  RowBox[{"{", 
   RowBox[{
    RowBox[{
     RowBox[{"HeavisideTheta", "[", 
      RowBox[{"1", "-", "x"}], "]"}], " ", 
     RowBox[{"HeavisideTheta", "[", "x", "]"}]}], ",", 
    RowBox[{"\[LeftSkeleton]", "1", "\[RightSkeleton]"}], ",", 
    RowBox[{
     FractionBox["1", "2"], " ", 
     RowBox[{"(", 
      RowBox[{"\[LeftSkeleton]", "1", "\[RightSkeleton]"}], ")"}]}]}], "}"}],
  Short]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{400, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]//Short=",
 CellID->630594162]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", "%", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "3"}], ",", "3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->316066683],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{225, 154},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->457859981]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->20770],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["DiracDelta",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/DiracDelta"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["UnitStep",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/UnitStep"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["PrincipalValue",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/PrincipalValue"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->32014]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->83728325],

Cell[TextData[ButtonBox["Generalized Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/GeneralizedFunctions"]], "MoreAbout",
 CellID->38907142],

Cell[TextData[ButtonBox["Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/MathematicalFunctions"]], "MoreAbout",
 CellID->100883443],

Cell[TextData[ButtonBox["New in 6.0: Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicalFunctions"]], "MoreAbout",
 CellID->206110422]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"HeavisideTheta - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 47, 27.4752904}", "context" -> "System`", 
    "keywords" -> {"generalized functions", "Heaviside theta", "unit step"}, 
    "index" -> True, "label" -> "Built-in Mathematica Symbol", "language" -> 
    "en", "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "HeavisideTheta[x] represents the Heaviside theta function \\[Theta](x), \
equal to 0 for x < 0 and 1 for x > 0. HeavisideTheta[x_1, x_2, ...] \
represents the multidimensional Heaviside theta function which is 1 only if \
none of the x_i are not positive. ", "synonyms" -> {"heaviside theta"}, 
    "title" -> "HeavisideTheta", "type" -> "Symbol", "uri" -> 
    "ref/HeavisideTheta"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[7503, 235, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->122089080]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 115673, 3476}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 1749, 41, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2372, 68, 1579, 37, 70, "ObjectNameGrid"],
Cell[3954, 107, 1576, 49, 70, "Usage",
 CellID->26346]
}, Open  ]],
Cell[CellGroupData[{
Cell[5567, 161, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6045, 178, 339, 12, 70, "Notes",
 CellID->20509],
Cell[6387, 192, 256, 8, 70, "Notes",
 CellID->30080],
Cell[6646, 202, 321, 12, 70, "Notes",
 CellID->7981],
Cell[6970, 216, 496, 14, 70, "Notes",
 CellID->23207]
}, Closed]],
Cell[CellGroupData[{
Cell[7503, 235, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->122089080],
Cell[CellGroupData[{
Cell[7888, 249, 148, 5, 70, "ExampleSection",
 CellID->465933886],
Cell[CellGroupData[{
Cell[8061, 258, 245, 8, 28, "Input",
 CellID->255218453],
Cell[8309, 268, 797, 17, 117, "Output",
 Evaluatable->False,
 CellID->512734480]
}, Open  ]],
Cell[9121, 288, 125, 3, 70, "ExampleDelimiter",
 CellID->229644424],
Cell[9249, 293, 211, 8, 70, "ExampleText",
 CellID->198163040],
Cell[CellGroupData[{
Cell[9485, 305, 160, 5, 70, "Input",
 CellID->4240533],
Cell[9648, 312, 242, 8, 36, "Output",
 CellID->914881128]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[9939, 326, 216, 7, 70, "ExampleSection",
 CellID->18618],
Cell[10158, 335, 221, 8, 70, "ExampleText",
 CellID->552355627],
Cell[CellGroupData[{
Cell[10404, 347, 278, 9, 70, "Input",
 CellID->389936162],
Cell[10685, 358, 1539, 48, 88, "Output",
 CellID->58312727]
}, Open  ]],
Cell[12239, 409, 125, 3, 70, "ExampleDelimiter",
 CellID->136559208],
Cell[12367, 414, 221, 7, 70, "ExampleText",
 CellID->87642531],
Cell[CellGroupData[{
Cell[12613, 425, 187, 6, 70, "Input",
 CellID->223027789],
Cell[12803, 433, 254, 8, 36, "Output",
 CellID->8957073]
}, Open  ]],
Cell[13072, 444, 125, 3, 70, "ExampleDelimiter",
 CellID->647755012],
Cell[13200, 449, 85, 1, 70, "ExampleText",
 CellID->31876294],
Cell[CellGroupData[{
Cell[13310, 454, 324, 10, 70, "Input",
 CellID->185869160],
Cell[13637, 466, 597, 21, 36, "Output",
 CellID->542186415]
}, Open  ]],
Cell[CellGroupData[{
Cell[14271, 492, 395, 13, 70, "Input",
 CellID->311609859],
Cell[14669, 507, 305, 11, 51, "Output",
 CellID->80028367]
}, Open  ]],
Cell[CellGroupData[{
Cell[15011, 523, 307, 10, 70, "Input",
 CellID->17399961],
Cell[15321, 535, 206, 7, 36, "Output",
 CellID->484103998]
}, Open  ]],
Cell[15542, 545, 65, 1, 70, "ExampleText",
 CellID->474749005],
Cell[CellGroupData[{
Cell[15632, 550, 443, 14, 70, "Input",
 CellID->183728440],
Cell[16078, 566, 223, 7, 36, "Output",
 CellID->123425078]
}, Open  ]],
Cell[16316, 576, 122, 3, 70, "ExampleDelimiter",
 CellID->182430],
Cell[16441, 581, 252, 8, 70, "ExampleText",
 CellID->243133372],
Cell[CellGroupData[{
Cell[16718, 593, 451, 14, 70, "Input",
 CellID->480136610],
Cell[17172, 609, 252, 8, 36, "Output",
 CellID->725572585]
}, Open  ]],
Cell[17439, 620, 125, 3, 70, "ExampleDelimiter",
 CellID->611122967],
Cell[17567, 625, 97, 1, 70, "ExampleText",
 CellID->259982338],
Cell[CellGroupData[{
Cell[17689, 630, 170, 5, 70, "Input",
 CellID->147978867],
Cell[17862, 637, 222, 7, 36, "Output",
 CellID->483347866]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[18133, 650, 239, 7, 70, "ExampleSection",
 CellID->31720],
Cell[18375, 659, 208, 8, 70, "ExampleText",
 CellID->461353717],
Cell[CellGroupData[{
Cell[18608, 671, 251, 6, 70, "Input",
 CellID->105408681],
Cell[18862, 679, 206, 7, 36, "Output",
 CellID->133075188]
}, Open  ]],
Cell[19083, 689, 125, 3, 70, "ExampleDelimiter",
 CellID->141883382],
Cell[CellGroupData[{
Cell[19233, 696, 490, 16, 70, "Input",
 CellID->5135933],
Cell[19726, 714, 330, 10, 36, "Output",
 CellID->20206465]
}, Open  ]],
Cell[20071, 727, 125, 3, 70, "ExampleDelimiter",
 CellID->429962081],
Cell[20199, 732, 224, 8, 70, "ExampleText",
 CellID->5507245],
Cell[CellGroupData[{
Cell[20448, 744, 231, 8, 70, "Input",
 CellID->240921601],
Cell[20682, 754, 458, 15, 36, "Output",
 CellID->91629666]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[21189, 775, 223, 7, 70, "ExampleSection",
 CellID->30703],
Cell[21415, 784, 354, 13, 70, "ExampleText",
 CellID->138952721],
Cell[CellGroupData[{
Cell[21794, 801, 833, 27, 70, "Input",
 CellID->33955874],
Cell[22630, 830, 347, 12, 51, "Output",
 CellID->76629991]
}, Open  ]],
Cell[22992, 845, 124, 3, 70, "ExampleText",
 CellID->48591970],
Cell[CellGroupData[{
Cell[23141, 852, 429, 13, 70, "Input",
 CellID->597358624],
Cell[23573, 867, 483, 17, 53, "Output",
 CellID->225673814]
}, Open  ]],
Cell[24071, 887, 214, 8, 70, "ExampleText",
 CellID->54497589],
Cell[CellGroupData[{
Cell[24310, 899, 877, 28, 70, "Input",
 CellID->517161024],
Cell[25190, 929, 530, 19, 53, "Output",
 CellID->77670538]
}, Open  ]],
Cell[25735, 951, 125, 3, 70, "ExampleDelimiter",
 CellID->453679385],
Cell[25863, 956, 84, 1, 70, "ExampleText",
 CellID->158401160],
Cell[25950, 959, 249, 8, 70, "Input",
 CellID->316758502],
Cell[CellGroupData[{
Cell[26224, 971, 232, 8, 70, "Input",
 CellID->458897865],
Cell[26459, 981, 1058, 23, 70, "Output",
 Evaluatable->False,
 CellID->306785250]
}, Open  ]],
Cell[27532, 1007, 143, 4, 70, "ExampleText",
 CellID->233767774],
Cell[CellGroupData[{
Cell[27700, 1015, 624, 19, 70, "Input",
 CellID->71106397],
Cell[28327, 1036, 747, 26, 54, "Output",
 CellID->125769134]
}, Open  ]],
Cell[29089, 1065, 78, 1, 70, "ExampleText",
 CellID->405412712],
Cell[CellGroupData[{
Cell[29192, 1070, 213, 7, 70, "Input",
 CellID->130181674],
Cell[29408, 1079, 2700, 50, 70, "Output",
 Evaluatable->False,
 CellID->65940727]
}, Open  ]],
Cell[32123, 1132, 124, 3, 70, "ExampleDelimiter",
 CellID->14364154],
Cell[32250, 1137, 114, 3, 70, "ExampleText",
 CellID->20125111],
Cell[32367, 1142, 259, 8, 70, "Input",
 CellID->22304118],
Cell[32629, 1152, 76, 1, 70, "ExampleText",
 CellID->138427790],
Cell[32708, 1155, 846, 26, 70, "Input",
 CellID->377461750],
Cell[CellGroupData[{
Cell[33579, 1185, 393, 13, 70, "Input",
 CellID->432287539],
Cell[33975, 1200, 5865, 155, 359, "Output",
 CellID->51535107]
}, Open  ]],
Cell[39855, 1358, 61, 1, 70, "ExampleText",
 CellID->158417198],
Cell[CellGroupData[{
Cell[39941, 1363, 388, 13, 70, "Input",
 CellID->75116359],
Cell[40332, 1378, 8228, 139, 70, "Output",
 Evaluatable->False,
 CellID->174562102]
}, Open  ]],
Cell[48575, 1520, 125, 3, 70, "ExampleDelimiter",
 CellID->221643494],
Cell[48703, 1525, 315, 10, 70, "ExampleText",
 CellID->150951800],
Cell[49021, 1537, 929, 33, 70, "Input",
 CellID->85726970],
Cell[49953, 1572, 140, 4, 70, "ExampleText",
 CellID->233537389],
Cell[CellGroupData[{
Cell[50118, 1580, 620, 20, 70, "Input",
 CellID->489441445],
Cell[50741, 1602, 10668, 179, 70, "Output",
 Evaluatable->False,
 CellID->632024652]
}, Open  ]],
Cell[61424, 1784, 125, 3, 70, "ExampleDelimiter",
 CellID->513749978],
Cell[61552, 1789, 130, 4, 70, "ExampleText",
 CellID->713047985],
Cell[61685, 1795, 805, 27, 70, "Input",
 CellID->101984099],
Cell[62493, 1824, 481, 16, 70, "Input",
 CellID->282973152],
Cell[62977, 1842, 62, 1, 70, "ExampleText",
 CellID->394152426],
Cell[CellGroupData[{
Cell[63064, 1847, 401, 14, 70, "Input",
 CellID->135872379],
Cell[63468, 1863, 158, 5, 36, "Output",
 CellID->440213334]
}, Open  ]],
Cell[63641, 1871, 60, 1, 70, "ExampleText",
 CellID->36716238],
Cell[CellGroupData[{
Cell[63726, 1876, 327, 11, 70, "Input",
 CellID->49751253],
Cell[64056, 1889, 12994, 217, 70, "Output",
 Evaluatable->False,
 CellID->452373150]
}, Open  ]],
Cell[77065, 2109, 125, 3, 70, "ExampleDelimiter",
 CellID->453076180],
Cell[77193, 2114, 131, 4, 70, "ExampleText",
 CellID->326857912],
Cell[77327, 2120, 345, 12, 70, "Input",
 CellID->237338708],
Cell[CellGroupData[{
Cell[77697, 2136, 234, 8, 70, "Input",
 CellID->573692869],
Cell[77934, 2146, 1140, 23, 70, "Output",
 Evaluatable->False,
 CellID->364979244]
}, Open  ]],
Cell[79089, 2172, 114, 3, 70, "ExampleText",
 CellID->534675154],
Cell[CellGroupData[{
Cell[79228, 2179, 151, 5, 70, "Input",
 CellID->105686975],
Cell[79382, 2186, 373, 13, 36, "Output",
 CellID->16012691]
}, Open  ]],
Cell[CellGroupData[{
Cell[79792, 2204, 127, 4, 70, "Input",
 CellID->35542566],
Cell[79922, 2210, 382, 13, 36, "Output",
 CellID->378209724]
}, Open  ]],
Cell[80319, 2226, 229, 8, 70, "ExampleText",
 CellID->21430366],
Cell[CellGroupData[{
Cell[80573, 2238, 269, 8, 70, "Input",
 CellID->712114601],
Cell[80845, 2248, 596, 14, 70, "Output",
 Evaluatable->False,
 CellID->38686201]
}, Open  ]],
Cell[CellGroupData[{
Cell[81478, 2267, 152, 5, 70, "Input",
 CellID->301431977],
Cell[81633, 2274, 968, 20, 70, "Output",
 Evaluatable->False,
 CellID->424893095]
}, Open  ]],
Cell[CellGroupData[{
Cell[82638, 2299, 128, 4, 70, "Input",
 CellID->706615872],
Cell[82769, 2305, 157, 5, 36, "Output",
 CellID->24952956]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[82975, 2316, 233, 7, 70, "ExampleSection",
 CellID->32544],
Cell[83211, 2325, 367, 13, 70, "ExampleText",
 CellID->875727382],
Cell[CellGroupData[{
Cell[83603, 2342, 209, 7, 70, "Input",
 CellID->827402583],
Cell[83815, 2351, 419, 15, 39, "Output",
 CellID->405353561]
}, Open  ]],
Cell[84249, 2369, 125, 3, 70, "ExampleDelimiter",
 CellID->446141051],
Cell[84377, 2374, 227, 8, 70, "ExampleText",
 CellID->218614874],
Cell[CellGroupData[{
Cell[84629, 2386, 365, 12, 70, "Input",
 CellID->147094569],
Cell[84997, 2400, 251, 9, 36, "Output",
 CellID->181494354]
}, Open  ]],
Cell[85263, 2412, 125, 3, 70, "ExampleDelimiter",
 CellID->323123013],
Cell[85391, 2417, 60, 1, 70, "ExampleText",
 CellID->991934362],
Cell[CellGroupData[{
Cell[85476, 2422, 262, 8, 70, "Input",
 CellID->349467808],
Cell[85741, 2432, 231, 8, 36, "Output",
 CellID->77440148]
}, Open  ]],
Cell[CellGroupData[{
Cell[86009, 2445, 401, 13, 70, "Input",
 CellID->25845485],
Cell[86413, 2460, 239, 9, 38, "Output",
 CellID->39808079]
}, Open  ]],
Cell[86667, 2472, 125, 3, 70, "ExampleDelimiter",
 CellID->146008241],
Cell[86795, 2477, 69, 1, 70, "ExampleText",
 CellID->314406036],
Cell[CellGroupData[{
Cell[86889, 2482, 215, 7, 70, "Input",
 CellID->227003087],
Cell[87107, 2491, 430, 16, 61, "Output",
 CellID->23624205]
}, Open  ]],
Cell[87552, 2510, 126, 3, 70, "ExampleDelimiter",
 CellID->1006516673],
Cell[87681, 2515, 69, 1, 70, "ExampleText",
 CellID->930209115],
Cell[CellGroupData[{
Cell[87775, 2520, 200, 6, 70, "Input",
 CellID->646218562],
Cell[87978, 2528, 226, 8, 51, "Output",
 CellID->133742640]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[88253, 2542, 227, 7, 70, "ExampleSection",
 CellID->32050],
Cell[88483, 2551, 231, 7, 70, "ExampleText",
 CellID->255283717],
Cell[CellGroupData[{
Cell[88739, 2562, 109, 3, 70, "Input",
 CellID->12162104],
Cell[88851, 2567, 247, 8, 36, "Output",
 CellID->190755669]
}, Open  ]],
Cell[89113, 2578, 124, 3, 70, "ExampleDelimiter",
 CellID->14924626],
Cell[89240, 2583, 425, 13, 70, "ExampleText",
 CellID->370442904],
Cell[CellGroupData[{
Cell[89690, 2600, 151, 4, 70, "Input",
 CellID->46088639],
Cell[89844, 2606, 199, 6, 36, "Output",
 CellID->615391083]
}, Open  ]],
Cell[90058, 2615, 125, 3, 70, "ExampleDelimiter",
 CellID->495916464],
Cell[90186, 2620, 121, 3, 70, "ExampleText",
 CellID->188188594],
Cell[CellGroupData[{
Cell[90332, 2627, 109, 3, 70, "Input",
 CellID->812151],
Cell[90444, 2632, 204, 7, 36, "Output",
 CellID->1528832]
}, Open  ]],
Cell[CellGroupData[{
Cell[90685, 2644, 134, 3, 70, "Input",
 CellID->351307062],
Cell[90822, 2649, 206, 7, 36, "Output",
 CellID->300225805]
}, Open  ]],
Cell[91043, 2659, 124, 3, 70, "ExampleDelimiter",
 CellID->78837792],
Cell[91170, 2664, 233, 7, 70, "ExampleText",
 CellID->673641459],
Cell[CellGroupData[{
Cell[91428, 2675, 397, 13, 70, "Input",
 CellID->236660448],
Cell[91828, 2690, 627, 19, 45, "Output",
 CellID->235289381]
}, Open  ]],
Cell[CellGroupData[{
Cell[92492, 2714, 104, 3, 70, "Input",
 CellID->123622996],
Cell[92599, 2719, 205, 7, 36, "Output",
 CellID->19711796]
}, Open  ]],
Cell[92819, 2729, 125, 3, 70, "ExampleDelimiter",
 CellID->110679831],
Cell[92947, 2734, 269, 9, 70, "ExampleText",
 CellID->216733786],
Cell[CellGroupData[{
Cell[93241, 2747, 397, 13, 70, "Input",
 CellID->510635538],
Cell[93641, 2762, 626, 19, 45, "Output",
 CellID->22536744]
}, Open  ]],
Cell[CellGroupData[{
Cell[94304, 2786, 97, 3, 70, "Input",
 CellID->238486270],
Cell[94404, 2791, 206, 7, 36, "Output",
 CellID->244609980]
}, Open  ]],
Cell[94625, 2801, 123, 3, 70, "ExampleText",
 CellID->140926449],
Cell[CellGroupData[{
Cell[94773, 2808, 124, 4, 70, "Input",
 CellID->497318653],
Cell[94900, 2814, 628, 11, 70, "Message",
 CellID->477412756],
Cell[95531, 2827, 206, 7, 36, "Output",
 CellID->663820968]
}, Open  ]],
Cell[95752, 2837, 418, 14, 70, "ExampleText",
 CellID->17465],
Cell[CellGroupData[{
Cell[96195, 2855, 250, 8, 70, "Input",
 CellID->809015494],
Cell[96448, 2865, 629, 11, 70, "Message",
 CellID->93301592],
Cell[97080, 2878, 206, 7, 36, "Output",
 CellID->125706083]
}, Open  ]],
Cell[97301, 2888, 125, 3, 70, "ExampleDelimiter",
 CellID->151128829],
Cell[97429, 2893, 372, 13, 70, "ExampleText",
 CellID->229130295],
Cell[CellGroupData[{
Cell[97826, 2910, 193, 6, 70, "Input",
 CellID->614562461],
Cell[98022, 2918, 282, 9, 36, "Output",
 CellID->918897257]
}, Open  ]],
Cell[CellGroupData[{
Cell[98341, 2932, 181, 6, 70, "Input",
 CellID->99294959],
Cell[98525, 2940, 245, 8, 36, "Output",
 CellID->89242928]
}, Open  ]],
Cell[98785, 2951, 124, 3, 70, "ExampleDelimiter",
 CellID->52897639],
Cell[98912, 2956, 167, 4, 70, "ExampleText",
 CellID->86757341],
Cell[CellGroupData[{
Cell[99104, 2964, 314, 10, 70, "Input",
 CellID->155725090],
Cell[99421, 2976, 395, 13, 51, "Output",
 CellID->752134003]
}, Open  ]],
Cell[99831, 2992, 124, 3, 70, "ExampleDelimiter",
 CellID->27336247],
Cell[99958, 2997, 281, 8, 70, "ExampleText",
 CellID->356648096],
Cell[CellGroupData[{
Cell[100264, 3009, 152, 5, 70, "Input",
 CellID->385445084],
Cell[100419, 3016, 258, 8, 36, "Output",
 CellID->110414120]
}, Open  ]],
Cell[100692, 3027, 125, 3, 70, "ExampleDelimiter",
 CellID->254059430],
Cell[100820, 3032, 116, 3, 70, "ExampleText",
 CellID->25094924],
Cell[CellGroupData[{
Cell[100961, 3039, 228, 7, 70, "Input",
 CellID->266815590],
Cell[101192, 3048, 421, 8, 70, "Message",
 CellID->490300623],
Cell[101616, 3058, 211, 7, 36, "Output",
 CellID->312914770]
}, Open  ]],
Cell[CellGroupData[{
Cell[101864, 3070, 251, 8, 70, "Input",
 CellID->294724839],
Cell[102118, 3080, 522, 9, 70, "Message",
 CellID->148451150],
Cell[102643, 3091, 622, 11, 70, "Message",
 CellID->166964921],
Cell[103268, 3104, 175, 5, 36, "Output",
 CellID->41941159]
}, Open  ]],
Cell[103458, 3112, 125, 3, 70, "ExampleDelimiter",
 CellID->100562957],
Cell[103586, 3117, 354, 12, 70, "ExampleText",
 CellID->542079127],
Cell[CellGroupData[{
Cell[103965, 3133, 494, 16, 70, "Input",
 CellID->530109224],
Cell[104462, 3151, 324, 12, 52, "Output",
 CellID->16349085]
}, Open  ]],
Cell[CellGroupData[{
Cell[104823, 3168, 433, 14, 70, "Input",
 CellID->202268354],
Cell[105259, 3184, 251, 9, 36, "Output",
 CellID->395979576]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[105559, 3199, 223, 7, 70, "ExampleSection",
 CellID->4301],
Cell[105785, 3208, 105, 2, 70, "ExampleText",
 CellID->168473678],
Cell[105893, 3212, 735, 23, 70, "Input",
 CellID->288154854],
Cell[CellGroupData[{
Cell[106653, 3239, 542, 17, 70, "Input",
 CellID->130872667],
Cell[107198, 3258, 615, 20, 51, "Output",
 CellID->630594162]
}, Open  ]],
Cell[CellGroupData[{
Cell[107850, 3283, 239, 8, 70, "Input",
 CellID->316066683],
Cell[108092, 3293, 4331, 76, 70, "Output",
 Evaluatable->False,
 CellID->457859981]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[112484, 3376, 311, 9, 70, "SeeAlsoSection",
 CellID->20770],
Cell[112798, 3387, 712, 26, 70, "SeeAlso",
 CellID->32014]
}, Open  ]],
Cell[CellGroupData[{
Cell[113547, 3418, 318, 9, 70, "MoreAboutSection",
 CellID->83728325],
Cell[113868, 3429, 152, 3, 70, "MoreAbout",
 CellID->38907142],
Cell[114023, 3434, 155, 3, 70, "MoreAbout",
 CellID->100883443],
Cell[114181, 3439, 174, 3, 70, "MoreAbout",
 CellID->206110422]
}, Open  ]],
Cell[114370, 3445, 27, 0, 70, "History"],
Cell[114400, 3447, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

