(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     28865,       1066]
NotebookOptionsPosition[     22691,        851]
NotebookOutlinePosition[     24091,        891]
CellTagsIndexPosition[     24006,        886]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"MatrixQ" :> Documentation`HelpLookup["paclet:ref/MatrixQ"], 
          "PositiveDefiniteMatrixQ" :> 
          Documentation`HelpLookup["paclet:ref/PositiveDefiniteMatrixQ"], 
          "ConjugateTranspose" :> 
          Documentation`HelpLookup["paclet:ref/ConjugateTranspose"]}, #, 2]& ,
         Evaluator -> Automatic], {1->"\<\"MatrixQ\"\>", 
       2->"\<\"PositiveDefiniteMatrixQ\"\>", 
       3->"\<\"ConjugateTranspose\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Matrices and Linear Algebra" :> 
          Documentation`HelpLookup["paclet:guide/MatricesAndLinearAlgebra"], 
          "New in 6.0: Matrix & Linear Algebra Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MatrixAndLinearAlgebraFunctions"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Matrices and Linear Algebra\"\>", 
       2->"\<\"New in 6.0: Matrix & Linear Algebra Functions\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["HermitianMatrixQ", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["HermitianMatrixQ",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/HermitianMatrixQ"], "[", 
       StyleBox["m", "TI"], "]"}]], "InlineFormula"],
     " \[LineSeparator]tests whether ",
     Cell[BoxData[
      StyleBox["m", "TI"]], "InlineFormula"],
     " is a Hermitian matrix."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["HermitianMatrixQ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/HermitianMatrixQ"], "[", 
   StyleBox["m", "TI"], "]"}]], "InlineFormula"],
 " gives ",
 Cell[BoxData[
  ButtonBox["True",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/True"]], "InlineFormula"],
 " if ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 " is explicitly Hermitian, and gives ",
 Cell[BoxData[
  ButtonBox["False",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/False"]], "InlineFormula"],
 " if it is a matrix that is not Hermitian."
}], "Notes",
 CellID->237854512],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["HermitianMatrixQ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/HermitianMatrixQ"], "[", 
   StyleBox["m", "TI"], "]"}]], "InlineFormula"],
 " is effectively equivalent to ",
 Cell[BoxData[
  RowBox[{
   StyleBox["m", "TI"], "==", 
   RowBox[{
    ButtonBox["ConjugateTranspose",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/ConjugateTranspose"], "[", 
    StyleBox["m", "TI"], "]"}]}]], "InlineFormula"],
 "."
}], "Notes",
 CellID->1067943069],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["HermitianMatrixQ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HermitianMatrixQ"]], "InlineFormula"],
 " works with ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 " objects."
}], "Notes",
 CellID->78903241],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["HermitianMatrixQ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HermitianMatrixQ"]], "InlineFormula"],
 " works for symbolic as well as numerical matrices."
}], "Notes",
 CellID->162284008]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->7146748],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(1)", "ExampleCount"]
}], "ExampleSection",
 CellID->1788233],

Cell["Test if a matrix is explicitly Hermitian:", "ExampleText",
 CellID->154603684],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"HermitianMatrixQ", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1", ",", 
      RowBox[{"2", "I"}], ",", 
      RowBox[{"3", "+", 
       RowBox[{"4", "I"}]}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"-", "2"}], "I"}], ",", "5", ",", 
      RowBox[{"6", "-", 
       RowBox[{"7", "I"}]}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"3", "-", 
       RowBox[{"4", " ", "I"}]}], ",", 
      RowBox[{"6", "+", 
       RowBox[{"7", " ", "I"}]}], ",", "8"}], "}"}]}], "}"}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->458787918],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->602877885]
}, Open  ]],

Cell["For a real matrix, this is equivalent to symmetry:", "ExampleText",
 CellID->519922740],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"HermitianMatrixQ", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1.", ",", "2."}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"2.", ",", "3."}], "}"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->400259259],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->76072822]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1293636265],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["HermitianMatrixQ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HermitianMatrixQ"]], "InlineFormula"],
 " works with ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 " objects:"
}], "ExampleText",
 CellID->180939996],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"s", "=", 
  RowBox[{"SparseArray", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"{", 
      RowBox[{"i_", ",", "j_"}], "}"}], "\[Rule]", 
     RowBox[{"i", "+", "j", "+", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{"i", "-", "j"}], ")"}], "I"}]}]}], ",", 
    RowBox[{"{", 
     RowBox[{"5", ",", "5"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->175368963],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "25", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 25, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"5", ",", "5"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{174, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->453451024]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"HermitianMatrixQ", "[", "s", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->387869650],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->51481789]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1020263627],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["HermitianMatrixQ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HermitianMatrixQ"]], "InlineFormula"],
 " works with symbolic matrices:"
}], "ExampleText",
 CellID->226023338],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"HermitianMatrixQ", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"a", ",", "b"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"b", ",", "a"}], "}"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->266106940],

Cell[BoxData["False"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->3788070]
}, Open  ]],

Cell["All symbolic quantities are assumed to be complex:", "ExampleText",
 CellID->293455183],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"HermitianMatrixQ", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"Re", "[", "a", "]"}], ",", "b"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"Conjugate", "[", "b", "]"}], ",", 
      RowBox[{"Re", "[", "a", "]"}]}], "}"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->192020097],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->205866709]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->258228157],

Cell["Use a different method for Hermitian matrices:", "ExampleText",
 CellID->19446229],

Cell[BoxData[
 RowBox[{
  RowBox[{"myLS", "[", 
   RowBox[{"m_", ",", "b_"}], "]"}], " ", ":=", " ", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", "x", "}"}], ",", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"If", "[", 
      RowBox[{
       RowBox[{"HermitianMatrixQ", "[", "m", "]"}], ",", 
       "\[IndentingNewLine]", 
       RowBox[{"Quiet", "[", 
        RowBox[{"x", "=", 
         RowBox[{"LinearSolve", "[", 
          RowBox[{"m", ",", "b", ",", 
           RowBox[{"Method", "\[Rule]", "\"\<Cholesky\>\""}]}], "]"}]}], 
        "]"}]}], "]"}], ";", "\[IndentingNewLine]", 
     RowBox[{"If", "[", 
      RowBox[{
       RowBox[{"Not", "[", 
        RowBox[{"VectorQ", "[", "x", "]"}], "]"}], ",", 
       RowBox[{"x", " ", "=", " ", 
        RowBox[{"LinearSolve", "[", 
         RowBox[{"m", ",", "b"}], "]"}]}]}], "]"}], ";", 
     "\[IndentingNewLine]", "x"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->158706631],

Cell["Construct complex-valued matrices for testing:", "ExampleText",
 CellID->251651921],

Cell[BoxData[{
 RowBox[{
  RowBox[{"n", "=", "1000"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"m", " ", "=", " ", 
   RowBox[{"RandomComplex", "[", 
    RowBox[{
     RowBox[{"1", "+", 
      RowBox[{"2", "I"}]}], ",", 
     RowBox[{"{", 
      RowBox[{"n", ",", "n"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"mi", "=", 
   RowBox[{"m", "+", 
    RowBox[{"ConjugateTranspose", "[", "m", "]"}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"mp", " ", "=", " ", 
   RowBox[{"m", ".", 
    RowBox[{"ConjugateTranspose", "[", "m", "]"}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"mp", "=", 
   RowBox[{"mp", "+", 
    RowBox[{"ConjugateTranspose", "[", "mp", "]"}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"b", "=", " ", 
   RowBox[{"N", "[", 
    RowBox[{"Range", "[", "n", "]"}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[2]:=",
 CellID->103529755],

Cell[TextData[{
 "For the non-Hermitian complex-valued matrix ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 ", the function just uses Gaussian elimination:"
}], "ExampleText",
 CellID->393229421],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Timing", "[", 
  RowBox[{
   RowBox[{"myLS", "[", 
    RowBox[{"m", ",", "b"}], "]"}], ";"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->22659715],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.7190000000000001`", ",", "Null"}], "}"}]], "Output",
 ImageSize->{93, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->36874128]
}, Open  ]],

Cell[TextData[{
 "For the Hermitian indefinite matrix ",
 Cell[BoxData[
  StyleBox["mi", "TI"]], "InlineFormula"],
 ", the function tries the Cholesky method first:"
}], "ExampleText",
 CellID->201736759],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Timing", "[", 
  RowBox[{
   RowBox[{"myLS", "[", 
    RowBox[{"mi", ",", "b"}], "]"}], ";"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->23456514],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.406`", ",", "Null"}], "}"}]], "Output",
 ImageSize->{93, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->4408350]
}, Open  ]],

Cell[TextData[{
 "For the Hermitian positive definite matrix ",
 Cell[BoxData[
  StyleBox["mp", "TI"]], "InlineFormula"],
 ", the function succeeds with the Cholesky method:"
}], "ExampleText",
 CellID->66297857],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Timing", "[", 
  RowBox[{
   RowBox[{"myLS", "[", 
    RowBox[{"mp", ",", "b"}], "]"}], ";"}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->96798091],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.5779999999999997`", ",", "Null"}], "}"}]], "Output",
 ImageSize->{93, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->376407176]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->2123667759],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["HermitianMatrixQ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/HermitianMatrixQ"], "[", 
   StyleBox["m", "TI"], "]"}]], "InlineFormula"],
 " is effectively equivalent to ",
 Cell[BoxData[
  RowBox[{
   StyleBox["m", "TI"], "==", 
   RowBox[{
    ButtonBox["ConjugateTranspose",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/ConjugateTranspose"], "[", 
    StyleBox["m", "TI"], "]"}]}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->258060763],

Cell[BoxData[{
 RowBox[{
  RowBox[{"m", " ", "=", " ", 
   RowBox[{"RandomComplex", "[", 
    RowBox[{
     RowBox[{"1", "+", "I"}], ",", 
     RowBox[{"{", 
      RowBox[{"10", ",", "10"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"m", "+", 
    RowBox[{"ConjugateTranspose", "[", "m", "]"}]}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->7435939],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"HermitianMatrixQ", "[", "m", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->324129958],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->136719165]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"m", "\[Equal]", 
  RowBox[{"ConjugateTranspose", "[", "m", "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->440498014],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->74395427]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1376953],

Cell["Hermitian matrices have all real eigenvalues:", "ExampleText",
 CellID->218741997],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"m", " ", "=", " ", 
   RowBox[{"RandomComplex", "[", 
    RowBox[{
     RowBox[{"1", "+", "I"}], ",", 
     RowBox[{"{", 
      RowBox[{"8", ",", "8"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"m", "+", 
    RowBox[{"ConjugateTranspose", "[", "m", "]"}]}]}], ";"}], "\n", 
 RowBox[{"HermitianMatrixQ", "[", "m", "]"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->390110785],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->725041407]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Eigenvalues", "[", "m", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->100590787],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"8.47366675148516`", ",", "2.335764006906478`", ",", 
   RowBox[{"-", "1.9228846797214605`"}], ",", "1.8077643658906417`", ",", 
   "0.8450602867708624`", ",", 
   RowBox[{"-", "0.6826597197510406`"}], ",", 
   RowBox[{"-", "0.45923985004613116`"}], ",", "0.07240652571089377`"}], 
  "}"}]], "Output",
 ImageSize->{526, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->611926356]
}, Open  ]],

Cell["\<\
This also means that their characteristic polynomials have real coefficients:\
\
\>", "ExampleText",
 CellID->205700727],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CharacteristicPolynomial", "[", 
  RowBox[{"m", ",", "x"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->375698726],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "1.319789440657332`"}], "+", 
  RowBox[{"15.746596428019478`", " ", "x"}], "+", 
  RowBox[{"37.91122026821656`", " ", 
   SuperscriptBox["x", "2"]}], "-", 
  RowBox[{"45.74829439362673`", " ", 
   SuperscriptBox["x", "3"]}], "-", 
  RowBox[{"67.65159524072962`", " ", 
   SuperscriptBox["x", "4"]}], "+", 
  RowBox[{"50.21517789261695`", " ", 
   SuperscriptBox["x", "5"]}], "+", 
  RowBox[{"11.99886222725953`", " ", 
   SuperscriptBox["x", "6"]}], "-", 
  RowBox[{"10.469877687245404`", " ", 
   SuperscriptBox["x", "7"]}], "+", 
  SuperscriptBox["x", "8"]}]], "Output",
 ImageSize->{332, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->106830447]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["MatrixQ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/MatrixQ"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["PositiveDefiniteMatrixQ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/PositiveDefiniteMatrixQ"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["ConjugateTranspose",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ConjugateTranspose"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Matrices and Linear Algebra",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/MatricesAndLinearAlgebra"]], "MoreAbout",
 CellID->8260772],

Cell[TextData[ButtonBox["New in 6.0: Matrix & Linear Algebra Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/NewIn60MatrixAndLinearAlgebraFunctions"]], "MoreAbout",
 CellID->425952660]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"HermitianMatrixQ - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 47, 34.6473490}", "context" -> "System`", 
    "keywords" -> {
     "Hermitian matrix", "symmetric matrix", "matrix property", 
      "test for Hermitian", "test for symmetry"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "HermitianMatrixQ[m] tests whether m is a Hermitian matrix.", 
    "synonyms" -> {
     "hermitian matrix q", "HermiteanMatrixQ", "HermiteanQ", "HermitianQ"}, 
    "title" -> "HermitianMatrixQ", "type" -> "Symbol", "uri" -> 
    "ref/HermitianMatrixQ"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[6743, 215, 358, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->7146748]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 23865, 879}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 1691, 39, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2314, 66, 1581, 37, 70, "ObjectNameGrid"],
Cell[3898, 105, 574, 18, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[4509, 128, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[4987, 145, 622, 22, 70, "Notes",
 CellID->237854512],
Cell[5612, 169, 514, 18, 70, "Notes",
 CellID->1067943069],
Cell[6129, 189, 337, 12, 70, "Notes",
 CellID->78903241],
Cell[6469, 203, 237, 7, 70, "Notes",
 CellID->162284008]
}, Closed]],
Cell[CellGroupData[{
Cell[6743, 215, 358, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->7146748],
Cell[CellGroupData[{
Cell[7126, 229, 146, 5, 70, "ExampleSection",
 CellID->1788233],
Cell[7275, 236, 84, 1, 70, "ExampleText",
 CellID->154603684],
Cell[CellGroupData[{
Cell[7384, 241, 631, 23, 28, "Input",
 CellID->458787918],
Cell[8018, 266, 161, 5, 36, "Output",
 CellID->602877885]
}, Open  ]],
Cell[8194, 274, 93, 1, 70, "ExampleText",
 CellID->519922740],
Cell[CellGroupData[{
Cell[8312, 279, 265, 9, 28, "Input",
 CellID->400259259],
Cell[8580, 290, 160, 5, 36, "Output",
 CellID->76072822]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[8789, 301, 221, 7, 70, "ExampleSection",
 CellID->1293636265],
Cell[9013, 310, 344, 12, 70, "ExampleText",
 CellID->180939996],
Cell[CellGroupData[{
Cell[9382, 326, 406, 14, 70, "Input",
 CellID->175368963],
Cell[9791, 342, 481, 17, 36, "Output",
 CellID->453451024]
}, Open  ]],
Cell[CellGroupData[{
Cell[10309, 364, 112, 3, 70, "Input",
 CellID->387869650],
Cell[10424, 369, 160, 5, 36, "Output",
 CellID->51481789]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[10633, 380, 244, 7, 70, "ExampleSection",
 CellID->1020263627],
Cell[10880, 389, 223, 7, 70, "ExampleText",
 CellID->226023338],
Cell[CellGroupData[{
Cell[11128, 400, 261, 9, 70, "Input",
 CellID->266106940],
Cell[11392, 411, 160, 5, 36, "Output",
 CellID->3788070]
}, Open  ]],
Cell[11567, 419, 93, 1, 70, "ExampleText",
 CellID->293455183],
Cell[CellGroupData[{
Cell[11685, 424, 367, 12, 70, "Input",
 CellID->192020097],
Cell[12055, 438, 161, 5, 36, "Output",
 CellID->205866709]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[12265, 449, 227, 7, 70, "ExampleSection",
 CellID->258228157],
Cell[12495, 458, 88, 1, 70, "ExampleText",
 CellID->19446229],
Cell[12586, 461, 955, 27, 70, "Input",
 CellID->158706631],
Cell[13544, 490, 89, 1, 70, "ExampleText",
 CellID->251651921],
Cell[13636, 493, 843, 28, 70, "Input",
 CellID->103529755],
Cell[14482, 523, 210, 6, 70, "ExampleText",
 CellID->393229421],
Cell[CellGroupData[{
Cell[14717, 533, 176, 6, 70, "Input",
 CellID->22659715],
Cell[14896, 541, 223, 7, 36, "Output",
 CellID->36874128]
}, Open  ]],
Cell[15134, 551, 204, 6, 70, "ExampleText",
 CellID->201736759],
Cell[CellGroupData[{
Cell[15363, 561, 177, 6, 70, "Input",
 CellID->23456514],
Cell[15543, 569, 209, 7, 36, "Output",
 CellID->4408350]
}, Open  ]],
Cell[15767, 579, 212, 6, 70, "ExampleText",
 CellID->66297857],
Cell[CellGroupData[{
Cell[16004, 589, 177, 6, 70, "Input",
 CellID->96798091],
Cell[16184, 597, 224, 7, 36, "Output",
 CellID->376407176]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[16457, 610, 238, 7, 70, "ExampleSection",
 CellID->2123667759],
Cell[16698, 619, 519, 18, 70, "ExampleText",
 CellID->258060763],
Cell[17220, 639, 389, 13, 70, "Input",
 CellID->7435939],
Cell[CellGroupData[{
Cell[17634, 656, 112, 3, 70, "Input",
 CellID->324129958],
Cell[17749, 661, 161, 5, 36, "Output",
 CellID->136719165]
}, Open  ]],
Cell[CellGroupData[{
Cell[17947, 671, 144, 4, 70, "Input",
 CellID->440498014],
Cell[18094, 677, 160, 5, 36, "Output",
 CellID->74395427]
}, Open  ]],
Cell[18269, 685, 123, 3, 70, "ExampleDelimiter",
 CellID->1376953],
Cell[18395, 690, 88, 1, 70, "ExampleText",
 CellID->218741997],
Cell[CellGroupData[{
Cell[18508, 695, 442, 14, 70, "Input",
 CellID->390110785],
Cell[18953, 711, 161, 5, 36, "Output",
 CellID->725041407]
}, Open  ]],
Cell[CellGroupData[{
Cell[19151, 721, 107, 3, 70, "Input",
 CellID->100590787],
Cell[19261, 726, 471, 12, 36, "Output",
 CellID->611926356]
}, Open  ]],
Cell[19747, 741, 130, 4, 70, "ExampleText",
 CellID->205700727],
Cell[CellGroupData[{
Cell[19902, 749, 143, 4, 70, "Input",
 CellID->375698726],
Cell[20048, 755, 742, 21, 54, "Output",
 CellID->106830447]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[20851, 783, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[21170, 794, 748, 26, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[21955, 825, 299, 8, 70, "MoreAboutSection"],
Cell[22257, 835, 161, 3, 70, "MoreAbout",
 CellID->8260772],
Cell[22421, 840, 198, 4, 70, "MoreAbout",
 CellID->425952660]
}, Open  ]],
Cell[22634, 847, 27, 0, 70, "History"],
Cell[22664, 849, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

