(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     38275,       1344]
NotebookOptionsPosition[     31556,       1110]
NotebookOutlinePosition[     32828,       1146]
CellTagsIndexPosition[     32742,       1141]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Special Functions" :> 
          Documentation`HelpLookup["paclet:tutorial/SpecialFunctions"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Special Functions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"KelvinBer" :> Documentation`HelpLookup["paclet:ref/KelvinBer"],
           "KelvinKei" :> Documentation`HelpLookup["paclet:ref/KelvinKei"], 
          "BesselJ" :> Documentation`HelpLookup["paclet:ref/BesselJ"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"KelvinBer\"\>", 
       2->"\<\"KelvinKei\"\>", 3->"\<\"BesselJ\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Bessel-Related Functions" :> 
          Documentation`HelpLookup["paclet:guide/BesselRelatedFunctions"], 
          "New in 6.0: Mathematical Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicalFunctions"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Bessel-Related Functions\"\>", 
       2->"\<\"New in 6.0: Mathematical Functions\"\>"}, "\<\"more about\"\>",
        
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["KelvinBei", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["KelvinBei",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/KelvinBei"], "[", 
       StyleBox["z", "TI"], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the Kelvin function ",
     Cell[BoxData[
      FormBox[
       RowBox[{"bei", "(", "z", ")"}], TraditionalForm]], "InlineMath"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["KelvinBei",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/KelvinBei"], "[", 
       RowBox[{
        StyleBox["n", "TI"], ",", 
        StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the Kelvin function ",
     Cell[BoxData[
      FormBox[
       RowBox[{
        SubscriptBox["bei", "n"], "(", "z", ")"}], TraditionalForm]], 
      "InlineMath"],
     ". "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->15419]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["\<\
Mathematical function, suitable for both symbolic and numerical manipulation.\
\
\>", "Notes",
 CellID->20640],

Cell[TextData[{
 "For positive real values of parameters, ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SubscriptBox["bei", "n"], "(", "z", ")"}], "=", 
    RowBox[{"Im", "(", 
     RowBox[{
      SuperscriptBox["e", 
       RowBox[{"n", "\[InvisibleSpace]", "\[Pi]", "\[InvisibleSpace]", "i"}]],
       
      RowBox[{
       SubscriptBox["J", "n"], "(", 
       RowBox[{"z", "\[InvisibleSpace]", 
        SuperscriptBox["e", 
         RowBox[{
          RowBox[{"-", "\[Pi]"}], "\[InvisibleSpace]", 
          RowBox[{"i", "/", "4"}]}]]}], ")"}]}], ")"}]}], TraditionalForm]], 
  "InlineMath"],
 ". For other values, ",
 Cell[BoxData[
  FormBox["bei", TraditionalForm]], "InlineMath"],
 " is defined by analytic continuation. "
}], "Notes",
 CellID->19040],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["KelvinBei",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/KelvinBei"], "[", 
   RowBox[{
    StyleBox["n", "TI"], ",", 
    StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
 " has a branch cut discontinuity in the complex ",
 Cell[BoxData[
  StyleBox["z", "TI"]], "InlineFormula"],
 " plane running from ",
 Cell[BoxData[
  FormBox[
   RowBox[{"-", "\[Infinity]"}], TraditionalForm]], "InlineMath"],
 " to ",
 Cell[BoxData[
  FormBox["0", TraditionalForm]], "InlineMath"],
 ". "
}], "Notes",
 CellID->24388],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["KelvinBei",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/KelvinBei"], "[", 
   StyleBox["z", "TI"], "]"}]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["KelvinBei",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/KelvinBei"], "[", 
   RowBox[{"0", ",", 
    StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
 "."
}], "Notes",
 CellID->83559840],

Cell[TextData[{
 "For certain special arguments, ",
 Cell[BoxData[
  ButtonBox["KelvinBei",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KelvinBei"]], "InlineFormula"],
 " automatically evaluates to exact values."
}], "Notes",
 CellID->792533805],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["KelvinBei",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KelvinBei"]], "InlineFormula"],
 " can be evaluated to arbitrary numerical precision."
}], "Notes",
 CellID->31226],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["KelvinBei",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KelvinBei"]], "InlineFormula"],
 " automatically threads over lists."
}], "Notes",
 CellID->28162]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->79473039],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(3)", "ExampleCount"]
}], "ExampleSection",
 CellID->263300735],

Cell["Evaluate numerically:", "ExampleText",
 CellID->64203606],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KelvinBei", "[", "2.5", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->25647115],

Cell[BoxData["1.4571820441598047`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->292417452]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->110621565],

Cell[TextData[{
 "Plot ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["bei", "0"], "(", "x", ")"}], TraditionalForm]], 
  "InlineMath"],
 ":"
}], "ExampleText",
 CellID->283017315],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"KelvinBei", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->273567397],

Cell[GraphicsData["CompressedBitmap", "\<\
eJydWE1oXVUQPnn33Zektdj6mqZJ05qioi5diCD+LESLSPEHigsRTdNqoqQp
aQpiN7rRjaJNWmtBLeIP+LOoKBWhglgoFsRCsSAqSAWpC4tgcREQrt/M3PPm
3HnnnrwmvNycOfebOfN/5mX7xPzU7pmJ+enJifFtcxN7p6Yn943fNzuHrazP
OfcFfndtdFgXWPHv+qIoXEYr/nFv08PZ3QP0aP7y819nf/gDn/M//rklinuO
cZcu/bu4cPrhB9+b3PlpY3bP5y7DAuQnH59zYx2+BtgEf+yd7x/d8SHegrEZ
SJieOn758tJo9KT9jLtw4W+IBv/S0n8DjIBq2AGbG4nyzTN96tvfIBz2uNVM
YwU9iWtjlGsf09DwwPNfEWoN01+f/PXFF07ScjjKNcc0tIMttLya6VdfOfXZ
8fO03FDxxZx/CwZarmUanoA9eAbo8Iy9xA2OttewPGsoip4t0bIPp8HvJDuO
3lOiNxjZ66PomRI9bGQrmqycMbLY39nRN8988P5ZWrajsp9l+ssTP7380je0
HPG+QVbVcz1juDYxvX/+xJnvfqflNczVwrMJpkapYctImWYaLIg9rJIMDmIv
UvrpmQtX58kyQ1mZoaeYRkVBKZK92fvugfuPES2yhU1+Gp5sqpT4OUQ+bbL7
WqahONSn5TpmHwy4e9H6KSN13HhkZVJ3M33x4j+oTMqbrcYX6zrRcq3Q9noP
hPQuLx064+muYxohRSsofLVxFL2HY7kwaaRcbyxfG83DnYbrBqbhOVhW5aIz
LfpGpkEgTQrfRbrOeNLHBN2CuG4qpBrjGj3h866M4M0BXX8GVR2l3WazP1pq
nntvIJtJ6lC53/aRvPeet2iZl9npKy6LnjZq8seeyj2gUgUSc/QSRIMyZqgL
0Ta+zLsQpBNdlX30txKXsYh+xMp6DpiMhPnILfS04agOFJlWj2eHZ1odRoyP
uMP1hzKREagnJDljyQ6RiR0oeCU+SOnhM2BTGVnu8DgX93MkThIFzf+VaNAt
dSyME71BFuKSQSKMdmFFGkoF7uGXK9DA27wlqhlC//hjH0nP9ppxPmbQCc4p
yh4VdhutA6J2sBwkKtRETsOOW03k1kT1eoT5YBhQyHQsbivKjoIbuZ7vIX8e
+HAkjrmdEQgSQkXLq6J82zmEgGCqAR+OpGy7g1F61xpemSO55WyjvRb4pY3i
OMxA0PquoqhMBiKBboF3fb/v0iU3dNPT8B6dl/MDNOpPPogGNwRG3X3nUXxc
A4+G38NfwlDFuhyzRKm5c5Tb9n6w53OHI/gg9mAKhCEGMBGOgUAaQF05O8S0
rpFaa2WPeEtnPj/KRmKtivdn2wPIVkof10R6i3Mxd4lvUQqBb10GJ2CfXvV7
P6NbVHzfhNcReQiEYu1lzhbdMAxicKi1sMuCW7zdZUwHOCXxDaJ8zxURzAir
o1JeM3mqKKri1/mtTggpGXJ716NEFi5RZDMtVyVQ2ttSKJ0EFUVaH+S3OqWn
ZKDAkc3LoXRWj6PkRO01gwmU3t+KIq0X+K3215QMNCu0meVQOhEJiio1/K5g
8+ygsVW4+Jvpst8NLL1gPMJSRGJy9o/LQXlJYpdyVgWoOm2yjk81T3rntjpQ
B1Uv9jLDxy3RWhroeDe8RW1MhAvGwwXKZeMtKJ1JFEUyD5lY9EdlLBpPp1Dq
0XoUeSyu7aLxhMpQbbWbpPTQXI2jRJb2nFYChS6PS3E5lPZTRZHWh4tq9aZk
aGdKoXRSEpSpnZp8sd6L8vaQrYdMRokc30Hk2EAc3+JZxw+aa3mHr6viKvrY
KUDk6D0gcny19Mqv0VoZv8Yzv4IYHDZRziuZ8oaJrr7tlqF1n0JpFcRRcqJW
QTOBwmyDkaKKIq2P8Fu9u1IyMDhh1EyjqDuktNUbLnWS3t4pFMoaxV1nk+Z4
XMYRnwklHZ/G4v95ll3X9z+Pp8qK\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 100},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->783022416]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->127075355],

Cell["Series at the origin:", "ExampleText",
 CellID->4255939],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"KelvinBei", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "15"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->258655594],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
   FractionBox[
    SuperscriptBox["x", "2"], "4"], "-", 
   FractionBox[
    SuperscriptBox["x", "6"], "2304"], "+", 
   FractionBox[
    SuperscriptBox["x", "10"], "14745600"], "-", 
   FractionBox[
    SuperscriptBox["x", "14"], "416179814400"], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "x", "]"}], "16"],
    SeriesData[$CellContext`x, 0, {}, 2, 16, 1],
    Editable->False]}],
  SeriesData[$CellContext`x, 0, {
    Rational[1, 4], 0, 0, 0, 
    Rational[-1, 2304], 0, 0, 0, 
    Rational[1, 14745600], 0, 0, 0, 
    Rational[-1, 416179814400]}, 2, 16, 1],
  Editable->False]], "Output",
 ImageSize->{301, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->5856940]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->22176],

Cell["Evaluate for complex arguments and orders:", "ExampleText",
 CellID->2708615],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KelvinBei", "[", 
  RowBox[{
   RowBox[{"1", "-", "I"}], ",", 
   RowBox[{"-", "2.5"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->17954271],

Cell[BoxData[
 RowBox[{"26.52206790267168`", "\[InvisibleSpace]", "-", 
  RowBox[{"153.52593805458847`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{126, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->379120653]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->464533498],

Cell["Evaluate to high precision:", "ExampleText",
 CellID->66248962],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"KelvinBei", "[", 
    RowBox[{"1", ",", "10"}], "]"}], ",", "50"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->170986264],

Cell[BoxData["131.8786391756869304066049974630014712083259749802487134302178`\
50."], "Output",
 ImageSize->{361, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->26921967]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->295287149],

Cell["\<\
The precision of the output tracks the precision of the input:\
\>", "ExampleText",
 CellID->29695085],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KelvinBei", "[", 
  RowBox[{"1", ",", "10.00000000000000000000000000000000000"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->465676590],

Cell[BoxData["131.878639175686930406604997463001471203114237369948`35.\
38692205845309"], "Output",
 ImageSize->{263, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->18382165]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->221466515],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TraditionalForm",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TraditionalForm"]], "InlineFormula"],
 " formatting:"
}], "ExampleText",
 CellID->231345657],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"KelvinBei", "[", 
   RowBox[{"n", ",", "r"}], "]"}], "//", "TraditionalForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->12842311],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["bei", "n"], "(", "r", ")"}], TraditionalForm]], "Output",
 ImageSize->{38, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//TraditionalForm=",
 CellID->895685377]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->22005],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["KelvinBei",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KelvinBei"]], "InlineFormula"],
 " can be applied to a power series:"
}], "ExampleText",
 CellID->90913756],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KelvinBei", "[", 
  RowBox[{
   RowBox[{"Log", "[", 
    RowBox[{"1", "+", "x"}], "]"}], "+", 
   RowBox[{
    RowBox[{"O", "[", "x", "]"}], "^", "7"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->221683768],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
   FractionBox[
    SuperscriptBox["x", "2"], "4"], "-", 
   FractionBox[
    SuperscriptBox["x", "3"], "4"], "+", 
   FractionBox[
    RowBox[{"11", " ", 
     SuperscriptBox["x", "4"]}], "48"], "-", 
   FractionBox[
    RowBox[{"5", " ", 
     SuperscriptBox["x", "5"]}], "24"], "+", 
   FractionBox[
    RowBox[{"243", " ", 
     SuperscriptBox["x", "6"]}], "1280"], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "x", "]"}], "7"],
    SeriesData[$CellContext`x, 0, {}, 2, 7, 1],
    Editable->False]}],
  SeriesData[$CellContext`x, 0, {
    Rational[1, 4], 
    Rational[-1, 4], 
    Rational[11, 48], 
    Rational[-5, 24], 
    Rational[243, 1280]}, 2, 7, 1],
  Editable->False]], "Output",
 ImageSize->{235, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->191359368]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->429],

Cell["Solve the Kelvin differential equation:", "ExampleText",
 CellID->127419602],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{
      SuperscriptBox["x", "4"], " ", 
      RowBox[{
       SuperscriptBox["f", 
        TagBox[
         RowBox[{"(", "4", ")"}],
         Derivative],
        MultilineFunction->None], "[", "x", "]"}]}], "+", 
     RowBox[{"2", " ", 
      SuperscriptBox["x", "3"], " ", 
      RowBox[{
       SuperscriptBox["f", 
        TagBox[
         RowBox[{"(", "3", ")"}],
         Derivative],
        MultilineFunction->None], "[", "x", "]"}]}], "-", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"1", "+", 
        RowBox[{"2", " ", 
         SuperscriptBox["n", "2"]}]}], ")"}], " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{
         RowBox[{"-", "x"}], " ", 
         RowBox[{
          SuperscriptBox["f", "\[Prime]",
           MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
         SuperscriptBox["x", "2"], " ", 
         RowBox[{
          SuperscriptBox["f", "\[Prime]\[Prime]",
           MultilineFunction->None], "[", "x", "]"}]}]}], ")"}]}], "+", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{
         RowBox[{"-", "4"}], " ", 
         SuperscriptBox["n", "2"]}], "+", 
        SuperscriptBox["n", "4"], "+", 
        SuperscriptBox["x", "4"]}], ")"}], " ", 
      RowBox[{"f", "[", "x", "]"}]}]}], "\[Equal]", "0"}], ",", 
   RowBox[{"f", "[", "x", "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->61770610],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"f", "[", "x", "]"}], "\[Rule]", 
    RowBox[{
     RowBox[{
      RowBox[{"C", "[", "4", "]"}], " ", 
      RowBox[{"KelvinBei", "[", 
       RowBox[{"n", ",", "x"}], "]"}]}], "+", 
     RowBox[{
      RowBox[{"C", "[", "3", "]"}], " ", 
      RowBox[{"KelvinBer", "[", 
       RowBox[{"n", ",", "x"}], "]"}]}], "+", 
     RowBox[{
      RowBox[{"C", "[", "2", "]"}], " ", 
      RowBox[{"KelvinKei", "[", 
       RowBox[{"n", ",", "x"}], "]"}]}], "+", 
     RowBox[{
      RowBox[{"C", "[", "1", "]"}], " ", 
      RowBox[{"KelvinKer", "[", 
       RowBox[{"n", ",", "x"}], "]"}]}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{471, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->241143927]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->809257294],

Cell["\<\
Plot the resistance of a wire with circular cross section versus AC frequency \
(skin effect):\
\>", "ExampleText",
 CellID->146335090],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"\[Omega]", "/", "2"}], " ", 
    RowBox[{
     RowBox[{"(", " ", 
      RowBox[{
       RowBox[{
        RowBox[{"KelvinBer", "[", "\[Omega]", "]"}], 
        RowBox[{
         RowBox[{"KelvinBei", "'"}], "[", "\[Omega]", "]"}]}], "-", 
       RowBox[{
        RowBox[{"KelvinBei", "[", "\[Omega]", "]"}], 
        RowBox[{
         RowBox[{"KelvinBer", "'"}], "[", "\[Omega]", "]"}]}]}], ")"}], "/", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"KelvinBer", "'"}], "[", "\[Omega]", "]"}], "^", "2"}], "+", 
       
       RowBox[{
        RowBox[{
         RowBox[{"KelvinBei", "'"}], "[", "\[Omega]", "]"}], "^", "2"}]}], 
      ")"}]}]}], ",", 
   RowBox[{"{", 
    RowBox[{"\[Omega]", ",", "0", ",", "4"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->673371864],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzdWGtsVFUQPt272zZVFHxWfFW6Lb4VjcVHsCICglZERTS10bWibZCCpTUK
QWmQZ4qv+qDxQYyvoP7ASDAmJTGSGEmMJEYSoyQGE4M/JCYSfzQxOX4z59ye
e87O3m3Xfzbbe8/cM/PNnJlvZm/bVujrWrai0NfdWWiY31tY1dXdubph3spe
PIqqlFJ78Ntfr7DWWPHvHq1xL7624BpBIY971t5Jnk67tND8ozKsPTN4mk/I
SrDKjwulKZCbA3majS0SrUPtRlwyEJOoFFt2XNZhxHmLJluTL5V1cnEGGj37
Fis1J2Isne1knSYHGu+kPJ2E64d0ZLd/8Lvf8FE5ukxAzrL86e5DeLTz7W9V
hi51mzd9ubLnM/PBxtw5b7bOGl665AOI/X17WcdijI7+o3JBjP83uUqsBLOv
ekw+evQvpOPAN79ydmqRJiQLicMHi+2D+5FjKNRLKCmy801smkFSHXIOJOAB
FdgoTXfX7qGXv943chhRPCZG28LyoR9+//ij7wfWj6gMqqki1BlGx4+Pqk7R
6nq61AD0870/wuq2BTvhD46PHftbPeRF1srZQFiIafGid6EGEwB3iLizOc3Q
RgDQxhWRqXZRdw7r4swAxDk7H/4EC3W/530+6/z80x8GD2ckXt4r4t3KujgB
MgZdVIvOf4+ou5B1sQ+tjvZdFvcu1rUNb38ygWVkFSi6NnqQNR6Bsv+rXxaz
aWgT8i4K5DsYB9VYu+YL8OvIkT8XVYRzJ+OAqsjl8I4DOFLbuHFI5OizMHv/
vYOAQBlvryiOuxkHxwBbUDikeWFFOEtYNvQgHi0whYlcfaptJXL2vpQt4BkW
yIDhRFH172MZh4QWMX6eqMVMpM4yqVBzPWYypyNY44ygj7pFxHiAZdADaSCG
3SxqdcTnBBbFM9vz9GAcB6LFydRNIobRAodsrlo9jALvohDwAJqqG3U600mO
mZ70YnBwEjCVunWWWNdsgBSi8EyiGYUS0PKGCbH0ERZQXfSujmdZkXV1IIf7
yygzQKjM+lFrfZ1ozQT13i3iidFl7a7VaUxOYsUWM8Wqd9vdFq/ay+3Ta0Sb
5XH+bPavFrUej2uEMtPyKs9DD++CBCATLWeIGCsSbMFUKqHVE2uBuZj06kqX
spLcTPb8Km07Efbgv7qioooaFDM56Hvj8opQntC2V9Fn1O+XiSgxG+gE9LiX
95FK5ImWl4pW5ablapYxizBuaHlJRSfoY9m81tDyYh0yNY46zv+TbIHvLZSY
lheJVe7XdgLZKSZrGSxUETOXqnChx7qneBezGM4ou9NFDKNl3qJo2SxqPc0y
UoWE0bLJ87Qm5iQ80exvEjGMlmNeXtRaG3CiUZfgd27Mxm7FGV7HCGhEtCMt
p1XEj3VBZS8QUcgqa8OLxnwjbkRP3yulrZL9GcbyTFDXBhEl2RfJ58/GdUcM
tDy/rDXdB3gfQw6jjpbn6ZDJNWUyZhDclDtXrK/Rcsx2WhTFBm3nCk5O1T9H
xNgQVPhsUes5lt1Mnup52hhUaqqIsTGoxFmi1ibtT6R6XYKzJWbyFu1315li
xcrNos3BiSpD2RKc+Ixxo5C4NcjE6WW5l3y+LcjDaSWt0+RtQR4MSsjk+H8V
dB8MznyqWOXBgHWniFrbA9ZN8Vj3fBwbPBH7p4gYRst1wWRR6wWW3eQ+2fP0
kva78SQR40VtuxGeaCbLWgbLzeRJOp3fzIgEv4d4z33XnijWVX4PTGoMBWc6
4T/h0J/l7bvoL4iJ4JDCK7ztJmxdSfv4mz/5/FVt3jdlK4ndlMPX+Tkcwi0t
a3Uaq5MIr7GMc+K0xO5ascYG33VujcemYe13ZrWIsUP7nSdrGSzXbTnP0xu8
u2/k8MD6kbF8FGEYLddlUYqW67bI8/QW7+Id2rIpI2Kk/f9RVf0L+iNz2g==
\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 117},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->755197575]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->29619],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["FullSimplify",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FullSimplify"]], "InlineFormula"],
 " to simplify expressions involving Kelvin functions:"
}], "ExampleText",
 CellID->540949068],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"D", "[", 
  RowBox[{
   RowBox[{"x", " ", 
    RowBox[{"KelvinBei", "[", 
     RowBox[{"1", ",", "x"}], "]"}]}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->275163834],

Cell[BoxData[
 RowBox[{
  RowBox[{"KelvinBei", "[", 
   RowBox[{"1", ",", "x"}], "]"}], "+", 
  FractionBox[
   RowBox[{"x", " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", 
       RowBox[{"KelvinBei", "[", 
        RowBox[{"0", ",", "x"}], "]"}]}], "+", 
      RowBox[{"KelvinBei", "[", 
       RowBox[{"2", ",", "x"}], "]"}], "+", 
      RowBox[{"KelvinBer", "[", 
       RowBox[{"0", ",", "x"}], "]"}], "-", 
      RowBox[{"KelvinBer", "[", 
       RowBox[{"2", ",", "x"}], "]"}]}], ")"}]}], 
   RowBox[{"2", " ", 
    SqrtBox["2"]}]]}]], "Output",
 ImageSize->{494, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->258721107]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FullSimplify", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->71003703],

Cell[BoxData[
 FractionBox[
  RowBox[{"x", " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", 
      RowBox[{"KelvinBei", "[", 
       RowBox[{"0", ",", "x"}], "]"}]}], "+", 
     RowBox[{"KelvinBer", "[", 
      RowBox[{"0", ",", "x"}], "]"}]}], ")"}]}], 
  SqrtBox["2"]]], "Output",
 ImageSize->{257, 34},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->213987913]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->545301568],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["FunctionExpand",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FunctionExpand"]], "InlineFormula"],
 " to expand Kelvin functions of half-integer orders:"
}], "ExampleText",
 CellID->127473317],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FunctionExpand", "[", 
  RowBox[{"KelvinBei", "[", 
   RowBox[{
    RowBox[{"1", "/", "2"}], ",", "x"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->788709929],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SqrtBox[
    FractionBox["2", "\[Pi]"]], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      FractionBox["1", "2"], " ", 
      SqrtBox[
       RowBox[{"2", "+", 
        SqrtBox["2"]}]], " ", 
      RowBox[{"Cosh", "[", 
       FractionBox["x", 
        SqrtBox["2"]], "]"}], " ", 
      RowBox[{"Sin", "[", 
       FractionBox["x", 
        SqrtBox["2"]], "]"}]}], "+", 
     RowBox[{
      FractionBox["1", "2"], " ", 
      SqrtBox[
       RowBox[{"2", "-", 
        SqrtBox["2"]}]], " ", 
      RowBox[{"Cos", "[", 
       FractionBox["x", 
        SqrtBox["2"]], "]"}], " ", 
      RowBox[{"Sinh", "[", 
       FractionBox["x", 
        SqrtBox["2"]], "]"}]}]}], ")"}]}], 
  SqrtBox["x"]]], "Output",
 ImageSize->{483, 62},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->476394900]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->370746783],

Cell["Integrate expressions involving Kelvin functions:", "ExampleText",
 CellID->122712053],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"r", " ", 
    RowBox[{"(", 
     RowBox[{
      SuperscriptBox[
       RowBox[{"KelvinBer", "[", "r", "]"}], "2"], "+", 
      SuperscriptBox[
       RowBox[{"KelvinBei", "[", "r", "]"}], "2"]}], ")"}]}], ",", "r"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->586709475],

Cell[BoxData[
 RowBox[{"-", 
  RowBox[{
   FractionBox["1", 
    SqrtBox["2"]], 
   RowBox[{"r", " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{
       RowBox[{"KelvinBer", "[", 
        RowBox[{"0", ",", "r"}], "]"}], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"-", 
          RowBox[{"KelvinBei", "[", 
           RowBox[{"1", ",", "r"}], "]"}]}], "+", 
         RowBox[{"KelvinBer", "[", 
          RowBox[{"1", ",", "r"}], "]"}]}], ")"}]}], "+", 
      RowBox[{
       RowBox[{"KelvinBei", "[", 
        RowBox[{"0", ",", "r"}], "]"}], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"KelvinBei", "[", 
          RowBox[{"1", ",", "r"}], "]"}], "+", 
         RowBox[{"KelvinBer", "[", 
          RowBox[{"1", ",", "r"}], "]"}]}], ")"}]}]}], ")"}]}]}]}]], "Output",\

 ImageSize->{414, 51},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->830901745]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->18830],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["KelvinBer",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/KelvinBer"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["KelvinKei",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/KelvinKei"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["BesselJ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BesselJ"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->5281]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->25938],

Cell[TextData[ButtonBox["Special Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/SpecialFunctions"]], "Tutorials",
 CellID->29228]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->167386683],

Cell[TextData[ButtonBox["Bessel-Related Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/BesselRelatedFunctions"]], "MoreAbout",
 CellID->113773963],

Cell[TextData[ButtonBox["New in 6.0: Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicalFunctions"]], "MoreAbout",
 CellID->92669646]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"KelvinBei - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 51, 44.1537362}", "context" -> "System`", 
    "keywords" -> {"Kelvin bei", "Kelvin functions"}, "index" -> True, 
    "label" -> "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "KelvinBei[z] gives the Kelvin function bei(z). KelvinBei[n, z] gives the \
Kelvin function bei_n (z). ", "synonyms" -> {"kelvin bei"}, "title" -> 
    "KelvinBei", "type" -> "Symbol", "uri" -> "ref/KelvinBei"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[8553, 281, 359, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->79473039]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 32600, 1134}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2126, 51, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2749, 78, 1574, 37, 70, "ObjectNameGrid"],
Cell[4326, 117, 1081, 36, 70, "Usage",
 CellID->15419]
}, Open  ]],
Cell[CellGroupData[{
Cell[5444, 158, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[5922, 175, 120, 4, 70, "Notes",
 CellID->20640],
Cell[6045, 181, 771, 25, 70, "Notes",
 CellID->19040],
Cell[6819, 208, 568, 21, 70, "Notes",
 CellID->24388],
Cell[7390, 231, 444, 17, 70, "Notes",
 CellID->83559840],
Cell[7837, 250, 250, 8, 70, "Notes",
 CellID->792533805],
Cell[8090, 260, 220, 7, 70, "Notes",
 CellID->31226],
Cell[8313, 269, 203, 7, 70, "Notes",
 CellID->28162]
}, Closed]],
Cell[CellGroupData[{
Cell[8553, 281, 359, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->79473039],
Cell[CellGroupData[{
Cell[8937, 295, 148, 5, 70, "ExampleSection",
 CellID->263300735],
Cell[9088, 302, 63, 1, 70, "ExampleText",
 CellID->64203606],
Cell[CellGroupData[{
Cell[9176, 307, 106, 3, 28, "Input",
 CellID->25647115],
Cell[9285, 312, 176, 5, 36, "Output",
 CellID->292417452]
}, Open  ]],
Cell[9476, 320, 125, 3, 70, "ExampleDelimiter",
 CellID->110621565],
Cell[9604, 325, 191, 9, 70, "ExampleText",
 CellID->283017315],
Cell[CellGroupData[{
Cell[9820, 338, 220, 7, 28, "Input",
 CellID->273567397],
Cell[10043, 347, 2193, 40, 121, "Output",
 Evaluatable->False,
 CellID->783022416]
}, Open  ]],
Cell[12251, 390, 125, 3, 70, "ExampleDelimiter",
 CellID->127075355],
Cell[12379, 395, 62, 1, 70, "ExampleText",
 CellID->4255939],
Cell[CellGroupData[{
Cell[12466, 400, 222, 7, 70, "Input",
 CellID->258655594],
Cell[12691, 409, 795, 26, 54, "Output",
 CellID->5856940]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[13535, 441, 216, 7, 70, "ExampleSection",
 CellID->22176],
Cell[13754, 450, 83, 1, 70, "ExampleText",
 CellID->2708615],
Cell[CellGroupData[{
Cell[13862, 455, 172, 6, 70, "Input",
 CellID->17954271],
Cell[14037, 463, 272, 7, 36, "Output",
 CellID->379120653]
}, Open  ]],
Cell[14324, 473, 125, 3, 70, "ExampleDelimiter",
 CellID->464533498],
Cell[14452, 478, 69, 1, 70, "ExampleText",
 CellID->66248962],
Cell[CellGroupData[{
Cell[14546, 483, 184, 6, 70, "Input",
 CellID->170986264],
Cell[14733, 491, 225, 6, 36, "Output",
 CellID->26921967]
}, Open  ]],
Cell[14973, 500, 125, 3, 70, "ExampleDelimiter",
 CellID->295287149],
Cell[15101, 505, 112, 3, 70, "ExampleText",
 CellID->29695085],
Cell[CellGroupData[{
Cell[15238, 512, 168, 5, 70, "Input",
 CellID->465676590],
Cell[15409, 519, 229, 6, 36, "Output",
 CellID->18382165]
}, Open  ]],
Cell[15653, 528, 125, 3, 70, "ExampleDelimiter",
 CellID->221466515],
Cell[15781, 533, 203, 7, 70, "ExampleText",
 CellID->231345657],
Cell[CellGroupData[{
Cell[16009, 544, 166, 5, 70, "Input",
 CellID->12842311],
Cell[16178, 551, 256, 8, 50, "Output",
 CellID->895685377]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[16483, 565, 239, 7, 70, "ExampleSection",
 CellID->22005],
Cell[16725, 574, 212, 7, 70, "ExampleText",
 CellID->90913756],
Cell[CellGroupData[{
Cell[16962, 585, 238, 8, 70, "Input",
 CellID->221683768],
Cell[17203, 595, 896, 32, 54, "Output",
 CellID->191359368]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[18148, 633, 221, 7, 70, "ExampleSection",
 CellID->429],
Cell[18372, 642, 82, 1, 70, "ExampleText",
 CellID->127419602],
Cell[CellGroupData[{
Cell[18479, 647, 1487, 49, 70, "Input",
 CellID->61770610],
Cell[19969, 698, 813, 26, 54, "Output",
 CellID->241143927]
}, Open  ]],
Cell[20797, 727, 125, 3, 70, "ExampleDelimiter",
 CellID->809257294],
Cell[20925, 732, 145, 4, 70, "ExampleText",
 CellID->146335090],
Cell[CellGroupData[{
Cell[21095, 740, 892, 29, 70, "Input",
 CellID->673371864],
Cell[21990, 771, 2165, 40, 70, "Output",
 Evaluatable->False,
 CellID->755197575]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[24204, 817, 233, 7, 70, "ExampleSection",
 CellID->29619],
Cell[24440, 826, 246, 8, 70, "ExampleText",
 CellID->540949068],
Cell[CellGroupData[{
Cell[24711, 838, 208, 7, 70, "Input",
 CellID->275163834],
Cell[24922, 847, 691, 23, 71, "Output",
 CellID->258721107]
}, Open  ]],
Cell[CellGroupData[{
Cell[25650, 875, 107, 3, 70, "Input",
 CellID->71003703],
Cell[25760, 880, 418, 15, 55, "Output",
 CellID->213987913]
}, Open  ]],
Cell[26193, 898, 125, 3, 70, "ExampleDelimiter",
 CellID->545301568],
Cell[26321, 903, 249, 8, 70, "ExampleText",
 CellID->127473317],
Cell[CellGroupData[{
Cell[26595, 915, 195, 6, 70, "Input",
 CellID->788709929],
Cell[26793, 923, 883, 34, 83, "Output",
 CellID->476394900]
}, Open  ]],
Cell[27691, 960, 125, 3, 70, "ExampleDelimiter",
 CellID->370746783],
Cell[27819, 965, 92, 1, 70, "ExampleText",
 CellID->122712053],
Cell[CellGroupData[{
Cell[27936, 970, 344, 12, 70, "Input",
 CellID->586709475],
Cell[28283, 984, 938, 32, 72, "Output",
 CellID->830901745]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[29282, 1023, 311, 9, 70, "SeeAlsoSection",
 CellID->18830],
Cell[29596, 1034, 697, 26, 70, "SeeAlso",
 CellID->5281]
}, Open  ]],
Cell[CellGroupData[{
Cell[30330, 1065, 314, 9, 70, "TutorialsSection",
 CellID->25938],
Cell[30647, 1076, 144, 3, 70, "Tutorials",
 CellID->29228]
}, Open  ]],
Cell[CellGroupData[{
Cell[30828, 1084, 319, 9, 70, "MoreAboutSection",
 CellID->167386683],
Cell[31150, 1095, 158, 3, 70, "MoreAbout",
 CellID->113773963],
Cell[31311, 1100, 173, 3, 70, "MoreAbout",
 CellID->92669646]
}, Open  ]],
Cell[31499, 1106, 27, 0, 70, "History"],
Cell[31529, 1108, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

