(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     93556,       2588]
NotebookOptionsPosition[     84031,       2257]
NotebookOutlinePosition[     85591,       2299]
CellTagsIndexPosition[     85505,       2294]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Solving Linear Systems" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/SolvingLinearSystems"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Solving Linear Systems\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"LinearSolve" :> 
          Documentation`HelpLookup["paclet:ref/LinearSolve"], "PseudoInverse" :> 
          Documentation`HelpLookup["paclet:ref/PseudoInverse"], "Fit" :> 
          Documentation`HelpLookup["paclet:ref/Fit"], "FindFit" :> 
          Documentation`HelpLookup["paclet:ref/FindFit"], 
          "SingularValueDecomposition" :> 
          Documentation`HelpLookup["paclet:ref/SingularValueDecomposition"], 
          "QRDecomposition" :> 
          Documentation`HelpLookup["paclet:ref/QRDecomposition"], 
          "CoefficientArrays" :> 
          Documentation`HelpLookup["paclet:ref/CoefficientArrays"], 
          "NMinimize" :> Documentation`HelpLookup["paclet:ref/NMinimize"], 
          "FindMinimum" :> 
          Documentation`HelpLookup["paclet:ref/FindMinimum"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"LinearSolve\"\>", 
       2->"\<\"PseudoInverse\"\>", 3->"\<\"Fit\"\>", 4->"\<\"FindFit\"\>", 
       5->"\<\"SingularValueDecomposition\"\>", 6->"\<\"QRDecomposition\"\>", 
       7->"\<\"CoefficientArrays\"\>", 8->"\<\"NMinimize\"\>", 
       9->"\<\"FindMinimum\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Curve Fitting & Approximate Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/CurveFittingAndApproximateFunctions"], 
          "Linear Systems" :> 
          Documentation`HelpLookup["paclet:guide/LinearSystems"], 
          "Matrices and Linear Algebra" :> 
          Documentation`HelpLookup["paclet:guide/MatricesAndLinearAlgebra"], 
          "Matrix-Based Minimization" :> 
          Documentation`HelpLookup["paclet:guide/MatrixBasedMinimization"], 
          "Optimization" :> 
          Documentation`HelpLookup["paclet:guide/Optimization"], 
          "Signal Processing" :> 
          Documentation`HelpLookup["paclet:guide/SignalProcessing"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"], 
          "New in 6.0: Matrix & Linear Algebra Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MatrixAndLinearAlgebraFunctions"], 
          "New in 6.0: Numerical Data Handling" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60NumericalDataHandling"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Curve Fitting & Approximate Functions\"\>", 
       2->"\<\"Linear Systems\"\>", 3->"\<\"Matrices and Linear Algebra\"\>", 
       4->"\<\"Matrix-Based Minimization\"\>", 5->"\<\"Optimization\"\>", 
       6->"\<\"Signal Processing\"\>", 
       7->"\<\"New in 6.0: Mathematics & Algorithms\"\>", 
       8->"\<\"New in 6.0: Matrix & Linear Algebra Functions\"\>", 
       9->"\<\"New in 6.0: Numerical Data Handling\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["LeastSquares", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["LeastSquares",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/LeastSquares"], "[", 
       RowBox[{
        StyleBox["m", "TI"], ",", 
        StyleBox["b", "TI"]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]finds an ",
     Cell[BoxData[
      StyleBox["x", "TI"]], "InlineFormula"],
     " which solves the linear least-squares problem for the matrix equation \
",
     Cell[BoxData[
      RowBox[{
       RowBox[{
        StyleBox["m", "TI"], ".", 
        StyleBox["x", "TI"]}], "==", 
       StyleBox["b", "TI"]}]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->11929]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["LeastSquares",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/LeastSquares"], "[", 
   RowBox[{
    StyleBox["m", "TI"], ",", 
    StyleBox["b", "TI"]}], "]"}]], "InlineFormula"],
 " gives a vector ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " which minimizes ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Norm",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Norm"], "[", 
   RowBox[{
    RowBox[{
     StyleBox["m", "TI"], ".", 
     StyleBox["x", "TI"]}], "-", 
    StyleBox["b", "TI"]}], "]"}]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->6427],

Cell[TextData[{
 "The vector ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " is uniquely determined by the minimization only if ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    ButtonBox["Length",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Length"], "[", 
    StyleBox["x", "TI"], "]"}], "==", 
   RowBox[{
    ButtonBox["MatrixRank",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/MatrixRank"], "[", 
    StyleBox["m", "TI"], "]"}]}]], "InlineFormula"],
 "."
}], "Notes",
 CellID->21903],

Cell[TextData[{
 "The argument ",
 Cell[BoxData[
  StyleBox["b", "TI"]], "InlineFormula"],
 " can be a matrix, in which case the least-squares minimization is done \
independently for each column in ",
 Cell[BoxData[
  StyleBox["b", "TI"]], "InlineFormula"],
 "."
}], "Notes",
 CellID->5176],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["LeastSquares",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LeastSquares"]], "InlineFormula"],
 " works on both numerical and symbolic matrices, as well as ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 " objects. "
}], "Notes",
 CellID->1120],

Cell[TextData[{
 "A ",
 Cell[BoxData[
  ButtonBox["Method",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Method"]], "InlineFormula"],
 " option can also be given. Settings for arbitrary-precision numerical \
matrices include ",
 Cell[BoxData["\"\<Direct\>\""], "InlineFormula"],
 " and ",
 Cell[BoxData["\"\<IterativeRefinement\>\""], "InlineFormula"],
 ", and for sparse arrays ",
 Cell[BoxData["\"\<Direct\>\""], "InlineFormula"],
 " and ",
 Cell[BoxData["\"\<Krylov\>\""], "InlineFormula"],
 ". The default setting of ",
 Cell[BoxData[
  ButtonBox["Automatic",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
 " switches between these methods depending on the matrix given. "
}], "Notes",
 CellID->22255]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->604415620],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(1)", "ExampleCount"]
}], "ExampleSection",
 CellID->527231416],

Cell["Solve a simple least-squares problem:", "ExampleText",
 CellID->13491],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LeastSquares", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "3"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"7", ",", "7", ",", "8"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->27830],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["19", "3"], ",", 
   FractionBox["1", "2"]}], "}"}]], "Output",
 ImageSize->{59, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->176376629]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->14781],

Cell["Use symbolic input:", "ExampleText",
 CellID->27465],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LeastSquares", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "3"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"7", ",", "7", ",", "x"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19659],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    FractionBox["35", "3"], "-", 
    FractionBox[
     RowBox[{"2", " ", "x"}], "3"]}], ",", 
   RowBox[{
    RowBox[{"-", 
     FractionBox["7", "2"]}], "+", 
    FractionBox["x", "2"]}]}], "}"}]], "Output",
 ImageSize->{124, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->11467411]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->120452940],

Cell[TextData[{
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 " is a 3\[Times]4 matrix and ",
 Cell[BoxData[
  StyleBox["b", "TI"]], "InlineFormula"],
 " is a length 4 vector:"
}], "ExampleText",
 CellID->56784250],

Cell[BoxData[{
 RowBox[{
  RowBox[{"m", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "2", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4", ",", "5", ",", "6"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"7", ",", "8", ",", "9"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"10", ",", "11", ",", "12"}], "}"}]}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"b", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "4", ",", "8"}], "}"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->174043150],

Cell[TextData[{
 "Use exact arithmetic to find a vector ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " that minimizes ",
 Cell[BoxData[
  FormBox[
   SubscriptBox[
    RowBox[{"\[LeftDoubleBracketingBar]", 
     RowBox[{
      RowBox[{"m", ".", "x"}], "-", "b"}], "\[RightDoubleBracketingBar]"}], 
    "2"], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->312376863],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LeastSquares", "[", 
  RowBox[{"m", ",", "b"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->274623602],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["157", "180"], ",", 
   FractionBox["23", "90"], ",", 
   RowBox[{"-", 
    FractionBox["13", "36"]}]}], "}"}]], "Output",
 ImageSize->{110, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->633443637]
}, Open  ]],

Cell["Use machine arithmetic:", "ExampleText",
 CellID->593509987],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LeastSquares", "[", 
  RowBox[{
   RowBox[{"N", "[", "m", "]"}], ",", " ", "b"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->441020697],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.8722222222222215`", ",", "0.25555555555555565`", ",", 
   RowBox[{"-", "0.36111111111111066`"}]}], "}"}]], "Output",
 ImageSize->{217, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->114081428]
}, Open  ]],

Cell["Use 20-digit precision arithmetic:", "ExampleText",
 CellID->26863089],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LeastSquares", "[", 
  RowBox[{
   RowBox[{"N", "[", 
    RowBox[{"m", ",", "20"}], "]"}], ",", " ", "b"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->122665625],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.87222222222222222222222222222092600714`20.", ",", 
   "0.25555555555555555555555555555318718654`20.", ",", 
   RowBox[{"-", "0.36111111111111111111111111110739844932`20."}]}], 
  "}"}]], "Output",
 ImageSize->{511, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->81608291]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->522607642],

Cell["Solve the least-squares problem for a random complex matrix:", \
"ExampleText",
 CellID->26582841],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LeastSquares", "[", 
  RowBox[{
   RowBox[{"RandomComplex", "[", 
    RowBox[{
     RowBox[{"1", " ", "+", " ", "I"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"4", ",", "7"}], "}"}]}], "]"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"1", ",", "1", ",", "1", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->993202514],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"0.1835553882856915`", "\[InvisibleSpace]", "-", 
    RowBox[{"0.13298990703752786`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{"0.19085573754384536`", "\[InvisibleSpace]", "-", 
    RowBox[{"0.28215650156303307`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{"0.013701590632593571`", "\[InvisibleSpace]", "-", 
    RowBox[{"0.20656345602177711`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{"0.20881715492770345`", "\[InvisibleSpace]", "-", 
    RowBox[{"0.1466381764319698`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{"0.13773003214716556`", "\[InvisibleSpace]", "+", 
    RowBox[{"0.05577910882216154`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{"0.06983933935771376`", "\[InvisibleSpace]", "-", 
    RowBox[{"0.05572186228128501`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "0.034241722046252074`"}], "-", 
    RowBox[{"0.24331802099860622`", " ", "\[ImaginaryI]"}]}]}], 
  "}"}]], "Output",
 ImageSize->{455, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->362601567]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->53301542],

Cell["Use a sparse matrix:", "ExampleText",
 CellID->811714018],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"s", " ", "=", " ", 
   RowBox[{"SparseArray", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"{", 
         RowBox[{"i_", ",", " ", "j_"}], "}"}], " ", "/;", " ", 
        RowBox[{"j", "\[Equal]", 
         RowBox[{"i", "+", "1"}]}]}], "\[Rule]", 
       RowBox[{"N", "[", "i", "]"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"10", ",", " ", "10"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"LeastSquares", "[", 
  RowBox[{
   RowBox[{"N", "[", "s", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "1", ",", "2", ",", "3", ",", "4", ",", "5", ",", "6", ",", "7", ",", "8",
      ",", "9", ",", "10"}], "}"}]}], "]"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->234977969],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.`", ",", "1.`", ",", "0.9999999999999999`", ",", "1.`", ",", "1.`", ",", 
   "0.9999999999999999`", ",", "1.`", ",", "1.`", ",", "1.`", ",", "1.`"}], 
  "}"}]], "Output",
 ImageSize->{259, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->601140228]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->5256],

Cell[TextData[{
 Cell[BoxData[
  StyleBox["b", "TI"]], "InlineFormula"],
 " can be a matrix:"
}], "ExampleText",
 CellID->116368703],

Cell[BoxData[
 RowBox[{
  RowBox[{"b", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"7", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"7", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"8", ",", "3"}], "}"}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->194601242],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LeastSquares", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "3"}], "}"}]}], "}"}], ",", "b"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->215002914],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     FractionBox["19", "3"], ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     FractionBox["1", "2"], ",", "1"}], "}"}]}], "}"}]], "Output",
 ImageSize->{123, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->324243934]
}, Open  ]],

Cell[TextData[{
 "The first column of the ",
 Cell[BoxData[
  StyleBox["b", "TI"]], "InlineFormula"],
 " matrix is used to generate the first column of the result:"
}], "ExampleText",
 CellID->495184610],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LeastSquares", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "3"}], "}"}]}], "}"}], ",", 
   RowBox[{"b", "[", 
    RowBox[{"[", 
     RowBox[{"All", ",", "1"}], "]"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->180347153],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["19", "3"], ",", 
   FractionBox["1", "2"]}], "}"}]], "Output",
 ImageSize->{59, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->185444001]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Options",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->282792129],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Tolerance",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->73904260],

Cell[TextData[{
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 " is a 20\[Times]20 Hilbert matrix and ",
 Cell[BoxData[
  StyleBox["b", "TI"]], "InlineFormula"],
 " is a vector such that the solution of ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    StyleBox["m", "TI"], ".", 
    StyleBox["x", "TI"]}], " ", "=", " ", 
   StyleBox["b", "TI"]}]], "InlineFormula"],
 " is known:"
}], "ExampleText",
 CellID->34283710],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"HilbertMatrix", "[", "20", "]"}]}], ";", 
  RowBox[{"b", "=", 
   RowBox[{"m", ".", 
    RowBox[{"ConstantArray", "[", 
     RowBox[{"1", ",", "20"}], "]"}]}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->112506301],

Cell["\<\
With the default tolerance, numerical roundoff is limited so errors are \
distributed:\
\>", "ExampleText",
 CellID->53502523],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"x", "=", 
  RowBox[{"LeastSquares", "[", 
   RowBox[{
    RowBox[{"N", "[", "m", "]"}], ",", "b"}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->155915323],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.9999999998092332`", ",", "1.000000014964043`", ",", 
   "0.9999997096147906`", ",", "1.0000023327618914`", ",", 
   "0.9999909255400236`", ",", "1.000016789334973`", ",", 
   "0.9999906300690925`", ",", "0.9999901016097482`", ",", 
   "1.0000056412889637`", ",", "1.000010691600367`", ",", 
   "1.000002552264348`", ",", "0.9999928245527323`", ",", 
   "0.9999908506885512`", ",", "0.9999971753997002`", ",", 
   "1.0000057134359275`", ",", "1.0000092180066953`", ",", 
   "1.0000042371755116`", ",", "0.999994015626889`", ",", 
   "0.9999892589322142`", ",", "1.0000073174256925`"}], "}"}]], "Output",
 ImageSize->{549, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->570014100]
}, Open  ]],

Cell[TextData[{
 "With ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Tolerance",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Tolerance"], "->", "0"}]], "InlineFormula"],
 ", numerical roundoff can introduce excessive error:"
}], "ExampleText",
 CellID->151699388],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"x0", "=", 
  RowBox[{"LeastSquares", "[", 
   RowBox[{
    RowBox[{"N", "[", "m", "]"}], ",", "b", ",", 
    RowBox[{"Tolerance", "\[Rule]", "0"}]}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->167441092],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1.0000018189521132`", ",", "0.9997896307294343`", ",", 
   "1.004915268297438`", ",", "0.979362045377577`", ",", 
   "0.5576850964635021`", ",", "7.045529598799857`", ",", 
   RowBox[{"-", "31.376234902852538`"}], ",", "85.23269920367379`", ",", 
   RowBox[{"-", "68.90115898982904`"}], ",", 
   RowBox[{"-", "173.65449400835706`"}], ",", "506.29844541139624`", ",", 
   RowBox[{"-", "357.9850368253945`"}], ",", 
   RowBox[{"-", "265.481009944776`"}], ",", "411.3547060262649`", ",", 
   "89.17976380211104`", ",", 
   RowBox[{"-", "23.272865787487895`"}], ",", 
   RowBox[{"-", "776.8157948669577`"}], ",", "1068.0780381616357`", ",", 
   RowBox[{"-", "568.2793593288253`"}], ",", "114.03502031766669`"}], 
  "}"}]], "Output",
 ImageSize->{467, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->248223234]
}, Open  ]],

Cell["\<\
Specifying a higher tolerance will limit roundoff errors at the expense of a \
larger residual:\
\>", "ExampleText",
 CellID->22816417],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"x1", "=", 
  RowBox[{"LeastSquares", "[", 
   RowBox[{
    RowBox[{"N", "[", "m", "]"}], ",", "b", ",", 
    RowBox[{"Tolerance", "\[Rule]", 
     SuperscriptBox["10", 
      RowBox[{"-", "8"}]]}]}], "]"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->174669911],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.9999997941350756`", ",", "1.000007907947448`", ",", 
   "0.9999318635617397`", ",", "1.000198048677443`", ",", 
   "0.9998575989513127`", ",", "0.9998550626627715`", ",", 
   "1.000028651777173`", ",", "1.0001441285579062`", ",", 
   "1.000135434078639`", ",", "1.0000432886512278`", ",", "0.99993572658515`",
    ",", "0.9998656501516787`", ",", "0.9998576520227442`", ",", 
   "0.9999097469618662`", ",", "1.0000005096852849`", ",", 
   "1.0000968404933523`", ",", "1.0001604436126499`", ",", 
   "1.0001525397049833`", ",", "1.0000369484246996`", ",", 
   "0.9997818784134389`"}], "}"}]], "Output",
 ImageSize->{450, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->175117973]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"Norm", "[", 
    RowBox[{
     RowBox[{"m", ".", "x"}], "-", "b"}], "]"}], ",", 
   RowBox[{"Norm", "[", 
    RowBox[{
     RowBox[{"m", ".", "x1"}], "-", "b"}], "]"}]}], "}"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->191459788],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"7.928578270080922`*^-16", ",", "7.379074840534493`*^-12"}], 
  "}"}]], "Output",
 ImageSize->{209, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->251299560]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->25767],

Cell["Here is some data:", "ExampleText",
 CellID->117238259],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"x", " ", "=", " ", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"10", ",", " ", "1000"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"y", " ", "=", " ", 
   RowBox[{
    RowBox[{"Sin", "[", "x", "]"}], "  ", 
    RowBox[{"RandomReal", "[", 
     RowBox[{"1", ",", "1000"}], "]"}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"data", " ", "=", " ", 
   RowBox[{"Transpose", "[", 
    RowBox[{"{", 
     RowBox[{"x", ",", "y"}], "}"}], "]"}]}], ";"}], "\n", 
 RowBox[{"ListPlot", "[", "data", "]"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->79982441],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 102},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->25588907]
}, Open  ]],

Cell[TextData[{
 "Define cubic basis functions centered at ",
 Cell[BoxData[
  StyleBox["t", "TI"]], "InlineFormula"],
 " with support on the interval ",
 Cell[BoxData[
  RowBox[{"[", 
   RowBox[{
    RowBox[{
     StyleBox["t", "TI"], " ", "-", " ", "2"}], ",", " ", 
    RowBox[{
     StyleBox["t", "TI"], " ", "+", " ", "2"}]}], "]"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->692511576],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"b", "[", "t_", "]"}], "[", "r_", "]"}], " ", "=", " ", 
   RowBox[{"Block", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"s", " ", "=", " ", 
        RowBox[{"r", " ", "-", " ", "t"}]}], ",", "sp1"}], "}"}], ",", 
     RowBox[{"Piecewise", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"0", ",", " ", 
          RowBox[{"s", " ", "\[LessEqual]", " ", 
           RowBox[{"-", "2"}]}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{
           RowBox[{"(", 
            RowBox[{"2", " ", "+", " ", "s"}], ")"}], "^", "3"}], ",", 
          RowBox[{
           RowBox[{"-", "2"}], " ", "<", " ", "s", " ", "\[LessEqual]", " ", 
           RowBox[{"-", "1"}]}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"1", " ", "+", " ", 
           RowBox[{"3", 
            RowBox[{"(", 
             RowBox[{"1", "+", "s"}], ")"}], 
            RowBox[{"(", 
             RowBox[{"1", " ", "-", " ", 
              RowBox[{"s", 
               RowBox[{"(", 
                RowBox[{"1", "+", "s"}], ")"}]}]}], ")"}]}]}], ",", 
          RowBox[{
           RowBox[{"-", "1"}], " ", "<", " ", "s", " ", "<", "0"}]}], "}"}], 
        ",", " ", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"1", " ", "+", " ", 
           RowBox[{"3", 
            RowBox[{"(", 
             RowBox[{"1", "-", "s"}], ")"}], " ", 
            RowBox[{"(", 
             RowBox[{"1", " ", "+", " ", 
              RowBox[{"s", "*", 
               RowBox[{"(", 
                RowBox[{"1", "-", "s"}], ")"}]}]}], ")"}]}]}], ",", 
          RowBox[{"0", " ", "<=", " ", "s", " ", "<", "1"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{
           RowBox[{"(", 
            RowBox[{"2", " ", "-", " ", "s"}], ")"}], "^", "3"}], ",", 
          RowBox[{"1", "\[LessEqual]", " ", "s", " ", "<", " ", "2"}]}], 
         "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"0", ",", 
          RowBox[{"s", " ", "\[GreaterEqual]", "  ", "2"}]}], "}"}]}], "}"}], 
      "]"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->3010787],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"b", "[", "0", "]"}], "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "2"}], ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->777638514],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 127},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->351073543]
}, Open  ]],

Cell["\<\
Set up a sparse design matrix for basis functions centered at 0, 1, ..., 10:\
\>", "ExampleText",
 CellID->21418062],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"s", " ", "=", " ", 
  RowBox[{"SparseArray", "[", 
   RowBox[{"Flatten", "[", "\[IndentingNewLine]", 
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{"Table", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"{", 
          RowBox[{"i", ",", " ", 
           RowBox[{"t", "+", "1"}]}], "}"}], "->", 
         RowBox[{
          RowBox[{"b", "[", "t", "]"}], "[", 
          RowBox[{"x", "[", 
           RowBox[{"[", "i", "]"}], "]"}], "]"}]}], ",", " ", 
        RowBox[{"{", 
         RowBox[{"t", ",", " ", 
          RowBox[{"Max", "[", 
           RowBox[{"0", ",", " ", 
            RowBox[{"Floor", "[", 
             RowBox[{
              RowBox[{"x", "[", 
               RowBox[{"[", "i", "]"}], "]"}], " ", "-", " ", "2"}], "]"}]}], 
           "]"}], ",", 
          RowBox[{"Min", "[", 
           RowBox[{"10", ",", " ", 
            RowBox[{"Ceiling", "[", 
             RowBox[{
              RowBox[{"x", "[", 
               RowBox[{"[", "i", "]"}], "]"}], " ", "+", " ", "2"}], "]"}]}], 
           "]"}]}], "}"}]}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"i", ",", 
        RowBox[{"Length", "[", "x", "]"}]}], "}"}]}], "]"}], "]"}], 
   "]"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->236955778],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "3802", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 3802, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"1000", ",", "11"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{216, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->98721170]
}, Open  ]],

Cell["Solve the least-squares problem:", "ExampleText",
 CellID->132830414],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"c", " ", "=", " ", 
  RowBox[{"LeastSquares", "[", 
   RowBox[{"s", ",", " ", "y"}], "]"}]}]], "Input",
 CellLabel->"In[5]:=",
 CellID->636202219],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "0.015761027638730794`"}], ",", "0.09613366766985312`", ",", 
   "0.08777754602829933`", ",", "0.013039988602631754`", ",", 
   RowBox[{"-", "0.07939051355828161`"}], ",", 
   RowBox[{"-", "0.0923856842281286`"}], ",", 
   RowBox[{"-", "0.02842059097721398`"}], ",", "0.07036580185877078`", ",", 
   "0.09873934829426786`", ",", "0.046685348804640314`", ",", 
   RowBox[{"-", "0.06730689842928876`"}]}], "}"}]], "Output",
 ImageSize->{475, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->67822912]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", 
  RowBox[{
   RowBox[{"ListPlot", "[", "data", "]"}], ",", 
   RowBox[{"Plot", "[", 
    RowBox[{
     RowBox[{"c", ".", 
      RowBox[{"Table", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"b", "[", "t", "]"}], "[", "x", "]"}], ",", 
        RowBox[{"{", 
         RowBox[{"t", ",", "0", ",", "10"}], "}"}]}], "]"}]}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "10"}], "}"}], ",", " ", 
     RowBox[{"PlotStyle", "\[Rule]", "Red"}]}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->146702974],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 102},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->483937585]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->6774],

Cell[TextData[{
 "When ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    StyleBox["m", "TI"], ".", 
    StyleBox["x", "TI"]}], "==", 
   StyleBox["b", "TI"]}]], "InlineFormula"],
 " can be solved exactly, ",
 Cell[BoxData[
  ButtonBox["LeastSquares",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LeastSquares"]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  ButtonBox["LinearSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LinearSolve"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->16389],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LeastSquares", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "3"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"7", ",", "7", ",", "7"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->15889],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"7", ",", "0"}], "}"}]], "Output",
 ImageSize->{44, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->82942956]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->42631810],

Cell[TextData[{
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 " is a 5\[Times]2 matrix and ",
 Cell[BoxData[
  StyleBox["b", "TI"]], "InlineFormula"],
 " is a length 5 vector:"
}], "ExampleText",
 CellID->735307411],

Cell[BoxData[{
 RowBox[{
  RowBox[{"m", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "4"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "5"}], "}"}]}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"b", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "4", ",", "3", ",", " ", "5"}], "}"}]}], 
  ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->7259997],

Cell["Solve the least-squares problem:", "ExampleText",
 CellID->4753688],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"a", " ", "=", " ", 
  RowBox[{"LeastSquares", "[", 
   RowBox[{"m", ",", "b"}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->322065917],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["3", "10"], ",", 
   FractionBox["9", "10"]}], "}"}]], "Output",
 ImageSize->{66, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->255041252]
}, Open  ]],

Cell[TextData[{
 "This is the minimizer of ",
 Cell[BoxData[
  FormBox[
   SubscriptBox[
    RowBox[{"\[LeftDoubleBracketingBar]", 
     RowBox[{
      RowBox[{"m", ".", "x"}], "-", "b"}], "\[RightDoubleBracketingBar]"}], 
    "2"], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->231560046],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", 
  RowBox[{
   RowBox[{"Plot3D", "[", 
    RowBox[{
     RowBox[{"Norm", "[", 
      RowBox[{
       RowBox[{"m", ".", 
        RowBox[{"{", 
         RowBox[{"x", ",", "y"}], "}"}]}], " ", "-", " ", "b"}], "]"}], ",", 
     " ", 
     RowBox[{"{", 
      RowBox[{"x", ",", 
       RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"y", ",", "0", ",", "2"}], "}"}], ",", " ", 
     RowBox[{"MeshFunctions", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{"#3", "&"}], "}"}]}], ",", " ", 
     RowBox[{"ColorFunction", "\[Rule]", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"Hue", "[", 
         RowBox[{".65", " ", 
          RowBox[{"(", 
           RowBox[{"1", "-", "#3"}], ")"}]}], "]"}], "&"}], ")"}]}]}], "]"}], 
   ",", " ", 
   RowBox[{"Graphics3D", "[", 
    RowBox[{"{", 
     RowBox[{"Red", ",", " ", 
      RowBox[{"PointSize", "[", "0.05", "]"}], ",", " ", 
      RowBox[{"Point", "[", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"a", "[", 
          RowBox[{"[", "1", "]"}], "]"}], ",", " ", 
         RowBox[{"a", "[", 
          RowBox[{"[", "2", "]"}], "]"}], ",", " ", 
         RowBox[{"Norm", "[", 
          RowBox[{
           RowBox[{"m", ".", "a"}], " ", "-", " ", "b"}], "]"}]}], "}"}], 
       "]"}]}], "}"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->20408039],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 144},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->324960660]
}, Open  ]],

Cell["\<\
It is also gives the coefficients for the line with least-squares distance to \
the points:\
\>", "ExampleText",
 CellID->91489318],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", 
  RowBox[{
   RowBox[{"ListPlot", "[", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"1", ",", "1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"2", ",", "2"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"3", ",", "4"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"4", ",", "3"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"5", ",", "5"}], "}"}]}], "}"}], "]"}], ",", " ", 
   RowBox[{"Plot", "[", 
    RowBox[{
     RowBox[{"a", ".", 
      RowBox[{"{", 
       RowBox[{"1", ",", "x"}], "}"}]}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"x", ",", "1", ",", "5"}], "}"}]}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->148203188],

Cell[GraphicsData["CompressedBitmap", "\<\
eJyVWH9oVmUUvvvu/VxW/sD8iSVDttVQa42yRGylpqYMs7RIG/I1lW/RXLpZ
sZIcUalMSYfoYkzNLLSIiWIEBpEQCZEQCVJBKIj9kQRKfwyC2/Oec9973x9n
+z7H9n33fd/nnPuc5z3nvPeuqdBZ3NhW6GxtKVQt3Vp4vdja0lG1pH0rpsKK
IAjO4G/71ADXMa7o70wc49v8fAifEZZq8R3iu07NqouYfoLqZD5P6HpjNRDQ
tc74fmecI7uHndkHhHsqTu48Wzc4szX4yGFo3kviJlu7jGsNb7k0bqVSEGUo
37vEgoxS3fSqpLK+a+Ttzx0O8sQIs5U0e/zTi1jY03M+OEmzKoKvMFNhYC78
eBUzX5+9HExxPI11xpNLrOdvc31UiXX3fu567rZVSWeHhv6zZ5XiQXj++z8T
/OcZPo+V69dvvt31Tfvm05d+/ctcq0zvcPHna63FQYAADY6LHIhvhHu8uOaz
3v0/3Ljx7zERRzsffXHyl+Z1J7CD4HrU4km6RbduDfUdugAIHB4R/dD+Rkwd
3EB9QMSNJhyW0wD6RdxdHv+PRdwYj/8hi/84mz8S8KDoZ7zmv63zLKT//be/
D4i4CYTDBrS88mX3jnPg1SviJhLu23N/gD9uDQL7RNwkwoE2yAOKQD6y+E/V
/CEC1iHIXtHPNEN/5t8j4qZ7/HcTLtJdwMDeZ8RweOAnkNhJ2KRdBMbAsJqh
I4IJFEdEjECAjQv6VH/YQYY5xy6MuUdVkb0SVkfMSFzD/ulF/Yl9KGW8MZ5J
aWkWEzYfF6r7lOehWmuFXIVWcNWdhGx2ZDf+Wq0agkcIUO1dBob8Feju7p5X
eWgGtRAnJ3KwXdzBOiOjsYkI6h0rY2bT+pUr/yCT8YtM6BL9zCEczOEEuwLB
3xJxfAadGryEeLClIPaGiKvXvABCDJB5m4hr0Pv74QffMf8Oi/+jtA7aaSVu
Ef3M9TK5XcQ9bmQxV2KbiJun8xbqI1QE/JqIm6/5QzTcGgRetfg/ofmDPOtf
FP00evpvEnFPaf3TKtwg4hZq/lx3MGgRcYsN/tzRCxb/pTp/0k6yXvSzzNCf
+TeLuOVe/qwjXCjUwQo7h1ANa+NyKrVJx5T2DH7SyXrO7GH9qOHK2Oqe6BX8
PJX1HLYfqYZXaT24W4BMDSGyrsM+pL5njp838pX1eq4sDVYb+8/1t2pYnV/Q
eqU1uNLKgZd0Dqc12CTu7VovB1aIuJftHAbJZ0Rcsx0DDJaJuPVGDjP/JSKu
4MWx2Ipzg8Gf92yh6GejvSdI0CdF3CaDP599jSKuaOQr1+ACEUedxarF+Rb/
No//PNHPZi+nHhNx7V4OzRVxW7z8eUTEdRj64xeBNFj86SRRPRAicP7Ui37e
1PmT9pAHRRydYFn/gAGdciVrrsvLp1ll2T0b212mzo4XFzSTPl0pU7bJOsss
rUFaQ9R7AvOd0q1f9pF1ljmGPvzwUBOX6lfdWiscirzX1SVt3vP2fWZZOr3v
5UGVlQc7vTyYIe7vLi8P7hVxu+08QGzTRVyPHQ8KaZqI26v5Q29+a5ki4ujZ
PXtrAYFJVpz0rhCZD3n3iH56jb1BqCjcCSKO3lHUuYXix7ED/uNFHL3zqLeu
lP9YEdfn8b/b4t/v8b9T9NPv5dZoETfg6V8p4o4Y+nP+5EXcJ0YegT/4hRZ/
eldW9QoP6nKkt3rp3ExWnFk3+8tDHdNckt4RDmulhozOukZSpCPUKltkPcK2
UJ9ydLlEq0Ea+f+5Cyr+B8TWaAY=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 125},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->172639169]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->25343],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["LinearSolve",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/LinearSolve"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["PseudoInverse",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/PseudoInverse"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Fit",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Fit"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["FindFit",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FindFit"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SingularValueDecomposition",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SingularValueDecomposition"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["QRDecomposition",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/QRDecomposition"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["CoefficientArrays",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/CoefficientArrays"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["NMinimize",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NMinimize"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["FindMinimum",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FindMinimum"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->21821]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->16350],

Cell[TextData[ButtonBox["Solving Linear Systems",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/SolvingLinearSystems"]], "Tutorials",
 CellID->15515]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->202379032],

Cell[TextData[ButtonBox["Curve Fitting & Approximate Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/CurveFittingAndApproximateFunctions"]], "MoreAbout",\

 CellID->483002166],

Cell[TextData[ButtonBox["Linear Systems",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/LinearSystems"]], "MoreAbout",
 CellID->589262117],

Cell[TextData[ButtonBox["Matrices and Linear Algebra",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/MatricesAndLinearAlgebra"]], "MoreAbout",
 CellID->2667426],

Cell[TextData[ButtonBox["Matrix-Based Minimization",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/MatrixBasedMinimization"]], "MoreAbout",
 CellID->9279249],

Cell[TextData[ButtonBox["Optimization",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/Optimization"]], "MoreAbout",
 CellID->198737940],

Cell[TextData[ButtonBox["Signal Processing",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/SignalProcessing"]], "MoreAbout",
 CellID->622895022],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->556118210],

Cell[TextData[ButtonBox["New in 6.0: Matrix & Linear Algebra Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/NewIn60MatrixAndLinearAlgebraFunctions"]], "MoreAbout",
 CellID->340301978],

Cell[TextData[ButtonBox["New in 6.0: Numerical Data Handling",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60NumericalDataHandling"]], "MoreAbout",
 CellID->495788959]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"LeastSquares - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 52, 25.8266780}", "context" -> "System`", 
    "keywords" -> {
     "curve fitting", "Gauss-Markov theorem", "gradient descent", 
      "iterative refinement", "Krylov method", "least mean squares", 
      "least-squares", "LMS", "numerical optimization", 
      "ordinate differences", "regression analysis", "residuals", 
      "weighted least squares"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "LeastSquares[m, b] finds an x which solves the linear least-squares \
problem for the matrix equation m.x == b.", "synonyms" -> {"least squares"}, 
    "title" -> "LeastSquares", "type" -> "Symbol", "uri" -> 
    "ref/LeastSquares"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[10312, 305, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->604415620]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 85361, 2287}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 4150, 88, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[4773, 115, 1577, 37, 70, "ObjectNameGrid"],
Cell[6353, 154, 843, 28, 70, "Usage",
 CellID->11929]
}, Open  ]],
Cell[CellGroupData[{
Cell[7233, 187, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[7711, 204, 629, 25, 70, "Notes",
 CellID->6427],
Cell[8343, 231, 515, 19, 70, "Notes",
 CellID->21903],
Cell[8861, 252, 291, 10, 70, "Notes",
 CellID->5176],
Cell[9155, 264, 373, 12, 70, "Notes",
 CellID->1120],
Cell[9531, 278, 744, 22, 70, "Notes",
 CellID->22255]
}, Closed]],
Cell[CellGroupData[{
Cell[10312, 305, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->604415620],
Cell[CellGroupData[{
Cell[10697, 319, 148, 5, 70, "ExampleSection",
 CellID->527231416],
Cell[10848, 326, 76, 1, 70, "ExampleText",
 CellID->13491],
Cell[CellGroupData[{
Cell[10949, 331, 404, 14, 28, "Input",
 CellID->27830],
Cell[11356, 347, 248, 9, 51, "Output",
 CellID->176376629]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[11653, 362, 216, 7, 70, "ExampleSection",
 CellID->14781],
Cell[11872, 371, 58, 1, 70, "ExampleText",
 CellID->27465],
Cell[CellGroupData[{
Cell[11955, 376, 404, 14, 70, "Input",
 CellID->19659],
Cell[12362, 392, 391, 15, 51, "Output",
 CellID->11467411]
}, Open  ]],
Cell[12768, 410, 125, 3, 70, "ExampleDelimiter",
 CellID->120452940],
Cell[12896, 415, 226, 8, 70, "ExampleText",
 CellID->56784250],
Cell[13125, 425, 576, 18, 70, "Input",
 CellID->174043150],
Cell[13704, 445, 401, 14, 70, "ExampleText",
 CellID->312376863],
Cell[CellGroupData[{
Cell[14130, 463, 131, 4, 70, "Input",
 CellID->274623602],
Cell[14264, 469, 308, 11, 51, "Output",
 CellID->633443637]
}, Open  ]],
Cell[14587, 483, 66, 1, 70, "ExampleText",
 CellID->593509987],
Cell[CellGroupData[{
Cell[14678, 488, 165, 5, 70, "Input",
 CellID->441020697],
Cell[14846, 495, 289, 8, 36, "Output",
 CellID->114081428]
}, Open  ]],
Cell[15150, 506, 76, 1, 70, "ExampleText",
 CellID->26863089],
Cell[CellGroupData[{
Cell[15251, 511, 191, 6, 70, "Input",
 CellID->122665625],
Cell[15445, 519, 371, 11, 36, "Output",
 CellID->81608291]
}, Open  ]],
Cell[15831, 533, 125, 3, 70, "ExampleDelimiter",
 CellID->522607642],
Cell[15959, 538, 104, 2, 70, "ExampleText",
 CellID->26582841],
Cell[CellGroupData[{
Cell[16088, 544, 365, 11, 70, "Input",
 CellID->993202514],
Cell[16456, 557, 1096, 23, 71, "Output",
 CellID->362601567]
}, Open  ]],
Cell[17567, 583, 124, 3, 70, "ExampleDelimiter",
 CellID->53301542],
Cell[17694, 588, 63, 1, 70, "ExampleText",
 CellID->811714018],
Cell[CellGroupData[{
Cell[17782, 593, 751, 23, 70, "Input",
 CellID->234977969],
Cell[18536, 618, 346, 10, 36, "Output",
 CellID->601140228]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[18931, 634, 238, 7, 70, "ExampleSection",
 CellID->5256],
Cell[19172, 643, 132, 5, 70, "ExampleText",
 CellID->116368703],
Cell[19307, 650, 328, 12, 70, "Input",
 CellID->194601242],
Cell[CellGroupData[{
Cell[19660, 666, 349, 12, 70, "Input",
 CellID->215002914],
Cell[20012, 680, 351, 13, 51, "Output",
 CellID->324243934]
}, Open  ]],
Cell[20378, 696, 203, 6, 70, "ExampleText",
 CellID->495184610],
Cell[CellGroupData[{
Cell[20606, 706, 431, 15, 70, "Input",
 CellID->180347153],
Cell[21040, 723, 248, 9, 51, "Output",
 CellID->185444001]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[21337, 738, 222, 7, 70, "ExampleSection",
 CellID->282792129],
Cell[CellGroupData[{
Cell[21584, 749, 229, 7, 70, "ExampleSubsection",
 CellID->73904260],
Cell[21816, 758, 425, 15, 70, "ExampleText",
 CellID->34283710],
Cell[22244, 775, 278, 9, 70, "Input",
 CellID->112506301],
Cell[22525, 786, 136, 4, 70, "ExampleText",
 CellID->53502523],
Cell[CellGroupData[{
Cell[22686, 794, 185, 6, 70, "Input",
 CellID->155915323],
Cell[22874, 802, 778, 17, 54, "Output",
 CellID->570014100]
}, Open  ]],
Cell[23667, 822, 267, 9, 70, "ExampleText",
 CellID->151699388],
Cell[CellGroupData[{
Cell[23959, 835, 235, 7, 70, "Input",
 CellID->167441092],
Cell[24197, 844, 902, 20, 71, "Output",
 CellID->248223234]
}, Open  ]],
Cell[25114, 867, 145, 4, 70, "ExampleText",
 CellID->22816417],
Cell[CellGroupData[{
Cell[25284, 875, 285, 9, 70, "Input",
 CellID->174669911],
Cell[25572, 886, 777, 17, 71, "Output",
 CellID->175117973]
}, Open  ]],
Cell[CellGroupData[{
Cell[26386, 908, 282, 10, 70, "Input",
 CellID->191459788],
Cell[26671, 920, 251, 8, 39, "Output",
 CellID->251299560]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[26983, 935, 223, 7, 70, "ExampleSection",
 CellID->25767],
Cell[27209, 944, 61, 1, 70, "ExampleText",
 CellID->117238259],
Cell[CellGroupData[{
Cell[27295, 949, 579, 18, 70, "Input",
 CellID->79982441],
Cell[27877, 969, 6848, 116, 70, "Output",
 Evaluatable->False,
 CellID->25588907]
}, Open  ]],
Cell[34740, 1088, 401, 14, 70, "ExampleText",
 CellID->692511576],
Cell[35144, 1104, 2223, 64, 70, "Input",
 CellID->3010787],
Cell[CellGroupData[{
Cell[37392, 1172, 262, 9, 70, "Input",
 CellID->777638514],
Cell[37657, 1183, 2592, 47, 70, "Output",
 Evaluatable->False,
 CellID->351073543]
}, Open  ]],
Cell[40264, 1233, 126, 3, 70, "ExampleText",
 CellID->21418062],
Cell[CellGroupData[{
Cell[40415, 1240, 1267, 37, 70, "Input",
 CellID->236955778],
Cell[41685, 1279, 488, 17, 36, "Output",
 CellID->98721170]
}, Open  ]],
Cell[42188, 1299, 75, 1, 70, "ExampleText",
 CellID->132830414],
Cell[CellGroupData[{
Cell[42288, 1304, 170, 5, 70, "Input",
 CellID->636202219],
Cell[42461, 1311, 607, 14, 54, "Output",
 CellID->67822912]
}, Open  ]],
Cell[CellGroupData[{
Cell[43105, 1330, 569, 17, 70, "Input",
 CellID->146702974],
Cell[43677, 1349, 8138, 137, 70, "Output",
 Evaluatable->False,
 CellID->483937585]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[51864, 1492, 232, 7, 70, "ExampleSection",
 CellID->6774],
Cell[52099, 1501, 517, 20, 70, "ExampleText",
 CellID->16389],
Cell[CellGroupData[{
Cell[52641, 1525, 404, 14, 70, "Input",
 CellID->15889],
Cell[53048, 1541, 202, 7, 36, "Output",
 CellID->82942956]
}, Open  ]],
Cell[53265, 1551, 124, 3, 70, "ExampleDelimiter",
 CellID->42631810],
Cell[53392, 1556, 227, 8, 70, "ExampleText",
 CellID->735307411],
Cell[53622, 1566, 612, 21, 70, "Input",
 CellID->7259997],
Cell[54237, 1589, 73, 1, 70, "ExampleText",
 CellID->4753688],
Cell[CellGroupData[{
Cell[54335, 1594, 165, 5, 70, "Input",
 CellID->322065917],
Cell[54503, 1601, 249, 9, 51, "Output",
 CellID->255041252]
}, Open  ]],
Cell[54767, 1613, 310, 11, 70, "ExampleText",
 CellID->231560046],
Cell[CellGroupData[{
Cell[55102, 1628, 1392, 43, 70, "Input",
 CellID->20408039],
Cell[56497, 1673, 19554, 325, 70, "Output",
 Evaluatable->False,
 CellID->324960660]
}, Open  ]],
Cell[76066, 2001, 141, 4, 70, "ExampleText",
 CellID->91489318],
Cell[CellGroupData[{
Cell[76232, 2009, 726, 24, 70, "Input",
 CellID->148203188],
Cell[76961, 2035, 2010, 37, 70, "Output",
 Evaluatable->False,
 CellID->172639169]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[79032, 2079, 311, 9, 70, "SeeAlsoSection",
 CellID->25343],
Cell[79346, 2090, 2246, 80, 70, "SeeAlso",
 CellID->21821]
}, Open  ]],
Cell[CellGroupData[{
Cell[81629, 2175, 314, 9, 70, "TutorialsSection",
 CellID->16350],
Cell[81946, 2186, 153, 3, 70, "Tutorials",
 CellID->15515]
}, Open  ]],
Cell[CellGroupData[{
Cell[82136, 2194, 319, 9, 70, "MoreAboutSection",
 CellID->202379032],
Cell[82458, 2205, 186, 4, 70, "MoreAbout",
 CellID->483002166],
Cell[82647, 2211, 139, 3, 70, "MoreAbout",
 CellID->589262117],
Cell[82789, 2216, 161, 3, 70, "MoreAbout",
 CellID->2667426],
Cell[82953, 2221, 158, 3, 70, "MoreAbout",
 CellID->9279249],
Cell[83114, 2226, 136, 3, 70, "MoreAbout",
 CellID->198737940],
Cell[83253, 2231, 145, 3, 70, "MoreAbout",
 CellID->622895022],
Cell[83401, 2236, 179, 3, 70, "MoreAbout",
 CellID->556118210],
Cell[83583, 2241, 198, 4, 70, "MoreAbout",
 CellID->340301978],
Cell[83784, 2247, 175, 3, 70, "MoreAbout",
 CellID->495788959]
}, Open  ]],
Cell[83974, 2253, 27, 0, 70, "History"],
Cell[84004, 2255, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

