(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    129857,       2804]
NotebookOptionsPosition[    123216,       2570]
NotebookOutlinePosition[    124585,       2608]
CellTagsIndexPosition[    124498,       2603]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Continuous Distributions" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/ContinuousDistributions"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Continuous Distributions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Continuous Statistical Distributions" :> 
          Documentation`HelpLookup[
           "paclet:guide/ContinuousStatisticalDistributions"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"Continuous Statistical Distributions\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["LogisticDistribution", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["LogisticDistribution",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/LogisticDistribution"], "[", 
       RowBox[{
        StyleBox["\[Mu]", "TR"], ",", 
        StyleBox["\[Beta]", "TR"]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]represents a logistic distribution with mean ",
     Cell[BoxData[
      StyleBox["\[Mu]", "TR"]], "InlineFormula"],
     " and scale parameter ",
     Cell[BoxData[
      StyleBox["\[Beta]", "TR"]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->28313]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "The probability density for value ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " in a logistic distribution is proportional to ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox["\[ExponentialE]", 
     RowBox[{
      RowBox[{"-", 
       RowBox[{"(", 
        RowBox[{"x", "-", "\[Mu]"}], ")"}]}], "/", "\[Beta]"}]], "/", 
    RowBox[{"(", 
     RowBox[{"1", "+", 
      SuperscriptBox["\[ExponentialE]", 
       RowBox[{
        RowBox[{"-", 
         RowBox[{"(", 
          RowBox[{"x", "-", "\[Mu]"}], ")"}]}], "/", "\[Beta]"}]]}], ")"}]}], 
   TraditionalForm]], "InlineMath"],
 ".",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->12472,
  ButtonNote->"12472"]
}], "Notes",
 CellID->429991538],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["LogisticDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LogisticDistribution"]], "InlineFormula"],
 " allows ",
 Cell[BoxData[
  StyleBox["\[Mu]", "TR"]], "InlineFormula"],
 " to be any real number and ",
 Cell[BoxData[
  StyleBox["\[Beta]", "TR"]], "InlineFormula"],
 " to be any positive real number."
}], "Notes",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0.7, 0},
 ScriptMinSize->9,
 CellID->22312],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["LogisticDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LogisticDistribution"]], "InlineFormula"],
 " can be used with such functions as ",
 Cell[BoxData[
  ButtonBox["Mean",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Mean"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["CDF",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CDF"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["RandomReal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RandomReal"]], "InlineFormula"],
 ".",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->990,
  ButtonNote->"990"]
}], "Notes",
 CellID->33143606]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->4094141],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->517378609],

Cell["The mean and variance of a logistic distribution:", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 CellFrameLabels->{{None, None}, {None, None}},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->990],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"LogisticDistribution", "[", 
   RowBox[{"\[Mu]", ",", "\[Beta]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->3873],

Cell[BoxData["\[Mu]"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->66471315]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Variance", "[", 
  RowBox[{"LogisticDistribution", "[", 
   RowBox[{"\[Mu]", ",", "\[Beta]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->3413],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox["\[Pi]", "2"], " ", 
   SuperscriptBox["\[Beta]", "2"]}], "3"]], "Output",
 ImageSize->{38, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->52987697]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->31896],

Cell["Probability density function:", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 CellFrameLabels->{{None, None}, {None, None}},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->12472],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"LogisticDistribution", "[", 
    RowBox[{"\[Mu]", ",", "\[Beta]"}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->24450],

Cell[BoxData[
 FractionBox[
  SuperscriptBox["\[ExponentialE]", 
   RowBox[{"-", 
    FractionBox[
     RowBox[{"x", "-", "\[Mu]"}], "\[Beta]"]}]], 
  RowBox[{
   SuperscriptBox[
    RowBox[{"(", 
     RowBox[{"1", "+", 
      SuperscriptBox["\[ExponentialE]", 
       RowBox[{"-", 
        FractionBox[
         RowBox[{"x", "-", "\[Mu]"}], "\[Beta]"]}]]}], ")"}], "2"], " ", 
   "\[Beta]"}]]], "Output",
 ImageSize->{88, 61},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->394144508]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"PDF", "[", 
    RowBox[{
     RowBox[{"LogisticDistribution", "[", 
      RowBox[{"0", ",", "1"}], "]"}], ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "5"}], ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->4641],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 155},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->412329999]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->28213],

Cell["\<\
Generate a set of pseudorandom numbers that are logistically distributed:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->16824],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RandomReal", "[", 
  RowBox[{
   RowBox[{"LogisticDistribution", "[", 
    RowBox[{"0", ",", "1"}], "]"}], ",", "5"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->17929],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.7001762235917457`", ",", "0.26506499314110576`", ",", 
   RowBox[{"-", "2.318208509802954`"}], ",", "0.27616579180894885`", ",", 
   RowBox[{"-", "3.036225023407514`"}]}], "}"}]], "Output",
 ImageSize->{345, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->82500071]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->25403],

Cell["\<\
Properties based on higher\[Hyphen]order moments:\
\>", "ExampleText",
 CellID->20699375],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", 
  RowBox[{"LogisticDistribution", "[", 
   RowBox[{"\[Mu]", ",", "\[Beta]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->12594],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->83786101]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Kurtosis", "[", 
  RowBox[{"LogisticDistribution", "[", 
   RowBox[{"\[Mu]", ",", "\[Beta]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->22187],

Cell[BoxData[
 FractionBox["21", "5"]], "Output",
 ImageSize->{23, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->239233800]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->25079],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  StyleBox["q", "TI"]], "InlineFormula"],
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Null]", "th"], TraditionalForm]]],
 " quantile of a logistic distribution:"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->10041],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Quantile", "[", 
  RowBox[{
   RowBox[{"LogisticDistribution", "[", 
    RowBox[{"\[Mu]", ",", "\[Beta]"}], "]"}], ",", "q"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->18125],

Cell[BoxData[
 RowBox[{"\[Mu]", "-", 
  RowBox[{"\[Beta]", " ", 
   RowBox[{"Log", "[", 
    RowBox[{"1", "-", "q"}], "]"}], " ", 
   RowBox[{"Sign", "[", "\[Beta]", "]"}]}], "+", 
  RowBox[{"\[Beta]", " ", 
   RowBox[{"Log", "[", "q", "]"}], " ", 
   RowBox[{"Sign", "[", "\[Beta]", "]"}]}]}]], "Output",
 ImageSize->{264, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->596618132]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->4907],

Cell["\<\
Plot the cumulative distribution function of the random variable:\
\>", "ExampleText",
 CellID->29591],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"CDF", "[", 
    RowBox[{
     RowBox[{"LogisticDistribution", "[", 
      RowBox[{"0", ",", "2"}], "]"}], ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->25867],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], LineBox[CompressedData["
1:eJwVzns0lHkcx/GR6xlqDYUo84yZXAajbLqp/f7CSki2skqnixmXSFRnp0Tb
djJd6Iis3chhzrShdhsWOTW7UghDjFs7p0ieMNQw8yimK9nf/PE5n/P6783i
p+yIXUCj0cLw9F98hm+yKubydxl21QtpNBIe9xGhkUQC+K0conMMSZhvqc8g
iFNguVNhGmRCQvGjbKYxkQkP1Z/ncyxISNokPPKBWQjQ3DLMXEpC35dD+VPM
26Bt0PmCNwmmA8MuGqYMnm4KvJwWQ4KkXXxsgtkGGUZJdko5CRWVmQNJvH7o
M5CkHZe8gnjoKZw8PgJ5e//4syRhGJJVbiN3K95A8foA+ix3BOgskcvQuikY
WjK4L+vjCFyxfMKVHJwGhWaL4fWaUeAP7Hpo2PEehHOCusBUFRxrFvFTXD7D
vc1d9895j0HVUYuddlvn4JlZgEPO8zGQpB9Wr4ykobdMr/Z3F8bhnHj2h5lJ
A2SdYuTqw3oNtD4S1aUZonL1xmCbqtfg/FNc6i6uMXKxMnNTBryBUd21H80f
mCB5aaxTd8sbCFYl/px3zAwJphmJziFqCBvNdXd0pKPwmRtBWY/VoG7in5VL
zVGUrE60nTcBPIZyuf+BhWjf5pM1roUTQAdd9IvpRejVrbZPTboJ2KKJ+uAU
YYk8H9xsqtgxCc8aYwyiPBjoe9vK6IaySai99Hg/oWMgmc8qDzY1CXHGQlKh
sEIBnNJ1sWs1IDieXmtx3RrxSvJm9gs14BXvrgiNXozu+M0lb5dqoMvKw1y4
fgkaPHTtkG5UA/9WnCvyMLRBZ76pLzq9WAvpBtzwjkEbREkyBdn+WqC/SAtK
vmuLVm4ghwqPaEFVxsoXn7ZDtA+ELPaqFvJdlxrJIpaidabZV1rvaiH3elxr
NGGP2Jftq5S9WojK36ewGLdHg4Jf/3qo0cLHgNJP4noHFGkRtGKHGQUjKr/U
xovL0N+ceN9+RwpubFrl0Bm9HFnfo1mb+lAgPS9kxHs6orLoRbLOLRRcjBCE
GOocUfCXwKCc3RT4vJDcvtjJRD0SZZJFAgUFbkXSzhAC5UXeL/x6goJfHmUX
hMgI9HF1a5NURME2h4jSdg4LkYvGPTtzKdic3LqXkclCapXVjHkRBeJlfE6H
joWKvYuy7Usp4DSV/x621wl9K26+UCClINezkdtb74RUB2hhrrUUxE0b9bRx
2Sim26v7tzoK+FXviMwcNlJ0xff3N1IgbzfLqJtlo/PLm67OyylI8n+V24Zd
jRYG0NoosKyTlSixn6+IeK/3bmniP1PYq9e8jFqAPZ7T8ZYzx0a9Kf+xjbFN
InIPZmPPhu6upWP7D9l+t/8rG/WQlcol2GORUyGHsWfDqy/ZYGd1y/ekYpuI
qjfYYvc2nBLmYT/xvlNih82/OXCnBfttjSjOAftsgtjBa56Nnga36whszvBJ
t43Y3Mlb5SxseVT42q3YXeHn9zjp+0MNdgqwiZi1D9jYYp4g6xp2dc1ZkTO2
X7lvwU3swKPb1rjo+5iLy6qw+VO2r/XmMZobnmCH+ZaFuOn7Mou7nmHzTBLn
9D6x4MSgCrsq372Si22fHjbxDvvlmPqgO3b9tPOneez62XIrD+z/ARiRXWQ=

     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0.5},
  ImageMargins->0.,
  ImageSize->{225, Automatic},
  PlotRange->{{0, 10}, {0.5000000255102041, 0.9933071483973419}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{229, 149},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->314678191]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->226342353],

Cell[TextData[{
 "A contour plot as both ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["\[Beta]", "TR"]], "InlineFormula"],
 " are varied:"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->26444],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ContourPlot", "[", 
  RowBox[{
   RowBox[{"CDF", "[", 
    RowBox[{
     RowBox[{"LogisticDistribution", "[", 
      RowBox[{"0", ",", "\[Beta]"}], "]"}], ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "3"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\[Beta]", ",", 
     RowBox[{"1", "/", "10"}], ",", "3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->10654],

Cell[BoxData[
 GraphicsBox[GraphicsComplexBox[CompressedData["
1:eJxkXQVUVU8TfzTq3xYLsQNbVMSeiwoY2Ak22Njdih3YoGAhWCiYCCIgDdLd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   "], {{{}, 
     {RGBColor[0.33611150406573054`, 0.1316403510089098, 0.5894256959680103], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNVk1slFUUfdPOzDffzEdCIIBYkjoaMP6kEPFno6laE8RAoq6oP23BSmBQ
Rxcuigu7Mhq2bZyhriCdkpgUQWkXJm7oNAaInTIdNy6gdIOGsrAmLCqBc3IO
KYubd+d99/+ee98UD5Xf/awthHAN1A76OgphCRc3wf8B/t8khL/TOg8VxP+c
D+Er0EAqhAuQq4NmQM1MCB3Q6QI/DmNbINuArUHofQT6B78H8L0fdBC6z+RC
+CUv/kRKus/i/Bw6YzhfgZ1F2Lye0e9X05K9E8S/4d/DkeLuixTX0znFtsu2
aOcd8C2cs6BV0G3QsmPn/QJoLqvfU6ASfD7SJr0vEHsn8l+BvynYn87LN8/H
E/E1yHfDfxW6n2TFj+NuvW3QbxF3o7gbATUg8/ZD9wdB+8Ffwv1l0HHHzBgY
XwI/H4LOgJ9HPL9Gst0EfzlSvNSj/jHI7MC3DeA7wV/MqF7daeltSqR7zP5K
QT1nDVm388jr5Vh3rPu+oBpuzMombZ9CjTsg9zz43ZD9sqAeslfMhzmyzwO2
Oer7/e4p86VsL+gkbOyBzHuIK5+ohnfw/UBePVyAzLZI+KKf03n5upaSHmu1
BLzdS6vfxNE2Y4k8ZYnJXcYU+d/ahGXmQr3d7dIlXveC1kPmJPxVQR8Qb8jx
iVjf/jN2iJlPUY9ypJ4zFubynb/f9lz8iO+ZRPGypm95vgYjxbGd/lDPmYJs
XDdG9hmfy7ZDvTet2wd7G0A/QS6B7hHUpIcYRxwfw+5CVkQftD+ZEa4niHP4
aRSEv17oXYxUv2KimeRMLbkerBlx1eM6n/BcdTl++iIGmMeg+RnYmU1Umx73
ibo52E7j/oeMasuTGNya1r5gfszjqHNh3Vn/94MwWrV81frkOWdlzx3jJua4
H1aM95SxfT4t+13Ie2dBv8s51Y6+KMsZoW6/dwn5Ye8rYvg1xH4Y8svuz6ix
wZpy1s84Hu6CcVPN98xhxPJxSjuIOCJWhoLqQzvfgypBu5L4ZF1fhN//C5od
4ox2K5b71jYY05jtj7k+5P+KNVdV+7nk3cKeXnF/zmaED8a5CbXfDKq5H+wL
MUefE/bJXfhkTjWOwa9LlCfxVXMetDVpnVJeM0Ac9HlX33SuE9bh3NHnXs8f
ccwZ/AZ+tuaE1ZYxzXvi/mxmzQ/56SDZOWN72jt9ynkPGUe082dWWOUepsy0
Z+2C9z9na3NOfiqIfzESTrfgbiO+nXO+k/bLOWjZJnHPPt3yPuD5YCfUvTf4
NjLfhme64Z0zn1VMtFmCn6OJ9m47ziin3nIftjLab5Sd91vCnbfg+znbmfUs
8A0gxhlLxZjhLqc83wieTfN7HO9x46viXnFHLvkt5rlovpAIB8QA8brDO63l
fBnDo4j9akFvWt194T3fgaLfAvK0SZzXYuGe/Jh3IvdhL+y8gPsb9n/D8q/j
/qlEdRlyfoyduDjnHq169ppBO4N94lxsd8yd3i0lzyvft5di9ZT9HfRefSyv
9+DBW7Dq3hJTxADvOUNFz9E6zzv9noLe6XjtveHJ/O6iNqsF7ZR6orllbOxp
h98+9pjvNutGXB1uVz+v+D8D9yFna9j76ipsNGPJjT+EtRHcPRcLJ/2W51zW
jH3OQDmr+txy7MyB71kdMf4e6d0e8W5mXxj7Pce/EskH7dN32XZY8+5obTZ4
Elv3AW16aHU=
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwlk81vTVEUxfd7fbf3vndvpVQVo5aEEkGoMDBokQiJBCMVhJImrWgj/gRh
YmIiKKM2GrNSwohJW5EQ0RZTqiMSo0ZqUOq3sgYrWefsj7POOvu09QyeHChH
RCeogHdpxCuwls3+JCLNIspFxBD8AWirRbSCI8RH6yKWiE8Sv1IfMQj64F3s
rYI3gQH4b/bewy/T/2MeMQ0aqT9eirgI/8B5Pzj8J/gGb665dwvrU/BuMM96
J/k95F9jvansmoTejZxxmNjXks9uZ+9TYg2T5E+B7sQ5/+BdxH+xPoOezeS3
sp5gvVCx1kH2DpasuR3+gvMOEesEM9TP5vbmBDnN5K/PzHWnJtY3WL+k91Ny
tsLHpZ/YeXJG4Iup73KrZK0HyPmcWLNqF9mbrncP3bW/sJe683LuO0j7HfJv
02u4GrEbf5bRt4b6leSPEXsCavAcjMLvU/OW/DEwVLYnG4i1gQsl33GXelWt
bRvYD9/OeeMV30HabmZ+O2m8i757YB3xo2V7Kc/klTwdof406/nEGmernhnN
Si/rPdLG3jG0fU+sbabqmDTuhe8Dr+EdoJfausKzqDfsgHdnrp0DS2j9m9sb
9awUnkHN3sPE3rRkfht5tCP3HaRdd5xI7am81Mz1yf+aZ0sz0UB9AR6HPdXf
yDL31h9ZXfMMa3Yf1fnu0iht8uANfKrwW+jPbIEvpPZaM/Cc+hUNzErqN5lL
3VO95LG8uJT7r8gT5T6rebZUcx1czT3LbMVZ+p8rPAvS/Ac+nNnLL5y/seo9
cf3p/wHShPo=
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.4215025121971913, 0.28011225302672904`, 0.7094530879040305], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFl2mIVmUUx++7XN/3LjMZNS1GofmhKdo0o6BIxyiKosU+OE5pVI7mNGhq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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlVtsVVUQhtc5+xzOPntvKlFaqHih9ME7Aoo+YITWaDQY1BojIGBEaG1J
aFrkhZIYQYlCYkIEJOHBhILXRgwtBaMvXEqrKCIUEwGNGlRMNBhMbA0F/X5/
H/5kZv+zZmbNmplds7i1YXk+hLAWFECShTAf5EaF8HYxhGfTEJaA7zH6Cf4k
8iCo1aEohL5yCF2lEGZi+yT6NPRN4Ebkb7FZEIcwlITwIs7XggZ8PwYezIVw
kTNXIY/DZg56jP0Jzh4H63HfiT4AfwZ9G3o7OfUTewBsy9uHYv1J/At5x5xE
rDq+/YK+B/0zuI/ALvQlxNtErKVl576ZfE4j9xFjOb5bI99d3ySrBsP4exd9
DkfWYN+JvB3ci/4B8T9HPgJWo79OjJvx35G4mHVgL3IvWITtfejPIWfYzEA+
T4yN5LaD83ciB/KbCz/Et1r4H/nWzV13p/atGi8kvwy8g1wgvwXYLwRt8FOJ
8SFnB/HXFLlGzdi2gGH0F+C7sC0TfxbyQbAf+1fBfvjvIvtanToX+TyNfiqx
7xZy2lHym+utb8WmFnkW+Bnbw6DI2Y34vw75NzAWeQ3II3dh308ul2K/ZRs4
jH4o81vrDargrlT/5d0Ti/F3QTajfAfVrhUcy7mG1yC/Bqqx34f/Smw/Jsd+
rtCD/iu5nQOfovdicwx/X4JGuKaie+MsfEfkHilwPsbfzJxjTEbek/jtHgBF
+Pc504G/NznfCF/HtzeQD2D/BL42wA8X3JO3wNcof/h2cDf8BPztijxT18O1
kc8qzh6Ar8Z+XexZ6EbvSexDZ+vxeQbbLHV/reDbPvhB4v3Fp/v5NgHbKnz2
wu0FG9DHZZaL1HBS5p7UfZSTerU7sW/5/AffF8GMnO9UkfkN9XbvgefhpmG/
Puc7qhYt6H9ErkmGnILLwTNYz/mm2L02FfsfsN8KPx65Mufa3hY7d+Wkt/ik
5LfSm3TCryTm3wXPRJq6Brq77twA/3ji3lXP3RC7pqqlZm4pXCP4PXLPf8HZ
o6l3wSPoK7F/NPMu0gwPqL/h5xa9g0bDlcHsnGfuCPwJ+Gbkr4N77avUvaSe
U+8uyjyb6mH13tOZa68evAJ5TOZaaqeNkFsOfSv6lIJrfwlsKfoNdpa9Q7U7
tSOUS5p5Nysnzd5EfPREnkH11vbYs6geG42vKegVyFvkP3FMxdJOe5nz68BI
3m/WV/KMa7ZVM+3up2LvPu3wV7CdjX4K/lr4l9BbQBG5GrRj/3DZsz+EPp1c
58XeldPRR8jnLr6dLfqOkt8qm9O3ErZR5lnSDDYn3pnaleqxVuSK1HfRP0h3
087XrtcdJ6I/A3+u4DtVwVUm7rU+9NvhJoP5Offg5dg7SbtoBViV+B+x+f95
0G7UTGuWtSNvwrY+9i7VN82adpB2j2ZOu0UzrNnVjjlZdgz51k4+BD8m9u5Q
j43l/PjYvagdeRz+6ti7Sz2m3r+j7H+pZiBgH8WeJfWMduPRkv/N2pHLyt5Z
/+0q4j2EPJj4rTSzqo3eSG+jGuktv0m82/Wm93B+d+J/l2pUk7iGqp167CCx
5iX+N+kf+C+gWA20
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.5068935203286523, 0.42858415504454866`, 0.8294804798400511], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmGuMVdUVx8/cc+89555z0FYGtdN+gKaxaYwtqYlQKTbqB9OOpHYmmuEh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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlntolnUUx3/v++69PM/7TN2wtEngIoyMKAqcZRecf0Q1qRbFcKbmbbN5
qciJZVGRutxaCNUflVK5kTXBtQXaWjciM10StGlq/WG1yksWUaHZss+X7/44
cM7v3C/POU/lwlW1K9MhhHagBKgqhvBuIYTdELuAi5IQKoC+bAgD8GvB7wSO
pUL4l7dbkb0mDuHTTAhLkV8JPpAP4XtkRzA8B/oM9KXwfkTmCPiaKIRS8CJv
h+AfBK5GvgH6Y/kihlfRXZaz7hH4TSW2MQ9/Z6GfAd8AvIBsA2+niWU8cC/0
TPgPo38E/tzYMcj3MPAe+FrgPPgt8G9Ddyi272XAUfDDwHMp+2wDnwrUgSdA
LfhmdNrBv8PGWOKtB9LE2oX/veAj8Dvw/yD8VvD5vD0NXs7bSfB23mqx34PO
dOK9Fnp5yjWvxv4sYFrGOaxAfjmwAN4E5BdTuyVAZdo1Lkf3o8S12ob/XOI3
4a+rZ8j+Bl0Hfz42m7HdiM9TqnXauV0BfJBxjqep9+zIvMuIaXvkmqvWiuk4
dAVQlnZOY8E3wl8H3s3bOWxVEcMwvjeiM4S9dbzl4Z8h3mp4GeRvSLln6sVC
bPyXdk+WQEfwp6F7SjrQ9dA7ka+C7kL2Omz8jO4d2Psb/HH4s+EfIMfdyF8J
vShlexPhXwzdn/UMnyWee4qejSnAL+oN/Brku9CfrH7h4zHoupR7+23kXqrH
TeAPRM5FPjYrv6Jn62beTiDfz1sf+E7kZ0BfD9yI7E/4X4D96Ylx5bCIeBcD
T6UtE+ClEtdONRoBP4/OFlg/KD9sH8N/M/7eRmcYfBzQH9yDdQXXRLWQznZi
exNoQ/aqrH29BH1h1j53QL8IfUHWNd+G7knVE/0aYA+2T0CvxFYd9hv0bUK/
XOIZUm2fByrSrvFD4L3YmIvu18T/CPKtxDw5eEeshr8/8ixqh4xHdhf8Lvg9
Wc/OW5F7rRk6jvwbiX0rh8+xtweohzc3613TDL8y5+9Bs94D/6asZ/598H6g
MeOd0oPsO8ATyM7JOXftEO0O1eB39LsT85RDKbwxwJ/oN6o/kXuiXpxOOXbN
kGZHOahWqqFqp5pthd+Pzu0Z1zwPr5D421QMvQXvXO1a5fAo/UgUU4m/8S/B
JxZcm0VAC/gA+muQLcKvjz3TU0b3q2YhVRrCHznPhL5VzaRmUd+seGWRdSWz
Ht150OfS3gEJ9qqhZ0Bvhb4ffDDvb0nf6FHw1ti7cG3Wui28HU7bRge8ztF9
q52eJf5f894l2pnr5Tu27QpgEvQXxLc655zke2bkWiuGFvBZ5Pxkyju5G1t7
i+59G/aLRceo2FQzzUYZ8q9lPSOXF7zzteu1g3R7lINi1w36DHxp7F2zAnuH
IvdYvdXMXIL+X9CbRvMfKnoHa9aasuZVFnzrJCPfdyW+jYphP/i+xLdTOXYi
3xF5V2in/RN7p2mXrSfGVdCDRdvWzPWC78g7Ns10Bn46dj1V847E35y+NX2z
Y6DHJb5F2mkf5n1zdGsOY2Mq+Dexb91BZJ4F3xT7FusGfRVbRjzdHN3a+wqO
TTdXu3FD5N5pR+4nvoGid5F2sHZTX963QDtqH/iBomdXO3EStiYkvoW6gert
YOx/B/W4HNktBd+6V3j7BP27Y99a7RDtHsmIpx2kb6sz792mb2xI8xp5NnSD
NXs1kXe5avY/tWA0Zg==
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.5847482423373828, 0.5611899048399556, 0.9099758210116741], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmH+QVmUVx++7777vvfd9L9Xyo9Fc/4BmcZB0WqqJjQktUGRbU6u1aRwQ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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlm1ol1UYxs///Xme/yNhafhWoNUHUaKswClotcqXNCtJWWLO/z606TZX
g4GgUYRlZRmR80uTFG1WjJpOKzWtPhipqUmpWCoZkW5phmkgiv2uLj/ccN/P
db+dc+5znWd4ZdGTzdkQQjuSR26IQziThtBVDCEBiLCTfiFUl0K4kglhbDmE
niiEaWBTCdjF9z/x34L/L8Qfw349IQasgP9H+LaCDwM/x7dm8i0Bz6BXwGeR
7ylkDPqBXAhzwDYgd+T97RH0SchYsFZqfkWuXcgUsMmFEAYSOwBpwn6PmGPU
m4t9Cv0uajYR24N9P76riH8FfRmyOeOcy/HvT0/dYEuxb0JPWe8k1lHg29fU
WoxPA9gC8i0kthE5S/655JiN/mPk3kbhsxZ9CN/eB1+HjMCukHMevjXIFuwO
ehoG9gXyONgTSIX4XvZvMvUmIS3oF+h5PXofMW+C15C/jtgK0ovehwyl1qGS
Y9eSbwXYG4n3fgd4E7nHXa9fzPmsVpJvZsZnVkF/mG9/U2trxmf1Md8+zPvM
GtCr+fZXwT6jyDWcmhvB76anW8HfRobqbMBng7cjI6m1gnzL6G1U4l760+MR
9J8SY/XkWKqzwe7AzhRcq5585wuuqbXmscfkveaTsWdSs9iKfQD//cgBci+m
36n4HiWmLusam7HbwC/kPLP3gt2H7MZeCb6H+gPx6aJWLXIcrIb8F/FtynuW
fgb/Le+Z+gN8RuSzbsNuJ/cp7InUnoCcTFxDuZvBx4OPRXqwT5P/CPmmk/+s
8mMfxLcR/Oaiez4MtgOfKWBvsabP2b992POxu7AH41uLFPHfrnj0H1KvvV/W
a1mKXM54TefRN6eenXfBp6NPQx4C/xe8hD4TuYq+TjWot58e5ud8ZzZSewPf
5lH7TuxnseuRK+CjybcdfVvZd1E9q7de4l/OucdO9GrWeFD5yHEZfS49dVO/
SvMLvi/13fss616+51ttzj39GnmPtbe1xBfRo9S9igNawR9NvRbdmRZxTdl3
bzAxa+i9vey7ojU0gXdE7l17Pg/9Kj2Nz5kTBoEPSb23uiNfEvtN6rvfQUwL
/uuwp2IfAp8iPfVd/RZ7b9kzpVnSmQ0DuwXZWfCMDmBtWfjlEvG7sG8jXx3f
RmTND+sTz6BmTxz4IPoDiblRe7iC9TyN/6WsZ3hE2WvWWnVnlXtg7FlQjWtg
SWwu68Nuxl6UmisHUa+W+lWpz0Iz+j93ReYScVhb4pnVrOrMxV1nsF8qmMOa
sWuv96871hWbg8W92tMGfG/n24tFc+ZE9AmxuVOcLW6qir12cdQW1nci9t3u
zPrtKWF/EvwGjYz8hlSu+8ex16i1yecMeGfZs6ozOkdMlh5X53zn9LZtK/ku
6Y3bWfYZ62ynFcxty0vmKnHcJvSj1FhA/Bp89hL/XepZbSN+E/qnqe/WnKLf
jhtjn63ekBnoj4kzgzleb8nFyLn0pmjWu1PHauZ1d/akzq07JG48nZhrxJHa
6+GR90t7Pis254rrVEPcdTgxt4rDGhO/gXr7dOdXgf9TMiaOFreOS81N4tgM
+At8ex7sA3yWoB/n23M5c8rQ1JwvrtdM62wXlP2264xfo9Y9kf8N9A+QS/xm
6K3QGWh23ymZ6zXDvcReiz2LmqnV5F8Y+S6JU4/E5kRx4QatAXxrbC7QGyGu
6i757MRZu2PfUd1NvZla64nEXKw16yx7Sl6bzvR37Fcj/+voH2BjbM4SV4mj
RoMtiv12auaeKfvN1FupN0F7lY09q9oz7dW1yL1qz1bpfpX91uu8/gNFYDmp

         "],
         VertexColors->None]}]}, 
     {RGBColor[0.6475303921006732, 0.6620633501905816, 0.911407060654613], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmHvQVlUVxg/v+73vOec9B5sxQQiUoTSJkVtqJYOlXAMRQggcyyCZmgyF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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlntsVFUQxs/u0t29e28x4U0AEYyAxooViGKIirQgCFRIUaP4AKMRijSi
ESICBlDeiEh5JYSKClrARKDIP6I0IcQYQXlFraEBA4hGGxBERMDfl48/Jnfm
fDNzzsyZmXO7ja8ePTkdQvgQagE9HodwJB/CqyyWZEIYA3+lEMJ8wA2pEOqR
+6OzCGwEOmPh50P5ohB6Z0P4KAlhCjpPgI9EngA2CWrG/hv0d4Dvy4Uwm70q
sSmPQhgClYBl8N8T3V7QZvR7s1YBvx9/VfDd8DkO+9/Rnwufw39b8NbQHeDr
sOkA3xF6Cv4ce6xAfyL2w9EfgP53yAuQZyFPR96F/Dl0mr2fRX4GbBTxNmYc
8/vInfHXgP8+UBVYMWsDwM6i04R8DCrBfi/yGnTXQp2JbS76m+G3QP2Q30Fe
qNxBCfIbyHvYuwF6GLkZH/PAxuF/UMY5XQV2AnkOuiNY667caP+0z6SzDEfn
nyKfqRN4bd5nvYz+ZLBr5Os9+A7E9wfyMvBK9tqF3B3sB2yqsZ+BPKngGBXb
Od2xzgYdTXmPCvS/xr4O7E/2WMxd9is49tLg3L2Fv1zR9RwmjlmxluJ/Cbbj
WYvgC1APsFugbMp3fgD7/dDYlO/wfvbri00rfJ3ERznYYeRGsIOstce2Xexa
uYLcGv054NPgt6V9121Z2572nQ9H/rngWlrKHt9ylu+T67WAzm58fUZMbwbf
SR36U8A7ZZ3zm/B1MHGu9iCvQr8mcW3NRu4Ffiv+t+P/CPJZ7Hcmri3VfBn8
i6ztZe/B6CyEfy5xLm5Iu5arE9+VanoTZ9mHzyczrqFPwDcm7q2l4F0LPpPO
oj3rYusIuwv9COwENBVsC2tRMX2GvIGzDMT3SXzt4LsM+QN8xuAx+Dbkctbb
wJ+CZmacwwmctxybsoxjuAz/N2vL0W/C/rjijRybarZz4p5Vr37JeVbHzply
1bHItTQK/ea0a2pB7JmiWRKBD8W2Ku/avyCZeIdBRzOeUafhf1XNIL/LebZx
5l+wWZ91TGuR6/FfnnHPFcFfVb9nnYNU5Bwo9suqcfb6D/l41jHVY38KeSNy
LfILnO15qCd7X1JPYZ8hZ/+CNwTnqlXk2lPOiuEDeHPWOa3EV0v2WMdZrkJP
46sReXraOb2bWO6Bbst45k0Ffy12rZ5HngG/POdeU8z1yGMS+zqZsu+G2LNV
e3QC65I49+r5Os1b/B/CdiA+avH1BWsjM55Zjdj+GHt2voz8G/yZ2LNeM+Ri
3neuu9bMrALrwVqLjGf8VznPGM2WJnzMAp8JrU85psFgwxLf5eGUe2ti7LdB
Paa7u5D3LNQd7owdo2IbhM1FztofOo7+EvzdSSylBedmNjo3560j7BprLcFG
Rp5VtayVwQ+Czga/OY+on3OelarBFPr3RZ41K1l7O3bOlCvNNMUqHWGKuQTs
dqhXyjPrUfhFUFewv9J+q/YT0+tZv1lNyMci19LmtHV1Rp1NNpp1bWK/ZZp5
7cGLqZ+hnDGnngYrjf1WbkenmfXR6DyY9hlWIG9BvjfjHtmKvCP2LNdMaxF5
Bqj3z6GfRO559fpV5JeQJ6nng3O8GL4s79xqjxrNq8i51T/AmsRvnt66B4hx
cMEzWrNZNdYF/saCZ6FmvGbZpznfpWaaZnUh8mzSzNbbsJi1g1m/EeqdlpFj
Vw9Vgp0p+N9ENVUDtjJyrvUGq7ezkWNRj6t305F7Wz2s3r6Ud+zq8Vfgd4NX
ZPzGqZc7Rp7F6umPE7/hersrkB9KXMOq3fPEcIBYjsaeBcqZZm0u8t1q5uru
2kWeDbpDvbU/xe4tvbnae0hiXzqD/p02Jd5L/1CHsJlW8NummlWvbsx5tqtn
dddrcp6VuvOtkf/Z9K+mmaJ/iXmJ3179UzzG3hV5z0q9oQcSv4F6+1Sj+lda
nfgu9c+kWqiG8tfj0yxdn/Ps1kztg7++sWtRM/F/N6ta4A==
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.7103125418639638, 0.762936795541208, 0.9128383002975519], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmGuwlWUVx9+z9z57v+9+94HGG9jVaBIKnA4c8EJcRhQbs4lLNvUBaDhc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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlnmwjmUYxt/zfd8537thSlmmRZSlkD0mpAmVFlua8Yc0DlJDGJ0cSTnG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         "],
         VertexColors->None]}]}, 
     {RGBColor[0.7720281052631939, 0.8501316842105384, 0.9098211052631417], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmGuMVdUVx8/ce+fec885pi9qE8swCmMlKtDETonlIVCqlgrIa/qQ8KGg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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlnuQzWUYx397ztk95/x+x2gaaRLKqEioSRjD7iKpjNBiVYyZYvlDky4Y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         "],
         VertexColors->None]}]}, 
     {RGBColor[0.8284107368421468, 0.8689337894736981, 0.8845617368420866], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmGuwlVUZx9+997vPfve7d6LxqRkuIhwVZDSskAMWFHBCLWIYOlyM6TJ1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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlmuQjmUYx5/3sO8+h7ecPtQ0IbOoRUlFpFmEteggsVJjavqAL5RKTM7W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         "],
         VertexColors->None]}]}, 
     {RGBColor[0.8847933684210999, 0.8877358947368579, 0.8593023684210315], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFl22wVlUVx899nvM85zkvTkDMFBVCeIFA0a68lLeGD800cIe4iBdCbRJq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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlmtMlmUYxx9eHnifw9uS6kurjBpgmVp4bJZf2poySstDWK60ZmVtKbpw
ohWeivqiARsdTEWzoloLra2FbUUHM50rpfLQBqYgNU1J0crK9fvv34dru677
fx3uw3W4r3lowdT5mSAIWqAQmhAFwQ5oNotVLOyH3xQHwcj8IOhCLkyCoAAK
84JgPDrTwb5E57rCIPgdOQN/Pg2CDuFQgu4EdHrA1uNjaS4IaqHVYFWsvZYN
gj/RbywIgs9YOwt/BhoLv4a1Odjfi9xN7F+wfxF+HTYT2esI1hqQDxPzFuIX
oX8M/R6oGfsa8B3wlcQ/TqxpyH+h/0fqvSlmNfz14PvB+8G3IG/G/0z8j0a+
kPUeFPsG5E/hxxFvMvyd0Du6j8i6o7RH+C/wNwP9CvbUCT6RtYfB5kJ74BvB
y8A7kWuJN5L76GIvi5CvApvNWi98H/QJ9uXYPM5+u9ljC7qFyGPAJuF/HnIz
cgnYIHwORx6Rc6yhegNhyFPAa3Tm1Hemu7oEujnymXSWKXpvdA/pffD9LTbl
8EvRvxB6j8XwrZHfXm+axfbv2PmwLt9v0Y5cke832Yr8JD4Gw18NDYCdgYrR
fwT7Juzrka9Av4gYFfjfic2DGa8NAr8U2gI/MePcex6KQ8dcHPvOdFc5dGKw
Uugj+F3o1xF7ObQPvIw3fU/3Dz4IuQ16nfd9k5hDOdtj4HVgw1Jjc/G/M3FN
qBbugWrw1YZ8O76LOc858AGoFd3loWvhN+xXF7gmZkbOIeXOu+i8kTpHlBuK
2Y+/z7PeWztrq/D1LJQN/eYHss5h5a7W+jnrYOgpdOs54zR8nUPnWrCX0G9A
dwxrlaHPOAV5MnQQ3UOqH+I1QeOxr+MMrWBB5FoaHlpXNa9al82/+D6Xeu+q
yTZi35XYl2pQsRayFuc7pmpjbOxcVY3kRY4h3zdCp/B1DFqMr+c4fxXYVOQD
YD/jownb3sS1OwR5O3JT6lxVTaqXtINPCp1/J3WfOn+e7/w0cn/Wua8eo1q6
EptjoWtqI3wz+Ntg5fj/jv2tyDnWE+pn6K9HZxyxDqJ/BN2lsWurDv8d4CfR
/5j7uxt5DdhaaB62pdh0g5eB7wJbQoyVYEOQdyNfBv5TzjWvWq/Hx4ms70R3
UYb+stQ+5Us1txH9Pvb4QsY9ZRvYB1AtcgU6q9F/BXksfGGB+Vdjn0Vrp7mr
UTljz4SuxW2Rc1c1eWvkHFdua4/V8AsT94oqqCv1mXSWR6E58Cfwf3me7/Ti
xDWq2tQblOJ/P2eaEfoO9sCPjm17lP1UYj8/da/di837xC/C/nzGNbUCrAsf
a9nbmDzntnJEuaEc19tsgBZk/EZHUue8cn0Z8crBbordW9TTh4EtQmcgdE85
he9Vsd9Cb9ir+49dS+qR6g3qSUX/14962z9QJt/9pjHnnFYuq4Y0a45CI/I8
c6rR3Zt1bqom7khd46rtbuSt6A5G7mCvBfj4Ef576Bv1qkL3kjOJz66eortY
mfrtdCeTU88QzQ71tF+J1RF5lj6QcW9Qjio31SNC9U7VZMYzsAF/3ZFrXz1a
e2lL/NbaU+FFvEHivekNVat9qXNfNVsC9lbkXq8eux356cS9UjWp2hpIPbtV
Y5pVmpGajZpZ9fA/5HxW9fAN8JtynmXK8ZLIfwb9FXTnp/E3PfHdFaP/Nfik
1H+NOdDBnGeMZotmVgu6mxJjs6AD4KWxZ5lmyFeR70x3pT9OH/yGxLqaSbXo
Lomdq5qpa7PuGeoVyrlpxN4XuTeoZ7WAnU3c+/VH0qzfnHivmvn3qbexdjj0
H2Z35BpQ7qtnRIlnlmaV3mhW6hpXbcumAX5N1r1df4b7kWtizzrVzMtgHyLf
lu+e0IN8PPUsV40qtzpz7i3KMenqz6W/lmyWpO5J6kXK4f8AiA1RYA==
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.941176, 0.906538, 0.834043], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNlktslFUYhv+ZYeafmb+0EgK9QKtGEOlY60IDQlmwMlHjgku8AIlxJS4Q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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwlk0tsTHEUxs/M7Z25d+ZKWaBigbTVhHgsJFRY2Fh4LdouPHY2xKI0Qjpd
FJUo9aiWKBIWEq/EgthZISm1kKiJR8WUSEQkotPSVps08ftyFif3nHu+c77z
+i/Ze7ChOW1mm5EKJB2ZTebNekKzS8j+xKwtNpuXMpudMRvH9xdZB/Y8/h3o
x8Esw3eARH1Zswn+deOrB9ONXcJ/LuMxY/gK2DXYXWAOkbsFaSb/T/xl9BFk
IfYT8l0l/iv4y+C34B/FHiXHGWLPKj5n9hDZSPETgdkM/j95r0017Ebfhawl
Xyv5MrPMPoCZIdcCvk343tKzhrA68FqL2NdSXvMNMNN5n8VWYrajP+ffG/gG
kQK19oGfA7aS+A3odeSYj14gx4rEa1atQ8hn7AvYdeTqIqYau56YAXwvkRHs
34n3XkWObfA10d/ylHNO4XsGvhr7E/HT2C8i5yphF8GuTJzrFP8aiW9AVqW8
xyn0f3mfpXasXm/G3rt67kA/GTu3evhGrT/At4d+A7XkXoocDX2G2kU567vU
Tr4kfgPafS/1tGE/hWOu+c7H8N3KeW3qaR/+K5HfViUyhP4Y/k1wP8CfAXuf
f3fp/XrgtyCMfLqJxfAPZv127oB5BL4DGQ/8Jo6Bv0i+2sBvVLNuyfmtaOav
iW2NfXfDzGd95DvRLl6FnivkZqqynnMRfPci57pd4bl1E7oFcZQ0a/z9xB5B
+iPfkXajne2MfEaajW6yh/je2N+ObqITvtM573Ug7bnWxF6bcjZGfkO6He20
DN9hZBL+ztBnrRvRbWjmmp1qVq2aobBF/O0Zj1FvJ3L+FtTjHvwfI38r2sFw
3ntSL9r5r6xziks38R27JvFZ6c2+R3+XeO3i0NvQjrVbvZH/Zuifxw==
         "],
         VertexColors->None]}]}}, {{}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1949, 3066, 2050, 3063, 3065, 3064, 1948, 2484, 1944, 3296, 
         2265, 2483, 2482, 2907, 2264, 2906, 2905, 2656, 2046, 2136, 3295, 
         1943, 3062, 2045, 3059, 3061, 3060, 1942, 2463, 1935, 3293, 2259, 
         2462, 2461, 2900, 2258, 2899, 2898, 2647, 2038, 2131, 3292, 1934, 
         2449, 2450, 2253, 2894, 3200, 3199, 2448, 2893, 1986, 3312, 2892, 
         2644, 2034, 2722, 2119, 2771, 3118, 3117, 2033, 3057, 2032, 3058, 
         3193, 3194, 2240, 3321, 2241, 2025, 2245, 3322, 2246, 2109, 2760, 
         2108, 3111, 2434, 2435, 3311, 3310, 3009, 2334, 2544, 2543, 2021, 
         3108, 2106, 3107, 2105, 2673, 2672, 2736, 3102, 2737, 2690, 3333, 
         2689, 3334, 2007, 3109, 3110, 3039, 3038, 3037, 2095, 2356, 1997, 
         2424, 2425, 2199, 3175, 2198, 3176, 2426, 2018, 3047, 3046, 3055, 
         2028, 3056, 2931, 3283, 3284, 3286, 3285, 2695, 2696, 2565, 2566, 
         2217, 2847, 2216, 2848, 2001, 2416, 2417, 2101, 2705, 3042}],
        "0.95`"],
       Annotation[#, 0.95, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1926, 2930, 2604, 2248, 2889, 2110, 2439, 2707, 2706, 2221, 
         2860, 2687, 2685, 2747, 2746, 2365, 2570, 2084, 2408, 2556, 2555, 
         2567, 2215, 2596, 2597, 2569, 2568, 2213, 2595, 2212, 2564, 2379, 
         2758, 2008, 3040, 2693, 2691, 2097, 2748, 2357, 2358, 2103, 2751, 
         2104, 2752, 3051, 2020, 2540, 2912, 2332, 2911, 2333, 2422, 2421, 
         2397, 2577, 2223, 2584, 2224, 3189, 2226, 3188, 2225, 3289, 3290, 
         3287, 2341, 3288, 2578, 2579, 2576, 2575, 2598, 2583, 2024, 3054, 
         3053, 2244, 2388, 2387, 2888, 2247, 2721, 2720, 2719, 2243, 2718, 
         2023, 2547, 2548, 2250, 2586, 2352, 2353, 2345, 2928, 2925, 2926, 
         2891, 2251, 2617, 2252, 2447, 2446, 2939, 2938, 3121, 3122, 2937, 
         3291, 2936, 2037, 2645, 2787, 2788, 2790, 2789, 2451, 3201, 2256, 
         2897, 2257, 2458, 2457, 2948, 2947, 3123, 2802, 2946, 3294, 2945, 
         2044, 2655, 2803, 2804, 2806, 2805, 2464, 2465, 2263, 2638, 2639, 
         2640, 2480, 1947, 2960, 2729, 2048, 2728, 2049, 3218, 3219, 3216, 
         2271, 3217, 2489, 2490, 2272, 2961, 2962, 2963, 2491, 1954, 2980, 
         2731, 2053, 2730, 2054, 3232, 3233, 3230, 2279, 3231, 2498, 2499, 
         2280, 2981, 2982, 1955, 2500, 1962}],
        "0.9`"],
       Annotation[#, 0.9, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2019, 3048, 2713, 2107, 2759, 2098, 2427, 1924, 2605, 2346, 
         2111, 2378, 2376, 2006, 2857, 2831, 2832, 2686, 2201, 2407, 2406, 
         2000, 2677, 2553, 2202, 2599, 2203, 2554, 2762, 2761, 2214, 3185, 
         2846, 2692, 2211, 2845, 2079, 2749, 2750, 2709, 2710, 2013, 2430, 
         2014, 2883, 2884, 2882, 2239, 2542, 2541, 2022, 3052, 2881, 2238, 
         2879, 2237, 2880, 2441, 2442, 2236, 2878, 2235, 3192, 2015, 2545, 
         2546, 2242, 2585, 2350, 2351, 3196, 3195, 2887, 2885, 2886, 2825, 
         3174, 2195, 2826, 3116, 2770, 3198, 3197, 2890, 2618, 2128, 2785, 
         2120, 2786, 2130, 2622, 2623, 1933, 2395, 1941, 2726, 2727, 2646, 
         2043, 2800, 2801, 2798, 2134, 2799, 2459, 2460, 2262, 2635, 2636, 
         2637, 2478, 2959, 2958, 3136, 2814, 2957, 3298, 2956, 2047, 2657, 
         2815, 2816, 2818, 2817, 2481, 3213, 3214, 3212, 2270, 3215, 2149, 
         2488, 2977, 2976, 2277, 3228, 2157, 3229, 2278, 2978, 2979, 1953, 
         2496, 1961, 2734, 2735, 2733, 2059, 3246, 3247, 3244, 2285, 3245, 
         2504, 2505, 2286, 2991, 2992, 2993, 2506, 3001, 3000, 2999, 3159, 
         3158, 2063, 3011, 3160, 3161, 2168}],
        "0.85`"],
       Annotation[#, 0.85, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2177, 1993, 3164, 3017, 2067, 2176, 3303, 1974, 2522, 1971, 
         3300, 2299, 2171, 3267, 2298, 3007, 3008, 2513, 2167, 3262, 2292, 
         3261, 2291, 2512, 3258, 2290, 3257, 3260, 3259, 2164, 3242, 3243, 
         3241, 2284, 2497, 2820, 1990, 3332, 2819, 2663, 2058, 3150, 3151, 
         2989, 2990, 1960, 2495, 2975, 2974, 2973, 2276, 3227, 2156, 2971, 
         3336, 2275, 3226, 2972, 2487, 2148, 3211, 2269, 3209, 2268, 3210, 
         2479, 3133, 2147, 3132, 3135, 3134, 2135, 3205, 3206, 3202, 3204, 
         3203, 2396, 2797, 1988, 3313, 2796, 2654, 2042, 2794, 2795, 2632, 
         2944, 1939, 2456, 2621, 2620, 2619, 2255, 2896, 2118, 2445, 2125, 
         2784, 2126, 3120, 3119, 3316, 3315, 3115, 2769, 2824, 3172, 3173, 
         2823, 2194, 2822, 2222, 2861, 2231, 2875, 2232, 2876, 2230, 2877, 
         2234, 3191, 2233, 3335, 3043, 3044, 3045, 2754, 2755, 2753, 2757, 
         2756, 2102, 2432, 2433, 2429, 2428, 1930, 2409, 1927, 2607, 2606, 
         2764, 2763, 2112, 2557, 2558, 2342, 2600, 3179, 2833, 2343, 2927, 
         2363, 2851, 2220, 2701, 2702, 2703, 2552, 2012, 3041, 2704, 2859, 
         2858, 2688, 2377, 2420, 2419, 2026, 2560, 2027, 2374, 2694, 2869, 
         2870, 2714, 2715, 2716, 2550, 2588, 2335}],
        "0.8`"],
       Annotation[#, 0.8, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1999, 3030, 2676, 2918, 2917, 2200, 2867, 2868, 2864, 2866, 
         2865, 2009, 2373, 2375, 2206, 2372, 2371, 2418, 2003, 2680, 2679, 
         2839, 2838, 2207, 2855, 2856, 2852, 2854, 2853, 2573, 2589, 2590, 
         2344, 2601, 2337, 2591, 2336, 3282, 2338, 3032, 3034, 3035, 3033, 
         2840, 2841, 2765, 2767, 2766, 2085, 2349, 2348, 2431, 2031, 2386, 
         2708, 2444, 2116, 2874, 2608, 2609, 2392, 2935, 2934, 2611, 2124, 
         2780, 2117, 2895, 2254, 2614, 2615, 2616, 2455, 2943, 2942, 2628, 
         2132, 2793, 2129, 2904, 2261, 2633, 2634, 1940, 2476, 1945, 2953, 
         2908, 2952, 3297, 2951, 2145, 3207, 3208, 2954, 2955, 1946, 2485, 
         1951, 2968, 2967, 3221, 3220, 2154, 3224, 3225, 2969, 2970, 1952, 
         2493, 1958, 2986, 2985, 3238, 3237, 2162, 3239, 3240, 2987, 2988, 
         1959, 2503, 1966, 2998, 2997, 3157, 3156, 2062, 3010, 3252, 1991, 
         3253, 2511, 2289, 3254, 3256, 3255, 2166, 2295, 3323, 2294, 2521, 
         2296, 3324, 2297, 2170, 2308, 3337, 2307, 2528, 2309, 3326, 2310, 
         2175, 2316, 3339, 2315, 2531, 2317, 3328, 2318, 2180, 2327, 3341, 
         2326, 2535, 2328, 3331, 2329, 2186, 3171, 2189, 3170, 2188}],
        "0.75`"],
       Annotation[#, 0.75, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1984, 3309, 2192, 2078, 3099, 3101, 3100, 3169, 2539, 3278, 
         3279, 3277, 3281, 3280, 2185, 2325, 3330, 2324, 2534, 2322, 3340, 
         2323, 2179, 2314, 3327, 2313, 2530, 2311, 3338, 2312, 2527, 2526, 
         1973, 3302, 2306, 2174, 2305, 3301, 1972, 2520, 1970, 3299, 2293, 
         2169, 3265, 3266, 3005, 3006, 1969, 2510, 1965, 2996, 2669, 2668, 
         2667, 2061, 3148, 3149, 3147, 2163, 2494, 3143, 3144, 3142, 3146, 
         3145, 2155, 3223, 2274, 3222, 2273, 2486, 3140, 2152, 3139, 2153, 
         3141, 2146, 2910, 2267, 2909, 2266, 2477, 2133, 2813, 2143, 2641, 
         2949, 2950, 2474, 2631, 2630, 2629, 2260, 2903, 2127, 2791, 2792, 
         2624, 2941, 1938, 2394, 1932, 2613, 2612, 2783, 2782, 2036, 2453, 
         2454, 2391, 2390, 2443, 2574, 2382, 2383, 2249, 2385, 2384, 2002, 
         2843, 2208, 2842, 2209, 2844, 3184, 2850, 2219, 3187, 2218, 3186, 
         2922, 2340, 2921, 2924, 2923, 2086, 2561, 2366, 2367, 2100, 2836, 
         2837, 2835, 3183, 3182, 2559, 3050, 3049, 2717, 2229, 2873, 2197, 
         2403, 2083, 2551, 2361, 2362, 2413, 2412, 2415, 2092}],
        "0.7`"],
       Annotation[#, 0.7, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1983, 3308, 2191, 2077, 3096, 3098, 3097, 3168, 2538, 3274, 
         3275, 3273, 2331, 3276, 2184, 2321, 3329, 2320, 2533, 2183, 2319, 
         3305, 1978, 2529, 1976, 3020, 3019, 3018, 2069, 2173, 3318, 2172, 
         2525, 2303, 3325, 2304, 2519, 2518, 3004, 3003, 3002, 3264, 3263, 
         2066, 3078, 3080, 3079, 3155, 2509, 3249, 3250, 3248, 2288, 3251, 
         2161, 3236, 2283, 3235, 2282, 2502, 2160, 3234, 2281, 2983, 2984, 
         1957, 2492, 1950, 2966, 2660, 2659, 2658, 2052, 3130, 3131, 3128, 
         2144, 3129, 2475, 2810, 2141, 2809, 2142, 2472, 2471, 2627, 2626, 
         2625, 2902, 2901, 2041, 2651, 2725, 2724, 2781, 2393, 2774, 2775, 
         2773, 2122, 2776, 3114, 2768, 3180, 3181, 2205, 2834, 2204, 2919, 
         2920, 2678, 3031, 2381, 2030, 2562, 2563, 2347, 2849, 1928, 2398, 
         1925, 2603, 2602, 2700, 2699, 2011, 2423, 2010, 2698, 2697, 2099, 
         2872, 2228, 2871, 2227, 3190, 1996, 2670, 2400, 2081, 2549, 2082, 
         2402, 2681, 2745, 2089, 2414, 2684, 2005}],
        "0.65`"],
       Annotation[#, 0.65, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1982, 3307, 2190, 2076, 3094, 2075, 3095, 2187, 3167, 2536, 
         2537, 2330, 3306, 1980, 2532, 1977, 1994, 3022, 3021, 2072, 3092, 
         1975, 3304, 2178, 2068, 3086, 3088, 3087, 3163, 2524, 3272, 2301, 
         3271, 2302, 2517, 2516, 1968, 3016, 3015, 3014, 2065, 3153, 3154, 
         3152, 2165, 2507, 2508, 2287, 2994, 2995, 1964, 2501, 1956, 1989, 
         2662, 2661, 2057, 2732, 2965, 2964, 2151, 3138, 2051, 3072, 3073, 
         2812, 2811, 3127, 2473, 2808, 2139, 2807, 2140, 2469, 2468, 1937, 
         2940, 2653, 2652, 2650, 2040, 2778, 2779, 2777, 2123, 2440, 2115, 
         2772, 2121, 2610, 2932, 1931, 2389, 1929, 1985, 2643, 2642, 2193, 
         2821, 2029, 2380, 2674, 2913, 2080, 2399, 2438, 2437, 2571, 2572, 
         2370, 2369, 2594, 2096, 2829, 2830, 2671, 3028, 3029, 2404, 2405, 
         2196, 2682, 2828, 3036, 2827, 2401, 2004, 2744, 2093, 2741, 2094, 
         2411, 2410, 2436}],
        "0.6`"],
       Annotation[#, 0.6, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1923, 2929, 3103, 2087, 3104, 2088, 2742, 2743, 3106, 3105, 
         2091, 2740, 3342, 2739, 2683, 2090, 2738, 2364, 2862, 2863, 2711, 
         2712, 2017, 2580, 2016, 2581, 2582, 2593, 2592, 2368, 2339, 3320, 
         2210, 2587, 2359, 2360, 2916, 2915, 1998, 3314, 2914, 2675, 3178, 
         3177, 2113, 3112, 2114, 3113, 2035, 2452, 2354, 2355, 2933, 2723, 
         2039, 2648, 2649, 1987, 1936, 2466, 2467, 2138, 3124, 2137, 3125, 
         2470, 3126, 3068, 3069, 3067, 3071, 3070, 3137, 2150, 3075, 2055, 
         3074, 2056, 3076, 2158, 3317, 2159, 2060, 2664, 2665, 2666, 1963, 
         3077, 2064, 3012, 3013, 1992, 1967, 2514, 2515, 2300, 3268, 3270, 
         3269, 2523, 3162, 3082, 3083, 3081, 3085, 3084, 3165, 3166, 3090, 
         2070, 3089, 2071, 3091, 2181, 3319, 2182, 2073, 3023, 3024, 3025, 
         1979, 3093, 2074, 3026, 3027, 1995, 1981}],
        "0.55`"],
       Annotation[#, 0.55, "Tooltip"]& ]}, {}, {}, {}}}],
  AspectRatio->1,
  Frame->True,
  ImageMargins->0.,
  ImageSize->{225, Automatic},
  PlotRange->NCache[{{0, 3}, {
      Rational[1, 10], 3}}, {{0, 3}, {0.1, 3}}],
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{229, 224},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->36265762]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->24353],

Cell["The probability density function integrates to unity:", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->16061],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"PDF", "[", 
    RowBox[{
     RowBox[{"LogisticDistribution", "[", 
      RowBox[{"\[Mu]", ",", "\[Beta]"}], "]"}], ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "\[Infinity]"}], ",", "\[Infinity]"}], "}"}], ",", 
   RowBox[{"Assumptions", "->", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"Im", "[", "\[Mu]", "]"}], "==", "0"}], ",", 
      RowBox[{"\[Beta]", ">", "0"}]}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->3904],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->790652750]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->17571],

Cell["Moments can be obtained from the characteristic function:", \
"ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->3912],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ExpectedValue", "[", 
  RowBox[{
   RowBox[{"x", "^", "4"}], ",", 
   RowBox[{"LogisticDistribution", "[", 
    RowBox[{"0", ",", "2"}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->16405],

Cell[BoxData[
 FractionBox[
  RowBox[{"112", " ", 
   SuperscriptBox["\[Pi]", "4"]}], "15"]], "Output",
 ImageSize->{46, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->44923017]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"cf", "=", 
  RowBox[{"CharacteristicFunction", "[", 
   RowBox[{
    RowBox[{"LogisticDistribution", "[", 
     RowBox[{"0", ",", "2"}], "]"}], ",", "t"}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->59192641],

Cell[BoxData[
 RowBox[{"2", " ", "\[Pi]", " ", "t", " ", 
  RowBox[{"Csch", "[", 
   RowBox[{"2", " ", "\[Pi]", " ", "t"}], "]"}]}]], "Output",
 ImageSize->{105, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->103553116]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Limit", "[", 
  RowBox[{
   RowBox[{"D", "[", 
    RowBox[{"cf", ",", 
     RowBox[{"{", 
      RowBox[{"t", ",", "4"}], "}"}]}], "]"}], ",", 
   RowBox[{"t", "->", "0"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->22056],

Cell[BoxData[
 FractionBox[
  RowBox[{"112", " ", 
   SuperscriptBox["\[Pi]", "4"]}], "15"]], "Output",
 ImageSize->{46, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->374236841]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[Cell[BoxData[
   InterpretationBox[Cell[TextData[{
     "Possible Issues",
     "\[NonBreakingSpace]\[NonBreakingSpace]",
     Cell["(2)", "ExampleCount"]
    }], "ExampleSection"],
    $Line = 0; Null]]]], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1267],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["LogisticDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LogisticDistribution"]], "InlineFormula"],
 " is not defined when ",
 Cell[BoxData[
  StyleBox["\[Mu]", "TR"]], "InlineFormula"],
 " is not a real number:"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->3751],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"LogisticDistribution", "[", 
   RowBox[{
    RowBox[{"3", "+", "I"}], ",", "4"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2019],

Cell[BoxData[
 RowBox[{
  RowBox[{"LogisticDistribution", "::", "\<\"realparm\"\>"}], ":", 
  " ", "\<\"Parameter \\!\\(3 + \[ImaginaryI]\\) is expected to be \
real.\"\>"}]], "Message", "MSG",
 CellID->251088736],

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"LogisticDistribution", "[", 
   RowBox[{
    RowBox[{"3", "+", "\[ImaginaryI]"}], ",", "4"}], "]"}], "]"}]], "Output",
 ImageSize->{246, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->465520786]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["LogisticDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LogisticDistribution"]], "InlineFormula"],
 " is not defined when ",
 Cell[BoxData[
  StyleBox["\[Beta]", "TR"]], "InlineFormula"],
 " is not a positive real number:"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->5717],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"LogisticDistribution", "[", 
   RowBox[{"3", ",", 
    RowBox[{"-", "4"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->25846],

Cell[BoxData[
 RowBox[{
  RowBox[{"LogisticDistribution", "::", "\<\"posparm\"\>"}], ":", 
  " ", "\<\"Parameter \\!\\(-4\\) is expected to be positive.\"\>"}]], \
"Message", "MSG",
 CellID->110400633],

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"LogisticDistribution", "[", 
   RowBox[{"3", ",", 
    RowBox[{"-", "4"}]}], "]"}], "]"}]], "Output",
 ImageSize->{233, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->9687584]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->19046],

Cell["\<\
Substitution of invalid parameters into symbolic outputs gives results that \
are not meaningful:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->16727],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Mean", "[", 
   RowBox[{"LogisticDistribution", "[", 
    RowBox[{"\[Mu]", ",", "\[Beta]"}], "]"}], "]"}], "/.", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"\[Mu]", "\[Rule]", 
     RowBox[{"2", "I"}]}], ",", 
    RowBox[{"\[Beta]", "\[Rule]", 
     RowBox[{"3", "+", "I"}]}]}], "}"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->25838],

Cell[BoxData[
 RowBox[{"2", " ", "\[ImaginaryI]"}]], "Output",
 ImageSize->{22, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->264656321]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->24618],

Cell[TextData[ButtonBox["Continuous Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/ContinuousDistributions"]], "Tutorials",
 CellID->7471]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Continuous Statistical Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/ContinuousStatisticalDistributions"]], "MoreAbout",\

 CellID->125755044]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"LogisticDistribution - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 58, 45.3363932}", "context" -> "System`", 
    "keywords" -> {"sech squared distribution", "sech square distribution"}, 
    "index" -> True, "label" -> "Built-in Mathematica Symbol", "language" -> 
    "en", "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "LogisticDistribution[\\[Mu], \\[Beta]] represents a logistic \
distribution with mean \\[Mu] and scale parameter \\[Beta].", 
    "synonyms" -> {"logistic distribution"}, "title" -> 
    "LogisticDistribution", "type" -> "Symbol", "uri" -> 
    "ref/LogisticDistribution"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[6810, 224, 358, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->4094141]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 124357, 2596}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 1286, 34, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[1909, 61, 1585, 37, 70, "ObjectNameGrid"],
Cell[3497, 100, 756, 23, 70, "Usage",
 CellID->28313]
}, Open  ]],
Cell[CellGroupData[{
Cell[4290, 128, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[4768, 145, 770, 27, 70, "Notes",
 CellID->429991538],
Cell[5541, 174, 531, 17, 70, "Notes",
 CellID->22312],
Cell[6075, 193, 698, 26, 70, "Notes",
 CellID->33143606]
}, Closed]],
Cell[CellGroupData[{
Cell[6810, 224, 358, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->4094141],
Cell[CellGroupData[{
Cell[7193, 238, 148, 5, 70, "ExampleSection",
 CellID->517378609],
Cell[7344, 245, 271, 6, 70, "ExampleText",
 CellID->990],
Cell[CellGroupData[{
Cell[7640, 255, 176, 5, 28, "Input",
 CellID->3873],
Cell[7819, 262, 161, 5, 36, "Output",
 CellID->66471315]
}, Open  ]],
Cell[CellGroupData[{
Cell[8017, 272, 180, 5, 28, "Input",
 CellID->3413],
Cell[8200, 279, 260, 9, 54, "Output",
 CellID->52987697]
}, Open  ]],
Cell[8475, 291, 121, 3, 70, "ExampleDelimiter",
 CellID->31896],
Cell[8599, 296, 253, 6, 70, "ExampleText",
 CellID->12472],
Cell[CellGroupData[{
Cell[8877, 306, 201, 6, 70, "Input",
 CellID->24450],
Cell[9081, 314, 535, 19, 82, "Output",
 CellID->394144508]
}, Open  ]],
Cell[CellGroupData[{
Cell[9653, 338, 331, 11, 28, "Input",
 CellID->4641],
Cell[9987, 351, 3372, 59, 176, "Output",
 Evaluatable->False,
 CellID->412329999]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[13408, 416, 216, 7, 70, "ExampleSection",
 CellID->28213],
Cell[13627, 425, 257, 7, 70, "ExampleText",
 CellID->16824],
Cell[CellGroupData[{
Cell[13909, 436, 198, 6, 70, "Input",
 CellID->17929],
Cell[14110, 444, 361, 9, 36, "Output",
 CellID->82500071]
}, Open  ]],
Cell[14486, 456, 121, 3, 70, "ExampleDelimiter",
 CellID->25403],
Cell[14610, 461, 99, 3, 70, "ExampleText",
 CellID->20699375],
Cell[CellGroupData[{
Cell[14734, 468, 181, 5, 70, "Input",
 CellID->12594],
Cell[14918, 475, 157, 5, 36, "Output",
 CellID->83786101]
}, Open  ]],
Cell[CellGroupData[{
Cell[15112, 485, 181, 5, 70, "Input",
 CellID->22187],
Cell[15296, 492, 179, 6, 51, "Output",
 CellID->239233800]
}, Open  ]],
Cell[15490, 501, 121, 3, 70, "ExampleDelimiter",
 CellID->25079],
Cell[15614, 506, 376, 13, 70, "ExampleText",
 CellID->10041],
Cell[CellGroupData[{
Cell[16015, 523, 206, 6, 70, "Input",
 CellID->18125],
Cell[16224, 531, 436, 13, 36, "Output",
 CellID->596618132]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[16709, 550, 222, 7, 70, "ExampleSection",
 CellID->4907],
Cell[16934, 559, 112, 3, 70, "ExampleText",
 CellID->29591],
Cell[CellGroupData[{
Cell[17071, 566, 312, 10, 70, "Input",
 CellID->25867],
Cell[17386, 578, 2195, 46, 170, "Output",
 CellID->314678191]
}, Open  ]],
Cell[19596, 627, 125, 3, 70, "ExampleDelimiter",
 CellID->226342353],
Cell[19724, 632, 361, 13, 70, "ExampleText",
 CellID->26444],
Cell[CellGroupData[{
Cell[20110, 649, 426, 13, 70, "Input",
 CellID->10654],
Cell[20539, 664, 94612, 1586, 70, "Output",
 CellID->36265762]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[115200, 2256, 233, 7, 70, "ExampleSection",
 CellID->24353],
Cell[115436, 2265, 229, 5, 70, "ExampleText",
 CellID->16061],
Cell[CellGroupData[{
Cell[115690, 2274, 561, 17, 70, "Input",
 CellID->3904],
Cell[116254, 2293, 158, 5, 36, "Output",
 CellID->790652750]
}, Open  ]],
Cell[116427, 2301, 121, 3, 70, "ExampleDelimiter",
 CellID->17571],
Cell[116551, 2306, 234, 6, 70, "ExampleText",
 CellID->3912],
Cell[CellGroupData[{
Cell[116810, 2316, 235, 7, 70, "Input",
 CellID->16405],
Cell[117048, 2325, 232, 8, 54, "Output",
 CellID->44923017]
}, Open  ]],
Cell[CellGroupData[{
Cell[117317, 2338, 240, 7, 70, "Input",
 CellID->59192641],
Cell[117560, 2347, 274, 8, 36, "Output",
 CellID->103553116]
}, Open  ]],
Cell[CellGroupData[{
Cell[117871, 2360, 253, 9, 70, "Input",
 CellID->22056],
Cell[118127, 2371, 233, 8, 54, "Output",
 CellID->374236841]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[118409, 2385, 326, 9, 70, "ExampleSection",
 CellID->1267],
Cell[118738, 2396, 442, 14, 70, "ExampleText",
 CellID->3751],
Cell[CellGroupData[{
Cell[119205, 2414, 191, 6, 70, "Input",
 CellID->2019],
Cell[119399, 2422, 213, 5, 70, "Message",
 CellID->251088736],
Cell[119615, 2429, 297, 9, 36, "Output",
 CellID->465520786]
}, Open  ]],
Cell[119927, 2441, 453, 14, 70, "ExampleText",
 CellID->5717],
Cell[CellGroupData[{
Cell[120405, 2459, 187, 6, 70, "Input",
 CellID->25846],
Cell[120595, 2467, 201, 5, 70, "Message",
 CellID->110400633],
Cell[120799, 2474, 278, 9, 36, "Output",
 CellID->9687584]
}, Open  ]],
Cell[121092, 2486, 121, 3, 70, "ExampleDelimiter",
 CellID->19046],
Cell[121216, 2491, 281, 8, 70, "ExampleText",
 CellID->16727],
Cell[CellGroupData[{
Cell[121522, 2503, 369, 12, 70, "Input",
 CellID->25838],
Cell[121894, 2517, 192, 6, 36, "Output",
 CellID->264656321]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[122147, 2530, 314, 9, 70, "TutorialsSection",
 CellID->24618],
Cell[122464, 2541, 157, 3, 70, "Tutorials",
 CellID->7471]
}, Open  ]],
Cell[CellGroupData[{
Cell[122658, 2549, 299, 8, 70, "MoreAboutSection"],
Cell[122960, 2559, 184, 4, 70, "MoreAbout",
 CellID->125755044]
}, Open  ]],
Cell[123159, 2566, 27, 0, 70, "History"],
Cell[123189, 2568, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

