(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     48188,       1795]
NotebookOptionsPosition[     38229,       1436]
NotebookOutlinePosition[     39703,       1481]
CellTagsIndexPosition[     39588,       1475]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Combinatorial Functions" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/CombinatorialFunctions"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Combinatorial Functions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Fibonacci" :> Documentation`HelpLookup["paclet:ref/Fibonacci"],
           "GoldenRatio" :> 
          Documentation`HelpLookup["paclet:ref/GoldenRatio"], "RSolve" :> 
          Documentation`HelpLookup["paclet:ref/RSolve"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Fibonacci\"\>", 2->"\<\"GoldenRatio\"\>", 
       3->"\<\"RSolve\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Integer Functions" :> 
          Documentation`HelpLookup["paclet:guide/IntegerFunctions"], 
          "Recurrence and Sum Functions" :> 
          Documentation`HelpLookup["paclet:guide/RecurrenceAndSumFunctions"], 
          "New in 6.0: Mathematical Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicalFunctions"], 
          "New in 6.0: Number Theory & Integer Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60NumberTheoryAndIntegerFunctions"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Integer Functions\"\>", 
       2->"\<\"Recurrence and Sum Functions\"\>", 
       3->"\<\"New in 6.0: Mathematical Functions\"\>", 
       4->"\<\"New in 6.0: Number Theory & Integer Functions\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["LucasL", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["LucasL",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/LucasL"], "[", 
       StyleBox["n", "TI"], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the Lucas number ",
     Cell[BoxData[
      FormBox[
       SubscriptBox["L", "n"], TraditionalForm]], "InlineMath"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["\<\
Mathematical function, suitable for both symbolic and numerical manipulation.\
\
\>", "Notes",
 CellID->20640],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["L", "n"], TraditionalForm]], "InlineMath"],
 " satisfy the recurrence relation ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["L", "n"], "=", 
    RowBox[{
     SubscriptBox["L", 
      RowBox[{"n", "-", "1"}]], "+", 
     SubscriptBox["L", 
      RowBox[{"n", "-", "2"}]]}]}], TraditionalForm]], "InlineMath"],
 " with ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["L", "1"], "=", "1"}], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["L", "2"], "=", "3"}], TraditionalForm]], "InlineMath"],
 "."
}], "Notes",
 CellID->475],

Cell[TextData[{
 "For any complex value of ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " the ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["L", "n"], TraditionalForm]], "InlineMath"],
 " are given by the general formula ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["L", "n"], "=", 
    RowBox[{"(", 
     RowBox[{
      SuperscriptBox["\[Phi]", "n"], "+", 
      RowBox[{
       RowBox[{"cos", "(", 
        RowBox[{"\[Pi]", " ", "n"}], ")"}], 
       SuperscriptBox["\[Phi]", 
        RowBox[{"-", "n"}]]}]}], ")"}]}], TraditionalForm]], "InlineMath"],
 ", where ",
 Cell[BoxData[
  FormBox["\[Phi]", TraditionalForm]], "InlineMath"],
 " is the golden ratio. "
}], "Notes",
 CellID->1255],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["LucasL",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LucasL"]], "InlineFormula"],
 " automatically threads over lists."
}], "Notes",
 CellID->28162]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->47268906],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(1)", "ExampleCount"]
}], "ExampleSection",
 CellID->302918085],

Cell["Compute Lucas numbers:", "ExampleText",
 CellID->625931072],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"LucasL", "[", "n", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "20"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->227574515],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1", ",", "3", ",", "4", ",", "7", ",", "11", ",", "18", ",", "29", ",", 
   "47", ",", "76", ",", "123", ",", "199", ",", "322", ",", "521", ",", 
   "843", ",", "1364", ",", "2207", ",", "3571", ",", "5778", ",", "9349", 
   ",", "15127"}], "}"}]], "Output",
 ImageSize->{348, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->6710460]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(7)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1293636265],

Cell["Evaluate large Lucas numbers:", "ExampleText",
 CellID->21966],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LucasL", "[", "1000", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->30804],

Cell[BoxData[\
"97194177735908175207981982079326473737797879155345685082728081084772518818444\
815269080619149045968297679578305403209347401163036907660573971740862463751801\
641201490284097309096322681531675707666695323797578127"], "Output",
 ImageSize->{583, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->744611745]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->5025],

Cell["Lucas numbers of negative argument:", "ExampleText",
 CellID->423],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"LucasL", "[", 
    RowBox[{"-", "n"}], "]"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"n", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->15809],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "1"}], ",", "3", ",", 
   RowBox[{"-", "4"}], ",", "7", ",", 
   RowBox[{"-", "11"}], ",", "18", ",", 
   RowBox[{"-", "29"}], ",", "47", ",", 
   RowBox[{"-", "76"}], ",", "123"}], "}"}]], "Output",
 ImageSize->{277, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->70483682]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->7030],

Cell["Non-integer arguments:", "ExampleText",
 CellID->23375],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LucasL", "[", "1.5", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->31384],

Cell[BoxData["2.0581710272714924`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->15118738]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"LucasL", "[", 
    RowBox[{"3", "/", "2"}], "]"}], ",", "50"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1326],

Cell[BoxData["2.\
0581710272714922503219810475804504212387300996778196449639794643526`50."], \
"Output",
 ImageSize->{361, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->74873172]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->11923],

Cell["Complex arguments:", "ExampleText",
 CellID->21468],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LucasL", "[", 
  RowBox[{"1.5", "+", "I"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->26278],

Cell[BoxData[
 RowBox[{"4.421583233252993`", "\[InvisibleSpace]", "+", 
  RowBox[{"5.926564258367858`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{126, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->861489839]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->2398],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["LucasL",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LucasL"]], "InlineFormula"],
 " threads element-wise over lists:"
}], "ExampleText",
 CellID->22387],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LucasL", "[", 
  RowBox[{"{", 
   RowBox[{"1", ",", "2", ",", "3", ",", "4", ",", "5"}], "}"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->24086],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1", ",", "3", ",", "4", ",", "7", ",", "11"}], "}"}]], "Output",
 ImageSize->{105, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->21649052]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->11161],

Cell["Series expansion at generic point:", "ExampleText",
 CellID->7541],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Series", "[", 
   RowBox[{
    RowBox[{"LucasL", "[", "n", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"n", ",", "5", ",", "2"}], "}"}]}], "]"}], "//", 
  "FullSimplify"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->5915],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"11", "+", 
   RowBox[{"5", " ", 
    SqrtBox["5"], " ", 
    RowBox[{"ArcCsch", "[", "2", "]"}], " ", 
    RowBox[{"(", 
     RowBox[{"n", "-", "5"}], ")"}]}], "+", 
   FractionBox[
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       SuperscriptBox["\[Pi]", "2"], "+", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          RowBox[{"-", "1"}], "+", 
          SuperscriptBox["GoldenRatio", "10"]}], ")"}], " ", 
        SuperscriptBox[
         RowBox[{"ArcCsch", "[", "2", "]"}], "2"]}]}], ")"}], " ", 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"n", "-", "5"}], ")"}], "2"]}], 
    RowBox[{"2", " ", 
     SuperscriptBox["GoldenRatio", "5"]}]], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", 
      RowBox[{"n", "-", "5"}], "]"}], "3"],
    SeriesData[$CellContext`n, 5, {}, 0, 3, 1],
    Editable->False]}],
  SeriesData[$CellContext`n, 5, {
   11, 5 5^Rational[1, 2] ArcCsch[2], Rational[1, 2] 
    GoldenRatio^(-5) (Pi^2 + (-1 + GoldenRatio^10) ArcCsch[2]^2)}, 0, 3, 1],
  Editable->False]], "Output",
 ImageSize->{561, 37},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->640733637]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->804171535],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TraditionalForm",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TraditionalForm"]], "InlineFormula"],
 " formatting:"
}], "ExampleText",
 CellID->126097607],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"LucasL", "[", "n", "]"}], "//", "TraditionalForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->619062489],

Cell[BoxData[
 FormBox[
  SubscriptBox[
   TagBox["L",
    LucasL], "n"], TraditionalForm]], "Output",
 ImageSize->{17, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//TraditionalForm=",
 CellID->72509466]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->258228157],

Cell["Solve the Fibonacci recurrence equation:", "ExampleText",
 CellID->11321],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"f", "[", "n", "]"}], "==", 
      RowBox[{
       RowBox[{"f", "[", 
        RowBox[{"n", "-", "1"}], "]"}], "+", 
       RowBox[{"f", "[", 
        RowBox[{"n", "-", "2"}], "]"}]}]}], ",", 
     RowBox[{
      RowBox[{"f", "[", "1", "]"}], "\[Equal]", "a"}], ",", 
     RowBox[{
      RowBox[{"f", "[", "2", "]"}], "\[Equal]", "b"}]}], "}"}], ",", 
   RowBox[{"f", "[", "n", "]"}], ",", "n"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->10265],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"f", "[", "n", "]"}], "\[Rule]", 
    RowBox[{
     FractionBox["1", "2"], " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"3", " ", "a", " ", 
        RowBox[{"Fibonacci", "[", "n", "]"}]}], "-", 
       RowBox[{"b", " ", 
        RowBox[{"Fibonacci", "[", "n", "]"}]}], "-", 
       RowBox[{"a", " ", 
        RowBox[{"LucasL", "[", "n", "]"}]}], "+", 
       RowBox[{"b", " ", 
        RowBox[{"LucasL", "[", "n", "]"}]}]}], ")"}]}]}], "}"}], 
  "}"}]], "Output",
 ImageSize->{486, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->205190814]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FullSimplify", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"First", "[", 
     RowBox[{
      RowBox[{"f", "[", "n", "]"}], "/.", "%"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"n", ",", "10"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->32369],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"a", ",", "b", ",", 
   RowBox[{"a", "+", "b"}], ",", 
   RowBox[{"a", "+", 
    RowBox[{"2", " ", "b"}]}], ",", 
   RowBox[{
    RowBox[{"2", " ", "a"}], "+", 
    RowBox[{"3", " ", "b"}]}], ",", 
   RowBox[{
    RowBox[{"3", " ", "a"}], "+", 
    RowBox[{"5", " ", "b"}]}], ",", 
   RowBox[{
    RowBox[{"5", " ", "a"}], "+", 
    RowBox[{"8", " ", "b"}]}], ",", 
   RowBox[{
    RowBox[{"8", " ", "a"}], "+", 
    RowBox[{"13", " ", "b"}]}], ",", 
   RowBox[{
    RowBox[{"13", " ", "a"}], "+", 
    RowBox[{"21", " ", "b"}]}], ",", 
   RowBox[{
    RowBox[{"21", " ", "a"}], "+", 
    RowBox[{"34", " ", "b"}]}]}], "}"}]], "Output",
 ImageSize->{511, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->344988741]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->113800316],

Cell["Find ratios of successive Lucas numbers:", "ExampleText",
 CellID->29408],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"LucasL", "[", 
     RowBox[{"n", "+", "1"}], "]"}], "/", 
    RowBox[{"LucasL", "[", "n", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "15"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1563],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"3", ",", 
   FractionBox["4", "3"], ",", 
   FractionBox["7", "4"], ",", 
   FractionBox["11", "7"], ",", 
   FractionBox["18", "11"], ",", 
   FractionBox["29", "18"], ",", 
   FractionBox["47", "29"], ",", 
   FractionBox["76", "47"], ",", 
   FractionBox["123", "76"], ",", 
   FractionBox["199", "123"], ",", 
   FractionBox["322", "199"], ",", 
   FractionBox["521", "322"], ",", 
   FractionBox["843", "521"], ",", 
   FractionBox["1364", "843"], ",", 
   FractionBox["2207", "1364"]}], "}"}]], "Output",
 ImageSize->{485, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->358058281]
}, Open  ]],

Cell["Compare with continued fractions:", "ExampleText",
 CellID->13103],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"FromContinuedFraction", "[", 
    RowBox[{"PadLeft", "[", 
     RowBox[{
      RowBox[{"{", "3", "}"}], ",", "n", ",", "1"}], "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "15"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->25323],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"3", ",", 
   FractionBox["4", "3"], ",", 
   FractionBox["7", "4"], ",", 
   FractionBox["11", "7"], ",", 
   FractionBox["18", "11"], ",", 
   FractionBox["29", "18"], ",", 
   FractionBox["47", "29"], ",", 
   FractionBox["76", "47"], ",", 
   FractionBox["123", "76"], ",", 
   FractionBox["199", "123"], ",", 
   FractionBox["322", "199"], ",", 
   FractionBox["521", "322"], ",", 
   FractionBox["843", "521"], ",", 
   FractionBox["1364", "843"], ",", 
   FractionBox["2207", "1364"]}], "}"}]], "Output",
 ImageSize->{485, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->7080173]
}, Open  ]],

Cell["Convergence to the Golden Ratio:", "ExampleText",
 CellID->31752],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", "%", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->6867],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "3.`", ",", "1.3333333333333333`", ",", "1.75`", ",", "1.5714285714285714`",
    ",", "1.6363636363636365`", ",", "1.6111111111111112`", ",", 
   "1.6206896551724137`", ",", "1.6170212765957446`", ",", 
   "1.618421052631579`", ",", "1.6178861788617886`", ",", 
   "1.6180904522613064`", ",", "1.6180124223602483`", ",", 
   "1.618042226487524`", ",", "1.6180308422301304`", ",", 
   "1.6180351906158357`"}], "}"}]], "Output",
 ImageSize->{433, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->155967553]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->5556],

Cell[TextData[{
 "Calculate the number of ways to write an integer as a sum of Lucas numbers \
",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["L", "1"], ",", 
    SubscriptBox["L", "2"], ",", "\[Ellipsis]"}], TraditionalForm]], 
  "InlineMath"],
 ":"
}], "ExampleText",
 CellID->11617],

Cell[BoxData[
 RowBox[{
  RowBox[{"lucasSumCount", "[", "n_", "]"}], ":=", 
  RowBox[{"SeriesCoefficient", "[", 
   RowBox[{
    RowBox[{"Series", "[", 
     RowBox[{
      RowBox[{"Product", "[", 
       RowBox[{
        RowBox[{"1", "+", 
         RowBox[{"z", "^", 
          RowBox[{"LucasL", "[", "k", "]"}]}]}], ",", 
        RowBox[{"{", 
         RowBox[{"k", ",", 
          RowBox[{
           RowBox[{"Ceiling", "[", 
            RowBox[{"Log", "[", 
             RowBox[{"GoldenRatio", ",", "n"}], "]"}], "]"}], "+", "2"}]}], 
         "}"}]}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"z", ",", "0", ",", "n"}], "}"}]}], "]"}], ",", "n"}], 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->29865],

Cell["Plot the counts for the first hundred integers:", "ExampleText",
 CellID->14612],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"lucasSumCount", "[", "n", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"n", ",", "100"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->430777473],

Cell[BoxData[
 GraphicsBox[{{}, 
   {Hue[0.67, 0.6, 0.6], PointBox[CompressedData["
1:eJxdlMlKQ0EQRQtXLl24cOFCRUREJM6zKaMxcYr6NA5b0bW/0J/Wn+QnqFjn
Qd2GUBzuvV3V3eHNv381nxNm9vH7+6v/67ubq7mlNelZn/LsmxZ9RvKz4p8T
XhD/ouy35HmOZcmvSH5V9DXJd2T/dclviH9TeEv829JvJyr3tCv6nuT3Zf8D
4UPxHwUz/7H0O5F8V/Ke9QLzjqd53tJyrF7uV4TtLPtLy1HPc78CM29f8n3R
L6Q/zH0MJD/IfhtKfpj9dpn7FZj/9ZXoMOs6mPMJ243M13LU29yvwMw/kvxI
9DvJw+x/n/MOM69wFb89BMf5Haaf6BVmNXleFy5N9leY933M8zlMHiYvuj1F
je+Sw7wnTB7m/ceZfZz9Beb8MN+9Z5kfJi96hfmOvuCPfMuRh2O+Kmyv2e8w
9wfHvFV0e2vzP0oeaH4=
     "]]}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  PlotRange->{{0., 100.}, {0., 8.}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 123},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->93145189]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->11112],

Cell["Find the first Lucas number above 1000000:", "ExampleText",
 CellID->17425],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NestWhile", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{"#", "+", "1"}], ")"}], "&"}], ",", "1", ",", 
   RowBox[{
    RowBox[{
     RowBox[{"LucasL", "[", "#", "]"}], "<=", 
     RowBox[{"10", "^", "6"}]}], "&"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->28156],

Cell[BoxData["29"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1024475]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->32664],

Cell["Expand in terms of elementary functions:", "ExampleText",
 CellID->14367],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FunctionExpand", "[", 
  RowBox[{"LucasL", "[", "n", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->5783],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{
     FractionBox["1", "2"], " ", 
     RowBox[{"(", 
      RowBox[{"1", "+", 
       SqrtBox["5"]}], ")"}]}], ")"}], "n"], "+", 
  RowBox[{
   SuperscriptBox[
    RowBox[{"(", 
     FractionBox["2", 
      RowBox[{"1", "+", 
       SqrtBox["5"]}]], ")"}], "n"], " ", 
   RowBox[{"Cos", "[", 
    RowBox[{"n", " ", "\[Pi]"}], "]"}]}]}]], "Output",
 ImageSize->{231, 37},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->210422843]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FullSimplify", "[", 
  RowBox[{"Table", "[", 
   RowBox[{"%", ",", 
    RowBox[{"{", 
     RowBox[{"n", ",", "10"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->22744],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1", ",", "3", ",", "4", ",", "7", ",", "11", ",", "18", ",", "29", ",", 
   "47", ",", "76", ",", "123"}], "}"}]], "Output",
 ImageSize->{238, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->23793776]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->5521],

Cell["Explicit recursive definition:", "ExampleText",
 CellID->31328],

Cell[BoxData[
 RowBox[{
  RowBox[{"f", "[", "n_", "]"}], ":=", 
  RowBox[{
   RowBox[{"f", "[", "n", "]"}], "=", 
   RowBox[{
    RowBox[{"f", "[", 
     RowBox[{"n", "-", "1"}], "]"}], "+", 
    RowBox[{"f", "[", 
     RowBox[{"n", "-", "2"}], "]"}]}]}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->16041],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"f", "[", "1", "]"}], "=", "2"}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"f", "[", "2", "]"}], "=", "1"}], ";"}]}], "Input",
 CellLabel->"In[2]:=",
 CellID->992],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"f", "[", "n", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "20"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->32629],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "2", ",", "1", ",", "3", ",", "4", ",", "7", ",", "11", ",", "18", ",", 
   "29", ",", "47", ",", "76", ",", "123", ",", "199", ",", "322", ",", "521",
    ",", "843", ",", "1364", ",", "2207", ",", "3571", ",", "5778", ",", 
   "9349"}], "}"}]], "Output",
 ImageSize->{581, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->227930339]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->12267],

Cell["Simplify some expressions involving Lucas numbers:", "ExampleText",
 CellID->28825],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FullSimplify", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"LucasL", "[", 
      RowBox[{"n", "+", "1"}], "]"}], 
     RowBox[{"LucasL", "[", 
      RowBox[{"n", "-", "1"}], "]"}]}], "-", 
    SuperscriptBox[
     RowBox[{"LucasL", "[", "n", "]"}], "2"]}], ",", 
   RowBox[{
    RowBox[{"n", ">", "0"}], "&&", 
    RowBox[{"n", "\[Element]", "Integers"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19977],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "5"}], " ", 
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"-", "1"}], ")"}], "n"]}]], "Output",
 ImageSize->{56, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->91173419]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->32717],

Cell["Generating function:", "ExampleText",
 CellID->32653],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sum", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"LucasL", "[", "k", "]"}], " ", 
    SuperscriptBox["t", "k"]}], ",", 
   RowBox[{"{", 
    RowBox[{"k", ",", "Infinity"}], "}"}]}], "]"}]], "Input",
 CellTags->"Fibonacci",
 CellLabel->"In[1]:=",
 CellID->28134],

Cell[BoxData[
 FractionBox[
  RowBox[{
   RowBox[{"-", "t"}], "-", 
   RowBox[{"2", " ", 
    SuperscriptBox["t", "2"]}]}], 
  RowBox[{
   RowBox[{"-", "1"}], "+", "t", "+", 
   SuperscriptBox["t", "2"]}]]], "Output",
 ImageSize->{68, 34},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"Fibonacci",
 CellLabel->"Out[1]=",
 CellID->401717101]
}, Open  ]],

Cell["Extract Lucas numbers as coefficients:", "ExampleText",
 CellID->10858],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{"%", ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->32494],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"t", "+", 
   RowBox[{"3", " ", 
    SuperscriptBox["t", "2"]}], "+", 
   RowBox[{"4", " ", 
    SuperscriptBox["t", "3"]}], "+", 
   RowBox[{"7", " ", 
    SuperscriptBox["t", "4"]}], "+", 
   RowBox[{"11", " ", 
    SuperscriptBox["t", "5"]}], "+", 
   RowBox[{"18", " ", 
    SuperscriptBox["t", "6"]}], "+", 
   RowBox[{"29", " ", 
    SuperscriptBox["t", "7"]}], "+", 
   RowBox[{"47", " ", 
    SuperscriptBox["t", "8"]}], "+", 
   RowBox[{"76", " ", 
    SuperscriptBox["t", "9"]}], "+", 
   RowBox[{"123", " ", 
    SuperscriptBox["t", "10"]}], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "t", "]"}], "11"],
    SeriesData[$CellContext`t, 0, {}, 1, 11, 1],
    Editable->False]}],
  SeriesData[$CellContext`t, 0, {1, 3, 4, 7, 11, 18, 29, 47, 76, 123}, 1, 11, 
   1],
  Editable->False]], "Output",
 ImageSize->{433, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->7829229]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1305812373],

Cell["\<\
Large arguments can give results too large to be computed explicitly:\
\>", "ExampleText",
 CellID->10552],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LucasL", "[", 
  RowBox[{"10", "^", "12."}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->28284],

Cell[BoxData[
 RowBox[{"Overflow", "[", "]"}]], "Output",
 ImageSize->{74, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->452152549]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->13255],

Cell["\<\
Results for integer arguments may not hold for non-integers: \
\>", "ExampleText",
 CellID->5540],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FullSimplify", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"2", 
     RowBox[{"LucasL", "[", 
      RowBox[{"2", "n"}], "]"}]}], "\[Equal]", 
    RowBox[{
     RowBox[{"5", 
      SuperscriptBox[
       RowBox[{"Fibonacci", "[", "n", "]"}], "2"]}], "+", 
     SuperscriptBox[
      RowBox[{"LucasL", "[", "n", "]"}], "2"]}]}], ",", 
   RowBox[{"n", "\[Element]", "Integers"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->5618],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->135072367]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"2", 
    RowBox[{"LucasL", "[", 
     RowBox[{"2", "n"}], "]"}]}], "\[Equal]", 
   RowBox[{
    RowBox[{"5", 
     SuperscriptBox[
      RowBox[{"Fibonacci", "[", "n", "]"}], "2"]}], "+", 
    SuperscriptBox[
     RowBox[{"LucasL", "[", "n", "]"}], "2"]}]}], "/.", 
  RowBox[{"n", "->", "2.5"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->5826],

Cell[BoxData["False"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->140416266]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Neat Examples",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->589267740],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"Mod", "[", 
     RowBox[{
      RowBox[{"LucasL", "[", 
       RowBox[{"n", "+", "1"}], "]"}], ",", "n"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"n", ",", "200"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->251863325],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], LineBox[CompressedData["
1:eJxdlkluFEEQRUusWLJgwYJFYSGEEELMMzhsMGBjG094BFzdbmOz4gp5lDpK
LjlGHYEjAOL9kuK31Pp6GZGRUZGRWXVt+nPr9ELTNL/+/v/p/9/v+Sb/IuPF
yH6XIo9fNr5i41eN28hx5mzedRu/YfFvWrxb5nfb4t1B9Vx3Ld4987tveT8w
/4cW55HNe2zxnpjfU1T1fGbzn6N6zheWz0vL55XxaxufN3vk+EWs9RfyeFmw
9RdR6lYWze8Nyv4WserwFmW/ilj5LeW8y5L5vUP5FbHmv480XoybDyj1L2LV
aRlln8qy2VcYV51WzO+jrQuP9VtF6aOyavmtwar/mj3fOuPUo6yb/ROq+ovV
/xsofR1irSsm37phfps5TmzmeUV2+q+Kld8WSv1jK/sV2RcZF6t+2yj9E2Li
F1j1qWZvdnKc2LHngLUfdSf7NZ+xs+8h1rqw+rjCY/128cMeu5b/bs67yq77
Yy/HiT3Lfy+PV1TntNlHVf9985Od+6DKrjoe4Ee/x4HlL+a81wPzO0TJJ8Ss
Vw7zeDVujvBT/Y9sfTHns8JjH37JzxVfsl8Rs18VHs/VV5T+Cng8x2L8q1j3
7DdU9Rer78Sc3ypW/x+j6n+x9h9W39Zj8+tQnrft8rzoct6d/FXfLsfr5U+e
VfGo3yDmPmwm+LGf7ST7hezE7ya2vhjtJzZPyr0xyF/1m6LUvxXTfyFmfiem
D8o059/LTn51mscHszcnKOephVW/OMl5d7LDBVbf9ifmp/j0zyDW+2lGHLid
5fVCdt4r3Sz7FVTvwV6s+2KWxwfj5pR1OH8trO+pgHWPdfD4/LLzXdGfmp/i
8f4ZNK7743uO04r1PoN1f3bw2H+w6tGbvYrV/2LdP2fEZV/aM1sX1r3fweP3
xFmO08uu+0r2OdYXq//OUfX/efYL2dmfDtZ9WGDVpT83P1Tvu0HxtP4PdDv+
AB9xRz0=
     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  PlotRange->{{0., 200.}, {0., 195.}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 113},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->115121952]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->261980604],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sum", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"LucasL", "[", "k", "]"}], "/", 
    RowBox[{"3", "^", "k"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"k", ",", "1", ",", "Infinity"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->121146353],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->118884903]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["Fibonacci",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Fibonacci"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["GoldenRatio",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/GoldenRatio"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["RSolve",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RSolve"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->250839057],

Cell[TextData[ButtonBox["Combinatorial Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/CombinatorialFunctions"]], "Tutorials",
 CellID->118]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->449221642],

Cell[TextData[ButtonBox["Integer Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/IntegerFunctions"]], "MoreAbout",
 CellID->244521211],

Cell[TextData[ButtonBox["Recurrence and Sum Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/RecurrenceAndSumFunctions"]], "MoreAbout",
 CellID->11115255],

Cell[TextData[ButtonBox["New in 6.0: Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicalFunctions"]], "MoreAbout",
 CellID->47310307],

Cell[TextData[ButtonBox["New in 6.0: Number Theory & Integer Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/NewIn60NumberTheoryAndIntegerFunctions"]], "MoreAbout",
 CellID->152811463]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"LucasL - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 59, 17.0090790}", "context" -> "System`", 
    "keywords" -> {
     "Fibonacci recurrence", "Lucas number", "Lucas numbers", 
      "Lucas sequence"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "LucasL[n] gives the Lucas number L_n.", "synonyms" -> {"Lucas L"}, 
    "title" -> "LucasL", "type" -> "Symbol", "uri" -> "ref/LucasL"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[7583, 242, 359, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->47268906]},
 "Fibonacci"->{
  Cell[28656, 1071, 283, 10, 70, "Input",
   CellTags->"Fibonacci",
   CellID->28134],
  Cell[28942, 1083, 371, 14, 55, "Output",
   CellTags->"Fibonacci",
   CellID->401717101]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 39250, 1461},
 {"Fibonacci", 39385, 1465}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2551, 60, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3174, 87, 1571, 37, 70, "ObjectNameGrid"],
Cell[4748, 126, 575, 19, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[5360, 150, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[5838, 167, 120, 4, 70, "Notes",
 CellID->20640],
Cell[5961, 173, 664, 27, 70, "Notes",
 CellID->475],
Cell[6628, 202, 718, 26, 70, "Notes",
 CellID->1255],
Cell[7349, 230, 197, 7, 70, "Notes",
 CellID->28162]
}, Closed]],
Cell[CellGroupData[{
Cell[7583, 242, 359, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->47268906],
Cell[CellGroupData[{
Cell[7967, 256, 148, 5, 70, "ExampleSection",
 CellID->302918085],
Cell[8118, 263, 65, 1, 70, "ExampleText",
 CellID->625931072],
Cell[CellGroupData[{
Cell[8208, 268, 208, 7, 28, "Input",
 CellID->227574515],
Cell[8419, 277, 431, 11, 54, "Output",
 CellID->6710460]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[8899, 294, 221, 7, 70, "ExampleSection",
 CellID->1293636265],
Cell[9123, 303, 68, 1, 70, "ExampleText",
 CellID->21966],
Cell[CellGroupData[{
Cell[9216, 308, 101, 3, 70, "Input",
 CellID->30804],
Cell[9320, 313, 373, 8, 71, "Output",
 CellID->744611745]
}, Open  ]],
Cell[9708, 324, 120, 3, 70, "ExampleDelimiter",
 CellID->5025],
Cell[9831, 329, 72, 1, 70, "ExampleText",
 CellID->423],
Cell[CellGroupData[{
Cell[9928, 334, 229, 8, 70, "Input",
 CellID->15809],
Cell[10160, 344, 385, 12, 36, "Output",
 CellID->70483682]
}, Open  ]],
Cell[10560, 359, 120, 3, 70, "ExampleDelimiter",
 CellID->7030],
Cell[10683, 364, 61, 1, 70, "ExampleText",
 CellID->23375],
Cell[CellGroupData[{
Cell[10769, 369, 100, 3, 70, "Input",
 CellID->31384],
Cell[10872, 374, 175, 5, 36, "Output",
 CellID->15118738]
}, Open  ]],
Cell[CellGroupData[{
Cell[11084, 384, 175, 6, 70, "Input",
 CellID->1326],
Cell[11262, 392, 234, 7, 36, "Output",
 CellID->74873172]
}, Open  ]],
Cell[11511, 402, 121, 3, 70, "ExampleDelimiter",
 CellID->11923],
Cell[11635, 407, 57, 1, 70, "ExampleText",
 CellID->21468],
Cell[CellGroupData[{
Cell[11717, 412, 123, 4, 70, "Input",
 CellID->26278],
Cell[11843, 418, 271, 7, 36, "Output",
 CellID->861489839]
}, Open  ]],
Cell[12129, 428, 120, 3, 70, "ExampleDelimiter",
 CellID->2398],
Cell[12252, 433, 202, 7, 70, "ExampleText",
 CellID->22387],
Cell[CellGroupData[{
Cell[12479, 444, 178, 6, 70, "Input",
 CellID->24086],
Cell[12660, 452, 234, 7, 36, "Output",
 CellID->21649052]
}, Open  ]],
Cell[12909, 462, 121, 3, 70, "ExampleDelimiter",
 CellID->11161],
Cell[13033, 467, 72, 1, 70, "ExampleText",
 CellID->7541],
Cell[CellGroupData[{
Cell[13130, 472, 255, 9, 70, "Input",
 CellID->5915],
Cell[13388, 483, 1233, 39, 58, "Output",
 CellID->640733637]
}, Open  ]],
Cell[14636, 525, 125, 3, 70, "ExampleDelimiter",
 CellID->804171535],
Cell[14764, 530, 203, 7, 70, "ExampleText",
 CellID->126097607],
Cell[CellGroupData[{
Cell[14992, 541, 140, 4, 70, "Input",
 CellID->619062489],
Cell[15135, 547, 248, 9, 50, "Output",
 CellID->72509466]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[15432, 562, 227, 7, 70, "ExampleSection",
 CellID->258228157],
Cell[15662, 571, 79, 1, 70, "ExampleText",
 CellID->11321],
Cell[CellGroupData[{
Cell[15766, 576, 559, 18, 70, "Input",
 CellID->10265],
Cell[16328, 596, 673, 22, 51, "Output",
 CellID->205190814]
}, Open  ]],
Cell[CellGroupData[{
Cell[17038, 623, 305, 10, 70, "Input",
 CellID->32369],
Cell[17346, 635, 806, 28, 36, "Output",
 CellID->344988741]
}, Open  ]],
Cell[18167, 666, 125, 3, 70, "ExampleDelimiter",
 CellID->113800316],
Cell[18295, 671, 79, 1, 70, "ExampleText",
 CellID->29408],
Cell[CellGroupData[{
Cell[18399, 676, 289, 10, 70, "Input",
 CellID->1563],
Cell[18691, 688, 681, 21, 51, "Output",
 CellID->358058281]
}, Open  ]],
Cell[19387, 712, 72, 1, 70, "ExampleText",
 CellID->13103],
Cell[CellGroupData[{
Cell[19484, 717, 318, 10, 70, "Input",
 CellID->25323],
Cell[19805, 729, 679, 21, 51, "Output",
 CellID->7080173]
}, Open  ]],
Cell[20499, 753, 71, 1, 70, "ExampleText",
 CellID->31752],
Cell[CellGroupData[{
Cell[20595, 758, 92, 3, 70, "Input",
 CellID->6867],
Cell[20690, 763, 599, 14, 54, "Output",
 CellID->155967553]
}, Open  ]],
Cell[21304, 780, 120, 3, 70, "ExampleDelimiter",
 CellID->5556],
Cell[21427, 785, 296, 11, 70, "ExampleText",
 CellID->11617],
Cell[21726, 798, 721, 23, 70, "Input",
 CellID->29865],
Cell[22450, 823, 86, 1, 70, "ExampleText",
 CellID->14612],
Cell[CellGroupData[{
Cell[22561, 828, 255, 8, 70, "Input",
 CellID->430777473],
Cell[22819, 838, 891, 23, 144, "Output",
 CellID->93145189]
}, Open  ]],
Cell[23725, 864, 121, 3, 70, "ExampleDelimiter",
 CellID->11112],
Cell[23849, 869, 81, 1, 70, "ExampleText",
 CellID->17425],
Cell[CellGroupData[{
Cell[23955, 874, 312, 11, 70, "Input",
 CellID->28156],
Cell[24270, 887, 157, 5, 36, "Output",
 CellID->1024475]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[24476, 898, 233, 7, 70, "ExampleSection",
 CellID->32664],
Cell[24712, 907, 79, 1, 70, "ExampleText",
 CellID->14367],
Cell[CellGroupData[{
Cell[24816, 912, 138, 4, 70, "Input",
 CellID->5783],
Cell[24957, 918, 549, 21, 58, "Output",
 CellID->210422843]
}, Open  ]],
Cell[CellGroupData[{
Cell[25543, 944, 212, 7, 70, "Input",
 CellID->22744],
Cell[25758, 953, 297, 9, 36, "Output",
 CellID->23793776]
}, Open  ]],
Cell[26070, 965, 120, 3, 70, "ExampleDelimiter",
 CellID->5521],
Cell[26193, 970, 69, 1, 70, "ExampleText",
 CellID->31328],
Cell[26265, 973, 306, 11, 70, "Input",
 CellID->16041],
Cell[26574, 986, 232, 8, 70, "Input",
 CellID->992],
Cell[CellGroupData[{
Cell[26831, 998, 199, 7, 70, "Input",
 CellID->32629],
Cell[27033, 1007, 429, 11, 36, "Output",
 CellID->227930339]
}, Open  ]],
Cell[27477, 1021, 121, 3, 70, "ExampleDelimiter",
 CellID->12267],
Cell[27601, 1026, 89, 1, 70, "ExampleText",
 CellID->28825],
Cell[CellGroupData[{
Cell[27715, 1031, 450, 15, 70, "Input",
 CellID->19977],
Cell[28168, 1048, 265, 10, 36, "Output",
 CellID->91173419]
}, Open  ]],
Cell[28448, 1061, 121, 3, 70, "ExampleDelimiter",
 CellID->32717],
Cell[28572, 1066, 59, 1, 70, "ExampleText",
 CellID->32653],
Cell[CellGroupData[{
Cell[28656, 1071, 283, 10, 70, "Input",
 CellTags->"Fibonacci",
 CellID->28134],
Cell[28942, 1083, 371, 14, 55, "Output",
 CellTags->"Fibonacci",
 CellID->401717101]
}, Open  ]],
Cell[29328, 1100, 77, 1, 70, "ExampleText",
 CellID->10858],
Cell[CellGroupData[{
Cell[29430, 1105, 181, 6, 70, "Input",
 CellID->32494],
Cell[29614, 1113, 1005, 33, 36, "Output",
 CellID->7829229]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[30668, 1152, 231, 7, 70, "ExampleSection",
 CellID->1305812373],
Cell[30902, 1161, 116, 3, 70, "ExampleText",
 CellID->10552],
Cell[CellGroupData[{
Cell[31043, 1168, 124, 4, 70, "Input",
 CellID->28284],
Cell[31170, 1174, 187, 6, 36, "Output",
 CellID->452152549]
}, Open  ]],
Cell[31372, 1183, 121, 3, 70, "ExampleDelimiter",
 CellID->13255],
Cell[31496, 1188, 107, 3, 70, "ExampleText",
 CellID->5540],
Cell[CellGroupData[{
Cell[31628, 1195, 454, 15, 70, "Input",
 CellID->5618],
Cell[32085, 1212, 161, 5, 36, "Output",
 CellID->135072367]
}, Open  ]],
Cell[CellGroupData[{
Cell[32283, 1222, 392, 14, 70, "Input",
 CellID->5826],
Cell[32678, 1238, 162, 5, 36, "Output",
 CellID->140416266]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[32889, 1249, 228, 7, 70, "ExampleSection",
 CellID->589267740],
Cell[CellGroupData[{
Cell[33142, 1260, 340, 11, 70, "Input",
 CellID->251863325],
Cell[33485, 1273, 1430, 32, 134, "Output",
 CellID->115121952]
}, Open  ]],
Cell[34930, 1308, 125, 3, 70, "ExampleDelimiter",
 CellID->261980604],
Cell[CellGroupData[{
Cell[35080, 1315, 272, 9, 70, "Input",
 CellID->121146353],
Cell[35355, 1326, 158, 5, 36, "Output",
 CellID->118884903]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[35574, 1338, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[35893, 1349, 704, 26, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[36634, 1380, 318, 9, 70, "TutorialsSection",
 CellID->250839057],
Cell[36955, 1391, 154, 3, 70, "Tutorials",
 CellID->118]
}, Open  ]],
Cell[CellGroupData[{
Cell[37146, 1399, 319, 9, 70, "MoreAboutSection",
 CellID->449221642],
Cell[37468, 1410, 145, 3, 70, "MoreAbout",
 CellID->244521211],
Cell[37616, 1415, 164, 3, 70, "MoreAbout",
 CellID->11115255],
Cell[37783, 1420, 173, 3, 70, "MoreAbout",
 CellID->47310307],
Cell[37959, 1425, 198, 4, 70, "MoreAbout",
 CellID->152811463]
}, Open  ]],
Cell[38172, 1432, 27, 0, 70, "History"],
Cell[38202, 1434, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

