(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     77520,       2544]
NotebookOptionsPosition[     66790,       2167]
NotebookOutlinePosition[     68163,       2206]
CellTagsIndexPosition[     68077,       2201]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Basic Matrix Operations" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/BasicMatrixOperations"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Basic Matrix Operations\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"MatrixPower" :> 
          Documentation`HelpLookup["paclet:ref/MatrixPower"], "Dot" :> 
          Documentation`HelpLookup["paclet:ref/Dot"], "JordanDecomposition" :> 
          Documentation`HelpLookup["paclet:ref/JordanDecomposition"], 
          "QRDecomposition" :> 
          Documentation`HelpLookup["paclet:ref/QRDecomposition"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"MatrixPower\"\>", 2->"\<\"Dot\"\>", 
       3->"\<\"JordanDecomposition\"\>", 
       4->"\<\"QRDecomposition\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Integer Functions" :> 
          Documentation`HelpLookup["paclet:guide/IntegerFunctions"], 
          "Matrices and Linear Algebra" :> 
          Documentation`HelpLookup["paclet:guide/MatricesAndLinearAlgebra"], 
          "Matrix Operations" :> 
          Documentation`HelpLookup["paclet:guide/MatrixOperations"], 
          "Recurrence and Sum Functions" :> 
          Documentation`HelpLookup["paclet:guide/RecurrenceAndSumFunctions"], 
          "New in 6.0: Matrix & Linear Algebra Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MatrixAndLinearAlgebraFunctions"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Integer Functions\"\>", 
       2->"\<\"Matrices and Linear Algebra\"\>", 
       3->"\<\"Matrix Operations\"\>", 
       4->"\<\"Recurrence and Sum Functions\"\>", 
       5->"\<\"New in 6.0: Matrix & Linear Algebra Functions\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["MatrixExp", "ObjectName"], Cell[BoxData[
     TooltipBox[
      TogglerBox[Dynamic[
        FrontEnd`CurrentValue[
         FrontEnd`EvaluationNotebook[], {
         TaggingRules, "ModificationHighlight"}]], {True->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVVly2zAMDQhxEylSsizLaxJn73KafvQAuUDv/1eApJuosceeTDAWF4B8
eFgk71///HoVV1dXK3p+0wNfIzdCLL4I6p1sYBYuPy3SI2AmRBpuaE/TPRsE
3Isg8mrB9l6ILQhhLkbd93MBuwVxAthdZwPr5wVVzBc8wAvMX8TFqPkqE8q6
rDCsKPjZ/g1guz+PKvLFfGsDU1R4A8zDhgLangX9x/WGM7A1qSQ7rraYkaLf
LHZ5FWY9H7rew/n4/6vWPKT2mXFBaH/NGejzqi/VnFEFJ2Rb1Lo+jtofllPD
RVJjdZLr51Eb/ED1K+QI109LcIg+sdSIWqa5g85L6MiNoRloTws+pdBKiFGl
mx3N0jcAlcsKeFivVkteVOgMSNsC18sYS/iIMdZoK43WSdQVK1SLlYy09bZ2
aEO+aoPiACNak0EBbtfLRIz9KJRcroaSKwE11BprpRHqiJ0BrxtJG4pLYqWc
RZO4BQwSbGT4rFivaLjjQdscjSvlih48Mu+WDisOBcB5ehgeDEZQlikkKh3b
KA1FMTDg9yFxLSe4XC37begHTDh5seS1I/gaW025Yo/yUFbmSalShzon1Kch
JTNn2hKS1hyWVIkf5n2LHEQgt0zeUNTkMaYrDdkMZdo3fJCDHx7hxzAC+8rs
XYBokTPfEJk6JVlZx3Yl0UnTYMPvSUt6zghf8dhIi5VDnwO+Hcb1MHK1msK+
oy5qkYS7wVC3cfiq8ojBcMtFMJXnDiIcLPFJClDFGkJBhXGEZV6VcCQFoey7
tyvBF6lw0uKH6nyQ5fqhrKx/Kxe8+3coucpGN4U5+RYux7vnhGO5P7hKenqz
xSO3ivjisf345fg5Pt1ScCRkkzR1U1Dur+NCec4LZ49Ynx9P04GmOvlnqcKZ
79BfJfcjkg==
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlLlKQ0EYRi/iA9gpWIgvIXbjM1iKWAjW2vgeZnWN+AgWZhEULCxELFyi
ZnfJavbeJOr/33xXcj8CNnY5EA7k3GWYmTuzq+uLa2OWZS38/MYt0DbDHH8Q
8uTB64bRNrmsQkaPRoQSGX1ZqZPR9/eEazL6klIlowf8wiUZ3ecV3snowYDw
SEafUmJk9HmlQh4cX4GM7vUId2T0VFIho28or2T0w5DQIKN3PoUaGX1FaZLR
0ymh6sz7hHLk9PMzoU5Gz6SFmvFsCbd9Xzj9qyc0nfe49xfuTyYUEzoQbmh8
+N8e37HTbVpk9MSzUCajbypFMvq0UiOjzyklMnqvKzRMJCwU+/vjzemTSpns
nh97feJmd0e4cvppTKiQ0bsdoW5ecorZDgpPtD6xqFA2fp+Qp/vtdUj1v797
Oqdwnb2Ov++fUVpk9PCJUCC7n4t5wz5xj9/+rj6ceXbPP86NbEb58/k56qP+
n/0bdGyYqQ==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}], False->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVdd22zAMDUBxSaJm5HgljrPTJN177/nWh35AfqD//1aAVGrrJLbbk+BI
FEmQFxcXlDQ9//3zHNfW1gZ0/6IbbsYOhahvCGrO9qHM/mnhmMMLoKv0RD6V
cPyZHTTcDI5NkYnQq4UQ0BdiAkKYZagl1OOw5fWtcgyvPp7AnefsGG0DBIeA
9RZVrNfcwBasb4nlbP9u4cbc/XD/zXcIgC2qYYazJTQiGSYHS0Hr8Yu5LV/h
ePvR5oMOKswAQ0M8R5OloFMWSNRE+IAVeALv38KX26wAV1uU5Ojv16PQy8q+
gMP+9ABW5C9KbkZi6qv19DGc/ID9b+CrZTjbbVagH3p91oIcL+GZ6JDNUet4
HpWUXxx5BaWZxRh1FQgH6JqoKcarF/2/dblez7IE0XmWGlFL/6ygchIqCmPo
CTSmDq9SaCUUhfI7K3pKlwJESZiAveHGxoA7ESYGpM2B62WMJXzEoojRRhpt
IlFHPKFyjGRBQ2fjBG0WttpMcYIFWhNAAXaHA0+M4yiUXK6UxJWAGmKNsdII
cYGVAadTSQPKS2KkEovGc8swk2ALhg8Tww1qdrjRNmSTtOUqHDhk3jktVpwK
QOLoZngwWICyTMFTqdhHMrQTDQPeazzXdgWXK+e4KV3AhH0US1Ergo8x16QV
R5QXZWWeJJW6qLNHPWq8mEFpS0hac1pSeX4YxjlyEhmFZfKGsqaIhd+Sks+Q
0i7lhZx8cwYPmx5wrMA+yaCwyMqnRCb2IiubsF9JTKRJMeX3JKd5VoS3OEyl
xShBFxLebXrDpsfVSlv2FZ2iHMn4NBg6bZy+ihxiZvjIFWAixyeIcLDNT1KC
qogha1Gh14NB6LXpSEpC2bm3y8O3FmHniF9U55INhnttz7pZuWDu79BqFZxJ
F2bhWzjo7Zx6HMvng6ukuztzvGJXa66NmF/+crzrHe1ScmTkk/SouqB8vq42
0jl0EnuF9/RsMR1Io4U/S5Wt+A79AaDyK/w=
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlUtrE1EYhgfxBwguFBTEvyAiFhfzHxRFvC0E17rQhYLiT7D3u/gTXNg2
3sBFFyIutE1qm17UZJKZZJK0KFjBtur7Td4jnhe7dTMPhAfyQObMnO9Mjl69
fvraniAI7v/+7A3IRri7/8VGWJwHVTH72qohZp+ZBnUx+yWjJWYfHwNvxOwX
jKaYfXAAzIrZ+/tARcw+NAhKYvaTRiJmP2gUxH+vLxKzl5cMMXtfL3gvZn84
Cdpi9hvGJzH7yxegJWa/YnTE7Mtl0HTPfZ/xyPWVZZCGvQ/Au65fSV9aNMLJ
CfDW9a0fIBWz/9wBHbcOf/74PX83W99j128ZNTH71BMQidmfFkAiZj9irIvZ
Fz+AWMx+yEjF7KeMVOw/32x/iuHoCHjt+s42aIfTU6DWna/Prm9vgVb4cc0I
h4fAgusZ62L2A0YsZj9h1MX+9bN9LHfP75zrZ42a2L9/3+wXjVTM7q8jm9NE
5q8wA+JwoB/8eX+eNxIx+00jErP3GA2xvy7uG88R+z2jI2Y/bLTE/v1l76WG
mxP/+r7ZN7+Bupj9mFERsz9/Bppi9myeEzH7QgnUxOy3jYaY/a7RFrNfNtpi
9nNGXcw+NgoaYnbOk2///PJ/bXXFcP37JojF7CPDIBazf/0CKmL2+TlQEbPf
MZpi9olxkIrZS0UQidn3G1Ux+3EjErOfMapi9mBX8p73vOc973n/P/0XnFJ2
Rg==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}]}, "\"\<Click me\>\""],
      "Toggle changes in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["MatrixExp",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/MatrixExp"], "[", 
       StyleBox["m", "TI"], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the matrix exponential of ",
     Cell[BoxData[
      StyleBox["m", "TI"]], "InlineFormula"],
     ". "
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["MatrixExp",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/MatrixExp"], "[", 
       RowBox[{
        StyleBox["m", "TI"], ",", 
        StyleBox["v", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the matrix exponential of ",
     Cell[BoxData[
      StyleBox["m", "TI"]], "InlineFormula"],
     " applied to the vector ",
     Cell[BoxData[
      StyleBox["v", "TI"]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->29857]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["MatrixExp",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/MatrixExp"], "[", 
   StyleBox["m", "TI"], "]"}]], "InlineFormula"],
 " effectively evaluates the power series for the exponential function, with \
ordinary powers replaced by matrix powers. "
}], "Notes",
 CellID->23041],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["MatrixExp",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MatrixExp"]], "InlineFormula"],
 " works only on square matrices. "
}], "Notes",
 CellID->8891],

Cell[TextData[{
 "In ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["MatrixExp",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/MatrixExp"], "[", 
   RowBox[{
    StyleBox["m", "TI"], ",", 
    StyleBox["v", "TI"]}], "]"}]], "InlineFormula"],
 " the matrix ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 " can be a ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 " object."
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->4360]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->817203149],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->696617278],

Cell["Exponential of a 2\[Times]2 matrix:", "ExampleText",
 CellID->32653817],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixExp", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1.2", ",", "5.6"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"3", ",", "4"}], "}"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->745925929],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"346.55748729806976`", ",", "661.7345909344509`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"354.50067371488444`", ",", "677.4247827652953`"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{276, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->76528396]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixExp", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"0", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "}"}], " ", "t"}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->394999781],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Cos", "[", "t", "]"}], ",", 
     RowBox[{"Sin", "[", "t", "]"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", 
      RowBox[{"Sin", "[", "t", "]"}]}], ",", 
     RowBox[{"Cos", "[", "t", "]"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{256, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->571556928]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->88858306],

Cell["Exponential applied to a vector:", "ExampleText",
 CellID->47953683],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixExp", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1.2", ",", "5.6"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "4"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->746853787],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1670.0266691669717`", ",", "1709.350239245475`"}], "}"}]], "Output",\

 ImageSize->{128, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->605926166]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->27759],

Cell["Use exact arithmetic to compute the matrix exponential:", "ExampleText",
 CellID->8689360],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2", ",", "0", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "1", ",", 
       RowBox[{"-", "1"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "1", ",", "1"}], "}"}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->194557409],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixExp", "[", "m", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->36731986],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     SuperscriptBox["\[ExponentialE]", "2"], ",", "0", ",", "0"}], "}"}], ",",
    
   RowBox[{"{", 
    RowBox[{"0", ",", 
     RowBox[{"\[ExponentialE]", " ", 
      RowBox[{"Cos", "[", "1", "]"}]}], ",", 
     RowBox[{
      RowBox[{"-", "\[ExponentialE]"}], " ", 
      RowBox[{"Sin", "[", "1", "]"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", 
     RowBox[{"\[ExponentialE]", " ", 
      RowBox[{"Sin", "[", "1", "]"}]}], ",", 
     RowBox[{"\[ExponentialE]", " ", 
      RowBox[{"Cos", "[", "1", "]"}]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{406, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->153926014]
}, Open  ]],

Cell["Use machine arithmetic:", "ExampleText",
 CellID->24188808],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixExp", "[", 
  RowBox[{"N", "[", "m", "]"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->263630791],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"7.38905609893065`", ",", "0.`", ",", "0.`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.`", ",", "1.4686939399158851`", ",", 
     RowBox[{"-", "2.2873552871788423`"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.`", ",", "2.2873552871788423`", ",", "1.4686939399158851`"}], 
    "}"}]}], "}"}]], "Output",
 ImageSize->{459, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->662258082]
}, Open  ]],

Cell["Use 24-digit precision arithmetic:", "ExampleText",
 CellID->667858031],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixExp", "[", 
  RowBox[{"N", "[", 
   RowBox[{"m", ",", " ", "24"}], "]"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->288600981],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "7.38905609893065022723042746057266019005`24.", ",", "0", ",", "0"}], 
    "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1.4686939399158851571389675973266176736`24.", ",", 
     RowBox[{"-", "2.2873552871788423912081719067004895307`24."}]}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
    "0", ",", "2.2873552871788423912081719067004895307`24.", ",", 
     "1.4686939399158851571389675973266176736`24."}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{420, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->559204]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->146026522],

Cell["Find the matrix exponential of a complex matrix:", "ExampleText",
 CellID->85928973],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"RandomComplex", "[", 
    RowBox[{
     RowBox[{"1", "+", "I"}], ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "3"}], "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->320512040],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixExp", "[", "m", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->450199809],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"2.2063534403702727`", "\[InvisibleSpace]", "+", 
      RowBox[{"1.434321021178073`", " ", "\[ImaginaryI]"}]}], ",", 
     RowBox[{
      RowBox[{"-", "0.19070102801835606`"}], "+", 
      RowBox[{"1.7571540642450083`", " ", "\[ImaginaryI]"}]}], ",", 
     RowBox[{"0.40471183393502236`", "\[InvisibleSpace]", "+", 
      RowBox[{"0.9732596148891882`", " ", "\[ImaginaryI]"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"1.6285145849013194`", "\[InvisibleSpace]", "+", 
      RowBox[{"0.82391840221739`", " ", "\[ImaginaryI]"}]}], ",", 
     RowBox[{"1.1732705769554106`", "\[InvisibleSpace]", "+", 
      RowBox[{"1.2075408672493355`", " ", "\[ImaginaryI]"}]}], ",", 
     RowBox[{"1.1341745079185575`", "\[InvisibleSpace]", "+", 
      RowBox[{"0.823089079946296`", " ", "\[ImaginaryI]"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"0.04584228539797135`", "\[InvisibleSpace]", "+", 
      RowBox[{"0.7656752547275396`", " ", "\[ImaginaryI]"}]}], ",", 
     RowBox[{
      RowBox[{"-", "0.45159246975907025`"}], "+", 
      RowBox[{"0.9085914174073243`", " ", "\[ImaginaryI]"}]}], ",", 
     RowBox[{"1.286264697331338`", "\[InvisibleSpace]", "+", 
      RowBox[{"1.438789594330586`", " ", "\[ImaginaryI]"}]}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{462, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->306686202]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->554953936],

Cell["The exponential of a symbolic matrix:", "ExampleText",
 CellID->82753195],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2", ",", "0", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "1", ",", 
       RowBox[{"-", "1"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "1", ",", "1"}], "}"}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->411257602],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixExp", "[", 
  RowBox[{"m", " ", "t"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->296806254],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     SuperscriptBox["\[ExponentialE]", 
      RowBox[{"2", " ", "t"}]], ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", 
     RowBox[{
      SuperscriptBox["\[ExponentialE]", "t"], " ", 
      RowBox[{"Cos", "[", "t", "]"}]}], ",", 
     RowBox[{
      RowBox[{"-", 
       SuperscriptBox["\[ExponentialE]", "t"]}], " ", 
      RowBox[{"Sin", "[", "t", "]"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", 
     RowBox[{
      SuperscriptBox["\[ExponentialE]", "t"], " ", 
      RowBox[{"Sin", "[", "t", "]"}]}], ",", 
     RowBox[{
      SuperscriptBox["\[ExponentialE]", "t"], " ", 
      RowBox[{"Cos", "[", "t", "]"}]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{435, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->115025016]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->200642322],

Cell["\<\
Exponential of a sparse 100\[Times]100 matrix:\
\>", "ExampleText",
 CellID->140797825],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixExp", "[", 
  RowBox[{
   RowBox[{"SparseArray", "[", 
    RowBox[{
     RowBox[{"Band", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"1", ",", 
         RowBox[{"-", "1"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"100", ",", "100"}], "}"}]}], "]"}], "\[Rule]", "2"}], "]"}], 
   ",", 
   RowBox[{"Range", "[", "100", "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->594768040],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "201", ",", "2", ",", "3", ",", "4", ",", "5", ",", "6", ",", "7", ",", "8",
    ",", "9", ",", "10", ",", "11", ",", "12", ",", "13", ",", "14", ",", 
   "15", ",", "16", ",", "17", ",", "18", ",", "19", ",", "20", ",", "21", 
   ",", "22", ",", "23", ",", "24", ",", "25", ",", "26", ",", "27", ",", 
   "28", ",", "29", ",", "30", ",", "31", ",", "32", ",", "33", ",", "34", 
   ",", "35", ",", "36", ",", "37", ",", "38", ",", "39", ",", "40", ",", 
   "41", ",", "42", ",", "43", ",", "44", ",", "45", ",", "46", ",", "47", 
   ",", "48", ",", "49", ",", "50", ",", "51", ",", "52", ",", "53", ",", 
   "54", ",", "55", ",", "56", ",", "57", ",", "58", ",", "59", ",", "60", 
   ",", "61", ",", "62", ",", "63", ",", "64", ",", "65", ",", "66", ",", 
   "67", ",", "68", ",", "69", ",", "70", ",", "71", ",", "72", ",", "73", 
   ",", "74", ",", "75", ",", "76", ",", "77", ",", "78", ",", "79", ",", 
   "80", ",", "81", ",", "82", ",", "83", ",", "84", ",", "85", ",", "86", 
   ",", "87", ",", "88", ",", "89", ",", "90", ",", "91", ",", "92", ",", 
   "93", ",", "94", ",", "95", ",", "96", ",", "97", ",", "98", ",", "99", 
   ",", "100"}], "}"}]], "Output",
 ImageSize->{511, 84},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->846559194]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->24241],

Cell["A system of first-order linear differential equations:", "ExampleText",
 CellID->40126531],

Cell[BoxData[
 RowBox[{
  RowBox[{"system", " ", "=", " ", GridBox[{
     {
      RowBox[{
       RowBox[{
        RowBox[{"x", "'"}], "[", "t", "]"}], " ", "\[Equal]", " ", 
       RowBox[{"z", "[", "t", "]"}]}]},
     {
      RowBox[{
       RowBox[{
        RowBox[{"y", "'"}], "[", "t", "]"}], " ", "\[Equal]", " ", 
       RowBox[{
        RowBox[{"y", "[", "t", "]"}], " ", "-", " ", 
        RowBox[{"z", "[", "t", "]"}]}]}]},
     {
      RowBox[{
       RowBox[{
        RowBox[{"z", "'"}], "[", "t", "]"}], " ", "\[Equal]", " ", 
       RowBox[{"-", 
        RowBox[{"x", "[", "t", "]"}]}]}]}
    },
    GridBoxAlignment->{"Columns" -> {{Left}}}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->605903113],

Cell[TextData[{
 "Write the system in the form ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SuperscriptBox["v", "\[Prime]",
      MultilineFunction->None], "(", "t", ")"}], "\[LongEqual]", 
    RowBox[{"a", ".", 
     RowBox[{"v", "(", "t", ")"}]}]}], TraditionalForm]], "InlineMath"],
 " with ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"v", "(", "t", ")"}], "=", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"x", "(", "t", ")"}], ",", 
      RowBox[{"y", "(", "t", ")"}], ",", 
      RowBox[{"z", "(", "t", ")"}]}], "}"}]}], TraditionalForm]], 
  "InlineMath"],
 ":"
}], "ExampleText",
 CellID->380841397],

Cell[BoxData[
 RowBox[{
  RowBox[{"a", " ", "=", " ", 
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {"0", "0", "1"},
       {"0", "1", 
        RowBox[{"-", "1"}]},
       {
        RowBox[{"-", "1"}], "0", "0"}
      },
      GridBoxAlignment->{
       "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
        "RowsIndexed" -> {}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]]}], ";"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->295767112],

Cell["\<\
The matrix exponential gives the basis for the general solution:\
\>", "ExampleText",
 CellID->213645615],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"MatrixExp", "[", 
   RowBox[{"a", " ", "t"}], "]"}], "//", "MatrixForm"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->43135533],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      RowBox[{"Cos", "[", "t", "]"}], "0", 
      RowBox[{"Sin", "[", "t", "]"}]},
     {
      RowBox[{
       FractionBox["1", "2"], " ", 
       RowBox[{"(", 
        RowBox[{
         SuperscriptBox["\[ExponentialE]", "t"], "-", 
         RowBox[{"Cos", "[", "t", "]"}], "-", 
         RowBox[{"Sin", "[", "t", "]"}]}], ")"}]}], 
      SuperscriptBox["\[ExponentialE]", "t"], 
      RowBox[{
       FractionBox["1", "2"], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"-", 
          SuperscriptBox["\[ExponentialE]", "t"]}], "+", 
         RowBox[{"Cos", "[", "t", "]"}], "-", 
         RowBox[{"Sin", "[", "t", "]"}]}], ")"}]}]},
     {
      RowBox[{"-", 
       RowBox[{"Sin", "[", "t", "]"}]}], "0", 
      RowBox[{"Cos", "[", "t", "]"}]}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 ImageSize->{359, 51},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]//MatrixForm=",
 CellID->487100079]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{"system", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "[", "t", "]"}], ",", 
     RowBox[{"y", "[", "t", "]"}], ",", 
     RowBox[{"z", "[", "t", "]"}]}], "}"}], ",", "t"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->36189721],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{
     RowBox[{"x", "[", "t", "]"}], "\[Rule]", 
     RowBox[{
      RowBox[{
       RowBox[{"C", "[", "1", "]"}], " ", 
       RowBox[{"Cos", "[", "t", "]"}]}], "+", 
      RowBox[{
       RowBox[{"C", "[", "3", "]"}], " ", 
       RowBox[{"Sin", "[", "t", "]"}]}]}]}], ",", 
    RowBox[{
     RowBox[{"y", "[", "t", "]"}], "\[Rule]", 
     RowBox[{
      RowBox[{
       SuperscriptBox["\[ExponentialE]", "t"], " ", 
       RowBox[{"C", "[", "2", "]"}]}], "+", 
      RowBox[{
       FractionBox["1", "2"], " ", 
       RowBox[{"C", "[", "1", "]"}], " ", 
       RowBox[{"(", 
        RowBox[{
         SuperscriptBox["\[ExponentialE]", "t"], "-", 
         RowBox[{"Cos", "[", "t", "]"}], "-", 
         RowBox[{"Sin", "[", "t", "]"}]}], ")"}]}], "+", 
      RowBox[{
       FractionBox["1", "2"], " ", 
       RowBox[{"C", "[", "3", "]"}], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"-", 
          SuperscriptBox["\[ExponentialE]", "t"]}], "+", 
         RowBox[{"Cos", "[", "t", "]"}], "-", 
         RowBox[{"Sin", "[", "t", "]"}]}], ")"}]}]}]}], ",", 
    RowBox[{
     RowBox[{"z", "[", "t", "]"}], "\[Rule]", 
     RowBox[{
      RowBox[{
       RowBox[{"C", "[", "3", "]"}], " ", 
       RowBox[{"Cos", "[", "t", "]"}]}], "-", 
      RowBox[{
       RowBox[{"C", "[", "1", "]"}], " ", 
       RowBox[{"Sin", "[", "t", "]"}]}]}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{508, 74},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->518096503]
}, Open  ]],

Cell["\<\
The matrix exponential applied to a vector gives a particular solution:\
\>", "ExampleText",
 CellID->70516776],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"MatrixExp", "[", 
   RowBox[{
    RowBox[{"a", " ", "t"}], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "0", ",", "0"}], "}"}]}], "]"}], "//", 
  "MatrixForm"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->633566986],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", 
   TagBox[GridBox[{
      {
       RowBox[{"Cos", "[", "t", "]"}]},
      {
       RowBox[{
        FractionBox["1", "2"], " ", 
        RowBox[{"(", 
         RowBox[{
          SuperscriptBox["\[ExponentialE]", "t"], "-", 
          RowBox[{"Cos", "[", "t", "]"}], "-", 
          RowBox[{"Sin", "[", "t", "]"}]}], ")"}]}]},
      {
       RowBox[{"-", 
        RowBox[{"Sin", "[", "t", "]"}]}]}
     },
     GridBoxAlignment->{
      "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
       "RowsIndexed" -> {}},
     GridBoxSpacings->{"Columns" -> {
         Offset[0.27999999999999997`], {
          Offset[0.5599999999999999]}, 
         Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
         Offset[0.2], {
          Offset[0.4]}, 
         Offset[0.2]}, "RowsIndexed" -> {}}],
    Column], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 ImageSize->{173, 51},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]//MatrixForm=",
 CellID->269278080]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"system", ",", " ", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"x", "[", "0", "]"}], " ", "\[Equal]", " ", "1"}], ",", " ", 
       RowBox[{
        RowBox[{"y", "[", "0", "]"}], " ", "\[Equal]", " ", "0"}], ",", " ", 
       RowBox[{
        RowBox[{"z", "[", "0", "]"}], " ", "\[Equal]", " ", "0"}]}], "}"}]}], 
    "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "[", "t", "]"}], ",", " ", 
     RowBox[{"y", "[", "t", "]"}], ",", " ", 
     RowBox[{"z", "[", "t", "]"}]}], "}"}], ",", "t"}], "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->344428191],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{
     RowBox[{"x", "[", "t", "]"}], "\[Rule]", 
     RowBox[{"Cos", "[", "t", "]"}]}], ",", 
    RowBox[{
     RowBox[{"y", "[", "t", "]"}], "\[Rule]", 
     RowBox[{
      FractionBox["1", "2"], " ", 
      RowBox[{"(", 
       RowBox[{
        SuperscriptBox["\[ExponentialE]", "t"], "-", 
        RowBox[{"Cos", "[", "t", "]"}], "-", 
        RowBox[{"Sin", "[", "t", "]"}]}], ")"}]}]}], ",", 
    RowBox[{
     RowBox[{"z", "[", "t", "]"}], "\[Rule]", 
     RowBox[{"-", 
      RowBox[{"Sin", "[", "t", "]"}]}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{428, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->485172378]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->93211559],

Cell[TextData[{
 Cell[BoxData[
  StyleBox["s", "TI"]], "InlineFormula"],
 " is a matrix for approximating the second derivative with periodically on \
",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{
     RowBox[{"-", "L"}], ",", "L"}], "]"}], TraditionalForm]], "InlineMath"],
 " on the grid ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->225486379],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"L", " ", "=", " ", "10"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"n", " ", "=", " ", "500"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"x", " ", "=", " ", 
   RowBox[{"L", "*", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"2", 
       RowBox[{
        RowBox[{"N", "[", 
         RowBox[{"Range", "[", "n", "]"}], "]"}], "/", "n"}]}], " ", "-", " ",
       "1"}], ")"}]}]}], ";"}], "\n", 
 RowBox[{"s", " ", "=", " ", 
  RowBox[{"N", "[", 
   RowBox[{"Block", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"h", " ", "=", " ", 
       RowBox[{"2.", " ", 
        RowBox[{"L", "/", " ", "n"}]}]}], "}"}], ",", 
     RowBox[{
      FractionBox["1", 
       SuperscriptBox["h", "2"]], 
      RowBox[{"SparseArray", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{
           RowBox[{"{", 
            RowBox[{"i_", ",", " ", "i_"}], "}"}], "\[Rule]", 
           RowBox[{"-", "2"}]}], ",", " ", 
          RowBox[{
           RowBox[{
            RowBox[{"{", 
             RowBox[{"i_", ",", " ", "j_"}], "}"}], "/;", 
            RowBox[{
             RowBox[{"Abs", "[", 
              RowBox[{"i", "-", "j"}], "]"}], "\[Equal]", "1"}]}], "\[Rule]", 
           "1"}], ",", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"1", ",", "n"}], "}"}], "\[Rule]", "1"}], ",", " ", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"n", ",", "1"}], "}"}], "\[Rule]", "1"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"n", ",", "n"}], "}"}]}], "]"}]}]}], "]"}], 
   "]"}]}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->113944751],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "1500", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 1500, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"500", ",", "500"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{216, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->40666312]
}, Open  ]],

Cell[TextData[{
 "A vector representing a soliton on the grid ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->780525454],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"u", " ", "=", " ", 
   RowBox[{
    RowBox[{"Sech", "[", 
     RowBox[{"x", " ", "+", "Pi"}], " ", "]"}], " ", 
    RowBox[{"Exp", "[", 
     RowBox[{"Pi", " ", "I", " ", "x"}], "]"}]}]}], ";"}], "\n", 
 RowBox[{"ListPlot", "[", 
  RowBox[{"Transpose", "[", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"Abs", "[", "u", "]"}]}], "}"}], "]"}], "]"}]}], "Input",
 CellLabel->"In[2]:=",
 CellID->168958847],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 124},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->359007177]
}, Open  ]],

Cell[TextData[{
 "Propagate the solution of ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"\[ImaginaryI]", " ", 
     FractionBox[
      RowBox[{"\[PartialD]", "u"}], 
      RowBox[{"\[PartialD]", "t"}],
      MultilineFunction->None]}], "\[LongEqual]", 
    RowBox[{
     RowBox[{"2", "  ", 
      SuperscriptBox[
       RowBox[{"\[LeftBracketingBar]", "u", "\[RightBracketingBar]"}], "2"], 
      "u"}], "+", 
     FractionBox[
      RowBox[{
       SuperscriptBox["\[PartialD]", "2"], "u"}], 
      RowBox[{"\[PartialD]", 
       SuperscriptBox["x", "2"]}],
      MultilineFunction->None]}]}], TraditionalForm]], "InlineMath"],
 "using a splitting ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox["\[ExponentialE]", 
     RowBox[{"A", "+", "B"}]], "\[LongEqual]", 
    RowBox[{
     RowBox[{
      SuperscriptBox["\[ExponentialE]", 
       FractionBox["A", "2"]], "  ", 
      SuperscriptBox["\[ExponentialE]", "B"], 
      SuperscriptBox["\[ExponentialE]", 
       FractionBox["A", "2"]]}], "+", 
     RowBox[{"O", "(", 
      SuperscriptBox["\[CapitalDelta]t", "2"], ")"}]}]}], TraditionalForm]], 
  "InlineMath"],
 ": "
}], "ExampleText",
 CellID->69576445],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Timing", "[", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"\[CapitalDelta]t", " ", "=", " ", "0.05"}], ",", " ", "m"}], 
     "}"}], ",", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"m", " ", "=", " ", 
      RowBox[{"\[CapitalDelta]t", " ", 
       RowBox[{"(", 
        RowBox[{"0.", " ", "+", " ", 
         RowBox[{"1.", " ", "I"}]}], ")"}], " ", "s"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"Do", "[", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{
        RowBox[{"u", " ", "=", " ", 
         RowBox[{
          RowBox[{"Exp", "[", 
           RowBox[{"\[CapitalDelta]t", " ", "I", " ", "u", " ", 
            RowBox[{"Conjugate", "[", "u", "]"}]}], "]"}], " ", "u"}]}], ";", 
        "\[IndentingNewLine]", 
        RowBox[{"u", " ", "=", " ", 
         RowBox[{"MatrixExp", "[", 
          RowBox[{"m", ",", "u"}], "]"}]}], ";", "\[IndentingNewLine]", 
        RowBox[{"u", " ", "=", " ", 
         RowBox[{
          RowBox[{"Exp", "[", 
           RowBox[{"\[CapitalDelta]t", " ", "I", " ", "u", " ", 
            RowBox[{"Conjugate", "[", "u", "]"}]}], "]"}], " ", "u"}]}]}], 
       ",", "\[IndentingNewLine]", 
       RowBox[{"{", 
        RowBox[{"1", "/", "\[CapitalDelta]t"}], "}"}]}], "]"}]}]}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->255233404],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"2.1250000000000018`", ",", "Null"}], "}"}]], "Output",
 ImageSize->{93, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->480520070]
}, Open  ]],

Cell["\<\
Plot the solution and 10 times the error from the solution of the cubic Schr\
\[ODoubleDot]dinger equation:\
\>", "ExampleText",
 CellID->205579354],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Transpose", "[", 
      RowBox[{"{", 
       RowBox[{"x", ",", 
        RowBox[{"Abs", "[", "u", "]"}]}], "}"}], "]"}], ",", " ", 
     RowBox[{"Transpose", "[", 
      RowBox[{"{", 
       RowBox[{"x", ",", " ", 
        RowBox[{"10", 
         RowBox[{"Abs", "[", 
          RowBox[{"u", " ", "-", " ", 
           RowBox[{
            RowBox[{"Sech", "[", 
             RowBox[{"x", "-", " ", "Pi"}], "]"}], 
            RowBox[{"Exp", "[", 
             RowBox[{"I", 
              RowBox[{"(", 
               RowBox[{
                RowBox[{"Pi", "  ", "x"}], " ", "+", " ", 
                RowBox[{"(", 
                 RowBox[{"1", " ", "-", " ", 
                  RowBox[{"Pi", "^", "2"}]}], ")"}]}], ")"}]}], "]"}]}]}], 
          "]"}]}]}], "}"}], "]"}]}], "}"}], ",", " ", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->187026982],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 122},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->590663588]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->11829],

Cell["The matrix exponential of a diagonal matrix is diagonal:", "ExampleText",
 CellID->556033161],

Cell[BoxData[
 RowBox[{
  RowBox[{"d", "=", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "3", ",", "4"}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->264242556],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixExp", "[", 
  RowBox[{"DiagonalMatrix", "[", 
   RowBox[{"d", " ", "t"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->5177623],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     SuperscriptBox["\[ExponentialE]", "t"], ",", "0", ",", "0", ",", "0"}], 
    "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", 
     SuperscriptBox["\[ExponentialE]", 
      RowBox[{"2", " ", "t"}]], ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", 
     SuperscriptBox["\[ExponentialE]", 
      RowBox[{"3", " ", "t"}]], ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0", ",", 
     SuperscriptBox["\[ExponentialE]", 
      RowBox[{"4", " ", "t"}]]}], "}"}]}], "}"}]], "Output",
 ImageSize->{397, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->638462165]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DiagonalMatrix", "[", 
  RowBox[{"Exp", "[", 
   RowBox[{"d", " ", "t"}], "]"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->88887168],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     SuperscriptBox["\[ExponentialE]", "t"], ",", "0", ",", "0", ",", "0"}], 
    "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", 
     SuperscriptBox["\[ExponentialE]", 
      RowBox[{"2", " ", "t"}]], ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", 
     SuperscriptBox["\[ExponentialE]", 
      RowBox[{"3", " ", "t"}]], ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0", ",", 
     SuperscriptBox["\[ExponentialE]", 
      RowBox[{"4", " ", "t"}]]}], "}"}]}], "}"}]], "Output",
 ImageSize->{397, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->601292095]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->362655316],

Cell["\<\
The matrix exponential of a nilpotent matrix is a polynomial matrix:\
\>", "ExampleText",
 CellID->663812163],

Cell[BoxData[
 RowBox[{
  RowBox[{"n", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"0", ",", "2", ",", "3", ",", "4"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "0", ",", "5", ",", "6"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "0", ",", "0", ",", "7"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "0", ",", "0", ",", "0"}], "}"}]}], "}"}]}], 
  ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->214394689],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"MatrixExp", "[", 
   RowBox[{"n", " ", "t"}], "]"}], "//", "MatrixForm"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->44531182],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {"1", 
      RowBox[{"2", " ", "t"}], 
      RowBox[{
       RowBox[{"3", " ", "t"}], "+", 
       RowBox[{"5", " ", 
        SuperscriptBox["t", "2"]}]}], 
      RowBox[{
       FractionBox["1", "6"], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"24", " ", "t"}], "+", 
         RowBox[{"99", " ", 
          SuperscriptBox["t", "2"]}], "+", 
         RowBox[{"70", " ", 
          SuperscriptBox["t", "3"]}]}], ")"}]}]},
     {"0", "1", 
      RowBox[{"5", " ", "t"}], 
      RowBox[{
       FractionBox["1", "2"], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"12", " ", "t"}], "+", 
         RowBox[{"35", " ", 
          SuperscriptBox["t", "2"]}]}], ")"}]}]},
     {"0", "0", "1", 
      RowBox[{"7", " ", "t"}]},
     {"0", "0", "0", "1"}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 ImageSize->{261, 75},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]//MatrixForm=",
 CellID->118030690]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->110225441],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["MatrixExp",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/MatrixExp"], "[", 
   StyleBox["m", "TI"], "]"}]], "InlineFormula"],
 " is always invertible, and the inverse is given by ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["MatrixExp",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/MatrixExp"], "[", 
   RowBox[{"-", 
    StyleBox["m", "TI"]}], "]"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->754033753],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"KroneckerProduct", "[", 
    RowBox[{
     RowBox[{"RandomReal", "[", 
      RowBox[{"1", ",", "3"}], "]"}], ",", 
     RowBox[{"RandomReal", "[", 
      RowBox[{"1", ",", "3"}], "]"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->193303670],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"MatrixRank", "[", "m", "]"}], ",", 
   RowBox[{"MatrixRank", "[", 
    RowBox[{"MatrixExp", "[", "m", "]"}], "]"}]}], "}"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->135061812],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1", ",", "3"}], "}"}]], "Output",
 ImageSize->{44, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->956558069]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"MatrixExp", "[", "m", "]"}], ".", 
   RowBox[{"MatrixExp", "[", 
    RowBox[{"-", "m"}], "]"}]}], "//", "Chop"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->269986019],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1.0000000000000002`", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0.9999999999999998`", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1.`"}], "}"}]}], "}"}]], "Output",
 ImageSize->{234, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->33329679]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->805649711],

Cell[TextData[{
 "If ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 " is diagonalizable with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"m", "=", 
    RowBox[{
     SuperscriptBox["v", 
      RowBox[{"-", "1"}]], ".", "d", ".", "v"}]}], TraditionalForm]], 
  "InlineMath"],
 " then ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"exp", "(", "m", ")"}], "=", 
    RowBox[{"v", ".", 
     RowBox[{"exp", "(", "d", ")"}], ".", "v"}]}], TraditionalForm]], 
  "InlineMath"],
 ":"
}], "ExampleText",
 CellID->175955871],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "3"}], "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->569559691],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"d", ",", " ", "vt"}], "}"}], " ", "=", " ", 
  RowBox[{"Eigensystem", "[", "m", "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->72273480],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1.584513709481664`", ",", 
     RowBox[{"-", "0.30352340050491877`"}], ",", "0.20037516942064118`"}], 
    "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "0.6870704165833948`"}], ",", 
       RowBox[{"-", "0.630798328955157`"}], ",", 
       RowBox[{"-", "0.36059355352432737`"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "0.1202268601713266`"}], ",", 
       RowBox[{"-", "0.42452642725886014`"}], ",", "0.897397801787018`"}], 
      "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "0.7373907083513089`"}], ",", "0.31085984679553946`", ",",
        "0.5996841659468829`"}], "}"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{468, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->304523435]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"v", " ", "=", " ", 
   RowBox[{"Transpose", "[", "vt", "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"v", ".", 
    RowBox[{"DiagonalMatrix", "[", 
     RowBox[{"Exp", "[", "d", "]"}], "]"}], ".", 
    RowBox[{"Inverse", "[", "v", "]"}]}], " ", "-", " ", 
   RowBox[{"MatrixExp", "[", "m", "]"}]}], "//", "Chop"}]}], "Input",
 CellLabel->"In[3]:=",
 CellID->165172772],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0"}], "}"}]}], "}"}]], "Output",
 ImageSize->{213, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->894231516]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->12448],

Cell["\<\
For a large sparse matrix, computing the matrix exponential may take a long \
time:\
\>", "ExampleText",
 CellID->530307707],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Block", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"n", " ", "=", " ", "1000"}], "}"}], ",", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"v", " ", "=", " ", 
     RowBox[{"RandomComplex", "[", 
      RowBox[{
       RowBox[{"1", " ", "+", " ", "I"}], ",", " ", "1000"}], "]"}]}], ";", 
    "\[IndentingNewLine]", 
    RowBox[{"s", " ", "=", " ", 
     RowBox[{"N", "[", 
      RowBox[{"SparseArray", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{
           RowBox[{"{", 
            RowBox[{"i_", ",", " ", "i_"}], "}"}], "\[Rule]", 
           RowBox[{
            RowBox[{"-", "2"}], " ", "I"}]}], ",", " ", 
          RowBox[{
           RowBox[{
            RowBox[{"{", 
             RowBox[{"i_", ",", " ", "j_"}], "}"}], "/;", 
            RowBox[{
             RowBox[{"Abs", "[", 
              RowBox[{"i", "-", "j"}], "]"}], "\[Equal]", "1"}]}], "\[Rule]", 
           "I"}], ",", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"1", ",", "n"}], "}"}], "\[Rule]", "I"}], ",", " ", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"n", ",", "1"}], "}"}], "\[Rule]", "I"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"n", ",", "n"}], "}"}], ",", " ", 
        RowBox[{"(", 
         RowBox[{"0.", " ", "+", " ", 
          RowBox[{"0.", " ", "I"}]}], ")"}]}], "]"}], "]"}]}]}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->268186946],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "3000", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 3000, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"1000", ",", "1000"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{230, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->12888226]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Timing", "[", 
  RowBox[{
   RowBox[{"m", " ", "=", " ", 
    RowBox[{"MatrixExp", "[", "s", "]"}]}], ";"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->14606463],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"6.625`", ",", "Null"}], "}"}]], "Output",
 ImageSize->{93, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->146193481]
}, Open  ]],

Cell["\<\
Computing the application of it to a vector uses less memory and is much \
faster:\
\>", "ExampleText",
 CellID->786992911],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Timing", "[", 
  RowBox[{
   RowBox[{"ev", " ", "=", " ", 
    RowBox[{"MatrixExp", "[", 
     RowBox[{"s", ",", " ", "v"}], "]"}]}], ";"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->649272123],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.04700000000000029`", ",", "Null"}], "}"}]], "Output",
 ImageSize->{93, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->481933799]
}, Open  ]],

Cell["The results are essentially the same:", "ExampleText",
 CellID->328434215],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Max", "[", 
  RowBox[{"Abs", "[", 
   RowBox[{"ev", " ", "-", " ", 
    RowBox[{"m", ".", "v"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->196582990],

Cell[BoxData["1.159106867033638`*^-15"], "Output",
 ImageSize->{94, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->373371724]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Neat Examples",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->11798],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"Reverse", "@", 
    RowBox[{"IdentityMatrix", "[", "20", "]"}]}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->257009288],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FullSimplify", "[", 
  RowBox[{"MatrixExp", "[", 
   RowBox[{"m", " ", "t"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->18151500],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{465, 339},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->338605806]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->17929],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["MatrixPower",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/MatrixPower"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Dot",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Dot"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["JordanDecomposition",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/JordanDecomposition"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["QRDecomposition",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/QRDecomposition"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->841]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->25264],

Cell[TextData[ButtonBox["Basic Matrix Operations",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/BasicMatrixOperations"]], "Tutorials",
 CellID->1623]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection",
 CellID->6915],

Cell[TextData[ButtonBox["Implementation notes: Numerical and Related \
Functions",
 BaseStyle->"Hyperlink",
 ButtonData->{
   URL["http://reference.wolfram.com/mathematica/note/\
SomeNotesOnInternalImplementation.html#14568"], None}]], "RelatedLinks",
 CellID->269338167],

Cell[TextData[ButtonBox["Implementation notes: Algebra and Calculus",
 BaseStyle->"Hyperlink",
 ButtonData->{
   URL["http://reference.wolfram.com/mathematica/note/\
SomeNotesOnInternalImplementation.html#12949"], None}]], "RelatedLinks"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->802043143],

Cell[TextData[ButtonBox["Integer Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/IntegerFunctions"]], "MoreAbout",
 CellID->572525692],

Cell[TextData[ButtonBox["Matrices and Linear Algebra",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/MatricesAndLinearAlgebra"]], "MoreAbout",
 CellID->90385971],

Cell[TextData[ButtonBox["Matrix Operations",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/MatrixOperations"]], "MoreAbout",
 CellID->161672957],

Cell[TextData[ButtonBox["Recurrence and Sum Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/RecurrenceAndSumFunctions"]], "MoreAbout",
 CellID->172055981],

Cell[TextData[ButtonBox["New in 6.0: Matrix & Linear Algebra Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/NewIn60MatrixAndLinearAlgebraFunctions"]], "MoreAbout",
 CellID->22375997]
}, Open  ]],

Cell["New in 2  |  Last modified in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"MatrixExp - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 59, 57.3694872}", "context" -> "System`", 
    "keywords" -> {
     "exponentials of matrices", "Jordan decomposition", "matrix exponential",
       "Putzer's method"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "MatrixExp[m] gives the matrix exponential of m. MatrixExp[m, v] gives \
the matrix exponential of m applied to the vector v.", 
    "synonyms" -> {"matrix exp"}, "title" -> "MatrixExp", "type" -> "Symbol", 
    "uri" -> "ref/MatrixExp"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[10434, 283, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->817203149]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 67933, 2194}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2822, 65, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3445, 92, 4193, 83, 70, "ObjectNameGrid"],
Cell[7641, 177, 1070, 35, 70, "Usage",
 CellID->29857]
}, Open  ]],
Cell[CellGroupData[{
Cell[8748, 217, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[9226, 234, 340, 10, 70, "Notes",
 CellID->23041],
Cell[9569, 246, 200, 7, 70, "Notes",
 CellID->8891],
Cell[9772, 255, 625, 23, 70, "Notes",
 CellID->4360]
}, Closed]],
Cell[CellGroupData[{
Cell[10434, 283, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->817203149],
Cell[CellGroupData[{
Cell[10819, 297, 148, 5, 70, "ExampleSection",
 CellID->696617278],
Cell[10970, 304, 77, 1, 70, "ExampleText",
 CellID->32653817],
Cell[CellGroupData[{
Cell[11072, 309, 258, 9, 28, "Input",
 CellID->745925929],
Cell[11333, 320, 374, 12, 36, "Output",
 CellID->76528396]
}, Open  ]],
Cell[CellGroupData[{
Cell[11744, 337, 309, 12, 28, "Input",
 CellID->394999781],
Cell[12056, 351, 456, 16, 36, "Output",
 CellID->571556928]
}, Open  ]],
Cell[12527, 370, 124, 3, 70, "ExampleDelimiter",
 CellID->88858306],
Cell[12654, 375, 74, 1, 70, "ExampleText",
 CellID->47953683],
Cell[CellGroupData[{
Cell[12753, 380, 336, 12, 70, "Input",
 CellID->746853787],
Cell[13092, 394, 241, 8, 36, "Output",
 CellID->605926166]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[13382, 408, 216, 7, 70, "ExampleSection",
 CellID->27759],
Cell[13601, 417, 96, 1, 70, "ExampleText",
 CellID->8689360],
Cell[13700, 420, 381, 13, 70, "Input",
 CellID->194557409],
Cell[CellGroupData[{
Cell[14106, 437, 104, 3, 70, "Input",
 CellID->36731986],
Cell[14213, 442, 754, 24, 39, "Output",
 CellID->153926014]
}, Open  ]],
Cell[14982, 469, 65, 1, 70, "ExampleText",
 CellID->24188808],
Cell[CellGroupData[{
Cell[15072, 474, 133, 4, 70, "Input",
 CellID->263630791],
Cell[15208, 480, 513, 15, 36, "Output",
 CellID->662258082]
}, Open  ]],
Cell[15736, 498, 77, 1, 70, "ExampleText",
 CellID->667858031],
Cell[CellGroupData[{
Cell[15838, 503, 163, 5, 70, "Input",
 CellID->288600981],
Cell[16004, 510, 648, 20, 71, "Output",
 CellID->559204]
}, Open  ]],
Cell[16667, 533, 125, 3, 70, "ExampleDelimiter",
 CellID->146026522],
Cell[16795, 538, 90, 1, 70, "ExampleText",
 CellID->85928973],
Cell[16888, 541, 255, 9, 70, "Input",
 CellID->320512040],
Cell[CellGroupData[{
Cell[17168, 554, 105, 3, 70, "Input",
 CellID->450199809],
Cell[17276, 559, 1492, 34, 71, "Output",
 CellID->306686202]
}, Open  ]],
Cell[18783, 596, 125, 3, 70, "ExampleDelimiter",
 CellID->554953936],
Cell[18911, 601, 79, 1, 70, "ExampleText",
 CellID->82753195],
Cell[18993, 604, 381, 13, 70, "Input",
 CellID->411257602],
Cell[CellGroupData[{
Cell[19399, 621, 128, 4, 70, "Input",
 CellID->296806254],
Cell[19530, 627, 890, 28, 39, "Output",
 CellID->115025016]
}, Open  ]],
Cell[20435, 658, 125, 3, 70, "ExampleDelimiter",
 CellID->200642322],
Cell[20563, 663, 97, 3, 70, "ExampleText",
 CellID->140797825],
Cell[CellGroupData[{
Cell[20685, 670, 441, 15, 70, "Input",
 CellID->594768040],
Cell[21129, 687, 1341, 23, 105, "Output",
 CellID->846559194]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[22519, 716, 223, 7, 70, "ExampleSection",
 CellID->24241],
Cell[22745, 725, 96, 1, 70, "ExampleText",
 CellID->40126531],
Cell[22844, 728, 719, 24, 70, "Input",
 CellID->605903113],
Cell[23566, 754, 632, 23, 70, "ExampleText",
 CellID->380841397],
Cell[24201, 779, 791, 24, 70, "Input",
 CellID->295767112],
Cell[24995, 805, 115, 3, 70, "ExampleText",
 CellID->213645615],
Cell[CellGroupData[{
Cell[25135, 812, 161, 5, 70, "Input",
 CellID->43135533],
Cell[25299, 819, 1463, 44, 86, "Output",
 CellID->487100079]
}, Open  ]],
Cell[CellGroupData[{
Cell[26799, 868, 291, 9, 70, "Input",
 CellID->36189721],
Cell[27093, 879, 1585, 49, 95, "Output",
 CellID->518096503]
}, Open  ]],
Cell[28693, 931, 121, 3, 70, "ExampleText",
 CellID->70516776],
Cell[CellGroupData[{
Cell[28839, 938, 251, 9, 70, "Input",
 CellID->633566986],
Cell[29093, 949, 1121, 35, 86, "Output",
 CellID->269278080]
}, Open  ]],
Cell[CellGroupData[{
Cell[30251, 989, 676, 20, 70, "Input",
 CellID->344428191],
Cell[30930, 1011, 744, 24, 51, "Output",
 CellID->485172378]
}, Open  ]],
Cell[31689, 1038, 124, 3, 70, "ExampleDelimiter",
 CellID->93211559],
Cell[31816, 1043, 406, 15, 70, "ExampleText",
 CellID->225486379],
Cell[CellGroupData[{
Cell[32247, 1062, 1626, 52, 70, "Input",
 CellID->113944751],
Cell[33876, 1116, 488, 17, 36, "Output",
 CellID->40666312]
}, Open  ]],
Cell[34379, 1136, 165, 6, 70, "ExampleText",
 CellID->780525454],
Cell[CellGroupData[{
Cell[34569, 1146, 448, 14, 70, "Input",
 CellID->168958847],
Cell[35020, 1162, 2380, 43, 70, "Output",
 Evaluatable->False,
 CellID->359007177]
}, Open  ]],
Cell[37415, 1208, 1181, 39, 70, "ExampleText",
 CellID->69576445],
Cell[CellGroupData[{
Cell[38621, 1251, 1374, 37, 70, "Input",
 CellID->255233404],
Cell[39998, 1290, 224, 7, 36, "Output",
 CellID->480520070]
}, Open  ]],
Cell[40237, 1300, 158, 4, 70, "ExampleText",
 CellID->205579354],
Cell[CellGroupData[{
Cell[40420, 1308, 991, 29, 70, "Input",
 CellID->187026982],
Cell[41414, 1339, 2799, 50, 70, "Output",
 Evaluatable->False,
 CellID->590663588]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[44262, 1395, 233, 7, 70, "ExampleSection",
 CellID->11829],
Cell[44498, 1404, 99, 1, 70, "ExampleText",
 CellID->556033161],
Cell[44600, 1407, 179, 6, 70, "Input",
 CellID->264242556],
Cell[CellGroupData[{
Cell[44804, 1417, 168, 5, 70, "Input",
 CellID->5177623],
Cell[44975, 1424, 750, 23, 39, "Output",
 CellID->638462165]
}, Open  ]],
Cell[CellGroupData[{
Cell[45762, 1452, 163, 5, 70, "Input",
 CellID->88887168],
Cell[45928, 1459, 750, 23, 39, "Output",
 CellID->601292095]
}, Open  ]],
Cell[46693, 1485, 125, 3, 70, "ExampleDelimiter",
 CellID->362655316],
Cell[46821, 1490, 119, 3, 70, "ExampleText",
 CellID->663812163],
Cell[46943, 1495, 474, 15, 70, "Input",
 CellID->214394689],
Cell[CellGroupData[{
Cell[47442, 1514, 161, 5, 70, "Input",
 CellID->44531182],
Cell[47606, 1521, 1475, 47, 110, "Output",
 CellID->118030690]
}, Open  ]],
Cell[49096, 1571, 125, 3, 70, "ExampleDelimiter",
 CellID->110225441],
Cell[49224, 1576, 479, 17, 70, "ExampleText",
 CellID->754033753],
Cell[49706, 1595, 313, 10, 70, "Input",
 CellID->193303670],
Cell[CellGroupData[{
Cell[50044, 1609, 229, 7, 70, "Input",
 CellID->135061812],
Cell[50276, 1618, 203, 7, 36, "Output",
 CellID->956558069]
}, Open  ]],
Cell[CellGroupData[{
Cell[50516, 1630, 213, 7, 70, "Input",
 CellID->269986019],
Cell[50732, 1639, 428, 13, 36, "Output",
 CellID->33329679]
}, Open  ]],
Cell[51175, 1655, 125, 3, 70, "ExampleDelimiter",
 CellID->805649711],
Cell[51303, 1660, 530, 22, 70, "ExampleText",
 CellID->175955871],
Cell[51836, 1684, 226, 8, 70, "Input",
 CellID->569559691],
Cell[CellGroupData[{
Cell[52087, 1696, 192, 6, 70, "Input",
 CellID->72273480],
Cell[52282, 1704, 908, 27, 54, "Output",
 CellID->304523435]
}, Open  ]],
Cell[CellGroupData[{
Cell[53227, 1736, 413, 12, 70, "Input",
 CellID->165172772],
Cell[53643, 1750, 391, 13, 36, "Output",
 CellID->894231516]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[54083, 1769, 226, 7, 70, "ExampleSection",
 CellID->12448],
Cell[54312, 1778, 134, 4, 70, "ExampleText",
 CellID->530307707],
Cell[CellGroupData[{
Cell[54471, 1786, 1461, 43, 70, "Input",
 CellID->268186946],
Cell[55935, 1831, 490, 17, 36, "Output",
 CellID->12888226]
}, Open  ]],
Cell[CellGroupData[{
Cell[56462, 1853, 191, 6, 70, "Input",
 CellID->14606463],
Cell[56656, 1861, 211, 7, 36, "Output",
 CellID->146193481]
}, Open  ]],
Cell[56882, 1871, 133, 4, 70, "ExampleText",
 CellID->786992911],
Cell[CellGroupData[{
Cell[57040, 1879, 224, 7, 70, "Input",
 CellID->649272123],
Cell[57267, 1888, 225, 7, 36, "Output",
 CellID->481933799]
}, Open  ]],
Cell[57507, 1898, 80, 1, 70, "ExampleText",
 CellID->328434215],
Cell[CellGroupData[{
Cell[57612, 1903, 189, 6, 70, "Input",
 CellID->196582990],
Cell[57804, 1911, 180, 5, 36, "Output",
 CellID->373371724]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[58033, 1922, 224, 7, 70, "ExampleSection",
 CellID->11798],
Cell[58260, 1931, 184, 6, 70, "Input",
 CellID->257009288],
Cell[CellGroupData[{
Cell[58469, 1941, 167, 5, 70, "Input",
 CellID->18151500],
Cell[58639, 1948, 4145, 72, 70, "Output",
 Evaluatable->False,
 CellID->338605806]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[62845, 2027, 311, 9, 70, "SeeAlsoSection",
 CellID->17929],
Cell[63159, 2038, 971, 35, 70, "SeeAlso",
 CellID->841]
}, Open  ]],
Cell[CellGroupData[{
Cell[64167, 2078, 314, 9, 70, "TutorialsSection",
 CellID->25264],
Cell[64484, 2089, 154, 3, 70, "Tutorials",
 CellID->1623]
}, Open  ]],
Cell[CellGroupData[{
Cell[64675, 2097, 320, 9, 70, "RelatedLinksSection",
 CellID->6915],
Cell[64998, 2108, 271, 6, 70, "RelatedLinks",
 CellID->269338167],
Cell[65272, 2116, 238, 4, 70, "RelatedLinks"]
}, Open  ]],
Cell[CellGroupData[{
Cell[65547, 2125, 319, 9, 70, "MoreAboutSection",
 CellID->802043143],
Cell[65869, 2136, 145, 3, 70, "MoreAbout",
 CellID->572525692],
Cell[66017, 2141, 162, 3, 70, "MoreAbout",
 CellID->90385971],
Cell[66182, 2146, 145, 3, 70, "MoreAbout",
 CellID->161672957],
Cell[66330, 2151, 165, 3, 70, "MoreAbout",
 CellID->172055981],
Cell[66498, 2156, 197, 4, 70, "MoreAbout",
 CellID->22375997]
}, Open  ]],
Cell[66710, 2163, 50, 0, 70, "History"],
Cell[66763, 2165, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

