(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     52570,       1929]
NotebookOptionsPosition[     41694,       1542]
NotebookOutlinePosition[     42929,       1578]
CellTagsIndexPosition[     42843,       1573]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Solving Linear Systems" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/SolvingLinearSystems"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Solving Linear Systems\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"NullSpace" :> Documentation`HelpLookup["paclet:ref/NullSpace"],
           "Det" :> Documentation`HelpLookup["paclet:ref/Det"], "Eigensystem" :> 
          Documentation`HelpLookup["paclet:ref/Eigensystem"], "RowReduce" :> 
          Documentation`HelpLookup["paclet:ref/RowReduce"], 
          "SingularValueList" :> 
          Documentation`HelpLookup["paclet:ref/SingularValueList"], "Inverse" :> 
          Documentation`HelpLookup["paclet:ref/Inverse"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"NullSpace\"\>", 2->"\<\"Det\"\>", 
       3->"\<\"Eigensystem\"\>", 4->"\<\"RowReduce\"\>", 
       5->"\<\"SingularValueList\"\>", 
       6->"\<\"Inverse\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Linear Systems" :> 
          Documentation`HelpLookup["paclet:guide/LinearSystems"], 
          "Matrices and Linear Algebra" :> 
          Documentation`HelpLookup[
           "paclet:guide/MatricesAndLinearAlgebra"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Linear Systems\"\>", 
       2->"\<\"Matrices and Linear Algebra\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["MatrixRank", "ObjectName"], Cell[BoxData[
     TooltipBox[
      TogglerBox[Dynamic[
        FrontEnd`CurrentValue[
         FrontEnd`EvaluationNotebook[], {
         TaggingRules, "ModificationHighlight"}]], {True->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVVly2zAMDQhxEylSsizLaxJn73KafvQAuUDv/1eApJuosceeTDAWF4B8
eFgk71///HoVV1dXK3p+0wNfIzdCLL4I6p1sYBYuPy3SI2AmRBpuaE/TPRsE
3Isg8mrB9l6ILQhhLkbd93MBuwVxAthdZwPr5wVVzBc8wAvMX8TFqPkqE8q6
rDCsKPjZ/g1guz+PKvLFfGsDU1R4A8zDhgLangX9x/WGM7A1qSQ7rraYkaLf
LHZ5FWY9H7rew/n4/6vWPKT2mXFBaH/NGejzqi/VnFEFJ2Rb1Lo+jtofllPD
RVJjdZLr51Eb/ED1K+QI109LcIg+sdSIWqa5g85L6MiNoRloTws+pdBKiFGl
mx3N0jcAlcsKeFivVkteVOgMSNsC18sYS/iIMdZoK43WSdQVK1SLlYy09bZ2
aEO+aoPiACNak0EBbtfLRIz9KJRcroaSKwE11BprpRHqiJ0BrxtJG4pLYqWc
RZO4BQwSbGT4rFivaLjjQdscjSvlih48Mu+WDisOBcB5ehgeDEZQlikkKh3b
KA1FMTDg9yFxLSe4XC37begHTDh5seS1I/gaW025Yo/yUFbmSalShzon1Kch
JTNn2hKS1hyWVIkf5n2LHEQgt0zeUNTkMaYrDdkMZdo3fJCDHx7hxzAC+8rs
XYBokTPfEJk6JVlZx3Yl0UnTYMPvSUt6zghf8dhIi5VDnwO+Hcb1MHK1msK+
oy5qkYS7wVC3cfiq8ojBcMtFMJXnDiIcLPFJClDFGkJBhXGEZV6VcCQFoey7
tyvBF6lw0uKH6nyQ5fqhrKx/Kxe8+3coucpGN4U5+RYux7vnhGO5P7hKenqz
xSO3ivjisf345fg5Pt1ScCRkkzR1U1Dur+NCec4LZ49Ynx9P04GmOvlnqcKZ
79BfJfcjkg==
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlLlKQ0EYRi/iA9gpWIgvIXbjM1iKWAjW2vgeZnWN+AgWZhEULCxELFyi
ZnfJavbeJOr/33xXcj8CNnY5EA7k3GWYmTuzq+uLa2OWZS38/MYt0DbDHH8Q
8uTB64bRNrmsQkaPRoQSGX1ZqZPR9/eEazL6klIlowf8wiUZ3ecV3snowYDw
SEafUmJk9HmlQh4cX4GM7vUId2T0VFIho28or2T0w5DQIKN3PoUaGX1FaZLR
0ymh6sz7hHLk9PMzoU5Gz6SFmvFsCbd9Xzj9qyc0nfe49xfuTyYUEzoQbmh8
+N8e37HTbVpk9MSzUCajbypFMvq0UiOjzyklMnqvKzRMJCwU+/vjzemTSpns
nh97feJmd0e4cvppTKiQ0bsdoW5ecorZDgpPtD6xqFA2fp+Qp/vtdUj1v797
Oqdwnb2Ov++fUVpk9PCJUCC7n4t5wz5xj9/+rj6ceXbPP86NbEb58/k56qP+
n/0bdGyYqQ==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}], False->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVdd22zAMDUBxSaJm5HgljrPTJN177/nWh35AfqD//1aAVGrrJLbbk+BI
FEmQFxcXlDQ9//3zHNfW1gZ0/6IbbsYOhahvCGrO9qHM/mnhmMMLoKv0RD6V
cPyZHTTcDI5NkYnQq4UQ0BdiAkKYZagl1OOw5fWtcgyvPp7AnefsGG0DBIeA
9RZVrNfcwBasb4nlbP9u4cbc/XD/zXcIgC2qYYazJTQiGSYHS0Hr8Yu5LV/h
ePvR5oMOKswAQ0M8R5OloFMWSNRE+IAVeALv38KX26wAV1uU5Ojv16PQy8q+
gMP+9ABW5C9KbkZi6qv19DGc/ID9b+CrZTjbbVagH3p91oIcL+GZ6JDNUet4
HpWUXxx5BaWZxRh1FQgH6JqoKcarF/2/dblez7IE0XmWGlFL/6ygchIqCmPo
CTSmDq9SaCUUhfI7K3pKlwJESZiAveHGxoA7ESYGpM2B62WMJXzEoojRRhpt
IlFHPKFyjGRBQ2fjBG0WttpMcYIFWhNAAXaHA0+M4yiUXK6UxJWAGmKNsdII
cYGVAadTSQPKS2KkEovGc8swk2ALhg8Tww1qdrjRNmSTtOUqHDhk3jktVpwK
QOLoZngwWICyTMFTqdhHMrQTDQPeazzXdgWXK+e4KV3AhH0US1Ergo8x16QV
R5QXZWWeJJW6qLNHPWq8mEFpS0hac1pSeX4YxjlyEhmFZfKGsqaIhd+Sks+Q
0i7lhZx8cwYPmx5wrMA+yaCwyMqnRCb2IiubsF9JTKRJMeX3JKd5VoS3OEyl
xShBFxLebXrDpsfVSlv2FZ2iHMn4NBg6bZy+ihxiZvjIFWAixyeIcLDNT1KC
qogha1Gh14NB6LXpSEpC2bm3y8O3FmHniF9U55INhnttz7pZuWDu79BqFZxJ
F2bhWzjo7Zx6HMvng6ukuztzvGJXa66NmF/+crzrHe1ScmTkk/SouqB8vq42
0jl0EnuF9/RsMR1Io4U/S5Wt+A79AaDyK/w=
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlUtrE1EYhgfxBwguFBTEvyAiFhfzHxRFvC0E17rQhYLiT7D3u/gTXNg2
3sBFFyIutE1qm17UZJKZZJK0KFjBtur7Td4jnhe7dTMPhAfyQObMnO9Mjl69
fvraniAI7v/+7A3IRri7/8VGWJwHVTH72qohZp+ZBnUx+yWjJWYfHwNvxOwX
jKaYfXAAzIrZ+/tARcw+NAhKYvaTRiJmP2gUxH+vLxKzl5cMMXtfL3gvZn84
Cdpi9hvGJzH7yxegJWa/YnTE7Mtl0HTPfZ/xyPWVZZCGvQ/Au65fSV9aNMLJ
CfDW9a0fIBWz/9wBHbcOf/74PX83W99j128ZNTH71BMQidmfFkAiZj9irIvZ
Fz+AWMx+yEjF7KeMVOw/32x/iuHoCHjt+s42aIfTU6DWna/Prm9vgVb4cc0I
h4fAgusZ62L2A0YsZj9h1MX+9bN9LHfP75zrZ42a2L9/3+wXjVTM7q8jm9NE
5q8wA+JwoB/8eX+eNxIx+00jErP3GA2xvy7uG88R+z2jI2Y/bLTE/v1l76WG
mxP/+r7ZN7+Bupj9mFERsz9/Bppi9myeEzH7QgnUxOy3jYaY/a7RFrNfNtpi
9nNGXcw+NgoaYnbOk2///PJ/bXXFcP37JojF7CPDIBazf/0CKmL2+TlQEbPf
MZpi9olxkIrZS0UQidn3G1Ux+3EjErOfMapi9mBX8p73vOc973n/P/0XnFJ2
Rg==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}]}, "\"\<Click me\>\""],
      "Toggle changes in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["MatrixRank",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/MatrixRank"], "[", 
       StyleBox["m", "TI"], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the rank of the matrix ",
     Cell[BoxData[
      StyleBox["m", "TI"]], "InlineFormula"],
     ". "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->28498]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["MatrixRank",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MatrixRank"]], "InlineFormula"],
 " works on both numerical and symbolic matrices. "
}], "Notes",
 CellID->17369],

Cell["\<\
The rank of a matrix is the number of linearly independent rows or columns. \
\>", "Notes",
 CellID->17291],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["MatrixRank",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/MatrixRank"], "[", 
   RowBox[{
    StyleBox["m", "TI"], ",", 
    RowBox[{
     ButtonBox["Modulus",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Modulus"], "->", 
     StyleBox["n", "TI"]}]}], "]"}]], "InlineFormula"],
 " finds the rank for integer matrices modulo ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->8400],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["MatrixRank",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/MatrixRank"], "[", 
   RowBox[{
    StyleBox["m", "TI"], ",", 
    RowBox[{
     ButtonBox["ZeroTest",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/ZeroTest"], "->", 
     StyleBox["test", "TI"]}]}], "]"}]], "InlineFormula"],
 " evaluates ",
 Cell[BoxData[
  RowBox[{
   StyleBox["test", "TI"], "[", 
   RowBox[{
    StyleBox["m", "TI"], "[", 
    RowBox[{"[", 
     RowBox[{
      StyleBox["i", "TI"], ",", 
      StyleBox["j", "TI"]}], "]"}], "]"}], "]"}]], "InlineFormula"],
 " to determine whether matrix elements are zero. The default setting is ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["ZeroTest",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ZeroTest"], "->", 
   ButtonBox["Automatic",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Automatic"]}]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->14179],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["MatrixRank",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/MatrixRank"], "[", 
   RowBox[{
    StyleBox["m", "TI"], ",", 
    RowBox[{
     ButtonBox["Tolerance",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Tolerance"], "->", 
     StyleBox["t", "TI"]}]}], "]"}]], "InlineFormula"],
 " gives the minimum rank with each element in a numerical matrix assumed to \
be correct only to within tolerance ",
 Cell[BoxData[
  StyleBox["t", "TI"]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->24117],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["MatrixRank",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MatrixRank"]], "InlineFormula"],
 " works with sparse arrays."
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->65797184]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->4247027],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(1)", "ExampleCount"]
}], "ExampleSection",
 CellID->2585440],

Cell["Find the number of linearly independent rows:", "ExampleText",
 CellID->2042804],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixRank", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1", ",", "2", ",", "3"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"4", ",", "5", ",", "6"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"7", ",", "8", ",", "9"}], "}"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->49101319],

Cell[BoxData["2"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->468679026]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->17981],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["MatrixRank",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MatrixRank"]], "InlineFormula"],
 " can handle rectangular matrices:"
}], "ExampleText",
 CellID->434305127],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixRank", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1", ",", "0"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"3", ",", "2"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"7", ",", "2"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"8", ",", "1"}], "}"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->928474369],

Cell[BoxData["2"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->864250741]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->14621536],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["MatrixRank",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MatrixRank"]], "InlineFormula"],
 " assumes all symbols to be independent:"
}], "ExampleText",
 CellID->149084181],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixRank", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"a", ",", "b"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"c", ",", "d"}], "}"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->273900892],

Cell[BoxData["2"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->54893953]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixRank", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"a", ",", "b"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"2", "a"}], ",", 
      RowBox[{"2", "b"}]}], "}"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->341506674],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->382953675]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->193813815],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["MatrixRank",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MatrixRank"]], "InlineFormula"],
 " for a machine\[Hyphen]number matrix:"
}], "ExampleText",
 CellID->56269021],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1.", ",", "2.", ",", "3."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4.", ",", "5.", ",", "6."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"7.", ",", "8.", ",", "9."}], "}"}]}], "}"}]}], ";"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->211886945],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixRank", "[", "m", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->57428873],

Cell[BoxData["2"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->52373403]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->711910507],

Cell["\<\
Find the matrix rank for a complex\[Hyphen]valued matrix:\
\>", "ExampleText",
 CellID->233207340],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixRank", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1", ",", " ", "I"}], "}"}], ",", " ", 
    RowBox[{"{", 
     RowBox[{"I", ",", " ", 
      RowBox[{"-", "1"}]}], "}"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->763939373],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->773088137]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->541429861],

Cell["Find the rank of a sparse matrix:", "ExampleText",
 CellID->332961116],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"SparseArray", "[", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"Band", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"1", ",", "1"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"50", ",", "50"}], "}"}]}], "]"}], "\[Rule]", "1"}], ",", 
      RowBox[{
       RowBox[{"Band", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"1", ",", "50"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"50", ",", "1"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"1", ",", 
           RowBox[{"-", "1"}]}], "}"}]}], "]"}], "\[Rule]", "1"}]}], "}"}], 
    "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->633320560],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixRank", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->183040289],

Cell[BoxData["25"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->16537343]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", "m", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->143914957],

Cell[GraphicsData["CompressedBitmap", "\<\
eJy9l7FPAjEUxnuehBAMEkSUEHJgkCAaGRwcHIyDGgcGBwcHh4aYwGA0yP9f
+rjP3Fkul3rvxSbH1/b7tTTtXfs61av5+4deLWa697jUX/PF7Lv38Lm0VWGg
lNL0HCubNza3eajKppDym6Te7E+QV7sb18aFV8qluN9MnHZs4QUabvWTy6R6
DuE+Q0sMqgR3Ci0zqDLcJ2iFQVXg3kOrDKrqrMydU85utYc+b6E1BlWDewOt
M6g63Gtog0E14F5BmwyqCXcCbTGolrNC505ZstURRnIGbWdSbbin0A6D6sDt
Q7sMqptySSMGFaVm4Oefi1J9Z+4PnbJkqxOM5AA6yKQGcPehQwY1NMl+Qzpi
UCOT7KmkYwY1Nsm5QXrBoKiWipfiruQoJWdPclUl3za/99uPKvaFRUZuZ5Dc
sSR3Uskd3u9M8aP+8yz0O50lowbJaEYyypKM/orFn8Va+UXEkpG65A2CfbOR
um394WaXy2TeG3HTDNbohxJP\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{97, 97},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->276217885]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Options",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->26569],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Modulus",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->6816],

Cell["The rank of a matrix depends on the modulus used:", "ExampleText",
 CellID->104071873],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "0", ",", "4"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "0", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "1", ",", "2"}], "}"}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->266557437],

Cell[TextData[{
 "With ordinary arithmetic, ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 " has the full rank of 3:"
}], "ExampleText",
 CellID->573850144],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixRank", "[", "m", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->105389421],

Cell[BoxData["3"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->56543857]
}, Open  ]],

Cell["With arithmetic modulo 5, the rank is only 2:", "ExampleText",
 CellID->735194580],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixRank", "[", 
  RowBox[{"m", ",", 
   RowBox[{"Modulus", "\[Rule]", "5"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->184529261],

Cell[BoxData["2"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->68893419]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Tolerance",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->1883],

Cell[TextData[{
 "The setting of ",
 Cell[BoxData[
  ButtonBox["Tolerance",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Tolerance"]], "InlineFormula"],
 " can affect the estimated rank for numerical ill-conditioned matrices:"
}], "ExampleText",
 CellID->224001787],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "1", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", 
       SuperscriptBox["10", 
        RowBox[{"-", "10"}]], ",", " ", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "0", ",", 
       SuperscriptBox["10", 
        RowBox[{"-", "20"}]]}], "}"}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->108806433],

Cell[TextData[{
 "In exact arithmetic, ",
 Cell[BoxData["m"], "InlineFormula"],
 " has full rank:"
}], "ExampleText",
 CellID->98733497],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixRank", "[", "m", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->152750405],

Cell[BoxData["3"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->138687001]
}, Open  ]],

Cell["\<\
With machine arithmetic the default is to consider elements that are too \
small as zero:\
\>", "ExampleText",
 CellID->677218086],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixRank", "[", 
  RowBox[{"N", "[", "m", "]"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->229465423],

Cell[BoxData["2"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->763579186]
}, Open  ]],

Cell["\<\
With zero tolerance even small terms may be taken into account:\
\>", "ExampleText",
 CellID->692496789],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixRank", "[", 
  RowBox[{
   RowBox[{"N", "[", "m", "]"}], ",", " ", 
   RowBox[{"Tolerance", "\[Rule]", "0"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->22912155],

Cell[BoxData["3"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->111102934]
}, Open  ]],

Cell["\<\
With a tolerance greater than the pivot in the middle row, the last two rows \
are considered zero:\
\>", "ExampleText",
 CellID->38630175],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixRank", "[", 
  RowBox[{
   RowBox[{"N", "[", "m", "]"}], ",", " ", 
   RowBox[{"Tolerance", "\[Rule]", 
    SuperscriptBox["10", 
     RowBox[{"-", "8"}]]}]}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->802999688],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->415804416]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->5385],

Cell["\<\
Most but not all random 10\[Times]10 0-1 matrices have full rank:\
\>", "ExampleText",
 CellID->991741],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"MatrixRank", "[", 
    RowBox[{"RandomInteger", "[", 
     RowBox[{"1", ",", 
      RowBox[{"{", 
       RowBox[{"10", ",", "10"}], "}"}]}], "]"}], "]"}], ",", 
   RowBox[{"{", "20", "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->102017883],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "8", ",", "10", ",", "10", ",", "10", ",", "10", ",", "10", ",", "10", ",", 
   "10", ",", "10", ",", "10", ",", "10", ",", "10", ",", "10", ",", "10", 
   ",", "10", ",", "9", ",", "10", ",", "8", ",", "9", ",", "9"}], 
  "}"}]], "Output",
 ImageSize->{476, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->328702433]
}, Open  ]],

Cell["\<\
Estimate the average rank of a random 10\[Times]10 0-1 matrix:\
\>", "ExampleText",
 CellID->488020517],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Block", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"n", " ", "=", " ", "10000"}], "}"}], ",", " ", 
   RowBox[{"N", "[", 
    RowBox[{
     RowBox[{"Sum", "[", 
      RowBox[{
       RowBox[{"MatrixRank", "[", 
        RowBox[{"RandomInteger", "[", 
         RowBox[{"1", ",", 
          RowBox[{"{", 
           RowBox[{"10", ",", "10"}], "}"}]}], "]"}], "]"}], ",", 
       RowBox[{"{", "n", "}"}]}], "]"}], "/", "n"}], "]"}]}], "]"}]], "Input",\

 CellLabel->"In[2]:=",
 CellID->442485436],

Cell[BoxData["9.686`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->277384079]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->374348564],

Cell["Find the ranks of coprimality arrays:", "ExampleText",
 CellID->294558787],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"If", "[", 
      RowBox[{
       RowBox[{"CoprimeQ", "[", 
        RowBox[{"x", ",", "y"}], "]"}], ",", "1", ",", "0"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "20"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", "20"}], "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->33525623],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixRank", "[", "m", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->15120155],

Cell[BoxData["13"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->676214662]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", "m", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->568791851],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzzTSzJSM1NLMlMTlRwL0osyMhMLlZwyy8CCjEzMjAwRIKwBAOQ/R/IAmOQ
EBAwg9hgwBAKJBiJEIUxWYBsFiDNC6VZoTQLEs0KpHmgNEycbdREuIneSCqY
BlAFM5KP0H1IehiNmkTYJCdo2uHBGSf0UYGcptHTMq60jYtPONeM2oDLBnYk
lRxAmhNKw0zgRKNHqknYYgI9R1IvjkeuyYZQE2UZcJUcg0XFYCzdB6NJmlAV
ajhDkj4qhkK7bCiYCCtT0ctODjzi2GNkeJs0HFoHozYQY4MEVIUUlJbEmSYG
VuVgrB0Go0kCUBWCWOihpGM45K1RG0br5NE6ebjZEIwzhqgn8x9lFBk67swI
AF2qWDE=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{89, 89},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->869805102]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"MatrixRank", "[", 
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{"If", "[", 
       RowBox[{
        RowBox[{"CoprimeQ", "[", 
         RowBox[{"x", ",", "y"}], "]"}], ",", "1", ",", "0"}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", "n"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"y", ",", "n"}], "}"}]}], "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "50"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->363833727],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1", ",", "2", ",", "3", ",", "3", ",", "4", ",", "5", ",", "6", ",", "6", 
   ",", "6", ",", "7", ",", "8", ",", "8", ",", "9", ",", "10", ",", "11", 
   ",", "11", ",", "12", ",", "12", ",", "13", ",", "13", ",", "14", ",", 
   "15", ",", "16", ",", "16", ",", "16", ",", "17", ",", "17", ",", "17", 
   ",", "18", ",", "19", ",", "20", ",", "20", ",", "21", ",", "22", ",", 
   "23", ",", "23", ",", "24", ",", "25", ",", "26", ",", "26", ",", "27", 
   ",", "28", ",", "29", ",", "29", ",", "29", ",", "30", ",", "31", ",", 
   "31", ",", "31", ",", "31"}], "}"}]], "Output",
 ImageSize->{435, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->211987892]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", "%", "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->318729659],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], 
    LineBox[{{1., 1.}, {2., 2.}, {3., 3.}, {4., 3.}, {5., 4.}, {6., 5.}, {7., 
     6.}, {8., 6.}, {9., 6.}, {10., 7.}, {11., 8.}, {12., 8.}, {13., 9.}, {
     14., 10.}, {15., 11.}, {16., 11.}, {17., 12.}, {18., 12.}, {19., 13.}, {
     20., 13.}, {21., 14.}, {22., 15.}, {23., 16.}, {24., 16.}, {25., 16.}, {
     26., 17.}, {27., 17.}, {28., 17.}, {29., 18.}, {30., 19.}, {31., 20.}, {
     32., 20.}, {33., 21.}, {34., 22.}, {35., 23.}, {36., 23.}, {37., 24.}, {
     38., 25.}, {39., 26.}, {40., 26.}, {41., 27.}, {42., 28.}, {43., 29.}, {
     44., 29.}, {45., 29.}, {46., 30.}, {47., 31.}, {48., 31.}, {49., 31.}, {
     50., 31.}}]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  PlotRange->{{0., 50.}, {0., 31.}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 118},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->6512877]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->28402],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["MatrixRank",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/MatrixRank"], "[", 
   StyleBox["m", "TI"], "]"}]], "InlineFormula"],
 " is equal to ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Length",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Length"], "[", 
   RowBox[{
    ButtonBox["SingularValueList",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/SingularValueList"], "[", 
    StyleBox["m", "TI"], "]"}], "]"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->55263802],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", " ", "=", " ", 
   RowBox[{"N", "[", 
    RowBox[{"RandomInteger", "[", 
     RowBox[{"1", ",", " ", 
      RowBox[{"{", 
       RowBox[{"4", ",", "4"}], "}"}]}], "]"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->142726290],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SingularValueList", "[", "m", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->467505575],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "2.052880840033716`", ",", "1.2086401975729215`", ",", 
   "0.5699729199123005`"}], "}"}]], "Output",
 ImageSize->{195, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->4710411]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"MatrixRank", "[", "m", "]"}], " ", "\[Equal]", " ", 
  RowBox[{"Length", "[", "%", "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->102473470],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->160979876]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1170489],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["MatrixRank",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/MatrixRank"], "[", 
   StyleBox["m", "TI"], "]"}]], "InlineFormula"],
 " plus the dimension of the null space is equal to the number of columns of \
",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->803370024],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", " ", "=", " ", 
   RowBox[{"N", "[", 
    RowBox[{"RandomInteger", "[", 
     RowBox[{"1", ",", " ", 
      RowBox[{"{", 
       RowBox[{"4", ",", "4"}], "}"}]}], "]"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->401911608],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"r", ",", " ", "c"}], "}"}], " ", "=", " ", 
   RowBox[{"Dimensions", "[", "m", "]"}]}], ";"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->354050287],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ns", " ", "=", " ", 
  RowBox[{"NullSpace", "[", "m", "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->165632243],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0.`", ",", "0.`", ",", "0.`", ",", "1.`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.`", ",", 
     RowBox[{"-", "1.`"}], ",", "0.`", ",", "0.`"}], "}"}]}], "}"}]], "Output",\

 ImageSize->{244, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->782890309]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"c", " ", "\[Equal]", " ", 
  RowBox[{
   RowBox[{"Length", "[", "ns", "]"}], " ", "+", " ", 
   RowBox[{"MatrixRank", "[", "m", "]"}]}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->223716098],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->225209597]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->53178406],

Cell["The column and row rank of a matrix are equal:", "ExampleText",
 CellID->443631555],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "2", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4", ",", "5", ",", "6"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"7", ",", "8", ",", "9"}], "}"}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->73542669],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"MatrixRank", "[", "m", "]"}], ",", 
   RowBox[{"MatrixRank", "[", 
    RowBox[{"m", "\[Transpose]"}], "]"}]}], "}"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->239521825],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"2", ",", "2"}], "}"}]], "Output",
 ImageSize->{44, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->26878093]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->17211467],

Cell["The outer product of vectors have matrix rank 1:", "ExampleText",
 CellID->614734970],

Cell[BoxData[
 RowBox[{
  RowBox[{"v", "=", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", "1000"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->104557984],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixRank", "[", 
  RowBox[{"KroneckerProduct", "[", 
   RowBox[{"v", ",", "v"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->224450679],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->153627330]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->5230],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["MatrixRank",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MatrixRank"]], "InlineFormula"],
 " may depend on the precision of the given matrix:"
}], "ExampleText",
 CellID->82016794],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "2", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4", ",", "5", ",", "6"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"7", ",", "8", ",", 
       RowBox[{"9", " ", "+", " ", 
        SuperscriptBox["10", 
         RowBox[{"-", "20"}]]}]}], "}"}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->236527520],

Cell["Use exact arithmetic to compute the matrix rank exactly:", "ExampleText",
 CellID->987359563],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixRank", "[", "m", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->40280618],

Cell[BoxData["3"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->173238885]
}, Open  ]],

Cell[TextData[{
 "Use machine arithmetic. Machine numbers cannot distinguish between ",
 Cell[BoxData[
  FormBox["9", TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"9", "+", 
    SuperscriptBox["10", 
     RowBox[{"-", "20"}]]}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->544761386],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixRank", "[", 
  RowBox[{"N", "[", "m", "]"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->267466789],

Cell[BoxData["2"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->2395284]
}, Open  ]],

Cell["\<\
Use 24\[Hyphen]digit precision arithmetic:\
\>", "ExampleText",
 CellID->249357343],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixRank", "[", 
  RowBox[{"N", "[", 
   RowBox[{"m", ",", " ", "24"}], "]"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->142945797],

Cell[BoxData["3"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->563090871]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->5936],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["NullSpace",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NullSpace"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Det",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Det"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Eigensystem",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Eigensystem"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["RowReduce",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RowReduce"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SingularValueList",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SingularValueList"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Inverse",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Inverse"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->21867]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->2317],

Cell[TextData[ButtonBox["Solving Linear Systems",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/SolvingLinearSystems"]], "Tutorials",
 CellID->19914]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection",
 CellID->1595],

Cell[TextData[ButtonBox["Implementation notes: Numerical and Related \
Functions",
 BaseStyle->"Hyperlink",
 ButtonData->{
   URL["http://reference.wolfram.com/mathematica/note/\
SomeNotesOnInternalImplementation.html#30506"], None}]], "RelatedLinks",
 CellID->20859006],

Cell[TextData[ButtonBox["Implementation notes: Algebra and Calculus",
 BaseStyle->"Hyperlink",
 ButtonData->{
   URL["http://reference.wolfram.com/mathematica/note/\
SomeNotesOnInternalImplementation.html#22549"], None}]], "RelatedLinks"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->538383311],

Cell[TextData[ButtonBox["Linear Systems",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/LinearSystems"]], "MoreAbout",
 CellID->137059175],

Cell[TextData[ButtonBox["Matrices and Linear Algebra",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/MatricesAndLinearAlgebra"]], "MoreAbout",
 CellID->5486831]
}, Open  ]],

Cell["New in 5  |  Last modified in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"MatrixRank - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 0, 5.9478318}", "context" -> "System`", 
    "keywords" -> {"basis dimension", "column rank", "row rank"}, "index" -> 
    True, "label" -> "Built-in Mathematica Symbol", "language" -> "en", 
    "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "MatrixRank[m] gives the rank of the matrix m. ", 
    "synonyms" -> {"matrix rank"}, "title" -> "MatrixRank", "type" -> 
    "Symbol", "uri" -> "ref/MatrixRank"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[11042, 315, 358, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->4247027]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 42701, 1566}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2434, 57, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3057, 84, 4194, 83, 70, "ObjectNameGrid"],
Cell[7254, 169, 551, 18, 70, "Usage",
 CellID->28498]
}, Open  ]],
Cell[CellGroupData[{
Cell[7842, 192, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[8320, 209, 219, 7, 70, "Notes",
 CellID->17369],
Cell[8542, 218, 117, 3, 70, "Notes",
 CellID->17291],
Cell[8662, 223, 490, 18, 70, "Notes",
 CellID->8400],
Cell[9155, 243, 950, 34, 70, "Notes",
 CellID->14179],
Cell[10108, 279, 564, 19, 70, "Notes",
 CellID->24117],
Cell[10675, 300, 330, 10, 70, "Notes",
 CellID->65797184]
}, Closed]],
Cell[CellGroupData[{
Cell[11042, 315, 358, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->4247027],
Cell[CellGroupData[{
Cell[11425, 329, 146, 5, 70, "ExampleSection",
 CellID->2585440],
Cell[11574, 336, 86, 1, 70, "ExampleText",
 CellID->2042804],
Cell[CellGroupData[{
Cell[11685, 341, 345, 11, 28, "Input",
 CellID->49101319],
Cell[12033, 354, 158, 5, 36, "Output",
 CellID->468679026]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[12240, 365, 216, 7, 70, "ExampleSection",
 CellID->17981],
Cell[12459, 374, 214, 7, 70, "ExampleText",
 CellID->434305127],
Cell[CellGroupData[{
Cell[12698, 385, 377, 13, 70, "Input",
 CellID->928474369],
Cell[13078, 400, 158, 5, 36, "Output",
 CellID->864250741]
}, Open  ]],
Cell[13251, 408, 124, 3, 70, "ExampleDelimiter",
 CellID->14621536],
Cell[13378, 413, 220, 7, 70, "ExampleText",
 CellID->149084181],
Cell[CellGroupData[{
Cell[13623, 424, 255, 9, 70, "Input",
 CellID->273900892],
Cell[13881, 435, 157, 5, 36, "Output",
 CellID->54893953]
}, Open  ]],
Cell[CellGroupData[{
Cell[14075, 445, 299, 11, 70, "Input",
 CellID->341506674],
Cell[14377, 458, 158, 5, 36, "Output",
 CellID->382953675]
}, Open  ]],
Cell[14550, 466, 125, 3, 70, "ExampleDelimiter",
 CellID->193813815],
Cell[14678, 471, 217, 7, 70, "ExampleText",
 CellID->56269021],
Cell[14898, 480, 369, 13, 70, "Input",
 CellID->211886945],
Cell[CellGroupData[{
Cell[15292, 497, 105, 3, 70, "Input",
 CellID->57428873],
Cell[15400, 502, 157, 5, 36, "Output",
 CellID->52373403]
}, Open  ]],
Cell[15572, 510, 125, 3, 70, "ExampleDelimiter",
 CellID->711910507],
Cell[15700, 515, 108, 3, 70, "ExampleText",
 CellID->233207340],
Cell[CellGroupData[{
Cell[15833, 522, 292, 10, 70, "Input",
 CellID->763939373],
Cell[16128, 534, 158, 5, 36, "Output",
 CellID->773088137]
}, Open  ]],
Cell[16301, 542, 125, 3, 70, "ExampleDelimiter",
 CellID->541429861],
Cell[16429, 547, 76, 1, 70, "ExampleText",
 CellID->332961116],
Cell[16508, 550, 747, 25, 70, "Input",
 CellID->633320560],
Cell[CellGroupData[{
Cell[17280, 579, 106, 3, 70, "Input",
 CellID->183040289],
Cell[17389, 584, 158, 5, 36, "Output",
 CellID->16537343]
}, Open  ]],
Cell[CellGroupData[{
Cell[17584, 594, 105, 3, 70, "Input",
 CellID->143914957],
Cell[17692, 599, 723, 16, 70, "Output",
 Evaluatable->False,
 CellID->276217885]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[18464, 621, 218, 7, 70, "ExampleSection",
 CellID->26569],
Cell[CellGroupData[{
Cell[18707, 632, 223, 7, 70, "ExampleSubsection",
 CellID->6816],
Cell[18933, 641, 92, 1, 70, "ExampleText",
 CellID->104071873],
Cell[19028, 644, 368, 12, 70, "Input",
 CellID->266557437],
Cell[19399, 658, 170, 6, 70, "ExampleText",
 CellID->573850144],
Cell[CellGroupData[{
Cell[19594, 668, 106, 3, 70, "Input",
 CellID->105389421],
Cell[19703, 673, 157, 5, 36, "Output",
 CellID->56543857]
}, Open  ]],
Cell[19875, 681, 88, 1, 70, "ExampleText",
 CellID->735194580],
Cell[CellGroupData[{
Cell[19988, 686, 165, 5, 70, "Input",
 CellID->184529261],
Cell[20156, 693, 157, 5, 36, "Output",
 CellID->68893419]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[20362, 704, 225, 7, 70, "ExampleSubsection",
 CellID->1883],
Cell[20590, 713, 269, 8, 70, "ExampleText",
 CellID->224001787],
Cell[20862, 723, 483, 16, 70, "Input",
 CellID->108806433],
Cell[21348, 741, 136, 5, 70, "ExampleText",
 CellID->98733497],
Cell[CellGroupData[{
Cell[21509, 750, 106, 3, 70, "Input",
 CellID->152750405],
Cell[21618, 755, 158, 5, 36, "Output",
 CellID->138687001]
}, Open  ]],
Cell[21791, 763, 140, 4, 70, "ExampleText",
 CellID->677218086],
Cell[CellGroupData[{
Cell[21956, 771, 134, 4, 70, "Input",
 CellID->229465423],
Cell[22093, 777, 158, 5, 36, "Output",
 CellID->763579186]
}, Open  ]],
Cell[22266, 785, 114, 3, 70, "ExampleText",
 CellID->692496789],
Cell[CellGroupData[{
Cell[22405, 792, 200, 6, 70, "Input",
 CellID->22912155],
Cell[22608, 800, 158, 5, 36, "Output",
 CellID->111102934]
}, Open  ]],
Cell[22781, 808, 149, 4, 70, "ExampleText",
 CellID->38630175],
Cell[CellGroupData[{
Cell[22955, 816, 249, 8, 70, "Input",
 CellID->802999688],
Cell[23207, 826, 158, 5, 36, "Output",
 CellID->415804416]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[23426, 838, 222, 7, 70, "ExampleSection",
 CellID->5385],
Cell[23651, 847, 113, 3, 70, "ExampleText",
 CellID->991741],
Cell[CellGroupData[{
Cell[23789, 854, 312, 10, 70, "Input",
 CellID->102017883],
Cell[24104, 866, 413, 11, 36, "Output",
 CellID->328702433]
}, Open  ]],
Cell[24532, 880, 113, 3, 70, "ExampleText",
 CellID->488020517],
Cell[CellGroupData[{
Cell[24670, 887, 519, 17, 70, "Input",
 CellID->442485436],
Cell[25192, 906, 163, 5, 36, "Output",
 CellID->277384079]
}, Open  ]],
Cell[25370, 914, 125, 3, 70, "ExampleDelimiter",
 CellID->374348564],
Cell[25498, 919, 80, 1, 70, "ExampleText",
 CellID->294558787],
Cell[25581, 922, 423, 14, 70, "Input",
 CellID->33525623],
Cell[CellGroupData[{
Cell[26029, 940, 105, 3, 70, "Input",
 CellID->15120155],
Cell[26137, 945, 159, 5, 36, "Output",
 CellID->676214662]
}, Open  ]],
Cell[CellGroupData[{
Cell[26333, 955, 105, 3, 70, "Input",
 CellID->568791851],
Cell[26441, 960, 585, 14, 70, "Output",
 Evaluatable->False,
 CellID->869805102]
}, Open  ]],
Cell[CellGroupData[{
Cell[27063, 979, 534, 17, 70, "Input",
 CellID->363833727],
Cell[27600, 998, 752, 15, 71, "Output",
 CellID->211987892]
}, Open  ]],
Cell[CellGroupData[{
Cell[28389, 1018, 108, 3, 70, "Input",
 CellID->318729659],
Cell[28500, 1023, 1046, 23, 139, "Output",
 CellID->6512877]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[29595, 1052, 233, 7, 70, "ExampleSection",
 CellID->28402],
Cell[29831, 1061, 551, 20, 70, "ExampleText",
 CellID->55263802],
Cell[30385, 1083, 277, 9, 70, "Input",
 CellID->142726290],
Cell[CellGroupData[{
Cell[30687, 1096, 113, 3, 70, "Input",
 CellID->467505575],
Cell[30803, 1101, 272, 9, 36, "Output",
 CellID->4710411]
}, Open  ]],
Cell[CellGroupData[{
Cell[31112, 1115, 179, 5, 70, "Input",
 CellID->102473470],
Cell[31294, 1122, 161, 5, 36, "Output",
 CellID->160979876]
}, Open  ]],
Cell[31470, 1130, 123, 3, 70, "ExampleDelimiter",
 CellID->1170489],
Cell[31596, 1135, 372, 13, 70, "ExampleText",
 CellID->803370024],
Cell[31971, 1150, 277, 9, 70, "Input",
 CellID->401911608],
Cell[32251, 1161, 212, 7, 70, "Input",
 CellID->354050287],
Cell[CellGroupData[{
Cell[32488, 1172, 139, 4, 70, "Input",
 CellID->165632243],
Cell[32630, 1178, 381, 13, 36, "Output",
 CellID->782890309]
}, Open  ]],
Cell[CellGroupData[{
Cell[33048, 1196, 215, 6, 70, "Input",
 CellID->223716098],
Cell[33266, 1204, 161, 5, 36, "Output",
 CellID->225209597]
}, Open  ]],
Cell[33442, 1212, 124, 3, 70, "ExampleDelimiter",
 CellID->53178406],
Cell[33569, 1217, 89, 1, 70, "ExampleText",
 CellID->443631555],
Cell[33661, 1220, 357, 12, 70, "Input",
 CellID->73542669],
Cell[CellGroupData[{
Cell[34043, 1236, 222, 7, 70, "Input",
 CellID->239521825],
Cell[34268, 1245, 202, 7, 36, "Output",
 CellID->26878093]
}, Open  ]],
Cell[34485, 1255, 124, 3, 70, "ExampleDelimiter",
 CellID->17211467],
Cell[34612, 1260, 91, 1, 70, "ExampleText",
 CellID->614734970],
Cell[34706, 1263, 176, 6, 70, "Input",
 CellID->104557984],
Cell[CellGroupData[{
Cell[34907, 1273, 173, 5, 70, "Input",
 CellID->224450679],
Cell[35083, 1280, 158, 5, 36, "Output",
 CellID->153627330]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[35290, 1291, 225, 7, 70, "ExampleSection",
 CellID->5230],
Cell[35518, 1300, 229, 7, 70, "ExampleText",
 CellID->82016794],
Cell[35750, 1309, 463, 15, 70, "Input",
 CellID->236527520],
Cell[36216, 1326, 99, 1, 70, "ExampleText",
 CellID->987359563],
Cell[CellGroupData[{
Cell[36340, 1331, 105, 3, 70, "Input",
 CellID->40280618],
Cell[36448, 1336, 158, 5, 36, "Output",
 CellID->173238885]
}, Open  ]],
Cell[36621, 1344, 342, 12, 70, "ExampleText",
 CellID->544761386],
Cell[CellGroupData[{
Cell[36988, 1360, 134, 4, 70, "Input",
 CellID->267466789],
Cell[37125, 1366, 156, 5, 36, "Output",
 CellID->2395284]
}, Open  ]],
Cell[37296, 1374, 93, 3, 70, "ExampleText",
 CellID->249357343],
Cell[CellGroupData[{
Cell[37414, 1381, 164, 5, 70, "Input",
 CellID->142945797],
Cell[37581, 1388, 158, 5, 36, "Output",
 CellID->563090871]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[37800, 1400, 310, 9, 70, "SeeAlsoSection",
 CellID->5936],
Cell[38113, 1411, 1447, 53, 70, "SeeAlso",
 CellID->21867]
}, Open  ]],
Cell[CellGroupData[{
Cell[39597, 1469, 313, 9, 70, "TutorialsSection",
 CellID->2317],
Cell[39913, 1480, 153, 3, 70, "Tutorials",
 CellID->19914]
}, Open  ]],
Cell[CellGroupData[{
Cell[40103, 1488, 320, 9, 70, "RelatedLinksSection",
 CellID->1595],
Cell[40426, 1499, 270, 6, 70, "RelatedLinks",
 CellID->20859006],
Cell[40699, 1507, 238, 4, 70, "RelatedLinks"]
}, Open  ]],
Cell[CellGroupData[{
Cell[40974, 1516, 319, 9, 70, "MoreAboutSection",
 CellID->538383311],
Cell[41296, 1527, 139, 3, 70, "MoreAbout",
 CellID->137059175],
Cell[41438, 1532, 161, 3, 70, "MoreAbout",
 CellID->5486831]
}, Open  ]],
Cell[41614, 1538, 50, 0, 70, "History"],
Cell[41667, 1540, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

