(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     34221,       1224]
NotebookOptionsPosition[     27644,        996]
NotebookOutlinePosition[     28949,       1033]
CellTagsIndexPosition[     28863,       1028]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Algebraic Number Fields" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/AlgebraicNumberFields"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Algebraic Number Fields\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Root" :> Documentation`HelpLookup["paclet:ref/Root"], 
          "RootReduce" :> Documentation`HelpLookup["paclet:ref/RootReduce"], 
          "AlgebraicNumber" :> 
          Documentation`HelpLookup["paclet:ref/AlgebraicNumber"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Root\"\>", 2->"\<\"RootReduce\"\>", 
       3->"\<\"AlgebraicNumber\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Algebraic Numbers" :> 
          Documentation`HelpLookup["paclet:guide/AlgebraicNumbers"], 
          "Algebraic Number Theory" :> 
          Documentation`HelpLookup["paclet:guide/AlgebraicNumberTheory"], 
          "Number Recognition" :> 
          Documentation`HelpLookup["paclet:guide/NumberRecognition"], 
          "Number Theory" :> 
          Documentation`HelpLookup["paclet:guide/NumberTheory"], 
          "Polynomial Algebra" :> 
          Documentation`HelpLookup["paclet:guide/PolynomialAlgebra"], 
          "New in 6.0: Number Theory & Integer Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60NumberTheoryAndIntegerFunctions"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Algebraic Numbers\"\>", 
       2->"\<\"Algebraic Number Theory\"\>", 3->"\<\"Number Recognition\"\>", 
       4->"\<\"Number Theory\"\>", 5->"\<\"Polynomial Algebra\"\>", 
       6->"\<\"New in 6.0: Number Theory & Integer Functions\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["MinimalPolynomial", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["MinimalPolynomial",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/MinimalPolynomial"], "[", 
       RowBox[{
        StyleBox["s", "TI"], ",", 
        StyleBox["x", "TI"]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the minimal polynomial in ",
     Cell[BoxData[
      StyleBox["x", "TI"]], "InlineFormula"],
     " for which the algebraic number ",
     Cell[BoxData[
      StyleBox["s", "TI"]], "InlineFormula"],
     " is a root. "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["MinimalPolynomial",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/MinimalPolynomial"], "[", 
   StyleBox["s", "TI"], "]"}]], "InlineFormula", "TemplateInclusion"],
 " gives a pure function representation of the minimal polynomial of ",
 Cell[BoxData[
  StyleBox["s", "TI"]], "InlineFormula"],
 "."
}], "Notes",
 CellID->1067943069],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["MinimalPolynomial",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/MinimalPolynomial"], "[", 
   RowBox[{
    StyleBox["s", "TI"], ",", 
    StyleBox["x", "TI"], ",", 
    RowBox[{
     ButtonBox["Extension",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Extension"], "->", 
     StyleBox["a", "TI"]}]}], "]"}]], "InlineFormula"],
 " finds the characteristic polynomial of ",
 Cell[BoxData[
  StyleBox["s", "TI"]], "InlineFormula"],
 " over the field ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[DoubleStruckCapitalQ]", "[", "a", "]"}], TraditionalForm]], 
  "InlineMath"],
 "."
}], "Notes",
 CellID->251581752]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->72029767],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->536030079],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MinimalPolynomial", "[", 
  RowBox[{
   RowBox[{"Sqrt", "[", "3", "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->505147914],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "3"}], "+", 
  SuperscriptBox["x", "2"]}]], "Output",
 ImageSize->{44, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->616132984]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->580643178],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MinimalPolynomial", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Sqrt", "[", "2", "]"}], "+", 
    RowBox[{"Sqrt", "[", "3", "]"}]}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->204998416],

Cell[BoxData[
 RowBox[{"1", "-", 
  RowBox[{"10", " ", 
   SuperscriptBox["x", "2"]}], "+", 
  SuperscriptBox["x", "4"]}]], "Output",
 ImageSize->{79, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->180916254]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1293636265],

Cell["Radical expressions:", "ExampleText",
 CellID->928188102],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MinimalPolynomial", "[", 
  RowBox[{
   RowBox[{"Sqrt", "[", 
    RowBox[{"1", "+", 
     RowBox[{"Sqrt", "[", "3", "]"}]}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->736218506],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "2"}], "-", 
  RowBox[{"2", " ", 
   SuperscriptBox["x", "2"]}], "+", 
  SuperscriptBox["x", "4"]}]], "Output",
 ImageSize->{79, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1970120473]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MinimalPolynomial", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{"1", "+", "I"}], ")"}], "/", 
    RowBox[{"Sqrt", "[", "2", "]"}]}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1935708990],

Cell[BoxData[
 RowBox[{"1", "+", 
  SuperscriptBox["x", "4"]}]], "Output",
 ImageSize->{37, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1911920265]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1866821203],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Root",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Root"]], "InlineFormula"],
 " objects:"
}], "ExampleText",
 CellID->927876151],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MinimalPolynomial", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Root", "[", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"2", 
         RowBox[{"#1", "^", "3"}]}], "-", 
        RowBox[{"2", " ", "#1"}], "+", "7"}], "&"}], ",", "1"}], "]"}], "+", 
    "17"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->71816797],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "9785"}], "+", 
  RowBox[{"1732", " ", "x"}], "-", 
  RowBox[{"102", " ", 
   SuperscriptBox["x", "2"]}], "+", 
  RowBox[{"2", " ", 
   SuperscriptBox["x", "3"]}]}]], "Output",
 ImageSize->{175, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->846083794]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->668446203],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["AlgebraicNumber",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AlgebraicNumber"]], "InlineFormula"],
 " objects:"
}], "ExampleText",
 CellID->1088376364],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MinimalPolynomial", "[", 
  RowBox[{
   RowBox[{"AlgebraicNumber", "[", 
    RowBox[{
     RowBox[{"Root", "[", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"-", "2"}], "+", 
         RowBox[{"#1", "^", "3"}]}], "&"}], ",", "2"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "2", ",", "1"}], "}"}]}], "]"}], ",", "x"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1985789751],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "4"}], "-", 
  RowBox[{"6", " ", 
   SuperscriptBox["x", "2"]}], "+", 
  SuperscriptBox["x", "3"]}]], "Output",
 ImageSize->{79, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1641098971]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->638919951],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["MinimalPolynomial",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MinimalPolynomial"]], "InlineFormula"],
 " automatically threads over lists:"
}], "ExampleText",
 CellID->844953649],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MinimalPolynomial", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Sqrt", "[", "3", "]"}], ",", 
     RowBox[{
      RowBox[{"Root", "[", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{"-", "2"}], "+", 
          RowBox[{"#1", "^", "3"}]}], "&"}], ",", "2"}], "]"}], "+", "1"}]}], 
    "}"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->890427757],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"-", "3"}], "+", 
    SuperscriptBox["x", "2"]}], ",", 
   RowBox[{
    RowBox[{"-", "3"}], "+", 
    RowBox[{"3", " ", "x"}], "-", 
    RowBox[{"3", " ", 
     SuperscriptBox["x", "2"]}], "+", 
    SuperscriptBox["x", "3"]}]}], "}"}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{174, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1817927176]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1020263627],

Cell["Express the minimal polynomial as a pure function:", "ExampleText",
 CellID->930701266],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MinimalPolynomial", "[", 
  RowBox[{"Sqrt", "[", 
   RowBox[{"2", "+", 
    RowBox[{"Sqrt", "[", "3", "]"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->25292914],

Cell[BoxData[
 RowBox[{
  RowBox[{"1", "-", 
   RowBox[{"4", " ", 
    SuperscriptBox["#1", "2"]}], "+", 
   SuperscriptBox["#1", "4"]}], "&"}]], "Output",
 ImageSize->{96, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1652201163]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Options",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->2061341341],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Extension",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->1295379749],

Cell[TextData[{
 "Find the characteristic polynomial of ",
 Cell[BoxData[
  SqrtBox["2"]], "InlineFormula"],
 " over the extension ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["E",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/E"], "^", 
   RowBox[{"(", 
    RowBox[{
     ButtonBox["I",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/I"], " ", 
     RowBox[{
      ButtonBox["Pi",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/Pi"], "/", "4"}]}], ")"}]}]], 
  "InlineFormula"],
 " of ",
 Cell[BoxData["\[DoubleStruckCapitalQ]"], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->854615781],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MinimalPolynomial", "[", 
  RowBox[{
   RowBox[{"Sqrt", "[", "2", "]"}], ",", "x", ",", 
   RowBox[{"Extension", " ", "\[Rule]", " ", 
    RowBox[{"E", "^", 
     RowBox[{"(", " ", 
      RowBox[{"I", " ", 
       RowBox[{"Pi", "/", "4"}]}], ")"}]}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1889701305],

Cell[BoxData[
 RowBox[{"4", "-", 
  RowBox[{"4", " ", 
   SuperscriptBox["x", "2"]}], "+", 
  SuperscriptBox["x", "4"]}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{72, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1581390831]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->258228157],

Cell[TextData[{
 "Construct a polynomial with a root ",
 Cell[BoxData[
  SqrtBox[
   RowBox[{"1", "+", 
    SqrtBox["2"]}]]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->2054899579],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MinimalPolynomial", "[", 
  RowBox[{
   RowBox[{"Sqrt", "[", 
    RowBox[{"1", "+", 
     RowBox[{"Sqrt", "[", "2", "]"}]}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->20031],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "1"}], "-", 
  RowBox[{"2", " ", 
   SuperscriptBox["x", "2"]}], "+", 
  SuperscriptBox["x", "4"]}]], "Output",
 ImageSize->{79, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1178081345]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{"%", ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", 
   RowBox[{"Epilog", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"Blue", ",", 
      RowBox[{"PointSize", "[", "Large", "]"}], ",", 
      RowBox[{"Point", "[", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"Sqrt", "[", 
          RowBox[{"1", "+", 
           RowBox[{"Sqrt", "[", "2", "]"}]}], "]"}], ",", "0"}], "}"}], 
       "]"}]}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->23760],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 114},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->873888761]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->544251230],

Cell[TextData[{
 "The degree of the number field generated by ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"(", 
    RowBox[{"2", "-", 
     ButtonBox["I",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/I"]}], ")"}], "/", 
   RowBox[{
    ButtonBox["Sqrt",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Sqrt"], "[", "5", "]"}]}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->1194050958],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Exponent", "[", 
  RowBox[{
   RowBox[{"MinimalPolynomial", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"(", 
       RowBox[{"2", "-", "I"}], ")"}], "/", 
      RowBox[{"Sqrt", "[", "5", "]"}]}], ",", "x"}], "]"}], ",", "x"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2050735036],

Cell[BoxData["4"], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->803215447]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->2123667759],

Cell[TextData[{
 "Compute the extension that defines the number field ",
 Cell[BoxData[
  FormBox[
   RowBox[{"F", "=", 
    RowBox[{"\[DoubleStruckCapitalQ]", "[", 
     RowBox[{
      SqrtBox["3"], ",", 
      SuperscriptBox["\[ExponentialE]", 
       FractionBox[
        RowBox[{"\[ImaginaryI]", " ", "\[Pi]"}], "4"]]}], "]"}]}], 
   TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->182138168],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"F", "=", 
  RowBox[{
   RowBox[{"ToNumberField", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"Sqrt", "[", "3", "]"}], ",", " ", 
       RowBox[{"E", "^", 
        RowBox[{"(", 
         RowBox[{"I", " ", 
          RowBox[{"Pi", "/", "4"}]}], ")"}]}]}], "}"}], ",", " ", "All"}], 
    "]"}], "[", 
   RowBox[{"[", 
    RowBox[{"1", ",", "1"}], "]"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->431021432],

Cell[BoxData[
 RowBox[{"Root", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"100", "-", 
     RowBox[{"72", " ", 
      SuperscriptBox["#1", "2"]}], "+", 
     RowBox[{"56", " ", 
      SuperscriptBox["#1", "4"]}], "-", 
     RowBox[{"12", " ", 
      SuperscriptBox["#1", "6"]}], "+", 
     SuperscriptBox["#1", "8"]}], "&"}], ",", "8"}], "]"}]], "Output",
 ImageSize->{276, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->559812687]
}, Open  ]],

Cell[TextData[{
 "Find the characteristic polynomial of ",
 Cell[BoxData[
  RowBox[{
   SqrtBox["2"], "+", 
   SqrtBox["3"]}]], "InlineFormula"],
 " over ",
 Cell[BoxData["F"], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->502094902],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MinimalPolynomial", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Sqrt", "[", "2", "]"}], "+", 
    RowBox[{"Sqrt", "[", "3", "]"}]}], ",", " ", "x", ",", 
   RowBox[{"Extension", " ", "\[Rule]", " ", "F"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->228651492],

Cell[BoxData[
 RowBox[{"1", "-", 
  RowBox[{"20", " ", 
   SuperscriptBox["x", "2"]}], "+", 
  RowBox[{"102", " ", 
   SuperscriptBox["x", "4"]}], "-", 
  RowBox[{"20", " ", 
   SuperscriptBox["x", "6"]}], "+", 
  SuperscriptBox["x", "8"]}]], "Output",
 ImageSize->{170, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->645972902]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["Root",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Root"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["RootReduce",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RootReduce"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["AlgebraicNumber",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AlgebraicNumber"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->250839057],

Cell[TextData[ButtonBox["Algebraic Number Fields",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/AlgebraicNumberFields"]], "Tutorials",
 CellID->341631938]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->277758029],

Cell[TextData[ButtonBox["Algebraic Numbers",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/AlgebraicNumbers"]], "MoreAbout",
 CellID->337947800],

Cell[TextData[ButtonBox["Algebraic Number Theory",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/AlgebraicNumberTheory"]], "MoreAbout",
 CellID->475253647],

Cell[TextData[ButtonBox["Number Recognition",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NumberRecognition"]], "MoreAbout",
 CellID->53039487],

Cell[TextData[ButtonBox["Number Theory",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NumberTheory"]], "MoreAbout",
 CellID->130969135],

Cell[TextData[ButtonBox["Polynomial Algebra",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/PolynomialAlgebra"]], "MoreAbout",
 CellID->471850283],

Cell[TextData[ButtonBox["New in 6.0: Number Theory & Integer Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/NewIn60NumberTheoryAndIntegerFunctions"]], "MoreAbout",
 CellID->378877077]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"MinimalPolynomial - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 0, 46.2301130}", "context" -> "System`", 
    "keywords" -> {"characteristic polynomial"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "MinimalPolynomial[s, x] gives the minimal polynomial in x for which the \
algebraic number s is a root. ", "synonyms" -> {"minimal polynomial"}, 
    "title" -> "MinimalPolynomial", "type" -> "Symbol", "uri" -> 
    "ref/MinimalPolynomial"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[7329, 217, 359, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->72029767]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 28721, 1021}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2753, 63, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3376, 90, 1582, 37, 70, "ObjectNameGrid"],
Cell[4961, 129, 743, 23, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[5741, 157, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6219, 174, 392, 12, 70, "Notes",
 CellID->1067943069],
Cell[6614, 188, 678, 24, 70, "Notes",
 CellID->251581752]
}, Closed]],
Cell[CellGroupData[{
Cell[7329, 217, 359, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->72029767],
Cell[CellGroupData[{
Cell[7713, 231, 148, 5, 70, "ExampleSection",
 CellID->536030079],
Cell[CellGroupData[{
Cell[7886, 240, 168, 5, 28, "Input",
 CellID->505147914],
Cell[8057, 247, 222, 8, 36, "Output",
 CellID->616132984]
}, Open  ]],
Cell[8294, 258, 125, 3, 70, "ExampleDelimiter",
 CellID->580643178],
Cell[CellGroupData[{
Cell[8444, 265, 226, 7, 28, "Input",
 CellID->204998416],
Cell[8673, 274, 263, 9, 36, "Output",
 CellID->180916254]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[8985, 289, 221, 7, 70, "ExampleSection",
 CellID->1293636265],
Cell[9209, 298, 63, 1, 70, "ExampleText",
 CellID->928188102],
Cell[CellGroupData[{
Cell[9297, 303, 227, 7, 70, "Input",
 CellID->736218506],
Cell[9527, 312, 281, 10, 36, "Output",
 CellID->1970120473]
}, Open  ]],
Cell[CellGroupData[{
Cell[9845, 327, 245, 8, 70, "Input",
 CellID->1935708990],
Cell[10093, 337, 205, 7, 36, "Output",
 CellID->1911920265]
}, Open  ]],
Cell[10313, 347, 126, 3, 70, "ExampleDelimiter",
 CellID->1866821203],
Cell[10442, 352, 178, 7, 70, "ExampleText",
 CellID->927876151],
Cell[CellGroupData[{
Cell[10645, 363, 369, 13, 70, "Input",
 CellID->71816797],
Cell[11017, 378, 346, 12, 36, "Output",
 CellID->846083794]
}, Open  ]],
Cell[11378, 393, 125, 3, 70, "ExampleDelimiter",
 CellID->668446203],
Cell[11506, 398, 201, 7, 70, "ExampleText",
 CellID->1088376364],
Cell[CellGroupData[{
Cell[11732, 409, 440, 15, 70, "Input",
 CellID->1985789751],
Cell[12175, 426, 281, 10, 36, "Output",
 CellID->1641098971]
}, Open  ]],
Cell[12471, 439, 125, 3, 70, "ExampleDelimiter",
 CellID->638919951],
Cell[12599, 444, 229, 7, 70, "ExampleText",
 CellID->844953649],
Cell[CellGroupData[{
Cell[12853, 455, 425, 15, 70, "Input",
 CellID->890427757],
Cell[13281, 472, 490, 18, 39, "Output",
 CellID->1817927176]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[13820, 496, 244, 7, 70, "ExampleSection",
 CellID->1020263627],
Cell[14067, 505, 93, 1, 70, "ExampleText",
 CellID->930701266],
Cell[CellGroupData[{
Cell[14185, 510, 200, 6, 70, "Input",
 CellID->25292914],
Cell[14388, 518, 286, 10, 36, "Output",
 CellID->1652201163]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[14723, 534, 223, 7, 70, "ExampleSection",
 CellID->2061341341],
Cell[CellGroupData[{
Cell[14971, 545, 231, 7, 70, "ExampleSubsection",
 CellID->1295379749],
Cell[15205, 554, 612, 24, 70, "ExampleText",
 CellID->854615781],
Cell[CellGroupData[{
Cell[15842, 582, 339, 10, 70, "Input",
 CellID->1889701305],
Cell[16184, 594, 311, 11, 36, "Output",
 CellID->1581390831]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[16556, 612, 227, 7, 70, "ExampleSection",
 CellID->258228157],
Cell[16786, 621, 188, 8, 70, "ExampleText",
 CellID->2054899579],
Cell[CellGroupData[{
Cell[16999, 633, 223, 7, 70, "Input",
 CellID->20031],
Cell[17225, 642, 281, 10, 36, "Output",
 CellID->1178081345]
}, Open  ]],
Cell[CellGroupData[{
Cell[17543, 657, 562, 18, 70, "Input",
 CellID->23760],
Cell[18108, 677, 2827, 50, 70, "Output",
 Evaluatable->False,
 CellID->873888761]
}, Open  ]],
Cell[20950, 730, 125, 3, 70, "ExampleDelimiter",
 CellID->544251230],
Cell[21078, 735, 403, 15, 70, "ExampleText",
 CellID->1194050958],
Cell[CellGroupData[{
Cell[21506, 754, 317, 11, 70, "Input",
 CellID->2050735036],
Cell[21826, 767, 206, 7, 36, "Output",
 CellID->803215447]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[22081, 780, 238, 7, 70, "ExampleSection",
 CellID->2123667759],
Cell[22322, 789, 415, 14, 70, "ExampleText",
 CellID->182138168],
Cell[CellGroupData[{
Cell[22762, 807, 460, 16, 70, "Input",
 CellID->431021432],
Cell[23225, 825, 484, 16, 39, "Output",
 CellID->559812687]
}, Open  ]],
Cell[23724, 844, 238, 10, 70, "ExampleText",
 CellID->502094902],
Cell[CellGroupData[{
Cell[23987, 858, 289, 8, 70, "Input",
 CellID->228651492],
Cell[24279, 868, 383, 13, 36, "Output",
 CellID->645972902]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[24723, 888, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[25042, 899, 710, 26, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[25789, 930, 318, 9, 70, "TutorialsSection",
 CellID->250839057],
Cell[26110, 941, 159, 3, 70, "Tutorials",
 CellID->341631938]
}, Open  ]],
Cell[CellGroupData[{
Cell[26306, 949, 319, 9, 70, "MoreAboutSection",
 CellID->277758029],
Cell[26628, 960, 145, 3, 70, "MoreAbout",
 CellID->337947800],
Cell[26776, 965, 156, 3, 70, "MoreAbout",
 CellID->475253647],
Cell[26935, 970, 146, 3, 70, "MoreAbout",
 CellID->53039487],
Cell[27084, 975, 137, 3, 70, "MoreAbout",
 CellID->130969135],
Cell[27224, 980, 147, 3, 70, "MoreAbout",
 CellID->471850283],
Cell[27374, 985, 198, 4, 70, "MoreAbout",
 CellID->378877077]
}, Open  ]],
Cell[27587, 992, 27, 0, 70, "History"],
Cell[27617, 994, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

