(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    834299,      15341]
NotebookOptionsPosition[    822605,      14942]
NotebookOutlinePosition[    824320,      14985]
CellTagsIndexPosition[    824232,      14980]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Numerical Mathematics in Mathematica" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/NumericalMathematicsInMathematica"], 
          "Numerical Optimization" :> 
          Documentation`HelpLookup["paclet:tutorial/NumericalOptimization"], 
          "Constrained Optimization" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/ConstrainedOptimizationOverview"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"Numerical Mathematics in Mathematica\"\>", 
       2->"\<\"Numerical Optimization\"\>", 
       3->"\<\"Constrained Optimization\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"NMinimize" :> Documentation`HelpLookup["paclet:ref/NMinimize"],
           "Maximize" :> Documentation`HelpLookup["paclet:ref/Maximize"], 
          "FindMaximum" :> Documentation`HelpLookup["paclet:ref/FindMaximum"],
           "LinearProgramming" :> 
          Documentation`HelpLookup["paclet:ref/LinearProgramming"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"NMinimize\"\>", 
       2->"\<\"Maximize\"\>", 3->"\<\"FindMaximum\"\>", 
       4->"\<\"LinearProgramming\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Optimization" :> 
          Documentation`HelpLookup["paclet:guide/Optimization"], 
          "Polynomial Systems" :> 
          Documentation`HelpLookup["paclet:guide/PolynomialSystems"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"Optimization\"\>", 
       2->"\<\"Polynomial Systems\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["NMaximize", "ObjectName"], Cell[BoxData[
     TooltipBox[
      TogglerBox[Dynamic[
        FrontEnd`CurrentValue[
         FrontEnd`EvaluationNotebook[], {
         TaggingRules, "ModificationHighlight"}]], {True->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVVly2zAMDQhxEylSsizLaxJn73KafvQAuUDv/1eApJuosceeTDAWF4B8
eFgk71///HoVV1dXK3p+0wNfIzdCLL4I6p1sYBYuPy3SI2AmRBpuaE/TPRsE
3Isg8mrB9l6ILQhhLkbd93MBuwVxAthdZwPr5wVVzBc8wAvMX8TFqPkqE8q6
rDCsKPjZ/g1guz+PKvLFfGsDU1R4A8zDhgLangX9x/WGM7A1qSQ7rraYkaLf
LHZ5FWY9H7rew/n4/6vWPKT2mXFBaH/NGejzqi/VnFEFJ2Rb1Lo+jtofllPD
RVJjdZLr51Eb/ED1K+QI109LcIg+sdSIWqa5g85L6MiNoRloTws+pdBKiFGl
mx3N0jcAlcsKeFivVkteVOgMSNsC18sYS/iIMdZoK43WSdQVK1SLlYy09bZ2
aEO+aoPiACNak0EBbtfLRIz9KJRcroaSKwE11BprpRHqiJ0BrxtJG4pLYqWc
RZO4BQwSbGT4rFivaLjjQdscjSvlih48Mu+WDisOBcB5ehgeDEZQlikkKh3b
KA1FMTDg9yFxLSe4XC37begHTDh5seS1I/gaW025Yo/yUFbmSalShzon1Kch
JTNn2hKS1hyWVIkf5n2LHEQgt0zeUNTkMaYrDdkMZdo3fJCDHx7hxzAC+8rs
XYBokTPfEJk6JVlZx3Yl0UnTYMPvSUt6zghf8dhIi5VDnwO+Hcb1MHK1msK+
oy5qkYS7wVC3cfiq8ojBcMtFMJXnDiIcLPFJClDFGkJBhXGEZV6VcCQFoey7
tyvBF6lw0uKH6nyQ5fqhrKx/Kxe8+3coucpGN4U5+RYux7vnhGO5P7hKenqz
xSO3ivjisf345fg5Pt1ScCRkkzR1U1Dur+NCec4LZ49Ynx9P04GmOvlnqcKZ
79BfJfcjkg==
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlLlKQ0EYRi/iA9gpWIgvIXbjM1iKWAjW2vgeZnWN+AgWZhEULCxELFyi
ZnfJavbeJOr/33xXcj8CNnY5EA7k3GWYmTuzq+uLa2OWZS38/MYt0DbDHH8Q
8uTB64bRNrmsQkaPRoQSGX1ZqZPR9/eEazL6klIlowf8wiUZ3ecV3snowYDw
SEafUmJk9HmlQh4cX4GM7vUId2T0VFIho28or2T0w5DQIKN3PoUaGX1FaZLR
0ymh6sz7hHLk9PMzoU5Gz6SFmvFsCbd9Xzj9qyc0nfe49xfuTyYUEzoQbmh8
+N8e37HTbVpk9MSzUCajbypFMvq0UiOjzyklMnqvKzRMJCwU+/vjzemTSpns
nh97feJmd0e4cvppTKiQ0bsdoW5ecorZDgpPtD6xqFA2fp+Qp/vtdUj1v797
Oqdwnb2Ov++fUVpk9PCJUCC7n4t5wz5xj9/+rj6ceXbPP86NbEb58/k56qP+
n/0bdGyYqQ==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}], False->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVdd22zAMDUBxSaJm5HgljrPTJN177/nWh35AfqD//1aAVGrrJLbbk+BI
FEmQFxcXlDQ9//3zHNfW1gZ0/6IbbsYOhahvCGrO9qHM/mnhmMMLoKv0RD6V
cPyZHTTcDI5NkYnQq4UQ0BdiAkKYZagl1OOw5fWtcgyvPp7AnefsGG0DBIeA
9RZVrNfcwBasb4nlbP9u4cbc/XD/zXcIgC2qYYazJTQiGSYHS0Hr8Yu5LV/h
ePvR5oMOKswAQ0M8R5OloFMWSNRE+IAVeALv38KX26wAV1uU5Ojv16PQy8q+
gMP+9ABW5C9KbkZi6qv19DGc/ID9b+CrZTjbbVagH3p91oIcL+GZ6JDNUet4
HpWUXxx5BaWZxRh1FQgH6JqoKcarF/2/dblez7IE0XmWGlFL/6ygchIqCmPo
CTSmDq9SaCUUhfI7K3pKlwJESZiAveHGxoA7ESYGpM2B62WMJXzEoojRRhpt
IlFHPKFyjGRBQ2fjBG0WttpMcYIFWhNAAXaHA0+M4yiUXK6UxJWAGmKNsdII
cYGVAadTSQPKS2KkEovGc8swk2ALhg8Tww1qdrjRNmSTtOUqHDhk3jktVpwK
QOLoZngwWICyTMFTqdhHMrQTDQPeazzXdgWXK+e4KV3AhH0US1Ergo8x16QV
R5QXZWWeJJW6qLNHPWq8mEFpS0hac1pSeX4YxjlyEhmFZfKGsqaIhd+Sks+Q
0i7lhZx8cwYPmx5wrMA+yaCwyMqnRCb2IiubsF9JTKRJMeX3JKd5VoS3OEyl
xShBFxLebXrDpsfVSlv2FZ2iHMn4NBg6bZy+ihxiZvjIFWAixyeIcLDNT1KC
qogha1Gh14NB6LXpSEpC2bm3y8O3FmHniF9U55INhnttz7pZuWDu79BqFZxJ
F2bhWzjo7Zx6HMvng6ukuztzvGJXa66NmF/+crzrHe1ScmTkk/SouqB8vq42
0jl0EnuF9/RsMR1Io4U/S5Wt+A79AaDyK/w=
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlUtrE1EYhgfxBwguFBTEvyAiFhfzHxRFvC0E17rQhYLiT7D3u/gTXNg2
3sBFFyIutE1qm17UZJKZZJK0KFjBtur7Td4jnhe7dTMPhAfyQObMnO9Mjl69
fvraniAI7v/+7A3IRri7/8VGWJwHVTH72qohZp+ZBnUx+yWjJWYfHwNvxOwX
jKaYfXAAzIrZ+/tARcw+NAhKYvaTRiJmP2gUxH+vLxKzl5cMMXtfL3gvZn84
Cdpi9hvGJzH7yxegJWa/YnTE7Mtl0HTPfZ/xyPWVZZCGvQ/Au65fSV9aNMLJ
CfDW9a0fIBWz/9wBHbcOf/74PX83W99j128ZNTH71BMQidmfFkAiZj9irIvZ
Fz+AWMx+yEjF7KeMVOw/32x/iuHoCHjt+s42aIfTU6DWna/Prm9vgVb4cc0I
h4fAgusZ62L2A0YsZj9h1MX+9bN9LHfP75zrZ42a2L9/3+wXjVTM7q8jm9NE
5q8wA+JwoB/8eX+eNxIx+00jErP3GA2xvy7uG88R+z2jI2Y/bLTE/v1l76WG
mxP/+r7ZN7+Bupj9mFERsz9/Bppi9myeEzH7QgnUxOy3jYaY/a7RFrNfNtpi
9nNGXcw+NgoaYnbOk2///PJ/bXXFcP37JojF7CPDIBazf/0CKmL2+TlQEbPf
MZpi9olxkIrZS0UQidn3G1Ux+3EjErOfMapi9mBX8p73vOc973n/P/0XnFJ2
Rg==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}]}, "\"\<Click me\>\""],
      "Toggle changes in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["NMaximize",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/NMaximize"], "[", 
       RowBox[{
        StyleBox["f", "TI"], ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["x", "TI"], ",", 
          StyleBox["y", "TI"], ",", 
          StyleBox["\[Ellipsis]", "TR"]}], "}"}]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]maximizes ",
     Cell[BoxData[
      StyleBox["f", "TI"]], "InlineFormula"],
     " numerically with respect to ",
     Cell[BoxData[
      StyleBox["x", "TI"]], "InlineFormula"],
     ", ",
     Cell[BoxData[
      StyleBox["y", "TI"]], "InlineFormula"],
     ", ",
     StyleBox["\[Ellipsis]", "TR"],
     ". "
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["NMaximize",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/NMaximize"], "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          StyleBox["f", "TI"], ",", 
          StyleBox["cons", "TI"]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["x", "TI"], ",", 
          StyleBox["y", "TI"], ",", 
          StyleBox["\[Ellipsis]", "TR"]}], "}"}]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]maximizes ",
     Cell[BoxData[
      StyleBox["f", "TI"]], "InlineFormula"],
     " numerically subject to the constraints ",
     Cell[BoxData[
      StyleBox["cons", "TI"]], "InlineFormula"],
     ". "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->12975]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NMaximize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NMaximize"]], "InlineFormula"],
 " returns a list of the form ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    SubscriptBox[
     StyleBox["f", "TI"], 
     StyleBox["max", "TI"]], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       StyleBox["x", "TI"], "->", 
       SubscriptBox[
        StyleBox["x", "TI"], 
        StyleBox["max", "TI"]]}], ",", 
      RowBox[{
       StyleBox["y", "TI"], "->", 
       SubscriptBox[
        StyleBox["y", "TI"], 
        StyleBox["max", "TI"]]}], ",", 
      StyleBox["\[Ellipsis]", "TR"]}], "}"}]}], "}"}]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->4813],

Cell[TextData[{
 Cell[BoxData[
  StyleBox["cons", "TI"]], "InlineFormula"],
 " can contain equations, inequalities or logical combinations of these. "
}], "Notes",
 CellID->1183],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NMaximize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NMaximize"]], "InlineFormula"],
 " always attempts to find a global maximum of ",
 Cell[BoxData[
  StyleBox["f", "TI"]], "InlineFormula"],
 " subject to the constraints given. "
}], "Notes",
 CellID->10391],

Cell["By default, all variables are assumed to be real. ", "Notes",
 CellID->23646],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   StyleBox["x", "TI"], "\[Element]", 
   ButtonBox["Integers",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Integers"]}]], "InlineFormula"],
 " can be used to specify that a variable can take on only integer values. "
}], "Notes",
 CellID->21374],

Cell[TextData[{
 "If ",
 Cell[BoxData[
  StyleBox["f", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["cons", "TI"]], "InlineFormula"],
 " are linear, ",
 Cell[BoxData[
  ButtonBox["NMaximize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NMaximize"]], "InlineFormula"],
 " can always find global maxima, over both real and integer values."
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->14292637],

Cell[TextData[{
 "Otherwise, ",
 Cell[BoxData[
  ButtonBox["NMaximize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NMaximize"]], "InlineFormula"],
 " may sometimes find only a local maximum. "
}], "Notes",
 CellID->29994],

Cell[TextData[{
 "If ",
 Cell[BoxData[
  ButtonBox["NMaximize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NMaximize"]], "InlineFormula"],
 " determines that the constraints cannot be satisfied, it returns ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    RowBox[{"-", 
     ButtonBox["Infinity",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Infinity"]}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       StyleBox["x", "TI"], "->", 
       ButtonBox["Indeterminate",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Indeterminate"]}], ",", 
      StyleBox["\[Ellipsis]", "TR"]}], "}"}]}], "}"}]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->3559],

Cell["The following options can be given: ", "Notes",
 CellID->19127],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["AccuracyGoal",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/AccuracyGoal"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "number of digits of final accuracy sought ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["EvaluationMonitor",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/EvaluationMonitor"], 
    ButtonBox["None",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/None"], Cell[TextData[{
     "expression to evaluate whenever ",
     Cell[BoxData[
      StyleBox["f", "TI"]], "InlineFormula"],
     " is evaluated "
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["MaxIterations",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/MaxIterations"], "100", Cell[
    "maximum number of iterations to use ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["Method",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Method"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "method to use ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["PrecisionGoal",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/PrecisionGoal"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "number of digits of final precision sought ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["StepMonitor",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/StepMonitor"], 
    ButtonBox["None",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/None"], Cell[
    "expression to evaluate whenever a step is taken ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["WorkingPrecision",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/WorkingPrecision"], 
    ButtonBox["MachinePrecision",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/MachinePrecision"], Cell[
    "the precision used in internal computations ", "TableText"]}
  },
  GridBoxAlignment->{
   "Columns" -> {Left, Left, {Left}}, "ColumnsIndexed" -> {}, 
    "Rows" -> {{Baseline}}, "RowsIndexed" -> {}}]], "3ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, None, None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{True, True, True, True, True, True, True, True}}}},
 CellID->11773],

Cell[TextData[{
 "The default settings for ",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " are ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["WorkingPrecision",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/WorkingPrecision"], "/", "2"}]], 
  "InlineFormula"],
 ". "
}], "Notes",
 CellID->28601],

Cell[TextData[{
 "The settings for ",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " specify the number of digits to seek in both the value of the position of \
the maximum, and the value of the function at the maximum. "
}], "Notes",
 CellID->32207],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NMaximize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NMaximize"]], "InlineFormula"],
 " continues until either of the goals specified by ",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 " or ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " is achieved. "
}], "Notes",
 CellID->22495],

Cell[TextData[{
 "Possible settings for the ",
 Cell[BoxData[
  ButtonBox["Method",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Method"]], "InlineFormula"],
 " option include ",
 Cell[BoxData["\"\<NelderMead\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<DifferentialEvolution\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<SimulatedAnnealing\>\""], "InlineFormula"],
 " and ",
 Cell[BoxData["\"\<RandomSearch\>\""], "InlineFormula"],
 ". "
}], "Notes",
 CellID->552]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->6487599],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->60213890],

Cell["Find the global maximum of an unconstrained problem:", "ExampleText",
 CellID->1917251726],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMaximize", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"-", 
     RowBox[{"x", "^", "4"}]}], "-", 
    RowBox[{"3", 
     RowBox[{"x", "^", "2"}]}], "+", "x"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1746774899],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.08258881886826769`", ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", "0.16374000103603134`"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{179, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->22990837]
}, Open  ]],

Cell["Extract the maximizing argument:", "ExampleText",
 CellID->1664023518],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"x", "/.", 
  RowBox[{"Last", "[", 
   RowBox[{"NMaximize", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"-", 
       RowBox[{"x", "^", "4"}]}], "-", 
      RowBox[{"3", 
       RowBox[{"x", "^", "2"}]}], "+", "x"}], ",", "x"}], "]"}], 
   "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1557760567],

Cell[BoxData["0.16374000103603134`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->39490870]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1112598013],

Cell["Find the global maximize of problems with constraints:", "ExampleText",
 CellID->1719552892],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMaximize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"x", "^", "2"}], "-", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{"y", "-", "1"}], ")"}], "^", "2"}]}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "^", "2"}], "+", 
       RowBox[{"y", "^", "2"}]}], "\[LessEqual]", "4"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->886900649],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"3.5000000037251753`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "1.9364917355778475`"}], ",", 
     RowBox[{"y", "\[Rule]", "0.4999997617639282`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{192, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->37726438]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMaximize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"E", "^", 
       RowBox[{"(", 
        RowBox[{"Sin", "[", 
         RowBox[{"50", " ", "x"}], "]"}], ")"}]}], "+", 
      RowBox[{"Sin", "[", 
       RowBox[{"60", " ", 
        RowBox[{"E", "^", "y"}]}], "]"}], "+", 
      RowBox[{"Sin", "[", 
       RowBox[{"70", " ", 
        RowBox[{"Sin", "[", "x", "]"}]}], "]"}], "+", 
      RowBox[{"Sin", "[", 
       RowBox[{"Sin", "[", 
        RowBox[{"80", " ", "y"}], "]"}], "]"}], "-", 
      RowBox[{"Sin", "[", 
       RowBox[{"10", " ", 
        RowBox[{"(", 
         RowBox[{"x", "+", "y"}], ")"}]}], "]"}], "+", 
      RowBox[{
       RowBox[{"1", "/", "4"}], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"x", "^", "2"}], "+", 
         RowBox[{"y", "^", "2"}]}], ")"}]}]}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "^", "2"}], "+", 
       RowBox[{"y", "^", "2"}]}], "\[LessEqual]", "1"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1759366686],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"4.9556706998312725`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "0.4623685540615059`"}]}], ",", 
     RowBox[{"y", "\[Rule]", "0.40380539618670336`"}]}], "}"}]}], 
  "}"}]], "Output",
 GeneratedCell->False,
 ImageSize->{270, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->47497397]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->26512],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Or",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Or"]], "InlineFormula"],
 " constraints can be specified:"
}], "ExampleText",
 CellID->1077996427],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NMaximize", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"x", "+", "y"}], ",", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"x", "^", "2"}], "+", 
         RowBox[{"y", "^", "2"}]}], "\[LessEqual]", "1"}], "||", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{"(", 
           RowBox[{"x", "+", "2"}], ")"}], "^", "2"}], "+", 
         RowBox[{
          RowBox[{"(", 
           RowBox[{"y", "+", "2"}], ")"}], "^", "2"}]}], "\[LessEqual]", 
        "1"}]}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "y"}], "}"}]}], "]"}], "//", "Quiet"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1881132418],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.4142135642356175`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "0.7071067819987362`"}], ",", 
     RowBox[{"y", "\[Rule]", "0.7071067822368814`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{262, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->503175292]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->145340294],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["NMaximize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NMaximize"]], "InlineFormula"],
 " for linear objective and constraints:"
}], "ExampleText",
 CellID->1627702079],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMaximize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"-", "x"}], "-", "y"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"3", "x"}], "+", 
        RowBox[{"2", "y"}]}], "\[GreaterEqual]", "7"}], "&&", 
      RowBox[{
       RowBox[{"x", "+", 
        RowBox[{"2", "y"}]}], "\[GreaterEqual]", "6"}], "&&", 
      RowBox[{"x", "\[GreaterEqual]", "0"}], "&&", 
      RowBox[{"y", "\[GreaterEqual]", "0"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->555314442],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "3.25`"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "0.5000000000000001`"}], ",", 
     RowBox[{"y", "\[Rule]", "2.75`"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{186, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->767200545]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->592370280],

Cell["Integer constraints can be imposed:", "ExampleText",
 CellID->1554484569],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMaximize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"-", "x"}], "-", "y"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"3", "x"}], "+", 
        RowBox[{"2", "y"}]}], "\[GreaterEqual]", "7"}], "&&", 
      RowBox[{
       RowBox[{"x", "+", 
        RowBox[{"2", "y"}]}], "\[GreaterEqual]", "6"}], "&&", 
      RowBox[{"x", "\[GreaterEqual]", "0"}], "&&", 
      RowBox[{"y", "\[GreaterEqual]", "0"}], "&&", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"x", ",", "y"}], "}"}], "\[Element]", "Integers"}]}]}], "}"}],
    ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1605455941],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "4.`"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "2"}], ",", 
     RowBox[{"y", "\[Rule]", "2"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{137, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->32758853]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMaximize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"-", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"x", "-", 
          RowBox[{"1", "/", "3"}]}], ")"}], "^", "2"}]}], "-", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{"y", "-", 
         RowBox[{"1", "/", "3"}]}], ")"}], "^", "2"}]}], ",", 
     RowBox[{"x", "\[Element]", "Integers"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1044023529],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "0.1111111111111111`"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "0"}], ",", 
     RowBox[{"y", "\[Rule]", "0.3333333333333333`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{228, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->147056107]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Options",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->453790855],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "AccuracyGoal & PrecisionGoal",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->61701773],

Cell[TextData[{
 "This enforces a convergence criteria ",
 Cell[BoxData[
  FormBox[
   RowBox[{"||", 
    RowBox[{
     SubscriptBox["x", "k"], "-", 
     SuperscriptBox["x", "*"]}], "||", " ", 
    RowBox[{"\[LessEqual]", " ", 
     RowBox[{"max", "(", 
      RowBox[{
       SuperscriptBox["10", 
        RowBox[{"-", "9"}]], ",", 
       RowBox[{
        SuperscriptBox["10", 
         RowBox[{"-", "8"}]], "||", 
        SubscriptBox["x", "k"], "||"}]}], ")"}]}]}], TraditionalForm]], 
  "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"\[EmptyDownTriangle]f", "(", 
     SubscriptBox["x", "k"], ")"}], "\[LessEqual]", 
    SuperscriptBox["10", 
     RowBox[{"-", "9"}]]}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->764461149],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMaximize", "[", 
  RowBox[{
   RowBox[{"-", 
    RowBox[{"Sin", "[", 
     RowBox[{"x", "/", "2"}], "]"}]}], ",", "x", ",", 
   RowBox[{"AccuracyGoal", "\[Rule]", "9"}], ",", 
   RowBox[{"PrecisionGoal", "\[Rule]", "8"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1625295724],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.`", ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{"-", "3.141592653589793`"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{137, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->33024731]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1055000278],

Cell[TextData[{
 "This enforces a convergence criteria ",
 Cell[BoxData[
  FormBox[
   RowBox[{"||", 
    RowBox[{
     SubscriptBox["x", "k"], "-", 
     SuperscriptBox["x", "*"]}], "||", " ", 
    RowBox[{"\[LessEqual]", " ", 
     RowBox[{"max", "(", 
      RowBox[{
       SuperscriptBox["10", 
        RowBox[{"-", "20"}]], ",", 
       RowBox[{
        SuperscriptBox["10", 
         RowBox[{"-", "18"}]], "||", 
        SubscriptBox["x", "k"], "||"}]}], ")"}]}]}], TraditionalForm]], 
  "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"\[EmptyDownTriangle]f", "(", 
     SubscriptBox["x", "k"], ")"}], "\[LessEqual]", 
    SuperscriptBox["10", 
     RowBox[{"-", "20"}]]}], TraditionalForm]], "InlineMath"],
 ", which is not achievable with the default machine-precision computation:"
}], "ExampleText",
 CellID->1329553122],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMaximize", "[", 
  RowBox[{
   RowBox[{"-", 
    RowBox[{"Sin", "[", 
     RowBox[{"x", "/", "2"}], "]"}]}], ",", "x", ",", 
   RowBox[{"AccuracyGoal", "\[Rule]", "20"}], ",", 
   RowBox[{"PrecisionGoal", "\[Rule]", "18"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1468387723],

Cell[BoxData[
 RowBox[{
  RowBox[{"NMaximize", "::", "\<\"cvmit\"\>"}], ":", 
  " ", "\<\"Failed to converge to the requested accuracy or precision within \
\\!\\(100\\) iterations.\"\>"}]], "Message", "MSG",
 CellID->501329],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.`", ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{"-", "3.1415926607337976`"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{137, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->28621414]
}, Open  ]],

Cell[TextData[{
 "Setting a high ",
 Cell[BoxData[
  ButtonBox["WorkingPrecision",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/WorkingPrecision"]], "InlineFormula"],
 " makes the process convergent:"
}], "ExampleText",
 CellID->1199314865],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMaximize", "[", 
  RowBox[{
   RowBox[{"-", 
    RowBox[{"Sin", "[", 
     RowBox[{"x", "/", "2"}], "]"}]}], ",", "x", ",", 
   RowBox[{"AccuracyGoal", "\[Rule]", "20"}], ",", 
   RowBox[{"PrecisionGoal", "\[Rule]", "18"}], ",", 
   RowBox[{"WorkingPrecision", "\[Rule]", "40"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->39134653],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.`40.", ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{
     "-", "3.1415926535897932384626433832795034721592982030009607408185`40."}]\
}], "}"}]}], "}"}]], "Output",
 ImageSize->{350, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->320301033]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "EvaluationMonitor",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->405919056],

Cell["\<\
Record all the points evaluated during the solution process of a function \
with a ring of minima:\
\>", "ExampleText",
 CellID->1914018609],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"f", "[", 
    RowBox[{"x_", ",", "y_"}], "]"}], ":=", " ", 
   RowBox[{"-", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"x", "^", "2"}], "+", 
       RowBox[{"y", "^", "2"}], "-", "16"}], ")"}], "^", "2"}]}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"sol", ",", " ", "pts"}], "}"}], "=", 
   RowBox[{"Reap", "[", "\[IndentingNewLine]", 
    RowBox[{"NMaximize", "[", 
     RowBox[{
      RowBox[{"f", "[", 
       RowBox[{"x", ",", "y"}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"x", ",", 
          RowBox[{"-", "5"}], ",", "5"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"y", ",", 
          RowBox[{"-", "5"}], ",", "5"}], "}"}]}], "}"}], ",", "  ", 
      RowBox[{"Method", "->", "\"\<DifferentialEvolution\>\""}], ",", 
      RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
       RowBox[{"Sow", "[", 
        RowBox[{"{", 
         RowBox[{"x", ",", "y"}], "}"}], "]"}]}]}], "]"}], "]"}]}], 
  ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->157454682],

Cell["\<\
Plot all the visited points that are close in objective function value to the \
final solution.\
\>", "ExampleText",
 CellID->738842830],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ContourPlot", "[", 
  RowBox[{
   RowBox[{"f", "[", 
    RowBox[{"x", ",", "y"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "5"}], ",", "5"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "5"}], ",", "5"}], "}"}], ",", 
   RowBox[{"ColorFunction", "\[Rule]", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"Hue", "[", 
       RowBox[{".6", 
        RowBox[{"(", 
         RowBox[{"1", "-", 
          RowBox[{"0.1", "*", 
           RowBox[{"(", 
            RowBox[{"#1", "+", "3"}], ")"}]}]}], ")"}]}], "]"}], "&"}], 
     ")"}]}], ",", "\[IndentingNewLine]", 
   RowBox[{"Epilog", "->", 
    RowBox[{"Map", "[", 
     RowBox[{"Point", ",", 
      RowBox[{"Cases", "[", 
       RowBox[{
        RowBox[{"First", "[", "pts", "]"}], ",", 
        RowBox[{"x_", "/;", 
         RowBox[{
          RowBox[{"Abs", "[", 
           RowBox[{
            RowBox[{"f", "@@", "x"}], "-", 
            RowBox[{"First", "[", "sol", "]"}]}], "]"}], "\[LessEqual]", 
          ".05"}]}]}], "]"}]}], "]"}]}], ",", " ", 
   RowBox[{"Contours", "\[Rule]", 
    RowBox[{
     RowBox[{"-", "4"}], 
     RowBox[{
      RowBox[{"Range", "[", 
       RowBox[{"0", ",", "10"}], "]"}], "^", "2"}]}]}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->629793524],

Cell[BoxData[
 GraphicsBox[GraphicsComplexBox[CompressedData["
1:eJyc3VvIptm2H/TvUPWdz1WgsEBQSyNe6U3dWFIvCJI2gtAgXvSVYQueyqjI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   "], {{
     {Hue[0.42], EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["

1:eJwtlEtslVUUhbdArQqUSNFIrqltpd5Ttb3+18DFmJAa77VxZGIcaKSJQdA4
aZ2oUYwPfOOj2gLaCKJtUCES0dBqIFQRjchDRawD4wydGBN5DNSBxG+5zmDn
nH//++y99tqPjlVDtw7OiohhZDZy0bUR118V0bYg4rL+iIHeiDnn8829HTnb
GfFiitiBbEe/lv+tjYjj7RFXVyN6kBUtEbfUIm6/IaJrfsTvyFJ0c/E/xrsR
ZBDdq5xfIEu4D+JnEX4W4XNdX8Q0GHaBYU53RCdyiLdD/D+NXeqI2My7p5FL
0N9bRKwu7OefHmIRe+w8bCoRNd52nONzQ+F7mTyuRJouj3gH3TjxRksRn9cj
mvG/jbdf4ntfsn0F7BNg4Ro/YnccKXNfQq5dVee2CtncGjGD/jf8nUCOYPfr
vIhN6PehHwH3a0gXjn5oN2eH0d+3PGI+OKawbQbXH7w9xtuz5PIv8iZ4XgDL
t8n8yPfeuv2PlIxdPsfIdwvy8gURF8PVwobjVJaCO3P1P8YW4zwGJ3fV8jd5
3Hyd7+J1a7Y/CR9H6QmgxSZ0ryCTfJTJo7vDPTDTaV7Eieqtui3mPgGWb8Aw
it1K/A9UzZPyO5lzfB5/R5J7bC1x7oGHn/WW/+PcN2CzkbMHOUHcz7B9KTnH
vZzPJvPzEPYD2Kwp+Y14UY9N0kflPtfvIFjGK67De8mcis87ifsAdrctcP56
J36O8n9Xcs8/Ag/L8N+G/bZk3MJ8Bv39DXPxPbp3kQNw/h3nR8k99mnyzIgr
YT+Y8b/N+Vxyv+0AwxX4/5C4Z8B7oG5O1YPTuQ9Vh5nkWpSpVaVwf5Xg9dKq
+3QWWIap9zQ2NWo7t+r5u5F7b7ZvKdxzU7k3lbf6U3OrWqqOin+6zxi6+/N8
gn+Y+F8l1+IxzsPJsU5hu7/u2TiE7onkmb0DfWO546gv2nJvBDhnN8zVKLbP
JM+s6nd3yT22kzweJ+5NF3pu1EfiXzEfzXG1G94qHEuz29RvPtfxfyPyE/op
zveTZ0G+V5esX8m7B/v8PQ9ul9EDi8G3tfAsqUZ76t4pwvYn/k/1uo6v4++N
5LlbT35rCve+OFZtxPMvybOnOVJNrimM+WH0T+Y67ud8Krnntdc0V8Im3Z6s
X8+5Mzn33QuNUbv062Sulfu5Fc+P7Hcnv5G99ndTw72qXpvI/aa5+TjPjrB8
kvFo3zRXrFdNtuS6aJcrT+WoXbWi5v75IJkL8fAXur+RgVbvYtmq1tor2jWa
reaGa6+6ayblQ3t1qOZ9ql2q3hJ/6kn9n8w2+qf8lfv25BqLq/8A+o4PrQ==

         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlUts1GUUxa9oKcpYIqCRYLBVx95q2mHa1DExIZPYcXBlgm6MNjEI4qo1
MWAEXzFx5ZO2EJFCX75AUURABa3iC6r4hLpQd+jGjTw2xgXxdzyzuMn3/e/9
zj33+W9bPbhqYE5EDCIXIccyYgYZL0Y8f0nE6WrE79yvRPfT0ojpGyMu74lY
tiBiL7KzHPHcStt/g80Utq8iJ7B9bV5EP/r1YNzPfS0yU4vY2MX7iyOG2yIm
0U+gH0I3gvxRh0eHsU5dG3EU+6mSbTcjZ3k7wLd2zoms4f0dlYjipRHXITuw
XQ+fDQ3+4+inqsYeQH8cbt8h56/BDxymOX+Wtm+7IKKE/d3Yb10El0LEQnxt
wueJ1oifkfsqxhTWbEvEOnydRo9prALjGe7ryuYyUnTs93Jfs9Q5aCW+q+v2
LQ6K5RzvO9oc08tw2Z7ORxE+t9+Mn277GoVTjfs95P+uBeZ4A1gdyGLwmsjX
Tt6ONup1HIyHsR2sOVe/8G0I3WZkNhzDbs5vpX3vAmMv53clYX7vc340Hd8M
+GuJ5emqa6+cv4fuCWQ67P8f8rO7x+/74deJ/a0V5/JUwbn+NF0b5XyW8291
Y8tHE1ybkS9anaPH0P+YvsvHk5wPpmvxLffF2F5R8105UC+OlRu1B/8F+UJI
ZRzAwfXoltUdmzh+hO7DdOyq31z0BfgvIb79CyNeQbc1/Vb1O9PlmstWbz5I
Y+gu/gXeP1J1r+tNsduYwuqlhjeV3ePqbc3AlrQPYas+ndj/u5y8wG8F9vO5
D1b89i/yvYhY9/U4dvXkVegPMJNT8C1h85S4pvE1QxdiP6fm3CmH/d3uKfXS
6hZj7emNuO0yY06WPHOatSHNJ7Hc0uFeHIbjIbA/Ru4k3l+xb8b+wZW+b2fe
N6F7Kd3/S9B/z/mHdG+oPs9iu6XsXhjjzXLO7XXrxVG9uCudC/XkO5zfTvsX
p5Oc/6y6t9XjO9I9r/u2ed49k+rn8A5qAru57t2lffQVuq/Tvv/nwHkirRee
evF8p2NRTz5ELv7ucq01o/vSMyGdbHqpTaXi2sj/4+heTGN/At5ZuJ6pmstJ
ZCO6kbT9fPQjZedkrLF/tAsnys61dsADZe+UkUa+1RvnqsZSj3yu2UjrlXPF
Mp6OVTG9yfmNdO41I/I9nOYqDu1gteDjYME9tKHqnaH8a2eptnNL9q0avy7s
9Gyqn9QLh9O5kF69cqzm3lHNjvQx833moxysqHhGNBvyqVp8mY5dOdC/52i6
dzW/iu1I2r9iPAzWoT7zV833pHtEvaF50u4fLRlP+0WzsD+t00xot25L7z7l
9z/FeSvA
         "],
         VertexColors->None]}]}, 
     {Hue[0.07253898759132957], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFllls1FUUxg8t04KU1k5RwFbaUrqMDVP+M5QpArMIkyKihiUxBpAlYhvB
thFBBUxcWAJoXCIJiSi0JCpFMRGJERFRSxSNohV4UCEYgSdDWYIxPqi/k2+M
Dzdz596zf98591+9rHNuR56ZnWbls96Omy2OmoWGmn08zWx/o9mOErN3m83e
j2v/GvtRGbNKZLLlZm0jzHaHzCZxPjdiNnmQWVeC+7TZ5bDZAc5DE83OonuF
s6ust4aY/dLAPft+bHRNwh+6V9C9xNlA7rwTnxenmC1Ht4P/p8rMDuLrQFi2
9xaZ5WF7A3Gu5Pxe/D4YmJ3g7gX8vsn5S+h2Zs1WYauNfcUtZtuLzV68wexQ
RL7d75wYvtFfwF1N0uzhFrMPy6S3Ji3d6dyPx/5R7OejNxX9OnQfQ2YTvtqR
qSqXnp93o9eDfOdw1eBMo+pwaAb5sp6oMivl3lq1j2N//+1mveRVwFkZd+fG
ml0ntlnE00uOp9jfyX4v+13Y3816GZ9HsfcZax12ninW2SP4vXCj2bwKs8Pg
tYA4sy3C62ZqMoJ1EvkBMO9i311qdvd07omhmxiOYO8T1pPIzCHX+zKy5TjM
rxDWrdRqTotw+S6sOo3Bbx2ro0QySXTno3sPOtfI8TrrJ+z3YfsYaz32Gyaj
M1l8up+cmgLxqx4ca8h5Ijmt5nxRIPtfEe8U7O7gfFzKrJY1Fmy3UuP1cfl+
FvnlgbhTgPxK8jyGr1/D0nNcimLiq/PJ8/cYe6nVwkD+3FcX9vIniodto6lr
ierpv9+Hlatj7/m7fPtMs8ImyQzGTktCXJ2L7jslsj+7XDj9znkErG9jFdSI
m2ews2y4uOKxepwx7DTmZGriwskx8rtzyAylkXcjsySh/0+T63PkVo2vfdRk
UVy4e3wfhcXPB8hvMesuYilCN5nQ3UHkK2Lim3NtUUx8ayOvK1HF/zn2GpHL
47ySuKpY/8DVgbD60/MtQqYgEJ/rA2HvuHeAxVXs7Buq2oQCyd8KhhUp9ajz
y/vY+eN9+VurejMJR46D+1p4MqNcvej96zg45o5FL37/TKj31+EnnNW82gMm
X7J/hd7/A5labPWXKGfvW6+J9251uXL3OTesRbp/TaD3Y6p5W863+z2aw8r3
nmttTPl6HxfHte8ONAscK4+7NKzenECus4hhE2fFyNSn5TuPGPNZfXB1IzV/
KFBeK8hlcO58NeevN2mOOZed+56X9+LIuPqxwecG83MK988j/0Ygv94f6YTq
5b31Y4P6y2etc915niGub6jzU8S2Dd1Xg/856Xk437z/etLyX0hcoaz6y2fA
vNHCt71F89H7oxAMQ62aaf1xxeexuY87ShW/16avQnmNQbYuEGceT4ujjpXP
wrFJYXQVG5vJZQj2TzcKV8f0a2L7NCNbdcjWJxVTAtnmhDh6E7JHGlUrx6Ev
Iyy8RzePMtvJO3WoTHPBc9kVqI5ew1W8j8OiesuKovrv+7ZAmDleJ3KzwTlc
i/9xSc0An/eHc/N/Kv5/II6RxHCBHE+Cxbc+u8kxFZH/pdheNk0x7Umrz8/l
eOe5Ovec1z1NquGglHre+93fwNlp9cK0iN4t53kx9gv5vwL7W9jPTGqub2vR
f99vZfZuY40n//PEPCnQvLqWVj94L3jtr8VU/w/w1ZTSrBiI6031ns1gO53U
7PTviLKsviX+Hm/2c6vqvD0QzzynNdi5hNzCkGx/EZf9an43surZ9/g3SbN4
uJb6H8+KQwcTknMZt32qQfb7I9JxeX9/tjSLV94fZxvEww3oTcd3ZUi2fC56
vo/GlZvn1f7f+0YNq+Ly5746+R0W0/zz99LfXfe7lFotSWveX+T3fFr4FjTp
nfA3wmfM+qjmjH9zvdes+lyOaub6vN3ZpJ73vvD3wXH3WqUS+vZyjg2OCCeP
OYGdOs6LvIZRzVznsHM22yyu+nubl9I88++deRnNv38BH1aPBQ==
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtllts1GUQxQekCFqKFFFhkV6gLc3aLdvSLhW6+6+0ViNBLSRqoHKJCIZL
jYmogMELoEAIaCTRiFAgRqkaEwkxgqICJqJBhYgPJBiN8mYslxh9UfwdDg+T
fP+db2bOzJyZbysW9nStGBgRW5BByEdNEfuRn4dHDEXxZDpiTmNEb1HEXKSq
IWICsq84og9ZmfB7NmLcsIjF2LxVH/EGMqI0Yut1Ed3YrkzbVj7mc38t9s+X
RMxIRXxbG/E28e4YEfHsjREvoiuZErF0ZETF0IhVzRHfcOfYWPtcRKxN6E+j
X4H9g20R9+Yivkc3mztZ9BM7I2qQSZwPtUf8RswzlRFPl9tWNrorH6cnRpwA
4/tg3zEkYk0Gm9aI1/kuI/5Y8KSQgWDvQ/bgcy/+fiFeDznvIdevO7hPPq+S
7yjObzbZ9kfifUn8C9x/B9+r+Z5F7FbyqR4Q8THxj6K/hH43tVyD/k+w9CNz
Slyz9ZzXIWWca8CUppZ/NRiLMD0KntXol6OrBs9x4u+tNxbV9CS6U8hU/N3M
/YRaFZAW4q/ApiYfsYyavAfeWaPxgb8+7m9DdwZMD4D3ZfSfSk/Njte6J+rF
Kvr1GfgbyffcDRHPgH8k8VdnnL9qeC1Y/q2zXj50/q/OtdZvlfgbhPxRbE5k
iD8dfF/gv53+luLvpg7XUj5nUKuarLEJYzffCxJzST7K6fu4TusvVxrb/bXG
rnjXU7sc/vfjvx+px1dNC9wll46UubkLf6+kjKca+xpkMudi6lfg7q0FeDHG
/N6j3mXNBXGg93a42u5eqqdLqV0VNe6itn3gv5i459KLE+vo5xPUex++lyEd
xDrSbq5IP7vNPmQ7kxzqwHdbg/kg/3lyGUb8cvB+UupcVEPVTjlNxf8o/B8o
cv+b+S5u9Owqn27uPpXYVjNcx/d2ZOEw92C35p18K8c4P3GhjXwWj3Y/D1Hb
iVfrdwFObcT/fWD6odSc7MZ+boNzU47r0VfnzTXti43kW9HpXqln8xL3VL28
J2VuD653LPkrAltjzvi1g+4m9ukG+94AhgJ4BsPP3XD/IPN1F7E6885f8375
Trib9S7QTljIrM9vNR9fuiXiMfIrQV+R8nzOJ9bJWu9Czfxm8L4LpiLw1BHj
AlzvzzieenSMfp26el94qsB3sNa1OQX+EmxzaddeOWj3PZw4V81rGedBWed2
vtS55Gvde8XYgO1MMH2HbslwYz/c7tlTDofRv5B4N17pOfGmpd17fYsrylG5
iTMJ+NpavHu1M6Zzd3yjuSWOfQi+5zLexZrnz9s8A+K+dnIX2DZf3cfaKaqV
eqLcVYNWzvkW74rlcOoQ/he0utbauRfx3dNhLmgH1cO1SQXbq6dVBWMUNnFQ
u/wcciLl/olrm9LeZeLcWvx90GSswrwz6x2jWVb8R/henPWu1Jsxj/tbmrxL
1NMFOXNeXP9V80SsA2nvW3HiKLU50ujdOwSbndztxuYsd7djkyPWjnrXR99j
C75z5TzGb+OurLFohw5EP6Dgt1Ez9BW9LM549rUjOqjtQ4l3k+78Q6y/c8Z6
Zf9x/iltbgmD3rLzGXNR9RVXNHOqrzjzWtYYhUczfs1kMEz2fC7iTjP6VRlz
UTP4e+KZUa1V80pmaXze/FFNmvFf3GCuaqf2Ju658tMO1X+Fnpx3rTg9IW8O
CJt24rYmv9l6q7dyPpt2D1R7/QeZBtYlU8x1vZmPw5VLGXNF86C3a/kUz7p+
S/LeQdoV2lH/A5C1WF0=
         "],
         VertexColors->None], 
        PolygonBox[{{17665, 11883, 8388, 17667, 17664}, {12563, 16621, 16620, 
         18183, 8928}, {11784, 18182, 8928, 18181, 11783}, {11870, 17670, 
         8389, 17669, 11869}, {16621, 16622, 7122, 16624, 16620}, {13616, 
         16614, 7117, 10452, 13615}, {18216, 12594, 8966, 18218, 18215}, {
         18209, 11870, 11869, 18207, 8959}, {17596, 11770, 11769, 17594, 
         8312}, {11884, 17665, 17664, 17671, 8389}, {17602, 11784, 11783, 
         17600, 8315}, {18219, 11887, 11886, 18217, 8966}, {11770, 14134, 
         4787, 14133, 11769}, {16623, 13616, 13615, 10460, 7122}, {17668, 
         13712, 13711, 17666, 8388}, {11887, 17685, 8404, 17684, 11886}},
         VertexColors->None]}]}, 
     {Hue[0.08415357395127578], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFl3twVdUVxndiEpMaEkLAShIMeZFcTrgh93LJDTH3npTE1Pgi1NZRikqc
sXSEUuKjA+IUNTAiLdUO9g+dQvDt1PrAkWoQH6gzbe1LRhlnip3RFlFbpJ2p
Tu1Y7O/ju53+seecs886a6/1rW89TtPYuuXfKQ4hDBWFcBrXniiEqnQIddUh
rMmG0J4L4bnaEHrjELKsqD6En/J88ewQHqkIoXgR37GOIH9Nta/vlYawHz2t
fPsk9+3IvcP+n7kvQXZu3s/j6KtIhHB1VQgfcZ9gtaB/+owQ7voSujHsD+yN
94TwD/aW9YXwInr3oSeuCeGXDSHsRG5Vv8/TWe8jvzDDWdNDOITuVwcs9yn7
65G766wQjnG/E51TlSHs6QrhzLzPO4b+I5Ht34zMGXy/GttGanye/OjA7u+x
NiBzSyqEhznrVmQmwGxL2u+WoXsbesaReRasHquwbbs4q5Gz7uSsTTN9lY+S
jXOWX4++h9CzAz1fYa+Cbw7w/RzWG8IXmbPZb8w5Bi9h8xt8/xb7n/Ddf3KW
S2L/AlYbeG7E7x9FtvkgMq+wHkSmotrXn7F/G3u3sm7k+fqhEK5jdTaBLXsb
+fYy9sfwZQk4HAW3Hdi+AR/u49vF+PvvBSH8dVoIZdhfzDczq+23/G+HWxsX
h/Ar8CzC3+PIX4rMKDKfgGEU2Zb3WkK4G/nH4eEUsm29cIozflFpDq4t/T92
4tJNM42LMNFVz9Lfiu5BdCbQmQCPfyZty0rwuJI1Aibb4fZyzjiM3tXcn8P9
Pu5HsKdO9hHTpewPZm3TD/HrSvQk8X9aj3ERJkPoLyFGj7E/zt5E5DxYjS2l
cPZm7G3k+YFa83Y7sp/zzSb2F3Dm8qzf/QCsd1ZbPqOcYH8v+1+v9lXxHeC7
29l/k+f9xHQT9vwJPn+GPeVZY/Ap9yc7QvhmreOjOAmTMt7PzJrDzXDwSWz9
CbqbyY3PwWMn8kXEfA06ZyDzVLXtls1tnHtt1rGcx/2WyLF4Cyy39Dg+DfDw
fGR2ITObve+C12Tw+dka2/BtdD/X7+dF4PwZtWApOP8YH26eG8Kjs8z1rWc5
X0bBIY2eLPw5Dr/mc/YR8roSPa8N+puD6HuFdZRvytk/yP7fkemI/Y3k5Z/8
FK+UK5P9hXxJmh/ixqrY/BA3lOvSJ6wW4UuKtZdY9sPHydi4ngF+H6J/ohx7
Us4B8T+AQXHCufIE5zyus9B1nO9qu/EPrGZlXQdVA1+vdoxlT1vGeagcXEZu
765xjX23xbkh+1Uf/9gMjpXmpfYv41qRdf4r9w8R/1J0Af2pvG/NOPcVJ+Gh
fNS3wkU6T4MPJazyOmo/em5LuaY9j+1T/ebYveiYNeT8v0ixZi07M4Qr8OvB
jOuYsFMdVU2QzwtrzCVxrS5vvum9rpLZkDR3xVtx//Q68/8ecuRfOXMnD+Yv
Dji3P46NozAUBsJCmItP0iGOfQN790WuaavQ/U7kWqH6If/lu3rJM6y/8e0x
8FgL1i/w/XmcOZLz8yPYvD3jOjzM3ldZa9i/v9ZXyaeIZ8ew6+G1SeeP/FUP
+bDPfeRpdOzNuIe8Pd19VZzZhm23s4rR1w9fvoaPFzaY/6ol4nwSzBawOsDl
WeUZq63W9eaW2HKlda4zqifii3SIMx9x/reI5TPIb8a2J7BhuMa98UTs/qjc
OjDosy5C99ZCXovXzSnbqZoovXPAqgZ/vzjX3Mthc9li47m2kTjChTeD8eju
NiayS/Vbtkk2n7C8ZgidrzgG9mbg+8l69/n1hV7fjm3zMsbskHiddk1Sn3l+
0HFX7Vb8FIvOGusQ/gl0DhKviSrXrUsGXLvUzxQ/ya/knF0p944T6P447Vxs
Be89KXPmavy4BD0tnFXe5X6s+CiHVDPki+rxpQOuyapbei889V7nq45djr4V
rFVVrpfV3eZiEWcV58171c55CddP8be+zxxWH3i5zz1AuS0sZbN6bDpyn5W9
4rpsfjTjuUT5q/nlg9gzjHzQVTPGYvZ6WPPrHaPyQt4pJq31jq/irblMfDgv
5b6onqh57XDsd8qh2fXOoxJ8aoz9vJvr91PO8c0p2yR7VJd1puL+W2R+x5qq
N3ayT7blel1nVWNVy4cy/u5V7ieXuEe8wP1V/cZ7DvrPZi2scj5Ij/BXnVE9
Ul2SLVfFtiedcu9R37kB3Q9EzvGI952x/RcvGvLGWXXo3i7XohuRXZl2vV7I
+y5WAuzu47m9zrPoOch8Oe3eqblyssu2RNx35i2nGXB3l2eZw5G5qLNGOP98
Vi82tIPnBYV71bmpQq1bP2R+i1vi5li3fe4As/0Jzx0t7DezRqlTa7Oe41X/
VL/XJV3Dy7i/Ium8mSrEUPqF61gBW+X6X2Lz4XKuK1jnIjvAWXGv5zDVeO0p
38uG3YMVr27OX9FtHql+jNX7/jp0XB/7eRo4/z722ZpTlcuayVvAb2nkWVu1
T/1Y2NbnPUOo3ufBLGbth4O/Tni+VP2/G1xb8v6H2AVfdhc4I92SlX71zHcL
fVPYiX/iTBP2b8sYk4e4VqL/N5Sol8HqpSHPaJpb10WeXcf7/V+hfwphrf8i
1V7V+KdnuX6qBi9JuA5rrr+h0FOUP6oByi/NgKN9ngPfX4rdna6pTXn7I1/U
M3cUeqvyWH4oX/Zh403Y0VrruaGj8G+i3jzKYhQ49c90YZ9nOXFO+pQXXxCj
kzn3kPmx/8HEtTl59131XM166g2y8+20Z2vN1Yp/W94c0Bwjvao/8vOawj9g
KX5c0Gc7VP8mCjXwKHsf9Lln3AkeJUPmURO2zM159ikFs9nU1eIG81SxUVz2
4OuJHsduq+Z0notq3UdPXYPnkdcT9vF/+Sn+3BH530y4qc5WFP59lBPRsPOi
GZmfL3H81GuFK9Q6lW8HCv9xqjevpV1z0ugr7fZ8+F/RO0QX
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtl32Q1WUVxw8LF1iEXWFXcl8Q9o3dy1122b1c72XXvfe3sSu1Rbn2wpSW
sc04TEkCMhapM2QbijiNNuYfOsJKVjQaFQYKhqHhTGVvkhAzYTNFgL1BL9Lk
RNDny/f+cWZ+z32e837O95zbNHb7jZ+piIjj0BRoeSZiPBvx2eqIDmgoF3Ft
ErGoIeLvV0bcmObMb2f4PlYT0ZuPaIHn+1dF7JkT0cb9NcWI99dFPMH957oi
Xh6IOHV1RDI7osD7KT0Rdcjby/tUj3/Tt/h3o/tm6EPofgS6Df6pwxGd8M6p
jHiFu8OQbF5SFXGxI+L+nO9vRt8Y8jZj7xruWmZF7OiOaCpFXInsh2ZgJ7Zu
gP9WZO+EnuY8iftp9dZ3C/p2cf9lvruQ+RT3D+b8rd/Wc3coa17JmIa+pXnb
Lh+ux/eNsrcp4kXsXUg83uqNqOTtdux7HHsa0FeQPOy5H94t0CTuKqDT/RG3
I/91/Ht7bsRIIeJ8mV/xvA55k6/FL/j3cT65IuIs/B/h+/ctEc3wzsfmXdc4
f7uxfTP89xKPFfDczf0DikEq4sDMiG3cPwBNRNm/vH2SL/+AnuO8D2pFfhv0
Xuz7DTIIe4xiX+DLS/j7IP7eM8O5CGy82OCcrIN3Jvr/Qi7+Cu3hfmfOtTAC
rSbeHyVf1/N+Ced7s/ZBtt8JdSJrrnzi+2vonyB+zeh8lPPD6Psmsr4B/Szs
4yjxWkEObqvz+zfwNbU0op1cnMTns+i7A31jDY6hYjmAjpWNPr8T/r9h86jq
D/5x+CvgP1HO9xbOG7FnHr6sRV6G+PcSr1PEsm+2e6ME/wtz3CN7+V7Ae9Rc
juG/+T6GjBPw3j3DsTiC/h83OiZ9adeYdO2tcS/k+G1PjXtCtZfB//Z61+BZ
zndmfPeq+hX7qwcjLuDfV+GpR39d3vldRXzWYcgB5N1aS1ywsZJ4DhOvZ9B9
kBjV4MvUFe71g/h0aXHEu7lfW2d//ss5Q/xakX+S82PYn0JeRaP9r4a/g/sT
3D0H/02cN3D+ZJVjdh/6JmP/9Hrrb4X39HLkz3fN9/D+oZxrQTHdg+0bwI+H
wY+vQAuKzqFydQM2LeJ8nvMzldbZznlh0bnZj//NRfeQemeU96+ib3AZscf/
CkBk94BrUPqOIr8R2+ZBd9U6Rwn8g9DrNe6Z//G9rdu5e414/45GGKEnbpjr
nqiH97Fu97ZqdAHn7d3Gnjc4Xyw6BvI9RQxWwvu+Mv8+5E9gz5MD7uejnL9E
fpugTSlj8o4+x0Sx+BaY+yPu/kV/Pc3919W/A8Zs5U8xW8X5wwO+F8ZcgjeI
x1iN43uF+mHA8vT+bWr9P0tdr8vJWQO2NfYbuyRjTPGCvsv3Ft6/C3+2Fox1
6rnJ8K7Dpj+k3DP7ebsJHfMqjXl6u63g2SEexa6j174rBj8hP0Xys4n4TyI/
Jc5rC85l+yTqBX1t0PM1tmEhbzugNbXGnz8h78/97h3ZoFmxNW9s1cx4R9YY
IWwQpmTonZezjp0wdpbwh9+eJbb11caexdjwVPn9HcPGJPEL46dw/nSXZ5Nq
SFivGah6k7yd6F5G/Wca3E+f4Hs1NMJ5J/a9kHZPqh7kwwi+LKFm0ppH+LOI
7y6oo3wW9h6H0tWutxS5ngrVcub6MvZv73NtaAYEvTybntqP7t82U3vY889e
x0L1sJLeaynjrd7PT4zZwmrNaM2W0V7XompStfF83rNANfJF7jZ3ebYIfz+W
9UzVLBUmqXb3lueHavg95CbV6dmhfP8Sfb9KPGukM833z6ED6G6BxnuNEcKG
1ch/C13rifl4lXu+mfsjHfZ9fLp74b7yPFW9TvC9KLEsxWAasqoKtlUzYzu2
nOr3rBUGPcL7Lt63NXg+CuvXFDxrhV+vcb4u41r4XrV7ZyY23bPAPTRS9A6g
2hZmvoTsQ/2OnzB8OOeeU6+pRzsT2yjbhKmjwoJ+94Nq7tvk8oqsfVNNFhJj
sLBXNbW45B1Ls0M18ia8Z/pd+/JJs+/ZnGetZuAHmA2fwp9dlZ5587F1XtFn
7Wzq3dPleEjGOWxbXvAsUf67S96hNHtUo4sTzwjNBsXs8KBrWLWqmdZZcg3r
rJn1R96fTmy/fF6IbfmMe+04v/1gyDWrWtXOcGjYO6HqSTtGRdozR7NGGNZC
vttz5lWMppS8kwgLNGPalnnmCq818w4i/8UhY1cBPLmAf98ZMParh6t4u7vP
+6xsGiI207s9G1Vzt2D7xxPPDvVwPnGPq7e1I/+6wzbJFtXkRJ97TL11Ga/R
NbXgXGgn3gh/T9G1pZ2klrurCp4VeqPeO5f1bFcNTfB+R2LsEEZ+QbO1x7uu
Zvg6YrE+511OM/+mxJgvrNeO1UZsWkvehTQDlfumomepauAc788m9kc5EjYf
ybrWVZOqnZaiZ6nev8nbM4l75fLOnTGGKnaKoWKZLe/HOv+Ct7N63evq8QvI
ulT0LFSOfkj9/DRt7NcMrigZ45VP7eivkLvDQ46lei7Je6eVfu1ccwadU+VS
O+hdxPvzA8YeYdBWYnM+7/8K2k9n8j1L+3O18eCD8BfTnjWaMS3of7Lbu6b+
Q7TC35bzbiqZ04lHZcG7gGbcKvhH064t7QzaxbRTapfUTnYM/48m3o20E13N
/ePletYO9X+ebgPF
         "],
         VertexColors->None], 
        PolygonBox[{{11175, 16458, 6969, 16460, 11174}, {16665, 14348, 14347, 
         16664, 7170}, {14348, 15804, 6366, 11900, 14347}, {13708, 11175, 
         11174, 13709, 4398}, {16459, 14324, 14323, 12583, 6969}, {11183, 
         13742, 4423, 13743, 11182}, {15802, 11183, 11182, 15803, 6366}, {
         14324, 14326, 4956, 14325, 14323}},
         VertexColors->None]}]}, 
     {Hue[0.1073827466711684], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmguYj2Xexx9mRkMYxnnGYRA5zMHMGIN55yBmU6TVudS+Hfbaklap3q3N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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtmHuUz2Uexx/MuEzmIrcxM5ihdcwMM35zZxrz05iDSCVFabeLTpJy77Js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         "],
         VertexColors->None], 
        PolygonBox[{{11902, 12930, 12929, 13737, 4420}, {12930, 11901, 4419, 
         13735, 12929}, {11854, 13223, 13222, 13681, 4378}, {13223, 11855, 
         4380, 13685, 13222}},
         VertexColors->None]}]}, 
     {Hue[0.14222650575100718`], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFm3d8lUXWxx+qEFKAkAASIRESEiC53FxCmgkJfV8RFNfuiqhrwRUL4qvi
uiJWsO3rrit2egfpKFIUFV1RQRAFRECaCIptUcDV9/flN3z8Yz7P3HnmmTlz
5vRzbs4VNw4eXjeKonMaRFE9PeOlUTSgOIoeSo6iQ7lRdFBtfFYU3annHWq3
a84zyf5Nf1NH9xclRdGQhp7zgxZa2y+KNtRE0TLN/XebKHpXrbHmz9Fzrtoz
9aPotFT/ps8ao8I6q9VWqdUIsP3t9TvF/av1/hq17nrXpTqKCtUuaB1FFzaL
orsE4/A6UXS4s9bs5f3Wd/Q3zG/fKIr+rP4Q7dUw3TABz3+TPWds5OefQz87
P4oq1P9K84/Ud3+8np/Ho2ipzjVPMB1Qf2WNzxPT/PmaM1fjTUL/5VzPo/8H
fZssuH6qiKL6mVG0kLOqXae9XtFzWegv1XOJ2gLNL853/50UP3n3rvojNb5A
/U7CycZC3VmV8NBKMHb0uqxTGfp8+55gXKf2iuDs0srPPRr/UvCv0vjr6s8R
nhdrfK/2zRJ8b7QxPfxyquBUf7PmTKFpzYy6xkd5wEl2qr/dmGwYOAvwfav1
f1R7X9993TeKvlFbfXoUnd/K8PbWnYzR73q6v71NougB0eA4rXex9j2SF0VT
c6LoEf3+WPMe1nOAxj/P8LxStcu13xC1R/XufeH2Q7XB7XQn2u+Q1ntS4zNz
/H60aGNTI6/HHnMF3xy1GZp7mda4ONnvvuoRRceE37NFI0u01yjBeb++Oyg8
/7HKv1/Tu7v1LNCagzR2ttr/6vfZGX7eo/kDNba/0P0Vel5c4fOWCz9lahOE
703Cx0a1GcLB7DzjFFprm2raeVrrT9fzTtF2tcafaS4a0NxLBOterTlAe9yt
Nd/SudeqndnOY3sKPf5f3d27mv+hYNqlfXaqLdTvg7qTr+OmpaHZvo+/C7e7
9f6Y4DxF9/+69t2p9ztSTEtLAu0dE5xRqnH7bIrvc6jgOVU8tkPvnqjj5848
0+cBrfml2nLN+zXAg2xZl+J5zNkV5rPGPtHRitN9t4WCq7PaZN31MsmTpWov
CW/faM5rp/veeA9Nvy+wxgpPiwXn43UM7+LAR7NyTbdnac5O3e1PovlvBMtd
yX62bWA59bX6o/TM1P0eDu+hZfgTGgbmV3N9xqEt1PT7RX37pWA6oPaw1h9c
qzsuEM9ojW9FS1v7ed2ZklW9YlHUWmf8VjL2v3pXr7HvCB4DJxN1rklq/QTz
dt3P/Br/rlV/gfrHNGeY1t2g+dmNDesh9Sfqrjum+nu+Teqj+aKDT4WXLdr/
YI3nQQfcwWfJvof/hn2fFu28Kbk5VPc+VnB2E5wNBOeKXMtfzvVHja3Ru48k
b7vqfOU6Z6nwvaG1xxKav0R7LlPbpn02C4Z4F9GjaHKKYB6jOZsF827kvL7L
rOt7+K5HuIssjyPDoa8NYU3kLzAg/2/O8fhUzfmPYO2q+c+mm3e4A+au0743
C76pqf7uywD/Ko131/g/ND5X5+iv82QleZ2Y1tkYWdeh83por1t1v3/R/U7Q
/T6T5XH0YKc0n3uFxnOEs+c6B57VOZ6L+5sbu4s/4qaP2bmmP2hvSugjc5A/
89Sq6ltWQmfQF3sgV9/WGVsK1im60xv07aEOoiO1XbrrEsF/mdZvp/l7egtP
RcJtI6/NfjM1Pkx83L7GNMJ+yLob6vg5N+yLXEG+gPMZue4jZ2YG+PhuZq7f
Md5dcmBq7u96YGo4F++ZN1XjXfL9mzm8nxbmrAryln2PC/7/C/jZJ/jLBP9W
ZGwHn/MuzV8jWsnT2SuRgeKXFuha7A7R6pXC+ZnC/2C9P1dtreZcViIYdd4O
urstor+6Gv+75rdBt4i+ftD6zfqYRqHP2b1sK2AnXCV7p1jfjtCa++obDmA4
r433YK9XJNuPdbCcSNVYstoL6h8W/AVlUXRvuu8H+NfVd2Ody+r7u+M09R/T
/LWdZCMIhmM6Vxed6wutM0Y4uFQw5OncAwVPftx8+kuyYeCMA9v5zgYKthTx
dRvNeUfvv0W/6Fz3IMdPt1w+3CTo5yzr6Oe159T+sj3U36fxcVnWoWeKVv8p
Xu6GXBJ+1osvVqrfTvT/aJZlXRfR0ix9e4747Arh85Esy1jst6xU22ejdf5B
edZdyfm2gfhdKjjL1JqLD5ek+PmbYJ4QdDt6fYr0w+S+tgXm6jlHbaz6HYSP
JRXG/Xj1TxP/3q317xPM7wnmaQFOdFRJXcM1LsAGDGfn2e7ieU7oYx+Ak4ci
7zGiuW2FhVr/5RrLUPQougQ7pDDfNl2sruUA38KbyAN+IyuQV8iqH+uZzrdq
/ha1NcLlA4J1YZLt5I0dbR9jk3wievxc8zNSrT/Rp8u0/6Ia83wLnWuM1r9P
ez2vsRyd/Z5Un29sOGNRQ68BPbzdyHYOv5/VmZo0s220RePbNXaZ5qzWOitq
bHMv1zOlzPpkTEPDDk5eFO5HaK/pnElzYjXmw56NjQPgeVYy8OI06+WJ7W3L
YMNDC/Sh2Q6NbHMPCfYhtiG8tbmV9RK8Sf8TtXStfVu+7dquauOSzKPbtP7F
wuFm0d528C6ePdLJOn1ttr/9ALtXMKaJf75vapv/w462498NNgd67ybd/8S2
ss8EU5tU0+D79Q3/JLVfk82vwAjPvqu9Fnay3MNmmxjmTBOfTa/ymt+Lfy8p
sh6/Qvf1qmA8LPz0EDyPScZNk656UjKmRDR8t86yUmcZIjk1S+fbI1gf0xpT
Bc8NWm94lX8X6z6mtPL8JK19Y5XnLNa9LOpr/oY2clMt6zpozr1V/mainpPU
1rXyd42L/O21es7R702tbNs8EfZ9XTBk6B5LGvlONoe74CzM3RR03/ZA5y9r
jflVnoe9tiPYbPgG7Ake0LfoY3TiPMH7z66Si+KvEp07RXBMb2Q7lzno68Xi
4yW1xt1wjd2S43v5FPkXty7CfyrtGHwr0fXfhIM9WrO75EqJWrP21pXMwd+C
BrGxoUPsQOxX/CD8IfwT9G37cul+3dFVLax7GcdvekkwT+jrPbjz8mCro8tO
6L8U6zf66Lhqnf9AnuXz5/puh9oC5IrOsTecBzuVMXwM5n6VZ3vmQLrxsbSB
ZQi2NbYNvme7fMvvL7Te5q62+V9Q//+6+ux78rz+LM1vlOrzQp8N1T+e4v6V
OtuoJPPRjT3F66LnafXsHz18uuUeshWdhp3AHqMbWCafItlQIvwfxEaWfrmw
yj5PvXTrmTma90FP6xV0yizBNruv1wWW8gDPMd15A8nAeaKNl/V+vtqj+DsZ
9lH+qrY83bbMcK1ZrnvZVil5INg7iu/v0Pqv1LOc5jto6cTYGR5fKfweFx5W
5Bh/xwMOD2n+rhT7MAeCHQhNXio6PDXoTe4GPEJvZ4lHFxaaF6DDG/Msu5gz
PPR7NXCf+beJDp8NuuCjGtv62PnTdd7dwlVd8f5Gjbco9p00jTsmAq5v1pwv
NSezsfdBNiGX3tf7D9SWt/cY79izTmuvh28+UnhZo30j0c+FWn+Y1ioXLO9I
Xv27l20E+GB+iD2sEG+trPU3GwXDx3HT8L5sr4uPyTmeCWcpEqyFqZYvyMub
AgzgqW6qz46OrxPgwZ/EL4W2wfH+YIfjv/+kdpfWbJfqb5iP7TOnv+0f7IYD
gReQJ/iB85JsQx0MdlSU4v3Zl/33BJpHv7AGOgb5Dc7AM3shw7HfJuYYBmwN
YglHAjwHBe+2FOvpCWH8hO+v58E8x58OBj6Fv1N1J8XyZQYJt4+IL+4X3j9J
/53m0N1b9PsFwTJfsC7Wuomg71J0X3dk2fbGv7kz+Dj4ZId72C+7U2uvEs2v
z7QtiEzCr4TPixtafwP/1CTHLuBJ+APfBP8KewVbhRjapmBj/JAQTRSYTs5t
4bWR5zc1891z79h56H5k+xFiP22N69mSi08X2dbrlG85R8yHs2DXdNWcz2TH
bld7Mtv2KbIWO7ltvu+fO31IfL8h2JPDhbfzas3rH8vOXKTxyTm2iYCZuN+h
TNP3Lq2zVHrzrDLLGXwq/LHf6li3o+OJ6WHbzM0xfpCVyExomGd2gGO+3s/L
8VzG24Y5y7X/54Lj5fCeJzEieG54oHlikMznLC8LJ1OLjL+54RvmEzNcH+wN
7NEB4S6wf7FBT9qfycEeRl6tyrGMur2Fz8sdvSV49vfxuxUBL+AEPm4S1nkr
1XY9d9Y26AjOiA6/qa1lF/ywL8N3QVyN+8E2HCB6uE44/Z803/PEcNcfSx5O
ws7Q3Bna/wLBMVL7t9P8M3o5BvKF7nm32rPZ9nOaZdreryearV/p3zUl9jnR
I3M1d57a9Zr/Ra7tVGzUJK05r8LxtOclA7LUxovPl2Y41oZ/dGob95nDc1kY
b9vG8+gv0hqLK/xuomjk6lrr0gkam1hhnTKylXUM/WHZ7v+ob5/T++crHP/E
3weX2L3ET/LUjiY7RjAu6E3sYOw1+PA+4WeMWnWgOcawq4n1XCW87mxg/UQc
nBg48mpdsCc/056z+/tuXkgyToiBfCJ5mNPL99cm07FX4q7QCjFA5Oa9urdU
4XmY1j8qfdJFv5ukORaakul46HKt/araOH1XR3P39/W7rsL5QM2Pp1m/E9tD
x/9FdPqT+kfUlul9s0rvXaR+c+61gWMoxGWB9Tzh+Y3OjnvgfyBzsMORNdAq
522h7zLV2mVa9k7KsbzFP8NPw5eM5zuufcIeE61t6uN5D+ibB9X0M7pRsK5u
bH88X7JxdKXfwWPwP7q4lWTnP/R9WnPLbuQjsvHf2JMx8+IyPbdowRk5jrvh
WxBvxKcrCn4dMUJgwp96VfM/0/w5OY4NMQ+fB93cL9s6Bz8lHuBnjVhYBz6l
z53RePd0OG/XsP466ZE75TP8LJinBZ0Efma0MH7wYYnBdM63jpit97OCnJkZ
+viNyOTOQS7z7BL6DWRTrCp03DRD+uVoV8cqTkm1XEAmAFdhgO3REB/HZ8RO
5l5OxE/Fu0/09pnvlwzfWWDYKkUDSbqLx9Pt35TF7ONc18n2Jbblnj72e/F5
gXdmgJ94InCjRx7V2rMLjU/if8N6OQZ4meD/j8bva+VzdQp4OEfjgzRntOYv
IHeS6Xg1+JsW9CI0jV4iR/Cd+GBbiW20EaLT5Ja2GYYmWReiB58TXe2uNA98
Kx6ZqzNux5ZWv5G+bZnmmMi57Swf4GXWgZ95poT+BeKJjyscS5mnNY9W+h0x
RXiLuOKJmE07+5J1uomGtVejuo713N/C/eNBN6AX4P0mgZZYAxiRLcQ7/xPW
ZH14mPd8eyzYwLu0/0X9PW+I4JlRaXwR89mba97FjgEeZDI0MDfc0dHWwfdr
4HMPDmf/JdNwope50zmBDgt0d7dJd1QJz9PDXUDPPGeEe8F+/jDoa3wRfJJv
NWd1js+LjtvYyvviOxMTHBr6O/vYZ0b+ttBe3cttB1SpvzpmW/GIzrq5wrLw
etHqKwVef6buZW2l7SjkE7hBRmH3Dwpnxw/Nzv89TnYiNlTfsTpidvhGH0ku
v53hHMFpWnO89pqcYb3d+lTjJEPw/CtmnTElxE2h86eJ9/fwvG5FjtUSp0Vv
4LujO/LEU6dIZn2QbP8UX5U8S3mRY6PERQeJ346Kj1/U/DeTzUusP0X4GdDF
dwX86LMTukx75qt10L4vpfn5pL7J09jkmPvcJ/QH/Mn6tkmW48wf64zfVVv3
fS/cNCvw+IZczyMHkZnq+ehK9Cf7YkMS+1od4l/E+dH9d4aY3L5AewsFzxuN
7fOeq30eK7PtPFx8N6az9Sj6GtsB/3B00N/o7ubNfR7gh2ahXfQwuaVGosE/
4f9ip+vez4Xmte5AbErtdURnSaq1nXuR6PxFvZsrOCakW5fOCXoNHsF+my1Y
0nvYxmYM+OFx/BHWwHaq1vuNMfsL6yQzxuvOrmxmeT40xzJ9Tpp9X/yC+oKt
X6Vhekh6vH+l8fC98HB6uXmDGDl43BJyTeTh0J/naU5qzLirr3N8k+88WH/J
xX69bM99LRzeHrOcw3efXGJfYYe+vSOMcx/cC/TPfXKv3CX0h85D390v2AZX
G+b1uZ7DvedrLDlm3dxL/bSY4/Sd1f+5yPgEr+Tk6R/XWDe9+5N+R4L5q3zn
98ifMoa/X1ztGAE4Ok1zsmoN19Uav6baeSPiFs/1t36pFM7OUOtJjjrgZ1F9
nwFYoUlsufUBZnC4PMzh+Vro46eS38IHZY1Xwzh5PuaRi+H9ylznx68RTn6s
cQ6C+exNbGF1uC/yGszlG/JNZZJFv9SYX7EH0Z/US8wSz14Ss1+ZGexNcM6d
wDPYhNhbX4T+A4H+TuiXJNsx2DDPaJ3eMcff8Ct3B17DTvq8je2oBeEczB8h
+rqxv2lzj2zDvX2dSz6Ybl6BRwYI34ekf7c19bcjg52DDXNbsGNGhj7jxGU3
1To+SAzhthzHEbA74cWTsPOEp1aIHh/VXZ9PDgHeL3Ltxvk6x3QBfGvw1ycG
PUIMjTF09z+0z+CYdfKlMeMRHBKrGh30PrbtJxn+TT5zfC/nNImT7QmxuMvR
z+LDKzT+XCfHsIhf4fOwH3oKfcVc7Ey+HRFgqB/kLr4862Lnypw7EYN+Ndv5
mk8qrJPQR29n+x3j43X2pn3sw3IPfWO+i2nqx9QWd7B/sL3CPsJN2HK9Hatd
KxkyWG29znWWaKpxH8dpkXn4ZNQ1vJLm3CT2Oe9vKPIcYFgb4AAv4Ac7ab32
2VDh37xnHvMfFJz3Vjimu1/24xlFzjmObeWYNblmYmvoQOJr3wjGqjDnQX33
cIVj+8Tm14Y18b2GBXn+nmCbFbMP9Lhk/6Wig62iwZ163qbxn3WmC9RvoP6c
Zqap7SE+Ra4OvqduZ6nWWaJ2v97nCW/5yIcO1jVDcswvQyqd2+auFgf9g89C
vcWSEKOp1T7j+/gbbARiOdgJy4NMgH/IORGXwZa4RWdNKnPsbpHkbRMRwPg0
52aQteiLPjHzJ7xZov7f1R8E7em7Dzr7HuFZxuDxC3Xevybs9/xc4TohaoSK
NH5E+01s5ljUgRCPgkZG9TKdwC9HQozrS+3zdsy2FTmbNcGHekq0dKbWeb2V
Yy7wFzU1NL6Dh7br3udqzu3gX/3Z4Y5GCOZ2wuNK3dEOrf9azDGV30QDmU0d
K5yl+WnCQx99+4Hg+qKT7SjsTupLsNVL2/ke4Bd47/JM0+pWdH6Z7Vt8SWQl
cvKkrEF2MZfv+XZJqn0L5oEj/FpyJUeFq4djtjF31Ti/C70QpydeT8wzQ+cY
pP06a/5UneXcmGsKVsRsa57083lin71S6Bw/ceNP9X5pzHbu8YR9cvxxYu/4
V8S/tmnOKzH7bfV0pvma9y/B94zw2qHMObSbhJu65IDq2caBL7Bx8G8mV7h/
T7B7sHkmaWxgL/PQjSFOjFyq1f73JGyXDtJ6x7p5PvYH+5CnW6R7X1jlfDg5
AWrdqG3LCvCSsyDfib5EV2JTz6q2nhml9a9MOM+XovVT1d5uarmO3MHX3hni
tdwR+bdrq52DuyXDcSbqnbifu8IdpQsHUTfbnWdIH1WWOwaWrb0eSBh+fAne
4xOBp6sSxhU1Te1LLEPGamxcwuv+Jhvs1x6uoZlKHqzKsi4SPPeVOiaLTZHo
Z7uCnCg5JOxG+AOeg0fwVYgZoKe4M/Ki1NiQmyVfglzgO2KyfEu+lJwv9jY1
icQBwW/DAudW+Z4cI7lG4j/okFuCfsGXbBNi6btFD/vizp+SE6gbZAs5eXQe
+o59uoW9yBNlhPor4g7YENgP2D/0sdULhKcGwk9mmsdahnHyAvT5rm2IlRMn
Rx+j29Br2WGcWD05W2LZxLHfFpwf9fVvYv98TwyBeizwA54qSp3nIMeBzXRa
qA0gV09cBvzu1j2elnBcDvmLHL4s5I6ZDx74ht/QJzIL/weZCVzZIb5PvWF2
gBPZeWa175JaD2r3fghxEnwlYgiJmGub+E0+LyvsRRyzbcBDlXB/htoT6aZH
+AqaTBLtnF9iWX2X7LpR3e2L3F/i2AFxA3IW6EDWGd3ZMgL5QJx4bog/X6vx
PRo/u4FjLvjQ2OG/auw3tS2Cf4joZ6P6Z9WxrNqRsLxaqvGDCfvFxHWJ6bIG
9jS1cvxGrk8Isp2aoEtK7G+S6+CJjXSZ1vlY6wyo49wj/hg+yCaNnV/t3/ic
wIt/iozfGvrYAuh7zglt4wtBw1ME523a63F9e0j3/3o351K5W+Zxv2/EHBtC
ds/QXt+Vev7KoHdP1g/zHluZnG4UvscehY8r6tgeIodNXJp7Zg3uemmVfVf8
VuqJ/lRiuPOwl+KOHyOrfw0ysJXg+XO1a9K+Cjkf4vzUIJNHocZ5ttaYWWK9
RhwUvINzdCN2DPpxc9Cj4PYm7X9RwrYa8nFqkCfIQmQn8jBNOqh/mevWcqjz
0fpNGtin4pycMbPW+deDwTfAR0Cex6lxKrEdlq51mlY6ZlxX5+1ZYtuMeCj5
EuzMWzQ2osQ1UJMTpldoFR8GOQ2NTSM+XGQcska9Iq8DbRIrgT7R58sKrdPJ
OwEXOYuuwm1ZiWu0GleavqFt9AM2KTj4QOsnqu0H31pme2Jl8LmnB7+7q97H
qu2Xc8/Mxfd8R9++m/C8Tugf9fPr2O8lp84exAu2hNj7+gLH19Ad+Kzrg2+L
HNzV1HjGv7myyrguFLzrEo51v57wHqy/WvdyX9zynDrNS0ONx0DBnl/gmAG1
DE+Wmod+rHa8CftwUJFxCj55PyXM2ZIwb4PzV4NPTzyc3ClxeWyw7+OumcYH
he9Lis37O+OuOyefW1HuHDn38JLgmaD2peD5g2j7h+bWb/goxKCQ58gG3jF+
rs49WO2OVo7bUkeCnfOd9jkWt9+6Sc/D+a7ZIwfMHGgvR+MjS60T5oT4BOu/
qP1fUNsfahnIwyBbiL18GfrIs8vD2beEvCv8gq2JPYucPxx3XTKxrwmtnWvE
T5mstSeV2U9tIxzv7W15Rg4nVuo8DvYXvA1fLxCt/hq3P04NJrWGxBSLxbP9
Sq3n2H9bgAF53K/YdMu5t8R9duAEBs5InSb1kNSiHI+7bh79NiuMgevifq7T
xY56WHt2jfkc67V+bYFrxb/V+hXFvo/FNY6voZfBa3bcuKX2mffc20Gt81DM
cdlP5Yvt62v7i1jR0lrHi4iV838Gcn/w4nkFoWZS/Tmdbcvgf/Md/Hiirj/d
OSb4Bz4if8L97Qv3Nbq1+8QEyKmTr8B+7CkY/1ZqvkAHJ4Jsf7vG9AH8xOSg
NWykdoJzUa3juUcE/z2llqtL1N/Q198Tz8L/Qwc9LL06p8hx+0/irlfAbt8i
2fJUF+eHDxLT7eKYNPVZm+OOq2+Nu2ZoZqh9pAYSG+1W4XtAqXO50NOWcO+P
ljmXTx7/RE1rrnMoyMaD4ezkakaFfA315gtO/x1fFwee5b8R1D7hL+M3F5S6
T93r1OB3PFFmX5dzdgm5G/I82Kf/DDYq8S/kInIP/fld3DqUnCT2OXXdd4oW
dokmrkiy/CD+hK9EzeCSGtcNEvuhhoz4D/KvY4gx0VgLXl4ivO2IO6ecF2o9
GP9J4/WKHQdbEeJeq0PdFPEgYj78Z4Xf8GlxyIXhW/eLua76ZN6MJ/YhNa3Y
/dj01PiQ/0fGEjfmnrij/qKTNbWOSXcod20WNRBFISfL2YkLcG70FzFOYprE
MzkzNi2ykblF4TzoU+4W2x4Zxxg2DPGCR9K8FrmgoWHNWMgJknM5pcS6GX2B
LbIj4Dk31Mawzju6t3dLTVPcw64wh7gvdbPEfgsDPMBPXjYW8jcpovNzC6zr
uXvi3/Q5R2awz6mhahT8NPJN+AzEoMjTE1dAdxAnxC7BJrlK+GgZM+/ia+YG
f/NJ8drEvp6H3Ub+HhuVmgTyTNi/2Lvk9MgvkGd4IvRPE699LhnSUTBfp3vs
HXdNNHEI6AK5jR9CHRkxE2BFFpNnpF7r6zCObXp5kA3kIU8JshpYRgd4+A8H
svx99f8W/mNDfPgi6avDhR6jbhUfAbqcXGW7E/vvtRDjRW/iG+EjQVc9Y/7f
xcncL0/ysPhedYNfRg3juqCL/yVcLehruuTMywMM2HPzgw+C/YgsBP6motO0
ctPR9VW2z7DNJoVaJOauSPdZkP/4SBkBtmW5v/+npJto4Yoy78H/G4CfPODV
Zf4/CfUF1L1+E3f8arH6Z3YzXRHb6BLkyTuVlrnIW+w7bFt0LvUK/AeD3z/E
zefw+EzNn17pnCrxeuJM8ObuUv8PCvt6svrb821zU2dGfIX82myd95oi5+5a
FNiGwH5Axm8rtpy/r8a19fAg31FXgX1+R/ivFHEOYmNN+jkWBi/C5/AjccTn
SxxLJI4+rtqx9Ou0/rWlruF6Xd9Fxf7vWpbGbg22LnUBL9TYFh4a4nfYZsTS
bq/xOfl/z8n/AF2vb4eVOq6Jf0hMGp8R3UQMEn1KPPKiav+mDr1Zrdeltgv/
GTthWYFzkOAHG5yYIPHAQ139v0r+a/PHAvM//M5/FDaHWmhin19U+/yX6/1Q
taPNLKuOBtmH7fJmsO/INXI+7HzqDg+F/y4Rn6BmhTgW9Ts3VfiMxNpXlfhs
1Cf2rXFN4TXqX13l+O7RIudj0EV7BOeD3Z3X+0Ol62yw7dBRvIeeiXl/GOLb
LXW+zFLXnr+gbxf2db0StV88kff18S9KXHe8UHAtqHBND3KCuB7+X7bgahR3
HRD/gRhY4/XRyehm6HmM9rm31DFI8s/zSxw3Ij/8Wq1zxMTGenZ2fGxtgePv
4Ir/Bk3q69oW6tx39TG+PtP8bQnH0YmdPF5qv4FcI2vjU+ArUB+Dv9BY/VMS
rhNoWuv/chAjfj7u/wJx/9QwsQZ6BP+jZegTq8JGgB6+0hoHEo7xE9vrEOJ7
t5Y7Z0OOgP8ttexh/uH/ELk1zk0j3z84mT9KNn9Ce9g7H8Vs8+CjclZ0Bb5m
SWfDUFvqmA7xnIOik7ldXCsxTnvt7eP7iGmfeI1jydh9+EXYftScHqv2ftRg
EodGrmNfx2OmU2p1GENOdgNnCdepnSVa21fo+ixqeKOE63iba58WNf5fDbqf
83AWdCgxCHTu2JjjvMiSp0rtW+Irknf9tMqxz1SN/7XIsTvs38VVtoGxveBn
eHlhwrH78eF/DDz5LwP/q8PX51tqhPmPDT4w+g19hi67Vzi4O6yPnJsUt6wD
19it1BZS37EyYZpZoufihP8zca/46J5K1zrhqz0b/DVkypoAGzqKukv0FH55
m1L7kkl6vtzZcPJfkJa1li/E7N+sti+FTYo/eaJ+W+/nVZs+qOV8rySsq3Ua
htjg1krnuZFR1BNyBuDfEmQScUjsGs6HbfOW7mdtjWNv5EL5XxY8mRDt7O/t
e6JGcnOJ6ySnljquREyJ/7JwP9iZ/GcLeLEfagrse6N3sP2p3TlRWx1iiujK
v4g+r+/lWgzyh+XdLXOpLeU/S8iYlJhz5Nwx9hd2Ejq9acy56jVBv2GDslcf
nTsv4ZzZUzHXcCBT21Y6/40dnl7jeD3x5yFx/++O7y8VD37VyXFuaifnVHs/
bF7Ojd2LLcP/w7BnqHG4qNZ2HjYptinyn/+4sCdyaVvcvit+KzUFI4sdk7qk
2rUC+EfQNfQH7fH/vyHFhucWrX9zmW2NJjHn6dnj/wGsXlGt
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtmnmAztX+x4+MGsMY65CZMCWZxnjMPPOYYZoxllm6tqKSkq1+abmVSqt+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         "],
         VertexColors->None]}]}, 
     {Hue[0.18868485119079234`], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFnHd8V0X2/i9ViAkhBCIhEEhCSEIglZCQHkICrgoq2FEUWXXtYEHZtaAo
iqJ+9SfqqsCusCpFEZGidARRAUVBWCyAigVRcQEF6+958wwv/7ivO5+5c2fO
zJw55TnnftJGXnP61U2jKBrWMoqa6f5NYxTtzY6iISp/OiCKCvOi6P1WUXSw
Mooe0e8uHaPoyx5R9IWux9Rmutr3VZvtajM4xvVnto6iKfrdWW1fax5FXVXf
NiGKblH7Q7FRdEBtalQ+PDCK/p0TRZua+L6x0uXLOnicS3S/oCSKRuiK0/tv
xvr+YSvT8JWuKvV9W1UUZYiG5ao/VK5x9Ptc9dM2NYrmqk1/1W9oiKLZ+aI5
TfcuUTRV9euPi6K8TJd/bBNFZ2mcM3U10Rjp2R6D/h+PMc2P6b4sxvXM/eqi
KCrQuJvVf8vcKOqucddovh0To+hp0f+86ndqTvdr3Vrp97BU389Rm9NExw71
c7X6GRnK89R362LR0lv9691K0VKla19b7Y36XqH6E1XfVGNWh/rH00zvGtWv
66917xlFv2kj12mtmqr+BG3ugqQoytK4hzXft2u1TqEfxvwwjDs0zWXo+k19
HF/lvRut8l7NYY76rNa4I0u8l2V9xRtaz3dV7jrI7WgzXOtwqdr0bxFFOWr/
iebeXG2mx/jeUfTUD/IY9D9S7W9W+7+o/T+034Wi8xeVP9J72Xp/rehcJXpj
qvyss+r+UeX1fVxrNUC/96rNPaJnoq4u7aKoWHTdVx1FzyRrjU7QOmaaB+CF
ilBe2Nbln47z2tCG9cnXvSDTvAGfUF57gi+evaH610OZNvBOXijPauvyOpXv
bC+eVPlZzWWO5resIIrKtScf6v6NmP/LTlF0sZ6P1JWitbkolMeKH7Z1Ft+r
PKO96aoKNH9Wqj3T+5epn/dU3qbycJVPE89vUnmIyg0qL1e5QuV/J3sNtBTR
F5085r9Unt8nip4S7+4QbV918jP6bKkz+JLWdKXazEj2+310lstae27M6yat
1Y26DmnvLlTbRM1tscpDU10Pb+/Vfg3THu0Ia8y6/6Lz9Yvm/bDqt6jNLtH/
F+3dfs1rS65pgh76GBf6f0f1U9X+fdV/qvYnqf0Pav9kryh6VTz0hmRWlure
aTCvz0o1fbz7gN5NEW036d0H9O6h3pZJtWHv2fdry6Loe9VPVt1xOrN3dfS5
O6y6EVX+faS79i/VNJ0Xr7XRu73UJlPlD1TWENFqnamOKvyqcT/Tuxklfnad
3rleV37in79pf2y/T9G63iY652juf2vhtfmj0uuzNEProTHWq/2HahP18e8v
RMtored9qrpf5Qt7+CwfTHH9m+rnv1qf/fWSJdq/yWpzUQ+3/UD9b9N1GNna
2s83Re6DNmPFA8928G/KX6vtq3lu+6XeH9Pjz/HHhLGeEB/9Xx/vUbd+Ovtq
31btT1H9H5Kh52jf68UnF4umCdqvadIXD4s/b00Xj2teV+ndgXr3M9V/oPqF
qt+muR9EXoq2jbGmt1Z9vit6Uhr8bJHKi3V9pPJi8dUKlY+TbJmo8SKtdRuN
tVT9Z6j/+er/Fu3BEbWtUz+Dte+n6Jquuj/U9rDq9HoUX6H3dCVLXk7THhW0
tPzMFu23q75Mv4v1XpGue/VuTKbXZZbKuZI5y8rcVyfRmKxrezetu/Zhv87k
71rXy3S/tNxyrlSH8grRPUJ7sDTZ86vRtVg6cZ/WYoXWYa+I+kZXKz3/Jzq2
ynMYHu869mWUxm6p8giV8zXfOM23hebbWXLwNc2hWvSvVp9NtB+fxGltNf5W
XR+JhqfU9mld52kNVqlNg/i4RHy0QGOfo3XPo58srbmu+1R3r/ooVPvFqs/X
vaCP1/pMjVulconq96jf03v4XB/QOm4Vre9F5hHOM3zSVnwRn+126KsZg6yz
Vmp934m3HPhWc3pL5bl698U+lhHUv6v73Co/g8/Wx7vPA+Kx5XV+P079LIg3
ndC7Mtfls9Xnmbrm6r1DGvtgD+v20aI5u6PP3TKd38Iq/34k3mszW/211No8
3MeycYB4bKmeaZuiter7k2L/flvlz1RerfIejfdivHlvfmuPu13tz2zj5w+p
XCe51K1QY2qvRlWZX2m/V+3vSLb9kCUeWaY+M2TPnCeeGa5rnJ4N1n2IrtEq
/5pnfoKXztf9Al1/V/muNn6+qrXXfKjme35z34eFcoH2tJv4u7fmeEGy++AI
vRvvs5Widz8Sv8wd5HcyRHO6rhmiOQ1dXO0ztEZ887zeaY48SXZfrHNHnYvL
1GZCsuc0IcwLefTPeOuOk9Osa9BTV4kfpmqs58UPb+k8Fej9ZtqjhSpn5Htt
OOOswesqX5LisaD567aeN/WD1eeTKv+C3ND6nV3l/UDHonfHxPl8rNSLN2ru
j0n37W3w70zpwwxdnfTuDRXmb3gbu+b8XNs2J3V2G/ZlvOyOs1VfBr9prCIu
jXVchWUc8u0M6adctX9UY67RHPOLrSuLtL/FNX5WIBoyVb8vsr15MNjJo3XG
B2mee5HJ2sc9qixR3Zval7d0vS/+faGD9Qy67+MU6xp+s3aliZZvn4uep7XX
h5OsrzJzrbOuVh/X6LpTfdyqNplqc6faDBQtg7jibVv3VT/3qP0cbVS99mJ5
svvuG/o/JLp2Z1vHTdbzVvodq3lNE93H675BOvcV6bc/xHOLmtlWwu7GrkCn
Hm7jNsNEw416V8tx1H7AdkBfL5SM+lTnZZr0wz7NcZ2e5Wtt6zSPleXWyemJ
vkPD4nS3wX/pq/7+k28bPksdHy7yeUReFuRaZv6iulPFZ3HIELVfJ35IEN19
1P5srUO89vpKyYFG0XCWxj+5s5/DA9iwF+Xajv1G+3uS2j+jtYpXP211JXby
+YgNZ4T1iA1rwj4XdPJePxrvtpNEZ3yK2/2gF34tMn3Q9pb6XpLr/jK0VrtU
ztV+faD9a6n77aLhEdXdoU1pod+ndfAd3+RdrdPP5d5HzlLTJPPb5erzb7pG
qs8TdL7jNdZBnZE87fXhPMtO5NRBXYdUf0Rtt+davsMfHdTPoy3M1z+Vm7db
i7aYCj+7XW1jK2znoHc7qa6n1m2R6s+t8m/qf+3gfemkM5maZD+ug95L0tUt
yeeK+0r1c4LqnpF+TdPve9RPv2q/f4wPsP3mJfr5i4m2EZIrbCdgR2Yk2d/8
Q2t8bqJt47mpPkucI87P3FB+roNtI/ZiovZ6uNYkV2f6V23a7Zrv3eLxN6Xv
1+t6TO36a+8uSbT8TNRe1qh8jvp/X2u7pdS/2SfOLnr2bJ3XB1S+TfPeoP42
oqf1e63abi61j4mtTx08v1jyrS92u/r8Xjbqd939/iT5TffpqhMd89THfGjT
u7t0ZnbqOkvj7ld/P5R6zi+289yY1/Pau5+zvQ5fQafWc1ii5cSicssKfNwW
uealE0XzpA62i77Q+Hu6+zf2I2uFXXqP3jtlkNdko2juJ5ofV//7RMv2TJ+j
bTovzTT2OPV5m9qlqG6u5pqms5Ze698/6r2DuspUv1Bn9hVdV0vev6J9aFVr
+wdbckCR5S22TGOwZ07RvZXOcrnKTXSWE+tc/jjOz7eoz8Xi76IU25tNVH9j
G9PJGsEf7C/7/XMof6h1ilW/YqGoIMtnlTVsp7ojBZZ58FKrTPPT7GAvYiti
M1LGfr5e9N5Q5PH2Fpif6H+brtIU4wzoSeaOrmyTaz8KW+WBdl4X1uRT0V+m
9ttj7c+MamOdWCB6CnVlpdimfbaN7VqwEfxKfF/wktk1PmubcmxPc0bGqLy5
we+AD+CTghUMbm3/BV/1e407OtOysbWenZdpG/sujTcm0885g7S5MMb3MaF8
CXul83iF+h+e6XdHqM9NWf4dozbdNd8h4p8u6u+FWN/hjQv07ogiz/Nl8Vjr
Qj8fprZnDPK6Hyd+aKlrZ7xtA9bgNbWZE+P+GYv7+WEsfIW7i2yDoUP3d7Ie
OaD1OFjj35fl2c/B9tyaYtrRU+iuuBTve4P0zFydnZWa+xZdW3WdpfYtpFua
5VgucHZXJPv8fqj6E/ONk/XT/cFq07NN9TX5lh9JMW6Pz7g8+Az4C/gNS0M/
76p9Wb73gXPG2PDwGmxAXUPU9rQY10PPdp2748K5Q0++EXTlR9JJifgY2OEp
rkffHq6xzw8fvoyu0DVK5YEa80XRvFr9jhKfvK57l0DvykAz+vOZoEOhhbbH
sK9Vocw8loW59Fefs6v9u0k7FILojLFPPznXsr2HaLw+3mX4fidzVJtY7c/Y
fPMYcgYZsy/ZPAk/jojxfX8o3ykarsv085uqjB3QJ3x6XeDV75Ldx4jQJ/4J
vkmizlb7Wu/9OI35abX7Paj7IWzjZOvIi4IftFF1m6pN63C1f73a/IG9/Yvu
N6v/b1X3XbV/gw2gO9Gb4HCXFBtnwzehb9akueZ7Rb7xV3yFq4K/gE2Fr3iJ
3v1R/f1U7XXM0T7OyLedW65314snx7b3uduY5XO4Qedyo64PJeAely74q9Yl
Vnx7XfAr8ClWaV/axtku3a++v0f/JluWIkfh2dkBOwHXQX4gR8C1Dst5OiB5
N7ed8c6WxcZemsv2fKrMPtwK0bihzud1isbflGudjfweUmsZvi3Ja4GP9o7O
/Ve9jdfs132S6H5AfV6UEHSwaO7QyTYW9tVi9ddWc3xB63Of7vfoelD9tYv3
GYD/H5Qu7ax2X4nPN2QY00KfYvCeV27f8apulvXMc7DqT63yfLFvwc3ABsGJ
1wY9/mia8doM9V/eVXymMUfruh67V9fNKm9PtbzBBumkur9qTV5ONHYLFoxu
BbO5MGA7f0811glO+L7a5KvNpSq/onWer+tB2cL/SDdODwbL3OYfr7umcorm
kVZvvJ3zlBnO1HLJ/mW6JqUZq5tebL9+vn4XtrS/A2+BQ4Aj/axx1qj9ZD0/
K8G2Nvhhjuq26NxenWA/CB+oWPUPg91VGV+cqXf/rWu8aByfZIyf/l9Q3Sxd
96h+t9ZhUJkx1YG679Tvf2pNDuh5k1rr6Vc197PLjfMdjPVYR/HPGttV2E/4
Cl2KPc9vq4yXgYU9iX+ka4bGL9G5mCaa7+FMid/K9HtFS/t92PzU91Cf2TX2
A3fJN8kXTRMTPeaUMC4xC+aCrhmb4DNKfOQbrcl68XdKO/Ml/Il9GGkeJxcZ
N2utcgtdzcTnd9dZH9Busu5DdWa3a6y3RM+D7T0efsSBgGlMxParst1FHXEX
9gP8+LoS470z8+0XUY8txbkAg3lZfHpzlW3or7W2X+hakm5snfjH0hjbU4PD
GcyTzOita7naXKj1W51r2/1a9XFt8CvjKrxn7Fe65vJTtteqRDTcrN8t1ef1
kiUXavwq8eft6cYkwN+6Sre30/puFs3rOUPl9jnBlbFj8A8nptuP5110Fbb1
SVrOdppfgq722qNs1f+tjW0A8Dh8BXyObZIb5+VbZz6l/o/XvBJU3138+bp+
r2xlvwpbnX63qv2AfOvw3Vqbj3W9pPHvSXJ8AHt4rMZ8Ve/2SHRs6vIi+9CF
WT4f8UGvrgu6FX/+7IC5Lax2rIU4y4Bulj1fNjcuiX8G/xfJ1s5MMgbeV2vT
R1fbDONqvHO2eOrDRNs94CHEIohJ0H49dnGZ2+yWzNh1guVd7zRjK7QfKH5f
VON3vultX+KoH6G29ZxPtf9YvDRC9ZeL7jqtx73ax47ax5mifVeV7cSxadZv
2Bw3qp+FlX739BjbIE+A+YAV9HPdc3r3U2Sonr3axXYItB0SzZerfnJH2+dD
wjlCb70T+JY9g0b8lEfUZ2E/n9MVicZ+wOTa1NmHxyY4Q2vQT7//k2h5AcZw
h+j5rrdjA9jR2+CFCuuRF6QLZpTbJ4QfUwvNk+9xtmIdh5omGveV2s8C8wP7
AwfD/n8i1jb/x9rfi8psF3O2wXDxX5CP6EzspWT1f20v+wCHNWZJuf14sGPi
GMTOclK8v/DzgTjb9cj2yaJ/TZHjWZ/WOv7BO9j7lcHmv1DjjyyzX/y/DPvf
zHF9K8t2+kHfswb4EWBlnC/w5FzR37va82NdaYPNnKh1aqerS5JjJdgf6Iex
oj2jyr4h8YMi0TtA9edU2Y7Hhn9PND9fYZ31mc7XVaL7bM2vtfYnRtchlc9P
tT1Dv+dJb76hcdYl2fcFr+Xdqfo9P8YYHT4RdiE27wUa6+Xejkn31Dg9dPVG
RnVwPIj9n5fu880ZPLXe9gS2xATJur9WG78cGmK4aWpfovPRR1cvyYGz1N9Q
XWclOT7QqsIxgi2SDZt1Pau+/w72UmM/Hf+cfthTYkr4D8wrdqBjWsSzWojm
ebn2/14TLecM8jlGbsFX4CTttH7zZCP+Fmeewy8BDz9HY33Z0/P7PMVxVGQp
MfxDlY6v/xbsTmzOnUEGcDanVlgXogeJLa+Ntb0yTvM6rq/HLjnBa8Z5AW+4
N2AOzbVXzersN4xV+5c01upErxE6Ct3IGmHfgclzvo/aISpP0jrt1L6v17ul
xFt0VSXZf8Pm5PzS19Gz2AIAXTpf7feo/RxiJcWOBwxJshwjBsQ+Z4e9/mt7
+xLYzMha5DE4GDgh2Aq4CnH/tGznMTyQZowODK6F6ppnOy43U/vQW9d88V33
LJ//M1o7FkzMGEyTWAvyFbn9ZLVxHzAf4nTIAOpLRHufWp/jp7RfG2qNmzTL
tkzD3iB+ODdgu8T09lXZVrlKvDy90vghfHdG4L05XRyfxuZ8o9R4MboF/Vea
YLs3QfzcVtdm7dfTmsdTunqJ9hjVXZVnHGuy9vMhXQ3djJNsCTJuCVhKqbGH
9cHuht+ul6zdJZtkltZ4uOZybrCNiZ8TKyd+Plz8cF6ZMZBVwUfEJ70wzX4i
/iKxrIoEy0MwBnQI8vBZ0fVBifVCrOTiBRXGCPlNH+SW/J7oOnAM6D011G/J
sI0JzrxYa7ao0vz+b93/pevNE4wjdqwwVjhG+/VQrrHRMaLnhQT7O1egB4us
4xdprXZX24/kfc4ue0HcqU3g7Ula1wUN9i8X038v47zIMvgAvY5MAYcFg0Uu
IMPAAJeUOebBsxqtebWuAvAFsBBd/xTvPVFq3ww5jf/AGQVDxobGfuLM/080
/lDts/6CdOLGBuNAnJuPB/jszAvYCrjKVO3hu2q3Vmv4djvjUuhbZAA2Gv3f
VOfYDLGF9VqzNyrsh43RHK+pt+8yTM9/LLdu3dRgLAkciWt6iK1+0s2+GRgG
/iJ3fP90rVNlkfUQPgv30U0sx3iP3KSOwZ/DZp6lfelbbpmMXwFdxCPSNG63
Bo+Dz4U9eY36GSwZewr2eobt4pbBT+kuHl6SYznRU/zWpM768JQ0x6iQX2Cc
YOnYhDuzHEfFp8N3nBT8R2xTZAs0XK91qMuxDYKMBx+CT/5b4bg+Mf3TtD+n
VpvXFpTa/sb2vld1b5Y6xgLOhMwEa8LmOGpbqdysvX0t4lzpec6FYuxftL/j
K0wneTfrg9/XVHOq7ePf7zU4R4fnxKrxVTknN6r+hgbbg1tDLJ76XwMWg764
RPMaFXzSD4JdjPw8qyzEwOKt7xkfe5g4DnoW/3ScaFhR4nX/XBPaUme/4kXy
iRr+5A/u0HyS+KG62L4BehH+RZcN1fqcXm1M4nMx5pBy544d7uF4Ne1HJxhX
Q0bgm64O/ik66rMB1lOj1CY2xrjleOzcEmO9LcUPz4q+k7XGMzXOjGpjG+gG
/Diw2U7B78E2mlpnLIZn2IB1ZdbJxBQXB/tph+omqM+09sYB0b/ECJ7Vmf6P
rir1PVV68F+6bkl33lBcmXOHkNfIcNpnBB+UGCW+GTrsaJ5enG1B/Pw7koyd
4HsS78QuOpZfNLO9sdx3yB0q9x6C+7QO7cFEwUZ5/1Wt15I0vx8nvnpYc2sl
2vvprJfVO8eEvDFyLFhH8AIwGfTszQm2v7D1/6Z3LxbPD9Oaj9CePldhWUfe
2Y6Qe/ZymvEFfOa14rXNCfbj3xRvfN3gddyodb6sn2mbKRv83oH2HVlfnuMD
vqP23zXYbt2g8j6VX41xnJmzwnkfX2fMlDmTm/JqyJfAplwa9pd8juKQH9Ke
OF6NMaT39N7GEvNOseZVVGv/MkZ0fVLieZMLAV6Cj/O+eOe9atukXcQzb6V5
TOIr+D20IaY+Pda5N1NqjWOCPU4SnW+XWOaBhWfWGgf/Q2v2ew/3BVY5OuDt
PWqNm9MG3gV3GBJwX/YCTLi72mTo6qw299cZg+4afPtlYe6/SzbcVGZ7j/PD
c/TpUtW9VmZ926fMuVzMc5XKV+RYJsEvnD/s8BWqX1pi7KeuvWNvxN3i6pzX
xe+PO3h+yHPiRXeXG7voIBoPFzhOVV1jXx0/fQo0lzg/kzxQcjKhmzj5iQnG
JIl3HcsTOYqDtrY/8p9022v4ID3UX2WFfdqfQxyQ+MuIDOttMGHyVLDlmOMv
msvPZZ7HVdqfK/sZz8R2wXdDf12cZlyb84UvRMz2hYBRDwnr/5D6OCPHPjr5
mfhqnIUTwbgqbMPCv5vyzcPIC2QZNtO3qn873zk+5DNgJ5FTVlVjbAJcYpr6
31pse+SBdPMavN0rxfmp5GCMCv7nysg4BfXol5/Uz5Eax5ywbfC7sG/wy7HD
sKvA25oFzO1byaop2qPvZMe31X4m1Dv/EFxnXMB28rNs28LbxFJ+DPGUZSW2
J9DZu7KsU4eHvSNmCcbOHu4MZWy5rcGe+yDDuWLkiV3XwTgEMfeLwPdK/Rv+
JqaIbcN7W0J7fCpwU3BUYt3lId59AfEZ0TOqs9eOO/Ycchwa4CWwjRl51sHI
PvohJkIu5JGeLhPb3BdoJlZAmfgBfjb+NrHv1cH3xp8BP5oX8q7Ak3YGTGmL
1nV8lf0V8HFyPInXDCt33ie/6eNA6PNTvbdH1yvp1ttbw3zJ6aIP/J099c6h
BAuaTXvJ0ImRdQr4JnolIZSJI4B/D6y3vcq+Lolz3Ic25EBgA4BHFXd33BCM
vkt3n3HwI54xFnhVUWhD3u4PA4yZYF+f0dk2Nnt2WdjHCWk+h5xB7MBbAn49
vo9lOvIamwWfB3/nR/k6HXKcG4l9D36ET3t/uu2BGU3shz+Hv9fcucvoFnBa
ZBVt0B1X4jf3dw4SOuetfOuV1Ha2hcGdOMeUOcvI4bfTLDs5Z8hj8kCoo4wt
fUvA37FjM7NMBzSglzaGuXQlN7DA/jB+0+wwr2N2MLYQerRpnPUF/W8I43LG
sGWxzx8LZXQcWAvyDlnHGV3Q6HP6mMovq/xQumMD3KET2Qpmh65IzfIz6pem
2Q7ABjjSw2XWs7907xmFjgOt1jz2oHP1rFM7vzM/2A+vhfbYgMQqsAPJLWRv
vjne+DFyChm1TP3sVj8Lgj3AnXVbKJrfa/T+8R3AK8FOgM4u4X3iheRGIIfB
Gthf9Dd0oJPQR9CPjchv9gIfnjV6Rf2/3+g1q9ecLqv1WhNn4Zwju4aJv86p
M7/OarAeQgfhH+AnYFtgE0zobFsIbP6U4DtgE8zsbB1Mbho5ajwn3oGtht1V
k+DcEWw88nGWhpycg6Ltx8Y/MUTuYAjYMyeHcZ8TbdPrPMayDGMV6yP7GJTB
E4hfEccibwG6Bod3yfeCDmIux4ecLfIZsOmwy4jPJklGvqv9Xp7h/rmDO5H/
gHzHdgEnWRbq7+orXdrXvFBR5Dw5+j9R8uTWvo5/gddNS7JdCi8TY4Cf2dfx
YU2wx8D6OFPETCYNdNwEbJRn2CqjJc8WVbgd+Y3kO6K3wcMpk+uI3Q62w/nF
bloX+gQ3Zk3AjtGr3zdYt45Q2wt1TWpu+5l5wEfTk0w3vNRd437a4P27Q+N3
HuQ4AHqM9/EPwVbAP6hHjkA7cgac55Js57Wi+xiL95AFjM9ZYLypYX1KB5pH
4U/iQseF2BC0bwx7in+wurNxHuwAsBVsk8XijUXijdf07HTVDUszf3IfGsrY
4CPDXmNTzyi0XY1+WDTQOuLRnv7OhHzdoaGfGQFXo298Q+z5ocH+WaRxF2jc
JXpWRk5dneNTg4sc6yLOtaHEtjh2OHsAFgGGD17dNsgK4sAbQywYLJG4F2fq
DfW/tM5zXqv+oiLjVfep7skG20W36QylFjtf4Nx8n1vOLPFofCHoxGYjdwH7
Cp0E1oWtm6/+72wwPkN8Y1KIcWwt8Rj037TIOdZgflOCLECngOff29l24T/U
zx11lgvExMGRwSuOfRcEDkyZenLn7lT7e+r8Pjl3y0Pe3YSwf+wXGNrIQDM8
xV7BV9iE8Cs8wx6O7ONn7Mlp4V3euzi8O7U45HrGG9Njr8D1Xg/5HPiK4Hrw
P/3y++Kwv3Vahxpd30quDCu0PkAX7Miyb0xcbjt5oYW2G4hNNYSYFrgpc2Kt
2XNwKPYdDHhXwIGJ4RODwUZiDRaGvK35AbfCrkNf0pZYdmPon/NOnmBCV+cK
4qu07Wp8EFseOpCf3bTOHYosJ4mZE4cAg3sR+6vR+mlZWHv09YgK44/0Q1x6
TohNf1JrzBq8mvEHBBraq+6JetNBu4lB3xHH6dnVcRny2+aHuVwaZAKy4S3x
3bP5Pi/EIcF0wFOwm8lPAhMmz5cyeUp8k0FuGfoBDAB7Fqzsq1TXYwPs7ODf
lLH3yWHC5gfXIoePvD5wG34zFu99Fb5xQN4Sd0Fufl/hfAVyFfhe4btGf7Pw
mfZ5qeicrbYfaA5fq/xOinFPch3AKWel+DmYxu5ax6hYB2Kz74QcrQr1PVfr
ltvV+b6zgp+CPYKex4a5T4Q/WmC7Av+Ntnw7Bs1fhDmuU99v1Aa+KbFeR6dj
e9aHb8fIx/8pxTn5YIx7QrzkqV62lbFdsdewEZC9VWX2XfFbOa+zw7dPfBcD
rov9Ca5EXJcYNFgW/YJZHa/3Yus83gTRdZeua1VeU+m4HX4g+eyXpjhuSE4J
uSXkoGKPDAj0g002Dfjk4nLHrRNCTinjgsXhh/E9GDZ7TqVzesjnIVZcFuLF
7AHPsedHqO+baj32XvY02/QzP76Dghb0dn6RdTdxO/LviQ/y/cPVXf1NBDg2
MTnOEd+FlBY5f36V+l5T63NL7Js1ArMiF+yHFOsd4ni8S34j9+9D+VfR0jnH
/hRrfCis80zt0Zf1HhuZwjPONfTODjJ2YKFjisQT8Wnx/ciZnBPmRRvWFFwK
u5p4y6yw5sicj4OMIlcJGU//yLKysG74dfRNbtjTlY4TECPgm5s3a/3O/yPX
qdFnn2/ydqVYR1wr2kfXO0cU++5QgW07/JiSgZ4jvMxewXvg93PzjYHO6WYZ
Cg//pnE6Fvq7T2QV9cirGPVxvK7P0x3L4w6WTy7ewgbn40H7UV9LfU4udNzi
WE4cdzAoYgtrg3+BDkc2IZfu0v7+r9E5w1PSjLODsZ+kOR1ptE0I7g5d8D95
D8TVsDH5FndKaP9oKDPO77Wez7rwLs/gN54/HtpgM4KDo+OIw++u8u9HtH7P
aNw70o0BoYuQ7bcG/xE/IiXoJXQlsRLaQsM42TY3y7ZZ0sWxAeRRVpy/Q53R
aJmAXjsqxyPHLmmL/iEXBz+W9vOD38JY0H976J/7+FDG1iE3CPuWuM6x75vB
v+flG/smH5lv34iNMDbvs4YbxDgZhc4RxrbgOf4OZx8MgTMLjkA5L9S9FOo7
ihe6h3dr+nit14UcJejGPt+dZXsUTJk1QD4xL/gBnBF/HD8cHYNPyrnZn2q8
k/j6Z92dnxCjdVshOTtOfNpNPJNWZ9wtS/yVreu57s4n4Y78JtZ5ZTefvz06
Qy8O8G/OKTl2yJ8JAZODV1nv0YG2uzWPibX+zVkiBoV+RJYje9GDxLLIQcdn
H1Tv3ALkCPGcPUH3PdjR9JKzhNw+EHToOLX/e73nyVnkTHI2OWecQ+aF/lse
eAMsennwZ8FiwF/AXsBgFoR94f5KwHz4xgd+hVcXhnrabxU/72p0G3D+V9Pt
j5A3xP4g518J/MYe4ovfWmS+vzvgoMQIwBDmB55kf5eEfuAjeAJ+4Fv/rxot
kz9ptA6AZ/Zn+38BWAd8V3BM1nC7aPuy0TSRe74z5Fs0rXUuOzgquWErgr5G
hm4JdjX+9Jp0+9TESxgfP+4K4tt5zvPBnkGnstb4bcTmsds3S+b/XOkchKm6
T6t0zLu/ZOYNZcbBwMPAxcDY2CN8fOQh6wE+zJpcWuQcdGJrg0P+DbHmj2Qn
TKszbo69kRqwBfzcwpDrj/zAvmWf7xS9L5W4X3Ki3+5pvTC/0jRBD9gnPjI+
NTEFsHBwCejKDLRhX/AbXsIuB+8Fd2efdwVcFDskM+AVxGvI1yVutTzQA++B
P1BPDjk5UCUhD+qHcn+P+nugpSDQg4zmN3vBXuWFmNY2cq3rTAf4DPXs41uq
n1jn2C3fHbHGR3No0y1bkat8Ywb2R64a8nJHkLlF5Nn1dVyHb6T29jTWNyr4
I/gWZ4jeM/sZG8GHa8y3n3Wi9j1xoGV/OvnAvf0NCnY09jS2HxgkGCNjo6PJ
U0BPH8WUsxyrxc8eEXztzxt9xjhf7HHvMEfWckdYT+rBl8D29mruS/J8tvAN
wLCOfhurOb2Xb4yQNaAeXT8xrAllctxahzw3cKnNgQfACOkfnI19Y//YU+Jv
5GQf2+dNYU/BHuHNM0P+P/tALHyX9MLN+c5V7y/5OmCQZTH2zoNhHYiPkucL
hvyT1vLIQGO15Js0C1gE2AiYJjgMea9Tehk7AkMiB4Yy8mN30BFgpXy3CV6K
z0isAv5tHvwY2oN3ERcCa8orMX4Ndj1d5X+VGPcCz/wjxLP4BmJzmeeLv/F7
oAebm9gL35rRjnownPocnzfO2tie/s6QbwyXq+/Smj9j5NzBRckxnxryzIkH
Fwb8CprBsMCvRmufny5xPgq5sd8mOc+w+yCPQf/gLTUhJovv1jy8e5r488ve
/j+C54qc+0huFf9x8kyOsfEd2OMV7heZMavEcoPvyJeHb8k596wzsnpzlb+B
B9fHv4wPuNawEn+Ly3e4YExgTdif4K5HAs45JOw9+05+++6Q78V3V3wHQj15
Oq+HdwuDfGBN9mn9Pirwf0VwdnYFeuAt9D38VCF6K8u8L1+X+vsxYpvLw5lC
Dx7D4/C/sX/B37GB8cvxz/HT+Q7p0hzn8KRqnbvperK784XJsSZnGHyRPYIf
X9d6rq107ntjifNmyJlBhtIWH5CcR57jWzFnbA72AgyY9cF/5AwfDmXydpeE
3F3WjnriA0+p/92BH5ppnKZljqFxdlPD+eW/cJJKbR810d5+UO614nvT7Bzn
7OHfXxr4hLOXEb5PJIaJjgOnvFxn7mW9v0rP/it9ur3W31hnEP/VNbO754c/
zxzJtybXhnU7K1c+RINjqvjutMF/J2bP/1yAExIrJv8U+5/cmfKQP/Ntsb+x
J+Y2psTfsiKfLyf2mm1fD6wRzBF/s2kYF3wsVfR2Jd8uyX2/GPrfwXfxFY7/
ECviTgyIvFFyl8gdzSnx/4aAT5eRr5Tn78g566vCt6XIbvBUYsRDqvwdNVjZ
E8E3+zLEdMgbQL7S9uPQnhhyfqnjyPwfwJkVztGap7EG5fsbKTBvnqMT+TZl
zUD3i0x/O8R+sHWxc/ET4SHsVvQOcoT8O7AUvuEGN2RNelX7e2P2vUj3nxqd
C3y/6vvk+39WnhAtf+llXJn/3fm9p+3nWq1Jszxja13VZ/9cY5hZmke2rudE
0wOykef0Nu6F34e/ip11ZZVzprGjUittM2Ev8a1zToW/dx6gPuoH+jse9Bl3
7KN5jfZX0VkLNH7vUufjFfHfLb39rSrfGPXLcXwQvJ2zzbrs03s5gbZkjXl1
hb9vwb+/NdE+fg+dm8wy/yZHkNgtPEauwbiQb1Au2tsXOl69Cwytl/9P4vQQ
RyEORu75RSH/HEyTWAV6k/zE1iFHEblycfhemBxSePNoHrj6u6jCz8gljAuY
3uAQx8H2wD4lJwO7iP88OLnev2P4RqDUOrV5OMvouaulK6/pZxuGud0S5ku+
zP+FnBnwL9YDf/Z80f5tjefP96DwBdgd6813raw5+UDka5MjwHe3N5TbT5kl
mvMHuZ74Dv4DvgO+8ejQP9j86QGfB3+jHv+d70T4/wvq/j8ez/y6
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtmnd4lUUWxodOYiDBABJCC0EkhBQSLimX3CRCCquCuiCiKBbEtoJYsLuL
haaIq48FpakgSBdROlJEsKFItVEEOyCuSLPt7+Xlj0m+uTNz5rQ5c8qkXTP0
4iE1Qwh7+FOb/wdLQ/gqN4SbEkK4vUkI6SUhVFSH8G27EMbS/4GxY8w50jyE
o7T13UJYRzteL4QDZ4UQVxHCBWUhFKeGsLFNCMOLWdcjhHK+L2L9CdY3YXz+
mSFMYY/CSAi59M9h/g9JIUxgfXlOCM3iwYP5vRjr0DmEaKphjswLYRTtmzoh
LKTVKQqhLu38xiH0Zs2oriHszwhhJd8t2aNbYQgltF/qh/AM8y8Dn5VZwGwW
wjDgn6S/rzKEt9qGMBoGPMp+jzP/nAYh3Mp+Jxh/j7aFuXWbhjA1M4TJ+SFU
JxreR4x9QMvhextzmrF3PuvHJIfwJPw8Buwj0DwBWmfz2wH42QF+bWIsLiWE
7zuG8E438+6+ViEMhB+roP/NtBAaNIJ/2SFkwO+z4XXzuBA+Zez8AvaEtla0
HcDbRitj7Dg0jYgCvxN8rRvCamgeyNgx6H2O70cZ/zkWwiPAqMHaP9i/FvQM
Af+Hwf0p9n+V/RYwZx1jvYE5FN5+Bj3v0J8Oj3PYexLzZzaxDF9AXhXAW834
VeDclfmNwbcJ+I5FBl9C+w/0v6X/Pe0XYG9jz0/g35+s6YluNaS/Fd7/jI7V
g/91aavhTy34XwNe3wuPlrYMIZk1R4H1Gw02hcPAGwwu5wJjH2sHg9N38LsG
PBgLvGXIdBmw/0BfGrB2Nm04Y3fSIsjrHuQ5jbXTaQtZ/2etEOZB37/g2Th4
NZ6WDm23gXNN5v4IPyZ0hxfAXM36Rq1DKAWRGvT/zd4H0d+b0b8l6MDyFubx
LdDyJjx6BN5cCj3J4D6H+c8kmke14Pe5wDiU5PX3sPfdasBvi758Az37af3R
nzehJ4H9jxf7LPVpZVkeBb/Q3PstBfZJ6E0A/mu0BujCVdDbCvyboxOfAutd
+DkA/swE3s/oRn/2G9QQmtGRfZzVr2lb+D4Ivmez/gHWj2D9w7RDrD9IO3RG
CG+zfj/yTYdn09h7Hf2/gH0D+N0PfsvZ4yFgt8u2LtZhfQZzH2L8acZnsuZD
5JuVaXwln3HiJb+1TzQPxJuarD+QZB7dD7yG9F8D3iTkU5vv8oh5p/GP4MdB
+N0C2MvQv0bwezBzCjhLjeHRCNa3pb+K9ZfC093Qsot2Hix8Hfy/hpfDgTGQ
tYfRz87SNVon1j7DbwnwogHtBWRzI/hNANaH7B/HWDn60RJdeozx8dD6Ajo1
iLPzOTSdTf8N1lzC3Pgq+MZeNcDpBLCP05bUtk0N0L4Ue7AXe3AmPFhE/w3a
IMYLZS/B7QlwfC/BezZk7Zmlpk3nrR22MgX4BchuB3NqgN8VzL8vxfa1J+vf
op/E3q2Zfymy6E+7FXmUwZOFfL9Bm8fakfQXA2tJlWU9CvQGsHa87AG03gvM
Fuj6i9C88+wQ5jJeCO3pOlPQXgLN29GPryutu7KxN7A+8bT8HmF9febXo1XV
sc1szffthbbvHVi/AN7FozPjZWvgxxbGbsHGTMMWXcyZGs9YHfjD1FCP9X/D
u1Bi2y6bsgtd7sh+G9mvNjTvhpYx7LEjHdxob4DbZGhYXwN73dZjm1l/pJnn
rAD2QnCYAT7XsP9q+FcCv7vC7yXs/6POUrH3e4X9vmK/L2nr2e8bZNAXXtzK
mZ3HeH14dB/fXzH/G/qHOb8N2e+hTMNOgYhu9KO0TGxL96a29Xdwnoc3tM2P
13ih754mjE8Bvx+Y0wfaJ0D6JMYnCibf0xjPYv/JUX/Lxh3qAB/Kfbal87rr
xtF2JvvOK4K3xbS74O9q7MVkeDGF1o/xLez5GrAGYhP+zdj10DgX+BvgwRXQ
3qc9fIC+22g7W5nmPdi/r2hMDRvZcxryL6efx/cS5D+XuT9D/+91TPNe8FtU
YNsnGz8D2mfSvqjnOUdYez39m+HHNmB0Rx6zgLkCWKtoVazN47fNyHq9fAz0
YRJtA+vf4UydLPWdo7vmCfB7le9J4H8O8i4B/zewhxcU+y5pwW/NOE8J6ExN
4L3LmRiDbHcDr34z+zR/gE8hNv8TxveAz+/o+wXg1x/8Xmf/IsbGoX/5ccbp
WXj3GTz8HHm/DQ9HMr4OfFegS82Z/yrfheD4PGfzOdoSeP8WLZX1X7Ln4+w/
CxzT6huHBew1P8/3vWicyPekPMvuc/jVGX4cyLWtFT4fwNTnwac0zj5WPfQn
Lt/+wb3IsCXzP2P+DQn2AVcwPw98MtH3vuD/Nv0nWN+F9XVZfzu6cgdtFvpy
G/rzD+h7Hvomy1ei1UPX2qNfDyLPX+DPHHAfBw0V8K6S9iTf42nf1/Zvs7Hd
A+DBNmR5FzwZwF5/sv9J9s9k/2rG5iHvNSn2UdKQz7vI4C1gr9L5YL+j6NBj
6M5v/HYP83cz/wBzDwkm/a/oLwCfq8A/Gdm2BsZmaJ0Cvp8y1lMyZe3XzL8f
XoyttO4u5M67BFr60fIS7QN8BD/ux348wn0/CPwCvF8DzAjzv4IfW4F3C3te
HO8zWs3axKjPumzKN9D7GOunYX+WQ+9JdGtZlm1xBH0dLt8VffwLfZzK+gXw
6iLOzFzO1y3wa2eJ7yDdPSfQh2LmXsj4nFbWkST2Sqa1ZL957He37kfk/T74
H4S+55k7MGrfIInzF1dgH1++/RPw8yjwfqNdBn6vsn9vvo+AY0v2uqGJfcPP
aWmN7SNex3cD5NWoue+7uvm20bLN0rG90Hs79A4D3iLo7Q6v+sD/JHi1hv26
Q8thaJwDbT8yv5r1H2If1iXa594Us88vX383++1A1hOAnwD8hrp/dPfR8pBn
Eva7DbDvZPwIsFfS5sKvG9gjKc48eAXZDgOfoeDzEPa/Gvl9wH5rGbsEgFXw
p7q7dUk6+j70vwu87DqOCWYh/zm04gTf0etYvzbP9lEx0VbGHgb+GPRjCPrx
JLD7FJqXp3xK8A/lvit68xlfYpsnW9cZHm9ANrNY8zT47GB+ArrZQPY02Xfe
d9iuI13Q/yTf4YvBbwmtEeN/1LZt23Qu9LayjbuVfrJgsN/xhr5r76C9new7
d2ixfWD5vooJ3oHf8+BXvzifyV3QMxN6pkPPPY1s21byW61427g85PUu+K5M
9BlZgL59WuG7dypz7uQ7m/Wj2DsL+V/P9wr2/yvZOtahwDZUtko2qwje5DLn
QDDPXkdeC2g/cRYfR15nFNhHkm8kH2Ep+PQosC6NZM8J4LaB9Wtb2AesBe01
aSOZ/zc6egm6tJr9X4S28ejwXuYvAsZAaIkyvwP8Pyfq2DgLhm/lbN5X7Lvj
Uni0B1rvYv1acMtkPBd6x8CzgmTzRL7nVGCWNvL+nZHvlizfnYpBN7H+KtbP
Z34662/m7NxUal0emGJd20CLJFvnFIvX1Xmo7Zh8ILpyJTo+ONU+/N3QdpQ9
54L/9fzWgPulD/T3hf557N8UWhpHHbu0Yb9t0NIv07H2l7Q/2Ws0vz3XxPH7
fnA9H5zvANfbm/n7G377Vxv/Vg/4x07vt5CWAz65tCz0Px185yCr2ZX2HUcj
r3j4MxIch6Uan++A1QuYw4F1t/iB7PeX2rb0RkYXMtab1r+252xi/l3FzkX0
pD8DeJ/Dk9/a+87MkS8DDbsZHwXPXmLvqbSd8PsB9v9IsRDzX4R3X0B/RLFI
pmMTxVRTkNVd2Y61RwPvdXSrFjAHJTsmfQr8Jlca1oO0DuB6Di0d3U1Hn/qA
609Z9p0/50x2YK/21fblZraz7/wk7Ytk+9CNwGdJpm3zE6xpw/l4Hxyuhje7
4PGlwDuU5dhsPuepc459HPk28jkuz/GdqbuyiP2zMx1TKZZSzJuOfrejXYdu
36v7Gti3Reyr6o5XLKs7QLZf99nrEeccRH9P4I+k3yjT8ZruX/lqc5jfsbVt
1NBq+wC6+6/Fv+2R45zAqVxAc/uGT/TwXS8f8TLoOZxlWpvDk6bQcpbscwPL
qAe6fCP60Qf9qMlvV7Nfd+h5BH36Rx3nWj6mbW/inMs09mtfat4rR3Jm1DwQ
7brzUsHlKegpjLNPX8346Dzbn96Md2Htds7H+y0c4w2nfxetPfLh51Cls49P
06Cufc4Cxqbn2DfrBz9SgH8A+zsLWe+ExldY9DvyLYffm9CJudiCV9j/gjjn
SGRLPoO+dam2KUkVtlGyDZtZvxbc+56+HxSzN+B7OfNnpfqOuSXbOTadzfdo
/+U7kmNfbD3jzcB1KfIpRj65zR0rV0acu1DM1Zf9+0RNWz/gX6fYv8S5NN2h
k5HnlGrHrq/V4lx1c45F/rDu/KHMjQfmvW18R8Rl+zd9K2dyM7h8GHPs2Quc
hnL33IgOHIV3s1ty/vi+DnkOQD5XcCelyPeI2leoVxOdhP64qGMb3fd70Yel
ZfZFlqIPf7JfV+BfqXyTclTMbxK1LdN9swvChhQ5V7WjhXMzaaxPTXWO5uqo
fTjp+rWsaQO/SrvYt5fNel25OPyHX9Nt04vY669s27Y/mjg3tweev5HmHN1Q
eP1rqXNRinnkOx8oda5L52sI+/XMtC28jf2WZjhHpFyNzuRReL2j2LZ3YAfb
8shp+mTTlzG+nJbO2doAfufn2afX/agYQ7z4I9t3hdb8DYwLy5wv1XhCuXOS
p3KR/LYY3P6Gvnzk+3Bj28oB8GNWsm1mHPu3Z33LVOcA9iHbmcC/EPq30t9L
fzr9XnH2mefAj34R5zKeT/NZ6lvku1Rnai28epw5icAbx/i79J/OsW2Vj6jc
Ujtk8FBT55guBJfehbaHOsPXQV+zHOcKHqJFGOtCO6z8TZzP3s3o1JyWPoOH
5OvT7gH+78xfDP1ti9xXTu5s1rajPZDsmOQa1p+nHBm8vQP9msr8esy/KN4+
zn/Q1XLlb8DlDukXsKs6O5enmGIv/XtznCv6mTYHWX1Z7Fj4n7RjjP8as+1V
jk6xyCkfPMXrV7HfynLHgh8rB8bYB/DoWXj1HG03/MljPLuFc06DkP8jZc5N
34h8J0PLFNqiRNuopchzBq1Zsm3YL+AzF3ldAq/S0I2PkV/zCtsq2cSf0IdH
4cFL8iXBbxR7jaQlx/sOiumsQv+5dcyjicC7CX6vJzbp1N6557QK31XKQXcF
v/XMXwE+PcCvMsd3hO6GVGDuQT8e7epYVzHjGcoPoI8fpdtnHUX/4TLnIuQj
7OT7/XznMmUzP6T/Aa0n8p+Bvv4HXo8o9t0kn+4Ye83v4th3MPQuhn/XZPru
2IB+XcBeCRnmXY5yQJzfYZ08djk2cwfn99tS5/IHAK8L+I9FZi8jq2m0btAX
Vbwt/xx+nIAXRcAY18o5vMsVG2Y4168zdW6mdUi6o5xmq3zn3JVrV/9iYD8K
vEfli6f47A/sbN9N5/lq4J0FDWlNHbNeI/0HfmIN+4yiZbvkelof3uauvbXE
vrts9EnGfjydf5eML0a2gwttb89Hp07wfQyetGxneJuR5fwc++ozGlrWsuGy
3ZL5Rvb7Nd+6qhxRzYhjCsUSqzl/Y4E1vbPPUioy/hh484A3Jd45lCpo3Z/v
eE7xlWIX1UBU+1AMU5RhHIWbfKoLY/YZ5SveBj/KGf8k376QYpabkH3Z6fyv
7PsqxTKFzlUcRx+/Bp8/Ojg3uQd5XwrvH4w6ly0ZXsJ3v6jvwiX0WygfVG3d
Hq2ct3L3FbY9ikFeBnbPqH2Hhsg/h7kzqp07WtTUuT3ZAJ195fiUay8vcOyp
nPt/lJuLGrZyNtKtfOh7JcU6pty9ahby35QT3wovtgPv+/auaT3B/v+MWVfW
QN/gcuMkXAbBswsYuyhmXmn8AsYWZ9rXUv5AvnphzHenfPZc+D82z+tDkn27
vZyvZ+vbx8sA145Rxz516tg2p0cd68hGL+jiM6ez9mcD254W2c7NywY9FnOO
TLkx0XhGlX0W+SrKgTdEnn2Q399Jp9K2p3JZbUvs24s/o6FvJ3s2jXcM/VC5
c8DK/SrnciXwL4vZl1aOuyTimoBqAaoZpEccwyh22U7/i4htomxhiWqE6NK+
LI/Jxs9UvB21LBXDK9dUhXzWpjjnNA359oZHs9DXucj3EHN/jrqWRFgQthf7
TtddrhqUYrFsYFx6mt+XM35FsWuryhkrd3ku8FemOIdZWeY7V3etckgb6K8v
cy5JOa4n4e/xLNfylFOZH3OOTbk14ShbP6ara4Oy+Vcg+wlR5+Jkw5WbyCtx
7Cr6bgWXbt1tj2Qzrqe/Jsu5B9VwZhY456pcq+R5tnJtzJmbYB+hreJNzvvC
VPtYqs3WQyd2JzqfcBhZb5ZPiLw+oG2S74zMV3P2l8dbdsMrnAuQDJU77NnJ
uTjlEHchn+n0h7T2nZJU6pqHah3J2LT9rF2b49zZMlpn8MmhLU84XVPm+7kC
92UjvwbXZaf96Snprh1kRByrKZ9yDd/P8dsDwTWSp5RbqfJcxX/rlAuGxl3I
Yy48fiPqmrDy8WtU74i4hqraqXymqfnO0Ss3L5/lHfAdB74Jyq+lOXZd0t2+
umJY5XobVTgWUM5Xudy3wGFIknO6d9LfwvqP0lwTaEL/Y3BaCX4bg2sJv7Pf
jkTXFORr/wjM+in2uVWL/SLXd73uhMbw4swy1+KVE5oE/mOQ7/NNnP8o7O4a
n2p7uuOUy99V5tqOcvryhVfmOJcnn1h3zb4C+76CvwjdbI0/NVu+Cef1W8Xi
ua7NLqXdw93Suti5VdV4Z8dcY1Ft5dSZKHdNWfJWDmdQvu8c3TWXQ88kePNC
nuFfzm+bVcvPdW1zBm0h/edyXdsZT3uZ/tO5zvWNoB0rdk1VtVTlGKQb+9CR
2anWkfxS1wBV+1MO/HvW/1Bp3FUj7V9umyxbqxxLmvzNcuce+sbZl9wScy5Z
PqVi6XMKnNtSTK1a6KiI/Rf1Z3A2vlQOopHfNExjr+mVxlU1pRPI+niVY0vd
2cqlTql0LkI51brwo2OG/Qmdwfn4A/N62FYpZ/MIYzuAP6yRc7ZvIb/FFfZd
5VMoV6qcunLpypkG4CVD37Bk1yw2M/5pgX2jzar/IKtNFfYVdMd/n+UctXLT
ykErV92n2rUv5ayTFJ/leW1/wM9FnhO7WHbKccpX/1/MbwPks5cC/0X6pcn2
SfX2YGKxfT+9QVCtdk/MuX3VbLuUueapWmcG8ssrs88s36gD/cadbdNkyzql
WtdHnc53SeeV+/8o5tyyagD3dfQbBL09WNbSsVWNDL9FUYw1F97Pq3RuSznJ
s5DvxIjPjuD1LnNOTfG61qxA95LoRxMczz8Ibm0yfHaVA5/PXomltm16I9AG
3rWmzanlnNXZfLejLa7lGqvexvQ6XW8SPpn06yCvwfI96Q+QP53tXE51qmuD
rbq5tqAa4a2d7APL91XNWLakKfBWpdumxFh/ZcRvUXIZH1lsGkWbeHY9sOdg
XzYmOqeu2t72Er9FUY2vsrNjJsU6/VJdK25Y7m/VjBWrHY75LYrkfRLcb+pk
W6ocit5KfA7OB5uaH8vl63Ry7K6c8Tfwamuua/nKkcpXuizHd7t8pjuL/QZJ
uT3lTJtF7fPK15VPvanYb3z0tkc5JdW2Ho/Z11eNaw+y3VvptwF6g/E1uv42
8OfEO2bdhryuzLDvpRhMtTzdSbqLVNNTbDq5m2uPilEV63zRwXeNYp40eL+j
o2tvyrH8M8M5W/m7qsn+CKz3ZA/jfWfewH6fZdq+X53ou7v16fcSusPPA/9n
S5xL1psn1c4Oo2+XJbuGVsZZa9vJsYtq4u8D+6cK++raYzO8+bTQtax4+h3A
p1ehfcedwb7PRSXO1coHUq70x4hz7coZ/sTYD7TBcb5zsuBlp5jXXkL/bexD
nRLX8hXDKRZ/Idu1K8XkypVtizhWVM5MtY8jrP87xfbhfPY/jzY1wW9Ovlet
P9vnrW6KfcXpMdeW5TOqtvNhhWNp1XhOIMuTlc5FCMZmZP9Jsd9+dVXNGfl0
rXLtSDXEX+Dda/mOncTDOwtdkxM/OgLvi6hr5rqbpaOqrc6JuRatGqti/1Xs
PzbNOYDfYs7J6ls0LWP+wRK/BRPPjtPPjJlX8hkVG7aK+i2EbHoP5ZKr/bZG
b54+ybGPJd9KbxiU27w85rcZynHKtx+T59hGPv7KCttc8VI46e6ckOfavu5Q
1WZyujgW0Pq30N0ltN1nuQZfH326BRl+WM8+zCrO3s5S5+Zk0xS796pybK0Y
/rB4m+ncmWqYR9j7YMy5Gb0B01uwxSWubav/nmxlgWWhGsoivR2p9N2tN0fK
haSD46fBORGdhZvy7evoTMj3GZHps6G+fNV24NQi1T7rEHhXlGPdvgL7868i
5wyVK9SbkvvBLTPHsaNikrFRvxnRWxG96Xmpm3OYyl2+d5bfdv2b8YHpfuNV
K8M0ihbZ4HXsvbbMtbgF4PAJc8/Mc6wkG7tQ+dQK53JlT28oMo7CTTamd8wx
nfyLYfDsvwWOAeT7qwb4ZhffGbKN8hHLCx3zKtbVm4332HtjmXNl8lmV+9uU
4VyrcoCLu/mO090mGcuXX1NmXOXTT2X85W6mVTJXbrR7hnVfOdJFylWVuZb+
MPTXLPSZ1FlUzWQc8nq83G+N5HMol103w770KR+cscnlzhXL54nAm935rh8q
fyTfQ3eY7i75IOL1uM7eSzz/DV28ssRvNRSzFrP+cAfXUlVT+K67YzTFZooh
7tFbj2rXYvZAU31gx5X5raFqfvL9n8937VkxwLXo0r3d/VZHNQnVUicC4/eG
rqmuKPOdrrtcPmPfDL9plf+hGs8WNt5aaV9UNemfgL04w76w7ky9hSousy+i
/PxNeY6JFQvrjZpq36pJK1einIlyqYohFDsop6pcoXKwyr0qZ6i3ntkR185U
b5fvvTHq2rHOt96eZnRz7UtvUFVrb1NuX1U1d/mOvWLWNdmbrQX22eSryQdT
LfiriGszqgnvg54zyp1bVs38N3hxvNKxkd68/o/vXytda5UNVW0ipZNz4apR
vBSxzyJfRTXSOPCNL3JtWW8mJjD3BHtsA/avrNmZ6TtQ51u/KdbLL7Nvpxiv
P7LdG/XbCuXMnwGf+ZWuFeuNxTS9p61wLV45bL21k05IF/TmbhL7VRU5N36c
/fYBa2In5971hk+2Z0/UuMgGSVdVg1DtQTqrt2L7o14rHBSrxUpdm1PMdi2w
RnTx20TVtFQrV85Uuq71ettXWmZfS2/80suc41JuSzbsZvZfFXHtqVZjx7aR
Mvu6ol+x9uiufvusmPtnaP04x7G23kQq96Yai2or0ifVvtcUGpZq4P8H6Uyl
6A==
         "],
         VertexColors->None], 
        PolygonBox[{{13618, 13628, 4332, 13630, 13617}, {15722, 13618, 13617, 
         15723, 6285}},
         VertexColors->None]}]}, 
     {Hue[0.24675778299052378`], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFnXd4VcXWxjdgEnpoKSSYhE5IgxTSTk4SkwDqtVzbtfcuYPsAFRALNmwU
GypgAdu9Yu/lCtKrIKJYUFGvggURpCnyvT/e4fGP/ex9Zs+eWbNmzZpV53Q/
+9JjRjSPoqhlqyhqofse3QsyoujsZlG0b0gUvVYbRf/W7+0VUfRdSRTNT1al
oVGUUhpFm/R8ae8oOqJLFB3bOopG6PkwPR+t5+NaRtHqdlH0zkFR9F5qFHXJ
UntJUfR11yga3911ji2LovbF6i8hilbFo+iFJr+br3dXtY+i93Xvqb4qqqMo
rjb+EAyP50bRIepjpspn6CroE0VZuq9Qnc9VZ21RFI2pjKK5+rZpkOqrzQcE
wxmC5f9UdqWuDLUxuth93K4x7BKsDwiGQxqiqF7Xx6pbp3aeEExpqj+8o8ar
+13CyWXqe5Jg/FL1xws/r6j9U1V/jGDrLRgWdI6iDYKpTDibo2/O1LeddL9T
336iOrfWC161kaMrS+XTwHO6+6H9CuGwtXBVI5j+yDAst7SNoj8Hqk5dFH2k
3wt7+L5O4/pO+Lyuu/E/W3iYpSum/qcK/oVNhjVR+EjQdbj6/FXj7lijMo37
vra+M0eVVepb17EpUTQ5MYp2qI81gifq6+dDRCTTmBvB1TvBZTtD+e7wzHfP
qO89wuvWJJftCnV+1X2rrtNUJ1tj6TzEv5PVZ3tdRep3vL7r0VN0JbgeUTs7
9XtJkstyQjn37rrOE51+2E34FCy91f7lXVz2l2B+Q3j6Wn08nRlFqZ2i6DHh
5lFdP+dE0Vca74Wq94pwPEztvKbygzRn7VWupqJfNd6zVT5K5Wlqs1F1B6j9
TpqLs1SerN93DvTcM+/vCd/xAuE7yd+dE74dpTGN1HVrisv5dpa+3am57hLo
YbLauX6Iv3mrMIp+0bxcoDrDexq2mwXPEy1c9oGe92rurhUNJQvWpzTvMwVf
iq5N+u5HXUkqK+jg7/mWNkaEMV7TP4qaaf2+qfaOU78NQ4zHVLW5Ss+thet1
aneh5meDEFGt8piuw1Q+qa/vR6vN90UDc7u7Tdob099tXpyveRdstycax68e
rPaEk9NUvlXl41R+tOZlXz+tc7WzEFrS9QbzkuXn+1T+k2B5Iaa1kiaazI6i
e1S+vJv7nKfrdeFtiWh7c5N/v5jld5Q/oP6OFpxXqJ2zRO9fCb53BcvJeh4n
XtBZtLC7k8f6puCp0/iaV/sb+l8QYBin+teIN3XX2o1r7HM0Z5erTgfxjbXl
ft4r+lwc4N+t8aWrnTrNxQzBfolw8pRwMlp9LFKdQ4XP4WrvCvV3t55rREsr
A5/rIBq5VPWfUf12an+3cDRM7a/SGLdojEtU7/RUl12jb9/R+huh53EJ7n9J
gOEcwbxe308TDO11v7jY9Y4UPm9Um5dqLQzRvG8QT0sQ3T4rmM9WvW2i/SbR
cE/Rz2Ea5za936nrL9HSHl1J7cwz+Q3/TBWe79W37YXP54SfdcJDnWBYo3c3
C+b/1Zj+nxDMv2k+hgsH96v/zEp/f7bG21x1RmtdzNYecoXaOUs0u0NwTtd8
3aL5ukPtzGMvUP2dQ4xfcDtM9Vbq/dtq81nVfafJ8LBmt/T1Grtb9Janvl5V
nfa6P6sxHqNvSwRjqa6FWkv13Q3zx3oeozbGNvndIcLD+YJvjGBL0reP1Htv
G9vR/HWt+trSx/PH3L0tuC5Q+/uEw2s6ek/7UHWG69sPK8w/XsoyjUILd2pc
88O4DlVfe7S37lRbDwjf8SLvE4vT/H6b2rlaOK8VXN+xLlTnl7jnBNhrA/zP
iqe9rnl7jUvvTu/uvWqR+h8kHJWxpntF0RTR/Zuay7aC9wjhVOQQZQe+Ct+E
t/L7QXip5mS16Od5wfyhePRqXQ/o/WyNY10Pw/ZtX5cnsK/1cPmziS7/RtcH
+h0L9yMS/P7j8O2pqrdFz4/rOSYamKBxtWRuhbcVunYJbz20XhYKhodUN01z
mqGx5Oj5DY3zTsF7h+A/XWNsKvHvU7QulxSbXper7RW6JgrmLhp3gXAwjXUq
XC8I5cC+JsD/oPB2nOZvt9qZpj4mtzdd3ab2Juq6Vb+n6f5gsd8VCebCXM/L
u5rDd3Q9r7ZzVHZtsfexqwVzu0q3d7PWwU269gm3Fwnxn+p5va4B2jd+VJ1b
9XyfYDhGMOwQDLeIRm5uMp3uFS2+J/r6n3BYorqLKr3XDcw2XMCwfYDoaLDH
U5Tq/WGT6t8gOEsEUyuNZUqK5QZkhgMyBHLFS40ag2D+P8HZJHh2A7PgeVXf
Tlf5VPWxW9/u09xfrXuB1u4tkpHOUfufqd9NGvtFbb2euc9S+0tF24t1vai1
0kawXysafUZ861vVr9C3V6neXWpvSQc/D9aa66TyM1I83hbZHvPdid4b4Mm0
MT60s7qPf0PDi4S3KzpalqLuL1nmt/9V/+/oukd1L1Od4rD2nxONz4HOI+9V
U1oZP5Td08rl0MmiHsbZXsHxqvp8RXM3U7TXV2U3tRHswtNiXds1F82Ek/nC
1R2JxuuigNue5R4zYzxSNN5fc/G52gfwZYEOm2WaVu9N8XeLw7cXqs/5wv3c
ZpYDKQeueern/WK/uyfF7dzW1vfloU3a4pn3h4qHnVHhsbHvQZusdfghOPlA
eCvs7v1hgercp7m/V1eh9qMuwuurev+DYPmneMK5eVH0D+H2Y41jkdpd2c38
+L1u5slXChdzu1kGW6A6b9X7HWudNV8qHB4vnK3Tuw7NLWvvHWgZt4XaztJa
fk/0c3AX88I16vcw0cJdouk7Uy3rIfNdqfbXa0yXVlqO3p3mfpHNrxPfOD/P
PKeHfh+jMW1UOz3U/zLVOUffnq33r2m/bhTsO1v6jqxbIn5Vm2y+NFtwTRxi
+Wm16PQM1Ttd1xOa+6Qk7++ZanOM2v8tMm+i/Y4a10LB1aV/4KVdzFfhsR+J
Tj6usQxYr7H+pDHUJxu/8WTLbNDr1D6WJ5F/kH2AnfuK0H6j4GlI9rpApmEe
HhS+TtW44iVuC1iWh/oThe9Yice1rp/3ANbOmb2ti7BfdNb7LroO1rd9RA+X
q48nBPObmvdZoredwudPklGb6/dslT+pstYay8z2lgOgnf+28jvk3Gz1tSHN
7wc1M33MDuUnaS5fUT+dVfan2hyE7tfKcsotQVZhzS4MbR6h90fFXYZ+Rd08
tfMce5Sup9VHtvjVUbnua5NksB90dUz3nPF+VDPrINAidHiReEPCIaI7zefv
wsNa4WqQ4BmhuduQa5m/tK+/QWd5UXB207vTwYlg3DjQ++nXbV32VuQ9hr0G
GvuP5uVa4WWq3s9LMV5ThMOTxOtOROZL8f48ut579Ej1WV5i2o6E132CPzfd
Mv5H3azjAOOH4flQzfWIavfH+7WhfLvGMijZ7cAXoX90zFzN21B900MwFKvO
s5LPlyVZjkDOQJYYJ5iOEr6OaGk97t9BlxspeDoUmvdWiy7WsD6TrKdSB111
uebnEskwG9RWnn6vV51M1elY6O/5dpjaf0D3DVrjw1Q2R/wxS/P5jPD6b13d
VP9clZ+n6zaNvZXw3FJXgughRe1vVv3jVb9Ce84T6qtI/DBTeDpYV7ForX2d
9Vt02m3aD3/TtVTz11brOF3z9Z2ei8UfLlb7QkF0vmA/r491wVR9O0f7X3v1
+776uD7dNDZc+/CwJq+JhdorFzV6H0rU3G/RGO8Szv+V4edmwuELosOUsPYn
6bl9f9dpVJvHCZeXiJauER6uUPtjhYulGvcyXQ0a+381xrm6VgmH56nuDapz
qeZrq+A6pcYw9VKbqWrzYbX5o/jHpgq3+bNgXFFkverX1pahHlH7PwlX67UH
zRQ93iu6S9B8zFL5nGTjmDqJWgPNU7zHXak6S9XfSS0tZyLfQyPM8zNhrpeK
lkYEnWVLB9sRHoc2BHtbwbdRbQ/sbZ1+qca1RrA9L9geUTsD0BX6eM9hTZyb
7nXxQh/LC8gVi1R3Y4X3HuSjn4KM9D/dv9c1IckyCfLI4er3Nc3J67quyPGa
hY+zzj+ttWyBXNFWMF8lmF9v4fvBVX5mbb6p65aWlqk+6mNdpgI9SLh6XGso
R7jvrmtounFBHeSuCW0NNzBTti58e2ma8Ql/O7+l66w+yPsu8gK6J/sJ7Z2m
+22MI91lhVrT32mdlolHfqTxfKvyQuAXvO2q/Ey9wwJvG62ylQV+Xqq5ODKs
X+SUfkFWScyzjsc372utLND1o2C8Xu/Hpnvs32p83+lq0dXru76l1/h28aQW
ueZvf+i7WcJ/5yR/zx35ZK7m6j1dzfXdF6pzg3C3TOMtFm/7UuP5TWN5V3Qx
SPOaJPxcofc99Xy3+npPOJnW1bJ3nsrG6t39+n2t7uPjfjceGV9juFL0kqN+
elRYFrhZa+om1qpge0+wd8QGp3b6655XYd63LdP1qPOAYL+v0vv3/6l+uXjB
D8LZU6p7Udw88E/tjxvFL6bqmw7iN28PsO60XjxiuObxY90be/v5BuGoWOP6
o8ZrB57Ad/CB8eInY+os72P/2BZsIFEXjw89vlHjGqTyqRpLqvj3LtW9Xd9/
r9+P6b5e1Y/S+6Mr/PsE3c8SPp/U84Xqs6HM62+82pymsgTBkK65aiu8T9Tv
UtUvq3Cb76d53OyDwP+64D9e8/+L6Ge0cLtTz2lq7+y4+TM2uafrvIbvQm4p
Nw+brvmdGXN72Ohm9vE+ckPgm/CriSrL13h+STVO2V9marxpgv0swbMqwbQ8
t8b0XNTL/A9bF3sWNMjcze/sNuHD6ANzgx40W7rVBuHuY+Hz3i7WHXl+Wu09
WmKb7mkV1g/RDX9m745bnsG2hY2rB3QrPA2u9G/eLQ7lyMXIGsjqrP/5AYZ3
1M7WGtsPv9D4oyTbvhL0PinB/KS1aPjouOWW8er/el2b9W6Txnu58LxV4y3W
HJ1SYd0TWQaZBn7+l9qv1Ldnqe/n9M3yRNPAkYLxH5X+fX0X2+kWwENU954K
y0nIUHsL3V6DcHBWnmXEcXpfFbc+CC5pmzlaIFrrK35XpTn6QPxjyVDzqutE
zx/p90V6HilY56Vbp/uH2r6/xr8nIvOn22a+SLh/X89NLS2zIrvmCQ/te/m5
SGX/UL/zQ33sFNgrsIMhG8KrkNm2C5/b0uwjmKS+VqivMo0l0nqcn2C5BR0d
/wS+iVtUb6C+Z0lla88/L+bfkwXzO2r/JvGVuaLfHQPth4A37Q7fPiH6OV34
/EQfD1Zf7eO2K7yV7m9/EUxvBpsKcJ6WYlkPWwRl1MX+0LGbf/OMrW1Wuu1y
T4e9Eht4D7V/vcbyeLrlymGVli3rNdYLVPcC4XYR+p3ef9bMMt9bAYZj9e0D
NcZdmcZyf7n1aWyK2JOweR6tcS1J95rC7zE2+D4qRYcjhv5tI+aOfIiNYafw
cJa+XdTR/Jff8KUdofzo7tZ54NWfixe2El+Yr9+Hqf864XOHvvlE5R/pmt/T
+/oMXYsE8/TwfFkz28BuqLcd7AvVbaMxJGUanjZq+ybVqRKck+L+3Vr4ObPY
sg+2zUXBrosu8ozk1Vnay9YXem2zrv+nNr8VfeQJJ1/qeaL4yyTR83Nqc7Vk
jpkBlpkBnge724fB89eq/73WycLkv+EeiN0A2UPfvtrdPpzXutuP014081aF
dT5sWvBsdMw2gq1O8NaxB6nNn7VfPK39or/auSpunBZqTi5QX2cmW059OtAG
Mjv9oHdgg08pto7O/lNf4b2kKNEw43PhzhjwwTwUnhkLdaaH51f17RSt4X92
8PuHQ/05gmdVk38/mm56jKG3Cp/XicYeS3f5Y4GGL5NufvxQ73szks134Dlv
qfxtXTXiHY9qDNd18b5zrcY0vo/tCRcJV89pLiaqfryj7QLYBE4UHk7S9VBr
2+Xh3fDtlsLTg8LPTcnWldGZkZ9Zo2eGdYrevU2wLdNYNqZbLmKvX6O+5hZb
FinMNgz5uvdKdT3qsMdQH/r5XOO9sdR2ny/Zm1V+WORxjA9jwb6EjIpuuFf9
/4UtOMk2p9WhHDvZnmArQ5alHJ16u8p+13WPyseI568tsC0HWR/76G/Igam2
VeBDHJBunesfuo7Js08Uf2g/9jHBWtzK1y9hvlZqrtKbDPcyPd8R6owU7h5S
2ctqZ4v6/1XXZMFwqPDzQLV12KtU/+oa1/tRbSwtsj/nuxz7IPE/jtI4Rgf7
Q11n7+/44NjrHw0+Ofj6PYG3U/ZIkAFmhmfql3S23+SmBOuaj4Y6B+qhOxwr
2O6ssf6CD+3doe739hzb2l5r6fuM8Myd38i0swK9Qqu/hGfwk699J0/XIq2z
icLBHZXep9FbWHf4gjN6209ymXj6COzUoo+pen9rsm2C2AOLhZ+BTfYZrmtr
WyT2ltnC99eCOepqnHHHnoD/lt88n8c+lW7f4a+q+4XGeZvW43TxwqjSbWFz
wl+K3ekcwToi6AXYsVgv0Ad+bvZDYBrSwfZVbIzr1eanNe6DPRPfCGsNvyK2
pdzmXgOsSWQY5Jlrw9rEN1IQ/CMj+3i+2a8XiXj7ZdvOkyN6+TLPstZrmZZN
kSF3hWdspOmp5mdPJFivLe/gedwk3HYW/p9v5jXI+oPOZ5Xa34OvZ6vWxagK
y3SdunkfZQ/9QPrajAHeD+/XuNr2su2vTS8/347NR23sFJ1crHYqtF88W2qZ
DVgu7OBxFQkfXXpZN5iudjr1sk3zIM1dTZl1VeyWtAHOz802XPg7+C4lwM/+
8Hy194ih4jHLhZMVCd5vKWPPzU733liuNpeI9y7Os7wCz4LHzU2wHxEawtZ4
nWD6TXhc1dZxEcwvc4duQGwF7c9KMm89Q/PyeJrhYUz/p7FeWWobzErR5rNF
3gdYDwd18N40I8iGzAv7dlKN4UNGXZi5fylEt2lcGcL/GNWtUp1KXZtV53vN
XTv19XAzy+4F3axXdBPeniy3HXyY7ifm21/AOmO9YbcZoPtAXS1Ee7/1tkzP
87lpLt+g9qarvG8vyzmntPMzvsiHNY/jS61LYjvKC21mC65L8k3zU9XOIJVX
pxo29IvVGktZL5e3VV/J/W1zwd6yC7+hrqxeXiPcWSf70mz/wP6DTMc+iVwH
fIzhC90PV59nl9qXDq9DlmWNHK9vk3uZ7/2gBjroeWYz01bHXraZUzc54BA7
LG1g1+V9p1C/WV/HVcxrbVgeD/BgK7ojwI8fsDDwsVYZxjPPjJ8xI7dfkG24
gXmbGigNeMA+xV7FPrW1t9vB70YbBQG3+AzgKeAWWRM5Hhl+U0vbo4F5qWh7
SbL12SwJOAfrSs/0mv4k2ev68LC3sa+9RXxBiWkQuWt6kAmXtrV8h/z5p97v
LfH3F+u+Ps862TtB7kVux1+Cb4i2J4k3r+lgfaRrR/uHf25p3Yb1BE9uI776
TYnLbxYv2kaMUqL3VWwi6BgZedZv0W1zVP/XRvOd32psU0P/wia3PtXj2YDM
qXYmtbR8iRyHzp7V3zo/+v5U1f1V+DhH3fVEn6v3WN8SzCs62K6CfQU7F8/Y
vJupvItgay0e3ErXUaLJjWrj60zLaZNzvLezf7ZV3c2ZjrtgvbBWoI1NKvsh
lH+Z5WdkV2wS6PbYKPhuU6jzQ3hmLvCt3NrN8Sf4sImxamhuO+imMO/ZnezP
5t1a8N/TMsld4ttNkv2mqywBXUnwZembx4TbE4T/+1XnT+Fmb6nfNXS2TIf/
BZ6E3spc71HdefX+fSG0EGigNvhC8FN82NP9IputEG7Hqn5ponHzYShPFG/+
vsK8Mafj3/6cv7RO1mXatkwb1EeX/Dluexa2rJvV3i31xsUg8faFeebV2JhW
dbD9E58ePIg1Av1Bh8hy6EXoR9jcsHV1K7O96wutj6/qvJeigwEL+/sd6udO
XXepr2TV7VDm9Zab7brsucsE11GVprmUjva9MR9/CoY382wPgBZ5z/Melb8R
yh8rse0Du0cPfXdQO8e0QXPQIbF4+MLoH1/ATZqfH2u9/ol5wZ/EOoK/wHvP
x/4jnj9WbW7t4LbQ+dHf0f8+HmIdcKb6faTE7/b7FVpZBpuU5XXH+sPG9nSF
7WzI4BuDHM5aZS1P0Xd5g6QvDbI+WgrtB5kYuoF+9uvgwnOzQtuRrtbed1WT
bcmjdD85bhvT3XpfUuK57ir8rBOcCzT3D5c6XgSf/mv6/WqgZ2ib5x24ZDtY
txiaaPkN2Q39lHjGWd0c1/Gb+PDuDpaxbwt2WOy3V2tdXKXr9hTHsDQM8tjY
69nnaStPeHu0zP6iLeIbZyR7/9pXYbkQmRA6hb6Bf1qJ9SLGj8yFHIbcNUN1
Z9YbJvYO7H/sH4+IFl6qtgy7V3v0UaLDM9X/ZvwpJZ7TKapzWJllZugcGsff
+nCJ42+AGRq/Kfifb9c4Jg4yP6gUDu8osa/zn3o+utQ4q+lrnzrxIcgXC4KM
gS1tZrJls6Ws8TrT70ua+y87WN7GTwNtEn/I+qJPZIBnhMsBGsPznWznPzHZ
Nnb8CvDlSWENsv6wZ45T/eQ87xW7UmzHwt5+ocr/6m+/VqlwPj/TMX7A+3yA
uZm+qy0zDNhT76kyTtEJgBU40cPQtaDv+WH/QU47RLiZkWcavkJ88XJdn2bZ
1wW9Q6t/dvYaOklwdko1r0IuPafKsVDETdT2dZzZ+4KpvsxyKnP6otp+u8R4
7FPuGFBiN/AVHBb8Bdjp0X1nCv7mxAYM8djuzbJ/4YzgDzgy2NmIkSKWER8t
a5g+8bMPIy4rzz6B2mzbVPtpDvcM1hzoWis9abD21lbwLZV/kxlsH8JnD8lL
daXer6pLHdPD98QMUsYedHzQH/bHXYRn7IInSPY7WfR8gui5QfyvTbX1cOyx
2H2gI+yUyD3IPPhyCsC/xrwt3/4zbMfQFPHIzNesRts0oYONYQ/H/kkcJXoY
sZTYsvEN4BfAbwTtIiNPUlvXq865Kv9Me+l24WdAM8fv/N7TMTyDSxxXQUzF
t6rTOugmR2rujigzj0CuH9zButsBHYYYBmhhX0/Tw3Lk6Jj7o+0doS/0xWtj
hvV6zct1VY5FuSvNdYBlqdrb2MH+VGyC0Bz0tkvjnlTu8s+lJ9WJzzfvax55
euCT/xKdntjg2DTiFPBZYm8h1ocYO3z1O9XOGTXu4/rgH2OtXpLisjEH2WY5
p5vl+Zsz/bzfHimcH93BsU93CPauoquHUmwDeyPYwZIF13MljgeA/rEjYito
3s88Gh1qsYhrSZ153/2qe7ZwO0Hfnp1hfQXfJXGYf/axrfCETMfqsf/R3oLQ
JvbmCWEfGZPv2DJk/l/6OlYR/z3+UGgQuZH5YZ7Qd+iHd/TVU3ge39b8Hzv+
vkzHslI/6uU5ndVkPy4+XMb3SqnHSBwR9WjzJ+HkJl33qY12ne2vpX30Rexh
yCGjtC7eKTV/e7fJcZnEYM5t61gT4jr+Jxx+LH35mwzbcbFvE1eDnaRFqm0m
0CK2IWAmVoQ4EeIU0P9uLbEOiH0T2wF2g6SO5gvwhCFad03ZlruQv95r8PNx
ZY4xxeadKXy+qfLe2JcGWQdA/i/Ktj0OfHZVnTcaHMcypa/bxfYyWrBvEY3N
bmP7LnY6ZINYtu2LtNkszfwIXvRUkW2I2A+RaYEN3tJf/HKD1vw9Obb/0S+2
jppB3qvYp/6T4TwG2gcH9AHvKsg2rMCJ7oDdBJsJY+2Z7/FiH7gq2/EM8Brg
xVeFDf9qla+KLJfhn0EeQ/+YE3SQ+zQPo1TnSuwwupdkW/6PlTkWAb0vS/N1
aI5tCMCeH+CBHgvCM7oUdt6BwdeMrxofNDGAA0ObkzWncxr8DWXFoXyh+MFS
0VtpZ+sZsYBb8jzwp0BL8GvwDP0ROwBNohMlCX+3FBs3+DPHBZ9mP/XVKdcy
znzibfp7D8pX+e/9LPsjUxeXmJ67qc7X/W1/w7eDrIacg0yInxudklyS+UEf
uUT9XFznNd1Oe0LbatuckNeQ25DfsCGNDHakk7U3vqs63dp5jyKmhn1qHjF6
9Y7PKKmwfxTfKLHI1EUm/Elj7NrkmK6vVPflSvvx3xHOLhA+J6TY9jO0xPs+
eRIbYs6VOLzM9hTWEf5/5oYYgU/Fe58scgzXNYK7LtBwgmjmMj0fr7nLELxd
q20nw7a9Pscy6Olq91Tx5vOEq7EVzi3BR4gct6K/ZbmOmtNV/e0/guavDXbv
pYInp8n63Rcay4uVXgtThO/MJvdB7s5bg7xfEhvVJsd85us8+yzxV24Tb7lU
Y++n5wnYG7W+2qTaH4B+SXnLTPtz7m1ruWd9KJ+ue2KmY5/g++NDHBT44tuT
BfMxavsJ4fFxlT/W1nd4KXLGk6rzRJZpdW2WfR+zQj3iVzeKhzR1diwiei3r
Ht2WPQ9YNoQY11mhPnCjV6NTI1cRr4K8N1X9T6ryb3Rn4Men//Mg29qwsz0o
nNwlWCeqr756fkV42HeQZZTW2baPIbPc0+Cyk8QzXmi0fw78YlNFJukhGA7O
ts1wV4bjvpGz/6nvnm90v9nC84oC6xQnqfxllb+a5VjlD0IMNn6SEWXWC9j3
fg7xrt8MdFwaNq1/d7UOg81lcqpjcNBrVhdYNkIuIv+A/CfW27Qs+02BZ5fa
WN3ovQMd/5Ac57BcW2Y9Ex0TWscWgh6RLfxdfYjlZWKgkcP2y/jIL422DxyS
ZlsVdqr+GfY3sk8lpjm+E7noG433W12X6Pdm9TW60L6juYL5girLsNj8eI/+
hHxNbAEyzzHquyHHfaF80D9++Wf07R2Nfsf6wofEGidP7Lxs23ux+54beHtt
mp8pH6M5HdVgfsteg52FvYAY3YUhNhibN/Gb2L3R/ZANsAPgj5iUY5/EjcQ2
5zkWF98Fvg1ic4gJIn4IPeVijfeiMuv9xKfjSCKO93LR0roG+wmI983NsX5E
LlxusNUT44PtFxswMb/LQtwvsB+bZ/iJgUPXvy3EGxMfTMztoxmWFfj2oRLH
pNaHWHn2SPbH4apzYqpjxIiX/LDSdkXsdXNybDvEV4b/Cz8YslTLdraxHCqa
PKLcvxdqLi6psgxJDBSxUMicn5XbJoI9hHhJeAe8ApllapZ11BFVjg9GV8Wn
fWGl/dnY78h/wAe/qNz7HHvc5VXm3fBt+BBtfnaQ5T30b3K4ZmqtzaixjfM8
1T23yvFkxMwuCzG3h6r+0L62U/cVL62rsGx5ZBffyQch14k8J2J2oGvkMGgb
X/2mOsvm6DHIHegy5Jf8HnOOCTaSu7Osm83Nt78K+oV2DvhjwV1SwOcfmfYJ
E2eE3xA7C3aSfdoHqhrcFnFYJ1Q63gF7YWKwGXaqsO0VuysyKPOJDfMg4bWm
wbjGpoXeiM6ITDey3nICe/v2Uu/v2GjWZ3qcxLMnllpeegA7TJnlR/Kf2FNZ
a6uJ5Qx++ULx1T11lmFfaG2fGf4ybB74YuHP70nOvbXAuUP4eZ4vtZ8mUhv7
qszLyW/4Jtn71EiNqX2p42S66E5uKnyXGB3mhnmJd3b72AyJAd0YbK3YZsgb
IB4DPs13rKmnBOOwAvNN4k+a9bM+Qm4CsQnEz6D7kTNADMCmctvCsIOxL33S
03sTthR4K3wVfQRdG9kamQscIHcBB/scexx3YGPd0cb6nvaLUfZNKCfu4ZNQ
ntDRPA2ZYZVws6wq2GvJ/SqwjjxbzyMLnG+C3Pd1aAe79awc+3iZz2khFo/5
GV5t+NaQ49PXeb3Eih0f9m7WA/Zo9D5ySt8PeU81BfY943cmJ4+8J/IEW6X7
W/j0jN72A6EvzpV89GTctEB+wAuBHtaI7q6pdPznMNHMpIG2V5BrO7zOeZes
iTNCbhoxGi93tR5zv2hyZZHjRohvnxP0mhsqnD8J7ZDrx75J3twrXZ2ji/7z
hmjn9aGOX8O2CD8jrh2dZXJf6xv4jckpRX85XHRxWD/nRxKzTn1iAE4R/KOb
/O1bXZ0/DK0Tl00eKrHZd2tMD9R5DMSPTAv8dqDkrpJa75OL1c65lc5ffSjo
bcSW03bHIre/JsAKT0YGxNfLemyvNjrU2tZDn4+FfsmnXBFyA8hhGtlkPQK7
O/jG9n5rlW2I2A8vzbeNEvvkXulrT1ZZriKuqkedY6vILV4ZNy20U1nbOsuG
5LGRM4bviVxgcobZy+DLq7Jsn5+n+/tZjgEmFpdn4nGhHewTyFrksRKjjW8C
eeiVUI4s1i/Pz8Ry8w6/5GiNe1St9y7k03mh/R9V/+dG0z22eXyy2L03aVyt
yr0fP6T575ZrXotc9kOBZTN8O5Pj9u9sVf3bq7xPYduFj8BD8Osu7Orf0P2L
Ac7jhlrOo53X9d1rVR4jc0UeDvF709T/x/1sD3hZz1/3s70hkZjtKssMn+Y7
Vh7/HfGvwIJPbXaQl/HBEWMyOzwjgwIDa+Fw9f+Phr9l6ycP1AkxAMi3P5Q4
Lgrf9vJK6+fousQUEgdJDCTy5MI080Z4ITyROAFytniGN96qOfm1ys+nB5ss
cs5d4is1GsM9auMq1Sms8Tv0D+JJ0UHg2/Bs+DnxqpSzj5B/hi0YOzD24N0h
9hv/BjSIXPGHYD5b8/Vqkn3ugzI9xxMbnY+NHPtoo31U+Kc+0D68Oub8IfJi
Bw/0nvkyMR5llkfa15smoAfiMbaHGMUhovGhde6DNfF0WBcNuc5hJn95dtw5
/+TYELu6Ps2yAt/jk8bvjy8NnQX6vEPzflqN89Gnal1OaXIs3jqV1deZj03X
80mFtr31C/4j9Av4GfoAPI0+6B88UrY3lJ8qWD4p9x71tGB7Km7+RKzd3mBD
m0asUJ59ccOFy5a19kVj7/otxTyBHAl+o69txpbV4HfEthHXhq3jTLXxWZV1
k8Fl9idhU3tFfb4cd97eZxrTMw32GYAf4IaWjhOck/Kc77xEbeyuso+GmHrk
FuLq2wmu8yrsg/6d91V/27y48+3YOsf9E5dLLH/SINMx/nDGBF/iGf86vi/y
TNBVp2g+jkl3TDQ6OOM4I89jQf9+NGYdHLkSGkG2ZD2/Feyk+HvnhWf4Gc/M
74G4eeSoycJzrdbC/anOpT613nlUuZrzWYWOyULepz4yPz5x6J01G9PYq2sd
n0gu6Wdp3t/xlxzQp8HZyoA3YpXJxcL+vz8HsqPjk8ndfy1mXQk9+LcB1oWh
A+iG8RL3NivEomSp7sOCe3aqY7rII4IXEFO6OMSVHieauUn1Zuub5Z3sO0I+
IO+ZvG72BHI5eH+m2jyceJkqyz/YL7AzYe8q1NwWVDh2H7sm8ixy9WTR/5BC
x+fCx+B1tLkfz2HvmCcYckotV2PHgn9hy3qcuOYax4D0EL671zv+CH8xZftj
QtTn3JjttcTZE1ODzwsesSfNcYzEaO0Oz+RPE7OIPgOvxMeAbRD9AFs0OgLj
x5+OjPR9jfUi5PqpWgdT4tbXySFHD6O/NcHGhbyKH/uvAFtxsOthm+ooGvs8
zfN/c53jzpnTR9TezLjlGnSJL8K8N2rdv1HoPMQXG+0DYH/dNVjrZ4DzvBML
nSOEX/Eh4WpNzGPmjABiF7AlYqMjBx67AXnwPJMbT/7NqyEHZ52++zhmuPHT
XjzUMgpy/dvFlu3Ray/LtzyQLhqYqH6naczXC/br4s7Jhkcnpluv7yv4O+bZ
nlM70LHIyNLUwU6BzYS1Sxus34fFc9LyvDa531zl5wkNtr9je9+ltb+zl59X
hPxZdOMJet/Q4Gd8+63zrNenCt7Gatvf/62xbNQYWwfZFhkX2XhUiuPmiZmH
J94XeCa2VWzZ2FdbiQZa1liHwx6IXw1b5etq79WY5+xawTC+ynaZMcLHNXHn
iENTnPlybJCrW4d+v8K+GnMZsVpvhHgtdHR09cnBPoGdAl8AdjrscYyLHKzp
gUddAG+L2dd0Jz7fgZb1yUEm/xi7BLZ5dAlsBOQ97M9/iOzTw8+HX2/oQOfk
I6dPxX5YblvBaaKjzun24d2UYlkQvZ48RnRu5Ft4GHsM+8tJoR51OA/gfzH7
bg7knBObgb6eE74lz21UwD9tI2PyLTkNY6vsZ8bWjs0dHRwc31Nj/MC/kLuJ
gX+9wTZc7LePBP6H3xCdeEGF9eLlWi9Vpc4BJgeCeSeGijw27DvY2bApXBNk
lf129GBLJy6LWCr2HmL+wQ3fnq82O9baRgHtozNB/5fG7C/EV4hv/NgG8/s1
GZaxkK/AKf5RbMK9VHdctWPL2cdOD3vZol4+D4KcBs42GFBqH8LF+n1Riv16
+JTJTyX/ax15UXHTHPQD3UAz5AntiNnWir9xcvA5viA+tLLC+sZSYog6Wd/d
rrpHVdnH0V2wja22PwOdgFg4Yoe4NwvPq0OMHGcHcSZSua6erJcKx1Jgx2ke
6hNH1yzX58tAMy+q7ReqHV+Jzob9DtvdCsF8aJ39C8hc0DzwY+NMCHzmQcE2
s9g5QdOCz4W4+lGa9yUVXqfYQr4P/o6vMow3xvWb+ny/wXbN4cLDWvX3TVfr
8ejsyF2rgj8OuiYf7s64YxkWYzcutsz7pmh1j/DVSzCdqDb/o35P1li26PmX
avORxqCTs/bJ1+uT7nnbC9+N+TdzSBudgh0D2xjyAHm3l1Z5beyPjQjxNujL
nF2FzsyYLgn0wDphvRCj3iPdtEI5dYh/ZezEmGaF52W97J8m3/dY9XNMlb8h
z5r8ZHKY/xl3Xjd5VsgO8zp5/QFHSegrpnluKjcNMT/I7/D5i6q8x7O/c84M
dMGeAo3gd+V5t3DwSoHh/ayf80/QBVuXOacFeInlIKaQeA7sDeSdogNfqDof
1jsGq0WlzxQjnov9hbxu9D70cmgdOif2kvhU5Jr7q23XwJbxuubtNV1/dHaM
ALEC2EaSQo40NufjNZZdMeOK98Q9U+fCYp8HxLk/cb2viTn+lHgE4hLIDyWv
9+By87zmousWdbYdLy22fQHbwsXiJZfoOkt0er3o9px8x3vV6jkn3zEcyPqd
O9oG0F/w51b7HJG71ecvlY6JIBatS6gzvoXLiJGAD9M//BbYGQOxEJxJhnzA
PoVOS//IYpxPg9zKmQPYoMmV4bwLbIdDy20/7FntvEdyHtfovlpXqmi8tsD5
FeRWoLOODHrracLhKVXmn8TYEmtLTC4+NHI34Xv4E2aW2afAt+QZENNLfufb
da7HmRP0j+8bPntPGCP+Y2xt2NnI7+lfbrjn6bv34TnC+StJxgm4IVea3Hhy
56FHcg/we67sb78d/Bl5D/s7Mh97LXNK7Ac+J9Yr5ybRPzHFwEBcPbHg5Juw
rlnr8J/3hJPEGq+X3I7ORSdel/hZ9GT2HfYfYloPD2uL9Uz90ZrfnZXGFTbC
a+usR+A3wn9E/Fsv7YG9dT0hnJ5abfkY2Rjb4Nje5r3EfWDPwx4IHtlf9utN
af4Of0VXrbleA31WHDnibUKe+I2i8bVlzj35l9o/odrnQ1SKj+7u73Ni3hI9
/zTIOGHfon38XPWi66sF27jePl9wcIi7Aa5xATZs+NgbsRmyxuBL7FPc4VPw
sWK183aZ9310DfIMyRMk5hfdhnJ0e/ZbdCtw+G0v4xF/yh9BLrpP+Hu5ybig
zxmhX+BdFHBC7MzS0G+DYG8c6rYOfAMOP5Mcvb7RPss1jbYBYf/Bh409n/lF
7kYWZ30xXmy0R4e8f+JjiCc5WjisqvI+QSwRcgwyDLZObJ74iNEf8OejI2Mb
+ynYx8ZqXlaUmQau1ZycmGf7JfGQ9MW+zLllnAVALg/nZAEPeyt2Seyh6CMF
orH+McdKnNxgvyZ2Ouzu5PoSj8q+xP5EDCm5/cRYaaqjItHm/Fznxhw4hxE6
yS/wWQDU3Ry37x9fPnsXexi8BZsLz9hMkMGR85HDL1T/5zfY/viS2hgm/P+r
t/XRW4JOyrkHlKFn/ay19aZ4R+uuztdExmL9sjbvC+uzXvBcFfPvA/nh7Mvz
VHaj3m1Oc+xqaZ39lbc22OeEbIgO91nQ4x4ocZwTsUpviCbf1DUnwzwOXse5
dq8iw2h/WSx4vtX7lQMsgxBX+58Q37Iy12d3Eu+CfE7b+3OTsyxLsV++HmLF
sUneQNBulXNO8Busr7Xv4Hfdd+jammG72pSYbWsPCYZawfBiV5/z1z/Vtht0
C2Jk0S++rXb8IrGL8KHfM2xfZazwA/gMcUDkYmFL53wj1hF7wVf6tmeuzw3Y
WWv7LN/u0vPuWreFPA685OthJwVG5FPkiEXBFkFux0GZ1gc5CxJZGTkZvQ+/
IPog9gvOqsCeeZ1oeUm97fvY5Dk7DJs7suL+vNUE2+E7h3Jy4/mWnLZjhZ8b
Y7aVFKmdAYMs/xLj8OcA6zTULQ714TFLQz4zPgT6xE7/lMa9ocx2A9ZbWigH
N/Aa9syXOJtxkH36nAnyQoZtsxtEV7fl29+eKFpLqvP42bugI3xF4wTLf2v9
zQuan9GDnCuUr3vBIMti6MLNgg+6U7r95dADexkyFvsZ94rwvCz4rPEvt1af
vw82fifqPnaAc+zwgRHzQIxv+zyff8dv8jiI3ccez1linw+wTLFdNLC12vox
slf3UGeMcNOv3nHuWSH3ir0M3WtThmWAA3HwxK2Tr745w/IJ9oOby2xDaCr2
WVH4767S87gyx7d/of5/rnVb3Zt8ZhM5dRtU/kut28oW7Dm6HhYdZajOqhrH
F6DXdgq67Wj8pw2OG7ithc+mZP1OFo3sqvRv1hJ33vMd7YFr1jBrF38xZxhx
lg7y4iNl1g/RDbHjIMtgg/qv9uR3dTVTvQbRW16N90li+fkOW+WGOucnYC/e
ofU7oci+sx8L7WPDv5av71YV2PaJ//PlYq8LaHZgsPUNyfNZQvDqeRrf3Aav
Y/yI5H2SC0OuJ/ZBckWxeX8S7N7Ef3EmL7LQ4nBOGjaFXOGkX8zxF6wp1gd7
CrpUfr71qfPyfAYZflXyhHaF+JlpAY/gELn5lSCDrZGctbrcvt8XhO9/ab5S
ensfYP2z9u/F793gOJ4bhdsJZZahiF/lzj5LPB7fEUMF34Tu0S/OIs425v6g
9V3BjoQ9e3Oebdojin1mKOeFcvYh+j42Z3KvN2Y4V2Kt6Oq7Wv9+ssE58PhV
Lo75XAPONMC+xB6OLQCfM+cqQrf41cl7x7deIDzdUu1YY/BFTM3xav92lR08
1H4b5ItDQzwDcQ08I5+Qo0fOHvl6Wzv7/Bps7ORcErdMm+RpkatDng52hd75
ti08qvuoEsc+Hmg3IcTfEteLz/u/5db50aWJc6ecuFzyg+PhmXo8gyvi88kr
YKwjicOpdgxd9zrH6xO7RXwfZcTUYT9+t9w25E619n3i9wRPtEPc1IHzKHlG
B0UXBX7seZwvif2Is744P2hLOCeMc4Twh36g9bI67rOrgRMY0RGIRyA2jpgE
dCjqk2dN7h1xH/iCidf+PcRsEzOAfezByPLXC12tcxCPiA5KHgG6MfDhqyWu
B9sSNuHnNIdzh9pfx/maxMBxBgHnxwwu9vOvcZ8Tyhmhezr73Fj8IMCAjoF+
gb0F+wp5SbXCU7zCvzeRF1xgODmzmXO8kPE42+aU6nC+TRvHAkEbx6ns+Gr/
JqYV+wT24UJs7DGfecn5Br8HfR9Z7eYg200IOWHEY6Pf4rMBTtYhuZ3E1MGP
mENipxfkOUeLOFJyjYndJ26fXDpsV10CraJ/ontepv7jjf59ao3jGokVISad
8zuJS0evjRVYZ/sw7vNekWGJiVsefBbw6cYC8+ifK+3bQE7E9o5MhWy1WeVF
BZYdd1RaFkQORJ8D5+jpDcLzIbXWh0tSnYuCuQtdnveUEWM4Lc+y3ichBhy/
A7mt2BbI6/yh0vgFj52Ejw+G+DxjZClkKmJXLtHYBzT6m32VlkuQSYg/xK97
INeYOFby3MnFh3fDb7trPntU/y3PcccfR/4Z+WbENhM3Qn34PDE47UMcDvLj
0QWWyfCpYd/EttlX7fXTVZjq/FRsJdhJsAEgryOrE5sDnyb+h1wN6pKvMQa9
odF6L+cH31TkuF1olpxD6Bb/EHk15NQkEGfU4N/wWmCA374S4lDwxWPDy8q3
HY9Yty6pXnfgBhwRnzladb7Ks/80SW22rHY98rCBhdxwcompDz6X5Tk/HJxy
ljRzyF4PPygNPAH7BWd2p4S9HBsQ+/KMEudukbd1ILYMuYWzBl8qdDxdYZnz
MdAd0UP4FtsRtvkFxbbPf6L6iYfYR0pOKe0gQ4LzW4Neg56O35gcjR+7+s54
k0t9thG+d/zMs+LWm7ADElfTJeSJEt+BfgdvQA+EPxALSn4s8aCfC84nc+3n
Qh9jnRFfjY56RIAHHZVvqXOV5urqBp/Zytnnbxfaf3a8cPJErnnUMSGfhLaQ
55Hlsf9z/hB+BmS2XtXO48XHR4wt88L8TNc4ZsQ9BmwDnPlBPg70QL4r8SSc
H3N13GP4U/Jqi4E+z/juBueXkq/1idbuuwOsP5DHkJ5vXwU+PfYExkQsfWGG
7Qbs2fA7dNh5Fc5bQG+YrXt5resRg8P5hcSAocuRnwlPBC5iWYhjIVc7P/in
2Z/Zp8k/QH+mDuPAnr0m7nkZr/XSWOmzTPbV2D/B/sE+0rurn+mzIPQbkVMZ
97vlqp9d6NhV6JxzEBj7pgAT8HAmF3Cis3+MvJTrZ2TJLXHLkwd0b/T0I0Sb
fwywzQRdkdxj9s0E4SCp1uPinADGxriIpeGsIGwOrJeU8Iz8S5xuZqDjWwJu
82LWedB3OJt5eqn5O/YrviU2hv2VfrF1dBaMXXTVdHXcCHY7Ykc6xX0uDPTB
OdX4lImj4dzcDYXet6fhk4x7zya+mHhn/Oatw3m6nCMBjdE27QwXXCMaHL/e
VfQ8LNf+qRvVxsIKyy0DBHNRsXVN8kuIPUYGfle4vSTXtPSteENBtfcM7CCc
W0Ff5B5tDb5F7AvYmMk/Yt//MzyjD3CuE/aru4NMwD7L2VSchYLuhk2Fc8OZ
tzmlPiuN3DvsM5ytS1zTjnA+FGucs3tLa+2L/6TaZ8Jyriv97wn9kgtFrhN2
wsc03sfj7qNHjc9sJecAfzKyJrIEPJ3cJuinuMZnHBPvzB7GfsHe933Mug16
zYSY913o4F8VPhcPX/0X6md0kXnbhTHLNMgz5LWR04YsgZyCPxuZYX2d9yf2
m7M0rn9XGKbmIeaTXBPOCWgodKwxuT7QGvk+yF/IYZzhtjNmOz72Rc6hJ84Y
HQ2ZjHroLNT9NtS/Jub/HmCv+LTKOQPEaiGjfhPkOmKkOReGueZMDs7mIG6K
XADaoe4QjW96vvVE/GH0y37BHTjIbWO/wnbMnoUfjDwJYsLJA90S4HxM8DwS
cwwMZb+Gb7F31YfxElu6IsSX8n8W1KENbDvYZojL4ow69AF0AXwVwMw5GHeW
+Mw15Jq64AeH/vFlfd/g3+cF2if2njP58MPhg8NntjHgDTyAgymceVHts/45
7wVbEWdNYi/C/oZfGz8lcRTkGJErRE5h8zCn+NuxK6AfXalxv13pMz3QQX+q
MRzk6CSGOsgL26otM0AX2Crxo/FMm8QG026zUE4+HvIM8RuLJaMtHeL+J6qd
ldU+s4MYQWIFWWfEkRJfih2M8zU504pYIfZF4rqIveSsEWKF7k21rQibEfXB
MbgGL8R2FVUYZ7RHXCr5cZyLQ745uKrRXAzNd7wtbRwe2omp7M5qn2kR1/Ok
aueu3av7kFAfPG3ON64444R+mV/Od2FvQn/kXBTO3DkAK20QD3Y4fskG1+O8
ec6RYU9hLukTefJH8brSauf6kjNHPBnfHlXgc7iwX9M2fXCOCrEE++2xCeE/
SvrZVsbYpwR8YpvgGT/wvcX+PxLkX3Kn8mO25R74jxO+5aysw/Its/HN5IBD
7uCKevhniIHCH4F+GQu6KjwVmLFv4O8lVpu1/1Sxz8vmvEPO6+XcK3JGiKWa
HuKp+M8e7Av4M5kf8A7O8cvFw7kY0ABjB7cz9L5gqOt/WuHz0OHXL4oHvhQ3
bRWEMzg5Z4C9m5w8ZAzi6dEb4C1ltY45Ju5ht2hjWr7P6rg/z2etEmfKfyxh
+9lvQ9A4Hg7/C4A/DNsDdkjslOARHCbWOlaVOFX8Tpypwn6KTx+/EX4Z/E51
BfY9ESOCzkqOwwD8e/nm18iw+M+Rh9eGfDDiLqBlcoCgZ2Bl3WPDxI+AXQq7
CvZ94o/wcZPzwJlrxArih50cbEScUzUt2I6gI2xw2Dk5J4AYTNQabLDbg60A
//NtwQdNLtKWgNutlbYHYQuqKDHuwFtppf+zgfPssBGg32Dr4IyoviXmG/hu
wAk+FXyX0De0DRzYAuGHrC/OJoJvjNLgRsZcj72InCr4M+uRs4dYm+iMnBHE
OsIORd68ivf3wRwgtxNbsTrMxdPFzj+hEjhhvwAv6HbgAf0OOa5fuWW5paK9
ZdWe1zTxyyuqHKPD2brEMRDD8Dm2gaGmvyV6vjnfNPD/NScMLA==
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtm2d4lcXTxp8AoUtCC00SCKKENEghyUlyEkgCsfNHsfdKx0bvCkiXKmAB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         "],
         VertexColors->None]}]}, 
     {Hue[0.31644530115020136`], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFnXmczlX7x+8k+zKYmXvumWEGg8aMGWaG2e65Z5gZRmkjS+pB/dJCpdVS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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtnAmYz9Xbxr/IzphhjDVblmbFzBjzW2fGzFjSSrYWVFRCWhSyFkplj5Il
WtBGiUoqouxCCSWFf5JWFe3R+7nf23U5l++Zc85znvOc7Vnu82t647But5cN
guDMBUHAv+C79CB4sTgIWjbmb+WD4JacIMgKBcGQ5CD4u0wQ3BgPgjrtgmBX
YhCMrxcEqZSlkW6jfFKVIHgyPwjmk47XCoJxlI/sEAQNw0GwomoQrL0wCBYW
BEHHNkHQk/zxhCD4nrIrs6FfKQhO1w6CCbEgaJIVBO+Qv69GEMyifAbpKcoW
k85S9wHob4d+BvzcnRYEI+jjljpBsKZcECyH/g3Qvwf6J6B/D/SSoDehSRDM
rQtPjO02xtiNsZ1uxDgo+486zeG1JelKxnIVaRv0qyKYRqX0xXj7JQXBF9DY
Tv1/qd+Iuk1Ie9oHwV5SjPrp9NcCXnfBw6b6yA8efiA/iT7/aeTxlUF+Aeme
cu7zHLSK6eMstEdVDoLy1K9AalXLdWZSdg/1Z1O3a9MgmMjYJ5F2Uf4m9Wsy
lonU78x8HGQCf0Y+CYVBULkB40CG8+ClFHl0g5d9jKEGZY1o8zN1q1DnEdrW
JX+YfDdo3BMNgntJTyHPXhWC4M+SIHgJHj9Gdmvg8SfofQm9IdWYW8bTCXnM
onw4Y/+B/hLgtWOWv/Ph+T3GPqYIuTfk76ypG1sHwUrGNJqxPE2fRdR9Fnnv
Y/28Th9L6e+VVOQPvRvp72r4/Rqap6FVAL/dyI/JgJ8UxEg+p20QvAFPLyLv
TuS3dQ6CHaTVLYOgO/SKWL8LSA+yllaxXobR/6fQOwG9k6S384Kgdi5rCPlc
VhN5MZbu8PQX/I+Ch/HIZxxpP7x2RubTVE7/z9L/GujfAr2arI9tyO4w9AqR
/33I7xnkF2eNjUBWuxjfH4xvBTw+QftLof9rJcvwMLx8TprLXH5L+ynU3QD9
5rR/lzbfxC0Djf0M5XNp34X2p2hfnf7HsxaOMX9tUrxmD1L/AOla6B2n/gvw
vpz0KnO1ijSP77HFXvujoVFA+7H0t4/2s2ifxHhGMD+tmZ+2pD7QHh8JgsmU
hxhPDmUb6L8mtH9ift5gvtaShvM9G3p9KFsEjxcwtgAeapPfRv4kvK6k/Dh9
f0XaxNr6GZox+p4O/e7ku5Fegr+lpFfg9TXSduZyb2YQlGvoOX2dvlrB0xr4
v4M+B0L/Rejvres1Kt6qMJ/HapjHYvqaFPbeWEibKHV/g2afi4NgDjz+Qt9n
SBWR9+f0f4ayzfTXHV57UKcA/mZFvBeuJiXGXEdlPaiTifyydN7xPQ35PIV8
niHdjuxSOQ97UXcVc3qEufyZMVwL749DbwFjL6L+Itbm752CoH5zaFH/U9bS
DNb/Y9S9m/1ahfoN+dte1tdU5HEt/f8CfzPobw1r9jbWxmDS7Fqe8191lkE/
A/qJjL8/a7Gv9nRF5pAxvoN8fqJOVcqqk+Zo7ZHqwV91+q9C38uR2UXsVUQY
fEDdEDy8R98nkG9j2u/ib6fq+ox+kb13c0fWZDPOZ9Z3ZeS1hvbtaF8beT2G
bPJpv4H2X8LjGO190lH6G0P/G5HPK6QF8PY48zuQ+X0efveQr878/ot8akLj
DtpuhZ8x2ouMvw/9z4BmCrz3g15KU++hrR18Buvsrcp8PwQvmTGfBefI50Fv
K/Sq0P+tnAf/UPdfUtXzed01R6A/uK7vnGrQnsbfurRg3PAzJ+YzQWfBO9SZ
q/MamUzl+2nqz0J+30G/TbLPm9eRz/2UH63rM2o13w1ZD/0oL8/4U6i7OWbZ
psJ/H/i9hvQmvD6MfPdRt7zGxN15Ltm0SvnbA8mm+S9r/Wlk1hfe+C8YSH83
kX5iLq+nvyv5Yy/G8xvzcZr0DmujBPlei3z7kV+J7KvR/5X0dSvyXwH/CboD
oX+4jMuehMbVTVxnNrwtZj7HwF9l2t9MX9M5Mypzdvalv/nUHYpMOiCvnk18
1t9E+ZimPvNnUvYYaUlty3ca85dH+TJksZR0C2u5C/fpVuhdxZl3H3UL6PMJ
+ptG/X70/Tl/O873V6Sv4e846SpkN5E52RH3mayz+HPSIco+0x1L+XjKjyKL
x8OeqyF1Pdcp7K+diZ7zr1kfZ5iTEON/j/Evo243+l/ZyGfSYM73idBbrrOX
/fsB7ReUeKzdkM9G+t9Aao3899bzWXOQOkMr+cw5TjrBfj9xEecG81UTeV+c
b96GkX+e/v7jzqjVwGf2OvpuwJ6KI+uI7k/mbgBpKvM3kT2dwlzVIV0Lv0Po
80baPs+cVKlpHegAd28/eIgzH29QpyZ1a5F6ab1Tf5vkS2pD/3vpL8L3Gmh8
yFz2JH8hfTUgJdJXju4kmK7BmjvO4VCD8fXk/nyVOR7H3C69wHtzAuXFid6j
A6E3pdh7b0gt0/6P8vWJ7mMjsqxOnWGUfUX9bvRVCRp309ddpCzKRzLnfZF1
D8pfYH0kUH9addYlMjrBWDKoP426CfB7jvZnSf3Q3VpQfxCy+JH1cKqK1+h9
zPVY2qxi/EXUOcv3FbRfSvsmtD9DvoD8QvL1tD5p3xKe85FlImPuzl5ch84x
mfl7lD05Bt72UWcXY/lba5L5HAWNFdCPQH8G8vqQMfwMLwdYIwPI92QMRfD+
EX+bSdtU6Ieg/wLn/T/UzUY+ZZHPhch3MeVdKO9D+ReU72H/7iUNhv4F0N8E
vW7ke3Le16f+IMZ3G6mA9lHKjyH7jyk/RvmtrJ+nWQtj0JnfRrdsw34/FfYY
xPs/pFPM58+kDrQfx/ldHX7ez/Ta1h75PWobQLp/E/hPzLKOLd16Dql8ls9E
7a9HyUcoW1ZiXU57oCFtf6TOZbS/VPoXdZuSEpj/jbQZzvdi6l9H/WupfzP9
3UKawXz8whrohfwryyZAHtOR/wPIelLIa7uINX6asl/Dtg2kD02lr53w8ENd
31lVw9YZpStm0t8X7L0vSS1o34w1Upe1vJv6m+j7IDycilmHk+5WDX4fhF49
6t9byzrNo+Qv5szIYG38SP52zq4qnBHfsD5/qumzbxf8/1rHZ2Au9HJIE5Dt
bfTxMHv7S+boZuZmFX/7JXZ+j9FXAukz5HUcGp/T9hvyY/k+zt++QHe5BX7H
MlfXMP7Z2tvM9w7Olm2kn+h/LWt0QNh7UHvvLfbnJOj/R74Zuu3z8HsX33X4
WyHfveCnAWNrSBpJ/QeQ9zHmfnbYttft1PmM9X0T/c2jv3r0d5j8LeQXkW9Y
1rrTQuavTxPrUGc4fzZR/g281mC+jiKrG8g/Qf1k6m9hLF8xpkOM7yDj+4j8
XtI1nNdHyT+GbOaSqiKfpfDzHLQ/yvRZ0J8+boS3m8K2RTpW8tp4hT3zbH2v
kbLI5wfk2RVaT7F/RrH3yrb2XruVNdAU+W9gzjIZ7wT2SCPy2/OtD4wmX5a9
MQSaE6C/HvnVgbej0CuRbkmqQ/0S9ldX6t5Buov+lrJ/JzDeCsx/B9bTV5z5
abVs442j/3vgf4XuG/pfzPcnrI9/WR9hyruwNr+JWTfoQ3kZ6K9i/G0Y/1T+
dkfcf9P3NNJZ6hdRvoDz9WXm+5qIdTDpXis4s3PJf59uW34Pf7uXuqP523h4
PUqbTL6PU/4j5bNkv8H/XMZ/BH5j7KHbOUvmM2ffU1YHmV2RbZtdtuw1tInH
bZPKFh0IP38xt6WM6Rj8XkidO5BV9VLrTrWpM0C2NWtiD/rtfGT8eKl1UOme
vVgDI+B9BvK7gP6egN/JYdvQ6k8+gr7w/yDyWoG8NsD/h8z1W+ifc9A/32fO
T8R9x+luk804mPxtcdvWD5F/hvNvIGMewPjzOP8e0F6CXkXW00KtN3g7wXgy
aD8HHmsw/s9ZH12Qx2zu5+7plqFkd4z0DLIZR53myOZv+JtI+/2078NYJtN+
OrLYCv2f4DcNGVRnv9yKDOsyNqoF4+l/nPQFvufR/8/Qr0H9FtT/F/lOht4B
KibBz0PQK8PYypK6w89HzM889sOV0NxX67yOBu2m8PNJouX1Jm3Xkr6t6D5H
IstRYdsSE6rZN/MOMnylvn00O3T38rcHa/tM+ADaremvA+3P0t/L0OqN/P7H
3P0qmyPiO0t3VT9kOpvy3xnDJXx/QXlb1lMMer1ku7B/8vkOka4m3wT+0qE9
gzk/muI7rQX0RkPjZHnbMHG+C+AvUt/2aYT9EyO1Zm9shf9ByO556pyibDI8
xBDkE3GXSWbF9PWN9BfGkkZ/K+jvsPYs8j1Mf1OYv/rnz3PpZNJF17P+7km0
Tirb+Sznf9J5fex2aA/Rmcx6eoTyHpT/EfVdGIL+v6yNR2jflfYPsQdmki8f
sW1YUfo281kbHgbD22rms2qBfWjynSUwxuXQ3ggP7fleB/36jLUhqQ79L2dM
DXR2kn+IfF/4r5hpnUm60gLonxO/YZ/FOsN2FtjHIt+KfFIr6D8p5LvjNfpv
wPjXU39zbZ//F/OdSroceVRjvk4wtw/Cz5/wcgGpbbptBtkKlRnTKWjF4OGw
7D3d+bT9BXntYT/Ol75O27uofxt3cxF9vkv5nWH3pT43M97rW/vsP0Caxdja
t7avahxjPE1/TXNtG/1Hf4Npu4X+wnV9Js+n/ZOkGOWv0v4R+NnI/LbU+cD4
XyR/iHwh+UPke3Pe7Ece2bJ/2aOPheyzlK9yc7J9QQH8RprbJ/QstJ8hdYX+
G7pD+F4kG4Nqq8k/AX/5yLA9938C63WIfE/QPAat35BvJeRXmZQlexl5vkbZ
F0X2BcgGHgw/38HPJdBqAT+V4PVF+Pm5lnW6pYx1s3yKjDWAxv3c7bM48/tX
95xUpP4W6H9DWQ/qH2MtHCU1Rr7NSN9AqwLrrxZlg9j//6PspUyXaQ92YX10
Im3XXqlvX9J10JtY1z6lVoxtm+x3xpdOair/LOkDxtJJPsES26yyVYdx/vxH
f8PgaTNtSxnfTObzPviNw+/uJPvebm9tXUY+uL6Mfwzn1UVJtuFPwN+WbPv/
tGe/jPrM0FmhM7oc8j3DmXQ967tDc6/Vo6STtb1mV0P/tbhtqc3ylzH2zvCz
CH6eJPVG/s/Tf1udJ/C3UvYfPCay1lrRxwvQfxn9uCnju4Q1/jFjbVxq32+c
NfAl+YvJHyd/aWXrUgF9VEuxTtUk32tCa2E0/NxHfx8z34eZ71forz+83UD6
H2W75fPL8hrQ3G+Hp5PMxTcF3nvSMWZSNwy/L8qXR5qOrF4O2zf2B3V6F1gm
ksUe6L9N/jL6nJLsM/0S2T/I62v6LyF/B7Reo7/76a8n+R7w2os5PMv+zWF/
vQ//H+j+g78BnPd/om//1cG8SSaz4Kc84znVwDw9wtw/Sior/QyediHrYczZ
4+y3qtwZa1k/8/jbL8hjIevnEca7Axme5ixry/g7wt+v8LMHforoY23cOrl0
8e2yL+Vvhuej5MOyOWk7kP77N7SO2Z+yFbQfRfsetP+N8XRgPo7QdxnW+53k
vw17bqWv9GW+ezLefxp7zK04q6bTx7Ik23gjqP+TdOZKtjmfSfcdpLsnhTF8
jS5wor3PYsUQvuf7O+T/VivoUqcF43sZes/AyzLtD+biaJHPAt2Zn2T5DNLZ
UyobO9M+GvlmzjS2r00+M/nK5HO7tMBtVLcj6V3kc0Vr24K7kc839D8R+Ver
a51gOn0do31z5DWDPq+Cdna6z2r5gFZJ1nHr1lHdMdS9LuK7ux7yasPYS2P2
jdRBBktovx0ee1eyTbZO+hbpavbfdsYzubXvJN1Ff0AzgJ8ypE+Yqy/o82nm
/4Xz/tnnKW7M3r6N9TGb9VGB8qH0/Tf77xDraRg8dOa7U6Ftc50h38rWbG3f
xhFk3ouxvgZPt8PLFHh6jLV1EBobme+34b8lsuoHP48n20czkvJDxfZ1zaP8
A8bXFXnsR3fbiXxvbm2fkXxF8mF8It0k3XuhP/T3IJ/Zhb6LOnOGXZThmIZi
GZ1JVyOLbqQj8Hcp8tyErAYpZgDtbPr/kHzjmM9K+egW8t1TMSRop7HevmXt
PBmz7ri2lXWfMaQttawDRQvtE5MvTDpPDNnFST3r2Gf+GX3fTh8vo2/so/80
eI8h/7XYHmuQVynlp+mjOzbISPjvD//baH+qjm3QLfC6g/r5tXzH9NTdTEqU
r5b9fj20N1Helvy/jPdj2i7gb9Xg9y5kvAH6Y9vYFy2e5bs/R75mA/vLNLa/
Shw70hiXIevC1tZt3iJdxdi6FzqWI5tWvpXp9LGmjn0s9RjP1LB1CcWo5tF3
I+a3P33/SZ31WY5JKBaxmvG35Tu71Lbz6nLoday/pxnzOdbSbVqTfP8cs++7
MWvmXcb2PvS31faZtw36tSnfneg1fwf8TIG/u+BvOKmT9g99pmmts172Qe8n
1sOIxtbxRzC+e+M+25bUdaxtQdhno2Juo+H9E8YfhtdXGcM10P8u7r3TH/pt
4OfFsGM1Ot/LxmzjyrZNg97DjKcrd8DviY6npELrbtrPrOcY4Rb0g0rssZPM
9Qbuo53IcjHnf0GyfRA14Gds2L5zxbTuJl///P49w9/2MP4M+VPr+gwcS345
6/EnZPMN7WeQ3wLNH+t4Dd0I7RX5tgWu4L4oRFb7aP877dsynoPUPUC6sLzP
k68UKyGl098u+F3F3n2NtJ75qQo/k2ULMabF0rXkP2XdZijmqXgyIr+Pu2h0
B8+1fFiXY19cQUpJso16J32VZb0Xch/uSvbdcIP8C/V8R/RCNr2z7Tu9EZ4+
ov5e0rsVvKYPIr9C5qQ5stvC+jtAvh71L0rwHb8DeS2H31f53lXevt7n6GNY
U/t8w/DakjO4gWw/eG5G3aYRr8VU2qyLW0aSzU6d38znMfr4gr6eCjyXF8Pv
rHqeU+kqm6B5QbJ1llWtbUPLds4mP4qxFhX4LG9In7cUW8eWbr23kW2JJ+h/
YYptikHMzbOZngvFLC6gv4PszxlNvKZfp+wv5NGUsjB1tra2DiPdJb++1245
8l2SvYZ1thzgb1ur+IzpiKzy4HcqY++BzFZJP2SPvJTiM3sk/aVRf65iT6Ru
8No97L7ySUfg75ks61Z7OQ8eDTnGpNiS9PlyzM2/2PuHoPc85ZewXwr42xbO
u22kGLKPksLIIkK6AX0+jIwj7P0M9kwp9aPU/4W225Mca30WnhckOOaq2ENr
+p/fxDGI5+H3hbjPKtkAJZxtBR2NXVBMvaNi/9D7tZp1mvLQ2lHo2PpX7MHO
1O/c0bqLfLhJ1N+NftmXuZkD/ZYx3zm6ay5F5tPh994M+z4Us6tPeV3SNZR3
oPwy9v7l7Wy7b2S/jKf+Qvbg+wn2uQ+mbWPK9yUaEzCT8hpp9r1dkei76h/Z
QE19Z02lPIHyn5DtpYm+O1JZ46ur+Q6JsZ4uSHdf8glspv4HpN/Y6zdXty9r
KWNelGCf1hTy78Ud+1hN/myuY1SKTckmOUZfDaOOJdyLDN6nrB78j6D9fdTf
wV4eRp2V8Nunpu+2j6H/cYLvuE25nlPN5SjSnjTLVLLsB721zP3GVGNV3mH+
JyKP6yg/THkf6leUPYZ8ZE61Rp7129kGlj2qM38QtGem2veTS/5p8ktIN6OP
RHU/8X0J5VcqPkv5u5ov8i8m2GcwBNkOLbVveyl7/m1oj2RM1el/MvJ+ONdj
1lgvITVhL0ylfqemXrNjKVsEvUiyffDNo/YRyjcYRWZnEOQ51tNn2NdvNbLt
1CDVWAXZUBcz/tHtrGs10h6g7+vT7Cv/hPIe0O9JWs5a7Kn4Qdg2pmxL3YFR
6q9G3q9Abz/134bW0Lh9Hd3JryPfucC24lXky5A/yRo/rf0Jv0fIv5pq38Jo
yitBb3iasT/Pkb+Us65ynrE6mZx5Y6k/Mtv8y4e2m/z4VPvObk7yXSMfmnxn
unP2Uf5wqnXbIZQfJb8m1b6UsUnG6jyQal+ZMDtPwdte+N2aYEzCcfkOM42d
kM48nvpXs9//qXReJ2Ot/EibR6m7nDm7gb3dFZ4zFK8j9WY9DVGdmo6xnEOW
t9PfvQ0d0wqzX/5Ls60vHXAG661vhmMpbyheC72W7U17NWfEKsoeZ41MlG0j
na7YOoDufsUch8Dr84WOdb0E/cR29knKFykMycPQz0Oee+Tbgn4m3xmk1pxV
PzOGSdSd2M6xXPngJ/Bdmu2zuSXyuUG+jmJjGRSzrVdoG1P9KwZ5N+MbTjpd
1TG1VGh3SrcvYEuSfS0vQG9Jgn0ubdQ3qU1ln7FlWN+xuH3DsrGGR20jyjZU
jG0S30OpXw55XMP46hRaRpKNfG6KDVYLey8pRvgqd8udcfu+v6aPfuS7sOZL
kP9KYQTYH1tKHCsX5qsU2falzgrKOsLzAe2lmLEvg0kL6G9eW59lFWval96h
2FgR+dST6KsGaTp9hesZ+/Pt+fi+MEAjmetq8P8SY4vUdmxINq1sWfkUr+7g
mKFihVNJq0uN+RHWRza4YlOKISl2pBhVH+i1y7OvJE3+xjSfiToL25C/SmuL
/OXkWyg+yd0ezTB25stKxuK8SQrXNibnpdB5Hxj8fk4qGzFGSdikRPh/iHwm
7aO0v4w0MGobTbaZYqqTKJ8YMTZCPpwU+m/OejiDLP9K9t1+KOxYvO74+6Wb
S/9iLf0KvZ3kd5B+TzYGSr7eLyLG/sjnOwV6z1K/vXwtzHFxxDFNxTLlQ1bs
8nnKu7R0DHNaqX368uXrzGwFvab59rUIo9SSfAj55yL/SeQj0o0K7LuTz24G
c72D8X3Iep3C+ro43ZgiYYmEuRC2bxv9N65njN+1ij21tS21jvK5rJfxOhOQ
xRHm6MKw14DmXjIfwFq8nvPghob2UfRk7ntTfm85x7QV61zCeFq3dMxzP3P5
SZ59/aeQ6Q98vw1/b8PfCOa4lPEU02cuvM5PMXbktSz7YoUh+Z76T6QZ63Iv
9dcyN2/m2NdyB/vpTfIDCowFGUa+GvkI8rtR9zNrerXuM8V04W1oomNXz5EG
lnUM63fo/5ZnXXcjNI/xfTTP2IdxNWy7Z+VZl5INf12q7wzdFcKA7of2mxle
C2fpYwW086nzOOMbRH+V5R/nvLua8+YKxrCK8mTGk8JeGZzovd2R8XSq4T1+
DfmetI+yHg/XMpbp7SzHZoVpSkB23en/XKJjoDk5jsEr9p4j+SK/70k1mata
sslC9mHLd51C+iZkTIGwBDVIx1gbR0l/V7dOeC5kzICwAk1JNzKXJcXGkgkj
NBzeRsPjWHi9Cx7nU14Zns4yvs2Uj4vaBpXtKczB3yHHzBQra0TqlGGfpXyV
0tmF/RkEjatbGAM0k7J4xnksGjR+CxmzIKxC/dq2XaQTSxeWDSNZbqH8dLJl
uoG56tHBZ5HW5CV8dyGNrOozRWu3KvLbnOg13JG+5pznR5g1YUn2F9mXIEyJ
fIuzQ/ady8eos3AMa+K2Bj4T36PurlRjFcbpDGA+3ic/NNk+ovnQ7xgynukM
9Pdz137aybaXbC7dzYfixmrqju5dbB+QfD/CAK7LsY4l3aoCfcxULKnYWIe0
ho69PpRjLLFisKPixuAIeyMbZD6yWhc29qAG5evb20ck35BiwhuFRQobGyMf
yGzOn8dKjVWSD7wt42tT5L7SSWG+2zK+EN/5DY1lWx51LF2YtnTKCuWTZqxt
KX9WZzfjn8r4/0uxr+JS8g8l22fxI7zeDw/nahpjrNjRybixoIohVYd2AunB
KtZ5H893DE6xt6OkF6H3GuvvJWRZif6vTbUOJ91NPugJ0Lo/7tij7uz2yOoe
6iQlO6YxB1r94sbqKSYcpfwRylMpj1GeFzOmQViGa7m/3soxZly8LKK8kuIz
Obbd29bzXHSm/zOJnpO5Oda5pWtLB6oWtQ4v3V3xQMU2L8z12agY50z0ic3I
v1pNj3kZ45sGjbGUl2V8i4ocg1Ds4VloTKUsRPuKia4zgnwpNHdAazD50xHH
/BXrFwa0hLqfU76Q9omJnssc/lYu0XOaBO33YsYqCZPYTHdhumNT8jkLu9OI
PdfqQmN4TrJf6vC3y1ibr7TizOG7IzSfZD5O1nCsKZm/XV7WMadm+Y4h6D4b
Q0pQLDLHvithrjYzvk9SjU3XmmlIeQPJr4z3cOVcrzGtLWFg7ykyhkzYsXv4
WwjZh0m/VPEZ+hDzMyVuX8v4yr7bhSEVdlR3/NkcYzCFvZSPJI/yE6zJz9nf
5RnfH9BfkGvbKvlC2/4lknEN+wDSyaflG5snH84PzF/FjsbmC5OVxnmRnmdd
VTHwesx/qu5w+spiPh6h7FHph9WNCZHuvZ71vD7BOvhw8ne3N+/TWdMD+R5A
ery8bcqpnIfTcm3bHtL5xvdDyGeN9FX+Vkz+F8bTlXwC+Xvb20aTbfYb9WtQ
93vKv29gTMxWxnsw1Vibmcj/h3xjzIX1zCZ/Le070+eHtP2cdB20Jss+0dsA
8UD5YzFjR4UpSGIsOztZd9cd05v6o6j/AHPVJNG2607q7E6wDauzoAfzFyT5
TDiC0Hpxvx9Icowkqvs919iagPzF2DJNOxq7JoyZsER9Cxz7EaZoAGv3KuT7
bZLnLARv7WPWja+Hp2sY213pjp0ohlKZtVKaZV+bYqjbyfeFZmEV+9iGQOv7
LGNtysLfRPIP5Bk7Lhm/y9j2ZtuWlg/hEtp2L7DvXGO4D3qj4/YNyYY5Tf2n
OPN2I9sTDT32X9t5riWDP6OOwSv2Lp9VZ+RzDeW9A/uEpAt0S/fdL52gAmN7
L9O2v2Kyh5F9lXb2Pe3gztlK39vijkVL59/HfN/azlg6xfAG852S6vcO8jnq
bcOHrIEayX7jMIDym9oZ2y2fk2Q7MN22h2T8XoZ9WvJlCeOhtzT1KV/cyG9q
tskepHwr5a11X8SMYRB2QT7ZJtLPo8Zatqc8ifbPwc+WBL9B+Jj2z4X8niKP
8uuKjOkRlqcX/DXmu1Gefcc6gx8WtiDPvv67Gxj7sj7u2JIwMLuz7XOXr30P
fdzB94ASY7Ol4zSj/9EZfjuyn7Sd/ibkGRsmjI98UR9kmz/5pIRlF+ZGWBth
2nfSdkHIvtq29LGX/JKQY5m58hFB65Y0Y/V05stW/DDbvMhmnMH6qVJg7IQw
ryMoX5Xuu0c+l0LOvgH8be8FxhDXTrOOKd3yN/p4gfG2l0+tjDEkpXn2Uck3
dZT56sT+WQT9T5P8ZqkUetOFV63kOZOv4L1Oxl7JZ3AFvO9kz1Rh/A8nO/bZ
IWTdRzFQYY87MqdD6xoj2Zvz/3DE2LZmLexLqBezr08+hQ7ytRT4Lcdy1vTb
yHdzqrE4w6D5lOKd8Pxlkt+4bILWEP7Ws7zfdFyo9z6F9pWOYz77wduodPu+
FTPLgbfckGM531H/Hvmz5FPkLHpNPr50+wzkK5AO3ilmTKCwgBrDJvj5O8e6
1xb4vYjxNQ/bFhGN9dQ/mO5Y4DHqz0EebULm/WSKbfPrSS/Xso1eyFgPd7Yv
W5ifnfTVIGYsqHygcfkXO/utQDnumzWspdfjjvULA/4a81GRObqcuXmuGeuV
up9KB2XsA5BvV9bLm3HHljWnc+C1mPF9dYEx6dXaOeajWI904Evka0m3rqKY
i7CK1xc41izMYqdU3+m6y/UmS7bqRxFjB2Szvhqxj1++/Q0pxlJs72RspjAV
evuxMMdYc70BuQpasTz73mpUsa6ymv1+qI51lvvz/eZH2Fa9gfu71HeM7hb5
dBX7fTvu2IJiwA8KzwgPn6bYZ67YqGLUik0rRprGXkoVHo35yUR+XSlfQpuz
KdZB//+tQZbnTjr6W8irN3Owso7fdMyW7Uh+Lfk3STPJ30D+Db5fJyXDS624
YztReNpFXwMoX1fHd9o26G8PeS8KY3ZYb4virhsnrUDWFelzSlXH1BW7V0xS
sUjF8N9F3htIc5H1fGh+mWWfkHxBhbQvgfZ4eFpG2QfoFD+RnxE9zxv8v0P+
7qjbfpvsu2MeMr2+qe+QQZRl0d8sjY30F/OxMmTsmHzCJyn/NmrfimzmivBb
vsB7rUp9Y51X5trWFeb5WeoOg94uaO0U5pO6g6C3oop9FiXs1Q6Fjn3I5p9I
/e7Uf5G6Lwh/T/31pPeSrTN+wnca8mlUzRj0ydS/WvFg6r5Ux2/rOmTZ96g7
oRe0exQ6li4MelXWRnHIsqkjewP5fl9iX9LyJvbdnGY9PNzYPpyFUe8x7a1N
0F+TbR+lfJPyGa+Enydosy3ZGNwPdXeH7CuSzils5CLyHyYbIyns05pc+x6E
gTqWZYytsLWlsheixkgIGyHMp7C8n0Z8dgjT+yBz9VCpsX/CXL0fsk4qXfRn
8us7+8zQWaE1co9itcXGburOXgH9uyk/AO39dXxXlYk51qo7S3drfsy6kPLC
JunO010njJJsY/lo5p/HF11I/1v424EUx6grYFt/yt+6VbRNIF9t+zy/FZXP
dm227zzdddJJFYsbF3PsWDG5d+D17WLrmnoDUgTtgoixLbIhp4aMeZM+oTXx
eqp94vKF602IsEBdz9uTwgSV6C4n37C23ygKW1xWNmeKMcbSrVeXGMslHfsz
vj8nzW5iTPyDzN99heex05zvh0qMKReWXHUyoPU+NLs0tk/xIuR7MMOxi4jO
BOhPgMcpZRyDWEd+dqrvmizWx4ZU+7Dku9KbYuky/0T9lkPrd2SqdWTpxrJp
myLLZtJv+G5Q3livWI5tU2G+RqIPjmpvXf4P+A3YKw9EvZeaycYPGYMqW1Nv
nK+AfmPo/1zDPu3vKJsU9V5Kqu23uR8z5kea+I3uc/C/Ls9vRxUDk29jQdR7
Qz6OrqmOGSlWpBjjTRFjNoXVHCh8C/RvjfpsSaztWIrePOmtk2Iql1J2hjHd
TPlA0hLW+tMh32VD2T9/8D1CMSLKGtDmO/p7OOS3o7I5D9DfJxFjY07QX6He
E+Y4/quYfEf6OlRg7EismrGSFQp8dgkzKd/MxiLf7Tr/X6GvV6PeK7Kx6uba
xpVtu477Y3Gp30Dq7aPeMN5G+znkezb1GVeW/sqF7bseVdVnV7p0oto+w6T7
zgsZWyQdWG8j0oUfqOg3EnpLMzPLbzP0pmZxyJgCYQlkI7yD7tYael1q26e4
NWQdR7qNfGA/lPiM09nWljrVKRvFmPek+M2bsC/9io0tFwamZ44xYMJ+XcGa
WJxtn5V8VfJhtip1TEqxKGECpXtUKrYvUzrIJzFjCIUdFCaiZrFlLlkrBnxE
tku+sQLCME6nfH66ffMVGG+DQjlOOM8aGAO8VlizztYV9YZ2OHX7Rf228zfh
meT7KnbsX/PbN2QfiXwjuu/1dq4/bU418hu6pbobo76LLhI+POSYlGJResMk
X9TlIeuyOkPWYUu2hMbfVY2538TY329nX4ls9uI8v2nVW1bppOJdb9qEndYY
JsPbd2Hbk3oTf3PIPir5pvSGJI3yb9P8NlxvDJvkGlMlLJX2ZJtS+9zka/sB
ef8i/VFvJJBXPu03Z9vml62vmN+jub5TdJcIkz0/ZsyVsFbC7E3Mtc4hXaM0
0djwfdDYl2CMeEbUMWbFlovJP13qPai9pze+c2WrZdhXofUahOyDku+pY7Jj
Zz/m2ZevGNor7MVBGdYFhXm5DV5G6v1/sn28I+mviPEuqeM9Lez0/0LGiglD
vTHbNp5sO8X8dZbPilr30pkuLI/WhNaCMD0HOfumx/xWXW+Ub+noPai9pze7
H7EX/mSNfdrAZ9ppeJmp+7uKMZn/yzcGRNgP+fwU+/o+374OxcAmxe1zkK9h
ETL882LWQtyxso61/LZlSaZ9/3rjIl2uUtixLel0p0sc01YsW2/ax6b7TZre
oglTqrXWitShttfcA6XG0Ao7G27qt+rXFtm21Zt12RKfZfm3B2RT1M/xGx29
zWnG+hkQ8RsQvf1QzEmxgDZ6o9HYMYEwtMdqzLUcI2kB/y2jvrtkQyq2pTdn
emumGFcV1tqGbGOthJlYmm9MtrDYP0Lj13y/UdHbFK1PvV2sEXYsT28Yny21
zSlbUxiiU8W28WXbC1P2V6HfjEDm/+foLvoa2s5vceSDOIK8mkSMLdKb7Rml
/s0I/VbEJaStzH2dfOv2wohPKDKmUVhG+ZhuijoGq9ir3pTqrlnQ3r5w3TnS
fU+GHAuRDtw603ei7kJhOM/Bb1dobKxgHfHXNn7jq7e9+g2KG+GtRobvOr25
WR81pkxYshOkd7KNGRFWRDEvYaFOMmf67QBhouRb6pbntwPyMS1gL5bkOfam
NZGb69/k0G9xyEe7JuqYvWL1spk2Ro1xFbb1W52RGb5TdZfqN0N+Yz9UKfRb
Gb1ZmNnWPiz5roSxn1fomKFihXpzsYy56JHju0AYKL21vqXAb5/15nqlfDOp
fpsjnbIH8ulZ7LeAwuTJ1rw15reRsjlXZdvnI1+PMCSKxe2P+K5WTG5cvn/z
RO8ZtSalW/0a8W8VSMeSb1I+TPku5aNULFQx0Enn8edVcx3zU6xPPglhty/K
t2/5/zHcfL8b9VzozcHXrKesiN9a6k3Yf8j2rlJjN9ZAIzPTOot0Fb3JbE3d
NhGXicdopm0s2VbCgOttXEqGsb16I7eS+bs7z7F23fnvRozRFTZXb77WZvmM
19muO1xvc4R5lL9K8fSeRdZBpXsKg/MMfV1WYGy0bPwCyuPCgFEWEyYjYoyl
sJW64xezv5fE/XZDb0aEBZIPUL4/YYLGMt+jSc2qO4a1m/aPZPhtpta4sCD9
2xq7IkzId+yV2nHbmvrNEGGphLkS1kqYqtmUPRY3dl9v0HdnGUMq7Kh4fDBu
nsWrMI876W9HxG81ZbN/wXwfLrYtc1NjY6n15kxYWmGqFUtuEvZbXsWUZXtM
znBd2SB629G1wNh/vfFYJl9TxP5qxbD12xp6U6m3lPqNDWEX9cZfb/uFYdTb
P8VwFLuRz7e2bKsiv+XTG7NC5qJVkbFG+g0KYWWHtbUvUZjZaNhvyvWWXG/A
+kl+nXw2CTNXl/qF5/H5wgzrLXo87LdiepN+J7K8K9e2gd58DI8b4ydsnzCS
l7X3m0m9lZTPXb8FojfWelst+ewWNibPsZ47yhtblhi27ieMmXSHueyv6QnW
IZZhry3tYCyCMAD6rZmC8/h//ebMROGhoz5L9KauKefTsLixKHoTdl2Gz3yd
9cKIvpltH6d8m4oB/NvGvykgbLTenOzON6ZTWE7F3PowN5Eixw71Gxr6LZyg
nbFVsjflO/kxbKyufChp8JIZtW4iHVm+mH9CjiXLJ6O1PL7Qb021pjfqt1eK
jV3XG95nsh2DVOxRMalZtJ3A+gw1tQ4wPdW/SaLfIhFmV76dX0KOlcvHI1/D
jyHH0uVz0FujGcV+66Y3RxWgv7mTsXpf0/8I6t4b8tt46Q/yJT2R71ilfEoP
pjpGo9iM3mRPk7+Ev1VONiZcutKoUv82jXSm050cc1esvXJz5itkjLWw1Xrj
r9+y+rzQv02j37R6Mt06tnRr/eaIxnp/qXUJjbkw0zaVbCn9BtLCkGMQij3I
ZyHfxtzz/nn5OC6jbFm2bR/F4EbwXT/VvlrFrBQrkE4kXUj3tXS730OO1UvH
K2LvjY76bZbezEZYz9H2lpUwA/2z/ZssChXqN1+E9fvnYr81EWZuTrF9OPLd
6I10n5DfOOltk96E9w/ZBpDurzfXP1D3hmzvBfn4chRryrXuqz2g2GeLmLGo
ioEKS/1LqrGCwlTHhZWJGiukN3+y7eaW+u2ubLw4a3le3G/x5EM7nW8dXftB
v6mRyFwkFRobK8zw5nzHKBWbrER5ONe/yaXf4lLM9mxHY3CEvRHmcTZr8bF2
tuXl416VasyhsIb6TYukmH8DS799pTeCest9L/RHNPR7SL3NFyZDWAy90f8u
wz5A+f70pqI46jecerup30jZnu2YnmJ5wgxFhKcuNZYyUf4K+H0v19/yaYyJ
+E253pLrjXUb5DU46rfX0jGvzLWPV75dvaH5CPp35vmtuDCu61ON2RFWRzbI
ndAbFrFtNjzFsdc7Iv5WDDbGeq6b47fe8uEqljE85t9iUExjl7D5ehPQ3G88
RhT5N3CE55JMVjLWWfnG0ug3isalGoMp7KV+E2hZvnVa6bJ6AzBE8Ykc69LC
pCfzXSvHb+MVE+mcboygsIH6TQS9PTiWY+yQ3iAI29ouZqyAMK6j6H9ksW0l
2aQv0dcNqX5LIJ/Z7ZQNLfba0JudUuFf0u3rkc9Kb1cL0m3rCvN+ZaZ9PPLt
yOf7HLIfmuq3bPK5ShY7831WSCbyrf4XMpZHPtYtilXHbcvoN4J09kzPMdZJ
Z9D18D61vd/K643kkYsdE1csXD6o/wPSqNhV
         "],
         VertexColors->None]}]}, 
     {Hue[0.3977474056698254], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFnXV8ltX7xx9QydEbC7YxGAzGNgZjG6y3Z4mJKJKKlP5MUCxU7AD1C1iE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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtnHe8z+X7x98U2ftwzrG340zH2XsftFDJjMyWkK2M0pIio8zKyCpFRoQo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         "],
         VertexColors->None]}]}, 
     {Hue[0.4906640965493956], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxMnXd4VsXWxV9qIIGQhJBASK+kEBJIAQKEtDexIM1CV0EQQREUROygcC1g
V1BBr11Br71j72LHgl3A3isqon7rxxoevz/Oc+bMmXbmTNll7T05048eN699
JBIpTYxEOui+NiMSuULXlZ0jkUnDIpHJuk5NjUSisb4viIlETtR9ia6LlOF0
3Zfq2r+D78t0fdNO7woikeqiSKRH90hkosrI7O+8J+n9ybpWK31Tsst5Venb
6f0YpZun5/nUoytVaZb09fNkpencKxI5XuFj1LZFij9K4X3a+U6+Z5X+665O
T96OSr9Y4apOrueEUFe66powzO2YHOs20Z7m3pFIi66e6pCk4ZHI022RSJ2e
Px0aifys9Md2jESmp0ciw3q73lGlkUiG0hWp/EdLIpF+Ch+kNDPSnY80b3R1
+hSlacmJRLYUqh7d3y50+DP1yel93YZ3lL5PfCTyGN+lNsxRO9cM8/Ow3Egk
Vv1Zp/srcc5P3k5KM3aY+2tPuS/qW9bG+nv4lklKt1DvO+l+nco/TuES9eF2
hRel+v17Kmek2j9LeWb29v1Lfcuq7Ehko8pcqXtTt0ikUG1o1P2Lskjk16ZI
5JJUxxURr/HzVpbjflb4pcxI5Ejl3aT73EKHj+sSibRX2el9IpEjVH77ERp3
uvrpube+Y3O5xpD67Md89bXS/5Dv5/56/7vKXKS68xSep7z9Qr20a08d9P8T
+Q5T1wv6LyX6rhmKf1Xf+x+17RZ9+3kan8XKe5/G55qESOQOxf9P/fBhrdKq
L+5XOEXve+l6Tf29UN86Rv+7TP9xQazHOuP8+2KN3Tb374Oq61J9wzjV9Y3a
fBtjWf8iOZTzssrcojG/QvGLY9x2viGq9u+nNl+n+BX6X8f3dV7G/J1Fzv+K
ynk9zmHKOYnxO8x5Htb1SGjP0bF+Jvy0wtPV19/pe+dqTj+pPpnPN6Y7vll1
vaZ6V6svTlWbv4hxmLq4rwnx3/Vyf72i8Hi1dZXCA/Uv3i9yeGSIOzeEF6Wp
73VdqTbEV0Qip9X7eXGa36XwHYo/XfFL9Pybxu0Jug9VezYkO5ylNMen+X2r
+vLnHD9TZqzynqK8C/VcmOwyid8a6zSkz9B4aNT3ftPe/5n279PNZZ8Yyn+5
p8a6wpXK+3kP/XeFf1EZh6c5/jeFZ+t+hK4/FP5pgNYa1XtYmtMeFtJ/kuNn
yiHfrBD+RemPrvfzFzkuaxDtVP+fpPBealsffcvyerepX7zXzdSuXjsJx2t8
Hpvmb01Qm09Jc94BCr/W02XSth9znK5SZS5O8Xw6SeUUhH54XeP2FI3PhzRm
HungOZTa2/Nop+ZHjsZwX8W9pjQ/Ks1cxd+s+3pdr2k8pXT3/TWNkQ2Ku4W5
qudeWhsXaL68rvgZGidpKvM45f1I33hoGGOj9B3Pqw3r1Obxnby+sLbMCWvN
xRpLd6qMn4Y5/73ZHqufKu99Cj9V6LhUrX0x6pfTuvr9U2E8D1f7N7W5Pr61
KXwvabvo2qL+OT/Rz4T7qs2rVV8nrQ+T9PyrvuNGtefMKu0tin9DZW5SHdsU
X9/O9+0h3Ft5z1eanUpzRl/nZd2uV9/1UP0LVM5VWkNuV5pylX+D7u8O85rX
R2m7qg29KSfXbeObWNOJv1p1Zma6/YzflV3dx5tj/D425L1OZW5Rmb1V/n69
XDZraYv+ex/WRuX7TX3y+3A/36y16wutXT+oTz7SOnBxnsaHyny2Uv+x1c91
qu9y3f+rb3pG8RsUf5GeX8vwfbvSf61/+pWu2fov60fqHyjdBRoXtw2JRP4s
djz/gPWF/z4l1eOAcDvNrytVzt1K319l36VrvNq8Js3zYa+Ql/R3ak6/0qD1
WtdnWrt2qA2XKG9LZ68P76Z7rThebb5O8Rfou65XH65T+C7mS57bmaRx9ara
eJvqWqO4tzT+rlDflqqc5eqbMu01z6sPFxY5/ljNi3MUP1Lx2xW/TnFrdd2m
by/WGL6QMRfjtGtDOdu1361ReJvuqwsdHqQ0x0c1JkI5G9RXL6kd67r7/9IW
xucuPafneY7/o/z9VcffWV7nByn8ldp/h9p+Z6v/zZ46eqn85QkOU9fjqW7b
shj35zZ9+/3qpwkhTN+eo/xn6/pa4+c/up/Js+p5RGX/rrZtVht2Ffkd8aRd
yXhQ+DG9v77Vz/0y/Y74wzXO/wh5n9D/f1D/61HV977+15G6v6D+nKx5Oam/
+/AqtfmqdNOS83Q/StdYtWeu8h2pa66etxY4/j2N4fNVz7l5nqfEHRnSz0l3
2jHtvBYyfhaoPatC+mc1rp7r6GfCp+t+hq6TleY+tfdKfctSPa9Sfy7T/U39
xz+KnI40jyvNjUpznp53Fnjc7+rs96Tvof6cmOEyTuzkOs8LdZ2k+ym6quLc
lyt0naB2Ls1yuF7lxNaovDBfxquc0xS/UOXcyxhp9TPfTPuTVeYmrUuXKf4p
jYe7leaKVtfxh/aXY+q9P/1P5V+gMXCr7oVq8yqFn1UfjtJ9P12LVX6x/l1L
of9hc6bjm3Q/qpvXL9au3e8KPU7uSXc4povvLSF8qtJMUfgU3ScXOnyZyl+m
56kKX6TvytH69JbG/1lqw0S1Z6LiT1f5b+Y7/Eb+v/lXKf3OXIcphzKm6bqy
k/NXq20d4BHy/I74V7WmHSAa9Gj15yVa22v7eL3ke/hmvYr07Cf6QeHzO/l+
QghPifPzctXboP5aqvAvSt+o/XSX1qA2lfO+1oEPhvv58Wr1rb6lVvGpzD/V
9WHEZRwfyqlOdrhF5f+ttW4fhW9XX72S7/bQ/8TtG+JHZjpcn+n6l+l6K9bP
pNN2Hvk50MBbGMOZ7hP6eEK6/2+u5tSX6ocN+tgP1A9NWS6Hb/pZZV2n8AD1
2+H6htm6DunodeUaxS/iGzUvG4f73RG9nY59+WGNyQeUZpTK79fF8eRlbJ2v
+LM1r09S+DyFT+Se4fhRSpOt8s9UeCp7mcKnK7w6xrQF9UK7kudchWd18J1y
hirv0mzXe4v6s3O63xFP2ReE8pO7u7/oK3iSh6ANlP6dDIfHq/x9FH+Twnvr
frTaPr+3+b48Pd9Iv+meVug0i5T3KPXfXuqHVZ1dxsZQzmDNuz7673eoD+Kz
3Z/soT0Uvr7w37jrQz/T9gdDvy3LdphvOjzbtA50zr6xbtPu9mgc3qA0dyl8
bab/Af1P+4hnbHO/MaQZqz6ZE2jII9Mc7qd+7quxt0Dh7nGem8wr6BloG8LM
r0yt1d+XOL63wp8ovE7hYZo7r/Ad+hcva87+ENK8ne/9lb2V+U25zM17tU5u
VPqHY3x/SNdYpX9B75/XldbJ92dD+B+17WmFO6ptT4lWeSbTcb/3dJqD9C3V
KuNlXUuhMdSG6RrT6Z2cj/R9OpkmoK7LOri9L4c2My+YE+s7er1aHsK8fzWU
+Xiu41m7Xs40X0r5x+n9Il3TOzi8WNdeSh8t8DNh8q0IZTL/qI/1p1JtHNjk
d2dkOp40Z+p+lq7/KXxRsp8JD8l2fHyX0H+hD+fFuv2smRcVOX0+MgSNu7+b
nW6H+u0vxcf1Mr0Hrwuf+1ym+51+PlHj/50C7xELtb901b4wUd9VlO742Urz
htJ8qPCWrt5P2WvZU4n7KMQfnu7wrJDv3VDmrJDuOZXZL93xlM26t7mP18DR
Kv+PAu+VF9ZpPoiv79/BtBjx0GMzRQNMgn9SXR8XuA3s9c2a14MUn0969ckO
lXlBe/+rTeF/0VbyQFd82tNyH2Q+E9RXf6qv7ldfDVX4C4Xf72LegTh4hF+U
/qkwlp4P/RZRXy4o8nOnOL9/OqR5W/23S+XUdAtjV/H7a6zmaO78UuJ/+oX2
hf20H3dT20cP0r3UPDKyE+Yzc7lO7fkqtAdZE3zMp2G9+jh8O3zMfaGd3EmD
PIo+2xn681SNtbml5p9+yPK4vam9/1djjv9dmuiBM1v8/GOj+CldV6ttG9Te
HQpnd/R/b8pxvauzLDNCXoRsivAp8C99zQ9kKbxeeX9V3qF6Xp7ke2/1yfOi
b1PKvZcxBr5U/MPqn5f1rakhnrXz1Wavn4sU7tHiOQkvVau6XlAbYtQPizOD
3CPTYdal7bp/omsgfKvyndriPC2BdqnraD4Jehpa+iZ9V7bCPTuaTikO8z0r
02mIr8o1X4VM4C1909u6unTz+4yQZgP0qcIbY7w/s0/v2a/Zm/OgkXqazoDG
gIaiPezvrZnOO7qjZXv0KTI63jeFNh9c6//Bvzg7y2Wy1ydrzt5a4vLZO9g3
5jAXMt230BXbNF6/Ubitk/ntmUrTuaPvs7K9v1yhcbJWV7LKnsyeovglXcyj
b2h2vllhTyL9l/nOT3hVjPM9ofJvVNqbdM0Ie9dh2ebV3+3qtrFHU+5vmR6T
p6nOpU1+3pXp9Wqq4mPVn51UZtd0y/+Yz8jl4BEpGznUOo3rC5pMm32d6W/m
e9k/6Qv2U/ZkZJHsy1t6eWx8luy05OnSy8+Mm3eQAxX5XbHm7tbQb1/rfbLG
4Uv6rlvVho97Oe/uvs10XsbbparnEl0nqv/bCvw8Ocb1Q3Mg07si2XlIT32E
09QnY0stK0ROCC9HOU8qvqfq3RTqRTZIOdAtlL06lH+P8s0XH5Sq+d6g8bBR
62dlB9PG36sdU1XP15rj07U2TdZa2aI58VK55TxL9XxqmvkOaKRz0kzTQntP
0vVtjO88Q4fDF64JMijynaZrb4XvUr8tLbDM84FkvyMeep28b6k9F6j+C3X9
R+8m5vvOWpSkes9MM9/aLdfhJpXTK9ft2Vfh/AqN+3rXx/uzQnr+CX1E//TQ
v75U8T+1d76V4VvSkYvoqkOGlOHwYRm+eDcz4j0rp9D73d6BzoNHO0RrTi7z
Ocb3vBDmfd+QBp6O8ATdI6FM6vqv2nt1vdt0CXddqxS+TPfLdZ2v8OR8x/Gv
X1bcFfrOXfre0xL8fjzjMMd7H/3JfXLYB0/KcJvZwyfovx+lvBfyH4eqTIW3
xJnPPTn8ryzF/afez8hdVwZ57KR8j1fGEnLd1iDLZYwRz1jk3xLPmpykts1T
GWdr3f5T42qhypyb5rijQ/z3OaYtCXdWvScpzTFpjjsm0Jz7x1n2iHyyndIs
qXf+Rs2JvVVXTKBb5wbadX6oIy7O+0PvQGPDOyArZN8/UDzUjUqzWmnuVHl3
1fuZvZQ9nv0UfgJ+Bp7iOf6D6v5A6S/T3vSCnrso/kndlyr+CcVvixc9kObw
xRrn1QWmxzaL5pzWpr2+wHE1IR4aEFqQvfYOvqnC7flS/PfMNvM4J6uuu/Xu
R83ZCflOD92IvJNvIc1Tev90vet+nLGvch6DpkhRe0XnXKh/dEqGx2WG+udA
hXsrfIDu8eq711VOidKkKvyWwl00f/dWWW2sAV0tW3hZ8SsUf0Oy0xPPfXMI
90h3mPJI0xZkxdz3CmHKeCWUU6L6S3U9is4ow+FrM7xf/600j6idhXEOn9DZ
MhJoFWi9Luku670OXvcpk7j3NcbG1Lu++G6ug/IZHy8WmOfbI7earjb31vPb
Bf7ucbpaw/ihD4hfrnpvUfwPBd63WSN/COvknmf24tmqd9MI9zV1bCrwOrO/
yhvH3ERe2sPxWeyJSn8I9HOa2zIxtGdXjOum3o+VZrLSjKfvElwOMsmPc5yH
ebq77BBP2S+G8llf0gLvGaf5Xq+2tVNfXaX4YuQbGZYxIGuYq3p76Z5SaBnR
uAw/zw2yI+KJS9Tau0J57lD8hXouhGeL+P/8E/4R44kykV28n+tyKJN1jTW3
ey+/7x3q3arxHF/odZh7jxAmHc8nqj+6ZLh8xgWy+2gfy/9PzfAay7iG1ztX
8d07eAzx369UOE3f3VdXm94laq0Yofi1att49We/Qs8p7sUhDP8yqNCya/qI
d/co/UUKFxX6u4+N97cjd+VeFMIfhrzHtPe9KpRzg+JHEo/uK8NtOCrG9/oQ
Tsp0mDam6mrgf2Ralk45lI1sn+9Ahs87ylwf8b0hlL9K/fB+ntaFnq6LMN87
LM8yTWirKaJJDxpmHftErXWTSkyH7Cv6rVXPV6qcyUozqdHf/Zf66gzNwwVK
M0txMxs9jn4RXz+hu2UAgxQ3UNd0xffU/WaNueGi5bMUbqfwOqVt0Do2Ule2
xkV/xZc1ej+c3k//MM7/bkY/P/+mtfMAtaum2e9mKu4wXTsVv6/yjdJ1uvI+
rbIfEX90ttqxr8rdiqyTvULvf0Wmpnonq5wElXOO4qsVX9XovT2qe0uj98ZS
5X1HZZfo3qi4/ZTn+AzLUpBdIrfcT2WMKzEvX680U4cZmzAr1liGvRS/j+JH
K/7IVPfjDyEeeTTYhJEK7686vot3nx6tOubpqtG7I5T3WX3PgUn+Vtr7jMbS
cYr/o9oyyDaVM1dpG1TOXMXPafTa2ap29dF+MRP6R3F3qZxWlZOguqbp/dQM
74fsx+yJJ3f3/UCNjST9k2Rd3+T4gi5hz83XvUxltFd4Zyffq9uZzmAfnh7x
PTGEPxmieTDEMoR4taF7o9/B/84vsiymi567hnq76H1Mo5/r1E+x6GNVfn/V
W57k5yHqp1365p6w0nruqetypTkm3uH5EdNv0KPQfgAE7ql2nizRuZkN3pe+
0tobiXM5N2Z4TjJf2jM+ufQuTX3YVX34j/L+pfR/6x7f3vd/QribxsIlnb22
UF67UCZ9Ewn9wz+5ptX/JWuw+lT/siXeelv0weh80en+qiuR9VNt/KvBz5/m
BBmHwrH6J92ijjupu9shMiHyU6xlr8gDCzMs60J+cAS65DiPt69U34w4y+Wg
kdj/kHO+GGgl9ghkn+xVyEiJ3xT2x69U3qh8y+W4gxuAd4ZvHhXC/dSe28Hz
oM/KMp/J0KnVeLtca8xlXU1rwitMVppl0H7q2/kB1wC+ATwPc6wpybqVT9UH
3+jfnalyR6ueMbrWtLfuBT6CNF8qTQf9p7/Y6+m3Yu3B6aYB2P8PihivQJv5
jtZ484fE/6L/0FPpF0RMpzAu9tAt0CzEf53nsZGFPibZ9Ba01vRAJ0EjwUfm
NZiXhEaAhmC/Zu3tk+719sdkf0Nv1f9thtsGrRIb2kp7oFkIQwsdmGcaBLor
Js95+XbK6x3K5J4awtAfM8P/XZJhORn8CH05P866+znqh+f1fYep3lFau0br
+jHL+I15cZYHLlfeI5T3Jn3jWNGMozXeDg20NjQ3NNtzyvdCs/lk9totqmea
yqyEZ4x3+n31T/q2udx9khzHWletPaWatTfG6zXjC7qoTenbmj1ev1YZc0J7
wMeUxHiv4Rv4lif1vcM1p0boOkZlvKu+f6fB9OAb+hebdU2E/1d/rs0yX3yL
3l9SpT1E612T8jU3up8OFz38sso+StfPShOndnRX/A8Kf99gWuMv8by7dL2l
Af2n4i7UWpqvb3pW61phL8tmalRera7ZSt+iMgY3WucDLzY+rIdzlOftOOuE
52g9OELX3vq3b+i732o2r79W5V+tMXkN/1llD0py/sGxLg+ZJ7qNSQpXdPiX
H2Zszu/qeY/MsEzjL0bhIxUuUht6Kzwr4nufEF4UsCxgYwrU5m5tfneq5nJG
WEu5Z+q6VeW8r/+SpvDA9r73DuF+ytujzen6arxk6PpFfbUUvbvi7lU5V6pt
ZQp/rnK66rtWJ1pOchx4pDjL8dLUZ30a/QyttRaZB3JC9VOzrrw8y1gWF1k+
dUEPl/+J8m7MMoaQPAPUh1nFxgHC7yMjQCZwr9JH4yxHAmsxIoSRS7bGWTYC
rq280di2GSrnIY2Zheq/w1T/DF0tasPdqusu9EkdTP9xX6VrDPtwnOlh8EB9
FV7InqjyejT6eUCC+5T+pO3lof0D1YeJbX6m7w7PMF5osOKT29zWhWrPbH3X
Qo2NCdBpzdbhtyrtmynu37NVbxHf0s40Vt/u7vuBSlup6zml/0y87HSN56sU
f3WuMXkZ3Sy3TdX9vojvyFw/Q3evNv9Xafsgn1QZ++j6XuVcV2GZJvLM+5W3
VvcqXSM0ZgugF3TdrbF5l65ZMU6DbBtcHHLumFo/P5XvezVYv0zjh5Cv3qfy
36x3mUdmGE+GvLpe9/Up/s4q9CCqL0Vprte8H6M+2jvdfCL1wzOepnpzw79I
UXt7Nbp9xKEPQ88Fz1IY0o/I8Dfw/Lbol7eGWG5PeeQ7TY/VCX4mfJD68EBd
Kxm3akOz2tCgNtyi9i/Vv8tR29YpvkHx9YrvqO++Q/91mPrhEO1vdyvciBw4
xmmR2Z4b5/LQTbRoTHZW/qkak+3UjnMSrTs4SOX8b6C+W/+vTmWktHkcw+u/
XWN+H1knmDzwMOhp0WOiqz2um+Phmxarf55RuEVpDiky7qa9rmqN9Wb18xKl
+UTrxMpE60DXi74+T+UvUb6e6qvLFP+l4m/obD0KshT2Y/BPYJ8+117fLup3
0L9fhrrAMyEbRp68Ltb4UbCjyJ7AyOyW+yUaBwb+ChzMaTHWwYF/JT1YMsqk
/dA5J3Q3PQQd9WkP01WEH1R47zDH58eb7oTmvDbs7VehY1J4VaLXTWjNLoH+
BIvFegGGDexRD9X9mML56seB6p9KXU+rT86AP9L7DhoLm1XvFyr/LdX1msLd
Ve9olf29+r2sq2mUfdCD6t3YVH8/396W6vIe72DdK/IH0pyfYZ032KRe4ECU
92vFD841jfWFwnfGmyaGHob/eUB9dYnKGcS+Hee5vCnez4QL1d4iXWNU3s8q
46dcY1DBLYOTBmt9d5F1hsjVTwo0F/vwsu6+sxd3057eQ/N5mtK16Bqm+PYd
XOfAsP4P0vhMavPzmgzzovC58KW08WJdByY4D+nvyzJWGozyL539rW3IMBX/
oPrnAV1nsJ4o/dMRjw9k4sjD/063XJ5n9ObozO9V+UOU5l3FFbU35q7PUOPu
4KPAcbB2olf8JNb77NfQn7raqa2TRLfvpb0gOcF4T3Cf4D+RVSOzhq49WfPx
24HGneXpe2P1vcmBR0kO++lWrSUfD7HugLbXxhl33UXpOyv9Byq3rqvv/dqb
94e379HR33Ftvr9lYndjAcEBFtNH+uYi9DOas4fp2qD0D6jtV2hczlXa3qmO
76v4rYrfpmuH8mzX/RNdb6q+1ALHoQtAvr04z7TBLSrjfO19i/T8j/afj5T+
Z6X7W/12SJ71yTP07TdXei05JcHvC2MsmwMbB4/zjujtc1utk/25yG3tqnn+
nPr1olbL0km7KKTvgA2Awu/qe1er7NOVZlae880Kebeqjso8z8eRqrcYGkHt
L8y2fIj1/GD2zUrzKLPV5s4Z/n8TFLeXyuxHWer/7xr8rkuQOcLXIGfiGVlW
VGPmR6WJhW5Untl51qnFdvEzusvtKQ4T/32M2/+O2jZZ9f6T7vE3J+QlH7xJ
bKiL/bokz3t2UZ7bBV3BOsqYZjzPV3sX6BqbZ0wT2KbdupV8y84e7Og7z8jK
eP9WSAOv9nbg49B7EoaHq8t2HuRsr+u+Od+Ya/Kig9ESsRsv/mYID+puWwhs
DdDjoc97Bt3iSO1Nuo7r67ivQvwEjZlD1G9T9A2lDZZlI6Nmjad81vkIthJ9
jUPYR9+2r652Ch+q9NN1VarO/RV3gK4d+q5PEt0HYxT/S4zzRJT+aJV9eqXj
6fvWMH7uTXEaytwvlE++vUIa/tVDSjNe4Q5Kc7H+9WKV9ZfiT9D9RF1T9O7b
IreDNqxQPce2uk2kg2eDX4MnI9w18HEHhXBOuvlJ9hvKoK4rFf9rostmTg/s
4r5DN8pYLgrjeZ76tUF1ZeYZc3JvuvFyYOTuCWH0wMigkT8zL0nD3LxT/XeX
rvvgKbQ+vaFyngjv7wt5kWmT912Fb9Z9fQg/ytzKs43IlFjrvNFZX63yrmmw
7IP3pBsfeFXC8LnEPxHy0g7aCV7lIc2111ud/nqVcUOD65uW57qp5zbF3a7r
bj1PTfYd3gG+FP4UHCb9+VWeZZ7w6F+HMBgN7GEYn8g3SY9ee/c95MVWZnpI
Q5j04DvezjDuBMwJOIdZIT0yfvoIPnu92rWhwXK+GxKMP2WNBRO1It16K/As
5AcjcWu689Kfs9MdT5nIXT7Js+wFbG088pEY9/HNIT0yE/Cd7C/I4JDFndLe
ekLCrJlVej8wz2vhZMaprvo8v28X0ndSn3/V4DXo8US/r+3utZO80BsHajy3
Km+BnvPzfEdGMVbxIxXfPWCHGYPgh6FTKkPewpAHrCm8UVFYu5p0b9S1vKP3
zsKQBv1Rfih/Y/hmvvdk1TUdfi/Pebl/F2Pc7OA8YxXgx4bmGUM7QWkn6qrL
8/shIQ16R7DNyLFvU5mXKM0SvTtK4YNanW5oyEM5Ryr+AMXX6vmBRMvHGTtf
FbkdtIH9cEnYE9EtUCfr0udKU6Pwthjfa0P4uBR/+5YOjoeehpaGpib8tOKj
Md7Tr9I/+lz3L3T9qbhdeV5H4eVZu0amG49aovfFura3M5/REngNeF/ikTuD
kwIv9RTLap7D6JJ4T/4jAnagNt/0EjQmui5sO6iTusEunZ/gNZt9E/znxHTj
e2lLQ2jPb0r7O/+qnfFK5KXN1IcMCjzY7FA+ejR0stC10LRgMn8LeZnH2eme
v7QfnTr5GUu9870G9sl3eKPSp+iequvudt5vN+Z7/+X+cAjP0Jp5mK799E2f
pPgd+zNlUBYyFfClKaEc+LPGwKOxluSE9vC/Bqb731Xp/eomfYueX8/yfXvE
tMqIdNuLETcwxBNXH+IrYl3Xgs6m/WnXt9pXX+rqfuiEbiDd/Cs62VVBx99B
tOK+olU/0Rh9iL1X1+q+5unQv5MGDABjflIII6dELgRGgvy708X4n1/S1//9
OvXN9SP9DH8On/5me4/H6wKdiT7n9EBnwuPekW+bCNLC2/+jNF3Fn64dabku
GC7sOeh/bChI/2uG8aV3hfStYQwwHvhXV4X013R3XtaZKLJYXd2Upkn3RmSh
8C8pfkc8+zXx7KGt+i9XNTktMoCxQQ7wWZbjyNsGP5UebGFSHd4QaLa9Qp+/
Jr72qWBjcou+6ds2t+8nxb8c7CnoL/oRPM5BnbyfI/+/b4h5Y/ji4bmuj7pG
qS3XNfl5cHfbl2JnBwawQ4F5eTBKHQuMU9pRJ3p4pGXL0/XvnkOGovjfExzG
Tor78yGM/OTpfMtVwH6/mm9M3RK15dyRpqs2iug+WPdU1T9fcQtG+hn+elpf
89h3iIZ/pkL9rjE5MNZYamSMyGDur7Ac5j2Fn1S4oZsx1+Anru9svDl1g09j
v84OmAqw2JRzbGfzxCkF5ovBAb4S2jm5l78B+Q+YDHAkYEu4vx7Co3rZxjK9
i+lT4rEThP5k3USGwH52fOhb1tHPA/35mO6P6zqpk200KQcsK7Jh3mlqRR7R
/dF8y2SIfySE+feMUdaP22Mcj80Otoq0D6zLHhkW/U/ZT4byqfOxUA5yG/41
cqTdPFaB+6Oz+vxZ9WdrN/Oi2J5eF7EtKmFsMX7N9zPhXRobLyn9KKVP6eJ3
xP+Sb+w+uP0fQnhPOT+FeHD9P4Yw+Xboekfh1eqAcW0ef8hm4wrMn0bF8+6t
MdS1u3FtxCOzXRtk6/QD5f0ayvnfSNMKzN/SAqdB9s4/53vhK2kXbd1X7f9Y
Y+kFfcve3ayHKCkwzhRsAfYkyBAWFbksbHnAyeTr6hhju4M+BZZHIZsivD3Z
ckzSwEdgr0A52CwM0vzdUOH3YGuoC73tS3GhfZ1tZ9OvwLY2y6BhCv61oSGM
Tc25WU4DPQJNNCZ8L7ZQ2GthDwW2nDByqkc07+8d4ratyTVeFHoH/Ty8PTpH
bE3Y49lPoamQKcHPwq8QD18DNoN34ChKB6lc0XMHdvRYY8wxP6DFKkPeA/S9
hw3Q+qX5PV2TcoPGzX3IiFsUX2b9FPJT5KjMzWv1fnKZ+VzqvC/U272jZS3I
Z7g/GMI79V2TQnrWad6xJq9u1vdhH64+6Kh94QGlSUs1lhocOTaDfyjvl4o/
I8u20WA7sI8+QeOhfYWxWshm0oJ8ZlBUc0zxPdR/KYpLHWr7xEvRR/dye+aq
/T1V5pHsD6XiLdr+1c1wR7+DrSRrDbr7EUr/k9JFU7x2Mn95d4naf5/aP0rt
L0TPoz58Un34cIr3G3jD+WrnrwNss4v9/WMlnu8j++m/9vMcQUdBGP0GF+/O
0z7/nMp4Xle80h+r9t5f52fkO8+F+Bd1fyGEkcOB2QKvNU/p76nzM7QS6aFP
liuutMw68331f0tC+G3VO1X1tgevoLE2J8VrHfQMdpvQJ+i/Zuv6XuEjUvxM
X6ETOzzE98j09/Ptt2lNiFX5KSr/NI3JUYp/hnkKdn2AbUQ+qbV9C7YtCR39
LYyNEWpLfT/vd8iwR4TwLKW9pc5j6+gixy/saowA6YlDXzQy9O3VSntNnccu
+z//hf0d2yjsr7CJ6pXpfqQPF6r8B+tMOzyl+9N1/+pLuLN/QRtCR24Pfb8p
5V8aY2MoP6mfy0fPSx08g+MF3/FXmTEe/ao0vjWufu+h9+rsnkNtv7xaY3i5
xvDiXMtoq5MsFwS3UBXC49EJVpqfm6fwskrLKpBzgt1AXjhb8adUWm6BfgI9
xW49h8q/vsJrzFK1Z6zG8E2p1sE1JFnXlKq8ucr7q9LMVvqZSr9PrmW66ImR
5yIbbkmyHBa9CZgd9CTIZptD/AbNi1vAkCVZD1If2rBecamNbhM6xKFJ1nUd
lWEdMPrf0apzXNTPD2tjv7RYd/XBVMVN07umXMuP90ry+l+iuHtr/Dxc8+y+
GvvZWKx5eo6+sUjz9DWtr69iD6y5kqC1sYeuvj2so6xNsp4SHejgEH5fad/T
1UnzvVppq7iU/lO15xNd38RbTzcktB87ztFJttN8Q+8fU5vfwK9DhucM8wVd
5MjQz/BA8ELwUP/TWJtd5vmDrcbnQ22v8b7myDtDjRNv1ZgZUGYsDhhUbG/w
1TBa8dWsn4qfrXISy1wfuHLygckHXw5GH7lBYw+Hie+vvH+XGrcC/qZbKBO6
7ZBetuOGfoSOhMbcprbMKzU9e5jy7lfmPmB9/byX5fT4MsAvAz5JntT6unKY
7ZXB3dNmdASb9F2vDzVevrindeSswTcrfuNQ22MdovKbVH6Jyp+scJ3C+QnG
SzP/oMP/6GkMH7Za2GORDxuZx2rtAwX9z7HqkwzlPV55emodHlpqnyfdm42J
h/Z4SGkWlXktmKW6xpQZOwnNge8A9F/4WXmzxN+FH4JXS0xXg/fnO8Db47eA
fkLflKAyDq/zuvlKRthDOpjnaw7raoHGcUwv81Pg/cH9Q1Mx5vAR81x701PQ
W6xppO0c0sOLEcbnzCExTo8/Gfg5+Dp0/Uek+t6zk/sHGyJwP2O1Dn8wwFhd
7I2eCzZH0J7QoKy38Ivwg/CF+GXhjm+WbPVbb82pe7Uvv6H58eYQlwk2MK2/
8YE3KO5GXalqV2auv4dvaZ/r+Ycc4xG9f3SIx+Zjuj8+xHi2DZpnN5V6rizX
fF+kuX248t2i+JsVf2EPj23GMHmRjexMtr7tfpXxwBDrdaD1oPl2037BhoL4
IrU/X+18RO3vrH/44BDLvobW+JkwOCDk4GCBDkr1PQquSe3ZT+3pr/YM0fd+
rn88Tu87qsydCp+fZXxfdn9j/K7RWPq02DqpYqUpUvzjStMx1/o/eJzbs4wP
ABMLrqFvhbENBaqrUNc/OVoHtJ7kRm1vBQa7Xa5xsGWkjfoZPCm4SzCl3CkT
2dql1aKrlK407KexucY5gFntnmvcabnKGBD1O+gn4pFJ9lfcwAqn/zbebWHd
hSeh/WBJDlabD9F1WYZtN8AYg5XOV97SCqcHUzQ3+KkYrPjhFba9qFOfDFGf
vI3cTuFahTcrPEXjc+sA46AzlT5V6Xeo7l4Kd1T4O4V7K32S0t+i9BWKr6qw
bUeS4nso/gbF71K7lpcbyzZM+9qoSsu6sPV7MNn2Dt2V9zfV9XmO4x4I8chS
CSOvgBdgTiEnOVNty66wv5NzFC6osI3LRxpHHw7xHGWM7wzjvEXt2Vvt+Vzt
6cX+rzVhHvtBiu+bO3htQg6IDPDLDD8THp3qdNBXfxYZzwqW9Y0Ojifv+FTn
xa8R98UhDCYaLDR0c1R1digzHYYtIzKJ3ZgBbH+TrVNGVoG874pgG4qNKPwR
mFLqxffCX8GGir0C207WXvC3NyYbuw/ui/aRBwwqdOU0XTH670PUhu9KjQll
LWwO6+Ew0aLfhvhXVN6QFOMdByr9oDo/X9XdZVHOdqUtrTP/Az69us4YdWj6
OaFPuM8NdOwXGX5HPJhgcMInh/uMEAY3zDPfAn72sPC9W4rcfmjkJPXfnDqX
+3uR85IPvCG4O7B7uRpX1yZbJ7tT69SqZOv0m7TvfNDfdtTYp6ADQP6/v9JP
SDa2Zyn64GTr5Q8V73Nrrdfljbo/VOt98nytgYuLjTc4CLlrsnX00KbIU9B9
fK/1547+3ovurzEtBR11isb/OZXWn3yqNLf1N6+DPh/+Ad4Be/TmZNuA7xAN
/9tg84DYvFI+shreN4U0jJ2LwvgpSjZODH33XK17VxRbrzpJ82uS2jBC9UxQ
eILCQ1m/FJ6t8H7M2VTbUWNDjW0IYwm51j/qk3OTjaHF/8qUkG6c1rTMcmPM
/9I/ubvMZeBDCT8s2DUgf6c92PnWJNsWFPwYNDlhdPfLwIYlG2e1VHO5R4V9
QaH3LQrf0k913JRsDBvYxhvDOD9J7Z+n9FPV/v0UHqNwucLLFD5G4cMUblA/
N+qalGrZyPEBk4Zc/7Mij3+w44yz87tahvJjkflc7j8VWX/NO8bgCsW30z+5
s8y+s9ArMAYZt6wPzD30JuBD9kq1HABZHvHMxy91/6LIawu6RfSU6Bdf1Rw5
tMg8Gvp/6gUDgEzjz1Am76cXGffzkeqvGOY6+P9jUj0GuI8NYeQw2DIhZ+Gb
4XXp/1O1Rt5bZrsJ5BBtoZ3MKb4FPQvtRKeCPgW7KdoM7UG7fgxtg2+mzD18
9HGhfOQIyG1Jw9ydGubvbpluqmWtpD8+5OU/fBr+BXbutBkbK+J4h44GjBTP
0FHoeWgbOhv0Neh1+J/YRO6ss13kGnACNZZj/6S8izT/ftD99hLjupD15Whc
Ha34rYo/X/GfKf1HSn+V5s6Z6p9pen+DwisVnpFg3B12zmDuSrQO3MV4U3ia
0owsM++C7dJpScabR3SdrPDfKv9vjZm/BnvuQj+dEcppVFseqLBMdbvSzQvt
2U/lP1RhrN5BKr9W5Wcl2OcSY4Dx8p7SzVX6d3S/VG07Isn+gdgrGDOsn9Ax
/BdkGsTtCvFPqB37pBqLAt/M/ID2Q1/BHEdngY8a0jMGT1X/VA6zDpE9jj0P
XeHv4i8211nuurq7sbPgZi9X+BiFv4iz3c3C0P9fqY+fbPT/gF9aENJ8ovg7
G/39yHE3Blnuan37aWXma9mDTkmynAS80z1Fxjwhe0AGAc38Yx+vQ6xj6Ho6
phrXAc81MTXobVXeD3V+h40kvuqwk+R6r8j/9NQs7/3s+yO0d9wV5vsOrakd
9c0T9K4802Odcc7cwU8Qexl2lNjKYkvJHvV7iGcusM6whzKuPw9jG79X1It/
vCO0Fz9YZiwVfg35BnRS32g/2qj4nZ1N7zLP8TMAjTM71bQ3cg3oPHDju+mH
MI+uU75v64zdSO9hnoR5xLpAGmiMKv3bT8qM4a5WuEbXKHxHlfkf8y9G9jAt
Dh0OPn1OoCfB+YATYt2if+kb/FB1UhkPhW8B49Eh/Av8T8QMs86R734/fDt9
807oH9aZn8Pai/0J/xceE1k1PrmQaTNOwVriY+RirWnPlNmPJfvTlBD/tdJ+
U2QcCHLj38P6xti5t+hfXyWE8SFCPUeHuvB9WZZqOXZ9N8cj7yId+RljpF8Q
xh56IsK0l7Xpy7DOs77TBuRy3L8OYdbzQ8Kajz0w8fRtivomdZjrxp8Y34tc
/XJ94/Nltl0Fb8//4X9R5rfhG0m7KKRHHk+YPsNXJGMS2+3vy4xVpe/4jrvD
t2C/Shj5AP6sGM/0Gban8Pzw+9erDW+U+f/iy5N1AFq3NdSBngUbWr4LbCr7
xpdhH2H8Rod5DLMP9Al7AWkPDfsaPizvDO2s1PpXpr08Jte4YfDDyd1sA7gs
hNdq3ejaaLsj5DIDkmwfBb4U7D8Y0/+0qu8qjecCL0paMKNniC+4WjTYcP3P
71XPxxX2B4PNTnGS7XbAphYkGZ+6WnWt0VWaZCw1eCD8dlymuA6N/9pcccem
6HLFX6Grf5KxrP1DvctU79YK+31B7lMS6sLWqTTkfV/rbUaN9W7LxBf9lmhM
C/YW4PnA6WDTiewKHPt4tX+cyhyUa4xLfJJtPX5UOdNqrT9N0Fr6Wam/6Uj1
ycRa46w+U5pJtdbPoh9eVGOeC6xrryTjdy8Uf7akxuViC/RPou1R5ijuS8Bm
Scb27ko0vveYWMdhLwTWMCe8P1zpZ9f4G8A2ZycZ3zy6xL5pWBOWt9oXBtg8
/JE8mGjs/7tq/zu6LkPmpzKfTjQfho3FtkTr7T9UOVU1AUcHjjnRtoAxquvn
RMu530s0Lg3d6YHQe4nWn6DLHVFjujemxPIIZBH4qqmvMR27t+IfqLENFPYe
2H1gS3eT2vxEpf3STVOakTX2hbV3d9eHfhY9z20t1vVE9b61xu1uKrFPB+iW
V1XWxkTL9RepzPE1xqRdrn1nTI3pK+wC0XX06WA7CXAwYGAS9H6p/sUdep6p
9Nk1XsfeiDOWCP/HfRSXVuM8C9XevBrrZ5PUth41tg3D9wplsMa/qOvOROOq
31TfvwEGNME2atiRYkOK7uXcFrcJ7PYTifZjkaTyeuq6S8/zM/zd2D+OxXYh
0f6PscPgPdiKv5T2u8ZgJ6B6vin1nHgc3wK13iPmD9S4G2h5/Cx8FPQ0rjhR
4bqexopflmVeDR03drRgmJGLlgX5KHoo7JOO6mlcNDh+ZJ/Ih9/XPHpd9d2m
ebRLdfftads95JCrkyx77Kr/kqK+2640R6ktRw70/K5uMO4EXMltau/ttcYD
lKttDT2NKz5XaVfpOjvebewX2tkHWW6x7RAHJ9jfHHYQzdgtJtmv8iyNk+dr
jIHHXyr2plfofqHavExtPjbX9gjjk2yT8AS0Vo2f8Qs9O8l6VGxVn6tx/pdV
77fogBLs93FckunDecPsx5k1Hqz8IYrfGGd9yaEhfLLa81KN8fZTtS9MGWJ+
9HTFb6mx/6WB+tanKtz+rQp/rOsLfEsrzaYaY1eS1U/Ta21rsGWo5nit5bDI
SsD67d4flT6/1rhZ/JDhswxfZD+X2O8StNBzqmd7vf9TV42dglrjClKQMdda
9/1piX2c4UcIe4w36Vt937n0R63tsNAr/EkZun8mOvwl5X9a393WYPwKOJVZ
GgMTao1ZXcfaWWuZ9uP6FzeqHZfrXxSrb3cq7x/Ku1+D8SVgS1ajT9F1nspv
1L1B11iFC5T+N6X/Vel7ImOpNQago+pK0nj7CBsC8RCDB9tmcYLuEwebv8zQ
PXOw/RCC4QLLBU4MvCQ4S/oRP8bId/A/zLygjLUaYweq/gN0TYs3nhvZD/4n
31Jfjas1nmqs2jC21vQ//QVOBX72wQrzMPAv5/NNus5ROR8p7/615hGo8+VQ
7wrRsR0HW4aF/BI57AK1+Szl6RTksZ1CPOELtFdeqKst27TO3tm2MUTvu0+I
g04fnGV5HHrCQ7Isj3lSNO97TX7Gnx9+FcjfvofLw38hvAp2NviQwe4GvALy
6quV7xpdfbNcz77Z9slAOfuE8DmaIw832dcr+SiH8Fdqe7tsy9uJb84yroB7
U0izLsZlb+pk/4ykx96Be/sQ3grtrPJPzrJPRe7YAuMT8aQQxr9rbLZtbv7I
NlYPu2Box0VZpl3B2uzIst/wcerL8c1+xiYNP0TIsjrmOAwmjfo7hDYcIvpo
SrOfrwWfr+fHYozZJD+4zYP1/vD+9tfxcJl5FWi8qzVmZoV4bKemNtt+an/d
J/c3TmedwlfqmpRtnC93fOKv0/ulih+s5//qP76mMg/Ksi6ZO1gI8BgTsoyD
eEn/tFZpd7X3vSaEuVMGeA/ovGHZ1gXUhTDrSW6On8HV40sSf6n4k5yZY9wh
+zg6fOw70OODSQEHAy7lQ/XJtjLbUJzAWO1vXChl14W6oPV+iJreA28xOts+
BcBjYCcDJmOLvu8dXVOz/H5Mtn3BYUs3OoQPLLNvU3w/fllq3834nPxNe+bo
BmPj/qu450s9rsCiYPuEjAdbBcoGB06fVIf+wTalKtu+qsEqTMnyt76nsX1j
f+O4ftd8vai/fZycom88tdl5qkM/07e/gcUL3/5yeAeu5qRmY/TB3dOvt4W+
7aXyv23yGP0QWVST8T/guhfmGj/+ttaEYxR+RHP0dPXhRVG/m1WtvtbzBsUv
yLBtMfRYjxjbY2IPDo22T4njz1e+C6Iu61T11SnK+0DQpS/KMC2AjyX8SWCL
3UHtvThqf2LfZNl2Gt4HunhMiWnjP9Xmc6L2P4b9J/VCv+HbhPTQ3qdl+Jkw
fo+PU/2fhjbODukfV3vWqD0/qowv1Vf36/mPDOvX+S58VEJDLwjfiN8JaCno
KGSdRwZbS+j+o4M/CvxSHB3S7M6f4X/NWCI9+n3G39wQvk/fcb+ui1TGA/G+
IzMD93RprrFP69TGh5Tmklz71sIPB/6voIsbS0wb43edMhmr+EUhLbZjrFWs
S+CHwQpjzw2dPwo6udH/AD+medm28cSmpzDbezS4AcLIsc/SfzkW3WG2dbrI
aJDPHMmY7G9/epfBRzRb7kd5+aHMBZofp6EDUppy0atn9Lffs1vK7Cf3gyBv
OTrLcopnREdMb7b/KNqGr9XcbGNjuaP/PV5lzFWaZD3P031pf5+TMF/h5aH8
M9Rvl0b97/EFi50b/h/gp4/Ish59Vo7Xmt2+czNsS8AYA28EPgZsDLYFxD+l
712r8tbpOlFlvhvvO7LD29WXF6i+bQoP19o7Ovif7Kk6k7LtFwjMHmHmGu3B
Ry32ameLfj8l9CFtTAjtPE79ObvZdBr5kkNesMfk242Hj3cdlL9QZRze7PyM
taPCGGPus5aC30Yvenuwy75cff9sqf0KTlYb3ivXN8CniOf6otw+c6Yofqqu
5BzvjzE53lvBGhFHevYu9jDwVMjTxqnfBnSzTXRWju2iWY9Z65FHpereJ8e2
sE/oG+OanKd3jt/1Up8/keUwad5Tf2+qMLaC/x+b4zEwQ+36vNzlH6bwzBbX
h0/jXdn2pYyPg8Qc6xC6q574Jp8tgp71tSzvHVyvMvZiHAdmfXvEca+FMNhG
8I7gG0vUz+XNXnvB5xGPPwr8+78a0uO/oaq/fTgM0r1Kz5uy7Bv7wBbvI7dq
zIxsMP59k8JNDbYbQK/wQpb1AtyfD2H8SY9t8XwhjvLu6mT/Uc+H8KPgdbNs
nw5N8me2MfPgesG3oRPEdhwb8j7sv/jOzHSfp4RnwpzXUBB8cq6p9hkqnJ/y
uu4p6sOZ0JDQk7p6dbHf+YJQJvR3YbBPxyc9vsuxn/pA3/ihrpxMYzxzgm9P
cBLUS5vJVxTK2aa02xv9DO6mvMXYm5JM+xElDfd+IfyG2pau9hyVab6NNnIG
DWcs1LZ4v2Yt/CP0CXzmMZnuG3yKLMr0f0R/Sl8t6+j7MSGcleiy27rZ/8lH
2faFcgn8TpPz1w0z/Q3tnae4fF3HKn5ntuuFPnxT4/npCrdtx3DtQ+UeO+B6
f8q2LriP8qU12aaQuf59mO+05djQnnXKl9vidqxX3g26FkWMsWAc8D+fzPJc
+kL9/2cP+53RdIhcB52u69CIfZ3ii+bWjr7A7WIfhP3iJVnmYcB2nBvSgO0l
DAaYfCv/XzwyS/L/AWPd7Hfg2Ndkmdf+PcN1Uy+65CuzPCaxjY7rb/tocJPk
wf9tJ/aTMvuNwCbg0iyfIwUufnUoE6wDZX6s9BvE9/wd0rPGYRsDDiRZZac1
mwYCM7E+0EJ3ZHk9BH+C7fj60IcnFyuP0q9Qv92ouJtD+QkqJ6XZz/jfeCDL
/jQezHIYv2dT4v1MmPrxhYE9DjY1d4Uwdd4Z1uFUaL1mPw9KcJtoDzYhN4Z6
sS/B3htb73tDmeRl3jwa5jt9uS7rX6w+GHRk9d9mmZaCjsLe49sQhlYiHt80
3L8LYWiYT7OM54TWZlwiG2FsEgYHwvvPQhr82XyRZb5qCn42siyXZtw9GcZh
P31jf33j02Fd512vMD6fDmnuVZ+XNnus7qf0eze7Tfj9eiPLfnDBLW0Mfb42
fC8YNvAltAnam7M/PsnyGRzYfJEeW7A8lZmnMh/K8vkFnNuBLJ3/9lAok3F6
WRhXlL029Cd43thmj03KJy9Y9CXouPvb3wxnjGwP9Q5T2jrFnxdjn4tg5LHj
e0njc2iIp5/eCn0FBohn9uI2vW9q9jfU694UymcPQh9IWb2Dzzsw0ZvCP4Pm
vFrrysaoaUfomsNzbEuFnzzoQmjCi7Q+PaZ1tUOm6YKZIQ3v8aWHzPM5vX9e
V7dM22UiB98tA090vq/AjWTYfgm9LTZMhLF7QvcYUZov4zzf/whp8A3KM3Of
Mf17iGeMM9aZa9gF0wZkI5T3a0hzodr8SKPLfVD3hxqt5wSDTxvB4V+uNC80
Wq6OT3Lofmj+DzIcxj/5u+huQ/gfrZ0R9dU7+rZtYMd0bepgv1O7/U/FGIu8
LZQzOtc+BcGMrYZ3aHRZ+Bv8MOS9X2U+oGtFjuXx3LGBxR8h6fBt+HHIQ3rK
xf8e56egE+AbaD9x20IafPNtDeEd2ovPLfFYwlf5u+Fb8Ht6R7ltxjkrgXf4
8VigNh6ja32c5YLLSiwbBJ+BTzf4C84WXJdhP0LYnb2WY9vD9QNVzgD7PcA3
4xvh29GBowtHZ0Tcmxm2Beb+VghfoDovbPTzlgzzlsRjK0wYHnMh/EejeclJ
2kN/bzFfgz/7N0J65Kpv7um3An8X3ztD+Q5rDO3OCOcjQsOrH86qsA4LmzN8
l2B3hr9H7J+RIeC3El+MYDCeUZ/MKHHew1Xe7EbLd/FDR17kveS7MYQp84YQ
xvcq6ZCx4K8DDCB+OnboO35rcT9iT0r7WH92+2oK7aT+K0N4Wq7bj10qMuaZ
JS7/gbDHsI/Q3+AO+HcPqeyNus6CtlZ7r2702EV/i09GdLhgNAlDn3OBswGH
QFxZSAPOMT7X/xQ7aPwYzoR2Vf8NjPodPj8Kc+33Y5XW4X7IB9v5XhTCnH/F
OVicl4WvlSG6EtvZdzu+3bFVx74ELDhy/n2h5VV+ca59Gf4zwP4MKY+4c7r6
W9BJ0+7qXJcDLYrfwb8H2IcgvhK7VNhfIr4Afxhgf4C/aY4cXOJ9f7HCh5bY
Tws+jdIq7J8In2DoNtnLkHd2DrLQCrCiUT8jK8DPFDIN/I6iy2Bd4r5mz7/O
cRrkEPivSqqwDyvwlPggxCchvgkJsyzu8TFKXnCc9Dk267+q/Z8NsI8abDyw
20FXwvmB/Gd0erv5qFz7mKoNfU0/Yzubnmuswu4zXzLMz/ZFXqBraVef1XJe
iCdt35D+A83xaNTymzO1lx1YYj/qjfqO4VHnB+PF97J34PuF9RL/L8jgV5VY
Dg+e46sMYwyuUT9c22jdF+9vaXQa8J7gEcEi4qOH9kBroWf7b4n3nfWsB40u
a2j4RvokLbSZb+GeFtrPmCkM47AwjEnGz9/Z9uOJD88D9C0H6ltqcu1nsTb0
G+USxu8i46smjFXOf+McOPwyce/z/8KcC/dbxHbHKbmW2Var7LoK+9KsUbhW
V89c+0niTvwIdM5R5+HcTNoPXmKk4uuirgP7UsrHT88eWRL0FfoD5j66GHSB
nAOE3gdZ99e5luvCm38Vwt+GeHDxaxP8TPiUFPtvwnfTfxLtBxNfeKfiEybD
mAtkS/i1QaZUp7kzvsSYFnwQ4osQ/gUbZ/x5sV9PDGF0XuzVB4Z4nglDD7Bv
js31PgbeEj9KzEfOkTsu6rGAnmxiKAf/g6yJrIe0m+9C5oDfJfKD1UT/tzXX
/uHRKxBGN4H/5A9y3eZitT1ZY2mo2hUruqlLns9wjAlhfGtwthV+/PEJSRnb
QjkfhXIWBrzN+6HMOJXXrdG+YK+qsc9W1h98NpMGOcyr1fZnxzqAbRD140MN
3OhP4dt/Df3ANyWybjTalgi/IDGhbej7kWMiw1yocXKG+moW65LCc5B/Kjxa
+cboWpFhX77EQUvgo5I4fHo0KG6krp6KPzfBz4RP0Z57aNRyV2Slbbn2vQ8t
Mzr8L/Cp+4YwfqNJR5oVyNXUjmvj7DNkeagLW6nlYX+fofczoy63PrQBf3pb
Q1tp54vN9k3M2EKfekiuZT4jQx7aOSbX7WGfwhZiZu6/8oaZYV8g36G55h2I
mxXi4TMOCfH/UVvO1DWHvo/3nfbPCHmvCHrdGSE8R+2fF/XYpT2Nof34VW3J
tU3YFKWZEvW7g3U/JOp3+O5uzrVeA/k/34KPwS1ab5dUmB47RPdpUafbPafD
vMYnY2wYq71S7T8Of3C56qe8Rtu6rU20T0Ps8zIUl9loG4dlTaJrovaRgw9y
fJEzhvEZjX9z7MKIQxaJ7Vi28uU02lcZ9nP7ZdjfFvJG0jCen1Z5z+i6Mtfn
ON6Sa934ixWWzSGX40wc9jn2uGrNu4JGl8W5DdTNPOU9ftTAq+/ZFwl/o7Tr
sYnoZtu1YSVuJ5gJ2oy+7zTFnal0mzKsCwUTiT4U3ThxD8cZE/9Uhm170Y1z
rir9yflf9C9+LJcx1hqdDl90yxutYwefzVnO7LHQM4R3+0AotM0J9ibY1+0f
+g2sxkEZ9uWL7A0ZHDI64g4M8aQdH9JT3vBQJu8PCGkGtGp9qLTPuxfi3cfg
rT5Xu25WnyQp/I/WSZxK4Z95icZPeqvP/GLd+jPXvow4f49z+PAHe3GN/SND
C4ETIh7Z3cnKW9Dqc7Lwwb8z5C1Q/X0VvyuUtyvE71kr8ak7RO/3VbpY9VWC
+nWHyk/KsF9D7uylfyhuZ4PxX9TZPtTbW+l3KT4tw7gw7vhixjcwPoKhPb7X
t/4Stc07PpquzbUNOPK9K3J9ZhD4oIYSY4RGgolVnxyaYZ8z7FnQrm+rjC1R
j9GbVOYTUefHHuvRqGWN4IbIB54dfhuZBXge7lND+H/K+5LSX5Prc4K4c5bQ
HYp/Oer2kXZaSA+9wTNnvlycaP+b2I7zTLnw+D01Nx+rMJ5xi3i1iS2mVcCQ
gw8FCwoeE1wmeNiv9X1fNRrrylkIFQn2pQlmH7tDcPvT9F+m6hqZ5/MuOZuS
MyrHlmldr7I86TOV8Wmj5aY3635To+1skA0MTzDvf7zKWNhqP1zYsoKDAQNz
fILtV6n3omKNpyrrdE5W2pOoW+kXK+6CBpe1v+qdV+X86NWoH1kWGDp8LyND
GDtIa3GVfS+AI8QOBBuQ/6m8JYofQR6Vc3GV3+GT+NIq0wWcLQAGEZ5otNLM
rHK/0Of0N3zKzSpnSz/RUJpPL+lbX2y0fTbn02LXiH3EG4rb3Gg5yofiGZ+t
97ni+DzDpwB+z/A3BaYWGdGZrcbhsSfvVL4Xoj57Ct6edkGr3K8x+Y3m2N/4
d+7iO/4nwF5ja8H/yovapyTrILhs4nmfoPiigfbd3hq1r2T0lJzHiy8w6AH0
wZEetkt9WHHv1/v5cc2Lx2qMH3oEnFqNbXGvU95HK33G7JUqb12jZQ+JKr8C
HGmC/WxiZ4KNCbbI+CfGnhdbVexswaOCbQV/DD62Xv/upzY/YxP8Y5Xtghen
+o4NZnPU/Dy8PGe8jxpkvx/P8R2DfGYl5+iAC8cOtyhqG+a9go03tkDYAd2u
uAtKLZ8fqzRjovafPSNq2gW6pU1jcskg68vw91A8yD4foLuguRjLl6sf1rRa
j3eO7jcFH4zwp9D76CAa1R9faX4Wq0/26uo7vEmD4g+ush3M6Qm2M5gVsU0M
trnYxWAv3S+EkaljK4CMC4wduCT8OUC/Q8fDv4MhwY88ugj8DcUm2L5jcJCd
g5HG3gJ8OljxD/SN46uMCVmabbwhtBb0ZWmYp//o3/7d6HPYKlnLSy3vZ0Nm
D0GPPVTfMknl5CVYtoUNxkFKMkTxE6tsv4Wtw62t/oZmhS9tdR25+hc922z7
Xqi43Fav76Owz6gyzq69+q9dk89bYwwRB+4OvCo+ztmLRrca78j+ia0YNs3Y
i52n+FWt9m05QuU9X+9+AWOLTTM4W3yL7l1lH2ngQBYFeTQYLOKw0ykH51xl
TCJ+5sAlY2sGPQddiQ1XhupJa/Veh6C3vMpzYKX675xG2ytfofvljZYdXppo
Ww7kJ5co7uJGy7TggfGnid+tE1XGSVV+xv8Raxi2ycjP8B+KDA0/K9BU6HBH
Ke2piu+d4Ll4QavnI/hp2g6W8nOtYbVV9u2D32OwkshPPg86EvQjVcp7tuZY
nwTLUHqFvNh19AnjELs67D2YUzXY7KrM9ARjqz8P/n/Zl9mTwWzjA5b30yPG
sUGrQadx1jfthafrE/X4YGyA20G+gCw5W/FNavA4znQSH1TRap++4Ei4s96c
neg9EDw/fn9H6blU4aFd7QOU+nfj9kI8/qFvrbDfZ87wHRPKOVbxRwww1vhQ
7aXby61Dvnuw1s3Bto/n/ISlA4ybPk7hmwcYW40vGNZT1tKvY4xlRi7xg/J9
P9g2c6cp/bUVxi8PU31TE+13drrWmaXVPguym97HtdlfxVx82Q4wbnfGCP2j
Afb/cZjiLyl3GfNFh/xWZxu/U/GbpfydsG/QWJjX4PO307V+960x9voUpTlH
ad4EV6/wmRXGKR+o8OGlxtUiq8VXGmfOHqt6rqz2+bFHgBlrsD/FvFbrcaHH
Jir9fcxXffvxVd6/2bt3dLGO89cu1rHnZltP3avVcjT4VnDe0AtgqsF3c84L
GO/9te6OKXdesErY6uHnc6Ta+XuJ8eYVCq9V+/9R+9OUdvFw++uqVPzVJbYx
wvcCcfgWGKj4G5S+ezeftzOx2pi3esVvU/wixY9SP+2r6zO1pQ3ZabX9uuPb
puNQyx2LFV4B3Rbns2j6Vxs/VkJ8ie23vh2m/bK/x8M6jaV9Bxgb8oP69PYE
j5GEVvsiR8bSvtXyEWQCB6veaTWmccpU5rmq613WZmTRJbbNAoOPPgAc/scq
86YEz2mwg/jHxGc3/qhvTrBPavhS/iE4ros5x6ba9q0PqLwLqm1fc1KCMdqc
pcLej+0u+GHwgZdkG5eBTAZdCzqUKzQ3Bw/0+R741ATrmsU6o3H0eZvXgkvQ
HTVar8BZOxNiLA/BN8LzwU8CNql/BZphZbbx5sgM/2I8Vhm7zf5TWuU9CPsK
4pQ0sqbAfsHA6oC3Xd/qdjylescNtO4Umyp8k2BXhZ+Sb6v8/Jr69fuB9ruL
Dh4sKbQE+yS2VeyViUqfNMjP90ZNB0ODpio8Qnn3VflXqN6LB5n2elLxm0uN
YXxM4Y8ajYfAdga7GfZl5GbIzDgvgjPAWYPQoa9RvosGmV6BT4VfhW+Fl/sz
pAGr/VvoZ/Ri6MRYm8+Keq7yj49JsF2Igrvb8uMgtwfs88NR45/xf0M90EfY
TOMfApoZW2rkx9hKrFTakxstR8BvKHpC9sT9qrzfsHavSLA9370R+8W/daCx
2IuVd0KjZcerFP5woLFF06M+CwTcGHQE7QLPNiXYUmJ/DU7/j/B/sfkGp47d
9+9q/3HD/f2F8ARai3b0MCaFOxiMP5WmpMr24qz1O0Ne/MyhwwanOiPNYxMM
M2c8cEYFvPwCtW1+1GdDsS7nBdoM/CZ9BhYOX//4tUcXxhr6kMKPdrKPa+yc
OYMCmfeYao9h7D9zA00yQmPzHI2f6XqeqfXgigafJQUWFZwmulf8IGHXAk7s
TNGVk7H70DyujXE8/pGwvSAf+lnWXtZd3qHLx4YJv5HYx/COeM41uSrBeTiv
8pca26iBE50U0v+3wvsHe8exatuuVufH3g1/TNgTgSUCn4s+EXspfCdTH2dh
cbYW2LD5QUbwjcp8Y6ToSjDGqbZZwVcgYxq7J2yekC2AjQF7wl6J/AB+Dtkp
fju+HWw6Cb+8+KqFtgZf2z30w1WKOzXVdb9S6XWfNX/3+W3plv9f0t+4VTCr
4GHB4dHX2HzRdv4pWHP2a/CH2MJMCm0Aj4w+mG+GrmA8MZbmhPMB2B93hTtr
JudX428Nv2uch8maiL/hP8JYZDw397YNFr50sC3E5wO+H8aF8wXhy6F19u9q
HnmD5vQ71X7mGh/iSwJfAY+AvcHeXc1fXFBu+wFsB6AboRnRZUMvQePsPkur
h8OsJ9jgduhmO1zOI7293rwU/vPuCT708Ok4M9v2F9g/4wcO34+cu8Q5ifT1
0HjbdbNmYifNnkxa/JGxn+P/DX85+GhCR4DfphPj7bsJHz/Y1uDnp1V77HfD
7C8JOoy8+EY4IdXPhLHlPrSr7X3wp42P63lBLnV0GFf4R4FuwqYVX7n4en0p
2Chxh/5HNo6fbPxm46MauTO8EDZTz4e86DTRHeO/BSwiOERsipH/oTth3diq
9WZNsXE4+IHEfx7+IeHZ4def6+izsF8IZdKuTSGMjJ+6kIti/80+jw04fn/x
BYzv34dT/V3QWsht0S9hX/OK/tfQNttglXfxuIM3fLaj9VLopLApeze083LN
wTWJfkce+EnSvygea5OusVk+axW8I1jHM7NsV4r9LTh+0uO7G76DdiGHPE9j
5Jti26BjG1Q2yPZB2LuPGWSbd+w59vhowZYWm3PwbIxR+Ht0fNi7b46zzTu6
yQVhDQHvTJg9ArtU/H/irxSaB3oH2zrW35fD/J3S3zYJrPPPV5pmgl7izHPw
XbttMPsb4wvN8Fyl6TZotmWpXr+oFz3pMSGMnVVq4I+wu4LuIw58MXpKMMac
C8C5yfjW31RpTAl4EuQ4+H7GjhJ6bVYIc+4YOB6wIZzhgBwBGQJn226vsZ8G
xgF+7PDV/6nG2J3F1vuD76ROdKDgHlkbkW2CWfs90Gf/1TpQ0OJ1AbwM8eBl
sNVHjoaMKC/bNDn4amz1wRqzX3HuB/MLewfOtsWvLDgQytorrDP7au+qarIf
iP1Uz5gWY3nBA6BjByOAvh1fEWBlwczC9xIG67cqYPx+1vipL/G5W/hHBLe4
PWJMLb5v8dvWK8whxjN8L36S8HeNzRE6dp6fC+OeNNcHvQJzFf+r+D9lXoKL
TB7mOmgjbYEHxw8s6xRrVFTfVdxkH4fHaYwsaTZmfc4A84esQdsC3gu8Vn+l
fVj/7nT9h7dUfuwIv8OfxsjgixLfwOALkfcOUvnZTfZbhs9f9gHWvXHKu6PZ
PNPm7p6jfM/2CvM88Dv4n8V2Ef9p2FxyJgRzcrz6/oAW81i9RP98V2H7uLah
Xk9ZSwfpPnCo9yd8tq0LNm4f1Jt3gkfC1xHyZWTLnOWI7dTJXX2WF+sB/DV6
JvTJ+LI+HTxjs+1jsP3FRzb2v/juYU3FbugejeW7dbUlGbvMXod/lV3Z5tsp
l7OduX8Vsf+HEcH/JL5y8XPLGrpZbew0wrruF0WjbYr6HC30KZxli/7jMaX5
ebh1K/j44Mxt7Elv6m8bGPjzfdQHew+1H7oq9VlNi7GwY4LvFWwP8Q+BXxh8
RHCeBmdDoP9Cjx9TZJ069Dt9g43h+gGWBSAH2FZv20XsFtHJ015wBcjRGaPs
C8szvWehgyjVWHhI4+dUxR0Gnlfr7Exd+xXav3NTpu1Z8M2OzQC82vRy82vI
DA4qt9wAvRg2cfAN+D9GxgnOME/tODzecpgP6+3LjzGHLIf1AHkOvs44Qxcs
P2cqVg62v81Fyjc+3meXwCv3Lze/PKXQeu5TMn2eCb7dONPkhEL7A+XMPHSF
6LCRT63UeHu1wuMfmQzyGGRB8E5DQxjZOr7n4NHgzbaGMLazbV0tk0TeAV0O
Tc48QTaAzB+ZH/HQ2B9nmxYnD7qfHSEN+G3w0exN+HqpHOh9nHOxkZ2AuyZP
a6hrVLzro656ffc5w+0LDhk2vhWQY3+iDv5sqP0zbgi6AmQFtdo/N6ca6zRU
427YMPtWgPeH74fePby3bRa/7OizYjjnBWw25wZgV4Mvf7DZCwKeHN/b7H3Y
d2MfsNtHssKdsowTB29+FvofXS05xq5QPziWa/Xfr9G1Os3x6Iho64maQyfo
mpLr/wxPgEx+t0+lZPtZxc8ffsw4B2emxuphul7O9FmqYCnYv5HTfx/C+O8H
0waunvqwy8Yma5Lmx5Q2+7HopjUjvtl4WOyPOFcDfRs+CJHhoyPB/yU2reCb
6jQ//qM5OVX/+ahS+82kfObJ+E4+twX7XvSL6BaRG40b5PGBf2HsWVnboZXe
KzHtg7+rY6L2eYXvAM6u4d8hi8LOH3lUl4CdoFzOVW6KWr/4sNpwaZPtI+aq
bUe1Bj2P+n6vFtuunK95dF6DyxrX4j2S/RGeH5tb7G3hkwmzhnA24KflxgOg
A/0i0HkHtXh9Z23nLOglUfsj5KwA7HaYM21gA8uNZUIGDR+FPhc7cPghsHmc
57o57J3o52+rsY4ejAp4F/Tp+NlNCr58kddDv7A+cP7Iru72uQ3vCl6AdfXs
CmOzwGWVtJhXgUe5QGWfX+O1B34S3hWe7UXRMPdXGuf/nsIPVdr+BRtjaDTs
dpHdoD9HvsHZBNR7t/rsFo2X9bpmqa1vldknJrrXAaq3pMHjrl+T/V/Aq76i
tP+UmtdhT2Jvwg8SPB98KzzvDap/VZvfYVPO3sm+OUJlPllu2gKdBHwFOm74
wgkhb53SDGkxhv1m0WU7W+1/Ap3ncUHvidyNOL4d++mTKk33Xi+++MaRPtPg
oxz7g+OcvlP0HSfralaemSrj8Fafk7Sh0rpw9ODoeN8u9xxGbk5afA9yNh26
dOi2QU32FYKfEHyBLK90n8DXoR8D74Q9fZdS83RT9R/T2syX41/k0krnPUP1
r2j1uVnY6/9ZYhtbZIv4JQFHdJbSntXmfkF/ge4CX3D4+GTfYD3BBp/5ydxE
Hv9u0Akg/95z9ho6dPTn+N/AjxcyE2jRdH3vReXeJ8CKgxOHBoW2g8ZDN3dl
nu294M/OSLAfZPREyLA56wP+HB0471nzwVtdWWPMFWelYSePjfy6POv5KRc9
BPgp9gLwNeCeoX3e1fx4ot78LudVsG+xT4GrgTfjXETOe0wO45mxBx0BzXar
xu0rDcan4XcI30bs40NUfmqb/W2A7SMvtAV91i+sa5x/TJ3w12BV0esic8C/
JL5yGQPoe3tE/S4u6rPfwM9wbhDlMI84X/fPJp+vBh4N/hM8z9R4+61Gjgcd
OrbceydYafhf8NKsv6wZ8N2cv3B/oNGxgXur3jT7HxoPj7Sahj05wf53kHt8
o2//qcG2DPiSxXYe2cIt5V47WDcuLbdug32XM5TAEKPzfRcca4Mxk/jagFeD
Vry43OODscH5d9jdYXPHefKHRW1/WRvveYMs9LwCtxGegnPgrglrC+efEOY8
FPwm4BsO+xf0SN91s+wGPRjjBV0YfNVV5eaHxuvf9W6z3Ij1mrNi4M3Rd8cF
/8+1oa/QWWc0+VwZ0mU12S8DPhn430vCXvBZg8814Xlwi+lUaNT1Wmf+bLU8
EJkfd2SG6EI5vxm5x25eLdk+E/BBztm2yFgO0T+9qcQ+7fBR/k2Yj8+Irnta
1xR9a3rU53ODT+ZMYmgg6B/8Sdze6r4BU4VMHbwE50oybhgz6G/zgw43WfeU
VuMRsbEBEwOmBd8YhS2WVyGbHx/k89Cm0Kj8o7Ni7BuXMy0bo8YTg+cZoPZf
Psw+EsoULmm2rc9Nmsc3gvFCfqA9ZVKxbZPxm45PT3ynT1cZvwx1HcsyLSvi
TLTv1B8j2qxzwhcgZ+/ixw89BTZC6Cp66DsSW41/xQ6SPmBsPBFsjLA5er3c
ezD7by+1q2ezbVDA6+4b+IQ/1K+v1JvOKlN55a2mL05Q204c+K8s7MTQn60q
Yx/xLVfG2L4d2Tkyon2zg88VhQdo/JTrOiPT541wLgj713V5pt1Zf6Hr8dkH
ZgYZ/EdBDs++/GW89+a/En0GOHIKfNvgj4f1Ex88+MJhzOPzGFwgmMAZatvO
4ZYT4QNmTdRtQk9b2mL5JbaD/GdkBZxDjk0XcuJrY4yVZA+hffQHewQYevIh
23yw1bg0aJj30B3pyso03uereMtmj9X71GJ/Zzul79Dq9ewkxedqnJyk+A7i
F6NqT78+9jEG9gd80DS1f3KzfWOgk7ku3noZfLQkl9pPC3oVznZFx31egn3P
s1Yg+8G/Cf5M4H+gCfBdgw+Y/zTbDwz2CtyRRXQBnzPQ6xAyUWweoMk/x38X
mKUEn4/CedD49xipMppL7AeLsXhSGG8bW423gw65N9t0DHwffsQnB73ScOWd
UmK/QfiC3BZvv22cj/p4ic9ILVCau4Ic6SLFXzzQbfqiyvZy4FqRQa6MNy6d
vZM9lL0Gv9XIApAD4Mvv1Xj7VsWHCPQ4djEbVcYrA/2ONapH0GWwDoFHRMfE
PONMXHQ9K9NM/7FXXqB8Fw503Welea1nnecsI2h0/IRwDsBvVf4f7fUtbxab
nvwn1j4mkG+ju4UH5awxxh7jDt3E+1W2wwRPznksO4I/3l9LrANG/4svCuyI
4OeQ0xOHfAa6flSg7Y9P9fkJyNZXio47rNjyKLBO+FwAl/W30m4bYJqI8xZ+
qnKeTLX5siArfklxlxTbjj52qHXw6N+jfSzvgWbAjyB4KeYItBI+AuBzoYM4
Gx75/y8t9mmOP/Mb1J53iy2fAcPKWcjoLM6hXwd67HbUmOwQNe2JP3x86eIn
Hx4efw34apivMm4Z6LlRrTWmRtdZmT7PAWwSGKU/WnxeL3v18mb7dYI+hPbl
DE50Lr8p3wnFpvNZaz+q8HoLlrQ2rPn4GMbeBR8a+CpjvUMm/qPG/PGV9g/3
g8JHFJuH/k7hGcWWAWfpO3qrzN9y7D8MuQ+8SW/Fd1X8jznWua1tNe8CPY3/
SvhV/EVtDD6j8B8wIcGYHs5+hd9FznWZ/lH7qN9h14BNA3YW3QIvRF2Pquy/
qy1fT1HaTuiVc8wT/KfSPECx4ssr7LM/R/3wR1g/b1b/3dTg8yqRzUMXQEdA
qyBXAAu9ReOoU5t1UGkqp6fK+TnHNAG23Nihc7bwvU0+5wp6nXNVoHF6FdkP
I/Zr+H5D34nuAUw+GHxsANGHoDtFJ/K12jy+0phx1mrWaej8K9UP/VXHFH3z
UvXnmirvr9h+wWfutv/qZnsAMPYTi70fsxdzRvrgCutOF2fb5xz07PPN9r2C
3xXo6AsSzM9eXOy5wbxgP1wR9MvJKqd/hb+BeQ8/ydzPVHhTlf1pQG/hJwKb
ptWFxlyCt+Qc7JwK0yzQBLSd9ec15Xu9ynX/AH1e5W/gHF++lfUN3Axtx2/h
ZvXPa60+Z3Uv/Yu9K3zuxFEKH1Fh2wL4MGhl6GT0JOhxoGeQYaxuDXKMntZj
PRlkF+wx7C+91M6hJdbbQOuhA4bO4xz4JYEvOAAddYXt2EagT6/wuRDg6apK
jKk7WnEzaiyr+lRjc2irecS1hfZDik/R/ZX222rb4HyEX6A6+/h+SeW8qOte
pUsEL9Pmc13gOcHFIjtCDo9MAXkC545ihw7vx9l9t44M5/eB6xxkbOd3Cg8J
WBFoCM5dxn4BuyH0gZTzlPrn3CGm4ZGDrA1patXOuBpjxp5VOc/our3Q/ixZ
n1mnkVXTR/TPaVHvZ+xlyAanBZzV5VWez8xl7HKQ/YBF+QEsTIPzf6dwrybL
ppFDc8YD+x2ym3bZ9pWBrPrsEP+y2vKu6LQ5vW1rBE8OLwavyDl/rCHIjM/f
E44zXUg70fcuSrRO90+N4dga86FgZ5KqjFFAPnFjpWUU8NlnV5rXPlPhxZU+
2/AM5leleZUVCh9Tad1AUZVxEmAjkG0iF0W+iX6HsYCsgDG4LIQfCHIa7ATv
oZwy+5FiTV2eaHn63YqfWWZ/V5/Rb9XOfydYtTL7b+NsW85FQ4Y7UnvE0+X2
VwFtzzzAToH99ZIwlnaqjL6N/+qHuKOX2a0HCfrK6zUG/qk2vY1eZXVIgx58
bqJlArt1o4leP3fh077Ja+pSjeGUGvuyPVftfLjS2Ki9g0yBvM3wkRXG3S8I
45IxSd1rQpnw6CNKzKfD80VLzPclKpxRYRrnHI39VSN9pivn1eJTHn0Hvvfm
VprXG6O5OK7V/QQeAlwE5yeCs0GvzHrI+fLQcNBv6Hj4n+heF1T6H/MM/gTd
J9gF/k23IJdrYOz387nh6GHQcXPePecqIM8kDn/R0J3QnOhhkeciv62t8BrE
+nNcpccZYwxM1uDQTs7AYf1GFlEFRjhqO7/aSo9FxuFT5ZajLQgyb/Zw2sn/
52xbxsCcYu/xvMPPLnIN/OX+D91p1Dbz/YLsiLVx5wDbroC5uFV1dWmzfGG2
6pnVYn9AyOSGB7ncTJVzStS20ftV2o8SfbRA+87KEvtcZ9zHBplGucr+qtUy
eHxGsiZTNz7yOUsA3+DoSFnDwDnvU2k7FvIPqbTtDVitsgaf40gecPOck4HP
cDBw2JvjiwyeYGWraaAT1Z6r1B58zNcqb42uxO7ek7mzL4OzYw2l7qOUfnmJ
/YAim6C/kbFw/iG+2vAjlFXsvZx9HNzrHjzUIuU9ptm6X3Cv/wQsCD6SoaHB
1mEDBe0I1gxbRPQ06GguKnB5yKOOQBbbbP9Su/unS9gr1fenD7DunHLpD3Am
YOUyWlzWCq3xscN8Hgzy3fggh9mab79O0JBnNtsmAXsEaLTUIDPvxNqu8j/O
8dlsswINBl5pQ7ExS+C2Tg/YreQ4ywThZzsr71fKu1V5P9Ac+bDK9qxfKN/6
Ks8R/MHXD7Pf80lpPtuG82uQHYCrg15dpXXupiq/O1fhm6vcDmi0v4tNp20h
zRDTIfGq93dkWKq3r9IcXGGZzcKoeRL4EXgydPzwZdvFt544wP7vZ4FBrTGP
Dn4LHBf8DDwEuBP4CPZJ7PauDPOYb0I3A2Z8XYLPtvpIaT+u8jP0OPJ3aPLV
2fbBDA/ZR207sMJ89nesewmWgSAb3qvE8uE3m+1TjzHcTvH7q68Wqq+66lt+
hN/RN/ZU+Lkq209hp4i9IjRhXp79XDOmsQ3ELpB/eJ/m6b2N9v8xuq9xqPCc
7Dkzgx43VnXFVfh/luTZ/yHlcjYSujJ80O32RR7WOvZt+FX2aPS12ELDz3Lu
7mupfn5H8+xXeBjN95N17znAGB3WbM47x27iIfq4ys9d1Ya/Bnie9ACjXWGe
D/qosMQ0UrzCRSWWQfMNfAu8M5ju/BLjupmvyEaZs6zxyC6heRKUprLEcljw
8qcleN6xzvUp8VqHbRI2VMhJoEOSgiy0M7qtAeYRuymcUGEZHjIp+BZ4Fs6D
ANMDngdcNphXdGQXa+5c2Gq7t2mFtv1DTna+/vML+HgPMjnOnmL9RueGPhy9
G3Yl4L4474P9A74KPA80L+s86TcW2ofIymyfp92r1nwOmCxsVnbzHo32ecf6
iu9J/G/if7J7hml45MbwvvjGg//dob399ybrQsGj0UYwaegMKIP1ubjBdAw0
DDIwMC/IwfB5DIaU9W2F6ny9xmve5gKfl4MdIWML/5qMW+xfP63wNzMn8NfP
vMCOBL9QrM/oLzkjBzv3o3sbtwamZKP670Fd21J9vgP8DX63fgoYHGQF3H8M
bYMWgbeHr0cGuSrUdVWuZSbITjj7gvUJPSl0ML5W61TneWrjc/XGDmFbja9/
fDLhMwTZHlikfI3zvBbryfGxgd9+/PJxbgg+AvAPcLPKeHqI7bLBLVMeZx8c
nP2vLRpnTHEGFb7FOPcDPAE45KeU77RhxoAgX4wJ+j/+29bw755Wn1+s8dw1
0/Qm78Fznq34pYp/WW2Oz/B5dHwj+kD+FTrBnKAX4v9Cx6M3g2ffJ9t6THwI
w7d8qD45tLf9l++hX7Dtob34nsK2ckqwrwTXjq9BfOfiVwffNB8GW2jk6ODk
+TZkgsgDsf9G/ku98Ijs8+x92Asio0KPDGbp0IBbgv7es5+Dl8SeAVsG/Irh
j4wxhO9G7GXg5cEFLAnYAPwO4dMH23ZkhMgfoNk4V4QzOfB/CIYM+yv6hP2T
OLBb4NHwWwl+DD+C4J5YZ+D/8JWMLy9sF37ub/uFjer/L2qMK+esGM6YwT/U
zgLbY6MrA+fHmkU54FtYX1hb8FUFvw02Cl/+0KbQe8gewPRxlsShqfaVybhC
fod8gn2fs2uQVVIuOCH2UvaFlcM1tgZ67i7ItSwLuiRT8enDjaVF3v9xmeUM
32dZZoHPsajWhveH+Wx38BbQoGCMkVOjr0Z+ji9G+gW9UUof+9gFO8RZGZwt
gu85fIDhNwOfWE+qnoRhpslpL/wE4V8yjJ9jjDcPN08IP/hLutd2vuvIQp+D
DkYYv7APNBqX92Ch7cmXZtuWA/0h+qNjteYsitoXRjv23Gbj5qIFPjsPW7W0
MOcYk8hU0A9Cl9+ofDfoWq6876i9lUNtn4SPMGTD8Kfn6f25Ufcr5y9xrgfY
THwZYROLnyB8cOFzA5sPbFag0+nHF+p9fgljDpoQGh0ajv2cPZj9lzUUvQ26
pKZC+3/EPpKzs6DlsCPAfgD9NTzz3fU+ywofSveon/+uc1/crDbepGsFOAX1
bdtwr6+vqv9eabSPMXAK9ANYhd6KK0Gvk+FxtzmMPfwAgr2AhiwVb7Jc5Tzf
xdg4bISwD2oO+CB0yo9UmleET/xiqLE+yOM4QxgaBLsJMKDoA9AF4C92UaUx
pdgZsmYik5+g/j9UbTpZY/yJRvsIZq2/vdK0AnQCZ8v/X1H3FmNnVcUBHMHQ
2laC08tQmV4sc0516HTaZpy2zGmnc3p6VSMxIfGGPhhQESQmGisKSBSaGEsT
o0FI9KGYGGwQqvhoVEIEYixYDK1tDBB5MPGSAiokVev6zX83PjSdOXPO9+1v
7332Xnut/+WlmeD7dlU/DbrRoLyt+uBL0+GLeA7X9izqye9r+eGzk8nVytOu
rTnfrX8HViZ/Ko+qZvT1sXCWtBUWZk/Dw5zs5gwB1yoOEA/gvzxZ7XlsOu9b
W+PVXRvs1FfqGd/Wi06XcyfeGNztZ+r9t+5O3UyuRfwt36Km6juurooPsWJ9
OBG8a/jCwPPyBeIPxG9oyc5w2nH5eAiLX+SZecX/tvUz/cjPDkdDEiYXd1H7
1efouoh7T6t917+H14Sf8Ww7J9PDs+arR/54Y7RX1HRf6EQjjJbY3yfD+cH3
2VNtHlofbuxX+9Gb1qe/6SR+eHh+dHZpFtPapUn8y17a9FIn+mLW72faXNHO
3ze8MqwyXCqMGnza5fXcT9c4fWRl9NB46OD4qtnALfDLeLz6+cxM5uu9nfhd
4S49W8/6zO7gXqdq3A5uCc9yX83Zs73g62khvD7I/b5cc+H2vfG2uqLu+1zd
91Mrg1vH9YdRfG833CRansuqD47XZx+dH9wkDCrvjKerPS/O5Hfaos9tCIbt
4IZwC/X7javDHTgxL1yrxY1vxSfn343L9YtePM5xUIc7GW9YpQODeDyoR8C/
+RzuKpw6Li6uknrLqy2Oss6t3Za17sr18e3AK4BzhY+HdX98NHsCLOhrk+Gf
72jcMPkqWh/8meFh5/w+hsKD5fVDyxGGXDufWpV8FgyYPX5Z6zceLH8cC05v
cT3XUC+5nCdqvv1oRzAj8AvWJnG7863xxBPk+8T/CedGvfoTm3LufKDG9APr
opugrii2UFucq9lfHiwNvOl72ric60dTgJ7AwurDtwzSdvUqOgT4ILyV8ILx
g2mgfm44OqiP1hz+ay/9NVtteNe2YOhgYgZtbVxQ115Tzzi+IvMGttLf9Neu
Vi97tebDxyr+/uBIfKjkoflS/WsyNUL1wQerXUcGiStP1TWvnwqvyJpjHslf
/brm0gszuS5PBnELzwWawc/vjiYvzQja93BduCz/afGVvOAF/v26mg/9yaz5
6j32AlgLvm3823gOWi/m/CQWZe9U/xXz4cDd0U9+zvdTbCZG817/WzOff0c8
LnHArq3+O9zPOvXm+v/K6qvR+uyqGsMvTufeeFnqo2qjPBLnvNs64dFewHdc
Wn2zdSycFVhO5xy148295FvlWkeWB89obp/qZk3HdVi/PBro8Mf4oPBaajkn
F4YbIfbD0YVPhHl0bXwd18cF0o/6UK0ed9Rn+RyqR6tL88KVr1En+Vs/mjuw
IaPV5u4gGqn8z9Vi1GSOw5TtCCfpnuHgaL3+RjuHqVGqhX+01V7wKObG6qLo
VtJ4EgfTVRJ/49/jcF+yOTzuDnz+bPIWvDHE+3KVR6r/bh/keb5RP183SB/g
ZdEg47txtK5zYib5UHhrPj648fK8roPr50xygXf++ZGcA90LJuX+hkvhmeQ1
WJfuRGqW6pXf3BDetZgYNpsGLM9KXlVwGM5AeB6nFiZW1x8/b32yotq7ahB9
3vvqPt/phQ90ZEP45M5XfDn4dtCNnO1GmxKnkJ+53A9t829100b8YDXnxeOp
r/E5t2fSQ8P3FSeJkf7cj2YQjibuI74crhxN5QPTed/eTjzNcYOd4bTFvH6g
2nZ8JnhimpZj3Whf8hv50zXBC6jV60v1+q1D0Xag8aAf8F7kajbvjC84j8Rz
POW3plbM11SORn5mtMZ8vM6zV4yEA2A9llvGOzGP8cYP1rjfOkgMteuqeDaL
O3mK8hjAf9k/m9qGuoaau7bz9Vs6ER4+ztZ36z33zeaaP9mYfI1cDW+3kYlw
i/ijwty6Nw0dfi3y+2rnfBx5OPL24BXAE41vAD9Gv8uFipngbOWn9IV+UOvA
p1HvsB/jzdDT3l9tePndOX++VmN1fio4JnNwLj9xSeJI2mb0zMVYvPpgO3mm
wTzDVfDsc5ahBwijYI+wV/Du8p2Gmfld3euVfrBxtKfoRtK2ouu/tpccgnVR
bsi5VW79D20+ixFxtWhNwF9OtH1EblCdT97y6pa/dNam++y8NcdvXZWakD7H
cfdddy3nbG131r6z5tuvZpKH2ldt2dvLWfMh2gg1108tSd79TPsuHxmP1rx9
VD/J56gDvlFjOyRHtyJ74V/aGiIPJTcvvwQzpA6Is0976diWxKs8cB7ZkLVV
Wy9u66fPWVd9Vi1YPgxvzhnU9Z1DecTBMuinQ3WNe7ZlLVg5EayDvy0bjR6G
tfpsJ3pbR0eiL073kvY4r7zxifjl3Vhr7Q07kk+dX/1x99bkSmnRWjvpluCl
0/uGo1a7hZeB3YXToRkGq4OTbb1XK7+z9tu7dmcNhk+i3QAnL4+DVyYO45cL
B7CoPff5tqd8fyx6DX7nobepffZrNa/2TWUu0h3XHri6Y8PxxDQ2w6N5fvf7
6cZozMGhqnXwRMEN3DX0/9o+zQOYC/VM/t4fH09u59763KF+tG+N+6H2HZm3
vcZne/ZMZwBag9fW+99e/bZlXXJO8IU0qOj/jNVrd08nZyRHaD2DfXqq+ubJ
3cnV0WN1f7VUOu1bGjb6nVNtb6gxfVNd/6JesMN7esmdvdL4tudaPpNXCd0h
uhlTy+OThF9GI5yWNx1v2k+07+B/YKW8pm/hB89PB0O4qJ5v4fZcAy//Q/1w
89USfN981x7pp/am7rap2jPRi580DYMX+5kvl7Wx9bw/qNdGp8LtPrYxupMw
0/5uDsid8qy4pRfuF6wwLCz8rZqY19RG6St22jr1WF3j8FhyKjfXz7f0o5Ws
RrGg1Sloty9p+u14Yr5/vBFxCMSHsGTiFXELvQu6V59sunBwjvoDB1OeSL4I
vgWGWn5EbsSYLmrtt3dd1p73psnE4uLwi+uZ7lgX7bDtrcZizTxf/f/fTnBU
sPb0OOHt+QFeN57c4w/HokWij/gC4bje1c4SzhT2Lx5N988mV2ittb6qJbmf
9Rj+dsHqxHb29Lm4bF6wgp+u119vcSpfPr4vPF/wfPBr8ADgMY278T9bffxQ
vWdyZbh4zia8cp42N6/J2e1nG6MXifsBvw8HQT+KJy2fWr606o/aos/xRr/X
1hntXdrazKuHp4685BPd+AzBM6rf3tRquPi+4g7fa/Uu/H7tNLYnW97yUDdn
C2vq6jXxkccfeOvSxI/ysWJ/WHsYVPkjeDm59X/W+vqPbcn5vX91fI54KC1r
e511b/9VWafEx7Tu6TXTFeBLwOtN3x1tvNvTOCxj0WqBnaTHT6//cItlTrd4
5uaaLy9vTl3AniT+NaYfrtcfrHl126XRzeX3Js/7heHU1vzsLHV9O0/puzNt
bz1xdc2BnZnL63aGT4hLCFeiVgRbckO159u9+CPzRuYTjH/wPzfIE90=
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtnXd8lEXXhpcaIPSe0BNalpRNSAKBAMlms6F3BaQoIF1RKQqKVCkKokhX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         "],
         VertexColors->None]}]}, 
     {Hue[0.5951953737889122], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFnXd4VVXWxm8SSMCQGEIv6QmBCyE3AUILRlJuQi8qNuwVBGw46igWBBHF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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtm3mcz9X+xz+WsY1lDMYSxjIT853t+52NYSzznZnvt+xLJIpSsqek5Yqy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         "],
         VertexColors->None]}]}}, {{}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[LineBox[CompressedData["
1:eJwl12eUVdUZxvFN7yC9VyF0ZICBaQzS+zQ6AgqGMgy9mShFkKqiURAC4qAR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         "]],
        RowBox[{"-", "324"}]],
       Annotation[#, -324, "Tooltip"]& ], 
      TagBox[
       TooltipBox[LineBox[CompressedData["
1:eJwl1nlwkNUVBfDH5qBQSQybQFwIIUgCBAgQtqKowAg6I0pBQ5W6oCKgKFSk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         "]],
        RowBox[{"-", "324"}]],
       Annotation[#, -324, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[LineBox[CompressedData["
1:eJwl13eUV8UZxvFBjARMIkGDSDMigqF3IgrLLrCg51CyQKS3BUSkGIHoQmgC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         "]],
        RowBox[{"-", "256"}]],
       Annotation[#, -256, "Tooltip"]& ], 
      TagBox[
       TooltipBox[LineBox[CompressedData["
1:eJwl1nmUz1UYx/FLTudQOsWMNcRpsQwizDCVGTsZ2Rozoyyz2cKMfac6bfaO
XfWHss2CbDMmslVCWUoKobJzIv9kj15z+uN9Pvfe57nPc5/n3u9vpmbq8O7D
SoQQYkuGsCEmhFoNQ0hqH0Iy+seHkIo0bG0SwjYMaxRCIttQ2otGNAthTd0Q
7rYK4Q72twthHyo0COFgmxAq0kP0Qb0QippKFBVCA3F6Ng+hPr1vfaEY/9Io
8/zoEOrQO+aF/O/SuuY9+Nemt82P8L9Fx7wkv/gHxC9Pv6c/oLNzJKCnc10V
I65tCN+yt6R76HPiPIsYdUWjGdqqpQ1qqH+tPkyPC2EGZmEm0uuEkNs4hOPs
Z/mfw3n8JX4Jef40PoOC1iFsRpZxNqbVD2HniyHswDpxy6i/jDM+gtJ4mH2b
mrdiV4sQ2qmrR4cQShl3U8PA2BCesCeTPk4H0PM0WrwN/CKMC+WLpG+yrbJn
CC1vPoiWo4NpaXkele8xlEU3eUbZ15WmiPM6eiOF39d0pvi76Qz6Fb8d2I4C
97AZJdV8Xe0X1XgBhdZyvYk57mY2EtXTC2v1bA1eM67InqWX2UjSi2SkoKO7
6IRe7O3cQSJtT2+IP1uedWpai6Xe1RLsb2ks3z61VXG+GHpezuY0WZwhbL1p
rPleMVrQFPMDtA+No8v5x9O+5iP496OtzX/i34peUesaekhthzFG3rFIco5k
9HGuODE+9aY6exOf0AQ6Tpyx+MieuZiDwWoZhC7snezJMs7GcLyj1o7WOiBD
7kxMeN7dyDEaY3DbWtmXQ8hTaz4qeBsVUQlz3dd+a/sQaZ5tnud+85GL0+Zt
fCcH3eUJ4xxrv9E4ax3V2JqeMo+nJ2kruldtPV7w/aCr76U77Yb31fVBcb/4
VZarCqpikflqtexWy0q6i66ime58AG6bbze/RXfQ++q5h+ryV0NnMc6xRfKN
QCrSkGNtBv/VdJb+TNK/LfpchFx9zUM+FjjTQmTpW6p+pSEde+xJU0umcQam
uL/J2OT9THXnBe63pL5Noff4TqXzxYlgy1Pvat/GGusTvN+JOMq+iH0aXW/9
LbrYvBL/fP45/O8Xn5NvLXF/YV/Jtgo75N2J353jD6xwrgd60F8tZ/W4Hz1H
l4q3BCX5XlRvCVoKD2GaeWWx1qn5C8y3fwEW4km1VxO3OmqghHN00rtj+nsc
y/h/hs8x0H2d1fOhznuGDqMN5G+IuuyX7KtDN3szI9Twq/0j6Ra/Iz+KvZWO
Ni+io+iX9BX+nbDJngxrR+3JpPlsb9hTRp5EtgFynTaO9t3H4Bl7nkZ9uaOQ
bs8N6zdxC/OcdQn7YizCz+IeQQrfO+quKnYVVEZVPbhqz1K/1V3Vf934b1wr
/q3Ruwf2BH28T8+YT9TnSUj2vjqoOan4W6Q3xe3rfnqLmYLx9oxDS/czw/cR
R2fSbPd8WZ6P1TTf/Z+XJ128LmpYYT2Bppkf9Zaz+A7HMuuL+c/jf5L/q+xN
+V203oROFzdW/Cu+0xb0Er2MLc5RhLnOdYFvoXEBNhf/zXXuLuIkYKoeTcFk
pKslDRvUmuHdLPe7uwKrsBIn9OmYM43Xt2xxRmAk/lH/cPYcbzgX5dxXeURg
Ft/3nLGes71Lo2greeNxqbb37Y4bi/kNvyg11sN2Z65D62Kj9Ubsb/O7yD+6
+O8wKotzWJ2V6ES2yZiED/mfpt/5Zk/RPTSVXxruyF/T2m0a706vOXektxGB
8thp73o9zlDnU3q2yXhjzP//5/wH1Q6gCg==
         "]],
        RowBox[{"-", "256"}]],
       Annotation[#, -256, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[LineBox[CompressedData["
1:eJwl13uUz3Uex/EPTgchtYVxSaai0LqEXAbjMoNcTgxm6LjNGGzbicIyRKuI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         "]],
        RowBox[{"-", "196"}]],
       Annotation[#, -196, "Tooltip"]& ], 
      TagBox[
       TooltipBox[LineBox[CompressedData["
1:eJwl1WtM1mUYx/FL1yrthVLThppGL1S0Yp0wgQdNBKHshGI1E7BwLU/gIayl
kjIPtFJTtKTStKymadrhja61IZKHhk2x0FpooGbOF0lttpb1edaLr7///7p/
1+G+//eDac9UFs/uFhHb/bMhO6Lg9oh6mk/zRkeMxRjccldEeyJi650RP9Oh
+RFDcPy+iLr0iFUPRJy5P2JfbsRejJZznK8VxzCIdyAO8S/lX8b/PX8iI+JN
/lT1s+UkkIPnUDQiIkO/M2MjTiPl7ogj4yJWqleHzMyIEegnfpT3O1waGrGE
luhzGV2Y731ecl2tFXqtxDJk6L2cjjJLLo4VRrTwjDPnUXqNtR/0XazXAs/V
qBM/5P1VepjmmrtA7QSdQHeYpcyM5Vitzhq8IC9dr3l0KJ1PE/rloFXPW8Ve
43sdM6xNx2H1r6qfJzYGHfa14p6IQvvplFNEB9n7QPThPci7nKY604JR9i6n
K9l/pD3o8wZ9xfsSLEW5HlPxFv9k32OveoXyijAOD+FBXLK2Vu46bORtwNNi
+/hnyZ+NmUizh0o6V6/dPHM8V6Gbe7QnL6I7zXevcofzmvk2DHdW7faVeW/E
OVplfQ4q0Y9/W07Eh3gflXdEfEBn6b1f/SZ85bz3Z5lTzUa6jecAHe+9mR62
/i3fZd/iCO2i8+W/bMYmZ1jFX+H7TsPHan+Eern9eQegF+9Fc+WZ73d6Vk7C
zLVmX4qneGvMOsSsafyD6UhnloXnnVeLPtNpvfNejw2odybr0cq/WP0W51hD
a3lPiL3L/w66xDbRzegQ78QqsTb+1clvwr9ZvU3Yii14L6n8VeZss77V88Nm
GY9Wd+NJ6ydoD/3z6SP28ChmqjkDfdW9Yp+9raegF07xfUMP4gCm+M7NtINO
UbdNr3LPO9WeSg9ZK6XF6jaY4RPxMu9z+X4xz4Ax9ueuTlL3bVpCa/SeSE/J
reRtS94repJ20lJ9Tia/l+fP1PscP1n7Ee3y+tIysVIsVKu/ffxtH//6rhfd
gd/kNuh1gW6kf4lfwQ7P0+RvpxV0kr85T6AEw/yeL6hbo2dfc9+ML9U9rf4X
dJe99RH7lec8Jso5S8/hRvGbsIevnX833cmfIrZIvc7k3wP1H5czAcVY6O6M
8o0X0UZ3aqa7tVbuOvR25160j5ewANWocyf3I1XtauvTxCqwRd6n7vFu7MQ8
d3wX3cbXTf/uOM9X6zfZId6Z9PGsdwZFzuBP51JIu+gfuFoQ8Q8m89+QnNkZ
DtMjoUY6fcx7T/GvzdDs93U9nchbgkmYxTcbPcUbeHvQwfKyxXJwnfcmedfS
A7TRnqYiy9+6c/zZtMIZPIvJw///P+o/jVE57A==
         "]],
        RowBox[{"-", "196"}]],
       Annotation[#, -196, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[LineBox[CompressedData["
1:eJwl13lwV9UVwPELNoGyBIIobWV1bGcKSYiAbLInUDuKUqmVArZAqyRCobIl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         "]],
        RowBox[{"-", "144"}]],
       Annotation[#, -144, "Tooltip"]& ], 
      TagBox[
       TooltipBox[LineBox[CompressedData["
1:eJwl1F1M1mUYx/GrNbeOOGjLgg4SOeARECpCeVHA5OEhOEpWqZvRzJwh4Nam
8p5tbSm6CTiDKCtoIzkqfDsoZ+lqoiWVigi02kw96Ky1Tjqxz38dfPn9r/v+
Xdd93S8Pudv3bOp4KCLK/NlcEXGyJOJSeUTbqojs0ogcPF4UUVcVsZy2039r
I4rXRBTxptdGrHsm4mlxCQ4WRxyojngHfXgb/biXiigwX4gq/tVJPt7j7zbf
g04UqNlF7/Lfz6iX9KCfQ7SpLKIRw4XWWB+xQu7tuoin6I/iI/QHepim+Bv4
VlkjQ/PFM/SKmlcxg6xnI5rV22S81PxRedfkD9In0hGzvhfUX8Ii3hU/qHRY
zuB9ObfUyFZjjt7GPGrtrdGZ1NAGulX9UbWq7WvUvsbwAUas8Qn9FB/hY5zA
H7xVNc4U4+LPMIFx/pN0CpOoVe9zepGO6Cfjbo7TBrreXtZhwB3ecXaXebrd
7fUNETewS09vohXdz7l3OW16vWAvp+3zDLZs1DtS9nlWnE83i1/BKfF04pG3
6DxazC3R12iX+U6E9U5Y64H19/CdS3rzbjLotm6f+V704Kb+HqaT+n1J3y9j
Qq1quTXOYanBfumvdC55H3SDuJO2iaed2U/O4hf8jC+c1YB7P4y0e9hvXzlq
/8X3vLyNmOObxy18yd+qr92otWa+vlPo0ee4Xqbc+wTtT/pGp3qjxprl/e7c
/sRj6vwmbpH/gtx6/p3JO+VP89djjbk64zuMv6i3N2ha/KrxbViQn6VOPR1U
P0OH6Fty29XpwBj/I8kbov/4jRTJK0QB7vHexZS8MzxnMY3TOIW/+R91Dqlk
j1hM3jc+5B82fwyDqHSmQ3SZfeTxrcQKjOi33LprsVedWWPTvDvc3QzvFRyw
rwVzrxvbjovewiXkeQPf0l6ea/L2+t6PZjX30R7voxe5fN8kbxHfeVNN5jvM
t+P7JE/+V76f5DtPv07epHUuG5+3bp/1t4q3VPz//+w/AhLc4g==
         "]],
        RowBox[{"-", "144"}]],
       Annotation[#, -144, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{10860, 17008, 10861, 18590, 18591, 18461, 15243, 15240, 
         15242, 18399, 15241, 15825, 15824, 17691, 11922, 15823, 17690, 17689,
          13548, 10859, 15518, 15519, 15517, 11627, 12976, 12975, 15650, 
         11726, 14967, 15651, 18501, 11014, 15652, 11013, 14965, 14966, 17546,
          17547, 17545, 15583, 17548, 11690, 14902, 14906, 17232, 18639, 
         11752, 17584, 17233, 18516, 18517, 17264, 17265, 17266, 17263, 16371,
          16372, 16374, 16373, 10465, 14260, 14259, 15737, 15736, 11150, 
         13595, 11149, 13183, 13086, 11011, 14958, 14959, 14254, 15724, 16629,
          15722, 13618, 13628, 13629, 11780, 13633, 13631, 11009, 16417, 
         16418, 12790, 16577, 16579, 16578, 13592, 11045, 15031, 15032, 14236,
          16395, 18397, 10459, 16396, 16397, 14226, 14225, 14224, 11768, 
         15711, 11767, 13614, 14932, 14929, 12494, 16356, 14927, 15604, 17555,
          15605, 14950, 18498, 18500, 18499, 11005, 15627, 11006, 14952, 
         14951, 11713, 15626, 12967, 12968, 11638, 15542, 15544, 15543, 10875,
          13057, 10874, 17017, 14879, 12412, 18058, 16607, 16608, 16605, 
         16606, 15159, 15160, 18467, 18469, 18468, 10871, 15541, 13055}],
        RowBox[{"-", "100"}]],
       Annotation[#, -100, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{10901, 17094, 10902, 18596, 18597, 18477, 15262, 15259, 
         15261, 15260, 15390, 11473, 15610, 11706, 14936, 14937, 10983, 13556,
          10984, 15556, 15557, 15555, 11661, 12995, 12994, 15666, 11732, 
         14971, 16378, 18505, 11022, 16379, 15762, 15615, 15616, 14939, 14941,
          16365, 12509, 14943, 14945, 14109, 12505, 16364, 12508, 16361, 
         12507, 16363, 13445, 18162, 18163, 16385, 16387, 16386, 12522, 16380,
          14976, 14977, 13702, 11025, 14119, 14120, 12539, 12877, 10495, 
         16464, 10497, 16659, 16463, 14333, 14984, 18509, 14982, 11031, 13724,
          13793, 11956, 14125, 11957, 16392, 16393, 16389, 16391, 16390, 
         13458, 18358, 12770, 16570, 12771, 16572, 12476, 14096, 17169, 14917,
          12478, 16344, 12769, 16569, 14088, 14089, 12521, 16375, 16377, 
         16376, 13152, 13587, 11001, 17209, 17208, 12516, 14115, 14114, 17228,
          17227, 18169, 18170, 17859, 17858, 12172, 17860, 12173, 16136, 
         11629, 15520, 15522, 15521, 10957, 13157, 15850, 15851, 15849, 11962,
          15853, 11481, 17392, 15852, 15400, 15401, 15308, 15309, 18462, 
         18464, 18463, 10865, 15594, 13158}],
        RowBox[{"-", "100"}]],
       Annotation[#, -100, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{11055, 13325, 11056, 15145, 15146, 15142, 15144, 15143, 
         16601, 16600, 13258, 11193, 13050, 11194, 13052, 13051, 14416, 14415,
          15139, 11232, 14455, 15537, 18412, 10563, 15538, 10562, 14453, 
         14454, 15089, 17292, 15088, 11144, 15929, 12030, 14456, 14458, 13831,
          12023, 15928, 12029, 15926, 12028, 15927, 16235, 16234, 10455, 
         14233, 14232, 15717, 15716, 11196, 13347, 12934, 12935, 13319, 13318,
          10538, 15950, 15951, 15949, 15953, 15952, 12042, 16518, 16519, 
         16515, 18254, 18253, 10508, 10454, 16617, 14231, 16437, 16436, 11777,
          15965, 15966, 15962, 17782, 17781, 10542, 13359, 13360, 12938, 
         12937, 13361, 11200, 15190, 15191, 15187, 15189, 15188, 16246, 16247,
          14266, 14265, 14247, 11804, 15740, 11790, 13638, 14521, 14518, 
         12067, 15986, 11152, 14420, 15094, 15093, 15741, 16299, 10599, 14524,
          14523, 11278, 15201, 14422, 14423, 13105, 13106, 11206, 12854, 
         11205, 13278, 11429, 17383, 15342, 15343, 15220, 15222, 15221, 11094,
          13373, 12855}],
        RowBox[{"-", "100"}]],
       Annotation[#, -100, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{11135, 13484, 11136, 15116, 15117, 14407, 15327, 16690, 
         10531, 15328, 11417, 13305, 11224, 13160, 11225, 13162, 13161, 13244,
          11186, 15332, 11421, 14663, 16311, 18431, 10702, 16312, 15807, 
         15129, 15130, 14437, 16141, 17862, 12179, 14641, 14645, 13727, 11895,
          13802, 11961, 14148, 18243, 12628, 18242, 12629, 12926, 10528, 
         12821, 18383, 12822, 18384, 12818, 13488, 18679, 13487, 11220, 13490,
          12956, 12957, 13494, 13493, 13231, 16143, 16144, 16142, 16146, 
         16145, 11878, 13728, 13729, 12587, 14341, 10499, 13300, 16109, 16110,
          16108, 16112, 16111, 12164, 16118, 16119, 16115, 17854, 17853, 
         13229, 13468, 13469, 12954, 12953, 13457, 11215, 15774, 15775, 14316,
          14401, 18402, 10492, 10524, 18587, 16654, 16492, 12614, 16094, 
         12615, 18659, 16493, 17845, 18303, 12685, 18570, 17844, 16534, 14604,
          12151, 16093, 14433, 15127, 17315, 15128, 14585, 18423, 18425, 
         18424, 10658, 15565, 10659, 14587, 14586, 11370, 15283, 11168, 13214,
          13127, 13128, 11212, 12872, 11211, 13283, 16651, 16652, 10520, 
         14387, 14386, 15112, 15111, 11102, 13421, 12868}],
        RowBox[{"-", "100"}]],
       Annotation[#, -100, "Tooltip"]& ], 
      TagBox[
       TooltipBox[LineBox[CompressedData["
1:eJwllElMU1EUhv8ScYoxRjdlo23BaMVYKC2jtpDiUNwZNSbiwrrQiBR1Z4Ka
qnGKxiUYE2HRihuHlUOi4BCRQgc72CIaUk001tk4oAuN34uLj6/3vnPPOffe
97AGujYETZJK+TPaKA3XSY+huFR67ZWyVVJolWRbIR3G5Xi3U5prlza6JSvj
bo9U6ZDu1kuDMAR3eOZkfQ1UgxtckGPdpgZpM9wj7j50MDdIfB3PI+SpxZdw
PR5uoa/V0gDjiXXSLZ9kXi7dxmU42CT5K6VDtZKDvqrAwrzXJf1tlv6AuVqa
s1ZaQq9Jcl0lVxxfwQm8n/opfI3xE3wdp7Gd+MlWaRk20UcJNJPXRv6V1GmC
EHW3GPXpYzHzB3EFfsX6IfLEWBOHXew3SP8NzJ3jvFLsOw1x9t1N/SPEZBh3
ErcHJpkL4i54xvwEHGMuT/xZYs/Ac+b85LtBPg89BnjWUyP1Qivjr/Tuw1+w
hXgr2GCKuy1wHg7Wj3PX7XgrOMk/tUb6zp3/xD/wL/wNPyS+hP0+wCYc5VxH
wM06FxTI00atfmr3QZHxW8jSWyc9zmQfs2C2sTf6vEytfnqZwThDTBr85Ivj
BJQyPx0GiHtBfB+xF2Eacx3GuRJTzr2a6acCl+G9nHuO92Ifvsk9BLiXA6zd
jn30tIP5EPcXY/0H7ieOP+Ix/Bkn8SecwD3UfcfvKL+LeBS/xyP4NLUi1DyF
wzhAzvXk9lCjnVo7qbkNn6eHNuZ7jXcULyB+EfHz8UL8ku8pwpkVcBjbyZ9l
7VFqn2Sv45xrhH2GoYWzyRGXh6dgJdZi9EX8CWKPQ554L/1dID5jnD3MM94j
47sg/yPGKUhDnjt9w/cV45uPwxgkIQG/Idr4///BP3b8yEI=
         "]],
        RowBox[{"-", "100"}]],
       Annotation[#, -100, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{10857, 17007, 10858, 18588, 18589, 18460, 15234, 15232, 
         11304, 15233, 16635, 14276, 11812, 15746, 11155, 14901, 15096, 17164,
          15095, 13103, 10947, 15695, 15696, 15694, 11751, 15697, 17231, 
         15012, 17582, 17583, 11749, 15692, 11748, 13603, 11747, 13085, 15040,
          11073, 13184, 11074, 13087, 17212, 14960, 15728, 17599, 11786, 
         15727, 11785, 15723, 13617, 13630, 11778, 13634, 11779, 13632, 14954,
          16419, 16420, 15030, 16421, 18511, 11044, 16422, 12553, 15027, 
         15026, 15024, 11755, 15700, 11754, 13609, 13061, 14223, 13062, 11766,
          13065, 13063, 10968, 13059, 10967, 17185, 14925, 11699, 15710, 
         11765, 14222, 16610, 16609, 17324, 11247, 17325, 15163, 15164, 18470,
          18472, 18471, 10873, 15545, 13056}],
        RowBox[{"-", "64"}]],
       Annotation[#, -64, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{10908, 17102, 10909, 18598, 18599, 18479, 15274, 15272, 
         11359, 15273, 11730, 15663, 11731, 15665, 11709, 14938, 14940, 10986,
          13131, 10987, 13682, 13683, 12506, 14307, 16362, 10490, 13225, 
         14321, 14320, 16381, 11868, 13703, 14978, 14121, 12524, 14129, 12876,
          16384, 12878, 12584, 16462, 17672, 16461, 17673, 11885, 17674, 
         15794, 14983, 14985, 13725, 11889, 15848, 14338, 14339, 13233, 10502,
          16571, 14349, 12477, 13740, 13738, 10955, 13154, 13235, 13236, 
         14091, 14090, 11184, 15621, 11711, 15619, 11710, 16587, 16588, 15004,
          15005, 14127, 12536, 16405, 12537, 16402, 16404, 16403, 10864, 
         13570, 13156}],
        RowBox[{"-", "64"}]],
       Annotation[#, -64, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{11053, 13260, 11054, 15122, 15123, 15120, 11192, 15121, 
         16599, 14218, 11760, 15706, 11191, 14413, 14414, 10536, 13048, 10535,
          15916, 15917, 15915, 12024, 15918, 12022, 16504, 12639, 15925, 
         12638, 12895, 12894, 15705, 11759, 14448, 11758, 15704, 13264, 10561,
          14452, 14451, 15135, 15134, 12933, 15954, 15955, 14478, 16729, 
         10580, 13837, 10579, 14477, 14476, 16517, 16516, 12900, 16434, 16435,
          16431, 16433, 16432, 13624, 10581, 14480, 14479, 15964, 15963, 
         12936, 15176, 15177, 14498, 16742, 10590, 13269, 14495, 14494, 14492,
          11268, 15175, 12902, 12903, 16628, 14248, 12564, 16442, 11791, 
         15976, 12056, 15973, 17796, 17795, 10543, 13093, 10544, 16697, 14421,
          11202, 15200, 11277, 15198, 11276, 15199, 17304, 11204, 17305, 
         15124, 15126, 15125, 11087, 13279, 12853}],
        RowBox[{"-", "64"}]],
       Annotation[#, -64, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{11137, 13307, 11138, 15118, 15119, 14408, 15329, 18521, 
         18619, 18618, 11419, 17381, 11420, 15331, 11226, 14436, 14438, 16722,
          10553, 13165, 13245, 16167, 16168, 16166, 12196, 16169, 11894, 
         13745, 11907, 16476, 14355, 13312, 11227, 15321, 11414, 14648, 16829,
          16830, 13299, 10696, 14650, 14649, 15323, 15322, 11180, 16147, 
         16148, 14646, 16828, 10695, 13721, 17680, 17681, 15798, 15800, 15799,
          11221, 16113, 16114, 14626, 16823, 10680, 13931, 10681, 14628, 
         14627, 16117, 16116, 11178, 15303, 15304, 14623, 16820, 10679, 13291,
          15788, 11879, 14624, 11880, 15789, 11216, 13293, 12152, 16095, 
         12153, 16096, 12150, 16098, 12154, 16097, 17847, 17846, 13219, 13144,
          10548, 14434, 14432, 11214, 15766, 11851, 14306, 16653, 14397, 
         14396, 14395, 15114, 15113, 11112, 13284, 12871}],
        RowBox[{"-", "64"}]],
       Annotation[#, -64, "Tooltip"]& ], 
      TagBox[
       TooltipBox[LineBox[CompressedData["
1:eJwl0zlsDnAYx/FHSASDCYuBBHEELb1FqZ6DyaAJQsLCQkSbUJPBUVX3LXFf
ad8ebkvdN0NbbTWhlLZuDYtBBJ/G8M3v/T/X//f83/cdvXTVvJX9IuIPmrIi
HqVHPEZ5dkTS1IgnGRFrcyIaUiMqp0RkpkVMEy9Uk0wznH/PjqhQP915K02h
2+h29Tf1lelfh1TxXfkRQwojBmO+XDES4k/Fq+kzunpCxKmCiD1JESfpGZzG
ePNe5UV04K87w8zP4yO+yH2nWdMihmIEb8ORpv643AkcxW7zjtE15r91T4P7
OukNuoSPxcg2dxa2858h3mj/De55JnfIvPaZEYfpGPGxGIeB8nXqC9XX0iJa
T8+qa9N3jr7Td5520QLz83HbGwS/931+iAcYnRxRyvsomkMfTIxo5vkq7020
Bc9xxTldfwZSkIZUDLPbT28xmKdBKMrlCyMnRezwbjtRicRkO9KH5tcUeVff
fYJ20PXqy7BMzwqe8nhZTnNph31q7XHPfovsWe39FtIXerNn8eZzqZ1KMJf/
Zudyv5EaPT/UvNS/n68qfQvk2sRanLv53qxuC5rEv8s30qPil+x7wL4X+/bG
Zex3/kZ78RXX7X4NPTxWOydQhVrUYK/6enoBddjnXGF2j3vuuKeb3qWb3L8R
nXIlmX4rWMVjgdwRNRW8ttrhlH1qvEG++GHxPNqqbqY3uNX3LnJzaBXNpQl6
UM9TvVvNKEeO+CG9xd651bu/wE8z2mi/GREf+Gzn8z39hI9I7vst+d660N+M
Aej1fX+We4036MBbdKJVfzftwTu0OXfRX+5ozPr/n/8HvYvQkA==
         "]],
        RowBox[{"-", "64"}]],
       Annotation[#, -64, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{10449, 14221, 12832, 13572, 13571, 11060, 13060, 14926, 
         13064, 13576, 13575, 12833, 13608, 11061, 13066, 15023, 16416, 12552,
          15025, 15028, 15029, 14137, 18510, 14138, 12560, 18375, 16593, 
         16594, 16368, 18394, 18393, 16619, 16367, 14241, 15719, 15718, 11772,
          15725, 15726, 14243, 14244, 10461, 16370, 10464, 16631, 16369, 
         17258, 17257, 17260, 17259, 15039, 15009, 15010, 15693, 11750, 15011,
          17230, 11040, 13190, 13647, 13562, 13563, 15699, 15698, 14900, 
         13104, 11086, 13194, 13195, 13102, 14275, 13101}],
        RowBox[{"-", "36"}]],
       Annotation[#, -36, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{10466, 15197, 12846, 13276, 13275, 11077, 13094, 14419, 
         15974, 15975, 13274, 13273, 10591, 13267, 10592, 14500, 14499, 11270,
          15183, 11147, 15972, 12055, 14493, 14496, 14497, 13620, 13621, 
         11789, 14227, 14245, 14246, 13268, 15958, 12045, 15957, 12046, 15959,
          12037, 15961, 12048, 15939, 12047, 15960, 13181, 15947, 12043, 
         15912, 12044, 15948, 11756, 15715, 11776, 14212, 14229, 14230, 13262,
          15923, 12025, 15922, 12026, 13829, 13828, 14450, 14449, 14447, 
         12027, 15924, 11143, 15133, 11230, 14446, 16725, 10560, 13249, 13610,
          13254, 13255, 15920, 15919, 14412, 13049, 11052, 13256, 13257, 
         13047, 14219, 13046}],
        RowBox[{"-", "36"}]],
       Annotation[#, -36, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{10493, 14305, 12875, 13290, 13289, 11120, 13145, 11171, 
         13920, 13921, 13221, 13220, 13701, 13296, 10670, 14608, 14607, 11386,
          15295, 12955, 16107, 12168, 14625, 16821, 16822, 13934, 13935, 
         12691, 16123, 12690, 16537, 10525, 16657, 16656, 14330, 12588, 16472,
          11882, 16122, 12170, 16121, 12169, 17855, 10550, 16151, 12182, 
         16149, 12183, 16152, 12180, 16542, 12701, 16158, 12700, 16541, 10529,
          16663, 16662, 14343, 12597, 13732, 13731, 14652, 14651, 14647, 
         12184, 16153, 12960, 15339, 11423, 14664, 16835, 10704, 13313, 10703,
          13246, 13247, 13948, 13947, 11187, 13166, 11139, 13310, 13311, 
         13164, 15330, 13163}],
        RowBox[{"-", "36"}]],
       Annotation[#, -36, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{10504, 15620, 12889, 13238, 13237, 11131, 13155, 14914, 
         13739, 14095, 14094, 11901, 12930, 11902, 13736, 11181, 15797, 11890,
          14346, 15796, 15795, 16388, 15792, 15793, 14334, 15782, 15781, 
         11872, 15783, 15784, 14331, 16383, 16382, 16580, 15778, 15779, 14322,
          11867, 15780, 11173, 13686, 11855, 13223, 11854, 14107, 14108, 
         13684, 14942, 13132, 11113, 13581, 13582, 13130, 15664, 13129}],
        RowBox[{"-", "36"}]],
       Annotation[#, -36, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{17877, 12212, 17875, 12211, 17876, 13202, 17880, 12215, 
         17879, 12216, 17881, 12213, 13956, 16843, 16842, 12214, 17878, 11313,
          17349, 17351, 17350, 10998, 12782, 18580, 12783, 12779, 12781, 
         18579, 12780, 13505, 12785, 18581, 12786, 12784, 12788, 18582, 12787,
          10999, 12803, 18585, 12804, 14187, 14186, 11037, 15001, 14126, 
         12534, 14208, 13019, 12707, 18551, 12228, 17896, 12230, 17894, 12229,
          17895, 10636, 17830, 12115, 16066, 12116, 16067, 12113, 13899, 
         16789, 16788, 12114, 17829, 11340, 17357, 17359, 17358, 10978, 11704,
          11705, 11041, 11164, 18407, 10516, 13041, 12794, 18583, 12795, 
         12776, 16575, 16576, 16573, 12777, 16574, 10979, 12774, 18361, 12775,
          18381, 18382, 16595, 12820, 18380, 14202, 14203, 14155, 12610, 
         14207, 12992, 12676, 18545, 12117, 17833, 12119, 17831, 12118, 17832,
          10637, 17387, 11438, 15352, 11439, 15353, 11019, 18504, 12519, 
         11836, 17646, 11837, 17647, 14377, 15107, 15108, 14893, 18602, 14892,
          18603, 10934, 12767, 12768, 12766, 16567, 16568, 16564, 16566, 
         16565, 10482, 10513, 16669, 16668, 18227, 18226, 12609, 16562, 16563,
          16560, 12765, 16561, 10933, 13387, 13388, 14211, 14210, 11315, 
         17639, 11833, 17638, 11832, 13661, 11831, 17887, 12220, 17885, 12219,
          17886, 10619, 17883, 12217, 17882, 12218, 17884, 11829, 13660, 
         12570, 18189, 12571, 18190, 11314, 17352, 17354, 17353, 10931, 12758,
          18576, 12759, 12755, 12757, 18575, 12756, 10479, 12761, 18577, 
         12762, 12760, 12764, 18578, 12763, 10930, 12798, 18584, 12799, 14185,
          14184, 11035, 17637, 11828, 17636, 11827, 13659, 11826, 17877}],
        RowBox[{"-", "36"}]],
       Annotation[#, -36, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{11059, 13058, 13574, 13573, 10450, 12834, 11753, 13067, 
         15021, 11042, 13069, 11043, 14130, 14132, 12551, 14193, 14194, 16592,
          16591, 12813, 16590, 16589, 16425, 16618, 16426, 14240, 16428, 
         16427, 11771, 16440, 16441, 14242, 16439, 16438, 13636, 15729, 16630,
          15730, 14258, 15007, 15006, 11039, 13090, 13186, 13187, 13605, 
         13604, 11151, 15738, 15739, 14262, 14263, 14264, 13560, 13561, 12852,
          12851}],
        RowBox[{"-", "16"}]],
       Annotation[#, -16, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{11076, 13092, 13272, 13271, 15180, 11269, 15179, 15182, 
         15181, 11199, 15184, 15185, 14418, 10541, 13078, 10540, 17779, 17780,
          15942, 15946, 15945, 11775, 16429, 16430, 14228, 16615, 16616, 
         10507, 16512, 12642, 15956, 12643, 16513, 12036, 15941, 12041, 15938,
          12040, 15940, 10537, 17777, 12020, 15913, 12021, 15914, 11757, 
         16424, 12559, 14213, 16597, 16598, 10505, 16498, 12635, 15921, 16500,
          16499, 12014, 15904, 15905, 15901, 15903, 15902, 10533, 13044, 
         10534, 14411, 15132, 15131, 11190, 15702, 15703, 14214, 14216, 14217,
          13252, 13253, 12830, 12829}],
        RowBox[{"-", "16"}]],
       Annotation[#, -16, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{11119, 13143, 14317, 14402, 16655, 14404, 14403, 14319, 
         15777, 15776, 11217, 15296, 15297, 14435, 10549, 13148, 13230, 17856,
          17857, 16128, 16132, 16131, 12167, 16126, 16127, 16124, 12171, 
         16125, 13301, 14345, 14344, 14329, 11897, 15801, 11881, 16155, 12187,
          16120, 12186, 16154, 13232, 16162, 12192, 16150, 12193, 16163, 
         12181, 16161, 12191, 16159, 12190, 16160, 13316, 14357, 14356, 14342,
          15811, 15810, 11896, 16173, 16174, 16170, 16172, 16171, 13248, 
         13169, 10554, 14439, 15338, 15337, 11228, 15335, 15336, 15333, 11422,
          17382, 10532, 12932, 12891, 12890}],
        RowBox[{"-", "16"}]],
       Annotation[#, -16, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{11130, 13153, 14093, 14092, 13734, 13735, 12929, 13737, 
         12928, 13742, 11183, 15802, 15804, 14348, 16665, 10501, 13722, 10498,
          14336, 14335, 16475, 16474, 11873, 15790, 15791, 14332, 16658, 
         10496, 13704, 10494, 14326, 14324, 16459, 16458, 11175, 13708, 11172,
          13685, 13222, 13681, 13680, 14105, 14106, 12874, 12873}],
        RowBox[{"-", "16"}]],
       Annotation[#, -16, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{10451, 12835, 13607, 13606, 11063, 13070, 15022, 14131, 
         12549, 14136, 12548, 16415, 14192, 12812, 14134, 11770, 17596, 18176,
          12561, 18175, 16612, 16614, 13616, 16623, 16622, 16621, 12563, 
         18182, 11784, 17602, 13637, 11787, 15731, 11788, 17605, 17604, 15008,
          17229, 13091, 11075, 13188, 13189, 13089, 14261, 13088}],
        RowBox[{"-", "4"}]],
       Annotation[#, -4, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{10462, 15178, 12840, 13266, 13265, 11068, 13079, 14417, 
         15943, 15944, 14473, 14475, 10578, 13622, 14143, 14144, 13180, 11146,
          14160, 11145, 15936, 15937, 14472, 16727, 10577, 13619, 17597, 
         17598, 15712, 15714, 15713, 11195, 15908, 15909, 14443, 16724, 10558,
          13830, 10559, 14445, 14444, 15911, 15910, 11141, 14158, 11142, 
         13172, 14440, 10556, 13827, 10557, 16723, 14442, 15907, 15906, 14410,
          13045, 11051, 13250, 13251, 13043, 14215, 13042}],
        RowBox[{"-", "4"}]],
       Annotation[#, -4, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{10503, 14350, 12931, 11904, 13743, 11182, 15803, 11900, 
         14347, 16664, 13723, 17685, 11887, 18219, 12594, 18216, 12593, 13712,
          17668, 11883, 17665, 11884, 17670, 11870, 18209, 13705, 14325, 
         14323, 12583, 16460, 11174, 13709, 11871, 13224, 13707, 13706}],
        RowBox[{"-", "4"}]],
       Annotation[#, -4, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{13167, 15334, 13168, 13315, 13314, 11140, 13170, 11188, 
         16175, 16176, 14666, 16836, 10706, 13730, 13746, 13747, 13317, 11229,
          14177, 12961, 16164, 16165, 14656, 14657, 10699, 13733, 16478, 
         16479, 16477, 16481, 16480, 12927, 16543, 16544, 14653, 14654, 10697,
          13946, 10698, 16831, 14655, 16157, 16156, 12958, 14173, 11222, 
         13302, 14629, 10682, 13936, 10683, 14632, 14630, 16130, 16129, 11179,
          13149, 11126, 13294, 13295, 12881, 14318, 18403}],
        RowBox[{"-", "4"}]],
       Annotation[#, -4, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{11062, 13068, 12550, 12837, 12836, 14191, 12811, 14135, 
         14133, 11769, 17594, 17595, 16611, 16613, 10452, 13615, 10460, 16624,
          16620, 18183, 18181, 11783, 17600, 17601, 16625, 16626, 17603, 
         16627, 12538, 12845, 12844}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{11067, 13077, 11198, 16514, 12645, 14474, 16728, 14159, 
         12644, 12647, 18565, 12646, 13179, 12038, 18536, 12039, 11773, 11774,
          18395, 10453, 13261, 12016, 18535, 12017, 12015, 17776, 12019, 
         17774, 12018, 17775, 13171, 16502, 12636, 16501, 12637, 16503, 12556,
          14142, 12557, 14441, 12558, 16423, 11189, 12828, 12827}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{11125, 13147, 14405, 16473, 12591, 14631, 12590, 14147, 
         12589, 12693, 18571, 12692, 10551, 12188, 18549, 12189, 12185, 12703,
          18572, 12702, 10530, 10500, 18404, 12598, 11898, 17865, 12195, 
         17863, 12194, 17864, 10555, 16546, 12705, 16545, 12706, 16547, 12704,
          14176, 10705, 14665, 13013, 14409, 12893, 12892}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{11903, 13741, 13234, 17683, 11899, 17682, 17686, 17684, 
         11886, 18217, 18218, 18215, 12592, 13711, 17666, 17667, 17664, 17671,
          17669, 11869, 18207, 18208, 18206, 12582, 13226, 14328, 14327}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ]}, {}, {}}}],
  AspectRatio->1,
  Epilog->{
    PointBox[{-2.7366639988756276`, -2.942345201011201}], 
    PointBox[{-2.7366639988756276`, -2.942345201011201}], 
    PointBox[{-3.7498967229198605`, 1.4552469382196032`}], 
    PointBox[{-2.7366639988756276`, -2.942345201011201}], 
    PointBox[{-2.621362398525856, 2.9956327630491804`}], 
    PointBox[{-3.7498967229198605`, 1.4552469382196032`}], 
    PointBox[{-2.7366639988756276`, -2.942345201011201}], 
    PointBox[{-3.9890161957970904`, -0.37621255361854944`}], 
    PointBox[{-3.7498967229198605`, 1.4552469382196032`}], 
    PointBox[{-2.7366639988756276`, -2.942345201011201}], 
    PointBox[{-3.9890161957970904`, -0.37621255361854944`}], 
    PointBox[{-3.7498967229198605`, 1.4552469382196032`}], 
    PointBox[{-2.621362398525856, 2.9956327630491804`}], 
    PointBox[{-3.7498967229198605`, 1.4552469382196032`}], 
    PointBox[{-2.7366639988756276`, -2.942345201011201}], 
    PointBox[{-3.7498967229198605`, 1.4552469382196032`}], 
    PointBox[{-3.7498967229198605`, 1.4552469382196032`}], 
    PointBox[{-3.9890161957970904`, -0.37621255361854944`}], 
    PointBox[{1.8268708012611898`, 3.5562606439414575`}], 
    PointBox[{-2.7366639988756276`, -2.942345201011201}], 
    PointBox[{-3.7498967229198605`, 1.4552469382196032`}], 
    PointBox[{-3.7498967229198605`, 1.4552469382196032`}], 
    PointBox[{-3.9890161957970904`, -0.2899152158516196}], 
    PointBox[{-3.9890161957970904`, -0.2899152158516196}], 
    PointBox[{-2.621362398525856, 2.9956327630491804`}], 
    PointBox[{-3.7498967229198605`, 1.4552469382196032`}], 
    PointBox[{1.8268708012611898`, 3.5562606439414575`}], 
    PointBox[{-3.7498967229198605`, 1.4552469382196032`}], 
    PointBox[{-3.7498967229198605`, 1.4552469382196032`}], 
    PointBox[{-3.9890161957970904`, -0.2899152158516196}], 
    PointBox[{-3.9890161957970904`, -0.2899152158516196}], 
    PointBox[{-2.7366639988756276`, -2.942345201011201}], 
    PointBox[{-3.8420774300766407`, -1.1947259500599068`}], 
    PointBox[{-3.9890161957970904`, 0.4462628177704646}], 
    PointBox[{-2.7366639988756276`, -2.942345201011201}], 
    PointBox[{-3.7498967229198605`, 1.4552469382196032`}], 
    PointBox[{-3.9890161957970904`, -0.2899152158516196}], 
    PointBox[{-3.8420774300766407`, -1.1947259500599068`}], 
    PointBox[{-3.7498967229198605`, 1.4552469382196032`}], 
    PointBox[{-3.9890161957970904`, -0.2899152158516196}], 
    PointBox[{-2.621362398525856, 2.9956327630491804`}], 
    PointBox[{-2.7366639988756276`, -2.942345201011201}], 
    PointBox[{-3.9890161957970904`, -0.2899152158516196}], 
    PointBox[{-2.621362398525856, 2.9956327630491804`}], 
    PointBox[{-3.7498967229198605`, 1.4552469382196032`}], 
    PointBox[{-3.7498967229198605`, 1.4552469382196032`}], 
    PointBox[{-3.9890161957970904`, -0.2899152158516196}], 
    PointBox[{-3.8420774300766407`, -1.1947259500599068`}], 
    PointBox[{1.8268708012611898`, 3.5562606439414575`}], 
    PointBox[{-2.621362398525856, 2.9956327630491804`}], 
    PointBox[{-3.9890161957970904`, -0.2899152158516196}], 
    PointBox[{-2.7366639988756276`, -2.942345201011201}], 
    PointBox[{-3.9468878428150775`, 0.47974909276527367`}], 
    PointBox[{-3.7498967229198605`, 1.4552469382196032`}], 
    PointBox[{-3.8420774300766407`, -1.1947259500599068`}], 
    PointBox[{1.8268708012611898`, 3.5562606439414575`}], 
    PointBox[{-3.9468878428150775`, 0.47974909276527367`}], 
    PointBox[{-3.9468878428150775`, 0.47974909276527367`}], 
    PointBox[{-2.621362398525856, 2.9956327630491804`}], 
    PointBox[{-2.621362398525856, 2.9956327630491804`}], 
    PointBox[{-3.8420774300766407`, -1.1947259500599068`}], 
    PointBox[{-3.9890161957970904`, 0.3952577829077992}], 
    PointBox[{-3.7498967229198605`, 1.4552469382196032`}], 
    PointBox[{-3.9890161957970904`, -0.2899152158516196}], 
    PointBox[{-3.7498967229198605`, 1.4552469382196032`}], 
    PointBox[{-0.6340958904513947, 3.937673811980489}], 
    PointBox[{-3.9890161957970904`, -0.2899152158516196}], 
    PointBox[{-3.9890161957970904`, -0.2899152158516196}], 
    PointBox[{-3.9890161957970904`, -0.2899152158516196}], 
    PointBox[{-3.7498967229198605`, 1.4552469382196032`}], 
    PointBox[{-3.9890161957970904`, -0.2899152158516196}], 
    PointBox[{-3.9761217920422003`, -0.2899152158516196}], 
    PointBox[{-3.9468878428150775`, 0.47974909276527367`}], 
    PointBox[{-3.9468878428150775`, 0.47974909276527367`}], 
    PointBox[{-3.8420774300766407`, -1.1947259500599068`}], 
    PointBox[{-3.9890161957970904`, -0.2899152158516196}], 
    PointBox[{-3.9890161957970904`, -0.2899152158516196}], 
    PointBox[{-3.9890161957970904`, -0.2899152158516196}], 
    PointBox[{-3.9890161957970904`, -0.2899152158516196}], 
    PointBox[{-3.9890161957970904`, -0.2899152158516196}], 
    PointBox[{-3.989477455767874, -0.28994873947685196`}], 
    PointBox[{-3.9894773757783164`, -0.2899487336633409}], 
    PointBox[{-3.9894773757783164`, -0.2899487336633409}], 
    PointBox[{-3.9890161957970904`, -0.2899152158516196}], 
    PointBox[{-3.9890161957970904`, -0.2899152158516196}], 
    PointBox[{-3.9894773757783115`, -0.2899487336633404}], 
    PointBox[{-3.9894773757783115`, -0.2899487336633404}], 
    PointBox[{-3.9894773757783115`, -0.2899487336633404}]},
  Frame->True,
  PlotRange->{{-5, 5}, {-5, 5}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 180},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->289924982]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Method",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->23837268],

Cell["\<\
Specifying a non-default method could give a better solution:\
\>", "ExampleText",
 CellID->1424437279],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMaximize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"Sin", "[", 
       RowBox[{".4", " ", "x"}], "]"}], "+", 
      RowBox[{".7", 
       RowBox[{"Cos", "[", 
        RowBox[{".6", " ", "x"}], "]"}], " ", "x"}]}], ",", 
     RowBox[{
      RowBox[{"-", "10"}], "<", "x", "<", "10"}]}], "}"}], ",", 
   RowBox[{"{", "x", "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1467809973],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.2538744443565584`", ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", "1.7658460394529054`"}], "}"}]}], "}"}]], "Output",\

 ImageSize->{165, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->36353381]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMaximize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"Sin", "[", 
       RowBox[{".4", " ", "x"}], "]"}], "+", 
      RowBox[{".7", 
       RowBox[{"Cos", "[", 
        RowBox[{".6", " ", "x"}], "]"}], " ", "x"}]}], ",", 
     RowBox[{
      RowBox[{"-", "10"}], "<", "x", "<", "10"}]}], "}"}], ",", 
   RowBox[{"{", "x", "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<DifferentialEvolution\>\""}]}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->394924060],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"5.964389511244633`", ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", "10.`"}], "}"}]}], "}"}]], "Output",
 ImageSize->{137, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->179384147]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Sin", "[", 
     RowBox[{".4", " ", "x"}], "]"}], "+", 
    RowBox[{".7", 
     RowBox[{"Cos", "[", 
      RowBox[{".6", " ", "x"}], "]"}], " ", "x"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "10"}], ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->2988854],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 116},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->34517295]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "StepMonitor",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->151017442],

Cell[TextData[{
 "Steps taken by ",
 Cell[BoxData[
  RowBox[{"NMaximize", " "}]], "InlineFormula"],
 "in finding the maximum of a function:"
}], "ExampleText",
 CellID->922642942],

Cell[BoxData[{
 RowBox[{
  RowBox[{"pts", " ", "=", " ", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"NMaximize", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"-", 
         RowBox[{
          RowBox[{"(", 
           RowBox[{"1", "-", "x"}], ")"}], "^", "2"}]}], "-", 
        RowBox[{"100", 
         RowBox[{
          RowBox[{"(", 
           RowBox[{
            RowBox[{"-", 
             RowBox[{"x", "^", "2"}]}], "-", "y"}], ")"}], "^", "2"}]}]}], 
       ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "y"}], "}"}], ",", 
       RowBox[{"StepMonitor", "\[RuleDelayed]", 
        RowBox[{"Sow", "[", 
         RowBox[{"{", 
          RowBox[{"x", ",", "y"}], "}"}], "]"}]}]}], "]"}], "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "1"}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"pts", " ", "=", 
   RowBox[{"Join", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1.2"}], ",", "1"}], "}"}], "}"}], ",", "pts"}], "]"}]}],
   ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->1534545049],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ContourPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"-", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"1", "-", "x"}], ")"}], "^", "2"}]}], "-", 
    RowBox[{"100", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", 
         RowBox[{"x", "^", "2"}]}], "-", "y"}], ")"}], "^", "2"}]}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1.3"}], ",", "1.5"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "1.5"}], ",", "1.4"}], "}"}], ",", 
   RowBox[{"Epilog", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"Arrow", "[", "pts", "]"}], ",", 
      RowBox[{"Point", "[", "pts", "]"}]}], "}"}]}], ",", 
   RowBox[{"Contours", "\[Rule]", 
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{"-", 
       RowBox[{"10", "^", 
        RowBox[{"-", "i"}]}]}], ",", 
      RowBox[{"{", 
       RowBox[{"i", ",", 
        RowBox[{"-", "2"}], ",", "10"}], "}"}]}], "]"}]}], ",", 
   RowBox[{"ColorFunction", "\[Rule]", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"Hue", "[", 
       RowBox[{"0.5", "*", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{
           RowBox[{"Log", "[", 
            RowBox[{"10", ",", 
             RowBox[{"-", "#"}]}], "]"}], "+", "11"}], ")"}], "/", "13"}]}], 
       "]"}], "&"}], ")"}]}], ",", 
   RowBox[{"ColorFunctionScaling", "\[Rule]", "False"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1614037545],

Cell[BoxData[
 GraphicsBox[GraphicsComplexBox[CompressedData["
1:eJxknQeUVTUTxwHpUpUiKlWQJqCgiCKGLiD4oQKCgNIFpKoIiNKki4BIkd6k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   "], {{
     {Hue[0.5213266254695816], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmot31dWVx8+V3Jub5L5ycwNoWyUBdaZrzT8wTqc6jh0HnE5bC1Spy7Vq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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwllulv1UUUhqe0tyv3d2/vvbghsrrxzT/ABUGCZW8pi8aYiKCggiAllrKD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         "],
         VertexColors->None]}]}, 
     {Hue[0.4900139495959325], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNnXe4VFXytRv6dvcNHW83jjIqg4wKoqOCAQMZRUAUEEExB7IgJgyAqKAY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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtmXecVdURx++y5e3bZe+9rxCRBBAbKiRRRAUiUsRGUYgCilhQioCVjvQF
BBVUjAkiGBUQlSwoSDVKREOiERGkS29SpINU0Xx/+eWP+bwzb+bMzJl7zply
qj30xB8fLxcEwcmcIMjjd0dJEBxPBUGXRBB0BBrFQbAyGwQDi4JgMLAL2qww
CPbkB0Fb6HUqBMFUeGbkBsFtxUEwgHHFDGNo78LTC/6ewJXI/xIFVZBVFfgg
GQQL0JePrEPofBraI8wZAW1KFATrGFdF3ybmToBncEEQtOK/gPEK4BRGVwbd
Df0f4D2Rdw/07eAzwLeiuzX4o+CPAR/nec73jOtg36TCILgbehreMB0EwxiX
z/Ha92FPm3z7YBv4m/C0Q/5d4Aux7Xls3I4945HZF9ps5NWC3hj5M1P2mXx1
GfyHwY+k7MsNQA/mt8Bnb+CrUfD/CO0E0B3aTuAk44ux51XGe4BT4JeCv8Z4
L/Ac49XofAJ/bwJfzngU/+1nfBD4GvzX2DcTe74Hj8Hvxb4vWO8x/NUY3UXY
sIz5fbBhGfSRzN8F727gKcYv8983jJcBx+Cdjrz9rPUHoCHzF0Nvzvz7mX8J
/M3BJyZs40XgTcEnJLyGpYwbMedheB9mzml8WwD+Of55HnsiZJ+B5z1oW+Cv
xfyrgLcZTwNaoH8V/uyCv/ezpsmh95z22iL8NxtaKT4K4X0AeJa5o9PeW/JJ
I9beGJjG+p/E/n6MZwBXIOtFbHhVew0bShl/hY4OyHoQ+BWyz8K/BX05Wft2
CDbug7YX6JvvPVkXWj1gHfIvZ/+8jz1zoNeEvx3wFbb0xN5K5YPgFfATrD8f
/FP0dcHGTtobwOXo+wCZf8OefszZyHg0+s/Cn4B/Cfwj4D8Kb4+0v81GYDjj
L/SNc30Gv2T+eOypA/885q+CvxX0t5B3C/xPsPYngZ3YOxae8fA3QH77Yu8J
ffslzKmb8B7oyrgbsBBZJ7HxZugRc76Fdzz0s+hahrxm+HMG8q7J+kzpLC3l
v4WMX4G/A/OrMedCnTf0rYP3KniqwX8B8Eiu74RnGL8N/2D4Z2HzWsYl8K+B
fwH8s8FHw3OJ/AE+Dfw78OHgQ5hzBlvOAh+xvur4K4m+AB/1w9YBDF+Cdx//
/Rldv2fOGPAt+LgJ37IG+AnkzeC/Z6AfQl43dHcFyrBlkvYs9Fbg0/FXW+hr
kPUT+krhnw/sAM9n/nLWMwo4Il+AvwZtJbA69p2nu+4RdPYCPwB9E+PX0JGH
/Az4ZHinAmt19uBpAW0Y+ldB74q+71hfA/Bi8AvhmQPvPOBixhcBLXL8XyH0
88H/pu8DfIesIvQvYH53dM6B3lp3FGsZBzRHdgvgaewbhP0v4budQDP4y6Df
CH8j+QxZ24AfkNVfPmJ/j4d/EvgAeKYwLivn2NEQnYOKHEOWo2tFymv/BBun
Qt+mb4jsXfz3HvjRyHfNHvDrsOUPwOvoP4MPvme8G6iBfbeBv4u+H2KPt6Hv
sPYO9JPwt0DfY4zHAuWxtQn4Duhvo6Mp4+bA9dhynPmj0LUEmAXtp8hn/37w
1swdl/VemcZ/G9D3C/+NRl5b/tspX7H+Hqz/Vmyoj7xR8mme76xPkL0I6ACt
LnAn89fH9u23rP86+DOhdX0KzERXMXhbxncjfzqyWsKzDnwkMt+Hfpb/zjC+
T3OQtRjohOx6QB3dXcx/ANpiYBjfrhTYjL+uQN/V0L6GpwH4C0Br8I2x77YN
0NuCn0H+ceQfAB+Avir8txl/VsSeLxmPQF5Zru+cAmQd1H3PeD3wOPyjI5+N
f6N/A/hGYC7zx/BfEfxJYDtnsRf2VoL2Y+y7rQyfHmHcAPqXzB0BPAt9N/LH
IPtS9B8FHxr7rhyD/RXRVYP/PmHuRngWhc4JlAsMlw/0Lfg+c4pt40zmLlQ+
Av195JUydyI8vbCtMvgLrK0YGAvvCXj6Q+8HXKu1oSOl/ZO2rpVAA75/Q2Cp
8hNs6oOu3hWcC1zFnjiNPS/D/w20d+Bfi/51wF/Bn1MOhS13g3fT/kXGQHT9
ljn78df52NMQeqT7Hdp/gLHofxKe+vC3w+brmVsXfc1YXwL5q6CVaI8FPnN/
Qvf3itHImw3/NOinY+cCzaB/EDqnUy5XWmDbVoLfnLCNCeg/g/cGb8+cqqH3
hPZCB/hXwP+b0LGvHPIbQ6sBfgv4Vvx1AeM1ke9u5YBayxTsCQu9plraX9D3
5voM1ga/Avxorvfwv+A/hn/7J/2NfxfaR/JNR+hZ3RexY+kU6O3hvU/fo9Ax
MA/aoazv7s5ALvg88E6M3wL6gq9OOTZOLPJZ+gz82oTPVHvGQ4ES5V/o206s
WADP49CqofNhxkuhrwIfCnRE9iuhc7unsadG2jmTcqWT+OdW+OfDvwZZd8A/
CNqG2HexcuamjN/I+m6fm+OzcACdTQt8Jn5kbV8hfxCyBwKfQ5+reML4OubP
09lDRldkb4V/BuMb0HE3+u/nv8PMPwT0hr8PUICuncpxlA8B1bF3IHO+yHVO
p9iRiHw2FUN09zeO/G0VA5R7D2LO+oRzcN2Faeirc30nXgx+PHKuXAEbBmed
kykXq8QatzLeAuzAtjbK7+GtwpwCeOvhw8HYUintu2okPMdY7+ehc9/Oif/n
EsxZmOOc4hXlzkAe338uNtSHtwT6CsYLch1bcyPvFcXYVfBeACyCdkOhfXNL
7FgrH60HXwe00X0PhJFjtGLz/2TqrGZ8N1QF/w78HPiIpHOIpinHHMWaVYov
Gee0ymVVMzwIXjNyrvQN+r8GHwheAH9T5QDYNjntu0Znpj62TdN9XM4xRmd/
SuhcWHeAbC1lfrLINh+Cdhlr/jvj8cA5xRL+O5zwmVYudD9zlhU6J8pAzwJb
8n1H10b3NWnnrkl0TI6doys3156/nvnD0Xe74h3foDu6VkKflWsbuujulA5o
+Tpv6OoD/DXpO65Ud1HWZ1F31uWMq2cdS2RzedZXkfkXM38FPBeh6yb+G1ls
H7REVivtJ9UfqiGZ+7Xqh1z7cCi0YRnnEvug56RdY6m2Uo6j2kc1k2ol1UBD
wZeErkUUM28D71XBd7lqxL+kvce0t3Kx/1eiwX8n9h0scq7fELxHjnP+X9C1
CfxowjnoHdCfjpwrTWUNIxXf+e+dYt+BWtuF/KeaXGt8jfHn+KgV9j+F/eXS
rglVCw4A3gpdo6o2vRd/zdV+Zs4F2FYVOBk6h1PuJp90znhPaC+shb4PfH7o
vdud/8Yxt03Guc4Deb5bNjH/5kLfMZOgD1fOjz19of8T+hH4+0LrmvTZ1Z7Q
XtAZrl7BZ1hntwr4YWhvKEdXrgg8yrg78GrSOcbtGcdkxWLl8BHzL2JOLeRt
4L8DyDoIbFe+g4/uiX3n6a5TDP0I/DD3ZVtoa8E/Bv977Lu2JVBbtTqwj/EE
5D+qs4mOSdg2BKevQfc06I2gD0FfTcb7S6zrJfjfw5YZKdf2isHTkP2Q6pNi
5+iqhXSGdXZVE+nbv4uOoXneAx0r+I7X3b6BOaflz5RzX+Vcyj3Kpx3blYMo
9v07dCxRDGzM3OmhazHdsWcZT4enAP7pqqHUn5E8bMkBr4C+c8pZ0bef9WxQ
7ZCxb9TTqIeuorRznTLgP6HvTN2V9ymHh/ep0LWYauay0D0W9VZ0R6j2WRua
VzVQo9A5tXLpvygf0vdS/YPuYTn27dsZ5wrysXLpjcx5Numc+pfQOYtyFd0h
6l0kso5N6mEMhLad/x5OukdUDfzCtGvRxfDXxp/XZ11raY/p29wQO/fWN4qz
//dZru+gOO0ejXozwxPu/bylGFfkHlBpxjW1amn1QEaCH4l8l6im7oCsumn7
LqUcBfygfJZ0zqha9cPIuatq1jHQ7tEeyndNcgn23R66FqpYbF8pJ1EuIp/V
gP+N2HvvRWBRyjFdsfxqdB5XroX+n/Jcc/1GtTBzluS6B9RauTL6N/I9HmFO
HLkmVC2oGl69GflQvlOPZjr857LOvbWHbopd86nWawL+MrL/BGSTrkE6K/eD
3oz56XI+mztTjuU6o/tVb2dd245XvZr1HaCzrzN/KfrvCl0L/brYuUeb0LWI
cpBB6GqinFO9OdbfFV1XMad2sXtQym1mh859lePUzbhmVK2oHl9l+K/M+q7e
pf2g+gX74oTXMETxPvTeVM3YLPYZ1dlUj22rcp+sewGqURvDuxp/NYB3Qr6/
5QvIKEn6mzZPOcdSbqUcL4f5c9LWPR+YF7tHpN7QjdA7Ibtz1rmcYsY1imXy
ueJZoXXdGDo3kc7d4FdnfHe1AD6GdknWtX8tfUNo5/HfKmgrgQMZ1zyqdbQn
tXbtWe1V+WB82jFEsUM5k+ZGaZ8FyeiSccxQrFCPq0y1Y+xaUz2qbtAqh64t
1ZP4Cnr10LVZfeBA6Biu2K2e2o6M7yjdTTrDGfWKY9/N6oE2Ab8R6JHnmupm
xntLnGuXqWYIHVMUS3SnzUbWnNC1t3oCAb4Yk3avUXfKgpR7yOodq+Z7WWcZ
/srQl6pnjfzjJa5tZun7g58qca9INestGfek1YueWehcpiL6TxQ5p1Ev5VyJ
Y7F6Kv0z7jGqt3io0LV2rdBj1dx/Rv9HoXthqqH3MPemjHPZKujsHftO0F2g
nuhloX0qX6pnp1xgUNa5s3KCD1PuYap3qZ7jPNUDwG8Z/053UugaRrWLaiLR
NpV474mnFfJuzXjt6qF9C/2TlO8W1YA9oPeMvdebgM9H172qV/LdAx+cdg9b
vWvVAKdT7lGqN6mcrED9sZRrS8Vkxd5rM97bisHNQtcIqg3G48/NJe7JqRen
nm3/yDmTciXlgG2U68S+y7Yg8x5ktcu4Np6hBwx4t5X4beBDZHwDbTnwx4Rr
8GYp9wzVK1RMvgO8pfLhPPdY6mW8Zq1VPacm6r8rxyp0zpOD/J0l7i3OLXIv
8LnIvSf1BNXrOwr/xKR7fpuR91PWvVHVOOepds76LlNPo4Lu0ox7veoRLUbf
GeS3LHANWgna+RnXjupB3pp2D1+9e8V83RVPxY61ujOG6a1D8anAPfHN6lUA
2SLncOepXxX7bUY9evV6t2DPc0Xu+b4ZO+Yo1owDEtALU/ZtTeytydqy6N9d
5BpftfEPWd/tqpH/mfIbi95W6gAPqpYEf6jAPfMtKeuQbH2Tz0L7VL5Ujtc7
6xxUuadsLlHtElq3fHpnym8IejtQTFPv8Y6Mc3/1ID+L3QNU708+fDzjHqB6
f/qGe5E1D39UY/yCcpyU96D2nnymWuiBjGsx1UR9FI+zzo3Vw97KeHPWvSv5
TL3E7ZF7KeopXofsOmmv/X14PlQuk3bvR28am9H1TNpvLccS7kWoR6HehHoS
X8R+09BbhnqAz2Tck1MvTjW6cvvlsXt/yvHV2/qU/6aVc4+rWuQaQLm/1riN
8euRc3XlCJPR92bWb3HKcdRbrxa6F68ee4XQPUT1DvVmtov5TWLHfvWQPg19
xnS2pPPFjN8U9JagGNgm9hnV2VQP4srQOahyT9VIO/R+lXVurZ7yImybELm2
eQe4K+Uev3r76omfiuwz+Uo9ar2FzI3cq9abSDZ0D1W9U71R/By6JlMtpjcV
1VLNgUp5rql09+/JuFZRDGgXuwen3pvutPaxeyrqpajH0y92jqLcRDlVWeye
gHoBegNV7TYicm2sGi5SbpFy71E9eOUq40LXFspZOsHbMXKupB6Xeg+TgMJC
9yAOpd0DVu+3jXKWEvtIvlGPUm81ivmK9XqzeT7tGKXYpJp+Kry3pdxr0BvK
emTdkHFvUm9centST1m9ZL1BKZZ1id2LVEy7CdkrY+9d5Vx6m81Fxtik32iP
ha7JVIup53I04xit2KwacGLaOYpyE61Jb5090Vc3z2+ep+A9DSQL/ebzZOSe
nnp54nk9bR9o7eLRW+GdsXvHejN8E1oycuzWnTJE/eCscx3lDCeQ3TN070g9
Mt1VHWP35nRn/az4HLkXMRL6BOSVi5ybqYfRLXKPU73Nq/XmFbnnr16/YoRq
xcci01Qzlo/8BqS3H735qPezPrRs9YDejf0morcQvUnnYWu5rN+G5NP/AhnY
cm0=
         "],
         VertexColors->None]}]}, 
     {Hue[0.45155241113439404`], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFnXW8lNX2/+fknJk5Z/qIXuOKqIgiKBIiBqIIKmF/RVAp9dqg2F4lFOva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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtmnnATdXXx69ncJ97H845dxAlTZSxknlI/MwaFEkJmSmkgUy/IpSSSspY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         "],
         VertexColors->None]}]}, 
     {Hue[0.41309087267285555`], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmgm4VNWRx1vee939Frrvvd3NIgiICEoAQaOC4BaXuERUcJdVQFAUAffd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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtmXnc1XMWx389Pfc+9ynub7m/p10lbdKqlCzZKoRKi9BCRSLaZE0YS/u+
UJGZvIZkKUYphVS0aCqVEcXImi2MEQ1D8/7M5/njvO73/M75nu9+zuece/yg
ET2GFwVBsLxCEBTz2z0Jgt+jIKgLs7okCAbBv1oIgjNLg2ADfLt8ELSMkWWD
4EN0TodvBf8y/GfwK+jbKQyChvCNoN/oH/NtcA5bUEfZh66lXQH9ichGYOM2
dFdngqAxfWtj71VkP1QMggHwFeHnIG/Ct5bM5U36jEY2DxsZZP/C3hW0n0De
Kw2Chwqe/4/M+V1k07FxDvKzoVPp24xvj7OWlozXAlkdbGxA/zA2ByEfx7dT
0R2HfCu6w+Fbw58MvQF/LnQU3abwmxlvI9QY3ZGVguBExp6HjenIpkGr0D2F
/ifQrgfdDH8S47VlvLeZw8vofnNsELRn7yvy7TxkP/PtFtrbWfMB2h9D9cr3
4xHsPwrVYG3VoVMZ+wD9a9F/B/wN6A4tC4IvKwdBZfr0Rd4PasnY39L/GPq2
QGck/HDoFWSvQdfTPswdqMhZvIG8QbHnVJ35fsD8e2k9zPcK2p8iv5W5NUSn
PvZmwk9GPgm6n3YRNnoiX8GeNKB9Imt6AX6XbMK3hl+rtcFfh/3XsXEWc9+E
/auZy8DUe/07823DWh6mzywu6Rms6TvG78O31bRH8q01/VvQfy76S+jfHvl/
+XZNzmdUT3eF/ncx3nHM53h0p/FtPPIHoU+1duhuZNM5vx/pW4S8b853KoN+
M/hhOe/RQ+hOxd5NjN0O/bnws6EXWEtr+La0m4U+i+3QRHQ3YuMc2lUZ4z3s
74c6Yqs/fRYjqwQ1QFYHndvo2xdqjKwDtBPd3dAZ6HeBr46sCvu3Cv0PWd9W
5vYr8quRPwA/g/GnpX4LzZjPZmSD6XOk1HegLvoDmNMI9qMWNs5k7Bf49hdk
f4aWwj9V8F1sRJ9vGWsd8mL0+7O/8+l7Cd/2wp9E/4bIruLbSPi68GWMt4fx
emBrYonv4vXoDCnynbwP3Ub0X571HXsCvgv61Wj3gNYz9w2p72ZH7uN05C2w
0Yq+zZAPQvcrvdGs+RdpT0HnxiL7nIi5n6b3Q/9RUAjfUnegnL8w9B5rbzXm
EOb7GjodWOubzHcBYy+EukoXmwu1v9BUbG9hfbV0v5n/bPga8LPgv4TGwq+D
v5m+Y1L7mgz29zFWjP2b4ffAV6BvAj8GvgH267C2Dsx/OGvNscaHsHUQugN7
b2DvYmx1w8b3zO8QdIr2Q+eP/H3sXYj8IuRfI/sKWkTfk5GvQf5Wxmd3GP8w
KvAZNpC/Qb+UtdZGHrP+AvQg/HrsTUX+Hv0vhr8EmgLfmPlOzfnbPmST815r
V/iHkT8SeW+6w2ewPTD02SzK+myas8bmlX1GJyMbgY1fkD0rn47tOfTfkLHP
rI+sBn0mlFh+cmQfoLcfYrMG8gbYOMRdL8rYV76j/aton7klsU+XL2+l9yp/
Q//TKtuHlNG/MrQA24uhXrrr9NnL/qxDvxqyO/KOfX/VHSv4zemtzUb+VeIY
JV/cDf4i+NvR74TtlfTJwZdAzxR7jG7aW2x2Rr4K+Vr6fhP7LR1E/g/st2V+
5zG/C+lzD7K7Y/u2ehmf1behY6vO7Cvax0d+uyXwh+AbRX57leD3Jr5juluK
eUeRXYZOffrPzDq2H2C+91Z0jBxKe2HBsaQafX5D3jN17JyGfh5bx0LfB/Zx
P6M/CP4QsoVZx97i2Psnf6m3UgOqnvGb0VupBl+W8ZtRLK8NXzPjmD4XPoQ/
Vr6D+W9nrvMZ/+1SY4BHae/i20baz6DfCb42Y85B91N0ljP3zqnbMTodaZ+X
+q7I555N+6zQb3GnfCp8u9CxRfelM+3z+fY5ul/IR4aOsdoPzeEyZL2h79Af
rfuC/HL4n5AdLnXsuCf02SmG9Fasjz3Xn5F30ftM/TZHsL9tkf0K/zS+uTdv
/ihvs6vslzqGj6Y9KvVZKMafj72rQmOfg4oh9P8P8qWVfCarmOslqX2D5rgo
9RvT2+lQyb6jFn2ezNqHXIr8Ws7wZp0d/BvwI+HvhB+B/gXyr3ybS7uMMd+i
vRU6k/mM5tsfnNUVqdeuOa7jLHqktlWn2G/5g9i+S296C/PrU75fvzC/7bRP
l4/K2Me1k+9IbXsM9DLtOalj6VCoa2gf+P/1Qf9G/4LUZyWfKd+3DJpSbB/4
IuN1R/4D7dp82637A+2lfbv2g/ZHkdcvnZXw01PHysFQhcgYQHd5EPwAZP1T
z/1p+vRgbc8y3uRijzkT/sa88VA17uQO+tdhDg/Qvx/fHkTWFJ0V8uclxq6b
oCszxrDjkQ9B/gPyrchvZexbFPNp7ylxbG0c2Rcoxk6Ab4b+Svl35IPzXoPm
Popvg+k7MDFW3whdqbUoxtFeB/Wm3StxbBsHfzntPlB3+MXwl9G+G5vd4J/D
XmP5Xr7VxT/VqWzsUD9y7BCGaIFsXLm+fPBb8NfD1y4yZq7C3MpSY7Ml7HcP
5Jcm9t3fZH0Wp7Ce17I+k1rIakIz0J8pjE778ryxqeZ3Q96YSVhJYwgLtI6M
RYQJ9nM29ZBPYOx3OJ8NiTG0/PWJOceWurHPUjFGucBTUJJzTlA97xik96OY
WCUxphGWmQRdh/w6xmiF7Vuy9kUL5fNz9knCAmuEKSsZEwgrz8Ze9RLfEd2V
Teg3y5lX7jMI/aOlzoGEVU9RjpIxZtVbbaN8qHx/9Fb1RvQ29GZl+zTm83zW
Y5yZ+Mx0VopB2ttGyJdkvcdf0D4uMnbJlPju6U7pLukOHkq8h9o7Yf7PaH8e
OtcrRmcs+p+kxsoV4Hfq/mNvJvb2Qf9E96PQ2KSiYhztL0Nj/Sz8HegPjJxb
Hc0aC7QOfdeECb6W/4yM1XMljjWH846lijnz4QvII+S3Kz4VjLGErZTTrIO/
DZ0+RcZ0yrU6IC8uds41Ev6RvHO/I/TPob8b+ZvFxvgXs/4uiWP5SyV+u9/C
98z5DQt7jUH/jGJjsDbM5bHIWKs++otjY35h/brwj6O7KXVuuFs5smKN/GXO
PukPnX1q7C2MprO+Ju9YozNXrjQf+5szzpkWwZfBx/BVSoz1zkJ/dsaYT7n5
qfBTM87RN8Nvio1PdAeaKJfOO/f8W9a5+2nw0zPO4ZWr1c87Vipnawh/buxc
dBn0Cu1uigHo75QPgt8K9S7yHRS2vz/v3FAYX3d7eN65hO5489A5knKjnVD/
0JheeFEYf4buImeyjfmNw2b1yHdKd0kYoqveV+y5aw51It8x3S1hij6yL7yT
M+a5VfuZ2BdqjqsYe3VqrL8fnzAc2cjEa1MONht7DRl/CP1fhy9OjPF0H4SZ
FkTGvMK6OpM7kS2NXIt4V5hV/pPx28OX5YwNp/HtpxKfQVZ3qWCsKf/bmfld
mzg26E4Phb8k8d2T/lrNFf1qGfuUWxTPofWlvoOvont24lpKQ+WotG9IHEvW
Kj+hb3/mEGArLHVtZF7BtQDhv18S55DKHQfQP8F3FKBPGSunGoXwKjppzjWS
sfRtkzd20R1W7rgycq1HOaRy/4PHGlupBnAv/Fr6vM58ejCHSayvb+JYpJij
2JShf7+cY9RY+Tf019C+AHmKbEFsrK4734/5vIjOgGLnyEfFp459AfKeii+h
fb1sKJZMxsb9OccUYfHxsXWFyUcz1qiC16qY2j3ym9VbVYzuGhuzC6sr5z0A
f1vi2LyC/v1oZyPvndao2BvBD8w5Bg+DHxI6F9GdHh4Zswur682qlvF9aKyl
moawqDCfsJ4w6TLWOjbx3TqqHJr2jtC5h3LUeehvCV2LaAs1EbYruFbTiTNq
z3yrJo5djeAnyD/Frs0oZj/PfM4oGKv8rnoY9u5jTjuO4e1hf1hknyxfLB+h
2H56Yuyv+Kvc/q7YuYhyfOVaj0auPSnnGho55ijWyCcp15hf8NkKYyt3eFJ7
lnMOEdP/T6rvHOM7KGzxtHLUnOc7M3SOqdxSd2Ab7WGxc7U1xcZOiyPnasJQ
Oflb+UfGmgW1g19QsG+RD1qaGAMJWwgDDKV9JTabq7an+kVonyVfpZjfNrLP
l69XjFZue3Xo3EM5bs3IMVOxUjmQYtHtsWtfikl1C36jepvCiGNoj0C/De1d
JY49f6jGlbPPuRb9mnmfnWqIyn2rx16rcuA9yN+BepQaU39N/52hc3nt2T74
GvBVc645KZa9kthXKKa9QPvvOg/6j1chmPbRxG9V9UrN5b+Rc0HNSbWdUbFj
p2o87wurha5lTVb8KDNG090QZmwVuQaj2otqIHqLJ8BPzPlNKvd9VzWAnHNg
1fKeiIwlVNNT7UGYUFhQNQjV7t6PXZtQDS+JXGNWbVk1P9Ui98euTagmOT1y
zU+1PmFC1YK+iIwdVBOqGrkGqdqjatKqVXeKHdtUsx4hfdX4Kjkn+pL2wdS5
nL7NiFwDVe1TOZlyg22pcxnlCI/Qfix1LqYc9hj2ZkXq3EL3fady3dS5iXz0
EeUaqd+i9uhn2pMi18YvYo5zI9eYVVtWDilfE8TO7eVz8pFr9vI9qgmqlnlV
bKytmqZ87SeRsat8rmLJzIJjg2JKE42V+OwVAzeH9jHyLcoBVet+M3WuqJq3
am13xsZyqrm1j1yD03mrxrVcc4E/G9v9ivxfwpTYub7+U1gJv6LgXE056vqC
36Denvoot32p4NxYOeDHtJfn7QsVY/fCP5a3L9YdUq65ueDcVDmnastTY2Ml
1ZhVuy1VjlnqGu6kvDGbsFpH1QDy9hnyFdKR7xgfujYpH9IEWdXQ2EGYtBl8
09S1SsV7+c7mqX2vfOiivDGzsLLeaE/VMljf4RLX9B8u2CdtL8//txXsE+QL
9CYnqNZT8Nmr5qvcYVje/z0oh9Dbfibx2HrjZfBNy/2ZYvYneWMOYQ2NKaw7
Oja2EeZV7pINnZsph9F/H8+W29N/IDdFrlGqNimMv0TxutyefPZkbM2Knfuq
hvM2sntD16pvytq3HYnsG+Tj5Fs/jCyTj12v3DN07UAYSdhtVfn9E4arqdwm
cSyqCd1bZp8lX6UYsSuxjmSqkS+Dfy7xXRPmF3baVPDZCkOpdqMajWozuh//
A2dBdxc=
         "],
         VertexColors->None]}]}, 
     {Hue[0.3746293342113171], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFlmmIllUUx686M+87Or2+z31mdBynXHLLNDUCM61AUT9E5ZJmggmSZjOm
pX4oKlFJw70N0jZaFFqoDCvRplyCMrVSKXBpsUWDli8tbpn1+/u/0YfDPc+5
5/7PuWd7brdpc8bNbh1C+BFqA1WXQ3guC6G2EMKGqhC2sI5GYWH7EPbFEIYW
Q3gvD2EafE1lCIMrQrgK2dlW7KPfHtlE9MeyPy76fGfO7of/BgP16HaAvm5j
Gzn6XdBvh8066Ht86FMXwtXov87+cHRHJPzerH2gRZzdjg+3R59v5twebPyB
/W217EHNbUPIuMtf0T7Jx6EJ52dk+9Afy3dbcJ6HZsGf5nwPdBfgTz8w+kPb
qkNYy/6TUD18d9YjnA34thf+TbAaOXsdawV0P/JZ2L4TWS+w7kDWFb1dyPvj
T2VuGy+wXqs9MKeX8AVbGXbPVfncEvj98L0T/y1nj0LzkQ3j7Eb4Z7HRE18a
MufqLvBmQ59g6x/WoBih8zn4J9Bfhs5HyG7G1kjsjswcI8VHd/o1+l6t+F5d
5bzn5CJCP3CnlkrvT2ltHzdV2bdF7GfQlhr7ey7JD4NZyv0t+yMy+3AAeTF3
vI/i299l29tbsEzxV247lp3ffvB1nDlI3gew7uL7ZfSWFrz2auXcb0g+3wtm
fWZ+ORidVYfgNCG/pM65fk31FV2LjyHvm9ln1UtMMemS2Z5sSE+1oLpV7ZfK
rv9G+D658Yej3yuz/w1g9sjs2zVgXJo5Xp0y50y+NoC5vdp90QuMR8Fcgewk
+TkFvUTMFxe9Cn8V+91y50H1pFW5uBvZXOgYuu9zrgVq4sxZfBuCrZ5F1/qD
BcfwEXxYnno8y50n5Uj7iql8lu+dSuZVL7vyVDNtbecL+r4AfhFawJl1wsO/
C+Ef4NwA5C8WHPvGFP9h0b2tvt4C3n3oDQGzd4X7/HyPI++ZOw5j8GtHdM93
zRxT+bMG+aHouTAJ2UD49WAOxs8nUp82oHM4Ol9NRded+m4y8pnRNjtc4Lqe
ULSv61N83gVjIbjDwBmnGRA9e1qQb80tn4E/S0uO/+Si61d904zsw+iZ9lXZ
9mRrHuf6sncc3dGcHwUNRP52O/eU4qn+kT+qi/9yvLXStvsWPR92gvNG9ByV
7rHkv+prXnSNqV6FrVgpX7IpnZz9LDrP/dF/KPl/Cj/ntPcsH6jaz93HO7n7
jlrPs+tZNyMfjN/3lDzvNOvasN6Umx+gnsg9G74Er1P0vTQ7Z5acl37qs5L/
Ecr9lVC3Cs81rbrjZdhaXOfe0GwYEx3fD1jHq5eKntnqZ/XybnT+ZO8WdFZk
/j+pN+dmnjWaKZqV6qWtaYYp7rr7LDAezpyvraw3gDWI/Ytzzw7V7cesJxN+
U/RMV266oPNp2ZhT8PlW6Ey14zum6Fr6DPs/RX9rnncvOLZjkY+PvvOSGvuj
WD2FD09DFxVcN/XpX6mcDCr77PGy8yReNb0szUz9fzsXPLty9teC06hv+MfV
P/B18M8k/DPRd9O9doO/puTafIt71MBPKrrX5Z9805xbkWKivP2ecncQvJUp
DqpJ9ZVqWP+rndG5nc+6J3eNba6xjmpSvaIcKP6qcemqBg4g/yz6vy3SOc1J
+aR+U68dS7NIOLqz5pzqqojd05ydWnSv78j/nzFaFc8jyA9Gy06wVqc5pr48
k3pzAnGYWOt7trD/SzS+6q93eodMRzYqc+xvxG4L/OXgVCH/rmSdqchal+2z
1t+iedWoYq83zxWZ3wG6+6roHlNPqS8VO91/DTqroI4Fvy/mpDfGocz/Bs0M
9bRmk2r7tpJxhak4jUyzQv+Q1Un/1ehZqZisyz1D1aevRM9xzcXz/ib/V2Z+
e+m/rBwqlw3Jl03p7aLeHVDr/p0BdnN0PI8nH1RL43PHSPERhnSFszH6vahc
610lmXj15P70rlAsGpNcOdFMUV70FltY5713cs8szat/AVDzUwc=
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlltsVFUUhjfQMjNtmZmzzxQBQShqkVIK+IIXJAK2fVMoQpAIJMZbKIiC
L16C1IiJooAhgvCACBpRI5J4gYJc1YiCAkYT8IKiwoNGYwIKEkW/P/95WJm1
zrrt9e+11p6GOxZ03NczhLAaqoJq0xCGF0Pomw9hPvR1DOEbaCL8iV4hLKkP
4TFoAPLAvPkuqLvO3xbDt1RC2FsIYRvfpsB3QMPQJST6k1hnoZ/IdVvvEH6P
9pHtNGwGJPjwrTsXwivom5B3QGOQl+KfYJtCP9c457JyCIM48+P4dleH8BG+
k3Re6EOojlp68fso/jOwOQP/A9+WE3sm8g2lEPrgvyCrdzW5WqESsS7FZzP8
aXIMQx6KPBn/h/Bv7+kYs9HPgXpU+1s98X7FZgqxjoFXC/wM7F/GdxP0ObHG
kW89+h349EB+p+KzKcaP2H6AzwjkLvwPYnsoNdZjuaBN8J3YDCbXEDCeCf+d
zofunx4h7EO/H+qX9zdh/ZcwzhnzNejeg8bi2w/6IxoTYaEz/Ab/LTV8pWbA
5xyxz5edW/H7ky/F5gl0jcgH4O/GfjT618BjObZDU/u+Tz0F+D3U18l9zSN+
HnkkNleg+xv7EfDdxNgONcEvB8tlUB36S4iR53tt4rOrhk7kT4nRXnDORcir
iu6Nt4nXG92XyE/BzyVfpFcmcL5TNe6ZDviT5HlQvYV+OvqRxJ+F/7u1nJV4
k7G5Ku+eXYnuOaiSM+b7dbfoG6oco5lY4/m2Bf096nH4w+gn541RVTTGwlY9
KFthIizkcxDb/ol7/SxURDeKmI3UfxF5AfYb0S8m92fYr4IvlH2eATn37viS
a1cPb4nuafWyZng22DckxuoCmC0Fm7uwOYL9dGIuxL8F/VJm6RT2T6v3y46t
HM/AH098l+pXYflAalthugFdX2zSatu/iNwHuQ5aD78G2lByvoHo+5N7Ef6n
8+6xdvS3YHs1uqPYtKK/CP0L3QQV0V2I3hXqienUc6Js/M9QTyOxp1XMXwPt
jb7jpuz+5iEvTNxrh8gxhNyDU9dyALke3VpoUDYfb2H/ZjR2miHtwmbi7yn4
W4XYr0fPsuZ7FnyO8/SGboe/ueyaVIt21hHsYzYv2glF4h0suxbl2yld0Xdx
JfoJ0Rio9huhR9BtT13btVAb3+Zn8y/MdbdDEs+i7ngn/Esl5xem5zn75al3
42bua2rJGAkb7Zg67Qts5tW451YQ+xO+tRU8Y5fBn6vYVzu2gG+e88/Jex73
IZ8seldpZo5E7yTtiuuRt5YdQ77aeVO121PXojsczvyNy+ZHM7GLXFWp98Nc
ck6inolZ/2umteuPJb4/xewi13/Rs6cYR+G/gK5D/p4zjaHeWRX3vmpaif3a
1LuvteBa9a0104/Cdmu0r96wNnKlZfeuelh3WYGezPlO1Yt3Fv22qZ5tqXtS
96UdOwZ9knqW9YbcCj81NV7aBwdSn7EtO49iD068S5VDd70x9a7XnTeiuz96
FwgTYf9s9r4pZjW2uyvGThiOhH8j+u1WD4+Gb47uXb1JTUW/wXp7NZPHo3e+
7m8S8mHkJSXfpTA9hryiZJ3+E+yO7ln16i7hQry50bOlnardoh2r3aodo13w
fOK3VXjei/3HqWvXnah3X0381qqH1fsPF42lMBX/QnZ/+rYu8UxpljTDmq0V
id8OzVg/+M7Eu1Xv8y/RM6z6taOjao2ePb356r1c6rdD+HVEv/Hy1X+S85r3
7D3VDLTgX5Ptc71x+yr+D6X9q5nS7p4fvXt0X/8DluInMw==
         "],
         VertexColors->None], 
        PolygonBox[{{4103, 4298, 1625, 4855, 4102}, {4035, 4704, 1966, 4669, 
         4034}, {4703, 5052, 2364, 3712, 4702}, {4501, 3999, 3998, 4502, 
         1819}, {5050, 4701, 4700, 5051, 2364}, {4519, 4325, 4324, 4520, 
         1826}, {4118, 4511, 4510, 4492, 1299}, {3716, 4798, 4797, 4796, 
         598}, {4867, 4866, 4865, 4654, 1955}, {4301, 4506, 1821, 4116, 
         4300}, {4516, 4719, 1997, 4718, 4515}, {4857, 4035, 4034, 4856, 
         2143}, {4793, 4140, 2151, 4864, 4792}, {4866, 5309, 2666, 5308, 
         4865}, {4107, 4499, 1818, 4500, 4106}, {3999, 4539, 1842, 4540, 
         3998}, {4109, 4507, 1822, 4509, 4108}, {4901, 4139, 4138, 4902, 
         2187}, {4325, 4503, 1820, 4504, 4324}, {4704, 4703, 4702, 3713, 
         1966}, {4675, 4795, 2085, 3897, 4674}, {4671, 4805, 4804, 5027, 
         2336}, {4717, 4518, 4517, 4716, 1996}, {4862, 4678, 1299, 4119, 
         4861}, {4871, 3795, 3794, 4870, 2154}, {4518, 4903, 2187, 4242, 
         4517}, {3795, 4867, 1955, 4655, 3794}, {4508, 4301, 4300, 4117, 
         1822}, {5312, 4516, 4515, 4142, 2667}, {4493, 4107, 4106, 4494, 
         1815}},
         VertexColors->None]}]}, 
     {Hue[0.3361677957497786], EdgeForm[None], 
      GraphicsGroupBox[{
        PolygonBox[{{4873, 4670, 1315}, {4495, 4299, 1626}, {4139, 4901, 
         1315}, {4798, 3716, 1967}, {4107, 4493, 1291}, {4795, 4675, 1967}, {
         3598, 4506, 1626}, {4511, 4118, 1823}, {4506, 4301, 1626}, {4675, 
         4805, 1967}, {4104, 4617, 1290}, {5311, 4873, 1315}, {4539, 3999, 
         1144}, {4105, 4499, 1291}, {4103, 4701, 1290}, {4503, 4325, 1144}, {
         5052, 4703, 1985}, {3999, 4501, 1144}, {4000, 4503, 1144}, {4719, 
         4516, 1825}, {4867, 3795, 2152}, {4499, 4107, 1291}, {4717, 4514, 
         1825}, {4670, 4140, 1315}, {4704, 4035, 1985}, {4903, 4518, 1825}, {
         4301, 4508, 1626}, {4514, 4719, 1825}, {4673, 4798, 1967}, {4518, 
         4717, 1825}, {4678, 4862, 1968}, {5309, 4866, 2152}, {4852, 4853, 
         2141}, {3716, 4795, 1967}, {4507, 4109, 1291}, {3693, 4678, 1968}, {
         4862, 4677, 1968}, {4866, 4867, 2152}, {4868, 5309, 2152}, {4001, 
         4539, 1144}, {3795, 4871, 2152}, {4805, 4671, 1967}, {4871, 4868, 
         2152}, {4793, 4139, 1315}, {4701, 5050, 1290}, {4140, 4793, 1315}, {
         4519, 4001, 1144}, {4298, 4103, 1290}, {4501, 4000, 1144}, {4035, 
         4857, 1985}, {4299, 3598, 1626}, {4325, 4519, 1144}, {4516, 5312, 
         1825}, {5026, 3693, 1968}, {4118, 4512, 1823}, {4854, 4852, 2141}, {
         4703, 4704, 1985}},
         VertexColors->None], 
        PolygonBox[{{4512, 4105, 1291, 1823}, {4677, 4673, 1967, 1968}, {4508,
          4507, 1291, 1626}, {4857, 4104, 1290, 1985}, {4493, 4495, 1626, 
         1291}, {5050, 5052, 1985, 1290}, {4853, 4298, 1290, 2141}, {4671, 
         5026, 1968, 1967}, {4901, 4903, 1825, 1315}, {5312, 5311, 1315, 
         1825}, {4109, 4511, 1823, 1291}, {4617, 4854, 2141, 1290}},
         VertexColors->None]}]}}, {{}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{5312, 4516, 4719, 4514, 4717, 4518, 4903, 4901, 4139, 4793, 
         4140, 4670, 4873, 5311, 5312}],
        RowBox[{"-", 
          FractionBox["1", "100"]}]],
       Annotation[#, 
        Rational[-1, 100], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{4798, 4673, 4677, 4862, 4678, 3693, 5026, 4671, 4805, 4675, 
         4795, 3716, 4798}],
        RowBox[{"-", 
          FractionBox["1", "100"]}]],
       Annotation[#, 
        Rational[-1, 100], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{4035, 4704, 4703, 5052, 5050, 4701, 4103, 4298, 4853, 4852, 
         4854, 4617, 4104, 4857, 4035}],
        RowBox[{"-", 
          FractionBox["1", "100"]}]],
       Annotation[#, 
        Rational[-1, 100], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{4118, 4511, 4109, 4507, 4508, 4301, 4506, 3598, 4299, 4495, 
         4493, 4107, 4499, 4105, 4512, 4118}],
        RowBox[{"-", 
          FractionBox["1", "100"]}]],
       Annotation[#, 
        Rational[-1, 100], "Tooltip"]& ], 
      TagBox[
       TooltipBox[LineBox[{3795, 4867, 4866, 5309, 4868, 4871, 3795}],
        RowBox[{"-", 
          FractionBox["1", "100"]}]],
       Annotation[#, 
        Rational[-1, 100], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{4325, 4503, 4000, 4501, 3999, 4539, 4001, 4519, 4325}],
        RowBox[{"-", 
          FractionBox["1", "100"]}]],
       Annotation[#, 
        Rational[-1, 100], "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[LineBox[CompressedData["
1:eJwl1FmIj1EYgPEz9syYMVuMfSwTYzZFUsjuQraxJkvJPrJfWMKMQtm3bEVp
UiJbyTq2IXu2XNhDcmFLFJMlficXzzzvec973nO+M9//y544u3hWQgihkz+N
00JYUz+EVukhVGWE8APNxMfkuvJdbs4tMF3tDKxKDmFKagirOZ9HZoZQwPPU
zEXPlBDS5NKRijIsRymOW39HzW0sEM9WW2WPH3hivNH4MT/DUzQyPsT5zpWH
dvZMtPayuBJJ4kJrk/mp2lTuYJzCRfwOB5ztLe/kdO7Hk/VcqVee+KO4AufR
A33wBw3UTuVLeq1TtxdJcns4k+8gA7viGImoo742qlDpPN24refuyu14hj33
6VcXBdaWWzPL3BzkiHuhN76be6S2lrqHXGh+KR/kDH3f8734vJyOLHOtrWvE
LTgb493POHRQd0OfUeLrPE3Nbt6JxeLTvITP8DC9hqqv5FfO8BrFxpeMh/EV
fhPPFcfqh6O6OEnuhP4J6r8aV+MJ8Qwojj3j/1LcRo8svs0l3BCLrB0S79ya
V+Jv4i+ogQFYKLeeX/NlHmLtRS6231BMlu+ozyR+qcdIuRHIUXfK+QaKT/IO
dBbv4k1qb/JGvsUl+k2J98SF3Fy/DHEa7ht/4EzO4yKMtq6AE6yt0PMCaorz
7V8n/s/UztNjppr9vEU+Rz6XT8sF8V8u02c8l/I5ubHiWvH3wDf5JyfzxPh+
ml+r1wbUldvGKbyGN2NT3E+fa2pLuFz9VXEvdHGe3/F51P/iVO6rfnt8r81V
8GC5/jyIz3IutuozX317cXd19fTsxkfkDuNofM/j7zG+V+Ix5l5YP8a6+/G5
jO/xQzxAY+Nj8Y7d14r43eBl/FkuN+6lT5b4ubpP8TvALeU26JnN6+N3wlnW
8WNuGu9E3JabpP3/nv0DAlGhjQ==
         "]],
        RowBox[{"-", 
          FractionBox["1", "10"]}]],
       Annotation[#, 
        Rational[-1, 10], "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[LineBox[CompressedData["
1:eJwV1nmQz3Ucx/EvpdFMtBEb6z5y1LLt2rXIMROSe7RYdh27KlfWESnnTMpR
ucK6zWhSzIgKI0z5AysahGI6yS2qaYqZjqHH+4/nvD7v9+v1/ny/39/32K1f
PL5fSbkkSeqXT5IRDybJ4SpJUvhwkhSgkXp51SRZgVLk6T2Lf1OS5D/c4XeX
74leGPVQkpThMGqjCT+bnpPtY64vdthnJ3YhjVcLfeR68nrT3vRupSQ5j1zr
dL0c2oKOQo9qSTKSjqycJBfsWwGDHHswCrDAfrN5A63zsQWbkY5c2Wzza/kT
5CbitL2q8aojFW300vjPO7+96mz+bppDB6Kb+QH0rr0S2RRaxB+B5/Cy3hEc
RVO0lX2HXozztH7c/Bz7N9F7jT6mztc/yL9HfcN177F+Um+zc9iCEv1iv8F4
Op93y3oBvU3foLVk5tE0OpdWt3cN8+nqjeopcV6YivO8DfY5y2uBM8hxDt/Q
UziJ2WZH2/siLmC6mfVmvuadRjt7HtO/SkuRaX4F/we5Zerm8czQZnQ5/Uc2
h9dKZrreNFzWu4QfHeskPYXj+AonUCg7hg6hS+WXoJbslHhGHG8ybWnPbH6q
+pz6J9RwvHz1I7R8PF90sLo2vRLPEK2pzjWXwW9Na6gv8y4hLe6TumacN9bg
xXhm6Eqsxlj1Kpplvo35NPU1s1dRV79AXSfuJ39ovCPqYfTpwDV1jeeI/inT
TqaDPerKjItj4YTee3qbrFu67iz3rZp8ddxndo26DrrEXtgu8xGG81rY5y2z
O8x2te6it9O6xHoX/RjtZe+au4PBjlVR/pbnbj/vpvoG9sXvTHthFT+Ldx3F
+kOQYb+WWMQrUlfAEuxQF9JneAV0aqzjutXd6SZ+W+u51petKzqHK/R++jYt
pV/wVtDD8Z47/mT9I+qj4aG18/iSbuX3439I8+y5Lb4PcTxeD3THQrMzZa5a
X4tjxXsU10K7yf4e77XZ3+is+HagGX+4XlM6zPyjoeom9Fe5ElpP3ZGXGe+G
epx6A22FnPgmqLPpJJkGdIi6If3F/Nh4Z9VteU+Yb0drq2/ybqAxf6i6USiv
Hi1U14/z5s8wM9p6lN4Y2llvXTyP6lL5LfyVtJU6U3+b+/2U6+2Mg9YHMJ+/
UW4BLVMfwj7+3vidzCxFuvk3+d/JfY8leo3kFtO/5VL1KqFyfPfjO6e3HMuw
BxPMHqSHcCDuJ8rwrmzfeD/id/fclVkfQkZ87+g0z/Z0bDO/3d7b6c+843qz
zG6lM2mx7Eh7FNFX5abF99Y6RbYNrULL6Z2VXyR/hk52/JfQSd2RdqSd4m+d
YzSTbUxT7Pmt/ivqqdhvn8/xgvlPZXfH9zd+Q6TKNVffim+07G36QNxT2dHW
R2mJzBFaJDccuer28RzTDvEd9RuUl53jnP9Sv07n22MeZukPMNvYOkt+PM2k
6+U2YC1Oxv2nDXgN0V+dr86jg+I9t99EOgnrnfM6rEUGf5XeMVqkLkad+Lul
vu6Yx+k1ukZmNUp4M5xPnl5/VMAHqCr3Pr2X/0f06Wd6C80sxiLswoj4LsXf
d94nNHFec2XfiP8BzPXG/3P5JQI=
         "]],
        RowBox[{"-", "1"}]],
       Annotation[#, -1, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[LineBox[CompressedData["
1:eJwV1Hloz3Ecx/GPf8hi7KDlP5QhfzhyZfwhJrIchea+RW0zmW2UaHOfc5Wa
ko1ph1iOUfgDuf2BcmTYnNtstYUiWx7fP557vd+vz/v49P3u++u7PGt2ZqcQ
QrI/ObEhDO4VwvQeIfTDZHF/WsaviAvhHB3fM4RdvPHyNPlLeZr8UkII1cjl
zU8MYRPN0b8JGzE1PoTuiEW8s5FqhuhbqmcJFqORf5hfhDnyeZiLInU5vHd6
36MWV9TecYeF9tc6W0RrUN89hHHyBpqqtw2tuKs+1z2+i6+Kj5g5yJyB6Co+
TZ/rf2DmUtpAGyPEJ9XXimtwHRd4zea04KazFbw6O/OQjyb+D9Q42yy/Rhvl
DdHziZ4Njtp5DNednaA36HH6wT0K8VBe6L4PaIH8Hy21dwzvjLjczhI6Vp7C
X4VnvGreBF6V/Jb8Jp7jBV7zXqFFvJquQZLaUXrWRPe3v8Md38jb6Vtaxf/N
r6C/aCX9SEtoHS2lN2iBmX9oM71v7l/xPP07zcmz47O4s9ouWMDrKY8T/1F7
jtarP0s/0WlmTEWpeAfvLN1Js/VU6q2KZiIfw3lP1Y6gfTynreYl27dAns5f
LG8X9+INpdvNGkZ7yzvoPflKdauwXW2rPdtoG012/pg/kK63KwuHcBDpvP30
ADKQiQG8SeoL9Bab+9F9TtE6WuZ8Cv2pJtb5E/EWZ8Xy8/alRO9MXK+/ko6T
l/ET5Qnoihgs4b2P3gE+4B7viPps8zagLXoPyNL/1Tcwy96ZmBE9L16T2hh9
P6L/Tftn8PJpkrybuJnejr5LM5ahXXxR/b/oG6cJzkfzE6M56h/x18nXopw3
Mfp9iJ6F2nhnm6N3L7/srFrNN/FXfEGq2iacd7ZX7Sy1++geZOjfTTPpBH3/
AYcKt9E=
         "]],
        RowBox[{"-", "10"}]],
       Annotation[#, -10, "Tooltip"]& ], 
      TagBox[
       TooltipBox[LineBox[CompressedData["
1:eJwV1HlM13Ucx/GP6ayJBCgYtZLQWil2eIJEyaELN93K1dZhq5BIMxPFaS0Q
M1334VGmOfEuhcqjdEtLcK7WxNQB3WXOAy0VDLXLrR6fP56/1/t4vT+fz/f7
+/x+mSVTxz3VKYRQ5+PE5SG80SOEk3QbzUwK4RN6LT2QHMJ+fJQawr/yhT1D
WIQS3nkpITxKh/PmohteRwK641X+qeZeo+V0A+8HZj5EgTWb1PJpM03Umyvu
EHem22gX2kbb8URaCJNQpXaV/a9Elj3y1RrMVovbrDvJXgPVFqg9KR4jnkIn
4+G4Ds3Qe1bcO3qsMxkJcR95jXglVmAIb576WfXu9A+aSHfQS+11Gfbw3Md7
Px5AvlqS/hrnWYvVSOFZR1/Rfxnp8izrfMHXIf+SnqOlaovF0/XHOncFnYYJ
6uX0Zr1K8U10Fv9MJMfvRb5RXIti+xSa7ca/StwfS8Qj1HbyXc+fIO9Dr0Nf
3KKe5NzJaPd+P5e3infRKvOzcV5+AR2JIfS0z2i+ndiBE+rFdDvdEN95XIen
DXeqf2uPU3Q9SpzjDrqL57O4F8YhjTcVL/DWWuNpez6DOr4R+vW0njZgt7gB
/zjLbnk7/2yz1XE/Z260RjGdqXdGPIu28DfjbvX5fGfNvKc3R/wcHoxnj9+x
+vu0g26ko+O91P9KvkQ+XvwQUtTHm5mD3+Pz4TfMsNdpupn/Hr0t9EX5VvoS
bTV3BU1HpXWqMDLeQ/ko2mS+GsM8ezZysMc6x5HDOxzZuA25mM5bgT7ea6H5
Fv4i2lfeHH/DWI530YrB8V5jEIZhKMrNT0OmmQKzefY6Jx8UfyPx90Qz6Hzc
xTOPPsI3xHpDkYG9qNDLj+9d/zEzZThv/b1qF2gFDXq9+fLEz4t/Uv8ZS81f
Iu9hphP9WH4NvVX+lvhtLMY78T5jlfnDzrmS/mj+B6xx5k/jfXAvuqpf1G+T
bxIfE2+mt9u7Tu17/u8wN/432KMp7of/eBbR4ziGkbwTzIyiRbwFvIW0kGbx
5lp3AE3lKY1eNKrVqGWrrY7v0ToDsRVb8JfeAvo3XUj3O8cB/GLuUHwmM1ny
GjqAnvI8XXhPx/uMozxH8KveoTiHi/F/AF+L96HSGSfK91k/HX+af5MexRG0
6H2D9bzrsDbeKTMHrX91/E/Xy8VE9cdRFn/P6jeq9cMN6MXf1TnOqDeI7/W8
9bTIGsvU6uP3rXbY+YP4IGrjXTczBY28CfppajnyXvFOWbe/+uD4fuM9QinK
9PrR7db+H3XC9II=
         "]],
        RowBox[{"-", "10"}]],
       Annotation[#, -10, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{3551, 5260, 3550, 4930, 4929, 5246, 5245, 4291, 3994, 5458, 
         3993, 4030, 3557, 5378, 3995, 5250, 4942, 5115, 5114, 5249, 3585, 
         3556, 5377, 3778, 3619, 5391, 3827, 4173, 4056, 5390, 3618, 4469, 
         3617, 5389, 4055, 4172, 4054, 3616, 4467, 4468, 4053, 5280, 5282, 
         5281, 3562, 4596, 3555, 4593, 4592, 4624, 4623, 3607, 4963, 4964, 
         5248, 5247, 4931, 4932, 4417, 4928, 4927, 4926, 5337, 5336, 3992, 
         4531, 4532, 4271, 4278, 3549, 4533, 4534}],
        RowBox[{"-", "100"}]],
       Annotation[#, -100, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{3560, 4933, 4595, 3779, 4746, 3797, 4602, 4603, 4171, 5476, 
         4170, 5279, 4052, 4969, 4970, 3613, 4440, 4834, 4737, 4738, 4736, 
         3781, 5135, 3834, 4980, 4981, 4213, 5413, 3662, 4560, 4561, 4310, 
         5384, 4309, 3589, 3567, 5123, 5124, 5122, 3811, 5125, 3663, 3852, 
         3798, 5116, 3799, 3569, 5380, 3786, 3645, 3853, 5443, 3854, 3844, 
         5139, 3843, 5140, 4338, 4369, 3574, 5381, 4010, 4432, 4431, 3664, 
         4087, 3803, 5119, 3804, 5120, 4748, 5111, 3788, 5441, 4641, 4642, 
         3654, 3855, 5444, 3856, 3848, 5145, 3847, 5146, 4341, 4370, 3579, 
         5383, 4021, 4020, 4437, 3666, 4088, 3806, 4742, 3790, 4741, 3789, 
         5147, 5148, 4995, 4996, 4216, 5414, 3668, 4566, 3667, 4313, 4940, 
         4941, 3580, 5488, 4609, 4599, 4644, 4643, 3658, 4887, 4218, 4886, 
         4219, 4888, 4217, 5501, 4315, 5490, 4314, 3594, 3581, 5489, 4610, 
         4600, 4646, 4645, 3660, 4647, 4220, 3670, 4567, 3669, 4316, 5385, 
         4946, 3807, 5121, 3792, 4743, 4745, 4744, 4767, 3582, 5150, 5151, 
         5149, 5153, 5152, 3791, 5112, 5113, 4766, 4765, 4847, 5294, 3595, 
         4296, 4274, 4482, 4481, 4028, 4445, 3659, 5412, 4084, 4024, 5258, 
         4025, 5259, 4023, 4187, 5478, 4186, 4439, 4184, 5500, 4185, 4022, 
         4183, 5477, 4182, 4438, 4180, 5499, 4181, 4016, 4435, 3577, 5382, 
         4179, 4015, 5339, 5340, 4938, 4939, 3576, 4433, 3575, 3593, 4608, 
         4607, 3805, 4008, 5459, 4007, 4427, 4006, 5338, 4177, 4936, 4937, 
         3572, 4424, 3570, 3591, 4606, 4605, 3800, 5507, 4747, 4935, 5252, 
         4003, 4732, 5090, 5091, 4528, 4529, 4267, 4913, 4266, 5373, 4337, 
         4415, 3565}],
        RowBox[{"-", "100"}]],
       Annotation[#, -100, "Tooltip"]& ]}, {}, {}}}],
  AspectRatio->1,
  Epilog->{
    ArrowBox[{{-1.2, 1}, {0.3656261737333204, 0.13270366218664575`}, {
      0.06771914560170122, -0.17705447588506507`}, {
      0.17254118152723774`, -0.16633346652518247`}, {
      0.24287816864889494`, -0.019495154509239}, {
      0.24287816864889494`, -0.019495154509239}, {
      0.23391518406945047`, -0.09023405817724009}, {
      0.23391518406945047`, -0.09023405817724009}, {
      0.2657225584129874, -0.04334199104476876}, {
      0.24634851994505694`, -0.043141589560121714`}, {
      0.24497536162423633`, -0.06673792423984266}, {
      0.24497536162423633`, -0.06673792423984266}, {
      0.25431909127799646`, -0.07273720865209643}, {
      0.25266973802496673`, -0.05943922020917251}, {
      0.2705325207059721, -0.06478879481221807}, {
      0.2705325207059721, -0.06478879481221807}, {
      0.2873994704916241, -0.07944316392207587}, {
      0.29997188087586324`, -0.07680635065289809}, {
      0.3399919856392869, -0.10479668223802481`}, {
      0.3399919856392869, -0.10479668223802481`}, {
      0.3925845007869496, -0.13015020055397375`}, {
      0.39373593759230274`, -0.14810058625030378`}, {
      0.39373593759230274`, -0.14810058625030378`}, {
      0.47492758392450307`, -0.22203163511688728`}, {
      0.47492758392450307`, -0.22203163511688728`}, {
      0.5584221038674096, -0.3318634553761308}, {
      0.6110146190150723, -0.3572169736920798}, {
      0.6945091389579789, -0.46704879395132337`}, {
      0.6945091389579789, -0.46704879395132337`}, {
      0.7281224575472753, -0.5441364979411998}, {
      0.7281224575472753, -0.5441364979411998}, {
      0.850652923667583, -0.7162748262834834}, {
      0.850652923667583, -0.7162748262834834}, {
      0.850652923667583, -0.7162748262834834}, {
      0.850652923667583, -0.7162748262834834}, {
      0.8728979578705099, -0.7689919521458588}, {
      0.9213203028348589, -0.8426184470116205}, {
      0.990021543722887, -0.9848659461692522}, {
      0.990021543722887, -0.9848659461692522}, {
      0.990021543722887, -0.9848659461692522}, {
      1.0296490608620266`, -1.0569282252306726`}, {
      1.0034463541377554`, -1.0035072022563436`}, {
      1.003284625611389, -1.0075418299563803`}, {
      1.003284625611389, -1.0075418299563803`}, {
      1.003284625611389, -1.0075418299563803`}, {
      1.003284625611389, -1.0075418299563803`}, {
      0.996370017785795, -0.9922171316881888}, {
      0.9941965704216871, -0.9884338284465501}, {
      0.999283959857565, -0.9989336550118749}, {
      0.999283959857565, -0.9989336550118749}, {
      1.0016425308985883`, -1.0034502632740256`}, {
      0.9985091934203936, -0.9970711979258255}, {
      0.9985091934203936, -0.9970711979258255}, {
      1.0006655049197468`, -1.0013898826654632`}, {
      1.0002275568157435`, -1.0004091732418103`}, {
      1.0002275568157435`, -1.0004091732418103`}, {
      1.0002591071998266`, -1.0005435753781171`}, {
      0.9998605970759271, -0.9997308686248475}, {
      0.9998605970759271, -0.9997308686248475}, {
      0.9998605970759271, -0.9997308686248475}, {
      1.0000696587544113`, -1.0001375170297055`}, {
      1.0000696587544113`, -1.0001375170297055`}, {
      1.0000696587544113`, -1.0001375170297055`}, {
      0.9999814631422528, -0.9999597066953214}, {
      0.9999926218939035, -0.9999871500335543}, {
      0.9999926218939035, -0.9999871500335543}, {
      0.9999959747036634, -0.9999905090303173}, {
      1.0000113244675142`, -1.0000221511145038`}, {
      0.9999981357397462, -0.9999967400529823}, {
      0.9999981357397462, -0.9999967400529823}, {
      0.9999981357397462, -0.9999967400529823}, {
      1.0000010312041612`, -1.0000017373910746`}, {
      1.0000024338707796`, -1.0000047468096418`}, {
      0.9999999341386083, -0.9999999910766703}, {
      0.9999999341386083, -0.9999999910766703}, {
      0.9999999341386083, -0.9999999910766703}, {
      1.00000018930493, -1.0000003487112612`}, {
      1.00000018930493, -1.0000003487112612`}, {
      1.00000018930493, -1.0000003487112612`}, {
      1.00000018930493, -1.0000003487112612`}, {
      1.0000000561587097`, -1.000000136010001}, {
      0.9999998674736963, -0.9999997345975675}, {
      1.0000000755605665`, -1.0000001420075226`}, {
      1.0000000138379206`, -1.0000000371562732`}, {
      0.9999999560864699, -0.9999999120897327}, {
      1.000000030261381, -1.0000000583152628`}, {
      1.000000030261381, -1.0000000583152628`}, {
      0.9999999864850609, -0.9999999734185284}, {
      0.9999999864850609, -0.9999999734185284}, {
      0.9999999864850609, -0.9999999734185284}, {
      1.0000000029002634`, -1.0000000065894645`}, {
      1.0000000039647554`, -1.0000000079485225`}, {
      1.0000000039647554`, -1.0000000079485225`}, {
      1.0000000039647554`, -1.0000000079485225`}, {
      0.9999999974764008, -0.9999999950847158}, {
      0.9999999983719277, -0.999999996609719}, {
      1.0000000009444598`, -1.0000000018978699`}, {
      1.0000000009444598`, -1.0000000018978699`}, {
      1.0000000009444598`, -1.0000000018978699`}, {
      0.9999999992857393, -0.9999999985770052}, {
      0.9999999995895013, -0.9999999991545392}, {1., -1.}, {
      1.00000000019104, -1.000000000381821}}], 
    PointBox[{{-1.2, 1}, {0.3656261737333204, 0.13270366218664575`}, {
      0.06771914560170122, -0.17705447588506507`}, {
      0.17254118152723774`, -0.16633346652518247`}, {
      0.24287816864889494`, -0.019495154509239}, {
      0.24287816864889494`, -0.019495154509239}, {
      0.23391518406945047`, -0.09023405817724009}, {
      0.23391518406945047`, -0.09023405817724009}, {
      0.2657225584129874, -0.04334199104476876}, {
      0.24634851994505694`, -0.043141589560121714`}, {
      0.24497536162423633`, -0.06673792423984266}, {
      0.24497536162423633`, -0.06673792423984266}, {
      0.25431909127799646`, -0.07273720865209643}, {
      0.25266973802496673`, -0.05943922020917251}, {
      0.2705325207059721, -0.06478879481221807}, {
      0.2705325207059721, -0.06478879481221807}, {
      0.2873994704916241, -0.07944316392207587}, {
      0.29997188087586324`, -0.07680635065289809}, {
      0.3399919856392869, -0.10479668223802481`}, {
      0.3399919856392869, -0.10479668223802481`}, {
      0.3925845007869496, -0.13015020055397375`}, {
      0.39373593759230274`, -0.14810058625030378`}, {
      0.39373593759230274`, -0.14810058625030378`}, {
      0.47492758392450307`, -0.22203163511688728`}, {
      0.47492758392450307`, -0.22203163511688728`}, {
      0.5584221038674096, -0.3318634553761308}, {
      0.6110146190150723, -0.3572169736920798}, {
      0.6945091389579789, -0.46704879395132337`}, {
      0.6945091389579789, -0.46704879395132337`}, {
      0.7281224575472753, -0.5441364979411998}, {
      0.7281224575472753, -0.5441364979411998}, {
      0.850652923667583, -0.7162748262834834}, {
      0.850652923667583, -0.7162748262834834}, {
      0.850652923667583, -0.7162748262834834}, {
      0.850652923667583, -0.7162748262834834}, {
      0.8728979578705099, -0.7689919521458588}, {
      0.9213203028348589, -0.8426184470116205}, {
      0.990021543722887, -0.9848659461692522}, {
      0.990021543722887, -0.9848659461692522}, {
      0.990021543722887, -0.9848659461692522}, {
      1.0296490608620266`, -1.0569282252306726`}, {
      1.0034463541377554`, -1.0035072022563436`}, {
      1.003284625611389, -1.0075418299563803`}, {
      1.003284625611389, -1.0075418299563803`}, {
      1.003284625611389, -1.0075418299563803`}, {
      1.003284625611389, -1.0075418299563803`}, {
      0.996370017785795, -0.9922171316881888}, {
      0.9941965704216871, -0.9884338284465501}, {
      0.999283959857565, -0.9989336550118749}, {
      0.999283959857565, -0.9989336550118749}, {
      1.0016425308985883`, -1.0034502632740256`}, {
      0.9985091934203936, -0.9970711979258255}, {
      0.9985091934203936, -0.9970711979258255}, {
      1.0006655049197468`, -1.0013898826654632`}, {
      1.0002275568157435`, -1.0004091732418103`}, {
      1.0002275568157435`, -1.0004091732418103`}, {
      1.0002591071998266`, -1.0005435753781171`}, {
      0.9998605970759271, -0.9997308686248475}, {
      0.9998605970759271, -0.9997308686248475}, {
      0.9998605970759271, -0.9997308686248475}, {
      1.0000696587544113`, -1.0001375170297055`}, {
      1.0000696587544113`, -1.0001375170297055`}, {
      1.0000696587544113`, -1.0001375170297055`}, {
      0.9999814631422528, -0.9999597066953214}, {
      0.9999926218939035, -0.9999871500335543}, {
      0.9999926218939035, -0.9999871500335543}, {
      0.9999959747036634, -0.9999905090303173}, {
      1.0000113244675142`, -1.0000221511145038`}, {
      0.9999981357397462, -0.9999967400529823}, {
      0.9999981357397462, -0.9999967400529823}, {
      0.9999981357397462, -0.9999967400529823}, {
      1.0000010312041612`, -1.0000017373910746`}, {
      1.0000024338707796`, -1.0000047468096418`}, {
      0.9999999341386083, -0.9999999910766703}, {
      0.9999999341386083, -0.9999999910766703}, {
      0.9999999341386083, -0.9999999910766703}, {
      1.00000018930493, -1.0000003487112612`}, {
      1.00000018930493, -1.0000003487112612`}, {
      1.00000018930493, -1.0000003487112612`}, {
      1.00000018930493, -1.0000003487112612`}, {
      1.0000000561587097`, -1.000000136010001}, {
      0.9999998674736963, -0.9999997345975675}, {
      1.0000000755605665`, -1.0000001420075226`}, {
      1.0000000138379206`, -1.0000000371562732`}, {
      0.9999999560864699, -0.9999999120897327}, {
      1.000000030261381, -1.0000000583152628`}, {
      1.000000030261381, -1.0000000583152628`}, {
      0.9999999864850609, -0.9999999734185284}, {
      0.9999999864850609, -0.9999999734185284}, {
      0.9999999864850609, -0.9999999734185284}, {
      1.0000000029002634`, -1.0000000065894645`}, {
      1.0000000039647554`, -1.0000000079485225`}, {
      1.0000000039647554`, -1.0000000079485225`}, {
      1.0000000039647554`, -1.0000000079485225`}, {
      0.9999999974764008, -0.9999999950847158}, {
      0.9999999983719277, -0.999999996609719}, {
      1.0000000009444598`, -1.0000000018978699`}, {
      1.0000000009444598`, -1.0000000018978699`}, {
      1.0000000009444598`, -1.0000000018978699`}, {
      0.9999999992857393, -0.9999999985770052}, {
      0.9999999995895013, -0.9999999991545392}, {1., -1.}, {
      1.00000000019104, -1.000000000381821}}]},
  Frame->True,
  PlotRange->{{-1.3, 1.5}, {-1.5, 1.4}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 169},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->273159911]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "WorkingPrecision",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->792799295],

Cell[TextData[{
 "With the working precision set to ",
 Cell[BoxData[
  FormBox["20", TraditionalForm]], "InlineMath"],
 ", by default ",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " are set to ",
 Cell[BoxData[
  FormBox[
   FractionBox["20", "2"], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->1087085504],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMaximize", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Cos", "[", 
     RowBox[{
      RowBox[{"x", "^", "2"}], "-", 
      RowBox[{"3", " ", "y"}]}], "]"}], "+", 
    RowBox[{"Sin", "[", 
     RowBox[{
      RowBox[{"x", "^", "2"}], "+", 
      RowBox[{"y", "^", "2"}]}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}], ",", 
   RowBox[{"WorkingPrecision", "\[Rule]", "20"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2145912710],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"2.`20.", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "1.16792859722224602765129447424632375347`20."}],
      ",", 
     RowBox[{
     "y", "\[Rule]", "0.45468573606984113052448267911632798802`20."}]}], 
    "}"}]}], "}"}]], "Output",
 ImageSize->{549, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->386185489]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1677],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NMaximize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NMaximize"]], "InlineFormula"],
 " aims to find a global minimum, while ",
 Cell[BoxData["FindMaximize"], "InlineFormula"],
 " attempts to find a local minimum:"
}], "ExampleText",
 CellID->316126506],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMaximize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"-", 
       RowBox[{"x", "^", "2"}]}], "+", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{"y", "-", "0.5"}], ")"}], "^", "2"}]}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "^", "2"}], "+", 
       RowBox[{"y", "^", "2"}]}], "<=", "4"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1726927757],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"6.250000005820451`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "1.9559659271231472`*^-10"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "2.00000000116409`"}]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{248, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->526508987]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMaximum", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"-", 
       RowBox[{"x", "^", "2"}]}], "+", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{"y", "-", "0.5"}], ")"}], "^", "2"}]}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "^", "2"}], "+", 
       RowBox[{"y", "^", "2"}]}], "<=", "4"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->283369064],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"2.2500000420815556`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "7.028915517410935`*^-12"}]}], ",", 
     RowBox[{"y", "\[Rule]", "2.000000014027185`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{240, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->411653134]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ContourPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"-", 
     RowBox[{"x", "^", "2"}]}], "+", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"y", "-", "0.5"}], ")"}], "^", "2"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", 
   RowBox[{"RegionFunction", "->", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"#1", "^", "2"}], "+", 
        RowBox[{"#2", "^", "2"}]}], "<=", "4"}], "&"}], ")"}]}], ",", 
   RowBox[{"Epilog", "\[Rule]", 
    RowBox[{"(", 
     RowBox[{"{", 
      RowBox[{"Red", ",", 
       RowBox[{"PointSize", "[", ".02", "]"}], ",", 
       RowBox[{"Text", "[", 
        RowBox[{"\"\<global maximum\>\"", ",", 
         RowBox[{"{", 
          RowBox[{"0", ",", 
           RowBox[{"-", "1.8"}]}], "}"}]}], "]"}], ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{"0", ",", 
          RowBox[{"-", "2"}]}], "}"}], "]"}], ",", 
       RowBox[{"Text", "[", 
        RowBox[{"\"\<local maximum\>\"", ",", 
         RowBox[{"{", 
          RowBox[{"0", ",", "1.8"}], "}"}]}], "]"}], ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{"0", ",", "2"}], "}"}], "]"}]}], "}"}], ")"}]}], ",", 
   RowBox[{"ContourLabels", "\[Rule]", "True"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->288919193],

Cell[BoxData[
 GraphicsBox[GraphicsComplexBox[CompressedData["
1:eJyFnQecVcX1xwElaJCoiERRIyp27KCCxkEEpQQT0SgQE5WIiNIEgVBUUGMB
MYCICBgRbIhKEVBZZBBEkCZt6UjvZZey+97usvv+782c71zv2f9+4odPbn5v
5s6cOXPanJk7e1HbLi3bVShXrtyGU8uVyzyH1WoyrFbTCqac/2/OlhmdNs7o
dMBucs8iO6JW03T59oBHObwyYHltFnigLzfgN/37Aa/17Qc83JUftd9k4MZ8
S/9g+gfTP5j+wfQPpn8w/YOH+P4D5gld+gl9YOgDQx8Y+sDQB4Y+MPSBoQ88
zdMXMHzUfNN0gvUT+sHQD4Z+MPSDoR8M/WDoB0M/eJ2n35Qld1oONN/1OPST
8YAZD5jxgBkPmPGAGQ+Y8YAZT1l6o+VYy42eBz0O/WQ8YMYDZjxgxgNmPGDG
A2Y88f7KzRn2YQZXLWUHtF5qPdBypedFj0s/GR+Y8YEZH5jxgRkfmPHF+ynX
YNiHGbtY1Wi7pu2M1mutJ1rO9Dzpcekn4wMzPjDjAzM+MOMryy5rO6ntktZz
rSdazvS86HHoJ+MBMx4w4wEznrL8irbz2q5qO6X1XOuJlis9D3oc+sl4wIwH
zHjK8kvaD2i7qu2S1mutB1puNN813foJ/WX5JW3ntZ3UdkbrpZZjLQeaj5ou
7Ve0XdZ2TdsBrTda7vS8DavVLF1eAbvY4KTM/56/xw7LvLe22FZwP+faZu6H
ArvDvbfRds7874xiO861P7cUvizz2J+yL3psKF/l3w+4KJX+77w9hv5OZHAq
x9Af8wN27Q47Hp4X+6cBI39g3mMcd3tswOgfGDrgwxCPDRi64UNHj01ZfKFc
80Vho/lYFt80X2lfY+ozfvjK+MHIAVjzVT+RU7Dmq+4H+QZrvmo+/g++lTlO
zQfqT3XFu42rtqO41LyjX/AZ/dLzAN/RL8rRLzDjp30tl/CDeYGPlGu5h9+0
Ax8pZ/4YF3QwLvhS1jgp1+NmnGXxvSw+UK79m9ZrLRd6HaP1WsudXkdovdZy
re2pHrcelx4HeLNr55gt7/pJhie/O/qfPmHfybzX4ZD14lliFrv/s9j8x7VX
aO5x9B4xT7t288KT36nHe7TT3Jfbt3y5PeDL7de+3FIfTHvU7+L7Cc/1nm7D
OKCDct6j32LHz0TgA3RSPs2P3zB+6IBOxlEW3U4+0vxz7XQ45OUj3R4YehgX
dDMO6AMzDsYF3boemHEwLsbBuP7jsQHrcfGc7Z5r7KzM86atno95JZbfGfeX
Hof54km9Jk7cS8yVGXJujNqZ5tsN/dEOT35v5t+3vE/7yL2mA/1wj8+icdAe
74HRF/BFvh8D3egP7an2S9FBPINd6ebKpwb+0B9YtW91P5pu2qUf2pnq+wl2
gHLehw49bvrT6wfaoV3eA3vxSgWs6ZDyECdo+cBvgqn3v/qhXLeLHwUzXujQ
84A/0P1qPql2gh+hHEy7zCPzp+ddv0d9+I9/men9fvDz4Eud/zhh/+Tq7Qv+
ljiB+mDqN/L1De3gb8D4F9r1fupEiD+oD6Y+7VKf/mq6587Qflnjor2yxlEW
nWD6he6y6ARTH7rRJ+i9wj+t5hsYOlnnYM/1fCDfjANMOyc7Pdsa6iO/1AdT
H/vG+of3oA86kCvKwbTL78g9v4PpD32Dn7QDP8HwEz2hfeQZuuAvfKAeegKG
/7yHX4Uunvgt3Q7vaf5iP3XcofkERl+pD//1vNIP9fS6hXbAjEfXpz36oZx5
oV/oBPOknn7q97Td48nv0KN/1+Vg9FBj6sF3TTfl+BtNF+Xwryw69BM9px6Y
cujjd+ijHPoohz7NF8qhT/NF9wvWT+ghzoceyqGHcuihHHoohx7NB8qhB0w9
3b/uT7cPhj7WNdADjtuJ46F/MO3TDv1r+Y3nh+fM0vKr87hafvU8guE/79Gf
bkfLoZ53zXcw7WMPaR9M+2DaB9M+mPbBtI9foH0w7YNpH0z7YNrX+Vg9X2DG
CcZvYMfwC2DKk05+tgSMf6C+xro+dFMOxq5Tn3L4hf2lHH6AGZ+mV8fH0Afm
ffit2yMOJG4oy2/pOJB1H3kF548bFpbaf9T+jn7BzCf+XOff0Wf6BzMenYck
r4G8gHUeDnnkPdoDw38w8wimfbDO29FfPB48XsrOazsKhh/a7kMn5WDkh+dK
9/s429jV21bKzvF+LP/5K76xPiWO0+PU4+qRgdWz/TrwzZTp5uTjUJCfpzw2
GiMn5BNi+bqGhfFzAM0L/LiHH/N5guYFBkw9t05Oy6Frf/dh337FE2G+/aMk
jBus13163abXrfDH/dwvFfjtx5OyRe6ZFTD6CiYuhx/wU8elYOYFzBM6wIzD
87HQ9nR83RzKsZu8hx3CPiE/2k5SDl3UA1MfeUHvmWfsDhi55X30DQw/n/D0
G/jp7cMOk+WxAbf09Q3yDuZ95FHvu/8vuwTfkC/0hPnSGD3hfa1Xet9O77to
O0a72EPKoR8M/WD8Hxj7CSbPq/UaP8B6Rq87NR163Mgf49NxJO/zO/XBOi7Q
9k/v02l7qc9H6DhG+0H0kvfho9YDymP5+/T8aPtJP8gv8gm/dZzJuLB72v6i
L7yv43P4HM8jRPOo/RLtMB+Ugykn/4s+Iwfkh+GP+3l5KuR/df5b23XooH2d
/waT/6Bf1vPYFf0edGnMvDE/0MO4weg97fIe2JcXS5y0w+tb1xKb5czQjkCf
plfvi1Cu++nm/ZXFX/n6R8ReFpiPHf9+NBu9/wj2WPhvT3EELQ/+5UdfP2A1
X5b2KPfjyxM7J/FcrTwD9vNRYLt6uix0gfXvij4DPdC/IfO8tMTLZddd9jdO
HNZ5OtqmbDsfTxiJJ+zrGTo6rPf574dThvHSXq6b958Df9T7Ol9q8Q+MH38C
3uTps9CHXKEP3k6csC28vwnyjX4j39ov4n/c/nxqj8FPI9fYI/SfecOugSnH
b9KPtls6P8HvjLeBo2t9mC/oBGNXaJdy7CCYcp1HwD7RHv1jh3X+CnvH72D4
CiYegH4w/h4+ghmvxnq9Rjnygf2mHEz74Pj515RFHsHxc1apsN7Sfl7HGfq8
pva3+ryj9qf6HJZeVzIuvS5kPqEbf0Y72E8w8qjPOVEPuvT6C72JnQdIl4O1
Xmm/q8+lIFf8DkYfmbey+gFTHz7odST1wdTH78A37FE8r50K9CDPZflhMPaH
+trPxfPpqeA/8VfURx8Zl6YnHpcmJV44KvQlDRh9ZnzIGdjLxf5g35Azvb9P
Ozovx/jB0E87YMaB/aRdnXfgSX/4M/wI79E+dp162CMtF2DmG4w90nk4bc+x
n6x7mE/0mvEzHuhDf8DajjFO+sXfn+/6WW30vs0ah2dYX74p2D/0G/nHDhE3
EcdQTn+Ua/9HXIi9gF/Yn6Cn4j+wR/AH/cW+oYfEVXoe4TcY+rD/8AM94wlf
wNCv9Ztx0w/j1nxjfJTzO+Pjd+ikHDooR/9Zj8X3MYvtM658V9B/MHJU1r4Z
/RK3Mh7awZ+515uWBKz3yXVeKuxzyrzRflzvI0ycTP+aHn3uStsR9IB4k/gA
esDok7Yf8AV/R33oB6MvXg42BPnQ+Q3mEX0HI/9aHrSf5D3q6TwMGDs21M9P
0C/kBMx81vZyb/V88j7Yk1MS7BPjgv/Qif5BJxj9ZD6Ir8D/a98H+Yevmr+0
Dx/oX+eJ9b4D88W8lpUXRX6xY3qfnn71PhV0Uq73tfld7xszP9BFud4Hx27o
eJJ8FXpBPe2n4Jv2a2G/RfrXeQDW+eT/yathF5l36iN3nPsCI188aQ87iXwS
R9AP9dy4i4v9+qvRbvsP9/tS2Z9I2b94PQ3rOvQWfIbnS9Bf5LGDt5fBX4Kx
D/AT/UdOiEfgH/MJ/9Af/CV+kDww8kI/tMf8INe0w3u04/l+MOR9fb2DIf7C
PsBv/B78RW7Lkhv6Y/51HiP4L5kn6vEe9XiPeoyXesgH9fQ6FBz2HUVO4ZeO
P7Tf5b3znOOaKO1uD/IHf5Ef+Mj8UA+/sdHrn8H/xvNkxYb4G7rRV+S1iZdX
c6d7rggYf0x96I+fy0hY5hF9Y77RHx33gZEH8hDIBRh94D0w8og+wZ8RXh8M
8eNoR99EA38ZH/qp19n4PeRNr8N0XoWnXt/p9/T6SWPqF3m7EOwB+ybQRx6M
cbPu5z0/H8Uix7uCnjHeOo4vP9iE+32tedL9/q3wc0PgP/Ovx4Uc6/1YcPw7
ocJgf9Fr9kWIa5Bj2tF2Cv+g95Npp6p3RGK3tgW/iR9AfonXsLvEF6wnyXOC
WX8in8Qfz3t9sfDnPi9PATNPtI9doH/miXL4AEa+NaZ/xkM5egRGjnQ572MX
WIfG43S+v0mGOAn543fWoxrHz/3tKLVepRy55X30Ggz/kLP4dw0FQf5oD78I
vcgHeg1/0VvsLfahqisfZpAf7Kve99B2AT8Q+y4g7CscD/lr/CLlJY7+Y0GP
wzpR/BD1wJTH9+OS8oz252iP/AV4daaZ7FV+/TQmys8jr9jHSa7fXwJf0R/6
Qz+gL75vngj+CfyVq3c4vI8+tXLt/hT6xx+B8cO8F/gR80MJlTdPhPjK822b
2MFI/mgv9h1CepzoB+XYPzDyT/vIAZhy6EPfka9G7jlN5HZjyOvSPn6OeXHP
mis8/dNTdqefP8v8xfK7E1O+3QtXGOrH8rvpcuQGPuo4SedRdF4ROaCceQTD
D9rR+2LoLxg9pD76TDl+IPa9ycb8sM+IvyBOA7PuwP7TL/E+dLGO0hg/R33k
EqzPecFn+kfOoFfv1zIu/D3+jPpg6KYf7HU8vt5Z6nvb2P5d2m6EPKVgvc+u
n8wH86/PD1Gu8/pg6ICPeh+d8rLOxVKuz6GCKUfu0GPkNJ5HTgQ9pn48Ls23
8X2wRFjfET+77xjL7fV6NiHCxAf+mRv4BL+xy9SDf8RH2EXiI7D+nhC7z34o
dILj94FwbiM6r4R8eLbI96E50XeJ5M85Lxfbv18s33nmRN/vUf9Wb79DfIjd
GubjjeBfyMcRLxF/4qepD6Y+/oH62Gfqg6kPPdRHX9kHJh4gHkIewfgRV71T
FL/NkHgJ/+jDhJIgT667AYlg95lfMHzGT2Dv4/naZNBz5pv9buYbzHxjb+jf
lfcvNJdn2jl8pNS5F+wcdhyszztTH3tMfewWmHLex04TBxEXgMN+gdgh7CoY
foHDeS/B+tw0dgyM3tEuGD9M3IJcQB/65/nMuYuE07v+WYX220ytEdF3mMgT
v2f6eSErkjPad/SOy/Hzs6Io5BP0d0nx75P3h3wJdgas86a8B9brbDB5AzBy
wXqJ+AT7xTizvD0K78Xj0sJS+yys95hv5ETPP/2E837SLn4VzLyB0V9w2EcR
jD6Bw36yYOwX/kP7RbA+f8Y8MT6dP9TnvZA//BZ0468ZF/4dTLmWZ95H33V+
T5+/Ij4Aoyc6Top/v3bcMN8+DoryhoxL7/PqvAV2Kn6O5GhYj+A3wP69orCv
iR+N+9Wk8qMnwrjIZ7FuZV0Nphy+Mj70hvW6zoeAq3kc/Imnn+/U8607L1Rc
bMj3Mo6V3m8EPUNO0UP8FnIEph7zy++sk4mPwJRjJ/U5COJI5AE5AbNeIe5g
PQLe6ubHGvJP8Iv8H/kzsM6zomcao186D8u8gvFz/I6/5T3sjsbolc4LE7eC
KdfnXJg36NB2K5yLEn6iX8SR6Bf6A45//3k8Okct+h++6xRMXIA8oV96nar1
Rp+HRH49P6nPPTTo2dEQ79CuziszTjB2gvgaDH36uzzyNsR5jBdMud5PJB7R
58rBtIN+QT9+EEw8H/+ulvszdqv7NKJy7Ek4RybzQ76pyNXLClifI6cceUCO
kAfkLJ5nifbntV9GfrFjYO8nToS8mz7nBB/BOg7U3xFhT8D6HLu+hwD/C9b7
OswrfkTvSzJf5I+wJ/BH533Jd6Mf2HviJmefJ0SYuDp+30gy5OFi66HF0T0R
8Xtxkob6sfVQuj7jxg8xbtYXYM/+KB+M3DGOYb5+wKxvkCPKweE+AZFPnSfX
eXTGSb/xc0nFwY8Rb4KZd/wPcgEe4fqdEfwF9fFno91zYtiPyfL0B7nSckZ9
8rHUD/e7SX/s58AP3gdTf42vH/bXGF/8u8kTIc+HX9H7jfAPucJuoc96f5r6
xEXUZ/71vj/zqO0K/KE+mPrwh/rILf5Q543QO+JY4jlwWMeKHoLj9w8VhPwz
71EfTH3OYev60EP+DUx8iD0rq10w7dKPyp+G+GWjj1cCf8ry09qvY491nAbW
cZk+14GfQw54j3kA6+9PeZ92eR//hx0HM4/4Q+K5sr4r1Pd5Uo68h/1ukU/0
i/0T9BP+6e9DwPAHewjGHrIPST5Jf0cbzneI3NM+WPsh7C1+iHHAX/JT5Pv0
d676XDb8pR/4BqY9+l2See+aZT4+np+Kn0+zqaB/Ot+uzynqfDkYe0V8gn5j
z/V3aNDFE3rxt8QHyAVyRX+Ml/gJfiAnzFP4nlTkhXL9nTfyHl9fR+fY4QPz
wboRPaA8/h1bftAv7C52A3/POtDhq/LUvtrxEGeD4/csR/lV4gDqg+P3v0Xr
RfQK/81608Fh0b1Z6Af5TykPGH1B7vFjYPw6805/yDEYvjF+9AC7zjj0fqeO
u8H6u3v0SdtP9A96wv6u+HP2H/H/Xj63Bkx97A/xBhg9qenonRz223if+joO
pt34Pm1h8G/O/1UsUnkvvl8rjN8nVrEoxKX4Y70vE84dybyBsXN6nwt7QTus
++ifJ3TpdSj2h3kmHtTnjLC7xOP4D3D8XqpVIV8C1ueD9DlMxqXzgvAPeQKH
c+XCN5031N8JEbczrvj9mgUhbgbrcy3xc7HJsI+hz72B0RP93TTyg33GXzEe
6GT8zKvOL+h7McDIK/YTOUPu9HfLjh8vJsP8Eu8T74L1d1+MC/sOxr7AR/QX
v8l4mGf2k7x7ORriNL3/q++HjO9P54cn9DDvjBP7H18XR/d7M37kCznR+8zo
G3EpeoCehXPqIof8jn/RdLO/gD846OMDK/FB8MeOn7NTId7AbxLPocdg/V0U
dFMP/858hu/iBPMeekF8wLyxftBxMXxkvvS9ltQDE+/hL2kP/WPeoEOvg8ra
pwJjt9HLcB5E5gM/GfvO7So5H/fA/iD/6FNZ914w78wPeWn4rs8ta/+KXdHn
esBhv1DoRr/Qe/3Uf48Ae6f3IeETcsl8xPPxiTAusP7enf70uV/GAd2c/4V+
+tXfZcfvb84LT/wNcqzPaSM3+ntOfe5b8xV+IK/IDfIF/2gXTD3e13kBMPwh
TsUe6nUF+XXiHeZBf/esv7NhHOg18kS/0AH/KdffBeh7D+CXvgdE38PGOOAf
fELf6Bf51/fkMT/MHxh7H98XTYR90dj94L861wB2cVe7/FJ/H+Ut/3uw62A5
F2XUOSoT+x5h34kwDtr/f++P33cift97GrtqY1KGc1vkITin5exKVqHPJ46Q
ewHG5oen88trVvnzcGOi92jH2ZPm8j1aqti3v/hYkHPHn3py3/TipJejF5NB
33zeMunlrp6se6YXxfM7JcVe79P9dPTYgGmXfrCLLu/aN9fj6UXx70cXy3na
esdDv/wOX+Ez9Shf58cXv1/zV3Tdlfl9zp6Qf439nZl9st/et/Q98Iw7dl45
XT92v3e6nP7195vwnfgK/jL+2D0XabkoS26gu9CZhT3hHBD8hA59Lg9+aTmP
3e/+q7gPDN3wE/7oe8b1PIKZP+SAJ7+799+ReyqyCi1yDkbOm/lnOOeo83b6
HCR68P+2OzY/tOt+71JiuA/DDfPOVPBnTq/apgz3PFCOn7w388OsHf6+iK4l
vt1ZOwxY6sf/TsKY6Jwm90ao9sP6FL1+wuu5gZ5853bX+bipa4mh36/8OMK4
ONcZuzdojHzHkG6P/mlPjwO+0J6Tu/u2uHzcod+kQvuxv/+Sjtu9O4jOkYFd
u8/KeaXvEpZnM/+7l6PvEgbsykeLHl4rcjZa5O7a3BCfNfLxmaF92nF0XyN2
ZrTcx5uO4y727xneA1Mv9l66HdplPKxTYnSmx8E44QPtltWvHrcep5vvP5WE
OIr3GDf8gV7ogF7ogF70P3ZfWceCkBe82+P4985Zm7zcXpvy69msTQZMeUeP
DeXgJt5eBDsVu3++otyf2zGKpxkvcUYTX25UeYjbYnqR1pMYH0ZHeRj6g7+U
Oz5UlnXi9QlvFyvL33FJY3U+OqyXWH/puECtz+L3QPfNDXkf6HKvvyj7zVl7
g9/FD8fu+wFflRfiSvhLP7G/F5DuB77TP3KI/Hg7X2zoX/cb2w9N9xvb70xj
x891x+0vmX4ai32/Ws6lbs8Nz2n+d0M9N09rxc9sl/xUupwn723w9Q318UNO
bvBLHaP8gSsekAjxMPaeJ34b/+ru3dsu9z8vOeH1d0dxyE+6+/o7yLrgV3+H
AOzysHklhvv3eXIvP/fsc/+9o29t9HduiE+5/w8/oO08dtrpW+W1fn+qdyrk
l9w09Yv8lvZT2q9pPwNW7Vl1P2I4x8i5xun+afgdvrarvOD8rj8X2quWLfrz
a1/tD/mbrrIeoN6rw3PWv9w2up9t0ooaX6yoEd0j4feVo+/Byok8kq/x+6tb
w/7Pu7W79R+yPfp7Znu9QTb6++FKN2X+WxXiycGCO+//OP0v+jtN39f45dzv
HkiEuO9zz5+ALxQ7xDy29+uvEF+98/4Z7XddftDmjqyx7raPj1rXzU0HzBPu
v6O29eYOPT7J/sJOeeCHlmc2PyR5/3w5P3zQumZmHJHzxhKfNN1vLnHlq61v
p9A85p5rfBxSa5usw8bZ0afsHHXKznyb+uayCt9ext9nWx6era5qO+G1j/LN
mvce7NVm8zbzaouPLtpzR8Leklhf8YeP9wsdq0O/0NFZ6ILOON1JeebayU80
uP3SzcX2pyqvdetduNvyO/U++/P7dzxXWPrvNM1d+vxFDw6Pvk+aIzj75UbP
jdiQVPkk/m5KgWn13nNnZtdfbHZ2PTf3p22cvyavcET4WWA6xXD0+9lvTi0a
PHWx8K3EdPft2V3SXtXKOW2z/pBtGwy7uvasQSl7ssfGCL6qQWHlT5ZlmwcH
n1uzuHHkHyb8mOxy69nRuvFTwd3fzfw3wXi5SNl6Tz2aM7LGN6ba9s9PvHxr
yrb7a+a/6XbiZ5n/SswskZfWbt5KzDH/vuX93v59y/usn/sd2nPW0u9KzDX+
fZn36Pu/vlI+aP3IwetH7jJ/3zQ//a/AVneCGd2337t/pX/1r3TCPjz07NZD
z95mq/z+8YeHPLPPJHfvHT2+S6HduPbG1jvL7TMLyzccWNSm0H469/5tE3KT
dsoHPy9vv3eeXbsk898mu9Q9k/bNk+YP2N4g+h7wrD9sO/MP26L75T+bWO2s
Zx9ab/r/+XfvlH8/Kecx19vZjp4sH6/WWi/2P8tMd2T+KPO5y3zkcTjv4OV2
s+xPHA720ZcfkrgnS/b/d5iZ0m64D9HRFd27MFXwasGfeFzqXpt9grsI1uem
9Hfhbwt+9axbB5x167ZS9xcXPl79o+K/JWyH6t+M3pD2G0+6Z3R/8YtbEs9t
SWy2i+Y/1vzv06P7jGP3X1Q9Yra233vT8XMX2e0Tch/411VJO2nhwM8XDsyV
eT5uX1q46aLur26ydXpl/k+OfVX0YfG2G7e+Ne+QrfpOtyvf6bbe1Opa4fKu
FXKsa65wp/nEzTv3ex43KZ+3DXzvJPPQWeZlmsybfy9p3hF5GbLo0zcWfZo0
25dmP7w0e5alHdqt5/XNom973PjDfpO5Yuey66Y0PGouO/n6wSMaRuf5L676
fvEn90T39xYLP8dl7V+3vmGh7db0ujcOpPk5+jcrqxU8UmDnvPLuRf36HTOj
PDbfCZb2Le2Tf9zQYueIxF9TZs2qXgduuXadbd29+Y1ndEyZ9vVfOrv+S2/b
9/vckP6Xsj/u6f3fPb1n25+P3Z7+F9232+yhV2+7sG7K1vD6ZQtFv7Z5/bI/
iX6Rl8vqU6H20oICe92Xz3Zp8O7ekJcb7+zGdvOAsyPR3wOZKfWv9fVD3q69
9ye2k/gT77cK7Y3morQnX2evOPbl/CdvLLIz0v+b02ZT6OfjrGY33NkrO9ib
y2W90kNwWffl9ld4hfgt8p/VGl6w+s4LPrSzH+o3/6F+m819vjyc/8q6+LMe
+Y0/NLXHtWn/SKNtZteTyVMGtI/Ob5PXrSXzjV25pOJr1Wq2TNp751y6tWrX
6PuEjl5+7QSRX/jV809VDl5w/xHqh7wJ+/+c86oseqbPzYz86eJKp1x5OPo7
qoKlfql7Tp+v/+lL9T89ZO/flJnwE3aJq74j7E+82795mpLtIY/O/hx+0cWX
L+y0Bee1S5zXLjrnT37m0AdT7qizZkvYd5wv+tfiy5qXPXRGdD+txFvq3vUf
iQtMF4mXsHfEX/CtidhriRMoD/E1+zmciyAu/MLJbXr92mZsxTZjo3NC63Pu
uHfaw8ds93G7CjcvyLO5RbNzimYfsy/WPCX9L9+u8eWmm5Tv8uWmr5T38v7L
tBL/dbesl/Q9APf8ep0R8nhHZD1Z+r7Vu+PlYX3X0s+f+Unm7zUf76rvNk+Y
N2U+71btNlft7jvw0DkXDsqxNw+4e/Sez4tsfvmvKi35w3L7WH7/9Sc9kbJ9
3HzOF71M2Qc9lvMcKdvhL5V6/KVStpmUHJj+lxK/tUr8aNo+1XP2yWCfXvH2
yWCfJD40N0h8KPGl6T/+4PPjD+42a+7M6OsJW3D2I0fOfmSbFfkI/pL9mk9E
jqzYS+wC9vPfgpEz5E7KS90LOkEwv8+V38eqel7Ofha53mu+9+uDUve0iL6Y
w7LePyFyiv42lnXJa4LZp1wo6w19/+hXgheInrF/ir2cJPR9KFjfL/qM+K2V
gvFjxB3gLIlX/ipx0cdiL7Gn36h45jOJd8D9vHzYySIfb7j4LPqeY7eXN/Oo
yNsIXx6+7/hmwYrs5jevNpePucL2ezllr/Drv1L3gA58u9Xrb7c6YvNqLHr6
ikl5YT0u679S94NOdfZgjf2ttwchHt13+Q+tC07dV+rc2Ugn4Iskvtgs64Vx
sftCa/acezvz81+Zxzbif/V92M3E3yCfv8j6b7CPT+wzEp/csqWK+ax5Ot5t
u6Zh+Qsm2nmyfkQuffySCHb4Zqlf9avqv927LjfIw23vVemek16PV82EfwMS
9o3m3w5v/m3aft72+OEh3yZs/czjrsPmLF9uRkj5SF9u6ko5/dIe4/3xjlPa
PPJQyn435Kyd/2mYbU/+qPHGCoOj9f96iV/aVdn3p8FXrjODT71/Rs7TKTPB
zcN+4x6fFdqCn6cUdE5OMU17vN66csUk4w39Cr2m1axzjzzxw0RzttB7dFzd
NuPrzgznWC6Z8s4nm4uSYR1woM8Xa/t8scC+8/njl17dOvreY8DwnpfU/KjA
zjqv/JWH7l5u257zymPnvFJgpzg7tcyOEv6cyPL9VRP+mfG+v4H9Mw1km2ZZ
Zw5Z2Tlpf/DxryH+nf5U73+efNZS83zP8lMm1Im+v80Rvo0UrO/DlPykeUUw
9gC94rtS5PZZift3CmYdgJ2cJXiHWkccFD5AF3wZJLilio9G+PHa5jJe1jUb
RO8fdHFhsW3lnrvsecfGvt7p2eh7lpTUC/cH+cA9nKcjH7p1/6j6LQtDXrbU
9y0S5wW7Wxyzw3NmfbR/0qr110f3uR47nPlv2ay1T3+35unvovveal6V98nq
96faVnPalH86LW83LHfyZ5G/fB+vmtYSr7aW/EaDTHql8SZzYEXJky9M+9mc
N3TwwrsfT9lnLp+4a/CZS8yCUy4rvK11yhbLeO+U8a6R8Q6S8c4V+Sb/s8HL
i31B5OW/346a+O2oHWagswu5kjfINy3S0fHbbVeK3uTbzyQu//01a9MRy0E7
p+jWFd2KC23LVXVarKpzyC6ZWnvx1NpFtqovN1bKG/ly85OU3+rzSfZJySd9
7vu3g6V/zgew3873x5yXZ98cv4zfmi9ylaX8InZR1rHY6bBuHSXxHfIp+SW7
TPJL4JYyH5JvC+0OlXaGCRZ+mXHCr/g55Og7s+xzUlU+2V5iZq+cfdvrv4/u
M+O8zuTFMz9fPDO6l63XrIZ9mqXt390blhVVvyvbXPpKzfdOStu/w4Nd/seM
kvzPyOL9DdvU/tmU/93znbfnlZjWzt6sMHOvP23e9aelzPo2Fye+HrHcdD75
w2nnlY/uP2P9JvNvjcw/duGkzrVfavlYYfiu7QKJB96S9zeK3j9TK54fwH4M
rnzjtQu/ie4Na1r+4vSKJrpvjPJb5jzYcOb60veNebubiz8q9Z1ujuAipQ/r
lP5PFHl+zz03mEkHZn55YGa+Xd3j72vT/4J9uHNlr9RN7VJ2vIp3wt9DEvpr
fHDh2EVnbjcnP1CvoWkQ/d3Dd/rt+HeTe6K/b9Kz9ad/fGN5oa069/FJ3ceX
vj/sVbHD8L91enU4fVn03dMLTl/S8arYF+zUsku8v8d+3qvs6VCJB5oKvtbn
50ROS0zjao3vq9b4e/Ol40MJ8YFZ+3AmQDhi7Klj5pw6Jml+zoQ5XxwJ+0XY
z2le3izy9oSXNztP5O2QlzfbReRtyspMwD3Y/G7/2PS/6HsxH3eXmKc9PXaS
0DNltatvqf9Nr8zCdp69/6S+C1bPie4Pe2/XkC69lkXnx//cYsq7wz8pMe/5
+ob6/TLp7R9+Ct9ncJ5pjLyPHp429M22b48usEN9ftH+Q/KL52aWCUd2WbfM
X11g92fSlldvsS9998+nHs1J2mon33nGyXdutQ3PyPyfgvCd2bV/T7X+vmaB
/dLbBfFjBXb0HzOR6Q57xsjM/ymw0zNp7wrrzakZdR6VtLszy5ifVxnH1sei
+8NekTiRuLPP8Ht6D78nuj/sm/9svePFdHyyTOZf3xcGvkDdF3ap2CvWwZKP
KPV3A4jr2cd6/dyCFj1W5tk6dSenOX7cts3JOLbofrBXpfxGX24elXL/fsLd
OxHdy5DP+kF+j+4LqyD2QP99k/KC1z9z/x2JmtF9Yes8DufQLh9Q9YNhY1aa
51rMWfFBr5R8l8B5383mnM/O+fPpT0T3yw+SeHmF6MP3Xh/sCtGH5+9tNe+2
8Ynou7PEv9L/Enbmgr6zFvQ9KOu3hD+vkLZPq/1+QKnzfeNPGjJwatofvCDt
IcfsH9Vwz8mzhkj7Bd9fdP7ci2aZi7z9sRXF/oT7RR4dlf3sgKStfXTtwU8b
ryt1P9jOF99IW9yktdcVr7yueK7pMKD6kwOq833fCrHrybBu6S70kM9m3Xah
kp+vRJ+x6wt9/tyQP1/l9cUMEH0ZNOuji3+6ZqU9fcXxtQ89m7L1/PxY5od4
td/z+e+c1yNl+/r6hvr4bcrbDz3p5WqJI5KHiu5RX1a/+tL61fPsGxWXftD2
zuic5uO+vmkp9ZmXJfH64Xxp7SeuqbZ8yjZZN0V/F6KGk5tce5kvl3VS9Peg
zpVyb4fzxC7vD/nxbFk3XubzleZzyVeyT94+nh8P65D+fn1hJsv6QvYjQpzP
d4QvSr5T/J/5UvyfxKsh37BT4r+W4n/IY90e9z8h/9lJ4m4/3UeNupdA7Oxh
ieOje0lEr0MesUrGYUw+FtaZlQUjf39wfiAnxAW+vUMhntggGLk7X+ov9hMc
5N77pYS9wiWIlhjK8cPe7iX838WZsc/U9etdc7qsd8kvir1R940l/XmjVE44
Z1wnY+4OHjLODd1fZHf7fG8453BapQufvnfzYVN+Ye6tk18pso2nL5v92lN5
dttb89rNuTy6D6vPpJ8njUnk20WDZi0cNOu4rXb1lKpXT8m3jVP/avXaIwX2
6Q51r/+2zzFzNLMMubbArsqo22nHTF72jouXnhPdh9XI1zftpf5+X9/Mkfr5
vn64Dyvcd755duXa6Xio8kq3HrGsR4b69YhlPYI/nnTf1DGDmqf197mrx/Rc
t8BunzZv9tn3puw3Tu4Wif+L7r+ifh9f31B/pK9vqO/XAwWl7r9q0Ol419pj
o/uuGn10Sd/fjCsxX5+e0ewJ5m+XDTq/5odRfHB//ybpCC5p117p1mfmIVmf
/eDs+vemXmYZfla0nq7b8t6LN7RM2rv8/JgtMj+yHxz/+5xp/e4l8zV9+El/
vKVxoWl+/nkDGzU+ShzGvrJ8H7olyL26l8PE/65krvm33583B2R/fm3bOu/2
uuawudzZjWO20K8H7M1nLqx75sJD9sdNfv90suzXPyzxn3v89ZDtdviryq+1
yDZnjG5217LnDttbbnvqzKNrkyHOlXyP+ZPkh+oksndd0fioaZmoM6mkUWHI
E3SUeVlcoU3Xwz3TdnXc+x/cfn3Cft+3evG1jQttj1NLft+8Mecpor9vdbNv
zz4g7cX/zlWBeeqmvnWGzswxk7te2fexr4tsbbd+zDEj3T5ydP/Urfue+fHC
9PrE27X99nPJ90xrNnbc189G92pMyyxDFxfax0a0az2iXXR/1P2Xbmp66ab1
tp7XV9tb9JV8tj/3Ed0HNdW1m16/+v1H8y/Zf4z/fZRce6Xfb7SXy37jC57f
9kzh9zDZL39Q5gN5PrTqjsw/21HOByyS/ctVst6rI/Pby+93mhtlvxM7i93F
Lul7/LBTnFcQ+8e+Uqn7ob4Q/kl+M+QVyfuX711lV93hC+wglyeN7oti3c66
7Nf3P617fdvt6rs58tAhbsRP4jfvErr0PVbQ7d8/LPvi0TqxJD7usG4cJedN
esp+wFzxD9dlHvP2mg1+f8BUqu73B7zeFIv+7LI1h4ztNHh5sflj8zb7zlm1
2x6udc6JGs9G9zXJ+t7o+8o/lvX92d7+25PE/l9Xr9vOhUPn2WbDr7hvXKOU
PctjA17o7b+pKvZ/kj+vYn6U8yrsT7wkfvEayZPE76+ZK/sTa0y+xHH6O//r
ZH3POQT8AecWmkp97h/oPGHZiP0do/uTOgne5O2T7MNG9yVNE/3ZMjgTweyV
POkR2/ofd6UlaZeN7yPlyfkC7pE7GvJ9fBfKOmuwryf7YXnqHHqeWSOYuKuG
xPusQ1+X80SnXjJyb87j82WcCXuHe/5kJjVsP7lh++g+Iu5PGdzi5vWPv3VE
zhEkbWZX9rTnjpobd25q//szCuwY+1b20R+Pmo/lvMD1Ul7Xl5vxUr7ExyMh
TzRG9qsOyHkoiV9MI4lfXpP9rVpCr+dXwlQTev18F9gP3HORTUg8gJ9Gj3+T
2V7un7Sz/fkVO0DOr8z161H7W1mPFvr1qD1X1qPrOry5tsObxfa92ZVGza60
2254pnH/0xcW2029p3R/4dvddvXISi06ji99/9Bfs97bMv6DXeIHd8p+39zw
vedYoU/sR8jrbPDyYvqLvDAP/xb7zPo3Jfbtvsz2UI8T9o+Z7emXtge/ix8u
J/vZ5eS/TcL3YeL3Zwh/NorcDJN4fpjIy/Ujqwy4d81qc6xNiyXP3RL9feJl
8z+65sAN0d8nPvf9B4p71E+FvE1lyS80u67/DSsrR/fReL2cIe1v8OdPw/2c
m8zuondz28+dZGa/f+oLl53YZv4r5dxX8Za8z30wX0n+a4ZgWy9zMqPI9jm+
dMUt92yxlTPs+cse69Irvyu0FT02+wQPlXlhngaI/WAcX4tdYB2g/75nU/EP
TQT/xc+HqSfz0VR+byL1JB4PeUfs1PBYnL9YxrPbLJL5Ig+8UdYLnWU97v3x
XltJ4lhvvyZK+TbW72E/Mq9n1wG771prZvbsuWfBwynW4yFuRC42Sr9DZZ3V
ROiV/oXe0vsf7K98LXKh7z8ffGPzBWe3O26vuW1IzkM98+1Aj01twV2639y+
+80LbPkqy8tVWV5i6hw4OOjTNdH9CdyncMnQKQ/XXVdiXpd+2f+t/J8dbzap
lm0/KOz95SODUrbT1Pve+9vS1fbD+l+Pqvl6yl7R7R8b9r61yo49f871lV5I
hTw83+V6eUyoezSjfM2jv9vz0mVL88zL5dtvmPN0wr6RO+jjZ37Jscu2J++6
yxbZ/CV1Oh3/Je1/i7uubpvG//T17StSX+ff9X6Ijm/0eSl9HkqvD3S8pc8P
6vOS/wdIoex2
   "], {{{}, 
     {RGBColor[0.3689498373283029, 0.18873734406798132`, 0.6355839459205629], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmF2MVtUVhveZOXPO+TDxJ+qNrVoVBqil1WFgphcVpM6QSDQ42oEPhRng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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwllVmIjmEUx5/v+9753udFluJCsgzSiJBMlpIsNUUkZZlh1sI0Y8kwmrHv
ZI8h5cKSnZkRQxgx4gopIcaSC7tRBheWGcvv+F/863/ec56zPec8b2rB/Enz
4s65QhCA9ZFzHb1z1QnnmvgwMHQuFeQgjwnEi9DvT+hbK/gV8AP+FX1Z0rl7
IDPmXEtsX8KPgAf4XkygavhxcA65HTaribcKvEDO5PwB04H26Jr45vFdjv4z
ukK+LYM38W0s8jfQBn4dVBL/Mf7HwdeBG8j5nI/gw0E2ciP2Z/B9CgzAV2tw
kxyn47OUs0VgB3wb6Ix9mOLcLGwPgmZ89cC+El4DOsDTQSG8DmTA52KzhVjL
I+V2Cx9JeAqYjDwFzMa2Kqna08B5+CUwIiafc5OyMV0D/sZzdgJI42yTxcT/
TuSahHI8i205SKGOp9i/R/8B9LV41FPP90PYD6GWUcjDkLdHqq04odiHwG+n
HKz3p5Pqjd3BDvgdMC2mGJu9arTa7M568m2s3VFCMd/BB/BtNLregc5utxyd
fKSj6wXysM8I1JsNyA+T6tFg5D3Iu+G7wCD4KK9aivGxCF4KniLXIF+F14Ln
TjOYzfmZ4DZnbwbKLRNsTSjHpcSZBz5hfwGswHZlpLsxH2ssn0iz30CMTfCs
ULP3h2/fbda9duE++nx4FfoSeAX+94aqwXKvALeRc5Fvwfegb/aqwXK3Gf4J
/+XVi3x81GL/Ebkf8jkwkbMX+TYR3UewBt1ls8FXHbgbKob5thle62VjOtsB
q30tPn4F6sF05Cyv3ajA/hq8tdfu2J08CZWDxd4VV6xMztcGitkN2y7gL/Jn
9J/gOV530996Dk8FjXHtXAnnT3A+g3nLi2mXbSdtF22np6AfGqlXuTG9PZ28
arU3aBC8wKu3NvNLI92B9f4LZxZ6vUH29tgbcJS7PQZa4OskOXW3WQVnEsrJ
W/xIb8008Arda6/dHxdXrgtC5WI5z0nqDbDdN59t7a7sDhOawRnY50SaNXvf
DmO7EzwjtsM+G19dvXrVJa7ebcT+d6AeliOXed1FLj6zIs3E/1kItLtTQ/XO
djgvUkyLZTNls1cdardtBi23x6HuznJsgbwkUq+sB2+Q33rtpr0ptqsjvWq3
nV0XaQds9m1mNobqmfXqUVJvf49Qudo/wGrp7LUbVlMf5BWh3qr/M4uuwWs2
bKZtt/eF6oXtuP0rLoB6p3/GPw4j3vA=
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.5301949530204012, 0.46909905359403087`, 0.8622334623327463], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNm3e4ldWVxr97ztcOMjNiTJBcUC8gItiSwQTMM0aKVMFEqsAFxn6xNxAw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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwll3mwl2MUx9/7W+7zvDfrMGZooVLdVJi0KGNXohihxU1KlhYtdLUI0YIs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         "],
         VertexColors->None]}]}, 
     {RGBColor[0.6576505082106406, 0.6783235608270402, 0.9116377681315365], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmge0V9WVxu/7t3vvrKArYNRopCjSbPBwKeXxQLEB4qNETNZKU6RoRFCw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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlmeMVVUUhe+89+ade38oiZEiCkqRooCARAJILwpWhuoPoxEGpBdpioVe
BgUFBJEqRaqGKgHBgDQxFGmx0xERpUsM1W9l8WPl7nX2PvXus/Yp9VqvvJ6p
KIo2gwxYEkfRs0kULYY0x1GQjaLp4Ay+PaAD9ixwFftEThQNCtj0aUj8ENAU
/gF8ZzqKEvyfYE8DB+Dn6dONvlvBc/gmRx67P5gbeY75zN2NMXKIb5AbRVvw
dQXj8bWgT1n8D4NazFU88lrywd7IaxqGvQKcxt4CAnNPJ/5BxnqJ/RyFt4Av
on8DeGfmakbb48x3kPHX0vdnUAhfcXhz/DOJr0z/e+C9sB+h7V7GrskYediT
6f8D/TvBB+EfCN7HfxY+UWvF3wj/CXCN+D60Hc54jAHMdRvcYPz/4IXxTSJ+
L7H5xGTgdelzGd4K3hFeAvTB3gm6Yh/HP4v13gJPBvdRbB7ojP07bdPxzWSM
cVnPqbl0Zm/ge4iY/sS+Ax8BrwIfDY/psxHeCr4GPgr/L6ztV5CPrybYl7iP
Yk8y7kDsK/jrEv8uGAsvAHeBopzfAO0Ffx3NBzbRvz18HGtbpX+OvzJtI/Gd
Y77O+F5XfuG7j/7LsdeCK/BezPkN8W2JX5fxmfeHX4z9bwaD7fi2Jc7F5Zxx
aexSYF6O91yL+KfgC7UW0Aa7NUinvGfl6vCsc0k5+4XWBzowf31i/sTeQfwh
xp4LX6HzAx3xN4VPw16K/0X8M+j/E2v7EVTH1xE0wbeUmHnMNRfUC/7H+rfa
YwPsJbTNhs8Ck+HFg3P/bsa8wFgVg/9NDZ0ZvELwv6wGv8HaL4AT8P25zr3r
+I9knIPHYt8J3YUy8MX46iX+N7Vp66q1KQfpe4j1ryJ2BSircVnfxazn0Nj6
X7vwPQqu07cH/p7Y3RljJXwBfD1jFY59l96jz1Z824L7rgfldBfhnYm9Sp8n
4H/DF2IvBZexO4H9OR7zY/xbE/9b3cFvg3NEuaEcHYU9m7bT8oGviD0frAX9
6DNUuQgKsd+N+Keyl2vge3zHI/vagoK0Y36j7yvBe9OdKhJ7T9pLQcpa2BQ+
P2NNzMJHgy2aH/9A7AGx9yLNqab7BzYpl+EvYH+Ef3faGjAfXidxLitnVmDv
pm2dtJg5l8XWbGn1mbTnGhl7r5rzaeKbJc5V5fgUfK3o3wV7f9pnUy223uqM
FsBbxv73NcAGziKHtuiORnfkm5I+Z93WGnss8duln6AocxUBu9KeYxX+wWAQ
vBnjLcM3XXrF/2vF+hfSdyZ8CrEnlU/Yc4mpg292ylpRP3HuSzP+hTdOfFfq
gw9ja6a0UnMOpf+QO1pTDFTH1wSexR5HzATix8f2qUY1xrdTmsxcw0EKXxVw
Q+elfMH+EqhISoO19wNZ653O4LtgDZF2aIzbyj8wmtgN8DGxz1xnrTO6JG1k
TTMyrmGV4a8S3zPtOXXXLkljUr5zqj0Hg7VPNUhaUTFxrZVmNMSuRFuZyJr7
DPwxeH7aNbwlPC+xFkjT9gTnkHJHZ3IUflj/g/iZ+Nvja5f4LkpTd0n7WePF
tO/8WWKrwm8S+xltGelxcO3ZDNKxz1Bnp5zW2W0K1jqd4cvw3Nh3YQIxzRn/
LfwD067JWkv52Hdfa1LtLk9b38g1fEZwDil3pAGdVItivy1UI25i34qthboj
x5TriXNJNVK5tgj/qbRzrjy+ColzSWd4Bv8a1ddc5+i5YM2QVqhmnIKvhveA
z2G8XvA98FNZv4emYC+ThqasAZ8Ha7a0+nKOtbxc4tooTR8GrwQvFlzzUqo9
iWu3anBV1UJ8CeMNk96pVgXXXuVQxF5agzHsZ2PK2lk6+F9IQ5VbR4L/rXJM
sbfCndyE14ZPSlx79UbJg6+Mre0jQEl8DyTem94c0oo5wW8zacbU2Bqgu683
ns52FvyPtM94tbQi9trGwtvF1jxpnWp+NcYeE3w3CzJ++6lmqFZofGnh8zrj
tDVxYuw3l95aauubWLOkVappbfC/HfzWkmZLS7oE1y5pimrxp/B9Oa7JPYJz
Urm4ijFu6r0DHwX/Gl5Ba6fPqFznoGp7iWAtU41X7q4MrvXKYd29ksG1RHfw
RvCZ6izXEbMD3ij4X/ZOOZdOBr8NlFOaq2Ls3NKcvfHdm1ibTub4rXGatn6R
3xz5em8Fr117ktYujV0LpLnNsd8Mzj3lpN4u9wfXer1hlOt/Bb9dlPP/YHeP
fbdVc/8HjBmhEw==
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.7748195583371988, 0.8510625594298178, 0.9085705367874998], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmHmMV1cVx9/M+/3eez/AAYXZ2KyxdZuWtsIMqTN0CZBAZ6Y4gIkLMQEH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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlW1o1mUUxu89/2fPfd9fSooaG9oLaW9G0JtUTs3KSMvmiiBKog0s6YPP
ZmoFi1aSuFKzpjmX86XCWhjRlzIqStOiWcwgNadZSIiQTpvWh4Lsd3X14YJz
/te5z/1yrnP+F7dW751XCiF0gDLorYTwBNiAfRR8HkN4L4XwWxHCpwRswz+A
34D9GdiEf3MOoQd7JzjK2jZQJumv4A34mcR3sH4tfAvcOnAyeI9fsI+APcQu
B0PYB8ApuAFwGPsFsCN4zULyTWS/XnIdq/HadrCs5Byr4SaBYbiza0O4lr2X
4XeyfwTXYe8GSyvO34g9npytcK+T8xb8KeAE6yeS7y24veQ4v+w9b4W7im8b
dXfwLFwneIb11xDfRd4F4A9yfwsuIH4M+A67jfgnWXtpdu6T7PEV9g7wPes/
4LyXZe/xX24wQu5x+B1l3+lG/Lmgh/it4FXse8g5Hu4mMA17G9/OwA2Asdl3
1N2UYzH8Ga3BXgIew74r+u2U88fkmqvWdeD95Jqr1se1n+5Ovhs462H2ewm7
K/ttC749AF+X/BZz4H/CbgaL4ceQYzP2vOi1iW+7k7/JPgv+TexLwAjrT4MS
ua8Gj8M1g4dZ2w7/LvGHwCDcYLQ+lujd4S7Pfrvf2WM9/uhkcQ8Rvx/77Wht
1PPtEfw5YA1cd2GtfwF/rLDm78avx59dWMNV/Bez7/oOfhNcN9/OKbkGOttg
8l10xufgZ0T3gnpiNdyq5Fqp5lW40dnamK/7498Jv7CwZtVb6iH1jnrsDuyd
2Vr5B/5l6Rv8ib1LGsbuBXux94EJxK5QTUq+w/XZPSDtN5HvELFb4EeR70L8
Wfht8Aexnye+B38q/IP4e/Dr4O5Tz3P+uboj/qroWs+sdazeVG+pNQvwO/A/
Lnwn2VNAS+Fv47AHkmt9qmTt6o66mzSs2vWB/YVreLziGaXZNAu/nv0bQD/8
EdbPx/6b9VtL1lw/9kFpgNg+0IXfGV2Lc0E/8Svxmzj/DM4/ndifwUXELirc
+/uSe0EzQNq/n/jmGveA3upKzcSy30x7SfPSuvZcBDc5e2/NlA/htuu8tdZY
H3dprXhWaOatgV+b/NaNNT7buuRa6oyaLcOgWvaMeQr+Cs087jKieULs18m9
oR5Sb7Rkz2b1yGvY07JjVfPbsW/L7hXlmEz8l/hbaj2zZuNvSO6ddmK6o2uu
WusNHoruCfWCajSU3GPqLf0jHk2eWZpV6jG9rTQnremNtddH+MOF95SWlyfP
Lmlas+mH6H+DZtQ3cGOTtaKa/xWtWWlVNV+pWZ2tTfXkdPjz+Dbq/xxV3TX5
36GaqTdXJPeOenR79MzTrNOZXonWiLShHpqKfTp5dusfsBS7K7lW6gnN1pDd
W5qx+rc0gAkl/2M2wW9Mfkv1aCNckT3bdrF0M/7T0drTmqbkma1ZrX+MtDMp
OlYaOoEWPql4fkgT/wJa4RND
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.8812877234392763, 0.8865668555466382, 0.8608728942313656], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFlktsV1UQxu/tvf3f+49UEDQC8rBUl7YVjQtjqYkuXEBEEqG2WKUkNoZS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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlEtIlVEUhc/1/tf//KRZWqQ2qKiGlUg1i4KCgrQXBIX20miS3kQaRJA1
CKI0e0JQqZkUNK8G1TRfE80oKAspaBQNovesb7UCF+511j5n73P2+u+SlmO7
iiUhhK0gAa/SEBrAolwIM2AgC2EnfAVxfSGEn8TtrA2T/JCNV+CXwGV4NXwc
bSpamyb/GtpuUAOvBS+Jt4ByzlsMNpJ7EjzKh9AGdqD1c0ZdwTU3oK0Hn4nr
2D8LrQzso14zWEPcqx6Ie8FceCVoJW4BV9G62P807x56Ut9Jd1nFmXvh11P3
dgNsJ3cb4C+k8HkEm9HLyK3g/xDoYX+i/ejlqkXO1+Ce7hHfAe+o9wEU0X+z
5zG599UDvBq0kdsOKslthY8Sr6PGQuJx1YcXwSTxBOhC+8j+KuJN0W/1Hf15
dE3VSuD1xOc5o6TgHruJL4COnNc60Gsz1z7C2nL4KChNXLMCbTY4QHwQ5IhD
5rcoBz+4y1jm2b4Fzal7UO3pvN/6FHiS95trtr9S310zllf+zSyxZ06jLwCf
OPsZ+lTqGWp2epNG+OvUbyNPHpd/ot9enphM7Ql5QZ7ULOZk7l0zGYYvBTOc
XWBtdeY76S498H60gejev7C2h/OOwm/n3dOf1J6VVzXD0cxvoLu/KfjtloE0
8RvK24fh8/97fCh6ppqlerils6O/lc6ca8mD8p5qriW3Bq5vUp5uUa/RXh2B
V2X2gGYvjzeR2xfdu2Ysr6pH9SbPni0N4Ru4mPjM/eh3yX+ft6ceoHWCPrQx
MEF8DjSDppy/hTPRs9E3odkWU3tRMx6E34y+izwvb1dr5ok9PkjuIdZOUHwl
GIv2oLynN2sgbozW1J96G4melXqUF1+kvrs8qd+G7tS/TfqN+Ateo48R
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.941176, 0.906538, 0.834043], EdgeForm[None], 
      GraphicsGroupBox[{
        PolygonBox[{{1420, 1421, 693}, {654, 655, 1}, {1868, 2005, 655}, {
         2322, 2321, 639}, {1864, 2003, 649}, {732, 656, 199}, {2403, 2066, 
         732}, {638, 639, 199}, {1863, 2068, 1814}, {2316, 2315, 636}, {726, 
         637, 154}, {735, 1420, 693}, {2462, 1872, 532}, {2258, 2506, 1041}, {
         2413, 2038, 1241}, {637, 635, 154}, {1421, 1043, 693}, {637, 918, 
         635}, {2039, 2170, 693}, {2040, 2414, 1242}, {639, 732, 199}, {2061, 
         2402, 726}, {533, 726, 154}, {1043, 1242, 693}, {2067, 2323, 735}, {
         1420, 735, 638}, {735, 639, 638}, {534, 533, 302}, {2001, 2318, 
         637}, {1040, 1769, 636}, {656, 921, 654}, {2147, 2317, 918}, {1769, 
         635, 636}, {1873, 2463, 533}, {2463, 1932, 533}, {2318, 2319, 637}, {
         1241, 1040, 636}, {918, 636, 635}, {2505, 2037, 1814}, {656, 654, 
         199}, {859, 532, 1}, {2148, 1864, 649}, {655, 859, 1}, {532, 534, 
         302}, {2320, 2147, 918}, {2321, 2002, 639}, {532, 302, 1}, {2006, 
         1869, 656}, {2116, 2067, 735}, {2504, 1863, 1814}, {2170, 2115, 
         693}, {2003, 2041, 649}, {921, 655, 654}, {2315, 2000, 636}, {302, 
         533, 154}, {2037, 2504, 1814}, {1931, 2462, 532}},
         VertexColors->None], 
        PolygonBox[{{2261, 2260, 1043, 1044}, {1872, 1933, 534, 532}, {1933, 
         1873, 533, 534}, {2000, 2413, 1241, 636}, {2066, 2006, 656, 732}, {
         2002, 2403, 732, 639}, {2506, 2505, 1814, 1041}, {2115, 2116, 735, 
         693}, {2259, 2258, 1041, 1042}, {2317, 2316, 636, 918}, {1869, 2153, 
         921, 656}, {2038, 2257, 1040, 1241}, {2005, 2108, 859, 655}, {2319, 
         2320, 918, 637}, {2414, 2039, 693, 1242}, {2257, 2259, 1042, 1040}, {
         2323, 2322, 639, 735}, {2232, 2261, 1044, 1013}, {2402, 2001, 637, 
         726}, {2260, 2040, 1242, 1043}, {2041, 2232, 1013, 649}, {2153, 1868,
          655, 921}, {2108, 1931, 532, 859}, {1932, 2061, 726, 533}},
         VertexColors->None]}]}}, {{}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2068, 1863, 2504, 2037, 2505, 2506, 2258, 2259, 2257, 2038, 
         2413, 2000, 2315, 2316, 2317, 2147, 2320, 2319, 2318, 2001, 2402, 
         2061, 1932, 2463, 1873, 1933, 1872, 2462, 1931, 2108, 2005, 1868, 
         2153, 1869, 2006, 2066, 2403, 2002, 2321, 2322, 2323, 2067, 2116, 
         2115, 2170, 2039, 2414, 2040, 2260, 2261, 2232, 2041, 2003, 1864, 
         2148}],
        "6"],
       Annotation[#, 6, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2511, 1999, 2401, 1953, 2362, 1954, 2125, 2124, 1817, 2358, 
         1934, 2357, 1935, 1937, 2473, 1936, 2204, 2203, 2202, 2209, 2208, 
         2213, 2245, 1884, 2207, 2206, 2076, 2205, 1883, 2159, 2442, 2443, 
         2441, 2098, 2444, 1955, 2120, 1908, 2172, 2111, 2112, 897, 2072, 
         2328, 2329, 2330, 2160, 2249, 2084, 2248, 2085, 2250, 2331, 2211, 
         2083, 2246, 2247, 2210, 1886, 2161, 2255, 2256, 2214, 1902, 2173, 
         2174, 1938, 2464, 1877, 1939, 2474, 1940, 1881, 2155, 1880, 2071, 
         2175, 2176, 2126, 2127, 1923, 2303, 2304, 2305, 2356, 2355, 2512}],
        "4"],
       Annotation[#, 4, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1930, 1929, 1920, 1921, 1866, 2190, 2191, 2188, 2189, 1861, 
         2241, 1862, 2501, 2502, 2500, 2197, 2196, 1874, 2194, 1870, 2193, 
         1871, 2195, 2325, 2192, 2326, 2327, 1875, 2198, 2184, 2185, 1876, 
         2199, 2201, 2200, 1857, 2144, 2149, 2150, 2104, 2152, 2151, 1867, 
         2004, 2461, 1865, 2069, 2490, 2070}],
        "2"],
       Annotation[#, 2, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{2510, 1903, 2332, 2334, 2333, 2404, 1854, 2499, 2231, 2224, 
         1986, 2223, 1985, 2267, 2266, 2265, 1959, 2453, 2365, 2290, 2341, 
         2340, 2364, 1835, 1818, 2452, 1958, 1913, 2121, 2271, 2270, 1964, 
         2454, 2373, 2294, 1912, 2345, 1963, 2372, 2371, 2293, 2344, 2343, 
         2370, 1836, 2369, 2292, 1911, 2342, 1962, 2368, 2367, 2291, 1910, 
         2366, 1961, 2268, 2269, 2129, 1957, 2418, 2043, 2417, 2042, 2128, 
         2416, 1834, 2415, 2289, 1909, 2363, 1956, 2262, 2263, 2264, 1983, 
         2221, 1984, 2222, 2230, 2498, 1853, 2308, 1852, 2307, 2306, 2350, 
         1924, 2513}],
        "2"],
       Annotation[#, 2, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2335, 1904, 2324, 1859, 1993, 2479, 1992, 1849, 2375, 2376, 
         2374, 1965, 1837, 1819, 2273, 2272, 2378, 2377, 1914, 1970, 2477, 
         1969, 1838, 1821, 2456, 1971, 1915, 2219, 2220, 2218, 2380, 2379, 
         1839, 1822, 2177, 2107, 2134, 2133, 1916, 2450, 2105, 2449, 2106, 
         2451, 2099, 2496, 2136, 2460, 2135, 1841, 1824, 2457, 2051, 2050, 
         2094, 2494, 2093, 1888, 2015, 2483, 2016, 2014, 2087, 2492, 2086, 
         1843, 1826, 2458, 2052, 1975, 2410, 2018, 2408, 2017, 2409, 2412, 
         2033, 2466, 1897, 2157, 1898, 2467, 2034, 2233, 2212, 477, 2215, 477,
          2181, 2183, 2054, 2488, 2053, 2141, 1901, 2471, 2036, 1893, 1946, 
         2475, 1945, 1889, 2020, 2484, 2021, 2019, 2089, 2493, 2088, 1846, 
         1831, 2459, 2055, 1978, 2031, 2486, 2030, 1891, 2337, 2338, 2336, 
         1907, 1847, 1832, 2286, 2285, 2386, 2385, 1919, 2388, 2389, 2387, 
         1981, 1848, 1833, 2288, 2287, 2302, 2301, 1851, 1997, 2481, 1998, 
         1860, 2354, 1928, 2353}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{2348, 1922, 2349, 1856, 1989, 1988, 1215, 2391, 2392, 2390, 
         1982, 2143, 2423, 2424, 2422, 1261, 1980, 1237, 2411, 1280, 2090, 
         1145, 2339, 1146, 1239, 2032, 1892, 931, 767, 895, 2023, 2503, 2024, 
         2022, 2026, 2485, 2025, 1890, 1947, 2476, 1948, 1894, 2035, 2470, 
         1900, 2158, 1899, 2468, 2469, 1952, 2097, 1951, 2182}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{2351, 1925, 2352, 1858, 1991, 2478, 1990, 1010, 2228, 2497, 
         2229, 2508, 2225, 2227, 2509, 2226, 2130, 2044, 2419, 1250, 1960, 
         1254, 2420, 1253, 2131, 2045, 2421, 1257, 1968, 2047, 2507, 2046, 
         2132, 2048, 2487, 2049, 1972, 1199, 2383, 1198, 1227, 882, 945, 2092,
          2251, 2091, 2252, 1887, 2012, 2406, 2013, 2407, 2011, 666, 894, 
         2095, 2495, 2096, 2027, 2156, 2029, 2171, 2028, 2102, 2117, 2166, 
         2167, 2103, 2118, 1895, 2254, 1896, 1476, 2253, 2217, 1950, 2216, 
         1949, 2215}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1850, 2446, 2447, 2445, 2101, 2448, 2100, 2146, 2145, 1966, 
         1967, 2455, 1820, 2009, 2465, 1879, 1906, 2472, 1905, 1943, 1878, 
         2361, 2360, 1942, 2359, 1941, 1944, 1885, 2079, 2080, 2078, 2439, 
         2077, 2440, 1882, 2162, 2056, 2425, 2057, 2109, 1823, 2243, 2081, 
         2242, 2082, 2244, 2058, 2426, 2427, 2295, 2296, 1842, 1825, 2154, 
         1827, 2110, 2169, 2168, 2010, 2139, 2138, 2140, 2123, 2180, 2179, 
         2178, 1977, 2393, 2394, 2297, 2298, 1845, 1830, 2238, 2062, 2237, 
         2063, 2239, 2059, 2163, 2060, 2234, 2236, 2235, 2240, 2430, 2431, 
         2311, 2310, 2309, 1926}],
        RowBox[{"-", "2"}]],
       Annotation[#, -2, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{1927, 2312, 2313, 2314, 2433, 2432, 1855, 2186, 2187, 2429, 
         2428, 2299, 2300, 2164, 2165, 2065, 2489, 2064, 2119, 2007, 2482, 
         2008, 1979, 2282, 2283, 2284, 2142, 1829, 1844, 2114, 1976, 2280, 
         2281, 1828, 2122, 2137, 2435, 2436, 2434, 2438, 2437, 2073, 2075, 
         2491, 2074, 2384, 1974, 2347, 1918, 2346, 1917, 2381, 2382, 2277, 
         2278, 2279, 1840, 2113, 1973, 2274, 2275, 2276, 1987, 2395, 1995, 
         2480, 1996, 1994, 2399, 2400, 2396, 2398, 2397, 2405}],
        RowBox[{"-", "2"}]],
       Annotation[#, -2, "Tooltip"]& ]}, {}, {}, {}}, {InsetBox[
      RowBox[{"-", "2"}], 2514], InsetBox[
      RowBox[{"-", "2"}], 2515], InsetBox["0", 2516], InsetBox["0", 2517], 
     InsetBox["0", 2518], InsetBox["2", 2519], InsetBox["2", 2520], 
     InsetBox["4", 2521], InsetBox["6", 2522]}}],
  AspectRatio->1,
  Epilog->{
    RGBColor[1, 0, 0], 
    PointSize[0.02], 
    InsetBox[
     FormBox["\"global maximum\"", TraditionalForm], {0, -1.8}], 
    PointBox[{0, -2}], 
    InsetBox[
     FormBox["\"local maximum\"", TraditionalForm], {0, 1.8}], 
    PointBox[{0, 2}]},
  Frame->True,
  PlotRange->{{-2, 2}, {-2, 2}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 183},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->286241299]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Maximize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Maximize"]], "InlineFormula"],
 " finds a global minimum and can work in infinite precision:"
}], "ExampleText",
 CellID->1543511006],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Maximize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"x", "^", "2"}], "-", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{"y", "-", 
         RowBox[{"1", "/", "2"}]}], ")"}], "^", "2"}]}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "^", "2"}], "+", 
       RowBox[{"y", "^", "2"}]}], "<=", "4"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->1936029081],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["31", "8"], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"3", " ", 
         SqrtBox["7"]}], "4"]}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      FractionBox["1", "4"]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{176, 35},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->147824252]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1659867246],

Cell[TextData[{
 "For nonlinear functions, ",
 Cell[BoxData[
  ButtonBox["NMaximize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NMaximize"]], "InlineFormula"],
 " may sometimes find only a local maximum:"
}], "ExampleText",
 CellID->1669385690],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMaximize", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"x", " ", 
     RowBox[{"Sin", "[", "x", "]"}]}], "-", 
    RowBox[{"3", " ", 
     RowBox[{"Cos", "[", "x", "]"}]}], "-", 
    RowBox[{"x", "/", "1000"}]}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1982487473],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"3.9108501677923626`", ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", "2.5702528588727858`"}], "}"}]}], "}"}]], "Output",\

 ImageSize->{165, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->10018934]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"x", " ", 
     RowBox[{"Sin", "[", "x", "]"}]}], "-", 
    RowBox[{"3", " ", 
     RowBox[{"Cos", "[", "x", "]"}]}], "-", 
    RowBox[{"x", "/", "1000"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "20"}], ",", "20"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1912208167],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 108},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->48544354]
}, Open  ]],

Cell["\<\
Specifying a starting interval can help in achieving a better local maximum:\
\>", "ExampleText",
 CellID->657148640],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMaximize", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"x", " ", 
     RowBox[{"Sin", "[", "x", "]"}]}], "-", 
    RowBox[{"3", " ", 
     RowBox[{"Cos", "[", "x", "]"}]}], "-", 
    RowBox[{"x", "/", "1000"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "100"}], ",", "100"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->723115657],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"26.97352558860916`", ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", "26.851381348909204`"}], "}"}]}], "}"}]], "Output",\

 ImageSize->{165, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->181556204]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->993018304],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NMaximize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NMaximize"]], "InlineFormula"],
 " finds a local maximum of a 2-dimensional function on a disk:"
}], "ExampleText",
 CellID->220009434],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMaximize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"-", 
       RowBox[{"Cos", "[", "x", "]"}]}], "+", 
      RowBox[{"Exp", "[", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"x", "-", "0.5"}], ")"}], " ", "y"}], "]"}]}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "^", "2"}], "+", 
       RowBox[{"y", "^", "2"}]}], "<", "1"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->33483118],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.5734522220470394`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "0.9270953353893807`"}], ",", 
     RowBox[{"y", "\[Rule]", "0.37482561490660843`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{269, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->15713487]
}, Open  ]],

Cell["\<\
Specifying a starting interval helps in achieving the global maximum:\
\>", "ExampleText",
 CellID->1122017447],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMaximize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"-", 
       RowBox[{"Cos", "[", "x", "]"}]}], "+", 
      RowBox[{"Exp", "[", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"x", "-", "0.5"}], ")"}], " ", "y"}], "]"}]}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "^", "2"}], "+", 
       RowBox[{"y", "^", "2"}]}], "<", "1"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"x", ",", 
       RowBox[{"-", "1"}], ",", "0"}], "}"}], ",", "y"}], "}"}]}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->559037039],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.6004629972522912`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "0.657218786081364`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "0.7536998524938943`"}]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{263, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->26332952]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ContourPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"-", 
     RowBox[{"Cos", "[", "x", "]"}]}], "+", 
    RowBox[{"Exp", "[", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"x", "-", "0.5"}], ")"}], "y"}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
   RowBox[{"RegionFunction", "->", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"#1", "^", "2"}], "+", 
        RowBox[{"#2", "^", "2"}]}], "<=", "1"}], "&"}], ")"}]}], ",", 
   RowBox[{"Epilog", "\[Rule]", 
    RowBox[{"(", 
     RowBox[{"{", 
      RowBox[{"Red", ",", 
       RowBox[{"Text", "[", 
        RowBox[{"\"\<local maximum\>\"", ",", 
         RowBox[{"{", 
          RowBox[{".927", ",", ".374"}], "}"}]}], "]"}], ",", 
       RowBox[{"Text", "[", 
        RowBox[{"\"\<global maximum\>\"", ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"-", "0.657"}], ",", 
           RowBox[{"-", "0.753"}]}], "}"}]}], "]"}]}], "}"}], ")"}]}], ",", 
   RowBox[{"PlotRange", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1.6"}], ",", "1.6"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1.6"}], ",", "1.6"}], "}"}]}], "}"}]}]}], 
  "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->186971854],

Cell[BoxData[
 GraphicsBox[GraphicsComplexBox[CompressedData["
1:eJyFfQVYVc/zPmJ3d4vdjYq6Kja2Yndhd3e3qKiY2GLxUZRUxEFQWqQbiQvc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   "], {{{}, 
     {RGBColor[0.34087078348212874`, 0.1399154525992207, 0.5961154389321128], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNVk1slFUUfd83X+f7aZu4EQr1J1MNYCjdkLgBKiZGtEVBtAtZMSWFlBBY
saAWdp3WTaEkJLpw5Y5ExcREdAjQUEoiCQktwpKGaVEqhrgh4WfBOZwzkcXN
e/O+++7PuefeN5XBw7sOxSGEOUgJsikKoYBcxP41rA+x/gW5mIZwCzKC/VhL
CLUW7euQuzzz+RDkTtD9e5CTkAuQJcgJyBHY2AHZDtvb8hAuYV9FABl+N/B9
EtKJ/epI91dCZz4LoR3BxZF80H4L9gv2W4GN3ani+C6VD9o/x7zK0pnH+gh+
vsI+tIZwHTa/ga3FsnR4HuHe5aCcmceYc2zmwPgfQOdoplhG8fs+1mNYN2Dt
hpzGfgI645CbSQhTxoI2GFN/pBjbkdcKyCmcrynrnPaJcZ7LL3M+WBL+/M77
k8aV+DCmQ4nq9KrrxvpdcjybXqrjZCydMyXtafPjIoSPIL/AxjOsV+H7Kr49
wX4G+yuxMNpnHIa8J45c93u/5BxOmD8bHEOv+UNMqXsYshL+l2H/VqZ9F6QC
aYVsgf7mSPoPoPNvoTuPsf4A/ZlYeE5Avjam4/7N+neZAwNY30rFJ9ZmGTaP
Y/0DuF6DvJLI94qS4r+B2Btl1ZPr39H/+0Vz43PEsAtyHnc/xfoJ5Fwi7lCH
NtYjxg8L3Wft2kqqL33QF+3wW8M2NyLGb1PVsc0YUH/R8dDmcKxz5s0eedt5
rS0Lc2LfYV+s6Sl8X1fW9y6fTwfFujcWX7vNiylz6YKx/AJ330jVVx9gfT8V
z4gVMWMOncDvv1RcYiwDjoe8emJuMV/2G/vvLPR/zvWNsbQ7zu9TcZd22J/M
tRbEm55UXOZsqXkmvI6zL1P1Pvk6GAtL4rcqEg6sDTnCerMu88Y5wXmcCMu+
SJwhV0OqPFf7bNR8uWEu0Dex513W5c1UGBEf9jZnAOMZyTTLqs57zrlXcXcW
Z0OxOErb5CFz6/R8ewq+/JRJr8e9c9r9xBpyVoya54wnsU/av2LbtLkTdj6D
1BP54jl75MdM/Uxd4rTVnDnPumTCphXfi0I1Yf8P++7ZzHMhFndoe9k93eE6
cuVvcmw2VT7MhfX5NRHfOFdac9nZ7xlAjvyGs3oufPeUFONx90KHeUt/Mz6/
nGoW8Hc/1u2FfDDXWcdMW6wZc1gwTmPG/J5nMTmzZN4Pe6bwLvld9RvRrBn9
8o3aVuhbxTNr2tyouDfZUxWfb3aPTTuXLvf2Mc8q1pJz/YwxTCPNrbr7ep3r
PuT5yhy2mg+M+45ncfOcPc9v+/z+jtjHSfO8z+9GzfOHtRl1LFPWY51YY8b4
LmoykSnGwZLqRnxqfhNjz+lez2rG3uI+fui5z9x/h502v3EN58kcWT/W+0Wt
c72F1CF+xG3K/fWne2zBvGe+fAfXGJ+GMWvmVnet30PsB1Cv24ly73f+48aC
uqvg851c84drp/fsvx6/X93+Tc5wrvEdbb7FfX6viVNHLqx6jcu0e5P9Tx51
+5x9/aI3SsKTZ1t8/k+h94lzNuN/Lef/tKy5z/jn/CYwx6rnH7lKvdvGatAz
nvXd61nImj4Hpj5FHw==
         "],
         VertexColors->None], 
        PolygonBox[{{1837, 1825, 226, 263}, {1986, 2640, 1211, 611}, {2665, 
         2111, 1239, 1237}, {2507, 2471, 790, 1040}, {1857, 1843, 190, 282}, {
         2171, 2222, 709, 864}, {2636, 2634, 489, 1207}, {2641, 2639, 490, 
         1211}, {2326, 2327, 781, 146}, {2642, 1987, 708, 1212}, {2262, 2795, 
         1040, 791}, {2797, 2794, 791, 1391}, {2314, 1990, 710, 865}, {2790, 
         2255, 1383, 1382}, {1985, 2635, 1207, 610}, {2808, 1983, 122, 
         1414}, {1838, 1858, 283, 266}, {1926, 1859, 284, 1134}, {2225, 2489, 
         1026, 1028}, {2313, 2816, 1432, 122}, {1822, 1836, 260, 224}, {2631, 
         2091, 488, 609}, {2173, 2224, 711, 866}, {1871, 1787, 142, 332}, {
         1827, 1837, 263, 227}, {2481, 2330, 654, 1018}, {1859, 1793, 239, 
         284}, {1881, 1928, 1236, 223}, {2174, 2492, 1028, 617}, {1988, 2314, 
         865, 709}, {1927, 1800, 1237, 1236}, {2254, 2790, 1382, 1432}, {2639,
          2643, 1212, 490}, {2221, 2171, 864, 708}, {2259, 2358, 876, 781}, {
         2315, 2174, 617, 711}, {2634, 2638, 1208, 489}, {2169, 2808, 1414, 
         258}, {2328, 2390, 938, 876}, {2666, 2028, 258, 1239}, {2110, 2665, 
         1237, 1238}, {2033, 2571, 1955, 277}, {1858, 1792, 282, 283}, {2491, 
         2490, 146, 1026}, {1804, 1823, 224, 261}, {2223, 2173, 866, 710}, {
         2632, 1985, 610, 1204}, {2637, 1986, 611, 1208}, {2390, 2480, 1018, 
         938}, {1836, 1821, 223, 260}, {2255, 2796, 1391, 1383}, {2330, 2329, 
         277, 654}, {1826, 1804, 261, 226}, {2471, 2170, 609, 790}, {2630, 
         2633, 1204, 488}},
         VertexColors->None]}]}, 
     {RGBColor[0.4400435369881514, 0.3123500878563094, 0.7355147412112962], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmG1snlUZx+/nft7u5y5fNNEx2mGMss3gGO7NV8Bo4lrp2tElsxJlunWj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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwllFuIFVQUhvc5Z86cvfcomZDjHaTCRCQIstLURNAGdZwUZIIiURObnIvR
WI5l1IsXQo2khwJvYOroeBm7jorCSIWCoAOjD1KGZZEa+pSmWN/P/7Bgrb1u
e631rzVmcev8lmIIYTlUBQ3KIWyNITxVDqGdh+8qIVSnEF4thXAKw2fRP4b+
50II85DPw1+APoPvgGZh24F8C/svoOnYP4F8gFid0HPI3xCzAdtdyHeRe5Hf
IvcGco6N9pHtfugW8W5Dy7C/Ts4z8NfRt2D7h/6A3AA1ob+JfAj/e+g3oV/O
2374dt5GIY+EjmB7GFqPrpn/9ZP7JLQI+fOSfX9A/iA4RkaugVr5yxZsTlVc
o2prxeYisT+Krn00b+fgQ00Ir8H/w9tq5Bv4bC46x3v43kEeBn8Bm0fR/81b
HfG3QW/APxldi2qeRu5j2DfC70V/APlx9L8WXFMN9j+i30jaJcT7D347+nex
b8Pm4eyZaparoPXoJuKzDtt12KyQnJ37kbJj3Uf+pOyYc+HrNTP0K5EjvpVk
vPTin5J7qN6pZmHleMU6YaaN+J3R2HkF+UVi1UHdxGsjxnz4BVBPlTHwG/7f
4j8R+WnoU3y/Tu7tVmIMgO+BSuiW8vYSvg3Zf1OMqfAvZPdKPSpiuxbqw7YR
moNudnZu1XSE+MeTe/cT8Y/CT2B+m6qNqSb+0gYNha8reFad+A8ve2a30Y3g
raVkjFyFv4h+HvqTRc+6lhwnCp75GvSroVrtDj5d2B7Mxrp6+gz8pOzd0I5N
gZ8M/QK/m7cPidUtDJQ942vJOZVLPetG/zJyF7HLyA+Rfwr0J7FPQ5f0l2Sd
dnYmscbzdrTKM9kG/6Bi7KwUfujFO9G7d46cffC1+L9ZMoa0y89n/0073Ye8
JxtLC8l3hz7Oqbh3zfoDvmeJsQT9AHxuYnsje1eaeevPxoywoh7KdnA2duVz
NvkGaPd1Y/rR/5Xtq5twmlz/In9c9g5r9h3JvRYGjsF/nzxrYWhndM2qdR9v
q6JrVq3t+I/XfKHL8AVsvsT392Rs1le7173JtajnXdE3TbdsKfGuRPdEvahH
fjv6JukWaYYDkzEtLL9eMrZao2+rMCYsfpWMfWFySLKPbDUDYen95FsiTO1T
L6JzLeRtaPLOa9d187Tbh6Nvo3Z8XI1rUi1zq927g9F/Vw9nZd883Tph5AT8
juheCdMzsnumXunGjyPWsOTbuqbk3dCbeO2IbllI/ptuWhP+0/AvFn0j/wd6
2tsK
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.5392162904941741, 0.48478472311339815`, 0.8749140434904796], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmguUlVUVx7+5936vSykDpkAWIGpkPgIUSlQGTVAwdYUoiBqJQVhpZmEE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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwllnuQj2UUx5/d3++3v/d9NwZhmSG0Qm5bkS0pFBktMXLJdYQGXZikyGXE
oBn90Uw01IxcQyVkmWTcZpaualDTUtFulKbYXZeYzGT7fPv+ceY95zmX5zzn
Oc/3vK0mTB8yLTeEsB1KQ29GIeyKQ/g3FcIDLDREvpSEMCsTwtycEAryQ1jL
2nhsK1mrhzwV+WHkr5F3Z0O4gv0c+P6svQz/CnSeWM3ZaBCxi/BZnxfCBvYY
gzwWWgX/BvE/h/8MWgZfB59V+CaRdRExzyIfRn4KfnzGe9/F2qq0c9gC/zxr
TdAVQMvh97J2N/zNlPcuYu1QyjnUkm8L9puBvIX8KpGXoX+X7xbWuuB7L7Sa
+D+RwzX4fehGs9dC7Y/vEeRFwTFb4VuEzcq0cypFPzj2WVXTe5QLNDjYZjj8
PHxaEesQciXyEeQJyGOhUfiOhlakXJMKdNOgpui6sVYN3xn9AvhpnGcPuVwl
xvyMc9yH/lfkccgnOd9hbMugL+EvQPclvjPd1Tr274TcGeoH/yhrNdhWQyXI
A6ApxKtCP5N4p8jnIfg7WNuMbhN0U70DTYSflHZtLsX2VY1KkSuQOyJ3Q66P
f4PEuayEvke/FypGX8yZfkBXDnVFHoL+C+UeuzcuI+9k7/3I7VM+UzG2rSOf
ZS10Ev4ElE7Z5iv4a8TvTe51sf8U/m98FmR8JxeJ9R7yrRnXoAL+QOS7GANN
hF/JWha+CPs+yLWxz1qOXBa5R9WbhA5d4RuxdiDlHupI/3XId+63cF9H8d0d
uzfro/8Y+1exb41/IdQc2x0EKiTWdeSh2L6F/m38N+PzAnxl5FpVcKYW8Bew
6UusFbov4t0euXe/Qz4O3xX9kpRrUh45B+1diNyM/bayXxP2u5pxrHXEqJdx
zPvhuyeu7WPYtMx3D6r34jz3xk78uwf3iPZaFDm29lRvfhD5LtWjjbFNQ/9g
fwyai/0cqFQy9n2xrcseZRnf2Tv4Py28wX8o+l+Q9yOfyvUb2gR/jLX+8CXQ
M5ExSthUjn81ObbRGxFeQfvhr2O/OGNMq0V/AzoHXwWdQHc88V0q5kj4JxP3
bhk1GEiuj+uNsv9Z4pfEvjPdlTBNvTIpMnapZ/SWOqpn035TwiZhrrBWGKVa
V0MvZVxzYel09KdzjKnqVd257lo92we5PfIO/LdBW+WLPCLXb1JYIUwUFgoz
IvbfytpI9OOEb7JNfJYj0Ab49YlrpTfaVrVGHoX9OdbW6O4T95bueASFaQBh
Fsaib6f3RP1OIN8Irm0BysE5rvG6yDNEs2MzMZti/wQ0Av2d2Bei74bPwFyf
eTi6XtCP6EYjP4iuR+Le20gOzwl7EmOhfNRLBdDx4J7S7NqVNZZphnWH76n+
Jlax9oxcQ9VuO3SJ3NuqJ7D/NhjLekR+O8K019ljI3FysVmtnoj9ZvRWhNFX
81wDnV05qNdqWPsmuOcWEOtDfCZjOyPXe13Jc+9rzxRyo6z3vgY9Qm7tItd+
TK6xpANry9PGlBr4n4nXL9c90wn9UGEW+oNQF3KbjLwI/YvE/Ssyhgg7ekPn
4H8TxsMPCH5rwrT/sYx4qdiYKawcic1RzSrivUbsGta6oK/KOtZiqBZ9y8iz
oQT9+9gGbIYFz+RO8PmRfTWzGmZ9Zp1VNdIsvpl1b2gmN4s9kzWLp6d89sb4
70m5BlPQt4n89obh8wfyDmxaZPwm1bvboNsy7uG+8LOVY8Zv5nzsGPLVzPkd
/pOsa6GZ0xP7Xolnm86gs7eElqZdg3axZ6xmqzBocNp9KvzKI85MdDuxb5Px
m5yAfRT5X0pnuIzu2ci10sw5jf5MbGzTnU7N+t50f3ojHwmLYr9dnWl74h5W
7+rMdfRvFLu3SnOMNbMi944wZyn2SxLrdIb6sTFTWKmeVO90jvwvoh4SVuif
R/86wowa3X++Z5dmUlXkGSDs18yYR7z5sbFbb/5g5Bmq2al/EP3b/An1Svkf
Zxa6QZH/DXXGM8LvrGerMH5h5JmoWaga6t+veexe0D+g7nJY4l7XnWp2ron8
L6cZOjsxxgnb9A96MeueVa8qh/8A2Plp2g==
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.6186457294702818, 0.6156537345864066, 0.9107485793126169], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmXu0VdV1xve557XPuSrEilV8gQgRFHyhBUYard7rq6aKYmKwBmOb+kDB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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwll3uQjnUUx59333c9v+dZ2ghdlOyWXJd2Q6WLGjHp7hrrMigUhSlT0U2h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         "],
         VertexColors->None]}]}, 
     {RGBColor[0.6915606510679697, 0.7328077248011576, 0.9124108149440162], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJw1mHuMVdUVxu/cO/eec+5FKCBoEWR4tQWqA/JowjBQJE2DiQaxI1Zbirbi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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlllsVVUUhve9517uOecaFIoMKlRAYhyxIiRaKhY1BhKNLYIIGhUFHoxg
AqIgFigPRDtAKKBYKA7wAg4JIUySSIsWDApEYwS08KKlIAgyxAmN35/fhz/5
117DPnuttdc+A6fMrJ6RDSGsADnQkobQDO5nsXs+hH/hvyQh3IdyJXIf5Gwc
wiuZEOaxth5eKIbwEbrtrMXwq2Lzrax1FkIow782CuEH4s/Dvwr9KXxPggnw
bvhsxHYHPv3QlyAfQq5E/wC+D4K38C/nmw5h3wUqIsd8Qd+qGOjKkbdiuw3U
wwkRzrH/ROQPkXsR/x743aAuss1AeB7/Rew1G3kT9heRx8CbQEnqM+ksg9jj
SuzfR1+JPA39SfhJfE6z12fEr0O/FBxDdxv20+DTwafIt2JzAdvtyJmcv+kT
5D+I8XDkM06Hr2HPOVnXIMfeS1gbRuz32PM5fJ8HLdiWEq8ZXgmGYftzZD46
8bdr7QpiFYmxC98eyKeJ9TL6r9CPBidir4krp5vR3Yj9kG4hrOPbfkPXN/bZ
mrLO3ePIxyPnsDF1jVSby5ypM3bNVesL+Bwr+Az69o2srcf2WWx+xfcM2A1v
Yy3Ft4MY1xHvWvBQ3jXRt5dgszPjM8zEdgaYS6x2/AegW5u6V3chD8F+MDgA
780ZRqB/MbHtPrAGeQ/25dhXgH3wval7f1TetW1PzVXjDfh2sHYw42+uhzeA
xXxrD9Yy8FrszxNgTsa1WgqWZ12znkXfId0d9bRyf5yc7I9cg8nYvon+EfgI
fCYiL0EeizwU+Qjy0di8jXiPxr5DujvKwTri/cja7dh3gcfgNal7dSBoRD7P
fnfk/U1TkJvQR8h5MAv/CtYy6CaABNuFoIr424jfAc+o5pHP/B3844J7rz82
jcgbwYXIPX9Y9py5mdirsZnEXpNTx9Md2cBepchbiLUZvAGfxNrwrM+8hljv
Jj7LXayNQZdHng/vmfVZ30nc6zrz1MQ9oNrrji0i3sLUvXYu51kzDlyf9cwp
g3+RuvZnyed4/APy1JxzMFb3G5yKfMd2pL5DujvKYd/UM0qzSTPwJ+L9g9yU
dw8dgPdOPRtrwMHUZ9ZZ1dN1yPWpc3cA/9/p0bnkrFX1Bd1izyTNosXqKXhV
4m/9mrXx6BekPpt6rn9qH9nWgovYXgLVOef8Mrr52JyBV0fu5VWpa6+efgJ5
P/sPjVyDm+D3gj7oLgfHeinx3VFM9YJiKpZ6YhN8Vuze0fd/i/6bxL2smMvZ
6xnNGHiWPW9GdwvoRD+KPY6w1w1F39Ui8duw3Y0+znkmlKO/EyRZz+9SbAcU
PSuUc9WqIfXsUc124N8PfRe6sox7sT02V09KtzN2L8nmbWKvAHvYvw3MJt64
xHvpjuawvSZ1LV9l7W/kv9RP5KYi61l5tuBZoJmpWb4W/xORZ7reknrNkMhv
iu5K98Rvh+7MB/D+sXtBOdJsakYemfeM0iwbVHTtNdOWJZ4JmgWawU/HzrFy
qzf689RvoN4+xdBboxqpNnpzlqF/SvGznjnKdStyknPOjySe0ZrNg5FbEt9J
3UWd6TD8+8RnL0N+Ut9eMFfML5EvFVxbvcmaJQv+fx/Vj3/Ch8d+G0rRV8de
E9cdVe/VgNWRe3AkvAC2BL8p+tdQTVQL/XP0Qr666N7Xm1lQbWP/S2jGrOS8
K1K/vZp5+tfQzNas1j/H6+heSz0rdUcaEr/hertVs72xe1S9qZ5tTTzjNNuU
M/XC0cS5Uk/8BylbKy8=
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.7642875514468763, 0.8475504157671568, 0.9132888664362893], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFl12MnlUVhd/5/t/302KkVCGxxEToSCApxJuOSdPW1lSLSdvglRel0yEp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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlUtsllUQhs//ff/v/30HoYa2Sghy2agQTJQlJkCEAiomQCAuXCAtG4Fq
LDFqrIYFN4VyBxFQJG3CvRISbWCh0hgEbRsXbdS2YowVIVJIuETlFp43r4s3
mTlzOTNzZuaMq319/mtJCKEeFMEPMYRz4AiHM0ohDIUeBlYgbADNWQgThoRQ
90AIrehUIHuRs5qCdR6BfxjMxPZd+NM5QH4e3T/A4XII1+FnpyFsASn0WHRW
QL+H/gnol8BQdG9zth+6E5unoC9JB9+N4CCxXkZ/APuzyNchmwy6omNUbIph
Ofws+H50e8Eh/H0fnds19Evkshv5fvQPlJzbSOTHE+d4p2wdyRaRY0C3Cvnb
+Dqa2Pds8GDiO76AvoR8GfpdnHVE20i3hrNj+PsHfg537wDjkT8R7Xsz9i3I
v4Ofhm5b4tx0Jlo5dhJ/F6hCdz04gGwV+jeKPnsc2Q3O/kX/HfRbuKuPs8eK
zkm1TjK/jWr+DfRo8iuTayP8WXydAYcTxzAB+iF03kJ/q3oA+2fg7xHvXu4Z
lJ1qhO0Yzn6FTvG3r+Q3Wv2/jmT12Hfr7bD/Cv+D4Evonzmbh/4C0A7fnjsW
9Yx6sTO6durJ/5CPAp0FxyD6DeS/FX2mu9dwNqngGH7i/nrkfch3Ec8iZFvg
C8im4K+buzbA30Y+GX499KPozE19tgt5H/yT8APgI/gdYJL6GX4d9BowEb4X
fg/0bvAt929Nfdff8G8mvrMCehhoKLjGPblj0N2ji+7FQTA9dU/eIv4BfCwp
+ewu/PvwV9CtxUedZKAD/krBuTcpx8Q1+FrxI1+I/Y+c7YP/DFxIHWODZju6
9zTDh7DtidbVGynXj3Pnrpw/5/4a+J3wVdTzeegl4BT8OeyHU/9K0Fh0zhuR
bQL9qXNWrT5UTKlrdgR/z+auFWZhKvS03LvhaeRV0NXgBfUm9tXRM6De1wzW
wi+OflvVoDK6RqqNevZm5pnTrMnHn9E9qN5TTTU7K/HXk3qG6nLvHO0a5aRZ
OpN5d2mmLqI/MnMsS7H/S/4y06+qhzVb0btJb6RZ35m7VzTzy6GX5n6r9tS9
tQ3+99Q91gz9SubeVA/+Er1TtEvml7w71+buNe3Q7blrplrJx6fQn+R+G73x
BvSvlr1L1RO90TOnWVNPKPdluWNRDT7IvSO1G/VG2r2tZfemdvBG7Jui31o9
oN2iHaHdoB0zJ/Ofob9CM9mTuUaqjXZik+Yt91urJzT7bZlz0w44mXmHaXd1
E+9z8Kcy0y9rx2B4N3rXa2eNiP5D9Heo52Zm3vna9Zr5+3PV97k=
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.8297700998551583, 0.8693871008170272, 0.8839527434304271], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFlklslVUUx7/3Xl/f+z6CJnRYEF2oNLY2ETakGI2akgALwFhXJKVGdEGR
qgyuWIhShxZJWwfmWoIJg3EIYUqDEyExFNqSmAIiaghqVBJAoqVojPH/4/yJ
i5N77zn3nnuG/zn33rXk+Zbn8kmS/CsqiGamSXKxnCTDYl4pJsll0ZSKJBnQ
eq2oLZckhyX/JkuSU1ofr0ySEY136myNzlaL2qXogPbUpSF7SOM80Qbxr0pf
v86Ma75CtCMfug9I337RvZa/J5oinV3S87HOvqk946JBUVM+zl8XrcrHHcO2
AVtbc+ELOn4vhp566W4Q9VUE75r52MR6msaT0nFB44x82MXew9J1QmOm9YTm
r8iWdaKzhbgX/1dr7Lecu7EJXVXSX1kROlnv0npdKUnmaE+nbNldCju572Ih
9n8m/kb5/ILWf9rf95Pw74xzcETy77JYI2cf8flBvO9Fi+TPeY1fyM5SRcjw
j7h1VYZP2PCi70D/iPbW6EyV9o9p/Fm0QHq26q7ipCTZmY98jzv+vdIzKupx
jK86hlCd7zqos+ekp6YQdMpnR8QbEs0uxlnyxPmqQuSV+F2SPR+mkSNk8JEP
iqo9JydbFcOvkuDV+N4672E+Jht+yWLdJ31vpZGPZumcKnmL4vAlsRI1FcMm
ZKvFH00jLsRkseRt2K35Ien81vg/onGz1ts0Xyw/WkX1Ot9XGTFCF3ofKAZG
FmqcL5qVj5EYU1PUE3rRCYapO2wj3/Xes6ccueEMeX3CehgXeo4NDcXIaUId
Z+EL9uDb47mQ3+c9jykWX1cENk9r795y6Koi9qKlkk8Wf774NdK/XPM5mk/S
fJ7GuZ53ajyUhd7GYtyB7cT+dDnif9mxxa8fy4HjM8Z1o/cTM2zFzmmF/3Ex
6njCr3cMicndsnF3ObDcLflQZWD8vHibJTuWCz48MH/ce6jZ+4shg09uyC01
TL6vuO89JfuXiEYqAqf9xuoe4e5EKfCJX1eMmWb3TObc0+27GNd7ji8jrp01
+dB9AwzIh2uiDuccLJHT2cZTs3nkmNzhE/7Q9+gDreavkF3Ti3EXtd7lPZwn
huAe4v59uehJxII4YCPnOUsOFxvP1VngCb8G3QewDRyOuzdsz0f/a7Ev+NHo
3kaPIxe1+cARd3H3OepYqu9Jo9fTqxYZQ+DimnsEfRpdvYXQM+L+Ax6Wur/S
N/aX4w2jX4KJl93jToDtNPbTz/4pRU+jf0zORQ8ZdB+lb7yqvZ1p+EE/PppF
T35XvE1p+AQG61136O6y/vXO+018ifdaGn72OO/M293LwPYjkr+Thm9tGp8U
7XP8GInT0ULEFjunSz4jDd9GFf9jhcDPw+LtlT89Wj+ahl50btQ4UY735SPJ
xz3/ST79XYo6+i2Lfn2zV+ejL9AT3k6jlog5WOUe4nWHeH+VYx91cMm9utW2
8q7RG+gX9Ic/NP+gFLmmVqgxsDRuDBFDeNQcZy+X4+3ZaUzBB1c9jh8xmSr5
r1nEEQzw1oKDLWnkiRy9nsbd3EucsI0YzhJ/l+zp1vqTNP4oYGyZ33XwD2bn
Gm/0d/KPDdjE32XA8kZju9dz7iLnbzjvL2n8VMwx/jxZxI64EceZpYgPsm2u
ywWS355FrSzT2eVp2L1KvJWiC/6X1LqO1voPRY1jz1nbwRt70u9sr/sN+vGZ
+5h3SH62FLmt8/sNJuklg67x27J4A27VbY3/MtzRYX+flY1r08gdPZS3mTjT
z25h+RltfDoLzJ50fMEheJ3rHkFMbhjn5H3iFuaLkRvywh/xuv8K3MlfinuJ
4xrruG5M0BOG3Cfpg+g+Zgw8qDiv1Pkdpehd/cbasP9ZxJMYbfM/YYPjQ4+o
9j+HWG2Rns+z0NHgnsD5TeX4H3C+ye8/e9p9Dtv4F/DuINvhfs29A14z3+n8
UqdgcLvtoaapZ2IIJmp9/mbfyEVeeNuHXMvdzjmy/wCBOohE
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwllFtMDmAYx199fX3f+7axVZ8LNptjaCMby2HMZqOLYg43Noq4IMfElQuL
GhojTDVsjU1lDmshNkq6ispGSU5rYePCRUaOF37/PRfP9n8O73N+3olFe9bs
TnLOrYOSoaneuea4c10Iu6G3wbn30Pqocy/h62PO9WCTjnEE+xpwLTQE0zHK
uRvg29hnYv8DWQb8OPzdB2+HjsPPScV3inOrsb+DbTn6mdjPgL6BD2PTj+0K
qAxcAb0CZxF/mvzHLbdiZPXg2cjKwAegCeC/5FiHvgQ+Db4VSiHfbfDXwbOI
f4n4aeQ/BX4ydAL779AD8qnG5wVwL/b9+NqMrJv3B5GNoEt4q6UU/gh8ujff
I1A5uAEajljO44nVELNYw1GrvRL9QMR60Mj7F/jPQ7dSPYtbDPlO580f3n6F
vwxfQL+Wg3cFyyUV2gfORzYWPEiOC8El+O/E/3P4LeizkXXA98DnSY9sFXyC
N2/wP8lb764QbwBdCzbP0CWQLUa3BDoFzk4yX8Xw7RHzeZX3h7z1qhJqjNkM
NTu9qUZ33lst2hH1pgBqiliPPhHvY7Da8qn/M7iX9xkR25lz2J71tlun4fvQ
fQmWm2xuorsFnYzYDK8Rf6k3W8zcTvAOb7uqnDV77ah2UztwDHzUm61mkgOe
762WJmJmoigN1tsW+HvE7ws2q7XQO3AFsqyo3UihNx96qxo3od8arPe/kJ3x
1lP1chD977jNTLPSTO4Gm4lmoZtohX8E5YAvat+83YRuQTcyl3py47YL82I2
+0XwdTHbgRnBalDuVcTo0i5Dy/C3H1kBeAP0E9yJvihYzspVO9/mrWbVGku2
W6rylrtuSrvbELfZaYezvd2Ubkk3mUCfESyW/owx4NHBdk872w6ujVtte7F/
Cv8k2N+gHNvAj6EFUeuBeq0/Q3+Fet4c7AZ1e/pzasAPg/nSDf+L2Z+gv0A3
lOvtJnWL2hn9FdOD9UZ/hmJtDNYLxRzC34e4/X36E9WrQuWYbD17rXkF641u
5j+xLLi7
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.8952526482634402, 0.8912237858668975, 0.8546166204245649], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJw1lUuIl2UUxj/nf/n+3yfdFwppQ2otK7DUaGaoVhHUokUj419HglqFtqim
Tcl4YwRnomDGLkIWTBl0WWgttOwCAxLCbHIGumFqE81KjHAkFz2/nsfF+73n
e895z/U5573zmZ1P7egpiuK0VkPrt7ooftEabIluFsW41gc639MpimM6f63H
vKe1RqHF36w1I5nvxf9Ga0r0Q23zDolej7zodyS/qm0amwcaluHuVdEva/0u
xrqqKO6r/N8n+QOSuVR4H9N6T/QXsrOvYx8uaD+v9aDk/yyLYpl4f0lmTv9P
RB6/D4o+Ujimg4nrj5Ztzoo+of1c6GvLTC/F//tb9vkl6b4gG2cK634y+nfL
38vyQVtxw/KieER0UxdelWwpP0a0/hWzqfvbdWeN9nW1Y6q0b5T889GHXvzn
f1Zrq1af1hblYlePfV6I312d9+v8n8RyUeukzt+Wvu+kd5P+N7ecm+nCtoeb
9p/8D6UWW5qmqcVEwzLEsrpt3vul/cEXbGGzm3wi343ObmSgsVnGbm/btq/X
ZSzYOCQ/v6qd330yOFqZT7z9Wj/0OO6+0M9K9rnaeSH+E6nXKd27tbI/hwFy
03lYahhbyN4r/kcd34EPD5k3Kt8/Ep/3x++9HeNsNHbOBg8bWs4reD5fGhPE
e0zyd1WuNfbAL/euVa498a+tjG94t1XuK7B5XPvd+n9F559qv0cYupL+oE7k
apfOPyxdr3nZn2tZZo3Ofy7t3+PSs6M2JsHjx6XrSO4OJ96jOrt9uf/RQ27B
1aLOW7V7DX1f1tZ5R9uYRM9K0Su0pnV3a+2en0rdwQE5nGmYvyj6c136rHKu
0UFt0TOgT3/lO+icbbgfz7Zcb/zdJv7RyrGPkJvKcpztzTmzZH16c5PONlbO
F703Ex/AYze4RT++ES9zazCzhTnDzAKXA23jmxkxrPi6WqebzsVoMIDMeHRS
q08q1wL9+L6YOTMRGfwajm9jyQM1mmp7ntIjg+lVaPp7KTjarVrcXDpW+mNP
euRR1ffhzJke+Xi58vygto9Vru9k2zbQie/M7aXYhcf/TWCvtq8jsjNeGtvv
lp6DyFOrN1Ovls4nS+v8qTTmiOWBlmvB3R9bzh/YZk7/mj4iNyeTn25pG+j/
Wvy3OtZBTMzQZnqfnFKvW0rPopnMJd4SZhZ9gD1sMU/7UsedtecsZ/g7mTxM
p06LmUvgm3k3URrLG9JXc9H5gnTc2PF7SD5WZeYQH3bBbVsyS5UxNZQegdeo
/SYx985ljhAvuociQ93+Tu32561g/hyvPQuupHfwEz3DoavM7dXJA2e9kcGX
q/Fnbe3eBufk/mJiZ+fNmI1vxPY/LdkXa+cZDC7kTcHn+dSCXL5eGj/EMZdz
5hdvHjHQN9vSO8yY+eR1IXbRSZ6G8q4xH3tTC3qROl5KnunH62/3QHpzNrNj
Oj3FHWZCGYxSa/KLDWrEvjL0cObVmeBhRXz4Vv6eqo0BbA7kbQVr5Jl3cDC2
8I28w2P+8F7Rc2B8KHN7Jn09lV77D/npSQ0=
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtkz1IlVEYx0/3vtf3PSfoc2lwKbO1qEwFK2oKAgcHc0iLoGgIbYiuU6FZ
4OCNGrQi6AP6gqhBacg+ByEiuFMNURHaki1BRBoN/f78hffP+3/O8/2c56w/
MtDVXwoh7AYZeJuHcB9cL4ewk4NNKYQfyL/R1ZDvFSFsiCG8x+kpeIn+FdhR
CWEcm2n4BDbbkbeBX9h+xP8Guk78/yLPIr9D7kXej+1JfL7Ab3O2Ed4EfsJH
OcvgZfAd+TDyC/hVfNoqzrmGeGvBGWq5ic1gdEzFqtPDfHRP6qUDtKPrBI3Y
H+d8HP6cmC3EusbZMngpuVblbMF/EpsqsS6AdcvJjTyP7hs+zdiuLDybMdAN
GhtCuEXsIXCl8Ew0C+V4Ah/hbKjkGa4oHEO+moFi7wVZaSlH7pzKVUfuop5q
dG93wGfNDhyoOOYpeCv2q/Gbwf4i/4NgAd6NzTSx50A7vj3IY+hquWep/veg
2wpmyq5hFbpqbp16U1/Dubn66yNfb/Ldv8k8W9WsWjXjY8l3rruug8vwS9F3
VSPHw+g70t38Qa5g3wB6Mu/AOXTDgC98QH8I0gcmtB/IW+Cbo3fxNPII/Hy0
rWyeEesssnZcO6neNEPNTj3OFr5j3a1mMon9VHIt2qm79NIWnUsxWrE/mtzL
CfAA/a7oXlTjp+QZa7a6k1F4U3Rti+BR9Jn4ApjD/1/0butNvE7eSc1bO96P
PJD8FvSGOrB9HO2rB7sv+kxcb7YZPoX/YNk7q7cbk2vVG15E/3VpNzTj/70b
ho8=
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.941176, 0.906538, 0.834043], EdgeForm[None], 
      GraphicsGroupBox[{
        PolygonBox[{{2104, 2840, 509}, {2595, 2596, 1162}, {1245, 531, 432}, {
         1103, 1217, 496}, {2661, 2103, 506}, {2562, 2561, 1126}, {517, 1162, 
         432}, {515, 1127, 507}, {2563, 2564, 1127}, {2566, 2565, 1128}, {
         1943, 1893, 507}, {1942, 1892, 506}, {2423, 2386, 497}, {1217, 1126, 
         496}, {2018, 2554, 1118}, {1232, 1121, 507}, {539, 1118, 506}, {2098,
          2113, 497}, {1127, 1232, 507}, {2569, 2027, 519}, {531, 1103, 
         496}, {2027, 2598, 519}, {2010, 2547, 1103}, {2016, 2553, 1113}, {
         1121, 539, 507}, {2020, 2126, 509}, {1121, 1251, 539}, {2123, 2424, 
         532}, {1126, 515, 496}, {2671, 2670, 1245}, {2597, 2062, 436}, {2554,
          2662, 1118}, {2871, 2676, 1250}, {1936, 1887, 497}, {1251, 1118, 
         539}, {1891, 1942, 506}, {2114, 2104, 509}, {2672, 2123, 532}, {2126,
          2105, 509}, {1883, 1778, 436}, {2556, 2019, 1121}, {2063, 2597, 
         436}, {2386, 2098, 497}, {1894, 1943, 507}, {2840, 2020, 509}, {1886,
          531, 496}, {1472, 1936, 497}},
         VertexColors->None], 
        PolygonBox[{{2546, 2562, 1126, 1217}, {2026, 2595, 1162, 517}, {2103, 
         2569, 519, 506}, {2677, 2018, 1118, 1251}, {2594, 2671, 1245, 432}, {
         1892, 1874, 539, 506}, {2565, 2026, 517, 1128}, {1778, 1902, 519, 
         436}, {1893, 1900, 515, 507}, {2552, 2566, 1128, 1223}, {2676, 2016, 
         1113, 1250}, {2564, 2555, 1232, 1127}, {2553, 2552, 1223, 1113}, {
         2019, 2677, 1251, 1121}, {1874, 1894, 507, 539}, {2561, 2023, 515, 
         1126}, {2424, 2423, 497, 532}, {2670, 2122, 531, 1245}, {2547, 2546, 
         1217, 1103}, {1902, 1891, 506, 519}, {2062, 2672, 532, 436}, {2596, 
         2594, 432, 1162}, {1887, 1864, 532, 497}, {2662, 2661, 506, 1118}, {
         2023, 2563, 1127, 515}, {2122, 2010, 1103, 531}, {1900, 1885, 496, 
         515}, {2113, 2829, 1472, 497}, {2829, 2112, 1901, 1472}, {2555, 2556,
          1121, 1232}, {2598, 2063, 436, 519}, {1864, 1883, 436, 532}},
         VertexColors->None], PolygonBox[{{2112, 2114, 509, 1895, 1901}},
         VertexColors->None]}]}}, {{}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2105, 2126, 2020, 2840, 2104, 2114, 2112, 2829, 2113, 2098, 
         2386, 2423, 2424, 2123, 2672, 2062, 2597, 2063, 2598, 2027, 2569, 
         2103, 2661, 2662, 2554, 2018, 2677, 2019, 2556, 2555, 2564, 2563, 
         2023, 2561, 2562, 2546, 2547, 2010, 2122, 2670, 2671, 2594, 2596, 
         2595, 2026, 2565, 2566, 2552, 2553, 2016, 2676, 2871}],
        "1.5`"],
       Annotation[#, 1.5, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2342, 2228, 2341, 2227, 2357, 2274, 2275, 2235, 2780, 2782, 
         2781, 2181, 2741, 2180, 2742, 2011, 2338, 2215, 2351, 2352, 2340, 
         2339, 2185, 2478, 2479, 2364, 2376, 2843, 1960, 1994, 2538, 2476, 
         2862, 2477, 2740, 2739, 2179, 2323, 2322, 2494, 2449, 2450, 2365, 
         2366, 1961, 2290, 2434, 2435, 2433, 2437, 2436, 2472, 2398, 2399, 
         2359, 2360, 1958, 2431, 2748, 2199, 2747, 2200, 2749, 2198, 2753, 
         2201, 2750, 2752, 2751, 2754, 2653, 2654, 2652, 2655}],
        "1.25`"],
       Annotation[#, 1.25, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2418, 2107, 2417, 2106, 2297, 2076, 2673, 2124, 2573, 2574, 
         2575, 2125, 2674, 2675, 2599, 2601, 2600, 2043, 2334, 2044, 2821, 
         2202, 2285, 2042, 2788, 2789, 2772, 2216, 2771, 2217, 2773, 2232, 
         2495, 2779, 2778, 2230, 2776, 2231, 2777, 2229, 2793, 2261, 2791, 
         2260, 2792, 2074, 2324, 2325, 2294, 2293, 1965, 2295, 2454, 2455, 
         2453, 2457, 2456, 2267, 2451, 2452, 2367, 2368, 1962, 2291, 2439, 
         2440, 2438, 2442, 2865, 2441, 2473, 2762, 2763, 2761, 2765, 2764, 
         2048, 2580, 2581, 2509, 2510, 1959, 2692, 2432, 2411, 2412, 2410, 
         2096, 2659, 2660, 2656, 2658, 2657, 2007, 2669, 2118, 2668, 2119, 
         2120, 2868}],
        "1"],
       Annotation[#, 1, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2309, 2121, 2234, 2841, 2233, 2343, 2036, 2246, 2817, 2008, 
         2497, 2241, 2496, 2242, 2498, 2240, 2846, 2847, 2813, 2276, 1992, 
         1956, 2759, 2760, 2758, 2210, 2319, 2318, 2474, 2446, 2447, 2443, 
         2445, 2444, 2292, 1963, 2370, 2369, 2459, 2458, 2268, 2463, 2464, 
         2460, 2462, 2461, 2296, 1967, 2373, 2372, 2702, 2701, 2466, 2378, 
         2404, 2403, 2465, 1997, 1966, 2517, 2516, 2700, 2699, 2077, 2467, 
         2144, 2706, 2143, 2705, 2539, 2814, 2834, 2835, 2585, 2584, 2051, 
         2336, 2050, 2583, 2582, 2766, 2211, 2858, 2586, 2052, 2823, 2212, 
         2523, 2602, 2064, 2824, 2579, 2045, 2686, 2132, 2683, 2685, 2684, 
         2300, 2305, 2013, 2819, 2182, 2189, 2188, 2031, 2855, 2482, 2391}],
        "0.75`"],
       Annotation[#, 0.75, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2021, 2560, 2389, 2109, 2419, 2162, 2163, 2219, 2488, 2035, 
         2392, 2032, 2387, 2420, 2854, 2015, 2421, 2284, 2279, 2864, 2280, 
         2256, 2353, 2022, 2278, 2100, 2828, 2101, 2041, 2130, 2831, 2131, 
         2129, 2682, 2128, 2005, 1982, 2414, 2415, 2413, 2102, 2416, 2549, 
         2548, 2651, 2650, 2024, 2354, 2025, 2281, 2072, 2289, 2192, 2838, 
         2193, 2870}],
        "0.5`"],
       Annotation[#, 0.5, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{2277, 2667, 2115, 2567, 2568, 2249, 2037, 2346, 2347, 2248, 
         2842, 2247, 2009, 2852, 2853, 2851, 2850, 2818, 2422, 2385, 2097, 
         2757, 2204, 2755, 2203, 2756, 2486, 2487, 2363, 2362, 2361, 2135, 
         2448, 2265, 2266, 2512, 2511, 2136, 2695, 2073, 2697, 2138, 2696, 
         2137, 1995, 2515, 2514, 2513, 2139, 2698, 2075, 2704, 2141, 2703, 
         2140, 1998, 2520, 2519, 2518, 2142, 2299, 2298, 2522, 2521, 2468, 
         2145, 2469, 2269, 2710, 2146, 2707, 2709, 2708, 2301, 1969, 2406, 
         2407, 2405, 2078, 2717, 2718, 2715, 2151, 2716, 2815, 2836, 2837, 
         2590, 2589, 2054, 2337, 2053, 2588, 2587, 2767, 2213, 2591, 2055, 
         2066, 2825, 2065, 2134, 2822, 2049, 2690, 2133, 2687, 2689, 2688, 
         2691, 2308, 2014, 2820, 2183, 2191, 2190, 2034, 2745, 2572, 2827, 
         2067, 2856, 2826}],
        "0.5`"],
       Annotation[#, 0.5, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2187, 2186, 2017, 2332, 2333, 2244, 2787, 2243, 1993, 1957, 
         2769, 2770, 2768, 2214, 2321, 2320, 2475, 2175, 2401, 2061, 2400, 
         2060, 2693, 2694, 2592, 2593, 2252, 2059, 2349, 2350, 2287, 2848, 
         2286, 1996, 1964, 2844, 2377, 2371, 2395, 2394, 2057, 2502, 2503, 
         2501, 2505, 2504, 2253, 2859, 2860, 2849, 2288, 1999, 1968, 2845, 
         2379, 2374, 2397, 2396, 2058, 2712, 2148, 2711, 2147, 2000, 2270, 
         2150, 2713, 2149, 2714, 2302, 2526, 2525, 2524, 2800, 2799, 2721, 
         2540, 2079, 2604, 2720, 2719, 2001, 1970, 2798, 2152, 2080, 2303, 
         2530, 2529, 2155, 2802, 2725, 2541, 2606, 2605, 2724, 2002, 1972, 
         2801, 2068, 2056, 2402, 2070, 2603, 2069, 2612, 2613, 2611, 2082, 
         2664, 2108, 2557, 2558, 2559, 2116, 2830, 2117, 2039, 2348, 2038, 
         2250, 2307, 2869}],
        "0.25`"],
       Annotation[#, 0.25, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{2866, 2197, 2388, 2283, 2030, 2355, 2029, 2282, 2245, 2785, 
         2238, 2784, 2239, 2786, 2236, 2344, 2551, 2550, 2237, 2783, 2172, 
         2646, 2094, 2644, 2093, 2645, 2168, 2629, 2090, 2628, 2089, 2735, 
         2736, 2536, 2537, 2382, 2381, 2380, 2086, 2408, 2535, 1978, 2004, 
         2273, 2161, 2805, 1977, 2205, 2726, 2609, 2610, 2607, 2081, 2608, 
         2153, 2832, 2154, 2271, 2722, 2723, 2527, 2528, 1971, 2743, 2744, 
         2648, 2649, 2647, 2099, 2680, 2681, 2678, 2127, 2679, 2012, 2663, 
         2867}],
        "0.25`"],
       Annotation[#, 0.25, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2196, 2576, 2577, 2578, 2251, 2500, 2499, 2393, 2226, 2493, 
         2264, 2258, 2257, 2356, 2178, 2263, 2508, 2485, 2195, 2746, 2194, 
         2484, 2861, 2483, 2384, 2383, 2863, 2095, 2409, 1989, 2545, 2544, 
         2543, 2092, 2167, 2833, 2166, 2627, 2088, 2625, 2087, 2626, 2734, 
         2733, 2311, 2312, 2220, 2806, 1980, 2310, 1976, 2804, 2218, 2160, 
         2774, 2775, 2531, 2534, 1975, 2532, 2533, 2470, 2857, 75, 2619, 2621,
          2620, 1974, 2617, 2618, 2615, 2084, 2616, 1973, 2803, 2159, 2730, 
         2542, 2083, 2614, 2156, 2727, 2003, 2272, 2158, 2728, 2157, 2729, 
         2304, 1979, 2623, 2624, 2622, 2085, 2732, 2165, 2731, 2164, 2738, 
         2177, 2737, 2176, 2207, 2047, 2335, 2046, 2206, 2209, 2839, 2208, 
         1981, 2807, 2071, 2040, 2428, 2429, 2425, 2427, 2426, 2430, 2306}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2110, 2665, 2111, 2666, 2028, 2169, 2808, 1983, 2006, 2313, 
         2816, 2254, 2790, 2255, 2796, 2797, 2794, 2262, 2795, 2506, 2507, 
         2471, 2170, 2375, 1984, 2631, 2091, 2630, 2633, 2632, 1985, 2635, 
         2636, 2634, 2638, 2637, 1986, 2640, 2641, 2639, 2643, 2642, 1987, 
         2809, 2221, 2171, 2222, 2810, 1988, 2314, 1990, 2811, 2223, 2173, 
         2224, 2812, 1991, 2315, 2174, 2492, 2225, 2489, 2491, 2490, 2316, 
         2317, 2326, 2327, 2259, 2358, 2184, 2328, 2390, 2480, 2481, 2330, 
         2329, 2345, 2331, 2033, 2571, 2570}],
        RowBox[{"-", "0.25`"}]],
       Annotation[#, -0.25, "Tooltip"]& ]}, {}, {}, {}}}],
  AspectRatio->1,
  Epilog->{
    RGBColor[1, 0, 0], 
    InsetBox[
     FormBox["\"local maximum\"", TraditionalForm], {0.927, 0.374}], 
    InsetBox[
     FormBox["\"global maximum\"", TraditionalForm], {-0.657, -0.753}]},
  Frame->True,
  PlotRange->{{-1.6, 1.6}, {-1.6, 1.6}},
  PlotRangeClipping->True,
  PlotRangePadding->{Automatic, Automatic}]], "Output",
 ImageSize->{184, 172},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->490439125]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Neat Examples",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->21890],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Solve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Solve"]], "InlineFormula"],
 " cannot work with this system of equations because they are highly \
nonalgebraic:"
}], "ExampleText",
 CellID->1680507868],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Solve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"Floor", "[", 
      SuperscriptBox["x", "2"], "]"}], "\[Equal]", "196"}], "&&", 
    RowBox[{
     RowBox[{"Mod", "[", 
      RowBox[{"x", ",", "10"}], "]"}], "\[Equal]", "4"}]}], ",", 
   RowBox[{"{", "x", "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->398952527],

Cell[BoxData[
 RowBox[{
  RowBox[{"InverseFunction", "::", "\<\"ifun\"\>"}], ":", 
  " ", "\<\"Inverse functions are being used. Values may be lost for \
multivalued inverses. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/InverseFunction/ifun\\\", ButtonNote -> \\\
\"InverseFunction::ifun\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->15685951],

Cell[BoxData[
 RowBox[{
  RowBox[{"Solve", "::", "\<\"ifun\"\>"}], ":", 
  " ", "\<\"Inverse functions are being used by \\!\\(Solve\\), so some \
solutions may not be found; use Reduce for complete solution information. \\!\
\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/Solve/ifun\\\", \
ButtonNote -> \\\"Solve::ifun\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->41064949],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"x", "\[Rule]", 
    RowBox[{
     RowBox[{"InverseFunction", "[", 
      RowBox[{"Mod", ",", "1", ",", "2"}], "]"}], "[", 
     RowBox[{"4", ",", "10"}], "]"}]}], "}"}], "}"}]], "Output",
 ImageSize->{278, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->32046333]
}, Open  ]],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["NMaximize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NMaximize"]], "InlineFormula"],
 " with a trivial objective function to find a solution:"
}], "ExampleText",
 CellID->1220382311],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMaximize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", 
     RowBox[{
      RowBox[{
       RowBox[{"Floor", "[", 
        SuperscriptBox["x", "2"], "]"}], "\[Equal]", "196"}], "&&", 
      RowBox[{
       RowBox[{"Mod", "[", 
        RowBox[{"x", ",", "10"}], "]"}], "\[Equal]", "4"}]}]}], "}"}], ",", 
   RowBox[{"{", "x", "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1696840635],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.`", ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", "14.`"}], "}"}]}], "}"}]], "Output",
 ImageSize->{102, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->465961250]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->7273],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["NMinimize",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NMinimize"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Maximize",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Maximize"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["FindMaximum",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FindMaximum"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["LinearProgramming",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/LinearProgramming"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->16088]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->18835],

Cell[TextData[{
 ButtonBox["Numerical Mathematics in ",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NumericalMathematicsInMathematica"],
 StyleBox[ButtonBox["Mathematica",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NumericalMathematicsInMathematica"],
  FontSlant->"Italic"]
}], "Tutorials",
 CellID->19652],

Cell[TextData[ButtonBox["Numerical Optimization",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/NumericalOptimization"]], "Tutorials",
 CellID->23171],

Cell[TextData[ButtonBox["Constrained Optimization",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/ConstrainedOptimizationOverview"]], "Tutorials",\

 CellID->78894921]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection"],

Cell[TextData[{
 ButtonBox["Demonstrations with NMaximize",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/symbol.html?symbol=NMaximize"], 
    None}],
 " (",
 ButtonBox["Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None}],
 ")"
}], "RelatedLinks"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->403692888],

Cell[TextData[ButtonBox["Optimization",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/Optimization"]], "MoreAbout",
 CellID->300660481],

Cell[TextData[ButtonBox["Polynomial Systems",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/PolynomialSystems"]], "MoreAbout",
 CellID->203492113]
}, Open  ]],

Cell["New in 5  |  Last modified in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"NMaximize - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 2, 11.1697874}", "context" -> "System`", 
    "keywords" -> {
     "constrained optimization", "cost function", "differential evolution", 
      "extremization", "flexible polyhedron method", "goal functions", 
      "integer programming", "linear programming", "maximization", 
      "minimization", "Nelder-Mead", "numerical maximization", 
      "objective functions", "operations research", "optimization", 
      "pay-off functions", "random search", "simulated annealing"}, "index" -> 
    True, "label" -> "Built-in Mathematica Symbol", "language" -> "en", 
    "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "NMaximize[f, {x, y, ...}] maximizes f numerically with respect to x, y, \
.... NMaximize[{f, cons}, {x, y, ...}] maximizes f numerically subject to the \
constraints cons. ", "synonyms" -> {}, "title" -> "NMaximize", "type" -> 
    "Symbol", "uri" -> "ref/NMaximize"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[17306, 519, 358, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->6487599]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 824090, 14973}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2597, 60, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3220, 87, 4193, 83, 70, "ObjectNameGrid"],
Cell[7416, 172, 1688, 56, 70, "Usage",
 CellID->12975]
}, Open  ]],
Cell[CellGroupData[{
Cell[9141, 233, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[9619, 250, 714, 27, 70, "Notes",
 CellID->4813],
Cell[10336, 279, 178, 5, 70, "Notes",
 CellID->1183],
Cell[10517, 286, 311, 10, 70, "Notes",
 CellID->10391],
Cell[10831, 298, 83, 1, 70, "Notes",
 CellID->23646],
Cell[10917, 301, 295, 9, 70, "Notes",
 CellID->21374],
Cell[11215, 312, 521, 17, 70, "Notes",
 CellID->14292637],
Cell[11739, 331, 227, 8, 70, "Notes",
 CellID->29994],
Cell[11969, 341, 685, 24, 70, "Notes",
 CellID->3559],
Cell[12657, 367, 69, 1, 70, "Notes",
 CellID->19127],
Cell[12729, 370, 2553, 69, 70, "3ColumnTableMod",
 CellID->11773],
Cell[15285, 441, 520, 20, 70, "Notes",
 CellID->28601],
Cell[15808, 463, 472, 14, 70, "Notes",
 CellID->32207],
Cell[16283, 479, 503, 17, 70, "Notes",
 CellID->22495],
Cell[16789, 498, 480, 16, 70, "Notes",
 CellID->552]
}, Closed]],
Cell[CellGroupData[{
Cell[17306, 519, 358, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->6487599],
Cell[CellGroupData[{
Cell[17689, 533, 147, 5, 70, "ExampleSection",
 CellID->60213890],
Cell[17839, 540, 96, 1, 70, "ExampleText",
 CellID->1917251726],
Cell[CellGroupData[{
Cell[17960, 545, 255, 9, 28, "Input",
 CellID->1746774899],
Cell[18218, 556, 299, 10, 36, "Output",
 CellID->22990837]
}, Open  ]],
Cell[18532, 569, 76, 1, 70, "ExampleText",
 CellID->1664023518],
Cell[CellGroupData[{
Cell[18633, 574, 327, 12, 28, "Input",
 CellID->1557760567],
Cell[18963, 588, 176, 5, 36, "Output",
 CellID->39490870]
}, Open  ]],
Cell[19154, 596, 126, 3, 70, "ExampleDelimiter",
 CellID->1112598013],
Cell[19283, 601, 98, 1, 70, "ExampleText",
 CellID->1719552892],
Cell[CellGroupData[{
Cell[19406, 606, 483, 17, 70, "Input",
 CellID->886900649],
Cell[19892, 625, 373, 12, 36, "Output",
 CellID->37726438]
}, Open  ]],
Cell[CellGroupData[{
Cell[20302, 642, 1119, 36, 70, "Input",
 CellID->1759366686],
Cell[21424, 680, 419, 14, 36, "Output",
 CellID->47497397]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[21892, 700, 216, 7, 70, "ExampleSection",
 CellID->26512],
Cell[22111, 709, 196, 7, 70, "ExampleText",
 CellID->1077996427],
Cell[CellGroupData[{
Cell[22332, 720, 706, 24, 70, "Input",
 CellID->1881132418],
Cell[23041, 746, 374, 12, 36, "Output",
 CellID->503175292]
}, Open  ]],
Cell[23430, 761, 125, 3, 70, "ExampleDelimiter",
 CellID->145340294],
Cell[23558, 766, 227, 8, 70, "ExampleText",
 CellID->1627702079],
Cell[CellGroupData[{
Cell[23810, 778, 615, 20, 70, "Input",
 CellID->555314442],
Cell[24428, 800, 362, 12, 36, "Output",
 CellID->767200545]
}, Open  ]],
Cell[24805, 815, 125, 3, 70, "ExampleDelimiter",
 CellID->592370280],
Cell[24933, 820, 79, 1, 70, "ExampleText",
 CellID->1554484569],
Cell[CellGroupData[{
Cell[25037, 825, 731, 24, 70, "Input",
 CellID->1605455941],
Cell[25771, 851, 337, 12, 36, "Output",
 CellID->32758853]
}, Open  ]],
Cell[CellGroupData[{
Cell[26145, 868, 549, 19, 70, "Input",
 CellID->1044023529],
Cell[26697, 889, 375, 13, 36, "Output",
 CellID->147056107]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[27121, 908, 222, 7, 70, "ExampleSection",
 CellID->453790855],
Cell[CellGroupData[{
Cell[27368, 919, 248, 7, 70, "ExampleSubsection",
 CellID->61701773],
Cell[27619, 928, 782, 28, 70, "ExampleText",
 CellID->764461149],
Cell[CellGroupData[{
Cell[28426, 960, 309, 9, 70, "Input",
 CellID->1625295724],
Cell[28738, 971, 298, 10, 36, "Output",
 CellID->33024731]
}, Open  ]],
Cell[29051, 984, 126, 3, 70, "ExampleDelimiter",
 CellID->1055000278],
Cell[29180, 989, 858, 28, 70, "ExampleText",
 CellID->1329553122],
Cell[CellGroupData[{
Cell[30063, 1021, 311, 9, 70, "Input",
 CellID->1468387723],
Cell[30377, 1032, 225, 5, 70, "Message",
 CellID->501329],
Cell[30605, 1039, 299, 10, 36, "Output",
 CellID->28621414]
}, Open  ]],
Cell[30919, 1052, 244, 8, 70, "ExampleText",
 CellID->1199314865],
Cell[CellGroupData[{
Cell[31188, 1064, 365, 10, 70, "Input",
 CellID->39134653],
Cell[31556, 1076, 356, 12, 54, "Output",
 CellID->320301033]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[31961, 1094, 238, 7, 70, "ExampleSubsection",
 CellID->405919056],
Cell[32202, 1103, 150, 4, 70, "ExampleText",
 CellID->1914018609],
Cell[32355, 1109, 1114, 36, 70, "Input",
 CellID->157454682],
Cell[33472, 1147, 146, 4, 70, "ExampleText",
 CellID->738842830],
Cell[CellGroupData[{
Cell[33643, 1155, 1320, 42, 70, "Input",
 CellID->629793524],
Cell[34966, 1199, 460392, 7636, 70, "Output",
 CellID->289924982]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[495407, 8841, 226, 7, 70, "ExampleSubsection",
 CellID->23837268],
Cell[495636, 8850, 113, 3, 70, "ExampleText",
 CellID->1424437279],
Cell[CellGroupData[{
Cell[495774, 8857, 449, 15, 70, "Input",
 CellID->1467809973],
Cell[496226, 8874, 296, 10, 36, "Output",
 CellID->36353381]
}, Open  ]],
Cell[CellGroupData[{
Cell[496559, 8889, 524, 17, 70, "Input",
 CellID->394924060],
Cell[497086, 8908, 279, 9, 36, "Output",
 CellID->179384147]
}, Open  ]],
Cell[CellGroupData[{
Cell[497402, 8922, 378, 13, 70, "Input",
 CellID->2988854],
Cell[497783, 8937, 2875, 51, 70, "Output",
 Evaluatable->False,
 CellID->34517295]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[500707, 8994, 232, 7, 70, "ExampleSubsection",
 CellID->151017442],
Cell[500942, 9003, 179, 6, 70, "ExampleText",
 CellID->922642942],
Cell[501124, 9011, 1080, 37, 70, "Input",
 CellID->1534545049],
Cell[CellGroupData[{
Cell[502229, 9052, 1462, 48, 70, "Input",
 CellID->1614037545],
Cell[503694, 9102, 156547, 2654, 70, "Output",
 CellID->273159911]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[660290, 11762, 237, 7, 70, "ExampleSubsection",
 CellID->792799295],
Cell[660530, 11771, 552, 20, 70, "ExampleText",
 CellID->1087085504],
Cell[CellGroupData[{
Cell[661107, 11795, 482, 16, 70, "Input",
 CellID->2145912710],
Cell[661592, 11813, 425, 14, 36, "Output",
 CellID->386185489]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[662078, 11834, 232, 7, 70, "ExampleSection",
 CellID->1677],
Cell[662313, 11843, 305, 9, 70, "ExampleText",
 CellID->316126506],
Cell[CellGroupData[{
Cell[662643, 11856, 499, 18, 70, "Input",
 CellID->1726927757],
Cell[663145, 11876, 417, 13, 39, "Output",
 CellID->526508987]
}, Open  ]],
Cell[CellGroupData[{
Cell[663599, 11894, 500, 18, 70, "Input",
 CellID->283369064],
Cell[664102, 11914, 399, 13, 39, "Output",
 CellID->411653134]
}, Open  ]],
Cell[CellGroupData[{
Cell[664538, 11932, 1465, 45, 70, "Input",
 CellID->288919193],
Cell[666006, 11979, 59831, 1014, 70, "Output",
 CellID->286241299]
}, Open  ]],
Cell[725852, 12996, 237, 7, 70, "ExampleText",
 CellID->1543511006],
Cell[CellGroupData[{
Cell[726114, 13007, 503, 18, 70, "Input",
 CellID->1936029081],
Cell[726620, 13027, 461, 17, 56, "Output",
 CellID->147824252]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[727130, 13050, 231, 7, 70, "ExampleSection",
 CellID->1659867246],
Cell[727364, 13059, 251, 8, 70, "ExampleText",
 CellID->1669385690],
Cell[CellGroupData[{
Cell[727640, 13071, 307, 10, 70, "Input",
 CellID->1982487473],
Cell[727950, 13083, 296, 10, 36, "Output",
 CellID->10018934]
}, Open  ]],
Cell[CellGroupData[{
Cell[728283, 13098, 384, 13, 70, "Input",
 CellID->1912208167],
Cell[728670, 13113, 4892, 84, 70, "Output",
 Evaluatable->False,
 CellID->48544354]
}, Open  ]],
Cell[733577, 13200, 127, 3, 70, "ExampleText",
 CellID->657148640],
Cell[CellGroupData[{
Cell[733729, 13207, 390, 13, 70, "Input",
 CellID->723115657],
Cell[734122, 13222, 296, 10, 36, "Output",
 CellID->181556204]
}, Open  ]],
Cell[734433, 13235, 125, 3, 70, "ExampleDelimiter",
 CellID->993018304],
Cell[734561, 13240, 240, 7, 70, "ExampleText",
 CellID->220009434],
Cell[CellGroupData[{
Cell[734826, 13251, 540, 19, 70, "Input",
 CellID->33483118],
Cell[735369, 13272, 374, 12, 36, "Output",
 CellID->15713487]
}, Open  ]],
Cell[735758, 13287, 121, 3, 70, "ExampleText",
 CellID->1122017447],
Cell[CellGroupData[{
Cell[735904, 13294, 630, 23, 70, "Input",
 CellID->559037039],
Cell[736537, 13319, 413, 13, 36, "Output",
 CellID->26332952]
}, Open  ]],
Cell[CellGroupData[{
Cell[736987, 13337, 1461, 48, 70, "Input",
 CellID->186971854],
Cell[738451, 13387, 77195, 1295, 70, "Output",
 CellID->490439125]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[815695, 14688, 224, 7, 70, "ExampleSection",
 CellID->21890],
Cell[815922, 14697, 254, 8, 70, "ExampleText",
 CellID->1680507868],
Cell[CellGroupData[{
Cell[816201, 14709, 363, 12, 70, "Input",
 CellID->398952527],
Cell[816567, 14723, 427, 8, 70, "Message",
 CellID->15685951],
Cell[816997, 14733, 452, 8, 70, "Message",
 CellID->41064949],
Cell[817452, 14743, 374, 12, 36, "Output",
 CellID->32046333]
}, Open  ]],
Cell[817841, 14758, 243, 8, 70, "ExampleText",
 CellID->1220382311],
Cell[CellGroupData[{
Cell[818109, 14770, 431, 14, 70, "Input",
 CellID->1696840635],
Cell[818543, 14786, 264, 9, 36, "Output",
 CellID->465961250]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[818868, 14802, 310, 9, 70, "SeeAlsoSection",
 CellID->7273],
Cell[819181, 14813, 967, 35, 70, "SeeAlso",
 CellID->16088]
}, Open  ]],
Cell[CellGroupData[{
Cell[820185, 14853, 314, 9, 70, "TutorialsSection",
 CellID->18835],
Cell[820502, 14864, 324, 9, 70, "Tutorials",
 CellID->19652],
Cell[820829, 14875, 154, 3, 70, "Tutorials",
 CellID->23171],
Cell[820986, 14880, 171, 4, 70, "Tutorials",
 CellID->78894921]
}, Open  ]],
Cell[CellGroupData[{
Cell[821194, 14889, 305, 8, 70, "RelatedLinksSection"],
Cell[821502, 14899, 363, 12, 70, "RelatedLinks"]
}, Open  ]],
Cell[CellGroupData[{
Cell[821902, 14916, 319, 9, 70, "MoreAboutSection",
 CellID->403692888],
Cell[822224, 14927, 136, 3, 70, "MoreAbout",
 CellID->300660481],
Cell[822363, 14932, 147, 3, 70, "MoreAbout",
 CellID->203492113]
}, Open  ]],
Cell[822525, 14938, 50, 0, 70, "History"],
Cell[822578, 14940, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

