(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     66664,       2349]
NotebookOptionsPosition[     56426,       1998]
NotebookOutlinePosition[     57969,       2039]
CellTagsIndexPosition[     57883,       2034]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Vector Operations" :> 
          Documentation`HelpLookup["paclet:tutorial/VectorOperations"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Vector Operations\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Projection" :> 
          Documentation`HelpLookup["paclet:ref/Projection"], "Normalize" :> 
          Documentation`HelpLookup["paclet:ref/Normalize"], "Cross" :> 
          Documentation`HelpLookup["paclet:ref/Cross"], "Dot" :> 
          Documentation`HelpLookup["paclet:ref/Dot"], "Inner" :> 
          Documentation`HelpLookup["paclet:ref/Inner"], "QRDecomposition" :> 
          Documentation`HelpLookup["paclet:ref/QRDecomposition"], 
          "LinearSolve" :> 
          Documentation`HelpLookup["paclet:ref/LinearSolve"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Projection\"\>", 
       2->"\<\"Normalize\"\>", 3->"\<\"Cross\"\>", 4->"\<\"Dot\"\>", 
       5->"\<\"Inner\"\>", 6->"\<\"QRDecomposition\"\>", 
       7->"\<\"LinearSolve\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Calculus" :> Documentation`HelpLookup["paclet:guide/Calculus"],
           "Matrices and Linear Algebra" :> 
          Documentation`HelpLookup["paclet:guide/MatricesAndLinearAlgebra"], 
          "Operations on Vectors" :> 
          Documentation`HelpLookup["paclet:guide/OperationsOnVectors"], 
          "New in 6.0: Symbolic Computation" :> 
          Documentation`HelpLookup["paclet:guide/NewIn60AlgebraicComputing"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"], 
          "New in 6.0: Matrix & Linear Algebra Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MatrixAndLinearAlgebraFunctions"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Calculus\"\>", 
       2->"\<\"Matrices and Linear Algebra\"\>", 
       3->"\<\"Operations on Vectors\"\>", 
       4->"\<\"New in 6.0: Symbolic Computation\"\>", 
       5->"\<\"New in 6.0: Mathematics & Algorithms\"\>", 
       6->"\<\"New in 6.0: Matrix & Linear Algebra Functions\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["Orthogonalize", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["Orthogonalize",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Orthogonalize"], "[", 
       RowBox[{"{", 
        RowBox[{
         SubscriptBox[
          StyleBox["v", "TI"], 
          StyleBox["1", "TR"]], ",", 
         SubscriptBox[
          StyleBox["v", "TI"], 
          StyleBox["2", "TR"]], ",", 
         StyleBox["\[Ellipsis]", "TR"]}], "}"}], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives an orthonormal basis found by orthogonalizing \
the vectors ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["v", "TI"], 
       StyleBox["i", "TI"]]], "InlineFormula"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["Orthogonalize",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Orthogonalize"], "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          SubscriptBox[
           StyleBox["e", "TI"], 
           StyleBox["1", "TR"]], ",", 
          SubscriptBox[
           StyleBox["e", "TI"], 
           StyleBox["2", "TR"]], ",", 
          StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
        StyleBox["f", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives a basis for the ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["e", "TI"], 
       StyleBox["i", "TI"]]], "InlineFormula"],
     " orthonormal with respect to the inner product function ",
     Cell[BoxData[
      StyleBox["f", "TI"]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["Orthogonalize",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Orthogonalize"], "[", 
   RowBox[{"{", 
    RowBox[{
     SubscriptBox[
      StyleBox["v", "TI"], 
      StyleBox["1", "TR"]], ",", 
     SubscriptBox[
      StyleBox["v", "TI"], 
      StyleBox["2", "TR"]], ",", 
     StyleBox["\[Ellipsis]", "TR"]}], "}"}], "]"}]], "InlineFormula"],
 " uses the ordinary scalar product ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    OverscriptBox[
     SubscriptBox["v", "1"], "_"], ".", 
    SubscriptBox["v", "2"]}], TraditionalForm]], "InlineMath"],
 " as an inner product."
}], "Notes",
 CellID->1067943069],

Cell[TextData[{
 "The output from ",
 Cell[BoxData[
  ButtonBox["Orthogonalize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Orthogonalize"]], "InlineFormula"],
 " always contains the same number of vectors as the input. If some of the \
input vectors are not linearly independent, the output will contain zero \
vectors."
}], "Notes",
 CellID->46203962],

Cell["\<\
All nonzero vectors in the output are normalized to unit length.\
\>", "Notes",
 CellID->114100630],

Cell[TextData[{
 "The inner product function ",
 Cell[BoxData[
  StyleBox["f", "TI"]], "InlineFormula"],
 " is applied to pairs of linear combinations of the ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["e", "TI"], 
   StyleBox["i", "TI"]]], "InlineFormula"],
 "."
}], "Notes",
 CellID->6068880],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["e", "TI"], 
   StyleBox["i", "TI"]]], "InlineFormula"],
 " can be any expressions for which ",
 Cell[BoxData[
  StyleBox["f", "TI"]], "InlineFormula"],
 " always yields real results."
}], "Notes",
 CellID->150878744],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["Orthogonalize",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Orthogonalize"], "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      SubscriptBox[
       StyleBox["v", "TI"], 
       StyleBox["1", "TR"]], ",", 
      SubscriptBox[
       StyleBox["v", "TI"], 
       StyleBox["2", "TR"]], ",", 
      StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
    ButtonBox["Dot",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Dot"]}], "]"}]], "InlineFormula"],
 " effectively assumes that all elements of the ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["v", "TI"], 
   StyleBox["i", "TI"]]], "InlineFormula"],
 " are real."
}], "Notes",
 CellID->226788037],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Orthogonalize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Orthogonalize"]], "InlineFormula"],
 " by default generates a Gram-Schmidt basis."
}], "Notes",
 CellID->42557509],

Cell[TextData[{
 "Other bases can be obtained by giving alternative settings for the ",
 Cell[BoxData[
  ButtonBox["Method",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Method"]], "InlineFormula"],
 " option. Possible settings include: ",
 Cell[BoxData["\"\<GramSchmidt\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<ModifiedGramSchmidt\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<Reorthogonalization\>\""], "InlineFormula"],
 " and ",
 Cell[BoxData["\"\<Householder\>\""], "InlineFormula"],
 "."
}], "Notes",
 CellID->644381659],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["Orthogonalize",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Orthogonalize"], "[", 
   RowBox[{
    StyleBox["list", "TI"], ",", 
    RowBox[{
     ButtonBox["Tolerance",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Tolerance"], "->", 
     StyleBox["t", "TI"]}]}], "]"}]], "InlineFormula"],
 " sets to zero elements whose relative norm falls below ",
 Cell[BoxData[
  StyleBox["t", "TI"]], "InlineFormula"],
 "."
}], "Notes",
 CellID->10340163]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->42362211],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(1)", "ExampleCount"]
}], "ExampleSection",
 CellID->174169758],

Cell["Find an orthonormal basis for two 3D vectors:", "ExampleText",
 CellID->194959924],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Orthogonalize", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "1", ",", "1"}], "}"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->395964526],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     FractionBox["1", 
      SqrtBox["2"]], ",", "0", ",", 
     FractionBox["1", 
      SqrtBox["2"]]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "0"}], "}"}]}], "}"}]], "Output",
 ImageSize->{186, 34},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->557960280]
}, Open  ]],

Cell["\<\
Find the coefficients of a general vector with respect to this basis:\
\>", "ExampleText",
 CellID->248283644],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"%", " ", ".", 
  RowBox[{"{", 
   RowBox[{"x", ",", "y", ",", "z"}], "}"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->603557343],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    FractionBox["x", 
     SqrtBox["2"]], "+", 
    FractionBox["z", 
     SqrtBox["2"]]}], ",", "y"}], "}"}]], "Output",
 ImageSize->{105, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->72009712]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->274073923],

Cell["Use exact arithmetic to find an orthonormal basis:", "ExampleText",
 CellID->407300717],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "1", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", 
       RowBox[{"1", "/", "2"}], ",", 
       RowBox[{"1", "/", "3"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "2", ",", "3"}], "}"}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->151485997],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"u", "=", 
  RowBox[{"Orthogonalize", "[", "m", "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->256789355],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     FractionBox["1", 
      SqrtBox["3"]], ",", 
     FractionBox["1", 
      SqrtBox["3"]], ",", 
     FractionBox["1", 
      SqrtBox["3"]]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     FractionBox["7", 
      SqrtBox["78"]], ",", 
     RowBox[{"-", 
      SqrtBox[
       FractionBox["2", "39"]]}], ",", 
     RowBox[{"-", 
      FractionBox["5", 
       SqrtBox["78"]]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     FractionBox["1", 
      SqrtBox["26"]], ",", 
     RowBox[{
      RowBox[{"-", "2"}], " ", 
      SqrtBox[
       FractionBox["2", "13"]]}], ",", 
     FractionBox["3", 
      SqrtBox["26"]]}], "}"}]}], "}"}]], "Output",
 ImageSize->{483, 41},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->140063395]
}, Open  ]],

Cell["Use machine arithmetic:", "ExampleText",
 CellID->259368292],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Orthogonalize", "[", 
  RowBox[{"N", "[", "m", "]"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->28854865],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "0.5773502691896258`", ",", "0.5773502691896258`", ",", 
     "0.5773502691896258`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.7925939239012164`", ",", 
     RowBox[{"-", "0.22645540682891968`"}], ",", 
     RowBox[{"-", "0.5661385170722983`"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.1961161351381805`", ",", 
     RowBox[{"-", "0.7844645405527391`"}], ",", "0.5883484054145494`"}], 
    "}"}]}], "}"}]], "Output",
 ImageSize->{461, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->181900957]
}, Open  ]],

Cell["\<\
Use 25\[Hyphen]digit precision arithmetic:\
\>", "ExampleText",
 CellID->298298005],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Orthogonalize", "[", 
  RowBox[{"N", "[", 
   RowBox[{"m", ",", "25"}], "]"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->272541],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "0.57735026918962576450914878050195745565`25.", ",", 
     "0.57735026918962576450914878050195745565`25.", ",", 
     "0.57735026918962576450914878050195745565`25."}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.7925939239012170254275422960091638173`25.", ",", 
     RowBox[{"-", "0.2264554068289191501221549417169039478`25."}], ",", 
     RowBox[{"-", "0.5661385170722978753053873542922598695`25."}]}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"0.19611613513818403192416246573164546114`25.", ",", 
     RowBox[{"-", "0.78446454055273612769664986292658184454`25."}], ",", 
     "0.58834840541455209577248739719493638342`25."}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{436, 84},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->461567044]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->15707969],

Cell["Orthogonalize complex vectors:", "ExampleText",
 CellID->219177784],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"u", "=", 
  RowBox[{"Orthogonalize", "[", 
   RowBox[{"RandomComplex", "[", 
    RowBox[{
     RowBox[{"1", "+", "I"}], ",", 
     RowBox[{"{", 
      RowBox[{"4", ",", "4"}], "}"}]}], "]"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->245640227],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"0.23721651043828343`", "\[InvisibleSpace]", "+", 
      RowBox[{"0.18469407985333283`", " ", "\[ImaginaryI]"}]}], ",", 
     RowBox[{"0.4066354231310899`", "\[InvisibleSpace]", "+", 
      RowBox[{"0.33742856222533013`", " ", "\[ImaginaryI]"}]}], ",", 
     RowBox[{"0.01708044866920961`", "\[InvisibleSpace]", "+", 
      RowBox[{"0.5254366338658611`", " ", "\[ImaginaryI]"}]}], ",", 
     RowBox[{"0.5632709313281585`", "\[InvisibleSpace]", "+", 
      RowBox[{"0.19171980095399863`", " ", "\[ImaginaryI]"}]}]}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{
     RowBox[{"0.47075681097687494`", "\[InvisibleSpace]", "+", 
      RowBox[{"0.2617675383701195`", " ", "\[ImaginaryI]"}]}], ",", 
     RowBox[{
      RowBox[{"-", "0.06610518163730586`"}], "+", 
      RowBox[{"0.08726059501060207`", " ", "\[ImaginaryI]"}]}], ",", 
     RowBox[{"0.6205203359895175`", "\[InvisibleSpace]", "-", 
      RowBox[{"0.0342348974801783`", " ", "\[ImaginaryI]"}]}], ",", 
     RowBox[{
      RowBox[{"-", "0.4059578079429441`"}], "+", 
      RowBox[{"0.383226064982228`", " ", "\[ImaginaryI]"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"-", "0.15819857603718965`"}], "+", 
      RowBox[{"0.20916922257858345`", " ", "\[ImaginaryI]"}]}], ",", 
     RowBox[{"0.4939290766414861`", "\[InvisibleSpace]", "+", 
      RowBox[{"0.6004871716888996`", " ", "\[ImaginaryI]"}]}], ",", 
     RowBox[{"0.028490654810754232`", "\[InvisibleSpace]", "-", 
      RowBox[{"0.42867025001168985`", " ", "\[ImaginaryI]"}]}], ",", 
     RowBox[{
      RowBox[{"-", "0.2135055764734218`"}], "-", 
      RowBox[{"0.31067046674724824`", " ", "\[ImaginaryI]"}]}]}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{
     RowBox[{"0.32218133901818735`", "\[InvisibleSpace]", "-", 
      RowBox[{"0.6685079180318437`", " ", "\[ImaginaryI]"}]}], ",", 
     RowBox[{"0.04804501272500963`", "\[InvisibleSpace]", "+", 
      RowBox[{"0.31929013797179673`", " ", "\[ImaginaryI]"}]}], ",", 
     RowBox[{
      RowBox[{"-", "0.1755263105491085`"}], "+", 
      RowBox[{"0.34932463430358907`", " ", "\[ImaginaryI]"}]}], ",", 
     RowBox[{
      RowBox[{"-", "0.41830421623599034`"}], "-", 
      RowBox[{"0.13124873224628764`", " ", "\[ImaginaryI]"}]}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{491, 101},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->394260877]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1020263627],

Cell["Find a symbolic basis, assuming all variables are real:", "ExampleText",
 CellID->11099593],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Orthogonalize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"a", ",", "b"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "y"}], "}"}]}], "}"}], ",", "Dot"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->289639357],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     FractionBox["a", 
      SqrtBox[
       RowBox[{
        SuperscriptBox["a", "2"], "+", 
        SuperscriptBox["b", "2"]}]]], ",", 
     FractionBox["b", 
      SqrtBox[
       RowBox[{
        SuperscriptBox["a", "2"], "+", 
        SuperscriptBox["b", "2"]}]]]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     FractionBox[
      RowBox[{"x", "-", 
       FractionBox[
        RowBox[{"a", " ", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"a", " ", "x"}], "+", 
           RowBox[{"b", " ", "y"}]}], ")"}]}], 
        RowBox[{
         SuperscriptBox["a", "2"], "+", 
         SuperscriptBox["b", "2"]}]]}], 
      SqrtBox[
       RowBox[{
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{"x", "-", 
           FractionBox[
            RowBox[{"a", " ", 
             RowBox[{"(", 
              RowBox[{
               RowBox[{"a", " ", "x"}], "+", 
               RowBox[{"b", " ", "y"}]}], ")"}]}], 
            RowBox[{
             SuperscriptBox["a", "2"], "+", 
             SuperscriptBox["b", "2"]}]]}], ")"}], "2"], "+", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{"y", "-", 
           FractionBox[
            RowBox[{"b", " ", 
             RowBox[{"(", 
              RowBox[{
               RowBox[{"a", " ", "x"}], "+", 
               RowBox[{"b", " ", "y"}]}], ")"}]}], 
            RowBox[{
             SuperscriptBox["a", "2"], "+", 
             SuperscriptBox["b", "2"]}]]}], ")"}], "2"]}]]], ",", 
     FractionBox[
      RowBox[{"y", "-", 
       FractionBox[
        RowBox[{"b", " ", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"a", " ", "x"}], "+", 
           RowBox[{"b", " ", "y"}]}], ")"}]}], 
        RowBox[{
         SuperscriptBox["a", "2"], "+", 
         SuperscriptBox["b", "2"]}]]}], 
      SqrtBox[
       RowBox[{
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{"x", "-", 
           FractionBox[
            RowBox[{"a", " ", 
             RowBox[{"(", 
              RowBox[{
               RowBox[{"a", " ", "x"}], "+", 
               RowBox[{"b", " ", "y"}]}], ")"}]}], 
            RowBox[{
             SuperscriptBox["a", "2"], "+", 
             SuperscriptBox["b", "2"]}]]}], ")"}], "2"], "+", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{"y", "-", 
           FractionBox[
            RowBox[{"b", " ", 
             RowBox[{"(", 
              RowBox[{
               RowBox[{"a", " ", "x"}], "+", 
               RowBox[{"b", " ", "y"}]}], ")"}]}], 
            RowBox[{
             SuperscriptBox["a", "2"], "+", 
             SuperscriptBox["b", "2"]}]]}], ")"}], "2"]}]]]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{474, 110},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->748680665]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->295368580],

Cell["\<\
Orthogonalize symbolic expressions with a symbolic scalar product:\
\>", "ExampleText",
 CellID->598483379],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Orthogonalize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "x", ",", 
     RowBox[{"x", "^", "2"}], ",", 
     RowBox[{"x", "^", "3"}]}], "}"}], ",", 
   RowBox[{
    RowBox[{"Integrate", "[", 
     RowBox[{
      RowBox[{"#1", " ", "#2"}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", 
        RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "]"}], "&"}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->220798853],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["1", 
    SqrtBox["2"]], ",", 
   RowBox[{
    SqrtBox[
     FractionBox["3", "2"]], " ", "x"}], ",", 
   RowBox[{
    FractionBox["3", "2"], " ", 
    SqrtBox[
     FractionBox["5", "2"]], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", 
       FractionBox["1", "3"]}], "+", 
      SuperscriptBox["x", "2"]}], ")"}]}], ",", 
   RowBox[{
    FractionBox["5", "2"], " ", 
    SqrtBox[
     FractionBox["7", "2"]], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", 
       FractionBox[
        RowBox[{"3", " ", "x"}], "5"]}], "+", 
      SuperscriptBox["x", "3"]}], ")"}]}]}], "}"}]], "Output",
 ImageSize->{343, 41},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->264764501]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Options",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->2061341341],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Tolerance",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->1757724783],

Cell["\<\
Below the tolerance, two vectors are not recognized as linearly independent:\
\>", "ExampleText",
 CellID->95302197],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Orthogonalize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "1.*^-10"}], "}"}]}], "}"}], ",", 
   RowBox[{"Tolerance", "\[Rule]", 
    RowBox[{"10", "^", 
     RowBox[{"-", "5"}]}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->391104580],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1.`", ",", "0.`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.`", ",", "0.`"}], "}"}]}], "}"}]], "Output",
 ImageSize->{136, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1143497]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Orthogonalize", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1", ",", "0"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "1.*^-10"}], "}"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->243545844],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1.`", ",", "0.`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.`", ",", "1.`"}], "}"}]}], "}"}]], "Output",
 ImageSize->{136, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1744680]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Method",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->1295379749],

Cell[TextData[{
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 " forms a set of vectors that are nearly linearly dependent:"
}], "ExampleText",
 CellID->291190620],

Cell[BoxData[
 RowBox[{
  RowBox[{"n", "=", "10"}], ";", 
  RowBox[{"m", "=", 
   RowBox[{"N", "[", 
    RowBox[{"HilbertMatrix", "[", "n", "]"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->590618219],

Cell["Deviation from orthonormality for the default method:", "ExampleText",
 CellID->126067530],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"gs", "=", 
   RowBox[{"Orthogonalize", "[", "m", "]"}]}], ";", 
  RowBox[{"Max", "[", 
   RowBox[{"Abs", "[", 
    RowBox[{
     RowBox[{"gs", ".", 
      RowBox[{"Transpose", "[", "gs", "]"}]}], "-", 
     RowBox[{"IdentityMatrix", "[", "n", "]"}]}], "]"}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->588840553],

Cell[BoxData["0.0004095897552118706`"], "Output",
 ImageSize->{74, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->150429528]
}, Open  ]],

Cell["Deviation for all of the methods:", "ExampleText",
 CellID->187348485],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"methods", "=", 
   RowBox[{"{", 
    RowBox[{
    "\"\<GramSchmidt\>\"", ",", "\"\<ModifiedGramSchmidt\>\"", ",", 
     "\"\<Householder\>\"", ",", "\"\<Reorthogonalization\>\""}], "}"}]}], 
  ";"}], "\n", 
 RowBox[{"TableForm", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"method", ",", 
      RowBox[{
       RowBox[{"gs", "=", 
        RowBox[{"Orthogonalize", "[", 
         RowBox[{"m", ",", 
          RowBox[{"Method", "\[Rule]", "method"}]}], "]"}]}], ";", 
       RowBox[{"Max", "[", 
        RowBox[{"Abs", "[", 
         RowBox[{
          RowBox[{"gs", ".", 
           RowBox[{"Transpose", "[", "gs", "]"}]}], "-", 
          RowBox[{"IdentityMatrix", "[", "n", "]"}]}], "]"}], "]"}]}]}], 
     "}"}], ",", "\[IndentingNewLine]", 
    RowBox[{"{", 
     RowBox[{"method", ",", "methods"}], "}"}]}], "]"}], "]"}]}], "Input",
 CellLabel->"In[3]:=",
 CellID->5675005],

Cell[BoxData[
 TagBox[GridBox[{
    {"\<\"GramSchmidt\"\>", "0.9999812543144961`"},
    {"\<\"ModifiedGramSchmidt\"\>", "0.0004095897552118706`"},
    {"\<\"Householder\"\>", "4.440892098500626`*^-16"},
    {"\<\"Reorthogonalization\"\>", "2.220446049250313`*^-16"}
   },
   GridBoxAlignment->{
    "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
     "RowsIndexed" -> {}},
   GridBoxSpacings->{"Columns" -> {
       Offset[0.27999999999999997`], {
        Offset[0.7]}, 
       Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
       Offset[0.2], {
        Offset[0.4]}, 
       Offset[0.2]}, "RowsIndexed" -> {}}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 ImageSize->{235, 63},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]//TableForm=",
 CellID->107933660]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->530742545],

Cell["\<\
For a large numerical matrix, the Householder method is usually fastest:\
\>", "ExampleText",
 CellID->664258031],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", 
     RowBox[{"{", 
      RowBox[{"1000", ",", "1000"}], "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->335241345],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"methods", "=", 
   RowBox[{"{", 
    RowBox[{
    "\"\<GramSchmidt\>\"", ",", "\"\<ModifiedGramSchmidt\>\"", ",", 
     "\"\<Householder\>\"", ",", "\"\<Reorthogonalization\>\""}], "}"}]}], 
  ";"}], "\n", 
 RowBox[{"TableForm", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"method", ",", 
      RowBox[{"First", "[", 
       RowBox[{"Timing", "[", 
        RowBox[{"Orthogonalize", "[", 
         RowBox[{"m", ",", 
          RowBox[{"Method", "->", "method"}]}], "]"}], "]"}], "]"}]}], "}"}], 
    ",", "\[IndentingNewLine]", 
    RowBox[{"{", 
     RowBox[{"method", ",", "methods"}], "}"}]}], "]"}], "]"}]}], "Input",
 CellLabel->"In[2]:=",
 CellID->201751166],

Cell[BoxData[
 TagBox[GridBox[{
    {"\<\"GramSchmidt\"\>", "1.4219999999999997`"},
    {"\<\"ModifiedGramSchmidt\"\>", "1.3119999999999992`"},
    {"\<\"Householder\"\>", "0.9849999999999998`"},
    {"\<\"Reorthogonalization\"\>", "2.6250000000000004`"}
   },
   GridBoxAlignment->{
    "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
     "RowsIndexed" -> {}},
   GridBoxSpacings->{"Columns" -> {
       Offset[0.27999999999999997`], {
        Offset[0.7]}, 
       Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
       Offset[0.2], {
        Offset[0.4]}, 
       Offset[0.2]}, "RowsIndexed" -> {}}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 ImageSize->{180, 57},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]//TableForm=",
 CellID->974156184]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->258228157],

Cell[TextData[{
 "Derive normalized Legendre polynomials by orthogonalizing powers of ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->369412078],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Orthogonalize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "x", ",", 
     RowBox[{"x", "^", "2"}], ",", 
     RowBox[{"x", "^", "3"}]}], "}"}], ",", 
   RowBox[{
    RowBox[{"Integrate", "[", 
     RowBox[{
      RowBox[{"#1", " ", "#2"}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", 
        RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "]"}], "&"}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->110850738],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["1", 
    SqrtBox["2"]], ",", 
   RowBox[{
    SqrtBox[
     FractionBox["3", "2"]], " ", "x"}], ",", 
   RowBox[{
    FractionBox["3", "2"], " ", 
    SqrtBox[
     FractionBox["5", "2"]], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", 
       FractionBox["1", "3"]}], "+", 
      SuperscriptBox["x", "2"]}], ")"}]}], ",", 
   RowBox[{
    FractionBox["5", "2"], " ", 
    SqrtBox[
     FractionBox["7", "2"]], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", 
       FractionBox[
        RowBox[{"3", " ", "x"}], "5"]}], "+", 
      SuperscriptBox["x", "3"]}], ")"}]}]}], "}"}]], "Output",
 ImageSize->{343, 41},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->258849475]
}, Open  ]],

Cell["Derive normalized Hermite polynomials:", "ExampleText",
 CellID->214651176],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Orthogonalize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "x", ",", 
     RowBox[{"x", "^", "2"}], ",", 
     RowBox[{"x", "^", "3"}]}], "}"}], ",", 
   RowBox[{
    RowBox[{"Integrate", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"Exp", "[", 
        RowBox[{"-", 
         RowBox[{"x", "^", "2"}]}], "]"}], " ", "#1", " ", "#2"}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", 
        RowBox[{"-", "\[Infinity]"}], ",", "\[Infinity]"}], "}"}]}], "]"}], 
    "&"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->15261190],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["1", 
    SuperscriptBox["\[Pi]", 
     RowBox[{"1", "/", "4"}]]], ",", 
   FractionBox[
    RowBox[{
     SqrtBox["2"], " ", "x"}], 
    SuperscriptBox["\[Pi]", 
     RowBox[{"1", "/", "4"}]]], ",", 
   FractionBox[
    RowBox[{
     SqrtBox["2"], " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", 
        FractionBox["1", "2"]}], "+", 
       SuperscriptBox["x", "2"]}], ")"}]}], 
    SuperscriptBox["\[Pi]", 
     RowBox[{"1", "/", "4"}]]], ",", 
   FractionBox[
    RowBox[{"2", " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", 
        FractionBox[
         RowBox[{"3", " ", "x"}], "2"]}], "+", 
       SuperscriptBox["x", "3"]}], ")"}]}], 
    RowBox[{
     SqrtBox["3"], " ", 
     SuperscriptBox["\[Pi]", 
      RowBox[{"1", "/", "4"}]]}]]}], "}"}]], "Output",
 ImageSize->{283, 44},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->585273161]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(6)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->2123667759],

Cell[TextData[{
 "In ",
 Cell[BoxData[
  FormBox["d", TraditionalForm]], "InlineMath"],
 " dimensions, there can be at most ",
 Cell[BoxData[
  FormBox["d", TraditionalForm]], "InlineMath"],
 " elements in the orthonormal basis:"
}], "ExampleText",
 CellID->565470712],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Orthogonalize", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1", ",", "2"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"3", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"6", ",", "9"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"7", ",", "8"}], "}"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->175634434],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     FractionBox["1", 
      SqrtBox["5"]], ",", 
     FractionBox["2", 
      SqrtBox["5"]]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     FractionBox["2", 
      SqrtBox["5"]], ",", 
     RowBox[{"-", 
      FractionBox["1", 
       SqrtBox["5"]]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0"}], "}"}]}], "}"}]], "Output",
 ImageSize->{300, 34},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->649550955]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->370862024],

Cell[TextData[{
 "Most sets of random ",
 Cell[BoxData[
  FormBox["d", TraditionalForm]], "InlineMath"],
 "-dimensional vectors are spanned by exactly ",
 Cell[BoxData[
  FormBox["d", TraditionalForm]], "InlineMath"],
 " basis vectors:"
}], "ExampleText",
 CellID->69631862],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Orthogonalize", "[", 
  RowBox[{"RandomReal", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"10", ",", "3"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->263507827],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "0.8387797547536097`"}], ",", 
     RowBox[{"-", "0.4779316835354596`"}], ",", "0.2608252842199847`"}], 
    "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.215819809237878`", ",", 
     RowBox[{"-", "0.7316499171295082`"}], ",", 
     RowBox[{"-", "0.6466144204275913`"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.4998703161380964`", ",", 
     RowBox[{"-", "0.48607582190163`"}], ",", "0.7168402628247483`"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"0.`", ",", "0.`", ",", "0.`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.`", ",", "0.`", ",", "0.`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.`", ",", "0.`", ",", "0.`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.`", ",", "0.`", ",", "0.`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.`", ",", "0.`", ",", "0.`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.`", ",", "0.`", ",", "0.`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.`", ",", "0.`", ",", "0.`"}], "}"}]}], "}"}]], "Output",
 ImageSize->{455, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->183093870]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->54768347],

Cell["\<\
With the default method, the first element of the basis is always a multiple \
of the first vector: \
\>", "ExampleText",
 CellID->15704052],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Orthogonalize", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"2", ",", "3"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"2", ",", "7"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"4", ",", "5"}], "}"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->49524972],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     FractionBox["2", 
      SqrtBox["13"]], ",", 
     FractionBox["3", 
      SqrtBox["13"]]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", 
      FractionBox["3", 
       SqrtBox["13"]]}], ",", 
     FractionBox["2", 
      SqrtBox["13"]]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0"}], "}"}]}], "}"}]], "Output",
 ImageSize->{278, 34},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->236853727]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->394678611],

Cell["\<\
For linearly independent vectors, the result is an orthonormal set:\
\>", "ExampleText",
 CellID->293590904],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"u", "=", 
  RowBox[{"Orthogonalize", "[", 
   RowBox[{"RandomComplex", "[", 
    RowBox[{"1", ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "5"}], "}"}]}], "]"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->41764138],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"0.13402255397395663`", "\[InvisibleSpace]", "+", 
      RowBox[{"0.`", " ", "\[ImaginaryI]"}]}], ",", 
     RowBox[{"0.34415482392808255`", "\[InvisibleSpace]", "+", 
      RowBox[{"0.`", " ", "\[ImaginaryI]"}]}], ",", 
     RowBox[{"0.5952205673822926`", "\[InvisibleSpace]", "+", 
      RowBox[{"0.`", " ", "\[ImaginaryI]"}]}], ",", 
     RowBox[{"0.7100294446707283`", "\[InvisibleSpace]", "+", 
      RowBox[{"0.`", " ", "\[ImaginaryI]"}]}], ",", 
     RowBox[{"0.07187542040925612`", "\[InvisibleSpace]", "+", 
      RowBox[{"0.`", " ", "\[ImaginaryI]"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"0.8621408727364672`", "\[InvisibleSpace]", "+", 
      RowBox[{"0.`", " ", "\[ImaginaryI]"}]}], ",", 
     RowBox[{"0.20305959742311497`", "\[InvisibleSpace]", "+", 
      RowBox[{"0.`", " ", "\[ImaginaryI]"}]}], ",", 
     RowBox[{"0.01196924401904816`", "\[InvisibleSpace]", "+", 
      RowBox[{"0.`", " ", "\[ImaginaryI]"}]}], ",", 
     RowBox[{
      RowBox[{"-", "0.3064653348069827`"}], "+", 
      RowBox[{"0.`", " ", "\[ImaginaryI]"}]}], ",", 
     RowBox[{"0.3484474870203854`", "\[InvisibleSpace]", "+", 
      RowBox[{"0.`", " ", "\[ImaginaryI]"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"0.0158829190333883`", "\[InvisibleSpace]", "+", 
      RowBox[{"0.`", " ", "\[ImaginaryI]"}]}], ",", 
     RowBox[{
      RowBox[{"-", "0.7514335298173177`"}], "+", 
      RowBox[{"0.`", " ", "\[ImaginaryI]"}]}], ",", 
     RowBox[{"0.06791907814938743`", "\[InvisibleSpace]", "+", 
      RowBox[{"0.`", " ", "\[ImaginaryI]"}]}], ",", 
     RowBox[{"0.24257749058713934`", "\[InvisibleSpace]", "+", 
      RowBox[{"0.`", " ", "\[ImaginaryI]"}]}], ",", 
     RowBox[{"0.6096216392427756`", "\[InvisibleSpace]", "+", 
      RowBox[{"0.`", " ", "\[ImaginaryI]"}]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{567, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->183531080]
}, Open  ]],

Cell["Verify using matrix multiplication:", "ExampleText",
 CellID->12179607],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Chop", "[", 
  RowBox[{"u", ".", 
   RowBox[{"ConjugateTranspose", "[", "u", "]"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->266322687],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1.`", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1.`", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1.`"}], "}"}]}], "}"}]], "Output",
 ImageSize->{234, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->405091768]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->698648664],

Cell["\<\
For linearly independent vectors, the result is a set orthonormal with the \
given inner product:\
\>", "ExampleText",
 CellID->805989730],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"u", "=", 
  RowBox[{"Orthogonalize", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1", ",", "x", ",", 
      RowBox[{"x", "^", "2"}], ",", 
      RowBox[{"x", "^", "3"}]}], "}"}], ",", 
    RowBox[{
     RowBox[{"Integrate", "[", 
      RowBox[{
       RowBox[{"Times", "[", "##", "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", 
         RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "]"}], "&"}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->53028191],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["1", 
    SqrtBox["2"]], ",", 
   RowBox[{
    SqrtBox[
     FractionBox["3", "2"]], " ", "x"}], ",", 
   RowBox[{
    FractionBox["3", "2"], " ", 
    SqrtBox[
     FractionBox["5", "2"]], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", 
       FractionBox["1", "3"]}], "+", 
      SuperscriptBox["x", "2"]}], ")"}]}], ",", 
   RowBox[{
    FractionBox["5", "2"], " ", 
    SqrtBox[
     FractionBox["7", "2"]], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", 
       FractionBox[
        RowBox[{"3", " ", "x"}], "5"]}], "+", 
      SuperscriptBox["x", "3"]}], ")"}]}]}], "}"}]], "Output",
 ImageSize->{343, 41},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->561095229]
}, Open  ]],

Cell["Verify orthonormality:", "ExampleText",
 CellID->27311458],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Outer", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Integrate", "[", 
     RowBox[{
      RowBox[{"Times", "[", "##", "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", 
        RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "]"}], "&"}], ",", "u", ",", 
   "u"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->551996848],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0", ",", "1"}], "}"}]}], "}"}]], "Output",
 ImageSize->{354, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->279673394]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->738094],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["Orthogonalize",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Orthogonalize"], "[", 
   StyleBox["m", "TI"], "]"}]], "InlineFormula"],
 " is related to ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["QRDecomposition",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/QRDecomposition"], "[", 
   RowBox[{
    ButtonBox["Transpose",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Transpose"], "[", 
    StyleBox["m", "TI"], "]"}], "]"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->822074023],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "3"}], "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->639586099],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"q1", "=", 
   RowBox[{"Orthogonalize", "[", "m", "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"r1", "=", 
   RowBox[{"q1", ".", 
    RowBox[{"Transpose", "[", "m", "]"}]}]}], ";"}], "\n", 
 RowBox[{"{", 
  RowBox[{
   RowBox[{"MatrixForm", "[", "q1", "]"}], ",", 
   RowBox[{"MatrixForm", "[", 
    RowBox[{"Chop", "[", "r1", "]"}], "]"}]}], "}"}]}], "Input",
 CellLabel->"In[2]:=",
 CellID->350486289],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {"0.37131540186913464`", "0.05224779243783977`", 
        "0.9270356198766774`"},
       {"0.48455789243047653`", "0.8407707194801761`", 
        RowBox[{"-", "0.24147100477722255`"}]},
       {
        RowBox[{"-", "0.7920407320448215`"}], "0.5388643293540228`", 
        "0.2868740373957575`"}
      },
      GridBoxAlignment->{
       "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
        "RowsIndexed" -> {}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]], ",", 
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {"0.9776216326739131`", "1.0351792773778805`", 
        "0.28066926183060803`"},
       {"0", "0.7569439926558588`", "0.871766872314077`"},
       {"0", "0", "0.16878823861816467`"}
      },
      GridBoxAlignment->{
       "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
        "RowsIndexed" -> {}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]]}], "}"}]], "Output",
 ImageSize->{460, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->38685493]
}, Open  ]],

Cell["They are the same up to sign:", "ExampleText",
 CellID->471826964],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"q", ",", "r"}], "}"}], "=", 
   RowBox[{"QRDecomposition", "[", 
    RowBox[{"Transpose", "[", "m", "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{"{", 
  RowBox[{
   RowBox[{"MatrixForm", "[", "q", "]"}], ",", 
   RowBox[{"MatrixForm", "[", "r", "]"}]}], "}"}]}], "Input",
 CellLabel->"In[3]:=",
 CellID->335234509],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {
        RowBox[{"-", "0.3713154018691347`"}], 
        RowBox[{"-", "0.05224779243783977`"}], 
        RowBox[{"-", "0.9270356198766774`"}]},
       {
        RowBox[{"-", "0.48455789243047653`"}], 
        RowBox[{"-", "0.8407707194801761`"}], "0.2414710047772227`"},
       {
        RowBox[{"-", "0.7920407320448222`"}], "0.5388643293540213`", 
        "0.2868740373957585`"}
      },
      GridBoxAlignment->{
       "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
        "RowsIndexed" -> {}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]], ",", 
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {
        RowBox[{"-", "0.9776216326739132`"}], 
        RowBox[{"-", "1.0351792773778807`"}], 
        RowBox[{"-", "0.28066926183060803`"}]},
       {"0.`", 
        RowBox[{"-", "0.7569439926558587`"}], 
        RowBox[{"-", "0.871766872314077`"}]},
       {"0.`", "0.`", "0.1687882386181633`"}
      },
      GridBoxAlignment->{
       "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
        "RowsIndexed" -> {}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]]}], "}"}]], "Output",
 ImageSize->{492, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->132766]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["Projection",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Projection"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Normalize",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Normalize"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Cross",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Cross"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Dot",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Dot"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Inner",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Inner"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["QRDecomposition",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/QRDecomposition"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["LinearSolve",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/LinearSolve"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->250839057],

Cell[TextData[ButtonBox["Vector Operations",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/VectorOperations"]], "Tutorials",
 CellID->513408734]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->43178843],

Cell[TextData[ButtonBox["Calculus",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/Calculus"]], "MoreAbout",
 CellID->519431487],

Cell[TextData[ButtonBox["Matrices and Linear Algebra",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/MatricesAndLinearAlgebra"]], "MoreAbout",
 CellID->626062243],

Cell[TextData[ButtonBox["Operations on Vectors",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/OperationsOnVectors"]], "MoreAbout",
 CellID->482558],

Cell[TextData[ButtonBox["New in 6.0: Symbolic Computation",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60AlgebraicComputing"]], "MoreAbout",
 CellID->219016046],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->547157338],

Cell[TextData[ButtonBox["New in 6.0: Matrix & Linear Algebra Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/NewIn60MatrixAndLinearAlgebraFunctions"]], "MoreAbout",
 CellID->283719392]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Orthogonalize - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 5, 26.5654144}", "context" -> "System`", 
    "keywords" -> {
     "Gram-Schmidt", "modified Gram-Schmidt", "orthogonalization", 
      "orthonormal basis", "orthonormal matrix", "unitary matrix", 
      "reorthogonalization", "Householder"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "Orthogonalize[{v_1, v_2, ...}] gives an orthonormal basis found by \
orthogonalizing the vectors v_i. Orthogonalize[{e_1, e_2, ...}, f] gives a \
basis for the e_i orthonormal with respect to the inner product function f.", 
    "synonyms" -> {}, "title" -> "Orthogonalize", "type" -> "Symbol", "uri" -> 
    "ref/Orthogonalize"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[11487, 362, 359, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->42362211]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 57740, 2027}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 3242, 71, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3865, 98, 1578, 37, 70, "ObjectNameGrid"],
Cell[5446, 137, 1747, 56, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[7230, 198, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[7708, 215, 668, 24, 70, "Notes",
 CellID->1067943069],
Cell[8379, 241, 359, 10, 70, "Notes",
 CellID->46203962],
Cell[8741, 253, 109, 3, 70, "Notes",
 CellID->114100630],
Cell[8853, 258, 296, 11, 70, "Notes",
 CellID->6068880],
Cell[9152, 271, 285, 11, 70, "Notes",
 CellID->150878744],
Cell[9440, 284, 716, 26, 70, "Notes",
 CellID->226788037],
Cell[10159, 312, 223, 7, 70, "Notes",
 CellID->42557509],
Cell[10385, 321, 545, 16, 70, "Notes",
 CellID->644381659],
Cell[10933, 339, 517, 18, 70, "Notes",
 CellID->10340163]
}, Closed]],
Cell[CellGroupData[{
Cell[11487, 362, 359, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->42362211],
Cell[CellGroupData[{
Cell[11871, 376, 148, 5, 70, "ExampleSection",
 CellID->174169758],
Cell[12022, 383, 88, 1, 70, "ExampleText",
 CellID->194959924],
Cell[CellGroupData[{
Cell[12135, 388, 278, 9, 28, "Input",
 CellID->395964526],
Cell[12416, 399, 402, 15, 55, "Output",
 CellID->557960280]
}, Open  ]],
Cell[12833, 417, 120, 3, 70, "ExampleText",
 CellID->248283644],
Cell[CellGroupData[{
Cell[12978, 424, 154, 5, 28, "Input",
 CellID->603557343],
Cell[13135, 431, 303, 12, 53, "Output",
 CellID->72009712]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[13487, 449, 220, 7, 70, "ExampleSection",
 CellID->274073923],
Cell[13710, 458, 93, 1, 70, "ExampleText",
 CellID->407300717],
Cell[13806, 461, 414, 14, 70, "Input",
 CellID->151485997],
Cell[CellGroupData[{
Cell[14245, 479, 132, 4, 70, "Input",
 CellID->256789355],
Cell[14380, 485, 853, 35, 62, "Output",
 CellID->140063395]
}, Open  ]],
Cell[15248, 523, 66, 1, 70, "ExampleText",
 CellID->259368292],
Cell[CellGroupData[{
Cell[15339, 528, 136, 4, 70, "Input",
 CellID->28854865],
Cell[15478, 534, 633, 19, 54, "Output",
 CellID->181900957]
}, Open  ]],
Cell[16126, 556, 93, 3, 70, "ExampleText",
 CellID->298298005],
Cell[CellGroupData[{
Cell[16244, 563, 159, 5, 70, "Input",
 CellID->272541],
Cell[16406, 570, 868, 22, 105, "Output",
 CellID->461567044]
}, Open  ]],
Cell[17289, 595, 124, 3, 70, "ExampleDelimiter",
 CellID->15707969],
Cell[17416, 600, 73, 1, 70, "ExampleText",
 CellID->219177784],
Cell[CellGroupData[{
Cell[17514, 605, 277, 9, 70, "Input",
 CellID->245640227],
Cell[17794, 616, 2489, 56, 122, "Output",
 CellID->394260877]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[20332, 678, 244, 7, 70, "ExampleSection",
 CellID->1020263627],
Cell[20579, 687, 97, 1, 70, "ExampleText",
 CellID->11099593],
Cell[CellGroupData[{
Cell[20701, 692, 289, 10, 70, "Input",
 CellID->289639357],
Cell[20993, 704, 2886, 96, 131, "Output",
 CellID->748680665]
}, Open  ]],
Cell[23894, 803, 125, 3, 70, "ExampleDelimiter",
 CellID->295368580],
Cell[24022, 808, 117, 3, 70, "ExampleText",
 CellID->598483379],
Cell[CellGroupData[{
Cell[24164, 815, 457, 16, 70, "Input",
 CellID->220798853],
Cell[24624, 833, 795, 31, 62, "Output",
 CellID->264764501]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[25468, 870, 223, 7, 70, "ExampleSection",
 CellID->2061341341],
Cell[CellGroupData[{
Cell[25716, 881, 231, 7, 70, "ExampleSubsection",
 CellID->1757724783],
Cell[25950, 890, 126, 3, 70, "ExampleText",
 CellID->95302197],
Cell[CellGroupData[{
Cell[26101, 897, 378, 13, 70, "Input",
 CellID->391104580],
Cell[26482, 912, 308, 11, 36, "Output",
 CellID->1143497]
}, Open  ]],
Cell[CellGroupData[{
Cell[26827, 928, 264, 9, 70, "Input",
 CellID->243545844],
Cell[27094, 939, 308, 11, 36, "Output",
 CellID->1744680]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[27451, 956, 228, 7, 70, "ExampleSubsection",
 CellID->1295379749],
Cell[27682, 965, 174, 5, 70, "ExampleText",
 CellID->291190620],
Cell[27859, 972, 215, 7, 70, "Input",
 CellID->590618219],
Cell[28077, 981, 96, 1, 70, "ExampleText",
 CellID->126067530],
Cell[CellGroupData[{
Cell[28198, 986, 355, 11, 70, "Input",
 CellID->588840553],
Cell[28556, 999, 179, 5, 36, "Output",
 CellID->150429528]
}, Open  ]],
Cell[28750, 1007, 76, 1, 70, "ExampleText",
 CellID->187348485],
Cell[CellGroupData[{
Cell[28851, 1012, 943, 28, 70, "Input",
 CellID->5675005],
Cell[29797, 1042, 864, 23, 98, "Output",
 CellID->107933660]
}, Open  ]],
Cell[30676, 1068, 125, 3, 70, "ExampleDelimiter",
 CellID->530742545],
Cell[30804, 1073, 123, 3, 70, "ExampleText",
 CellID->664258031],
Cell[30930, 1078, 232, 8, 70, "Input",
 CellID->335241345],
Cell[CellGroupData[{
Cell[31187, 1090, 729, 22, 70, "Input",
 CellID->201751166],
Cell[31919, 1114, 853, 23, 92, "Output",
 CellID->974156184]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[32833, 1144, 227, 7, 70, "ExampleSection",
 CellID->258228157],
Cell[33063, 1153, 196, 6, 70, "ExampleText",
 CellID->369412078],
Cell[CellGroupData[{
Cell[33284, 1163, 457, 16, 70, "Input",
 CellID->110850738],
Cell[33744, 1181, 795, 31, 62, "Output",
 CellID->258849475]
}, Open  ]],
Cell[34554, 1215, 81, 1, 70, "ExampleText",
 CellID->214651176],
Cell[CellGroupData[{
Cell[34660, 1220, 577, 19, 70, "Input",
 CellID->15261190],
Cell[35240, 1241, 977, 37, 65, "Output",
 CellID->585273161]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[36266, 1284, 238, 7, 70, "ExampleSection",
 CellID->2123667759],
Cell[36507, 1293, 268, 9, 70, "ExampleText",
 CellID->565470712],
Cell[CellGroupData[{
Cell[36800, 1306, 380, 13, 70, "Input",
 CellID->175634434],
Cell[37183, 1321, 603, 24, 55, "Output",
 CellID->649550955]
}, Open  ]],
Cell[37801, 1348, 125, 3, 70, "ExampleDelimiter",
 CellID->370862024],
Cell[37929, 1353, 274, 9, 70, "ExampleText",
 CellID->69631862],
Cell[CellGroupData[{
Cell[38228, 1366, 295, 10, 70, "Input",
 CellID->263507827],
Cell[38526, 1378, 1189, 34, 71, "Output",
 CellID->183093870]
}, Open  ]],
Cell[39730, 1415, 124, 3, 70, "ExampleDelimiter",
 CellID->54768347],
Cell[39857, 1420, 150, 4, 70, "ExampleText",
 CellID->15704052],
Cell[CellGroupData[{
Cell[40032, 1428, 318, 11, 70, "Input",
 CellID->49524972],
Cell[40353, 1441, 548, 22, 55, "Output",
 CellID->236853727]
}, Open  ]],
Cell[40916, 1466, 125, 3, 70, "ExampleDelimiter",
 CellID->394678611],
Cell[41044, 1471, 118, 3, 70, "ExampleText",
 CellID->293590904],
Cell[CellGroupData[{
Cell[41187, 1478, 250, 8, 70, "Input",
 CellID->41764138],
Cell[41440, 1488, 2050, 45, 71, "Output",
 CellID->183531080]
}, Open  ]],
Cell[43505, 1536, 77, 1, 70, "ExampleText",
 CellID->12179607],
Cell[CellGroupData[{
Cell[43607, 1541, 169, 5, 70, "Input",
 CellID->266322687],
Cell[43779, 1548, 397, 13, 36, "Output",
 CellID->405091768]
}, Open  ]],
Cell[44191, 1564, 125, 3, 70, "ExampleDelimiter",
 CellID->698648664],
Cell[44319, 1569, 148, 4, 70, "ExampleText",
 CellID->805989730],
Cell[CellGroupData[{
Cell[44492, 1577, 500, 17, 70, "Input",
 CellID->53028191],
Cell[44995, 1596, 795, 31, 62, "Output",
 CellID->561095229]
}, Open  ]],
Cell[45805, 1630, 64, 1, 70, "ExampleText",
 CellID->27311458],
Cell[CellGroupData[{
Cell[45894, 1635, 347, 12, 70, "Input",
 CellID->551996848],
Cell[46244, 1649, 500, 15, 36, "Output",
 CellID->279673394]
}, Open  ]],
Cell[46759, 1667, 122, 3, 70, "ExampleDelimiter",
 CellID->738094],
Cell[46884, 1672, 562, 20, 70, "ExampleText",
 CellID->822074023],
Cell[47449, 1694, 226, 8, 70, "Input",
 CellID->639586099],
Cell[CellGroupData[{
Cell[47700, 1706, 436, 14, 70, "Input",
 CellID->350486289],
Cell[48139, 1722, 1799, 48, 64, "Output",
 CellID->38685493]
}, Open  ]],
Cell[49953, 1773, 72, 1, 70, "ExampleText",
 CellID->471826964],
Cell[CellGroupData[{
Cell[50050, 1778, 374, 12, 70, "Input",
 CellID->335234509],
Cell[50427, 1792, 1998, 55, 64, "Output",
 CellID->132766]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[52486, 1854, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[52805, 1865, 1684, 62, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[54526, 1932, 318, 9, 70, "TutorialsSection",
 CellID->250839057],
Cell[54847, 1943, 148, 3, 70, "Tutorials",
 CellID->513408734]
}, Open  ]],
Cell[CellGroupData[{
Cell[55032, 1951, 318, 9, 70, "MoreAboutSection",
 CellID->43178843],
Cell[55353, 1962, 128, 3, 70, "MoreAbout",
 CellID->519431487],
Cell[55484, 1967, 163, 3, 70, "MoreAbout",
 CellID->626062243],
Cell[55650, 1972, 149, 3, 70, "MoreAbout",
 CellID->482558],
Cell[55802, 1977, 169, 3, 70, "MoreAbout",
 CellID->219016046],
Cell[55974, 1982, 179, 3, 70, "MoreAbout",
 CellID->547157338],
Cell[56156, 1987, 198, 4, 70, "MoreAbout",
 CellID->283719392]
}, Open  ]],
Cell[56369, 1994, 27, 0, 70, "History"],
Cell[56399, 1996, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

