(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     53877,       1837]
NotebookOptionsPosition[     45554,       1545]
NotebookOutlinePosition[     47008,       1584]
CellTagsIndexPosition[     46922,       1579]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Pad\[EAcute] Approximation" :> 
          Documentation`HelpLookup["paclet:tutorial/PadeApproximation"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Pad\[EAcute] Approximation\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Series" :> Documentation`HelpLookup["paclet:ref/Series"], 
          "Rationalize" :> 
          Documentation`HelpLookup["paclet:ref/Rationalize"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Series\"\>", 
       2->"\<\"Rationalize\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Rational Functions" :> 
          Documentation`HelpLookup["paclet:guide/RationalFunctions"], 
          "New in 6.0: Symbolic Computation" :> 
          Documentation`HelpLookup["paclet:guide/NewIn60AlgebraicComputing"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Rational Functions\"\>", 
       2->"\<\"New in 6.0: Symbolic Computation\"\>", 
       3->"\<\"New in 6.0: Mathematics & Algorithms\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["PadeApproximant", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["PadeApproximant",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/PadeApproximant"], "[", 
       RowBox[{
        StyleBox["expr", "TI"], ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["x", "TI"], ",", 
          SubscriptBox[
           StyleBox["x", "TI"], 
           StyleBox["0", "TR"]], ",", 
          RowBox[{"{", 
           RowBox[{
            StyleBox["m", "TI"], ",", 
            StyleBox["n", "TI"]}], 
           StyleBox["}", "TI"]}]}], "}"}]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the Pad\[EAcute] approximant to ",
     Cell[BoxData[
      StyleBox["expr", "TI"]], "InlineFormula"],
     " about the point ",
     Cell[BoxData[
      RowBox[{
       StyleBox["x", "TI"], "=", 
       SubscriptBox[
        StyleBox["x", "TI"], 
        StyleBox["0", "TR"]]}]], "InlineFormula"],
     ", with numerator order ",
     Cell[BoxData[
      StyleBox["m", "TI"]], "InlineFormula"],
     " and denominator order ",
     Cell[BoxData[
      StyleBox["n", "TI"]], "InlineFormula"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["PadeApproximant",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/PadeApproximant"], "[", 
       RowBox[{
        StyleBox["expr", "TI"], ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["x", "TI"], ",", 
          SubscriptBox[
           StyleBox["x", "TI"], 
           StyleBox["0", "TR"]], ",", 
          StyleBox["n", "TI"]}], "}"}]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the diagonal Pad\[EAcute] approximant to ",
     Cell[BoxData[
      StyleBox["expr", "TI"]], "InlineFormula"],
     " about the point ",
     Cell[BoxData[
      RowBox[{
       StyleBox["x", "TI"], "=", 
       SubscriptBox[
        StyleBox["x", "TI"], 
        StyleBox["0", "TR"]]}]], "InlineFormula"],
     " of order ",
     Cell[BoxData[
      StyleBox["n", "TI"]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->12187]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " can find the Pad\[EAcute] approximant about the point ",
 Cell[BoxData[
  RowBox[{
   StyleBox["x", "TI"], "=", 
   SubscriptBox[
    StyleBox["x", "TI"], 
    StyleBox["0", "TR"]]}]], "InlineFormula"],
 " only when it can evaluate power series at that point."
}], "Notes",
 CellID->14462],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["PadeApproximant",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PadeApproximant"]], "InlineFormula"],
 " produces a ratio of ordinary polynomial expressions, not a special ",
 Cell[BoxData[
  ButtonBox["SeriesData",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SeriesData"]], "InlineFormula"],
 " object."
}], "Notes",
 CellID->18258]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->41793019],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->553444322],

Cell[TextData[{
 "Order [2/3] Pad\[EAcute] approximant for ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Exp",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Exp"], "[", 
   StyleBox["x", "TI"], "]"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->13976],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PadeApproximant", "[", 
  RowBox[{
   RowBox[{"Exp", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "3"}], "}"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->881659],

Cell[BoxData[
 FractionBox[
  RowBox[{"1", "+", 
   FractionBox[
    RowBox[{"2", " ", "x"}], "5"], "+", 
   FractionBox[
    SuperscriptBox["x", "2"], "20"]}], 
  RowBox[{"1", "-", 
   FractionBox[
    RowBox[{"3", " ", "x"}], "5"], "+", 
   FractionBox[
    RowBox[{"3", " ", 
     SuperscriptBox["x", "2"]}], "20"], "-", 
   FractionBox[
    SuperscriptBox["x", "3"], "60"]}]]], "Output",
 ImageSize->{106, 53},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->39478702]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->39],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["PadeApproximant",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PadeApproximant"]], "InlineFormula"],
 " can handle functions with poles:"
}], "ExampleText",
 CellID->31283],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PadeApproximant", "[", 
  RowBox[{
   FractionBox[
    RowBox[{"Exp", "[", "x", "]"}], "x"], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "3"}], "}"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->35112178],

Cell[BoxData[
 FractionBox[
  RowBox[{"1", "+", 
   FractionBox["x", "2"], "+", 
   FractionBox[
    SuperscriptBox["x", "2"], "12"]}], 
  RowBox[{"x", "-", 
   FractionBox[
    SuperscriptBox["x", "2"], "2"], "+", 
   FractionBox[
    SuperscriptBox["x", "3"], "12"]}]]], "Output",
 ImageSize->{70, 53},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->239775505]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->5769],

Cell["\<\
Pad\[EAcute] approximant of an arbitrary function:\
\>", "ExampleText",
 CellID->23675],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PadeApproximant", "[", 
  RowBox[{
   FractionBox[
    RowBox[{"f", "[", "x", "]"}], "x"], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->917],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzFWTlv1EAUnrV3N8ciYBMB4Q4COpKGS1RQkQYq0oK0WZA2BSIKoQhKF4Wg
SIhDQhAJCgj8BVrKIP5AfkbKlIPn8Nr+/DzH2issre03M++Yb9689zx7v7PS
e/K0s7LY7UzPLXeWeovd59N3ny1HTWGNsdoDxtjtKRa98+hN/sQ1H/3Uq3r7
Fd2DqHNEPz/0W1kY83DZm6bfyVE7mqcRPUXnGIwKLLSS8hNaQ6BHpQF9WvIw
dc+NVRJ/QGu9tES0MdAzljOqJ+2a30v2TqqVaTRZAyQoTbnWEHjd8DBoSM8x
lldPj0x8RFDKhzZ0e6Cff2Trqxw2banh4ODgt7zYjLiF+/v7esSGHLbJE48U
RjYl197e3h15sRlxCyNWkutvge5rPOvnW6TlVwED9PvXkutLTvqo9odJxNBC
K3lftfZx6MXVbAAt/drR18ya0C6ct49XpzBqpDGKh9V5au+ADXE0ydnmpfsb
hVvO35sua+Wl9zulN+MnODeUE2NA77stUsso0BjtcF6fchix2CMcZ/uZtGOc
RBl5t3k2dlBvymIRHNQlBN2LfrVo59eivpqDjBc8u9/fAn2rP8rk928s/Tel
lAVeLg+uAr9tdyKtrHgMrWXy6GqKZoTfjnnIUtZ1obVMTq7eOsQuzu8+kshV
9LJigZpXRZXAKtCONYD3HHCd8zEQaiZuinhr+tmkpThHTrXbD2mqpZ/zOakf
cy1zfc60thbQ4qbmL/QLWNG/RXtI+CpmoW2g30upKI3MkbnVRC7Upejd3V1d
WW3KyqrX6zmOX4fxQYW21nm6+ktqxk1ZM0bVoGXcOoyr0rb/uWZHgX4JNOae
Y0AvUfILvdbNximg14A+D/QyhUxFeFVjSxzDhfSAJTVrTJ8ErkmerS7aQB8G
+jTQONezQB+H8eeAPmWRJ1GS30Botam6OWLpx6iI/YgS+VVhmbWJ/4ylH33B
pl+hNCNf0zmgqEqPvQTl2uq1OBf41olNzUfZ46LP9+sU9bRzyOC3ByLjW8m2
eLLPyiA7UXKmyhNmebL7TbUrcsuY6lg1edfBmdpGEGVqNttMypw4ID2h0USv
t9ngU4M3NSaqQw0jq5gh4eNzTuLnczbLhupzhhhaxtuqPHVy9TabTT7ehvmv
yjWxYeOD+6B5jrIpLBlxQl7FyXkmEw6iuSBnVYWwzJSFp5/OGGfwscbeTGYQ
MjA7mxCoFEXv0wnaQ7FeDUAKrcsnR9VJCYOdsJSOjEM9oUYaa2tT/CnG2tui
nIQxJz3l15TEckh+SlYJpK7iE6sRN7wL+cfJVoOuCv+nQO4TFmluePmcG6I3
494ceEXsKDla6PaF/pAaZYgI+N1jQ/4y0Fc89bXJWVynZuG4e00zYARuF6F/
1kMXWt8grX/EbZFCvOK3NZ49IJKXgJ510oLoKItvQCueCV4Amj4DFK/qvzNW
+wcagnW+\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{339, 63},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->33113755]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->15020],

Cell["\<\
Pad\[EAcute] approximant with a complex-valued expansion point:\
\>", "ExampleText",
 CellID->6030],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PadeApproximant", "[", 
  RowBox[{
   RowBox[{"Cos", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "I", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->17442],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{456, 75},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->91878968]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->192591776],

Cell["\<\
Pad\[EAcute] approximant with an expansion point at infinity:\
\>", "ExampleText",
 CellID->252993701],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PadeApproximant", "[", 
  RowBox[{
   RowBox[{"Log", "[", 
    RowBox[{"1", "+", 
     RowBox[{"1", "/", "x"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "Infinity", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->352078885],

Cell[BoxData[
 FractionBox[
  RowBox[{
   FractionBox["1", 
    RowBox[{"2", " ", 
     SuperscriptBox["x", "2"]}]], "+", 
   FractionBox["1", "x"]}], 
  RowBox[{"1", "+", 
   FractionBox["1", 
    RowBox[{"6", " ", 
     SuperscriptBox["x", "2"]}]], "+", 
   FractionBox["1", "x"]}]]], "Output",
 ImageSize->{73, 51},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->256709090]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->47068930],

Cell["\<\
Find a Pad\[EAcute] approximant to a given series:\
\>", "ExampleText",
 CellID->242726366],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sum", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"BernoulliB", "[", 
     RowBox[{"2", "n"}], "]"}], 
    RowBox[{"x", "^", "n"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "0", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->736465434],

Cell[BoxData[
 RowBox[{"1", "+", 
  FractionBox["x", "6"], "-", 
  FractionBox[
   SuperscriptBox["x", "2"], "30"], "+", 
  FractionBox[
   SuperscriptBox["x", "3"], "42"], "-", 
  FractionBox[
   SuperscriptBox["x", "4"], "30"], "+", 
  FractionBox[
   RowBox[{"5", " ", 
    SuperscriptBox["x", "5"]}], "66"], "-", 
  FractionBox[
   RowBox[{"691", " ", 
    SuperscriptBox["x", "6"]}], "2730"], "+", 
  FractionBox[
   RowBox[{"7", " ", 
    SuperscriptBox["x", "7"]}], "6"], "-", 
  FractionBox[
   RowBox[{"3617", " ", 
    SuperscriptBox["x", "8"]}], "510"], "+", 
  FractionBox[
   RowBox[{"43867", " ", 
    SuperscriptBox["x", "9"]}], "798"], "-", 
  FractionBox[
   RowBox[{"174611", " ", 
    SuperscriptBox["x", "10"]}], "330"]}]], "Output",
 ImageSize->{470, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->192536676]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PadeApproximant", "[", 
  RowBox[{"%", ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->694626828],

Cell[BoxData[
 FractionBox[
  RowBox[{"1", "+", 
   FractionBox[
    RowBox[{"997", " ", "x"}], "42"], "+", 
   FractionBox[
    RowBox[{"316156", " ", 
     SuperscriptBox["x", "2"]}], "1995"], "+", 
   FractionBox[
    RowBox[{"4665797", " ", 
     SuperscriptBox["x", "3"]}], "13566"], "+", 
   FractionBox[
    RowBox[{"17344727", " ", 
     SuperscriptBox["x", "4"]}], "79135"], "+", 
   FractionBox[
    RowBox[{"10045320", " ", 
     SuperscriptBox["x", "5"]}], "323323"]}], 
  RowBox[{"1", "+", 
   FractionBox[
    RowBox[{"165", " ", "x"}], "7"], "+", 
   FractionBox[
    RowBox[{"2937", " ", 
     SuperscriptBox["x", "2"]}], "19"], "+", 
   FractionBox[
    RowBox[{"103015", " ", 
     SuperscriptBox["x", "3"]}], "323"], "+", 
   FractionBox[
    RowBox[{"385836", " ", 
     SuperscriptBox["x", "4"]}], "2261"], "+", 
   FractionBox[
    RowBox[{"43200", " ", 
     SuperscriptBox["x", "5"]}], "4199"]}]]], "Output",
 ImageSize->{327, 53},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->272732061]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->32389],

Cell[TextData[{
 "Pad\[EAcute] approximant centered at the point ",
 Cell[BoxData[
  RowBox[{
   StyleBox["x", "TI"], "=", 
   StyleBox["a", "TI"]}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->20373],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PadeApproximant", "[", 
  RowBox[{
   RowBox[{"Exp", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "a", ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "3"}], "}"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->3480],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox["\[ExponentialE]", "a"], "+", 
   RowBox[{
    FractionBox["2", "5"], " ", 
    SuperscriptBox["\[ExponentialE]", "a"], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "a"}], "+", "x"}], ")"}]}], "+", 
   RowBox[{
    FractionBox["1", "20"], " ", 
    SuperscriptBox["\[ExponentialE]", "a"], " ", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "a"}], "+", "x"}], ")"}], "2"]}]}], 
  RowBox[{"1", "-", 
   RowBox[{
    FractionBox["3", "5"], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "a"}], "+", "x"}], ")"}]}], "+", 
   RowBox[{
    FractionBox["3", "20"], " ", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "a"}], "+", "x"}], ")"}], "2"]}], "-", 
   RowBox[{
    FractionBox["1", "60"], " ", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "a"}], "+", "x"}], ")"}], "3"]}]}]]], "Output",
 ImageSize->{257, 47},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->744968702]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->45664635],

Cell["\<\
Pad\[EAcute] approximant in fractional powers:\
\>", "ExampleText",
 CellID->205718317],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PadeApproximant", "[", " ", 
  RowBox[{
   RowBox[{"Sin", "[", 
    RowBox[{"Sqrt", "[", "x", "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->468002459],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SqrtBox["x"], "-", 
   FractionBox[
    RowBox[{"2363", " ", 
     SuperscriptBox["x", 
      RowBox[{"3", "/", "2"}]]}], "18183"], "+", 
   FractionBox[
    RowBox[{"12671", " ", 
     SuperscriptBox["x", 
      RowBox[{"5", "/", "2"}]]}], "4363920"]}], 
  RowBox[{"1", "+", 
   FractionBox[
    RowBox[{"445", " ", "x"}], "12122"], "+", 
   FractionBox[
    RowBox[{"601", " ", 
     SuperscriptBox["x", "2"]}], "872784"], "+", 
   FractionBox[
    RowBox[{"121", " ", 
     SuperscriptBox["x", "3"]}], "16662240"]}]]], "Output",
 ImageSize->{167, 53},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->133632258]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->463412606],

Cell["\<\
Pad\[EAcute] approximant of a function containing logarithmic terms:\
\>", "ExampleText",
 CellID->350172998],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Hypergeometric2F1", "[", 
  RowBox[{"2", ",", "2", ",", "4", ",", 
   RowBox[{"1", "-", "x"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->644854751],

Cell[BoxData[
 FractionBox[
  RowBox[{"6", " ", 
   RowBox[{"(", 
    RowBox[{"2", "-", 
     RowBox[{"2", " ", "x"}], "+", 
     RowBox[{"Log", "[", "x", "]"}], "+", 
     RowBox[{"x", " ", 
      RowBox[{"Log", "[", "x", "]"}]}]}], ")"}]}], 
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "x"}], ")"}], "3"]]], "Output",
 ImageSize->{188, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->567139884]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PadeApproximant", "[", 
  RowBox[{
   RowBox[{"Hypergeometric2F1", "[", 
    RowBox[{"2", ",", "2", ",", "4", ",", 
     RowBox[{"1", "-", "x"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->382005369],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", 
   FractionBox["12", 
    RowBox[{"1", "-", 
     RowBox[{"2", " ", "x"}], "+", 
     SuperscriptBox["x", "2"]}]]}], "+", 
  FractionBox[
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "6"}], "-", 
      RowBox[{"6", " ", "x"}]}], ")"}], " ", 
    RowBox[{"Log", "[", "x", "]"}]}], 
   RowBox[{"1", "-", 
    RowBox[{"3", " ", "x"}], "+", 
    RowBox[{"3", " ", 
     SuperscriptBox["x", "2"]}], "-", 
    SuperscriptBox["x", "3"]}]]}]], "Output",
 ImageSize->{198, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->4282066]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->31785],

Cell[TextData[{
 "Plot successive Pad\[EAcute] approximants to ",
 Cell[BoxData[
  FormBox[
   RowBox[{"sin", "(", "x", ")"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->522],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{"PadeApproximant", "[", 
       RowBox[{
        RowBox[{"Sin", "[", "x", "]"}], ",", 
        RowBox[{"{", 
         RowBox[{"x", ",", "0", ",", "n"}], "}"}]}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"n", ",", "5"}], "}"}]}], "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", 
     RowBox[{"2", "\[Pi]"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->18345],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 137},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->878228846]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->385597818],

Cell["\<\
Construct discrete orthogonal polynomials with respect to discrete weighted \
measure:\
\>", "ExampleText",
 CellID->28757800],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"\[Omega]", "[", "k_", "]"}], "=", 
   RowBox[{
    SuperscriptBox[
     RowBox[{"(", 
      FractionBox[
       RowBox[{"-", "1"}], "2"], ")"}], "k"], " ", 
    FractionBox["1", 
     SqrtBox["k"]]}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->89607175],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"moment", "[", "n_", "]"}], "=", 
  RowBox[{"Sum", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"\[Omega]", "[", "k", "]"}], "/", 
     RowBox[{"k", "^", "n"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"k", ",", "1", ",", "Infinity"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->458210890],

Cell[BoxData[
 RowBox[{"PolyLog", "[", 
  RowBox[{
   RowBox[{
    FractionBox["1", "2"], "+", "n"}], ",", 
   RowBox[{"-", 
    FractionBox["1", "2"]}]}], "]"}]], "Output",
 ImageSize->{128, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->357917763]
}, Open  ]],

Cell[BoxData[
 RowBox[{
  RowBox[{"p", "[", 
   RowBox[{"m_", ",", "x_"}], "]"}], ":=", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"x", "^", "m"}]}], "  ", 
   RowBox[{"Denominator", "[", 
    RowBox[{"PadeApproximant", "[", 
     RowBox[{
      RowBox[{"Sum", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"x", "^", "k"}], " ", 
         RowBox[{"N", "[", 
          RowBox[{
           RowBox[{"moment", "[", "k", "]"}], ",", "30"}], "]"}]}], ",", 
        RowBox[{"{", 
         RowBox[{"k", ",", "0", ",", "10"}], "}"}]}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", "0", ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"m", "-", "1"}], ",", "m"}], "}"}]}], "}"}]}], "]"}], 
    "]"}]}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->365158402],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"Sum", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"\[Omega]", "[", "x", "]"}], 
      RowBox[{"p", "[", 
       RowBox[{"n", ",", "x"}], "]"}], " ", 
      RowBox[{"p", "[", 
       RowBox[{"m", ",", "x"}], "]"}]}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "1", ",", "Infinity"}], "}"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "5"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"m", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->260037025],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "0.032155736681259849086158569560192457174477009658`27.91004300466064", 
     ",", "0``29.281742954492568", ",", "0``29.115313240382292", ",", 
     "0``28.99216782213593", ",", "0``28.89612290019834"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0``29.281742954492568", ",", 
     RowBox[{
     "-", "0.000577133370115906722894097837422404791872027686`25.\
953232443791375"}], ",", "0``29.028471504413965", ",", 
     "0``28.907866469377446", ",", "0``28.81345002926065"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "0``29.115313240382292", ",", "0``29.028471504413965", ",", 
     "4.39656660905964761704313180964587`23.51520786246989*^-6", ",", 
     "0``28.756082100756093", ",", "0``28.66457689164602"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "0``28.99216782213593", ",", "0``28.907866469377446", ",", 
     "0``28.756082100756093", ",", 
     RowBox[{"-", "1.834044672490095998765416177623`20.906484727199697*^-8"}],
      ",", "0``28.553478285627566"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "0``28.89612290019834", ",", "0``28.81345002926065", ",", 
     "0``28.66457689164602", ",", "0``28.553478285627566", ",", 
     "4.810747564841982847607538524`18.147305259812967*^-11"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{521, 94},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->82621380]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->16085],

Cell[TextData[{
 "The Pad\[EAcute] approximant agrees with the ordinary series for ",
 Cell[BoxData[
  FormBox[
   RowBox[{"m", "+", "n"}], TraditionalForm]], "InlineMath"],
 " terms:"
}], "ExampleText",
 CellID->15713],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"PadeApproximant", "[", 
    RowBox[{
     RowBox[{"Exp", "[", "x", "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", 
       RowBox[{"{", 
        RowBox[{"2", ",", "3"}], "}"}]}], "}"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "6"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->30935],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"1", "+", "x", "+", 
   FractionBox[
    SuperscriptBox["x", "2"], "2"], "+", 
   FractionBox[
    SuperscriptBox["x", "3"], "6"], "+", 
   FractionBox[
    SuperscriptBox["x", "4"], "24"], "+", 
   FractionBox[
    SuperscriptBox["x", "5"], "120"], "+", 
   FractionBox[
    RowBox[{"11", " ", 
     SuperscriptBox["x", "6"]}], "7200"], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "x", "]"}], "7"],
    SeriesData[$CellContext`x, 0, {}, 0, 7, 1],
    Editable->False]}],
  SeriesData[$CellContext`x, 0, {1, 1, 
    Rational[1, 2], 
    Rational[1, 6], 
    Rational[1, 24], 
    Rational[1, 120], 
    Rational[11, 7200]}, 0, 7, 1],
  Editable->False]], "Output",
 ImageSize->{250, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->936942803]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"Exp", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "6"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->30883],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"1", "+", "x", "+", 
   FractionBox[
    SuperscriptBox["x", "2"], "2"], "+", 
   FractionBox[
    SuperscriptBox["x", "3"], "6"], "+", 
   FractionBox[
    SuperscriptBox["x", "4"], "24"], "+", 
   FractionBox[
    SuperscriptBox["x", "5"], "120"], "+", 
   FractionBox[
    SuperscriptBox["x", "6"], "720"], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "x", "]"}], "7"],
    SeriesData[$CellContext`x, 0, {}, 0, 7, 1],
    Editable->False]}],
  SeriesData[$CellContext`x, 0, {1, 1, 
    Rational[1, 2], 
    Rational[1, 6], 
    Rational[1, 24], 
    Rational[1, 120], 
    Rational[1, 720]}, 0, 7, 1],
  Editable->False]], "Output",
 ImageSize->{242, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->576095441]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->16096],

Cell[TextData[{
 "For ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", "=", "0"}], TraditionalForm]], "InlineMath"],
 " ",
 Cell[BoxData[
  ButtonBox["PadeApproximant",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PadeApproximant"]], "InlineFormula"],
 " gives an ordinary series:"
}], "ExampleText",
 CellID->20648],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PadeApproximant", "[", 
  RowBox[{
   RowBox[{"Exp", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", 
     RowBox[{"{", 
      RowBox[{"5", ",", "0"}], "}"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2085],

Cell[BoxData[
 RowBox[{"1", "+", "x", "+", 
  FractionBox[
   SuperscriptBox["x", "2"], "2"], "+", 
  FractionBox[
   SuperscriptBox["x", "3"], "6"], "+", 
  FractionBox[
   SuperscriptBox["x", "4"], "24"], "+", 
  FractionBox[
   SuperscriptBox["x", "5"], "120"]}]], "Output",
 ImageSize->{158, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->459824844]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->521],

Cell["\<\
Pad\[EAcute] approximants often have spurious poles not present in the \
original function:\
\>", "ExampleText",
 CellID->9260],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PadeApproximant", "[", 
  RowBox[{
   RowBox[{"Exp", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->21982],

Cell[BoxData[
 FractionBox[
  RowBox[{"1", "+", 
   FractionBox["x", "2"], "+", 
   FractionBox[
    SuperscriptBox["x", "2"], "10"], "+", 
   FractionBox[
    SuperscriptBox["x", "3"], "120"]}], 
  RowBox[{"1", "-", 
   FractionBox["x", "2"], "+", 
   FractionBox[
    SuperscriptBox["x", "2"], "10"], "-", 
   FractionBox[
    SuperscriptBox["x", "3"], "120"]}]]], "Output",
 ImageSize->{97, 53},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->6474849]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NSolve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Denominator", "[", "%", "]"}], "==", "0"}], ",", "x"}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->7688],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{"3.677814645373914`", "\[InvisibleSpace]", "-", 
      RowBox[{"3.5087619195674433`", " ", "\[ImaginaryI]"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{"3.677814645373914`", "\[InvisibleSpace]", "+", 
      RowBox[{"3.5087619195674433`", " ", "\[ImaginaryI]"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", "4.644370709252172`"}], "}"}]}], "}"}]], "Output",\

 ImageSize->{444, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->162020982]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->17236],

Cell["\<\
Pad\[EAcute] approximants of a given order may not exist:\
\>", "ExampleText",
 CellID->26792],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PadeApproximant", "[", 
  RowBox[{
   RowBox[{"Cos", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->27065],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->494235113]
}, Open  ]],

Cell["\<\
Perturbing the order slightly is usually sufficient to produce an \
approximant:\
\>", "ExampleText",
 CellID->31496],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PadeApproximant", "[", 
  RowBox[{
   RowBox[{"Cos", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->26386],

Cell[BoxData[
 FractionBox[
  RowBox[{"1", "-", 
   FractionBox[
    RowBox[{"5", " ", 
     SuperscriptBox["x", "2"]}], "12"]}], 
  RowBox[{"1", "+", 
   FractionBox[
    SuperscriptBox["x", "2"], "12"]}]]], "Output",
 ImageSize->{50, 53},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->14473967]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->22832],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["Series",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Series"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Rationalize",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Rationalize"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->24441]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->16859],

Cell[TextData[ButtonBox["Pad\[EAcute] Approximation",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/PadeApproximation"]], "Tutorials",
 CellID->8871]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection"],

Cell[TextData[{
 ButtonBox["Demonstrations with PadeApproximant",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/symbol.html?symbol=\
PadeApproximant"], None}],
 " (",
 ButtonBox["Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None}],
 ")"
}], "RelatedLinks"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->68500462],

Cell[TextData[ButtonBox["Rational Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/RationalFunctions"]], "MoreAbout",
 CellID->261541416],

Cell[TextData[ButtonBox["New in 6.0: Symbolic Computation",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60AlgebraicComputing"]], "MoreAbout",
 CellID->878000925],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->192595385]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"PadeApproximant - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 5, 44.4721228}", "context" -> "System`", 
    "keywords" -> {"Pade approximant"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "PadeApproximant[expr, {x, x_0, {m, n}}] gives the Pade' approximant to \
expr about the point x = x_0, with numerator order m and denominator order n. \
PadeApproximant[expr, {x, x_0, n}] gives the diagonal Pade' approximant to \
expr about the point x = x_0 of order n.", "synonyms" -> {"pade approximant"},
     "title" -> "PadeApproximant", "type" -> "Symbol", "uri" -> 
    "ref/PadeApproximant"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[7973, 246, 359, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->41793019]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 46780, 1572}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2225, 54, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2848, 81, 1580, 37, 70, "ObjectNameGrid"],
Cell[4431, 120, 2245, 73, 70, "Usage",
 CellID->12187]
}, Open  ]],
Cell[CellGroupData[{
Cell[6713, 198, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[7191, 215, 357, 12, 70, "Notes",
 CellID->14462],
Cell[7551, 229, 385, 12, 70, "Notes",
 CellID->18258]
}, Closed]],
Cell[CellGroupData[{
Cell[7973, 246, 359, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->41793019],
Cell[CellGroupData[{
Cell[8357, 260, 148, 5, 70, "ExampleSection",
 CellID->553444322],
Cell[8508, 267, 261, 10, 70, "ExampleText",
 CellID->13976],
Cell[CellGroupData[{
Cell[8794, 281, 274, 9, 28, "Input",
 CellID->881659],
Cell[9071, 292, 521, 19, 74, "Output",
 CellID->39478702]
}, Open  ]],
Cell[9607, 314, 118, 3, 70, "ExampleDelimiter",
 CellID->39],
Cell[9728, 319, 220, 7, 70, "ExampleText",
 CellID->31283],
Cell[CellGroupData[{
Cell[9973, 330, 299, 10, 70, "Input",
 CellID->35112178],
Cell[10275, 342, 412, 15, 74, "Output",
 CellID->239775505]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[10736, 363, 215, 7, 70, "ExampleSection",
 CellID->5769],
Cell[10954, 372, 97, 3, 70, "ExampleText",
 CellID->23675],
Cell[CellGroupData[{
Cell[11076, 379, 239, 8, 70, "Input",
 CellID->917],
Cell[11318, 389, 1683, 32, 70, "Output",
 Evaluatable->False,
 CellID->33113755]
}, Open  ]],
Cell[13016, 424, 121, 3, 70, "ExampleDelimiter",
 CellID->15020],
Cell[13140, 429, 109, 3, 70, "ExampleText",
 CellID->6030],
Cell[CellGroupData[{
Cell[13274, 436, 220, 7, 70, "Input",
 CellID->17442],
Cell[13497, 445, 2431, 44, 70, "Output",
 Evaluatable->False,
 CellID->91878968]
}, Open  ]],
Cell[15943, 492, 125, 3, 70, "ExampleDelimiter",
 CellID->192591776],
Cell[16071, 497, 112, 3, 70, "ExampleText",
 CellID->252993701],
Cell[CellGroupData[{
Cell[16208, 504, 282, 9, 70, "Input",
 CellID->352078885],
Cell[16493, 515, 426, 16, 72, "Output",
 CellID->256709090]
}, Open  ]],
Cell[16934, 534, 124, 3, 70, "ExampleDelimiter",
 CellID->47068930],
Cell[17061, 539, 101, 3, 70, "ExampleText",
 CellID->242726366],
Cell[CellGroupData[{
Cell[17187, 546, 286, 10, 70, "Input",
 CellID->736465434],
Cell[17476, 558, 884, 31, 54, "Output",
 CellID->192536676]
}, Open  ]],
Cell[CellGroupData[{
Cell[18397, 594, 193, 6, 70, "Input",
 CellID->694626828],
Cell[18593, 602, 1063, 36, 74, "Output",
 CellID->272732061]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[19705, 644, 239, 7, 70, "ExampleSection",
 CellID->32389],
Cell[19947, 653, 208, 8, 70, "ExampleText",
 CellID->20373],
Cell[CellGroupData[{
Cell[20180, 665, 272, 9, 70, "Input",
 CellID->3480],
Cell[20455, 676, 1084, 39, 68, "Output",
 CellID->744968702]
}, Open  ]],
Cell[21554, 718, 124, 3, 70, "ExampleDelimiter",
 CellID->45664635],
Cell[21681, 723, 97, 3, 70, "ExampleText",
 CellID->205718317],
Cell[CellGroupData[{
Cell[21803, 730, 262, 8, 70, "Input",
 CellID->468002459],
Cell[22068, 740, 704, 25, 74, "Output",
 CellID->133632258]
}, Open  ]],
Cell[22787, 768, 125, 3, 70, "ExampleDelimiter",
 CellID->463412606],
Cell[22915, 773, 119, 3, 70, "ExampleText",
 CellID->350172998],
Cell[CellGroupData[{
Cell[23059, 780, 180, 5, 70, "Input",
 CellID->644854751],
Cell[23242, 787, 484, 17, 54, "Output",
 CellID->567139884]
}, Open  ]],
Cell[CellGroupData[{
Cell[23763, 809, 309, 9, 70, "Input",
 CellID->382005369],
Cell[24075, 820, 632, 23, 53, "Output",
 CellID->4282066]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[24756, 849, 223, 7, 70, "ExampleSection",
 CellID->31785],
Cell[24982, 858, 198, 7, 70, "ExampleText",
 CellID->522],
Cell[CellGroupData[{
Cell[25205, 869, 533, 17, 70, "Input",
 CellID->18345],
Cell[25741, 888, 5280, 91, 70, "Output",
 Evaluatable->False,
 CellID->878228846]
}, Open  ]],
Cell[31036, 982, 125, 3, 70, "ExampleDelimiter",
 CellID->385597818],
Cell[31164, 987, 136, 4, 70, "ExampleText",
 CellID->28757800],
Cell[31303, 993, 308, 12, 70, "Input",
 CellID->89607175],
Cell[CellGroupData[{
Cell[31636, 1009, 337, 11, 70, "Input",
 CellID->458210890],
Cell[31976, 1022, 304, 11, 51, "Output",
 CellID->357917763]
}, Open  ]],
Cell[32295, 1036, 789, 26, 70, "Input",
 CellID->365158402],
Cell[CellGroupData[{
Cell[33109, 1066, 548, 18, 70, "Input",
 CellID->260037025],
Cell[33660, 1086, 1443, 35, 115, "Output",
 CellID->82621380]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[35152, 1127, 233, 7, 70, "ExampleSection",
 CellID->16085],
Cell[35388, 1136, 219, 7, 70, "ExampleText",
 CellID->15713],
Cell[CellGroupData[{
Cell[35632, 1147, 401, 13, 70, "Input",
 CellID->30935],
Cell[36036, 1162, 866, 30, 54, "Output",
 CellID->936942803]
}, Open  ]],
Cell[CellGroupData[{
Cell[36939, 1197, 211, 7, 70, "Input",
 CellID->30883],
Cell[37153, 1206, 836, 29, 54, "Output",
 CellID->576095441]
}, Open  ]],
Cell[38004, 1238, 121, 3, 70, "ExampleDelimiter",
 CellID->16096],
Cell[38128, 1243, 316, 12, 70, "ExampleText",
 CellID->20648],
Cell[CellGroupData[{
Cell[38469, 1259, 272, 9, 70, "Input",
 CellID->2085],
Cell[38744, 1270, 408, 14, 54, "Output",
 CellID->459824844]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[39201, 1290, 224, 7, 70, "ExampleSection",
 CellID->521],
Cell[39428, 1299, 137, 4, 70, "ExampleText",
 CellID->9260],
Cell[CellGroupData[{
Cell[39590, 1307, 220, 7, 70, "Input",
 CellID->21982],
Cell[39813, 1316, 504, 18, 74, "Output",
 CellID->6474849]
}, Open  ]],
Cell[CellGroupData[{
Cell[40354, 1339, 188, 7, 70, "Input",
 CellID->7688],
Cell[40545, 1348, 640, 18, 36, "Output",
 CellID->162020982]
}, Open  ]],
Cell[41200, 1369, 121, 3, 70, "ExampleDelimiter",
 CellID->17236],
Cell[41324, 1374, 104, 3, 70, "ExampleText",
 CellID->26792],
Cell[CellGroupData[{
Cell[41453, 1381, 220, 7, 70, "Input",
 CellID->27065],
Cell[41676, 1390, 158, 5, 36, "Output",
 CellID->494235113]
}, Open  ]],
Cell[41849, 1398, 127, 4, 70, "ExampleText",
 CellID->31496],
Cell[CellGroupData[{
Cell[42001, 1406, 220, 7, 70, "Input",
 CellID->26386],
Cell[42224, 1415, 347, 13, 74, "Output",
 CellID->14473967]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[42632, 1435, 311, 9, 70, "SeeAlsoSection",
 CellID->22832],
Cell[42946, 1446, 453, 17, 70, "SeeAlso",
 CellID->24441]
}, Open  ]],
Cell[CellGroupData[{
Cell[43436, 1468, 314, 9, 70, "TutorialsSection",
 CellID->16859],
Cell[43753, 1479, 153, 3, 70, "Tutorials",
 CellID->8871]
}, Open  ]],
Cell[CellGroupData[{
Cell[43943, 1487, 305, 8, 70, "RelatedLinksSection"],
Cell[44251, 1497, 372, 12, 70, "RelatedLinks"]
}, Open  ]],
Cell[CellGroupData[{
Cell[44660, 1514, 318, 9, 70, "MoreAboutSection",
 CellID->68500462],
Cell[44981, 1525, 147, 3, 70, "MoreAbout",
 CellID->261541416],
Cell[45131, 1530, 169, 3, 70, "MoreAbout",
 CellID->878000925],
Cell[45303, 1535, 179, 3, 70, "MoreAbout",
 CellID->192595385]
}, Open  ]],
Cell[45497, 1541, 27, 0, 70, "History"],
Cell[45527, 1543, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

