(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     51265,       1709]
NotebookOptionsPosition[     42300,       1390]
NotebookOutlinePosition[     43678,       1429]
CellTagsIndexPosition[     43592,       1424]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Discrete Distributions" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/DiscreteDistributions"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Discrete Distributions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"BinomialDistribution" :> 
          Documentation`HelpLookup["paclet:ref/BinomialDistribution"], 
          "GammaDistribution" :> 
          Documentation`HelpLookup["paclet:ref/GammaDistribution"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"BinomialDistribution\"\>", 
       2->"\<\"GammaDistribution\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Discrete Statistical Distributions" :> 
          Documentation`HelpLookup[
           "paclet:guide/DiscreteStatisticalDistributions"], 
          "Functions Used in Statistics" :> 
          Documentation`HelpLookup["paclet:guide/FunctionsUsedInStatistics"], 
          "Mathematical Functions" :> 
          Documentation`HelpLookup["paclet:guide/MathematicalFunctions"], 
          "Statistical Distributions" :> 
          Documentation`HelpLookup["paclet:guide/StatisticalDistributions"], 
          "Statistics" :> Documentation`HelpLookup["paclet:guide/Statistics"],
           "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"], 
          "New in 6.0: Statistics" :> 
          Documentation`HelpLookup["paclet:guide/NewIn60Statistics"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Discrete Statistical Distributions\"\>", 
       2->"\<\"Functions Used in Statistics\"\>", 
       3->"\<\"Mathematical Functions\"\>", 
       4->"\<\"Statistical Distributions\"\>", 5->"\<\"Statistics\"\>", 
       6->"\<\"New in 6.0: Mathematics & Algorithms\"\>", 
       7->"\<\"New in 6.0: Statistics\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["PoissonDistribution", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["PoissonDistribution",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/PoissonDistribution"], "[", 
       StyleBox["\[Mu]", "TR"], "]"}]], "InlineFormula"],
     " \[LineSeparator]represents a Poisson distribution with mean ",
     Cell[BoxData[
      StyleBox["\[Mu]", "TR"]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->28458]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "The probability for integer value ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " in a Poisson distribution is ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox["\[ExponentialE]", 
     RowBox[{"-", "\[Mu]"}]], 
    RowBox[{
     SuperscriptBox["\[Mu]", "x"], "/", 
     RowBox[{"x", "!"}]}]}], TraditionalForm]], "InlineMath"],
 " for ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "\[GreaterEqual]", "0"}], TraditionalForm]], "InlineMath"],
 ".",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->7184,
  ButtonNote->"7184"]
}], "Notes",
 CellID->429991538],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["PoissonDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PoissonDistribution"]], "InlineFormula"],
 " allows ",
 Cell[BoxData[
  StyleBox["\[Mu]", "TR"]], "InlineFormula"],
 " to be any positive real number."
}], "Notes",
 CellID->132305407],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["PoissonDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PoissonDistribution"]], "InlineFormula"],
 " can be used with such functions as ",
 Cell[BoxData[
  ButtonBox["Mean",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Mean"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["CDF",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CDF"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["RandomInteger",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RandomInteger"]], "InlineFormula"],
 ".",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->1864,
  ButtonNote->"1864"]
}], "Notes",
 CellID->33143606]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->619271192],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->609804345],

Cell["\<\
The mean and variance of a Poisson distribution are the same:\
\>", "ExampleText",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->1864],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"PoissonDistribution", "[", "\[Mu]", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->20566],

Cell[BoxData["\[Mu]"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->20978886]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Variance", "[", 
  RowBox[{"PoissonDistribution", "[", "\[Mu]", "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->27931],

Cell[BoxData["\[Mu]"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->137000638]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->26756],

Cell["Probability density function:", "ExampleText",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->7184],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"PoissonDistribution", "[", "\[Mu]", "]"}], ",", "k"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2065],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"-", "\[Mu]"}]], " ", 
   SuperscriptBox["\[Mu]", "k"]}], 
  RowBox[{"k", "!"}]]], "Output",
 ImageSize->{43, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->86344222]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"k", ",", 
       RowBox[{"PDF", "[", 
        RowBox[{
         RowBox[{"PoissonDistribution", "[", "10", "]"}], ",", "k"}], "]"}]}],
       "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"k", ",", "0", ",", "30"}], "}"}]}], "]"}], ",", 
   RowBox[{"Filling", "->", "Axis"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->9265],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztXDtv1EAQds53AToiAZcQLqSABmiA8CoQBSJUqfJAUHGKgKRIgkIqUtGQ
8MgPQAIK8vgBIELLoyFCSkpCkSKhSIGQCFIkRISOXXvtWY/3bN+dfd49+6Sz
b7y7s983uzP78Nk9xYmhWyPFieHBYufV8eLdoeHBe53dY+Pkkt6kado6+c60
auR3ifwyvuulEjnzx9PkqJOkAjtnyDlDzgeZrGWNbOzTRoQsl8uS3bloWZ3l
0o3Es/RIL5eMj9ZOD1mQTa18jlYk60jO00POlO2jS0uek22oXHobSqflqfma
6Dki5K01a4gDtajOcun0p6glMlY5NxKti6V3BEMKOVi5Q5XiIz/FVnFasguV
KwTDl+PxacyntGa+nLj0AV/97QKmTsxnOJtYPmj5dYZ5geXBXumUiduX+Qiy
H2Fb8ri6z7g6N7tMEp48/sh8cX5u5dLFZzNPP2lTSMbpujCd6CMy6ANZrzHd
DGst5LjCrGtxWPq8Qa4svv3Kypjyu8VVwEjkR9MfWPrOzj/WXHb82dr645CZ
pczyJBHn397+C+nq6uPtA2040LdA5CuXn7N0sKfYvuJ03B5U1mtMzwr78x5D
vta/QHoKRW238traT2C9ufnbz4okC6R/3/iFraioPt2yD2tV26pjo29oq+Gx
pkNUfxmrg5ZgVv+2+iNUK8mjj4+HUw/fu6zcLbTiXsVYRmM18N0e20p0hGtR
jE001oE+BNZx96HZV8uOkVheVuHoA769HlYhY5ljVJCNRdj6wJO8rEJ61OgI
F7VlYxG2PvAgsVXyUqIOT991m7W1jpAJXXRsG7mN+fkGXdvcSBBriHJJYm1E
7dc0iiWD9U2bZSNGLWDX+G3IlpDGB+9kWruQWc29H2zsvnFrqlxjWMOxn2iU
KbvDqJW1CsTAZFgF9xssm1YxYyTdK0qGVXDfoH2Hih1Ksn/54gtb01XP3r2T
pZoVBvrmmWeXtUKAOxeNYQtYublHkuaS835JTnHGtw1q1nznsLLoG6MNUhYp
i3BZpF6dskhZqMmC322j89M7NgsVvBrvmgF69doAZoRiFgXUUlnJWIwYqP12
RPDKBucpoBaVlGWNa/oCanFlWOY0tD6zdrcouSOqsNjtXmWaR7zSUoaRa59N
tG5M2dQ3/lnreL+YJycTaw5wVFp0atgwRZk0lKnXVPvfwlGJUcKcVGaUMKeU
E+WEVF4CaOSzkd9KqtzdAUnQh3JvJ+UiDxcZVnhlObjWpbpKnuH5HBZePZgs
4I5y/NGJX+eI1jsmYrynI0eUx3swcY2EJ6SwyfEUhQQo0v4QBwp47iJOFPBc
R5wo4DmK+vbK+7FwjrPWamf5dUIXysxXdqxUjA1joL1u2S2YYg3q437/7atf
rEvjeePWesyQ4RmqbKS9yORY6XsKwuEY3XP64lpPprXYtfC797SXTTp6xalQ
tEOvmoyQA/QisxavWZnuMU7XiuaBWa1jbGs2CnrtEGE56SiomHQbRICigllM
ODWG7eugNYz4FDXGVGvUrVT9m0W8tOJ5kJ9W2Mn30ornOX5a+3utZ62c9rwQ
qDTUJsZ03kfLVA2l/MZePl10tyFgLe6Yusu+Yl5k1YSm3yNCU0pUDEO3axcp
AvyR1VHRuFZ5f/Ibt84ZMn7XFfznerrE+5JZP/Z3kCG3uw78phqQxaW83nio
Nf0HtukCFw==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 141},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->215144501]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->4389],

Cell["\<\
Generate a set of pseudorandom numbers that are Poisson distributed:\
\>", "ExampleText",
 CellID->25402],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RandomInteger", "[", 
  RowBox[{
   RowBox[{"PoissonDistribution", "[", "2", "]"}], ",", "10"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->25411],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "3", ",", "2", ",", "1", ",", "3", ",", "1", ",", "0", ",", "1", ",", "3", 
   ",", "0", ",", "4"}], "}"}]], "Output",
 ImageSize->{189, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->316718784]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->31807],

Cell["Properties based on higher-order moments:", "ExampleText",
 CellID->20699375],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", 
  RowBox[{"PoissonDistribution", "[", "\[Mu]", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->24407],

Cell[BoxData[
 FractionBox["1", 
  SqrtBox["\[Mu]"]]], "Output",
 ImageSize->{32, 35},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->447878400]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Kurtosis", "[", 
  RowBox[{"PoissonDistribution", "[", "\[Mu]", "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->7943],

Cell[BoxData[
 RowBox[{"3", "+", 
  FractionBox["1", "\[Mu]"]}]], "Output",
 ImageSize->{35, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->748461605]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->3112],

Cell["Third moment of a Poisson distribution:", "ExampleText",
 CellID->8231],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ExpectedValue", "[", 
  RowBox[{
   RowBox[{"k", "^", "3"}], ",", 
   RowBox[{"PoissonDistribution", "[", "\[Mu]", "]"}], ",", "k"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->10677],

Cell[BoxData[
 RowBox[{"\[Mu]", "+", 
  RowBox[{"3", " ", 
   SuperscriptBox["\[Mu]", "2"]}], "+", 
  SuperscriptBox["\[Mu]", "3"]}]], "Output",
 ImageSize->{72, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->9267117]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1206],

Cell[TextData[{
 "The 0.75 quantile of a Poisson distribution with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Mu]", "=", "\[Pi]"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->20401],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Quantile", "[", 
  RowBox[{
   RowBox[{"PoissonDistribution", "[", "\[Pi]", "]"}], ",", "0.75"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->21435],

Cell[BoxData["4"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->74850141]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->16811],

Cell["\<\
Plot the cumulative distribution function of a Poisson distribution:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->13370],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"CDF", "[", 
    RowBox[{
     RowBox[{"PoissonDistribution", "[", "10", "]"}], ",", "k"}], "]"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"k", ",", "0", ",", "20"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->26124],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWU1oE0EU3maTJgcPelLQQxU82Na0Fz0oYrVWUdEiWFREJQlCItZKWxUU
D3qwB734i4ViD/5Afw6CUBFaEQWxIC0UFdGCRJB6sAgWDwVhfW92tvOzs9vZ
JBtUXJLZvH3v+978vHkzs9mV6swebU115jKpqm3tqZPZXKajamtbOzwyKwzD
yMP3xhIDflvwi3zzlgV3d9kApQkGSXqPwD0C91oqG1FiRq8aEKKclUmUjVii
sUUugjWiTLbteYtqSTYleRUWMVueKwkLNq8C7yF5rlYyhO21hpPVLd1Ee7xO
zxezoLiklgeeJ6nnKcZ7MmjkGJU8zqsfN9O6rbZYNDmxF6GhF6FRx+vtqOOj
eKFUszGfpwugfEPr4baJe8njY1/gY8SxQHlfy4O2E4+MBBY6+BLJiT+c31T2
6lIiz87+Enqx+/Zo16VnKq+LlHgmL5vHvl6rViuJDFXJpAexQglSq+npn1R/
MwALoE51DmEwxElEACXIqOpRouoc32AtWmGE13tq9TiUVhCxvXdeGwkszMdD
76n+qhJl12CgfwKsERAnqI0bun1R64gcthX20PqyeFJaCT1y/944hDWqunxQ
L55/ElA7t/dOTf3wRm0hMgyYywrb3uSp1eMolRWdEOTiVzQi0/ytwSKsMTGJ
pxCOSgmDdUNRByvnM3nNKynHyPAky4Zv33yl+vMKTtIP7icuVohHyAYsD0Gk
2cQOKzdmlTSeYhLTbiIDHOrFkHJMFGuFnpvL4ul/29xW16+9hFQGHyOBBWYo
qm9XovYTGZZre+EwOoS2HvDU6nGUysqVkVy7UznWPXjmzSdBOXC2RukdoboY
PhcWUveYpZ7jIWC5vlfuCssaCa7IhM0m2+PNzMxSdNaHc/TVZyGX2hsoVOWU
qDT6BItjZdBi2zJl9Pe31NVrT4I/VbGcJjLAYYTZ7thOoKg6bHnMS9+12GaF
KBN2fefOPqHnnSMFsR4nMoR5YBYk0UWbil7SwcoYzdpLWSOi3W45Doq1Qs+t
ZfEUuhWcPFmuszeqqNqraCtohXNrLvvw0ME+VLUofZx2UJAM/yUr7JEzoXsK
mp08WALmjTmWPc13WVzk89+pfkcRrHD4hZhkb8KaGnt8WPkdBVJdIBzs5F1Y
TUJl0TgvqZFyRBRrhZ4vBuMQ3rax8W7wQcEaJZ8f7RdiHqjLRGbn4XCssO1X
wvAExyC26rNd4RofFHSmsKoDCz1vrVWi7LeI7FRWEquB/glnJieFXrpFLEAt
jCJM/JHhSVTVKnl7icz20PNYQZCwXvv44RvVL7fYxHIu+d2Hav7YvMDjsVta
QQhlJjIxfWd0nzNewkkXhpzqFxfIS1624nyCZMd4M+lBof4wBjyjph80eErU
MBQCH3tjG4SPtxgrCqfKhO/II/e/ikbFb5Hh5tM=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 148},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->368591478]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->745802587],

Cell[TextData[{
 "The variance is the same as the mean, so Poisson density functions spread \
out as ",
 Cell[BoxData[
  StyleBox["\[Mu]", "TR"]], "InlineFormula"],
 " increases:"
}], "ExampleText",
 CellID->25168],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"PDF", "[", 
     RowBox[{
      RowBox[{"PoissonDistribution", "[", "\[Mu]", "]"}], ",", "k"}], "]"}], 
    ",", 
    RowBox[{"{", 
     RowBox[{"k", ",", "0", ",", "50"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"\[Mu]", ",", "1", ",", "50"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->26934],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 229},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->378924046]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->8611],

Cell["\<\
The probability of getting negative integers or non-integer real numbers is \
zero:\
\>", "ExampleText",
 CellID->4200],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"PoissonDistribution", "[", "\[Mu]", "]"}], ",", "\[Pi]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->13041],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->775847824]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1273],

Cell["Moments can be obtained from the characteristic function:", \
"ExampleText",
 CellID->3912],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ExpectedValue", "[", 
  RowBox[{
   RowBox[{"k", "^", "4"}], ",", 
   RowBox[{"PoissonDistribution", "[", "\[Mu]", "]"}], ",", "k"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->7189],

Cell[BoxData[
 RowBox[{"\[Mu]", "+", 
  RowBox[{"7", " ", 
   SuperscriptBox["\[Mu]", "2"]}], "+", 
  RowBox[{"6", " ", 
   SuperscriptBox["\[Mu]", "3"]}], "+", 
  SuperscriptBox["\[Mu]", "4"]}]], "Output",
 ImageSize->{107, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->83785060]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Limit", "[", 
  RowBox[{
   RowBox[{"D", "[", 
    RowBox[{
     RowBox[{"CharacteristicFunction", "[", 
      RowBox[{
       RowBox[{"PoissonDistribution", "[", "\[Mu]", "]"}], ",", "t"}], "]"}], 
     ",", 
     RowBox[{"{", 
      RowBox[{"t", ",", "4"}], "}"}]}], "]"}], ",", 
   RowBox[{"t", "\[Rule]", "0"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->7223],

Cell[BoxData[
 RowBox[{"\[Mu]", " ", 
  RowBox[{"(", 
   RowBox[{"1", "+", 
    RowBox[{"7", " ", "\[Mu]"}], "+", 
    RowBox[{"6", " ", 
     SuperscriptBox["\[Mu]", "2"]}], "+", 
    SuperscriptBox["\[Mu]", "3"]}], ")"}]}]], "Output",
 ImageSize->{125, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->92611849]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->932691717],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NegativeBinomialDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NegativeBinomialDistribution"]], "InlineFormula"],
 " is a mixture of ",
 Cell[BoxData[
  ButtonBox["PoissonDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PoissonDistribution"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["GammaDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/GammaDistribution"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->217493462],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"poispdf", "=", 
  RowBox[{"PDF", "[", 
   RowBox[{
    RowBox[{"PoissonDistribution", "[", "\[Mu]", "]"}], ",", "k"}], 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->364101735],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"-", "\[Mu]"}]], " ", 
   SuperscriptBox["\[Mu]", "k"]}], 
  RowBox[{"k", "!"}]]], "Output",
 ImageSize->{43, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->397040062]
}, Open  ]],

Cell[BoxData[
 RowBox[{
  RowBox[{"gdist", "=", 
   RowBox[{"GammaDistribution", "[", 
    RowBox[{"n", ",", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"1", "-", "p"}], ")"}], "/", "p"}]}], "]"}]}], ";"}]], "Input",\

 CellLabel->"In[2]:=",
 CellID->437860854],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ev", "=", 
  RowBox[{"ExpectedValue", "[", 
   RowBox[{"poispdf", ",", "gdist", ",", "\[Mu]", ",", 
    RowBox[{"Assumptions", "\[Rule]", 
     RowBox[{"k", ">", "0"}]}]}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->221897965],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox[
    RowBox[{"(", 
     RowBox[{"1", "-", "p"}], ")"}], "k"], " ", 
   SuperscriptBox["p", "n"], " ", 
   RowBox[{"Gamma", "[", 
    RowBox[{"k", "+", "n"}], "]"}]}], 
  RowBox[{
   RowBox[{"k", "!"}], " ", 
   RowBox[{"Gamma", "[", "n", "]"}]}]]], "Output",
 ImageSize->{150, 35},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->108516494]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"PDF", "[", 
    RowBox[{
     RowBox[{"NegativeBinomialDistribution", "[", 
      RowBox[{"n", ",", "p"}], "]"}], ",", "k"}], "]"}], "\[Equal]", "ev"}], "//",
   "FullSimplify"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->15690664],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->454494932]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->213522293],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["GeometricDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/GeometricDistribution"]], "InlineFormula"],
 " is a mixture of ",
 Cell[BoxData[
  ButtonBox["PoissonDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PoissonDistribution"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["GammaDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/GammaDistribution"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->544004688],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"poispdf", "=", 
  RowBox[{"PDF", "[", 
   RowBox[{
    RowBox[{"PoissonDistribution", "[", "\[Mu]", "]"}], ",", "k"}], 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->117157743],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"-", "\[Mu]"}]], " ", 
   SuperscriptBox["\[Mu]", "k"]}], 
  RowBox[{"k", "!"}]]], "Output",
 ImageSize->{43, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->33719711]
}, Open  ]],

Cell[BoxData[
 RowBox[{
  RowBox[{"gdist", "=", 
   RowBox[{"GammaDistribution", "[", 
    RowBox[{"1", ",", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"1", "-", "p"}], ")"}], "/", "p"}]}], "]"}]}], ";"}]], "Input",\

 CellLabel->"In[2]:=",
 CellID->631382887],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ExpectedValue", "[", 
  RowBox[{"poispdf", ",", "gdist", ",", "\[Mu]", ",", 
   RowBox[{"Assumptions", "\[Rule]", 
    RowBox[{"k", ">", "0"}]}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->418493666],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"1", "-", "p"}], ")"}], "k"], " ", "p"}]], "Output",
 ImageSize->{62, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->122531099]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"GeometricDistribution", "[", "p", "]"}], ",", "k"}], 
  "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->245155988],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"1", "-", "p"}], ")"}], "k"], " ", "p"}]], "Output",
 ImageSize->{62, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->287015012]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->493488621],

Cell[TextData[{
 "For large ",
 Cell[BoxData[
  FormBox["\[Mu]", TraditionalForm]], "InlineMath"],
 " ",
 Cell[BoxData[
  ButtonBox["PoissonDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PoissonDistribution"]], "InlineFormula"],
 " can be approximated by ",
 Cell[BoxData[
  ButtonBox["NormalDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NormalDistribution"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->248323822],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"k", ",", 
      RowBox[{"PDF", "[", 
       RowBox[{
        RowBox[{"PoissonDistribution", "[", 
         RowBox[{"10", "^", "3"}], "]"}], ",", "k"}], "]"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"k", ",", "900", ",", "1100"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->260560305],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 106},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->167647690]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"PDF", "[", 
    RowBox[{
     RowBox[{"NormalDistribution", "[", 
      RowBox[{
       RowBox[{"10", "^", "3"}], ",", 
       RowBox[{"Sqrt", "[", 
        RowBox[{"10", "^", "3"}], "]"}]}], "]"}], ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "900", ",", "1100"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->49751438],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 106},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->386378781]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[Cell[BoxData[
   InterpretationBox[Cell[TextData[{
     "Possible Issues",
     "\[NonBreakingSpace]\[NonBreakingSpace]",
     Cell["(2)", "ExampleCount"]
    }], "ExampleSection"],
    $Line = 0; Null]]]], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->18178],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["PoissonDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PoissonDistribution"]], "InlineFormula"],
 " is not defined when ",
 Cell[BoxData[
  StyleBox["\[Mu]", "TR"]], "InlineFormula"],
 " is non-positive:"
}], "ExampleText",
 CellID->30739],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"PoissonDistribution", "[", 
   RowBox[{"-", "1.5"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->31503],

Cell[BoxData[
 RowBox[{
  RowBox[{"PoissonDistribution", "::", "\<\"posparm\"\>"}], ":", 
  " ", "\<\"Parameter \\!\\(-1.5`\\) is expected to be positive.\"\>"}]], \
"Message", "MSG",
 CellID->44389108],

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"PoissonDistribution", "[", 
   RowBox[{"-", "1.5`"}], "]"}], "]"}]], "Output",
 ImageSize->{222, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->522187181]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->2463],

Cell["\<\
Substitution of invalid parameters into symbolic outputs gives results that \
are not meaningful:\
\>", "ExampleText",
 CellID->16727],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Mean", "[", 
   RowBox[{"PoissonDistribution", "[", "\[Mu]", "]"}], "]"}], "/.", 
  RowBox[{"{", 
   RowBox[{"\[Mu]", "->", 
    RowBox[{"-", "1"}]}], "}"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->29801],

Cell[BoxData[
 RowBox[{"-", "1"}]], "Output",
 ImageSize->{19, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->319596984]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->8597],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["BinomialDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BinomialDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["GammaDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/GammaDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->4322]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->16981],

Cell[TextData[ButtonBox["Discrete Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/DiscreteDistributions"]], "Tutorials",
 CellID->4738]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection"],

Cell[TextData[{
 ButtonBox["Demonstrations with PoissonDistribution",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/symbol.html?symbol=\
PoissonDistribution"], None}],
 " (",
 ButtonBox["Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None}],
 ")"
}], "RelatedLinks"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Discrete Statistical Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DiscreteStatisticalDistributions"]], "MoreAbout",
 CellID->14967796],

Cell[TextData[ButtonBox["Functions Used in Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/FunctionsUsedInStatistics"]], "MoreAbout",
 CellID->281584308],

Cell[TextData[ButtonBox["Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/MathematicalFunctions"]], "MoreAbout",
 CellID->294608179],

Cell[TextData[ButtonBox["Statistical Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/StatisticalDistributions"]], "MoreAbout",
 CellID->64237961],

Cell[TextData[ButtonBox["Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/Statistics"]], "MoreAbout",
 CellID->762927168],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->557306967],

Cell[TextData[ButtonBox["New in 6.0: Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60Statistics"]], "MoreAbout",
 CellID->110748785]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"PoissonDistribution - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 10, 31.3857176}", "context" -> "System`", 
    "keywords" -> {
     "infrequent events", "non-frequent events", "Poisson distribution", 
      "Poisson process", "rare events"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "PoissonDistribution[\\[Mu]] represents a Poisson distribution with mean \
\\[Mu].", "synonyms" -> {"poisson distribution"}, "title" -> 
    "PoissonDistribution", "type" -> "Symbol", "uri" -> 
    "ref/PoissonDistribution"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[7936, 241, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->619271192]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 43449, 1417}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2948, 67, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3571, 94, 1584, 37, 70, "ObjectNameGrid"],
Cell[5158, 133, 592, 18, 70, "Usage",
 CellID->28458]
}, Open  ]],
Cell[CellGroupData[{
Cell[5787, 156, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6265, 173, 625, 23, 70, "Notes",
 CellID->429991538],
Cell[6893, 198, 299, 10, 70, "Notes",
 CellID->132305407],
Cell[7195, 210, 704, 26, 70, "Notes",
 CellID->33143606]
}, Closed]],
Cell[CellGroupData[{
Cell[7936, 241, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->619271192],
Cell[CellGroupData[{
Cell[8321, 255, 148, 5, 70, "ExampleSection",
 CellID->609804345],
Cell[8472, 262, 155, 4, 70, "ExampleText",
 CellID->1864],
Cell[CellGroupData[{
Cell[8652, 270, 146, 4, 28, "Input",
 CellID->20566],
Cell[8801, 276, 161, 5, 36, "Output",
 CellID->20978886]
}, Open  ]],
Cell[CellGroupData[{
Cell[8999, 286, 150, 4, 28, "Input",
 CellID->27931],
Cell[9152, 292, 162, 5, 36, "Output",
 CellID->137000638]
}, Open  ]],
Cell[9329, 300, 121, 3, 70, "ExampleDelimiter",
 CellID->26756],
Cell[9453, 305, 115, 2, 70, "ExampleText",
 CellID->7184],
Cell[CellGroupData[{
Cell[9593, 311, 171, 6, 70, "Input",
 CellID->2065],
Cell[9767, 319, 310, 11, 54, "Output",
 CellID->86344222]
}, Open  ]],
Cell[CellGroupData[{
Cell[10114, 335, 456, 15, 28, "Input",
 CellID->9265],
Cell[10573, 352, 1933, 36, 162, "Output",
 Evaluatable->False,
 CellID->215144501]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[12555, 394, 215, 7, 70, "ExampleSection",
 CellID->4389],
Cell[12773, 403, 115, 3, 70, "ExampleText",
 CellID->25402],
Cell[CellGroupData[{
Cell[12913, 410, 179, 6, 70, "Input",
 CellID->25411],
Cell[13095, 418, 291, 9, 36, "Output",
 CellID->316718784]
}, Open  ]],
Cell[13401, 430, 121, 3, 70, "ExampleDelimiter",
 CellID->31807],
Cell[13525, 435, 83, 1, 70, "ExampleText",
 CellID->20699375],
Cell[CellGroupData[{
Cell[13633, 440, 150, 4, 70, "Input",
 CellID->24407],
Cell[13786, 446, 194, 7, 56, "Output",
 CellID->447878400]
}, Open  ]],
Cell[CellGroupData[{
Cell[14017, 458, 149, 4, 70, "Input",
 CellID->7943],
Cell[14169, 464, 205, 7, 53, "Output",
 CellID->748461605]
}, Open  ]],
Cell[14389, 474, 120, 3, 70, "ExampleDelimiter",
 CellID->3112],
Cell[14512, 479, 77, 1, 70, "ExampleText",
 CellID->8231],
Cell[CellGroupData[{
Cell[14614, 484, 216, 7, 70, "Input",
 CellID->10677],
Cell[14833, 493, 272, 9, 36, "Output",
 CellID->9267117]
}, Open  ]],
Cell[15120, 505, 120, 3, 70, "ExampleDelimiter",
 CellID->1206],
Cell[15243, 510, 205, 7, 70, "ExampleText",
 CellID->20401],
Cell[CellGroupData[{
Cell[15473, 521, 180, 6, 70, "Input",
 CellID->21435],
Cell[15656, 529, 157, 5, 36, "Output",
 CellID->74850141]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[15862, 540, 223, 7, 70, "ExampleSection",
 CellID->16811],
Cell[16088, 549, 252, 7, 70, "ExampleText",
 CellID->13370],
Cell[CellGroupData[{
Cell[16365, 560, 289, 10, 70, "Input",
 CellID->26124],
Cell[16657, 572, 1656, 31, 70, "Output",
 Evaluatable->False,
 CellID->368591478]
}, Open  ]],
Cell[18328, 606, 125, 3, 70, "ExampleDelimiter",
 CellID->745802587],
Cell[18456, 611, 214, 7, 70, "ExampleText",
 CellID->25168],
Cell[CellGroupData[{
Cell[18695, 622, 412, 13, 70, "Input",
 CellID->26934],
Cell[19110, 637, 3315, 58, 70, "Output",
 Evaluatable->False,
 CellID->378924046]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[22474, 701, 232, 7, 70, "ExampleSection",
 CellID->8611],
Cell[22709, 710, 129, 4, 70, "ExampleText",
 CellID->4200],
Cell[CellGroupData[{
Cell[22863, 718, 176, 6, 70, "Input",
 CellID->13041],
Cell[23042, 726, 158, 5, 36, "Output",
 CellID->775847824]
}, Open  ]],
Cell[23215, 734, 120, 3, 70, "ExampleDelimiter",
 CellID->1273],
Cell[23338, 739, 97, 2, 70, "ExampleText",
 CellID->3912],
Cell[CellGroupData[{
Cell[23460, 745, 215, 7, 70, "Input",
 CellID->7189],
Cell[23678, 754, 336, 11, 36, "Output",
 CellID->83785060]
}, Open  ]],
Cell[CellGroupData[{
Cell[24051, 770, 396, 13, 70, "Input",
 CellID->7223],
Cell[24450, 785, 366, 12, 39, "Output",
 CellID->92611849]
}, Open  ]],
Cell[24831, 800, 125, 3, 70, "ExampleDelimiter",
 CellID->932691717],
Cell[24959, 805, 528, 17, 70, "ExampleText",
 CellID->217493462],
Cell[CellGroupData[{
Cell[25512, 826, 208, 7, 70, "Input",
 CellID->364101735],
Cell[25723, 835, 311, 11, 54, "Output",
 CellID->397040062]
}, Open  ]],
Cell[26049, 849, 267, 10, 70, "Input",
 CellID->437860854],
Cell[CellGroupData[{
Cell[26341, 863, 258, 7, 70, "Input",
 CellID->221897965],
Cell[26602, 872, 446, 16, 56, "Output",
 CellID->108516494]
}, Open  ]],
Cell[CellGroupData[{
Cell[27085, 893, 278, 9, 70, "Input",
 CellID->15690664],
Cell[27366, 904, 161, 5, 36, "Output",
 CellID->454494932]
}, Open  ]],
Cell[27542, 912, 125, 3, 70, "ExampleDelimiter",
 CellID->213522293],
Cell[27670, 917, 514, 17, 70, "ExampleText",
 CellID->544004688],
Cell[CellGroupData[{
Cell[28209, 938, 208, 7, 70, "Input",
 CellID->117157743],
Cell[28420, 947, 310, 11, 54, "Output",
 CellID->33719711]
}, Open  ]],
Cell[28745, 961, 267, 10, 70, "Input",
 CellID->631382887],
Cell[CellGroupData[{
Cell[29037, 975, 231, 6, 70, "Input",
 CellID->418493666],
Cell[29271, 983, 253, 9, 36, "Output",
 CellID->122531099]
}, Open  ]],
Cell[CellGroupData[{
Cell[29561, 997, 174, 6, 70, "Input",
 CellID->245155988],
Cell[29738, 1005, 253, 9, 36, "Output",
 CellID->287015012]
}, Open  ]],
Cell[30006, 1017, 125, 3, 70, "ExampleDelimiter",
 CellID->493488621],
Cell[30134, 1022, 457, 16, 70, "ExampleText",
 CellID->248323822],
Cell[CellGroupData[{
Cell[30616, 1042, 426, 13, 70, "Input",
 CellID->260560305],
Cell[31045, 1057, 2559, 46, 70, "Output",
 Evaluatable->False,
 CellID->167647690]
}, Open  ]],
Cell[CellGroupData[{
Cell[33641, 1108, 412, 13, 70, "Input",
 CellID->49751438],
Cell[34056, 1123, 2539, 46, 70, "Output",
 Evaluatable->False,
 CellID->386378781]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[36644, 1175, 327, 9, 70, "ExampleSection",
 CellID->18178],
Cell[36974, 1186, 299, 10, 70, "ExampleText",
 CellID->30739],
Cell[CellGroupData[{
Cell[37298, 1200, 163, 5, 70, "Input",
 CellID->31503],
Cell[37464, 1207, 202, 5, 70, "Message",
 CellID->44389108],
Cell[37669, 1214, 257, 8, 36, "Output",
 CellID->522187181]
}, Open  ]],
Cell[37941, 1225, 120, 3, 70, "ExampleDelimiter",
 CellID->2463],
Cell[38064, 1230, 144, 4, 70, "ExampleText",
 CellID->16727],
Cell[CellGroupData[{
Cell[38233, 1238, 243, 8, 70, "Input",
 CellID->29801],
Cell[38479, 1248, 175, 6, 36, "Output",
 CellID->319596984]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[38715, 1261, 310, 9, 70, "SeeAlsoSection",
 CellID->8597],
Cell[39028, 1272, 492, 17, 70, "SeeAlso",
 CellID->4322]
}, Open  ]],
Cell[CellGroupData[{
Cell[39557, 1294, 314, 9, 70, "TutorialsSection",
 CellID->16981],
Cell[39874, 1305, 153, 3, 70, "Tutorials",
 CellID->4738]
}, Open  ]],
Cell[CellGroupData[{
Cell[40064, 1313, 305, 8, 70, "RelatedLinksSection"],
Cell[40372, 1323, 380, 12, 70, "RelatedLinks"]
}, Open  ]],
Cell[CellGroupData[{
Cell[40789, 1340, 299, 8, 70, "MoreAboutSection"],
Cell[41091, 1350, 177, 3, 70, "MoreAbout",
 CellID->14967796],
Cell[41271, 1355, 165, 3, 70, "MoreAbout",
 CellID->281584308],
Cell[41439, 1360, 155, 3, 70, "MoreAbout",
 CellID->294608179],
Cell[41597, 1365, 160, 3, 70, "MoreAbout",
 CellID->64237961],
Cell[41760, 1370, 132, 3, 70, "MoreAbout",
 CellID->762927168],
Cell[41895, 1375, 179, 3, 70, "MoreAbout",
 CellID->557306967],
Cell[42077, 1380, 151, 3, 70, "MoreAbout",
 CellID->110748785]
}, Open  ]],
Cell[42243, 1386, 27, 0, 70, "History"],
Cell[42273, 1388, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

