(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    150846,       4099]
NotebookOptionsPosition[    134802,       3574]
NotebookOutlinePosition[    137509,       3650]
CellTagsIndexPosition[    137204,       3640]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Special Functions" :> 
          Documentation`HelpLookup["paclet:tutorial/SpecialFunctions"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Special Functions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Gamma" :> Documentation`HelpLookup["paclet:ref/Gamma"], 
          "LogGamma" :> Documentation`HelpLookup["paclet:ref/LogGamma"], 
          "EulerGamma" :> Documentation`HelpLookup["paclet:ref/EulerGamma"], 
          "HarmonicNumber" :> 
          Documentation`HelpLookup["paclet:ref/HarmonicNumber"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Gamma\"\>", 2->"\<\"LogGamma\"\>", 
       3->"\<\"EulerGamma\"\>", 
       4->"\<\"HarmonicNumber\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Gamma Functions and Related Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/GammaFunctionsAndRelatedFunctions"], 
          "Recurrence and Sum Functions" :> 
          Documentation`HelpLookup["paclet:guide/RecurrenceAndSumFunctions"], 
          "Special Functions" :> 
          Documentation`HelpLookup["paclet:guide/SpecialFunctions"], 
          "New in 6.0: Mathematical Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicalFunctions"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Gamma Functions and Related Functions\"\>", 
       2->"\<\"Recurrence and Sum Functions\"\>", 
       3->"\<\"Special Functions\"\>", 
       4->"\<\"New in 6.0: Mathematical Functions\"\>"}, "\<\"more about\"\>",
        
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["PolyGamma", "ObjectName"], Cell[BoxData[
     TooltipBox[
      TogglerBox[Dynamic[
        FrontEnd`CurrentValue[
         FrontEnd`EvaluationNotebook[], {
         TaggingRules, "ModificationHighlight"}]], {True->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVVly2zAMDQhxEylSsizLaxJn73KafvQAuUDv/1eApJuosceeTDAWF4B8
eFgk71///HoVV1dXK3p+0wNfIzdCLL4I6p1sYBYuPy3SI2AmRBpuaE/TPRsE
3Isg8mrB9l6ILQhhLkbd93MBuwVxAthdZwPr5wVVzBc8wAvMX8TFqPkqE8q6
rDCsKPjZ/g1guz+PKvLFfGsDU1R4A8zDhgLangX9x/WGM7A1qSQ7rraYkaLf
LHZ5FWY9H7rew/n4/6vWPKT2mXFBaH/NGejzqi/VnFEFJ2Rb1Lo+jtofllPD
RVJjdZLr51Eb/ED1K+QI109LcIg+sdSIWqa5g85L6MiNoRloTws+pdBKiFGl
mx3N0jcAlcsKeFivVkteVOgMSNsC18sYS/iIMdZoK43WSdQVK1SLlYy09bZ2
aEO+aoPiACNak0EBbtfLRIz9KJRcroaSKwE11BprpRHqiJ0BrxtJG4pLYqWc
RZO4BQwSbGT4rFivaLjjQdscjSvlih48Mu+WDisOBcB5ehgeDEZQlikkKh3b
KA1FMTDg9yFxLSe4XC37begHTDh5seS1I/gaW025Yo/yUFbmSalShzon1Kch
JTNn2hKS1hyWVIkf5n2LHEQgt0zeUNTkMaYrDdkMZdo3fJCDHx7hxzAC+8rs
XYBokTPfEJk6JVlZx3Yl0UnTYMPvSUt6zghf8dhIi5VDnwO+Hcb1MHK1msK+
oy5qkYS7wVC3cfiq8ojBcMtFMJXnDiIcLPFJClDFGkJBhXGEZV6VcCQFoey7
tyvBF6lw0uKH6nyQ5fqhrKx/Kxe8+3coucpGN4U5+RYux7vnhGO5P7hKenqz
xSO3ivjisf345fg5Pt1ScCRkkzR1U1Dur+NCec4LZ49Ynx9P04GmOvlnqcKZ
79BfJfcjkg==
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlLlKQ0EYRi/iA9gpWIgvIXbjM1iKWAjW2vgeZnWN+AgWZhEULCxELFyi
ZnfJavbeJOr/33xXcj8CNnY5EA7k3GWYmTuzq+uLa2OWZS38/MYt0DbDHH8Q
8uTB64bRNrmsQkaPRoQSGX1ZqZPR9/eEazL6klIlowf8wiUZ3ecV3snowYDw
SEafUmJk9HmlQh4cX4GM7vUId2T0VFIho28or2T0w5DQIKN3PoUaGX1FaZLR
0ymh6sz7hHLk9PMzoU5Gz6SFmvFsCbd9Xzj9qyc0nfe49xfuTyYUEzoQbmh8
+N8e37HTbVpk9MSzUCajbypFMvq0UiOjzyklMnqvKzRMJCwU+/vjzemTSpns
nh97feJmd0e4cvppTKiQ0bsdoW5ecorZDgpPtD6xqFA2fp+Qp/vtdUj1v797
Oqdwnb2Ov++fUVpk9PCJUCC7n4t5wz5xj9/+rj6ceXbPP86NbEb58/k56qP+
n/0bdGyYqQ==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}], False->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVdd22zAMDUBxSaJm5HgljrPTJN177/nWh35AfqD//1aAVGrrJLbbk+BI
FEmQFxcXlDQ9//3zHNfW1gZ0/6IbbsYOhahvCGrO9qHM/mnhmMMLoKv0RD6V
cPyZHTTcDI5NkYnQq4UQ0BdiAkKYZagl1OOw5fWtcgyvPp7AnefsGG0DBIeA
9RZVrNfcwBasb4nlbP9u4cbc/XD/zXcIgC2qYYazJTQiGSYHS0Hr8Yu5LV/h
ePvR5oMOKswAQ0M8R5OloFMWSNRE+IAVeALv38KX26wAV1uU5Ojv16PQy8q+
gMP+9ABW5C9KbkZi6qv19DGc/ID9b+CrZTjbbVagH3p91oIcL+GZ6JDNUet4
HpWUXxx5BaWZxRh1FQgH6JqoKcarF/2/dblez7IE0XmWGlFL/6ygchIqCmPo
CTSmDq9SaCUUhfI7K3pKlwJESZiAveHGxoA7ESYGpM2B62WMJXzEoojRRhpt
IlFHPKFyjGRBQ2fjBG0WttpMcYIFWhNAAXaHA0+M4yiUXK6UxJWAGmKNsdII
cYGVAadTSQPKS2KkEovGc8swk2ALhg8Tww1qdrjRNmSTtOUqHDhk3jktVpwK
QOLoZngwWICyTMFTqdhHMrQTDQPeazzXdgWXK+e4KV3AhH0US1Ergo8x16QV
R5QXZWWeJJW6qLNHPWq8mEFpS0hac1pSeX4YxjlyEhmFZfKGsqaIhd+Sks+Q
0i7lhZx8cwYPmx5wrMA+yaCwyMqnRCb2IiubsF9JTKRJMeX3JKd5VoS3OEyl
xShBFxLebXrDpsfVSlv2FZ2iHMn4NBg6bZy+ihxiZvjIFWAixyeIcLDNT1KC
qogha1Gh14NB6LXpSEpC2bm3y8O3FmHniF9U55INhnttz7pZuWDu79BqFZxJ
F2bhWzjo7Zx6HMvng6ukuztzvGJXa66NmF/+crzrHe1ScmTkk/SouqB8vq42
0jl0EnuF9/RsMR1Io4U/S5Wt+A79AaDyK/w=
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlUtrE1EYhgfxBwguFBTEvyAiFhfzHxRFvC0E17rQhYLiT7D3u/gTXNg2
3sBFFyIutE1qm17UZJKZZJK0KFjBtur7Td4jnhe7dTMPhAfyQObMnO9Mjl69
fvraniAI7v/+7A3IRri7/8VGWJwHVTH72qohZp+ZBnUx+yWjJWYfHwNvxOwX
jKaYfXAAzIrZ+/tARcw+NAhKYvaTRiJmP2gUxH+vLxKzl5cMMXtfL3gvZn84
Cdpi9hvGJzH7yxegJWa/YnTE7Mtl0HTPfZ/xyPWVZZCGvQ/Au65fSV9aNMLJ
CfDW9a0fIBWz/9wBHbcOf/74PX83W99j128ZNTH71BMQidmfFkAiZj9irIvZ
Fz+AWMx+yEjF7KeMVOw/32x/iuHoCHjt+s42aIfTU6DWna/Prm9vgVb4cc0I
h4fAgusZ62L2A0YsZj9h1MX+9bN9LHfP75zrZ42a2L9/3+wXjVTM7q8jm9NE
5q8wA+JwoB/8eX+eNxIx+00jErP3GA2xvy7uG88R+z2jI2Y/bLTE/v1l76WG
mxP/+r7ZN7+Bupj9mFERsz9/Bppi9myeEzH7QgnUxOy3jYaY/a7RFrNfNtpi
9nNGXcw+NgoaYnbOk2///PJ/bXXFcP37JojF7CPDIBazf/0CKmL2+TlQEbPf
MZpi9olxkIrZS0UQidn3G1Ux+3EjErOfMapi9mBX8p73vOc973n/P/0XnFJ2
Rg==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}]}, "\"\<Click me\>\""],
      "Toggle changes in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["PolyGamma",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/PolyGamma"], "[", 
       StyleBox["z", "TI"], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the digamma function ",
     Cell[BoxData[
      FormBox[
       RowBox[{"\[Psi]", "(", "z"}], TraditionalForm]], "InlineMath"],
     "). "
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["PolyGamma",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/PolyGamma"], "[", 
       RowBox[{
        StyleBox["n", "TI"], ",", 
        StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the ",
     Cell[BoxData[
      FormBox[
       SuperscriptBox[
        StyleBox["n", "TI"], "th"], TraditionalForm]]],
     " derivative of the digamma function ",
     Cell[BoxData[
      FormBox[
       RowBox[{
        SuperscriptBox["\[Psi]", 
         RowBox[{"(", "n", ")"}]], "(", "z", ")"}], TraditionalForm]], 
      "InlineMath"],
     ". "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->10657]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["\<\
Mathematical function, suitable for both symbolic and numerical manipulation.\
\
\>", "Notes",
 CellID->20640],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["PolyGamma",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/PolyGamma"], "[", 
   StyleBox["z", "TI"], "]"}]], "InlineFormula"],
 " is the logarithmic derivative of the gamma function, given by ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"\[Psi]", "(", "z", ")"}], "=", 
    RowBox[{
     RowBox[{
      SuperscriptBox["\[CapitalGamma]", "\[Prime]"], "(", "z", ")"}], "/", 
     RowBox[{"\[CapitalGamma]", "(", "z", ")"}]}]}], TraditionalForm]], 
  "InlineMath"],
 ". "
}], "Notes",
 CellID->23946],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["PolyGamma",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/PolyGamma"], "[", 
   RowBox[{
    StyleBox["n", "TI"], ",", 
    StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
 " is given for positive integer ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " by ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SuperscriptBox["\[Psi]", 
      RowBox[{"(", "n", ")"}]], "(", "z", ")"}], "=", 
    RowBox[{
     SuperscriptBox["d", "n"], 
     RowBox[{
      RowBox[{"\[Psi]", "(", "z", ")"}], "/", "d"}], 
     SuperscriptBox["z", "n"]}]}], TraditionalForm]], "InlineMath"],
 ". "
}], "Notes",
 CellID->29958],

Cell[TextData[{
 "For arbitrary complex ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " the polygamma function is defined by fractional calculus analytic \
continuation."
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->358863611],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["PolyGamma",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/PolyGamma"], "[", 
   StyleBox["z", "TI"], "]"}]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["PolyGamma",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/PolyGamma"], "[", 
   RowBox[{
    StyleBox["n", "TI"], ",", 
    StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
 " are meromorphic functions of ",
 Cell[BoxData[
  StyleBox["z", "TI"]], "InlineFormula"],
 " with no branch cut discontinuities. "
}], "Notes",
 CellID->2263],

Cell[TextData[{
 "For certain special arguments, ",
 Cell[BoxData[
  ButtonBox["PolyGamma",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PolyGamma"]], "InlineFormula"],
 " automatically evaluates to exact values."
}], "Notes",
 CellID->283848419],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["PolyGamma",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PolyGamma"]], "InlineFormula"],
 " can be evaluated to arbitrary numerical precision."
}], "Notes",
 CellID->31226],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["PolyGamma",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PolyGamma"]], "InlineFormula"],
 " automatically threads over lists."
}], "Notes",
 CellID->28162]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->30414486],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->33776721],

Cell["Evaluate the digamma function:", "ExampleText",
 CellID->121247567],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PolyGamma", "[", "5", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->402316662],

Cell[BoxData[
 RowBox[{
  FractionBox["25", "12"], "-", "EulerGamma"}]], "Output",
 ImageSize->{105, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->39210521]
}, Open  ]],

Cell["Evaluate quadro\[Hyphen]gamma:", "ExampleText",
 CellID->323317997],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PolyGamma", "[", 
  RowBox[{"3", ",", "5"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->58503667],

Cell[BoxData[
 RowBox[{"6", " ", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"-", 
     FractionBox["22369", "20736"]}], "+", 
    FractionBox[
     SuperscriptBox["\[Pi]", "4"], "90"]}], ")"}]}]], "Output",
 ImageSize->{108, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->279621243]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->573998195],

Cell["Derivative of the gamma function:", "ExampleText",
 CellID->3935701],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"D", "[", 
  RowBox[{
   RowBox[{"Gamma", "[", "z", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"z", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->407145307],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"Gamma", "[", "z", "]"}], " ", 
   SuperscriptBox[
    RowBox[{"PolyGamma", "[", 
     RowBox[{"0", ",", "z"}], "]"}], "2"]}], "+", 
  RowBox[{
   RowBox[{"Gamma", "[", "z", "]"}], " ", 
   RowBox[{"PolyGamma", "[", 
    RowBox[{"1", ",", "z"}], "]"}]}]}]], "Output",
 ImageSize->{345, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->397183923]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter"],

Cell["The digamma function:", "ExampleText",
 CellID->864686355],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"PolyGamma", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "3"}], ",", "3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:="],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 145},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->380678060]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(9)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection"],

Cell["Evaluate for integer arguments of any size:", "ExampleText"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PolyGamma", "[", "100", "]"}]], "Input",
 CellLabel->"In[1]:="],

Cell[BoxData[
 RowBox[{
  FractionBox["360968703235711654233892612988250163157207", 
   "69720375229712477164533808935312303556800"], "-", 
  "EulerGamma"}]], "Output",
 ImageSize->{413, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->114353727]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->127828245],

Cell["Evaluate numerically:", "ExampleText",
 CellID->215230372],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PolyGamma", "[", "100.5", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->466830376],

Cell[BoxData["4.605174352581845`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->40856817]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->351643097],

Cell["Evaluate for complex arguments and orders:", "ExampleText",
 CellID->334604611],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PolyGamma", "[", 
  RowBox[{"2.5", "+", 
   RowBox[{"3", "I"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->317365560],

Cell[BoxData[
 RowBox[{"1.2812739190662308`", "\[InvisibleSpace]", "+", 
  RowBox[{"0.9798053153445595`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{133, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->644053106]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PolyGamma", "[", 
  RowBox[{
   RowBox[{"6", "+", "I"}], ",", 
   RowBox[{"2.5", "+", 
    RowBox[{"3", "I"}]}]}], "]"}]], "Input",
 CellTags->"Ex--21958729656347312745,1",
 CellLabel->"In[2]:=",
 CellID->6334],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "0.47438897416335507`"}], "-", 
  RowBox[{"1.9894416905356176`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{141, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"Ex--21958729656347312745,1",
 CellLabel->"Out[2]=",
 CellID->362645953]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->80376993],

Cell["Evaluate to any precision:", "ExampleText",
 CellID->104147080],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"PolyGamma", "[", 
    RowBox[{"100", "/", "3"}], "]"}], ",", "50"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->123865482],

Cell[BoxData["3.\
4914829040670915487856413468881046026679932936762495177107318734198`50."], \
"Output",
 ImageSize->{361, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->539038579]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->547872452],

Cell["\<\
The precision of the output tracks the precision of the input:\
\>", "ExampleText",
 CellID->230370911],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PolyGamma", "[", "2.20000000000000000000000", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->82935731],

Cell[BoxData["0.54429343674114503778612537088338119282`25.988029395705418"], \
"Output",
 ImageSize->{200, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->44141884]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->788580244],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["PolyGamma",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PolyGamma"]], "InlineFormula"],
 " threads element-wise over lists:"
}], "ExampleText",
 CellID->67387947],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PolyGamma", "[", 
  RowBox[{"{", 
   RowBox[{"1", ",", "2", ",", "3", ",", "4", ",", "5"}], "}"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->277880383],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "EulerGamma"}], ",", 
   RowBox[{"1", "-", "EulerGamma"}], ",", 
   RowBox[{
    FractionBox["3", "2"], "-", "EulerGamma"}], ",", 
   RowBox[{
    FractionBox["11", "6"], "-", "EulerGamma"}], ",", 
   RowBox[{
    FractionBox["25", "12"], "-", "EulerGamma"}]}], "}"}]], "Output",
 ImageSize->{526, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->987201016]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->506155396],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["FunctionExpand",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FunctionExpand"]], "InlineFormula"],
 " to expand higher-order polygamma functions:"
}], "ExampleText",
 CellID->462536501],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"PolyGamma", "[", 
    RowBox[{"n", ",", "1"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "0", ",", "4"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->334350941],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "EulerGamma"}], ",", 
   FractionBox[
    SuperscriptBox["\[Pi]", "2"], "6"], ",", 
   RowBox[{"PolyGamma", "[", 
    RowBox[{"2", ",", "1"}], "]"}], ",", 
   FractionBox[
    SuperscriptBox["\[Pi]", "4"], "15"], ",", 
   RowBox[{"PolyGamma", "[", 
    RowBox[{"4", ",", "1"}], "]"}]}], "}"}]], "Output",
 ImageSize->{381, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->3501085]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FunctionExpand", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->11183202],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "EulerGamma"}], ",", 
   FractionBox[
    SuperscriptBox["\[Pi]", "2"], "6"], ",", 
   RowBox[{
    RowBox[{"-", "2"}], " ", 
    RowBox[{"Zeta", "[", "3", "]"}]}], ",", 
   FractionBox[
    SuperscriptBox["\[Pi]", "4"], "15"], ",", 
   RowBox[{
    RowBox[{"-", "24"}], " ", 
    RowBox[{"Zeta", "[", "5", "]"}]}]}], "}"}]], "Output",
 ImageSize->{317, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->122176922]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->500660245],

Cell["Series expansion:", "ExampleText",
 CellID->115422345],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"PolyGamma", "[", "z", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"z", ",", "1", ",", "3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->41981680],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
   RowBox[{"-", "EulerGamma"}], "+", 
   RowBox[{
    FractionBox["1", "6"], " ", 
    SuperscriptBox["\[Pi]", "2"], " ", 
    RowBox[{"(", 
     RowBox[{"z", "-", "1"}], ")"}]}], "+", 
   RowBox[{
    FractionBox["1", "2"], " ", 
    RowBox[{"PolyGamma", "[", 
     RowBox[{"2", ",", "1"}], "]"}], " ", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"z", "-", "1"}], ")"}], "2"]}], "+", 
   RowBox[{
    FractionBox["1", "90"], " ", 
    SuperscriptBox["\[Pi]", "4"], " ", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"z", "-", "1"}], ")"}], "3"]}], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", 
      RowBox[{"z", "-", "1"}], "]"}], "4"],
    SeriesData[$CellContext`z, 1, {}, 0, 4, 1],
    Editable->False]}],
  SeriesData[$CellContext`z, 
   1, {-EulerGamma, Rational[1, 6] Pi^2, Rational[1, 2] PolyGamma[2, 1], 
    Rational[1, 90] Pi^4}, 0, 4, 1],
  Editable->False]], "Output",
 ImageSize->{506, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->180030687]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->760443585],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TraditionalForm",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TraditionalForm"]], "InlineFormula"],
 " formatting:"
}], "ExampleText",
 CellID->653625731],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"PolyGamma", "[", "z", "]"}], "//", "TraditionalForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->35954786],

Cell[BoxData[
 FormBox[
  RowBox[{
   SuperscriptBox[
    TagBox["\[Psi]",
     PolyGamma], 
    RowBox[{"(", "0", ")"}]], "(", "z", ")"}], TraditionalForm]], "Output",
 ImageSize->{36, 17},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//TraditionalForm=",
 CellID->322398403]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"PolyGamma", "[", 
   RowBox[{"n", ",", "x"}], "]"}], "//", "TraditionalForm"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->633500976],

Cell[BoxData[
 FormBox[
  RowBox[{
   SuperscriptBox[
    TagBox["\[Psi]",
     PolyGamma], 
    RowBox[{"(", "n", ")"}]], "(", "x", ")"}], TraditionalForm]], "Output",
 ImageSize->{36, 17},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]//TraditionalForm=",
 CellID->236937360]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  InterpretationBox[Cell[TextData[{
    "Digamma Function",
    "\[NonBreakingSpace]\[NonBreakingSpace]",
    Cell["(2)", "ExampleCount"]
   }], "ExampleSubsection"],
   $Line = 0; Null], " "}]], "ExampleSubsection",
 CellID->136184792],

Cell["Infinite arguments can give symbolic results:", "ExampleText"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PolyGamma", "[", "Infinity", "]"}]], "Input",
 CellLabel->"In[1]:="],

Cell[BoxData["\[Infinity]"], "Output",
 ImageSize->{12, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->283554744]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter"],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["PolyGamma",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PolyGamma"]], "InlineFormula"],
 " can be applied to a power series:"
}], "ExampleText"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PolyGamma", "[", 
  RowBox[{
   RowBox[{"Exp", "[", "x", "]"}], "+", 
   SuperscriptBox[
    RowBox[{"O", "[", "x", "]"}], "4"]}], "]"}]], "Input",
 CellLabel->"In[1]:="],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
   RowBox[{"-", "EulerGamma"}], "+", 
   FractionBox[
    RowBox[{
     SuperscriptBox["\[Pi]", "2"], " ", "x"}], "6"], "+", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      FractionBox[
       SuperscriptBox["\[Pi]", "2"], "12"], "+", 
      RowBox[{
       FractionBox["1", "2"], " ", 
       RowBox[{"PolyGamma", "[", 
        RowBox[{"2", ",", "1"}], "]"}]}]}], ")"}], " ", 
    SuperscriptBox["x", "2"]}], "+", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      FractionBox[
       SuperscriptBox["\[Pi]", "2"], "36"], "+", 
      FractionBox[
       SuperscriptBox["\[Pi]", "4"], "90"], "+", 
      RowBox[{
       FractionBox["1", "2"], " ", 
       RowBox[{"PolyGamma", "[", 
        RowBox[{"2", ",", "1"}], "]"}]}]}], ")"}], " ", 
    SuperscriptBox["x", "3"]}], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "x", "]"}], "4"],
    SeriesData[$CellContext`x, 0, {}, 0, 4, 1],
    Editable->False]}],
  SeriesData[$CellContext`x, 
   0, {-EulerGamma, Rational[1, 6] Pi^2, Rational[1, 12] Pi^2 + 
    Rational[1, 2] PolyGamma[2, 1], Rational[1, 36] Pi^2 + 
    Rational[1, 90] Pi^4 + Rational[1, 2] PolyGamma[2, 1]}, 0, 4, 1],
  Editable->False]], "Output",
 ImageSize->{578, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->165117740]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter"],

Cell["Series expansion at poles:", "ExampleText"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"PolyGamma", "[", "z", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"z", ",", 
     RowBox[{"-", "5"}], ",", "3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:="],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
   RowBox[{"-", 
    FractionBox["1", 
     RowBox[{"z", "+", "5"}]]}], "+", 
   RowBox[{"(", 
    RowBox[{
     FractionBox["137", "60"], "-", "EulerGamma"}], ")"}], "+", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      FractionBox["5269", "3600"], "+", 
      FractionBox[
       SuperscriptBox["\[Pi]", "2"], "6"]}], ")"}], " ", 
    RowBox[{"(", 
     RowBox[{"z", "+", "5"}], ")"}]}], "+", 
   RowBox[{
    FractionBox["1", "2"], " ", 
    RowBox[{"PolyGamma", "[", 
     RowBox[{"2", ",", "6"}], "]"}], " ", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"z", "+", "5"}], ")"}], "2"]}], "+", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      FractionBox["14001361", "12960000"], "+", 
      FractionBox[
       SuperscriptBox["\[Pi]", "4"], "90"]}], ")"}], " ", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"z", "+", "5"}], ")"}], "3"]}], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", 
      RowBox[{"z", "+", "5"}], "]"}], "4"],
    SeriesData[$CellContext`z, -5, {}, -1, 4, 1],
    Editable->False]}],
  SeriesData[$CellContext`z, -5, {-1, Rational[137, 60] - EulerGamma, 
    Rational[5269, 3600] + Rational[1, 6] Pi^2, Rational[1, 2] 
    PolyGamma[2, 6], Rational[14001361, 12960000] + Rational[1, 90] Pi^4}, -1,
    4, 1],
  Editable->False]], "Output",
 ImageSize->{414, 66},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->759018436]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->59853664],

Cell["Series expansion at infinities:", "ExampleText"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"PolyGamma", "[", "z", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"z", ",", "Infinity", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:="],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
   RowBox[{"-", 
    RowBox[{"Log", "[", 
     FractionBox["1", "z"], "]"}]}], "-", 
   FractionBox["1", 
    RowBox[{"2", " ", "z"}]], "-", 
   RowBox[{
    FractionBox["1", "12"], " ", 
    SuperscriptBox[
     RowBox[{"(", 
      FractionBox["1", "z"], ")"}], "2"]}], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", 
      FractionBox["1", "z"], "]"}], "3"],
    SeriesData[$CellContext`z, 
     DirectedInfinity[1], {}, 0, 3, 1],
    Editable->False]}],
  SeriesData[$CellContext`z, 
   DirectedInfinity[1], {-Log[$CellContext`z^(-1)], 
    Rational[-1, 2], 
    Rational[-1, 12]}, 0, 3, 1],
  Editable->False]], "Output",
 ImageSize->{207, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->711016451]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"PolyGamma", "[", "z", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"z", ",", 
     RowBox[{"-", "Infinity"}], ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->163711927],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"-", "\[Pi]"}], " ", 
   RowBox[{"Cot", "[", 
    RowBox[{"\[Pi]", " ", "z"}], "]"}]}], "+", 
  RowBox[{"(", 
   InterpretationBox[
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"\[ImaginaryI]", " ", "\[Pi]"}], "-", 
       RowBox[{"Log", "[", 
        FractionBox["1", "z"], "]"}]}], ")"}], "-", 
     FractionBox["1", 
      RowBox[{"2", " ", "z"}]], "-", 
     RowBox[{
      FractionBox["1", "12"], " ", 
      SuperscriptBox[
       RowBox[{"(", 
        FractionBox["1", "z"], ")"}], "2"]}], "+", 
     InterpretationBox[
      SuperscriptBox[
       RowBox[{"O", "[", 
        FractionBox["1", "z"], "]"}], "3"],
      SeriesData[$CellContext`z, 
       DirectedInfinity[-1], {}, 0, 3, 1],
      Editable->False]}],
    SeriesData[$CellContext`z, 
     DirectedInfinity[-1], {Complex[0, 1] Pi - Log[$CellContext`z^(-1)], 
      Rational[-1, 2], 
      Rational[-1, 12]}, 0, 3, 1],
    Editable->False], ")"}]}]], "Output",
 ImageSize->{343, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->176302054]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Higher-Order Polygamma Functions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection"],

Cell["Special cases:", "ExampleText"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PolyGamma", "[", 
  RowBox[{
   RowBox[{"-", "1"}], ",", "z"}], "]"}]], "Input",
 CellLabel->"In[1]:="],

Cell[BoxData[
 RowBox[{"LogGamma", "[", "z", "]"}]], "Output",
 ImageSize->{81, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->522078183]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter"],

Cell["Evaluate at exact arguments:", "ExampleText"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"PolyGamma", "[", 
   RowBox[{
    RowBox[{"-", "2"}], ",", 
    RowBox[{"7", "/", "2"}]}], "]"}], "//", "FunctionExpand", " "}]], "Input",\

 CellTags->"Ex--21958729656347312745,5",
 CellLabel->"In[1]:="],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", 
   FractionBox["9", "2"]}], "-", 
  FractionBox[
   RowBox[{"67", " ", 
    RowBox[{"Log", "[", "2", "]"}]}], "24"], "+", 
  FractionBox[
   RowBox[{"3", " ", 
    RowBox[{"Log", "[", "3", "]"}]}], "2"], "+", 
  FractionBox[
   RowBox[{"5", " ", 
    RowBox[{"Log", "[", "5", "]"}]}], "2"], "+", 
  FractionBox[
   RowBox[{"3", " ", 
    RowBox[{"Log", "[", "Glaisher", "]"}]}], "2"], "+", 
  FractionBox[
   RowBox[{"7", " ", 
    RowBox[{"Log", "[", "\[Pi]", "]"}]}], "4"]}]], "Output",
 ImageSize->{424, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"Ex--21958729656347312745,5",
 CellLabel->"Out[1]=",
 CellID->297058935]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"PolyGamma", "[", 
   RowBox[{"1", ",", 
    FractionBox["1", "4"]}], "]"}], "//", "FunctionExpand"}]], "Input",
 CellTags->"Ex--21958729656347312745,6",
 CellLabel->"In[2]:="],

Cell[BoxData[
 RowBox[{
  RowBox[{"8", " ", "Catalan"}], "+", 
  SuperscriptBox["\[Pi]", "2"]}]], "Output",
 ImageSize->{90, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"Ex--21958729656347312745,6",
 CellLabel->"Out[2]=",
 CellID->98942505]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter"],

Cell["Series expansion at a generic point:", "ExampleText"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"PolyGamma", "[", 
    RowBox[{"a", ",", "z"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"z", ",", "z0", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:="],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
   RowBox[{"PolyGamma", "[", 
    RowBox[{"a", ",", "z0"}], "]"}], "+", 
   RowBox[{
    RowBox[{"PolyGamma", "[", 
     RowBox[{
      RowBox[{"1", "+", "a"}], ",", "z0"}], "]"}], " ", 
    RowBox[{"(", 
     RowBox[{"z", "-", "z0"}], ")"}]}], "+", 
   RowBox[{
    FractionBox["1", "2"], " ", 
    RowBox[{"PolyGamma", "[", 
     RowBox[{
      RowBox[{"2", "+", "a"}], ",", "z0"}], "]"}], " ", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"z", "-", "z0"}], ")"}], "2"]}], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", 
      RowBox[{"z", "-", "z0"}], "]"}], "3"],
    SeriesData[$CellContext`z, $CellContext`z0, {}, 0, 3, 1],
    Editable->False]}],
  SeriesData[$CellContext`z, $CellContext`z0, {
    PolyGamma[$CellContext`a, $CellContext`z0], 
    PolyGamma[1 + $CellContext`a, $CellContext`z0], Rational[1, 2] 
    PolyGamma[2 + $CellContext`a, $CellContext`z0]}, 0, 3, 1],
  Editable->False]], "Output",
 ImageSize->{316, 46},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->218737218]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->11942],

Cell["Series expansion near a singularity:", "ExampleText",
 CellID->4827],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"PolyGamma", "[", 
    RowBox[{"3", ",", 
     RowBox[{
      RowBox[{"-", "n"}], "+", "\[Epsilon]"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"\[Epsilon]", ",", "0", ",", "2"}], "}"}], ",", 
   RowBox[{"Assumptions", "\[Rule]", 
    RowBox[{
     RowBox[{"n", "\[Element]", "Integers"}], " ", "&&", " ", 
     RowBox[{"n", "\[GreaterEqual]", "0"}]}]}]}], "]"}]], "Input",
 CellTags->"Ex--21958729656347312745,7",
 CellLabel->"In[1]:=",
 CellID->5436],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
   FractionBox["6", 
    SuperscriptBox["\[Epsilon]", "4"]], "+", 
   RowBox[{"(", 
    RowBox[{
     FractionBox[
      SuperscriptBox["\[Pi]", "4"], "15"], "+", 
     RowBox[{"6", " ", 
      RowBox[{"HarmonicNumber", "[", 
       RowBox[{"n", ",", "4"}], "]"}]}]}], ")"}], "+", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"24", " ", 
       RowBox[{"HarmonicNumber", "[", 
        RowBox[{"n", ",", "5"}], "]"}]}], "+", 
      RowBox[{"PolyGamma", "[", 
       RowBox[{"4", ",", "1"}], "]"}]}], ")"}], " ", "\[Epsilon]"}], "+", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      FractionBox[
       RowBox[{"4", " ", 
        SuperscriptBox["\[Pi]", "6"]}], "63"], "+", 
      RowBox[{"60", " ", 
       RowBox[{"HarmonicNumber", "[", 
        RowBox[{"n", ",", "6"}], "]"}]}]}], ")"}], " ", 
    SuperscriptBox["\[Epsilon]", "2"]}], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[Epsilon]", "]"}], "3"],
    SeriesData[$CellContext`\[Epsilon], 0, {}, -4, 3, 1],
    Editable->False]}],
  SeriesData[$CellContext`\[Epsilon], 0, {
   6, 0, 0, 0, Rational[1, 15] Pi^4 + 6 HarmonicNumber[$CellContext`n, 4], 
    24 HarmonicNumber[$CellContext`n, 5] + PolyGamma[4, 1], 
    Rational[4, 63] Pi^6 + 60 HarmonicNumber[$CellContext`n, 6]}, -4, 3, 1],
  Editable->False]], "Output",
 ImageSize->{541, 67},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"Ex--21958729656347312745,7",
 CellLabel->"Out[1]=",
 CellID->447976926]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection"],

Cell[TextData[{
 "Plot of the absolute value of ",
 Cell[BoxData[
  ButtonBox["PolyGamma",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PolyGamma"]], "InlineFormula"],
 " over the complex plane: "
}], "ExampleText"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"Abs", "[", 
    RowBox[{"PolyGamma", "[", 
     RowBox[{"x", "+", 
      RowBox[{"I", " ", "y"}]}], "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "5"}], ",", "2"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:="],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{300, 247},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->437872115]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter"],

Cell[TextData[{
 "Electric field energy of a charge a fraction ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 " of the way between parallel conducting plates:"
}], "ExampleText"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sum", "[", 
  RowBox[{
   RowBox[{
    FractionBox["1", "k"], "-", 
    FractionBox["1", 
     RowBox[{
      RowBox[{"2", "k"}], " ", "-", 
      RowBox[{"2", " ", "x"}]}]], "-", 
    FractionBox["1", 
     RowBox[{
      RowBox[{"2", 
       RowBox[{"(", 
        RowBox[{"k", "-", "1"}], ")"}]}], " ", "+", 
      RowBox[{"2", " ", "x"}]}]]}], ",", 
   RowBox[{"{", 
    RowBox[{"k", ",", "Infinity"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:="],

Cell[BoxData[
 RowBox[{"EulerGamma", "+", 
  RowBox[{
   FractionBox["1", "2"], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"PolyGamma", "[", 
      RowBox[{"0", ",", 
       RowBox[{"1", "-", "x"}]}], "]"}], "+", 
     RowBox[{"PolyGamma", "[", 
      RowBox[{"0", ",", "x"}], "]"}]}], ")"}]}]}]], "Output",
 ImageSize->{352, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->560384993]
}, Open  ]],

Cell["Expand near the left wall:", "ExampleText"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Series", "[", 
   RowBox[{"%", ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "3"}], "}"}]}], "]"}], "//", 
  "FullSimplify"}]], "Input",
 CellLabel->"In[2]:="],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
   RowBox[{"-", 
    FractionBox["1", 
     RowBox[{"2", " ", "x"}]]}], "-", 
   RowBox[{
    RowBox[{"Zeta", "[", "3", "]"}], " ", 
    SuperscriptBox["x", "2"]}], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "x", "]"}], "4"],
    SeriesData[$CellContext`x, 0, {}, -1, 4, 1],
    Editable->False]}],
  SeriesData[$CellContext`x, 0, {
    Rational[-1, 2], 0, 0, -Zeta[3]}, -1, 4, 1],
  Editable->False]], "Output",
 ImageSize->{157, 31},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->611432459]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter"],

Cell[TextData[{
 "Final speed of a rocket with ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " discrete propulsion events:"
}], "ExampleText"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sum", "[", 
  RowBox[{
   RowBox[{
    FractionBox[
     RowBox[{
      SubscriptBox["m", "i"], "-", 
      SubscriptBox["m", "f"]}], "n"], " ", "u", " ", 
    FractionBox["1", 
     RowBox[{
      SubscriptBox["m", "f"], "+", 
      RowBox[{
       FractionBox["k", "n"], 
       RowBox[{"(", 
        RowBox[{
         SubscriptBox["m", "i"], "-", 
         SubscriptBox["m", "f"]}], ")"}]}]}]]}], ",", 
   RowBox[{"{", 
    RowBox[{"k", ",", "n"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:="],

Cell[BoxData[
 FractionBox[
  RowBox[{"u", " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"PolyGamma", "[", 
      RowBox[{"0", ",", 
       RowBox[{"1", "-", 
        FractionBox[
         RowBox[{"n", " ", 
          SubscriptBox["m", "f"]}], 
         RowBox[{
          SubscriptBox["m", "f"], "-", 
          SubscriptBox["m", "i"]}]]}]}], "]"}], "-", 
     RowBox[{"PolyGamma", "[", 
      RowBox[{"0", ",", 
       RowBox[{"1", "+", "n", "-", 
        FractionBox[
         RowBox[{"n", " ", 
          SubscriptBox["m", "f"]}], 
         RowBox[{
          SubscriptBox["m", "f"], "-", 
          SubscriptBox["m", "i"]}]]}]}], "]"}]}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", 
      SubscriptBox["m", "f"]}], "+", 
     SubscriptBox["m", "i"]}], ")"}]}], 
  RowBox[{
   SubscriptBox["m", "f"], "-", 
   SubscriptBox["m", "i"]}]]], "Output",
 ImageSize->{421, 48},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->62019231]
}, Open  ]],

Cell["\<\
Final velocity in the limit of constant continuous propulsion:\
\>", "ExampleText"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Assuming", "[", 
  RowBox[{
   RowBox[{
    SubscriptBox["m", "i"], ">", 
    SubscriptBox["m", "f"], ">", "0"}], ",", 
   RowBox[{
    RowBox[{"Series", "[", 
     RowBox[{"%", ",", 
      RowBox[{"{", 
       RowBox[{"n", ",", "Infinity", ",", "1"}], "}"}], ",", 
      RowBox[{"Assumptions", "->", 
       RowBox[{"n", ">", "1"}]}]}], "]"}], "//", "FullSimplify"}]}], 
  "]"}]], "Input",
 CellLabel->"In[2]:="],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzVWM1uEzEQdrLrhDSNQklUUOlPKK3acgIOcM0B6IkXgFMUVWqlVqDSpygP
kUfJpe+12F5713z5vHE3LRIrZXdnPDsz38x4bOfr5Ob87GpyczGdjE6vJz/P
L6a/Rl9+XCtW0hBC3Knf6xdCvWfqzfzuskw98/tbdU8U842m9EtmLtG2/J59
NmH8s/n6HXCPge4A3QMatX4yWrWWNrH5Ut2kIrVvIi34Aek0wE+InCbNi3DX
PuX2DXGi7zLXUNyNHf3FtjdqTTifEXlQRys42lRPCboktXhAuU8rEGBktoF+
EoUBtUgqhZVxfwz7fjQW9Kck61r6OfDj8oLa0ijkVZEo0ewQHOWotKOdAvNy
P7ogtQl0KwpzNzpvzcBcRU+ZHEa2TXHqSCZFiHNnECVm1sTzry6BGtYID/E5
v52MvnaJdyiDdcVxDYEe0BrAeYf2sev2a9ke2misAR+7wEmUtS2rDXsVWr2M
0rYDXKyielpdP/XzpvMtLSuxdEo6AFr8DnQrgJ974laZnKl9MnO0rF43cxZq
g3Pv6y3vPagl4Ls3e+jsfjAUeT+dz+c2Vx2Tu/F4bMfRukYlXSy978bmu9ls
5umVItSBl3mRGm2KDngRGSW08D9mOg6DP4//MQK3AhUzjOwDWcfHlRx9nK7g
0zpwB49oq2sxok7Hx5jsES2a3Mz4qSDU9eNWJIwEasF1D9dq3Oegd69qeaVz
z84R6N2zbLFuoIsHNPUCfKy6EdBurcKscRx8r+hmA+5XNjCWRKb6DNOvtIfR
Q2yIaRWMuFPc8KQsKxodOz3i2lFKY4XWRxk+6TCPtDTfS9Y77/SiLLN6ZXXj
zgyu4xofAmfSvRVQDKNygvtZrJbcu+Ns8aywm5VnmgJn5YljQHjoj5s/B8DH
7tgwKr4B92hJTuK04pp7SDLo065C/f933GwX4jdIv6c2b+Hbj/TbD4ab/3cl
Gn8AMuSvng==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{196, 35},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->168125987]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection"],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["FullSimplify",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FullSimplify"]], "InlineFormula"],
 " to simplify polygamma functions:"
}], "ExampleText"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FullSimplify", "[", 
  RowBox[{
   RowBox[{"PolyGamma", "[", 
    RowBox[{"1", "-", "z"}], "]"}], "-", 
   RowBox[{"PolyGamma", "[", "z", "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->276607508],

Cell[BoxData[
 RowBox[{"\[Pi]", " ", 
  RowBox[{"Cot", "[", 
   RowBox[{"\[Pi]", " ", "z"}], "]"}]}]], "Output",
 ImageSize->{68, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->484759132]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter"],

Cell["Express rational arguments through elementary functions:", "ExampleText"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FunctionExpand", "[", 
  RowBox[{"PolyGamma", "[", 
   RowBox[{"1", "/", "5"}], "]"}], " ", "]"}]], "Input",
 CellLabel->"In[1]:="],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "EulerGamma"}], "-", 
  RowBox[{
   FractionBox["1", "8"], " ", 
   SqrtBox[
    FractionBox["5", 
     RowBox[{
      FractionBox["5", "8"], "-", 
      FractionBox[
       SqrtBox["5"], "8"]}]]], " ", "\[Pi]"}], "-", 
  FractionBox["\[Pi]", 
   RowBox[{"8", " ", 
    SqrtBox[
     RowBox[{
      FractionBox["5", "8"], "-", 
      FractionBox[
       SqrtBox["5"], "8"]}]]}]], "-", 
  RowBox[{"Log", "[", "10", "]"}], "-", 
  RowBox[{
   FractionBox["1", "4"], " ", 
   RowBox[{"Log", "[", 
    RowBox[{
     FractionBox["5", "8"], "-", 
     FractionBox[
      SqrtBox["5"], "8"]}], "]"}]}], "+", 
  RowBox[{
   FractionBox["1", "4"], " ", 
   SqrtBox["5"], " ", 
   RowBox[{"Log", "[", 
    RowBox[{
     FractionBox["5", "8"], "-", 
     FractionBox[
      SqrtBox["5"], "8"]}], "]"}]}], "-", 
  RowBox[{
   FractionBox["1", "4"], " ", 
   RowBox[{"Log", "[", 
    RowBox[{
     FractionBox["5", "8"], "+", 
     FractionBox[
      SqrtBox["5"], "8"]}], "]"}]}], "-", 
  RowBox[{
   FractionBox["1", "4"], " ", 
   SqrtBox["5"], " ", 
   RowBox[{"Log", "[", 
    RowBox[{
     FractionBox["5", "8"], "+", 
     FractionBox[
      SqrtBox["5"], "8"]}], "]"}]}]}]], "Output",
 ImageSize->{507, 99},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->325767126]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter"],

Cell["Numerically find a root of a transcendental equation:", "ExampleText"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindRoot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"PolyGamma", "[", "z", "]"}], "-", 
     RowBox[{"2", " ", 
      RowBox[{"PolyGamma", "[", 
       RowBox[{"z", "+", "2"}], "]"}]}], "+", "z"}], "\[Equal]", "2"}], ",", 
   RowBox[{"{", 
    RowBox[{"z", ",", " ", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:="],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"z", "\[Rule]", "4.169684419023774`"}], "}"}]], "Output",
 ImageSize->{90, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->41422527]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter"],

Cell["Sums and integrals:", "ExampleText"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  UnderoverscriptBox["\[Sum]", 
   RowBox[{"k", "=", "1"}], "n"], 
  RowBox[{"PolyGamma", "[", "k", "]"}]}]], "Input",
 CellLabel->"In[1]:="],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "n"}], "+", 
  RowBox[{"n", " ", 
   RowBox[{"PolyGamma", "[", 
    RowBox[{"0", ",", 
     RowBox[{"1", "+", "n"}]}], "]"}]}]}]], "Output",
 ImageSize->{164, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->423626892]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  UnderoverscriptBox["\[Sum]", 
   RowBox[{"k", "=", "1"}], "n"], 
  RowBox[{
   RowBox[{"PolyGamma", "[", "k", "]"}], 
   RowBox[{"z", "^", "k"}]}]}]], "Input",
 CellLabel->"In[2]:="],

Cell[BoxData[
 RowBox[{
  FractionBox[
   RowBox[{
    RowBox[{"EulerGamma", " ", "z"}], "+", 
    RowBox[{"z", " ", 
     RowBox[{"Log", "[", 
      RowBox[{"1", "-", "z"}], "]"}]}]}], 
   RowBox[{
    RowBox[{"-", "1"}], "+", "z"}]], "+", 
  FractionBox[
   RowBox[{
    SuperscriptBox["z", 
     RowBox[{"1", "+", "n"}]], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"z", " ", 
       RowBox[{"Hypergeometric2F1", "[", 
        RowBox[{
         RowBox[{"1", "+", "n"}], ",", "1", ",", 
         RowBox[{"2", "+", "n"}], ",", "z"}], "]"}]}], "+", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{"1", "+", "n"}], ")"}], " ", 
       RowBox[{"PolyGamma", "[", 
        RowBox[{"0", ",", 
         RowBox[{"1", "+", "n"}]}], "]"}]}]}], ")"}]}], 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"1", "+", "n"}], ")"}], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "1"}], "+", "z"}], ")"}]}]]}]], "Output",
 ImageSize->{477, 65},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->337091828]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"x", "^", "2"}], " ", 
    RowBox[{"PolyGamma", "[", "x", "]"}]}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[3]:="],

Cell[BoxData[
 RowBox[{
  RowBox[{
   SuperscriptBox["x", "2"], " ", 
   RowBox[{"LogGamma", "[", "x", "]"}]}], "+", 
  RowBox[{"2", " ", 
   RowBox[{"PolyGamma", "[", 
    RowBox[{
     RowBox[{"-", "3"}], ",", "x"}], "]"}]}], "-", 
  RowBox[{"2", " ", "x", " ", 
   RowBox[{"PolyGamma", "[", 
    RowBox[{
     RowBox[{"-", "2"}], ",", "x"}], "]"}]}]}]], "Output",
 ImageSize->{372, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->213311815]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter"],

Cell[TextData[{
 "Generate ",
 Cell[BoxData[
  ButtonBox["PolyGamma",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PolyGamma"]], "InlineFormula"],
 " from integrals, sums, and limits: "
}], "ExampleText"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{"1", " ", "-", " ", 
      RowBox[{"t", "^", 
       RowBox[{"(", 
        RowBox[{"z", " ", "-", " ", "1"}], ")"}]}]}], ")"}], "/", 
    RowBox[{"(", 
     RowBox[{"1", " ", "-", " ", "t"}], ")"}]}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"t", ",", " ", "0", ",", " ", "1"}], "}"}], ",", " ", 
   RowBox[{"Assumptions", "->", 
    RowBox[{
     RowBox[{"Re", "[", "z", "]"}], ">", "0"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:="],

Cell[BoxData[
 RowBox[{"EulerGamma", "+", 
  RowBox[{"PolyGamma", "[", 
   RowBox[{"0", ",", "z"}], "]"}]}]], "Output",
 ImageSize->{189, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->66066784]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"Exp", "[", 
      RowBox[{"-", "t"}], "]"}], "/", "t"}], " ", "-", " ", 
    RowBox[{"1", "/", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{
        RowBox[{"(", 
         RowBox[{"t", " ", "+", " ", "1"}], ")"}], "^", "z"}], " ", "t"}], 
      ")"}]}]}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"t", ",", " ", "0", ",", " ", "Infinity"}], "}"}]}], 
  "]"}]], "Input",
 CellLabel->"In[2]:="],

Cell[BoxData[
 RowBox[{"If", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Re", "[", "z", "]"}], ">", "0"}], ",", 
   RowBox[{"PolyGamma", "[", 
    RowBox[{"0", ",", "z"}], "]"}], ",", 
   RowBox[{"Integrate", "[", 
    RowBox[{
     RowBox[{
      FractionBox[
       SuperscriptBox["\[ExponentialE]", 
        RowBox[{"-", "t"}]], "t"], "-", 
      FractionBox[
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"1", "+", "t"}], ")"}], 
        RowBox[{"-", "z"}]], "t"]}], ",", 
     RowBox[{"{", 
      RowBox[{"t", ",", "0", ",", "\[Infinity]"}], "}"}], ",", 
     RowBox[{"Assumptions", "\[Rule]", 
      RowBox[{
       RowBox[{"Re", "[", "z", "]"}], "\[LessEqual]", "0"}]}]}], "]"}]}], 
  "]"}]], "Output",
 ImageSize->{415, 54},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->96656509]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sum", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"k", "^", "2"}], " ", "-", " ", "1"}], ")"}], "/", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"k", "^", "2"}], "  ", "+", " ", "1"}], ")"}]}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"k", ",", " ", "1", ",", " ", "n"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:="],

Cell[BoxData[
 RowBox[{"1", "+", "n", "-", 
  RowBox[{"\[Pi]", " ", 
   RowBox[{"Coth", "[", "\[Pi]", "]"}]}], "+", 
  RowBox[{"\[ImaginaryI]", " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"PolyGamma", "[", 
      RowBox[{"0", ",", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"1", "-", "\[ImaginaryI]"}], ")"}], "+", "n"}]}], "]"}], "-",
      
     RowBox[{"PolyGamma", "[", 
      RowBox[{"0", ",", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"1", "+", "\[ImaginaryI]"}], ")"}], "+", "n"}]}], "]"}]}], 
    ")"}]}]}]], "Output",
 ImageSize->{468, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->713550654]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Product", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{"1", " ", "+", " ", 
      RowBox[{"y", "/", 
       RowBox[{"(", 
        RowBox[{"x", "+", "n"}], ")"}]}]}], ")"}], 
    RowBox[{"Exp", "[", 
     RowBox[{
      RowBox[{"-", "y"}], "/", 
      RowBox[{"(", 
       RowBox[{"x", "+", "n"}], ")"}]}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "Infinity"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[4]:="],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"y", " ", 
     RowBox[{"PolyGamma", "[", 
      RowBox[{"0", ",", 
       RowBox[{"1", "+", "x"}]}], "]"}]}]], " ", 
   RowBox[{"Gamma", "[", 
    RowBox[{"1", "+", "x"}], "]"}]}], 
  RowBox[{"Gamma", "[", 
   RowBox[{"1", "+", "x", "+", "y"}], "]"}]]], "Output",
 ImageSize->{182, 35},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->76513197]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Limit", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Zeta", "[", 
     RowBox[{"s", ",", "z"}], "]"}], "-", 
    FractionBox["1", 
     RowBox[{"s", "-", "1"}]]}], ",", 
   RowBox[{"s", "\[Rule]", "1"}]}], "]"}]], "Input",
 CellLabel->"In[5]:="],

Cell[BoxData[
 RowBox[{"-", 
  RowBox[{"PolyGamma", "[", 
   RowBox[{"0", ",", "z"}], "]"}]}]], "Output",
 ImageSize->{114, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->594749800]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter"],

Cell["Generating function:", "ExampleText"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  UnderoverscriptBox["\[Sum]", 
   RowBox[{"k", "=", "1"}], "\[Infinity]"], 
  RowBox[{
   RowBox[{"PolyGamma", "[", "k", "]"}], " ", 
   SuperscriptBox["x", "k"]}]}]], "Input",
 CellTags->"Fibonacci",
 CellLabel->"In[1]:="],

Cell[BoxData[
 FractionBox[
  RowBox[{
   RowBox[{"EulerGamma", " ", "x"}], "+", 
   RowBox[{"x", " ", 
    RowBox[{"Log", "[", 
     RowBox[{"1", "-", "x"}], "]"}]}]}], 
  RowBox[{
   RowBox[{"-", "1"}], "+", "x"}]]], "Output",
 ImageSize->{173, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"Fibonacci",
 CellLabel->"Out[1]=",
 CellID->98925211]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter"],

Cell["Obtain as special cases of hypergeometric functions:", "ExampleText"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"HypergeometricPFQ", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "1", ",", "1", ",", 
     RowBox[{"2", "-", "z"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"2", ",", "2", ",", "2"}], "}"}], ",", "1"}], "]"}]], "Input",
 CellLabel->"In[1]:="],

Cell[BoxData[
 RowBox[{
  FractionBox["1", 
   RowBox[{"12", " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "1"}], "+", "z"}], ")"}]}]], 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"6", " ", 
     SuperscriptBox["EulerGamma", "2"]}], "+", 
    SuperscriptBox["\[Pi]", "2"], "+", 
    RowBox[{"12", " ", "EulerGamma", " ", 
     RowBox[{"PolyGamma", "[", 
      RowBox[{"0", ",", "z"}], "]"}]}], "+", 
    RowBox[{"6", " ", 
     SuperscriptBox[
      RowBox[{"PolyGamma", "[", 
       RowBox[{"0", ",", "z"}], "]"}], "2"]}], "-", 
    RowBox[{"6", " ", 
     RowBox[{"PolyGamma", "[", 
      RowBox[{"1", ",", "z"}], "]"}]}]}], ")"}]}]], "Output",
 ImageSize->{480, 51},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->70919708]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection"],

Cell["\<\
The one\[Hyphen]argument form evaluates to the two-argument form:\
\>", "ExampleText"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PolyGamma", "[", "z", "]"}]], "Input",
 CellLabel->"In[1]:="],

Cell[BoxData[
 RowBox[{"PolyGamma", "[", 
  RowBox[{"0", ",", "z"}], "]"}]], "Output",
 ImageSize->{107, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->606037572]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter"],

Cell["\<\
Large orders can give results too large to be computed explicitly:\
\>", "ExampleText"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PolyGamma", "[", 
  RowBox[{
   RowBox[{"10.", "^", "10"}], ",", "1"}], "]"}]], "Input",
 CellLabel->"In[1]:="],

Cell[BoxData[
 RowBox[{
  RowBox[{"General", "::", "\<\"ovfl\"\>"}], ":", 
  " ", "\<\"Overflow occurred in computation. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/General/ovfl\\\", ButtonNote -> \
\\\"General::ovfl\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->5867454],

Cell[BoxData[
 RowBox[{"Overflow", "[", "]"}]], "Output",
 ImageSize->{74, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->405274523]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter"],

Cell["\<\
Machine\[Hyphen]number inputs can give high\[Hyphen]precision results:\
\>", "ExampleText"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PolyGamma", "[", 
  RowBox[{
   RowBox[{"10.", "^", "3"}], ",", "1"}], "]"}]], "Input",
 CellLabel->"In[1]:="],

Cell[BoxData[
 RowBox[{"-", 
  "4.023872600769655110507501855823193`15.954589770191005*^2567"}]], "Output",\

 ImageSize->{177, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->427028192]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MachineNumberQ", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:="],

Cell[BoxData["False"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->13110509]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->10637],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["Gamma",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Gamma"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["LogGamma",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/LogGamma"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["EulerGamma",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/EulerGamma"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["HarmonicNumber",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/HarmonicNumber"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->21100]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->1627],

Cell[TextData[ButtonBox["Special Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/SpecialFunctions"]], "Tutorials",
 CellID->17831]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection",
 CellID->6441],

Cell[TextData[ButtonBox["Implementation notes: Numerical and Related \
Functions",
 BaseStyle->"Hyperlink",
 ButtonData->{
   URL["http://reference.wolfram.com/mathematica/note/\
SomeNotesOnInternalImplementation.html#23847"], None}]], "RelatedLinks",
 CellID->780547280],

Cell[TextData[StyleBox[ButtonBox["MathWorld",
 BaseStyle->"MathWorldLink",
 ButtonData->{
   URL["http://mathworld.wolfram.com/DigammaFunction.html"], 
   None}], "IT"]], "RelatedLinks"],

Cell[TextData[ButtonBox["The Wolfram Functions Site",
 BaseStyle->"FunctionsSiteLink",
 ButtonData->{
   URL["http://functions.wolfram.com/GammaBetaErf/PolyGamma/"], 
   None}]], "RelatedLinks"],

Cell[TextData[{
 ButtonBox["NKS|Online",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.wolframscience.com/nksonline/index/search.cgi?SearchIndex=\
PolyGamma"], None}],
 " (",
 StyleBox[ButtonBox["A New Kind of Science",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.wolframscience.com/nksonline/"], None}],
  FontSlant->"Italic"],
 ")"
}], "RelatedLinks"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->238537373],

Cell[TextData[ButtonBox["Gamma Functions and Related Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/GammaFunctionsAndRelatedFunctions"]], "MoreAbout",
 CellID->603989183],

Cell[TextData[ButtonBox["Recurrence and Sum Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/RecurrenceAndSumFunctions"]], "MoreAbout",
 CellID->264127284],

Cell[TextData[ButtonBox["Special Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/SpecialFunctions"]], "MoreAbout",
 CellID->35776035],

Cell[TextData[ButtonBox["New in 6.0: Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicalFunctions"]], "MoreAbout",
 CellID->1593466]
}, Open  ]],

Cell["New in 1  |  Last modified in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"PolyGamma - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 10, 48.2611496}", "context" -> "System`", 
    "keywords" -> {
     "digamma function", "gamma function", "poly-gamma", "psi", 
      "psi function"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "PolyGamma[z] gives the digamma function \\[Psi](z). PolyGamma[n, z] \
gives the n^th derivative of the digamma function \\[Psi] (n) (z). ", 
    "synonyms" -> {"poly gamma"}, "title" -> "PolyGamma", "type" -> "Symbol", 
    "uri" -> "ref/PolyGamma"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[12250, 358, 359, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->30414486]},
 "Ex--21958729656347312745,1"->{
  Cell[21379, 641, 234, 8, 70, "Input",
   CellTags->"Ex--21958729656347312745,1",
   CellID->6334],
  Cell[21616, 651, 312, 9, 36, "Output",
   CellTags->"Ex--21958729656347312745,1",
   CellID->362645953]},
 "Ex--21958729656347312745,5"->{
  Cell[37238, 1264, 239, 8, 70, "Input",
   CellTags->"Ex--21958729656347312745,5"],
  Cell[37480, 1274, 700, 24, 51, "Output",
   CellTags->"Ex--21958729656347312745,5",
   CellID->297058935]},
 "Ex--21958729656347312745,6"->{
  Cell[38217, 1303, 210, 6, 70, "Input",
   CellTags->"Ex--21958729656347312745,6"],
  Cell[38430, 1311, 277, 9, 36, "Output",
   CellTags->"Ex--21958729656347312745,6",
   CellID->98942505]},
 "Ex--21958729656347312745,7"->{
  Cell[40503, 1387, 516, 15, 70, "Input",
   CellTags->"Ex--21958729656347312745,7",
   CellID->5436],
  Cell[41022, 1404, 1536, 45, 88, "Output",
   CellTags->"Ex--21958729656347312745,7",
   CellID->447976926]},
 "Fibonacci"->{
  Cell[125990, 3235, 248, 8, 70, "Input",
   CellTags->"Fibonacci"],
  Cell[126241, 3245, 382, 14, 51, "Output",
   CellTags->"Fibonacci",
   CellID->98925211]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 135940, 3601},
 {"Ex--21958729656347312745,1", 136093, 3605},
 {"Ex--21958729656347312745,5", 136335, 3612},
 {"Ex--21958729656347312745,6", 136563, 3618},
 {"Ex--21958729656347312745,7", 136789, 3624},
 {"Fibonacci", 137019, 3631}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2621, 61, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3244, 88, 4193, 83, 70, "ObjectNameGrid"],
Cell[7440, 173, 1258, 42, 70, "Usage",
 CellID->10657]
}, Open  ]],
Cell[CellGroupData[{
Cell[8735, 220, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[9213, 237, 120, 4, 70, "Notes",
 CellID->20640],
Cell[9336, 243, 566, 19, 70, "Notes",
 CellID->23946],
Cell[9905, 264, 696, 26, 70, "Notes",
 CellID->29958],
Cell[10604, 292, 348, 10, 70, "Notes",
 CellID->358863611],
Cell[10955, 304, 576, 21, 70, "Notes",
 CellID->2263],
Cell[11534, 327, 250, 8, 70, "Notes",
 CellID->283848419],
Cell[11787, 337, 220, 7, 70, "Notes",
 CellID->31226],
Cell[12010, 346, 203, 7, 70, "Notes",
 CellID->28162]
}, Closed]],
Cell[CellGroupData[{
Cell[12250, 358, 359, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->30414486],
Cell[CellGroupData[{
Cell[12634, 372, 147, 5, 70, "ExampleSection",
 CellID->33776721],
Cell[12784, 379, 73, 1, 70, "ExampleText",
 CellID->121247567],
Cell[CellGroupData[{
Cell[12882, 384, 105, 3, 28, "Input",
 CellID->402316662],
Cell[12990, 389, 212, 7, 51, "Output",
 CellID->39210521]
}, Open  ]],
Cell[13217, 399, 73, 1, 70, "ExampleText",
 CellID->323317997],
Cell[CellGroupData[{
Cell[13315, 404, 127, 4, 28, "Input",
 CellID->58503667],
Cell[13445, 410, 335, 12, 54, "Output",
 CellID->279621243]
}, Open  ]],
Cell[13795, 425, 125, 3, 70, "ExampleDelimiter",
 CellID->573998195],
Cell[13923, 430, 74, 1, 70, "ExampleText",
 CellID->3935701],
Cell[CellGroupData[{
Cell[14022, 435, 202, 7, 70, "Input",
 CellID->407145307],
Cell[14227, 444, 444, 15, 36, "Output",
 CellID->397183923]
}, Open  ]],
Cell[14686, 462, 105, 2, 70, "ExampleDelimiter"],
Cell[14794, 466, 64, 1, 70, "ExampleText",
 CellID->864686355],
Cell[CellGroupData[{
Cell[14883, 471, 220, 7, 28, "Input"],
Cell[15106, 480, 4308, 75, 166, "Output",
 Evaluatable->False,
 CellID->380678060]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[19463, 561, 200, 6, 70, "ExampleSection"],
Cell[19666, 569, 66, 0, 70, "ExampleText"],
Cell[CellGroupData[{
Cell[19757, 573, 87, 2, 70, "Input"],
Cell[19847, 577, 299, 9, 51, "Output",
 CellID->114353727]
}, Open  ]],
Cell[20161, 589, 125, 3, 70, "ExampleDelimiter",
 CellID->127828245],
Cell[20289, 594, 64, 1, 70, "ExampleText",
 CellID->215230372],
Cell[CellGroupData[{
Cell[20378, 599, 109, 3, 70, "Input",
 CellID->466830376],
Cell[20490, 604, 174, 5, 36, "Output",
 CellID->40856817]
}, Open  ]],
Cell[20679, 612, 125, 3, 70, "ExampleDelimiter",
 CellID->351643097],
Cell[20807, 617, 85, 1, 70, "ExampleText",
 CellID->334604611],
Cell[CellGroupData[{
Cell[20917, 622, 149, 5, 70, "Input",
 CellID->317365560],
Cell[21069, 629, 273, 7, 36, "Output",
 CellID->644053106]
}, Open  ]],
Cell[CellGroupData[{
Cell[21379, 641, 234, 8, 70, "Input",
 CellTags->"Ex--21958729656347312745,1",
 CellID->6334],
Cell[21616, 651, 312, 9, 36, "Output",
 CellTags->"Ex--21958729656347312745,1",
 CellID->362645953]
}, Open  ]],
Cell[21943, 663, 124, 3, 70, "ExampleDelimiter",
 CellID->80376993],
Cell[22070, 668, 69, 1, 70, "ExampleText",
 CellID->104147080],
Cell[CellGroupData[{
Cell[22164, 673, 185, 6, 70, "Input",
 CellID->123865482],
Cell[22352, 681, 235, 7, 36, "Output",
 CellID->539038579]
}, Open  ]],
Cell[22602, 691, 125, 3, 70, "ExampleDelimiter",
 CellID->547872452],
Cell[22730, 696, 113, 3, 70, "ExampleText",
 CellID->230370911],
Cell[CellGroupData[{
Cell[22868, 703, 128, 3, 70, "Input",
 CellID->82935731],
Cell[22999, 708, 218, 6, 36, "Output",
 CellID->44141884]
}, Open  ]],
Cell[23232, 717, 125, 3, 70, "ExampleDelimiter",
 CellID->788580244],
Cell[23360, 722, 211, 7, 70, "ExampleText",
 CellID->67387947],
Cell[CellGroupData[{
Cell[23596, 733, 185, 6, 70, "Input",
 CellID->277880383],
Cell[23784, 741, 466, 15, 51, "Output",
 CellID->987201016]
}, Open  ]],
Cell[24265, 759, 125, 3, 70, "ExampleDelimiter",
 CellID->506155396],
Cell[24393, 764, 242, 8, 70, "ExampleText",
 CellID->462536501],
Cell[CellGroupData[{
Cell[24660, 776, 245, 8, 70, "Input",
 CellID->334350941],
Cell[24908, 786, 489, 16, 54, "Output",
 CellID->3501085]
}, Open  ]],
Cell[CellGroupData[{
Cell[25434, 807, 109, 3, 70, "Input",
 CellID->11183202],
Cell[25546, 812, 522, 18, 54, "Output",
 CellID->122176922]
}, Open  ]],
Cell[26083, 833, 125, 3, 70, "ExampleDelimiter",
 CellID->500660245],
Cell[26211, 838, 60, 1, 70, "ExampleText",
 CellID->115422345],
Cell[CellGroupData[{
Cell[26296, 843, 220, 7, 70, "Input",
 CellID->41981680],
Cell[26519, 852, 1102, 36, 51, "Output",
 CellID->180030687]
}, Open  ]],
Cell[27636, 891, 125, 3, 70, "ExampleDelimiter",
 CellID->760443585],
Cell[27764, 896, 203, 7, 70, "ExampleText",
 CellID->653625731],
Cell[CellGroupData[{
Cell[27992, 907, 142, 4, 70, "Input",
 CellID->35954786],
Cell[28137, 913, 315, 11, 52, "Output",
 CellID->322398403]
}, Open  ]],
Cell[CellGroupData[{
Cell[28489, 929, 167, 5, 70, "Input",
 CellID->633500976],
Cell[28659, 936, 315, 11, 52, "Output",
 CellID->236937360]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[29023, 953, 223, 6, 70, "ExampleSection"],
Cell[CellGroupData[{
Cell[29271, 963, 260, 8, 70, "ExampleSubsection",
 CellID->136184792],
Cell[29534, 973, 68, 0, 70, "ExampleText"],
Cell[CellGroupData[{
Cell[29627, 977, 92, 2, 70, "Input"],
Cell[29722, 981, 168, 5, 36, "Output",
 CellID->283554744]
}, Open  ]],
Cell[29905, 989, 105, 2, 70, "ExampleDelimiter"],
Cell[30013, 993, 193, 6, 70, "ExampleText"],
Cell[CellGroupData[{
Cell[30231, 1003, 194, 6, 70, "Input"],
Cell[30428, 1011, 1367, 43, 54, "Output",
 CellID->165117740]
}, Open  ]],
Cell[31810, 1057, 105, 2, 70, "ExampleDelimiter"],
Cell[31918, 1061, 49, 0, 70, "ExampleText"],
Cell[CellGroupData[{
Cell[31992, 1065, 222, 7, 70, "Input"],
Cell[32217, 1074, 1485, 48, 87, "Output",
 CellID->759018436]
}, Open  ]],
Cell[33717, 1125, 124, 3, 70, "ExampleDelimiter",
 CellID->59853664],
Cell[33844, 1130, 54, 0, 70, "ExampleText"],
Cell[CellGroupData[{
Cell[33923, 1134, 208, 6, 70, "Input"],
Cell[34134, 1142, 825, 29, 53, "Output",
 CellID->711016451]
}, Open  ]],
Cell[CellGroupData[{
Cell[34996, 1176, 249, 8, 70, "Input",
 CellID->163711927],
Cell[35248, 1186, 1122, 37, 53, "Output",
 CellID->176302054]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[36419, 1229, 233, 6, 70, "ExampleSubsection"],
Cell[36655, 1237, 37, 0, 70, "ExampleText"],
Cell[CellGroupData[{
Cell[36717, 1241, 127, 4, 70, "Input"],
Cell[36847, 1247, 192, 6, 36, "Output",
 CellID->522078183]
}, Open  ]],
Cell[37054, 1256, 105, 2, 70, "ExampleDelimiter"],
Cell[37162, 1260, 51, 0, 70, "ExampleText"],
Cell[CellGroupData[{
Cell[37238, 1264, 239, 8, 70, "Input",
 CellTags->"Ex--21958729656347312745,5"],
Cell[37480, 1274, 700, 24, 51, "Output",
 CellTags->"Ex--21958729656347312745,5",
 CellID->297058935]
}, Open  ]],
Cell[CellGroupData[{
Cell[38217, 1303, 210, 6, 70, "Input",
 CellTags->"Ex--21958729656347312745,6"],
Cell[38430, 1311, 277, 9, 36, "Output",
 CellTags->"Ex--21958729656347312745,6",
 CellID->98942505]
}, Open  ]],
Cell[38722, 1323, 105, 2, 70, "ExampleDelimiter"],
Cell[38830, 1327, 59, 0, 70, "ExampleText"],
Cell[CellGroupData[{
Cell[38914, 1331, 227, 7, 70, "Input"],
Cell[39144, 1340, 1121, 34, 67, "Output",
 CellID->218737218]
}, Open  ]],
Cell[40280, 1377, 121, 3, 70, "ExampleDelimiter",
 CellID->11942],
Cell[40404, 1382, 74, 1, 70, "ExampleText",
 CellID->4827],
Cell[CellGroupData[{
Cell[40503, 1387, 516, 15, 70, "Input",
 CellTags->"Ex--21958729656347312745,7",
 CellID->5436],
Cell[41022, 1404, 1536, 45, 88, "Output",
 CellTags->"Ex--21958729656347312745,7",
 CellID->447976926]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[42619, 1456, 207, 6, 70, "ExampleSection"],
Cell[42829, 1464, 219, 7, 70, "ExampleText"],
Cell[CellGroupData[{
Cell[43073, 1475, 397, 13, 70, "Input"],
Cell[43473, 1490, 65153, 1072, 70, "Output",
 Evaluatable->False,
 CellID->437872115]
}, Open  ]],
Cell[108641, 2565, 105, 2, 70, "ExampleDelimiter"],
Cell[108749, 2569, 199, 5, 70, "ExampleText"],
Cell[CellGroupData[{
Cell[108973, 2578, 478, 17, 70, "Input"],
Cell[109454, 2597, 444, 15, 51, "Output",
 CellID->560384993]
}, Open  ]],
Cell[109913, 2615, 49, 0, 70, "ExampleText"],
Cell[CellGroupData[{
Cell[109987, 2619, 205, 7, 70, "Input"],
Cell[110195, 2628, 616, 21, 52, "Output",
 CellID->611432459]
}, Open  ]],
Cell[110826, 2652, 105, 2, 70, "ExampleDelimiter"],
Cell[110934, 2656, 164, 5, 70, "ExampleText"],
Cell[CellGroupData[{
Cell[111123, 2665, 524, 19, 70, "Input"],
Cell[111650, 2686, 1002, 35, 69, "Output",
 CellID->62019231]
}, Open  ]],
Cell[112667, 2724, 93, 2, 70, "ExampleText"],
Cell[CellGroupData[{
Cell[112785, 2730, 437, 14, 70, "Input"],
Cell[113225, 2746, 1200, 24, 70, "Output",
 Evaluatable->False,
 CellID->168125987]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[114474, 2776, 217, 6, 70, "ExampleSection"],
Cell[114694, 2784, 207, 7, 70, "ExampleText"],
Cell[CellGroupData[{
Cell[114926, 2795, 230, 7, 70, "Input",
 CellID->276607508],
Cell[115159, 2804, 242, 8, 36, "Output",
 CellID->484759132]
}, Open  ]],
Cell[115416, 2815, 105, 2, 70, "ExampleDelimiter"],
Cell[115524, 2819, 79, 0, 70, "ExampleText"],
Cell[CellGroupData[{
Cell[115628, 2823, 155, 4, 70, "Input"],
Cell[115786, 2829, 1349, 53, 120, "Output",
 CellID->325767126]
}, Open  ]],
Cell[117150, 2885, 105, 2, 70, "ExampleDelimiter"],
Cell[117258, 2889, 76, 0, 70, "ExampleText"],
Cell[CellGroupData[{
Cell[117359, 2893, 360, 11, 70, "Input"],
Cell[117722, 2906, 225, 7, 36, "Output",
 CellID->41422527]
}, Open  ]],
Cell[117962, 2916, 105, 2, 70, "ExampleDelimiter"],
Cell[118070, 2920, 42, 0, 70, "ExampleText"],
Cell[CellGroupData[{
Cell[118137, 2924, 165, 5, 70, "Input"],
Cell[118305, 2931, 310, 11, 36, "Output",
 CellID->423626892]
}, Open  ]],
Cell[CellGroupData[{
Cell[118652, 2947, 208, 7, 70, "Input"],
Cell[118863, 2956, 1054, 37, 86, "Output",
 CellID->337091828]
}, Open  ]],
Cell[CellGroupData[{
Cell[119954, 2998, 195, 6, 70, "Input"],
Cell[120152, 3006, 497, 17, 36, "Output",
 CellID->213311815]
}, Open  ]],
Cell[120664, 3026, 105, 2, 70, "ExampleDelimiter"],
Cell[120772, 3030, 208, 7, 70, "ExampleText"],
Cell[CellGroupData[{
Cell[121005, 3041, 529, 16, 70, "Input"],
Cell[121537, 3059, 249, 8, 36, "Output",
 CellID->66066784]
}, Open  ]],
Cell[CellGroupData[{
Cell[121823, 3072, 488, 17, 70, "Input"],
Cell[122314, 3091, 853, 28, 75, "Output",
 CellID->96656509]
}, Open  ]],
Cell[CellGroupData[{
Cell[123204, 3124, 377, 12, 70, "Input"],
Cell[123584, 3138, 689, 23, 36, "Output",
 CellID->713550654]
}, Open  ]],
Cell[CellGroupData[{
Cell[124310, 3166, 461, 16, 70, "Input"],
Cell[124774, 3184, 483, 16, 56, "Output",
 CellID->76513197]
}, Open  ]],
Cell[CellGroupData[{
Cell[125294, 3205, 266, 9, 70, "Input"],
Cell[125563, 3216, 236, 8, 36, "Output",
 CellID->594749800]
}, Open  ]],
Cell[125814, 3227, 105, 2, 70, "ExampleDelimiter"],
Cell[125922, 3231, 43, 0, 70, "ExampleText"],
Cell[CellGroupData[{
Cell[125990, 3235, 248, 8, 70, "Input",
 CellTags->"Fibonacci"],
Cell[126241, 3245, 382, 14, 51, "Output",
 CellTags->"Fibonacci",
 CellID->98925211]
}, Open  ]],
Cell[126638, 3262, 105, 2, 70, "ExampleDelimiter"],
Cell[126746, 3266, 75, 0, 70, "ExampleText"],
Cell[CellGroupData[{
Cell[126846, 3270, 280, 8, 70, "Input"],
Cell[127129, 3280, 784, 26, 72, "Output",
 CellID->70919708]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[127962, 3312, 210, 6, 70, "ExampleSection"],
Cell[128175, 3320, 96, 2, 70, "ExampleText"],
Cell[CellGroupData[{
Cell[128296, 3326, 85, 2, 70, "Input"],
Cell[128384, 3330, 217, 7, 36, "Output",
 CellID->606037572]
}, Open  ]],
Cell[128616, 3340, 105, 2, 70, "ExampleDelimiter"],
Cell[128724, 3344, 97, 2, 70, "ExampleText"],
Cell[CellGroupData[{
Cell[128846, 3350, 135, 4, 70, "Input"],
Cell[128984, 3356, 355, 7, 70, "Message",
 CellID->5867454],
Cell[129342, 3365, 187, 6, 36, "Output",
 CellID->405274523]
}, Open  ]],
Cell[129544, 3374, 105, 2, 70, "ExampleDelimiter"],
Cell[129652, 3378, 101, 2, 70, "ExampleText"],
Cell[CellGroupData[{
Cell[129778, 3384, 134, 4, 70, "Input"],
Cell[129915, 3390, 240, 8, 36, "Output",
 CellID->427028192]
}, Open  ]],
Cell[CellGroupData[{
Cell[130192, 3403, 90, 2, 70, "Input"],
Cell[130285, 3407, 161, 5, 36, "Output",
 CellID->13110509]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[130507, 3419, 311, 9, 70, "SeeAlsoSection",
 CellID->10637],
Cell[130821, 3430, 951, 35, 70, "SeeAlso",
 CellID->21100]
}, Open  ]],
Cell[CellGroupData[{
Cell[131809, 3470, 313, 9, 70, "TutorialsSection",
 CellID->1627],
Cell[132125, 3481, 144, 3, 70, "Tutorials",
 CellID->17831]
}, Open  ]],
Cell[CellGroupData[{
Cell[132306, 3489, 320, 9, 70, "RelatedLinksSection",
 CellID->6441],
Cell[132629, 3500, 271, 6, 70, "RelatedLinks",
 CellID->780547280],
Cell[132903, 3508, 186, 4, 70, "RelatedLinks"],
Cell[133092, 3514, 194, 4, 70, "RelatedLinks"],
Cell[133289, 3520, 387, 13, 70, "RelatedLinks"]
}, Open  ]],
Cell[CellGroupData[{
Cell[133713, 3538, 319, 9, 70, "MoreAboutSection",
 CellID->238537373],
Cell[134035, 3549, 182, 3, 70, "MoreAbout",
 CellID->603989183],
Cell[134220, 3554, 165, 3, 70, "MoreAbout",
 CellID->264127284],
Cell[134388, 3559, 144, 3, 70, "MoreAbout",
 CellID->35776035],
Cell[134535, 3564, 172, 3, 70, "MoreAbout",
 CellID->1593466]
}, Open  ]],
Cell[134722, 3570, 50, 0, 70, "History"],
Cell[134775, 3572, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

