(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     36855,       1407]
NotebookOptionsPosition[     29784,       1163]
NotebookOutlinePosition[     31388,       1205]
CellTagsIndexPosition[     31302,       1200]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"PolynomialGCD" :> 
          Documentation`HelpLookup["paclet:ref/PolynomialGCD"], "ExtendedGCD" :> 
          Documentation`HelpLookup["paclet:ref/ExtendedGCD"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"PolynomialGCD\"\>", 
       2->"\<\"ExtendedGCD\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Polynomial Division" :> 
          Documentation`HelpLookup["paclet:guide/PolynomialDivision"], 
          "New in 6.0: Symbolic Computation" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60AlgebraicComputing"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Polynomial Division\"\>", 
       2->"\<\"New in 6.0: Symbolic Computation\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["PolynomialExtendedGCD", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["PolynomialExtendedGCD",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/PolynomialExtendedGCD"], "[", 
       RowBox[{
        SubscriptBox[
         StyleBox["poly", "TI"], 
         StyleBox["1", "TR"]], ",", 
        SubscriptBox[
         StyleBox["poly", "TI"], 
         StyleBox["2", "TR"]], ",", 
        StyleBox["x", "TI"]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the extended GCD of ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["poly", "TI"], 
       StyleBox["1", "TR"]]], "InlineFormula"],
     " and ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["poly", "TI"], 
       StyleBox["2", "TR"]]], "InlineFormula"],
     " treated as univariate polynomials in ",
     Cell[BoxData[
      StyleBox["x", "TI"]], "InlineFormula"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["PolynomialExtendedGCD",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/PolynomialExtendedGCD"], "[", 
       RowBox[{
        SubscriptBox[
         StyleBox["poly", "TI"], 
         StyleBox["1", "TR"]], ",", 
        SubscriptBox[
         StyleBox["poly", "TI"], 
         StyleBox["2", "TR"]], ",", 
        StyleBox["x", "TI"], ",", 
        RowBox[{
         ButtonBox["Modulus",
          BaseStyle->"Link",
          ButtonData->"paclet:ref/Modulus"], "->", 
         StyleBox["p", "TI"]}]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the extended GCD over the integers mod prime ",
     Cell[BoxData[
      StyleBox["p", "TI"]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->159]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->1],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(1)", "ExampleCount"]
}], "ExampleSection",
 CellID->128247013],

Cell["Compute the extended GCD:", "ExampleText",
 CellID->468623224],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"f", ",", "g"}], "}"}], "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"2", 
       RowBox[{"x", "^", "5"}]}], "-", 
      RowBox[{"2", "x"}]}], ",", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{"x", "^", "2"}], "-", "1"}], ")"}], "^", "2"}]}], "}"}]}], 
  ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->241327354],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"d", ",", 
    RowBox[{"{", 
     RowBox[{"a", ",", "b"}], "}"}]}], "}"}], "=", 
  RowBox[{"PolynomialExtendedGCD", "[", 
   RowBox[{"f", ",", "g", ",", "x"}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->138577927],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"-", "1"}], "+", 
    SuperscriptBox["x", "2"]}], ",", 
   RowBox[{"{", 
    RowBox[{
     FractionBox["x", "4"], ",", 
     RowBox[{
      RowBox[{"-", "1"}], "-", 
      FractionBox[
       SuperscriptBox["x", "2"], "2"]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{150, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->853511675]
}, Open  ]],

Cell["\<\
The second part gives coefficients of a linear combination of polynomials \
that yields the GCD:\
\>", "ExampleText",
 CellID->313637936],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"a", " ", "f"}], "+", 
    RowBox[{"b", " ", "g"}]}], "\[Equal]", "d"}], "//", "Expand"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->162916650],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->65137498]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1293636265],

Cell["Polynomials with numeric coefficients:", "ExampleText",
 CellID->400754486],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PolynomialExtendedGCD", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{"x", "-", "1"}], ")"}], 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"x", "-", "2"}], ")"}], "^", "2"}]}], ",", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"x", "-", "1"}], ")"}], 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"x", "^", "2"}], "-", "3"}], ")"}]}], ",", "x"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1555133],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"-", "1"}], "+", "x"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"7", "+", 
      RowBox[{"4", " ", "x"}]}], ",", 
     RowBox[{"9", "-", 
      RowBox[{"4", " ", "x"}]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{162, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->431258031]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->564006120],

Cell["Polynomials with symbolic coefficients:", "ExampleText",
 CellID->45833577],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PolynomialExtendedGCD", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{"x", "-", "a"}], ")"}], 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"b", " ", "x"}], "-", "c"}], ")"}], "^", "2"}]}], ",", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"x", "-", "a"}], ")"}], 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"x", "^", "2"}], "-", 
      RowBox[{"b", " ", "c"}]}], ")"}]}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->102916897],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWc1uEzEQ9u5m0zRJCUnTghBCaUnVAP0RHED0QACJvwISvQGnRBVSe0Cg
0leANwAOXBAvkheCc47L2mtnt9/Oxs7uKg0VK8W7Y8+MvxmP7bHzsn988O59
//hwv996ctT/eHC4/6n1+MORX+VYjFmfGWOdi8z/9vwv8ePPF//H2AteOqrO
Yx2grwDdAvpmRvm2QPEcaq8DfRnoVaC3M8pf/Y8CUOxC7QbQl4BuA72ZUV6g
Yl2/5JUlaJ33C9cnF6C+KPldqrfCiJbSywncVVm/eOq9M3aPl25U1v9mcyBT
JrmqwOWSXAXNyNikVNNIV4Xkqp9BmxreaLQgUrBfLuqQOsqyFSOimMoaU20V
oBfPgA1No9GYIMIyW9LMeRRmGXvE+0az3PZJm7Ivl3lOaW8Yacu6ev1bdgUr
WNcvC0mrsdzDmHgciaEuJYzW3Ik0VGV9Y0YxBa2c+5Z8n9eMNs6cZQIPJ5co
nCn0YDZ+2nhq0k9u5C13kgg3ninqQJekNMYD9op6Y6hS6Lg2Q1jQLxirdirv
NoDGscdesB1jTCePMTrt/nHO5uPFqFa+rlyY0IrShPLoxWn3n48XcexR67ix
pFDiyU0nj2flafefbkavAY254iOgHwBdAxrP/sifbqw5Sr7K4fz8Iet/pdKq
Tgi4A6mdWWVKqA1H5hXQz4D+aYSOzPa0uaLYBcP7ARbuj4hyj0Bp4jsrUOsE
L9WnK/0TR6hai1I/RkhwrhgMBix4HvLC6fV6st05MSoh347g63Q6w+GQNy3k
pAfPOdx+yye53RY5MjwHpnJKZTHGOHr+j5HHMVrnU8YDosFo/a1Bkzx3olmZ
mis42ry4n6oHXN/Ptiaer427F7wj2wteeKsR9fuGRl7NSnVDuAb6yqBPl/mY
WSX2tJT3cLfJWpo3HvVbGXqma+n1uD2phhMoMUvP08tJ9193gS6S0motW/dM
buwwa0jmjnKtGyGpSCS6LDLwFZ6qJ4jACL1DtSf6aNuoj3Q+wtycRqJ8tORR
Z+Q9L8t9EsZLIUGbsCc2B036nZN8xQT+EokLoycJVymGa3XklfH3NxgFTkwC
V+HNGAdGKZ5S4jp5jSO9wRswAmgJm0CzEtqJsyxCPyW9q1s18PRRM9KC3qBX
ad3Ns7hHEXkMdfOx64U7ni1HkcqHzwEf6tnStIvzHnsLtV8lN/4zillCYMNr
8OJ34GpBexfoG+L9BqS+SeQrUN/1KAv4Z/B/PrP+AqTA9AQ=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{389, 40},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->245168198]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->331605263],

Cell["Relatively prime polynomials:", "ExampleText",
 CellID->233999856],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PolynomialExtendedGCD", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"a", " ", 
     RowBox[{"x", "^", "2"}]}], "+", 
    RowBox[{"b", " ", "x"}], "+", "c"}], ",", 
   RowBox[{"x", "-", "r"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->180588769],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1", ",", 
   RowBox[{"{", 
    RowBox[{
     FractionBox["1", 
      RowBox[{"c", "+", 
       RowBox[{"b", " ", "r"}], "+", 
       RowBox[{"a", " ", 
        SuperscriptBox["r", "2"]}]}]], ",", 
     RowBox[{
      FractionBox[
       RowBox[{
        RowBox[{"-", "b"}], "-", 
        RowBox[{"a", " ", "r"}]}], 
       RowBox[{"c", "+", 
        RowBox[{"b", " ", "r"}], "+", 
        RowBox[{"a", " ", 
         SuperscriptBox["r", "2"]}]}]], "-", 
      FractionBox[
       RowBox[{"a", " ", "x"}], 
       RowBox[{"c", "+", 
        RowBox[{"b", " ", "r"}], "+", 
        RowBox[{"a", " ", 
         SuperscriptBox["r", "2"]}]}]]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{302, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->461590073]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Options",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->257656258],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Modulus",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->266299794],

Cell["Extended GCD over the integers:", "ExampleText",
 CellID->348062617],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PolynomialExtendedGCD", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"(", 
      RowBox[{"x", "-", "1"}], ")"}], "^", "2"}], 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"x", "-", "2"}], ")"}], "^", "2"}]}], ",", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"x", "-", "1"}], ")"}], 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"x", "^", "2"}], "-", "3"}], ")"}]}], ",", "x"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->456546624],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"-", "1"}], "+", "x"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      FractionBox["19", "2"], "+", 
      FractionBox[
       RowBox[{"11", " ", "x"}], "2"]}], ",", 
     RowBox[{
      RowBox[{"-", "13"}], "+", 
      RowBox[{"18", " ", "x"}], "-", 
      FractionBox[
       RowBox[{"11", " ", 
        SuperscriptBox["x", "2"]}], "2"]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{253, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->61483159]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->112350548],

Cell["Extended GCD over the integers modulo 2:", "ExampleText",
 CellID->331001379],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PolynomialExtendedGCD", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"(", 
      RowBox[{"x", "-", "1"}], ")"}], "^", "2"}], 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"x", "-", "2"}], ")"}], "^", "2"}]}], ",", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"x", "-", "1"}], ")"}], 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"x", "^", "2"}], "-", "3"}], ")"}]}], ",", "x", ",", 
   RowBox[{"Modulus", "\[Rule]", "2"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->103109475],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"1", "+", 
    SuperscriptBox["x", "2"]}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", 
     RowBox[{"1", "+", "x"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{121, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->290167073]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->258228157],

Cell[TextData[{
 "Given polynomials ",
 Cell[BoxData[
  StyleBox["a", "TI"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  StyleBox["b", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["c", "TI"]], "InlineFormula"],
 ", find polynomials ",
 Cell[BoxData[
  StyleBox["f", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["g", "TI"]], "InlineFormula"],
 " such that ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    RowBox[{
     StyleBox["a", "TI"], " ", 
     StyleBox["f", "TI"]}], "+", 
    RowBox[{
     StyleBox["b", "TI"], " ", 
     StyleBox["g", "TI"]}]}], "==", 
   StyleBox["c", "TI"]}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->43578932],

Cell[BoxData[{
 RowBox[{
  RowBox[{"a", "=", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"x", "-", "1"}], ")"}], 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"x", "^", "2"}], "+", 
      RowBox[{"7", "x"}], "-", "9"}], ")"}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"b", "=", 
   RowBox[{
    RowBox[{
     RowBox[{"(", 
      RowBox[{"x", "-", "1"}], ")"}], "^", "2"}], 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"x", "^", "3"}], "-", 
      RowBox[{"5", "x"}], "+", "3"}], ")"}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"c", "=", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"x", "-", "1"}], ")"}], 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"x", "^", "2"}], "-", "7"}], ")"}]}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->71781401],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"d", ",", 
    RowBox[{"{", 
     RowBox[{"r", ",", "s"}], "}"}]}], "}"}], "=", 
  RowBox[{"PolynomialExtendedGCD", "[", 
   RowBox[{"a", ",", "b", ",", "x"}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->19525921],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"-", "1"}], "+", "x"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      FractionBox["1244", "579"], "-", 
      FractionBox[
       RowBox[{"2360", " ", "x"}], "579"], "+", 
      FractionBox[
       RowBox[{"53", " ", 
        SuperscriptBox["x", "2"]}], "579"], "+", 
      FractionBox[
       RowBox[{"484", " ", 
        SuperscriptBox["x", "3"]}], "579"]}], ",", 
     RowBox[{
      RowBox[{"-", 
       FractionBox["3925", "579"]}], "-", 
      FractionBox[
       RowBox[{"484", " ", "x"}], "579"]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{363, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->194170090]
}, Open  ]],

Cell[TextData[{
 "Solution exists if and only if ",
 Cell[BoxData[
  StyleBox["c", "TI"]], "InlineFormula"],
 " is divisible by ",
 Cell[BoxData[
  StyleBox["d", "TI"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->469370764],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"h", "=", 
  RowBox[{"Cancel", "[", 
   RowBox[{"c", "/", "d"}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->726405255],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "7"}], "+", 
  SuperscriptBox["x", "2"]}]], "Output",
 ImageSize->{44, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->296123971]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"f", ",", "g"}], "}"}], "=", 
  RowBox[{"h", 
   RowBox[{"{", 
    RowBox[{"r", ",", "s"}], "}"}]}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->703447182],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "7"}], "+", 
      SuperscriptBox["x", "2"]}], ")"}], " ", 
    RowBox[{"(", 
     RowBox[{
      FractionBox["1244", "579"], "-", 
      FractionBox[
       RowBox[{"2360", " ", "x"}], "579"], "+", 
      FractionBox[
       RowBox[{"53", " ", 
        SuperscriptBox["x", "2"]}], "579"], "+", 
      FractionBox[
       RowBox[{"484", " ", 
        SuperscriptBox["x", "3"]}], "579"]}], ")"}]}], ",", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", 
       FractionBox["3925", "579"]}], "-", 
      FractionBox[
       RowBox[{"484", " ", "x"}], "579"]}], ")"}], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "7"}], "+", 
      SuperscriptBox["x", "2"]}], ")"}]}]}], "}"}]], "Output",
 ImageSize->{445, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->411660741]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"a", " ", "f"}], "+", 
    RowBox[{"b", " ", "g"}]}], "\[Equal]", "c"}], "//", "Expand"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->419327087],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->646939392]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->2123667759],

Cell[TextData[{
 "The extended ",
 "GCD",
 " of ",
 Cell[BoxData[
  StyleBox["f", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["g", "TI"]], "InlineFormula"],
 " is ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    StyleBox["d", "TI"], ",", 
    RowBox[{"{", 
     RowBox[{
      StyleBox["r", "TI"], ",", 
      StyleBox["s", "TI"]}], "}"}]}], "}"}]], "InlineFormula"],
 ", such that ",
 Cell[BoxData[
  RowBox[{
   StyleBox["d", "TI"], "=", 
   RowBox[{
    ButtonBox["GCD",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/GCD"], 
    RowBox[{"(", 
     RowBox[{
      StyleBox["f", "TI"], ",", 
      StyleBox["g", "TI"]}], ")"}]}]}]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    RowBox[{
     StyleBox["r", "TI"], " ", 
     StyleBox["f", "TI"]}], "+", 
    RowBox[{
     StyleBox["s", "TI"], " ", 
     StyleBox["g", "TI"]}]}], "==", 
   StyleBox["d", "TI"]}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->203403629],

Cell[BoxData[{
 RowBox[{
  RowBox[{"f", "=", 
   RowBox[{"2", "a", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"x", "^", "2"}], "-", "1"}], ")"}], "^", "2"}], 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"x", "^", "3"}], "-", "2"}], ")"}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"g", "=", 
   RowBox[{"4", "a", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"x", "-", "1"}], ")"}], "^", "3"}], 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"x", "^", "2"}], "-", "3"}], ")"}]}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->365995544],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"d", ",", 
    RowBox[{"{", 
     RowBox[{"r", ",", "s"}], "}"}]}], "}"}], "=", 
  RowBox[{"PolynomialExtendedGCD", "[", 
   RowBox[{"f", ",", "g", ",", "x"}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->681022368],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"1", "-", 
    RowBox[{"2", " ", "x"}], "+", 
    SuperscriptBox["x", "2"]}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"-", 
       FractionBox["83", 
        RowBox[{"368", " ", "a"}]]}], "+", 
      FractionBox["x", 
       RowBox[{"23", " ", "a"}]], "+", 
      FractionBox[
       RowBox[{"21", " ", 
        SuperscriptBox["x", "2"]}], 
       RowBox[{"368", " ", "a"}]]}], ",", 
     RowBox[{
      FractionBox["3", 
       RowBox[{"368", " ", "a"}]], "-", 
      FractionBox[
       RowBox[{"47", " ", "x"}], 
       RowBox[{"368", " ", "a"}]], "-", 
      FractionBox[
       RowBox[{"7", " ", 
        SuperscriptBox["x", "2"]}], 
       RowBox[{"46", " ", "a"}]], "-", 
      FractionBox[
       RowBox[{"79", " ", 
        SuperscriptBox["x", "3"]}], 
       RowBox[{"736", " ", "a"}]], "-", 
      FractionBox[
       RowBox[{"21", " ", 
        SuperscriptBox["x", "4"]}], 
       RowBox[{"736", " ", "a"}]]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{467, 34},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->418368787]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"r", " ", "f"}], "+", 
    RowBox[{"s", " ", "g"}]}], "\[Equal]", "d"}], "//", "Expand"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->143477264],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->115951279]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  StyleBox["d", "TI"]], "InlineFormula"],
 " is equal to ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["PolynomialGCD",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/PolynomialGCD"], "[", 
   RowBox[{
    StyleBox["f", "TI"], ",", 
    StyleBox["g", "TI"]}], "]"}]], "InlineFormula"],
 " up to a factor not containing x:"
}], "ExampleText",
 CellID->11469175],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PolynomialGCD", "[", 
  RowBox[{"f", ",", "g"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->613567837],

Cell[BoxData[
 RowBox[{"2", " ", "a", " ", 
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "x"}], ")"}], "2"]}]], "Output",
 ImageSize->{78, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->376414353]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Cancel", "[", 
  RowBox[{"d", "/", "%"}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->167665404],

Cell[BoxData[
 FractionBox["1", 
  RowBox[{"2", " ", "a"}]]], "Output",
 ImageSize->{26, 31},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->269553677]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  StyleBox["r", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["s", "TI"]], "InlineFormula"],
 " are uniquely determined by the following ",
 Cell[BoxData[
  ButtonBox["Exponent",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Exponent"]], "InlineFormula"],
 " conditions:"
}], "ExampleText",
 CellID->122740513],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Exponent", "[", 
   RowBox[{"r", ",", "x"}], "]"}], "<", 
  RowBox[{
   RowBox[{"Exponent", "[", 
    RowBox[{"g", ",", "x"}], "]"}], "-", 
   RowBox[{"Exponent", "[", 
    RowBox[{"d", ",", "x"}], "]"}]}]}]], "Input",
 CellLabel->"In[6]:=",
 CellID->20797891],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->84239404]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Exponent", "[", 
   RowBox[{"s", ",", "x"}], "]"}], "<", 
  RowBox[{
   RowBox[{"Exponent", "[", 
    RowBox[{"f", ",", "x"}], "]"}], "-", 
   RowBox[{"Exponent", "[", 
    RowBox[{"d", ",", "x"}], "]"}]}]}]], "Input",
 CellLabel->"In[7]:=",
 CellID->803721],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->517464938]
}, Open  ]],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["Cancel",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Cancel"]], "InlineFormula"],
 " or ",
 Cell[BoxData[
  ButtonBox["PolynomialRemainder",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PolynomialRemainder"]], "InlineFormula"],
 " to prove that ",
 Cell[BoxData[
  StyleBox["d", "TI"]], "InlineFormula"],
 " divides ",
 Cell[BoxData[
  StyleBox["f", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["g", "TI"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->301351900],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Cancel", "[", 
  RowBox[{"f", "/", "d"}], "]"}]], "Input",
 CellLabel->"In[8]:=",
 CellID->25767822],

Cell[BoxData[
 RowBox[{"2", " ", "a", " ", 
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"1", "+", "x"}], ")"}], "2"], " ", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"-", "2"}], "+", 
    SuperscriptBox["x", "3"]}], ")"}]}]], "Output",
 ImageSize->{127, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->337860317]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PolynomialRemainder", "[", 
  RowBox[{"g", ",", "d", ",", "x"}], "]"}]], "Input",
 CellLabel->"In[9]:=",
 CellID->456197911],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[9]=",
 CellID->234074271]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["PolynomialGCD",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/PolynomialGCD"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["ExtendedGCD",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ExtendedGCD"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->414484924],

Cell[TextData[ButtonBox["Polynomial Division",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/PolynomialDivision"]], "MoreAbout",
 CellID->159967961],

Cell[TextData[ButtonBox["New in 6.0: Symbolic Computation",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60AlgebraicComputing"]], "MoreAbout",
 CellID->516479351]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"PolynomialExtendedGCD - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 11, 50.8721274}", "context" -> "System`", 
    "keywords" -> {
     "Aryabhatta's identity", "Bezout identity", "Euclidean algorithm", 
      "extended greatest common divisor", "polynomial Diophantine equations", 
      "polynomial extended GCD"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "PolynomialExtendedGCD[poly_1, poly_2, x] gives the extended GCD of \
poly_1 and poly_2 treated as univariate polynomials in x. \
PolynomialExtendedGCD[poly_1, poly_2, x, Modulus -> p] gives the extended GCD \
over the integers mod prime p.", "synonyms" -> {"polynomial extended"}, 
    "title" -> "PolynomialExtendedGCD", "type" -> "Symbol", "uri" -> 
    "ref/PolynomialExtendedGCD"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[5567, 165, 352, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->1]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 31167, 1193}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 1464, 35, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2087, 62, 1586, 37, 70, "ObjectNameGrid"],
Cell[3676, 101, 1854, 59, 70, "Usage",
 CellID->159]
}, Open  ]],
Cell[CellGroupData[{
Cell[5567, 165, 352, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->1],
Cell[CellGroupData[{
Cell[5944, 179, 148, 5, 70, "ExampleSection",
 CellID->128247013],
Cell[6095, 186, 68, 1, 70, "ExampleText",
 CellID->468623224],
Cell[6166, 189, 419, 17, 70, "Input",
 CellID->241327354],
Cell[CellGroupData[{
Cell[6610, 210, 272, 9, 28, "Input",
 CellID->138577927],
Cell[6885, 221, 449, 17, 54, "Output",
 CellID->853511675]
}, Open  ]],
Cell[7349, 241, 147, 4, 70, "ExampleText",
 CellID->313637936],
Cell[CellGroupData[{
Cell[7521, 249, 202, 7, 28, "Input",
 CellID->162916650],
Cell[7726, 258, 160, 5, 36, "Output",
 CellID->65137498]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[7935, 269, 221, 7, 70, "ExampleSection",
 CellID->1293636265],
Cell[8159, 278, 81, 1, 70, "ExampleText",
 CellID->400754486],
Cell[CellGroupData[{
Cell[8265, 283, 447, 17, 70, "Input",
 CellID->1555133],
Cell[8715, 302, 403, 15, 36, "Output",
 CellID->431258031]
}, Open  ]],
Cell[9133, 320, 125, 3, 70, "ExampleDelimiter",
 CellID->564006120],
Cell[9261, 325, 81, 1, 70, "ExampleText",
 CellID->45833577],
Cell[CellGroupData[{
Cell[9367, 330, 501, 18, 70, "Input",
 CellID->102916897],
Cell[9871, 350, 1480, 28, 70, "Output",
 Evaluatable->False,
 CellID->245168198]
}, Open  ]],
Cell[11366, 381, 125, 3, 70, "ExampleDelimiter",
 CellID->331605263],
Cell[11494, 386, 72, 1, 70, "ExampleText",
 CellID->233999856],
Cell[CellGroupData[{
Cell[11591, 391, 284, 9, 70, "Input",
 CellID->180588769],
Cell[11878, 402, 839, 29, 53, "Output",
 CellID->461590073]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[12766, 437, 222, 7, 70, "ExampleSection",
 CellID->257656258],
Cell[CellGroupData[{
Cell[13013, 448, 228, 7, 70, "ExampleSubsection",
 CellID->266299794],
Cell[13244, 457, 74, 1, 70, "ExampleText",
 CellID->348062617],
Cell[CellGroupData[{
Cell[13343, 462, 476, 18, 70, "Input",
 CellID->456546624],
Cell[13822, 482, 573, 21, 54, "Output",
 CellID->61483159]
}, Open  ]],
Cell[14410, 506, 125, 3, 70, "ExampleDelimiter",
 CellID->112350548],
Cell[14538, 511, 83, 1, 70, "ExampleText",
 CellID->331001379],
Cell[CellGroupData[{
Cell[14646, 516, 519, 18, 70, "Input",
 CellID->103109475],
Cell[15168, 536, 329, 12, 39, "Output",
 CellID->290167073]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[15558, 555, 227, 7, 70, "ExampleSection",
 CellID->258228157],
Cell[15788, 564, 677, 29, 70, "ExampleText",
 CellID->43578932],
Cell[16468, 595, 748, 29, 70, "Input",
 CellID->71781401],
Cell[CellGroupData[{
Cell[17241, 628, 271, 9, 70, "Input",
 CellID->19525921],
Cell[17515, 639, 740, 26, 54, "Output",
 CellID->194170090]
}, Open  ]],
Cell[18270, 668, 231, 9, 70, "ExampleText",
 CellID->469370764],
Cell[CellGroupData[{
Cell[18526, 681, 149, 5, 70, "Input",
 CellID->726405255],
Cell[18678, 688, 222, 8, 36, "Output",
 CellID->296123971]
}, Open  ]],
Cell[CellGroupData[{
Cell[18937, 701, 206, 8, 70, "Input",
 CellID->703447182],
Cell[19146, 711, 947, 34, 54, "Output",
 CellID->411660741]
}, Open  ]],
Cell[CellGroupData[{
Cell[20130, 750, 202, 7, 70, "Input",
 CellID->419327087],
Cell[20335, 759, 161, 5, 36, "Output",
 CellID->646939392]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[20545, 770, 238, 7, 70, "ExampleSection",
 CellID->2123667759],
Cell[20786, 779, 975, 43, 70, "ExampleText",
 CellID->203403629],
Cell[21764, 824, 561, 21, 70, "Input",
 CellID->365995544],
Cell[CellGroupData[{
Cell[22350, 849, 272, 9, 70, "Input",
 CellID->681022368],
Cell[22625, 860, 1157, 40, 55, "Output",
 CellID->418368787]
}, Open  ]],
Cell[CellGroupData[{
Cell[23819, 905, 202, 7, 70, "Input",
 CellID->143477264],
Cell[24024, 914, 161, 5, 36, "Output",
 CellID->115951279]
}, Open  ]],
Cell[24200, 922, 391, 14, 70, "ExampleText",
 CellID->11469175],
Cell[CellGroupData[{
Cell[24616, 940, 132, 4, 70, "Input",
 CellID->613567837],
Cell[24751, 946, 284, 10, 36, "Output",
 CellID->376414353]
}, Open  ]],
Cell[CellGroupData[{
Cell[25072, 961, 125, 4, 70, "Input",
 CellID->167665404],
Cell[25200, 967, 201, 7, 52, "Output",
 CellID->269553677]
}, Open  ]],
Cell[25416, 977, 360, 13, 70, "ExampleText",
 CellID->122740513],
Cell[CellGroupData[{
Cell[25801, 994, 295, 10, 70, "Input",
 CellID->20797891],
Cell[26099, 1006, 160, 5, 36, "Output",
 CellID->84239404]
}, Open  ]],
Cell[CellGroupData[{
Cell[26296, 1016, 293, 10, 70, "Input",
 CellID->803721],
Cell[26592, 1028, 161, 5, 36, "Output",
 CellID->517464938]
}, Open  ]],
Cell[26768, 1036, 548, 22, 70, "ExampleText",
 CellID->301351900],
Cell[CellGroupData[{
Cell[27341, 1062, 124, 4, 70, "Input",
 CellID->25767822],
Cell[27468, 1068, 367, 13, 39, "Output",
 CellID->337860317]
}, Open  ]],
Cell[CellGroupData[{
Cell[27872, 1086, 148, 4, 70, "Input",
 CellID->456197911],
Cell[28023, 1092, 158, 5, 36, "Output",
 CellID->234074271]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[28242, 1104, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[28561, 1115, 471, 17, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[29069, 1137, 319, 9, 70, "MoreAboutSection",
 CellID->414484924],
Cell[29391, 1148, 149, 3, 70, "MoreAbout",
 CellID->159967961],
Cell[29543, 1153, 169, 3, 70, "MoreAbout",
 CellID->516479351]
}, Open  ]],
Cell[29727, 1159, 27, 0, 70, "History"],
Cell[29757, 1161, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

