(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     39628,       1496]
NotebookOptionsPosition[     30924,       1190]
NotebookOutlinePosition[     32328,       1229]
CellTagsIndexPosition[     32242,       1224]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"SameQ" :> Documentation`HelpLookup["paclet:ref/SameQ"], 
          "Equal" :> Documentation`HelpLookup["paclet:ref/Equal"], "Simplify" :> 
          Documentation`HelpLookup["paclet:ref/Simplify"], "FullSimplify" :> 
          Documentation`HelpLookup["paclet:ref/FullSimplify"], "N" :> 
          Documentation`HelpLookup["paclet:ref/N"], "ZeroTest" :> 
          Documentation`HelpLookup["paclet:ref/ZeroTest"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"SameQ\"\>", 2->"\<\"Equal\"\>", 
       3->"\<\"Simplify\"\>", 4->"\<\"FullSimplify\"\>", 5->"\<\"N\"\>", 
       6->"\<\"ZeroTest\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Formula Manipulation" :> 
          Documentation`HelpLookup["paclet:guide/FormulaManipulation"], 
          "Testing Expressions" :> 
          Documentation`HelpLookup["paclet:guide/TestingExpressions"], 
          "New in 6.0: Symbolic Computation" :> 
          Documentation`HelpLookup["paclet:guide/NewIn60AlgebraicComputing"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Formula Manipulation\"\>", 
       2->"\<\"Testing Expressions\"\>", 
       3->"\<\"New in 6.0: Symbolic Computation\"\>", 
       4->"\<\"New in 6.0: Mathematics & Algorithms\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["PossibleZeroQ", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["PossibleZeroQ",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/PossibleZeroQ"], "[", 
       StyleBox["expr", "TI"], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives ",
     Cell[BoxData[
      ButtonBox["True",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/True"]], "InlineFormula"],
     " if basic symbolic and numerical methods suggest that ",
     Cell[BoxData[
      StyleBox["expr", "TI"]], "InlineFormula"],
     " has value zero, and gives ",
     Cell[BoxData[
      ButtonBox["False",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/False"]], "InlineFormula"],
     " otherwise. "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "The general problem of determining whether an expression has value zero is \
undecidable; ",
 Cell[BoxData[
  ButtonBox["PossibleZeroQ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PossibleZeroQ"]], "InlineFormula"],
 " provides a quick but not always accurate test."
}], "Notes",
 CellID->397840670],

Cell[TextData[{
 "With the setting ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Method",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Method"], "->", "\"\<ExactAlgebraics\>\""}]], 
  "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["PossibleZeroQ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PossibleZeroQ"]], "InlineFormula"],
 " will use exact guaranteed methods in the case of explicit algebraic \
numbers. "
}], "Notes",
 CellID->227118640]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->210254011],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(1)", "ExampleCount"]
}], "ExampleSection",
 CellID->7373557],

Cell["Test whether a numeric expression is zero:", "ExampleText",
 CellID->97978609],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PossibleZeroQ", "[", 
  RowBox[{
   RowBox[{"E", "^", 
    RowBox[{"(", 
     RowBox[{"I", " ", 
      RowBox[{"Pi", "/", "4"}]}], ")"}]}], "-", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"-", "1"}], ")"}], "^", 
    RowBox[{"(", 
     RowBox[{"1", "/", "4"}], ")"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->192470371],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->30562916]
}, Open  ]],

Cell["\<\
Test whether a symbolic expression is likely to be identically zero:\
\>", "ExampleText",
 CellID->352539858],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PossibleZeroQ", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{"x", "+", "1"}], ")"}], 
    RowBox[{"(", 
     RowBox[{"x", "-", "1"}], ")"}]}], "-", 
   RowBox[{"x", "^", "2"}], "+", "1"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->3414988],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->13815301]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1293636265],

Cell["Show that a numeric expression is zero:", "ExampleText",
 CellID->311946735],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PossibleZeroQ", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{"E", "+", "Pi"}], ")"}], "^", "2"}], "-", 
   RowBox[{"E", "^", "2"}], "-", 
   RowBox[{"Pi", "^", "2"}], "-", 
   RowBox[{"2", " ", "E", " ", "Pi"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->85421391],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->410372298]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->333474514],

Cell["Show that a numeric expression is nonzero: ", "ExampleText",
 CellID->45724352],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PossibleZeroQ", "[", 
  RowBox[{
   RowBox[{"E", "^", "Pi"}], "-", 
   RowBox[{"Pi", "^", "E"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->234510],

Cell[BoxData["False"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->17296814]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->443186859],

Cell["\<\
Decide that a numeric expression is zero based on approximate computations:\
\>", "ExampleText",
 CellID->70979351],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PossibleZeroQ", "[", 
  RowBox[{
   RowBox[{"2", "^", 
    RowBox[{"(", 
     RowBox[{"2", " ", "I"}], ")"}]}], "-", 
   RowBox[{"2", "^", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "2"}], "I"}], ")"}]}], "-", 
   RowBox[{"2", "I", " ", 
    RowBox[{"Sin", "[", 
     RowBox[{"Log", "[", "4", "]"}], "]"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->115766131],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->63138589]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->162065703],

Cell["\<\
Test whether symbolic expressions are likely to be identically zero:\
\>", "ExampleText",
 CellID->75729155],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PossibleZeroQ", "[", 
  RowBox[{
   RowBox[{"1", "/", "x"}], "+", 
   RowBox[{"1", "/", "y"}], "-", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"x", "+", "y"}], ")"}], "/", 
    RowBox[{"(", 
     RowBox[{"x", " ", "y"}], ")"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->124769453],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->50488657]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PossibleZeroQ", "[", 
  RowBox[{
   RowBox[{"Sqrt", "[", 
    RowBox[{"x", "^", "2"}], "]"}], "-", "x"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->646680615],

Cell[BoxData["False"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->592615501]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Options",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->258300861],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Assumptions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->422654719],

Cell[TextData[{
 "For arbitrary complex ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  StyleBox["f", "TI"]], "InlineFormula"],
 " is not identically zero:"
}], "ExampleText",
 CellID->300705768],

Cell[BoxData[
 RowBox[{
  RowBox[{"f", "=", 
   RowBox[{
    RowBox[{"Sqrt", "[", 
     RowBox[{"x", "^", "2"}], "]"}], "-", "x"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->164361993],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PossibleZeroQ", "[", "f", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->26439952],

Cell[BoxData["False"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->25780141]
}, Open  ]],

Cell[TextData[{
 "When ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    ButtonBox["Re",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Re"], "[", 
    StyleBox["x", "TI"], "]"}], ">", "0"}]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  StyleBox["f", "TI"]], "InlineFormula"],
 " is identically zero:"
}], "ExampleText",
 CellID->772202845],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PossibleZeroQ", "[", 
  RowBox[{"f", ",", 
   RowBox[{"Assumptions", "\[Rule]", 
    RowBox[{
     RowBox[{"Re", "[", "x", "]"}], ">", "0"}]}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->2258349],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->152196139]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Method",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->137015073],

Cell["\<\
By default, numeric approximations may be used to decide that an algebraic \
number is zero:\
\>", "ExampleText",
 CellID->554591652],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"zero", "=", 
  RowBox[{
   RowBox[{"Sqrt", "[", "2", "]"}], "+", 
   RowBox[{"Sqrt", "[", "3", "]"}], "-", 
   RowBox[{"RootReduce", "[", 
    RowBox[{
     RowBox[{"Sqrt", "[", "2", "]"}], "+", 
     RowBox[{"Sqrt", "[", "3", "]"}]}], "]"}]}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->146676826],

Cell[BoxData[
 RowBox[{
  SqrtBox["2"], "+", 
  SqrtBox["3"], "-", 
  RowBox[{"Root", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"1", "-", 
      RowBox[{"10", " ", 
       SuperscriptBox["#1", "2"]}], "+", 
      SuperscriptBox["#1", "4"]}], "&"}], ",", "4"}], "]"}]}]], "Output",
 ImageSize->{236, 20},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->162432838]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PossibleZeroQ", "[", "zero", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->17619470],

Cell[BoxData[
 RowBox[{
  RowBox[{"PossibleZeroQ", "::", "\<\"ztest\"\>"}], ":", 
  " ", "\<\"Unable to decide whether numeric quantities \\!\\({\\@2 + \\@3 - \
\\(\\(Root[\\(\\(\\(\\(\\(\\(1 - \\(\\(10\\\\ \\(\\(Power[\\(\\(\
\[LeftSkeleton] 2 \[RightSkeleton]\\)\\)]\\)\\)\\)\\) + \\(Slot[\\(\\(\
\[LeftSkeleton] 1 \[RightSkeleton]\\)\\)]\\)\\^4\\)\\) &\\)\\), 4, \
0\\)\\)]\\)\\)}\\) are equal to zero. Assuming they are.\"\>"}]], "Message", \
"MSG",
 CellID->445481445],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->55234279]
}, Open  ]],

Cell["Approximate methods may give incorrect positive answers:", "ExampleText",
 CellID->19131711],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PossibleZeroQ", "[", 
  RowBox[{"zero", "+", 
   RowBox[{"10", "^", 
    RowBox[{"-", "100"}]}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->20717127],

Cell[BoxData[
 RowBox[{
  RowBox[{"PossibleZeroQ", "::", "\<\"ztest\"\>"}], ":", 
  " ", "\<\"Unable to decide whether numeric quantities \
\\!\\({1\\/\\(100000000000000000000000 \\(\\(\[LeftSkeleton] 53 \
\[RightSkeleton]\\)\\) 000000000000000000000000\\) + \\@2 + \\@3 - \
\\(\\(Root[\\(\\(\\(\\(\\(\\(1 - \\(\\(10\\\\ \\(\\(Power[\\(\\(\
\[LeftSkeleton] 2 \[RightSkeleton]\\)\\)]\\)\\)\\)\\) + \\(Slot[\\(\\(\
\[LeftSkeleton] 1 \[RightSkeleton]\\)\\)]\\)\\^4\\)\\) &\\)\\), 4, \
0\\)\\)]\\)\\)}\\) are equal to zero. Assuming they are.\"\>"}]], "Message", \
"MSG",
 CellID->481438934],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->407315866]
}, Open  ]],

Cell[TextData[{
 "With ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Method",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Method"], "->", "\"\<ExactAlgebraics\>\""}]], 
  "InlineFormula"],
 " exact methods are used for explicit algebraic numbers:"
}], "ExampleText",
 CellID->647697057],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PossibleZeroQ", "[", 
  RowBox[{"zero", ",", 
   RowBox[{"Method", "\[Rule]", "\"\<ExactAlgebraics\>\""}]}], "]"}]], "Input",\

 CellLabel->"In[4]:=",
 CellID->129626469],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->330727358]
}, Open  ]],

Cell["\<\
For explicit algebraic numbers the answer is provably correct:\
\>", "ExampleText",
 CellID->161883608],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PossibleZeroQ", "[", 
  RowBox[{
   RowBox[{"zero", "+", 
    RowBox[{"10", "^", 
     RowBox[{"-", "100"}]}]}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<ExactAlgebraics\>\""}]}], "]"}]], "Input",\

 CellLabel->"In[5]:=",
 CellID->49013207],

Cell[BoxData["False"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->355892957]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->258228157],

Cell["\<\
Solving polynomial equations requires deciding whether coefficients are zero:\
\
\>", "ExampleText",
 CellID->462942576],

Cell[BoxData[
 RowBox[{
  RowBox[{"QuadraticRoots", "[", 
   RowBox[{"eq_", ",", "x_"}], "]"}], ":=", "\[IndentingNewLine]", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"a", ",", "b", ",", "c", ",", "\[CapitalDelta]"}], "}"}], ",", 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{
      RowBox[{"{", 
       RowBox[{"c", ",", "b", ",", "a"}], "}"}], "=", 
      RowBox[{"CoefficientList", "[", 
       RowBox[{
        RowBox[{"Subtract", "@@", "eq"}], ",", "x"}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"If", "[", 
      RowBox[{
       RowBox[{"!", 
        RowBox[{"PossibleZeroQ", "[", "a", "]"}]}], ",", 
       "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"\[CapitalDelta]", "=", 
         RowBox[{
          RowBox[{"b", "^", "2"}], "-", 
          RowBox[{"4", "a", " ", "c"}]}]}], ";", "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"x", "==", 
          RowBox[{
           RowBox[{"(", 
            RowBox[{
             RowBox[{"-", "b"}], "-", 
             RowBox[{"Sqrt", "[", "\[CapitalDelta]", "]"}]}], ")"}], "/", 
           RowBox[{"(", 
            RowBox[{"2", "a"}], ")"}]}]}], "||", 
         RowBox[{"x", "==", 
          RowBox[{
           RowBox[{"(", 
            RowBox[{
             RowBox[{"-", "b"}], "+", 
             RowBox[{"Sqrt", "[", "\[CapitalDelta]", "]"}]}], ")"}], "/", 
           RowBox[{"(", 
            RowBox[{"2", "a"}], ")"}]}]}]}]}], ",", "\[IndentingNewLine]", 
       RowBox[{"If", "[", 
        RowBox[{
         RowBox[{"!", 
          RowBox[{"PossibleZeroQ", "[", "b", "]"}]}], ",", 
         RowBox[{"x", "==", 
          RowBox[{
           RowBox[{"-", "c"}], "/", "b"}]}], ",", 
         RowBox[{"PossibleZeroQ", "[", "c", "]"}]}], "]"}]}], "]"}]}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->140644417],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"QuadraticRoots", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"2", 
      RowBox[{"Log", "[", "2", "]"}], 
      RowBox[{"x", "^", "2"}]}], "-", 
     RowBox[{
      RowBox[{"Log", "[", "4", "]"}], 
      RowBox[{"x", "^", "2"}]}], "+", "x", "-", "1"}], "\[Equal]", "0"}], ",",
    "x"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->111842500],

Cell[BoxData[
 RowBox[{"x", "\[Equal]", "1"}]], "Output",
 ImageSize->{35, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->13719009]
}, Open  ]],

Cell[TextData[{
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " equation solvers use zero testing automatically:"
}], "ExampleText",
 CellID->414928791],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Reduce", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"2", 
      RowBox[{"Log", "[", "2", "]"}], 
      RowBox[{"x", "^", "2"}]}], "-", 
     RowBox[{
      RowBox[{"Log", "[", "4", "]"}], 
      RowBox[{"x", "^", "2"}]}], "+", "x", "-", "1"}], "\[Equal]", "0"}], ",",
    "x"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->49751432],

Cell[BoxData[
 RowBox[{"x", "\[Equal]", "1"}]], "Output",
 ImageSize->{35, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->79431490]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->2123667759],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["SameQ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SameQ"], "[", 
   RowBox[{
    StyleBox["e", "TI"], ",", " ", "0"}], "]"}]], "InlineFormula"],
 " returns ",
 Cell[BoxData[
  ButtonBox["True",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/True"]], "InlineFormula"],
 " only if ",
 Cell[BoxData[
  StyleBox["e", "TI"]], "InlineFormula"],
 " is explicitly identical to zero:"
}], "ExampleText",
 CellID->608840551],

Cell[BoxData[
 RowBox[{
  RowBox[{"e", "=", 
   RowBox[{
    RowBox[{
     RowBox[{"(", 
      RowBox[{"E", "+", "Pi"}], ")"}], "^", "2"}], "-", 
    RowBox[{"E", "^", "2"}], "-", 
    RowBox[{"Pi", "^", "2"}], "-", 
    RowBox[{"2", " ", "E", " ", "Pi"}]}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->220005133],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"e", "===", "0"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->517452360],

Cell[BoxData["False"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->121624924]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["Equal",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Equal"], "[", 
   RowBox[{
    StyleBox["e", "TI"], ",", " ", "0"}], "]"}]], "InlineFormula"],
 " uses simple tests to decide whether ",
 Cell[BoxData[
  StyleBox["e", "TI"]], "InlineFormula"],
 " is zero:"
}], "ExampleText",
 CellID->9449965],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"e", "\[Equal]", "0"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->732756206],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->130669541]
}, Open  ]],

Cell[TextData[{
 "When ",
 Cell[BoxData[
  ButtonBox["Equal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Equal"]], "InlineFormula"],
 " cannot decide whether an expression is zero it returns unchanged:"
}], "ExampleText",
 CellID->28259697],

Cell[BoxData[
 RowBox[{
  RowBox[{"ee", "=", 
   RowBox[{
    RowBox[{"Log", "[", "4", "]"}], "+", 
    RowBox[{"2", 
     RowBox[{"Log", "[", 
      RowBox[{"Sin", "[", 
       RowBox[{"Pi", "/", "8"}], "]"}], "]"}]}], "-", 
    RowBox[{"Log", "[", 
     RowBox[{"2", "-", 
      RowBox[{"Sqrt", "[", "2", "]"}]}], "]"}]}]}], ";"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->149017419],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ee", "\[Equal]", "0"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->484210522],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"Log", "[", "4", "]"}], "-", 
   RowBox[{"Log", "[", 
    RowBox[{"2", "-", 
     SqrtBox["2"]}], "]"}], "+", 
   RowBox[{"2", " ", 
    RowBox[{"Log", "[", 
     RowBox[{"Sin", "[", 
      FractionBox["\[Pi]", "8"], "]"}], "]"}]}]}], "\[Equal]", 
  "0"}]], "Output",
 ImageSize->{266, 28},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->144428369]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["PossibleZeroQ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PossibleZeroQ"]], "InlineFormula"],
 " uses numeric methods to test whether ",
 Cell[BoxData[
  StyleBox["ee", "TI"]], "InlineFormula"],
 " is zero:"
}], "ExampleText",
 CellID->6783521],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PossibleZeroQ", "[", "ee", "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->176397997],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->110227583]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FullSimplify",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FullSimplify"]], "InlineFormula"],
 " proves symbolically that ",
 Cell[BoxData[
  StyleBox["ee", "TI"]], "InlineFormula"],
 " is zero:"
}], "ExampleText",
 CellID->238594230],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FullSimplify", "[", "ee", "]"}]], "Input",
 CellLabel->"In[7]:=",
 CellID->115975075],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->2517264]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1305812373],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["PossibleZeroQ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PossibleZeroQ"]], "InlineFormula"],
 " may return ",
 Cell[BoxData[
  ButtonBox["True",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/True"]], "InlineFormula"],
 " for nonzero numeric expressions that are close to zero:"
}], "ExampleText",
 CellID->339350021],

Cell[BoxData[
 RowBox[{
  RowBox[{"e", "=", 
   RowBox[{
    RowBox[{"Sqrt", "[", "2", "]"}], "+", 
    RowBox[{"Sqrt", "[", "3", "]"}], "-", 
    RowBox[{"RootReduce", "[", 
     RowBox[{
      RowBox[{"Sqrt", "[", "2", "]"}], "+", 
      RowBox[{"Sqrt", "[", "3", "]"}]}], "]"}], "+", 
    RowBox[{"10", "^", 
     RowBox[{"-", "100"}]}]}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->381678745],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PossibleZeroQ", "[", "e", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->96288496],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->29152809]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FullSimplify", "[", "e", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->849838554],

Cell[BoxData[
 RowBox[{"1", "/", 
  "100000000000000000000000000000000000000000000000000000000000000000000000000\
00000000000000000000000000"}]], "Output",
 ImageSize->{566, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->166017766]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["SameQ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SameQ"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Equal",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Equal"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Simplify",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Simplify"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["FullSimplify",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FullSimplify"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["N",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/N"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["ZeroTest",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ZeroTest"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Formula Manipulation",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/FormulaManipulation"]], "MoreAbout",
 CellID->850038841],

Cell[TextData[ButtonBox["Testing Expressions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/TestingExpressions"]], "MoreAbout",
 CellID->272284170],

Cell[TextData[ButtonBox["New in 6.0: Symbolic Computation",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60AlgebraicComputing"]], "MoreAbout",
 CellID->104215038],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->3068840]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"PossibleZeroQ - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 12, 23.5292134}", "context" -> "System`", 
    "keywords" -> {
     "equivalence testing", "zero recognition", "zero testing", 
      "exact algebraics"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "PossibleZeroQ[expr] gives True if basic symbolic and numerical methods \
suggest that expr has value zero, and gives False otherwise. ", 
    "synonyms" -> {"possible zero q", "zero q", "ZeroQ"}, "title" -> 
    "PossibleZeroQ", "type" -> "Symbol", "uri" -> "ref/PossibleZeroQ"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[6599, 194, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->210254011]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 32099, 1217}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2150, 46, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2773, 73, 1578, 37, 70, "ObjectNameGrid"],
Cell[4354, 112, 910, 28, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[5301, 145, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[5779, 162, 323, 9, 70, "Notes",
 CellID->397840670],
Cell[6105, 173, 457, 16, 70, "Notes",
 CellID->227118640]
}, Closed]],
Cell[CellGroupData[{
Cell[6599, 194, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->210254011],
Cell[CellGroupData[{
Cell[6984, 208, 146, 5, 70, "ExampleSection",
 CellID->7373557],
Cell[7133, 215, 84, 1, 70, "ExampleText",
 CellID->97978609],
Cell[CellGroupData[{
Cell[7242, 220, 356, 13, 28, "Input",
 CellID->192470371],
Cell[7601, 235, 160, 5, 36, "Output",
 CellID->30562916]
}, Open  ]],
Cell[7776, 243, 119, 3, 70, "ExampleText",
 CellID->352539858],
Cell[CellGroupData[{
Cell[7920, 250, 285, 10, 28, "Input",
 CellID->3414988],
Cell[8208, 262, 160, 5, 36, "Output",
 CellID->13815301]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[8417, 273, 221, 7, 70, "ExampleSection",
 CellID->1293636265],
Cell[8641, 282, 82, 1, 70, "ExampleText",
 CellID->311946735],
Cell[CellGroupData[{
Cell[8748, 287, 311, 10, 70, "Input",
 CellID->85421391],
Cell[9062, 299, 161, 5, 36, "Output",
 CellID->410372298]
}, Open  ]],
Cell[9238, 307, 125, 3, 70, "ExampleDelimiter",
 CellID->333474514],
Cell[9366, 312, 85, 1, 70, "ExampleText",
 CellID->45724352],
Cell[CellGroupData[{
Cell[9476, 317, 179, 6, 70, "Input",
 CellID->234510],
Cell[9658, 325, 161, 5, 36, "Output",
 CellID->17296814]
}, Open  ]],
Cell[9834, 333, 125, 3, 70, "ExampleDelimiter",
 CellID->443186859],
Cell[9962, 338, 125, 3, 70, "ExampleText",
 CellID->70979351],
Cell[CellGroupData[{
Cell[10112, 345, 403, 14, 70, "Input",
 CellID->115766131],
Cell[10518, 361, 160, 5, 36, "Output",
 CellID->63138589]
}, Open  ]],
Cell[10693, 369, 125, 3, 70, "ExampleDelimiter",
 CellID->162065703],
Cell[10821, 374, 118, 3, 70, "ExampleText",
 CellID->75729155],
Cell[CellGroupData[{
Cell[10964, 381, 316, 11, 70, "Input",
 CellID->124769453],
Cell[11283, 394, 160, 5, 36, "Output",
 CellID->50488657]
}, Open  ]],
Cell[CellGroupData[{
Cell[11480, 404, 189, 6, 70, "Input",
 CellID->646680615],
Cell[11672, 412, 162, 5, 36, "Output",
 CellID->592615501]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[11883, 423, 222, 7, 70, "ExampleSection",
 CellID->258300861],
Cell[CellGroupData[{
Cell[12130, 434, 232, 7, 70, "ExampleSubsection",
 CellID->422654719],
Cell[12365, 443, 231, 9, 70, "ExampleText",
 CellID->300705768],
Cell[12599, 454, 193, 7, 70, "Input",
 CellID->164361993],
Cell[CellGroupData[{
Cell[12817, 465, 108, 3, 70, "Input",
 CellID->26439952],
Cell[12928, 470, 161, 5, 36, "Output",
 CellID->25780141]
}, Open  ]],
Cell[13104, 478, 339, 14, 70, "ExampleText",
 CellID->772202845],
Cell[CellGroupData[{
Cell[13468, 496, 227, 7, 70, "Input",
 CellID->2258349],
Cell[13698, 505, 161, 5, 36, "Output",
 CellID->152196139]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[13908, 516, 227, 7, 70, "ExampleSubsection",
 CellID->137015073],
Cell[14138, 525, 143, 4, 70, "ExampleText",
 CellID->554591652],
Cell[CellGroupData[{
Cell[14306, 533, 323, 10, 70, "Input",
 CellID->146676826],
Cell[14632, 545, 413, 15, 41, "Output",
 CellID->162432838]
}, Open  ]],
Cell[CellGroupData[{
Cell[15082, 565, 111, 3, 70, "Input",
 CellID->17619470],
Cell[15196, 570, 473, 9, 70, "Message",
 CellID->445481445],
Cell[15672, 581, 160, 5, 36, "Output",
 CellID->55234279]
}, Open  ]],
Cell[15847, 589, 98, 1, 70, "ExampleText",
 CellID->19131711],
Cell[CellGroupData[{
Cell[15970, 594, 181, 6, 70, "Input",
 CellID->20717127],
Cell[16154, 602, 587, 11, 70, "Message",
 CellID->481438934],
Cell[16744, 615, 161, 5, 36, "Output",
 CellID->407315866]
}, Open  ]],
Cell[16920, 623, 290, 10, 70, "ExampleText",
 CellID->647697057],
Cell[CellGroupData[{
Cell[17235, 637, 194, 6, 70, "Input",
 CellID->129626469],
Cell[17432, 645, 161, 5, 36, "Output",
 CellID->330727358]
}, Open  ]],
Cell[17608, 653, 113, 3, 70, "ExampleText",
 CellID->161883608],
Cell[CellGroupData[{
Cell[17746, 660, 266, 9, 70, "Input",
 CellID->49013207],
Cell[18015, 671, 162, 5, 36, "Output",
 CellID->355892957]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[18238, 683, 227, 7, 70, "ExampleSection",
 CellID->258228157],
Cell[18468, 692, 130, 4, 70, "ExampleText",
 CellID->462942576],
Cell[18601, 698, 1854, 54, 70, "Input",
 CellID->140644417],
Cell[CellGroupData[{
Cell[20480, 756, 382, 13, 70, "Input",
 CellID->111842500],
Cell[20865, 771, 186, 6, 36, "Output",
 CellID->13719009]
}, Open  ]],
Cell[21066, 780, 156, 5, 70, "ExampleText",
 CellID->414928791],
Cell[CellGroupData[{
Cell[21247, 789, 373, 13, 70, "Input",
 CellID->49751432],
Cell[21623, 804, 186, 6, 36, "Output",
 CellID->79431490]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[21858, 816, 238, 7, 70, "ExampleSection",
 CellID->2123667759],
Cell[22099, 825, 481, 18, 70, "ExampleText",
 CellID->608840551],
Cell[22583, 845, 320, 11, 70, "Input",
 CellID->220005133],
Cell[CellGroupData[{
Cell[22928, 860, 94, 3, 70, "Input",
 CellID->517452360],
Cell[23025, 865, 162, 5, 36, "Output",
 CellID->121624924]
}, Open  ]],
Cell[23202, 873, 358, 13, 70, "ExampleText",
 CellID->9449965],
Cell[CellGroupData[{
Cell[23585, 890, 99, 3, 70, "Input",
 CellID->732756206],
Cell[23687, 895, 161, 5, 36, "Output",
 CellID->130669541]
}, Open  ]],
Cell[23863, 903, 246, 8, 70, "ExampleText",
 CellID->28259697],
Cell[24112, 913, 386, 13, 70, "Input",
 CellID->149017419],
Cell[CellGroupData[{
Cell[24523, 930, 100, 3, 70, "Input",
 CellID->484210522],
Cell[24626, 935, 444, 16, 49, "Output",
 CellID->144428369]
}, Open  ]],
Cell[25085, 954, 295, 10, 70, "ExampleText",
 CellID->6783521],
Cell[CellGroupData[{
Cell[25405, 968, 110, 3, 70, "Input",
 CellID->176397997],
Cell[25518, 973, 161, 5, 36, "Output",
 CellID->110227583]
}, Open  ]],
Cell[25694, 981, 283, 10, 70, "ExampleText",
 CellID->238594230],
Cell[CellGroupData[{
Cell[26002, 995, 109, 3, 70, "Input",
 CellID->115975075],
Cell[26114, 1000, 156, 5, 36, "Output",
 CellID->2517264]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[26319, 1011, 231, 7, 70, "ExampleSection",
 CellID->1305812373],
Cell[26553, 1020, 371, 12, 70, "ExampleText",
 CellID->339350021],
Cell[26927, 1034, 404, 13, 70, "Input",
 CellID->381678745],
Cell[CellGroupData[{
Cell[27356, 1051, 108, 3, 70, "Input",
 CellID->96288496],
Cell[27467, 1056, 160, 5, 36, "Output",
 CellID->29152809]
}, Open  ]],
Cell[CellGroupData[{
Cell[27664, 1066, 108, 3, 70, "Input",
 CellID->849838554],
Cell[27775, 1071, 286, 8, 71, "Output",
 CellID->166017766]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[28122, 1086, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[28441, 1097, 1417, 53, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[29895, 1155, 299, 8, 70, "MoreAboutSection"],
Cell[30197, 1165, 151, 3, 70, "MoreAbout",
 CellID->850038841],
Cell[30351, 1170, 149, 3, 70, "MoreAbout",
 CellID->272284170],
Cell[30503, 1175, 169, 3, 70, "MoreAbout",
 CellID->104215038],
Cell[30675, 1180, 177, 3, 70, "MoreAbout",
 CellID->3068840]
}, Open  ]],
Cell[30867, 1186, 27, 0, 70, "History"],
Cell[30897, 1188, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

