(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     37986,       1404]
NotebookOptionsPosition[     29461,       1102]
NotebookOutlinePosition[     30725,       1138]
CellTagsIndexPosition[     30639,       1133]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Integer and Number\[Hyphen]Theoretical Functions" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/IntegerAndNumberTheoreticalFunctions"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"Integer and Number\[Hyphen]Theoretical Functions\"\>"}, \
"\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"PrimeQ" :> Documentation`HelpLookup["paclet:ref/PrimeQ"], 
          "FactorInteger" :> 
          Documentation`HelpLookup["paclet:ref/FactorInteger"], "SquareFreeQ" :> 
          Documentation`HelpLookup["paclet:ref/SquareFreeQ"], "MoebiusMu" :> 
          Documentation`HelpLookup["paclet:ref/MoebiusMu"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"PrimeQ\"\>", 
       2->"\<\"FactorInteger\"\>", 3->"\<\"SquareFreeQ\"\>", 
       4->"\<\"MoebiusMu\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Number Recognition" :> 
          Documentation`HelpLookup["paclet:guide/NumberRecognition"], 
          "Number Theoretic Functions" :> 
          Documentation`HelpLookup["paclet:guide/NumberTheoreticFunctions"], 
          "New in 6.0: Number Theory & Integer Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60NumberTheoryAndIntegerFunctions"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Number Recognition\"\>", 
       2->"\<\"Number Theoretic Functions\"\>", 
       3->"\<\"New in 6.0: Number Theory & Integer Functions\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["PrimePowerQ", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["PrimePowerQ",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/PrimePowerQ"], "[", 
       StyleBox["expr", "TI"], "]"}]], "InlineFormula"],
     "\[LineSeparator]yields ",
     Cell[BoxData[
      ButtonBox["True",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/True"]], "InlineFormula"],
     " if ",
     Cell[BoxData[
      StyleBox["expr", "TI"]], "InlineFormula"],
     " is a power of a prime number, and yields ",
     Cell[BoxData[
      ButtonBox["False",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/False"]], "InlineFormula"],
     " otherwise. "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->21618]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["PrimePowerQ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/PrimePowerQ"], "[", "1", "]"}]], 
  "InlineFormula"],
 " gives ",
 Cell[BoxData[
  ButtonBox["False",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/False"]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->7718],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["PrimePowerQ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/PrimePowerQ"], "[", 
   RowBox[{
    StyleBox["n", "TI"], ",", 
    RowBox[{
     ButtonBox["GaussianIntegers",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/GaussianIntegers"], "->", 
     ButtonBox["True",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/True"]}]}], "]"}]], "InlineFormula"],
 " determines whether ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " is a Gaussian prime power. "
}], "Notes",
 CellID->20536],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["PrimePowerQ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/PrimePowerQ"], "[", 
   RowBox[{
    StyleBox["m", "TI"], "+", 
    RowBox[{
     ButtonBox["I",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/I"], 
     StyleBox["n", "TI"]}]}], "]"}]], "InlineFormula"],
 " automatically works over the Gaussian integers. "
}], "Notes",
 CellID->9747]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->214675637],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(1)", "ExampleCount"]
}], "ExampleSection",
 CellID->272748317],

Cell["Test whether a number is a power of a prime number:", "ExampleText",
 CellID->175606003],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PrimePowerQ", "[", "8", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1019277352],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->97685465]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PrimePowerQ", "[", "8388609", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->65203526],

Cell[BoxData["False"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->459228979]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->153725027],

Cell["Integers:", "ExampleText",
 CellID->38810103],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PrimePowerQ", "[", "123456789", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1237727275],

Cell[BoxData["False"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->423270053]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->22743318],

Cell["Gaussian integers:", "ExampleText",
 CellID->230920432],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PrimePowerQ", "[", 
  RowBox[{"2", "+", "I"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2086156370],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->337307321]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->179622490],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["PrimePowerQ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrimePowerQ"]], "InlineFormula"],
 " works with numbers of any size:"
}], "ExampleText",
 CellID->6193544],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"n", "=", 
  RowBox[{"RandomInteger", "[", 
   RowBox[{"{", 
    RowBox[{"1*^100", ",", "1*^200"}], "}"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->114974305],

Cell[BoxData[\
"95155780888333325047361754727045221965162335412858482972340499229645756474473\
641487378585717332478836321941291824671737937136259913779482798689449976839940\
233400795707820563366481966201265693023378414"], "Output",
 ImageSize->{583, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->184746926]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PrimePowerQ", "[", "n", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->757301146],

Cell[BoxData["False"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->222474344]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->374233358],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["PrimePowerQ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrimePowerQ"]], "InlineFormula"],
 " threads automatically over lists:"
}], "ExampleText",
 CellID->74429818],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PrimePowerQ", "[", 
  RowBox[{"{", 
   RowBox[{"1", ",", 
    RowBox[{"2", "+", "I"}], ",", "3", ",", 
    RowBox[{"4", "-", 
     RowBox[{"2", "I"}]}], ",", "5", ",", "6", ",", 
    RowBox[{"7", "+", 
     RowBox[{"9", "I"}]}], ",", "8", ",", "9"}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1839322466],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "False", ",", "True", ",", "True", ",", "False", ",", "True", ",", "False", 
   ",", "False", ",", "True", ",", "True"}], "}"}]], "Output",
 ImageSize->{388, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->354796876]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->508157837],

Cell["Gaussian rationals:", "ExampleText",
 CellID->1900032098],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PrimePowerQ", "[", 
  RowBox[{
   RowBox[{"2", "/", "125"}], "-", 
   RowBox[{"11", 
    RowBox[{"I", "/", "125"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1034125700],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->8263169]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FactorInteger", "[", 
  RowBox[{
   RowBox[{"2", "/", "125"}], "-", 
   RowBox[{"11", 
    RowBox[{"I", "/", "125"}]}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->221343178],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"2", "+", "\[ImaginaryI]"}], ",", 
    RowBox[{"-", "3"}]}], "}"}], "}"}]], "Output",
 ImageSize->{86, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->82746588]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Options",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->319279168],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "GaussianIntegers",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->422652588],

Cell["Test whether a number is a power of a Gaussian prime:", "ExampleText",
 CellID->52816472],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PrimePowerQ", "[", 
  RowBox[{"5", ",", " ", 
   RowBox[{"GaussianIntegers", "\[Rule]", " ", "True"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->312668629],

Cell[BoxData["False"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->824930832]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FactorInteger", "[", 
  RowBox[{"5", ",", " ", 
   RowBox[{"GaussianIntegers", "\[Rule]", " ", "True"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1403626551],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "\[ImaginaryI]"}], ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"1", "+", 
      RowBox[{"2", " ", "\[ImaginaryI]"}]}], ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"2", "+", "\[ImaginaryI]"}], ",", "1"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{218, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->361160643]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->3590128],

Cell["The first prime powers that are not prime:", "ExampleText",
 CellID->25177758],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Select", "[", 
  RowBox[{
   RowBox[{"Range", "[", "100", "]"}], ",", 
   RowBox[{
    RowBox[{
     RowBox[{"PrimePowerQ", "[", "#", "]"}], " ", "&&", " ", 
     RowBox[{"!", 
      RowBox[{"PrimeQ", "[", "#", "]"}]}]}], " ", "&"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1515955449],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "4", ",", "8", ",", "9", ",", "16", ",", "25", ",", "27", ",", "32", ",", 
   "49", ",", "64", ",", "81"}], "}"}]], "Output",
 ImageSize->{238, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->249966741]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->698482399],

Cell["\<\
The infinite sum of reciprocals of prime powers which are not prime \
converges:\
\>", "ExampleText",
 CellID->1467261848],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Total", "[", 
  RowBox[{"1.", "/", 
   RowBox[{"Select", "[", 
    RowBox[{
     RowBox[{"Range", "[", 
      RowBox[{"10", "^", "5"}], "]"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"PrimePowerQ", "[", "#", "]"}], " ", "&&", " ", 
       RowBox[{"!", 
        RowBox[{"PrimeQ", "[", "#", "]"}]}]}], " ", "&"}]}], "]"}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->650280108],

Cell[BoxData["0.7725824496683651`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->86575546]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->647522041],

Cell[TextData[{
 "The number of prime powers in intervals of size ",
 Cell[BoxData[
  FormBox["1000", TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->73017792],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"Count", "[", 
    RowBox[{
     RowBox[{"Range", "[", 
      RowBox[{
       RowBox[{"10", "^", "i"}], ",", 
       RowBox[{
        RowBox[{"10", "^", "i"}], "+", "999"}]}], "]"}], ",", 
     RowBox[{"_", "?", "PrimePowerQ"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"i", ",", "0", ",", "20"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1828713773],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "193", ",", "187", ",", "175", ",", "140", ",", "108", ",", "82", ",", "75",
    ",", "61", ",", "54", ",", "49", ",", "44", ",", "47", ",", "37", ",", 
   "34", ",", "30", ",", "24", ",", "20", ",", "27", ",", "23", ",", "28", 
   ",", "24"}], "}"}]], "Output",
 ImageSize->{571, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->547206596]
}, Open  ]],

Cell["A graph showing the growth of the prime powers:", "ExampleText",
 CellID->141085145],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", 
  RowBox[{"Accumulate", "[", "%", "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->859547160],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], 
    LineBox[{{1., 193.}, {2., 380.}, {3., 555.}, {4., 695.}, {5., 803.}, {6., 
     885.}, {7., 960.}, {8., 1021.}, {9., 1075.}, {10., 1124.}, {11., 
     1168.}, {12., 1215.}, {13., 1252.}, {14., 1286.}, {15., 1316.}, {16., 
     1340.}, {17., 1360.}, {18., 1387.}, {19., 1410.}, {20., 1438.}, {21., 
     1462.}}]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 200.},
  PlotRange->{{0., 21.}, {193., 1462.}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 112},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->405812396]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->71114196],

Cell["Recognize Mersenne numbers:", "ExampleText",
 CellID->168530013],

Cell[BoxData[
 RowBox[{
  RowBox[{"mersenneQ", "[", "n_", "]"}], ":=", 
  RowBox[{
   RowBox[{"EvenQ", "[", 
    RowBox[{"n", "+", "1"}], "]"}], "&&", " ", 
   RowBox[{"PrimePowerQ", "[", 
    RowBox[{"n", "+", "1"}], "]"}]}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->386840400],

Cell[TextData[{
 "The number ",
 Cell[BoxData["2147483647"], "InlineFormula"],
 " is a Mersenne number, ",
 Cell[BoxData["524285"], "InlineFormula"],
 " is not:"
}], "ExampleText",
 CellID->96348099],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"mersenneQ", "[", "2147483647", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1246437490],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->91288127]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"mersenneQ", "[", "524285", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->140203617],

Cell[BoxData["False"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->11653156]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->416435302],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["FactorInteger",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FactorInteger"]], "InlineFormula"],
 " to get the decomposition into primes:"
}], "ExampleText",
 CellID->41296261],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PrimePowerQ", "[", "96889010407", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->792875640],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->229040980]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FactorInteger", "[", "96889010407", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->636651582],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"7", ",", "13"}], "}"}], "}"}]], "Output",
 ImageSize->{65, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->406523856]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->114537685],

Cell["The only square-free prime powers are prime numbers:", "ExampleText",
 CellID->594829705],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Select", "[", 
  RowBox[{
   RowBox[{"Range", "[", "100", "]"}], ",", 
   RowBox[{
    RowBox[{
     RowBox[{"SquareFreeQ", "[", "#", "]"}], " ", "&&", " ", 
     RowBox[{"PrimePowerQ", "[", "#", "]"}]}], " ", "&"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->766475150],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "2", ",", "3", ",", "5", ",", "7", ",", "11", ",", "13", ",", "17", ",", 
   "19", ",", "23", ",", "29", ",", "31", ",", "37", ",", "41", ",", "43", 
   ",", "47", ",", "53", ",", "59", ",", "61", ",", "67", ",", "71", ",", 
   "73", ",", "79", ",", "83", ",", "89", ",", "97"}], "}"}]], "Output",
 ImageSize->{360, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->185700289]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PrimeQ", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1927799435],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "True", ",", "True", ",", "True", ",", "True", ",", "True", ",", "True", 
   ",", "True", ",", "True", ",", "True", ",", "True", ",", "True", ",", 
   "True", ",", "True", ",", "True", ",", "True", ",", "True", ",", "True", 
   ",", "True", ",", "True", ",", "True", ",", "True", ",", "True", ",", 
   "True", ",", "True", ",", "True"}], "}"}]], "Output",
 ImageSize->{517, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->466451271]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->727226463],

Cell[TextData[{
 "The sum of divisors of a prime power ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " is less than ",
 Cell[BoxData[
  StyleBox[
   RowBox[{"2", "n"}], "TI"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->290451798],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"DivisorSigma", "[", 
      RowBox[{"1", ",", "#"}], "]"}], " ", "<", " ", 
     RowBox[{"2", "#"}]}], ")"}], " ", "&"}], " ", "/@", " ", 
  RowBox[{"Select", "[", 
   RowBox[{
    RowBox[{"Range", "[", "100", "]"}], ",", "PrimePowerQ"}], 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->421323488],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "True", ",", "True", ",", "True", ",", "True", ",", "True", ",", "True", 
   ",", "True", ",", "True", ",", "True", ",", "True", ",", "True", ",", 
   "True", ",", "True", ",", "True", ",", "True", ",", "True", ",", "True", 
   ",", "True", ",", "True", ",", "True", ",", "True", ",", "True", ",", 
   "True", ",", "True", ",", "True", ",", "True", ",", "True", ",", "True", 
   ",", "True", ",", "True", ",", "True", ",", "True", ",", "True", ",", 
   "True", ",", "True"}], "}"}]], "Output",
 ImageSize->{478, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->36093671]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Neat Examples",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->589267740],

Cell["The distribution of prime powers:", "ExampleText",
 CellID->1741935],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"Boole", "[", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"PrimePowerQ", "[", 
      RowBox[{"a", "+", 
       RowBox[{"b", " ", "I"}]}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"a", ",", "50"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"b", ",", "50"}], "}"}]}], "]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1748729376],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJyNVNsRwjAME2zCCozCCCzA/n8cpbR6uaV37YU4tiVZ4fZ8PZ5XAPf1vSwP
1vfzXbdoT9dyyEPYorBzcRyyC406GAeWRYfmDo+KMAwYHE4f2inBvqItRHrC
ApPvUwnSukR8SxlrF1+nH50mMJKobSceJLLXVn1cVOnQUhIoJpJteJQ+6VFI
u68Cc3fx5J7vuGqaA/KV9j64vM0HYpRBSiENDvQu1c+pRkukvqXlKETUhK2y
BPYfOcAjYIWgS8HdpYrD8ujJDRk1zAGh7NWZhKmDOnmjOYoTTK2mhgsqkhgM
u8Ya3e7gf9eVtHJazfZ1GmeTtplnoN+UUt1ZlL39L22n2r1VfTWDkSh+YzGq
Dnpbtfvn4oppzsenD37V3b2QFZNfOKQdDoXAGFSWRAQBxQ+b9xqL7Ek93kFP
Ce8=
   "], {{0, 0}, {50, 50}}, {0, 1}],
  Frame->Automatic,
  FrameLabel->{None, None},
  FrameTicks->{{None, None}, {None, None}},
  PlotRangeClipping->True]], "Output",
 ImageSize->{184, 180},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->334399586]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["PrimeQ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/PrimeQ"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["FactorInteger",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FactorInteger"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SquareFreeQ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SquareFreeQ"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["MoebiusMu",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/MoebiusMu"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->250839057],

Cell[TextData[ButtonBox["Integer and Number\[Hyphen]Theoretical Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/IntegerAndNumberTheoreticalFunctions"]], "Tutorials",
 CellID->28609]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->59497507],

Cell[TextData[ButtonBox["Number Recognition",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NumberRecognition"]], "MoreAbout",
 CellID->40078263],

Cell[TextData[ButtonBox["Number Theoretic Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NumberTheoreticFunctions"]], "MoreAbout",
 CellID->483987476],

Cell[TextData[ButtonBox["New in 6.0: Number Theory & Integer Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/NewIn60NumberTheoryAndIntegerFunctions"]], "MoreAbout",
 CellID->16044660]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"PrimePowerQ - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 12, 53.4674798}", "context" -> "System`", 
    "keywords" -> {"power of a prime"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "PrimePowerQ[expr] yields True if expr is a power of a prime number, and \
yields False otherwise. ", "synonyms" -> {"prime power q"}, "title" -> 
    "PrimePowerQ", "type" -> "Symbol", "uri" -> "ref/PrimePowerQ"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[7481, 233, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->214675637]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 30496, 1126}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2533, 59, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3156, 86, 1576, 37, 70, "ObjectNameGrid"],
Cell[4735, 125, 867, 28, 70, "Usage",
 CellID->21618]
}, Open  ]],
Cell[CellGroupData[{
Cell[5639, 158, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6117, 175, 333, 14, 70, "Notes",
 CellID->7718],
Cell[6453, 191, 573, 20, 70, "Notes",
 CellID->20536],
Cell[7029, 213, 415, 15, 70, "Notes",
 CellID->9747]
}, Closed]],
Cell[CellGroupData[{
Cell[7481, 233, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->214675637],
Cell[CellGroupData[{
Cell[7866, 247, 148, 5, 70, "ExampleSection",
 CellID->272748317],
Cell[8017, 254, 94, 1, 70, "ExampleText",
 CellID->175606003],
Cell[CellGroupData[{
Cell[8136, 259, 108, 3, 28, "Input",
 CellID->1019277352],
Cell[8247, 264, 160, 5, 36, "Output",
 CellID->97685465]
}, Open  ]],
Cell[CellGroupData[{
Cell[8444, 274, 112, 3, 28, "Input",
 CellID->65203526],
Cell[8559, 279, 162, 5, 36, "Output",
 CellID->459228979]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[8770, 290, 220, 7, 70, "ExampleSection",
 CellID->153725027],
Cell[8993, 299, 51, 1, 70, "ExampleText",
 CellID->38810103],
Cell[CellGroupData[{
Cell[9069, 304, 116, 3, 70, "Input",
 CellID->1237727275],
Cell[9188, 309, 162, 5, 36, "Output",
 CellID->423270053]
}, Open  ]],
Cell[9365, 317, 124, 3, 70, "ExampleDelimiter",
 CellID->22743318],
Cell[9492, 322, 61, 1, 70, "ExampleText",
 CellID->230920432],
Cell[CellGroupData[{
Cell[9578, 327, 131, 4, 70, "Input",
 CellID->2086156370],
Cell[9712, 333, 161, 5, 36, "Output",
 CellID->337307321]
}, Open  ]],
Cell[9888, 341, 125, 3, 70, "ExampleDelimiter",
 CellID->179622490],
Cell[10016, 346, 213, 7, 70, "ExampleText",
 CellID->6193544],
Cell[CellGroupData[{
Cell[10254, 357, 191, 6, 70, "Input",
 CellID->114974305],
Cell[10448, 365, 364, 8, 71, "Output",
 CellID->184746926]
}, Open  ]],
Cell[CellGroupData[{
Cell[10849, 378, 107, 3, 70, "Input",
 CellID->757301146],
Cell[10959, 383, 162, 5, 36, "Output",
 CellID->222474344]
}, Open  ]],
Cell[11136, 391, 125, 3, 70, "ExampleDelimiter",
 CellID->374233358],
Cell[11264, 396, 216, 7, 70, "ExampleText",
 CellID->74429818],
Cell[CellGroupData[{
Cell[11505, 407, 342, 10, 70, "Input",
 CellID->1839322466],
Cell[11850, 419, 312, 9, 36, "Output",
 CellID->354796876]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[12211, 434, 243, 7, 70, "ExampleSection",
 CellID->508157837],
Cell[12457, 443, 63, 1, 70, "ExampleText",
 CellID->1900032098],
Cell[CellGroupData[{
Cell[12545, 448, 204, 7, 70, "Input",
 CellID->1034125700],
Cell[12752, 457, 159, 5, 36, "Output",
 CellID->8263169]
}, Open  ]],
Cell[CellGroupData[{
Cell[12948, 467, 205, 7, 70, "Input",
 CellID->221343178],
Cell[13156, 476, 283, 10, 36, "Output",
 CellID->82746588]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[13488, 492, 222, 7, 70, "ExampleSection",
 CellID->319279168],
Cell[CellGroupData[{
Cell[13735, 503, 237, 7, 70, "ExampleSubsection",
 CellID->422652588],
Cell[13975, 512, 95, 1, 70, "ExampleText",
 CellID->52816472],
Cell[CellGroupData[{
Cell[14095, 517, 188, 5, 70, "Input",
 CellID->312668629],
Cell[14286, 524, 162, 5, 36, "Output",
 CellID->824930832]
}, Open  ]],
Cell[CellGroupData[{
Cell[14485, 534, 191, 5, 70, "Input",
 CellID->1403626551],
Cell[14679, 541, 500, 18, 36, "Output",
 CellID->361160643]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[15240, 566, 225, 7, 70, "ExampleSection",
 CellID->3590128],
Cell[15468, 575, 84, 1, 70, "ExampleText",
 CellID->25177758],
Cell[CellGroupData[{
Cell[15577, 580, 320, 10, 70, "Input",
 CellID->1515955449],
Cell[15900, 592, 298, 9, 36, "Output",
 CellID->249966741]
}, Open  ]],
Cell[16213, 604, 125, 3, 70, "ExampleDelimiter",
 CellID->698482399],
Cell[16341, 609, 132, 4, 70, "ExampleText",
 CellID->1467261848],
Cell[CellGroupData[{
Cell[16498, 617, 419, 14, 70, "Input",
 CellID->650280108],
Cell[16920, 633, 175, 5, 36, "Output",
 CellID->86575546]
}, Open  ]],
Cell[17110, 641, 125, 3, 70, "ExampleDelimiter",
 CellID->647522041],
Cell[17238, 646, 178, 6, 70, "ExampleText",
 CellID->73017792],
Cell[CellGroupData[{
Cell[17441, 656, 430, 14, 70, "Input",
 CellID->1828713773],
Cell[17874, 672, 435, 11, 36, "Output",
 CellID->547206596]
}, Open  ]],
Cell[18324, 686, 90, 1, 70, "ExampleText",
 CellID->141085145],
Cell[CellGroupData[{
Cell[18439, 691, 145, 4, 70, "Input",
 CellID->859547160],
Cell[18587, 697, 753, 20, 133, "Output",
 CellID->405812396]
}, Open  ]],
Cell[19355, 720, 124, 3, 70, "ExampleDelimiter",
 CellID->71114196],
Cell[19482, 725, 70, 1, 70, "ExampleText",
 CellID->168530013],
Cell[19555, 728, 281, 9, 70, "Input",
 CellID->386840400],
Cell[19839, 739, 199, 7, 70, "ExampleText",
 CellID->96348099],
Cell[CellGroupData[{
Cell[20063, 750, 115, 3, 70, "Input",
 CellID->1246437490],
Cell[20181, 755, 160, 5, 36, "Output",
 CellID->91288127]
}, Open  ]],
Cell[CellGroupData[{
Cell[20378, 765, 110, 3, 70, "Input",
 CellID->140203617],
Cell[20491, 770, 161, 5, 36, "Output",
 CellID->11653156]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[20701, 781, 237, 7, 70, "ExampleSection",
 CellID->416435302],
Cell[20941, 790, 233, 8, 70, "ExampleText",
 CellID->41296261],
Cell[CellGroupData[{
Cell[21199, 802, 117, 3, 70, "Input",
 CellID->792875640],
Cell[21319, 807, 161, 5, 36, "Output",
 CellID->229040980]
}, Open  ]],
Cell[CellGroupData[{
Cell[21517, 817, 119, 3, 70, "Input",
 CellID->636651582],
Cell[21639, 822, 228, 8, 36, "Output",
 CellID->406523856]
}, Open  ]],
Cell[21882, 833, 125, 3, 70, "ExampleDelimiter",
 CellID->114537685],
Cell[22010, 838, 95, 1, 70, "ExampleText",
 CellID->594829705],
Cell[CellGroupData[{
Cell[22130, 843, 302, 9, 70, "Input",
 CellID->766475150],
Cell[22435, 854, 470, 11, 54, "Output",
 CellID->185700289]
}, Open  ]],
Cell[CellGroupData[{
Cell[22942, 870, 103, 3, 70, "Input",
 CellID->1927799435],
Cell[23048, 875, 528, 12, 54, "Output",
 CellID->466451271]
}, Open  ]],
Cell[23591, 890, 125, 3, 70, "ExampleDelimiter",
 CellID->727226463],
Cell[23719, 895, 253, 10, 70, "ExampleText",
 CellID->290451798],
Cell[CellGroupData[{
Cell[23997, 909, 382, 13, 70, "Input",
 CellID->421323488],
Cell[24382, 924, 665, 14, 71, "Output",
 CellID->36093671]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[25096, 944, 228, 7, 70, "ExampleSection",
 CellID->589267740],
Cell[25327, 953, 74, 1, 70, "ExampleText",
 CellID->1741935],
Cell[CellGroupData[{
Cell[25426, 958, 410, 13, 70, "Input",
 CellID->1748729376],
Cell[25839, 973, 786, 19, 201, "Output",
 CellID->334399586]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[26686, 999, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[27005, 1010, 959, 35, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[28001, 1050, 318, 9, 70, "TutorialsSection",
 CellID->250839057],
Cell[28322, 1061, 198, 4, 70, "Tutorials",
 CellID->28609]
}, Open  ]],
Cell[CellGroupData[{
Cell[28557, 1070, 318, 9, 70, "MoreAboutSection",
 CellID->59497507],
Cell[28878, 1081, 146, 3, 70, "MoreAbout",
 CellID->40078263],
Cell[29027, 1086, 162, 3, 70, "MoreAbout",
 CellID->483987476],
Cell[29192, 1091, 197, 4, 70, "MoreAbout",
 CellID->16044660]
}, Open  ]],
Cell[29404, 1098, 27, 0, 70, "History"],
Cell[29434, 1100, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

