(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     41901,       1543]
NotebookOptionsPosition[     33666,       1249]
NotebookOutlinePosition[     34958,       1286]
CellTagsIndexPosition[     34872,       1281]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Descriptive Statistics" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/DescriptiveStatistics"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Descriptive Statistics\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Quantile" :> Documentation`HelpLookup["paclet:ref/Quantile"], 
          "InterquartileRange" :> 
          Documentation`HelpLookup["paclet:ref/InterquartileRange"], 
          "QuartileDeviation" :> 
          Documentation`HelpLookup["paclet:ref/QuartileDeviation"], 
          "QuartileSkewness" :> 
          Documentation`HelpLookup["paclet:ref/QuartileSkewness"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Quantile\"\>", 
       2->"\<\"InterquartileRange\"\>", 3->"\<\"QuartileDeviation\"\>", 
       4->"\<\"QuartileSkewness\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Descriptive Statistics" :> 
          Documentation`HelpLookup["paclet:guide/DescriptiveStatistics"], 
          "New in 6.0: Statistics" :> 
          Documentation`HelpLookup["paclet:guide/NewIn60Statistics"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"Descriptive Statistics\"\>", 
       2->"\<\"New in 6.0: Statistics\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["Quartiles", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["Quartiles",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Quartiles"], "[", 
       StyleBox["list", "TI"], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives a list of the 1/4, 1/2, and 3/4 quantiles of the \
elements in ",
     Cell[BoxData[
      StyleBox["list", "TI"]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->11812]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["Quartiles",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Quartiles"], "[", 
   StyleBox["list", "TI"], "]"}]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Quantile",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Quantile"], "[", 
   RowBox[{
    StyleBox["list", "TI"], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"1", "/", "4"}], ",", 
      RowBox[{"1", "/", "2"}], ",", 
      RowBox[{"3", "/", "4"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"1", "/", "2"}], ",", "0"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "1"}], "}"}]}], "}"}]}], "]"}]], "InlineFormula"],
 "."
}], "Notes",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->14560],

Cell[TextData[{
 "The second quartile is equivalent to ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Median",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Median"], "[", 
   StyleBox["list", "TI"], "]"}]], "InlineFormula"],
 "."
}], "Notes",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->724085966],

Cell[TextData[{
 "For even ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Length",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Length"], "[", 
   StyleBox["list", "TI"], "]"}]], "InlineFormula"],
 ", the first quartile is equivalent to the median of the ",
 Cell[BoxData[
  FormBox[
   FractionBox["n", "2"], TraditionalForm]], "InlineMath"],
 " smallest elements in ",
 Cell[BoxData[
  StyleBox["list", "TI"]], "InlineFormula"],
 "."
}], "Notes",
 CellID->365953173],

Cell[TextData[{
 "For odd ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Length",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Length"], "[", 
   StyleBox["list", "TI"], "]"}]], "InlineFormula"],
 ", the first quartile is equivalent to the average of the median of the ",
 Cell[BoxData[
  FormBox[
   FractionBox[
    RowBox[{"n", "-", "1"}], "2"], TraditionalForm]], "InlineMath"],
 " smallest elements and the median of the ",
 Cell[BoxData[
  FormBox[
   FractionBox[
    RowBox[{"n", "+", "1"}], "2"], TraditionalForm]], "InlineMath"],
 " smallest elements in ",
 Cell[BoxData[
  StyleBox["list", "TI"]], "InlineFormula"],
 "."
}], "Notes",
 CellID->42875104],

Cell["\<\
The third quartile is defined as for the first, but with the largest rather \
than smallest elements.\
\>", "Notes",
 CellID->215460642],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Quartiles",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Quartiles"]], "InlineFormula"],
 " works with ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 " objects."
}], "Notes",
 CellID->11686772]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->109541307],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(1)", "ExampleCount"]
}], "ExampleSection",
 CellID->47281233],

Cell["Quartiles for a list of exact numbers:", "ExampleText",
 CellID->31273586],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Quartiles", "[", 
  RowBox[{"{", 
   RowBox[{"1", ",", "3", ",", "4", ",", "2", ",", "5", ",", "6"}], "}"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->31907],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"2", ",", 
   FractionBox["7", "2"], ",", "5"}], "}"}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{66, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->632913276]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->16338],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Quartiles",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Quartiles"]], "InlineFormula"],
 " works with any real numeric quantities:"
}], "ExampleText",
 CellID->13924],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Quartiles", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"Sqrt", "[", "2", "]"}], ",", "E", ",", "Pi", ",", 
    RowBox[{"Pi", "^", "2"}], ",", "1", ",", "2", ",", "3"}], "}"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19854],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    SqrtBox["2"], "+", 
    RowBox[{
     FractionBox["1", "4"], " ", 
     RowBox[{"(", 
      RowBox[{"2", "-", 
       SqrtBox["2"]}], ")"}]}]}], ",", "\[ExponentialE]", ",", 
   RowBox[{"3", "+", 
    RowBox[{
     FractionBox["3", "4"], " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "3"}], "+", "\[Pi]"}], ")"}]}]}]}], "}"}]], "Output",
 ImageSize->{239, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->107049843]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->64612447],

Cell["Quartiles by column for a matrix:", "ExampleText",
 CellID->281606070],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Quartiles", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"3", ",", "5"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "8"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"5", ",", "6"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"7", ",", "8"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"2", ",", "4"}], "}"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->421309609],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     FractionBox["7", "4"], ",", "3", ",", 
     FractionBox["11", "2"]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     FractionBox["19", "4"], ",", "6", ",", "8"}], "}"}]}], "}"}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{171, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->679059439]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->158391064],

Cell["Obtain results at any precision: ", "ExampleText",
 CellID->160553898],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Quartiles", "[", 
  RowBox[{"N", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"5", ",", "10", ",", "4", ",", "25", ",", "2", ",", "1"}], "}"}],
     ",", "30"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->190699768],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"2.`30.", ",", "4.5`29.72379358806105", ",", "10.`30."}], 
  "}"}]], "Output",
 ImageSize->{464, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->817440611]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->650909219],

Cell["Compute results for a large vector or matrix:", "ExampleText",
 CellID->285474028],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Quartiles", "[", 
  RowBox[{"RandomReal", "[", 
   RowBox[{"1", ",", 
    RowBox[{"10", "^", "5"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->315224713],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.25115962921260415`", ",", "0.5005372347322116`", ",", 
   "0.7538770493168108`"}], "}"}]], "Output",
 ImageSize->{202, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->237324472]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Quartiles", "[", 
  RowBox[{"RandomReal", "[", 
   RowBox[{"1", ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"10", "^", "4"}], ",", "5"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->594479966],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "0.25324590160101956`", ",", "0.5047302522495375`", ",", 
     "0.7502173852628918`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "0.25374660966365825`", ",", "0.5048979797138218`", ",", 
     "0.7475229585151127`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "0.25470212206812437`", ",", "0.4995996007421626`", ",", 
     "0.7543753943174064`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "0.243940751723966`", ",", "0.49121050576461145`", ",", 
     "0.7543473839012669`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "0.24667972970884122`", ",", "0.4963241792466667`", ",", 
     "0.7495940022941152`"}], "}"}]}], "}"}]], "Output",
 ImageSize->{431, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->29249989]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->22424],

Cell[TextData[{
 "Compute results for a ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->625271644],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sp", "=", 
  RowBox[{"SparseArray", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"{", 
        RowBox[{"i_", ",", "i_"}], "}"}], ":>", "i"}], ",", 
      RowBox[{
       RowBox[{
        RowBox[{"{", 
         RowBox[{"i_", ",", "j_"}], "}"}], "/;", 
        RowBox[{"j", "<", "i"}]}], "\[RuleDelayed]", 
       RowBox[{"i", "+", "j"}]}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"100", ",", "10"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19878761],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "955", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 955, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"100", ",", "10"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{202, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->163631235]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Quartiles", "[", "sp", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->235263230],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     FractionBox["53", "2"], ",", 
     FractionBox["103", "2"], ",", 
     FractionBox["153", "2"]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     FractionBox["55", "2"], ",", 
     FractionBox["105", "2"], ",", 
     FractionBox["155", "2"]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     FractionBox["57", "2"], ",", 
     FractionBox["107", "2"], ",", 
     FractionBox["157", "2"]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     FractionBox["59", "2"], ",", 
     FractionBox["109", "2"], ",", 
     FractionBox["159", "2"]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     FractionBox["61", "2"], ",", 
     FractionBox["111", "2"], ",", 
     FractionBox["161", "2"]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     FractionBox["63", "2"], ",", 
     FractionBox["113", "2"], ",", 
     FractionBox["163", "2"]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     FractionBox["65", "2"], ",", 
     FractionBox["115", "2"], ",", 
     FractionBox["165", "2"]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     FractionBox["67", "2"], ",", 
     FractionBox["117", "2"], ",", 
     FractionBox["167", "2"]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     FractionBox["69", "2"], ",", 
     FractionBox["119", "2"], ",", 
     FractionBox["169", "2"]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     FractionBox["71", "2"], ",", 
     FractionBox["121", "2"], ",", 
     FractionBox["171", "2"]}], "}"}]}], "}"}]], "Output",
 ImageSize->{471, 90},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->164473528]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->26059],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Quartiles",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Quartiles"]], "InlineFormula"],
 " are given by linearly interpolated ",
 Cell[BoxData[
  ButtonBox["Quantile",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Quantile"]], "InlineFormula"],
 " values:",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->14560,
  ButtonNote->"14560"]
}], "ExampleText",
 CellID->38722882],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"data", "=", 
  RowBox[{"RandomReal", "[", 
   RowBox[{"10", ",", "20"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->133352675],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.9736327064468453`", ",", "0.9849637870466044`", ",", "9.33201757016192`",
    ",", "6.376307177628258`", ",", "6.019169547927166`", ",", 
   "5.774683632253847`", ",", "8.193152964290372`", ",", 
   "1.8023521186246239`", ",", "5.2346214404904945`", ",", 
   "6.431938402897073`", ",", "6.775857766393707`", ",", "3.107833726333693`",
    ",", "9.028407210056226`", ",", "7.6343162686734845`", ",", 
   "3.401081145087459`", ",", "7.451667806782565`", ",", 
   "1.4111252864484358`", ",", "3.1127351673631054`", ",", 
   "7.6085423255390765`", ",", "5.704894849279789`"}], "}"}]], "Output",
 ImageSize->{429, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->73587842]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Quartiles", "[", "data", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->632052870],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "3.110284446848399`", ",", "5.896926590090507`", ",", 
   "7.530105066160821`"}], "}"}]], "Output",
 ImageSize->{188, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->6564546]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Quantile", "[", 
  RowBox[{"data", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"1", "/", "4"}], ",", 
     RowBox[{"1", "/", "2"}], ",", 
     RowBox[{"3", "/", "4"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"1", "/", "2"}], ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "1"}], "}"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->155558425],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "3.110284446848399`", ",", "5.896926590090507`", ",", 
   "7.530105066160821`"}], "}"}]], "Output",
 ImageSize->{188, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->359744003]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->181243417],

Cell[TextData[{
 "The second quartile is the ",
 Cell[BoxData[
  ButtonBox["Median",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Median"]], "InlineFormula"],
 ":",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->724085966,
  ButtonNote->"724085966"]
}], "ExampleText",
 CellID->381938433],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"data", "=", 
  RowBox[{"RandomReal", "[", 
   RowBox[{"10", ",", "20"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->27788033],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "5.944414993056439`", ",", "8.312506306560092`", ",", "6.082554521865696`", 
   ",", "4.8490813665060255`", ",", "0.3940692536413781`", ",", 
   "5.671046055239753`", ",", "8.260062603588034`", ",", "9.843785537127232`",
    ",", "7.349529263771961`", ",", "4.463779450963767`", ",", 
   "6.074667770989192`", ",", "2.9851848442712625`", ",", 
   "7.198336882201291`", ",", "4.134654738497359`", ",", "8.844380246213694`",
    ",", "2.444549573932686`", ",", "5.777939739027671`", ",", 
   "2.779218957552283`", ",", "1.3127701320521257`", ",", 
   "2.058582602064345`"}], "}"}]], "Output",
 ImageSize->{429, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->861386740]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Quartiles", "[", "data", "]"}], "[", 
  RowBox[{"[", "2", "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->91770469],

Cell[BoxData["5.724492897133712`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->540519151]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Median", "[", "data", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->239513108],

Cell[BoxData["5.724492897133712`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->406471397]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->103804862],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["InterquartileRange",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/InterquartileRange"]], "InlineFormula"],
 " is the difference between the first and third quartiles:"
}], "ExampleText",
 CellID->15299726],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"data", "=", 
  RowBox[{"RandomReal", "[", 
   RowBox[{"10", ",", "20"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->118537781],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "9.015200919531638`", ",", "6.840491579691066`", ",", "6.571783066597343`", 
   ",", "2.3713873986982392`", ",", "6.3625023650713715`", ",", 
   "8.108717247648185`", ",", "9.082051671805152`", ",", 
   "2.4743234941209624`", ",", "3.101810950356154`", ",", 
   "9.842396046528751`", ",", "2.31953703748657`", ",", "6.71159600830173`", 
   ",", "0.8988420499533167`", ",", "6.489397098251464`", ",", 
   "2.0399184046610097`", ",", "7.0844693465431625`", ",", 
   "8.302931683954471`", ",", "3.6455071961983854`", ",", 
   "7.883245583950769`", ",", "6.755509640249688`"}], "}"}]], "Output",
 ImageSize->{422, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->447399351]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Apply", "[", 
  RowBox[{"Subtract", ",", 
   RowBox[{
    RowBox[{"Quartiles", "[", "data", "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"{", 
      RowBox[{"3", ",", "1"}], "}"}], "]"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->23051440],

Cell[BoxData["5.207914193560919`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->8718643]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"InterquartileRange", "[", "data", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->481300380],

Cell[BoxData["5.207914193560919`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->568061497]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->631682428],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["QuartileDeviation",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/QuartileDeviation"]], "InlineFormula"],
 " is half the difference between the first and third quartiles:"
}], "ExampleText",
 CellID->533304287],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"data", "=", 
  RowBox[{"RandomReal", "[", 
   RowBox[{"10", ",", "20"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->239649738],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "8.3672843182738`", ",", "8.22770216605443`", ",", "0.0937371209993243`", 
   ",", "3.9027191438508946`", ",", "2.415605291140317`", ",", 
   "9.075268149332782`", ",", "1.1721104225632994`", ",", 
   "5.427277882050068`", ",", "3.4119127506373603`", ",", 
   "8.689972357570191`", ",", "1.7561187588212035`", ",", 
   "8.449764272826945`", ",", "8.283508604349914`", ",", "7.200100422276137`",
    ",", "1.4434168214521925`", ",", "6.992654579930086`", ",", 
   "6.013818941927509`", ",", "0.746235934361541`", ",", "3.002918366308962`",
    ",", "3.8259047358057896`"}], "}"}]], "Output",
 ImageSize->{429, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->270057498]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Apply", "[", 
   RowBox[{"Subtract", ",", 
    RowBox[{
     RowBox[{"Quartiles", "[", "data", "]"}], "[", 
     RowBox[{"[", 
      RowBox[{"{", 
       RowBox[{"3", ",", "1"}], "}"}], "]"}], "]"}]}], "]"}], "/", 
  "2"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->501255430],

Cell[BoxData["3.0848716801107057`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->78731539]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"QuartileDeviation", "[", "data", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->333901997],

Cell[BoxData["3.0848716801107057`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->1041664135]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->174408474],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["QuartileSkewness",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/QuartileSkewness"]], "InlineFormula"],
 " is a skewness measure obtained from the quartiles:"
}], "ExampleText",
 CellID->75587225],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"data", "=", 
  RowBox[{"RandomReal", "[", 
   RowBox[{"10", ",", "20"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->371742929],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.6146537444701927`", ",", "7.789172877823378`", ",", "5.807718229114778`",
    ",", "8.517261695460917`", ",", "4.275281807981495`", ",", 
   "3.4568418425849345`", ",", "8.059244829550295`", ",", 
   "1.7685210765065684`", ",", "4.185119665377148`", ",", 
   "0.6490503502551781`", ",", "7.900362031035229`", ",", 
   "4.665629861574985`", ",", "5.323519572507298`", ",", "5.717125917738015`",
    ",", "2.224667419842725`", ",", "1.4202046624782083`", ",", 
   "7.661258999639172`", ",", "1.2533131510987783`", ",", 
   "4.222035820365695`", ",", "7.336097758170254`"}], "}"}]], "Output",
 ImageSize->{429, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->807612021]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"q1", ",", "q2", ",", "q3"}], "}"}], "=", 
  RowBox[{"Quartiles", "[", "data", "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->250626437],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1.9965942481746468`", ",", "4.47045583477824`", ",", 
   "7.498678378904713`"}], "}"}]], "Output",
 ImageSize->{188, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->405288132]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"(", 
   RowBox[{"q1", "-", 
    RowBox[{"2", "q2"}], "+", "q3"}], ")"}], "/", 
  RowBox[{"(", 
   RowBox[{"q3", "-", "q1"}], ")"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->156230570],

Cell[BoxData["0.10075472209279408`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->804052371]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"QuartileSkewness", "[", "data", "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->46759148],

Cell[BoxData["0.10075472209279412`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->678686761]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->7483],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Quartiles",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Quartiles"]], "InlineFormula"],
 " requires numeric values:"
}], "ExampleText",
 CellID->9090],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Quartiles", "[", 
  RowBox[{"{", 
   RowBox[{"a", ",", "b", ",", "c"}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->9034],

Cell[BoxData[
 RowBox[{
  RowBox[{"Quartiles", "::", "\<\"rectn\"\>"}], ":", 
  " ", "\<\"Rectangular array of real numbers is expected at position \\!\\(1\
\\) in \\!\\(Quartiles[\\(\\({a, b, c}\\)\\)]\\).\"\>"}]], "Message", "MSG",
 CellID->48208341],

Cell[BoxData[
 RowBox[{"Quartiles", "[", 
  RowBox[{"{", 
   RowBox[{"a", ",", "b", ",", "c"}], "}"}], "]"}]], "Output",
 ImageSize->{139, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->467596327]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->26126],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["Quantile",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Quantile"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["InterquartileRange",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/InterquartileRange"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["QuartileDeviation",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/QuartileDeviation"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["QuartileSkewness",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/QuartileSkewness"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->18497]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->3256],

Cell[TextData[ButtonBox["Descriptive Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/DescriptiveStatistics"]], "Tutorials",
 CellID->4063544]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Descriptive Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DescriptiveStatistics"]], "MoreAbout",
 CellID->567145432],

Cell[TextData[ButtonBox["New in 6.0: Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60Statistics"]], "MoreAbout",
 CellID->99021359]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Quartiles - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 13, 31.3434494}", "context" -> "System`", 
    "keywords" -> {
     "hinges", "lower quartile", "median", "percentiles", "upper quartile"}, 
    "index" -> True, "label" -> "Built-in Mathematica Symbol", "language" -> 
    "en", "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "Quartiles[list] gives a list of the 1/4, 1/2, and 3/4 quantiles of the \
elements in list.", "synonyms" -> {}, "title" -> "Quartiles", "type" -> 
    "Symbol", "uri" -> "ref/Quartiles"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[8420, 274, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->109541307]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 34729, 1274}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2308, 55, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2931, 82, 1574, 37, 70, "ObjectNameGrid"],
Cell[4508, 121, 594, 19, 70, "Usage",
 CellID->11812]
}, Open  ]],
Cell[CellGroupData[{
Cell[5139, 145, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[5617, 162, 835, 30, 70, "Notes",
 CellID->14560],
Cell[6455, 194, 312, 11, 70, "Notes",
 CellID->724085966],
Cell[6770, 207, 470, 17, 70, "Notes",
 CellID->365953173],
Cell[7243, 226, 665, 23, 70, "Notes",
 CellID->42875104],
Cell[7911, 251, 146, 4, 70, "Notes",
 CellID->215460642],
Cell[8060, 257, 323, 12, 70, "Notes",
 CellID->11686772]
}, Closed]],
Cell[CellGroupData[{
Cell[8420, 274, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->109541307],
Cell[CellGroupData[{
Cell[8805, 288, 147, 5, 70, "ExampleSection",
 CellID->47281233],
Cell[8955, 295, 80, 1, 70, "ExampleText",
 CellID->31273586],
Cell[CellGroupData[{
Cell[9060, 300, 191, 6, 28, "Input",
 CellID->31907],
Cell[9254, 308, 283, 10, 51, "Output",
 CellID->632913276]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[9586, 324, 216, 7, 70, "ExampleSection",
 CellID->16338],
Cell[9805, 333, 215, 7, 70, "ExampleText",
 CellID->13924],
Cell[CellGroupData[{
Cell[10045, 344, 261, 8, 70, "Input",
 CellID->19854],
Cell[10309, 354, 538, 20, 51, "Output",
 CellID->107049843]
}, Open  ]],
Cell[10862, 377, 124, 3, 70, "ExampleDelimiter",
 CellID->64612447],
Cell[10989, 382, 76, 1, 70, "ExampleText",
 CellID->281606070],
Cell[CellGroupData[{
Cell[11090, 387, 437, 15, 70, "Input",
 CellID->421309609],
Cell[11530, 404, 444, 16, 51, "Output",
 CellID->679059439]
}, Open  ]],
Cell[11989, 423, 125, 3, 70, "ExampleDelimiter",
 CellID->158391064],
Cell[12117, 428, 76, 1, 70, "ExampleText",
 CellID->160553898],
Cell[CellGroupData[{
Cell[12218, 433, 256, 8, 70, "Input",
 CellID->190699768],
Cell[12477, 443, 248, 8, 54, "Output",
 CellID->817440611]
}, Open  ]],
Cell[12740, 454, 125, 3, 70, "ExampleDelimiter",
 CellID->650909219],
Cell[12868, 459, 88, 1, 70, "ExampleText",
 CellID->285474028],
Cell[CellGroupData[{
Cell[12981, 464, 192, 6, 70, "Input",
 CellID->315224713],
Cell[13176, 472, 276, 9, 36, "Output",
 CellID->237324472]
}, Open  ]],
Cell[CellGroupData[{
Cell[13489, 486, 245, 8, 70, "Input",
 CellID->594479966],
Cell[13737, 496, 857, 27, 71, "Output",
 CellID->29249989]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[14643, 529, 239, 7, 70, "ExampleSection",
 CellID->22424],
Cell[14885, 538, 211, 8, 70, "ExampleText",
 CellID->625271644],
Cell[CellGroupData[{
Cell[15121, 550, 533, 18, 70, "Input",
 CellID->19878761],
Cell[15657, 570, 486, 17, 36, "Output",
 CellID->163631235]
}, Open  ]],
Cell[CellGroupData[{
Cell[16180, 592, 106, 3, 70, "Input",
 CellID->235263230],
Cell[16289, 597, 1644, 57, 111, "Output",
 CellID->164473528]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[17982, 660, 233, 7, 70, "ExampleSection",
 CellID->26059],
Cell[18218, 669, 452, 16, 70, "ExampleText",
 CellID->38722882],
Cell[CellGroupData[{
Cell[18695, 689, 158, 5, 70, "Input",
 CellID->133352675],
Cell[18856, 696, 765, 16, 71, "Output",
 CellID->73587842]
}, Open  ]],
Cell[CellGroupData[{
Cell[19658, 717, 108, 3, 70, "Input",
 CellID->632052870],
Cell[19769, 722, 270, 9, 36, "Output",
 CellID->6564546]
}, Open  ]],
Cell[CellGroupData[{
Cell[20076, 736, 460, 16, 70, "Input",
 CellID->155558425],
Cell[20539, 754, 272, 9, 36, "Output",
 CellID->359744003]
}, Open  ]],
Cell[20826, 766, 125, 3, 70, "ExampleDelimiter",
 CellID->181243417],
Cell[20954, 771, 320, 12, 70, "ExampleText",
 CellID->381938433],
Cell[CellGroupData[{
Cell[21299, 787, 157, 5, 70, "Input",
 CellID->27788033],
Cell[21459, 794, 763, 16, 71, "Output",
 CellID->861386740]
}, Open  ]],
Cell[CellGroupData[{
Cell[22259, 815, 158, 5, 70, "Input",
 CellID->91770469],
Cell[22420, 822, 175, 5, 36, "Output",
 CellID->540519151]
}, Open  ]],
Cell[CellGroupData[{
Cell[22632, 832, 105, 3, 70, "Input",
 CellID->239513108],
Cell[22740, 837, 175, 5, 36, "Output",
 CellID->406471397]
}, Open  ]],
Cell[22930, 845, 125, 3, 70, "ExampleDelimiter",
 CellID->103804862],
Cell[23058, 850, 253, 7, 70, "ExampleText",
 CellID->15299726],
Cell[CellGroupData[{
Cell[23336, 861, 158, 5, 70, "Input",
 CellID->118537781],
Cell[23497, 868, 764, 16, 71, "Output",
 CellID->447399351]
}, Open  ]],
Cell[CellGroupData[{
Cell[24298, 889, 278, 9, 70, "Input",
 CellID->23051440],
Cell[24579, 900, 173, 5, 36, "Output",
 CellID->8718643]
}, Open  ]],
Cell[CellGroupData[{
Cell[24789, 910, 117, 3, 70, "Input",
 CellID->481300380],
Cell[24909, 915, 175, 5, 36, "Output",
 CellID->568061497]
}, Open  ]],
Cell[25099, 923, 125, 3, 70, "ExampleDelimiter",
 CellID->631682428],
Cell[25227, 928, 257, 7, 70, "ExampleText",
 CellID->533304287],
Cell[CellGroupData[{
Cell[25509, 939, 158, 5, 70, "Input",
 CellID->239649738],
Cell[25670, 946, 763, 16, 71, "Output",
 CellID->270057498]
}, Open  ]],
Cell[CellGroupData[{
Cell[26470, 967, 311, 11, 70, "Input",
 CellID->501255430],
Cell[26784, 980, 175, 5, 36, "Output",
 CellID->78731539]
}, Open  ]],
Cell[CellGroupData[{
Cell[26996, 990, 116, 3, 70, "Input",
 CellID->333901997],
Cell[27115, 995, 177, 5, 36, "Output",
 CellID->1041664135]
}, Open  ]],
Cell[27307, 1003, 125, 3, 70, "ExampleDelimiter",
 CellID->174408474],
Cell[27435, 1008, 243, 7, 70, "ExampleText",
 CellID->75587225],
Cell[CellGroupData[{
Cell[27703, 1019, 158, 5, 70, "Input",
 CellID->371742929],
Cell[27864, 1026, 765, 16, 71, "Output",
 CellID->807612021]
}, Open  ]],
Cell[CellGroupData[{
Cell[28666, 1047, 191, 6, 70, "Input",
 CellID->250626437],
Cell[28860, 1055, 272, 9, 36, "Output",
 CellID->405288132]
}, Open  ]],
Cell[CellGroupData[{
Cell[29169, 1069, 221, 8, 70, "Input",
 CellID->156230570],
Cell[29393, 1079, 177, 5, 36, "Output",
 CellID->804052371]
}, Open  ]],
Cell[CellGroupData[{
Cell[29607, 1089, 114, 3, 70, "Input",
 CellID->46759148],
Cell[29724, 1094, 177, 5, 36, "Output",
 CellID->678686761]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[29950, 1105, 225, 7, 70, "ExampleSection",
 CellID->7483],
Cell[30178, 1114, 199, 7, 70, "ExampleText",
 CellID->9090],
Cell[CellGroupData[{
Cell[30402, 1125, 157, 5, 70, "Input",
 CellID->9034],
Cell[30562, 1132, 252, 5, 70, "Message",
 CellID->48208341],
Cell[30817, 1139, 251, 8, 36, "Output",
 CellID->467596327]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[31129, 1154, 311, 9, 70, "SeeAlsoSection",
 CellID->26126],
Cell[31443, 1165, 995, 35, 70, "SeeAlso",
 CellID->18497]
}, Open  ]],
Cell[CellGroupData[{
Cell[32475, 1205, 313, 9, 70, "TutorialsSection",
 CellID->3256],
Cell[32791, 1216, 156, 3, 70, "Tutorials",
 CellID->4063544]
}, Open  ]],
Cell[CellGroupData[{
Cell[32984, 1224, 299, 8, 70, "MoreAboutSection"],
Cell[33286, 1234, 155, 3, 70, "MoreAbout",
 CellID->567145432],
Cell[33444, 1239, 150, 3, 70, "MoreAbout",
 CellID->99021359]
}, Open  ]],
Cell[33609, 1245, 27, 0, 70, "History"],
Cell[33639, 1247, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

