(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     76280,       1925]
NotebookOptionsPosition[     68895,       1671]
NotebookOutlinePosition[     70229,       1709]
CellTagsIndexPosition[     70143,       1704]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"RamanujanTauZ" :> 
          Documentation`HelpLookup["paclet:ref/RamanujanTauZ"], 
          "RamanujanTauTheta" :> 
          Documentation`HelpLookup["paclet:ref/RamanujanTauTheta"], 
          "RamanujanTau" :> 
          Documentation`HelpLookup["paclet:ref/RamanujanTau"], "Zeta" :> 
          Documentation`HelpLookup["paclet:ref/Zeta"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"RamanujanTauZ\"\>", 
       2->"\<\"RamanujanTauTheta\"\>", 3->"\<\"RamanujanTau\"\>", 
       4->"\<\"Zeta\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Number Theoretic Functions" :> 
          Documentation`HelpLookup["paclet:guide/NumberTheoreticFunctions"], 
          "New in 6.0: Number Theory & Integer Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60NumberTheoryAndIntegerFunctions"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Number Theoretic Functions\"\>", 
       2->"\<\"New in 6.0: Number Theory & Integer Functions\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["RamanujanTauL", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["RamanujanTauL",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/RamanujanTauL"], "[", 
       StyleBox["s", "TI"], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the Ramanujan tau Dirichlet L-function ",
     Cell[BoxData[
      FormBox[
       RowBox[{"L", "(", "s", ")"}], TraditionalForm]], "InlineMath"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["\<\
Mathematical function, suitable for both symbolic and numerical manipulation.\
\
\>", "Notes",
 CellID->20640],

Cell[TextData[{
 "For ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     ButtonBox["Re",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Re"], 
     RowBox[{"(", "s", ")"}]}], ">", "6"}], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{"L", 
    RowBox[{"(", "s", ")"}]}], TraditionalForm]], "InlineMath"],
 " is given by the Dirichlet series ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    UnderoverscriptBox["\[Sum]", 
     RowBox[{"n", "=", "1"}], "\[Infinity]"], 
    FractionBox[
     RowBox[{"\[Tau]", "(", "n", ")"}], 
     SuperscriptBox["n", "s"]]}], TraditionalForm]], "InlineMath"],
 "."
}], "Notes",
 CellID->547407586],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RamanujanTauL",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTauL"]], "InlineFormula"],
 " can be evaluated to arbitrary numerical precision."
}], "Notes",
 CellID->31226],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RamanujanTauL",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTauL"]], "InlineFormula"],
 " automatically threads over lists."
}], "Notes",
 CellID->28162]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->266811228],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->159905332],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RamanujanTauL", "[", 
  RowBox[{"6", "+", 
   RowBox[{"9.22", "I"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->25001],

Cell[BoxData[
 RowBox[{"0.0004030903798357173`", "\[InvisibleSpace]", "-", 
  RowBox[{"0.002390127652106469`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{168, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->475104261]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->248295503],

Cell["Plot on the critical line:", "ExampleText",
 CellID->22932358],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Abs", "[", 
    RowBox[{"RamanujanTauL", "[", 
     RowBox[{"6", "+", 
      RowBox[{"I", " ", "x"}]}], "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "20"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->7191511],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{187, 120},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->528376808]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->247168195],

Cell["Evaluate for complex arguments:", "ExampleText",
 CellID->218580873],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RamanujanTauL", "[", 
  RowBox[{"1.2", "+", "I"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1491696141],

Cell[BoxData[
 RowBox[{"0.011724537411606846`", "\[InvisibleSpace]", "+", 
  RowBox[{"0.08607097622907532`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{147, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->36511352]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->466082509],

Cell["Evaluate to high precision:", "ExampleText",
 CellID->1575868154],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"RamanujanTauL", "[", 
    RowBox[{"3", "/", "2"}], "]"}], ",", "50"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->700245827],

Cell[BoxData["0.\
0825600102475561761214388628875512982282428343063272515622584242586`50."], \
"Output",
 ImageSize->{375, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->17265666]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->212522826],

Cell["\<\
The precision of the output tracks the precision of the input:\
\>", "ExampleText",
 CellID->605831621],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RamanujanTauL", "[", "1.500000000000000000000000000000000", 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->992083720],

Cell[BoxData["0.082560010247556176121438862887551298225756663639`32.\
184894189666295"], "Output",
 ImageSize->{249, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->233206786]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->2002302686],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RamanujanTauL",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTauL"]], "InlineFormula"],
 " threads over lists:"
}], "ExampleText",
 CellID->1632495112],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RamanujanTauL", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"-", "0.1"}], ",", 
    RowBox[{"3.1", "+", 
     RowBox[{"2.7", " ", "I"}]}], ",", "1.0"}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->181627411],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "0.0008681277588742591`"}], ",", 
   RowBox[{"0.3334622848854827`", "\[InvisibleSpace]", "+", 
    RowBox[{"0.6918231138749945`", " ", "\[ImaginaryI]"}]}], ",", 
   "0.03744128126851549`"}], "}"}]], "Output",
 ImageSize->{324, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->594390]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->174424782],

Cell["Plot on the critical line:", "ExampleText",
 CellID->10621695],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Abs", "[", 
    RowBox[{"RamanujanTauL", "[", 
     RowBox[{"6", "+", 
      RowBox[{"I", " ", "x"}]}], "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "20"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->390078074],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 116},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->20675269]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->955861864],

Cell[TextData[{
 "Find a zero of ",
 Cell[BoxData[
  ButtonBox["RamanujanTauL",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTauL"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->1992668688],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindRoot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"RamanujanTauL", "[", "t", "]"}], "==", "0"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", 
     RowBox[{"6", "+", 
      RowBox[{"9", "I"}]}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->961223457],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"t", "\[Rule]", 
   RowBox[{"6.0000000000000115`", "\[InvisibleSpace]", "+", 
    RowBox[{"9.222379399921131`", " ", "\[ImaginaryI]"}]}]}], "}"}]], "Output",\

 ImageSize->{128, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->3960099]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1388085803],

Cell[TextData[{
 "The number of zeros on the critical strip from ",
 Cell[BoxData["0"], "InlineFormula"],
 " to ",
 Cell[BoxData[
  FormBox["t", TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->1443233660],

Cell[BoxData[
 RowBox[{
  RowBox[{"n", "[", "t_", "]"}], ":=", " ", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"RamanujanTauTheta", "[", "t", "]"}], "+", 
     RowBox[{"Im", "[", 
      RowBox[{"Log", "[", 
       RowBox[{"RamanujanTauL", "[", 
        RowBox[{"6", "+", 
         RowBox[{"I", " ", "t"}]}], "]"}], "]"}], "]"}]}], ")"}], "/", 
   "Pi"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->600330755],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"Chop", "[", 
    RowBox[{"n", "[", "t", "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0.", ",", "20"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->2084621857],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", 
   ",", "0", ",", "0", ",", "1.0000000000000002`", ",", "1.0000000000000007`",
    ",", "1.0000000000000009`", ",", "1.0000000000000013`", ",", 
   "2.0000000000000027`", ",", "2.000000000000002`", ",", 
   "1.9999999999999996`", ",", "2.0000000000000004`", ",", "3.`", ",", 
   "3.0000000000000044`", ",", "4.000000000000004`"}], "}"}]], "Output",
 ImageSize->{466, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1885843744]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->179178048],

Cell["Plot of the real part:", "ExampleText",
 CellID->673379642],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"Re", "[", 
    RowBox[{"RamanujanTauL", "[", 
     RowBox[{"x", "+", 
      RowBox[{"I", " ", "y"}]}], "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "5"}], ",", " ", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "2"}], ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1622422540],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{264, 185},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->25222214]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->797259246],

Cell[TextData[{
 "Ramanujan ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[Tau]", "z"], TraditionalForm]], "InlineMath"],
 " function:"
}], "ExampleText",
 CellID->1385673889],

Cell[BoxData[
 RowBox[{
  RowBox[{
   SubscriptBox["\[Tau]", "z"], "[", "t_", "]"}], ":=", " ", 
  RowBox[{
   RowBox[{"Gamma", "[", 
    RowBox[{"6", "+", 
     RowBox[{"I", " ", "t"}]}], "]"}], 
   RowBox[{
    RowBox[{
     RowBox[{"(", 
      RowBox[{"2", " ", "Pi"}], ")"}], "^", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "I"}], " ", "t"}], ")"}]}], "/", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"RamanujanTauL", "[", 
       RowBox[{"6", "+", 
        RowBox[{"I", " ", "t"}]}], "]"}], " ", 
      RowBox[{"Sqrt", "[", 
       RowBox[{
        RowBox[{"Sinh", "[", 
         RowBox[{"Pi", " ", "t"}], "]"}], "/", 
        RowBox[{"(", 
         RowBox[{"Pi", " ", "t", " ", 
          RowBox[{"Product", "[", 
           RowBox[{
            RowBox[{
             RowBox[{"k", "^", "2"}], "+", 
             RowBox[{"t", "^", "2"}]}], ",", 
            RowBox[{"{", 
             RowBox[{"k", ",", "1", ",", "5"}], "}"}]}], "]"}]}], ")"}]}], 
       "]"}]}], ")"}]}]}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1708384355],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Abs", "[", 
    RowBox[{
     SubscriptBox["\[Tau]", "z"], "[", "t", "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", 
     RowBox[{"-", "5"}], ",", " ", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->959494095],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 124},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->7645892]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->145403997],

Cell["Functional equation:", "ExampleText",
 CellID->1437919821],

Cell[BoxData[
 RowBox[{
  RowBox[{"a", "[", "s_", "]"}], ":=", 
  RowBox[{
   RowBox[{"RamanujanTauL", "[", "s", "]"}], 
   RowBox[{
    RowBox[{"Gamma", "[", "s", "]"}], "/", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"2", "Pi"}], ")"}], "^", "s"}]}]}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1484951856],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"With", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"s", "=", "2.5"}], "}"}], ",", 
   RowBox[{
    RowBox[{"a", "[", "s", "]"}], " ", "\[Equal]", " ", 
    RowBox[{"a", "[", 
     RowBox[{"12", "-", "s"}], "]"}]}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1678382005],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->532691433]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->298497921],

Cell[TextData[{
 "Approximation of ",
 Cell[BoxData[
  ButtonBox["RamanujanTauL",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTauL"]], "InlineFormula"],
 " using Euler product formula:"
}], "ExampleText",
 CellID->332758855],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"With", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"s", "=", "7.5"}], ",", 
     RowBox[{"n", "=", "500"}]}], "}"}], ",", 
   RowBox[{"Product", "[", 
    RowBox[{
     RowBox[{"1", "/", 
      RowBox[{"(", 
       RowBox[{"1", "-", 
        RowBox[{
         RowBox[{"RamanujanTau", "[", "p", "]"}], 
         RowBox[{"p", "^", 
          RowBox[{"(", 
           RowBox[{"-", "s"}], ")"}]}]}], "+", 
        RowBox[{"p", "^", 
         RowBox[{"(", 
          RowBox[{"11", "-", 
           RowBox[{"2", "s"}]}], ")"}]}]}], ")"}]}], ",", 
     RowBox[{"{", 
      RowBox[{"p", ",", 
       RowBox[{"Prime", "[", 
        RowBox[{"Range", "[", "n", "]"}], "]"}]}], "}"}]}], "]"}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->630659197],

Cell[BoxData["0.9073756132071458`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->54518388]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RamanujanTauL", "[", "7.5", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1343136856],

Cell[BoxData["0.9073756962700319`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->336353569]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1542421952],

Cell[TextData[{
 "On the critical line, ",
 Cell[BoxData[
  ButtonBox["RamanujanTauL",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTauL"]], "InlineFormula"],
 " splits:"
}], "ExampleText",
 CellID->1210644327],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"With", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"s", " ", "=", " ", "9.22`20"}], "}"}], ",", " ", 
   RowBox[{
    RowBox[{"RamanujanTauL", "[", 
     RowBox[{"6", "+", 
      RowBox[{"I", " ", "s"}]}], "]"}], " ", "\[Equal]", " ", 
    RowBox[{
     RowBox[{"RamanujanTauZ", "[", "s", "]"}], 
     RowBox[{"Exp", "[", 
      RowBox[{
       RowBox[{"-", "I"}], " ", 
       RowBox[{"RamanujanTauTheta", "[", "s", "]"}]}], "]"}]}]}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2039132211],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->868581952]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1687896406],

Cell["Evaluate derivatives numerically:", "ExampleText",
 CellID->215918772],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"RamanujanTauL", "'"}], "[", 
  RowBox[{"6.", "+", "I"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->612067025],

Cell[BoxData[
 RowBox[{"0.10087234580527563`", "\[InvisibleSpace]", "-", 
  RowBox[{"0.04222779962507472`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{147, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->311203571]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"RamanujanTauL", "''"}], "[", 
  RowBox[{"6.", "+", "I"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->2019024088],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "0.04485020481650111`"}], "+", 
  RowBox[{"0.007808288691001777`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{169, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->14462166]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["RamanujanTauZ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RamanujanTauZ"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["RamanujanTauTheta",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RamanujanTauTheta"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["RamanujanTau",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RamanujanTau"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Zeta",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Zeta"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->43325779],

Cell[TextData[ButtonBox["Number Theoretic Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NumberTheoreticFunctions"]], "MoreAbout",
 CellID->73042907],

Cell[TextData[ButtonBox["New in 6.0: Number Theory & Integer Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/NewIn60NumberTheoryAndIntegerFunctions"]], "MoreAbout",
 CellID->87523766]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"RamanujanTauL - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 13, 41.0780736}", "context" -> "System`", 
    "keywords" -> {
     "Ramanujan L function", "Ramanujan tau", "Ramanujan tau L-function", 
      "Ramanujan tau L-series"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "RamanujanTauL[s] gives the Ramanujan tau Dirichlet L-function L(s).", 
    "synonyms" -> {"ramanujan tau L"}, "title" -> "RamanujanTauL", "type" -> 
    "Symbol", "uri" -> "ref/RamanujanTauL"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[6381, 203, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->266811228]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 70000, 1697}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 1767, 41, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2390, 68, 1578, 37, 70, "ObjectNameGrid"],
Cell[3971, 107, 617, 19, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[4625, 131, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[5103, 148, 120, 4, 70, "Notes",
 CellID->20640],
Cell[5226, 154, 673, 26, 70, "Notes",
 CellID->547407586],
Cell[5902, 182, 228, 7, 70, "Notes",
 CellID->31226],
Cell[6133, 191, 211, 7, 70, "Notes",
 CellID->28162]
}, Closed]],
Cell[CellGroupData[{
Cell[6381, 203, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->266811228],
Cell[CellGroupData[{
Cell[6766, 217, 148, 5, 70, "ExampleSection",
 CellID->159905332],
Cell[CellGroupData[{
Cell[6939, 226, 150, 5, 28, "Input",
 CellID->25001],
Cell[7092, 233, 278, 7, 36, "Output",
 CellID->475104261]
}, Open  ]],
Cell[7385, 243, 125, 3, 70, "ExampleDelimiter",
 CellID->248295503],
Cell[7513, 248, 68, 1, 70, "ExampleText",
 CellID->22932358],
Cell[CellGroupData[{
Cell[7606, 253, 307, 10, 28, "Input",
 CellID->7191511],
Cell[7916, 265, 4365, 76, 141, "Output",
 Evaluatable->False,
 CellID->528376808]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[12330, 347, 220, 7, 70, "ExampleSection",
 CellID->247168195],
Cell[12553, 356, 74, 1, 70, "ExampleText",
 CellID->218580873],
Cell[CellGroupData[{
Cell[12652, 361, 135, 4, 70, "Input",
 CellID->1491696141],
Cell[12790, 367, 275, 7, 36, "Output",
 CellID->36511352]
}, Open  ]],
Cell[13080, 377, 125, 3, 70, "ExampleDelimiter",
 CellID->466082509],
Cell[13208, 382, 71, 1, 70, "ExampleText",
 CellID->1575868154],
Cell[CellGroupData[{
Cell[13304, 387, 187, 6, 70, "Input",
 CellID->700245827],
Cell[13494, 395, 234, 7, 36, "Output",
 CellID->17265666]
}, Open  ]],
Cell[13743, 405, 125, 3, 70, "ExampleDelimiter",
 CellID->212522826],
Cell[13871, 410, 113, 3, 70, "ExampleText",
 CellID->605831621],
Cell[CellGroupData[{
Cell[14009, 417, 146, 4, 70, "Input",
 CellID->992083720],
Cell[14158, 423, 229, 6, 36, "Output",
 CellID->233206786]
}, Open  ]],
Cell[14402, 432, 126, 3, 70, "ExampleDelimiter",
 CellID->2002302686],
Cell[14531, 437, 208, 7, 70, "ExampleText",
 CellID->1632495112],
Cell[CellGroupData[{
Cell[14764, 448, 245, 8, 70, "Input",
 CellID->181627411],
Cell[15012, 458, 392, 11, 36, "Output",
 CellID->594390]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[15453, 475, 227, 7, 70, "ExampleSection",
 CellID->174424782],
Cell[15683, 484, 68, 1, 70, "ExampleText",
 CellID->10621695],
Cell[CellGroupData[{
Cell[15776, 489, 309, 10, 70, "Input",
 CellID->390078074],
Cell[16088, 501, 4193, 73, 70, "Output",
 Evaluatable->False,
 CellID->20675269]
}, Open  ]],
Cell[20296, 577, 125, 3, 70, "ExampleDelimiter",
 CellID->955861864],
Cell[20424, 582, 209, 8, 70, "ExampleText",
 CellID->1992668688],
Cell[CellGroupData[{
Cell[20658, 594, 291, 10, 70, "Input",
 CellID->961223457],
Cell[20952, 606, 327, 10, 36, "Output",
 CellID->3960099]
}, Open  ]],
Cell[21294, 619, 126, 3, 70, "ExampleDelimiter",
 CellID->1388085803],
Cell[21423, 624, 223, 8, 70, "ExampleText",
 CellID->1443233660],
Cell[21649, 634, 421, 14, 70, "Input",
 CellID->600330755],
Cell[CellGroupData[{
Cell[22095, 652, 248, 8, 70, "Input",
 CellID->2084621857],
Cell[22346, 662, 598, 13, 36, "Output",
 CellID->1885843744]
}, Open  ]],
Cell[22959, 678, 125, 3, 70, "ExampleDelimiter",
 CellID->179178048],
Cell[23087, 683, 65, 1, 70, "ExampleText",
 CellID->673379642],
Cell[CellGroupData[{
Cell[23177, 688, 426, 14, 70, "Input",
 CellID->1622422540],
Cell[23606, 704, 33566, 554, 70, "Output",
 Evaluatable->False,
 CellID->25222214]
}, Open  ]],
Cell[57187, 1261, 125, 3, 70, "ExampleDelimiter",
 CellID->797259246],
Cell[57315, 1266, 176, 7, 70, "ExampleText",
 CellID->1385673889],
Cell[57494, 1275, 1050, 35, 70, "Input",
 CellID->1708384355],
Cell[CellGroupData[{
Cell[58569, 1314, 299, 10, 70, "Input",
 CellID->959494095],
Cell[58871, 1326, 2797, 50, 70, "Output",
 Evaluatable->False,
 CellID->7645892]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[61717, 1382, 237, 7, 70, "ExampleSection",
 CellID->145403997],
Cell[61957, 1391, 64, 1, 70, "ExampleText",
 CellID->1437919821],
Cell[62024, 1394, 314, 11, 70, "Input",
 CellID->1484951856],
Cell[CellGroupData[{
Cell[62363, 1409, 303, 10, 70, "Input",
 CellID->1678382005],
Cell[62669, 1421, 161, 5, 36, "Output",
 CellID->532691433]
}, Open  ]],
Cell[62845, 1429, 125, 3, 70, "ExampleDelimiter",
 CellID->298497921],
Cell[62973, 1434, 238, 8, 70, "ExampleText",
 CellID->332758855],
Cell[CellGroupData[{
Cell[63236, 1446, 786, 27, 70, "Input",
 CellID->630659197],
Cell[64025, 1475, 175, 5, 36, "Output",
 CellID->54518388]
}, Open  ]],
Cell[CellGroupData[{
Cell[64237, 1485, 112, 3, 70, "Input",
 CellID->1343136856],
Cell[64352, 1490, 176, 5, 36, "Output",
 CellID->336353569]
}, Open  ]],
Cell[64543, 1498, 126, 3, 70, "ExampleDelimiter",
 CellID->1542421952],
Cell[64672, 1503, 223, 8, 70, "ExampleText",
 CellID->1210644327],
Cell[CellGroupData[{
Cell[64920, 1515, 528, 17, 70, "Input",
 CellID->2039132211],
Cell[65451, 1534, 161, 5, 36, "Output",
 CellID->868581952]
}, Open  ]],
Cell[65627, 1542, 126, 3, 70, "ExampleDelimiter",
 CellID->1687896406],
Cell[65756, 1547, 76, 1, 70, "ExampleText",
 CellID->215918772],
Cell[CellGroupData[{
Cell[65857, 1552, 151, 5, 70, "Input",
 CellID->612067025],
Cell[66011, 1559, 275, 7, 36, "Output",
 CellID->311203571]
}, Open  ]],
Cell[CellGroupData[{
Cell[66323, 1571, 153, 5, 70, "Input",
 CellID->2019024088],
Cell[66479, 1578, 272, 8, 36, "Output",
 CellID->14462166]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[66812, 1593, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[67131, 1604, 973, 35, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[68141, 1644, 318, 9, 70, "MoreAboutSection",
 CellID->43325779],
Cell[68462, 1655, 161, 3, 70, "MoreAbout",
 CellID->73042907],
Cell[68626, 1660, 197, 4, 70, "MoreAbout",
 CellID->87523766]
}, Open  ]],
Cell[68838, 1667, 27, 0, 70, "History"],
Cell[68868, 1669, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

