(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     58462,       1530]
NotebookOptionsPosition[     51875,       1302]
NotebookOutlinePosition[     53136,       1338]
CellTagsIndexPosition[     53050,       1333]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Special Functions" :> 
          Documentation`HelpLookup["paclet:tutorial/SpecialFunctions"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Special Functions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"RamanujanTauTheta" :> 
          Documentation`HelpLookup["paclet:ref/RamanujanTauTheta"], 
          "RamanujanTauL" :> 
          Documentation`HelpLookup["paclet:ref/RamanujanTauL"], 
          "RiemannSiegelZ" :> 
          Documentation`HelpLookup["paclet:ref/RiemannSiegelZ"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"RamanujanTauTheta\"\>", 
       2->"\<\"RamanujanTauL\"\>", 
       3->"\<\"RiemannSiegelZ\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Number Theoretic Functions" :> 
          Documentation`HelpLookup["paclet:guide/NumberTheoreticFunctions"], 
          "New in 6.0: Number Theory & Integer Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60NumberTheoryAndIntegerFunctions"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Number Theoretic Functions\"\>", 
       2->"\<\"New in 6.0: Number Theory & Integer Functions\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["RamanujanTauZ", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["RamanujanTauZ",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/RamanujanTauZ"], "[", 
       StyleBox["t", "TI"], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the Ramanujan tau Z-function ",
     Cell[BoxData[
      FormBox[
       RowBox[{"Z", "(", "t", ")"}], TraditionalForm]], "InlineMath"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["\<\
Mathematical function, suitable for both symbolic and numerical manipulation.\
\
\>", "Notes",
 CellID->20640],

Cell[TextData[{
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"Z", "(", "t", ")"}], "=", 
    RowBox[{
     SuperscriptBox["e", 
      RowBox[{"i", " ", 
       RowBox[{"\[Theta]", "(", " ", "t", ")"}]}]], " ", 
     RowBox[{"L", "(", 
      RowBox[{
       RowBox[{"i", " ", "t"}], "+", "6"}], ")"}]}]}], TraditionalForm]], 
  "InlineMath"],
 ", where ",
 Cell[BoxData[
  FormBox["\[Theta]", TraditionalForm]], "InlineMath"],
 " is the Ramanujan tau theta function, and ",
 Cell[BoxData[
  FormBox["L", TraditionalForm]], "InlineMath"],
 " is the Ramanujan tau L-function."
}], "Notes",
 CellID->25772179],

Cell[TextData[{
 "For certain special arguments, ",
 Cell[BoxData[
  ButtonBox["RamanujanTauZ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTauZ"]], "InlineFormula"],
 " automatically evaluates to exact values."
}], "Notes",
 CellID->67140068],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RamanujanTauZ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTauZ"]], "InlineFormula"],
 " can be evaluated to arbitrary numerical precision."
}], "Notes",
 CellID->31226],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RamanujanTauZ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTauZ"]], "InlineFormula"],
 " automatically threads over lists."
}], "Notes",
 CellID->28162]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->439689604],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->19732612],

Cell["Evaluate numerically:", "ExampleText",
 CellID->967458888],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RamanujanTauZ", "[", "9.22", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1482839179],

Cell[BoxData["0.002423879544796747`"], "Output",
 ImageSize->{74, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->530358018]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1670073476],

Cell["Plot on the real line:", "ExampleText",
 CellID->1313644842],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"RamanujanTauZ", "[", "t", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", 
     RowBox[{"-", "25"}], ",", "25"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->247528612],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 114},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->678350067]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->106913306],

Cell["Evaluate for complex arguments:", "ExampleText",
 CellID->180085176],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RamanujanTauZ", "[", 
  RowBox[{"1.", "+", "I"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->887094409],

Cell[BoxData[
 RowBox[{"0.7921163051364698`", "\[InvisibleSpace]", "+", 
  RowBox[{"0.054683421748137555`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{147, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->282572924]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1564254589],

Cell["Evaluate to high precision:", "ExampleText",
 CellID->1538466473],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"RamanujanTauZ", "[", "1", "]"}], ",", "50"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1302901831],

Cell[BoxData["0.\
8194349386751963056125375916205154235730118236441275123726904380562`50."], \
"Output",
 ImageSize->{368, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->734042052]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->260118207],

Cell["\<\
The precision of the output tracks the precision of the input:\
\>", "ExampleText",
 CellID->1677483404],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RamanujanTauZ", "[", "1.5000000000000000000000", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1298253212],

Cell[BoxData["0.85352190207383081041470826006188956172`21.071750091208465"], \
"Output",
 ImageSize->{165, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->45365268]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1669533896],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RamanujanTauZ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTauZ"]], "InlineFormula"],
 " evaluates to exact values for certain special arguments:"
}], "ExampleText",
 CellID->834858370],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RamanujanTauZ", "[", 
  RowBox[{"5", " ", "I"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1398162741],

Cell[BoxData[
 FractionBox[
  RowBox[{"45", " ", 
   SqrtBox["7"], " ", 
   RowBox[{"RamanujanTauL", "[", "11", "]"}]}], 
  RowBox[{"2", " ", 
   SuperscriptBox["\[Pi]", "5"]}]]], "Output",
 ImageSize->{171, 37},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->62611036]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1168224184],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RamanujanTauZ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTauZ"]], "InlineFormula"],
 " threads over lists:"
}], "ExampleText",
 CellID->2048359614],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RamanujanTauZ", "[", 
  RowBox[{"{", 
   RowBox[{"5.4", ",", " ", "9.", ",", "0.5"}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->469687441],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1.4269594889455104`", ",", "0.21980070240438523`", ",", 
   "0.7989520417325605`"}], "}"}]], "Output",
 ImageSize->{202, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->32226234]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->433842623],

Cell[TextData[{
 "Plot of the absolute value of ",
 Cell[BoxData[
  ButtonBox["RamanujanTauZ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTauZ"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->968656494],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"Abs", "[", 
    RowBox[{"RamanujanTauZ", "[", 
     RowBox[{"x", "+", 
      RowBox[{"I", " ", "y"}]}], "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "9"}], ",", "9"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", " ", 
     RowBox[{"-", "1"}], ",", " ", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->136166751],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{248, 184},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->194271789]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->907062050],

Cell["Find a numerical root:", "ExampleText",
 CellID->5464297],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindRoot", "[", 
  RowBox[{
   RowBox[{"RamanujanTauZ", "[", "t", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1644126505],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"t", "\[Rule]", "9.222379399921103`"}], "}"}]], "Output",
 ImageSize->{90, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->750353114]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Chop", "[", 
  RowBox[{"RamanujanTauL", "[", 
   RowBox[{
    RowBox[{"6", "+", 
     RowBox[{"I", " ", "t"}]}], " ", "/.", "%"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->617381836],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->101331076]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->7357994],

Cell[TextData[{
 "On the critical line, ",
 Cell[BoxData[
  ButtonBox["RamanujanTauZ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTauZ"]], "InlineFormula"],
 " is the modulus of ",
 Cell[BoxData[
  ButtonBox["RamanujanTauL",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTauL"]], "InlineFormula"],
 " up to a sign:"
}], "ExampleText",
 CellID->654548207],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Abs", "[", 
  RowBox[{"RamanujanTauL", "[", 
   RowBox[{"6", "+", 
    RowBox[{"7.5", "I"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1580475115],

Cell[BoxData["1.2386246756080384`"], "Output",
 GeneratedCell->False,
 CellAutoOverwrite->False,
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->66834345]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RamanujanTauZ", "[", "7.5", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1809075235],

Cell[BoxData["1.23862467560804`"], "Output",
 GeneratedCell->False,
 CellAutoOverwrite->False,
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->378336743]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RamanujanTauZ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTauZ"]], "InlineFormula"],
 " can be expressed in terms of ",
 Cell[BoxData[
  ButtonBox["RamanujanTauL",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTauL"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["RamanujanTauTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTauTheta"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->1184776935],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"With", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"t", "=", "7.5`20"}], "}"}], ",", 
   RowBox[{
    RowBox[{"RamanujanTauZ", "[", "t", "]"}], " ", "==", 
    RowBox[{
     RowBox[{"Exp", "[", 
      RowBox[{"I", " ", 
       RowBox[{"RamanujanTauTheta", "[", "t", "]"}]}], "]"}], " ", 
     RowBox[{"RamanujanTauL", "[", 
      RowBox[{"6", "+", 
       RowBox[{"I", " ", "t"}]}], "]"}]}]}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->1503064476],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->290917793]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->895693104],

Cell["Evaluate derivatives numerically:", "ExampleText",
 CellID->53633458],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"RamanujanTauZ", "'"}], "[", "9.22", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1621880513],

Cell[BoxData[
 RowBox[{"-", "1.0183888028812527`"}]], "Output",
 ImageSize->{61, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->600949410]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"RamanujanTauZ", "'''"}], "[", "9.22", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->671139712],

Cell[BoxData["0.2769929607617225`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->9007446]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["RamanujanTauTheta",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RamanujanTauTheta"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["RamanujanTauL",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RamanujanTauL"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["RiemannSiegelZ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RiemannSiegelZ"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->250839057],

Cell[TextData[ButtonBox["Special Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/SpecialFunctions"]], "Tutorials",
 CellID->23022]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->45879745],

Cell[TextData[ButtonBox["Number Theoretic Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NumberTheoreticFunctions"]], "MoreAbout",
 CellID->95083468],

Cell[TextData[ButtonBox["New in 6.0: Number Theory & Integer Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/NewIn60NumberTheoryAndIntegerFunctions"]], "MoreAbout",
 CellID->456437903]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"RamanujanTauZ - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 13, 45.5938142}", "context" -> "System`", 
    "keywords" -> {"Ramanujan tau", "Ramanujan Tau Z function"}, "index" -> 
    True, "label" -> "Built-in Mathematica Symbol", "language" -> "en", 
    "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "RamanujanTauZ[t] gives the Ramanujan tau Z-function Z(t).", 
    "synonyms" -> {"ramanujan tau Z"}, "title" -> "RamanujanTauZ", "type" -> 
    "Symbol", "uri" -> "ref/RamanujanTauZ"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[7056, 222, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->439689604]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 52907, 1326}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2255, 55, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2878, 82, 1578, 37, 70, "ObjectNameGrid"],
Cell[4459, 121, 607, 19, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[5103, 145, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[5581, 162, 120, 4, 70, "Notes",
 CellID->20640],
Cell[5704, 168, 610, 21, 70, "Notes",
 CellID->25772179],
Cell[6317, 191, 257, 8, 70, "Notes",
 CellID->67140068],
Cell[6577, 201, 228, 7, 70, "Notes",
 CellID->31226],
Cell[6808, 210, 211, 7, 70, "Notes",
 CellID->28162]
}, Closed]],
Cell[CellGroupData[{
Cell[7056, 222, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->439689604],
Cell[CellGroupData[{
Cell[7441, 236, 147, 5, 70, "ExampleSection",
 CellID->19732612],
Cell[7591, 243, 64, 1, 70, "ExampleText",
 CellID->967458888],
Cell[CellGroupData[{
Cell[7680, 248, 113, 3, 28, "Input",
 CellID->1482839179],
Cell[7796, 253, 178, 5, 36, "Output",
 CellID->530358018]
}, Open  ]],
Cell[7989, 261, 126, 3, 70, "ExampleDelimiter",
 CellID->1670073476],
Cell[8118, 266, 66, 1, 70, "ExampleText",
 CellID->1313644842],
Cell[CellGroupData[{
Cell[8209, 271, 246, 8, 28, "Input",
 CellID->247528612],
Cell[8458, 281, 5215, 90, 135, "Output",
 Evaluatable->False,
 CellID->678350067]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[13722, 377, 220, 7, 70, "ExampleSection",
 CellID->106913306],
Cell[13945, 386, 74, 1, 70, "ExampleText",
 CellID->180085176],
Cell[CellGroupData[{
Cell[14044, 391, 133, 4, 70, "Input",
 CellID->887094409],
Cell[14180, 397, 275, 7, 36, "Output",
 CellID->282572924]
}, Open  ]],
Cell[14470, 407, 126, 3, 70, "ExampleDelimiter",
 CellID->1564254589],
Cell[14599, 412, 71, 1, 70, "ExampleText",
 CellID->1538466473],
Cell[CellGroupData[{
Cell[14695, 417, 163, 5, 70, "Input",
 CellID->1302901831],
Cell[14861, 424, 235, 7, 36, "Output",
 CellID->734042052]
}, Open  ]],
Cell[15111, 434, 125, 3, 70, "ExampleDelimiter",
 CellID->260118207],
Cell[15239, 439, 114, 3, 70, "ExampleText",
 CellID->1677483404],
Cell[CellGroupData[{
Cell[15378, 446, 133, 3, 70, "Input",
 CellID->1298253212],
Cell[15514, 451, 218, 6, 36, "Output",
 CellID->45365268]
}, Open  ]],
Cell[15747, 460, 126, 3, 70, "ExampleDelimiter",
 CellID->1669533896],
Cell[15876, 465, 244, 7, 70, "ExampleText",
 CellID->834858370],
Cell[CellGroupData[{
Cell[16145, 476, 133, 4, 70, "Input",
 CellID->1398162741],
Cell[16281, 482, 319, 11, 58, "Output",
 CellID->62611036]
}, Open  ]],
Cell[16615, 496, 126, 3, 70, "ExampleDelimiter",
 CellID->1168224184],
Cell[16744, 501, 208, 7, 70, "ExampleText",
 CellID->2048359614],
Cell[CellGroupData[{
Cell[16977, 512, 176, 5, 70, "Input",
 CellID->469687441],
Cell[17156, 519, 275, 9, 36, "Output",
 CellID->32226234]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[17480, 534, 227, 7, 70, "ExampleSection",
 CellID->433842623],
Cell[17710, 543, 223, 8, 70, "ExampleText",
 CellID->968656494],
Cell[CellGroupData[{
Cell[17958, 555, 431, 14, 70, "Input",
 CellID->136166751],
Cell[18392, 571, 26386, 437, 70, "Output",
 Evaluatable->False,
 CellID->194271789]
}, Open  ]],
Cell[44793, 1011, 125, 3, 70, "ExampleDelimiter",
 CellID->907062050],
Cell[44921, 1016, 63, 1, 70, "ExampleText",
 CellID->5464297],
Cell[CellGroupData[{
Cell[45009, 1021, 219, 7, 70, "Input",
 CellID->1644126505],
Cell[45231, 1030, 226, 7, 36, "Output",
 CellID->750353114]
}, Open  ]],
Cell[CellGroupData[{
Cell[45494, 1042, 221, 7, 70, "Input",
 CellID->617381836],
Cell[45718, 1051, 158, 5, 36, "Output",
 CellID->101331076]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[45925, 1062, 235, 7, 70, "ExampleSection",
 CellID->7357994],
Cell[46163, 1071, 381, 13, 70, "ExampleText",
 CellID->654548207],
Cell[CellGroupData[{
Cell[46569, 1088, 186, 6, 70, "Input",
 CellID->1580475115],
Cell[46758, 1096, 225, 7, 36, "Output",
 CellID->66834345]
}, Open  ]],
Cell[CellGroupData[{
Cell[47020, 1108, 112, 3, 70, "Input",
 CellID->1809075235],
Cell[47135, 1113, 224, 7, 36, "Output",
 CellID->378336743]
}, Open  ]],
Cell[47374, 1123, 500, 17, 70, "ExampleText",
 CellID->1184776935],
Cell[CellGroupData[{
Cell[47899, 1144, 482, 15, 70, "Input",
 CellID->1503064476],
Cell[48384, 1161, 161, 5, 36, "Output",
 CellID->290917793]
}, Open  ]],
Cell[48560, 1169, 125, 3, 70, "ExampleDelimiter",
 CellID->895693104],
Cell[48688, 1174, 75, 1, 70, "ExampleText",
 CellID->53633458],
Cell[CellGroupData[{
Cell[48788, 1179, 131, 4, 70, "Input",
 CellID->1621880513],
Cell[48922, 1185, 193, 6, 36, "Output",
 CellID->600949410]
}, Open  ]],
Cell[CellGroupData[{
Cell[49152, 1196, 132, 4, 70, "Input",
 CellID->671139712],
Cell[49287, 1202, 174, 5, 36, "Output",
 CellID->9007446]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[49522, 1214, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[49841, 1225, 740, 26, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[50618, 1256, 318, 9, 70, "TutorialsSection",
 CellID->250839057],
Cell[50939, 1267, 144, 3, 70, "Tutorials",
 CellID->23022]
}, Open  ]],
Cell[CellGroupData[{
Cell[51120, 1275, 318, 9, 70, "MoreAboutSection",
 CellID->45879745],
Cell[51441, 1286, 161, 3, 70, "MoreAbout",
 CellID->95083468],
Cell[51605, 1291, 198, 4, 70, "MoreAbout",
 CellID->456437903]
}, Open  ]],
Cell[51818, 1298, 27, 0, 70, "History"],
Cell[51848, 1300, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

