(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     62838,       1942]
NotebookOptionsPosition[     55534,       1701]
NotebookOutlinePosition[     57212,       1743]
CellTagsIndexPosition[     57126,       1738]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"ScalingTransform" :> 
          Documentation`HelpLookup["paclet:ref/ScalingTransform"], "Rescale" :> 
          Documentation`HelpLookup["paclet:ref/Rescale"], 
          "TranslationTransform" :> 
          Documentation`HelpLookup["paclet:ref/TranslationTransform"], 
          "TransformationMatrix" :> 
          Documentation`HelpLookup["paclet:ref/TransformationMatrix"], 
          "TransformationFunction" :> 
          Documentation`HelpLookup["paclet:ref/TransformationFunction"], 
          "Cuboid" :> Documentation`HelpLookup["paclet:ref/Cuboid"], 
          "GeometricTransformation" :> 
          Documentation`HelpLookup["paclet:ref/GeometricTransformation"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"ScalingTransform\"\>", 
       2->"\<\"Rescale\"\>", 3->"\<\"TranslationTransform\"\>", 
       4->"\<\"TransformationMatrix\"\>", 5->"\<\"TransformationFunction\"\>",
        6->"\<\"Cuboid\"\>", 
       7->"\<\"GeometricTransformation\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Geometric Transforms" :> 
          Documentation`HelpLookup["paclet:guide/GeometricTransforms"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Geometric Transforms\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["RescalingTransform", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["RescalingTransform",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/RescalingTransform"], "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            SubscriptBox[
             StyleBox["x", "TI"], 
             StyleBox["min", "TI"]], ",", 
            SubscriptBox[
             StyleBox["x", "TI"], 
             StyleBox["max", "TI"]]}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{
            SubscriptBox[
             StyleBox["y", "TI"], 
             StyleBox["min", "TI"]], ",", 
            SubscriptBox[
             StyleBox["y", "TI"], 
             StyleBox["max", "TI"]]}], "}"}], ",", 
          StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            SubscriptBox[
             StyleBox["xp", "TI"], 
             StyleBox["min", "TI"]], ",", 
            SubscriptBox[
             StyleBox["xp", "TI"], 
             StyleBox["max", "TI"]]}], "}"}], ",", 
          StyleBox["\[Ellipsis]", "TR"]}], "}"}]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives a ",
     Cell[BoxData[
      ButtonBox["TransformationFunction",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/TransformationFunction"]], "InlineFormula"],
     " that rescales the region with coordinate ranges ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["min", "TI"]]], "InlineFormula"],
     " to ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["max", "TI"]]], "InlineFormula"],
     ", etc. to the region with coordinate ranges ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["xp", "TI"], 
       StyleBox["min", "TI"]]], "InlineFormula"],
     " to ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["xp", "TI"], 
       StyleBox["max", "TI"]]], "InlineFormula"],
     ", etc."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["RescalingTransform",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/RescalingTransform"], "[", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           SubscriptBox[
            StyleBox["x", "TI"], 
            StyleBox["min", "TI"]], ",", 
           SubscriptBox[
            StyleBox["x", "TI"], 
            StyleBox["max", "TI"]]}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
           SubscriptBox[
            StyleBox["y", "TI"], 
            StyleBox["min", "TI"]], ",", 
           SubscriptBox[
            StyleBox["y", "TI"], 
            StyleBox["max", "TI"]]}], "}"}], ",", 
         StyleBox["\[Ellipsis]", "TR"]}], "}"}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives a ",
     Cell[BoxData[
      ButtonBox["TransformationFunction",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/TransformationFunction"]], "InlineFormula"],
     " that rescales to the unit square, cube, etc."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RescalingTransform",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RescalingTransform"]], "InlineFormula"],
 " gives a ",
 Cell[BoxData[
  ButtonBox["TransformationFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TransformationFunction"]], "InlineFormula"],
 " which can be applied to vectors."
}], "Notes",
 CellID->257351590],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RescalingTransform",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RescalingTransform"]], "InlineFormula"],
 " works in any number of dimensions. In 2D, it transforms rectangles to \
rectangles, and in 3D cuboids to cuboids."
}], "Notes",
 CellID->89947050]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->199386652],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->688769686],

Cell[TextData[{
 "Rescaling the rectangle ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"[", 
    RowBox[{
     StyleBox["xmin", "TI"], ",", 
     StyleBox["xmax", "TI"]}], "]"}], "\[Times]", 
   RowBox[{"[", 
    RowBox[{
     StyleBox["ymin", "TI"], ",", 
     StyleBox["ymax", "TI"]}], "]"}]}]], "InlineFormula"],
 " to ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"[", 
    RowBox[{"0", ",", "1"}], "]"}], "\[Times]", 
   RowBox[{"[", 
    RowBox[{"0", ",", "1"}], "]"}]}]], "InlineFormula",
  ShowStringCharacters->False],
 ":"
}], "ExampleText",
 CellID->365930361],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"t", "=", 
  RowBox[{"RescalingTransform", "[", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"xmin", ",", "xmax"}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"ymin", ",", "ymax"}], "}"}]}], "}"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->25519729],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {
      FractionBox["1", 
       RowBox[{"xmax", "-", "xmin"}]], "0", 
      RowBox[{"-", 
       FractionBox["xmin", 
        RowBox[{"xmax", "-", "xmin"}]]}]},
     {"0", 
      FractionBox["1", 
       RowBox[{"ymax", "-", "ymin"}]], 
      RowBox[{"-", 
       FractionBox["ymin", 
        RowBox[{"ymax", "-", "ymin"}]]}]},
     {"0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}}], ")"}],
   "]"}]], "Output",
 ImageSize->{366, 65},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->798569326]
}, Open  ]],

Cell[TextData[{
 "This now maps ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    StyleBox["xmax", "TI"], ",", 
    StyleBox["ymax", "TI"]}], "}"}]], "InlineFormula"],
 " to ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{"0", ",", "1"}], "}"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->162675123],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"t", "[", 
   RowBox[{"{", 
    RowBox[{"xmax", ",", "ymax"}], "}"}], "]"}], "//", "Simplify"}]], "Input",\

 CellLabel->"In[2]:=",
 CellID->41177558],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1", ",", "1"}], "}"}]], "Output",
 ImageSize->{44, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->226683430]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->542113416],

Cell[TextData[{
 "Rescaling the cube ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"[", 
    RowBox[{
     StyleBox["xmin", "TI"], ",", 
     StyleBox["xmax", "TI"]}], "]"}], "\[Times]", 
   RowBox[{"[", 
    RowBox[{
     StyleBox["ymin", "TI"], ",", 
     StyleBox["ymax", "TI"]}], "]"}], "\[Times]", 
   RowBox[{"[", 
    RowBox[{
     StyleBox["zmin", "TI"], ",", 
     StyleBox["zmax", "TI"]}], "]"}]}]], "InlineFormula"],
 " to ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"[", 
    RowBox[{"0", ",", "1"}], "]"}], "\[Times]", 
   RowBox[{"[", 
    RowBox[{"0", ",", "1"}], "]"}], "\[Times]", 
   RowBox[{"[", 
    RowBox[{"0", ",", "1"}], "]"}]}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->124938820],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"t", "=", 
  RowBox[{"RescalingTransform", "[", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"xmin", ",", "xmax"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"ymin", ",", "ymax"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"zmin", ",", "zmax"}], "}"}]}], "}"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->25142373],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {
      FractionBox["1", 
       RowBox[{"xmax", "-", "xmin"}]], "0", "0", 
      RowBox[{"-", 
       FractionBox["xmin", 
        RowBox[{"xmax", "-", "xmin"}]]}]},
     {"0", 
      FractionBox["1", 
       RowBox[{"ymax", "-", "ymin"}]], "0", 
      RowBox[{"-", 
       FractionBox["ymin", 
        RowBox[{"ymax", "-", "ymin"}]]}]},
     {"0", "0", 
      FractionBox["1", 
       RowBox[{"zmax", "-", "zmin"}]], 
      RowBox[{"-", 
       FractionBox["zmin", 
        RowBox[{"zmax", "-", "zmin"}]]}]},
     {"0", "0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}}], ")"}],
   "]"}]], "Output",
 ImageSize->{424, 89},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->609599009]
}, Open  ]],

Cell[TextData[{
 "This maps ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    StyleBox["xmax", "TI"], ",", 
    StyleBox["ymax", "TI"], ",", 
    StyleBox["zmax", "TI"]}], "}"}]], "InlineFormula"],
 " to ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{"1", ",", "1", ",", "1"}], "}"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->167739699],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"t", "[", 
   RowBox[{"{", 
    RowBox[{"xmax", ",", "ymax", ",", "zmax"}], "}"}], "]"}], "//", 
  "Simplify"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->479210254],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1", ",", "1", ",", "1"}], "}"}]], "Output",
 ImageSize->{62, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->538443706]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1293636265],

Cell[TextData[{
 "Transforming the rectangle ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"[", 
     RowBox[{
      SubscriptBox["l", "1"], ",", 
      SubscriptBox["u", "1"]}], "]"}], "\[Times]", 
    RowBox[{"[", 
     RowBox[{
      SubscriptBox["l", "2"], ",", 
      SubscriptBox["u", "2"]}], "]"}]}], TraditionalForm]], "InlineMath"],
 " to the rectangle ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"[", 
     RowBox[{
      SubscriptBox["L", "1"], ",", 
      SubscriptBox["U", "1"]}], "]"}], "\[Times]", 
    RowBox[{"[", 
     RowBox[{
      SubscriptBox["L", "2"], ",", 
      SubscriptBox["U", "2"]}], "]"}]}], TraditionalForm]], "InlineMath"],
 ": "
}], "ExampleText",
 CellID->694427974],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"t", "=", 
  RowBox[{"RescalingTransform", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"l1", ",", "u1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"l2", ",", "u2"}], "}"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"L1", ",", "U1"}], "}"}], ",", " ", 
      RowBox[{"{", 
       RowBox[{"L2", ",", "U2"}], "}"}]}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->72129810],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {
      FractionBox[
       RowBox[{
        RowBox[{"-", "L1"}], "+", "U1"}], 
       RowBox[{
        RowBox[{"-", "l1"}], "+", "u1"}]], "0", 
      RowBox[{"L1", "-", 
       FractionBox[
        RowBox[{"l1", " ", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"-", "L1"}], "+", "U1"}], ")"}]}], 
        RowBox[{
         RowBox[{"-", "l1"}], "+", "u1"}]]}]},
     {"0", 
      FractionBox[
       RowBox[{
        RowBox[{"-", "L2"}], "+", "U2"}], 
       RowBox[{
        RowBox[{"-", "l2"}], "+", "u2"}]], 
      RowBox[{"L2", "-", 
       FractionBox[
        RowBox[{"l2", " ", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"-", "L2"}], "+", "U2"}], ")"}]}], 
        RowBox[{
         RowBox[{"-", "l2"}], "+", "u2"}]]}]},
     {"0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}}], ")"}],
   "]"}]], "Output",
 ImageSize->{362, 65},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->312260653]
}, Open  ]],

Cell["\<\
This transforms the midpoint in the source rectangle to the midpoint in the \
target rectangle:\
\>", "ExampleText",
 CellID->512428900],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"t", "[", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"(", 
       RowBox[{"l1", "+", "u1"}], ")"}], "/", "2"}], ",", " ", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"l2", "+", "u2"}], ")"}], "/", "2"}]}], "}"}], "]"}], "//", 
  "Simplify"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->503646145],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox[
    RowBox[{"L1", "+", "U1"}], "2"], ",", 
   FractionBox[
    RowBox[{"L2", "+", "U2"}], "2"]}], "}"}]], "Output",
 ImageSize->{119, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->33994031]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->472240829],

Cell["Transformation applied to a 2D shape:", "ExampleText",
 CellID->479507293],

Cell[BoxData[
 RowBox[{
  RowBox[{"gr", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Rectangle", "[", "]"}], ",", 
     RowBox[{"AbsolutePointSize", "[", "10", "]"}], ",", 
     RowBox[{"Opacity", "[", "1", "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"Magenta", ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{"0", ",", "0"}], "}"}], "]"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"Green", ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{"1", ",", "1"}], "}"}], "]"}]}], "}"}]}], "}"}]}], 
  ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->419323043],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"Opacity", "[", ".35", "]"}], ",", "Blue", ",", "gr"}], "}"}], 
    ",", 
    RowBox[{"GeometricTransformation", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"Opacity", "[", ".85", "]"}], ",", "Red", ",", "gr"}], "}"}], 
      ",", 
      RowBox[{"RescalingTransform", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"0", ",", "1"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"0", ",", "1"}], "}"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"1.1", ",", "2.1"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"0", ",", "1.5"}], "}"}]}], "}"}]}], "]"}]}], "]"}]}], 
   "}"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->150504143],

Cell[BoxData[
 GraphicsBox[{
   {RGBColor[0, 0, 1], Opacity[0.35], RectangleBox[{0, 0}], 
    {AbsolutePointSize[10], Opacity[1], 
     {RGBColor[1, 0, 1], PointBox[{0, 0}]}, 
     {RGBColor[0, 1, 0], PointBox[{1, 1}]}}}, GeometricTransformationBox[
    {RGBColor[1, 0, 0], Opacity[0.85], RectangleBox[{0, 0}], 
     {AbsolutePointSize[10], Opacity[1], 
      {RGBColor[1, 0, 1], PointBox[{0, 0}]}, 
      {RGBColor[0, 1, 0], PointBox[{1, 1}]}}}, {{{1., 0.}, {0., 1.5}}, {1.1, 
     0.}}]}]], "Output",
 ImageSize->{184, 132},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->490800187]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->99099084],

Cell["Transformation applied to a 3D shape:", "ExampleText",
 CellID->137131831],

Cell[BoxData[
 RowBox[{
  RowBox[{"gr", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Cuboid", "[", "]"}], ",", 
     RowBox[{"AbsolutePointSize", "[", "10", "]"}], ",", 
     RowBox[{"Opacity", "[", "1", "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"Magenta", ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{"0", ",", "0", ",", "0"}], "}"}], "]"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"Green", ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{"1", ",", "1", ",", "1"}], "}"}], "]"}]}], "}"}]}], "}"}]}], 
  ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->367829949],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics3D", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"Opacity", "[", ".35", "]"}], ",", "Blue", ",", "gr"}], "}"}], 
     ",", 
     RowBox[{"GeometricTransformation", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"Opacity", "[", ".85", "]"}], ",", "Red", ",", "gr"}], "}"}],
        ",", 
       RowBox[{"RescalingTransform", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"0", ",", "1"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"0", ",", "1"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"0", ",", "1"}], "}"}]}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"1.2", ",", "2.2"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"0", ",", "1.5"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"0", ",", ".5"}], "}"}]}], "}"}]}], "]"}]}], "]"}]}], 
    "}"}], ",", 
   RowBox[{"Boxed", "\[Rule]", "False"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->167437831],

Cell[BoxData[
 Graphics3DBox[{
   {RGBColor[0, 0, 1], Opacity[0.35], CuboidBox[{0, 0, 0}], 
    {AbsolutePointSize[10], Opacity[1], 
     {RGBColor[1, 0, 1], Point3DBox[{0, 0, 0}]}, 
     {RGBColor[0, 1, 0], Point3DBox[{1, 1, 1}]}}}, 
   InterpretationBox[
    {RGBColor[1., 0., 0.], Opacity[0.85], 
     CuboidBox[{1.2, 0., 0.}, {2.2, 1.5, 0.5}], 
     {AbsolutePointSize[10.], Opacity[1.], 
      {RGBColor[1., 0., 1.], Point3DBox[{1.2, 0., 0.}]}, 
      {RGBColor[0., 1., 0.], Point3DBox[{2.2, 1.5, 0.5}]}}},
    GeometricTransformation[{
      Opacity[0.85], 
      RGBColor[1, 0, 0], {
       Cuboid[{0, 0, 0}], 
       AbsolutePointSize[10], 
       Opacity[1], {
        RGBColor[1, 0, 1], 
        Point[{0, 0, 0}]}, {
        RGBColor[0, 1, 0], 
        Point[{1, 1, 1}]}}}, {{{1.0000000000000002`, 0., 0.}, {0., 1.5, 0.}, {
      0., 0., 0.5}}, {1.2, 0., 0.}}]]},
  Boxed->False]], "Output",
 ImageSize->{184, 145},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->88957685,
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->656977541],

Cell["Transforming graphics primitives:", "ExampleText",
 CellID->85187644],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics3D", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"Opacity", "[", ".5", "]"}], ",", 
       RowBox[{"Cuboid", "[", "]"}]}], "}"}], ",", 
     RowBox[{"GeometricTransformation", "[", 
      RowBox[{
       RowBox[{"Cylinder", "[", "]"}], ",", 
       RowBox[{"RescalingTransform", "[", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "}"}], "]"}]}], "]"}]}], 
    "}"}], ",", 
   RowBox[{"Boxed", "\[Rule]", "False"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->460199753],

Cell[BoxData[
 Graphics3DBox[{
   {Opacity[0.5], CuboidBox[{0, 0, 0}]}, 
   InterpretationBox[CylinderBox[{{0.5, 0.5, 0.}, {0.5, 0.5, 1.}}, 0.5],
    GeometricTransformation[
     Cylinder[{{0, 0, -1}, {0, 0, 1}}], {{{
        Rational[1, 2], 0, 0}, {0, 
        Rational[1, 2], 0}, {0, 0, 
        Rational[1, 2]}}, {
       Rational[1, 2], 
       Rational[1, 2], 
       Rational[1, 2]}}]]},
  Boxed->False]], "Output",
 ImageSize->{184, 195},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->68610979,
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->197584706],

Cell[TextData[{
 "Transform from user coordinates to ",
 Cell[BoxData[
  ButtonBox["Scaled",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Scaled"]], "InlineFormula"],
 " coordinates with a particular ",
 Cell[BoxData[
  ButtonBox["PlotRange",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PlotRange"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->29647397],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"pr", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "1"}], ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "1"}], ",", "4"}], "}"}]}], "}"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{"scaled", "=", 
  RowBox[{"RescalingTransform", "[", "pr", "]"}]}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->26433637],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {
      FractionBox["1", "4"], "0", 
      FractionBox["1", "4"]},
     {"0", 
      FractionBox["1", "5"], 
      FractionBox["1", "5"]},
     {"0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}}], ")"}],
   "]"}]], "Output",
 ImageSize->{242, 61},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->192519565]
}, Open  ]],

Cell[TextData[{
 "Specify the disk in user coordinates and the circle in ",
 Cell[BoxData[
  ButtonBox["Scaled",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Scaled"]], "InlineFormula"],
 " coordinates: "
}], "ExampleText",
 CellID->705323807],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"Cyan", ",", 
     RowBox[{"Disk", "[", 
      RowBox[{"{", 
       RowBox[{"2", ",", "2"}], "}"}], "]"}], ",", "Thick", ",", "Red", ",", 
     RowBox[{"Circle", "[", 
      RowBox[{"Scaled", "[", 
       RowBox[{"scaled", "[", 
        RowBox[{"{", 
         RowBox[{"2", ",", "2"}], "}"}], "]"}], "]"}], "]"}]}], "}"}], ",", 
   RowBox[{"PlotRange", "\[Rule]", "pr"}], ",", 
   RowBox[{"Axes", "\[Rule]", "True"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->242362159],

Cell[BoxData[
 GraphicsBox[{
   {RGBColor[0, 1, 1], DiskBox[{2, 2}]}, 
   {RGBColor[1, 0, 0], Thickness[Large], 
    CircleBox[Scaled[{Rational[3, 4], Rational[3, 5]}]]}},
  Axes->True,
  PlotRange->{{-1, 3}, {-1, 4}}]], "Output",
 ImageSize->{176, 216},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->146771700]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->2123667759],

Cell[TextData[{
 "The inverse of ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RescalingTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RescalingTransform"], "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        SubscriptBox[
         StyleBox["l", "TI"], 
         StyleBox["1", "TR"]], ",", 
        SubscriptBox[
         StyleBox["u", "TI"], 
         StyleBox["1", "TR"]]}], "}"}], ",", 
      StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        SubscriptBox[
         StyleBox["L", "TI"], 
         StyleBox["1", "TR"]], ",", 
        SubscriptBox[
         StyleBox["U", "TI"], 
         StyleBox["1", "TR"]]}], "}"}], ",", 
      StyleBox["\[Ellipsis]", "TR"]}], "}"}]}], "]"}]], "InlineFormula"],
 " is given by ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RescalingTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RescalingTransform"], "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        SubscriptBox[
         StyleBox["L", "TI"], 
         StyleBox["1", "TR"]], ",", 
        SubscriptBox[
         StyleBox["U", "TI"], 
         StyleBox["1", "TR"]]}], "}"}], ",", 
      StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        SubscriptBox[
         StyleBox["l", "TI"], 
         StyleBox["1", "TR"]], ",", 
        SubscriptBox[
         StyleBox["u", "TI"], 
         StyleBox["1", "TR"]]}], "}"}], ",", 
      StyleBox["\[Ellipsis]", "TR"]}], "}"}]}], "]"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->3912962],

Cell[BoxData[{
 RowBox[{
  RowBox[{"t1", "=", 
   RowBox[{"RescalingTransform", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"l1", ",", "u1"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"l2", ",", "u2"}], "}"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"L1", ",", "U1"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"L2", ",", "U2"}], "}"}]}], "}"}]}], "]"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"t2", "=", 
   RowBox[{"RescalingTransform", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"L1", ",", "U1"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"L2", ",", "U2"}], "}"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"l1", ",", "u1"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"l2", ",", "u2"}], "}"}]}], "}"}]}], "]"}]}], ";"}]}], "Input",\

 CellLabel->"In[1]:=",
 CellID->3689491],

Cell[TextData[{
 "This shows that ",
 Cell[BoxData["t1"], "InlineFormula"],
 " and ",
 Cell[BoxData["t2"], "InlineFormula"],
 " are inverses:"
}], "ExampleText",
 CellID->123222125],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Composition", "[", 
   RowBox[{"t1", ",", "t2"}], "]"}], "//", "Simplify"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->769770174],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {"1", "0", "0"},
     {"0", "1", "0"},
     {"0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}}], ")"}],
   "]"}]], "Output",
 ImageSize->{230, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->741627207]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->119516733],

Cell["\<\
Rescaling transformation is a composition of scaling and translation:\
\>", "ExampleText",
 CellID->523719973],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"RescalingTransform", "[", 
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{"a", ",", "b"}], "}"}], "}"}], "]"}], "\[Equal]", 
  RowBox[{"Composition", "[", 
   RowBox[{
    RowBox[{"TranslationTransform", "[", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "a"}], "/", 
       RowBox[{"(", 
        RowBox[{"b", "-", "a"}], ")"}]}], "}"}], "]"}], ",", 
    RowBox[{"ScalingTransform", "[", 
     RowBox[{"{", 
      RowBox[{"1", "/", 
       RowBox[{"(", 
        RowBox[{"b", "-", "a"}], ")"}]}], "}"}], "]"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->87731033],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->177067120]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->335493545],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Rescale",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Rescale"]], "InlineFormula"],
 " provides a scalar version of ",
 Cell[BoxData[
  ButtonBox["RescalingTransform",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RescalingTransform"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->201197035],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"Rescale", "[", 
    RowBox[{"x", ",", 
     RowBox[{"{", 
      RowBox[{
       SubscriptBox["l", "1"], ",", 
       SubscriptBox["u", "1"]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       SubscriptBox["L", "1"], ",", 
       SubscriptBox["U", "1"]}], "}"}]}], "]"}], "\[Equal]", 
   RowBox[{"First", "@", 
    RowBox[{
     RowBox[{"RescalingTransform", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{
          SubscriptBox["l", "1"], ",", 
          SubscriptBox["u", "1"]}], "}"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{
          SubscriptBox["L", "1"], ",", 
          SubscriptBox["U", "1"]}], "}"}], "}"}]}], "]"}], "[", 
     RowBox[{"{", "x", "}"}], "]"}]}]}], "//", "Simplify"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->435329008],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->441604376]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Neat Examples",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->589267740],

Cell["A collection of randomly rescaled cuboids:", "ExampleText",
 CellID->93217913],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics3D", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"GeometricTransformation", "[", 
     RowBox[{
      RowBox[{"Cuboid", "[", "]"}], ",", 
      RowBox[{"RescalingTransform", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"0", ",", "1"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"0", ",", "1"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"0", ",", "1"}], "}"}]}], "}"}], ",", 
        RowBox[{"RandomReal", "[", 
         RowBox[{"1", ",", 
          RowBox[{"{", 
           RowBox[{"3", ",", "2"}], "}"}]}], "]"}]}], "]"}]}], "]"}], ",", 
    RowBox[{"{", "30", "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->36107864],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 197},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->251588034]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["ScalingTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ScalingTransform"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Rescale",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Rescale"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["TranslationTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/TranslationTransform"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["TransformationMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/TransformationMatrix"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["TransformationFunction",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/TransformationFunction"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Cuboid",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Cuboid"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["GeometricTransformation",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/GeometricTransformation"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection"],

Cell[TextData[{
 ButtonBox["Demonstrations with RescalingTransform",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/symbol.html?symbol=\
RescalingTransform"], None}],
 " (",
 ButtonBox["Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None}],
 ")"
}], "RelatedLinks"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->198374323],

Cell[TextData[ButtonBox["Geometric Transforms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/GeometricTransforms"]], "MoreAbout",
 CellID->13965720]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"RescalingTransform - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 16, 29.9417714}", "context" -> "System`", 
    "keywords" -> {
     "box transform", "rescaling coordinates", "resize", "scale", 
      "scaled coordinates", "windowing transform"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "RescalingTransform[{{x_min, x_max}, {y_min, y_max}, ...}, {{xp_min, \
xp_max}, ...}] gives a TransformationFunction that rescales the region with \
coordinate ranges x_min to x_max, etc. to the region with coordinate ranges \
xp_min to xp_max, etc. RescalingTransform[{{x_min, x_max}, {y_min, y_max}, \
...}] gives a TransformationFunction that rescales to the unit square, cube, \
etc.", "synonyms" -> {"rescaling transform"}, "title" -> "RescalingTransform",
     "type" -> "Symbol", "uri" -> "ref/RescalingTransform"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[8731, 261, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->199386652]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 56983, 1731}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 1985, 44, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2608, 71, 1583, 37, 70, "ObjectNameGrid"],
Cell[4194, 110, 3293, 102, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[7524, 217, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[8002, 234, 385, 12, 70, "Notes",
 CellID->257351590],
Cell[8390, 248, 304, 8, 70, "Notes",
 CellID->89947050]
}, Closed]],
Cell[CellGroupData[{
Cell[8731, 261, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->199386652],
Cell[CellGroupData[{
Cell[9116, 275, 148, 5, 70, "ExampleSection",
 CellID->688769686],
Cell[9267, 282, 560, 22, 70, "ExampleText",
 CellID->365930361],
Cell[CellGroupData[{
Cell[9852, 308, 308, 10, 28, "Input",
 CellID->25519729],
Cell[10163, 320, 695, 24, 86, "Output",
 CellID->798569326]
}, Open  ]],
Cell[10873, 347, 305, 13, 70, "ExampleText",
 CellID->162675123],
Cell[CellGroupData[{
Cell[11203, 364, 184, 7, 70, "Input",
 CellID->41177558],
Cell[11390, 373, 203, 7, 36, "Output",
 CellID->226683430]
}, Open  ]],
Cell[11608, 383, 125, 3, 70, "ExampleDelimiter",
 CellID->542113416],
Cell[11736, 388, 706, 27, 70, "ExampleText",
 CellID->124938820],
Cell[CellGroupData[{
Cell[12467, 419, 372, 12, 70, "Input",
 CellID->25142373],
Cell[12842, 433, 882, 30, 110, "Output",
 CellID->609599009]
}, Open  ]],
Cell[13739, 466, 345, 14, 70, "ExampleText",
 CellID->167739699],
Cell[CellGroupData[{
Cell[14109, 484, 199, 7, 70, "Input",
 CellID->479210254],
Cell[14311, 493, 213, 7, 36, "Output",
 CellID->538443706]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[14573, 506, 221, 7, 70, "ExampleSection",
 CellID->1293636265],
Cell[14797, 515, 717, 27, 70, "ExampleText",
 CellID->694427974],
Cell[CellGroupData[{
Cell[15539, 546, 495, 17, 70, "Input",
 CellID->72129810],
Cell[16037, 565, 1131, 40, 86, "Output",
 CellID->312260653]
}, Open  ]],
Cell[17183, 608, 146, 4, 70, "ExampleText",
 CellID->512428900],
Cell[CellGroupData[{
Cell[17354, 616, 351, 13, 70, "Input",
 CellID->503646145],
Cell[17708, 631, 301, 11, 51, "Output",
 CellID->33994031]
}, Open  ]],
Cell[18024, 645, 125, 3, 70, "ExampleDelimiter",
 CellID->472240829],
Cell[18152, 650, 80, 1, 70, "ExampleText",
 CellID->479507293],
Cell[18235, 653, 620, 20, 70, "Input",
 CellID->419323043],
Cell[CellGroupData[{
Cell[18880, 677, 913, 30, 70, "Input",
 CellID->150504143],
Cell[19796, 709, 634, 15, 153, "Output",
 CellID->490800187]
}, Open  ]],
Cell[20445, 727, 124, 3, 70, "ExampleDelimiter",
 CellID->99099084],
Cell[20572, 732, 80, 1, 70, "ExampleText",
 CellID->137131831],
Cell[20655, 735, 637, 20, 70, "Input",
 CellID->367829949],
Cell[CellGroupData[{
Cell[21317, 759, 1154, 36, 70, "Input",
 CellID->167437831],
Cell[22474, 797, 4177, 81, 166, 901, 23, "CachedBoxData", "BoxData", "Output",
 
 CellID->88957685]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[26700, 884, 227, 7, 70, "ExampleSection",
 CellID->656977541],
Cell[26930, 893, 75, 1, 70, "ExampleText",
 CellID->85187644],
Cell[CellGroupData[{
Cell[27030, 898, 854, 27, 70, "Input",
 CellID->460199753],
Cell[27887, 927, 7984, 140, 216, 422, 12, "CachedBoxData", "BoxData", \
"Output",
 CellID->68610979]
}, Open  ]],
Cell[35886, 1070, 125, 3, 70, "ExampleDelimiter",
 CellID->197584706],
Cell[36014, 1075, 370, 13, 70, "ExampleText",
 CellID->29647397],
Cell[CellGroupData[{
Cell[36409, 1092, 417, 15, 70, "Input",
 CellID->26433637],
Cell[36829, 1109, 505, 18, 82, "Output",
 CellID->192519565]
}, Open  ]],
Cell[37349, 1130, 247, 8, 70, "ExampleText",
 CellID->705323807],
Cell[CellGroupData[{
Cell[37621, 1142, 559, 16, 70, "Input",
 CellID->242362159],
Cell[38183, 1160, 362, 11, 237, "Output",
 CellID->146771700]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[38594, 1177, 238, 7, 70, "ExampleSection",
 CellID->2123667759],
Cell[38835, 1186, 1679, 61, 70, "ExampleText",
 CellID->3912962],
Cell[40517, 1249, 1011, 36, 70, "Input",
 CellID->3689491],
Cell[41531, 1287, 181, 7, 70, "ExampleText",
 CellID->123222125],
Cell[CellGroupData[{
Cell[41737, 1298, 164, 5, 70, "Input",
 CellID->769770174],
Cell[41904, 1305, 405, 14, 64, "Output",
 CellID->741627207]
}, Open  ]],
Cell[42324, 1322, 125, 3, 70, "ExampleDelimiter",
 CellID->119516733],
Cell[42452, 1327, 120, 3, 70, "ExampleText",
 CellID->523719973],
Cell[CellGroupData[{
Cell[42597, 1334, 617, 20, 70, "Input",
 CellID->87731033],
Cell[43217, 1356, 161, 5, 36, "Output",
 CellID->177067120]
}, Open  ]],
Cell[43393, 1364, 125, 3, 70, "ExampleDelimiter",
 CellID->335493545],
Cell[43521, 1369, 350, 12, 70, "ExampleText",
 CellID->201197035],
Cell[CellGroupData[{
Cell[43896, 1385, 867, 29, 70, "Input",
 CellID->435329008],
Cell[44766, 1416, 161, 5, 36, "Output",
 CellID->441604376]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[44976, 1427, 228, 7, 70, "ExampleSection",
 CellID->589267740],
Cell[45207, 1436, 84, 1, 70, "ExampleText",
 CellID->93217913],
Cell[CellGroupData[{
Cell[45316, 1441, 768, 23, 70, "Input",
 CellID->36107864],
Cell[46087, 1466, 5967, 102, 70, "Output",
 Evaluatable->False,
 CellID->251588034]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[52115, 1575, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[52434, 1586, 1796, 62, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[54267, 1653, 305, 8, 70, "RelatedLinksSection"],
Cell[54575, 1663, 378, 12, 70, "RelatedLinks"]
}, Open  ]],
Cell[CellGroupData[{
Cell[54990, 1680, 319, 9, 70, "MoreAboutSection",
 CellID->198374323],
Cell[55312, 1691, 150, 3, 70, "MoreAbout",
 CellID->13965720]
}, Open  ]],
Cell[55477, 1697, 27, 0, 70, "History"],
Cell[55507, 1699, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

