(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    168428,       3794]
NotebookOptionsPosition[    159625,       3493]
NotebookOutlinePosition[    161224,       3535]
CellTagsIndexPosition[    161137,       3530]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"EllipticTheta" :> 
          Documentation`HelpLookup["paclet:ref/EllipticTheta"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"EllipticTheta\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Elliptic Functions" :> 
          Documentation`HelpLookup["paclet:guide/EllipticFunctions"], 
          "Special Functions" :> 
          Documentation`HelpLookup["paclet:guide/SpecialFunctions"], 
          "New in 6.0: Mathematical Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicalFunctions"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Elliptic Functions\"\>", 
       2->"\<\"Special Functions\"\>", 
       3->"\<\"New in 6.0: Mathematical Functions\"\>"}, "\<\"more about\"\>",
        
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["SiegelTheta", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["SiegelTheta",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/SiegelTheta"], "[", 
       RowBox[{
        StyleBox["\[CapitalOmega]", "TR"], ",", 
        StyleBox["s", "TI"]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the Siegel theta function ",
     Cell[BoxData[
      FormBox[
       RowBox[{"\[CapitalTheta]", "(", 
        RowBox[{
         StyleBox["\[CapitalOmega]", "TR"], ",", "s"}], ")"}], 
       TraditionalForm]], "InlineMath"],
     " with Riemann modular matrix ",
     Cell[BoxData[
      FormBox[
       StyleBox["\[CapitalOmega]", "TR"], TraditionalForm]], "InlineMath"],
     " and vector ",
     Cell[BoxData[
      StyleBox["s", "TI"]], "InlineFormula"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["SiegelTheta",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/SiegelTheta"], "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          SubscriptBox[
           StyleBox["\[Nu]", "TR"], 
           StyleBox["1", "TR"]], ",", 
          SubscriptBox[
           StyleBox["\[Nu]", "TR"], 
           StyleBox["2", "TR"]]}], "}"}], ",", 
        StyleBox["\[CapitalOmega]", "TR"], ",", 
        StyleBox["s", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the Siegel theta function ",
     Cell[BoxData[
      FormBox[
       RowBox[{
        RowBox[{"\[CapitalTheta]", "[", 
         RowBox[{
          SubscriptBox["\[Nu]", "1"], ",", 
          SubscriptBox["\[Nu]", "2"]}], "]"}], 
        RowBox[{"(", " ", 
         RowBox[{
          StyleBox["\[CapitalOmega]", "TR"], ",", "s"}], ")"}]}], 
       TraditionalForm]], "InlineMath"],
     " with characteristics ",
     Cell[BoxData[
      FormBox[
       SubscriptBox["\[Nu]", "1"], TraditionalForm]], "InlineMath"],
     " and ",
     Cell[BoxData[
      FormBox[
       SubscriptBox["\[Nu]", "2"], TraditionalForm]]],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->9985]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["\<\
Mathematical function, suitable for both symbolic and numerical manipulation.\
\
\>", "Notes",
 CellID->20640],

Cell[TextData[{
 "The matrix",
 " ",
 Cell[BoxData[
  FormBox[
   StyleBox["\[CapitalOmega]", "TR"], TraditionalForm]], "InlineMath"],
 " ",
 "must be symmetric, with positive definite imaginary part. "
}], "Notes",
 CellID->70201465],

Cell[TextData[{
 "If ",
 Cell[BoxData[
  FormBox[
   StyleBox["\[CapitalOmega]", "TR"], TraditionalForm]], "InlineMath"],
 " is a ",
 Cell[BoxData[
  RowBox[{
   StyleBox["p", "TI"], "\[Times]", 
   StyleBox["p", "TI"]}]], "InlineFormula"],
 " matrix, the vectors ",
 Cell[BoxData[
  StyleBox["s", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["v", "TI"]], "InlineFormula"],
 " or ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["\[Nu]", "TR"], 
   StyleBox["i", "TI"]]], "InlineFormula"],
 " must have length ",
 Cell[BoxData[
  StyleBox["p", "TI"]], "InlineFormula"],
 "."
}], "Notes",
 CellID->19750],

Cell[TextData[{
 " ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"\[CapitalTheta]", "(", 
     RowBox[{
      StyleBox["\[CapitalOmega]", "TR"], ",", "s"}], ")"}], "=", 
    RowBox[{
     UnderscriptBox["\[Sum]", "n"], 
     SuperscriptBox["\[ExponentialE]", 
      RowBox[{" ", 
       RowBox[{"\[ImaginaryI]", " ", "\[Pi]", " ", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"n", ".", 
           StyleBox["\[CapitalOmega]", "TR"], ".", "n"}], "+", 
          RowBox[{"2", " ", 
           RowBox[{"n", ".", "s"}]}]}], ")"}]}]}]]}]}], TraditionalForm]], 
  "InlineMath",
  LimitsPositioningTokens->{
   "\[Product]", "\[Intersection]", "\[Union]", "\[UnionPlus]", "\[Wedge]", 
    "\[Vee]", "lim", "max", "min", "\[CirclePlus]", "\[CircleMinus]", 
    "\[CircleTimes]", "\[CircleDot]"}],
 ", where ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " ranges over all possible vectors in the ",
 Cell[BoxData[
  StyleBox["p", "TI"]], "InlineFormula"],
 "-dimensional integer lattice."
}], "Notes",
 CellID->771240644],

Cell[TextData[{
 " ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{"\[CapitalTheta]", "[", 
      RowBox[{
       SubscriptBox["\[Nu]", "1"], ",", 
       SubscriptBox["\[Nu]", "2"]}], "]"}], 
     RowBox[{"(", 
      RowBox[{
       StyleBox["\[CapitalOmega]", "TR"], ",", "s"}], ")"}]}], "=", 
    RowBox[{
     UnderscriptBox["\[Sum]", "n"], 
     SuperscriptBox["\[ExponentialE]", 
      RowBox[{" ", 
       RowBox[{"\[ImaginaryI]", " ", "\[Pi]", " ", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{
           RowBox[{"(", 
            RowBox[{"n", "+", 
             SubscriptBox["\[Nu]", "1"]}], ")"}], ".", 
           StyleBox["\[CapitalOmega]", "TR"], ".", 
           RowBox[{"(", 
            RowBox[{"n", "+", 
             SubscriptBox["\[Nu]", "1"]}], ")"}]}], "+", 
          RowBox[{"2", " ", 
           RowBox[{
            RowBox[{"(", 
             RowBox[{"n", "+", 
              SubscriptBox["\[Nu]", "1"]}], ")"}], ".", 
            RowBox[{"(", 
             RowBox[{"s", "+", 
              SubscriptBox["\[Nu]", "2"]}], ")"}]}]}]}], ")"}]}]}]]}]}], 
   TraditionalForm]], "InlineMath",
  LimitsPositioningTokens->{
   "\[Product]", "\[Intersection]", "\[Union]", "\[UnionPlus]", "\[Wedge]", 
    "\[Vee]", "lim", "max", "min", "\[CirclePlus]", "\[CircleMinus]", 
    "\[CircleTimes]", "\[CircleDot]"}],
 ", where ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " ranges over all possible vectors in the ",
 Cell[BoxData[
  StyleBox["p", "TI"]], "InlineFormula"],
 "-dimensional integer lattice. "
}], "Notes",
 CellID->98455669],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SiegelTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SiegelTheta"]], "InlineFormula"],
 " can be evaluated to arbitrary numerical precision."
}], "Notes",
 CellID->31226]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->394591841],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->65399100],

Cell["Evaluate numerically:", "ExampleText",
 CellID->359970644],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SiegelTheta", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"I", ",", 
       RowBox[{
        RowBox[{"-", "1"}], "/", "2"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"-", "1"}], "/", "2"}], ",", "I"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.5", ",", "0"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->31370694],

Cell[BoxData[
 RowBox[{"1.007483720344903`", "\[InvisibleSpace]", "+", 
  RowBox[{"0.`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{91, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->53054217]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->941397941],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Re", "[", 
    RowBox[{"SiegelTheta", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"I", ",", 
          RowBox[{
           RowBox[{"-", "1"}], "/", "2"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{
           RowBox[{"-", "1"}], "/", "2"}], ",", " ", "I"}], "}"}]}], "}"}], 
      ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", 
        RowBox[{"2", " ", "x"}]}], "}"}]}], "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->305506288],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 119},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->716492446]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->663858768],

Cell[TextData[{
 "Evaluate ",
 Cell[BoxData[
  ButtonBox["SiegelTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SiegelTheta"]], "InlineFormula"],
 " for higher-dimensional arguments:"
}], "ExampleText",
 CellID->661921282],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SiegelTheta", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"IdentityMatrix", "[", "4", "]"}], " ", "I"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.5", ",", "0", ",", "0", ",", "0"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->945695114],

Cell[BoxData[
 RowBox[{"1.1715401901901221`", "\[InvisibleSpace]", "+", 
  RowBox[{"0.`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{91, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->19043360]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->68546449],

Cell["Evaluate for complex arguments:", "ExampleText",
 CellID->110643755],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SiegelTheta", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"I", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "I"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"0.5", "+", "I"}], ",", 
     RowBox[{"2", " ", "I"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->61459423],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "6.5861499712015875`*^6"}], "-", 
  RowBox[{"8.065441101960532`*^-10", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{205, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->485532050]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->113702518],

Cell["Evaluate to high precision:", "ExampleText",
 CellID->1885257843],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"SiegelTheta", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"I", ",", "0"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"0", ",", 
         RowBox[{
          RowBox[{"-", "1"}], "+", 
          RowBox[{"4", "I"}]}]}], "}"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"6", "/", "5"}], ",", 
       RowBox[{"1", "+", 
        RowBox[{"3", "I"}]}]}], "}"}]}], "]"}], ",", "50"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1554839463],

Cell[BoxData[
 RowBox[{
  RowBox[{
  "-", "548.7636629006731053166200760600436049968587247406080466788368`50.\
15051499783199"}], "+", 
  RowBox[{"0``47.41112965155493", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{448, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->748128496]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->23239488],

Cell["\<\
The precision of the output tracks the precision of the input:\
\>", "ExampleText",
 CellID->1891534709],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SiegelTheta", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"I", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", 
       RowBox[{
        RowBox[{"-", "1"}], "+", 
        RowBox[{"4", "I"}]}]}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1.2000000000000000000000", ",", 
     RowBox[{"1", "+", 
      RowBox[{"3", " ", "I"}]}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1823427622],

Cell[BoxData[
 RowBox[{
  RowBox[{
  "-", "548.76366290067310531662007606004589859187`22.229696243879616"}], "+",
   
  RowBox[{"0``19.49031089760256", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{252, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->766411675]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->497907993],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SiegelTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SiegelTheta"]], "InlineFormula"],
 " with characteristics ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox[
    StyleBox["\[Nu]", "TR"], 
    StyleBox["1", "TR"]], "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"1", "/", "2"}], ",", "0"}], "}"}]}]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox[
    StyleBox["\[Nu]", "TR"], 
    StyleBox["2", "TR"]], "=", 
   RowBox[{"{", 
    RowBox[{"0", ",", 
     RowBox[{"1", "/", "2"}]}], "}"}]}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->364931836],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SiegelTheta", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"1", "/", "2"}], ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", 
       RowBox[{"1", "/", "2"}]}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"I", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "I"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"0.1", "+", "I"}], ",", "0"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->196071820],

Cell[BoxData[
 RowBox[{"14.849894009124256`", "\[InvisibleSpace]", "-", 
  RowBox[{"10.7890795340008`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{126, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->138341053]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->262151406],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SiegelTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SiegelTheta"]], "InlineFormula"],
 " with characteristics simplifies symbolically for special arguments:"
}], "ExampleText",
 CellID->1489720645],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SiegelTheta", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"0", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"a", ",", "b"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"I", ",", 
       RowBox[{"1", "/", "2"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"1", "/", "2"}], ",", "I"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->223040142],

Cell[BoxData[
 RowBox[{"SiegelTheta", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"\[ImaginaryI]", ",", 
       FractionBox["1", "2"]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       FractionBox["1", "2"], ",", "\[ImaginaryI]"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"a", ",", "b"}], "}"}]}], "]"}]], "Output",
 ImageSize->{260, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->17771563]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->403588778],

Cell[TextData[{
 "Plot of the absolute value of ",
 Cell[BoxData[
  ButtonBox["SiegelTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SiegelTheta"]], "InlineFormula"],
 " in the complex plane:"
}], "ExampleText",
 CellID->179005300],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"Abs", "[", 
    RowBox[{"SiegelTheta", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"I", ",", "0.5"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"0.5", ",", "I"}], "}"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", "y"}], "}"}]}], "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->36951453],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{265, 210},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->301342322]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1688517432],

Cell["Define an Abelian function:", "ExampleText",
 CellID->2063568166],

Cell[BoxData[{
 RowBox[{
  RowBox[{"m", " ", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"1", "+", "I"}], ",", " ", 
       RowBox[{
        RowBox[{"-", "1"}], "/", "2"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"-", "1"}], "/", "2"}], ",", 
       RowBox[{"1", "+", "I"}]}], "}"}]}], "}"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"a", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"1", "/", "3"}], ",", 
     RowBox[{"1", "/", "4"}]}], "}"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"b", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"1", "/", "5"}], ",", 
     RowBox[{"1", "/", "6"}]}], "}"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"c", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"2", "/", "5"}], ",", 
     RowBox[{"1", "/", "12"}]}], "}"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"d", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"2", "/", "15"}], ",", 
     RowBox[{"1", "/", "3"}]}], "}"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->2022594182],

Cell[BoxData[
 RowBox[{
  RowBox[{"f", "[", "s_", "]"}], ":=", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"SiegelTheta", "[", 
      RowBox[{"m", ",", 
       RowBox[{"s", "+", "a"}]}], "]"}], 
     RowBox[{"SiegelTheta", "[", 
      RowBox[{"m", ",", 
       RowBox[{"s", "+", "b"}]}], "]"}]}], ")"}], "/", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"SiegelTheta", "[", 
      RowBox[{"m", ",", 
       RowBox[{"s", "+", "c"}]}], "]"}], 
     RowBox[{"SiegelTheta", "[", 
      RowBox[{"m", ",", 
       RowBox[{"s", "+", "d"}]}], "]"}]}], ")"}]}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->529266912],

Cell["Plot of the real part:", "ExampleText",
 CellID->1221768511],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Re", "[", 
    RowBox[{"f", "[", 
     RowBox[{"x", " ", "I"}], "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->762399044],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 106},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->165474680]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->124660801],

Cell[TextData[{
 "In one dimension, ",
 Cell[BoxData[
  ButtonBox["SiegelTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SiegelTheta"]], "InlineFormula"],
 " coincides with the ",
 Cell[BoxData[
  ButtonBox["EllipticTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/EllipticTheta"]], "InlineFormula"],
 " functions:"
}], "ExampleText",
 CellID->1757091478],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SiegelTheta", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{"0.5", "+", "I"}], "}"}], "}"}], ",", 
   RowBox[{"{", "2", "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->235086758],

Cell[BoxData[
 RowBox[{"1.0000069746847124`", "\[InvisibleSpace]", "+", 
  RowBox[{"0.08642783652754452`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{140, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->446262312]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"EllipticTheta", "[", 
  RowBox[{"3", ",", 
   RowBox[{"2", "Pi"}], ",", 
   RowBox[{"E", "^", 
    RowBox[{"(", " ", 
     RowBox[{"I", "*", "Pi", "*", 
      RowBox[{"(", 
       RowBox[{"0.5", "+", "I"}], ")"}]}], ")"}]}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->363070318],

Cell[BoxData[
 RowBox[{"1.0000069746847124`", "\[InvisibleSpace]", "+", 
  RowBox[{"0.08642783652859562`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{140, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->101020158]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1442857924],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SiegelTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SiegelTheta"]], "InlineFormula"],
 " satisfies the equations:"
}], "ExampleText",
 CellID->727786744],

Cell[BoxData[{
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"I", ",", " ", 
       RowBox[{"1", "/", "2"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"1", "/", "2"}], ",", "I"}], "}"}]}], "}"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"s", "=", 
   RowBox[{"{", 
    RowBox[{"0.5`20", ",", " ", "0.6`20"}], "}"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->1628072117],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"SiegelTheta", "[", 
   RowBox[{"m", ",", 
    RowBox[{"-", "s"}]}], "]"}], "\[Equal]", 
  RowBox[{"SiegelTheta", "[", 
   RowBox[{"m", ",", "s"}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->614088907],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->192025603]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"SiegelTheta", "[", 
   RowBox[{"m", ",", 
    RowBox[{"s", "+", "1"}]}], "]"}], "\[Equal]", 
  RowBox[{"SiegelTheta", "[", 
   RowBox[{"m", ",", "s"}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->422826564],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->254824041]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"SiegelTheta", "[", 
   RowBox[{"m", ",", 
    RowBox[{"s", "+", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"1", ",", "2"}], "}"}], ".", " ", "m"}]}]}], "]"}], " ", 
  "\[Equal]", " ", 
  RowBox[{
   RowBox[{"E", "^", 
    RowBox[{"(", 
     RowBox[{"Pi", " ", "I", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"1", ",", "2"}], "}"}], ".", "m", ".", 
          RowBox[{"{", 
           RowBox[{"1", ",", "2"}], "}"}]}]}], "-", 
        RowBox[{"2", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"1", ",", "2"}], "}"}], ".", "s"}]}]}], ")"}]}], ")"}]}], 
   RowBox[{"SiegelTheta", "[", 
    RowBox[{"m", ",", "s"}], "]"}]}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->1122135009],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->328891423]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->208392064],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SiegelTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SiegelTheta"]], "InlineFormula"],
 " requires a symmetric ",
 Cell[BoxData[
  StyleBox["\[Tau]", "TR"]], "InlineFormula"],
 " matrix:"
}], "ExampleText",
 CellID->810605206],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"I", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", 
       RowBox[{"2", "I"}]}], "}"}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->321849602],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SiegelTheta", "[", 
  RowBox[{"m", ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2."}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->312846986],

Cell[BoxData[
 RowBox[{
  RowBox[{"SiegelTheta", "::", "\<\"invmat\"\>"}], ":", 
  " ", "\<\"\\!\\({\\(\\({\[ImaginaryI], 1}\\)\\), \\(\\({0, \\(\\(2\\\\ \
\[ImaginaryI]\\)\\)}\\)\\)}\\) must be a symmetric matrix with a positive \
definite imaginary part.\"\>"}]], "Message", "MSG",
 CellID->157722668],

Cell[BoxData[
 RowBox[{"SiegelTheta", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"\[ImaginaryI]", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", 
       RowBox[{"2", " ", "\[ImaginaryI]"}]}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2.`"}], "}"}]}], "]"}]], "Output",
 ImageSize->{268, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->73666234]
}, Open  ]],

Cell["The symmetric part: ", "ExampleText",
 CellID->95037792],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SiegelTheta", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{"m", "+", 
      RowBox[{"m", "\[Transpose]"}]}], ")"}], "/", "2"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2."}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->937563001],

Cell[BoxData[
 RowBox[{"1.0898469246793634`", "\[InvisibleSpace]", "+", 
  RowBox[{"2.3473152817982554`*^-37", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{167, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->687522528]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1633651548],

Cell["\<\
Machine precision may be insufficient to obtain a correct answer:\
\>", "ExampleText",
 CellID->1203140341],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SiegelTheta", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{
      SuperscriptBox["10", 
       RowBox[{"-", "3"}]], " ", "I"}], "}"}], "}"}], ",", 
   RowBox[{"{", "0.5", "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->473691035],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "2.3104369528681445`*^-9"}], "+", 
  RowBox[{"8.38164711797325`*^-31", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{210, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->69564747]
}, Open  ]],

Cell["Use arbitrary precision to check the result:", "ExampleText",
 CellID->21240796],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Block", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"$MaxExtraPrecision", "=", "500"}], "}"}], ",", 
   RowBox[{"N", "[", 
    RowBox[{
     RowBox[{"SiegelTheta", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{
          SuperscriptBox["10", 
           RowBox[{"-", "3"}]], " ", "I"}], "}"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"1", "/", "2"}], "}"}]}], "]"}], ",", "50"}], "]"}]}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->154347460],

Cell[BoxData[
 RowBox[{"5.0926230946447430804394208683662863171433801931524074406465154255`\
50.15051499783199*^-340", "+", 
  RowBox[{"0``389.4435734626284", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{491, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->533944107]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"EllipticTheta", "[", 
    RowBox[{"3", ",", 
     RowBox[{"\[Pi]", "/", "2"}], ",", 
     RowBox[{"Exp", "[", 
      RowBox[{
       RowBox[{"-", "\[Pi]"}], " ", 
       SuperscriptBox["10", 
        RowBox[{"-", "3"}]]}], "]"}]}], "]"}], ",", "50"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->423840250],

Cell[BoxData[
 RowBox[{"5.0926230946447430804394208683662863171433801931524074031412215207`\
50.15051499783199*^-340", "+", 
  RowBox[{"0``389.4435734626284", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{491, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->114237547]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Neat Examples",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->706393597],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ContourPlot", "[", 
  RowBox[{
   RowBox[{"Arg", "[", 
    RowBox[{"Nest", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"SiegelTheta", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"I", ",", 
             RowBox[{
              RowBox[{"Re", "[", "#", "]"}], "-", 
              RowBox[{"Im", "[", "#", "]"}]}]}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{
             RowBox[{
              RowBox[{"Re", "[", "#", "]"}], "-", 
              RowBox[{"Im", "[", "#", "]"}]}], ",", "I"}], "}"}]}], "}"}], 
         ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"Re", "[", "#", "]"}], ",", 
           RowBox[{"Im", "[", "#", "]"}]}], "}"}]}], "]"}], " ", "&"}], ",", 
      RowBox[{"x", "+", 
       RowBox[{"y", " ", "I"}]}], ",", "5"}], "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1328152518],

Cell[BoxData[
 GraphicsBox[GraphicsComplexBox[CompressedData["
1:eJyNvQk4VV30P640SCNJA00kjfI2j5aihKRZadI8UVQqlSKluSRNSCFFUmZl
HkLmeczsXve6l0saKNP/ft++65zf2ef1+/09nuc8n7PP2Xutvde0h7PuxL0n
NhzoLSEh4T9AQuJ/ruSf4yRd8X8gMHE8hb+Gmon/04jyXAr/W/y1kMKlf29Q
uKv7f/4qKfzg3/drKKz8P9CRQ+GQf1+vo7DOvw3WU7jk3/qFFD72L4GNFP77
10xgms//vsZTGPllludSGPlFjPwiRn4RI7+IkV/EyC9i5Bcx8osY+WWOWzOB
6XH8bz7pK5PfeILfeILfeILfeILfeILfeILfeILfeILfeILfeILfeIJftpz+
9zjSfJJX5Bcx8osY+UWM/CJGfhEjv4iRX8TIL2LkFzHyy9S7ZgLTevjfchrP
4otZTl+Z45tLjG8uMb65xPjmEuObS4xvLjG+ucT45hLjm0uML9vO/LcexrPG
jeQLMXlFfhEjv4iRX8TIL2LkFzHyixj5RYz8Mu1mM4FpO/rfdiaeJZfkuJF8
ISavyC9i5Bcx8osY+UWM/CJGfhEjv0y/0Aw9+Yn/tqPxLL0j5ZIcN5IvxOQV
+UWM/CJGfhEjv4iRX8TIL9PvNUNPfhD5Je0maVdIvSPlkhw3ki/E5BX5RYz8
IkZ+ESO/iJFfpl9vhp78PPJL+gXSbpJ2hdQ7Ui7JcSP5QkxekV/EyC9i5Bcx
8suMW5qhpzgG+SX9HukXSLtJ2hVS70i5JMeN5AsxeUV+ESO/iJFfZlzWDD3F
acgv6ddJv0f6BdJuknaF1DtSLslxI/lCTF6RX8TILzPubIae4lDkl4xbSL9O
+j3SL5B2k7QrpN6RckmOG8kXYvKK/DLj6mboKc5Gfsm4jIxbSL9O+j3SL5B2
k7QrpN6RckmOG8kXYvKK/PY0j0B+/zvuJOMyMm4h/Trp90i/QNpN0q6QekfK
JTluJF/NrOtfej2oK3KO+C+5IqKc/R4+h+8j7ulK1kO+R9JB0s2ky5G6/h2P
ph7vk1dmO4493ifbJ9uRYPw1s+7/d7tYTwTRj0kU/isfPAo7/HsVUFjp32Ya
WOOB9f5tJ5vCf8nKpzDWjxjrR4z93BOdWD/i1f8+UEJhpA8xUw6SWHSS9SK9
iLF+Jq6m8F/9qqUwts+sX0Rhsh3TfwkqpzDWjxjr74kufB8xvs8s51IY+x8x
SQdZL7aPGOtDjPUhJvuJ5IukE+tHzKSX3c9kv5B0kvUifYhR3pjPl7D4IulA
zOSXS403YlKPSLknx4HsR5Jusl2kHzG235PeknpG1oPy2ZOek3pFyjlJB9aH
mNRrUi9JPSHpIutFf0jaHVKvmXY8m/KrzDiQtHdkXOVBxBUePfKF7+OVtF9I
N2mHmPUksa5M+uMp+SHtK2kP8XnkD8uZcWIE0d9JRFwVQdj7+B7HgaSTyWca
iy+87/i/8QuzvJDCTP+RRsTtSURcm0TxQ8oVs71AFr3M5wtZdpq0k1iOGPlg
xlO0HcRy1Gfm+1yqfmY8lU3EU9lUPaQ/QLpJu860vyWsK9LPfL+aNe8l7S5p
x5j9wyXWRXisfv3vdkuoeJWkk1xnIP0I1of0Mt+rJdZleBRmjgtND2Jsj6wf
McbHpByQ/otcPyHpJv0j6d/IdQXSz5F+A8sRk+s5ZNxCjhvpb7GcuZ5HywHZ
L+R6COmfmc9zWXww/TGX5QfxfbRPpH8n/TO5vkPyS+oh6a/JcUZM9jPJF/M9
Lus+2jO8j/qPGPkh13dIf0/aRTL+IMuZdr+QpRek3JD8IWauG9BxBenHmXZA
xPLD5HoGGb+Q8Qfpt0i7Sa5HkPEPlmO9JN14n7wi/eR6DDkvItcryLiBXL8g
+WOuT/BY7eLz+BxJJ7l+gu+Tfpzpn0WEnIgI/4rxGb7nS12Z9yOIOCiNFT9h
OdKD7zHfp+MIJi5kxT9YztSLeCKuJePGNELO0oj6con1k2xifYS2m/g80kf6
X9LPkXYWMeo56d+ZfrGasMNfWf6DtLvkOi3pn0g/T65bk/aZtHPkuiBpj8n6
STrI9WoybiX1mFx3I+0duR6K7+F9Ut9JvSLtGjMOi6Xk/u/zfJY+4HP/kh9a
QOFV/9LdyCr/2w8pFP632kk5/9u/1RTGKz73lz4c31LWe3gfMT5PlpP3EQf/
25CAwtju336ro8oRk/2C5YiRT/I+YrKf/rYrZNkZfO74vwUZ/ytXn+Hfx7uL
KIxXfA7fR/xXLsuo95B/xHjF57AfEP+Vc16P9eB4/JVj3F/LY9WD5WQ7WB/2
L7b3tx+5VP8g/ltPKNWvOL4knWQ9iLFf/9JfT/U7s50UCmM/4nhh/2I51kf2
O/kcOa7IB17J8SbpwueQDsToZ/B9lAfE5H3kj5QfvI/jwbQDeSz5Y/p9PlWO
co4Y+xnrR/qZ/tCXJedo17A+LEf9RLrwivdx/FEucfwRM/U4jyrH99HPIUZ+
EZP3Uf5IfcL7KPd4HzEpt6RdJPWNef6Anh+iHjLlnraXeJ/Ue5IOUj+ZcXEt
VY5XvI/0k/YA+7WnfibLcVyY/ULaoVoWXUw7Ucu6j/UiJu+jfjH9Qz3L/jD3
0/NY8yusD/kj+4W0R+R8h9QjEpN6hJjUO7xiOV6ZfkxE1Y92pCc7g3qH72Nc
wvSbQlb7eJ/0c+R9pj2n5w9k+8x5AJ9lX8l5AtO+FlL2DPUIx4XUI2b8Svsz
Mv5Defnb//nU+P99L4k1Xsx1u1hq3Jn20JfCWI7yhOOMfCN/aCexHPkl5YaU
EzJOwXpIO9iTnjLl/CsrLiDjMtJ+kXaaeY6mlKXniPF5vM+0lxxW3Mj0CxwW
Hcz+4FJ8kHqKGMcfMY4nyiVi1BOm/eYR8xoeS6/Qr5Lyw4z3fVlyg/KP+kLq
BanfKEfM+SN97hH1EPuX5AvLSb7/XpsJ/95ElTPPLSQRek/v+2M5vo/9gH4A
6cT9ftTDv+NVQY07yg2WozxiOXN86X0yHCd8j3n+rZbCzHU7mg7yHA2+h/3F
9J9fqXpQ/pn+iW4H+cNyHDfmPN+XmG80E+VNVDn2M7kvQo4DWc9/l0tIoJ35
S5cjNW6Ime9LSKCcYznShZiM05nrSHjOUUg9z7SfjsT8JJTl97B+xMx+ocuZ
8XATEbfjOYskFt9ID/YLOc9APcRyrJ+09/g8c38sjbIvqPcoP4jRzjHtJnte
gxjlC+39XzqqiLinmlWO+kT6TZRzxFjOtNOVxL5OFUsPSLtMxs+kncVybBf5
wvYRY33oj7A9xEw/wO4Xpn9m9xO+j+VYP/k82S+ImfJK+0eSbpLOnvjCdnui
C/WLeR5SSPUrygtzPi0hQdqfnvT6/6V3KP9MvZOQYK6rOrL0iown/1961JPe
Me1NE+HvfAk6fVl+mWlHyXjOg2oX7fZfeul4Dvkj6Ua7zHw/npIf5IspT0mE
PNH7zeQ8AsuZ8U0Ky08w6UljxWUoT6S+Yrvox5AulKO/9Rey4kAyLmfuu9Hz
VnLfjlwHwnJsl7QHzHMqtVQ587wv7XcR/x2HXFZcgRj5QozPk/Er0o3jR/KB
dDDjbDpeJeMY5ry3hOoHrI+5b1dPPc+cN3KJ+SS9X8eMP0g/HUGsG0ew7AG5
fsb0+/Gs9VJsl/SjKJekXyP3r8h4mdzPYuppPMuekPERaU9IP/rf81ARK45g
6nETocdNLLv4l09cX/cn4j8RdR+fY8bXvsS+zWdiXOJZ/YyYuS7KnnchRn1n
6gWX5edQL1AOUW+Yfoo+V4f6jveRDiZ9XIoPMs4j90/Rr+B4oVyiXWPKKY/w
Q3yqnDlvcwRmOXneypGw+0nEuHymMPKBz+F7+BzTX8QTfi+NklMcN5Q7Zjwe
T8z34wn644n90VBi/hbK0htSrrF/kG6cN+A44PPMde8mFh+IkQ/muMUT895K
qn600yifPdltxMgvue7EbC+XsqtYL9phxCjfzPNqudRzTH0oJOz+V2IeWUrM
Z0tZfgUxuU7JXJei921Jf8jcpylkrQOTfoj0I4ix33EcyLgA+wEx85xVJdU/
WM6ku4rSR+Z8nEeVM/eX6ffR7jDXxTlAzutRPnqKA5jnpbisOJ+5Hs8h5vU1
1HNID/M7LHr/mrm/XEfYO/o7JPQP5PdczPXrSqrfkB/sN8TM76d8Wd8PMeOo
WmJ+UEvEFxxiftBMtY/6zYwP2HE2eb6Fef6DT/l7xOT+NvP7JiHl55jnj+j9
dfQjaM+wXxGT++1Y7185C6b8LdojMk4m19uwXXJdnrl/LyLiokDCf2UTdjSN
opcZR7C/U0A9xPFDjP2HzzPjpFzWPg3T/5ZQ/Y6Y3KdFe4NySa5zMe1cDaHv
PEKOyXiXTUdP5+LJ7+fIfsPx6akfEKN9I889oNwz/W8za12OGR9FEHoYQcyP
Qln72H/p8aD861/sSJw3DqXoRPqZcpNH+TfmeZ0kYl6SRtkppr+tZa07M/0g
fW4b60GM76HdJ9exMP4j/QIpX1jeE92IyXUxkg/s957GpadxINehyHUbxEy7
xKPsEL6P40nOl8jvF8h1T1IemXaPy1rXYZ4r5VHPk+s8KFdol9G+IWbKnQcR
P+J3Ta//9/kEqr97iuOYdi22x7iVjGtx3JF/5vO5hD3MI+RZxKIL6+kpriPL
kW5cR8BylFcSM+mqZo0b1ofjgOV4H9vFuKunOA7pwXJsH8tRDrE+pv2l40zS
fpPxLxlXk3E3SQfzuyZ2HELGKWS/knaBeT6wpziMLifPpzPPDdDPMfWDHecw
/RL5vRavx7iNjOuYesqnypnnAun1GfJ7cea5UHIeSs5TPVh2k3keupS1T/X/
L+6iy8lzvSQ9+B7aD3J+TtZDnqPqKW7C9sh9brR3PcV1ZNxHnldGf03OU5n+
hD4Hjs//3+PcJsJPiIj1VhFhHz0IexxB+SO8j5hppyOI8iTKzjHpjaXkCfnG
8SLpIP0maZf+vpZP2Vu8jxjtCj7H1OMS1voweZ6XPOfDjFt6nmcxv9Mgv7uo
pexjT/NK5vlkHqsfmXJKn8PpaR7CPLcvZMkp9g/2O9pXxGgP0R8zx5H8DimJ
wuj/0F/geDK/j0qiMD6PdpQ5/jyqPaZdyqbsFtO/8Sg+sB9RLpj9Vk58P1JO
6QtzX0tA0cmcf1RTzzPXeTyocUdMzn+Z45tN0Y39gJhpX2h7Sdo/5nkxEdUe
2i+yPeQD+5l5jiefqgf7ETH2M9aLmIyzyXUOlHPkg+l32f2EmDl+vqx+Is9F
M+0YW66Rj/87n2y5RMzsr2qWXDL7h/7eqSe5w3p7kjPm+Rcyf0Q2NW5MP8HW
Q8TkuW3km9mveazxIeUKMT5HygPadea8wxHIOAL9CdPvRRDzRR5RnkfRj+NI
2hmUN+wnUv4Qk+2gPSfXo5j9Uc9ab0FMzuuZ+0K0XyL1g7lO+JWql7lPWU/E
Dc2UXf9bfxOlJ0y73cSKo0m7hZh8D/sPMY4bOc7Mfs6n/Dzyjc8x10m+UnyR
9pZ5LqmeGi+sn7lflkf1P8o/U//jCbpiWf4Ay8nv+rB9tGvkPiUz3qa/8yDt
KdkOc70vj4hH6TiT2Y8RRH/HUhj5/HsNJOIjX6odlBtmeRNBRyVFL8oL8oPr
Zkx7Vk2VM+1FEoXRDqFd/G97Rb/HjIdSCP1MouzWXz4+U/RjPUz7yqXoY+4v
iQj9bKL0C/H6c2eClJ3o87QWB4+udxtC53Fce/pBW3MkvR/X57tK3115dJ7D
3Sed5zWNZOfLLFIIsf4yLhiu7NBtnnyR3N9Kg8v7t5W2zU2AuA8vR9itYue7
vL3qRe4LM3r/69IbfqTx/XgYV5i4buoOeh0Z+8mvTUcoOJUGw3ZGjOsfxc4f
adu1ZHr6jXRqfEcPl/o0UDID/siv4I3j0vtX8mcDvcoH03nPkJ+L8x9P1XpJ
55GKUgsfu3RQBkXvJjXFZVlRdJ60bBmXYG0OvR/19kmbU+jJIlgV80PiRhx9
nndhe5TMshg+hZG+47byXU0X6f0hqdwL/8TPK4OtcxPcbifS532nBtuFvFpF
76u02ZX+SC/nUxjpkyhY0nuPdw10bL7RvuwDvX+Tz7fYM8qdnldvV9s68Ol+
+rw+tn9mvYf5RXcO5PVSW9lXmZ2f7WI/s851j+l8ejsVJ0X13cQDnaFTfVSt
/Gm//q5ASqlfHIUbuRr7p49OZuVXyz3gpr54LI+6P9Z7lmV0WR1UvD6SEjeb
nR8tMV2vyL+N/j5wv93pt9/X0+vTV1Y1Tw5ZJADnJ4LN/3xMpeiWrvRMO21P
njsSx6dOqdZno4WwIjns2g+jHFaeLsEepeGtpfT5iMHxK4Oe6tD7FSMqSqW/
NjeChNXO+DvnYqHEuZZTb+gHacZfRjSG5IDoz23Nu20f4EZBwcM0IR+cNfJ4
oiv+EOttnXf7lh9YdUhqzLseCzuvrzAfrZoDqoIla4f0jYV7px9NiSkuhOWz
oyyTteLhZ7VKZJRUPejvW7srtyQOJrsfi7u7uxFslihWNwpjwTr0qb/F6lwI
KB00ar2pWO+KogfFivVj+70Z/DPTc2Htl5iZaoNqYJDp/JQFWTnQ1OxXsm0r
B6adtulYWZADpdxnYcpbG8Fi0dDJIbdzoOPIOlfHlhyYEFQT+/VqIexo6Twc
P4UDBgNW7vdxLoWWCXoHDT7yQT3/44iZj0pB83q3/+rOHOBvfuU+X7kGpiuY
v1LT5MCzX5NlpVur4dHYl7usxH658sPnNwPDquGOysMB+YalcFHCK8JIkgO6
j5Ulk76J49Rw00eCxRwoygqY9+GYECbuLby03IEDisNu5LkpFAI/+p+V9hfq
IG+D9OmxYrlYrhZX3VZaB5n5XeeOQQCsXd2usl2yHtR+rD7Yf0QNcPZMGmvZ
zAfDF+a+bgM4cIrfO5vnwodHOzovHxHLl+TC4Ebtej7MHWYbHyF+L2lu6qaq
NULYnit8G7YvFsrtK+fYLGuESpv2JaMWi/VxgMOKzbMaIaGP0ufAQUlwo1fy
PQtFN9A6OSxiSXgyRE8S9C5xeQ71M7XtAxrqYe7p7548yWdw81tsjIWXAIQ+
Ty09pZ7BndqXL7SS66FqbkbknENvQc7I2er6tQbwl0ytGWDtDj7fzVb0cWmA
CRfvBZ+McodVd8rnqN7mw9NNtkut5cJA0cWp7fITHnwr/yIfFhANBua75L7l
10Oc3KiGMelh4PTuV7/FWfUQf6bX/IHLYmDZqLD8KRwReOx1aChXjYCtY7ZZ
T7Bzg3KdH+43RifBhguKzjuHuYG56rQpvr7JMDkHnu0a1QA+BiqckQlf4NsS
x0kHT4rg9etQM8MFieC/eCMnfEETZC3qkFh0VIxnzPl8Z5sIUiwu2n6r+QLP
Bry/NWxNMXiIZO061maCtaX83k25tWDRdyf3plkmNO/bfb+VXwMFXopPLw3I
gaMFJ7aajxX3c+ZN9X15GaDJmTw2ZWoDJKYUDpHIyYbw1W9eLMgQwTDTWZ6q
yVmwAKZtMZKqgG3ccUszNPPhTMa687mpVXA5RmCfUpEHm/6Jjqt5XQ0DB+6e
4303D67c0mpfnV4FAVs3xX5dUQC552/aCRtqQDYn5XSgQx5sfe634bdSNRz2
nr/DZGIJrNiUc8ZKuRo2D0lrTJhcAu7R3vbP9/Oh14i8JUbbi0HO71uQ5gUe
jF24vHqYWimYGYtWr1oojt9TjXUrhlVA/rkxA9xMa8HKaKfs4E/lMNjV4/pT
HS6823HPfXNxGZw7v65u+qI62Pjc7uLiwDJ4nVcw3nEXF34Lzg04fqgC5l79
qT68hQfFte5ZvVzKYcMGlVktYj9TKX88sy6jCrhd2tc2LCuAwwvSng7OqgIT
g/x9b5RLIDI4PaK/SjXYDypqDnnAhb0J2umm3VWw/m7znmeVXNhXJdiU314F
HjeTOtdHZ4DWtkpj/fIa+D7KcEhadiY8by4ZefVTLUh4910xpisPVh9UPfpC
uRZUVcb66/bOh+LV3o9CQ2pBV+mWsnJwOegaG4Uai/m8+vjgjeaQclhbMmLt
jOO1oH/8Q76jUzEcNfum+rWLC0Uyv2oGfC4DjzNbvE8u48LMB0cyOXsq4I7N
zqgte7ggTM785i2ma6jMHvmWu1xodN7w1aG1CpJzb3wtq+WCxuNNr+ZxokB2
Uu1k9QAexI9bk916pxjiFpkuzdDnw6xR25SG3SqH3ePWGzvX82BNrvalsN3P
oPSjfF+HJrG/kXqb3c/4GTge5L+q8xBA1v1jO084fIIPlTc8LgwWAM+8pr/b
6CgwvBv+7s1wAZS85y+YOLwJzGqXyFjICCD8oVGKq6s7jFCX9La53gBn1Qtf
Tcxzh/3njmz0edYAkakjDPYEJEN1gCSk72iAkrBFgS+eZ8N8M0t9K60GuHvL
x/nuMx4E5C2JKZzeABlLg/eHzI6ANtdAbkaNCK4lTn3PlUkEH8kPwQOPiMDv
wC7TA6cT4dOH8ymz5on17MEB/6ZXWTA194eHz2cRKJ35OW/ECAGoKM4abj2U
PkdkN0e7a8lDT6jMUNV3zqTzHPAcCpUyI7KgtUrUr2GbJzXPcbbgGD7Y8gou
ZYxeqTKPzn+F89uCQ19lGo1eUX601bLTe9+gt5R/dNquMH7NUV94r1hueC2m
Ac6MGWlbIBkO2npDNLbrkvmwPMB23AvFHd/iQftM1s75vRIhzuvaXRWlGLBc
/kmRLxEDR7znyReI++H4tT03Wp9nsfa7bD/Ib06Qj4cLz849MrufD/uHeD8N
LUiChKiIMS4X+eB+pkG/V/1nWO0T7FQbJIBC45l+FuqJcOnrwqhiJQF0P1Ac
+Px0MuRlnza1/0mfU3SbYp64aDd9XnjC+n1GJu2eUH760xCLhzQdU51em/jJ
pMCPJ3nrLAfGUXE8xilqYc4dFd3JEHRj+dT0Cj5VTvXf+2K+jWsyuFztdejI
dPpc5uaaPn4xiWL/ICXz6OHgTKrfNvWWi9R8lwMTc5o0lNK9qH7Y3Wm8bcjg
ODoP27ZT73ZZ0ftrQyZsUEoVz0O1b84/Uh6WAdvu8WssLpXDcIPMW0cWiu3p
6PJD61b+H3kfYvO6plbS+22jDkbLJ2lwwVvQ7DetORs2vlo8dDufC1vCblXr
zsqHpkerp5zazoVx+cc/jNYtgBvrZiy8cJ6dr6p7f7plyKMcuKJbo2Ybzs5/
2NHoO03rRyHobxu8IHJPMUy84jN//OYCcHsUm+OwuwAyV/5y0TMqhvl2FX9g
MTsvk8e5rRoicVyg3Dp1+/hyer/O/Ovo5qR2Os+E+5krHjK6tRAu2x5eZlYC
b2/19xFc50H6HLvcU1cKqPmWom76+JNShaBYdUDK6jd93hr1YVeYhazweyEM
n7bpnWd5JpxprtLyUykHw/lRF7SVs2Gs1+9vjx/SdBZcldRfK1sKD8c4q7jM
os+5Ydz+xVV6y+vIEniq/rR20Cj6e2XngD/57ybS3z3h+k26W9bCuiw6L8/1
9JXquy3pfFOW/eNhwdAKKBnP++fRbLqfqHNPMu9041bS+4ehl7f2fre8AmQ/
/U6MPJdB0YXj03t11/2sHfS5uQ8rtzisC6wGxezv+23XlUDzSHuDGCM6D8i+
aIfeVXur4X1CXpVUC52fyrLk6eK0Wh7kfLw9JFmTzpM006VPPl9c/4eoffuS
LtLnu5tvV77UciuAfosdZlWoc8HmbNel3hb5oPHZJvZSMheurrzdbBpD94vr
6Y2Vu1zoc3fYrqz5oEMhdRwQafjuN9/Mha0brTgTs2pBvrxPoHF+LaiHjH+y
YB2dT8psY5LPOi0O9Em5ET1ORwBVl389Kwimz+l2L15ZtqdI7L9MYlblDqTP
7SZ6LszpVU3nj8JxVJ24NtN1OQ/MZ0penzu/Ggr3T4zfKeLBZom1d3bNps+9
9Vm8/EFohhdFh+toydYbl11Z3xGOe1zaZ1EKH9KnpmklJybC4+HXfoZLCqCr
8HVAZn0SaOg93nhnpQDuz0vQlrai9ytRjj49sM3MdudD2wnJKpNVdL5JHPeO
LTUhM2TqYePykRMOrqTHSWLA26MV19j5njzceGOvjOLD2E3h0wzP0/uQCWdG
nFW8K4JBAxbHvhf762aZB7AnVQSft+udHSTDB6sto2ps5V1Y+Z7W88wHDDcV
wGF1yI3bGEetz+TvuDtR/0Q47NuU6SIf2AAOWr9LggzZef84c8xvD/UVgr/G
bftTK7NY+6VW932c6yJpOl9r+hyJFONV+oZrIxQawFTpzuHAGAHUPjoZfitC
HCcUHe6oGd8AyZEKmulJ7P1T+0HzzbatoeeHfQafvmNX3wBPJGyOt93yp9Zh
0p7sdq0N+sg6Hy/ontXyUiSex7151KDwNZ21n7q7Qfh4kqcIkj3WRyor8KEx
9eYf3ViajojrLX3f8RugbPe722Fq9L4Y9udNyb7f9FXpc8KtD0bdKBhN768m
vGiYP7t/E+xcWNzmrUXzpelUMedbHxGUR8jHN/h/AO2Lfz4ejP8Ai7eaO5VI
0vk4N1ptc03f6A+Wwz4evqieQPE1LP5Y9ovwj6z9VXubtTkTVyZQ4+XvdGdV
jFocNR75UWvHDxf7d838lMTKUfR3DigHNxt/7zPekAARN37U8e++oPiavaFE
U0pCQKzPCWBG+W/jrKvp4J7yy0htFZ2vEe1EAsxtGjw4GwwnJZ+Rtmfnc1p2
yN/pmVcOZNgNq5Xzp/dr8bqhex9vhQu9H1vQd+7+QUqFYJL+Zt6cmXQ+RKlX
Z6YYRNC/l4HtbLbOiZljSefbXboz/LruBNr+zj/gVn5ei/ZjCu/3jbx8nwNy
ndyM3yJ6n9Zp2aXS4l/s/duQltlms/m0nRkZffbeT1UOdN601Fm6ns6/hPXz
DF+Wyf+pA+2OcSb726qIfax42KC+c5ynGh/k4rc9OjAojti/S4L+jQZ5ySPq
gfuxl/BxOL3PX6b/JbDKmx6X3NIdauuC6X3dst0bvJ0i2Pu9/UY2+ipF0XJd
1Z7DrZojhPyun+6VU3msvE+NvWXKlon15c+kuSPnD+HDhtO3+9rP8IQXpd6v
lorn8QujypX9FovnwxLD0zc9FvP7tazvtdokkB0ya9Tl2dXg1CrnVqIg9uOP
dIyON9XCxj+fanWOlsKyhJ36aTOygXcqT9S6uBqO6a9/qKdaAiWixcdF4vmW
s84iBU5UKfBTNHwuXKyGxMjrB88Yl8CG+0/sZqvVgNpcxTU8/VKYdU7e062r
FsqvLdUs9CqArYb9pr5ZXwcHOgq2OeoJYHmQ1jzdj1wwKg/MPhDiCbdc6oZ/
lufDywDhVLlvyXBr3uBZ9+4JwHnF1u+tw1Pghe3aNU6VAhhY7DCm/W0MrDyb
y509qxEq1uycOrVFCOX8pysn6b8DLfmVjh+UGmCyQkz3h6d+YG7xOsJgiQhs
0m8Zb18QC/pD+khYThZBqUGRmZ0gGhrGz9JtME4Hz7vSEsbzxfq2ZGNv68pC
cI1dOFVrfTyUzyh6dyk9HibHtP8c9i0Npj3tq+zR1AjFPt7+OevSIUU28KtQ
pgy8UsaHeVwuAmkbGYUdGdWQEKrn1ZVbBDd5H5NmfquGhaP6BL46XQwrb00u
mHarCI7kffx9SrEMXF8Z+9nIc4DXXZaic6AMar53vvZLrAWB+r2Tr8aWQVvK
foX9o3hgcsLmlSfvK5T85n1NTyyCyxXx3T/yqkHLdf6YNMdiuDc194+mhDiu
Ktl//+SDHJivF/Gw81UN7Oi/+daOvTkA3qeKNvypgVKnJT8mHxf792HPvidf
5sAM0yt+Uv3L4Z8d+1oGu9eAUXV2m7PnV3CZGJn+eUMtdK44LyG/tQxkJAc5
jh7Lgcp326UvdX+FsgDOSefQWugqzxhlLlEED+bVXGv4UgfZH+tvawvqQMPp
5e/x7hwwC3JQGnJCCGErQ469fMiBBU8aej+PCIYW5WFLtAr5UDlv6fQTKh9B
de+4jTcj+OBw7GZxn9BSMH5o8HHVAh5c+jBqvfHAUlh3sSA8Zy0PZmwKi3T8
9BV2LbV5s70XD46/vjfeMUpsz9KKAkL7ieOJlNWWa3LjoK/P/nWrV9TDNRdJ
LZPX7yDI3f6IZYMQ3m15tbT0wHtoPzJ8/pVZDcDdbhZd8SsVnmT41ZjeFoKa
eczw76/TQd7MY/edw42w7ax9UKlMBiy3ejpmQ3sDrEsK2NNrugii3Q9nlVUL
YZoVd9u0tbGwMimr/dk8EYz0328osouGh+HzdniMFkFUhezhhy1psF+lykG2
uBGqJC77z64QwkmVLyuMVEWQuN1QU3NGHMwIXTq/8UowPFvd+WSFWTDckz1x
f8SCOBj37eP1Poni/vvl8eOBZAgYrb66/e2oBhBN+1Q6Rj4UHsZvDDTZ3Qhr
hXvM/vQJob8nap49uMAjFiSCOow7o1PBbEq4bEVrPHx+qh/1PSgefFYc8+f1
TgMjq3W/NH/R+x9+ZxP0+Yr10H4v43r6oXiIuFV4MUvsj68Zel7b+zEeZLWG
rVh2rx7ef49ulU5OgTKnUq1VQnaelpLf01a9FUTBm8ZhNX0L6H0ScB1UMeig
ELQntmTvupMGijOe7OKtF8LgAq9B8Q5ZoHxk1JQXnwtBZeqgid9350Kv7XqG
q6/lgs6C0t+6Lwohon3k4ZOeHNgTs2xg7uRc8NW89+ihNweGNbVWO4nt1swz
Fu2KVnXgPnzsgKTBefR+iuPFBM6NQjDgajbvGPQVtEDKI3lpEch/PrAmU6sI
zpvUrtzS9ytsvVBu4BtYA1oKfdQuvC+Cgt7KQ8pSayCuwHy98sRiGBUpuczQ
pwbSr7dd2+NXCgYT9vx53FAH16SyZyl8KoQzUn3uh9SL461LtfnF7wtBVLD8
fdFFOl/TqW3uQ9dOpvPI/r1dA/2vX19mJp7/Jelc2z6pvRpOmP9cHTaAznMk
bxBVtS+oBM4vXbpHN5/O5+RyPdHicb8yUF86Njm4oxr6dIfcWPe7CBaazW8f
UVwDrkPzPsX3LgYe99u2QaU1oBm7eEKgRylUvKhpev2+BkQK5glq59jf9QR2
BriqG4vtR1W7museDmy9W3k9x0rcD4ZDyjdZ18D3sRklo3Zx4JCt2r041Rz4
5Pzi9R8lLkiHDH5htjIHvJVH/+iey4UZV1wtv0vlQXPrksfq4nmS/p7vfgE7
8mBYVHWl3Ewu7DXbNDszvRBWbD/le/sDF87pjvbhv8wHg5SCuqfPOLD5ht2y
dSvyIG+JRnXRrjr4OVT7aPd5Ol8w9sOGqEvvdTpq4fUJeHwzpJwoF/fviMop
Tss5cExodaJfdh2YpX0/5v+cA9s5spLb/TlgnWtcvD2qDgZF1ybFnqT3iy7v
VJHbtjGf2BfNgwF6toUT59P5nr7PuFtUEFMHt1NjA6XDC0A/hr9wvCkP3n3M
GCnxuACsVKzUdXfywDt0udvdCA4RbwRC6xDJ5GVCPtTfnhSQcP4zWHw0s532
lA8jz08YYx2cCLsLfk4sLOXDmd+Jme9tP4NR01TJ3uI4QnbofP+ELSmw6GLL
cMO39dAECzyWWX+GXuonn/hXCaDGcv0FqRVC0MjT9NFcVw+bZnGHr4J6SB/t
N0xyo3i+kCK5p2t4KBwauHKELE8IcStNcw7bh0Inx8/1pXQDxG99d/r0lBDg
nj5q8c24EeaM0/2z+UUGSJaHDjesFsCFt1POvXFKh3PVZz9VOdJ5azYn3b8e
rScEzs+N2zxONIKJa3ZTkzgOOrYUXBcHCqFkM/ed0clGuDw57UzjqU+s72Nt
tj+IOzmWtiszXSavv5IjtnvKlt+PyMVQdkbh8/daqY3pVHyUrRUUqVfEzv8k
vW16dYbXR0h4uTjs7Qp2PqjE/hacjrn07771a9Fx4ydUw9M+rklu+uz8UNNX
V81dokLjPm3lejYramHrtd6e5v2LKb0e+Zq7bpMN/ftaGPdtSCi6WnmVjndh
faPlxkIuCI51boh8SH/3ivHm+MjNg41XCODcUJ/UJ7/ofMYnbuU9G36oEbbM
vZTtvvIN3Csb2O4Y0QBzb16sKOjyhwMVblPefqfzz2K/dVddMVE+EwQlTXMn
xEYUQIfFvZuyrp8hIKb50JmgAngn02/05duf4ZNiW43XbRGcfrfvn33JwVB0
XF1bpaoRrBXcVQ8rh8Ooqan+Qbl0PirtKu9yCTUBxM1L3R8UEg+9JRWnpT+p
h7S3nbPthyTC3ZMZkx9uEAL30o6bHTrJoPdI89SuSAE0f79vlPWSzi9wrm3g
y5MSqXDTpCjz4FoubBsjlT14TB7MqBlotjOSA8NHvng4QSMfclraR/4+WQdC
jfV7Z+8rgNOn1UObqrjgtYSvsvJ6Idw30fX5MKUG9MdMUrh4oARmfFrhV3K+
GmSP/DN8ShydT2CQVkduwXpxfNVi1rLcQBy3apj5vXz/FSz1D/069qIGXt09
77VGuhxypswzqvEsYeUluKek0BJwowZuHhI9MwjJgV0jXIw8wmpBau0Gn4vz
smG8Un3WrbMcCDMawx9uRM/L/r1MyoET6YsKj0/gwq6rgkP2TvlgP73uyrQH
HGg9OcNc91E+Md/5Co+PHcgfolYLVeEpCq4iOu+B44XoRRPKBTD5Zf907clc
uLTPdWd7sQBkb3Cbxovtc7P3nD9vywshK0o6PTKEC8pczyXB1fT30iEHNRrl
S+LA8sOKI5KK4nisadnOY2eiQP224OOFWjqvU87FR31udPEBClLWRTgUw+J0
qTnbVHgQHTHTRDCsCP4Ub/FdzuXBJu9eZyxM2fnqzm+96lHLEYB2nHBrsXIA
3PkZ5dsS2gD7Tk3bVbPtDVxKanPIOkB/n39+U3VhUXAjKK/4adxyPwvWHfTX
Dr8phPl5mxVlTqbDoBVbnWavb4C7G4d5HF0dDgXXa35c/doIf9IWnbPLCYYv
W8eW7rlFn//6Nv7AtZ1PfOh8uDc1X552CAGpxR2y6wpzKLuzWi5LvlEtB/q2
bU5xKoyBt/a3z8T/zoZlI2d7vdgcC0W3LrX+ekjnNbD/ZHBdMYFex7Ls/GNq
FsmHYzvDI8vfR0PtoY5Tp/vwoeOqUfyGBfFQlTPun+g6Op/H5915e2bxRbBG
Z+acz3kR8Cmq892siU1wgxPzSXF2HMSLWo99M6DzZXX7+ajeOkifo0J7Vz46
pebu9y8Ufp7fV2mSfwaYLA/LGrKkiXp+1/6q+KoFTbBpdVPdIoMvcMdrTuBG
1SawKxokGqeRSvnRKw5D2uxH0euWZk9maqaZ5MELe+c+UZ307/lsbJ+eUZpM
n/tatsvx0OReFfAzzWycU1k+JHcHiJa50r+TY6yw6EE/Hzof15uchbMKHlYD
Jzm8OdO8AKJ1T+fxnlZDatcSj/iWAlg9t363syH9nV9UWJtd5Ck6bz36ad2U
u+UtfqUUPi7hqnQ1mT5/qHNIVB3Qt4LC40uu7QxLq6L4G9ZYnhn0if5dnC0D
vA/omVdQ+H1koPXEXjQfJrJVpTK+NLao/z5w3Pca2GNUMrbTn0/Vu1l7SaWm
KR/+afcOby2pBYWZo/gJo2k6l34FvTKfMgqbzgmYHvexBLqmd1a0SNRBlMEP
7QeHaTq2p0sNVe2sorBGrvK+2fsroGCLGd9+cB24OnO//M+699gh2gP6Pq6D
iz+Fk15cfkvJvZZC10/98R8pbJQ4NET7hA+43rN0so8QgFNqzVbjtz5gpdjn
smerAM7bzZyjPriJev578s1TikObIONS69q8i3RemEdLNW3lO+jf03mjGTC4
31Zazme03Q/6xaHldu2u07LHF9D1ts3VWek7gJZTTwdD59/b3OBcUZ81GS6p
MGStvk9HbzfqeVHeyAGT3Z7C2NSL2ufM6+Hw2tufbo5wgU2N7/t5jhbC41Gf
7mQOdYYvQ0c92yaOk9w7bB4883pL6UdbtaxI0ckdyvo55JwLEkLR6tUm0qUu
YKqbPSZmZC5MeTDDPHxIEPT+tHqJVUsOuEiZXh4wPBy6z91sSLzFB/cGp18b
X4TCx+hPwWfnitvj8pPHDg8Gc6+yU7PtBPBwOtft4ehgaP5yX9mDUw9DQ/cV
jQmLhednozTao9zgrWLQuJlBqaBX21vH1boR5PqOnjlRLQFuGyvfybjYCPOP
3lWf/z0Rzsuq6i3KFcKisliLs2ZfIHda5jdtMxEoql96ndzrMyxZ4HzbdlYT
vNLT5p56lgBzm3kHX28WQVHsC5dnW+h8Yac8nG8MXJoBw1qe25wX+48TmoMf
zDJNhw63iQ15KSWgnX1o4uKP6VCjF5A0L6MG3NdkH1aSyoHph0wP3xbbXaVz
h4cOKE+DdcGHV0+a1gDK0nJr9IZkQPS4o4bTxfHzUZMh6mqiLDiyd6btiBmN
8Fuh1ICfnwHewp+SRf0qIHPujSkTnuXC5CmvdxeXVEFR28oDTytyYay78J+F
b6th2Yqifeab8mD+B/vIlsIqcHzz4rmaZiGExfg83ivBgSMLxrz/ZpoHY84m
P5rdXQuyXh39FMU4zLL0zS6xHz6a9vH27x2FEGe5f9eXKzVgJ/wS6dC3AHQ5
LVOui+Oquzbvchd3F8FpP9uOb2L8RcYi37lXMeg6Ky+2y+HAPTl9/xVNpbDs
lov7AmsuXJBboHXiTynUJ8b02jmCB+PGvVqzxrAIbIKdB3DbeRDU57WnZ20h
uEprfbPX58GFrZvzd2sXw/uSJ2miCzxYdPT5SacDpZCqPmttgFcuvOK5Tj08
vAKeSdS8d+vHBRvTkdvUM8ugcpdcZvLmOoiLmuJ637cM5s1QD+BXcsDBvcV+
zIYKiK892/gzLxf8Z991+Z99zH3xxoryKwvhxVLOhYPifuSGW/yjtz8PDpUE
xz+IqAZu78dXVMzzwKKXfpzsx2p4wo0Qngiqg2nSfS85dVTBCa/5lk+VCiHk
hsKc+141YP7NQtJYrwDC2j72minGvyOuKPzg8KHyjyAmXtyvjQNfn/2Wwwcb
9aYB8gq18FD05Vl1cCno+O1PDBKJ+23IvQ6vkFJY1V+nK9eYCym3vfrOiiqD
n83OW8cP5UKM/QKpOQYVkOGW++10OQdmGjw+X3ajDG4r6fBKDepgdOcR69L7
YtwVrNGoUwcz7vwj1P9VBc8O8beq+tHfqf6psf1VJMeD030tTCO7hLAuskLn
lHi+1alZcHBmX/G8wfLp6ZuHeTDrfJD8U4EQfJaGfx5nx4Nisz8BU9waYfXX
Jycn/KgDwSXZbLeoBrD23L43oR8POG2VCooRvqAf9mF038B6WG9hezD3kq84
3jw8uTGsHpaGPr5we6MzrChVEcUqCOGw80jh514hMI6/MdtDXgjpG95/KVMJ
AaPUEc8iXgngvlfIGL7YPk5I7Xo3aFs9WO/2jz7tlQD+g57vTd7dCM8KvV1k
zBLBc57goPb2RpC91yR5cuJn2JxnYWx8QASpSsufCRekwAmJFRprtUXw1nP3
Qf0JKVDPf1Iiu1oETY8P63Q4JICBnvKrfTObIG/hxo9zN9bDCjnpycOk6f2P
hh9gf+7teyhbUtZleZPOq7Bpi+OkGYECyNUsC31e4g2/TpnI3+sWQPiGsbX+
032gUjI22zeOzmc2YHbWvfFhdFymba2xc7IfO29S3rtlv0+e86LOeXuesL8p
qUPnN/syYpB8xuNw+EfFNPucN53vDJ+/0Nwdue53LISVBIa1HssDieL3veQm
psA8mVchE8Ty3LnqQeeBqSmw8PJ7Pcsb9O+B4fzcyCvn7O9RKVDZHpxz7A1d
/reZAgjhrnxzKuYz5a/Pln9brt47GWL6Ny4xtKLzp42KymtTlKD9GsaxQ1+e
XZ7/MJGarxRNixzZvygRPki4nt06gp3n4eaw+32mlKZQ84aip8Pkw4/kQGJI
is/qV3S+tLdqr2bf5FRBy9yy2svb6Xxps8JOt73Ky4eDrxLtZj6m86VJlIQq
rWql87mApF7WqsvsvCYgN3+MdB09T7j8+9iek72+wsOg1h9K+uw8tCedTN8d
DyuFivJleQqcfJhsvlfxvVQFSEXtubzpKP1dK9ZXanHbKmlZOaT6cnIMT9TC
msRXxoPDy+EaV2aI5gYOeHuEOTfOrwCNl7eWugyrA5kDuwZuP1oBjpI7M4Ic
6iDgQ5Cm4Rr6e+H83c5P+0rVwCfZjR/GXCsA2darQ9qfV8Pwcalr46oLYPTi
mdaH3auh8c76bec72fnUHujGjqnexYGy+RYCXhWdHzjDXerx2lJ2PrUbi4zy
VtjS+dRWz5OeMKOdCyWml1PFvohar8Fyh/3J0neG0nHnyA/twcWydbBJMy3z
RxP9+9Hw3sXmdUg97Gud6dUktmdduk7d4WI7ovRw8qQ+enXw8oZd0ymg86mh
fI27OGDvVq06SDyu9mfeWroc5e3a6cLDEtPo7xr2xnobJ0ZwwfvxQI+4T/T3
wRkPu/dXRrlAi+rEJ/OO0fOSle2lmem9BSCVU+WuJOMKzivshsYGCGFvCGfX
TG4c7NBU2q20VwD93zhaPTP9zMq/9v2+wXLbsQKw7zpjd+ZsEnjlrvIOthfC
1NiUP39uJrPyr+2p0FqxTiCAJoebWSFBQaw8iC6KUCXv0wAcK52A6w+9WHmk
Ju0fXFa8h84zVZkxZNXNm42w1LbMUiY1GSTvbZBryWgAGYX+XavDwll5FE1u
CK8mvxaBzuW6mD8J4nmW/dE588Xzrvs/W+K6F8bB2d/NgjPjm+CBopcwZvkX
MCsZrN17XhOom1zwnLoxFfZIFveRUWqCfqIrRc+tBDBzwYiOuoFNoLDj5+rl
2+h8a0Ej5w68nRcAP64c3GOxWATPsp/U1299B/2tn7ndS6S/M6rVXjVySnIS
tMhvVOv2pddPdH4OW6xxi51v7enAAMU73cnwZ6RHk7QL/R2zV+cMNbV7HOh8
nNowTyobYjcWjMgur4VEgdXYlsN0PrYOA8fwUzK5EObX5WQbTv8uGNoXqSOK
yYJS+pxH17IFJRsL8iD/eYHt2ov0umflT07LnzvsfG33X/Uu7LhVCOfTLFxv
OdL527pN9g/fMqKO+K6mDjZ/3tSxt38BdX6l4drSXkbTS0BGZ3GC4h46v9ux
R8KYNwPY+d2GXFEZqfdOLP/2avLK2fT6hbl986/oLXS+/NQKp/GLvgVT6277
++yLrB2fDtVlPMUaCXpf3fFqlUbffW6wzu2wvVkdH7akDs/wz6DPpYxeluU3
QTwfwHqH7VrzakFzCFXv0sAZJ9+XOEPJt7SOWI4nKy/G1C/GN9xfxsKK3z/H
y5TEw4XBcrHlBuHUc//csFN/yvsoji+9/rl/N46S2+O+Y1ulnZKhtK7fhusO
cdCUZSYtG/KFlffixHzR9KA7dP62o8fMZWZZfYZV95XvL+5P52+jvvcYNCDi
/d6PUO7Z9HDJA1qvzJyP35k9lT5nWdWr06j2Mp2vzW/PEyvH0U0QYr+zTvdg
Ooy0f/BHbSJ97qL59CDDYY/jYNe6hEjhGvo84VlhxZelAzPBMsw+zNSSzs9m
d85A4U57OVy4Hsg51ScX1ocqjOX3Zudj+9oSNn/L4Dyqv6vjLRJG1GbBktNn
Bv5eTudjy8nOd1gxLBuua/jr+AsLoVu5l2zIEvr3ItHPljScSbYRy+sNHZho
d6gINpxQC5/TLx+Gt9zmjJSiv0s/GL3Vcu3jEni6/M2tgCOFkCVaMmX7BHYe
DK/ajuGzfxdCZX6g08C37Dyfmpn5Zd0Liyh9HMJVerm6Mw9eZIXOWt6Lzs92
9q7ow/ZetD9HfYk0cXxp3VkE0/pODUzxLISqk32OLLhMn7/72D4jUHZ1ORzf
536/+wydB+Rcha38CCN2vjaNLc6FfPH8ouLpIxmbfXT+Nod+rzibrOj8bYci
+vYKLsyBoPIrEsZC+tyk7twjew+K45SuVpVsK6MimGRweemdOvpc4p4h044a
iecn39sDByd11oF0H4OWRb3p370qnHTy2+2uKgg+0Os6b3YFdEs2jtTaxIEl
tb7T1y+n86vdS7LW369ZDUeshZMV3nNA6Ur961/b2Xk8dNbOWmijzYHZu98t
Vk3jQNrLjYnFc+nzek+kao7lNtXAjHu9fh20o/OrRb7LUqrr4ELeWasTGmvo
PBzx0smDf1yg86mdFDqrvpan93vskup1todw4Yn2ny79qRWgr2Td7SWejxx7
MnPJnAUVxHe3haCc3zLwTD4Pxh+6s1Ephs5jEFGlWqZ6iwOPnzwKq2il84h0
m3WfUanhgeNNbdeYcDp/2qi4UU1t/ZuAq8Eb1r+ND1/bpQp5g+g4Pye6IeTC
lbesPMT9Moq7N74RQt30rGGJM/gQX7O04YKzECas/JKhbs3O2zFhw7jcV3uF
oLPq0vQKF/H9RwtXdI6hfzes95391hG59bD3ZJ95eTvoPAZrL3XqaIc2gJ9t
748RukJovdf0e1BtAyuvRvLNS3NcXYUQMOhtY2XfdCpe0P80ZkKoYSD8/nbP
ZMNq+pyyzFx37WIjEZSnuUr4742j/G34Atlru0yE8ER02WREJn1erCNMtzko
TgRRM6TciorYeUKsylZNGBgYBAc6L3pc7NcICRr275e8+0LFRz+TDnwo06fj
xmSLjbpNVjS20LBtsjfNo+aj5/+RMZXUzYHTI10SXy+oA2PpxrYpk0uhf+wg
/2wXPhUnaslVnB/nxwfDewFcq+F0/pyDer7bHZ/T52Lb9HfEa5+l1w1dRqiM
kuvgw1jlorZLchms/GTZCUnNjjoi4F+VuGnTQf9eNV/qs/TX2Tnwq3AT32xs
Iyw/1Kp7aTv9u3Gr9rWZqiwWy1WOulaNRgjITfNqdNLlAS9vvIp1RxgEu3x7
lLntJWX/VYw3+WvJu1HzoRcTfJPzHgeL57nNnz+G5EDbdc5LOfuPUFL8WnLp
/Rw472m9dIheLrwLLpx16wSdd6usOx2iHDKg7pGy5ZGQGlC/WHQwdVwGFC1U
Haz1qwbq31+Xs9fPgFs+x158GJpO9cvUXT9iTMTv7Ygb5xPfUAT+w46UjOmV
Bbu4k/WqjIvANifRfOrADPh59VxX8eZicBG8NTR+lwZ/FA0nHlj7FTy2XH8q
4qTB9L0BqmomJeD/ZplhZSudn/riuqejtJTpfb3R2fd+NShlUnQP3LlXsl9U
BhjeNZepLamh+vFW1OWIGFsemARdFm6+Fgr3oXfN9708mLN6W5SDXBjcan9+
3vEgD2wttp7I2FoDRitHmNbL1UP9WtUGvmEtbJ5Wse1zv3rYkNLxSSjW49hX
QarJYvuxI/X1T9m+zpR+Fc27m/Nc7hkVB6nvsdysr+FG6U/2n8BzdRHu1Phf
VTTwljGmf698gd/158oqPKh3kb6uYiaOrx/O6NqzkAddx2uPLWtqgOiLtq0P
TsWD1rVpqqUtHmDhNHziu/v07/+hHMy2vtEgM/UlFOzrm/xcjc7vhfbDxbjw
a9uAl+A9qF1TXc4Tnrr9tOZdiIeJA3JXGVVnQdBMhZIL0+g44OzFA8Y5IxLg
V/8Xw1NOxsGGW9fmpQQkw8fHuRKym9j5vWrV7lye8yCR8kdxSfo/7Y6kwduy
6aEDDtHzd+7Bd3v/rKG/x0U/mz/Q5trrpSnUPMHTXFB7Y2UWbHFecdKkm86z
c2TvxH567tFQK3iXyknPhuKjB+rOLSkCpc6bswaa0HHF9JGCXlELc2Bn7bLL
/SvY+cB2BjbkFuVmUvNm5+rIQq+UbIj4ti/4azadx2rP6nMa+8V+NMGh+eKX
dDqv1Yzrw2ZM254N6TpLBpd0V7HygyUtKfL97ZMNNn13bDeXaIQC35BJhaY5
8Oqx8o3os+UgmrI6Z50rncf13vSJgq3HimBP/MDztQV0Hi6pPjJSNuvoPFBT
Kl3mXrIohMIJ+9Ns75VASvYM1xWSRaz8X6nOz0wjPYrBWl0l+0dbDthmlc4a
2L8CvIJWmr20Z+dB+/58do5fNd3ucRgrv6m8HJ6HXY9zGc2DJSeG2i2YUwGH
I1vrj+bS+b7QD+43v8k1kKiAly5DzTffpn93HOtPOb9m/Iv+1fD62D/zTCfQ
v5uNfE28Mj7i5+8quC7c/qQogAPv5ooiJmhXw4KjTbc26rHzg8WO7qU1ZASd
H2y63iFuyolaiGiPvSk3iM4LoWPy7cdmGfo9s15LDv/JEvtxvYw7Wns5cOSH
9GX3fey8qIZD9mh2OHDgUJ72QD19PpGXoBzUZrfpbA7nwVHpQ3FxzXQ/iMqH
pg/azAH3VYuSRv6gf5e06JvuvrvGdeA5pk9D9B12frGFBde+Z56nv7u/k+6T
eaWkDgZ0vthV6E+vH6wd4tf7hQr9fTTaIR8ZwaQLa+pAymSK99Cjz6FrxXLr
dcL/ww68cDIRrOLDtAscp832L1n5xbybfGM4U/lw1mC53pGVpfBo5z2bYw58
MOGOK1+0phQiDqtejhZjw7sNpfc3cCA3wkv3E/DBLirBZnhBLTy/nek5L4oP
D+Z+GrDHUgih1vpqb8XznKqXB31uL6bjGNTveN46+TANPqgGrlt48ruIyCvy
GQzqXrqXcASQrpo86OFWOt+YxdDM/c0RdB4nX6O7O3OzRZD5cvwVhX10XFEz
JqJVrkwIM84tqD7eSccZNbJOn9N/JFP2d/B1GevQVUIQnRNd0f3UAN5BhuEp
T9n5xr5MzMm4sqgBur//kY7ekA7dkdFXPsg1wQk7P+stnExWPjCDjwGvnVtF
cD1odNP2Tj6cWVM37FXvJrBIv/sifCfdD1SetkVfFhsJaTq1617K7/sjgmaT
yNfKfbOoccT6bcp/nC/7QscBMVpfzt8+FA3WC+/wg+aFQ1ztp4GTuXTecbd1
CiNLXOPA99uZB/Vzoqn33m1KmmFllkjZcYW9ctMkfem8WUoW82utfcXz8n+e
aTlIp1N0zM78YjviZiaY7L248Y6eOyvfWFFTRGV3VCYlX9oDP+599D0fwk/e
PZU9Mh8maFcdqttcBGsUTRdx9dh5tkNSW1Wv1BXC3c0/Ru2Vpn/P8XDwiB9W
gcWUnfrnvrvcTBc6/xiV78j2i4rjyGowWr6ul9nWLFY+slSVn30TXtfCwyaD
MTELa+HosECuZjI9D3n2pD5g1yUOPJsdlOL+nn5vxx/PtfPs6XxkGk9lD5/v
qILYQ41PrknSdMyYvDPlQyQP+gQcHDl1KDsf2ShTV6vQPFqOc49mBru/EsCK
3EuBd4cLQHXAwggDTzqO369gc7n9qBBW/lCcEORC5yOzb33tMGQOnY+M03j7
sodrNgQ/9L63UO0NHK4wOa2qkwPczbdtB297S7U/pnrf/PE3suFXnzP+/Adl
MHzjpWtjxOPn8m3J9gNif2R9a5+Epxjn7jYwfRBTDX0bhx0OKsmAPjfsT76u
5sLeNJ8b977ngMEY34iF/Xjwo6llTNDSfIj83K7y8CYftmoeUC23KYN3OwO9
pMTy/aupUOQstt+7xvLASIzV++/dkCHG6lX5u41S38GJqcG1NnY8uNzZePHn
7yTYLHph1KxYD4aardPPSojgWa6C5QBxnPRSj6d6wlgEaQ5ZK0138ODVKht3
jTtp4Gbz+eWV5QIY5/dkYVdbNrQH9L5lPEcEXb/07ELF/qfolWSNclgGzAgJ
djolxnMmFvVxE/MXsCD10lYxPUe8vOYlicfni/zl/gvE2OaqcfQrMfaY6Sc1
/iQPdiof+LAo2BdKHFztLCXqwexUe3rVsWTwDJQ8dmpcJiySkTaLy66GhBBZ
4/FTcyFfNndARRIXlJTj0vurCOBh5N1vU7vS4OC547ena6TDi5HH53wr+Apr
zhQp3DmUAcHtthnNn8pgc+n2Bq2T+TDW69cGc2EdGGgs1Msx58HY55eUpVeJ
QFJN/svOF76w3+7+sZ3jciDh+tBdU82qweqnQO7JarH97A3P+ntWw1El6d+K
r/nw/wG/YoBQ
   "], {{
     {RGBColor[0.293416, 0.0574044, 0.529412], EdgeForm[None], 
      GraphicsGroupBox[{
        PolygonBox[{{264, 359, 43}, {1766, 1767, 583}, {539, 988, 415}, {346, 
         583, 334}, {1802, 1857, 620}, {702, 1002, 141}, {1956, 1958, 299}, {
         412, 404, 141}, {1850, 1535, 334}, {415, 808, 299}, {1816, 1713, 
         539}, {1544, 2149, 594}, {1535, 1776, 334}, {2268, 1929, 264}, {2032,
          2033, 808}, {1991, 1553, 359}, {334, 349, 43}, {806, 412, 294}, {
         988, 808, 415}, {1463, 1916, 259}, {1767, 1851, 583}, {1927, 2268, 
         264}, {1464, 1921, 259}, {1501, 2107, 294}, {1873, 1668, 489}, {1989,
          1990, 771}, {1669, 1873, 489}, {346, 334, 43}, {2109, 1615, 872}, {
         349, 771, 264}, {766, 346, 259}, {1875, 1554, 359}, {1705, 1430, 
         530}, {1958, 2112, 299}, {945, 771, 349}, {1921, 1463, 259}, {2029, 
         2276, 806}, {530, 349, 334}, {259, 346, 43}, {943, 583, 346}, {359, 
         594, 43}, {1553, 1875, 359}, {489, 259, 43}, {1685, 2166, 920}, {
         1856, 1598, 404}, {1777, 1705, 530}, {2112, 1504, 299}, {698, 412, 
         141}, {628, 699, 141}, {698, 920, 294}, {1713, 1442, 539}, {530, 945,
          349}, {702, 415, 299}, {1598, 1815, 404}, {539, 415, 404}, {412, 
         698, 294}, {2276, 2028, 806}, {1929, 1468, 264}, {699, 698, 141}, {
         412, 620, 404}, {2267, 1985, 766}, {1801, 1802, 620}, {1002, 628, 
         141}, {766, 943, 346}, {1812, 1604, 628}, {872, 702, 299}, {1668, 
         1987, 489}, {349, 264, 43}, {1986, 2267, 766}, {987, 620, 412}, {415,
          702, 141}, {1500, 1953, 294}, {806, 987, 412}, {1604, 1813, 628}, {
         2080, 1544, 594}, {594, 489, 43}, {2107, 1500, 294}, {404, 415, 
         141}},
         VertexColors->None], 
        PolygonBox[{{1916, 1986, 766, 259}, {1468, 1991, 359, 264}, {2239, 
         1812, 628, 1002}, {1815, 1816, 539, 404}, {1893, 1685, 920, 698}, {
         1430, 1375, 945, 530}, {1985, 1374, 943, 766}, {1776, 1777, 530, 
         334}, {1990, 1927, 264, 771}, {1400, 2032, 808, 988}, {1615, 1899, 
         702, 872}, {1899, 2239, 1002, 702}, {1953, 2029, 806, 294}, {1375, 
         1989, 771, 945}, {2033, 1956, 299, 808}, {1857, 1856, 404, 620}, {
         1554, 2080, 594, 359}, {1851, 1850, 334, 583}, {1399, 1801, 620, 
         987}, {2028, 1399, 987, 806}, {1442, 1400, 988, 539}, {1894, 1893, 
         698, 699}, {2166, 1501, 294, 920}, {1504, 2109, 872, 299}, {1374, 
         1766, 583, 943}, {2149, 1669, 489, 594}, {1987, 1464, 259, 489}, {
         1813, 1894, 699, 628}},
         VertexColors->None]}]}, 
     {RGBColor[
      0.40248095264917916`, 0.24703889803025533`, 0.6827159848639923], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJw1VUtIVVEUPe8+7z1XzQaWFRTPDBoUjYTX55mllH1BQ2pkZak9K/tBYQ3K
Soigj1BhhKBW5MiCEILCwkoqE8SIrFFQQlKRoH2owZNaq3UcHPa+++y9z95r
r3NuXtXB8gOeMeYyVhRrizVmM9ZtGPfCMAJ5J2LMJd+YjNCY2fC5EhhzF+sP
7LXwHYDeBn0l9ldg7UHcBPynQi+Gvgn7h6zy2DRjimCbgzxTsJ8V6jsXshw+
nTivC/4l0GPQm6Hvgp5C7EvoR5D3BGJLoMdgb41oP+l85iNXS0Q+ccSfxurE
93rsb8Aajqj2Sqv6D3uKYx7mbnf5+7D/CLofVX3Frua5qDMb8WX4LoNPBvSt
iO2Bvc7qLPYfd7idg+0V1hC+GyFP8dsTxrWhcC5EjqXQqxH3E/Zi6K04620g
7JhnB+yZWPtdTzHX12vYelFnFmKPQn7DdwNiN0J+Ruw6+I1CbsPeQtgLkfuD
r/pGYN/uq6+GQBgRn2FPOBKHWsgFWE3Qn2G/Ev6LjGp8F6jO77B/9OXXCDnk
7Ky1zRPORezL6rvD14wne6Ak/gnsF2C1eOp1t+uXHPzkKedqq9nTv9pXb+wr
A7x6A58Co3mRZwPE2RMGnNEsxM7EqnezGXE5ORvGM5Z8aHecOIb8N331fjEq
bvIs5mONxKov0H1gj+QE51kREW9YD7Hi/Vjj7shy4LY2EK4/IGuQPwGf95D1
2MtHTBX2ZqRpDpzrkOvrLGp/bpWXvOt23OOcL4Sqaxz6WKC+c2DLJAeQI83l
4Vl1vs6mPuGJj7zXxOX/zJAzD7E9gc4ld5odf6bh/GwrnDh/zpsY3oJt3GpO
ScSmB8JhFWR/oDv529NMeBZ7JN/J1cXwXxLKRg4xBzEnBlXO5zFyn4RPCnv3
fH2nXL4Jl3PQqjf29RT6cav9ZFS4soZR2G5YYZtr1Q97iSP3FyvOs/ZYqPrJ
96uO85mwdVv18SDQnSEHSh2+PJezsVHhdi3U/eQMOQNygnmI91/HpTErnrHf
dKs7zH55L59Y3c0XkA99ndED2W91Rgfkr0nMIXPcG0j+5jsOk3eDjnvkb1NU
fXG2+3zVeR+y16om1pZyePI9Kwt0R/k2N4XKwdw17p0lHl2+MDnDtzrUGXw/
ks6nwtf/gzl3WnGC9rj7x/Ausqd5ofpaBpkIdcZ0q7eVNZwP9eaS87xnpQ7z
r7BddzPl+81Zco7sh/eWOP8Dx8ndHA==
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlFuIj1EUxc9833z/85lBmdyKxlAeyJNym3EZIhNFbi8od4PcSnlxHZH7
vJGmXEMJD5TcE5N7mAe8GMXkElFDxAPxW63/w669ztpnn33W3uf0Xbhm+uok
hDAbK8WaYwjbCyHsT0P4B9Gah7CTtef4ZQTMw5+PrQL/LglhCXxO/Bz8Rvbc
hnvFWiS2N/mmwo0CDyH+LDGd8a8UzI0l/gz+N2wm/Fz4Mvhy7Be4FzH/shCq
ydkErsPaC16Trz0H8LfBt+APJP44fitr/cndxtpi/FmsncOvhq/B/0iOthLn
bMGvYq0hcY1bOHtO5rP/JOZOgSsTxzwi/iG2Cb6JHM/wn2KN4CPg9ew/AP6D
P5j4evDS3PqohvFwXaO1OIw9Bo+AX5w6/gK4J/z6xPmO4Q+FX1i8Tyf8v5m1
q8WqwQuia1GO++x/UHBvNlPTLrhbxMfUPXyDX8ueFanve53Ya9igYP0rie+D
rUtc3z5i9+bmpHklfh/sbOI7bSC2Y+5anrD2BfwO3L3U59fgv858t6WpZ2ta
9H7N2Gn8k9h78IfEe79G31U5NFtVrE1Nfd5PuB/Y4eA9NzPfUXfLiLkB3g1+
AS4Fl0uPaK4Om4A/Pvpu6qlmRTWqNs3MuOieqN6jiWvdUfBsq2bN9p3ou2rG
D2aeYekjDZvBe4rnd0qtbbfcZ4tfBlcf3Vu9Ic3OSPj61DO0Fm4MeDn4PDET
4b+zNiB1jHonzaSVejg69wyrl9ozPPcMSPtF4HvE3o0+Wxp1gbsfrc0UrAI8
o/g+pHEj+xdlfku6bwvcs+i3ozsNgx9VcG7NpN7Sh+J71psaS2xZ9PwdIb5b
9MxLb72JT+DPxf7qDels1aRaVMNo/NrovdKnAf9K5r9Ib7wiukfqjWZ0Jdyh
3FrXkG8S3OToWVU9+muuFt+7ajyR+Y1Jf+n9FnyqOH/SuB+5+ubOpTdzMfOM
q9/6kzRb+hP1F0qPb+D26FzKsRX/Uua/Q/VuBF8G/02cQ9p0iO6PNHpBLS8L
7rVmWrENxf9Me3pEayIt9Cf8B804vNo=
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.5202710437428251, 0.45184405546090606`, 0.8482842073179402], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmH3M1mUVx3/Pff9eVd5t6QoRAg11U6yFgIh7RMlMk0eowKWgBpECtWSD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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtl3nQV2MUx+/v/u69z73tKdsM8r5kKTP2pZRISrKkZKkhJYVUjK1Qb0pZ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         "],
         VertexColors->None]}]}, 
     {RGBColor[0.6184046403594226, 0.6152663714690145, 0.9107430832234539], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmXvY1vMdx3/3777v333/eHrqmVJPtHqkPCJz2BxrTiURtTFX4hpF1+yi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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwll3nQjmUUxp/3eb5na4Qvpvhk+KgI9YcaJcTYy9ZUZrJM2Wra0LRMiZC9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         "],
         VertexColors->None]}]}, 
     {RGBColor[0.7050076129849243, 0.7544132509447526, 0.912717364258514], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmXvUVcV5xvd3zt7nO7ukSXRFMIohghfUcpNwUQRERVBuAipYARVNUAQD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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlnuwTmUUxve3v29/Z++Iakoup9S4TQqHI7cmjZE4LuEkRjg47pIw7pzK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         "],
         VertexColors->None]}]}, 
     {RGBColor[0.7886566347535708, 0.8556768557127201, 0.9023715389528207], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJw1WHmQV8URfu/33sybZySkAAmwSuFJKXLsgrKmVIKxLDwqpaCsHIISMYC6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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtl3m0lWMUxr/vnPN95/2Ma6HMVpGxuvc26o8MYZFYlkpzZlFXw52alJSk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         "],
         VertexColors->None]}]}, 
     {RGBColor[0.8664319820074635, 0.8816128616243769, 0.8675282530801031], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFVE1sjUEUnfd975sZsWm6lLSJaCgtC1okxE6t/LZaETbUT4u+ts8CG2wa
tXg2lGos/HbR1l9CwrKh1UqKtijVShsWiFhIWBXnOCMWk3vn3DNz79x7587d
ldnSEBljclgx1qg1ZivWg5Qx+yA3eWP6U9qXgdAJzmcc+IR1H3q3MybltT8G
WeqEV0OPsfqg1+CedcAfhjvI5T2zoY9BXwvdgXsXnK/YZ2Px6HMJsErYcsBX
BxvxH7HioK/1sNfCxyPgNV5nyOddq2LFMAmsHWsS2BjkGSfffTiXTYzZA84Q
5DTwOTjT6BkU3gExA3k8Ucy0L/biTONsDng/8NZI/vnGgrQxJ2EfgX4I3AL6
SAmjjXzGkZeWX+KFacU7CP6A1/uOwlcTzg7jbAbyMPbPoF+Ez3pwKnD2CLCs
la8eYB+xL4yVq74QUznsO4G3Ql8EzgT0Wpx9CdmGfR7445Eka1ENvAH4LWBP
gd2OlIdiYO9DTtoRwzwvfl2iPXX6H04UQw56nVeenyTyRx98e0GsGDbC/iVR
vDdh73Gq6wtgH5x43Yn6iXVkziYi5a0I2JBTH7VBnsN6FylPjVa5SoNzJ/RV
s1VOmc8s8G1Ob/uO+xOvXikM8fF+1tZ61WsK3AtOvrlYf3IqY+WJ9WUsvcn/
vs4P+eR/anDq24Ne+eWZEui/YauF3o+1JtJfK4e9jO+JZF/oxWEfNXv5ugF5
3as+7FPWhnW5lKjGzG0G+Anse1k/nK0If/As5FunmueH2jPOlU49zfcOxMpl
lVHvlFn1zxvoGS+ccsZKP8/ch/x8g8/NXu8pAjZi9Xe2Q18GvSWld/LPMB4L
3GHVY/8T9gOhtxkvZwc5rFNVrLczd4OxOJ2JfPD+/ZHyxPt3W80F9v+OWcih
kz/20/PQe1dgv+z1Vs6Ltki9YbxmGWdLF2S3D2cixcg88L4NXndehbzmFVNv
mFOsdQliWOAVdxr82CnGpVa5YJysMX2zV5cDW+EURxfkL6dZyRm81ypnrMl4
qEsL3t7s9M85D5g75oTzYDLMhI606kCd/R2l9Y9egf/aa38a94xy3gUuJfmF
YY6xH/ifxsOf4iy/F+oyZfWvmLf5uKPY6k88tpo7zBVr3hvqzj/X5FXPLPRT
iXq/1GsW/5uHHSGGv3M3zN4Jqx5lPH8Av6/mMw==
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlEtsTlEUhc9////eexqRNIaSNhHiEe1EigExYyIhVDwSI1pvLToxZCJq
UANaVAy8B37PBGFGSqnEo228+iKReEUMJGbEt6w7OMlee+27z977rH2nbWpf
3ZaEEDZyKpzRPIQRzlrsMYidMYRdnOvYy/D1wr1KQ6grhzCO7z34HWcN3Aj4
KbHXwN+wfxNzFtwDHhUH7gFv5/ulxNeC72QhdMD3w/eXQjiF/ZGY2orz/wX/
4Vwm/iu4BBc4dwrcDXcs9926oxF7Pqcb+wX5LhK7AXwD/Iz7+rjvEacTroUc
h6llFLwkcQ1HiR3AN4XYt+A/cH85h0rusQPuKjnr4VvBE9hvc8dOpuZv8NXc
tTURcwH+Te7ZDYDryTUTXzN2BzmbwHPBLYlrUi9PUs9GPa2A20xMH1wnuBrd
g2p/Scw71Ub8Y7jFnIfgB5lrVc2bakL4mbq/09R3F66qePhB+BHw+8zz1Lyq
2GdS51c9b7jvQ+b4hO9nSRvg+yXXNAy/H99QyQL6zrfL8XWUXY9meyQ1pxo+
w43h68dVhy8Fv47OvajsWgaitaOaWsAro3PpvjngjPt2JJ7Z7cwaknZU0xD2
MOdeyZrJc/ekXvSN3rKCb1vi+RxIPcP/s0usvRO5tS8NfsK+Ej1raV6zPRet
Zc14VvSb6i3X4MvAX+SruEdpb25mThpcBz4frQXt1GDqnZLWpaly7h7Vm2rU
LKUpaUkzbYJvhe9KrPEt2BtrrA3NYJz43tR6kx7HlLvIrzed0H4V/WnHb2Kv
55uD5P6Brw17d3Q/qm869gzOpLJ3thm7PbNWVMPszD0pVppeBb+1eA+990m4
59Qztew7H2J3FfuinHOIn1fsl3rep7ujZ6Uarmr+xT7pn6LeF4B7Et+/l9g9
0b3oG+3u8eJ/oB2Wdj9m1ro0vD16RpqNatir+qNzKUa7OlHwmmFD7n+K+pUe
F+beIcWqhvHMmpaWtQP7cu+ItKqdbMNuz60tabJMbCV61qpHs9eOajf1BreI
/ZV6FxQjLeVFvDTViN0Q/e/RTPWvvhT9b9MM/gE2d9JU
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.941176, 0.906538, 0.834043], EdgeForm[None], 
      GraphicsGroupBox[{
        PolygonBox[{{1519, 2402, 317}, {888, 889, 317}, {1639, 2129, 889}, {
         465, 888, 317}, {1798, 1577, 382}, {1837, 1836, 653}, {1173, 888, 
         183}, {1579, 1881, 384}, {499, 277, 85}, {465, 453, 183}, {281, 392, 
         85}, {1790, 1789, 611}, {2271, 2007, 787}, {825, 894, 465}, {2402, 
         2136, 317}, {1566, 2087, 605}, {611, 916, 499}, {2214, 2011, 791}, {
         856, 535, 392}, {653, 923, 520}, {894, 548, 465}, {381, 605, 85}, {
         888, 465, 183}, {2052, 2051, 825}, {605, 384, 85}, {611, 499, 382}, {
         1708, 1799, 535}, {1943, 2273, 281}, {1964, 1514, 313}, {2211, 1879, 
         381}, {277, 381, 85}, {520, 821, 313}, {2048, 2049, 821}, {1846, 
         1637, 453}, {2011, 2012, 791}, {384, 281, 85}, {2396, 1638, 1173}, {
         825, 465, 317}, {1796, 1790, 611}, {384, 791, 281}, {2273, 1945, 
         281}, {2009, 2010, 788}, {2127, 1636, 887}, {1635, 2126, 886}, {1637,
          1844, 453}, {453, 520, 183}, {2123, 1515, 313}, {392, 535, 382}, {
         465, 548, 453}, {2136, 1969, 317}, {392, 382, 85}, {2008, 2271, 
         787}, {916, 788, 499}, {382, 499, 85}, {1437, 1708, 535}, {2007, 
         2212, 787}, {1940, 1482, 277}, {923, 821, 520}, {787, 381, 277}, {
         2086, 1566, 605}, {885, 520, 313}, {1452, 1719, 548}, {499, 788, 
         277}, {1482, 1935, 277}, {792, 856, 392}, {2014, 2013, 792}, {1577, 
         1795, 382}, {885, 886, 183}, {1845, 1837, 653}, {1879, 1576, 381}, {
         1719, 1847, 548}, {792, 392, 281}, {1881, 2213, 384}, {653, 520, 
         453}, {1514, 2123, 313}, {887, 885, 313}, {520, 885, 183}},
         VertexColors->None], 
        PolygonBox[{{1576, 2086, 605, 381}, {2087, 1579, 384, 605}, {1795, 
         1796, 611, 382}, {2090, 1437, 535, 856}, {1935, 2008, 787, 277}, {
         2125, 1635, 886, 885}, {2212, 2211, 381, 787}, {1515, 2127, 887, 
         313}, {2126, 1833, 183, 886}, {2134, 1452, 548, 894}, {1969, 2052, 
         825, 317}, {2128, 1639, 889, 888}, {1789, 2159, 916, 611}, {1799, 
         1798, 382, 535}, {1844, 1845, 653, 453}, {1836, 2170, 923, 653}, {
         2010, 1940, 277, 788}, {2051, 2134, 894, 825}, {1636, 2125, 885, 
         887}, {2213, 2214, 791, 384}, {1833, 2396, 1173, 183}, {2170, 2048, 
         821, 923}, {2013, 2090, 856, 792}, {1945, 2014, 792, 281}, {2049, 
         1964, 313, 821}, {2129, 1519, 317, 889}, {2159, 2009, 788, 916}, {
         1638, 2128, 888, 1173}, {2012, 1943, 281, 791}, {1847, 1846, 453, 
         548}},
         VertexColors->None]}]}}, {{}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2012, 1943, 2273, 1945, 2014, 2013, 2090, 1437, 1708, 1799, 
         1798, 1577, 1795, 1796, 1790, 1789, 2159, 2009, 2010, 1940, 1482, 
         1935, 2008, 2271, 2007, 2212, 2211, 1879, 1576, 2086, 1566, 2087, 
         1579, 1881, 2213, 2214, 2011, 2012}],
        "1.5`*^-9"],
       Annotation[#, 1.5*^-9, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{2127, 1636, 2125, 1635, 2126, 1833, 2396, 1638, 2128, 1639, 
         2129, 1519, 2402, 2136, 1969, 2052, 2051, 2134, 1452, 1719, 1847, 
         1846, 1637, 1844, 1845, 1837, 1836, 2170, 2048, 2049, 1964, 1514, 
         2123, 1515, 2127}],
        "1.5`*^-9"],
       Annotation[#, 1.5*^-9, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1939, 1480, 2005, 2198, 1385, 1433, 2083, 2084, 1998, 1474, 
         1934, 2153, 2270, 1434, 2154, 1783, 2155, 2156, 1878, 2158, 2157, 
         1565, 2162, 2163, 1880, 2161, 2160, 1792, 2088, 2015, 2017, 1944, 
         1486, 2018, 2093, 2092, 1734, 1395, 2221, 2021, 1490, 1946, 1489, 
         2020, 2219, 2220, 1732, 1733, 1589, 1885, 1588, 2091, 1791, 2089, 
         1586, 1882, 1587, 1730, 1729, 2210, 2209, 1942, 1941, 1939}],
        "1.`*^-9"],
       Annotation[#, 1.*^-9, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{2046, 1963, 1965, 1966, 2249, 2250, 1754, 1755, 1648, 1908, 
         1647, 2131, 1838, 2135, 1649, 1913, 1650, 1761, 1760, 2263, 2262, 
         2053, 1523, 1970, 2055, 2280, 2054, 1422, 1762, 1848, 1757, 2279, 
         1758, 1651, 1912, 2260, 1418, 1756, 1416, 1907, 1718, 1717, 1863, 
         1623, 1862, 1622, 1831, 1906, 1414, 1753, 1408, 1903, 1902, 1621, 
         1748, 1747, 1828, 1447, 1407, 2045, 2046}],
        "1.`*^-9"],
       Annotation[#, 1.*^-9, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2327, 1550, 2325, 1549, 2326, 1771, 1778, 1726, 2195, 2194, 
         2193, 1558, 2450, 2334, 2269, 2289, 2288, 1556, 2478, 1557, 1548, 
         2422, 2423, 2421, 1670, 1988, 1467, 2287, 1466, 2286, 1537, 2322, 
         2177, 2178, 1426, 1769, 1765, 2417, 1664, 2416, 1665, 2418, 1539, 
         2327}],
        "5.`*^-10"],
       Annotation[#, 5.*^-10, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{2453, 1562, 1707, 2342, 2343, 2341, 1563, 2004, 1479, 1937, 
         1481, 2006, 1575, 2085, 2300, 1483, 2301, 1484, 2164, 2165, 1883, 
         1884, 1392, 1797, 1394, 2218, 1886, 2096, 2095, 1488, 1949, 2097, 
         2098, 1593, 2022, 1491, 1950, 2099, 2100, 1594, 2357, 2359, 2358, 
         1492, 1590, 2456, 1393, 1438, 2275, 2274, 1485, 2349, 1581, 2016, 
         1580, 1793, 1578, 1680, 2455, 1390, 1785, 1386, 2454, 1676, 1564, 
         1784, 1674, 2152, 1675, 2424, 1473, 2291, 2293, 2292, 1384, 2453}],
        "5.`*^-10"],
       Annotation[#, 5.*^-10, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{2352, 1585, 2350, 1584, 2351, 1788, 1794, 1731, 2217, 2216, 
         2215, 1592, 2356, 2094, 1947, 1487, 2019, 1591, 2353, 2355, 2354, 
         1583, 1682, 1582, 2475, 2474, 1681, 2298, 2299, 2294, 2297, 2295, 
         2296, 1570, 2201, 2202, 2203, 1436, 1786, 1782, 2427, 2428, 2426, 
         1678, 2429, 1572, 2352}],
        "5.`*^-10"],
       Annotation[#, 5.*^-10, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{2379, 1618, 2376, 2378, 2377, 1820, 1827, 1752, 2246, 2245, 
         2244, 1629, 2390, 1449, 2391, 2278, 2308, 2307, 1627, 1448, 2480, 
         2479, 1628, 1617, 2440, 2441, 2439, 1692, 2038, 1508, 1960, 1507, 
         2369, 1608, 2230, 2231, 2232, 1443, 1818, 1807, 2436, 1687, 2435, 
         1688, 2437, 1610, 2379}],
        "5.`*^-10"],
       Annotation[#, 5.*^-10, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{2459, 1619, 1716, 2385, 2386, 2384, 1620, 2122, 2310, 1512, 
         2311, 1513, 2047, 1634, 2124, 2312, 1517, 2313, 1518, 2132, 2133, 
         1909, 1910, 1417, 2146, 1661, 2145, 1421, 2261, 2144, 1914, 2138, 
         2137, 1522, 1971, 2139, 2140, 1656, 2056, 1524, 1972, 2141, 2142, 
         1657, 2406, 2408, 2407, 1525, 1652, 2464, 1419, 1453, 2282, 2281, 
         1520, 2397, 1641, 1911, 1640, 2130, 1839, 1840, 1697, 2463, 1415, 
         1832, 1409, 2460, 1695, 1830, 1829, 2169, 1693, 1901, 1694, 2442, 
         1509, 2304, 2306, 2305, 1406, 2459}],
        "5.`*^-10"],
       Annotation[#, 5.*^-10, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{2405, 1655, 2403, 1654, 2404, 1843, 1849, 1763, 2266, 2265, 
         2264, 1659, 2412, 2143, 1973, 1526, 2057, 1658, 2409, 2411, 2410, 
         1653, 1703, 2477, 1702, 2315, 2316, 2314, 1521, 2398, 1642, 2251, 
         2252, 2253, 1451, 1841, 1835, 2445, 1699, 2444, 1700, 2446, 1644, 
         2405}],
        "5.`*^-10"],
       Annotation[#, 5.*^-10, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[LineBox[{23, 38, 53, 954, 68, 729, 83, 789, 98}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{107, 108, 109, 978, 110, 735, 1044, 111, 981, 112}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{1311, 115, 737, 1045, 116, 984, 117, 1328, 118, 119}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{128, 989, 143, 743, 158, 1003, 173, 1352, 188, 203}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{327, 33, 340, 49, 1032, 2465, 65, 365, 81, 378, 97}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ], 
      TagBox[
       TooltipBox[LineBox[{24, 40, 1356, 56, 1297, 1233, 72, 1335, 88, 104}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{122, 138, 1359, 154, 1303, 1247, 170, 1341, 186, 202}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{407, 145, 421, 161, 1047, 2467, 177, 448, 193, 461, 209}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ], 
      TagBox[
       TooltipBox[LineBox[{327, 17}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ], 
      TagBox[
       TooltipBox[LineBox[{407, 129}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ], 
      TagBox[
       TooltipBox[LineBox[{114, 736, 1311}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{2075, 1868, 1872, 1378, 1725, 1772, 590, 1852, 1428, 1372, 
         2185, 1869, 2148, 2147, 1542, 2072, 2073, 2070, 2329, 2328, 1431, 
         1377, 2466, 1032}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{2119, 1897, 1900, 1405, 1746, 1821, 637, 1859, 1444, 1402, 
         2238, 1898, 2168, 2167, 1612, 2116, 2117, 2114, 2381, 2380, 1446, 
         1404, 2468, 1047}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ], 
      TagBox[
       TooltipBox[LineBox[{2331, 2071, 2077, 2076, 1706, 2074, 2075}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ], 
      TagBox[
       TooltipBox[LineBox[{2383, 2115, 2121, 2120, 1715, 2118, 2119}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ], 
      TagBox[
       TooltipBox[LineBox[{1381, 2290, 2330, 2331}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ], 
      TagBox[
       TooltipBox[LineBox[{1411, 2309, 2382, 2383}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ], 
      TagBox[
       TooltipBox[LineBox[{1381, 2465}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ], 
      TagBox[
       TooltipBox[LineBox[{1411, 2467}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1975, 1455, 1974, 1454, 2319, 1528, 2317, 1527, 2318, 1423, 
         1369, 1764, 2181, 2180, 2179, 1663, 2415, 1538, 2419, 1666, 2182, 
         2183, 2184, 1770, 1376, 1724, 2324, 1540, 2323, 1541, 1919, 1427, 
         1918, 1917, 1983, 1982, 1370, 1424, 2449, 2448, 1530, 1529, 2413, 
         2414, 2171, 2172, 1975}],
        RowBox[{"-", "5.`*^-10"}]],
       Annotation[#, -5.*^-10, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{2321, 2173, 2174, 1977, 1456, 1976, 1457, 1978, 2176, 1981, 
         1459, 1980, 1460, 2064, 2065, 1864, 1534, 2063, 1660, 2066, 1536, 
         1870, 2068, 2067, 1465, 1924, 1925, 1926, 2189, 1993, 1470, 1931, 
         1932, 1933, 2192, 2191, 2333, 2332, 1477, 1569, 1568, 1435, 2345, 
         2272, 1476, 2344, 1567, 2001, 2199, 1780, 1555, 1672, 2473, 1671, 
         1775, 1551, 2470, 1552, 1543, 1774, 2186, 1923, 1667, 2420, 1462, 
         2283, 2285, 2284, 1531, 2469, 1532, 1704, 2320, 2321}],
        RowBox[{"-", "5.`*^-10"}]],
       Annotation[#, -5.*^-10, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{1936, 1478, 2003, 2197, 2196, 2340, 2339, 1561, 1673, 1560, 
         2452, 2451, 1383, 1996, 1997, 1995, 1472, 2338, 1559, 2335, 2337, 
         2336, 1432, 1382, 1781, 1389, 2205, 2204, 1677, 2425, 1571, 2430, 
         1679, 2206, 2207, 2208, 1787, 1391, 1728, 2347, 1573, 2346, 1574, 
         2348, 1938, 1936}],
        RowBox[{"-", "5.`*^-10"}]],
       Annotation[#, -5.*^-10, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{2361, 2222, 2223, 2023, 1493, 1948, 1494, 2024, 2225, 2224, 
         2363, 2362, 1495, 1855, 1710, 1711, 1854, 1853, 1800, 1804, 1805, 
         1740, 1502, 1817, 1503, 2367, 2368, 2228, 2229, 2035, 1506, 1959, 
         1961, 1962, 2241, 2240, 2375, 2374, 1511, 1626, 1625, 2389, 2388, 
         2277, 1510, 2387, 1624, 1905, 2242, 2243, 1825, 1826, 1691, 2476, 
         1690, 1814, 1613, 2472, 1614, 1809, 1808, 1892, 1891, 1890, 1684, 
         2433, 1499, 2302, 1498, 2303, 1595, 2471, 1596, 1709, 2360, 2361}],
        RowBox[{"-", "5.`*^-10"}]],
       Annotation[#, -5.*^-10, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{2027, 1497, 1951, 1496, 2366, 1600, 2364, 1599, 2365, 1440, 
         1397, 1806, 2235, 2234, 2233, 1686, 2434, 1609, 2438, 1689, 2236, 
         2237, 1401, 1819, 1403, 1745, 2371, 2372, 2370, 1611, 2373, 1955, 
         1954, 2031, 2030, 1398, 1441, 2458, 2457, 1602, 1601, 2431, 2432, 
         2226, 2227, 2027}],
        RowBox[{"-", "5.`*^-10"}]],
       Annotation[#, -5.*^-10, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{1967, 1516, 2050, 2248, 2247, 2395, 2394, 1633, 1696, 1632, 
         2462, 2461, 1413, 2043, 2044, 2039, 2041, 2040, 2042, 1631, 2392, 
         1630, 2393, 1450, 1412, 1834, 2256, 2255, 2254, 1698, 2443, 1643, 
         2447, 1701, 2257, 2258, 2259, 1842, 1420, 1759, 2400, 1645, 2399, 
         1646, 2401, 1968, 1967}],
        RowBox[{"-", "5.`*^-10"}]],
       Annotation[#, -5.*^-10, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1920, 1461, 1984, 2175, 1371, 1425, 2058, 2059, 1979, 1458, 
         1915, 2060, 2061, 1533, 2062, 1720, 1721, 1662, 1865, 1866, 1867, 
         1768, 2188, 2187, 1871, 1546, 1723, 1722, 2069, 1547, 1992, 1469, 
         1928, 1471, 1994, 2082, 2081, 1727, 1388, 2200, 2002, 1475, 1930, 
         2000, 1999, 1387, 1779, 1380, 1877, 1876, 2151, 2150, 1545, 2078, 
         2079, 1874, 2190, 1379, 1773, 1373, 1429, 1922, 1920}],
        RowBox[{"-", "1.`*^-9"}]],
       Annotation[#, -1.*^-9, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{1738, 1603, 1889, 2106, 2105, 1739, 1858, 1810, 1811, 1823, 
         1822, 1605, 1860, 1861, 1714, 1445, 1824, 1744, 2110, 2111, 1904, 
         1616, 1750, 1749, 1410, 1751, 2113, 2036, 2037, 1957, 1505, 2034, 
         1607, 2108, 1742, 1743, 1606, 1741, 1895, 1896, 1803, 1888, 1887, 
         1712, 1683, 1736, 1735, 2104, 1597, 2103, 2102, 1952, 2026, 2025, 
         2101, 1439, 1396, 1737, 1738}],
        RowBox[{"-", "1.`*^-9"}]],
       Annotation[#, -1.*^-9, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2080, 1554, 1875, 1553, 1991, 1468, 1929, 2268, 1927, 1990, 
         1989, 1375, 1430, 1705, 1777, 1776, 1535, 1850, 1851, 1767, 1766, 
         1374, 1985, 2267, 1986, 1916, 1463, 1921, 1464, 1987, 1668, 1873, 
         1669, 2149, 1544, 2080}],
        RowBox[{"-", "1.5`*^-9"}]],
       Annotation[#, -1.5*^-9, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{1812, 1604, 1813, 1894, 1893, 1685, 2166, 1501, 2107, 1500, 
         1953, 2029, 2276, 2028, 1399, 1801, 1802, 1857, 1856, 1598, 1815, 
         1816, 1713, 1442, 1400, 2032, 2033, 1956, 1958, 2112, 1504, 2109, 
         1615, 1899, 2239, 1812}],
        RowBox[{"-", "1.5`*^-9"}]],
       Annotation[#, -1.5*^-9, "Tooltip"]& ]}, {}, {}}}],
  AspectRatio->1,
  Frame->True,
  PlotRange->{{0, 1}, {0, 1}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 179},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->462461144]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->26279],

Cell[TextData[Cell[BoxData[
 StyleBox[
  ButtonBox["EllipticTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/EllipticTheta"],
  FontFamily->"Verdana"]], "InlineFormula"]], "SeeAlso",
 CellID->24884]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->496806995],

Cell[TextData[ButtonBox["Elliptic Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/EllipticFunctions"]], "MoreAbout",
 CellID->86202938],

Cell[TextData[ButtonBox["Special Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/SpecialFunctions"]], "MoreAbout",
 CellID->62013597],

Cell[TextData[ButtonBox["New in 6.0: Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicalFunctions"]], "MoreAbout",
 CellID->468091418]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"SiegelTheta - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 20, 41.9794734}", "context" -> "System`", 
    "keywords" -> {
     "Riemann-Siegel theta", "Riemann theta function", "Siegel theta", 
      "Siegel theta function"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "SiegelTheta[\\[CapitalOmega], s] gives the Siegel theta function \
\\[CapitalTheta](\\[CapitalOmega], s) with Riemann modular matrix \
\\[CapitalOmega] and vector s. SiegelTheta[{\\[Nu]_1, \\[Nu]_2}, \
\\[CapitalOmega], s] gives the Siegel theta function \
\\[CapitalTheta][\\[Nu]_1, \\[Nu]_2] ( \\[CapitalOmega], s) with \
characteristics \\[Nu]_1 and \\[Nu]_2.", "synonyms" -> {}, "title" -> 
    "SiegelTheta", "type" -> "Symbol", "uri" -> "ref/SiegelTheta"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[10329, 338, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->394591841]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 160993, 3523}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 1514, 38, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2137, 65, 1576, 37, 70, "ObjectNameGrid"],
Cell[3716, 104, 2202, 70, 70, "Usage",
 CellID->9985]
}, Open  ]],
Cell[CellGroupData[{
Cell[5955, 179, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6433, 196, 120, 4, 70, "Notes",
 CellID->20640],
Cell[6556, 202, 234, 9, 70, "Notes",
 CellID->70201465],
Cell[6793, 213, 617, 26, 70, "Notes",
 CellID->19750],
Cell[7413, 241, 1048, 32, 70, "Notes",
 CellID->771240644],
Cell[8464, 275, 1601, 49, 70, "Notes",
 CellID->98455669],
Cell[10068, 326, 224, 7, 70, "Notes",
 CellID->31226]
}, Closed]],
Cell[CellGroupData[{
Cell[10329, 338, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->394591841],
Cell[CellGroupData[{
Cell[10714, 352, 147, 5, 70, "ExampleSection",
 CellID->65399100],
Cell[10864, 359, 64, 1, 70, "ExampleText",
 CellID->359970644],
Cell[CellGroupData[{
Cell[10953, 364, 439, 16, 28, "Input",
 CellID->31370694],
Cell[11395, 382, 254, 7, 36, "Output",
 CellID->53054217]
}, Open  ]],
Cell[11664, 392, 125, 3, 70, "ExampleDelimiter",
 CellID->941397941],
Cell[CellGroupData[{
Cell[11814, 399, 684, 24, 28, "Input",
 CellID->305506288],
Cell[12501, 425, 3938, 69, 140, "Output",
 Evaluatable->False,
 CellID->716492446]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[16488, 500, 220, 7, 70, "ExampleSection",
 CellID->663858768],
Cell[16711, 509, 231, 8, 70, "ExampleText",
 CellID->661921282],
Cell[CellGroupData[{
Cell[16967, 521, 268, 8, 70, "Input",
 CellID->945695114],
Cell[17238, 531, 255, 7, 36, "Output",
 CellID->19043360]
}, Open  ]],
Cell[17508, 541, 124, 3, 70, "ExampleDelimiter",
 CellID->68546449],
Cell[17635, 546, 74, 1, 70, "ExampleText",
 CellID->110643755],
Cell[CellGroupData[{
Cell[17734, 551, 387, 14, 70, "Input",
 CellID->61459423],
Cell[18124, 567, 277, 8, 36, "Output",
 CellID->485532050]
}, Open  ]],
Cell[18416, 578, 125, 3, 70, "ExampleDelimiter",
 CellID->113702518],
Cell[18544, 583, 71, 1, 70, "ExampleText",
 CellID->1885257843],
Cell[CellGroupData[{
Cell[18640, 588, 568, 20, 70, "Input",
 CellID->1554839463],
Cell[19211, 610, 337, 10, 36, "Output",
 CellID->748128496]
}, Open  ]],
Cell[19563, 623, 124, 3, 70, "ExampleDelimiter",
 CellID->23239488],
Cell[19690, 628, 114, 3, 70, "ExampleText",
 CellID->1891534709],
Cell[CellGroupData[{
Cell[19829, 635, 487, 17, 70, "Input",
 CellID->1823427622],
Cell[20319, 654, 319, 10, 36, "Output",
 CellID->766411675]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[20687, 670, 243, 7, 70, "ExampleSection",
 CellID->497907993],
Cell[20933, 679, 631, 25, 70, "ExampleText",
 CellID->364931836],
Cell[CellGroupData[{
Cell[21589, 708, 583, 21, 70, "Input",
 CellID->196071820],
Cell[22175, 731, 271, 7, 36, "Output",
 CellID->138341053]
}, Open  ]],
Cell[22461, 741, 125, 3, 70, "ExampleDelimiter",
 CellID->262151406],
Cell[22589, 746, 252, 7, 70, "ExampleText",
 CellID->1489720645],
Cell[CellGroupData[{
Cell[22866, 757, 555, 20, 70, "Input",
 CellID->223040142],
Cell[23424, 779, 498, 17, 51, "Output",
 CellID->17771563]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[23971, 802, 227, 7, 70, "ExampleSection",
 CellID->403588778],
Cell[24201, 811, 240, 8, 70, "ExampleText",
 CellID->179005300],
Cell[CellGroupData[{
Cell[24466, 823, 623, 21, 70, "Input",
 CellID->36951453],
Cell[25092, 846, 39757, 656, 70, "Output",
 Evaluatable->False,
 CellID->301342322]
}, Open  ]],
Cell[64864, 1505, 126, 3, 70, "ExampleDelimiter",
 CellID->1688517432],
Cell[64993, 1510, 71, 1, 70, "ExampleText",
 CellID->2063568166],
Cell[65067, 1513, 1119, 41, 70, "Input",
 CellID->2022594182],
Cell[66189, 1556, 617, 21, 70, "Input",
 CellID->529266912],
Cell[66809, 1579, 66, 1, 70, "ExampleText",
 CellID->1221768511],
Cell[CellGroupData[{
Cell[66900, 1584, 289, 10, 70, "Input",
 CellID->762399044],
Cell[67192, 1596, 3230, 57, 70, "Output",
 Evaluatable->False,
 CellID->165474680]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[70471, 1659, 237, 7, 70, "ExampleSection",
 CellID->124660801],
Cell[70711, 1668, 372, 13, 70, "ExampleText",
 CellID->1757091478],
Cell[CellGroupData[{
Cell[71108, 1685, 231, 8, 70, "Input",
 CellID->235086758],
Cell[71342, 1695, 274, 7, 36, "Output",
 CellID->446262312]
}, Open  ]],
Cell[CellGroupData[{
Cell[71653, 1707, 310, 10, 70, "Input",
 CellID->363070318],
Cell[71966, 1719, 274, 7, 36, "Output",
 CellID->101020158]
}, Open  ]],
Cell[72255, 1729, 126, 3, 70, "ExampleDelimiter",
 CellID->1442857924],
Cell[72384, 1734, 208, 7, 70, "ExampleText",
 CellID->727786744],
Cell[72595, 1743, 464, 17, 70, "Input",
 CellID->1628072117],
Cell[CellGroupData[{
Cell[73084, 1764, 243, 8, 70, "Input",
 CellID->614088907],
Cell[73330, 1774, 161, 5, 36, "Output",
 CellID->192025603]
}, Open  ]],
Cell[CellGroupData[{
Cell[73528, 1784, 248, 8, 70, "Input",
 CellID->422826564],
Cell[73779, 1794, 161, 5, 36, "Output",
 CellID->254824041]
}, Open  ]],
Cell[CellGroupData[{
Cell[73977, 1804, 811, 28, 70, "Input",
 CellID->1122135009],
Cell[74791, 1834, 161, 5, 36, "Output",
 CellID->328891423]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[75001, 1845, 230, 7, 70, "ExampleSection",
 CellID->208392064],
Cell[75234, 1854, 280, 10, 70, "ExampleText",
 CellID->810605206],
Cell[75517, 1866, 288, 11, 70, "Input",
 CellID->321849602],
Cell[CellGroupData[{
Cell[75830, 1881, 180, 6, 70, "Input",
 CellID->312846986],
Cell[76013, 1889, 303, 6, 70, "Message",
 CellID->157722668],
Cell[76319, 1897, 476, 16, 36, "Output",
 CellID->73666234]
}, Open  ]],
Cell[76810, 1916, 62, 1, 70, "ExampleText",
 CellID->95037792],
Cell[CellGroupData[{
Cell[76897, 1921, 288, 10, 70, "Input",
 CellID->937563001],
Cell[77188, 1933, 278, 7, 36, "Output",
 CellID->687522528]
}, Open  ]],
Cell[77481, 1943, 126, 3, 70, "ExampleDelimiter",
 CellID->1633651548],
Cell[77610, 1948, 117, 3, 70, "ExampleText",
 CellID->1203140341],
Cell[CellGroupData[{
Cell[77752, 1955, 283, 10, 70, "Input",
 CellID->473691035],
Cell[78038, 1967, 276, 8, 36, "Output",
 CellID->69564747]
}, Open  ]],
Cell[78329, 1978, 86, 1, 70, "ExampleText",
 CellID->21240796],
Cell[CellGroupData[{
Cell[78440, 1983, 522, 18, 70, "Input",
 CellID->154347460],
Cell[78965, 2003, 326, 8, 36, "Output",
 CellID->533944107]
}, Open  ]],
Cell[CellGroupData[{
Cell[79328, 2016, 368, 12, 70, "Input",
 CellID->423840250],
Cell[79699, 2030, 326, 8, 36, "Output",
 CellID->114237547]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[80074, 2044, 228, 7, 70, "ExampleSection",
 CellID->706393597],
Cell[CellGroupData[{
Cell[80327, 2055, 1083, 33, 70, "Input",
 CellID->1328152518],
Cell[81413, 2090, 76730, 1343, 70, "Output",
 CellID->462461144]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[158204, 3440, 311, 9, 70, "SeeAlsoSection",
 CellID->26279],
Cell[158518, 3451, 206, 6, 70, "SeeAlso",
 CellID->24884]
}, Open  ]],
Cell[CellGroupData[{
Cell[158761, 3462, 319, 9, 70, "MoreAboutSection",
 CellID->496806995],
Cell[159083, 3473, 146, 3, 70, "MoreAbout",
 CellID->86202938],
Cell[159232, 3478, 144, 3, 70, "MoreAbout",
 CellID->62013597],
Cell[159379, 3483, 174, 3, 70, "MoreAbout",
 CellID->468091418]
}, Open  ]],
Cell[159568, 3489, 27, 0, 70, "History"],
Cell[159598, 3491, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

