(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    942704,      20271]
NotebookOptionsPosition[    923060,      19657]
NotebookOutlinePosition[    927241,      19733]
CellTagsIndexPosition[    927153,      19728]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Sound" :> Documentation`HelpLookup["paclet:tutorial/Sound"], 
          "The Representation of Sound" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/TheRepresentationOfSound"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Sound\"\>", 
       2->"\<\"The Representation of Sound\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Sound" :> Documentation`HelpLookup["paclet:ref/Sound"], 
          "SampledSoundList" :> 
          Documentation`HelpLookup["paclet:ref/SampledSoundList"], 
          "SampledSoundFunction" :> 
          Documentation`HelpLookup["paclet:ref/SampledSoundFunction"], "Beep" :> 
          Documentation`HelpLookup["paclet:ref/Beep"], "EmitSound" :> 
          Documentation`HelpLookup["paclet:ref/EmitSound"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Sound\"\>", 
       2->"\<\"SampledSoundList\"\>", 3->"\<\"SampledSoundFunction\"\>", 
       4->"\<\"Beep\"\>", 5->"\<\"EmitSound\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Sound and Sonification" :> 
          Documentation`HelpLookup["paclet:guide/SoundAndSonification"], 
          "New in 6.0: Visualization & Graphics" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60VisualizationAndGraphics"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Sound and Sonification\"\>", 
       2->"\<\"New in 6.0: Visualization & Graphics\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["SoundNote", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["SoundNote",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/SoundNote"], "[", 
       StyleBox["pitch", "TI"], "]"}]], "InlineFormula"],
     " \[LineSeparator]represents a music-like sound note with the specified \
pitch."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["SoundNote",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/SoundNote"], "[", 
       RowBox[{
        StyleBox["pitch", "TI"], ",", 
        StyleBox["t", "TI"]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]takes the note to have duration ",
     Cell[BoxData[
      StyleBox["t", "TI"]], "InlineFormula"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["SoundNote",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/SoundNote"], "[", 
       RowBox[{
        StyleBox["pitch", "TI"], ",", 
        RowBox[{"{", 
         RowBox[{
          SubscriptBox[
           StyleBox["t", "TI"], 
           StyleBox["min", "TI"]], ",", 
          SubscriptBox[
           StyleBox["t", "TI"], 
           StyleBox["max", "TI"]]}], "}"}]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]takes the note to occupy the time interval ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["t", "TI"], 
       StyleBox["min", "TI"]]], "InlineFormula"],
     " to ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["t", "TI"], 
       StyleBox["max", "TI"]]], "InlineFormula"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["SoundNote",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/SoundNote"], "[", 
       RowBox[{
        StyleBox["pitch", "TI"], ",", 
        StyleBox["tspec", "TI"], ",", "\"\<\!\(\*
StyleBox[\"style\", \"TI\"]\)\>\""}], "]"}]], "InlineFormula"],
     " \[LineSeparator]takes the note to be in the specified style."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["SoundNote",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/SoundNote"], "[", 
       RowBox[{
        StyleBox["pitch", "TI"], ",", 
        StyleBox["tspec", "TI"], ",", "\"\<\!\(\*
StyleBox[\"style\", \"TI\"]\)\>\"", ",", 
        StyleBox["opts", "TI"]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]uses the specified rendering options for the note."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, {None}}, "RowsIndexed" -> {}}},
 CellID->15192]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["The pitch specification can be as follows:", "Notes",
 CellID->21502],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "0", Cell["middle C ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    StyleBox["n", "TI"], Cell[TextData[{
     Cell[BoxData[
      StyleBox["n", "TI"]], "InlineFormula"],
     " semitones from middle C"
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], Cell[TextData[{
     Cell[BoxData["\"\<C\>\""], "InlineFormula"],
     ", ",
     Cell[BoxData["\"\<C#\>\""], "InlineFormula"],
     ", ",
     Cell[BoxData["\"\<D\>\""], "InlineFormula"],
     ", etc."
    }], "TableText"], Cell["notes in the middle C octave", "TableText"]},
   {Cell["      ", "TableRowIcon"], Cell[TextData[{
     Cell[BoxData["\"\<\!\(\*
StyleBox[\(C\*
StyleBox[\"m\", \"TI\"]\)]\)\>\""], "InlineFormula"],
     ", \"C#",
     StyleBox["m", "TI"],
     "\", \"D",
     StyleBox["m", "TI"],
     "\", etc."
    }], "TableText"], Cell[TextData[{
     "notes in the ",
     Cell[BoxData[
      StyleBox["m", "TI"]], "InlineFormula"],
     " octave (",
     Cell[BoxData["\"\<C4\>"], "InlineFormula"],
     "\" is middle C)"
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{
      SubscriptBox[
       StyleBox["p", "TI"], 
       StyleBox["1", "TR"]], ",", 
      SubscriptBox[
       StyleBox["p", "TI"], 
       StyleBox["2", "TR"]], ",", 
      StyleBox["\[Ellipsis]", "TR"]}], "}"}], Cell[TextData[{
     "a chord containing pitches ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["p", "TI"], 
       StyleBox["i", "TI"]]], "InlineFormula"]
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["None",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/None"], Cell[
    "silence (a musical rest)", "TableText"]},
   {Cell[" ", "TableRowIcon"], "\"\<\!\(\*
StyleBox[\"percussion\", \"TI\"]\)\>\"", Cell[
    "a percussion event", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, None, None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{True, True, True, True, True, True, True, True}}}},
 CellID->31815],

Cell[TextData[{
 "The forms ",
 Cell[BoxData["\"\<C#\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<C\[Sharp]\>\""], "InlineFormula"],
 " (entered as ",
 Cell[BoxData["\"\<C\!\(\*Cell[TextData[ButtonBox[\"\\\\[Sharp]\",
BaseStyle->\"Link\",
ButtonData:>\"paclet:ref/character/Sharp\"]], \
\"InlineCharacterName\"]\)\>\""], "InlineFormula"],
 " ) and ",
 Cell[BoxData["\"\<CSharp\>\""], "InlineFormula"],
 " are all equivalent."
}], "Notes",
 CellID->10670],

Cell[TextData[{
 Cell[BoxData["\"\<Cb\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<C\[Flat]\>\""], "InlineFormula"],
 " (entered as ",
 Cell[BoxData["\"\<C\!\(\*Cell[TextData[ButtonBox[\"\\\\[Flat]\",
BaseStyle->\"Link\",
ButtonData:>\"paclet:ref/character/Flat\"]], \"InlineCharacterName\"]\)\>\""],
   "InlineFormula"],
 ") and ",
 Cell[BoxData["\"\<CFlat\>\""], "InlineFormula"],
 " are also equivalent."
}], "Notes",
 CellID->30696],

Cell["Negative numbers specify pitches below middle C.", "Notes",
 CellID->28254],

Cell[TextData[{
 "The C one octave above middle C can be specified as ",
 Cell[BoxData["12"], "InlineFormula"],
 " or ",
 Cell[BoxData["\"\<C5\>\""], "InlineFormula"],
 "."
}], "Notes",
 CellID->31293],

Cell[TextData[{
 Cell[BoxData["\"\<C+4\>\""], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData["\"\<C4\>\""], "InlineFormula"],
 "; low notes can be specified as ",
 Cell[BoxData["\"\<C-1\>\""], "InlineFormula"],
 " etc."
}], "Notes",
 CellID->129119450],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["SoundNote",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SoundNote"], "[", 
   RowBox[{
    StyleBox["pitch", "TI"], ",", 
    StyleBox["t", "TI"]}], "]"}]], "InlineFormula"],
 " specifies a total duration ",
 Cell[BoxData[
  StyleBox["t", "TI"]], "InlineFormula"],
 "; the actual sound of the note may decay before time ",
 Cell[BoxData[
  StyleBox["t", "TI"]], "InlineFormula"],
 " has elapsed."
}], "Notes",
 CellID->4300],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["SoundNote",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SoundNote"], "[", 
   RowBox[{
    StyleBox["pitch", "TI"], ",", 
    StyleBox["tspec", "TI"]}], "]"}]], "InlineFormula"],
 " normally is by default taken to be in a piano style."
}], "Notes",
 CellID->32573],

Cell[TextData[{
 "Style ",
 Cell[BoxData[
  StyleBox["s", "TI"]], "InlineFormula"],
 " between 1 and 128 represents General MIDI instrument ",
 Cell[BoxData[
  StyleBox["s", "TI"]], "InlineFormula"],
 "."
}], "Notes",
 CellID->12361],

Cell[TextData[{
 "Style ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    StyleBox["bank", "TI"], ",", 
    StyleBox["patch", "TI"], ",", 
    StyleBox["s", "TI"]}], "}"}]], "InlineFormula"],
 " represents an instrument in the specified bank and patch."
}], "Notes",
 CellID->4331],

Cell[TextData[{
 "In ",
 Cell[BoxData[
  ButtonBox["Sound",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Sound"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{"\"\<\!\(\*
StyleBox[\"style\", \"TI\"]\)\>\"", ",", 
    RowBox[{"SoundNote", "[", 
     StyleBox["\[Ellipsis]", "TR"], "]"}], ",", 
    StyleBox["\[Ellipsis]", "TR"]}], "}"}]], "InlineFormula"],
 " can be used to specify that many notes should be taken to be in the given \
style."
}], "Notes",
 CellID->19175],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["SoundNote",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SoundNote"], "[", "]"}]], "InlineFormula"],
 " by default represents middle C in piano style with duration 1 second."
}], "Notes",
 CellID->9143],

Cell["Typical possible styles include:", "Notes",
 CellID->133771515],

Cell[BoxData[GridBox[{
   {"\"\<Accordion\>\"", "\"\<Agogo\>\"", "\"\<AltoSax\>\"", 
    "\"\<Applause\>\""},
   {"\"\<Atmosphere\>\"", "\"\<Bagpipe\>\"", "\"\<Bandoneon\>\"", 
    "\"\<Banjo\>\""},
   {"\"\<BaritoneSax\>\"", "\"\<Bass\>\"", "\"\<BassAndLead\>\"", 
    "\"\<Bassoon\>\""},
   {"\"\<Bird\>\"", "\"\<BlownBottle\>\"", "\"\<Bowed\>\"", 
    "\"\<BrassSection\>\""},
   {"\"\<Breath\>\"", "\"\<Brightness\>\"", "\"\<BrightPiano\>\"", 
    "\"\<Calliope\>\""},
   {"\"\<Celesta\>\"", "\"\<Cello\>\"", "\"\<Charang\>\"", "\"\<Chiff\>\""},
   {"\"\<Choir\>\"", "\"\<Clarinet\>\"", "\"\<Clavi\>\"", 
    "\"\<Contrabass\>\""},
   {"\"\<Crystal\>\"", "\"\<DrawbarOrgan\>\"", "\"\<Dulcimer\>\"", 
    "\"\<Echoes\>\""},
   {"\"\<ElectricBass\>\"", "\"\<ElectricGrandPiano\>\"", 
    "\"\<ElectricGuitar\>\"", "\"\<ElectricPiano\>\""},
   {"\"\<ElectricPiano2\>\"", "\"\<EnglishHorn\>\"", "\"\<Fiddle\>\"", 
    "\"\<Fifths\>\""},
   {"\"\<Flute\>\"", "\"\<FrenchHorn\>\"", "\"\<FretlessBass\>\"", 
    "\"\<FretNoise\>\""},
   {"\"\<Glockenspiel\>\"", "\"\<Goblins\>\"", "\"\<Guitar\>\"", 
    "\"\<GuitarDistorted\>\""},
   {"\"\<GuitarHarmonics\>\"", "\"\<GuitarMuted\>\"", 
    "\"\<GuitarOverdriven\>\"", "\"\<Gunshot\>\""},
   {"\"\<Halo\>\"", "\"\<Harmonica\>\"", "\"\<Harp\>\"", 
    "\"\<Harpsichord\>\""},
   {"\"\<Helicopter\>\"", "\"\<HonkyTonkPiano\>\"", "\"\<JazzGuitar\>\"", 
    "\"\<Kalimba\>\""},
   {"\"\<Koto\>\"", "\"\<Marimba\>\"", "\"\<MelodicTom\>\"", 
    "\"\<Metallic\>\""},
   {"\"\<MusicBox\>\"", "\"\<MutedTrumpet\>\"", "\"\<NewAge\>\"", 
    "\"\<Oboe\>\""},
   {"\"\<Ocarina\>\"", "\"\<OrchestraHit\>\"", "\"\<Organ\>\"", 
    "\"\<PanFlute\>\""},
   {"\"\<PercussiveOrgan\>\"", "\"\<Piano\>\"", "\"\<Piccolo\>\"", 
    "\"\<PickedBass\>\""},
   {"\"\<PizzicatoStrings\>\"", "\"\<Polysynth\>\"", "\"\<Rain\>\"", 
    "\"\<Recorder\>\""},
   {"\"\<ReedOrgan\>\"", "\"\<ReverseCymbal\>\"", "\"\<RockOrgan\>\"", 
    "\"\<Sawtooth\>\""},
   {"\"\<SciFi\>\"", "\"\<Seashore\>\"", "\"\<Shakuhachi\>\"", 
    "\"\<Shamisen\>\""},
   {"\"\<Shanai\>\"", "\"\<Sitar\>\"", "\"\<SlapBass\>\"", 
    "\"\<SlapBass2\>\""},
   {"\"\<SopranoSax\>\"", "\"\<Soundtrack\>\"", "\"\<Square\>\"", 
    "\"\<Steeldrums\>\""},
   {"\"\<SteelGuitar\>\"", "\"\<Strings\>\"", "\"\<Strings2\>\"", 
    "\"\<Sweep\>\""},
   {"\"\<SynthBass\>\"", "\"\<SynthBass2\>\"", "\"\<SynthBrass\>\"", 
    "\"\<SynthBrass2\>\""},
   {"\"\<SynthDrum\>\"", "\"\<SynthStrings\>\"", "\"\<SynthStrings2\>\"", 
    "\"\<SynthVoice\>\""},
   {"\"\<Taiko\>\"", "\"\<Telephone\>\"", "\"\<TenorSax\>\"", 
    "\"\<Timpani\>\""},
   {"\"\<Tinklebell\>\"", "\"\<TremoloStrings\>\"", "\"\<Trombone\>\"", 
    "\"\<Trumpet\>\""},
   {"\"\<Tuba\>\"", "\"\<TubularBells\>\"", "\"\<Vibraphone\>\"", 
    "\"\<Viola\>\""},
   {"\"\<Violin\>\"", "\"\<Voice\>\"", "\"\<VoiceAahs\>\"", 
    "\"\<VoiceOohs\>\""},
   {"\"\<Warm\>\"", "\"\<Whistle\>\"", "\"\<Woodblock\>\"", 
    "\"\<Xylophone\>\""}
  },
  GridBoxAlignment->{
   "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
    "RowsIndexed" -> {}}]], "Notes",
 CellID->360898806],

Cell["Typical possible percussion events include:", "Notes",
 CellID->69401903],

Cell[BoxData[GridBox[{
   {"\"\<BassDrum\>\"", "\"\<BassDrum2\>\"", "\"\<BellTree\>\"", 
    "\"\<Cabasa\>\""},
   {"\"\<Castanets\>\"", "\"\<ChineseCymbal\>\"", "\"\<Clap\>\"", 
    "\"\<Claves\>\""},
   {"\"\<Cowbell\>\"", "\"\<CrashCymbal\>\"", "\"\<CrashCymbal2\>\"", 
    "\"\<ElectricSnare\>\""},
   {"\"\<GuiroLong\>\"", "\"\<GuiroShort\>\"", "\"\<HighAgogo\>\"", 
    "\"\<HighBongo\>\""},
   {"\"\<HighCongaMute\>\"", "\"\<HighCongaOpen\>\"", "\"\<HighFloorTom\>\"", 
    "\"\<HighTimbale\>\""},
   {"\"\<HighTom\>\"", "\"\<HighWoodblock\>\"", "\"\<HiHatClosed\>\"", 
    "\"\<HiHatOpen\>\""},
   {"\"\<HiHatPedal\>\"", "\"\<JingleBell\>\"", "\"\<LowAgogo\>\"", 
    "\"\<LowBongo\>\""},
   {"\"\<LowConga\>\"", "\"\<LowFloorTom\>\"", "\"\<LowTimbale\>\"", 
    "\"\<LowTom\>\""},
   {"\"\<LowWoodblock\>\"", "\"\<Maracas\>\"", "\"\<MetronomeBell\>\"", 
    "\"\<MetronomeClick\>\""},
   {"\"\<MidTom\>\"", "\"\<MidTom2\>\"", "\"\<MuteCuica\>\"", 
    "\"\<MuteSurdo\>\""},
   {"\"\<MuteTriangle\>\"", "\"\<OpenCuica\>\"", "\"\<OpenSurdo\>\"", 
    "\"\<OpenTriangle\>\""},
   {"\"\<RideBell\>\"", "\"\<RideCymbal\>\"", "\"\<RideCymbal2\>\"", 
    "\"\<ScratchPull\>\""},
   {"\"\<ScratchPush\>\"", "\"\<Shaker\>\"", "\"\<SideStick\>\"", 
    "\"\<Slap\>\""},
   {"\"\<Snare\>\"", "\"\<SplashCymbal\>\"", "\"\<SquareClick\>\"", 
    "\"\<Sticks\>\""},
   {"\"\<Tambourine\>\"", "\"\<Vibraslap\>\"", "\"\<WhistleLong\>\"", 
    "\"\<WhistleShort\>\""}
  },
  GridBoxAlignment->{
   "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
    "RowsIndexed" -> {}}]], "Notes",
 CellID->179316135],

Cell[TextData[{
 "The option ",
 Cell[BoxData[
  RowBox[{"SoundVolume", "->", 
   StyleBox["v", "TI"]}]], "InlineFormula"],
 " specifies an relative sound volume ",
 Cell[BoxData[
  StyleBox["v", "TI"]], "InlineFormula"],
 " for a note."
}], "Notes",
 CellID->726003106],

Cell["\<\
What notes can actually play as sounds may depend on the setup of your \
computer system.\
\>", "Notes",
 CellID->475884268]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->1351969],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(5)", "ExampleCount"]
}], "ExampleSection",
 CellID->202604790],

Cell["Produce a middle C:", "ExampleText",
 CellID->108123467],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"SoundNote", "[", "0", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->690209635],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[
         RectangleBox[{0., 0.43198696602798653`}, {0.95, 0.4438917279327484}],
         
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"1", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{1, " s"}]],
       Row[{
         Row[{1, " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[
   SoundNote[0]]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1825030]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->116191540],

Cell["Produce a middle G for 1 second in a violin style:", "ExampleText",
 CellID->714566645],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"SoundNote", "[", 
   RowBox[{"\"\<G\>\"", ",", "1", ",", "\"\<Violin\>\""}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19781668],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {Hue[-0.1599553194190092, 0.9, 0.7], 
        StyleBox[
         RectangleBox[{0., 0.5153202993613198}, {0.95, 0.5272250612660818}],
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"1", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{1, " s"}]],
       Row[{
         Row[{1, " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 40], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 40], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[
   SoundNote["G", 1, "Violin"]]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->181507460]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->314676768],

Cell["Produce a chord for 1 second:", "ExampleText",
 CellID->106519206],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"SoundNote", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"\"\<C\>\"", ",", "\"\<G\>\""}], "}"}], ",", "1", ",", 
    "\"\<Harpsichord\>\""}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2768197],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {Hue[-0.14074811787423214`, 0.9, 0.7], 
        StyleBox[{
          RectangleBox[{0., 0.43198696602798653`}, \
{0.95, 0.4438917279327484}], 
          RectangleBox[{0., 0.5153202993613198}, {0.95, 0.5272250612660818}]},
         
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"1", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{1, " s"}]],
       Row[{
         Row[{1, " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 6], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 6], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[
   SoundNote[{"C", "G"}, 1, "Harpsichord"]]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->52871950]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->212688541],

Cell["Produce a sequence of three notes:", "ExampleText",
 CellID->88221508],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"SoundNote", "[", "\"\<C\>\"", "]"}], ",", 
    RowBox[{"SoundNote", "[", "\"\<G\>\"", "]"}], ",", 
    RowBox[{"SoundNote", "[", "\"\<C5\>\"", "]"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->701800872],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[{
          RectangleBox[{0., 0.43198696602798653`}, \
{0.95, 0.4438917279327484}], 
          RectangleBox[{1., 0.5153202993613198}, {1.95, 0.5272250612660818}], 
          RectangleBox[{2., 0.5748441088851295}, {2.95, 0.5867488707898912}]},
         
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"3", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{3, " s"}]],
       Row[{
         Row[{3, " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 2, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 2, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {-164, 0}},
   PlotRangePadding->1],
  Sound[{
    SoundNote["C"], 
    SoundNote["G"], 
    SoundNote["C5"]}]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->85083618]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->654504802],

Cell["Produce the sound of a bell tree:", "ExampleText",
 CellID->804139014],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"SoundNote", "[", "\"\<BellTree\>\"", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->903232243],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[
         RectangleBox[{0., 0.7177012517422724}, {0.95, 0.7296060136470344}],
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"1", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{1, " s"}]],
       Row[{
         Row[{1, " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 9, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 84, "Channel" -> 9, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 84, "Channel" -> 9, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 9, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 84, "Channel" -> 9, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 84, "Channel" -> 9, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[
   SoundNote["BellTree"]]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->77763871]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(14)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->9374],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Basic Notes and Chords",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(6)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->793326280],

Cell["The note 7 semitones above middle C:", "ExampleText",
 CellID->117974802],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"SoundNote", "[", "7", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->67249194],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[
         RectangleBox[{0., 0.5153202993613198}, {0.95, 0.5272250612660818}],
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"1", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{1, " s"}]],
       Row[{
         Row[{1, " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[
   SoundNote[7]]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->701233626]
}, Open  ]],

Cell["The note 7 semitones below middle C:", "ExampleText",
 CellID->172890070],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"SoundNote", "[", 
   RowBox[{"-", "7"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->367949343],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[
         RectangleBox[{0., 0.34865363269465305`}, {0.95, 0.3605583945994151}],
         
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"1", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{1, " s"}]],
       Row[{
         Row[{1, " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 53, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 53, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 53, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 53, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[
   SoundNote[-7]]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->125515990]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->323930242],

Cell["The G above middle C:", "ExampleText",
 CellID->24225762],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"SoundNote", "[", "\"\<G\>\"", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->93228870],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[
         RectangleBox[{0., 0.5153202993613198}, {0.95, 0.5272250612660818}],
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"1", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{1, " s"}]],
       Row[{
         Row[{1, " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[
   SoundNote["G"]]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->259920389]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->34259455],

Cell["Middle C:", "ExampleText",
 CellID->499030287],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"SoundNote", "[", "\"\<C4\>\"", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->454451608],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[
         RectangleBox[{0., 0.43198696602798653`}, {0.95, 0.4438917279327484}],
         
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"1", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{1, " s"}]],
       Row[{
         Row[{1, " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[
   SoundNote["C4"]]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->532719665]
}, Open  ]],

Cell["One octave above middle C:", "ExampleText",
 CellID->74519669],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"SoundNote", "[", "\"\<C5\>\"", "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->188282611],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[
         RectangleBox[{0., 0.5748441088851295}, {0.95, 0.5867488707898912}],
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"1", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{1, " s"}]],
       Row[{
         Row[{1, " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[
   SoundNote["C5"]]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->90480637]
}, Open  ]],

Cell["One octave below middle C:", "ExampleText",
 CellID->285967126],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"SoundNote", "[", "\"\<C3\>\"", "]"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->185619107],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[
         RectangleBox[{0., 0.2891298231708436}, {0.95, 0.3010345850756056}],
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"1", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{1, " s"}]],
       Row[{
         Row[{1, " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[
   SoundNote["C3"]]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->286581249]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->217100863],

Cell["A chord of C and G:", "ExampleText",
 CellID->260011936],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"SoundNote", "[", 
   RowBox[{"{", 
    RowBox[{"\"\<C\>\"", ",", "\"\<G\>\""}], "}"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->35700781],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[{
          RectangleBox[{0., 0.43198696602798653`}, \
{0.95, 0.4438917279327484}], 
          RectangleBox[{0., 0.5153202993613198}, {0.95, 0.5272250612660818}]},
         
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"1", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{1, " s"}]],
       Row[{
         Row[{1, " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[
   SoundNote[{"C", "G"}]]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->79394894]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->228695725],

Cell["A chord of C, E, G, and B\[Flat]:", "ExampleText",
 CellID->461969822],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"SoundNote", "[", 
   RowBox[{"{", 
    RowBox[{
    "\"\<C\>\"", ",", "\"\<E\>\"", ",", "\"\<G\>\"", ",", "\"\<Bb\>\""}], 
    "}"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->651703356],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[{
          RectangleBox[{0., 0.43198696602798653`}, \
{0.95, 0.4438917279327484}], 
          RectangleBox[{0., 0.4796060136470343}, \
{0.95, 0.49151077555179606`}], 
          RectangleBox[{0., 0.5153202993613198}, {0.95, 0.5272250612660818}], 
          RectangleBox[{0., 0.5510345850756055}, {0.95, 0.5629393469803675}]},
         
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"1", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{1, " s"}]],
       Row[{
         Row[{1, " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[
   SoundNote[{"C", "E", "G", "Bb"}]]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->227768562]
}, Open  ]],

Cell["A chord of D, F\[Sharp], A, and D:", "ExampleText",
 CellID->65294128],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"SoundNote", "[", 
   RowBox[{"{", 
    RowBox[{
    "\"\<D\>\"", ",", "\"\<F#\>\"", ",", "\"\<A\>\"", ",", "\"\<D6\>\""}], 
    "}"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->579743871],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[{
          RectangleBox[{0., 0.45579648983751037`}, \
{0.95, 0.4677012517422721}], 
          RectangleBox[{0., 0.5034155374565581}, {0.95, 0.5153202993613198}], 
          RectangleBox[{0., 0.5391298231708438}, {0.95, 0.5510345850756055}], 
          RectangleBox[{0., 0.7415107755517961}, {0.95, 0.7534155374565579}]},
         
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"1", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{1, " s"}]],
       Row[{
         Row[{1, " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 86, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 86, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 86, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 86, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[
   SoundNote[{"D", "F#", "A", "D6"}]]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->883613282]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->94975604],

Cell["Low and high C notes:", "ExampleText",
 CellID->207328570],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"SoundNote", "[", "\"\<C0\>\"", "]"}], ",", 
    RowBox[{"SoundNote", "[", "\"\<C9\>\"", "]"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->608930050],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[{
          RectangleBox[{0., -0.13944160540058487`}, \
{0.95, -0.127536843495823}], 
          RectangleBox[{1., 1.1462726803137009`}, \
{1.95, 1.1581774422184625`}]},
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"2", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{2, " s"}]],
       Row[{
         Row[{2, " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 12, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 12, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOn", "Note" -> 120, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 120, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 12, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 12, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOn", "Note" -> 120, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 120, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[{
    SoundNote["C0"], 
    SoundNote["C9"]}]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->148631695]
}, Open  ]],

Cell["Lowest C typically supported:", "ExampleText",
 CellID->36500127],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"SoundNote", "[", "\"\<C-1\>\"", "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->201698962],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[
         RectangleBox[{0., -0.2822987482577278}, \
{0.95, -0.27039398635296596`}],
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"1", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{1, " s"}]],
       Row[{
         Row[{1, " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 0, "Channel" -> 0, "Velocity" -> 127],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 0, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 0, "Channel" -> 0, "Velocity" -> 127],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 0, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[
   SoundNote["C-1"]]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->142102853]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Timing",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->65845541],

Cell["C for 0.2 seconds, the G for 0.3 seconds:", "ExampleText",
 CellID->303449142],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"SoundNote", "[", 
     RowBox[{"\"\<C\>\"", ",", "0.2"}], "]"}], ",", 
    RowBox[{"SoundNote", "[", 
     RowBox[{"\"\<G\>\"", ",", "0.3"}], "]"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->188841332],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[{
          RectangleBox[{0., 0.43198696602798653`}, \
{0.19, 0.4438917279327484}], 
          RectangleBox[{0.2, 0.5153202993613198}, \
{0.485, 0.5272250612660818}]},
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"\<\"0.5\"\>", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{"0.5", " s"}]],
       Row[{
         Row[{"0.5", " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 2, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 10] =!= 
            None, 
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 2, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 10]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[{
    SoundNote["C", 0.2], 
    SoundNote["G", 0.3]}]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->284965416]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->594028497],

Cell["Notes with a rest in the middle:", "ExampleText",
 CellID->576630885],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"SoundNote", "[", 
     RowBox[{"\"\<C\>\"", ",", "0.2"}], "]"}], ",", 
    RowBox[{"SoundNote", "[", 
     RowBox[{"None", ",", "0.2"}], "]"}], ",", 
    RowBox[{"SoundNote", "[", 
     RowBox[{"\"\<G\>\"", ",", "0.3"}], "]"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->44056557],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[{
          RectangleBox[{0., 0.43198696602798653`}, \
{0.19, 0.4438917279327484}], 
          RectangleBox[{0.4, 0.5153202993613198}, \
{0.6849999999999999, 0.5272250612660818}]},
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"\<\"0.7\"\>", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{"0.7", " s"}]],
       Row[{
         Row[{"0.7", " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 4, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 7, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 10] =!= 
            None, 
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 4, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 7, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 10]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[{
    SoundNote["C", 0.2], 
    SoundNote[None, 0.2], 
    SoundNote["G", 0.3]}]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->235247701]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->136274979],

Cell["Specify absolute times, with an overlap in the middle:", "ExampleText",
 CellID->137808773],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"SoundNote", "[", 
     RowBox[{"\"\<C\>\"", ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "0.3"}], "}"}]}], "]"}], ",", 
    RowBox[{"SoundNote", "[", 
     RowBox[{"\"\<G\>\"", ",", 
      RowBox[{"{", 
       RowBox[{"0.1", ",", "0.5"}], "}"}]}], "]"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->520380054],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[{
          RectangleBox[{0., 0.43198696602798653`}, \
{0.285, 0.4438917279327484}], 
          RectangleBox[{0.1, 0.5153202993613198}, \
{0.48, 0.5272250612660818}]},
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"\<\"0.5\"\>", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{"0.5", " s"}]],
       Row[{
         Row[{"0.5", " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 5, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 10] =!= 
            None, 
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 5, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 10]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[{
    SoundNote["C", {0, 0.3}], 
    SoundNote["G", {0.1, 0.5}]}]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->386566153]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Instrumentation",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(7)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->151168849],

Cell["Middle C on the organ for 1 second:", "ExampleText",
 CellID->139672495],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"SoundNote", "[", 
   RowBox[{"\"\<C\>\"", ",", "1", ",", "\"\<Organ\>\""}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->178763726],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {Hue[-0.19103322434723669`, 0.9, 0.7], 
        StyleBox[
         RectangleBox[{0., 0.43198696602798653`}, {0.95, 0.4438917279327484}],
         
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"1", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{1, " s"}]],
       Row[{
         Row[{1, " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 19], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 19], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[
   SoundNote["C", 1, "Organ"]]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->278040955]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->69789477],

Cell["A sequence of two notes, on two instruments:", "ExampleText",
 CellID->110887593],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"SoundNote", "[", 
     RowBox[{"\"\<C\>\"", ",", ".2", ",", "\"\<Tuba\>\""}], "]"}], ",", 
    RowBox[{"SoundNote", "[", 
     RowBox[{"\"\<G\>\"", ",", ".3", ",", "\"\<Cello\>\""}], "]"}]}], "}"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->13674992],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[{
        {Hue[0.22538543343129763`, 0.9, 0.7], 
         StyleBox[
          RectangleBox[{0.2, 0.5153202993613198}, \
{0.485, 0.5272250612660818}],
          Antialiasing->True]}, 
        {Hue[0.3881114562337579, 0.9, 0.7], 
         StyleBox[
          RectangleBox[{0., 0.43198696602798653`}, \
{0.19, 0.4438917279327484}],
          Antialiasing->True]}},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"\<\"0.5\"\>", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{"0.5", " s"}]],
       Row[{
         Row[{"0.5", " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 42], 
                 Sound`MIDIEvent[
                 2, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 1, "Value" -> 58], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 1, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 60, "Channel" -> 1, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 10] =!= 
            None, 
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 42], 
                 Sound`MIDIEvent[
                 2, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 1, "Value" -> 58], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 1, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 60, "Channel" -> 1, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 10]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[{
    SoundNote["C", 0.2, "Tuba"], 
    SoundNote["G", 0.3, "Cello"]}]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->461283651]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->398357265],

Cell["Notes on two instruments, with overlap:", "ExampleText",
 CellID->333445715],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"SoundNote", "[", 
     RowBox[{"\"\<C\>\"", ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "0.3"}], "}"}], ",", "\"\<Tuba\>\""}], "]"}], ",", 
    RowBox[{"SoundNote", "[", 
     RowBox[{"\"\<G\>\"", ",", 
      RowBox[{"{", 
       RowBox[{"0.1", ",", "0.5"}], "}"}], ",", "\"\<Cello\>\""}], "]"}]}], 
   "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->41520424],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[{
        {Hue[0.22538543343129763`, 0.9, 0.7], 
         StyleBox[
          RectangleBox[{0.1, 0.5153202993613198}, {0.48, 0.5272250612660818}],
          
          Antialiasing->True]}, 
        {Hue[0.3881114562337579, 0.9, 0.7], 
         StyleBox[
          RectangleBox[{0., 0.43198696602798653`}, \
{0.285, 0.4438917279327484}],
          Antialiasing->True]}},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"\<\"0.5\"\>", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{"0.5", " s"}]],
       Row[{
         Row[{"0.5", " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 42], 
                 Sound`MIDIEvent[
                 1, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 1, "Value" -> 58], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 1, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOff", "Note" -> 60, "Channel" -> 1, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 10] =!= 
            None, 
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 42], 
                 Sound`MIDIEvent[
                 1, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 1, "Value" -> 58], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 1, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOff", "Note" -> 60, "Channel" -> 1, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 10]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[{
    SoundNote["C", {0, 0.3}, "Tuba"], 
    SoundNote["G", {0.1, 0.5}, "Cello"]}]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->788909719]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->264388839],

Cell["Give styles for multiple notes:", "ExampleText",
 CellID->224690550],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"{", 
   RowBox[{"\"\<Tuba\>\"", ",", 
    RowBox[{"SoundNote", "[", "0", "]"}], ",", 
    RowBox[{"SoundNote", "[", "1", "]"}], ",", "\"\<Cello\>\"", ",", 
    RowBox[{"SoundNote", "[", "2", "]"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->371217081],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[{
        {Hue[0.22538543343129763`, 0.9, 0.7], 
         StyleBox[
          RectangleBox[{2., 0.45579648983751037`}, \
{2.95, 0.4677012517422721}],
          Antialiasing->True]}, 
        {Hue[0.3881114562337579, 0.9, 0.7], 
         StyleBox[{
           RectangleBox[{0., 0.43198696602798653`}, \
{0.95, 0.4438917279327484}], 
           RectangleBox[{1., 0.4438917279327484}, \
{1.95, 0.45579648983751037`}]},
          Antialiasing->True]}},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"3", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{3, " s"}]],
       Row[{
         Row[{3, " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 42], 
                 Sound`MIDIEvent[
                 2, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 1, "Value" -> 58], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 1, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 1, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOn", "Note" -> 61, "Channel" -> 1, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 61, "Channel" -> 1, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 42], 
                 Sound`MIDIEvent[
                 2, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 1, "Value" -> 58], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 1, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 1, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOn", "Note" -> 61, "Channel" -> 1, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 61, "Channel" -> 1, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[{"Tuba", 
    SoundNote[0], 
    SoundNote[1], "Cello", 
    SoundNote[2]}]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->23075324]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->936293599],

Cell[TextData[{
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " supports many instruments and sound effects:"
}], "ExampleText",
 CellID->477514323],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"SoundNote", "[", 
   RowBox[{"\"\<C\>\"", ",", "1", ",", "\"\<PanFlute\>\""}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->345467706],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {Hue[0.378507855461369, 0.9, 0.7], 
        StyleBox[
         RectangleBox[{0., 0.43198696602798653`}, {0.95, 0.4438917279327484}],
         
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"1", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{1, " s"}]],
       Row[{
         Row[{1, " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 75], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 75], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[
   SoundNote["C", 1, "PanFlute"]]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->500844854]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"SoundNote", "[", 
   RowBox[{"\"\<C\>\"", ",", "1", ",", "\"\<Voice\>\""}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->90695030],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {Hue[0.3474299505331442, 0.9, 0.7], 
        StyleBox[
         RectangleBox[{0., 0.43198696602798653`}, {0.95, 0.4438917279327484}],
         
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"1", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{1, " s"}]],
       Row[{
         Row[{1, " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 54], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 54], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[
   SoundNote["C", 1, "Voice"]]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->165537014]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"SoundNote", "[", 
   RowBox[{"\"\<C\>\"", ",", "1", ",", "\"\<Telephone\>\""}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->119585382],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {Hue[-0.03564369970609926, 0.9, 0.7], 
        StyleBox[
         RectangleBox[{0., 0.43198696602798653`}, {0.95, 0.4438917279327484}],
         
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"1", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{1, " s"}]],
       Row[{
         Row[{1, " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 124], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 124], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[
   SoundNote["C", 1, "Telephone"]]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->16028380]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->45046013],

Cell["Percussion is specified without a pitch:", "ExampleText",
 CellID->166729684],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"SoundNote", "[", "\"\<Snare\>\"", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->126948112],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[
         RectangleBox[{0., 0.1700822041232246}, {0.95, 0.18198696602798659`}],
         
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"1", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{1, " s"}]],
       Row[{
         Row[{1, " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 9, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 38, "Channel" -> 9, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 38, "Channel" -> 9, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 9, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 38, "Channel" -> 9, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 38, "Channel" -> 9, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[
   SoundNote["Snare"]]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->83281232]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->351332568],

Cell["Play middle C for 1 second using MIDI instrument 99:", "ExampleText",
 CellID->537980816],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"SoundNote", "[", 
   RowBox[{"\"\<C\>\"", ",", "1", ",", "99"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->137084481],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {Hue[0.06492651323989562, 0.9, 0.7], 
        StyleBox[
         RectangleBox[{0., 0.43198696602798653`}, {0.95, 0.4438917279327484}],
         
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"1", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{1, " s"}]],
       Row[{
         Row[{1, " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 98], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 98], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[
   SoundNote["C", 1, 99]]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->237581513]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Options",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->349216433],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "SoundVolume",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->136293263],

Cell["Play a sequence of progressively louder notes:", "ExampleText",
 CellID->322677866],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"SoundNote", "[", 
     RowBox[{"\"\<C\>\"", ",", ".2", ",", 
      RowBox[{"SoundVolume", "\[Rule]", "v"}]}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"v", ",", "0", ",", "1", ",", "0.1"}], "}"}]}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->561880915],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[{
          RectangleBox[{0., 0.43198696602798653`}, \
{0.19, 0.4438917279327484}], 
          RectangleBox[{0.2, 0.43198696602798653`}, \
{0.39, 0.4438917279327484}], 
          RectangleBox[{0.4, 0.43198696602798653`}, \
{0.59, 0.4438917279327484}], 
          RectangleBox[{0.6, 0.43198696602798653`}, \
{0.79, 0.4438917279327484}], 
          RectangleBox[{0.8, 0.43198696602798653`}, \
{0.99, 0.4438917279327484}], 
          RectangleBox[{1., 0.43198696602798653`}, \
{1.19, 0.4438917279327484}], 
          RectangleBox[{1.2, 0.43198696602798653`}, \
{1.39, 0.4438917279327484}], 
          RectangleBox[{1.4, 0.43198696602798653`}, \
{1.59, 0.4438917279327484}], 
          RectangleBox[{1.6, 0.43198696602798653`}, \
{1.79, 0.4438917279327484}], 
          RectangleBox[{1.8, 0.43198696602798653`}, \
{1.99, 0.4438917279327484}], 
          RectangleBox[{2., 0.43198696602798653`}, \
{2.1900000000000004`, 0.4438917279327484}]},
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"\<\"2.2\"\>", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{"2.2", " s"}]],
       Row[{
         Row[{"2.2", " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0], 
                 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 12],
                  
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 2, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 25],
                  
                 Sound`MIDIEvent[
                 3, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 3, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 38],
                  
                 Sound`MIDIEvent[
                 4, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 4, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 50],
                  
                 Sound`MIDIEvent[
                 5, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 5, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 63],
                  
                 Sound`MIDIEvent[
                 6, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 6, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 76],
                  
                 Sound`MIDIEvent[
                 7, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 7, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 88],
                  
                 Sound`MIDIEvent[
                 8, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 8, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  101], 
                 Sound`MIDIEvent[
                 9, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 9, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  114], 
                 Sound`MIDIEvent[
                 10, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 10, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 11, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 5] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0], 
                 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 12],
                  
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 2, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 25],
                  
                 Sound`MIDIEvent[
                 3, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 3, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 38],
                  
                 Sound`MIDIEvent[
                 4, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 4, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 50],
                  
                 Sound`MIDIEvent[
                 5, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 5, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 63],
                  
                 Sound`MIDIEvent[
                 6, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 6, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 76],
                  
                 Sound`MIDIEvent[
                 7, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 7, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 88],
                  
                 Sound`MIDIEvent[
                 8, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 8, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  101], 
                 Sound`MIDIEvent[
                 9, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 9, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  114], 
                 Sound`MIDIEvent[
                 10, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 10, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 11, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 5]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[{
    SoundNote["C", 0.2, SoundVolume -> 0.], 
    SoundNote["C", 0.2, SoundVolume -> 0.1], 
    SoundNote["C", 0.2, SoundVolume -> 0.2], 
    SoundNote["C", 0.2, SoundVolume -> 0.30000000000000004`], 
    SoundNote["C", 0.2, SoundVolume -> 0.4], 
    SoundNote["C", 0.2, SoundVolume -> 0.5], 
    SoundNote["C", 0.2, SoundVolume -> 0.6000000000000001], 
    SoundNote["C", 0.2, SoundVolume -> 0.7000000000000001], 
    SoundNote["C", 0.2, SoundVolume -> 0.8], 
    SoundNote["C", 0.2, SoundVolume -> 0.9], 
    SoundNote["C", 0.2, SoundVolume -> 1.]}]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->75785747]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(7)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->30057],

Cell["Play a chromatic scale starting at middle C:", "ExampleText",
 CellID->398354286],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"SoundNote", "[", "i", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"i", ",", "0", ",", "12"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->215259133],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[{
          RectangleBox[{0., 0.43198696602798653`}, \
{0.95, 0.4438917279327484}], 
          RectangleBox[{1., 0.4438917279327484}, \
{1.95, 0.45579648983751037`}], 
          RectangleBox[{2., 0.45579648983751037`}, \
{2.95, 0.4677012517422721}], 
          RectangleBox[{3., 0.4677012517422721}, {3.95, 0.4796060136470343}], 
          RectangleBox[{4., 0.4796060136470343}, \
{4.95, 0.49151077555179606`}], 
          RectangleBox[{5., 0.49151077555179606`}, \
{5.95, 0.5034155374565581}], 
          RectangleBox[{6., 0.5034155374565581}, {6.95, 0.5153202993613198}], 
          RectangleBox[{7., 0.5153202993613198}, {7.95, 0.5272250612660818}], 
          RectangleBox[{8., 0.5272250612660818}, {8.95, 0.5391298231708438}], 
          RectangleBox[{9., 0.5391298231708438}, {9.95, 0.5510345850756055}], 
          RectangleBox[{10., 0.5510345850756055}, \
{10.95, 0.5629393469803675}], 
          RectangleBox[{11., 0.5629393469803675}, \
{11.95, 0.5748441088851295}], 
          RectangleBox[{12., 0.5748441088851295}, \
{12.95, 0.5867488707898912}]},
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"13", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{13, " s"}]],
       Row[{
         Row[{13, " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOn", "Note" -> 61, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 61, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 2, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 3, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 4, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 4, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 5, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 6, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 6, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 7, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 7, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 8, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 8, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 9, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 9, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 10, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 10, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 11, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 11, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 12, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 12, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 13, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOn", "Note" -> 61, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 61, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 2, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 3, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 4, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 4, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 5, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 6, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 6, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 7, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 7, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 8, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 8, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 9, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 9, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 10, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 10, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 11, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 11, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 12, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 12, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 13, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[{
    SoundNote[0], 
    SoundNote[1], 
    SoundNote[2], 
    SoundNote[3], 
    SoundNote[4], 
    SoundNote[5], 
    SoundNote[6], 
    SoundNote[7], 
    SoundNote[8], 
    SoundNote[9], 
    SoundNote[10], 
    SoundNote[11], 
    SoundNote[12]}]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->9648212]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->462149],

Cell["Play a scale on a violin, with each note taking 0.1 seconds:", \
"ExampleText",
 CellID->672107903],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"SoundNote", "[", 
     RowBox[{"i", ",", "0.1", ",", "\"\<Violin\>\""}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"i", ",", "0", ",", "12"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->125790237],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {Hue[-0.1599553194190092, 0.9, 0.7], 
        StyleBox[{
          RectangleBox[{0., 0.43198696602798653`}, \
{0.095, 0.4438917279327484}], 
          RectangleBox[{0.1, 0.4438917279327484}, \
{0.195, 0.45579648983751037`}], 
          RectangleBox[{0.2, 0.45579648983751037`}, \
{0.295, 0.4677012517422721}], 
          RectangleBox[{0.3, 0.4677012517422721}, \
{0.395, 0.4796060136470343}], 
          RectangleBox[{0.4, 0.4796060136470343}, \
{0.495, 0.49151077555179606`}], 
          RectangleBox[{0.5, 0.49151077555179606`}, \
{0.595, 0.5034155374565581}], 
          RectangleBox[{0.6, 0.5034155374565581}, \
{0.695, 0.5153202993613198}], 
          RectangleBox[{0.7, 0.5153202993613198}, \
{0.795, 0.5272250612660818}], 
          RectangleBox[{0.8, 0.5272250612660818}, \
{0.895, 0.5391298231708438}], 
          RectangleBox[{0.9, 0.5391298231708438}, \
{0.995, 0.5510345850756055}], 
          RectangleBox[{1., 0.5510345850756055}, \
{1.0950000000000002`, 0.5629393469803675}], 
          RectangleBox[{1.1, 0.5629393469803675}, \
{1.195, 0.5748441088851295}], 
          RectangleBox[{1.2, 0.5748441088851295}, \
{1.295, 0.5867488707898912}]},
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"\<\"1.3\"\>", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{"1.3", " s"}]],
       Row[{
         Row[{"1.3", " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 40], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOn", "Note" -> 61, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 61, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 2, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 3, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 4, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 4, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 5, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 6, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 6, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 7, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 7, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 8, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 8, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 9, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 9, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 10, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 10, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 11, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 11, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 12, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 12, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 13, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 10] =!= 
            None, 
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 40], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOn", "Note" -> 61, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 61, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 2, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 3, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 4, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 4, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 5, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 6, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 6, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 7, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 7, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 8, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 8, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 9, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 9, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 10, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 10, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 11, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 11, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 12, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 12, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 13, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 10]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[{
    SoundNote[0, 0.1, "Violin"], 
    SoundNote[1, 0.1, "Violin"], 
    SoundNote[2, 0.1, "Violin"], 
    SoundNote[3, 0.1, "Violin"], 
    SoundNote[4, 0.1, "Violin"], 
    SoundNote[5, 0.1, "Violin"], 
    SoundNote[6, 0.1, "Violin"], 
    SoundNote[7, 0.1, "Violin"], 
    SoundNote[8, 0.1, "Violin"], 
    SoundNote[9, 0.1, "Violin"], 
    SoundNote[10, 0.1, "Violin"], 
    SoundNote[11, 0.1, "Violin"], 
    SoundNote[12, 0.1, "Violin"]}]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->61475641]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->363938610],

Cell["Play the first 20 primes as notes in woodblock style:", "ExampleText",
 CellID->306183704],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"SoundNote", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"Prime", "[", "i", "]"}], "-", "20"}], ",", "0.1", ",", 
      "\"\<Woodblock\>\""}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"i", ",", "20"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->268546659],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {Hue[0.05532291246752746, 0.9, 0.7], 
        StyleBox[{
          RectangleBox[{0., 0.2177012517422722}, \
{0.095, 0.22960601364703406`}], 
          RectangleBox[{0.1, 0.22960601364703406`}, \
{0.195, 0.24151077555179604`}], 
          RectangleBox[{0.2, 0.25341553745655804`}, \
{0.295, 0.2653202993613199}], 
          RectangleBox[{0.3, 0.27722506126608165`}, \
{0.395, 0.2891298231708436}], 
          RectangleBox[{0.4, 0.3248441088851295}, \
{0.495, 0.33674887078989135`}], 
          RectangleBox[{0.5, 0.34865363269465305`}, \
{0.595, 0.3605583945994151}], 
          RectangleBox[{0.6, 0.3962726803137009}, \
{0.695, 0.40817744221846264`}], 
          RectangleBox[{0.7, 0.4200822041232246}, \
{0.795, 0.43198696602798653`}], 
          RectangleBox[{0.8, 0.4677012517422721}, \
{0.895, 0.4796060136470343}], 
          RectangleBox[{0.9, 0.5391298231708438}, \
{0.995, 0.5510345850756055}], 
          RectangleBox[{1., 0.5629393469803675}, \
{1.0950000000000002`, 0.5748441088851295}], 
          RectangleBox[{1.1, 0.6343679184089389}, {1.195, 0.646272680313701}],
           RectangleBox[{1.2, 0.6819869660279864}, \
{1.295, 0.6938917279327484}], 
          RectangleBox[{1.3, 0.7057964898375104}, \
{1.395, 0.7177012517422724}], 
          RectangleBox[{1.4, 0.7534155374565579}, \
{1.495, 0.7653202993613198}], 
          RectangleBox[{1.5, 0.8248441088851293}, \
{1.5950000000000002`, 0.8367488707898912}], 
          RectangleBox[{1.6, 0.896272680313701}, {1.695, 0.9081774422184627}],
           RectangleBox[{1.7, 0.9200822041232247}, \
{1.795, 0.9319869660279865}], 
          RectangleBox[{1.8, 0.9915107755517961}, {1.895, 1.003415537456558}],
           RectangleBox[{1.9, 1.039129823170844}, \
{1.995, 1.0510345850756055`}]},
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"2", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{2, " s"}]],
       Row[{
         Row[{2, " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 115], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 42, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOn", "Note" -> 43, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 43, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 2, "NoteOn", "Note" -> 45, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOff", "Note" -> 45, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 3, "NoteOn", "Note" -> 47, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 4, "NoteOff", "Note" -> 47, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 4, "NoteOn", "Note" -> 51, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOff", "Note" -> 51, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 5, "NoteOn", "Note" -> 53, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 6, "NoteOff", "Note" -> 53, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 6, "NoteOn", "Note" -> 57, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 7, "NoteOff", "Note" -> 57, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 7, "NoteOn", "Note" -> 59, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 8, "NoteOff", "Note" -> 59, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 8, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 9, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 9, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 10, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 10, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 11, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 11, "NoteOn", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 12, "NoteOff", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 12, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 13, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 13, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 14, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 14, "NoteOn", "Note" -> 87, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 15, "NoteOff", "Note" -> 87, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 15, "NoteOn", "Note" -> 93, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 16, "NoteOff", "Note" -> 93, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 16, "NoteOn", "Note" -> 99, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 17, "NoteOff", "Note" -> 99, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 17, "NoteOn", "Note" -> 101, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 18, "NoteOff", "Note" -> 101, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 18, "NoteOn", "Note" -> 107, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 19, "NoteOff", "Note" -> 107, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 19, "NoteOn", "Note" -> 111, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 20, "NoteOff", "Note" -> 111, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 10] =!= 
            None, 
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 115], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 42, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOn", "Note" -> 43, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 43, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 2, "NoteOn", "Note" -> 45, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOff", "Note" -> 45, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 3, "NoteOn", "Note" -> 47, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 4, "NoteOff", "Note" -> 47, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 4, "NoteOn", "Note" -> 51, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOff", "Note" -> 51, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 5, "NoteOn", "Note" -> 53, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 6, "NoteOff", "Note" -> 53, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 6, "NoteOn", "Note" -> 57, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 7, "NoteOff", "Note" -> 57, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 7, "NoteOn", "Note" -> 59, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 8, "NoteOff", "Note" -> 59, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 8, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 9, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 9, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 10, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 10, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 11, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 11, "NoteOn", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 12, "NoteOff", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 12, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 13, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 13, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 14, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 14, "NoteOn", "Note" -> 87, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 15, "NoteOff", "Note" -> 87, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 15, "NoteOn", "Note" -> 93, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 16, "NoteOff", "Note" -> 93, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 16, "NoteOn", "Note" -> 99, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 17, "NoteOff", "Note" -> 99, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 17, "NoteOn", "Note" -> 101, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 18, "NoteOff", "Note" -> 101, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 18, "NoteOn", "Note" -> 107, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 19, "NoteOff", "Note" -> 107, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 19, "NoteOn", "Note" -> 111, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 20, "NoteOff", "Note" -> 111, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 10]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[{
    SoundNote[-18, 0.1, "Woodblock"], 
    SoundNote[-17, 0.1, "Woodblock"], 
    SoundNote[-15, 0.1, "Woodblock"], 
    SoundNote[-13, 0.1, "Woodblock"], 
    SoundNote[-9, 0.1, "Woodblock"], 
    SoundNote[-7, 0.1, "Woodblock"], 
    SoundNote[-3, 0.1, "Woodblock"], 
    SoundNote[-1, 0.1, "Woodblock"], 
    SoundNote[3, 0.1, "Woodblock"], 
    SoundNote[9, 0.1, "Woodblock"], 
    SoundNote[11, 0.1, "Woodblock"], 
    SoundNote[17, 0.1, "Woodblock"], 
    SoundNote[21, 0.1, "Woodblock"], 
    SoundNote[23, 0.1, "Woodblock"], 
    SoundNote[27, 0.1, "Woodblock"], 
    SoundNote[33, 0.1, "Woodblock"], 
    SoundNote[39, 0.1, "Woodblock"], 
    SoundNote[41, 0.1, "Woodblock"], 
    SoundNote[47, 0.1, "Woodblock"], 
    SoundNote[51, 0.1, "Woodblock"]}]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->15889590]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->583997264],

Cell["Play the first notes in Beethoven's Fifth Symphony:", "ExampleText",
 CellID->43962524],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"SoundNote", "[", "\"\<G\>\"", "]"}], ",", 
     RowBox[{"SoundNote", "[", "\"\<G\>\"", "]"}], ",", 
     RowBox[{"SoundNote", "[", "\"\<G\>\"", "]"}], ",", 
     RowBox[{"SoundNote", "[", 
      RowBox[{"\"\<Eb\>\"", ",", "4"}], "]"}]}], "}"}], ",", "1.5"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->156060250],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[{
          RectangleBox[{0.6428571428571429, 0.4677012517422721}, \
{1.457142857142857, 0.4796060136470343}], 
          RectangleBox[{0., 0.5153202993613198}, \
{0.20357142857142857`, 0.5272250612660818}], 
          RectangleBox[{0.21428571428571427`, 0.5153202993613198}, \
{0.4178571428571428, 0.5272250612660818}], 
          RectangleBox[{0.42857142857142855`, 0.5153202993613198}, \
{0.6321428571428572, 0.5272250612660818}]},
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"\<\"1.5\"\>", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{"1.5", " s"}]],
       Row[{
         Row[{"1.5", " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 3, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 6, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 6, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 9, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 9, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 21, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 14] =!= 
            None, 
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 3, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 6, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 6, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 9, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 9, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 21, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 14]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[{
    SoundNote["G"], 
    SoundNote["G"], 
    SoundNote["G"], 
    SoundNote["Eb", 4]}, 1.5]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->68245836]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->203735909],

Cell["Play a sequence of five notes on a synthesizer:", "ExampleText",
 CellID->479018329],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"SoundNote", "[", 
     RowBox[{"#", ",", "0.5", ",", " ", "\"\<Polysynth\>\""}], "]"}], "&"}], "/@",
    
   RowBox[{"{", 
    RowBox[{"2", ",", "4", ",", "0", ",", 
     RowBox[{"-", "12"}], ",", 
     RowBox[{"-", "5"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->211432710],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {Hue[-0.016436498161331914`, 0.9, 0.7], 
        StyleBox[{
          RectangleBox[{1.5, 0.2891298231708436}, \
{1.975, 0.3010345850756056}], 
          RectangleBox[{2., 0.37246315650417705`}, \
{2.475, 0.3843679184089389}], 
          RectangleBox[{1., 0.43198696602798653`}, \
{1.475, 0.4438917279327484}], 
          RectangleBox[{0., 0.45579648983751037`}, \
{0.475, 0.4677012517422721}], 
          RectangleBox[{0.5, 0.4796060136470343}, \
{0.975, 0.49151077555179606`}]},
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"\<\"2.5\"\>", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{"2.5", " s"}]],
       Row[{
         Row[{"2.5", " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 90], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 2, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 3, "NoteOn", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 4, "NoteOff", "Note" -> 48, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 4, "NoteOn", "Note" -> 55, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOff", "Note" -> 55, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 2] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 90], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 2, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 3, "NoteOn", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 4, "NoteOff", "Note" -> 48, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 4, "NoteOn", "Note" -> 55, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOff", "Note" -> 55, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 2]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[{
    SoundNote[2, 0.5, "Polysynth"], 
    SoundNote[4, 0.5, "Polysynth"], 
    SoundNote[0, 0.5, "Polysynth"], 
    SoundNote[-12, 0.5, "Polysynth"], 
    SoundNote[-5, 0.5, "Polysynth"]}]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->44615779]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->364666654],

Cell["Generate a sequence of random notes:", "ExampleText",
 CellID->433073870],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{
   RowBox[{"SoundNote", "/@", 
    RowBox[{"RandomInteger", "[", 
     RowBox[{"12", ",", "30"}], "]"}]}], ",", "2"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->74789739],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[{
          RectangleBox[{0.4666666666666667, 0.4438917279327484}, \
{0.53, 0.45579648983751037`}], 
          RectangleBox[{0.26666666666666666`, 0.45579648983751037`}, \
{0.32999999999999996`, 0.4677012517422721}], 
          RectangleBox[{1.2666666666666666`, 0.45579648983751037`}, \
{1.3299999999999998`, 0.4677012517422721}], 
          RectangleBox[{1.3333333333333333`, 0.45579648983751037`}, \
{1.3966666666666665`, 0.4677012517422721}], 
          RectangleBox[{0.4, 0.4677012517422721}, \
{0.4633333333333333, 0.4796060136470343}], 
          RectangleBox[{0.8, 0.4677012517422721}, \
{0.8633333333333334, 0.4796060136470343}], 
          RectangleBox[{1.4666666666666666`, 0.4677012517422721}, \
{1.53, 0.4796060136470343}], 
          RectangleBox[{1.2, 0.4796060136470343}, \
{1.2633333333333332`, 0.49151077555179606`}], 
          RectangleBox[{1.6, 0.4796060136470343}, \
{1.6633333333333333`, 0.49151077555179606`}], 
          RectangleBox[{1.8, 0.4796060136470343}, \
{1.8633333333333333`, 0.49151077555179606`}], 
          RectangleBox[{1.8666666666666667`, 0.4796060136470343}, \
{1.93, 0.49151077555179606`}], 
          RectangleBox[{0., 0.49151077555179606`}, \
{0.06333333333333334, 0.5034155374565581}], 
          RectangleBox[{0.3333333333333333, 0.5034155374565581}, \
{0.39666666666666667`, 0.5153202993613198}], 
          RectangleBox[{0.9333333333333333, 0.5034155374565581}, \
{0.9966666666666667, 0.5153202993613198}], 
          RectangleBox[{1., 0.5034155374565581}, \
{1.0633333333333332`, 0.5153202993613198}], 
          RectangleBox[{1.1333333333333333`, 0.5034155374565581}, \
{1.1966666666666665`, 0.5153202993613198}], 
          RectangleBox[{0.06666666666666667, 0.5153202993613198}, \
{0.13, 0.5272250612660818}], 
          RectangleBox[{0.7333333333333333, 0.5153202993613198}, \
{0.7966666666666667, 0.5272250612660818}], 
          RectangleBox[{0.2, 0.5272250612660818}, \
{0.2633333333333333, 0.5391298231708438}], 
          RectangleBox[{0.6, 0.5272250612660818}, \
{0.6633333333333333, 0.5391298231708438}], 
          RectangleBox[{1.0666666666666667`, 0.5272250612660818}, \
{1.13, 0.5391298231708438}], 
          RectangleBox[{1.4, 0.5272250612660818}, \
{1.4633333333333332`, 0.5391298231708438}], 
          RectangleBox[{1.5333333333333334`, 0.5272250612660818}, \
{1.5966666666666667`, 0.5391298231708438}], 
          RectangleBox[{0.5333333333333333, 0.5391298231708438}, \
{0.5966666666666667, 0.5510345850756055}], 
          RectangleBox[{1.7333333333333334`, 0.5391298231708438}, \
{1.7966666666666666`, 0.5510345850756055}], 
          RectangleBox[{0.8666666666666667, 0.5510345850756055}, \
{0.93, 0.5629393469803675}], 
          RectangleBox[{1.6666666666666667`, 0.5510345850756055}, \
{1.73, 0.5629393469803675}], 
          RectangleBox[{0.13333333333333333`, 0.5629393469803675}, \
{0.19666666666666668`, 0.5748441088851295}], 
          RectangleBox[{0.6666666666666666, 0.5629393469803675}, \
{0.73, 0.5748441088851295}], 
          RectangleBox[{1.9333333333333333`, 0.5629393469803675}, \
{1.9966666666666666`, 0.5748441088851295}]},
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"2", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{2, " s"}]],
       Row[{
         Row[{2, " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 2, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 3, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 4, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 4, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 5, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 6, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 6, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 7, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 7, "NoteOn", "Note" -> 61, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 8, "NoteOff", "Note" -> 61, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 8, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 9, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 9, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 10, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 10, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 11, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 11, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 12, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 12, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 13, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 13, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 14, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 14, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 15, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 15, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 16, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 16, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 17, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 17, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 18, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 18, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 19, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 19, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 20, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 20, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 21, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 21, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 22, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 22, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 23, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 23, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 24, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 24, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 25, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 25, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 26, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 26, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 27, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 27, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 28, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 28, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 29, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 29, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 30, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 15] =!= 
            None, 
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 2, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 3, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 4, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 4, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 5, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 6, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 6, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 7, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 7, "NoteOn", "Note" -> 61, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 8, "NoteOff", "Note" -> 61, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 8, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 9, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 9, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 10, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 10, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 11, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 11, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 12, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 12, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 13, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 13, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 14, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 14, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 15, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 15, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 16, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 16, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 17, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 17, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 18, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 18, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 19, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 19, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 20, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 20, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 21, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 21, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 22, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 22, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 23, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 23, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 24, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 24, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 25, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 25, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 26, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 26, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 27, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 27, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 28, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 28, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 29, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 29, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 30, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 15]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[{
    SoundNote[5], 
    SoundNote[7], 
    SoundNote[11], 
    SoundNote[8], 
    SoundNote[2], 
    SoundNote[6], 
    SoundNote[3], 
    SoundNote[1], 
    SoundNote[9], 
    SoundNote[8], 
    SoundNote[11], 
    SoundNote[7], 
    SoundNote[3], 
    SoundNote[10], 
    SoundNote[6], 
    SoundNote[6], 
    SoundNote[8], 
    SoundNote[6], 
    SoundNote[4], 
    SoundNote[2], 
    SoundNote[2], 
    SoundNote[8], 
    SoundNote[3], 
    SoundNote[8], 
    SoundNote[4], 
    SoundNote[10], 
    SoundNote[9], 
    SoundNote[4], 
    SoundNote[4], 
    SoundNote[11]}, 2]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->442109752]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->221919902],

Cell["Generate a simple WolframTones-like composition:", "ExampleText",
 CellID->421247838],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"SoundNote", "[", 
     RowBox[{"#", ",", 
      RowBox[{"1", "/", "6"}], ",", "\"\<Warm\>\""}], "]"}], "&"}], "/@", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      RowBox[{"Pick", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"0", ",", "5", ",", "9", ",", "12", ",", "16", ",", "21"}], 
         "}"}], ",", "#", ",", "1"}], "]"}], "&"}], "/@", 
     RowBox[{"CellularAutomaton", "[", 
      RowBox[{"30", ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"1", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0"}], 
          "}"}], ",", "0"}], "}"}], ",", "13", ",", 
       RowBox[{"{", 
        RowBox[{"13", ",", "5"}], "}"}]}], "]"}]}], ")"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->333718560],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {Hue[0.15589312541351202`, 0.9, 0.7], 
        StyleBox[{
          RectangleBox[{0., 0.43198696602798653`}, \
{0.15833333333333333`, 0.4438917279327484}], 
          RectangleBox[{0.16666666666666666`, 0.43198696602798653`}, \
{0.32499999999999996`, 0.4438917279327484}], 
          RectangleBox[{0.5, 0.43198696602798653`}, \
{0.6583333333333333, 0.4438917279327484}], 
          RectangleBox[{0.6666666666666666, 0.43198696602798653`}, \
{0.8250000000000001, 0.4438917279327484}], 
          RectangleBox[{0.8333333333333334, 0.43198696602798653`}, \
{0.9916666666666667, 0.4438917279327484}], 
          RectangleBox[{1.3333333333333333`, 0.43198696602798653`}, \
{1.4916666666666667`, 0.4438917279327484}], 
          RectangleBox[{1.5, 0.43198696602798653`}, \
{1.6583333333333334`, 0.4438917279327484}], 
          RectangleBox[{2.1666666666666665`, 0.43198696602798653`}, \
{2.325, 0.4438917279327484}], 
          RectangleBox[{0.16666666666666666`, 0.49151077555179606`}, \
{0.32499999999999996`, 0.5034155374565581}], 
          RectangleBox[{0.5, 0.49151077555179606`}, \
{0.6583333333333333, 0.5034155374565581}], 
          RectangleBox[{0.8333333333333334, 0.49151077555179606`}, \
{0.9916666666666667, 0.5034155374565581}], 
          RectangleBox[{1.3333333333333333`, 0.49151077555179606`}, \
{1.4916666666666667`, 0.5034155374565581}], 
          RectangleBox[{1.6666666666666667`, 0.49151077555179606`}, \
{1.825, 0.5034155374565581}], 
          RectangleBox[{1.8333333333333333`, 0.49151077555179606`}, \
{1.9916666666666667`, 0.5034155374565581}], 
          RectangleBox[{2., 0.49151077555179606`}, \
{2.158333333333333, 0.5034155374565581}], 
          RectangleBox[{2.1666666666666665`, 0.49151077555179606`}, \
{2.325, 0.5034155374565581}], 
          RectangleBox[{0.3333333333333333, 0.5391298231708438}, \
{0.49166666666666664`, 0.5510345850756055}], 
          RectangleBox[{0.5, 0.5391298231708438}, \
{0.6583333333333333, 0.5510345850756055}], 
          RectangleBox[{1.1666666666666667`, 0.5391298231708438}, \
{1.325, 0.5510345850756055}], 
          RectangleBox[{1.3333333333333333`, 0.5391298231708438}, \
{1.4916666666666667`, 0.5510345850756055}], 
          RectangleBox[{1.6666666666666667`, 0.5391298231708438}, \
{1.825, 0.5510345850756055}], 
          RectangleBox[{2., 0.5391298231708438}, \
{2.158333333333333, 0.5510345850756055}], 
          RectangleBox[{0.5, 0.5748441088851295}, \
{0.6583333333333333, 0.5867488707898912}], 
          RectangleBox[{0.8333333333333334, 0.5748441088851295}, \
{0.9916666666666667, 0.5867488707898912}], 
          RectangleBox[{1., 0.5748441088851295}, \
{1.1583333333333334`, 0.5867488707898912}], 
          RectangleBox[{1.1666666666666667`, 0.5748441088851295}, \
{1.325, 0.5867488707898912}], 
          RectangleBox[{1.5, 0.5748441088851295}, \
{1.6583333333333334`, 0.5867488707898912}], 
          RectangleBox[{1.6666666666666667`, 0.5748441088851295}, \
{1.825, 0.5867488707898912}], 
          RectangleBox[{2.1666666666666665`, 0.5748441088851295}, \
{2.325, 0.5867488707898912}], 
          RectangleBox[{0.6666666666666666, 0.6224631565041769}, \
{0.8250000000000001, 0.6343679184089389}], 
          RectangleBox[{0.8333333333333334, 0.6224631565041769}, \
{0.9916666666666667, 0.6343679184089389}], 
          RectangleBox[{1.1666666666666667`, 0.6224631565041769}, \
{1.325, 0.6343679184089389}], 
          RectangleBox[{1.6666666666666667`, 0.6224631565041769}, \
{1.825, 0.6343679184089389}], 
          RectangleBox[{2.1666666666666665`, 0.6224631565041769}, \
{2.325, 0.6343679184089389}], 
          RectangleBox[{0.8333333333333334, 0.6819869660279864}, \
{0.9916666666666667, 0.6938917279327484}], 
          RectangleBox[{1.1666666666666667`, 0.6819869660279864}, \
{1.325, 0.6938917279327484}], 
          RectangleBox[{2., 0.6819869660279864}, \
{2.158333333333333, 0.6938917279327484}], 
          RectangleBox[{2.1666666666666665`, 0.6819869660279864}, \
{2.325, 0.6938917279327484}]},
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"\<\"2.33\"\>", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{"2.33", " s"}]],
       Row[{
         Row[{"2.33", " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 89], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 2, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 3, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 4, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 4, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 4, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 4, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 4, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 4, "NoteOn", "Note" -> 76, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 5, "NoteOff", "Note" -> 76, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 5, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOn", "Note" -> 76, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 6, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 6, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 6, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 6, "NoteOff", "Note" -> 76, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 6, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 6, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 7, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 7, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 7, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 7, "NoteOn", "Note" -> 76, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 7, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 8, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 8, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 8, "NoteOff", "Note" -> 76, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 8, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 8, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 8, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 8, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 9, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 9, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 9, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 9, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 9, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 10, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 10, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 10, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 10, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 10, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 10, "NoteOn", "Note" -> 76, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 11, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 11, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 11, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 11, "NoteOff", "Note" -> 76, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 11, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 12, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 12, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 12, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 12, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 13, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 13, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 13, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 13, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 13, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 13, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 13, "NoteOn", "Note" -> 76, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 13, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 14, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 14, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 14, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 14, "NoteOff", "Note" -> 76, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 14, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 6] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 89], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 2, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 3, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 4, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 4, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 4, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 4, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 4, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 4, "NoteOn", "Note" -> 76, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 5, "NoteOff", "Note" -> 76, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 5, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOn", "Note" -> 76, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 6, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 6, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 6, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 6, "NoteOff", "Note" -> 76, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 6, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 6, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 7, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 7, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 7, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 7, "NoteOn", "Note" -> 76, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 7, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 8, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 8, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 8, "NoteOff", "Note" -> 76, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 8, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 8, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 8, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 8, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 9, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 9, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 9, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 9, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 9, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 10, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 10, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 10, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 10, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 10, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 10, "NoteOn", "Note" -> 76, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 11, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 11, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 11, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 11, "NoteOff", "Note" -> 76, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 11, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 12, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 12, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 12, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 12, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 13, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 13, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 13, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 13, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 13, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 13, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 13, "NoteOn", "Note" -> 76, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 13, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 14, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 14, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 14, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 14, "NoteOff", "Note" -> 76, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 14, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 6]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[{
    SoundNote[{0}, 
     Rational[1, 6], "Warm"], 
    SoundNote[{0, 5}, 
     Rational[1, 6], "Warm"], 
    SoundNote[{9}, 
     Rational[1, 6], "Warm"], 
    SoundNote[{0, 5, 9, 12}, 
     Rational[1, 6], "Warm"], 
    SoundNote[{0, 16}, 
     Rational[1, 6], "Warm"], 
    SoundNote[{0, 5, 12, 16, 21}, 
     Rational[1, 6], "Warm"], 
    SoundNote[{12}, 
     Rational[1, 6], "Warm"], 
    SoundNote[{9, 12, 16, 21}, 
     Rational[1, 6], "Warm"], 
    SoundNote[{0, 5, 9}, 
     Rational[1, 6], "Warm"], 
    SoundNote[{0, 12}, 
     Rational[1, 6], "Warm"], 
    SoundNote[{5, 9, 12, 16}, 
     Rational[1, 6], "Warm"], 
    SoundNote[{5}, 
     Rational[1, 6], "Warm"], 
    SoundNote[{5, 9, 21}, 
     Rational[1, 6], "Warm"], 
    SoundNote[{0, 5, 12, 16, 21}, 
     Rational[1, 6], "Warm"]}]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->257577381]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->17018],

Cell["Play a sampled waveform followed by a note:", "ExampleText",
 CellID->20068085],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"Play", "[", 
     RowBox[{
      RowBox[{"Sin", "[", 
       RowBox[{"500", "t", " ", 
        RowBox[{"Sin", "[", 
         RowBox[{"20", "t"}], "]"}]}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"t", ",", "0", ",", "0.3"}], "}"}]}], "]"}], ",", 
    RowBox[{"SoundNote", "[", 
     RowBox[{"0", ",", "0.1"}], "]"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->8917284],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[InsetBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJzs2mk0leH7L3ApDVJCSpQkDSoaKA3qWyg0qFBIGUoSShqUShIhidBkiMzz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          "],
          ColorFunction->(Blend[{
             RGBColor[1, 1, 1], 
             RGBColor[0, 0, 1]}, #]& )],
         AspectRatio->Full,
         ImageSize->NCache[{
            Rational[3, 10]}, {0.3}],
         PlotRangePadding->None], {0, 0}, {0, 0}, 
        NCache[{Rational[3, 10], 1}, {0.3, 1}]],
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Epilog->{{{
           RGBColor[0.5, 0, 0.5], Antialiasing -> True, {{
             
             RectangleBox[{0.3, 0.43198696602798653`}, {0.395, 
              0.4438917279327484}]}}}}},
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->NCache[{{0, 
           Rational[2, 5]}, {0, 1}}, {{0, 0.4}, {0, 1}}],
       PlotRangePadding->{
         Scaled[0.017142857142857144`], 
         Scaled[0.1]}], {180, -35}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[InsetBox[
        GraphicsBox[
         {RGBColor[0.325101, 0.364492, 0.820256], LineBox[CompressedData["
1:eJw0W3dcjH8cT2ZGEiKUUEYpIXu8hURCSUr8jFIZEbI3USlNpTQ0taSlvfee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           "]]},
         AspectRatio->Full,
         PlotRange->{Automatic, {0, 1}},
         PlotRangePadding->None], {0, 0}, {0, 0}, 
        NCache[{Rational[3, 10], 1}, {0.3, 1}]],
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->NCache[{{0, 
           Rational[2, 5]}, {0, 1}}, {{0, 0.4}, {0, 1}}],
       PlotRangePadding->{
         Scaled[0.017142857142857144`], 
         Scaled[0.1]}], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       RowBox[{
        InterpretationBox[
         RowBox[{"\<\"0.4\"\>", "\[InvisibleSpace]", "\<\" s\"\>"}],
         Row[{"0.4", " s"}]], 
        StyleBox["\<\" | \"\>",
         StripOnInput->False,
         FontSize->14,
         FontColor->GrayLevel[0.75]], 
        InterpretationBox[
         RowBox[{"8000", "\[InvisibleSpace]", "\<\" Hz\"\>"}],
         Row[{8000, " Hz"}]]}],
       Row[{
         Row[{"0.4", " s"}], 
         Row[{8000, " Hz"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 3, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 4, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 10] =!= 
            None, 
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 3, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 4, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 10]]]; 
          Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[CompressedData["
1:eJwsmwk0lO/7/6V8KBWRJSnZElEJaeMqSpY2a0mkJCXSoiJFhcq+lCghbZZk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             "], 8000], "Preemptive"]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[{
    Sound[
     SampledSoundFunction[
      CompiledFunction[{
        Blank[Integer]}, {{2, 0, 0}, {3, 0, 2}}, {0, 3, 3, 0, 
       0}, {{1, 5}, {8, 0., 0}, {8, 0.000125, 1}, {15, 1, 0, 2}, {21, 1, 2, 
        1}, {18, 0, 1, 0}, {7, 500, 1}, {7, 20, 2}, {15, 1, 2, 1}, {21, 1, 0, 
        1}, {93, 1, 3, 0, 1, 3, 0, 2}, {15, 1, 1, 1}, {21, 1, 0, 2, 1}, {93, 
        1, 3, 0, 1, 3, 0, 2}, {8, 2.3693340150399145`*^-6, 1}, {18, 2, 1, 
        2}, {8, 1.000007126479909, 1}, {21, 2, 1, 2}, {2}}, 
       Function[{Play`Time58}, 
        Block[{$CellContext`t = 0. + 0.000125 Play`Time58}, (
          Sin[500 $CellContext`t Sin[20 $CellContext`t]] + 
          2.3693340150399145`*^-6) 1.000007126479909]], Evaluate], 2400, 
      8000]], 
    SoundNote[0, 0.1]}]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->323937806]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->134976226],

Cell["Play a sampled sound and a note at the same time:", "ExampleText",
 CellID->17035733],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"Sound", "[", 
     RowBox[{
      RowBox[{"Play", "[", 
       RowBox[{
        RowBox[{"Sin", "[", 
         RowBox[{"500", "t", " ", 
          RowBox[{"Sin", "[", 
           RowBox[{"20", "t"}], "]"}]}], "]"}], ",", 
        RowBox[{"{", 
         RowBox[{"t", ",", "0", ",", "0.5"}], "}"}]}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "0.5"}], "}"}]}], "]"}], ",", 
    RowBox[{"SoundNote", "[", 
     RowBox[{"0", ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "0.5"}], "}"}]}], "]"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->292662339],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[InsetBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJzs2/lXzVHcL/AiSYZUQtKIQlGGpNDbFIWUkpBMlZBMhWZFGpCKBk0aNI+n
0zzP8zzP86lzOkUZm8jt+Rfuus+9z32e/dNe3/Pd6/vd+7M+r/da+5x1RG8+
0DBYxMLCwsrKwvJgYbzFtoPDZYwJ/pDcbhkdBoqzj03kLh/Gh/ZMngv8A4h+
7fRbXbcbssEcu4/V9OCt2ZMfxRmtMF/feezAeCukPqdNapd34KTr7vh/5XWo
0XsrdYetCzI84cLuHg3wmTPgCrPoxMwOZY/pmV58ZnvjKkTrQZDjx2WPBoYQ
K/AdFyzpoCS7bHs2OgoDHuNFjbvGcdRqK+fYmi8QKX53KmXdFzSnfu+hin/B
6v0GgpKjY3jW7pSdkcFA9n1RZTwdhnuW5UvXvgFsnt4gfDu/DTFpJeHrHLog
uKG4NkqsBSM3l9UuUWxFTorU400xLegZSslwV27FyRYZsTfnWvAjsTz4dmAL
hAVFbon/akEq89FJj8WtaLhiNnuK3gKTrR4fs5pasL/PZUN7ewvU1vuqrFwY
PwUfLeepacGvpFVHrBbu+7w022myMD9RwsyQ81sLPtvqLhNkbUWs9991XbUt
0L9/s2elaAtm7/9lXjvaBrqr7czf4gZsX9yeaqDRiWvnFK49iOxHtdX45DOB
QRidP3agr3IYZmxZ93Ybj2K/4zeGXwMT7wYeNTprj8Lme5nk+ehhhMTNvGkN
G0C8m3st83A/NgiNfPHu6QAl+Hn2itxGpMrwuu8ub0XB0r1HY1+0gH//pmGD
Dy3QnOw9XZPVgsxmifNVCyOXbaHrnfyF/d5rCzjb0oLLGe21igvXoYsj1vvV
t0CPpS0pra4FXfsDI+wqW3Dc2/PstdIWxIQ4zseXtGD9Hkf2W+UtiD7NApOq
FszP3zFZuTBPvMF/dsvC/Wte3hqSC/X6/Tu+v2phXtehfHmPhToKzLxbvGPh
PRU3BBN6oluQbOCwhjetBXs1Nx0aet+KlcuvFRttaULs0OY4l+ROZKlIX5f5
MIC98hUlD2xpUH9SeS/BbBhcL0ayH/kN4/0uYd+jN4dhfDWxJTyOhqpf28a4
Vg5C/uI06/113fgwZEiR+dSEoutCVm9etELPo2lOcKEea2sFn/Is9Ieil+2u
1wvr2b53mx8W6mCxVa3l0cI+734bKTcta0Fljta3oIVx2znu5EUVLZgwny+N
X9hX08RlLraFfV8I6RhfsbDvDWnMuI8Lo6xqgq7qwufOWZsslBdG4o64I+6I
O+KOuCPuiDvijrgj7og74o64I+6IO+KOuCPuiDvijrgj7og74o64I+6IO+KO
uCPuiDvijrgj7og74o64I+6IO+KOuCPuiDvijrgj7og74o64I+6IO+KOuCPu
iDvijrgj7og74o64I+6IO+KOuCPuiLv/ru7OCvS7Nw8y8dGtNL2dbRxym5xZ
7CPG8JVTqTvzBhPOn80Hfpky8PbGmn0DZ0awN/eoZsMKGmitD3TjBQcRMiGX
q1LWD4rxH8fTev0w2WBsobVxAJ/uHFo3mTKISV/+YzE1NKRuDj1ft4+OcdNL
bKxWoyj5WTR0R3wMm088PXO1fgzb+NQNTRXHkN9pcu10CwO7NLQ7m3/RsLcn
j3Lm9QAWRT+ws1w9jK2iS3PVj4ziaEjvqjX7x7C4yGMueScTm1TrdMNBh5Tf
xs1O7kMI4zfeZR/Uh8qayCAGSwfseR4etnJvQn6Jildtcx0GuK6WvZ+tQNGO
PfkiDmWIbHtrd6S/EAnXqjzTCvLBCK8aHpMswAc+wbYd5VlQEOwvrHtKBde1
hOq1Vck4G7wr9HNBCjq+uIf66iZDTNnA+EV3Eio13h+09srGud+ZUqwD+fj7
IbF3eL4Azw7x6Bu8LMHW6NgMAcMq3PTTO/2tswF+PisfH5BuQzD78PP1j3qR
/qMv4mn8ENxsosYcr9PhEdjq6/xlFMaBWknxV5nQ9rge57yQT7wPljMOYyEv
684fd3uy4E+841V+zgjWcLMd/dI2iKfJddr5l7ox7tn6T56tBUObXf8UF9Ti
2lHmHffMUkjYcgxa3CzEqKLbSQWzbGgvLe5YeysDYna4uToiFXrDg/tT1yXC
26es5tuPFNiljfq+cgxDgHrJ79Z/UdBXZMn0vpSAy+LeEXVRVMhoXuMy40sC
T5/4gL5dIli3b8yaPhaPdYFZV7ZwUEBzDl/qaJ+EaJOW2oplsbCRXOwjEkDF
3nHeM+fYkzH18EbVud3p2Cgjt0O3Nhef3HhMukMKkXnEKnDwXBX6PfMlN8Q0
QTCjXPriXBeKFrPlJkgNYYv3+vHDTcOIrDry5BLXMN4w1or9nB6AbeabVek3
aHj5wcwjOXEYKvte1ql5DmEyI/i0t1oPbHuvhnYuakYP+6ZSvWVV4O0slb80
VYikoMiixX056DcbNty/Ow1r75nfbKNTcO5fDAutngq9iUVxaexUpLLyftGs
S0R2an/GVD0F5reM6T6cEagIP9p3XSAeHXPVkl6dCXho2yqyrSUeAz/HhAZN
4vBG+HtDr3UccUfcEXfEHXFH3BF3xB1xR9wRd8QdcUfcEXfEHXFH3BF3xB1x
R9wRd8QdcUfcEXfEHXFH3BF3xB1xR9wRd8QdcUfcEXfEHXFH3BF3xB1xR9wR
d8QdcUfcEXfEHXFH3BF3xB1xR9wRd8QdcUfcEXfEHXFH3BF3xB1xR9wRd8Qd
cUfcEXfEHXFH3BF3xB1xR9wRd8QdcUfcEXfEHXH3/9idSNiHsoGNNHhHxs4Y
LqGD2fcs4ErAKPo+3HvBfWoM1j9GA1QExtHm6/b4875xvJhnkRWaHEPnn9e9
eS/G0HokYNJt6xhErg/yJXIuXJcZJJbKjaG+KXr65qcxlHTap+SuHMeejgPi
i/eM41akWNLzvDEY2dbTbK8wsX/XRZFyOh2PgzqltHxpuFj+KfBq6cDC+s7k
7NLsQw/ru2yI9+IE6+rHQxf7ceWjIPs69SFo3p71yZsagZzM7PqOPUy8makw
rB1nYvf65akOPxkYW1sdyH1tGO6CL35ZBQyALVHl+aufXZDNr1nyzrQVf/w9
qYdjG5B+fc30S9FaRPucKhD8W4bfrXaGn/hLIK3YmC2nXoyuCzFZB/zzkWBd
YNJ9Ix3i3D3L5RIzkKWsYviJkglZidok5d0ZeMC/va1JJgO1YQHnytcVwPt0
jeyy/GKcdTbzPzRcCiH/EmExg2ps+t6okL2rEXFcnM6qeW2YLOBxX7WoDzZi
57T4OoawdkIi9Ww0HW37H8zUtYwipZXfWK2cgWy+75MqSjRYlaZrC9yrgXJ1
E+uVSBp6L3SmBiYyILEomyNglgHeN4wog8sjmCg4z9/FNwiKp8Tp9i1d2Gb7
jnK1sglnAxNV5r9W4w13UeqfuRJ8br/uUPonHzGOhzROns/GGPcWC9sV6XhU
YiHw0CEZFt0O72mMFCR9rJj2S0tAZdDkpT9cC304+nnueUU00qYlWd1NE6H7
xOGvNgcVXl+Euo4ZU5Ce5l3DZR0L+cVzzzoKY1BfxKF3rI2KP1+T7hQUxSPO
65Lom7EkvAwR3tuhnAZnne2X3u/IgvdWLmXriAKc+rdf+ePxMhxTmLLL7KnH
rQtFoacN2/GQs19BV2gAljf6L84/GYaEWO9PvSA6nO5/6Tr2kIEEw5Kzix0Y
kOpVGio/x8ABzWG9ACc6lno/0HvrMgwh+qOO4yYD6BXeb+ns0Q4DhULF3e/q
EX+irksopRRd/+LVPQPzUbHv5dp7uzIgsHm7jb58MngP8mbP8lLhsfvFFrMS
CvpE2fJC5SmIXE2Tao6Ig6jfmoygqhCIxF04uDcoAlIfHu46HReN8O2u5tY5
kTDf9VezdGk42ve91nm+MZy4I+6IO+KOuCPuiDvijrgj7og74o64I+6IO+KO
uCPuiDvijrgj7og74o64I+6IO+KOuCPuiDvijrgj7og74o64I+6IO+KOuCPu
iDvijrgj7og74o64I+6IO+KOuCPuiDvijrgj7v5LuDt4viAhe7QDx2vylHlX
9mKieRlbhugAqjm/7e30GYJ9k9xFv40jsBrk31q9lYGcQypXZTxHkSVdvjHO
nIk61V7VwW8LfT+yPWGQewxTcvsGPXqYKM+vlnJRYmJ9zbSizvZRfA8e+6X4
iA63i/tO2vMteIAjR4vxIF5aB1ZN7+yH7tBMHmd7Fzh/rN0yE9AKtdtOxubO
jWAPZSiVXm2AVyL3nrquZvx4LvupHl14rJFc/Pv1ALKoH78fOzSMnj8461TP
wPunZx8xGUy42O27v1+Fia35Z/riz9ARxRN/Vc9jCEX2Mf2bGvogj3HOwn0d
0L+0LIsa04wXm5J1Eo/WY/XUFrfckSoIPD2z59zOCqg87VS5qFcMWmza5gHd
PLQoOqfr5+YhJ4dH6YxOPpSix55R4/Iw57vqufT6fCw5XeBqyl2CumZsf3qp
EttHhGN9fGpRdjguRFi+CU4OHO9fsnZg6Vr17h6OfqQaTEtdVqchOy6vpoh3
Yf11DUL9HEzQ19Xtj21ggE2MeiTCehgahtJX0TOA+pWMugiBfqx5pCvb5DoI
n64BD+GTI5gdS3l3u4aBTLrzB5/ohX4r2/e3v2QYWoqnrlBPDODqqYaov1kd
qNxxXuJ0QSPOpPAUi/lUoaX5rtivsWJcsVXwF03KR+YRxKrrZ+GEmnJbrFoa
vkrL5z08lQYVSxlplg1UWPy4tahbiIrWixRubs4Y2KtlLmGWJsKO+eexowkV
V/j+9ifuSoTa4CevymsJKHZIFvjMEYt5565LOxOSIK3w2lB4IBEf941zc/im
gm2rwMGjNploUVe9/G0qD7s/VtGdZEswx8r/98SpGmRQVH5n7m2BUqQhX5J8
L0TmvJ+JctFg0B3uv1+RjrSisS332BkwaNQ9pPBxGLal8T9mlvfgxvX3Ory/
aQh8XeC1YyHH5MS6dvUupaOBx59VZzENh3Z8WVu/rBdRUTvebfdsRoZ5l8HJ
jiqcfOcdKfukCDukXnmebc+Gr5f28aGsVJj4MpZacyVhXn5caZV8IjpXHWq2
BQW1RrJ8g8pxWCWozdz0JAIKgb25By+GYXu4x5pi3SgI0ORr31qGQ5aWZ7Pz
UAiKEi4n9F8LIe6IO+KOuCPuiDvijrgj7og74o64I+6IO+KOuCPuiDvijrgj
7og74o64I+6IO+KOuCPuiDvijrgj7og74o64I+6IO+KOuCPu/g+7o+15MbQm
rR7+FStOlNY0osLCF3IL/f15/kFbyOJ2cFx97sX41gGn4pHruwO6QDG+fNph
qAeOOZMVuzf3w23i5/P9v/uR6Luc+nD3Ql9oxsSuqu1H2+nlD/Oy+jByUOcv
w7AH3OY/jptadyGEVaR53qEDkmpnJcWaWtFiJS3+froJXy43sIQFN2IfjfOZ
pXItWqQk3gpeLYNXmqbtu94CXC7pUb79sAZJmbGbaZ+a4MMS31DJ34H6q13L
eyX6oP062sH13kJO+M5lvRei40+WxcigJBOHNjaMTi7kgaamfv7So6N4knmm
TD93GJKS5k9k1w9ix5pfmQ/v9eCK04tXV7TawKrIXNfj0YRnQ3KPYheEd/fV
8DDNK9E8cFy9YG8JzK5ZGS2PL8ZxRxW/wOliOL4R4ThRXQz63bfiXNQSsINy
XtuoCvb9fvt3uDcgt+k+Rf9eC97KzPD6V3RCPWucz2DvAOa6VdnrpIYh3GlT
dLyKAQ2+S+GVF5jgUq7SXFLMQMfoQRtH5WHsblnv/7OvHx1DAZu/FnTgl+KU
+1xxK04w8mbvZndhv87TaTWPQfSmscpwMEYgYvvQzfkPA/mxByZUdBm4Uepi
obVmGAEFkkccXvbj+HDTAyfLdrDq/LbY2tKAn0c3fPj1qBIzcqY21leK4eOD
jak782E5xRDc2ZYJE1NvR+/3GVBos3yg8iAZEpXUJBPJZEjW6rvuV4tDzKOE
hCP5VJx7LUkbt0yCl3Jr9jEPKnZlsWZ3P02EuPNgyYxyPDwqvndNyqXgsalM
WqdfMqgPZ46ZTmRg/u7lGz+ouXiZnMO5/FQRbrs3NvIqVWJrWPZHpcBGbOX0
cdRJ6cQGNqObF1UGMX6ay4liOAKK9GDRxgVvNy7fl1/ORofCogLVrskhTHDa
3qZEDyCi+d9FWvoQNN9b9n7wHkHQ6jg9Mz86yrLo+QasI6goFNvBMjCATj1b
VzWnDiT4j4v5qjSAx3SMLdu8DGa+L2yy6fkY+8K886swA/TDgu553cmYOJ1y
/tI1Kvo+sJR+plGgrOcqqakYj2m1lwYVrVFwtb736y0zFLOt/0rFX0ahIzTw
oFR6GDxb/40OnA7Eh8wf0qwZgcQdcUfcEXfEHXFH3BF3xB1xR9wRd8QdcUfc
EXfEHXFH3BF3xB1xR9wRd8QdcUfcEXfEHXFH3BF3xB1x93/V3Y1TZk+aXpdh
XSp7U5ZPBVLWKTAVL1TBelLm0cZddXBWWBVdEFgL78rKTMzVgOVaMz1CoB5S
kcWukqX1kPkr/kPrdj2+f+f/1V9Thzclrf7O++vRJfHM7qxhPWof/pV/eKUO
CZn7VETu1uLIhtsWGhvq0G679UXybA2WvqupNRmvgL/D6Q81k5W4y9fj/GRX
Kd7pm6S+KipAzjWHkW+bM/FgTPpr41ApVB3S0m2ka2Dnzzrhld4Ajd6EVzyK
rVjDfW54pLsLUqtCdr9/OgCNx6vG9kgPw1/speZAEgPRzp5v1P8x0drmmVTS
zUT/nR35+voM3Cr6c9EumQZ2gSlV1f0DOHtl5WxsVjde0ldeO/WoDbfmCtQu
BDRCQ8u9vH1pHb4K2aSYHK/BfOwJHe8VNeBUtFua97gGMVIsb2iudaB8L3Kv
kmzC7KvJMC6fdqy90UdvLumF5hWZisSqQYSYi8X7UEbw5n3H1TbmKN7LH1V1
dmfCXDnDfp8eA37SDy+tT6fheQDblIV7P0rDjUeivTpwwdxgy6edjRCtDv+w
+H45yn9v3JrQ2gwdGP8YT+3BgRnLF9sNhkDfYzk0aExHnXp63iX+UdB+KK/g
5mTgmu3m82W3aajSZBoIBfch9xTr6IdLbfB7EnzismADBOUXy336VAHzYtFj
s3lFoDc1Ko0l5mHq/KiWa2c2hFo9X8WfT0NGq9PArtepEMwSlIARBSGHs5MG
5ZNxYeim++HZZJiWrnnLeTkZ7pZfSmUWHHZOMm32jVLw2NNMW/xiGp6osSzR
Gk2DUoys/KhULjIc3D9GHSkEP6u68/PBMkRnis6kh9fDR8HnSV1RG0z4Fh96
sFAHc/Usn6+Kw2iPtM3520rHky1re9Jb6DgzxtRwWfj8L2/9aJFQP96rTeuF
HWlFW++fg0F/+2DVE94DsWG8fyd7aeQGHX+rN+iu2kXHhP5tntLHNKisM34k
saUPApUSrYHOLdA3n7VlfKnGt7qDY0byxRC/vLZ1PD0HE/7DjoI30uB0uPJB
FnsyRqNWFy//kYhrWZzvJ78lYMXjkmNL+mOQXKG1dqVjGN5PpArfvxkNhYmI
pRKFYcjf2CHp8dUX37OiZsx8/Yg74o64I+6IO+KOuCPuiDvijrgj7og74o64
I+6IO+KOuCPuiDvijrgj7og74o64+y/k7tsiBa5QzkK4nQnOOXmmCEZPp2uP
iRVDksU4KuBSGR7fZehnO5bA6XI3U6u/CIZmlcvnTEvBJr5tC5diOV78250b
+7Ycg6eSAn7KlyPtX/2MAbMMHWmHNOVUyrD9tBKPhnEJuMd1qoRuFoM7/syq
bRKliHLnaAh8XIo4zaLFUh8Lcc/5iYvhiRIc4XxUM/eyALvf0uQGEnJgNPS5
1dc8HfTgJZb+aYUoMQxa1pVZCuqj6P7ExAoIRMXo79tVDyOX4yWBrk3YbFAl
21bRjg3203k3y3uRFNCqf45lCA1qHrXN7iNY69Xy6VHVKGbyaPMexmOYuv2s
k7uTiW/r+ZttjRgIO+HEUvGNBieB72d2nB/AzdZNfe1F3UiYW3QgZVUHygVk
ul9ItsG5uTmn1b0VNbL6MgXSbei3y5uK29UB5tDIzqLtPfjjy5/WWjyAo+WG
5w9HDqOZ/8qKIOlRSD/TMfVePQaNwAbZyQtMCGuZLt5oudAH8acCRpOH4L74
sNxntz48to/KFrvfDpfk98bVK5rQdH79WCajCvG+5XyDXwrR7txvraNXDzPW
1I2CEwvrULRLX7akDyJiD5JfiNEQbdDEc3mYDll5/AveOoqg0X2Jm+Po6Lha
nk4rHULWkZ5Jc4NetAWGis4/a8VmKZklXB/qoSBZs8mOXo6GSnPR6LNFcFD9
47IqPx/yirlVkeJZePvPlc5mlgF6vNmF/moqnh4MfiWikIolqR0fKr6lIuFQ
nvd9h1R8mJfpe92ZgkvKGWuU1ZOQupZtbpFUJsynG1Z9l8zG6oC717YzCxAT
pDC37FopZrqMm2bVaiFOuc3s029BpuvaJYJcvRhn9bh5a2gIn7WGbdbr0xEq
KC8ueXAhL+P4th9+O4K83B+OYXqDMOoK33lkohMmbXmFXnwNiFN87DZ1ugvO
92tNtIoG4XyoTY1uOYLCEIfauiA6NhRX1O7XGMHsqco1X40HMb5EWutJWCfK
xW3YpFY0gh4S/c8yuhxe5q29Ewv9+L2pbdea7Ex8OCL5+P2eVIQ8zuKl3kzC
svfyGe0bE5G6KL/C9F8cbrhpBC2vCMfvJZKyrEkx+PNAN6vgTDhwvPf32WQv
CO+U4hzK9CbuiDvijrgj7og74o64I+6IO+KOuCPuiDvijrgj7og74o64+//G
XRzX17DfB3KwJ0wv7oBoHk4sTd/5a1E+gm7khCziKIL54+M5vzrz8WKkqj6u
IgcuTwo7zXMKoCv4l9XueRF2XC1U4SwrwuBBWuof/SLM9Om0DKwogrTJb30+
x0Jw8HLtjkrPx6P4oC0n63MhrtI+N2VYAI8Um/HTJwoRmuT68V5gNorGeOt8
KguQdlefo3VpDhpPhG3Vz87A2WL/28fWpaInZvHN02l5qDm/Kb/qP547uVbg
kkkR1sxlxa1MLkeH57u/d2WrwVvw1qL/eQP0Nhszx7xb4HygtOBzTifWHuEw
HfmP/33mGHuyDw+B6iZpOK5Ox/U9TV+GwcSPD1qN/cljoO5rvffBbgx3Y111
fbmYMPv7LGD7FB3Bb8DlWzGMwvRfVKyg4cbvk8PvTw1hdl6iKHxgCFwpD56u
9x3G8v2Pq0yG6MiVfrPjhgATLSu+VFUuPC/z5EXtVafH0BL9LfmJyCj+hgbG
UuuHsaJXVe7ZxkFo5ZhL9on2gJ+a1B/S04qjrrVp3hUNsFqTpn7arRqPTu+U
HV3IqefugiuvuBVio023SopCJVyKq9f6mTXA2TL8sNeydnyOyNFddqgfgYFp
Na/raXCzfvlTUZUByXo156hdoziYdPbr3Dk6vst8of5SGIJ53ZfRP8Y9cBo1
P+TX0oKvon9Tu3fWo0u0fAmrbjm+2aht28NfjHgdzgfKe/LxlH21qq5SDjQo
NnMNMyk4ckTrUMVAOqzXJXyetsrA7UcmLTpyGfA4uHnnmuJ0sFJXdMs4piL2
mPZg+XQOxMQ3/OWczkds63xNdlsJvoWsceM6Wo2E5sAUL+8mPGHcEjEx71rw
tnmjn8MgJmaixsrjRsAnrLB0WoeBLlajNf1OdERVqiSEstDg8+XN1et7emH5
cb7cYLgFR90mrDIuN4BNF1mhPq0Y3lWT+Ly4D3fvfXjPSKdBvPOHLOMsHY9i
RIqsdekwXfnU8/coDcYv6iSFdvQjj8a2o8OtFVKrvIXV3GohZNHV3ydQgopZ
7p6x3XnIrbDkes6eAa5Pe/6sPpwCynGeGvM3VBiaKDFM9lBw1d/z/hbHSPyr
H5h+cDkOKVy8MwZ7IiBYvZX1/CVPGHlVxXZkeRJ3xB1xR9wRd8QdcUfcEXfE
HXFH3BF3xB1xR9wRd8QdcUfc/X/j7kX2t5lP79KxOKRNRnBPJuwiU/9cTMjE
eI+5hc3BfDz4c5wpN5yFCUOOAK78NGzTvnCVIywHrEmrHCOf5aPmX36aRUw+
9tzZGWErl49dvVU79+Xn4UdwTt0ujjzMuE/rXsnJgobP4fVeCelYXxBvQy3O
xoSYPI/Otjy8zwipYi5Kg6xq6hvdhlzUdAg/etiRjiVtFDXeylSYrxvr2EJL
gv/z5DiK98JzVnNectTOQ6DDm9bKpFyIyHz5LtlZhEjlbweVbpXA0+ua7EH/
CjhpHmTx+lIL9g6nrBXWjdCvtvo73tyKmU/PUmzedGGVlKb1aH4/jjjETl/O
G4JRMNXkAnMEgVb7HD0SR3HjFm9l/cMx1PByd4VuH4fU+p7N+xXG0aV7esMO
znGsm8l/VzE+hiuL2kaV2MYhEUvbObJ3HDbRd11rtoxD7t+BxQ0XxqAqwT/y
6sooyu48e1h7bQSxugHGNZJDUGB/6eeQ1geLzbH10z0doJlqBPAkN+Pc8HNX
xaf14E//Fa5pXonNhTtlspaX4tBvJaNv7wrx5+z9Oxov8qHUfIdD+mYJNvsl
37/TUolVXnbrnj5sxNnv2vf27euA1Y4jsf70fjxJuKKx59Ywyt+0yW37zIAJ
369BbflRVKXRLI2X0LEzcVaysW4QWVOPvF5XdWMuo/KfrHkLCvx3VN5rqoPd
WUdx3eUV+OZnas/7owirNh7M7ZcvwOmBpBcn1TLhVvAjSPJRNupafnofqMtG
hMyekZOfs/HLwPqLrG02XuV2WMuPZcJdq1qJ81MB/IQKtt9cWoJxu5gOV5Mq
aPJsWnNZvRGzamefZVh14G9EbtcmjQFcjqn5Nd8wDDuZyudV5xi4WSP0vOEg
A7bvb/Fk3BwGTTzIRjW7HzKaudGla9vxkW/D7i17G3DSLdWs+lAN5po4B2LZ
G2Els62OV7wTrHevFTbuGoR1nGv/mwMjkHJJp2+Io6PSNugIM2wESjqfrS5x
DGF5m5OZy7cuiLV9S5ve2ARf7f7p0ekKOC7ZvyXyYyGMt13++ywzGxtUb1ZJ
R6ZBwaX0bqpGMnZRT8lv+5SI4K7yju6MaAhe6GnG+QRI375zNf5WJDJFZ6Rt
4n3As0y4TnaTD3FH3BF3xB1xR9wRd8QdcUfcEXfEHXFH3BF3xB1x97/pruTa
W+Um8RSYTL47oJWYiuRHQuvGJlMhXnNvQ59NNmy7/L+dHEpDwvaLUlFZSThm
3fKXGpgB3rf8reJx2Vh5fn7E6Fs2+IPkN08xs7D1wtbzQ4ezcHzQx+h2SybG
DufVHxlIw90zEs80bZOx9axt+16HdGy4nc47eCELr1O3BQitTIJ0mXnnOfUs
9N2KyzlVlALBZr6yf1bJmCySWLySkwpRr6XPNlulQzu1VCf3YBaWfozperI8
A4sm1/4Yj8zHfTYHe431BdjH2+D7LqcIt97d+RGlUwZvh2NrLtlWwkPqLU9k
Xx36bO3vH0lpxDHXZEFx3Vbk0WoWF2t0YuVAKFNHpA8rOOreTdYPICDoTLmM
Gg05HwxWhlJG0Dh0zoLrIQMBwr+fLTUZRf2FF8xX1aOIaLz2OtFhFCyNXYU3
3Rk4QElwpzaNoP0F5wDzPg2ClnNJD34PgHukd5vBkj5InHxE1+TuRFfUI7p2
UQtO17jweAo3wule2h2nlBpsf7tZn9uoHHkqXJZjTsXg1HN0lIoqwJctMvvF
TfMglxpfzFycA+tXkUdtOwqg1He08RN3CQ6tq+3/9qoaPHs9HN8oNeHwSAG1
+nAnTFUk/r43G8D9YP6azoFhsIR8KtQcY2D9+k3XlyqPYiz0mMvnlhFEHbkz
7OQ5iNk/r56EeHVjzIhVk67Vgse++pVpCgs50F92Ud66Ai8PmKVOThZjW8gc
46tCPmjLv5zSj8/HpThWNoPgfMRvYq77/X4hh+uiMnYU5UPqQWT1S/d8xFYu
Dhg3LAHbLZ+9IqOV4EmXT60ubEDxj0XyguvbsUTb001Xvx8GKrvMmudpWBzq
sVZ3GwMx1svKhpMZ+COaKx5gMwKh43fik7YNouPYWju+7k7kb0gJmZprBGO3
+eKe0kp4jjt+vZdRhqytwX1HjKrxb3Vv1l6pFnxJShQIjO7F4Rwj+a1HaNjZ
n+92Zg8dfbfLan6/p8NpFTOt5PEwwtblGNxk9mOCv9iriN6GK+INBzkP1mP3
sTKhleGlOP3YP+JeSz4MJtYPMqsyMXM9YYz+NhVlqhXP884n4Wpuwo1T9+Pw
MTFZWLeZAsULR7ZcoEXh9AH+HZ+DPkFBRel3EMsn4o64I+6IO+KOuCPuiDvi
jrgj7og74u6/ibvT3+1n/d9R8e/9oXKP1cmI4rCQui+ZDN8XW5U/WGcAmoOR
6d+SQd+9I/vFGAWvL2w6+SQqFQnrQi4zbTJwlSMiRT0iA5afdPjWOafDqJu9
6cuC08bQ7G5Kehrqz1OGzvOmQJ3KEdhmmoii6HsdHTopqDZjn2V7kg4uJc/J
k8IUuLpxCHD7peOkufADd70kNImx/koPoSJ19HX9YzsKfFdLl59blYq5M5+m
BsQX3vP7wqoVE8mwNzAfG+/JxlM73kQri2yw/55acZSZh+BbwlXzMYWIbPy2
rkWsGHdv3L8oZFGG1Eu7OLLjKgE8vRl/vhaXGbe3d+k3QIPmvtgmqQlX/hlz
v+ltQflb19ZX1HYs3aa97pF2J4pHFGPFTnTBK0NJUOJGN9h+rlU7xtaDuysb
y8v8u3FIrNlJPqYL1ked+SKqOsHnE/2Fb2UHNs/Ka40cakXt7OT6+TXNuLyK
ejb+eAOCFiV9qFWrxffY5+Z2Jyvx80np+jUnSyF18bZs/lQhZNfeLcj+nI9C
vqV20idzIPZVlWJzIAtsleMPnCMzcLvAynr2RS5EFIpeMtgKwDJ+qW/oTSly
tuUpP/SuQdH659NWs03ot2eXXyTSBZFmxAsMD8DNVyU5WmEEAiw3Xl/ZOopN
0sbe7Eaj+DrHy2G6kBc+mfM3V4YMoqp7aPhYYzc+rLhwYltOC57lVFb/ZdTj
g7iIRahTJS7aftzCU1cMt677i64fKkbIxGKFv1lFmPgbaiadUISRzq3tHOeL
kfT7tKrb9hL0nS/dr8RThdi3++fbCxrwPss6WcOiDf9GK3Q5/BbybpOqmt8N
GsojbyaLT9FhtMjFcWLVKITok/wSunSkMlaE7700BA9dw7DDp3qwz6hrY8fy
FmRYiPu9mqmBUc1N9ltLSxDKayT68XQR3u6VCAp4XYpu46qjt9gboPxl8pA9
XwfMVgWcq34/ACWzyPbukWFsrn96nz2Vjl6puz3dHHSE6EsbzxQNQd3lklnQ
ux5k1n3fKmDajCe9307WC1bjaehdg687iuFe+ji2fCoXdk59K9/qZyBhGW0/
i0kKetQti7lvULCcRyhqkywVJ597SrQnxYD9++AFw/JgVCvGhF21CSbuiDvi
jrgj7og74o64I+6IO+KOuCPu/pu4O/9qW7HKSwq25MovyWKjIv2E1D/VZVRw
xofOOwun4Ur+oZs6CVREqznPeZbFIbZgOd3mejI03bt7udnS8NPSIkZ5XRp4
E07F3KSngHLsqX3ZiRSsYoQFcqinwPbRH9ber1Tk7vqzxFMlHioPuKJ+Dy84
P/RvtYlMKrY1z7p/qIqBw/fRmd4HqXi8ZNOSj3aJCBCyOUQXSATVOaD29tN4
uCxiYzTyJ+NVg8wBz9wUZC/782VLXSICS4Ocbbwy0LvqvLKkRTpq57S3XizK
hBKvZ2e6fB5abkcsStiUj8UxzeHT1wpxaNjTzcK0GPe1d/P84ykFb2+a80BV
OWgBdHvpu5WY+PzdRkSiGmf3FrMEfKxFxn2mjQBPDZ4eFKqssK9EjfWS8Avs
lWh+lfa8UKQCX1OESpuvVSDjfO6RafFKrBWZzXFfW4OXXjqLpCm1cE+56yO6
qxoxamZP/NSrcHuJ5VLK3nKIT0SzU1+VomP3xi7vA8UI/ZEWzetbgAmpP8nF
VbkQUx9QG/ydhc1J88vmJ9Lx/E8ar+Z8GkbZTiiNjKXiw+GUZYGfMsFsqfaa
Vs5Gk6Y9y/0LhRA9weC4sb0MJ4R3FrSZ1eJrQH34kbfN8L+/dfRKfBcYS8qf
uIsM4v0ZhuLtWyM4xda4S017FGyvhJczKaM42PQnz4eXjprdjI2yXEOgqkzr
G3n3IGnTC4oWfxsWe4mcKAhsAJf1R8G0n1UQ+GNi2xlXAd14Nk+rU+XoO5xu
Hiq7UM95F0Y8pQLyXkLWKkeq8afKgj1rsAHvFhl0FE22gRqsobNGoQ9JvaIy
+Vto0Gz3qVgzSMeW2QEFB8NRzLqwFHiwMeDT1fhw41EaWp48LndV7cMV38Sv
or2t+P23oSnwRj1eqt/84vWnDO/OGqv6tOTjqNYmu28LuXt7x+fbXZKF2Bi+
+Ifa4yrYVZcfS5Noxh+9l4oyC26b4uvWfA0bwhGeEumoZXScl89iK6iko+rd
7qHG1SNYudzknXzTACxSJjj0Ejog+vhKbSSlAZvWztqF5ZWj/9Ge5R/VCsHC
VzbeKZCDLUmPRfaKpkNb6HMxiwkVsbfNLu3uSkIprbnv13AcytsF1SruhELf
PX+KY2socUfcEXfEHXFH3BF3xB1xR9wRd8QdcfffxB2vAFNSNz0OJ3Y/t/O/
TgFrIkus4kUKmD97K47vSQGrJd/2fhoFXxPWUkcrouF/8m2ovwsVyh5rt3Eu
SYFKW9VBbY4UnHs8YdJdlYSMtUsfm0gnwZm67somvSQcbzpsMXsgETjJ5TEV
HY2Yw5fi1b9QsOgejUPqUjLoWw4fndSLxPEi+cpdEcnYv3Zn5v6fCTjifSPl
Q0QCelY7FPBpxsJx0fOsa9ZU/NF5qYi6JPyIulxVmxCPMzVrspRXp6F51mKn
g3MKCt0dDDK50nD82WWx2KlMjBe/Wjn7NRP6F9bigUQuknc2GdpE5INOj4hY
caAAzwTMx5YcK4J6xmENq8AibGA3ffDFuxgH6S5TytdLEZb4XinLsBgczRZt
OlUFmJdbnr1sthARSf2KnZ+LcGp/yTK3r4WwCOxziK8ugB37rdjVj4pB3Q7h
MxdLofFT9vWhI8Xwdao0nftYjG6K94GvVwvxU+0czWGkAGnrL10vMsmH5CeO
gs7lOWhsuHmIfygDJ5dGlL7amo6oZZ4PGAkpCHcPn7vwMAUK9GV9gQnJ4I9v
ufjoUxoOJyXTs9ang7Lz5U3dpFzQSjN85z8XYt3e7+YHqGXosv8nGS1Yh4wo
Mev81mbMVB83WtneBZ5xiS4XgUGwzmQu2qc1As3Yu5L/8b3mHk67qgObmOBL
Z6TX5tAhP73Jt6B/CM4MkU3F1X2oFv+4u8ysHS5V0z9KM5tQ75xxJn5vA36v
U1AL1KqDHtulnKWb6rDZgur7iLcBuvdDJmuzmyDqXVg149cOYSutZfY7+lEy
HvNTRJ+G2IhaHmsxBo6aj2RpNI3iLOVa3SMTBjpqrwrta6Shh0fPJFm7H3f/
mm0MPdcOKeqyEy9CGjByOpTenF6BVRdtDh7+VASNH8/t+92ysJgiHiY4kYlb
HddHh+k5OLBFSy32SineyEVZZa6vx3OLH/Y/c9pg2Ccer1/Wj2YBl+ELBsPg
++xaPJdGR4Bd+O/nVnTodPHPsMbRIC9XsNkpsw9Xi504NHRb0bvDHCK/a8Gd
Eksb31OKG6+YomdWFEDz1t44Jf8snNxQ/NSLJwUSNKX9Td0peHjKe+OV0xTY
l/FfNb8RgVfdu2yltkUQd8QdcUfcEXfEHXFH3BF3xN1/iju/d379HPPRqEjZ
K2jIFQ+q1FMp3W9x2PYaLEyfJExwnLZ+2x+PK67hPlveROLa+bvbRLgS4ZYX
cfX2tSTk3TbxqruQBBuLI9b93AvnybAcx7vmidglhAN76xLhWNNx9zI1Ab7n
nLesKIvA7nfpXssC4+HxwrX9ZgMVOco8YRr1YYhefHV1jEQStp9zmGUaxWGE
FvthjVkcjJU++n5/EY3BhzKsDLFEmB2f3+UhR8X69YMfL8RE48flMratK1NA
c8tzNzuVhM0N1kcVlJMh8E69e7QlDbanS0bzNqXhth/niNWHDNRFY2e6UjYs
avZnSx/LhsGxyDcVGXloHhLJtFTPw4pbmvymRvlYL7f5wSq2Qmhnag7piOej
7PiGP1s2Z0EqS6bIWScXzaH1lVnleTjVZ2/Iq5UL0dAPL56qZkFIJVHunnQ+
hjnXP92/qBDbDwecu9Wch5w0veJX6flwTPeWtXfPwXK5A49qunMg/Hx/ck1b
Fr4uldn5LzcdKfIKH+O0U5HdslHNSi0FWbGJJlu3JyE40vrx+utJCAjwVVJq
oyJey7JQRTEFxrqXHbf+Ssb0YFNBkk4mOPeUstb05CI2t9c+croQycb1zF//
ynCJYhR1WbcO0srt0permuEZ+eiarl0X+l4py7IXD+Dum7MqpuPDCB+Si+rG
KKj3BI49j2HiHfeyRXu4RmEjaHXujdYwVp068Nnj6QAm33DLydG6UacoY+R5
pB1+P7piQte3IE9U//uHhmaMsUzsKyhpQ5Tp68ulr7qxmKX2qeS1AUy7cOOr
xjC+TyonneIcBXvlxFCxFhMdPpp6mf4MyEVFmefO0sAvOZl8M7wfZy/TZxW2
doDXMr2lyKURIXsa9l7UrcImraxrgz7F+Ex9917cNB9TVlL0kYW8rT7A/lB2
axrWDGWyqGzKgL3o8LSAQyGC+TJOpDRVYGYPxfrYVCNchPd3C/t04Yj/Nb+9
9YOwLSt6IpQ+AjZFjutT/+hguXPnS3DNCMz0e7S5WYZAl/u7321NNwLmB130
A5ug5yt7jNO7Ct9eeP96Y14M6+n+ShbRfHxeHnz6qEU6kh51zjTrpCPm3fJo
+ZlEbFU/8rt7MBot/NoBnSnRxB1xR9wRd8Qdcfc/wt2Zm4yh5cmReBS7S1+6
NgbVI2ub3sbE4M6KTctFQUVtNzez/mkspPic4wUaw8Gv05Fk+SkByRGhp0P/
LPj7UvJZaiwRu1dqSPIFU9C+ayDywQoKIgx/NNtYUzB+96et+s047L2hI9Xz
MAymYr+vdI7FILvh7qTtq0RMqVlWlY1+xgzHuSyp/kQ8/acW5HgqBhVlpUFt
HjGQWKOxhfdXJF7Nn9ny7HcC1u/KDnQaoEBb+5xOn2oEOCo38QkaJkHn8Fjp
uH0ivGyefd0kTEVymfeHMeMUuN/W+9alngy/lyEah2RTIZyqkPgzMR1h3opX
habT4BvB0PYSzELj1XOvktdl4khmfsy7F1m4VNDGp3EnF/2Tq3LVv2WCdV/W
86t9Kfh81N5i8lUG0qkBOm9HMuG67AGrgkMGuvl4n24TTIX74j5/he+ZeOc9
l5/yNBebp/NjBNQz8Th496LQq9lYl6a6ZzQzHT8zHnSbZmXg4kpOC/UD6WAW
4f2UZwp29K/iuB+XhPd6519ybkjCNbVG34PiiRgSWZ1eG5uIxq53Ui/OJCJv
9eq/d84l4XrMrdZGZyokyouL6BvSsO/uBNdh2Ux82x/SJKqZt5AzcQUCXEVo
C6tv2MVfjp9nE+juAnWIv1f+/c6FZohajf3a79wJrRGXX4asA/g7vC5gftEw
JEz9r7i7MvCvKFUujsZEq+WNPVrNTLh8UMvcYsWAbH+d58xXGqzbjmid5B1A
ILvfeLV3J1yZ9+9JoQNatX8s117vh/UoxcjyBg02Q9dlvosy8KFMXCngMxPs
0yknGuKYCGb3F7t0mgHLXZu3J3nRUCM3/+qbWT8seqv8np/uwMhI6t3rM42Y
jDSKPm1YDWbmg/ND0SWgfLXpth/PRwd/t+iWE9lIKmFpOx+VBIWjt2jyCslQ
yQ7yd3ZJgd2vArkxlTwYpYtO8/wtRrCd2I8vCrVg7r9199K2Vgg9PsIb7dAH
xwCV5KCPNOg5s6ne+kQH+zyNuvgXHbKV8lTzomEovfO8tklpAOo1K3+t+taO
MSNr/4vWDUDgyrLUIxWgVUjPsZ4rwhv3yVh29hzklMkGp7zNwrPmfLsq4WR8
qUiYUt2TgAS9C9E+axKIO+KOuCPuiDvijrgj7og74u4/xV3XB963TwvC4Z11
cO75UBTcbzZfNCyMglXU59tDaxKhzB40d+N+NBQbNnDWPA2DhNP1/hs2cWC+
tFQ9X0OBooZ+p1IKBY6GWKfUEI/2Dfw2dglxyP27cfW/7Hi4xWszDYxiwJHc
fd3+9WfIl2uIjQVHwXrzqqBVtymQ7nj0bdfbEJRdct2XWEzBT1/+k2/MoyCp
sPph29eFeduyu1/6R0B2s6mqwFQcvs/VhA1ZJoDpfo01704oUi4zRdKXUDHT
bKZUyEdBR25W5AEDCvycm9rO2Cchtux7vbsYFcylQiwcn5IwL56aeZAtFbcD
5poVG5Jx8FCs6e6iNIwaUr/46aaCxz/K+kxlGlI51KIOGGXiQIfI5TDZNPSJ
iV3dqEzFRmFm2KltqfDvev5y96M0mEsuuyIplYrO/kjasY9UPHRSdBYUTYPp
nStQdc5Eg1q7d5xlKlr1ess2F6UjyILz9WbFFCwbez2XG5+Kla0X82ZfpcCj
hcdTQjIJ4wZ1T0f9EiHzK7Ar+EIiSvwWtb1hSYDc5CmpNYcp2OO8/+L6vwlY
EWC5nJGRiFTtAxY2QRSIHDYfGvFOxiqfIr13aalIWTvRNPshEyF3Lu/cL58H
u7BMVwvhIlCL+OSFOsqgq39TcdHhWqw8PkelXGmCSuTVln0N7WhSasgue9yH
nc0bpHJuDEHcr/l1wtcRhH3bq6PwZxQUvw+hVjFj8KvP0NLLGMNYYPD4xQEm
PpdWWrvzMjGw7mNm8t9RLDrE5GRNZuJynHbXiO8Y1gqXbneJH4PmKy3WyoV5
64wyrFJW06Elcim89fYQ/nSbbVHU6MNSD5ELZpbtuBbD4tlZ34j3f68c0iit
hsG8uRG/TCkE2Kzt334ugCqNa+fOV9m4ENh9fldtOjS+/cy/uzQRc7bDkTPD
C33bXscnVkvFVJrcOoWF3IzIzAtTu1oAk1xPVdNT5fi1bWV1gH8DLrxe9XbC
vgMbbl02fBU+gK+2XHKHuUdgLNAaNyrBAHPIV0w6nI7uxsKK4zY0uLB4PYu7
0ofIurS7d+61QqeVLtE+WQdxsRORguVlSPmacrV7YX10ZgNTrigPL828UkOm
U3HvnrFCky4VS/y5bo9qU4k74o64I+6IO+Luf4S74+d8Iw+vCcPn+MK7FQcj
wc4oF/ASjcTrsg45KW4KvlilZ3xLjERa5Oot73RDAX9vrR1RMXATbti9ziUB
v3K+SBzWTEBSr4v3fHosnJ2z/pR+igHXknRjm9pYCBW3n+nMiAK7qqn+3/lg
qE6vZFkeE4FLTluyh3rjEeF0kj/5fTAKZ54u/fcxAX4+1VmvGRH4MrF5pNoi
Eo9FP39XexCOCyYbfb7JxWJ7c5zEavs47Huw7rF5bzDUF7U9cGmn4NYLf032
6jgIm1UKf5iOw8yTDkW3n4mwVfyaGX2LgnOLTl1XfpeI38u0s5q/JKH1dv/7
FYZU+NqPBhnvSYFR3B697y5JKA16r+izLAXemZUOHbfTUBS/eTD2YTKmo1wv
Vt9IQLhecdb6hfPXb8e9ht87ktG1/e2KF5Qk/Ji78dhrDQUeLzL3OCslY+mS
huf7vNIQK7tnyikmCSXKp5bqr0jFj1DBuK3+VHym9cU+PZ8MRyaTmX8nCX3m
nwL7jybimPls6uh4AsyltKbsqhLwXbuslu9ILPaa3TY8vSYeQ6H6e2IK4sC2
/9HAQT0KVj8SkxORjIeScP10hQ8VDIHnoU5sydhTkf2de3UaWlh+tCmoZKIk
jiv5pH4e2FtnN579VYgXNfonTsqUIYEqcjsoqBo72XNvjDQ1gNVnc/TUyVbU
H9u640FiF1RbPLKeNfTDNAorDowP4V0wpe/O0Ah+aS7vTjEdRU7iIcnIZWO4
+fcW9/e0MazKK1hiUTKGQ26HHRv2jaHhR7meTMkoVD0d3hw6Q0d5Swp/ykUa
lMaWdZVJDIAn6VCw/vZuPFDsPnTcvRU5zvefeHU24I7q0q3aB6uhfEquZM3N
UvS9vl/G3VGA5NSOw16vc7BdQmbbvvfpMF2s8/zKplRcmNywe34wDhtKQ6X6
WChgEx3r442kIDh8SmJlQBqG71/XOq+Zg1evjnLytBTh/tomS6fl1TCKbZnQ
PtqMXG7Vr9lj3RBKEl/KvDiEjUYBcJCmY9hs6PAnAwZ+j7mqn1pLh0Pu4LV/
O4bwOHmF1tvEblzqeHEj1r0ZbwbnNuqtq8W8k6zjRe5SKCuo3a0oKoT2rYLe
x8ML5/sgnVsVvKk4t6Tgu8uGVOKOuCPuiDvijrj7H+EudZm3D2fgZ/i+p0ua
VYdDKfmY1dHMcMhJzZnJX4qH7O/WHNbdEbDyML3+0uQz6nVOPKzlisbT11PR
Qflx6C9/9jX+eRwCHXmio+nROGFu7by2JQqvBxvuOAjHgJEhVmmLSBQ+ENLR
zg0CR/WhF1O/w1AU7imisC0OVsOJ91d2B2FTUvapnQt1q7eyGDj2JhxVuQb5
a2QioLHKc0WeQRhk5FJcxeOi8TOIw72lKgY1GRlsEtsCYXi5eE49MgE6LEnR
qI6BzWvDfXJBMdjA13BZKIICJTb+4cjZOGBUz5Ezd+GcpNfhqniYCqMXvU7D
1hS8Pl++V1gqCcsEWBkNFxPhunPGtWKUitRNWzzWrEjBLX8R6sPTVCS84ul4
axODK1tjDwwWJGJdV//GT81UsBjn9TkUJyJGybXE6GEsrOzuzoUvo+JXNLvs
tpMpkFGxqOz/j983LsjkaI4ngVkVEvHLhYKzgR5+x02pMOQMjeJLT8Q2IwGP
mQcJSG7d17TzcBwiKSLC//TjoB0ofMX5SxQ+7ItlP/AvBp5Oe843N8VA6Gyb
Yo91PALqJnxmg2Pg+utM4Os5CvzrOdeL9iWiqKV14t+1ZAQ1TOi6B6Ui0e2l
XbJTJm6Uh1pvOZaHz5k8q91eFmIujZXR/acE+dkhi84crcScYw3r9Vd1ePk6
Kf/v0iasMFVFVVkrnM8n2ybZd8Li8G7LtqReqKsquUSyD6D/9va/E28HcbXB
/tO20CG8rHfdsKlwCCx5ZVsDOwcxMqPbNm04gDv8jF037Pvw+Ddj/OiFLhhs
5nafvNWGxYIe1lNuTeCVXsY/JVSPzmCNbbctKhFX8valRVcJjFMSqjo4CpFt
nBfN7ZWDYe7tei91MmCxR6tHyDUFB5a67hXgSEZq3bKNwvbRGOAUe72kOxam
J2+UXh2Kw9fynydyF6fAOXLXBs5DGfj3LvLmxVf5CExvu3bGpxTukdu87Vnq
8eTZp2vcrG3wNJANe8PVj4vH9UfV62gQ6mD+2bOEgbLe586CzxjoCR3cFqA7
AkVZn/dqrIMovcKxODy6E3ZTRygbnjWB9eaiRYom1dATb8jS7CqF6EvDp1DJ
x2w9b4rBuSyEuIas+nc7i7gj7og74o64I+7+R7gTL+pYStUOwXlGeM5NpzBU
uk3VX7QLQ73Rsu2qRbGwcVOgLhkKw+L6WfuxmBCIDFdVro6IxEX7vFfnt8XC
f/LiDpWhGLhMRIQtVo9CwTenlbJKkfhf7Nz3P5X//wfwSkZoSGkqSqSU2RI9
SkYhlEoUUVkNTZLMokhJNmfv49h7b0mIjMzsImkqVKTv+fwV7x++fjo31+06
13Wu83ren6/nw83NJLPSWIUZj3mFHqTB72z8ZD4/WeVGQdjqe8Ved5g49UKA
7stMgIf3fF0FUyr8PlxoEjydiIV+t4JsT7HwhSLH+iHMxpKmD9LP+ecrdcan
pxyJR8ONT+POBjx0Soa9lJ+KA2HB3g9hfUnoH3519KoUD6b3VRjnWPH48+ru
thUrU6CwQ/n2B3oCts5t0JH4kojdl/1CzHtSwdl9yVOC7zxRvGqhT0AaFuLE
dZt1KUi8X5fE3J0G0zap7tENGRBg2m/dFJGKgoQ2PR1vLohxvlQpzVT8oCjK
pGikYUdQseP8vfxcJxta102MBz6PT8hopSJuqdh89YsZiGAOW7uwUnD7d4WQ
5o50vMjwc9dsTEKXlvsJ/9FUjHv36j0155+vVShm9y4RxWc1kxxFE5Chr0H8
eiYBMekBIh1fOFhaWTGxTIUHZXLMqS2beJArL7EXDkjE9bNvuuX845En3Sj8
+lAKVhXuFGpfn4o9/QMW9Trp0I1bsPNfUwaGd8Vcr53IxrjHfO6riHz46Jpp
1uUWo/IgxbI/oQxNck8V/A9XIj9mR9Y37RfonsqJCnpdg/J9DOX0L/W4tuoH
p3zqNfI952uHhjXhzwn56KKOJrDKbFUk/jahg/c2qEu3AWqkO8E7E2uxXnL7
gDi1CYPmAbSzv5ugwtmXItbShKcOlFdydo1YZ3FB6A+3AVGj9k9T42uRab75
xVTjC9j2VOu3Xq+ENO1I7SVCGbqPnR3etKwYGWLBS/ZuyUN75YmH+TuycXkp
M1ldLQPSmz4Q35qnY5XrYtHxEg4e2dude63Gw5X6PxX+axOwgTvXjPo9HUUD
V3ckr8zGb0/ZmiurC1F4UDFv8Ek5dqe0rG1yfwlX1eW2lxc2wbfb79PW+A6k
KPLyvnj3gyhRq0CZfo/tx94e203+gMXSXs8Y1z5g3XHfI0kJ7zG5TmLdwzX9
uCOlEfLGowPvi4RfOTc34X6hjPPaN3WgRK3JPUKsxMLnmsUykWWQELn51YH/
fc+6m3U3627W3ay7WXf/H9ypzQwZXdhJg9Wr0ZQqJyZEH7HNLrow4bfvfobg
qQRsNq7PUS9k4n3gC876GRrMDzriqgMHtHXEnxLFPAwzl8vdvsWDig/aeoc5
KDljd+3vOzaaeg+lXDLnIj3XzmPNSxakhYwOLHhDwnxdd00DRQb2b1CefHSI
h8Pjawd351DQ1EQ/8HqKh73TwQHMQ0x4DVmWL//LxLOqo10xbAb+fnrmcH8p
Fy+yiZZ1n7mQPqD3wlA7GrcnFp9JOJKIj9GMmxNxXOTfbHy16CcHIzONQhdV
kkAPUPs8Pz4etkmetk8e8dDzpFP6hXMKWoX+/ZU7moBEYYbhWvlUnHZNSxmW
T0SaW8CJ6aQURHem1kmGpMGcNHZV8lQyLF05OCjDxsJe+Q0GP5JwrWD/pZDC
FAjG6Hlpf0vC+8ztwemXOAgOGvOv9UxCc86zHsW+NERuPVnUPZyIAyUDdk41
qbB5MXVktWwCYpfVhjoYpMDloVSIw9Fk+JxVbukq46GpwXGBhEo8pCkvyW/J
8RhLPCsDDTb+3anIkeQ/n8Dwq+pgHhc7LhjPu1XHg4b4MeOaPA5aOmKEMoSS
IF1Z8br8FL8P5bbNbQ5JxbvFtz7qkNJgeD6UHIBMHLi/LuKpQg6CJpeeSTDM
h8wfCyGtx0X4SJ3Yl3OzFJ3Lbb95yZQjeZfpVg3pSkSu8E7P5+egFQZG24kB
L3Ds/Ms6Wddq1FV46WQmV8G7efET4rYX/O/DKU74XRWGVws73v5SBVqW0Mue
Yf6rjfWuQM8qPNyyJUroVDWCfOPLs4teYDkpwcXhF//4KR5zdXUlVvupHTDk
OzAQCJYq9itFVtbQ8V2XijChdnLYvSEP+vowqerMwqGFUs/zKjOQRar9F2aU
hrWT24948vvXFwWSo+UtFvaUrLvfNcKBZ1lpiVszF8daz6wb1OGfp5Om6UXL
wJaz6TcirHMxs0I4X0C2BNuX1iyqpFSCZaoX5W5bh+GO+0UjS1sw94f+GlOB
t4j8yI4j8QYwEzb/eNO7IVhZlIwULx/Bwdh4LcbFD/CfkLm9Yud7BMV/PyNM
7oP9zaCi3kMd+POVwV3Z3oSy/WY/GY9qQRQZX3Cq+QV2JVY8jPn5YtbdrLtZ
d7PuZt3Nuvt/4W6rl7LdjAYVLVFzKURHBjY/Jx356sHAj0zDb8lLeFC4v55b
mM2Aw3Rn1fYLNDh8fM42NWDD622rXtHteMh/8N5TsTUeNfMj9p/gsCHX2L7t
QSILeuscO1TkOXjaJyG7pISJK0ODLvn3iLjB8dFZ9I+G6jLTBWOL4jHf7ouR
/lcyXFv0k12L4yHn0lthc5IBwmGThTvEmHhcK33L4QMd8pmjY0sm2NisSHww
ncNB4NQbwWNXI0DYe9Hu9BAPeeWXzSxN+P2AQBp/6MU/75nC0KaHCRAWEz15
+hcH11tMeyLvctGhHbM3/EoSghG6gbQ5Hmabbi9+/jsJYtXlpyX4efTKQbn2
PtskCAa727XtTkXeNRNVkZkE2I0qPBXUZGLVdx0Hn7cJ8Heo/zHfOwlqInWr
HZoT0Pnce45UKgu3y4gXWn7xYDK3juvFz4WvQomu/iweQhzz1jWqpmD0EWuH
4xC/b9RF5S9ck4R60/6D6scTIful4ePS71xsaB1tNCFx4KbMIr1Zw4Xexw1h
+9KY+CVUsMBWnYM61ObqnOagND3zqeOheFy1+pzmp8rGgqPidqyVCeAtHxmx
fc/Dz+huCPA938tOGVaMT8HcOWoX8telIy9gQMTkSSbyym6tn1qQg28iZfOf
uuWB86j6mF1+AZQ4i0R02EWwfmNp2jWvFO6tTy7O6y2D1VqO6FzJcsgvOK+t
kF0Oq4kg67mfS2ETwZAvjClHSterm5uryuErWHFPN5f/eqDT1z+iHCH1Sgtz
/pRiRl1weWF0OfIOK/kQ15fDRDZkjLG2HIN206nihFIc7LuycrVRMSx+cAlb
uQWQ/rj4eZ5RHn5s5QnUsrLxU7hDyc0oA/qMZWu4TWlI3+cqrHo7BX9sdQVq
3VOQWrRIty6IgY2M5MeZn1jYdW3qWsQoG+zwrCs7ivj5WuqvyTxuGtr+HJDR
8MjCL+dla2J5+Sh7oi9ElCyDaUqPVQ+pCkEqKSPqBg1Y4vS1Y/vLN9ihfkQ0
Y6QbYne/mu0MHISMw92/c58M43vW25rLwSPI1Bl8kh/zAfKO6j+y5d9D8vjY
+fiBPriqfRgKutKBZc/uzVkX2Iw5j1cyhtc24pii0M4Vqo2z7mbdzbr7D9xN
bfkQpUUiYzfTuHeDLx1ZXiN9z2l0DMxJahwb5kJz7asXuq/oCO7TOPOtg4rJ
s72x3roshOU+CMs242LvFer6wwJc5P2Va9gQyILQ4uBPwZFMVAUqeI5LsbGj
wXrsVz0DIvOOP2q7QsBM6aH3S8ep2MXWtiONcbD/0xqjI3vJyNet/L6UzIVa
5WiCoBcdpecndnRsZ+D+vJsJUlp0nE0uaO7/wcLyHNmvN6hsWAsKd37zCofy
suwfXox4ON2NnfdbgY1gjs1IxU4WRKgX05z4fcOsY/OHPj82SsfbFfpS+HNi
lOaJK1kJ0HUkGmUf5eCzbun8mqkEUDiXpl57cODSfy1YxiUB764OFTYoJmNK
jPQweSYeDhUXXqgl08HzuS72RYqHlZ8mXlyPSIBI+d6UViEe2t8cNXhjyMSS
lj+/+l9zceBwzhrBsWScMqJejCvl+6mUbzZTSEJF9fWjMsIcWJQTm25oJQDs
+AtuRB7upn9MW2nHQZY1h3TzIBtdnvct06rZOB/YIHz6DAPPjO3/6v5kwVcq
QsZJhQ0PWcVVHV85aLgRzz7Sw7+vuPyybVfiERm4sV+kjoulbY/yV/xJwGOD
1MHUvEQo5qk+sXROxXxJZzOz++mwW1nEPB2SieOqGTaaN7IxtrMwLnhBLvxl
gw5+XJWH3FNX1IKj8rFS8MW+eS+LsPR8RMQqahHOcueg7WUxXOitgkqaBXDc
YTDNUypB32k5ScmuEuQuPS3/trEEtK6SbAWdEvxS7PGc51sA18EdwxejimG3
TNQtNbYIDj4N0g0qxWhtWXpmd0EBjh+RG95ByENX3/v5bY05KO+TuHNu7/9+
/9goMLQrE/qvYl4R+fOi4Ol2uQCtVAQ+jNGhdCQi6VpG2uqbSVjRnuM/+JUG
mbH4g/uUmIh7GvVP0IyFgH1v8x0dktDt6rwxWiwV7Zxk0zHVDDzxD6TU6eWg
oNN1efj3Qqza++1o1+VyiN43zmTRqpEybLhRofY1nknt7L4j3YYM86kTa9b2
4l+WyIYtfwYhnv5vT9n7YUztVO1jiH5E8iol98ZjIygZc9B5XvMerd9tTiUz
+kF3mHkbI9KN0P40qSxKB/7VNJ0cSOuYdTfrbtbdrLtZd7Pu/l+4w42LnfEz
RES3y0YFJ9PgsnFrQlsPDRqrRv7e6+Bg3Z2s6oApGj8fLZx78QwVSRp6gi2W
TAQ84Rh76HIwcPTxkBk/d/2ecylJP4B/vPJZgwiBAfXipHnGsiy0isVUCY7T
YXNsnumdh3EgnLrEyhOiQvN+vdO1r2z4awlPikSScPoJkZYSxUFsm+mWb7k0
GAZGrVS0p2OxTUyMHIUG8UD2+ZMiLDS9G+oto7Mg/7l3TspkOOyN9zTd8uFi
SKTfXZt/v6B0xUyTFUyIbTj4TqOCi+rIzpCzmiwM7JNNjVrNAuVOAPO5Og/1
Jr2HfxewcGnNjcZlB3hYU3TvdUs0C2argvslyuMRff9+n4RKIowbEyae6nPx
ZYnRvlpHGpZd/njibygXZ8OaT9d/jkepXwx1yx0uFkoXtlVt4j/36FSMy3c2
7i07tGqXcBIGegVWr5LjIObUF8mEcwkQ/Sso4HaNBf/4sxe+E+Ixx1G+XW9t
PNLSJi9bCrJBjE2fuLqChQK9TXJfiCxEWTSNYjsdlqvoCoxWJpz2n75wU45/
fODPjewCNso1dORN+fk7a+nR3+aTHDg2eezf+4eN6MImcRd+v+i8GfjxvR4P
gV6Rr44LJWOXvnF/QW8qtFLHD5vGpuOt/bs7UrKZEL2dfizSIAvrI60Ld9dk
Yf7v8Fan/GxYLqApXYvPg8L3LaKD2nnYvd7O+t+xfMhdajdb9jobL+JPjV+c
yMd+CZOP8XkFGC+4M3aXnxNLN833/LGqANVzv9lkbMsBQ+e+lZh8Pm7drLEw
l8+DEnvuR8eF+TDY4t19oysH24QEXRJUsjH5qe/l6e+Z6Li2YaNVTwZ0+3oD
wo3SwY7dcNV5ZwpCQ3R76t4kwTHr1ZaHVB50ZdZy5DISMCfX/2hNHBXBvx5P
LiPRoSEz747pdwZ8c22ObnRKgERi6tpFl5Kg11FoamiVBuEezdtdtEx431w2
umRRHp7HPn2o8rwYF/ZJtTi8rkBFfPiy4g018Bs29ckgNSJhsW3GoZw2GLmK
26050YuXqi9iwrsH8aK61bCzbBjMj7o5knYf0f3pz6PRko8Yj38b0hj8AbXj
LOXlC9/jXJl0gcdgP/xi6k9kD/TPupt1N+vuP3D3W6jkXNdDAoJWhduUf6Xi
oN6FoI7tNMS6bVzR1c/GSj97MfF9NP76Onxoa6IgJdXLf+w+AyOG3i7WpmzE
X9TYKjyXDf3IEFZ6HAMvUh2fjeTSITVZME0+wAShZfdbWX7d7LNYXDpYEYuT
TRuSgm3JqI/Otlk8j41L3rpfP/8iIvf5dYIIiw3RMKVdG4T59z+48GR1Ag3m
stenvRbTMBhZJu6qwgTi1i89VsDE7o6XkefSIpAoL/SswpUD5ZOSTwV3MJE5
fmxB+EoGDBKUrAgxHKiXTAQcW8UEZ31D5VQaA7elj7h79XDhWK/8Svk0E2e3
m74dLuTisSSm3n1i4MCNivPOJlywk1z1zrvyUDk9FHfnBRurR0OzrCyosJj+
4zigzcEHpy1mQ55crFO1VC6V5sAsPvLAPFU6vu9kU1SNWTgrI+SebZkAqZbN
YRWlLMj8LvRTa42HbbaTnvMyJixkN4kMbuPi4j0/5t6XHDSNOa41amHi37oj
MrVLmMjqvCL64xkTJ6uKcpVkaVhcstP0yRsGTvlpxn5QY2L8oaPfOTYLCkpC
9KE0OloEetuNkvhzfXLmDu1zLOy/clF92JCLXBVNgznqXExaLRudSUvAP7XY
H/cXpeDqdvWtHQJpaPyqKTVlmI5UrUY/BbkMVHIiVPX0M2BY/1DBoS0DKsrM
heGkbOyhlG5XWpaNs9MS6gckc9Df/eu9Gy8D9/61uW57k4Mb++1jpaNyUb71
kTiJlAuDyGsxU/9yYElsEH+xIhNZW3aJJo9mI0gm7NimsSwE3TR98E4gBzq/
mi+aTWSitUg8ZUdUBv7NadtXlJmOJ3PWOez8mwahpQuWzSlMxWm9kLUWdxKx
MWKg6gghAZ6lW3k+2VxkCrqrr9nKgwLTuSZjHgWZBpDw3cnvj+JsUm4FHRcu
D8hOpMRDYWYRz3trAjoth9r1E1Nw9uf1yqqH6fhUpPrDZXk2Gq/ds/5inA9h
1VpPidOlULxVGePcUIn630d+QbwWfc/s0nZUN4LnnX1F+WobUkxuDQ2a9qAu
NdpvenwAe9maynezhpCZZ/nHvXYEezynvyxnjsL9a/2A6pdRsA5ftvwSO4rO
Z3pzJOJGZ93Nupt19x+4k2zTvPZIPQ5ur5Z/P3CAiqsXwz6vfUrFp+GrFZIz
LGQ7dRgLhfHnwvaybNoMmf8cw2UrS+l4YVFw6owLC9bjWwQPy7DwQqRMZPlz
OrTWxn1uHaXB6UD3g3+3GLgl+GtD9W0aRA5tLdyzOha52fYRR7qJeNlEO5Op
wELoyo9pXy4QwfwkleVUxcIKb/TK21Ox6G+5f9EcGt4pqBpeCqZCcuyDYZUz
Ay0Wx+tUhxn4ECDkteJgJD48ccivvseGdWnhJt2zDMiukeWc2UHHQbNFcVuD
2WDqvIqQ4zv8R0VigAsd3AuvX/QkcdCfPLduhTwDx6cPDcTZcxBoqWNy8Cod
7YmvH/iKcNCbUSjh+IWLyAVPj/7v/6gJ2acfDR0gI6/a3mjpMjaykum0xl0c
WG4uXa88ysK+1NyDonY0XFP7OHSrlYH4Fcu9T/bEQyL3l4xpMBNDiy4FUW9z
MZWXM33kAx07RgNrnEfYEA7PULhIYGOhzDb90CoG7i9mZXzbzID8Bru+J3QG
FqyUqn+iQsX70qUzcz7TMUDqPL32FAMZ9qR0ywQmJhu/HF9TS4Ndqu3Yw4cs
6B9rNeb+ZiB9Ecf7TCMbJA8fr5Qr/P6oqa5Wlh+PI3OpMzX1iSiJCZ+n4ZyC
77lN1uV1qbhy/vr3CZc0XHFQkpt/JA3SuoNPrWPTYP7sGNFCJxP0L9dPrk3J
QK5F+EGbqEwcWx93LuRoGgj3K+vpjln4La5Ii5PORr/wIcFVW7JBMUiXJUZl
wSCP8PThyzR07h/w9rmZid8J9prsBxmIMzjSVpGUiR0jboG2Wemo6Pn9Pjcw
DWuuuno//pgK92Nnjoi6pmKpsULU1/5k6FgrPDykxIN6haMhKz0eXpr3770R
4+A8z96d+poL04gL4oa7SLiX/t7fa5CC1voQAa1iGgJm9h/eqMLFqbgif/0D
8bi3WuJj6IYk3Fb6m3W0KRWmR7xiCGUZ+CkZpS33LRvTZhNjHakFeOcvoctZ
XIZ3Xid6O3Seg+ew3cP3Uw2CTAg7rhk0YmDR4oFI0VaED35WCeB0QU1zaLB8
uA+vDjXv3rDoHZZlUfLiOoZgf6xYTEhuBM3PNgQdN/8IIb+boYXHPs66m3U3
6+4/cKee9nLFib8xmFg1Vj9ygALFN30jnX8oUGl7alG2lYXDwmnrdT9ScEKj
q4izhwyTjYF7/4nRcXOaHTtBZiLxB4nmdZQJUmNuf8pcOuSzJ7ctVKXB45V7
w5w8Op5fvWg12EDF06wHz6bux0BnNFya6U2Aovr2e4tt+PNSboJcfRsBsl8J
p4J/MSH2WTt9mMa/f01mqM05KmJGJ056SVCxK1Uxd086HYKaB95+28jAU9ua
vFuMSLzV9pt5xGDhvuGBl/lE/jzrLO7g60xDoWfxj9AoFhKOu7Um8Oe/rzce
+OgZ0lBZykr/zK/zw5MJO4KW07HySEVRtjobTxWOtJClaMDMkwdaH1joOyvW
aRPGgcI8Z7Orxkx0id8v8rckYaXHtKaBOAulD90Jf6TY2PB1Rsq0k4nKOely
D5KoEHF92jI/kI7AN70v7j3m4u+yk1s17jJwsDBdfhk4YHXf6m8epGFm2YZT
rBcsTNs9deU9ZuHbKb9951rpCF308E6mFR3LB48nxdTQ0X0yf4d9GhnDnNo9
6avoyBvMnhB6Rsfhuc+2feM7pSg9cFb5S8WVJUlkGV8mnI5Eudzmz/ka+wqN
1vIdzm90qeMOMxGi37tPdiG/PkOom68O8XDFpmm6RSMJo+s/CXasS8H1xVr3
nBekojlwybexyyn4I7XPdZKYArkFtTLMs+mYzNhtpFKfhjnqHaI8YjqszUY7
rOenwMG6e7jDNQObaAGmA9KZ8H9bvcBleyasjf6KXuFk4Pn+ZRPr5qbimZ7w
+KRXOvYduFT4m5YG+QZ/eZfGdBzwrkhaOpQK1ZY/Gz2KU7DJTsHn5VAydp6w
fJNvnYymlqiSqYxENFRZvbbZxcUOel6VghgXeQWufyojWNCYP29FFYGD+eNF
xyQrCdjsq68pF06C9HTziVcdVIjq7Mk60MPGBaHh7Xr889RtDqQ3/uFhTp+1
+r+GZLTfqLatHUyDs7rlUYpFJthipYHnxnNwssB18/qaQpiWTbpla5dhb7+i
I2e6EnsP3xs5/O0ldlFMGl+2N8D33oVLBaea8VpYR1ZVrA325/Q2O5Z2QpOb
J3lpXw8+/myyDd7eh5yuEwsoWX34pMVMF8zvm3U3627W3X/gbtfm/pyn/dGw
quzP0lpNxpuddYZfvpJRPry8pNOJif1JX2b6a8jQi1IdHvEjISlROb/Yngb5
f3KvJHsZ+Lb7ZMCdMAbKv7KMNE1o+NOsWqL9gIreF76tRL5P/V3tF7M1qCjq
mCTcnY7Gd6u/H47sj4O8RdJcZQYDRxN/Xi8zI+D2hxW7H+xhQrVX7AvnGRlq
jaa4T6ZgkUDyjZn9FPg8NYuUEKHDKvoVNYA/Dzof7tOM7orElZJUyYg3TNwX
KNte/ZGGDMX286Q0Kt48XjNinMPEbiPOrVf8+S9PZ16Y134qfulzVcf4TrmP
fKIGNtKwFanxZ2RZ6NWwci3NpWDo3COHE/w6faP+Nfa6FRuhhB22EjoMEH5T
5RYKERHieX3eWxkmtiY4yrsvY8HTxPRl+HsGrt/9PW65hIqbn0fXe1nREH8n
WPqhNQdySzbJvw+i447aI2NHeTb6/tUeP8efn1PP8hS3VDDha95ovTGGie22
q8wV/9EgefzBoj80GlTdnu/OE6JDktopp7yNhB/LGpjvTtHQeEg3aEMXDUsO
k+NlJ+hYNCiVc0qXitchXd0O/HVg9RHunIuhQaeL7Tt6nomJL8buUlEMCMkv
MUiKY+OSqsG2Fp147CxfKk84loCq6jt2CUJJOFf6Y5lLcRKi1qjm2jcmYnJ5
xGJqciKck92iVZtTkW74ScDZMBVulJVZU/9S8bnq3IUf8olwehjkptWdhsFP
h/JK3NKB4zX9ZY/SMbdFT+nxknSkbf+hP6SUBGfnDluB96nw2btm0HAt//02
Pe39Rmngpfz2oTYmQ7xbZW7yTCLYig98m24l8vNM4tWZHYk4HzY19qOVhwii
JXNvEBvaZanMSzFsiB6uCD16iImwd4s5tg/YuDPdPl4VGwetngHb4ZsEJAXe
vNbOXw8lC2kcSWehnFp7M1aJjZfpkv7HZOKxWII+utQlEadXG5QEnUmF3cUF
R5sOp2NOYaV+1eosDFv3i9m9z0XHhdQZ2eZCbGVe2u7yqxRe5JbKo/srUVP5
tsfoxQtoCtd7xN+vhcCG3VpGU/V4uk1U9Qr1NX5ckXS9urcRVe7N6QNH6/Fd
ofNekNpL6PmYy2+rqp51N+tu1t1/4C76Z5WFbHcU/uU/Ng4VIeHy93tV9n9I
OC6ecpydwcA1126/+HISlhJDHOveEOGysH2msJaKH4Gp874qMVBy9VihUz8d
r4yNVr/kUjGXvE/gWxMFlKLJ6vWXaDh0qVU+8jAF+65dnDznG42DZM3Haeti
0Wca03F5nI55f0UXaTfEwaRLSzH3IQN3ezZpJEWR8F5cnnDuMRlhIgcjW13I
aCgwFm/mz49viMSI3dU0dD5anHx5PBKCGu6yb6SYMAo4oKNrwK9bETHjr6JU
7JRN2BXwkQGMOv/ey6KCS3YYS4ojg7Xl6fdN5UxcTluhzjCmIrvJ50C3Ir8+
P7w/Z+VGQpSKalX7HCYOC8gYbjvAwhRppUHuMTr2b6neUtAdh2LXFWlnjjCQ
tP91P3ErE+p2N8zJIgysibLKFxClwMVNy67hEBWPl5TVSxjz15khW6eXTkN0
+IXQ/M0siN2Rb+DupiJzya8mehsDC/5OrR7PY2Dxk/pd6to0iDkfaLg1RYWp
rdRxteM0eD3UItxtICA78vQ2E/73u/pIiMpRDRrs6M8YV3bSQT0vUzCjSUFL
EYcUnU9H7/Wmu5cjqNhW2TVkaceAxuNHL59eo6Ps3aDjjVMs2Iua6uu0chBQ
OL3gUFo8vw4t1s/dmwCrhb90WisTkPTEXUGFyQM3bP6vv9o8rIl9co6wMAWa
B89IP1iejEPPTCLbTVOQeGbd5jWt8TD57Xxx28NUeLd+C4temobTc5VFuYpp
OKiaZJdZkIq1wps3n+/locfV+nSkTApU/K6JieQmYce6ZvIjoVRIG2e8ox9N
RMnQ+vOXknk480PO1f4cD5dFTP7kn+HhErWHJ6kZj/2G4ic/8vtcaoVK7byj
LGTwRgd91jMQUiq7+FkwCwc29NqHxseCM3z8T59PLH6WRp6rKyKjKfit/Tw2
E49MdtEcRpnw1r/dV3WfA/P8TrkH9jw0LSq1Iyslo/qk6SUSNRW3XiuEJ7xN
B+Wd+HNMZME6+1zD7txcvMgSrtx6oRAzsZLfI+6XYmXJl7w/meV4YvvuwNu6
SpxR29LVYlkFa/XGC9b6VWCvmPcqancVeAKGX93MyhBy4ZfU8epS3FVzGDhd
VzrrbtbdrLv/wN3LkRVLnD5HQuqJ96OLokSkZyT2J68gIXyNrWSCKAMj2v0f
e+qJOPQtY/OMOhFf1lqMkbSoGHm34J3QQzqSRZWlirXoGLet6S5czJ+z+sbq
tqeTsevNuy0rW6n4vXQmrkKbjIO6OommC6OR8UR1qEAqBqmpZZm25nSYxQiK
dRyLg7n5u/oCvt9HTWXpAclEkFjWrfqPSZiZeSD6nEdCpVHvtz386437PoqM
UOLPjU3Ta1p/RcLJe+XYM3sGYphzOPapVFyQObDIcoiM9B1V5r3bGBAYirrx
dIyCZxHkcYsuIppM1CU8fjAwbWIF/ZUUGPpOx7kcZmDDhP7RtdYEDEq4rmhW
YcDMe571dh0mAsvUgh08aPAvk1Vyb4+F4fZC5z0hdOy1fWzGPcWARWFh1DfQ
8azw5NK9S8hQLW943k0ig6bk1tZlzsL0ubExo29UXNzzdWAj/3riY9u76Eso
EDHO/ft4LgOPtYUs/n6hQypi7aDXEyrCmeEM9wtUSJuPu5Txn4dsSy5ZGRGH
1DkLOHOE+PO628XlL+Oo2MLOtCsMoOHRWOIrbXEy1oi53WSO839urOPKH6Xg
Z5h1vJEHHcRssx6HazRYCgb4TWkz4ea608Ihio24xrPWm3S5GHQwj/iVEI8p
h9NH9y7lQWTq5ZfIOi5knJMndbZzIRrh6NremohTO2y4w9MJyMlYMbhFJAl7
mgM/an7hoEf9ksiNar7HMzpn8itT0Cnq0mn1KwVKinOjtNVTUOR77bHQJBcP
wooFP/YkQuPU45PvKAkw2+z8y/hdEv4G3z2YpMzDsN3xG5WZXBDqP1n9juNi
yizyfG8ZFz0nRIK/vuagxql0U+BrBsbOvt5D38ZE3cvqjfXL6PCV/6x7ncNE
7Tp9zt/3Mci6719vnBGNU39Cr2b5kuC83c72UiYDpjn94f8aGHiivaMifhUb
CvUFG/oZXBA/LHBaOj8R+k/3bgryT8an5SPbghalwTzo67385Axs++EXoUXP
AqVhZ/0yp1x0dsnhx+cC/CnzXzPQXcyf/+8PSZSX4uT5wygmlqFZWPPRnslS
bLIvulKTXobHYSdKTv8ugN3fD0Oan/Mx1h73Zm95/qy7WXez7v4Dd+s/br6x
cUkkAjIveSVLE3Ba1DqcoEfEooAT81iX6SB82Gyv/JmA2Mq3Aw4EAuTn55pc
c6HgXrrWWYlPNExYrV4nz6DBz0iizFGAAkLXi18XOSSsXCfxxNCIipxT6x61
HyTh2ZZXn4NpUVhyKKv3+aZoEOfb9rYU0HBj8Vkhna5YMFbsWWOtT4dh9Zfu
1W0EXCOlGZwnEnFphYSt0Q8iVmosXa/MnwsdrO0SrQhUfHK+LJE6GolXnm9F
jpby/bvWDCXJUdEntzCl4zUJBpLah+x96Lh8S5JwO4cMbLVOKtMmoHBtk9CV
HQzsGz+tU1JMwqfbFkqTnnQURx1adsw+Fut0zssuvUXHlhH1S32WfLebLnX1
ZlHx5Xma/MTXGCj7xLCe9tHQsOsLNSSGjrZc0Z/ZgTQIbR3YprWdhMUbr93e
0kzE1/Ydqy5eZ+K6kMSCwv1UmBmaPL51lYGhjcm/wttIeODoKbhgHx1z9Z5m
J2jQUb7/0gLZLgrcaB0+SVkU6IgPt/uvpmJrlmbo9duxKEmt+p7aR8aeMffe
+QupaJLYc8pwgIoqUc45vV9ELBBOkxU7QENXklflVgYJCy+VlLF4NFxLk7q5
PZSKulPLw5m6DHAc2tJO3mXBZ1L0D2UuB0rlEeTXR7jwmJyp3BrDhf+ToJ6k
3RyQ9NrO9l9k4/j8fXWfP/GwZGdTcqQ6D3s1H/z6J5EAUzfvsFsn2QhwT53s
akpEqsvT7z7ZSdjEaXCT+ZqEVWOrKYP7kvDrt92SktMcrLWr6932hYevIRIH
1drj8VoR8+eJJMLMfNWnQAsufN/f2138j42xDSUy/os5IPtaJ3lYcvBkiPBq
QxgbxRePiNtl0hGddULi6WYGrB4uNX6zkYZTx8n17a0MxH60pxYfjIHOBZ+M
WvEoZFgIqD26QASH5mkZ0sFfJ9mmT+Yt/Lww/XuZQgcTEdbXRBzXcJAk6Rv6
RpGHSpurZmN7EkFts5gTvikFy9bseq3QnQY3q0uCpVoZsPiqtK7lShbaLvs8
EtfPxV7tX01pZ/Nxp3uTYCwKkRQ8197brgiXVutIFtIL4FeREkR6XYQdbhYW
VUI5OCbyUi46PxuPo7pebQ3JnnU3627W3X/gzlf6rNv5gxFYPXkpIXh/HGgJ
SgJm9whQkOYt8mynod+zSI68hgDf7Z0nV4vx/b3fl7zClYyngepj361osJ7e
0+MpSYPvttu6M39IEDzTNaCdSkSt5ZEm03AK5N1ld82xIkJxxlOsc28UCk49
6PQ5HAUbks2Tom00UEZvfFZxisWPtpbfoZn868R99aQtIiDVyV9NoYCAZbea
W9cbEPE2gcXzi6SA9sXtprskFd7KT14ebopEuaKbacMmOr4qSnZn6lFgem/z
EqUmIhC9Q/d6Nw0f/nLfPowkYbdjnfGTIr4r05GE0QA6PGMMdiQ8JIL1buTz
UBkNf7Z8u7QoMBp9P4sif72gQfo4XciMf94a2WObvOZRYWT0KuXg+hhQra4Q
5+nQILj1+uuGDzQQjz1cQ+ujQs1ikLTSlgiPi19alZUIoLpP03tpDES9uV/4
7QAFQUvqfy1KpuN5taHg7lIiPs+Xccx+RMOS+ktt/Xy33ABOfWE2Gd8pbgsf
5JGRpv3tzuv1FNh06Mc4u8VgNPSCQk8tCT/nzfvqIknBPJ2zWjGGVNikW/hT
pggINi7xzeHPn3cNTdzCVxExzzr2ncT//s494cKi2igKGs9e69voyH+ecx8F
P3ozsfs4O9pmgIWrE3NVOSs46BSee3vAiANzxIUodbHgvXFwwYt3TAxuWZ/i
vT8eBOKazzVcLo4pGbi8PBOP7yndvl+mmRjyfHKBLp2AhILclkcfEyDhfLGq
SCERPSpV570fJUDJR0L7wjQL9ul9A0Pa8fioWu9fbMrFtl3pYtlOPJjzmmdc
89moKcgNEb3HwpyTCx6WZrIg72eywGs1G7fPvui45saCx7tnXSfyaQgjuPnu
1qZjw/3uMMHDVOy9quKvuoIB5UUrRj/kRcMzJ1f345MIDG24FOJ2gYCsMeET
akvpCP1E+hv4g4ao8MdVNgUMmGttup9UxoLc8wctkiFc0N6kDykd4qH3zO9Y
L14itpoEBm6zS8XQF7N1f4+lIbTE+9YBmQzcXnjtc83DLLAfayW8Wp2DPSZ2
SoFBucjqYnV8sskDu9Nod6RlDvZLdSU6vM+D5jwbZ5PADPT0htWfq87Ayw0F
XB1Cxqy7WXez7v4Dd98aI3o9AsIhOqr34dr1WKxWEFqhUReHlwteFmsd4edx
/Xs3XU/F4UiaoIDpvTg8Lve9ezuYhALtrz/TmqgQCRQve/6YikdNX1TU55OQ
yHLjRdYQkL/l899gKhk5Do7VLG8CRj6pCJp1RSJDuu9xv08k7vhYvFPlUnHy
gqunw+8YHFtZR3XZSsN307Yvj83jYP/I/YDGcBzecUTpciwCXt46r7CHTkZ0
1OjYnM0ULK4SsWlMiQR59ZyrExE02I7MW55uTgZt6Klk3DABt51d/5L0adiz
POFUWhARYb0tP0UtYmBroyqrOUhDTdG3V7UuBPxMoV1ZLUXDndrlGwuLI2HT
ZnTZegsNypr9d0XKabg82mq/spAM5gVDKY9z0VhftX49OZWKjiPfb1sa0LBe
p2z/cz0qzm9qeltCJ0C4W2VpBDMWRntJzV0DdDz4JdCXL81//1GO74spGqK/
vxkkPSfgkiNRNfY9Fc2/11zoGqFikmbRZscj4WuN2w2fRhIY14xNtq4nY4Xs
/t0HQqKxb6LyvkwtET+OT5D8+cencjIVijwoUD13bo6DJAG73A41fZpDhUp3
QeRkShzaF/SFNulSEX2xNsdDi4yf3QmaO2k01HfVVesQGVj/LEqe1cdEc8GR
tex5bIj4TBK2b2KDm95vPhPPhDKvrIoZysCPs8Faa6o46GkoLFcx48DqoPDi
/W856P+CzU9bGPjdI727lRgPJ4u1Pa+P8xDVyt6kEcVD2I2dJttEechRJHy8
UcfE2o78671lHGyZ2z8x8JuNT+qrFYf/cOH0RdHK5QgL2nFaiFVlonNtwGaJ
e0xEqY23v//LxOm0OzuU3ZkQiAj9c72biv7uz85fHWnQEFAo+DiPggMLLtMt
HOioXk9OqD8QDcWj3VOpe8OxInDeMS/POPjvO9mUcZqG155b1qQq8utB8+ae
gXw6uEUbphJCmAikHoyekOOg7m8YvYnIhaWHlMuDpzwc0ClZvsUwGTtTPzWM
BKSgwWbTMhWjNCzeeaNv5kIG2stOWC0hZYLw/DE7rjILHiGVy59ezkbNzNY/
jnsycbmh+zV7Mht9qstYcz+n4t9TafP0uDQYLpFZpeaShiGlA4d/toRhUPRt
/dyUGHjsEvw0LBuH56ZXjy2toaJVILNEkx6L5vRnsJgTB9vV9jWL+PvZgN62
Sx7HqLCS6JfYtIiKCU7UaR1pIuI/WCtxfsahZbfWrfRCEoYE5/Y7pcbBOPTf
mX18byvvS5+WfhEBAW9lO4ctVAT9+FXFDYnBRg91xS08KtQZCaP+/LqVkVxf
dXJ9HEw5h5psxQmYqHAxW1ZEgoAc7dih3WS0HdoTynwQib6H65XeiNPgLkz/
4nWdBCPOVFYLf79cG95Pys+jgrI5jGj7jIC7wptiVghHY+fkE9MBI/5+1VnF
8rsWhxu0PROVt6n4N9f8FGVeBHaIXve/GUtFfeGeBX8W0rD4HrHZ5QkJGwrF
j/tkR+HXW3ptKf9zZ4ZVyV5Lp+LtN6Xdy69QUHmykvz8Rxz+KL9Jf7U7BjE+
j/tt1fg5sXy83ESCBLb3qvG6kzTkFytE0gfjkHp6y7DgESrWH9z+Y81JKo4l
plhsyyDCmfWPu2aKiMvF26ubVfi5OHn/d9/CKEQqLDQ4/5aA1qIf209rkbBP
rrJa+CYZfzfl9cprx8FDQ/+D0zsyVEL6XNPOxcKK6f9G15yCUpHtwu9ESSgK
rnxxjr//ZUx/XeDVQMcbH0/xkUl+3/cY8L61gAVEqPXmSbPQMTASc4bEgG2+
j1GTPR3GMkUbF7uy8aZP2cBXhg0rovdNyYdsbBWzCAlupuP6wwePrAy4ePfm
rMLiT1wku9Q8OaQaj/0i5xa+zeTCfFy/mVjOwOjJYr3Wa2xkjiS1rH/Fwrtd
5ZfbszjI2GeULiLCRG6WeT5XiAEZQ6XlFdcYyJs+E7tjhoEOE/O27jAGbGw3
faWuosLRVHBzOosKiedZch78OSZy4b0Bv//1xxu+k4TWKCRdd0p2WxeGwAS7
P0fiY7E7zi7oXQYVFoTBJc9uUhFaVK17qpmG/Q6jlis8GYg49oLg/p4F48uh
JWpbOGA/S6WalXJx/VXM93T1RHyNSy0fG01E/OYen8S4FMgJ7de8k5yG1XNG
9L1Z6ZB8ujd6HikDXQblnOcKmdA17l8V1JaGd5a0zBvJmei7T1cOqUtCJSX+
5r3OFKyobbpi558y627W3ay7/8BdzT7Hw0EqYeCdMNBb9CsaZXeHQjXvxcL8
gfbMARMqkko3/WVOxkDtd4EY0T8WOkuDpZ70E6BNPiu6LoOCuLR+z6xtFHy1
sdpJ1yVgh0pX4oBSHBwGAp+cHSLC5HoDln+Jxcq5zIp1CpH4eHinR5R0BLws
lsSYOVJQdT6gu0whBta6PIUlKlQIWjTusJ4Ti13Gb6x9+PVzeK3qsz934zC2
+NWRhndEqHubFFufJOH1Yzf9R6cisb9skXEsf84l+niU3ycRQZ1Jqu3Xi0PG
A5Z52Q4q2lxbdjrw4uA12j8tUBuJKwc1vdjFVPTKhQQmBcViZ6dDQkkNBT3r
Bu3kjMNwSpypmsPPj5tKPig7OFJxvGRFyZArESTCEtG2lVE4ldMlssKGgiZx
Y3HyZirCGrsOq50nY8lnRdbuo/x+I3o9UmgoCr3kD1dLQ2gwSLEULuTnLbn6
iudRRVQkFzQbvJKKw3KDihVyPhTwkh1RHkPBEXP39sR6Am7lL+w6oEHEQ/bi
qtZjRHwfOX/v40wktqlsTWr7GwejM6/GDzoRwe14XqXoSULogE1PvVssToz8
FKN1knBXnDydtT0GnK5lB3ftIuPwgbr8s9MEPCZ1ddTz19PngfXFm8vpUL78
4XyAHAMb9nhqtSgwofAzPSdIhQkd63hX5RQ6Ap6Hb35mTcPy5Q/+xOqzsMQj
Kou0lIX3h9rtFp9koUv/OPPmGA1vn08qrJLioMacPSQUz0GaQOivsu8c6Jyh
fr5zn4MLOmunWtvouCzcs555kAXBBLKJajkTJ03CwqrC2FB7EmBQOEgH7fh5
8038OdxqSvLh47t0iL6tdRHj57ba9v5f3eV0rG93P/mjlYxXrwYcxD9QoLfA
biZHjwj3swtOKvNzgZxM3avhW1Fortk/5oVnmF9qW7PxAz8X76Ov+bGSiogn
G5UFSRSc31Wrt3cuDT2U6Fdf/ekoljQNJ1YzEds10PCB7y/r/Ot1Xds4YAoK
SMU48+Ce3F3xMpeHCD31v1tkkxD+XCjQTD8VW0OMNvPE0zAw57fS0oE0fG8d
7VtpmQ7RVb4q2V9TELTGgfmtNR3zL/1M+hTMQ4G7qW6schJy/539tKc7EZva
PDTWRD9DmkxjTLBRNJY4Lb+/4VMMBmRfipmkU7BjpLE1gp+LXhiHLYldFIuN
919W16wjoE5Jbk1EPRnGMQ7ynsZksJ5fpP5xj8PiQ3HEj/ycaCIcs+n0WiIy
bUa+E3bGYjlBc1tgUwTOTIwbdviF4/7zzDmZD8iQm9O50KcyGil+rdLz71BQ
7bykaZFDDH/dBtKL+Psv8V6bo9qPWPS4eu40XkmEwabO1wJ+RJTzrq7sko+E
l08U9xHfScV+6bQLrQToy0Z6LgyOhev32nzKTQrMRBOd3zXFYg0pV/seNQIF
WzrfuWlSkUpYaRGbE4Ml3sx6zQIyugxHnTKfheLWns2KpcoUMEIzpqixFPhb
6Fj33yAgIjWn1/thJBYMR7nHu5ORqh313OgoBVID97bHXSKBsOXk8oC8WEgq
+G8Q5UWiRWrHs8cTVORtpO0P1iCgdLpKq207FX150x5qZ2IR4PfcZAn/OgpO
Uj7K0WTIpz6revknDrft9c5vvUtAoGr0VMB9AlSTZOtdzCOhcede+8i2OIhr
0ua+oBFQ/VbE1jOC32cMjF1iimMwrjT099wAEX6SD2fuSETjpZShiaEaCRWD
/bcE+POCxWHhH3v4edBjouNX+CUaTH/YZPifp+Pc4qzrZ6wYuCecupZjyQAj
WHGc0kkDmSX5KPgGFZ7rVuQ77WDC5mTP7n8yTDSmFOwNOshE4lXPkcf8PM4I
6KG7C7BRoHfi4mQAG16tnXveNbJh+3OXy3X+/nb87dMNOUJ0yDSJ+8drMLHO
8Nz49WYGPHT6jpx9zMLTN8l/bbtoCCZFXdMUpWH+1Rf+amQaGs2W3erVpcNx
idN1bUE6VKgr56pRSZgonViTQyNjm/ieEIMlBPglxruvIlKx9Iz+taOro7Cu
agOJxghF0B2BzoSdMXA6nLMsbSkFV65xXGKvkbGzu2JZ/AEqPotYNyqzaEjl
nfs3UM3AxNRDzc8vmPhc23umq46FE/v/0XXfcrGUsOvq9lYu5C/cVD4rlgCW
RoiMk2oy3JJ/K1NupEDLoOKWv2sq1mlZJeT2pMKINLfz98dEsBo9H2YcTcPN
8uGlV3o4ILQ0t/vOS8A5AXeZQ5m8WXez7mbd/Qfujv7eq5Qk/Aw29MnOEG4U
dBsXheTYxGCLx7qFZ1vI0P1KPLKoLBoz9RsCOkgxqC6eWFHhEIcYMXk5tQl+
7hm953HrNgk+xRvOSfPr7qvyGuHH5TGYk18vEWZNwEuqQ9nCoBiUHReLlveL
gGL7rTuRY2GYr6Sb35tGghaV7JZ6IRoXbeY4DJHIeLhopWt6SzRueCrJagnF
oPR5nOVZvuP4hdlxT04SIPC+786GHAJsBMIXksciEHjk8tlFOhRcflZW9WU1
gf98mtSetzEIdutXO/iIjI2/tAr2L4hFx5Drp5iAcFw8mCTV7EmBj4Wn+9Bk
NDLu9rUWFZPwcBtlZ/GjJzgtpo7Ew2QkP/c0f3+HjICb2X2y9+JwaNk2z+Y5
kZjppitpM0g4pN1d++0cGX4dOx9Z+RPxe21Hn83WWGxdYSitfT8CQ34/JB87
8Z0dUnrEPhuHhpuLHvYfo+CCYPnlM2kxWNd2N7PwHgn393x/y2bx3ao8Ns7b
we9bl/Oaj9fz5+TmdufLZXFQ3n07QCE3Ak0pr6p2XeLnlVyb8rjhOBwjifmE
5BNA26arMbM0BiO/DX90/SbgKH1t9luRKExG/7OsPkBEu7HfaCD/ur2VJXUc
ft0tOzixvbSBio+5UxZ30mloG2l1MSDTIedyO783kg7dC8IWNWtoqJ/zxfPH
fgrG025U7NZmQMjqJ++gAQOvfTdITBkycHvFs4CzHlQ8ejPoOi3KwuIkuc/w
ZiFGLsF8KT+/pabUqx++zcLVgpk58/l5+kuB8NKL4OdGK0EVl2n+fY7++n6O
zIT91O3TrR+psDxCvzifnzMGnc3sD72hQnOV2K3X/jTslFi2PdSCBsfXJZcW
PiLioFS92ZdbJDiMh64/OhaLi7du2yiuoEKDqvpM7XkkHu2jbJPRD8UMc/ud
b2HR8H5p2hYuTMau7RvD5WxICCZ8+uktR0FGzoLqZx38fO0vETvST0eu11Ti
2EsG6G5l0vERTCT7UgJW+3LQxfJu6VfnIC/eLzruYDympQ0PC6okQrOgcGhK
IwkeZpYD/x4k41Jtkv8TvRQcznbYXXONh1HFW2qbRFPxOcDJcROZhbvnt624
qxYPlW6nKZ8PXDg3zd911i8UOQ/Wax9aFMWvm5cV2zqiob1iDjFjLhnmJv/c
1dWj8e7m6bxx1Rj+fqD9NyQ3FseDt+j83EzCkinRpqgkIr4xr03/E4hF15jA
8VzpGCRPhxcviY+DmNpSStQ7viNdjWNZahGwki/xinUJQxRxt1vXMBE5Gq+r
Novw99nC6Li6VyRMyqp9uMvfdxMeRy7Vso+Gvn59/Oh4DIKut/9JI8ehJjYk
qmwyDkftUtRIhRH8vFGy558TP8dUSLUtvxiH/dORxxdqxkBFguK1PJmEcs1x
95XGMRD4vjaN4B4Gv58doY7hZJj6j532PRQNM6fO36eaiWCt8s3vIwfBKUV/
7OhNEq70yNk4XCHBNvzOxkJaLH4Lsn/d5XtyqVtgsO4tEXG8B6Gn/ElwDn//
tyqBAKPSeX1prBhUTXU/JV0Lx7G5UqHnXvD7AVHtmEFsLISu2L/tPEVGWpEr
a9eyGNRzbF9Z8/MoSzov92k5EaeMT9786hWLo+J3JG/w8/Gq7Zuk9MTiINPK
JphvjYDDfMl3QZkxuPrYaTNpdxyIphq31n6OQ16hu3rqjWhkp7ZYu68nwMNz
fYmUZCSmGkG0O0+A88Ua672usdAeNc+K8iJBZsvdtxr7qZjYWapuu5iGJP0T
sibfaVCqSb4bM0CDjZh0b8BVKoZduzdWvCAh8puAbJQVHVduPU7+4EfHsl9V
dwud6Ki5YlH5MpuCn0fODM9RZAIPHy9eG8LE6OWRPcG9THh7RVra8uvXz3TR
TwcaFUYn/Ml6tnRMo7o3eg8dbP2LrXbPGdi3V+9w5SIq/mytyEttImNByd5v
c9ZRkVVQNd+nhYqVflohzTlUhG+hrtUN5c9Z82cyKm2IWN8vdM55JAZKvAdX
ejQo0OpcNGZ/KxIr2zvezFsViuRbtxMPTUVh7vY/oVaLSMhnFlr6nCPCrKp5
ocAACYWKetvt1/8fe+f9TvX///EU2lYaVoWMKITKqnuINIiKjKhESDRISsle
GWdPexclykoULRVKkRYSDVSUFYrv8/NXfH95/3Surss553Wer8f9cb/djutS
Otzq6ZeUxTLxpJy2b+RLJgwToxsjgrPATPnJ37orF3NXZlXde5iDF+6xdUK3
8/DKSX+J3JmrSJD78/m1RQF+Ll2U3nmhEEtTPnr0rrmGYSXXg7aH8nE1Y1PN
np5rYJVHLJIvzkJ+Y034DuK/rQriswOO5P2Xu/9y91/u/h9yp2RKUz82i4I+
y08bp4OYmDPmZNvsyEZIf46Zm34KvmQaa265RvizSHo59SkbfnOvCB5exMWv
tvi/v92TIak4q/rPDz6O5a+NKbPn4PjXbctCwth4sXhbxdQ/LvY2uLacI/Nd
XCAsPPmFjhh+utrzYSosL71NeaKSDCsJ+qLq6yxwIpY3v56dgrQ5jbplT1iY
CKcMSjSw8OHDtNqeEA4c7wj9GxjgospS54ygCQ+1vlrh/pEMBK6VievlpUBi
VVPRjzou/L492huUwUZLf9/kt/ZkcL7eDNnIY2PY305KLZCKazouG3aUp8B7
cmiQmsfCgejlFs5jfMQ6K4YFzYqE3Q+2tURBMtSNP2Uank6GnKlSt/ALDr5I
eGk5ijCguLn0TIt8MpovtPoM3EzG2L/rK30+8vB6+9MJpgoHi9dXxtafomEL
LWinEvHfz4GDIt86OaDNmowYJDle6cV8NjOcDcY/U1+3R3xsHvbqjfrOh0T+
8BPJlxycLUjudGNwkcV0DR06yAU9nRZvW0CHsnDSl4Y5HGiOvPnVSOdig7y2
+lx1HuJ/qItd/sAi/KoZ+NaGhwH7/oOFGgy471hvKM3kYVaYhtm3xxyk3dq5
zCA1GfVuwukDAWmgd8nkxgWm45K6z9d4mwzsCFZaUrgpA+cpmQJnc9PwPvIc
o8kvGRPV9lKFcRmwCsx5srgpA/024b7PuRlYd8RMjke8NKSYVxxL/PBKTUzc
6ZIsPC91vKEnnI16+uE91Q+yEDHyUlt3djo8je9tNGNkYP71w9miMRno3uLo
FD+ViZsGXRHvYsg+mpkkvMEjBfekR/yWfk9FbvTpub/XpUNPdWGD/ap0nOow
HFQu4kHvbYzynQNkn1gsmBM8xIZu6wqljTtS8bf27Z3/fV/uvKL4ndJ8CvBb
9eMOXxZaJALK5JWS8edFXcrEeT6azNgpURXJkM19GNH+LRWyJUoOZdYZEBad
nNUvTfLHCm5dGpkJkSq57h/6OdBQPkuLC8tGmYHGznrXXDSO/uUn/c7Hs+g2
O9OUKwicFtt0rPIqtgfc4Je7FMDpw3u+U2Yu3pQMBZ6aLsDG6cdp04KZuPo9
SXb3/lzMlPd4fXJxLko2NVL1pJOQcM3QTvQXA3tuzl05q5uFbr/aFxnnk7G3
RMozSoOFd61lhfYebEibiT8PDeCgPX/uiYOFfDiPWAo92cjHVM31kyfK2Agq
otaJDLKwR2fnb54TF+XH6K9FbrFwuVr646MUOk74+wrqR1BR88fwYfdJPk4G
9ldstWehn/7XxGhnMrJCWk7es2IhTTIzScWAhYXlOlMyYhxse3V9E307yTvt
9xJlJhemq2rbki0ZcBAYu9H3OhlXao0cclZxYZ2mdXWtBBuO0cPXbi5LxoLm
TU2U3yzo3eJ6bQin4M412ykz0rOo0th6cRELAf2Z1ntW8/HmcIWzVfMlOB1y
QuQAH+Ui8zSqovhYpbn+NmU+B/1m9pXB6XRMfV++VMqPjz+aP23tBvmQgm/3
PHkevvS2GZnfYKO0++7kjGgqdGhtjU+6CQ806PzQ0+aggL4up4iSjC2cOhXj
vyxsW+GzzP8vD8P/mCvtFPi47/tDcEqLg60CiqrrhbhwYE1euljDwcUzjWkl
G+ig+XXcXEDOn3ZQ/vv2PxxskjyVZO/Hxf40gVHKbhYsr4S0NCRxEbyev8bV
kQ7JkwwRo1YuOl4PRDarcdBms7hk3Qs+CvvKj9NDUiGypDZR5m0a/p04JnWi
Oh13i7dK7MlJh6CF1mRCbCoGwp+12G7h4/38s04Kz9IxpWbUeWZ1Bk7OO1Vy
+306agOKvuTRUrC9sz5bLiUTegVHeDe+ZcJ6/YGmuM1ZyBityu4VysJyub2f
IoZTcflS0fORN+lQU3T9c+Ezef7y3+VJ2zKxu3yjjtSiFBhHnNy6VSoZYR87
d4+WpaBNUKFt1sY0CH/4Pe1JHqvMzYLDu7no0qWUFB/nYZNCveMjETbkPvnN
HT6egiHx7YcbPjCw0ubMhb8LKRBpql9Z18fE54FxA4sdfHhsm169I4cHjsS8
It0cPjijTjMyrpFeedGevJH08Y2oP4xY1wxEfZYbkM7OQFXgALvCNBuHGZNP
xY9kYdDUcs1b3RyIj+wZf56Sh6rKZXmPlfLRN0fwLlvjCkTOJzz8+uwKvBZd
rOtWysEsjWTjMburcDFuKz0Unw67rD+WRuty8O3gV4OlQ9n/5e6/3P2Xu/+H
3CWvpt8qNExEYjI/8vNxBt6ZiDasP8VC+uZr0iUP+AigWG/wLmPCp2k9R4B4
2M/yXkXLLjZ2tz8RDZnkQbYo3XZXHA/m3752npAlfNq3diLXncxz+c08vVoO
uqYDkmuUWTj1+QX9px0dtsoXlAZlqLi+/0Hh2Toe0hZpgyvIgumWv/YWbD6e
/BETmf+OiYL8OdMD15kIe3v6Wkc2G09z/6SOX+VAtUlhfsAgB06eKQL+ixm4
qhAupCebjN2f5NeEJ3LwUG40cttlFnKNzk7HOvNxOtPfW28fC/M6lGo1tZOg
F9cosVMhGVVublda4ph430Er/eFBfCk2sXS/XRjeKT65dhJ8KNzb2HiC+KlD
r5je0r1scMfOpqzQp+PNvK1mz5t4eOY3cdpRjw/RGWnNUce5cL39erY0CPd8
2eOkUUwBM3T80nfhFKRatjxaksRG0QOre6/u8lG9N+xIUTALSQ+m5Z0287DZ
MvnHd1ce8YeK1TlcNloPv+6aF8zB5V/nVwsrc9DZsvKCRAMN5pcUZU82sTC4
uHDC3J2DyXBet9pDDt7Lznza/ZSJooqbF83ecOC5ocBQh0nDmRk5GjsVuMQj
Ntul89loODNSESDOh2rUs40b+Cm4Xvh58lhLKu5YHvRN00nHI8Gor/OXpUPk
S+l0DfHWWItHhw+s5sH6SpjALOl03NLIK5KmpWNfm/KozxriXWn1zqbZyZCe
OuRT8DOD+MYlv+samZicszDsQBTxpJKVd17bZkKgYuZl3q8U2HocOrWLPO/n
E8frg7vTIazFvRuUmoGiAr3w8xV8XHKV7jb4zMODy1WHDnOToTvzhBd1RSru
Het9fEQ/FWureNp0NS7SK+RLG+O5SCz+0Vqmz8Li6IW8t/xk5OruOTCbyUBu
Zmpc+lIKjiXPtss5wcTovdLTbhd4+CvC6Pbr5GJamjGcSfI3EBBA/RmbjNlR
+3e7TKfhp6s2X7YqHcqXDv1cTPYDN8vmnYt7Forfrkm6fjgTgbmfDq4nHrtl
adapV765cN3VpPH3TS6SDhS+6m/OQ/dazoLtXvnIeXZMQb8jCwufnkrq6MzH
RcngsbTzqVA7P5hdrJGNu9dCf8wazsITDdtkiWMJePdE7+W/n3T8UDRx7Sc5
6MlVPfFjCR8+Y4eM3LYyIT9uqmdRSPJxd3egJplD85kuI9f28NA5feToux4u
XqbdrRKLY6FuIiUp7RMTa1ZzK400ONgXX+4ikM6E7u6q3l0SdLx1jgtTqKIg
yzGivEeGh8i1/iasm0zYhnBU7ft56AhvZiz2Jv92P1AwsJoJNyl7qsFmNnZ5
r5pBXcQBa5lXve1eDqTl1neYtNPRwP9VJHOMj9pgT1rXNBvnl27/KrSABTm/
iUBeAQ/594sCXWqYuC84Pujkl4DhQ8vNvDz42MCPEbRdwMT6P8+3LSjh4oOB
G2N2YSTK+oR9a7k87Jg3Kin+loszFvEZl3JYMKw3UaW20aA2UXzgnw553XZp
u59k35w89e9RL9kvwu3jLZdesXBy41hOVHcSpl/7D+/bkAy3GSkfwkdYEDYQ
I7LEQy43SMlgLguU21t2LLjMRaWC8myna1zEJGoNKc5n43XH/inVKeJ/K8/m
PWGy8S38MnPzMRr06BtN129m4eOQw12xd2yEqmq0V6zgYO9NSfd7lkzsctCy
syO9Vj/sY2bRQYVNdNeNv+c5eHFqLN9+GRszhUcrdx3iYeS0BG32w2T4Dp7v
GP6YQny+6tOYO+m9Hu78tavTEKXyzno+k3i7/vbHoepc3Ny54rFlfyrOhkyc
8RBOR1qgoouDZBo0lgp1uj/ho1tEvNffMgP8IdPNvLAMzIwVYXS8z4B7HAVP
SjJwqpq+t2ZmCg4Lv4xQkErD7T0OqSmX0zBT9OL8IoEMvHGrONJ4huy98dyf
c+u5WBCtbcWn8OGgRbmoLpOCqJyyWuouwg2q36N/hHJwkDtqdLaKg3fqVcvG
TjPBS3PKe9nKh2PU2bURuxn4TZ0O36xAwcM2yaVifxmoDlCTM7zNxS790m11
KlxU/Wirqykn3rzu83GpID78Hd+OfyH752j2/jZ3hXQYFo+MaS1Px3utLd98
+ZlIqpJx+HAuA3/+fbhpZ5CFr69Ccw4czIGhR19PTlEOlm3WfOLOz0UGtVr7
gEYeLm22csypzUTsF2rBLWoexDLvPExYmYxhs60pdtuzcEeofk2HeBba1efs
98yOh1WG6YWuADpUlt8xuZnCBJutbnbnJNkT5bOjRtoYsNzcLD+wlwXtrbGv
s56xcNW3K3JPMRft7imc9+ZcQH+D9J0pJuJeT/ccOsrESJuVv0wmG775hUUH
VjAhbed9895zGq7nixwXP0yBxl6ROu4FLsZXrHaUcmdCZnH87jfmPNj2NJQd
F2Ai4nj7w9KrDHBCov3fvmeh6dfmna1hbKh8X+mzoZoN017tQeMcOjbt8Z/x
+y4PX49q2qwJZAOzHwrfoZMeM3RbrTaTB/58WcmdWkw4vOk6p/fsMu5ydl94
VsqDaubprx9ZDDz+EWZfKMzFJ4Pfn560RGNho6tj5D8ueJTFzRckuZiRNj39
QoCF+XlNFluCaLggLVz0OIOLKQn2oVVfyf5sWffEYSUH7B4/ufHjLNgsN1n8
i5GIRzc6QwYIR3v5+/zKcGMh8l/esTHSbyeGGZtGeUzs6b3sw+3iYMjah/1s
BheHRldqFUUQzmCvt+yNYKOszLxCaiEbBxZ+1ZYXo6FEuoVzvZQJkeTUR0J2
bKx+YnbImpxH9ebtGefJ/RGzvue3NpKN7bOMFAd0qcj+pSat2cFGiVivvmUa
C12lbbbrq7gIFzJ6dfYvH4/1OtWG/yZDRHjUQOpMKvg5K8xrdFKx+/vbrf3l
fGwMzjzbbcoBR8t8hmVlCjb33LcUmJeGauewt5NfUpDDf7g2cZoHFfMlptPl
6Ri5c7z966d0yBUUn4relIE7Alnq22Uy0HCxqydUKZnsx6xPqYMpmL/iLifo
cio+VK+vf3wiHRR5tZADFlwIzvZ6sOQuBw61li1WPB70ck8q1qskI1023nvN
iWQ0nT/ufOErG3kb6ru2j7EhyzPKd6hkIPLLEWqfFB8j6yzC5i1goEPY3Oux
GgXlcj5PjJIYWDQxKlcxk4ujob4XKy9wEGS4ebL6LQeaRvOXvAnmYdklxTr9
eynwSfEsnLc2DUbP6u4ofksleX0iKd2dAf22U5NDeemQ3lch3XIkEzKzGlNC
jmXDqHCVnFx2NlbMDjJojMjBTOleqUTJXJzWu6XMqyc8vvwTr8s9F7wjl1KW
TnPR26mtaHYhEwd8S1BH+NpkTL3gQe9lZLedtFMSpKPpQ7C3qiETiQuTdzOb
uRAd/Xyr6CQDv77sdKshucq7NiWpvYOFv6dULn5ewkX8gpf9j4o4qH72cWn/
OSbeiF2udB1kwNAv/ood4c4DKw3fH84mzzcaF1qTRMOetGodMcLf8/2sti/q
4eCfuqjATCkm7GSy5EyvkOszPFiQn8zAvrSTn99rMRCs43HM5yIL+4zx1OoP
Cw9G21pYa9n4uEWiO9mHjlAb5loVOR50mYomHaMs2KbNtX4hz0TGGo9s6iEu
nop/HXK8wsAq6bE5OzbFYbWLZt/NBTy01ilYXVjNwPEw75m9bhwwfy1SlN4T
i5V+TmFfvMg+aFBgFjtykLXvoIGmBxOnzCSUrivScNckVfH4Ui6Ks5erqO3i
Yia/7+niaDb4biPyS+ezsD/k/O2UcwnY8sF56DHJt0jci3PX3jMRPnDxj0wC
F9J7hlZK65CeVT+zUtOYA9EjtvtDD3CQuNeqjyZE8mEdsbRSnA3xVtH7joSX
56fXjNy/Q4XL8IDZFl0mFqTayzPfsLD3jFe1zXfi3Yv4Euc9GNhbamsm9o0F
FetZFjQOBW8t5qj17CC5G/99XFKThXuLsky+rOTisc4n1bXE26o3mv/4qJyM
Fw9sB8eIpwU/1/k9bJkCi7XW/nqEO2YsEo0QOsmG9eKBmGTCcVEl/PZgqVT8
e3A6BU3JEL0ro5ezlYfo74Yc4bXp+KDqbLLVKh0uouHhwlfTkaK3//TixHQ4
187xf7efj4li63Dqh2QUeDdp6KSkgLpj9hvN52kQnyVyuW0tBzaJ5neU69m4
8zxA/cIt4qlf5ybkmPKRu6bq9NU0Ph64xdafciC8r/gmeiHheSfaoRWt8xng
rzgWZ0E8QeNck+GJp3R07flW26JJwauyQ/WXVBlwtxqP/baPg0BVJ2+Rdjb2
49lfiXkcPLKK6dWmcxH0qEPid00ynP2mjJvUU/F+V5p+RFEKPm1UrxQzyMBU
8/OeX81psDotFyqXmYGXoUd1bOKzYLq7waunOAsiOv7n9GKzsdi4u65kSQ5O
aOaeWPkjHUerpReo789Btf02uYyFHPDkh/qy72fgiGEr/T0147/c/Ze7/3L3
/5A7z/cR8vv0L8NqWO+wJ4uG4jFFgcEPDAjM+P5qUo+LZScFT2uIMHA76GBP
WCETmGm226eJCTGlLYb9YRwk6RpiJfGMrBj2tOIkA3PX7dimFsLAW/M14heS
WShR7Qr73++TCrZuy/WzpoGGeQ1CF5MgsF6q5a0NB6+82i6se87AkTjzGSsk
uJCLV6n3NmKAq13YJniHjraplBdFCiyEf5OroPix0Df+eFggmwXdM7U4s4GO
1PdaHutDuSiVtfC4FMTCweRHLZWlDHy0jDxv8Ij45fgk5S85/0P/JLZseBED
qXeRXYHeXGzZ7ffW4j4dfTY3DEafsSG5duuNNe5xeCz6a7VKBweL5ZdseHqV
jU8u+bPaWhg4F/6Wd/wFFYUP1FibGRxoKQmmbq3koEPFTeU08bd3ww8cAq8x
kTLFyHpJi8fY0afqQoJkLvPd88cdmOg6WH31SzcHF8+fDP7zgoGZx4sExa+w
0aZevvvVXTa6glo1NROYaKOGWi8hn0/RjOUqIMZCsnjmd8sTVBw/4T3pTLiq
oeupvdFhwt0fDZ3WOpP9NyG2feM/Op465R6w30M89HHjWNEcCv73v64a3mPh
jvo2reBHTNwy4i4zpnDw0nze44lEHlpU1PWOOvMRduzuIQUyd9t8XUoNA5Kx
9dayc7JreXieigv3rrHQrZ9RqkZ8a3317cHd+inI0L59aPIxH2+trq7sZXGx
t3JlyDe7NLjd6OtYEp0G4QZVLYOV6Xh1ex+k56XD++5Tn4UcHh60+rDa3vFx
dXKnwSvyfisPLBne+CIVUyMD54xXsbHU9ZylVRcL8ob7LI61c6AQ3tq7yYeH
pTZbCks/8KDo+EQYr1mQqM/NLaKywOlSDAv2oiOq0fyKejUX4uGDhyhxdDhU
vPH6rkWB0o/D7qaP6VANOVI7UsRGws5Hrmnb2BgN6eh1JY+ht99vCCjjwCV9
w+DM53wYKQiKHtucgp9iaos3JCVDIPTSTo/UdIQGJct880+FjG6t9Y7f6Thi
ZG2YTvytXr3yjUtrJljldUsqsrPQ/brS+4pWNp4drY3xU0uHW8mmtc6Hs7H+
6YWgWeYshGQZTbnLZmDYxfXycpLLwFqvIDtaHEwSGjl71tLwcHaQSEUYA2N1
VobBuRzQE8qWLb1Bx0TMUNNCZyaOCqqxfu9n4q9m8O6hYTa+n1vxMoDBRsMJ
gbI/oQx8EIzOsVjMwK7GsB5FkpfQBzHeXdV06K5kc4MX0XBo9bjc7d7/fX96
kxHykI0FL18/jItm4PnWh6v+hXBQKiRgefgjHUbpCbz3O+no/7fD/9AzJiqU
ZgrqDTPxQ3eD38KVLOhE9EX2TNOgncvQyuon81wztT2L9IVD7J5VU1YMhNWY
SaZoc/Bmh8qja9fpiFvYqO+eGI0dkpoaYc0cFGxRPBl2hI5bQeHhcobEi351
2pzVugwXz31+qQ7k9WA4PT6LDXOPHn/pHQws5Rb4yYRSEV7c/zxnKQdhE1Zf
Gsie/v3X91brMRY0ZkkcebSXifnS1adGOi6jrWyu72dH4i3rG1R/dDBwUHeG
cq4JB4235lVP+jEwGihRylpG+mj2Gnk1NTZOni87NCLNxFNpbSQYsnBFX8SG
Qjzw33WpXQIqVFQuuy0eu5WBGk7XtcgBJmZufF5xuIV47EXhCH0OHdSsmqqT
90kfvtCLzJybhCXfPgSd2MRCWLDnZJwbE0qjkZRdczhoPf7h/lvipa8522Y6
pvGQ2Hb7ud0IH6Nod9UpID24zl9tFvHu1ff8NupMMiEw/eneGiYP00sOJt33
SEa7VHOt9UseTvk1Tn4Y5mDGLbW1191SYez9YHZoeOr/vnd7sXtzGt58Gjj/
UjYN+yxp/hk9XJRkz4rg9PKQpVLzuvgnH5MBU4e/fkzBqjV5UZ9WszB2PSy9
ew4Lg9VxC/mLOOhx+py5KYsLlacafxSUCM80qh6KP8DCye6ee5t+MmFm8u1c
RDMNzL8KWV+lCZ+sLD6wZxeZU1/RE2IaFGRYHGMJetORedHGVX4RG+91Tov5
VrFgvb2n7iKFhR//xpaqEF8cpJRV5f3koSw/8hfLNRlq65p1fCL4cJQvnxUm
ko5c6Yn5RbopcFeR+jpsk479S5rPv5qdicQGLSX+/EwkNMV8lm7JRM3egCsy
B7Pg+aG4/fR4Kmx2L6ctCM1CprFneyThvZzOC/dFI9KxxUBx0dL9hAfKjkTI
Dsdiqdf5H/R6Kj5H9p3asIaBytvbW29JkT577kxdakdH/tCnFbpiTBy7MKPJ
4wsDd5t3bWAfZ+NI/f0/R0TZkJPpcxYivTiZy3dzKKKD1ipfMED6MTTR+cw1
Gzo09lk+X/+GiqUJI/OMDyZi8pS3fvZmNg5bS5/4bczAhT1u2wNH2OgvmONo
FENHQOWdQImPNBzrMW/depYJP8HHm/sIx072hp60z2DioZT2i39PaGjU6x8W
IJyW0MlmfWUwISzH89f8Qcd6ueOdm3PYOKH88Pm4Ph2MY/MPaUdGITbnTJv5
Zg48CxL1LsylY65BuUJVMQtHvv88Z11/GWkXe6ml79m4HxoZEuvOwvbZw9ou
j+hYODxH5eh6KvR+sf4uy2Djamtp6lrSVzPjpnaVf2FiABN2t/6RvLfnHFrj
cBlBol7bq0gf1kYv9Jz2YuDR6MuLVflszEoI3pQpw0D3s/w1q5NYiMrLrvzH
Y6E+Vuib+3UGZkQfiy1rY8J/TDCbpcbEywf7BvW6KZBfWNxv1Ur20Xc5jnA4
EzfLH1Wes2HC6f65Wkk9Os6dd0x6ZcAEU3uj0kypRFxS6HB7XstEJmqHj8xh
4kwEtVcjho2LP+xMurdz8XtsQ2xZPxcBX8rrgzby0brDcuVVwpeui04lJr0m
/DLrDrfcjom4Y53yi/73d5jv7bQ9ksUHZdf8N2nkeaeiDBYZOXNwZk1c2u3Q
FBi+iC0WoRBOreyI7N+ZCu9dZ58d1U1F7Us+f48RF1keIRz92SR37suHD6rz
Yco8714xnQyR4IJlmzYx0WCkVudHHvNzIh7W2rIRFDZ8Moh4iPRCzez+s1zk
0E50rutn4nnnreaPZL9ZtFOt1U1p+JO2acGnSxzMObRJNlWU7NesgYJLKhSy
x1foWC+iI0Rt17da4imL3p48ydvIwkfOw5/7e5i4lnnNa0KR8Ko/bb69LA8F
e+IcD7P40BTq738UT7g1tyNunkQatGJi0l8tTIbxJQOvzcFpGLEZ2/zgQAZ2
KpmuX7snA0luNvQ6yUzYi6lmW6VlYijHbUlKbQoafRcv8bqZCYcoEaV1qnSI
a+2nGP5Kw1PGkX1x/DRscD2hJewSi1fiVw92e1BRFYVfKm/J9W/7eH8LjY2U
uOh9n0mvKOfv0296yMCGZfU3z59joGrmK+NzvYT3uv79iyXz89NIfM1AKh1u
tyMcBazomKnZoBVrxER3pnL5g14a9tVrdISnUrFXySCo4kMCpMa/9vTeYSEn
8FOi1RQdmiV+pTon2BiQFZ9XrkOHzsVWE9dAGgzMboapKJO9++d44T5BJkxU
es/FqTBxzsJr+XMGDS0bWhq/N7OR2VzT/1uVCXXpC2L9SXTMeKZ774YcG/kV
h7hnn9IwU0FixbzaSNQ4FOrvLGbjxZ7ojym3aAiIef3AkPjPB7f7L+/qx4Nq
0O7z+QgbNnqB714Rnn6hyxNeup2OMPq3OTN/UJDZNxkpp8VG87UvaR9JTy3Y
9rKG78mEtF6V97+rDAxXnHm5cCIOgkK3J/sIf1MaZXeXTdBRdmJnC3sxGyH3
l2wKIvyTe34vfcsCFtijf468Emeh2NHFSmQbAw9rhJPWBzKh53xjx/W7DJy5
YPDUPYuC/bJ69G1kf5+p0Q90kWGCMzdEaz7h39qnLVKeH2joaYw3nlNB+PrM
o+fRmxPwR8TmZs0usq8G7NftK2ag/cl549VkP3o8L8vfeYsDUyODzwxjLq7O
7eznhvPgsUnGiLGBh7nGnZ9phhxscmOlvK1hwP43RSHkKQfNM8Zrevt4CH1h
o3GR+AD7Mf/9QBMbf241jqQWJqNyUV3h5oJk3LrNyXvqkYI/v1Kmox1SyJ4P
l5mbysHNgpwl+7S5mF3C9Enz42H4dTvPTj0ZstLSI4LOhI903LsrLzLA2ajq
XpXJwuGk0c+/NDlwuNE4pdLCAav670utUCbua9+78LaWgTtxl2KfVFPRHJA7
m9LPxs+7rg8PttDw12DdAroc8dscp/x7d2k4SQ14C8IHiVcDGC5VTFAzhr4J
mjCh4Tv7VKM3C+P3x79U23HxwNVvtcwbHrzu7Ny3MZcLIX1FGTrxWReXC5XH
/vEQmbroneHJVLwMsJHPq0nHoapfNPG8dDic7i0Kd8uAVOiHxhkDGejOiH+6
KT8ZAfmdobKjGeD0c2afekXFrL13Rp4NpiJo+PWgUFoqrOQFJ0Oex0Byu5TB
1Dwq2r8889kdS/bqS/35z8h9Cpw9lmR0jQbbo3Ux5eRcvqrIzfooxsAzrR+5
puS6xTx3qtElWcg943HkiC4dxq9EOT8GaRgOz0/NIB7DXreqvCuShoipT037
jlKhaP99q93hBKx+fM5iCizovyjNiyIe56mwfO+7HyxcTVpeQP1Ew9UdJRmx
ksQGT0zEt7Ux4BZwx0+dzsDCiyfT9pQwsEbroqPrIRr4Sup+sy3ZmCMq2OJH
7kfMwYmpNuJ9Jhk/bifwWTjfbrer/iCNcK3F8nrzSIx05HusIHnREi9Zc9iN
BjvZO+ekbzIRaNP7NyMhHlN7v2Xd+sVCQITlqa1gQiRzovkx4ZnTO+40FeZR
cMTpmonZAxau/QwV8ykhXPTuzRzJQQaYq56/OuLEgGbmy9dClXGo+1il9y2e
jXmL4hyyKXQ0de63vR3DQor+782rztLxZVaps2sSE7/V9eWNEpnQ+kZnRvcQ
7v2yVbhclgkhnx33zzsw8POEqe0CDwo8mf/CJYXp6DA7fG49yVeikQq/lLzf
z5YShw8RNJSEur7OMGDA4fwTBYVz8ehrcXErbif9+fro3zoXBvi/dt2gE9+e
sA656qLMQeSDBopqOgf948/evH7DxdYFVFvhy1xInvU9LES81nqiSsFam/D/
8jqDLX/Y2GXWY2ewiYfCuzOTr5hxMHXk3cTs7WwI6Yx/XtbDh5NYSUDNKz4e
Cgd3pVKSYdjwYfl4XDLoyrVfmgnfel+ap3TNi4OokF3iTo+4WKA3ac125WOj
xE3zZYRvhtPZupx6OvZGrdoYM85E6s7Zy29HsPFyVO39PH0ORMwr5l2XI/uo
hdZxYSMDPTcWTz/YQsWcjR+s/Z3JPrvIa4nk0XCrMHpBgRgFixOm5vw9QYOm
ioz7Ig8mtEc3LJlvzoTXKs6bknyyT5ZTG9OqmTh5+VHbtWQOsuc1WFgr8zAx
r+nYRBMHgnf/TWWtS8ErbvPw2AweFoaJS4ifJVxsR70+rp6OIt11Y+w56Rg6
9O9BQi3xWJUCW+tdGVjMruCLX+Gjt6E5dN22DDg+qnnCKyT3r3Lx3PmCqbgu
a6jWXZqCJZlfLghbxOB3Kt89r5iC455PdM4Y0TEr3ePuLBrx9EPnPDudaciW
TdvYRD7vzw2o2FNIx7sVAlfyfjGxS7WpTyuVicqJdX5LmmjIyBINfcmmIfCg
cMmgJ9nXq8urHyjRwIox/MXUpIIlt3TE/Xs8bn6VeM94zESjxH5a+AWSO+fC
z3KnWNAuEBKnkR4bsl0dvOsWFX+y5eepxjOw28L1m6oyA/Kr5+QfJVxa+nCJ
as0aGoo2nLP610D4wUyvewWZr2WSp97ldNGw9saKxnPLWfBsPybJGqeiN2F/
wDrRSOyZJVPvnM9CkJLXk93LCMceCprsIL38hq1ttPVpPIJs7ZpLI1h4WSI1
uq+M8O/r5lfTJLdTCo326u4UnOyw61vjzILKTl6dO9kbk4uF2qsuEC5/fKmp
iHC2rrigdG1CHByOfzfTnUF8/qna+FF1wt0S48HXBVhIq1PvNlejwzWktWTF
MiZue47ilSjJnZVLs38UHQJM46aBBwwodHZN5Y3R8Wx6rlrdWgqsz42lH8yh
QeOsKcuQ5HGHnNZC+W90nCiTbVZYR4PW9byITXWEq3LqTyU/vozZNHFRNX8G
vGZfnqtJ9qT8gg1C70ivJ8wXnQ5KY8PRdby4SpiDEFf2v5MbuLALlqFqfeNA
aMeEygx54j13AiXHiC9sXpH30HQdGxbh2bZOXC5qbpZkzCd5CBo703HzKfFQ
3r9lC1T4GKw5v+/IHD7eHk28ZPyQD/GPCTOyHvPR/img+E0QGyLpHXqahcRj
Nx5QTlfgop3tZuyXzYONbfGK89U0fLIMF5KQI3O3PzHL3IGJpZUbHWd/YqF3
kN/WTnxhtdbA4++k/32cPNvX3aJDRmmNH+UZBQOvValyrSwcs669NMOVhucn
RvWVBSjwPZV8skSRhvnZrw7X9TDwwCF+iSDZ34GVql25SxkYMRcOSSTnf2S0
48zKH2xoM5OOVV/ignVj3ZoYIQ4WHf3UFWufjAW/Nr52WMLFL8sveQ5kn4wt
vJn7zTwNfR76H3P6SB+6Ka13UEvHy7Op7/JK08Gpv1wZf4/0+WVvxoycdMQq
9Xbon0/C/Zp/GdmqKTD5e1q8tT35v9z9l7v/cvf/kLsVC+K0ZB5HY/pZ6stc
JwpcfvUeOvqbhqdKQzszlrHgBf19THEaznQ0ff49REdoW7UXfScdnUfWfzxw
gYmr6woWyaxhwsr/oPZ9bxoW+P8V8DahoUnWY4GuEAP+iQ6bdJ6QvIm0Vayf
oOB1c2iJ+cV49MT0qXoQPz56MMJTczMddY7G9mV/iI+cvPIvbDsNJ170nvfb
T8W8qtgrAyYMPO570Legmo4NHdMuYS10UJMC476QPDlOKBWG2LCwZenq3yXE
E8V+pryYT6Fhlb3Wz5W5TIRErRWns6k4o3fqc/zcSDy2y+y5os6C6Yyt9gIv
qCgp26Uj20C4LWPXne3D8RjW3LoiSpaFEw+Myi/rMFA+p1/NYYiK7qKyGxZK
FHzy23z//AQTrl3yjR7NTFy6vshBj+RtA/9A5/UHdCg5GnNkT8Xhc52p04Fz
LPSrvtuQQ/yytPBs3rwQJmKPlkmzP9Lwc2HHfMlcBmTZOsqbWAxoBIQNha6h
Q7+p6gf/JANdL5u9e7l0rLU6GLz/ZxKkR7R2vLOggTlv3j0V8jkjZ9E3/Quk
47vE713DH6kIL9AMCyVePY3+23zVy7hUd1e0QpRwuXK+v9pDOvJW7p1zrp6J
a+vrMiRXsKEauGpmoT8bTlcnpe/wOTjW55fpaMmBbOsdr4o0Fs7WXWA+Jby+
13tFJ9efhbneumdOTnHAwftPV8ieuyhzgbNpLwtdEr8KLvjz8DfvLH2VBQ8i
Ka+vXxPk442N9dNEET4iPdyXHxklc7T+VmbPH5L70hFNbjR5HeoiH4FBLvZu
jTO2+ksFvfbLZ/PzhJcNUjs0iC+Edcx0UNzGwr+DIcZTUmwonInZu8ODgVip
bd2+G4kHucSFXnGmIFLrxq9Ve1iINmDtqFWjYYudrE1lRxI+CMjnJL+jwqrQ
m91BnldcoHijxof48NJHYgaJdFyfbrn7PIiBqXtakmu3Ea5Wsg6w6ObgceKj
iTgLNmIol74+i+djasGm5beJ75rUp2+fd5sP644FHVPbUrFL81bMkjcp+OVX
xdM2TIOxcmF60ep0hIa/fFnez4VNZrWDvEg6yiuunZ0hk4iFZlXz0w8kIzFN
NqpXMhkS5nIulruikfPKaqR+DgUaLgkuw1dpeKuV1nwwi4lQm45YtXoq1lDd
js8poUNwklPm8ZOGzp/arJXzmeiuPK0vSvw7zmeLjtpCGszvLOg8O0jmlH+E
q5FDx9ab7pZjJ6goPz47XfERBbuNJ+G1KB7P+v31dLsZOCD65OQZATKvAfXy
hyOYeFAenqwjQIPvw93Bz/4Rn3ozum/OX7IHvxk7Je+nY3RW9NhRbzpeiGZ4
tj+m4sWFa8I73jERQF3m9zOTjjUaTg8eG9MgzVLX3qDNxMd1748H6lOxPe7F
rgGFSOy8f8+1iPjcQKFDqdRlKix7qb99DzHAXe0yqiOagJ/uSTMbKojvpT1Y
c76Mjs8Hc5POkp+rr5bzuvo8CSHGMZfa+Uzs47nbS3mSvjoz/WNDNtkHc/6s
0rlIh93fgKBQ5zhMPkvb/GeMcECHaYyIL430SnOl7ywmio/dzWnmkr1w5dWs
Hj0Glij6mCvLM7AyZl6d62saqK+Pl19cwYC589WV46DjvYR79XBkEiqnaqhG
5FwvTeWsKM+l47Cf580ZInRYWURuEGNQsWHVugmZThrYPRWvMnhxeD+nc2yE
9JU6O+XUfnJdS4J3rf7rzsSfmBXh+dksvB+S0fHsZSHf6Z7zcuJfq9e+DftY
yUal/TXzICUWOPr587sf0tDW93bh8Voman7kJ5d5c3C1701GLuGI6Is7i4Q+
MWHyeio97hkXtt+GGpbQuDiWFn1v0pIHn+gdB7tcePj3/MnzhrOk5/t9JYas
WFi81yEpdpgNE7mhxtNbuWhc+n7ua1MqYp69Yft2kj3rsMc6SYuBE65h5t7k
PjllWwW8Y7Awy/KvmsEiBvL1Yn+V36Ohq9b03dofSaBp7/i0poWJz9qHqm4P
U7HwzTo1o/IkpBlfGzhPzuW7p+BOwUE6ThepPauZS/a39paHr+fTsV6+NKro
Cx2SKg+dha4STy8/ZT2xmwOD0/sb2lgsFDq5KCo08pByp0kv9ATJ/cdxo8O/
edh2cFW0rUMKqhwerz/eloxiaU6Pq24qBm/83BxhlYaboVMUSVUulvVor6+Z
mYaIB/tT/DclwKXkzTerVD6sdM+sDHTgw/f1ptW6bVHI6fHOunMqCT7xHceH
3WnY8WAtewaZ24q1qaEnQqmI3padN3SKjsD1jQ+5hAMdYp02lKYwoN/9cfmL
vQw4PXu7RryECqbv42V3M6iQXP1CON+C9BKjWLdTiooP9Oci1nQK0iU9V1hd
v4zwbbYC9wJI7oI3zTd5RIPI5MDtj4uYkPcdWHi2iszR8ZbtSnkUKKcqt1uV
0yG8VSo1fYSGkdQZ4e+E6Giamj+eyaKCIa3hsZR4s/7eVx84O+gozNJLSBml
4qMTa45fHQPtvn9pQp0USP4Nob/dGQmT3hkr5bcwESX76MS27VQcn8VouzBO
x8CRAKMKuQTcalJQ3UA4p2/HvPIq0sNKYWsMZqtS8WMTHu90SEIPXcN0vjET
BmJHGKokRwueTPgcJz93d7vaUwk9OiQaT3W1WcVhcNaU6JUgMjdjr587LKZh
/MB7ufDLDOx6L9F825aG/Ts32Bi9oOPK5QA9kxt0mCxZ8bs5nIbKvO4Fzs10
7KrwVmT30rDm/OvcFvkkqLScitnKp4Lzy/5E4R46zutxRL7l0kgfhL8w3EH2
moSzSp0fyfPxRqElsnHYZPZBN2YfHX7msynu5Lq0VYXjJmcz0a7bzHpAePM6
/lX7uLGQLLupfyHhQGvDO9c017JRYL/45NVikk+OSOmpAzR89Jm6NbGECTav
1T7kIxtRngLfDc8x0foL1y8FMGHzb1LCV52Lp62CfR6fOWBnGmo4p3Oxc+0I
ra2ci3StzSoR00zYnxJ/yb7GRPPBu6NPPNkwr8x/IpLBQbeeoa0vmQtBBcvC
1p1UiLopxz0heyX+5/7VlspMyD42KxpYzIJxTlOZyX06yk+WjCfvpuHm4F7/
JvUkJHlUzvvoyIRtd5jSx7tUzDlt+NM1Pgn27pbykjZUVJRWTL0hHvNC4+0W
5lU6EuUKz40m0aCeej8r3p7w6oz7hb8kWRhvp1wbf8CGZ9a8Oo8BssdHXFp9
lvHgEfx55a4bLIhbMMffbuRh/u/UA5bnkiESnuH/7gsfeSbr4bA5BeY5nxY8
ckxFpVR7SbQfB2seBoUdnExBcLbSbrPGeEx72LbiOw8H7mR/kLzKQ1D28ypf
jyjoOy9RD2hLxGv1jUYgXKyjWBa3nXjG05spkXqbqGioGTPn69ChVTGvdWwT
DUs6m5q6CIexWi/qKP0gc6MoHmDiSHqF4nlkvh0Vp0sMLI8P0CBh9ligmvTc
2ePC5oWHKSiWP/0+3PIypGQPD71cyIBWk1FaWDy5v+VDin8Jdy3oD52xKpCK
Xyud5Gz3UmBVsupGpj8dUbN1135l0fDttdbZvCwakt6MLdlwlIpl29lvC8bI
PN+5oHKR5NKW+sfK7joVRyPz6Sw7Bob2Pw+djqZgQbodve1cJOK0m70rXjDw
3rjwpMk8KvasObZ/iEXHHFHlO2PyCbCI3B11/h/Zq/TP0cPPaFi078fPqnoK
2rn/KCpfEyExQ/7ZvV4GZjjpvNqRSfoqR8mQ8YGG49d/X1Yh779Ujnt4m0kc
SjUSNi8TZGKSXhuy9R4V0i8yHo4Rr/iY5ughJ0nD9uhtXS9Ib/fX9w/uMKXD
mCZ2mKJLQ5xMUZUw8bwn1ROPY8jntW3JTw2rTcTNuZ4ZpWRP/P425PGc8MER
mSVHz5jSMNGWnGg8iwrn15HxmgsIbywT9bUpiMWI7cxemX803IuvDxIdpcH+
bP4e6f/9fvDhjrUC95gwreuZs+0zyc+A2xuRYRb2fZN4/SCXhV2HrbdWmjLB
e2PbXE04PnLdDVEe8cQfu49/1D7Axs5Xnj4V7wkfP6pw0BQj++nLYQF/Ogfr
pQ40OBlzYG1vsVVqkoO95S7Ddcu4COaH8GcnMJFu8SZjlST5+cpFxx92s5Da
L6mtRTyq2b/gk2xrEv4U1PDf1VBQzTap361Ixz0TowqzNAaMTB8p1CczoXDB
6vxEAB19EulnKrqpcGn6KXswIBEBRRsPD35mYDDXJlIpngrlzbNYV12ScGXU
qEZXjFx/+IDLPMIFxe/vmnbspqPZTLl3hTQNoVXztr1+QbxYSPaPZTQTed+W
N0UZs4FBl849NsR3xBNkbvpwMdF+eE3PXyZy1r5XUYzlwt8/1v9iLh8nlk5W
NhOePvP7yR6f/cnwrj61/WRACrakOCZua2AjkFlp4yCYgsclzAUu+fH4tAuv
dcx4ONa0mhspxIP70ejYrL+RCBJ6HuRjkohFk8L0G5+oOPJRq1nQkdwndshu
50kKvs08lPmV3L9WeYuWvX1UPEjrSjvVSEfA3AD5d7Gkr71ZlHghKmKPeV1X
X0gFT/mL2U7CUy/cN2seDaAA4jm71NYRjr1e4p8zGId9oosu6+fTkeq6I05p
Hw1BLbMCgzcxsHg3zU5yIxUD7Mdu2jMp2Cgty/lM8r79K2WDsSENycf+mCRv
JbmTvRrD2ECFf9newSbSI2JeFg7HSB4vKgt+vetJReLPKkn3n8R/Kp20tXQp
6Ojo3VueGwn/6g+Ouu4M5O7b2fC0kYKjraZfo0gfJC3ZdymH9J1vXEuCbQ4D
omv6il440xBYt17d2ocCf8fmodHARIQ/vFZ8j8nA3bbYWkOz/3HLJ+qJYOKz
zvXWVrdo6HCV+ta/Pg7vJHdljFKIp8h7dFF9qbCpbXIMJ5/3HP9Gw5MWKpb7
HBI/QHin3XCdxIx3NPQkTR91+0ZFjbTU4M5N5Fw08xU45mQuFl9fwHNLhNpD
SRV7ch+Y2UP4fpMGU7WU2yfI/B04aBU+SOY13nLUXCmf7Kvyv72F5rH4flyq
1/0K6dUfSxWlygifJIduE7Jn4JayVPwMWzLH15QaYr2ZOC5sdaT4NAtTu4d+
rFZmwfdsiL8aydVGTTllhRQqdn/dnV30lo5XCl4c0w4WFjGU710gn9so18o+
l+S4uC00dXImB0yTI0Hbc9iYSPhrFefMwbaFW5fPDuXgXPGT4O4VTLx1bBH9
F8pAzhmRplHSs/J89WqTk2zw86gyFGoiLAe+OfZvpKDv3dTdzBxyjjfsznfI
MUAZWe81rsrEuw+R14U06Fgu5qVx/QIVb2805y9uSIDeeR2XJWcYsFl81O2S
AxVhOgqeIRpJOB7lv+16MwXJFzptLclcPKx/MoVx4k/jN/w/XyHnzxDsm0t6
U7z/hHn0DLIXMpW7xh6wkJlwpOBZFQMmWk61ixo4CBYT2Ke3j4nHaQJxS0if
r/1QGH+1k4fP37XOjWjysMct8MraYD5G9GMvzE5Pxo112WONWmwcmbXiupRs
Mjxsvg3VXYtHvMzl/sX5XFRrXLtReYyLtX+e5V9iRcJHMDpxf0kCuAstppSz
qUjlhLjOGCI8FxVaXFFFweCHvYy6KhoEPV8tCOdSsTh0VYXOMToyr6h61qwm
+0Q/rbb2JgUn/W9V/ib9Zoa+XcvI3IT4ae+5qUqBx4/6zjsCFEQ82z5iz47D
WtNZv++b0fFSr0JUWo4G1xB2/IY2OnZsqGxd9YeCFluZdg2PJMxupYRODpN+
+JJkFk72QQGP/zOOzKd4Yv/GqtlU9L7ttdulyMBEkYa8Dsnv1YeJ3oVKVDhM
PffKI/tgkc195YZHSTglu2dozfNI2IRxNB3+0WETzKvII3wjJ0bb8rudBm3b
X5MrJRKguWPfmVhr0neeZjtmEV/wW7HMJ2IxBY+qAw+cE0lEWrz/05dbGah/
frHiTj/ht68ufQ8IH/ivzGo6eoYG+d2dWjTVOBjqxBROEm/THYk+Om8lFV7V
rVGfjOio8d60mcmm4k/sCoHU64TnnXfsVfan4b6b5RapNNJnh+ZUCYzRsNx3
Tjyd9I1w+4LR+/MSYd6/rfZPEQVnLjjqv/Kkob/v6rnMKCquxi37KHORgttX
fc94baOi8ZD2scTvMYjaPNkh70DDT7/MdVaBNLhJyNNPCTHQX7uiwucnA3b3
6mpaRxko3S14kTLKxHoBxbDhq0wcWDY9lnCWgTed11rnmVNhWPBt+Q1ynyoZ
UhPrj7IQOfOp4ngpHSJFEWZhNgxEp9wQ8j/HRvdkQTVHjI0RF9X1Mo/ZuLnz
3PGlg2w823dd/kY5g/COhD5jlI44Wc07X/qZeNEz1H2vi4USXdNAilMChqV/
hgj7JGHwUVbr7rU0zGTPMijMoCOskimveJuB36tvbJT8Ru5vUsk6NWkqkvkl
stprEvB2nsMusTkMbBnd+fiTChU0e9kI07+J6BNXSrVjUsCZlWfkbESDx7Jf
7l6Ey7//iH3UvZkKg8/SYcUvyfnNEDsnFcLAsjapOWXEP0d8l0fJaDBQPkwd
e6rNgdfLVfn7S8k5/erdU7KDg8l5Y9/vKvKwMNJ+paQHF7306Qm7Yh4+FBT8
xms+WGf82xrSWLAYf5+xxZiP0KPKl0sexmOZqJXF8CIuns0VNb3/hgOR3xKb
y3QjUapZ3PBTOQHb1265HO5Bxfrl224sTKJjd9be7arBxMt+VexfGkaD7955
L50Jh69W4PMn5tJRlbxgZk4DDfV/fpSWu1Fw2N6q/H9/T2Hl0NMfV0aomPU9
KWv6dhJe3r5Wfz03CfGBnSpWxnE4fL/qitkXGmLsfbZbf6Xin7Qyf9SPjsO+
bzp1Sc5tCyj76poS0Z03vO1pOQ02fYdGIghHvD+p+9giifiW1ImDuW8p2NTd
emKomg4h5akL12YT37a7dPtXDwXed+fy1VTp8FX0Uf+qlgR1iZPHin9Fomxd
fesAh473717m/O/7WwP5gDsjUTQU6GWtiJ2RgMSBCX77FB1fH869b5ZIPFVn
jtKmsCRsqRC865eXgAcpffT7ZH74k0+VGEzic2uO+R5soMJUTmrTXj0yL15b
dMOk4hApVFLbU0auy+XhqqAWChTWLNjT1kLDeokbGg4HqNjb+nPciXie2KKf
41HiNCCsNCbYnorp9MTcGaTP2o5P6VgWUSGguuf7rZsJkPn9aI6nBwW1wbuN
7FfS4Pk5SrR9DRUxhs6RvE0UZIo23nzUS4HD4vWL/fkxqItYFi1OuLP5+q2m
EDJ3140ay2XI9ax8UO03Sfhg0N/U7VUUyd+AZ5j1JSaGVkXtfqDLxLT3x/2X
lpE581I+PzREOMHv0zm5ChoyvU8/4w6Rva8uH1+whg4NGzsx7z+Eh6lbtjmO
stDQtEj0XhgLUgafT5htZOPc9hXzz7iz8ev9eROxfQykVi/pOHaKjtzPV16u
DiTeNuErou3AwpN0Aa2b2vFQ+dV0bI9/Ir5q5cQNVBLeuvU311KNjowgifs/
dpPccof6OrNpGLSZ/tZJ5iM08u79f8x4FC1oKcxIpWPyjOsKszEKnr94edq8
PhFDO22CZ5P7m9S1V23yDRUl0Raa4vtp2NxTnXbvDQU3I5gzXZ2ouHhG43wC
2UftwaqPDDqYKI+uYqfl0aEQc75udQYb9/9Qa4+vYMDY+0rFoko2VIaC05+c
4kKoTCdx5RUOylYQpR/g4pOAne3oUj4+z9zbuWoZC+XOYWzOaR6WbLX+86Mr
HhpSUa0T0Ry8r6ZXN1tx8DQzmC59MQL2F7QqleXicVpjyiaYnPevXbdfljyn
oarOY+zRHAosvO55d0nQYFFO64ooIz1hOrBj0IaG9Hdr81oIL3NmmoWzk5Nw
ttMiOYXw4cHTVw+Fk/nCUeMtCsmJeFs3N6P3dyLeWwg/+5ISCw8//px4TRr+
ZInKmV+iYn5vXMWXUhrmcEVufTCn4HPQUNjWggSMeGSf0TahYW/HRSedbxSw
7Y41MEmuXvNmn664RMFMjmHbgr803KC28eUCqKjebNLw04uCd/5+mga+NHyd
u7ana2YiNgzcKjuiGAXPsXk9V9+S/pTyji39lITPJnn/R8GZh1P5dWE4JSHJ
PDWhfojKEFLKk0IDKWlOZayUEkqFEM48n4MokUoqIpkyRIUGoUwVRaiQKSpK
yLe/v1zXmd53773Weu67znU2fCI56rpiLOZ2DRvP7Uoea3iKsOaiZcMY4dCY
3Kt2R9Zz8dJs79rxDjbWUXCL5i7CsRvypwJ/CGHjVeaYoiOA7M6jTy3LBFi6
61DDl18M5P2Xk+jfK0RHXnEbx5YPb+vai8myQuh6t1qs6+UjZAbl29l00q9H
wxONdgoQ+GKhdHodH9t5RevsjYXwyOKN7tAWQEYn5Hm/DAer21smZYp50HdQ
M06IFUA/6pB+wUU+FKcKsv4yeKisbGFKE0474SZm2iNLx+otaQfuBQkQYr7u
jXWlAH21zfqWmiIsjLg/4VknguPBn3Lp2SLcuaN8amliNEzubz42n/hwQcr7
uWmHRZi8G8MZceFD1feQVDjxZf/KzsDF82LwPiVF4qu+ECUfrHOXORGeaHp0
7ZpYLHRDymfePx+DyoYrLszFpJ+uK1vtcYvFgiebD6+LFyFc9uB6JZK7hXNn
KRsSb9s/63jSuj0x+KUr0pOQZOL7wdhl766y8bf5Q3JcGx8a5/u+3d8oxO9S
zcGldBHiYpw8Sv8jfSN7c0mnNHm+pNHzRyQLU15HFmQ2CpF3r2hnaxwfjfX6
vQtcuIhu21z76x0PSx2v8//NIXVoEWn//gnhgSb7sAZ1Pjava41zDeJjVlKg
RyDZl+srVgSWycTgyQz1ZhnCIcI4qZ4Lwlj0Gf/ujyE+UOb7qm3uvVioTBaq
+evFocGuS9dr8yU0nS6wf8OMg9cEd/pIQTyiuysHNbVi8IeWIMgZjkPRqP+K
vbM4qL9Q4H+XQvq2XfX58m2X8Pbr86AXOlHooppaOieywO/oVvtG5snZOUV2
4wFC3OC+nTifzEO7bsrs5S0CPN1y5B7Xlw/3BxYv6ggHLW0WetVVCHDEaN+s
DbN4GN59OGeEycOCgw0lFEUBVi+a5mWsyMVMRbH8p1e4mEif9V5uGwOs/XZf
ppM5dDKuLPGbgwC2jzQSX+4jdakhbW41mw97Xs3BRws46J3KKkiSIv0fOc0h
IZ4P5/vpR1uEfDz8ORHb6MzH2oBnV5cmCSEzZcC62oj40XjXNzdD0ncr3s3x
Vya+V+WWvySQg7FtGga9JlTYVZyMO018zKl6ZQn1Ag+PDkcdOhAjQGuxu/iL
TDZ6HvwrTtUg9anbcX7cnw8t67mKyekcTJrxivUpbBgXzlw4Y64Ii9U6DRYS
DxH5XDv/9x2Zs821sRIsAZJ07r7V/MRAyWLL+3eZQhyos5/dQvj6jPwPffn7
AmzLrXqun0v2rzvxwxkyl9SOBkkvIN7236Zy2n+Ee5PZZTt3/RRg2fz2oN2f
+ChfwZOrLGZDf96+6pqVPKh+uLNagfQp3WTqRmkjPq6OdfxZrcqDbtPZd8Zl
PCT7WTfq1tKQ9Lb67bCZANc+nRCyRQJ8tKhvsCPzZsf7qREVQcTHjUvrx7YS
P/+uIF+wPBqr4r+pyfeLsKo1y3GzIuHhZTfcN6jy0Vtz9WaGsQD/CV68SnkY
DUVB8/3ELOL3o0s/XJwqwmVWxxMTXgwoSqeTrk1Ew/nynS11hFczdjQeVe2K
gclauXmR1iLo9BZpLZgmxON9UyY3rYxG1ZkZokWfouGjoYwaCQYsHN/+jCph
IdfgNWPyPB/TGfc+vyP1Ju5yerkayT2zvsMUn28CfF2+e9dZXx7O52UI3kmw
YJy6Sewl4Xupja4dYsf5+J5edNF7ERdRktyGf4RbJ64ZvJp4QryltnNvVLAA
Z4MH5P6E85AjYXtCXI4P0c+Tc4zJvN983VP09y7hzZ/nfkysFEI11OyntxLp
O41+b2+2EPuPSaufMY6Fsu+yjBTeJWwyXOk76zbhZHv38YdfL+HjoPg0gVo8
3Jc6hpvcj8bkAQ29DNs4SBeermhV5aCOnmg+R/YS9Lt+nN/RHIvcA/X73tRH
Iufg1DfrF7LI4wEldRQ+/vppri4kOTaNaeN7x5SHh7bi43nE2/1j91wt0Sd9
kdzxV5N4YH7vGQ/LAPL4xaTWAxFccDe88YtV4qGo7dOBJ1V8+DSPsjdxOVjZ
mqa03Y6Li+lJ6nRxBpqF2dPbIwR46PqCFUvmkfWWphz1qUJ8/juU4vmAB5P9
y8wU49goDXq95QTh8M1Jct2rt/CR8XpBoud6Pvqz6A03iDeWxq+ObLARIjWZ
UZncz8cFa/Rs6OdByy/ukHUpmW/TV7aN9bPxSmXntxerqVglESf9l8xNPWdX
lyNzeDC72NNwfbMA/+Jttz7gsLEvtl4/oV4I65FSPZ05fBxs4P9eMoeDpIPs
2fL6bNx7qiP26B3xhX+le7zJnKAVKRR3sYj/PRA/s24H8d68R6fUnpO5kuQY
0reE5ILH4s3j5JwbqhSWbibP/xouFReGE37QG59hOJtcd8N+KeEN0nc73jhn
7eRjR2TIM718sq/rRv3DE/lgzKV9FJ1kI8s3+v7PIi5qfjb8EykLIFdUkBf5
hIclBrtpare5eCM0CZo/jwcd52/TKi7RkLmoKPkO4Zaepdp7HQ4K8LbOqL4o
WojIJ6dKHXVEOLspYc/X70LIflSli1UTP/+YoP6eRnLQo+OPVZkQSzzX9Ynf
5eFjj+hxYcH/f6dpQYDJnmjMpOxbP7Sa9N3+3ttv8oQY0q8Pjybzfa3DsZt5
hFtlvk3emmkcQ7wr8Vm0XwxC+Ad/mPQTjjZzO/tOKEAsZQrlCPH4GeOqoSHH
ohHwvEN/XJFw8aY0cdNOJn6KR21QIP2uvnnusiuEX7rfvPNkEe5izjC/MeP/
3C3bZpH6mosIp83jn/lMdPuO6j1QEyK3U5A9QXj75p+wPR96OZiSyY9aNIOH
4w+iMx8d4///97n+PlshwLvKlANljVzcOS0Vm7udhyP8aMXdjuT+pKYV7d8S
jctOSlcNXwlgKhM+petKDAYOH72YriiEw/ZE+YGiGDxQEP03NBGLrXsf9ITM
Ink+o9omceMlPGvtzLUNi4PKswtfTtpGozDc8K9l4iUcPatjGKDBgcFLw19r
EmJxu9K/VvZYLEx3z6kdj4xE6M8NSc/TmNBMlL3znxMfCz4m+r0i3DRzQVxI
XyUXkvSjhQfOCMChG64Yes3Di7eMm4XbiI9EL+v5rv3/vKhUTB7jwH2UWtKW
zEVyeeI3NTofOU+GF6wgOcsTz3gU+YMDk0WTK+YX0bGc+vzJhKEASz5bHF1L
uGvspsgsL43Uk9OjcJtNPAgCjhkqKRHu9fNyoyeQPP25j/+H8Ov8EZ9jGn08
FPaw3C3HefAL8Zj9qU+ArCvRWRfS+fhWa3vIOIyHgnNXsqOOCxDtZquicIwN
nxNKBkHWVAQ81hQpvyF5Mul6aW4xF2K3WmqjppLXdb9/+eEoG7+buBvruELc
O7j6gIjNg967gV0lXDbSH+6ZRq9l4dnFczXUGCE4q29raBPuTrdUv7lpHR/b
ej8UNc8nnlrhMfw1nYGtyTetZpJ54X759urEqTxsYN3ark1yTUW6q+AQmR9K
qb+rg8v54MSnZ3s48uGVc1TZUIPwu0RkXWoI8T//H573D/FhuaNoaKE2G8pL
5C6sWsdFyq6jCtOInyxMupYVokv4Yp+qzq7VXKgn3bAWBHNRH/HvxLAbDYoV
9sevZfPxw6nC/4i+AHcX3LY8tV2IKQeoX2uIZ1YPT/CMRMTXmz7HVnmL8Lfy
3rQVpB9vT9++2yFQiKW+i6bmGfBg+PuX6U7C/Z22C1oYYyLMUVXzZpJ5elzX
+N/XE0KkzHdoCiA5mL4v0ShBKRqzpG+0dxRFY97WufUXp8TAem7xWbUEIUyX
ra3uJzz+k3r8mu15ERq/TpeYHBVh0ZxDpi+MaHgVv33TGiUmbvUuSNIkeTZH
3cS+WEEA+aZDZ7YSPg01+XjqGOnDB2bFpn/MuGDMmhrC0GLCmqH9w7pEAJWw
u+0v5Pl4qdrXl/CAgx2+jy+a3iB+kO28VkmFj92USRf/ET58JdkZl4y5CJfJ
rF7D5sJgR+vNxF4B4uMnXl36KYKB5YjOOjcBeotXd3D1YkD/u3/oyzXimx8/
RUo5EI5262qp9P//95mYa+cGxeCK+6fs9gexcMNaBf/+SzhG2eBr1ybCm7Sz
S7P+xaKRYznCVeZA9lGQ11XDWKxza7n1/V8MCid2z5m7IhIX/pwcub+CCcms
tOTqeXxsTWhPk/MVIEHveJiKJxdtPLWbausF0HapSrVx5OHaNx7r4iQfbcaU
TXUNhEfEre80nePgyNTUiiQTLlaqun0+RbzmmIVH7UseGxPnXjFsrnHgKH3U
404AHZdk3rlbEY7yep5t84FwraG6yo7w3QKsSFmza6iDC0fz0hiBiIVvHrJr
Ph0R4Pq64q9Hz/BwY9G1/DM7ebAzGFwUfJuHmCP7B5ZcEsDi7kdn2xPEo74t
e2JP8nZ83VKDORoCzD9pF2rew0KNlsKPYNJ33SmzFt4kdb02xcLC7xAXlTGu
vA/FpK7q16ulWpM+19eX13UQQsd1T3RgNxcvrMYPy85g41v4eovjwSz8Kzkh
sNlJfKIpevviKgHmnDG//m+YhySq3/ziHj7U0p/OT+Mz0NZQfaTlPLl+71iz
K50LreyRDf43CSew/ZIukBzN3Zmxd0UIHzXvrhi2j5Ccim8XipH8Miw8xSsk
+2zMvvXzpTYfP6sOeu9+x4L866f+h6s4+BA5OGd9Asl1r0s9jwVcbJJdwnla
z4HymfSVtq0cJLBNF84ypGEix2lK7hkyP/85JT37zYfsg9Upa8icWO33TEyN
KgRdanPrHcJThX5BeV+kRPjC9zkz8lyI1o7/pC1JTicbhsqqZXOhYn3g7bEM
Ht7EzNnj8//vA7kk7Z/vSrzozmtvga4QC1oKRBTnaNTELGJMTRUhyOcZe6Z9
NOZWCkpjRdHYYjBHa3KLELvO5Dupl5J66FwynDpfhOS4DOUjHBFsq5rLXhyk
QnfEc9vUjQx8CTwoeyGPi42K5bFiD4ivznr13GiREBdxL9YfArzp22cse52D
vp19ldYPGShoW/Td6bAAa6a9iM6t4qF0vp3eG1KHAYWDq8XInCpmJ6vm0ngw
/eMR6ZLPx8V/H+bPjObA59Vsb22yn6Wzn/mWs8m8lP4tceSqCJujh9tbyXzU
bl0vNr0wGuMeckduLhdAIypxfGd7NIasbzbn9cfA3XZkkUd/NK6dY7f/WhiL
47ttT0a5XyJcXsZZFU64QW7NTv/DsXjYfCrwqxS53ri01ZvnMRij7YvhxcWg
+eCh15N9EdDSd5BofMbAw653R3yJj95bVMv0mUt8tLbr7sQkB6/fqtQ2yQsQ
pesx1+4NF1MTR7btvU/6ZrTm3jOSa2WuaToNo2zYn928MvkZB1sqvwVlivMR
PjS0r0GZjcCKZuWIrRzUhtx/Pm0Z4YotbyapfDI3e07l7OcQ3/kXqLx/Oplb
XSuXrg/jYnzoKv+QCgtRxbMN3EwFGF6bUJVB+uDkg0XjKU+58Jgxxhoj/ff6
k9fTJFKnQfFvy1uJ17me7bU7nM6F2k0Xsc5KPg7tObnR9CQLrV8m5GzXUFEd
5ypNMSC+8/j3gOl0Li6vkJGNID6h42BqcFSNDefxttHzM4kvOCvl39zEhUPw
lTsfiL8vk17eHqfLgn35ac57FSGUPYsPHKCT/Vi42b/hOA/VsWZRDeRcOZWb
FnD8GShX3OW0epEAh2arhkkpcNE/a/cu5W18DEhUbsho5EFyyg93mJM6KLg1
sYys49etQQVlJx5a9Fc/2iRDeCxQ182DcMWbTb1Ss7ksJM2+32S1jwOZddev
rHQn3BQi3a08hYtf1/ZNXXaSg8zUH69GbTjgt73+FfKPCsv+zikVKwknU9Xz
dzwjrzcdfjalQYC8zhkJk6Tf9i9FqtZnAZL/W3BF7Z4Ql8t82J99SC4Z8869
/yLA7zzLPxngwqY/3d34PRf2FStPJTmQPnEX7aoZ4EF7mva1W+T9VuVK3Kgf
IlCqpngmryHeeFKGj1YRwtJePRLoR+Pdda9HecQfZi6P0lHdz4fchKXZbtLf
JzO3mp9ZJEKVbX58UxIFCQ+svHrD6bjMHXCpnkZ8pMK8xGwHH9d0+Y8mCTek
XnigIyYtwPQOZbloRQ464+qCuTsYMCWrLpETIHJqoequszxsU1WSF1px8Ep9
gOr6mdSh4XDh+y7CAanp7QeD+DjFDF5UPM7GqM7b2/3yHBT/lChiEn89pfFp
KX2zCDm7F2vnX+ZDeL1j/fDWaAgGbirRXhGvmSHccCI4Gu3N+vNsT8Zg8c8H
oW8Pk+c1/3juJjy61Szm+GhLLELlAg59IOtafM/A4WcDyUfVdNqqP2xs8dZP
POsZg6Fo82CBeQyiDILjQ1IiIEhwTp25l4Efge3aN0muFJZLepSR612d9eBV
UCIHdPkbktHtfCyZVqH+fCcXYX7lm3U9+LBdcnn0gRWf+G7YdtdQNrb7uE7L
O8DBjQMBjedYPKQN7cWSJBbqe37k6k/hYP7vfLmZPTS0G3TvfkQ8TV1OpTyP
8FRK1GV9g1ySA5deXlCaTzzRuu37jgQmpmuPuP4TE6BkeAVfRPJpQev4uvPL
uWj546GaJc7DzVdKYxo/+Dh54dYOj24eOhO3nXPewAX8bh97cYEPc1f+7uZh
JsRfDXiuN6Pi3ev75fEfSf3vbS+0vMcheb/k7U4LPoJdxFMO97Nw4fHnvHSS
Y0N3766qy/x/PW8+802cBY2BmsihRiaGSi/I6DYT39w842OiHeHwq7K9H75y
YX00sE+ByofUPw3lB9sZyJj1ZWxaLR/b15/juyVxMOvNX4e6MR4cWzcFPgjk
4bdUw5euLzywCk6f9M/nIvbE+2zPL1x4tUwm4i2ZZ59lO6648CAj8eHRMjsW
6mo/3+H2spF2TKK8ejGf1N/hYRt/DnbGrZb0mcmBdbyRk38mG9eNdjBrXlNR
IStr5jbJw+JDlp7/YvjITDPVjidckPk32OXgoAB2osLe4zziv0nDVauchVjV
knPmtZIQNztKtowkCZCWGTx9Zi0HrQGmWyOluPhjtdq9f1SIvyNSazcu4uFd
tu+MiWQBzHiD841jRHi6v8nS/L0QdydWRlqdEWF+89kj5hUiyB1WFlMsJOe3
KZPlTLgg7dKOS5oBQvwplxgbKBHihqynXtdwFLT1f7/wKqEhVKW64d4uDv78
tybR8QcPvVKrnPhRAkgondX/20jO03TBVEsmG4sf647/GaZDPrrDx+sR8fyY
MAmPhYSHJuJPjk3joOZiJQxoHDw97i22zYHkXmzF5E7wkc+zadx2nI0/ZvES
JgfYGP6c0/iwiQ/HD2ovSv8KcWfXbI9I4omhp1fJKfaQ+88su2BylI9fN45L
/VSPRmq+q6Hcr2gYpJ+tl+gUoUxFwiBGLQbrdoyu2neIeN+OO4kV1UJ8FS3Z
lbspBjOOjCZvaSeeoqRSuWZ6DMyW3jZKfh+NhOsvl1yfF4Gurx5T51LpeHDi
/vK0h4QvgmhSYyQ/jrdrVqV0sFFz8OGrJ5F8VEx5N+/KMQ7MNQ53UBJ4cIk4
YPCfKw9v76meCRawsGXR0aLj0WzcsKj1ronnompVe4VOEROuxoEV573Y+MQZ
kbh1mAZe6sbTCw+R9yXO0ayX4GPjEt0btsRvVFJtPO+6cyBcsmX5vCwGvkte
uLr1Gh87L/8JZe7mwHQ4lLl1AQcHs7iOHC8urFyWBj8mc3FmXvLE9mmE07L7
y/9FcTD9oNiR+jpSd99yQ4/PZkJN85SD1xwq/u3zPTZgw8cuh9sKZ8j5uJhU
nJh1jodNVTsWSBKuvbK/MOz//z4SLqdbr0LO19mtPvebAhMtqxVM7lswoZCW
WMbxEmBERK9++4LMxTfz32Z+5SDP8N+1p//xYdec1b1HlgGLgtlnNxmTfo4o
LdpdzsaK9fMubizl4q/BNp2F5O/+9jUbm9u5GF0pnvaDykHVNz/RqCIXzCn2
drNP8qE4VHuaGc4l7586a30hE3e7o4bfqJJ91bLp6gziIUtT/vgRNzZKtfes
Lt7ChqrouEoNycXoN+aWNwOpMPAy0d+ixcNndbnqg2SutJsklNQvFKBBN/zN
kAfJaYZD/7vvfEhcOLskJVWAk5csj/71EeBcQG6gzDwBus/L8f///5nC1c93
zrpI/Fgx2Uv1gBBJlzRi5hLO23J7ucYHkuceCsWH078QjtQz3OekJoRRTFpx
DslP086PHdfFRSiUVfBRNyL+9O7Q+wM0sv7gzuDN9cRXhsUGnSYE0Jptf/l3
ZySs16ZMvrGhIsOrGfkz2bjXXfJkTgzhm1BnvV3kPvfQlZa/8OMjoT12TjWF
hannjj4NsKOj5spaXvdc4jljXY6RhJNctnyjKRMf53kv6QtoZCP3Rdn8dVoc
LJ+XscLmKg9PWN5rP+1hQbnrskL0VBZWT/TKhi/nI+zRlqirHkLYHV0s82MO
D8daIhgVi0U41fqleP0BHsKVNnRO9RDhgdqDgF+zovG8/tsFhV9CUj8r/5ux
MBoMqfZfK/1j8PDLpv3zDYRYOIXnbWkTjd+hlU1ut9i42GP9dMQpGvei7w2V
6kYj8jW9QKzlIsZ1S9vva9Lx395JubUHuEgf6cvb/ZKHsMyDe+TobKxfdeXt
vF18XHmo13WV8ORhT419FTI8sG2UdR+85YKye7IrfwEL050UVhUvZ+PjiZkO
ef9x4ZSTEhS+hYmNic1jk+psNBe2ik3TpeGU75UWccKr6UWGRmL5PIxmfFz5
JJHwy+xq52WSHCSyftndAQNSs46Od53iY77KLXf792zMyF56fjyeDd8CbnWC
KhduHc1uBZLEG5jGvRvJOc0T5H/z1eVAz/XD8MB6HgKZllbiyQw0V5UtWS5N
xUI7tzXDv3no+anW6JrFxk1d0UpnMR7WJXXdkvFkISjuyzF/4oXcdDutzAkW
vv7bGXg2iYEHna802/vI/fy5pH52qQBemxRV4th8aJ44/HPucQ4eNK6TX0r4
yyEuV0azh47h7lfzlMn+aTgffxu/n408ydDiZB0ufCN7vm49zMXYkuo9v8nc
CJ4mnl32k42z8+jyXSRfX+XsGyyz5GNtjH97twoXRoo/D88MZOJz6dm53g9Y
OMFUN8lV4IFWuD9+ooUF56VjKpE/WAiImcXaOI2Fvx5BkXH2VHT80/nxm3hS
ytulQ8PSZJ7ZdJ9d20c4r9e5+4sm8ZWrhpX3b/BhZJx+6vpO0hefvUv6iFd8
aE2NL2jlg/962/TJq+S8VKNOhDxjYTJ2764rCv/3wuqsxG9sOHL+/4NifJzx
yDutIBBiwDYq8dgj4udRHS1PHUnfqQbskLwjxJMuV4XgAT7Os987nm/i4O0Z
33l7aOT6M5yMPlwToONMX+NwRCT+yAem3mBTsNQ+k73mBAtWk+lNh/o5uHzC
0nvdbcJdppUtp6z5SMzL32cmzYLw0XfZR79oeGXi+FoliQeTDRvtzxlysVX1
s1TXdjLfuRl298PYWPLo00Z7ARs9ha3PzxvzULN4ySzbd0ws3Naz6s9+JppX
O18+T87pKaoXfpkjRNWQ/B63QC6Ui7+yPtWQ/IugLbqXyYXF/sDSMRkRTLks
oxPJInzfMJe6JkIIjZGsK1GZIrT8eOU3MTUGmquklv1uFaBpzLVyxUsRAnQd
PvIi2bhdxGO6T4iQLzMWtrlBBHud/RFHEi/i+gaZDTWPaJBUPbWukPD1pVGp
vgEz0lc+Ju8ZhqRfxQ7Wryd84yX9yiWMxYbsVtOZK8k8frdqr2XwNtI/sgtj
2u4zsXW995PQRhb8pJ7UC3M4EFt0dHdMB+HXKw6P7r9mIfVqe1tiNxUH5Kom
ZtpwcUX62/cnbmQ/EtsfLyR97Hk+R+M+4aWL7pHVNW/oEPRcXe65jg8t2ygz
SRc2FLMN25LI3JccC1iYXcmBkx5cM0N4eF/9OVBKn4sTMvXO6lVsKJw78vI0
ye2jbq/Ur5kxoN7+aDp7hIJflcbtKd48KDfedxEjeTHr3f050lwuXp1sn65k
ykKJ85FLczqIlx7eFbbxDAs3O31zgw0ZULnrGVZ3nYFU2ZQvxSMk929wlow7
8ZGjIX9xnPTNihnzp6UJebDry6nPLqdDK5q9/MxyHk5+kqytG2ah+I/aHD8B
BwwNi8otswkHtxv4RJO6+mW2cPUDDzakaoJvuDpyIJtg3xZN+iTm3ivelPsc
hIvvy+0wYaJBSYlStpUFLWe/10r3uKhttPj84AAL02YGuVPiWTjiKyrNCWLi
jJ+v8yYtKt6OpJtTtpK8+HXMqa6Yh/owxT6VPD5qlm8NfvWJ9Mt5i08v9xKu
j7NL/U28ujjt0uD613y4SW5X2nadD3nVPVEtVmz8uHbb47M8C9/epS1mVAqw
x+Dy013mbGTkzUzWzuQjyEjsl/NaIVaqJ4snEQ4wGvJ5/eKHAOXe+p+dtgmh
Ozqt8nsGHyNrNZIXG3OgsfKOX9MaAfrurToze4MAB/lCg31mkdj8NDHx2dso
8LojGLPqmXBK8yuu3cRB6ilR0QHiM0flvz7SVyC+J5f7N1/IRNZelZqYWzTc
Wn56uG8GD3t2y39LaOOAYsXdeJ3Ux8YtPYcbDNiQrvgtVS7Ohkg6Y0bmCy4y
Bsouee1jYt7yGjG3AgY+qj16upX0o8eNfNuLbwV4a1pM/f6WePTcoFijs0Ko
7zQRP0o4O8sx4ptVoRBxx8/Ma7MQ4b7T4xllhMsZ9rmvT6wVQS3YrHZ5XDRK
yz686Y0R4HxCS87n7SKYnJ3W10h49kneDX5LGuHSNu7h95EiSJSuCspxuwhR
YFXIrgM0OGldHPmXxoGblFepIIULO9U09eH3xC+KfY5s/0v4HGdi52qysXRz
nHbBCOG9DKu8vdUcSP627a21YWLXnh23+edZuGNVGwU7DlZbmbfbBjFAPSFc
cYHKQmZQzEmPNCqK57aflC/ioDrjrdvbWTxodUW4MEK4aNwaGq+4l40NP0KW
+3rSMWVyUPyhEh9dUy80XO1gQSFkygmlOBYiupZXl4dycHvqWMXMXi4KVl/T
OfWEgwDLMdVmfza646vKVxlx8VZ1ls/cajps7iUfzGyjgJMszHzaysULO8tb
KxXY2BR2mDqF+OTyGa9dQfLi/h/XVXppfGhbpXjeG2Tijf7O8KqndOyYnMw5
sY+BZbaX1O2f8PE8NmpXnxof+74E2OuEs7F5uaLbVwseViS8jTVKpOMWJGPD
krm4eKnyygchC8etTvdETrKxrGdDrlghB99DRc6R5zh4/kc4h/eWBWlV89a1
39lImi678EsOD61rZzxOsyc899xpOG+QAXv1hV4F34mnBn0zad/ExTHj6VsO
tTGxLsWg7/o6FhK23636//dkWm9lnWT/pmDryGL3Fz0cZCT2++7y4sG28nHZ
2oskN2QDvzIIt3fxLIvESR2/iLpZRHnIh9gGhQNtxCMSthZIZR3hY0y782Y7
2W+9kOzT392ZOOxwzrWYeNZ5K8aLbhoLM5hDjysCCZ/Rf/jZkj4LPTcS9ERO
gLO74xyuxAvQeaPQbPdfARZXf7+8l/DhUx8V7hzCfyxVi53pZG5RZrqOPSP8
2Nzh3fNiLAIdx3YqX9OOIjxdk/tlNbleavN9/btsfE89z2No8pHx+ETa8+c8
vNgdbD1/ARPHlw6sS99PQ2xzk6/2aS5CO0/2dQg5KC8KjHD6yIKRUOBX3sQC
a3u0QkgwC/naa29nu3NRoZ+fOo3kwMqwPbNDNRg4MfVTvnkimbvHnyg6igS4
80E2yXoNBw/np41RtYXYdNKhTox4VaHEr0jhMSEEisOSsxqE8JZ8dNoljXiy
Y8/4+UbiqQcz9YTG0ThYP/P3060CLLAuEX5vE2J3iS83yYyN8uJLncYHRLB0
cykxNRfB+raEg6A/HGrJIp7FDSpyEqUXRopxYCmTk+3C4eDcZWnecmMWrNOq
KhUdebiwLYk5w4mFrVMscjf1seHnJXPp2Ws2cjyW9DNOM3ChVmKd1gsyd0/9
uBbux0YCfUGceBbhrrGVq41+M2HuKcxWNqUixO3c20jCeeWy9ytzvLl44BTV
8cSLgzv/HbZVyWKhU+FRWoaQBk/vI+tOneVBbPW/A2/I+3c8LA/4eo2JWxIF
29MJ77xmPb8a8YaDl0WpDqm/2XilOtVYl3CRz+bwuz165HEqRXqFOB2KprfN
jTIpaBuIibtax0H/8x0eM8h8eNhw2OQXmQ8WusFU3yQmbvhN/NYgPKO5PX69
0g8GtoxkmBb+oOFGUuTQlho6mIeGTEdITuTVen+02MWD7KlfIeXJLPwrHL+y
jMmFrI7Pvk2OdMR/UDRqIF6fsCpVrKqBiSTDZKu8L+S+flpX6yhxMHurun7a
JTbx8Mca7uVMPMvQ33V3BRuTq8oz9f9ysdcFbw2JX+5uodsq+TJwumjd3C5z
JjoFzS0rmBzkH/RokBxgwLxm78/8y0z8Xq8y9/MgHaezn05sSKFAXCloShVZ
/574iaeNhVz07NLxWD2HD9O6XSZ6unzYC3idn+R4uFTy+GvzQuKhyXccezt5
yJbefLOM8DLF/LLH8eWEW9/6p79UIr464Lf3xCQfjbPuzbEhnku9YNTHIrms
mR68NP6AAFOL3tcb+JI+un1aV5L0k/nTxXun7iL+d/RRzxrC9Zvu25hH7yJe
NjhuVkn81aTF13zZRpKD9n2VrfkR8FGiL3lXTP7GrNN/OIeBbWdWXWAS3tj8
mEW1seYhb/8uh0X//z6GXUlh1ToGKqd3DByvocIwjKH1aycHJUiovPKXDfXO
Nc5pXiwcbtMRT17FwgUGb/ZvPyYcg2e2nE/kIHKbZ9v7CTq2nnSbd6uTBgnl
79naxK9P8n1ufVMQwGzD7Gc9xIduLD48xLsowL0yu4vmJPcHDmoMXCZemrt+
3aN7JOe07JPGqz7zEfjladOwtBB3O753PfQV4cfQoQOOr/gI/XCmV1JKCNaD
a//++8RCVt7sfTefCxG+lr798nXiwTWq+n254ag7/T7r4CYqPJ7fWyjKYEPZ
+032qukcTOsJCKj/xISRHCY+jxKOy3VoU+1n4nnH8Ril42xEfzS7Kb+TDRN9
t+sKcgykZ90K8fBnQmv13foGaTaqFfL/fNhKR/kH/eVjhEMdclZ7tI5SEBGw
ujjNiA2n7+ttD6hz4Vnj7WHfykbdx20fIw+xsDvPMkltKQ2qdUfbEjR5GOFK
yBlRmajb+jR5ymImVnyR9D1+jY3K9hIHpw0cbGWOJ6+IJfwbPeYe5M/CxqeD
paw7xN+sppdevkHDo3d/qL0iCkLdxOTlnDi4ojak7LOMBa5vvELCJzYyDdba
J/swMeX556MfZvHB3Z48tegsA1oLuj/I8WgY7H6g6cOkwzG+dM8gqdusj7dn
H+vjojXg+7Vf5ixEP51XPduMi/MfF62U/48OLzanroLkW8vFrdGnzjOxPlG1
IWQ/C5rCDQNHy9iIvLJgTzThoK6uTF1xMJE7+9PnH60sVIb6PwxNIp5aW9Tz
bAkbLt9634USzqX5TPI+tDAgJp6xPWIpB48OWpQ+PsfA4t8WiWc3MfFjldHW
Q/508M2m9XNDKBjnK3isfcVGe/TqQzeOEF85Mdv3TT3x589xRVlNPEhMNchv
YnNx4cGmXx8e8rDE6W5zgAsPpW23d49H8KD7gRZKJeeuNdWnUPYMHYMvWW/u
lfLxJq9/2CeWgZkKHhlT7/HweMlJxwcqAgzRa630tPn4V38iYt9VwrOSKs5e
MwSIfv/5Mp/Kw0fnB795T5io6Jy+/uNSPiyzXm81/0744IiBdfqtCKzx/ilW
IBWBdpUPMhv5dNT+YWirkDz3Pap8Ke8ZF5IWvJy6Yi4oqtyoh0102De5/+az
qFCaq1i+iqzzgmtA4U/iJyOJ3kFu+izEdGeHHCB1Wr1jp/yRIQaKypZPfAMH
OxX6P90R0LHScZyl5UTDtRdVT+5M4eB9zyf9c418HHjwQC6TR/rkQve7QVMB
9Ku1e1VJDtTlX/8yxCIcnam0pqBAAOlNml7GFD4UPMdavt8VoOy1of8MJREu
LZvz9zbx/jzFQt+Y6wJ4z+r7fo6837X85qNj4UL8arWhb3ER4iAl4F9ZWDii
ysN/04YoWLBo8KIs4dHD3LBnPOKlaxqO4B+PiZlSi4a/3uIiUvL64FEWE9/N
+W6ufSysdbuq2Uty/bPUxuque3Q4/O5oDZ7PhHbJ+uBzt1hoyLu9vH2QhsMq
0nfvHmfiZWPPIcpjCv7qfeFseMBC64d+VxXiabLH7FtsCF8qpjU9853Nglvf
EzHRSyouD+le0nzFxZOyO639KkwsrRafeyybgT/b1h36s5uNp6fihlIfs3Hd
gx7dT7xjSYHQxV+Thcn1Y9M+kLrV5zkFeNvSoGbW57jsLAUhhyPm7K5j43PG
x2SJr0x0XAzY8j6IjW122yedVjJRbCUnPJLPQ3iJuUfgFAZ+qaw21F1GQ8Bg
rWPnBjr2xjV5Nffz8KvFKE+ZwsWQQq5uZw0TbI+edqd2Dha/3iqvMUY8Oe/t
4vvkc484n+Qd1GRi8mcWp4HkXqdjQ8HU02x4Nt04bfiUhU23VcZ7ixjIbR72
0qSzcLpnsijIjguTICvBiVoWnA/Jqd0hOXbZZq3HFyqZa10GVwrI/QdVNql+
nc7A9qsyCR2jDAw4tr/4O0xDJyW2aeoOChpzajTqfNlwHhToZCtzEdFRzjUP
ILlhiz3TjxLvPGKbulGGi4D0l9Y3lvLgH9f7KbuR+LZFhfuYOg9VTwNVhzhM
9LpHf/vaRIPzx3d3HhJO/XWr8O3SP3RIf35cfm4DD252gtzuWj6chrzvjGTw
kBRu2N29hQ8b95HBiUI+nJnh88XJ503cLgrOM2JCwrl8zZUWHm7tdGqQ3s7D
1euxWZSECJLjr/M277wIq7fdX6ZL0hE0cumBYAsTSnu8c84TT61MvzN9Bpkf
atLyt2sCyFyTWpbas5kKS59+PbP1bGieXbh4xWFSDx1L5i4kPG595HYUJZkJ
Df3XCqcCGDA0vzCT+pl4ae+ncIPFdBgG9041LqHCe9rFGhtS51XoP5Qbz0fS
EdcrctIsiDaL6WT3kHV485s9FrMQajinLMFOgOOJi+zuuQlwzqfes2wxHw7f
zMM0HQXo3t2txygSYtl/+Y+Nt5HH6/sbQ0HyXizzc2gEC0Xn/FbtsxTC3iDL
XVuNeGHZGkWBajj88od3MzdQYESxnv3zEQtDpuPb5xIudl/cZXuZeMM2Q586
C3Eu6oVl591qGHhxxrVNbIQJYZjYWMNPJvZ+Z/TPJP3VUkzJqQgm9WpktWHN
FyYobwb8VFbS8NFG71zkZwaiitK6DbeRvnv8g+VezMTZWBOLkGWkXuE6OZ1w
yJ7HV8SOBzChqLu29IAkFVQZ3ljLQi7+8H51qYMBz0g/lzzCe13SNtWOL1lg
f1MYO1LIQpjy2oK9oSxUeL4+3B/OhIr/4YUzLFmwn9vh1MmjYuI8ZcPrlRQs
93Ty7yBzwl+iOpu5k/CHfpBh8EMWSnIURo7WMpBjsIb/9hcXj6NcB/kadMgm
vv1vwWkqtpya4p7+jAZjCe5ZL0MeGlRXXN/xiIMkjZfdlyWY+LxtqW0q8YKz
/LmlS1No2Czc0/PuHAuvA+4G3yT8HRJ5s/0y2bd5Jzagl/jywIFKkx9k/1yl
t6YPF9HhdqySW0jyha+aoiUZz0GWFmcikMwPxeaIDzNI3lh/MnQpaaWjrb7c
KkybjRVsbZNxHToWlD7s2XuIAbn9l7XGptLw3nNUxnIaBdHjvROvSlhIlNPL
2nSGAw3ldrV7+VzcmLk4uoPk3IavrzWaCO9uvXfpoLcWF5dqt1XdI/zdE8CJ
SiZ1rilp7CdLcmJvjsIin/tUfPfu1HpOuNLBL7FyXwUNHnHHDp6M5OKMjkO/
iSMfW18rSG4TcGHrkX6MeoyH449+Vbnr8bFO42vxLTJHLHTjPBOJT8UsumvY
Rny8WX1EtkKb+IXzSNZPUQT+WgSfH34ehu+t/t/dp9CwcFqqVZgRA2MteyfX
E7/abxeGhFIOvpzK8Pt0n4ZVCftPnHhNwYOVcSfvE57cOUXl+JZGMsdCNexq
zzFRKZphXzGLCcNijW+loXR8lMyuf0j4ynaD4FCcOw27HktliidRsKkhaNr/
fXXVR+f2Wh0+go4yUTaTiauwuvXWhY+NG2edUCB8Zb7u4+Cup3wwbzMVUwv4
mOLRKjz5m4v7sV5rvUm+F255J+WmKMQ9Mcd4SiEPfcphGvERJD/l884e1yKe
bCL01MkX4E8B+0lKjADiOX6XCyrDoP1ca+/K4ShkHnyqYHqKhRBDyyLlcZLz
bZppbbcZePVdLOdIDgcOlvn9jf6Erxmdj+eHMmFTys7NiGBipPjiCr1oGph2
uzwqljBgeuP1kB7hwt8vdUbU26ioeclPFl1jILRdsE1FnYKURu2J2zak7mr2
3pXuIB540H8W/x3xt+iS+9t0mfhGn3o88i4FMz0bREO1HIiPhRzf9ZaOZVWB
dzwO0bEkPOubWCSp3ykpu4Vk362y+i83LyK8d5GZZL2ciWmGdcI7FUzkCYIS
jlhQcWv+VPPD5LqR8+wke//P/2/cWFtmMzHaUTr36j4W3q8rq/maQO7bRVy0
O5bUz2x/4e00Gsa92+tfaVAx5Ye8WxONBmXPwZ+P6rgwXjqsfduFg2V9t9Yu
vsVAUIxLhqifeEBR2Xrn8zQ8HXJsqhljokiYPf+5FvHeNScXlDoyUOC5XVOR
w0K1/SufpmgmvH9fLq+xoqPx1M7bP0i+ZF6YX37MhoPnaebZNyqZ+Dt22mmh
BR16G35G7uHQsVrl3Y/3VSzI5QfsyC+kIcYqr9CC+Je9zZqjamwqZCNq9/Y0
R+Hrz8WC30dZ6PlT//CuDgeJaWc1L7pw8bPCQUaLzLHlWBF89yobJeduQ/0a
Bw/mryxX28XBys6a2NF+DvLuJPkFpjCQUNexNU+LigjPhpvmN3i4PFagZWVE
QwXH8GjqIi4GKXKjs2bzIbvMyJMnycVsHda6rHfEc4vNtP+W8eBe0t1BJxxw
vPab60g0Hcobi4t7z3JRPNjzcT2fgwA16s05wghYxGoF2mwMxeZjtRcfhlNR
YPU4hJFCh3O6ac3oGBtSrVH/qk9wsLpDuUzoTIPbkiifVdEU9CdMN+gtYGJR
SbnhDC7xbIvjh0rXkL4prJnMrWCg5dSYRMUMOtiG1RWzm1k4uW6zt7k4DWb0
AduNCqRvd1lZC+RZ6P+XWDDlEw91895MTxcycGZP48u/anzYyXfTBzIZaBy3
hVYkHyF/b3i7evNxfs9CaaYTF77LXb4abCJ+nDV7xYEyAXm9Wqz6Fh5OnL9Y
cpzMnZN15dNTSP6uioHK07MCDIQ7X1XaLcDyQf5AiWQYHM9HjfN3RmGKy5I1
v0j9nz97n/JxGhMKXkv0Rsm5XmndbR+qykFq35+NK57QsfYaV+ATzcASf6mW
dFJ3uVYDn/+1U7FKpC2aLqLjb9X1f9GvGVjGf3qItZyKn8bLFErmMrDWR1X4
JD0K1Zc/uC3cz8ANbf/xaHs27CtPvvZqYyB7sr2UQdYd9GJ9V9/PKEwquXvo
GnOw+1bxyG1pkgs7C0e3UmmYHOaorP1D1hOsc7lrLRPiPcyrvwKZuPv38807
VxgIU7cyCm8n15/y/AKTR8GsoaGqmMYoSChWzB0l80KskpIfRGEgWHjXvOMm
8cW0jOZeEwY4LcWxB1o5GP8v++LGRiqeF4oEs30p0O8UzA2fRcNL7xWXZs/n
otc4LVM7mfjpaROKWR8dxjmpHqJtbGSXT2yYqUnD5xuUlW2STNhGtdfxL9IR
vSC8vmkHHT9HY6NPkvv29DnzZt47BuwWtbsHbaLhxU5H45IBBo4EjV4dTSS+
sp17oWYhExGbwr+OlNJwT1Up9/B3Gr6YuUT++4+FzVPvyst0k321XHKgjnz+
r3O3Frddp6Bkj8upjcwoDDCzAxY8JVwvWSYGNhtZcyou1NzlYKO0bi2P9JfW
b/VPqfkswlV+a11vsXFOr0hXnXDaSuVZqpNGHGzbb/zRWZEBm+b37POzKSjw
GF/QQTzrj911yx2zqLDPcw/95cNBXfhl0X4DHvyYl60NphJuKXYJyyrm4Pfp
93VauVwc5Wy7WjqfgzVzVZaFi2goP7bERo7keXrEht/xhLebwrmMFk4Esq1s
lK8rB+O9klmvBoMCn4SY9qoYst8xwaH2umwINQeUhdVsUJr/3E6/RursjOCL
tjYFpRsjS/JKGODFqkRfFmOhI3Pk5LPnDJQ8vNfYYsEA82hU/Y8FNPhk7buf
YsxC5i2rc6JtVCyMpmTbTUTCfbpzTfEyJtyk5BtcpHig9zmb5KfR0XrN7VoR
4e9XGp9ToirpmPL3kfywFB+lcZf2WCfxoO+xnRFLeMH1YX3VgW08FHeu2Z6m
L0BXtISbWQgXyyI3SwpJHnqc3nHchsnEQfE5/bUTfHx3OTFe3MrHFo3xzs67
oXAYaJ+mLBmFCye2jTuSuRtr7Ez1ZzKwNNnv680qOqJavi2yq2TjvebCbyPH
6LjwOXiljzbpD50cieuWDNxa+LyVTTzX9KuCssZGOrzjz9R+DWRgim1LUd0n
CtrX59s+/kBH/maFfp+zURhys3v93zc6htLLFU6Lkb7L3HUq6AgDj69J38vf
xMAd1qCHzJUo7A1IC9IheeigWf5PlEqD33bvY65zSV2PznqVk81ErmDhpeZn
DJw8zXM9qM1EqLm1WLQTA9+uy7tHkFwOmhd7foYVBcpz5iXseBCF+W1th6pk
mTDZqK9fZs2AsI3+5KUD6QOWfmL4BB0XNyV9SI/iYM4eSWr0CSr0qHIDwXMp
SDiaFvuplopArU7m2woOvBj79BTs2LgzyL5FE9JxP2PjushppH5nuzEzflAx
X7b0YgmZH5vqVvlJm9LxLFpsHaeZBl6GtG9YJhMrs99Lryc8Xrur+Isk+VyX
vnT+wmQG1PcH/xnbQvbjJ+1bQh0Dsocq9OKiaFjw8yf17nUaxOnz8q4QT4yU
Vf175iIVpctvWXmtomP+i5JZG4zIPucYiHrdolDk6HSo6xQTwSdljLetZcN0
Yv1JxR0c+BZZ+If9ZGNOZ94e/kYWzH5GDeQtZUPvnFblyR8sVC0pNDnWyEYm
a+11Ljn3uWtlQm+yonBs5IxfDMlLgb3L5ulhFCRaXNaqVuRgrsn+9ZQ8LuzH
XOR1o9hw941iLTXl4FOGvl65CRe/V3RYNxD/1jmW1LpflYYVJZF325dysF5j
xGpOLwsrGWuHTZgRSMtPYj52Po9FcssEo3IUtFBejqbI0WAeOm56/TEL2GK4
ZyyUjUDF7C/viNfpxq32+dMTBZeWVd1SNgxIXDGb5kK8xerAwNUPbAZaC/n5
P37SMSn+admXDCo6dy1ckEE4PuVQsEBxjAL1aP7+yNORSHH8Fjf0hAGlQhl3
6etcsCMrwVhDx8zVIufG+1xMxKperj9Ax9Oj8YNs0m/dsbTZDTN5cD11qHgy
hI189dbhnWT9jmfMv2a38/H12aC3ixoXiamtybO3cyF1tXnOp11MbC50rJIr
48Niy16Ppzf5GH8sCGz7dQFWbhH3z/pFIs2oyv76XwaavJdJhCfQ0b2n+9kn
wkGr9lvPFzNho3Cp4IZeGQ0NrYnDd9bTsfuojI70cTruDe49QflMQXMN/UpZ
Lg3euRKKDmRO7P6kn7VoDQV2/bLrPfbT0aIrE2UiGwWmXsM0a5JffQXxmxf4
stD0bb3M6SA60mxsMz4W0uFw/rB6dX8kNlaEf7xE6tsztdCU0UnFQW+fmo1b
qLCqcFwjv4R4HJ9ZXvWOjpd8259BFxhIzQiRCHhMh5hmQ+Lay3RssR7Uol6L
gpsONeaEdxTmmR/TjDNlYE33E5cjxXR84ft9iiX1XnrnRM4nAR0BJV1fPEnd
vWr/8ir+JvFQr+bZzAtR2K43ohd0gIo7/+anL1bnQDNx7ru6aBaG51zZ5PqV
BrsUq9v9x0ndNosp3IilghW1PPNyOh1r5rn2PrpKw4CM2uakHirOSUsdUdJk
IifTZ7PcSzou0V/oDZJ9uxfMLgyXYsAyVjZSIZMFwfyjMy3JfU59dHqewmwa
VMUvN64keft9WbpOiBkTCyxrVfYVULCdy6x7mEFDibioOMAxCnkTv22myUfh
Ki/CyJP0bWPWpesZd1nQWi0TdPAuG8cGFhhYLGdDjtneG+fHRJXojcZKGxYc
yop5ZoosfKzI6Di7kI21Nl+Y6uR8TwsEd75wImE19dSUSS4HgzqDKSePRMH5
iir2knzUUnlyTraRgw0tZ3USBCycODd4afsSNnYwIuRPrSBcqCIqODCTjVlW
DR6hulTIT1GOO3SIjb4Z3CzxNiZk6l+9kQmNgJsM57nc8jOoqbx5/5luFGa+
6Jz/QZqKaAVuk/RfJg58uTkt+AsLknQF+88pFHwULrKtDovC6u63jqPOdIje
MCOExC/OjF+EhirhlnedIdU+dMy5ftVnopaCBrFZzEIw8VROm6a3h4L6C025
K70iMANunG+DdKT/FPx3u4kDyzXud1SO0/AxJ1Ra+JgDqq/zNCaPBsUXTFe/
UuJzgQtp9EEOKk6aengRPxl4ulj9H9mXvVv89Nt38jH6dsE+ib0cOGpG7hsm
c+nRyHip4ncGxBqz8iOc+Fil6HfnoSkfRaIXCYEJFxC+xUDGXC8SlkeOjp97
wED5Ebp+hS6Zo89+5dqP0qBn35zU2MlC/fjhZ35+pM7uzZny6C0NTdE/nzpP
p+PVvlznFwIK7vLFG4xO0rBpcf5e80A69rV/os4aiMLWsJMX+ubQocB8Fm7W
HImK/Wv0lpF6vGt52v21Dgvt+yrqdSTI9Xb6m2edJvsVYnpua3wkWBa5X7pH
WcjNXvP6BJ0K89Mtdb+7KNiyu3tjRzcD3GOeofHudPy5sqdh6lIGMhUzZNed
p+Pg4+VxtUZ0uLesXntsRxQWPtzi3bY+CqdNgtwPk3mgfSna4Sbp8z/zo69v
IN6VtL/f7x6ZC9cVXjpfDWPj8bHRlwvWU/DEV/RIyjgKPdWZbYmLqJDoqj1K
L2fj5ZLhTH8rFhIK0uOfCmiI9zWQ30D8+UJgv9HEESoMZ4/d9lxNx8CuN/fe
7KCBK15GKT1NhRpzX/S5JsIHUTm1VH86GEa3TkkQjlW+Gr50aykdBqtkLKa4
sfAkIeD9p0462oKpUxsbqJiUObNoXzkVayTtZ8V0MfAi7bFK2m4K6l44Ze48
RsPbwX7eSGskZEfvBp7qjoSx2KvrPVEM7Pcpl//oyoLsknOvzXazoWm9OPJF
IwtvLO6vCJNiope1eb5RPRNDH5a7BKQykW1zU9RO/PGu/udnq6bQ8WnSKeCJ
YiS+3/7865QB8T8Hp+/JLZFoafC50yTPRpHEooA5RzkoyF/iVTaXhcamsDdZ
6Szc1ChxsihhI3jvvb6EbBYU8xeZhGVTkHzN78KbMRaCCrjyiw4yse4D3xBn
IhAWVNk2j+6PqEvDz+/kRcJidK9kE+HNr0sZo23xTExVYG/2iWPBWiL5jWAX
BYNnm9Uv2UfB8dToqoOtNBROqyv/Tup7+/LPVy0/0nHT+E7KaX06ps9Vd/nt
RcHDFQfvcocIT2keGGVIU+Crv4Nm0noRH+1dpU6w6ZjYcevM9nMc3A5cMy9G
goYTsikF7Q4cvKG/K7uylIay1GV1O5250D+tXmFL1rupcX3pjxomtnSFf4IY
B4acYw0vFPhYvK1Lfc0EG79K3qe3B7GRsbde7XUhAyb1V8QZGnxo+ek8Hhvn
4fe04OkOgyGwecqOCI+PwPucl3VLFzEAyYNmzutosJF8tv34KRrmJd+s1dzO
QpbSxT1T66houZJ677AYDX+u+fXct6JB7sT0ZyE/orDLx+vyR+J5WtJqUfJ1
tP+xbN7hVL//Hw8lSkmKlNGQpIhCUnoqiijZu6E+DSFkO3s7RkZEGSUlJVlJ
kaZklNGwKxSVEUkRjd/9va5f/5zLFe9zv9/3/Xo+H4+ToJP9kClNzn0s+1LC
hgwBXDt9/EwCOFCuKH928BkfWu/eZ2UkR+Ja/ho6Y5kABtJ7wiP7BQi+Fu5q
M8aG1amEl56HojCWYVVX1MiDUtEcY69lZL4H9I/HHRZC9kQ3xYMqQGT+51sZ
sRGofuXANh0i69LWnvv9kAD0c8qKho84+P3y1fjjnxxcXGg3M2CKrNe/yOza
hAAdohGaDwnPd/zXHsD7KoDksMk6u5YouC2z7CzncbHru8KVm+kc/AvIiz9c
woNc5OCP7crR2KNQuPJeUiQeCotWLBvhIzfm9/7/fe6+0P3Ea7okH9nHQ7Zu
9BCg1/Kc1Rxyn3I/lLuSonkI3HykTY3M+XW9ZnvlWgE0ZeZmNAq4GGn0/lCt
GYHnReFbTJ6R694W0Tm5l/BClF6Utgcfnw5ljn3U48O1v8utiPw8g8Oen5vC
haXv0JjKRz7y+nm9GqIc/CicXv7nLAdKlxoXKgwRbuha5dxLzkdMP8xbyfnv
XS1iVKcShZman/cKIYR8b+hf2QkhFh3dLqVDfPPKptdVP1ZEoV37j2IdTYAq
+iCXo85GjmKrtPaJaLhneP25+pyN5lkJEy3eUThkLDYxPSkax97X9q3dHInf
ONDpURIJu/6jPYtfk/m9dXdpkmQU7l7i/3n8kgvVQH5LzOYomJrNHJD2EEK9
O6z8yz42pv3/ny+Zg0d4HaSH1h/deoN46nXj2U9M3wvx/K3xSmfRKNxcx2is
r+BiMGvXl4BeDhzmhwXQx/loqZz2ZNRTCJuA+16eR0nuRq9dOES8ZKebWMn+
BC5Gf/vs9nAW4ny8ezg/mAvBQMbwkkYmlETid4ySffjknskvK4vG+1ejv2k7
+XjUMPPUZR/C412emt4B5Lwn3H+0lh0DYUT0t0yPaARZ5i6teizEhFyQIHsk
Cqmr3lx3WxELt6JDFZW7o/HknfeKRLsoeNlkqdQS7zkzuogVfyAWijO9V/pq
xIKy4/g65zQqlJ/kZQ/bsXHupSVj4G0Eju9/au1DvEO4p/D+n7Wk14JkN4VJ
RCGh/fD7FVw+Zr5rj2tJ4WOlS1Ddhbd8KG1fL157mYuMsPVaP9P4eJoWZtPO
EiDK7OTNWZJcVCZRKtsOCDC0xNbEwJgD38rZTMZ+PpyaNpVnWkdCadqPMz8L
+AjSEB4/mS1AhZ5K3+0swv36w2Zh8sSXb00b+RbMg8hLxrHZD7k4Nq9m3GEZ
8X5ppz1aswXICHz1IXJ7BLw3ppTa5wig9ZWuv/8vH2r05eFzwzmw8/qFbQ0c
zHu8RPX0WTKPYvu17O8IIPDImeXsHoH7ErIl0aUCSN0R7criRuFp6eyRK6u5
KArUXXPXkYN3uktjRug81F4T8Q0kfaCkXTBhYBIJA922wn+ZfFjoPjiyfEck
/tYl3clp48HsRKVw/k8+ngXkWy1k8/Gq9OZt6cU8wkBm2baEJ70KryXODxVg
eevid1jIxeV+trhVnwADh6YUAymRWDDvipo24QhNly5GMeEzoyW6r1aNEO9J
1DGLFRPCe+DOHv3NJOfq5visvcSHC9ds1tlENlTDA5YtOslBeYzL7bOXIjB0
SfbQJX4krovWWYcejkJxruSDWWSus9Z8fxFO9lmkrTq5OUaIqNCm1RHEhw+r
6u0/2RSJH1NpogtInj70dC8wK2Xh+H9217ZJRWPkbMqTD6ZswhtHFFaTfgsJ
3FmXokVy6PX+ber1hNd3KMcFGUVisVDu+TTyfvWblnUvLY2E4qKQXfVHuDha
831/zrtIlD07JpfRGwH/7Hyvo3vZMBTv9vY38sFMl6a8y/5stM+z8FSW4mL6
33n30klvFz5tbV53JxIj5TezNgRy8WTPbad9xRy8+MSaPpNPOKJnZQxzlRDO
aTd+e66JQNLMoysuHRNgJMX57lUNLvp0PkfqzhHiAF+WOb6OzN3qDzdPmzMR
u+Q894GPAKFSO+8b7o+GhXZIYnk/D+9iP/xrFInGjr2eU72L+eA2HJrbqBiD
Fb43Hg31RMFp5MOSgm1CyI0n1/j7RMGg4g71a/ZpXNpHNWkbj0KH6Yru75WR
aPm29IaykhDf3g1pl46fxoKlG5bMqTwN3sl1tC0TFBgek8hRfM7CmcuH2ztt
I9A44Nb2ZpQHyzNnRJuv8kFZsMHXJDgS2pvvulQM8MCeJRQ9V8eDnW3Orqsq
fHSVVnSOkudU50897biQj7sLAp89JDkMMfvQm8Ec9I2KRg1/4ENZOuJBVTsb
R72xwTuOB8n+tBfJT4TY1mGxSuIV6YN2rx+vVATgHIifOU+cjcc7D8j3siMh
ebpjyPUZFzoyVPkn07gw7J9fwEqPgNgddaxey4dP9Y+BrbkC9Dd97tyyVIAk
cVXzhE188LzCytxH2KAFsJ750zm4+8q9Kp3kya9w6VdFBgKspnen1z0kfpQ2
vvSJuQAns+ltT7ojcWHgcbslWffqOcvHxOrZmLOLv0V3AQ9Ff9T3LloThbzk
NO0Q4oVGccuSW4mHfI5WXDJ0S4iKo489X4TyYHJerWk74dIuR97Ta8TjtU4a
/BVu5ELuAHtQghWB8WlPnnS18HFyav0QQ4f0P6PTONpFgJahv63ne4XoCxZJ
3BUgAGMH4/1a0rMiA5sp5sd44N0M0OL5RuBW1J1BaS4HhwdW7TBdykfMF9g6
lLHg/G5spG8JB9lJajYT0hFwDuj0MB4XYnxil/IQOQcznx2PjFaIBM0xk/1Q
OQJmDh9nXSB8KyfzpfN5fgT+U7wfXLWO5IeIebD1I7I+K4fv/71lwvPX0vdb
HIiHRwbXq+mxoH2n6vYMeiSmy8sZ2B6Lwr/ke30zpiJQ8Nl8i9CerH/lmrXb
IiIRnSp2ZVguEg/gVWOUwIHMxY0tPRqRyM67uc51TICZ954s/LCW5EWDhWbs
pwCYpr2V+H2FhSOD+5fPmcdBW0X3lmWviWfm+Q/vXkk42FUhXqOPg0Sexfsa
aw4Y1x1Pnc7mocF33cLQrAhwxQ9oKJYIwFRqLL77mY/XnjUjcgc4eHBiOGMn
8Y/njW7uB69xEH7bveXRSTrKRY4MjN/lQ2P6s2PbrkVBM+Wd54nNPBQc/Ryi
pBOFU8Mm9odO8XDwVdfwC8Lp/v4domJfI+F+wPJYkXkEdhbQen38ImGWpy1O
vx0D5ljMSSX7KCSt3GTTXCBE6gsV8QvFEVhc1autoHkaW1JZrR6/Y2C4e9Hx
DGcKtm6d/cLVnIU93eMnHGoEcK9+XTcsz0PU2eYLfHJ+DulvO6BEzkP8VICz
LfGK//q+CeTFyP0e/c+s/igP+s9TR46FcXDQxqBvFZmnlR0SC89ZEs8xe2PV
9ZmNB0YW4w+O86Hz62z0J8LzRv2v1jdWcWGxTeqm90YhJPZ4vC+e4EJ2xLTi
x3k+tpf5+HZ5s3Byk37QBcLtAV+lFp9X5KLjQ8XB/B0cpKrWV6xbHAFpw2X3
FtjwsKG25ZipvACWantlZMnP/6nfHaYZzsMv+zjVLG+Sn1ORS23WcKAv6XPn
6XzCa2Na5fPL+LDlpC+asVoACdv2+PJW4lEpLu9ztkUiNOfM765qNsQ+7rb5
Z8lG8NO5Gmo3uHD66XxBLzoSHcVjd/mkdxwrzsltIO9TpFoouk5NCBfbnx9s
5Hhw4SO+jzyn/fd/98eEkf5bM9P+rBMHOX0uTesmBVC2oVc6beFjYsrrfLIV
G53202u2NfBx+P7zC9mESzY66i5zJrllJav3fWw78cxToy+OdHMhnOG8ffyt
AEeEilxBExsxOntG0y/yUPj+QQK6mOjVC5NXrGGjX7h/owST8LS+9vrfx4Wg
tJm216hHonrTpZDnfkJIHjwpnnmU5JvPuR574u+PBSnZAT8EOL9vj6psghDr
FoUYF5vx4XtU029wLhPfzzMO/M0ieUSvLOIvZyI+fWFx7gchLC+oPs7IjiR+
0uxgTnz6hUbMXx3i55Zq2cKB20IE31wnjKQSnjwlIp1exYb6/puaqXQhzLud
Iz5KCmB0/sIWfzFyPs421r4vD8Z2cafBwj4mFLovP++UZUORqVEVuDoCLMX8
rvdnhKgSDPxK2s9B9/aD/w0Os3G/8p70wjouvozqbtm+NAIRR6vktUlPVw9s
eVTixUdN6tCvfSlsfPHIcJH6LICxbvo3KXIO8mc0+pXsoGKdw9XxwLl8HHGf
TxmciETwTJsLWWe5eHrY7UffcTJPtrNjI19z0SujP8w/R+ayqiODJxYJ05zv
kWOuhE+cy2+IMISo0Ti3ZXFjNBpX74v9Qvp9Lvt9+IurEUjbqtQ03yICH9hb
RT2IF3pqbt5z1ZC8bny71HIWBZk3XGldM1l4LRF+dYorwB/eapucXC6SFETd
jn3gQTlFEFidJsTdRvG3J+fyMJHdGxZykYtqcU19NSkevGqL1ywkuS3R7fbD
yooHsVkNudbT+LjuvDT+NOHFTV2bY63W8EmvCVsSTNgournUnX+YixRlaPK+
RSDvWfSKG9FcXNvGuDudHNoVfvSet8tI/0aes7POFsJqu3h6XzUHi/xXC858
YYPfLvxwtlWADW4rdjd85AKiX3841vORX1D/NtqJj8CX51XdFXkIvvhgcoTk
qfL9ufd6p3Ggd2TTFJv0h36iVPvfcDLfy36GOLwjfeGTq7D+Ih8DKnqb/UaF
oLaaKWwKZONGru6uE3OI978KeTc3jIsh/ep5DMJRRUVd/NqSCFx98Pq3vi4P
w1WWqbrvI+Arvne3H1mPk972scTv5P7aoqR/byI59VrjginJH6dZYVvVCc/6
DZ89sZRwYzNVf5vUWxasvEt5nZF8DJupp+hJEO+a7ajwIp2P5S0D1m2SPJys
XLLqyAUuHCiZicNJhJ8KDT63h7Fxamu3E82DcMPvxh/Np5ionfTdG3yBjbkV
qse6SJ8ff+f3Qmm5EJUNx5S9O4RwqKgPnCLc8+CrVV2zFDmHrnFRM84J8POu
1j2LVAG2hCWVMUyJP/6uDB6ezYdm+y17OWcG6hQn5lruiUT89EVNHzUZuOp4
vbksWYhoiW4/H/NInLd8x3v6gnD6U/22GTMjkP70onEayY+Co8uOuOgIYZ3m
Tks/zMYGkWGh/BohWuu+iv86y0ds4UXTnYMsdKesXH+iKgTOFk9ncKKZyHTt
3SOfxIKW2ir7Bc0CZNtXp0ZaCnG1Sk9vnhIHWQHBe/88IPe5a0Fp+n4ufmcE
hTa/E+DVjJbDx0UFSKu9sbhGk48vfVd07+uzwWreN2vbRQHOVRiNNn9lQ7N6
RVjmOgpObTijlF7Og8E/uYAtlyIhdTHMf6E5F56nFd7emhkJo02ylENCLuZ/
NfyUsiUKsi6ust3xQuwvV/v44xMfDTf1je1mkefp4TXN5GA06s/Fdp70Jj4t
pR0nvYo8B5lBf/uFETiiMpT3ZWYMlvpclrvdEQ1JM8/RLX7h6Dl1k0MRMtFa
UhFU/ZuPHwdHO0TfcPD20d5vfWRfC2wWTjOXEiLOsbvxO52LndEs4fp3HHDP
/zKVdeXCr4T7TITM1+4XJ0MekfzJcVtsM+HFQ2Tm2N/WJWy82enhG1xAcvmR
a1jvOxayLtdK5l/gYE/fw7onRyKwr4J//dRtDtRHF7s8f8PDqo/Rf9tPM7G0
880/0QVCzBub+ypJlfBYu0zwtw3E6xPmzjR2IBzmG/7yv7lceEEjYetW0g/J
eZfMCK9uf3tHSv8/LoLmLKJXFrKgFmlHcbzORtfccQ2Ff1xol3R/nP6HB77B
vqxvFnxcPdFnOEz26ce/NdO3Oghxjf7No6iehX661NREIgtvRAs3ZS7g4uyq
XuVlhC8/9G92PaYYgQQVyf4HpAeNPvv8vW1DOIufy7VicLFrt/ITWy0uNjPn
MxIquJitUqi/exYbkW3P9yfrCLCtuabf05WHErq5ZecEE0kzTvoZiPGRl2ZP
mRVM+G9040F90ssrE3uMjyQTH+E0RnwhfW/dv4v2T1yAPurURd8XLCjfOp3v
QOY87cu2kItrGVh8VMkgjfTzt10mNg25fNxJs3l2NjEC36vbKgqJjySHJf2Q
pETAU2U9w3MfHxWuMu0txM8nygtW1w7ywUm/fJOeGwGnm8cijM7y8MAq9EzF
Oxqim8sbdmQJ0Vwq3fG/z+GOd73dNi4iBG1mgOKTWCFWFozRB+YIcHiLj4n5
egHMwx6fOhETga9NhewbGYSzzl54OyOLhcCFyZd/R0TA7t0KKyuSf+HT7E2d
77KQOs/LsccrDA6f+lbH1zLQpCneWZLIxL/tpwSvyPWM9m4Ysygn+eya1WSW
wYbXWI/c24NsSL519a8ScFD/zFV2ibsAZ/SDukRJfh3vd/jYXsTD6cYJx4Fg
FhZ1v6obkBEgjCcqOY/Khulejo7frDBEqZSe8lrCg/th0QNWhMdLU28jqIgD
d93LnzIthCg/8yF33SQHu3XrEEs8+Y/16bgtJG/ralYqKEjykS6S1xOkHoEp
bw9tFj8KGrkrave3C+FedeWS71oBLpp+z50k/rNR7d88dYtoxMSYHpVbFo2G
Lfb/tNrC8Ft5GU2b8MTtC6/9ngfysat8zVlH0g+3cyv3jnZxEfz31cpIsl8Z
wduc549ycDGcIpGgQviaJnp7OJeDJ18ZUoPybPz0KcjcZsHFEfkBgVw7OY8R
1KaKZBa+3nh7bK8e6cHTtzR6wlg47DEupvKejdZDeulX2gS4vU1roVwPGy/d
lntMt+Nh9Eeu8b5/DAS89s3PIT4UbXogdDaX+J7suv9OsFgwHPrEtSeepr1j
0NfUkIPFrdZpImSuL4lVP0/aS3quNWHBwQKy/t9JGxdvYWF27LVoHxtyHtsX
Lhvcw4X3i6G4LjoPx+PUnC7d5UFqX117B3l120LfoHo3Atjhu69JkwXHmd0Z
BctZSNIuGz1G9mOdmblo5gwhFrB+dZjzBIjN8T+qsIaLu4HGj45WCiBDLfms
vYhcPzx//gofDjzZ7qVMUy7++QwXWa5l4aY3ZM1u8HHtwwGnl8RXRfNTzKpl
mHjxTvFPFpWHPK2LJQ4fBfDhik0tj+DBqKnjjrkqFwEBxfMmUzm45ygaJBXG
x57krQfTVVlQ2OcaY3yci0NOShK/vGiwuyoh9+0n8QXzwZc9xLv3NT1x6hWP
AEdpbOXjnAjcmUyUVukQIPlvgrt4MQ9BvwTJeav5SDVVbn+5g4+P72vevFkQ
AeGl6e6Fy8n7pyev2cug4ulEquuaPxEQCDNdShPCUVcUTj1zMgIpA5bPdnZE
YL2O3art9nzY/RF8ivmPj5x3bS9DmwRYoGn0ImleBGRbnC6Hf2FCYDamLD0k
QM2cduHVVi4uBVxcfoXJwpcU+1878sKxMXGTlGU2HX7W7QEmigxszPRwLj7D
xwLe6qtL10RgVL7oX4cqGwkif+K9RNlY7ZnMVylmY90LV5p6G+Ekm19iG+YR
L/2Rj5cGJFevq56tKWdiZ9Sk7AMuH9GytEfseWzYeTxx2SAfAnWLXJHVbC6U
7eYd/2klhLK4y4ssNQ6GPIRK4iSvFt1qP/M3gANfs+o1na+F+BjudKiyS4Br
GXGrf63nYah2ddDXvQLMDrw9frk4EpQlRn7+5Dor4ssVlQz5mFSROdm7XYBb
+kvnt8RGwaomzEz6RBRG/KY1ZJiG4VTA83vGawg/8P2uBA7ycPhb5cGvhM/0
9LclTh7gQu72XP1fXwTYrX4mbdMxDp6X66cY/kc8AnWyZiIcVFNVAi1iWVA5
cFHMsIaDFbPlveaS+TsxTfbBBzkWch7bNHrc5cJu+9TQsgUszL0/baPPMjbG
j5jxUncLwP+hIHg7l43kpFt+So1cfKYndu7/TMcm0a7v2V/J/kk3GHT3sdD2
/MZEehPxmFUi4Xa7+fgqW6L59BgbHmcy6eIreBBtn2H8gfTtYRuTuHCSG30r
QpXa65gwi9yYKEXOI/tXUs/DTA4+f2uNmjuNhxan1tulq3k48LZKJ8uEh8lz
axxOKkWg9OmA7fJ4JuJSG+cH3GbC8ntpynZbDrSUDh20PhEBP/k9NS+H+QiO
qexakM+BXJ+L8Ja+ADGF+0rab3Iwrv59pSCfjcbkI1+C6jko/7LGfs4eJu4U
6WWVqPFhHv/N+/5GLmjXVXUZlnR8nKPUKfxJ+FSyxDzMljyPA40zp/5ysfpb
ocV8kh92L1X7/i7hIPMINSRwgAfD3ye3STOY0NexP/hkkAPmV3a20wUKDv69
9HJfJgtRSPgUksiD6fGajyGhAuIbJZ1fpSPQlf62ItZIADnR9ekKc3hwaL3y
ey+Z988u1y5syOYh/vWJJV8YAqjL7Av4VcBFvW2xbO4WCn5xWRZ7XCPg/8xt
4Pu/ELz5MLd+FvHMvLtcP0mNCLTFvRn9eYEHGVnJb27pPOSc2PqJoiTAev3P
NiV84kEa4QHTNjKx0TIy7JOjAN+ODsw5MY2LNsP2rZHbWZB9tUXOLZeCvblb
GjRKaCijjR90m6Ji+7SXj+dP8BBm2Zp66jzhRAf5r8EFLEgFrDW1u8zC3fNm
T76+Z+Fh4MtbEQ58bIgonogl971gYuyvxQMu8k7EjTfMIHNnNtq5iFznRv7M
hne5LBT/9hSNehgE5z++eSeHOag5m3vvP5JLI4v3WdEus1Evua9d64MAHqu/
zLP+Sbymp/N+BvH3wcr3AaWyAgQObau8R3jq7eev1bOYfOjeoihw+oSIWL7k
kEl1BPJ3Xp6pQfJf3bdOXbOdjyvSLpPTWyPh0dgfc78kEmtidq+XKwrFHt2/
w61b6djYFdWy7DAPHd9M/0i5svBhf+nBo4Qnm5fYvxohz+u47YT8q7dsePst
Evxv/SdYo2KXXdioDnKrj5ZgoUV7yQwpcw4i7i9J3FzOwSdZEccfKUyc9pbg
aWzhQmbbPCvdPCZOecxqNTzKwttq3/yZ98n5tXbii5Ne4riu0aLt5sJZZ482
v5UG/o4bSg7EX2kFawWFuwnvJxSo6asyMUv6b37eUx7y1aN3Gp5l4VmSwC6P
+DD9c4JLvBkX7sWquZr2bMSl+S35dZAJ7bgOH/EUFiKSV/ufnCSelZDvPI3J
ReOTz1Ph57lw0YlQeV9L5r2y8Lo7SwCRSWtxvxEGKMauV7WsmThz9kTrkjE2
viql84LqBTjEoi4/5sbHlPz3qXgDDoJXqcuK55AeYmYYTjPhQOpB2asZP1iY
zPJ+utaNAx93O/FjzXTs21ak63KR+GDCrJy4axzUbrOZv/4/Kj46xyvt8eNC
RbAhvO4BH5cct6xdGkyel5fv7P/1ucMUv805jY2BjMtSUft4uOpiXThxhwHv
s5LbD/qTucv46L/rURh6j2s311mw4PBPvb9HgvTZqGZm/mc+Dojdm0sPFuCS
K12tIJNwgpvyLVkvwq89JzddeMMFZ05aRCOZw9VR708rE958MKIT3mjMhViv
+UTU4nDw9OcUiJUJMCcr3rxxfhAu5Bose0fy8rlX5sZFFAFmlF1ebDhC/Hdf
tv/uTi5yf0YPGPvzURnlP/3eNAHSVWUTa+UZWL7PKVS7io+7d6tmzV7LwaSb
2Arbf0wwivX/Gw+lwmi7pqTDX8Kx85KkS++Fg08G2v4AD4eSJiipEmR+TbKn
moxJP+7OvWa2k/Dp0xSjoDksXO63Er/XyMO7tv+mX5TjYSwq/F/XDi66d97K
682lIzZ99+29xHv8+7o7uvaw4Jsj03WPHgSlO97fPx3iwJURTKkUjcDm9S/y
BtewQX+gzwjZKED314anZUw2slZtrJ0KJ7zeFl91ZA8fsudWX0xNJ/zzMbty
wx0eHjUm08eIN48cvC0iNCMcsL/iaMUJLtwc5SQHyXMYENCV7y+LxMPO5Icm
YpHI8R0/EuAZio16j3K1amlw8dos91GJB8Heq4fiRph42nWmSpNcf3bMX8ur
xMMPXX5pF3aOjU0LQurKrFnY/vNzW704G6pv6FeKnzIR1baerzOTg4+Xr1VL
B3KQN6D4+rcrE/7S0y37fnMQuMfn58ow0iM/7omESLJQWP8063sQH3WX1/+k
v2RC6o6i2YZZxJOT3WqM7Wgo2P3aNVBagObVZ2OTp7EwpHry7MU0Bg5+vO4U
xeJhvgy3IIXwoMUFh98R1lz8mVWlOD6dixcLByd3/mOh+a2iXbQm8cQApY3f
j7EwTaNnQWwOG1fN1i+zBReBnMt3um25uJ7Goh6O4cJ3Ro/SN+IT0VGy5+Vd
GXj0pIRtpsLElfjXtZVP2Sg4/mlgIVOA+uVmbw/J8HFMsHah2z82NjAkVyke
5MNDQcl8hhQHzuniO2rJnMfOVS2mL+fgyp/0mVnfaHh07/qHAnce5v4XmvGH
7PfrTT/VCo5T0HRcqWJoLZn7Yx6Fg+R5GDhcO5OqQ/Lj/FLrlL/keTezpvu7
s3HFS+yLETlXxbIXn1MWMfBb/XNN7AYOokYdRfNrQlFa9V+FogoL139tSGp4
zEWUu6GK7EU+lp5oW7V7tQDn7RzuryE8OHp/iZ7bEuIBRystK7lcmM98mm1b
ycWt66Fbgi/xcaPElwFJ4s06jq8nz4ahrOTemawT5L6bubaFRwOQeT5tgeoM
AcpFRB5pqgrQbmw80nCOi/8adWLHaVzYtCapiinz8fOprd9ykiOlZjY3kk/R
4T8n4WsNgw9xY9cl6k/YWLGR6vO2g5wP11Om9CYqdmVO03t6l4pDU/5jyqfD
YGaWf6ldlofWaTf3S5Dr2Kzc0F4yi4VzKVoyOvIs7NWSSMRlJva8XpF/N4Z4
mtHaSxeaSY5WFmTJk/X7yfpJJ4jQMbeud3a2Kg+7s64esVrGwisJuTcVpO9m
buXe61HmwOrv9HrJAgEYqdqnDb6T/jCzkHL+wId/o6XSO3M25CXc2V0rI3BX
ap5XHOFB2dWHueFbODh6rl5ppRPp15F/D2YS//x9ZVJDTTICZ3pW0MuIF26y
Cn0pspUPe5PZH7qqhOh7vWXE97oQ0/ObT+i3hUDU0L8k8AMVyzOX3HoWz0Xc
96FfE4pMGGznRPoocHAwd6ba7GA+1snLPJGWIXw9W8njRBoTWyk3y18cYoG+
IFnvijET3wMcsljEw8o3KkXd72XjE619e8EjBt7reL7KoRIeC30SwZVm4u02
3ypvGyZmxIiuqPhAuHZi1hynPwxk23/MNeBysOOVQcwsKyr0PcbtHofwYbpc
Y4fSCSYU21l6o2Z0LN5hNMNgnIsfBzeHqbszIakcONL1hAOzzmP7T9A5yJmz
uLXGiQVK30QBO4mBftGjhYwZLCz1bNL1kGAj6PiZoc33SC+PPI40ID8XqqBZ
1i3NRVhA0NzqRD4a9Bvps9aT/dqnMdHFYWB8+b0d8SfYiN+ft6C/mw/VIuk6
BxrhiAXU+r9xbMzV07xdUE9yds++tsJUNpx5sc4LepgQmXcv7XsRG0Wh+hoq
S6m4cDX/vsRrLkpXqEUHN7OR2HLXfrN7GHad+9W45QoHjpX8TZldxDMlbteN
XeVAR7NrtwOPjRzdtP4ZnaSv12etDORzsX55abZ/Bw1jMf8tci9l44hH0vLm
ymCcbssx+VTFJD7uGp+qx8XGIP/cESk+pAt52d4xfLyekyMSSTw29ODjf6vI
c5I59zZ03mcOLLYaGlTqc2Hy0+QGpPk4tznstgTxyb4tSip78kIRHHSxWPIF
H4bXd9WIzPRFz9rJfBVyHgqfDc5oJK+f7xkfP0B4LfRcWEHUUw5G9O30TMjz
keFaXDlHzp3EmZe1+no0VD3ZOaP6Jw//dE3qSsn5zmwPO7mf8Px/B+u+hRA/
Lbo+6+Zrdyqmwj/KyEuEYEAqmmFN52KdWqusvAYflhnFbo2ZTLQtW6F57w4T
v5Rfbq28RHrUYu1DZXHCzdf1mavtuYhpVLmoFMvBZgOxo41lVJxhqcyyJjzk
5XBtQe9zJi5MjdYM3wjG/rRPPnpkv7Y5flXWnCWAn7Ho6dEIFhQdf80wJOtO
6eSHvX7HgndAUsFpqgD77W9e1lrHg6ujtPZVOhsLbqZNZxJekvs8o3HluADv
DX4u3BYngLfG8bdXyDw339n6p6iZh4L/wqXmyQixTtrMcPavCBRGHh9RsAhB
VMrphNh/FFxzfNBYK8rFYEWnRGIsHTpPJHaZnmEjSSe9cKKD8GeySJI/Wdd7
5tsFO94wYEz5ryzuIRObWvNXFYUwoDGR4dxCesYzI9ikzpGNVfO3bCptoiO0
3ubp6V9s7Ksz8SzdyUBAfcjyA/oMhN2MN5C24mGVGSVM3JaOzWfebiyeInyY
FxmtYE2By+392aWdPDzfMHedWi4DUZSrjW9+UpEljIuhenIRkkiNyJNlYNBV
jrlIn4OipT0fDv1g42ziyd+riknPFTo0f2inY/s/UfXiC0yUhQYMifqwsNax
xn4j+f6hwhvH9dZzsGti4dLA0xwYXViUOXuMhweSxS0caRqmIh9cWPuGjmvx
9/ceJXmTvKqnYc42Pu5lCIKKP3OhsSKsXE6dDZG/yQKqIQ9CuVtfGlaRfirT
4tuSnl0vEvrs3g42QgPMprcZh2O3OX18xi4uFq7+OqFtQXj9wr+UCvUQdIv2
qF0h/sbSU1cUmPNQ1HJNtEORg/Xrti4dIjnhfb/0oirh/i8XMpJcxsi8iDYz
/bKoeFD4J2doExsHHP9pul4IROyprdOFPkzYlTrus8jmQNd417+oYB6sbzOi
n33lIWteflGLMg9dC6z86R/YaH3CDFtFvDXVlm0VSTwy98Sut7kUHjLcep9/
1+Sg22r6bgyF4FjCbG6oNh+Bmcm7XN554+7yOhmHTzxoLD7uqFrDw96Z+02P
mXFwqC3Z+/J8DhaxKqr1P3LRlGhDi3rEQ0Ja6ezQ2VTIm14WvX+KB17QqrcK
q1g4tY+qumE7E6vnNveLD9KwyKFCV0uXCtclMems7YF4Yk1bubafg1U7GuZJ
JPBw1NT9takRE8uetlTec2HCfv+MoIBJGgyTaq50ULhwKDMcbnjBgcdVHSez
BRx8TeiLr46iwHS/8+5vM7jo2pEWm0snuZ/ENld6EIIJe9XVxkps7Lx85PzY
IT7+vrnsuFKdhYHEL7Xv0ng4lvgmWYfKwv4r2ltKX/FR/q7mzLwQLvT+sx/V
rGLhdMiCuA8fOfBdLlzZoStAyKOVXxmKAmR9VNi41YyN9G0rj2334yGQpXrT
kPjxuJ7NHQPCoQt2vvjcWx4MnxyZvttKFHT+MnEb8yVedn8s4sd7Kv5ru3G9
WpINz1s8w6NGPCyUW9bR8peJg+evUquC6TD/eWMPnXDYLxWvFlEf4ofxEqfN
Q1nQ/xkjb0vWFTVdWSKil4Zciaiq2lNs8DpDbb8fo+Nr6su7ui9oSC7e/uVT
MXke2Z5arRep8LVIsr7ty4ZY/D/3epdwaKiP1u8k77u5x01MjEfHCm8Z24XX
KJDJTbzi3cLBGesdt8+TnHyYsCDcLZvM+Xe8Hz7OBjtXeXypDMm1TeVijDV0
zC93+qmuz4Sz9bjqhWYm5O3fPjt6g4324Grxtxls0A7Ju7hIcLDxg6XoBeLD
N1pPvrr9jQJz3hL5JNJ7n+dUvvEsIz7+Wu2DD9kPXdOB4WrCpxaP04vLK1lY
pr2Rc/4S4Vw1gUJDPgu0w6/myozRMWNAt7mpnYU6lSaurXUocl47zftSxoHi
3XmsUxUsjD49PV5cEQhPSaVSy1g2PDzeWaff4CI5ueVaGPlaJ+SbWSzJOdY5
U7vBN0y8WHMiemg/B79/my/d50dB5s7dEXlkXZZVtBTr46dQ3GRvMk2WidBT
7mJ9shy0WNqKDXRwEWpD0TfbzQNUr9+y53CRvSy/4A2Z+9K7ZgY7Ce+pBJ4w
fqND/CDgkP+yT1zslVn7eOIR6fe+pz2xTiE4MTDQuS+RB69EvFFJ9kc39rp4
OPMwMbTl+zcFHvZVKMlOy2Rjqrd/eN0RNhbKMpgSdlzE9nhemjTjQU0+xbBv
IhzJHrHUa9+50N/VHSW1j4k716dUPncSHlrtfn2mMx3ZzN8p1mTuSt8cF5gJ
/HHj+SvHdkcOtjSs6en8wcWV75LqimUMqN88Hig/xkClbk1Vwx4qjp1cfH/j
Vw4mRuwelmznYKHRWL1uChu1Le8oR6nhmGY1eT3jFAeiMs90uOQc2B3caHHv
QSg8fPNpk5dZqD14xs6ScMDX3yP99xqZ+OlxSKz9L/GO687WL5aysDt771g4
6dvWe/Vhdwkfnbjx3fXYfBY+q08pLCH53Vt0uezNKT6ucDq/jebxETXruk9r
AAtya6WCx2fxQJVaNqabKYDvNwP/v2cECNrMnGJrBWPL9gy1pUbhOFwSZrO4
lY0PeYu6r6lTEKwruquXwoJUa8FPWcLvyRUy6w8FMHE9fyw+R4mGkD/1x6r+
0GFjJdjucoAGgWeNz8NBJlQf3Vy9ewML3AFObPFvKrynRR8+8YmFmPsmagtZ
NDRxHtpaHaNC0X/R9HkLiN/fdzfU/BSOkGu29urviZf1MlOqvcLwKrS3f00K
F+n1wl3VB2loLc/xKtsfjoQHL0cGwMGCN6+vb7lPgfvjbn6ZLJm7QzvrR1tY
eN3/dtl4Dx2Dl2/o5wtooPbr6TXHMaByNNWFs4PkBE9q2ROSc37Wc+ZLkT4J
XSxcn01lw2BdU5Z2Nhf1NUmejS3haGb/olmm0eBwXG6i4wgLs57cKXo6yoXW
RkXVDjI/SVtUFi8m3qgalNL5j+RpveWOLjFy36PKtuddYmlYlvr+jdkpFrRl
sjcW7AnG9ON72tvUOYjXP7P21DoWXt69XnxA+hTiv8Zb6BJOmDJOuFE8m/h4
Ge3O+78s5JSquRqLsfDrotk5RyvC9baLF0wjnmmbMORmtz0caxxOzp4yYGFw
ectjqSQ/eP+qOl12koE3Y/uuWtFIjp9i3f6wmVxPND310mUuKi2aBjf2cXDX
d01dTjFZd/hPqZrNbNw4v+LmQBkb2iXKy2fZcNH98PBnTSc2TCkyc8e/BKMt
p+Ko7gSXcLt+X/pYAFw2fJw1Vk3ut2V5yctjXDxrNudrT7Hg7PfthhHJkyjR
4wtFajlYmWxAmfOSC179qHydSDhq25NsrodxURvWN9mtwcBHm2EZn7UMuGh7
3pArpmN0kPrwnDUVrEOLXAOf+IFdGcapJ7mw44hB0jH7//07qOlPpigDGx80
bluzgQGrh5IeCqnh6AlOCvNw58BkSfvm54VsRFhM/olZTDzHoVaxLCkMhzYv
7TbvYKNmQ5DErc8M7J1neGXaxTCsV/wzHLqeheUOc2g/Z/OwNWSxZDKbif3T
uuXaD3MRbcqtjyVcWmEkGvCDwcOkR4n8A9KjyzMyJ28eYuKSwFFW7jQbPyWk
Ii/m87Dz6/3XNaZ8DCe1Dj7PYmLBpYPFele5EDHQisv/yEfC+Jxi23d83N3Q
UyFxKQjd60xmzNgXBkv1Xy/qCbcoSNgdfHg0DL9crdLNPjOhYzsVGjLEQYWP
6AvlWgY6DT7J5tRScOzYD836chrenJ4x7eRuKt7bFdl/dWPC4nqJEof434u2
zAvf5KhYlvTLxsLxf+doYbRCHOk1Tm3GxHQK7K/f+ibuw0Fm0if/mmVh+PRv
YmehOQvh4ydWjvFDcSQvorJlgINZMV/uWRpRMb5nuHP1ojC4nD31JI3k6868
HnaIZDj8lThxK2iED3asD5+zjQU9TZvGeTk0OP/JEO8bpuKIlOR///tcK21j
tYtVIAM6OoOic4QshBZsDTjmzcKjGjnzTf0sLGpJVVn5k4MHzA9fvz8MQ2L7
Np+o+TScNbXqUlxAvu8VrSB6OxfOLzNFVi8lHFNn7B0gzcLx72+j/A5zULwk
cH5WARPP529yNlxDRc6y0EeuUiz8UdHz/WUSCDMx/ZSeeNIzqR+FjoR7mR9m
5HjP98X1xydv6J9gYej51SiX/zjY21tgPe7JQtjMbH5tCBMuev8SbbgMlGc8
1FNZyUaW6AOjOwphkGv9IaN5i4lHTfI6j/b44YqE+obzJXS402rTNLpZcK2U
kJ+dxMHn1Xc0d5H7Cp6pVVGwk8z99kkXqhILFSdb5mndY6Hkx5dVGrsJ17i3
H2m7QzhxKrHq5RALpfVHjttfCMbHQ+ZGMW5cXJPy3lZ4OQi1GaPzOgy4+FTp
Y65SyIFIWF7rJ3sWlFRlNWcuYYH9ZHSncBcHKb+OrZU/yMWimrScKxphcM0Y
l5xF8qmjwdu28CINOk8fuGgG0NFi9Sls9B8dsR2zM2toVEzbbtPfRjmFmhTb
IaE28SiN85GbbnCgdDhOScmejirJZ6tr/Og4a2Vf/6grFNtsvs04QfIoWOXd
zUcKbNTJ6CXOTmXhqq370o/vQ3FRtUqrYCvhiQNzXsGPgX3+kWErD4SjW+lS
8OhTJjxE/94QuHBx3fX5SzrhWObPVZtv1XFwWa6j0I/BRJK2lghecdF8ToHW
5MnGnCtHcyq9Gbi1WY42MsjC2dntCxO/cMG61sVx7ObBSaCd1n2DAZ+wdYf5
O7hIVZwVvmYFH6zQJdkKK/nYp6Yj/0ouCL/qRvcWUUMRlGsSZ3KGhe9+7pmc
syE4cGjn6Z17mODPzX+WZsCBXjb/iucVOpL01uUvYoRj6eKdupPB5Hxt+avl
pUPBgPtFoXMEAx7b9jcxphG/SfU4s1mXgoqMmf7i95mwrvh4npFAwfQphQSR
q2Fgx642/XiPjdcHGtLZu0Nw+xtLLPM6Ew0uPs9TboTgkvcpQ2vS44N36tmb
FShoXLOFubs5BIxp7berJlngPZzzrdUiFAdzlyx26WZicwDjH/8SEzer55+3
9ic5YDppdO0/KqLHeQud7pG+VNBW9T5BR2d+XWsp6WWqVXbqjlom5jde/HRt
LwtIFAkLMSK8ckcj3/BGKPqGI9ySk6io2/BY/3+f1yranVnVIeTgp1D80SiL
DUNtVaoKuS/WjE32fnfZWPLq0qOKVUxMfv27V6EqHDTRF8Nh18g+bnrJnO55
CoXZq2UHx1hYvDRrib4UE0rKlM65LSexnZUQNlJP9vePZnzlLTYiJSdfUBrI
vCkvOL6nhoE1Z1ai4gAdCzbPnkEh+cJ8HpNuPx4C/QFPh+Q1TOQU9lme9fVH
W9AcA7spGrbz+481GLFwf2SLbVov6fX5MqOmhA/WedqaDWWxYeXdc7xfwESC
hEOH3A4W8rKPSYv2sRCUxXRcrMxBVOLwtAN8Fs4zYsQf+QVj4yH652X3OeCp
3S748ykYsu+uV8vlcBBX15Tw8jvhyrEfJ3/dYMKNv5Pzh/hTLKdnXOMlG0M1
vNxLoxxwcgyqdUNCsX/xWKHpWQ4Wd87Yp6tI8lDCIiu2kIbFHlov7hozYG55
tUC5mAr/g08yae6BOLAi7OG1NMJzc5g3I4n3vwpUDk3Lp4GvU37pQB4NBVDK
/6sUgjnXi9W3E8+e6M51+kQ854fD1la+AgtpCVZ5EgdCcX5yQYEdmUNpZsz6
C2IMjGx9KmE0jQLFH/Nqqg4wkVUW0WlxkQO5ZuoVDREmPkc1PAkh5y7v++vb
QZuY0BJe6AxYwcVX7crOmscsuGuJzqlSoCM0znzDNDsWktYPSj1W5uKHXm6b
NJeHZ3/SnBTk6LjIFpufRfiC9XuaXb8rD/zebyqrjvGw0jmYmhkTiNpP5Yuv
poZA+vJdF6UeJoS1Islzi4PQc2XsTZANA3qTlR9/hLAha6r0oj+IBheHuSXK
RmEwMsp8kb6Wgo0pvU65s8KxPqiyJ28nHUUzczd+LKPjxU/xsgl7wkNLXkp8
VGSi/+lLE3leOGZ2Tlb9sApFXnPR+1UibERLM1ff8QjCnxVD+s9EmVA70Gvk
9S4YtLp2yt9ANsq5vK/K38MwzfeiyvPzwbhrP/Rt3S6Sx5W7doUxgpGcp1b9
ZSsTh6c0lE0I91vFP55ZspWC4fosp9g+0sufFmr/eUhFIS1QV9+Xhg2PO2fV
7WSijVJXGE389MPzR5rauUzMaOgyE4SxcTE3YV1FSghmq0Xm31pJxSho2bYU
JrLLvbdu+9/v94p290QRHpb/NdkRTbi7ba7Xg3XibFhsPVbD0GKgNJI76znh
B0+jkRYnCyaC5bLfd1n7o4Rxh1lhTfju78+Rzf102K4TtGps80f53ejuIjI/
N+XPt10neRJZvXHmsCYTc8YWfyuKpeNK0ZMu2TU0JBTen7pbx4TJ8sY2amcw
Nng+lNYxYuCcQUHIl6encCbXWHL5Nhp8gm1Pq8Qzcb73xGO7NWz8jstbIUW4
89VXb6k1f1gwuKQWHdDLwOUlrYt2tTCRuykkdJx4ZKihv0gUk42nJ0Y3nF3G
Qtw9d/Exx2CM2Wroj6lwYKRYd2dTdwgG7eJ9p+Q4uG/y6e4+TcKns9lhS8i+
marmny57xcD1WZ5J8f+xkVRV4smI5MB7w7OgnV9CsMdiJG2+Fpn/2a4rejLC
of8slbH/MxWfhGk9C0IYoHm9K97XT8VdURPxZR1BCKbp3xKRYGFiWCBcYMwm
Xu3m/XshDfMa95/+9IWKo9elNjjZB2HrgGNQOYeFAPuzfyReMNEV97Nx4Vkm
1ubZ+nyZF4pq7TIDt2EmjKfN2910iQ79Ow+ujvApaHjZIuCRfji9YyJF6x0b
jtipG3aenB8Zu23Rl9kQql6wOvSDga8Kg3ue+ZL+ztog9lSFBZMS0bPGdVTE
XF4+8ukhE/8t8Wxp282B3d5ngWc0ecg5QunPYFGRtjTRnhXFwdQ7U+NOIRfy
UbuW7coivLT/ZmN+VQAK7jpFvzIPRsOFqnincAa22o1UKV4/BV+pgB+P19Ng
2ui6uI94xOJTF6+cOENBSHJeaVZACLY/6pNuIfzwZL1x3OF1oTjNcwv9Qrw4
JkLGcPQGlZyLtaW7pMPQ174xZMcaOiQbFsx+XxNKeFRP3cA9GCbL4tJ/1zAh
eSW8+PzOU3CO3TGnoYN40Xe+1XBlEHhxDx0adVkwq0+te7Y+FOMNnyy/vgqE
99VN69pWkL4azYyKvBkANvauMA6mQ91f7/vyahoSJxO9XlPCkHrU1sDXhIJb
ezMev1pHgXJ1ifhJeSq6776NU/Gmo1givUB/Ph0Fi9un+W9iwFHVf5GzPguP
I6qz9H8HQb+uesvPsxT0zVJZ77yQiW+/hh9L8VhYarDCx+QDyUV7T1O2DBMv
I3z+TpC88jNUV7J1oyF67QVzLd0QOA2VX5sbSfx/5adBqTZ/qIhXNC0kc5hX
XLl9G+HXmrI13WcyAlD08NtfNsmroPVba79Uk328SpsaJjzV3O90tD+CnEvx
eWr26ynwDPOdqtjDwIru1HJzsyCwVu553MOhoSH37ta2Q4FgNWHsvhYVGoua
afGDdLxe/OfqowniCb0Xmj5sYWFyBmNd6CQT5oY9sdrqdDyPluKKkvx5uuqm
UiXh5c/bLuX4xxIPf/hjkWwCExJdYY81dgVj04t0nXPr2dheyHvm9iMUIybV
tHrCbUMrqN/r+5morRDJyRkl8/7TLiluAx2Nq75WbXvNwlb6p9OvJkke3bG2
/t/nMwPrZtJcL7LxT0JqY0xwKI4WNR2RLiB+rnNqvLWUgf2+j7eoGNBw6XfM
pAk1BHI7l6zdps5EzaHJ852zWXgxtm9wcicVC7lTc5qrKWCplYQeuHkKei9p
C09Qmbg++cyLdoABC9F1VmbpxOcT049v/RQCtUw5hVUFDHRIyt7uWEXHnryT
bslqVCh8cE4sncaA1+b6sUuXWbAe82oyVWTg7/u9zcovWVilkp982Z6BinbH
6eMubCw9XURZo8vEoHXZ5a2d4XhzPzb06RADba0b9KyIj3NPX2JblHFxYNgw
rn15OEx+d+RnSXHw/tqfjj2GhFurqoLk/DjgplMWSlgEwEqNUvD1WBBU9mb1
Fz+kYa0+o0rH3w8bnyRbSKdS8FnLLVfFnHDj4/ICb8NwlB7Jj9d1C8ZP030S
Ko9CsFTKyz+oKRjxg/6j49nhKGhb+zOR9KD90pbGTcaheHtt3sHf2VRoHmGn
MBghSHn25bzD8iBErhul9VQwEHqoYuXa777YVntY+HwzFXL/RL//UglC2Vsp
F0kTJlamMi4EioRg3Mroy/Q7AVhzf3+WL+mBN6Jxpa8L/aE5+0vKIhEazAMd
XQNIPz1TlnCdcAqF3MJv+X8/h6PhjpjEwQXhGD6j+FJ5DwV2p3+nPnpDRc/N
6Xo2QVS0vtka+e8qDTvZLLWV5P1O5oX9OvslEOUJrQbd5hRIlp+a3VHEwPqb
Zk9tkpiozGh836PHxOWbKicptxjEI6JCWuYwkTo1ZrBlFhWSkRb7zhQFQVrD
oNH+C/EtGY0jMjWnsGcqHx/Jvjb7H7d5XkPBrkObPLruBOKXrG2eYiUVBUoq
52IJV54YsAhyeEfFJvXUiRFRCgKzfm1+VxaGebc1VJ8n0TBoVFC5yjIQn8vi
16rZUjFn6Jr0DK0gvC5YuNiYSYF8c+Xsx2akF96tcnoly4RrU8UwdTcT2or2
vjfJPn45bmImNp2GXkPs2PmJgZM/u5YoPmfiXyzv8sEhJjb1ivOWGTKR0tMo
cCBzd/LHjPmNZ1l4ucktx5sahp8bvuzKLWLBmaG2w3ucgZi6yap2NuEJx1nn
jpK5iYrxDq9hs6C58u/J4itsOHnLtoQphoBa/6Mmbg8bKbZelNzFIeCdVku7
/TYcyp5BK0zaGJhaYSgmdYSGpoPMJDPTUHS3Wl7pVWdArMRDpH8lE037RZ7o
N1CQfFlrslOa9FXcU/FLyv445D/zXjWTgbUX+ZWfCmgIEmGVdSfS8azxy7lF
gyG4KrPWl0K4b4FlW1TDRxpsx28xokOpcAkTORNB8nflaWPplgdMFDuNpgS3
0xHD6jvpZEV8X3Uk3vI3HQdeXV1SSzh18swil5KthFNvOPSbJoeh/+gKid3k
79/XdfQWE49/cvPdDK9QLkJlQgVPI0PRFNFPm3zIRkvx6PE8MzacnZqWlZ1j
477ZrqbBllOwfKu5c2t0INYdfJkvVkeBnrmP18UfXnBocdta+yIMS70GVcbI
+nZ83O51qC4UBu631acfCMKzoej2BuNgXA/hPdC1DELpQWd18Y+hiCv4IRIv
CMOEOLVuOpmz3lr/Iz6/w2H7ZM6O9dKkV5l8Te9pgXB1UWgVI+edR9fNyGk8
Ad+FvhPmR8JxIuEZVSs+EN1lresLiNcE0B5659cFYYbn3M02t05h6baampIL
hEeOqXo80/LFcoPEmoMBFFy3U4zdahEOtPMC/jMmPGT0ZtlYejhUvOYu3j8c
iqHI/LPa5GuPw2kH7s2jYJymcVDtWjg+FDWejBikoGvWSOH4Pjruyy5iRrYF
YL20akOJBAUCtUf7ZCwZMBIeP6wvw4D96f7RWXMZGCib8zHNiYF1ATm7Xv7v
9ywDvmxZfDkcff9Klu9WCIQpRePaDDc6ZPrmdfhmBeDa0osjp+JpsA5MPLoz
NBwSgXM+XhEG4U5k1fUiFQoyJA1ym2ppsHmsfvrVbgrkXtmvELcMw8CF/KJG
uVB8VK+fjMui4IXZaLbSfwGoNKnZH7iEgvcvu64q1QTBSnQN4/m7cFz06Pg1
4k/Bl8OpRvYk3+tEpK7vCaMjMcPqpo4GA54yiwP+faHgZ1bTvj3uDNS73dIo
OkM4W9+kaL4VEwrKadasZgbKc2+c/bA7GOc4Ha0mIiw0fBzuNf8YhvX21P1L
tFjIeC7RqN5FA+Xv3NQCcQpS972SvelMxeIn93+YbGVhY43v5SNuhFtVTJx1
JoLhNRbp9L//z2zqL3YmrTUIBhckF36YFw7/pNYHd7+T59qcS/2/Csw8nqqt
D+OINChKpuIWlQZSGeqGPDqUqJB0Dd0yRSE5zrDP5GzHmCRKmtyUbhIls1CG
KI6hgai3Qdwm4ib1SqHwrvff9dl/rLX2/j3P97s7EsToYVydv7GTj+BZ8u15
M2hcGFiZd92e9P+ptEMJB0VYvs3m91g/IbRseo98VmQitSzysM6wGJmtr0uU
SZ7V6BYccyP5lNO69l36OA8Xfn246P1HOIIHEhc+IF6oJ/dWxroiHFG1gymR
A2LMHusetD0eAf+lvakmWTTOWfHF1k8lkB8sT/ErpiFHH6L4nRLw7Btcz6wg
nPpRTq3Xg48HXrl6cuR+eZFXtLunSlAoH1J2xDoGfcl3+mcNUFDT/2FeRkfB
rLnDPpYRiekBgh6Pqkh8LdHPceOy8H6t/oRfNRsukoY7RwmffSp/WNUaGYB3
ds2NLYU8pP7woKW9Ityz9txdasfDn85ZddJ4Dgq92GY3r3Bw34jfaxnPxi8f
hc1nSe+9G3HmiEuIHz03EPe0UyifNCh/skQA04aBk1tCOPgkN1XbUY6NuilO
rdaE/4anbzR1PeMPRqV8jSyIB6+KtPTR5sDhGu9qVocISW2HksRnOTiTK6d8
sj0Ms1LLfxz0Jd54u9uVcTMICqpdR+IfE5492lZ7eD4f2QbM3V+sKBRk6J7/
QJ5rzq2T+DWTc6yKaT6iJERmost0l9kC0DrjVY1/8XFLOd773oQA7lNjE4ba
RJDV9bBf2s4CDp1SyXgmBGtHn9eG7zRydV6NLLgWDta4bfXC22JsrtA6VzIr
Alrbukq7Se+c9zf8UmwkwGDiWs0Kdxau2EYsWkTytT2+W9K5nYPBGTIF30ZE
MDJdNJ/SF6D0Y3qU8iIKf42+9XbZTfYfq8v2jRSh9tXk5pjLAhytrCy8v50H
w9+nrV9mSmGkekZRrCfJwY95o6/TWCjsGefc7RcgUaSz0s6CQo+ffuiPnULM
LS70tvEg/u2iHLjvtQip202mhY2LEKN5ePHPFjEeUhtNsj+QuTc2k+4aoDH0
W+fQggMSOFsUhf9kRqCp9l6qBvF6l7HU2ridFHzPnda6zZcgNCDf12WtENmt
lcdcyiQoEzCmWHBF8Dh9P8kkhvTgtbuxfYSPDLTu3c9TjETsz8Ygg9+icN3P
WvHDOwrS0jcbSlqI3/oVDyyv4sCyvo9h5iqA+sU29fdTJCh9a8MsyxZjxT/l
x7+RvPkud+2VnZwYg4+N7ewmSL8tNG91VBNh4l3Bja8tAqx+P/uTmlUoNHO/
uKfsDYfteGuHXrIA3Wnnkm9cD4fR3mHF0bl8nFTI2ecaKIT1zcO3e6LEOGE4
eX4l8cjReEeriDgx2GdnfD8RLIa+1sCZczTJz/WPXHuCJfC9oUIxhDSC7w8s
/qAnwfAY9+kXEzE0ZpcwqFU8yKnUj/zFEUOyr15Q3irGycTJU7mqMcja+eT7
HAcuuj7NU5LdGoWO2nGLZSslOHlYUKT/RoJ57pmed2aykNBpkGHSyMLh8yVC
WSkHs+zmqOom+uHj5m1rxqPZ8PTd5Xy1iQLutUlLZnCh0mD/35XX2Oj/Y07V
GhUWbI3fb4+dz8Ruo8LiwnQWLvwYsza7xgGj5KFuiDoFxoSPg70ehdLFNfrq
x8PQ5KGoVPlvGF6pLjDzJeuP3hq2XmjxB6SXtLpkOSjfeTDRLIvM8Ssn+Jwm
OZnc+GvHLxb6JzJjc98w0b/iyOXHhhQKOQts1NT9YPXsXUpJAA/4uiflM9kX
rXFKpUOZizEZabr7CsLZDqKAWw4UlExif9sqEODY/u/TXki4KHXbmt3fzUGa
tFw+vYVC0pkbjlqpFI7p97Dc4sLwwMErbDXhZsu+q6/iMmjUWnd9mLWK9I75
9HLp7HDo2TLa++/TSLo1R3higwAOx3TXO+jwYG2/wp5x8DCGV3cobAkUo8Up
sJf+lwNjGYaORrUA8TLLTyST/Qax6bK92RSKt/sy3CQUgoJq129z4eG96MAL
2cUkJ27VdZyOYcNjk/9AWjELhi8si8tqOXBb3fLBsyYMql0732QE8NE6u6bb
O5OCefsmcU6FAC5r6T2qiVz83G0gq1rOQ6Xyf9r71HjgdOrzXquEw8rZf8nT
EgFajytwCiNoRMxXHQ23kOBSoWGwqxKNmdUF1NF6Gi4VTUIvNwqV09Ne9RH/
yr5cv6ua9LT6ZvaYlzHx1Kb+RvZjCqNr171uD+WhbsMfvtNJTp21Kcj6p43M
6WVpxYNu4o3xuXOn/ofCZ8fzeg9OROK00c27IOf6Xe/uywwhH9rBPWEehNP1
7uZUddaLIR8k1/qwiOwvK+N70EMRZnOTXl5+wUMjo0n8slYIw80+mu4ktzQ9
5VjKqaGoSx46U7FMiMrQ2IRT5lzscZbWVZD5z2l/8m4PeV+VGlRRtQMPJ71H
8pkeYqhl1j800xRjXf/ckKULxVjZo7Ktf1KA8Ymaf508aFzZPC81k/TvAyND
vWYGjRP0KtNxNllneRVX6oUjLsNpouoOFxYqHDbrTjicatIUCozIXLflb9Lv
jUZEs/evK19ZOP/U+oCZShQkNRH6+BmO2/nO0iJxBNJG7tRM6Q/Dn6tiquPC
WPik3L9oyCUUST2GkcnL/dEw4vW+h+RJhhFfwcM9DEG2vvLXFTnw2OEbKufN
RsOSFTsqSpiYLAiRJWaLqV0874ubDsFynVdX2hcWllg9jzYmPFGUf4q/6ygH
PkvXXRuNDoXEys6o6UQYrs6T6/MWMWFqvdDNx/kArpfulf5ayoS7mgN3ahQb
Q9rLhLsdmah2s1feRfpD7mJsw8xUJr6u/qEgI8+GcmLzOD/FHTfTjJfO3UVB
/1HVucziEDx35/dqZHGQ+Y+FyipNIfbkqhY7lXGhaCPNHCM5+iROyVdqw0Z+
gonMqBMLieau3mNKXFy6mx4X8CcT0c7PbF9OC8PnmHD3R2eFsPLxyHaNo7Hr
3t9W279y0HknZ+YSwnGGB1yZM/NpLAlxqnxdwMNFbsUKbKRQ70+PbssJxMM+
Ol7TRgwnlRB1Vy0uwgP3vaFa+Vig4/NWuYzCbo8lW6zId1vRO1ZZOo+Lvo8X
v+AQB17QUMib4GK50adhx7NMDBb0mdeQe/GZM/Y5WpMJ5+R0mfe8MCQGNDem
PuXBZ+unyYw6ClXrDelb8eR7XfJmrtcoC/PPbqlaMMhBtu5V64LTLHSJ1qSY
HBLhwM6vSzPIPbj1tZjn7SMc1TmawjSSQMBLHk5IFsNXqTqk6TyN9KaYGDcP
Cu4NC2Of7YiAd8TfFfuPC5HatC8vgHCVqq5ddu8gC57++WvWkLza+7bHsZbc
V6y26VBRrQSdj66fTn8QifoXe0oFHYSDRJ2Pm+lIxHyPUfTUYeNb3ghvcg0f
AUqab7RVJBh57JmQ+Ij4WYq9c1CrAFY1gRvM/UQ4tTpvzTl1LuRXqp+jsoUY
G8l5fvMLH1Ut16V93aGwW9ZnZGFFOO2S6fjaJOJ7B978KoUIU1w/phmt4+Pb
iyHP7Gwu2pSKevR2iDGp/IRxdakYwwcvP8+bDEdjR7/zBks+ajbNVQ5zpPE9
+mCVxvUIaOb7WdgZEx7oVJ7r1ijGnh+7UqIbRVB0MaBaTLnone4w5///LT/3
ixYFWvPhZnlsXvGLaKy7FfPB4W0YtOv3Y8ZkJHg/g6027heB3ul9eeoFGv8D
dHOu+A==
          "],
          ColorFunction->(Blend[{
             RGBColor[1, 1, 1], 
             RGBColor[0, 0, 1]}, #]& )],
         AspectRatio->Full,
         ImageSize->NCache[{
            Rational[1, 2]}, {0.5}],
         PlotRangePadding->None], {0, 0}, {0, 0}, 
        NCache[{Rational[1, 2], 1}, {0.5, 1}]],
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Epilog->{{{
           RGBColor[0.5, 0, 0.5], Antialiasing -> True, {{
             
             RectangleBox[{0., 0.43198696602798653`}, {0.475, 
              0.4438917279327484}]}}}}},
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->NCache[{{0, 
           Rational[1, 2]}, {0, 1}}, {{0, 0.5}, {0, 1}}],
       PlotRangePadding->{
         Scaled[0.017142857142857144`], 
         Scaled[0.1]}], {180, -35}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[InsetBox[
        GraphicsBox[
         {RGBColor[0.325101, 0.364492, 0.820256], LineBox[CompressedData["
1:eJw0WwdYTW0cF0rIJp9KKjJCQpGMn5mZiuyspElJSnYK0UApmprapb3nbe97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           "]]},
         AspectRatio->Full,
         PlotRange->{Automatic, {0, 1}},
         PlotRangePadding->None], {0, 0}, {0, 0}, 
        NCache[{Rational[1, 2], 1}, {0.5, 1}]],
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->NCache[{{0, 
           Rational[1, 2]}, {0, 1}}, {{0, 0.5}, {0, 1}}],
       PlotRangePadding->{
         Scaled[0.017142857142857144`], 
         Scaled[0.1]}], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       RowBox[{
        InterpretationBox[
         RowBox[{"\<\"0.5\"\>", "\[InvisibleSpace]", "\<\" s\"\>"}],
         Row[{"0.5", " s"}]], 
        StyleBox["\<\" | \"\>",
         StripOnInput->False,
         FontSize->14,
         FontColor->GrayLevel[0.75]], 
        InterpretationBox[
         RowBox[{"8000", "\[InvisibleSpace]", "\<\" Hz\"\>"}],
         Row[{8000, " Hz"}]]}],
       Row[{
         Row[{"0.5", " s"}], 
         Row[{8000, " Hz"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 2] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 2]]]; 
          Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[CompressedData["
1:eJwcmwk0VP//xhOVlMpeSkkpe4tKhd5FspRCqbQSqRSSUFpEafG1hOzZI6SQ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             "], 8000], "Preemptive"]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[{
    Sound[
     Sound[
      SampledSoundFunction[
       CompiledFunction[{
         Blank[Integer]}, {{2, 0, 0}, {3, 0, 2}}, {0, 3, 3, 0, 
        0}, {{1, 5}, {8, 0., 0}, {8, 0.000125, 1}, {15, 1, 0, 2}, {21, 1, 2, 
         1}, {18, 0, 1, 0}, {7, 500, 1}, {7, 20, 2}, {15, 1, 2, 1}, {21, 1, 0,
          1}, {93, 1, 3, 0, 1, 3, 0, 2}, {15, 1, 1, 1}, {21, 1, 0, 2, 1}, {93,
          1, 3, 0, 1, 3, 0, 2}, {8, -0.000014258243185127917`, 1}, {18, 2, 1, 
         2}, {8, 1.0000181082271948`, 1}, {21, 2, 1, 2}, {2}}, 
        Function[{Play`Time59}, 
         Block[{$CellContext`t = 0. + 0.000125 Play`Time59}, (
           Sin[500 $CellContext`t Sin[20 $CellContext`t]] - 
           0.000014258243185127917`) 1.0000181082271948`]], Evaluate], 4000, 
       8000]], {0, 0.5}], 
    SoundNote[0, {0, 0.5}]}]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->966825440]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->18870],

Cell[TextData[{
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " renders MIDI notes even if they would not be played by the specified \
instrument:"
}], "ExampleText",
 CellID->96937903],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"SoundNote", "[", 
   RowBox[{"\"\<C3\>\"", ",", "1", ",", "\"\<Violin\>\""}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->322244040],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {Hue[-0.1599553194190092, 0.9, 0.7], 
        StyleBox[
         RectangleBox[{0., 0.2891298231708436}, {0.95, 0.3010345850756056}],
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"1", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{1, " s"}]],
       Row[{
         Row[{1, " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 40], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 40], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[
   SoundNote["C3", 1, "Violin"]]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->203917200]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->11195117],

Cell[TextData[{
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " specified the first MIDI instrument as 1, rather than 0:"
}], "ExampleText",
 CellID->91461644],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"SoundNote", "[", 
   RowBox[{"\"\<C\>\"", ",", "0.2", ",", "1"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->545407989],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[
         RectangleBox[{0., 0.43198696602798653`}, {0.19, 0.4438917279327484}],
         
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"\<\"0.2\"\>", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{"0.2", " s"}]],
       Row[{
         Row[{"0.2", " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 5] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 5]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[
   SoundNote["C", 0.2, 1]]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->363200134]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Neat Examples",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->9241],

Cell["Play each of the first 15 MIDI instruments for 0.1 seconds:", \
"ExampleText",
 CellID->29124222],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"SoundNote", "[", 
     RowBox[{"0", ",", "0.1", ",", "i"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"i", ",", "15"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->18839667],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[{
        {RGBColor[0.5, 0, 0.5], 
         StyleBox[
          RectangleBox[{0., 0.43198696602798653`}, \
{0.095, 0.4438917279327484}],
          Antialiasing->True]}, 
        {Hue[-0.009100000000000052, 0.9, 0.7], 
         StyleBox[
          RectangleBox[{0.1, 0.43198696602798653`}, \
{0.195, 0.4438917279327484}],
          Antialiasing->True]}, 
        {Hue[-0.1814296235748465, 0.9, 0.7], 
         StyleBox[
          RectangleBox[{0.2, 0.43198696602798653`}, \
{0.295, 0.4438917279327484}],
          Antialiasing->True]}, 
        {Hue[0.3762407528503072, 0.9, 0.7], 
         StyleBox[
          RectangleBox[{0.3, 0.43198696602798653`}, \
{0.395, 0.4438917279327484}],
          Antialiasing->True]}, 
        {Hue[0.203911129275461, 0.9, 0.7], 
         StyleBox[
          RectangleBox[{0.4, 0.43198696602798653`}, \
{0.495, 0.4438917279327484}],
          Antialiasing->True]}, 
        {Hue[0.031581505700614465`, 0.9, 0.7], 
         StyleBox[
          RectangleBox[{0.5, 0.43198696602798653`}, \
{0.595, 0.4438917279327484}],
          Antialiasing->True]}, 
        {Hue[-0.14074811787423214`, 0.9, 0.7], 
         StyleBox[
          RectangleBox[{0.6, 0.43198696602798653`}, \
{0.695, 0.4438917279327484}],
          Antialiasing->True]}, 
        {Hue[0.416922258550922, 0.9, 0.7], 
         StyleBox[
          RectangleBox[{0.7, 0.43198696602798653`}, \
{0.795, 0.4438917279327484}],
          Antialiasing->True]}, 
        {Hue[0.24459263497607545`, 0.9, 0.7], 
         StyleBox[
          RectangleBox[{0.8, 0.43198696602798653`}, \
{0.895, 0.4438917279327484}],
          Antialiasing->True]}, 
        {Hue[0.07226301140122887, 0.9, 0.7], 
         StyleBox[
          RectangleBox[{0.9, 0.43198696602798653`}, \
{0.995, 0.4438917279327484}],
          Antialiasing->True]}, 
        {Hue[-0.10006661217361768`, 0.9, 0.7], 
         StyleBox[
          RectangleBox[{1., 0.43198696602798653`}, \
{1.0950000000000002`, 0.4438917279327484}],
          Antialiasing->True]}, 
        {Hue[0.45760376425153576`, 0.9, 0.7], 
         StyleBox[
          RectangleBox[{1.1, 0.43198696602798653`}, \
{1.195, 0.4438917279327484}],
          Antialiasing->True]}, 
        {Hue[0.2852741406766892, 0.9, 0.7], 
         StyleBox[
          RectangleBox[{1.2, 0.43198696602798653`}, \
{1.295, 0.4438917279327484}],
          Antialiasing->True]}, 
        {Hue[0.11294451710184394`, 0.9, 0.7], 
         StyleBox[
          RectangleBox[{1.3, 0.43198696602798653`}, \
{1.395, 0.4438917279327484}],
          Antialiasing->True]}, 
        {Hue[-0.05938510647300388, 0.9, 0.7], 
         StyleBox[
          RectangleBox[{1.4, 0.43198696602798653`}, \
{1.495, 0.4438917279327484}],
          Antialiasing->True]}},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"\<\"1.5\"\>", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{"1.5", " s"}]],
       Row[{
         Row[{"1.5", " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 1, "Value" -> 1], 
                 Sound`MIDIEvent[
                 1, "NoteOn", "Note" -> 60, "Channel" -> 1, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 60, "Channel" -> 1, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 2, "Value" -> 2], 
                 Sound`MIDIEvent[
                 2, "NoteOn", "Note" -> 60, "Channel" -> 2, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOff", "Note" -> 60, "Channel" -> 2, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 3, "Value" -> 3], 
                 Sound`MIDIEvent[
                 3, "NoteOn", "Note" -> 60, "Channel" -> 3, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 4, "NoteOff", "Note" -> 60, "Channel" -> 3, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 4, "Value" -> 4], 
                 Sound`MIDIEvent[
                 4, "NoteOn", "Note" -> 60, "Channel" -> 4, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOff", "Note" -> 60, "Channel" -> 4, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 5, "Value" -> 5], 
                 Sound`MIDIEvent[
                 5, "NoteOn", "Note" -> 60, "Channel" -> 5, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 6, "NoteOff", "Note" -> 60, "Channel" -> 5, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 6, "Value" -> 6], 
                 Sound`MIDIEvent[
                 6, "NoteOn", "Note" -> 60, "Channel" -> 6, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 7, "NoteOff", "Note" -> 60, "Channel" -> 6, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 7, "Value" -> 7], 
                 Sound`MIDIEvent[
                 7, "NoteOn", "Note" -> 60, "Channel" -> 7, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 8, "NoteOff", "Note" -> 60, "Channel" -> 7, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 8, "Value" -> 8], 
                 Sound`MIDIEvent[
                 8, "NoteOn", "Note" -> 60, "Channel" -> 8, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 9, "NoteOff", "Note" -> 60, "Channel" -> 8, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 10, "Value" -> 9], 
                 Sound`MIDIEvent[
                 9, "NoteOn", "Note" -> 60, "Channel" -> 10, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 10, "NoteOff", "Note" -> 60, "Channel" -> 10, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 11, "Value" -> 10], 
                 Sound`MIDIEvent[
                 10, "NoteOn", "Note" -> 60, "Channel" -> 11, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 11, "NoteOff", "Note" -> 60, "Channel" -> 11, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 12, "Value" -> 11], 
                 Sound`MIDIEvent[
                 11, "NoteOn", "Note" -> 60, "Channel" -> 12, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 12, "NoteOff", "Note" -> 60, "Channel" -> 12, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 13, "Value" -> 12], 
                 Sound`MIDIEvent[
                 12, "NoteOn", "Note" -> 60, "Channel" -> 13, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 13, "NoteOff", "Note" -> 60, "Channel" -> 13, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 14, "Value" -> 13], 
                 Sound`MIDIEvent[
                 13, "NoteOn", "Note" -> 60, "Channel" -> 14, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 14, "NoteOff", "Note" -> 60, "Channel" -> 14, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 15, "Value" -> 14], 
                 Sound`MIDIEvent[
                 14, "NoteOn", "Note" -> 60, "Channel" -> 15, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 15, "NoteOff", "Note" -> 60, "Channel" -> 15, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 10] =!= 
            None, 
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 1, "Value" -> 1], 
                 Sound`MIDIEvent[
                 1, "NoteOn", "Note" -> 60, "Channel" -> 1, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 60, "Channel" -> 1, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 2, "Value" -> 2], 
                 Sound`MIDIEvent[
                 2, "NoteOn", "Note" -> 60, "Channel" -> 2, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOff", "Note" -> 60, "Channel" -> 2, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 3, "Value" -> 3], 
                 Sound`MIDIEvent[
                 3, "NoteOn", "Note" -> 60, "Channel" -> 3, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 4, "NoteOff", "Note" -> 60, "Channel" -> 3, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 4, "Value" -> 4], 
                 Sound`MIDIEvent[
                 4, "NoteOn", "Note" -> 60, "Channel" -> 4, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOff", "Note" -> 60, "Channel" -> 4, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 5, "Value" -> 5], 
                 Sound`MIDIEvent[
                 5, "NoteOn", "Note" -> 60, "Channel" -> 5, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 6, "NoteOff", "Note" -> 60, "Channel" -> 5, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 6, "Value" -> 6], 
                 Sound`MIDIEvent[
                 6, "NoteOn", "Note" -> 60, "Channel" -> 6, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 7, "NoteOff", "Note" -> 60, "Channel" -> 6, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 7, "Value" -> 7], 
                 Sound`MIDIEvent[
                 7, "NoteOn", "Note" -> 60, "Channel" -> 7, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 8, "NoteOff", "Note" -> 60, "Channel" -> 7, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 8, "Value" -> 8], 
                 Sound`MIDIEvent[
                 8, "NoteOn", "Note" -> 60, "Channel" -> 8, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 9, "NoteOff", "Note" -> 60, "Channel" -> 8, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 10, "Value" -> 9], 
                 Sound`MIDIEvent[
                 9, "NoteOn", "Note" -> 60, "Channel" -> 10, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 10, "NoteOff", "Note" -> 60, "Channel" -> 10, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 11, "Value" -> 10], 
                 Sound`MIDIEvent[
                 10, "NoteOn", "Note" -> 60, "Channel" -> 11, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 11, "NoteOff", "Note" -> 60, "Channel" -> 11, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 12, "Value" -> 11], 
                 Sound`MIDIEvent[
                 11, "NoteOn", "Note" -> 60, "Channel" -> 12, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 12, "NoteOff", "Note" -> 60, "Channel" -> 12, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 13, "Value" -> 12], 
                 Sound`MIDIEvent[
                 12, "NoteOn", "Note" -> 60, "Channel" -> 13, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 13, "NoteOff", "Note" -> 60, "Channel" -> 13, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 14, "Value" -> 13], 
                 Sound`MIDIEvent[
                 13, "NoteOn", "Note" -> 60, "Channel" -> 14, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 14, "NoteOff", "Note" -> 60, "Channel" -> 14, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 15, "Value" -> 14], 
                 Sound`MIDIEvent[
                 14, "NoteOn", "Note" -> 60, "Channel" -> 15, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 15, "NoteOff", "Note" -> 60, "Channel" -> 15, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 10]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[{
    SoundNote[0, 0.1, 1], 
    SoundNote[0, 0.1, 2], 
    SoundNote[0, 0.1, 3], 
    SoundNote[0, 0.1, 4], 
    SoundNote[0, 0.1, 5], 
    SoundNote[0, 0.1, 6], 
    SoundNote[0, 0.1, 7], 
    SoundNote[0, 0.1, 8], 
    SoundNote[0, 0.1, 9], 
    SoundNote[0, 0.1, 10], 
    SoundNote[0, 0.1, 11], 
    SoundNote[0, 0.1, 12], 
    SoundNote[0, 0.1, 13], 
    SoundNote[0, 0.1, 14], 
    SoundNote[0, 0.1, 15]}]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->6342098]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->179542823],

Cell["Play all semitones in the middle octave together:", "ExampleText",
 CellID->176046018],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"SoundNote", "[", 
   RowBox[{"Range", "[", "12", "]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->269501302],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[{
          RectangleBox[{0., 0.4438917279327484}, \
{0.95, 0.45579648983751037`}], 
          RectangleBox[{0., 0.45579648983751037`}, \
{0.95, 0.4677012517422721}], 
          RectangleBox[{0., 0.4677012517422721}, {0.95, 0.4796060136470343}], 
          RectangleBox[{0., 0.4796060136470343}, \
{0.95, 0.49151077555179606`}], 
          RectangleBox[{0., 0.49151077555179606`}, \
{0.95, 0.5034155374565581}], 
          RectangleBox[{0., 0.5034155374565581}, {0.95, 0.5153202993613198}], 
          RectangleBox[{0., 0.5153202993613198}, {0.95, 0.5272250612660818}], 
          RectangleBox[{0., 0.5272250612660818}, {0.95, 0.5391298231708438}], 
          RectangleBox[{0., 0.5391298231708438}, {0.95, 0.5510345850756055}], 
          RectangleBox[{0., 0.5510345850756055}, {0.95, 0.5629393469803675}], 
          RectangleBox[{0., 0.5629393469803675}, {0.95, 0.5748441088851295}], 
          RectangleBox[{0., 0.5748441088851295}, {0.95, 0.5867488707898912}]},
         
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"1", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{1, " s"}]],
       Row[{
         Row[{1, " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 61, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 61, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 61, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 61, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[
   SoundNote[{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12}]]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->784654678]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->24733363],

Cell["Play several octaves together:", "ExampleText",
 CellID->287204505],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"SoundNote", "[", 
   RowBox[{"Range", "[", 
    RowBox[{
     RowBox[{"-", "50"}], ",", "50", ",", "12"}], "]"}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->4385851],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[{
          RectangleBox[{0., -0.1632511292101088}, \
{0.95, -0.15134636730534692`}], 
          RectangleBox[{0., -0.020393986352965857`}, \
{0.95, -0.008489224448203993}], 
          RectangleBox[{0., 0.12246315650417702`}, \
{0.95, 0.13436791840893875`}], 
          RectangleBox[{0., 0.2653202993613199}, \
{0.95, 0.27722506126608165`}], 
          RectangleBox[{0., 0.40817744221846264`}, \
{0.95, 0.4200822041232246}], 
          RectangleBox[{0., 0.5510345850756055}, {0.95, 0.5629393469803675}], 
          RectangleBox[{0., 0.6938917279327484}, {0.95, 0.7057964898375104}], 
          RectangleBox[{0., 0.8367488707898912}, {0.95, 0.8486536326946532}], 
          RectangleBox[{0., 0.9796060136470341}, {0.95, 0.9915107755517961}]},
         
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"1", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{1, " s"}]],
       Row[{
         Row[{1, " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 10, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 22, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 34, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 46, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 58, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 94, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 106, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 10, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 22, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 34, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 46, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 58, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 94, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 106, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1] =!= None,
             
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 10, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 22, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 34, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 46, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 58, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 94, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 106, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 10, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 22, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 34, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 46, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 58, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 94, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 106, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[
   SoundNote[{-50, -38, -26, -14, -2, 10, 22, 34, 46}]]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->455522655]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->118240812],

Cell["Play a cellular automaton sound:", "ExampleText",
 CellID->120978039],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"SoundNote", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"DeleteCases", "[", 
        RowBox[{
         RowBox[{"3", " ", 
          RowBox[{"Range", "[", "31", "]"}], " ", 
          RowBox[{"Reverse", "[", "#", "]"}]}], ",", " ", "0"}], "]"}], " ", 
       "-", " ", "48"}], ",", " ", ".1"}], "]"}], " ", "&"}], " ", "/@", " ", 
   
   RowBox[{"Transpose", "[", 
    RowBox[{"CellularAutomaton", "[", 
     RowBox[{"90", ",", " ", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", "1", "}"}], ",", " ", "0"}], "}"}], ",", " ", "30"}], 
     "]"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->135191944],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[{
          RectangleBox[{0., -0.10372731968629915`}, \
{0.095, -0.09182255778153729}], 
          RectangleBox[{0.4, -0.10372731968629915`}, \
{0.495, -0.09182255778153729}], 
          RectangleBox[{0.8, -0.10372731968629915`}, \
{0.895, -0.09182255778153729}], 
          RectangleBox[{1.2, -0.10372731968629915`}, \
{1.295, -0.09182255778153729}], 
          RectangleBox[{1.6, -0.10372731968629915`}, \
{1.695, -0.09182255778153729}], 
          RectangleBox[{2., -0.10372731968629915`}, \
{2.095, -0.09182255778153729}], 
          RectangleBox[{2.4, -0.10372731968629915`}, \
{2.495, -0.09182255778153729}], 
          RectangleBox[{2.8, -0.10372731968629915`}, \
{2.895, -0.09182255778153729}], 
          RectangleBox[{3.2, -0.10372731968629915`}, \
{3.295, -0.09182255778153729}], 
          RectangleBox[{3.6, -0.10372731968629915`}, \
{3.6950000000000003`, -0.09182255778153729}], 
          RectangleBox[{4., -0.10372731968629915`}, \
{4.095, -0.09182255778153729}], 
          RectangleBox[{4.4, -0.10372731968629915`}, \
{4.495, -0.09182255778153729}], 
          RectangleBox[{4.8, -0.10372731968629915`}, \
{4.8950000000000005`, -0.09182255778153729}], 
          RectangleBox[{5.2, -0.10372731968629915`}, \
{5.295, -0.09182255778153729}], 
          RectangleBox[{5.6, -0.10372731968629915`}, \
{5.695, -0.09182255778153729}], 
          RectangleBox[{6., -0.10372731968629915`}, \
{6.095, -0.09182255778153729}], 
          RectangleBox[{0.1, -0.06801303397201344}, \
{0.195, -0.05610827206725157}], 
          RectangleBox[{0.3, -0.06801303397201344}, \
{0.395, -0.05610827206725157}], 
          RectangleBox[{0.9, -0.06801303397201344}, \
{0.995, -0.05610827206725157}], 
          RectangleBox[{1.1, -0.06801303397201344}, \
{1.195, -0.05610827206725157}], 
          RectangleBox[{1.7, -0.06801303397201344}, \
{1.795, -0.05610827206725157}], 
          RectangleBox[{1.9, -0.06801303397201344}, \
{1.995, -0.05610827206725157}], 
          
          RectangleBox[{2.5, -0.06801303397201344}, \
{2.595, -0.05610827206725157}], 
          RectangleBox[{2.7, -0.06801303397201344}, \
{2.795, -0.05610827206725157}], 
          RectangleBox[{3.3, -0.06801303397201344}, \
{3.395, -0.05610827206725157}], 
          RectangleBox[{3.5, -0.06801303397201344}, \
{3.595, -0.05610827206725157}], 
          RectangleBox[{4.1, -0.06801303397201344}, \
{4.195, -0.05610827206725157}], 
          RectangleBox[{4.3, -0.06801303397201344}, \
{4.3950000000000005`, -0.05610827206725157}], 
          RectangleBox[{4.9, -0.06801303397201344}, \
{4.995, -0.05610827206725157}], 
          RectangleBox[{5.1, -0.06801303397201344}, \
{5.195, -0.05610827206725157}], 
          RectangleBox[{5.7, -0.06801303397201344}, \
{5.795, -0.05610827206725157}], 
          RectangleBox[{5.9, -0.06801303397201344}, \
{5.995, -0.05610827206725157}], 
          RectangleBox[{0.2, -0.03229874825772772}, \
{0.295, -0.020393986352965857`}], 
          RectangleBox[{1., -0.03229874825772772}, \
{1.0950000000000002`, -0.020393986352965857`}], 
          RectangleBox[{1.8, -0.03229874825772772}, \
{1.895, -0.020393986352965857`}], 
          RectangleBox[{2.6, -0.03229874825772772}, \
{2.6950000000000003`, -0.020393986352965857`}], 
          RectangleBox[{3.4, -0.03229874825772772}, \
{3.495, -0.020393986352965857`}], 
          RectangleBox[{4.2, -0.03229874825772772}, \
{4.295, -0.020393986352965857`}], 
          RectangleBox[{5., -0.03229874825772772}, \
{5.095, -0.020393986352965857`}], 
          RectangleBox[{5.8, -0.03229874825772772}, \
{5.8950000000000005`, -0.020393986352965857`}], 
          RectangleBox[{0.3, 0.0034155374565578715`}, \
{0.395, 0.015320299361319862`}], 
          RectangleBox[{0.5, 0.0034155374565578715`}, \
{0.595, 0.015320299361319862`}], 
          RectangleBox[{0.7, 0.0034155374565578715`}, \
{0.795, 0.015320299361319862`}], 
          RectangleBox[{0.9, 0.0034155374565578715`}, \
{0.995, 0.015320299361319862`}], 
          RectangleBox[{1.9, 0.0034155374565578715`}, \
{1.995, 0.015320299361319862`}], 
          RectangleBox[{2.1, 0.0034155374565578715`}, \
{2.1950000000000003`, 0.015320299361319862`}], 
          
          RectangleBox[{2.3, 0.0034155374565578715`}, \
{2.395, 0.015320299361319862`}], 
          RectangleBox[{2.5, 0.0034155374565578715`}, \
{2.595, 0.015320299361319862`}], 
          RectangleBox[{3.5, 0.0034155374565578715`}, \
{3.595, 0.015320299361319862`}], 
          RectangleBox[{3.7, 0.0034155374565578715`}, \
{3.795, 0.015320299361319862`}], 
          RectangleBox[{3.9, 0.0034155374565578715`}, \
{3.995, 0.015320299361319862`}], 
          RectangleBox[{4.1, 0.0034155374565578715`}, \
{4.195, 0.015320299361319862`}], 
          RectangleBox[{5.1, 0.0034155374565578715`}, \
{5.195, 0.015320299361319862`}], 
          RectangleBox[{5.3, 0.0034155374565578715`}, \
{5.3950000000000005`, 0.015320299361319862`}], 
          RectangleBox[{5.5, 0.0034155374565578715`}, \
{5.595, 0.015320299361319862`}], 
          RectangleBox[{5.7, 0.0034155374565578715`}, \
{5.795, 0.015320299361319862`}], 
          RectangleBox[{0.4, 0.03912982317084359}, \
{0.495, 0.051034585075605585`}], 
          RectangleBox[{0.8, 0.03912982317084359}, \
{0.895, 0.051034585075605585`}], 
          RectangleBox[{2., 0.03912982317084359}, \
{2.095, 0.051034585075605585`}], 
          RectangleBox[{2.4, 0.03912982317084359}, \
{2.495, 0.051034585075605585`}], 
          RectangleBox[{3.6, 0.03912982317084359}, \
{3.6950000000000003`, 0.051034585075605585`}], 
          RectangleBox[{4., 0.03912982317084359}, \
{4.095, 0.051034585075605585`}], 
          RectangleBox[{5.2, 0.03912982317084359}, \
{5.295, 0.051034585075605585`}], 
          RectangleBox[{5.6, 0.03912982317084359}, \
{5.695, 0.051034585075605585`}], 
          RectangleBox[{0.5, 0.07484410888512931}, \
{0.595, 0.0867488707898913}], 
          RectangleBox[{0.7, 0.07484410888512931}, \
{0.795, 0.0867488707898913}], 
          RectangleBox[{2.1, 0.07484410888512931}, \
{2.1950000000000003`, 0.0867488707898913}], 
          RectangleBox[{2.3, 0.07484410888512931}, \
{2.395, 0.0867488707898913}], 
          RectangleBox[{3.7, 0.07484410888512931}, \
{3.795, 0.0867488707898913}], 
          RectangleBox[{3.9, 0.07484410888512931}, \
{3.995, 0.0867488707898913}], 
          RectangleBox[{5.3, 0.07484410888512931}, \
{5.3950000000000005`, 0.0867488707898913}], 
          RectangleBox[{5.5, 0.07484410888512931}, \
{5.595, 0.0867488707898913}], 
          RectangleBox[{0.6, 0.11055839459941504`}, \
{0.695, 0.12246315650417702`}], 
          RectangleBox[{2.2, 0.11055839459941504`}, \
{2.295, 0.12246315650417702`}], 
          RectangleBox[{3.8, 0.11055839459941504`}, \
{3.895, 0.12246315650417702`}], 
          RectangleBox[{5.4, 0.11055839459941504`}, \
{5.495, 0.12246315650417702`}], 
          RectangleBox[{0.7, 0.14627268031370075`}, \
{0.795, 0.1581774422184626}], 
          RectangleBox[{0.9, 0.14627268031370075`}, \
{0.995, 0.1581774422184626}], 
          RectangleBox[{1.1, 0.14627268031370075`}, \
{1.195, 0.1581774422184626}], 
          RectangleBox[{1.3, 0.14627268031370075`}, \
{1.395, 0.1581774422184626}], 
          RectangleBox[{1.5, 0.14627268031370075`}, \
{1.5950000000000002`, 0.1581774422184626}], 
          RectangleBox[{1.7, 0.14627268031370075`}, \
{1.795, 0.1581774422184626}], 
          RectangleBox[{1.9, 0.14627268031370075`}, \
{1.995, 0.1581774422184626}], 
          RectangleBox[{2.1, 0.14627268031370075`}, \
{2.1950000000000003`, 0.1581774422184626}], 
          RectangleBox[{3.9, 0.14627268031370075`}, \
{3.995, 0.1581774422184626}], 
          RectangleBox[{4.1, 0.14627268031370075`}, \
{4.195, 0.1581774422184626}], 
          RectangleBox[{4.3, 0.14627268031370075`}, \
{4.3950000000000005`, 0.1581774422184626}], 
          RectangleBox[{4.5, 0.14627268031370075`}, \
{4.595, 0.1581774422184626}], 
          RectangleBox[{4.7, 0.14627268031370075`}, \
{4.795, 0.1581774422184626}], 
          RectangleBox[{4.9, 0.14627268031370075`}, \
{4.995, 0.1581774422184626}], 
          RectangleBox[{5.1, 0.14627268031370075`}, \
{5.195, 0.1581774422184626}], 
          RectangleBox[{5.3, 0.14627268031370075`}, \
{5.3950000000000005`, 0.1581774422184626}], 
          RectangleBox[{0.8, 0.18198696602798659`}, \
{0.895, 0.19389172793274845`}], 
          RectangleBox[{1.2, 0.18198696602798659`}, \
{1.295, 0.19389172793274845`}], 
          RectangleBox[{1.6, 0.18198696602798659`}, \
{1.695, 0.19389172793274845`}], 
          RectangleBox[{2., 0.18198696602798659`}, \
{2.095, 0.19389172793274845`}], 
          RectangleBox[{4., 0.18198696602798659`}, \
{4.095, 0.19389172793274845`}], 
          RectangleBox[{4.4, 0.18198696602798659`}, \
{4.495, 0.19389172793274845`}], 
          RectangleBox[{4.8, 0.18198696602798659`}, \
{4.8950000000000005`, 0.19389172793274845`}], 
          RectangleBox[{5.2, 0.18198696602798659`}, \
{5.295, 0.19389172793274845`}], 
          RectangleBox[{0.9, 0.2177012517422722}, \
{0.995, 0.22960601364703406`}], 
          RectangleBox[{1.1, 0.2177012517422722}, \
{1.195, 0.22960601364703406`}], 
          RectangleBox[{1.7, 0.2177012517422722}, \
{1.795, 0.22960601364703406`}], 
          RectangleBox[{1.9, 0.2177012517422722}, \
{1.995, 0.22960601364703406`}], 
          RectangleBox[{4.1, 0.2177012517422722}, \
{4.195, 0.22960601364703406`}], 
          RectangleBox[{4.3, 0.2177012517422722}, \
{4.3950000000000005`, 0.22960601364703406`}], 
          RectangleBox[{4.9, 0.2177012517422722}, \
{4.995, 0.22960601364703406`}], 
          RectangleBox[{5.1, 0.2177012517422722}, \
{5.195, 0.22960601364703406`}], 
          RectangleBox[{1., 0.25341553745655804`}, \
{1.0950000000000002`, 0.2653202993613199}], 
          RectangleBox[{1.8, 0.25341553745655804`}, \
{1.895, 0.2653202993613199}], 
          RectangleBox[{4.2, 0.25341553745655804`}, \
{4.295, 0.2653202993613199}], 
          RectangleBox[{5., 0.25341553745655804`}, \
{5.095, 0.2653202993613199}], 
          RectangleBox[{1.1, 0.2891298231708436}, \
{1.195, 0.3010345850756056}], 
          RectangleBox[{1.3, 0.2891298231708436}, \
{1.395, 0.3010345850756056}], 
          RectangleBox[{1.5, 0.2891298231708436}, \
{1.5950000000000002`, 0.3010345850756056}], 
          RectangleBox[{1.7, 0.2891298231708436}, \
{1.795, 0.3010345850756056}], 
          RectangleBox[{4.3, 0.2891298231708436}, \
{4.3950000000000005`, 0.3010345850756056}], 
          RectangleBox[{4.5, 0.2891298231708436}, \
{4.595, 0.3010345850756056}], 
          RectangleBox[{4.7, 0.2891298231708436}, \
{4.795, 0.3010345850756056}], 
          RectangleBox[{4.9, 0.2891298231708436}, \
{4.995, 0.3010345850756056}], 
          RectangleBox[{1.2, 0.3248441088851295}, \
{1.295, 0.33674887078989135`}], 
          RectangleBox[{1.6, 0.3248441088851295}, \
{1.695, 0.33674887078989135`}], 
          RectangleBox[{4.4, 0.3248441088851295}, \
{4.495, 0.33674887078989135`}], 
          RectangleBox[{4.8, 0.3248441088851295}, \
{4.8950000000000005`, 0.33674887078989135`}], 
          RectangleBox[{1.3, 0.3605583945994151}, \
{1.395, 0.37246315650417705`}], 
          RectangleBox[{1.5, 0.3605583945994151}, \
{1.5950000000000002`, 0.37246315650417705`}], 
          RectangleBox[{4.5, 0.3605583945994151}, \
{4.595, 0.37246315650417705`}], 
          RectangleBox[{4.7, 0.3605583945994151}, \
{4.795, 0.37246315650417705`}], 
          RectangleBox[{1.4, 0.3962726803137009}, \
{1.495, 0.40817744221846264`}], 
          RectangleBox[{4.6, 0.3962726803137009}, \
{4.695, 0.40817744221846264`}], 
          RectangleBox[{1.5, 0.43198696602798653`}, \
{1.5950000000000002`, 0.4438917279327484}], 
          RectangleBox[{1.7, 0.43198696602798653`}, \
{1.795, 0.4438917279327484}], 
          RectangleBox[{1.9, 0.43198696602798653`}, \
{1.995, 0.4438917279327484}], 
          RectangleBox[{2.1, 0.43198696602798653`}, \
{2.1950000000000003`, 0.4438917279327484}], 
          RectangleBox[{2.3, 0.43198696602798653`}, \
{2.395, 0.4438917279327484}], 
          RectangleBox[{2.5, 0.43198696602798653`}, \
{2.595, 0.4438917279327484}], 
          RectangleBox[{2.7, 0.43198696602798653`}, \
{2.795, 0.4438917279327484}], 
          RectangleBox[{2.9, 0.43198696602798653`}, \
{2.995, 0.4438917279327484}], 
          RectangleBox[{3.1, 0.43198696602798653`}, \
{3.1950000000000003`, 0.4438917279327484}], 
          RectangleBox[{3.3, 0.43198696602798653`}, \
{3.395, 0.4438917279327484}], 
          RectangleBox[{3.5, 0.43198696602798653`}, \
{3.595, 0.4438917279327484}], 
          RectangleBox[{3.7, 0.43198696602798653`}, \
{3.795, 0.4438917279327484}], 
          RectangleBox[{3.9, 0.43198696602798653`}, \
{3.995, 0.4438917279327484}], 
          RectangleBox[{4.1, 0.43198696602798653`}, \
{4.195, 0.4438917279327484}], 
          RectangleBox[{4.3, 0.43198696602798653`}, \
{4.3950000000000005`, 0.4438917279327484}], 
          RectangleBox[{4.5, 0.43198696602798653`}, \
{4.595, 0.4438917279327484}], 
          RectangleBox[{1.6, 0.4677012517422721}, \
{1.695, 0.4796060136470343}], 
          RectangleBox[{2., 0.4677012517422721}, {2.095, 0.4796060136470343}],
           RectangleBox[{2.4, 0.4677012517422721}, \
{2.495, 0.4796060136470343}], 
          RectangleBox[{2.8, 0.4677012517422721}, \
{2.895, 0.4796060136470343}], 
          RectangleBox[{3.2, 0.4677012517422721}, \
{3.295, 0.4796060136470343}], 
          RectangleBox[{3.6, 0.4677012517422721}, \
{3.6950000000000003`, 0.4796060136470343}], 
          RectangleBox[{4., 0.4677012517422721}, {4.095, 0.4796060136470343}],
           RectangleBox[{4.4, 0.4677012517422721}, \
{4.495, 0.4796060136470343}], 
          RectangleBox[{1.7, 0.5034155374565581}, \
{1.795, 0.5153202993613198}], 
          RectangleBox[{1.9, 0.5034155374565581}, \
{1.995, 0.5153202993613198}], 
          RectangleBox[{2.5, 0.5034155374565581}, \
{2.595, 0.5153202993613198}], 
          RectangleBox[{2.7, 0.5034155374565581}, \
{2.795, 0.5153202993613198}], 
          RectangleBox[{3.3, 0.5034155374565581}, \
{3.395, 0.5153202993613198}], 
          RectangleBox[{3.5, 0.5034155374565581}, \
{3.595, 0.5153202993613198}], 
          RectangleBox[{4.1, 0.5034155374565581}, \
{4.195, 0.5153202993613198}], 
          RectangleBox[{4.3, 0.5034155374565581}, \
{4.3950000000000005`, 0.5153202993613198}], 
          RectangleBox[{1.8, 0.5391298231708438}, \
{1.895, 0.5510345850756055}], 
          RectangleBox[{2.6, 0.5391298231708438}, \
{2.6950000000000003`, 0.5510345850756055}], 
          RectangleBox[{3.4, 0.5391298231708438}, \
{3.495, 0.5510345850756055}], 
          RectangleBox[{4.2, 0.5391298231708438}, \
{4.295, 0.5510345850756055}], 
          RectangleBox[{1.9, 0.5748441088851295}, \
{1.995, 0.5867488707898912}], 
          RectangleBox[{2.1, 0.5748441088851295}, \
{2.1950000000000003`, 0.5867488707898912}], 
          RectangleBox[{2.3, 0.5748441088851295}, \
{2.395, 0.5867488707898912}], 
          RectangleBox[{2.5, 0.5748441088851295}, \
{2.595, 0.5867488707898912}], 
          RectangleBox[{3.5, 0.5748441088851295}, \
{3.595, 0.5867488707898912}], 
          RectangleBox[{3.7, 0.5748441088851295}, \
{3.795, 0.5867488707898912}], 
          RectangleBox[{3.9, 0.5748441088851295}, \
{3.995, 0.5867488707898912}], 
          RectangleBox[{4.1, 0.5748441088851295}, \
{4.195, 0.5867488707898912}], 
          RectangleBox[{2., 0.610558394599415}, {2.095, 0.6224631565041769}], 
          RectangleBox[{2.4, 0.610558394599415}, {2.495, 0.6224631565041769}],
           RectangleBox[{3.6, 0.610558394599415}, \
{3.6950000000000003`, 0.6224631565041769}], 
          RectangleBox[{4., 0.610558394599415}, {4.095, 0.6224631565041769}], 
          RectangleBox[{2.1, 0.646272680313701}, \
{2.1950000000000003`, 0.658177442218463}], 
          RectangleBox[{2.3, 0.646272680313701}, {2.395, 0.658177442218463}], 
          RectangleBox[{3.7, 0.646272680313701}, {3.795, 0.658177442218463}], 
          RectangleBox[{3.9, 0.646272680313701}, {3.995, 0.658177442218463}], 
          RectangleBox[{2.2, 0.6819869660279864}, \
{2.295, 0.6938917279327484}], 
          RectangleBox[{3.8, 0.6819869660279864}, \
{3.895, 0.6938917279327484}], 
          RectangleBox[{2.3, 0.7177012517422724}, \
{2.395, 0.7296060136470344}], 
          RectangleBox[{2.5, 0.7177012517422724}, \
{2.595, 0.7296060136470344}], 
          RectangleBox[{2.7, 0.7177012517422724}, \
{2.795, 0.7296060136470344}], 
          RectangleBox[{2.9, 0.7177012517422724}, \
{2.995, 0.7296060136470344}], 
          RectangleBox[{3.1, 0.7177012517422724}, \
{3.1950000000000003`, 0.7296060136470344}], 
          RectangleBox[{3.3, 0.7177012517422724}, \
{3.395, 0.7296060136470344}], 
          RectangleBox[{3.5, 0.7177012517422724}, \
{3.595, 0.7296060136470344}], 
          RectangleBox[{3.7, 0.7177012517422724}, \
{3.795, 0.7296060136470344}], 
          RectangleBox[{2.4, 0.7534155374565579}, \
{2.495, 0.7653202993613198}], 
          RectangleBox[{2.8, 0.7534155374565579}, \
{2.895, 0.7653202993613198}], 
          RectangleBox[{3.2, 0.7534155374565579}, \
{3.295, 0.7653202993613198}], 
          RectangleBox[{3.6, 0.7534155374565579}, \
{3.6950000000000003`, 0.7653202993613198}], 
          RectangleBox[{2.5, 0.7891298231708438}, \
{2.595, 0.8010345850756058}], 
          RectangleBox[{2.7, 0.7891298231708438}, \
{2.795, 0.8010345850756058}], 
          RectangleBox[{3.3, 0.7891298231708438}, \
{3.395, 0.8010345850756058}], 
          RectangleBox[{3.5, 0.7891298231708438}, \
{3.595, 0.8010345850756058}], 
          RectangleBox[{2.6, 0.8248441088851293}, \
{2.6950000000000003`, 0.8367488707898912}], 
          RectangleBox[{3.4, 0.8248441088851293}, \
{3.495, 0.8367488707898912}], 
          RectangleBox[{2.7, 0.860558394599415}, {2.795, 0.872463156504177}], 
          RectangleBox[{2.9, 0.860558394599415}, {2.995, 0.872463156504177}], 
          RectangleBox[{3.1, 0.860558394599415}, \
{3.1950000000000003`, 0.872463156504177}], 
          RectangleBox[{3.3, 0.860558394599415}, {3.395, 0.872463156504177}], 
          RectangleBox[{2.8, 0.896272680313701}, {2.895, 0.9081774422184627}],
           RectangleBox[{3.2, 0.896272680313701}, \
{3.295, 0.9081774422184627}], 
          RectangleBox[{2.9, 0.9319869660279865}, \
{2.995, 0.9438917279327484}], 
          RectangleBox[{3.1, 0.9319869660279865}, \
{3.1950000000000003`, 0.9438917279327484}], 
          RectangleBox[{3., 0.9677012517422724}, \
{3.095, 0.9796060136470341}]},
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"\<\"6.1\"\>", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{"6.1", " s"}]],
       Row[{
         Row[{"6.1", " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 2, "NoteOn", "Note" -> 21, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOff", "Note" -> 21, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 3, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 4, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 4, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 4, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 4, "NoteOn", "Note" -> 27, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 5, "NoteOff", "Note" -> 27, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 5, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOn", "Note" -> 30, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 6, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 6, "NoteOff", "Note" -> 30, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 6, "NoteOn", "Note" -> 33, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 7, "NoteOff", "Note" -> 33, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 7, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 7, "NoteOn", "Note" -> 30, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 7, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 8, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 8, "NoteOff", "Note" -> 30, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 8, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 8, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 8, "NoteOn", "Note" -> 27, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 8, "NoteOn", "Note" -> 39, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 9, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 9, "NoteOff", "Note" -> 27, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 9, "NoteOff", "Note" -> 39, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 9, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 9, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 9, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 9, "NoteOn", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 10, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 10, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 10, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 10, "NoteOff", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 10, "NoteOn", "Note" -> 21, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 10, "NoteOn", "Note" -> 45, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 11, "NoteOff", "Note" -> 21, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 11, "NoteOff", "Note" -> 45, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 11, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 11, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 11, "NoteOn", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 11, "NoteOn", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 12, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 12, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 12, "NoteOff", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 12, "NoteOff", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 12, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 12, "NoteOn", "Note" -> 39, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 12, "NoteOn", "Note" -> 51, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 13, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 13, "NoteOff", "Note" -> 39, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 13, "NoteOff", "Note" -> 51, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 13, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 13, "NoteOn", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 13, "NoteOn", "Note" -> 54, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 14, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 14, "NoteOff", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 14, "NoteOff", "Note" -> 54, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 14, "NoteOn", "Note" -> 57, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 15, "NoteOff", "Note" -> 57, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 15, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 15, "NoteOn", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 15, "NoteOn", "Note" -> 54, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 15, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 16, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 16, "NoteOff", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 16, "NoteOff", "Note" -> 54, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 16, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 16, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 16, "NoteOn", "Note" -> 39, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 16, "NoteOn", "Note" -> 51, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 16, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 17, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 17, "NoteOff", "Note" -> 39, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 17, "NoteOff", "Note" -> 51, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 17, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 17, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 17, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 17, "NoteOn", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 17, "NoteOn", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 17, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 17, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 18, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 18, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 18, "NoteOff", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 18, "NoteOff", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 18, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 18, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 18, "NoteOn", "Note" -> 21, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 18, "NoteOn", "Note" -> 45, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 18, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 19, "NoteOff", "Note" -> 21, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 19, "NoteOff", "Note" -> 45, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 19, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 19, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 19, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 19, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 19, "NoteOn", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 19, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 19, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 19, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 20, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 20, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 20, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 20, "NoteOff", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 20, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 20, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 20, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 20, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 20, "NoteOn", "Note" -> 27, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 20, "NoteOn", "Note" -> 39, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 20, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 20, "NoteOn", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 21, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 21, "NoteOff", "Note" -> 27, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 21, "NoteOff", "Note" -> 39, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 21, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 21, "NoteOff", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 21, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 21, "NoteOn", "Note" -> 30, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 21, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 21, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 21, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 21, "NoteOn", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 22, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 22, "NoteOff", "Note" -> 30, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 22, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 22, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 22, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 22, "NoteOff", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 22, "NoteOn", "Note" -> 33, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 22, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 23, "NoteOff", "Note" -> 33, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 23, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 23, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 23, "NoteOn", "Note" -> 30, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 23, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 23, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 23, "NoteOn", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 23, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 24, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 24, "NoteOff", "Note" -> 30, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 24, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 24, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 24, "NoteOff", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 24, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 24, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 24, "NoteOn", "Note" -> 27, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 24, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 24, "NoteOn", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 24, "NoteOn", "Note" -> 87, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 25, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 25, "NoteOff", "Note" -> 27, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 25, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 25, "NoteOff", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 25, "NoteOff", "Note" -> 87, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 25, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 25, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 25, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 25, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 25, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 25, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 25, "NoteOn", "Note" -> 90, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 26, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 26, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 26, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 26, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 26, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 26, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 26, "NoteOff", "Note" -> 90, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 26, "NoteOn", "Note" -> 21, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 26, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 26, "NoteOn", "Note" -> 93, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 27, "NoteOff", "Note" -> 21, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 27, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 27, "NoteOff", "Note" -> 93, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 27, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 27, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 27, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 27, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 27, "NoteOn", "Note" -> 90, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 27, "NoteOn", "Note" -> 96, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 28, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 28, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 28, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 28, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 28, "NoteOff", "Note" -> 90, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 28, "NoteOff", "Note" -> 96, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 28, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 28, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 28, "NoteOn", "Note" -> 87, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 28, "NoteOn", "Note" -> 99, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 29, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 29, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 29, "NoteOff", "Note" -> 87, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 29, "NoteOff", "Note" -> 99, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 29, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 29, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 29, "NoteOn", "Note" -> 96, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 29, "NoteOn", "Note" -> 102, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 30, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 30, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 30, "NoteOff", "Note" -> 96, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 30, "NoteOff", "Note" -> 102, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 30, "NoteOn", "Note" -> 105, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 31, "NoteOff", "Note" -> 105, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 31, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 31, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 31, "NoteOn", "Note" -> 96, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 31, "NoteOn", "Note" -> 102, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 32, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 32, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 32, "NoteOff", "Note" -> 96, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 32, "NoteOff", "Note" -> 102, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 32, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 32, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 32, "NoteOn", "Note" -> 87, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 32, "NoteOn", "Note" -> 99, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 33, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 33, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 33, "NoteOff", "Note" -> 87, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 33, "NoteOff", "Note" -> 99, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 33, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 33, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 33, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 33, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 33, "NoteOn", "Note" -> 90, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 33, "NoteOn", "Note" -> 96, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 34, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 34, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 34, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 34, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 34, "NoteOff", "Note" -> 90, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 34, "NoteOff", "Note" -> 96, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 34, "NoteOn", "Note" -> 21, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 34, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 34, "NoteOn", "Note" -> 93, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 35, "NoteOff", "Note" -> 21, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 35, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 35, "NoteOff", "Note" -> 93, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 35, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 35, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 35, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 35, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 35, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 35, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 35, "NoteOn", "Note" -> 90, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 36, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 36, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 36, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 36, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 36, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 36, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 36, "NoteOff", "Note" -> 90, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 36, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 36, "NoteOn", "Note" -> 27, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 36, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 36, "NoteOn", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 36, "NoteOn", "Note" -> 87, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 37, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 37, "NoteOff", "Note" -> 27, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 37, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 37, "NoteOff", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 37, "NoteOff", "Note" -> 87, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 37, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 37, "NoteOn", "Note" -> 30, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 37, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 37, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 37, "NoteOn", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 37, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 38, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 38, "NoteOff", "Note" -> 30, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 38, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 38, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 38, "NoteOff", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 38, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 38, "NoteOn", "Note" -> 33, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 38, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 39, "NoteOff", "Note" -> 33, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 39, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 39, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 39, "NoteOn", "Note" -> 30, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 39, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 39, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 39, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 39, "NoteOn", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 40, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 40, "NoteOff", "Note" -> 30, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 40, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 40, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 40, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 40, "NoteOff", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 40, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 40, "NoteOn", "Note" -> 27, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 40, "NoteOn", "Note" -> 39, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 40, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 40, "NoteOn", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 41, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 41, "NoteOff", "Note" -> 27, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 41, "NoteOff", "Note" -> 39, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 41, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 41, "NoteOff", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 41, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 41, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 41, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 41, "NoteOn", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 41, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 41, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 41, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 42, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 42, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 42, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 42, "NoteOff", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 42, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 42, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 42, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 42, "NoteOn", "Note" -> 21, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 42, "NoteOn", "Note" -> 45, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 42, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 43, "NoteOff", "Note" -> 21, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 43, "NoteOff", "Note" -> 45, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 43, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 43, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 43, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 43, "NoteOn", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 43, "NoteOn", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 43, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 43, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 44, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 44, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 44, "NoteOff", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 44, "NoteOff", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 44, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 44, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 44, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 44, "NoteOn", "Note" -> 39, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 44, "NoteOn", "Note" -> 51, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 44, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 45, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 45, "NoteOff", "Note" -> 39, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 45, "NoteOff", "Note" -> 51, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 45, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 45, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 45, "NoteOn", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 45, "NoteOn", "Note" -> 54, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 45, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 46, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 46, "NoteOff", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 46, "NoteOff", "Note" -> 54, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 46, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 46, "NoteOn", "Note" -> 57, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 47, "NoteOff", "Note" -> 57, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 47, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 47, "NoteOn", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 47, "NoteOn", "Note" -> 54, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 48, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 48, "NoteOff", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 48, "NoteOff", "Note" -> 54, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 48, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 48, "NoteOn", "Note" -> 39, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 48, "NoteOn", "Note" -> 51, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 49, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 49, "NoteOff", "Note" -> 39, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 49, "NoteOff", "Note" -> 51, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 49, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 49, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 49, "NoteOn", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 49, "NoteOn", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 50, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 50, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 50, "NoteOff", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 50, "NoteOff", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 50, "NoteOn", "Note" -> 21, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 50, "NoteOn", "Note" -> 45, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 51, "NoteOff", "Note" -> 21, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 51, "NoteOff", "Note" -> 45, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 51, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 51, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 51, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 51, "NoteOn", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 52, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 52, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 52, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 52, "NoteOff", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 52, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 52, "NoteOn", "Note" -> 27, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 52, "NoteOn", "Note" -> 39, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 53, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 53, "NoteOff", "Note" -> 27, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 53, "NoteOff", "Note" -> 39, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 53, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 53, "NoteOn", "Note" -> 30, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 53, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 54, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 54, "NoteOff", "Note" -> 30, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 54, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 54, "NoteOn", "Note" -> 33, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 55, "NoteOff", "Note" -> 33, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 55, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 55, "NoteOn", "Note" -> 30, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 56, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 56, "NoteOff", "Note" -> 30, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 56, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 56, "NoteOn", "Note" -> 27, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 57, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 57, "NoteOff", "Note" -> 27, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 57, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 57, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 58, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 58, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 58, "NoteOn", "Note" -> 21, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 59, "NoteOff", "Note" -> 21, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 59, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 60, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 60, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 61, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 10] =!= 
            None, 
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 1, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 1, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 2, "NoteOn", "Note" -> 21, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOff", "Note" -> 21, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 3, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 3, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 4, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 4, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 4, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 4, "NoteOn", "Note" -> 27, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 5, "NoteOff", "Note" -> 27, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 5, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 5, "NoteOn", "Note" -> 30, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 6, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 6, "NoteOff", "Note" -> 30, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 6, "NoteOn", "Note" -> 33, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 7, "NoteOff", "Note" -> 33, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 7, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 7, "NoteOn", "Note" -> 30, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 7, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 8, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 8, "NoteOff", "Note" -> 30, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 8, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 8, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 8, "NoteOn", "Note" -> 27, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 8, "NoteOn", "Note" -> 39, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 9, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 9, "NoteOff", "Note" -> 27, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 9, "NoteOff", "Note" -> 39, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 9, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 9, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 9, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 9, "NoteOn", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 10, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 10, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 10, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 10, "NoteOff", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 10, "NoteOn", "Note" -> 21, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 10, "NoteOn", "Note" -> 45, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 11, "NoteOff", "Note" -> 21, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 11, "NoteOff", "Note" -> 45, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 11, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 11, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 11, "NoteOn", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 11, "NoteOn", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 12, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 12, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 12, "NoteOff", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 12, "NoteOff", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 12, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 12, "NoteOn", "Note" -> 39, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 12, "NoteOn", "Note" -> 51, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 13, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 13, "NoteOff", "Note" -> 39, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 13, "NoteOff", "Note" -> 51, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 13, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 13, "NoteOn", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 13, "NoteOn", "Note" -> 54, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 14, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 14, "NoteOff", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 14, "NoteOff", "Note" -> 54, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 14, "NoteOn", "Note" -> 57, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 15, "NoteOff", "Note" -> 57, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 15, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 15, "NoteOn", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 15, "NoteOn", "Note" -> 54, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 15, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 16, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 16, "NoteOff", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 16, "NoteOff", "Note" -> 54, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 16, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 16, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 16, "NoteOn", "Note" -> 39, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 16, "NoteOn", "Note" -> 51, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 16, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 17, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 17, "NoteOff", "Note" -> 39, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 17, "NoteOff", "Note" -> 51, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 17, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 17, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 17, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 17, "NoteOn", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 17, "NoteOn", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 17, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 17, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 18, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 18, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 18, "NoteOff", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 18, "NoteOff", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 18, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 18, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 18, "NoteOn", "Note" -> 21, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 18, "NoteOn", "Note" -> 45, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 18, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 19, "NoteOff", "Note" -> 21, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 19, "NoteOff", "Note" -> 45, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 19, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 19, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 19, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 19, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 19, "NoteOn", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 19, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 19, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 19, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 20, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 20, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 20, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 20, "NoteOff", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 20, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 20, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 20, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 20, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 20, "NoteOn", "Note" -> 27, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 20, "NoteOn", "Note" -> 39, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 20, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 20, "NoteOn", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 21, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 21, "NoteOff", "Note" -> 27, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 21, "NoteOff", "Note" -> 39, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 21, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 21, "NoteOff", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 21, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 21, "NoteOn", "Note" -> 30, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 21, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 21, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 21, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 21, "NoteOn", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 22, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 22, "NoteOff", "Note" -> 30, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 22, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 22, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 22, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 22, "NoteOff", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 22, "NoteOn", "Note" -> 33, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 22, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 23, "NoteOff", "Note" -> 33, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 23, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 23, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 23, "NoteOn", "Note" -> 30, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 23, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 23, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 23, "NoteOn", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 23, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 24, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 24, "NoteOff", "Note" -> 30, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 24, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 24, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 24, "NoteOff", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 24, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 24, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 24, "NoteOn", "Note" -> 27, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 24, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 24, "NoteOn", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 24, "NoteOn", "Note" -> 87, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 25, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 25, "NoteOff", "Note" -> 27, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 25, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 25, "NoteOff", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 25, "NoteOff", "Note" -> 87, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 25, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 25, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 25, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 25, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 25, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 25, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 25, "NoteOn", "Note" -> 90, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 26, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 26, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 26, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 26, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 26, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 26, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 26, "NoteOff", "Note" -> 90, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 26, "NoteOn", "Note" -> 21, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 26, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 26, "NoteOn", "Note" -> 93, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 27, "NoteOff", "Note" -> 21, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 27, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 27, "NoteOff", "Note" -> 93, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 27, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 27, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 27, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 27, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 27, "NoteOn", "Note" -> 90, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 27, "NoteOn", "Note" -> 96, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 28, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 28, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 28, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 28, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 28, "NoteOff", "Note" -> 90, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 28, "NoteOff", "Note" -> 96, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 28, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 28, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 28, "NoteOn", "Note" -> 87, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 28, "NoteOn", "Note" -> 99, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 29, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 29, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 29, "NoteOff", "Note" -> 87, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 29, "NoteOff", "Note" -> 99, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 29, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 29, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 29, "NoteOn", "Note" -> 96, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 29, "NoteOn", "Note" -> 102, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 30, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 30, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 30, "NoteOff", "Note" -> 96, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 30, "NoteOff", "Note" -> 102, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 30, "NoteOn", "Note" -> 105, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 31, "NoteOff", "Note" -> 105, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 31, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 31, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 31, "NoteOn", "Note" -> 96, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 31, "NoteOn", "Note" -> 102, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 32, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 32, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 32, "NoteOff", "Note" -> 96, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 32, "NoteOff", "Note" -> 102, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 32, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 32, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 32, "NoteOn", "Note" -> 87, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 32, "NoteOn", "Note" -> 99, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 33, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 33, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 33, "NoteOff", "Note" -> 87, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 33, "NoteOff", "Note" -> 99, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 33, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 33, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 33, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 33, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 33, "NoteOn", "Note" -> 90, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 33, "NoteOn", "Note" -> 96, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 34, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 34, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 34, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 34, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 34, "NoteOff", "Note" -> 90, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 34, "NoteOff", "Note" -> 96, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 34, "NoteOn", "Note" -> 21, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 34, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 34, "NoteOn", "Note" -> 93, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 35, "NoteOff", "Note" -> 21, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 35, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 35, "NoteOff", "Note" -> 93, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 35, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 35, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 35, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 35, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 35, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 35, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 35, "NoteOn", "Note" -> 90, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 36, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 36, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 36, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 36, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 36, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 36, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 36, "NoteOff", "Note" -> 90, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 36, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 36, "NoteOn", "Note" -> 27, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 36, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 36, "NoteOn", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 36, "NoteOn", "Note" -> 87, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 37, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 37, "NoteOff", "Note" -> 27, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 37, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 37, "NoteOff", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 37, "NoteOff", "Note" -> 87, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 37, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 37, "NoteOn", "Note" -> 30, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 37, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 37, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 37, "NoteOn", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 37, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 38, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 38, "NoteOff", "Note" -> 30, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 38, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 38, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 38, "NoteOff", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 38, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 38, "NoteOn", "Note" -> 33, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 38, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 39, "NoteOff", "Note" -> 33, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 39, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 39, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 39, "NoteOn", "Note" -> 30, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 39, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 39, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 39, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 39, "NoteOn", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 40, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 40, "NoteOff", "Note" -> 30, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 40, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 40, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 40, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 40, "NoteOff", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 40, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 40, "NoteOn", "Note" -> 27, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 40, "NoteOn", "Note" -> 39, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 40, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 40, "NoteOn", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 41, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 41, "NoteOff", "Note" -> 27, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 41, "NoteOff", "Note" -> 39, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 41, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 41, "NoteOff", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 41, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 41, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 41, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 41, "NoteOn", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 41, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 41, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 41, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 42, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 42, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 42, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 42, "NoteOff", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 42, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 42, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 42, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 42, "NoteOn", "Note" -> 21, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 42, "NoteOn", "Note" -> 45, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 42, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 43, "NoteOff", "Note" -> 21, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 43, "NoteOff", "Note" -> 45, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 43, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 43, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 43, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 43, "NoteOn", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 43, "NoteOn", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 43, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 43, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 44, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 44, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 44, "NoteOff", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 44, "NoteOff", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 44, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 44, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 44, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 44, "NoteOn", "Note" -> 39, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 44, "NoteOn", "Note" -> 51, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 44, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 45, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 45, "NoteOff", "Note" -> 39, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 45, "NoteOff", "Note" -> 51, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 45, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 45, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 45, "NoteOn", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 45, "NoteOn", "Note" -> 54, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 45, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 46, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 46, "NoteOff", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 46, "NoteOff", "Note" -> 54, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 46, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 46, "NoteOn", "Note" -> 57, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 47, "NoteOff", "Note" -> 57, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 47, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 47, "NoteOn", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 47, "NoteOn", "Note" -> 54, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 48, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 48, "NoteOff", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 48, "NoteOff", "Note" -> 54, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 48, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 48, "NoteOn", "Note" -> 39, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 48, "NoteOn", "Note" -> 51, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 49, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 49, "NoteOff", "Note" -> 39, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 49, "NoteOff", "Note" -> 51, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 49, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 49, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 49, "NoteOn", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 49, "NoteOn", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 50, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 50, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 50, "NoteOff", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 50, "NoteOff", "Note" -> 48, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 50, "NoteOn", "Note" -> 21, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 50, "NoteOn", "Note" -> 45, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 51, "NoteOff", "Note" -> 21, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 51, "NoteOff", "Note" -> 45, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 51, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 51, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 51, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 51, "NoteOn", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 52, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 52, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 52, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 52, "NoteOff", "Note" -> 42, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 52, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 52, "NoteOn", "Note" -> 27, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 52, "NoteOn", "Note" -> 39, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 53, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 53, "NoteOff", "Note" -> 27, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 53, "NoteOff", "Note" -> 39, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 53, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 53, "NoteOn", "Note" -> 30, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 53, "NoteOn", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 54, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 54, "NoteOff", "Note" -> 30, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 54, "NoteOff", "Note" -> 36, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 54, "NoteOn", "Note" -> 33, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 55, "NoteOff", "Note" -> 33, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 55, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 55, "NoteOn", "Note" -> 30, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 56, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 56, "NoteOff", "Note" -> 30, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 56, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 56, "NoteOn", "Note" -> 27, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 57, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 57, "NoteOff", "Note" -> 27, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 57, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 57, "NoteOn", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 58, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 58, "NoteOff", "Note" -> 24, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 58, "NoteOn", "Note" -> 21, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 59, "NoteOff", "Note" -> 21, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 59, "NoteOn", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 60, "NoteOff", "Note" -> 18, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 60, "NoteOn", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 61, "NoteOff", "Note" -> 15, "Channel" -> 0, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 10]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[{
    SoundNote[{-45}, 0.1], 
    SoundNote[{-42}, 0.1], 
    SoundNote[{-39}, 0.1], 
    SoundNote[{-42, -36}, 0.1], 
    SoundNote[{-45, -33}, 0.1], 
    SoundNote[{-36, -30}, 0.1], 
    SoundNote[{-27}, 0.1], 
    SoundNote[{-36, -30, -24}, 0.1], 
    SoundNote[{-45, -33, -21}, 0.1], 
    SoundNote[{-42, -36, -24, -18}, 0.1], 
    SoundNote[{-39, -15}, 0.1], 
    SoundNote[{-42, -24, -18, -12}, 0.1], 
    SoundNote[{-45, -21, -9}, 0.1], 
    SoundNote[{-24, -12, -6}, 0.1], 
    SoundNote[{-3}, 0.1], 
    SoundNote[{-24, -12, -6, 0}, 0.1], 
    SoundNote[{-45, -21, -9, 3}, 0.1], 
    SoundNote[{-42, -24, -18, -12, 0, 6}, 0.1], 
    SoundNote[{-39, -15, 9}, 0.1], 
    SoundNote[{-42, -36, -24, -18, 0, 6, 12}, 0.1], 
    SoundNote[{-45, -33, -21, 3, 15}, 0.1], 
    SoundNote[{-36, -30, -24, 0, 12, 18}, 0.1], 
    SoundNote[{-27, 21}, 0.1], 
    SoundNote[{-36, -30, 0, 12, 18, 24}, 0.1], 
    SoundNote[{-45, -33, 3, 15, 27}, 0.1], 
    SoundNote[{-42, -36, 0, 6, 12, 24, 30}, 0.1], 
    SoundNote[{-39, 9, 33}, 0.1], 
    SoundNote[{-42, 0, 6, 24, 30, 36}, 0.1], 
    SoundNote[{-45, 3, 27, 39}, 0.1], 
    SoundNote[{0, 24, 36, 42}, 0.1], 
    SoundNote[{45}, 0.1], 
    SoundNote[{0, 24, 36, 42}, 0.1], 
    SoundNote[{-45, 3, 27, 39}, 0.1], 
    SoundNote[{-42, 0, 6, 24, 30, 36}, 0.1], 
    SoundNote[{-39, 9, 33}, 0.1], 
    SoundNote[{-42, -36, 0, 6, 12, 24, 30}, 0.1], 
    SoundNote[{-45, -33, 3, 15, 27}, 0.1], 
    SoundNote[{-36, -30, 0, 12, 18, 24}, 0.1], 
    SoundNote[{-27, 21}, 0.1], 
    SoundNote[{-36, -30, -24, 0, 12, 18}, 0.1], 
    SoundNote[{-45, -33, -21, 3, 15}, 0.1], 
    SoundNote[{-42, -36, -24, -18, 0, 6, 12}, 0.1], 
    SoundNote[{-39, -15, 9}, 0.1], 
    SoundNote[{-42, -24, -18, -12, 0, 6}, 0.1], 
    SoundNote[{-45, -21, -9, 3}, 0.1], 
    SoundNote[{-24, -12, -6, 0}, 0.1], 
    SoundNote[{-3}, 0.1], 
    SoundNote[{-24, -12, -6}, 0.1], 
    SoundNote[{-45, -21, -9}, 0.1], 
    SoundNote[{-42, -24, -18, -12}, 0.1], 
    SoundNote[{-39, -15}, 0.1], 
    SoundNote[{-42, -36, -24, -18}, 0.1], 
    SoundNote[{-45, -33, -21}, 0.1], 
    SoundNote[{-36, -30, -24}, 0.1], 
    SoundNote[{-27}, 0.1], 
    SoundNote[{-36, -30}, 0.1], 
    SoundNote[{-45, -33}, 0.1], 
    SoundNote[{-42, -36}, 0.1], 
    SoundNote[{-39}, 0.1], 
    SoundNote[{-42}, 0.1], 
    SoundNote[{-45}, 0.1]}]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->9157080]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->164005995],

Cell["Play a random sequence of notes from different instruments:", \
"ExampleText",
 CellID->314946192],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"SoundNote", "[", 
      RowBox[{"#", ",", "1", ",", 
       RowBox[{"RandomChoice", "[", 
        RowBox[{"{", 
         RowBox[{"\"\<Piano\>\"", ",", "\"\<Cello\>\"", ",", "\"\<Tuba\>\""}],
          "}"}], "]"}]}], "]"}], "&"}], " ", "/@", " ", 
    RowBox[{"RandomInteger", "[", 
     RowBox[{"12", ",", " ", "30"}], "]"}]}], ",", " ", "4"}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->84590558],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {GrayLevel[0.9], RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[{
        {RGBColor[0.5, 0, 0.5], 
         StyleBox[{
           RectangleBox[{1.2, 0.43198696602798653`}, \
{1.3266666666666667`, 0.4438917279327484}], 
           RectangleBox[{1.3333333333333333`, 0.43198696602798653`}, \
{1.46, 0.4438917279327484}], 
           RectangleBox[{3.7333333333333334`, 0.45579648983751037`}, \
{3.86, 0.4677012517422721}], 
           RectangleBox[{0.8, 0.4677012517422721}, \
{0.9266666666666666, 0.4796060136470343}], 
           RectangleBox[{0., 0.4796060136470343}, \
{0.12666666666666668`, 0.49151077555179606`}], 
           RectangleBox[{3.6, 0.49151077555179606`}, \
{3.7266666666666666`, 0.5034155374565581}], 
           RectangleBox[{1.4666666666666666`, 0.5034155374565581}, \
{1.5933333333333335`, 0.5153202993613198}], 
           RectangleBox[{2.533333333333333, 0.5153202993613198}, \
{2.6599999999999997`, 0.5272250612660818}], 
           RectangleBox[{3.3333333333333335`, 0.5153202993613198}, \
{3.46, 0.5272250612660818}], 
           RectangleBox[{0.5333333333333333, 0.5272250612660818}, \
{0.6599999999999999, 0.5391298231708438}], 
           RectangleBox[{1.7333333333333334`, 0.5391298231708438}, \
{1.86, 0.5510345850756055}], 
           RectangleBox[{0.26666666666666666`, 0.5510345850756055}, \
{0.39333333333333337`, 0.5629393469803675}], 
           RectangleBox[{1.6, 0.5510345850756055}, \
{1.7266666666666668`, 0.5629393469803675}], 
           RectangleBox[{2.6666666666666665`, 0.5629393469803675}, \
{2.793333333333333, 0.5748441088851295}]},
          Antialiasing->True]}, 
        {Hue[0.22538543343129763`, 0.9, 0.7], 
         StyleBox[{
           RectangleBox[{0.13333333333333333`, 0.45579648983751037`}, \
{0.26, 0.4677012517422721}], 
           RectangleBox[{0.9333333333333333, 0.4677012517422721}, \
{1.06, 0.4796060136470343}], 
           RectangleBox[{3.066666666666667, 0.4677012517422721}, \
{3.1933333333333334`, 0.4796060136470343}], 
           RectangleBox[{3.2, 0.4796060136470343}, \
{3.3266666666666667`, 0.49151077555179606`}], 
           RectangleBox[{3.8666666666666667`, 0.49151077555179606`}, \
{3.993333333333333, 0.5034155374565581}], 
           RectangleBox[{2.1333333333333333`, 0.5153202993613198}, \
{2.26, 0.5272250612660818}], 
           RectangleBox[{2.8, 0.5153202993613198}, \
{2.9266666666666663`, 0.5272250612660818}], 
           RectangleBox[{2.2666666666666666`, 0.5510345850756055}, \
{2.393333333333333, 0.5629393469803675}], 
           RectangleBox[{3.466666666666667, 0.5510345850756055}, \
{3.5933333333333333`, 0.5629393469803675}], 
           RectangleBox[{1.8666666666666667`, 0.5629393469803675}, \
{1.9933333333333334`, 0.5748441088851295}]},
          Antialiasing->True]}, 
        {Hue[0.3881114562337579, 0.9, 0.7], 
         StyleBox[{
           RectangleBox[{1.0666666666666667`, 0.43198696602798653`}, \
{1.1933333333333334`, 0.4438917279327484}], 
           RectangleBox[{0.4, 0.45579648983751037`}, \
{0.5266666666666666, 0.4677012517422721}], 
           RectangleBox[{2.933333333333333, 0.45579648983751037`}, \
{3.06, 0.4677012517422721}], 
           RectangleBox[{2., 0.4677012517422721}, \
{2.1266666666666665`, 0.4796060136470343}], 
           RectangleBox[{0.6666666666666666, 0.4796060136470343}, \
{0.7933333333333333, 0.49151077555179606`}], 
           RectangleBox[{2.4, 0.5153202993613198}, \
{2.5266666666666664`, 0.5272250612660818}]},
          Antialiasing->True]}},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       PlotRange->All,
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]], InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->GrayLevel[0.9]]}, InsetBox[
     StyleBox[
      InterpretationBox[
       InterpretationBox[
        RowBox[{"4", "\[InvisibleSpace]", "\<\" s\"\>"}],
        Row[{4, " s"}]],
       Row[{
         Row[{4, " s"}]}, 
        Style[" | ", FontSize -> 14, FontColor -> GrayLevel[0.75]]]], 
      "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{5, -158}, {29, -135}]}, 
        {GrayLevel[0.3], 
         PolygonBox[
          NCache[{{13, -153}, {13, -140}, {21, Rational[-293, 2]}, {
            13, -153}}, {{13, -153}, {13, -140}, {21, -146.5}, {
            13, -153}}]]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{5, -158}, {5, -135}, {29, -135}, {29, -158}, {5, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{"MouseClicked" :> (If[Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 4, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 6, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 8, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 10, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 12, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 14, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 18, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 20, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 20, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 22, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 22, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 24, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 24, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 26, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 26, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 28, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 38, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 40, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 40, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 42, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 50, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 52, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 54, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 56, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 56, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 58, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 1, "Value" -> 42], 
                 Sound`MIDIEvent[
                 2, "NoteOn", "Note" -> 62, "Channel" -> 1, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 4, "NoteOff", "Note" -> 62, "Channel" -> 1, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 14, "NoteOn", "Note" -> 63, "Channel" -> 1, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 16, "NoteOff", "Note" -> 63, "Channel" -> 1, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 28, "NoteOn", "Note" -> 71, "Channel" -> 1, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 30, "NoteOff", "Note" -> 71, "Channel" -> 1, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 32, "NoteOn", "Note" -> 67, "Channel" -> 1, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 34, "NoteOff", "Note" -> 67, "Channel" -> 1, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 34, "NoteOn", "Note" -> 70, "Channel" -> 1, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 36, "NoteOff", "Note" -> 70, "Channel" -> 1, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 42, "NoteOn", "Note" -> 67, "Channel" -> 1, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 44, "NoteOff", "Note" -> 67, "Channel" -> 1, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 46, "NoteOn", "Note" -> 63, "Channel" -> 1, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 48, "NoteOff", "Note" -> 63, "Channel" -> 1, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 48, "NoteOn", "Note" -> 64, "Channel" -> 1, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 50, "NoteOff", "Note" -> 64, "Channel" -> 1, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 52, "NoteOn", "Note" -> 70, "Channel" -> 1, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 54, "NoteOff", "Note" -> 70, "Channel" -> 1, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 58, "NoteOn", "Note" -> 65, "Channel" -> 1, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 60, "NoteOff", "Note" -> 65, "Channel" -> 1, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 2, "Value" -> 58], 
                 Sound`MIDIEvent[
                 6, "NoteOn", "Note" -> 62, "Channel" -> 2, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 8, "NoteOff", "Note" -> 62, "Channel" -> 2, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 10, "NoteOn", "Note" -> 64, "Channel" -> 2, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 12, "NoteOff", "Note" -> 64, "Channel" -> 2, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 16, "NoteOn", "Note" -> 60, "Channel" -> 2, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 18, "NoteOff", "Note" -> 60, "Channel" -> 2, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 30, "NoteOn", "Note" -> 63, "Channel" -> 2, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 32, "NoteOff", "Note" -> 63, "Channel" -> 2, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 36, "NoteOn", "Note" -> 67, "Channel" -> 2, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 38, "NoteOff", "Note" -> 67, "Channel" -> 2, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 44, "NoteOn", "Note" -> 62, "Channel" -> 2, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 46, "NoteOff", "Note" -> 62, "Channel" -> 2, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 15] =!= 
            None, 
            Sound`EmitMIDI[
             Sound`MIDISequence[{
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                 Sound`MIDIEvent[
                 0, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 2, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 4, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 6, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 8, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 10, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 12, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 14, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 18, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 20, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 20, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 22, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 22, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 24, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 24, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 26, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 26, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 28, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 38, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 40, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 40, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 42, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 50, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 52, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 54, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 56, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 56, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 58, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 1, "Value" -> 42], 
                 Sound`MIDIEvent[
                 2, "NoteOn", "Note" -> 62, "Channel" -> 1, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 4, "NoteOff", "Note" -> 62, "Channel" -> 1, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 14, "NoteOn", "Note" -> 63, "Channel" -> 1, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 16, "NoteOff", "Note" -> 63, "Channel" -> 1, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 28, "NoteOn", "Note" -> 71, "Channel" -> 1, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 30, "NoteOff", "Note" -> 71, "Channel" -> 1, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 32, "NoteOn", "Note" -> 67, "Channel" -> 1, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 34, "NoteOff", "Note" -> 67, "Channel" -> 1, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 34, "NoteOn", "Note" -> 70, "Channel" -> 1, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 36, "NoteOff", "Note" -> 70, "Channel" -> 1, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 42, "NoteOn", "Note" -> 67, "Channel" -> 1, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 44, "NoteOff", "Note" -> 67, "Channel" -> 1, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 46, "NoteOn", "Note" -> 63, "Channel" -> 1, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 48, "NoteOff", "Note" -> 63, "Channel" -> 1, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 48, "NoteOn", "Note" -> 64, "Channel" -> 1, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 50, "NoteOff", "Note" -> 64, "Channel" -> 1, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 52, "NoteOn", "Note" -> 70, "Channel" -> 1, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 54, "NoteOff", "Note" -> 70, "Channel" -> 1, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 58, "NoteOn", "Note" -> 65, "Channel" -> 1, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 60, "NoteOff", "Note" -> 65, "Channel" -> 1, "Velocity" -> 
                  0]}], 
               Sound`MIDITrack[{
                 Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                 Sound`MIDIEvent[
                 0, "ProgramCommand", "Channel" -> 2, "Value" -> 58], 
                 Sound`MIDIEvent[
                 6, "NoteOn", "Note" -> 62, "Channel" -> 2, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 8, "NoteOff", "Note" -> 62, "Channel" -> 2, "Velocity" -> 0],
                  
                 Sound`MIDIEvent[
                 10, "NoteOn", "Note" -> 64, "Channel" -> 2, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 12, "NoteOff", "Note" -> 64, "Channel" -> 2, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 16, "NoteOn", "Note" -> 60, "Channel" -> 2, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 18, "NoteOff", "Note" -> 60, "Channel" -> 2, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 30, "NoteOn", "Note" -> 63, "Channel" -> 2, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 32, "NoteOff", "Note" -> 63, "Channel" -> 2, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 36, "NoteOn", "Note" -> 67, "Channel" -> 2, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 38, "NoteOff", "Note" -> 67, "Channel" -> 2, "Velocity" -> 
                  0], 
                 Sound`MIDIEvent[
                 44, "NoteOn", "Note" -> 62, "Channel" -> 2, "Velocity" -> 
                  127], 
                 Sound`MIDIEvent[
                 46, "NoteOff", "Note" -> 62, "Channel" -> 2, "Velocity" -> 
                  0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 15]]]; 
          Sound`SoundDump`emitSoundThroughFE[None, "Preemptive"]), 
         PassEventsDown -> Automatic, PassEventsUp -> True}]],
      "\"Play\""],
     Annotation[#, "Play", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      TagBox[{
        {GrayLevel[0.9], RectangleBox[{34, -158}, {58, -135}]}, 
        {GrayLevel[0.3], RectangleBox[{42, -150}, {50, -143}]}, 
        {GrayLevel[0.5], 
         StyleBox[
          LineBox[{{34, -158}, {34, -135}, {58, -135}, {58, -158}, {
           34, -158}}],
          Antialiasing->False]}},
       EventHandlerTag[{
        "MouseClicked" :> (Sound`SoundDump`emitSoundThroughFE[
            SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
          Sound`MIDIDump`stopMIDI[]), PassEventsDown -> Automatic, 
         PassEventsUp -> True}]],
      "\"Stop\""],
     Annotation[#, "Stop", "Tooltip"]& ], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->250,
   PlotRange->{{0, 360}, {0, -164}},
   PlotRangePadding->1],
  Sound[{
    SoundNote[4, 1, "Piano"], 
    SoundNote[2, 1, "Cello"], 
    SoundNote[10, 1, "Piano"], 
    SoundNote[2, 1, "Tuba"], 
    SoundNote[8, 1, "Piano"], 
    SoundNote[4, 1, "Tuba"], 
    SoundNote[3, 1, "Piano"], 
    SoundNote[3, 1, "Cello"], 
    SoundNote[0, 1, "Tuba"], 
    SoundNote[0, 1, "Piano"], 
    SoundNote[0, 1, "Piano"], 
    SoundNote[6, 1, "Piano"], 
    SoundNote[10, 1, "Piano"], 
    SoundNote[9, 1, "Piano"], 
    SoundNote[11, 1, "Cello"], 
    SoundNote[3, 1, "Tuba"], 
    SoundNote[7, 1, "Cello"], 
    SoundNote[10, 1, "Cello"], 
    SoundNote[7, 1, "Tuba"], 
    SoundNote[7, 1, "Piano"], 
    SoundNote[11, 1, "Piano"], 
    SoundNote[7, 1, "Cello"], 
    SoundNote[2, 1, "Tuba"], 
    SoundNote[3, 1, "Cello"], 
    SoundNote[4, 1, "Cello"], 
    SoundNote[7, 1, "Piano"], 
    SoundNote[10, 1, "Cello"], 
    SoundNote[5, 1, "Piano"], 
    SoundNote[2, 1, "Piano"], 
    SoundNote[5, 1, "Cello"]}, 4]]], "Output",
 ImageSize->{254, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->317603711]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->17587],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["Sound",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Sound"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SampledSoundList",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SampledSoundList"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SampledSoundFunction",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SampledSoundFunction"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Beep",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Beep"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["EmitSound",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/EmitSound"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->23064]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->21541],

Cell[TextData[ButtonBox["Sound",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/Sound"]], "Tutorials",
 CellID->439901354],

Cell[TextData[ButtonBox["The Representation of Sound",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/TheRepresentationOfSound"]], "Tutorials",
 CellID->5171]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection"],

Cell[TextData[{
 ButtonBox["Demonstrations with SoundNote",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/symbol.html?symbol=SoundNote"], 
    None}],
 " (",
 ButtonBox["Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None}],
 ")"
}], "RelatedLinks"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->124941842],

Cell[TextData[ButtonBox["Sound and Sonification",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/SoundAndSonification"]], "MoreAbout",
 CellID->593729836],

Cell[TextData[ButtonBox["New in 6.0: Visualization & Graphics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60VisualizationAndGraphics"]], "MoreAbout",
 CellID->247522525]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"SoundNote - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 21, 52.2000210}", "context" -> "System`", 
    "keywords" -> {
     "middle C", "music", "musical note", "clef", "scale", "arpeggio", 
      "sharp", "flat", "accordion", "agogo", "algorithmic composition", 
      "alto sax", "applause", "atmosphere", "audio", "bagpipe", "bandoneon", 
      "banjo", "baritone sax", "bass", "bassoon", "bell tree", "bird", 
      "blown bottle", "bowed", "brass section", "breath", "brightness", 
      "bright piano", "cabasa", "calliope", "castanets", "celesta", "cello", 
      "charang", "chiff", "chinese cymbal", "choir", "clap", "clarinet", 
      "claves", "clavi", "contrabass", "cowbell", "crystal", "drawbar organ", 
      "drum", "dulcimer", "echoes", "electric bass", "electric guitar", 
      "electric snare", "english horn", "fiddle", "fifths", "flute", 
      "french horn", "fretless bass", "fret noise", "glockenspiel", "goblins",
       "guiro long", "guiro short", "guitar", "guitar distorted", 
      "guitar harmonics", "guitar muted", "guitar overdriven", "gunshot", 
      "halo", "harmonica", "harp", "harpsichord", "helicopter", "high agogo", 
      "high bongo", "high timbale", "high tom", "high woodblock", 
      "instrument", "jazz guitar", "jingle bell", "kalimba", "koto", 
      "low agogo", "low bongo", "low conga", "low timbale", "low tom", 
      "low woodblock", "maracas", "marimba", "melodic tom", "metallic", 
      "metronome bell", "metronome click", "MIDI", "Moog", "music", 
      "music box", "mute cuica", "muted trumpet", "mute surdo", 
      "mute triangle", "new age", "note", "oboe", "ocarina", "open cuica", 
      "open surdo", "open triangle", "orchestra hit", "orchestration", 
      "organ", "pan flute", "percussion", "percussive organ", "piano", 
      "piccolo", "picked bass", "pizzicato strings", "polysynth", "rain", 
      "recorder", "reed organ", "reverse cymbal", "ride bell", "rock organ", 
      "sawtooth", "sci fi", "scratch pull", "scratch push", "seashore", 
      "sequencing", "shaker", "shakuhachi", "shamisen", "shanai", 
      "side stick", "sitar", "slap", "snare", "soprano sax", "sound note", 
      "soundtrack", "splash cymbal", "square", "square click", "steeldrums", 
      "steel guitar", "sticks", "string instrument", "strings1", "strings2", 
      "sweep", "symbolic sound", "synth", "synth drum", "synthesizer", 
      "synth voice", "taiko", "tambourine", "telephone", "tenor sax", 
      "timpani", "tinklebell", "tone", "tremolo strings", "trombone", 
      "trumpet", "tuba", "tubular bells", "vibraphone", "vibraslap", "viola", 
      "violin", "voice", "voice aahs", "voice oohs", "warm", "whistle", 
      "whistle long", "whistle short", "woodblock", "xylophone"}, "index" -> 
    True, "label" -> "Built-in Mathematica Symbol", "language" -> "en", 
    "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "SoundNote[pitch] represents a music-like sound note with the specified \
pitch. SoundNote[pitch, t] takes the note to have duration t. \
SoundNote[pitch, {t_min, t_max}] takes the note to occupy the time interval \
t_min to t_max. SoundNote[pitch, tspec, \" style\"] takes the note to be in \
the specified style. SoundNote[pitch, tspec, \" style\", opts] uses the \
specified rendering options for the note.", "synonyms" -> {"sound note"}, 
    "title" -> "SoundNote", "type" -> "Symbol", "uri" -> "ref/SoundNote"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[19062, 570, 358, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->1351969]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 927011, 19721}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2515, 58, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3138, 85, 1574, 37, 70, "ObjectNameGrid"],
Cell[4715, 124, 2652, 83, 70, "Usage",
 CellID->15192]
}, Open  ]],
Cell[CellGroupData[{
Cell[7404, 212, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[7882, 229, 75, 1, 70, "Notes",
 CellID->21502],
Cell[7960, 232, 2175, 65, 70, "2ColumnTableMod",
 CellID->31815],
Cell[10138, 299, 460, 14, 70, "Notes",
 CellID->10670],
Cell[10601, 315, 442, 13, 70, "Notes",
 CellID->30696],
Cell[11046, 330, 81, 1, 70, "Notes",
 CellID->28254],
Cell[11130, 333, 201, 7, 70, "Notes",
 CellID->31293],
Cell[11334, 342, 260, 8, 70, "Notes",
 CellID->129119450],
Cell[11597, 352, 487, 17, 70, "Notes",
 CellID->4300],
Cell[12087, 371, 327, 11, 70, "Notes",
 CellID->32573],
Cell[12417, 384, 233, 9, 70, "Notes",
 CellID->12361],
Cell[12653, 395, 281, 10, 70, "Notes",
 CellID->4331],
Cell[12937, 407, 498, 17, 70, "Notes",
 CellID->19175],
Cell[13438, 426, 264, 8, 70, "Notes",
 CellID->9143],
Cell[13705, 436, 69, 1, 70, "Notes",
 CellID->133771515],
Cell[13777, 439, 3135, 68, 70, "Notes",
 CellID->360898806],
Cell[16915, 509, 79, 1, 70, "Notes",
 CellID->69401903],
Cell[16997, 512, 1618, 35, 70, "Notes",
 CellID->179316135],
Cell[18618, 549, 270, 10, 70, "Notes",
 CellID->726003106],
Cell[18891, 561, 134, 4, 70, "Notes",
 CellID->475884268]
}, Closed]],
Cell[CellGroupData[{
Cell[19062, 570, 358, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->1351969],
Cell[CellGroupData[{
Cell[19445, 584, 148, 5, 70, "ExampleSection",
 CellID->202604790],
Cell[19596, 591, 62, 1, 70, "ExampleText",
 CellID->108123467],
Cell[CellGroupData[{
Cell[19683, 596, 137, 4, 28, "Input",
 CellID->690209635],
Cell[19823, 602, 4289, 110, 136, "Output",
 CellID->1825030]
}, Open  ]],
Cell[24127, 715, 125, 3, 70, "ExampleDelimiter",
 CellID->116191540],
Cell[24255, 720, 93, 1, 70, "ExampleText",
 CellID->714566645],
Cell[CellGroupData[{
Cell[24373, 725, 194, 6, 70, "Input",
 CellID->19781668],
Cell[24570, 733, 4309, 109, 70, "Output",
 CellID->181507460]
}, Open  ]],
Cell[28894, 845, 125, 3, 70, "ExampleDelimiter",
 CellID->314676768],
Cell[29022, 850, 72, 1, 70, "ExampleText",
 CellID->106519206],
Cell[CellGroupData[{
Cell[29119, 855, 259, 8, 70, "Input",
 CellID->2768197],
Cell[29381, 865, 4948, 124, 70, "Output",
 CellID->52871950]
}, Open  ]],
Cell[34344, 992, 125, 3, 70, "ExampleDelimiter",
 CellID->212688541],
Cell[34472, 997, 76, 1, 70, "ExampleText",
 CellID->88221508],
Cell[CellGroupData[{
Cell[34573, 1002, 297, 8, 70, "Input",
 CellID->701800872],
Cell[34873, 1012, 5565, 139, 70, "Output",
 CellID->85083618]
}, Open  ]],
Cell[40453, 1154, 125, 3, 70, "ExampleDelimiter",
 CellID->654504802],
Cell[40581, 1159, 76, 1, 70, "ExampleText",
 CellID->804139014],
Cell[CellGroupData[{
Cell[40682, 1164, 152, 4, 70, "Input",
 CellID->903232243],
Cell[40837, 1170, 4287, 109, 70, "Output",
 CellID->77763871]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[45173, 1285, 216, 7, 70, "ExampleSection",
 CellID->9374],
Cell[CellGroupData[{
Cell[45414, 1296, 243, 7, 70, "ExampleSubsection",
 CellID->793326280],
Cell[45660, 1305, 79, 1, 70, "ExampleText",
 CellID->117974802],
Cell[CellGroupData[{
Cell[45764, 1310, 136, 4, 70, "Input",
 CellID->67249194],
Cell[45903, 1316, 4279, 109, 70, "Output",
 CellID->701233626]
}, Open  ]],
Cell[50197, 1428, 79, 1, 70, "ExampleText",
 CellID->172890070],
Cell[CellGroupData[{
Cell[50301, 1433, 156, 5, 70, "Input",
 CellID->367949343],
Cell[50460, 1440, 4292, 110, 70, "Output",
 CellID->125515990]
}, Open  ]],
Cell[54767, 1553, 125, 3, 70, "ExampleDelimiter",
 CellID->323930242],
Cell[54895, 1558, 63, 1, 70, "ExampleText",
 CellID->24225762],
Cell[CellGroupData[{
Cell[54983, 1563, 144, 4, 70, "Input",
 CellID->93228870],
Cell[55130, 1569, 4281, 109, 70, "Output",
 CellID->259920389]
}, Open  ]],
Cell[59426, 1681, 124, 3, 70, "ExampleDelimiter",
 CellID->34259455],
Cell[59553, 1686, 52, 1, 70, "ExampleText",
 CellID->499030287],
Cell[CellGroupData[{
Cell[59630, 1691, 146, 4, 70, "Input",
 CellID->454451608],
Cell[59779, 1697, 4294, 110, 70, "Output",
 CellID->532719665]
}, Open  ]],
Cell[64088, 1810, 68, 1, 70, "ExampleText",
 CellID->74519669],
Cell[CellGroupData[{
Cell[64181, 1815, 146, 4, 70, "Input",
 CellID->188282611],
Cell[64330, 1821, 4281, 109, 70, "Output",
 CellID->90480637]
}, Open  ]],
Cell[68626, 1933, 69, 1, 70, "ExampleText",
 CellID->285967126],
Cell[CellGroupData[{
Cell[68720, 1938, 146, 4, 70, "Input",
 CellID->185619107],
Cell[68869, 1944, 4282, 109, 70, "Output",
 CellID->286581249]
}, Open  ]],
Cell[73166, 2056, 125, 3, 70, "ExampleDelimiter",
 CellID->217100863],
Cell[73294, 2061, 62, 1, 70, "ExampleText",
 CellID->260011936],
Cell[CellGroupData[{
Cell[73381, 2066, 201, 6, 70, "Input",
 CellID->35700781],
Cell[73585, 2074, 4915, 124, 70, "Output",
 CellID->79394894]
}, Open  ]],
Cell[78515, 2201, 125, 3, 70, "ExampleDelimiter",
 CellID->228695725],
Cell[78643, 2206, 76, 1, 70, "ExampleText",
 CellID->461969822],
Cell[CellGroupData[{
Cell[78744, 2211, 249, 8, 70, "Input",
 CellID->651703356],
Cell[78996, 2221, 6153, 151, 70, "Output",
 CellID->227768562]
}, Open  ]],
Cell[85164, 2375, 76, 1, 70, "ExampleText",
 CellID->65294128],
Cell[CellGroupData[{
Cell[85265, 2380, 250, 8, 70, "Input",
 CellID->579743871],
Cell[85518, 2390, 6150, 150, 70, "Output",
 CellID->883613282]
}, Open  ]],
Cell[91683, 2543, 124, 3, 70, "ExampleDelimiter",
 CellID->94975604],
Cell[91810, 2548, 64, 1, 70, "ExampleText",
 CellID->207328570],
Cell[CellGroupData[{
Cell[91899, 2553, 242, 7, 70, "Input",
 CellID->608930050],
Cell[92144, 2562, 4934, 125, 70, "Output",
 CellID->148631695]
}, Open  ]],
Cell[97093, 2690, 71, 1, 70, "ExampleText",
 CellID->36500127],
Cell[CellGroupData[{
Cell[97189, 2695, 147, 4, 70, "Input",
 CellID->201698962],
Cell[97339, 2701, 4283, 110, 70, "Output",
 CellID->142102853]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[101671, 2817, 226, 7, 70, "ExampleSubsection",
 CellID->65845541],
Cell[101900, 2826, 84, 1, 70, "ExampleText",
 CellID->303449142],
Cell[CellGroupData[{
Cell[102009, 2831, 296, 9, 70, "Input",
 CellID->188841332],
Cell[102308, 2842, 4957, 125, 70, "Output",
 CellID->284965416]
}, Open  ]],
Cell[107280, 2970, 125, 3, 70, "ExampleDelimiter",
 CellID->594028497],
Cell[107408, 2975, 75, 1, 70, "ExampleText",
 CellID->576630885],
Cell[CellGroupData[{
Cell[107508, 2980, 374, 11, 70, "Input",
 CellID->44056557],
Cell[107885, 2993, 4997, 126, 70, "Output",
 CellID->235247701]
}, Open  ]],
Cell[112897, 3122, 125, 3, 70, "ExampleDelimiter",
 CellID->136274979],
Cell[113025, 3127, 97, 1, 70, "ExampleText",
 CellID->137808773],
Cell[CellGroupData[{
Cell[113147, 3132, 408, 13, 70, "Input",
 CellID->520380054],
Cell[113558, 3147, 4969, 125, 70, "Output",
 CellID->386566153]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[118576, 3278, 236, 7, 70, "ExampleSubsection",
 CellID->151168849],
Cell[118815, 3287, 78, 1, 70, "ExampleText",
 CellID->139672495],
Cell[CellGroupData[{
Cell[118918, 3292, 194, 6, 70, "Input",
 CellID->178763726],
Cell[119115, 3300, 4322, 110, 70, "Output",
 CellID->278040955]
}, Open  ]],
Cell[123452, 3413, 124, 3, 70, "ExampleDelimiter",
 CellID->69789477],
Cell[123579, 3418, 87, 1, 70, "ExampleText",
 CellID->110887593],
Cell[CellGroupData[{
Cell[123691, 3423, 339, 10, 70, "Input",
 CellID->13674992],
Cell[124033, 3435, 5511, 136, 70, "Output",
 CellID->461283651]
}, Open  ]],
Cell[129559, 3574, 125, 3, 70, "ExampleDelimiter",
 CellID->398357265],
Cell[129687, 3579, 82, 1, 70, "ExampleText",
 CellID->333445715],
Cell[CellGroupData[{
Cell[129794, 3584, 454, 14, 70, "Input",
 CellID->41520424],
Cell[130251, 3600, 5532, 136, 70, "Output",
 CellID->788909719]
}, Open  ]],
Cell[135798, 3739, 125, 3, 70, "ExampleDelimiter",
 CellID->264388839],
Cell[135926, 3744, 74, 1, 70, "ExampleText",
 CellID->224690550],
Cell[CellGroupData[{
Cell[136025, 3749, 315, 8, 70, "Input",
 CellID->371217081],
Cell[136343, 3759, 6114, 151, 70, "Output",
 CellID->23075324]
}, Open  ]],
Cell[142472, 3913, 125, 3, 70, "ExampleDelimiter",
 CellID->936293599],
Cell[142600, 3918, 152, 5, 70, "ExampleText",
 CellID->477514323],
Cell[CellGroupData[{
Cell[142777, 3927, 197, 6, 70, "Input",
 CellID->345467706],
Cell[142977, 3935, 4321, 110, 70, "Output",
 CellID->500844854]
}, Open  ]],
Cell[CellGroupData[{
Cell[147335, 4050, 193, 6, 70, "Input",
 CellID->90695030],
Cell[147531, 4058, 4319, 110, 70, "Output",
 CellID->165537014]
}, Open  ]],
Cell[CellGroupData[{
Cell[151887, 4173, 198, 6, 70, "Input",
 CellID->119585382],
Cell[152088, 4181, 4326, 110, 70, "Output",
 CellID->16028380]
}, Open  ]],
Cell[156429, 4294, 124, 3, 70, "ExampleDelimiter",
 CellID->45046013],
Cell[156556, 4299, 83, 1, 70, "ExampleText",
 CellID->166729684],
Cell[CellGroupData[{
Cell[156664, 4304, 149, 4, 70, "Input",
 CellID->126948112],
Cell[156816, 4310, 4296, 110, 70, "Output",
 CellID->83281232]
}, Open  ]],
Cell[161127, 4423, 125, 3, 70, "ExampleDelimiter",
 CellID->351332568],
Cell[161255, 4428, 95, 1, 70, "ExampleText",
 CellID->537980816],
Cell[CellGroupData[{
Cell[161375, 4433, 180, 5, 70, "Input",
 CellID->137084481],
Cell[161558, 4440, 4315, 110, 70, "Output",
 CellID->237581513]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[165934, 4557, 222, 7, 70, "ExampleSection",
 CellID->349216433],
Cell[CellGroupData[{
Cell[166181, 4568, 232, 7, 70, "ExampleSubsection",
 CellID->136293263],
Cell[166416, 4577, 89, 1, 70, "ExampleText",
 CellID->322677866],
Cell[CellGroupData[{
Cell[166530, 4582, 359, 11, 70, "Input",
 CellID->561880915],
Cell[166892, 4595, 10985, 260, 70, "Output",
 CellID->75785747]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[177938, 4862, 223, 7, 70, "ExampleSection",
 CellID->30057],
Cell[178164, 4871, 87, 1, 70, "ExampleText",
 CellID->398354286],
Cell[CellGroupData[{
Cell[178276, 4876, 257, 8, 70, "Input",
 CellID->215259133],
Cell[178536, 4886, 11901, 285, 70, "Output",
 CellID->9648212]
}, Open  ]],
Cell[190452, 5174, 122, 3, 70, "ExampleDelimiter",
 CellID->462149],
Cell[190577, 5179, 105, 2, 70, "ExampleText",
 CellID->672107903],
Cell[CellGroupData[{
Cell[190707, 5185, 308, 9, 70, "Input",
 CellID->125790237],
Cell[191018, 5196, 12171, 290, 70, "Output",
 CellID->61475641]
}, Open  ]],
Cell[203204, 5489, 125, 3, 70, "ExampleDelimiter",
 CellID->363938610],
Cell[203332, 5494, 96, 1, 70, "ExampleText",
 CellID->306183704],
Cell[CellGroupData[{
Cell[203453, 5499, 373, 12, 70, "Input",
 CellID->268546659],
Cell[203829, 5513, 16844, 392, 70, "Output",
 CellID->15889590]
}, Open  ]],
Cell[220688, 5908, 125, 3, 70, "ExampleDelimiter",
 CellID->583997264],
Cell[220816, 5913, 93, 1, 70, "ExampleText",
 CellID->43962524],
Cell[CellGroupData[{
Cell[220934, 5918, 414, 12, 70, "Input",
 CellID->156060250],
Cell[221351, 5932, 6330, 155, 70, "Output",
 CellID->68245836]
}, Open  ]],
Cell[227696, 6090, 125, 3, 70, "ExampleDelimiter",
 CellID->203735909],
Cell[227824, 6095, 90, 1, 70, "ExampleText",
 CellID->479018329],
Cell[CellGroupData[{
Cell[227939, 6100, 367, 12, 70, "Input",
 CellID->211432710],
Cell[228309, 6114, 6958, 170, 70, "Output",
 CellID->44615779]
}, Open  ]],
Cell[235282, 6287, 125, 3, 70, "ExampleDelimiter",
 CellID->364666654],
Cell[235410, 6292, 79, 1, 70, "ExampleText",
 CellID->433073870],
Cell[CellGroupData[{
Cell[235514, 6297, 226, 7, 70, "Input",
 CellID->74789739],
Cell[235743, 6306, 23434, 545, 70, "Output",
 CellID->442109752]
}, Open  ]],
Cell[259192, 6854, 125, 3, 70, "ExampleDelimiter",
 CellID->221919902],
Cell[259320, 6859, 91, 1, 70, "ExampleText",
 CellID->421247838],
Cell[CellGroupData[{
Cell[259436, 6864, 835, 25, 70, "Input",
 CellID->333718560],
Cell[260274, 6891, 28738, 655, 70, "Output",
 CellID->257577381]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[289061, 7552, 233, 7, 70, "ExampleSection",
 CellID->17018],
Cell[289297, 7561, 85, 1, 70, "ExampleText",
 CellID->20068085],
Cell[CellGroupData[{
Cell[289407, 7566, 469, 15, 70, "Input",
 CellID->8917284],
Cell[289879, 7583, 182174, 3041, 70, "Output",
 CellID->323937806]
}, Open  ]],
Cell[472068, 10627, 125, 3, 70, "ExampleDelimiter",
 CellID->134976226],
Cell[472196, 10632, 91, 1, 70, "ExampleText",
 CellID->17035733],
Cell[CellGroupData[{
Cell[472312, 10637, 659, 21, 70, "Input",
 CellID->292662339],
Cell[472974, 10660, 229435, 3817, 70, "Output",
 CellID->966825440]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[702458, 14483, 226, 7, 70, "ExampleSection",
 CellID->18870],
Cell[702687, 14492, 189, 6, 70, "ExampleText",
 CellID->96937903],
Cell[CellGroupData[{
Cell[702901, 14502, 196, 6, 70, "Input",
 CellID->322244040],
Cell[703100, 14510, 4310, 109, 70, "Output",
 CellID->203917200]
}, Open  ]],
Cell[707425, 14622, 124, 3, 70, "ExampleDelimiter",
 CellID->11195117],
Cell[707552, 14627, 163, 5, 70, "ExampleText",
 CellID->91461644],
Cell[CellGroupData[{
Cell[707740, 14636, 181, 5, 70, "Input",
 CellID->545407989],
Cell[707924, 14643, 4319, 110, 70, "Output",
 CellID->363200134]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[712292, 14759, 223, 7, 70, "ExampleSection",
 CellID->9241],
Cell[712518, 14768, 103, 2, 70, "ExampleText",
 CellID->29124222],
Cell[CellGroupData[{
Cell[712646, 14774, 284, 9, 70, "Input",
 CellID->18839667],
Cell[712933, 14785, 20666, 474, 70, "Output",
 CellID->6342098]
}, Open  ]],
Cell[733614, 15262, 125, 3, 70, "ExampleDelimiter",
 CellID->179542823],
Cell[733742, 15267, 92, 1, 70, "ExampleText",
 CellID->176046018],
Cell[CellGroupData[{
Cell[733859, 15272, 171, 5, 70, "Input",
 CellID->269501302],
Cell[734033, 15279, 11067, 257, 70, "Output",
 CellID->784654678]
}, Open  ]],
Cell[745115, 15539, 124, 3, 70, "ExampleDelimiter",
 CellID->24733363],
Cell[745242, 15544, 73, 1, 70, "ExampleText",
 CellID->287204505],
Cell[CellGroupData[{
Cell[745340, 15549, 230, 8, 70, "Input",
 CellID->4385851],
Cell[745573, 15559, 9252, 219, 70, "Output",
 CellID->455522655]
}, Open  ]],
Cell[754840, 15781, 125, 3, 70, "ExampleDelimiter",
 CellID->118240812],
Cell[754968, 15786, 75, 1, 70, "ExampleText",
 CellID->120978039],
Cell[CellGroupData[{
Cell[755068, 15791, 705, 22, 70, "Input",
 CellID->135191944],
Cell[755776, 15815, 137918, 3100, 70, "Output",
 CellID->9157080]
}, Open  ]],
Cell[893709, 18918, 125, 3, 70, "ExampleDelimiter",
 CellID->164005995],
Cell[893837, 18923, 104, 2, 70, "ExampleText",
 CellID->314946192],
Cell[CellGroupData[{
Cell[893966, 18929, 494, 15, 70, "Input",
 CellID->84590558],
Cell[894463, 18946, 24890, 567, 70, "Output",
 CellID->317603711]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[919414, 19520, 311, 9, 70, "SeeAlsoSection",
 CellID->17587],
Cell[919728, 19531, 1214, 44, 70, "SeeAlso",
 CellID->23064]
}, Open  ]],
Cell[CellGroupData[{
Cell[920979, 19580, 314, 9, 70, "TutorialsSection",
 CellID->21541],
Cell[921296, 19591, 125, 3, 70, "Tutorials",
 CellID->439901354],
Cell[921424, 19596, 161, 3, 70, "Tutorials",
 CellID->5171]
}, Open  ]],
Cell[CellGroupData[{
Cell[921622, 19604, 305, 8, 70, "RelatedLinksSection"],
Cell[921930, 19614, 363, 12, 70, "RelatedLinks"]
}, Open  ]],
Cell[CellGroupData[{
Cell[922330, 19631, 319, 9, 70, "MoreAboutSection",
 CellID->124941842],
Cell[922652, 19642, 154, 3, 70, "MoreAbout",
 CellID->593729836],
Cell[922809, 19647, 179, 3, 70, "MoreAbout",
 CellID->247522525]
}, Open  ]],
Cell[923003, 19653, 27, 0, 70, "History"],
Cell[923033, 19655, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

