(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     30494,       1061]
NotebookOptionsPosition[     24565,        856]
NotebookOutlinePosition[     25901,        894]
CellTagsIndexPosition[     25816,        889]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Special Functions" :> 
          Documentation`HelpLookup["paclet:tutorial/SpecialFunctions"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Special Functions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"BesselY" :> Documentation`HelpLookup["paclet:ref/BesselY"], 
          "SphericalBesselJ" :> 
          Documentation`HelpLookup["paclet:ref/SphericalBesselJ"], 
          "SphericalHankelH1" :> 
          Documentation`HelpLookup["paclet:ref/SphericalHankelH1"], 
          "SphericalHankelH2" :> 
          Documentation`HelpLookup["paclet:ref/SphericalHankelH2"], 
          "SpheroidalS2" :> 
          Documentation`HelpLookup["paclet:ref/SpheroidalS2"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"BesselY\"\>", 
       2->"\<\"SphericalBesselJ\"\>", 3->"\<\"SphericalHankelH1\"\>", 
       4->"\<\"SphericalHankelH2\"\>", 
       5->"\<\"SpheroidalS2\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Bessel-Related Functions" :> 
          Documentation`HelpLookup["paclet:guide/BesselRelatedFunctions"], 
          "Functions for Separable Coordinate Systems" :> 
          Documentation`HelpLookup[
           "paclet:guide/FunctionsForSeparableCoordinateSystems"], 
          "New in 6.0: Mathematical Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicalFunctions"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Bessel-Related Functions\"\>", 
       2->"\<\"Functions for Separable Coordinate Systems\"\>", 
       3->"\<\"New in 6.0: Mathematical Functions\"\>"}, "\<\"more about\"\>",
        
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["SphericalBesselY", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["SphericalBesselY",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/SphericalBesselY"], "[", 
       RowBox[{
        StyleBox["n", "TI"], ",", 
        StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the spherical Bessel function of the second kind \
",
     Cell[BoxData[
      FormBox[
       RowBox[{
        SubscriptBox["y", "n"], "(", "z", ")"}], TraditionalForm]], 
      "InlineMath"],
     ". "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->26990]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["\<\
Mathematical function, suitable for both symbolic and numerical manipulation.\
\
\>", "Notes",
 CellID->20640],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SphericalBesselY",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SphericalBesselY"]], "InlineFormula"],
 " is given in terms of ordinary Bessel functions by ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SubscriptBox["y", "n"], "(", "z", ")"}], "=", 
    RowBox[{
     RowBox[{
      SqrtBox[
       RowBox[{"\[Pi]", "/", "2"}]], "/", 
      SqrtBox["z"]}], 
     RowBox[{
      SubscriptBox["Y", 
       RowBox[{"n", "+", 
        FractionBox["1", "2"]}]], "(", "z", ")"}]}]}], TraditionalForm]], 
  "InlineMath"],
 ". "
}], "Notes",
 CellID->27007],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["SphericalBesselY",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SphericalBesselY"], "[", 
   RowBox[{
    StyleBox["n", "TI"], ",", 
    StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
 " has a branch cut discontinuity in the complex ",
 Cell[BoxData[
  FormBox["z", TraditionalForm]], "InlineMath"],
 " plane running from ",
 Cell[BoxData[
  FormBox[
   RowBox[{"-", "\[Infinity]"}], TraditionalForm]], "InlineMath"],
 " to ",
 Cell[BoxData[
  FormBox["0", TraditionalForm]], "InlineMath"],
 ". "
}], "Notes",
 CellID->24653],

Cell[TextData[{
 "Explicit symbolic forms for integer ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " can be obtained using ",
 Cell[BoxData[
  ButtonBox["FunctionExpand",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FunctionExpand"]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->28997],

Cell[TextData[{
 "For certain special arguments, ",
 Cell[BoxData[
  ButtonBox["SphericalBesselY",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SphericalBesselY"]], "InlineFormula"],
 " automatically evaluates to exact values."
}], "Notes",
 CellID->124429815],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SphericalBesselY",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SphericalBesselY"]], "InlineFormula"],
 " can be evaluated to arbitrary numerical precision."
}], "Notes",
 CellID->31226],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SphericalBesselY",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SphericalBesselY"]], "InlineFormula"],
 " automatically threads over lists."
}], "Notes",
 CellID->28162]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->639159750],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->9718709],

Cell["Evaluate numerically:", "ExampleText",
 CellID->310104354],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SphericalBesselJ", "[", 
  RowBox[{"1", ",", "5.5"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->693268908],

Cell[BoxData[
 RowBox[{"-", "0.15217269699743205`"}]], "Output",
 ImageSize->{68, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->213136633]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->141994335],

Cell[TextData[{
 "Plot ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["y", "1"], "(", "x", ")"}], TraditionalForm]], 
  "InlineMath"],
 ":"
}], "ExampleText",
 CellID->49178],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"SphericalBesselY", "[", 
    RowBox[{"1", ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "20"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->241358785],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 129},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->179590838]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(6)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->29732],

Cell[TextData[{
 "Find explicit symbolic forms using ",
 Cell[BoxData[
  ButtonBox["FunctionExpand",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FunctionExpand"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->620889108],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FunctionExpand", "[", 
  RowBox[{"SphericalBesselY", "[", 
   RowBox[{"1", ",", "x"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->39824096],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", 
   FractionBox[
    RowBox[{"Cos", "[", "x", "]"}], 
    SuperscriptBox["x", "2"]]}], "-", 
  FractionBox[
   RowBox[{"Sin", "[", "x", "]"}], "x"]}]], "Output",
 ImageSize->{117, 31},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->74659649]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FunctionExpand", "[", 
  RowBox[{"SphericalBesselY", "[", 
   RowBox[{
    RowBox[{"1", "/", "2"}], ",", "x"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->86659175],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SqrtBox[
    FractionBox["\[Pi]", "2"]], " ", 
   RowBox[{"BesselY", "[", 
    RowBox[{"1", ",", "x"}], "]"}]}], 
  SqrtBox["x"]]], "Output",
 ImageSize->{129, 44},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->54905880]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->551907091],

Cell["Evaluate for complex arguments and orders:", "ExampleText",
 CellID->168756706],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SphericalBesselY", "[", 
  RowBox[{
   RowBox[{"3", "+", 
    RowBox[{"2", " ", "I"}]}], ",", 
   RowBox[{"2.5", "-", " ", "I"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->593547515],

Cell[BoxData[
 RowBox[{"0.02090509685725091`", "\[InvisibleSpace]", "+", 
  RowBox[{"0.5328661674949319`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{147, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->211218239]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->133153095],

Cell["Evaluate to high precision:", "ExampleText",
 CellID->339175458],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", " ", 
  RowBox[{
   RowBox[{"SphericalBesselY", "[", 
    RowBox[{"1", ",", " ", "3"}], "]"}], ",", " ", "50"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->89131405],

Cell[BoxData["0.\
0629591636023159767743709318118811625392088115398657307064931121368`50."], \
"Output",
 ImageSize->{375, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->728010557]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->282337929],

Cell["The precision of the output tracks precision of the input:", \
"ExampleText",
 CellID->175906109],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SphericalBesselY", "[", 
  RowBox[{"1", ",", "3.0000000000000000000000000000000000"}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->272242306],

Cell[BoxData["0.06295916360231597677437093181188116253920881154`33.\
15051991754505"], "Output",
 ImageSize->{256, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->149231144]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->716662293],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SphericalBesselY",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SphericalBesselY"]], "InlineFormula"],
 " threads element-wise over lists:"
}], "ExampleText",
 CellID->17043502],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SphericalBesselY", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "3"}], "}"}], ",", " ", "4.5"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->552427445],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.22763858414438107`", ",", "0.10491554511163632`", ",", 
   RowBox[{"-", "0.11106575624256294`"}]}], "}"}]], "Output",
 ImageSize->{217, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->216925252]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->142510687],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TraditionalForm",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TraditionalForm"]], "InlineFormula"],
 " formatting:"
}], "ExampleText",
 CellID->68674788],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"SphericalBesselY", "[", 
   RowBox[{"n", ",", "r"}], "]"}], "//", "TraditionalForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->149723510],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["y", "n"], "(", "r", ")"}], TraditionalForm]], "Output",
 ImageSize->{30, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//TraditionalForm=",
 CellID->302444414]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->15248],

Cell["\<\
Solve the radial part of the three-dimensional Laplace operator:\
\>", "ExampleText",
 CellID->1979980135],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"f", "''"}], "[", "r", "]"}], "+", 
     RowBox[{
      FractionBox[
       RowBox[{"2", " "}], "r"], " ", 
      RowBox[{
       RowBox[{"f", "'"}], "[", "r", "]"}]}], "+", 
     RowBox[{"f", "[", "r", "]"}], "-", 
     RowBox[{
      FractionBox[
       RowBox[{"n", 
        RowBox[{"(", 
         RowBox[{"n", "+", "1"}], ")"}]}], 
       SuperscriptBox["r", "2"]], 
      RowBox[{"f", "[", "r", "]"}]}]}], "\[Equal]", "0"}], ",", 
   RowBox[{"f", "[", "r", "]"}], ",", "r"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->64238625],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"f", "[", "r", "]"}], "\[Rule]", 
    RowBox[{
     RowBox[{
      RowBox[{"C", "[", "1", "]"}], " ", 
      RowBox[{"SphericalBesselJ", "[", 
       RowBox[{"n", ",", "r"}], "]"}]}], "+", 
     RowBox[{
      RowBox[{"C", "[", "2", "]"}], " ", 
      RowBox[{"SphericalBesselY", "[", 
       RowBox[{"n", ",", "r"}], "]"}]}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{453, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->605133443]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->29999],

Cell["Integrate expressions involving spherical Bessel functions:", \
"ExampleText",
 CellID->280135973],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Integrate", "[", 
   RowBox[{
    RowBox[{
     SuperscriptBox["r", "2"], 
     SuperscriptBox[
      RowBox[{"SphericalBesselY", "[", 
       RowBox[{"1", ",", "r"}], "]"}], "2"]}], ",", "r"}], "]"}], "//", 
  "Simplify"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->110856366],

Cell[BoxData[
 RowBox[{"-", 
  FractionBox[
   RowBox[{"2", "-", 
    RowBox[{"2", " ", 
     SuperscriptBox["r", "2"]}], "+", 
    RowBox[{"2", " ", 
     RowBox[{"Cos", "[", 
      RowBox[{"2", " ", "r"}], "]"}]}], "+", 
    RowBox[{"r", " ", 
     RowBox[{"Sin", "[", 
      RowBox[{"2", " ", "r"}], "]"}]}]}], 
   RowBox[{"4", " ", "r"}]]}]], "Output",
 ImageSize->{208, 34},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->321157779]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->12753],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["BesselY",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BesselY"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SphericalBesselJ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SphericalBesselJ"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SphericalHankelH1",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SphericalHankelH1"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SphericalHankelH2",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SphericalHankelH2"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SpheroidalS2",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SpheroidalS2"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->14578]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->8804],

Cell[TextData[ButtonBox["Special Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/SpecialFunctions"]], "Tutorials",
 CellID->22116]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->232781770],

Cell[TextData[ButtonBox["Bessel-Related Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/BesselRelatedFunctions"]], "MoreAbout",
 CellID->179906310],

Cell[TextData[ButtonBox["Functions for Separable Coordinate Systems",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/FunctionsForSeparableCoordinateSystems"]], "MoreAbout",
 CellID->380358325],

Cell[TextData[ButtonBox["New in 6.0: Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicalFunctions"]], "MoreAbout",
 CellID->192565865]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"SphericalBesselY - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 22, 23.0914368}", "context" -> "System`", 
    "keywords" -> {
     "spherical Bessel", "spherical Bessel function", "spherical Bessel Y"}, 
    "index" -> True, "label" -> "Built-in Mathematica Symbol", "language" -> 
    "en", "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "SphericalBesselY[n, z] gives the spherical Bessel function of the second \
kind y_n (z). ", "synonyms" -> {"spherical bessel Y"}, "title" -> 
    "SphericalBesselY", "type" -> "Symbol", "uri" -> 
    "ref/SphericalBesselY"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[8528, 273, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->639159750]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 25673, 882}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2691, 63, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3314, 90, 1581, 37, 70, "ObjectNameGrid"],
Cell[4898, 129, 720, 24, 70, "Usage",
 CellID->26990]
}, Open  ]],
Cell[CellGroupData[{
Cell[5655, 158, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6133, 175, 120, 4, 70, "Notes",
 CellID->20640],
Cell[6256, 181, 609, 23, 70, "Notes",
 CellID->27007],
Cell[6868, 206, 589, 21, 70, "Notes",
 CellID->24653],
Cell[7460, 229, 307, 11, 70, "Notes",
 CellID->28997],
Cell[7770, 242, 264, 8, 70, "Notes",
 CellID->124429815],
Cell[8037, 252, 234, 7, 70, "Notes",
 CellID->31226],
Cell[8274, 261, 217, 7, 70, "Notes",
 CellID->28162]
}, Closed]],
Cell[CellGroupData[{
Cell[8528, 273, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->639159750],
Cell[CellGroupData[{
Cell[8913, 287, 146, 5, 70, "ExampleSection",
 CellID->9718709],
Cell[9062, 294, 64, 1, 70, "ExampleText",
 CellID->310104354],
Cell[CellGroupData[{
Cell[9151, 299, 137, 4, 28, "Input",
 CellID->693268908],
Cell[9291, 305, 194, 6, 36, "Output",
 CellID->213136633]
}, Open  ]],
Cell[9500, 314, 125, 3, 70, "ExampleDelimiter",
 CellID->141994335],
Cell[9628, 319, 185, 9, 70, "ExampleText",
 CellID->49178],
Cell[CellGroupData[{
Cell[9838, 332, 252, 8, 28, "Input",
 CellID->241358785],
Cell[10093, 342, 3185, 56, 150, "Output",
 Evaluatable->False,
 CellID->179590838]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[13327, 404, 216, 7, 70, "ExampleSection",
 CellID->29732],
Cell[13546, 413, 230, 8, 70, "ExampleText",
 CellID->620889108],
Cell[CellGroupData[{
Cell[13801, 425, 176, 5, 70, "Input",
 CellID->39824096],
Cell[13980, 432, 330, 12, 52, "Output",
 CellID->74659649]
}, Open  ]],
Cell[CellGroupData[{
Cell[14347, 449, 201, 6, 70, "Input",
 CellID->86659175],
Cell[14551, 457, 313, 12, 65, "Output",
 CellID->54905880]
}, Open  ]],
Cell[14879, 472, 125, 3, 70, "ExampleDelimiter",
 CellID->551907091],
Cell[15007, 477, 85, 1, 70, "ExampleText",
 CellID->168756706],
Cell[CellGroupData[{
Cell[15117, 482, 215, 7, 70, "Input",
 CellID->593547515],
Cell[15335, 491, 274, 7, 36, "Output",
 CellID->211218239]
}, Open  ]],
Cell[15624, 501, 125, 3, 70, "ExampleDelimiter",
 CellID->133153095],
Cell[15752, 506, 70, 1, 70, "ExampleText",
 CellID->339175458],
Cell[CellGroupData[{
Cell[15847, 511, 204, 6, 70, "Input",
 CellID->89131405],
Cell[16054, 519, 235, 7, 36, "Output",
 CellID->728010557]
}, Open  ]],
Cell[16304, 529, 125, 3, 70, "ExampleDelimiter",
 CellID->282337929],
Cell[16432, 534, 103, 2, 70, "ExampleText",
 CellID->175906109],
Cell[CellGroupData[{
Cell[16560, 540, 172, 5, 70, "Input",
 CellID->272242306],
Cell[16735, 547, 227, 6, 36, "Output",
 CellID->149231144]
}, Open  ]],
Cell[16977, 556, 125, 3, 70, "ExampleDelimiter",
 CellID->716662293],
Cell[17105, 561, 225, 7, 70, "ExampleText",
 CellID->17043502],
Cell[CellGroupData[{
Cell[17355, 572, 204, 7, 70, "Input",
 CellID->552427445],
Cell[17562, 581, 290, 8, 36, "Output",
 CellID->216925252]
}, Open  ]],
Cell[17867, 592, 125, 3, 70, "ExampleDelimiter",
 CellID->142510687],
Cell[17995, 597, 202, 7, 70, "ExampleText",
 CellID->68674788],
Cell[CellGroupData[{
Cell[18222, 608, 174, 5, 70, "Input",
 CellID->149723510],
Cell[18399, 615, 254, 8, 50, "Output",
 CellID->302444414]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[18702, 629, 223, 7, 70, "ExampleSection",
 CellID->15248],
Cell[18928, 638, 116, 3, 70, "ExampleText",
 CellID->1979980135],
Cell[CellGroupData[{
Cell[19069, 645, 642, 22, 70, "Input",
 CellID->64238625],
Cell[19714, 669, 555, 18, 36, "Output",
 CellID->605133443]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[20318, 693, 233, 7, 70, "ExampleSection",
 CellID->29999],
Cell[20554, 702, 104, 2, 70, "ExampleText",
 CellID->280135973],
Cell[CellGroupData[{
Cell[20683, 708, 312, 11, 70, "Input",
 CellID->110856366],
Cell[20998, 721, 487, 17, 55, "Output",
 CellID->321157779]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[21546, 745, 311, 9, 70, "SeeAlsoSection",
 CellID->12753],
Cell[21860, 756, 1244, 44, 70, "SeeAlso",
 CellID->14578]
}, Open  ]],
Cell[CellGroupData[{
Cell[23141, 805, 313, 9, 70, "TutorialsSection",
 CellID->8804],
Cell[23457, 816, 144, 3, 70, "Tutorials",
 CellID->22116]
}, Open  ]],
Cell[CellGroupData[{
Cell[23638, 824, 319, 9, 70, "MoreAboutSection",
 CellID->232781770],
Cell[23960, 835, 158, 3, 70, "MoreAbout",
 CellID->179906310],
Cell[24121, 840, 195, 4, 70, "MoreAbout",
 CellID->380358325],
Cell[24319, 846, 174, 3, 70, "MoreAbout",
 CellID->192565865]
}, Open  ]],
Cell[24508, 852, 27, 0, 70, "History"],
Cell[24538, 854, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

