(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     35719,       1203]
NotebookOptionsPosition[     29284,        980]
NotebookOutlinePosition[     30606,       1017]
CellTagsIndexPosition[     30520,       1012]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Spheroidal Functions" :> 
          Documentation`HelpLookup["paclet:tutorial/SpheroidalFunctions"]}, #,
          2]& , Evaluator -> Automatic], {
       1->"\<\"Spheroidal Functions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"SpheroidalS1" :> 
          Documentation`HelpLookup["paclet:ref/SpheroidalS1"], "SpheroidalS2" :> 
          Documentation`HelpLookup["paclet:ref/SpheroidalS2"], "SpheroidalPS" :> 
          Documentation`HelpLookup["paclet:ref/SpheroidalPS"], "SpheroidalQS" :> 
          Documentation`HelpLookup["paclet:ref/SpheroidalQS"], 
          "MathieuCharacteristicA" :> 
          Documentation`HelpLookup["paclet:ref/MathieuCharacteristicA"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"SpheroidalS1\"\>", 
       2->"\<\"SpheroidalS2\"\>", 3->"\<\"SpheroidalPS\"\>", 
       4->"\<\"SpheroidalQS\"\>", 
       5->"\<\"MathieuCharacteristicA\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Functions Used in Quantum Mechanics" :> 
          Documentation`HelpLookup[
           "paclet:guide/FunctionsUsedInQuantumMechanics"], 
          "Special Functions" :> 
          Documentation`HelpLookup["paclet:guide/SpecialFunctions"], 
          "Spheroidal and Related Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/SpheroidalAndRelatedFunctions"], 
          "New in 6.0: Mathematical Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicalFunctions"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Functions Used in Quantum Mechanics\"\>", 
       2->"\<\"Special Functions\"\>", 
       3->"\<\"Spheroidal and Related Functions\"\>", 
       4->"\<\"New in 6.0: Mathematical Functions\"\>"}, "\<\"more about\"\>",
        
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["SpheroidalEigenvalue", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["SpheroidalEigenvalue",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/SpheroidalEigenvalue"], "[", 
       RowBox[{
        StyleBox["n", "TI"], ",", 
        StyleBox["m", "TI"], ",", 
        StyleBox["\[Gamma]", "TR"]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the spheroidal eigenvalue with degree ",
     Cell[BoxData[
      FormBox["n", TraditionalForm]], "InlineMath"],
     " and order ",
     Cell[BoxData[
      FormBox["m", TraditionalForm]], "InlineMath"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->2861]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["\<\
Mathematical function, suitable for both symbolic and numerical manipulation.\
\
\>", "Notes",
 CellID->20640],

Cell[TextData[{
 "The spheroidal eigenvalues for successive ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " correspond to the successive values of ",
 Cell[BoxData[
  FormBox["\[Lambda]", TraditionalForm]], "InlineMath"],
 " for which there exist normalizable solutions to the differential equation \
",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"(", 
       RowBox[{"1", "-", 
        SuperscriptBox["z", "2"]}], ")"}], 
      SuperscriptBox["y", "\[DoublePrime]"]}], "-", 
     RowBox[{"2", "z", " ", 
      SuperscriptBox["y", "\[Prime]"]}], " ", "+", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"\[Lambda]", "+", 
        RowBox[{
         SuperscriptBox["\[Gamma]", "2"], "(", 
         RowBox[{"1", "-", 
          SuperscriptBox["z", "2"]}], ")"}], "-", 
        FractionBox[
         SuperscriptBox["m", "2"], 
         RowBox[{"1", "-", 
          SuperscriptBox["z", "2"]}]]}], ")"}], "y"}]}], "\[Equal]", "0"}], 
   TraditionalForm]], "InlineMath"],
 "."
}], "Notes",
 CellID->29660],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["SpheroidalEigenvalue",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SpheroidalEigenvalue"], "[", 
   RowBox[{
    StyleBox["n", "TI"], ",", 
    StyleBox["m", "TI"], ",", 
    StyleBox["0", "TR"]}], "]"}]], "InlineFormula"],
 "is equal to ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", 
    RowBox[{"(", 
     RowBox[{"n", "+", "1"}], ")"}]}], TraditionalForm]], "InlineMath"],
 "."
}], "Notes",
 CellID->15931],

Cell[TextData[{
 "For certain special arguments, ",
 Cell[BoxData[
  ButtonBox["SpheroidalEigenvalue",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalEigenvalue"]], "InlineFormula"],
 " automatically evaluates to exact values."
}], "Notes",
 CellID->33960842],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SpheroidalEigenvalue",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalEigenvalue"]], "InlineFormula"],
 " can be evaluated to arbitrary numerical precision."
}], "Notes",
 CellID->31226],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SpheroidalEigenvalue",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalEigenvalue"]], "InlineFormula"],
 " automatically threads over lists."
}], "Notes",
 CellID->28162]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->245183646],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->528141429],

Cell["Evaluate numerically:", "ExampleText",
 CellID->8813137],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalEigenvalue", "[", 
  RowBox[{"2", ",", "1", ",", "0.5"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->383517724],

Cell[BoxData["5.856900130405332`"], "Output",
 ImageSize->{46, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->621716545]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->328096302],

Cell["Series expansion in the spherical limit:", "ExampleText",
 CellID->871859212],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"SpheroidalEigenvalue", "[", 
    RowBox[{"n", ",", "m", ",", "\[Gamma]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"\[Gamma]", ",", "0", ",", "3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->12929764],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
   RowBox[{"n", " ", 
    RowBox[{"(", 
     RowBox[{"1", "+", "n"}], ")"}]}], "-", 
   FractionBox[
    RowBox[{"2", " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "1"}], "+", 
       SuperscriptBox["m", "2"], "+", "n", "+", 
       SuperscriptBox["n", "2"]}], ")"}], " ", 
     SuperscriptBox["\[Gamma]", "2"]}], 
    RowBox[{
     RowBox[{"-", "3"}], "+", 
     RowBox[{"4", " ", "n"}], "+", 
     RowBox[{"4", " ", 
      SuperscriptBox["n", "2"]}]}]], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[Gamma]", "]"}], "3"],
    SeriesData[$CellContext`\[Gamma], 0, {}, 0, 3, 1],
    Editable->False]}],
  SeriesData[$CellContext`\[Gamma], 
   0, {$CellContext`n (1 + $CellContext`n), 
    0, (-2) (-1 + $CellContext`m^2 + $CellContext`n + $CellContext`n^2)/(-3 + 
    4 $CellContext`n + 4 $CellContext`n^2)}, 0, 3, 1],
  Editable->False]], "Output",
 ImageSize->{244, 37},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1772401]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(6)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->22705],

Cell["Evaluate for complex arguments:", "ExampleText",
 CellID->295604286],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalEigenvalue", "[", 
  RowBox[{"2", ",", "1", ",", 
   RowBox[{"1.5", "-", "I"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->235032259],

Cell[BoxData[
 RowBox[{"5.31416398529885`", "\[InvisibleSpace]", "+", 
  RowBox[{"1.7436892722745148`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{126, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->91571986]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->506423259],

Cell["Evaluate to high precision:", "ExampleText",
 CellID->638651861],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"SpheroidalEigenvalue", "[", 
    RowBox[{"5", ",", "2", ",", "10"}], "]"}], ",", "50"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->235062895],

Cell[BoxData[
 RowBox[{"-", 
  "30.6969237612034490155045493829118489324241006257672951099967187808245`50.\
"}]], "Output",
 ImageSize->{369, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->490208110]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->140516092],

Cell["\<\
The precision of the output tracks the precision of the input:\
\>", "ExampleText",
 CellID->173158339],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalEigenvalue", "[", 
  RowBox[{"5", ",", "2", ",", "10.0000000000000000000000000000"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->508879473],

Cell[BoxData[
 RowBox[{"-", 
  "30.69692376120344901550454938291184893242`29.000000000000004"}]], "Output",\

 ImageSize->{222, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->46493696]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->180299003],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SpheroidalEigenvalue",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalEigenvalue"]], "InlineFormula"],
 " threads element\[Hyphen]wise over lists:"
}], "ExampleText",
 CellID->169866854],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalEigenvalue", "[", 
  RowBox[{"5", ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "2", ",", "3", ",", "4", ",", "5"}], "}"}], 
   ",", "0.5"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->165480890],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "29.87613901304222`", ",", "29.871850414242004`", ",", 
   "29.858990350517676`", ",", "29.83757594380256`", ",", 
   "29.807635481191625`", ",", "29.76920805440443`"}], "}"}]], "Output",
 ImageSize->{361, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->7635562]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->165479942],

Cell[TextData[{
 "For certain parameters ",
 Cell[BoxData[
  ButtonBox["SpheroidalEigenvalue",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalEigenvalue"]], "InlineFormula"],
 " evaluates exactly:"
}], "ExampleText",
 CellID->160938561],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"SpheroidalEigenvalue", "[", 
    RowBox[{"n", ",", "1", ",", 
     RowBox[{"n", " ", 
      RowBox[{"Pi", "/", "2"}]}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->231054440],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0", ",", "0", ",", "0", ",", "0", ",", "0"}], "}"}]], "Output",
 ImageSize->{98, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->343822029]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalEigenvalue", "[", 
  RowBox[{
   FractionBox["3", "2"], ",", 
   FractionBox["1", "2"], ",", "c"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->607652],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", 
   FractionBox["1", "4"]}], "-", 
  FractionBox[
   SuperscriptBox["c", "2"], "2"], "+", 
  RowBox[{"MathieuCharacteristicA", "[", 
   RowBox[{"2", ",", 
    FractionBox[
     SuperscriptBox["c", "2"], "4"]}], "]"}]}]], "Output",
 ImageSize->{268, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->72114811]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->5584233],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TraditionalForm",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TraditionalForm"]], "InlineFormula"],
 " formatting:"
}], "ExampleText",
 CellID->54774772],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"SpheroidalEigenvalue", "[", 
   RowBox[{"n", ",", "m", ",", "c"}], "]"}], "//", 
  "TraditionalForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->314081060],

Cell[GraphicsData["CompressedBitmap", "\<\
eJydUs0OgjAMLiuDi4khHPRoeBg9+RALMYGD0SDvj+1+YK47EJus2770+/qz
3c08PJ5mHntzuU3mPYz953J9TQRhAQAdreYMdF7oZFe3LLQ7f2CPjFqD1qIV
e+1QsBbCAB3EXtFeJnxM7mrTw02v/NWP0Yirk3vh4soVIVMrMcjktW0dgsmx
PCRWFuq+WfRxWqoERMe6a7akLhWqjSeY7UBmqTJ1xoxmByPksrF1UrGvGwUr
fc9j5n3VX8xa9uQnw9bu+gunBM3/vk54KL66Q4Vk\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{34, 17},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//TraditionalForm=",
 CellID->196727239]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->30862],

Cell["Solve the spheroidal differential equation:", "ExampleText",
 CellID->640948785],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"(", 
       RowBox[{"1", "-", 
        RowBox[{"x", "^", "2"}]}], ")"}], 
      RowBox[{
       RowBox[{"y", "''"}], "[", "x", "]"}]}], "-", 
     RowBox[{"2", "x", " ", 
      RowBox[{
       RowBox[{"y", "'"}], "[", "x", "]"}]}], "+", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{"SpheroidalEigenvalue", "[", 
         RowBox[{"n", ",", "m", ",", "c"}], "]"}], "+", 
        RowBox[{
         RowBox[{"c", "^", "2"}], 
         RowBox[{"(", 
          RowBox[{"1", "-", 
           RowBox[{"x", "^", "2"}]}], ")"}]}], " ", "-", " ", 
        FractionBox[
         RowBox[{"m", "^", "2"}], 
         RowBox[{"1", "-", 
          RowBox[{"x", "^", "2"}]}]]}], ")"}], 
      RowBox[{"y", "[", "x", "]"}]}]}], "\[Equal]", "0"}], ",", "y", ",", 
   "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->297005226],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    RowBox[{"Function", "[", 
     RowBox[{
      RowBox[{"{", "x", "}"}], ",", 
      RowBox[{
       RowBox[{
        RowBox[{"C", "[", "1", "]"}], " ", 
        RowBox[{"SpheroidalPS", "[", 
         RowBox[{"n", ",", "m", ",", "c", ",", "x"}], "]"}]}], "+", 
       RowBox[{
        RowBox[{"C", "[", "2", "]"}], " ", 
        RowBox[{"SpheroidalQS", "[", 
         RowBox[{"n", ",", "m", ",", "c", ",", "x"}], "]"}]}]}]}], "]"}]}], 
   "}"}], "}"}]], "Output",
 ImageSize->{551, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->28394361]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->226971874],

Cell[TextData[{
 "Find a branch point of ",
 Cell[BoxData[
  ButtonBox["SpheroidalEigenvalue",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalEigenvalue"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->610482613],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindRoot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"SpheroidalEigenvalue", "[", 
     RowBox[{"2", ",", "0", ",", "c"}], "]"}], "\[Equal]", 
    RowBox[{"SpheroidalEigenvalue", "[", 
     RowBox[{"0", ",", "0", ",", "c"}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"c", ",", 
     RowBox[{"2", "+", 
      RowBox[{"3", " ", "I"}]}]}], "}"}], ",", 
   RowBox[{"WorkingPrecision", "\[Rule]", "30"}], ",", 
   RowBox[{"PrecisionGoal", "\[Rule]", "6"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->660230029],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"c", "\[Rule]", 
   RowBox[{"1.824770749208804330916894612153286887411594790285`30.", "+", 
    RowBox[{
    "2.601670692890318735986736488856461707699171535828`30.", " ", 
     "\[ImaginaryI]"}]}]}], "}"}]], "Output",
 ImageSize->{499, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->14892753]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->293],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SpheroidalEigenvalue",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalEigenvalue"]], "InlineFormula"],
 " does not evaluate for half-integer ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " and for generic ",
 Cell[BoxData[
  FormBox["m", TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->150678595],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalEigenvalue", "[", 
  RowBox[{
   FractionBox["1", "2"], ",", "0", ",", "0.1"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->15346465],

Cell[BoxData[
 RowBox[{"SpheroidalEigenvalue", "[", 
  RowBox[{
   FractionBox["1", "2"], ",", "0", ",", "0.1`"}], "]"}]], "Output",
 ImageSize->{220, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->14673758]
}, Open  ]],

Cell[TextData[{
 "The half-integer values of ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " are singular for the near-spherical expansion:"
}], "ExampleText",
 CellID->4281175],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SeriesCoefficient", "[", 
  RowBox[{
   RowBox[{"SpheroidalEigenvalue", "[", 
    RowBox[{"n", ",", "m", ",", "c"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"c", ",", "0", ",", "6"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->169082372],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{500, 106},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->188184389]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->7066],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["SpheroidalS1",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SpheroidalS1"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SpheroidalS2",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SpheroidalS2"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SpheroidalPS",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SpheroidalPS"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SpheroidalQS",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SpheroidalQS"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["MathieuCharacteristicA",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/MathieuCharacteristicA"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->11499]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->22136],

Cell[TextData[ButtonBox["Spheroidal Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/SpheroidalFunctions"]], "Tutorials",
 CellID->12303]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->92758295],

Cell[TextData[ButtonBox["Functions Used in Quantum Mechanics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/FunctionsUsedInQuantumMechanics"]], "MoreAbout",
 CellID->39865917],

Cell[TextData[ButtonBox["Special Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/SpecialFunctions"]], "MoreAbout",
 CellID->573473224],

Cell[TextData[ButtonBox["Spheroidal and Related Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/SpheroidalAndRelatedFunctions"]], "MoreAbout",
 CellID->20660978],

Cell[TextData[ButtonBox["New in 6.0: Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicalFunctions"]], "MoreAbout",
 CellID->700400264]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"SpheroidalEigenvalue - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 23, 30.9837998}", "context" -> "System`", 
    "keywords" -> {"eigenvalue", "spheroidal functions"}, "index" -> True, 
    "label" -> "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "SpheroidalEigenvalue[n, m, \\[Gamma]] gives the spheroidal eigenvalue \
with degree n and order m.", "synonyms" -> {"spheroidal eigenvalue"}, "title" -> 
    "SpheroidalEigenvalue", "type" -> "Symbol", "uri" -> 
    "ref/SpheroidalEigenvalue"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[8786, 270, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->245183646]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 30377, 1005}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2840, 65, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3463, 92, 1585, 37, 70, "ObjectNameGrid"],
Cell[5051, 131, 780, 24, 70, "Usage",
 CellID->2861]
}, Open  ]],
Cell[CellGroupData[{
Cell[5868, 160, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6346, 177, 120, 4, 70, "Notes",
 CellID->20640],
Cell[6469, 183, 1059, 34, 70, "Notes",
 CellID->29660],
Cell[7531, 219, 471, 18, 70, "Notes",
 CellID->15931],
Cell[8005, 239, 271, 8, 70, "Notes",
 CellID->33960842],
Cell[8279, 249, 242, 7, 70, "Notes",
 CellID->31226],
Cell[8524, 258, 225, 7, 70, "Notes",
 CellID->28162]
}, Closed]],
Cell[CellGroupData[{
Cell[8786, 270, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->245183646],
Cell[CellGroupData[{
Cell[9171, 284, 148, 5, 70, "ExampleSection",
 CellID->528141429],
Cell[9322, 291, 62, 1, 70, "ExampleText",
 CellID->8813137],
Cell[CellGroupData[{
Cell[9409, 296, 151, 4, 28, "Input",
 CellID->383517724],
Cell[9563, 302, 175, 5, 36, "Output",
 CellID->621716545]
}, Open  ]],
Cell[9753, 310, 125, 3, 70, "ExampleDelimiter",
 CellID->328096302],
Cell[9881, 315, 83, 1, 70, "ExampleText",
 CellID->871859212],
Cell[CellGroupData[{
Cell[9989, 320, 280, 8, 70, "Input",
 CellID->12929764],
Cell[10272, 330, 1065, 33, 58, "Output",
 CellID->1772401]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[11386, 369, 216, 7, 70, "ExampleSection",
 CellID->22705],
Cell[11605, 378, 74, 1, 70, "ExampleText",
 CellID->295604286],
Cell[CellGroupData[{
Cell[11704, 383, 175, 5, 70, "Input",
 CellID->235032259],
Cell[11882, 390, 270, 7, 36, "Output",
 CellID->91571986]
}, Open  ]],
Cell[12167, 400, 125, 3, 70, "ExampleDelimiter",
 CellID->506423259],
Cell[12295, 405, 70, 1, 70, "ExampleText",
 CellID->638651861],
Cell[CellGroupData[{
Cell[12390, 410, 205, 6, 70, "Input",
 CellID->235062895],
Cell[12598, 418, 254, 8, 36, "Output",
 CellID->490208110]
}, Open  ]],
Cell[12867, 429, 125, 3, 70, "ExampleDelimiter",
 CellID->140516092],
Cell[12995, 434, 113, 3, 70, "ExampleText",
 CellID->173158339],
Cell[CellGroupData[{
Cell[13133, 441, 182, 5, 70, "Input",
 CellID->508879473],
Cell[13318, 448, 239, 8, 36, "Output",
 CellID->46493696]
}, Open  ]],
Cell[13572, 459, 125, 3, 70, "ExampleDelimiter",
 CellID->180299003],
Cell[13700, 464, 242, 7, 70, "ExampleText",
 CellID->169866854],
Cell[CellGroupData[{
Cell[13967, 475, 244, 7, 70, "Input",
 CellID->165480890],
Cell[14214, 484, 358, 10, 36, "Output",
 CellID->7635562]
}, Open  ]],
Cell[14587, 497, 125, 3, 70, "ExampleDelimiter",
 CellID->165479942],
Cell[14715, 502, 248, 8, 70, "ExampleText",
 CellID->160938561],
Cell[CellGroupData[{
Cell[14988, 514, 310, 10, 70, "Input",
 CellID->231054440],
Cell[15301, 526, 233, 7, 36, "Output",
 CellID->343822029]
}, Open  ]],
Cell[CellGroupData[{
Cell[15571, 538, 190, 6, 70, "Input",
 CellID->607652],
Cell[15764, 546, 399, 14, 54, "Output",
 CellID->72114811]
}, Open  ]],
Cell[16178, 563, 123, 3, 70, "ExampleDelimiter",
 CellID->5584233],
Cell[16304, 568, 202, 7, 70, "ExampleText",
 CellID->54774772],
Cell[CellGroupData[{
Cell[16531, 579, 191, 6, 70, "Input",
 CellID->314081060],
Cell[16725, 587, 512, 12, 70, "Output",
 Evaluatable->False,
 CellID->196727239]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[17286, 605, 223, 7, 70, "ExampleSection",
 CellID->30862],
Cell[17512, 614, 86, 1, 70, "ExampleText",
 CellID->640948785],
Cell[CellGroupData[{
Cell[17623, 619, 938, 31, 70, "Input",
 CellID->297005226],
Cell[18564, 652, 668, 21, 36, "Output",
 CellID->28394361]
}, Open  ]],
Cell[19247, 676, 125, 3, 70, "ExampleDelimiter",
 CellID->226971874],
Cell[19375, 681, 230, 8, 70, "ExampleText",
 CellID->610482613],
Cell[CellGroupData[{
Cell[19630, 693, 538, 15, 70, "Input",
 CellID->660230029],
Cell[20171, 710, 387, 11, 36, "Output",
 CellID->14892753]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[20607, 727, 224, 7, 70, "ExampleSection",
 CellID->293],
Cell[20834, 736, 393, 13, 70, "ExampleText",
 CellID->150678595],
Cell[CellGroupData[{
Cell[21252, 753, 172, 5, 70, "Input",
 CellID->15346465],
Cell[21427, 760, 262, 8, 51, "Output",
 CellID->14673758]
}, Open  ]],
Cell[21704, 771, 199, 6, 70, "ExampleText",
 CellID->4281175],
Cell[CellGroupData[{
Cell[21928, 781, 278, 8, 70, "Input",
 CellID->169082372],
Cell[22209, 791, 3844, 67, 70, "Output",
 Evaluatable->False,
 CellID->188184389]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[26114, 865, 310, 9, 70, "SeeAlsoSection",
 CellID->7066],
Cell[26427, 876, 1246, 44, 70, "SeeAlso",
 CellID->11499]
}, Open  ]],
Cell[CellGroupData[{
Cell[27710, 925, 314, 9, 70, "TutorialsSection",
 CellID->22136],
Cell[28027, 936, 150, 3, 70, "Tutorials",
 CellID->12303]
}, Open  ]],
Cell[CellGroupData[{
Cell[28214, 944, 318, 9, 70, "MoreAboutSection",
 CellID->92758295],
Cell[28535, 955, 177, 3, 70, "MoreAbout",
 CellID->39865917],
Cell[28715, 960, 145, 3, 70, "MoreAbout",
 CellID->573473224],
Cell[28863, 965, 172, 3, 70, "MoreAbout",
 CellID->20660978],
Cell[29038, 970, 174, 3, 70, "MoreAbout",
 CellID->700400264]
}, Open  ]],
Cell[29227, 976, 27, 0, 70, "History"],
Cell[29257, 978, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

