(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     36516,       1232]
NotebookOptionsPosition[     30256,       1015]
NotebookOutlinePosition[     31603,       1052]
CellTagsIndexPosition[     31517,       1047]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Spheroidal Functions" :> 
          Documentation`HelpLookup["paclet:tutorial/SpheroidalFunctions"]}, #,
          2]& , Evaluator -> Automatic], {
       1->"\<\"Spheroidal Functions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"SpheroidalPS" :> 
          Documentation`HelpLookup["paclet:ref/SpheroidalPS"], 
          "SpheroidalQSPrime" :> 
          Documentation`HelpLookup["paclet:ref/SpheroidalQSPrime"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"SpheroidalPS\"\>", 
       2->"\<\"SpheroidalQSPrime\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Spheroidal and Related Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/SpheroidalAndRelatedFunctions"], 
          "New in 6.0: Mathematical Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicalFunctions"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Spheroidal and Related Functions\"\>", 
       2->"\<\"New in 6.0: Mathematical Functions\"\>"}, "\<\"more about\"\>",
        
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["SpheroidalPSPrime", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["SpheroidalPSPrime",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/SpheroidalPSPrime"], "[", 
       RowBox[{
        StyleBox["n", "TI"], ",", 
        StyleBox["m", "TI"], ",", 
        StyleBox["\[Gamma]", "TR"], ",", 
        StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the derivative with respect to ",
     Cell[BoxData[
      FormBox["z", TraditionalForm]], "InlineMath"],
     " of the angular spheroidal function ",
     Cell[BoxData[
      FormBox[
       RowBox[{
        SubscriptBox[
         StyleBox["PS",
          FontSlant->"Italic"], 
         RowBox[{"n", ",", "m"}]], "(", 
        RowBox[{"\[Gamma]", ",", "z"}], ")"}], TraditionalForm]], 
      "InlineMath"],
     " of the first kind."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->9067]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["\<\
Mathematical function, suitable for both symbolic and numerical manipulation.\
\
\>", "Notes",
 CellID->20640],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["SpheroidalPSPrime",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SpheroidalPSPrime"], "[", 
   RowBox[{
    StyleBox["n", "TI"], ",", 
    StyleBox["m", "TI"], ",", 
    StyleBox["a", "TI"], ",", 
    StyleBox["\[Gamma]", "TR"], ",", 
    StyleBox["z", "TI"]}], "]"}]], "InlineFormula", "TemplateInclusion"],
 " uses spheroidal functions of type ",
 Cell[BoxData[
  FormBox["a", TraditionalForm]], "InlineMath"],
 ". The types are specified as for ",
 Cell[BoxData[
  ButtonBox["SpheroidalPS",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalPS"]], "InlineFormula"],
 "."
}], "Notes",
 CellID->16338],

Cell[TextData[{
 "For certain special arguments, ",
 Cell[BoxData[
  ButtonBox["SpheroidalPSPrime",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalPSPrime"]], "InlineFormula"],
 " automatically evaluates to exact values."
}], "Notes",
 CellID->152512156],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SpheroidalPSPrime",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalPSPrime"]], "InlineFormula"],
 " can be evaluated to arbitrary numerical precision."
}], "Notes",
 CellID->31226],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SpheroidalPSPrime",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalPSPrime"]], "InlineFormula"],
 " automatically threads over lists."
}], "Notes",
 CellID->28162]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->28944332],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(3)", "ExampleCount"]
}], "ExampleSection",
 CellID->171533154],

Cell["Evaluate numerically:", "ExampleText",
 CellID->102308445],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalPSPrime", "[", 
  RowBox[{"3", ",", "2", ",", "1", ",", "0.5"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->43173829],

Cell[BoxData["3.45432632111244367742114652645`15.954589770191"], "Output",
 ImageSize->{123, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->451625242]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->835947008],

Cell["Expansion about the spherical case:", "ExampleText",
 CellID->802528156],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"SpheroidalPSPrime", "[", 
    RowBox[{"2", ",", "0", ",", "\[Gamma]", ",", "z"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"\[Gamma]", ",", "0", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->148566906],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
   RowBox[{"3", " ", "z"}], "+", 
   FractionBox[
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{
        RowBox[{"-", "324"}], " ", "z"}], "+", 
       RowBox[{"756", " ", 
        SuperscriptBox["z", "3"]}]}], ")"}], " ", 
     SuperscriptBox["\[Gamma]", "2"]}], "1764"], "+", 
   FractionBox[
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{
        RowBox[{"-", "19682"}], " ", "z"}], "-", 
       RowBox[{"179172", " ", 
        SuperscriptBox["z", "3"]}], "+", 
       RowBox[{"203742", " ", 
        SuperscriptBox["z", "5"]}]}], ")"}], " ", 
     SuperscriptBox["\[Gamma]", "4"]}], "11409552"], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[Gamma]", "]"}], "5"],
    SeriesData[$CellContext`\[Gamma], 0, {}, 0, 5, 1],
    Editable->False]}],
  SeriesData[$CellContext`\[Gamma], 0, {
   3 $CellContext`z, 0, 
    Rational[1, 1764] ((-324) $CellContext`z + 756 $CellContext`z^3), 0, 
    Rational[1, 11409552] ((-19682) $CellContext`z - 179172 $CellContext`z^3 + 
     203742 $CellContext`z^5)}, 0, 5, 1],
  Editable->False]], "Output",
 ImageSize->{450, 35},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->59767461]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->434247032],

Cell[TextData[{
 "Plot ",
 Cell[BoxData[
  FormBox[
   TagBox[
    RowBox[{
     SuperscriptBox[
      SubscriptBox[
       StyleBox["PS", "IT"], 
       RowBox[{
        TagBox["2",
         SpheroidalPSPrime,
         Editable->True,
         Selectable->True], ",", 
        TagBox["0",
         SpheroidalPSPrime,
         Editable->True,
         Selectable->True]}]], "\[Prime]"], "(", 
     RowBox[{
      TagBox["1",
       SpheroidalPSPrime,
       Editable->True,
       Selectable->True], ",", 
      TagBox["x",
       SpheroidalPSPrime,
       Editable->True,
       Selectable->True]}], ")"}],
    InterpretTemplate[SpheroidalPSPrime[
      SlotSequence[1]]& ],
    Editable->False,
    Selectable->False], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->6381167],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"SpheroidalPSPrime", "[", 
    RowBox[{"2", ",", "0", ",", "1", ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->213350775],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzNWE1sVUUUHt67j9bS2lKsVFLeexSLtVDUKggWahEooGIBxWoaF7UxaU0U
g3WjG93oRje60Y0uUBPUBUYCIcHESGIkMZAQSYyaGEgILiQkEhdNSK7fnDMz
b2bu6etd1ERC77t35jv/Z8495+6dmJl64aWJmenJiequwxOvTE1PvlodOXQY
S8VFSqnj+HuxU+E+xR39HU9T/I7ra1E/0z/1NK020eoH7//w7enf9e0TuBQA
Gqvtln779a+33jy9f/TIl19cmJ29qR4XOTUT9tq1f8AM2E8+/gn36lEReyth
b9yYBWrs4Oeg0Njdgew2wkAepNYwIyK/dg8L2e+8/d3Vq3+rHSK2Q1+Tkyd+
AVMGPiLillvcc+NHWfhwoF9nqB8sgT1qSOS1wmLZXuPHLSK2i+SCF4QCi7gM
irgy4aAWYjM9deziz39uDvSr0j6WsQkIgOuxkmC/JPLrJjzEsTFQsN/DqTnp
VhMd2wZXnfn+jzVGTjHCJ7joR6broSW4/43XT702c+LSpevrInxMz3R3Ed3Z
Hy9PPv/VRx+eheC+XHS9Vs/PPj0PE42eKqnh5rKwz3raadpjPL048Pg6TzP2
4FqRX7/nMUARotgCxt1DOIhDBN979wxSojeQdx9RfX3sItwOqeyZDJcBqz1U
hwG46RFxD9Aq3AJuyHp1p4jaYLMTR4d90R3o9CBRsQ/gaLVK5LLJegABhHHg
UhFxD9EqHARuWqeyiBq03FgnWNgV6LTV6mQsW2byUI72UGghuLULeZylG46k
6No5/ynYlpHWmkvadk8aYsYVdV6qHVYaEgECcdOci46rLqzidCXbsvm/KzUn
GiHlk9kmcttDq8iOQy9/g9Cr2GJGNRg/FaJdLa9i5Pq+1RVKr7V5OYNCm/FM
tQ5tpY5MwqvGaIdWgxqyMrPSENGQRjpfUeE55s1Gp+q8tCvnlV/gn6Ixy1gW
Z2GWTr/vEktfVwf2MJ9xXXeWRPuVBbNiIby9NK3VP87KW/5X+uoqpCN0e2pO
BbyqT0VjOldOlPyd/1Arn4a1g16mrsceyvow5lDJqW1DGtaWivm169yX8buD
T08sqRpRlCOOfDSeStOgXsuV7yBJ4/rMFbOQ+jWPriXQcwvGhqhn6Jeu+v2N
Oqd9Ztef9SzAIiwYc5YfxQJpYjrzmj6xlXKE8uNzPZfw7BvP3ekCSmmR9jO2
66lELXbP7kX0JKHHTUS4avP7B62VOuB2s90h9oHSE8U+EcXZABHIDvTEalRE
lWx2oMFmB+0NNOKT57KDJ6cMF0Zxn6f1fkxENXl66+lmj4haYvU2HdPuQJ8W
q6/rNkZELvzO5LOhK/xOEdVquYEVGCKhtwfS2q31pmugOYvOXdFmucetPfIC
dUZ8/ur09rd5XtFV8+FcVB2el/TJHJqTqmgrc9CtbsklpTOSMpiL6o5QGkK9
2fmt4FX/rijSm8QYuWnSRXyjiCt7EdfcNoioSobb/UHEuyMuAyKXbo8L+/Ne
EUcTSMIW8jy1XsT1WI+5eao/0IonP84stFwQvFbkczc9o2c0PXWfiIqmQcSn
N5DG0x37AFGHWmtEPm4KRJy5kSI7lK5piXA+3DTo5rjVLi/8PjamoykxgT1u
rl9VN9sHPG+yI6jP4Peh3xNHzzQ/mnmey105F91GG2X3naLL+SHsD2LLeIr0
v1iscB7xTwpNkZmvL51iVAZtdJEonMTLRdxWay1ymL3UEWTBsI0We+P8uSvL
RD7brPXoEfAfuKUijiprgm2HaxVxO0Mcsqol0IveByXsI4vYPu2LJpEXvWFK
sAE2wm9IZt0BNIpYemfpr3/I5v2jR/iYUceVxdI7sshfzqCJh6r1VfLXSrXo
X5zgy28=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 106},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->44825000]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->23897],

Cell["Evaluate for complex arguments and parameters:", "ExampleText",
 CellID->94393633],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalPSPrime", "[", 
  RowBox[{"3", ",", "2", ",", 
   RowBox[{"1", "-", "I"}], ",", " ", "3.5"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->314925903],

Cell[BoxData[
 RowBox[{"267.1617133702218`", "\[InvisibleSpace]", "-", 
  RowBox[{"829.2928471737238`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{126, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->12410620]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->217504807],

Cell["Evaluate to high precision:", "ExampleText",
 CellID->283569038],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"SpheroidalPSPrime", "[", 
    RowBox[{"2", ",", "1", ",", 
     RowBox[{"1", "/", "3"}], ",", 
     RowBox[{"1", "/", "2"}]}], "]"}], ",", "50"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->852018633],

Cell[BoxData[
 RowBox[{"-", 
  "1.7241824277681511140021730468310520630861948354788020361981084616357`50."}\
]], "Output",
 ImageSize->{369, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->489148550]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->3535261],

Cell["\<\
The precision of the output tracks the precision of the input:\
\>", "ExampleText",
 CellID->173158339],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalPSPrime", "[", 
  RowBox[{"2", ",", "1", ",", 
   RowBox[{"1", "/", "3"}], ",", "0.50000000000000000000000"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->429954945],

Cell[BoxData[
 RowBox[{"-", 
  "1.72418242776815111400217304683105287662`22.698970004336022"}]], "Output",
 ImageSize->{180, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->212466528]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->165479942],

Cell[TextData[{
 "For certain parameters ",
 Cell[BoxData[
  ButtonBox["SpheroidalPS",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalPS"]], "InlineFormula"],
 " evaluates exactly:"
}], "ExampleText",
 CellID->160938561],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"With", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"n", "=", "3"}], "}"}], ",", 
   RowBox[{"SpheroidalPSPrime", "[", 
    RowBox[{"n", ",", "1", ",", 
     RowBox[{"n", " ", 
      RowBox[{"Pi", "/", "2"}]}], ",", " ", "x"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->10306910],

Cell[BoxData[
 RowBox[{
  FractionBox[
   RowBox[{"2", " ", "x", " ", 
    RowBox[{"Cos", "[", 
     FractionBox[
      RowBox[{"3", " ", "\[Pi]", " ", "x"}], "2"], "]"}], " ", 
    SqrtBox[
     FractionBox["3", 
      RowBox[{"7", " ", 
       RowBox[{"(", 
        RowBox[{"EulerGamma", "-", 
         RowBox[{"CosIntegral", "[", 
          RowBox[{"6", " ", "\[Pi]"}], "]"}], "+", 
         RowBox[{"Log", "[", 
          RowBox[{"6", " ", "\[Pi]"}], "]"}]}], ")"}]}]]]}], 
   SuperscriptBox[
    RowBox[{"(", 
     RowBox[{"1", "-", 
      SuperscriptBox["x", "2"]}], ")"}], 
    RowBox[{"3", "/", "2"}]]], "-", 
  FractionBox[
   RowBox[{"3", " ", "\[Pi]", " ", 
    SqrtBox[
     FractionBox["3", 
      RowBox[{"7", " ", 
       RowBox[{"(", 
        RowBox[{"EulerGamma", "-", 
         RowBox[{"CosIntegral", "[", 
          RowBox[{"6", " ", "\[Pi]"}], "]"}], "+", 
         RowBox[{"Log", "[", 
          RowBox[{"6", " ", "\[Pi]"}], "]"}]}], ")"}]}]]], " ", 
    RowBox[{"Sin", "[", 
     FractionBox[
      RowBox[{"3", " ", "\[Pi]", " ", "x"}], "2"], "]"}]}], 
   SqrtBox[
    RowBox[{"1", "-", 
     SuperscriptBox["x", "2"]}]]]}]], "Output",
 ImageSize->{318, 131},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->53074574]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalPSPrime", "[", 
  RowBox[{
   RowBox[{"3", "/", "2"}], ",", 
   RowBox[{"1", "/", "2"}], ",", "\[Gamma]", ",", "x"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->283753173],

Cell[BoxData[
 RowBox[{
  FractionBox[
   RowBox[{"x", " ", 
    RowBox[{"MathieuC", "[", 
     RowBox[{
      RowBox[{"MathieuCharacteristicA", "[", 
       RowBox[{"2", ",", 
        FractionBox[
         SuperscriptBox["\[Gamma]", "2"], "4"]}], "]"}], ",", 
      FractionBox[
       SuperscriptBox["\[Gamma]", "2"], "4"], ",", 
      RowBox[{"ArcCos", "[", "x", "]"}]}], "]"}]}], 
   RowBox[{"2", " ", 
    SqrtBox["\[Pi]"], " ", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"1", "-", 
       SuperscriptBox["x", "2"]}], ")"}], 
     RowBox[{"5", "/", "4"}]]}]], "-", 
  FractionBox[
   RowBox[{"MathieuCPrime", "[", 
    RowBox[{
     RowBox[{"MathieuCharacteristicA", "[", 
      RowBox[{"2", ",", 
       FractionBox[
        SuperscriptBox["\[Gamma]", "2"], "4"]}], "]"}], ",", 
     FractionBox[
      SuperscriptBox["\[Gamma]", "2"], "4"], ",", 
     RowBox[{"ArcCos", "[", "x", "]"}]}], "]"}], 
   RowBox[{
    SqrtBox["\[Pi]"], " ", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"1", "-", 
       SuperscriptBox["x", "2"]}], ")"}], 
     RowBox[{"3", "/", "4"}]]}]]}]], "Output",
 ImageSize->{425, 99},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->650758621]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->129130650],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TraditionalForm",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TraditionalForm"]], "InlineFormula"],
 " formatting:"
}], "ExampleText",
 CellID->54774772],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"SpheroidalPSPrime", "[", 
   RowBox[{"n", ",", "m", ",", "\[Gamma]", ",", "\[Eta]"}], "]"}], "//", 
  "TraditionalForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->187276340],

Cell[GraphicsData["CompressedBitmap", "\<\
eJyNVMuqAjEMrW3HqwsFHYULunDhV8wH6MqPGETQhShe/7+36cNJ0oxamLYe
0nOSk+ChfZ5P1/Z5Obab/aO9ny/Hv83u9vCQGSilGv/NfpW/O38LX+OcP+Ne
w24ADUttAzqBvYqoCiuHKRMh2LU/R3Cx3fsUgBh/5IioYToNSzUTOkLoiyDG
Io2hGKXDaXK8xfWYr9SHTEfTWvwCD6qiQmAD8we44EBq0xvNmHOWlaAqOlpU
a4hmwU/foA6Sjqbsoh3ZBikfiBqLne3PgfPM39TZ33tdaFhhIuoiNy2wvzpB
IlH28kwhnQV93dPdKvuavO7pl6xGcpsyZq60JKzmrQsuTcCEYXzeasRZus5n
6BPbjP2OjGuGrlz3H9WktvwDstVbTg==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{58, 17},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//TraditionalForm=",
 CellID->34779494]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->29996],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SpheroidalPSPrime",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalPSPrime"]], "InlineFormula"],
 " of different types have different branch cut structures:"
}], "ExampleText",
 CellID->277064462],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalPSPrime", "[", 
  RowBox[{"n", ",", "m", ",", "1", ",", "0", ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->135549909],

Cell[BoxData[
 FractionBox[
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "1"}], "-", "n"}], ")"}], " ", "x", " ", 
    RowBox[{"LegendreP", "[", 
     RowBox[{"n", ",", "m", ",", "2", ",", "x"}], "]"}]}], "+", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"1", "-", "m", "+", "n"}], ")"}], " ", 
    RowBox[{"LegendreP", "[", 
     RowBox[{
      RowBox[{"1", "+", "n"}], ",", "m", ",", "2", ",", "x"}], "]"}]}]}], 
  RowBox[{
   RowBox[{"-", "1"}], "+", 
   SuperscriptBox["x", "2"]}]]], "Output",
 ImageSize->{443, 31},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->8731514]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalPSPrime", "[", 
  RowBox[{"n", ",", "m", ",", "2", ",", "0", ",", "x"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->591864808],

Cell[BoxData[
 FractionBox[
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "1"}], "-", "n"}], ")"}], " ", "x", " ", 
    RowBox[{"LegendreP", "[", 
     RowBox[{"n", ",", "m", ",", "3", ",", "x"}], "]"}]}], "+", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"1", "-", "m", "+", "n"}], ")"}], " ", 
    RowBox[{"LegendreP", "[", 
     RowBox[{
      RowBox[{"1", "+", "n"}], ",", "m", ",", "3", ",", "x"}], "]"}]}]}], 
  RowBox[{
   RowBox[{"-", "1"}], "+", 
   SuperscriptBox["x", "2"]}]]], "Output",
 ImageSize->{443, 31},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->138811151]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->21801],

Cell["\<\
Plot prolate and oblate versions of the same angular function:\
\>", "ExampleText",
 CellID->470010545],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"SpheroidalPSPrime", "[", 
      RowBox[{"2", ",", "1", ",", "3", ",", "x"}], "]"}], ",", 
     RowBox[{"SpheroidalPSPrime", "[", 
      RowBox[{"2", ",", "1", ",", 
       RowBox[{"3", "I"}], ",", "x"}], "]"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
   RowBox[{"MaxRecursion", "\[Rule]", "0"}], ",", 
   RowBox[{"PlotPoints", "\[Rule]", "30"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->663030826],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], 
    LineBox[{{-0.933695908885904, 7.75604722713986}, {-0.933668662146553, 
     5.51389396156141}, {-0.861757965711209, 
     3.49146707482107}, {-0.794612669314658, 
     2.39822413293659}, {-0.728784497094833, 
     1.52484192307931}, {-0.657376897327593, 
     0.668550634833131}, {-0.590734697599148, -0.079816937571547}, \
{-0.518513070323288, -0.841956914569757}, {-0.447608567224154, \
-1.5340781805207}, {-0.381469464163813, -2.1188964648741}, \
{-0.309750933556058, -2.67341754583842}, {-0.242797802987097, \
-3.10426027425852}, {-0.17716179659486, -3.43518337851188}, \
{-0.10594636265521, -3.68276391354436}, {-0.0394963287543533, \
-3.80313551568053}, {0.0325331326939167, -3.80941253413652}, {
     0.103245469965462, -3.68977054188475}, {
     0.169192407198214, -3.46879919615271}, {
     0.24071877197838, -3.1161778521914}, {
     0.307479736719752, -2.68947262157744}, {
     0.379820129008537, -2.13263100406329}, {
     0.450843397120598, -1.50387463768265}, {
     0.517101265193866, -0.856315486961688}, {
     0.588938560814547, -0.0993970016175838}, {0.656010456396434, 
     0.652764901623703}, {0.721765227801597, 1.43747985081308}, {
     0.793099426754174, 2.37667215067068}, {0.859668225667957, 
     3.45154774940672}, {0.931816452129153, 5.43175226622555}, {
     0.931845485321906, 7.75604722713986}}]}, 
   {Hue[0.90606797749979, 0.6, 0.6], 
    LineBox[{{-0.933675544083987, 7.75604722713986}, {-0.933668662146553, 
     6.58284050010937}, {-0.861757965711209, 
     1.74858698237228}, {-0.794612669314658, -0.173586820948419}, \
{-0.728784497094833, -1.17904574870697}, {-0.657376897327593, \
-1.77834686002714}, {-0.590734697599148, -2.07885215919645}, \
{-0.518513070323288, -2.24107658273876}, {-0.447608567224154, \
-2.30344458595028}, {-0.381469464163813, -2.31256857035013}, \
{-0.309750933556058, -2.29473114532988}, {-0.242797802987097, \
-2.26746336127298}, {-0.17716179659486, -2.24025001061181}, \
{-0.10594636265521, -2.21692774930666}, {-0.0394963287543533, \
-2.20478330708747}, {0.0325331326939167, -2.20413652767114}, {
     0.103245469965462, -2.21623461405708}, {
     0.169192407198214, -2.23722236585095}, {
     0.24071877197838, -2.26657037646023}, {
     0.307479736719752, -2.29390405381092}, {
     0.379820129008537, -2.31240499876973}, {
     0.450843397120598, -2.30201769464872}, {
     0.517101265193866, -2.24307851174522}, {
     0.588938560814547, -2.08459313927581}, {
     0.656010456396434, -1.78653987429661}, {
     0.721765227801597, -1.25579223166154}, {
     0.793099426754174, -0.204003750392509}, {0.859668225667957, 
     1.66623501554888}, {0.931816452129153, 6.37419585503844}, {
     0.931824784149993, 7.75604722713986}}]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.618033988749895],
  Axes->True,
  ImageMargins->0.,
  ImageSize->Small,
  Method->{},
  PlotRange->{{-1, 1}, {-3.80941253413652, 7.75604722713986}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 117},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->12628216]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->30403],

Cell[TextData[{
 "Spheroidal functions do not evaluate for half-integer values of ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " and generic values of ",
 Cell[BoxData[
  FormBox["m", TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->348355882],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalPSPrime", "[", 
  RowBox[{
   RowBox[{"3", "/", "2"}], ",", "1", ",", "1", ",", "0.5"}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->175415402],

Cell[BoxData[
 RowBox[{"SpheroidalPSPrime", "[", 
  RowBox[{
   FractionBox["3", "2"], ",", "1", ",", "1", ",", "0.5`"}], "]"}]], "Output",\

 ImageSize->{217, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->280525813]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->26407],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["SpheroidalPS",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SpheroidalPS"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SpheroidalQSPrime",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SpheroidalQSPrime"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->9699]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->17735],

Cell[TextData[ButtonBox["Spheroidal Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/SpheroidalFunctions"]], "Tutorials",
 CellID->12303]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->854494349],

Cell[TextData[ButtonBox["Spheroidal and Related Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/SpheroidalAndRelatedFunctions"]], "MoreAbout",
 CellID->16063298],

Cell[TextData[ButtonBox["New in 6.0: Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicalFunctions"]], "MoreAbout",
 CellID->10371604]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"SpheroidalPSPrime - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 23, 32.7963462}", "context" -> "System`", 
    "keywords" -> {"spheroidal functions"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "SpheroidalPSPrime[n, m, \\[Gamma], z] gives the derivative with respect \
to z of the angular spheroidal function PS n, m (\\[Gamma], z) of the first \
kind.", "synonyms" -> {"spheroidal prime"}, "title" -> "SpheroidalPSPrime", 
    "type" -> "Symbol", "uri" -> "ref/SpheroidalPSPrime"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[7447, 234, 359, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->28944332]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 31375, 1040}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2130, 53, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2753, 80, 1582, 37, 70, "ObjectNameGrid"],
Cell[4338, 119, 1031, 32, 70, "Usage",
 CellID->9067]
}, Open  ]],
Cell[CellGroupData[{
Cell[5406, 156, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[5884, 173, 120, 4, 70, "Notes",
 CellID->20640],
Cell[6007, 179, 673, 22, 70, "Notes",
 CellID->16338],
Cell[6683, 203, 266, 8, 70, "Notes",
 CellID->152512156],
Cell[6952, 213, 236, 7, 70, "Notes",
 CellID->31226],
Cell[7191, 222, 219, 7, 70, "Notes",
 CellID->28162]
}, Closed]],
Cell[CellGroupData[{
Cell[7447, 234, 359, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->28944332],
Cell[CellGroupData[{
Cell[7831, 248, 148, 5, 70, "ExampleSection",
 CellID->171533154],
Cell[7982, 255, 64, 1, 70, "ExampleText",
 CellID->102308445],
Cell[CellGroupData[{
Cell[8071, 260, 157, 4, 28, "Input",
 CellID->43173829],
Cell[8231, 266, 205, 5, 36, "Output",
 CellID->451625242]
}, Open  ]],
Cell[8451, 274, 125, 3, 70, "ExampleDelimiter",
 CellID->835947008],
Cell[8579, 279, 78, 1, 70, "ExampleText",
 CellID->802528156],
Cell[CellGroupData[{
Cell[8682, 284, 288, 8, 70, "Input",
 CellID->148566906],
Cell[8973, 294, 1274, 39, 56, "Output",
 CellID->59767461]
}, Open  ]],
Cell[10262, 336, 125, 3, 70, "ExampleDelimiter",
 CellID->434247032],
Cell[10390, 341, 796, 33, 70, "ExampleText",
 CellID->6381167],
Cell[CellGroupData[{
Cell[11211, 378, 293, 9, 28, "Input",
 CellID->213350775],
Cell[11507, 389, 2111, 39, 127, "Output",
 Evaluatable->False,
 CellID->44825000]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[13667, 434, 216, 7, 70, "ExampleSection",
 CellID->23897],
Cell[13886, 443, 88, 1, 70, "ExampleText",
 CellID->94393633],
Cell[CellGroupData[{
Cell[13999, 448, 187, 5, 70, "Input",
 CellID->314925903],
Cell[14189, 455, 270, 7, 36, "Output",
 CellID->12410620]
}, Open  ]],
Cell[14474, 465, 125, 3, 70, "ExampleDelimiter",
 CellID->217504807],
Cell[14602, 470, 70, 1, 70, "ExampleText",
 CellID->283569038],
Cell[CellGroupData[{
Cell[14697, 475, 263, 8, 70, "Input",
 CellID->852018633],
Cell[14963, 485, 253, 8, 36, "Output",
 CellID->489148550]
}, Open  ]],
Cell[15231, 496, 123, 3, 70, "ExampleDelimiter",
 CellID->3535261],
Cell[15357, 501, 113, 3, 70, "ExampleText",
 CellID->173158339],
Cell[CellGroupData[{
Cell[15495, 508, 207, 6, 70, "Input",
 CellID->429954945],
Cell[15705, 516, 237, 7, 36, "Output",
 CellID->212466528]
}, Open  ]],
Cell[15957, 526, 125, 3, 70, "ExampleDelimiter",
 CellID->165479942],
Cell[16085, 531, 232, 8, 70, "ExampleText",
 CellID->160938561],
Cell[CellGroupData[{
Cell[16342, 543, 320, 10, 70, "Input",
 CellID->10306910],
Cell[16665, 555, 1289, 42, 152, "Output",
 CellID->53074574]
}, Open  ]],
Cell[CellGroupData[{
Cell[17991, 602, 211, 6, 70, "Input",
 CellID->283753173],
Cell[18205, 610, 1239, 41, 120, "Output",
 CellID->650758621]
}, Open  ]],
Cell[19459, 654, 125, 3, 70, "ExampleDelimiter",
 CellID->129130650],
Cell[19587, 659, 202, 7, 70, "ExampleText",
 CellID->54774772],
Cell[CellGroupData[{
Cell[19814, 670, 210, 6, 70, "Input",
 CellID->187276340],
Cell[20027, 678, 625, 14, 70, "Output",
 Evaluatable->False,
 CellID->34779494]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[20701, 698, 239, 7, 70, "ExampleSection",
 CellID->29996],
Cell[20943, 707, 252, 7, 70, "ExampleText",
 CellID->277064462],
Cell[CellGroupData[{
Cell[21220, 718, 166, 4, 70, "Input",
 CellID->135549909],
Cell[21389, 724, 654, 22, 52, "Output",
 CellID->8731514]
}, Open  ]],
Cell[CellGroupData[{
Cell[22080, 751, 166, 4, 70, "Input",
 CellID->591864808],
Cell[22249, 757, 656, 22, 52, "Output",
 CellID->138811151]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[22954, 785, 223, 7, 70, "ExampleSection",
 CellID->21801],
Cell[23180, 794, 113, 3, 70, "ExampleText",
 CellID->470010545],
Cell[CellGroupData[{
Cell[23318, 801, 564, 16, 70, "Input",
 CellID->663030826],
Cell[23885, 819, 3191, 66, 138, "Output",
 CellID->12628216]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[27125, 891, 226, 7, 70, "ExampleSection",
 CellID->30403],
Cell[27354, 900, 284, 9, 70, "ExampleText",
 CellID->348355882],
Cell[CellGroupData[{
Cell[27663, 913, 184, 6, 70, "Input",
 CellID->175415402],
Cell[27850, 921, 272, 9, 51, "Output",
 CellID->280525813]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[28183, 937, 311, 9, 70, "SeeAlsoSection",
 CellID->26407],
Cell[28497, 948, 476, 17, 70, "SeeAlso",
 CellID->9699]
}, Open  ]],
Cell[CellGroupData[{
Cell[29010, 970, 314, 9, 70, "TutorialsSection",
 CellID->17735],
Cell[29327, 981, 150, 3, 70, "Tutorials",
 CellID->12303]
}, Open  ]],
Cell[CellGroupData[{
Cell[29514, 989, 319, 9, 70, "MoreAboutSection",
 CellID->854494349],
Cell[29836, 1000, 172, 3, 70, "MoreAbout",
 CellID->16063298],
Cell[30011, 1005, 173, 3, 70, "MoreAbout",
 CellID->10371604]
}, Open  ]],
Cell[30199, 1011, 27, 0, 70, "History"],
Cell[30229, 1013, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

