(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     41728,       1408]
NotebookOptionsPosition[     34891,       1170]
NotebookOutlinePosition[     36188,       1206]
CellTagsIndexPosition[     36102,       1201]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Spheroidal Functions" :> 
          Documentation`HelpLookup["paclet:tutorial/SpheroidalFunctions"]}, #,
          2]& , Evaluator -> Automatic], {
       1->"\<\"Spheroidal Functions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"LegendreQ" :> Documentation`HelpLookup["paclet:ref/LegendreQ"],
           "SpheroidalPS" :> 
          Documentation`HelpLookup["paclet:ref/SpheroidalPS"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"LegendreQ\"\>", 
       2->"\<\"SpheroidalPS\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Special Functions" :> 
          Documentation`HelpLookup["paclet:guide/SpecialFunctions"], 
          "Spheroidal and Related Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/SpheroidalAndRelatedFunctions"], 
          "New in 6.0: Mathematical Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicalFunctions"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Special Functions\"\>", 
       2->"\<\"Spheroidal and Related Functions\"\>", 
       3->"\<\"New in 6.0: Mathematical Functions\"\>"}, "\<\"more about\"\>",
        
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["SpheroidalQS", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["SpheroidalQS",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/SpheroidalQS"], "[", 
       RowBox[{
        StyleBox["n", "TI"], ",", 
        StyleBox["m", "TI"], ",", 
        StyleBox["\[Gamma]", "TR"], ",", 
        StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the angular spheroidal function ",
     Cell[BoxData[
      FormBox[
       RowBox[{
        SubscriptBox[
         StyleBox["QS",
          FontSlant->"Italic"], 
         RowBox[{"n", ",", "m"}]], "(", 
        RowBox[{"\[Gamma]", ",", "z"}], ")"}], TraditionalForm]], 
      "InlineMath"],
     " of the second kind."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->4145]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["\<\
Mathematical function, suitable for both symbolic and numerical manipulation.\
\
\>", "Notes",
 CellID->20640],

Cell[TextData[{
 "The angular spheroidal functions satisfy the differential equation ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"(", 
       RowBox[{"1", "-", 
        SuperscriptBox["z", "2"]}], ")"}], 
      SuperscriptBox["y", "\[DoublePrime]"]}], "-", 
     RowBox[{"2", "z", " ", 
      SuperscriptBox["y", "\[Prime]"]}], " ", "+", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"\[Lambda]", "+", 
        RowBox[{
         SuperscriptBox["\[Gamma]", "2"], "(", 
         RowBox[{"1", "-", 
          SuperscriptBox["z", "2"]}], ")"}], "-", 
        FractionBox[
         SuperscriptBox["m", "2"], 
         RowBox[{"1", "-", 
          SuperscriptBox["z", "2"]}]]}], ")"}], "y"}]}], "\[Equal]", "0"}], 
   TraditionalForm]], "InlineMath"],
 " with the spheroidal eigenvalue ",
 Cell[BoxData[
  FormBox["\[Lambda]", TraditionalForm]], "InlineMath"],
 " given by ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["SpheroidalEigenvalue",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SpheroidalEigenvalue"], "[", 
   RowBox[{
    StyleBox["n", "TI"], ",", 
    StyleBox["m", "TI"], ",", 
    StyleBox["\[Gamma]", "TR"]}], "]"}]], "InlineFormula"],
 "."
}], "Notes",
 CellID->28734],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["SpheroidalQS",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SpheroidalQS"], "[", 
   RowBox[{
    StyleBox["n", "TI"], ",", 
    StyleBox["m", "TI"], ",", "0", ",", 
    StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["LegendreQ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/LegendreQ"], "[", 
   RowBox[{
    StyleBox["n", "TI"], ",", 
    StyleBox["m", "TI"], ",", 
    StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
 "."
}], "Notes",
 CellID->32522],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["SpheroidalQS",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SpheroidalQS"], "[", 
   RowBox[{
    StyleBox["n", "TI"], ",", 
    StyleBox["m", "TI"], ",", 
    StyleBox["a", "TI"], ",", 
    StyleBox["\[Gamma]", "TR"], ",", 
    StyleBox["z", "TI"]}], "]"}]], "InlineFormula", "TemplateInclusion"],
 " gives spheroidal functions of type ",
 Cell[BoxData[
  FormBox["a", TraditionalForm]], "InlineMath"],
 ". The types are specified as for ",
 Cell[BoxData[
  ButtonBox["LegendreP",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LegendreP"]], "InlineFormula"],
 "."
}], "Notes",
 CellID->6268],

Cell[TextData[{
 "For certain special arguments, ",
 Cell[BoxData[
  ButtonBox["SpheroidalQS",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalQS"]], "InlineFormula"],
 " automatically evaluates to exact values."
}], "Notes",
 CellID->111660345],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SpheroidalQS",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalQS"]], "InlineFormula"],
 " can be evaluated to arbitrary numerical precision."
}], "Notes",
 CellID->31226],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SpheroidalQS",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalQS"]], "InlineFormula"],
 " automatically threads over lists."
}], "Notes",
 CellID->28162]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->199562261],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(3)", "ExampleCount"]
}], "ExampleSection",
 CellID->300794885],

Cell["Evaluate numerically:", "ExampleText",
 CellID->48231665],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalQS", "[", 
  RowBox[{"3", ",", "2", ",", "1", ",", "0.5"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->18458256],

Cell[BoxData[
 RowBox[{"-", 
  "0.26301932392342705306443351119`15.954589770191003"}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{138, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->168659276]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->290329948],

Cell["Expansion about the spherical case:", "ExampleText",
 CellID->543845061],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"SpheroidalQS", "[", 
    RowBox[{"n", ",", "m", ",", "c", ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"c", ",", "0", ",", "3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->656162328],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
   RowBox[{"LegendreQ", "[", 
    RowBox[{"n", ",", "m", ",", "2", ",", "x"}], "]"}], "+", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", 
       FractionBox[
        RowBox[{
         RowBox[{"(", 
          RowBox[{
           RowBox[{"-", "1"}], "+", "m", "+", "n"}], ")"}], " ", 
         RowBox[{"(", 
          RowBox[{"m", "+", "n"}], ")"}], " ", 
         RowBox[{"LegendreQ", "[", 
          RowBox[{
           RowBox[{
            RowBox[{"-", "2"}], "+", "n"}], ",", "m", ",", "2", ",", "x"}], 
          "]"}]}], 
        RowBox[{"2", " ", 
         SuperscriptBox[
          RowBox[{"(", 
           RowBox[{
            RowBox[{"-", "1"}], "+", 
            RowBox[{"2", " ", "n"}]}], ")"}], "2"], " ", 
         RowBox[{"(", 
          RowBox[{"1", "+", 
           RowBox[{"2", " ", "n"}]}], ")"}]}]]}], "+", 
      FractionBox[
       RowBox[{
        RowBox[{"(", 
         RowBox[{"1", "-", "m", "+", "n"}], ")"}], " ", 
        RowBox[{"(", 
         RowBox[{"2", "-", "m", "+", "n"}], ")"}], " ", 
        RowBox[{"LegendreQ", "[", 
         RowBox[{
          RowBox[{"2", "+", "n"}], ",", "m", ",", "2", ",", "x"}], "]"}]}], 
       RowBox[{"2", " ", 
        RowBox[{"(", 
         RowBox[{"1", "+", 
          RowBox[{"2", " ", "n"}]}], ")"}], " ", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{"3", "+", 
           RowBox[{"2", " ", "n"}]}], ")"}], "2"]}]]}], ")"}], " ", 
    SuperscriptBox["c", "2"]}], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "c", "]"}], "3"],
    SeriesData[$CellContext`c, 0, {}, 0, 3, 1],
    Editable->False]}],
  SeriesData[$CellContext`c, 0, {
    LegendreQ[$CellContext`n, $CellContext`m, 2, $CellContext`x], 0, 
    Rational[-1, 
       2] (-1 + $CellContext`m + $CellContext`n) ($CellContext`m + \
$CellContext`n) (-1 + 2 $CellContext`n)^(-2) (1 + 2 $CellContext`n)^(-1) 
     LegendreQ[-2 + $CellContext`n, $CellContext`m, 2, $CellContext`x] + 
    Rational[1, 2] (1 - $CellContext`m + $CellContext`n) (
      2 - $CellContext`m + $CellContext`n) (1 + 2 $CellContext`n)^(-1) (3 + 
       2 $CellContext`n)^(-2) 
     LegendreQ[2 + $CellContext`n, $CellContext`m, 2, $CellContext`x]}, 0, 3, 
   1],
  Editable->False]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{465, 68},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->386070490]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->132275320],

Cell[TextData[{
 "Plot ",
 Cell[BoxData[
  FormBox[
   TagBox[
    RowBox[{
     SubscriptBox[
      StyleBox["QS", "IT"], 
      RowBox[{
       TagBox["2",
        SpheroidalQS,
        Editable->True,
        Selectable->True], ",", 
       TagBox["0",
        SpheroidalQS,
        Editable->True,
        Selectable->True]}]], "(", 
     RowBox[{
      TagBox["10",
       SpheroidalQS,
       Editable->True,
       Selectable->True], ",", 
      TagBox["x",
       SpheroidalQS,
       Editable->True,
       Selectable->True]}], ")"}],
    InterpretTemplate[SpheroidalQS[
      SlotSequence[1]]& ],
    Editable->False,
    Selectable->False], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->193592132],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"SpheroidalQS", "[", 
    RowBox[{"2", ",", "0", ",", "10", ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
   RowBox[{"MaxRecursion", "\[Rule]", "1"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->457603330],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], LineBox[CompressedData["
1:eJwBcQSO+yFib1JlAgAAAEYAAAACAAAAPaz8aA1i6792VZ0GmS48v7VrxAUP
3eq/fyqsfDJGOL8K/Pp69C/qvwCPeM2XkDS/YIwx8NmC6b/sb77FXcoxv6yP
VrxM4ei/SyA5nSKHL7/4knuIvz/ovw/KOyilNyy/ztAs0auQ57+6Qrv1zjQp
v6MO3hmY4ea/dUl/tQiaJr+grAfGtTXmv+k7o5RDTiS/nEoxctOJ5b+zBXre
4Swiv4JsYXgpSeS/HDyR2f3dHL97A6p3cu3iv1mjiIy9+xS/YYDPJNao4b9z
xEO1Kz4Lv5W95ZicauC/1TtPdKCe+b663ygMrCLev9kP0Ez7LNQ+IxBAQlSe
279H4FuKBQUAP2odZFYbQdq/Tw6DNx/NBj+yKohq4uPYv/EtetBmFQ0/SPic
RQyN17+JGMJTEFYRP9/FsSA2Nta/cypKMdHMEz9ieaOpevbUv75SSv64wxU/
5SyVMr+2079V8xnhnmAXP3tVn7T2W9K/ZwDOAlmzGD8Rfqk2LgHRv/yPk9ZZ
jhk/JxkhzQB7z79gdJDU1eoZPyw27yyl88y/su9SsFLZGT9YPe5+LzbKv8Ro
3JUcShk/g0Tt0Ll4x7/Pvp0p9DsYP0zMzbAJyMS/GXjA69i7Fj8VVK6QWRfC
v3tBcrTdzBQ/bk+RmL0pv7/SlezoGKMSP7L2xQ/IJLq/fpAFa7EpED9Cclxr
nrO0v/mRD14pWgo/pdvljemErr/3R2eP9tsDP11RHLTbY5S/D82jWK8o6z65
CnvVwnWTP2HtjYxD7em+NeIBHoFYrz+2mRb7Il4Ev59wXsQ9t7Q/U53V5VJe
Cr8j8Lv5OsK5P8wPOFI67A+/Wpu3Smw5vz9nosIJW6oSv0mj2c1OWMI/jnpH
BQ8AFb9H+HXoIQfFP+cHvnzF4xa/RU0SA/W1xz85JMQNfVgYv2rW9MGSNso/
9LhV6TlKGb+PX9eAMLfMP+u7J8YN0hm/jv6ITehtzz8DWB4Lk+sZv8dOHQ1Q
EtE/KYWFEmuGGb9aOJlFkVbSP5clYcevtxi/7SEVftKa0z+m9TF4MIAXvzFL
oO+w2NQ/MFpwWRnuFb91dCthjxbWP4Luy6NpAhS/piie2Xpv1z9VdWpy848R
v9fcEFJmyNg/3pJjidKNDb8cq6YcNwraP18ZYogE0we/YHk85wdM2z8vtasZ
WKgBv8MrN4rDBd4/8zynv0Yw2b4mCbzopEjgP4teEfPMjvY+HbxrRQWI4T+H
Wg+WYcAJPwD6Aqly4uI/+8fn/gO8FD/3Ub1exSXkP4nyhZFlDxw/2zRfGyWE
5T8TRrgCXxsiPybGN5YjMOY/zl8NXgc8JD9wVxARItzmPzRhSZt9hiY/xHX6
NJN85z9wkS68xeMoPxmU5FgEHeg/d3RaHUqUKz/kd0IA/MroP0eQBDBMCC8/
rlugp/N46T/mzFGpIqcxP4LMD/hdG+o/re9OX4UxND9XPX9IyL3qP9Zfz43j
gzc/NQz3YdVG6z9YVkDNAUw7P34qJQI=
     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  Method->{},
  PlotRange->{{-1, 1}, {-0.0004300235365140773, 0.00041651767719605914`}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{184, 108},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->71263860]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->13683],

Cell["Evaluate for complex arguments and parameters:", "ExampleText",
 CellID->94393633],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalQS", "[", 
  RowBox[{"3", ",", "2", ",", 
   RowBox[{"1", "-", "I"}], ",", " ", "3.5"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->314925903],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "1024.0105859579035`"}], "+", 
  RowBox[{"296.9878796870481`", " ", "\[ImaginaryI]"}]}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{134, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->666346520]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->217504807],

Cell["Evaluate to high precision:", "ExampleText",
 CellID->283569038],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"SpheroidalQS", "[", 
    RowBox[{"2", ",", "1", ",", 
     RowBox[{"1", "/", "3"}], ",", 
     RowBox[{"1", "/", "2"}]}], "]"}], ",", "50"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->852018633],

Cell[BoxData["0.\
7187055748595749711171456146673072298121200928891831537556993109601`50."], \
"Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{368, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->71864951]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->3535261],

Cell["\<\
The precision of the output tracks the precision of the input:\
\>", "ExampleText",
 CellID->173158339],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalQS", "[", 
  RowBox[{"2", ",", "1", ",", 
   RowBox[{"1", "/", "3"}], ",", "0.50000000000000000000000"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->429954945],

Cell[BoxData["0.71870557485957497111714561466730632715`22.698970004336022"], \
"Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{172, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->191763801]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->165479942],

Cell[TextData[{
 "For certain parameters ",
 Cell[BoxData[
  ButtonBox["SpheroidalPS",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalPS"]], "InlineFormula"],
 " evaluates exactly:"
}], "ExampleText",
 CellID->160938561],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"With", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"n", "=", "3"}], "}"}], ",", 
   RowBox[{"SpheroidalQS", "[", 
    RowBox[{"n", ",", "1", ",", 
     RowBox[{"n", " ", 
      RowBox[{"Pi", "/", "2"}]}], ",", " ", "x"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->10306910],

Cell[BoxData[
 FractionBox[
  RowBox[{"2", " ", 
   SqrtBox[
    FractionBox["3", 
     RowBox[{"7", " ", 
      RowBox[{"(", 
       RowBox[{"EulerGamma", "-", 
        RowBox[{"CosIntegral", "[", 
         RowBox[{"6", " ", "\[Pi]"}], "]"}], "+", 
        RowBox[{"Log", "[", 
         RowBox[{"6", " ", "\[Pi]"}], "]"}]}], ")"}]}]]], " ", 
   RowBox[{"Sin", "[", 
    FractionBox[
     RowBox[{"3", " ", "\[Pi]", " ", "x"}], "2"], "]"}]}], 
  SqrtBox[
   RowBox[{"1", "-", 
    SuperscriptBox["x", "2"]}]]]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{298, 66},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->114371194]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalQS", "[", 
  RowBox[{
   RowBox[{"3", "/", "2"}], ",", 
   RowBox[{"1", "/", "2"}], ",", "c", ",", "x"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->283753173],

Cell[BoxData["0"], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->45561341]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->129130650],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TraditionalForm",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TraditionalForm"]], "InlineFormula"],
 " formatting:"
}], "ExampleText",
 CellID->54774772],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"SpheroidalQS", "[", 
   RowBox[{"n", ",", "m", ",", "c", ",", "\[Eta]"}], "]"}], "//", 
  "TraditionalForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->187276340],

Cell[GraphicsData["CompressedBitmap", "\<\
eJyNUksOgjAQHdoiGP9GExcuOAYn0JWHIMZEFkaj3B87/WCZqWgTpp3H9L35
9FQ118utaupzVRyf1eNan1/F4f7UkEwAoNTfegf63OqT+cq21bu1W7QSUbOg
MGimgRxPqf0DZvlQkBZCK/S+t26AKsKaBj4SqD63Q1NySxI/M9Qf3/GAIHGT
v9QoO1UfBbc6QVckKirvqbBHiEiniZdFP2e9hEOFr5hVhGNLcO/1NZaz54IR
nZONFV35HhVMkXbve93xvLp5D7LOWcSUIX9oDLyOLwqsEjGokkbvrHDPCOr+
jYOZRnrAOiFZ3JpO1THmfoLkBfSz/81PI2bE3zBO9s6inJjbgkTSWiwfzk65
pxhWhJYyLGPaUDILyRt9AmRa\
\>"], "Output",
 Evaluatable->False,
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{56, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//TraditionalForm=",
 CellID->9774668]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->5698],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SpheroidalQS",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalQS"]], "InlineFormula"],
 " of different types have different branch cut structures:"
}], "ExampleText",
 CellID->277064462],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalQS", "[", 
  RowBox[{"n", ",", "m", ",", "1", ",", "0", ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->135549909],

Cell[BoxData[
 RowBox[{"LegendreQ", "[", 
  RowBox[{"n", ",", "m", ",", "2", ",", "x"}], "]"}]], "Output",
 ImageSize->{143, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->117363349]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalQS", "[", 
  RowBox[{"n", ",", "m", ",", "2", ",", "0", ",", "x"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->591864808],

Cell[BoxData[
 RowBox[{"LegendreQ", "[", 
  RowBox[{"n", ",", "m", ",", "3", ",", "x"}], "]"}]], "Output",
 ImageSize->{143, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->459538625]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->9065],

Cell[TextData[{
 "Solve the spheroidal differential equation in terms of ",
 Cell[BoxData[
  ButtonBox["SpheroidalQS",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalQS"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->640948785],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"(", 
       RowBox[{"1", "-", 
        RowBox[{"x", "^", "2"}]}], ")"}], 
      RowBox[{
       RowBox[{"y", "''"}], "[", "x", "]"}]}], "-", 
     RowBox[{"2", "x", " ", 
      RowBox[{
       RowBox[{"y", "'"}], "[", "x", "]"}]}], "+", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{"SpheroidalEigenvalue", "[", 
         RowBox[{"n", ",", "m", ",", "c"}], "]"}], "+", 
        RowBox[{
         RowBox[{"c", "^", "2"}], 
         RowBox[{"(", 
          RowBox[{"1", "-", 
           RowBox[{"x", "^", "2"}]}], ")"}]}], " ", "-", " ", 
        FractionBox[
         RowBox[{"m", "^", "2"}], 
         RowBox[{"1", "-", 
          RowBox[{"x", "^", "2"}]}]]}], ")"}], 
      RowBox[{"y", "[", "x", "]"}]}]}], "\[Equal]", "0"}], ",", "y", ",", 
   "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->297005226],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    RowBox[{"Function", "[", 
     RowBox[{
      RowBox[{"{", "x", "}"}], ",", 
      RowBox[{
       RowBox[{
        RowBox[{"C", "[", "1", "]"}], " ", 
        RowBox[{"SpheroidalPS", "[", 
         RowBox[{"n", ",", "m", ",", "c", ",", "x"}], "]"}]}], "+", 
       RowBox[{
        RowBox[{"C", "[", "2", "]"}], " ", 
        RowBox[{"SpheroidalQS", "[", 
         RowBox[{"n", ",", "m", ",", "c", ",", "x"}], "]"}]}]}]}], "]"}]}], 
   "}"}], "}"}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{551, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->59291492]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->377793566],

Cell["\<\
Plot prolate and oblate versions of the same angular function:\
\>", "ExampleText",
 CellID->470010545],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"SpheroidalQS", "[", 
    RowBox[{"2", ",", "1", ",", "5", ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
   RowBox[{"MaxRecursion", "\[Rule]", "1"}], ",", 
   RowBox[{"PlotPoints", "\[Rule]", "25"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->663030826],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], 
    LineBox[{{-0.955478952857643, 0.436446485095354}, {-0.919849633427084, 
     0.275165531567269}, {-0.876403587664064, 
     0.197597431589825}, {-0.832957541901044, 
     0.155665594302539}, {-0.792390591994795, 
     0.129677883588256}, {-0.751823642088545, 
     0.109915219474608}, {-0.712052454705734, 
     0.0937948395757773}, {-0.672281267322923, 
     0.079471400093673}, {-0.585997084270842, 
     0.0513275481006646}, {-0.505471092932303, 
     0.0266565360808744}, {-0.418203293307307, 
     0.00119194293276159}, {-0.332527018729186, -0.0217541790200582}, \
{-0.292567977296896, -0.0314069547382603}, {-0.252608935864607, \
-0.0401979215701576}, {-0.209278990289088, -0.0485964671342299}, \
{-0.16594904471357, -0.0556567233061112}, {-0.125498194994823, \
-0.0609141105310718}, {-0.0850473452760752, -0.0647861690345242}, \
{-0.0453922580807655, -0.0671687078758898}, {-0.00573717088545587, \
-0.068109044192892}, {0.0372888204530827, -0.0674790393917436}, {
     0.0803148117916213, -0.0651451252575489}, {
     0.120461707273389, -0.0614734115043658}, {
     0.160608602755156, -0.0564270340609685}, {
     0.204126402380153, -0.0495091427736305}, {
     0.247644202005149, -0.0412230655265612}, {
     0.290366239106708, -0.0319150617532459}, {
     0.333088276208267, -0.0216131381104379}, {
     0.412774158697842, -0.000334582501133561}, {0.499201849473876, 
     0.0247788645615709}, {0.579871348536367, 0.0494121116202269}, {
     0.667282655885316, 0.0777546739094992}, {0.710192547036353, 
     0.093092795750521}, {0.75310243818739, 0.110476755521484}, {
     0.793133233481656, 0.130085606498542}, {0.833164028775922, 
     0.15582206391872}, {0.876565728213417, 0.197799107587446}, {
     0.919967427650911, 0.275470377688674}, {0.955444106347454, 
     0.436446485095354}}]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.618033988749895],
  Axes->True,
  Method->{},
  PlotRange->{{-1, 1}, {-0.068109044192892, 0.436446485095354}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{184, 106},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->162317617]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"SpheroidalQS", "[", 
    RowBox[{"2", ",", "1", ",", 
     RowBox[{"5", "I"}], ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
   RowBox[{"MaxRecursion", "\[Rule]", "1"}], ",", 
   RowBox[{"PlotPoints", "\[Rule]", "25"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->201523365],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{180, 106},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->215784466]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->18975],

Cell[TextData[{
 "Spheroidal functions do not generically evaluate for half-integer values of \
",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->348355882],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalQS", "[", 
  RowBox[{
   RowBox[{"3", "/", "2"}], ",", "1", ",", "1", ",", "0.5"}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->175415402],

Cell[BoxData[
 RowBox[{"SpheroidalQS", "[", 
  RowBox[{
   FractionBox["3", "2"], ",", "1", ",", "1", ",", "0.5`"}], "]"}]], "Output",\

 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{182, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->8797745]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->12931],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["LegendreQ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/LegendreQ"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SpheroidalPS",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SpheroidalPS"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->18720]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->19800],

Cell[TextData[ButtonBox["Spheroidal Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/SpheroidalFunctions"]], "Tutorials",
 CellID->12303]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->91738450],

Cell[TextData[ButtonBox["Special Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/SpecialFunctions"]], "MoreAbout",
 CellID->54573673],

Cell[TextData[ButtonBox["Spheroidal and Related Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/SpheroidalAndRelatedFunctions"]], "MoreAbout",
 CellID->228278880],

Cell[TextData[ButtonBox["New in 6.0: Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicalFunctions"]], "MoreAbout",
 CellID->466210729]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"SpheroidalQS - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 23, 33.2651082}", "context" -> "System`", 
    "keywords" -> {"spheroidal functions", "spheroidal harmonics"}, "index" -> 
    True, "label" -> "Built-in Mathematica Symbol", "language" -> "en", 
    "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "SpheroidalQS[n, m, \\[Gamma], z] gives the angular spheroidal function \
QS n, m (\\[Gamma], z) of the second kind.", "synonyms" -> {}, "title" -> 
    "SpheroidalQS", "type" -> "Symbol", "uri" -> "ref/SpheroidalQS"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[9195, 299, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->199562261]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 35959, 1194}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2239, 55, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2862, 82, 1577, 37, 70, "ObjectNameGrid"],
Cell[4442, 121, 906, 29, 70, "Usage",
 CellID->4145]
}, Open  ]],
Cell[CellGroupData[{
Cell[5385, 155, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[5863, 172, 120, 4, 70, "Notes",
 CellID->20640],
Cell[5986, 178, 1223, 40, 70, "Notes",
 CellID->28734],
Cell[7212, 220, 586, 22, 70, "Notes",
 CellID->32522],
Cell[7801, 244, 657, 22, 70, "Notes",
 CellID->6268],
Cell[8461, 268, 256, 8, 70, "Notes",
 CellID->111660345],
Cell[8720, 278, 226, 7, 70, "Notes",
 CellID->31226],
Cell[8949, 287, 209, 7, 70, "Notes",
 CellID->28162]
}, Closed]],
Cell[CellGroupData[{
Cell[9195, 299, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->199562261],
Cell[CellGroupData[{
Cell[9580, 313, 148, 5, 70, "ExampleSection",
 CellID->300794885],
Cell[9731, 320, 63, 1, 70, "ExampleText",
 CellID->48231665],
Cell[CellGroupData[{
Cell[9819, 325, 152, 4, 28, "Input",
 CellID->18458256],
Cell[9974, 331, 276, 9, 36, "Output",
 CellID->168659276]
}, Open  ]],
Cell[10265, 343, 125, 3, 70, "ExampleDelimiter",
 CellID->290329948],
Cell[10393, 348, 78, 1, 70, "ExampleText",
 CellID->543845061],
Cell[CellGroupData[{
Cell[10496, 353, 269, 8, 70, "Input",
 CellID->656162328],
Cell[10768, 363, 2477, 71, 89, "Output",
 CellID->386070490]
}, Open  ]],
Cell[13260, 437, 125, 3, 70, "ExampleDelimiter",
 CellID->132275320],
Cell[13388, 442, 729, 32, 70, "ExampleText",
 CellID->193592132],
Cell[CellGroupData[{
Cell[14142, 478, 340, 10, 70, "Input",
 CellID->457603330],
Cell[14485, 490, 2100, 44, 129, "Output",
 CellID->71263860]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[16634, 540, 216, 7, 70, "ExampleSection",
 CellID->13683],
Cell[16853, 549, 88, 1, 70, "ExampleText",
 CellID->94393633],
Cell[CellGroupData[{
Cell[16966, 554, 182, 5, 70, "Input",
 CellID->314925903],
Cell[17151, 561, 317, 10, 36, "Output",
 CellID->666346520]
}, Open  ]],
Cell[17483, 574, 125, 3, 70, "ExampleDelimiter",
 CellID->217504807],
Cell[17611, 579, 70, 1, 70, "ExampleText",
 CellID->283569038],
Cell[CellGroupData[{
Cell[17706, 584, 258, 8, 70, "Input",
 CellID->852018633],
Cell[17967, 594, 282, 9, 36, "Output",
 CellID->71864951]
}, Open  ]],
Cell[18264, 606, 123, 3, 70, "ExampleDelimiter",
 CellID->3535261],
Cell[18390, 611, 113, 3, 70, "ExampleText",
 CellID->173158339],
Cell[CellGroupData[{
Cell[18528, 618, 202, 6, 70, "Input",
 CellID->429954945],
Cell[18733, 626, 267, 8, 36, "Output",
 CellID->191763801]
}, Open  ]],
Cell[19015, 637, 125, 3, 70, "ExampleDelimiter",
 CellID->165479942],
Cell[19143, 642, 232, 8, 70, "ExampleText",
 CellID->160938561],
Cell[CellGroupData[{
Cell[19400, 654, 315, 10, 70, "Input",
 CellID->10306910],
Cell[19718, 666, 700, 24, 87, "Output",
 CellID->114371194]
}, Open  ]],
Cell[CellGroupData[{
Cell[20455, 695, 199, 6, 70, "Input",
 CellID->283753173],
Cell[20657, 703, 205, 7, 36, "Output",
 CellID->45561341]
}, Open  ]],
Cell[20877, 713, 125, 3, 70, "ExampleDelimiter",
 CellID->129130650],
Cell[21005, 718, 202, 7, 70, "ExampleText",
 CellID->54774772],
Cell[CellGroupData[{
Cell[21232, 729, 198, 6, 70, "Input",
 CellID->187276340],
Cell[21433, 737, 664, 16, 70, "Output",
 Evaluatable->False,
 CellID->9774668]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[22146, 759, 238, 7, 70, "ExampleSection",
 CellID->5698],
Cell[22387, 768, 242, 7, 70, "ExampleText",
 CellID->277064462],
Cell[CellGroupData[{
Cell[22654, 779, 161, 4, 70, "Input",
 CellID->135549909],
Cell[22818, 785, 237, 7, 36, "Output",
 CellID->117363349]
}, Open  ]],
Cell[CellGroupData[{
Cell[23092, 797, 161, 4, 70, "Input",
 CellID->591864808],
Cell[23256, 803, 237, 7, 36, "Output",
 CellID->459538625]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[23542, 816, 222, 7, 70, "ExampleSection",
 CellID->9065],
Cell[23767, 825, 246, 8, 70, "ExampleText",
 CellID->640948785],
Cell[CellGroupData[{
Cell[24038, 837, 938, 31, 70, "Input",
 CellID->297005226],
Cell[24979, 870, 716, 23, 36, "Output",
 CellID->59291492]
}, Open  ]],
Cell[25710, 896, 125, 3, 70, "ExampleDelimiter",
 CellID->377793566],
Cell[25838, 901, 113, 3, 70, "ExampleText",
 CellID->470010545],
Cell[CellGroupData[{
Cell[25976, 908, 389, 11, 70, "Input",
 CellID->663030826],
Cell[26368, 921, 2301, 49, 127, "Output",
 CellID->162317617]
}, Open  ]],
Cell[CellGroupData[{
Cell[28706, 975, 410, 12, 70, "Input",
 CellID->201523365],
Cell[29119, 989, 2501, 46, 70, "Output",
 Evaluatable->False,
 CellID->215784466]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[31669, 1041, 226, 7, 70, "ExampleSection",
 CellID->18975],
Cell[31898, 1050, 206, 7, 70, "ExampleText",
 CellID->348355882],
Cell[CellGroupData[{
Cell[32129, 1061, 179, 6, 70, "Input",
 CellID->175415402],
Cell[32311, 1069, 313, 11, 51, "Output",
 CellID->8797745]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[32685, 1087, 311, 9, 70, "SeeAlsoSection",
 CellID->12931],
Cell[32999, 1098, 461, 17, 70, "SeeAlso",
 CellID->18720]
}, Open  ]],
Cell[CellGroupData[{
Cell[33497, 1120, 314, 9, 70, "TutorialsSection",
 CellID->19800],
Cell[33814, 1131, 150, 3, 70, "Tutorials",
 CellID->12303]
}, Open  ]],
Cell[CellGroupData[{
Cell[34001, 1139, 318, 9, 70, "MoreAboutSection",
 CellID->91738450],
Cell[34322, 1150, 144, 3, 70, "MoreAbout",
 CellID->54573673],
Cell[34469, 1155, 173, 3, 70, "MoreAbout",
 CellID->228278880],
Cell[34645, 1160, 174, 3, 70, "MoreAbout",
 CellID->466210729]
}, Open  ]],
Cell[34834, 1166, 27, 0, 70, "History"],
Cell[34864, 1168, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

