(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     64367,       2315]
NotebookOptionsPosition[     55027,       1984]
NotebookOutlinePosition[     56534,       2024]
CellTagsIndexPosition[     56448,       2019]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Symmetric Polynomials" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/SymmetricPolynomials"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Symmetric Polynomials\"\>"}, "\<\"tutorials\"\>",
        
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"SymmetricPolynomial" :> 
          Documentation`HelpLookup["paclet:ref/SymmetricPolynomial"], 
          "PolynomialReduce" :> 
          Documentation`HelpLookup["paclet:ref/PolynomialReduce"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"SymmetricPolynomial\"\>", 
       2->"\<\"PolynomialReduce\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Polynomial Algebra" :> 
          Documentation`HelpLookup["paclet:guide/PolynomialAlgebra"], 
          "Polynomial Factoring & Decomposition" :> 
          Documentation`HelpLookup["paclet:guide/PolynomialFactoring"], 
          "New in 6.0: Symbolic Computation" :> 
          Documentation`HelpLookup["paclet:guide/NewIn60AlgebraicComputing"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Polynomial Algebra\"\>", 
       2->"\<\"Polynomial Factoring & Decomposition\"\>", 
       3->"\<\"New in 6.0: Symbolic Computation\"\>", 
       4->"\<\"New in 6.0: Mathematics & Algorithms\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["SymmetricReduction", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["SymmetricReduction",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/SymmetricReduction"], "[", 
       RowBox[{
        StyleBox["f", "TI"], ",", 
        RowBox[{"{", 
         RowBox[{
          SubscriptBox[
           StyleBox["x", "TI"], 
           StyleBox["1", "TR"]], ",", 
          StyleBox["\[Ellipsis]", "TR"], ",", 
          SubscriptBox[
           StyleBox["x", "TI"], 
           StyleBox["n", "TI"]]}], "}"}]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives a pair of polynomials ",
     Cell[BoxData[
      FormBox[
       RowBox[{"{", 
        RowBox[{"p", ",", "q"}], "}"}], TraditionalForm]], "InlineMath"],
     " in ",
     Cell[BoxData[
      FormBox[
       RowBox[{
        SubscriptBox["x", "1"], ",", "\[Ellipsis]", ",", 
        SubscriptBox["x", "n"]}], TraditionalForm]], "InlineMath"],
     " such that ",
     Cell[BoxData[
      FormBox[
       RowBox[{"f", "==", 
        RowBox[{"p", "+", "q"}]}], TraditionalForm]], "InlineMath"],
     ", where ",
     Cell[BoxData[
      FormBox["p", TraditionalForm]], "InlineMath"],
     " is the symmetric part and ",
     Cell[BoxData[
      FormBox["q", TraditionalForm]], "InlineMath"],
     " is the remainder."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["SymmetricReduction",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/SymmetricReduction"], "[", 
       RowBox[{
        StyleBox["f", "TI"], ",", 
        RowBox[{"{", 
         RowBox[{
          SubscriptBox[
           StyleBox["x", "TI"], 
           StyleBox["1", "TR"]], ",", 
          StyleBox["\[Ellipsis]", "TR"], ",", 
          SubscriptBox[
           StyleBox["x", "TI"], 
           StyleBox["n", "TI"]]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          SubscriptBox[
           StyleBox["s", "TI"], 
           StyleBox["1", "TR"]], ",", 
          StyleBox["\[Ellipsis]", "TR"], ",", 
          SubscriptBox[
           StyleBox["s", "TI"], 
           StyleBox["n", "TI"]]}], "}"}]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the pair ",
     Cell[BoxData[
      FormBox[
       RowBox[{"{", 
        RowBox[{"p", ",", "q"}], "}"}], TraditionalForm]], "InlineMath"],
     " with the elementary symmetric polynomials in ",
     Cell[BoxData[
      FormBox["p", TraditionalForm]], "InlineMath"],
     " replaced by ",
     Cell[BoxData[
      FormBox[
       RowBox[{
        SubscriptBox["s", "1"], ",", "\[Ellipsis]", ",", 
        SubscriptBox["s", "n"]}], TraditionalForm]], "InlineMath"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->23775]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "If ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]], "InlineMath"],
 " is a symmetric polynomial, then ",
 Cell[BoxData[
  FormBox["p", TraditionalForm]], "InlineMath"],
 " is the unique polynomial in elementary symmetric polynomials equal to ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]], "InlineMath"],
 ", and ",
 Cell[BoxData[
  FormBox["q", TraditionalForm]], "InlineMath"],
 " is zero."
}], "Notes",
 CellID->13000],

Cell[TextData[{
 "If ",
 Cell[BoxData[
  FormBox[
   RowBox[{"f", " "}], TraditionalForm]], "InlineMath"],
 " is not a symmetric polynomial, then the output ",
 Cell[BoxData[
  FormBox["p", TraditionalForm]], "InlineMath"],
 " is not unique, but depends on the ordering of its variables."
}], "Notes",
 CellID->12212],

Cell[TextData[{
 "For a given ordering, a nonsymmetric polynomial ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]], "InlineMath"],
 " can be expressed uniquely as a sum of its symmetric part ",
 Cell[BoxData[
  FormBox["p", TraditionalForm]], "InlineMath"],
 " and a remainder ",
 Cell[BoxData[
  FormBox["q", TraditionalForm]], "InlineMath"],
 " that does not contain descending monomials. A monomial ",
 Cell[BoxData[
  FormBox[
   RowBox[{"c", " ", 
    SubsuperscriptBox["x", "1", 
     SubscriptBox["e", "1"]], " ", "\[Ellipsis]", " ", 
    SubsuperscriptBox["x", "n", 
     SubscriptBox["e", "n"]]}], TraditionalForm]], "InlineMath"],
 " is called descending if ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["e", "1"], "\[GreaterEqual]", "\[Ellipsis]", 
    "\[GreaterEqual]", 
    SubscriptBox["e", "n"]}], TraditionalForm]], "InlineMath"],
 "."
}], "Notes",
 CellID->21100],

Cell[TextData[{
 "Changing the ordering of the variables may produce different pairs ",
 Cell[BoxData[
  FormBox[
   RowBox[{"{", 
    RowBox[{"p", ",", "q"}], "}"}], TraditionalForm]], "InlineMath"],
 "."
}], "Notes",
 CellID->7551],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SymmetricReduction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SymmetricReduction"]], "InlineFormula"],
 " does not check to see that ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]], "InlineMath"],
 " is a polynomial, and will attempt to symmetrize the polynomial part of ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]], "InlineMath"],
 "."
}], "Notes",
 CellID->25476]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->288700245],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(3)", "ExampleCount"]
}], "ExampleSection",
 CellID->748647321],

Cell["\<\
Write a symmetric polynomial as a sum of elementary symmetric polynomials:\
\>", "ExampleText",
 CellID->958837676],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SymmetricReduction", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"x", "^", "2"}], "+", 
    RowBox[{"y", "^", "2"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->462908957],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"-", "2"}], " ", "x", " ", "y"}], "+", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"x", "+", "y"}], ")"}], "2"]}], ",", "0"}], "}"}]], "Output",
 ImageSize->{130, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->29256572]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->20448923],

Cell["\<\
Write a nonsymmetric polynomial as a symmetric part and a remainder:\
\>", "ExampleText",
 CellID->57567550],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SymmetricReduction", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"x", "^", "2"}], "-", 
    RowBox[{"y", "^", "2"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->752424637],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"-", "2"}], " ", "x", " ", "y"}], "+", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"x", "+", "y"}], ")"}], "2"]}], ",", 
   RowBox[{
    RowBox[{"-", "2"}], " ", 
    SuperscriptBox["y", "2"]}]}], "}"}]], "Output",
 ImageSize->{153, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->35756098]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->53173094],

Cell[TextData[{
 "Name the first two elementary symmetric polynomials ",
 Cell[BoxData["s1"], "InlineFormula"],
 " and ",
 Cell[BoxData["s2"], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->118134771],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SymmetricReduction", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"x", "^", "2"}], "-", 
    RowBox[{"y", "^", "2"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"s1", ",", "s2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->49828701],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    SuperscriptBox["s1", "2"], "-", 
    RowBox[{"2", " ", "s2"}]}], ",", 
   RowBox[{
    RowBox[{"-", "2"}], " ", 
    SuperscriptBox["y", "2"]}]}], "}"}]], "Output",
 ImageSize->{116, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->16405196]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->6461],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SymmetricReduction", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"x", "^", "10"}], "-", 
    RowBox[{"y", "^", "10"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->230645080],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"-", "2"}], " ", 
     SuperscriptBox["x", "5"], " ", 
     SuperscriptBox["y", "5"]}], "+", 
    RowBox[{"25", " ", 
     SuperscriptBox["x", "4"], " ", 
     SuperscriptBox["y", "4"], " ", 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"x", "+", "y"}], ")"}], "2"]}], "-", 
    RowBox[{"50", " ", 
     SuperscriptBox["x", "3"], " ", 
     SuperscriptBox["y", "3"], " ", 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"x", "+", "y"}], ")"}], "4"]}], "+", 
    RowBox[{"35", " ", 
     SuperscriptBox["x", "2"], " ", 
     SuperscriptBox["y", "2"], " ", 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"x", "+", "y"}], ")"}], "6"]}], "-", 
    RowBox[{"10", " ", "x", " ", "y", " ", 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"x", "+", "y"}], ")"}], "8"]}], "+", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"x", "+", "y"}], ")"}], "10"]}], ",", 
   RowBox[{
    RowBox[{"-", "2"}], " ", 
    SuperscriptBox["y", "10"]}]}], "}"}]], "Output",
 ImageSize->{325, 37},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->832974987]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SymmetricReduction", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"x", "^", "10"}], "-", 
    RowBox[{"y", "^", "10"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"s1", ",", "s2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->629681612],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    SuperscriptBox["s1", "10"], "-", 
    RowBox[{"10", " ", 
     SuperscriptBox["s1", "8"], " ", "s2"}], "+", 
    RowBox[{"35", " ", 
     SuperscriptBox["s1", "6"], " ", 
     SuperscriptBox["s2", "2"]}], "-", 
    RowBox[{"50", " ", 
     SuperscriptBox["s1", "4"], " ", 
     SuperscriptBox["s2", "3"]}], "+", 
    RowBox[{"25", " ", 
     SuperscriptBox["s1", "2"], " ", 
     SuperscriptBox["s2", "4"]}], "-", 
    RowBox[{"2", " ", 
     SuperscriptBox["s2", "5"]}]}], ",", 
   RowBox[{
    RowBox[{"-", "2"}], " ", 
    SuperscriptBox["y", "10"]}]}], "}"}]], "Output",
 ImageSize->{412, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->379122588]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->123114972],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SymmetricReduction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SymmetricReduction"]], "InlineFormula"],
 " will reduce the polynomial part of an expression:"
}], "ExampleText",
 CellID->16991],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SymmetricReduction", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"x", " ", "y"}], "+", 
    RowBox[{"Sin", "[", 
     RowBox[{
      RowBox[{"x", " ", "y"}], "+", 
      RowBox[{"x", " ", "z"}], "+", 
      RowBox[{"y", " ", "z"}]}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y", ",", "z"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"s1", ",", "s2", ",", "s3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->31461],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"s2", "+", 
    RowBox[{"Sin", "[", 
     RowBox[{
      RowBox[{"x", " ", "y"}], "+", 
      RowBox[{"x", " ", "z"}], "+", 
      RowBox[{"y", " ", "z"}]}], "]"}]}], ",", 
   RowBox[{
    RowBox[{
     RowBox[{"-", "x"}], " ", "z"}], "-", 
    RowBox[{"y", " ", "z"}]}]}], "}"}]], "Output",
 ImageSize->{222, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->381886287]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->14179],

Cell[TextData[{
 "Let the roots of the equation ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SuperscriptBox["x", "3"], "+", 
     RowBox[{"a", " ", 
      SuperscriptBox["x", "2"]}], "+", 
     RowBox[{"b", " ", "x"}], "+", "c"}], "==", "0"}], TraditionalForm]], 
  "InlineMath"],
 " be ",
 Cell[BoxData[
  StyleBox["\[Alpha]", "TR"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  StyleBox["\[Beta]", "TR"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  StyleBox["\[Gamma]", "TR"]], "InlineFormula"],
 ". The coefficients ",
 Cell[BoxData[
  StyleBox["a", "TI"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  StyleBox["b", "TI"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  StyleBox["c", "TI"]], "InlineFormula"],
 " are trivially related to the symmetric polynomials of  ",
 Cell[BoxData[
  StyleBox["\[Alpha]", "TR"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  StyleBox["\[Beta]", "TR"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  StyleBox["\[Gamma]", "TR"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->24117],

Cell[BoxData[
 RowBox[{
  RowBox[{"poly1", "=", 
   RowBox[{
    SuperscriptBox["x", "3"], "+", 
    RowBox[{"a", " ", 
     SuperscriptBox["x", "2"]}], "+", 
    RowBox[{"b", " ", "x"}], "+", "c"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2317],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SolveAlways", "[", 
  RowBox[{
   RowBox[{"poly1", "==", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"x", "-", "\[Alpha]"}], ")"}], 
     RowBox[{"(", 
      RowBox[{"x", "-", "\[Beta]"}], ")"}], 
     RowBox[{"(", 
      RowBox[{"x", "-", "\[Gamma]"}], ")"}]}]}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->19914],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"a", "\[Rule]", 
     RowBox[{
      RowBox[{"-", "\[Alpha]"}], "-", "\[Beta]", "-", "\[Gamma]"}]}], ",", 
    RowBox[{"b", "\[Rule]", 
     RowBox[{
      RowBox[{"\[Alpha]", " ", "\[Beta]"}], "+", 
      RowBox[{"\[Alpha]", " ", "\[Gamma]"}], "+", 
      RowBox[{"\[Beta]", " ", "\[Gamma]"}]}]}], ",", 
    RowBox[{"c", "\[Rule]", 
     RowBox[{
      RowBox[{"-", "\[Alpha]"}], " ", "\[Beta]", " ", "\[Gamma]"}]}]}], "}"}],
   "}"}]], "Output",
 ImageSize->{293, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->94316343]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"SymmetricPolynomial", "[", 
    RowBox[{"#", ",", 
     RowBox[{"{", 
      RowBox[{"\[Alpha]", ",", "\[Beta]", ",", "\[Gamma]"}], "}"}]}], "]"}], 
   "&"}], "/@", 
  RowBox[{"Range", "[", "3", "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->1441],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"\[Alpha]", "+", "\[Beta]", "+", "\[Gamma]"}], ",", 
   RowBox[{
    RowBox[{"\[Alpha]", " ", "\[Beta]"}], "+", 
    RowBox[{"\[Alpha]", " ", "\[Gamma]"}], "+", 
    RowBox[{"\[Beta]", " ", "\[Gamma]"}]}], ",", 
   RowBox[{"\[Alpha]", " ", "\[Beta]", " ", "\[Gamma]"}]}], "}"}]], "Output",
 ImageSize->{194, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->583222659]
}, Open  ]],

Cell[TextData[{
 "A similar expression holds for the monic polynomial with roots ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Alpha]", "2"], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Beta]", "2"], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Gamma]", "2"], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->21867],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SolveAlways", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     SuperscriptBox["x", "3"], "+", 
     RowBox[{
      OverscriptBox["a", "_"], 
      SuperscriptBox["x", "2"]}], "+", 
     RowBox[{
      OverscriptBox["b", "_"], "x"}], "+", 
     OverscriptBox["c", "_"]}], "==", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"x", "-", 
       SuperscriptBox["\[Alpha]", "2"]}], ")"}], 
     RowBox[{"(", 
      RowBox[{"x", "-", 
       SuperscriptBox["\[Beta]", "2"]}], ")"}], 
     RowBox[{"(", 
      RowBox[{"x", "-", 
       SuperscriptBox["\[Gamma]", "2"]}], ")"}]}]}], ",", "x"}], 
  "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->7028],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzVWNtOwkAQXVoqlCKoVFCJiT7pg/pCNH6Bxhg/gqCJPhiN8v+1287i5uxM
F6qSaELbmZ3LmTO767YP0/nz0+t0/jKbHt1+TN+fX2afRzdvH7kqbCjVmCml
dvZU/pzlT8VP/z3mP6Wa+TXIVRHd9cgd6KLCMtTX0Hhnhact68u9xyZYc6Rr
kPeooi7oz0E+Ibsh6PukT0B/CXKP7Db/rAojK0Kjs10wVdh2u0w19niH4lwx
1dh2yF69qvR4lIsx6LuEIgD9NqDA/nDIIsrBIeZytJkc2MNI8I3Bt5+59VbF
Np1EPCGjlzji8Hezao7qda+wihaygL+DXmYWOp7D7Htl6nuLtULsWP+iXq+n
Uw91I2A612QjGmuz+tq1GZERc3sV3y9kglttbg5tFSu1WCelWTlVMGKHbHg2
EhptgdeAtY4oL4+K56M6v71H6cdkJesByBsr8sLPDsRT1Z9Gtsyq0+NNaVY5
GOSVZu8HqdwD73pfYa1VMsT1ZDmvFLyQd54ZP25u7f7mTon7QMJaFR1sfiOi
7jl2ev76/idl4i7QdbKw55YlWWR93ehsDXV5ciuQ95DfOblo8YDu9u6QWkhK
relNBFG2HMwDJl7ixEsZq7aQY8fx7giW+nSlDfeXwhALUXD3r8fvCWGwGSrz
BjQyojvuGWPSt0TPId03vTl6dJ94Y5pYyMbPZtcx06FAuf4jkM255BD0bcEf
33Fw/Up5kfu+gFvKy52rf87eqac6ZOvMg8rHDo4jKxOPv+8NcT0smLk+/pds
4DcMfO/FE88I5DOQ8YyB/r5xfF+eePxRDjPu+wt+Y/hvM3urrK7xBe9ig+Q=
\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{355, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->21971802]
}, Open  ]],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["SymmetricReduction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SymmetricReduction"]], "InlineFormula"],
 " to solve for ",
 Cell[BoxData[
  OverscriptBox[
   StyleBox["a", "TI"], "_"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  OverscriptBox[
   StyleBox["b", "TI"], "_"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  OverscriptBox[
   StyleBox["c", "TI"], "_"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->27909],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{
    OverscriptBox["a", "_"], ",", 
    OverscriptBox["b", "_"], ",", 
    OverscriptBox["c", "_"]}], "}"}], "=", 
  RowBox[{"Part", "[", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"SymmetricReduction", "[", 
       RowBox[{"#", ",", 
        RowBox[{"{", 
         RowBox[{"\[Alpha]", ",", "\[Beta]", ",", "\[Gamma]"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", "a"}], ",", "b", ",", 
          RowBox[{"-", "c"}]}], "}"}]}], "]"}], "&"}], "/@", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"-", 
         SuperscriptBox["\[Alpha]", "2"]}], "-", 
        SuperscriptBox["\[Beta]", "2"], "-", 
        SuperscriptBox["\[Gamma]", "2"]}], ",", 
       RowBox[{
        RowBox[{
         SuperscriptBox["\[Alpha]", "2"], 
         SuperscriptBox["\[Beta]", "2"]}], "+", 
        RowBox[{
         SuperscriptBox["\[Alpha]", "2"], 
         SuperscriptBox["\[Gamma]", "2"]}], "+", 
        RowBox[{
         SuperscriptBox["\[Beta]", "2"], 
         SuperscriptBox["\[Gamma]", "2"]}]}], ",", 
       RowBox[{
        RowBox[{"-", 
         SuperscriptBox["\[Alpha]", "2"]}], 
        SuperscriptBox["\[Beta]", "2"], 
        SuperscriptBox["\[Gamma]", "2"]}]}], "}"}]}], ",", "All", ",", "1"}], 
   "]"}]}]], "Input",
 CellLabel->"In[5]:=",
 CellID->13973],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"-", 
     SuperscriptBox["a", "2"]}], "+", 
    RowBox[{"2", " ", "b"}]}], ",", 
   RowBox[{
    SuperscriptBox["b", "2"], "-", 
    RowBox[{"2", " ", "a", " ", "c"}]}], ",", 
   RowBox[{"-", 
    SuperscriptBox["c", "2"]}]}], "}"}]], "Output",
 ImageSize->{163, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->568762279]
}, Open  ]],

Cell[TextData[{
 "The monic polynomial with roots ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Alpha]", "2"], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Beta]", "2"], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Gamma]", "2"], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->11503],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"poly2", "=", 
  RowBox[{
   SuperscriptBox["x", "3"], "+", 
   RowBox[{
    OverscriptBox["a", "_"], 
    SuperscriptBox["x", "2"]}], "+", 
   RowBox[{
    OverscriptBox["b", "_"], "x"}], "+", 
   OverscriptBox["c", "_"]}]}]], "Input",
 CellLabel->"In[6]:=",
 CellID->26569],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", 
   SuperscriptBox["c", "2"]}], "+", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     SuperscriptBox["b", "2"], "-", 
     RowBox[{"2", " ", "a", " ", "c"}]}], ")"}], " ", "x"}], "+", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", 
      SuperscriptBox["a", "2"]}], "+", 
     RowBox[{"2", " ", "b"}]}], ")"}], " ", 
   SuperscriptBox["x", "2"]}], "+", 
  SuperscriptBox["x", "3"]}]], "Output",
 ImageSize->{230, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->232487381]
}, Open  ]],

Cell["Check:", "ExampleText",
 CellID->1883],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   SuperscriptBox["x", "2"], "/.", 
   RowBox[{"NSolve", "[", 
    RowBox[{
     RowBox[{"poly1", "/.", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"a", "->", "3"}], ",", 
        RowBox[{"b", "->", "5"}], ",", 
        RowBox[{"c", "->", "7"}]}], "}"}]}], ",", "x"}], "]"}]}], "//", 
  "Sort"}]], "Input",
 CellLabel->"In[7]:=",
 CellID->25367],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"-", "2.8751297941627785`"}], "-", 
    RowBox[{"1.4313819935719294`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "2.8751297941627785`"}], "+", 
    RowBox[{"1.4313819935719294`", " ", "\[ImaginaryI]"}]}], ",", 
   "4.750259588325557`"}], "}"}]], "Output",
 ImageSize->{348, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->147536678]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"x", "/.", 
   RowBox[{"NSolve", "[", 
    RowBox[{
     RowBox[{"poly2", "/.", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"a", "->", "3"}], ",", 
        RowBox[{"b", "->", "5"}], ",", 
        RowBox[{"c", "->", "7"}]}], "}"}]}], ",", "x"}], "]"}]}], "//", 
  "Sort"}]], "Input",
 CellLabel->"In[8]:=",
 CellID->28402],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"-", "2.875129794162779`"}], "-", 
    RowBox[{"1.4313819935719294`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "2.875129794162779`"}], "+", 
    RowBox[{"1.4313819935719294`", " ", "\[ImaginaryI]"}]}], ",", 
   "4.750259588325558`"}], "}"}]], "Output",
 ImageSize->{348, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->33062638]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->17157],

Cell["\<\
Consider solving the following symmetric system of equations:\
\>", "ExampleText",
 CellID->28681],

Cell[BoxData[
 RowBox[{
  RowBox[{"system", "=", 
   RowBox[{"{", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"Cos", "[", 
        SubscriptBox["a", "1"], "]"}], "+", 
       RowBox[{"Cos", "[", 
        SubscriptBox["a", "2"], "]"}], "+", 
       RowBox[{"Cos", "[", 
        SubscriptBox["a", "3"], "]"}], "-", "k"}], "==", "0"}], ",", 
     "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{
       RowBox[{"Cos", "[", 
        RowBox[{"5", 
         SubscriptBox["a", "1"]}], "]"}], "+", 
       RowBox[{"Cos", "[", 
        RowBox[{"5", 
         SubscriptBox["a", "2"]}], "]"}], "+", 
       RowBox[{"Cos", "[", 
        RowBox[{"5", 
         SubscriptBox["a", "3"]}], "]"}]}], "==", "0"}], ",", 
     "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{
       RowBox[{"Cos", "[", 
        RowBox[{"7", 
         SubscriptBox["a", "1"]}], "]"}], "+", 
       RowBox[{"Cos", "[", 
        RowBox[{"7", 
         SubscriptBox["a", "2"]}], "]"}], "+", 
       RowBox[{"Cos", "[", 
        RowBox[{"7", 
         SubscriptBox["a", "3"]}], "]"}]}], "==", "0"}]}], 
    "\[IndentingNewLine]", "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->15567],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["ChebyshevT",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ChebyshevT"]], "InlineFormula"],
 " to convert to a symmetric system of polynomials:"
}], "ExampleText",
 CellID->8310],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"eqns", "=", 
  RowBox[{"system", "/.", 
   RowBox[{
    RowBox[{"Cos", "[", 
     RowBox[{"n_.", 
      SubscriptBox["a", "i_"]}], "]"}], ":>", 
    RowBox[{"ChebyshevT", "[", 
     RowBox[{"n", ",", 
      SubscriptBox["x", "i"]}], "]"}]}]}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1866],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzVWttuEzEQ3e5mA0g09E4pqtIWpSlQ2i9A8ASiBf4hKkjlAYGAn+PywpeZ
eONpvGeP1/ZmE9pKqWPPfTw7a8/k3ejH5cfPox+fLkZ7r7+Nvl5+uvi+9+rL
t/FStpQk6cMkSQbbyfi7Gn8rPvK3O/5Mp1vjTzoG56XVbT3LrqiStIC+Gf/P
x4vHZjw0o17XyCdmHJrxzPA+NeNjM+5S7n3DbQjQE5gfwbxvSa2jQ/ie0eZJ
pLwdqv2+R1qIFXVwnzUo/zBS/kNq1SqsdozPUjNmAL8F89zsTceMuYe+pyYx
kMIYK7dr5MmYeehXIu3sOuTN206X3HnZGbufHcBfDrQT6WLtRPo1Sz9/VBdY
nem8yg/9UMBzO6smRpvqqp8725WK1g34st2r8kFfZ9H6+jVBGVyTUDvtKNUY
LEs18R/ybct/Pn2rmmDcoozZ4m9R/rs9J/9JFpgoKn9dyp3aUmiWEz9LPmPr
uVjk1E18JxkpNTGUEytRL5Z5GR/3k9NNpvlYqLq23sROWwvZU41/J9hO9FTH
cKAZM8BeF78wu13UofaX/NXIfvFg0/128Wtm/xqlQqylYn2txkepxIQFD/cR
3ibQF+xtoenuEY1Cs57weOCwypdhw5+Aptb1FmgdPo03Y+/06XG2PI8x3anY
zXZBTpgdkVaiuO/xAMtBfpkdsaF6Diit0P0oYSyTHZm3RSizXYtu2h4VT2Vg
hK6TVftmswlwdiO08bc8lmOe88mz8cUKG59Viv6n/hs3XP9G/g+MtXsUt+/R
eNlIornUe0vAzEDvjpZ9iN+7BjrTd9k11znWz/yk6I6j3FgkOTEjObOnpqcR
GTOQnRiLMwuP+bNNeasLluezb13VVavOYfUU5lgJPvPgY2XYh4/8eaX4Lawe
G1ufeqQJHlbhXXjIj1fjbWq9A3XVbw334Q8j8ZH/jrK7LLijz9Rk/4/M6IIP
PXChL3oBVw/yLpnJW0RWi/NS5d1WD/2pJk/JQJW7QLaWqdFKjy8pl0eGetNQ
ST8Ke0T3Fe8O4bxvaVGHt6943CEe3rD2qRUDmONbHE8ZEg8YvSHWueKM4R/A
HJ99xBe9DqiVtw20fB9jlZ6YHkbb+PoEyLLpLcV7Fq7eTBaIz2rbs9B3A+li
9WT7NAt9W3r67mY+f4b2fvYUi+lqnYdWreO7GzNSsBtqxp89empLnVC/dvz8
2rQLhbZiHLXFl1eY5m2tr+fB7nvz66X442i5VWtxL5vu7awdHh3tkzdzChpo
ztJ5Rs2lTl/XORC4q1Pigm9QXV2/p6iTjta466XY9QnbDRc11vvtzpbLE+G6
hvZN6nV2cVms7q6eV5zu/AZenNDoWysjduJesmwU0v2o82/oW8gV6zJinaPe
W3j3jvM1UrviA+3FKkq9FKwTxcUyUs9Hx9n8GNORcUdu7HmLdRdiMXaojj01
/V0R6UpU8NGa6m8VUBPmX96zcPVt2tMN957FJ9eNUV4Xv/FuS030Wflfngj7
PrHumXN5rF9UqdPXdlyxE4CdBDzZYOcBaw6L1wDxF68Bjz5/LOivo6B97VMs
njXvKv6mYzdke45vgkXIxLfIImTGnHik9vRBTTKP9pEOpb4q/yIVfSdPt2Ss
uxZcS1tpic9qS3ykenVEvfAb5lj/e+OBY1Ud4c+p1D8wHxgb3jnW3zvWXxbc
5RzzCyyv62Mw+LkH/qJY/wurh4rX0GUd+xiyPuGWLP0D9ywBOg==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{539, 37},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->961039723]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Solve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Solve"]], "InlineFormula"],
 " is able to solve the equations in the variables ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox[
    StyleBox["x", "TI"], 
    StyleBox["1", "TR"]], ",", 
   SubscriptBox[
    StyleBox["x", "TI"], 
    StyleBox["2", "TR"]], ",", 
   SubscriptBox[
    StyleBox["x", "TI"], 
    StyleBox["3", "TR"]]}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->13171],

Cell[BoxData[
 RowBox[{
  RowBox[{"SetOptions", "[", 
   RowBox[{"Roots", ",", 
    RowBox[{"Cubics", "->", "False"}], ",", 
    RowBox[{"Quartics", "->", "False"}]}], "]"}], ";"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->3477],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"r1", "=", 
    RowBox[{"Solve", "[", 
     RowBox[{"eqns", ",", 
      RowBox[{"{", 
       RowBox[{
        SubscriptBox["x", "1"], ",", 
        SubscriptBox["x", "2"], ",", 
        SubscriptBox["x", "3"]}], "}"}]}], "]"}]}], ";"}], "//", 
  "Timing"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->31245],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"4.39`", ",", "Null"}], "}"}]], "Output",
 ImageSize->{86, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->3267357]
}, Open  ]],

Cell["The leaf count of the solution is enormous:", "ExampleText",
 CellID->6238],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LeafCount", "[", "r1", "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->27671],

Cell[BoxData["816175"], "Output",
 ImageSize->{49, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->500996804]
}, Open  ]],

Cell[TextData[{
 "Convert to a system of equations of symmetric polynomials ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox[
    StyleBox["s", "TI"], 
    StyleBox["1", "TR"]], ",", 
   SubscriptBox[
    StyleBox["s", "TI"], 
    StyleBox["2", "TR"]], ",", 
   SubscriptBox[
    StyleBox["s", "TI"], 
    StyleBox["3", "TR"]]}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->26115],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"symeqns", "=", 
  RowBox[{"Map", "[", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"SymmetricReduction", "[", 
       RowBox[{"#", ",", 
        RowBox[{"{", 
         RowBox[{
          SubscriptBox["x", "1"], ",", 
          SubscriptBox["x", "2"], ",", 
          SubscriptBox["x", "3"]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          SubscriptBox["e", "1"], ",", 
          SubscriptBox["e", "2"], ",", 
          SubscriptBox["e", "3"]}], "}"}]}], "]"}], "[", 
      RowBox[{"[", "1", "]"}], "]"}], "&"}], ",", "eqns", ",", 
    RowBox[{"{", "2", "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[6]:=",
 CellID->4592],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"-", "k"}], "+", 
     SubscriptBox["e", "1"]}], "\[Equal]", "0"}], ",", 
   RowBox[{
    RowBox[{
     RowBox[{"5", " ", 
      SubscriptBox["e", "1"]}], "-", 
     RowBox[{"20", " ", 
      SubsuperscriptBox["e", "1", "3"]}], "+", 
     RowBox[{"16", " ", 
      SubsuperscriptBox["e", "1", "5"]}], "+", 
     RowBox[{"60", " ", 
      SubscriptBox["e", "1"], " ", 
      SubscriptBox["e", "2"]}], "-", 
     RowBox[{"80", " ", 
      SubsuperscriptBox["e", "1", "3"], " ", 
      SubscriptBox["e", "2"]}], "+", 
     RowBox[{"80", " ", 
      SubscriptBox["e", "1"], " ", 
      SubsuperscriptBox["e", "2", "2"]}], "-", 
     RowBox[{"60", " ", 
      SubscriptBox["e", "3"]}], "+", 
     RowBox[{"80", " ", 
      SubsuperscriptBox["e", "1", "2"], " ", 
      SubscriptBox["e", "3"]}], "-", 
     RowBox[{"80", " ", 
      SubscriptBox["e", "2"], " ", 
      SubscriptBox["e", "3"]}]}], "\[Equal]", "0"}], ",", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"-", "7"}], " ", 
      SubscriptBox["e", "1"]}], "+", 
     RowBox[{"56", " ", 
      SubsuperscriptBox["e", "1", "3"]}], "-", 
     RowBox[{"112", " ", 
      SubsuperscriptBox["e", "1", "5"]}], "+", 
     RowBox[{"64", " ", 
      SubsuperscriptBox["e", "1", "7"]}], "-", 
     RowBox[{"168", " ", 
      SubscriptBox["e", "1"], " ", 
      SubscriptBox["e", "2"]}], "+", 
     RowBox[{"560", " ", 
      SubsuperscriptBox["e", "1", "3"], " ", 
      SubscriptBox["e", "2"]}], "-", 
     RowBox[{"448", " ", 
      SubsuperscriptBox["e", "1", "5"], " ", 
      SubscriptBox["e", "2"]}], "-", 
     RowBox[{"560", " ", 
      SubscriptBox["e", "1"], " ", 
      SubsuperscriptBox["e", "2", "2"]}], "+", 
     RowBox[{"896", " ", 
      SubsuperscriptBox["e", "1", "3"], " ", 
      SubsuperscriptBox["e", "2", "2"]}], "-", 
     RowBox[{"448", " ", 
      SubscriptBox["e", "1"], " ", 
      SubsuperscriptBox["e", "2", "3"]}], "+", 
     RowBox[{"168", " ", 
      SubscriptBox["e", "3"]}], "-", 
     RowBox[{"560", " ", 
      SubsuperscriptBox["e", "1", "2"], " ", 
      SubscriptBox["e", "3"]}], "+", 
     RowBox[{"448", " ", 
      SubsuperscriptBox["e", "1", "4"], " ", 
      SubscriptBox["e", "3"]}], "+", 
     RowBox[{"560", " ", 
      SubscriptBox["e", "2"], " ", 
      SubscriptBox["e", "3"]}], "-", 
     RowBox[{"1344", " ", 
      SubsuperscriptBox["e", "1", "2"], " ", 
      SubscriptBox["e", "2"], " ", 
      SubscriptBox["e", "3"]}], "+", 
     RowBox[{"448", " ", 
      SubsuperscriptBox["e", "2", "2"], " ", 
      SubscriptBox["e", "3"]}], "+", 
     RowBox[{"448", " ", 
      SubscriptBox["e", "1"], " ", 
      SubsuperscriptBox["e", "3", "2"]}]}], "\[Equal]", "0"}]}], 
  "}"}]], "Output",
 ImageSize->{573, 54},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->184682544]
}, Open  ]],

Cell["Solve the new system of equations:", "ExampleText",
 CellID->32657],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"r2", "=", 
    RowBox[{"Solve", "[", 
     RowBox[{"symeqns", ",", 
      RowBox[{"{", 
       RowBox[{
        SubscriptBox["e", "1"], ",", 
        SubscriptBox["e", "2"], ",", 
        SubscriptBox["e", "3"]}], "}"}]}], "]"}]}], ";"}], "//", 
  "Timing"}]], "Input",
 CellLabel->"In[7]:=",
 CellID->1405],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.06199999999999972`", ",", "Null"}], "}"}]], "Output",
 ImageSize->{93, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->485878766]
}, Open  ]],

Cell["The leaf count of the symmetric solution is much smaller:", \
"ExampleText",
 CellID->7171],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LeafCount", "[", "r2", "]"}]], "Input",
 CellLabel->"In[8]:=",
 CellID->20580],

Cell[BoxData["2533"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->488016089]
}, Open  ]],

Cell[TextData[{
 "Solving for the variables ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["x", "1"], ",", 
    SubscriptBox["x", "2"], ",", 
    SubscriptBox["x", "3"]}], TraditionalForm]], "InlineMath"],
 " in terms of the symmetric polynomials ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["s", "1"], ",", 
    SubscriptBox["s", "2"], ",", 
    SubscriptBox["s", "3"]}], TraditionalForm]], "InlineMath"],
 " is also quick:"
}], "ExampleText",
 CellID->22530],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"Solve", "[", 
    RowBox[{
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"SymmetricPolynomial", "[", 
         RowBox[{"k", ",", 
          RowBox[{"{", 
           RowBox[{
            SubscriptBox["x", "1"], ",", 
            SubscriptBox["x", "2"], ",", 
            SubscriptBox["x", "3"]}], "}"}]}], "]"}], "==", 
        SubscriptBox["e", "k"]}], ",", 
       RowBox[{"{", 
        RowBox[{"k", ",", "3"}], "}"}]}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{
       SubscriptBox["x", "1"], ",", 
       SubscriptBox["x", "2"], ",", 
       SubscriptBox["x", "3"]}], "}"}]}], "]"}], ";"}], "//", 
  "Timing"}]], "Input",
 CellLabel->"In[9]:=",
 CellID->13675],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.016000000000000333`", ",", "Null"}], "}"}]], "Output",
 ImageSize->{93, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[9]=",
 CellID->693529561]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->25790],

Cell["\<\
The order of variables can effect the decomposition into symmetric and \
nonsymmetric parts:\
\>", "ExampleText",
 CellID->470406655],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SymmetricReduction", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"x", "^", "2"}], "-", 
    RowBox[{"y", "^", "2"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->35468958],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"-", "2"}], " ", "x", " ", "y"}], "+", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"x", "+", "y"}], ")"}], "2"]}], ",", 
   RowBox[{
    RowBox[{"-", "2"}], " ", 
    SuperscriptBox["y", "2"]}]}], "}"}]], "Output",
 ImageSize->{153, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->122791480]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SymmetricReduction", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"x", "^", "2"}], "-", 
    RowBox[{"y", "^", "2"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "x"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->138114613],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"2", " ", "x", " ", "y"}], "-", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"x", "+", "y"}], ")"}], "2"]}], ",", 
   RowBox[{"2", " ", 
    SuperscriptBox["x", "2"]}]}], "}"}]], "Output",
 ImageSize->{138, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->238979362]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->136422718],

Cell[TextData[{
 "Another basis for the symmetric polynomials consists of the complete \
symmetric polynomials. They are the sum of all monomials of a given degree, \
and can be defined by the generating function ",
 Cell[BoxData[
  SuperscriptBox[
   RowBox[{
    ButtonBox["Product",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Product"], "[", 
    RowBox[{
     RowBox[{"1", "-", 
      RowBox[{
       SubscriptBox[
        StyleBox["x", "TI"], 
        StyleBox["i", "TI"]], 
       StyleBox["t", "TI"]}]}], ",", 
     RowBox[{"{", 
      RowBox[{
       StyleBox["i", "TI"], ",", 
       StyleBox["n", "TI"]}], "}"}]}], "]"}], 
   RowBox[{"-", "1"}]]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->13377],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"complete", "=", 
  RowBox[{
   RowBox[{"Solve", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"1", "+", 
       RowBox[{
        SubscriptBox["h", "1"], "t"}], "+", 
       RowBox[{
        SubscriptBox["h", "2"], 
        SuperscriptBox["t", "2"]}], "+", 
       RowBox[{
        SubscriptBox["h", "3"], 
        SuperscriptBox["t", "3"]}], "+", 
       SuperscriptBox[
        RowBox[{"O", "[", "t", "]"}], "4"]}], "==", 
      RowBox[{
       FractionBox["1", 
        RowBox[{
         UnderoverscriptBox["\[Product]", 
          RowBox[{"i", "=", "1"}], "3"], 
         RowBox[{"(", 
          RowBox[{"1", "-", 
           RowBox[{
            SubscriptBox["x", "i"], "t"}]}], ")"}]}]], "+", 
       SuperscriptBox[
        RowBox[{"O", "[", "t", "]"}], "4"]}]}], ",", 
     RowBox[{"{", 
      RowBox[{
       SubscriptBox["h", "1"], ",", 
       SubscriptBox["h", "2"], ",", 
       SubscriptBox["h", "3"]}], "}"}]}], "]"}], "[", 
   RowBox[{"[", "1", "]"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->10998],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzVWktvEzEQ3iS7eTSPCpE2AS4RgiLxuJErvcGJlwQcoDlEASk9IFDbXwEc
gAP9uWa9a3en39o7a2e3gUhb7zw8nhnbs/anPl+erT99Xp4dr5azZyfLr+vj
1ens6ZeTmNVqBEHjTxAEd6dB/C7it+SRv/P4SV9/xk8zZkcXnN/yraU1RdBM
JPfjv1HMHKv2cdxKpX3SSv6RoufK7j3VTlTbNlrfVdJbIN0HeqCs76nR+oz+
FOi+Gkd7q+0NiJdUfwj6Q5Dvqf4jxo+WMeprjLem6CmN3qIc+5uiw2iK7GGU
qM/JOf/QH5SHxix21BykYv1rgu6OYUaaSk+2OEO4TkNGn5O72uP862yov+n4
Vefnf5sPXItV24sYe6hvrjDdxNQFV6QbBPua9kYQZjQfbcT24DXcbYY5DZoF
GWl34x4+XuDcGekNs1XFqBh7u4ZIribjaBOrHR87Rla0f5qGvj3V6r2pzzqR
aim/K/J7WNuh+qZs2/gRGd8mRz86nn60Lf1sfF97m+aD89NUY7nxKN0oWCNY
j/GkqucAz0xtwrflypY7Oqe7llwhX/cfeo7XtfSzzY3NXtXxcmvGlnc8o2Z1
QH99i068+cqEZ9p8JRp49OnnOLwnqFHGqo/31VipIgfVRIg7l3wZwqymZDdc
fUK7DvQOI+8x8rGjPo7nqs/55+OPJY+EOxPZTVyy5yLDFyR/qugFyA9Az4wz
6GpxB6RjoOdA480cb7YLpv8BY898en8IXLwP4yjcfd/Vy3L370dMbnAU9ApH
cfUS7Zlz+YBwA7VOKD1n5BOgF0z/A8aevv9F/+RM9xKv5Os5vOnvruT8SjiY
b7vkm0jPV09EulfXytY7o3ZXaU9Uq7E2vcPxyz5S8pvAp1hhC2aCyikG2CJ0
Yi93vntt9Blr3A3DWEV1QNdKPM1gTDjf+I3i9hh+KbH+cXim9vNVloWQZkiq
bheP88GL8BxQfRTbRuFcUcNto5oc6oXz8UKYdmWi5XxDqwMF43tgVdMZ69QU
A4+U8OiLe5TuOB2PkNWBVXG3UT4yvYJfJjSu10SbnVVpg+JLHG7B4SgoHxm9
KItydUQ5lMYH3SlCtXyjLeIX+fNGlK8tul7h7TIUl3OH674IISujNyy1nmxY
lA1DQaylrHc2DIrDjspGa1sZWNN0/yPrDNaF7fAa9BbtNno9aAc/7lVhNR+s
s+WGu5joYguuyEbVSMp7e+SEXgE9E+mJ9VBktyvZvs2sXTqjy5E+qvY22DoE
eg00sUm4PxytmCuqnxVdU74TaaCyQek10Gkk+D8v5cYMGn8BWASrTQ==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{406, 37},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->750969559]
}, Open  ]],

Cell["\<\
A determinant formula expresses the elementary symmetric polynomials in the \
basis of the complete symmetric polynomials:\
\>", "ExampleText",
 CellID->21604],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{
    SubscriptBox["e", "1"], ",", 
    SubscriptBox["e", "2"], ",", 
    SubscriptBox["e", "3"]}], "}"}], "==", 
  RowBox[{"{", 
   RowBox[{
    SubscriptBox["h", "1"], ",", 
    RowBox[{"Det", "[", GridBox[{
       {
        SubscriptBox["h", "1"], 
        SubscriptBox["h", "2"]},
       {"1", 
        SubscriptBox["h", "1"]}
      }], "]"}], ",", 
    RowBox[{"Det", "[", GridBox[{
       {
        SubscriptBox["h", "1"], 
        SubscriptBox["h", "2"], 
        SubscriptBox["h", "3"]},
       {"1", 
        SubscriptBox["h", "1"], 
        SubscriptBox["h", "2"]},
       {"0", "1", 
        SubscriptBox["h", "1"]}
      }], "]"}]}], "}"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->4489],

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{
    SubscriptBox["e", "1"], ",", 
    SubscriptBox["e", "2"], ",", 
    SubscriptBox["e", "3"]}], "}"}], "\[Equal]", 
  RowBox[{"{", 
   RowBox[{
    SubscriptBox["h", "1"], ",", 
    RowBox[{
     SubsuperscriptBox["h", "1", "2"], "-", 
     SubscriptBox["h", "2"]}], ",", 
    RowBox[{
     SubsuperscriptBox["h", "1", "3"], "-", 
     RowBox[{"2", " ", 
      SubscriptBox["h", "1"], " ", 
      SubscriptBox["h", "2"]}], "+", 
     SubscriptBox["h", "3"]}]}], "}"}]}]], "Output",
 ImageSize->{270, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->284558397]
}, Open  ]],

Cell["Check:", "ExampleText",
 CellID->29744],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"%", "[", 
    RowBox[{"[", "2", "]"}], "]"}], "/.", "complete"}], "//", 
  "Expand"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->8388],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    SubscriptBox["x", "1"], "+", 
    SubscriptBox["x", "2"], "+", 
    SubscriptBox["x", "3"]}], ",", 
   RowBox[{
    RowBox[{
     SubscriptBox["x", "1"], " ", 
     SubscriptBox["x", "2"]}], "+", 
    RowBox[{
     SubscriptBox["x", "1"], " ", 
     SubscriptBox["x", "3"]}], "+", 
    RowBox[{
     SubscriptBox["x", "2"], " ", 
     SubscriptBox["x", "3"]}]}], ",", 
   RowBox[{
    SubscriptBox["x", "1"], " ", 
    SubscriptBox["x", "2"], " ", 
    SubscriptBox["x", "3"]}]}], "}"}]], "Output",
 ImageSize->{256, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->100161202]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"SymmetricPolynomial", "[", 
    RowBox[{"#", ",", 
     RowBox[{"{", 
      RowBox[{
       SubscriptBox["x", "1"], ",", 
       SubscriptBox["x", "2"], ",", 
       SubscriptBox["x", "3"]}], "}"}]}], "]"}], "&"}], "/@", 
  RowBox[{"Range", "[", "3", "]"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->18718],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    SubscriptBox["x", "1"], "+", 
    SubscriptBox["x", "2"], "+", 
    SubscriptBox["x", "3"]}], ",", 
   RowBox[{
    RowBox[{
     SubscriptBox["x", "1"], " ", 
     SubscriptBox["x", "2"]}], "+", 
    RowBox[{
     SubscriptBox["x", "1"], " ", 
     SubscriptBox["x", "3"]}], "+", 
    RowBox[{
     SubscriptBox["x", "2"], " ", 
     SubscriptBox["x", "3"]}]}], ",", 
   RowBox[{
    SubscriptBox["x", "1"], " ", 
    SubscriptBox["x", "2"], " ", 
    SubscriptBox["x", "3"]}]}], "}"}]], "Output",
 ImageSize->{256, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->75209739]
}, Open  ]],

Cell["\<\
Any symmetric polynomial can also be expressed in terms of the complete \
symmetric polynomials:\
\>", "ExampleText",
 CellID->18259],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"SymmetricReduction", "[", 
    RowBox[{
     RowBox[{"x", " ", "y", " ", "z", 
      RowBox[{"(", 
       RowBox[{"x", "+", "y"}], ")"}], 
      RowBox[{"(", 
       RowBox[{"x", "+", "z"}], ")"}], 
      RowBox[{"(", 
       RowBox[{"y", "+", "z"}], ")"}]}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "y", ",", "z"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       SubscriptBox["e", "1"], ",", 
       SubscriptBox["e", "2"], ",", 
       SubscriptBox["e", "3"]}], "}"}]}], "]"}], "/.", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      SubscriptBox["e", "1"], "->", 
      SubscriptBox["h", "1"]}], ",", 
     RowBox[{
      SubscriptBox["e", "2"], "->", 
      RowBox[{"Det", "[", GridBox[{
         {
          SubscriptBox["h", "1"], 
          SubscriptBox["h", "2"]},
         {"1", 
          SubscriptBox["h", "1"]}
        }], "]"}]}], ",", 
     RowBox[{
      SubscriptBox["e", "3"], "->", 
      RowBox[{"Det", "[", GridBox[{
         {
          SubscriptBox["h", "1"], 
          SubscriptBox["h", "2"], 
          SubscriptBox["h", "3"]},
         {"1", 
          SubscriptBox["h", "1"], 
          SubscriptBox["h", "2"]},
         {"0", "1", 
          SubscriptBox["h", "1"]}
        }], "]"}]}]}], "}"}]}], "//", "Expand"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->5927],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{
     SubsuperscriptBox["h", "1", "4"], " ", 
     SubscriptBox["h", "2"]}], "-", 
    RowBox[{"2", " ", 
     SubsuperscriptBox["h", "1", "2"], " ", 
     SubsuperscriptBox["h", "2", "2"]}], "-", 
    RowBox[{
     SubsuperscriptBox["h", "1", "3"], " ", 
     SubscriptBox["h", "3"]}], "+", 
    RowBox[{"3", " ", 
     SubscriptBox["h", "1"], " ", 
     SubscriptBox["h", "2"], " ", 
     SubscriptBox["h", "3"]}], "-", 
    SubsuperscriptBox["h", "3", "2"]}], ",", "0"}], "}"}]], "Output",
 ImageSize->{247, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->429561899]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->361],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["SymmetricPolynomial",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SymmetricPolynomial"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["PolynomialReduce",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/PolynomialReduce"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->11385]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->6379],

Cell[TextData[ButtonBox["Symmetric Polynomials",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/SymmetricPolynomials"]], "Tutorials",
 CellID->15275]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->573745420],

Cell[TextData[ButtonBox["Polynomial Algebra",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/PolynomialAlgebra"]], "MoreAbout",
 CellID->38988161],

Cell[TextData[ButtonBox["Polynomial Factoring & Decomposition",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/PolynomialFactoring"]], "MoreAbout",
 CellID->57437975],

Cell[TextData[ButtonBox["New in 6.0: Symbolic Computation",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60AlgebraicComputing"]], "MoreAbout",
 CellID->179911398],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->851466778]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"SymmetricReduction - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 25, 38.0651780}", "context" -> "System`", 
    "keywords" -> {}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "SymmetricReduction[f, {x_1, ..., x_n}] gives a pair of polynomials {p, \
q} in x_1, ..., x_n such that f == p + q, where p is the symmetric part and q \
is the remainder. SymmetricReduction[f, {x_1, ..., x_n}, {s_1, ..., s_n}] \
gives the pair {p, q} with the elementary symmetric polynomials in p replaced \
by s_1, ..., s_n.", "synonyms" -> {"symmetric reduction"}, "title" -> 
    "SymmetricReduction", "type" -> "Symbol", "uri" -> 
    "ref/SymmetricReduction"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[10412, 322, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->288700245]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 56304, 2012}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2479, 59, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3102, 86, 1583, 37, 70, "ObjectNameGrid"],
Cell[4688, 125, 2843, 89, 70, "Usage",
 CellID->23775]
}, Open  ]],
Cell[CellGroupData[{
Cell[7568, 219, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[8046, 236, 446, 15, 70, "Notes",
 CellID->13000],
Cell[8495, 253, 317, 10, 70, "Notes",
 CellID->12212],
Cell[8815, 265, 895, 27, 70, "Notes",
 CellID->21100],
Cell[9713, 294, 233, 8, 70, "Notes",
 CellID->7551],
Cell[9949, 304, 426, 13, 70, "Notes",
 CellID->25476]
}, Closed]],
Cell[CellGroupData[{
Cell[10412, 322, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->288700245],
Cell[CellGroupData[{
Cell[10797, 336, 148, 5, 70, "ExampleSection",
 CellID->748647321],
Cell[10948, 343, 125, 3, 70, "ExampleText",
 CellID->958837676],
Cell[CellGroupData[{
Cell[11098, 350, 260, 9, 28, "Input",
 CellID->462908957],
Cell[11361, 361, 362, 13, 39, "Output",
 CellID->29256572]
}, Open  ]],
Cell[11738, 377, 124, 3, 70, "ExampleDelimiter",
 CellID->20448923],
Cell[11865, 382, 118, 3, 70, "ExampleText",
 CellID->57567550],
Cell[CellGroupData[{
Cell[12008, 389, 260, 9, 70, "Input",
 CellID->752424637],
Cell[12271, 400, 432, 16, 39, "Output",
 CellID->35756098]
}, Open  ]],
Cell[12718, 419, 124, 3, 70, "ExampleDelimiter",
 CellID->53173094],
Cell[12845, 424, 204, 7, 70, "ExampleText",
 CellID->118134771],
Cell[CellGroupData[{
Cell[13074, 435, 320, 11, 70, "Input",
 CellID->49828701],
Cell[13397, 448, 350, 13, 39, "Output",
 CellID->16405196]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[13796, 467, 215, 7, 70, "ExampleSection",
 CellID->6461],
Cell[CellGroupData[{
Cell[14036, 478, 262, 9, 70, "Input",
 CellID->230645080],
Cell[14301, 489, 1203, 40, 58, "Output",
 CellID->832974987]
}, Open  ]],
Cell[CellGroupData[{
Cell[15541, 534, 323, 11, 70, "Input",
 CellID->629681612],
Cell[15867, 547, 761, 25, 39, "Output",
 CellID->379122588]
}, Open  ]],
Cell[16643, 575, 125, 3, 70, "ExampleDelimiter",
 CellID->123114972],
Cell[16771, 580, 243, 7, 70, "ExampleText",
 CellID->16991],
Cell[CellGroupData[{
Cell[17039, 591, 462, 15, 70, "Input",
 CellID->31461],
Cell[17504, 608, 473, 17, 36, "Output",
 CellID->381886287]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[18026, 631, 223, 7, 70, "ExampleSection",
 CellID->14179],
Cell[18252, 640, 1023, 40, 70, "ExampleText",
 CellID->24117],
Cell[19278, 682, 256, 9, 70, "Input",
 CellID->2317],
Cell[CellGroupData[{
Cell[19559, 695, 359, 12, 70, "Input",
 CellID->19914],
Cell[19921, 709, 646, 20, 36, "Output",
 CellID->94316343]
}, Open  ]],
Cell[CellGroupData[{
Cell[20604, 734, 297, 10, 70, "Input",
 CellID->1441],
Cell[20904, 746, 471, 13, 36, "Output",
 CellID->583222659]
}, Open  ]],
Cell[21390, 762, 424, 15, 70, "ExampleText",
 CellID->21867],
Cell[CellGroupData[{
Cell[21839, 781, 666, 24, 70, "Input",
 CellID->7028],
Cell[22508, 807, 1065, 22, 70, "Output",
 Evaluatable->False,
 CellID->21971802]
}, Open  ]],
Cell[23588, 832, 479, 20, 70, "ExampleText",
 CellID->27909],
Cell[CellGroupData[{
Cell[24092, 856, 1370, 43, 70, "Input",
 CellID->13973],
Cell[25465, 901, 440, 16, 39, "Output",
 CellID->568762279]
}, Open  ]],
Cell[25920, 920, 393, 15, 70, "ExampleText",
 CellID->11503],
Cell[CellGroupData[{
Cell[26338, 939, 298, 11, 70, "Input",
 CellID->26569],
Cell[26639, 952, 575, 21, 39, "Output",
 CellID->232487381]
}, Open  ]],
Cell[27229, 976, 44, 1, 70, "ExampleText",
 CellID->1883],
Cell[CellGroupData[{
Cell[27298, 981, 389, 14, 70, "Input",
 CellID->25367],
Cell[27690, 997, 469, 14, 36, "Output",
 CellID->147536678]
}, Open  ]],
Cell[CellGroupData[{
Cell[28196, 1016, 364, 13, 70, "Input",
 CellID->28402],
Cell[28563, 1031, 466, 14, 36, "Output",
 CellID->33062638]
}, Open  ]],
Cell[29044, 1048, 121, 3, 70, "ExampleDelimiter",
 CellID->17157],
Cell[29168, 1053, 108, 3, 70, "ExampleText",
 CellID->28681],
Cell[29279, 1058, 1198, 39, 70, "Input",
 CellID->15567],
Cell[30480, 1099, 234, 8, 70, "ExampleText",
 CellID->8310],
Cell[CellGroupData[{
Cell[30739, 1111, 317, 11, 70, "Input",
 CellID->1866],
Cell[31059, 1124, 1972, 36, 70, "Output",
 Evaluatable->False,
 CellID->961039723]
}, Open  ]],
Cell[33046, 1163, 488, 19, 70, "ExampleText",
 CellID->13171],
Cell[33537, 1184, 229, 7, 70, "Input",
 CellID->3477],
Cell[CellGroupData[{
Cell[33791, 1195, 352, 13, 70, "Input",
 CellID->31245],
Cell[34146, 1210, 208, 7, 36, "Output",
 CellID->3267357]
}, Open  ]],
Cell[34369, 1220, 81, 1, 70, "ExampleText",
 CellID->6238],
Cell[CellGroupData[{
Cell[34475, 1225, 102, 3, 70, "Input",
 CellID->27671],
Cell[34580, 1230, 163, 5, 36, "Output",
 CellID->500996804]
}, Open  ]],
Cell[34758, 1238, 384, 15, 70, "ExampleText",
 CellID->26115],
Cell[CellGroupData[{
Cell[35167, 1257, 668, 21, 70, "Input",
 CellID->4592],
Cell[35838, 1280, 2902, 87, 75, "Output",
 CellID->184682544]
}, Open  ]],
Cell[38755, 1370, 73, 1, 70, "ExampleText",
 CellID->32657],
Cell[CellGroupData[{
Cell[38853, 1375, 354, 13, 70, "Input",
 CellID->1405],
Cell[39210, 1390, 225, 7, 36, "Output",
 CellID->485878766]
}, Open  ]],
Cell[39450, 1400, 97, 2, 70, "ExampleText",
 CellID->7171],
Cell[CellGroupData[{
Cell[39572, 1406, 102, 3, 70, "Input",
 CellID->20580],
Cell[39677, 1411, 161, 5, 36, "Output",
 CellID->488016089]
}, Open  ]],
Cell[39853, 1419, 484, 17, 70, "ExampleText",
 CellID->22530],
Cell[CellGroupData[{
Cell[40362, 1440, 744, 25, 70, "Input",
 CellID->13675],
Cell[41109, 1467, 226, 7, 36, "Output",
 CellID->693529561]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[41384, 1480, 233, 7, 70, "ExampleSection",
 CellID->25790],
Cell[41620, 1489, 143, 4, 70, "ExampleText",
 CellID->470406655],
Cell[CellGroupData[{
Cell[41788, 1497, 259, 9, 70, "Input",
 CellID->35468958],
Cell[42050, 1508, 433, 16, 39, "Output",
 CellID->122791480]
}, Open  ]],
Cell[CellGroupData[{
Cell[42520, 1529, 260, 9, 70, "Input",
 CellID->138114613],
Cell[42783, 1540, 392, 14, 39, "Output",
 CellID->238979362]
}, Open  ]],
Cell[43190, 1557, 125, 3, 70, "ExampleDelimiter",
 CellID->136422718],
Cell[43318, 1562, 725, 24, 70, "ExampleText",
 CellID->13377],
Cell[CellGroupData[{
Cell[44068, 1590, 1049, 35, 70, "Input",
 CellID->10998],
Cell[45120, 1627, 1549, 29, 70, "Output",
 Evaluatable->False,
 CellID->750969559]
}, Open  ]],
Cell[46684, 1659, 169, 4, 70, "ExampleText",
 CellID->21604],
Cell[CellGroupData[{
Cell[46878, 1667, 752, 29, 70, "Input",
 CellID->4489],
Cell[47633, 1698, 666, 23, 39, "Output",
 CellID->284558397]
}, Open  ]],
Cell[48314, 1724, 45, 1, 70, "ExampleText",
 CellID->29744],
Cell[CellGroupData[{
Cell[48384, 1729, 181, 7, 70, "Input",
 CellID->8388],
Cell[48568, 1738, 685, 25, 36, "Output",
 CellID->100161202]
}, Open  ]],
Cell[CellGroupData[{
Cell[49290, 1768, 355, 12, 70, "Input",
 CellID->18718],
Cell[49648, 1782, 684, 25, 36, "Output",
 CellID->75209739]
}, Open  ]],
Cell[50347, 1810, 143, 4, 70, "ExampleText",
 CellID->18259],
Cell[CellGroupData[{
Cell[50515, 1818, 1355, 47, 70, "Input",
 CellID->5927],
Cell[51873, 1867, 687, 22, 39, "Output",
 CellID->429561899]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[52621, 1896, 309, 9, 70, "SeeAlsoSection",
 CellID->361],
Cell[52933, 1907, 489, 17, 70, "SeeAlso",
 CellID->11385]
}, Open  ]],
Cell[CellGroupData[{
Cell[53459, 1929, 313, 9, 70, "TutorialsSection",
 CellID->6379],
Cell[53775, 1940, 152, 3, 70, "Tutorials",
 CellID->15275]
}, Open  ]],
Cell[CellGroupData[{
Cell[53964, 1948, 319, 9, 70, "MoreAboutSection",
 CellID->573745420],
Cell[54286, 1959, 146, 3, 70, "MoreAbout",
 CellID->38988161],
Cell[54435, 1964, 166, 3, 70, "MoreAbout",
 CellID->57437975],
Cell[54604, 1969, 169, 3, 70, "MoreAbout",
 CellID->179911398],
Cell[54776, 1974, 179, 3, 70, "MoreAbout",
 CellID->851466778]
}, Open  ]],
Cell[54970, 1980, 27, 0, 70, "History"],
Cell[55000, 1982, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

