(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     90103,       2187]
NotebookOptionsPosition[     83280,       1952]
NotebookOutlinePosition[     84729,       1993]
CellTagsIndexPosition[     84643,       1988]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"AffineTransform" :> 
          Documentation`HelpLookup["paclet:ref/AffineTransform"], 
          "RotationTransform" :> 
          Documentation`HelpLookup["paclet:ref/RotationTransform"], 
          "GeometricTransformation" :> 
          Documentation`HelpLookup["paclet:ref/GeometricTransformation"], 
          "TransformationFunction" :> 
          Documentation`HelpLookup["paclet:ref/TransformationFunction"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"AffineTransform\"\>", 
       2->"\<\"RotationTransform\"\>", 3->"\<\"GeometricTransformation\"\>", 
       4->"\<\"TransformationFunction\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Geometric Transforms" :> 
          Documentation`HelpLookup["paclet:guide/GeometricTransforms"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Geometric Transforms\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["TranslationTransform", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["TranslationTransform",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/TranslationTransform"], "[", 
       StyleBox["v", "TI"], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives a ",
     Cell[BoxData[
      ButtonBox["TransformationFunction",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/TransformationFunction"]], "InlineFormula"],
     " that represents translation of points by a vector ",
     Cell[BoxData[
      StyleBox["v", "TI"]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TranslationTransform",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TranslationTransform"]], "InlineFormula"],
 " gives a ",
 Cell[BoxData[
  ButtonBox["TransformationFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TransformationFunction"]], "InlineFormula"],
 " which can be applied to vectors."
}], "Notes",
 CellID->257351590],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["TranslationTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/TranslationTransform"], "[", 
   RowBox[{"{", 
    RowBox[{
     SubscriptBox[
      StyleBox["x", "TI"], 
      StyleBox["1", "TR"]], ",", 
     StyleBox["\[Ellipsis]", "TR"], ",", 
     SubscriptBox[
      StyleBox["x", "TI"], 
      StyleBox["n", "TI"]]}], "}"}], "]"}]], "InlineFormula"],
 " gives a transformation for vectors with dimension ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 "."
}], "Notes",
 CellID->232415095],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   RowBox[{"TranslationTransform", "[", 
    StyleBox["v", "TI"], "]"}], "[", 
   StyleBox["r", "TI"], "]"}]], "InlineFormula"],
 " for vectors ",
 Cell[BoxData[
  StyleBox["v", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["r", "TI"]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  RowBox[{
   StyleBox["r", "TI"], "+", 
   StyleBox["v", "TI"]}]], "InlineFormula"],
 "."
}], "Notes",
 CellID->212580077]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->702574],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(1)", "ExampleCount"]
}], "ExampleSection",
 CellID->282139695],

Cell[TextData[{
 "Generate a function representing a translation by the vector ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    StyleBox["a", "TI"], ",", 
    StyleBox["b", "TI"]}], "}"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->143900862],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TranslationTransform", "[", 
  RowBox[{"{", 
   RowBox[{"a", ",", "b"}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->750626971],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {"1", "0", "a"},
     {"0", "1", "b"},
     {"0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}}], ")"}],
   "]"}]], "Output",
 ImageSize->{230, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->197447562]
}, Open  ]],

Cell["Apply the transformation function to a vector:", "ExampleText",
 CellID->608535536],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"%", "[", 
  RowBox[{"{", 
   RowBox[{"x", ",", "y"}], "}"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->22306003],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"a", "+", "x"}], ",", 
   RowBox[{"b", "+", "y"}]}], "}"}]], "Output",
 ImageSize->{83, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->246473222]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->54466981],

Cell["Translation in four dimensions:", "ExampleText",
 CellID->25456152],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"t", "=", 
  RowBox[{"TranslationTransform", "[", 
   RowBox[{"{", 
    RowBox[{"x0", ",", "y0", ",", "z0", ",", "w0"}], "}"}], "]"}]}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->330826147],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {"1", "0", "0", "0", "x0"},
     {"0", "1", "0", "0", "y0"},
     {"0", "0", "1", "0", "z0"},
     {"0", "0", "0", "1", "w0"},
     {"0", "0", "0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}}], ")"}],
   "]"}]], "Output",
 ImageSize->{267, 73},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->450901510]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"t", "[", 
  RowBox[{"{", 
   RowBox[{"x", ",", "y", ",", "z", ",", "w"}], "}"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->15646037],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"x", "+", "x0"}], ",", 
   RowBox[{"y", "+", "y0"}], ",", 
   RowBox[{"z", "+", "z0"}], ",", 
   RowBox[{"w", "+", "w0"}]}], "}"}]], "Output",
 ImageSize->{187, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->36046072]
}, Open  ]],

Cell["The inverse transform:", "ExampleText",
 CellID->63688802],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"InverseFunction", "[", "t", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->234230738],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {"1", "0", "0", "0", 
      RowBox[{"-", "x0"}]},
     {"0", "1", "0", "0", 
      RowBox[{"-", "y0"}]},
     {"0", "0", "1", "0", 
      RowBox[{"-", "z0"}]},
     {"0", "0", "0", "1", 
      RowBox[{"-", "w0"}]},
     {"0", "0", "0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}}], ")"}],
   "]"}]], "Output",
 ImageSize->{275, 73},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->408114852]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"InverseFunction", "[", "t", "]"}], "[", 
  RowBox[{"{", 
   RowBox[{"x", ",", "y", ",", "z", ",", "w"}], "}"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->17728099],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"x", "-", "x0"}], ",", 
   RowBox[{"y", "-", "y0"}], ",", 
   RowBox[{"z", "-", "z0"}], ",", 
   RowBox[{"w", "-", "w0"}]}], "}"}]], "Output",
 ImageSize->{187, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->84982390]
}, Open  ]],

Cell["Apply the transform 5 times:", "ExampleText",
 CellID->513676756],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Nest", "[", 
  RowBox[{"t", ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y", ",", "z", ",", "w"}], "}"}], ",", "5"}], 
  "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->129763148],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"x", "+", 
    RowBox[{"5", " ", "x0"}]}], ",", 
   RowBox[{"y", "+", 
    RowBox[{"5", " ", "y0"}]}], ",", 
   RowBox[{"z", "+", 
    RowBox[{"5", " ", "z0"}]}], ",", 
   RowBox[{"w", "+", 
    RowBox[{"5", " ", "w0"}]}]}], "}"}]], "Output",
 ImageSize->{227, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->340236158]
}, Open  ]],

Cell["Use matrix operations and homogeneous coordinates:", "ExampleText",
 CellID->812525649],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"MatrixPower", "[", 
   RowBox[{
    RowBox[{"TransformationMatrix", "[", "t", "]"}], ",", "n"}], "]"}], ".", 
  RowBox[{"{", 
   RowBox[{"x", ",", "y", ",", "z", ",", "w", ",", "1"}], "}"}]}]], "Input",
 CellLabel->"In[6]:=",
 CellID->65750619],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"x", "+", 
    RowBox[{"n", " ", "x0"}]}], ",", 
   RowBox[{"y", "+", 
    RowBox[{"n", " ", "y0"}]}], ",", 
   RowBox[{"z", "+", 
    RowBox[{"n", " ", "z0"}]}], ",", 
   RowBox[{"w", "+", 
    RowBox[{"n", " ", "w0"}]}], ",", "1"}], "}"}]], "Output",
 ImageSize->{245, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->233290623]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->153692911],

Cell["Transformation applied to a 2D shape:", "ExampleText",
 CellID->479507293],

Cell[BoxData[
 RowBox[{
  RowBox[{"gr", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Rectangle", "[", "]"}], ",", 
     RowBox[{"AbsolutePointSize", "[", "10", "]"}], ",", 
     RowBox[{"Opacity", "[", "1", "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"Magenta", ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{"0", ",", "0"}], "}"}], "]"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"Green", ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{"1", ",", "1"}], "}"}], "]"}]}], "}"}]}], "}"}]}], 
  ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->419323043],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"Opacity", "[", ".35", "]"}], ",", "Blue", ",", "gr"}], "}"}], 
    ",", 
    RowBox[{"GeometricTransformation", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"Opacity", "[", ".85", "]"}], ",", "Red", ",", "gr"}], "}"}], 
      ",", 
      RowBox[{"TranslationTransform", "[", 
       RowBox[{"{", 
        RowBox[{".75", ",", 
         RowBox[{"-", ".25"}]}], "}"}], "]"}]}], "]"}]}], "}"}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->150504143],

Cell[BoxData[
 GraphicsBox[{
   {RGBColor[0, 0, 1], Opacity[0.35], RectangleBox[{0, 0}], 
    {AbsolutePointSize[10], Opacity[1], 
     {RGBColor[1, 0, 1], PointBox[{0, 0}]}, 
     {RGBColor[0, 1, 0], PointBox[{1, 1}]}}}, GeometricTransformationBox[
    {RGBColor[1, 0, 0], Opacity[0.85], RectangleBox[{0, 0}], 
     {AbsolutePointSize[10], Opacity[1], 
      {RGBColor[1, 0, 1], PointBox[{0, 0}]}, 
      {RGBColor[0, 1, 0], PointBox[{1, 1}]}}}, {{{1, 0}, {0, 1}}, {
     0.75, -0.25}}]}]], "Output",
 ImageSize->{184, 132},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->8574980]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->236951139],

Cell["Transformation applied to a 3D shape:", "ExampleText",
 CellID->137131831],

Cell[BoxData[
 RowBox[{
  RowBox[{"gr", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Cuboid", "[", "]"}], ",", 
     RowBox[{"AbsolutePointSize", "[", "10", "]"}], ",", 
     RowBox[{"Opacity", "[", "1", "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"Magenta", ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{"0", ",", "0", ",", "0"}], "}"}], "]"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"Green", ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{"1", ",", "1", ",", "1"}], "}"}], "]"}]}], "}"}]}], "}"}]}], 
  ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->367829949],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics3D", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"Opacity", "[", ".35", "]"}], ",", "Blue", ",", "gr"}], "}"}], 
     ",", 
     RowBox[{"GeometricTransformation", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"Opacity", "[", ".85", "]"}], ",", "Red", ",", "gr"}], "}"}],
        ",", 
       RowBox[{"TranslationTransform", "[", 
        RowBox[{"{", 
         RowBox[{".75", ",", 
          RowBox[{"-", ".25"}], ",", 
          RowBox[{"-", ".5"}]}], "}"}], "]"}]}], "]"}]}], "}"}], ",", 
   RowBox[{"Boxed", "\[Rule]", "False"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->167437831],

Cell[BoxData[
 Graphics3DBox[{
   {RGBColor[0, 0, 1], Opacity[0.35], CuboidBox[{0, 0, 0}], 
    {AbsolutePointSize[10], Opacity[1], 
     {RGBColor[1, 0, 1], Point3DBox[{0, 0, 0}]}, 
     {RGBColor[0, 1, 0], Point3DBox[{1, 1, 1}]}}}, 
   InterpretationBox[
    {RGBColor[1., 0., 0.], Opacity[0.85], 
     CuboidBox[{0.75, -0.25, -0.5}, {1.75, 0.75, 0.5}], 
     {AbsolutePointSize[10.], Opacity[1.], 
      {RGBColor[1., 0., 1.], Point3DBox[{0.75, -0.25, -0.5}]}, 
      {RGBColor[0., 1., 0.], Point3DBox[{1.75, 0.75, 0.5}]}}},
    GeometricTransformation[{
      Opacity[0.85], 
      RGBColor[1, 0, 0], {
       Cuboid[{0, 0, 0}], 
       AbsolutePointSize[10], 
       Opacity[1], {
        RGBColor[1, 0, 1], 
        Point[{0, 0, 0}]}, {
        RGBColor[0, 1, 0], 
        Point[{1, 1, 1}]}}}, {{{1, 0, 0}, {0, 1, 0}, {0, 0, 1}}, {
      0.75, -0.25, -0.5}}]]},
  Boxed->False]], "Output",
 ImageSize->{184, 184},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->495922966,
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->367482409],

Cell["Transforming graphics primitives:", "ExampleText",
 CellID->304565429],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics", "[", 
  RowBox[{
   RowBox[{"GeometricTransformation", "[", 
    RowBox[{
     RowBox[{"Circle", "[", "]"}], ",", 
     RowBox[{"TranslationTransform", "[", 
      RowBox[{"{", 
       RowBox[{"1", ",", "1"}], "}"}], "]"}]}], "]"}], ",", 
   RowBox[{"Axes", "\[Rule]", "True"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->62193242],

Cell[BoxData[
 GraphicsBox[
  GeometricTransformationBox[CircleBox[{0, 0}], {{{1, 0}, {0, 1}}, {1, 1}}],
  Axes->True]], "Output",
 ImageSize->{154, 149},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{150, Automatic}},
 CellLabel->"Out[1]=",
 CellID->36989502]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->2123667759],

Cell[TextData[{
 "Translating by ",
 Cell[BoxData[
  StyleBox["p", "TI"]], "InlineFormula"],
 " and then by ",
 Cell[BoxData[
  StyleBox["q", "TI"]], "InlineFormula"],
 " is the same as translating by ",
 Cell[BoxData[
  RowBox[{
   StyleBox["p", "TI"], "+", 
   StyleBox["q", "TI"]}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->799714906],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Composition", "[", 
  RowBox[{
   RowBox[{"TranslationTransform", "[", 
    RowBox[{"{", 
     RowBox[{"px", ",", "py"}], "}"}], "]"}], ",", 
   RowBox[{"TranslationTransform", "[", 
    RowBox[{"{", 
     RowBox[{"qx", ",", "qy"}], "}"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->195725743],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {"1", "0", 
      RowBox[{"px", "+", "qx"}]},
     {"0", "1", 
      RowBox[{"py", "+", "qy"}]},
     {"0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}}], ")"}],
   "]"}]], "Output",
 ImageSize->{264, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->26172361]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->409950494],

Cell[TextData[{
 "The inverse of translating by ",
 Cell[BoxData[
  StyleBox["p", "TI"]], "InlineFormula"],
 " is the same as translating by ",
 Cell[BoxData[
  RowBox[{"-", 
   StyleBox["p", "TI"]}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->269979487],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"InverseFunction", "[", 
  RowBox[{"TranslationTransform", "[", 
   RowBox[{"{", 
    RowBox[{"px", ",", "py"}], "}"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->404297754],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {"1", "0", 
      RowBox[{"-", "px"}]},
     {"0", "1", 
      RowBox[{"-", "py"}]},
     {"0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}}], ")"}],
   "]"}]], "Output",
 ImageSize->{245, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->149137767]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->888034987],

Cell[TextData[{
 "For geometric transformations use ",
 Cell[BoxData[
  ButtonBox["Translate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Translate"]], "InlineFormula"],
 " directly: "
}], "ExampleText",
 CellID->579020194],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"Graphics", "[", 
    RowBox[{
     RowBox[{"Translate", "[", 
      RowBox[{
       RowBox[{"Circle", "[", "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"1", ",", "1"}], "}"}]}], "]"}], ",", 
     RowBox[{"Frame", "\[Rule]", "True"}]}], "]"}], ",", 
   RowBox[{"Graphics", "[", 
    RowBox[{
     RowBox[{"GeometricTransformation", "[", 
      RowBox[{
       RowBox[{"Circle", "[", "]"}], ",", 
       RowBox[{"TranslationTransform", "[", 
        RowBox[{"{", 
         RowBox[{"1", ",", "1"}], "}"}], "]"}]}], "]"}], ",", 
     RowBox[{"Frame", "\[Rule]", "True"}]}], "]"}]}], "}"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->178089668],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   GraphicsBox[GeometricTransformationBox[CircleBox[{0, 0}], {1, 1}],
    Frame->True], ",", 
   GraphicsBox[
    GeometricTransformationBox[CircleBox[{0, 0}], {{{1, 0}, {0, 1}}, {1, 1}}],
    
    Frame->True]}], "}"}]], "Output",
 ImageSize->{230, 100},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{100, Automatic}},
 CellLabel->"Out[1]=",
 CellID->61658455]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"Graphics3D", "[", 
    RowBox[{
     RowBox[{"Translate", "[", 
      RowBox[{
       RowBox[{"Sphere", "[", "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"1", ",", "1", ",", "1"}], "}"}]}], "]"}], ",", 
     RowBox[{"Axes", "\[Rule]", "True"}]}], "]"}], ",", 
   RowBox[{"Graphics3D", "[", 
    RowBox[{
     RowBox[{"GeometricTransformation", "[", 
      RowBox[{
       RowBox[{"Sphere", "[", "]"}], ",", 
       RowBox[{"TranslationTransform", "[", 
        RowBox[{"{", 
         RowBox[{"1", ",", "1", ",", "1"}], "}"}], "]"}]}], "]"}], ",", 
     RowBox[{"Axes", "\[Rule]", "True"}]}], "]"}]}], "}"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->422850197],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   Graphics3DBox[
    InterpretationBox[SphereBox[{1., 1., 1.}, 1.],
     Translate[
      Sphere[{0, 0, 0}], {1, 1, 1}]],
    Axes->True,
    ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]], ",", 
   Graphics3DBox[
    InterpretationBox[SphereBox[{1., 1., 1.}, 1.],
     GeometricTransformation[
      Sphere[{0, 0, 0}], {{{1, 0, 0}, {0, 1, 0}, {0, 0, 1}}, {1, 1, 1}}]],
    Axes->True,
    ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]]}], "}"}]], "Output",
 ImageSize->{330, 175},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 Graphics3DBoxOptions->{ImageSize->{150, Automatic}},
 CellLabel->"Out[2]=",
 CellID->168877736]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Neat Examples",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->335428132],

Cell["A random translation walk:", "ExampleText",
 CellID->116213242],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics3D", "[", 
  RowBox[{
   RowBox[{"FoldList", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Normal", "@", 
       RowBox[{"GeometricTransformation", "[", 
        RowBox[{"#", ",", 
         RowBox[{"TranslationTransform", "[", "#2", "]"}]}], "]"}]}], "&"}], 
     ",", 
     RowBox[{"Cuboid", "[", "]"}], ",", 
     RowBox[{"RandomChoice", "[", 
      RowBox[{
       RowBox[{"Permutations", "[", 
        RowBox[{"{", 
         RowBox[{"1", ",", "0", ",", "0"}], "}"}], "]"}], ",", "50"}], 
      "]"}]}], "]"}], ",", 
   RowBox[{"Boxed", "\[Rule]", "False"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->272968209],

Cell[BoxData[
 Graphics3DBox[{CuboidBox[{0, 0, 0}], CuboidBox[{0, 1, 0}, {1, 2, 1}], 
   CuboidBox[{1, 1, 0}, {2, 2, 1}], CuboidBox[{1, 1, 1}, {2, 2, 2}], 
   CuboidBox[{2, 1, 1}, {3, 2, 2}], CuboidBox[{2, 2, 1}, {3, 3, 2}], 
   CuboidBox[{3, 2, 1}, {4, 3, 2}], CuboidBox[{4, 2, 1}, {5, 3, 2}], 
   CuboidBox[{5, 2, 1}, {6, 3, 2}], CuboidBox[{5, 2, 2}, {6, 3, 3}], 
   CuboidBox[{6, 2, 2}, {7, 3, 3}], CuboidBox[{7, 2, 2}, {8, 3, 3}], 
   CuboidBox[{7, 3, 2}, {8, 4, 3}], CuboidBox[{7, 4, 2}, {8, 5, 3}], 
   CuboidBox[{7, 5, 2}, {8, 6, 3}], CuboidBox[{7, 5, 3}, {8, 6, 4}], 
   CuboidBox[{8, 5, 3}, {9, 6, 4}], CuboidBox[{8, 6, 3}, {9, 7, 4}], 
   CuboidBox[{8, 7, 3}, {9, 8, 4}], CuboidBox[{8, 8, 3}, {9, 9, 4}], 
   CuboidBox[{8, 9, 3}, {9, 10, 4}], CuboidBox[{8, 10, 3}, {9, 11, 4}], 
   CuboidBox[{8, 11, 3}, {9, 12, 4}], CuboidBox[{8, 11, 4}, {9, 12, 5}], 
   CuboidBox[{8, 11, 5}, {9, 12, 6}], CuboidBox[{8, 12, 5}, {9, 13, 6}], 
   CuboidBox[{8, 12, 6}, {9, 13, 7}], CuboidBox[{8, 12, 7}, {9, 13, 8}], 
   CuboidBox[{8, 13, 7}, {9, 14, 8}], CuboidBox[{8, 13, 8}, {9, 14, 9}], 
   CuboidBox[{8, 14, 8}, {9, 15, 9}], CuboidBox[{8, 14, 9}, {9, 15, 10}], 
   CuboidBox[{9, 14, 9}, {10, 15, 10}], CuboidBox[{9, 14, 10}, {10, 15, 11}], 
   CuboidBox[{9, 14, 11}, {10, 15, 12}], CuboidBox[{9, 14, 12}, {10, 15, 13}],
    CuboidBox[{9, 15, 12}, {10, 16, 13}], 
   CuboidBox[{9, 16, 12}, {10, 17, 13}], CuboidBox[{9, 16, 13}, {10, 17, 14}],
    CuboidBox[{9, 16, 14}, {10, 17, 15}], 
   CuboidBox[{10, 16, 14}, {11, 17, 15}], 
   CuboidBox[{11, 16, 14}, {12, 17, 15}], 
   CuboidBox[{12, 16, 14}, {13, 17, 15}], 
   CuboidBox[{13, 16, 14}, {14, 17, 15}], 
   CuboidBox[{13, 17, 14}, {14, 18, 15}], 
   CuboidBox[{14, 17, 14}, {15, 18, 15}], 
   CuboidBox[{14, 18, 14}, {15, 19, 15}], 
   CuboidBox[{14, 19, 14}, {15, 20, 15}], 
   CuboidBox[{15, 19, 14}, {16, 20, 15}], 
   CuboidBox[{15, 20, 14}, {16, 21, 15}], 
   CuboidBox[{15, 21, 14}, {16, 22, 15}]},
  Boxed->False]], "Output",
 ImageSize->{184, 194},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->241609396,
 ImageCache->GraphicsData["CompressedBitmap", "\<\
eJzlWctyFTcQFdLMGIMh4ECALMAF6+xSRSAQFgk2TkJBAQmYZ27Mww4PG+M8
dvkTKFjksecL8gn5n6lEarV0W63WzBgDRRVTvuM7M6e7Tz/U0uieHa0v3Xk4
Wl9eHM3MrY1Wl5YXn8zMrqzZW2aLUuql/fyzX9nv/9lv8HnZtq198J6dp+3Z
WPefwdUue9b2Sq988rv7/xTu7sS7xt5Vlf3SPl+zf/ZLhdL78P8kIh9HOeVu
WzknqlN5+1cRWx+iDlU5GIDsuUGN3ri9a1WuDtSugzIT1SkdpFo41HbUX2W+
aeRjkJ9sFTxwAuSeYTYrjIwhfnCcgzpczWRXolV3s3rxb5totujAkd5vqFcs
RiHmGePE+0ept9auM13bU6gPE9hmUUv8CBaEeyHzGnP1EGx+wGw6nwdwRh1G
QJmSNOXiwjbh7BFnMJoP4tigvLQ9mUKMgQmrilif/hp5cenAl2c61hjW7TJw
2o3R1gFNbXHvTPQuq1jAN1iWcHgsz1iFQdJFRtNClLilzkpm+CwepBJVTbPv
tPwEHFwnAW881ruCekIUaDWHjuMPHTRT6cgDSqSiT33PopXjdPpo7CHRAN0u
psyTClk0qV1MhqYxIb6boKmOMcQYLIHlvdgxJpgsj0FD/PH+N0xC4/iL9wsV
eQfs7hdGrxG8NiFyBsQiF2epEnTQPnEXRA64c5KJSvAXIs70GdQ3IdRq0s/R
U+/Zx6EHFzzcKuR6G47PSsyUxi49Av2HM3/csyk2X021hY6MsaHz5A+g9wjy
PihI0vnNo0+LUeWz0i3AzpIZsns2uRnxnT2YzQc3hkkpVezc14sahvTYBVG6
qwcuJHHhPYvX8dVUP+1sWR1osYK8vblCZusCy+9B6kyWaT4eeW1dhkfz6F1p
VPHR461dEEaH7NMYH7qi5MNFQF2UYpfNCpcAe0nga4h3XuN3Ak8j8gSPPFuW
466aOgdSl3s63HlAXRHHYopZoLWWcTwXMaWeQbl9C+irPdzOAuoaj2Yxn17r
dcxn3ROfeUDf6OCQo28W45Rjb7Xpqt4v03y//DoiSt1MwfFNL85DN4d7530g
by6OBbGcrXJD7YwxVXFczacWe2oWuqefuwbUi0ePhNr1UxwcyAP6o/qxHb9J
8VjPRUR3rM/04jx0czipXmYjslQvb9EHVi/EslgvtEuM/Yh109FDvgT04sCa
8OjbwuxQRrsVYS1Wzen4vDue/TgP7cCxeH61WaSj1cR1VDyPPeZrCkf0FDy9
K4xWurfzBaDuiX2ary1PAnapHbq2PBHxXavEV5Wja1JaB8d7paEyxRqRLcc3
46xWxnioOXx7A1jBq2WWD/4WXooGvD9n7775Lo9H3yf59EzjbosYrQeBFVwp
v2YQetLn8PxR7xg60YujbN8kDqJU9ILvL7429mxcd8StoRr9+biIDlVol/cB
i4yPRbzzVAs92HH4DFArvb3XSx0F9Gpv76XoxwQdxpjEhtbfpyC5NnBGmMac
zYDUk4Ezw06cncIbuDsOgYZ1gXHeFXaEDsv6oUG9+6IuuZ4mEVfc6cRU0pXA
Rym/Yg3WmPVsVzdHMvZ7i6zbuF8cu96G+nfelWC3Tf0seFOTvMS+uZGu3zHq
avRtSrQu7oUL19KYArlasZFLfXHH9mi3OF92XCe7GWHsC6sAntdtxVj3Xce3
VTYSIVLIJPRRCACzPLlhy6AQTi+SvV86Jg2NP91NIJabbMxtLXNJ1rRx/7R3
v5LOItH7gTaTtTGLbGnXkvpIM2QFSPYnirXN+Vcq3wk1ivzuRuzXyJdmHOKc
VDSvFYMe7SCrCVi28DVoYJe9YQJS/VJgdICsdPkaqgpEiV3wTP3KtSl5x7X8
3gsZUr9hNzki6JP25uiKwz9+ihoasjob11uO/6Md71HryONPgXvu9V+iDa/h
b4yjr9d34bduelZb/gcnnLLG\
\>"]]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["AffineTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AffineTransform"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["RotationTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationTransform"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["GeometricTransformation",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/GeometricTransformation"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["TransformationFunction",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/TransformationFunction"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection"],

Cell[TextData[{
 ButtonBox["Demonstrations with TranslationTransform",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/symbol.html?symbol=\
TranslationTransform"], None}],
 " (",
 ButtonBox["Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None}],
 ")"
}], "RelatedLinks"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->90515769],

Cell[TextData[ButtonBox["Geometric Transforms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/GeometricTransforms"]], "MoreAbout",
 CellID->118880463]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"TranslationTransform - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 27, 37.9276214}", "context" -> "System`", 
    "keywords" -> {
     "affine transformation", "geometric transformation", 
      "homogeneous coordinates", "moving", "rigid motion", "sliding", 
      "translation"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "TranslationTransform[v] gives a TransformationFunction that represents \
translation of points by a vector v.", 
    "synonyms" -> {"translation transform"}, "title" -> 
    "TranslationTransform", "type" -> "Symbol", "uri" -> 
    "ref/TranslationTransform"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[6610, 209, 357, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->702574]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 84503, 1981}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 1628, 38, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2251, 65, 1585, 37, 70, "ObjectNameGrid"],
Cell[3839, 104, 777, 23, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[4653, 132, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[5131, 149, 389, 12, 70, "Notes",
 CellID->257351590],
Cell[5523, 163, 573, 20, 70, "Notes",
 CellID->232415095],
Cell[6099, 185, 474, 19, 70, "Notes",
 CellID->212580077]
}, Closed]],
Cell[CellGroupData[{
Cell[6610, 209, 357, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->702574],
Cell[CellGroupData[{
Cell[6992, 223, 148, 5, 70, "ExampleSection",
 CellID->282139695],
Cell[7143, 230, 252, 9, 70, "ExampleText",
 CellID->143900862],
Cell[CellGroupData[{
Cell[7420, 243, 163, 5, 28, "Input",
 CellID->750626971],
Cell[7586, 250, 405, 14, 64, "Output",
 CellID->197447562]
}, Open  ]],
Cell[8006, 267, 89, 1, 70, "ExampleText",
 CellID->608535536],
Cell[CellGroupData[{
Cell[8120, 272, 143, 5, 28, "Input",
 CellID->22306003],
Cell[8266, 279, 251, 9, 36, "Output",
 CellID->246473222]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[8566, 294, 219, 7, 70, "ExampleSection",
 CellID->54466981],
Cell[8788, 303, 73, 1, 70, "ExampleText",
 CellID->25456152],
Cell[CellGroupData[{
Cell[8886, 308, 214, 7, 70, "Input",
 CellID->330826147],
Cell[9103, 317, 503, 16, 94, "Output",
 CellID->450901510]
}, Open  ]],
Cell[CellGroupData[{
Cell[9643, 338, 163, 5, 70, "Input",
 CellID->15646037],
Cell[9809, 345, 323, 11, 36, "Output",
 CellID->36046072]
}, Open  ]],
Cell[10147, 359, 64, 1, 70, "ExampleText",
 CellID->63688802],
Cell[CellGroupData[{
Cell[10236, 364, 111, 3, 70, "Input",
 CellID->234230738],
Cell[10350, 369, 591, 20, 94, "Output",
 CellID->408114852]
}, Open  ]],
Cell[CellGroupData[{
Cell[10978, 394, 205, 6, 70, "Input",
 CellID->17728099],
Cell[11186, 402, 323, 11, 36, "Output",
 CellID->84982390]
}, Open  ]],
Cell[11524, 416, 71, 1, 70, "ExampleText",
 CellID->513676756],
Cell[CellGroupData[{
Cell[11620, 421, 205, 7, 70, "Input",
 CellID->129763148],
Cell[11828, 430, 424, 15, 36, "Output",
 CellID->340236158]
}, Open  ]],
Cell[12267, 448, 93, 1, 70, "ExampleText",
 CellID->812525649],
Cell[CellGroupData[{
Cell[12385, 453, 279, 8, 70, "Input",
 CellID->65750619],
Cell[12667, 463, 434, 15, 36, "Output",
 CellID->233290623]
}, Open  ]],
Cell[13116, 481, 125, 3, 70, "ExampleDelimiter",
 CellID->153692911],
Cell[13244, 486, 80, 1, 70, "ExampleText",
 CellID->479507293],
Cell[13327, 489, 620, 20, 70, "Input",
 CellID->419323043],
Cell[CellGroupData[{
Cell[13972, 513, 596, 20, 70, "Input",
 CellID->150504143],
Cell[14571, 535, 631, 15, 153, "Output",
 CellID->8574980]
}, Open  ]],
Cell[15217, 553, 125, 3, 70, "ExampleDelimiter",
 CellID->236951139],
Cell[15345, 558, 80, 1, 70, "ExampleText",
 CellID->137131831],
Cell[15428, 561, 637, 20, 70, "Input",
 CellID->367829949],
Cell[CellGroupData[{
Cell[16090, 585, 709, 22, 70, "Input",
 CellID->167437831],
Cell[16802, 609, 4868, 93, 205, 895, 23, "CachedBoxData", "BoxData", "Output",
 
 CellID->495922966]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[21719, 708, 227, 7, 70, "ExampleSection",
 CellID->367482409],
Cell[21949, 717, 76, 1, 70, "ExampleText",
 CellID->304565429],
Cell[CellGroupData[{
Cell[22050, 722, 374, 11, 70, "Input",
 CellID->62193242],
Cell[22427, 735, 313, 9, 170, "Output",
 CellID->36989502]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[22789, 750, 238, 7, 70, "ExampleSection",
 CellID->2123667759],
Cell[23030, 759, 348, 14, 70, "ExampleText",
 CellID->799714906],
Cell[CellGroupData[{
Cell[23403, 777, 332, 10, 70, "Input",
 CellID->195725743],
Cell[23738, 789, 462, 16, 64, "Output",
 CellID->26172361]
}, Open  ]],
Cell[24215, 808, 125, 3, 70, "ExampleDelimiter",
 CellID->409950494],
Cell[24343, 813, 263, 10, 70, "ExampleText",
 CellID->269979487],
Cell[CellGroupData[{
Cell[24631, 827, 209, 6, 70, "Input",
 CellID->404297754],
Cell[24843, 835, 451, 16, 64, "Output",
 CellID->149137767]
}, Open  ]],
Cell[25309, 854, 125, 3, 70, "ExampleDelimiter",
 CellID->888034987],
Cell[25437, 859, 229, 8, 70, "ExampleText",
 CellID->579020194],
Cell[CellGroupData[{
Cell[25691, 871, 695, 21, 70, "Input",
 CellID->178089668],
Cell[26389, 894, 454, 14, 121, "Output",
 CellID->61658455]
}, Open  ]],
Cell[CellGroupData[{
Cell[26880, 913, 717, 21, 70, "Input",
 CellID->422850197],
Cell[27600, 936, 47773, 795, 196, "Output",
 CellID->168877736]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[75422, 1737, 228, 7, 70, "ExampleSection",
 CellID->335428132],
Cell[75653, 1746, 69, 1, 70, "ExampleText",
 CellID->116213242],
Cell[CellGroupData[{
Cell[75747, 1751, 657, 20, 70, "Input",
 CellID->272968209],
Cell[76407, 1773, 4150, 73, 215, 1983, 33, "CachedBoxData", "BoxData", \
"Output",
 CellID->241609396]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[80618, 1853, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[80937, 1864, 1035, 35, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[82009, 1904, 305, 8, 70, "RelatedLinksSection"],
Cell[82317, 1914, 382, 12, 70, "RelatedLinks"]
}, Open  ]],
Cell[CellGroupData[{
Cell[82736, 1931, 318, 9, 70, "MoreAboutSection",
 CellID->90515769],
Cell[83057, 1942, 151, 3, 70, "MoreAbout",
 CellID->118880463]
}, Open  ]],
Cell[83223, 1948, 27, 0, 70, "History"],
Cell[83253, 1950, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

