(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     43724,       1585]
NotebookOptionsPosition[     35566,       1294]
NotebookOutlinePosition[     36896,       1331]
CellTagsIndexPosition[     36810,       1326]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Putting Expressions into Different Forms" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/PuttingExpressionsIntoDifferentForms"], 
          "Trigonometric Expressions" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/TrigonometricExpressions"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Putting Expressions into Different Forms\"\>", 
       2->"\<\"Trigonometric Expressions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"TrigExpand" :> 
          Documentation`HelpLookup["paclet:ref/TrigExpand"], "TrigFactor" :> 
          Documentation`HelpLookup["paclet:ref/TrigFactor"], "TrigToExp" :> 
          Documentation`HelpLookup["paclet:ref/TrigToExp"], "Simplify" :> 
          Documentation`HelpLookup["paclet:ref/Simplify"], "FullSimplify" :> 
          Documentation`HelpLookup["paclet:ref/FullSimplify"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"TrigExpand\"\>", 
       2->"\<\"TrigFactor\"\>", 3->"\<\"TrigToExp\"\>", 4->"\<\"Simplify\"\>",
        5->"\<\"FullSimplify\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Algebraic Transformations" :> 
          Documentation`HelpLookup["paclet:guide/AlgebraicTransformations"], 
          "Hyperbolic Functions" :> 
          Documentation`HelpLookup["paclet:guide/HyperbolicFunctions"], 
          "Trigonometric Functions" :> 
          Documentation`HelpLookup["paclet:guide/TrigonometricFunctions"]}, #,
          2]& , Evaluator -> Automatic], {
       1->"\<\"Algebraic Transformations\"\>", 
       2->"\<\"Hyperbolic Functions\"\>", 
       3->"\<\"Trigonometric Functions\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["TrigReduce", "ObjectName"], Cell[BoxData[
     TooltipBox[
      TogglerBox[Dynamic[
        FrontEnd`CurrentValue[
         FrontEnd`EvaluationNotebook[], {
         TaggingRules, "ModificationHighlight"}]], {True->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVVly2zAMDQhxEylSsizLaxJn73KafvQAuUDv/1eApJuosceeTDAWF4B8
eFgk71///HoVV1dXK3p+0wNfIzdCLL4I6p1sYBYuPy3SI2AmRBpuaE/TPRsE
3Isg8mrB9l6ILQhhLkbd93MBuwVxAthdZwPr5wVVzBc8wAvMX8TFqPkqE8q6
rDCsKPjZ/g1guz+PKvLFfGsDU1R4A8zDhgLangX9x/WGM7A1qSQ7rraYkaLf
LHZ5FWY9H7rew/n4/6vWPKT2mXFBaH/NGejzqi/VnFEFJ2Rb1Lo+jtofllPD
RVJjdZLr51Eb/ED1K+QI109LcIg+sdSIWqa5g85L6MiNoRloTws+pdBKiFGl
mx3N0jcAlcsKeFivVkteVOgMSNsC18sYS/iIMdZoK43WSdQVK1SLlYy09bZ2
aEO+aoPiACNak0EBbtfLRIz9KJRcroaSKwE11BprpRHqiJ0BrxtJG4pLYqWc
RZO4BQwSbGT4rFivaLjjQdscjSvlih48Mu+WDisOBcB5ehgeDEZQlikkKh3b
KA1FMTDg9yFxLSe4XC37begHTDh5seS1I/gaW025Yo/yUFbmSalShzon1Kch
JTNn2hKS1hyWVIkf5n2LHEQgt0zeUNTkMaYrDdkMZdo3fJCDHx7hxzAC+8rs
XYBokTPfEJk6JVlZx3Yl0UnTYMPvSUt6zghf8dhIi5VDnwO+Hcb1MHK1msK+
oy5qkYS7wVC3cfiq8ojBcMtFMJXnDiIcLPFJClDFGkJBhXGEZV6VcCQFoey7
tyvBF6lw0uKH6nyQ5fqhrKx/Kxe8+3coucpGN4U5+RYux7vnhGO5P7hKenqz
xSO3ivjisf345fg5Pt1ScCRkkzR1U1Dur+NCec4LZ49Ynx9P04GmOvlnqcKZ
79BfJfcjkg==
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlLlKQ0EYRi/iA9gpWIgvIXbjM1iKWAjW2vgeZnWN+AgWZhEULCxELFyi
ZnfJavbeJOr/33xXcj8CNnY5EA7k3GWYmTuzq+uLa2OWZS38/MYt0DbDHH8Q
8uTB64bRNrmsQkaPRoQSGX1ZqZPR9/eEazL6klIlowf8wiUZ3ecV3snowYDw
SEafUmJk9HmlQh4cX4GM7vUId2T0VFIho28or2T0w5DQIKN3PoUaGX1FaZLR
0ymh6sz7hHLk9PMzoU5Gz6SFmvFsCbd9Xzj9qyc0nfe49xfuTyYUEzoQbmh8
+N8e37HTbVpk9MSzUCajbypFMvq0UiOjzyklMnqvKzRMJCwU+/vjzemTSpns
nh97feJmd0e4cvppTKiQ0bsdoW5ecorZDgpPtD6xqFA2fp+Qp/vtdUj1v797
Oqdwnb2Ov++fUVpk9PCJUCC7n4t5wz5xj9/+rj6ceXbPP86NbEb58/k56qP+
n/0bdGyYqQ==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}], False->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVdd22zAMDUBxSaJm5HgljrPTJN177/nWh35AfqD//1aAVGrrJLbbk+BI
FEmQFxcXlDQ9//3zHNfW1gZ0/6IbbsYOhahvCGrO9qHM/mnhmMMLoKv0RD6V
cPyZHTTcDI5NkYnQq4UQ0BdiAkKYZagl1OOw5fWtcgyvPp7AnefsGG0DBIeA
9RZVrNfcwBasb4nlbP9u4cbc/XD/zXcIgC2qYYazJTQiGSYHS0Hr8Yu5LV/h
ePvR5oMOKswAQ0M8R5OloFMWSNRE+IAVeALv38KX26wAV1uU5Ojv16PQy8q+
gMP+9ABW5C9KbkZi6qv19DGc/ID9b+CrZTjbbVagH3p91oIcL+GZ6JDNUet4
HpWUXxx5BaWZxRh1FQgH6JqoKcarF/2/dblez7IE0XmWGlFL/6ygchIqCmPo
CTSmDq9SaCUUhfI7K3pKlwJESZiAveHGxoA7ESYGpM2B62WMJXzEoojRRhpt
IlFHPKFyjGRBQ2fjBG0WttpMcYIFWhNAAXaHA0+M4yiUXK6UxJWAGmKNsdII
cYGVAadTSQPKS2KkEovGc8swk2ALhg8Tww1qdrjRNmSTtOUqHDhk3jktVpwK
QOLoZngwWICyTMFTqdhHMrQTDQPeazzXdgWXK+e4KV3AhH0US1Ergo8x16QV
R5QXZWWeJJW6qLNHPWq8mEFpS0hac1pSeX4YxjlyEhmFZfKGsqaIhd+Sks+Q
0i7lhZx8cwYPmx5wrMA+yaCwyMqnRCb2IiubsF9JTKRJMeX3JKd5VoS3OEyl
xShBFxLebXrDpsfVSlv2FZ2iHMn4NBg6bZy+ihxiZvjIFWAixyeIcLDNT1KC
qogha1Gh14NB6LXpSEpC2bm3y8O3FmHniF9U55INhnttz7pZuWDu79BqFZxJ
F2bhWzjo7Zx6HMvng6ukuztzvGJXa66NmF/+crzrHe1ScmTkk/SouqB8vq42
0jl0EnuF9/RsMR1Io4U/S5Wt+A79AaDyK/w=
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlUtrE1EYhgfxBwguFBTEvyAiFhfzHxRFvC0E17rQhYLiT7D3u/gTXNg2
3sBFFyIutE1qm17UZJKZZJK0KFjBtur7Td4jnhe7dTMPhAfyQObMnO9Mjl69
fvraniAI7v/+7A3IRri7/8VGWJwHVTH72qohZp+ZBnUx+yWjJWYfHwNvxOwX
jKaYfXAAzIrZ+/tARcw+NAhKYvaTRiJmP2gUxH+vLxKzl5cMMXtfL3gvZn84
Cdpi9hvGJzH7yxegJWa/YnTE7Mtl0HTPfZ/xyPWVZZCGvQ/Au65fSV9aNMLJ
CfDW9a0fIBWz/9wBHbcOf/74PX83W99j128ZNTH71BMQidmfFkAiZj9irIvZ
Fz+AWMx+yEjF7KeMVOw/32x/iuHoCHjt+s42aIfTU6DWna/Prm9vgVb4cc0I
h4fAgusZ62L2A0YsZj9h1MX+9bN9LHfP75zrZ42a2L9/3+wXjVTM7q8jm9NE
5q8wA+JwoB/8eX+eNxIx+00jErP3GA2xvy7uG88R+z2jI2Y/bLTE/v1l76WG
mxP/+r7ZN7+Bupj9mFERsz9/Bppi9myeEzH7QgnUxOy3jYaY/a7RFrNfNtpi
9nNGXcw+NgoaYnbOk2///PJ/bXXFcP37JojF7CPDIBazf/0CKmL2+TlQEbPf
MZpi9olxkIrZS0UQidn3G1Ux+3EjErOfMapi9mBX8p73vOc973n/P/0XnFJ2
Rg==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}]}, "\"\<Click me\>\""],
      "Toggle changes in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["TrigReduce",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/TrigReduce"], "[", 
       StyleBox["expr", "TI"], "]"}]], "InlineFormula"],
     "\[LineSeparator]rewrites products and powers of trigonometric functions \
in ",
     Cell[BoxData[
      StyleBox["expr", "TI"]], "InlineFormula"],
     " in terms of trigonometric functions with combined arguments. "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->4550]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TrigReduce",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TrigReduce"]], "InlineFormula"],
 " operates on both circular and hyperbolic functions. "
}], "Notes",
 CellID->6941],

Cell[TextData[{
 "Given a trigonometric polynomial, ",
 Cell[BoxData[
  ButtonBox["TrigReduce",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TrigReduce"]], "InlineFormula"],
 " typically yields a linear expression involving trigonometric functions \
with more complicated arguments. "
}], "Notes",
 CellID->16502],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TrigReduce",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TrigReduce"]], "InlineFormula"],
 " automatically threads over lists, as well as equations, inequalities and \
logic functions."
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->82730543]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->139105191],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(1)", "ExampleCount"]
}], "ExampleSection",
 CellID->47633551],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TrigReduce", "[", 
  RowBox[{"2", 
   RowBox[{
    RowBox[{"Cos", "[", "x", "]"}], "^", "2"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->24764],

Cell[BoxData[
 RowBox[{"1", "+", 
  RowBox[{"Cos", "[", 
   RowBox[{"2", " ", "x"}], "]"}]}]], "Output",
 ImageSize->{76, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->31437095]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TrigReduce", "[", 
  RowBox[{"2", 
   RowBox[{"Sin", "[", "x", "]"}], 
   RowBox[{"Cos", "[", "y", "]"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->591097374],

Cell[BoxData[
 RowBox[{
  RowBox[{"Sin", "[", 
   RowBox[{"x", "-", "y"}], "]"}], "+", 
  RowBox[{"Sin", "[", 
   RowBox[{"x", "+", "y"}], "]"}]}]], "Output",
 ImageSize->{140, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->228737111]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->12833],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TrigReduce",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TrigReduce"]], "InlineFormula"],
 " operates on hyperbolic trigonometric functions:"
}], "ExampleText",
 CellID->196779600],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TrigReduce", "[", 
  RowBox[{"2", 
   RowBox[{"Sinh", "[", "u", "]"}], 
   RowBox[{"Cosh", "[", "v", "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->32588323],

Cell[BoxData[
 RowBox[{
  RowBox[{"Sinh", "[", 
   RowBox[{"u", "-", "v"}], "]"}], "+", 
  RowBox[{"Sinh", "[", 
   RowBox[{"u", "+", "v"}], "]"}]}]], "Output",
 ImageSize->{154, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->599350]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Options",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->20002],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Modulus",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->19202],

Cell["\<\
Manipulation with polynomials is performed using modular arithmetic:\
\>", "ExampleText",
 CellID->9379777],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Cos", "[", "x", "]"}], "^", "50"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->466802486],

Cell[BoxData[
 SuperscriptBox[
  RowBox[{"Cos", "[", "x", "]"}], "50"]], "Output",
 ImageSize->{57, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->218094279]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TrigReduce", "[", 
  RowBox[{"%", ",", 
   RowBox[{"Modulus", "\[Rule]", "7"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->204620218],

Cell[BoxData[
 RowBox[{"4", " ", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"Cos", "[", 
     RowBox[{"48", " ", "x"}], "]"}], "+", 
    RowBox[{"Cos", "[", 
     RowBox[{"50", " ", "x"}], "]"}]}], ")"}]}]], "Output",
 ImageSize->{159, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->802163236]
}, Open  ]],

Cell["Compare with the reduction over rationals:", "ExampleText",
 CellID->265159979],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TrigReduce", "[", "%%", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->2804612],

Cell[BoxData[
 RowBox[{
  FractionBox["1", "562949953421312"], 
  RowBox[{"(", 
   RowBox[{"63205303218876", "+", 
    RowBox[{"121548660036300", " ", 
     RowBox[{"Cos", "[", 
      RowBox[{"2", " ", "x"}], "]"}]}], "+", 
    RowBox[{"108043253365600", " ", 
     RowBox[{"Cos", "[", 
      RowBox[{"4", " ", "x"}], "]"}]}], "+", 
    RowBox[{"88749815264600", " ", 
     RowBox[{"Cos", "[", 
      RowBox[{"6", " ", "x"}], "]"}]}], "+", 
    RowBox[{"67327446062800", " ", 
     RowBox[{"Cos", "[", 
      RowBox[{"8", " ", "x"}], "]"}]}], "+", 
    RowBox[{"47129212243960", " ", 
     RowBox[{"Cos", "[", 
      RowBox[{"10", " ", "x"}], "]"}]}], "+", 
    RowBox[{"30405943383200", " ", 
     RowBox[{"Cos", "[", 
      RowBox[{"12", " ", "x"}], "]"}]}], "+", 
    RowBox[{"18053528883775", " ", 
     RowBox[{"Cos", "[", 
      RowBox[{"14", " ", "x"}], "]"}]}], "+", 
    RowBox[{"9847379391150", " ", 
     RowBox[{"Cos", "[", 
      RowBox[{"16", " ", "x"}], "]"}]}], "+", 
    RowBox[{"4923689695575", " ", 
     RowBox[{"Cos", "[", 
      RowBox[{"18", " ", "x"}], "]"}]}], "+", 
    RowBox[{"2250829575120", " ", 
     RowBox[{"Cos", "[", 
      RowBox[{"20", " ", "x"}], "]"}]}], "+", 
    RowBox[{"937845656300", " ", 
     RowBox[{"Cos", "[", 
      RowBox[{"22", " ", "x"}], "]"}]}], "+", 
    RowBox[{"354860518600", " ", 
     RowBox[{"Cos", "[", 
      RowBox[{"24", " ", "x"}], "]"}]}], "+", 
    RowBox[{"121399651100", " ", 
     RowBox[{"Cos", "[", 
      RowBox[{"26", " ", "x"}], "]"}]}], "+", 
    RowBox[{"37353738800", " ", 
     RowBox[{"Cos", "[", 
      RowBox[{"28", " ", "x"}], "]"}]}], "+", 
    RowBox[{"10272278170", " ", 
     RowBox[{"Cos", "[", 
      RowBox[{"30", " ", "x"}], "]"}]}], "+", 
    RowBox[{"2505433700", " ", 
     RowBox[{"Cos", "[", 
      RowBox[{"32", " ", "x"}], "]"}]}], "+", 
    RowBox[{"536878650", " ", 
     RowBox[{"Cos", "[", 
      RowBox[{"34", " ", "x"}], "]"}]}], "+", 
    RowBox[{"99884400", " ", 
     RowBox[{"Cos", "[", 
      RowBox[{"36", " ", "x"}], "]"}]}], "+", 
    RowBox[{"15890700", " ", 
     RowBox[{"Cos", "[", 
      RowBox[{"38", " ", "x"}], "]"}]}], "+", 
    RowBox[{"2118760", " ", 
     RowBox[{"Cos", "[", 
      RowBox[{"40", " ", "x"}], "]"}]}], "+", 
    RowBox[{"230300", " ", 
     RowBox[{"Cos", "[", 
      RowBox[{"42", " ", "x"}], "]"}]}], "+", 
    RowBox[{"19600", " ", 
     RowBox[{"Cos", "[", 
      RowBox[{"44", " ", "x"}], "]"}]}], "+", 
    RowBox[{"1225", " ", 
     RowBox[{"Cos", "[", 
      RowBox[{"46", " ", "x"}], "]"}]}], "+", 
    RowBox[{"50", " ", 
     RowBox[{"Cos", "[", 
      RowBox[{"48", " ", "x"}], "]"}]}], "+", 
    RowBox[{"Cos", "[", 
     RowBox[{"50", " ", "x"}], "]"}]}], ")"}]}]], "Output",
 ImageSize->{559, 166},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->33316194]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Collect", "[", 
  RowBox[{"%", ",", "_Cos", ",", 
   RowBox[{
    RowBox[{"Mod", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"Numerator", "[", "#", "]"}], 
       RowBox[{"PowerMod", "[", 
        RowBox[{
         RowBox[{"Denominator", "[", "#", "]"}], ",", 
         RowBox[{"-", "1"}], ",", "7"}], "]"}]}], ",", "7"}], "]"}], "&"}]}], 
  "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->520713546],

Cell[BoxData[
 RowBox[{
  RowBox[{"4", " ", 
   RowBox[{"Cos", "[", 
    RowBox[{"48", " ", "x"}], "]"}]}], "+", 
  RowBox[{"4", " ", 
   RowBox[{"Cos", "[", 
    RowBox[{"50", " ", "x"}], "]"}]}]}]], "Output",
 ImageSize->{155, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->240783857]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->13048],

Cell["Find the period of a trigonometric polynomial:", "ExampleText",
 CellID->71413300],

Cell[BoxData[
 RowBox[{
  RowBox[{"p", "=", 
   RowBox[{
    RowBox[{"15", 
     RowBox[{
      RowBox[{"Sin", "[", 
       RowBox[{"12", " ", "x"}], "]"}], "^", "2"}]}], "+", 
    RowBox[{"12", " ", 
     RowBox[{
      RowBox[{"Sin", "[", 
       RowBox[{"15", " ", "x"}], "]"}], "^", "2"}]}]}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->381546912],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TrigReduce", "[", "p", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->47621341],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", 
   FractionBox["3", "2"]}], " ", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"-", "9"}], "+", 
    RowBox[{"5", " ", 
     RowBox[{"Cos", "[", 
      RowBox[{"24", " ", "x"}], "]"}]}], "+", 
    RowBox[{"4", " ", 
     RowBox[{"Cos", "[", 
      RowBox[{"30", " ", "x"}], "]"}]}]}], ")"}]}]], "Output",
 ImageSize->{218, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->86321052]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"2", 
  RowBox[{"Pi", "/", 
   RowBox[{"GCD", "[", 
    RowBox[{"24", ",", "30"}], "]"}]}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->75991878],

Cell[BoxData[
 FractionBox["\[Pi]", "3"]], "Output",
 ImageSize->{17, 28},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->78356109]
}, Open  ]],

Cell["Verification:", "ExampleText",
 CellID->34644411],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"(", 
    RowBox[{"p", "/.", 
     RowBox[{"x", "\[Rule]", 
      RowBox[{"x", "+", "%"}]}]}], ")"}], "-", "p"}], "//", 
  "Simplify"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->111920163],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->96340095]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->13506],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   RowBox[{
    ButtonBox["ChebyshevT",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/ChebyshevT"], "[", 
    RowBox[{
     StyleBox["n", "TI"], ",", 
     RowBox[{
      ButtonBox["Cos",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/Cos"], "[", 
      StyleBox["x", "TI"], "]"}]}], "]"}], " "}]], "InlineFormula"],
 "reduces to ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Cos",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Cos"], "[", 
   RowBox[{
    StyleBox["n", "TI"], " ", 
    StyleBox["x", "TI"]}], "]"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->19589731],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ChebyshevT", "[", 
  RowBox[{"10", ",", 
   RowBox[{"Cos", "[", "x", "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->137546738],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "1"}], "+", 
  RowBox[{"50", " ", 
   SuperscriptBox[
    RowBox[{"Cos", "[", "x", "]"}], "2"]}], "-", 
  RowBox[{"400", " ", 
   SuperscriptBox[
    RowBox[{"Cos", "[", "x", "]"}], "4"]}], "+", 
  RowBox[{"1120", " ", 
   SuperscriptBox[
    RowBox[{"Cos", "[", "x", "]"}], "6"]}], "-", 
  RowBox[{"1280", " ", 
   SuperscriptBox[
    RowBox[{"Cos", "[", "x", "]"}], "8"]}], "+", 
  RowBox[{"512", " ", 
   SuperscriptBox[
    RowBox[{"Cos", "[", "x", "]"}], "10"]}]}]], "Output",
 ImageSize->{452, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->482867625]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TrigReduce", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->934346010],

Cell[BoxData[
 RowBox[{"Cos", "[", 
  RowBox[{"10", " ", "x"}], "]"}]], "Output",
 ImageSize->{63, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->112479127]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["ChebyshevU",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ChebyshevU"], "[", 
   RowBox[{
    StyleBox["n", "TI"], ",", 
    RowBox[{
     ButtonBox["Cos",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Cos"], "[", 
     StyleBox["x", "TI"], "]"}]}], "]"}]], "InlineFormula"],
 " is related to ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Sin",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Sin"], "[", 
   RowBox[{
    StyleBox["n", "TI"], " ", 
    StyleBox["x", "TI"]}], "]"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->788628854],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TrigReduce", "[", 
  RowBox[{
   RowBox[{"Sin", "[", "x", "]"}], 
   RowBox[{"ChebyshevU", "[", 
    RowBox[{"10", ",", 
     RowBox[{"Cos", "[", "x", "]"}]}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->21117297],

Cell[BoxData[
 RowBox[{"Sin", "[", 
  RowBox[{"11", " ", "x"}], "]"}]], "Output",
 ImageSize->{63, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->821970794]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->432339429],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TrigReduce",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TrigReduce"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["TrigExpand",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TrigExpand"]], "InlineFormula"],
 " are, generically, inverses of each other:"
}], "ExampleText",
 CellID->710105148],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TrigExpand", "[", 
  RowBox[{"Sin", "[", 
   RowBox[{"5", "x"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->49485222],

Cell[BoxData[
 RowBox[{
  RowBox[{"5", " ", 
   SuperscriptBox[
    RowBox[{"Cos", "[", "x", "]"}], "4"], " ", 
   RowBox[{"Sin", "[", "x", "]"}]}], "-", 
  RowBox[{"10", " ", 
   SuperscriptBox[
    RowBox[{"Cos", "[", "x", "]"}], "2"], " ", 
   SuperscriptBox[
    RowBox[{"Sin", "[", "x", "]"}], "3"]}], "+", 
  SuperscriptBox[
   RowBox[{"Sin", "[", "x", "]"}], "5"]}]], "Output",
 ImageSize->{295, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->47463080]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TrigReduce", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->562793634],

Cell[BoxData[
 RowBox[{"Sin", "[", 
  RowBox[{"5", " ", "x"}], "]"}]], "Output",
 ImageSize->{56, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->399666239]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->40442951],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TrigReduce",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TrigReduce"]], "InlineFormula"],
 " threads over lists, inequalities, equations and logical operations:"
}], "ExampleText",
 CellID->269834246],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TrigReduce", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"4", " ", 
     RowBox[{
      RowBox[{"Sin", "[", "x", "]"}], "^", "4"}]}], "\[Equal]", "1"}], "&&", 
   RowBox[{
    RowBox[{"2", 
     RowBox[{
      RowBox[{"Cos", "[", "x", "]"}], "^", "2"}]}], "\[GreaterEqual]", 
    "1"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->148369671],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{
    FractionBox["1", "2"], " ", 
    RowBox[{"(", 
     RowBox[{"3", "-", 
      RowBox[{"4", " ", 
       RowBox[{"Cos", "[", 
        RowBox[{"2", " ", "x"}], "]"}]}], "+", 
      RowBox[{"Cos", "[", 
       RowBox[{"4", " ", "x"}], "]"}]}], ")"}]}], "\[Equal]", "1"}], "&&", 
  RowBox[{
   RowBox[{"1", "+", 
    RowBox[{"Cos", "[", 
     RowBox[{"2", " ", "x"}], "]"}]}], "\[GreaterEqual]", "1"}]}]], "Output",
 ImageSize->{315, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->10209496]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->16913],

Cell[TextData[{
 "The value of the option ",
 Cell[BoxData[
  ButtonBox["Modulus",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Modulus"]], "InlineFormula"],
 " must be an integer:"
}], "ExampleText",
 CellID->924552947],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TrigReduce", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Sinh", "[", "t", "]"}], "^", "2"}], ",", 
   RowBox[{"Modulus", "\[Rule]", "Pi"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->267185323],

Cell[BoxData[
 RowBox[{
  RowBox[{"TrigReduce", "::", "\<\"modn\"\>"}], ":", 
  " ", "\<\"Value of option \\!\\(Modulus\\) -> \\!\\(\[Pi]\\) should be an \
integer. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\
\", ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/General/modn\\\", \
ButtonNote -> \\\"TrigReduce::modn\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->394626467],

Cell[BoxData[
 RowBox[{"TrigReduce", "[", 
  RowBox[{
   SuperscriptBox[
    RowBox[{"Sinh", "[", "t", "]"}], "2"], ",", 
   RowBox[{"Modulus", "\[Rule]", "\[Pi]"}]}], "]"}]], "Output",
 ImageSize->{227, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->593667220]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->240475609],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TrigReduce",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TrigReduce"]], "InlineFormula"],
 " requires explicit trigonometric functions:"
}], "ExampleText",
 CellID->103331699],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TrigReduce", "[", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"Exp", "[", 
      RowBox[{"I", " ", "x"}], "]"}], "+", 
     RowBox[{"Exp", "[", 
      RowBox[{
       RowBox[{"-", "I"}], " ", "x"}], "]"}]}], ")"}], "^", "3"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->93796402],

Cell[BoxData[
 SuperscriptBox[
  RowBox[{"(", 
   RowBox[{
    SuperscriptBox["\[ExponentialE]", 
     RowBox[{
      RowBox[{"-", "\[ImaginaryI]"}], " ", "x"}]], "+", 
    SuperscriptBox["\[ExponentialE]", 
     RowBox[{"\[ImaginaryI]", " ", "x"}]]}], ")"}], "3"]], "Output",
 ImageSize->{81, 21},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->215565831]
}, Open  ]],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["ExpToTrig",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ExpToTrig"]], "InlineFormula"],
 " to convert exponential to trigonometric functions:"
}], "ExampleText",
 CellID->1008052319],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"ExpToTrig", "[", "%", "]"}], "//", "TrigReduce"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->898172079],

Cell[BoxData[
 RowBox[{"2", " ", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"3", " ", 
     RowBox[{"Cos", "[", "x", "]"}]}], "+", 
    RowBox[{"Cos", "[", 
     RowBox[{"3", " ", "x"}], "]"}]}], ")"}]}]], "Output",
 ImageSize->{145, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->675228340]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->26695912],

Cell["Reducing constants might not always give the desired effect:", \
"ExampleText",
 CellID->23468404],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TrigReduce", "[", 
  RowBox[{
   RowBox[{"Cosh", "[", "1", "]"}], "^", "3"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->678517677],

Cell[BoxData[
 FractionBox[
  RowBox[{"1", "+", 
   RowBox[{"3", " ", 
    SuperscriptBox["\[ExponentialE]", "2"]}], "+", 
   RowBox[{"3", " ", 
    SuperscriptBox["\[ExponentialE]", "4"]}], "+", 
   SuperscriptBox["\[ExponentialE]", "6"]}], 
  RowBox[{"8", " ", 
   SuperscriptBox["\[ExponentialE]", "3"]}]]], "Output",
 ImageSize->{111, 35},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->169401436]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"TrigReduce", "[", 
   RowBox[{
    RowBox[{"Cosh", "[", "x", "]"}], "^", "3"}], "]"}], "/.", 
  RowBox[{"x", "\[Rule]", "1"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->8943582],

Cell[BoxData[
 RowBox[{
  FractionBox["1", "4"], " ", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"3", " ", 
     RowBox[{"Cosh", "[", "1", "]"}]}], "+", 
    RowBox[{"Cosh", "[", "3", "]"}]}], ")"}]}]], "Output",
 ImageSize->{153, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->87469608]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Neat Examples",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->25210],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"4", " ", 
  RowBox[{"Product", "[", 
   RowBox[{
    RowBox[{"Cos", "[", 
     RowBox[{
      RowBox[{"2", "^", "k"}], " ", "x"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"k", ",", "0", ",", "2"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->387029661],

Cell[BoxData[
 RowBox[{"4", " ", 
  RowBox[{"Cos", "[", "x", "]"}], " ", 
  RowBox[{"Cos", "[", 
   RowBox[{"2", " ", "x"}], "]"}], " ", 
  RowBox[{"Cos", "[", 
   RowBox[{"4", " ", "x"}], "]"}]}]], "Output",
 ImageSize->{166, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->255286787]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TrigReduce", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->614758385],

Cell[BoxData[
 RowBox[{
  RowBox[{"Cos", "[", "x", "]"}], "+", 
  RowBox[{"Cos", "[", 
   RowBox[{"3", " ", "x"}], "]"}], "+", 
  RowBox[{"Cos", "[", 
   RowBox[{"5", " ", "x"}], "]"}], "+", 
  RowBox[{"Cos", "[", 
   RowBox[{"7", " ", "x"}], "]"}]}]], "Output",
 ImageSize->{240, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->502538659]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->17803],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["TrigExpand",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/TrigExpand"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["TrigFactor",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/TrigFactor"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["TrigToExp",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/TrigToExp"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Simplify",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Simplify"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["FullSimplify",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FullSimplify"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->24466]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->9793],

Cell[TextData[ButtonBox["Putting Expressions into Different Forms",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/PuttingExpressionsIntoDifferentForms"]], "Tutorials",
 CellID->25994],

Cell[TextData[ButtonBox["Trigonometric Expressions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/TrigonometricExpressions"]], "Tutorials",
 CellID->16157]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->63029351],

Cell[TextData[ButtonBox["Algebraic Transformations",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/AlgebraicTransformations"]], "MoreAbout",
 CellID->339949304],

Cell[TextData[ButtonBox["Hyperbolic Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/HyperbolicFunctions"]], "MoreAbout",
 CellID->70328264],

Cell[TextData[ButtonBox["Trigonometric Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/TrigonometricFunctions"]], "MoreAbout",
 CellID->503139429]
}, Open  ]],

Cell["New in 3  |  Last modified in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"TrigReduce - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 27, 54.1311612}", "context" -> "System`", 
    "keywords" -> {"multiple angle formulas", "reduction"}, "index" -> True, 
    "label" -> "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "TrigReduce[expr] rewrites products and powers of trigonometric functions \
in expr in terms of trigonometric functions with combined arguments. ", 
    "synonyms" -> {"trig reduce"}, "title" -> "TrigReduce", "type" -> 
    "Symbol", "uri" -> "ref/TrigReduce"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[9694, 250, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->139105191]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 36667, 1319}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2732, 61, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3355, 88, 4194, 83, 70, "ObjectNameGrid"],
Cell[7552, 173, 648, 19, 70, "Usage",
 CellID->4550]
}, Open  ]],
Cell[CellGroupData[{
Cell[8237, 197, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[8715, 214, 223, 7, 70, "Notes",
 CellID->6941],
Cell[8941, 223, 317, 9, 70, "Notes",
 CellID->16502],
Cell[9261, 234, 396, 11, 70, "Notes",
 CellID->82730543]
}, Closed]],
Cell[CellGroupData[{
Cell[9694, 250, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->139105191],
Cell[CellGroupData[{
Cell[10079, 264, 147, 5, 70, "ExampleSection",
 CellID->47633551],
Cell[CellGroupData[{
Cell[10251, 273, 176, 6, 28, "Input",
 CellID->24764],
Cell[10430, 281, 233, 8, 36, "Output",
 CellID->31437095]
}, Open  ]],
Cell[CellGroupData[{
Cell[10700, 294, 191, 6, 28, "Input",
 CellID->591097374],
Cell[10894, 302, 289, 10, 36, "Output",
 CellID->228737111]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[11232, 318, 216, 7, 70, "ExampleSection",
 CellID->12833],
Cell[11451, 327, 229, 7, 70, "ExampleText",
 CellID->196779600],
Cell[CellGroupData[{
Cell[11705, 338, 192, 6, 70, "Input",
 CellID->32588323],
Cell[11900, 346, 288, 10, 36, "Output",
 CellID->599350]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[12237, 362, 218, 7, 70, "ExampleSection",
 CellID->20002],
Cell[CellGroupData[{
Cell[12480, 373, 224, 7, 70, "ExampleSubsection",
 CellID->19202],
Cell[12707, 382, 117, 3, 70, "ExampleText",
 CellID->9379777],
Cell[CellGroupData[{
Cell[12849, 389, 123, 4, 70, "Input",
 CellID->466802486],
Cell[12975, 395, 212, 7, 36, "Output",
 CellID->218094279]
}, Open  ]],
Cell[CellGroupData[{
Cell[13224, 407, 165, 5, 70, "Input",
 CellID->204620218],
Cell[13392, 414, 346, 12, 36, "Output",
 CellID->802163236]
}, Open  ]],
Cell[13753, 429, 85, 1, 70, "ExampleText",
 CellID->265159979],
Cell[CellGroupData[{
Cell[13863, 434, 105, 3, 70, "Input",
 CellID->2804612],
Cell[13971, 439, 2862, 83, 187, "Output",
 CellID->33316194]
}, Open  ]],
Cell[CellGroupData[{
Cell[16870, 527, 431, 14, 70, "Input",
 CellID->520713546],
Cell[17304, 543, 341, 12, 36, "Output",
 CellID->240783857]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[17706, 562, 223, 7, 70, "ExampleSection",
 CellID->13048],
Cell[17932, 571, 88, 1, 70, "ExampleText",
 CellID->71413300],
Cell[18023, 574, 359, 13, 70, "Input",
 CellID->381546912],
Cell[CellGroupData[{
Cell[18407, 591, 105, 3, 70, "Input",
 CellID->47621341],
Cell[18515, 596, 469, 17, 51, "Output",
 CellID->86321052]
}, Open  ]],
Cell[CellGroupData[{
Cell[19021, 618, 168, 6, 70, "Input",
 CellID->75991878],
Cell[19192, 626, 181, 6, 49, "Output",
 CellID->78356109]
}, Open  ]],
Cell[19388, 635, 55, 1, 70, "ExampleText",
 CellID->34644411],
Cell[CellGroupData[{
Cell[19468, 640, 235, 9, 70, "Input",
 CellID->111920163],
Cell[19706, 651, 157, 5, 36, "Output",
 CellID->96340095]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[19912, 662, 233, 7, 70, "ExampleSection",
 CellID->13506],
Cell[20148, 671, 640, 25, 70, "ExampleText",
 CellID->19589731],
Cell[CellGroupData[{
Cell[20813, 700, 161, 5, 70, "Input",
 CellID->137546738],
Cell[20977, 707, 651, 22, 36, "Output",
 CellID->482867625]
}, Open  ]],
Cell[CellGroupData[{
Cell[21665, 734, 106, 3, 70, "Input",
 CellID->934346010],
Cell[21774, 739, 211, 7, 36, "Output",
 CellID->112479127]
}, Open  ]],
Cell[22000, 749, 616, 24, 70, "ExampleText",
 CellID->788628854],
Cell[CellGroupData[{
Cell[22641, 777, 251, 8, 70, "Input",
 CellID->21117297],
Cell[22895, 787, 211, 7, 36, "Output",
 CellID->821970794]
}, Open  ]],
Cell[23121, 797, 125, 3, 70, "ExampleDelimiter",
 CellID->432339429],
Cell[23249, 802, 356, 12, 70, "ExampleText",
 CellID->710105148],
Cell[CellGroupData[{
Cell[23630, 818, 154, 5, 70, "Input",
 CellID->49485222],
Cell[23787, 825, 514, 17, 36, "Output",
 CellID->47463080]
}, Open  ]],
Cell[CellGroupData[{
Cell[24338, 847, 106, 3, 70, "Input",
 CellID->562793634],
Cell[24447, 852, 210, 7, 36, "Output",
 CellID->399666239]
}, Open  ]],
Cell[24672, 862, 124, 3, 70, "ExampleDelimiter",
 CellID->40442951],
Cell[24799, 867, 249, 7, 70, "ExampleText",
 CellID->269834246],
Cell[CellGroupData[{
Cell[25073, 878, 369, 13, 70, "Input",
 CellID->148369671],
Cell[25445, 893, 591, 20, 51, "Output",
 CellID->10209496]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[26085, 919, 226, 7, 70, "ExampleSection",
 CellID->16913],
Cell[26314, 928, 224, 8, 70, "ExampleText",
 CellID->924552947],
Cell[CellGroupData[{
Cell[26563, 940, 223, 7, 70, "Input",
 CellID->267185323],
Cell[26789, 949, 402, 7, 70, "Message",
 CellID->394626467],
Cell[27194, 958, 316, 10, 39, "Output",
 CellID->593667220]
}, Open  ]],
Cell[27525, 971, 125, 3, 70, "ExampleDelimiter",
 CellID->240475609],
Cell[27653, 976, 224, 7, 70, "ExampleText",
 CellID->103331699],
Cell[CellGroupData[{
Cell[27902, 987, 323, 12, 70, "Input",
 CellID->93796402],
Cell[28228, 1001, 406, 13, 42, "Output",
 CellID->215565831]
}, Open  ]],
Cell[28649, 1017, 240, 8, 70, "ExampleText",
 CellID->1008052319],
Cell[CellGroupData[{
Cell[28914, 1029, 138, 4, 70, "Input",
 CellID->898172079],
Cell[29055, 1035, 344, 12, 36, "Output",
 CellID->675228340]
}, Open  ]],
Cell[29414, 1050, 124, 3, 70, "ExampleDelimiter",
 CellID->26695912],
Cell[29541, 1055, 104, 2, 70, "ExampleText",
 CellID->23468404],
Cell[CellGroupData[{
Cell[29670, 1061, 161, 5, 70, "Input",
 CellID->678517677],
Cell[29834, 1068, 451, 14, 56, "Output",
 CellID->169401436]
}, Open  ]],
Cell[CellGroupData[{
Cell[30322, 1087, 214, 7, 70, "Input",
 CellID->8943582],
Cell[30539, 1096, 340, 12, 51, "Output",
 CellID->87469608]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[30928, 1114, 224, 7, 70, "ExampleSection",
 CellID->25210],
Cell[CellGroupData[{
Cell[31177, 1125, 296, 10, 70, "Input",
 CellID->387029661],
Cell[31476, 1137, 339, 11, 36, "Output",
 CellID->255286787]
}, Open  ]],
Cell[CellGroupData[{
Cell[31852, 1153, 106, 3, 70, "Input",
 CellID->614758385],
Cell[31961, 1158, 393, 13, 36, "Output",
 CellID->502538659]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[32415, 1178, 311, 9, 70, "SeeAlsoSection",
 CellID->17803],
Cell[32729, 1189, 1204, 44, 70, "SeeAlso",
 CellID->24466]
}, Open  ]],
Cell[CellGroupData[{
Cell[33970, 1238, 313, 9, 70, "TutorialsSection",
 CellID->9793],
Cell[34286, 1249, 190, 4, 70, "Tutorials",
 CellID->25994],
Cell[34479, 1255, 160, 3, 70, "Tutorials",
 CellID->16157]
}, Open  ]],
Cell[CellGroupData[{
Cell[34676, 1263, 318, 9, 70, "MoreAboutSection",
 CellID->63029351],
Cell[34997, 1274, 161, 3, 70, "MoreAbout",
 CellID->339949304],
Cell[35161, 1279, 150, 3, 70, "MoreAbout",
 CellID->70328264],
Cell[35314, 1284, 157, 3, 70, "MoreAbout",
 CellID->503139429]
}, Open  ]],
Cell[35486, 1290, 50, 0, 70, "History"],
Cell[35539, 1292, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

