(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     41576,       1481]
NotebookOptionsPosition[     34294,       1226]
NotebookOutlinePosition[     35502,       1262]
CellTagsIndexPosition[     35416,       1257]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Special Functions" :> 
          Documentation`HelpLookup["paclet:tutorial/SpecialFunctions"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Special Functions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"WhittakerW" :> 
          Documentation`HelpLookup["paclet:ref/WhittakerW"], 
          "Hypergeometric1F1" :> 
          Documentation`HelpLookup["paclet:ref/Hypergeometric1F1"], 
          "HypergeometricU" :> 
          Documentation`HelpLookup["paclet:ref/HypergeometricU"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"WhittakerW\"\>", 
       2->"\<\"Hypergeometric1F1\"\>", 
       3->"\<\"HypergeometricU\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Functions Used in Quantum Mechanics" :> 
          Documentation`HelpLookup[
           "paclet:guide/FunctionsUsedInQuantumMechanics"], 
          "Hypergeometric Functions" :> 
          Documentation`HelpLookup["paclet:guide/HypergeometricFunctions"], 
          "New in 6.0: Mathematical Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicalFunctions"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"Functions Used in Quantum Mechanics\"\>", 
       2->"\<\"Hypergeometric Functions\"\>", 
       3->"\<\"New in 6.0: Mathematical Functions\"\>", 
       4->"\<\"New in 6.0: Mathematics & Algorithms\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["WhittakerM", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["WhittakerM",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/WhittakerM"], "[", 
       RowBox[{
        StyleBox["k", "TI"], ",", 
        StyleBox["m", "TI"], ",", 
        StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the Whittaker function ",
     Cell[BoxData[
      FormBox[
       RowBox[{
        SubscriptBox["M", 
         RowBox[{"k", ",", "m"}]], "(", "z", ")"}], TraditionalForm]], 
      "InlineMath"],
     ". "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->32699]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["\<\
Mathematical function, suitable for both symbolic and numerical manipulation.\
\
\>", "Notes",
 CellID->20640],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["WhittakerM",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/WhittakerM"]], "InlineFormula"],
 " is related to the hypergeometric function by ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SubscriptBox["M", 
      RowBox[{"k", ",", "m"}]], "(", "z", ")"}], "=", 
    RowBox[{
     SuperscriptBox["e", 
      RowBox[{"-", "z", "/", "2"}]], 
     SuperscriptBox["z", 
      RowBox[{"m", "+", "1", "/", "2"}]], " ", 
     RowBox[{
      SubscriptBox["\[InvisiblePrefixScriptBase]", "1"], 
      SubscriptBox["F", "1"], "(", 
      RowBox[{
       RowBox[{"m", "-", "k", "+", 
        FractionBox["1", "2"]}], ";", 
       RowBox[{"1", "+", "2", "m"}], ";", "z"}], ")"}]}]}], TraditionalForm]],
   "InlineMath"],
 ". "
}], "Notes",
 CellID->460],

Cell[TextData[{
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["M", 
     RowBox[{"k", ",", "m"}]], "(", "z", ")"}], TraditionalForm]], 
  "InlineMath"],
 " vanishes at ",
 Cell[BoxData[
  FormBox[
   RowBox[{"z", "=", "0"}], TraditionalForm]], "InlineMath"],
 " for ",
 Cell[BoxData[
  FormBox[
   RowBox[{"m", ">", "0"}], TraditionalForm]], "InlineMath"],
 ". "
}], "Notes",
 CellID->6737],

Cell[TextData[{
 "For certain special arguments, ",
 Cell[BoxData[
  ButtonBox["WhittakerM",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/WhittakerM"]], "InlineFormula"],
 " automatically evaluates to exact values."
}], "Notes",
 CellID->22518045],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["WhittakerM",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/WhittakerM"]], "InlineFormula"],
 " can be evaluated to arbitrary numerical precision."
}], "Notes",
 CellID->31226],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["WhittakerM",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/WhittakerM"]], "InlineFormula"],
 " automatically threads over lists."
}], "Notes",
 CellID->28162],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["WhittakerM",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/WhittakerM"], "[", 
   RowBox[{
    StyleBox["k", "TI"], ",", 
    StyleBox["m", "TI"], ",", 
    StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
 " has a branch cut discontinuity in the complex ",
 Cell[BoxData[
  FormBox["z", TraditionalForm]], "InlineMath"],
 " plane running from ",
 Cell[BoxData[
  FormBox[
   RowBox[{"-", "\[Infinity]"}], TraditionalForm]], "InlineMath"],
 " to ",
 Cell[BoxData[
  FormBox["0", TraditionalForm]], "InlineMath"],
 ". "
}], "Notes",
 CellID->17881]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->37481813],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(3)", "ExampleCount"]
}], "ExampleSection",
 CellID->225764064],

Cell["Evaluate numerically:", "ExampleText",
 CellID->231069639],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"WhittakerM", "[", 
  RowBox[{"2", ",", "3", ",", "4.5"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->22333397],

Cell[BoxData["68.96703443494356`"], "Output",
 ImageSize->{46, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->601898098]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1249957],

Cell[TextData[{
 "Plot ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox[
     TagBox["M",
      WhittakerM], 
     RowBox[{"2", ",", 
      RowBox[{"1", "/", "2"}]}]], "(", "x", ")"}], TraditionalForm]], 
  "InlineMath"],
 ":"
}], "ExampleText",
 CellID->469428808],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"WhittakerM", "[", 
    RowBox[{"2", ",", 
     RowBox[{"1", "/", "2"}], ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "15"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->319759576],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 102},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->78004867]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->19586552],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["FunctionExpand",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FunctionExpand"]], "InlineFormula"],
 " to expand in terms of hypergeometric functions:"
}], "ExampleText",
 CellID->308640546],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{" ", 
  RowBox[{"FunctionExpand", "[", 
   RowBox[{"WhittakerM", "[", 
    RowBox[{"k", ",", "m", ",", "x"}], "]"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->52307413],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["\[ExponentialE]", 
   RowBox[{
    RowBox[{"-", "x"}], "/", "2"}]], " ", 
  SuperscriptBox["x", 
   RowBox[{
    FractionBox["1", "2"], "+", "m"}]], " ", 
  RowBox[{"Hypergeometric1F1", "[", 
   RowBox[{
    RowBox[{
     FractionBox["1", "2"], "-", "k", "+", "m"}], ",", 
    RowBox[{"1", "+", 
     RowBox[{"2", " ", "m"}]}], ",", "x"}], "]"}]}]], "Output",
 ImageSize->{314, 31},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->370176735]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(7)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->3982],

Cell["Evaluate for complex arguments and parameters:", "ExampleText",
 CellID->192155330],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"WhittakerM", "[", 
  RowBox[{
   RowBox[{"1", "+", "I"}], ",", "1.5", ",", 
   RowBox[{"2", "-", "I"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->130939040],

Cell[BoxData[
 RowBox[{"0.572523470925181`", "\[InvisibleSpace]", "-", 
  RowBox[{"2.4017650442097525`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{133, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->38441589]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->181032742],

Cell["Evaluate to high precision:", "ExampleText",
 CellID->29664063],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"WhittakerM", "[", 
    RowBox[{"1", ",", "2", ",", "3"}], "]"}], ",", "50"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->477389994],

Cell[BoxData["10.\
1760515204264052008367034830106989089351489824732728379575268879631`50."], \
"Output",
 ImageSize->{361, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->45077174]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->505722001],

Cell["\<\
The precision of the output tracks the precision of the input:\
\>", "ExampleText",
 CellID->121593466],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"WhittakerM", "[", 
  RowBox[{"1", ",", "2", ",", "3.00000000000000000000000"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->121305533],

Cell[BoxData["10.17605152042640520083670324278833908306`22.700922784656708"], \
"Output",
 ImageSize->{172, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->34831967]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->374030955],

Cell["Evaluate symbolically at the origin:", "ExampleText",
 CellID->11121899],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"WhittakerM", "[", 
  RowBox[{"k", ",", "3", ",", "0"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->253284775],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->861694016]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->9036255],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["WhittakerM",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/WhittakerM"]], "InlineFormula"],
 " threads element\[Hyphen]wise over lists:"
}], "ExampleText",
 CellID->221234742],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"WhittakerM", "[", 
  RowBox[{"2", ",", "3", ",", 
   RowBox[{"{", 
    RowBox[{"1.5", ",", "2.5", ",", "3.5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->772982833],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "2.7614913988531646`", ",", "13.00959997532238`", ",", 
   "34.2393848640866`"}], "}"}]], "Output",
 ImageSize->{188, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->6478970]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->312869665],

Cell["Series expansion:", "ExampleText",
 CellID->114119141],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"WhittakerM", "[", 
    RowBox[{"k", ",", "m", ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "4"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->11716619],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["x", "m"], " ", 
  RowBox[{"(", 
   InterpretationBox[
    RowBox[{
     SqrtBox["x"], "-", 
     FractionBox[
      RowBox[{"k", " ", 
       SuperscriptBox["x", 
        RowBox[{"3", "/", "2"}]]}], 
      RowBox[{"1", "+", 
       RowBox[{"2", " ", "m"}]}]], "+", 
     FractionBox[
      RowBox[{
       RowBox[{"(", 
        RowBox[{"1", "+", 
         RowBox[{"4", " ", 
          SuperscriptBox["k", "2"]}], "+", 
         RowBox[{"2", " ", "m"}]}], ")"}], " ", 
       SuperscriptBox["x", 
        RowBox[{"5", "/", "2"}]]}], 
      RowBox[{"16", " ", 
       RowBox[{"(", 
        RowBox[{"1", "+", "m"}], ")"}], " ", 
       RowBox[{"(", 
        RowBox[{"1", "+", 
         RowBox[{"2", " ", "m"}]}], ")"}]}]], "+", 
     FractionBox[
      RowBox[{
       RowBox[{"(", 
        RowBox[{
         RowBox[{
          RowBox[{"-", "5"}], " ", "k"}], "-", 
         RowBox[{"4", " ", 
          SuperscriptBox["k", "3"]}], "-", 
         RowBox[{"6", " ", "k", " ", "m"}]}], ")"}], " ", 
       SuperscriptBox["x", 
        RowBox[{"7", "/", "2"}]]}], 
      RowBox[{"48", " ", 
       RowBox[{"(", 
        RowBox[{"1", "+", "m"}], ")"}], " ", 
       RowBox[{"(", 
        RowBox[{"1", "+", 
         RowBox[{"2", " ", "m"}]}], ")"}], " ", 
       RowBox[{"(", 
        RowBox[{"3", "+", 
         RowBox[{"2", " ", "m"}]}], ")"}]}]], "+", 
     InterpretationBox[
      SuperscriptBox[
       RowBox[{"O", "[", "x", "]"}], 
       RowBox[{"9", "/", "2"}]],
      SeriesData[$CellContext`x, 0, {}, 1, 9, 2],
      Editable->False]}],
    SeriesData[$CellContext`x, 0, {
     1, 0, -$CellContext`k/(1 + 2 $CellContext`m), 0, 
      Rational[1, 16] (1 + $CellContext`m)^(-1) (1 + 2 $CellContext`m)^(-1) (
       1 + 4 $CellContext`k^2 + 2 $CellContext`m), 0, 
      Rational[1, 48] (1 + $CellContext`m)^(-1) (1 + 2 $CellContext`m)^(-1) (
        3 + 2 $CellContext`m)^(-1) ((-5) $CellContext`k - 4 $CellContext`k^3 - 
       6 $CellContext`k $CellContext`m)}, 1, 9, 2],
    Editable->False], ")"}]}]], "Output",
 ImageSize->{478, 40},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->8939783]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->410864798],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TraditionalForm",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TraditionalForm"]], "InlineFormula"],
 " formatting:"
}], "ExampleText",
 CellID->237226052],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"WhittakerM", "[", 
   RowBox[{"m", ",", "k", ",", "x"}], "]"}], "//", 
  "TraditionalForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->658442963],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox[
    TagBox["M",
     WhittakerM], 
    RowBox[{"m", ",", "k"}]], "(", "x", ")"}], TraditionalForm]], "Output",
 ImageSize->{42, 17},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//TraditionalForm=",
 CellID->114843525]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->610],

Cell["Bound state Coulomb eigenfunction in parabolic coordinates:", \
"ExampleText",
 CellID->18836974],

Cell[BoxData[
 RowBox[{
  RowBox[{
   SubscriptBox["\[Psi]", "par"], "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"n1_", ",", "n2_", ",", "m_"}], "}"}], ",", "\[Zeta]_", ",", 
    "\[Eta]_", ",", "\[Phi]_"}], "]"}], ":=", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"mm", "=", 
       RowBox[{"Abs", "[", "m", "]"}]}], ",", 
      RowBox[{"n", "=", 
       RowBox[{"n1", "+", "n2", "+", 
        RowBox[{"Abs", "[", "m", "]"}], "+", "1"}]}]}], "}"}], ",", 
    RowBox[{
     FractionBox[
      RowBox[{"Exp", "[", 
       RowBox[{"\[ImaginaryI]", " ", "m", " ", "\[Phi]"}], "]"}], 
      RowBox[{
       SqrtBox["\[Pi]"], 
       SqrtBox[
        RowBox[{"\[Zeta]", " ", "\[Eta]"}]], "n"}]], 
     SqrtBox[
      RowBox[{
       FractionBox[
        RowBox[{
         RowBox[{"(", 
          RowBox[{"n1", "+", "mm"}], ")"}], "!"}], 
        RowBox[{"n1", "!"}]], 
       FractionBox[
        RowBox[{
         RowBox[{"(", 
          RowBox[{"n2", "+", "mm"}], ")"}], "!"}], 
        RowBox[{"n2", "!"}]]}]], 
     RowBox[{"WhittakerM", "[", 
      RowBox[{
       RowBox[{"n1", "+", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{"mm", "+", "1"}], ")"}], "/", "2"}]}], ",", 
       RowBox[{"mm", "/", "2"}], ",", 
       RowBox[{
        FractionBox["1", "n"], " ", "\[Eta]"}]}], "]"}], 
     RowBox[{"WhittakerM", "[", 
      RowBox[{
       RowBox[{"n2", "+", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{"mm", "+", "1"}], ")"}], "/", "2"}]}], ",", 
       RowBox[{"mm", "/", "2"}], ",", 
       RowBox[{
        FractionBox["1", "n"], " ", "\[Zeta]"}]}], "]"}]}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->95832167],

Cell["\<\
Decompose the eigenfunction in terms of spherical eigenfunctions:\
\>", "ExampleText",
 CellID->388729592],

Cell[BoxData[
 RowBox[{
  RowBox[{
   SubscriptBox["\[Psi]", "sph"], "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"n_", ",", "l_", ",", "m_"}], "}"}], ",", "r_", ",", "\[Theta]_",
     ",", "\[Phi]_"}], "]"}], ":=", 
  RowBox[{
   SqrtBox[
    FractionBox[
     RowBox[{" ", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{"n", "-", "l", "-", "1"}], ")"}], "!"}]}], 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"n", "+", "l"}], ")"}], "!"}]]], " ", 
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"-", 
     FractionBox["r", "n"]}]], " ", 
   SuperscriptBox[
    RowBox[{"(", 
     FractionBox[
      RowBox[{"2", "r"}], "n"], ")"}], "l"], 
   FractionBox["2", 
    SuperscriptBox["n", "2"]], " ", 
   RowBox[{"LaguerreL", "[", 
    RowBox[{
     RowBox[{"n", "-", "l", "-", "1"}], ",", " ", 
     RowBox[{
      RowBox[{"2", "l"}], "+", "1"}], ",", 
     FractionBox[
      RowBox[{"2", "r"}], "n"]}], "]"}], 
   RowBox[{"SphericalHarmonicY", "[", 
    RowBox[{"l", ",", "m", ",", "\[Theta]", ",", "\[Phi]"}], 
    "]"}]}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->383683020],

Cell[TextData[{
 "Parabolic coordinates relate to radial coordinates as ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Zeta]", "=", 
    RowBox[{"r", " ", 
     RowBox[{"(", 
      RowBox[{"1", "+", 
       RowBox[{"cos", "(", "\[Theta]", ")"}]}], ")"}]}]}], TraditionalForm]], 
  "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Eta]", "=", 
    RowBox[{"r", " ", 
     RowBox[{"(", 
      RowBox[{"1", "-", 
       RowBox[{"cos", "(", "\[Theta]", ")"}]}], ")"}]}]}], TraditionalForm]], 
  "InlineMath"],
 ":"
}], "ExampleText",
 CellID->59479238],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"FunctionExpand", "[", 
   RowBox[{
    SubscriptBox["\[Psi]", "par"], "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "1", ",", 
       RowBox[{"-", "1"}]}], "}"}], ",", 
     RowBox[{"r", 
      RowBox[{"(", 
       RowBox[{"1", "+", 
        RowBox[{"Cos", "[", "\[Theta]", "]"}]}], ")"}]}], ",", 
     RowBox[{"r", 
      RowBox[{"(", 
       RowBox[{"1", "-", 
        RowBox[{"Cos", "[", "\[Theta]", "]"}]}], ")"}]}], ",", "\[Phi]"}], 
    "]"}], "]"}], "//", 
  RowBox[{
   RowBox[{"FullSimplify", "[", 
    RowBox[{"#", ",", 
     RowBox[{
      RowBox[{"0", "<=", "\[Theta]", "<", "\[Pi]"}], " ", "&&", " ", 
      RowBox[{"r", "\[GreaterEqual]", "0"}]}]}], "]"}], "&"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->275214311],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{
     RowBox[{"-", 
      FractionBox["r", "4"]}], "-", 
     RowBox[{"\[ImaginaryI]", " ", "\[Phi]"}]}]], " ", "r", " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], "+", 
     RowBox[{"Cos", "[", "\[Theta]", "]"}]}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{"1", "+", 
     RowBox[{"Cos", "[", "\[Theta]", "]"}]}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{"8", "-", "r", "+", 
     RowBox[{"r", " ", 
      RowBox[{"Cos", "[", "\[Theta]", "]"}]}]}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "8"}], "+", "r", "+", 
     RowBox[{"r", " ", 
      RowBox[{"Cos", "[", "\[Theta]", "]"}]}]}], ")"}], " ", 
   RowBox[{"Csc", "[", "\[Theta]", "]"}]}], 
  RowBox[{"2048", " ", 
   SqrtBox["\[Pi]"]}]]], "Output",
 ImageSize->{490, 41},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->52074424]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Simplify", "[", 
  RowBox[{"%", "+", 
   RowBox[{
    FractionBox["3", 
     SqrtBox["15"]], " ", 
    RowBox[{
     SubscriptBox["\[Psi]", "sph"], "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"4", ",", "3", ",", 
        RowBox[{"-", "1"}]}], "}"}], ",", "r", ",", "\[Theta]", ",", 
      "\[Phi]"}], "]"}]}], "-", 
   RowBox[{
    FractionBox["2", 
     SqrtBox["10"]], 
    RowBox[{
     SubscriptBox["\[Psi]", "sph"], "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"4", ",", "1", ",", 
        RowBox[{"-", "1"}]}], "}"}], ",", "r", ",", "\[Theta]", ",", 
      "\[Phi]"}], "]"}]}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->203396454],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->15193325]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->29538],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["FunctionExpand",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FunctionExpand"]], "InlineFormula"],
 " to expand ",
 Cell[BoxData[
  ButtonBox["WhittakerM",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/WhittakerM"]], "InlineFormula"],
 " into other functions:"
}], "ExampleText",
 CellID->156298838],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FunctionExpand", "[", 
  RowBox[{"WhittakerM", "[", 
   RowBox[{"5", ",", 
    RowBox[{"1", "/", "2"}], ",", "x"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->42482196],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["\[ExponentialE]", 
   RowBox[{
    RowBox[{"-", "x"}], "/", "2"}]], " ", "x", " ", 
  RowBox[{"(", 
   RowBox[{"1", "-", 
    RowBox[{"2", " ", "x"}], "+", 
    SuperscriptBox["x", "2"], "-", 
    FractionBox[
     SuperscriptBox["x", "3"], "6"], "+", 
    FractionBox[
     SuperscriptBox["x", "4"], "120"]}], ")"}]}]], "Output",
 ImageSize->{188, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->37741644]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FunctionExpand", "[", 
  RowBox[{"WhittakerM", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"-", "1"}], "/", "4"}], ",", 
    RowBox[{"1", "/", "4"}], ",", "x"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->417632044],

Cell[BoxData[
 RowBox[{
  FractionBox["1", "2"], " ", 
  SuperscriptBox["\[ExponentialE]", 
   RowBox[{"x", "/", "2"}]], " ", 
  SqrtBox["\[Pi]"], " ", 
  SuperscriptBox["x", 
   RowBox[{"1", "/", "4"}]], " ", 
  RowBox[{"Erf", "[", 
   SqrtBox["x"], "]"}]}]], "Output",
 ImageSize->{156, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->74977671]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FunctionExpand", "[", 
  RowBox[{"WhittakerM", "[", 
   RowBox[{"0", ",", "m", ",", "x"}], "]"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->285678544],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["2", 
   RowBox[{"2", " ", "m"}]], " ", 
  SqrtBox["x"], " ", 
  RowBox[{"BesselI", "[", 
   RowBox[{"m", ",", 
    FractionBox["x", "2"]}], "]"}], " ", 
  RowBox[{"Gamma", "[", 
   RowBox[{"1", "+", "m"}], "]"}]}]], "Output",
 ImageSize->{226, 28},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->739710008]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->30204456],

Cell["Integrate expressions involving Whittaker functions:", "ExampleText",
 CellID->696244865],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Exp", "[", 
     RowBox[{"-", "x"}], "]"}], 
    RowBox[{"WhittakerM", "[", 
     RowBox[{"3", ",", "2", ",", "x"}], "]"}], 
    RowBox[{"WhittakerW", "[", 
     RowBox[{"1", ",", 
      RowBox[{"-", "2"}], ",", "x"}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->305925733],

Cell[BoxData[
 FractionBox["128", 
  RowBox[{"21", " ", 
   SqrtBox["\[Pi]"]}]]], "Output",
 ImageSize->{50, 34},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->30745694]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->2012],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["WhittakerW",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/WhittakerW"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Hypergeometric1F1",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Hypergeometric1F1"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["HypergeometricU",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/HypergeometricU"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->1255]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->31406],

Cell[TextData[ButtonBox["Special Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/SpecialFunctions"]], "Tutorials",
 CellID->12755]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->23010574],

Cell[TextData[ButtonBox["Functions Used in Quantum Mechanics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/FunctionsUsedInQuantumMechanics"]], "MoreAbout",
 CellID->109050834],

Cell[TextData[ButtonBox["Hypergeometric Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/HypergeometricFunctions"]], "MoreAbout",
 CellID->356123203],

Cell[TextData[ButtonBox["New in 6.0: Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicalFunctions"]], "MoreAbout",
 CellID->153686154],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->209050681]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"WhittakerM - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 29, 36.3994042}", "context" -> "System`", 
    "keywords" -> {}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "WhittakerM[k, m, z] gives the Whittaker function M k, m (z). ", 
    "synonyms" -> {"Whittaker M"}, "title" -> "WhittakerM", "type" -> 
    "Symbol", "uri" -> "ref/WhittakerM"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[8755, 286, 359, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->37481813]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 35274, 1250}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2636, 64, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3259, 91, 1575, 37, 70, "ObjectNameGrid"],
Cell[4837, 130, 745, 25, 70, "Usage",
 CellID->32699]
}, Open  ]],
Cell[CellGroupData[{
Cell[5619, 160, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6097, 177, 120, 4, 70, "Notes",
 CellID->20640],
Cell[6220, 183, 798, 27, 70, "Notes",
 CellID->460],
Cell[7021, 212, 399, 17, 70, "Notes",
 CellID->6737],
Cell[7423, 231, 251, 8, 70, "Notes",
 CellID->22518045],
Cell[7677, 241, 222, 7, 70, "Notes",
 CellID->31226],
Cell[7902, 250, 205, 7, 70, "Notes",
 CellID->28162],
Cell[8110, 259, 608, 22, 70, "Notes",
 CellID->17881]
}, Closed]],
Cell[CellGroupData[{
Cell[8755, 286, 359, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->37481813],
Cell[CellGroupData[{
Cell[9139, 300, 148, 5, 70, "ExampleSection",
 CellID->225764064],
Cell[9290, 307, 64, 1, 70, "ExampleText",
 CellID->231069639],
Cell[CellGroupData[{
Cell[9379, 312, 140, 4, 28, "Input",
 CellID->22333397],
Cell[9522, 318, 175, 5, 36, "Output",
 CellID->601898098]
}, Open  ]],
Cell[9712, 326, 123, 3, 70, "ExampleDelimiter",
 CellID->1249957],
Cell[9838, 331, 274, 13, 70, "ExampleText",
 CellID->469428808],
Cell[CellGroupData[{
Cell[10137, 348, 282, 9, 28, "Input",
 CellID->319759576],
Cell[10422, 359, 2721, 49, 123, "Output",
 Evaluatable->False,
 CellID->78004867]
}, Open  ]],
Cell[13158, 411, 124, 3, 70, "ExampleDelimiter",
 CellID->19586552],
Cell[13285, 416, 246, 8, 70, "ExampleText",
 CellID->308640546],
Cell[CellGroupData[{
Cell[13556, 428, 200, 6, 70, "Input",
 CellID->52307413],
Cell[13759, 436, 532, 18, 52, "Output",
 CellID->370176735]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[14340, 460, 215, 7, 70, "ExampleSection",
 CellID->3982],
Cell[14558, 469, 89, 1, 70, "ExampleText",
 CellID->192155330],
Cell[CellGroupData[{
Cell[14672, 474, 189, 6, 70, "Input",
 CellID->130939040],
Cell[14864, 482, 271, 7, 36, "Output",
 CellID->38441589]
}, Open  ]],
Cell[15150, 492, 125, 3, 70, "ExampleDelimiter",
 CellID->181032742],
Cell[15278, 497, 69, 1, 70, "ExampleText",
 CellID->29664063],
Cell[CellGroupData[{
Cell[15372, 502, 194, 6, 70, "Input",
 CellID->477389994],
Cell[15569, 510, 235, 7, 36, "Output",
 CellID->45077174]
}, Open  ]],
Cell[15819, 520, 125, 3, 70, "ExampleDelimiter",
 CellID->505722001],
Cell[15947, 525, 113, 3, 70, "ExampleText",
 CellID->121593466],
Cell[CellGroupData[{
Cell[16085, 532, 163, 4, 70, "Input",
 CellID->121305533],
Cell[16251, 538, 219, 6, 36, "Output",
 CellID->34831967]
}, Open  ]],
Cell[16485, 547, 125, 3, 70, "ExampleDelimiter",
 CellID->374030955],
Cell[16613, 552, 78, 1, 70, "ExampleText",
 CellID->11121899],
Cell[CellGroupData[{
Cell[16716, 557, 139, 4, 70, "Input",
 CellID->253284775],
Cell[16858, 563, 158, 5, 36, "Output",
 CellID->861694016]
}, Open  ]],
Cell[17031, 571, 123, 3, 70, "ExampleDelimiter",
 CellID->9036255],
Cell[17157, 576, 222, 7, 70, "ExampleText",
 CellID->221234742],
Cell[CellGroupData[{
Cell[17404, 587, 204, 6, 70, "Input",
 CellID->772982833],
Cell[17611, 595, 270, 9, 36, "Output",
 CellID->6478970]
}, Open  ]],
Cell[17896, 607, 125, 3, 70, "ExampleDelimiter",
 CellID->312869665],
Cell[18024, 612, 60, 1, 70, "ExampleText",
 CellID->114119141],
Cell[CellGroupData[{
Cell[18109, 617, 256, 8, 70, "Input",
 CellID->11716619],
Cell[18368, 627, 2188, 66, 61, "Output",
 CellID->8939783]
}, Open  ]],
Cell[20571, 696, 125, 3, 70, "ExampleDelimiter",
 CellID->410864798],
Cell[20699, 701, 203, 7, 70, "ExampleText",
 CellID->237226052],
Cell[CellGroupData[{
Cell[20927, 712, 181, 6, 70, "Input",
 CellID->658442963],
Cell[21111, 720, 309, 11, 52, "Output",
 CellID->114843525]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[21469, 737, 221, 7, 70, "ExampleSection",
 CellID->610],
Cell[21693, 746, 103, 2, 70, "ExampleText",
 CellID->18836974],
Cell[21799, 750, 1706, 57, 70, "Input",
 CellID->95832167],
Cell[23508, 809, 116, 3, 70, "ExampleText",
 CellID->388729592],
Cell[23627, 814, 1096, 38, 70, "Input",
 CellID->383683020],
Cell[24726, 854, 560, 21, 70, "ExampleText",
 CellID->59479238],
Cell[CellGroupData[{
Cell[25311, 879, 783, 25, 70, "Input",
 CellID->275214311],
Cell[26097, 906, 957, 31, 62, "Output",
 CellID->52074424]
}, Open  ]],
Cell[CellGroupData[{
Cell[27091, 942, 690, 24, 70, "Input",
 CellID->203396454],
Cell[27784, 968, 157, 5, 36, "Output",
 CellID->15193325]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[27990, 979, 233, 7, 70, "ExampleSection",
 CellID->29538],
Cell[28226, 988, 359, 13, 70, "ExampleText",
 CellID->156298838],
Cell[CellGroupData[{
Cell[28610, 1005, 205, 6, 70, "Input",
 CellID->42482196],
Cell[28818, 1013, 502, 17, 54, "Output",
 CellID->37741644]
}, Open  ]],
Cell[CellGroupData[{
Cell[29357, 1035, 252, 8, 70, "Input",
 CellID->417632044],
Cell[29612, 1045, 400, 14, 51, "Output",
 CellID->74977671]
}, Open  ]],
Cell[CellGroupData[{
Cell[30049, 1064, 181, 5, 70, "Input",
 CellID->285678544],
Cell[30233, 1071, 398, 14, 49, "Output",
 CellID->739710008]
}, Open  ]],
Cell[30646, 1088, 124, 3, 70, "ExampleDelimiter",
 CellID->30204456],
Cell[30773, 1093, 95, 1, 70, "ExampleText",
 CellID->696244865],
Cell[CellGroupData[{
Cell[30893, 1098, 446, 14, 70, "Input",
 CellID->305925733],
Cell[31342, 1114, 220, 8, 55, "Output",
 CellID->30745694]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[31623, 1129, 310, 9, 70, "SeeAlsoSection",
 CellID->2012],
Cell[31936, 1140, 731, 26, 70, "SeeAlso",
 CellID->1255]
}, Open  ]],
Cell[CellGroupData[{
Cell[32704, 1171, 314, 9, 70, "TutorialsSection",
 CellID->31406],
Cell[33021, 1182, 144, 3, 70, "Tutorials",
 CellID->12755]
}, Open  ]],
Cell[CellGroupData[{
Cell[33202, 1190, 318, 9, 70, "MoreAboutSection",
 CellID->23010574],
Cell[33523, 1201, 178, 3, 70, "MoreAbout",
 CellID->109050834],
Cell[33704, 1206, 159, 3, 70, "MoreAbout",
 CellID->356123203],
Cell[33866, 1211, 174, 3, 70, "MoreAbout",
 CellID->153686154],
Cell[34043, 1216, 179, 3, 70, "MoreAbout",
 CellID->209050681]
}, Open  ]],
Cell[34237, 1222, 27, 0, 70, "History"],
Cell[34267, 1224, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

