(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     49455,       1532]
NotebookOptionsPosition[     41693,       1270]
NotebookOutlinePosition[     43321,       1312]
CellTagsIndexPosition[     43235,       1307]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Special Functions" :> 
          Documentation`HelpLookup["paclet:tutorial/SpecialFunctions"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Special Functions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Zeta" :> Documentation`HelpLookup["paclet:ref/Zeta"], 
          "RiemannSiegelZ" :> 
          Documentation`HelpLookup["paclet:ref/RiemannSiegelZ"], "PrimePi" :> 
          Documentation`HelpLookup["paclet:ref/PrimePi"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Zeta\"\>", 2->"\<\"RiemannSiegelZ\"\>", 
       3->"\<\"PrimePi\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Inverse Functions" :> 
          Documentation`HelpLookup["paclet:guide/InverseFunctions"], 
          "Number Theoretic Functions" :> 
          Documentation`HelpLookup["paclet:guide/NumberTheoreticFunctions"], 
          "Number Theory" :> 
          Documentation`HelpLookup["paclet:guide/NumberTheory"], 
          "Prime Numbers" :> 
          Documentation`HelpLookup["paclet:guide/PrimeNumbers"], 
          "Special Functions" :> 
          Documentation`HelpLookup["paclet:guide/SpecialFunctions"], 
          "Zeta Functions & Polylogarithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/ZetaFunctionsAndPolylogarithms"], 
          "New in 6.0: Mathematical Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicalFunctions"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"], 
          "New in 6.0: Number Theory & Integer Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60NumberTheoryAndIntegerFunctions"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Inverse Functions\"\>", 
       2->"\<\"Number Theoretic Functions\"\>", 3->"\<\"Number Theory\"\>", 
       4->"\<\"Prime Numbers\"\>", 5->"\<\"Special Functions\"\>", 
       6->"\<\"Zeta Functions & Polylogarithms\"\>", 
       7->"\<\"New in 6.0: Mathematical Functions\"\>", 
       8->"\<\"New in 6.0: Mathematics & Algorithms\"\>", 
       9->"\<\"New in 6.0: Number Theory & Integer Functions\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["ZetaZero", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["ZetaZero",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/ZetaZero"], "[", 
       StyleBox["k", "TI"], "]"}]], "InlineFormula"],
     " \[LineSeparator]represents the ",
     Cell[BoxData[
      StyleBox["k", "TI"]], "InlineFormula"],
     Cell[BoxData[
      FormBox[
       SuperscriptBox["\[Null]", "th"], TraditionalForm]]],
     " zero of the Riemann zeta function on the critical line."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["ZetaZero",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/ZetaZero"], "[", 
       RowBox[{
        StyleBox["k", "TI"], ",", 
        StyleBox["t", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]represents the ",
     Cell[BoxData[
      StyleBox["k", "TI"]], "InlineFormula"],
     Cell[BoxData[
      FormBox[
       SuperscriptBox["\[Null]", "th"], TraditionalForm]]],
     " zero with imaginary part greater than ",
     Cell[BoxData[
      FormBox["t", TraditionalForm]], "InlineMath"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["\<\
Mathematical function, suitable for both symbolic and numerical manipulation.\
\
\>", "Notes",
 CellID->20640],

Cell[TextData[{
 "For positive ",
 Cell[BoxData[
  StyleBox["k", "TI"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["ZetaZero",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ZetaZero"], "[", 
   StyleBox["k", "TI"], "]"}]], "InlineFormula"],
 " represents the zero of ",
 Cell[BoxData[
  FormBox[
   TagBox[
    RowBox[{"\[Zeta]", "(", 
     TagBox["s",
      Zeta,
      Editable->True,
      Selectable->True], ")"}],
    InterpretTemplate[
     Function[BoxForm`e$, 
      Zeta[BoxForm`e$]]],
    Editable->False,
    Selectable->False], TraditionalForm]], "InlineMath"],
 " on the critical line ",
 Cell[BoxData[
  FormBox[
   RowBox[{"s", "=", 
    RowBox[{
     FractionBox["1", "2"], "+", 
     RowBox[{"i", " ", "t"}]}]}], TraditionalForm]], "InlineMath"],
 " that has the ",
 Cell[BoxData[
  StyleBox["k", "TI"]], "InlineFormula"],
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Null]", "th"], TraditionalForm]]],
 " smallest positive imaginary part. "
}], "Notes",
 CellID->8759937],

Cell[TextData[{
 "For negative ",
 Cell[BoxData[
  StyleBox["k", "TI"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["ZetaZero",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ZetaZero"], "[", 
   StyleBox["k", "TI"], "]"}]], "InlineFormula"],
 " represents zeros with progressively larger negative imaginary parts."
}], "Notes",
 CellID->47592934],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["N",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/N"], "[", 
   RowBox[{"ZetaZero", "[", 
    StyleBox["k", "TI"], "]"}], "]"}]], "InlineFormula"],
 " gives a numerical approximation to the specified zero."
}], "Notes",
 CellID->43381924],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["ZetaZero",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ZetaZero"]], "InlineFormula"],
 " can be evaluated to arbitrary numerical precision."
}], "Notes",
 CellID->31226],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["ZetaZero",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ZetaZero"]], "InlineFormula"],
 " automatically threads over lists."
}], "Notes",
 CellID->28162]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->712695132],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->24628098],

Cell["Find numerically the position of the first zero:", "ExampleText",
 CellID->395660164],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{"ZetaZero", "[", "1", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->9495698],

Cell[BoxData[
 RowBox[{"0.5`", "\[InvisibleSpace]", "+", 
  RowBox[{"14.134725141734695`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{98, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->151140807]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->98201576],

Cell["Symbolic property:", "ExampleText",
 CellID->200274704],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Zeta", "[", 
  RowBox[{"ZetaZero", "[", "10", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->34670781],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->413582641]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1293636265],

Cell["Evaluate to high precision:", "ExampleText",
 CellID->456095757],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"ZetaZero", "[", "10", "]"}], ",", "50"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19960652],

Cell[BoxData[
 RowBox[{"0.5`48.152462008992615", "+", 
  RowBox[{
  "49.7738324776723021819167846785637240577231782996766620702079234947238`50.\
15049308642583", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{379, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->154497032]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->213745782],

Cell["\<\
Find the first zero with the imaginary part greater than 1000:\
\>", "ExampleText",
 CellID->294298831],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"ZetaZero", "[", 
    RowBox[{"1", ",", "1000"}], "]"}], ",", "50"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->357535962],

Cell[BoxData[
 RowBox[{"0.5`46.84889927025658", "+", 
  RowBox[{
  "1001.3494826377827371221033096531061705982745012854911253226929`50.\
150514943691405", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{386, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->274974575]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->204489201],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["ZetaZero",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ZetaZero"]], "InlineFormula"],
 " threads element-wise over lists:"
}], "ExampleText",
 CellID->567006562],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"ZetaZero", "[", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "3", ",", "4", ",", "5"}], "}"}], "]"}], "//",
   "N"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->91334672],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"0.5`", "\[InvisibleSpace]", "+", 
    RowBox[{"14.134725141734695`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{"0.5`", "\[InvisibleSpace]", "+", 
    RowBox[{"21.022039638771556`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{"0.5`", "\[InvisibleSpace]", "+", 
    RowBox[{"25.01085758014569`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{"0.5`", "\[InvisibleSpace]", "+", 
    RowBox[{"30.424876125859512`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{"0.5`", "\[InvisibleSpace]", "+", 
    RowBox[{"32.93506158773919`", " ", "\[ImaginaryI]"}]}]}], "}"}]], "Output",\

 ImageSize->{524, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->180245167]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1020263627],

Cell[TextData[{
 "Negative order is interpreted as a reflected root of the ",
 Cell[BoxData[
  ButtonBox["Zeta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Zeta"]], "InlineFormula"],
 " function:"
}], "ExampleText",
 CellID->430122750],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"ZetaZero", "[", 
    RowBox[{"-", "1"}], "]"}], "==", 
   RowBox[{"Conjugate", "[", 
    RowBox[{"ZetaZero", "[", "1", "]"}], "]"}]}], "//", "N"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->7341359],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->85319416]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"ZetaZero", "[", 
    RowBox[{
     RowBox[{"-", "3"}], ",", "20"}], "]"}], ",", "50"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->215553609],

Cell[BoxData[
 RowBox[{"0.5`48.526687344889034", "-", 
  RowBox[{
  "21.0220396387715549926284795938969027773343405249027817556817143133372`50.\
15039219120292", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{379, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->211283976]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"ZetaZero", "[", 
    RowBox[{"2", ",", 
     RowBox[{"-", "20"}]}], "]"}], ",", "50"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->409630597],

Cell[BoxData[
 RowBox[{"0.5`48.69892608550151", "+", 
  RowBox[{
  "14.1347251417346937904572519835624702707842571156992431854760153695438`50.\
1502434489547", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{379, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->360518618]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->258228157],

Cell["Plot distances between successive zeros:", "ExampleText",
 CellID->560227542],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", 
  RowBox[{"Differences", "[", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"Im", "[", 
      RowBox[{"ZetaZero", "[", "n", "]"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"n", ",", "50"}], "}"}]}], "]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->318207300],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], 
    LineBox[{{1., 6.887314497036863}, {2., 3.9888179413741334`}, {3., 
     5.414018545713823}, {4., 2.5101854618796793`}, {5., 4.651116571086483}, {
     6., 3.3325408533218237`}, {7., 2.4083542687675035`}, {8., 
     4.678077600252152}, {9., 1.7686815965051466`}, {10., 
     3.196489000042156}, {11., 3.475926219348935}, {12., 2.900796305538961}, {
     13., 1.4847345220074573`}, {14., 4.280765523471793}, {15., 
     1.9672664814125653`}, {16., 2.466591181679803}, {17., 
     2.520755963307934}, {18., 3.637533024602021}, {19., 
     1.4401493697908592`}, {20., 2.192534951374597}, {21., 
     3.573005833836646}, {22., 1.8251121264310228`}, {23., 
     2.689781632608188}, {24., 1.3838365945092335`}, {25., 
     3.6827880629240184`}, {26., 2.1594447699613823`}, {27., 
     1.2192901877254343`}, {28., 2.9605599899483934`}, {29., 
     2.4866567875376973`}, {30., 2.4076870347469566`}, {31., 
     1.721085011847748}, {32., 1.7219881319502974`}, {33., 
     3.8609243588932713`}, {34., 0.8451236338229791}, {35., 
     2.4455617384600856`}, {36., 1.9064594054048172`}, {37., 
     2.564102545118658}, {38., 2.5793421364444242`}, {39., 
     1.5767042911319606`}, {40., 1.3099892607931736`}, {41., 
     3.259865325250715}, {42., 2.0620203203595793`}, {43., 
     1.5089843309766309`}, {44., 2.410048672064903}, {45., 
     1.2587725503762783`}, {46., 3.359532301159561}, {47., 
     1.62016689758795}, {48., 1.3874984518997167`}, {49., 
     1.988138403599521}}]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 1.},
  PlotRange->{{0., 49.}, {0.8451236338229791, 6.887314497036863}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 122},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->610536658]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->514763383],

Cell["Visualize the first ten zeros:", "ExampleText",
 CellID->226328844],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"RiemannSiegelZ", "[", "t", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "50"}], "}"}], ",", 
   RowBox[{"Epilog", "->", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"PointSize", "[", ".02", "]"}], ",", "Red", ",", 
      RowBox[{"Point", "[", 
       RowBox[{"Table", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{"Im", "[", 
            RowBox[{"ZetaZero", "[", "n", "]"}], "]"}], ",", "0"}], "}"}], 
         ",", 
         RowBox[{"{", 
          RowBox[{"n", ",", "10"}], "}"}]}], "]"}], "]"}]}], "}"}]}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->5035842],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 108},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->160511189]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->336816641],

Cell[TextData[{
 "Compute ",
 ButtonBox["Gram points",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://mathworld.wolfram.com/GramPoint.html"], None}],
 ":"
}], "ExampleText",
 CellID->1546359],

Cell[BoxData[
 RowBox[{
  RowBox[{"gp", "[", "k_", "]"}], ":=", 
  RowBox[{"Block", "[", 
   RowBox[{
    RowBox[{"{", "t", "}"}], ",", 
    RowBox[{"t", "/.", 
     RowBox[{"FindRoot", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"RiemannSiegelTheta", "[", "t", "]"}], "-", 
        RowBox[{"Pi", " ", "k"}]}], ",", 
       RowBox[{"{", 
        RowBox[{"t", ",", 
         RowBox[{
          RowBox[{"(", 
           RowBox[{
            RowBox[{"1", "/", "4"}], "+", 
            RowBox[{"2", " ", "k"}]}], ")"}], " ", 
          RowBox[{"Pi", "/", 
           RowBox[{"ProductLog", "[", 
            RowBox[{
             RowBox[{"(", 
              RowBox[{
               RowBox[{"1", "/", "8"}], "+", "k"}], ")"}], "/", "E"}], 
            "]"}]}]}]}], "}"}]}], "]"}]}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->613438328],

Cell[TextData[{
 "Show good Gram points, where ",
 Cell[BoxData[
  ButtonBox["RiemannSiegelZ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RiemannSiegelZ"]], "InlineFormula"],
 " changes sign for consecutive points:"
}], "ExampleText",
 CellID->156674211],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"RiemannSiegelZ", "[", "t", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "40"}], "}"}], ",", 
   RowBox[{"Epilog", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"PointSize", "[", "0.03", "]"}], ",", "Red", ",", 
      RowBox[{"Point", "[", 
       RowBox[{"Table", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{"gp", "[", "k", "]"}], ",", 
           RowBox[{"RiemannSiegelZ", "[", 
            RowBox[{"1.0", " ", 
             RowBox[{"gp", "[", "k", "]"}]}], "]"}]}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"k", ",", 
           RowBox[{"-", "1"}], ",", "6"}], "}"}]}], "]"}], "]"}], ",", 
      "Black", ",", 
      RowBox[{"Point", "[", 
       RowBox[{"Table", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{"Im", "[", 
            RowBox[{"ZetaZero", "[", "k", "]"}], "]"}], ",", "0"}], "}"}], 
         ",", 
         RowBox[{"{", 
          RowBox[{"k", ",", "1", ",", "6"}], "}"}]}], "]"}], "]"}]}], 
     "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->159908230],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 109},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->237555958]
}, Open  ]],

Cell["Show a bad Gram point:", "ExampleText",
 CellID->26042395],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"RiemannSiegelZ", "[", "t", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", 
     RowBox[{"gp", "[", "125", "]"}], ",", 
     RowBox[{"gp", "[", "127", "]"}]}], "}"}], ",", 
   RowBox[{"Epilog", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"PointSize", "[", "0.03", "]"}], ",", "Black", ",", 
      RowBox[{"Point", "[", 
       RowBox[{"Table", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{"Im", "[", 
            RowBox[{"ZetaZero", "[", "k", "]"}], "]"}], ",", "0"}], "}"}], 
         ",", 
         RowBox[{"{", 
          RowBox[{"k", ",", "126", ",", "128"}], "}"}]}], "]"}], "]"}], ",", 
      "Red", ",", 
      RowBox[{"Point", "[", 
       RowBox[{"Table", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{"gp", "[", "k", "]"}], ",", 
           RowBox[{"RiemannSiegelZ", "[", 
            RowBox[{"1.0", " ", 
             RowBox[{"gp", "[", "k", "]"}]}], "]"}]}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"k", ",", "125", ",", "127"}], "}"}]}], "]"}], "]"}]}], 
     "}"}]}], ",", 
   RowBox[{"Ticks", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"282", ",", "283", ",", "284"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "6"}], ",", 
        RowBox[{"-", "4"}], ",", 
        RowBox[{"-", "2"}], ",", "0"}], "}"}]}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->36080142],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 111},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->127851478]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->69307833],

Cell[TextData[{
 "First occurrence of ",
 ButtonBox["Lehmer's phenomenon",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://mathworld.wolfram.com/LehmersPhenomenon.html"], None}],
 ": "
}], "ExampleText",
 CellID->577028046],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"ZetaZero", "[", "6709", "]"}], ",", 
     RowBox[{"ZetaZero", "[", "6710", "]"}]}], "}"}], ",", "20"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->91742678],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"0.5`16.004072963893837", "+", 
    RowBox[{
    "7005.0628661749205813830359943595`20.1505149967257", " ", 
     "\[ImaginaryI]"}]}], ",", 
   RowBox[{"0.5`16.004070626697764", "+", 
    RowBox[{
    "7005.1005646726467215652030458067`20.15051499672571", " ", 
     "\[ImaginaryI]"}]}]}], "}"}]], "Output",
 ImageSize->{315, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->450952053]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->2123667759],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RiemannSiegelZ", "[", 
  RowBox[{"Im", "[", 
   RowBox[{"ZetaZero", "[", "10", "]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->664055019],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->804900421]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1305812373],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["ZetaZero",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ZetaZero"], "[", "0", "]"}]], "InlineFormula"],
 " is not defined:"
}], "ExampleText",
 CellID->44117835],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{"ZetaZero", "[", "0", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->224899976],

Cell[BoxData[
 RowBox[{
  RowBox[{"ZetaZero", "::", "\<\"intnz\"\>"}], ":", 
  " ", "\<\"Nonzero integer expected at position \\!\\(1\\) in \\!\\(ZetaZero\
\\).\"\>"}]], "Message", "MSG",
 CellID->160543761],

Cell[BoxData[
 RowBox[{"ZetaZero", "[", "0", "]"}]], "Output",
 ImageSize->{81, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->219003875]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["Zeta",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Zeta"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["RiemannSiegelZ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RiemannSiegelZ"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["PrimePi",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/PrimePi"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->250839057],

Cell[TextData[ButtonBox["Special Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/SpecialFunctions"]], "Tutorials",
 CellID->23022]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Inverse Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/InverseFunctions"]], "MoreAbout",
 CellID->289729093],

Cell[TextData[ButtonBox["Number Theoretic Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NumberTheoreticFunctions"]], "MoreAbout",
 CellID->52414915],

Cell[TextData[ButtonBox["Number Theory",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NumberTheory"]], "MoreAbout",
 CellID->206358128],

Cell[TextData[ButtonBox["Prime Numbers",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/PrimeNumbers"]], "MoreAbout",
 CellID->468919898],

Cell[TextData[ButtonBox["Special Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/SpecialFunctions"]], "MoreAbout",
 CellID->35825252],

Cell[TextData[ButtonBox["Zeta Functions & Polylogarithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/ZetaFunctionsAndPolylogarithms"]], "MoreAbout",
 CellID->85762393],

Cell[TextData[ButtonBox["New in 6.0: Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicalFunctions"]], "MoreAbout",
 CellID->13351072],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->114057885],

Cell[TextData[ButtonBox["New in 6.0: Number Theory & Integer Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/NewIn60NumberTheoryAndIntegerFunctions"]], "MoreAbout",
 CellID->110343300]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"ZetaZero - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 30, 9.8846364}", "context" -> "System`", 
    "keywords" -> {
     "zeta function root", "Riemann-Siegel zeta", "Riemann hypothesis", 
      "Gram blocks", "Rosser rule", "GUE hypothesis", 
      "Montgomery pair correlation", "gaps between zeta zeros", 
      "critical line for zeta function", "complex zeros of zeta function", 
      "nontrivial zeros of zeta function"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "ZetaZero[k] represents the k\\[Null]^th zero of the Riemann zeta \
function on the critical line. ZetaZero[k, t] represents the k\\[Null]^th \
zero with imaginary part greater than t.", "synonyms" -> {"zeta zero"}, 
    "title" -> "ZetaZero", "type" -> "Symbol", "uri" -> "ref/ZetaZero"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[9609, 299, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->712695132]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 43092, 1300}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 3313, 74, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3936, 101, 1573, 37, 70, "ObjectNameGrid"],
Cell[5512, 140, 1302, 41, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[6851, 186, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[7329, 203, 120, 4, 70, "Notes",
 CellID->20640],
Cell[7452, 209, 1018, 40, 70, "Notes",
 CellID->8759937],
Cell[8473, 251, 372, 13, 70, "Notes",
 CellID->47592934],
Cell[8848, 266, 299, 10, 70, "Notes",
 CellID->43381924],
Cell[9150, 278, 218, 7, 70, "Notes",
 CellID->31226],
Cell[9371, 287, 201, 7, 70, "Notes",
 CellID->28162]
}, Closed]],
Cell[CellGroupData[{
Cell[9609, 299, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->712695132],
Cell[CellGroupData[{
Cell[9994, 313, 147, 5, 70, "ExampleSection",
 CellID->24628098],
Cell[10144, 320, 91, 1, 70, "ExampleText",
 CellID->395660164],
Cell[CellGroupData[{
Cell[10260, 325, 130, 4, 28, "Input",
 CellID->9495698],
Cell[10393, 331, 257, 7, 36, "Output",
 CellID->151140807]
}, Open  ]],
Cell[10665, 341, 124, 3, 70, "ExampleDelimiter",
 CellID->98201576],
Cell[10792, 346, 61, 1, 70, "ExampleText",
 CellID->200274704],
Cell[CellGroupData[{
Cell[10878, 351, 135, 4, 70, "Input",
 CellID->34670781],
Cell[11016, 357, 158, 5, 36, "Output",
 CellID->413582641]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[11223, 368, 221, 7, 70, "ExampleSection",
 CellID->1293636265],
Cell[11447, 377, 70, 1, 70, "ExampleText",
 CellID->456095757],
Cell[CellGroupData[{
Cell[11542, 382, 157, 5, 70, "Input",
 CellID->19960652],
Cell[11702, 389, 329, 9, 54, "Output",
 CellID->154497032]
}, Open  ]],
Cell[12046, 401, 125, 3, 70, "ExampleDelimiter",
 CellID->213745782],
Cell[12174, 406, 113, 3, 70, "ExampleText",
 CellID->294298831],
Cell[CellGroupData[{
Cell[12312, 413, 185, 6, 70, "Input",
 CellID->357535962],
Cell[12500, 421, 322, 9, 54, "Output",
 CellID->274974575]
}, Open  ]],
Cell[12837, 433, 125, 3, 70, "ExampleDelimiter",
 CellID->204489201],
Cell[12965, 438, 210, 7, 70, "ExampleText",
 CellID->567006562],
Cell[CellGroupData[{
Cell[13200, 449, 209, 7, 70, "Input",
 CellID->91334672],
Cell[13412, 458, 749, 18, 36, "Output",
 CellID->180245167]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[14210, 482, 244, 7, 70, "ExampleSection",
 CellID->1020263627],
Cell[14457, 491, 241, 8, 70, "ExampleText",
 CellID->430122750],
Cell[CellGroupData[{
Cell[14723, 503, 245, 8, 70, "Input",
 CellID->7341359],
Cell[14971, 513, 160, 5, 36, "Output",
 CellID->85319416]
}, Open  ]],
Cell[CellGroupData[{
Cell[15168, 523, 204, 7, 70, "Input",
 CellID->215553609],
Cell[15375, 532, 329, 9, 54, "Output",
 CellID->211283976]
}, Open  ]],
Cell[CellGroupData[{
Cell[15741, 546, 204, 7, 70, "Input",
 CellID->409630597],
Cell[15948, 555, 327, 9, 54, "Output",
 CellID->360518618]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[16324, 570, 227, 7, 70, "ExampleSection",
 CellID->258228157],
Cell[16554, 579, 83, 1, 70, "ExampleText",
 CellID->560227542],
Cell[CellGroupData[{
Cell[16662, 584, 329, 10, 70, "Input",
 CellID->318207300],
Cell[16994, 596, 1932, 39, 143, "Output",
 CellID->610536658]
}, Open  ]],
Cell[18941, 638, 125, 3, 70, "ExampleDelimiter",
 CellID->514763383],
Cell[19069, 643, 73, 1, 70, "ExampleText",
 CellID->226328844],
Cell[CellGroupData[{
Cell[19167, 648, 689, 22, 70, "Input",
 CellID->5035842],
Cell[19859, 672, 4328, 75, 70, "Output",
 Evaluatable->False,
 CellID->160511189]
}, Open  ]],
Cell[24202, 750, 125, 3, 70, "ExampleDelimiter",
 CellID->336816641],
Cell[24330, 755, 202, 8, 70, "ExampleText",
 CellID->1546359],
Cell[24535, 765, 849, 27, 70, "Input",
 CellID->613438328],
Cell[25387, 794, 260, 8, 70, "ExampleText",
 CellID->156674211],
Cell[CellGroupData[{
Cell[25672, 806, 1164, 35, 70, "Input",
 CellID->159908230],
Cell[26839, 843, 4226, 73, 70, "Output",
 Evaluatable->False,
 CellID->237555958]
}, Open  ]],
Cell[31080, 919, 64, 1, 70, "ExampleText",
 CellID->26042395],
Cell[CellGroupData[{
Cell[31169, 924, 1524, 46, 70, "Input",
 CellID->36080142],
Cell[32696, 972, 2669, 48, 70, "Output",
 Evaluatable->False,
 CellID->127851478]
}, Open  ]],
Cell[35380, 1023, 124, 3, 70, "ExampleDelimiter",
 CellID->69307833],
Cell[35507, 1028, 233, 8, 70, "ExampleText",
 CellID->577028046],
Cell[CellGroupData[{
Cell[35765, 1040, 254, 9, 70, "Input",
 CellID->91742678],
Cell[36022, 1051, 489, 15, 54, "Output",
 CellID->450952053]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[36560, 1072, 238, 7, 70, "ExampleSection",
 CellID->2123667759],
Cell[CellGroupData[{
Cell[36823, 1083, 176, 5, 70, "Input",
 CellID->664055019],
Cell[37002, 1090, 158, 5, 36, "Output",
 CellID->804900421]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[37209, 1101, 231, 7, 70, "ExampleSection",
 CellID->1305812373],
Cell[37443, 1110, 223, 8, 70, "ExampleText",
 CellID->44117835],
Cell[CellGroupData[{
Cell[37691, 1122, 132, 4, 70, "Input",
 CellID->224899976],
Cell[37826, 1128, 207, 5, 70, "Message",
 CellID->160543761],
Cell[38036, 1135, 192, 6, 36, "Output",
 CellID->219003875]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[38289, 1148, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[38608, 1159, 702, 26, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[39347, 1190, 318, 9, 70, "TutorialsSection",
 CellID->250839057],
Cell[39668, 1201, 144, 3, 70, "Tutorials",
 CellID->23022]
}, Open  ]],
Cell[CellGroupData[{
Cell[39849, 1209, 299, 8, 70, "MoreAboutSection"],
Cell[40151, 1219, 145, 3, 70, "MoreAbout",
 CellID->289729093],
Cell[40299, 1224, 161, 3, 70, "MoreAbout",
 CellID->52414915],
Cell[40463, 1229, 137, 3, 70, "MoreAbout",
 CellID->206358128],
Cell[40603, 1234, 137, 3, 70, "MoreAbout",
 CellID->468919898],
Cell[40743, 1239, 144, 3, 70, "MoreAbout",
 CellID->35825252],
Cell[40890, 1244, 172, 3, 70, "MoreAbout",
 CellID->85762393],
Cell[41065, 1249, 173, 3, 70, "MoreAbout",
 CellID->13351072],
Cell[41241, 1254, 179, 3, 70, "MoreAbout",
 CellID->114057885],
Cell[41423, 1259, 198, 4, 70, "MoreAbout",
 CellID->110343300]
}, Open  ]],
Cell[41636, 1266, 27, 0, 70, "History"],
Cell[41666, 1268, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

