(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     63503,       2343]
NotebookOptionsPosition[     47709,       1800]
NotebookOutlinePosition[     51194,       1880]
CellTagsIndexPosition[     51101,       1874]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Numbers" :> 
          Documentation`HelpLookup["paclet:tutorial/NumbersOverview"], 
          "Arbitrary\[Hyphen]Precision Calculations" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/ArbitraryPrecisionCalculations"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Numbers\"\>", 
       2->"\<\"Arbitrary\[Hyphen]Precision Calculations\"\>"}, "\<\"related \
tutorials\"\>", 
       StyleBox["\<\"related tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Numerical Evaluation & Precision" :> 
          Documentation`HelpLookup[
           "paclet:guide/NumericalEvaluationAndPrecision"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"Numerical Evaluation & Precision\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Infinity" :> Documentation`HelpLookup["paclet:ref/Infinity"], 
          "N" :> Documentation`HelpLookup["paclet:ref/N"], "SetAccuracy" :> 
          Documentation`HelpLookup["paclet:ref/SetAccuracy"], "SetPrecision" :> 
          Documentation`HelpLookup["paclet:ref/SetPrecision"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Infinity\"\>", 2->"\<\"N\"\>", 
       3->"\<\"SetAccuracy\"\>", 
       4->"\<\"SetPrecision\"\>"}, "\<\"functions\"\>", 
       StyleBox["\<\"functions\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["Arbitrary\[Hyphen]Precision Numbers", "Title",
 CellTags->{"S3.1.5", "5.1"},
 CellID->19964],

Cell[TextData[{
 "When you do calculations with arbitrary\[Hyphen]precision numbers,",
 " ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " ",
 "keeps track of precision at all points. In general, ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " tries to give you results which have the highest possible precision, given \
the precision of the input you provided. "
}], "Text",
 CellID->1340],

Cell[TextData[{
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " treats arbitrary\[Hyphen]precision numbers as representing the values of \
quantities where a certain number of digits are known, and the rest are \
unknown. In general, an arbitrary\[Hyphen]precision number ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 " is taken to have ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Precision",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Precision"], "[", 
   StyleBox["x", "TI"], "]"}]], "InlineFormula"],
 " digits which are known exactly, followed by an infinite number of digits \
which are completely unknown. "
}], "Text",
 CellID->5686],

Cell[TextData[{
 "This computes ",
 Cell[BoxData[
  FormBox["\[Pi]", TraditionalForm]], "InlineMath"],
 " to 10\[Hyphen]digit precision. "
}], "MathCaption",
 CellID->6021],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{"Pi", ",", "10"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->11662],

Cell[BoxData["3.1415926535897932385`10.000000000000004"], "Output",
 ImageSize->{81, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->705044459]
}, Open  ]],

Cell["After a certain point, all digits are indeterminate. ", "MathCaption",
 CellID->6064],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RealDigits", "[", 
  RowBox[{"%", ",", "10", ",", "13"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->29309],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "3", ",", "1", ",", "4", ",", "1", ",", "5", ",", "9", ",", "2", ",", "6",
      ",", "5", ",", "3", ",", "Indeterminate", ",", "Indeterminate", ",", 
     "Indeterminate"}], "}"}], ",", "1"}], "}"}]], "Output",
 ImageSize->{528, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->25027322]
}, Open  ]],

Cell[TextData[{
 "When you do a computation, ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " keeps track of which digits in your result could be affected by unknown \
digits in your input. It sets the precision of your result so that no \
affected digits are ever included. This procedure ensures that all digits \
returned by ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " are correct, whatever the values of the unknown digits may be. "
}], "Text",
 CellID->17902],

Cell[TextData[{
 "This evaluates ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[CapitalGamma]", "(", 
    RowBox[{"1", "/", "7"}], ")"}], TraditionalForm]], "InlineMath"],
 " to 30\[Hyphen]digit precision. "
}], "MathCaption",
 CellID->29873],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"Gamma", "[", 
    RowBox[{"1", "/", "7"}], "]"}], ",", "30"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->17124],

Cell[BoxData["6.548062940247824437714093349428996262611494602191`30."], \
"Output",
 ImageSize->{221, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->161312918]
}, Open  ]],

Cell["The result has a precision of exactly 30 digits. ", "MathCaption",
 CellID->32745],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Precision", "[", "%", "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->31762],

Cell[BoxData["30.`"], "Output",
 ImageSize->{25, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->344080330]
}, Open  ]],

Cell[TextData[{
 "If you give input only to a few digits of precision, ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " cannot give you such high\[Hyphen]precision output. "
}], "MathCaption",
 CellID->27531],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"Gamma", "[", "0.142", "]"}], ",", "30"}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->4806],

Cell[BoxData["6.589647294920399`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->532708995]
}, Open  ]],

Cell[TextData[{
 "If you want ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " to assume that the argument is ",
 StyleBox["exactly",
  FontSlant->"Italic"],
 " ",
 Cell[BoxData[
  RowBox[{"142", "/", "1000"}]], "InlineFormula"],
 ", then you have to say so explicitly. "
}], "MathCaption",
 CellID->9318],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"Gamma", "[", 
    RowBox[{"142", "/", "1000"}], "]"}], ",", "30"}], "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->5602],

Cell[BoxData["6.589647294920397883284819174959777852275991250324`30."], \
"Output",
 ImageSize->{221, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->103236222]
}, Open  ]],

Cell["\<\
In many computations, the precision of the results you get progressively \
degrades as a result of \[OpenCurlyDoubleQuote]roundoff error\
\[CloseCurlyDoubleQuote]. A typical case of this occurs if you subtract two \
numbers that are close together. The result you get depends on \
high\[Hyphen]order digits in each number, and typically has far fewer digits \
of precision than either of the original numbers. \
\>", "Text",
 CellID->24307],

Cell["\<\
Both input numbers have a precision of around 20 digits, but the result has \
much lower precision. \
\>", "MathCaption",
 CellID->23481],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"1.11111111111111111111", "-", "1.11111111111111111000"}]], "Input",
 CellLabel->"In[7]:=",
 CellID->1012],

Cell[BoxData["1.11`1.7442929831226766*^-18"], "Output",
 ImageSize->{66, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->521419611]
}, Open  ]],

Cell["\<\
Adding extra digits in one number but not the other is not sufficient to \
allow extra digits to be found in the result. \
\>", "MathCaption",
 CellID->26630],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"1.11111111111111111111345", "-", 
  "1.11111111111111111000"}]], "Input",
 CellLabel->"In[8]:=",
 CellID->24114],

Cell[BoxData["1.11345`2.0462366421290494*^-18"], "Output",
 ImageSize->{66, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->136368408]
}, Open  ]],

Cell["\<\
The precision of the output from a function can depend in a complicated way \
on the precision of the input. Functions that vary rapidly typically give \
less precise output, since the variation of the output associated with \
uncertainties in the input is larger. Functions that are close to constants \
can actually give output that is more precise than their input.\
\>", "Text",
 CellID->4084],

Cell["\<\
Here is a case where the output is less precise than the input. \
\>", "MathCaption",
 CellID->23556],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sin", "[", "111111111.0000000000000000", "]"}]], "Input",
 CellLabel->"In[9]:=",
 CellID->12290],

Cell[BoxData[
 RowBox[{"-", 
  "0.29753510333494323326806314389`15.493666339996429"}]], "Output",
 ImageSize->{138, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[9]=",
 CellID->70654409]
}, Open  ]],

Cell[TextData[{
 "Here is ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["e", 
    RowBox[{"-", "40"}]], TraditionalForm]], "InlineMath"],
 " evaluated to 20\[Hyphen]digit precision. "
}], "MathCaption",
 CellID->29996],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"Exp", "[", 
    RowBox[{"-", "40"}], "]"}], ",", "20"}], "]"}]], "Input",
 CellLabel->"In[10]:=",
 CellID->29152],

Cell[BoxData["4.248354255291588995329234782859`20.*^-18"], "Output",
 ImageSize->{192, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[10]=",
 CellID->128518029]
}, Open  ]],

Cell[TextData[{
 "The result you get by adding the exact integer ",
 Cell[BoxData["1"], "InlineFormula"],
 " has a higher precision. "
}], "MathCaption",
 CellID->25345],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"1", "+", "%"}]], "Input",
 CellLabel->"In[11]:=",
 CellID->14708],

Cell[BoxData["1.000000000000000004248354255291588995329234782859`37.\
37177927613008"], "Output",
 ImageSize->{277, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[11]=",
 CellID->240214073]
}, Open  ]],

Cell["\<\
It is worth realizing that different ways of doing the same calculation can \
end up giving you results with very different precisions. Typically, if you \
once lose precision in a calculation, it is essentially impossible to regain \
it; in losing precision, you are effectively losing information about your \
result. \
\>", "Text",
 CellID->7491],

Cell["\<\
Here is a 40\[Hyphen]digit number that is close to 1. \
\>", "MathCaption",
 CellID->13712],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"x", "=", 
  RowBox[{"N", "[", 
   RowBox[{
    RowBox[{"1", "-", 
     RowBox[{"10", "^", 
      RowBox[{"-", "30"}]}]}], ",", "40"}], "]"}]}]], "Input",
 CellLabel->"In[12]:=",
 CellID->5131],

Cell[BoxData["0.999999999999999999999999999999`40."], "Output",
 ImageSize->{298, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[12]=",
 CellID->741427351]
}, Open  ]],

Cell[TextData[{
 "Adding ",
 Cell[BoxData["1"], "InlineFormula"],
 " to it gives another 40\[Hyphen]digit number. "
}], "MathCaption",
 CellID->16439],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"1", "+", "x"}]], "Input",
 CellLabel->"In[13]:=",
 CellID->7958],

Cell[BoxData["1.999999999999999999999999999999`40.30102999566398"], "Output",
 ImageSize->{291, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[13]=",
 CellID->228191347]
}, Open  ]],

Cell["The original precision has been maintained. ", "MathCaption",
 CellID->29704],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Precision", "[", "%", "]"}]], "Input",
 CellLabel->"In[14]:=",
 CellID->6995],

Cell[BoxData["40.30102999566398`"], "Output",
 ImageSize->{46, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[14]=",
 CellID->45378020]
}, Open  ]],

Cell[TextData[{
 "This way of computing ",
 Cell[BoxData[
  RowBox[{"1", "+", "x"}]], "InlineFormula"],
 " loses precision. "
}], "MathCaption",
 CellID->25269],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"(", 
   RowBox[{
    RowBox[{"x", "^", "2"}], "-", "1"}], ")"}], "/", 
  RowBox[{"(", 
   RowBox[{"x", "-", "1"}], ")"}]}]], "Input",
 CellLabel->"In[15]:=",
 CellID->7479],

Cell[BoxData["2.`9.698970004336022"], "Output",
 ImageSize->{81, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[15]=",
 CellID->76680249]
}, Open  ]],

Cell["The result obtained in this way has quite low precision. ", \
"MathCaption",
 CellID->26423],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Precision", "[", "%", "]"}]], "Input",
 CellLabel->"In[16]:=",
 CellID->27918],

Cell[BoxData["9.698970004336022`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[16]=",
 CellID->246634169]
}, Open  ]],

Cell[TextData[{
 "The fact that different ways of doing the same calculation can give you \
different numerical answers means, among other things, that comparisons \
between approximate real numbers must be treated with care. In testing \
whether two real numbers are \[OpenCurlyDoubleQuote]equal\
\[CloseCurlyDoubleQuote], ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " effectively finds their difference, and tests whether the result is \
\[OpenCurlyDoubleQuote]consistent with zero\[CloseCurlyDoubleQuote] to the \
precision given."
}], "Text",
 CellID->17659],

Cell["These numbers are equal to the precision given. ", "MathCaption",
 CellID->32498],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"3", "==", "3.000000000000000000"}]], "Input",
 CellLabel->"In[17]:=",
 CellID->28486],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[17]=",
 CellID->33880555]
}, Open  ]],

Cell[TextData[{
 "The internal algorithms that ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " uses to evaluate mathematical functions are set up to maintain as much \
precision as possible. In most cases, built\[Hyphen]in ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " functions will give you results that have as much precision as can be \
justified on the basis of your input. In some cases, however, it is simply \
impractical to do this, and ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " will give you results that have lower precision. If you give higher\
\[Hyphen]precision input, ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " will use higher precision in its internal calculations, and you will \
usually be able to get a higher\[Hyphen]precision result. "
}], "Text",
 CellID->7462],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     ButtonBox["N",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/N"], "[", 
     StyleBox["expr", "TI"], "]"}], Cell[TextData[{
     "evaluate ",
     Cell[BoxData[
      StyleBox["expr", "TI"]], "InlineFormula"],
     " numerically to machine precision"
    }], "TableText"]},
   {
    RowBox[{
     ButtonBox["N",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/N"], "[", 
     StyleBox["expr", "TI"], ",", 
     StyleBox["n", "TI"], "]"}], Cell[TextData[{
     "evaluate ",
     Cell[BoxData[
      StyleBox["expr", "TI"]], "InlineFormula"],
     " numerically trying to get a result with ",
     Cell[BoxData[
      FormBox["n", TraditionalForm]], "InlineMath"],
     " digits of precision"
    }], "TableText"]}
  }]], "DefinitionBox",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {{False}}, 
   "RowsIndexed" -> {}},
 GridBoxSpacings->{
  "Columns" -> {0.28, {0.56}, 0.28}, "ColumnsIndexed" -> {}, 
   "Rows" -> {0.2, {0.16}, 0.2}, "RowsIndexed" -> {}}},
 CellID->26633],

Cell["Numerical evaluation. ", "Caption",
 CellID->12158],

Cell[TextData[{
 "If you start with an expression that contains only integers and other exact \
numeric quantities, then ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["N",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/N"], "[", 
   RowBox[{
    StyleBox["expr", "TI"], ",", 
    StyleBox["n", "TI"]}], "]"}]], "InlineFormula"],
 " will in almost all cases succeed in giving you a result to ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " digits of precision. You should realize, however, that to do this ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " sometimes has to perform internal intermediate calculations to much higher \
precision."
}], "Text",
 CellID->12022],

Cell[TextData[{
 "The global variable ",
 Cell[BoxData[
  ButtonBox["$MaxExtraPrecision",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/$MaxExtraPrecision"]], "InlineFormula"],
 " specifies how many additional digits should be allowed in such \
intermediate calculations. "
}], "Text",
 CellID->1146],

Cell[BoxData[GridBox[{
   {Cell["variable", "TableHeader"], Cell["default value", "TableHeader"], 
    ""},
   {
    ButtonBox["$MaxExtraPrecision",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/$MaxExtraPrecision"], "50", Cell[
    "maximum additional precision to use", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, {True}, False}, "RowsIndexed" -> {}},
 GridBoxSpacings->{
  "Columns" -> {0.28, {0.56}, 0.28}, "ColumnsIndexed" -> {}, 
   "Rows" -> {0.2, {0.16}, 0.2}, "RowsIndexed" -> {}}},
 CellID->18392],

Cell["Controlling precision in intermediate calculations. ", "Caption",
 CellID->13037],

Cell[TextData[{
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " automatically increases the precision that it uses internally in order to \
get the correct answer here. "
}], "MathCaption",
 CellID->3925],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"Sin", "[", 
    RowBox[{"10", "^", "40"}], "]"}], ",", "30"}], "]"}]], "Input",
 CellLabel->"In[18]:=",
 CellID->10647],

Cell[BoxData[
 RowBox[{"-", 
  "0.569633400953636327308034181573568723132921316046`30."}]], "Output",
 ImageSize->{236, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[18]=",
 CellID->61746896]
}, Open  ]],

Cell[TextData[{
 "Using the default setting ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["$MaxExtraPrecision",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/$MaxExtraPrecision"], "=", "50"}]], 
  "InlineFormula"],
 ", ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " cannot get the correct answer here. "
}], "MathCaption",
 CellID->6602],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"Sin", "[", 
    RowBox[{"10", "^", "100"}], "]"}], ",", "30"}], "]"}]], "Input",
 CellLabel->"In[19]:=",
 CellID->30807],

Cell[BoxData[
 RowBox[{
  RowBox[{"N", "::", "\<\"meprec\"\>"}], ":", 
  " ", "\<\"Internal precision limit $MaxExtraPrecision = \\!\\(50.`\\) \
reached while evaluating \
\\!\\(Sin[\
100000000000000000000000000000000000000000000000000000000000000000000000000000\
00000000000000000000000]\\). \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/General/meprec\\\", ButtonNote -> \
\\\"N::meprec\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[19]:=",
 CellID->146593588],

Cell[BoxData["0``0"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[19]=",
 CellID->986965512]
}, Open  ]],

Cell[TextData[{
 "This tells ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " that it can use more digits in its internal calculations. "
}], "MathCaption",
 CellID->29292],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"$MaxExtraPrecision", "=", "200"}]], "Input",
 CellLabel->"In[20]:=",
 CellID->14600],

Cell[BoxData["200"], "Output",
 ImageSize->{25, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[20]=",
 CellID->403534389]
}, Open  ]],

Cell["Now it gets the correct answer. ", "MathCaption",
 CellID->20799],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"Sin", "[", 
    RowBox[{"10", "^", "100"}], "]"}], ",", "30"}], "]"}]], "Input",
 CellLabel->"In[21]:=",
 CellID->18352],

Cell[BoxData[
 RowBox[{"-", 
  "0.372376123661276688262086695553164295719667883568`30."}]], "Output",
 ImageSize->{236, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[21]=",
 CellID->606156324]
}, Open  ]],

Cell[TextData[{
 "This resets ",
 Cell[BoxData[
  ButtonBox["$MaxExtraPrecision",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/$MaxExtraPrecision"]], "InlineFormula"],
 " to its default value. "
}], "MathCaption",
 CellID->13882],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"$MaxExtraPrecision", "=", "50"}]], "Input",
 CellLabel->"In[22]:=",
 CellID->540],

Cell[BoxData["50"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[22]=",
 CellID->127989118]
}, Open  ]],

Cell[TextData[{
 "Even when you are doing computations that give exact results, ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " still occasionally uses approximate numbers for some of its internal \
calculations, so that the value of ",
 Cell[BoxData[
  ButtonBox["$MaxExtraPrecision",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/$MaxExtraPrecision"]], "InlineFormula"],
 " can thus have an effect. "
}], "Text",
 CellID->4575],

Cell[TextData[{
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " works this out using bounds from approximate numbers. "
}], "MathCaption",
 CellID->8762],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Sin", "[", 
   RowBox[{"Exp", "[", "100", "]"}], "]"}], ">", "0"}]], "Input",
 CellLabel->"In[23]:=",
 CellID->9567],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[23]=",
 CellID->109907093]
}, Open  ]],

Cell[TextData[{
 "With the default value of ",
 Cell[BoxData[
  ButtonBox["$MaxExtraPrecision",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/$MaxExtraPrecision"]], "InlineFormula"],
 ", ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " cannot work this out. "
}], "MathCaption",
 CellID->18397],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Sin", "[", 
   RowBox[{"Exp", "[", "200", "]"}], "]"}], ">", "0"}]], "Input",
 CellLabel->"In[24]:=",
 CellID->31418],

Cell[BoxData[
 RowBox[{
  RowBox[{"N", "::", "\<\"meprec\"\>"}], ":", 
  " ", "\<\"Internal precision limit $MaxExtraPrecision = \\!\\(50.`\\) \
reached while evaluating \
\\!\\(-\\(\\(Sin[\\(\\(\[ExponentialE]\\^200\\)\\)]\\)\\)\\). \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/General/meprec\\\", \
ButtonNote -> \\\"N::meprec\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[24]:=",
 CellID->1946733],

Cell[BoxData[
 RowBox[{
  RowBox[{"Sin", "[", 
   SuperscriptBox["\[ExponentialE]", "200"], "]"}], ">", "0"}]], "Output",
 ImageSize->{83, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[24]=",
 CellID->165184517]
}, Open  ]],

Cell[TextData[{
 "Temporarily resetting ",
 Cell[BoxData[
  ButtonBox["$MaxExtraPrecision",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/$MaxExtraPrecision"]], "InlineFormula"],
 " allows ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " to get the result. "
}], "MathCaption",
 CellID->13851],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Block", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"$MaxExtraPrecision", "=", "100"}], "}"}], ",", 
   RowBox[{
    RowBox[{"Sin", "[", 
     RowBox[{"Exp", "[", "200", "]"}], "]"}], ">", "0"}]}], "]"}]], "Input",
 CellLabel->"In[25]:=",
 CellID->26816],

Cell[BoxData["False"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[25]=",
 CellID->39769224]
}, Open  ]],

Cell[TextData[{
 "In doing calculations that degrade precision, it is possible to end up with \
numbers that have no significant digits at all. But even in such cases, ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " still maintains information on the accuracy of the numbers. Given a number \
with no significant digits, but accuracy ",
 Cell[BoxData[
  FormBox["a", TraditionalForm]], "InlineMath"],
 ", ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " can then still tell that the actual value of the number must be in the \
range ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", 
      SuperscriptBox["10", 
       RowBox[{"-", 
        StyleBox["a", "TI"]}]]}], ",", 
     RowBox[{"+", 
      SuperscriptBox["10", 
       RowBox[{"-", 
        StyleBox["a", "TI"]}]]}]}], "}"}], "/", "2"}]], "InlineFormula"],
 ". ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " by default prints such numbers in the form ",
 Cell[BoxData[
  RowBox[{"0.", "\[Times]", 
   SuperscriptBox["10", 
    StyleBox["e", "TI"]]}]], "InlineFormula"],
 "."
}], "Text",
 CellID->4449],

Cell["\<\
Here is a number with 20\[Hyphen]digit precision. \
\>", "MathCaption",
 CellID->16925],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"x", "=", 
  RowBox[{"N", "[", 
   RowBox[{
    RowBox[{"Exp", "[", "50", "]"}], ",", "20"}], "]"}]}]], "Input",
 CellLabel->"In[26]:=",
 CellID->658],

Cell[BoxData["5.184705528587072464087453322934`20.*^21"], "Output",
 ImageSize->{187, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[26]=",
 CellID->18461276]
}, Open  ]],

Cell["Here there are no significant digits left. ", "MathCaption",
 CellID->24520],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Sin", "[", "x", "]"}], "/", "x"}]], "Input",
 CellLabel->"In[27]:=",
 CellID->10940],

Cell[BoxData["0``21.71472409516259"], "Output",
 ImageSize->{59, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[27]=",
 CellID->269308668]
}, Open  ]],

Cell[TextData[{
 "But ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " still keeps track of the accuracy of the result. "
}], "MathCaption",
 CellID->15147],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Accuracy", "[", "%", "]"}]], "Input",
 CellLabel->"In[28]:=",
 CellID->25162],

Cell[BoxData["21.71472409516259`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[28]=",
 CellID->83840535]
}, Open  ]],

Cell[TextData[{
 "Adding the result to an exact ",
 Cell[BoxData["1"], "InlineFormula"],
 " gives a number with quite high precision. "
}], "MathCaption",
 CellID->20675],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"1", "+", "%%"}]], "Input",
 CellLabel->"In[29]:=",
 CellID->792],

Cell[BoxData["1.`21.71472409516259"], "Output",
 ImageSize->{165, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[29]=",
 CellID->125966467]
}, Open  ]],

Cell["\<\
One subtlety in characterizing numbers by their precision is that any number \
that is consistent with zero must be treated as having zero precision. The \
reason for this is that such a number has no digits that can be recognized as \
significant, since all its known digits are just zero. \
\>", "Text",
 CellID->31754],

Cell["This gives a number whose value is consistent with zero. ", \
"MathCaption",
 CellID->16398],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"d", "=", 
  RowBox[{
   RowBox[{"N", "[", 
    RowBox[{"Pi", ",", "20"}], "]"}], "-", "Pi"}]}]], "Input",
 CellLabel->"In[30]:=",
 CellID->23146],

Cell[BoxData["0``19.502850127305866"], "Output",
 ImageSize->{59, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[30]=",
 CellID->36125067]
}, Open  ]],

Cell["\<\
The number has no recognizable significant digits of precision. \
\>", "MathCaption",
 CellID->26946],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Precision", "[", "d", "]"}]], "Input",
 CellLabel->"In[31]:=",
 CellID->21188],

Cell[BoxData["0.`"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[31]=",
 CellID->187280716]
}, Open  ]],

Cell["\<\
But it still has a definite accuracy, that characterizes the uncertainty in \
it. \
\>", "MathCaption",
 CellID->8638],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Accuracy", "[", "d", "]"}]], "Input",
 CellLabel->"In[32]:=",
 CellID->27663],

Cell[BoxData["19.502850127305866`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[32]=",
 CellID->126583283]
}, Open  ]],

Cell["\<\
If you do computations whose results are likely to be near zero, it can be \
convenient to specify the accuracy, rather than the precision, that you want \
to get. \
\>", "Text",
 CellID->24515],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     ButtonBox["N",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/N"], "[", 
     StyleBox["expr", "TI"], ",", 
     StyleBox["p", "TI"], "]"}], Cell[TextData[{
     "evaluate ",
     Cell[BoxData[
      StyleBox["expr", "TI"]], "InlineFormula"],
     " to precision ",
     Cell[BoxData[
      FormBox["p", TraditionalForm]], "InlineMath"]
    }], "TableText"]},
   {
    RowBox[{
     ButtonBox["N",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/N"], "[", 
     StyleBox["expr", "TI"], ",", "{", 
     StyleBox["p", "TI"], ",", 
     StyleBox["a", "TI"], "}", "]"}], Cell[TextData[{
     "evaluate ",
     Cell[BoxData[
      StyleBox["expr", "TI"]], "InlineFormula"],
     " to at most precision ",
     Cell[BoxData[
      FormBox["p", TraditionalForm]], "InlineMath"],
     " and accuracy ",
     Cell[BoxData[
      FormBox["a", TraditionalForm]], "InlineMath"]
    }], "TableText"]},
   {
    RowBox[{
     ButtonBox["N",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/N"], "[", 
     StyleBox["expr", "TI"], ",", "{", 
     ButtonBox["Infinity",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Infinity"], ",", 
     StyleBox["a", "TI"], "}", "]"}], Cell[TextData[{
     "evaluate ",
     Cell[BoxData[
      StyleBox["expr", "TI"]], "InlineFormula"],
     " to any precision but to accuracy ",
     Cell[BoxData[
      FormBox["a", TraditionalForm]], "InlineMath"]
    }], "TableText"]}
  }]], "DefinitionBox",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, False, {False}, False}, "RowsIndexed" -> {}},
 GridBoxSpacings->{
  "Columns" -> {0.28, {0.56}, 0.28}, "ColumnsIndexed" -> {}, 
   "Rows" -> {0.2, {0.16}, 0.2}, "RowsIndexed" -> {}}},
 CellID->11521],

Cell["Specifying accuracy as well as precision. ", "Caption",
 CellID->475],

Cell["Here is a symbolic expression. ", "MathCaption",
 CellID->15615],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"u", "=", 
  RowBox[{
   RowBox[{"ArcTan", "[", 
    RowBox[{"1", "/", "3"}], "]"}], "-", 
   RowBox[{"ArcCot", "[", "3", "]"}]}]}]], "Input",
 CellLabel->"In[33]:=",
 CellID->20528],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", 
   RowBox[{"ArcCot", "[", "3", "]"}]}], "+", 
  RowBox[{"ArcTan", "[", 
   FractionBox["1", "3"], "]"}]}]], "Output",
 ImageSize->{155, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[33]=",
 CellID->44979545]
}, Open  ]],

Cell["This shows that the expression is equivalent to zero. ", "MathCaption",
 CellID->12570],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FullSimplify", "[", "u", "]"}]], "Input",
 CellLabel->"In[34]:=",
 CellID->905],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[34]=",
 CellID->333704992]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["N",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/N"]], "InlineFormula"],
 " cannot guarantee to get a result to precision 20. "
}], "MathCaption",
 CellID->19557],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{"u", ",", "20"}], "]"}]], "Input",
 CellLabel->"In[35]:=",
 CellID->28962],

Cell[BoxData[
 RowBox[{
  RowBox[{"N", "::", "\<\"meprec\"\>"}], ":", 
  " ", "\<\"Internal precision limit $MaxExtraPrecision = \\!\\(50.`\\) \
reached while evaluating \\!\\(\\(\\(-\\(\\(ArcCot[3]\\)\\)\\)\\) + \
\\(\\(ArcTan[\\(\\(1\\/3\\)\\)]\\)\\)\\). \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/General/meprec\\\", ButtonNote -> \
\\\"N::meprec\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[35]:=",
 CellID->373567186],

Cell[BoxData["0``70.0153594413863"], "Output",
 ImageSize->{59, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[35]=",
 CellID->78614419]
}, Open  ]],

Cell["But it can get a result to accuracy 20. ", "MathCaption",
 CellID->12549],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{"u", ",", 
   RowBox[{"{", 
    RowBox[{"Infinity", ",", "20"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[36]:=",
 CellID->17469],

Cell[BoxData["0``20."], "Output",
 ImageSize->{59, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[36]=",
 CellID->301540253]
}, Open  ]],

Cell[TextData[{
 "When ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " works out the potential effect of unknown digits in \
arbitrary\[Hyphen]precision numbers, it assumes by default that these digits \
are completely independent in different numbers. While this assumption will \
never yield too high a precision in a result, it may lead to unnecessary loss \
of precision. "
}], "Text",
 CellID->31923],

Cell[TextData[{
 "In particular, if two numbers are generated in the same way in a \
computation, some of their unknown digits may be equal. Then, when these \
numbers are, for example, subtracted, the unknown digits may cancel. By \
assuming that the unknown digits are always independent, however, ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " will miss such cancellations. "
}], "Text",
 CellID->28350],

Cell["\<\
Here is a number computed to 20\[Hyphen]digit precision. \
\>", "MathCaption",
 CellID->14333],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"d", "=", 
  RowBox[{"N", "[", 
   RowBox[{
    RowBox[{"3", "^", 
     RowBox[{"-", "30"}]}], ",", "20"}], "]"}]}]], "Input",
 CellLabel->"In[37]:=",
 CellID->22925],

Cell[BoxData["4.856935749618861137906242664974575`20.*^-15"], "Output",
 ImageSize->{192, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[37]=",
 CellID->530934566]
}, Open  ]],

Cell[TextData[{
 "The quantity ",
 Cell[BoxData[
  RowBox[{"1", "+", "d"}]], "InlineFormula"],
 " has about 34\[Hyphen]digit precision. "
}], "MathCaption",
 CellID->19737],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Precision", "[", 
  RowBox[{"1", "+", "d"}], "]"}]], "Input",
 CellLabel->"In[38]:=",
 CellID->16336],

Cell[BoxData["34.31363764158988`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[38]=",
 CellID->151364406]
}, Open  ]],

Cell[TextData[{
 "This quantity has lower precision, since ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " assumes that the unknown digits in each number ",
 Cell[BoxData["d"], "InlineFormula"],
 " are independent."
}], "MathCaption",
 CellID->468393254],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Precision", "[", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{"1", "+", "d"}], ")"}], "-", "d"}], "]"}]], "Input",
 CellLabel->"In[39]:=",
 CellID->12393],

Cell[BoxData["34.0126076459259`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[39]=",
 CellID->771077801]
}, Open  ]],

Cell["\<\
Numerical algorithms sometimes rely on cancellations between unknown digits \
in different numbers yielding results of higher precision. If you can be sure \
that certain unknown digits will eventually cancel, then you can explicitly \
introduce fixed digits in place of the unknown ones. You can carry these \
fixed digits through your computation, then let them cancel, and get a result \
of higher precision. \
\>", "Text",
 CellID->30714],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     ButtonBox["SetPrecision",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/SetPrecision"], "[", 
     StyleBox["x", "TI"], ",", 
     StyleBox["n", "TI"], "]"}], Cell[TextData[{
     "create a number with ",
     Cell[BoxData[
      FormBox["n", TraditionalForm]], "InlineMath"],
     " decimal digits of precision, padding with base\[Hyphen]2 zeros if \
necessary"
    }], "TableText"]},
   {
    RowBox[{
     ButtonBox["SetAccuracy",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/SetAccuracy"], "[", 
     StyleBox["x", "TI"], ",", 
     StyleBox["n", "TI"], "]"}], Cell[TextData[{
     "create a number with ",
     Cell[BoxData[
      FormBox["n", TraditionalForm]], "InlineMath"],
     " decimal digits of accuracy"
    }], "TableText"]}
  }]], "DefinitionBox",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {{False}}, 
   "RowsIndexed" -> {}},
 GridBoxSpacings->{
  "Columns" -> {0.28, {0.56}, 0.28}, "ColumnsIndexed" -> {}, 
   "Rows" -> {0.2, {0.16}, 0.2}, "RowsIndexed" -> {}}},
 CellID->28164],

Cell["Functions for modifying precision and accuracy.", "Caption",
 CellID->31591],

Cell[TextData[{
 "This introduces 10 more digits in ",
 Cell[BoxData["d"], "InlineFormula"],
 ". "
}], "MathCaption",
 CellID->19949],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"d", "=", 
  RowBox[{"SetPrecision", "[", 
   RowBox[{"d", ",", "30"}], "]"}]}]], "Input",
 CellLabel->"In[40]:=",
 CellID->19135],

Cell[BoxData["4.8569357496188611379062426649745753872366046`30.*^-15"], \
"Output",
 ImageSize->{262, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[40]=",
 CellID->251267715]
}, Open  ]],

Cell["The digits that were added cancel out here. ", "MathCaption",
 CellID->13337],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"(", 
   RowBox[{"1", "+", "d"}], ")"}], "-", "d"}]], "Input",
 CellLabel->"In[41]:=",
 CellID->10004],

Cell[BoxData["1.`44.0126076459259"], "Output",
 ImageSize->{326, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[41]=",
 CellID->433192813]
}, Open  ]],

Cell["\<\
The precision of the result is now about 44 digits, rather than 34. \
\>", "MathCaption",
 CellID->2623],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Precision", "[", "%", "]"}]], "Input",
 CellLabel->"In[42]:=",
 CellID->28664],

Cell[BoxData["44.0126076459259`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[42]=",
 CellID->22178275]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SetPrecision",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SetPrecision"]], "InlineFormula"],
 " works by adding digits which are zero in base 2. Sometimes, ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " stores slightly more digits in an arbitrary\[Hyphen]precision number than \
it displays, and in such cases, ",
 Cell[BoxData[
  ButtonBox["SetPrecision",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SetPrecision"]], "InlineFormula"],
 " will use these extra digits before introducing zeros. "
}], "Text",
 CellID->25608],

Cell["\<\
This creates a number with a precision of 40 decimal digits. The extra digits \
come from conversion to base 10. \
\>", "MathCaption",
 CellID->30568],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SetPrecision", "[", 
  RowBox[{"0.400000000000000", ",", "40"}], "]"}]], "Input",
 CellLabel->"In[43]:=",
 CellID->19281],

Cell[BoxData["0.40000000000000002220446049250313080847263336181640625`40."], \
"Output",
 ImageSize->{298, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[43]=",
 CellID->48918553]
}, Open  ]],

Cell[BoxData[GridBox[{
   {Cell["variable", "TableHeader"], Cell["default value", "TableHeader"], 
    ""},
   {
    ButtonBox["$MaxPrecision",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/$MaxPrecision"], 
    ButtonBox["Infinity",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Infinity"], Cell[
    "maximum total precision to be used", "TableText"]},
   {
    ButtonBox["$MinPrecision",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/$MinPrecision"], 
    RowBox[{"-", 
     ButtonBox["Infinity",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Infinity"]}], Cell[
    "minimum precision to be used", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}},
 GridBoxSpacings->{
  "Columns" -> {0.28, {0.56}, 0.28}, "ColumnsIndexed" -> {}, 
   "Rows" -> {0.2, {0.16}, 0.2}, "RowsIndexed" -> {}}},
 CellID->9152],

Cell["Global precision control parameters. ", "Caption",
 CellID->18373],

Cell[TextData[{
 "By making the global assignment ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["$MinPrecision",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/$MinPrecision"], "=", 
   StyleBox["n", "TI"]}]], "InlineFormula"],
 ", you can effectively apply ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["SetPrecision",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SetPrecision"], "[", 
   RowBox[{
    StyleBox["expr", "TI"], ",", 
    StyleBox["n", "TI"]}], "]"}]], "InlineFormula"],
 " at every step in a computation. This means that even when the number of \
correct digits in an arbitrary\[Hyphen]precision number drops below ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 ", the number will always be padded to have ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " digits. "
}], "Text",
 CellID->28652],

Cell[TextData[{
 "If you set ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["$MaxPrecision",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/$MaxPrecision"], "=", 
   StyleBox["n", "TI"]}]], "InlineFormula"],
 " as well as ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["$MinPrecision",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/$MinPrecision"], "=", 
   StyleBox["n", "TI"]}]], "InlineFormula"],
 ", then you can force all arbitrary\[Hyphen]precision numbers to have a \
fixed precision of ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " digits. In effect, what this does is to make ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " treat arbitrary\[Hyphen]precision numbers in much the same way as it \
treats machine numbers\[LongDash]but with more digits of precision."
}], "Text",
 CellID->8194],

Cell["\<\
Fixed\[Hyphen]precision computation can make some calculations more \
efficient, but without careful analysis you can never be sure how many digits \
are correct in the results you get.\
\>", "Text",
 CellID->9876],

Cell["\<\
Here is a small number with 20\[Hyphen]digit precision. \
\>", "MathCaption",
 CellID->23826],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"k", "=", 
  RowBox[{"N", "[", 
   RowBox[{
    RowBox[{"Exp", "[", 
     RowBox[{"-", "60"}], "]"}], ",", "20"}], "]"}]}]], "Input",
 CellLabel->"In[44]:=",
 CellID->28396],

Cell[BoxData["8.7565107626965203384887328007392`20.*^-27"], "Output",
 ImageSize->{192, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[44]=",
 CellID->150947412]
}, Open  ]],

Cell[TextData[{
 "With ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 "\[CloseCurlyQuote]s usual arithmetic, this works fine. "
}], "MathCaption",
 CellID->6249],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Evaluate", "[", 
   RowBox[{"1", "+", "k"}], "]"}], "-", "1"}]], "Input",
 CellLabel->"In[45]:=",
 CellID->14640],

Cell[BoxData["8.7565107626965203384887328007392`19.999999999999996*^-27"], \
"Output",
 ImageSize->{192, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[45]=",
 CellID->138922340]
}, Open  ]],

Cell[TextData[{
 "This tells ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " to use fixed\[Hyphen]precision arithmetic. "
}], "MathCaption",
 CellID->18819],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"$MinPrecision", "=", 
  RowBox[{"$MaxPrecision", "=", "20"}]}]], "Input",
 CellLabel->"In[46]:=",
 CellID->943],

Cell[BoxData["20"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[46]=",
 CellID->783520870]
}, Open  ]],

Cell["The first few digits are correct, but the rest are wrong. ", \
"MathCaption",
 CellID->1941],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Evaluate", "[", 
   RowBox[{"1", "+", "k"}], "]"}], "-", "1"}]], "Input",
 CellLabel->"In[47]:=",
 CellID->20289],

Cell[BoxData["8.7565107626963908935`20.*^-27"], "Output",
 ImageSize->{192, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[47]=",
 CellID->700407683]
}, Open  ]],

Cell[CellGroupData[{

Cell["MORE ABOUT", "TutorialMoreAboutSection",
 CellID->339521319],

Cell[TextData[ButtonBox["Numerical Evaluation & Precision",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/NumericalEvaluationAndPrecision"]], "TutorialMoreAbout",
 CellID->171830946]
}, Open  ]],

Cell[CellGroupData[{

Cell["RELATED TUTORIALS", "RelatedTutorialsSection",
 CellID->107586942],

Cell[TextData[ButtonBox["Numbers",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/NumbersOverview"]], "RelatedTutorials",
 CellID->54483688],

Cell[TextData[ButtonBox["Arbitrary\[Hyphen]Precision Calculations",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/ArbitraryPrecisionCalculations"]], "RelatedTutorials",
 CellID->146018610]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Arbitrary-Precision Numbers - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, "Overview" -> {"toc" -> Cell[
      BoxData[
       ButtonBox[
        StyleBox["Numbers", "OverviewNavText"], BaseStyle -> "Link", 
        ButtonData -> "paclet:tutorial/NumbersOverview"]], "Text", FontFamily -> 
      "Verdana"], "prevnext" -> Cell[
      TextData[{
        ButtonBox[
         StyleBox["\[FilledLeftTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> "paclet:tutorial/NumericalPrecision"], 
        "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]", 
        ButtonBox[
         StyleBox["\[FilledRightTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> "paclet:tutorial/MachinePrecisionNumbers"]}], 
      "Text", FontFamily -> "Verdana"]}, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 32, 51.7169042}", "context" -> "", 
    "keywords" -> {
     "accuracy", "algorithms", "arbitrary precision functions", 
      "arbitrary precision numbers", "big floats", "big numbers", 
      "cancellations", "correctness", "digits", "equality testing", "errors", 
      "functions", "guard digits", "internal arithmetic", 
      "Kahanian arithmetic", "N", "numerical errors", "parameterized zeros", 
      "parametrized zeros", "Precision", "roundoff errors", "SetAccuracy", 
      "SetPrecision", "testing equality", "unknown digits", "zero", 
      "zero testing", "$MaxExtraPrecision", "$MaxPrecision", "$MinPrecision"},
     "index" -> True, "label" -> "Mathematica Tutorial", "language" -> "en", 
    "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "When you do calculations with arbitrary-precision numbers, Mathematica \
keeps track of precision at all points. In general, Mathematica tries to give \
you results which have the highest possible precision, given the precision of \
the input you provided. Mathematica treats arbitrary-precision numbers as \
representing the values of quantities where a certain number of digits are \
known, and the rest are unknown. In general, an arbitrary-precision number x \
is taken to have Precision[x] digits which are known exactly, followed by an \
infinite number of digits which are completely unknown. This computes \\[Pi] \
to 10-digit precision. ", "synonyms" -> {"3.1.5"}, "title" -> 
    "Arbitrary-Precision Numbers", "type" -> "Tutorial", "uri" -> 
    "tutorial/ArbitraryPrecisionNumbers"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "S3.1.5"->{
  Cell[2889, 81, 98, 2, 70, "Title",
   CellTags->{"S3.1.5", "5.1"},
   CellID->19964]},
 "5.1"->{
  Cell[2889, 81, 98, 2, 70, "Title",
   CellTags->{"S3.1.5", "5.1"},
   CellID->19964]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"S3.1.5", 50890, 1863},
 {"5.1", 50989, 1867}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 2264, 54, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2889, 81, 98, 2, 70, "Title",
 CellTags->{"S3.1.5", "5.1"},
 CellID->19964],
Cell[2990, 85, 403, 12, 70, "Text",
 CellID->1340],
Cell[3396, 99, 674, 18, 70, "Text",
 CellID->5686],
Cell[4073, 119, 172, 6, 70, "MathCaption",
 CellID->6021],
Cell[CellGroupData[{
Cell[4270, 129, 118, 4, 28, "Input",
 CellID->11662],
Cell[4391, 135, 197, 5, 36, "Output",
 CellID->705044459]
}, Open  ]],
Cell[4603, 143, 91, 1, 70, "MathCaption",
 CellID->6064],
Cell[CellGroupData[{
Cell[4719, 148, 137, 4, 28, "Input",
 CellID->29309],
Cell[4859, 154, 415, 12, 36, "Output",
 CellID->25027322]
}, Open  ]],
Cell[5289, 169, 481, 12, 70, "Text",
 CellID->17902],
Cell[5773, 183, 238, 8, 70, "MathCaption",
 CellID->29873],
Cell[CellGroupData[{
Cell[6036, 195, 175, 6, 70, "Input",
 CellID->17124],
Cell[6214, 203, 214, 6, 36, "Output",
 CellID->161312918]
}, Open  ]],
Cell[6443, 212, 88, 1, 70, "MathCaption",
 CellID->32745],
Cell[CellGroupData[{
Cell[6556, 217, 101, 3, 70, "Input",
 CellID->31762],
Cell[6660, 222, 161, 5, 36, "Output",
 CellID->344080330]
}, Open  ]],
Cell[6836, 230, 214, 6, 70, "MathCaption",
 CellID->27531],
Cell[CellGroupData[{
Cell[7075, 240, 153, 5, 70, "Input",
 CellID->4806],
Cell[7231, 247, 175, 5, 36, "Output",
 CellID->532708995]
}, Open  ]],
Cell[7421, 255, 311, 12, 70, "MathCaption",
 CellID->9318],
Cell[CellGroupData[{
Cell[7757, 271, 179, 6, 70, "Input",
 CellID->5602],
Cell[7939, 279, 214, 6, 36, "Output",
 CellID->103236222]
}, Open  ]],
Cell[8168, 288, 450, 8, 70, "Text",
 CellID->24307],
Cell[8621, 298, 147, 4, 70, "MathCaption",
 CellID->23481],
Cell[CellGroupData[{
Cell[8793, 306, 129, 3, 70, "Input",
 CellID->1012],
Cell[8925, 311, 185, 5, 36, "Output",
 CellID->521419611]
}, Open  ]],
Cell[9125, 319, 168, 4, 70, "MathCaption",
 CellID->26630],
Cell[CellGroupData[{
Cell[9318, 327, 136, 4, 70, "Input",
 CellID->24114],
Cell[9457, 333, 188, 5, 36, "Output",
 CellID->136368408]
}, Open  ]],
Cell[9660, 341, 407, 7, 70, "Text",
 CellID->4084],
Cell[10070, 350, 111, 3, 70, "MathCaption",
 CellID->23556],
Cell[CellGroupData[{
Cell[10206, 357, 120, 3, 70, "Input",
 CellID->12290],
Cell[10329, 362, 227, 7, 36, "Output",
 CellID->70654409]
}, Open  ]],
Cell[10571, 372, 219, 8, 70, "MathCaption",
 CellID->29996],
Cell[CellGroupData[{
Cell[10815, 384, 170, 6, 70, "Input",
 CellID->29152],
Cell[10988, 392, 200, 5, 36, "Output",
 CellID->128518029]
}, Open  ]],
Cell[11203, 400, 169, 5, 70, "MathCaption",
 CellID->25345],
Cell[CellGroupData[{
Cell[11397, 409, 89, 3, 70, "Input",
 CellID->14708],
Cell[11489, 414, 229, 6, 36, "Output",
 CellID->240214073]
}, Open  ]],
Cell[11733, 423, 359, 7, 70, "Text",
 CellID->7491],
Cell[12095, 432, 101, 3, 70, "MathCaption",
 CellID->13712],
Cell[CellGroupData[{
Cell[12221, 439, 216, 8, 70, "Input",
 CellID->5131],
Cell[12440, 449, 195, 5, 36, "Output",
 CellID->741427351]
}, Open  ]],
Cell[12650, 457, 150, 5, 70, "MathCaption",
 CellID->16439],
Cell[CellGroupData[{
Cell[12825, 466, 88, 3, 70, "Input",
 CellID->7958],
Cell[12916, 471, 209, 5, 36, "Output",
 CellID->228191347]
}, Open  ]],
Cell[13140, 479, 83, 1, 70, "MathCaption",
 CellID->29704],
Cell[CellGroupData[{
Cell[13248, 484, 101, 3, 70, "Input",
 CellID->6995],
Cell[13352, 489, 175, 5, 36, "Output",
 CellID->45378020]
}, Open  ]],
Cell[13542, 497, 160, 6, 70, "MathCaption",
 CellID->25269],
Cell[CellGroupData[{
Cell[13727, 507, 207, 8, 70, "Input",
 CellID->7479],
Cell[13937, 517, 177, 5, 36, "Output",
 CellID->76680249]
}, Open  ]],
Cell[14129, 525, 98, 2, 70, "MathCaption",
 CellID->26423],
Cell[CellGroupData[{
Cell[14252, 531, 102, 3, 70, "Input",
 CellID->27918],
Cell[14357, 536, 176, 5, 36, "Output",
 CellID->246634169]
}, Open  ]],
Cell[14548, 544, 571, 12, 70, "Text",
 CellID->17659],
Cell[15122, 558, 87, 1, 70, "MathCaption",
 CellID->32498],
Cell[CellGroupData[{
Cell[15234, 563, 109, 3, 70, "Input",
 CellID->28486],
Cell[15346, 568, 161, 5, 36, "Output",
 CellID->33880555]
}, Open  ]],
Cell[15522, 576, 824, 20, 70, "Text",
 CellID->7462],
Cell[16349, 598, 1088, 35, 70, "DefinitionBox",
 CellID->26633],
Cell[17440, 635, 57, 1, 70, "Caption",
 CellID->12158],
Cell[17500, 638, 698, 20, 70, "Text",
 CellID->12022],
Cell[18201, 660, 303, 9, 70, "Text",
 CellID->1146],
Cell[18507, 671, 623, 16, 70, "DefinitionBox3Col",
 CellID->18392],
Cell[19133, 689, 87, 1, 70, "Caption",
 CellID->13037],
Cell[19223, 692, 208, 6, 70, "MathCaption",
 CellID->3925],
Cell[CellGroupData[{
Cell[19456, 702, 176, 6, 70, "Input",
 CellID->10647],
Cell[19635, 710, 232, 7, 36, "Output",
 CellID->61746896]
}, Open  ]],
Cell[19882, 720, 346, 13, 70, "MathCaption",
 CellID->6602],
Cell[CellGroupData[{
Cell[20253, 737, 177, 6, 70, "Input",
 CellID->30807],
Cell[20433, 745, 572, 12, 70, "Message",
 CellID->146593588],
Cell[21008, 759, 162, 5, 36, "Output",
 CellID->986965512]
}, Open  ]],
Cell[21185, 767, 178, 6, 70, "MathCaption",
 CellID->29292],
Cell[CellGroupData[{
Cell[21388, 777, 108, 3, 70, "Input",
 CellID->14600],
Cell[21499, 782, 161, 5, 36, "Output",
 CellID->403534389]
}, Open  ]],
Cell[21675, 790, 71, 1, 70, "MathCaption",
 CellID->20799],
Cell[CellGroupData[{
Cell[21771, 795, 177, 6, 70, "Input",
 CellID->18352],
Cell[21951, 803, 233, 7, 36, "Output",
 CellID->606156324]
}, Open  ]],
Cell[22199, 813, 233, 8, 70, "MathCaption",
 CellID->13882],
Cell[CellGroupData[{
Cell[22457, 825, 105, 3, 70, "Input",
 CellID->540],
Cell[22565, 830, 160, 5, 36, "Output",
 CellID->127989118]
}, Open  ]],
Cell[22740, 838, 439, 12, 70, "Text",
 CellID->4575],
Cell[23182, 852, 157, 5, 70, "MathCaption",
 CellID->8762],
Cell[CellGroupData[{
Cell[23364, 861, 151, 5, 70, "Input",
 CellID->9567],
Cell[23518, 868, 162, 5, 36, "Output",
 CellID->109907093]
}, Open  ]],
Cell[23695, 876, 303, 11, 70, "MathCaption",
 CellID->18397],
Cell[CellGroupData[{
Cell[24023, 891, 152, 5, 70, "Input",
 CellID->31418],
Cell[24178, 898, 511, 10, 70, "Message",
 CellID->1946733],
Cell[24692, 910, 252, 8, 39, "Output",
 CellID->165184517]
}, Open  ]],
Cell[24959, 921, 302, 11, 70, "MathCaption",
 CellID->13851],
Cell[CellGroupData[{
Cell[25286, 936, 280, 9, 70, "Input",
 CellID->26816],
Cell[25569, 947, 162, 5, 36, "Output",
 CellID->39769224]
}, Open  ]],
Cell[25746, 955, 1107, 36, 70, "Text",
 CellID->4449],
Cell[26856, 993, 97, 3, 70, "MathCaption",
 CellID->16925],
Cell[CellGroupData[{
Cell[26978, 1000, 173, 6, 70, "Input",
 CellID->658],
Cell[27154, 1008, 198, 5, 36, "Output",
 CellID->18461276]
}, Open  ]],
Cell[27367, 1016, 82, 1, 70, "MathCaption",
 CellID->24520],
Cell[CellGroupData[{
Cell[27474, 1021, 119, 4, 70, "Input",
 CellID->10940],
Cell[27596, 1027, 178, 5, 36, "Output",
 CellID->269308668]
}, Open  ]],
Cell[27789, 1035, 162, 6, 70, "MathCaption",
 CellID->15147],
Cell[CellGroupData[{
Cell[27976, 1045, 101, 3, 70, "Input",
 CellID->25162],
Cell[28080, 1050, 175, 5, 36, "Output",
 CellID->83840535]
}, Open  ]],
Cell[28270, 1058, 170, 5, 70, "MathCaption",
 CellID->20675],
Cell[CellGroupData[{
Cell[28465, 1067, 88, 3, 70, "Input",
 CellID->792],
Cell[28556, 1072, 179, 5, 36, "Output",
 CellID->125966467]
}, Open  ]],
Cell[28750, 1080, 331, 6, 70, "Text",
 CellID->31754],
Cell[29084, 1088, 98, 2, 70, "MathCaption",
 CellID->16398],
Cell[CellGroupData[{
Cell[29207, 1094, 169, 6, 70, "Input",
 CellID->23146],
Cell[29379, 1102, 178, 5, 36, "Output",
 CellID->36125067]
}, Open  ]],
Cell[29572, 1110, 111, 3, 70, "MathCaption",
 CellID->26946],
Cell[CellGroupData[{
Cell[29708, 1117, 102, 3, 70, "Input",
 CellID->21188],
Cell[29813, 1122, 161, 5, 36, "Output",
 CellID->187280716]
}, Open  ]],
Cell[29989, 1130, 128, 4, 70, "MathCaption",
 CellID->8638],
Cell[CellGroupData[{
Cell[30142, 1138, 101, 3, 70, "Input",
 CellID->27663],
Cell[30246, 1143, 177, 5, 36, "Output",
 CellID->126583283]
}, Open  ]],
Cell[30438, 1151, 204, 5, 70, "Text",
 CellID->24515],
Cell[30645, 1158, 1804, 58, 70, "DefinitionBox",
 CellID->11521],
Cell[32452, 1218, 75, 1, 70, "Caption",
 CellID->475],
Cell[32530, 1221, 70, 1, 70, "MathCaption",
 CellID->15615],
Cell[CellGroupData[{
Cell[32625, 1226, 205, 7, 70, "Input",
 CellID->20528],
Cell[32833, 1235, 288, 10, 51, "Output",
 CellID->44979545]
}, Open  ]],
Cell[33136, 1248, 93, 1, 70, "MathCaption",
 CellID->12570],
Cell[CellGroupData[{
Cell[33254, 1253, 103, 3, 70, "Input",
 CellID->905],
Cell[33360, 1258, 159, 5, 36, "Output",
 CellID->333704992]
}, Open  ]],
Cell[33534, 1266, 210, 7, 70, "MathCaption",
 CellID->19557],
Cell[CellGroupData[{
Cell[33769, 1277, 118, 4, 70, "Input",
 CellID->28962],
Cell[33890, 1283, 536, 10, 70, "Message",
 CellID->373567186],
Cell[34429, 1295, 176, 5, 36, "Output",
 CellID->78614419]
}, Open  ]],
Cell[34620, 1303, 79, 1, 70, "MathCaption",
 CellID->12549],
Cell[CellGroupData[{
Cell[34724, 1308, 174, 6, 70, "Input",
 CellID->17469],
Cell[34901, 1316, 164, 5, 36, "Output",
 CellID->301540253]
}, Open  ]],
Cell[35080, 1324, 412, 10, 70, "Text",
 CellID->31923],
Cell[35495, 1336, 414, 9, 70, "Text",
 CellID->28350],
Cell[35912, 1347, 104, 3, 70, "MathCaption",
 CellID->14333],
Cell[CellGroupData[{
Cell[36041, 1354, 189, 7, 70, "Input",
 CellID->22925],
Cell[36233, 1363, 203, 5, 36, "Output",
 CellID->530934566]
}, Open  ]],
Cell[36451, 1371, 172, 6, 70, "MathCaption",
 CellID->19737],
Cell[CellGroupData[{
Cell[36648, 1381, 125, 4, 70, "Input",
 CellID->16336],
Cell[36776, 1387, 176, 5, 36, "Output",
 CellID->151364406]
}, Open  ]],
Cell[36967, 1395, 261, 8, 70, "MathCaption",
 CellID->468393254],
Cell[CellGroupData[{
Cell[37253, 1407, 174, 6, 70, "Input",
 CellID->12393],
Cell[37430, 1415, 175, 5, 36, "Output",
 CellID->771077801]
}, Open  ]],
Cell[37620, 1423, 452, 8, 70, "Text",
 CellID->30714],
Cell[38075, 1433, 1118, 34, 70, "DefinitionBox",
 CellID->28164],
Cell[39196, 1469, 82, 1, 70, "Caption",
 CellID->31591],
Cell[39281, 1472, 133, 5, 70, "MathCaption",
 CellID->19949],
Cell[CellGroupData[{
Cell[39439, 1481, 153, 5, 70, "Input",
 CellID->19135],
Cell[39595, 1488, 215, 6, 36, "Output",
 CellID->251267715]
}, Open  ]],
Cell[39825, 1497, 83, 1, 70, "MathCaption",
 CellID->13337],
Cell[CellGroupData[{
Cell[39933, 1502, 136, 5, 70, "Input",
 CellID->10004],
Cell[40072, 1509, 178, 5, 36, "Output",
 CellID->433192813]
}, Open  ]],
Cell[40265, 1517, 114, 3, 70, "MathCaption",
 CellID->2623],
Cell[CellGroupData[{
Cell[40404, 1524, 102, 3, 70, "Input",
 CellID->28664],
Cell[40509, 1529, 174, 5, 36, "Output",
 CellID->22178275]
}, Open  ]],
Cell[40698, 1537, 585, 16, 70, "Text",
 CellID->25608],
Cell[41286, 1555, 160, 4, 70, "MathCaption",
 CellID->30568],
Cell[CellGroupData[{
Cell[41471, 1563, 145, 4, 70, "Input",
 CellID->19281],
Cell[41619, 1569, 219, 6, 36, "Output",
 CellID->48918553]
}, Open  ]],
Cell[41853, 1578, 982, 28, 70, "DefinitionBox3Col",
 CellID->9152],
Cell[42838, 1608, 72, 1, 70, "Caption",
 CellID->18373],
Cell[42913, 1611, 852, 26, 70, "Text",
 CellID->28652],
Cell[43768, 1639, 829, 25, 70, "Text",
 CellID->8194],
Cell[44600, 1666, 224, 5, 70, "Text",
 CellID->9876],
Cell[44827, 1673, 103, 3, 70, "MathCaption",
 CellID->23826],
Cell[CellGroupData[{
Cell[44955, 1680, 196, 7, 70, "Input",
 CellID->28396],
Cell[45154, 1689, 201, 5, 36, "Output",
 CellID->150947412]
}, Open  ]],
Cell[45370, 1697, 167, 6, 70, "MathCaption",
 CellID->6249],
Cell[CellGroupData[{
Cell[45562, 1707, 148, 5, 70, "Input",
 CellID->14640],
Cell[45713, 1714, 218, 6, 36, "Output",
 CellID->138922340]
}, Open  ]],
Cell[45946, 1723, 163, 6, 70, "MathCaption",
 CellID->18819],
Cell[CellGroupData[{
Cell[46134, 1733, 135, 4, 70, "Input",
 CellID->943],
Cell[46272, 1739, 160, 5, 36, "Output",
 CellID->783520870]
}, Open  ]],
Cell[46447, 1747, 98, 2, 70, "MathCaption",
 CellID->1941],
Cell[CellGroupData[{
Cell[46570, 1753, 148, 5, 70, "Input",
 CellID->20289],
Cell[46721, 1760, 189, 5, 36, "Output",
 CellID->700407683]
}, Open  ]],
Cell[CellGroupData[{
Cell[46947, 1770, 66, 1, 70, "TutorialMoreAboutSection",
 CellID->339521319],
Cell[47016, 1773, 186, 4, 70, "TutorialMoreAbout",
 CellID->171830946]
}, Open  ]],
Cell[CellGroupData[{
Cell[47239, 1782, 72, 1, 70, "RelatedTutorialsSection",
 CellID->107586942],
Cell[47314, 1785, 143, 3, 70, "RelatedTutorials",
 CellID->54483688],
Cell[47460, 1790, 195, 4, 70, "RelatedTutorials",
 CellID->146018610]
}, Open  ]]
}, Open  ]],
Cell[47682, 1798, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

