(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     46843,       1653]
NotebookOptionsPosition[     37647,       1364]
NotebookOutlinePosition[     41305,       1446]
CellTagsIndexPosition[     41210,       1440]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Evaluation of Expressions" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/EvaluationOfExpressionsOverview"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"Evaluation of Expressions\"\>"}, "\<\"related tutorials\"\>", 
       StyleBox["\<\"related tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Compile" :> Documentation`HelpLookup["paclet:ref/Compile"], 
          "False" :> Documentation`HelpLookup["paclet:ref/False"], "True" :> 
          Documentation`HelpLookup["paclet:ref/True"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Compile\"\>", 2->"\<\"False\"\>", 
       3->"\<\"True\"\>"}, "\<\"functions\"\>", 
       StyleBox["\<\"functions\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[TextData[{
 "Compiling ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " Expressions"
}], "Title",
 CellTags->{"S2.6.16", "16.1"},
 CellID->23811],

Cell[TextData[{
 "If you make a definition like ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"f", "[", "x_", "]"}], ":=", 
   RowBox[{"x", " ", 
    RowBox[{
     ButtonBox["Sin",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Sin"], "[", "x", "]"}]}]}]], "InlineFormula"],
 ", ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " will store the expression ",
 Cell[BoxData[
  RowBox[{"x", " ", 
   RowBox[{
    ButtonBox["Sin",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Sin"], "[", "x", "]"}]}]], "InlineFormula"],
 " in a form that can be evaluated for any ",
 Cell[BoxData["x"], "InlineFormula"],
 ". Then when you give a particular value for ",
 Cell[BoxData["x"], "InlineFormula"],
 ", ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " substitutes this value into ",
 Cell[BoxData[
  RowBox[{"x", " ", 
   RowBox[{
    ButtonBox["Sin",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Sin"], "[", "x", "]"}]}]], "InlineFormula"],
 ", and evaluates the result. The internal code that ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " uses to perform this evaluation is set up to work equally well whether the \
value you give for ",
 Cell[BoxData["x"], "InlineFormula"],
 " is a number, a list, an algebraic object, or any other kind of expression. \
"
}], "Text",
 CellID->31322],

Cell[TextData[{
 "Having to take account of all these possibilities inevitably makes the \
evaluation process slower. However, if ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " could ",
 StyleBox["assume",
  FontSlant->"Italic"],
 " that ",
 Cell[BoxData["x"], "InlineFormula"],
 " will be a machine number, then it could avoid many steps, and potentially \
evaluate an expression like ",
 Cell[BoxData[
  RowBox[{"x", " ", 
   RowBox[{
    ButtonBox["Sin",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Sin"], "[", "x", "]"}]}]], "InlineFormula"],
 " much more quickly. "
}], "Text",
 CellID->30333],

Cell[TextData[{
 "Using ",
 Cell[BoxData[
  ButtonBox["Compile",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Compile"]], "InlineFormula"],
 ", you can construct ",
 StyleBox["compiled functions",
  FontSlant->"Italic"],
 " in ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 ", which evaluate ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " expressions assuming that all the parameters which appear are numbers (or \
logical variables). ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Compile",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Compile"], "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["1", "TR"]], ",", 
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["2", "TR"]], ",", 
      StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
    StyleBox["expr", "TI"]}], "]"}]], "InlineFormula"],
 " takes an expression ",
 Cell[BoxData[
  StyleBox["expr", "TI"]], "InlineFormula"],
 " and returns a \[OpenCurlyDoubleQuote]compiled function\
\[CloseCurlyDoubleQuote] which evaluates this expression when given arguments \
",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["x", "1"], ",", " ", 
    SubscriptBox["x", "2"], ",", "\[Ellipsis]"}], TraditionalForm]], 
  "InlineMath"],
 ". "
}], "Text",
 CellID->17673],

Cell[TextData[{
 "In general, ",
 Cell[BoxData[
  ButtonBox["Compile",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Compile"]], "InlineFormula"],
 " creates a ",
 Cell[BoxData[
  ButtonBox["CompiledFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CompiledFunction"]], "InlineFormula"],
 " object which contains a sequence of simple instructions for evaluating the \
compiled function. The instructions are chosen to be close to those found in \
the machine code of a typical computer, and can thus be executed quickly. "
}], "Text",
 CellID->4664],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     ButtonBox["Compile",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Compile"], "[", "{", 
     SubscriptBox[
      StyleBox["x", "TI"], "1"], ",", 
     SubscriptBox[
      StyleBox["x", "TI"], "2"], ",", 
     StyleBox["\[Ellipsis]", "TR"], "}", ",", 
     StyleBox["expr", "TI"], "]"}], Cell[TextData[{
     "create a compiled function which evaluates ",
     Cell[BoxData[
      StyleBox["expr", "TI"]], "InlineFormula"],
     " for numerical values of the ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["i", "TI"]]], "InlineFormula"]
    }], "TableText"]}
  }]], "DefinitionBox",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {False, False}, 
   "RowsIndexed" -> {}},
 GridBoxSpacings->{
  "Columns" -> {0.28, {0.56}, 0.28}, "ColumnsIndexed" -> {}, 
   "Rows" -> {0.2, {0.16}, 0.2}, "RowsIndexed" -> {}}},
 CellID->15141],

Cell["Creating compiled functions. ", "Caption",
 CellID->7711],

Cell[TextData[{
 "This defines ",
 Cell[BoxData["f"], "InlineFormula"],
 " to be a pure function which evaluates ",
 Cell[BoxData[
  RowBox[{"x", " ", 
   RowBox[{
    ButtonBox["Sin",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Sin"], "[", "x", "]"}]}]], "InlineFormula"],
 " for any ",
 Cell[BoxData["x"], "InlineFormula"],
 ". "
}], "MathCaption",
 CellID->28253],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"f", "=", 
  RowBox[{"Function", "[", 
   RowBox[{
    RowBox[{"{", "x", "}"}], ",", 
    RowBox[{"x", " ", 
     RowBox[{"Sin", "[", "x", "]"}]}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->6868],

Cell[BoxData[
 RowBox[{"Function", "[", 
  RowBox[{
   RowBox[{"{", "x", "}"}], ",", 
   RowBox[{"x", " ", 
    RowBox[{"Sin", "[", "x", "]"}]}]}], "]"}]], "Output",
 ImageSize->{159, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->506938155]
}, Open  ]],

Cell[TextData[{
 "This creates a compiled function for evaluating ",
 Cell[BoxData[
  RowBox[{"x", " ", 
   RowBox[{
    ButtonBox["Sin",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Sin"], "[", "x", "]"}]}]], "InlineFormula"],
 ". "
}], "MathCaption",
 CellID->27644],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"fc", "=", 
  RowBox[{"Compile", "[", 
   RowBox[{
    RowBox[{"{", "x", "}"}], ",", 
    RowBox[{"x", " ", 
     RowBox[{"Sin", "[", "x", "]"}]}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->32662],

Cell[BoxData[
 TagBox[
  RowBox[{"CompiledFunction", "[", 
   RowBox[{
    RowBox[{"{", "x", "}"}], ",", 
    RowBox[{"x", " ", 
     RowBox[{"Sin", "[", "x", "]"}]}], ",", "\<\"-CompiledCode-\"\>"}], "]"}],
  
  False,
  Editable->False]], "Output",
 ImageSize->{324, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->42440658]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData["f"], "InlineFormula"],
 " and ",
 Cell[BoxData["fc"], "InlineFormula"],
 " yield the same results, but ",
 Cell[BoxData["fc"], "InlineFormula"],
 " runs faster when the argument you give is a number. "
}], "MathCaption",
 CellID->20037],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"f", "[", "2.5", "]"}], ",", 
   RowBox[{"fc", "[", "2.5", "]"}]}], "}"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->12859],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.4961803602598913`", ",", "1.4961803602598913`"}], 
  "}"}]], "Output",
 ImageSize->{128, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->487674970]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Compile",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Compile"]], "InlineFormula"],
 " is useful in situations where you have to evaluate a particular numerical \
or logical expression many times. By taking the time to call ",
 Cell[BoxData[
  ButtonBox["Compile",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Compile"]], "InlineFormula"],
 ", you can get a compiled function which can be executed more quickly than \
an ordinary ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " function. "
}], "Text",
 CellID->9741],

Cell[TextData[{
 "For simple expressions such as ",
 Cell[BoxData[
  RowBox[{"x", " ", 
   RowBox[{
    ButtonBox["Sin",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Sin"], "[", "x", "]"}]}]], "InlineFormula"],
 ", there is usually little difference between the execution speed for \
ordinary and compiled functions. However, as the size of the expressions \
involved increases, the advantage of compilation also increases. For large \
expressions, compilation can speed up execution by a factor as large as 20. "
}], "Text",
 CellID->27529],

Cell[TextData[{
 "Compilation makes the biggest difference for expressions containing a large \
number of simple, say arithmetic, functions. For more complicated functions, \
such as ",
 Cell[BoxData[
  ButtonBox["BesselK",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BesselK"]], "InlineFormula"],
 " or ",
 Cell[BoxData[
  ButtonBox["Eigenvalues",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Eigenvalues"]], "InlineFormula"],
 ", most of the computation time is spent executing internal ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " algorithms, on which compilation has no effect. "
}], "Text",
 CellID->778],

Cell[TextData[{
 "This creates a compiled function for finding values of the tenth Legendre \
polynomial. The ",
 Cell[BoxData[
  ButtonBox["Evaluate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Evaluate"]], "InlineFormula"],
 " tells ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " to construct the polynomial explicitly before doing compilation. "
}], "MathCaption",
 CellID->12316],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"pc", "=", 
  RowBox[{"Compile", "[", 
   RowBox[{
    RowBox[{"{", "x", "}"}], ",", 
    RowBox[{"Evaluate", "[", 
     RowBox[{"LegendreP", "[", 
      RowBox[{"10", ",", "x"}], "]"}], "]"}]}], "]"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->3035],

Cell[BoxData[
 TagBox[
  RowBox[{"CompiledFunction", "[", 
   RowBox[{
    RowBox[{"{", "x", "}"}], ",", 
    RowBox[{
     FractionBox["1", "256"], " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "63"}], "+", 
       RowBox[{"3465", " ", 
        SuperscriptBox["x", "2"]}], "-", 
       RowBox[{"30030", " ", 
        SuperscriptBox["x", "4"]}], "+", 
       RowBox[{"90090", " ", 
        SuperscriptBox["x", "6"]}], "-", 
       RowBox[{"109395", " ", 
        SuperscriptBox["x", "8"]}], "+", 
       RowBox[{"46189", " ", 
        SuperscriptBox["x", "10"]}]}], ")"}]}], 
    ",", "\<\"-CompiledCode-\"\>"}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{521, 52},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->248586927]
}, Open  ]],

Cell[TextData[{
 "This finds the value of the tenth Legendre polynomial with argument ",
 Cell[BoxData["0.4"], "InlineFormula"],
 ". "
}], "MathCaption",
 CellID->1842],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"pc", "[", "0.4", "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->288],

Cell[BoxData["0.09683906440000015`"], "Output",
 ImageSize->{67, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->196716958]
}, Open  ]],

Cell["\<\
This uses built\[Hyphen]in numerical code. \
\>", "MathCaption",
 CellID->9040],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LegendreP", "[", 
  RowBox[{"10", ",", "0.4"}], "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->8942],

Cell[BoxData["0.0968390643999999`"], "Output",
 ImageSize->{67, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->679770097]
}, Open  ]],

Cell[TextData[{
 "Even though you can use compilation to speed up numerical functions that \
you write, you should still try to use built\[Hyphen]in ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " functions whenever possible. Built\[Hyphen]in functions will usually run \
faster than any compiled ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " programs you can create. In addition, they typically use more extensive \
algorithms, with more complete control over numerical precision and so on. "
}], "Text",
 CellID->22648],

Cell[TextData[{
 "You should realize that built\[Hyphen]in ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " functions quite often themselves use ",
 Cell[BoxData[
  ButtonBox["Compile",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Compile"]], "InlineFormula"],
 ". Thus, for example, ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " by default automatically uses ",
 Cell[BoxData[
  ButtonBox["Compile",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Compile"]], "InlineFormula"],
 " on the expression you tell it to integrate. Similarly, functions like ",
 Cell[BoxData[
  ButtonBox["Plot",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Plot"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["Plot3D",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Plot3D"]], "InlineFormula"],
 " use ",
 Cell[BoxData[
  ButtonBox["Compile",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Compile"]], "InlineFormula"],
 " on the expressions you ask them to plot. Built\[Hyphen]in functions that \
use ",
 Cell[BoxData[
  ButtonBox["Compile",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Compile"]], "InlineFormula"],
 " typically have the option ",
 Cell[BoxData[
  ButtonBox["Compiled",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Compiled"]], "InlineFormula"],
 ". Setting ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Compiled",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Compiled"], "->", 
   ButtonBox["False",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/False"]}]], "InlineFormula"],
 " tells the functions not to use ",
 Cell[BoxData[
  ButtonBox["Compile",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Compile"]], "InlineFormula"],
 ". "
}], "Text",
 CellID->27446],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     ButtonBox["Compile",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Compile"], "[", "{", "{", 
     SubscriptBox[
      StyleBox["x", "TI"], "1"], ",", 
     SubscriptBox[
      StyleBox["t", "TI"], "1"], "}", ",", "{", 
     SubscriptBox[
      StyleBox["x", "TI"], "2"], ",", 
     SubscriptBox[
      StyleBox["t", "TI"], "2"], "}", ",", 
     StyleBox["\[Ellipsis]", "TR"], "}", ",", 
     StyleBox["expr", "TI"], "]"}], "\[SpanFromLeft]"},
   {"", Cell[TextData[{
     "compile ",
     Cell[BoxData[
      StyleBox["expr", "TI"]], "InlineFormula"],
     " assuming that ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["i", "TI"]]], "InlineFormula"],
     " is of type ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["t", "TI"], 
       StyleBox["i", "TI"]]], "InlineFormula"]
    }], "TableText"]},
   {
    RowBox[{
     ButtonBox["Compile",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Compile"], "[", "{", "{", 
     SubscriptBox[
      StyleBox["x", "TI"], "1"], ",", 
     SubscriptBox[
      StyleBox["t", "TI"], "1"], ",", 
     SubscriptBox[
      StyleBox["n", "TI"], "1"], "}", ",", "{", 
     SubscriptBox[
      StyleBox["x", "TI"], "2"], ",", 
     SubscriptBox[
      StyleBox["t", "TI"], "2"], ",", 
     SubscriptBox[
      StyleBox["n", "TI"], "2"], "}", ",", 
     StyleBox["\[Ellipsis]", "TR"], "}", ",", 
     StyleBox["expr", "TI"], "]"}], "\[SpanFromLeft]"},
   {"", Cell[TextData[{
     "compile ",
     Cell[BoxData[
      StyleBox["expr", "TI"]], "InlineFormula"],
     " assuming that ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["i", "TI"]]], "InlineFormula"],
     " is a rank ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["n", "TI"], 
       StyleBox["i", "TI"]]], "InlineFormula"],
     " array of objects each of type ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["t", "TI"], 
       StyleBox["i", "TI"]]], "InlineFormula"]
    }], "TableText"]},
   {
    RowBox[{
     ButtonBox["Compile",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Compile"], "[", 
     StyleBox["vars", "TI"], ",", 
     StyleBox["expr", "TI"], ",", "{", "{", 
     SubscriptBox[
      StyleBox["p", "TI"], "1"], ",", 
     SubscriptBox[
      StyleBox["pt", "TI"], "1"], "}", ",", 
     StyleBox["\[Ellipsis]", "TR"], "}", "]"}], "\[SpanFromLeft]"},
   {"", Cell[TextData[{
     "compile ",
     Cell[BoxData[
      StyleBox["expr", "TI"]], "InlineFormula"],
     ", assuming that subexpressions which match ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["p", "TI"], 
       StyleBox["i", "TI"]]], "InlineFormula"],
     " are of type ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["pt", "TI"], 
       StyleBox["i", "TI"]]], "InlineFormula"]
    }], "TableText"]},
   {"_Integer", Cell["machine\[Hyphen]size integer", "TableText"]},
   {"_Real", Cell["\<\
machine\[Hyphen]precision approximate real number\
\>", "TableText"]},
   {"_Complex", Cell["\<\
machine\[Hyphen]precision approximate complex number\
\>", "TableText"]},
   {
    RowBox[{
     ButtonBox["True",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/True"], "|", 
     ButtonBox["False",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/False"]}], Cell[
    "logical variable", "TableText"]}
  }]], "DefinitionBox",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {
    False, False, False, False, True, False, True, False, False, {False}, 
     False}, "RowsIndexed" -> {}},
 GridBoxItemSize->{"Columns" -> {18, {
      Scaled[0.6]}}, "ColumnsIndexed" -> {}, "Rows" -> {{1.}}, 
   "RowsIndexed" -> {}}},
 CellID->23805],

Cell["Specifying types for compilation. ", "Caption",
 CellID->15890],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Compile",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Compile"]], "InlineFormula"],
 " works by making assumptions about the types of objects that occur in \
evaluating the expression you give. The default assumption is that all \
variables in the expression are approximate real numbers. "
}], "Text",
 CellID->6729],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Compile",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Compile"]], "InlineFormula"],
 " nevertheless also allows integers, complex numbers and logical variables \
(",
 Cell[BoxData[
  ButtonBox["True",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/True"]], "InlineFormula"],
 " or ",
 Cell[BoxData[
  ButtonBox["False",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/False"]], "InlineFormula"],
 "), as well as arrays of numbers. You can specify the type of a particular \
variable by giving a pattern which matches only values that have that type. \
Thus, for example, you can use the pattern ",
 Cell[BoxData["_Integer"], "InlineFormula"],
 " to specify the integer type. Similarly, you can use ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["True",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/True"], "|", 
   ButtonBox["False",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/False"]}]], "InlineFormula"],
 " to specify a logical variable that must be either ",
 Cell[BoxData[
  ButtonBox["True",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/True"]], "InlineFormula"],
 " or ",
 Cell[BoxData[
  ButtonBox["False",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/False"]], "InlineFormula"],
 ". "
}], "Text",
 CellID->24370],

Cell[TextData[{
 "This compiles the expression ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"5", "i"}], "+", "j"}]], "InlineFormula"],
 " with the assumption that ",
 Cell[BoxData["i"], "InlineFormula"],
 " and ",
 Cell[BoxData["j"], "InlineFormula"],
 " are integers. "
}], "MathCaption",
 CellID->15350],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Compile", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"i", ",", "_Integer"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"j", ",", "_Integer"}], "}"}]}], "}"}], ",", 
   RowBox[{
    RowBox[{"5", " ", "i"}], "+", "j"}]}], "]"}]], "Input",
 CellLabel->"In[7]:=",
 CellID->15006],

Cell[BoxData[
 TagBox[
  RowBox[{"CompiledFunction", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"i", ",", "j"}], "}"}], ",", 
    RowBox[{
     RowBox[{"5", " ", "i"}], "+", "j"}], ",", "\<\"-CompiledCode-\"\>"}], 
   "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{327, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->35532450]
}, Open  ]],

Cell["This yields an integer result. ", "MathCaption",
 CellID->24393],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"%", "[", 
  RowBox[{"8", ",", "7"}], "]"}]], "Input",
 CellLabel->"In[8]:=",
 CellID->3548],

Cell[BoxData["47"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->539226852]
}, Open  ]],

Cell["\<\
This compiles an expression that performs an operation on a matrix of \
integers. \
\>", "MathCaption",
 CellID->12623],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Compile", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{"m", ",", "_Integer", ",", "2"}], "}"}], "}"}], ",", 
   RowBox[{"Apply", "[", 
    RowBox[{"Plus", ",", 
     RowBox[{"Flatten", "[", "m", "]"}]}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[9]:=",
 CellID->24084],

Cell[BoxData[
 TagBox[
  RowBox[{"CompiledFunction", "[", 
   RowBox[{
    RowBox[{"{", "m", "}"}], ",", 
    RowBox[{"Plus", "@@", 
     RowBox[{"Flatten", "[", "m", "]"}]}], ",", "\<\"-CompiledCode-\"\>"}], 
   "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{389, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[9]=",
 CellID->347015329]
}, Open  ]],

Cell["\<\
The list operations are now carried out in a compiled way, and the result is \
an integer. \
\>", "MathCaption",
 CellID->18756],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"%", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1", ",", "2", ",", "3"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"7", ",", "8", ",", "9"}], "}"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[10]:=",
 CellID->11840],

Cell[BoxData["30"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[10]=",
 CellID->41753969]
}, Open  ]],

Cell[TextData[{
 "The types that ",
 Cell[BoxData[
  ButtonBox["Compile",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Compile"]], "InlineFormula"],
 " handles correspond essentially to the types that computers typically \
handle at a machine\[Hyphen]code level. Thus, for example, ",
 Cell[BoxData[
  ButtonBox["Compile",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Compile"]], "InlineFormula"],
 " can handle approximate real numbers that have machine precision, but it \
cannot handle arbitrary\[Hyphen]precision numbers. In addition, if you \
specify that a particular variable is an integer, ",
 Cell[BoxData[
  ButtonBox["Compile",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Compile"]], "InlineFormula"],
 " generates code only for the case when the integer is of \
\[OpenCurlyDoubleQuote]machine size\[CloseCurlyDoubleQuote], typically \
between ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[PlusMinus]", 
    SuperscriptBox["2", "31"]}], TraditionalForm]], "InlineMath"],
 ". "
}], "Text",
 CellID->7376],

Cell[TextData[{
 "When the expression you ask to compile involves only standard arithmetic \
and logical operations, ",
 Cell[BoxData[
  ButtonBox["Compile",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Compile"]], "InlineFormula"],
 " can deduce the types of objects generated at every step simply from the \
types of the input variables. However, if you call other functions, ",
 Cell[BoxData[
  ButtonBox["Compile",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Compile"]], "InlineFormula"],
 " will typically not know what type of value they return. If you do not \
specify otherwise, ",
 Cell[BoxData[
  ButtonBox["Compile",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Compile"]], "InlineFormula"],
 " assumes that any other function yields an approximate real number value. \
You can, however, also give an explicit list of patterns, specifying what \
type to assume for an expression that matches a particular pattern. "
}], "Text",
 CellID->13931],

Cell["\<\
This defines a function which yields an integer result when given an integer \
argument. \
\>", "MathCaption",
 CellID->26308],

Cell[BoxData[
 RowBox[{
  RowBox[{"com", "[", "i_", "]"}], ":=", 
  RowBox[{"Binomial", "[", 
   RowBox[{
    RowBox[{"2", "i"}], ",", "i"}], "]"}]}]], "Input",
 CellLabel->"In[11]:=",
 CellID->16944],

Cell[TextData[{
 "This compiles ",
 Cell[BoxData[
  RowBox[{"x", "^", 
   RowBox[{"com", "[", "i", "]"}]}]], "InlineFormula"],
 " using the assumption that ",
 Cell[BoxData[
  RowBox[{"com", "[", "_", "]"}]], "InlineFormula"],
 " is always an integer. "
}], "MathCaption",
 CellID->32439],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Compile", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"{", 
      RowBox[{"i", ",", "_Integer"}], "}"}]}], "}"}], ",", 
   RowBox[{"x", "^", 
    RowBox[{"com", "[", "i", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"com", "[", "_", "]"}], ",", "_Integer"}], "}"}], "}"}]}], 
  "]"}]], "Input",
 CellLabel->"In[12]:=",
 CellID->24626],

Cell[BoxData[
 TagBox[
  RowBox[{"CompiledFunction", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"x", ",", "i"}], "}"}], ",", 
    SuperscriptBox["x", 
     RowBox[{"com", "[", "i", "]"}]], ",", "\<\"-CompiledCode-\"\>"}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{328, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[12]=",
 CellID->1005455266]
}, Open  ]],

Cell["This evaluates the compiled function. ", "MathCaption",
 CellID->5537],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"%", "[", 
  RowBox[{"5.6", ",", "1"}], "]"}]], "Input",
 CellLabel->"In[13]:=",
 CellID->21538],

Cell[BoxData["31.359999999999996`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[13]=",
 CellID->217772394]
}, Open  ]],

Cell[TextData[{
 "The idea of ",
 Cell[BoxData[
  ButtonBox["Compile",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Compile"]], "InlineFormula"],
 " is to create a function which is optimized for certain types of arguments. \
",
 Cell[BoxData[
  ButtonBox["Compile",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Compile"]], "InlineFormula"],
 " is nevertheless set up so that the functions it creates work with whatever \
types of arguments they are given. When the optimization cannot be used, a \
standard ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " expression is evaluated to find the value of the function. "
}], "Text",
 CellID->2082],

Cell["\<\
Here is a compiled function for taking the square root of a variable. \
\>", "MathCaption",
 CellID->22929],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sq", "=", 
  RowBox[{"Compile", "[", 
   RowBox[{
    RowBox[{"{", "x", "}"}], ",", 
    RowBox[{"Sqrt", "[", "x", "]"}]}], "]"}]}]], "Input",
 CellLabel->"In[14]:=",
 CellID->16541],

Cell[BoxData[
 TagBox[
  RowBox[{"CompiledFunction", "[", 
   RowBox[{
    RowBox[{"{", "x", "}"}], ",", 
    SqrtBox["x"], ",", "\<\"-CompiledCode-\"\>"}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{296, 22},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[14]=",
 CellID->412970740]
}, Open  ]],

Cell["If you give a real number argument, optimized code is used. ", \
"MathCaption",
 CellID->31115],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sq", "[", "4.5", "]"}]], "Input",
 CellLabel->"In[15]:=",
 CellID->4639],

Cell[BoxData["2.1213203435596424`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[15]=",
 CellID->75401665]
}, Open  ]],

Cell[TextData[{
 "The compiled code cannot be used, so ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " prints a warning, then just evaluates the original symbolic expression. "
}], "MathCaption",
 CellID->22704],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sq", "[", 
  RowBox[{"1", "+", "u"}], "]"}]], "Input",
 CellLabel->"In[16]:=",
 CellID->9930],

Cell[BoxData[
 RowBox[{
  RowBox[{"CompiledFunction", "::", "\<\"cfsa\"\>"}], ":", 
  " ", "\<\"Argument \\!\\(1 + u\\) at position \\!\\(1\\) should be a \\!\\(\
\\\"machine-size real number\\\"\\). \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/CompiledFunction/cfsa\\\
\", ButtonNote -> \\\"CompiledFunction::cfsa\\\"]\\)\"\>"}]], "Message", \
"MSG",
 CellLabel->"During evaluation of In[16]:=",
 CellID->398997549],

Cell[BoxData[
 SqrtBox[
  RowBox[{"1", "+", "u"}]]], "Output",
 ImageSize->{48, 17},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[16]=",
 CellID->431318443]
}, Open  ]],

Cell[TextData[{
 "The compiled code generated by ",
 Cell[BoxData[
  ButtonBox["Compile",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Compile"]], "InlineFormula"],
 " must make assumptions not only about the types of arguments you will \
supply, but also about the types of all objects that arise during the \
execution of the code. Sometimes these types depend on the actual ",
 StyleBox["values",
  FontSlant->"Italic"],
 " of the arguments you specify. Thus, for example, ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Sqrt",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Sqrt"], "[", 
   StyleBox["x", "TI"], "]"}]], "InlineFormula"],
 " yields a real number result for real ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " if ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " is not negative, but yields a complex number if ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " is negative. "
}], "Text",
 CellID->31673],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Compile",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Compile"]], "InlineFormula"],
 " always makes a definite assumption about the type returned by a particular \
function. If this assumption turns out to be invalid in a particular case \
when the code generated by ",
 Cell[BoxData[
  ButtonBox["Compile",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Compile"]], "InlineFormula"],
 " is executed, then ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " simply abandons the compiled code in this case, and evaluates an ordinary \
",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " expression to get the result. "
}], "Text",
 CellID->22386],

Cell[TextData[{
 "The compiled code does not expect a complex number, so ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " has to revert to explicitly evaluating the original symbolic expression. "
}], "MathCaption",
 CellID->5021],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sq", "[", 
  RowBox[{"-", "4.5"}], "]"}]], "Input",
 CellLabel->"In[17]:=",
 CellID->28745],

Cell[BoxData[
 RowBox[{
  RowBox[{"CompiledFunction", "::", "\<\"cfn\"\>"}], ":", 
  " ", "\<\"Numerical error encountered at instruction \\!\\(2\\); proceeding \
with uncompiled evaluation. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/CompiledFunction/cfn\\\", ButtonNote -> \\\
\"CompiledFunction::cfn\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[17]:=",
 CellID->704415293],

Cell[BoxData[
 RowBox[{"0.`", "\[InvisibleSpace]", "+", 
  RowBox[{"2.1213203435596424`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{91, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[17]=",
 CellID->621729897]
}, Open  ]],

Cell[TextData[{
 "An important feature of ",
 Cell[BoxData[
  ButtonBox["Compile",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Compile"]], "InlineFormula"],
 " is that it can handle not only mathematical expressions, but also various \
simple ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " programs. Thus, for example, ",
 Cell[BoxData[
  ButtonBox["Compile",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Compile"]], "InlineFormula"],
 " can handle conditionals and control flow structures. "
}], "Text",
 CellID->6270],

Cell[TextData[{
 "In all cases, ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Compile",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Compile"], "[", 
   RowBox[{
    StyleBox["vars", "TI"], ",", 
    StyleBox["expr", "TI"]}], "]"}]], "InlineFormula"],
 " holds its arguments unevaluated. This means that you can explicitly give a \
\[OpenCurlyDoubleQuote]program\[CloseCurlyDoubleQuote] as the expression to \
compile. "
}], "Text",
 CellID->5829],

Cell[TextData[{
 "This creates a compiled version of a ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " program which implements Newton\[CloseCurlyQuote]s approximation to square \
roots. "
}], "MathCaption",
 CellID->26777],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"newt", "=", 
  RowBox[{"Compile", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"x", ",", 
      RowBox[{"{", 
       RowBox[{"n", ",", "_Integer"}], "}"}]}], "}"}], ",", 
    RowBox[{"Module", "[", 
     RowBox[{
      RowBox[{"{", "t", "}"}], ",", 
      RowBox[{
       RowBox[{"t", "=", "x"}], ";", 
       RowBox[{"Do", "[", 
        RowBox[{
         RowBox[{"t", "=", 
          RowBox[{
           RowBox[{"(", 
            RowBox[{"t", "+", 
             RowBox[{"x", "/", "t"}]}], ")"}], "/", "2"}]}], ",", 
         RowBox[{"{", "n", "}"}]}], "]"}], ";", "t"}]}], "]"}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[18]:=",
 CellID->15573],

Cell[BoxData[
 TagBox[
  RowBox[{"CompiledFunction", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"x", ",", "n"}], "}"}], ",", 
    RowBox[{"Module", "[", 
     RowBox[{
      RowBox[{"{", "t", "}"}], ",", 
      RowBox[{
       RowBox[{"t", "=", "x"}], ";", 
       RowBox[{"Do", "[", 
        RowBox[{
         RowBox[{"t", "=", 
          RowBox[{
           FractionBox["1", "2"], " ", 
           RowBox[{"(", 
            RowBox[{"t", "+", 
             FractionBox["x", "t"]}], ")"}]}]}], ",", 
         RowBox[{"{", "n", "}"}]}], "]"}], ";", "t"}]}], "]"}], 
    ",", "\<\"-CompiledCode-\"\>"}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{574, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[18]=",
 CellID->33467843]
}, Open  ]],

Cell["This executes the compiled code. ", "MathCaption",
 CellID->16512],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"newt", "[", 
  RowBox[{"2.4", ",", "6"}], "]"}]], "Input",
 CellLabel->"In[19]:=",
 CellID->23986],

Cell[BoxData["1.5491933384829668`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[19]=",
 CellID->481805538]
}, Open  ]],

Cell[CellGroupData[{

Cell["RELATED TUTORIALS", "RelatedTutorialsSection",
 CellID->193618375],

Cell[TextData[ButtonBox["Evaluation of Expressions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/EvaluationOfExpressionsOverview"]], "RelatedTutorials",
 CellID->14411223]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Compiling Mathematica Expressions - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, "Overview" -> {"toc" -> Cell[
      BoxData[
       ButtonBox[
        StyleBox["Evaluation of Expressions", "OverviewNavText"], BaseStyle -> 
        "Link", ButtonData -> 
        "paclet:tutorial/EvaluationOfExpressionsOverview"]], "Text", 
      FontFamily -> "Verdana"], "prevnext" -> Cell[
      TextData[{
        ButtonBox[
         StyleBox["\[FilledLeftTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> "paclet:tutorial/InterruptsAndAborts"], 
        "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]", 
        ButtonBox[
         StyleBox["\[FilledRightTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> "paclet:tutorial/ManipulatingCompiledCode"]}], 
      "Text", FontFamily -> "Verdana"]}, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 33, 15.7487694}", "context" -> "", 
    "keywords" -> {
     "compilation", "Compile", "Compiled", "compiled functions", "efficiency",
       "functions", "machine-size numbers", "numerical precision", "optimize",
       "procedures", "routines", "speed", "types"}, "index" -> True, "label" -> 
    "Mathematica Tutorial", "language" -> "en", "paclet" -> "Mathematica", 
    "status" -> "None", "summary" -> 
    "If you make a definition like f[x_]:=x Sin[x], Mathematica will store \
the expression x Sin[x] in a form that can be evaluated for any x. Then when \
you give a particular value for x, Mathematica substitutes this value into x \
Sin[x], and evaluates the result. The internal code that Mathematica uses to \
perform this evaluation is set up to work equally well whether the value you \
give for x is a number, a list, an algebraic object, or any other kind of \
expression. Having to take account of all these possibilities inevitably \
makes the evaluation process slower. However, if Mathematica could assume \
that x will be a machine number, then it could avoid many steps, and \
potentially evaluate an expression like x Sin[x] much more quickly. Using \
Compile, you can construct compiled functions in Mathematica, which evaluate \
Mathematica expressions assuming that all the parameters which appear are \
numbers (or logical variables). Compile[{x_1,x_2,...},expr] takes an \
expression expr and returns a 'compiled function' which evaluates this \
expression when given arguments x_1, x_2,.... ", "synonyms" -> {"2.6.15"}, 
    "title" -> "Compiling Mathematica Expressions", "type" -> "Tutorial", 
    "uri" -> "tutorial/CompilingMathematicaExpressions"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "S2.6.16"->{
  Cell[1989, 60, 156, 7, 70, "Title",
   CellTags->{"S2.6.16", "16.1"},
   CellID->23811]},
 "16.1"->{
  Cell[1989, 60, 156, 7, 70, "Title",
   CellTags->{"S2.6.16", "16.1"},
   CellID->23811]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"S2.6.16", 40992, 1429},
 {"16.1", 41095, 1433}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 1364, 33, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[1989, 60, 156, 7, 70, "Title",
 CellTags->{"S2.6.16", "16.1"},
 CellID->23811],
Cell[2148, 69, 1314, 43, 70, "Text",
 CellID->31322],
Cell[3465, 114, 615, 20, 70, "Text",
 CellID->30333],
Cell[4083, 136, 1319, 47, 70, "Text",
 CellID->17673],
Cell[5405, 185, 562, 15, 70, "Text",
 CellID->4664],
Cell[5970, 202, 972, 29, 70, "DefinitionBox",
 CellID->15141],
Cell[6945, 233, 63, 1, 70, "Caption",
 CellID->7711],
Cell[7011, 236, 375, 14, 70, "MathCaption",
 CellID->28253],
Cell[CellGroupData[{
Cell[7411, 254, 229, 8, 28, "Input",
 CellID->6868],
Cell[7643, 264, 296, 10, 36, "Output",
 CellID->506938155]
}, Open  ]],
Cell[7954, 277, 276, 10, 70, "MathCaption",
 CellID->27644],
Cell[CellGroupData[{
Cell[8255, 291, 230, 8, 70, "Input",
 CellID->32662],
Cell[8488, 301, 380, 14, 36, "Output",
 CellID->42440658]
}, Open  ]],
Cell[8883, 318, 267, 8, 70, "MathCaption",
 CellID->20037],
Cell[CellGroupData[{
Cell[9175, 330, 174, 6, 70, "Input",
 CellID->12859],
Cell[9352, 338, 243, 8, 36, "Output",
 CellID->487674970]
}, Open  ]],
Cell[9610, 349, 576, 17, 70, "Text",
 CellID->9741],
Cell[10189, 368, 549, 13, 70, "Text",
 CellID->27529],
Cell[10741, 383, 628, 18, 70, "Text",
 CellID->778],
Cell[11372, 403, 397, 12, 70, "MathCaption",
 CellID->12316],
Cell[CellGroupData[{
Cell[11794, 419, 275, 9, 70, "Input",
 CellID->3035],
Cell[12072, 430, 804, 27, 73, "Output",
 CellID->248586927]
}, Open  ]],
Cell[12891, 460, 168, 5, 70, "MathCaption",
 CellID->1842],
Cell[CellGroupData[{
Cell[13084, 469, 94, 3, 70, "Input",
 CellID->288],
Cell[13181, 474, 177, 5, 36, "Output",
 CellID->196716958]
}, Open  ]],
Cell[13373, 482, 89, 3, 70, "MathCaption",
 CellID->9040],
Cell[CellGroupData[{
Cell[13487, 489, 126, 4, 70, "Input",
 CellID->8942],
Cell[13616, 495, 176, 5, 36, "Output",
 CellID->679770097]
}, Open  ]],
Cell[13807, 503, 537, 12, 70, "Text",
 CellID->22648],
Cell[14347, 517, 1791, 61, 70, "Text",
 CellID->27446],
Cell[16141, 580, 3775, 123, 70, "DefinitionBox",
 CellID->23805],
Cell[19919, 705, 69, 1, 70, "Caption",
 CellID->15890],
Cell[19991, 708, 366, 9, 70, "Text",
 CellID->6729],
Cell[20360, 719, 1290, 41, 70, "Text",
 CellID->24370],
Cell[21653, 762, 300, 11, 70, "MathCaption",
 CellID->15350],
Cell[CellGroupData[{
Cell[21978, 777, 340, 12, 70, "Input",
 CellID->15006],
Cell[22321, 791, 400, 15, 36, "Output",
 CellID->35532450]
}, Open  ]],
Cell[22736, 809, 70, 1, 70, "MathCaption",
 CellID->24393],
Cell[CellGroupData[{
Cell[22831, 814, 115, 4, 70, "Input",
 CellID->3548],
Cell[22949, 820, 159, 5, 36, "Output",
 CellID->539226852]
}, Open  ]],
Cell[23123, 828, 129, 4, 70, "MathCaption",
 CellID->12623],
Cell[CellGroupData[{
Cell[23277, 836, 312, 10, 70, "Input",
 CellID->24084],
Cell[23592, 848, 390, 14, 36, "Output",
 CellID->347015329]
}, Open  ]],
Cell[23997, 865, 138, 4, 70, "MathCaption",
 CellID->18756],
Cell[CellGroupData[{
Cell[24160, 873, 263, 9, 70, "Input",
 CellID->11840],
Cell[24426, 884, 159, 5, 36, "Output",
 CellID->41753969]
}, Open  ]],
Cell[24600, 892, 1024, 28, 70, "Text",
 CellID->7376],
Cell[25627, 922, 967, 23, 70, "Text",
 CellID->13931],
Cell[26597, 947, 136, 4, 70, "MathCaption",
 CellID->26308],
Cell[26736, 953, 200, 7, 70, "Input",
 CellID->16944],
Cell[26939, 962, 288, 10, 70, "MathCaption",
 CellID->32439],
Cell[CellGroupData[{
Cell[27252, 976, 417, 15, 70, "Input",
 CellID->24626],
Cell[27672, 993, 407, 14, 39, "Output",
 CellID->1005455266]
}, Open  ]],
Cell[28094, 1010, 76, 1, 70, "MathCaption",
 CellID->5537],
Cell[CellGroupData[{
Cell[28195, 1015, 119, 4, 70, "Input",
 CellID->21538],
Cell[28317, 1021, 177, 5, 36, "Output",
 CellID->217772394]
}, Open  ]],
Cell[28509, 1029, 658, 19, 70, "Text",
 CellID->2082],
Cell[29170, 1050, 117, 3, 70, "MathCaption",
 CellID->22929],
Cell[CellGroupData[{
Cell[29312, 1057, 206, 7, 70, "Input",
 CellID->16541],
Cell[29521, 1066, 335, 12, 43, "Output",
 CellID->412970740]
}, Open  ]],
Cell[29871, 1081, 101, 2, 70, "MathCaption",
 CellID->31115],
Cell[CellGroupData[{
Cell[29997, 1087, 96, 3, 70, "Input",
 CellID->4639],
Cell[30096, 1092, 176, 5, 36, "Output",
 CellID->75401665]
}, Open  ]],
Cell[30287, 1100, 218, 6, 70, "MathCaption",
 CellID->22704],
Cell[CellGroupData[{
Cell[30530, 1110, 117, 4, 70, "Input",
 CellID->9930],
Cell[30650, 1116, 501, 10, 70, "Message",
 CellID->398997549],
Cell[31154, 1128, 193, 7, 38, "Output",
 CellID->431318443]
}, Open  ]],
Cell[31362, 1138, 967, 29, 70, "Text",
 CellID->31673],
Cell[32332, 1169, 702, 21, 70, "Text",
 CellID->22386],
Cell[33037, 1192, 236, 6, 70, "MathCaption",
 CellID->5021],
Cell[CellGroupData[{
Cell[33298, 1202, 115, 4, 70, "Input",
 CellID->28745],
Cell[33416, 1208, 488, 9, 70, "Message",
 CellID->704415293],
Cell[33907, 1219, 257, 7, 36, "Output",
 CellID->621729897]
}, Open  ]],
Cell[34179, 1229, 536, 17, 70, "Text",
 CellID->6270],
Cell[34718, 1248, 450, 14, 70, "Text",
 CellID->5829],
Cell[35171, 1264, 230, 7, 70, "MathCaption",
 CellID->26777],
Cell[CellGroupData[{
Cell[35426, 1275, 673, 23, 70, "Input",
 CellID->15573],
Cell[36102, 1300, 785, 27, 51, "Output",
 CellID->33467843]
}, Open  ]],
Cell[36902, 1330, 72, 1, 70, "MathCaption",
 CellID->16512],
Cell[CellGroupData[{
Cell[36999, 1335, 122, 4, 70, "Input",
 CellID->23986],
Cell[37124, 1341, 177, 5, 36, "Output",
 CellID->481805538]
}, Open  ]],
Cell[CellGroupData[{
Cell[37338, 1351, 72, 1, 70, "RelatedTutorialsSection",
 CellID->193618375],
Cell[37413, 1354, 180, 4, 70, "RelatedTutorials",
 CellID->14411223]
}, Open  ]]
}, Open  ]],
Cell[37620, 1362, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

