(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    760640,      15533]
NotebookOptionsPosition[    741557,      14887]
NotebookOutlinePosition[    743745,      14936]
CellTagsIndexPosition[    743702,      14933]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], Cell[TextData[Cell[
    BoxData[
     PopupMenuBox[
      Dynamic[{"ActionMenu", None}, 
       Part[{"Constrained Optimization" :> 
         Documentation`HelpLookup[
          "paclet:tutorial/ConstrainedOptimizationOverview"], 
         "Numerical Nonlinear Local Optimization" :> 
         Documentation`HelpLookup[
          "paclet:tutorial/ConstrainedOptimizationLocalNumerical"], 
         "Numerical Nonlinear Global Optimization" :> 
         Documentation`HelpLookup[
          "paclet:tutorial/ConstrainedOptimizationGlobalNumerical"], 
         "Exact Global Optimization" :> 
         Documentation`HelpLookup[
          "paclet:tutorial/ConstrainedOptimizationExact"]}, #, 2]& , 
       Evaluator -> Automatic], {1->"\<\"Constrained Optimization\"\>", 
      2->"\<\"Numerical Nonlinear Local Optimization\"\>", 
      3->"\<\"Numerical Nonlinear Global Optimization\"\>", 
      4->"\<\"Exact Global Optimization\"\>"}, "\<\"related tutorials\"\>", 
      StyleBox["\<\"related tutorials\"\>",
       Background->Automatic],
      Appearance->Automatic,
      ImageSize->Automatic,
      MenuAppearance->"Menu"]],
     FontSize->9]], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["Linear Programming", "Title",
 CellID->100661346],

Cell[CellGroupData[{

Cell["Introduction", "Section",
 CellID->430238072],

Cell["\<\
Linear programming problems are optimization problems where the objective \
function and constraints are all linear.\
\>", "Text",
 CellID->15733215],

Cell[TextData[{
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " has a collection of algorithms for solving linear optimization problems \
with real variables, accessed via ",
 Cell[BoxData[
  ButtonBox["LinearProgramming",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LinearProgramming"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["FindMinimum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindMinimum"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["FindMaximum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindMaximum"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["NMinimize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NMinimize"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["NMaximize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NMaximize"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["Minimize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Minimize"]], "InlineFormula"],
 ", and ",
 Cell[BoxData[
  ButtonBox["Maximize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Maximize"]], "InlineFormula"],
 ". ",
 Cell[BoxData[
  ButtonBox["LinearProgramming",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LinearProgramming"]], "InlineFormula"],
 " ",
 "gives direct access to linear programming algorithms, provides the most \
flexibility for specifying the methods used, and is the most efficient for \
large-scale problems.",
 " ",
 Cell[BoxData[
  ButtonBox["FindMinimum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindMinimum"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["FindMaximum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindMaximum"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["NMinimize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NMinimize"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["NMaximize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NMaximize"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["Minimize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Minimize"]], "InlineFormula"],
 ", and ",
 Cell[BoxData[
  ButtonBox["Maximize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Maximize"]], "InlineFormula"],
 " are convenient for solving linear programming problems in equation and \
inequality form."
}], "Text",
 CellID->706807348],

Cell[TextData[{
 "This solves a linear programming problem",
 "\n",
 Cell[BoxData[{
  FormBox[
   RowBox[{
    RowBox[{"Min", "                        ", "x"}], " ", "+", " ", 
    RowBox[{"2", " ", "y"}]}], TraditionalForm], "\n", 
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"s", ".", "t", ".", "                   ", 
       RowBox[{"-", "5"}]}], " ", "x"}], " ", "+", " ", "y"}], " ", "=", " ", 
    "7"}], TraditionalForm], "\n", 
  FormBox[
   RowBox[{"                            ", 
    RowBox[{
     RowBox[{"x", " ", "+", " ", "y"}], " ", "\[GreaterEqual]", " ", "26"}]}],
    TraditionalForm], "\n", 
  FormBox[
   RowBox[{"                            ", 
    RowBox[{
     RowBox[{"x", " ", "\[GreaterEqual]", " ", "3"}], ",", " ", 
     RowBox[{"y", " ", "\[GreaterEqual]", " ", "4"}]}]}], TraditionalForm]}], 
  "InlineMath"],
 "\n",
 "using ",
 Cell[BoxData[
  ButtonBox["Minimize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Minimize"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->292783491],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Minimize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "+", 
      RowBox[{"2", " ", "y"}]}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"-", "5"}], "x"}], "+", "y"}], "\[Equal]", "7"}], " ", "&&", 
      
      RowBox[{
       RowBox[{"x", "+", "y"}], "\[GreaterEqual]", "26"}], "&&", " ", 
      RowBox[{"x", "\[GreaterEqual]", "3"}], "&&", 
      RowBox[{"y", "\[GreaterEqual]", "4"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->138525404],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["293", "6"], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      FractionBox["19", "6"]}], ",", 
     RowBox[{"y", "\[Rule]", 
      FractionBox["137", "6"]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{169, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->482087615]
}, Open  ]],

Cell[TextData[{
 "This solves the same problem using ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["NMinimize",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NMinimize"],
   FontFamily->"Courier"]], "InlineFormula"],
 ". ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["NMinimize",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NMinimize"],
   FontFamily->"Courier"]], "InlineFormula"],
 " returns a machine-number solution",
 "."
}], "MathCaption",
 CellID->667638231],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMinimize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "+", 
      RowBox[{"2", " ", "y"}]}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"-", "5"}], "x"}], "+", "y"}], "\[Equal]", "7"}], " ", "&&", 
      
      RowBox[{
       RowBox[{"x", "+", "y"}], "\[GreaterEqual]", "26"}], "&&", " ", 
      RowBox[{"x", "\[GreaterEqual]", "3"}], "&&", 
      RowBox[{"y", "\[GreaterEqual]", "4"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->73921660],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"48.83333333333333`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "3.1666666666666665`"}], ",", 
     RowBox[{"y", "\[Rule]", "22.833333333333332`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{248, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->257888045]
}, Open  ]],

Cell[TextData[{
 "This solves the same problem using ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["LinearProgramming",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/LinearProgramming"],
   FontFamily->"Courier"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->92461611],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LinearProgramming", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "2"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "5"}], ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "1"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"7", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"26", ",", "1"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"3", ",", "Infinity"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4", ",", "Infinity"}], "}"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->14420703],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["19", "6"], ",", 
   FractionBox["137", "6"]}], "}"}]], "Output",
 ImageSize->{73, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->1740945228]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["The LinearProgramming Function", "Section",
 CellID->113426546],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["LinearProgramming",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LinearProgramming"]], "InlineFormula"],
 " i",
 "s the main function for linear programming with the most flexibility for \
specifying the methods used, and is the most efficient for large-scale \
problems."
}], "Text",
 CellID->138121570],

Cell["The following options are accepted.", "Text",
 CellID->1628457416],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    ""},
   {
    ButtonBox["Method",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Method"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "method used to solve the linear optimization problem", "TableText"]},
   {
    ButtonBox["Tolerance",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Tolerance"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "convergence tolerance", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->139774623],

Cell[TextData[{
 "Options for ",
 Cell[BoxData[
  ButtonBox["LinearProgramming",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LinearProgramming"]], "InlineFormula"],
 "."
}], "Caption",
 CellID->624514127],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  ButtonBox["Method",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Method"]], "InlineFormula"],
 " option specifies the algorithm used to solve the linear programming \
problem. Possible values are ",
 Cell[BoxData[
  ButtonBox["Automatic",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<Simplex\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<RevisedSimplex\>\""], "InlineFormula"],
 ", and ",
 Cell[BoxData["\"\<InteriorPoint\>\""], "InlineFormula"],
 ". The default is ",
 Cell[BoxData[
  ButtonBox["Automatic",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
 ", which automatically chooses from the other methods based on the problem \
size and precision. "
}], "Text",
 CellID->355015470],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  ButtonBox["Tolerance",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Tolerance"]], "InlineFormula"],
 " option specifies the convergence tolerance."
}], "Text",
 CellID->12397142]
}, Open  ]],

Cell[CellGroupData[{

Cell["Examples", "Section",
 CellID->414335858],

Cell[CellGroupData[{

Cell["\<\
Difference between Interior Point and Simplex and/or Revised Simplex\
\>", "Subsection",
 CellID->589526112],

Cell[TextData[{
 "The simplex and revised simplex algorithms solve a linear programming \
problem by moving along the edges of the polytope defined by the constraints, \
from vertices to vertices with successively smaller values of the objective \
function, until the minimum is reached. ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 "'s implementation of these algorithm uses dense linear algebra. A unique \
feature of the implementation is that it is possible to solve exact/extended \
precision problems. Therefore these methods are suitable for small-sized \
problems for which non-machine-number results are needed, or a solution on \
the vertex is desirable."
}], "Text",
 CellID->50752786],

Cell[TextData[{
 "Interior point algorithms for linear programming, loosely speaking, iterate \
from the interior of the polytope defined by the constraints. They get closer \
to the solution very quickly, but unlike the simplex/revised simplex \
algorithms, do not find the solution exactly. ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 "'s implementation of an interior point algorithm uses machine precision \
sparse linear algebra. Therefore for large-scale machine-precision linear \
programming problems, the interior point method is more efficient and should \
be used."
}], "Text",
 CellID->776588746],

Cell[TextData[{
 "This solves a linear programming problem that has multiple solutions (any \
point that lies on the line segment between ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{"1", ",", "0"}], "}"}]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{"1", ",", "0"}], "}"}]], "InlineFormula"],
 " is a solution); the interior point algorithm gives a solution that lies in \
the middle of the solution set",
 "."
}], "MathCaption",
 CellID->141642741],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LinearProgramming", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "1."}], ",", 
     RowBox[{"-", "1"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{"1.", ",", "1."}], "}"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{"1.", ",", 
      RowBox[{"-", "1"}]}], "}"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<InteriorPoint\>\""}]}], "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->644934419],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.49999999958034974`", ",", "0.49999999958034974`"}], 
  "}"}]], "Output",
 ImageSize->{72, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->1714401028]
}, Open  ]],

Cell[TextData[{
 "Using ",
 Cell[BoxData["Simplex"], "InlineFormula"],
 " or ",
 Cell[BoxData["RevisedSimplex"], "InlineFormula"],
 ", a solution at the boundary of the solution set is given",
 "."
}], "MathCaption",
 CellID->311332941],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LinearProgramming", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "1."}], ",", 
     RowBox[{"-", "1"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{"1.", ",", "1."}], "}"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{"1.", ",", 
      RowBox[{"-", "1"}]}], "}"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<RevisedSimplex\>\""}]}], "]"}]], "Input",\

 CellLabel->"In[7]:=",
 CellID->189718616],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.`", ",", "0.`"}], "}"}]], "Output",
 ImageSize->{58, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->208662731]
}, Open  ]],

Cell["\<\
This shows that interior point method is much faster for the following random \
sparse linear programming problem of 200 variables and gives similar optimal \
value.\
\>", "MathCaption",
 CellID->208795567],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"SparseArray", "[", 
    RowBox[{"RandomChoice", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"{", 
        RowBox[{"0.1", ",", "0.9"}], "}"}], "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"1.", ",", "0."}], "}"}]}], ",", 
      RowBox[{"{", 
       RowBox[{"50", ",", "200"}], "}"}]}], "]"}], "]"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"xi", "=", 
    RowBox[{"LinearProgramming", "[", 
     RowBox[{
      RowBox[{"Range", "[", "200", "]"}], ",", " ", "m", ",", " ", 
      RowBox[{"Table", "[", 
       RowBox[{"0", ",", 
        RowBox[{"{", "50", "}"}]}], "]"}], ",", " ", "\n", " ", 
      RowBox[{"Method", " ", "\[Rule]", " ", "\"\<InteriorPoint\>\""}]}], 
     "]"}]}], ";"}], "//", "Timing"}]}], "Input",
 CellLabel->"In[43]:=",
 CellID->873231639],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.012001000000000692`", ",", "Null"}], "}"}]], "Output",
 ImageSize->{114, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[44]=",
 CellID->37207525]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"xs", "=", 
    RowBox[{"LinearProgramming", "[", 
     RowBox[{
      RowBox[{"Range", "[", "200", "]"}], ",", " ", "m", ",", " ", 
      RowBox[{"Table", "[", 
       RowBox[{"0", ",", 
        RowBox[{"{", "50", "}"}]}], "]"}], ",", " ", "\n", " ", 
      RowBox[{"Method", " ", "\[Rule]", " ", "\"\<Simplex\>\""}]}], "]"}]}], 
   ";"}], "//", "Timing"}]], "Input",
 CellLabel->"In[45]:=",
 CellID->1653932530],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.5760360000000001`", ",", "Null"}], "}"}]], "Output",
 ImageSize->{114, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[45]=",
 CellID->1118648765]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"Range", "[", "200", "]"}], ".", "xi"}], "-", 
  RowBox[{
   RowBox[{"Range", "[", "200", "]"}], ".", "xs"}]}]], "Input",
 CellLabel->"In[46]:=",
 CellID->1454909102],

Cell[BoxData["2.1443106668955724`*^-7"], "Output",
 ImageSize->{89, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[46]=",
 CellID->901336229]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Finding Dual Variables", "Subsection",
 CellID->224700240],

Cell["Given the general linear programming problem", "Text",
 CellID->507175941],

Cell[BoxData[{
 FormBox[
  RowBox[{"Min", "                ", 
   SuperscriptBox["c", "T"], "x", "                              ", 
   RowBox[{"(", "P", ")"}]}], TraditionalForm], "\n", 
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{"s", ".", "t", ".", "                   ", 
     SubscriptBox["A", "1"]}], " ", "x"}], " ", "=", " ", 
   SubscriptBox["b", "1"]}], TraditionalForm], "\n", 
 FormBox[
  RowBox[{"                          ", 
   RowBox[{
    RowBox[{
     SubscriptBox["A", "2"], "x"}], " ", "\[GreaterEqual]", " ", 
    SubscriptBox["b", 
     RowBox[{"2", " "}]]}]}], TraditionalForm], "\n", 
 FormBox[
  RowBox[{"                          ", 
   RowBox[{
    FormBox[
     RowBox[{" ", 
      RowBox[{
      "l", " ", "\[LessEqual]", " ", "x", " ", "\[LessEqual]", " ", "u"}]}],
     TraditionalForm], " ", ",", "   "}]}], TraditionalForm]}], "DisplayMath",\

 CellID->204420850],

Cell["its dual is", "Text",
 CellID->16053491],

Cell[TextData[Cell[BoxData[{
 FormBox[
  RowBox[{
   RowBox[{"Max", "                ", 
    SuperscriptBox["b", "T"], " ", "y"}], " ", "+", " ", 
   RowBox[{
    SuperscriptBox["l", "T"], " ", "z"}], " ", "-", " ", 
   RowBox[{
    SuperscriptBox["u", "T"], " ", "w", "     ", 
    RowBox[{"(", "D", ")"}]}]}], TraditionalForm], "\n", 
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"s", ".", "t", ".", "                   ", 
      SuperscriptBox["A", "T"]}], " ", "y"}], " ", "+", " ", "z", " ", "-", 
    " ", "w"}], " ", "=", " ", "c"}], TraditionalForm], "\n", 
 FormBox[
  RowBox[{"                          ", 
   RowBox[{
    RowBox[{
     SubscriptBox["y", "2"], " ", "\[GreaterEqual]", " ", "0"}], ",", " ", 
    "z", ",", " ", 
    RowBox[{"w", " ", "\[GreaterEqual]", " ", "0"}]}]}], 
  TraditionalForm]}]]], "DisplayMath",
 CellID->693919847],

Cell["\<\
It is useful to know solutions for both for some applications.\
\>", "Text",
 CellID->40882290],

Cell["\<\
The relationship between the solutions of the primal and dual problems is \
given by the following table.\
\>", "Text",
 CellID->16683215],

Cell[BoxData[GridBox[{
   {Cell["if the primal is", "TableHeader"], Cell[
    "then the dual problem is", "TableHeader"]},
   {Cell["feasible", "TableText"], Cell["feasible", "TableText"]},
   {Cell["unbounded", "TableText"], Cell[
    "infeasible or unbounded", "TableText"]},
   {Cell["infeasible", "TableText"], Cell[
    "unbounded or infeasible", "TableText"]}
  },
  GridBoxDividers->{
   "Columns" -> {{True}}, "ColumnsIndexed" -> {}, "Rows" -> {{True}}, 
    "RowsIndexed" -> {}},
  GridBoxItemSize->{"Columns" -> {{
       Scaled[0.3]}}}]], "Text",
 CellID->250487882],

Cell[TextData[{
 "When both problems are feasible, then the optimal values of (P) and (D) are \
the same, and the following complementary conditions hold for the primal \
solution ",
 Cell[BoxData["x"], "InlineFormula"],
 ", and dual solution ",
 Cell[BoxData[
  RowBox[{"y", ",", " ", "z"}]], "InlineFormula"],
 "."
}], "Text",
 CellID->44218844],

Cell[BoxData[{
 FormBox[
  RowBox[{
   RowBox[{
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{
       RowBox[{
        SubscriptBox["A", "2"], " ", "x"}], " ", "-", " ", 
       SubscriptBox["b", "2"]}], ")"}], "T"], " ", 
    SubscriptBox["y", "2"]}], " ", "=", " ", "0"}], TraditionalForm], "\n", 
 FormBox[
  RowBox[{
   RowBox[{
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"l", " ", "-", " ", 
       SuperscriptBox["x", "*"]}], ")"}], "T"], " ", 
    SuperscriptBox["z", "*"]}], " ", "=", " ", 
   RowBox[{
    RowBox[{
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"u", "-", 
        SuperscriptBox["x", "*"]}], ")"}], "T"], " ", 
     SuperscriptBox["w", "*"]}], " ", "=", " ", "0"}]}], 
  TraditionalForm]}], "DisplayMath",
 CellID->28996498],

Cell[TextData[{
 Cell[BoxData["DualLinearProgramming"], "InlineFormula"],
 " returns a list ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{"x", ",", "y", ",", "z", ",", "w"}], "}"}]], "InlineFormula"],
 "."
}], "Text",
 CellID->80499807],

Cell[TextData[{
 "This solves the primal problem",
 "\n",
 Cell[BoxData[{
  FormBox[
   RowBox[{
    RowBox[{"Min", "               ", "3", " ", 
     SubscriptBox["x", "1"]}], "+", 
    RowBox[{"4", " ", 
     SubscriptBox["x", 
      RowBox[{"2", " "}]]}]}], TraditionalForm], "\n", 
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{"s", ".", "t", ".", "                  ", 
      SubscriptBox["x", "1"]}], "+", 
     RowBox[{"2", " ", 
      SubscriptBox["x", "2"]}]}], "\[GreaterEqual]", " ", "5"}], 
   TraditionalForm], "\n", 
  FormBox[
   RowBox[{"                         ", 
    RowBox[{
     RowBox[{"1", "\[LessEqual]", 
      SubscriptBox["x", "1"], "\[LessEqual]", "4"}], ",", " ", 
     RowBox[{"1", "\[LessEqual]", 
      SubscriptBox["x", "2"], "\[LessEqual]", "4"}], ","}]}], 
   TraditionalForm]}], "InlineMath"],
 "\n",
 "as well as the dual problem",
 "\n",
 Cell[BoxData[{
  FormBox[
   RowBox[{
    RowBox[{"Max", "          ", "5", 
     SubscriptBox["y", "1"]}], "+", " ", 
    SubscriptBox["z", "1"], "+", 
    SubscriptBox["z", "2"], " ", "-", " ", 
    RowBox[{"4", " ", 
     SubscriptBox["w", "1"]}], " ", "-", " ", 
    RowBox[{"4", " ", 
     SubscriptBox["w", 
      RowBox[{"2", " "}]]}]}], TraditionalForm], "\n", 
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{"s", ".", "t", ".", "                 ", 
      SubscriptBox["y", "1"]}], "+", " ", 
     SubscriptBox["z", "1"], "-", " ", 
     SubscriptBox["w", "1"]}], "=", " ", "3"}], TraditionalForm], "\n", 
  FormBox[
   RowBox[{"                        ", 
    RowBox[{
     RowBox[{
      RowBox[{"2", " ", 
       SubscriptBox["y", "1"]}], " ", "+", " ", 
      SubscriptBox["z", "2"], " ", "-", " ", 
      SubscriptBox["w", "2"]}], " ", "=", " ", "4"}]}], 
   TraditionalForm], "\n", 
  FormBox[
   RowBox[{"                       ", 
    RowBox[{
     SubscriptBox["y", "1"], ",", " ", 
     SubscriptBox["z", "1"], ",", " ", 
     SubscriptBox["z", "2"], ",", " ", 
     SubscriptBox["w", "1"], ",", " ", 
     RowBox[{
      SubscriptBox["w", "2"], "\[GreaterEqual]", " ", "0"}]}]}], 
   TraditionalForm]}], "InlineMath"]
}], "MathCaption",
 CellID->48134959],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"x", ",", "y", ",", "z", ",", "w"}], "}"}], "=", 
  RowBox[{"DualLinearProgramming", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"3", ",", "4"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"{", 
      RowBox[{"1", ",", "2"}], "}"}], "}"}], ",", 
    RowBox[{"{", "5", "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"1", ",", "4"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"1", ",", "4"}], "}"}]}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[14]:=",
 CellID->947100034],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "2"}], "}"}], ",", 
   RowBox[{"{", "2", "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0"}], "}"}]}], "}"}]], "Output",
 ImageSize->{191, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[14]=",
 CellID->1620664186]
}, Open  ]],

Cell["\<\
This confirms that the primal and dual give the same objective value.\
\>", "MathCaption",
 CellID->539582753],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"3", ",", "4"}], "}"}], ".", "x"}]], "Input",
 CellLabel->"In[15]:=",
 CellID->452933674],

Cell[BoxData["11"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[15]=",
 CellID->1650563956]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"5", ",", "1", ",", "1", ",", 
    RowBox[{"-", "4"}], ",", 
    RowBox[{"-", "4"}]}], "}"}], ".", 
  RowBox[{"Flatten", "[", 
   RowBox[{"{", 
    RowBox[{"y", ",", "z", ",", "w"}], "}"}], "]"}]}]], "Input",
 CellLabel->"In[16]:=",
 CellID->28001828],

Cell[BoxData["11"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[16]=",
 CellID->1891333427]
}, Open  ]],

Cell[TextData[{
 "The dual of the constraint is ",
 Cell[BoxData[
  RowBox[{"y", " ", "=", " ", 
   RowBox[{"{", "2.", "}"}]}]], "InlineFormula"],
 ", which means that for one unit of increase in the right-hand side of the \
constraint, there will be 2 units of increase in the objective. This can be \
confirmed by perturbing the right-hand side of the constraint by 0.001",
 "."
}], "MathCaption",
 CellID->648985],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"x", ",", "y", ",", "z", ",", "w"}], "}"}], "=", 
  RowBox[{"DualLinearProgramming", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"3", ",", "4"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"{", 
      RowBox[{"1", ",", "2"}], "}"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"5", "+", "0.001"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"1", ",", "4"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"1", ",", "4"}], "}"}]}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[17]:=",
 CellID->293155135],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1.`", ",", "2.0005`"}], "}"}], ",", 
   RowBox[{"{", "2.`", "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1.`", ",", "0.`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.`", ",", "0.`"}], "}"}]}], "}"}]], "Output",
 ImageSize->{268, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[17]=",
 CellID->1765940800]
}, Open  ]],

Cell["Indeed the objective value increases by twice that amount.", \
"MathCaption",
 CellID->309989897],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"3", ",", "4"}], "}"}], ".", "x"}], " ", "-", " ", 
  "11"}]], "Input",
 CellLabel->"In[18]:=",
 CellID->12741327],

Cell[BoxData["0.002000000000000668`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[18]=",
 CellID->1071294700]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Dealing with Infeasibility and Unboundedness in the Interior Point Method\
\>", "Subsection",
 CellID->512388034],

Cell["\<\
The primal-dual interior point method is designed for feasible problems; for \
infeasible/unbounded problems it will diverge, and with the current \
implementation, it is difficult to find out if the divergence is due to \
infeasibility, or unboundedness.\
\>", "Text",
 CellID->377803856],

Cell["\<\
A heuristic catches infeasible/unbounded problems and issues a suitable \
message.\
\>", "MathCaption",
 CellID->8926348],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LinearProgramming", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1.", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "1"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", 
       RowBox[{"-", "1"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "1"}], "}"}]}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<InteriorPoint\>\""}]}], "]"}]], "Input",
 CellLabel->"In[19]:=",
 CellID->452474303],

Cell[BoxData[
 RowBox[{
  RowBox[{"LinearProgramming", "::", "\<\"lpsnf\"\>"}], ":", 
  " ", "\<\"No solution can be found that satisfies the constraints. \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/LinearProgramming/lpsnf\
\\\", ButtonNote -> \\\"LinearProgramming::lpsnf\\\"]\\)\"\>"}]], "Message", \
"MSG",
 CellLabel->"During evaluation of In[19]:=",
 CellID->4018639],

Cell[BoxData[
 RowBox[{"LinearProgramming", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1.`", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "1"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", 
       RowBox[{"-", "1"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "1"}], "}"}]}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\<\"InteriorPoint\"\>"}]}], "]"}]], "Output",\

 ImageSize->{298, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[19]=",
 CellID->48032705]
}, Open  ]],

Cell["\<\
Sometimes the heuristic cannot tell with certainty if a problem is infeasible \
or unbounded.\
\>", "MathCaption",
 CellID->20528856],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LinearProgramming", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "1."}], ",", 
     RowBox[{"-", "1."}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{"1.", ",", "1."}], "}"}], "}"}], ",", 
   RowBox[{"{", "1.", "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<InteriorPoint\>\""}]}], "]"}]], "Input",
 CellLabel->"In[20]:=",
 CellID->181927198],

Cell[BoxData[
 RowBox[{
  RowBox[{"LinearProgramming", "::", "\<\"lpdinf\"\>"}], ":", 
  " ", "\<\"The dual of this problem is infeasible, which implies that this \
problem is either unbounded or infeasible. Setting the option Method -> \
\\!\\(\\\"Simplex\\\"\\) should give a more definite answer, though large \
problems may take longer computing time. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/LinearProgramming/lpdinf\\\", ButtonNote \
-> \\\"LinearProgramming::lpdinf\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[20]:=",
 CellID->76091856],

Cell[BoxData[
 RowBox[{"LinearProgramming", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "1.`"}], ",", 
     RowBox[{"-", "1.`"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{"1.`", ",", "1.`"}], "}"}], "}"}], ",", 
   RowBox[{"{", "1.`", "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\<\"InteriorPoint\"\>"}]}], "]"}]], "Output",\

 ImageSize->{484, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[20]=",
 CellID->6869406]
}, Open  ]],

Cell[TextData[{
 "Using the ",
 Cell[BoxData["Simplex"], "InlineFormula"],
 " method as suggested by the message shows that the problem is unbounded",
 "."
}], "MathCaption",
 CellID->727892506],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LinearProgramming", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "1."}], ",", 
     RowBox[{"-", "1."}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{"1.", ",", "1."}], "}"}], "}"}], ",", 
   RowBox[{"{", "1.", "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<Simplex\>\""}]}], "]"}]], "Input",
 CellLabel->"In[21]:=",
 CellID->51654650],

Cell[BoxData[
 RowBox[{
  RowBox[{"LinearProgramming", "::", "\<\"lpsub\"\>"}], ":", 
  " ", "\<\"This problem is unbounded. \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"MessageLink\\\", ButtonFrame->None, \
ButtonData:>{\\\"System`.`\\\", \\\"LinearProgramming\\\", \\\"lpsub\\\"}]\\)\
\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[21]:=",
 CellID->12449505],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"Indeterminate", ",", "Indeterminate"}], "}"}]], "Output",
 ImageSize->{212, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[21]=",
 CellID->1919329137]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["The Method Options of \"InteriorPoint\"", "Subsection",
 CellID->605660219],

Cell[TextData[{
 Cell[BoxData["\"\<TreatDenseColumn\>\""], "InlineFormula"],
 " is a method option of ",
 Cell[BoxData["\"\<InteriorPoint\>\""], "InlineFormula"],
 " that decides if dense columns are to be treated separately. Dense columns \
are columns of the constraint matrix that have many nonzero elements. By \
default, this method option has the value ",
 Cell[BoxData[
  ButtonBox["Automatic",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
 ", and dense columns are treated separately."
}], "Text",
 CellID->654114],

Cell["\<\
Large problems that contain dense columns typically benefit from dense column \
treatment.\
\>", "MathCaption",
 CellID->126549907],

Cell[BoxData[{
 RowBox[{
  RowBox[{"A", "=", 
   RowBox[{"SparseArray", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"{", 
         RowBox[{"i_", ",", "i_"}], "}"}], "\[Rule]", "1."}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"i_", ",", "1"}], "}"}], "\[Rule]", "1."}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"300", ",", "300"}], "}"}]}], "]"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"c", " ", "=", " ", 
   RowBox[{"Table", "[", 
    RowBox[{"1", ",", 
     RowBox[{"{", "300", "}"}]}], "]"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"b", " ", "=", " ", 
   RowBox[{"A", ".", 
    RowBox[{"Range", "[", "300", "]"}]}]}], ";"}]}], "Input",
 CellLabel->"In[95]:=",
 CellID->936774867],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"x1", "=", 
      RowBox[{"LinearProgramming", "[", 
       RowBox[{"c", ",", "A", ",", "b", ",", 
        RowBox[{"Method", "\[Rule]", "\"\<InteriorPoint\>\""}]}], "]"}]}], 
     ";"}], "//", "Timing"}], ",", " ", 
   RowBox[{
    RowBox[{
     RowBox[{"x2", "=", 
      RowBox[{"LinearProgramming", "[", 
       RowBox[{"c", ",", "A", ",", "b", ",", 
        RowBox[{"Method", "\[Rule]", 
         RowBox[{"{", 
          RowBox[{"\"\<InteriorPoint\>\"", ",", 
           RowBox[{"\"\<TreatDenseColumns\>\"", "\[Rule]", "False"}]}], 
          "}"}]}]}], "]"}]}], ";"}], "//", "Timing"}]}], "}"}]], "Input",
 CellLabel->"In[98]:=",
 CellID->191014035],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0.028001000000003294`", ",", "Null"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.20001300000000002`", ",", "Null"}], "}"}]}], "}"}]], "Output",
 ImageSize->{248, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[98]=",
 CellID->1182785113]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"x1", ".", "c"}], "-", 
  RowBox[{"x2", ".", "c"}]}]], "Input",
 CellLabel->"In[99]:=",
 CellID->1922830587],

Cell[BoxData["4.979483492206782`*^-11"], "Output",
 ImageSize->{94, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[99]=",
 CellID->1586095220]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Importing Large Datasets and Solving Large-Scale Problems", "Section",
 CellID->9813130],

Cell["\<\
A commonly used format for documenting linear programming problems is the \
Mathematical Programming System (MPS) format. This is a text format \
consisting of a number of sections.\
\>", "Text",
 CellID->82319969],

Cell[CellGroupData[{

Cell["Importing MPS Formatted Files in Equation Form", "Subsection",
 CellID->208553956],

Cell[TextData[{
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " is able to import ",
 Cell[BoxData[
  ButtonBox["MPS",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/format/MPS"]], "InlineFormula"],
 " formatted files. By default, ",
 Cell[BoxData[
  ButtonBox["Import",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Import"]], "InlineFormula"],
 " of ",
 Cell[BoxData[
  ButtonBox["MPS",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/format/MPS"]], "InlineFormula"],
 " data returns a linear programming problem in equation form, which can then \
be solved using ",
 Cell[BoxData[
  ButtonBox["Minimize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Minimize"]], "InlineFormula"],
 " or ",
 Cell[BoxData[
  ButtonBox["NMinimize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NMinimize"]], "InlineFormula"],
 "."
}], "Text",
 CellID->122417020],

Cell[TextData[{
 "This solves the linear programming problem specified by ",
 Cell[BoxData[
  ButtonBox["MPS",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/format/MPS"]], "InlineFormula"],
 " file ",
 Cell[BoxData["\"\<afiro.mps\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->448660818],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"p", "=", 
  RowBox[{
  "Import", "[", "\"\<Optimization/Data/afiro.mps\>\"", "]"}]}]], "Input",
 CellLabel->"In[25]:=",
 CellID->103415320],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{535, 458},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[25]=",
 CellID->148306818]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMinimize", "@@", "p"}]], "Input",
 CellLabel->"In[26]:=",
 CellID->299458573],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "464.75314285714285`"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      SubscriptBox["\<\"X01\"\>", "\<\"MPS\"\>"], "\[Rule]", "80.`"}], ",", 
     RowBox[{
      SubscriptBox["\<\"X02\"\>", "\<\"MPS\"\>"], "\[Rule]", "25.5`"}], ",", 
     RowBox[{
      SubscriptBox["\<\"X03\"\>", "\<\"MPS\"\>"], "\[Rule]", "54.5`"}], ",", 
     RowBox[{
      SubscriptBox["\<\"X04\"\>", "\<\"MPS\"\>"], "\[Rule]", 
      "84.80000000000001`"}], ",", 
     RowBox[{
      SubscriptBox["\<\"X06\"\>", "\<\"MPS\"\>"], "\[Rule]", 
      "18.214285714285715`"}], ",", 
     RowBox[{
      SubscriptBox["\<\"X07\"\>", "\<\"MPS\"\>"], "\[Rule]", "0.`"}], ",", 
     RowBox[{
      SubscriptBox["\<\"X08\"\>", "\<\"MPS\"\>"], "\[Rule]", "0.`"}], ",", 
     RowBox[{
      SubscriptBox["\<\"X09\"\>", "\<\"MPS\"\>"], "\[Rule]", "0.`"}], ",", 
     RowBox[{
      SubscriptBox["\<\"X10\"\>", "\<\"MPS\"\>"], "\[Rule]", "0.`"}], ",", 
     RowBox[{
      SubscriptBox["\<\"X11\"\>", "\<\"MPS\"\>"], "\[Rule]", "0.`"}], ",", 
     RowBox[{
      SubscriptBox["\<\"X12\"\>", "\<\"MPS\"\>"], "\[Rule]", "0.`"}], ",", 
     RowBox[{
      SubscriptBox["\<\"X13\"\>", "\<\"MPS\"\>"], "\[Rule]", "0.`"}], ",", 
     RowBox[{
      SubscriptBox["\<\"X14\"\>", "\<\"MPS\"\>"], "\[Rule]", 
      "18.214285714285715`"}], ",", 
     RowBox[{
      SubscriptBox["\<\"X15\"\>", "\<\"MPS\"\>"], "\[Rule]", "0.`"}], ",", 
     RowBox[{
      SubscriptBox["\<\"X16\"\>", "\<\"MPS\"\>"], "\[Rule]", 
      "19.30714285714286`"}], ",", 
     RowBox[{
      SubscriptBox["\<\"X22\"\>", "\<\"MPS\"\>"], "\[Rule]", "500.`"}], ",", 
     RowBox[{
      SubscriptBox["\<\"X23\"\>", "\<\"MPS\"\>"], "\[Rule]", "475.92`"}], ",",
      
     RowBox[{
      SubscriptBox["\<\"X24\"\>", "\<\"MPS\"\>"], "\[Rule]", "24.08`"}], ",", 
     
     RowBox[{
      SubscriptBox["\<\"X25\"\>", "\<\"MPS\"\>"], "\[Rule]", "0.`"}], ",", 
     RowBox[{
      SubscriptBox["\<\"X26\"\>", "\<\"MPS\"\>"], "\[Rule]", 
      "215.00000000000003`"}], ",", 
     RowBox[{
      SubscriptBox["\<\"X28\"\>", "\<\"MPS\"\>"], "\[Rule]", "0.`"}], ",", 
     RowBox[{
      SubscriptBox["\<\"X29\"\>", "\<\"MPS\"\>"], "\[Rule]", "0.`"}], ",", 
     RowBox[{
      SubscriptBox["\<\"X30\"\>", "\<\"MPS\"\>"], "\[Rule]", "0.`"}], ",", 
     RowBox[{
      SubscriptBox["\<\"X31\"\>", "\<\"MPS\"\>"], "\[Rule]", "0.`"}], ",", 
     RowBox[{
      SubscriptBox["\<\"X32\"\>", "\<\"MPS\"\>"], "\[Rule]", "0.`"}], ",", 
     RowBox[{
      SubscriptBox["\<\"X33\"\>", "\<\"MPS\"\>"], "\[Rule]", "0.`"}], ",", 
     RowBox[{
      SubscriptBox["\<\"X34\"\>", "\<\"MPS\"\>"], "\[Rule]", "0.`"}], ",", 
     RowBox[{
      SubscriptBox["\<\"X35\"\>", "\<\"MPS\"\>"], "\[Rule]", "0.`"}], ",", 
     RowBox[{
      SubscriptBox["\<\"X36\"\>", "\<\"MPS\"\>"], "\[Rule]", 
      "339.9428571428571`"}], ",", 
     RowBox[{
      SubscriptBox["\<\"X37\"\>", "\<\"MPS\"\>"], "\[Rule]", 
      "383.9428571428571`"}], ",", 
     RowBox[{
      SubscriptBox["\<\"X38\"\>", "\<\"MPS\"\>"], "\[Rule]", "0.`"}], ",", 
     RowBox[{
      SubscriptBox["\<\"X39\"\>", "\<\"MPS\"\>"], "\[Rule]", "0.`"}]}], 
    "}"}]}], "}"}]], "Output",
 ImageSize->{555, 101},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[26]=",
 CellID->934999415]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Large-Scale Problems: Importing in Matrix and Vector Form", "Subsection",
 CellID->445507631],

Cell[TextData[{
 "For large-scale problems, it is more efficient to import the ",
 Cell[BoxData[
  ButtonBox["MPS",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/format/MPS"]], "InlineFormula"],
 " data file and represent the linear programming using matrices and vectors, \
then solve using ",
 Cell[BoxData[
  ButtonBox["LinearProgramming",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LinearProgramming"]], "InlineFormula"],
 ". "
}], "Text",
 CellID->137995970],

Cell[TextData[{
 "This shows that for ",
 Cell[BoxData[
  ButtonBox["MPS",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/format/MPS"]], "InlineFormula"],
 " formatted data, the following 3 elements can be imported",
 "."
}], "MathCaption",
 CellID->297516826],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"p", "=", 
  RowBox[{"Import", "[", 
   RowBox[{"\"\<Optimization/Data/ganges.mps\>\"", ",", "\"\<Elements\>\""}], 
   "]"}]}]], "Input",
 CellLabel->"In[101]:=",
 CellID->129022199],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"ConstraintMatrix\"\>", ",", "\<\"Equations\"\>", 
   ",", "\<\"LinearProgrammingData\"\>"}], "}"}]], "Output",
 ImageSize->{363, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[101]=",
 CellID->90286632]
}, Open  ]],

Cell[TextData[{
 " This imports the problem \"ganges\", with 1309 constraints and 1681 \
variables, in a form suitable for ",
 Cell[BoxData[
  ButtonBox["LinearProgramming",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LinearProgramming"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->211685703],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"c", ",", "A", ",", "b", ",", "bounds"}], "}"}], "=", 
   RowBox[{"Import", "[", 
    RowBox[{
    "\"\<Optimization/Data/ganges.mps\>\"", ",", 
     "\"\<LinearProgrammingData\>\""}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[102]:=",
 CellID->205726187],

Cell["This solves the problem and finds the optimal value.", "MathCaption",
 CellID->5890066],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"x", "=", 
    RowBox[{"LinearProgramming", "[", 
     RowBox[{"c", ",", "A", ",", "b", ",", "bounds"}], "]"}]}], ";"}], 
  " "}]], "Input",
 CellLabel->"In[103]:=",
 CellID->52691358],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"c", ".", "x"}]], "Input",
 CellLabel->"In[104]:=",
 CellID->1056069432],

Cell[BoxData[
 RowBox[{"-", "109585.73602370448`"}]], "Output",
 ImageSize->{63, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[104]=",
 CellID->2101418431]
}, Open  ]],

Cell[TextData[{
 "The ",
 Cell[BoxData["\"\<ConstraintMatrix\>\""], "InlineFormula"],
 " specification can be used to get the sparse constraint matrix only."
}], "MathCaption",
 CellID->201121673],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"p", "=", 
  RowBox[{"Import", "[", 
   RowBox[{
   "\"\<Optimization/Data/ganges.mps\>\"", ",", "\"\<ConstraintMatrix\>\""}], 
   "]"}]}]], "Input",
 CellLabel->"In[105]:=",
 CellID->96551054],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "6912", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 6912, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"1309", ",", "1681"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{230, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[105]=",
 CellID->465458690]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Free Formatted MPS Files", "Subsection",
 CellID->298767088],

Cell[TextData[{
 "Standard ",
 Cell[BoxData[
  ButtonBox["MPS",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/format/MPS"]], "InlineFormula"],
 " formatted files use a fixed format, where each field occupies a strictly \
fixed character position. However some modeling systems output ",
 Cell[BoxData[
  ButtonBox["MPS",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/format/MPS"]], "InlineFormula"],
 " files with a free format, where fields are positioned freely. For such \
files, the option ",
 Cell[BoxData[
  RowBox[{"\"\<FreeFormat\>\"", "->", 
   ButtonBox["True",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/True"]}]], "InlineFormula"],
 " can be specified for ",
 Cell[BoxData[
  ButtonBox["Import",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Import"]], "InlineFormula"],
 "."
}], "Text",
 CellID->437791027],

Cell[TextData[{
 "This string describes an ",
 Cell[BoxData[
  ButtonBox["MPS",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/format/MPS"]], "InlineFormula"],
 " file in free format",
 "."
}], "MathCaption",
 CellID->23023199],

Cell[BoxData[
 RowBox[{
  RowBox[{
  "txt", " ", "=", " ", 
   "\"\<NAME TESTPROB\\nROWS\\n N COST\\n L CON1\\n G CON2\\n E CON3 \
\\nCOLUMNS\\n x COST 1 CON1 2\\n x CON2 3\\n y COST 4 CON1 5\\n y CON3 6\\n Z \
COST 7 CON2 8\\n Z CON3 9\\nRHS\\n RHS1 CON1 10 CON2 11\\n RHS1 CON3 \
12\\nBOUNDS\\n UP BND1 x 13\\n LO BND1 y 14\\n UP BND1 y \
15\\nENDATA\\n\>\""}], ";"}]], "Input",
 CellLabel->"In[122]:=",
 CellID->8788611],

Cell["\<\
This gets a temporary file name, and exports the string to the file.\
\>", "MathCaption",
 CellID->80037026],

Cell[BoxData[{
 RowBox[{
  RowBox[{"file", " ", "=", " ", 
   RowBox[{"Close", "[", 
    RowBox[{"OpenWrite", "[", "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"Export", "[", 
   RowBox[{"file", ",", "txt", ",", "\"\<Text\>\""}], "]"}], ";"}]}], "Input",\

 CellLabel->"In[123]:=",
 CellID->412264192],

Cell[TextData[{
 "This imports the file, using the ",
 Cell[BoxData[
  RowBox[{"\"\<FreeFormat\>\"", "->", 
   ButtonBox["True",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/True"]}]], "InlineFormula"],
 " option",
 "."
}], "MathCaption",
 CellID->467073014],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Import", "[", 
  RowBox[{"file", ",", " ", "\"\<MPS\>\"", ",", 
   RowBox[{"\"\<FreeFormat\>\"", "\[Rule]", "True"}]}], "]"}]], "Input",
 CellLabel->"In[126]:=",
 CellID->903242654],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"1.`", " ", 
       SubscriptBox["\<\"x\"\>", "\<\"MPS\"\>"]}], "+", 
      RowBox[{"4.`", " ", 
       SubscriptBox["\<\"y\"\>", "\<\"MPS\"\>"]}], "+", 
      RowBox[{"7.`", " ", 
       SubscriptBox["\<\"Z\"\>", "\<\"MPS\"\>"]}]}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"2.`", " ", 
         SubscriptBox["\<\"x\"\>", "\<\"MPS\"\>"]}], "+", 
        RowBox[{"5.`", " ", 
         SubscriptBox["\<\"y\"\>", "\<\"MPS\"\>"]}]}], "\[LessEqual]", 
       "10.`"}], "&&", 
      RowBox[{
       RowBox[{
        RowBox[{"3.`", " ", 
         SubscriptBox["\<\"x\"\>", "\<\"MPS\"\>"]}], "+", 
        RowBox[{"8.`", " ", 
         SubscriptBox["\<\"Z\"\>", "\<\"MPS\"\>"]}]}], "\[GreaterEqual]", 
       "11.`"}], "&&", 
      RowBox[{
       RowBox[{
        RowBox[{"6.`", " ", 
         SubscriptBox["\<\"y\"\>", "\<\"MPS\"\>"]}], "+", 
        RowBox[{"9.`", " ", 
         SubscriptBox["\<\"Z\"\>", "\<\"MPS\"\>"]}]}], "\[Equal]", "12.`"}], "&&",
       
      RowBox[{
       SubscriptBox["\<\"x\"\>", "\<\"MPS\"\>"], "\[GreaterEqual]", "0"}], "&&",
       
      RowBox[{
       SubscriptBox["\<\"x\"\>", "\<\"MPS\"\>"], "\[LessEqual]", "13.`"}], "&&",
       
      RowBox[{
       SubscriptBox["\<\"y\"\>", "\<\"MPS\"\>"], "\[GreaterEqual]", "14.`"}], 
      "&&", 
      RowBox[{
       SubscriptBox["\<\"y\"\>", "\<\"MPS\"\>"], "\[LessEqual]", "15.`"}], "&&",
       
      RowBox[{
       SubscriptBox["\<\"Z\"\>", "\<\"MPS\"\>"], "\[GreaterEqual]", "0"}]}]}],
     "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     SubscriptBox["\<\"x\"\>", "\<\"MPS\"\>"], ",", 
     SubscriptBox["\<\"y\"\>", "\<\"MPS\"\>"], ",", 
     SubscriptBox["\<\"Z\"\>", "\<\"MPS\"\>"]}], "}"}]}], "}"}]], "Output",
 ImageSize->{493, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[126]=",
 CellID->997080558]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Linear Programming Test Problems", "Subsection",
 CellID->820857862],

Cell[TextData[{
 "Through the ",
 Cell[BoxData[
  ButtonBox["ExampleData",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ExampleData"]], "InlineFormula"],
 " function, all ",
 ButtonBox["NetLib",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.netlib.org/lp/"], None}],
 " linear programming test problems can be accessed."
}], "Text",
 CellID->439047054],

Cell["This finds all problems in the Netlib set.", "MathCaption",
 CellID->1015647543],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ExampleData", "[", "\"\<LinearProgramming\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->489880541],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"25fv47\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"80bau3b\"\>"}], "}"}], ",",
    
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"adlittle\"\>"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"afiro\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"agg\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"agg2\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"agg3\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"bandm\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"beaconfd\"\>"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"bgdbg1\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"bgetam\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"bgindy\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"bgprtr\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"blend\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"bnl1\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"bnl2\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"boeing1\"\>"}], "}"}], ",",
    
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"boeing2\"\>"}], "}"}], ",",
    
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"bore3d\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"box1\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"brandy\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"capri\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"ceria3d\"\>"}], "}"}], ",",
    
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"chemcom\"\>"}], "}"}], ",",
    
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"cplex1\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"cplex2\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"cre-a\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"cre-b\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"cre-c\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"cre-d\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"cycle\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"czprob\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"d2q06c\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"d6cube\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"degen2\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"degen3\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"dfl001\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"e226\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"etamacro\"\>"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"ex72a\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"ex73a\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"fffff800\"\>"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"finnis\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"fit1d\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"fit1p\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"fit2d\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"fit2p\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"forest6\"\>"}], "}"}], ",",
    
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"forplan\"\>"}], "}"}], ",",
    
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"galenet\"\>"}], "}"}], ",",
    
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"ganges\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"gfrd-pnc\"\>"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"gosh\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"gran\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"greenbea\"\>"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"greenbeb\"\>"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"grow15\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"grow22\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"grow7\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"israel\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"itest2\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"itest6\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"kb2\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"ken-07\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"ken-11\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"ken-13\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"ken-18\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"klein1\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"klein2\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"klein3\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"lotfi\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"maros\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"maros-r7\"\>"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"modszk1\"\>"}], "}"}], ",",
    
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"mondou2\"\>"}], "}"}], ",",
    
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"nesm\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"osa-07\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"osa-14\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"osa-30\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"osa-60\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"pang\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"pds-02\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"pds-06\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"pds-10\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"pds-20\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"perold\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"pilot\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"pilot4\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"pilot4i\"\>"}], "}"}], ",",
    
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"pilot87\"\>"}], "}"}], ",",
    
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"pilot.ja\"\>"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"pilotnov\"\>"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"pilot.we\"\>"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"qual\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"reactor\"\>"}], "}"}], ",",
    
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"recipe\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"refinery\"\>"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"sc105\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"sc205\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"sc50a\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"sc50b\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"scagr25\"\>"}], "}"}], ",",
    
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"scagr7\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"scfxm1\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"scfxm2\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"scfxm3\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"scorpion\"\>"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"scrs8\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"scsd1\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"scsd6\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"scsd8\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"sctap1\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"sctap2\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"sctap3\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"seba\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"share1b\"\>"}], "}"}], ",",
    
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"share2b\"\>"}], "}"}], ",",
    
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"shell\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"ship04l\"\>"}], "}"}], ",",
    
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"ship04s\"\>"}], "}"}], ",",
    
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"ship08l\"\>"}], "}"}], ",",
    
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"ship08s\"\>"}], "}"}], ",",
    
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"ship12l\"\>"}], "}"}], ",",
    
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"ship12s\"\>"}], "}"}], ",",
    
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"sierra\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"stair\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"standata\"\>"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"standgub\"\>"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"standmps\"\>"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"stocfor1\"\>"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"stocfor2\"\>"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"truss\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"tuff\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"vol1\"\>"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"vtp.base\"\>"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"wood1p\"\>"}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"woodinfe\"\>"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"\<\"LinearProgramming\"\>", ",", "\<\"woodw\"\>"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{578, 1104},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]="]
}, Open  ]],

Cell[TextData[{
 "This imports the problem ",
 Cell[BoxData["\"\<afiro\>\""], "InlineFormula"],
 " and solves it."
}], "MathCaption",
 CellID->17729874],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ExampleData", "[", 
  RowBox[{"{", 
   RowBox[{"\"\<LinearProgramming\>\"", ",", "\"\<afiro\>\""}], "}"}], 
  "]"}]], "Input",
 CellLabel->"In[8]:=",
 CellID->1200455685],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{575, 152},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->3719019]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LinearProgramming", "@@", "%"}]], "Input",
 CellLabel->"In[9]:=",
 CellID->878467550],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "80.`", ",", "25.5`", ",", "54.5`", ",", "84.80000000000001`", ",", 
   "18.214285714285715`", ",", "0.`", ",", "0.`", ",", "0.`", ",", "0.`", ",",
    "0.`", ",", "0.`", ",", "0.`", ",", "18.214285714285715`", ",", "0.`", 
   ",", "19.30714285714286`", ",", "500.`", ",", "475.92`", ",", "24.08`", 
   ",", "0.`", ",", "215.00000000000003`", ",", "0.`", ",", "0.`", ",", "0.`",
    ",", "0.`", ",", "0.`", ",", "0.`", ",", "0.`", ",", "0.`", ",", 
   "339.9428571428571`", ",", "383.9428571428571`", ",", "0.`", ",", "0.`"}], 
  "}"}]], "Output",
 ImageSize->{578, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[9]=",
 CellID->1183238080]
}, Open  ]],

Cell[TextData[{
 "This shows other properties that can be imported for the ",
 Cell[BoxData["\"\<afiro\>\""], "InlineFormula"],
 " problem."
}], "MathCaption",
 CellID->1198832590],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ExampleData", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"\"\<LinearProgramming\>\"", ",", "\"\<afiro\>\""}], "}"}], ",", 
   "\"\<Properties\>\""}], "]"}]], "Input",
 CellLabel->"In[10]:=",
 CellID->855712891],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"Collection\"\>", ",", "\<\"ConstraintMatrix\"\>", 
   ",", "\<\"Dimensions\"\>", ",", "\<\"Equations\"\>", 
   ",", "\<\"LinearProgrammingData\"\>", ",", "\<\"Name\"\>", 
   ",", "\<\"Source\"\>"}], "}"}]], "Output",
 ImageSize->{332, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[10]=",
 CellID->1360471072]
}, Open  ]],

Cell[TextData[{
 "This imports ",
 Cell[BoxData["\"\<afiro\>\""], "InlineFormula"],
 " in equation form."
}], "MathCaption",
 CellID->295471405],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ExampleData", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"\"\<LinearProgramming\>\"", ",", "\"\<afiro\>\""}], "}"}], ",", 
   "\"\<Equations\>\""}], "]"}]], "Input",
 CellLabel->"In[11]:=",
 CellID->1377141912],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{535, 458},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[11]=",
 CellID->46806323]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Application Examples of Linear Programming", "Section",
 CellID->261990371],

Cell[CellGroupData[{

Cell["L1-Norm Minimization", "Subsection",
 CellID->293106634],

Cell[TextData[{
 "It is possible to solve an ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["l", "1"], TraditionalForm]], "InlineMath"],
 " minimization problem"
}], "Text",
 CellID->22273430],

Cell[TextData[Cell[BoxData[
 FormBox[
  RowBox[{"Min", "               ", "|", 
   RowBox[{"A", 
    FormBox[
     RowBox[{
      RowBox[{"x", "-", "b"}], 
      SubscriptBox["|", "1"]}],
     TraditionalForm]}]}], TraditionalForm]]]], "DisplayMath",
 CellID->38244179],

Cell["by turning the system into a linear programming problem", "Text",
 CellID->310897012],

Cell[TextData[Cell[BoxData[{
 FormBox[
  RowBox[{"Min", "                ", 
   SuperscriptBox["z", "T"], " ", "e"}], TraditionalForm], "\n", 
 FormBox[
  RowBox[{"                          ", 
   RowBox[{"z", "\[GreaterEqual]", 
    RowBox[{
     RowBox[{"A", " ", "x"}], "-", "b"}]}]}], TraditionalForm], "\n", 
 FormBox[
  RowBox[{"                          ", 
   RowBox[{"z", "\[GreaterEqual]", 
    RowBox[{
     RowBox[{
      RowBox[{"-", "A"}], " ", "x"}], "+", "b"}]}]}], 
  TraditionalForm]}]]], "DisplayMath",
 CellID->20392480],

Cell[TextData[{
 "This defines a function for solving an ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["l", "1"], TraditionalForm]], "InlineMath"],
 " minimization problem",
 "."
}], "MathCaption",
 CellID->1470753465],

Cell[BoxData[
 RowBox[{
  RowBox[{"L1Minimization", "[", 
   RowBox[{"A_", ",", "b_"}], "]"}], ":=", " ", 
  RowBox[{"Module", "[", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
     "B", ",", "c", ",", "All", ",", "ball", ",", "x", ",", "lb", ",", "AT"}],
      " ", "}"}], ",", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{
      RowBox[{"{", 
       RowBox[{"m", ",", "n"}], "}"}], "=", 
      RowBox[{"Dimensions", "[", "A", "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"AT", " ", "=", " ", 
      RowBox[{"Transpose", "[", "A", "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"B", "=", 
      RowBox[{"SparseArray", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"i_", ",", "i_"}], "}"}], "\[Rule]", "1"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"m", ",", "m"}], "}"}]}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"Aall", "=", 
      RowBox[{"Join", "[", 
       RowBox[{
        RowBox[{"Transpose", "[", 
         RowBox[{"Join", "[", 
          RowBox[{"B", ",", 
           RowBox[{"-", "AT"}]}], "]"}], "]"}], ",", 
        RowBox[{"Transpose", "[", 
         RowBox[{"Join", "[", 
          RowBox[{"B", ",", "AT"}], "]"}], "]"}]}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"ball", "=", 
      RowBox[{"Join", "[", 
       RowBox[{
        RowBox[{"-", "b"}], ",", "b"}], "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"c", "=", 
      RowBox[{"Join", "[", 
       RowBox[{
        RowBox[{"Table", "[", 
         RowBox[{"1", ",", 
          RowBox[{"{", "m", "}"}]}], "]"}], ",", 
        RowBox[{"Table", "[", 
         RowBox[{"0", ",", 
          RowBox[{"{", "n", "}"}]}], "]"}]}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"lb", " ", "=", " ", 
      RowBox[{"Table", "[", 
       RowBox[{
        RowBox[{"-", "Infinity"}], ",", 
        RowBox[{"{", 
         RowBox[{"m", "+", "n"}], "}"}]}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"x", "=", 
      RowBox[{"LinearProgramming", "[", 
       RowBox[{"c", ",", "Aall", ",", "ball", ",", "lb"}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"x", "=", 
      RowBox[{"Drop", "[", 
       RowBox[{"x", ",", "m"}], "]"}]}]}]}], "\[IndentingNewLine]", 
   "]"}]}]], "Input",
 CellLabel->"In[35]:=",
 CellID->169775421],

Cell["The following is an over-determined linear system.", "MathCaption",
 CellID->215389070],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{"(", GridBox[{
       {"1", "2", "3"},
       {"4", "5", "5"},
       {"7", "8", "9"},
       {"10", "11", "12"}
      }], ")"}], " ", "x"}], " ", "=", " ", 
   RowBox[{"(", GridBox[{
      {"1"},
      {"2"},
      {"3"},
      {"4"}
     }], ")"}]}], TraditionalForm]], "DisplayMath"],

Cell[TextData[{
 "A simple application of ",
 Cell[BoxData[
  ButtonBox["LinearSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LinearSolve"]], "InlineFormula"],
 " would fail."
}], "MathCaption",
 CellID->254791179],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"A", "=", 
   RowBox[{"SparseArray", "[", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"1", ",", "2", ",", "3"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"5", ",", "6", ",", "7"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"7", ",", "8", ",", "9"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"10", ",", "11", ",", "12"}], "}"}]}], "}"}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"b", "=", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "3", ",", "4"}], "}"}]}], ";"}], "\n", 
 RowBox[{"LinearSolve", "[", 
  RowBox[{"A", ",", "b"}], "]"}]}], "Input",
 CellLabel->"In[36]:=",
 CellID->228353195],

Cell[BoxData[
 RowBox[{
  RowBox[{"LinearSolve", "::", "\<\"nosol\"\>"}], ":", 
  " ", "\<\"Linear equation encountered which has no solution. \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"MessageLink\\\", ButtonFrame->None, \
ButtonData:>{\\\"System`.`\\\", \\\"LinearSolve\\\", \
\\\"nosol\\\"}]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[36]:=",
 CellID->1702910108],

Cell[BoxData[
 RowBox[{"LinearSolve", "[", 
  RowBox[{
   TagBox[
    RowBox[{"SparseArray", "[", 
     RowBox[{
      InterpretationBox[
       RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "12", 
        "\[InvisibleSpace]", "\<\">\"\>"}],
       SequenceForm["<", 12, ">"],
       Editable->False], ",", 
      RowBox[{"{", 
       RowBox[{"4", ",", "3"}], "}"}]}], "]"}],
    False,
    Editable->False], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "3", ",", "4"}], "}"}]}], "]"}]], "Output",
 ImageSize->{352, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[38]=",
 CellID->276395405]
}, Open  ]],

Cell[TextData[{
 "This finds the ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["l", "1"], TraditionalForm]], "InlineMath"],
 " minimization solution",
 "."
}], "MathCaption",
 CellID->345408722],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"x", "=", 
  RowBox[{"L1Minimization", "[", 
   RowBox[{"A", ",", "b"}], "]"}]}]], "Input",
 CellLabel->"In[39]:=",
 CellID->312111314],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0", ",", "0", ",", 
   FractionBox["1", "3"]}], "}"}]], "Output",
 ImageSize->{66, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[39]=",
 CellID->1486497184]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{
    RowBox[{"Norm", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"A", ".", "x"}], "-", "b"}], ",", "1"}], "]"}], ",", 
    RowBox[{"Norm", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"A", ".", "x"}], "-", "b"}], ",", "2"}], "]"}]}], "}"}], "//", 
  "N"}]], "Input",
 CellLabel->"In[40]:=",
 CellID->304875770],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.3333333333333333`", ",", "0.3333333333333333`"}], 
  "}"}]], "Output",
 ImageSize->{142, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[40]=",
 CellID->1715359613]
}, Open  ]],

Cell[TextData[{
 "The least squares solution can be found using ",
 Cell[BoxData[
  ButtonBox["PseudoInverse",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PseudoInverse"]], "InlineFormula"],
 ". This gives a large ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["l", "1"], TraditionalForm]], "InlineMath"],
 " norm, but a smaller ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["l", "2"], TraditionalForm]], "InlineMath"],
 " norm",
 "."
}], "MathCaption",
 CellID->62657209],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"x2", "=", 
  RowBox[{
   RowBox[{"PseudoInverse", "[", "A", "]"}], ".", "b"}]}]], "Input",
 CellLabel->"In[41]:=",
 CellID->693798693],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["4", "513"], ",", 
   FractionBox["58", "513"], ",", 
   FractionBox["112", "513"]}], "}"}]], "Output",
 ImageSize->{116, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[41]=",
 CellID->129057075]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{
    RowBox[{"Norm", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"A", ".", "x2"}], "-", "b"}], ",", "1"}], "]"}], ",", 
    RowBox[{"Norm", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"A", ".", "x2"}], "-", "b"}], ",", "2"}], "]"}]}], "}"}], "//",
   "N"}]], "Input",
 CellLabel->"In[42]:=",
 CellID->155493228],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.49122807017543857`", ",", "0.2861316917596507`"}], 
  "}"}]], "Output",
 ImageSize->{142, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[42]=",
 CellID->1366635944]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Design of an Optimal Anchor", "Subsection",
 CellID->326994842],

Cell[TextData[{
 "The example is adopted from ",
 ButtonBox["[2]",
  BaseStyle->"Link",
  ButtonData->
   "paclet:tutorial/ConstrainedOptimizationLinearProgramming#1592161131"],
 ". The aim is to design an anchor that uses as little material as possible \
to support a load."
}], "Text",
 CellID->595713038],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61838 
%%ImageSize: 360 223 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
-1.51788e-18 0.00278552 -0.00278552 0.00278552 [
[ 0 0 0 0 ]
[ 1 .61838 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .61838 L
0 .61838 L
closepath
clip
newpath
% Start of colorimage (RGB)
p
0 -0.00279 translate
1.00279 .62117 scale
1080 string
360 223 8 [360 0 0 223 0 0] {
\tcurrentfile
\t1 index
\treadhexstring
\tpop
}  false 3 Mcolorimage
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFA5A5FF0000FF7676FF9A9AFF7B7BFF4545FFFFFFFFADADFF5B5BFF9696FF5353FF
C6C6FFFDFDFF4B4BFF4B4BFF9797FF1C1CFF8282FFFFFFFFABABFF2F2FFF6A6AFF8D8DFF
0000FFB3B3FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFF0000FFE6E6FFFFFFFFFFFFFF9191FFF7F7FF0C0CFFDBDBFFFFFFFFB6B6FF
2A2AFFECECFF0000FFF4F4FFF7F7FF0202FFFEFEFFF6F6FF0B0BFFC5C5FFFFFFFFE9E9FF
0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFF0000FFE7E7FFFFFFFFFFFFFFF9F9FFDFDFFF0000FFFCFCFFFFFFFFD7D7FF
0404FFFFFFFFA9A9FF8D8DFFB3B3FF0303FFFFFFFFDFDFFF0000FFFBFBFFFFFFFFF7F7FF
0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFF0000FFE7E7FFFFFFFFFFFFFFFFFFFFF7F7FF0C0CFFDBDBFFFFFFFFB6B6FF
2A2AFFF7F7FF8585FFFEFEFFEDEDFF0303FFFFFFFFFAFAFF1818FFD4D4FFFFFFFFF4F4FF
0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFF0000FFE7E7FFFFFFFFFFFFFFFFFFFFFFFFFFAEAEFF5C5CFF9696FF5555FF
C7C7FFFCFCFF6D6DFF7E7EFF7777FF5B5BFFFFFFFFFFFFFFC7C7FF5E5EFF9595FF7171FF
0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFF0000FFE7E7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF7F7FF
0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFF0000FFE7E7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF6F6FF
0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFA4A4FF0000FF9292FFF2F2FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF9D9DFF
1313FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFF808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFF808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFF808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFF808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFF808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFF808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFF808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFF808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFF808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFF808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFF808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFF808080808080808080808080808080808080808080FFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFF808080808080808080808080808080808080808080808080FFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFF808080FFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFF808080FFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
8080808080808080808080800000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
8080808080808080808080800000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
8080808080808080808080800000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
8080808080808080808080800000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
8080808080808080808080800000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
8080808080808080808080800000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
8080808080808080808080800000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
8080808080808080808080800000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
8080808080808080808080800000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
8080808080808080808080800000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
8080808080808080808080800000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
8080808080808080808080800000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
8080808080808080808080800000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
8080808080808080808080800000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
8080808080808080808080800000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
8080808080808080808080800000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
8080808080808080808080800000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
8080808080808080808080800000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
8080808080808080808080800000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
8080808080808080808080800000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
8080808080808080808080800000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
8080808080808080808080800000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
8080808080808080808080800000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
8080808080808080808080800000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
8080808080808080808080800000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FFFFFFFFFFFFFFFFFFFFFFFFFF808080FFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080FFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080FFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080
808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080
808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080
808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080
808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080
808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080
808080808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
808080808080808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
808080808080808080FFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFF808080808080FFFFFF0000FF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFF808080FFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FF0000FFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FF0000FF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
0000FF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFF0000FF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FF0000FFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFD2D2FF3232FF9898FFFFFFFFFFFFFFAAAAFF0000FF
8282FFB5B5FF0000FFA4A4FFFFFFFFA3A3FF0000FFB5B5FFFFFFFFA9A9FF3333FF6868FF
7878FFA3A3FF0000FFA2A2FFFFFFFFA0A0FF0000FFB4B4FFFFFFFFADADFF5B5BFF9696FF
5353FFC6C6FFB5B5FF0000FFA1A1FFF8F8FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFA9A9FFB4B4FFFFFFFFFFFFFFB5B5FF1D1DFF
FDFDFFFFFFFF0000FFF7F7FFFFFFFFF7F7FF0000FFFFFFFFF6F6FF0A0AFFC2C2FFFFFFFF
F8F8FFCBCBFF0000FFF7F7FFFFFFFFF7F7FF0000FFFFFFFFF7F7FF0C0CFFDBDBFFFFFFFF
B6B6FF2A2AFFFFFFFF0000FFF7F7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFAAAAFF3535FFFFFFFFFFFFFFFEFEFF1818FFCACAFF
FFFFFFFDFDFF4B4BFF4B4BFF9797FF1C1CFF8282FFB5B5FF0000FFADADFFB5B5FF0000FF
ADADFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF6F6FF7070FFFFFFFFFFFFFF6363FF7272FF
FFFFFFFFFFFF0000FFF7F7FFFFFFFFF7F7FF0000FFFFFFFFDFDFFF0000FFFAFAFFFFFFFF
FFFFFFFFFFFF0000FFF7F7FFFFFFFFF7F7FF0000FFFFFFFFDFDFFF0000FFFCFCFFFFFFFF
D7D7FF0404FFFFFFFF0000FFF6F6FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080FFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF6868FF2727FFEEEEFFFFFFFFD1D1FF1A1AFF9191FF
FFFFFFECECFF0000FFF4F4FFF7F7FF0202FFFEFEFFFFFFFF0000FFF7F7FFFFFFFF0000FF
F7F7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF5F5FFF8F8FFF9090FF0303FFC7C7FF
FFFFFFFEFEFF0000FFDDDDFFFFFFFFE6E6FF0505FFFFFFFFF9F9FF1212FFDFDFFFFFFFFF
C2C2FFDCDCFF0000FFE1E1FFFFFFFFE6E6FF0505FFFFFFFFF7F7FF0C0CFFDBDBFFFFFFFF
B6B6FF2A2AFFFEFEFF0000FFB4B4FFDEDEFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFF808080808080808080808080808080808080808080FFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF2626FFA7A7FFADADFFFFFFFF8A8AFF6464FF8787FF
FFFFFFFFFFFFA9A9FF8D8DFFB3B3FF0303FFFFFFFFFFFFFF0000FFF7F7FFFFFFFF0000FF
F7F7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFB6B6FFC2C2FFC3C3FF1D1DFFFEFEFF
FFFFFFB1B1FF1C1CFFA3A3FF7F7FFF3434FF7878FFFFFFFFFFFFFFBBBBFF6565FF9393FF
3D3DFFCECEFF0000FF9D9DFF7E7EFF3434FF7878FFFFFFFFFFFFFFAEAEFF5C5CFF9696FF
5555FFC7C7FF9C9CFF2121FF9F9FFF1414FFF9F9FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFF808080808080
808080808080808080808080808080808080808080808080808080FFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFE4E4FF1111FFFCFCFF8686FFFFFFFF4242FFACACFF8888FF
F9F9FFF7F7FF8585FFFEFEFFEDEDFF0303FFFFFFFFFFFFFF0000FFF7F7FFFFFFFF0000FF
F7F7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFAFAFF8686FF7272FF7272FFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFF0000FFF7F7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFF808080808080808080808080808080
8080808080808080808080808080808080808080808080808080800000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF
0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FF0000FFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFA0A0FF4F4FFFFFFFFF8282FFF2F2FF0909FFEFEFFFBCBCFF
C1C1FFFCFCFF6D6DFF7E7EFF7777FF5B5BFFFFFFFFFFFFFF0000FFF7F7FFFFFFFF0000FF
F7F7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF8888FF2222FFC7C7FFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFEFEFF0000FFF7F7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFF808080808080808080
808080808080808080808080808080808080808080808080808080FFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFF5A5AFF8F8FFFFFFFFF9D9DFF9090FF3B3BFFFFFFFFF5F5FF
8787FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFF7F7FFFFFFFF0000FF
F7F7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFBBBBFF1D1DFFFEFEFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFA4A4FF1414FFF7F7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFF808080808080808080808080808080808080808080808080FFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFDFDFF1515FFCCCCFFFFFFFFDDDDFF1111FF8282FFFFFFFFFFFFFF
7777FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFEFF0000FFF7F7FFFEFEFF0000FF
F7F7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080FFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFF8888FF0000FFADADFFFFFFFFFEFEFF1E1EFFC8C8FFFFFFFFCDCDFF
3737FFA5A5FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFAFAFFF1111FFF7F7FFAFAFFF1111FF
F7F7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080FFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080808080808080808080808080808080808080
808080808080808080808080808080808080FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
pop
P
% End of image
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
 ImageSize->{360, 223},
 ImageMargins->{{43, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellID->58852609,
 ImageCache->GraphicsData["CompressedBitmap", "\<\
eJztWmloVUcUvr68RLNoaxLRWpUoraKUSt1Sa+0PwQqV4g9rrUuVNEqjIMYo
xprWpIrG2lrjFkW0bkltXLEuP6oUVNwStSgWf7gEQRTBBYJb3L7eb869713n
zktuRayFd8GXM+f7zsyZmTP3zpzx05xpeeMn5UybkJuTNaggJz9vQu7UrI8n
F9iqhCaW1STPsqzaLIsybNH5Uc8E/jgFg9ySf95DSUmJ0naJa+Pa/6f2rFI0
R2WlxPj06cCqVUBRETBwoLLBtm3AqFFAeTkwdChw5Ajw5AmsEAYPhpWCFSuA
du2A2bOFv2UL0L07MHEisHCh1LtrF/rqbf+tFK0VfvUqIs/ixcD9+0BaGnD+
vOj27gW6dQNu3JD67twRTigEPH0K3LsHpKYChw8L//p1xbM+aKjRa9eijdbX
A7W1Uvnjx6I7dw4Ih0WuqpLO2B1BQoLoXP6jR1KmQw00muBt1B49/qbH7Cn1
c+dK727ejDb68CHQpg1w9qyUORJ2vYHaVFXYM7Z9OzB6tMzosGHA0aPS6UmT
gFmzgMJCNePiYqKCx4wBli8Hpkz5F31Md3TOwKhSjDAwG3JaYxpK7IZRUSGk
JUuAujq4tmvWABkZMWyV+j9eeyccR10CS8Ofi1RDKURphBFS5l+8GKiaUvi5
oERKPucbgI5TSnrRUFNKn+vQMUrNYkNWoncOZCIsdxk/Oz+fxZpEd3vR4EzH
SXFSnBQnxUlxUpz0ckk9KGWTk9sIvTvpkwmPDcKcSnhUEGYh4eFBmEWEhwZh
FhMeEoQ5h/DgIMxSwoOCMH8iPCAIs4zwR0GYyxA58zTCXEm4TxDmasI9gjDX
EX43CLOScNcgzCrCbwdhbiecFYT5O+F2QZh7CbcJwvyDcGYQ5p+EXw/CPEg4
LQjzCOGvdG01tWN07V/UjtC1x6Nr1rPw1cl4P6GvdSiV0lZCk3VInS5V2BTq
0JuUVOQX69BblGYSmq9D71CaSKhMh3pRGkko7AyU98ixWKd/SOmTyLiqw+ES
naQSPH1JahUh+Wr6lFJXktpTUue7RTppGKU3IiGszko/66QvKSVH1o4iLdRJ
yyglyXvDBKnT2486tDwCLTBC6mD3gw6VRyDfTJRHPPS1pSA1/JJQcSPW8jW9
IjKqHiZPjWZmjCOmj7zaCQDvSinVSWsMpHlGUogcn/0vBvu5OmltxN4HrTfY
zzaSlP0cHdpgsP9eJ200kEqMJNWIz77CYF+skyoNpFk66VcD6TsjSXnis99k
sP9WJ/1mIBXppCoDaaaRpDzx2W822H+jk5T6pWXKFjkunTkD9LC/0CtXuomX
VGzeLMuK2ec9e0SuqBC/N26UMvVkh5jGtTLRsyeQlwfMmBHlbNggNuvWSXn3
bimbst9uO+vt+H3wAFN0f+Xl/ZpKGrv18KFd+/Yqey4OWZLn5ePJkTLtafua
kgIrAwcPis5uBzU1sW1i5ZE97VgFuqOyf0tWg9i7d9RRJp2ZZK6ulnI03Z/8
TLrfHQgOkke2VFVWksrf5udH6821t9UFBU5iuWnMvtTWsjm++zJUvtv7NGCj
ZFPynnqOqa//zSmpV3m2gjJVWHSxA/DWLTE9fRpo21YuJfjEuoDQXXNk2Xi2
wNixwCI7jMvKJJPPlLo70KRevChl3oMkJopcX28PVHo6vWLn+DAYLlwQk8uX
/SaxgsBthsF46ZI/YJtRUp/P952p6dwZ2LRJZpbPAvsrNG4cUFrqLr2wfifi
XhcYhsGScUhTDrEThw7JSnCHbudOCSquxuJioFcvtTpVlSHs28fAGz8eWLoU
6N8fOHVK6maSnzcWDt9xzXzDsdr+bnbsCKxdC2RmqvHN1wdCBUxqNCBa4MoV
oGVLiat588TbAfZpZutWYP58uTgy3NU0PhhhFdqs0l4R/OfdKjsm/u+Ms5Hg
q5KlNNlQ2iYZjr98fw0ZIg1yidbV2XuuZOD2bXuDux84cKDBux6Tv9mOY+xu
p05qMO2Xm+jcyCKvWHdW+ZoWHc1kjBwp64AP547roV8/Kd+9K+HVNIArdkS4
8RNxhW99zqvRFVWp2ur1cSKcL+FW9r73xAlpgkuQE20PWiSrX1gI97Fk253k
7bFXrtFbTKGUGG0xVY2f+7bkQ+sOHYBjx6R88iSQbfewspLdl9Z27BCv2JpH
9rXWQlaK+wltreNK+0peQb9qWiU38h9AXpRsNfkHyLBmwg==\
\>"],
 ImageRangeCache->{{{0, 359}, {222, 0}} -> {-0.00179501, 0.9981, 1.00001, \
1.00001}}],

Cell["\<\
This problem can be modeled by discretizing and simulating it using nodes and \
links. The modeling process is illustrated using the following figure. Here a \
grid of 7\[Times]10 nodes is generated. Each node is then connected by a link \
to all other nodes that are of Manhattan distance of less than or equal to \
three. The three red nodes are assumed to be fixed to the wall, while on all \
other nodes, compression and tension forces must balance.\
\>", "Text",
 CellID->504518801],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.119048 0.0952381 0.0147151 0.0981006 [
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
% Start of user PostScript
/mathtops {
gsave
MBeginOrig
moveto
MEndOrig
currentpoint
grestore
} bind def
/MAtocoords {
mathtops 4 2 roll mathtops
4 copy pop pop
3 -1 roll sub
/arry exch def
exch sub
/arrx exch def
arrx dup mul
arry dup mul
add sqrt
/arrl exch def
translate
} bind def
/MAarrowhead1 {
gsave
MAtocoords
arrl 0. eq
{ 0 0 Mdot }
{
[ arrx arrl div
arry arrl div
-1 arry mul arrl div
arrx arrl div
0 0 ]
concat
-0.05 0.0125 moveto 0 0 lineto
-0.05 -0.0125 lineto fill
-0.05 0.0125 moveto 0 0 lineto
-0.05 -0.0125 lineto
-0.05 0.0125 lineto stroke
} ifelse
grestore
} def
% End of user PostScript
.5 g
.11905 .01472 m
.11905 .60332 L
.02381 .60332 L
.02381 .01472 L
.11905 .01472 L
F
0 g
.5 Mabswid
[ ] 0 setdash
.11905 .11282 m
.11905 .01472 L
s
.11905 .21092 m
.11905 .11282 L
s
.11905 .30902 m
.11905 .21092 L
s
.11905 .40712 m
.11905 .30902 L
s
.11905 .50522 m
.11905 .40712 L
s
.11905 .60332 m
.11905 .50522 L
s
.21429 .01472 m
.11905 .01472 L
s
.21429 .01472 m
.11905 .11282 L
s
.21429 .01472 m
.11905 .21092 L
s
.21429 .01472 m
.11905 .30902 L
s
.21429 .11282 m
.11905 .01472 L
s
.21429 .11282 m
.11905 .11282 L
s
.21429 .11282 m
.11905 .21092 L
s
.21429 .11282 m
.11905 .30902 L
s
.21429 .11282 m
.11905 .40712 L
s
.21429 .11282 m
.21429 .01472 L
s
.21429 .21092 m
.11905 .01472 L
s
.21429 .21092 m
.11905 .11282 L
s
.21429 .21092 m
.11905 .21092 L
s
.21429 .21092 m
.11905 .30902 L
s
.21429 .21092 m
.11905 .40712 L
s
.21429 .21092 m
.11905 .50522 L
s
.21429 .21092 m
.21429 .11282 L
s
.21429 .30902 m
.11905 .01472 L
s
.21429 .30902 m
.11905 .11282 L
s
.21429 .30902 m
.11905 .21092 L
s
.21429 .30902 m
.11905 .30902 L
s
.21429 .30902 m
.11905 .40712 L
s
.21429 .30902 m
.11905 .50522 L
s
.21429 .30902 m
.11905 .60332 L
s
.21429 .30902 m
.21429 .21092 L
s
.21429 .40712 m
.11905 .11282 L
s
.21429 .40712 m
.11905 .21092 L
s
.21429 .40712 m
.11905 .30902 L
s
.21429 .40712 m
.11905 .40712 L
s
.21429 .40712 m
.11905 .50522 L
s
.21429 .40712 m
.11905 .60332 L
s
.21429 .40712 m
.21429 .30902 L
s
.21429 .50522 m
.11905 .21092 L
s
.21429 .50522 m
.11905 .30902 L
s
.21429 .50522 m
.11905 .40712 L
s
.21429 .50522 m
.11905 .50522 L
s
.21429 .50522 m
.11905 .60332 L
s
.21429 .50522 m
.21429 .40712 L
s
.21429 .60332 m
.11905 .30902 L
s
.21429 .60332 m
.11905 .40712 L
s
.21429 .60332 m
.11905 .50522 L
s
.21429 .60332 m
.11905 .60332 L
s
.21429 .60332 m
.21429 .50522 L
s
.30952 .01472 m
.11905 .11282 L
s
.30952 .01472 m
.11905 .30902 L
s
.30952 .01472 m
.21429 .01472 L
s
.30952 .01472 m
.21429 .11282 L
s
.30952 .01472 m
.21429 .21092 L
s
.30952 .01472 m
.21429 .30902 L
s
.30952 .11282 m
.11905 .01472 L
s
.30952 .11282 m
.11905 .21092 L
s
.30952 .11282 m
.11905 .40712 L
s
.30952 .11282 m
.21429 .01472 L
s
.30952 .11282 m
.21429 .11282 L
s
.30952 .11282 m
.21429 .21092 L
s
.30952 .11282 m
.21429 .30902 L
s
.30952 .11282 m
.21429 .40712 L
s
.30952 .11282 m
.30952 .01472 L
s
.30952 .21092 m
.11905 .11282 L
s
.30952 .21092 m
.11905 .30902 L
s
.30952 .21092 m
.11905 .50522 L
s
.30952 .21092 m
.21429 .01472 L
s
.30952 .21092 m
.21429 .11282 L
s
.30952 .21092 m
.21429 .21092 L
s
.30952 .21092 m
.21429 .30902 L
s
.30952 .21092 m
.21429 .40712 L
s
.30952 .21092 m
.21429 .50522 L
s
.30952 .21092 m
.30952 .11282 L
s
.30952 .30902 m
.11905 .01472 L
s
.30952 .30902 m
.11905 .21092 L
s
.30952 .30902 m
.11905 .40712 L
s
.30952 .30902 m
.11905 .60332 L
s
.30952 .30902 m
.21429 .01472 L
s
.30952 .30902 m
.21429 .11282 L
s
.30952 .30902 m
.21429 .21092 L
s
.30952 .30902 m
.21429 .30902 L
s
.30952 .30902 m
.21429 .40712 L
s
.30952 .30902 m
.21429 .50522 L
s
.30952 .30902 m
.21429 .60332 L
s
.30952 .30902 m
.30952 .21092 L
s
.30952 .40712 m
.11905 .11282 L
s
.30952 .40712 m
.11905 .30902 L
s
.30952 .40712 m
.11905 .50522 L
s
.30952 .40712 m
.21429 .11282 L
s
.30952 .40712 m
.21429 .21092 L
s
.30952 .40712 m
.21429 .30902 L
s
.30952 .40712 m
.21429 .40712 L
s
.30952 .40712 m
.21429 .50522 L
s
.30952 .40712 m
.21429 .60332 L
s
.30952 .40712 m
.30952 .30902 L
s
.30952 .50522 m
.11905 .21092 L
s
.30952 .50522 m
.11905 .40712 L
s
.30952 .50522 m
.11905 .60332 L
s
.30952 .50522 m
.21429 .21092 L
s
.30952 .50522 m
.21429 .30902 L
s
.30952 .50522 m
.21429 .40712 L
s
.30952 .50522 m
.21429 .50522 L
s
.30952 .50522 m
.21429 .60332 L
s
.30952 .50522 m
.30952 .40712 L
s
.30952 .60332 m
.11905 .30902 L
s
.30952 .60332 m
.11905 .50522 L
s
.30952 .60332 m
.21429 .30902 L
s
.30952 .60332 m
.21429 .40712 L
s
.30952 .60332 m
.21429 .50522 L
s
.30952 .60332 m
.21429 .60332 L
s
.30952 .60332 m
.30952 .50522 L
s
.40476 .01472 m
.11905 .11282 L
s
.40476 .01472 m
.11905 .21092 L
s
.40476 .01472 m
.21429 .11282 L
s
.40476 .01472 m
.21429 .30902 L
s
.40476 .01472 m
.30952 .01472 L
s
.40476 .01472 m
.30952 .11282 L
s
.40476 .01472 m
.30952 .21092 L
s
.40476 .01472 m
.30952 .30902 L
s
.40476 .11282 m
.11905 .01472 L
s
.40476 .11282 m
.11905 .21092 L
s
.40476 .11282 m
.11905 .30902 L
s
.40476 .11282 m
.21429 .01472 L
s
.40476 .11282 m
.21429 .21092 L
s
.40476 .11282 m
.21429 .40712 L
s
.40476 .11282 m
.30952 .01472 L
s
.40476 .11282 m
.30952 .11282 L
s
.40476 .11282 m
.30952 .21092 L
s
.40476 .11282 m
.30952 .30902 L
s
.40476 .11282 m
.30952 .40712 L
s
.40476 .11282 m
.40476 .01472 L
s
.40476 .21092 m
.11905 .01472 L
s
.40476 .21092 m
.11905 .11282 L
s
.40476 .21092 m
.11905 .30902 L
s
.40476 .21092 m
.11905 .40712 L
s
.40476 .21092 m
.21429 .11282 L
s
.40476 .21092 m
.21429 .30902 L
s
.40476 .21092 m
.21429 .50522 L
s
.40476 .21092 m
.30952 .01472 L
s
.40476 .21092 m
.30952 .11282 L
s
.40476 .21092 m
.30952 .21092 L
s
.40476 .21092 m
.30952 .30902 L
s
.40476 .21092 m
.30952 .40712 L
s
.40476 .21092 m
.30952 .50522 L
s
.40476 .21092 m
.40476 .11282 L
s
.40476 .30902 m
.11905 .11282 L
s
.40476 .30902 m
.11905 .21092 L
s
.40476 .30902 m
.11905 .40712 L
s
.40476 .30902 m
.11905 .50522 L
s
.40476 .30902 m
.21429 .01472 L
s
.40476 .30902 m
.21429 .21092 L
s
.40476 .30902 m
.21429 .40712 L
s
.40476 .30902 m
.21429 .60332 L
s
.40476 .30902 m
.30952 .01472 L
s
.40476 .30902 m
.30952 .11282 L
s
.40476 .30902 m
.30952 .21092 L
s
.40476 .30902 m
.30952 .30902 L
s
.40476 .30902 m
.30952 .40712 L
s
.40476 .30902 m
.30952 .50522 L
s
.40476 .30902 m
.30952 .60332 L
s
.40476 .30902 m
.40476 .21092 L
s
.40476 .40712 m
.11905 .21092 L
s
.40476 .40712 m
.11905 .30902 L
s
.40476 .40712 m
.11905 .50522 L
s
.40476 .40712 m
.11905 .60332 L
s
.40476 .40712 m
.21429 .11282 L
s
.40476 .40712 m
.21429 .30902 L
s
.40476 .40712 m
.21429 .50522 L
s
.40476 .40712 m
.30952 .11282 L
s
.40476 .40712 m
.30952 .21092 L
s
.40476 .40712 m
.30952 .30902 L
s
.40476 .40712 m
.30952 .40712 L
s
.40476 .40712 m
.30952 .50522 L
s
.40476 .40712 m
.30952 .60332 L
s
.40476 .40712 m
.40476 .30902 L
s
.40476 .50522 m
.11905 .30902 L
s
.40476 .50522 m
.11905 .40712 L
s
.40476 .50522 m
.11905 .60332 L
s
.40476 .50522 m
.21429 .21092 L
s
.40476 .50522 m
.21429 .40712 L
s
.40476 .50522 m
.21429 .60332 L
s
.40476 .50522 m
.30952 .21092 L
s
.40476 .50522 m
.30952 .30902 L
s
.40476 .50522 m
.30952 .40712 L
s
.40476 .50522 m
.30952 .50522 L
s
.40476 .50522 m
.30952 .60332 L
s
.40476 .50522 m
.40476 .40712 L
s
.40476 .60332 m
.11905 .40712 L
s
.40476 .60332 m
.11905 .50522 L
s
.40476 .60332 m
.21429 .30902 L
s
.40476 .60332 m
.21429 .50522 L
s
.40476 .60332 m
.30952 .30902 L
s
.40476 .60332 m
.30952 .40712 L
s
.40476 .60332 m
.30952 .50522 L
s
.40476 .60332 m
.30952 .60332 L
s
.40476 .60332 m
.40476 .50522 L
s
.5 .01472 m
.21429 .11282 L
s
.5 .01472 m
.21429 .21092 L
s
.5 .01472 m
.30952 .11282 L
s
.5 .01472 m
.30952 .30902 L
s
.5 .01472 m
.40476 .01472 L
s
.5 .01472 m
.40476 .11282 L
s
.5 .01472 m
.40476 .21092 L
s
.5 .01472 m
.40476 .30902 L
s
.5 .11282 m
.21429 .01472 L
s
.5 .11282 m
.21429 .21092 L
s
.5 .11282 m
.21429 .30902 L
s
.5 .11282 m
.30952 .01472 L
s
.5 .11282 m
.30952 .21092 L
s
.5 .11282 m
.30952 .40712 L
s
.5 .11282 m
.40476 .01472 L
s
.5 .11282 m
.40476 .11282 L
s
.5 .11282 m
.40476 .21092 L
s
.5 .11282 m
.40476 .30902 L
s
.5 .11282 m
.40476 .40712 L
s
.5 .11282 m
.5 .01472 L
s
.5 .21092 m
.21429 .01472 L
s
.5 .21092 m
.21429 .11282 L
s
.5 .21092 m
.21429 .30902 L
s
.5 .21092 m
.21429 .40712 L
s
.5 .21092 m
.30952 .11282 L
s
.5 .21092 m
.30952 .30902 L
s
.5 .21092 m
.30952 .50522 L
s
.5 .21092 m
.40476 .01472 L
s
.5 .21092 m
.40476 .11282 L
s
.5 .21092 m
.40476 .21092 L
s
.5 .21092 m
.40476 .30902 L
s
.5 .21092 m
.40476 .40712 L
s
.5 .21092 m
.40476 .50522 L
s
.5 .21092 m
.5 .11282 L
s
.5 .30902 m
.21429 .11282 L
s
.5 .30902 m
.21429 .21092 L
s
.5 .30902 m
.21429 .40712 L
s
.5 .30902 m
.21429 .50522 L
s
.5 .30902 m
.30952 .01472 L
s
.5 .30902 m
.30952 .21092 L
s
.5 .30902 m
.30952 .40712 L
s
.5 .30902 m
.30952 .60332 L
s
.5 .30902 m
.40476 .01472 L
s
.5 .30902 m
.40476 .11282 L
s
.5 .30902 m
.40476 .21092 L
s
.5 .30902 m
.40476 .30902 L
s
.5 .30902 m
.40476 .40712 L
s
.5 .30902 m
.40476 .50522 L
s
.5 .30902 m
.40476 .60332 L
s
.5 .30902 m
.5 .21092 L
s
.5 .40712 m
.21429 .21092 L
s
.5 .40712 m
.21429 .30902 L
s
.5 .40712 m
.21429 .50522 L
s
.5 .40712 m
.21429 .60332 L
s
.5 .40712 m
.30952 .11282 L
s
.5 .40712 m
.30952 .30902 L
s
.5 .40712 m
.30952 .50522 L
s
.5 .40712 m
.40476 .11282 L
s
.5 .40712 m
.40476 .21092 L
s
.5 .40712 m
.40476 .30902 L
s
.5 .40712 m
.40476 .40712 L
s
.5 .40712 m
.40476 .50522 L
s
.5 .40712 m
.40476 .60332 L
s
.5 .40712 m
.5 .30902 L
s
.5 .50522 m
.21429 .30902 L
s
.5 .50522 m
.21429 .40712 L
s
.5 .50522 m
.21429 .60332 L
s
.5 .50522 m
.30952 .21092 L
s
.5 .50522 m
.30952 .40712 L
s
.5 .50522 m
.30952 .60332 L
s
.5 .50522 m
.40476 .21092 L
s
.5 .50522 m
.40476 .30902 L
s
.5 .50522 m
.40476 .40712 L
s
.5 .50522 m
.40476 .50522 L
s
.5 .50522 m
.40476 .60332 L
s
.5 .50522 m
.5 .40712 L
s
.5 .60332 m
.21429 .40712 L
s
.5 .60332 m
.21429 .50522 L
s
.5 .60332 m
.30952 .30902 L
s
.5 .60332 m
.30952 .50522 L
s
.5 .60332 m
.40476 .30902 L
s
.5 .60332 m
.40476 .40712 L
s
.5 .60332 m
.40476 .50522 L
s
.5 .60332 m
.40476 .60332 L
s
.5 .60332 m
.5 .50522 L
s
.59524 .01472 m
.30952 .11282 L
s
.59524 .01472 m
.30952 .21092 L
s
.59524 .01472 m
.40476 .11282 L
s
.59524 .01472 m
.40476 .30902 L
s
.59524 .01472 m
.5 .01472 L
s
.59524 .01472 m
.5 .11282 L
s
.59524 .01472 m
.5 .21092 L
s
.59524 .01472 m
.5 .30902 L
s
.59524 .11282 m
.30952 .01472 L
s
.59524 .11282 m
.30952 .21092 L
s
.59524 .11282 m
.30952 .30902 L
s
.59524 .11282 m
.40476 .01472 L
s
.59524 .11282 m
.40476 .21092 L
s
.59524 .11282 m
.40476 .40712 L
s
.59524 .11282 m
.5 .01472 L
s
.59524 .11282 m
.5 .11282 L
s
.59524 .11282 m
.5 .21092 L
s
.59524 .11282 m
.5 .30902 L
s
.59524 .11282 m
.5 .40712 L
s
.59524 .11282 m
.59524 .01472 L
s
.59524 .21092 m
.30952 .01472 L
s
.59524 .21092 m
.30952 .11282 L
s
.59524 .21092 m
.30952 .30902 L
s
.59524 .21092 m
.30952 .40712 L
s
.59524 .21092 m
.40476 .11282 L
s
.59524 .21092 m
.40476 .30902 L
s
.59524 .21092 m
.40476 .50522 L
s
.59524 .21092 m
.5 .01472 L
s
.59524 .21092 m
.5 .11282 L
s
.59524 .21092 m
.5 .21092 L
s
.59524 .21092 m
.5 .30902 L
s
.59524 .21092 m
.5 .40712 L
s
.59524 .21092 m
.5 .50522 L
s
.59524 .21092 m
.59524 .11282 L
s
.59524 .30902 m
.30952 .11282 L
s
.59524 .30902 m
.30952 .21092 L
s
.59524 .30902 m
.30952 .40712 L
s
.59524 .30902 m
.30952 .50522 L
s
.59524 .30902 m
.40476 .01472 L
s
.59524 .30902 m
.40476 .21092 L
s
.59524 .30902 m
.40476 .40712 L
s
.59524 .30902 m
.40476 .60332 L
s
.59524 .30902 m
.5 .01472 L
s
.59524 .30902 m
.5 .11282 L
s
.59524 .30902 m
.5 .21092 L
s
.59524 .30902 m
.5 .30902 L
s
.59524 .30902 m
.5 .40712 L
s
.59524 .30902 m
.5 .50522 L
s
.59524 .30902 m
.5 .60332 L
s
.59524 .30902 m
.59524 .21092 L
s
.59524 .40712 m
.30952 .21092 L
s
.59524 .40712 m
.30952 .30902 L
s
.59524 .40712 m
.30952 .50522 L
s
.59524 .40712 m
.30952 .60332 L
s
.59524 .40712 m
.40476 .11282 L
s
.59524 .40712 m
.40476 .30902 L
s
.59524 .40712 m
.40476 .50522 L
s
.59524 .40712 m
.5 .11282 L
s
.59524 .40712 m
.5 .21092 L
s
.59524 .40712 m
.5 .30902 L
s
.59524 .40712 m
.5 .40712 L
s
.59524 .40712 m
.5 .50522 L
s
.59524 .40712 m
.5 .60332 L
s
.59524 .40712 m
.59524 .30902 L
s
.59524 .50522 m
.30952 .30902 L
s
.59524 .50522 m
.30952 .40712 L
s
.59524 .50522 m
.30952 .60332 L
s
.59524 .50522 m
.40476 .21092 L
s
.59524 .50522 m
.40476 .40712 L
s
.59524 .50522 m
.40476 .60332 L
s
.59524 .50522 m
.5 .21092 L
s
.59524 .50522 m
.5 .30902 L
s
.59524 .50522 m
.5 .40712 L
s
.59524 .50522 m
.5 .50522 L
s
.59524 .50522 m
.5 .60332 L
s
.59524 .50522 m
.59524 .40712 L
s
.59524 .60332 m
.30952 .40712 L
s
.59524 .60332 m
.30952 .50522 L
s
.59524 .60332 m
.40476 .30902 L
s
.59524 .60332 m
.40476 .50522 L
s
.59524 .60332 m
.5 .30902 L
s
.59524 .60332 m
.5 .40712 L
s
.59524 .60332 m
.5 .50522 L
s
.59524 .60332 m
.5 .60332 L
s
.59524 .60332 m
.59524 .50522 L
s
.69048 .01472 m
.40476 .11282 L
s
.69048 .01472 m
.40476 .21092 L
s
.69048 .01472 m
.5 .11282 L
s
.69048 .01472 m
.5 .30902 L
s
.69048 .01472 m
.59524 .01472 L
s
.69048 .01472 m
.59524 .11282 L
s
.69048 .01472 m
.59524 .21092 L
s
.69048 .01472 m
.59524 .30902 L
s
.69048 .11282 m
.40476 .01472 L
s
.69048 .11282 m
.40476 .21092 L
s
.69048 .11282 m
.40476 .30902 L
s
.69048 .11282 m
.5 .01472 L
s
.69048 .11282 m
.5 .21092 L
s
.69048 .11282 m
.5 .40712 L
s
.69048 .11282 m
.59524 .01472 L
s
.69048 .11282 m
.59524 .11282 L
s
.69048 .11282 m
.59524 .21092 L
s
.69048 .11282 m
.59524 .30902 L
s
.69048 .11282 m
.59524 .40712 L
s
.69048 .11282 m
.69048 .01472 L
s
.69048 .21092 m
.40476 .01472 L
s
.69048 .21092 m
.40476 .11282 L
s
.69048 .21092 m
.40476 .30902 L
s
.69048 .21092 m
.40476 .40712 L
s
.69048 .21092 m
.5 .11282 L
s
.69048 .21092 m
.5 .30902 L
s
.69048 .21092 m
.5 .50522 L
s
.69048 .21092 m
.59524 .01472 L
s
.69048 .21092 m
.59524 .11282 L
s
.69048 .21092 m
.59524 .21092 L
s
.69048 .21092 m
.59524 .30902 L
s
.69048 .21092 m
.59524 .40712 L
s
.69048 .21092 m
.59524 .50522 L
s
.69048 .21092 m
.69048 .11282 L
s
.69048 .30902 m
.40476 .11282 L
s
.69048 .30902 m
.40476 .21092 L
s
.69048 .30902 m
.40476 .40712 L
s
.69048 .30902 m
.40476 .50522 L
s
.69048 .30902 m
.5 .01472 L
s
.69048 .30902 m
.5 .21092 L
s
.69048 .30902 m
.5 .40712 L
s
.69048 .30902 m
.5 .60332 L
s
.69048 .30902 m
.59524 .01472 L
s
.69048 .30902 m
.59524 .11282 L
s
.69048 .30902 m
.59524 .21092 L
s
.69048 .30902 m
.59524 .30902 L
s
.69048 .30902 m
.59524 .40712 L
s
.69048 .30902 m
.59524 .50522 L
s
.69048 .30902 m
.59524 .60332 L
s
.69048 .30902 m
.69048 .21092 L
s
.69048 .40712 m
.40476 .21092 L
s
.69048 .40712 m
.40476 .30902 L
s
.69048 .40712 m
.40476 .50522 L
s
.69048 .40712 m
.40476 .60332 L
s
.69048 .40712 m
.5 .11282 L
s
.69048 .40712 m
.5 .30902 L
s
.69048 .40712 m
.5 .50522 L
s
.69048 .40712 m
.59524 .11282 L
s
.69048 .40712 m
.59524 .21092 L
s
.69048 .40712 m
.59524 .30902 L
s
.69048 .40712 m
.59524 .40712 L
s
.69048 .40712 m
.59524 .50522 L
s
.69048 .40712 m
.59524 .60332 L
s
.69048 .40712 m
.69048 .30902 L
s
.69048 .50522 m
.40476 .30902 L
s
.69048 .50522 m
.40476 .40712 L
s
.69048 .50522 m
.40476 .60332 L
s
.69048 .50522 m
.5 .21092 L
s
.69048 .50522 m
.5 .40712 L
s
.69048 .50522 m
.5 .60332 L
s
.69048 .50522 m
.59524 .21092 L
s
.69048 .50522 m
.59524 .30902 L
s
.69048 .50522 m
.59524 .40712 L
s
.69048 .50522 m
.59524 .50522 L
s
.69048 .50522 m
.59524 .60332 L
s
.69048 .50522 m
.69048 .40712 L
s
.69048 .60332 m
.40476 .40712 L
s
.69048 .60332 m
.40476 .50522 L
s
.69048 .60332 m
.5 .30902 L
s
.69048 .60332 m
.5 .50522 L
s
.69048 .60332 m
.59524 .30902 L
s
.69048 .60332 m
.59524 .40712 L
s
.69048 .60332 m
.59524 .50522 L
s
.69048 .60332 m
.59524 .60332 L
s
.69048 .60332 m
.69048 .50522 L
s
.78571 .01472 m
.5 .11282 L
s
.78571 .01472 m
.5 .21092 L
s
.78571 .01472 m
.59524 .11282 L
s
.78571 .01472 m
.59524 .30902 L
s
.78571 .01472 m
.69048 .01472 L
s
.78571 .01472 m
.69048 .11282 L
s
.78571 .01472 m
.69048 .21092 L
s
.78571 .01472 m
.69048 .30902 L
s
.78571 .11282 m
.5 .01472 L
s
.78571 .11282 m
.5 .21092 L
s
.78571 .11282 m
.5 .30902 L
s
.78571 .11282 m
.59524 .01472 L
s
.78571 .11282 m
.59524 .21092 L
s
.78571 .11282 m
.59524 .40712 L
s
.78571 .11282 m
.69048 .01472 L
s
.78571 .11282 m
.69048 .11282 L
s
.78571 .11282 m
.69048 .21092 L
s
.78571 .11282 m
.69048 .30902 L
s
.78571 .11282 m
.69048 .40712 L
s
.78571 .11282 m
.78571 .01472 L
s
.78571 .21092 m
.5 .01472 L
s
.78571 .21092 m
.5 .11282 L
s
.78571 .21092 m
.5 .30902 L
s
.78571 .21092 m
.5 .40712 L
s
.78571 .21092 m
.59524 .11282 L
s
.78571 .21092 m
.59524 .30902 L
s
.78571 .21092 m
.59524 .50522 L
s
.78571 .21092 m
.69048 .01472 L
s
.78571 .21092 m
.69048 .11282 L
s
.78571 .21092 m
.69048 .21092 L
s
.78571 .21092 m
.69048 .30902 L
s
.78571 .21092 m
.69048 .40712 L
s
.78571 .21092 m
.69048 .50522 L
s
.78571 .21092 m
.78571 .11282 L
s
.78571 .30902 m
.5 .11282 L
s
.78571 .30902 m
.5 .21092 L
s
.78571 .30902 m
.5 .40712 L
s
.78571 .30902 m
.5 .50522 L
s
.78571 .30902 m
.59524 .01472 L
s
.78571 .30902 m
.59524 .21092 L
s
.78571 .30902 m
.59524 .40712 L
s
.78571 .30902 m
.59524 .60332 L
s
.78571 .30902 m
.69048 .01472 L
s
.78571 .30902 m
.69048 .11282 L
s
.78571 .30902 m
.69048 .21092 L
s
.78571 .30902 m
.69048 .30902 L
s
.78571 .30902 m
.69048 .40712 L
s
.78571 .30902 m
.69048 .50522 L
s
.78571 .30902 m
.69048 .60332 L
s
.78571 .30902 m
.78571 .21092 L
s
.78571 .40712 m
.5 .21092 L
s
.78571 .40712 m
.5 .30902 L
s
.78571 .40712 m
.5 .50522 L
s
.78571 .40712 m
.5 .60332 L
s
.78571 .40712 m
.59524 .11282 L
s
.78571 .40712 m
.59524 .30902 L
s
.78571 .40712 m
.59524 .50522 L
s
.78571 .40712 m
.69048 .11282 L
s
.78571 .40712 m
.69048 .21092 L
s
.78571 .40712 m
.69048 .30902 L
s
.78571 .40712 m
.69048 .40712 L
s
.78571 .40712 m
.69048 .50522 L
s
.78571 .40712 m
.69048 .60332 L
s
.78571 .40712 m
.78571 .30902 L
s
.78571 .50522 m
.5 .30902 L
s
.78571 .50522 m
.5 .40712 L
s
.78571 .50522 m
.5 .60332 L
s
.78571 .50522 m
.59524 .21092 L
s
.78571 .50522 m
.59524 .40712 L
s
.78571 .50522 m
.59524 .60332 L
s
.78571 .50522 m
.69048 .21092 L
s
.78571 .50522 m
.69048 .30902 L
s
.78571 .50522 m
.69048 .40712 L
s
.78571 .50522 m
.69048 .50522 L
s
.78571 .50522 m
.69048 .60332 L
s
.78571 .50522 m
.78571 .40712 L
s
.78571 .60332 m
.5 .40712 L
s
.78571 .60332 m
.5 .50522 L
s
.78571 .60332 m
.59524 .30902 L
s
.78571 .60332 m
.59524 .50522 L
s
.78571 .60332 m
.69048 .30902 L
s
.78571 .60332 m
.69048 .40712 L
s
.78571 .60332 m
.69048 .50522 L
s
.78571 .60332 m
.69048 .60332 L
s
.78571 .60332 m
.78571 .50522 L
s
.88095 .01472 m
.59524 .11282 L
s
.88095 .01472 m
.59524 .21092 L
s
.88095 .01472 m
.69048 .11282 L
s
.88095 .01472 m
.69048 .30902 L
s
.88095 .01472 m
.78571 .01472 L
s
.88095 .01472 m
.78571 .11282 L
s
.88095 .01472 m
.78571 .21092 L
s
.88095 .01472 m
.78571 .30902 L
s
.88095 .11282 m
.59524 .01472 L
s
.88095 .11282 m
.59524 .21092 L
s
.88095 .11282 m
.59524 .30902 L
s
.88095 .11282 m
.69048 .01472 L
s
.88095 .11282 m
.69048 .21092 L
s
.88095 .11282 m
.69048 .40712 L
s
.88095 .11282 m
.78571 .01472 L
s
.88095 .11282 m
.78571 .11282 L
s
.88095 .11282 m
.78571 .21092 L
s
.88095 .11282 m
.78571 .30902 L
s
.88095 .11282 m
.78571 .40712 L
s
.88095 .11282 m
.88095 .01472 L
s
.88095 .21092 m
.59524 .01472 L
s
.88095 .21092 m
.59524 .11282 L
s
.88095 .21092 m
.59524 .30902 L
s
.88095 .21092 m
.59524 .40712 L
s
.88095 .21092 m
.69048 .11282 L
s
.88095 .21092 m
.69048 .30902 L
s
.88095 .21092 m
.69048 .50522 L
s
.88095 .21092 m
.78571 .01472 L
s
.88095 .21092 m
.78571 .11282 L
s
.88095 .21092 m
.78571 .21092 L
s
.88095 .21092 m
.78571 .30902 L
s
.88095 .21092 m
.78571 .40712 L
s
.88095 .21092 m
.78571 .50522 L
s
.88095 .21092 m
.88095 .11282 L
s
.88095 .30902 m
.59524 .11282 L
s
.88095 .30902 m
.59524 .21092 L
s
.88095 .30902 m
.59524 .40712 L
s
.88095 .30902 m
.59524 .50522 L
s
.88095 .30902 m
.69048 .01472 L
s
.88095 .30902 m
.69048 .21092 L
s
.88095 .30902 m
.69048 .40712 L
s
.88095 .30902 m
.69048 .60332 L
s
.88095 .30902 m
.78571 .01472 L
s
.88095 .30902 m
.78571 .11282 L
s
.88095 .30902 m
.78571 .21092 L
s
.88095 .30902 m
.78571 .30902 L
s
.88095 .30902 m
.78571 .40712 L
s
.88095 .30902 m
.78571 .50522 L
s
.88095 .30902 m
.78571 .60332 L
s
.88095 .30902 m
.88095 .21092 L
s
.88095 .40712 m
.59524 .21092 L
s
.88095 .40712 m
.59524 .30902 L
s
.88095 .40712 m
.59524 .50522 L
s
.88095 .40712 m
.59524 .60332 L
s
.88095 .40712 m
.69048 .11282 L
s
.88095 .40712 m
.69048 .30902 L
s
.88095 .40712 m
.69048 .50522 L
s
.88095 .40712 m
.78571 .11282 L
s
.88095 .40712 m
.78571 .21092 L
s
.88095 .40712 m
.78571 .30902 L
s
.88095 .40712 m
.78571 .40712 L
s
.88095 .40712 m
.78571 .50522 L
s
.88095 .40712 m
.78571 .60332 L
s
.88095 .40712 m
.88095 .30902 L
s
.88095 .50522 m
.59524 .30902 L
s
.88095 .50522 m
.59524 .40712 L
s
.88095 .50522 m
.59524 .60332 L
s
.88095 .50522 m
.69048 .21092 L
s
.88095 .50522 m
.69048 .40712 L
s
.88095 .50522 m
.69048 .60332 L
s
.88095 .50522 m
.78571 .21092 L
s
.88095 .50522 m
.78571 .30902 L
s
.88095 .50522 m
.78571 .40712 L
s
.88095 .50522 m
.78571 .50522 L
s
.88095 .50522 m
.78571 .60332 L
s
.88095 .50522 m
.88095 .40712 L
s
.88095 .60332 m
.59524 .40712 L
s
.88095 .60332 m
.59524 .50522 L
s
.88095 .60332 m
.69048 .30902 L
s
.88095 .60332 m
.69048 .50522 L
s
.88095 .60332 m
.78571 .30902 L
s
.88095 .60332 m
.78571 .40712 L
s
.88095 .60332 m
.78571 .50522 L
s
.88095 .60332 m
.78571 .60332 L
s
.88095 .60332 m
.88095 .50522 L
s
.97619 .01472 m
.69048 .11282 L
s
.97619 .01472 m
.69048 .21092 L
s
.97619 .01472 m
.78571 .11282 L
s
.97619 .01472 m
.78571 .30902 L
s
.97619 .01472 m
.88095 .01472 L
s
.97619 .01472 m
.88095 .11282 L
s
.97619 .01472 m
.88095 .21092 L
s
.97619 .01472 m
.88095 .30902 L
s
.97619 .11282 m
.69048 .01472 L
s
.97619 .11282 m
.69048 .21092 L
s
.97619 .11282 m
.69048 .30902 L
s
.97619 .11282 m
.78571 .01472 L
s
.97619 .11282 m
.78571 .21092 L
s
.97619 .11282 m
.78571 .40712 L
s
.97619 .11282 m
.88095 .01472 L
s
.97619 .11282 m
.88095 .11282 L
s
.97619 .11282 m
.88095 .21092 L
s
.97619 .11282 m
.88095 .30902 L
s
.97619 .11282 m
.88095 .40712 L
s
.97619 .11282 m
.97619 .01472 L
s
.97619 .21092 m
.69048 .01472 L
s
.97619 .21092 m
.69048 .11282 L
s
.97619 .21092 m
.69048 .30902 L
s
.97619 .21092 m
.69048 .40712 L
s
.97619 .21092 m
.78571 .11282 L
s
.97619 .21092 m
.78571 .30902 L
s
.97619 .21092 m
.78571 .50522 L
s
.97619 .21092 m
.88095 .01472 L
s
.97619 .21092 m
.88095 .11282 L
s
.97619 .21092 m
.88095 .21092 L
s
.97619 .21092 m
.88095 .30902 L
s
.97619 .21092 m
.88095 .40712 L
s
.97619 .21092 m
.88095 .50522 L
s
.97619 .21092 m
.97619 .11282 L
s
.97619 .30902 m
.69048 .11282 L
s
.97619 .30902 m
.69048 .21092 L
s
.97619 .30902 m
.69048 .40712 L
s
.97619 .30902 m
.69048 .50522 L
s
.97619 .30902 m
.78571 .01472 L
s
.97619 .30902 m
.78571 .21092 L
s
.97619 .30902 m
.78571 .40712 L
s
.97619 .30902 m
.78571 .60332 L
s
.97619 .30902 m
.88095 .01472 L
s
.97619 .30902 m
.88095 .11282 L
s
.97619 .30902 m
.88095 .21092 L
s
.97619 .30902 m
.88095 .30902 L
s
.97619 .30902 m
.88095 .40712 L
s
.97619 .30902 m
.88095 .50522 L
s
.97619 .30902 m
.88095 .60332 L
s
.97619 .30902 m
.97619 .21092 L
s
.97619 .40712 m
.69048 .21092 L
s
.97619 .40712 m
.69048 .30902 L
s
.97619 .40712 m
.69048 .50522 L
s
.97619 .40712 m
.69048 .60332 L
s
.97619 .40712 m
.78571 .11282 L
s
.97619 .40712 m
.78571 .30902 L
s
.97619 .40712 m
.78571 .50522 L
s
.97619 .40712 m
.88095 .11282 L
s
.97619 .40712 m
.88095 .21092 L
s
.97619 .40712 m
.88095 .30902 L
s
.97619 .40712 m
.88095 .40712 L
s
.97619 .40712 m
.88095 .50522 L
s
.97619 .40712 m
.88095 .60332 L
s
.97619 .40712 m
.97619 .30902 L
s
.97619 .50522 m
.69048 .30902 L
s
.97619 .50522 m
.69048 .40712 L
s
.97619 .50522 m
.69048 .60332 L
s
.97619 .50522 m
.78571 .21092 L
s
.97619 .50522 m
.78571 .40712 L
s
.97619 .50522 m
.78571 .60332 L
s
.97619 .50522 m
.88095 .21092 L
s
.97619 .50522 m
.88095 .30902 L
s
.97619 .50522 m
.88095 .40712 L
s
.97619 .50522 m
.88095 .50522 L
s
.97619 .50522 m
.88095 .60332 L
s
.97619 .50522 m
.97619 .40712 L
s
.97619 .60332 m
.69048 .40712 L
s
.97619 .60332 m
.69048 .50522 L
s
.97619 .60332 m
.78571 .30902 L
s
.97619 .60332 m
.78571 .50522 L
s
.97619 .60332 m
.88095 .30902 L
s
.97619 .60332 m
.88095 .40712 L
s
.97619 .60332 m
.88095 .50522 L
s
.97619 .60332 m
.88095 .60332 L
s
.97619 .60332 m
.97619 .50522 L
s
1 0 0 r
.97619 .30902 m
.97619 .21092 L
s
% Start of user PostScript
9. 3. 9. 2. MAarrowhead1
% End of user PostScript
.008 w
.11905 .21092 Mdot
.11905 .30902 Mdot
.11905 .40712 Mdot
% End of Graphics
MathPictureEnd
\
\>"], "Text",
 ImageSize->{454.25, 280.5},
 ImageMargins->{{20.25, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellID->272052816,
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"],
 ImageRangeCache->{{{0, 453.25}, {279.5, 0}} -> {-1.26174, -0.150032, \
0.0232178, 0.0225403}}],

Cell["\<\
Each link represents a rigid rod that has a thickness, with its weight \
proportional to the force on it and its length. The aim is to minimize the \
total material used, which is \
\>", "Text",
 CellID->316296097],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
   "Material", " ", "needed", " ", "to", " ", "construct", " ", "a", " ", 
    "link"}], "=", 
   RowBox[{"force", "*", "link_length"}]}], TraditionalForm]], "DisplayMath",
 SingleLetterItalics->False,
 CellID->627637036],

Cell[BoxData[
 FormBox[
  RowBox[{
  "subject", " ", "to", " ", "force", " ", "balance", " ", "on", " ", "each", 
   " ", "node", " ", "except", " ", "the", " ", "fixed", " ", 
   RowBox[{"ones", "."}]}], TraditionalForm]], "DisplayMath",
 CellID->142308570],

Cell["\<\
Hence mathematically this is a linearly constrained minimization problem, \
with objective function a sum of absolute values of linear functions.\
\>", "Text",
 CellID->390858683],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{"Minimize", " ", "\[Sum]"}], "|", 
   RowBox[{"force", "*", "link_length"}], "|"}], 
  TraditionalForm]], "DisplayMath",
 CellID->113688339],

Cell[BoxData[
 FormBox[
  RowBox[{
  "subject", " ", "to", " ", "force", " ", "balance", " ", "on", " ", "every",
    " ", "unanchored", " ", 
   RowBox[{"node", "."}]}], TraditionalForm]], "DisplayMath",
 CellID->287545174],

Cell[TextData[{
 "The absolute values ",
 Cell[BoxData[
  FormBox[
   RowBox[{"|", 
    RowBox[{"force", "*", "link_length"}], "|"}], TraditionalForm]], 
  "InlineMath"],
 " in the objective function can be replaced by breaking down ",
 Cell[BoxData["force"], "InlineFormula"],
 " into a combination of compression and tension forces, with each \
non-negative. Thus assume ",
 Cell[BoxData[
  ButtonBox["E",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/E"]], "InlineFormula"],
 " is the set of links, ",
 Cell[BoxData["V"], "InlineFormula"],
 " the set of nodes, ",
 Cell[BoxData[
  SubscriptBox["l", "ij"]], "InlineFormula"],
 " the length of link between nodes ",
 Cell[BoxData["i"], "InlineFormula"],
 " and ",
 Cell[BoxData["j"], "InlineFormula"],
 ", ",
 Cell[BoxData[
  SubscriptBox["c", "ij"]], "InlineFormula"],
 "and ",
 Cell[BoxData[
  SubscriptBox["t", "ij"]], "InlineFormula"],
 " the compression and tension forces on the link; then the above model can \
be converted to a linear programming problem"
}], "Text",
 CellID->535486842],

Cell[BoxData[
 FormBox[
  RowBox[{"Minimize", " ", 
   RowBox[{
    SubscriptBox["\[Sum]", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"i", ",", "j"}], "}"}], "\[Element]", " ", "E"}]], " ", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       SubscriptBox["c", "ij"], " ", "+", " ", 
       SubscriptBox["t", "ij"]}], ")"}], "  ", 
     SubscriptBox["l", "ij"]}]}]}], TraditionalForm]], "DisplayMath",
 CellID->249283724],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{"subject", " ", "to", " ", 
     RowBox[{
      SubscriptBox["\[Sum]", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"i", ",", "k"}], "}"}], "\[Element]", " ", "E"}]], " ", 
      RowBox[{"(", " ", 
       RowBox[{
        SubscriptBox["t", "ik"], " ", "-", " ", 
        SubscriptBox["c", "ik"]}], ")"}]}]}], " ", "=", " ", 
    SubscriptBox["load", "i"]}], ",", " ", 
   SubscriptBox["t", "ij"], ",", " ", 
   RowBox[{
    SubscriptBox["c", "ij"], " ", "\[GreaterEqual]", " ", "0"}], ",", " ", 
   RowBox[{
    RowBox[{"for", " ", "all", " ", "i"}], "\[Element]", 
    RowBox[{"V", " ", "and", " ", 
     RowBox[{"{", 
      RowBox[{"i", ",", "j"}], "}"}]}], "\[Element]", " ", 
    RowBox[{"E", "."}]}]}], TraditionalForm]], "DisplayMath",
 CellID->206317008],

Cell[TextData[{
 "The following sets up the model, solves it, and plots the result; it is \
based on an AMPL model ",
 ButtonBox["[2]",
  BaseStyle->"Link",
  ButtonData->
   "paclet:tutorial/ConstrainedOptimizationLinearProgramming#1592161131"],
 "."
}], "MathCaption",
 CellID->361102706],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"OptimalAnchorDesign", "[", 
     RowBox[{"X_", ",", "Y_", ",", "ANCHORS_", ",", "forcepoints_", ",", 
      RowBox[{"dist_:", " ", "3"}]}], "]"}], ":=", 
    RowBox[{"Module", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
       "a", ",", "c", ",", "ldist", ",", "p", ",", "NODES", ",", "UNANCHORED",
         ",", "setx", ",", "sety", ",", "length", ",", "xload", ",", "yload", 
        ",", "nnodes", ",", "getarcs", ",", "ARCS", ",", "comp", ",", "comps",
         ",", "tensions", ",", "const1", ",", "const2", ",", "lengths", ",", 
        "volume", ",", "inedges", ",", "outedges", ",", "nodebalance", ",", 
        "const3", ",", "vars", ",", "totalf", ",", "maxf", ",", "res", ",", 
        "tens", ",", "setInOutEdges", ",", "consts", ",", "sol", ",", "f", 
        ",", "xii", ",", "yii", ",", "xjj", ",", "yjj", ",", "t", ",", "rhs", 
        ",", "ma", ",", "obj", ",", "m", ",", "n"}], "}"}], ",", 
      "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"Clear", "[", 
        RowBox[{"comp", ",", "tensions", ",", "tens", ",", "vars"}], "]"}], 
       ";", "\[IndentingNewLine]", "\[IndentingNewLine]", 
       RowBox[{"(*", " ", 
        RowBox[{
        "need", " ", "at", " ", "least", " ", "2", " ", "nchor", " ", 
         "points"}], " ", "*)"}], "\[IndentingNewLine]", 
       RowBox[{"If", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"Length", "[", 
           RowBox[{"Union", "[", "ANCHORS", "]"}], "]"}], "\[Equal]", "1"}], 
         ",", 
         RowBox[{"Return", "[", 
          RowBox[{"{", "}"}], "]"}]}], "]"}], ";", "\[IndentingNewLine]", 
       "\[IndentingNewLine]", 
       RowBox[{"(*", " ", 
        RowBox[{"A", " ", "lattice", " ", "of", " ", "Nodes"}], " ", "*)"}], 
       "\[IndentingNewLine]", 
       RowBox[{"NODES", "=", 
        RowBox[{"Partition", "[", 
         RowBox[{
          RowBox[{"Flatten", "[", 
           RowBox[{"Outer", "[", 
            RowBox[{"List", ",", "X", ",", "Y"}], "]"}], "]"}], ",", "2"}], 
         "]"}]}], ";", "\[IndentingNewLine]", "\[IndentingNewLine]", 
       RowBox[{"(*", " ", 
        RowBox[{
        "these", " ", "are", " ", "the", " ", "nodes", " ", "near", " ", 
         "the", " ", "wall", " ", "that", " ", "will", " ", "be", " ", 
         "anchored"}], " ", "*)"}], "\[IndentingNewLine]", 
       RowBox[{"UNANCHORED", "=", 
        RowBox[{"Complement", "[", 
         RowBox[{"NODES", ",", "ANCHORS"}], "]"}]}], ";", 
       "\[IndentingNewLine]", 
       RowBox[{"(*", " ", 
        RowBox[{
        "the", " ", "many", " ", "linked", " ", "exist", " ", "in", " ", 
         "the", " ", "structure", " ", "that", " ", "we", " ", "try", " ", 
         "to", " ", "optimize", " ", "away"}], " ", "*)"}], 
       "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"setx", "[", 
         RowBox[{"{", 
          RowBox[{"x_", ",", "y_"}], "}"}], "]"}], ":=", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"xload", "[", 
           RowBox[{"x", ",", "y"}], "]"}], "=", "0"}], ")"}]}], ";", 
       "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"sety", "[", 
         RowBox[{"{", 
          RowBox[{"x_", ",", "y_"}], "}"}], "]"}], ":=", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"yload", "[", 
           RowBox[{"x", ",", "y"}], "]"}], "=", "0"}], ")"}]}], ";", 
       "\[IndentingNewLine]", 
       RowBox[{"Map", "[", 
        RowBox[{"setx", ",", "UNANCHORED"}], "]"}], ";", 
       "\[IndentingNewLine]", 
       RowBox[{"Map", "[", 
        RowBox[{"sety", ",", "UNANCHORED"}], "]"}], ";", 
       "\[IndentingNewLine]", 
       RowBox[{"Map", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"(", 
           RowBox[{
            RowBox[{"yload", "[", 
             RowBox[{
              RowBox[{"#", "[", 
               RowBox[{"[", "1", "]"}], "]"}], ",", 
              RowBox[{"#", "[", 
               RowBox[{"[", "2", "]"}], "]"}]}], "]"}], "=", 
            RowBox[{"-", "1"}]}], ")"}], "&"}], ",", "forcepoints"}], "]"}], 
       ";", "\[IndentingNewLine]", "\[IndentingNewLine]", 
       RowBox[{"(*", " ", 
        RowBox[{
         RowBox[{
         "get", " ", "the", " ", "edges", " ", "that", " ", "link", " ", 
          "nodes", " ", "with", " ", "neighbors", " ", "of", " ", 
          "distance"}], " ", "\[LessEqual]", " ", "3"}], " ", "*)"}], 
       "\[IndentingNewLine]", 
       RowBox[{"nnodes", "=", 
        RowBox[{"Length", "[", "NODES", "]"}]}], ";", "\[IndentingNewLine]", 
       RowBox[{"getarcs", "=", 
        RowBox[{"Compile", "[", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"{", 
            RowBox[{"NODES", ",", "_Integer", ",", "2"}], "}"}], "}"}], ",", 
          RowBox[{"Module", "[", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{
             "xi", ",", "yi", ",", "xj", ",", "yj", ",", "i", ",", "j", ",", 
              RowBox[{"nn", "=", "0"}], ",", "NN"}], "}"}], ",", 
            "\[IndentingNewLine]", 
            RowBox[{"(*", " ", 
             RowBox[{
             "we", " ", "use", " ", "a", " ", "two", " ", "sweep", " ", 
              "strategy", " ", "as", " ", "a", " ", "nexted", " ", "list", 
              " ", "would", " ", "not", " ", "be", " ", "allowed", " ", "to", 
              " ", "compile", " ", "by", " ", "Compile"}], " ", "*)"}], 
            "\[IndentingNewLine]", 
            RowBox[{
             RowBox[{"Do", "[", 
              RowBox[{
               RowBox[{"Do", "[", 
                RowBox[{
                 RowBox[{
                  RowBox[{
                   RowBox[{"{", 
                    RowBox[{"xi", ",", "yi"}], "}"}], "=", 
                   RowBox[{"NODES", "[", 
                    RowBox[{"[", "i", "]"}], "]"}]}], ";", 
                  "\[IndentingNewLine]", 
                  RowBox[{
                   RowBox[{"{", 
                    RowBox[{"xj", ",", "yj"}], "}"}], "=", 
                   RowBox[{"NODES", "[", 
                    RowBox[{"[", "j", "]"}], "]"}]}], ";", 
                  "\[IndentingNewLine]", 
                  RowBox[{"If", "[", 
                   RowBox[{
                    RowBox[{
                    RowBox[{
                    RowBox[{"Abs", "[", 
                    RowBox[{"xj", "-", "xi"}], "]"}], "\[LessEqual]", 
                    "dist"}], "&&", 
                    RowBox[{
                    RowBox[{"Abs", "[", 
                    RowBox[{"yj", "-", "yi"}], "]"}], "\[LessEqual]", 
                    "dist"}], "&&", 
                    RowBox[{
                    RowBox[{"Abs", "[", 
                    RowBox[{"GCD", "[", 
                    RowBox[{
                    RowBox[{"xj", "-", "xi"}], ",", 
                    RowBox[{"yj", "-", "yi"}]}], "]"}], "]"}], "\[Equal]", 
                    "1"}], "&&", 
                    RowBox[{"(", 
                    RowBox[{
                    RowBox[{"xi", ">", "xj"}], "||", 
                    RowBox[{"(", 
                    RowBox[{
                    RowBox[{"xi", "\[Equal]", "xj"}], "&&", 
                    RowBox[{"yi", ">", "yj"}]}], ")"}]}], ")"}]}], ",", 
                    RowBox[{"nn", "++"}]}], "]"}]}], ",", 
                 RowBox[{"{", 
                  RowBox[{"j", ",", 
                   RowBox[{"Length", "[", "NODES", "]"}]}], "}"}]}], "]"}], 
               ",", 
               RowBox[{"{", 
                RowBox[{"i", ",", 
                 RowBox[{"Length", "[", "NODES", "]"}]}], "}"}]}], "]"}], ";",
              "\[IndentingNewLine]", 
             RowBox[{"NN", "=", 
              RowBox[{"Table", "[", 
               RowBox[{
                RowBox[{"{", 
                 RowBox[{"1", ",", "1", ",", "1", ",", "1"}], "}"}], ",", 
                RowBox[{"{", "nn", "}"}]}], "]"}]}], ";", 
             "\[IndentingNewLine]", 
             RowBox[{"nn", "=", "1"}], ";", "\[IndentingNewLine]", 
             RowBox[{"Do", "[", 
              RowBox[{
               RowBox[{"Do", "[", 
                RowBox[{
                 RowBox[{
                  RowBox[{
                   RowBox[{"{", 
                    RowBox[{"xi", ",", "yi"}], "}"}], "=", 
                   RowBox[{"NODES", "[", 
                    RowBox[{"[", "i", "]"}], "]"}]}], ";", 
                  "\[IndentingNewLine]", 
                  RowBox[{
                   RowBox[{"{", 
                    RowBox[{"xj", ",", "yj"}], "}"}], "=", 
                   RowBox[{"NODES", "[", 
                    RowBox[{"[", "j", "]"}], "]"}]}], ";", 
                  "\[IndentingNewLine]", 
                  RowBox[{"If", "[", 
                   RowBox[{
                    RowBox[{
                    RowBox[{
                    RowBox[{"Abs", "[", 
                    RowBox[{"xj", "-", "xi"}], "]"}], "\[LessEqual]", 
                    "dist"}], "&&", 
                    RowBox[{
                    RowBox[{"Abs", "[", 
                    RowBox[{"yj", "-", "yi"}], "]"}], "\[LessEqual]", 
                    "dist"}], "&&", 
                    RowBox[{
                    RowBox[{"Abs", "[", 
                    RowBox[{"GCD", "[", 
                    RowBox[{
                    RowBox[{"xj", "-", "xi"}], ",", 
                    RowBox[{"yj", "-", "yi"}]}], "]"}], "]"}], "\[Equal]", 
                    "1"}], "&&", 
                    RowBox[{"(", 
                    RowBox[{
                    RowBox[{"xi", ">", "xj"}], "||", 
                    RowBox[{"(", 
                    RowBox[{
                    RowBox[{"xi", "\[Equal]", "xj"}], "&&", 
                    RowBox[{"yi", ">", "yj"}]}], ")"}]}], ")"}]}], ",", 
                    RowBox[{
                    RowBox[{"NN", "[", 
                    RowBox[{"[", 
                    RowBox[{"nn", "++"}], "]"}], "]"}], "=", 
                    RowBox[{"{", 
                    RowBox[{"xi", ",", "yi", ",", "xj", ",", "yj"}], 
                    "}"}]}]}], "]"}]}], ",", 
                 RowBox[{"{", 
                  RowBox[{"j", ",", 
                   RowBox[{"Length", "[", "NODES", "]"}]}], "}"}]}], "]"}], 
               ",", 
               RowBox[{"{", 
                RowBox[{"i", ",", 
                 RowBox[{"Length", "[", "NODES", "]"}]}], "}"}]}], "]"}], ";",
              "\[IndentingNewLine]", "NN"}]}], "]"}]}], "]"}]}], ";", 
       "\[IndentingNewLine]", 
       RowBox[{"ARCS", "=", 
        RowBox[{"Partition", "[", 
         RowBox[{
          RowBox[{"Flatten", "[", 
           RowBox[{"getarcs", "[", "NODES", "]"}], "]"}], ",", 
          RowBox[{"{", "4", "}"}]}], "]"}]}], ";", "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"length", "[", 
         RowBox[{"{", 
          RowBox[{"xi_", ",", "yi_", ",", "xj_", ",", "yj_"}], "}"}], "]"}], ":=",
         
        RowBox[{
         RowBox[{"Sqrt", "[", 
          RowBox[{
           RowBox[{
            RowBox[{"(", 
             RowBox[{"xj", "-", "xi"}], ")"}], "^", "2"}], "+", 
           RowBox[{
            RowBox[{"(", 
             RowBox[{"yj", "-", "yi"}], ")"}], "^", "2"}]}], "]"}], "//", 
         "N"}]}], ";", "\[IndentingNewLine]", 
       RowBox[{"(*", " ", 
        RowBox[{"the", " ", 
         RowBox[{"variables", ":", " ", 
          RowBox[{
          "compression", " ", "and", " ", "tension", " ", "forces"}]}]}], " ",
         "*)"}], "\[IndentingNewLine]", 
       RowBox[{"comps", "=", 
        RowBox[{"Map", "[", 
         RowBox[{
          RowBox[{
           RowBox[{"(", 
            RowBox[{"comp", "@@", "#"}], ")"}], "&"}], ",", "ARCS"}], "]"}]}],
        ";", "\[IndentingNewLine]", 
       RowBox[{"tensions", "=", 
        RowBox[{"Map", "[", 
         RowBox[{
          RowBox[{
           RowBox[{"(", 
            RowBox[{"tens", "@@", "#"}], ")"}], "&"}], ",", "ARCS"}], "]"}]}],
        ";", "\[IndentingNewLine]", 
       RowBox[{"const1", "=", 
        RowBox[{"Thread", "[", 
         RowBox[{"Greater", "[", 
          RowBox[{"comps", ",", "0"}], "]"}], "]"}]}], ";", 
       "\[IndentingNewLine]", 
       RowBox[{"const2", "=", 
        RowBox[{"Thread", "[", 
         RowBox[{"Greater", "[", 
          RowBox[{"tensions", ",", "0"}], "]"}], "]"}]}], ";", 
       "\[IndentingNewLine]", 
       RowBox[{"lengths", "=", 
        RowBox[{
         RowBox[{"Map", "[", 
          RowBox[{
           RowBox[{
            RowBox[{"(", 
             RowBox[{"length", "[", "#", "]"}], ")"}], "&"}], ",", "ARCS"}], 
          "]"}], "//", "N"}]}], ";", "\[IndentingNewLine]", 
       "\[IndentingNewLine]", 
       StyleBox[
        RowBox[{"(*", " ", 
         RowBox[{"objective", " ", "function"}], " ", "*)"}],
        FontColor->RGBColor[0, 0, 1]], 
       StyleBox["\[IndentingNewLine]",
        FontColor->RGBColor[0, 0, 1]], 
       RowBox[{"volume", "=", 
        RowBox[{"lengths", ".", 
         RowBox[{"(", 
          RowBox[{"comps", "+", "tensions"}], ")"}]}]}], ";", 
       "\[IndentingNewLine]", "\[IndentingNewLine]", 
       RowBox[{"Map", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"(", 
           RowBox[{
            RowBox[{"inedges", "[", "#", "]"}], "=", "False"}], ")"}], "&"}], 
         ",", "NODES"}], "]"}], ";", "\[IndentingNewLine]", 
       RowBox[{"Map", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"(", 
           RowBox[{
            RowBox[{"outedges", "[", "#", "]"}], "=", "False"}], ")"}], "&"}],
          ",", "NODES"}], "]"}], ";", "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"setInOutEdges", "[", 
         RowBox[{"{", 
          RowBox[{"xi_", ",", "yi_", ",", "xj_", ",", "yj_"}], "}"}], "]"}], ":=",
         
        RowBox[{"Module", "[", 
         RowBox[{
          RowBox[{"{", "}"}], ",", 
          RowBox[{
           RowBox[{"If", "[", 
            RowBox[{
             RowBox[{
              RowBox[{"outedges", "[", 
               RowBox[{"{", 
                RowBox[{"xj", ",", "yj"}], "}"}], "]"}], "===", "False"}], 
             ",", 
             RowBox[{
              RowBox[{"outedges", "[", 
               RowBox[{"{", 
                RowBox[{"xj", ",", "yj"}], "}"}], "]"}], "=", 
              RowBox[{"{", 
               RowBox[{"xi", ",", "yi", ",", "xj", ",", "yj"}], "}"}]}], ",", 
             
             RowBox[{
              RowBox[{"outedges", "[", 
               RowBox[{"{", 
                RowBox[{"xj", ",", "yj"}], "}"}], "]"}], "=", 
              RowBox[{"{", 
               RowBox[{
                RowBox[{"outedges", "[", 
                 RowBox[{"{", 
                  RowBox[{"xj", ",", "yj"}], "}"}], "]"}], ",", 
                RowBox[{"{", 
                 RowBox[{"xi", ",", "yi", ",", "xj", ",", "yj"}], "}"}]}], 
               "}"}]}]}], "]"}], ";", "\[IndentingNewLine]", 
           RowBox[{"If", "[", 
            RowBox[{
             RowBox[{
              RowBox[{"inedges", "[", 
               RowBox[{"{", 
                RowBox[{"xi", ",", "yi"}], "}"}], "]"}], "===", "False"}], 
             ",", 
             RowBox[{
              RowBox[{"inedges", "[", 
               RowBox[{"{", 
                RowBox[{"xi", ",", "yi"}], "}"}], "]"}], "=", 
              RowBox[{"{", 
               RowBox[{"xi", ",", "yi", ",", "xj", ",", "yj"}], "}"}]}], ",", 
             
             RowBox[{
              RowBox[{"inedges", "[", 
               RowBox[{"{", 
                RowBox[{"xi", ",", "yi"}], "}"}], "]"}], "=", 
              RowBox[{"{", 
               RowBox[{
                RowBox[{"inedges", "[", 
                 RowBox[{"{", 
                  RowBox[{"xi", ",", "yi"}], "}"}], "]"}], ",", 
                RowBox[{"{", 
                 RowBox[{"xi", ",", "yi", ",", "xj", ",", "yj"}], "}"}]}], 
               "}"}]}]}], "]"}], ";"}]}], "]"}]}], ";", "\[IndentingNewLine]",
        
       RowBox[{"Map", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"(", 
           RowBox[{"setInOutEdges", "[", "#", "]"}], ")"}], "&"}], ",", 
         "ARCS"}], "]"}], ";", "\[IndentingNewLine]", 
       RowBox[{"Map", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"(", 
           RowBox[{
            RowBox[{"inedges", "[", "#", "]"}], "=", 
            RowBox[{"Partition", "[", 
             RowBox[{
              RowBox[{"Flatten", "[", 
               RowBox[{"{", 
                RowBox[{"inedges", "[", "#", "]"}], "}"}], "]"}], ",", 
              RowBox[{"{", "4", "}"}]}], "]"}]}], ")"}], "&"}], ",", 
         "NODES"}], "]"}], ";", "\[IndentingNewLine]", 
       RowBox[{"Map", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"(", 
           RowBox[{
            RowBox[{"outedges", "[", "#", "]"}], "=", 
            RowBox[{"Partition", "[", 
             RowBox[{
              RowBox[{"Flatten", "[", 
               RowBox[{"{", 
                RowBox[{"outedges", "[", "#", "]"}], "}"}], "]"}], ",", 
              RowBox[{"{", "4", "}"}]}], "]"}]}], ")"}], "&"}], ",", 
         "NODES"}], "]"}], ";", "\[IndentingNewLine]", "\[IndentingNewLine]", 
       
       RowBox[{
        RowBox[{"nodebalance", "[", 
         RowBox[{"{", 
          RowBox[{"x_", ",", "y_"}], "}"}], "]"}], ":=", 
        RowBox[{"Module", "[", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"Inedges", ",", "Outedges", ",", "xforce", ",", "yforce"}],
            "}"}], ",", 
          RowBox[{
           RowBox[{"Inedges", "=", 
            RowBox[{"inedges", "[", 
             RowBox[{"{", 
              RowBox[{"x", ",", "y"}], "}"}], "]"}]}], ";", 
           "\[IndentingNewLine]", 
           RowBox[{"Outedges", "=", 
            RowBox[{"outedges", "[", 
             RowBox[{"{", 
              RowBox[{"x", ",", "y"}], "}"}], "]"}]}], ";", 
           "\[IndentingNewLine]", 
           RowBox[{
            RowBox[{"xforce", "[", 
             RowBox[{"{", 
              RowBox[{"xi_", ",", "yi_", ",", "xj_", ",", "yj_"}], "}"}], 
             "]"}], ":=", 
            RowBox[{
             RowBox[{"(", 
              RowBox[{
               RowBox[{"(", 
                RowBox[{"xj", "-", "xi"}], ")"}], "/", 
               RowBox[{"length", "[", 
                RowBox[{"{", 
                 RowBox[{"xi", ",", "yi", ",", "xj", ",", "yj"}], "}"}], 
                "]"}]}], ")"}], "*", 
             RowBox[{"(", 
              RowBox[{
               RowBox[{"comp", "[", 
                RowBox[{"xi", ",", "yi", ",", "xj", ",", "yj"}], "]"}], "-", 
               RowBox[{"tens", "[", 
                RowBox[{"xi", ",", "yi", ",", "xj", ",", "yj"}], "]"}]}], 
              ")"}]}]}], ";", "\[IndentingNewLine]", 
           RowBox[{
            RowBox[{"yforce", "[", 
             RowBox[{"{", 
              RowBox[{"xi_", ",", "yi_", ",", "xj_", ",", "yj_"}], "}"}], 
             "]"}], ":=", 
            RowBox[{
             RowBox[{"(", 
              RowBox[{
               RowBox[{"(", 
                RowBox[{"yj", "-", "yi"}], ")"}], "/", 
               RowBox[{"length", "[", 
                RowBox[{"{", 
                 RowBox[{"xi", ",", "yi", ",", "xj", ",", "yj"}], "}"}], 
                "]"}]}], ")"}], "*", 
             RowBox[{"(", 
              RowBox[{
               RowBox[{"comp", "[", 
                RowBox[{"xi", ",", "yi", ",", "xj", ",", "yj"}], "]"}], "-", 
               RowBox[{"tens", "[", 
                RowBox[{"xi", ",", "yi", ",", "xj", ",", "yj"}], "]"}]}], 
              ")"}]}]}], ";", "\[IndentingNewLine]", 
           StyleBox[
            RowBox[{"(*", " ", "constraints", " ", "*)"}],
            FontColor->RGBColor[0, 0, 1]], 
           StyleBox["\[IndentingNewLine]",
            FontColor->RGBColor[0, 0, 1]], 
           RowBox[{"{", 
            RowBox[{
             RowBox[{
              RowBox[{
               RowBox[{"Total", "[", 
                RowBox[{"Map", "[", 
                 RowBox[{"xforce", ",", "Inedges"}], "]"}], "]"}], "-", 
               RowBox[{"Total", "[", 
                RowBox[{"Map", "[", 
                 RowBox[{"xforce", ",", "Outedges"}], "]"}], "]"}]}], 
              "\[Equal]", 
              RowBox[{"xload", "[", 
               RowBox[{"x", ",", "y"}], "]"}]}], ",", 
             RowBox[{
              RowBox[{
               RowBox[{"Total", "[", 
                RowBox[{"Map", "[", 
                 RowBox[{"yforce", ",", "Inedges"}], "]"}], "]"}], "-", 
               RowBox[{"Total", "[", 
                RowBox[{"Map", "[", 
                 RowBox[{"yforce", ",", "Outedges"}], "]"}], "]"}]}], 
              "\[Equal]", 
              RowBox[{"yload", "[", 
               RowBox[{"x", ",", "y"}], "]"}]}]}], "}"}]}]}], 
         "\[IndentingNewLine]", "]"}]}], ";", 
       StyleBox["\[IndentingNewLine]",
        FontColor->RGBColor[0.0784314, 0.666667, 1]], 
       RowBox[{"const3", "=", 
        RowBox[{"Flatten", "[", 
         RowBox[{"Map", "[", 
          RowBox[{
           RowBox[{
            RowBox[{"nodebalance", "[", "#", "]"}], "&"}], ",", 
           "UNANCHORED"}], "]"}], "]"}]}], ";", "\[IndentingNewLine]", 
       RowBox[{"(*", " ", 
        RowBox[{
         RowBox[{
         "assemble", " ", "the", " ", "variables", " ", "and", " ", 
          "constraints"}], ",", " ", 
         RowBox[{"and", " ", "solve"}]}], " ", "*)"}], "\[IndentingNewLine]", 
       "\[IndentingNewLine]", "\[IndentingNewLine]", 
       RowBox[{"vars", "=", 
        RowBox[{"Union", "[", 
         RowBox[{"Flatten", "[", 
          RowBox[{"{", 
           RowBox[{"comps", ",", "tensions"}], "}"}], "]"}], "]"}]}], ";", 
       "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"rhs", ",", "ma"}], "}"}], "=", 
        RowBox[{"CoefficientArrays", "[", 
         RowBox[{"const3", ",", "vars"}], "]"}]}], ";", "\[IndentingNewLine]",
        
       RowBox[{"obj", "=", 
        RowBox[{
         RowBox[{"CoefficientArrays", "[", 
          RowBox[{"volume", ",", "vars"}], "]"}], "[", 
         RowBox[{"[", "2", "]"}], "]"}]}], ";", "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"m", ",", "n"}], "}"}], "=", 
        RowBox[{"Dimensions", "[", "ma", "]"}]}], ";", "\[IndentingNewLine]", 
       
       RowBox[{"Print", "[", 
        RowBox[{
        "\"\<Number of variables = \>\"", ",", "n", ",", 
         "\"\< number of constraints = \>\"", ",", "m"}], "]"}], ";", 
       "\[IndentingNewLine]", "\[IndentingNewLine]", 
       StyleBox[
        RowBox[{"(*", " ", "solve", " ", "*)"}],
        FontColor->RGBColor[0, 0, 1]], "\[IndentingNewLine]", 
       RowBox[{"t", "=", 
        RowBox[{"Timing", "[", 
         RowBox[{
          RowBox[{"sol", "=", 
           RowBox[{"LinearProgramming", "[", 
            RowBox[{"obj", ",", " ", "ma", ",", " ", 
             RowBox[{"Transpose", "[", 
              RowBox[{"{", 
               RowBox[{
                RowBox[{"-", "rhs"}], ",", 
                RowBox[{"Table", "[", 
                 RowBox[{"0", ",", 
                  RowBox[{"{", "m", "}"}]}], "]"}]}], "}"}], "]"}], ",", 
             RowBox[{"Table", "[", 
              RowBox[{
               RowBox[{"{", 
                RowBox[{"0", ",", "Infinity"}], "}"}], ",", 
               RowBox[{"{", "n", "}"}]}], "]"}]}], "]"}]}], ";"}], "]"}]}], 
       ";", "\[IndentingNewLine]", 
       RowBox[{"Print", "[", 
        RowBox[{"\"\<CPU time = \>\"", ",", 
         RowBox[{"t", "[", 
          RowBox[{"[", "1", "]"}], "]"}], ",", "\"\< Seconds\>\""}], "]"}], 
       ";", "\[IndentingNewLine]", 
       RowBox[{"Map", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"Set", "@@", "#"}], "&"}], ",", 
         RowBox[{"Transpose", "[", 
          RowBox[{"{", 
           RowBox[{"vars", ",", "sol"}], "}"}], "]"}]}], "]"}], ";", 
       "\[IndentingNewLine]", "\[IndentingNewLine]", 
       RowBox[{"(*", " ", 
        RowBox[{
         RowBox[{
         "Now", " ", "add", " ", "up", " ", "the", " ", "total", " ", "force",
           " ", "on", " ", "all", " ", "links"}], ",", " ", 
         RowBox[{
         "and", " ", "scale", " ", "them", " ", "to", " ", "be", " ", 
          "between", " ", "0", " ", "and", " ", "1."}]}], " ", "*)"}], 
       "\[IndentingNewLine]", 
       RowBox[{"maxf", "=", 
        RowBox[{"Max", "[", 
         RowBox[{"comps", "+", "tensions"}], "]"}]}], ";", 
       "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"Evaluate", "[", 
         RowBox[{"Map", "[", 
          RowBox[{
           RowBox[{
            RowBox[{"totalf", "[", "#", "]"}], "&"}], ",", "ARCS"}], "]"}], 
         "]"}], "=", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{"comps", "+", "tensions"}], ")"}], "/", "maxf"}]}], ";", 
       "\[IndentingNewLine]", "\[IndentingNewLine]", 
       RowBox[{"(*", " ", 
        RowBox[{
         RowBox[{
         "Now", " ", "we", " ", "plot", " ", "the", " ", "links", " ", "that",
           " ", "has", " ", "a", " ", "force", " ", "at", " ", "least", " ", 
          "0.001", " ", "and", " ", "get", " ", "the", " ", "optimal", " ", 
          "design", " ", "of", " ", "the", " ", 
          RowBox[{"anchor", ".", " ", "We"}], " ", "color", " ", "code", " ", 
          "the", " ", "drawing", " ", "so", " ", "that", " ", "red", " ", 
          "means", " ", "a", " ", "large", " ", "force", " ", "and", " ", 
          "blue", " ", "a", " ", "small", " ", 
          RowBox[{"one", ".", " ", "Also"}]}], ",", " ", 
         RowBox[{
         "links", " ", "with", " ", "large", " ", "forces", " ", "are", " ", 
          "drawn", " ", "thinker", " ", "than", " ", "those", " ", "with", 
          " ", "small", " ", 
          RowBox[{"forces", "."}]}]}], " ", "*)"}], "\[IndentingNewLine]", 
       "\[IndentingNewLine]", 
       RowBox[{"res", "=", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"EdgeForm", "[", "Black", "]"}], ",", "White", ",", 
          RowBox[{"Polygon", "[", 
           RowBox[{"{", 
            RowBox[{
             RowBox[{"{", 
              RowBox[{"0", ",", "0"}], "}"}], ",", 
             RowBox[{"{", 
              RowBox[{"0", ",", 
               RowBox[{"Length", "[", "Y", "]"}]}], "}"}], ",", 
             RowBox[{"{", 
              RowBox[{"1", ",", 
               RowBox[{"Length", "[", "Y", "]"}]}], "}"}], ",", 
             RowBox[{"{", 
              RowBox[{"1", ",", "0"}], "}"}]}], "}"}], "]"}], ",", 
          RowBox[{"Map", "[", 
           RowBox[{
            RowBox[{
             RowBox[{"(", 
              RowBox[{
               RowBox[{
                RowBox[{"{", 
                 RowBox[{"xii", ",", "yii", ",", "xjj", ",", "yjj"}], "}"}], 
                "=", "#"}], ";", 
               RowBox[{"f", "=", 
                RowBox[{"totalf", "[", 
                 RowBox[{"{", 
                  RowBox[{"xii", ",", "yii", ",", "xjj", ",", "yjj"}], "}"}], 
                 "]"}]}], ";", "\[IndentingNewLine]", 
               RowBox[{"If", "[", 
                RowBox[{
                 RowBox[{"f", ">", "0.001"}], ",", 
                 RowBox[{"{", 
                  RowBox[{
                   RowBox[{"Hue", "[", 
                    RowBox[{".7", "*", 
                    RowBox[{"(", 
                    RowBox[{"1", "-", "f"}], ")"}]}], "]"}], ",", 
                   RowBox[{"Thickness", "[", 
                    RowBox[{".02", " ", 
                    RowBox[{"Sqrt", "[", "f", "]"}]}], "]"}], ",", 
                   RowBox[{"Line", "[", 
                    RowBox[{"{", 
                    RowBox[{
                    RowBox[{"{", 
                    RowBox[{"xii", ",", "yii"}], "}"}], ",", 
                    RowBox[{"{", 
                    RowBox[{"xjj", ",", "yjj"}], "}"}]}], "}"}], "]"}]}], 
                  "}"}], ",", 
                 RowBox[{"{", "}"}]}], "]"}]}], ")"}], "&"}], ",", "ARCS"}], 
           "]"}], ",", 
          RowBox[{"GrayLevel", "[", ".5", "]"}], ",", 
          RowBox[{"PointSize", "[", "0.04", "]"}], ",", 
          RowBox[{"{", 
           RowBox[{"Black", ",", 
            RowBox[{"Map", "[", 
             RowBox[{
              RowBox[{
               RowBox[{"{", 
                RowBox[{"Arrow", "[", 
                 RowBox[{"{", 
                  RowBox[{"#", ",", 
                   RowBox[{"#", "+", 
                    RowBox[{"{", 
                    RowBox[{"0", ",", 
                    RowBox[{"-", "4"}]}], "}"}]}]}], "}"}], "]"}], "}"}], 
               "&"}], ",", "forcepoints"}], "]"}]}], "}"}], ",", 
          RowBox[{"Map", "[", 
           RowBox[{"Point", ",", "ANCHORS"}], "]"}]}], "}"}]}], ";", 
       "\[IndentingNewLine]", 
       RowBox[{"Graphics", "[", "res", "]"}]}]}], "\[IndentingNewLine]", 
     "]"}]}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->197698860],

Cell["\<\
This solves the problem by placing 30 nodes in the horizontal and vertical \
directions.\
\>", "MathCaption",
 CellID->185115299],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"m", "=", "30"}], ";"}], 
  RowBox[{"(*", " ", 
   RowBox[{"y", " ", 
    RowBox[{"direction", "."}]}], " ", "*)"}]}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"n", "=", "30"}], ";"}], 
  RowBox[{"(*", " ", 
   RowBox[{"x", " ", 
    RowBox[{"direction", "."}]}], " ", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"X", "=", 
   RowBox[{"Table", "[", 
    RowBox[{"i", ",", 
     RowBox[{"{", 
      RowBox[{"i", ",", "0", ",", "n"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"Y", "=", 
   RowBox[{"Table", "[", 
    RowBox[{"i", ",", 
     RowBox[{"{", 
      RowBox[{"i", ",", "0", ",", "m"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"res", "=", 
  RowBox[{"OptimalAnchorDesign", "[", 
   RowBox[{"X", ",", "Y", ",", 
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"1", ",", "i"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"i", ",", 
        RowBox[{"Round", "[", 
         RowBox[{"m", "/", "3"}], "]"}], ",", 
        RowBox[{"Round", "[", 
         RowBox[{
          RowBox[{"m", "/", "3"}], "*", "2"}], "]"}]}], "}"}]}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"{", 
      RowBox[{"n", ",", 
       RowBox[{"m", "/", "2"}]}], "}"}], "}"}], ",", "3"}], "]"}]}]}], "Input",\

 CellLabel->"In[2]:=",
 CellID->259327580],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Number of variables = \"\>", "\[InvisibleSpace]", "27496", 
   "\[InvisibleSpace]", "\<\" number of constraints\:ffff = \"\>", 
   "\[InvisibleSpace]", "1900"}],
  SequenceForm[
  "Number of variables = ", 27496, " number of constraints\:ffff = ", 1900],
  Editable->False]], "Print",
 CellLabel->"During evaluation of In[2]:=",
 CellID->2092791593],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"CPU time = \"\>", "\[InvisibleSpace]", "4.812301`", 
   "\[InvisibleSpace]", "\<\" Seconds\"\>"}],
  SequenceForm["CPU time = ", 4.812301, " Seconds"],
  Editable->False]], "Print",
 CellLabel->"During evaluation of In[2]:=",
 CellID->475838129]
}, Open  ]],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 185},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->675201650]
}, Open  ]],

Cell["\<\
If, however, the anchor is fixed not on the wall, but on some points in \
space, notice how the results resemble the shape of some leaves. Perhaps the \
structure of leaves is optimized in the process of evolution.\
\>", "MathCaption",
 CellID->253723770],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"m", "=", "40"}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"must", " ", "be", " ", "even"}], 
   "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"n", "=", "40"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"X", "=", 
   RowBox[{"Table", "[", 
    RowBox[{"i", ",", 
     RowBox[{"{", 
      RowBox[{"i", ",", "0", ",", "n"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"Y", "=", 
   RowBox[{"Table", "[", 
    RowBox[{"i", ",", 
     RowBox[{"{", 
      RowBox[{"i", ",", "0", ",", "m"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"res", "=", 
  RowBox[{"OptimalAnchorDesign", "[", 
   RowBox[{"X", ",", "Y", ",", 
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"Round", "[", 
         RowBox[{"n", "/", "3"}], "]"}], ",", "i"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"i", ",", 
        RowBox[{
         RowBox[{"Round", "[", 
          RowBox[{"m", "/", "2"}], "]"}], "-", "1"}], ",", 
        RowBox[{
         RowBox[{"Round", "[", 
          RowBox[{"m", "/", "2"}], "]"}], "+", "1"}]}], "}"}]}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"{", 
      RowBox[{"n", ",", 
       RowBox[{"m", "/", "2"}]}], "}"}], "}"}], ",", "3"}], "]"}]}]}], "Input",\

 CellLabel->"In[7]:=",
 CellID->2126659739],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Number of variables = \"\>", "\[InvisibleSpace]", "49456", 
   "\[InvisibleSpace]", "\<\" number of constraints\:ffff = \"\>", 
   "\[InvisibleSpace]", "3356"}],
  SequenceForm[
  "Number of variables = ", 49456, " number of constraints\:ffff = ", 3356],
  Editable->False]], "Print",
 CellLabel->"During evaluation of In[7]:=",
 CellID->400303040],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"CPU time = \"\>", "\[InvisibleSpace]", "9.832615000000002`", 
   "\[InvisibleSpace]", "\<\" Seconds\"\>"}],
  SequenceForm["CPU time = ", 9.832615000000002, " Seconds"],
  Editable->False]], "Print",
 CellLabel->"During evaluation of In[7]:=",
 CellID->1732451358]
}, Open  ]],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 184},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[11]=",
 CellID->462811892]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Algorithms for Linear Programming", "Section",
 CellID->38991160],

Cell[CellGroupData[{

Cell["Simplex and Revised Simplex Algorithms", "Subsection",
 CellID->160402625],

Cell["\<\
The simplex and revised simplex algorithms solve linear programming problems \
by constructing a feasible solution at a vertex of the polytope defined by \
the constraints, and then moving along the edges of the polytope to vertices \
with successively smaller values of the objective function until the minimum \
is reached. \
\>", "Text",
 CellID->391192688],

Cell["\<\
Although the sparse implementation of simplex and revised algorithms are \
quite efficient in practice, and are guaranteed to find the global optimum, \
they have a poor worst-case behavior: it is possible to construct a linear \
programming problem for which the simplex or revised simplex method takes a \
number of steps exponential in the problem size.\
\>", "Text",
 CellID->788121671],

Cell[TextData[{
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " implements simplex and revised simplex algorithms using dense linear \
algebra. The unique feature of this implementation is that it is possible to \
solve exact/extended precision problems. Therefore these methods are more \
suitable for small-sized problems for which non-machine number results are \
needed. "
}], "Text",
 CellID->65762130],

Cell["\<\
This sets up a random linear programming problem with 20 constraints and 200 \
variables.\
\>", "MathCaption",
 CellID->425368342],

Cell[BoxData[{
 RowBox[{
  RowBox[{"SeedRandom", "[", "123", "]"}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"m", ",", "n"}], "}"}], "=", 
   RowBox[{"{", 
    RowBox[{"20", ",", "200"}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"c", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"RandomInteger", "[", 
      RowBox[{"{", 
       RowBox[{"1", ",", "10"}], "}"}], "]"}], ",", 
     RowBox[{"{", "n", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"A", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"RandomInteger", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "100"}], ",", "100"}], "}"}], "]"}], ",", 
     RowBox[{"{", "m", "}"}], ",", 
     RowBox[{"{", "n", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"b", " ", "=", " ", 
   RowBox[{"A", ".", 
    RowBox[{"Table", "[", 
     RowBox[{"1", ",", 
      RowBox[{"{", "n", "}"}]}], "]"}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"bounds", " ", "=", " ", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "10"}], ",", "10"}], "}"}], ",", 
     RowBox[{"{", "n", "}"}]}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[12]:=",
 CellID->13393687],

Cell["\<\
This solves the problem. Typically, for a linear programming problem with \
many more variables than constraints, the revised simplex algorithm is \
faster. On the other hand, if there are many more constraints than variables, \
the simplex algorithm is faster.\
\>", "MathCaption",
 CellID->298559827],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"t", "=", 
    RowBox[{"Timing", "[", 
     RowBox[{
      RowBox[{"x", "=", 
       RowBox[{"LinearProgramming", "[", 
        RowBox[{"c", ",", "A", ",", "b", ",", "bounds", ",", 
         RowBox[{"Method", "\[Rule]", "\"\<Simplex\>\""}]}], "]"}]}], ";"}], 
     "]"}]}], ";", " ", 
   RowBox[{"Print", "[", 
    RowBox[{"\"\<time = \>\"", ",", 
     RowBox[{"t", "[", 
      RowBox[{"[", "1", "]"}], "]"}], ",", "\"\< optimal value = \>\"", ",", 
     RowBox[{"c", ".", "x"}], ",", "\"\< or \>\"", ",", 
     RowBox[{"N", "[", 
      RowBox[{"c", ".", "x"}], "]"}]}], "]"}]}], 
  "\[IndentingNewLine]"}]], "Input",
 CellLabel->"In[25]:=",
 CellID->18667422],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"time = \"\>", "\[InvisibleSpace]", "14.740920999999998`", 
   "\[InvisibleSpace]", "\<\" optimal value = \"\>", "\[InvisibleSpace]", 
   RowBox[{"-", 
    FractionBox["1151274037058983869972777363", "105283309229356027027010"]}],
    "\[InvisibleSpace]", "\<\" or \"\>", "\[InvisibleSpace]", 
   RowBox[{"-", "10935.009979131388`"}]}],
  SequenceForm["time = ", 14.740920999999998`, " optimal value = ", 
   Rational[-1151274037058983869972777363, 105283309229356027027010], 
   " or ", -10935.009979131388`],
  Editable->False]], "Print",
 CellLabel->"During evaluation of In[25]:=",
 CellID->1628362047]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"t", "=", 
    RowBox[{"Timing", "[", 
     RowBox[{
      RowBox[{"x", "=", 
       RowBox[{"LinearProgramming", "[", 
        RowBox[{"c", ",", "A", ",", "b", ",", "bounds", ",", 
         RowBox[{"Method", "\[Rule]", "\"\<RevisedSimplex\>\""}]}], "]"}]}], 
      ";"}], "]"}]}], ";", " ", 
   RowBox[{"Print", "[", 
    RowBox[{"\"\<time = \>\"", ",", 
     RowBox[{"t", "[", 
      RowBox[{"[", "1", "]"}], "]"}], ",", "\"\< optimal value = \>\"", ",", 
     RowBox[{"c", ".", "x"}], ",", "\"\< or \>\"", ",", 
     RowBox[{"N", "[", 
      RowBox[{"c", ".", "x"}], "]"}]}], "]"}]}], 
  "\[IndentingNewLine]"}]], "Input",
 CellLabel->"In[26]:=",
 CellID->192654607],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"time = \"\>", "\[InvisibleSpace]", "6.344395999999998`", 
   "\[InvisibleSpace]", "\<\" optimal value = \"\>", "\[InvisibleSpace]", 
   RowBox[{"-", 
    FractionBox["1151274037058983869972777363", "105283309229356027027010"]}],
    "\[InvisibleSpace]", "\<\" or \"\>", "\[InvisibleSpace]", 
   RowBox[{"-", "10935.009979131388`"}]}],
  SequenceForm["time = ", 6.344395999999998, " optimal value = ", 
   Rational[-1151274037058983869972777363, 105283309229356027027010], 
   " or ", -10935.009979131388`],
  Editable->False]], "Print",
 CellLabel->"During evaluation of In[26]:=",
 CellID->1516079829]
}, Open  ]],

Cell["\<\
If only machine-number results are desired, then the problem should be \
converted to machine numbers, and the interior point algorithm should be \
used.\
\>", "MathCaption",
 CellID->113254445],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"t", "=", 
   RowBox[{"Timing", "[", 
    RowBox[{
     RowBox[{"x", "=", 
      RowBox[{"LinearProgramming", "[", 
       RowBox[{
        RowBox[{"N", "[", "c", "]"}], ",", 
        RowBox[{"N", "[", "A", "]"}], ",", 
        RowBox[{"N", "[", "b", "]"}], ",", 
        RowBox[{"N", "[", "bounds", "]"}], ",", 
        RowBox[{"Method", "\[Rule]", "\"\<InteriorPoint\>\""}]}], "]"}]}], 
     ";"}], "]"}]}], ";", " ", 
  RowBox[{"Print", "[", 
   RowBox[{
   "\"\<time = \>\"", ",", "t", ",", "\"\< optimal value = \>\"", ",", 
    RowBox[{"c", ".", "x"}]}], "]"}]}]], "Input",
 CellLabel->"In[20]:=",
 CellID->318414133],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"time = \"\>", "\[InvisibleSpace]", 
   RowBox[{"{", 
    RowBox[{"0.036002000000001276`", ",", "Null"}], "}"}], 
   "\[InvisibleSpace]", "\<\" optimal value = \"\>", "\[InvisibleSpace]", 
   RowBox[{"-", "10935.009979058392`"}]}],
  SequenceForm[
  "time = ", {0.036002000000001276`, Null}, 
   " optimal value = ", -10935.009979058392`],
  Editable->False]], "Print",
 CellLabel->"During evaluation of In[20]:=",
 CellID->369115005]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Interior Point Algorithm", "Subsection",
 CellID->155201634],

Cell["\<\
Although the simplex and revised simplex algorithms can be quite efficient on \
average, they have a poor worst-case behavior. It is possible to construct a \
linear programming problem for which the simplex or revised simplex methods \
take a number of steps exponential in the problem size. The interior point \
algorithm, however, has been proven to converge in a number of steps that are \
polynomial in the problem size. \
\>", "Text",
 CellID->168129131],

Cell[TextData[{
 "Furthermore, the ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " simplex and revised simplex implementation use dense linear algebra, while \
its interior point implementation uses machine-number sparse linear algebra. \
Therefore for large-scale, machine-number linear programming problems, the \
interior point method is more efficient and should be used."
}], "Text",
 CellID->204209391],

Cell[CellGroupData[{

Cell["Interior Point Formulation", "Subsubsection",
 CellID->302688295],

Cell["Consider the standardized linear programming problem", "Text",
 CellID->576509835],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{"Min", "   ", 
    SuperscriptBox["c", "T"], "x"}], ",", " ", 
   RowBox[{
    RowBox[{
     RowBox[{"s", ".", "t", ".", "  ", "A"}], " ", "x"}], "=", "b"}], ",", 
   " ", 
   RowBox[{"x", "\[GreaterEqual]", "0"}], ","}], 
  TraditionalForm]], "DisplayMath",
 CellID->190488124],

Cell[TextData[{
 "where ",
 Cell[BoxData[
  FormBox[
   RowBox[{"c", ",", " ", 
    RowBox[{"x", "\[Element]", 
     SuperscriptBox["R", "n"]}]}], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{"A", "\[Element]", 
    SuperscriptBox["R", 
     RowBox[{"m", "\[Times]", "n"}]]}], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{"b", "\[Element]", 
    SuperscriptBox["R", "m"]}], TraditionalForm]], "InlineMath"],
 ". This problem can be solved using a barrier function formulation to deal \
with the positive constraints"
}], "Text",
 CellID->86065235],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{"Min", "    ", 
     SuperscriptBox["c", "T"], "x"}], " ", "-", " ", 
    RowBox[{"t", " ", 
     RowBox[{
      UnderoverscriptBox["\[Sum]", 
       RowBox[{"i", "=", "1"}], "n"], 
      RowBox[{"ln", 
       RowBox[{"(", 
        SubscriptBox["x", "i"], ")"}]}]}]}]}], ",", " ", 
   RowBox[{
    RowBox[{
     RowBox[{"s", ".", "t", ".", "   ", "A"}], " ", "x"}], " ", "=", " ", 
    "b"}], ",", " ", 
   RowBox[{"x", "\[GreaterEqual]", " ", "0"}], ",", " ", 
   RowBox[{"t", " ", ">", " ", "0"}], ",", " ", 
   RowBox[{"t", " ", "\[Rule]", " ", "0"}]}], TraditionalForm]], "DisplayMath",\

 CellID->162687495],

Cell["\<\
The first-order necessary condition for the above problem gives\
\>", "Text",
 CellID->347626854],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{"c", " ", "-", " ", 
     RowBox[{"t", " ", 
      SuperscriptBox["X", 
       RowBox[{"-", "1"}]], "e"}]}], " ", "=", " ", 
    RowBox[{
     SuperscriptBox["A", "T"], "y"}]}], ",", " ", 
   RowBox[{
    RowBox[{"and", " ", "A", " ", "x"}], " ", "=", " ", "b"}], ",", " ", 
   RowBox[{"x", " ", "\[GreaterEqual]", " ", "0"}]}], 
  TraditionalForm]], "DisplayMath",
 CellID->166218520],

Cell[TextData[{
 "Let ",
 Cell[BoxData[
  FormBox["X", TraditionalForm]], "InlineMath"],
 " denote the diagonal matrix made of the vector ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 ", and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"z", "=", 
    RowBox[{"t", " ", 
     SuperscriptBox["X", 
      RowBox[{"-", "1"}]], " ", "e"}]}], TraditionalForm]], "InlineMath"],
 ". "
}], "Text",
 CellID->11219003],

Cell[BoxData[{
 FormBox[
  RowBox[{
   RowBox[{"x", " ", "z"}], "                     ", "=", " ", 
   RowBox[{"t", " ", "e"}]}], TraditionalForm], "\n", 
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{
     SuperscriptBox["A", "T"], "y"}], " ", "+", " ", "z"}], "    ", "=", " ", 
   "c"}], TraditionalForm], "\n", 
 FormBox[
  RowBox[{
   RowBox[{"A", " ", "x"}], "                    ", "=", " ", "b"}], 
  TraditionalForm], "\n", 
 FormBox[
  RowBox[{"x", ",", " ", 
   RowBox[{"z", " ", "\[GreaterEqual]", " ", "0"}]}], 
  TraditionalForm]}], "DisplayMath",
 CellID->342570519],

Cell[TextData[{
 "This is a set of ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"2", "m"}], "+", "n"}], TraditionalForm]], "InlineMath"],
 " linear/nonlinear equations with constraints. It can be solved using \
Newton's method"
}], "Text",
 CellID->130945983],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{"(", 
    RowBox[{"x", ",", " ", "y", ",", " ", "z"}], ")"}], " ", ":=", " ", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"x", ",", " ", "y", ",", " ", "z"}], ")"}], " ", "+", " ", 
    RowBox[{"(", 
     RowBox[{
     "\[CapitalDelta]x", ",", "\[CapitalDelta]y", ",", " ", 
      "\[CapitalDelta]z"}], ")"}]}]}], TraditionalForm]], "DisplayMath",
 CellID->45999338],

Cell["with", "Text",
 CellID->414010665],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{"(", GridBox[{
       {"X", "Z", "0"},
       {"I", "0", 
        SuperscriptBox["A", "T"]},
       {"0", "A", "0"}
      }], ")"}], 
    RowBox[{"(", GridBox[{
       {"\[CapitalDelta]z"},
       {"\[CapitalDelta]x"},
       {"\[CapitalDelta]y"}
      }], ")"}]}], " ", "=", " ", 
   RowBox[{
    RowBox[{"(", GridBox[{
       {
        RowBox[{
         RowBox[{"t", " ", "e"}], " ", "-", " ", 
         RowBox[{"x", " ", "z"}]}]},
       {
        RowBox[{"c", " ", "-", " ", 
         RowBox[{
          SuperscriptBox["A", "T"], " ", "y"}], " ", "-", " ", "z"}]},
       {
        RowBox[{"b", " ", "-", " ", 
         RowBox[{"A", " ", "x"}]}]}
      }], ")"}], "."}]}], TraditionalForm]], "DisplayMath",
 CellID->634693562],

Cell["\<\
One way to solve this linear system is to use Gaussian elimination to \
simplify the matrix into block triangular form.\
\>", "Text",
 CellID->467052005],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{"(", GridBox[{
      {"X", "Z", "0"},
      {"I", "0", 
       SuperscriptBox["A", "T"]},
      {"0", "A", "0"}
     }], ")"}], "  ", "\[Rule]", 
   RowBox[{
    RowBox[{"(", GridBox[{
       {"X", "Z", "0"},
       {"0", 
        RowBox[{
         SuperscriptBox["X", 
          RowBox[{"-", "1"}]], "Z"}], 
        SuperscriptBox["A", "T"]},
       {"0", "A", "0"}
      }], ")"}], " ", "\[Rule]", 
    RowBox[{"(", GridBox[{
       {"X", "Z", "0"},
       {"0", 
        RowBox[{
         SuperscriptBox["X", 
          RowBox[{"-", "1"}]], "Z"}], 
        SuperscriptBox["A", "T"]},
       {"0", "0", 
        RowBox[{"A", " ", 
         SuperscriptBox["Z", 
          RowBox[{"-", "1"}]], "X", " ", 
         SuperscriptBox["A", "T"]}]}
      }], ")"}]}]}], TraditionalForm]], "DisplayMath",
 CellID->307989],

Cell["\<\
To solve this block triangular matrix, the so-called normal system can be \
solved, with the matrix in this normal system\
\>", "Text",
 CellID->474983116],

Cell[BoxData[
 FormBox[
  RowBox[{"B", " ", "=", " ", 
   RowBox[{
    StyleBox["A",
     FontSlant->"Italic"], 
    StyleBox[" ",
     FontSlant->"Italic"], 
    SuperscriptBox[
     StyleBox["Z",
      FontSlant->"Italic"], 
     RowBox[{"-", "1"}]], "X", " ", 
    SuperscriptBox["A", "T"], " "}]}], TraditionalForm]], "DisplayMath",
 CellID->212964988],

Cell[TextData[{
 "This matrix is positive definite, but becomes very ill-conditioned as the \
solution is approached. Thus numerical techniques are used to stabilize the \
solution process, and typically the interior point method can only be \
expected to solve the problem to a tolerance of about ",
 Cell[BoxData[
  SqrtBox["$MachineEpsilon"]], "InlineFormula"],
 ", with tolerance explained below. ",
 StyleBox["Mathematica ",
  FontSlant->"Italic"],
 "uses Mehrotra's predictor-corrector scheme [1]."
}], "Text",
 CellID->12794403]
}, Open  ]],

Cell[CellGroupData[{

Cell["Convergence Tolerance", "Subsubsection",
 CellID->360075058],

Cell["\<\
General Linear Programming problems are first converted to the standard form\
\>", "Text",
 CellID->53829880],

Cell[BoxData[{
 FormBox[
  RowBox[{
   RowBox[{"Min", "  ", 
    SuperscriptBox["c", "T"], " ", "x"}], " "}], TraditionalForm], "\n", 
 FormBox[
  RowBox[{
   RowBox[{"subject", " ", "to", " ", "A", " ", "x"}], " ", "=", " ", "b"}], 
  TraditionalForm], "\n", 
 FormBox[
  RowBox[{"                            ", 
   RowBox[{"x", " ", "\[GreaterEqual]", " ", "0"}]}], 
  TraditionalForm]}], "DisplayMath",
 CellID->447319230],

Cell["with the corresponding dual ", "Text",
 CellID->108782254],

Cell[BoxData[{
 FormBox[
  RowBox[{"Max", "  ", 
   SuperscriptBox["b", "T"], " ", "y"}], TraditionalForm], "\n", 
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{"subject", " ", "to", " ", 
     SuperscriptBox[
      RowBox[{"A", " "}], "T"], "y"}], " ", "+", " ", "z"}], " ", "=", " ", 
   "c"}], TraditionalForm], "\n", 
 FormBox[
  RowBox[{"                         ", 
   RowBox[{"z", "\[GreaterEqual]", " ", "0"}]}], 
  TraditionalForm]}], "DisplayMath",
 CellID->264301056],

Cell["\<\
The convergence criterion for the interior point algorithm is \
\>", "Text",
 CellID->59181923],

Cell[BoxData[
 FormBox[
  RowBox[{
   StyleBox[
    RowBox[{
     FractionBox[
      RowBox[{"||", 
       RowBox[{"b", "-", 
        RowBox[{"A", " ", "x"}]}], "||"}], 
      RowBox[{"max", "(", 
       RowBox[{"1", ",", " ", 
        RowBox[{"||", "b", "||"}]}], ")"}]], "+", " ", 
     FractionBox[
      RowBox[{"||", 
       RowBox[{"c", " ", "-", " ", 
        RowBox[{
         SuperscriptBox["A", "T"], "y"}], " ", "-", " ", "z"}], "||"}], 
      RowBox[{"max", "(", 
       RowBox[{"1", ",", " ", 
        RowBox[{"||", "c", "||"}]}], ")"}]], " ", "+", " ", 
     FractionBox[
      RowBox[{"||", 
       RowBox[{
        RowBox[{
         SuperscriptBox["c", "T"], "x"}], " ", "-", 
        RowBox[{
         SuperscriptBox["b", "T"], "y"}]}], " ", "||"}], 
      RowBox[{"max", "(", 
       RowBox[{"1", ",", " ", 
        RowBox[{"||", 
         RowBox[{
          SuperscriptBox["c", "T"], "x"}], "||"}], ",", 
        RowBox[{"||", 
         RowBox[{
          SuperscriptBox["b", "T"], "y"}], "||"}]}], ")"}]]}],
    FontFamily->"Utopia"], "\[LessEqual]", " ", "tolerance"}], 
  TraditionalForm]], "DisplayMath",
 CellID->830006935],

Cell[TextData[{
 "with the tolerance set, by default, to ",
 Cell[BoxData[
  SqrtBox["$MachineEpsilon"]], "InlineFormula"],
 ". "
}], "Text",
 CellID->34314487]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["References", "Section",
 CellID->174792224],

Cell[TextData[{
 "[1] R. Vanderbei, ",
 StyleBox["Linear Programming: Foundations and Extensions",
  FontSlant->"Italic"],
 ", New York: Springer-Verlag, 2001."
}], "Text",
 CellID->2034996284],

Cell[TextData[{
 "[2] S. Mehrotra, \"On the Implementation of a Primal-Dual Interior Point \
Method,'' ",
 StyleBox["SIAM Journal on Optimization",
  FontSlant->"Italic"],
 ", ",
 StyleBox["2",
  FontWeight->"Bold"],
 ", 1992 pp. 575\[Dash]601."
}], "Text",
 CellID->1592161131],

Cell[TextData[ButtonBox["Constrained Optimization",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/ConstrainedOptimizationOverview"]], "RelatedTutorials",
 CellID->1633042],

Cell[TextData[ButtonBox["Numerical Nonlinear Local Optimization",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/ConstrainedOptimizationLocalNumerical"]], \
"RelatedTutorials",
 CellID->511088424],

Cell[TextData[ButtonBox["Numerical Nonlinear Global Optimization",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/ConstrainedOptimizationGlobalNumerical"]], \
"RelatedTutorials",
 CellID->125693634],

Cell[TextData[ButtonBox["Exact Global Optimization",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/ConstrainedOptimizationExact"]], "RelatedTutorials",
 CellID->140984286]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Linear Programming - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 33, 32.9054586}", "context" -> "", 
    "keywords" -> {
     "linear programming", "interior point method", "simplex method", 
      "revised simplex method", "linear optimization"}, "index" -> True, 
    "label" -> "Mathematica Tutorial", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "Linear programming problems are optimization problems where the \
objective function and constraints are all linear. Mathematica has a \
collection of algorithms for solving linear optimization problems with real \
variables, accessed via LinearProgramming, FindMinimum, FindMaximum, \
NMinimize, NMaximize, Minimize, and Maximize. LinearProgramming gives direct \
access to linear programming algorithms, provides the most flexibility for \
specifying the methods used, and is the most efficient for large-scale \
problems. FindMinimum, FindMaximum, NMinimize, NMaximize, Minimize, and \
Maximize are convenient for solving linear programming problems in equation \
and inequality form. This solves a linear programming problem Min x + 2 y \
s.t. -5 x + y = 7 x + y >= 26 x >= 3, y >= 4 using Minimize.", 
    "synonyms" -> {}, "title" -> "Linear Programming", "type" -> "Tutorial", 
    "uri" -> "tutorial/ConstrainedOptimizationLinearProgramming"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 1262, 27, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[1887, 54, 55, 1, 70, "Title",
 CellID->100661346],
Cell[CellGroupData[{
Cell[1967, 59, 51, 1, 70, "Section",
 CellID->430238072],
Cell[2021, 62, 159, 4, 70, "Text",
 CellID->15733215],
Cell[2183, 68, 2319, 81, 70, "Text",
 CellID->706807348],
Cell[4505, 151, 1041, 34, 70, "MathCaption",
 CellID->292783491],
Cell[CellGroupData[{
Cell[5571, 189, 607, 20, 28, "Input",
 CellID->138525404],
Cell[6181, 211, 394, 14, 51, "Output",
 CellID->482087615]
}, Open  ]],
Cell[6590, 228, 471, 18, 70, "MathCaption",
 CellID->667638231],
Cell[CellGroupData[{
Cell[7086, 250, 607, 20, 28, "Input",
 CellID->73921660],
Cell[7696, 272, 373, 12, 36, "Output",
 CellID->257888045]
}, Open  ]],
Cell[8084, 287, 277, 10, 70, "MathCaption",
 CellID->92461611],
Cell[CellGroupData[{
Cell[8386, 301, 707, 25, 70, "Input",
 CellID->14420703],
Cell[9096, 328, 251, 9, 51, "Output",
 CellID->1740945228]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[9396, 343, 69, 1, 70, "Section",
 CellID->113426546],
Cell[9468, 346, 352, 10, 70, "Text",
 CellID->138121570],
Cell[9823, 358, 72, 1, 70, "Text",
 CellID->1628457416],
Cell[9898, 361, 823, 24, 70, "DefinitionBox3Col",
 CellID->139774623],
Cell[10724, 387, 209, 8, 70, "Caption",
 CellID->624514127],
Cell[10936, 397, 834, 26, 70, "Text",
 CellID->355015470],
Cell[11773, 425, 224, 8, 70, "Text",
 CellID->12397142]
}, Open  ]],
Cell[CellGroupData[{
Cell[12034, 438, 47, 1, 70, "Section",
 CellID->414335858],
Cell[CellGroupData[{
Cell[12106, 443, 118, 3, 70, "Subsection",
 CellID->589526112],
Cell[12227, 448, 704, 13, 70, "Text",
 CellID->50752786],
Cell[12934, 463, 617, 12, 70, "Text",
 CellID->776588746],
Cell[13554, 477, 477, 14, 70, "MathCaption",
 CellID->141642741],
Cell[CellGroupData[{
Cell[14056, 495, 478, 16, 70, "Input",
 CellID->644934419],
Cell[14537, 513, 245, 8, 36, "Output",
 CellID->1714401028]
}, Open  ]],
Cell[14797, 524, 236, 8, 70, "MathCaption",
 CellID->311332941],
Cell[CellGroupData[{
Cell[15058, 536, 481, 17, 70, "Input",
 CellID->189718616],
Cell[15542, 555, 207, 7, 36, "Output",
 CellID->208662731]
}, Open  ]],
Cell[15764, 565, 216, 5, 70, "MathCaption",
 CellID->208795567],
Cell[CellGroupData[{
Cell[16005, 574, 856, 26, 70, "Input",
 CellID->873231639],
Cell[16864, 602, 227, 7, 36, "Output",
 CellID->37207525]
}, Open  ]],
Cell[CellGroupData[{
Cell[17128, 614, 459, 13, 70, "Input",
 CellID->1653932530],
Cell[17590, 629, 227, 7, 36, "Output",
 CellID->1118648765]
}, Open  ]],
Cell[CellGroupData[{
Cell[17854, 641, 212, 7, 70, "Input",
 CellID->1454909102],
Cell[18069, 650, 181, 5, 36, "Output",
 CellID->901336229]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[18299, 661, 64, 1, 70, "Subsection",
 CellID->224700240],
Cell[18366, 664, 80, 1, 70, "Text",
 CellID->507175941],
Cell[18449, 667, 896, 27, 70, "DisplayMath",
 CellID->204420850],
Cell[19348, 696, 46, 1, 70, "Text",
 CellID->16053491],
Cell[19397, 699, 868, 25, 70, "DisplayMath",
 CellID->693919847],
Cell[20268, 726, 105, 3, 70, "Text",
 CellID->40882290],
Cell[20376, 731, 148, 4, 70, "Text",
 CellID->16683215],
Cell[20527, 737, 577, 14, 70, "Text",
 CellID->250487882],
Cell[21107, 753, 347, 10, 70, "Text",
 CellID->44218844],
Cell[21457, 765, 776, 27, 70, "DisplayMath",
 CellID->28996498],
Cell[22236, 794, 235, 8, 70, "Text",
 CellID->80499807],
Cell[22474, 804, 2158, 69, 70, "MathCaption",
 CellID->48134959],
Cell[CellGroupData[{
Cell[24657, 877, 567, 19, 70, "Input",
 CellID->947100034],
Cell[25227, 898, 397, 14, 36, "Output",
 CellID->1620664186]
}, Open  ]],
Cell[25639, 915, 120, 3, 70, "MathCaption",
 CellID->539582753],
Cell[CellGroupData[{
Cell[25784, 922, 140, 5, 70, "Input",
 CellID->452933674],
Cell[25927, 929, 161, 5, 36, "Output",
 CellID->1650563956]
}, Open  ]],
Cell[CellGroupData[{
Cell[26125, 939, 302, 10, 70, "Input",
 CellID->28001828],
Cell[26430, 951, 161, 5, 36, "Output",
 CellID->1891333427]
}, Open  ]],
Cell[26606, 959, 416, 10, 70, "MathCaption",
 CellID->648985],
Cell[CellGroupData[{
Cell[27047, 973, 597, 20, 70, "Input",
 CellID->293155135],
Cell[27647, 995, 415, 14, 36, "Output",
 CellID->1765940800]
}, Open  ]],
Cell[28077, 1012, 103, 2, 70, "MathCaption",
 CellID->309989897],
Cell[CellGroupData[{
Cell[28205, 1018, 178, 7, 70, "Input",
 CellID->12741327],
Cell[28386, 1027, 180, 5, 36, "Output",
 CellID->1071294700]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[28615, 1038, 123, 3, 70, "Subsection",
 CellID->512388034],
Cell[28741, 1043, 299, 6, 70, "Text",
 CellID->377803856],
Cell[29043, 1051, 131, 4, 70, "MathCaption",
 CellID->8926348],
Cell[CellGroupData[{
Cell[29199, 1059, 595, 20, 70, "Input",
 CellID->452474303],
Cell[29797, 1081, 458, 9, 70, "Message",
 CellID->4018639],
Cell[30258, 1092, 686, 24, 54, "Output",
 CellID->48032705]
}, Open  ]],
Cell[30959, 1119, 143, 4, 70, "MathCaption",
 CellID->20528856],
Cell[CellGroupData[{
Cell[31127, 1127, 407, 13, 70, "Input",
 CellID->181927198],
Cell[31537, 1142, 660, 11, 70, "Message",
 CellID->76091856],
Cell[32200, 1155, 501, 17, 36, "Output",
 CellID->6869406]
}, Open  ]],
Cell[32716, 1175, 194, 6, 70, "MathCaption",
 CellID->727892506],
Cell[CellGroupData[{
Cell[32935, 1185, 400, 13, 70, "Input",
 CellID->51654650],
Cell[33338, 1200, 399, 9, 70, "Message",
 CellID->12449505],
Cell[33740, 1211, 230, 7, 36, "Output",
 CellID->1919329137]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[34019, 1224, 81, 1, 70, "Subsection",
 CellID->605660219],
Cell[34103, 1227, 558, 13, 70, "Text",
 CellID->654114],
Cell[34664, 1242, 141, 4, 70, "MathCaption",
 CellID->126549907],
Cell[34808, 1248, 783, 26, 70, "Input",
 CellID->936774867],
Cell[CellGroupData[{
Cell[35616, 1278, 731, 21, 70, "Input",
 CellID->191014035],
Cell[36350, 1301, 349, 11, 36, "Output",
 CellID->1182785113]
}, Open  ]],
Cell[CellGroupData[{
Cell[36736, 1317, 142, 5, 70, "Input",
 CellID->1922830587],
Cell[36881, 1324, 182, 5, 36, "Output",
 CellID->1586095220]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[37124, 1336, 94, 1, 70, "Section",
 CellID->9813130],
Cell[37221, 1339, 224, 5, 70, "Text",
 CellID->82319969],
Cell[CellGroupData[{
Cell[37470, 1348, 88, 1, 70, "Subsection",
 CellID->208553956],
Cell[37561, 1351, 864, 31, 70, "Text",
 CellID->122417020],
Cell[38428, 1384, 301, 10, 70, "MathCaption",
 CellID->448660818],
Cell[CellGroupData[{
Cell[38754, 1398, 163, 5, 28, "Input",
 CellID->103415320],
Cell[38920, 1405, 12609, 211, 479, "Output",
 Evaluatable->False,
 CellID->148306818]
}, Open  ]],
Cell[CellGroupData[{
Cell[51566, 1621, 102, 3, 70, "Input",
 CellID->299458573],
Cell[51671, 1626, 3340, 84, 122, "Output",
 CellID->934999415]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[55060, 1716, 99, 1, 70, "Subsection",
 CellID->445507631],
Cell[55162, 1719, 472, 14, 70, "Text",
 CellID->137995970],
Cell[55637, 1735, 262, 9, 70, "MathCaption",
 CellID->297516826],
Cell[CellGroupData[{
Cell[55924, 1748, 205, 6, 70, "Input",
 CellID->129022199],
Cell[56132, 1756, 286, 8, 36, "Output",
 CellID->90286632]
}, Open  ]],
Cell[56433, 1767, 306, 9, 70, "MathCaption",
 CellID->211685703],
Cell[56742, 1778, 319, 10, 70, "Input",
 CellID->205726187],
Cell[57064, 1790, 93, 1, 70, "MathCaption",
 CellID->5890066],
Cell[57160, 1793, 230, 8, 70, "Input",
 CellID->52691358],
Cell[CellGroupData[{
Cell[57415, 1805, 95, 3, 70, "Input",
 CellID->1056069432],
Cell[57513, 1810, 196, 6, 36, "Output",
 CellID->2101418431]
}, Open  ]],
Cell[57724, 1819, 196, 5, 70, "MathCaption",
 CellID->201121673],
Cell[CellGroupData[{
Cell[57945, 1828, 216, 7, 70, "Input",
 CellID->96551054],
Cell[58164, 1837, 493, 17, 36, "Output",
 CellID->465458690]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[58706, 1860, 66, 1, 70, "Subsection",
 CellID->298767088],
Cell[58775, 1863, 836, 26, 70, "Text",
 CellID->437791027],
Cell[59614, 1891, 229, 9, 70, "MathCaption",
 CellID->23023199],
Cell[59846, 1902, 423, 10, 70, "Input",
 CellID->8788611],
Cell[60272, 1914, 118, 3, 70, "MathCaption",
 CellID->80037026],
Cell[60393, 1919, 306, 10, 70, "Input",
 CellID->412264192],
Cell[60702, 1931, 264, 10, 70, "MathCaption",
 CellID->467073014],
Cell[CellGroupData[{
Cell[60991, 1945, 205, 5, 70, "Input",
 CellID->903242654],
Cell[61199, 1952, 1957, 58, 71, "Output",
 CellID->997080558]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[63205, 2016, 74, 1, 70, "Subsection",
 CellID->820857862],
Cell[63282, 2019, 372, 13, 70, "Text",
 CellID->439047054],
Cell[63657, 2034, 86, 1, 70, "MathCaption",
 CellID->1015647543],
Cell[CellGroupData[{
Cell[63768, 2039, 131, 3, 70, "Input",
 CellID->489880541],
Cell[63902, 2044, 13778, 374, 1125, "Output"]
}, Open  ]],
Cell[77695, 2421, 152, 5, 70, "MathCaption",
 CellID->17729874],
Cell[CellGroupData[{
Cell[77872, 2430, 194, 6, 28, "Input",
 CellID->1200455685],
Cell[78069, 2438, 3387, 60, 173, "Output",
 Evaluatable->False,
 CellID->3719019]
}, Open  ]],
Cell[CellGroupData[{
Cell[81493, 2503, 109, 3, 70, "Input",
 CellID->878467550],
Cell[81605, 2508, 721, 15, 54, "Output",
 CellID->1183238080]
}, Open  ]],
Cell[82341, 2526, 180, 5, 70, "MathCaption",
 CellID->1198832590],
Cell[CellGroupData[{
Cell[82546, 2535, 237, 7, 70, "Input",
 CellID->855712891],
Cell[82786, 2544, 393, 10, 54, "Output",
 CellID->1360471072]
}, Open  ]],
Cell[83194, 2557, 144, 5, 70, "MathCaption",
 CellID->295471405],
Cell[CellGroupData[{
Cell[83363, 2566, 237, 7, 28, "Input",
 CellID->1377141912],
Cell[83603, 2575, 12608, 211, 479, "Output",
 Evaluatable->False,
 CellID->46806323]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[96272, 2793, 81, 1, 70, "Section",
 CellID->261990371],
Cell[CellGroupData[{
Cell[96378, 2798, 62, 1, 70, "Subsection",
 CellID->293106634],
Cell[96443, 2801, 190, 7, 70, "Text",
 CellID->22273430],
Cell[96636, 2810, 269, 9, 70, "DisplayMath",
 CellID->38244179],
Cell[96908, 2821, 91, 1, 70, "Text",
 CellID->310897012],
Cell[97002, 2824, 540, 16, 70, "DisplayMath",
 CellID->20392480],
Cell[97545, 2842, 217, 8, 70, "MathCaption",
 CellID->1470753465],
Cell[97765, 2852, 2369, 68, 70, "Input",
 CellID->169775421],
Cell[100137, 2922, 93, 1, 70, "MathCaption",
 CellID->215389070],
Cell[100233, 2925, 345, 15, 70, "DisplayMath"],
Cell[100581, 2942, 224, 8, 70, "MathCaption",
 CellID->254791179],
Cell[CellGroupData[{
Cell[100830, 2954, 680, 22, 70, "Input",
 CellID->228353195],
Cell[101513, 2978, 413, 9, 70, "Message",
 CellID->1702910108],
Cell[101929, 2989, 635, 21, 36, "Output",
 CellID->276395405]
}, Open  ]],
Cell[102579, 3013, 193, 8, 70, "MathCaption",
 CellID->345408722],
Cell[CellGroupData[{
Cell[102797, 3025, 158, 5, 70, "Input",
 CellID->312111314],
Cell[102958, 3032, 237, 8, 51, "Output",
 CellID->1486497184]
}, Open  ]],
Cell[CellGroupData[{
Cell[103232, 3045, 372, 14, 70, "Input",
 CellID->304875770],
Cell[103607, 3061, 245, 8, 36, "Output",
 CellID->1715359613]
}, Open  ]],
Cell[103867, 3072, 474, 17, 70, "MathCaption",
 CellID->62657209],
Cell[CellGroupData[{
Cell[104366, 3093, 158, 5, 70, "Input",
 CellID->693798693],
Cell[104527, 3100, 290, 10, 51, "Output",
 CellID->129057075]
}, Open  ]],
Cell[CellGroupData[{
Cell[104854, 3115, 374, 14, 70, "Input",
 CellID->155493228],
Cell[105231, 3131, 246, 8, 36, "Output",
 CellID->1366635944]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[105526, 3145, 69, 1, 70, "Subsection",
 CellID->326994842],
Cell[105598, 3148, 307, 9, 70, "Text",
 CellID->595713038],
Cell[105908, 3159, 491702, 6786, 70, 489203, 6741, "GraphicsData", \
"PostScript", "Graphics",
 CellID->58852609],
Cell[597613, 9947, 497, 8, 70, "Text",
 CellID->504518801],
Cell[598113, 9957, 52374, 2676, 70, 24859, 2221, "GraphicsData", \
"PostScript", "Text",
 CellID->272052816],
Cell[650490, 12635, 224, 5, 70, "Text",
 CellID->316296097],
Cell[650717, 12642, 268, 8, 70, "DisplayMath",
 CellID->627637036],
Cell[650988, 12652, 258, 6, 70, "DisplayMath",
 CellID->142308570],
Cell[651249, 12660, 189, 4, 70, "Text",
 CellID->390858683],
Cell[651441, 12666, 186, 6, 70, "DisplayMath",
 CellID->113688339],
Cell[651630, 12674, 224, 6, 70, "DisplayMath",
 CellID->287545174],
Cell[651857, 12682, 1049, 33, 70, "Text",
 CellID->535486842],
Cell[652909, 12717, 430, 14, 70, "DisplayMath",
 CellID->249283724],
Cell[653342, 12733, 831, 24, 70, "DisplayMath",
 CellID->206317008],
Cell[654176, 12759, 290, 9, 70, "MathCaption",
 CellID->361102706],
Cell[654469, 12770, 28868, 714, 70, "Input",
 CellID->197698860],
Cell[683340, 13486, 139, 4, 70, "MathCaption",
 CellID->185115299],
Cell[CellGroupData[{
Cell[683504, 13494, 1328, 45, 70, "Input",
 CellID->259327580],
Cell[CellGroupData[{
Cell[684857, 13543, 398, 9, 70, "Print",
 CellID->2092791593],
Cell[685258, 13554, 294, 7, 23, "Print",
 CellID->475838129]
}, Open  ]],
Cell[685567, 13564, 20501, 340, 206, "Output",
 Evaluatable->False,
 CellID->675201650]
}, Open  ]],
Cell[706083, 13907, 265, 5, 70, "MathCaption",
 CellID->253723770],
Cell[CellGroupData[{
Cell[706373, 13916, 1305, 44, 70, "Input",
 CellID->2126659739],
Cell[CellGroupData[{
Cell[707703, 13964, 397, 9, 70, "Print",
 CellID->400303040],
Cell[708103, 13975, 313, 7, 23, "Print",
 CellID->1732451358]
}, Open  ]],
Cell[708431, 13985, 13821, 231, 205, "Output",
 Evaluatable->False,
 CellID->462811892]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[722313, 14223, 71, 1, 70, "Section",
 CellID->38991160],
Cell[CellGroupData[{
Cell[722409, 14228, 80, 1, 70, "Subsection",
 CellID->160402625],
Cell[722492, 14231, 370, 7, 70, "Text",
 CellID->391192688],
Cell[722865, 14240, 400, 7, 70, "Text",
 CellID->788121671],
Cell[723268, 14249, 411, 9, 70, "Text",
 CellID->65762130],
Cell[723682, 14260, 140, 4, 70, "MathCaption",
 CellID->425368342],
Cell[723825, 14266, 1211, 42, 70, "Input",
 CellID->13393687],
Cell[725039, 14310, 312, 6, 70, "MathCaption",
 CellID->298559827],
Cell[CellGroupData[{
Cell[725376, 14320, 706, 20, 70, "Input",
 CellID->18667422],
Cell[726085, 14342, 654, 13, 70, "Print",
 CellID->1628362047]
}, Open  ]],
Cell[CellGroupData[{
Cell[726776, 14360, 715, 20, 70, "Input",
 CellID->192654607],
Cell[727494, 14382, 651, 13, 70, "Print",
 CellID->1516079829]
}, Open  ]],
Cell[728160, 14398, 204, 5, 70, "MathCaption",
 CellID->113254445],
Cell[CellGroupData[{
Cell[728389, 14407, 657, 19, 70, "Input",
 CellID->318414133],
Cell[729049, 14428, 482, 12, 70, "Print",
 CellID->369115005]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[729580, 14446, 66, 1, 70, "Subsection",
 CellID->155201634],
Cell[729649, 14449, 470, 8, 70, "Text",
 CellID->168129131],
Cell[730122, 14459, 415, 9, 70, "Text",
 CellID->204209391],
Cell[CellGroupData[{
Cell[730562, 14472, 71, 1, 70, "Subsubsection",
 CellID->302688295],
Cell[730636, 14475, 88, 1, 70, "Text",
 CellID->576509835],
Cell[730727, 14478, 324, 11, 70, "DisplayMath",
 CellID->190488124],
Cell[731054, 14491, 613, 21, 70, "Text",
 CellID->86065235],
Cell[731670, 14514, 671, 21, 70, "DisplayMath",
 CellID->162687495],
Cell[732344, 14537, 107, 3, 70, "Text",
 CellID->347626854],
Cell[732454, 14542, 444, 14, 70, "DisplayMath",
 CellID->166218520],
Cell[732901, 14558, 424, 16, 70, "Text",
 CellID->11219003],
Cell[733328, 14576, 580, 19, 70, "DisplayMath",
 CellID->342570519],
Cell[733911, 14597, 267, 9, 70, "Text",
 CellID->130945983],
Cell[734181, 14608, 412, 12, 70, "DisplayMath",
 CellID->45999338],
Cell[734596, 14622, 40, 1, 70, "Text",
 CellID->414010665],
Cell[734639, 14625, 789, 29, 70, "DisplayMath",
 CellID->634693562],
Cell[735431, 14656, 163, 4, 70, "Text",
 CellID->467052005],
Cell[735597, 14662, 859, 32, 70, "DisplayMath",
 CellID->307989],
Cell[736459, 14696, 165, 4, 70, "Text",
 CellID->474983116],
Cell[736627, 14702, 356, 13, 70, "DisplayMath",
 CellID->212964988],
Cell[736986, 14717, 535, 12, 70, "Text",
 CellID->12794403]
}, Open  ]],
Cell[CellGroupData[{
Cell[737558, 14734, 66, 1, 70, "Subsubsection",
 CellID->360075058],
Cell[737627, 14737, 119, 3, 70, "Text",
 CellID->53829880],
Cell[737749, 14742, 425, 13, 70, "DisplayMath",
 CellID->447319230],
Cell[738177, 14757, 64, 1, 70, "Text",
 CellID->108782254],
Cell[738244, 14760, 477, 15, 70, "DisplayMath",
 CellID->264301056],
Cell[738724, 14777, 105, 3, 70, "Text",
 CellID->59181923],
Cell[738832, 14782, 1147, 37, 70, "DisplayMath",
 CellID->830006935],
Cell[739982, 14821, 160, 6, 70, "Text",
 CellID->34314487]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[740203, 14834, 49, 1, 70, "Section",
 CellID->174792224],
Cell[740255, 14837, 193, 6, 70, "Text",
 CellID->2034996284],
Cell[740451, 14845, 278, 10, 70, "Text",
 CellID->1592161131],
Cell[740732, 14857, 178, 4, 70, "RelatedTutorials",
 CellID->1633042],
Cell[740913, 14863, 202, 5, 70, "RelatedTutorials",
 CellID->511088424],
Cell[741118, 14870, 204, 5, 70, "RelatedTutorials",
 CellID->125693634],
Cell[741325, 14877, 178, 4, 70, "RelatedTutorials",
 CellID->140984286]
}, Open  ]]
}, Open  ]],
Cell[741530, 14885, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

