(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     44916,       1293]
NotebookOptionsPosition[     35132,        953]
NotebookOutlinePosition[     41170,       1142]
CellTagsIndexPosition[     41077,       1136]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], Cell[TextData[Cell[
    BoxData[
     PopupMenuBox[
      Dynamic[{"ActionMenu", None}, 
       Part[{"Differential Equation Solving with DSolve" :> 
         Documentation`HelpLookup["paclet:tutorial/DSolveOverview"]}, #, 2]& ,
        Evaluator -> Automatic], {
      1->"\<\"Differential Equation Solving with DSolve\"\>"}, "\<\"related \
tutorials\"\>", 
      StyleBox["\<\"related tutorials\"\>",
       Background->Automatic],
      Appearance->Automatic,
      ImageSize->Automatic,
      MenuAppearance->"Menu"]],
     FontSize->9]], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["Nonlinear IVPs and BVPs", "Title",
 CellTags->{"c:48", "b:4.2"},
 CellID->28252],

Cell[TextData[{
 "Many real-world applications require the solution of IVPs and BVPs for \
nonlinear",
 " ",
 "ODEs. For example, consider the ",
 StyleBox["logistic equation, ",
  FontSlant->"Italic"],
 "which occurs in population dynamics."
}], "Text",
 CellTags->"b:4.2",
 CellID->25011],

Cell["This is the logistic equation.", "MathCaption",
 CellTags->"b:4.2",
 CellID->1560],

Cell[BoxData[
 RowBox[{
  RowBox[{"LogisticEquation", " ", "=", " ", 
   RowBox[{
    RowBox[{
     RowBox[{"y", "'"}], "[", "t", "]"}], "\[Equal]", " ", 
    RowBox[{"r", 
     RowBox[{"(", 
      RowBox[{"1", "-", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"y", "[", "t", "]"}], "/", "K"}], ")"}]}], ")"}], "*", 
     RowBox[{"y", "[", "t", "]"}]}]}]}], ";"}]], "Input",
 CellTags->"b:4.2",
 CellLabel->"In[1]:=",
 CellID->15834],

Cell[TextData[{
 "The right-hand side of the equation can be expanded to a quadratic \
polynomial in ",
 Cell[BoxData[
  FormBox[
   RowBox[{"y", "[", "t", "]"}], TraditionalForm]], "InlineMath"],
 ". Hence, the logistic equation is simply a Riccati equation, and its \
general solution can be easily found."
}], "MathCaption",
 CellTags->"b:4.2",
 CellID->1840],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{"LogisticEquation", ",", " ", "y", ",", "t"}], "]"}]], "Input",
 CellTags->"b:4.2",
 CellLabel->"In[2]:=",
 CellID->31497],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    RowBox[{"Function", "[", 
     RowBox[{
      RowBox[{"{", "t", "}"}], ",", 
      FractionBox[
       RowBox[{
        SuperscriptBox["\[ExponentialE]", 
         RowBox[{
          RowBox[{"r", " ", "t"}], "+", 
          RowBox[{"K", " ", 
           RowBox[{"C", "[", "1", "]"}]}]}]], " ", "K"}], 
       RowBox[{
        RowBox[{"-", "1"}], "+", 
        SuperscriptBox["\[ExponentialE]", 
         RowBox[{
          RowBox[{"r", " ", "t"}], "+", 
          RowBox[{"K", " ", 
           RowBox[{"C", "[", "1", "]"}]}]}]]}]]}], "]"}]}], "}"}], 
  "}"}]], "Output",
 ImageSize->{241, 34},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:4.2",
 CellLabel->"Out[2]=",
 CellID->461409372]
}, Open  ]],

Cell[TextData[{
 "This sets the intrinsic growth rate ",
 Cell[BoxData["r"], "InlineFormula"],
 " to ",
 Cell[BoxData[
  RowBox[{"1", "/", "2"}]], "InlineFormula"],
 " and the saturation level ",
 Cell[BoxData["K"], "InlineFormula"],
 " to ",
 Cell[BoxData["4"], "InlineFormula"],
 " and solves the initial value problem. The warning message from ",
 Cell[BoxData[
  ButtonBox["Solve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Solve"]], "InlineFormula"],
 " is issued while solving for the arbitrary constant ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["C",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/C"], "[", "1", "]"}]], "InlineFormula"],
 " from the general solution."
}], "MathCaption",
 CellTags->"b:4.2",
 CellID->18529],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"LogisticEquation", "/.", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"r", "\[Rule]", " ", 
         RowBox[{"(", 
          RowBox[{"1", "/", "2"}], ")"}]}], ",", " ", 
        RowBox[{"K", "\[Rule]", " ", "4"}]}], "}"}]}], ",", " ", 
     RowBox[{
      RowBox[{"y", "[", "0", "]"}], "\[Equal]", "1"}]}], "}"}], ",", " ", "y",
    ",", "t"}], "]"}]], "Input",
 CellTags->"b:4.2",
 CellLabel->"In[3]:=",
 CellID->31540],

Cell[BoxData[
 RowBox[{
  RowBox[{"Solve", "::", "\<\"ifun\"\>"}], ":", 
  " ", "\<\"Inverse functions are being used by \\!\\(Solve\\), so some \
solutions may not be found; use Reduce for complete solution information. \\!\
\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/Solve/ifun\\\", \
ButtonNote -> \\\"Solve::ifun\\\"]\\)\"\>"}]], "Message", "MSG",
 CellTags->"b:4.2",
 CellLabel->"During evaluation of In[3]:=",
 CellID->290779130],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    RowBox[{"Function", "[", 
     RowBox[{
      RowBox[{"{", "t", "}"}], ",", 
      FractionBox[
       RowBox[{"4", " ", 
        SuperscriptBox["\[ExponentialE]", 
         RowBox[{"t", "/", "2"}]]}], 
       RowBox[{"3", "+", 
        SuperscriptBox["\[ExponentialE]", 
         RowBox[{"t", "/", "2"}]]}]]}], "]"}]}], "}"}], "}"}]], "Output",
 ImageSize->{204, 34},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:4.2",
 CellLabel->"Out[3]=",
 CellID->256478497]
}, Open  ]],

Cell[TextData[{
 "This solves the initial value problem for the logistic equation with \
symbolic parameters ",
 Cell[BoxData[Cell["r", "InlineFormula"]]],
 " and ",
 Cell[BoxData["K"], "InlineFormula"],
 "."
}], "MathCaption",
 CellTags->"b:4.2",
 CellID->13392],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", "=", 
  RowBox[{"DSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"LogisticEquation", ",", " ", 
      RowBox[{
       RowBox[{"y", "[", "0", "]"}], "\[Equal]", 
       RowBox[{"a", "*", "K"}]}]}], "}"}], ",", " ", "y", ",", "t"}], 
   "]"}]}]], "Input",
 CellTags->"b:4.2",
 CellLabel->"In[4]:=",
 CellID->13210],

Cell[BoxData[
 RowBox[{
  RowBox[{"Solve", "::", "\<\"ifun\"\>"}], ":", 
  " ", "\<\"Inverse functions are being used by \\!\\(Solve\\), so some \
solutions may not be found; use Reduce for complete solution information. \\!\
\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/Solve/ifun\\\", \
ButtonNote -> \\\"Solve::ifun\\\"]\\)\"\>"}]], "Message", "MSG",
 CellTags->"b:4.2",
 CellLabel->"During evaluation of In[4]:=",
 CellID->1246817],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    RowBox[{"Function", "[", 
     RowBox[{
      RowBox[{"{", "t", "}"}], ",", 
      FractionBox[
       RowBox[{"a", " ", 
        SuperscriptBox["\[ExponentialE]", 
         RowBox[{"r", " ", "t"}]], " ", "K"}], 
       RowBox[{"1", "-", "a", "+", 
        RowBox[{"a", " ", 
         SuperscriptBox["\[ExponentialE]", 
          RowBox[{"r", " ", "t"}]]}]}]]}], "]"}]}], "}"}], "}"}]], "Output",
 ImageSize->{231, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:4.2",
 CellLabel->"Out[4]=",
 CellID->69060035]
}, Open  ]],

Cell["\<\
This verifies that the solution satisfies the equation and the initial \
condition.\
\>", "MathCaption",
 CellTags->"b:4.2",
 CellID->26979],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"LogisticEquation", ",", " ", 
     RowBox[{"y", "[", "0", "]"}]}], "}"}], "/.", 
   RowBox[{"sol", "[", 
    RowBox[{"[", "1", "]"}], "]"}]}], "//", "Simplify"}]], "Input",
 CellTags->"b:4.2",
 CellLabel->"In[5]:=",
 CellID->30971],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"True", ",", 
   RowBox[{"a", " ", "K"}]}], "}"}]], "Output",
 ImageSize->{75, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:4.2",
 CellLabel->"Out[5]=",
 CellID->282737580]
}, Open  ]],

Cell[TextData[{
 "Here is a plot of the solution for different values of ",
 Cell[BoxData[Cell["r", "InlineFormula"]]],
 " and ",
 Cell[BoxData["K"], "InlineFormula"],
 "."
}], "MathCaption",
 CellTags->"b:4.2",
 CellID->73],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"y", "[", "t", "]"}], "/.", 
        RowBox[{"sol", "[", 
         RowBox[{"[", "1", "]"}], "]"}]}], "/.", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"K", "\[Rule]", " ", "4"}], ",", " ", 
         RowBox[{"a", "\[Rule]", " ", "i"}], ",", " ", 
         RowBox[{"r", "\[Rule]", " ", 
          RowBox[{"(", 
           RowBox[{"1", "/", "3"}], ")"}]}]}], "}"}]}], ",", " ", 
      RowBox[{"{", 
       RowBox[{"i", ",", "2", ",", 
        RowBox[{"1", "/", "10"}], ",", " ", 
        RowBox[{
         RowBox[{"-", "1"}], "/", "3"}]}], "}"}]}], "]"}], "]"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "8"}], "}"}], ",", " ", 
   RowBox[{"PlotRange", " ", "\[Rule]", " ", "All"}]}], "]"}]], "Input",
 CellTags->"b:4.2",
 CellLabel->"In[6]:=",
 CellID->20193],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 223},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:4.2",
 CellLabel->"Out[6]=",
 CellID->41369543]
}, Open  ]],

Cell["\<\
Here is an example of an IVP for a second-order nonlinear ODE whose general \
solution can be obtained in explicit form.\
\>", "MathCaption",
 CellTags->"b:4.2",
 CellID->21497],

Cell[BoxData[
 RowBox[{
  RowBox[{"eqn", " ", "=", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"y", "''"}], "[", "x", "]"}], " ", "-", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"1", "/", "2"}], ")"}], "*", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{"y", "'"}], "[", "x", "]"}], "^", "2"}], "/", 
        RowBox[{"y", "[", "x", "]"}]}], ")"}]}], " ", "+", 
     RowBox[{"1", "/", 
      RowBox[{"(", 
       RowBox[{"2", "*", 
        RowBox[{"y", "[", "x", "]"}]}], ")"}]}]}], "\[Equal]", "0"}]}], 
  ";"}]], "Input",
 CellTags->"b:4.2",
 CellLabel->"In[7]:=",
 CellID->13826],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", "=", 
  RowBox[{"DSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"eqn", ",", " ", 
      RowBox[{
       RowBox[{"y", "[", "0", "]"}], "\[Equal]", "1"}], ",", " ", 
      RowBox[{
       RowBox[{
        RowBox[{"y", "'"}], "[", "0", "]"}], "\[Equal]", "2"}]}], "}"}], ",", 
    " ", "y", ",", "x"}], "]"}]}]], "Input",
 CellTags->"b:4.2",
 CellLabel->"In[8]:=",
 CellID->31276],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    RowBox[{"Function", "[", 
     RowBox[{
      RowBox[{"{", "x", "}"}], ",", 
      RowBox[{
       FractionBox["1", "4"], " ", 
       RowBox[{"(", 
        RowBox[{"4", "+", 
         RowBox[{"8", " ", "x"}], "+", 
         RowBox[{"3", " ", 
          SuperscriptBox["x", "2"]}]}], ")"}]}]}], "]"}]}], "}"}], 
  "}"}]], "Output",
 ImageSize->{258, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:4.2",
 CellLabel->"Out[8]=",
 CellID->24835181]
}, Open  ]],

Cell["\<\
This verifies that the solution satisfies the equation and the initial \
conditions.\
\>", "MathCaption",
 CellTags->"b:4.2",
 CellID->13578],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"eqn", ",", " ", 
     RowBox[{"y", "[", "0", "]"}], ",", " ", 
     RowBox[{
      RowBox[{"y", "'"}], "[", "0", "]"}]}], "}"}], "/.", 
   RowBox[{"sol", "[", 
    RowBox[{"[", "1", "]"}], "]"}]}], "//", "Simplify"}]], "Input",
 CellTags->"b:4.2",
 CellLabel->"In[9]:=",
 CellID->11159],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"True", ",", "1", ",", "2"}], "}"}]], "Output",
 ImageSize->{83, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:4.2",
 CellLabel->"Out[9]=",
 CellID->707166091]
}, Open  ]],

Cell[TextData[{
 "Finally, here is a boundary value problem for a nonlinear second-order ODE. \
The solution is required to satisfy boundary conditions at 0 and infinity. \
The ",
 StyleBox["Solve::ifun", "InlineCode"],
 " message is generated while finding the general solution in terms of ",
 Cell[BoxData[
  ButtonBox["JacobiSN",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/JacobiSN"]], "InlineFormula"],
 ", the inverse of ",
 Cell[BoxData[
  ButtonBox["EllipticF",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/EllipticF"]], "InlineFormula"],
 ". The ",
 StyleBox["DSolve::bvlim", "InlineCode"],
 " messages are given because the limit required for satisfying the condition \
",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    SuperscriptBox[
     StyleBox["y", "TI"], "\[Prime]",
     MultilineFunction->None], "[", 
    ButtonBox["Infinity",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Infinity"], "]"}], "\[Equal]", 
   StyleBox["0", "TR"]}]]],
 " cannot be calculated for either branch of the general solution. However, \
the solution for the boundary value problem is found using an alternative \
method to determine the values of the constants ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["C",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/C"], "[", "1", "]"}]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["C",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/C"], "[", "2", "]"}]], "InlineFormula"],
 " in the general solution."
}], "MathCaption",
 CellTags->"b:4.2",
 CellID->26489],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"generalsolution", "=", 
  RowBox[{"DSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"y", "''"}], "[", "x", "]"}], "/", "2"}], "\[Equal]", 
      RowBox[{
       RowBox[{
        RowBox[{"y", "[", "x", "]"}], "^", "3"}], "-", 
       RowBox[{"y", "[", "x", "]"}]}]}], "}"}], ",", 
    RowBox[{"y", "[", "x", "]"}], ",", "x"}], "]"}]}]], "Input",
 CellTags->"b:4.2",
 CellLabel->"In[10]:=",
 CellID->159],

Cell[BoxData[
 RowBox[{
  RowBox[{"Solve", "::", "\<\"ifun\"\>"}], ":", 
  " ", "\<\"Inverse functions are being used by \\!\\(Solve\\), so some \
solutions may not be found; use Reduce for complete solution information. \\!\
\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/Solve/ifun\\\", \
ButtonNote -> \\\"Solve::ifun\\\"]\\)\"\>"}]], "Message", "MSG",
 CellTags->"b:4.2",
 CellLabel->"During evaluation of In[10]:=",
 CellID->5543618],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{578, 359},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:4.2",
 CellLabel->"Out[10]=",
 CellID->122820998]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 StyleBox[
  RowBox[{"sol", "=", 
   RowBox[{"DSolve", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{"y", "''"}], "[", "x", "]"}], "/", "2"}], " ", "==", " ", 
        RowBox[{
         RowBox[{
          RowBox[{"y", "[", "x", "]"}], "^", "3"}], " ", "-", " ", 
         RowBox[{"y", "[", "x", "]"}]}]}], ",", " ", 
       RowBox[{
        RowBox[{"y", "[", "0", "]"}], " ", "==", " ", "0"}], ",", " ", 
       RowBox[{
        RowBox[{
         RowBox[{"y", "'"}], "[", "Infinity", "]"}], " ", "==", " ", "0"}]}], 
      "}"}], ",", " ", 
     RowBox[{"y", "[", "x", "]"}], ",", " ", "x"}], "]"}], " "}],
  FormatType->StandardForm]], "Input",
 CellTags->"b:4.2",
 CellLabel->"In[11]:=",
 CellID->9072],

Cell[BoxData[
 RowBox[{
  RowBox[{"Solve", "::", "\<\"ifun\"\>"}], ":", 
  " ", "\<\"Inverse functions are being used by \\!\\(Solve\\), so some \
solutions may not be found; use Reduce for complete solution information. \\!\
\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/Solve/ifun\\\", \
ButtonNote -> \\\"Solve::ifun\\\"]\\)\"\>"}]], "Message", "MSG",
 CellTags->"b:4.2",
 CellLabel->"During evaluation of In[11]:=",
 CellID->1197387],

Cell[BoxData[
 RowBox[{
  RowBox[{"DSolve", "::", "\<\"bvlim\"\>"}], ":", 
  " ", "\<\"For some branches of the general solution, unable to compute the \
limit at the given points. Some of the solutions may be lost. \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/DSolve/bvlim\\\", \
ButtonNote -> \\\"DSolve::bvlim\\\"]\\)\"\>"}]], "Message", "MSG",
 CellTags->"b:4.2",
 CellLabel->"During evaluation of In[11]:=",
 CellID->425493595],

Cell[BoxData[
 RowBox[{
  RowBox[{"DSolve", "::", "\<\"bvlim\"\>"}], ":", 
  " ", "\<\"For some branches of the general solution, unable to compute the \
limit at the given points. Some of the solutions may be lost. \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/DSolve/bvlim\\\", \
ButtonNote -> \\\"DSolve::bvlim\\\"]\\)\"\>"}]], "Message", "MSG",
 CellTags->"b:4.2",
 CellLabel->"During evaluation of In[11]:=",
 CellID->413996960],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"y", "[", "x", "]"}], "\[Rule]", 
     RowBox[{"-", 
      RowBox[{"Tanh", "[", 
       SqrtBox[
        SuperscriptBox["x", "2"]], "]"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"y", "[", "x", "]"}], "\[Rule]", 
     RowBox[{"Tanh", "[", 
      SqrtBox[
       SuperscriptBox["x", "2"]], "]"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{300, 27},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:4.2",
 CellLabel->"Out[11]=",
 CellID->931705147]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"y", "[", "x", "]"}], "/.", 
      RowBox[{"sol", "[", 
       RowBox[{"[", "2", "]"}], "]"}]}], ",", "1"}], "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", " ", 
     RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", " ", 
   RowBox[{"PlotRange", " ", "\[Rule]", " ", "All"}]}], "]"}]], "Input",
 CellTags->"b:4.2",
 CellLabel->"In[12]:=",
 CellID->6477],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 124},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:4.2",
 CellLabel->"Out[12]=",
 CellID->400078189]
}, Open  ]],

Cell["\<\
It may not always be possible to obtain a symbolic solution to an IVP or BVP \
for a nonlinear equation. Numerical methods may be necessary in such cases.\
\>", "Text",
 CellTags->"b:4.2",
 CellID->2862],

Cell[CellGroupData[{

Cell["RELATED TUTORIALS", "RelatedTutorialsSection",
 CellID->20345877],

Cell[TextData[ButtonBox["Differential Equation Solving with DSolve",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/DSolveOverview"]], "RelatedTutorials",
 CellID->439091251]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Nonlinear IVPs and BVPs - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, "Overview" -> {"toc" -> Cell[
      BoxData[
       ButtonBox[
        StyleBox[
        "Differential Equation Solving with DSolve", "OverviewNavText"], 
        BaseStyle -> "Link", ButtonData -> "paclet:tutorial/DSolveOverview"]],
       "Text", FontFamily -> "Verdana"], "prevnext" -> Cell[
      TextData[{
        ButtonBox[
         StyleBox["\[FilledLeftTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> "paclet:tutorial/DSolveLinearBVPs"], 
        "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]", 
        ButtonBox[
         StyleBox["\[FilledRightTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> "paclet:tutorial/DSolvePiecewiseBVPs"]}], 
      "Text", FontFamily -> "Verdana"]}, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 34, 41.0790788}", "context" -> "", 
    "keywords" -> {}, "index" -> True, "label" -> "Mathematica Tutorial", 
    "language" -> "en", "paclet" -> "Mathematica", "status" -> "None", 
    "summary" -> 
    "Many real-world applications require the solution of IVPs and BVPs for \
nonlinear ODEs. For example, consider the logistic equation, which occurs in \
population dynamics. This is the logistic equation. The right-hand side of \
the equation can be expanded to a quadratic polynomial in y[t]. Hence, the \
logistic equation is simply a Riccati equation, and its general solution can \
be easily found.", "synonyms" -> {}, "title" -> "Nonlinear IVPs and BVPs", 
    "type" -> "Tutorial", "uri" -> "tutorial/DSolveNonlinearBVPs"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "c:48"->{
  Cell[1280, 43, 86, 2, 70, "Title",
   CellTags->{"c:48", "b:4.2"},
   CellID->28252]},
 "b:4.2"->{
  Cell[1280, 43, 86, 2, 70, "Title",
   CellTags->{"c:48", "b:4.2"},
   CellID->28252],
  Cell[1369, 47, 290, 10, 70, "Text",
   CellTags->"b:4.2",
   CellID->25011],
  Cell[1662, 59, 88, 2, 70, "MathCaption",
   CellTags->"b:4.2",
   CellID->1560],
  Cell[1753, 63, 444, 15, 70, "Input",
   CellTags->"b:4.2",
   CellID->15834],
  Cell[2200, 80, 362, 10, 70, "MathCaption",
   CellTags->"b:4.2",
   CellID->1840],
  Cell[2587, 94, 171, 5, 28, "Input",
   CellTags->"b:4.2",
   CellID->31497],
  Cell[2761, 101, 798, 27, 55, "Output",
   CellTags->"b:4.2",
   CellID->461409372],
  Cell[3574, 131, 741, 24, 70, "MathCaption",
   CellTags->"b:4.2",
   CellID->18529],
  Cell[4340, 159, 522, 17, 70, "Input",
   CellTags->"b:4.2",
   CellID->31540],
  Cell[4865, 178, 517, 10, 34, "Message",
   CellTags->"b:4.2",
   CellID->290779130],
  Cell[5385, 190, 573, 19, 55, "Output",
   CellTags->"b:4.2",
   CellID->256478497],
  Cell[5973, 212, 263, 9, 70, "MathCaption",
   CellTags->"b:4.2",
   CellID->13392],
  Cell[6261, 225, 352, 12, 70, "Input",
   CellTags->"b:4.2",
   CellID->13210],
  Cell[6616, 239, 515, 10, 70, "Message",
   CellTags->"b:4.2",
   CellID->1246817],
  Cell[7134, 251, 623, 20, 54, "Output",
   CellTags->"b:4.2",
   CellID->69060035],
  Cell[7772, 274, 150, 5, 70, "MathCaption",
   CellTags->"b:4.2",
   CellID->26979],
  Cell[7947, 283, 296, 10, 70, "Input",
   CellTags->"b:4.2",
   CellID->30971],
  Cell[8246, 295, 250, 9, 36, "Output",
   CellTags->"b:4.2",
   CellID->282737580],
  Cell[8511, 307, 224, 8, 70, "MathCaption",
   CellTags->"b:4.2",
   CellID->73],
  Cell[8760, 319, 953, 28, 65, "Input",
   CellTags->"b:4.2",
   CellID->20193],
  Cell[9716, 349, 8194, 139, 244, "Output",
   Evaluatable->False,
   CellTags->"b:4.2",
   CellID->41369543],
  Cell[17925, 491, 187, 5, 70, "MathCaption",
   CellTags->"b:4.2",
   CellID->21497],
  Cell[18115, 498, 641, 23, 70, "Input",
   CellTags->"b:4.2",
   CellID->13826],
  Cell[18781, 525, 418, 14, 70, "Input",
   CellTags->"b:4.2",
   CellID->31276],
  Cell[19202, 541, 558, 20, 51, "Output",
   CellTags->"b:4.2",
   CellID->24835181],
  Cell[19775, 564, 151, 5, 70, "MathCaption",
   CellTags->"b:4.2",
   CellID->13578],
  Cell[19951, 573, 351, 12, 70, "Input",
   CellTags->"b:4.2",
   CellID->11159],
  Cell[20305, 587, 236, 8, 36, "Output",
   CellTags->"b:4.2",
   CellID->707166091],
  Cell[20556, 598, 1534, 46, 70, "MathCaption",
   CellTags->"b:4.2",
   CellID->26489],
  Cell[22115, 648, 483, 16, 70, "Input",
   CellTags->"b:4.2",
   CellID->159],
  Cell[22601, 666, 516, 10, 34, "Message",
   CellTags->"b:4.2",
   CellID->5543618],
  Cell[23120, 678, 5134, 89, 380, "Output",
   Evaluatable->False,
   CellTags->"b:4.2",
   CellID->122820998],
  Cell[28291, 772, 794, 25, 70, "Input",
   CellTags->"b:4.2",
   CellID->9072],
  Cell[29088, 799, 516, 10, 70, "Message",
   CellTags->"b:4.2",
   CellID->1197387],
  Cell[29607, 811, 517, 10, 70, "Message",
   CellTags->"b:4.2",
   CellID->425493595],
  Cell[30127, 823, 517, 10, 70, "Message",
   CellTags->"b:4.2",
   CellID->413996960],
  Cell[30647, 835, 579, 21, 48, "Output",
   CellTags->"b:4.2",
   CellID->931705147],
  Cell[31263, 861, 464, 15, 28, "Input",
   CellTags->"b:4.2",
   CellID->6477],
  Cell[31730, 878, 2844, 51, 145, "Output",
   Evaluatable->False,
   CellTags->"b:4.2",
   CellID->400078189],
  Cell[34589, 932, 213, 5, 70, "Text",
   CellTags->"b:4.2",
   CellID->2862]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"c:48", 37506, 1005},
 {"b:4.2", 37607, 1009}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 655, 16, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[1280, 43, 86, 2, 70, "Title",
 CellTags->{"c:48", "b:4.2"},
 CellID->28252],
Cell[1369, 47, 290, 10, 70, "Text",
 CellTags->"b:4.2",
 CellID->25011],
Cell[1662, 59, 88, 2, 70, "MathCaption",
 CellTags->"b:4.2",
 CellID->1560],
Cell[1753, 63, 444, 15, 70, "Input",
 CellTags->"b:4.2",
 CellID->15834],
Cell[2200, 80, 362, 10, 70, "MathCaption",
 CellTags->"b:4.2",
 CellID->1840],
Cell[CellGroupData[{
Cell[2587, 94, 171, 5, 28, "Input",
 CellTags->"b:4.2",
 CellID->31497],
Cell[2761, 101, 798, 27, 55, "Output",
 CellTags->"b:4.2",
 CellID->461409372]
}, Open  ]],
Cell[3574, 131, 741, 24, 70, "MathCaption",
 CellTags->"b:4.2",
 CellID->18529],
Cell[CellGroupData[{
Cell[4340, 159, 522, 17, 70, "Input",
 CellTags->"b:4.2",
 CellID->31540],
Cell[4865, 178, 517, 10, 34, "Message",
 CellTags->"b:4.2",
 CellID->290779130],
Cell[5385, 190, 573, 19, 55, "Output",
 CellTags->"b:4.2",
 CellID->256478497]
}, Open  ]],
Cell[5973, 212, 263, 9, 70, "MathCaption",
 CellTags->"b:4.2",
 CellID->13392],
Cell[CellGroupData[{
Cell[6261, 225, 352, 12, 70, "Input",
 CellTags->"b:4.2",
 CellID->13210],
Cell[6616, 239, 515, 10, 70, "Message",
 CellTags->"b:4.2",
 CellID->1246817],
Cell[7134, 251, 623, 20, 54, "Output",
 CellTags->"b:4.2",
 CellID->69060035]
}, Open  ]],
Cell[7772, 274, 150, 5, 70, "MathCaption",
 CellTags->"b:4.2",
 CellID->26979],
Cell[CellGroupData[{
Cell[7947, 283, 296, 10, 70, "Input",
 CellTags->"b:4.2",
 CellID->30971],
Cell[8246, 295, 250, 9, 36, "Output",
 CellTags->"b:4.2",
 CellID->282737580]
}, Open  ]],
Cell[8511, 307, 224, 8, 70, "MathCaption",
 CellTags->"b:4.2",
 CellID->73],
Cell[CellGroupData[{
Cell[8760, 319, 953, 28, 65, "Input",
 CellTags->"b:4.2",
 CellID->20193],
Cell[9716, 349, 8194, 139, 244, "Output",
 Evaluatable->False,
 CellTags->"b:4.2",
 CellID->41369543]
}, Open  ]],
Cell[17925, 491, 187, 5, 70, "MathCaption",
 CellTags->"b:4.2",
 CellID->21497],
Cell[18115, 498, 641, 23, 70, "Input",
 CellTags->"b:4.2",
 CellID->13826],
Cell[CellGroupData[{
Cell[18781, 525, 418, 14, 70, "Input",
 CellTags->"b:4.2",
 CellID->31276],
Cell[19202, 541, 558, 20, 51, "Output",
 CellTags->"b:4.2",
 CellID->24835181]
}, Open  ]],
Cell[19775, 564, 151, 5, 70, "MathCaption",
 CellTags->"b:4.2",
 CellID->13578],
Cell[CellGroupData[{
Cell[19951, 573, 351, 12, 70, "Input",
 CellTags->"b:4.2",
 CellID->11159],
Cell[20305, 587, 236, 8, 36, "Output",
 CellTags->"b:4.2",
 CellID->707166091]
}, Open  ]],
Cell[20556, 598, 1534, 46, 70, "MathCaption",
 CellTags->"b:4.2",
 CellID->26489],
Cell[CellGroupData[{
Cell[22115, 648, 483, 16, 70, "Input",
 CellTags->"b:4.2",
 CellID->159],
Cell[22601, 666, 516, 10, 34, "Message",
 CellTags->"b:4.2",
 CellID->5543618],
Cell[23120, 678, 5134, 89, 380, "Output",
 Evaluatable->False,
 CellTags->"b:4.2",
 CellID->122820998]
}, Open  ]],
Cell[CellGroupData[{
Cell[28291, 772, 794, 25, 70, "Input",
 CellTags->"b:4.2",
 CellID->9072],
Cell[29088, 799, 516, 10, 70, "Message",
 CellTags->"b:4.2",
 CellID->1197387],
Cell[29607, 811, 517, 10, 70, "Message",
 CellTags->"b:4.2",
 CellID->425493595],
Cell[30127, 823, 517, 10, 70, "Message",
 CellTags->"b:4.2",
 CellID->413996960],
Cell[30647, 835, 579, 21, 48, "Output",
 CellTags->"b:4.2",
 CellID->931705147]
}, Open  ]],
Cell[CellGroupData[{
Cell[31263, 861, 464, 15, 28, "Input",
 CellTags->"b:4.2",
 CellID->6477],
Cell[31730, 878, 2844, 51, 145, "Output",
 Evaluatable->False,
 CellTags->"b:4.2",
 CellID->400078189]
}, Open  ]],
Cell[34589, 932, 213, 5, 70, "Text",
 CellTags->"b:4.2",
 CellID->2862],
Cell[CellGroupData[{
Cell[34827, 941, 71, 1, 70, "RelatedTutorialsSection",
 CellID->20345877],
Cell[34901, 944, 177, 3, 70, "RelatedTutorials",
 CellID->439091251]
}, Open  ]]
}, Open  ]],
Cell[35105, 951, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

