(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     49009,       1389]
NotebookOptionsPosition[     38839,       1032]
NotebookOutlinePosition[     45165,       1231]
CellTagsIndexPosition[     45072,       1225]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], Cell[TextData[Cell[
    BoxData[
     PopupMenuBox[
      Dynamic[{"ActionMenu", None}, 
       Part[{"Differential Equation Solving with DSolve" :> 
         Documentation`HelpLookup["paclet:tutorial/DSolveOverview"]}, #, 2]& ,
        Evaluator -> Automatic], {
      1->"\<\"Differential Equation Solving with DSolve\"\>"}, "\<\"related \
tutorials\"\>", 
      StyleBox["\<\"related tutorials\"\>",
       Background->Automatic],
      Appearance->Automatic,
      ImageSize->Automatic,
      MenuAppearance->"Menu"]],
     FontSize->9]], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["IVPs with Piecewise Coefficients", "Title",
 CellTags->{"c:49", "b:4.3"},
 CellID->10635],

Cell[TextData[{
 "The differential equations that arise in modern applications often have \
discontinuous coefficients. ",
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 " can handle a wide variety of such ",
 StyleBox["ODEs with piecewise coefficients.",
  FontSlant->"Italic"],
 " Some of the functions used in these equations are ",
 Cell[BoxData[
  ButtonBox["UnitStep",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/UnitStep"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["Max",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Max"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["Min",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Min"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["Sign",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Sign"]], "InlineFormula"],
 ", and ",
 Cell[BoxData[
  ButtonBox["Abs",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Abs"]], "InlineFormula"],
 ". These functions and combinations of them can be converted into ",
 Cell[BoxData[
  ButtonBox["Piecewise",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Piecewise"]], "InlineFormula"],
 " objects. "
}], "Text",
 CellTags->"b:4.3",
 CellID->1643],

Cell[TextData[{
 "This converts the given expression into a ",
 Cell[BoxData[
  ButtonBox["Piecewise",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Piecewise"]], "InlineFormula"],
 " expression."
}], "MathCaption",
 CellTags->"b:4.3",
 CellID->28888],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PiecewiseExpand", "[", 
  RowBox[{
   RowBox[{"UnitStep", "[", "x", "]"}], "+", 
   RowBox[{"Max", "[", 
    RowBox[{"x", ",", 
     RowBox[{"x", "^", "2"}]}], "]"}]}], "]"}]], "Input",
 CellTags->"b:4.3",
 CellLabel->"In[1]:=",
 CellID->8153],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzNVrFuwjAQNSQmhJYKCSh0S4cWVZ26dGy3durQ/kGKKoWhakQZ2PIJjHxC
Rj6hn8Bn8Bmu7dhgLgc4jSphCV98d753vjufeQkn0cdnOBkNw+B5HMbRaPgd
PH2NOcupEEIi/rvrE/7N+Jf8RYxxas6OmN3pdPojx2q1Cjd8J0kSJgep8Yly
E57kC1sZX+hV+bKqaIdTIXaUvpZnBrMhdF1OeyhSHSBARFdOmzUh54hP+9ba
X5N3YeULVaei6lQ9G2S6XqsQ/A0bxqG8D/SfcyC8c3dHIMe186XJtitOUCFu
AT0tz1yQA+x01M6+FW4X8WM7BjCvN6hdT6H3gfYpnzy+vEV3XQLtB1QrMLTE
gR9L2Ho/Wm7RXqT1KZDr26zlMn8qi/P5PE1T8ekDlMN3QeDmqxvaye9rAI0a
WJv+baIB7cIe1UW90VVYV9Q36C7MOI5ns5lNTPZja2lDUe3LCYpdvjfh+dCo
Xk7fjElxf2B/aqPo0DasXSjHa+GQL9co9sDKlo6P6GKO2rXu6EfZF3AuvPVX
ByJN2b5XQeZhq1oGOY6uHw+NPv7KmdE274RfwobPivthcz7Yp/A3HrcPs6Ff
7wDwWwVsNoE/9wX2niF7QfbdxWKRyLFcLiHWq9RrG1yB9SZpB9XN/neTyi/T
aUAs\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{104, 49},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:4.3",
 CellLabel->"Out[1]=",
 CellID->52740304]
}, Open  ]],

Cell[TextData[{
 "Here is the general solution to a first-order ODE that contains ",
 Cell[BoxData[
  ButtonBox["UnitStep",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/UnitStep"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellTags->"b:4.3",
 CellID->4747],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"y", "'"}], "[", "x", "]"}], "\[Equal]", 
    RowBox[{"UnitStep", "[", "x", "]"}]}], ",", " ", "y", ",", " ", "x"}], 
  "]"}]], "Input",
 CellTags->"b:4.3",
 CellLabel->"In[2]:=",
 CellID->28680],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    RowBox[{"Function", "[", 
     RowBox[{
      RowBox[{"{", "x", "}"}], ",", 
      RowBox[{
       RowBox[{"C", "[", "1", "]"}], "+", 
       RowBox[{"x", " ", 
        RowBox[{"UnitStep", "[", "x", "]"}]}]}]}], "]"}]}], "}"}], 
  "}"}]], "Output",
 ImageSize->{285, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:4.3",
 CellLabel->"Out[2]=",
 CellID->9472051]
}, Open  ]],

Cell["\<\
Here is the solution to the same ODE with an initial condition.\
\>", "MathCaption",
 CellTags->"b:4.3",
 CellID->25678],

Cell[BoxData[
 RowBox[{
  RowBox[{"eqn", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"y", "'"}], "[", "x", "]"}], "\[Equal]", 
      RowBox[{"UnitStep", "[", "x", "]"}]}], ",", " ", 
     RowBox[{
      RowBox[{"y", "[", "0", "]"}], "\[Equal]", "1"}]}], "}"}]}], 
  ";"}]], "Input",
 CellTags->"b:4.3",
 CellLabel->"In[3]:=",
 CellID->6450],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", "=", 
  RowBox[{"DSolve", "[", 
   RowBox[{"eqn", ",", " ", "y", ",", " ", "x"}], "]"}]}]], "Input",
 CellTags->"b:4.3",
 CellLabel->"In[4]:=",
 CellID->14801],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    RowBox[{"Function", "[", 
     RowBox[{
      RowBox[{"{", "x", "}"}], ",", 
      RowBox[{"1", "+", 
       RowBox[{"x", " ", 
        RowBox[{"UnitStep", "[", "x", "]"}]}]}]}], "]"}]}], "}"}], 
  "}"}]], "Output",
 ImageSize->{264, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:4.3",
 CellLabel->"Out[4]=",
 CellID->151002502]
}, Open  ]],

Cell[TextData[{
 "The solution can be plotted in the usual way. Note that the solution is \
continuous but not differentiable at ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "=", "0"}], TraditionalForm]], "InlineMath"],
 "."
}], "MathCaption",
 CellTags->"b:4.3",
 CellID->14199],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"y", "[", "x", "]"}], "/.", "sol"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "3"}], ",", "3"}], "}"}]}], "]"}]], "Input",
 CellTags->"b:4.3",
 CellLabel->"In[5]:=",
 CellID->20855],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 235},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:4.3",
 CellLabel->"Out[5]=",
 CellID->727978110]
}, Open  ]],

Cell["This verifies the solution.", "MathCaption",
 CellTags->"b:4.3",
 CellID->5716],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Simplify", "[", 
  RowBox[{
   RowBox[{"eqn", "/.", 
    RowBox[{"sol", "[", 
     RowBox[{"[", "1", "]"}], "]"}]}], ",", 
   RowBox[{
    RowBox[{"x", " ", ">", "0"}], " ", "||", " ", 
    RowBox[{"x", " ", "<", "0"}]}]}], "]"}]], "Input",
 CellTags->"b:4.3",
 CellLabel->"In[6]:=",
 CellID->10573],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"True", ",", "True"}], "}"}]], "Output",
 ImageSize->{86, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:4.3",
 CellLabel->"Out[6]=",
 CellID->457850758]
}, Open  ]],

Cell[TextData[{
 "Here is a piecewise ODE that has ",
 Cell[BoxData[
  ButtonBox["Max",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Max"]], "InlineFormula"],
 " in its coefficients."
}], "MathCaption",
 CellTags->"b:4.3",
 CellID->23245],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", "=", 
  RowBox[{"DSolve", "[", " ", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"y", "'"}], "[", "x", "]"}], " ", "+", " ", 
        RowBox[{
         RowBox[{"Max", "[", 
          RowBox[{"x", ",", "1"}], "]"}], " ", 
         RowBox[{"y", "[", "x", "]"}]}]}], " ", "\[Equal]", " ", "0"}], ",", 
      " ", 
      RowBox[{
       RowBox[{"y", "[", "0", "]"}], "\[Equal]", "1"}]}], "}"}], ",", " ", 
    RowBox[{"y", "[", "x", "]"}], ",", " ", "x"}], " ", "]"}]}]], "Input",
 CellTags->"b:4.3",
 CellLabel->"In[7]:=",
 CellID->6473],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzdV71OIzEQdvaHTQg5/tGVEeLuxP9PBQUtFIgGdA8QISQoEBFHkS6PkDI1
Vco8Ao+Qx0iZRzD2eJzdHTuxs0JCnKX1eDz2zGfPeOy9brw+3D81Xh/vGvXL
l0bz4fHuX/3i+UV0hSXG2Jv4tn8y0eaiBd8b54Jm63NZR61W6x3KcDiMRU8s
Bu8LGgoaIi9pwLDBVAlEMxJ0KdUVttttDoXFwEu7asJxhoemgw/QQrZvxctS
FbHXkK76WI7HPGopZls2y2RmjfDF8cTeu03RJw5MKxYM0gPR5L0xeu3I5riK
kvkZEQSIQEGAgj1aIjHKCctWu9ldZhgNWR7QRqk9c882rXrXHOtYFFUi2C2Y
HSHWGOkR0aZWPRgM2lBYICvJ5/yayhOQdzodlP8h+mG8EVER4SmKGvoIJo5G
IzAClWC0DiIvg7zf76N822EzKITs7zfSKn0pD0VFbxf2m7mvmD2db7Wvq0g3
CA4aofM47uRTULj4KuHXvh5d6guezyB0NM1RiQM7eDrOeloP2PdaDc04dFTu
VNPZuQijI6gXlLzb7fZ6PdlcJ/IfhD9wICsps2G6aoqA3khUrm+YOYulyrg/
v/eBzvYOpPIBJBGWvHaZxIglLiZhte9ys9lUCRqyc1buigqF1ycPmLevddTU
GHFF+y/iQb+YtqLPncBwDFeX0Bgz7ZTSjPKbyPeK45ziW1de+5w7Ref0Sbn9
O92H/59W12vultteY7YXsTy+uzwf5/rfS8v9XvqHnJ5zeq5p1vLTe4SoKhk6
uxZ9F0oNCU9je3ZNNIvvGOsuz4RU/5eUDT114heqb8p+RuI9rF7q4sm+gBa0
njPrPB0zp0jVf+aNgeoKiPpvZ6UPOIyIkA==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{161, 41},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:4.3",
 CellLabel->"Out[7]=",
 CellID->347171023]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"y", "[", "x", "]"}], "/.", "sol"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "3"}], ",", "3"}], "}"}]}], "]"}]], "Input",
 CellTags->"b:4.3",
 CellLabel->"In[8]:=",
 CellID->1550],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 234},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:4.3",
 CellLabel->"Out[8]=",
 CellID->32291927]
}, Open  ]],

Cell["\<\
A piecewise ODE can be thought of as a collection of ODEs over disjoint \
intervals such that the expressions for the coefficients and the boundary \
conditions change from one interval to another. Thus, different intervals \
have different solutions, and the final solution for the ODE is obtained by \
patching together the solutions over the different intervals.\
\>", "Text",
 CellTags->"b:4.3",
 CellID->1181],

Cell[TextData[{
 "For this piecewise ODE, the expression for ",
 Cell[BoxData["FinalSol"], "InlineFormula"],
 " is obtained by patching together ",
 Cell[BoxData["SolFromMinusInfinityToTwo"], "InlineFormula"],
 " and ",
 Cell[BoxData["SolFromTwoToInfinity"], "InlineFormula"],
 ". The boundary condition for the interval ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", 
      ButtonBox["Infinity",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/Infinity"]}], ",", "2"}]}], "]"}]], 
  "InlineFormula"],
 " is simply ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"y", "[", "0", "]"}], "=", "1"}], TraditionalForm]], 
  "InlineMath"],
 ", while the initial condition for the interval ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"[", 
    RowBox[{"2", ",", 
     ButtonBox["Infinity",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Infinity"]}]}], ")"}]], "InlineFormula"],
 " is ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"y", "[", "2", "]"}], "\[LongEqual]", 
    SuperscriptBox["\[ExponentialE]", "2"]}], TraditionalForm]], 
  "InlineMath"],
 " (given by the final value for the solution over the first interval)."
}], "MathCaption",
 CellTags->"b:4.3",
 CellID->4287],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinalSol", " ", "=", 
  RowBox[{"DSolve", "[", " ", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"y", "'"}], "[", "t", "]"}], "\[Equal]", 
       RowBox[{"If", "[", " ", 
        RowBox[{
         RowBox[{"t", "\[LessEqual]", "2"}], ",", " ", 
         RowBox[{"y", "[", "t", "]"}], ",", " ", 
         RowBox[{
          RowBox[{"-", 
           RowBox[{"y", "[", "t", "]"}]}], "/", "2"}]}], "]"}]}], " ", ",", 
      " ", 
      RowBox[{
       RowBox[{"y", "[", "0", "]"}], "\[Equal]", "1"}]}], "}"}], ",", " ", 
    "y", ",", " ", "t"}], "]"}]}]], "Input",
 CellTags->"b:4.3",
 CellLabel->"In[9]:=",
 CellID->2699],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWDtvUzEUduJ782hJX1JpmgGCSoWQQGKCf8BD0KkbTFGF1A4IVLpn5Ccw
ZsyYMSM/IWN+QX+H8fGjcb57cu0GUjJwpVz7HPu8Pvue4/ikd3X++Uvv6uKs
131z2ft2fnH2vfv666VmyYoQ4lr/nrSF7ivdM79rpXRr3z/onQ0Gg1/mmU6n
H2d8qfnKPOITy63pV65VdnUrdSsdTW1VuI6wT1V3M90esJpyQ5N3VmDHadwN
+KYboavOUsg7TLYYSiVZzm9op2U529TdAkn0J91+nowyensvwQdCOFsce4HL
e0I7J4OoBbPeVWfRmjSP4/gR8okE2mZC5ri5a09dewz2D8FOi17ZjC5isx/E
HlowmBZWJwP6FOjH4E/Laevotu6jKdGHdPWveLGN8v4b1qnB71MhNeHGn/4T
L/FLebUWWL1ciRc+a54APwe6GdHTBO9bwTgtKo3T/nu+kiiQ/hAZ3wTa550N
iAJ3Ql3hV/zsTuLZYFZHBjkirINyPpu5TONnyyCHy4AuSqX4KL1QybwGYIqZ
Z7kdgdkTZ3lksOLgvl5Y3xOrD18lC1pvoTEWf5pFX0Us+uPxeDgcevQz5jtE
XNBqxdpx+4yTwNpPdFO3tST/SxAq8RLpP8E5tt8xDpvhFld8RBjlST8doAlZ
i25sTxt6LvtgHiuc+9YU6xj2aGvLY93v96n7LjKf370ptbDoO+ZeWkdZiu1N
DplbK35+PRlV7p9HbMfEcN32uDr6bUTfLapZFsZQrC57gQxGyJ/HiZGJWfXi
qhrWwti+8LqQv0JcArqjuLU+4Gat5xoHNGW/miYfAn9PlXuH58qdO/E2dvb+
b3XVVv3NziM1+4qJ9vyYF7xXqf/OsU4eKe5LbHA22ZPX4v/tsRuSI7DRUOG5
bcZvJ2mjbhcka04jnhWWu8F5sMBfrJL3We3+VsZj1XGrjVkAo99ntRXvVDBX
7rIrxkfeAtkXKrz3sfeYk8mkb57RaLTpYnlvxn+CtOXaO1BR+Q2nWmMG\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{229, 39},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:4.3",
 CellLabel->"Out[9]=",
 CellID->186644743]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SolFromMinusInfinityToTwo", "=", 
  RowBox[{"DSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"y", "'"}], "[", "t", "]"}], "\[Equal]", 
       RowBox[{"y", "[", "t", "]"}]}], ",", 
      RowBox[{
       RowBox[{"y", "[", "0", "]"}], "\[Equal]", "1"}]}], "}"}], ",", "y", 
    ",", "t"}], "]"}]}]], "Input",
 CellTags->"b:4.3",
 CellLabel->"In[10]:=",
 CellID->18643],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    RowBox[{"Function", "[", 
     RowBox[{
      RowBox[{"{", "t", "}"}], ",", 
      SuperscriptBox["\[ExponentialE]", "t"]}], "]"}]}], "}"}], 
  "}"}]], "Output",
 ImageSize->{170, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:4.3",
 CellLabel->"Out[10]=",
 CellID->689319669]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SolFromTwoToInfinity", "=", 
  RowBox[{"DSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"y", "'"}], "[", "t", "]"}], "\[Equal]", 
       RowBox[{
        RowBox[{"-", 
         RowBox[{"y", "[", "t", "]"}]}], "/", "2"}]}], ",", 
      RowBox[{
       RowBox[{"y", "[", "2", "]"}], "\[Equal]", 
       RowBox[{"E", "^", "2"}]}]}], "}"}], ",", "y", ",", "t"}], 
   "]"}]}]], "Input",
 CellTags->"b:4.3",
 CellLabel->"In[11]:=",
 CellID->7172],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    RowBox[{"Function", "[", 
     RowBox[{
      RowBox[{"{", "t", "}"}], ",", 
      SuperscriptBox["\[ExponentialE]", 
       RowBox[{"3", "-", 
        FractionBox["t", "2"]}]]}], "]"}]}], "}"}], "}"}]], "Output",
 ImageSize->{183, 26},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:4.3",
 CellLabel->"Out[11]=",
 CellID->96686323]
}, Open  ]],

Cell[TextData[{
 "If there are a large number of discontinuities in a problem, it is \
convenient to use ",
 Cell[BoxData[
  ButtonBox["Piecewise",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Piecewise"]], "InlineFormula"],
 " directly in the formulation of the problem."
}], "Text",
 CellTags->"b:4.3",
 CellID->9981],

Cell[TextData[{
 "This second-order ODE contains a ",
 Cell[BoxData[
  ButtonBox["Piecewise",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Piecewise"]], "InlineFormula"],
 " term."
}], "MathCaption",
 CellTags->"b:4.3",
 CellID->2112],

Cell[BoxData[
 RowBox[{
  RowBox[{"eqn", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"y", "''"}], "[", "t", "]"}], "+", 
       RowBox[{"y", "[", "t", "]"}]}], "\[Equal]", 
      RowBox[{"Piecewise", "[", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{"-", "1"}], ",", 
           RowBox[{"t", "<", "0"}]}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"1", ",", 
           RowBox[{"t", "<", "1"}]}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"Sin", "[", "t", "]"}], ",", 
           RowBox[{"t", "<", "2"}]}], "}"}]}], "}"}], "]"}]}], ",", 
     RowBox[{
      RowBox[{"y", "[", "0", "]"}], "\[Equal]", "1"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"y", "'"}], "[", "0", "]"}], "\[Equal]", "1"}]}], "}"}]}], 
  ";"}]], "Input",
 CellTags->"b:4.3",
 CellLabel->"In[12]:=",
 CellID->13476],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", "=", 
  RowBox[{"DSolve", "[", " ", 
   RowBox[{"eqn", ",", " ", "y", ",", " ", "t"}], " ", "]"}]}]], "Input",
 CellTags->"b:4.3",
 CellLabel->"In[13]:=",
 CellID->4381],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{633, 273},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:4.3",
 CellLabel->"Out[13]=",
 CellID->1246453]
}, Open  ]],

Cell[TextData[{
 "This ODE contains the ",
 Cell[BoxData[
  ButtonBox["Clip",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Clip"]], "InlineFormula"],
 " function. The solutions are given in terms of Airy functions."
}], "MathCaption",
 CellTags->"b:4.3",
 CellID->26890],

Cell[BoxData[
 RowBox[{
  RowBox[{"eqn", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"y", "''"}], "[", "x", "]"}], " ", "-", 
       RowBox[{
        RowBox[{"Clip", "[", "x", "]"}], "*", 
        RowBox[{"y", "[", "x", "]"}]}]}], "\[Equal]", "0"}], ",", 
     RowBox[{
      RowBox[{"y", "[", "0", "]"}], "\[Equal]", "0"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"y", "'"}], "[", "0", "]"}], "\[Equal]", 
      RowBox[{"-", "1"}]}]}], "}"}]}], ";"}]], "Input",
 CellTags->"b:4.3",
 CellLabel->"In[14]:=",
 CellID->16671],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{"eqn", ",", " ", "y", ",", "x"}], "]"}]], "Input",
 CellTags->"b:4.3",
 CellLabel->"In[15]:=",
 CellID->8805],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{633, 327},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:4.3",
 CellLabel->"Out[15]=",
 CellID->7177450]
}, Open  ]],

Cell["\<\
This concludes the discussion of different types of boundary value problems \
and their solutions.\
\>", "Text",
 CellTags->"b:4.3",
 CellID->30032],

Cell[TextData[ButtonBox["Differential Equation Solving with DSolve",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/DSolveOverview"]], "RelatedTutorials",
 CellID->439091251]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"IVPs with Piecewise Coefficients - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, "Overview" -> {"toc" -> Cell[
      BoxData[
       ButtonBox[
        StyleBox[
        "Differential Equation Solving with DSolve", "OverviewNavText"], 
        BaseStyle -> "Link", ButtonData -> "paclet:tutorial/DSolveOverview"]],
       "Text", FontFamily -> "Verdana"], "prevnext" -> Cell[
      TextData[{
        ButtonBox[
         StyleBox["\[FilledLeftTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> "paclet:tutorial/DSolveNonlinearBVPs"], 
        "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]", 
        ButtonBox[
         StyleBox["\[FilledRightTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> 
         "paclet:tutorial/WorkingWithDSolveIntroduction"]}], "Text", 
      FontFamily -> "Verdana"]}, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 34, 43.2978856}", "context" -> "", 
    "keywords" -> {}, "index" -> True, "label" -> "Mathematica Tutorial", 
    "language" -> "en", "paclet" -> "Mathematica", "status" -> "None", 
    "summary" -> 
    "The differential equations that arise in modern applications often have \
discontinuous coefficients. DSolve can handle a wide variety of such ODEs \
with piecewise coefficients. Some of the functions used in these equations \
are UnitStep, Max, Min, Sign, and Abs. These functions and combinations of \
them can be converted into Piecewise objects. This converts the given \
expression into a Piecewise expression. Here is the general solution to a \
first-order ODE that contains UnitStep.", "synonyms" -> {}, "title" -> 
    "IVPs with Piecewise Coefficients", "type" -> "Tutorial", "uri" -> 
    "tutorial/DSolvePiecewiseBVPs"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "c:49"->{
  Cell[1280, 43, 95, 2, 70, "Title",
   CellTags->{"c:49", "b:4.3"},
   CellID->10635]},
 "b:4.3"->{
  Cell[1280, 43, 95, 2, 70, "Title",
   CellTags->{"c:49", "b:4.3"},
   CellID->10635],
  Cell[1378, 47, 1245, 43, 70, "Text",
   CellTags->"b:4.3",
   CellID->1643],
  Cell[2626, 92, 254, 9, 70, "MathCaption",
   CellTags->"b:4.3",
   CellID->28888],
  Cell[2905, 105, 267, 9, 28, "Input",
   CellTags->"b:4.3",
   CellID->8153],
  Cell[3175, 116, 906, 20, 70, "Output",
   Evaluatable->False,
   CellTags->"b:4.3",
   CellID->52740304],
  Cell[4096, 139, 262, 9, 70, "MathCaption",
   CellTags->"b:4.3",
   CellID->4747],
  Cell[4383, 152, 283, 10, 28, "Input",
   CellTags->"b:4.3",
   CellID->28680],
  Cell[4669, 164, 474, 17, 36, "Output",
   CellTags->"b:4.3",
   CellID->9472051],
  Cell[5158, 184, 130, 4, 70, "MathCaption",
   CellTags->"b:4.3",
   CellID->25678],
  Cell[5291, 190, 376, 14, 70, "Input",
   CellTags->"b:4.3",
   CellID->6450],
  Cell[5692, 208, 189, 6, 28, "Input",
   CellTags->"b:4.3",
   CellID->14801],
  Cell[5884, 216, 443, 16, 36, "Output",
   CellTags->"b:4.3",
   CellID->151002502],
  Cell[6342, 235, 279, 9, 70, "MathCaption",
   CellTags->"b:4.3",
   CellID->14199],
  Cell[6646, 248, 281, 10, 28, "Input",
   CellTags->"b:4.3",
   CellID->20855],
  Cell[6930, 260, 3156, 56, 256, "Output",
   Evaluatable->False,
   CellTags->"b:4.3",
   CellID->727978110],
  Cell[10101, 319, 85, 2, 70, "MathCaption",
   CellTags->"b:4.3",
   CellID->5716],
  Cell[10211, 325, 323, 11, 70, "Input",
   CellTags->"b:4.3",
   CellID->10573],
  Cell[10537, 338, 229, 8, 36, "Output",
   CellTags->"b:4.3",
   CellID->457850758],
  Cell[10781, 349, 242, 9, 70, "MathCaption",
   CellTags->"b:4.3",
   CellID->23245],
  Cell[11048, 362, 629, 20, 28, "Input",
   CellTags->"b:4.3",
   CellID->6473],
  Cell[11680, 384, 1122, 23, 62, "Output",
   Evaluatable->False,
   CellTags->"b:4.3",
   CellID->347171023],
  Cell[12839, 412, 280, 10, 28, "Input",
   CellTags->"b:4.3",
   CellID->1550],
  Cell[13122, 424, 3481, 62, 255, "Output",
   Evaluatable->False,
   CellTags->"b:4.3",
   CellID->32291927],
  Cell[16618, 489, 424, 8, 70, "Text",
   CellTags->"b:4.3",
   CellID->1181],
  Cell[17045, 499, 1227, 41, 70, "MathCaption",
   CellTags->"b:4.3",
   CellID->4287],
  Cell[18297, 544, 689, 22, 28, "Input",
   CellTags->"b:4.3",
   CellID->2699],
  Cell[18989, 568, 1264, 25, 60, "Output",
   Evaluatable->False,
   CellTags->"b:4.3",
   CellID->186644743],
  Cell[20290, 598, 447, 15, 70, "Input",
   CellTags->"b:4.3",
   CellID->18643],
  Cell[20740, 615, 390, 14, 39, "Output",
   CellTags->"b:4.3",
   CellID->689319669],
  Cell[21167, 634, 522, 18, 70, "Input",
   CellTags->"b:4.3",
   CellID->7172],
  Cell[21692, 654, 441, 15, 47, "Output",
   CellTags->"b:4.3",
   CellID->96686323],
  Cell[22148, 672, 323, 10, 70, "Text",
   CellTags->"b:4.3",
   CellID->9981],
  Cell[22474, 684, 238, 9, 70, "MathCaption",
   CellTags->"b:4.3",
   CellID->2112],
  Cell[22715, 695, 957, 32, 70, "Input",
   CellTags->"b:4.3",
   CellID->13476],
  Cell[23697, 731, 199, 6, 28, "Input",
   CellTags->"b:4.3",
   CellID->4381],
  Cell[23899, 739, 6136, 105, 294, "Output",
   Evaluatable->False,
   CellTags->"b:4.3",
   CellID->1246453],
  Cell[30050, 847, 274, 9, 70, "MathCaption",
   CellTags->"b:4.3",
   CellID->26890],
  Cell[30327, 858, 595, 20, 70, "Input",
   CellTags->"b:4.3",
   CellID->16671],
  Cell[30947, 882, 158, 5, 28, "Input",
   CellTags->"b:4.3",
   CellID->8805],
  Cell[31108, 889, 7336, 125, 348, "Output",
   Evaluatable->False,
   CellTags->"b:4.3",
   CellID->7177450],
  Cell[38459, 1017, 158, 5, 70, "Text",
   CellTags->"b:4.3",
   CellID->30032]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"c:49", 41349, 1087},
 {"b:4.3", 41450, 1091}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 655, 16, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[1280, 43, 95, 2, 70, "Title",
 CellTags->{"c:49", "b:4.3"},
 CellID->10635],
Cell[1378, 47, 1245, 43, 70, "Text",
 CellTags->"b:4.3",
 CellID->1643],
Cell[2626, 92, 254, 9, 70, "MathCaption",
 CellTags->"b:4.3",
 CellID->28888],
Cell[CellGroupData[{
Cell[2905, 105, 267, 9, 28, "Input",
 CellTags->"b:4.3",
 CellID->8153],
Cell[3175, 116, 906, 20, 70, "Output",
 Evaluatable->False,
 CellTags->"b:4.3",
 CellID->52740304]
}, Open  ]],
Cell[4096, 139, 262, 9, 70, "MathCaption",
 CellTags->"b:4.3",
 CellID->4747],
Cell[CellGroupData[{
Cell[4383, 152, 283, 10, 28, "Input",
 CellTags->"b:4.3",
 CellID->28680],
Cell[4669, 164, 474, 17, 36, "Output",
 CellTags->"b:4.3",
 CellID->9472051]
}, Open  ]],
Cell[5158, 184, 130, 4, 70, "MathCaption",
 CellTags->"b:4.3",
 CellID->25678],
Cell[5291, 190, 376, 14, 70, "Input",
 CellTags->"b:4.3",
 CellID->6450],
Cell[CellGroupData[{
Cell[5692, 208, 189, 6, 28, "Input",
 CellTags->"b:4.3",
 CellID->14801],
Cell[5884, 216, 443, 16, 36, "Output",
 CellTags->"b:4.3",
 CellID->151002502]
}, Open  ]],
Cell[6342, 235, 279, 9, 70, "MathCaption",
 CellTags->"b:4.3",
 CellID->14199],
Cell[CellGroupData[{
Cell[6646, 248, 281, 10, 28, "Input",
 CellTags->"b:4.3",
 CellID->20855],
Cell[6930, 260, 3156, 56, 256, "Output",
 Evaluatable->False,
 CellTags->"b:4.3",
 CellID->727978110]
}, Open  ]],
Cell[10101, 319, 85, 2, 70, "MathCaption",
 CellTags->"b:4.3",
 CellID->5716],
Cell[CellGroupData[{
Cell[10211, 325, 323, 11, 70, "Input",
 CellTags->"b:4.3",
 CellID->10573],
Cell[10537, 338, 229, 8, 36, "Output",
 CellTags->"b:4.3",
 CellID->457850758]
}, Open  ]],
Cell[10781, 349, 242, 9, 70, "MathCaption",
 CellTags->"b:4.3",
 CellID->23245],
Cell[CellGroupData[{
Cell[11048, 362, 629, 20, 28, "Input",
 CellTags->"b:4.3",
 CellID->6473],
Cell[11680, 384, 1122, 23, 62, "Output",
 Evaluatable->False,
 CellTags->"b:4.3",
 CellID->347171023]
}, Open  ]],
Cell[CellGroupData[{
Cell[12839, 412, 280, 10, 28, "Input",
 CellTags->"b:4.3",
 CellID->1550],
Cell[13122, 424, 3481, 62, 255, "Output",
 Evaluatable->False,
 CellTags->"b:4.3",
 CellID->32291927]
}, Open  ]],
Cell[16618, 489, 424, 8, 70, "Text",
 CellTags->"b:4.3",
 CellID->1181],
Cell[17045, 499, 1227, 41, 70, "MathCaption",
 CellTags->"b:4.3",
 CellID->4287],
Cell[CellGroupData[{
Cell[18297, 544, 689, 22, 28, "Input",
 CellTags->"b:4.3",
 CellID->2699],
Cell[18989, 568, 1264, 25, 60, "Output",
 Evaluatable->False,
 CellTags->"b:4.3",
 CellID->186644743]
}, Open  ]],
Cell[CellGroupData[{
Cell[20290, 598, 447, 15, 70, "Input",
 CellTags->"b:4.3",
 CellID->18643],
Cell[20740, 615, 390, 14, 39, "Output",
 CellTags->"b:4.3",
 CellID->689319669]
}, Open  ]],
Cell[CellGroupData[{
Cell[21167, 634, 522, 18, 70, "Input",
 CellTags->"b:4.3",
 CellID->7172],
Cell[21692, 654, 441, 15, 47, "Output",
 CellTags->"b:4.3",
 CellID->96686323]
}, Open  ]],
Cell[22148, 672, 323, 10, 70, "Text",
 CellTags->"b:4.3",
 CellID->9981],
Cell[22474, 684, 238, 9, 70, "MathCaption",
 CellTags->"b:4.3",
 CellID->2112],
Cell[22715, 695, 957, 32, 70, "Input",
 CellTags->"b:4.3",
 CellID->13476],
Cell[CellGroupData[{
Cell[23697, 731, 199, 6, 28, "Input",
 CellTags->"b:4.3",
 CellID->4381],
Cell[23899, 739, 6136, 105, 294, "Output",
 Evaluatable->False,
 CellTags->"b:4.3",
 CellID->1246453]
}, Open  ]],
Cell[30050, 847, 274, 9, 70, "MathCaption",
 CellTags->"b:4.3",
 CellID->26890],
Cell[30327, 858, 595, 20, 70, "Input",
 CellTags->"b:4.3",
 CellID->16671],
Cell[CellGroupData[{
Cell[30947, 882, 158, 5, 28, "Input",
 CellTags->"b:4.3",
 CellID->8805],
Cell[31108, 889, 7336, 125, 348, "Output",
 Evaluatable->False,
 CellTags->"b:4.3",
 CellID->7177450]
}, Open  ]],
Cell[38459, 1017, 158, 5, 70, "Text",
 CellTags->"b:4.3",
 CellID->30032],
Cell[38620, 1024, 177, 3, 70, "RelatedTutorials",
 CellID->439091251]
}, Open  ]],
Cell[38812, 1030, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

