(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     44455,       1621]
NotebookOptionsPosition[     29812,       1094]
NotebookOutlinePosition[     38269,       1369]
CellTagsIndexPosition[     38176,       1363]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], Cell[TextData[Cell[
    BoxData[
     PopupMenuBox[
      Dynamic[{"ActionMenu", None}, 
       Part[{"Differential Equation Solving with DSolve" :> 
         Documentation`HelpLookup["paclet:tutorial/DSolveOverview"]}, #, 2]& ,
        Evaluator -> Automatic], {
      1->"\<\"Differential Equation Solving with DSolve\"\>"}, "\<\"related \
tutorials\"\>", 
      StyleBox["\<\"related tutorials\"\>",
       Background->Automatic],
      Appearance->Automatic,
      ImageSize->Automatic,
      MenuAppearance->"Menu"]],
     FontSize->9]], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["Verification of the Solution", "Title",
 CellTags->{"c:53", "b:5.2"},
 CellID->8055],

Cell[TextData[{
 "The solution given by ",
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 " can be verified using various methods. The easiest method involves \
substituting the solution back into the equation. If the result is ",
 Cell[BoxData[
  ButtonBox["True",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/True"]], "InlineFormula"],
 StyleBox[", the solution is valid.", "TR"]
}], "Text",
 CellTags->"b:5.2",
 CellID->3393],

Cell[TextData[{
 "In this simple example, the solution is verified by substitution. Note that \
the first argument to ",
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 " is assigned to ",
 Cell[BoxData[
  StyleBox["eqn", "TI"]], "InlineFormula"],
 " for convenience in later work."
}], "MathCaption",
 CellTags->"b:5.2",
 CellID->8738],

Cell[BoxData[
 RowBox[{
  RowBox[{"eqn", " ", "=", " ", 
   RowBox[{
    RowBox[{
     RowBox[{"y", "'"}], "[", "x", "]"}], "\[Equal]", " ", "x"}]}], 
  ";"}]], "Input",
 CellTags->"b:5.2",
 CellLabel->"In[1]:=",
 CellID->15279],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", " ", "=", " ", 
  RowBox[{"DSolve", "[", 
   RowBox[{"eqn", ",", " ", "y", ",", "x"}], "]"}]}]], "Input",
 CellTags->"b:5.2",
 CellLabel->"In[2]:=",
 CellID->19882],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    RowBox[{"Function", "[", 
     RowBox[{
      RowBox[{"{", "x", "}"}], ",", 
      RowBox[{
       FractionBox[
        SuperscriptBox["x", "2"], "2"], "+", 
       RowBox[{"C", "[", "1", "]"}]}]}], "]"}]}], "}"}], "}"}]], "Output",
 ImageSize->{215, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:5.2",
 CellLabel->"Out[2]=",
 CellID->225222214]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"eqn", "/.", "sol"}]], "Input",
 CellTags->"b:5.2",
 CellLabel->"In[3]:=",
 CellID->12654],

Cell[BoxData[
 RowBox[{"{", "True", "}"}]], "Output",
 ImageSize->{46, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:5.2",
 CellLabel->"Out[3]=",
 CellID->616809145]
}, Open  ]],

Cell["\<\
In this example, the equation and an initial condition are verified by \
substitution.\
\>", "MathCaption",
 CellTags->"b:5.2",
 CellID->32707],

Cell[BoxData[
 RowBox[{
  RowBox[{"eqn", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"y", "'"}], "[", "x", "]"}], "\[Equal]", " ", "x"}], ",", " ", 
     
     RowBox[{
      RowBox[{"y", "[", "0", "]"}], "\[Equal]", "1"}]}], "}"}]}], 
  ";"}]], "Input",
 CellTags->"b:5.2",
 CellLabel->"In[4]:=",
 CellID->24245],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", " ", "=", " ", 
  RowBox[{"DSolve", "[", 
   RowBox[{"eqn", ",", " ", "y", ",", "x"}], "]"}]}]], "Input",
 CellTags->"b:5.2",
 CellLabel->"In[5]:=",
 CellID->1338],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    RowBox[{"Function", "[", 
     RowBox[{
      RowBox[{"{", "x", "}"}], ",", 
      RowBox[{
       FractionBox["1", "2"], " ", 
       RowBox[{"(", 
        RowBox[{"2", "+", 
         SuperscriptBox["x", "2"]}], ")"}]}]}], "]"}]}], "}"}], 
  "}"}]], "Output",
 ImageSize->{218, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:5.2",
 CellLabel->"Out[5]=",
 CellID->280718462]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"eqn", "/.", "sol"}]], "Input",
 CellTags->"b:5.2",
 CellLabel->"In[6]:=",
 CellID->22290],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"True", ",", "True"}], "}"}], "}"}]], "Output",
 ImageSize->{100, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:5.2",
 CellLabel->"Out[6]=",
 CellID->410688148]
}, Open  ]],

Cell[TextData[{
 "Sometimes the result of the substitution is more complicated than ",
 Cell[BoxData[
  ButtonBox["True",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/True"]], "InlineFormula"],
 " or ",
 Cell[BoxData[
  ButtonBox["False",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/False"]], "InlineFormula"],
 ". Such examples can be verified by using ",
 Cell[BoxData[
  ButtonBox["Simplify",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Simplify"]], "InlineFormula"],
 " to simplify the result of the substitution. If the simplified result is ",
 Cell[BoxData[
  ButtonBox["True",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/True"]], "InlineFormula"],
 ", the solution is valid."
}], "Text",
 CellTags->"b:5.2",
 CellID->23154],

Cell["\<\
Here is the general solution for a second-order inhomogeneous equation.\
\>", "MathCaption",
 CellTags->"b:5.2",
 CellID->24604],

Cell[BoxData[
 RowBox[{
  RowBox[{"eqn", " ", "=", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"y", "''"}], "[", "x", "]"}], " ", "+", 
     RowBox[{"5", "*", 
      RowBox[{
       RowBox[{"y", "'"}], "[", "x", "]"}]}], " ", "+", 
     RowBox[{"6", 
      RowBox[{"y", "[", "x", "]"}]}]}], "\[Equal]", "1"}]}], ";"}]], "Input",
 CellTags->"b:5.2",
 CellLabel->"In[7]:=",
 CellID->4623],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", "=", 
  RowBox[{"DSolve", "[", 
   RowBox[{"eqn", ",", " ", "y", ",", "x"}], "]"}]}]], "Input",
 CellTags->"b:5.2",
 CellLabel->"In[8]:=",
 CellID->22225],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    RowBox[{"Function", "[", 
     RowBox[{
      RowBox[{"{", "x", "}"}], ",", 
      RowBox[{
       FractionBox["1", "6"], "+", 
       RowBox[{
        SuperscriptBox["\[ExponentialE]", 
         RowBox[{
          RowBox[{"-", "3"}], " ", "x"}]], " ", 
        RowBox[{"C", "[", "1", "]"}]}], "+", 
       RowBox[{
        SuperscriptBox["\[ExponentialE]", 
         RowBox[{
          RowBox[{"-", "2"}], " ", "x"}]], " ", 
        RowBox[{"C", "[", "2", "]"}]}]}]}], "]"}]}], "}"}], "}"}]], "Output",
 ImageSize->{305, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:5.2",
 CellLabel->"Out[8]=",
 CellID->46122401]
}, Open  ]],

Cell["This substitutes the solution back into the equation.", "MathCaption",
 CellTags->"b:5.2",
 CellID->21724],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"eqn", "/.", "sol"}]], "Input",
 CellTags->"b:5.2",
 CellLabel->"In[9]:=",
 CellID->31981],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzFWttu00AQdXxJWgoKTQNNm7aEVCCoeEBCCMSlSAXah5aPiCKk9AGKevkL
bm+Ib4A3eM+nGa+9m7jHs561vQmREvvMzp6dGc/seu28G5yP3n8YnB8PB72D
08Gn0fHwrLd/chqJvJrjuDuO42x1nOg8jM7ir/o8iL5TOEUX0deLFP9MJCfi
zJv0dVzxE6S5onPnM2htT/p/hBaP7P8FtPqx3I9+XSkcleLpSoYgbvegVcfr
R/AryDepOGQYfwP+wbD4gL8D3jAa9RfgnwxLAPjbXEath0luYWTNRnsYJldy
SR5fAd4C/SeMPsYArdgF/ceA66W9SOMlwC+Z9i7gpxX5MP+Qf5fp/6ggv1mU
hFTARnQUJb4qMV4zV7bjKA3AC7K/b4lPZfICyK9IeV3ydAqO05TyoKSdZe1u
ETxl7MfxbPmxotFLssclvMysFBKLNWItNg1XDs4GztdFyrbCo9Krq0mcs71I
C3O5BV63Ep2m0Vg2Y2InRlgJmDt2ooOjzjMuag5IGtQHZwSs1Fqi7CWH0DTf
c6qSs1PNHeJYdxy6ig2rnvJI8aMfujkK+dHfdUv+Yr8gtOt3Xpz5+kjYx+Ox
zBtxpxzjVHtAXC1b0SErxWI0bhB80VYqzv+kBnC1EQw+YZk4vVnRa7M5bB4j
c1VyDTBWhzhdrWjTck67Lb/ns4rYiAa3hswiGlXWDuoen89lrLa25ajF/f0p
rm4Beo9+m96TU/tUyh5qxioSES5XbftvUmE+EY+idlaNC1dfs44L2k/vfcyq
SsXU5gxchDfv3kmcXgeM1x5XF/WMrlPRnxUNb1V/yN1Qzmq6Zsmf5ozst5mL
Iocm81p25q28t8IcFdhTPCXGM9tlcRlM7731seDv9ExiQe/C5xORohG6M+NY
dP+L1+Uqx5O2inlPmEq1q2fQau/hpo6BPLbl8arUS+/f6zkcXgq3Qno9XjS0
scHY2AKeMjbi2qo4cQ3vVbS1U8FGnHPbGjmdEbjPVaxvNfLnIL+l0dvXyPsg
x+f2dzX9Xmjk2wwf7TU+NYi1LtXuG+JKXtZ4Bho9lhNjmuW8zXijnp/jO6ns
WHidsmNxV4KOXPpOQiTsIeDXoL8J7UeA7+Xoh2H2jmID+u8B3inlFVbBIeAh
4CPAB4zV6NUe4PtGVqucFyJ8u4wzVf77aiHC9+ccQ99CT3znj6uW/m3/hbT/
70QiPsn/C5zaPy97NRQ=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{554, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:5.2",
 CellLabel->"Out[9]=",
 CellID->104447085]
}, Open  ]],

Cell[TextData[{
 "The solution can be verified using ",
 Cell[BoxData[
  ButtonBox["Simplify",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Simplify"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellTags->"b:5.2",
 CellID->29733],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Simplify", "[", 
  RowBox[{"eqn", "/.", "sol"}], "]"}]], "Input",
 CellTags->"b:5.2",
 CellLabel->"In[10]:=",
 CellID->28223],

Cell[BoxData[
 RowBox[{"{", "True", "}"}]], "Output",
 ImageSize->{46, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:5.2",
 CellLabel->"Out[10]=",
 CellID->5276569]
}, Open  ]],

Cell[TextData[{
 "Here is a linear PDE whose solution can be verified using ",
 Cell[BoxData[
  ButtonBox["Simplify",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Simplify"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellTags->"b:5.2",
 CellID->29130],

Cell[BoxData[
 RowBox[{
  RowBox[{"PDE", " ", "=", 
   RowBox[{
    RowBox[{
     RowBox[{"D", "[", 
      RowBox[{
       RowBox[{"u", "[", 
        RowBox[{"x", ",", "y"}], "]"}], ",", " ", "x"}], "]"}], " ", "+", " ",
      
     RowBox[{
      RowBox[{"Sin", "[", "x", "]"}], "*", 
      RowBox[{"D", "[", 
       RowBox[{
        RowBox[{"u", "[", 
         RowBox[{"x", ",", "y"}], "]"}], ",", "y"}], "]"}]}]}], "\[Equal]", 
    RowBox[{"x", "^", "2"}]}]}], ";"}]], "Input",
 CellTags->"b:5.2",
 CellLabel->"In[11]:=",
 CellID->18846],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", "=", 
  RowBox[{"DSolve", "[", 
   RowBox[{"PDE", ",", " ", "u", ",", " ", 
    RowBox[{"{", 
     RowBox[{"x", ",", "y"}], "}"}]}], "]"}]}]], "Input",
 CellTags->"b:5.2",
 CellLabel->"In[12]:=",
 CellID->4987],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"u", "\[Rule]", 
    RowBox[{"Function", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"x", ",", "y"}], "}"}], ",", 
      RowBox[{
       FractionBox["1", "3"], " ", 
       RowBox[{"(", 
        RowBox[{
         SuperscriptBox["x", "3"], "+", 
         RowBox[{"3", " ", 
          RowBox[{
           RowBox[{"C", "[", "1", "]"}], "[", 
           RowBox[{"y", "+", 
            RowBox[{"Cos", "[", "x", "]"}]}], "]"}]}]}], ")"}]}]}], "]"}]}], 
   "}"}], "}"}]], "Output",
 ImageSize->{343, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:5.2",
 CellLabel->"Out[12]=",
 CellID->25892531]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Simplify", "[", 
  RowBox[{"PDE", "/.", "sol"}], "]"}]], "Input",
 CellTags->"b:5.2",
 CellLabel->"In[13]:=",
 CellID->18805],

Cell[BoxData[
 RowBox[{"{", "True", "}"}]], "Output",
 ImageSize->{46, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:5.2",
 CellLabel->"Out[13]=",
 CellID->173679566]
}, Open  ]],

Cell[TextData[{
 "If the equation involves special functions, it may be necessary to use ",
 Cell[BoxData[
  ButtonBox["FullSimplify",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FullSimplify"]], "InlineFormula"],
 " to verify the solution. "
}], "Text",
 CellTags->"b:5.2",
 CellID->5750],

Cell["\<\
Here is an example of this type involving Bessel\[CloseCurlyQuote]s \
functions.\
\>", "MathCaption",
 CellTags->"b:5.2",
 CellID->20489],

Cell[BoxData[
 RowBox[{
  RowBox[{"eqn", " ", "=", " ", 
   RowBox[{
    RowBox[{
     RowBox[{"x", "*", 
      RowBox[{
       RowBox[{"y", "''"}], "[", "x", "]"}]}], " ", "+", " ", 
     RowBox[{
      RowBox[{"y", "'"}], "[", "x", "]"}], "-", 
     RowBox[{"y", "[", "x", "]"}]}], "\[Equal]", "1"}]}], ";"}]], "Input",
 CellTags->"b:5.2",
 CellLabel->"In[14]:=",
 CellID->27338],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", "=", 
  RowBox[{"DSolve", "[", 
   RowBox[{"eqn", ",", " ", "y", ",", "x"}], "]"}]}]], "Input",
 CellTags->"b:5.2",
 CellLabel->"In[15]:=",
 CellID->21963],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    RowBox[{"Function", "[", 
     RowBox[{
      RowBox[{"{", "x", "}"}], ",", 
      RowBox[{
       RowBox[{"-", "1"}], "+", 
       RowBox[{
        RowBox[{"BesselI", "[", 
         RowBox[{"0", ",", 
          RowBox[{"2", " ", 
           SqrtBox["x"]}]}], "]"}], " ", 
        RowBox[{"C", "[", "1", "]"}]}], "+", 
       RowBox[{"2", " ", 
        RowBox[{"BesselK", "[", 
         RowBox[{"0", ",", 
          RowBox[{"2", " ", 
           SqrtBox["x"]}]}], "]"}], " ", 
        RowBox[{"C", "[", "2", "]"}]}]}]}], "]"}]}], "}"}], "}"}]], "Output",
 ImageSize->{500, 22},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:5.2",
 CellLabel->"Out[15]=",
 CellID->277979645]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FullSimplify", "[", 
  RowBox[{"eqn", "/.", "sol"}], "]"}]], "Input",
 CellTags->"b:5.2",
 CellLabel->"In[16]:=",
 CellID->14697],

Cell[BoxData[
 RowBox[{"{", "True", "}"}]], "Output",
 ImageSize->{46, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:5.2",
 CellLabel->"Out[16]=",
 CellID->82783163]
}, Open  ]],

Cell[TextData[{
 "If the solution is large or if ",
 Cell[BoxData[
  ButtonBox["Simplify",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Simplify"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["FullSimplify",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FullSimplify"]], "InlineFormula"],
 " do not succeed in verifying the solution, a numerical check can be made by \
using ",
 Cell[BoxData[
  ButtonBox["RandomReal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RandomReal"]], "InlineFormula"],
 " ",
 "or ",
 ButtonBox["RandomComplex",
  BaseStyle->"Link",
  ButtonData->"paclet:ref/RandomComplex"],
 " to generate values for all the variables and parameters in the problem. It \
is advisable in such cases to repeat the check with several sets of random \
values. "
}], "Text",
 CellTags->"b:5.2",
 CellID->21630],

Cell["Here is an example where numerical verification is useful.", \
"MathCaption",
 CellTags->"b:5.2",
 CellID->23224],

Cell[BoxData[
 RowBox[{"Clear", "[", 
  RowBox[{"a", ",", "y", ",", "x", ",", " ", "r"}], "]"}]], "Input",
 CellTags->"b:5.2",
 CellLabel->"In[17]:=",
 CellID->1908],

Cell[BoxData[
 RowBox[{
  RowBox[{"eqn", " ", "=", " ", 
   RowBox[{
    RowBox[{
     RowBox[{"y", "''"}], "[", "x", "]"}], " ", "-", " ", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"a", "*", 
        RowBox[{"x", "^", "6"}]}], "+", 
       RowBox[{"x", "^", "2"}]}], ")"}], "*", 
     RowBox[{"y", "[", "x", "]"}]}]}]}], ";"}]], "Input",
 CellTags->"b:5.2",
 CellLabel->"In[18]:=",
 CellID->26737],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", "=", 
  RowBox[{"DSolve", "[", 
   RowBox[{
    RowBox[{"eqn", "\[Equal]", "0"}], ",", "y", ",", "x"}], "]"}]}]], "Input",\

 CellTags->"b:5.2",
 CellLabel->"In[19]:=",
 CellID->24474],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    RowBox[{"Function", "[", 
     RowBox[{
      RowBox[{"{", "x", "}"}], ",", 
      RowBox[{
       FractionBox[
        RowBox[{
         SuperscriptBox["2", 
          RowBox[{"3", "/", "8"}]], " ", 
         SuperscriptBox["\[ExponentialE]", 
          FractionBox[
           RowBox[{
            SqrtBox["a"], " ", 
            SuperscriptBox["x", "4"]}], "4"]], " ", 
         SuperscriptBox[
          RowBox[{"(", 
           SuperscriptBox["x", "4"], ")"}], 
          RowBox[{"3", "/", "8"}]], " ", 
         RowBox[{"C", "[", "1", "]"}], " ", 
         RowBox[{"HypergeometricU", "[", 
          RowBox[{
           FractionBox[
            RowBox[{
             RowBox[{"-", "1"}], "+", 
             RowBox[{"3", " ", 
              SqrtBox["a"]}]}], 
            RowBox[{"8", " ", 
             SqrtBox["a"]}]], ",", 
           FractionBox["3", "4"], ",", 
           RowBox[{
            RowBox[{"-", 
             FractionBox["1", "2"]}], " ", 
            SqrtBox["a"], " ", 
            SuperscriptBox["x", "4"]}]}], "]"}]}], 
        SuperscriptBox["x", 
         RowBox[{"3", "/", "2"}]]], "+", 
       FractionBox[
        RowBox[{
         SuperscriptBox["2", 
          RowBox[{"3", "/", "8"}]], " ", 
         SuperscriptBox["\[ExponentialE]", 
          FractionBox[
           RowBox[{
            SqrtBox["a"], " ", 
            SuperscriptBox["x", "4"]}], "4"]], " ", 
         SuperscriptBox[
          RowBox[{"(", 
           SuperscriptBox["x", "4"], ")"}], 
          RowBox[{"3", "/", "8"}]], " ", 
         RowBox[{"C", "[", "2", "]"}], " ", 
         RowBox[{"LaguerreL", "[", 
          RowBox[{
           RowBox[{"-", 
            FractionBox[
             RowBox[{
              RowBox[{"-", "1"}], "+", 
              RowBox[{"3", " ", 
               SqrtBox["a"]}]}], 
             RowBox[{"8", " ", 
              SqrtBox["a"]}]]}], ",", 
           RowBox[{"-", 
            FractionBox["1", "4"]}], ",", 
           RowBox[{
            RowBox[{"-", 
             FractionBox["1", "2"]}], " ", 
            SqrtBox["a"], " ", 
            SuperscriptBox["x", "4"]}]}], "]"}]}], 
        SuperscriptBox["x", 
         RowBox[{"3", "/", "2"}]]]}]}], "]"}]}], "}"}], "}"}]], "Output",
 ImageSize->{557, 126},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:5.2",
 CellLabel->"Out[19]=",
 CellID->100588371]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Union", "[", 
  RowBox[{"Flatten", "[", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"Chop", "[", 
      RowBox[{
       RowBox[{"eqn", "/.", "sol"}], "/.", " ", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"x", "\[Rule]", " ", 
          RowBox[{"RandomReal", "[", "]"}]}], ",", " ", 
         RowBox[{"a", "\[Rule]", " ", 
          RowBox[{"RandomReal", "[", "]"}]}], ",", " ", 
         RowBox[{
          RowBox[{"C", "[", "1", "]"}], "\[Rule]", " ", 
          RowBox[{"RandomReal", "[", "]"}]}], ",", " ", 
         RowBox[{
          RowBox[{"C", "[", "2", "]"}], "\[Rule]", " ", 
          RowBox[{"RandomReal", "[", "]"}]}]}], "}"}]}], "]"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"i", ",", "1", ",", "10"}], "}"}]}], "]"}], "]"}], 
  "]"}]], "Input",
 CellTags->"b:5.2",
 CellLabel->"In[20]:=",
 CellID->27372],

Cell[BoxData[
 RowBox[{"{", "0", "}"}]], "Output",
 ImageSize->{25, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:5.2",
 CellLabel->"Out[20]=",
 CellID->354564864]
}, Open  ]],

Cell["\<\
Although numerical checks cannot verify a solution with certainty, more \
rigorous checks can be made by using higher precision or by allowing the \
variables to take complex values.\
\>", "Text",
 CellTags->"b:5.2",
 CellID->3855],

Cell["This verifies the previous solution with higher precision.", \
"MathCaption",
 CellTags->"b:5.2",
 CellID->6734],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Chop", "[", 
  RowBox[{
   RowBox[{"eqn", "/.", "sol"}], "/.", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", " ", 
      RowBox[{"RandomReal", "[", " ", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"1", ",", "2"}], "}"}], ",", 
        RowBox[{"WorkingPrecision", " ", "->", "20"}]}], "]"}]}], ",", " ", 
     RowBox[{"a", "\[Rule]", " ", 
      RowBox[{"RandomReal", "[", " ", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"1", ",", "2"}], "}"}], ",", " ", 
        RowBox[{"WorkingPrecision", " ", "->", "20"}]}], "]"}]}], ",", " ", 
     RowBox[{
      RowBox[{"C", "[", "1", "]"}], "\[Rule]", 
      RowBox[{"RandomReal", "[", " ", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"1", ",", "2"}], "}"}], ",", " ", 
        RowBox[{"WorkingPrecision", " ", "->", "20"}]}], "]"}]}], " ", ",", 
     " ", 
     RowBox[{
      RowBox[{"C", "[", "2", "]"}], "\[Rule]", " ", 
      RowBox[{"RandomReal", "[", " ", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"1", ",", "2"}], "}"}], ",", " ", 
        RowBox[{"WorkingPrecision", " ", "->", "20"}]}], "]"}]}]}], "}"}]}], 
  "]"}]], "Input",
 CellTags->"b:5.2",
 CellLabel->"In[21]:=",
 CellID->9561],

Cell[BoxData[
 RowBox[{"{", "0", "}"}]], "Output",
 ImageSize->{25, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:5.2",
 CellLabel->"Out[21]=",
 CellID->287517301]
}, Open  ]],

Cell["\<\
This uses complex random values to verify the previous solution.\
\>", "MathCaption",
 CellTags->"b:5.2",
 CellID->27606],

Cell[BoxData[
 RowBox[{"r", ":=", 
  RowBox[{"RandomComplex", "[", "]"}]}]], "Input",
 CellTags->"b:5.2",
 CellLabel->"In[22]:=",
 CellID->8184],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Chop", "[", 
  RowBox[{"N", "[", 
   RowBox[{
    RowBox[{"eqn", "/.", "sol"}], "/.", " ", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"x", "\[Rule]", " ", "r"}], ",", " ", 
      RowBox[{"a", "\[Rule]", "r"}], " ", ",", " ", 
      RowBox[{
       RowBox[{"C", "[", "1", "]"}], "\[Rule]", "r"}], " ", ",", " ", 
      RowBox[{
       RowBox[{"C", "[", "2", "]"}], "\[Rule]", " ", "r"}]}], "}"}]}], "]"}], 
  "]"}]], "Input",
 CellTags->"b:5.2",
 CellLabel->"In[23]:=",
 CellID->7075],

Cell[BoxData[
 RowBox[{"{", "0", "}"}]], "Output",
 ImageSize->{25, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:5.2",
 CellLabel->"Out[23]=",
 CellID->196828262]
}, Open  ]],

Cell["\<\
The previous methods are of use only when the solution is available in \
explicit form. The final example shows how to verify the solution of a \
first-order ODE when it is given in implicit form.\
\>", "Text",
 CellTags->"b:5.2",
 CellID->14119],

Cell["This solves a first-order ODE.", "MathCaption",
 CellTags->"b:5.2",
 CellID->6741],

Cell[BoxData[
 RowBox[{
  RowBox[{"eqn", "=", " ", 
   RowBox[{
    RowBox[{
     SuperscriptBox["y", "\[Prime]",
      MultilineFunction->None], "[", "x", "]"}], "+", 
    RowBox[{"2", "*", "x", " ", 
     SuperscriptBox[
      RowBox[{"y", "[", "x", "]"}], "2"]}], "+", " ", 
    SuperscriptBox[
     RowBox[{"y", "[", "x", "]"}], "3"]}]}], ";"}]], "Input",
 CellTags->"b:5.2",
 CellLabel->"In[24]:=",
 CellID->30432],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", "=", 
  RowBox[{"DSolve", "[", 
   RowBox[{
    RowBox[{"eqn", "\[Equal]", "0"}], ",", "y", ",", "x"}], "]"}]}]], "Input",\

 CellTags->"b:5.2",
 CellLabel->"In[25]:=",
 CellID->24684],

Cell[BoxData[
 RowBox[{
  RowBox[{"InverseFunction", "::", "\<\"ifun\"\>"}], ":", 
  " ", "\<\"Inverse functions are being used. Values may be lost for \
multivalued inverses. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/InverseFunction/ifun\\\", ButtonNote -> \\\
\"InverseFunction::ifun\\\"]\\)\"\>"}]], "Message", "MSG",
 CellTags->"b:5.2",
 CellLabel->"During evaluation of In[25]:=",
 CellID->231717383],

Cell[BoxData[
 RowBox[{
  RowBox[{"InverseFunction", "::", "\<\"ifun\"\>"}], ":", 
  " ", "\<\"Inverse functions are being used. Values may be lost for \
multivalued inverses. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/InverseFunction/ifun\\\", ButtonNote -> \\\
\"InverseFunction::ifun\\\"]\\)\"\>"}]], "Message", "MSG",
 CellTags->"b:5.2",
 CellLabel->"During evaluation of In[25]:=",
 CellID->184821555],

Cell[BoxData[
 RowBox[{
  RowBox[{"InverseFunction", "::", "\<\"ifun\"\>"}], ":", 
  " ", "\<\"Inverse functions are being used. Values may be lost for \
multivalued inverses. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/InverseFunction/ifun\\\", ButtonNote -> \\\
\"InverseFunction::ifun\\\"]\\)\"\>"}]], "Message", "MSG",
 CellTags->"b:5.2",
 CellLabel->"During evaluation of In[25]:=",
 CellID->395672],

Cell[BoxData[
 RowBox[{
  RowBox[{"General", "::", "\<\"stop\"\>"}], ":", 
  " ", "\<\"Further output of \\!\\(InverseFunction :: \\\"ifun\\\"\\) will \
be suppressed during this calculation. \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/General/stop\\\", \
ButtonNote -> \\\"General::stop\\\"]\\)\"\>"}]], "Message", "MSG",
 CellTags->"b:5.2",
 CellLabel->"During evaluation of In[25]:=",
 CellID->5144806],

Cell[BoxData[
 RowBox[{
  RowBox[{"Solve", "::", "\<\"tdep\"\>"}], ":", 
  " ", "\<\"The equations appear to involve the variables to be solved for in \
an essentially non-algebraic way. \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/Solve/tdep\\\", \
ButtonNote -> \\\"Solve::tdep\\\"]\\)\"\>"}]], "Message", "MSG",
 CellTags->"b:5.2",
 CellLabel->"During evaluation of In[25]:=",
 CellID->292648265],

Cell[BoxData[
 RowBox[{"Solve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     FractionBox[
      RowBox[{
       RowBox[{
        RowBox[{"-", "x"}], " ", 
        RowBox[{"AiryAi", "[", 
         RowBox[{
          SuperscriptBox["x", "2"], "-", 
          FractionBox["1", 
           RowBox[{"y", "[", "x", "]"}]]}], "]"}]}], "+", 
       RowBox[{"AiryAiPrime", "[", 
        RowBox[{
         SuperscriptBox["x", "2"], "-", 
         FractionBox["1", 
          RowBox[{"y", "[", "x", "]"}]]}], "]"}]}], 
      RowBox[{
       RowBox[{
        RowBox[{"-", "x"}], " ", 
        RowBox[{"AiryBi", "[", 
         RowBox[{
          SuperscriptBox["x", "2"], "-", 
          FractionBox["1", 
           RowBox[{"y", "[", "x", "]"}]]}], "]"}]}], "+", 
       RowBox[{"AiryBiPrime", "[", 
        RowBox[{
         SuperscriptBox["x", "2"], "-", 
         FractionBox["1", 
          RowBox[{"y", "[", "x", "]"}]]}], "]"}]}]], "+", 
     RowBox[{"C", "[", "1", "]"}]}], "\[Equal]", "0"}], ",", 
   RowBox[{"y", "[", "x", "]"}]}], "]"}]], "Output",
 ImageSize->{437, 55},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:5.2",
 CellLabel->"Out[25]=",
 CellID->155155578]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", "[", 
  RowBox[{"[", "1", "]"}], "]"}]], "Input",
 CellTags->"b:5.2",
 CellLabel->"In[26]:=",
 CellID->24118],

Cell[BoxData[
 RowBox[{
  RowBox[{
   FractionBox[
    RowBox[{
     RowBox[{
      RowBox[{"-", "x"}], " ", 
      RowBox[{"AiryAi", "[", 
       RowBox[{
        SuperscriptBox["x", "2"], "-", 
        FractionBox["1", 
         RowBox[{"y", "[", "x", "]"}]]}], "]"}]}], "+", 
     RowBox[{"AiryAiPrime", "[", 
      RowBox[{
       SuperscriptBox["x", "2"], "-", 
       FractionBox["1", 
        RowBox[{"y", "[", "x", "]"}]]}], "]"}]}], 
    RowBox[{
     RowBox[{
      RowBox[{"-", "x"}], " ", 
      RowBox[{"AiryBi", "[", 
       RowBox[{
        SuperscriptBox["x", "2"], "-", 
        FractionBox["1", 
         RowBox[{"y", "[", "x", "]"}]]}], "]"}]}], "+", 
     RowBox[{"AiryBiPrime", "[", 
      RowBox[{
       SuperscriptBox["x", "2"], "-", 
       FractionBox["1", 
        RowBox[{"y", "[", "x", "]"}]]}], "]"}]}]], "+", 
   RowBox[{"C", "[", "1", "]"}]}], "\[Equal]", "0"}]], "Output",
 ImageSize->{349, 55},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:5.2",
 CellLabel->"Out[26]=",
 CellID->265399845]
}, Open  ]],

Cell["This verifies the solution by simplifying its derivative.", \
"MathCaption",
 CellTags->"b:5.2",
 CellID->18028],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Simplify", "[", 
  RowBox[{"Solve", "[", 
   RowBox[{
    RowBox[{"D", "[", 
     RowBox[{
      RowBox[{"sol", "[", 
       RowBox[{"[", "1", "]"}], "]"}], ",", "x"}], "]"}], ",", " ", 
    RowBox[{
     RowBox[{"y", "'"}], "[", "x", "]"}]}], "]"}], "]"}]], "Input",
 CellTags->"b:5.2",
 CellLabel->"In[27]:=",
 CellID->32119],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{
     SuperscriptBox["y", "\[Prime]",
      MultilineFunction->None], "[", "x", "]"}], "\[Rule]", 
    RowBox[{
     RowBox[{"-", 
      SuperscriptBox[
       RowBox[{"y", "[", "x", "]"}], "2"]}], " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"2", " ", "x"}], "+", 
       RowBox[{"y", "[", "x", "]"}]}], ")"}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{196, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:5.2",
 CellLabel->"Out[27]=",
 CellID->393484568]
}, Open  ]],

Cell[CellGroupData[{

Cell["RELATED TUTORIALS", "RelatedTutorialsSection",
 CellID->38852426],

Cell[TextData[ButtonBox["Differential Equation Solving with DSolve",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/DSolveOverview"]], "RelatedTutorials",
 CellID->439091251]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Verification of the Solution - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, "Overview" -> {"toc" -> Cell[
      BoxData[
       ButtonBox[
        StyleBox[
        "Differential Equation Solving with DSolve", "OverviewNavText"], 
        BaseStyle -> "Link", ButtonData -> "paclet:tutorial/DSolveOverview"]],
       "Text", FontFamily -> "Verdana"], "prevnext" -> Cell[
      TextData[{
        ButtonBox[
         StyleBox["\[FilledLeftTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> "paclet:tutorial/DSolveSettingUpTheProblem"], 
        "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]", 
        ButtonBox[
         StyleBox["\[FilledRightTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> "paclet:tutorial/DSolvePlottingTheSolution"]}],
       "Text", FontFamily -> "Verdana"]}, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 34, 50.0480584}", "context" -> "", 
    "keywords" -> {}, "index" -> True, "label" -> "Mathematica Tutorial", 
    "language" -> "en", "paclet" -> "Mathematica", "status" -> "None", 
    "summary" -> 
    "The solution given by DSolve can be verified using various methods. The \
easiest method involves substituting the solution back into the equation. If \
the result is True, the solution is valid. In this simple example, the \
solution is verified by substitution. Note that the first argument to DSolve \
is assigned to eqn for convenience in later work. In this example, the \
equation and an initial condition are verified by substitution.", 
    "synonyms" -> {}, "title" -> "Verification of the Solution", "type" -> 
    "Tutorial", "uri" -> "tutorial/DSolveSolutionVerification"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "c:53"->{
  Cell[1280, 43, 90, 2, 70, "Title",
   CellTags->{"c:53", "b:5.2"},
   CellID->8055]},
 "b:5.2"->{
  Cell[1280, 43, 90, 2, 70, "Title",
   CellTags->{"c:53", "b:5.2"},
   CellID->8055],
  Cell[1373, 47, 501, 15, 70, "Text",
   CellTags->"b:5.2",
   CellID->3393],
  Cell[1877, 64, 404, 13, 70, "MathCaption",
   CellTags->"b:5.2",
   CellID->8738],
  Cell[2284, 79, 228, 9, 70, "Input",
   CellTags->"b:5.2",
   CellID->15279],
  Cell[2537, 92, 194, 6, 28, "Input",
   CellTags->"b:5.2",
   CellID->19882],
  Cell[2734, 100, 460, 16, 54, "Output",
   CellTags->"b:5.2",
   CellID->225222214],
  Cell[3231, 121, 113, 4, 28, "Input",
   CellTags->"b:5.2",
   CellID->12654],
  Cell[3347, 127, 203, 7, 36, "Output",
   CellTags->"b:5.2",
   CellID->616809145],
  Cell[3565, 137, 153, 5, 70, "MathCaption",
   CellTags->"b:5.2",
   CellID->32707],
  Cell[3721, 144, 359, 14, 70, "Input",
   CellTags->"b:5.2",
   CellID->24245],
  Cell[4105, 162, 193, 6, 28, "Input",
   CellTags->"b:5.2",
   CellID->1338],
  Cell[4301, 170, 488, 18, 51, "Output",
   CellTags->"b:5.2",
   CellID->280718462],
  Cell[4826, 193, 113, 4, 70, "Input",
   CellTags->"b:5.2",
   CellID->22290],
  Cell[4942, 199, 254, 9, 36, "Output",
   CellTags->"b:5.2",
   CellID->410688148],
  Cell[5211, 211, 749, 24, 70, "Text",
   CellTags->"b:5.2",
   CellID->23154],
  Cell[5963, 237, 138, 4, 70, "MathCaption",
   CellTags->"b:5.2",
   CellID->24604],
  Cell[6104, 243, 396, 14, 70, "Input",
   CellTags->"b:5.2",
   CellID->4623],
  Cell[6525, 261, 184, 6, 70, "Input",
   CellTags->"b:5.2",
   CellID->22225],
  Cell[6712, 269, 730, 24, 51, "Output",
   CellTags->"b:5.2",
   CellID->46122401],
  Cell[7457, 296, 112, 2, 70, "MathCaption",
   CellTags->"b:5.2",
   CellID->21724],
  Cell[7594, 302, 113, 4, 28, "Input",
   CellTags->"b:5.2",
   CellID->31981],
  Cell[7710, 308, 1472, 29, 51, "Output",
   Evaluatable->False,
   CellTags->"b:5.2",
   CellID->104447085],
  Cell[9197, 340, 234, 9, 70, "MathCaption",
   CellTags->"b:5.2",
   CellID->29733],
  Cell[9456, 353, 149, 5, 70, "Input",
   CellTags->"b:5.2",
   CellID->28223],
  Cell[9608, 360, 202, 7, 36, "Output",
   CellTags->"b:5.2",
   CellID->5276569],
  Cell[9825, 370, 257, 9, 70, "MathCaption",
   CellTags->"b:5.2",
   CellID->29130],
  Cell[10085, 381, 540, 19, 70, "Input",
   CellTags->"b:5.2",
   CellID->18846],
  Cell[10650, 404, 240, 8, 70, "Input",
   CellTags->"b:5.2",
   CellID->4987],
  Cell[10893, 414, 692, 24, 51, "Output",
   CellTags->"b:5.2",
   CellID->25892531],
  Cell[11622, 443, 149, 5, 70, "Input",
   CellTags->"b:5.2",
   CellID->18805],
  Cell[11774, 450, 204, 7, 36, "Output",
   CellTags->"b:5.2",
   CellID->173679566],
  Cell[11993, 460, 294, 9, 70, "Text",
   CellTags->"b:5.2",
   CellID->5750],
  Cell[12290, 471, 147, 5, 70, "MathCaption",
   CellTags->"b:5.2",
   CellID->20489],
  Cell[12440, 478, 381, 13, 70, "Input",
   CellTags->"b:5.2",
   CellID->27338],
  Cell[12846, 495, 185, 6, 70, "Input",
   CellTags->"b:5.2",
   CellID->21963],
  Cell[13034, 503, 783, 26, 43, "Output",
   CellTags->"b:5.2",
   CellID->277979645],
  Cell[13854, 534, 153, 5, 70, "Input",
   CellTags->"b:5.2",
   CellID->14697],
  Cell[14010, 541, 203, 7, 36, "Output",
   CellTags->"b:5.2",
   CellID->82783163],
  Cell[14228, 551, 840, 27, 70, "Text",
   CellTags->"b:5.2",
   CellID->21630],
  Cell[15071, 580, 119, 3, 70, "MathCaption",
   CellTags->"b:5.2",
   CellID->23224],
  Cell[15193, 585, 165, 5, 70, "Input",
   CellTags->"b:5.2",
   CellID->1908],
  Cell[15361, 592, 419, 15, 70, "Input",
   CellTags->"b:5.2",
   CellID->26737],
  Cell[15805, 611, 214, 8, 70, "Input",
   CellTags->"b:5.2",
   CellID->24474],
  Cell[16022, 621, 2457, 77, 147, "Output",
   CellTags->"b:5.2",
   CellID->100588371],
  Cell[18516, 703, 870, 25, 70, "Input",
   CellTags->"b:5.2",
   CellID->27372],
  Cell[19389, 730, 201, 7, 36, "Output",
   CellTags->"b:5.2",
   CellID->354564864],
  Cell[19605, 740, 241, 6, 70, "Text",
   CellTags->"b:5.2",
   CellID->3855],
  Cell[19849, 748, 118, 3, 70, "MathCaption",
   CellTags->"b:5.2",
   CellID->6734],
  Cell[19992, 755, 1230, 36, 70, "Input",
   CellTags->"b:5.2",
   CellID->9561],
  Cell[21225, 793, 201, 7, 36, "Output",
   CellTags->"b:5.2",
   CellID->287517301],
  Cell[21441, 803, 131, 4, 70, "MathCaption",
   CellTags->"b:5.2",
   CellID->27606],
  Cell[21575, 809, 144, 5, 70, "Input",
   CellTags->"b:5.2",
   CellID->8184],
  Cell[21744, 818, 511, 16, 70, "Input",
   CellTags->"b:5.2",
   CellID->7075],
  Cell[22258, 836, 201, 7, 36, "Output",
   CellTags->"b:5.2",
   CellID->196828262],
  Cell[22474, 846, 256, 6, 70, "Text",
   CellTags->"b:5.2",
   CellID->14119],
  Cell[22733, 854, 88, 2, 70, "MathCaption",
   CellTags->"b:5.2",
   CellID->6741],
  Cell[22824, 858, 419, 14, 70, "Input",
   CellTags->"b:5.2",
   CellID->30432],
  Cell[23268, 876, 214, 8, 70, "Input",
   CellTags->"b:5.2",
   CellID->24684],
  Cell[23485, 886, 493, 10, 70, "Message",
   CellTags->"b:5.2",
   CellID->231717383],
  Cell[23981, 898, 493, 10, 70, "Message",
   CellTags->"b:5.2",
   CellID->184821555],
  Cell[24477, 910, 490, 10, 70, "Message",
   CellTags->"b:5.2",
   CellID->395672],
  Cell[24970, 922, 491, 10, 70, "Message",
   CellTags->"b:5.2",
   CellID->5144806],
  Cell[25464, 934, 484, 10, 70, "Message",
   CellTags->"b:5.2",
   CellID->292648265],
  Cell[25951, 946, 1200, 39, 76, "Output",
   CellTags->"b:5.2",
   CellID->155155578],
  Cell[27188, 990, 139, 5, 70, "Input",
   CellTags->"b:5.2",
   CellID->24118],
  Cell[27330, 997, 1057, 36, 76, "Output",
   CellTags->"b:5.2",
   CellID->265399845],
  Cell[28402, 1036, 118, 3, 70, "MathCaption",
   CellTags->"b:5.2",
   CellID->18028],
  Cell[28545, 1043, 351, 12, 70, "Input",
   CellTags->"b:5.2",
   CellID->32119],
  Cell[28899, 1057, 571, 20, 39, "Output",
   CellTags->"b:5.2",
   CellID->393484568]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"c:53", 32263, 1147},
 {"b:5.2", 32363, 1151}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 655, 16, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[1280, 43, 90, 2, 70, "Title",
 CellTags->{"c:53", "b:5.2"},
 CellID->8055],
Cell[1373, 47, 501, 15, 70, "Text",
 CellTags->"b:5.2",
 CellID->3393],
Cell[1877, 64, 404, 13, 70, "MathCaption",
 CellTags->"b:5.2",
 CellID->8738],
Cell[2284, 79, 228, 9, 70, "Input",
 CellTags->"b:5.2",
 CellID->15279],
Cell[CellGroupData[{
Cell[2537, 92, 194, 6, 28, "Input",
 CellTags->"b:5.2",
 CellID->19882],
Cell[2734, 100, 460, 16, 54, "Output",
 CellTags->"b:5.2",
 CellID->225222214]
}, Open  ]],
Cell[CellGroupData[{
Cell[3231, 121, 113, 4, 28, "Input",
 CellTags->"b:5.2",
 CellID->12654],
Cell[3347, 127, 203, 7, 36, "Output",
 CellTags->"b:5.2",
 CellID->616809145]
}, Open  ]],
Cell[3565, 137, 153, 5, 70, "MathCaption",
 CellTags->"b:5.2",
 CellID->32707],
Cell[3721, 144, 359, 14, 70, "Input",
 CellTags->"b:5.2",
 CellID->24245],
Cell[CellGroupData[{
Cell[4105, 162, 193, 6, 28, "Input",
 CellTags->"b:5.2",
 CellID->1338],
Cell[4301, 170, 488, 18, 51, "Output",
 CellTags->"b:5.2",
 CellID->280718462]
}, Open  ]],
Cell[CellGroupData[{
Cell[4826, 193, 113, 4, 70, "Input",
 CellTags->"b:5.2",
 CellID->22290],
Cell[4942, 199, 254, 9, 36, "Output",
 CellTags->"b:5.2",
 CellID->410688148]
}, Open  ]],
Cell[5211, 211, 749, 24, 70, "Text",
 CellTags->"b:5.2",
 CellID->23154],
Cell[5963, 237, 138, 4, 70, "MathCaption",
 CellTags->"b:5.2",
 CellID->24604],
Cell[6104, 243, 396, 14, 70, "Input",
 CellTags->"b:5.2",
 CellID->4623],
Cell[CellGroupData[{
Cell[6525, 261, 184, 6, 70, "Input",
 CellTags->"b:5.2",
 CellID->22225],
Cell[6712, 269, 730, 24, 51, "Output",
 CellTags->"b:5.2",
 CellID->46122401]
}, Open  ]],
Cell[7457, 296, 112, 2, 70, "MathCaption",
 CellTags->"b:5.2",
 CellID->21724],
Cell[CellGroupData[{
Cell[7594, 302, 113, 4, 28, "Input",
 CellTags->"b:5.2",
 CellID->31981],
Cell[7710, 308, 1472, 29, 51, "Output",
 Evaluatable->False,
 CellTags->"b:5.2",
 CellID->104447085]
}, Open  ]],
Cell[9197, 340, 234, 9, 70, "MathCaption",
 CellTags->"b:5.2",
 CellID->29733],
Cell[CellGroupData[{
Cell[9456, 353, 149, 5, 70, "Input",
 CellTags->"b:5.2",
 CellID->28223],
Cell[9608, 360, 202, 7, 36, "Output",
 CellTags->"b:5.2",
 CellID->5276569]
}, Open  ]],
Cell[9825, 370, 257, 9, 70, "MathCaption",
 CellTags->"b:5.2",
 CellID->29130],
Cell[10085, 381, 540, 19, 70, "Input",
 CellTags->"b:5.2",
 CellID->18846],
Cell[CellGroupData[{
Cell[10650, 404, 240, 8, 70, "Input",
 CellTags->"b:5.2",
 CellID->4987],
Cell[10893, 414, 692, 24, 51, "Output",
 CellTags->"b:5.2",
 CellID->25892531]
}, Open  ]],
Cell[CellGroupData[{
Cell[11622, 443, 149, 5, 70, "Input",
 CellTags->"b:5.2",
 CellID->18805],
Cell[11774, 450, 204, 7, 36, "Output",
 CellTags->"b:5.2",
 CellID->173679566]
}, Open  ]],
Cell[11993, 460, 294, 9, 70, "Text",
 CellTags->"b:5.2",
 CellID->5750],
Cell[12290, 471, 147, 5, 70, "MathCaption",
 CellTags->"b:5.2",
 CellID->20489],
Cell[12440, 478, 381, 13, 70, "Input",
 CellTags->"b:5.2",
 CellID->27338],
Cell[CellGroupData[{
Cell[12846, 495, 185, 6, 70, "Input",
 CellTags->"b:5.2",
 CellID->21963],
Cell[13034, 503, 783, 26, 43, "Output",
 CellTags->"b:5.2",
 CellID->277979645]
}, Open  ]],
Cell[CellGroupData[{
Cell[13854, 534, 153, 5, 70, "Input",
 CellTags->"b:5.2",
 CellID->14697],
Cell[14010, 541, 203, 7, 36, "Output",
 CellTags->"b:5.2",
 CellID->82783163]
}, Open  ]],
Cell[14228, 551, 840, 27, 70, "Text",
 CellTags->"b:5.2",
 CellID->21630],
Cell[15071, 580, 119, 3, 70, "MathCaption",
 CellTags->"b:5.2",
 CellID->23224],
Cell[15193, 585, 165, 5, 70, "Input",
 CellTags->"b:5.2",
 CellID->1908],
Cell[15361, 592, 419, 15, 70, "Input",
 CellTags->"b:5.2",
 CellID->26737],
Cell[CellGroupData[{
Cell[15805, 611, 214, 8, 70, "Input",
 CellTags->"b:5.2",
 CellID->24474],
Cell[16022, 621, 2457, 77, 147, "Output",
 CellTags->"b:5.2",
 CellID->100588371]
}, Open  ]],
Cell[CellGroupData[{
Cell[18516, 703, 870, 25, 70, "Input",
 CellTags->"b:5.2",
 CellID->27372],
Cell[19389, 730, 201, 7, 36, "Output",
 CellTags->"b:5.2",
 CellID->354564864]
}, Open  ]],
Cell[19605, 740, 241, 6, 70, "Text",
 CellTags->"b:5.2",
 CellID->3855],
Cell[19849, 748, 118, 3, 70, "MathCaption",
 CellTags->"b:5.2",
 CellID->6734],
Cell[CellGroupData[{
Cell[19992, 755, 1230, 36, 70, "Input",
 CellTags->"b:5.2",
 CellID->9561],
Cell[21225, 793, 201, 7, 36, "Output",
 CellTags->"b:5.2",
 CellID->287517301]
}, Open  ]],
Cell[21441, 803, 131, 4, 70, "MathCaption",
 CellTags->"b:5.2",
 CellID->27606],
Cell[21575, 809, 144, 5, 70, "Input",
 CellTags->"b:5.2",
 CellID->8184],
Cell[CellGroupData[{
Cell[21744, 818, 511, 16, 70, "Input",
 CellTags->"b:5.2",
 CellID->7075],
Cell[22258, 836, 201, 7, 36, "Output",
 CellTags->"b:5.2",
 CellID->196828262]
}, Open  ]],
Cell[22474, 846, 256, 6, 70, "Text",
 CellTags->"b:5.2",
 CellID->14119],
Cell[22733, 854, 88, 2, 70, "MathCaption",
 CellTags->"b:5.2",
 CellID->6741],
Cell[22824, 858, 419, 14, 70, "Input",
 CellTags->"b:5.2",
 CellID->30432],
Cell[CellGroupData[{
Cell[23268, 876, 214, 8, 70, "Input",
 CellTags->"b:5.2",
 CellID->24684],
Cell[23485, 886, 493, 10, 70, "Message",
 CellTags->"b:5.2",
 CellID->231717383],
Cell[23981, 898, 493, 10, 70, "Message",
 CellTags->"b:5.2",
 CellID->184821555],
Cell[24477, 910, 490, 10, 70, "Message",
 CellTags->"b:5.2",
 CellID->395672],
Cell[24970, 922, 491, 10, 70, "Message",
 CellTags->"b:5.2",
 CellID->5144806],
Cell[25464, 934, 484, 10, 70, "Message",
 CellTags->"b:5.2",
 CellID->292648265],
Cell[25951, 946, 1200, 39, 76, "Output",
 CellTags->"b:5.2",
 CellID->155155578]
}, Open  ]],
Cell[CellGroupData[{
Cell[27188, 990, 139, 5, 70, "Input",
 CellTags->"b:5.2",
 CellID->24118],
Cell[27330, 997, 1057, 36, 76, "Output",
 CellTags->"b:5.2",
 CellID->265399845]
}, Open  ]],
Cell[28402, 1036, 118, 3, 70, "MathCaption",
 CellTags->"b:5.2",
 CellID->18028],
Cell[CellGroupData[{
Cell[28545, 1043, 351, 12, 70, "Input",
 CellTags->"b:5.2",
 CellID->32119],
Cell[28899, 1057, 571, 20, 39, "Output",
 CellTags->"b:5.2",
 CellID->393484568]
}, Open  ]],
Cell[CellGroupData[{
Cell[29507, 1082, 71, 1, 70, "RelatedTutorialsSection",
 CellID->38852426],
Cell[29581, 1085, 177, 3, 70, "RelatedTutorials",
 CellID->439091251]
}, Open  ]]
}, Open  ]],
Cell[29785, 1092, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

