(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     37328,       1118]
NotebookOptionsPosition[     30412,        894]
NotebookOutlinePosition[     33500,        970]
CellTagsIndexPosition[     33407,        964]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], Cell[TextData[Cell[
    BoxData[
     PopupMenuBox[
      Dynamic[{"ActionMenu", None}, 
       Part[{"Numerical Operations on Functions" :> 
         Documentation`HelpLookup[
          "paclet:tutorial/NumericalOperationsOnFunctionsOverview"]}, #, 2]& ,
        Evaluator -> Automatic], {
      1->"\<\"Numerical Operations on Functions\"\>"}, "\<\"related tutorials\
\"\>", 
      StyleBox["\<\"related tutorials\"\>",
       Background->Automatic],
      Appearance->Automatic,
      ImageSize->Automatic,
      MenuAppearance->"Menu"]],
     FontSize->9]], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["Functions with Sensitive Dependence on Their Input", "Title",
 CellTags->{"S3.9.11", "11.1"},
 CellID->15338],

Cell["\<\
Functions that are specified by simple algebraic formulas tend to be such \
that when their input is changed only slightly, their output also changes \
only slightly. But functions that are instead based on executing procedures \
quite often show almost arbitrarily sensitive dependence on their input. \
Typically the reason this happens is that the procedure \
\[OpenCurlyDoubleQuote]excavates\[CloseCurlyDoubleQuote] progressively less \
and less significant digits in the input. \
\>", "Text",
 CellID->17342],

Cell["\<\
This shows successive steps in a simple iterative procedure with input \
0.1111. \
\>", "MathCaption",
 CellID->14428],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NestList", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"FractionalPart", "[", 
     RowBox[{"2", "#"}], "]"}], "&"}], ",", "0.1111", ",", "10"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:= ",
 CellID->29795],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.1111`", ",", "0.2222`", ",", "0.4444`", ",", "0.8888`", ",", 
   "0.7776000000000001`", ",", "0.5552000000000001`", ",", 
   "0.11040000000000028`", ",", "0.22080000000000055`", ",", 
   "0.4416000000000011`", ",", "0.8832000000000022`", ",", 
   "0.7664000000000044`"}], "}"}]], "Output",
 ImageSize->{592, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]= ",
 CellID->30742]
}, Open  ]],

Cell["\<\
Here is the result with input 0.1112. Progressive divergence from the result \
with input 0.1111 is seen. \
\>", "MathCaption",
 CellID->30037],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NestList", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"FractionalPart", "[", 
     RowBox[{"2", "#"}], "]"}], "&"}], ",", "0.1112", ",", "10"}], 
  "]"}]], "Input",
 CellLabel->"In[2]:= ",
 CellID->5506],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.1112`", ",", "0.2224`", ",", "0.4448`", ",", "0.8896`", ",", 
   "0.7791999999999999`", ",", "0.5583999999999998`", ",", 
   "0.11679999999999957`", ",", "0.23359999999999914`", ",", 
   "0.4671999999999983`", ",", "0.9343999999999966`", ",", 
   "0.8687999999999931`"}], "}"}]], "Output",
 ImageSize->{592, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]= ",
 CellID->6767]
}, Open  ]],

Cell[TextData[{
 "The action of ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["FractionalPart",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FractionalPart"], "[", 
   RowBox[{"2", 
    StyleBox["x", "TI"]}], "]"}]], "InlineFormula"],
 " is particularly simple in terms of the binary digits of the number ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 ": it just drops the first one, and shifts the remaining ones to the left. \
After several steps, this means that the results one gets are inevitably \
sensitive to digits that are far to the right, and have an extremely small \
effect on the original value of ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->5053],

Cell[TextData[{
 "This shows the shifting process achieved by ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["FractionalPart",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FractionalPart"], "[", 
   RowBox[{"2", 
    StyleBox["x", "TI"]}], "]"}]], "InlineFormula"],
 " in the first 8 binary digits of ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 "."
}], "MathCaption",
 CellID->9793],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RealDigits", "[", 
  RowBox[{
   RowBox[{"Take", "[", 
    RowBox[{"%", ",", "5"}], "]"}], ",", "2", ",", "8", ",", 
   RowBox[{"-", "1"}]}], "]"}]], "Input",
 CellLabel->"In[3]:= ",
 CellID->28505],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWc1OwzAMzlrGxs/YxonjeBU0wYmHqCak7TCBxl4QOABPVZqWjso4tZ2k
P4hMWls7buzPdtvEvk/264dtst+sksXdLnlab1bPi9vHXcaKB0pFN0qp6yuV
XafZVf4vf8vs/0MW1Jk+xgcpdQToE0BPPdMzYhzaQ42fOuKh7HHF1zYeV/9T
tBRPrA8kKj0eZeSMeZ5XzlBbHV9qx5xpz2XDdpjwY3yOHVBfxIqSnk2TQ8A/
zg7DjITyUv7IML+JPxbaw9UH5cYM70lx+sBv4kv19skvTcZfyrf1C+9pyqWG
Bzq/p7BBP7HwPW/LH9VqqR8diy2UWcDFwdFiwo9FwdUv9aMSS/rnr3bzRcp3
89fvZ5TSEaNzS9dME3QWnHvB0gjfQOfoXVBqQqCF8tTaUYIL51IausHpP14U
bYcT4hrkcnbfne5zGtqJzyXdf1Dy7ed0Nzjbz2k7nD7XUiGn5bhCToecDjnd
Hc6Q0yGnQ073KaepjomWir9tNNVsy/EY0Hoc243EiHWcGq6pRlzVPwX65w3b
QeGH/uLagXcayvguG6M4FZYqHTpqfvFJ8bzqQ5Edgqj9t47Um5WXTB0hXx0L
Kd/UEeBW7n11Pmz80qcOmq1fXlKbLKrvhDRRi5fy6yvbsoq93/q+i+9MvZF+
9JjcfVe8+SPCg7yav8sqnKLxNS8+dzMVcQlm6ffdffVvh/k99fcu6kv0m9m/
STC3H307zB9piD6uMUQ/RD9EP0RfivlvRV9apalW8eDZtN+vymGex/ic6ll1
3rp9PpST2PGZe6n0o7QGpgZfzd4b2g==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{578, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]= ",
 CellID->482572570]
}, Open  ]],

Cell[TextData[{
 "If you give input only to a particular precision, you are effectively \
specifying only a certain number of digits. And once all these digits have \
been \[OpenCurlyDoubleQuote]excavated\[CloseCurlyDoubleQuote] you can no \
longer get accurate results, since to do so would require knowing more digits \
of your original input. So long as you use arbitrary\[Hyphen]precision \
numbers, ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " automatically keeps track of this kind of degradation in precision, \
indicating a number with no remaining significant digits by ",
 Cell[BoxData[
  RowBox[{"0.", "\[Times]", 
   SuperscriptBox["10", 
    StyleBox["e", "TI"]]}]], "InlineFormula"],
 ", as discussed in\[NonBreakingSpace]\"",
 ButtonBox["Arbitrary-Precision Numbers\"",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/ArbitraryPrecisionNumbers"],
 ". "
}], "Text",
 CellID->3609],

Cell["\<\
Successive steps yield numbers of progressively lower precision, and \
eventually no precision at all. \
\>", "MathCaption",
 CellID->13924],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NestList", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"FractionalPart", "[", 
     RowBox[{"40", "#"}], "]"}], "&"}], ",", 
   RowBox[{"N", "[", 
    RowBox[{
     RowBox[{"1", "/", "9"}], ",", "20"}], "]"}], ",", "20"}], "]"}]], "Input",\

 CellLabel->"In[4]:= ",
 CellID->19854],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.11111111111111111111111111111111111111`20.", ",", 
   "0.4444444444444444444444444444388347669`19.", ",", 
   "0.77777777777777777777777777755`17.640978057358332", ",", 
   "0.11111111111111111111111110214`15.19382002601611", ",", 
   "0.44444444444444444444444408543`14.193820026016109", ",", 
   "0.777777777777777777777763417`12.834798083374439", ",", 
   "0.11111111111111111111053668013`10.387640052032218", ",", 
   "0.44444444444444444443842109904`9.387640052032216", ",", 
   "0.7777777777777777775`8.028618109390548", ",", 
   "0.1111111111111111015`5.581460078048329", ",", 
   "0.444444444444444059`4.5814600780483286", ",", 
   "0.777777777777762358`3.222438135406653", ",", 
   "0.1111111111104943205`0.7752801040620314", ",", "0``0.12746262216834003", 
   ",", "0``-1.4745973691596213", ",", "0``-3.0766573604875824", ",", 
   "0``-4.678717351815544", ",", "0``-6.280777343143505", ",", 
   "0``-7.882837334471466", ",", "0``-9.484897325799427", ",", 
   "0``-11.08695731712739"}], "}"}]], "Output",
 ImageSize->{479, 71},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]= ",
 CellID->29465]
}, Open  ]],

Cell["\<\
This asks for the precision of each number. Zero precision indicates that \
there are no correct significant digits. \
\>", "MathCaption",
 CellID->12239],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Map", "[", 
  RowBox[{"Precision", ",", "%"}], "]"}]], "Input",
 CellLabel->"In[5]:= ",
 CellID->26206],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "20.`", ",", "19.`", ",", "17.640978057358332`", ",", "15.19382002601611`", 
   ",", "14.193820026016109`", ",", "12.834798083374439`", ",", 
   "10.387640052032218`", ",", "9.387640052032216`", ",", 
   "8.028618109390548`", ",", "5.581460078048329`", ",", 
   "4.5814600780483286`", ",", "3.222438135406653`", ",", 
   "0.7752801040620314`", ",", "0.`", ",", "0.`", ",", "0.`", ",", "0.`", ",",
    "0.`", ",", "0.`", ",", "0.`", ",", "0.`"}], "}"}]], "Output",
 ImageSize->{510, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]= ",
 CellID->24209]
}, Open  ]],

Cell["This shows that the exact result is a periodic sequence. ", \
"MathCaption",
 CellID->15492],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NestList", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"FractionalPart", "[", 
     RowBox[{"40", "#"}], "]"}], "&"}], ",", 
   RowBox[{"1", "/", "9"}], ",", "10"}], "]"}]], "Input",
 CellLabel->"In[6]:= ",
 CellID->16499],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["1", "9"], ",", 
   FractionBox["4", "9"], ",", 
   FractionBox["7", "9"], ",", 
   FractionBox["1", "9"], ",", 
   FractionBox["4", "9"], ",", 
   FractionBox["7", "9"], ",", 
   FractionBox["1", "9"], ",", 
   FractionBox["4", "9"], ",", 
   FractionBox["7", "9"], ",", 
   FractionBox["1", "9"], ",", 
   FractionBox["4", "9"]}], "}"}]], "Output",
 ImageSize->{253, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]= ",
 CellID->18175]
}, Open  ]],

Cell[TextData[{
 "It is important to realize that if you use approximate numbers of any kind, \
then in an example like the one above you will always eventually run out of \
precision. But so long as you use arbitrary\[Hyphen]precision numbers, ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " will explicitly show you any decrease in precision that is occurring. \
However, if you use machine\[Hyphen]precision numbers, then ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " will not keep track of precision, and you cannot tell when your results \
become meaningless. "
}], "Text",
 CellID->31036],

Cell[TextData[{
 "If you use machine\[Hyphen]precision numbers, ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " will no longer keep track of any degradation in precision. "
}], "MathCaption",
 CellID->23061],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NestList", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"FractionalPart", "[", 
     RowBox[{"40", "#"}], "]"}], "&"}], ",", 
   RowBox[{"N", "[", 
    RowBox[{"1", "/", "9"}], "]"}], ",", "20"}], "]"}]], "Input",
 CellLabel->"In[7]:= ",
 CellID->13647],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.1111111111111111`", ",", "0.44444444444444464`", ",", 
   "0.7777777777777857`", ",", "0.11111111111142691`", ",", 
   "0.4444444444570763`", ",", "0.7777777782830526`", ",", 
   "0.11111113132210448`", ",", "0.44444525288417935`", ",", 
   "0.7778101153671741`", ",", "0.11240461468696594`", ",", 
   "0.4961845874786377`", ",", "0.8473834991455078`", ",", 
   "0.8953399658203125`", ",", "0.8135986328125`", ",", "0.5439453125`", ",", 
   "0.7578125`", ",", "0.3125`", ",", "0.5`", ",", "0.`", ",", "0.`", ",", 
   "0.`"}], "}"}]], "Output",
 ImageSize->{471, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]= ",
 CellID->15618]
}, Open  ]],

Cell[TextData[{
 "By iterating the operation ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["FractionalPart",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FractionalPart"], "[", 
   RowBox[{"2", 
    StyleBox["x", "TI"]}], "]"}]], "InlineFormula"],
 " you extract successive binary digits in whatever number you start with. \
And if these digits are apparently random\[LongDash]as in a number like ",
 Cell[BoxData[
  FormBox["\[Pi]", TraditionalForm]], "InlineMath"],
 "\[LongDash]then the results will be correspondingly random. But if the \
digits have a simple pattern\[LongDash]as in any rational \
number\[LongDash]then the results you get will be correspondingly simple. "
}], "Text",
 CellID->25013],

Cell[TextData[{
 "By iterating an operation such as ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["FractionalPart",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FractionalPart"], "[", 
   RowBox[{
    RowBox[{"3", "/", "2"}], 
    StyleBox["x", "TI"]}], "]"}]], "InlineFormula"],
 " it turns out however to be possible to get seemingly random sequences even \
from very simple input. This is an example of a very general phenomenon first \
identified by Stephen Wolfram in the mid\[Hyphen]1980s, which has nothing \
directly to do with sensitive dependence on input."
}], "Text",
 CellID->26727],

Cell["\<\
This generates a seemingly random sequence, even starting from simple input. \
\
\>", "MathCaption",
 CellID->15712],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NestList", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"FractionalPart", "[", 
     RowBox[{
      RowBox[{"3", "/", "2"}], "#"}], "]"}], "&"}], ",", "1", ",", "15"}], 
  "]"}]], "Input",
 CellLabel->"In[8]:= ",
 CellID->27896],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1", ",", 
   FractionBox["1", "2"], ",", 
   FractionBox["3", "4"], ",", 
   FractionBox["1", "8"], ",", 
   FractionBox["3", "16"], ",", 
   FractionBox["9", "32"], ",", 
   FractionBox["27", "64"], ",", 
   FractionBox["81", "128"], ",", 
   FractionBox["243", "256"], ",", 
   FractionBox["217", "512"], ",", 
   FractionBox["651", "1024"], ",", 
   FractionBox["1953", "2048"], ",", 
   FractionBox["1763", "4096"], ",", 
   FractionBox["5289", "8192"], ",", 
   FractionBox["15867", "16384"], ",", 
   FractionBox["14833", "32768"]}], "}"}]], "Output",
 ImageSize->{567, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]= ",
 CellID->21909]
}, Open  ]],

Cell["\<\
After the values have been computed, one can safely find numerical \
approximations to them. \
\>", "MathCaption",
 CellID->27841],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", "%", "]"}]], "Input",
 CellLabel->"In[9]:= ",
 CellID->20130],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1.`", ",", "0.5`", ",", "0.75`", ",", "0.125`", ",", "0.1875`", ",", 
   "0.28125`", ",", "0.421875`", ",", "0.6328125`", ",", "0.94921875`", ",", 
   "0.423828125`", ",", "0.6357421875`", ",", "0.95361328125`", ",", 
   "0.430419921875`", ",", "0.6456298828125`", ",", "0.96844482421875`", ",", 
   "0.452667236328125`"}], "}"}]], "Output",
 ImageSize->{465, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[9]= ",
 CellID->9090]
}, Open  ]],

Cell["\<\
Here are the last 5 results after 1000 iterations, computed using exact \
numbers. \
\>", "MathCaption",
 CellID->9034],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Take", "[", 
  RowBox[{
   RowBox[{"N", "[", 
    RowBox[{"NestList", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"FractionalPart", "[", 
        RowBox[{
         RowBox[{"3", "/", "2"}], "#"}], "]"}], "&"}], ",", "1", ",", 
      "1000"}], "]"}], "]"}], ",", 
   RowBox[{"-", "5"}]}], "]"}]], "Input",
 CellLabel->"In[10]:= ",
 CellID->7915],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.021843931659136874`", ",", "0.03276589748870531`", ",", 
   "0.049148846233057965`", ",", "0.07372326934958695`", ",", 
   "0.11058490402438043`"}], "}"}]], "Output",
 ImageSize->{371, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[10]= ",
 CellID->6286]
}, Open  ]],

Cell["\<\
Using machine\[Hyphen]precision numbers gives completely incorrect results. \
\>", "MathCaption",
 CellID->3523],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Take", "[", 
  RowBox[{
   RowBox[{"NestList", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"FractionalPart", "[", 
       RowBox[{
        RowBox[{"3", "/", "2"}], "#"}], "]"}], "&"}], ",", "1.", ",", 
     "1000"}], "]"}], ",", 
   RowBox[{"-", "5"}]}], "]"}]], "Input",
 CellLabel->"In[11]:= ",
 CellID->20666],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.6706644021252268`", ",", "0.005996603187840233`", ",", 
   "0.008994904781760349`", ",", "0.013492357172640523`", ",", 
   "0.020238535758960785`"}], "}"}]], "Output",
 ImageSize->{371, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[11]= ",
 CellID->1344]
}, Open  ]],

Cell["\<\
Many kinds of iterative procedures yield functions that depend sensitively on \
their input. Such functions also arise when one looks at solutions to \
differential equations. In effect, varying the independent parameter in the \
differential equation is a continuous analog of going from one step to the \
next in an iterative procedure.\
\>", "Text",
 CellID->32736],

Cell["\<\
This finds a solution to the Duffing equation with initial condition 1. \
\>", "MathCaption",
 CellID->22037],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"x", "''"}], "[", "t", "]"}], "+", 
       RowBox[{"0.15", " ", 
        RowBox[{
         RowBox[{"x", "'"}], "[", "t", "]"}]}], "-", 
       RowBox[{"x", "[", "t", "]"}], "+", 
       RowBox[{
        RowBox[{"x", "[", "t", "]"}], "^", "3"}]}], "==", 
      RowBox[{"0.3", " ", 
       RowBox[{"Cos", "[", "t", "]"}]}]}], ",", 
     RowBox[{
      RowBox[{"x", "[", "0", "]"}], "==", 
      RowBox[{"-", "1"}]}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "'"}], "[", "0", "]"}], "==", "1"}]}], "}"}], ",", "x", 
   ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "50"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[13]:=",
 CellID->5416],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"x", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "50.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{316, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[13]=",
 CellID->569505467]
}, Open  ]],

Cell["Here is a plot of the solution. ", "MathCaption",
 CellID->1949],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"x", "[", "t", "]"}], "/.", "%"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "50"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[14]:=",
 CellID->30971],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 108},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[14]=",
 CellID->399960829]
}, Open  ]],

Cell["Here is the same equation with initial condition 1.001. ", "MathCaption",
 CellID->22777],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"x", "''"}], "[", "t", "]"}], "+", 
       RowBox[{"0.15", " ", 
        RowBox[{
         RowBox[{"x", "'"}], "[", "t", "]"}]}], "-", 
       RowBox[{"x", "[", "t", "]"}], "+", 
       RowBox[{
        RowBox[{"x", "[", "t", "]"}], "^", "3"}]}], "==", 
      RowBox[{"0.3", " ", 
       RowBox[{"Cos", "[", "t", "]"}]}]}], ",", 
     RowBox[{
      RowBox[{"x", "[", "0", "]"}], "==", 
      RowBox[{"-", "1"}]}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "'"}], "[", "0", "]"}], "==", "1.001"}]}], "}"}], ",", 
   "x", ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "50"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[15]:=",
 CellID->19285],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"x", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "50.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{316, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[15]=",
 CellID->607054951]
}, Open  ]],

Cell["\<\
The solution progressively diverges from the one shown above. \
\>", "MathCaption",
 CellID->14590],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"x", "[", "t", "]"}], "/.", "%"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "50"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[16]:=",
 CellID->10285],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 104},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[16]=",
 CellID->407599545]
}, Open  ]],

Cell[CellGroupData[{

Cell["RELATED TUTORIALS", "RelatedTutorialsSection",
 CellID->262288586],

Cell[TextData[ButtonBox["Numerical Operations on Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/NumericalOperationsOnFunctionsOverview"]], \
"RelatedTutorials",
 CellID->163046772]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Functions with Sensitive Dependence on Their Input - Wolfram \
Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, "Overview" -> {"toc" -> Cell[
      BoxData[
       ButtonBox[
        StyleBox["Numerical Operations on Functions", "OverviewNavText"], 
        BaseStyle -> "Link", ButtonData -> 
        "paclet:tutorial/NumericalOperationsOnFunctionsOverview"]], "Text", 
      FontFamily -> "Verdana"], "prevnext" -> Cell[
      TextData[{
        ButtonBox[
         StyleBox["\[FilledLeftTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> 
         "paclet:tutorial/MonitoringAndSelectingAlgorithms"]}], "Text", 
      FontFamily -> "Verdana"]}, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 36, 18.5503240}", "context" -> "", 
    "keywords" -> {
     "attractors", "autoplectic sequences", "chaos theory", "complexity", 
      "digit excavation", "Duffing equation", "FractionalPart", 
      "iterated maps", "random sequences", 
      "sensitive dependence on initial conditions", "shift map", 
      "strange attractors", "symbolic dynamics"}, "index" -> True, "label" -> 
    "Mathematica Tutorial", "language" -> "en", "paclet" -> "Mathematica", 
    "status" -> "None", "summary" -> 
    "Functions that are specified by simple algebraic formulas tend to be \
such that when their input is changed only slightly, their output also \
changes only slightly. But functions that are instead based on executing \
procedures quite often show almost arbitrarily sensitive dependence on their \
input. Typically the reason this happens is that the procedure 'excavates' \
progressively less and less significant digits in the input. This shows \
successive steps in a simple iterative procedure with input 0.1111. Here is \
the result with input 0.1112. Progressive divergence from the result with \
input 0.1111 is seen. ", "synonyms" -> {"3.9.11"}, "title" -> 
    "Functions with Sensitive Dependence on Their Input", "type" -> 
    "Tutorial", "uri" -> 
    "tutorial/FunctionsWithSensitiveDependenceOnTheirInput"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "S3.9.11"->{
  Cell[1299, 44, 115, 2, 70, "Title",
   CellTags->{"S3.9.11", "11.1"},
   CellID->15338]},
 "11.1"->{
  Cell[1299, 44, 115, 2, 70, "Title",
   CellTags->{"S3.9.11", "11.1"},
   CellID->15338]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"S3.9.11", 33189, 953},
 {"11.1", 33292, 957}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 674, 17, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[1299, 44, 115, 2, 70, "Title",
 CellTags->{"S3.9.11", "11.1"},
 CellID->15338],
Cell[1417, 48, 523, 9, 70, "Text",
 CellID->17342],
Cell[1943, 59, 128, 4, 70, "MathCaption",
 CellID->14428],
Cell[CellGroupData[{
Cell[2096, 67, 226, 8, 28, "Input",
 CellID->29795],
Cell[2325, 77, 462, 12, 36, "Output",
 CellID->30742]
}, Open  ]],
Cell[2802, 92, 153, 4, 70, "MathCaption",
 CellID->30037],
Cell[CellGroupData[{
Cell[2980, 100, 225, 8, 28, "Input",
 CellID->5506],
Cell[3208, 110, 461, 12, 36, "Output",
 CellID->6767]
}, Open  ]],
Cell[3684, 125, 734, 20, 70, "Text",
 CellID->5053],
Cell[4421, 147, 407, 14, 70, "MathCaption",
 CellID->9793],
Cell[CellGroupData[{
Cell[4853, 165, 222, 7, 28, "Input",
 CellID->28505],
Cell[5078, 174, 1038, 21, 54, "Output",
 Evaluatable->False,
 CellID->482572570]
}, Open  ]],
Cell[6131, 198, 908, 21, 70, "Text",
 CellID->3609],
Cell[7042, 221, 150, 4, 70, "MathCaption",
 CellID->13924],
Cell[CellGroupData[{
Cell[7217, 229, 302, 11, 70, "Input",
 CellID->19854],
Cell[7522, 242, 1186, 24, 92, "Output",
 CellID->29465]
}, Open  ]],
Cell[8723, 269, 164, 4, 70, "MathCaption",
 CellID->12239],
Cell[CellGroupData[{
Cell[8912, 277, 127, 4, 70, "Input",
 CellID->26206],
Cell[9042, 283, 633, 14, 54, "Output",
 CellID->24209]
}, Open  ]],
Cell[9690, 300, 98, 2, 70, "MathCaption",
 CellID->15492],
Cell[CellGroupData[{
Cell[9813, 306, 243, 8, 70, "Input",
 CellID->16499],
Cell[10059, 316, 533, 18, 51, "Output",
 CellID->18175]
}, Open  ]],
Cell[10607, 337, 610, 13, 70, "Text",
 CellID->31036],
Cell[11220, 352, 214, 6, 70, "MathCaption",
 CellID->23061],
Cell[CellGroupData[{
Cell[11459, 362, 273, 9, 70, "Input",
 CellID->13647],
Cell[11735, 373, 716, 16, 71, "Output",
 CellID->15618]
}, Open  ]],
Cell[12466, 392, 708, 17, 70, "Text",
 CellID->25013],
Cell[13177, 411, 597, 15, 70, "Text",
 CellID->26727],
Cell[13777, 428, 126, 4, 70, "MathCaption",
 CellID->15712],
Cell[CellGroupData[{
Cell[13928, 436, 248, 9, 70, "Input",
 CellID->27896],
Cell[14179, 447, 725, 22, 51, "Output",
 CellID->21909]
}, Open  ]],
Cell[14919, 472, 140, 4, 70, "MathCaption",
 CellID->27841],
Cell[CellGroupData[{
Cell[15084, 480, 94, 3, 70, "Input",
 CellID->20130],
Cell[15181, 485, 511, 12, 54, "Output",
 CellID->9090]
}, Open  ]],
Cell[15707, 500, 129, 4, 70, "MathCaption",
 CellID->9034],
Cell[CellGroupData[{
Cell[15861, 508, 373, 13, 70, "Input",
 CellID->7915],
Cell[16237, 523, 339, 10, 36, "Output",
 CellID->6286]
}, Open  ]],
Cell[16591, 536, 122, 3, 70, "MathCaption",
 CellID->3523],
Cell[CellGroupData[{
Cell[16738, 543, 339, 12, 70, "Input",
 CellID->20666],
Cell[17080, 557, 340, 10, 36, "Output",
 CellID->1344]
}, Open  ]],
Cell[17435, 570, 378, 7, 70, "Text",
 CellID->32736],
Cell[17816, 579, 119, 3, 70, "MathCaption",
 CellID->22037],
Cell[CellGroupData[{
Cell[17960, 586, 799, 27, 70, "Input",
 CellID->5416],
Cell[18762, 615, 468, 17, 36, "Output",
 CellID->569505467]
}, Open  ]],
Cell[19245, 635, 70, 1, 70, "MathCaption",
 CellID->1949],
Cell[CellGroupData[{
Cell[19340, 640, 273, 9, 28, "Input",
 CellID->30971],
Cell[19616, 651, 4532, 78, 129, "Output",
 Evaluatable->False,
 CellID->399960829]
}, Open  ]],
Cell[24163, 732, 95, 1, 70, "MathCaption",
 CellID->22777],
Cell[CellGroupData[{
Cell[24283, 737, 804, 27, 70, "Input",
 CellID->19285],
Cell[25090, 766, 468, 17, 36, "Output",
 CellID->607054951]
}, Open  ]],
Cell[25573, 786, 109, 3, 70, "MathCaption",
 CellID->14590],
Cell[CellGroupData[{
Cell[25707, 793, 273, 9, 28, "Input",
 CellID->10285],
Cell[25983, 804, 4065, 71, 125, "Output",
 Evaluatable->False,
 CellID->407599545]
}, Open  ]],
Cell[CellGroupData[{
Cell[30085, 880, 72, 1, 70, "RelatedTutorialsSection",
 CellID->262288586],
Cell[30160, 883, 198, 5, 70, "RelatedTutorials",
 CellID->163046772]
}, Open  ]]
}, Open  ]],
Cell[30385, 892, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

