(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     33309,       1122]
NotebookOptionsPosition[     26207,        895]
NotebookOutlinePosition[     29331,        971]
CellTagsIndexPosition[     29238,        965]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Calculus" :> 
          Documentation`HelpLookup["paclet:tutorial/CalculusOverview"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Calculus\"\>"}, "\<\"related tutorials\"\>", 
       StyleBox["\<\"related tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"DiracDelta" :> 
          Documentation`HelpLookup["paclet:ref/DiracDelta"], "DiscreteDelta" :> 
          Documentation`HelpLookup["paclet:ref/DiscreteDelta"], 
          "HeavisideTheta" :> 
          Documentation`HelpLookup["paclet:ref/HeavisideTheta"], 
          "KroneckerDelta" :> 
          Documentation`HelpLookup["paclet:ref/KroneckerDelta"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"DiracDelta\"\>", 
       2->"\<\"DiscreteDelta\"\>", 3->"\<\"HeavisideTheta\"\>", 
       4->"\<\"KroneckerDelta\"\>"}, "\<\"functions\"\>", 
       StyleBox["\<\"functions\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["Generalized Functions and Related Objects", "Title",
 CellTags->{"S3.5.13", "13.1"},
 CellID->16141],

Cell[TextData[{
 "In many practical situations it is convenient to consider limits in which a \
fixed amount of something is concentrated into an infinitesimal region. \
Ordinary mathematical functions of the kind normally encountered in calculus \
cannot readily represent such limits. However, it is possible to introduce ",
 StyleBox["generalized functions",
  FontSlant->"Italic"],
 " or ",
 StyleBox["distributions",
  FontSlant->"Italic"],
 " which can represent these limits in integrals and other types of \
calculations. "
}], "Text",
 CellID->2670],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     ButtonBox["DiracDelta",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/DiracDelta"], "[", 
     StyleBox["x", "TI"], "]"}], Cell[TextData[{
     "Dirac delta function ",
     Cell[BoxData[
      FormBox[
       RowBox[{"\[Delta]", "(", "x", ")"}], TraditionalForm]], "InlineMath"]
    }], "TableText"]},
   {
    RowBox[{
     ButtonBox["HeavisideTheta",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/HeavisideTheta"], "[", 
     StyleBox["x", "TI"], "]"}], Cell[TextData[{
     "Heaviside theta function ",
     Cell[BoxData[
      FormBox[
       RowBox[{"\[Theta]", "(", "x", ")"}], TraditionalForm]], "InlineMath"],
     ", equal to 0 for ",
     Cell[BoxData[
      FormBox[
       RowBox[{"x", "<", "0"}], TraditionalForm]], "InlineMath"],
     " and 1 for ",
     Cell[BoxData[
      FormBox[
       RowBox[{"x", ">", "0"}], TraditionalForm]], "InlineMath"]
    }], "TableText"]}
  }]], "DefinitionBox",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {{False}}, 
   "RowsIndexed" -> {}},
 GridBoxSpacings->{
  "Columns" -> {0.28, {0.56}, 0.28}, "ColumnsIndexed" -> {}, 
   "Rows" -> {0.2, {0.16}, 0.2}, "RowsIndexed" -> {}}},
 CellID->10845],

Cell["Dirac delta and Heaviside theta functions. ", "Caption",
 CellID->10506],

Cell[TextData[{
 "Here is a function concentrated around ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "=", "0"}], TraditionalForm]], "InlineMath"],
 ". "
}], "MathCaption",
 CellID->9088],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Sqrt", "[", 
     RowBox[{"50", "/", "Pi"}], "]"}], " ", 
    RowBox[{"Exp", "[", 
     RowBox[{
      RowBox[{"-", "50"}], " ", 
      RowBox[{"x", "^", "2"}]}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", 
   RowBox[{"PlotRange", "->", "All"}]}], "]"}]], "Input",
 CellLabel->"In[1]:= ",
 CellID->23256],

Cell[GraphicsData["CompressedBitmap", "\<\
eJytWD1MXEcQXu69g7MTRSgVhYtrbFOkoKREikIogjlz/FluQFakcxEFORQ0
UUpKSkpKSlxYoqSkpKSkpKSkPM/u7Nvv23nLYUs5ibu3u983Ozvz7ew+VvcO
Rn/+tXfw8cNe//dPe/ujjx/+6S///Um6qinn3Bf5+3fOyfNYnsLfl/FYftvf
0/JdC+C1/FbyuxZ/X8X+OqBm/LfvDi3n5qntH9dM+5XBqxVXxaZ8Ok2TWP04
p+231l+btpp9NsGHkhW7hs7/aMVH1dXofTpiL1sMRL4T+f53YNpdyuV7Y3Pr
u3uD39Xx8fF/4eN+9V/VycnJ7e2tH/rRsJ6H9sPDw/n5uYf+FvAK9kNTwarP
z2chTdEcsPHTE+2uaVv+zwF0eHi4v7+fgud6JSvTqX1zc7O4uCiO/vCd3jzl
HeWfepdDezAYSJz8o1WZeruwsHB9fe0fX5jxX4JV1YxABFiyortUppCJ/OMf
gTUwtm1klLWysnJxcWFZR0dHEtiG5fWm6Pv7+9nZWf+4Smh4ZufQeJyenu7u
7oL1NvSKITEHTyxrNBqJAC0LsSyzsF6wOqLOoi68gnu9MPSG5kAkm104bdbJ
6Lu7u7m5uZJHOocsQpYC1nroFYoQ4bdliahlO1rW0tLS5eXl4yzR9tXVlWXt
7OycnZ09zoIvzBIBiAwaVqo5FLM1QmOVdo5QzyrZePPz85Yl0hMBPs5CJpiF
PJRZEjitCRkL2i2zEHFlDY1Ku1kUEFNGyxJloYisnQP5Yxb2YJnV7/djHQar
Sp5YtCRHUmTnkFTGY9myKhOdAbFk5xKrSmc0MsBoWdrEOZBtZsnSJrJQPZgl
S5zIQgUDy2cvMDOkZUL3ytwwOaijJY22uCYOWjSqUF2cAxlnFtRRZmF3Mwse
lFnwnFlQe5kFFYHl62izCyyLbSNjFtVpMi5O+ce3xMJeL7OgOWZJwCRsj7Mw
zixUozILK2AWKn2ZhbqjrM1iZHOWj+k6oVGLS2jONs+BulNm4dRkFvRWZmGc
WbBVZkFvzILfVVRQZdYLtI9JOcJoK9pHrg67GpfAgMSNTz6at1A3tHrU6Q1F
3KpbeNXnkDIvoyok10brwDDPuGxVXqt6j/q3nqPjbqhatlWHw7RW1+U3BVlU
wR9VIVarVU8ZaZaaGT7eig9vA3TmtWOptzJGI9+dbLXI+DqhUcPatlVvbBsK
bKPVEqMxYxutXjIa98s2WiMAtI9QnkvkgVeHU6eTZ0uPV1VTxKLGGKzKL8Oi
NhqsREDikGFR8Qw2KpixqINN7oI81LUMiZptrOrFL2K387HQ51KdY0w6XwjD
p9KQsElhGRbZJmydKiGvR4/bzMekrMwm9Mbzp4qXYaF77M+qGe0m7zIOqh3V
i7gX9d7RoR6lbraQqjJkDHc3XWGz64OtXtLZuCkBmku1u5FnTEuUYy9UYm0v
VE6MVPJmOxq9pJ7kg3q1iSj30ioSRiuZYnZyP2Mt86qi8RnkJIzrFneMwEkZ
EHqNzxDQQEBoM0NAeQGhJhlBOtYMIe8bQE2jWjTH0RaNoj44p/HjUZwlzmmG
eBR3JecUSaO8l4N3kId6926MChe0K1HewkhXVyMDMb48plVZxnTTZGMx2M7F
iGHMz2Bv5+RLHQtSYG5Dt/Z+UMdyGGafhNN3WOmVebeTXtP/Pf0dR569jxjl
iIR3jlrfXPSzle08+4b5LaPWU3sDUlT7P7lu6iugGAIK\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 125},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]= ",
 CellID->1962564]
}, Open  ]],

Cell[TextData[{
 "As ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " gets larger, the functions become progressively more concentrated. "
}], "MathCaption",
 CellID->8266],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Sqrt", "[", 
       RowBox[{"n", "/", "Pi"}], "]"}], " ", 
      RowBox[{"Exp", "[", 
       RowBox[{
        RowBox[{"-", "n"}], " ", 
        RowBox[{"x", "^", "2"}]}], "]"}]}], "/.", 
     RowBox[{"n", "->", 
      RowBox[{"{", 
       RowBox[{"1", ",", "10", ",", "100"}], "}"}]}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", 
   RowBox[{"PlotRange", "->", "All"}]}], "]"}]], "Input",
 CellLabel->"In[2]:= ",
 CellID->5857],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 122},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]= ",
 CellID->90912667]
}, Open  ]],

Cell[TextData[{
 "For any ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", ">", "0"}], TraditionalForm]], "InlineMath"],
 ", their integrals are nevertheless always equal to 1. "
}], "MathCaption",
 CellID->14199],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Sqrt", "[", 
     RowBox[{"n", "/", "Pi"}], "]"}], " ", 
    RowBox[{"Exp", "[", 
     RowBox[{
      RowBox[{"-", "n"}], " ", 
      RowBox[{"x", "^", "2"}]}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "Infinity"}], ",", "Infinity"}], "}"}], ",", 
   RowBox[{"Assumptions", "->", 
    RowBox[{"n", ">", "0"}]}]}], "]"}]], "Input",
 CellLabel->"In[3]:= ",
 CellID->12674],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]= ",
 CellID->14155]
}, Open  ]],

Cell[TextData[{
 "The limit of the functions for infinite ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " is effectively a Dirac delta function, whose integral is again 1. "
}], "MathCaption",
 CellID->25191],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"DiracDelta", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "Infinity"}], ",", "Infinity"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[4]:= ",
 CellID->8990],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]= ",
 CellID->29832]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["DiracDelta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DiracDelta"]], "InlineFormula"],
 " evaluates to 0 at all real points except ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "=", "0"}], TraditionalForm]], "InlineMath"],
 ". "
}], "MathCaption",
 CellID->30858],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"DiracDelta", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "3"}], ",", "3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[5]:= ",
 CellID->10996],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0", ",", "0", ",", "0", ",", 
   RowBox[{"DiracDelta", "[", "0", "]"}], ",", "0", ",", "0", ",", "0"}], 
  "}"}]], "Output",
 ImageSize->{219, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]= ",
 CellID->8917]
}, Open  ]],

Cell["\<\
Inserting a delta function in an integral effectively causes the integrand to \
be sampled at discrete points where the argument of the delta function \
vanishes. \
\>", "Text",
 CellID->28290],

Cell[TextData[{
 "This samples the function ",
 Cell[BoxData["f"], "InlineFormula"],
 " with argument ",
 Cell[BoxData["2"], "InlineFormula"],
 ". "
}], "MathCaption",
 CellID->3967],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"DiracDelta", "[", 
     RowBox[{"x", "-", "2"}], "]"}], " ", 
    RowBox[{"f", "[", "x", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "4"}], ",", "4"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[6]:= ",
 CellID->1707],

Cell[BoxData[
 RowBox[{"f", "[", "2", "]"}]], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]= ",
 CellID->8234]
}, Open  ]],

Cell["Here is a slightly more complicated example. ", "MathCaption",
 CellID->28246],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"DiracDelta", "[", 
    RowBox[{
     RowBox[{"x", "^", "2"}], "-", "x", "-", "1"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[7]:= ",
 CellID->11462],

Cell[BoxData[
 FractionBox["1", 
  SqrtBox["5"]]], "Output",
 ImageSize->{32, 34},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]= ",
 CellID->19402]
}, Open  ]],

Cell[TextData[{
 "This effectively counts the number of zeros of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"cos", "(", "x", ")"}], TraditionalForm]], "InlineMath"],
 " in the region of integration. "
}], "MathCaption",
 CellID->8239],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"DiracDelta", "[", 
    RowBox[{"Cos", "[", "x", "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "30"}], ",", "30"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[8]:= ",
 CellID->3426],

Cell[BoxData["20"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]= ",
 CellID->19095]
}, Open  ]],

Cell[TextData[{
 "The ",
 StyleBox["Heaviside function",
  FontWeight->"Bold"],
 " ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["HeavisideTheta",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/HeavisideTheta"], "[", 
   StyleBox["x", "TI"], "]"}]], "InlineFormula"],
 " is the indefinite integral of the delta function. It is variously denoted \
",
 Cell[BoxData[
  FormBox[
   RowBox[{"H", "(", "x", ")"}], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Theta]", "(", "x", ")"}], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Mu]", "(", "x", ")"}], TraditionalForm]], "InlineMath"],
 ", and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"U", "(", "x", ")"}], TraditionalForm]], "InlineMath"],
 ". As a generalized function, the Heaviside function is defined only inside \
an integral. This distinguishes it from the unit step function ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["UnitStep",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/UnitStep"], "[", 
   StyleBox["x", "TI"], "]"}]], "InlineFormula"],
 ", which is a piecewise function."
}], "Text",
 CellID->216576651],

Cell["\<\
The indefinite integral of the delta function is the Heaviside theta \
function. \
\>", "MathCaption",
 CellID->8314],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"DiracDelta", "[", "x", "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[16]:=",
 CellID->10350],

Cell[BoxData[
 RowBox[{"HeavisideTheta", "[", "x", "]"}]], "Output",
 ImageSize->{123, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[16]=",
 CellID->346418128]
}, Open  ]],

Cell[TextData[{
 "The value of this integral depends on whether ",
 Cell[BoxData[
  FormBox["a", TraditionalForm]], "InlineMath"],
 " lies in the interval ",
 Cell[BoxData[
  FormBox[
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "2"}], ",", "2"}], ")"}], TraditionalForm]], "InlineMath"],
 ". "
}], "MathCaption",
 CellID->5676],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"f", "[", "x", "]"}], " ", 
    RowBox[{"DiracDelta", "[", 
     RowBox[{"x", "-", "a"}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", 
   RowBox[{"Assumptions", "->", 
    RowBox[{"a", " ", "\[Element]", " ", "Reals"}]}]}], "]"}]], "Input",
 CellLabel->"In[22]:=",
 CellID->3164],

Cell[BoxData[
 RowBox[{
  RowBox[{"f", "[", "a", "]"}], " ", 
  RowBox[{"HeavisideTheta", "[", 
   RowBox[{"2", "-", "a"}], "]"}], " ", 
  RowBox[{"HeavisideTheta", "[", 
   RowBox[{"2", "+", "a"}], "]"}]}]], "Output",
 ImageSize->{316, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[22]=",
 CellID->323275187]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["DiracDelta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DiracDelta"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["HeavisideTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HeavisideTheta"]], "InlineFormula"],
 " often arise in doing integral transforms. "
}], "Text",
 CellID->32570],

Cell["\<\
The Fourier transform of a constant function is a delta function. \
\>", "MathCaption",
 CellID->6038],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FourierTransform", "[", 
  RowBox[{"1", ",", "t", ",", "\[Omega]"}], "]"}]], "Input",
 CellLabel->"In[13]:= ",
 CellID->24312],

Cell[BoxData[
 RowBox[{
  SqrtBox[
   RowBox[{"2", " ", "\[Pi]"}]], " ", 
  RowBox[{"DiracDelta", "[", "\[Omega]", "]"}]}]], "Output",
 ImageSize->{133, 17},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[13]= ",
 CellID->10144]
}, Open  ]],

Cell[TextData[{
 "The Fourier transform of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"cos", "(", "t", ")"}], TraditionalForm]], "InlineMath"],
 " involves the sum of two delta functions. "
}], "MathCaption",
 CellID->16561],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FourierTransform", "[", 
  RowBox[{
   RowBox[{"Cos", "[", "t", "]"}], ",", "t", ",", "\[Omega]"}], 
  "]"}]], "Input",
 CellLabel->"In[14]:= ",
 CellID->28180],

Cell[BoxData[
 RowBox[{
  RowBox[{
   SqrtBox[
    FractionBox["\[Pi]", "2"]], " ", 
   RowBox[{"DiracDelta", "[", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "\[Omega]"}], "]"}]}], "+", 
  RowBox[{
   SqrtBox[
    FractionBox["\[Pi]", "2"]], " ", 
   RowBox[{"DiracDelta", "[", 
    RowBox[{"1", "+", "\[Omega]"}], "]"}]}]}]], "Output",
 ImageSize->{316, 39},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[14]= ",
 CellID->18200]
}, Open  ]],

Cell[TextData[{
 "Dirac delta functions can be used in ",
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 " to find the impulse response or Green\[CloseCurlyQuote]s function of \
systems represented by linear and certain other differential equations. "
}], "Text",
 CellID->30643],

Cell[TextData[{
 "This finds the behavior of a harmonic oscillator subjected to an impulse at \
",
 Cell[BoxData[
  FormBox[
   RowBox[{"t", "=", "0"}], TraditionalForm]], "InlineMath"],
 ". "
}], "MathCaption",
 CellID->21856],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"x", "''"}], "[", "t", "]"}], "+", 
       RowBox[{"r", " ", 
        RowBox[{"x", "[", "t", "]"}]}]}], "==", 
      RowBox[{"DiracDelta", "[", "t", "]"}]}], ",", 
     RowBox[{
      RowBox[{"x", "[", "0", "]"}], "==", "0"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "'"}], "[", "0", "]"}], "==", "1"}]}], "}"}], ",", 
   RowBox[{"x", "[", "t", "]"}], ",", "t"}], "]"}]], "Input",
 CellLabel->"In[15]:= ",
 CellID->18061],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"x", "[", "t", "]"}], "\[Rule]", 
    FractionBox[
     RowBox[{
      RowBox[{"HeavisideTheta", "[", "t", "]"}], " ", 
      RowBox[{"Sin", "[", 
       RowBox[{
        SqrtBox["r"], " ", "t"}], "]"}]}], 
     SqrtBox["r"]]}], "}"}], "}"}]], "Output",
 ImageSize->{271, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[15]= ",
 CellID->2647]
}, Open  ]],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     ButtonBox["DiracDelta",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/DiracDelta"], "[", 
     SubscriptBox[
      StyleBox["x", "TI"], "1"], ",", 
     SubscriptBox[
      StyleBox["x", "TI"], "2"], ",", 
     StyleBox["\[Ellipsis]", "TR"], "]"}], Cell[
    "multidimensional Dirac delta function", "TableText"]},
   {
    RowBox[{
     ButtonBox["HeavisideTheta",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/HeavisideTheta"], "[", 
     SubscriptBox[
      StyleBox["x", "TI"], "1"], ",", 
     SubscriptBox[
      StyleBox["x", "TI"], "2"], ",", 
     StyleBox["\[Ellipsis]", "TR"], "]"}], Cell[
    "multidimensional Heaviside theta function", "TableText"]}
  }]], "DefinitionBox",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {{False}}, 
   "RowsIndexed" -> {}},
 GridBoxSpacings->{
  "Columns" -> {0.28, {0.56}, 0.28}, "ColumnsIndexed" -> {}, 
   "Rows" -> {0.2, {0.16}, 0.2}, "RowsIndexed" -> {}}},
 CellID->5939],

Cell["Multidimensional Dirac delta and Heaviside theta functions. ", "Caption",
 CellID->19808],

Cell["\<\
Multidimensional generalized functions are essentially products of univariate \
generalized functions.\
\>", "Text",
 CellID->311274134],

Cell["\<\
Here is a derivative of a multidimensional Heaviside function.\
\>", "MathCaption",
 CellID->45962922],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"D", "[", 
  RowBox[{
   RowBox[{"HeavisideTheta", "[", 
    RowBox[{"x", ",", "y", ",", "z"}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[26]:=",
 CellID->525847495],

Cell[BoxData[
 RowBox[{
  RowBox[{"DiracDelta", "[", "x", "]"}], " ", 
  RowBox[{"HeavisideTheta", "[", 
   RowBox[{"y", ",", "z"}], "]"}]}]], "Output",
 ImageSize->{236, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[26]=",
 CellID->436973786]
}, Open  ]],

Cell[TextData[{
 "Related to the multidimensional Dirac delta function are two integer \
functions: discrete delta and Kronecker delta. Discrete delta ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Delta]", "(", 
    RowBox[{
     SubscriptBox["n", "1"], ",", 
     SubscriptBox["n", "2"], ",", "\[Ellipsis]"}], ")"}], TraditionalForm]], 
  "InlineMath"],
 " is 1 if all the ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["n", "i"], "=", "0"}], TraditionalForm]], "InlineMath"],
 ", and is zero otherwise. Kronecker delta ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[Delta]", 
    RowBox[{
     SubscriptBox["n", "1"], 
     SubscriptBox["n", "2"], "\[Ellipsis]"}]], TraditionalForm]], 
  "InlineMath"],
 " is 1 if all the ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["n", "i"], TraditionalForm]], "InlineMath"],
 " are equal, and is zero otherwise. "
}], "Text",
 CellID->9290],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     ButtonBox["DiscreteDelta",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/DiscreteDelta"], "[", 
     SubscriptBox[
      StyleBox["n", "TI"], "1"], ",", 
     SubscriptBox[
      StyleBox["n", "TI"], "2"], ",", 
     StyleBox["\[Ellipsis]", "TR"], "]"}], Cell[TextData[{
     "discrete delta ",
     Cell[BoxData[
      FormBox[
       RowBox[{"\[Delta]", "(", 
        RowBox[{
         SubscriptBox["n", "1"], ",", 
         SubscriptBox["n", "2"], ",", "\[Ellipsis]"}], ")"}], 
       TraditionalForm]], "InlineMath"]
    }], "TableText"]},
   {
    RowBox[{
     ButtonBox["KroneckerDelta",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/KroneckerDelta"], "[", 
     SubscriptBox[
      StyleBox["n", "TI"], "1"], ",", 
     SubscriptBox[
      StyleBox["n", "TI"], "2"], ",", 
     StyleBox["\[Ellipsis]", "TR"], "]"}], Cell[TextData[{
     "Kronecker delta ",
     Cell[BoxData[
      FormBox[
       SubscriptBox["\[Delta]", 
        RowBox[{
         SubscriptBox["n", "1"], 
         SubscriptBox["n", "2"], "\[Ellipsis]"}]], TraditionalForm]], 
      "InlineMath"]
    }], "TableText"]}
  }]], "DefinitionBox",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {{False}}, 
   "RowsIndexed" -> {}},
 GridBoxSpacings->{
  "Columns" -> {0.28, {0.56}, 0.28}, "ColumnsIndexed" -> {}, 
   "Rows" -> {0.2, {0.16}, 0.2}, "RowsIndexed" -> {}}},
 CellID->17452],

Cell["Integer delta functions. ", "Caption",
 CellID->979],

Cell[CellGroupData[{

Cell["RELATED TUTORIALS", "RelatedTutorialsSection",
 CellID->118549662],

Cell[TextData[ButtonBox["Calculus",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/CalculusOverview"]], "RelatedTutorials",
 CellID->808424779]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Generalized Functions and Related Objects - Wolfram \
Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, "Overview" -> {"toc" -> Cell[
      BoxData[
       ButtonBox[
        StyleBox["Calculus", "OverviewNavText"], BaseStyle -> "Link", 
        ButtonData -> "paclet:tutorial/CalculusOverview"]], "Text", 
      FontFamily -> "Verdana"], "prevnext" -> Cell[
      TextData[{
        ButtonBox[
         StyleBox["\[FilledLeftTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> 
         "paclet:tutorial/IntegralTransformsAndRelatedOperations"]}], "Text", 
      FontFamily -> "Verdana"]}, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 36, 20.6128768}", "context" -> "", 
    "keywords" -> {
     "causal signals", "characteristic function", "delta", "delta function", 
      "DiracDelta", "discontinuous functions", "DiscreteDelta", 
      "distributions", "generalized functions", "Greens functions", 
      "Heaviside function", "HeavisideTheta", "hyperfunctions", 
      "improper functions", "impulse function", "indicator function", 
      "KroneckerDelta", "Lebesgue integration", "limiting functions", 
      "Onesided functions", "piecewise continuous functions", 
      "theta function", "UnitStep"}, "index" -> True, "label" -> 
    "Mathematica Tutorial", "language" -> "en", "paclet" -> "Mathematica", 
    "status" -> "None", "summary" -> 
    "In many practical situations it is convenient to consider limits in \
which a fixed amount of something is concentrated into an infinitesimal \
region. Ordinary mathematical functions of the kind normally encountered in \
calculus cannot readily represent such limits. However, it is possible to \
introduce generalized functions or distributions which can represent these \
limits in integrals and other types of calculations. Dirac delta and \
Heaviside theta functions. Here is a function concentrated around x=0. ", 
    "synonyms" -> {"3.5.13"}, "title" -> 
    "Generalized Functions and Related Objects", "type" -> "Tutorial", "uri" -> 
    "tutorial/GeneralizedFunctionsAndRelatedObjects"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "S3.5.13"->{
  Cell[2148, 64, 106, 2, 70, "Title",
   CellTags->{"S3.5.13", "13.1"},
   CellID->16141]},
 "13.1"->{
  Cell[2148, 64, 106, 2, 70, "Title",
   CellTags->{"S3.5.13", "13.1"},
   CellID->16141]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"S3.5.13", 29020, 954},
 {"13.1", 29123, 958}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 1523, 37, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2148, 64, 106, 2, 70, "Title",
 CellTags->{"S3.5.13", "13.1"},
 CellID->16141],
Cell[2257, 68, 558, 13, 70, "Text",
 CellID->2670],
Cell[2818, 83, 1263, 39, 70, "DefinitionBox",
 CellID->10845],
Cell[4084, 124, 78, 1, 70, "Caption",
 CellID->10506],
Cell[4165, 127, 187, 7, 70, "MathCaption",
 CellID->9088],
Cell[CellGroupData[{
Cell[4377, 138, 438, 15, 28, "Input",
 CellID->23256],
Cell[4818, 155, 1964, 36, 146, "Output",
 Evaluatable->False,
 CellID->1962564]
}, Open  ]],
Cell[6797, 194, 193, 6, 70, "MathCaption",
 CellID->8266],
Cell[CellGroupData[{
Cell[7015, 204, 606, 20, 28, "Input",
 CellID->5857],
Cell[7624, 226, 2917, 52, 143, "Output",
 Evaluatable->False,
 CellID->90912667]
}, Open  ]],
Cell[10556, 281, 209, 7, 70, "MathCaption",
 CellID->14199],
Cell[CellGroupData[{
Cell[10790, 292, 480, 16, 70, "Input",
 CellID->12674],
Cell[11273, 310, 155, 5, 36, "Output",
 CellID->14155]
}, Open  ]],
Cell[11443, 318, 230, 6, 70, "MathCaption",
 CellID->25191],
Cell[CellGroupData[{
Cell[11698, 328, 256, 8, 70, "Input",
 CellID->8990],
Cell[11957, 338, 155, 5, 36, "Output",
 CellID->29832]
}, Open  ]],
Cell[12127, 346, 314, 11, 70, "MathCaption",
 CellID->30858],
Cell[CellGroupData[{
Cell[12466, 361, 239, 8, 70, "Input",
 CellID->10996],
Cell[12708, 371, 291, 9, 36, "Output",
 CellID->8917]
}, Open  ]],
Cell[13014, 383, 203, 5, 70, "Text",
 CellID->28290],
Cell[13220, 390, 182, 7, 70, "MathCaption",
 CellID->3967],
Cell[CellGroupData[{
Cell[13427, 401, 323, 11, 70, "Input",
 CellID->1707],
Cell[13753, 414, 181, 6, 36, "Output",
 CellID->8234]
}, Open  ]],
Cell[13949, 423, 84, 1, 70, "MathCaption",
 CellID->28246],
Cell[CellGroupData[{
Cell[14058, 428, 283, 9, 70, "Input",
 CellID->11462],
Cell[14344, 439, 187, 7, 55, "Output",
 CellID->19402]
}, Open  ]],
Cell[14546, 449, 231, 7, 70, "MathCaption",
 CellID->8239],
Cell[CellGroupData[{
Cell[14802, 460, 276, 9, 70, "Input",
 CellID->3426],
Cell[15081, 471, 156, 5, 36, "Output",
 CellID->19095]
}, Open  ]],
Cell[15252, 479, 1140, 38, 70, "Text",
 CellID->216576651],
Cell[16395, 519, 127, 4, 70, "MathCaption",
 CellID->8314],
Cell[CellGroupData[{
Cell[16547, 527, 163, 5, 70, "Input",
 CellID->10350],
Cell[16713, 534, 200, 6, 36, "Output",
 CellID->346418128]
}, Open  ]],
Cell[16928, 543, 331, 12, 70, "MathCaption",
 CellID->5676],
Cell[CellGroupData[{
Cell[17284, 559, 416, 13, 70, "Input",
 CellID->3164],
Cell[17703, 574, 350, 11, 36, "Output",
 CellID->323275187]
}, Open  ]],
Cell[18068, 588, 354, 12, 70, "Text",
 CellID->32570],
Cell[18425, 602, 112, 3, 70, "MathCaption",
 CellID->6038],
Cell[CellGroupData[{
Cell[18562, 609, 150, 4, 70, "Input",
 CellID->24312],
Cell[18715, 615, 263, 9, 38, "Output",
 CellID->10144]
}, Open  ]],
Cell[18993, 627, 221, 7, 70, "MathCaption",
 CellID->16561],
Cell[CellGroupData[{
Cell[19239, 638, 184, 6, 70, "Input",
 CellID->28180],
Cell[19426, 646, 466, 17, 60, "Output",
 CellID->18200]
}, Open  ]],
Cell[19907, 666, 348, 9, 70, "Text",
 CellID->30643],
Cell[20258, 677, 227, 8, 70, "MathCaption",
 CellID->21856],
Cell[CellGroupData[{
Cell[20510, 689, 577, 19, 70, "Input",
 CellID->18061],
Cell[21090, 710, 450, 16, 64, "Output",
 CellID->2647]
}, Open  ]],
Cell[21555, 729, 1040, 31, 70, "DefinitionBox",
 CellID->5939],
Cell[22598, 762, 95, 1, 70, "Caption",
 CellID->19808],
Cell[22696, 765, 146, 4, 70, "Text",
 CellID->311274134],
Cell[22845, 771, 112, 3, 70, "MathCaption",
 CellID->45962922],
Cell[CellGroupData[{
Cell[22982, 778, 198, 6, 70, "Input",
 CellID->525847495],
Cell[23183, 786, 284, 9, 36, "Output",
 CellID->436973786]
}, Open  ]],
Cell[23482, 798, 888, 29, 70, "Text",
 CellID->9290],
Cell[24373, 829, 1473, 47, 70, "DefinitionBox",
 CellID->17452],
Cell[25849, 878, 58, 1, 70, "Caption",
 CellID->979],
Cell[CellGroupData[{
Cell[25932, 883, 72, 1, 70, "RelatedTutorialsSection",
 CellID->118549662],
Cell[26007, 886, 146, 3, 70, "RelatedTutorials",
 CellID->808424779]
}, Open  ]]
}, Open  ]],
Cell[26180, 893, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

